
/*****************************************************************************/
/*                                                                           */
/*  THE HSEVAL HIGH SCHOOL TIMETABLE EVALUATOR                               */
/*  COPYRIGHT (C) 2009, Jeffrey H. Kingston                                  */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 3, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         time_group.c                                               */
/*  MODULE:       A time group in the hseval timetable model                 */
/*                                                                           */
/*****************************************************************************/
#include "externs.h"

#define DEBUG1 0


/*****************************************************************************/
/*                                                                           */
/*  WEEK                                                                     */
/*                                                                           */
/*****************************************************************************/

struct week_rec {
  KHE_TIME_GROUP	time_group;		/* tg for week, else NULL    */
  ARRAY_DAY		days;			/* days of week              */
};


/*****************************************************************************/
/*                                                                           */
/*  DAY                                                                      */
/*                                                                           */
/*****************************************************************************/

struct day_rec {
  KHE_TIME_GROUP	time_group;		/* tg for day                */
  ARRAY_KHE_TIME	times;			/* times of day              */
};


/*****************************************************************************/
/*                                                                           */
/*  Submodule "construction"                                                 */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  WEEK WeekMake(KHE_TIME_GROUP tg)                                         */
/*                                                                           */
/*  Make a week from tg; actually tg may be NULL.                            */
/*                                                                           */
/*****************************************************************************/

WEEK WeekMake(KHE_TIME_GROUP tg, HA_ARENA a)
{
  WEEK res;
  HaMake(res, a);
  res->time_group = tg;
  HaArrayInit(res->days, a);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  DAY DayMake(KHE_TIME_GROUP tg, HA_ARENA a)                               */
/*                                                                           */
/*  Make a day from tg.                                                      */
/*                                                                           */
/*****************************************************************************/

static DAY DayMake(KHE_TIME_GROUP tg, HA_ARENA a)
{
  DAY res;
  HnAssert(tg != NULL, "hseval internal error (DayMake)");
  HaMake(res, a);
  res->time_group = tg;
  HaArrayInit(res->times, a);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void WeeksAddDay(ARRAY_WEEK *weeks, DAY day)                             */
/*                                                                           */
/*  Add day to weeks.                                                        */
/*                                                                           */
/*****************************************************************************/

void WeeksAddDay(ARRAY_WEEK *weeks, DAY day)
{
  WEEK week;  int i;
  HaArrayForEach(*weeks, week, i)
    if( week->time_group == NULL ||
	KheTimeGroupSubset(day->time_group, week->time_group) )
    {
      HaArrayAddLast(week->days, day);
      return;
    }
}


/*****************************************************************************/
/*                                                                           */
/*  void WeeksAddTime(ARRAY_WEEK *weeks, KHE_TIME time)                      */
/*                                                                           */
/*  Add time to weeks.                                                       */
/*                                                                           */
/*****************************************************************************/

void WeeksAddTime(ARRAY_WEEK *weeks, KHE_TIME time)
{
  WEEK week;  DAY day;  int i, j, pos;
  HaArrayForEach(*weeks, week, i)
    HaArrayForEach(week->days, day, j)
      if( KheTimeGroupContains(day->time_group, time, &pos) )
      {
	HaArrayAddLast(day->times, time);
	return;
      }
}


/*****************************************************************************/
/*                                                                           */
/*  bool WeeksMakeSense(ARRAY_WEEK *weeks)                                   */
/*                                                                           */
/*  Return true if weeks is something you could build a timetable from.      */
/*                                                                           */
/*****************************************************************************/

bool WeeksMakeSense(ARRAY_WEEK *weeks)
{
  WEEK week;  DAY day;  int i, j;
  HaArrayForEach(*weeks, week, i)
  {
    if( HaArrayCount(week->days) == 0 )
      return false;
    HaArrayForEach(week->days, day, j)
      if( HaArrayCount(day->times) == 0 )
	return false;
  }
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "query"                                                        */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_TIME_GROUP WeekTimeGroup(WEEK week)                                  */
/*                                                                           */
/*  Return the time group associated with week, or NULL if none.             */
/*                                                                           */
/*****************************************************************************/

KHE_TIME_GROUP WeekTimeGroup(WEEK week)
{
  return week->time_group;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_TIME_GROUP DayTimeGroup(DAY day)                                     */
/*                                                                           */
/*  Return the time group associated with day; this is never NULL.           */
/*                                                                           */
/*****************************************************************************/

KHE_TIME_GROUP DayTimeGroup(DAY day)
{
  return day->time_group;
}


/*****************************************************************************/
/*                                                                           */
/*  int WeekDayCount(WEEK week)                                              */
/*                                                                           */
/*  Return the number of days in week.                                       */
/*                                                                           */
/*****************************************************************************/

int WeekDayCount(WEEK week)
{
  return HaArrayCount(week->days);
}


/*****************************************************************************/
/*                                                                           */
/*  DAY WeekDay(WEEK week, int day_index)                                    */
/*                                                                           */
/*  Return the day at this index in week.                                    */
/*                                                                           */
/*****************************************************************************/

DAY WeekDay(WEEK week, int day_index)
{
  return day_index < 0 || day_index >= HaArrayCount(week->days) ? NULL :
    HaArray(week->days, day_index);
}


/*****************************************************************************/
/*                                                                           */
/*  int WeekMaxTimesPerDayCount(WEEK week)                                   */
/*                                                                           */
/*  Return the maximum number of times on any day of week.                   */
/*                                                                           */
/*****************************************************************************/

int WeekMaxTimesPerDayCount(WEEK week)
{
  int i, res;  DAY day;
  res = 0;
  HaArrayForEach(week->days, day, i)
    if( HaArrayCount(day->times) > res )
      res = HaArrayCount(day->times);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_TIME WeekTime(WEEK week, int day_index, int time_index)              */
/*                                                                           */
/*  Return the time at position time_index on the day at position day_index, */
/*  or NULL if there is no such day or time.                                 */
/*                                                                           */
/*****************************************************************************/

KHE_TIME WeekTime(WEEK week, int day_index, int time_index)
{
  DAY day;
  if( HaArrayCount(week->days) <= day_index )
    return NULL;
  day = HaArray(week->days, day_index);
  if( HaArrayCount(day->times) <= time_index )
    return NULL;
  return HaArray(day->times, time_index);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "timetable skeleton"                                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool WeekTimetableSkeleton(KHE_INSTANCE ins, ARRAY_WEEK *weeks,          */
/*    HA_ARENA a)                                                            */
/*                                                                           */
/*  Return an array of weeks showing how to lay out a timetable for this tg. */
/*                                                                           */
/*****************************************************************************/

bool WeekTimetableSkeleton(KHE_INSTANCE ins, ARRAY_WEEK *weeks, HA_ARENA a)
{
  WEEK week;  DAY day;  KHE_TIME t;  KHE_TIME_GROUP tg;  int i;  bool res;
  if( DEBUG1 )
    fprintf(stderr, "[ WeekTimetableSkeleton\n");

  /* find the weeks; add a week with a NULL if there are no weeks */
  HaArrayInit(*weeks, a);
  for( i = 0;  i < KheInstanceTimeGroupCount(ins);  i++ )
  {
    tg = KheInstanceTimeGroup(ins, i);
    if( KheTimeGroupKind(tg) == KHE_TIME_GROUP_KIND_WEEK )
    {
      week = WeekMake(tg, a);
      HaArrayAddLast(*weeks, week);
    }
  }
  if( HaArrayCount(*weeks) == 0 )
  {
    week = WeekMake(NULL, a);
    HaArrayAddLast(*weeks, week);
  }

  /* find the days */
  for( i = 0;  i < KheInstanceTimeGroupCount(ins);  i++ )
  {
    tg = KheInstanceTimeGroup(ins, i);
    if( KheTimeGroupKind(tg) == KHE_TIME_GROUP_KIND_DAY )
    {
      day = DayMake(tg, a);
      WeeksAddDay(weeks, day);
    }
  }

  /* find the times */
  for( i = 0;  i < KheInstanceTimeCount(ins);  i++ )
  {
    t = KheInstanceTime(ins, i);
    WeeksAddTime(weeks, t);
  }
  res = WeeksMakeSense(weeks);
  if( DEBUG1 )
  {
    fprintf(stderr, "] WeekTimetableSkeleton returning %s:\n",
      res ? "true" : "false");
    WeeksDebug(weeks, 2, stderr);
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void WeeksDebug(ARRAY_WEEK *weeks, int indent, FILE *fp)                 */
/*                                                                           */
/*  Debug print of weeks.                                                    */
/*                                                                           */
/*****************************************************************************/

void WeeksDebug(ARRAY_WEEK *weeks, int indent, FILE *fp)
{
  WEEK week;  DAY day;  KHE_TIME time;  int i, j, k;
  fprintf(fp, "%*s[ Weeks\n", indent, "");
  HaArrayForEach(*weeks, week, i)
  {
    fprintf(fp, "%*s  [ Week %s\n", indent, "",
      week->time_group == NULL ? "?" : KheTimeGroupId(week->time_group));
    HaArrayForEach(week->days, day, j)
    {
      fprintf(fp, "%*s    [ Day %s\n", indent, "",
        day->time_group == NULL ? "?" : KheTimeGroupId(day->time_group));
      HaArrayForEach(day->times, time, k)
	fprintf(fp, "%*s        Time %s\n", indent, "", KheTimeName(time));
      fprintf(fp, "%*s    ]\n", indent, "");
    }
    fprintf(fp, "%*s  ]\n", indent, "");
  }
  fprintf(fp, "%*s]\n", indent, "");
}
