
/*****************************************************************************/
/*                                                                           */
/*  THE HSEVAL HIGH SCHOOL TIMETABLE EVALUATOR                               */
/*  COPYRIGHT (C) 2009, Jeffrey H. Kingston                                  */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 3, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         timetable.c                                                */
/*  MODULE:       Timetables                                                 */
/*                                                                           */
/*****************************************************************************/
#include "externs.h"
#include <float.h>
#include "space.h"

#define bool_show(x) ((x) ? "true" : "false")

#define DEBUG1 0	/* SolutionPlanningTimetablesHTML */
#define DEBUG2 0	/* EventGroupTimetablesHTML */
#define DEBUG3 0	/* LineDailyDisplay */
#define DEBUG4 0	/* ResourceTypeDailyPlanningTimetableHTML */
#define DEBUG5 0	/* soln_fn */
#define DEBUG6 0	/* LineDisplayTableEntry */
#define DEBUG7 0	/* TaskRunningOnDay */
#define DEBUG8 0	/* TaskEventResourceAsstIsDefective */


/*****************************************************************************/
/*                                                                           */
/*  Submodule "table entry background colour"                                */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheTableEntryBackgroundColourExplanation(HTML html)                 */
/*                                                                           */
/*  Explain how table entry background colours are calculated.               */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheTableEntryBackgroundColourExplanation(HTML html)
{
  HTMLText(html, "The background colour of each timetable cell is red if");
  HTMLText(html, "the resource is unavailable then (dark red for hard");
  HTMLText(html, "constraints, light red for soft ones), or else light green");
  HTMLText(html, "if the cell contains no events, or else the colour of the");
  HTMLText(html, "first event depicted (white if the event has no colour).");
  HTMLText(html, "If specific constraints are being highlighted, their times");
  HTMLText(html, "will appear in light blue, with defective times in blue.");
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool TaskEventAsstIsDefective(KHE_TASK task)                             */
/*                                                                           */
/*  Return true if the assignment of the event resource of task is           */
/*  defective.                                                               */
/*                                                                           */
/*****************************************************************************/

/* audited */
static bool TaskEventResourceAsstIsDefective(KHE_TASK task)
{
  KHE_EVENT_RESOURCE er;  KHE_SOLN soln;  KHE_MONITOR m;  int i;
  if( DEBUG8 )
  {
    fprintf(stderr, "[ TaskEventResourceAsstIsDefective(");
    KheTaskDebug(task, 1, -1, stderr);
    fprintf(stderr, ")\n");
  }
  soln = KheTaskSoln(task);
  er = KheTaskEventResource(task);
  for( i = 0;  i < KheSolnEventResourceMonitorCount(soln, er);  i++ )
  {
    m = KheSolnEventResourceMonitor(soln, er, i);
    if( KheMonitorCost(m) > 0 )
    {
      if( DEBUG8 )
      {
	fprintf(stderr, "] TaskEventResourceAsstIsDefective returning true:\n");
	KheMonitorDebug(m, 2, 2, stderr);
      }
      return true;
    }
  }
  if( DEBUG8 )
    fprintf(stderr, "] TaskEventResourceAsstIsDefective returning false\n");
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool TimeGroupIsUnavailable(KHE_TIME_GROUP tg, ARRAY_KHE_TIME *utimes)   */
/*                                                                           */
/*  Return true if tg is completely unavailable, according to *utimes.       */
/*                                                                           */
/*****************************************************************************/

/* audited */
static bool TimeGroupIsUnavailable(KHE_TIME_GROUP tg, ARRAY_KHE_TIME *utimes)
{
  KHE_TIME t;  int i, pos;
  for( i = 0;  i < KheTimeGroupTimeCount(tg);  i++ )
  {
    t = KheTimeGroupTime(tg, i);
    if( !HaArrayContains(*utimes, t, &pos) )
      return false;
  }
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheTableEntryBackgroundColour(KHE_TASK task, KHE_TIME_GROUP tg,    */
/*    ARRAY_KHE_TIME *hard_utimes, ARRAY_KHE_TIME *soft_utimes)              */
/*                                                                           */
/*  Return the background colour for a box representing the times of tg      */
/*  and containing task.                                                     */
/*                                                                           */
/*****************************************************************************/

/* audited */
static char *KheTableEntryBackgroundColour(KHE_TASK task, KHE_TIME_GROUP tg,
  ARRAY_KHE_TIME *hard_utimes, ARRAY_KHE_TIME *soft_utimes)
{
  KHE_EVENT e;  char *res;
  if( TimeGroupIsUnavailable(tg, hard_utimes) )
    return Red;
  else if( TimeGroupIsUnavailable(tg, soft_utimes) )
    return LightRed;
  else if( task == NULL )
    return LightGreen;
  else
  {
    e = KheMeetEvent(KheTaskMeet(task));
    res = KheEventColor(e);
    return (res != NULL ? res : White);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheTableEntrySpecialBackgroundColour(KHE_TASK task,                */
/*    KHE_TIME_GROUP tg, bool highlight_splits, int incident_count,          */
/*    ARRAY_KHE_TIME *hard_utimes, ARRAY_KHE_TIME *soft_utimes)              */
/*                                                                           */
/*  Return the background colour for a special case that requires all        */
/*  these parameters.  It would be good to simplify this some time.          */
/*                                                                           */
/*****************************************************************************/

/* audited */
static char *KheTableEntrySpecialBackgroundColour(KHE_TASK task,
  KHE_TIME_GROUP tg, bool highlight_splits, int incident_count,
  ARRAY_KHE_TIME *hard_utimes, ARRAY_KHE_TIME *soft_utimes)
{
  KHE_EVENT e;  char *res;
  if( TimeGroupIsUnavailable(tg, hard_utimes) )
    return Red;
  else if( TimeGroupIsUnavailable(tg, soft_utimes) )
    return LightRed;
  else if( task == NULL )
    return LightGreen;
  else
  {
    res = NULL;
    e = KheMeetEvent(KheTaskMeet(task));
    if( !highlight_splits || incident_count > 1 ||
	TaskEventResourceAsstIsDefective(task) )
      res = KheEventColor(e);
    /* ***
    else if( !KheTaskNeedsAssignment(task) )
      bgcolor = LightBlue;
    else
    *** */
    if( res == NULL )
      res = highlight_splits ? VeryLightGrey : White;
    return res;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheTableEntryDayBackgroundColour(KHE_TASK task, DAY day,           */
/*    ARRAY_KHE_TIME *hard_utimes, ARRAY_KHE_TIME *soft_utimes)              */
/*                                                                           */
/*  Return the background colour appropriate to task on day.                 */
/*                                                                           */
/*****************************************************************************/

/* audited */
static char *KheTableEntryDayBackgroundColour(KHE_TASK task, DAY day,
  ARRAY_KHE_TIME *hard_utimes, ARRAY_KHE_TIME *soft_utimes)
{
  return KheTableEntryBackgroundColour(task, DayTimeGroup(day),
    hard_utimes, soft_utimes);
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheTableEntrySpecialDayBackgroundColour(KHE_TASK task,             */
/*    DAY day, bool highlight_splits, int incident_count,                    */
/*    ARRAY_KHE_TIME *hard_utimes, ARRAY_KHE_TIME *soft_utimes)              */
/*                                                                           */
/*  Return the background colour for a special case that requires all        */
/*  these parameters.  It would be good to simplify this some time.          */
/*                                                                           */
/*****************************************************************************/

/* audited */
static char *KheTableEntrySpecialDayBackgroundColour(KHE_TASK task,
  DAY day, bool highlight_splits, int incident_count,
  ARRAY_KHE_TIME *hard_utimes, ARRAY_KHE_TIME *soft_utimes)
{
  return KheTableEntrySpecialBackgroundColour(task, DayTimeGroup(day),
    highlight_splits, incident_count, hard_utimes, soft_utimes);
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheTableEntryTimeBackgroundColour(KHE_TASK task, KHE_TIME time,    */
/*    ARRAY_KHE_TIME *hard_utimes, ARRAY_KHE_TIME *soft_utimes)              */
/*                                                                           */
/*  Return the background colour appropriate to task at time.                */
/*                                                                           */
/*****************************************************************************/

/* audited */
static char *KheTableEntryTimeBackgroundColour(KHE_TASK task, KHE_TIME time,
  ARRAY_KHE_TIME *hard_utimes, ARRAY_KHE_TIME *soft_utimes)
{
  return KheTableEntryBackgroundColour(task, KheTimeSingletonTimeGroup(time),
    hard_utimes, soft_utimes);
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheTableEntrySpecialTimeBackgroundColour(KHE_TASK task,            */
/*    KHE_TIME time, bool highlight_splits, int incident_count,              */
/*    ARRAY_KHE_TIME *hard_utimes, ARRAY_KHE_TIME *soft_utimes)              */
/*                                                                           */
/*  Return the background colour for a special case that requires all        */
/*  these parameters.  It would be good to simplify this some time.          */
/*                                                                           */
/*****************************************************************************/

/* audited */
static char *KheTableEntrySpecialTimeBackgroundColour(KHE_TASK task,
  KHE_TIME time, bool highlight_splits, int incident_count,
  ARRAY_KHE_TIME *hard_utimes, ARRAY_KHE_TIME *soft_utimes)
{
  return KheTableEntrySpecialBackgroundColour(task,
    KheTimeSingletonTimeGroup(time), highlight_splits,
    incident_count, hard_utimes, soft_utimes);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "table entry font"                                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheTableEntryFontExplanation(HTML html)                             */
/*                                                                           */
/*  Explain how table entry fonts are calculated.                            */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheTableEntryFontExplanation(HTML html)
{
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheTableEntryFontShowText(KHE_TASK task, char *buff, HTML html)     */
/*                                                                           */
/*  Show buff on html in a font that depends on task.                        */
/*                                                                           */
/*****************************************************************************/

/* audited */
static void KheTableEntryFontShowText(KHE_TASK task, char *buff, HTML html)
{
  if( TaskEventResourceAsstIsDefective(task) )
    HTMLTextBoldItalic(html, buff);
  else if( !KheTaskNeedsAssignment(task) )
    HTMLTextItalic(html, buff);
  else
    HTMLText(html, buff);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "individual timetables"                                        */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void AddUnavailableTimes(KHE_TIME_GROUP tg, bool reqd,                   */
/*    ARRAY_KHE_TIME *hard_utimes, ARRAY_KHE_TIME *soft_utimes)              */
/*                                                                           */
/*  Add the times of tg to *hard_utimes or *soft_utimes depending on reqd.   */
/*                                                                           */
/*****************************************************************************/

/* audited */
static void AddUnavailableTimes(KHE_TIME_GROUP tg, bool reqd,
  ARRAY_KHE_TIME *hard_utimes, ARRAY_KHE_TIME *soft_utimes)
{
  int i, pos;  KHE_TIME t;
  for( i = 0;  i < KheTimeGroupTimeCount(tg);  i++ )
  {
    t = KheTimeGroupTime(tg, i);
    if( reqd )
    {
      if( !HaArrayContains(*hard_utimes, t, &pos) )
	HaArrayAddLast(*hard_utimes, t);
    }
    else
    {
      if( !HaArrayContains(*soft_utimes, t, &pos) )
	HaArrayAddLast(*soft_utimes, t);
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void AvoidUnavailableTimesConstraintAddUnavailableTimes(                 */
/*    KHE_AVOID_UNAVAILABLE_TIMES_CONSTRAINT c,                              */
/*    ARRAY_KHE_TIME *hard_utimes, ARRAY_KHE_TIME *soft_utimes)              */
/*                                                                           */
/*  Add the times that c indicates to be unavailable to *hard_utimes or      */
/*  *soft_utimes depending on whether c is a hard or soft constraint.        */
/*                                                                           */
/*****************************************************************************/

/* audited */
static void AvoidUnavailableTimesConstraintAddUnavailableTimes(
  KHE_AVOID_UNAVAILABLE_TIMES_CONSTRAINT c,
  ARRAY_KHE_TIME *hard_utimes, ARRAY_KHE_TIME *soft_utimes)
{
  KHE_TIME_GROUP tg;
  tg = KheAvoidUnavailableTimesConstraintUnavailableTimes(c);
  AddUnavailableTimes(tg, KheConstraintRequired((KHE_CONSTRAINT) c),
    hard_utimes, soft_utimes);
}


/*****************************************************************************/
/*                                                                           */
/*  void ClusterBusyTimesConstraintAddUnavailableTimes(                      */
/*    KHE_CLUSTER_BUSY_TIMES_CONSTRAINT c,                                   */
/*    ARRAY_KHE_TIME *hard_utimes, ARRAY_KHE_TIME *soft_utimes)              */
/*                                                                           */
/*  Add the times that c indicates to be unavailable to *hard_utimes or      */
/*  *soft_utimes depending on whether c is a hard or soft constraint.        */
/*                                                                           */
/*****************************************************************************/

/* audited */
static void ClusterBusyTimesConstraintAddUnavailableTimes(
  KHE_CLUSTER_BUSY_TIMES_CONSTRAINT c,
  ARRAY_KHE_TIME *hard_utimes, ARRAY_KHE_TIME *soft_utimes)
{
  int i, j, offset;  KHE_TIME_GROUP tg;  KHE_POLARITY po;  bool reqd;
  if( KheClusterBusyTimesConstraintMaximum(c) == 0 )
  {
    reqd = KheConstraintRequired((KHE_CONSTRAINT) c);
    for( i = 0; i < KheClusterBusyTimesConstraintAppliesToOffsetCount(c); i++ )
    {
      offset = KheClusterBusyTimesConstraintAppliesToOffset(c, i);
      for( j = 0;  j < KheClusterBusyTimesConstraintTimeGroupCount(c);  j++ )
      {
	tg = KheClusterBusyTimesConstraintTimeGroup(c, j, offset, &po);
	if( po == KHE_POSITIVE )
	  AddUnavailableTimes(tg, reqd, hard_utimes, soft_utimes);
      }
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void LimitBusyTimesConstraintAddUnavailableTimes(                        */
/*    KHE_LIMIT_BUSY_TIMES_CONSTRAINT c,                                     */
/*    ARRAY_KHE_TIME *hard_utimes, ARRAY_KHE_TIME *soft_utimes)              */
/*                                                                           */
/*  Add the times that c indicates to be unavailable to *hard_utimes or      */
/*  *soft_utimes depending on whether c is a hard or soft constraint.        */
/*                                                                           */
/*****************************************************************************/

/* audited */
static void LimitBusyTimesConstraintAddUnavailableTimes(
  KHE_LIMIT_BUSY_TIMES_CONSTRAINT c,
  ARRAY_KHE_TIME *hard_utimes, ARRAY_KHE_TIME *soft_utimes)
{
  int i, j, offset;  KHE_TIME_GROUP tg;  bool reqd;
  if( KheLimitBusyTimesConstraintMaximum(c) == 0 )
  {
    reqd = KheConstraintRequired((KHE_CONSTRAINT) c);
    for( i = 0; i < KheLimitBusyTimesConstraintAppliesToOffsetCount(c); i++ )
    {
      offset = KheLimitBusyTimesConstraintAppliesToOffset(c, i);
      for( j = 0;  j < KheLimitBusyTimesConstraintTimeGroupCount(c);  j++ )
      {
	tg = KheLimitBusyTimesConstraintTimeGroup(c, j, offset);
	AddUnavailableTimes(tg, reqd, hard_utimes, soft_utimes);
      }
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void ResourceUnavailableTimes(KHE_RESOURCE r,                            */
/*    ARRAY_KHE_TIME *hard_utimes, ARRAY_KHE_TIME *soft_utimes)              */
/*                                                                           */
/*  Set *hard_utimes to the hard unavailable times of r, and *soft_utimes    */
/*  to the soft unavailable times of r.  The two arrays are assumed to be    */
/*  initialized; ResourceUnavailableTimes begins by clearing them.           */
/*                                                                           */
/*****************************************************************************/

/* audited */
static void ResourceUnavailableTimes(KHE_RESOURCE r,
  ARRAY_KHE_TIME *hard_utimes, ARRAY_KHE_TIME *soft_utimes)
{
  KHE_CONSTRAINT c;  int i;
  HaArrayClear(*hard_utimes);
  HaArrayClear(*soft_utimes);
  for( i = 0;  i < KheResourceConstraintCount(r);  i++ )
  {
    c = KheResourceConstraint(r, i);
    if( KheConstraintWeight(c) > 0 ) switch( KheConstraintTag(c) )
    {
      case KHE_AVOID_UNAVAILABLE_TIMES_CONSTRAINT_TAG:

        AvoidUnavailableTimesConstraintAddUnavailableTimes(
	  (KHE_AVOID_UNAVAILABLE_TIMES_CONSTRAINT) c, hard_utimes, soft_utimes);
	break;

      case KHE_CLUSTER_BUSY_TIMES_CONSTRAINT_TAG:

        ClusterBusyTimesConstraintAddUnavailableTimes(
          (KHE_CLUSTER_BUSY_TIMES_CONSTRAINT) c, hard_utimes, soft_utimes);
	break;

      case KHE_LIMIT_BUSY_TIMES_CONSTRAINT_TAG:

        LimitBusyTimesConstraintAddUnavailableTimes(
          (KHE_LIMIT_BUSY_TIMES_CONSTRAINT) c, hard_utimes, soft_utimes);
	break;

      default:

	/* nothing to do in these other cases */
	break;
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool ResourceTimetableHasAtMostOneTaskPerDay(                            */
/*    KHE_RESOURCE_TIMETABLE_MONITOR rtm, ARRAY_WEEK *weeks, HA_ARENA a)     */
/*                                                                           */
/*  Return true if rtm has at most one task per day of *weeks.               */
/*                                                                           */
/*****************************************************************************/

/* audited but still to do (if at all) */
static bool ResourceTimetableHasAtMostOneTaskPerDay(
  KHE_RESOURCE_TIMETABLE_MONITOR rtm, ARRAY_WEEK *weeks, HA_ARENA a)
{
  WEEK week;  DAY day;  KHE_TIME time;  int i, j, k, m, pos;  KHE_TIME_GROUP tg;
  ARRAY_KHE_TASK tasks;  KHE_TASK task;
  HaArrayInit(tasks, a);
  HaArrayForEach(*weeks, week, i)
    for( j = 0;  j < WeekDayCount(week);  j++ )
    {
      day = WeekDay(week, j);
      tg = DayTimeGroup(day);
      HaArrayClear(tasks);
      for( k = 0;  k < KheTimeGroupTimeCount(tg);  k++ )
      {
	time = KheTimeGroupTime(tg, k);
	for( m=0; m < KheResourceTimetableMonitorTimeTaskCount(rtm, time); m++ )
	{
	  task = KheResourceTimetableMonitorTimeTask(rtm, time, m);
	  if( !HaArrayContains(tasks, task, &pos) )
	    HaArrayAddLast(tasks, task);
	}
      }
      if( HaArrayCount(tasks) > 1 )
	return false;
    }
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  bool ResourceTimetableHasTaskOnDay(KHE_RESOURCE_TIMETABLE_MONITOR rtm,   */
/*    DAY day, KHE_TASK *task)                                               */
/*                                                                           */
/*  If rtm contains at least one task running on day, set *task to one such  */
/*  task and return true.  Otherwise set *task to NULL and return false.     */
/*                                                                           */
/*****************************************************************************/

/* ***
static bool ResourceTimetableHasTaskOnDay(KHE_RESOURCE_TIMETABLE_MONITOR rtm,
  DAY day, KHE_TASK *task)
{
  KHE_TIME time;  int i;  KHE_TIME_GROUP tg;
  tg = DayTimeGroup(day);
  for( i = 0;  i < KheTimeGroupTimeCount(tg);  i++ )
  {
    time = KheTimeGroupTime(tg, i);
    if( KheResourceTimetableMonitorTimeTaskCount(rtm, time) > 0 )
    {
      *task = KheResourceTimetableMonitorTimeTask(rtm, time, 0);
      return true;
    }
  }
  *task = NULL;
  return false;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_TASK ResourceTimetableTaskOnDay(KHE_RESOURCE_TIMETABLE_MONITOR rtm,  */
/*    DAY day)                                                               */
/*                                                                           */
/*  If rtm contains at least one task running on day, return one such task.  */
/*  Otherwise return NULL.                                                   */
/*                                                                           */
/*****************************************************************************/

/* audited */
static KHE_TASK ResourceTimetableTaskOnDay(KHE_RESOURCE_TIMETABLE_MONITOR rtm,
  DAY day)
{
  KHE_TIME time;  int i;  KHE_TIME_GROUP tg;
  tg = DayTimeGroup(day);
  for( i = 0;  i < KheTimeGroupTimeCount(tg);  i++ )
  {
    time = KheTimeGroupTime(tg, i);
    if( KheResourceTimetableMonitorTimeTaskCount(rtm, time) > 0 )
      return KheResourceTimetableMonitorTimeTask(rtm, time, 0);
  }
  return NULL;
}


/*****************************************************************************/
/*                                                                           */
/*  char *DayCvt(char *str)                                                  */
/*                                                                           */
/*  Do a quick and dirty conversion of the name of a specific day to the     */
/*  name of a generic day.                                                   */
/*                                                                           */
/*****************************************************************************/

/* audited */
static char *DayCvt(char *str)
{
  if( strstr(str, "Mon") != NULL )
    return "Mon";
  else if( strstr(str, "Tue") != NULL )
    return "Tue";
  else if( strstr(str, "Wed") != NULL )
    return "Wed";
  else if( strstr(str, "Thu") != NULL )
    return "Thu";
  else if( strstr(str, "Fri") != NULL )
    return "Fri";
  else if( strstr(str, "Sat") != NULL )
    return "Sat";
  else if( strstr(str, "Sun") != NULL )
    return "Sun";
  else
    return str;
}


/*****************************************************************************/
/*                                                                           */
/*  void ResourceTimetableHTMLOneForAllWeeks(KHE_SOLN soln, KHE_RESOURCE r,  */
/*    ARRAY_WEEK *weeks, HTML html, HA_ARENA a)                              */
/*                                                                           */
/*  Print the timetable of r in the form of one table containing one row     */
/*  per week.  Just the tables are printed, no headings or extra info.       */
/*                                                                           */
/*****************************************************************************/

/* audited */
static void ResourceTimetableHTMLOneForAllWeeks(KHE_SOLN soln, KHE_RESOURCE r,
  ARRAY_WEEK *weeks, HTML html, HA_ARENA a)
{
  WEEK week;  DAY day;  char *bgcolor;  int i, col, col_count;  KHE_TASK task;
  KHE_EVENT e;  KHE_RESOURCE_TIMETABLE_MONITOR rtm;
  ARRAY_KHE_TIME hard_utimes, soft_utimes;

  if( HaArrayCount(*weeks) == 0 )
  {
    HTMLParagraphBegin(html);
    HTMLText(html, "No weeks!");
    HTMLParagraphEnd(html);
    return;
  }

  /* get unavailable times */
  HaArrayInit(hard_utimes, a);
  HaArrayInit(soft_utimes, a);
  ResourceUnavailableTimes(r, &hard_utimes, &soft_utimes);

  HTMLParagraphBegin(html);
  HTMLTableBeginAttributed(html, 3, 0, 1, LightGreen);

  /* header row */
  HTMLTableRowVAlignBegin(html, "top");
  week = HaArrayFirst(*weeks);
  col_count = WeekDayCount(week);
  HTMLTableEntryBegin(html);
  HTMLHSpace(html, 2);
  HTMLTableEntryEnd(html);
  for( col = 0;  col < col_count;  col++ )
  {
    day = WeekDay(week, col);
    HTMLTableEntryTextBold(html, DayCvt(KheTimeGroupName(DayTimeGroup(day))));
  }
  HTMLTableRowEnd(html);

  /* other rows, one per week */
  rtm = KheResourceTimetableMonitor(soln, r);
  HaArrayForEach(*weeks, week, i)
  {
    HTMLTableRowVAlignBegin(html, "top");
    HTMLTableEntryBegin(html);
    HTMLTextBold(html, "Week %d", i + 1);
    HTMLTableEntryEnd(html);
    for( col = 0;  col < col_count;  col++ )
    {
      day = WeekDay(week, col);
      if( day == NULL )
      {
	HTMLTableEntryColouredBegin(html, Black);
	HTMLHSpace(html, 2);
	HTMLTableEntryEnd(html);
      }
      else
      {
        task = ResourceTimetableTaskOnDay(rtm, day);
        bgcolor = KheTableEntryDayBackgroundColour(task, day, &hard_utimes,
	  &soft_utimes);
	/* ***
	e = KheMeetEvent(KheTaskMeet(task));
        if( DayIsUnavailable(day, &hard_utimes) )
	  bgcolor = Red;
	else if( DayIsUnavailable(day, &soft_utimes) )
	  bgcolor = Ligh tRed;
	else
	{
	  bgcolor = KheEventColor(e);
	  if( bgcolor == NULL )
	    bgcolor = White;
	}
	*** */
	HTMLTableEntryColouredBegin(html, bgcolor);
	if( task != NULL )
	{
	  e = KheMeetEvent(KheTaskMeet(task));
	  KheTableEntryFontShowText(task, KheEventName(e), html);
	  /* HTMLText(html, KheEventName(e)); */
	}
	else
	  HTMLHSpace(html, 2);
	HTMLTableEntryEnd(html);
      }
      /* ***
      else
      {
        bgcolor = KheTableEntryDayBackgroundColour(NULL, day, &hard_utimes,
	  &soft_utimes, BrightGreen);
	** ***
        if( DayIsUnavailable(day, &hard_utimes) )
	  HTMLTableEntryColouredBegin(html, Red);
	else if( DayIsUnavailable(day, &soft_utimes) )
	  HTMLTableEntryColouredBegin(html, Ligh tRed);
	else
	  HTMLTableEntryBegin(html);
	*** **
	HTMLTableEntryColouredBegin(html, bgcolor);
	HTMLHSpace(html, 2);
	HTMLTableEntryEnd(html);
      }
      *** */
    }
    HTMLTableRowEnd(html);
  }

  HTMLTableEnd(html);
  HTMLParagraphEnd(html);
}


/*****************************************************************************/
/*                                                                           */
/*  void ResourceTimetableHTMLOnePerWeek(KHE_SOLN soln, KHE_RESOURCE r,      */
/*    ARRAY_WEEK *weeks, HTML html, HA_ARENA a)                              */
/*                                                                           */
/*  Print the timetable of r in the form of one table per week.              */
/*  Just the tables are printed, no headings or extra info.                 */
/*                                                                           */
/*****************************************************************************/

/* audited */
static void ResourceTimetableHTMLOnePerWeek(KHE_SOLN soln, KHE_RESOURCE r,
  ARRAY_WEEK *weeks, HTML html, HA_ARENA a)
{
  WEEK week;  DAY day;  KHE_TIME time;  char *bgcolor;  KHE_TASK task;
  int i, j, col, col_count, row, row_count;  KHE_EVENT e;
  KHE_RESOURCE_TIMETABLE_MONITOR rtm;  ARRAY_KHE_TIME hard_utimes, soft_utimes;

  /* get unavailable times */
  HaArrayInit(hard_utimes, a);
  HaArrayInit(soft_utimes, a);
  ResourceUnavailableTimes(r, &hard_utimes, &soft_utimes);

  /* timetables, one per week */
  rtm = KheResourceTimetableMonitor(soln, r);
  HaArrayForEach(*weeks, week, i)
  {
    HTMLParagraphBegin(html);
    HTMLTableBeginAttributed(html, 3, 0, 1, LightGreen);

    /* header row */
    HTMLTableRowVAlignBegin(html, "top");
    col_count = WeekDayCount(week);
    for( col = 0;  col < col_count;  col++ )
    {
      day = WeekDay(week, col);
      HTMLTableEntryTextBold(html, KheTimeGroupName(DayTimeGroup(day)));
    }
    HTMLTableRowEnd(html);

    /* other rows */
    row_count = WeekMaxTimesPerDayCount(week);
    for( row = 0;  row < row_count;  row++ )
    {
      HTMLTableRowVAlignBegin(html, "top");
      for( col = 0;  col < col_count;  col++ )
      {
	time = WeekTime(week, col, row);
	if( time == NULL )
	{
	  HTMLTableEntryBegin(html);
	  HTMLHSpace(html, 2);
	  HTMLTableEntryEnd(html);
	}
	else if( KheResourceTimetableMonitorTimeTaskCount(rtm, time) == 0 )
	{
	  bgcolor = KheTableEntryTimeBackgroundColour(NULL, time,
            &hard_utimes, &soft_utimes);
	  HTMLTableEntryColouredBegin(html, bgcolor);
	  HTMLHSpace(html, 2);
	  HTMLTableEntryEnd(html);
	}
	else
	{
	  task = KheResourceTimetableMonitorTimeTask(rtm, time, 0);
	  bgcolor = KheTableEntryTimeBackgroundColour(task, time,
            &hard_utimes, &soft_utimes);
	  HTMLTableEntryColouredBegin(html, bgcolor);
	  for( j=0; j < KheResourceTimetableMonitorTimeTaskCount(rtm, time);
	       j++ )
	  {
	    if( j > 0 )
	      HTMLNewLine(html);
	    task = KheResourceTimetableMonitorTimeTask(rtm, time, j);
	    e = KheMeetEvent(KheTaskMeet(task));
	    KheTableEntryFontShowText(task, KheEventName(e), html);
	  }
	  HTMLTableEntryEnd(html);
	}
      }
      HTMLTableRowEnd(html);
    }
    HTMLTableEnd(html);
    HTMLParagraphEnd(html);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void ResourceTimetableHTML(KHE_SOLN soln, KHE_RESOURCE r,                */
/*    KHE_FRAME frame, ARRAY_WEEK *weeks, HTML html, HA_ARENA a)             */
/*                                                                           */
/*  Print a timetable for r according to the pattern in weeks.               */
/*                                                                           */
/*****************************************************************************/

/* audited */
static void ResourceTimetableHTML(KHE_SOLN soln, KHE_RESOURCE r,
  /* KHE_FRAME frame, */ ARRAY_WEEK *weeks, HTML html, HA_ARENA a)
{
  bool header_printed, started;
  int i, j, points, pos, count, avail_times;  float avail_workload;
  KHE_MONITOR m;  KHE_RESOURCE_TIMETABLE_MONITOR rtm;
  ARRAY_KHE_MONITOR monitors;  KHE_EVENT_RESOURCE er;
  KHE_RESOURCE_GROUP rg;  KHE_COST cost;  KHE_INSTANCE ins;

  /* make sure the resource timetable monitor is attached */
  rtm = KheResourceTimetableMonitor(soln, r);
  if( !KheMonitorAttachedToSoln((KHE_MONITOR) rtm) )
    KheMonitorAttachToSoln((KHE_MONITOR) rtm);

  /* heading showing resource name, resource groups, and availability */
  HTMLParagraphBegin(html);
  HTMLTextBold(html, KheResourceName(r));
  KheResourceAvailableBusyTimes(soln, r, &avail_times);
  KheResourceAvailableWorkload(soln, r, &avail_workload);
  if( KheResourceResourceGroupCount(r) > 0 ||
      avail_times < INT_MAX || avail_workload < FLT_MAX )
  {
    HTMLTextNoBreak(html, "(");
    count = KheResourceResourceGroupCount(r);
    if( count > 5 )  count = 5;
    started = false;
    for( i = 0;  i < count;  i++ )
    {
      rg = KheResourceResourceGroup(r, i);
      if( started )
	HTMLTextNoBreak(html, ", ");
      HTMLTextNoBreak(html, KheResourceGroupName(rg));
      started = true;
    }
    if( count < KheResourceResourceGroupCount(r) )
      HTMLTextNoBreak(html, ", ...");
    if( avail_times < INT_MAX )
    {
      if( started )
	HTMLTextNoBreak(html, ", ");
      HTMLTextNoBreak(html, "Avail times %d", avail_times);
      started = true;
    }
    if( avail_workload < FLT_MAX )
    {
      if( started )
	HTMLTextNoBreak(html, ", ");
      HTMLTextNoBreak(html, "Avail workload %.1f", avail_workload);
      started = true;
    }
    HTMLText(html, ")");
  }
  HTMLParagraphEnd(html);

  /* the actual timetable */
  ins = KheSolnInstance(soln);
  if( KheInstanceModel(ins) == KHE_MODEL_EMPLOYEE_SCHEDULE &&
      ResourceTimetableHasAtMostOneTaskPerDay(rtm, weeks, a) )
    ResourceTimetableHTMLOneForAllWeeks(soln, r, weeks, html, a);
  else
    ResourceTimetableHTMLOnePerWeek(soln, r, weeks, html, a);

  /* gather constraint violations and print a table of them */
  HaArrayInit(monitors, a);
  for( i = 0;  i < KheResourceAssignedTaskCount(soln, r);  i++ )
  {
    er = KheTaskEventResource(KheResourceAssignedTask(soln, r, i));
    if( er != NULL )
      for( j = 0;  j < KheSolnEventResourceMonitorCount(soln, er);  j++ )
      {
	m = KheSolnEventResourceMonitor(soln, er, j);
	if( KheMonitorTag(m) != KHE_LIMIT_RESOURCES_MONITOR_TAG &&
	    KheMonitorCost(m) > 0 && !HaArrayContains(monitors, m, &pos) )
	  HaArrayAddLast(monitors, m);
      }
  }
  for( i = 0;  i < KheSolnResourceMonitorCount(soln, r);  i++ )
  {
    m = KheSolnResourceMonitor(soln, r, i);
    if( KheMonitorCost(m) > 0 && !HaArrayContains(monitors, m, &pos) )
      HaArrayAddLast(monitors, m);
  }
  MonitorsReportHTML(&monitors, false, html, a);

  /* lower bound table */
  header_printed = false;
  for( i = 0;  i < KheSolnResourceMonitorCount(soln, r);  i++ )
  {
    m = KheSolnResourceMonitor(soln, r, i);
    if( KheMonitorLowerBound(m) > 0 )
    {
      if( !header_printed )
      {
	HTMLParagraphBegin(html);
	HTMLTableBegin(html, LightRed);
	HTMLTableRowBegin(html);
	HTMLTableEntryTextBold(html, "Lower bounds");
	HTMLTableEntryTextBold(html, "Constraint name");
	HTMLTableEntryTextBold(html, "Inf.");
	HTMLTableEntryTextBold(html, "Obj.");
	HTMLTableRowEnd(html);
	header_printed = true;
      }
      MonitorLowerBoundReportHTML(m, false, &points, &cost, html);
    }
  }
  if( header_printed )
  {
    HTMLTableEnd(html);
    HTMLParagraphEnd(html);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void ResourceTypeTimetablesHTML(KHE_SOLN soln, KHE_RESOURCE_TYPE rt,     */
/*    ARRAY_WEEK *weeks, HTML html, HA_ARENA a)                              */
/*                                                                           */
/*  Print timetables for the resources of rt.                                */
/*                                                                           */
/*****************************************************************************/

/* audited */
static void ResourceTypeTimetablesHTML(KHE_SOLN soln, KHE_RESOURCE_TYPE rt,
  ARRAY_WEEK *weeks, HTML html, HA_ARENA a)
{
  KHE_RESOURCE r;  int i;  /* KHE_FRAME frame; */

  if( KheResourceTypeResourceCount(rt) > 0 )
  {
    /* heading */
    HTMLParagraphBegin(html);
    HTMLHeadingBegin(html);
    HTMLTextNoBreak(html, "Resource type ");
    HTMLText(html, KheResourceTypeName(rt));
    HTMLHeadingEnd(html);
    HTMLParagraphEnd(html);

    /* ***
    frame = KheFrameMakeCommon(soln);
    if( frame == NULL )
      frame = KheFrameMakeSingletons(soln);
    *** */

    /* a timetable for each resource */
    for( i = 0;  i < KheResourceTypeResourceCount(rt);  i++ )
    {
      r = KheResourceTypeResource(rt, i);
      ResourceTimetableHTML(soln, r, /* frame, */ weeks, html, a);
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool MeetRunningAtTime(KHE_MEET meet, KHE_TIME time)                     */
/*                                                                           */
/*  Return true if meet is running at time.                                  */
/*                                                                           */
/*****************************************************************************/

static bool MeetRunningAtTime(KHE_MEET meet, KHE_TIME time)
{
  KHE_TIME mt;
  mt = KheMeetAsstTime(meet);
  return mt != NULL && KheTimeIndex(mt) <= KheTimeIndex(time) &&
    KheTimeIndex(time) < KheTimeIndex(mt) + KheMeetDuration(meet);
}


/*****************************************************************************/
/*                                                                           */
/*  bool EventRunningAtTime(KHE_SOLN soln, KHE_EVENT e, KHE_TIME time)       */
/*                                                                           */
/*  Return true if e is running at time.                                     */
/*                                                                           */
/*****************************************************************************/

static bool EventRunningAtTime(KHE_SOLN soln, KHE_EVENT e, KHE_TIME time)
{
  int i;  KHE_MEET meet;
  for( i = 0;  i < KheEventMeetCount(soln, e);  i++ )
  {
    meet = KheEventMeet(soln, e, i);
    if( MeetRunningAtTime(meet, time) )
    {
      if( DEBUG2 )
	fprintf(stderr, "  event %s running at time %s\n",
	  KheEventName(e), KheTimeName(time));
      return true;
    }
  }
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  void EventDisplayAtTimeHTML(KHE_SOLN soln, KHE_EVENT e,                  */
/*    KHE_TIME time, HTML html)                                              */
/*                                                                           */
/*  Display e and its resource assignments at time onto html.                */
/*                                                                           */
/*****************************************************************************/

static void EventDisplayAtTimeHTML(KHE_SOLN soln, KHE_EVENT e,
  KHE_TIME time, HTML html)
{
  KHE_MEET meet;  int i, j;  KHE_TASK task;  KHE_RESOURCE r;  char *role;
  KHE_EVENT_RESOURCE er;
  for( i = 0;  i < KheEventMeetCount(soln, e);  i++ )
  {
    meet = KheEventMeet(soln, e, i);
    if( MeetRunningAtTime(meet, time) )
    {
      HTMLText(html, KheEventName(e));
      HTMLNewLine(html);
      for( j = 0;  j < KheEventResourceCount(e);  j++ )
      {
	er = KheEventResource(e, j);
	if( KheMeetFindTask(meet, er, &task) )
	{
	  HTMLHSpace(html, 2);
	  role = KheEventResourceRole(KheTaskEventResource(task));
	  if( role != NULL )
	  {
	    HTMLTextNoBreak(html, role);
	    HTMLTextNoBreak(html, ": ");
	  }
	  r = KheTaskAsstResource(task);
	  HTMLText(html, r != NULL ? KheResourceName(r) : "-");
	  HTMLNewLine(html);
	}
      }
    }
  }
}

/* *** old version which can reorder tasks so that assigned ones come first
static void EventDisplayAtTimeHTML(KHE_SOLN soln, KHE_EVENT e,
  KHE_TIME time, HTML html)
{
  KHE_MEET meet;  int i, j;  KHE_TASK task;  KHE_RESOURCE r;  char *role;
  for( i = 0;  i < KheEventMeetCount(soln, e);  i++ )
  {
    meet = KheEventMeet(soln, e, i);
    if( MeetRunningAtTime(meet, time) )
    {
      HTMLText(html, KheEventName(e));
      HTMLNewLine(html);
      for( j = 0;  j < KheMeetTaskCount(meet);  j++ )
      {
	task = KheMeetTask(meet, j);
	HTMLHSpace(html, 2);
	role = KheEventResourceRole(KheTaskEventResource(task));
	if( role != NULL )
	{
	  HTMLTextNoBreak(html, role);
	  HTMLTextNoBreak(html, ": ");
	}
	r = KheTaskAsstResource(task);
	HTMLText(html, r != NULL ? KheResourceName(r) : "-");
	HTMLNewLine(html);
      }
    }
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void EventGroupTimetablesHTML(KHE_SOLN soln, KHE_EVENT_GROUP eg,         */
/*    ARRAY_WEEK *weeks, HTML html, HA_ARENA a)                              */
/*                                                                           */
/*  Print timetables and reports for eg following weeks format onto html.    */
/*                                                                           */
/*****************************************************************************/

static void EventGroupTimetablesHTML(KHE_SOLN soln, KHE_EVENT_GROUP eg,
  ARRAY_WEEK *weeks, HTML html, HA_ARENA a)
{
  WEEK week;  DAY day;  KHE_TIME time;  bool header_printed, first;
  int i, j, k, col, col_count, row, row_count, points, pos;  KHE_COST cost;
  KHE_EVENT e;  char *bgcolor;  ARRAY_KHE_MONITOR monitors;  KHE_MONITOR m;
  KHE_EVENT_RESOURCE er;  bool has_monitors;

  if( DEBUG2 )
    fprintf(stderr, "[ EventGroupTimetablesHTML(soln, %s, weeks, html)\n",
      KheEventGroupName(eg));

  /* find the unique monitors and decide whether there is anything to do */
  has_monitors = false;
  HaArrayInit(monitors, a);
  for( i = 0;  i < KheEventGroupEventCount(eg);  i++ )
  {
    e = KheEventGroupEvent(eg, i);
    for( j = 0;  j < KheSolnEventMonitorCount(soln, e);  j++ )
    {
      m = KheSolnEventMonitor(soln, e, j);
      has_monitors = true;
      if( KheMonitorCost(m) > 0 && !HaArrayContains(monitors, m, &pos) )
	HaArrayAddLast(monitors, m);
    }
    for( j = 0;  j < KheEventResourceCount(e);  j++ )
    {
      er = KheEventResource(e, j);
      for( k = 0;  k < KheSolnEventResourceMonitorCount(soln, er);  k++ )
      {
	m = KheSolnEventResourceMonitor(soln, er, k);
	has_monitors = true;
	if( KheMonitorCost(m) > 0 && !HaArrayContains(monitors, m, &pos) )
	  HaArrayAddLast(monitors, m);
      }
    }
  }

  if( has_monitors )
  {
    /* heading showing event group name */
    HTMLParagraphBegin(html);
    HTMLTextBold(html, KheEventGroupName(eg));
    /* HTMLText(html, " (%d weeks)", HaArrayCount(*weeks)); */
    HTMLParagraphEnd(html);

    /* timetables, one per week */
    HaArrayForEach(*weeks, week, i)
    {
      HTMLParagraphBegin(html);
      HTMLTableBeginAttributed(html, 3, 0, 1, LightGreen);

      /* header row */
      HTMLTableRowVAlignBegin(html, "top");
      col_count = WeekDayCount(week);
      for( col = 0;  col < col_count;  col++ )
      {
	day = WeekDay(week, col);
	HTMLTableEntryTextBold(html, KheTimeGroupName(DayTimeGroup(day)));
      }
      HTMLTableRowEnd(html);

      /* other rows */
      row_count = WeekMaxTimesPerDayCount(week);
      for( row = 0;  row < row_count;  row++ )
      {
	HTMLTableRowVAlignBegin(html, "top");
	for( col = 0;  col < col_count;  col++ )
	{
	  time = WeekTime(week, col, row);
	  if( time != NULL )
	  {
	    first = true;
	    for( j = 0;  j < KheEventGroupEventCount(eg);  j++ )
	    {
	      e = KheEventGroupEvent(eg, j);
	      if( EventRunningAtTime(soln, e, time) )
	      {
		if( first )
		{
		  bgcolor = KheEventColor(e);
		  if( bgcolor == NULL )
		    bgcolor = White;
		  HTMLTableEntryColouredBegin(html, bgcolor);
		}
		else
		  HTMLNewLine(html);
		first = false;
		EventDisplayAtTimeHTML(soln, e, time, html);
	      }
	    }
	    if( first )
	    {
	      HTMLTableEntryBegin(html);
	      HTMLHSpace(html, 2);
	      HTMLTableEntryEnd(html);
	    }
	  }
	  else
	  {
	    HTMLTableEntryBegin(html);
	    HTMLHSpace(html, 2);
	    HTMLTableEntryEnd(html);
	  }
	}
	HTMLTableRowEnd(html);
      }
      HTMLTableEnd(html);
      HTMLParagraphEnd(html);
    }

    /* constraint violation table */
    MonitorsReportHTML(&monitors, false, html, a);

    /* lower bound table */
    header_printed = false;
    HaArrayForEach(monitors, m, i)
      if( KheMonitorLowerBound(m) > 0 )
      {
	if( !header_printed )
	{
	  HTMLParagraphBegin(html);
	  HTMLTableBegin(html, LightRed);
	  HTMLTableRowBegin(html);
	  HTMLTableEntryTextBold(html, "Lower bounds");
	  HTMLTableEntryTextBold(html, "Constraint name");
	  HTMLTableEntryTextBold(html, "Inf.");
	  HTMLTableEntryTextBold(html, "Obj.");
	  HTMLTableRowEnd(html);
	  header_printed = true;
	}
	MonitorLowerBoundReportHTML(m, false, &points, &cost, html);
      }
    if( header_printed )
    {
      HTMLTableEnd(html);
      HTMLParagraphEnd(html);
    }
  }
  /* MArrayFree(monitors); */
  if( DEBUG2 )
    fprintf(stderr, "] EventGroupTimetablesHTML returning\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void SolnHeader(KHE_SOLN soln, HTML html)                                */
/*                                                                           */
/*  Print a header paragraph (or two) for printing soln.                     */
/*                                                                           */
/*****************************************************************************/

/* audited */
static void SolnHeader(KHE_SOLN soln, HTML html)
{
  KHE_INSTANCE ins;
  ins = KheSolnInstance(soln);
  HTMLParagraphBegin(html);
  HTMLHeadingBegin(html);
  HTMLTextNoBreak(html, "Solution of instance ");
  HTMLLiteralText(html, KheInstanceId(ins));
  if( KheSolnDescription(soln) != NULL )
    HTMLText(html, " (cost % .5f, %s)", KheCostShow(KheSolnCost(soln)),
       KheSolnDescription(soln));
  else
    HTMLText(html, " (cost % .5f)", KheCostShow(KheSolnCost(soln)));
  HTMLHeadingEnd(html);
  HTMLParagraphEnd(html);
}


/*****************************************************************************/
/*                                                                           */
/*  void SolutionTimetablesHTML(KHE_SOLN soln, bool with_event_groups,       */
/*    HTML html)                                                             */
/*                                                                           */
/*  Print timetables for soln onto html.                                     */
/*                                                                           */
/*****************************************************************************/

static void SolutionTimetablesHTML(KHE_SOLN soln, bool with_event_groups,
  ARRAY_WEEK *weeks, HTML html, HA_ARENA a)
{
  KHE_RESOURCE_TYPE rt;  /* ARRAY_WEEK weeks; */ KHE_INSTANCE ins;  int i;
  KHE_EVENT_GROUP eg;

  ins = KheSolnInstance(soln);
  for( i = 0;  i < KheInstanceResourceTypeCount(ins);  i++ )
  {
    rt = KheInstanceResourceType(ins, i);
    ResourceTypeTimetablesHTML(soln, rt, weeks, html, a);
  }
  if( with_event_groups )
    for( i = 0;  i < KheInstanceEventGroupCount(ins);  i++ )
    {
      eg = KheInstanceEventGroup(ins, i);
      EventGroupTimetablesHTML(soln, eg, weeks, html, a);
    }

  /* header */
  /* ***
  ins = KheSolnInstance(soln);
  SolnHeader(soln, html);

  ** resources, or nothing if no skeleton **
  if( KheSolnType(soln) == KHE_SOLN_INVALID_PLACEHOLDER )
    SolnInvalidParagraph(soln, html);
  else if( WeekTimetableSkeleton(ins, &weeks, a) )
  {
  }
  else
  {
    HTMLParagraphBegin(html);
    HTMLText(html, "HSEval cannot print timetables for this instance");
    HTMLText(html, "because it cannot find, or cannot make sense of,");
    HTMLText(html, "its Days (and Weeks if present).");
    HTMLParagraphEnd(html);
  }
  *** */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSolnGroupHeader(KHE_SOLN_GROUP soln_group, HTML html)            */
/*                                                                           */
/*  Print a header and a metadata paragraph for soln_group.                  */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheSolnGroupHeader(KHE_SOLN_GROUP soln_group, HTML html)
{
  char buff[1000];
  sprintf(buff, "Solution Group %s", KheSolnGroupId(soln_group));
  HTMLHeading(html, buff);
  HTMLParagraphBegin(html);
  HTMLText(html, KheSolnGroupMetaDataText(soln_group));
  HTMLParagraphEnd(html);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void SolutionGroupTimetablesHTML(KHE_SOLN_GROUP soln_group,              */
/*     bool with_event_groups, HTML html, HA_ARENA_SET as)                   */
/*                                                                           */
/*  Print timetables for soln_group to html.                                 */
/*                                                                           */
/*****************************************************************************/

/* ***
void SolutionGroupTimetablesHTML(KHE_SOLN_GROUP soln_group,
   bool with_event_groups, HTML html, HA_ARENA_SET as)
{
  KHE_SOLN soln;  int i;
  KheSolnGroupHeader(soln_group, html);
  for( i = 0;  i < KheSolnGroupSolnCount(soln_group);  i++ )
  {
    soln = KheSolnGroupSoln(soln_group, i);
    SolutionTimetablesHTML(soln, with_event_groups, html, as);
    KheSolnTypeReduce(soln, KHE_SOLN_BASIC_PLACEHOLDER, NULL);
    ** KheSolnReduceToPlaceholder(soln, false); **
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void ArchiveTimetablesHTML(KHE_ARCHIVE archive, bool with_event_groups,  */
/*    HTML html)                                                             */
/*                                                                           */
/*  Print timetables for each solution in archive.                           */
/*                                                                           */
/*****************************************************************************/

/* ***
void ArchiveTimetablesHTML(KHE_ARCHIVE archive, bool with_event_groups,
  HTML html, HA_ARENA_SET as)
{
  KHE_SOLN_GROUP soln_group;  int i;

  ** general intro paragraph **
  HTMLParagraphBegin(html);
  HTMLText(html, "For each solution in the uploaded XML archive, this page");
  HTMLText(html, "contains a timetable for each resource of the");
  HTMLText(html, "corresponding instance.  If any of the constraints");
  HTMLText(html, "that apply to that resource are violated, a table of");
  HTMLText(html, "those violations is shown below the timetable.");
  HTMLParagraphEnd(html);

  ** with event groups intro paragraph **
  if( with_event_groups )
  {
    HTMLParagraphBegin(html);
    HTMLText(html, "Similarly, a timetable is shown for each event group of");
    HTMLText(html, "the instance to which at least one constraint applies;");
    HTMLText(html, "and if any of those constraints are violated, a table");
    HTMLText(html, "of those violations is shown.");
    HTMLParagraphEnd(html);
  }

  ** display format intro paragraph **
  HTMLParagraphBegin(html);
  KheTableEntryBackgroundColourExplanation(html);
  KheTableEntryFontExplanation(html);
  HTMLParagraphEnd(html);

  ** one page segment for each solution group **
  if( KheArchiveSolnGroupCount(archive) == 0 )
  {
    HTMLParagraphBegin(html);
    HTMLText(html, "The uploaded XML file contains no solution groups.");
    HTMLParagraphEnd(html);
  }
  else
  {
    for( i = 0;  i < KheArchiveSolnGroupCount(archive);  i++ )
    {
      soln_group = KheArchiveSolnGroup(archive, i);
      HTMLHorizontalRule(html);
      SolutionGroupTimetablesHTML(soln_group, with_event_groups, html, as);
    }
    HTMLHorizontalRule(html);
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  Submodule "planning timetables - one column per time"                    */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool TaskRunningAtTime(KHE_TASK task, KHE_TIME t)                        */
/*                                                                           */
/*  Return true if task is running at time t.                                */
/*                                                                           */
/*****************************************************************************/

static bool TaskRunningAtTime(KHE_TASK task, KHE_TIME t)
{
  KHE_MEET meet;  KHE_TIME mt;
  meet = KheTaskMeet(task);
  mt = KheMeetAsstTime(meet);
  if( mt == NULL )
    return false;
  else return KheTimeIndex(mt) <= KheTimeIndex(t) &&
    KheTimeIndex(mt) + KheMeetDuration(meet) > KheTimeIndex(t);
}


/*****************************************************************************/
/*                                                                           */
/*  bool LineIsConstant(ARRAY_KHE_TASK *line, ARRAY_KHE_TIME *hard_utimes,   */
/*    ARRAY_KHE_TIME *soft_utimes, KHE_TIME time1, KHE_TIME time2,           */
/*    ARRAY_KHE_EVENT *time1_events, ARRAY_KHE_EVENT *time2_events)          */
/*                                                                           */
/*  Return true if line, *hard_utimes, and *soft_utimes are constant between */
/*  these two times; *time1_events and *time2_events are scratch arrays.     */
/*                                                                           */
/*****************************************************************************/

static bool LineIsConstant(ARRAY_KHE_TASK *line, ARRAY_KHE_TIME *hard_utimes,
  ARRAY_KHE_TIME *soft_utimes, KHE_TIME time1, KHE_TIME time2,
  ARRAY_KHE_EVENT *time1_events, ARRAY_KHE_EVENT *time2_events)
{
  KHE_TASK task;  KHE_EVENT e;  int i, pos, pos2;

  /* find the set of distinct events running at time1 */
  HaArrayClear(*time1_events);
  HaArrayForEach(*line, task, i)
  {
    e = KheMeetEvent(KheTaskMeet(task));
    if( TaskRunningAtTime(task, time1) &&
	!HaArrayContains(*time1_events, e, &pos) )
      HaArrayAddLast(*time1_events, e);
  }

  /* find the set of distinct events running at time2 */
  HaArrayClear(*time2_events);
  HaArrayForEach(*line, task, i)
  {
    e = KheMeetEvent(KheTaskMeet(task));
    if( TaskRunningAtTime(task, time2) &&
	!HaArrayContains(*time2_events, e, &pos) )
      HaArrayAddLast(*time2_events, e);
  }

  /* if the two arrays differ, line is not constant */
  if( HaArrayCount(*time1_events) != HaArrayCount(*time2_events) )
    return false;
  HaArrayForEach(*time1_events, e, i)
    if( !HaArrayContains(*time2_events, e, &pos) )
      return false;

  /* check for constancy of hard_utimes */
  if( HaArrayContains(*hard_utimes, time1, &pos) !=
      HaArrayContains(*hard_utimes, time2, &pos2) )
    return false;

  /* check for constancy of soft_utimes */
  return HaArrayContains(*soft_utimes, time1, &pos) ==
    HaArrayContains(*soft_utimes, time2, &pos2);
}


/*****************************************************************************/
/*                                                                           */
/*  void LineDisplayTableEntry(HTML html, ARRAY_KHE_TASK *line,              */
/*    ARRAY_KHE_TIME *hard_utimes, ARRAY_KHE_TIME *soft_utimes,              */
/*    KHE_TIME time, bool highlight_splits, int count)                       */
/*                                                                           */
/*  Print the table entry for line, *hard_utimes, and *soft_utimes at time.  */
/*                                                                           */
/*****************************************************************************/

/* *** replaced by TasksDisplayTableEntry
static void LineDisplayTableEntry(HTML html, ARRAY_KHE_TASK *line,
  ARRAY_KHE_TIME *hard_utimes, ARRAY_KHE_TIME *soft_utimes,
  KHE_TIME time, bool highlight_splits, int count)
{
  char buff[200];  KHE_TASK task;  bool first;  KHE_EVENT e;
  int i;  char *bgcolor;  int incident_count;
  if( DEBUG6 )
    fprintf(stderr, "[ LineDisplayTableEntry(line %d, utimes %d, %s, %s, %d)\n",
      HaArrayCount(*line),
      HaArrayCount(*hard_utimes) + HaArrayCount(*soft_utimes) , KheTimeId(time),
      highlight_splits ? "true" : "false", count);
  HnAssert(count >= 1, "LineDisplayTableEntry internal error 1");
  HaArrayForEach(*line, task, i)
    if( TaskRunningAtTime(task, time) )
      incident_count++;
  HaArrayForEach(*line, task, i)
    if( TaskRunningAtTime(task, time) )
    {
      e = KheMeetEvent(KheTaskMeet(task));
      snprintf(buff, 1 + count * 4, "%s", KheEventName(e));
      if( first )
      {
	bgcolor = KheTableEntrySpecialTimeBackgroundColour(task, time,
	  highlight_splits, incident_count, hard_utimes, soft_utimes);
	** ***
	if( bgcolor == NULL && (!highlight_splits || incident_count > 1 ||
	    TaskEventResourceAsstIsDefective(task)) )
	  bgcolor = KheEventColor(e);
	if( bgcolor == NULL )
	  bgcolor = highlight_splits ? LightGrey : White;
	*** **
	HTMLTableEntrySpanBegin(html, count, bgcolor);
	first = false;
      }
      else
	HTMLNewLine(html);
      KheTableEntryFontShowText(task, buff, html);
      ** ***
      if( TaskEventResourceAsstIsDefective(task) )
	HTMLTextIt alic(html, buff);
      else if( !KheTaskNeedsAssignment(task) )
	HTMLTextMono(html, buff);
      else
	HTMLText(html, buff);
      *** **
    }
  if( first )
  {
    bgcolor = KheTableEntrySpecialTimeBackgroundColour(NULL, NULL, false, 0,
      hard_utimes, soft_utimes);
    HTMLTableEntrySpanBegin(html, count, bgcolor);
    HTMLHSpace(html, 2);
  }
  HTMLTableEntryEnd(html);
  if( DEBUG6 )
    fprintf(stderr, "] LineDisplayTableEntry\n");
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void TasksDisplayTableEntry(ARRAY_KHE_TASK *tasks,                       */
/*    ARRAY_KHE_TIME *hard_utimes, ARRAY_KHE_TIME *soft_utimes,              */
/*    KHE_TIME time, bool highlight_splits, int count, HTML html)            */
/*                                                                           */
/*  Print one table entry showing *tasks, *hard_utimes, and *soft_utimes.    */
/*                                                                           */
/*****************************************************************************/

static void TasksDisplayTableEntry(ARRAY_KHE_TASK *tasks,
  ARRAY_KHE_TIME *hard_utimes, ARRAY_KHE_TIME *soft_utimes,
  KHE_TIME time, bool highlight_splits, int count, HTML html)
{
  char buff[200];  KHE_TASK task;  KHE_EVENT e;  int i;  char *bgcolor;
  if( DEBUG6 )
    fprintf(stderr,
      "[ TasksDisplayTableEntry(tasks %d, utimes %d, %s, %s, %d)\n",
      HaArrayCount(*tasks),
      HaArrayCount(*hard_utimes) + HaArrayCount(*soft_utimes) , KheTimeId(time),
      highlight_splits ? "true" : "false", count);
  HnAssert(count >= 1, "LineDisplayTableEntry internal error 1");
  if( HaArrayCount(*tasks) > 0 )
  {
    /* at least one task, get background colour based on first and print */
    bgcolor = KheTableEntrySpecialTimeBackgroundColour(HaArrayFirst(*tasks),
      time, highlight_splits, HaArrayCount(*tasks), hard_utimes, soft_utimes);
    HTMLTableEntrySpanBegin(html, count, bgcolor);
    HaArrayForEach(*tasks, task, i)
    {
      if( i > 0 )
	HTMLNewLine(html);
      e = KheMeetEvent(KheTaskMeet(task));
      snprintf(buff, 1 + count * 4, "%s", KheEventName(e));
      KheTableEntryFontShowText(task, buff, html);
    }
    HTMLTableEntryEnd(html);
  }
  else
  {
    /* no tasks, print an empty box */
    bgcolor = KheTableEntrySpecialTimeBackgroundColour(NULL, NULL, false, 0,
      hard_utimes, soft_utimes);
    HTMLTableEntrySpanBegin(html, count, bgcolor);
    HTMLHSpace(html, 2);
    HTMLTableEntryEnd(html);
  }
  if( DEBUG6 )
    fprintf(stderr, "] LineDisplayTableEntry\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void LineSelectTasks(ARRAY_KHE_TASK *line, KHE_TIME time,                */
/*    ARRAY_KHE_TASK *tasks)                                                 */
/*                                                                           */
/*  Set *tasks to the tasks from *line that are running at time.             */
/*                                                                           */
/*****************************************************************************/

static void LineSelectTasks(ARRAY_KHE_TASK *line, KHE_TIME time,
  ARRAY_KHE_TASK *tasks)
{
  KHE_TASK task;  int i;
  HaArrayClear(*tasks);
  HaArrayForEach(*line, task, i)
    if( TaskRunningAtTime(task, time) )
      HaArrayAddLast(*tasks, task);
}


/*****************************************************************************/
/*                                                                           */
/*  void LineDisplay(HTML html, ARRAY_KHE_TASK *line,                        */
/*    ARRAY_KHE_TIME *hard_utimes, ARRAY_KHE_TIME *soft_utimes,              */
/*    ARRAY_KHE_TIME *permuted_times, bool highlight_splits, HA_ARENA a)     */
/*                                                                           */
/*  Display one table entry for each time of permuted_times, showing the     */
/*  events of line running at that time.  Parameters *hard_utimes and        */
/*  *soft_utimes contain sets of unavailable times, to be shown as           */
/*  background colour.  Adjacent table entries with the same events and      */
/*  unavailability are grouped into spanning entries.                        */
/*                                                                           */
/*****************************************************************************/

static void LineDisplay(HTML html, ARRAY_KHE_TASK *line,
  ARRAY_KHE_TIME *hard_utimes, ARRAY_KHE_TIME *soft_utimes,
  ARRAY_KHE_TIME *permuted_times, bool highlight_splits, HA_ARENA a)
{
  int i, count;  KHE_TIME prev_t, t;  ARRAY_KHE_EVENT e1, e2;
  ARRAY_KHE_TASK tasks;
  HaArrayInit(e1, a);
  HaArrayInit(e2, a);
  HaArrayInit(tasks, a);
  count = 0;  prev_t = NULL;
  HaArrayForEach(*permuted_times, t, i)
  {
    if( prev_t == NULL || LineIsConstant(line, hard_utimes, soft_utimes,
	prev_t, t, &e1, &e2) )
      count++;
    else
    {
      LineSelectTasks(line, prev_t, &tasks);
      TasksDisplayTableEntry(&tasks, hard_utimes, soft_utimes, prev_t,
	highlight_splits, count, html);
      count = 1;
    }
    prev_t = t;
  }
  if( count > 0 )
  {
    LineSelectTasks(line, prev_t, &tasks);
    TasksDisplayTableEntry(&tasks, hard_utimes, soft_utimes, prev_t,
      highlight_splits, count, html);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "planning timetables - one column per day"                     */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool TaskRunningOnDay(KHE_TASK task, DAY day)                            */
/*                                                                           */
/*  Return true if task is running on day.                                   */
/*                                                                           */
/*****************************************************************************/

static bool TaskRunningOnDay(KHE_TASK task, DAY day)
{
  KHE_MEET meet;  KHE_TIME mt;  bool res;
  if( DEBUG7 )
    fprintf(stderr, "[ TaskRunningOnDay(task, day)\n");
  meet = KheTaskMeet(task);
  mt = KheMeetAsstTime(meet);
  if( mt == NULL )
    res = false;
  else res =
    KheTimeGroupOverlap(DayTimeGroup(day), mt, KheMeetDuration(meet)) > 0;
  if( DEBUG7 )
    fprintf(stderr, "] TaskRunningOnDay(task, day) returning %s\n",
      res ? "true" : "false");
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void LineSelectDayTasks(ARRAY_KHE_TASK *line, DAY day,                   */
/*    ARRAY_KHE_TASK *tasks)                                                 */
/*                                                                           */
/*  Set *tasks to the tasks from *line that are running on day.              */
/*                                                                           */
/*****************************************************************************/

static void LineSelectDayTasks(ARRAY_KHE_TASK *line, DAY day,
  ARRAY_KHE_TASK *tasks)
{
  KHE_TASK task;  int i;
  HaArrayClear(*tasks);
  HaArrayForEach(*line, task, i)
    if( TaskRunningOnDay(task, day) )
      HaArrayAddLast(*tasks, task);
}


/*****************************************************************************/
/*                                                                           */
/*  void TasksDailyDisplayTableEntry(ARRAY_KHE_TASK *tasks,                  */
/*    ARRAY_KHE_TIME *hard_utimes, ARRAY_KHE_TIME *soft_utimes, DAY day,     */
/*    bool highlight_splits, HTML html)                                      */
/*                                                                           */
/*  Print one table entry showing *tasks, *hard_utimes, and *soft_utimes.    */
/*                                                                           */
/*****************************************************************************/

static void TasksDailyDisplayTableEntry(ARRAY_KHE_TASK *tasks,
  ARRAY_KHE_TIME *hard_utimes, ARRAY_KHE_TIME *soft_utimes, DAY day,
  bool highlight_splits, HTML html)
{
  char buff[200];  KHE_TASK task;  KHE_EVENT e;
  int i;  char *bgcolor, *s, *tg_name;
  if( DEBUG3 )
    fprintf(stderr, "[ TasksDailyDisplayTableEntry (%d tasks)\n",
      HaArrayCount(*tasks));

  if( HaArrayCount(*tasks) > 0 )
  {
    /* at least one task, get background colour based on first and print */
    tg_name = KheTimeGroupName(DayTimeGroup(day));
    bgcolor = KheTableEntrySpecialDayBackgroundColour(HaArrayFirst(*tasks),
      day, highlight_splits, HaArrayCount(*tasks), hard_utimes, soft_utimes);
    HTMLTableEntrySpanBegin(html, 1, bgcolor);
    HaArrayForEach(*tasks, task, i)
    {
      if( i > 0 )
	HTMLNewLine(html);
      e = KheMeetEvent(KheTaskMeet(task));
      s = strstr(KheEventName(e), tg_name);
      if( s == KheEventName(e) )
      {
	s += strlen(tg_name);
	if( *s == ':' )
	  s++;
	snprintf(buff, 10, "%s", s);
      }
      else
	snprintf(buff, 10, "%s", KheEventName(e));
      KheTableEntryFontShowText(task, buff, html);
    }
    HTMLTableEntryEnd(html);
  }
  else
  {
    /* no tasks, print an empty box */
    bgcolor = KheTableEntrySpecialDayBackgroundColour(NULL, day, false, 0,
      hard_utimes, soft_utimes);
    HTMLTableEntrySpanBegin(html, 1, bgcolor);
    HTMLHSpace(html, 2);
    HTMLTableEntryEnd(html);
  }
  if( DEBUG3 )
    fprintf(stderr, "] TasksDailyDisplayTableEntry\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void LineDailyDisplayTableEntry(HTML html, ARRAY_KHE_TASK *line,         */
/*    ARRAY_KHE_TIME *hard_utimes, ARRAY_KHE_TIME *soft_utimes, DAY day,     */
/*    bool highlight_splits)                                                 */
/*                                                                           */
/*  Display the table entry for line and utimes at time.                     */
/*                                                                           */
/*****************************************************************************/

/* *** replaced by TasksDailyDisplayTableEntry
static void LineDailyDisplayTableEntry(HTML html, ARRAY_KHE_TASK *line,
  ARRAY_KHE_TIME *hard_utimes, ARRAY_KHE_TIME *soft_utimes, DAY day,
  bool highlight_splits)
{
  char buff[200];  KHE_TASK task;  bool first;  KHE_EVENT e;
  int i;  char *bgcolor, *s, *tg_name;  int incident_count;
  if( DEBUG3 )
    fprintf(stderr, "[ LineDailyDisplayTableEntry (%d tasks on line)\n",
      HaArrayCount(*line));
  first = true;
  ** bgcolor = !HaArrayContains(*utimes, time, &pos) ? NULL : Grey; **
  ** ***
  bgcolor = DayIsUnavailable(day, hard_utimes) ? Red :
    DayIsUnavailable(day, soft_utimes) ? LightRed : NULL;
  *** **
  incident_count = 0;
  tg_name = NULL;

  HaArrayForEach(*line, task, i)
    if( TaskRunningOnDay(task, day) )
      incident_count++;
  tg_name =  KheTimeGroupName(DayTimeGroup(day));

  HaArrayForEach(*line, task, i)
  {
    if( DEBUG3 )
      fprintf(stderr, "  LineDailyDisplayTableEntry (1) task %d\n", i);
    if( TaskRunningOnDay(task, day) )
    {
      e = KheMeetEvent(KheTaskMeet(task));
      s = strstr(KheEventName(e), tg_name);
      if( s == KheEventName(e) )
      {
	s += strlen(tg_name);
	if( *s == ':' )
	  s++;
        snprintf(buff, 10, "%s", s);
      }
      else
	snprintf(buff, 10, "%s", KheEventName(e));
      if( DEBUG3 )
	fprintf(stderr, "  LineDailyDisplayTableEntry (2) task %d: %s\n",
	  i, buff);
      if( first )
      {
	bgcolor = KheTableEntrySpecialDayBackgroundColour(task, day,
	  highlight_splits, incident_count, hard_utimes, soft_utimes);
	** ***
	if( bgcolor == NULL )
	{
	  if( !highlight_splits || incident_count > 1 ||
	      TaskEventResourceAsstIsDefective(task) )
	    bgcolor = KheEventColor(e);
	  else if( !KheTaskNeedsAssignment(task) )
	    bgcolor = LightBlue;
	  else
	    bgcolor = highlight_splits ? VeryLightGrey : White;
	}
	*** **
	HTMLTableEntrySpanBegin(html, 1, bgcolor);
	first = false;
      }
      else
	HTMLNewLine(html);
      if( DEBUG3 )
	fprintf(stderr, "  LineDailyDisplayTableEntry (5)\n");
      KheTableEntryFontShowText(task, buff, html);
      ** ***
      if( TaskEventResourceAsstIsDefective(task) )
	HTMLTextIta lic(html, buff);
      else if( !KheTaskNeedsAssignment(task) )
	HTMLTextMono(html, buff);
      else
	HTMLText(html, buff);
      *** **
      if( DEBUG3 )
	fprintf(stderr, "  LineDailyDisplayTableEntry (6)\n");
    }
  }
  if( DEBUG3 )
    fprintf(stderr, "  LineDailyDisplayTableEntry (4)\n");
  if( first )
  {
    bgcolor = KheTableEntrySpecialDayBackgroundColour(NULL, NULL, false, 0,
      hard_utimes, soft_utimes);
    HTMLTableEntrySpanBegin(html, 1, bgcolor);
    HTMLHSpace(html, 2);
  }
  HTMLTableEntryEnd(html);
  if( DEBUG3 )
    fprintf(stderr, "] LineDailyDisplayTableEntry\n");
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void LineDailyDisplay(HTML html, ARRAY_KHE_TASK *line,                   */
/*    ARRAY_KHE_TIME *hard_utimes, ARRAY_KHE_TIME *soft_utimes,              */
/*    ARRAY_WEEK *weeks, bool highlight_splits)                              */
/*                                                                           */
/*  Display one table entry for each day of *weeks, showing the events of    */
/*  line running on that day.  Parameters hard_utimes and soft_utimes        */
/*  contains sets of unavailable times, to be shown as background colour.    */
/*                                                                           */
/*****************************************************************************/

static void LineDailyDisplay(HTML html, ARRAY_KHE_TASK *line,
  ARRAY_KHE_TIME *hard_utimes, ARRAY_KHE_TIME *soft_utimes,
  ARRAY_WEEK *weeks, bool highlight_splits, HA_ARENA a)
{
  int i, j;  WEEK week;  DAY day;  ARRAY_KHE_TASK tasks;
  if( DEBUG3 )
    fprintf(stderr, "[ LineDailyDisplay(%d weeks)\n", HaArrayCount(*weeks));
  HaArrayInit(tasks, a);
  HaArrayForEach(*weeks, week, i)
  {
    for( j = 0;  j < WeekDayCount(week);  j++ )
    {
      day = WeekDay(week, j);
      LineSelectDayTasks(line, day, &tasks);
      TasksDailyDisplayTableEntry(&tasks, hard_utimes, soft_utimes,
	day, highlight_splits, html);
    }
  }
  if( DEBUG3 )
    fprintf(stderr, "] LineDailyDisplay\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "planning timetables - general"                                */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  int SolutionEventIncreasingTimeCmp(const void *t1, const void *t2)       */
/*                                                                           */
/*  Comparison function for unassigned soln event sorting.                   */
/*                                                                           */
/*****************************************************************************/

/* *** unused
static int TaskIncreasingTimeCmp(const void *t1, const void *t2)
{
  KHE_TASK task1 = * (KHE_TASK *) t1;
  KHE_TASK task2 = * (KHE_TASK *) t2;
  KHE_TIME time1 = KheMeetAsstTime(KheTaskMeet(task1));
  KHE_TIME time2 = KheMeetAsstTime(KheTaskMeet(task2));
  return KheTimeIndex(time1) - KheTimeIndex(time2);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool TasksOverlap(KHE_TASK task1, KHE_TASK task2)                        */
/*                                                                           */
/*  Return true if task1 and task2 overlap in time.                          */
/*                                                                           */
/*****************************************************************************/

static bool TasksOverlap(KHE_TASK task1, KHE_TASK task2)
{
  return KheMeetOverlap(KheTaskMeet(task1), KheTaskMeet(task2));
}


/*****************************************************************************/
/*                                                                           */
/*  bool TaskOverlapsLine(ARRAY_KHE_TASK *line, KHE_TASK task)               */
/*                                                                           */
/*  Return true if task overlaps any of the tasks of *line.                  */
/*                                                                           */
/*****************************************************************************/

static bool TaskOverlapsLine(ARRAY_KHE_TASK *line, KHE_TASK task)
{
  KHE_TASK task2;  int i;
  HaArrayForEach(*line, task2, i)
    if( TasksOverlap(task, task2) )
      return true;
  return false;
}


/*****************************************************************************/
/*                                                                           */
/* void TasksPartitionIntoLines(ARRAY_KHE_TASK *tasks,                       */
/*    ARRAY_ARRAY_KHE_TASK *lines, HA_ARENA a)                               */
/*                                                                           */
/*  Partition tasks into non-overlapping lines.                              */
/*                                                                           */
/*****************************************************************************/

static void TasksPartitionIntoLines(ARRAY_KHE_TASK *tasks,
  ARRAY_ARRAY_KHE_TASK *lines, HA_ARENA a)
{
  ARRAY_KHE_TASK line;  int unfinished_count, i;  KHE_TASK task;
  HaArrayClear(*lines);
  unfinished_count = HaArrayCount(*tasks);
  while( unfinished_count > 0 )
  {
    /* build one line of non-overlapping tasks */
    HaArrayInit(line, a);
    HaArrayForEach(*tasks, task, i)
      if( task != NULL && !TaskOverlapsLine(&line, task) )
      {
	HaArrayAddLast(line, task);
	HaArrayPut(*tasks, i, NULL);
	unfinished_count--;
      }
    HaArrayAddLast(*lines, line);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void ResourceTasks(KHE_SOLN soln, KHE_RESOURCE r, ARRAY_KHE_TASK *tasks) */
/*                                                                           */
/*  Set *tasks to the tasks that r is assigned to.                           */
/*                                                                           */
/*****************************************************************************/

static void ResourceTasks(KHE_SOLN soln, KHE_RESOURCE r, ARRAY_KHE_TASK *tasks)
{
  int i;  KHE_TASK task;
  HaArrayClear(*tasks);
  for( i = 0;  i < KheResourceAssignedTaskCount(soln, r);  i++ )
  {
    task = KheResourceAssignedTask(soln, r, i);
    HaArrayAddLast(*tasks, task);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void ResourceTypeAvail(KHE_RESOURCE_TYPE rt, KHE_SOLN soln,              */
/*    bool *has_avail_times, bool *has_avail_workload)                       */
/*                                                                           */
/*  Find out whether any of the resources of rt have avails.                 */
/*                                                                           */
/*****************************************************************************/

static void ResourceTypeAvail(KHE_RESOURCE_TYPE rt, KHE_SOLN soln,
  bool *has_avail_times, bool *has_avail_workload)
{
  float avail_workload;  int avail_times, i;  KHE_RESOURCE r;
  *has_avail_times = *has_avail_workload = false;
  for( i = 0;  i < KheResourceTypeResourceCount(rt);  i++ )
  {
    r = KheResourceTypeResource(rt, i);
    KheResourceAvailableBusyTimes(soln, r, &avail_times);
    KheResourceAvailableWorkload(soln, r, &avail_workload);
    if( avail_times < INT_MAX )
      *has_avail_times = true;
    if( avail_workload < FLT_MAX )
      *has_avail_workload = true;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void DisplayAvailTimesAndWorkload(int avail_times,                       */
/*    float avail_workload, HTML html)                                       */
/*                                                                           */
/*  Display avail_times (if less than INT_MAX) and avail_workload (if        */
/*  less than FLT_MAX) onto html.                                            */
/*                                                                           */
/*****************************************************************************/

static void DisplayAvailTimesAndWorkload(int avail_times,
  float avail_workload, HTML html)
{
  if( avail_times < INT_MAX )
  {
    if( avail_workload < FLT_MAX )
    {
      /* both avail times and avail workload */
      HTMLText(html, "%d; %.1f", avail_times, avail_workload);
    }
    else
    {
      /* avail times only */
      HTMLText(html, "%d", avail_times);
    }
  }
  else
  {
    if( avail_workload < FLT_MAX )
    {
      /* avail workload only */
      HTMLText(html, "%.1f", avail_workload);
    }
    else
    {
      /* nothing at all */
      HTMLHSpace(html, 2);
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void DisplayAvail(KHE_RESOURCE r, KHE_SOLN soln, HTML html)              */
/*                                                                           */
/*  Display the content of a table entry giving r's avail, or blank if none. */
/*                                                                           */
/*****************************************************************************/

static void DisplayAvail(KHE_RESOURCE r, KHE_SOLN soln, HTML html)
{
  int avail_times;  float avail_workload;
  KheResourceAvailableBusyTimes(soln, r, &avail_times);
  KheResourceAvailableWorkload(soln, r, &avail_workload);
  DisplayAvailTimesAndWorkload(avail_times, avail_workload, html);
}


/*****************************************************************************/
/*                                                                           */
/*  void ColouredBox(char *text, char *bg_color, HTML html)                  */
/*                                                                           */
/*  Print a coloured box.                                                    */
/*                                                                           */
/*****************************************************************************/

/* *** done for testing, no longer needed
static void ColouredBox(char *text, char *bg_color, HTML html)
{
  HTMLColouredBoxBegin(html, bg_color);
  HTMLLiteralText(html, text);
  HTMLColouredBoxEnd(html);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void ColouredBoxPalette(HTML html)                                       */
/*                                                                           */
/*  Print a palette.                                                         */
/*                                                                           */
/*****************************************************************************/

/* *** done for testing, no longer needed
static void ColouredBoxPalette(HTML html)
{
  ColouredBox("BrightGreen", BrightGreen, html);
  ColouredBox("LightGreen", LightGreen, html);
  ColouredBox("LightYellow", LightYellow, html);
  ColouredBox("Red", Red, html);
  ColouredBox("LightRed", LightRed, html);
  ColouredBox("LightBlue", LightBlue, html);
  ColouredBox("White", White, html);
  ColouredBox("VeryLightGrey", VeryLightGrey, html);
  ColouredBox("LightGrey", LightGrey, html);
  ColouredBox("Grey", Grey, html);
  ColouredBox("Black", Black, html);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void ResourceTypeDailyPlanningTimetableHTML(KHE_SOLN soln,               */
/*    KHE_RESOURCE_TYPE rt, bool highlight_splits, char *defects_str,        */
/*    HTML html, HA_ARENA a)                                                 */
/*                                                                           */
/*  Print a planning timetable for the resources of rt.                      */
/*                                                                           */
/*****************************************************************************/

static void ResourceTypeDailyPlanningTimetableHTML(KHE_SOLN soln,
  KHE_RESOURCE_TYPE rt, bool highlight_splits, DEFECT_INFO di,
  ARRAY_WEEK *weeks, HTML html, HA_ARENA a)
{
  KHE_RESOURCE r;  int i, j, avail_times /* , limit */;  KHE_INSTANCE ins;
  bool show_avail_times, show_avail_workload;
  ARRAY_ARRAY_KHE_TASK lines; ARRAY_KHE_TASK unassigned_tasks, line;
  KHE_TASK task;  float avail_workload;
  ARRAY_KHE_TIME hard_utimes, soft_utimes;
  /* ARRAY_WEEK weeks; */  WEEK week;  DAY day;  KHE_TIME_GROUP tg;
  if( DEBUG4 )
    fprintf(stderr, "[ ResourceTypeDailyPlanningTimetableHTML\n");
  ins = KheSolnInstance(soln);
  /* ***
  if( KheSolnType(soln) == KHE_SOLN_INVALID_PLACEHOLDER )
  {
    if( DEBUG4 )
      fprintf(stderr, "  ResourceTypeDailyPlanningTimetableHTML invalid\n");
    SolnInvalidParagraph(soln, html);
  }
  else if( !WeekTimetableSkeleton(ins, &weeks, a) )
  {
    HTMLParagraphBegin(html);
    HTMLText(html, "HSEval cannot print timetables for this instance");
    HTMLText(html, "because it cannot find, or cannot make sense of,");
    HTMLText(html, "its Days (and Weeks if present).");
    HTMLParagraphEnd(html);
  }
  else
  *** */
  if( KheResourceTypeResourceCount(rt) > 0 )
  {
    /* work out whether we are going to have an avail column or not */
    if( DEBUG4 )
      fprintf(stderr, "  ResourceTypeDailyPlanningTimetableHTML (a)\n");
    HaArrayInit(line, a);
    HaArrayInit(hard_utimes, a);
    HaArrayInit(soft_utimes, a);
    ResourceTypeAvail(rt, soln, &show_avail_times, &show_avail_workload);

    /* heading (only if there are several resource types) */
    if( KheInstanceResourceTypeCount(ins) > 1 )
    {
      HTMLParagraphBegin(html);
      HTMLHeadingBegin(html);
      HTMLText(html, KheResourceTypeName(rt));
      HTMLTextNoBreak(html, " Daily Timetables");
      HTMLHeadingEnd(html);
      HTMLParagraphEnd(html);
    }

    /* ***
    HTMLParagraphBegin(html);
    ColouredBoxPalette(html);
    HTMLParagraphEnd(html);
    *** */

    /* header row */
    HTMLParagraphBegin(html);
    HTMLTableBegin(html, LightGreen);

    HTMLTableRowVAlignBegin(html, "top");
    HTMLTableEntryBegin(html);
    if( KheInstanceResourceTypeCount(ins) > 1 )
      HTMLHSpace(html, 2);
    else
      HTMLTextBold(html, KheResourceTypeName(rt));
    HTMLTableEntryEnd(html);
    if( show_avail_times || show_avail_workload )
    {
      HTMLTableEntryCentredBegin(html);
      HTMLTextBold(html, "Avail");
      HTMLTableEntryEnd(html);
    }
    HaArrayForEach(*weeks, week, i)
      for( j = 0;  j < WeekDayCount(week);  j++ )
      {
	day = WeekDay(week, j);
	tg = DayTimeGroup(day);
	HTMLTableEntryCentredBegin(html);
	HTMLTextBold(html, KheTimeGroupName(tg));
	HTMLTableEntryEnd(html);
      }
    HTMLTableRowEnd(html);

    /* resource rows */
    for( i = 0;  i < KheResourceTypeResourceCount(rt);  i++ )
    {
      r = KheResourceTypeResource(rt, i);
      HTMLTableRowVAlignBegin(html, "top");
      HTMLTableEntryBegin(html);
      HTMLTextBold(html, KheResourceName(r));
      HTMLTableEntryEnd(html);
      if( show_avail_times || show_avail_workload )
      {
	HTMLTableEntryCentredBegin(html);
        DisplayAvail(r, soln, html);
	HTMLTableEntryEnd(html);
      }

      /* individual tasks */
      ResourceTasks(soln, r, &line);
      ResourceUnavailableTimes(r, &hard_utimes, &soft_utimes);
      LineDailyDisplay(html, &line, &hard_utimes, &soft_utimes, weeks,
	highlight_splits, a);
      HTMLTableRowEnd(html);
    }

    /* find the defective unassigned tasks of type rt */
    if( DEBUG4 )
      fprintf(stderr, "  ResourceTypeDailyPlanningTimetableHTML (c)\n");
    HaArrayInit(unassigned_tasks, a);
    for( i = 0;  i < KheSolnTaskCount(soln);  i++ )
    {
      task = KheSolnTask(soln, i);
      if( KheTaskResourceType(task) == rt && KheTaskAsstResource(task) == NULL 
	  && TaskEventResourceAsstIsDefective(task) )
	HaArrayAddLast(unassigned_tasks, task);
    }

    /* print lines of unassigned tasks */
    HaArrayInit(lines, a);
    HaArrayClear(hard_utimes);
    HaArrayClear(soft_utimes);
    TasksPartitionIntoLines(&unassigned_tasks, &lines, a);
    HaArrayForEach(lines, line, i)
    {
      HTMLTableRowVAlignBegin(html, "top");
      HTMLTableEntryBegin(html);
      HTMLTextBold(html, "Unassigned");
      HTMLTableEntryEnd(html);
      if( show_avail_times || show_avail_workload )
      {
	if( show_avail_times )
	{
	  avail_times = 0;
	  HaArrayForEach(line, task, j)
	    avail_times += KheTaskDuration(task);
	}
	else
	  avail_times = INT_MAX;
	if( show_avail_workload )
	{
	  avail_workload = 0.0;
	  HaArrayForEach(line, task, j)
	    avail_workload += KheTaskWorkload(task);
	}
	else
	  avail_workload = FLT_MAX;
	HTMLTableEntryCentredBegin(html);
        DisplayAvailTimesAndWorkload(avail_times, avail_workload, html);
	HTMLTableEntryEnd(html);
      }
      LineDailyDisplay(html, &line, &hard_utimes, &soft_utimes, weeks,
	highlight_splits, a);
      HTMLTableRowEnd(html);
    }

    HTMLTableEnd(html);
    HTMLParagraphEnd(html);
  }
  if( DEBUG4 )
    fprintf(stderr, "] ResourceTypeDailyPlanningTimetableHTML\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void ResourceTypePlanningTimetableHTML(KHE_SOLN soln,                    */
/*    KHE_RESOURCE_TYPE rt, ARRAY_KHE_TIME *permuted_times,                  */
/*    bool highlight_splits, HTML html, HA_ARENA a)                          */
/*                                                                           */
/*  Print a planning timetable for the resources of rt at permuted_times.    */
/*                                                                           */
/*****************************************************************************/

static void ResourceTypePlanningTimetableHTML(KHE_SOLN soln,
  KHE_RESOURCE_TYPE rt, ARRAY_KHE_TIME *permuted_times,
  bool highlight_splits, HTML html, HA_ARENA a)
{
  KHE_TIME time;  KHE_RESOURCE r;  int i, j, avail_times /* , limit */;
  bool show_avail_times, show_avail_workload;  float avail_workload;
  ARRAY_ARRAY_KHE_TASK lines; ARRAY_KHE_TASK unassigned_tasks, line;
  KHE_TASK task;  ARRAY_KHE_TIME hard_utimes, soft_utimes;
  if( KheResourceTypeResourceCount(rt) > 0 )
  {
    /* work out whether we are going to have an avail column or not */
    HaArrayInit(line, a);
    HaArrayInit(hard_utimes, a);
    HaArrayInit(soft_utimes, a);
    ResourceTypeAvail(rt, soln, &show_avail_times, &show_avail_workload);

    /* heading */
    HTMLParagraphBegin(html);
    HTMLHeadingBegin(html);
    HTMLTextNoBreak(html, "Resource type ");
    HTMLText(html, KheResourceTypeName(rt));
    HTMLHeadingEnd(html);
    HTMLParagraphEnd(html);

    /* header row */
    HTMLParagraphBegin(html);
    HTMLTableBegin(html, LightGreen);
    HTMLTableRowVAlignBegin(html, "top");
    HTMLTableEntryBegin(html);
    HTMLHSpace(html, 2);
    HTMLTableEntryEnd(html);
    if( show_avail_times || show_avail_workload )
    {
      HTMLTableEntryCentredBegin(html);
      HTMLTextBold(html, "Avail");
      HTMLTableEntryEnd(html);
    }
    HaArrayForEach(*permuted_times, time, i)
    {
      HTMLTableEntryCentredBegin(html);
      HTMLTextBold(html, KheTimeName(time));
      HTMLTableEntryEnd(html);
    }
    HTMLTableRowEnd(html);

    /* resource rows */
    for( i = 0;  i < KheResourceTypeResourceCount(rt);  i++ )
    {
      r = KheResourceTypeResource(rt, i);
      HTMLTableRowVAlignBegin(html, "top");
      HTMLTableEntryBegin(html);
      HTMLTextBold(html, KheResourceName(r));
      HTMLTableEntryEnd(html);
      if( show_avail_times || show_avail_workload )
      {
	HTMLTableEntryCentredBegin(html);
        DisplayAvail(r, soln, html);
	HTMLTableEntryEnd(html);
      }

      /* individual tasks */
      ResourceTasks(soln, r, &line);
      ResourceUnavailableTimes(r, &hard_utimes, &soft_utimes);
      LineDisplay(html, &line, &hard_utimes, &soft_utimes, permuted_times,
	highlight_splits, a);
      HTMLTableRowEnd(html);
    }

    /* find the unassigned tasks of type rt */
    HaArrayInit(unassigned_tasks, a);
    for( i = 0;  i < KheSolnTaskCount(soln);  i++ )
    {
      task = KheSolnTask(soln, i);
      if( KheTaskResourceType(task) == rt && KheTaskAsstResource(task) == NULL )
	HaArrayAddLast(unassigned_tasks, task);
    }

    /* print lines of unassigned tasks */
    HaArrayInit(lines, a);
    HaArrayClear(hard_utimes);
    HaArrayClear(soft_utimes);
    TasksPartitionIntoLines(&unassigned_tasks, &lines, a);
    HaArrayForEach(lines, line, i)
    {
      HTMLTableRowVAlignBegin(html, "top");
      HTMLTableEntryBegin(html);
      HTMLTextBold(html, "Unassigned");
      HTMLTableEntryEnd(html);
      if( show_avail_times || show_avail_workload )
      {
	if( show_avail_times )
	{
	  avail_times = 0;
	  HaArrayForEach(line, task, j)
	    avail_times += KheTaskDuration(task);
	}
	else
	  avail_times = INT_MAX;
	if( show_avail_workload )
	{
	  avail_workload = 0.0;
	  HaArrayForEach(line, task, j)
	    avail_workload += KheTaskWorkload(task);
	}
	else
	  avail_workload = FLT_MAX;
	HTMLTableEntryRightBegin(html);
        DisplayAvailTimesAndWorkload(avail_times, avail_workload, html);
	HTMLTableEntryEnd(html);
      }
      LineDisplay(html, &line, &hard_utimes, &soft_utimes, permuted_times,
	highlight_splits, a);
      HTMLTableRowEnd(html);
    }

    HTMLTableEnd(html);
    HTMLParagraphEnd(html);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  int EventDecreasingDurationCmp(const void *t1, const void *t2)           */
/*                                                                           */
/*  Comparison function for sorting an array of events by decreasing         */
/*  duration.                                                                */
/*                                                                           */
/*****************************************************************************/

static int EventDecreasingDurationCmp(const void *t1, const void *t2)
{
  KHE_EVENT e1 = * (KHE_EVENT *) t1;
  KHE_EVENT e2 = * (KHE_EVENT *) t2;
  if( KheEventDuration(e2) != KheEventDuration(e1) )
    return KheEventDuration(e2) - KheEventDuration(e1);
  else
    return KheEventIndex(e1) - KheEventIndex(e2);
}


/*****************************************************************************/
/*                                                                           */
/*  void ResourceBusyTimesAndEvents(KHE_SOLN soln, KHE_RESOURCE r,           */
/*    int *r_busy_times, ARRAY_KHE_EVENT *events)                            */
/*                                                                           */
/*  Set *r_busy_times to the number of times that r is busy in soln, and     */
/*  set *events to the events that it is assigned to, wholly or partially.   */
/*                                                                           */
/*****************************************************************************/

static void ResourceBusyTimesAndEvents(KHE_SOLN soln, KHE_RESOURCE r,
  int *r_busy_times, ARRAY_KHE_EVENT *events)
{
  int i, pos;  KHE_TASK task;  KHE_EVENT e;
  HaArrayClear(*events);
  *r_busy_times = 0;
  for( i = 0;  i < KheResourceAssignedTaskCount(soln, r);  i++ )
  {
    task = KheResourceAssignedTask(soln, r, i);
    *r_busy_times += KheTaskDuration(task);
    e = KheMeetEvent(KheTaskMeet(task));
    if( e != NULL && !HaArrayContains(*events, e, &pos) )
      HaArrayAddLast(*events, e);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void SolutionPlanningTimetablesHTML(KHE_SOLN soln,                       */
/*    bool highlight_splits, char *defects_str, HTML html, HA_ARENA a)       */
/*                                                                           */
/*  Print planning timetables onto html, one per resource type.              */
/*                                                                           */
/*****************************************************************************/

static void SolutionPlanningTimetablesHTML(KHE_SOLN soln,
  bool highlight_splits, char *defects_str, ARRAY_WEEK *weeks,
  HTML html, HA_ARENA a)
{
  ARRAY_KHE_TIME permuted_times;  KHE_RESOURCE_TYPE rt, best_rt;
  ARRAY_KHE_EVENT events;  KHE_EVENT e;  KHE_RESOURCE r, best_r;
  int i, j, k, r_busy_times, best_busy_times, best_ecount, pos;
  KHE_TIME start_time, time;  KHE_INSTANCE ins;  KHE_MEET meet;
  MODEL model;  MODEL_RESOURCE_TYPE classes_mrt, mrt;  DEFECT_INFO di;

  /* header */
  ins = KheSolnInstance(soln);
  if( DEBUG1 || DEBUG3 )
    fprintf(stderr,"[ SolutionPlanningTimetablesHTML(soln to %s, %s, \"%s\")\n",
      KheInstanceId(ins), highlight_splits ? "true" : "false", defects_str);

  /* ***
  SolnHeader(soln, html);

  if( KheSolnType(soln) == KHE_SOLN_INVALID_PLACEHOLDER )
    SolnInvalidParagraph(soln, html);
  else
  *** */
    
  if( KheInstanceModel(ins) == KHE_MODEL_EMPLOYEE_SCHEDULE )
  {
    /* one timetable for each resource type */
    di = DefectInfoMake(soln, defects_str);
    for( i = 0;  i < KheInstanceResourceTypeCount(ins);  i++ )
    {
      rt = KheInstanceResourceType(ins, i);
      if( DEBUG1 || DEBUG3 )
	fprintf(stderr, "  SolutionPlanningTimetablesHTML resource type %s\n",
	  KheResourceTypeId(rt));
      ResourceTypeDailyPlanningTimetableHTML(soln, rt, highlight_splits,
	di, weeks, html, a);
    }
  }
  else
  {
    /* find a resource type called Class if there is one */
    model = ModelBuild(KheInstanceModel(ins), a);
    classes_mrt = ModelRetrieve(model, "Classes");
    best_rt = NULL;
    for( i = 0;  i < KheInstanceResourceTypeCount(ins);  i++ )
    {
      rt = KheInstanceResourceType(ins, i);
      mrt = ModelRetrieve(model, KheResourceTypeName(rt));
      if( mrt == classes_mrt )
      {
	best_rt = rt;
	break;
      }
    }

    /* if there is a resource type called Class, find its busiest resource */
    best_r = NULL;
    HaArrayInit(events, a);
    if( best_rt != NULL )
      for( i = 0;  i < KheResourceTypeResourceCount(best_rt);  i++ )
      {
	r = KheResourceTypeResource(best_rt, i);
	ResourceBusyTimesAndEvents(soln, r, &r_busy_times, &events);
	if( best_r == NULL || r_busy_times > best_busy_times ||
	 (r_busy_times == best_busy_times && HaArrayCount(events) < best_ecount) )
	{
	  best_r = r;
	  best_busy_times = r_busy_times;
	  best_ecount = HaArrayCount(events);
	}
      }

    /* if no best_r yet, search entire instance */
    if( best_r == NULL )
      for( i = 0;  i < KheInstanceResourceCount(ins);  i++ )
      {
	r = KheInstanceResource(ins, i);
	ResourceBusyTimesAndEvents(soln, r, &r_busy_times, &events);
	if( best_r == NULL || r_busy_times > best_busy_times ||
	 (r_busy_times == best_busy_times && HaArrayCount(events) < best_ecount) )
	{
	  best_r = r;
	  best_busy_times = r_busy_times;
	  best_ecount = HaArrayCount(events);
	}
      }

    /* if still no best_r, nothing to print */
    if( best_r == NULL )
    {
      HTMLParagraphBegin(html);
      HTMLText(html, "This solution's instance contains no resources.");
      HTMLParagraphEnd(html);
      return;
    }
    if( DEBUG1 )
      fprintf(stderr, "  best_r is %s\n", KheResourceId(best_r));
    ResourceBusyTimesAndEvents(soln, best_r, &r_busy_times, &events);
    HaArraySort(events, &EventDecreasingDurationCmp);

    /* build permuted times by following the busiest resource's timetable */
    HaArrayInit(permuted_times, a);
    HaArrayForEach(events, e, i)
    {
      if( DEBUG1 )
	fprintf(stderr, "  %s attends event %s\n", KheResourceId(best_r),
	  KheEventId(e));
      for( j = 0;  j < KheEventMeetCount(soln, e);  j++ )
      {
	meet = KheEventMeet(soln, e, j);
	start_time = KheMeetAsstTime(meet);
	if( start_time != NULL )
	{
	  for( k = 0;  k < KheMeetDuration(meet);  k++ )
	  {
	    time = KheTimeNeighbour(start_time, k);
	    if( !HaArrayContains(permuted_times, time, &pos) )
	    {
	      HaArrayAddLast(permuted_times, time);
	      if( DEBUG1 )
	      {
		fprintf(stderr, "  %s attends meet ", KheResourceId(best_r));
		KheMeetDebug(meet, 1, -1, stderr);
		fprintf(stderr, " at time %s\n", KheTimeId(time));
	      }
	    }
	  }
	}
      }
    }

    /* add leftover times, not already in permuted_times */
    for( i = 0;  i < KheInstanceTimeCount(ins);  i++ )
    {
      time = KheInstanceTime(ins, i);
      if( !HaArrayContains(permuted_times, time, &pos) )
      {
	HaArrayAddLast(permuted_times, time);
	if( DEBUG1 )
	  fprintf(stderr, "  extra time %s\n", KheTimeId(time));
      }
    }

    /* one timetable for each resource type */
    for( i = 0;  i < KheInstanceResourceTypeCount(ins);  i++ )
    {
      rt = KheInstanceResourceType(ins, i);
      ResourceTypePlanningTimetableHTML(soln, rt, &permuted_times,
	highlight_splits, html, a);
    }
    /* MArrayFree(permuted_times); */
    /* MArrayFree(events); */
  }
  if( DEBUG1 || DEBUG3 )
    fprintf(stderr, "] SolutionPlanningTimetablesHTML returning\n");
}


/*****************************************************************************/
/*                                                                           */
/*  PT_INFO - info for planning timetable printing                           */
/*                                                                           */
/*****************************************************************************/

/* ***
typedef struct pt_info_rec {
  HTML			html;
  HA_ARENA_SET		arena_set;
  bool			highlight_splits;
  char			*defects_str;
  char			*magic;
} *PT_INFO;
*** */


/*****************************************************************************/
/*                                                                           */
/*  void archive_begin_fn(KHE_ARCHIVE archive, void *impl)                   */
/*                                                                           */
/*  Do this at start of archive.                                             */
/*                                                                           */
/*****************************************************************************/

/* ***
static void archive_begin_fn(KHE_ARCHIVE archive, void *impl)
{
  PT_INFO pti;  char buff[100];
  pti = (PT_INFO) impl;

  ** create html writer object and start off the page **
  if( DEBUG3 )
    fprintf(stderr, "[ archive_begin_fn()\n");
  HnAssert(pti->html == NULL, "archive_begin_fn internal error");
  snprintf(buff, 100, "%s Planning Timetables",
    KheArchiveId(archive) != NULL && strlen(KheArchiveId(archive)) < 70 ?
    KheArchiveId(archive) : "HSEval");
  pti->html = PageBegin(buff);
  HTMLBigHeading(pti->html, buff);
  if( DEBUG3 )
    fprintf(stderr, "] archive_begin_fn()\n");
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void IntroParagraphs(KHE_ARCHIVE archive, HTML html)                     */
/*                                                                           */
/*  Print some introductory paragraphs.                                      */
/*                                                                           */
/*****************************************************************************/

/* ***
static void IntroParagraphs(KHE_ARCHIVE archive, HTML html)
{
  HTMLParagraphBegin(html);
  HTMLText(html, "For each solution in the uploaded XML archive, this page");
  HTMLText(html, "contains one planning timetable for each resource type");
  HTMLText(html, "of the corresponding instance.");
  HTMLParagraphEnd(html);

  HTMLParagraphBegin(html);
  HTMLText(html, "The Avail column, where present, shows an integer number");
  HTMLText(html, "of available times, based on cluster busy times and limit");
  HTMLText(html, "busy times constraints, or a decimal amount of available");
  HTMLText(html, "workload, based on limit workload constraints.  These are");
  HTMLText(html, "heuristic estimates, and are occasionally higher than the");
  HTMLText(html, "true values, but never lower.  A negative value proves that");
  HTMLText(html, "the resource is overloaded.  In Unassigned rows the Avail");
  HTMLText(html, "column shows the unassigned times or workload.  For more");
  HTMLText(html, "information about how availability is calculated, see the");
  HTMLText(html, "Availability Report, reached from the front page of HSEval.");
  HTMLParagraphEnd(html);

  HTMLParagraphBegin(html);
  HTMLText(html, "The background colour of each timetable cell is red if");
  HTMLText(html, "the resource is unavailable then (dark red for hard");
  HTMLText(html, "constraints, light red for soft ones), or else light green");
  HTMLText(html, "if the cell contains no events, or else the colour of the");
  HTMLText(html, "first event depicted (white if the event has no colour).");
  HTMLText(html, "If specific constraints are being highlighted, their times");
  HTMLText(html, "will appear in light blue, with defective times in blue.");
  HTMLParagraphEnd(html);

  HTMLParagraphBegin(html);
  HTMLText(html, "The text in each cell appears in bold italic font if the");
  HTMLText(html, "event resource it indicates is subject to an event");
  HTMLText(html, "resource constraint which is violated (the event resource");
  HTMLText(html, "may be unassigned when an assignment is wanted, assigned an");
  HTMLText(html, "unpreferred resource, part of a split assignment, etc.), or");
  HTMLText(html, "else in italic when unassigning the event resource would");
  HTMLText(html, "not violate an event resource constraint, or else in roman.");
  HTMLParagraphEnd(html);

  ** ***
  HTMLParagraphBegin(html);
  HTMLText(html, "Italic font indicates that there is a cost associated");
  HTMLText(html, "with the event resource depicted:  it may be unassigned");
  HTMLText(html, "when an assignment is wanted, assigned when an assignment");
  HTMLText(html, "is not wanted, assigned an unpreferred resource, part");
  HTMLText(html, "of a split assignment, etc.  Otherwise, typewriter font");
  HTMLText(html, "indicates that unassigning the event resource would not");
  HTMLText(html, "cause a violation of any event resource constraint.");
  HTMLParagraphEnd(html);

  HTMLParagraphBegin(html);
  if( KheArchiveModel(archive) == KHE_MODEL_EMPLOYEE_SCHEDULE )
  {
    HTMLText(html, "Each column represents one day.");
    HTMLText(html, "Red boxes denote completely unavailable days (light");
    HTMLText(html, "red for soft constraints, dark red for hard ones).");
    HTMLText(html, "Light blue boxes hold assigned tasks which could be");
    HTMLText(html, "unassigned without incurring a cost from an event");
    HTMLText(html, "resource constraint.");
  }
  else
  {
    HTMLText(html, "Grey boxes denote unavailable times (light");
    HTMLText(html, "grey for soft constraints, dark grey for hard ones).");
    HTMLText(html, "The times are reordered heuristically to");
    HTMLText(html, "bring split events together.");
  }
  HTMLParagraphEnd(html);
  *** **
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void soln_group_begin_fn(KHE_SOLN_GROUP soln_group, void *impl)          */
/*                                                                           */
/*  Do this at the start of each solution group.                             */
/*                                                                           */
/*****************************************************************************/

/* ***
static void soln_group_begin_fn(KHE_SOLN_GROUP soln_group, void *impl)
{
  PT_INFO pti;  HTML html;  KHE_ARCHIVE archive;
  pti = (PT_INFO) impl;
  html = pti->html;

  ** print introductory blurb if first soln group **
  if( DEBUG3 )
    fprintf(stderr, "[ soln_group_begin_fn()\n");
  archive = KheSolnGroupArchive(soln_group);
  if( KheArchiveSolnGroupCount(archive) == 1 )
    IntroParagraphs(archive, html);

  ** print soln group metadata **
  HTMLHorizontalRule(html);
  KheSolnGroupHeader(soln_group, html);
  if( DEBUG3 )
    fprintf(stderr, "] soln_group_begin_fn()\n");
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void soln_fn(KHE_SOLN soln, void *impl)                                  */
/*                                                                           */
/*  Do this for each each solution.                                          */
/*                                                                           */
/*****************************************************************************/

/* ***
static void soln_fn(KHE_SOLN soln, void *impl)
{
  PT_INFO pti = (PT_INFO) impl;  HA_ARENA a;
  if( DEBUG5 )
    fprintf(stderr, "[ soln_fn(%s, %s)\n",
      KheInstanceId(KheSolnInstance(soln)), pti->magic);
  a = HaArenaSetArenaBegin(pti->arena_set, false);
  SolutionPlanningTimetablesHTML(soln, pti->highlight_splits, pti->defects_str,
    pti->html, a);
  HaArenaSetArenaEnd(pti->arena_set, a);
  if( KheSolnType(soln) == KHE_SOLN_ORDINARY )
    ** KheSolnReduceToPlaceholder(soln, false); **
    KheSolnTypeReduce(soln, KHE_SOLN_BASIC_PLACEHOLDER, NULL);
  if( DEBUG5 )
    fprintf(stderr, "] soln_fn(%s)\n", KheInstanceId(KheSolnInstance(soln)));
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void PlanningTimetablesHTML(COMMAND c, bool highlight_splits,            */
/*    HA_ARENA_SET as)                                                       */
/*                                                                           */
/*  Print planning timetables for the solutions of the archive.              */
/*                                                                           */
/*****************************************************************************/

/* ***
void PlanningTimetablesHTML(COMMAND c, bool highlight_splits, HA_ARENA_SET as)
{
  KHE_ARCHIVE archive;  char buff[200];  HTML html;  FILE *fp;
  struct pt_info_rec pt_info;  char *op, *junk;  KML_ERROR ke;

  if( DEBUG3 || DEBUG6 )
    fprintf(stderr, "[ PlanningTimetablesHTML(cgi, %s)\n",
      highlight_splits ? "true" : "false");

  ** set up pt_info, can't create HTML object until page title known **
  pt_info.html = NULL;
  pt_info.arena_set = as;
  pt_info.highlight_splits = highlight_splits;
  pt_info.defects_str = "";
  pt_info.magic = "magic";

  ** get next cgi op and make sure it's a file **
  if( !CommandNextPair(c, &op, &junk, &fp) || strcmp(op, "file") != 0 )
    CommandError(c, "file expected but missing");

  ** read the file incrementally and generate the timetables **
  if( KheArchiveReadIncremental(fp, as, &archive, &ke, false, false, false,
	false, false, KHE_SOLN_ORDINARY, CommandRerunFile(c), archive_begin_fn,
	NULL, soln_group_begin_fn, NULL, soln_fn, &pt_info) )
  {
    if( DEBUG3 )
      fprintf(stderr,
	"  PlanningTimetablesHTML after KheArchiveReadIncremental (true)\n");
    HnAssert(pt_info.html != NULL, "PlanningTimetablesHTML internal error");
    html = pt_info.html;

    if( KheArchiveSolnGroupCount(archive) == 0 )
    {
      HTMLParagraphBegin(html);
      HTMLText(html, "The uploaded XML file contains no solution groups.");
      HTMLParagraphEnd(html);
    }
  }
  else
  {
    ** make an HTML object if not done yet **
    if( DEBUG3 )
      fprintf(stderr,
       "  PlanningTimetablesHTML after KheArchiveReadIncremental (false)\n");
    if( pt_info.html == NULL )
    {
      html = PageBegin("HSEval: Error in uploaded XML file");
      HTMLBigHeading(html, "HSEval: Error in uploaded XML file");
    }
    else
      html = pt_info.html;
    if( DEBUG3 )
      fprintf(stderr, "  PlanningTimetablesHTML (2)\n");
    HnAssert(html != NULL, "PlanningTimetablesHTML internal error 2");

    ** print the error **
    HTMLParagraphBegin(html);
    HTMLText(html, "The XML file you just uploaded has at least one error:");
    HTMLParagraphEnd(html);
    snprintf(buff, 200, "line %d col %d: %s", KmlErrorLineNum(ke),
      KmlErrorColNum(ke), KmlErrorString(ke));
    HTMLParagraphBegin(html);
    HTMLColouredBoxBegin(html, LightRed);
    HTMLLiteralText(html, buff);
    HTMLColouredBoxEnd(html);
    HTMLParagraphEnd(html);
  }

  ** print a back jump link **
  HTMLParagraphBegin(html);
  HTMLText(html, "Return to the ");
  HTMLJumpFront(html);
  HTMLText(html, ".");
  HTMLParagraphEnd(html);

  ** and quit **
  PageEnd(html);
  if( DEBUG3 || DEBUG6 )
    fprintf(stderr, "] PlanningTimetablesHTML\n");
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void PlanningTimetablesHTML2(COMMAND c, bool highlight_splits,           */
/*    HA_ARENA_SET as)                                                       */
/*                                                                           */
/*  Print planning timetables for the solutions of the archive.              */
/*                                                                           */
/*****************************************************************************/

/* ***
void PlanningTimetablesHTML2(COMMAND c, bool highlight_splits,
  HA_ARENA_SET as)
{
  KHE_ARCHIVE archive;  char buff[1000];  HTML html;  int i, j;
  KHE_SOLN_GROUP soln_group;  KHE_SOLN soln;  HA_ARENA a;
  char *defects_str;
  ** char *contributor, *date, *description, *publication, *remarks; **

  if( DEBUG3 )
    fprintf(stderr, "[ PlanningTimetablesHTML2(cgi, %s)\n",
      highlight_splits ? "true" : "false");

  archive = ReadAndVerifyArchive(c, true, KHE_SOLN_ORDINARY, as);
  a = HaArenaSetArenaBegin(as, false);
  snprintf(buff, 1000, "%s Planning Timetables",
    KheArchiveId(archive) != NULL && strlen(KheArchiveId(archive)) < 70 ?
    KheArchiveId(archive) : "HSEval");
  html = PageBegin(buff);
  HTMLBigHeading(html, buff);

  if( KheArchiveSolnGroupCount(archive) == 0 )
  {
    HTMLParagraphBegin(html);
    HTMLText(html, "The uploaded XML file contains no solution groups.");
    HTMLParagraphEnd(html);
  }
  else
  {
    ** print introductory stuff **
    IntroParagraphs(archive, html);

    defects_str = "";
    if( strcmp(defects_str, "") != 0 )
    {
      HTMLParagraphBegin(html);
      HTMLText(html, "Printing defects whose constraints match \"%s\".",
	defects_str);
      HTMLParagraphEnd(html);
    }

    for( i = 0;  i < KheArchiveSolnGroupCount(archive);  i++ )
    {
      ** print soln group metadata **
      soln_group = KheArchiveSolnGroup(archive, i);
      HTMLHorizontalRule(html);
      KheSolnGroupHeader(soln_group, html);

      ** print solutions **
      for( j = 0;  j < KheSolnGroupSolnCount(soln_group);  j++ )
      {
	soln = KheSolnGroupSoln(soln_group, j);
	SolutionPlanningTimetablesHTML(soln, highlight_splits, defects_str,
	  html, a);
	if( KheSolnType(soln) == KHE_SOLN_ORDINARY )
	  KheSolnTypeReduce(soln, KHE_SOLN_BASIC_PLACEHOLDER, NULL);
	  ** KheSolnReduceToPlaceholder(soln, false); **
      }
    }
  }

  ** print a back jump link **
  HTMLParagraphBegin(html);
  HTMLText(html, "Return to the ");
  HTMLJumpFront(html);
  HTMLText(html, ".");
  HTMLParagraphEnd(html);

  ** and quit **
  PageEnd(html);
  HaArenaSetArenaEnd(as, a);
  if( DEBUG3 )
    fprintf(stderr, "] PlanningTimetablesHTML2\n");
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  Submodule "timetables - new version"                                     */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*****************************************************************************/

void SolutionTimetables(KHE_SOLN soln, bool planning,
  bool highlight_splits, bool with_event_groups, char *constraints_str,
  HTML html, HA_ARENA_SET as)
{
  HA_ARENA a;  KHE_INSTANCE ins;  ARRAY_WEEK weeks;
  a = HaArenaSetArenaBegin(as, false);
  ins = KheSolnInstance(soln);
  SolnHeader(soln, html);
  if( KheSolnType(soln) == KHE_SOLN_INVALID_PLACEHOLDER )
    SolnInvalidParagraph(soln, html);
  else if( !WeekTimetableSkeleton(ins, &weeks, a) )
  {
    HTMLParagraphBegin(html);
    HTMLText(html, "HSEval cannot print timetables for this instance");
    HTMLText(html, "because it cannot find, or cannot make sense of,");
    HTMLText(html, "its Days (and Weeks if present).");
    HTMLParagraphEnd(html);
  }
  else
  {
    /* soln is good to print */
    if( planning )
      SolutionPlanningTimetablesHTML(soln, highlight_splits, constraints_str,
	&weeks, html, a);
    else
      SolutionTimetablesHTML(soln, with_event_groups, &weeks, html, a);
  }
  HaArenaSetArenaEnd(as, a);
}


/*****************************************************************************/
/*                                                                           */
/*  void SolutionGroupTimetables(KHE_SOLN_GROUP soln_group, bool planning,   */
/*    bool highlight_splits, bool with_event_groups, char *constraints_str,  */
/*    HTML html, HA_ARENA_SET as)                                            */
/*                                                                           */
/*  Print timetables for the solutions of soln_group, preceded by a header.  */
/*                                                                           */
/*  Horizontal rules are printed between solutions, but not at the start     */
/*  or end of this whole print.                                              */
/*                                                                           */
/*****************************************************************************/

void SolutionGroupTimetables(KHE_SOLN_GROUP soln_group, bool planning,
  bool highlight_splits, bool with_event_groups, char *constraints_str,
  HTML html, HA_ARENA_SET as)
{
  char buff[1000];  int i;  KHE_SOLN soln;

  /* heading and solution group metadata */
  sprintf(buff, "Solution Group %s", KheSolnGroupId(soln_group));
  HTMLHeading(html, buff);
  HTMLParagraphBegin(html);
  HTMLText(html, KheSolnGroupMetaDataText(soln_group));
  HTMLParagraphEnd(html);

  /* solutions */
  for( i = 0;  i < KheSolnGroupSolnCount(soln_group);  i++ )
  {
    if( i > 0 )
      HTMLHorizontalRule(html);
    soln = KheSolnGroupSoln(soln_group, i);
    SolutionTimetables(soln, planning, highlight_splits, with_event_groups,
      constraints_str, html, as);
    if( KheSolnType(soln) == KHE_SOLN_ORDINARY )
      KheSolnTypeReduce(soln, KHE_SOLN_BASIC_PLACEHOLDER, NULL);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void Timetables(COMMAND c, bool planning, bool highlight_splits,         */
/*    bool with_event_groups)                                                */
/*                                                                           */
/*  Print some timetables.                                                   */
/*                                                                           */
/*  Horizontal rules are printed between solution groups, and at the end.    */
/*                                                                           */
/*****************************************************************************/

void Timetables(COMMAND c, bool planning, bool highlight_splits,
  bool with_event_groups, char *constraints_str, HA_ARENA_SET as)
{
  KHE_ARCHIVE archive;  char buff[1000];  HTML html;  int i;
  KHE_SOLN_GROUP soln_group;

  if( DEBUG3 )
    fprintf(stderr, "[ Timetables(c, plannning %s, highlight_splits %s, "
      "with_event_groups %s, constraints_str \"%s\")\n",
      bool_show(planning), bool_show(highlight_splits),
      bool_show(with_event_groups), constraints_str);

  /* get the archive */
  archive = ReadAndVerifyArchive(c, true, KHE_SOLN_ORDINARY, as);

  /* page header */
  snprintf(buff, 1000, "%s %s Timetables",
    KheArchiveId(archive) != NULL && strlen(KheArchiveId(archive)) < 70 ?
    KheArchiveId(archive) : "HSEval", planning ? "Planning" : "Solution");
  html = PageBegin(buff);
  HTMLBigHeading(html, buff);

  if( KheArchiveSolnGroupCount(archive) == 0 )
  {
    HTMLParagraphBegin(html);
    HTMLText(html, "The uploaded XML file contains no solution groups.");
    HTMLParagraphEnd(html);
  }
  else
  {
    /* intro paragraphs */
    if( planning )
    {
      HTMLParagraphBegin(html);
      HTMLText(html, "For each solution in the uploaded XML archive, this");
      HTMLText(html, "page contains one planning timetable for each resource");
      HTMLText(html, "type of the corresponding instance.");
      HTMLParagraphEnd(html);

      HTMLParagraphBegin(html);
      HTMLText(html, "The Avail column, where present, shows an integer");
      HTMLText(html, "number of available times, based on cluster busy");
      HTMLText(html, "times and limit busy times constraints, or a decimal");
      HTMLText(html, "amount of available workload, based on limit workload");
      HTMLText(html, "constraints.  These are heuristic estimates, and are");
      HTMLText(html, "occasionally higher than the true values, but never");
      HTMLText(html, "lower.  A negative value proves that the resource is");
      HTMLText(html, "overloaded.  In Unassigned rows the Avail column shows");
      HTMLText(html, "the unassigned times or workload.  For more information");
      HTMLText(html, "about how availability is calculated, see the");
      HTMLText(html, "Availability Report, reached from the front page");
      HTMLText(html, "of HSEval.");
      HTMLParagraphEnd(html);
    }
    else
    {
      HTMLParagraphBegin(html);
      HTMLText(html, "For each solution in the uploaded XML archive, this");
      HTMLText(html, "page contains a timetable for each resource of the");
      HTMLText(html, "corresponding instance.  If any of the constraints");
      HTMLText(html, "that apply to that resource are violated, a table of");
      HTMLText(html, "those violations is shown below its timetable.");
      HTMLParagraphEnd(html);

      if( with_event_groups )
      {
	HTMLParagraphBegin(html);
	HTMLText(html, "Similarly, a timetable is shown for each event group");
	HTMLText(html, "of the instance to which at least one constraint");
	HTMLText(html, "applies; and if any of those constraints are");
	HTMLText(html, "violated, a table of those violations is shown.");
	HTMLParagraphEnd(html);
      }
    }

    HTMLParagraphBegin(html);
    HTMLText(html, "The background colour of each timetable cell is red if");
    HTMLText(html, "the resource is unavailable then (dark red for hard");
    HTMLText(html, "constraints, light red for soft ones), or else light");
    HTMLText(html, "green if the cell contains no events, or else the");
    HTMLText(html, "colour of the first event depicted (white if the event");
    HTMLText(html, "has no colour).  If specific constraints are being");
    HTMLText(html, "highlighted, their times will appear in light blue,");
    HTMLText(html, "with defective times in blue.");
    HTMLParagraphEnd(html);

    HTMLParagraphBegin(html);
    HTMLText(html, "The text in each cell appears in bold italic font if");
    HTMLText(html, "the event resource it indicates is subject to an event");
    HTMLText(html, "resource constraint which is violated (the event");
    HTMLText(html, "resource may be unassigned when an assignment is");
    HTMLText(html, "wanted, assigned an unpreferred resource, part of");
    HTMLText(html, "a split assignment, etc.), or else in italic when");
    HTMLText(html, "unassigning the event resource would not violate an");
    HTMLText(html, "event resource constraint, or else in roman.");
    HTMLParagraphEnd(html);

    /* print solution groups */
    for( i = 0;  i < KheArchiveSolnGroupCount(archive);  i++ )
    {
      HTMLHorizontalRule(html);
      soln_group = KheArchiveSolnGroup(archive, i);
      SolutionGroupTimetables(soln_group, planning, highlight_splits,
	with_event_groups, constraints_str, html, as);
    }
  }

  /* print a back jump link */
  HTMLParagraphBegin(html);
  HTMLText(html, "Return to the ");
  HTMLJumpFront(html);
  HTMLText(html, ".");
  HTMLParagraphEnd(html);

  /* and quit */
  PageEnd(html);
  if( DEBUG3 )
    fprintf(stderr, "] Timetables\n");
}
