
/*****************************************************************************/
/*                                                                           */
/*  THE NRCONV NURSE ROSTERING TO XHSTT CONVERTER                            */
/*  COPYRIGHT (C) 2016, Jeffrey H. Kingston                                  */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 3, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         inrc2.c                                                    */
/*  MODULE:       ConvertINRC2 (second international nurse rostering comp)   */
/*                                                                           */
/*****************************************************************************/
#include "externs.h"
#define DEBUG1 0
#define DEBUG2 0
#define DEBUG3 0
#define DEBUG4 0
#define DEBUG5 0
#define DEBUG6 0
#define DEBUG7 1
#define DEBUG9 0


/*****************************************************************************/
/*                                                                           */
/*  Note on constraints                                                      */
/*  -------------------                                                      */
/*                                                                           */
/*  Here is the relationship between what the competition documentation      */
/*  says about constraints, and the functions of this file:                  */
/*                                                                           */
/*  Competition documentation             Functions in this file       X  Y  */
/*  -----------------------------------------------------------------------  */
/*  H1. Single assignment per day         SingleAssignmentPerDay       -  -  */
/*  H2. Under-staffing                    AddCoverConstraints                */
/*  H3. Shift type successions            UnwantedPattern              -  H  */
/*  H4. Missing required skill            AddCoverConstraints                */
/*  -----------------------------------------------------------------------  */
/*  S1. Insuff staff for opt cover (30)   AddCoverConstraints                */
/*  S2. Consecutive assignments (15/30)   MinConsecutiveSameShiftDays  S  H  */
/*                                        MaxConsecutiveSameShiftDays  S  H  */
/*                                        MinConsecutiveWorkingDays    C  H  */
/*                                        MaxConsecutiveWorkingDays    C  H  */
/*  S3. Consecutive days off (30)         MinConsecutiveDaysOff        C  H  */
/*                                        MaxConsecutiveDaysOff        C  H  */
/*  S4. Preferences (10)                  AddShiftOffRequests          -  -  */
/*  S5. Complete week-end (30)            CompleteWeekends             C  -  */
/*  S6. Total assignments (20)            MinAssignments               C  H  */
/*                                        MaxAssignments               C  H  */
/*  S7. Total working week-ends (30)      MaxWorkingWeekends           C  H  */
/*  -----------------------------------------------------------------------  */
/*                                                                           */
/*  Hard constraints H1-4 have no weights; this code gives them weight 1     */
/*  throughout.  Soft constraints S1-7 have the weights shown.  The limits   */
/*  vary as indicated by the X column:                                       */
/*                                                                           */
/*     -    Trivial limit (1, pattern length - 1, etc.)                      */
/*     R    Limit stored in (so varies with) Requirement in weekly file      */
/*     S    Limit stored in (so varies with) Shift Type in scenario file     */
/*     C    Limit stored in (so varies with) Contract in scenario file       */
/*                                                                           */
/*  The constraints are affected by history as indicated in the Y column:    */
/*                                                                           */
/*     -    Unaffected by history                                            */
/*     H    Affected by history                                              */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  INRC2_INFO - miscellaneous information about a inrc2 instance            */
/*                                                                           */
/*****************************************************************************/

typedef struct inrc2_info_rec {
  HA_ARENA			arena;
  char				*sc_file_name;
  KML_ELT			sc_root_elt;
  char				*hy_file_name;
  KML_ELT			hy_root_elt;
  char				*wk_file_name;
  KML_ELT			wk_root_elt;
  NRC_INSTANCE			ins;
  NRC_DAY_SET			weekend;
  int				number_of_weeks;
  int				week_index;
  int				days_before;
  int				days_after;
} *INRC2_INFO;


/*****************************************************************************/
/*                                                                           */
/*  Submodule "cycle"                                                        */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void AddDaysAndWeekend(INRC2_INFO ii)                                    */
/*                                                                           */
/*  Add the cycle (it begins on a Monday) and the sole weekend.              */
/*                                                                           */
/*****************************************************************************/

static void AddDaysAndWeekend(INRC2_INFO ii)
{
  NrcCycleMake(ii->ins, 7, 1);
  ii->weekend = NrcDaySetMake(ii->ins, "Weekend", "Weekend");
  NrcDaySetAddDay(ii->weekend, NrcInstanceCycleDay(ii->ins, 5));
  NrcDaySetAddDay(ii->weekend, NrcInstanceCycleDay(ii->ins, 6));
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "skills"                                                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void AddSkills(INRC2_INFO ii)                                            */
/*                                                                           */
/*  Add skills to ii.                                                        */
/*                                                                           */
/*****************************************************************************/

static void AddSkills(INRC2_INFO ii)
{
  KML_ELT skills_elt, skill_elt;  int i;  KML_ERROR ke;  /*  SKILL skill; */
  NRC_WORKER_SET skill_ws;  char *name;
  if( DEBUG3 )
    fprintf(stderr, "[ AddSkills(ii)\n");
  if( KmlContainsChild(ii->sc_root_elt, "Skills", &skills_elt) )
  {
    if( !KmlCheck(skills_elt, ": *$Skill", &ke) )
      KmlFatalError(ke, ii->sc_file_name);
    for( i = 0;  i < KmlChildCount(skills_elt);  i++ )
    {
      skill_elt = KmlChild(skills_elt, i);
      name = KmlText(skill_elt);
      if( NrcInstanceSkillsRetrieveSkill(ii->ins, name, &skill_ws) )
	KmlEltFatalError(skill_elt, "skill name %s appears twice", name);
      skill_ws = NrcWorkerSetMake(ii->ins, name);
      NrcInstanceSkillsAddSkill(ii->ins, skill_ws);
    }
  }
  if( DEBUG3 )
    fprintf(stderr, "] AddSkills returning\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "shift types and shift type constraints"                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void AddSameShiftDaysHistory(NRC_CONSTRAINT c, NRC_SHIFT_TYPE st,        */
/*    INRC2_INFO ii)                                                         */
/*                                                                           */
/*  Add suitable history information to constraint c, which limits the       */
/*  number of consecutive assignments to shift type st.                      */
/*                                                                           */
/*****************************************************************************/

static void AddSameShiftDaysHistory(NRC_CONSTRAINT c, NRC_SHIFT_TYPE st,
  INRC2_INFO ii)
{
  int i, v;  NRC_WORKER_SET ws;  NRC_WORKER w;  NRC_SHIFT_TYPE last_st;
  NrcConstraintAddHistory(c, ii->days_before, ii->days_after);
  ws = NrcConstraintWorkerSet(c);
  for( i = 0;  i < NrcWorkerSetWorkerCount(ws);  i++ )
  {
    w = NrcWorkerSetWorker(ws, i);
    if( NrcWorkerRetrieveHistory(w, "LastAssignedShiftType", &v) )
    {
      last_st = NrcInstanceShiftType(NrcConstraintInstance(c), v);
      if( last_st == st &&
	  NrcWorkerRetrieveHistory(w, "NumberOfConsecutiveAssignments", &v)
	  && v > 0 )
	NrcConstraintAddHistoryWorker(c, w, v);
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void MinConsecutiveSameShiftDays(INRC2_INFO ii, KML_ELT contracts_elt)   */
/*                                                                           */
/*  Minimum consecutive same shift days.                                     */
/*                                                                           */
/*****************************************************************************/

static void MinConsecutiveSameShiftDays(INRC2_INFO ii, NRC_SHIFT_TYPE st,
  int limit)
{
  NRC_CONSTRAINT c;  NRC_WORKER_SET ws;  NRC_PENALTY p;
  if( DEBUG6 )
    fprintf(stderr, "[ MinConsecutiveSameShiftDays(ii, %s, %d)\n",
      NrcShiftTypeName(st), limit);
  ws = NrcInstanceStaffing(ii->ins);
  p = NrcPenalty(false, 15, NRC_COST_FUNCTION_LINEAR, ii->ins);
  c = NrcConstraintMake(ii->ins, "MinConsecutiveSameShiftDays", ws,
    NRC_CONSTRAINT_CONSECUTIVE, NrcBoundMakeMin(limit, false, p, ii->ins),
    NULL);
  /* ***
  c = NrcConstraintMake(ii->ins, "MinConsecutiveSameShiftDays", ws, p,
    NrcLimit(NRC_LIMIT_MIN_CONSECUTIVE, limit), NULL);
  *** */
  NrcConstraintAddShiftSetSet(c, NrcShiftTypeShiftSetSet(st), NRC_POSITIVE);
  AddSameShiftDaysHistory(c, st, ii);
  if( DEBUG6 )
    fprintf(stderr, "] MinConsecutiveSameShiftDays returning\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void MaxConsecutiveSameShiftDays(INRC2_INFO ii, KML_ELT contracts_elt)   */
/*                                                                           */
/*  Maximum consecutive same shift days.                                     */
/*                                                                           */
/*****************************************************************************/

static void MaxConsecutiveSameShiftDays(INRC2_INFO ii, NRC_SHIFT_TYPE st,
  int limit)
{
  NRC_CONSTRAINT c;  NRC_WORKER_SET ws;  NRC_PENALTY p;
  ws = NrcInstanceStaffing(ii->ins);
  p = NrcPenalty(false, 15, NRC_COST_FUNCTION_LINEAR, ii->ins);
  c = NrcConstraintMake(ii->ins, "MaxConsecutiveSameShiftDays", ws,
    NRC_CONSTRAINT_CONSECUTIVE, NrcBoundMakeMax(limit, p, ii->ins), NULL);
  /* ***
  c = NrcConstraintMake(ii->ins, "MaxConsecutiveSameShiftDays", ws, p,
    NrcLimit(NRC_LIMIT_MAX_CONSECUTIVE, limit), NULL);
  *** */
  NrcConstraintAddShiftSetSet(c, NrcShiftTypeShiftSetSet(st), NRC_POSITIVE);
  AddSameShiftDaysHistory(c, st, ii);
}


/*****************************************************************************/
/*                                                                           */
/*  void AddShiftTypes(INRC2_INFO ii)                                        */
/*                                                                           */
/*  Add shift types to ii.                                                   */
/*                                                                           */
/*****************************************************************************/

static void AddShiftTypes(INRC2_INFO ii)
{
  KML_ELT shift_types_elt, shift_type_elt, consec_elt, min_elt, max_elt;
  KML_ERROR ke;  int i, min_consec, max_consec;  char *id;
  if( DEBUG9 )
    fprintf(stderr, "[ AddShiftTypes(ii)\n");
  if( KmlContainsChild(ii->sc_root_elt, "ShiftTypes", &shift_types_elt) )
  {
    if( !KmlCheck(shift_types_elt, ": *ShiftType", &ke) )
      KmlFatalError(ke, ii->sc_file_name);
    for( i = 0;  i < KmlChildCount(shift_types_elt);  i++ )
    {
      /* find one shift type and its id */
      shift_type_elt = KmlChild(shift_types_elt, i);
      if( !KmlCheck(shift_type_elt, "Id : NumberOfConsecutiveAssignments",&ke) )
	KmlFatalError(ke, ii->sc_file_name);
      id = KmlAttributeValue(shift_type_elt, 0);

      /* find min_consecutive and max_consecutive */
      consec_elt = KmlChild(shift_type_elt, 0);
      if( !KmlCheck(consec_elt, ": #Minimum #Maximum", &ke) )
	KmlFatalError(ke, ii->sc_file_name);
      min_elt = KmlChild(consec_elt, 0);
      if( sscanf(KmlText(min_elt), "%d", &min_consec) != 1 )
	HnAbort("AddShiftTypes internal error 1");
      max_elt = KmlChild(consec_elt, 1);
      if( sscanf(KmlText(max_elt), "%d", &max_consec) != 1 )
	HnAbort("AddShiftTypes internal error 2");

      /* make the shift type */
      NrcShiftTypeMake(ii->ins, id, NRC_NO_WORKLOAD);
    }
  }
  if( DEBUG9 )
    fprintf(stderr, "] AddShiftTypes returning\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void AddShiftTypeConstraints(INRC2_INFO ii)                              */
/*                                                                           */
/*  Add shift type constraints to ii.                                        */
/*                                                                           */
/*****************************************************************************/

static void AddShiftTypeConstraints(INRC2_INFO ii)
{
  KML_ELT shift_types_elt, shift_type_elt, consec_elt, min_elt, max_elt;
  KML_ERROR ke;  int i, min_consec, max_consec;  char *id;  NRC_SHIFT_TYPE st;
  if( DEBUG9 )
    fprintf(stderr, "[ AddShiftTypeConstraints(ii)\n");
  if( KmlContainsChild(ii->sc_root_elt, "ShiftTypes", &shift_types_elt) )
  {
    if( !KmlCheck(shift_types_elt, ": *ShiftType", &ke) )
      KmlFatalError(ke, ii->sc_file_name);
    for( i = 0;  i < KmlChildCount(shift_types_elt);  i++ )
    {
      /* find one shift type and its id */
      shift_type_elt = KmlChild(shift_types_elt, i);
      if( !KmlCheck(shift_type_elt, "Id : NumberOfConsecutiveAssignments",&ke) )
	KmlFatalError(ke, ii->sc_file_name);
      id = KmlAttributeValue(shift_type_elt, 0);

      /* find min_consecutive and max_consecutive */
      consec_elt = KmlChild(shift_type_elt, 0);
      if( !KmlCheck(consec_elt, ": #Minimum #Maximum", &ke) )
	KmlFatalError(ke, ii->sc_file_name);
      min_elt = KmlChild(consec_elt, 0);
      if( sscanf(KmlText(min_elt), "%d", &min_consec) != 1 )
	HnAbort("AddShiftTypes internal error 1");
      max_elt = KmlChild(consec_elt, 1);
      if( sscanf(KmlText(max_elt), "%d", &max_consec) != 1 )
	HnAbort("AddShiftTypes internal error 2");

      /* make the shift type and its associated constraints */
      if( !NrcInstanceRetrieveShiftType(ii->ins, id, &st) )
	HnAbort("AddShiftTypeConstraints internal error (%s)", id);
      MinConsecutiveSameShiftDays(ii, st, min_consec);
      MaxConsecutiveSameShiftDays(ii, st, max_consec);
    }
  }
  if( DEBUG9 )
    fprintf(stderr, "] AddShiftTypeConstraints returning\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "patterns and unwanted pattern constraints"                    */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void UnwantedPatterns(INRC2_INFO ii)                                     */
/*                                                                           */
/*  Unwanted patterns.  NB the pattern objects are already built.            */
/*                                                                           */
/*****************************************************************************/

static void UnwantedPattern(INRC2_INFO ii, NRC_PATTERN p)
{
  NRC_CONSTRAINT c;  NRC_PENALTY penalty;
  penalty = NrcPenalty(true, 1, NRC_COST_FUNCTION_LINEAR, ii->ins);
  c = NrcConstraintMake(ii->ins, "UnwantedPattern",
    NrcInstanceStaffing(ii->ins), NRC_CONSTRAINT_ACTIVE,
    NrcBoundMakeMax(NrcPatternTermCount(p) - 1, penalty, ii->ins),
    NrcInstanceDailyStartingShiftSet(ii->ins));
  /* ***
  c = NrcConstraintMake(ii->ins, "UnwantedPattern",
    NrcInstanceStaffing(ii->ins), penalty,
    NrcLimit(NRC_LIMIT_MAX, NrcPatternTermCount(p) - 1),
    NrcInstanceDailyStartingShiftSet(ii->ins));
  *** */
  NrcConstraintAddPattern(c, p, NrcInstanceCycleDay(ii->ins, 0));
}


/*****************************************************************************/
/*                                                                           */
/*  void AddPatternsAndUnwantedPatternConstraints(INRC2_INFO ii)             */
/*                                                                           */
/*  Add patterns (here called forbidden shift type successions) to ii.       */
/*  Also add one unwanted pattern constraint for each pattern.               */
/*                                                                           */
/*****************************************************************************/

static void AddPatternsAndUnwantedPatternConstraints(INRC2_INFO ii)
{
  KML_ELT patterns_elt, pattern_elt, prec_elt, succs_elt, succ_elt;
  KML_ERROR ke;  int i, j;  NRC_SHIFT_TYPE_SET sts;  NRC_PATTERN p;
  NRC_SHIFT_TYPE st;
  if( KmlContainsChild(ii->sc_root_elt, "ForbiddenShiftTypeSuccessions",
	&patterns_elt) )
  {
    HnAssert(NrcInstanceCycleDayCount(ii->ins) >= 2,
      "AddPatterns: less than 2 days in cycle");
    if( !KmlCheck(patterns_elt, ": *ShiftTypeSuccession", &ke) )
      KmlFatalError(ke, ii->sc_file_name);
    for( i = 0;  i < KmlChildCount(patterns_elt);  i++ )
    {
      /* get one pattern element and make a pattern for it */
      pattern_elt = KmlChild(patterns_elt, i);
      if( !KmlCheck(pattern_elt, ": $PrecedingShiftType SucceedingShiftTypes",
	    &ke) )
	KmlFatalError(ke, ii->sc_file_name);

      /* get the two children */
      prec_elt = KmlChild(pattern_elt, 0);
      succs_elt = KmlChild(pattern_elt, 1);
      if( !KmlCheck(succs_elt, ": *$ShiftType", &ke) )
	KmlFatalError(ke, ii->sc_file_name);

      if( KmlChildCount(succs_elt) > 0 )
      {
	/* a pattern is needed, so make one */
	p = NrcPatternMake(ii->ins, NULL);

	/* get preceding element and add a term for that */
	if( !NrcInstanceRetrieveShiftType(ii->ins, KmlText(prec_elt), &st) )
	  KmlEltFatalError(prec_elt, "unknown shift type %s",KmlText(prec_elt));
	sts = NrcShiftTypeSingletonShiftTypeSet(st);
	NrcPatternAddTerm(p, sts, NRC_POSITIVE);

	/* get succeeding element and add a term for that */
	sts = NrcShiftTypeSetMake(ii->ins, NULL);
	for( j = 0;  j < KmlChildCount(succs_elt);  j++ )
	{
	  succ_elt = KmlChild(succs_elt, j);
	  if( !NrcInstanceRetrieveShiftType(ii->ins, KmlText(succ_elt), &st) )
	   KmlEltFatalError(succ_elt,"unknown shift type %s",KmlText(succ_elt));
	  NrcShiftTypeSetAddShiftType(sts, st);
	}
	NrcPatternAddTerm(p, sts, NRC_POSITIVE);

	/* add an unwanted pattern constraint for p */
        UnwantedPattern(ii, p);
      }
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "contracts and contract constraints"                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool KmlTextInteger(INRC2_INFO ii, KML_ELT elt, int *val)                */
/*                                                                           */
/*  Make sure that the text of elt is an integer, set *val to it, and        */
/*  return true.                                                             */
/*                                                                           */
/*****************************************************************************/

static bool KmlTextInteger(INRC2_INFO ii, KML_ELT elt, int *val)
{
  if( sscanf(KmlText(elt), "%d", val) != 1 )
    KmlEltFatalError(elt, ii->sc_file_name, "integer expected here");
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  bool ContractHasLimitConstraint(INRC2_INFO ii, KML_ELT contract_elt,     */
/*    char *label1, char *label2, int *limit)                                */
/*                                                                           */
/*  Return true and set *limit to the limit if contract_elt has a limit      */
/*  constraint with label label1 and optionally label2.                      */
/*                                                                           */
/*****************************************************************************/

static bool ContractHasLimitConstraint(INRC2_INFO ii, KML_ELT contract_elt,
  char *label1, char *label2, int *limit)
{
  KML_ELT label1_elt, label2_elt;
  if( KmlContainsChild(contract_elt, label1, &label1_elt) )
  {
    /* find label2_elt, which is just label1_elt again if label2 is NULL */
    if( label2 == NULL )
      label2_elt = label1_elt;
    else if( !KmlContainsChild(label1_elt, label2, &label2_elt) )
    {
      *limit = -1;
      return false;
    }

    /* return the limit */
    return KmlTextInteger(ii, label2_elt, limit);
  }
  else
  {
    *limit = -1;
    return false;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void MinAssignments(INRC2_INFO ii, KML_ELT contracts_elt)                */
/*                                                                           */
/*  Minimum number of assignments.                                           */
/*                                                                           */
/*  Implementation note.  This code follows MaxAssignments except that it    */
/*  is possible for the constraint to be already satisfied by previous       */
/*  weeks, in which case the constraint is omitted.                          */
/*                                                                           */
/*****************************************************************************/

static void MinAssignments(INRC2_INFO ii, KML_ELT contract_elt,
  NRC_WORKER_SET contract_ws)
{
  int limit;  NRC_CONSTRAINT c;  NRC_PENALTY p;
  if( ContractHasLimitConstraint(ii, contract_elt, "NumberOfAssignments",
       "Minimum", &limit) )
  {
    p = NrcPenalty(false, 20, NRC_COST_FUNCTION_LINEAR, ii->ins);
    c = NrcConstraintMake(ii->ins, "MinAssignments", contract_ws,
      NRC_CONSTRAINT_ACTIVE, NrcBoundMakeMin(limit, false, p, ii->ins), NULL);
    /* ***
    c = NrcConstraintMake(ii->ins, "MinAssignments", contract_ws, p,
      NrcLimit(NRC_LIMIT_MIN, limit), NULL);
    *** */
    NrcConstraintAddShiftSetSet(c, NrcInstanceShiftsShiftSetSet(ii->ins),
      NRC_POSITIVE);
    NrcConstraintAddHistoryAllWorkers(c, ii->days_before, ii->days_after,
      "NumberOfAssignments");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void MaxAssignments(INRC2_INFO ii, KML_ELT contracts_elt)                */
/*                                                                           */
/*  Maximum number of assignments.                                           */
/*                                                                           */
/*****************************************************************************/

static void MaxAssignments(INRC2_INFO ii, KML_ELT contract_elt,
  NRC_WORKER_SET contract_ws)
{
  int limit;  NRC_CONSTRAINT c;  NRC_PENALTY p;
  if( ContractHasLimitConstraint(ii, contract_elt, "NumberOfAssignments",
       "Maximum", &limit) )
  {
    p = NrcPenalty(false, 20, NRC_COST_FUNCTION_LINEAR, ii->ins);
    c = NrcConstraintMake(ii->ins, "MaxAssignments", contract_ws,
      NRC_CONSTRAINT_ACTIVE, NrcBoundMakeMax(limit, p, ii->ins), NULL);
    /* ***
    c = NrcConstraintMake(ii->ins, "MaxAssignments", contract_ws, p,
      NrcLimit(NRC_LIMIT_MAX, limit), NULL);
    *** */
    NrcConstraintAddShiftSetSet(c, NrcInstanceShiftsShiftSetSet(ii->ins),
      NRC_POSITIVE);
    NrcConstraintAddHistoryAllWorkers(c, ii->days_before, ii->days_after,
      "NumberOfAssignments");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void MinConsecutiveWorkingDays(INRC2_INFO ii, KML_ELT contract_elt)      */
/*                                                                           */
/*  Minimum consecutive working days.                                        */
/*                                                                           */
/*****************************************************************************/

static void MinConsecutiveWorkingDays(INRC2_INFO ii, KML_ELT contract_elt,
  NRC_WORKER_SET contract_ws)
{
  int limit;  NRC_CONSTRAINT c;  NRC_PENALTY p;
  if( ContractHasLimitConstraint(ii, contract_elt, "ConsecutiveWorkingDays",
	"Minimum", &limit) )
  {
    p = NrcPenalty(false, 30, NRC_COST_FUNCTION_LINEAR, ii->ins);
    c = NrcConstraintMake(ii->ins, "MinConsecutiveWorkingDays", contract_ws,
      NRC_CONSTRAINT_CONSECUTIVE, NrcBoundMakeMin(limit, false, p, ii->ins),
      NULL);
    /* ***
    c = NrcConstraintMake(ii->ins, "MinConsecutiveWorkingDays", contract_ws,
      p, NrcLimit(NRC_LIMIT_MIN_CONSECUTIVE, limit), NULL);
    *** */
    NrcConstraintAddShiftSetSet(c, NrcInstanceDaysShiftSetSet(ii->ins),
      NRC_POSITIVE);
    NrcConstraintAddHistoryAllWorkers(c, ii->days_before, ii->days_after,
      "NumberOfConsecutiveWorkingDays");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void MaxConsecutiveWorkingDays(INRC2_INFO ii, KML_ELT contracts_elt)     */
/*                                                                           */
/*  Maximum consecutive working days.                                        */
/*                                                                           */
/*****************************************************************************/

static void MaxConsecutiveWorkingDays(INRC2_INFO ii, KML_ELT contract_elt,
  NRC_WORKER_SET contract_ws)
{
  int limit;  NRC_CONSTRAINT c;  NRC_PENALTY p;
  if( ContractHasLimitConstraint(ii, contract_elt, "ConsecutiveWorkingDays",
	"Maximum", &limit) )
  {
    p = NrcPenalty(false, 30, NRC_COST_FUNCTION_LINEAR, ii->ins);
    c = NrcConstraintMake(ii->ins, "MaxConsecutiveWorkingDays", contract_ws,
      NRC_CONSTRAINT_CONSECUTIVE, NrcBoundMakeMax(limit, p, ii->ins), NULL);
    /* ***
    c = NrcConstraintMake(ii->ins, "MaxConsecutiveWorkingDays", contract_ws,
      p, NrcLimit(NRC_LIMIT_MAX_CONSECUTIVE, limit), NULL);
    *** */
    NrcConstraintAddShiftSetSet(c, NrcInstanceDaysShiftSetSet(ii->ins),
      NRC_POSITIVE);
    NrcConstraintAddHistoryAllWorkers(c, ii->days_before, ii->days_after,
      "NumberOfConsecutiveWorkingDays");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void MinConsecutiveDaysOff(INRC2_INFO ii, KML_ELT contracts_elt)         */
/*                                                                           */
/*  Minimum consecutive days off.                                            */
/*                                                                           */
/*****************************************************************************/

static void MinConsecutiveDaysOff(INRC2_INFO ii, KML_ELT contract_elt,
  NRC_WORKER_SET contract_ws)
{
  int limit;  NRC_CONSTRAINT c;  NRC_PENALTY p;
  if( ContractHasLimitConstraint(ii, contract_elt, "ConsecutiveDaysOff",
	"Minimum", &limit) )
  {
    p = NrcPenalty(false, 30, NRC_COST_FUNCTION_LINEAR, ii->ins);
    c = NrcConstraintMake(ii->ins, "MinConsecutiveFreeDays", contract_ws,
      NRC_CONSTRAINT_CONSECUTIVE, NrcBoundMakeMin(limit, false, p, ii->ins),
      NULL);
    /* ***
    c = NrcConstraintMake(ii->ins, "MinConsecutiveFreeDays", contract_ws,
      p, NrcLimit(NRC_LIMIT_MIN_CONSECUTIVE, limit), NULL);
    *** */
    NrcConstraintAddShiftSetSet(c, NrcInstanceDaysShiftSetSet(ii->ins),
      NRC_NEGATIVE);
    NrcConstraintAddHistoryAllWorkers(c, ii->days_before, ii->days_after,
      "NumberOfConsecutiveDaysOff");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void MaxConsecutiveDaysOff(INRC2_INFO ii, KML_ELT contracts_elt)         */
/*                                                                           */
/*  Maximum consecutive days off.                                            */
/*                                                                           */
/*****************************************************************************/

static void MaxConsecutiveDaysOff(INRC2_INFO ii, KML_ELT contract_elt,
  NRC_WORKER_SET contract_ws)
{
  int limit;  NRC_CONSTRAINT c;  NRC_PENALTY p;
  if( ContractHasLimitConstraint(ii, contract_elt, "ConsecutiveDaysOff",
	"Maximum", &limit) )
  {
    p = NrcPenalty(false, 30, NRC_COST_FUNCTION_LINEAR, ii->ins);
    c = NrcConstraintMake(ii->ins, "MaxConsecutiveDaysOff", contract_ws,
      NRC_CONSTRAINT_CONSECUTIVE, NrcBoundMakeMax(limit, p, ii->ins), NULL);
    /* ***
    c = NrcConstraintMake(ii->ins, "MaxConsecutiveDaysOff", contract_ws,
      p, NrcLimit(NRC_LIMIT_MAX_CONSECUTIVE, limit), NULL);
    *** */
    NrcConstraintAddShiftSetSet(c, NrcInstanceDaysShiftSetSet(ii->ins),
      NRC_NEGATIVE);
    NrcConstraintAddHistoryAllWorkers(c, ii->days_before, ii->days_after,
      "NumberOfConsecutiveDaysOff");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void MaxWorkingWeekends(INRC2_INFO ii, KML_ELT contracts_elt)            */
/*                                                                           */
/*  Maxiumum number of working weekends.                                     */
/*                                                                           */
/*****************************************************************************/

static void MaxWorkingWeekends(INRC2_INFO ii, KML_ELT contract_elt,
  NRC_WORKER_SET contract_ws)
{
  int limit;  NRC_CONSTRAINT c;  NRC_PENALTY p;
  if( ContractHasLimitConstraint(ii, contract_elt,
	"MaximumNumberOfWorkingWeekends", NULL, &limit) )
  {
    p = NrcPenalty(false, 30, NRC_COST_FUNCTION_LINEAR, ii->ins);
    c = NrcConstraintMake(ii->ins, "MaxWorkingWeekends", contract_ws,
      NRC_CONSTRAINT_ACTIVE, NrcBoundMakeMax(limit, p, ii->ins), NULL);
    /* ***
    c = NrcConstraintMake(ii->ins, "MaxWorkingWeekends", contract_ws,
      p, NrcLimit(NRC_LIMIT_MAX, limit), NULL);
    *** */
    NrcConstraintAddShiftSet(c, NrcDaySetShiftSet(ii->weekend), NRC_POSITIVE);
    NrcConstraintAddHistoryAllWorkers(c, ii->days_before, ii->days_after,
      "NumberOfWorkingWeekends");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void CompleteWeekends(INRC2_INFO ii, KML_ELT contracts_elt)              */
/*                                                                           */
/*  Complete weekends.                                                       */
/*                                                                           */
/*****************************************************************************/

static void CompleteWeekends(INRC2_INFO ii, KML_ELT contract_elt,
  NRC_WORKER_SET contract_ws)
{
  int limit, i;  NRC_CONSTRAINT c;  NRC_DAY d;  NRC_PENALTY p;
  if( ContractHasLimitConstraint(ii, contract_elt, "CompleteWeekends",
	NULL, &limit) && limit >= 1 )
  {
    limit = NrcDaySetDayCount(ii->weekend);
    p = NrcPenalty(false, 30, NRC_COST_FUNCTION_LINEAR, ii->ins);
    c = NrcConstraintMake(ii->ins, "CompleteWeekends", contract_ws,
      NRC_CONSTRAINT_ACTIVE, NrcBoundMakeMin(limit, true, p, ii->ins), NULL);
    /* ***
    c = NrcConstraintMake(ii->ins, "CompleteWeekends", contract_ws, p,
      NrcLimit(NRC_LIMIT_MIN_OR_ZERO, limit), NULL);
    *** */
    for( i = 0;  i < limit;  i++ )
    {
      d = NrcDaySetDay(ii->weekend, i);
      NrcConstraintAddShiftSet(c, NrcDayShiftSet(d), NRC_POSITIVE);
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void AddContracts(INRC2_INFO ii)                                         */
/*                                                                           */
/*  Add contracts.                                                           */
/*                                                                           */
/*****************************************************************************/

static void AddContracts(INRC2_INFO ii)
{
  KML_ELT contracts_elt, contract_elt;  KML_ERROR ke;
  int i;  char *id;  NRC_WORKER_SET contract_ws;

  if( KmlContainsChild(ii->sc_root_elt, "Contracts", &contracts_elt) )
  {
    if( !KmlCheck(contracts_elt, ": *Contract", &ke) )
      KmlFatalError(ke, ii->sc_file_name);
    for( i = 0;  i < KmlChildCount(contracts_elt);  i++ )
    {
      /* check syntax of one contract */
      contract_elt = KmlChild(contracts_elt, i);
      if( !KmlCheck(contract_elt, "Id : NumberOfAssignments "
	  "ConsecutiveWorkingDays ConsecutiveDaysOff "
	  "MaximumNumberOfWorkingWeekends CompleteWeekends", &ke) )
	KmlFatalError(ke, ii->sc_file_name);

      /* create worker set for this contract */
      id = HnStringMake(ii->arena, "Contract-%s",
	KmlAttributeValue(contract_elt, 0));
      contract_ws = NrcWorkerSetMake(ii->ins, id);
      NrcInstanceContractsAddContract(ii->ins, contract_ws);
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void AddContractConstraints(INRC2_INFO ii)                               */
/*                                                                           */
/*  Add contract constraints.                                                */
/*                                                                           */
/*****************************************************************************/

static void AddContractConstraints(INRC2_INFO ii)
{
  KML_ELT contracts_elt, contract_elt;  KML_ERROR ke;
  int i;  char *id;  NRC_WORKER_SET contract_ws;

  if( KmlContainsChild(ii->sc_root_elt, "Contracts", &contracts_elt) )
  {
    if( !KmlCheck(contracts_elt, ": *Contract", &ke) )
      KmlFatalError(ke, ii->sc_file_name);
    for( i = 0;  i < KmlChildCount(contracts_elt);  i++ )
    {
      /* check syntax of one contract */
      contract_elt = KmlChild(contracts_elt, i);
      if( !KmlCheck(contract_elt, "Id : NumberOfAssignments "
	  "ConsecutiveWorkingDays ConsecutiveDaysOff "
	  "MaximumNumberOfWorkingWeekends CompleteWeekends", &ke) )
	KmlFatalError(ke, ii->sc_file_name);

      /* create worker set for this contract */
      id = HnStringMake(ii->arena, "Contract-%s",
	KmlAttributeValue(contract_elt, 0));
      if( !NrcInstanceContractsRetrieveContract(ii->ins, id, &contract_ws) )
	HnAbort("AddContractConstraints internal error");

      /* ConsecutiveWorkingDays (S2) */
      MinConsecutiveWorkingDays(ii, contract_elt, contract_ws);
      MaxConsecutiveWorkingDays(ii, contract_elt, contract_ws);

      /* ConsecutiveDaysOff (S3) */
      MinConsecutiveDaysOff(ii, contract_elt, contract_ws);
      MaxConsecutiveDaysOff(ii, contract_elt, contract_ws);

      /* CompleteWeekends (S5) */
      CompleteWeekends(ii, contract_elt, contract_ws);

      /* NumberOfAssignments (S6) */
      MinAssignments(ii, contract_elt, contract_ws);
      MaxAssignments(ii, contract_elt, contract_ws);

      /* MaximumNumberOfWorkingWeekends (S7) */
      MaxWorkingWeekends(ii, contract_elt, contract_ws);
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "workers"                                                      */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_RESOURCE_GROUP RetrieveContractResourceGroup(INRC2_INFO ii,          */
/*    KML_ELT elt, char *id)                                                 */
/*                                                                           */
/*  Return the resource group for the contract with this id.  The elt        */
/*  parameter is only for error messages.                                    */
/*                                                                           */
/*****************************************************************************/

static NRC_WORKER_SET RetrieveContractWorkerSet(INRC2_INFO ii,
  KML_ELT elt, char *id)
{
  NRC_WORKER_SET ws;  char *contract_id;
  contract_id = HnStringMake(ii->arena, "Contract-%s", id);
  if( !NrcInstanceContractsRetrieveContract(ii->ins, contract_id, &ws) )
    KmlEltFatalError(elt, ii->sc_file_name, "unknown contract %s", contract_id);
  return ws;
}


/*****************************************************************************/
/*                                                                           */
/*  void AddNurses(INRC2_INFO ii)                                            */
/*                                                                           */
/*  Add one worker for each nurse, including adding them to their skills     */
/*  and contract resource groups.                                            */
/*                                                                           */
/*****************************************************************************/

static void AddNurses(INRC2_INFO ii)
{
  KML_ELT nurses_elt, nurse_elt, skills_elt, skill_elt, contract_elt;
  KML_ERROR ke; int i, j;  char *nurse_id;
  NRC_WORKER w;  NRC_WORKER_SET ws;

  /* record everything in the Nurses child */
  if( DEBUG4 )
    fprintf(stderr, "[ AddNurses(ii)\n");
  KmlContainsChild(ii->sc_root_elt, "Nurses", &nurses_elt);
  if( !KmlCheck(nurses_elt, ": *Nurse", &ke) )
    KmlFatalError(ke, ii->sc_file_name);
  for( i = 0;  i < KmlChildCount(nurses_elt);  i++ )
  {
    /* sort out names and create one resource for nurse_elt */
    nurse_elt = KmlChild(nurses_elt, i);
    if( !KmlCheck(nurse_elt, "Id : $Contract Skills", &ke) )
      KmlFatalError(ke, ii->sc_file_name);
    nurse_id = KmlAttributeValue(nurse_elt, 0);

    w = NrcWorkerMake(ii->ins, nurse_id);

    /* add w to its contract worker set */
    contract_elt = KmlChild(nurse_elt, 0);
    ws = RetrieveContractWorkerSet(ii, contract_elt, KmlText(contract_elt));
    NrcWorkerSetAddWorker(ws, w);

    /* add resource to skills */
    skills_elt = KmlChild(nurse_elt, 1);
    if( !KmlCheck(skills_elt, ": *$Skill", &ke) )
      KmlFatalError(ke, ii->sc_file_name);
    for( j = 0;  j < KmlChildCount(skills_elt);  j++ )
    {
      skill_elt = KmlChild(skills_elt, j);
      if( !NrcInstanceSkillsRetrieveSkill(ii->ins, KmlText(skill_elt), &ws) )
	KmlEltFatalError(skill_elt, ii->sc_file_name, "unknown skill %s",
	  KmlText(skill_elt));
      NrcWorkerSetAddWorker(ws, w);
    }
  }
  if( DEBUG4 )
    fprintf(stderr, "] AddNurses\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "history"                                                      */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void AddNursesHistoryAndInitialUnwantedPatterns(INRC2_INFO ii)           */
/*                                                                           */
/*  Add history information for the resources of the instance.  Also add     */
/*  initial unwanted pattern constraints, the ones that end on the first     */
/*  day of the cycle.                                                        */
/*                                                                           */
/*****************************************************************************/

static void AddNursesHistoryAndInitialUnwantedPatterns(INRC2_INFO ii)
{
  KML_ERROR ke;  int i, j;  NRC_WORKER w;  NRC_PATTERN p;  char *val;
  NRC_SHIFT_TYPE_SET sts0, sts1;  NRC_SHIFT_TYPE st;  NRC_POLARITY po;
  KML_ELT histories_elt, history_elt, elt;  NRC_SHIFT_SET ss;  NRC_DAY first_d;
  int total_shifts, total_weekends, consecutive_shifts, consecutive_days;
  int consecutive_days_off;  NRC_CONSTRAINT c;  NRC_PENALTY penalty;

  if( DEBUG5 )
    fprintf(stderr, "[ AddNursesHistoryAndInitialUnwantedPatterns(ii)\n");
  if( !KmlContainsChild(ii->hy_root_elt, "NursesHistory", &histories_elt) )
    KmlEltFatalError(ii->hy_root_elt, ii->hy_file_name,
      "missing NursesHistory element");
  if( !KmlCheck(histories_elt, ": *NurseHistory", &ke) )
    KmlFatalError(ke, ii->hy_file_name);
  first_d = NrcInstanceCycleDay(ii->ins, 0);
  for( i = 0;  i < KmlChildCount(histories_elt);  i++ )
  {
    history_elt = KmlChild(histories_elt, i);
    if( !KmlCheck(history_elt, ": $Nurse #NumberOfAssignments "
	  "#NumberOfWorkingWeekends $LastAssignedShiftType "
	  "#NumberOfConsecutiveAssignments #NumberOfConsecutiveWorkingDays "
	  "#NumberOfConsecutiveDaysOff", &ke) )
      KmlFatalError(ke, ii->hy_file_name);

    /* resource */
    elt = KmlChild(history_elt, 0);
    val = KmlText(elt);
    if( !NrcInstanceStaffingRetrieveWorker(ii->ins, val, &w) )
      KmlEltFatalError(elt, ii->hy_file_name, "unknown nurse %s", val);

    /* number of assignments */
    elt = KmlChild(history_elt, 1);
    sscanf(KmlText(elt), "%d", &total_shifts);
    NrcWorkerAddHistory(w, "NumberOfAssignments", total_shifts);

    /* number of working weekends */
    elt = KmlChild(history_elt, 2);
    sscanf(KmlText(elt), "%d", &total_weekends);
    NrcWorkerAddHistory(w, "NumberOfWorkingWeekends", total_weekends);

    /* last assigned shift type */
    elt = KmlChild(history_elt, 3);
    val = KmlText(elt);
    if( strcmp(val, "None") != 0 )
    {
      if( !NrcInstanceRetrieveShiftType(ii->ins, val, &st) )
        KmlEltFatalError(elt, ii->hy_file_name, "unknown shift type %s", val);
      NrcWorkerAddHistory(w, "LastAssignedShiftType", NrcShiftTypeIndex(st));

      /* unwanted pattern constraints ending on the first day of the cycle */
      /* NB every pattern applies to every worker */
      penalty = NrcPenalty(true, 1, NRC_COST_FUNCTION_LINEAR, ii->ins);
      for( j = 0;  j < NrcInstancePatternCount(ii->ins);  j++ )
      {
	p = NrcInstancePattern(ii->ins, j);
	NrcPatternTerm(p, 0, &sts0, &po);
	NrcPatternTerm(p, 1, &sts1, &po);
	if( NrcShiftTypeSetContainsShiftType(sts0, st) )
	{
	  /* make the second term of p unwanted on the first day */
	  c = NrcConstraintMake(ii->ins, "UnwantedPattern-Init",
	    NrcWorkerSingletonWorkerSet(w), NRC_CONSTRAINT_ACTIVE,
	    NrcBoundMakeMax(0, penalty, ii->ins), NULL);
	  /* ***
	  c = NrcConstraintMake(ii->ins, "UnwantedPattern-Init",
	    NrcWorkerSingletonWorkerSet(w), penalty,
	    NrcLimit(NRC_LIMIT_MAX, 0), NULL);
	  *** */
	  ss = NrcDayShiftSetFromShiftTypeSet(first_d, sts1);
	  NrcConstraintAddShiftSet(c, ss, NRC_POSITIVE);
	}
      }
    }

    /* number of consecutive assignments */
    elt = KmlChild(history_elt, 4);
    sscanf(KmlText(elt), "%d", &consecutive_shifts);
    NrcWorkerAddHistory(w, "NumberOfConsecutiveAssignments",consecutive_shifts);

    /* consecutive working days */
    elt = KmlChild(history_elt, 5);
    sscanf(KmlText(elt), "%d", &consecutive_days);
    NrcWorkerAddHistory(w, "NumberOfConsecutiveWorkingDays", consecutive_days);

    /* consecutive days off */
    elt = KmlChild(history_elt, 6);
    sscanf(KmlText(elt), "%d", &consecutive_days_off);
    NrcWorkerAddHistory(w, "NumberOfConsecutiveDaysOff", consecutive_days_off);
  }
  if( DEBUG5 )
    fprintf(stderr, "] AddNursesHistoryAndInitialUnwantedPatterns returning\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "cover"                                                        */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void AddCoverConstraints(INRC2_INFO ii)                                  */
/*                                                                           */
/*  Add all events to ins.                                                   */
/*                                                                           */
/*****************************************************************************/

static void AddCoverConstraints(INRC2_INFO ii)
{
  KML_ELT reqs_elt, req_elt, day_req_elt, shift_type_elt, skill_elt;
  KML_ERROR ke;  NRC_DAY_SET ds;  NRC_PENALTY p1, p2, p3;  NRC_BOUND b;
  NRC_WORKER_SET ws;  /* NRC_DEMAND dm; */  NRC_DEMAND_SET dms;  NRC_DAY d;
  int i, j, k, min_cover, opt_cover;  char *id;  NRC_SHIFT_TYPE st;

  /* add cover requirements to the shifts_on_days */
  if( KmlContainsChild(ii->wk_root_elt, "Requirements", &reqs_elt) )
  {
    if( !KmlCheck(reqs_elt, ": *Requirement", &ke) )
      KmlFatalError(ke, ii->wk_file_name);
    for( i = 0;  i < KmlChildCount(reqs_elt);  i++ )
    {
      req_elt = KmlChild(reqs_elt, i);
      if( !KmlCheck(req_elt, ": $ShiftType $Skill RequirementOnMonday "
	  "RequirementOnTuesday RequirementOnWednesday RequirementOnThursday "
	  "RequirementOnFriday RequirementOnSaturday RequirementOnSunday",&ke) )
	KmlFatalError(ke, ii->wk_file_name);
      shift_type_elt = KmlChild(req_elt, 0);
      if( !NrcInstanceRetrieveShiftType(ii->ins, KmlText(shift_type_elt), &st) )
	KmlEltFatalError(shift_type_elt, ii->wk_file_name,
	  "unknown shift type %s", KmlText(shift_type_elt));
      skill_elt = KmlChild(req_elt, 1);
      if( !NrcInstanceSkillsRetrieveSkill(ii->ins, KmlText(skill_elt), &ws) )
	KmlEltFatalError(skill_elt, ii->wk_file_name, "unknown skill id %s",
	  KmlText(skill_elt));
      p1 = NrcPenalty(true, 1, NRC_COST_FUNCTION_LINEAR, ii->ins);
      p2 = NrcPenalty(false, 30, NRC_COST_FUNCTION_LINEAR, ii->ins);
      p3 = NrcPenalty(false, 0, NRC_COST_FUNCTION_LINEAR, ii->ins);
      for( j = 2;  j < KmlChildCount(req_elt);  j++ )
      {
	day_req_elt = KmlChild(req_elt, j);
	id = KmlLabel(day_req_elt) + strlen("RequirementOn");
	if( !NrcInstanceDaysOfWeekRetrieveDaySetLong(ii->ins, id, &ds) )
	  KmlEltFatalError(day_req_elt, ii->wk_file_name,
	    "unknown day name %s in label %s", id, KmlLabel(day_req_elt));
	if( !KmlCheck(day_req_elt, ": #Minimum #Optimal", &ke) )
          KmlFatalError(ke, ii->wk_file_name);
	sscanf(KmlText(KmlChild(day_req_elt, 0)), "%d", &min_cover);
	sscanf(KmlText(KmlChild(day_req_elt, 1)), "%d", &opt_cover);
	if( min_cover < 0 )
	  KmlEltFatalError(day_req_elt, ii->wk_file_name, "negative Minimum");
	if( min_cover > opt_cover )
	  KmlEltFatalError(day_req_elt, ii->wk_file_name,
	    "Minimum exceeds Optimal");
	if( opt_cover > 0 )
	{
	  /* build a suitable demand-set, dms */
	  b = NrcBoundMakeMin(min_cover, false, p1, ii->ins);
	  NrcBoundAddPreferred(b, opt_cover, p2, p3);
	  dms = NrcDemandSetMakeFromBound(ii->ins, b, opt_cover * 2, ws, p1);
	  /* ***
	  dms = NrcDemandSetMake(ii->ins);
	  dm = NrcDemandMake(ii->ins, p1, p3, ws, p1);
	  NrcDemandSetAddDemandMulti(dms, dm, min_cover);
	  dm = NrcDemandMake(ii->ins, p2, p3, ws, p1);
	  NrcDemandSetAddDemandMulti(dms, dm, opt_cover - min_cover);
	  dm = NrcDemandMake(ii->ins, p3, p3, ws, p1);
	  NrcDemandSetAddDemandMulti(dms, dm, opt_cover);
	  *** */

	  /* add dms to each shift */
	  for( k = 0;  k < NrcDaySetDayCount(ds);  k++ )
	  {
	    d = NrcDaySetDay(ds, k);
	    NrcShiftAddDemandSet(NrcDayShiftFromShiftType(d, st), dms);
	  }
	}
      }
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "shifts off"                                                   */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void AddShiftOffRequests(INRC2_INFO ii)                                  */
/*                                                                           */
/*  Add avoid unavailable times constraints to ii.  These handle rule S4.    */
/*  They are grouped by resource, based on day off and shift off requests.   */
/*                                                                           */
/*****************************************************************************/

static void AddShiftOffRequests(INRC2_INFO ii)
{
  KML_ELT shifts_off_elt, shift_off_elt, nurse_elt, day_elt, shift_elt;
  NRC_DAY_SET ds;  KML_ERROR ke;  int i;  NRC_DAY d;  NRC_WORKER w;
  char *id;  NRC_SHIFT_TYPE st;  NRC_PENALTY p;

  /* accumulate shift off requests */
  if( KmlContainsChild(ii->wk_root_elt, "ShiftOffRequests", &shifts_off_elt) )
  {
    if( !KmlCheck(shifts_off_elt, ": *ShiftOffRequest", &ke) )
      KmlFatalError(ke, ii->wk_file_name);
    p = NrcPenalty(false, 10, NRC_COST_FUNCTION_LINEAR, ii->ins);
    for( i = 0;  i < KmlChildCount(shifts_off_elt);  i++ )
    {
      /* get shift_off_elt and convert its attributes into NRC ones */
      shift_off_elt = KmlChild(shifts_off_elt, i);
      if( !KmlCheck(shift_off_elt, ": $Nurse $ShiftType $Day", &ke) )
	KmlFatalError(ke, ii->wk_file_name);

      /* find the nurse */
      nurse_elt = KmlChild(shift_off_elt, 0);
      id = KmlText(nurse_elt);
      if( !NrcInstanceStaffingRetrieveWorker(ii->ins, id, &w) )
        KmlEltFatalError(nurse_elt, ii->wk_file_name, "unknown nurse %s", id);

      /* find the day */
      day_elt = KmlChild(shift_off_elt, 2);
      id = KmlText(day_elt);
      if( !NrcInstanceDaysOfWeekRetrieveDaySetLong(ii->ins, id, &ds) )
        KmlEltFatalError(day_elt, ii->wk_file_name, "unknown day %s", id);
      HnAssert(NrcDaySetDayCount(ds)==1,"AddShiftOffRequests internal error 1");
      d = NrcDaySetDay(ds, 0);

      /* find the shift, or "Any", and record shift off or day off */
      shift_elt = KmlChild(shift_off_elt, 1);
      id = KmlText(shift_elt);
      if( strcmp(id, "Any") == 0 )
      {
	/* day off */
	NrcWorkerAddDayOff(w, d, p);
      }
      else if( NrcInstanceRetrieveShiftType(ii->ins, id, &st) )
      {
	/* shift off */
	NrcWorkerAddShiftOff(w, NrcDayShiftFromShiftType(d, st), p);
      }
      else
        KmlEltFatalError(shift_elt, ii->wk_file_name, "unknown shift %s", id);
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "single assignment per day constraint"                         */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void SingleAssignmentPerDay(INRC2_INFO ii)                               */
/*                                                                           */
/*  Single assignment per day.  This handles competition rule H1.            */
/*                                                                           */
/*****************************************************************************/

static void SingleAssignmentPerDay(INRC2_INFO ii)
{
  NRC_CONSTRAINT c;  NRC_SHIFT_SET starting_ss, day_ss;  NRC_PENALTY p;
  starting_ss = NrcInstanceDailyStartingShiftSet(ii->ins);
  p = NrcPenalty(true, 1, NRC_COST_FUNCTION_LINEAR, ii->ins);
  c = NrcConstraintMake(ii->ins, "SingleAssignmentPerDay",
    NrcInstanceStaffing(ii->ins), NRC_CONSTRAINT_ACTIVE,
    NrcBoundMakeMax(1, p, ii->ins), starting_ss);
  /* ***
  c = NrcConstraintMake(ii->ins, "SingleAssignmentPerDay",
    NrcInstanceStaffing(ii->ins), p, NrcLimit(NRC_LIMIT_MAX, 1), starting_ss);
  *** */
  day_ss = NrcDayShiftSet(NrcInstanceCycleDay(ii->ins, 0));
  NrcConstraintAddShiftSet(c, day_ss, NRC_POSITIVE);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "main functions (dynamic)"                                     */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  NRC_INSTANCE INRC2ConvertInstance(INSTANCE_MODEL ins_model,              */
/*    char *instance_file_name, HA_ARENA_SET as)                             */
/*                                                                           */
/*  Read the files named by file_name_seq, which is supposed to have format  */
/*                                                                           */
/*    scenario_file_name,history_file_name,week_file_name                    */
/*                                                                           */
/*  naming three files in the format of the second international nurse       */
/*  rostering competition, and return the resulting instance.                */
/*                                                                           */
/*****************************************************************************/

NRC_INSTANCE INRC2ConvertInstance(INSTANCE_MODEL ins_model,
  char *instance_file_name, HA_ARENA_SET as)
{
  char *comma1, *comma2, *comma3;  INRC2_INFO ii;  int i;
  FILE *sc_fp, *hy_fp, *wk_fp;  KML_ERROR ke;  HA_ARENA a;
  char *sc_id, *hy_id, *wk_id, *str1, *str2, *ins_id;  char buff[21];
  KML_ELT sc_elt;
  if( DEBUG1 )
    fprintf(stderr, "[ INRC2ConvertInstance(%s)\n", instance_file_name);

  /* make an instance info object, initially more or less empty */
  a = HaArenaSetArenaBegin(as);
  HaMake(ii, a);
  ii->arena = a;
  ii->sc_file_name = NULL;
  ii->sc_root_elt = NULL;
  ii->hy_file_name = NULL;
  ii->hy_root_elt = NULL;
  ii->wk_file_name = NULL;
  ii->wk_root_elt = NULL;
  ii->ins = NULL;
  ii->weekend = NULL;
  ii->number_of_weeks = 0;
  ii->week_index = 0;
  ii->days_before = 0;
  ii->days_after = 0;

  /* dismantle instance_file_name */
  comma1 = strstr(instance_file_name, ",");
  if( comma1 == NULL )
  {
    fprintf(stderr, "nrconv:  error in command-line argument \"%s\" (%s)\n",
      instance_file_name, "missing first comma");
    exit(1);
  }
  comma2 = strstr(comma1 + 1, ",");
  if( comma2 == NULL )
  {
    fprintf(stderr, "nrconv:  error in command-line argument \"%s\" (%s)\n",
      instance_file_name, "missing second comma");
    exit(1);
  }
  comma3 = strstr(comma2 + 1, ",");
  if( comma3 != NULL )
  {
    fprintf(stderr, "nrconv:  error in command-line argument \"%s\" (%s)\n",
      instance_file_name, "too many commas");
    exit(1);
  }
  *comma1 = *comma2 = '\0';
  ii->sc_file_name = instance_file_name;
  ii->hy_file_name = comma1 + 1;
  ii->wk_file_name =  comma2 + 1;
  if( DEBUG1 )
  {
    fprintf(stderr, "  ii->sc_file_name = %s\n", ii->sc_file_name);
    fprintf(stderr, "  ii->hy_file_name = %s\n", ii->hy_file_name);
    fprintf(stderr, "  ii->wk_file_name = %s\n", ii->wk_file_name);
  }

  /* open the files and exit if can't */
  sc_fp = fopen(ii->sc_file_name, "r");
  if( sc_fp == NULL )
    FatalError(NULL, 0, "cannot open inrc2 scenario file \"%s\"\n",
      ii->sc_file_name);
  hy_fp = fopen(ii->hy_file_name, "r");
  if( hy_fp == NULL )
    FatalError(NULL, 0, "cannot open inrc2 history file \"%s\"\n",
      ii->hy_file_name);
  wk_fp = fopen(ii->wk_file_name, "r");
  if( wk_fp == NULL )
    FatalError(NULL, 0, "cannot open inrc2 week file \"%s\"\n",
      ii->wk_file_name);

  /* read XML from the files, and exit if can't */
  if( !KmlReadFile(sc_fp, NULL, &ii->sc_root_elt, &ke, a) )
    KmlFatalError(ke, ii->sc_file_name);
  fclose(sc_fp);
  if( !KmlReadFile(hy_fp, NULL, &ii->hy_root_elt, &ke, a) )
    KmlFatalError(ke, ii->hy_file_name);
  fclose(hy_fp);
  if( !KmlReadFile(wk_fp, NULL, &ii->wk_root_elt, &ke, a) )
    KmlFatalError(ke, ii->wk_file_name);
  fclose(wk_fp);

  /* make sure the root tags and their children are all present and correct */
  /* scenario */
  if( strcmp(KmlLabel(ii->sc_root_elt), "Scenario") != 0 )
    KmlEltFatalError(ii->sc_root_elt, ii->sc_file_name,
      "root tag %s where Scenario expected", KmlLabel(ii->sc_root_elt));
  if( !KmlCheck(ii->sc_root_elt,
      "Id +xsi:noNamespaceSchemaLocation +xmlns:xsi +xmlns "
      ": #NumberOfWeeks +Skills ShiftTypes +ForbiddenShiftTypeSuccessions "
      "Contracts Nurses", &ke) )
    KmlFatalError(ke, ii->sc_file_name);
  sc_id = KmlAttributeValue(ii->sc_root_elt, 0);
  sscanf(KmlText(KmlChild(ii->sc_root_elt, 0)), "%d", &ii->number_of_weeks);

  /* history */
  if( strcmp(KmlLabel(ii->hy_root_elt), "History") != 0 )
    KmlEltFatalError(ii->hy_root_elt, ii->hy_file_name,
      "root tag %s where History expected", KmlLabel(ii->hy_root_elt));
  if( !KmlCheck(ii->hy_root_elt,
      "+xsi:noNamespaceSchemaLocation +xmlns:xsi +xmlns "
      ": #Week $Scenario NursesHistory", &ke) )
    KmlFatalError(ke, ii->hy_file_name);
  hy_id = KmlText(KmlChild(ii->hy_root_elt, 0));
  sc_elt = KmlChild(ii->hy_root_elt, 1);
  if( strcmp(KmlText(sc_elt), sc_id) != 0 )
    KmlEltFatalError(sc_elt, ii->hy_file_name,
      "scenario %s in history file does not match Id %s in scenario file",
      KmlText(sc_elt), sc_id);
  sscanf(KmlText(KmlChild(ii->hy_root_elt, 0)), "%d", &ii->week_index);
  ii->days_before = (ii->week_index == 0 ? 1 : ii->week_index) * 7,
  ii->days_after = (ii->number_of_weeks - ii->week_index - 1) * 7;

  /* week data */
  if( strcmp(KmlLabel(ii->wk_root_elt), "WeekData") != 0 )
    KmlEltFatalError(ii->wk_root_elt, ii->wk_file_name,
      "root tag %s where WeekData expected", KmlLabel(ii->wk_root_elt));
  if( !KmlCheck(ii->wk_root_elt,
      "+xsi:noNamespaceSchemaLocation +xmlns:xsi +xmlns "
      ": $Scenario +Requirements +ShiftOffRequests", &ke) )
    KmlFatalError(ke, ii->wk_file_name);
  sc_elt = KmlChild(ii->wk_root_elt, 0);
  if( strcmp(KmlText(sc_elt), sc_id) != 0 )
    KmlEltFatalError(sc_elt, ii->wk_file_name,
      "scenario %s in week file does not match Id %s in scenario file",
      KmlText(sc_elt), sc_id);

  /* week ID, scraped from week file name */
  str1 = strrchr(ii->wk_file_name, '-');
  str2 = strrchr(ii->wk_file_name, '.');
  if( str1 != NULL && str2 != NULL && str2 - str1 >= 2 )
  {
    for( i = 1;  i < str2 - str1 && i <= 20;  i++ )
      buff[i-1] = str1[i];
    buff[i-1] = '\0';
    wk_id = HnStringCopy(buff, a);
  }
  else
    wk_id = "?";

  /* make instance */
  ins_id = HnStringMake(a, "C2-%s-w%s-h%s", sc_id, wk_id, hy_id);
  /* ***
  md = InstanceModelMetaData(ins_model, ins_id, NULL, NULL, NULL, NULL, NULL);
  *** */
  /* p = NrcPenalty(true, 1, NRC_COST_FUNCTION_LINEAR, ii->ins); */
  ii->ins = NrcInstanceMake(ins_id, "Nurses", "Nurse", as);
  NrcInstanceSetMetaData(ii->ins, ins_id, NULL, NULL, NULL, NULL, NULL);

  /* add time stuff */
  AddDaysAndWeekend(ii);
  AddShiftTypes(ii);

  /* add nurse stuff */
  AddSkills(ii);
  AddContracts(ii);
  AddNurses(ii);
  AddNursesHistoryAndInitialUnwantedPatterns(ii);

  /* add worker constraints */
  SingleAssignmentPerDay(ii);
  AddShiftTypeConstraints(ii);
  AddPatternsAndUnwantedPatternConstraints(ii);
  AddContractConstraints(ii);

  /* add cover constraints and shift-off requests */
  AddCoverConstraints(ii);
  AddShiftOffRequests(ii);

  if( DEBUG1 )
    fprintf(stderr, "] INRC2ConvertInstance returning\n");
  HaArenaSetArenaEnd(as, a);
  return ii->ins;
}


/*****************************************************************************/
/*                                                                           */
/*  void INRC2ConvertSoln(SOLN_MODEL soln_model, char *soln_file_name,       */
/*    NRC_INSTANCE ins, NRC_ARCHIVE archive, HA_ARENA_SET as)                */
/*                                                                           */
/*  Convert soln_file_name into a solution of ins and add it to a solution   */
/*  group of archive, which is known to have at least one solution group.    */
/*                                                                           */
/*****************************************************************************/

void INRC2ConvertSoln(SOLN_MODEL soln_model, char *soln_file_name,
  NRC_INSTANCE ins, NRC_ARCHIVE archive, HA_ARENA_SET as)
{
  FILE *fp;  KML_ELT soln_elt;  NRC_DEMAND dm;  HA_ARENA a;
  NRC_SOLN soln;  KML_ERROR ke; NRC_WORKER w;  int i, j;  NRC_DAY_SET ds;
  NRC_DAY d;  NRC_SHIFT_TYPE st;  NRC_WORKER_SET skill_ws;  NRC_SHIFT s;
  KML_ELT assts_elt, asst_elt, nurse_elt, day_elt, shift_type_elt, skill_elt;

  /* must have an instance in this model */
  if( ins == NULL )
  {
    fprintf(stderr, "nrconv:  -t not permitted with -minrc2\n");
    exit(1);
  }

  /* open the file and exit if can't */
  fp = fopen(soln_file_name, "r");
  if( fp == NULL )
  {
    fprintf(stderr,
      "nrconv:  ignoring inrc2 solution file \"%s\" (cannot open)\n",
      soln_file_name);
    return;
  }

  /* read XML from the file, and exit if can't */
  a = HaArenaSetArenaBegin(as);
  if( !KmlReadFile(fp, NULL, &soln_elt, &ke, a) )
    KmlFatalError(ke, soln_file_name);
  fclose(fp);

  /* make sure the root tags and their children are all present and correct */
  if( strcmp(KmlLabel(soln_elt), "Solution") != 0 )
    KmlEltFatalError(soln_elt, soln_file_name,
      "root tag %s where Solution expected", KmlLabel(soln_elt));
  if( !KmlCheck(soln_elt, "+xsi:noNamespaceSchemaLocation +xmlns:xsi +xmlns"
      " : #Week $Scenario Assignments", &ke) )
  {
    if( DEBUG2 )
      fprintf(stderr, "INRC2ConvertSoln failing at 1\n");
    KmlFatalError(ke, soln_file_name);
  }

  /* make a soln object and load the assignments */
  soln = NrcSolnMake(ins, as);
  assts_elt = KmlChild(soln_elt, 2);
  if( !KmlCheck(assts_elt, " : *Assignment", &ke) )
    KmlFatalError(ke, soln_file_name);
  for( i = 0;  i < KmlChildCount(assts_elt);  i++ )
  {
    asst_elt = KmlChild(assts_elt, i);
    if( !KmlCheck(asst_elt, " : $Nurse $Day $ShiftType $Skill", &ke) )
      KmlFatalError(ke, soln_file_name);

    /* get the nurse */
    nurse_elt = KmlChild(asst_elt, 0);
    if( !NrcInstanceStaffingRetrieveWorker(ins, KmlText(nurse_elt), &w) )
      KmlEltFatalError(nurse_elt, soln_file_name, "unknown Nurse %s",
	KmlText(nurse_elt));

    /* get the day */
    day_elt = KmlChild(asst_elt, 1);
    if( !NrcInstanceDaysOfWeekRetrieveDaySetShort(ins, KmlText(day_elt), &ds) )
      KmlEltFatalError(day_elt, soln_file_name, "unknown Day %s",
	KmlText(day_elt));
    if( NrcDaySetDayCount(ds) != 1 )
      KmlEltFatalError(day_elt, soln_file_name, "internal error (Day %s)",
        KmlText(day_elt));
    d = NrcDaySetDay(ds, 0);

    /* get the shift type */
    shift_type_elt = KmlChild(asst_elt, 2);
    if( !NrcInstanceRetrieveShiftType(ins, KmlText(shift_type_elt), &st) )
      KmlEltFatalError(shift_type_elt, soln_file_name, "unknown ShiftType %s",
	KmlText(shift_type_elt));

    /* get the skill */
    skill_elt = KmlChild(asst_elt, 3);
    if( !NrcInstanceSkillsRetrieveSkill(ins, KmlText(skill_elt), &skill_ws) )
      KmlEltFatalError(skill_elt, soln_file_name, "unknown Skill %s",
	KmlText(skill_elt));

    /* get the shift, find a suitable demand in it, and assign */
    s = NrcDayShiftFromShiftType(d, st);
    for( j = 0;  j < NrcShiftDemandCount(s);  j++ )
    {
      dm = NrcShiftDemand(s, j);
      if( NrcDemandPreferredWorkerSet(dm) == skill_ws &&
	  NrcSolnAssignment(soln, s, j) == NULL )
	break;
    }
    if( j == NrcShiftDemandCount(s) )
      KmlEltFatalError(asst_elt, soln_file_name,
	"cannot find a free slot for skill %s in shift %s:%s",
	NrcWorkerSetName(skill_ws), NrcDayShortName(d), NrcShiftTypeName(st));
    NrcSolnAddAssignment(soln, s, j, w);
  }

  /* add soln to the appropriate soln group */
  SolnModelAddSoln(soln_model, archive, "", 0.0, soln);
  HaArenaSetArenaEnd(as, a);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "main functions (static)"                                      */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool GrabInt(char **p, char *val)                                        */
/*                                                                           */
/*  If **p contains another integer, grab it, put it in **val, and move *p.  */
/*                                                                           */
/*****************************************************************************/

static bool GrabInt(char **p, char *val)
{
  if( **p == '\0' || !sscanf(*p, "%[0-9]", val) )
    return false;
  *p += strlen(val);
  while( **p == '-' ) (*p)++;
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  NRC_INSTANCE INRC2ConvertInstanceStatic(INSTANCE_MODEL ins_model,        */
/*    char *instance_file_name, HA_ARENA_SET as)                             */
/*                                                                           */
/*  Convert a static INRC2 instance.  Here the instance file name has form   */
/*                                                                           */
/*    <stem>/120-4-1-5-6-9-8                                                 */
/*                                                                           */
/*  for example, where 120 is the number of nurses, 4 is the number of       */
/*  weeks, 1 identifies the history, and the other numbers are weeks.        */
/*  The files actually read lie in directory                                 */
/*                                                                           */
/*    <stem>/n120w4                                                          */
/*                                                                           */
/*  Here is what this directory contains:                                    */
/*                                                                           */
/*    H0-n120w4-0.xml  WD-n120w4-0.xml  WD-n120w4-4.xml  WD-n120w4-8.xml     */
/*    H0-n120w4-1.xml  WD-n120w4-1.xml  WD-n120w4-5.xml  WD-n120w4-9.xml     */
/*    H0-n120w4-2.xml  WD-n120w4-2.xml  WD-n120w4-6.xml                      */
/*    Sc-n120w4.xml    WD-n120w4-3.xml  WD-n120w4-7.xml                      */
/*                                                                           */
/*****************************************************************************/

NRC_INSTANCE INRC2ConvertInstanceStatic(INSTANCE_MODEL ins_model,
  char *instance_file_name, HA_ARENA_SET as)
{
  char *suffix, *p;  int len, suffix_len, weeks, count;  char save_suffix;
  char field1[500], field2[500], field3[500], field4[500], tag[500];
  char dir_name[500], scen[500], hist[500], week[500];

  if( DEBUG7 )
    fprintf(stderr, "[ INRC2ConvertInstanceStatic(model, %s, -)\n",
      instance_file_name);

  /* let suffix point to just after the last / in the file name */
  len = strlen(instance_file_name);
  HnAssert(len > 0, "INRC2ConvertInstanceStatic:  empty file name");
  HnAssert(len < 400, "INRC2ConvertInstanceStatic:  file name too long");
  suffix = &instance_file_name[len - 1];
  while( suffix != instance_file_name && *suffix != '/' )
    suffix--;
  if( *suffix == '/' )  suffix++;

  /* build the tag (n120w4 or whatever) */
  suffix_len = strlen(suffix);
  HnAssert(suffix_len >= 7,
    "INRC2ConvertInstanceStatic: file name suffix \"%s\" is too short", suffix);
  p = suffix;
  GrabInt(&p, field1);
  GrabInt(&p, field2);
  GrabInt(&p, field3);
  sprintf(tag, "n%sw%s", field1, field2);
  if( DEBUG7 )
    fprintf(stderr, "  tag \"%s\"\n", tag);

  /* build the directory name */
  save_suffix = *suffix;
  *suffix = '\0';
  sprintf(dir_name, "%s%s", instance_file_name, tag);
  *suffix = save_suffix;
  if( DEBUG7 )
    fprintf(stderr, "  dir \"%s\"\n", dir_name);

  /* build the scenario and history file names */
  sprintf(scen, "%s/Sc-%s.xml", dir_name, tag);
  sprintf(hist, "%s/H0-%s-%s.xml", dir_name, tag, field3);
  if( DEBUG7 )
    fprintf(stderr, "  scen \"%s\"\n", scen);
  if( DEBUG7 )
    fprintf(stderr, "  hist \"%s\"\n", hist);

  /* weekly file names */
  sscanf(field2, "%d", &weeks);
  count = 0;
  while( GrabInt(&p, field4) )
  {
    count++;
    sprintf(week, "%s/WD-%s-%s.xml", dir_name, tag, field4);
    if( DEBUG7 )
      fprintf(stderr, "  week %d: \"%s\"\n", count, week);
  }
  HnAssert(count == weeks,
    "INRC2ConvertInstanceStatic: inconsistent suffix \"%s\"", suffix);

  /* remainder still to do */
  if( DEBUG7 )
    fprintf(stderr, "] INRC2ConvertInstanceStatic returning\n");
  return NULL;
}
