
/*****************************************************************************/
/*                                                                           */
/*  THE HOWARD OBJECT-ORIENTED COMPILER TOOLKIT                              */
/*  COPYRIGHT (C) 2011 Jeffrey H. Kingston                                   */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 3, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to Free Software Foundation       */
/*  Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA               */
/*                                                                           */
/*  FILE:     ha_arena.c                                                     */
/*  PURPOSE:  Memory arenas                                                  */
/*                                                                           */
/*****************************************************************************/
#include "howard_a.h"
#include <stdarg.h>

#define HA_INIT_CHUNK_WORDS 13
#define HA_INIT_BLOCK_WORDS  5
#define HA_CHUNK_OVERHEAD_WORDS 3

#define DEBUG1 0
#define DEBUG2 0
#define DEBUG3 0
#define DEBUG4 0
#define DEBUG5 0
#define DEBUG6 0
#define DEBUG7 0
#define DEBUG8 0
#define CHECK_CHUNK_SIZE 0
#define CHECK_CHUNK_SIZE_LIMIT 83886050L

/*****************************************************************************/
/*                                                                           */
/*  Private typedefs                                                         */
/*                                                                           */
/*****************************************************************************/

typedef struct ha_chunk_rec		*HA_CHUNK;
typedef struct ha_block_list_rec	*HA_BLOCK_LIST;
typedef struct ha_free_block_rec	*HA_FREE_BLOCK;
typedef struct ha_allocated_block_rec	*HA_ALLOCATED_BLOCK;


/*****************************************************************************/
/*                                                                           */
/*  HA_CHUNK - a chunk of memory obtained from malloc                        */
/*                                                                           */
/*****************************************************************************/

struct ha_chunk_rec {
  union {
    struct {
      HA_CHUNK		next_chunk;
      size_t		mem_total_words;	/* total words in mem[]      */
      size_t		mem_avail_words;	/* avail words in mem[]      */
    } s;
    HA_ALIGN_TYPE	align[HA_CHUNK_OVERHEAD_WORDS];
  } u;
  HA_ALIGN_TYPE		mem[1];		/* extends */
};


/*****************************************************************************/
/*                                                                           */
/*  HA_BLOCK_LIST - a list of free blocks containing block_mem_words each    */
/*                                                                           */
/*****************************************************************************/

struct ha_block_list_rec {
  HA_ARENA		arena;
  size_t		block_mem_words;
  HA_FREE_BLOCK		first_free_block;
  HA_BLOCK_LIST		next_block_list;
};


/*****************************************************************************/
/*                                                                           */
/*  union ha_arena_or_block_rec - common parent type of arenas and blocks    */
/*                                                                           */
/*  Implementation note.  To save memory, the unique block of 0 words is     */
/*  represented by the arena object itself, hence the need for a common      */
/*  parent type for arenas and blocks.                                       */
/*                                                                           */
/*****************************************************************************/

union ha_arena_or_block_rec {
  HA_BLOCK_LIST		block_list;
  HA_ALIGN_TYPE		align;
};


/*****************************************************************************/
/*                                                                           */
/*  HA_FREE_BLOCK - a resizable block, currently free                        */
/*                                                                           */
/*****************************************************************************/

struct ha_free_block_rec {
  union ha_arena_or_block_rec	u;
  HA_FREE_BLOCK			next_free_block;
};


/*****************************************************************************/
/*                                                                           */
/*  HA_ALLOCATED_BLOCK - a resizable block, currently allocated              */
/*                                                                           */
/*****************************************************************************/

struct ha_allocated_block_rec {
  union ha_arena_or_block_rec	u;
  HA_ALIGN_TYPE			mem[1];		/* extends */
};


/*****************************************************************************/
/*                                                                           */
/*  HA_ARENA - an arena                                                      */
/*                                                                           */
/*****************************************************************************/

struct ha_arena_rec {
  union ha_arena_or_block_rec	u;
  HA_CHUNK			first_chunk;
  HA_CHUNK			first_recycled_chunk;
  size_t			allocated_words;
  bool				large;
};


/*****************************************************************************/
/*                                                                           */
/*  void HaAssert(bool cond, char *fmt, ...)                                 */
/*                                                                           */
/*  If cond is false, print an error message on stderr and abort.            */
/*                                                                           */
/*****************************************************************************/

/* ***
static void HaAssert(bool cond, char *fmt, ...)
{
  va_list args;
  if( !cond )
  {
    va_start(args, fmt);
    vfprintf(stderr, fmt, args);
    fprintf(stderr, "\n");
    va_end(args);
    abort();
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  size_t HaBytesToWords(size_t size)                                       */
/*                                                                           */
/*  Convert size (measured in bytes) to words, rounding up to the next       */
/*  whole number of words.                                                   */
/*                                                                           */
/*****************************************************************************/

static size_t HaBytesToWords(size_t size)
{
  return (size + (sizeof(HA_ALIGN_TYPE) - 1)) / sizeof(HA_ALIGN_TYPE);
}


/*****************************************************************************/
/*                                                                           */
/*  HA_CHUNK HaCallocChunk(size_t words, HA_ARENA a)                         */
/*                                                                           */
/*  Get from calloc a chunk whose mem[] array contains words words.          */
/*                                                                           */
/*  Parameter a is for debugging only, and may be NULL.                      */
/*                                                                           */
/*****************************************************************************/

static HA_CHUNK HaCallocChunk(size_t words, HA_ARENA a)
{
  HA_CHUNK res;
  if( DEBUG7 && words > 500000 )
  {
    if( a != NULL )
      fprintf(stderr, "  %sarena %p requesting %lu words\n",
	a->large ? "large " : "", (void *) a, words);
    else
      fprintf(stderr, "  new arena requesting %lu words\n", words);
  }
  if( CHECK_CHUNK_SIZE && words >= CHECK_CHUNK_SIZE_LIMIT )
  {
    fprintf(stderr, "HaCallocChunk(%ld) at size limit (%ld)\n", words,
      CHECK_CHUNK_SIZE_LIMIT);
    abort();
  }
  res = (HA_CHUNK) calloc(words + HA_CHUNK_OVERHEAD_WORDS,
    sizeof(HA_ALIGN_TYPE));
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  HA_BLOCK_LIST HaBlockListMake(HA_ARENA arena, size_t words)              */
/*                                                                           */
/*  Make a block list for blocks of size words, or NULL if no memory avail.  */
/*                                                                           */
/*****************************************************************************/

static HA_BLOCK_LIST HaBlockListMake(HA_ARENA arena, size_t words)
{
  HA_BLOCK_LIST res;
  HaMake(res, arena);
  if( res == NULL )
    return NULL;
  res->arena = arena;
  res->block_mem_words = words;
  res->first_free_block = NULL;
  res->next_block_list = NULL;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  HA_ARENA HaArenaMake(bool large)                                         */
/*                                                                           */
/*  Make a new, empty arena.                                                 */
/*                                                                           */
/*****************************************************************************/

HA_ARENA HaArenaMake(bool large)
{
  HA_ARENA res;  HA_CHUNK first_chunk;  size_t mem_avail_words, arena_words;
  if( DEBUG6 )
    fprintf(stderr, "  HaArenaMake()\n");

  /* align type must hold at least a pointer */
  HaAssert(sizeof(HA_ALIGN_TYPE) >= sizeof(void *),
    L"HaArenaMake: HA_ALIGN_TYPE is too small (must hold pointer at least)");

  /* the space at the start of each chunk must be as expected */
  HaAssert(sizeof(first_chunk->u) == sizeof(first_chunk->u.align),
    L"HaArenaMake: HA_CHUNK.u.align needs to have its length increased:\n"
    "  sizeof(first_chunk->u) == %d, sizeof(first_chunk->u.align) == %d;\n"
    "  sizeof(size_t) == %d",
    sizeof(first_chunk->u), sizeof(first_chunk->u.align), sizeof(size_t));
  
  /* initial chunk size must hold at least an arena object */
  arena_words = HaBytesToWords(sizeof(struct ha_arena_rec));
  HaAssert(HA_INIT_CHUNK_WORDS >= arena_words,
   L"HaArenaMake: HA_INIT_CHUNK_WORDS is too small (must hold arena at least)");

  /* make the first chunk */
  first_chunk = HaCallocChunk(HA_INIT_CHUNK_WORDS, NULL);
  if( first_chunk == NULL )
    return NULL;
  first_chunk->u.s.next_chunk = NULL;
  first_chunk->u.s.mem_total_words = HA_INIT_CHUNK_WORDS;

  /* make the arena out of memory lying in (at the end of) the first chunk */
  mem_avail_words = HA_INIT_CHUNK_WORDS - arena_words;
  res = (HA_ARENA) &first_chunk->mem[mem_avail_words];
  res->first_chunk = first_chunk;
  res->first_recycled_chunk = NULL;
  res->allocated_words = HA_INIT_CHUNK_WORDS;
  res->large = large;
  if( DEBUG4 )
    HaArenaDebug(res, "initially", 0, stderr);
  first_chunk->u.s.mem_avail_words = mem_avail_words;
  /* this is point alpha (see HaArenaClear below) */

  /* make the first block list */
  res->u.block_list = HaBlockListMake(res, 0);
  if( res->u.block_list == NULL )
    return NULL;

  /* all set, return the arena */
  if( DEBUG1 )
  {
    fprintf(stderr, "HaArenaMake returning new arena:\n");
    HaArenaDebug(res, "initially", 2, stderr);
  }
  if( DEBUG8 && large )
    fprintf(stderr, "  HaArenaMake made new large arena %p\n", (void *) res);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  bool HaArenaLarge(HA_ARENA a)                                            */
/*                                                                           */
/*  Return true if a is intended to be large.                                */
/*                                                                           */
/*****************************************************************************/

bool HaArenaLarge(HA_ARENA a)
{
  return a->large;
}



/*****************************************************************************/
/*                                                                           */
/*  void HaArenaDelete(HA_ARENA a)                                           */
/*                                                                           */
/*  Delete arena a.  This just frees its chunks, being careful to extract    */
/*  the next_chunk pointer from each chunk before freeing that chunk.        */
/*                                                                           */
/*****************************************************************************/

void HaArenaDelete(HA_ARENA a)
{
  HA_CHUNK chunk, next_chunk;
  if( DEBUG3 )
    fprintf(stderr, "[ HaArenaDelete(%p)\n", (void *) a);
  chunk = a->first_recycled_chunk;
  while( chunk != NULL )
  {
    next_chunk = chunk->u.s.next_chunk;
    free(chunk);
    chunk = next_chunk;
  }
  chunk = a->first_chunk;
  do
  {
    next_chunk = chunk->u.s.next_chunk;
    free(chunk);
    chunk = next_chunk;
  } while( chunk != NULL );
  if( DEBUG3 )
    fprintf(stderr, "] HaArenaDelete\n");
}


/*****************************************************************************/
/*                                                                           */
/*  bool HaChunkContainsArena(HA_CHUNK chunk, HA_ARENA a)                    */
/*                                                                           */
/*  Return true if chunk contains a.  If a is present at all, it will        */
/*  be at the end of the arena; but it's hard to say exactly where,          */
/*  because of the alignment, so we test the whole range.                    */
/*                                                                           */
/*****************************************************************************/

static bool HaChunkContainsArena(HA_CHUNK chunk, HA_ARENA a)
{
  return (void *) &chunk->mem[0] <= (void *) a && 
    (void *) a < (void *) &chunk->mem[HA_INIT_CHUNK_WORDS];
}


/*****************************************************************************/
/*                                                                           */
/*  void HaArenaClear(HA_ARENA a)                                            */
/*                                                                           */
/*  Clear arena a, returning it to its initial state.                        */
/*                                                                           */
/*****************************************************************************/

void HaArenaClear(HA_ARENA a)
{
  HA_CHUNK chunk, next_chunk, arena_chunk;  HA_ARENA res;
  size_t mem_avail_words, arena_words;
  if( DEBUG1 )
    fprintf(stderr, "HaArenaClear(%p)\n", (void *) a);

  /* free all chunks except arena_chunk, the one holding a itself */
  /* NB the original arena_chunk could be anywhere by this time */
  arena_chunk = NULL;
  chunk = a->first_chunk;
  do
  {
    next_chunk = chunk->u.s.next_chunk;
    if( HaChunkContainsArena(chunk, a) )
      arena_chunk = chunk;
    else
      free(chunk);
    chunk = next_chunk;
  } while( chunk != NULL );
  HaAssert(arena_chunk != NULL, L"HaArenaClear internal error 1");

  /* free recycled chunks */
  chunk = a->first_recycled_chunk;
  while( chunk != NULL )
  {
    next_chunk = chunk->u.s.next_chunk;
    free(chunk);
    chunk = next_chunk;
  }
  a->first_recycled_chunk = NULL;

  /* return arena_chunk to its state at point alpha in HaArenaMake */
  arena_words = HaBytesToWords(sizeof(struct ha_arena_rec));
  mem_avail_words = HA_INIT_CHUNK_WORDS - arena_words;
  res = (HA_ARENA) &arena_chunk->mem[mem_avail_words];
  HaAssert(res == a, L"HaArenaClear internal error 2");
  res->first_chunk = arena_chunk;
  arena_chunk->u.s.mem_avail_words = mem_avail_words;
  arena_chunk->u.s.next_chunk = NULL;
  /* res->mem_total_words = HA_INIT_CHUNK_WORDS; already set in arena_chunk */

  /* continue from alpha as in HaArenaMake, except don't return res */
  res->u.block_list = HaBlockListMake(res, 0);
  HaAssert(res->u.block_list != NULL, L"HaArenaClear internal error 3");
}


/*****************************************************************************/
/*                                                                           */
/*  void HaArenaRecycle(HA_ARENA a)                                          */
/*                                                                           */
/*  Similar to HaArenaClear except that the chunks are recycled, not freed.  */
/*                                                                           */
/*  Since original chunks are allocated with calloc which initializes them   */
/*  to zero, we need to reset the contents of recycled chunks to zero,       */
/*  which we do by calling                                                   */
/*                                                                           */
/*    void *memset(void *s, int c, size_t n);                                */
/*                                                                           */
/*  The memset() function fills the first n bytes of the memory area         */
/*  pointed to by s with the constant byte c, according to "man memset".     */
/*                                                                           */
/*****************************************************************************/

void HaArenaRecycle(HA_ARENA a)
{
  HA_CHUNK chunk, next_chunk, arena_chunk;  HA_ARENA res;
  size_t mem_avail_words, arena_words;
  if( DEBUG2 )
    fprintf(stderr, "[ HaArenaRecycle(%p)\n", (void *) a);

  /* recycle all chunks except arena_chunk, the one holding a itself */
  /* NB the original arena_chunk could be anywhere by this time */
  arena_chunk = NULL;
  chunk = a->first_chunk;
  do
  {
    if( DEBUG2 )
      fprintf(stderr, "  HaArenaRecycle at (1), chunk %p\n", (void *) chunk);
    next_chunk = chunk->u.s.next_chunk;
    if( HaChunkContainsArena(chunk, a) )
      arena_chunk = chunk;
    else
    {
      chunk->u.s.next_chunk = a->first_recycled_chunk;
      chunk->u.s.mem_avail_words = chunk->u.s.mem_total_words;
      memset(chunk->mem, 0, chunk->u.s.mem_total_words * sizeof(HA_ALIGN_TYPE));
      a->first_recycled_chunk = chunk;
    }
    chunk = next_chunk;
  } while( chunk != NULL );
  HaAssert(arena_chunk != NULL, L"HaArenaRecycle internal error 1");

  /* return arena_chunk to its state at point alpha in HaArenaMake */
  arena_words = HaBytesToWords(sizeof(struct ha_arena_rec));
  mem_avail_words = HA_INIT_CHUNK_WORDS - arena_words;
  res = (HA_ARENA) &arena_chunk->mem[mem_avail_words];
  HaAssert(res == a, L"HaArenaRecycle internal error 2");
  res->first_chunk = arena_chunk;
  arena_chunk->u.s.mem_avail_words = mem_avail_words;
  arena_chunk->u.s.next_chunk = NULL;
  memset(arena_chunk->mem, 0, mem_avail_words * sizeof(HA_ALIGN_TYPE));
  /* res->mem_total_words = HA_INIT_CHUNK_WORDS; already set in arena_chunk */

  /* continue from alpha as in HaArenaMake, except don't return res */
  res->u.block_list = HaBlockListMake(res, 0);
  HaAssert(res->u.block_list != NULL, L"HaArenaRecycle internal error 3");
  if( DEBUG2 )
    fprintf(stderr, "] HaArenaRecycle(%p)\n", (void *) a);
}


/*****************************************************************************/
/*                                                                           */
/*  void *HaArenaAllocWords(HA_ARENA a, size_t words)                        */
/*                                                                           */
/*  Allocate an ordinary block containing words words.                       */
/*                                                                           */
/*****************************************************************************/

static void *HaArenaAllocWords(HA_ARENA a, size_t words)
{
  HA_CHUNK chunk;  size_t new_mem_total_words;

  /* try the current chunk */
  chunk = a->first_chunk;
  if( chunk->u.s.mem_avail_words >= words )
    return &chunk->mem[chunk->u.s.mem_avail_words -= words];

  /* increase the chunk size; it's always 3 less than a power of 2 */
  new_mem_total_words = chunk->u.s.mem_total_words * 2 + 3;

  /* but before making a new chunk, try recycled chunks */
  chunk = a->first_recycled_chunk;
  while( chunk != NULL )
  {
    a->first_recycled_chunk = chunk->u.s.next_chunk;
    chunk->u.s.next_chunk = a->first_chunk;
    a->first_chunk = chunk;
    if( DEBUG4 )
      HaArenaDebug(a, "after moving in one recycled chunk", 0, stderr);
    if( chunk->u.s.mem_avail_words >= words )
      return &chunk->mem[chunk->u.s.mem_avail_words -= words];
    chunk = a->first_recycled_chunk;
  }

  /* get a new ordinary or large chunk, and allocate from it */
  if( words >= new_mem_total_words )
  {
    /* get a large chunk, fill it completely, make it second in list */
    chunk = HaCallocChunk(words, a);
    if( chunk == NULL )
      return NULL;
    a->allocated_words += words;
    if( DEBUG4 )
      HaArenaDebug(a, "after allocating a large chunk", 0, stderr);
    chunk->u.s.mem_total_words = words;
    chunk->u.s.mem_avail_words = 0;
    chunk->u.s.next_chunk = a->first_chunk->u.s.next_chunk;
    a->first_chunk->u.s.next_chunk = chunk;
    return &chunk->mem[0];
  }
  else
  {
    /* get an ordinary chunk, make it first in the list */
    chunk = HaCallocChunk(new_mem_total_words, a);
    if( chunk == NULL )
      return NULL;
    a->allocated_words += new_mem_total_words;
    if( DEBUG4 )
      HaArenaDebug(a, "after allocating an ordinary chunk", 0, stderr);
    chunk->u.s.mem_total_words = new_mem_total_words;
    chunk->u.s.mem_avail_words = new_mem_total_words - words;
    chunk->u.s.next_chunk = a->first_chunk;
    a->first_chunk = chunk;
    return &chunk->mem[chunk->u.s.mem_avail_words];
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void *HaAlloc(HA_ARENA a, size_t size)                                   */
/*                                                                           */
/*  Return a pointer to at least size bytes of fresh memory from arena a.    */
/*                                                                           */
/*****************************************************************************/

void *HaAlloc(HA_ARENA a, size_t size)
{
  return HaArenaAllocWords(a, HaBytesToWords(size));
}


/*****************************************************************************/
/*                                                                           */
/*  HA_ARENA HaResizableArena(void *resizable)                               */
/*                                                                           */
/*  Return *resizable's arena.                                               */
/*                                                                           */
/*****************************************************************************/

HA_ARENA HaResizableArena(void *resizable)
{
  HA_ALLOCATED_BLOCK block;
  block = (HA_ALLOCATED_BLOCK) &(((HA_ALIGN_TYPE *) resizable)[-1]);
  if( DEBUG1 )
    fprintf(stderr, "HaResizableArena(%p) returning %p\n", resizable,
      (void *) block->u.block_list->arena);
  return block->u.block_list->arena;
}


/*****************************************************************************/
/*                                                                           */
/*  size_t HaResizableSize(void *resizable)                                  */
/*                                                                           */
/*  Return the size of *resizable, in bytes.                                 */
/*                                                                           */
/*****************************************************************************/

size_t HaResizableSize(void *resizable)
{
  HA_ALLOCATED_BLOCK block;
  block = (HA_ALLOCATED_BLOCK) &(((HA_ALIGN_TYPE *) resizable)[-1]);
  return block->u.block_list->block_mem_words * sizeof(HA_ALIGN_TYPE);
}


/*****************************************************************************/
/*                                                                           */
/*  void *HaResizableAlloc(HA_ARENA a)                                       */
/*                                                                           */
/*  Return a resizable block of size 0 from a.                               */
/*                                                                           */
/*  Implementation note.  We use arena a itself as this block.  All that     */
/*  a block of size 0 needs is a u field; and a has a suitable u field.      */
/*                                                                           */
/*****************************************************************************/

void *HaResizableAlloc(HA_ARENA a)
{
  HA_ALLOCATED_BLOCK arena_as_block;
  arena_as_block = (HA_ALLOCATED_BLOCK) a;
  if( DEBUG1 )
    fprintf(stderr, "HaResizableAlloc(%p) returning %p\n",
      (void *) a, (void *) &arena_as_block->mem[0]);
  return &arena_as_block->mem[0];
}


/*****************************************************************************/
/*                                                                           */
/*  HA_ALLOCATED_BLOCK HaGetBlock(HA_BLOCK_LIST init_bl, size_t words)       */
/*                                                                           */
/*  Get a resizable block with words words or more from init_bl or from      */
/*  a later block list.                                                      */
/*                                                                           */
/*****************************************************************************/

static HA_ALLOCATED_BLOCK HaGetBlock(HA_BLOCK_LIST init_bl, size_t words)
{
  HA_BLOCK_LIST bl;  HA_ALLOCATED_BLOCK res;

  /* find the right block list, creating block lists as needed */
  for( bl = init_bl;  bl->block_mem_words < words;  bl = bl->next_block_list )
  {
    /* make sure there is a next block list, since this one won't do */
    if( bl->next_block_list == NULL )
    {
      bl->next_block_list = HaBlockListMake(bl->arena,
       bl->block_mem_words==0 ? HA_INIT_BLOCK_WORDS : 2*bl->block_mem_words+1);
      if( bl->next_block_list == NULL )
	return NULL;
    }
  }

  /* find a free block in bl, or allocate one */
  if( bl->first_free_block != NULL )
  {
    /* use an existing free block of bl */
    res = (HA_ALLOCATED_BLOCK) bl->first_free_block;
    bl->first_free_block = bl->first_free_block->next_free_block;
  }
  else
  {
    /* allocate a block containing bl->block_mem_words available words */
    res = (HA_ALLOCATED_BLOCK)
      HaArenaAllocWords(bl->arena, bl->block_mem_words + 1);
    if( res == NULL )
      return NULL;
    res->u.block_list = bl;
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void *HaResizableReAlloc(void *resizable, size_t size)                   */
/*                                                                           */
/*  Reallocate *resizable to a new block of at least size bytes, assuming    */
/*  that *resizable was allocated by HaResizableAlloc or HaResizableReAlloc. */
/*                                                                           */
/*****************************************************************************/

void *HaResizableReAlloc(void *resizable, size_t size)
{
  HA_BLOCK_LIST old_bl;  HA_FREE_BLOCK old_block;  size_t words;
  HA_ALLOCATED_BLOCK new_block;

  /* find resizable's block and block list */
  old_block = (HA_FREE_BLOCK) &(((HA_ALIGN_TYPE *) resizable)[-1]);
  old_bl = old_block->u.block_list;
  if( DEBUG1 )
  {
    fprintf(stderr, "[ HaResizableReAlloc(%p, %ld)\n", resizable, size);
    HaArenaDebug(old_bl->arena, "-", 2, stderr);
  }

  /* return resizable if it's enough */
  words = HaBytesToWords(size);
  if( old_bl->block_mem_words >= words )
  {
    if( DEBUG1 )
      fprintf(stderr, "] HaResizableReAlloc returning orig. %p\n", resizable);
    return resizable;
  }

  /* get a sufficiently large new block */
  new_block = HaGetBlock(old_bl, words);
  if( new_block == NULL )
  {
    if( DEBUG1 )
      fprintf(stderr, "] HaResizableReAlloc returning NULL\n");
    return NULL;
  }

  /* copy resizable into the new block */
  memcpy(&new_block->mem[0], resizable,
    old_bl->block_mem_words * sizeof(HA_ALIGN_TYPE));

  /* add old_block to its block list's free list, if it has non-zero size */
  if( old_bl->block_mem_words > 0 )
  {
    old_block->next_free_block = old_bl->first_free_block;
    old_bl->first_free_block = old_block;
  }

  /* return new_block's mem */
  if( DEBUG1 )
  {
    fprintf(stderr, "arena after HaResizableReAlloc:\n");
    HaArenaDebug(old_bl->arena, "-", 2, stderr);
    fprintf(stderr, "] HaResizableReAlloc returning %p\n",
      (void *) &new_block->mem[0]);
  }
  if( DEBUG5 )
    fprintf(stderr, "  %p = HaResizableReAlloc(%p, %ld)\n",
      (void *) &new_block->mem[0], resizable, size);
  return &new_block->mem[0];
}


/*****************************************************************************/
/*                                                                           */
/*  void HaResizableFree(void *resizable)                                    */
/*                                                                           */
/*  Free resizable, assuming it was allocated by HaResizableAlloc or         */
/*  HaResizableReAlloc.                                                      */
/*                                                                           */
/*****************************************************************************/

void HaResizableFree(void *resizable)
{
  HA_BLOCK_LIST old_bl;  HA_FREE_BLOCK old_block;

  /* find resizable's block and block list */
  old_block = (HA_FREE_BLOCK) &(((HA_ALIGN_TYPE *) resizable)[-1]);
  old_bl = old_block->u.block_list;
  if( DEBUG1 )
  {
    fprintf(stderr, "[ HaResizableFree(%p)\n", resizable);
    HaArenaDebug(old_bl->arena, "-", 2, stderr);
  }

  /* add old_block to its block list's free list, if it has non-zero size */
  if( old_bl->block_mem_words > 0 )
  {
    old_block->next_free_block = old_bl->first_free_block;
    old_bl->first_free_block = old_block;
  }
  if( DEBUG1 )
  {
    fprintf(stderr, "arena after HaResizableFree:\n");
    HaArenaDebug(old_bl->arena, "-", 2, stderr);
    fprintf(stderr, "] HaResizableFree returning\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void HaArenaDebug(HA_ARENA a, char *message, int indent, FILE *fp)       */
/*                                                                           */
/*  Debug print of arena a.                                                  */
/*                                                                           */
/*****************************************************************************/

void HaArenaDebug(HA_ARENA a, char *message, int indent, FILE *fp)
{
  /* HA_BLOCK_LIST bl;  HA_FREE_BLOCK fb; */  HA_CHUNK c;
  fprintf(fp, "%*s[ HaArena(%p%s) %s\n", indent, "", (void *) a,
    a->large ? ", large" : "", message);
  fprintf(fp, "%*s  allocated:", indent, "");
  for( c = a->first_chunk;  c != NULL;  c = c->u.s.next_chunk )
    fprintf(fp, " %ld", c->u.s.mem_total_words);
  fprintf(fp, "\n");
  if( a->first_recycled_chunk != NULL )
  {
    fprintf(fp, "%*s  recycled:", indent, "");
    for( c = a->first_recycled_chunk;  c != NULL;  c = c->u.s.next_chunk )
      fprintf(fp, " %ld", c->u.s.mem_total_words);
    fprintf(fp, "\n");
  }
  /* ***
  fprintf(stderr, "%*s  first_chunk %p\n", indent, "",
    (void *) a->first_chunk);
  for( bl = a->u.block_list;  bl != NULL;  bl = bl->next_block_list )
  {
    HaAssert(bl->arena == a, L"HaArenaDebug internal error 1");
    fprintf(stderr, "%*s  block list %p (block_mem_words %ld)\n",
      indent, "", (void *) bl, bl->block_mem_words);
    for( fb = bl->first_free_block;  fb != NULL;  fb = fb->next_free_block )
    {
      HaAssert(fb->u.block_list == bl, L"HaArenaDebug internal error 2");
      fprintf(stderr, "%*s    free block %p\n", indent, "",
	(void *) &fb->next_free_block);
    }
  }
  *** */
  fprintf(stderr, "%*s]\n", indent, "");
}
