
/*****************************************************************************/
/*                                                                           */
/*  THE HOWARD OBJECT-ORIENTED COMPILER TOOLKIT                              */
/*  COPYRIGHT (C) 2011 Jeffrey H. Kingston                                   */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 3, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to Free Software Foundation       */
/*  Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA               */
/*                                                                           */
/*  FILE:         ha_arena_set.c                                             */
/*  DESCRIPTION:  Arena sets                                                 */
/*                                                                           */
/*****************************************************************************/
#include "howard_a.h"
#include "howard_n.h"

#define DEBUG1 0
#define DEBUG2 0

/*****************************************************************************/
/*                                                                           */
/*  HA_ARENA_SET                                                             */
/*                                                                           */
/*****************************************************************************/

typedef HA_ARRAY(HA_ARENA) ARRAY_HA_ARENA;

struct ha_arena_set_rec
{
  ARRAY_HA_ARENA		arenas;		       /* the arenas         */
};


/*****************************************************************************/
/*                                                                           */
/*  Submodule "construction and query"                                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  HA_ARENA_SET HaArenaSetMake(HA_ARENA a)                                  */
/*                                                                           */
/*  Make a new arena set in arena a.                                         */
/*                                                                           */
/*****************************************************************************/

HA_ARENA_SET HaArenaSetMake(HA_ARENA a)
{
  HA_ARENA_SET res;
  HaMake(res, a);
  HaArrayInit(res->arenas, a);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void HaArenaSetClear(HA_ARENA_SET as)                                    */
/*                                                                           */
/*  Clear as back to the empty set of arenas.                                */
/*                                                                           */
/*****************************************************************************/

void HaArenaSetClear(HA_ARENA_SET as)
{
  HaArrayClear(as->arenas);
}


/*****************************************************************************/
/*                                                                           */
/*  void HaArenaSetDropFromEnd(KHE_ARENA_SET as, int n)                      */
/*                                                                           */
/*  Remove the last n arenas from as.                                        */
/*                                                                           */
/*****************************************************************************/

void HaArenaSetDropFromEnd(HA_ARENA_SET as, int n)
{
  HaArrayDeleteLastSlice(as->arenas, n);
}


/*****************************************************************************/
/*                                                                           */
/*  void HaArenaSetAddArena(HA_ARENA_SET as, HA_ARENA arena)                 */
/*                                                                           */
/*  Add arena to as.                                                         */
/*                                                                           */
/*****************************************************************************/

void HaArenaSetAddArena(HA_ARENA_SET as, HA_ARENA arena)
{
  int pos;
  if( DEBUG2 )
    HnAssert(!HaArrayContains(as->arenas, arena, &pos),
      "HaArenaSetAddArena: arena %p already present", (void *) arena);
  HaArrayAddLast(as->arenas, arena);
}


/*****************************************************************************/
/*                                                                           */
/*  HA_ARENA HaArenaSetLastAndDelete(HA_ARENA_SET as)                        */
/*                                                                           */
/*  Delete the last arena from as and return it.                             */
/*                                                                           */
/*****************************************************************************/

HA_ARENA HaArenaSetLastAndDelete(HA_ARENA_SET as)
{
  return HaArrayLastAndDelete(as->arenas);
}


/*****************************************************************************/
/*                                                                           */
/*  void HaArenaSetMerge(HA_ARENA_SET dest_as, HA_ARENA_SET src_as)          */
/*                                                                           */
/*  Merge the arenas of src_as into dest_as.                                 */
/*                                                                           */
/*****************************************************************************/

void HaArenaSetMerge(HA_ARENA_SET dest_as, HA_ARENA_SET src_as)
{
  while( HaArrayCount(src_as->arenas) > 0 )
    HaArrayAddLast(dest_as->arenas, HaArrayLastAndDelete(src_as->arenas));
}


/*****************************************************************************/
/*                                                                           */
/*  HA_ARENA HaArenaSetArenaBegin(HA_ARENA_SET as, bool large)               */
/*                                                                           */
/*  Return a fresh arena, either taken from as or newly created.             */
/*                                                                           */
/*****************************************************************************/

HA_ARENA HaArenaSetArenaBegin(HA_ARENA_SET as, bool large)
{
  HA_ARENA a;  int i;

  /* find and delete the last arena of as which is large or not as requested */
  if( as != NULL )
    HaArrayForEach(as->arenas, a, i)
      if( HaArenaLarge(a) == large )
      {
	HaArrayDeleteAndPlug(as->arenas, i);
	return a;
      }
  return HaArenaMake(large);
}


/*****************************************************************************/
/*                                                                           */
/*  void HaArenaSetArenaEnd(HA_ARENA_SET as, HA_ARENA a)                     */
/*                                                                           */
/*  We're finished with a.  Either recycle it through as or delete it.       */
/*                                                                           */
/*****************************************************************************/

void HaArenaSetArenaEnd(HA_ARENA_SET as, HA_ARENA a)
{
  if( DEBUG1 )
    fprintf(stderr, "[ HaArenaSetArenaEnd(%s, a)\n", as!=NULL ? "as" : "NULL");
  if( as != NULL )
  {
    HaArenaRecycle(a);
    HaArenaSetAddArena(as, a);
  }
  else
    HaArenaDelete(a);
  if( DEBUG1 )
    fprintf(stderr, "] HaArenaSetArenaEnd\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void HaArenaSetDeleteArena(HA_ARENA_SET as, HA_ARENA arena)              */
/*                                                                           */
/*  Delete arena from as.                                                    */
/*                                                                           */
/*****************************************************************************/

void HaArenaSetDeleteArena(HA_ARENA_SET as, HA_ARENA arena)
{
  int pos;
  if( !HaArrayContains(as->arenas, arena, &pos) )
    HnAbort("HaArenaSetDeleteArena:  as does not contain arena");
  HaArrayDeleteAndPlug(as->arenas, pos);
}


/*****************************************************************************/
/*                                                                           */
/*  bool HaArenaSetContainsArena(HA_ARENA_SET as, HA_ARENA arena,            */
/*    int *pos)                                                              */
/*                                                                           */
/*  Return true and set *pos if as contains arena.                           */
/*                                                                           */
/*****************************************************************************/

bool HaArenaSetContainsArena(HA_ARENA_SET as, HA_ARENA arena, int *pos)
{
  return HaArrayContains(as->arenas, arena, pos);
}


/*****************************************************************************/
/*                                                                           */
/*  int HaArenaSetArenaCount(HA_ARENA_SET as)                                */
/*                                                                           */
/*  Return the number of arenas in as.                                       */
/*                                                                           */
/*****************************************************************************/

int HaArenaSetArenaCount(HA_ARENA_SET as)
{
  return HaArrayCount(as->arenas);
}


/*****************************************************************************/
/*                                                                           */
/*  HA_ARENA HaArenaSetArena(HA_ARENA_SET as, int i)                         */
/*                                                                           */
/*  Return the i'th arena of as.                                             */
/*                                                                           */
/*****************************************************************************/

HA_ARENA HaArenaSetArena(HA_ARENA_SET as, int i)
{
  return HaArray(as->arenas, i);
}


/*****************************************************************************/
/*                                                                           */
/* void HaArenaSetDebug(HA_ARENA_SET as, int verbosity, int indent,          */
/*   FILE *fp)                                                               */
/*                                                                           */
/*  Debug print of arena set as onto fp with the given verbosity and indent. */
/*                                                                           */
/*****************************************************************************/

void HaArenaSetDebug(HA_ARENA_SET as, int verbosity, int indent, FILE *fp)
{
  HA_ARENA a;  int i;
  if( verbosity >= 1 && indent >= 0 )
  {
    fprintf(fp, "%*s[ Arena Set:\n", indent, "");
    HaArrayForEach(as->arenas, a, i)
      HaArenaDebug(a, "-", indent + 2, fp);
    fprintf(fp, "%*s]\n", indent, "");
  }
}
