
/*****************************************************************************/
/*                                                                           */
/*  THE HOWARD OBJECT-ORIENTED COMPILER TOOLKIT                              */
/*  COPYRIGHT (C) 2011 Jeffrey H. Kingston                                   */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 3, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to Free Software Foundation       */
/*  Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA               */
/*                                                                           */
/*  FILE:     ha_other.c                                                     */
/*  PURPOSE:  array and assert functions                                     */
/*                                                                           */
/*****************************************************************************/
#include "howard_a.h"
#include <string.h>
#include <stdarg.h>


/*****************************************************************************/
/*                                                                           */
/*  Submodule "arrays"                                                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  int HaArrayUniqSort(void *base, size_t nmemb, size_t size,               */
/*      int(*compar)(const void *, const void *))                            */
/*                                                                           */
/*  The parameters of HaArrayUniqSort are the same as for qsort and have the */
/*  same meaning.  This function first sorts using qsort, then uniqueifies   */
/*  the result array by removing each element which compares equal to the    */
/*  preceding element.  It returns the final number of elements.             */
/*                                                                           */
/*****************************************************************************/

int HaArrayUniqSort(void *base, size_t nmemb, size_t size,
    int(*compar)(const void *, const void *))
{
  int i, j;
  if( nmemb > 0 )
  {
    qsort(base, nmemb, size, compar);
    i = 0;
    for( j = 1;  j < nmemb;  j++ )
      if( compar(&((char *) base)[i * size], &((char *) base)[j * size]) != 0 )
      {
	i++;
	if( i != j )
	  memcpy(&((char *) base)[i * size], &((char *) base)[j * size], size);
      }
    return i + 1;
  }
  else
    return 0;
}


/*****************************************************************************/
/*                                                                           */
/*  bool HaArrayImplFind(char *a, char *target, size_t elem_size,            */
/*    int count, int *pos)                                                   */
/*                                                                           */
/*  Scan a in chunks of size elem_size, looking for *target.                 */
/*                                                                           */
/*****************************************************************************/

bool HaArrayImplFind(char *a, char *target, size_t elem_size,
  int count, int *pos)
{
  char *p;
  for( *pos = 0, p = a;  *pos < count;  (*pos)++, p += elem_size )
    if( memcmp(p, target, elem_size) == 0 )
      return true;
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "Abort and assert"                                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void HaAbort(wchar_t *fmt, ...)                                          */
/*                                                                           */
/*  Print an error message on stderr and abort.                              */
/*                                                                           */
/*****************************************************************************/

void HaAbort(wchar_t *fmt, ...)
{
  va_list args;
  va_start(args, fmt);
  vfwprintf(stderr, fmt, args);
  fwprintf(stderr, L"\n");
  va_end(args);
  abort();
}


/*****************************************************************************/
/*                                                                           */
/*  void HaAssert(bool cond, wchar_t *fmt, ...)                              */
/*                                                                           */
/*  If cond is false, print an error message on stderr and abort.            */
/*                                                                           */
/*****************************************************************************/

void HaAssert(bool cond, wchar_t *fmt, ...)
{
  va_list args;
  if( !cond )
  {
    va_start(args, fmt);
    vfwprintf(stderr, fmt, args);
    fwprintf(stderr, L"\n");
    va_end(args);
    abort();
  }
}
