
/*****************************************************************************/
/*                                                                           */
/*  THE KHE HIGH SCHOOL TIMETABLING ENGINE                                   */
/*  COPYRIGHT (C) 2010 Jeffrey H. Kingston                                   */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 3, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         khe_avail.c                                                */
/*  DESCRIPTION:  Resource availability                                      */
/*                                                                           */
/*****************************************************************************/
#include "khe_interns.h"

#define DEBUG1 0	/* high-level stuff */
#define DEBUG2 0
#define DEBUG3 0
#define DEBUG4 0
#define DEBUG5 0
#define DEBUG6 0


/*****************************************************************************/
/*                                                                           */
/*  PREASST_TIME_INFO - information about preassignments at one time         */
/*                                                                           */
/*****************************************************************************/

typedef struct preasst_time_info_rec {
  KHE_TIME			time;			/* the time          */
  bool				all_preassigned;	/* all tasks preass  */
  ARRAY_KHE_RESOURCE		resources;		/* resources         */
} *PREASST_TIME_INFO;

typedef HA_ARRAY(PREASST_TIME_INFO) ARRAY_PREASST_TIME_INFO;


/*****************************************************************************/
/*                                                                           */
/*  PREASST_INFO - preassignment information for one resource type           */
/*                                                                           */
/*****************************************************************************/

typedef struct preasst_info_rec {
  KHE_RESOURCE_TYPE		resource_type;		/* the resource type */
  ARRAY_PREASST_TIME_INFO	time_info;		/* indexed by time   */
} *PREASST_INFO;

typedef HA_ARRAY(PREASST_INFO) ARRAY_PREASST_INFO;


/*****************************************************************************/
/*                                                                           */
/*  WORKLOAD_RESOURCE - workload info for one resource                       */
/*                                                                           */
/*****************************************************************************/

typedef struct workload_resource_rec {
  KHE_RESOURCE			resource;		/* the resource r    */
  float				workload;		/* wup(r) or wpp(t,r)*/
} *WORKLOAD_RESOURCE;

typedef HA_ARRAY(WORKLOAD_RESOURCE) ARRAY_WORKLOAD_RESOURCE;


/*****************************************************************************/
/*                                                                           */
/*  WORKLOAD_TIME - workload info for one time                               */
/*                                                                           */
/*  workload_per_resource is not indexed by resource, that would be too      */
/*  expensive in space.  Instead it contains, for the given time t, one      */
/*  record for each resource r such that wpp(t, r) < infinity.               */
/*                                                                           */
/*****************************************************************************/

typedef struct workload_time_rec {
  KHE_TIME			time;			/* the time t        */
  float				workload;		/* wpu(t)            */
  ARRAY_WORKLOAD_RESOURCE	workload_per_resource;	/* wpp(t, r)         */
} *WORKLOAD_TIME;

typedef HA_ARRAY(WORKLOAD_TIME) ARRAY_WORKLOAD_TIME;


/*****************************************************************************/
/*                                                                           */
/*  WORKLOAD - workload information for one resource type                    */
/*                                                                           */
/*****************************************************************************/

typedef struct workload_rec {
  KHE_RESOURCE_TYPE		resource_type;		/* the resource type  */
  float				workload;		/* wuu                */
  ARRAY_WORKLOAD_TIME   	workload_per_time;	/* wpu(t); wpp(t, r)  */
  ARRAY_WORKLOAD_RESOURCE	workload_per_resource;	/* wup(r)             */
} *WORKLOAD;

typedef HA_ARRAY(WORKLOAD) ARRAY_WORKLOAD;


/*****************************************************************************/
/*                                                                           */
/*  AVAIL_NODE (private)                                                     */
/*                                                                           */
/*  One node in the graph for which we need an independent set.              */
/*                                                                           */
/*****************************************************************************/

typedef struct avail_node_rec {
  KHE_AVAIL_NODE_TYPE		type;			/* node type         */
  int				limit;			/* the limit         */
  KHE_TIME_SET			time_set;		/* the times         */
  KHE_MONITOR			monitor;		/* where from        */
} *AVAIL_NODE;

typedef HA_ARRAY(AVAIL_NODE) ARRAY_AVAIL_NODE;
typedef HP_TABLE(AVAIL_NODE) TABLE_AVAIL_NODE;


/*****************************************************************************/
/*                                                                           */
/*  AVAIL_SET - an independent set of avail nodes (private)                  */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  AVAIL_SET_TIMES,
  AVAIL_SET_WORKLOAD,
} AVAIL_SET_TYPE;

typedef struct avail_set_rec {
  AVAIL_SET_TYPE		type;
  int				total_limit;
  int				total_times;
  ARRAY_AVAIL_NODE		nodes;
} *AVAIL_SET;

typedef HA_ARRAY(AVAIL_SET) ARRAY_AVAIL_SET;


/*****************************************************************************/
/*                                                                           */
/*  KHE_AVAIL_SOLVER - an avail solver                                       */
/*                                                                           */
/*****************************************************************************/

typedef HA_ARRAY(KHE_TIME_SET) ARRAY_KHE_TIME_SET;

struct khe_avail_solver_rec {

  /* general fields */
  HA_ARENA			arena;		/* memory used by solver     */
  KHE_SOLN			soln;		/* soln being worked on      */
  KHE_INSTANCE			ins;		/* soln's instance           */
  int				ins_time_count;	/* no of times in instance   */
  ARRAY_PREASST_INFO		preasst_info;	/* indexed by resource type  */
  ARRAY_WORKLOAD		workload_array;	/* also indexed by rt        */

  /* resource-specific fields */
  KHE_RESOURCE			resource;	/* resource being worked on  */
  ARRAY_KHE_MONITOR		limit_and_cluster_monitors;
  ARRAY_KHE_LIMIT_WORKLOAD_MONITOR workload_monitors;
  TABLE_AVAIL_NODE		node_table;	/* avail nodes for resource  */
  ARRAY_AVAIL_NODE		node_array;	/* avail nodes for resource  */
  ARRAY_AVAIL_NODE		zero_time_nodes; /* indexed by time          */
  AVAIL_SET			best_busy_times; /* best busy times avail set*/
  AVAIL_SET			best_workload;	/* best workloads avail set  */
  HA_ARRAY_INT			cluster_limits;	/* for cluster constraints   */
  ARRAY_KHE_TIME_SET		cluster_time_sets; /* for cluster constraints*/
  HA_ARRAY_FLOAT		min_workloads;	/* for workload constraints  */

  /* free lists */
  ARRAY_KHE_TIME_SET		free_time_sets;			/* free list */
  ARRAY_AVAIL_NODE		free_avail_nodes;		/* free list */
  ARRAY_AVAIL_SET		free_avail_sets;		/* free list */
  ARRAY_PREASST_TIME_INFO	free_preasst_time_infos;	/* free list */
  ARRAY_WORKLOAD_TIME		free_workload_times;		/* free list */
  ARRAY_WORKLOAD_RESOURCE	free_workload_resources;	/* free list */
  ARRAY_WORKLOAD		free_workloads;			/* free list */
};


/*****************************************************************************/
/*                                                                           */
/*  Submodule "avail solver time info"                                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  PREASST_TIME_INFO KheAvailSolverMakePreasstTimeInfo(KHE_AVAIL_SOLVER as, */
/*    KHE_TIME t)                                                            */
/*                                                                           */
/*  Make a new, initially empty preasst time info object.                    */
/*                                                                           */
/*****************************************************************************/

static PREASST_TIME_INFO KheAvailSolverMakePreasstTimeInfo(KHE_AVAIL_SOLVER as,
  KHE_TIME t)
{
  PREASST_TIME_INFO res;
  if( HaArrayCount(as->free_preasst_time_infos) > 0 )
  {
    res = HaArrayLastAndDelete(as->free_preasst_time_infos);
    HaArrayClear(res->resources);
  }
  else
  {
    HaMake(res, as->arena);
    HaArrayInit(res->resources, as->arena);
  }
  res->time = t;
  res->all_preassigned = true;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheAvailSolverFreeTimeInfo(KHE_AVAIL_SOLVER as, PREASST_TIME_INFO ti)*/
/*                                                                           */
/*  Free ti, recycling its memory through as's free lists.                   */
/*                                                                           */
/*****************************************************************************/

static void KheAvailSolverFreeTimeInfo(KHE_AVAIL_SOLVER as, PREASST_TIME_INFO ti)
{
  HaArrayAddLast(as->free_preasst_time_infos, ti);
}


/*****************************************************************************/
/*                                                                           */
/*  void TimeInfoDebug(PREASST_TIME_INFO ti, int verbosity, int indent,      */
/*    FILE *fp)                                                              */
/*                                                                           */
/*  Debug print of ti onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void TimeInfoDebug(PREASST_TIME_INFO ti, int verbosity, int indent, FILE *fp)
{
  int i;  KHE_RESOURCE r;
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  fprintf(fp, "[%s %sall preassigned: ", KheTimeId(ti->time),
    ti->all_preassigned ? "" : "not ");
  HaArrayForEach(ti->resources, r, i)
    fprintf(fp, "%s%s", i == 0 ? "" : ", ", KheResourceId(r));
  fprintf(fp, "]");
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "avail solver preasst info"                                    */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KhePreasstInfoAddEvent(PREASST_INFO pi, KHE_EVENT e)                */
/*                                                                           */
/*  Examine e and update pi depending on what is found.                      */
/*                                                                           */
/*****************************************************************************/

static void KhePreasstInfoAddEvent(PREASST_INFO pi, KHE_EVENT e)
{
  KHE_TIME start_t, t;  PREASST_TIME_INFO ti;  int i, j;  KHE_EVENT_RESOURCE er;
  KHE_RESOURCE r;
  start_t = KheEventPreassignedTime(e);
  HnAssert(start_t != NULL, "KhePreasstInfoAddEvent internal error");
  for( i = 0;  i < KheEventDuration(e);  i++ )
  {
    t = KheTimeNeighbour(start_t, i);
    ti = HaArray(pi->time_info, KheTimeIndex(t));
    for( j = 0;  j < KheEventResourceCount(e);  j++ )
    {
      er = KheEventResource(e, j);
      if( KheEventResourceResourceType(er) == pi->resource_type )
      {
	r = KheEventResourcePreassignedResource(er);
	if( r == NULL )
	  ti->all_preassigned = false;
	else
	  HaArrayAddLast(ti->resources, r);
      }
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KhePreasstInfoCompressTimeInfo(PREASST_INFO pi, KHE_AVAIL_SOLVER as)*/
/*                                                                           */
/*  Delete and free all unpreassigned time info records.                     */
/*                                                                           */
/*****************************************************************************/

static void KhePreasstInfoCompressTimeInfo(PREASST_INFO pi, KHE_AVAIL_SOLVER as)
{
  int j, k, dropped;  PREASST_TIME_INFO ti;
  /* j is where to put the next all preassigned ti; k is the next to try */
  j = dropped = 0;
  HaArrayForEach(pi->time_info, ti, k)
  {
    if( !ti->all_preassigned )
    {
      if( DEBUG3 )
      {
	fprintf(stderr, "  compression at %d freeing ", k);
	TimeInfoDebug(ti, 2, 0, stderr);
      }
      KheAvailSolverFreeTimeInfo(as, ti), dropped++;
    }
    else
    {
      if( DEBUG3 )
      {
	fprintf(stderr, "  compression at %d moving to %d ", k, j);
	TimeInfoDebug(ti, 2, 0, stderr);
      }
      HaArrayPut(pi->time_info, j++, ti);
    }
  }
  HaArrayDeleteLastSlice(pi->time_info, dropped);
}


/*****************************************************************************/
/*                                                                           */
/*  PREASST_INFO KheAvailSolverMakePreasstInfo(KHE_AVAIL_SOLVER as,          */
/*    KHE_RESOURCE_TYPE rt)                                                  */
/*                                                                           */
/*  Make and completely set up a preasst_info object for resource type rt.   */
/*                                                                           */
/*  Note.  There is no KheAvailSolverFreePreasstInfo, because preasst_info   */
/*  objects have the full lifetime of the solver.                            */
/*                                                                           */
/*****************************************************************************/
static void PreasstInfoDebug(PREASST_INFO pi, int verbosity, int indent,
  FILE *fp);

static PREASST_INFO KheAvailSolverMakePreasstInfo(KHE_AVAIL_SOLVER as,
  KHE_RESOURCE_TYPE rt)
{
  PREASST_INFO res;  KHE_INSTANCE ins;  PREASST_TIME_INFO ti;  KHE_TIME t;  int i;
  if( DEBUG3 )
    fprintf(stderr, "[ KheAvailSolverMakePreasstInfo(as, %s)\n",
      KheResourceTypeId(rt));

  /* make the object and its time info components, one for each time */
  ins = KheResourceTypeInstance(rt);
  HaMake(res, as->arena);
  res->resource_type = rt;
  HaArrayInit(res->time_info, as->arena);
  for( i = 0;  i < KheInstanceTimeCount(ins);  i++ )
  {
    t = KheInstanceTime(ins, i);
    ti = KheAvailSolverMakePreasstTimeInfo(as, t);
    HaArrayAddLast(res->time_info, ti);
  }

  /* set the all_preassigned and resources fields of the time info objects */
  for( i = 0;  i < KheInstanceEventCount(ins);  i++ )
    KhePreasstInfoAddEvent(res, KheInstanceEvent(ins, i));

  /* compress the time info objects */
  if( DEBUG3 )
  {
    fprintf(stderr, "  KheAvailSolverMakePreasstInfo before compressing:\n");
    PreasstInfoDebug(res, 2, 4, stderr);
  }
  KhePreasstInfoCompressTimeInfo(res, as);

  /* uniqueify the resources arrays of the time info fields */
  HaArrayForEach(res->time_info, ti, i)
    HaArraySortUnique(ti->resources, &KheResourceCmp);
  if( DEBUG3 )
  {
    fprintf(stderr, "  KheAvailSolverMakePreasstInfo at end\n");
    PreasstInfoDebug(res, 2, 4, stderr);
    fprintf(stderr, "] KheAvailSolverMakePreasstInfo returning\n");
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void PreasstInfoDebug(PREASST_INFO pi, int verbosity, int indent,        */
/*    FILE *fp)                                                              */
/*                                                                           */
/*  Debug print of pi onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void PreasstInfoDebug(PREASST_INFO pi, int verbosity, int indent,
  FILE *fp)
{
  PREASST_TIME_INFO ti;  int i;
  fprintf(fp, "%*s[ PreasstInfo(%s)\n", indent, "",
    KheResourceTypeId(pi->resource_type));
  HaArrayForEach(pi->time_info, ti, i)
    TimeInfoDebug(ti, verbosity, indent + 2, fp);
  fprintf(fp, "%*s]\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "avail solver workload info"                                   */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  WORKLOAD_RESOURCE WorkloadResourceMake(KHE_RESOURCE r, float w,          */
/*    KHE_AVAIL_SOLVER as)                                                   */
/*                                                                           */
/*  Make a new workload resource object with these attributes.               */
/*                                                                           */
/*****************************************************************************/

static WORKLOAD_RESOURCE WorkloadResourceMake(KHE_RESOURCE r, float w,
  KHE_AVAIL_SOLVER as)
{
  WORKLOAD_RESOURCE res;
  if( HaArrayCount(as->free_workload_resources) > 0 )
    res = HaArrayLastAndDelete(as->free_workload_resources);
  else
    HaMake(res, as->arena);
  res->resource = r;
  res->workload = w;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void WorkloadResourceFree(WORKLOAD_RESOURCE wr, KHE_AVAIL_SOLVER as)     */
/*                                                                           */
/*  Free wr by returning it to the free list in as.                          */
/*                                                                           */
/*****************************************************************************/

/* *** unused
static void WorkloadResourceFree(WORKLOAD_RESOURCE wr, KHE_AVAIL_SOLVER as)
{
  HaArrayAddLast(as->free_workload_resources, wr);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  WORKLOAD_TIME WorkloadTimeMake(KHE_TIME t, float w, KHE_AVAIL_SOLVER as) */
/*                                                                           */
/*  Make a new workload time object with these attributes.                   */
/*                                                                           */
/*****************************************************************************/

static WORKLOAD_TIME WorkloadTimeMake(KHE_TIME t, float w, KHE_AVAIL_SOLVER as)
{
  WORKLOAD_TIME res;
  if( HaArrayCount(as->free_workload_resources) > 0 )
  {
    res = HaArrayLastAndDelete(as->free_workload_times);
    HaArrayClear(res->workload_per_resource);
  }
  else
  {
    HaMake(res, as->arena);
    HaArrayInit(res->workload_per_resource, as->arena);
  }
  res->time = t;
  res->workload = w;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void WorkloadTimeFree(WORKLOAD_TIME wt, KHE_AVAIL_SOLVER as)             */
/*                                                                           */
/*  Free wt by returning it to the free list in as.                          */
/*                                                                           */
/*****************************************************************************/

/* *** unused
static void WorkloadTimeFree(WORKLOAD_TIME wt, KHE_AVAIL_SOLVER as)
{
  HaArrayAddLast(as->free_workload_times, wt);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  WORKLOAD WorkloadMake(KHE_RESOURCE_TYPE rt, float w, KHE_AVAIL_SOLVER as)*/
/*                                                                           */
/*  Make a new workload object with these attributes.                        */
/*                                                                           */
/*****************************************************************************/

static WORKLOAD WorkloadMake(KHE_RESOURCE_TYPE rt, float w, KHE_AVAIL_SOLVER as)
{
  WORKLOAD res;
  if( HaArrayCount(as->free_workloads) > 0 )
  {
    res = HaArrayLastAndDelete(as->free_workloads);
    HaArrayClear(res->workload_per_time);
    HaArrayClear(res->workload_per_resource);
  }
  else
  {
    HaMake(res, as->arena);
    HaArrayInit(res->workload_per_time, as->arena);
    HaArrayInit(res->workload_per_resource, as->arena);
  }
  res->resource_type = rt;
  res->workload = w;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void WorkloadFree(WORKLOAD wk, KHE_AVAIL_SOLVER as)                      */
/*                                                                           */
/*  Free wk by returning it to the free list in as.                          */
/*                                                                           */
/*****************************************************************************/

/* *** unused
static void WorkloadFree(WORKLOAD wk, KHE_AVAIL_SOLVER as)
{
  HaArrayAddLast(as->free_workloads, wk);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool WorkloadTimeContainsResource(WORKLOAD_TIME wt, KHE_RESOURCE r,      */
/*    WORKLOAD_RESOURCE *wr)                                                 */
/*                                                                           */
/*  If wt contains a workload resource object for r, return true with *wr    */
/*  set to that object.  Otherwise return false wit *wr set to NULL.         */
/*                                                                           */
/*****************************************************************************/

static bool WorkloadTimeContainsResource(WORKLOAD_TIME wt, KHE_RESOURCE r,
  WORKLOAD_RESOURCE *wr)
{
  WORKLOAD_RESOURCE wr2;  int i;
  HaArrayForEach(wt->workload_per_resource, wr2, i)
    if( wr2->resource == r )
      return *wr = wr2, true;
  return *wr = NULL, false;
}


/*****************************************************************************/
/*                                                                           */
/*  WORKLOAD_RESOURCE GetWorkloadResource(WORKLOAD_TIME wt, KHE_RESOURCE r,  */
/*    KHE_AVAIL_SOLVER as)                                                   */
/*                                                                           */
/*  Return the workload resource object for r within wt.  This may involve   */
/*  creating the object if it is not present already.                        */
/*                                                                           */
/*****************************************************************************/

static WORKLOAD_RESOURCE GetWorkloadResource(WORKLOAD_TIME wt, KHE_RESOURCE r,
  KHE_AVAIL_SOLVER as)
{
  WORKLOAD_RESOURCE wr;

  /* if present already, return it */
  if( WorkloadTimeContainsResource(wt, r, &wr) )
    return wr;

  /* otherwise create a new object, add it to wt, and return it */
  wr = WorkloadResourceMake(r, 0.0, as);
  HaArrayAddLast(wt->workload_per_resource, wr);
  return wr;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheAvailSolverInitialiseWorkloadInfo(KHE_AVAIL_SOLVER as,           */
/*    KHE_RESOURCE_TYPE rt)                                                  */
/*                                                                           */
/*  Initialize workload information for resource type rt.                    */
/*                                                                           */
/*****************************************************************************/
static void WorkloadDebug(WORKLOAD wk, int verbosity, int indent, FILE *fp);

static void KheAvailSolverInitialiseWorkloadInfo(KHE_AVAIL_SOLVER as,
  KHE_RESOURCE_TYPE rt)
{
  WORKLOAD wk;  int i, j, k;  KHE_RESOURCE r;  KHE_TIME t;  KHE_EVENT e;
  KHE_EVENT_RESOURCE er;  float workload_per_time;  WORKLOAD_RESOURCE wr;
  WORKLOAD_TIME wt;
  wk = HaArray(as->workload_array, KheResourceTypeIndex(rt));
  if( wk == NULL )
  {
    /* initialize the workload object for resource type rt */
    wk = WorkloadMake(rt, FLT_MAX, as);
    HaArrayPut(as->workload_array, KheResourceTypeIndex(rt), wk);

    /* one workload per resource object for each resource of type rt */
    for( i = 0;  i < KheResourceTypeResourceCount(rt);  i++ )
    {
      r = KheResourceTypeResource(rt, i);
      HaArrayAddLast(wk->workload_per_resource,
	WorkloadResourceMake(r, FLT_MAX, as));
    }

    /* one workload per time object for each time of the instance */
    for( i = 0;  i < KheInstanceTimeCount(as->ins);  i++ )
    {
      t = KheInstanceTime(as->ins, i);
      HaArrayAddLast(wk->workload_per_time, WorkloadTimeMake(t, FLT_MAX, as));
    }

    /* add every event resource to the structure */
    for( i = 0;  i < KheInstanceEventCount(as->ins);  i++ )
    {
      e = KheInstanceEvent(as->ins, i);
      t = KheEventPreassignedTime(e);
      for( j = 0;  j < KheEventResourceCount(e);  j++ )
      {
	er = KheEventResource(e, j);
	if( KheEventResourceResourceType(er) == rt )
	{
	  /* get er's preassigned resource and workload per time */
	  r = KheEventResourcePreassignedResource(er);
	  workload_per_time = (float) KheEventResourceWorkload(er) /
	    (float) KheEventDuration(e);

	  /* update wuu, wup, wpu, or wpp to take account of er's workload */
	  if( t == NULL )
	  {
	    if( r == NULL )
	    {
	      /* no preassigned time, no preassigned resource */
	      if( workload_per_time < wk->workload )
                wk->workload = workload_per_time;
	    }
	    else
	    {
	      /* no preassigned time, preassigned resource r */
	      wr = HaArray(wk->workload_per_resource,
		KheResourceResourceTypeIndex(r));
	      if( workload_per_time < wr->workload )
                wr->workload = workload_per_time;
	    }
	  }
	  else
	  {
	    for( k = 0;  k < KheEventDuration(e);  k++ )
	    {
	      wt = HaArray(wk->workload_per_time, KheTimeIndex(t) + k);
	      if( r == NULL )
	      {
		/* preassigned time t + k, no preassigned resource */
		if( workload_per_time < wt->workload )
		  wt->workload = workload_per_time;
	      }
	      else
	      {
		/* preassigned time t + k, preassigned resource r */
		wr = GetWorkloadResource(wt, r, as);
		wr->workload += workload_per_time;
	      }
	    }
	  }
	}
      }
    }
    if( DEBUG4 )
      WorkloadDebug(wk, 2, 2, stderr);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  WORKLOAD_TYPE KheAvailSolverWorkloadPerTime(KHE_AVAIL_SOLVER as,         */
/*    KHE_TIME t, KHE_RESOURCE r, float *workload)                           */
/*                                                                           */
/*  Assuming that the data structures have been initialized, return the      */
/*  workload per time of r at t in *workload, and its type as result:        */
/*                                                                           */
/*    WORKLOAD_PREASST      r is preassigned at time t                       */
/*    WORKLOAD_NOT_BUSY     r cannot be busy at time t                       */
/*    WORKLOAD_DFT          neither of the above                             */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  WORKLOAD_PREASST,
  WORKLOAD_NOT_BUSY,
  WORKLOAD_DFT
} WORKLOAD_TYPE;

static WORKLOAD_TYPE KheAvailSolverWorkloadPerTime(KHE_AVAIL_SOLVER as,
  KHE_TIME t, KHE_RESOURCE r, float *workload)
{
  WORKLOAD wk;  WORKLOAD_TIME wt;  WORKLOAD_RESOURCE wr;  float res;

  /* first, see if there is a non-zero wpp */
  wk = HaArray(as->workload_array,
    KheResourceTypeIndex(KheResourceResourceType(r)));
  wt = HaArray(wk->workload_per_time, KheTimeIndex(t));
  if( WorkloadTimeContainsResource(wt, r, &wr) )
    return *workload = wr->workload, WORKLOAD_PREASST;

  /* otherwise, return min(wuu, wup(r), wpu(t)) */
  res = wk->workload;
  if( wt->workload < res )
    res = wt->workload;
  wr = HaArray(wk->workload_per_resource, KheResourceResourceTypeIndex(r));
  if( wr->workload < res )
    res = wr->workload;
  return *workload = res, (res < FLT_MAX ? WORKLOAD_DFT : WORKLOAD_NOT_BUSY);
}


/*****************************************************************************/
/*                                                                           */
/*  void WorkloadResourceDebug(WORKLOAD_RESOURCE wr, int verbosity,          */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of wr onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static char *WorkloadShow(float workload, char buff[50])
{
  if( workload == FLT_MAX )
    sprintf(buff, "%s", "inf");
  else
    sprintf(buff, "%.1f", workload);
  return buff;
}

static void WorkloadResourceDebug(WORKLOAD_RESOURCE wr, int verbosity,
  int indent, FILE *fp)
{
  char buff[50];
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  fprintf(fp, "%s:%s", KheResourceId(wr->resource),
    WorkloadShow(wr->workload, buff));
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void WorkloadTimeDebug(WORKLOAD_TIME wt, int verbosity,                  */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of wt onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

void WorkloadTimeDebug(WORKLOAD_TIME wt, int verbosity, int indent, FILE *fp)
{
  WORKLOAD_RESOURCE wr;  int i;  char buff[50];
  fprintf(fp, "%*s[ Time %s, wpu %s: ", indent, "", KheTimeId(wt->time),
    WorkloadShow(wt->workload, buff));
  HaArrayForEach(wt->workload_per_resource, wr, i)
  {
    if( i > 0 )
      fprintf(fp, ", ");
    WorkloadResourceDebug(wr, verbosity, -1, fp);
  }
  fprintf(fp, " ]\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void WorkloadDebug(WORKLOAD wk, int verbosity, int indent, FILE *fp)     */
/*                                                                           */
/*  Debug print of wk onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void WorkloadDebug(WORKLOAD wk, int verbosity, int indent, FILE *fp)
{
  WORKLOAD_TIME wt;  WORKLOAD_RESOURCE wr;  int i;  char buff[50];
  fprintf(fp, "%*s[ Workload(%s) wuu = %s\n", indent, "",
    KheResourceTypeId(wk->resource_type), WorkloadShow(wk->workload, buff));
  HaArrayForEach(wk->workload_per_time, wt, i)
    WorkloadTimeDebug(wt, verbosity, indent + 2, fp);
  HaArrayForEach(wk->workload_per_resource, wr, i)
    WorkloadResourceDebug(wr, verbosity, indent + 2, fp);
  fprintf(fp, "%*s]\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "avail solver time sets" (private)                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_TIME_SET KheAvailSolverMakeTimeSet(KHE_AVAIL_SOLVER as)              */
/*                                                                           */
/*  Get a new empty time set from as's free list, or make one.               */
/*                                                                           */
/*****************************************************************************/

static KHE_TIME_SET KheAvailSolverMakeTimeSet(KHE_AVAIL_SOLVER as)
{
  KHE_TIME_SET res;
  if( HaArrayCount(as->free_time_sets) > 0 )
  {
    res = HaArrayLastAndDelete(as->free_time_sets);
    KheTimeSetClear(res);
  }
  else
    res = KheTimeSetMake(as->ins, as->arena);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheAvailSolverFreeTimeSet(KHE_AVAIL_SOLVER as, KHE_TIME_SET ts)     */
/*                                                                           */
/*  Free ts.                                                                 */
/*                                                                           */
/*****************************************************************************/

static void KheAvailSolverFreeTimeSet(KHE_AVAIL_SOLVER as, KHE_TIME_SET ts)
{
  HaArrayAddLast(as->free_time_sets, ts);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "avail solver avail nodes"                                     */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  AVAIL_NODE AvailSolverMakeAvailNode(KHE_AVAIL_SOLVER as,                 */
/*    KHE_AVAIL_NODE_TYPE type, int limit, KHE_TIME_SET ts, KHE_MONITOR m)   */
/*                                                                           */
/*  Make a new avail node with these attributes.                             */
/*                                                                           */
/*  No checks for suitability are made here.  This function is called        */
/*  from only one place, AvailSolverMakeAndAddAvailNode, and it is the       */
/*  job of that function to make those checks.                               */
/*                                                                           */
/*****************************************************************************/

static AVAIL_NODE AvailSolverMakeAvailNode(KHE_AVAIL_SOLVER as,
  KHE_AVAIL_NODE_TYPE type, int limit, KHE_TIME_SET ts, KHE_MONITOR m)
{
  AVAIL_NODE res;
  if( HaArrayCount(as->free_avail_nodes) > 0 )
    res = HaArrayLastAndDelete(as->free_avail_nodes);
  else
    HaMake(res, as->arena);
  res->type = type;
  res->limit = limit;
  res->time_set = ts;
  res->monitor = m;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void UpdateZeroTimeNodes(KHE_AVAIL_SOLVER as, AVAIL_NODE an)             */
/*                                                                           */
/*  If an is a zero time node, update the zero time nodes array.             */
/*                                                                           */
/*****************************************************************************/

static void UpdateZeroTimeNodes(KHE_AVAIL_SOLVER as, AVAIL_NODE an)
{
  KHE_TIME t;
  if( an->limit == 0 && KheTimeSetTimeCount(an->time_set) == 1 )
  {
    t = KheTimeSetTime(an->time_set, 0);
    HaArrayPut(as->zero_time_nodes, KheTimeIndex(t), an);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void AvailSolverMakeAndAddAvailNode(KHE_AVAIL_SOLVER as,                 */
/*    KHE_AVAIL_NODE_TYPE type, int limit, KHE_TIME_SET ts, KHE_MONITOR m)   */
/*                                                                           */
/*  Make and add an avail node with these attributes, unless it would add    */
/*  nothing useful or there is already something as good or better.          */
/*                                                                           */
/*  Care is needed with ts:  it is undefined after this operation, because   */
/*  if a new node is not created, it is freed.                               */
/*                                                                           */
/*****************************************************************************/

static void AvailSolverMakeAndAddAvailNode(KHE_AVAIL_SOLVER as,
  KHE_AVAIL_NODE_TYPE type, int limit, KHE_TIME_SET ts, KHE_MONITOR m)
{
  AVAIL_NODE an;  int pos;
  HnAssert(limit >= 0, "AvailSolverMakeAndAddAvailNode internal error");
  if( limit >= KheTimeSetTimeCount(ts) )
  {
    /* ignore this call because it adds nothing useful */
    KheAvailSolverFreeTimeSet(as, ts);
  }
  else if( !HpTableRetrieve(as->node_table, (void *) ts, an, pos) )
  {
    /* ts is new, make a node and add it to the table */
    an = AvailSolverMakeAvailNode(as, type, limit, ts, m);
    HpTableAdd(as->node_table, (void *) ts, an);
    HaArrayAddLast(as->node_array, an);
    UpdateZeroTimeNodes(as, an);
  }
  else if( limit < an->limit )
  {
    /* ts is already present but this call is stronger, so replace */
    an->type = type;
    an->limit = limit;
    an->monitor = m;
    KheAvailSolverFreeTimeSet(as, ts);
    UpdateZeroTimeNodes(as, an);
  }
  else
  {
    /* ignore this call because something at least as good is already there */
    KheAvailSolverFreeTimeSet(as, ts);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void AvailSolverMakeAndAddAvailNodeAndSubsets(KHE_AVAIL_SOLVER as,       */
/*    KHE_AVAIL_NODE_TYPE type, int limit, KHE_TIME_SET ts, KHE_MONITOR m)   */
/*                                                                           */
/*  Add an avail node with these attributes, and also add one avail node     */
/*  for each time set which is a subset of ts with one less element, and     */
/*  the same other attributes.  But only add the subsets if they are         */
/*  useful, that is, if they contain at least one time and limit places      */
/*  a non-trivial constraint on their number.                                */
/*                                                                           */
/*  Care is needed with ts:  it is undefined after this operation.           */
/*                                                                           */
/*****************************************************************************/

/* *** not doing subsets any more
static void AvailSolverMakeAndAddAvailNodeAndSubsets(KHE_AVAIL_SOLVER as,
  KHE_AVAIL_NODE_TYPE type, int limit, KHE_TIME_SET ts, KHE_MONITOR m)
{
  int omit, i, count;  AVAIL_NODE an;

  ** add a node containing ts; this makes ts undefined **
  an = AvailSolverMakeAndAddAvailNode(as, type, limit, ts, m);

  ** build and add subsets, where useful **
  if( an != NULL )
  {
    count = KheTimeSetTimeCount(an->time_set);
    if( count >= 2 && limit < count - 1 )
    {
      for( omit = 0;  omit < KheTimeSetTimeCount(an->time_set);  omit++ )
      {
	** build and add a time set, omitting the element at omit **
	ts = KheAvailSolv erMakeTimeSet(as);
	for( i = 0;  i < count;  i++ )
	  if( i != omit )
	    KheTimeSetAddTime(ts, KheTimeSetTime(an->time_set, i));
	AvailSolverMakeAndAddAvailNode(as, type, limit, ts, m);
      }
    }
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void AvailSolverFreeAvailNode(KHE_AVAIL_SOLVER as, AVAIL_NODE an)        */
/*                                                                           */
/*  Free an.  Also free its time set.                                        */
/*                                                                           */
/*****************************************************************************/

static void AvailSolverFreeAvailNode(KHE_AVAIL_SOLVER as, AVAIL_NODE an)
{
  KheAvailSolverFreeTimeSet(as, an->time_set);
  HaArrayAddLast(as->free_avail_nodes, an);
}


/*****************************************************************************/
/*                                                                           */
/*  int AvailNodeCmp(const void *t1, const void *t2)                         */
/*                                                                           */
/*  Comparison function for sorting avail nodes by decreasing value, with    */
/*  ties broken by KheTimeSetTypedCmp.                                       */
/*                                                                           */
/*****************************************************************************/

static int AvailNodeCmp(const void *t1, const void *t2)
{
  AVAIL_NODE an1 = * (AVAIL_NODE *) t1;
  AVAIL_NODE an2 = * (AVAIL_NODE *) t2;
  int an1_value = KheTimeSetTimeCount(an1->time_set) - an1->limit;
  int an2_value = KheTimeSetTimeCount(an2->time_set) - an2->limit;
  if( an1_value != an2_value )
    return an2_value - an1_value;
  else
    return KheTimeSetTypedCmp(an1->time_set, an2->time_set);
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheAvailNodeTypeShow(KHE_AVAIL_NODE_TYPE type)                     */
/*                                                                           */
/*  Return a short phrase in static memory describing type.                  */
/*                                                                           */
/*****************************************************************************/

char *KheAvailNodeTypeShow(KHE_AVAIL_NODE_TYPE type)
{
  switch( type )
  {
    case KHE_AVAIL_NODE_UNASSIGNABLE_TIME:

      return "Unassignable time";

    case KHE_AVAIL_NODE_UNAVAILABLE_TIME:

      return "Unavailable time";

    case KHE_AVAIL_NODE_LIMIT_BUSY_ZERO:

      return "Limit busy times constraint with zero max limit";

    case KHE_AVAIL_NODE_CLUSTER_BUSY_ZERO:

      return "Cluster busy times constraint with zero max limit";

    case KHE_AVAIL_NODE_LIMIT_BUSY:

      return "Limit busy times constraint with non-zero max limit";

    case KHE_AVAIL_NODE_CLUSTER_BUSY:

      return "Cluster busy times constraint with non-zero max limit";

    case KHE_AVAIL_NODE_CLUSTER_BUSY_MIN:

      return "Cluster busy times constraint with min limit";

    case KHE_AVAIL_NODE_WORKLOAD:

      return "Limit workload constraint";

    default:

      HnAbort("KheAvailNodeTypeShow: invalid type (%d)", type);
      return NULL;  /* keep compiler happy */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheAvailNodeTypeDebug(KHE_AVAIL_NODE_TYPE type)                    */
/*                                                                           */
/*  Return type as a string.                                                 */
/*                                                                           */
/*****************************************************************************/

static char *KheAvailNodeTypeDebug(KHE_AVAIL_NODE_TYPE type)
{
  switch( type )
  {
    case KHE_AVAIL_NODE_UNASSIGNABLE_TIME:

      return "UNASSIGNABLE_TIME";

    case KHE_AVAIL_NODE_UNAVAILABLE_TIME:

      return "UNAVAILABLE_TIME";

    case KHE_AVAIL_NODE_LIMIT_BUSY_ZERO:

      return "LIMIT_BUSY_ZERO";

    case KHE_AVAIL_NODE_CLUSTER_BUSY_ZERO:

      return "CLUSTER_BUSY_ZERO";

    case KHE_AVAIL_NODE_LIMIT_BUSY:

      return "LIMIT_BUSY";

    case KHE_AVAIL_NODE_CLUSTER_BUSY:

      return "CLUSTER_BUSY";

    case KHE_AVAIL_NODE_CLUSTER_BUSY_MIN:

      return "CLUSTER_BUSY_MIN";

    case KHE_AVAIL_NODE_WORKLOAD:

      return "WORKLOAD";

    default:

      HnAbort("KheAvailNodeTypeDebug: invalid type (%d)", type);
      return NULL;  /* keep compiler happy */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void AvailNodeDebug(AVAIL_NODE an, int verbosity, int indent, FILE *fp)  */
/*                                                                           */
/*  Debug print of an onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void AvailNodeDebug(AVAIL_NODE an, int verbosity, int indent, FILE *fp)
{
  if( indent >= 0 )
    fprintf(fp, "%*s", indent, "");
  fprintf(fp, "AvailNode(%s %d ", KheAvailNodeTypeDebug(an->type), an->limit);
  KheTimeSetDebug(an->time_set, verbosity, -1, fp);
  if( an->monitor != NULL )
    fprintf(fp, ", %s", KheMonitorId(an->monitor));
  fprintf(fp, ")\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "avail sets" - independent sets of avail nodes                 */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  AVAIL_SET AvailSolverMakeAvailSet(KHE_AVAIL_SOLVER as,                   */
/*    AVAIL_SET_TYPE type)                                                   */
/*                                                                           */
/*  Make a new, empty set of avail nodes.                                    */
/*                                                                           */
/*****************************************************************************/

static AVAIL_SET AvailSolverMakeAvailSet(KHE_AVAIL_SOLVER as,
  AVAIL_SET_TYPE type)
{
  AVAIL_SET res;
  if( HaArrayCount(as->free_avail_sets) > 0 )
  {
    res = HaArrayLastAndDelete(as->free_avail_sets);
    HaArrayClear(res->nodes);
  }
  else
  {
    HaMake(res, as->arena);
    HaArrayInit(res->nodes, as->arena);
  }
  res->type = type;
  res->total_limit = 0;
  res->total_times = 0;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  int AvailSetFinalLimit(KHE_AVAIL_SOLVER as, AVAIL_SET avail_set)         */
/*                                                                           */
/*  Return the final limit of avail_set.                                     */
/*                                                                           */
/*  Implementation note.  For clarity we have separated the two cases, but   */
/*  they are the same, because as->ins_time_count - avail_set->total_times   */
/*  is zero in workload nodes whose final limit matters.                     */
/*                                                                           */
/*****************************************************************************/

static int AvailSetFinalLimit(KHE_AVAIL_SOLVER as, AVAIL_SET avail_set)
{
  switch( avail_set->type )
  {
    case AVAIL_SET_TIMES:

      return avail_set->total_limit +
	as->ins_time_count - avail_set->total_times;

    case AVAIL_SET_WORKLOAD:

      return avail_set->total_limit;

    default:

      HnAbort("AvailSetFinalLimit internal error");
      return 0;  /* keep compiler happy */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool AvailSetIsCandidate(KHE_AVAIL_SOLVER as, AVAIL_SET avail_set)       */
/*                                                                           */
/*  Return true if avail_set is a candidate.                                 */
/*                                                                           */
/*****************************************************************************/

static bool AvailSetIsCandidate(KHE_AVAIL_SOLVER as, AVAIL_SET avail_set)
{
  switch( avail_set->type )
  {
    case AVAIL_SET_TIMES:

      /* the set's limit must be non-trivial */
      return avail_set->total_limit < avail_set->total_times;

    case AVAIL_SET_WORKLOAD:

      /* the set must cover the whole cycle */
      return avail_set->total_times == as->ins_time_count;

    default:

      HnAbort("AvailSetIsCandidate internal error");
      return false;  /* keep compiler happy */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void AvailSolverFreeAvailSet(KHE_AVAIL_SOLVER as, AVAIL_SET avail_set,   */
/*    bool nodes_too)                                                        */
/*                                                                           */
/*  Free avail_set.  If nodes_too, also free its nodes.                      */
/*                                                                           */
/*****************************************************************************/

static void AvailSolverFreeAvailSet(KHE_AVAIL_SOLVER as, AVAIL_SET avail_set,
  bool nodes_too)
{
  AVAIL_NODE an;  int i;
  if( nodes_too )
    HaArrayForEach(avail_set->nodes, an, i)
      AvailSolverFreeAvailNode(as, an);
  HaArrayAddLast(as->free_avail_sets, avail_set);
}


/*****************************************************************************/
/*                                                                           */
/*  bool AvailSetAcceptsNode(AVAIL_SET as, AVAIL_NODE an)                    */
/*                                                                           */
/*  Return true if as accepts an (if an is independent of as's nodes).       */
/*                                                                           */
/*****************************************************************************/

static bool AvailSetAcceptsNode(AVAIL_SET as, AVAIL_NODE an)
{
  int i;  AVAIL_NODE an2;
  HaArrayForEach(as->nodes, an2, i)
    if( !KheTimeSetDisjoint(an->time_set, an2->time_set) )
      return false;
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  void AvailSetAddNode(AVAIL_SET as, AVAIL_NODE an)                        */
/*                                                                           */
/*  Add an to as.                                                            */
/*                                                                           */
/*****************************************************************************/

static void AvailSetAddNode(AVAIL_SET as, AVAIL_NODE an)
{
  /* *** no longer true
  bool is_workload_monitor;
  is_workload_monitor = (an->type == KHE_AVAIL_NODE_WORKLOAD);
  HnAssert(is_workload_monitor == (as->type == AVAIL_SET_WORKLOAD),
    "AvailSetAddNode internal error");
  *** */
  as->total_limit += an->limit;
  as->total_times += KheTimeSetTimeCount(an->time_set);
  HaArrayAddLast(as->nodes, an);
}


/*****************************************************************************/
/*                                                                           */
/*  void AvailSetDebug(AVAIL_SET as, int verbosity, int indent, FILE *fp)    */
/*                                                                           */
/*  Debug print of as onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void AvailSetDebug(AVAIL_SET as, int verbosity, int indent, FILE *fp)
{
  AVAIL_NODE an;  int i;
  fprintf(fp,"%*s[ AvailSet%s (total_limit %d, total_times %d)\n",
    indent, "", as->type == AVAIL_SET_TIMES ? "Times" : "Workload",
    as->total_limit, as->total_times);
  HaArrayForEach(as->nodes, an, i)
    AvailNodeDebug(an, verbosity, indent + 2, fp);
  fprintf(fp, "%*s]\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "avail solvers"                                                */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_AVAIL_SOLVER AvailSolverMake(KHE_SOLN soln, KHE_RESOURCE r,          */
/*    KHE_FRAME frame)                                                       */
/*                                                                           */
/*  Make an avail solver for r in soln.                                      */
/*                                                                           */
/*****************************************************************************/

KHE_AVAIL_SOLVER KheAvailSolverMake(KHE_SOLN soln, HA_ARENA a)
{
  KHE_AVAIL_SOLVER res;  int i;  KHE_RESOURCE_TYPE rt;
  HaMake(res, a);

  /* free lists - must do these first */
  HaArrayInit(res->free_time_sets, a);
  HaArrayInit(res->free_avail_nodes, a);
  HaArrayInit(res->free_avail_sets, a);
  HaArrayInit(res->free_preasst_time_infos, a);
  HaArrayInit(res->free_workload_times, a);
  HaArrayInit(res->free_workload_resources, a);
  HaArrayInit(res->free_workloads, a);

  /* general fields */
  res->arena = a;
  res->soln = soln;
  res->ins = KheSolnInstance(soln);
  res->ins_time_count = KheInstanceTimeCount(res->ins);
  HaArrayInit(res->preasst_info, a);
  if( KheInstanceAllEventsHavePreassignedTimes(res->ins) )
  {
    for( i = 0;  i < KheInstanceResourceTypeCount(res->ins);  i++ )
    {
      rt = KheInstanceResourceType(res->ins, i);
      HaArrayAddLast(res->preasst_info, KheAvailSolverMakePreasstInfo(res, rt));
    }
  }
  HaArrayInit(res->workload_array, a);
  HaArrayFill(res->workload_array, KheInstanceResourceTypeCount(res->ins),NULL);

  /* resource-specific fields */
  res->resource = NULL;
  HaArrayInit(res->limit_and_cluster_monitors, a);
  HaArrayInit(res->workload_monitors, a);
  HpTableInit(res->node_table, (HP_HASH_FN) &KheTimeSetHash,
    (HP_EQUAL_FN) &KheTimeSetEqual, NULL, a);
  HaArrayInit(res->node_array, a);
  HaArrayInit(res->zero_time_nodes, a);
  HaArrayFill(res->zero_time_nodes, res->ins_time_count, NULL);
  res->best_busy_times = NULL;
  res->best_workload = NULL;
  HaArrayInit(res->cluster_limits, a);
  HaArrayInit(res->cluster_time_sets, a);
  HaArrayInit(res->min_workloads, a);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void AvailSolverDebug(KHE_AVAIL_SOLVER as, int verbosity, int indent,    */
/*    FILE *fp)                                                              */
/*                                                                           */
/*  Debug print of avail solver as.                                          */
/*                                                                           */
/*****************************************************************************/

static void AvailSolverDebug(KHE_AVAIL_SOLVER as, int verbosity, int indent,
  FILE *fp)
{
  AVAIL_NODE an;  int i;  PREASST_INFO pi;
  fprintf(fp, "%*s[ AvailSolver(%s resource %s)\n", indent, "",
    KheInstanceId(as->ins),
    as->resource == NULL ? "-" : KheResourceId(as->resource));
  HaArrayForEach(as->preasst_info, pi, i)
    PreasstInfoDebug(pi, verbosity, indent + 2, fp);
  HaArrayForEach(as->node_array, an, i)
    AvailNodeDebug(an, verbosity, indent + 2, fp);
  HaArrayForEach(as->zero_time_nodes, an, i)
    if( an != NULL )
    {
      fprintf(fp, "%*s  zero %s node: ", indent, "",
	KheTimeId(KheInstanceTime(as->ins, i)));
      AvailNodeDebug(an, verbosity, 0, fp);
    }
  if( as->best_busy_times != NULL )
  {
    fprintf(fp, "%*s  best busy times:\n", indent, "");
    AvailSetDebug(as->best_busy_times, verbosity, indent + 2, fp);
  }
  if( as->best_workload != NULL )
  {
    fprintf(fp, "%*s  best workload:\n", indent, "");
    AvailSetDebug(as->best_workload, verbosity, indent + 2, fp);
  }
  fprintf(fp, "%*s]\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "set resource" (does all the real work)                        */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool AvailSolverTimeIsZeroTime(KHE_AVAIL_SOLVER as, KHE_TIME t)          */
/*                                                                           */
/*  Return true if t is known to be a zero time node.                        */
/*                                                                           */
/*****************************************************************************/

static bool AvailSolverTimeIsZeroTime(KHE_AVAIL_SOLVER as, KHE_TIME t)
{
  return HaArray(as->zero_time_nodes, KheTimeIndex(t)) != NULL;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheTimeSetAddTimeGroup(KHE_TIME_SET ts, KHE_TIME_GROUP tg,          */
/*    KHE_AVAIL_SOLVER as)                                                   */
/*                                                                           */
/*  Add the times of tg to ts, excluding zero times.                         */
/*                                                                           */
/*****************************************************************************/

static void AvailSolverTimeSetAddTimeGroup(KHE_AVAIL_SOLVER as,
  KHE_TIME_GROUP tg, KHE_TIME_SET ts)
{
  KHE_TIME t;  int i;
  for( i = 0;  i < KheTimeGroupTimeCount(tg);  i++ )
  {
    t = KheTimeGroupTime(tg, i);
    if( !AvailSolverTimeIsZeroTime(as, t) )
      KheTimeSetAddTime(ts, t);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_TIME_SET AvailSolverTimeSetFromTimeGroup(KHE_AVAIL_SOLVER as,        */
/*    KHE_TIME_GROUP tg)                                                     */
/*                                                                           */
/*  Return a new time set containing the times of tg, excluding zero times.  */
/*                                                                           */
/*****************************************************************************/

static KHE_TIME_SET AvailSolverTimeSetFromTimeGroup(KHE_AVAIL_SOLVER as,
  KHE_TIME_GROUP tg)
{
  KHE_TIME_SET res;
  res = KheAvailSolverMakeTimeSet(as);
  AvailSolverTimeSetAddTimeGroup(as, tg, res);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_TIME_SET KheClusterMonitorTimeSet(KHE_CLUSTER_BUSY_TIMES_MONITOR m,  */
/*    HA_ARENA a)                                                            */
/*                                                                           */
/*  Return a time set containing m's times.                                  */
/*                                                                           */
/*****************************************************************************/

/* *** no longer used
static KHE_TIME_SET KheClusterMonitorTimeSet(KHE_CLUSTER_BUSY_TIMES_MONITOR m,
  KHE_AVAIL_SOLVER as)
{
  int i;  KHE_TIME_GROUP tg;  KHE_POLARITY po;  KHE_TIME_SET res;
  res = KheAvailSolverMakeTimeSet(as);
  for( i = 0;  i < KheClusterBusyTimesMonitorTimeGroupCount(m);  i++ )
  {
    tg = KheClusterBusyTimesMonitorTimeGroup(m, i, &po);
    AvailSolverTimeSetAddTimeGroup(as, tg, res);
  }
  return res;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  AVAIL_SET AvailSolverFindBestAvailSet(KHE_AVAIL_SOLVER as,               */
/*    AVAIL_SET_TYPE type)                                                   */
/*                                                                           */
/*  Find the best avail set we can from the nodes in as->node_array.         */
/*                                                                           */
/*****************************************************************************/

static AVAIL_SET AvailSolverFindBestAvailSet(KHE_AVAIL_SOLVER as,
  AVAIL_SET_TYPE type)
{
  AVAIL_SET avail_set, res;  int i, j, since_new_best;  AVAIL_NODE init_an, an;
  if( DEBUG2 )
    fprintf(stderr, "[ AvailSolverFindBestAvailSet(%s, -)\n",
      KheResourceId(as->resource));

  /* sort the nodes */
  HaArraySort(as->node_array, &AvailNodeCmp);

  /* find one independent set starting at each position and keep the best */
  res = NULL;
  since_new_best = 0;
  HaArrayForEach(as->node_array, init_an, i)
  {
    /* give up if too long since a new best appeared */
    if( since_new_best >= 20 )
      break;

    /* build an avail set starting from init_an */
    avail_set = AvailSolverMakeAvailSet(as, type);
    AvailSetAddNode(avail_set, init_an);
    for( j = i + 1;  j < HaArrayCount(as->node_array);  j++ )
    {
      an = HaArray(as->node_array, j);
      if( AvailSetAcceptsNode(avail_set, an) )
	AvailSetAddNode(avail_set, an);
    }

    /* compare avail_set (if it is a candidate) with res and keep the best */
    if( !AvailSetIsCandidate(as, avail_set) )
    {
      /* avail_set is not a candidate; delete it */
      if( DEBUG2 )
      {
	fprintf(stderr, "  avail set %d (not candidate):\n", i);
	if( DEBUG6 )
	  AvailSetDebug(avail_set, 2, 2, stderr);
      }
      AvailSolverFreeAvailSet(as, avail_set, false);
      since_new_best++;
    }
    else if( res == NULL )
    {
      /* avail_set is first best */
      if( DEBUG2 )
      {
	fprintf(stderr, "  avail set %d (first best):\n", i);
	if( DEBUG6 )
	  AvailSetDebug(avail_set, 2, 2, stderr);
      }
      res = avail_set;
      since_new_best = 0;
    }
    else if( AvailSetFinalLimit(as, avail_set) < AvailSetFinalLimit(as, res) )
    {
      /* avail_set is new best; delete old best */
      if( DEBUG2 )
      {
	fprintf(stderr, "  avail set %d (new best):\n", i);
	if( DEBUG6 )
	  AvailSetDebug(avail_set, 2, 2, stderr);
      }
      AvailSolverFreeAvailSet(as, res, false);
      res = avail_set;
      since_new_best = 0;
    }
    else
    {
      /* avail_set is not new best; delete it */
      if( DEBUG2 )
      {
	fprintf(stderr, "  avail set %d (not new best):\n", i);
	if( DEBUG6 )
	  AvailSetDebug(avail_set, 2, 2, stderr);
      }
      AvailSolverFreeAvailSet(as, avail_set, false);
      since_new_best++;
    }
  }
  if( DEBUG2 )
    fprintf(stderr, "] AvailSolverFindBestAvailSet returning\n");
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void AvailSolverAddTimeNodes(KHE_AVAIL_SOLVER as,                        */
/*    KHE_AVAIL_NODE_TYPE type, int limit, KHE_TIME_GROUP tg, KHE_MONITOR m) */
/*                                                                           */
/*  For each time t of tg not known to be a zero time, add a node with       */
/*  attributes type, limit, t, and m.  (This could make t a zero time.)      */
/*                                                                           */
/*****************************************************************************/

static void AvailSolverAddTimeNodes(KHE_AVAIL_SOLVER as,
  KHE_AVAIL_NODE_TYPE type, int limit, KHE_TIME_GROUP tg, KHE_MONITOR m)
{
  int i;  KHE_TIME t;  KHE_TIME_SET ts;
  for( i = 0;  i < KheTimeGroupTimeCount(tg);  i++ )
  {
    t = KheTimeGroupTime(tg, i);
    if( !AvailSolverTimeIsZeroTime(as, t) )
    {
      ts = KheAvailSolverMakeTimeSet(as);
      KheTimeSetAddTime(ts, t);
      AvailSolverMakeAndAddAvailNode(as, type, limit, ts, m);
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void AvailSolverAddSingleTimeNodes(KHE_AVAIL_SOLVER as)                  */
/*                                                                           */
/*  Add all nodes of type KHE_AVAIL_NODE_SINGLE_TIME to as.                  */
/*                                                                           */
/*****************************************************************************/

/* *** no longer doing this
static void AvailSolverAddSingleTimeNodes(KHE_AVAIL_SOLVER as)
{
  AvailSolverAddTimeNodes(as, KHE_AVAIL_NODE_SINGLE_TIME, 1,
    KheInstanceFullTimeGroup(as->ins), NULL);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void AvailSolverAddUnassignableTimeNodes(KHE_AVAIL_SOLVER as)            */
/*                                                                           */
/*  Add all nodes of type KHE_AVAIL_NODE_UNASSIGNABLE_TIME to as.            */
/*                                                                           */
/*****************************************************************************/

static void AvailSolverAddUnassignableTimeNodes(KHE_AVAIL_SOLVER as)
{
  KHE_RESOURCE_TYPE rt;  int i, pos;  PREASST_INFO pi;  PREASST_TIME_INFO ti;
  if( HaArrayCount(as->preasst_info) > 0 )
  {
    rt = KheResourceResourceType(as->resource);
    pi = HaArray(as->preasst_info, KheResourceTypeIndex(rt));
    HaArrayForEach(pi->time_info, ti, i)
    {
      HnAssert(ti->all_preassigned,
	"AvailSolverAddUnassignableTimeNodes internal error");
      if( !HaArrayContains(ti->resources, as->resource, &pos) )
      {
	/* all the slots at this time are preassigned, and as->resource is */
	/* not assigned any of them, so this is an unavailable time for it */
	AvailSolverAddTimeNodes(as, KHE_AVAIL_NODE_UNASSIGNABLE_TIME, 0,
	  KheTimeSingletonTimeGroup(ti->time), NULL);
      }
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void AvailSolverAddUnavailableTimeNodes(KHE_AVAIL_SOLVER as,             */
/*    KHE_AVOID_UNAVAILABLE_TIMES_MONITOR m)                                 */
/*                                                                           */
/*  Add nodes derived from m of type KHE_AVAIL_NODE_UNAVAILABLE_TIME to as.  */
/*                                                                           */
/*****************************************************************************/

static void AvailSolverAddUnavailableTimeNodes(KHE_AVAIL_SOLVER as,
  KHE_AVOID_UNAVAILABLE_TIMES_MONITOR m)
{
  KHE_AVOID_UNAVAILABLE_TIMES_CONSTRAINT c;  KHE_TIME_GROUP tg;
  c = KheAvoidUnavailableTimesMonitorConstraint(m);
  tg = KheAvoidUnavailableTimesConstraintUnavailableTimes(c);
  AvailSolverAddTimeNodes(as, KHE_AVAIL_NODE_UNAVAILABLE_TIME, 0,
    tg, (KHE_MONITOR) m);
}


/*****************************************************************************/
/*                                                                           */
/*  void AvailSolverAddLimitBusyZeroNodes(KHE_AVAIL_SOLVER as,               */
/*    KHE_LIMIT_BUSY_TIMES_MONITOR m)                                        */
/*                                                                           */
/*  Add nodes derived from m of type KHE_AVAIL_NODE_LIMIT_BUSY_ZERO to as.   */
/*                                                                           */
/*****************************************************************************/

static void AvailSolverAddLimitBusyZeroNodes(KHE_AVAIL_SOLVER as,
  KHE_LIMIT_BUSY_TIMES_MONITOR m)
{
  KHE_LIMIT_BUSY_TIMES_CONSTRAINT c;  int i, offset;  KHE_TIME_GROUP tg;
  c = KheLimitBusyTimesMonitorConstraint(m);
  offset = KheLimitBusyTimesMonitorOffset(m);
  for( i = 0;  i < KheLimitBusyTimesConstraintTimeGroupCount(c);  i++ )
  {
    /* max limit is 0, so all times in all time groups are unavailable */
    tg = KheLimitBusyTimesConstraintTimeGroup(c, i, offset);
    AvailSolverAddTimeNodes(as, KHE_AVAIL_NODE_LIMIT_BUSY_ZERO, 0,
      tg, (KHE_MONITOR) m);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void AvailSolverAddClusterBusyZeroNodes(KHE_AVAIL_SOLVER as,             */
/*    KHE_CLUSTER_BUSY_TIMES_MONITOR m)                                      */
/*                                                                           */
/*  Add nodes derived from m of type KHE_AVAIL_NODE_CLUSTER_BUSY_ZERO to as. */
/*                                                                           */
/*****************************************************************************/

static void AvailSolverAddClusterBusyZeroNodes(KHE_AVAIL_SOLVER as,
  KHE_CLUSTER_BUSY_TIMES_MONITOR m)
{
  int i;  KHE_TIME_GROUP tg;  KHE_POLARITY po;
  for( i = 0;  i < KheClusterBusyTimesMonitorTimeGroupCount(m);  i++ )
  {
    tg = KheClusterBusyTimesMonitorTimeGroup(m, i, &po);
    if( po == KHE_POSITIVE )
    {
      /* max limit is 0, so all times in positive time groups are unavailable */
      AvailSolverAddTimeNodes(as, KHE_AVAIL_NODE_CLUSTER_BUSY_ZERO, 0,
	tg, (KHE_MONITOR) m);
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void AvailSolverAddLimitBusyNodes(KHE_AVAIL_SOLVER as,                   */
/*    KHE_LIMIT_BUSY_TIMES_MONITOR m)                                        */
/*                                                                           */
/*  Add nodes derived from m of type KHE_AVAIL_NODE_LIMIT_BUSY to as.        */
/*                                                                           */
/*****************************************************************************/

static void AvailSolverAddLimitBusyNodes(KHE_AVAIL_SOLVER as,
  KHE_LIMIT_BUSY_TIMES_MONITOR m)
{
  KHE_LIMIT_BUSY_TIMES_CONSTRAINT c;  int i, offset, limit;  KHE_TIME_GROUP tg;
  c = KheLimitBusyTimesMonitorConstraint(m);
  limit = KheLimitBusyTimesConstraintMaximum(c);
  offset = KheLimitBusyTimesMonitorOffset(m);
  for( i = 0;  i < KheLimitBusyTimesConstraintTimeGroupCount(c);  i++ )
  {
    /* limit is non-zero, so the whole time group is limited */
    tg = KheLimitBusyTimesConstraintTimeGroup(c, i, offset);
    AvailSolverMakeAndAddAvailNode(as, KHE_AVAIL_NODE_LIMIT_BUSY,
      limit, AvailSolverTimeSetFromTimeGroup(as, tg), (KHE_MONITOR) m);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  int IntDecreasingCmp(const void *t1, const void *t2)                     */
/*                                                                           */
/*  Comparison function for sorting an array of integers by decreasing order.*/
/*                                                                           */
/*****************************************************************************/

static int IntDecreasingCmp(const void *t1, const void *t2)
{
  int i1 = * (int *) t1;
  int i2 = * (int *) t2;
  return i2 - i1;
}


/*****************************************************************************/
/*                                                                           */
/*  void AvailSolverAddClusterBusyNodes(KHE_AVAIL_SOLVER as,                 */
/*    KHE_CLUSTER_BUSY_TIMES_MONITOR m, KHE_POLARITY po)                     */
/*                                                                           */
/*  If po is KHE_POSITIVE, try to add a KHE_AVAIL_NODE_CLUSTER_BUSY avail    */
/*  node based on the positive time groups and maximum limit of m.           */
/*                                                                           */
/*  If po is KHE_NEGATIVE, try to add a KHE_AVAIL_NODE_CLUSTER_BUSY_MIN      */
/*  avail node based on the negative time groups and minimum limit of m.     */
/*                                                                           */
/*****************************************************************************/

static void AvailSolverAddClusterBusyNodes(KHE_AVAIL_SOLVER as,
  KHE_CLUSTER_BUSY_TIMES_MONITOR m, KHE_POLARITY po)
{
  int i, pos, limit, max_lim, total_limit, total_times, history, history_before;
  KHE_TIME_GROUP tg;  KHE_POLARITY po2; KHE_TIME_SET ts, total_ts;
  AVAIL_NODE an;  KHE_CLUSTER_BUSY_TIMES_CONSTRAINT c;

  /* find a limit and time set for each positive time group */
  HaArrayClear(as->cluster_limits);
  HaArrayClear(as->cluster_time_sets);
  total_times = 0;
  for( i = 0;  i < KheClusterBusyTimesMonitorTimeGroupCount(m);  i++ )
  {
    tg = KheClusterBusyTimesMonitorTimeGroup(m, i, &po2);
    if( po2 == po )
    {
      ts = AvailSolverTimeSetFromTimeGroup(as, tg);
      if( HpTableRetrieve(as->node_table, (void *) ts, an, pos) )
      {
	/* use the limit from the table */
	limit = an->limit;
      }
      else
      {
	/* the limit is just the time set size */
	limit = KheTimeSetTimeCount(ts);
      }
      HaArrayAddLast(as->cluster_limits, limit);
      HaArrayAddLast(as->cluster_time_sets, ts);
      total_times += KheTimeSetTimeCount(ts);
    }
  }

  /* sort the limits and sum the largest max_lim */
  c = KheClusterBusyTimesMonitorConstraint(m);
  history = KheClusterBusyTimesConstraintHistory(c, as->resource);
  history_before = KheClusterBusyTimesConstraintHistoryBefore(c);
  if( po == KHE_POSITIVE )
  {
    max_lim = KheClusterBusyTimesConstraintMaximum(c) - history;
  }
  else
    max_lim = KheClusterBusyTimesConstraintTimeGroupCount(c) -
      KheClusterBusyTimesConstraintMinimum(c) - (history_before - history);
  if( max_lim < 0 )
    max_lim = 0;  /* done to agree with doc but makes no difference in fact */
  HaArraySort(as->cluster_limits, &IntDecreasingCmp);
  total_limit = 0;
  for( i = 0;  i < max_lim && i < HaArrayCount(as->cluster_limits);  i++ )
    total_limit += HaArray(as->cluster_limits, i);
  if( total_limit < total_times )
  {
    /* make a new node whose times are the stored ones */
    total_ts = KheAvailSolverMakeTimeSet(as);
    HaArrayForEach(as->cluster_time_sets, ts, i)
      KheTimeSetUnion(total_ts, ts);
    AvailSolverMakeAndAddAvailNode(as, KHE_AVAIL_NODE_CLUSTER_BUSY,
      total_limit, total_ts, (KHE_MONITOR) m);
  }

  /* free the time sets */
  HaArrayForEach(as->cluster_time_sets, ts, i)
    KheAvailSolverFreeTimeSet(as, ts);
}


/*****************************************************************************/
/*                                                                           */
/*  int FloatCmp(const void *t1, const void *t2)                             */
/*                                                                           */
/*  Comparison function for sorting an array of floats into increasing order.*/
/*                                                                           */
/*****************************************************************************/

static int FloatCmp(const void *t1, const void *t2)
{
  float f1 = * (float *) t1;
  float f2 = * (float *) t2;
  float diff = f1 - f2;
  return diff < 0 ? -1 : diff > 0 ? 1 : 0;
}


/*****************************************************************************/
/*                                                                           */
/*  void AvailSolverAddLimitWorkloadNodeForTimeGroup(KHE_AVAIL_SOLVER as,    */
/*    KHE_TIME_GROUP tg, float max_lim, KHE_LIMIT_WORKLOAD_MONITOR m)        */
/*                                                                           */
/*  Add an avail node expressing (in the form of a limit on the number of    */
/*  busy times) the workload limit of at most max_lim on tg.                 */
/*                                                                           */
/*****************************************************************************/

static void AvailSolverAddLimitWorkloadNodeForTimeGroup(KHE_AVAIL_SOLVER as,
  KHE_TIME_GROUP tg, float max_lim, KHE_LIMIT_WORKLOAD_MONITOR m)
{
  KHE_TIME t;  int i;  float wk, total;  KHE_TIME_SET ts;  char buff[50];

  /* set ts to the relevant times, and min_workloads to their min workloads */
  if( DEBUG4 )
    fprintf(stderr,
      "[ AvailSolverAddLimitWorkloadNodeForTimeGroup(as, %s, %s)\n",
      KheTimeGroupId(tg), WorkloadShow(max_lim, buff));
  ts = KheAvailSolverMakeTimeSet(as);
  HaArrayClear(as->min_workloads);
  for( i = 0;  i < KheTimeGroupTimeCount(tg);  i++ )
  {
    t = KheTimeGroupTime(tg, i);
    if( AvailSolverTimeIsZeroTime(as, t) )
    {
      if( DEBUG4 )
	fprintf(stderr, "  %s: zero time\n", KheTimeId(t));
    }
    else
    {
      switch( KheAvailSolverWorkloadPerTime(as, t, as->resource, &wk) )
      {
	case WORKLOAD_PREASST:

	  /* r is preassigned at t; subtract wk from max_lim and ignore t */
	  max_lim -= wk;
	  if( DEBUG4 )
	    fprintf(stderr, "  %s: preassigned time, max_lim -= %s\n",
	      KheTimeId(t), WorkloadShow(wk, buff));
	  break;

	case WORKLOAD_NOT_BUSY:

	  /* r cannot be busy at t; ignore t */
	  if( DEBUG4 )
	    fprintf(stderr, "  %s: not busy time\n", KheTimeId(t));
	  break;

	case WORKLOAD_DFT:

	  /* include t and w(t, r), NB w(t, r) < infinity */
	  KheTimeSetAddTime(ts, t);
	  HaArrayAddLast(as->min_workloads, wk);
	  if( DEBUG4 )
	    fprintf(stderr, "  %s: adding time and %s\n",
	      KheTimeId(t), WorkloadShow(wk, buff));
	  break;

	  break;

	default:

	  HnAbort("AvailSolverAddLimitWorkloadNodeForTimeGroup internal error");
	  break;
      }
    }
  }

  /* sort min_workloads and find the largest k st sum(0 .. k) <= max_lim */
  HaArraySort(as->min_workloads, &FloatCmp);
  if( DEBUG4 )
  {
    fprintf(stderr, "  sorted workloads: ");
    HaArrayForEach(as->min_workloads, wk, i)
      fprintf(stderr, "%s%s", i == 0 ? "" : i % 8 == 0 ? ",\n  " : ", ",
	WorkloadShow(wk, buff));
    fprintf(stderr, "\n");
  }
  total = 0.0;
  HaArrayForEach(as->min_workloads, wk, i)
  {
    /* total is the sum of workloads 0 .. i - 1 */
    if( total + wk > max_lim )
      break;
    total += wk;
  }
  if( DEBUG4 )
    fprintf(stderr, "  stopping at %d (total workload %s)\n", i,
      WorkloadShow(total, buff));
  AvailSolverMakeAndAddAvailNode(as, KHE_AVAIL_NODE_WORKLOAD, i, ts,
    (KHE_MONITOR) m);
  if( DEBUG4 )
    fprintf(stderr,"] AvailSolverAddLimitWorkloadNodeForTimeGroup returning\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void AvailSolverAddLimitWorkloadNodes(KHE_AVAIL_SOLVER as,               */
/*    KHE_LIMIT_WORKLOAD_MONITOR lwm)                                        */
/*                                                                           */
/*  Add nodes based on m of type KHE_AVAIL_NODE_WORKLOAD to as.              */
/*  This is for the calculation of limits on busy times.                     */
/*                                                                           */
/*****************************************************************************/

static void AvailSolverAddLimitWorkloadNodes(KHE_AVAIL_SOLVER as,
  KHE_LIMIT_WORKLOAD_MONITOR lwm)
{
  KHE_LIMIT_WORKLOAD_CONSTRAINT lwc;  int i, offset, max_lim; KHE_TIME_GROUP tg;
  if( DEBUG4 )
    fprintf(stderr, "[ AvailSolverAddLimitWorkloadNodes(as, %s)\n",
      KheMonitorId((KHE_MONITOR) lwm));
  lwc = KheLimitWorkloadMonitorConstraint(lwm);
  offset = KheLimitWorkloadMonitorOffset(lwm);
  max_lim = KheLimitWorkloadConstraintMaximum(lwc);
  for( i = 0;  i < KheLimitWorkloadConstraintTimeGroupCount(lwc);  i++ )
  {
    tg = KheLimitWorkloadConstraintTimeGroup(lwc, i, offset);
    AvailSolverAddLimitWorkloadNodeForTimeGroup(as, tg, (float) max_lim, lwm);
  }
  if( DEBUG4 )
    fprintf(stderr, "] AvailSolverAddLimitWorkloadNodes returning\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void AvailSolverAddWorkloadNodes(KHE_AVAIL_SOLVER as,                    */
/*    KHE_LIMIT_WORKLOAD_MONITOR m)                                          */
/*                                                                           */
/*  Add nodes based on m of type KHE_AVAIL_NODE_WORKLOAD to as.              */
/*  This is for the calculation of limits on workload.                       */
/*                                                                           */
/*****************************************************************************/

static void AvailSolverAddWorkloadNodes(KHE_AVAIL_SOLVER as,
  KHE_LIMIT_WORKLOAD_MONITOR m)
{
  int i, max_lim, offset;  KHE_TIME_GROUP tg;
  KHE_LIMIT_WORKLOAD_CONSTRAINT c;
  c = KheLimitWorkloadMonitorConstraint(m);
  max_lim = KheLimitWorkloadConstraintMaximum(c);
  offset = KheLimitWorkloadMonitorOffset(m);
  for( i = 0;  i < KheLimitWorkloadConstraintTimeGroupCount(c);  i++ )
  {
    /* make and add in an avail node for tg */
    tg = KheLimitWorkloadConstraintTimeGroup(c, i, offset);
    AvailSolverMakeAndAddAvailNode(as, KHE_AVAIL_NODE_WORKLOAD,
      max_lim, AvailSolverTimeSetFromTimeGroup(as, tg), (KHE_MONITOR) m);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  AVAIL_SET AvailSolverFindBestBusyTimes(KHE_AVAIL_SOLVER as)              */
/*                                                                           */
/*  Return an avail set for the maximum number of busy times.  Also set up   */
/*  as->workload_monitors, ready for AvailSolverFindBestWorkload to use.     */
/*                                                                           */
/*****************************************************************************/

static AVAIL_SET AvailSolverFindBestBusyTimes(KHE_AVAIL_SOLVER as)
{
  KHE_CONSTRAINT c;  KHE_MONITOR m;  int i, attempts;  AVAIL_SET res;
  KHE_CLUSTER_BUSY_TIMES_CONSTRAINT cbtc;  KHE_CLUSTER_BUSY_TIMES_MONITOR cbtm;
  KHE_LIMIT_BUSY_TIMES_CONSTRAINT lbtc;  KHE_LIMIT_BUSY_TIMES_MONITOR lbtm;
  KHE_LIMIT_WORKLOAD_MONITOR lwm;

  /* add nodes for single times and preassigned times */
  if( DEBUG1 )
  {
    fprintf(stderr, "[ AvailSolverFindBestBusyTimes(as)\n");
    AvailSolverDebug(as, 1, 2, stderr);
  }
  /* AvailSolverAddSingleTimeNodes(as); */
  if( DEBUG1 )
    fprintf(stderr, "  avail calling AvailSolverAddUnassignableTimeNodes:\n");
  AvailSolverAddUnassignableTimeNodes(as);

  /* for each monitor, either add its nodes immediately or save it for later */
  if( DEBUG1 )
    fprintf(stderr, "  avail adding zero nodes:\n");
  HaArrayClear(as->limit_and_cluster_monitors);
  HaArrayClear(as->workload_monitors);
  for( i = 0;  i < KheSolnResourceMonitorCount(as->soln, as->resource);  i++ )
  {
    m = KheSolnResourceMonitor(as->soln, as->resource, i);
    c = KheMonitorConstraint(m);
    if( c != NULL && KheConstraintWeight(c) > 0 )
      switch( KheMonitorTag(m) )
      {
	case KHE_AVOID_UNAVAILABLE_TIMES_MONITOR_TAG:

	  /* handle immediately */
	  AvailSolverAddUnavailableTimeNodes(as,
	    (KHE_AVOID_UNAVAILABLE_TIMES_MONITOR) m);
	  break;

	case KHE_CLUSTER_BUSY_TIMES_MONITOR_TAG:

	  /* handle immediately if limit is 0, save for later otherwise */
	  cbtc = (KHE_CLUSTER_BUSY_TIMES_CONSTRAINT) c;
	  cbtm = (KHE_CLUSTER_BUSY_TIMES_MONITOR) m;
	  if( KheClusterBusyTimesConstraintMaximum(cbtc) == 0 )
	    AvailSolverAddClusterBusyZeroNodes(as, cbtm);
	  else if( KheClusterBusyTimesConstraintTimeGroupsDisjoint(cbtc) )
	    HaArrayAddLast(as->limit_and_cluster_monitors, m);
	  break;

	case KHE_LIMIT_BUSY_TIMES_MONITOR_TAG:

	  /* handle immediately if limit is 0, save for later otherwise */
	  lbtc = (KHE_LIMIT_BUSY_TIMES_CONSTRAINT) c;
	  lbtm = (KHE_LIMIT_BUSY_TIMES_MONITOR) m;
	  if( KheLimitBusyTimesConstraintMaximum(lbtc) == 0 )
	    AvailSolverAddLimitBusyZeroNodes(as, lbtm);
	  else
	    HaArrayAddLast(as->limit_and_cluster_monitors, m);
	    /* AvailSolverAddLimitBusyNodes(as, lbtm); */
	  break;

	case KHE_LIMIT_WORKLOAD_MONITOR_TAG:

	  /* save for later */
	  lwm = (KHE_LIMIT_WORKLOAD_MONITOR) m;
	  HaArrayAddLast(as->workload_monitors, lwm);
	  break;

	default:

	  /* ignore all other monitors */
	  break;
      }
  }

  /* try each monitor saved above twice:  one can open the way to another */
  if( DEBUG1 )
    fprintf(stderr, "  avail adding non-zero nodes:\n");
  for( attempts = 1;  attempts <= 2;  attempts++ )
  {
    /* limit and cluster busy times monitors */
    HaArrayForEach(as->limit_and_cluster_monitors, m, i)
      if( KheMonitorTag(m) == KHE_LIMIT_BUSY_TIMES_MONITOR_TAG )
      {
	lbtm = (KHE_LIMIT_BUSY_TIMES_MONITOR) m;
	AvailSolverAddLimitBusyNodes(as, lbtm);
      }
      else
      {
	cbtm = (KHE_CLUSTER_BUSY_TIMES_MONITOR) m;
        AvailSolverAddClusterBusyNodes(as, cbtm, KHE_POSITIVE);
        AvailSolverAddClusterBusyNodes(as, cbtm, KHE_NEGATIVE);
      }

    /* limit workload monitors */
    HaArrayForEach(as->workload_monitors, lwm, i)
      AvailSolverAddLimitWorkloadNodes(as, lwm);
  }


  /* find the best avail set and return */
  if( DEBUG1 )
    fprintf(stderr, "  avail finding best avail set:\n");
  res = AvailSolverFindBestAvailSet(as, AVAIL_SET_TIMES);
  if( DEBUG1 )
  {
    if( DEBUG5 )
    {
      fprintf(stderr, "AvailSolverFindBestBusyTimes returning:\n");
      if( res != NULL )
	AvailSetDebug(res, 1, 2, stderr);
      AvailSolverDebug(as, 1, 2, stderr);
    }
    fprintf(stderr, "] AvailSolverFindBestBusyTimes\n");
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  AVAIL_SET AvailSolverFindBestBusyTimes(KHE_AVAIL_SOLVER as)              */
/*                                                                           */
/*  Return an avail set for the maximum number of busy times.                */
/*                                                                           */
/*****************************************************************************/

/* *** old version
static AVAIL_SET AvailSolverFindBestBusyTimes(KHE_AVAIL_SOLVER as)
{
  int i, j, max_lim, offset;
  KHE_TIME_SET ts;  KHE_TIME_GROUP tg;  KHE_MONITOR m;
  KHE_LIMIT_BUSY_TIMES_CONSTRAINT lbtc;  KHE_LIMIT_BUSY_TIMES_MONITOR lbtm;
  KHE_CLUSTER_BUSY_TIMES_CONSTRAINT cbtc;  KHE_CLUSTER_BUSY_TIMES_MONITOR cbtm;

  ** handle all time groups from limit busy times monitors **
  for( i = 0;  i < KheSolnResourceMonitorCount(as->soln, as->resource);  i++ )
  {
    m = KheSolnResourceMonitor(as->soln, as->resource, i);
    if( KheMonitorTag(m) == KHE_LIMIT_BUSY_TIMES_MONITOR_TAG )
    {
      lbtm = (KHE_LIMIT_BUSY_TIMES_MONITOR) m;
      lbtc = KheLimitBusyTimesMonitorConstraint(lbtm);
      if( KheConstraintWeight((KHE_CONSTRAINT) lbtc) > 0 )
      {
	max_lim = KheLimitBusyTimesConstraintMaximum(lbtc);
	offset = KheLimitBusyTimesMonitorOffset(lbtm);
	for( j = 0;  j < KheLimitBusyTimesConstraintTimeGroupCount(lbtc);  j++ )
	{
	  ** make an avail node for tg and add it to the list **
	  tg = KheLimitBusyTimesConstraintTimeGroup(lbtc, j, offset);
	  ts = KheTimeSetFromTimeGroup(tg, as);
	  AvailSolverMakeAndAddAvailNodeAndSubsets(as,
	    KHE_AVAIL_NODE_LIMIT_BUSY, max_lim, ts, m);
	}
      }
    }
  }

  ** sort and uniqueify the limit 1 nodes we have so far **
  ** AvailSolverSortMaxOneTimeSets(as); **

  ** handle all cluster busy times monitors **
  for( i = 0;  i < KheSolnResourceMonitorCount(as->soln, as->resource);  i++ )
  {
    m = KheSolnResourceMonitor(as->soln, as->resource, i);
    if( KheMonitorTag(m) == KHE_CLUSTER_BUSY_TIMES_MONITOR_TAG )
    {
      cbtm = (KHE_CLUSTER_BUSY_TIMES_MONITOR) m;
      cbtc = KheClusterBusyTimesMonitorConstraint(cbtm);
      if( KheConstraintWeight((KHE_CONSTRAINT) cbtc) > 0 &&
          KheClusterBusyTimesConstraintTimeGroupsDisjoint(cbtc) &&
	  (KheClusterBusyTimesConstraintAllPositive(cbtc) ||
	   KheClusterBusyTimesConstraintAllNegative(cbtc)) &&
	  KheClusterMonitorHasMaxOneTimeGroups(as, cbtm) )
      {
	if( KheClusterBusyTimesConstraintAllPositive(cbtc) )
	{
	  ** all positive, limit is just the max limit **
	  max_lim = KheClusterBusyTimesConstraintMaximum(cbtc);
	}
	else
	{
	  ** all negative, limit is time groups minus the min limit **
	  max_lim = KheClusterBusyTimesMonitorTimeGroupCount(cbtm) -
            KheClusterBusyTimesConstraintMinimum(cbtc);
	}

	** make an avail node for cbtm and add it to the list **
	ts = KheClusterMonitorTimeSet(cbtm, as);
	AvailSolverMakeAndAddAvailNode(as, KHE_AVAIL_NODE_CLUSTER_BUSY,
          max_lim, ts, m);
      }
    }
  }

  ** 1 ** AvailSolverAddSingleTimeNodes(as);
  ** 2 ** AvailSolverAddUnassignableTimeNodes(as);
  AvailSolverAddNodesForMonitorsSimple(as);
  AvailSolverAddNodesForMonitorsComplex(as);
  ** 7 ** AvailSolverAddClusterBusyNodes(as);
  ** 8 ** AvailSolverAddClusterBusyMinNodes(as);
  return AvailSolverFindBestAvailSet(as, AVAIL_SET_TIMES);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  AVAIL_SET AvailSolverFindBestWorkload(KHE_AVAIL_SOLVER as)               */
/*                                                                           */
/*  Return an avail set for the maximum workload as found by as.             */
/*                                                                           */
/*  Implementation note.  At the time this function is called, all           */
/*  relevant limit workload monitors will be in as->workload_monitors.       */
/*                                                                           */
/*****************************************************************************/

static AVAIL_SET AvailSolverFindBestWorkload(KHE_AVAIL_SOLVER as)
{
  int i;  KHE_LIMIT_WORKLOAD_MONITOR lwm;  AVAIL_SET res; 
  if( DEBUG1 )
    fprintf(stderr, "[ AvailSolverFindBestWorkload(as)\n");
  HaArrayForEach(as->workload_monitors, lwm, i)
    AvailSolverAddWorkloadNodes(as, lwm);
  res = AvailSolverFindBestAvailSet(as, AVAIL_SET_WORKLOAD);
  if( DEBUG1 )
  {
    if( DEBUG5 )
    {
      fprintf(stderr, "AvailSolverFindBestWorkload returning:\n");
      if( res != NULL )
	AvailSetDebug(res, 1, 2, stderr);
      AvailSolverDebug(as, 1, 2, stderr);
    }
    fprintf(stderr, "] AvailSolverFindBestWorkload\n");
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheAvailSolverFreeNodes(KHE_AVAIL_SOLVER as, AVAIL_SET keep_set)    */
/*                                                                           */
/*  Free the nodes of as, except keep any in keep_set.                       */
/*                                                                           */
/*****************************************************************************/

static void KheAvailSolverFreeNodes(KHE_AVAIL_SOLVER as, AVAIL_SET keep_set)
{
  AVAIL_NODE an;  int i, pos;
  HpTableClear(as->node_table);
  HaArrayForEach(as->node_array, an, i)
    if( keep_set == NULL || !HaArrayContains(keep_set->nodes, an, &pos) )
      AvailSolverFreeAvailNode(as, an);
  HaArrayClear(as->node_array);
  HaArrayClear(as->zero_time_nodes);
  HaArrayFill(as->zero_time_nodes, as->ins_time_count, NULL);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheAvailSolverSetResource(KHE_AVAIL_SOLVER as, KHE_RESOURCE r)      */
/*                                                                           */
/*  Set as to work on resource r.                                            */
/*                                                                           */
/*****************************************************************************/

void KheAvailSolverSetResource(KHE_AVAIL_SOLVER as, KHE_RESOURCE r)
{
  if( DEBUG1 )
    fprintf(stderr, "[ KheAvailSolverSetResource(as, %s%s)\n",
      r == NULL ? "-" : KheResourceId(r),
      r == as->resource ? " unchanged" : "");
  HnAssert(r != NULL, "KheAvailSolverSetResource:  r is NULL");
  HnAssert(KheResourceInstance(r) == as->ins,
    "KheAvailSolverSetResource:  r is from the wrong instance");
  if( r != as->resource )
  {
    /* clear out retained information about any previous resource */
    if( as->best_busy_times != NULL )
    {
      AvailSolverFreeAvailSet(as, as->best_busy_times, true);
      as->best_busy_times = NULL;
    }
    if( as->best_workload != NULL )
    {
      AvailSolverFreeAvailSet(as, as->best_workload, true);
      as->best_workload = NULL;
    }

    /* build up the wanted information about r */
    as->resource = r;
    KheAvailSolverInitialiseWorkloadInfo(as, KheResourceResourceType(r));
    as->best_busy_times = AvailSolverFindBestBusyTimes(as);
    KheAvailSolverFreeNodes(as, as->best_busy_times);
    as->best_workload = AvailSolverFindBestWorkload(as);
    KheAvailSolverFreeNodes(as, as->best_workload);
  }
  if( DEBUG1 )
    fprintf(stderr, "] KheAvailSolverSetResource returning\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "public queries"                                               */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool KheAvailSolverMaxBusyTimes(KHE_AVAIL_SOLVER as, int *res)           */
/*                                                                           */
/*  If the set resource has a max busy times limit, return true and set      */
/*  *res to that limit.  Otherwise return false with *res set to INT_MAX.    */
/*                                                                           */
/*****************************************************************************/

bool KheAvailSolverMaxBusyTimes(KHE_AVAIL_SOLVER as, int *res)
{
  HnAssert(as->resource != NULL,
    "KheAvailSolverMaxBusyTimes: resource not set");
  if( as->best_busy_times != NULL )
    return *res = AvailSetFinalLimit(as, as->best_busy_times), true;
  else
    return *res = INT_MAX, false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheAvailSolverMaxWorkload(KHE_AVAIL_SOLVER as, float *res)          */
/*                                                                           */
/*  If the set resource has a max workload limit, return true and set        */
/*  *res to that limit.  Otherwise return false with *res set to FLT_MAX.    */
/*                                                                           */
/*****************************************************************************/

bool KheAvailSolverMaxWorkload(KHE_AVAIL_SOLVER as, float *res)
{
  HnAssert(as->resource != NULL,
    "KheAvailSolverMaxWorkload: resource not set");
  if( as->best_workload != NULL )
    return *res = (float) AvailSetFinalLimit(as, as->best_workload), true;
  else
    return *res = FLT_MAX, false;
}


/*****************************************************************************/
/*                                                                           */
/*  int KheAvailSolverMaxBusyTimesAvailNodeCount(KHE_AVAIL_SOLVER as)        */
/*                                                                           */
/*  Return the number of avail nodes in the independent set that defines     */
/*  the maximum busy times of the set resource.                              */
/*                                                                           */
/*****************************************************************************/

int KheAvailSolverMaxBusyTimesAvailNodeCount(KHE_AVAIL_SOLVER as)
{
  HnAssert(as->resource != NULL,
    "KheAvailSolverMaxBusyTimesAvailNodeCount: resource not set");
  return as->best_busy_times == NULL ? 0 :
    HaArrayCount(as->best_busy_times->nodes);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheAvailSolverMaxBusyTimesAvailNode(KHE_AVAIL_SOLVER as, int i,     */
/*    KHE_AVAIL_NODE_TYPE *type, int *limit, KHE_TIME_SET *ts,KHE_MONITOR *m)*/
/*                                                                           */
/*  Return the fields of the i'th avail node of the independent set that     */
/*  defines the maximum busy times of the set resource.                      */
/*                                                                           */
/*****************************************************************************/

void KheAvailSolverMaxBusyTimesAvailNode(KHE_AVAIL_SOLVER as, int i,
  KHE_AVAIL_NODE_TYPE *type, int *limit, KHE_TIME_SET *ts, KHE_MONITOR *m)
{
  AVAIL_NODE an;
  HnAssert(as->resource != NULL,
    "KheAvailSolverMaxBusyTimesAvailNode: resource not set");
  HnAssert(as->best_busy_times != NULL,
    "KheAvailSolverMaxBusyTimesAvailNode: no max busy times");
  an = HaArray(as->best_busy_times->nodes, i);
  *type = an->type;
  *limit = an->limit;
  *ts = an->time_set;
  *m = an->monitor;
}


/*****************************************************************************/
/*                                                                           */
/*  int KheAvailSolverMaxWorkloadAvailNodeCount(KHE_AVAIL_SOLVER as)         */
/*                                                                           */
/*  Return the number of avail nodes in the independent set that defines     */
/*  the maximum workload of the set resource.                                */
/*                                                                           */
/*****************************************************************************/

int KheAvailSolverMaxWorkloadAvailNodeCount(KHE_AVAIL_SOLVER as)
{
  HnAssert(as->resource != NULL,
    "KheAvailSolverMaxWorkloadAvailNodeCount: resource not set");
  return as->best_workload == NULL ? 0 :
    HaArrayCount(as->best_workload->nodes);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheAvailSolverMaxWorkloadAvailNode(KHE_AVAIL_SOLVER as, int i,      */
/*    KHE_AVAIL_NODE_TYPE *type, int *limit, KHE_TIME_SET *ts,KHE_MONITOR *m)*/
/*                                                                           */
/*  Return the fields of the i'th avail node of the independent set that     */
/*  defines the maximum workload of the set resource.                        */
/*                                                                           */
/*****************************************************************************/

void KheAvailSolverMaxWorkloadAvailNode(KHE_AVAIL_SOLVER as, int i,
  KHE_AVAIL_NODE_TYPE *type, int *limit, KHE_TIME_SET *ts, KHE_MONITOR *m)
{
  AVAIL_NODE an;
  HnAssert(as->resource != NULL,
    "KheAvailSolverMaxWorkloadAvailNode: resource not set");
  HnAssert(as->best_workload != NULL,
    "KheAvailSolverMaxWorkloadAvailNode: no max workload");
  an = HaArray(as->best_workload->nodes, i);
  *type = an->type;
  *limit = an->limit;
  *ts = an->time_set;
  *m = an->monitor;
}
