
/*****************************************************************************/
/*                                                                           */
/*  THE KHE HIGH SCHOOL TIMETABLING ENGINE                                   */
/*  COPYRIGHT (C) 2010 Jeffrey H. Kingston                                   */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 3, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         khe_cluster_busy_times_monitor.c                           */
/*  DESCRIPTION:  A cluster busy times monitor                               */
/*                                                                           */
/*****************************************************************************/
#include "khe_interns.h"

#define DEBUG1 0
#define DEBUG2 0
#define DEBUG3 0
#define DEBUG5 0
#define DEBUG6 0
#define DEBUG7 0
#define DEBUG8 0


/*****************************************************************************/
/*                                                                           */
/*  KHE_TIME_GROUP_INFO                                                      */
/*                                                                           */
/*  A time group info object, called a time group in the implementation      */
/*  chapter of the User's Guide.  It contains everything we need to know     */
/*  about the time group at one real point, notably its state:               */
/*                                                                           */
/*    KHE_INACTIVE                                                           */
/*      The time group is not active, or not attached (because the           */
/*      monitor is not attached).                                            */
/*                                                                           */
/*    KHE_ACTIVE                                                             */
/*      The time group is active.  It contributes a value of 1 to the        */
/*      active_group_count attribute.                                        */
/*                                                                           */
/*    KHE_OPEN                                                               */
/*      The time group is attached, at or after the cutoff index, and        */
/*      not busy, so its state is not clearly either active or inactive.     */
/*      It contributes a value of 1 to the open_group_count attribute.       */
/*                                                                           */
/*  When checking for violations of minimum limits, open time groups         */
/*  count as active, because they could be active, and this gives the        */
/*  smallest (most conservative) estimate of cost.  When checking for        */
/*  violations of maximum limits, open time groups count as inactive,        */
/*  because they could be inactive, and again this gives the smallest        */
/*  (most conservative) estimate of cost.                                    */
/*                                                                           */
/*  A time group may be virtual:  logically present, but without any         */
/*  time group info object.  Virtual time groups have no time group or       */
/*  polarity.  When the monitor is attached, virtual time groups in the      */
/*  history_before range have state KHE_ACTIVE, while time groups in the     */
/*  history_after range have state KHE_OPEN.  When the monitor is not        */
/*  attached, like other time groups virtual ones have state KHE_INACTIVE.   */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  KHE_INACTIVE,
  KHE_ACTIVE,
  KHE_OPEN
} KHE_TIME_GROUP_STATE;

typedef struct khe_time_group_info_rec {
  KHE_MONITORED_TIME_GROUP	monitored_time_group;	/* the time group    */
  KHE_POLARITY			polarity;	/* polarity of time group    */
  KHE_TIME_GROUP_STATE		state;		/* state of time group       */
  KHE_TIME_GROUP_STATE		not_busy_state;	/* from SetNotBusy           */
} KHE_TIME_GROUP_INFO;


/*****************************************************************************/
/*                                                                           */
/*  KHE_CLUSTER_BUSY_TIMES_MONITOR - monitors busy times                     */
/*                                                                           */
/*  I've deleted history_before, history_after, and history to save space.   */
/*  There is no significant hit to running time, as I've verified by         */
/*  looking at all the places where these attributes are used.               */
/*                                                                           */
/*****************************************************************************/

struct khe_cluster_busy_times_monitor_rec {
  INHERIT_MONITOR(allow_zero, deviation)
  KHE_RESOURCE_IN_SOLN		resource_in_soln;	/* enclosing rs      */
  KHE_CLUSTER_BUSY_TIMES_CONSTRAINT constraint;		/* monitoring this   */
  int				offset;			/* monitoring this   */
  int				minimum;		/* from constraint   */
  int				maximum;		/* from constraint   */
  int				multiplier;		/* multiplier        */
  /* int			history_before; */	/* history: ai       */
  /* int			history_after; */	/* history: ci       */
  /* int			history; */		/* history: xi       */
  int				cutoff_index;		/* cutoff index      */
  KHE_TIME                      cached_cutoff_time;     /* prev call         */
  int                           cached_cutoff_index;    /* prev call         */
  int				active_group_count;	/* active groups     */
  int				open_group_count;	/* open groups       */
  int				time_group_info_count;	/* len of tgi        */
  KHE_COST			history_adjustment;	/* prev inst cost    */
  /* KHE_FRAME			frame; */		/* optional frame    */
  KHE_TIME			first_time;		/* first time        */
  KHE_TIME			last_time;		/* last time         */
  KHE_CLUSTER_BUSY_TIMES_MONITOR copy;			/* used when copying */
  KHE_TIME_GROUP_INFO		time_group_info[1];	/* extends!          */
};


/*****************************************************************************/
/*                                                                           */
/*  Submodule "points and ranges"                                            */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool KhePointIsReal(KHE_CLUSTER_BUSY_TIMES_MONITOR m, int index)         */
/*                                                                           */
/*  Return true if point is real.                                            */
/*                                                                           */
/*****************************************************************************/

static bool KhePointIsReal(KHE_CLUSTER_BUSY_TIMES_MONITOR m, int index)
{
  return 0 <= index && index < m->time_group_info_count;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_TIME_GROUP_STATE KheRealPointState(                                  */
/*    KHE_CLUSTER_BUSY_TIMES_MONITOR m, int index)                           */
/*                                                                           */
/*  Return the state of this real point.                                     */
/*                                                                           */
/*****************************************************************************/

static KHE_TIME_GROUP_STATE KheRealPointState(
  KHE_CLUSTER_BUSY_TIMES_MONITOR m, int index)
{
  return m->time_group_info[index].state;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_TIME_GROUP_STATE KheRealAttachedPointNewState(                       */
/*    KHE_CLUSTER_BUSY_TIMES_MONITOR m, int index, int busy_count,           */
/*    int cutoff_index)                                                      */
/*                                                                           */
/*  Assuming that (m, index) is a real and attached point, work out          */
/*  its new state given these values of busy_count and cutoff_index.         */
/*                                                                           */
/*****************************************************************************/

static KHE_TIME_GROUP_STATE KheRealAttachedPointNewState(
  KHE_CLUSTER_BUSY_TIMES_MONITOR m, int index, int busy_count,
  int cutoff_index)
{
  KHE_POLARITY po;
  if( busy_count == 0 && index >= cutoff_index )
    return m->time_group_info[index].not_busy_state;
    /* return KHE_OPEN; */
  else
  {
    po = m->time_group_info[index].polarity;
    if( (po == KHE_NEGATIVE) == (busy_count == 0) )
      return KHE_ACTIVE;
    else
      return KHE_INACTIVE;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheRealPointSetState(KHE_CLUSTER_BUSY_TIMES_MONITOR m,              */
/*    int index, KHE_TIME_GROUP_STATE new_state)                             */
/*                                                                           */
/*  Set the state of the real point at index.                                */
/*                                                                           */
/*****************************************************************************/
static char *KheTimeGroupStateShow(KHE_TIME_GROUP_STATE state);

static void KheRealPointSetState(KHE_CLUSTER_BUSY_TIMES_MONITOR m,
  int index, KHE_TIME_GROUP_STATE new_state)
{
  HnAssert(KhePointIsReal(m, index), "KheRealPointSetState internal error: "
    "index %d out of range 0 .. %d", index, m->time_group_info_count - 1);
  m->time_group_info[index].state = new_state;
  if( DEBUG5 )
    fprintf(stderr, "KheRealPointSetState(%s, %d, %s) assigned %s\n",
      KheClusterBusyTimesMonitorId(m), index,
      KheTimeGroupStateShow(new_state),
      KheTimeGroupStateShow(KheRealPointState(m, index)));
}


/*****************************************************************************/
/*                                                                           */
/*  void KheRealPointCheck(KHE_CLUSTER_BUSY_TIMES_MONITOR m, int index)      */
/*                                                                           */
/*  Check that the real point at this index is consistent.                   */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused, but worth keeping just in case
static void KheRealPointCheck(KHE_CLUSTER_BUSY_TIMES_MONITOR m,
  int index)
{
  KHE_TIME_GROUP_INFO mi;  int busy_count;  KHE_TIME_GROUP_STATE ns;
  mi = MArrayGet(m->time_group_info, index);
  if( !m->attached )
    HnAssert(mi.state == KHE_INACTIVE, "KheRealPointCheck(%s, %d) error 1",
      KheClusterBusyTimesMonitorId(m), index);
  else
  {
    busy_count = KheMonitoredTimeGroupBusyCount(mi.monitored_time_group);
    ns = KheRealAttachedPointNewState(m, index, busy_count, m->cutoff_index);
    HnAssert(mi.state == ns, "KheRealPointCheck(%s, %d) error 2: recorded %s, "
      "calculated %s:\n  (polarity %s, busy_count %d, cutoff_index %d)",
      KheClusterBusyTimesMonitorId(m), index,
      KheTimeGroupStateShow(mi.state), KheTimeGroupStateShow(ns),
      mi.polarity == KHE_POSITIVE ? "positive" : "negative", busy_count,
      m->cutoff_index);
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  char *KheTimeGroupStateShow(KHE_TIME_GROUP_STATE state)                  */
/*                                                                           */
/*  Return a string value showing state.                                     */
/*                                                                           */
/*****************************************************************************/

static char *KheTimeGroupStateShow(KHE_TIME_GROUP_STATE state)
{
  switch( state )
  {
    case KHE_INACTIVE:	return "inactive";
    case KHE_ACTIVE:	return "active";
    case KHE_OPEN:	return "open";
    default:		return "?";
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "real point state changes"                                     */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  int KheClusterDev(KHE_CLUSTER_BUSY_TIMES_MONITOR m,                      */
/*    int active_group_count, int open_group_count)                          */
/*                                                                           */
/*  Return the deviation of m, given this number of active and open time     */
/*  groups.                                                                  */
/*                                                                           */
/*****************************************************************************/

static int KheClusterDev(KHE_CLUSTER_BUSY_TIMES_MONITOR m,
  int active_group_count, int open_group_count)
{
  if( m->allow_zero && active_group_count == 0 )
    return 0;
  else if( active_group_count > m->maximum )
    return active_group_count - m->maximum;
  else if( active_group_count + open_group_count < m->minimum )
    return m->minimum - (active_group_count + open_group_count);
  else
    return 0;
}


/*****************************************************************************/
/*                                                                           */
/*  void Flush(KHE_CLUSTER_BUSY_TIMES_MONITOR m, bool force)                 */
/*                                                                           */
/*  Assuming that m is attached and that m->active_group_count and           */
/*  m->open_group_count are up to date but that m->deviation may not be,     */
/*  bring m->deviation and m->cost up to date and inform parents of any      */
/*  change in cost.                                                          */
/*                                                                           */
/*  If force is true, change the cost even if the deviation has not          */
/*  changed.  This is used only when the multiplier has changed.             */
/*                                                                           */
/*****************************************************************************/

static void Flush(KHE_CLUSTER_BUSY_TIMES_MONITOR m, bool force)
{
  int new_dev;  KHE_COST new_cost;
  new_dev = KheClusterDev(m, m->active_group_count, m->open_group_count);
  if( new_dev != m->deviation || force )
  {
    m->deviation = new_dev;
    new_cost = (KheConstraintCost((KHE_CONSTRAINT) m->constraint, new_dev)
      - m->history_adjustment) * m->multiplier;
    if( DEBUG8 )
      fprintf(stderr, "  %s from %.5f --(active %d, open %d)--> %.5f\n",
	KheMonitorId((KHE_MONITOR) m), KheCostShow(m->cost),
          m->active_group_count, m->open_group_count, KheCostShow(new_cost));
    KheMonitorChangeCost((KHE_MONITOR) m, new_cost);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheRealPointChangeState(KHE_CLUSTER_BUSY_TIMES_MONITOR m,           */
/*    int index, KHE_TIME_GROUP_STATE new_state)                             */
/*                                                                           */
/*  Change the state of the real time group at index to new_state,           */
/*  and update everything.                                                   */
/*                                                                           */
/*****************************************************************************/

static void KheRealPointChangeState(KHE_CLUSTER_BUSY_TIMES_MONITOR m,
  int index, KHE_TIME_GROUP_STATE new_state)
{
  KHE_TIME_GROUP_STATE prev_state;

  if( DEBUG6 )
    fprintf(stderr, "[ KheRealPointChangeState(%s, %d, %s -> %s)\n",
      KheClusterBusyTimesMonitorId(m), index,
      KheTimeGroupStateShow(KheRealPointState(m, index)),
      KheTimeGroupStateShow(new_state));

  /* save the old state and change the state */
  prev_state = KheRealPointState(m, index);
  KheRealPointSetState(m, index, new_state);

  /* update active_group_count and open_group_count and flush */
  switch( prev_state )
  {
    case KHE_INACTIVE:

      switch( new_state )
      {
	case KHE_INACTIVE:

	  /* nothing to do here */
	  break;

	case KHE_ACTIVE:

          m->active_group_count++;
	  Flush(m, false);
	  break;

	case KHE_OPEN:

          m->open_group_count++;
	  Flush(m, false);
	  break;
      }
      break;

    case KHE_ACTIVE:

      switch( new_state )
      {
	case KHE_INACTIVE:

          m->active_group_count--;
	  Flush(m, false);
	  break;

	case KHE_ACTIVE:

	  /* nothing to do here */
	  break;

	case KHE_OPEN:

          m->active_group_count--;
          m->open_group_count++;
	  Flush(m, false);
	  break;
      }
      break;

    case KHE_OPEN:

      switch( new_state )
      {
	case KHE_INACTIVE:

          m->open_group_count--;
	  Flush(m, false);
	  break;

	case KHE_ACTIVE:

          m->open_group_count--;
          m->active_group_count++;
	  Flush(m, false);
	  break;

	case KHE_OPEN:

	  /* nothing to do here */
	  break;
      }
      break;
  }
  if( DEBUG6 )
    fprintf(stderr, "] KheRealPointChangeState returning\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "construction and query"                                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  int KheClusterMtgInfoCmp(const void *t1, const void *t2)                 */
/*                                                                           */
/*  Comparison function for sorting time groups by increasing index          */
/*  of last (latest) time.                                                   */
/*                                                                           */
/*****************************************************************************/

static int KheClusterMtgInfoCmp(const void *t1, const void *t2)
{
  KHE_TIME_GROUP tg1, tg2;  KHE_TIME time1, time2;
  KHE_TIME_GROUP_INFO mi1 = * (KHE_TIME_GROUP_INFO *) t1;
  KHE_TIME_GROUP_INFO mi2 = * (KHE_TIME_GROUP_INFO *) t2;
  tg1 = KheMonitoredTimeGroupTimeGroup(mi1.monitored_time_group);
  tg2 = KheMonitoredTimeGroupTimeGroup(mi2.monitored_time_group);
  if( KheTimeGroupTimeCount(tg1) == 0 )
    return -1;
  else if( KheTimeGroupTimeCount(tg2) == 0 )
    return 1;
  else
  {
    time1 = KheTimeGroupTime(tg1, KheTimeGroupTimeCount(tg1) - 1);
    time2 = KheTimeGroupTime(tg2, KheTimeGroupTimeCount(tg2) - 1);
    return KheTimeIndex(time1) - KheTimeIndex(time2);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_COST KheClusterBusyTimesMonitorGetHistoryAdjustment(                 */
/*    KHE_CLUSTER_BUSY_TIMES_MONITOR m)                                      */
/*                                                                           */
/*  Find the history adjustment for m (in the previous instance).            */
/*                                                                           */
/*****************************************************************************/

static KHE_COST KheClusterBusyTimesMonitorGetHistoryAdjustment(
  KHE_CLUSTER_BUSY_TIMES_MONITOR m)
{
  int history_before, history_after, history;  KHE_RESOURCE r;
  history_before = KheClusterBusyTimesConstraintHistoryBefore(m->constraint);
  history_after = KheClusterBusyTimesConstraintHistoryAfter(m->constraint);
  r = KheClusterBusyTimesMonitorResource(m);
  history = KheClusterBusyTimesConstraintHistory(m->constraint, r);
  if( history_before <= 0 )
    return 0;
  else
    return KheConstraintCost((KHE_CONSTRAINT) m->constraint, 
      KheClusterDev(m, history, history_after + m->cutoff_index));
  /* ***
  if( m->history_before <= 0 )
    return 0;
  else
    return KheConstraintCost((KHE_CONSTRAINT) m->constraint, 
      KheClusterDev(m, m->history, m->history_after + m->cutoff_index));
  *** */
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_CLUSTER_BUSY_TIMES_MONITOR KheClusterBusyTimesMonitorMake(           */
/*    KHE_RESOURCE_IN_SOLN rs, int offset,                                   */
/*    KHE_CLUSTER_BUSY_TIMES_CONSTRAINT c)                                   */
/*                                                                           */
/*  Make a new cluster busy times monitor for (rs, offset), monitoring c.    */
/*                                                                           */
/*****************************************************************************/

KHE_CLUSTER_BUSY_TIMES_MONITOR KheClusterBusyTimesMonitorMake(
  KHE_RESOURCE_IN_SOLN rs, int offset, KHE_CLUSTER_BUSY_TIMES_CONSTRAINT c)
{
  KHE_CLUSTER_BUSY_TIMES_MONITOR res;   KHE_SOLN soln;  /* KHE_RESOURCE r; */
  int i, count;  KHE_RESOURCE_TIMETABLE_MONITOR rtm;  KHE_TIME_GROUP tg;
  KHE_TIME_GROUP_INFO mi;  HA_ARENA a;

  count = KheClusterBusyTimesConstraintTimeGroupCount(c);
  soln = KheResourceInSolnSoln(rs);
  a = KheSolnArena(soln);
  res = HaAlloc(a, sizeof(struct khe_cluster_busy_times_monitor_rec)
    + (count - 1) * sizeof(KHE_TIME_GROUP_INFO));
  /* ***
  res = MMallocCount(sizeof(struct khe_cluster_busy_times_monitor_rec)
    + (count - 1) * sizeof(KHE_TIME_GROUP_INFO), 60);
  *** */
  HaArrayInit(res->parent_links, a);
  KheMonitorInitCommonFields((KHE_MONITOR) res, soln,
    KHE_CLUSTER_BUSY_TIMES_MONITOR_TAG);
  res->allow_zero = KheClusterBusyTimesConstraintAllowZero(c);
  res->deviation = 0;
  res->resource_in_soln = rs;
  res->constraint = c;
  res->offset = offset;
  res->minimum = KheClusterBusyTimesConstraintMinimum(c);
  res->maximum = KheClusterBusyTimesConstraintMaximum(c);
  res->multiplier = 1;
  /* ***
  res->history_before = KheClusterBusyTimesConstraintHistoryBefore(c);
  res->history_after = KheClusterBusyTimesConstraintHistoryAfter(c);
  *** */
  /* r = KheResourceInSolnResource(rs); */
  /* res->history = KheClusterBusyTimesConstraintHistory(c, r); */
  res->cutoff_index = KheClusterBusyTimesConstraintTimeGroupCount(c);
  res->cached_cutoff_index = -1;
  res->cached_cutoff_time = NULL;
  res->active_group_count = 0;
  res->open_group_count = 0;
  res->history_adjustment = KheClusterBusyTimesMonitorGetHistoryAdjustment(res);
  /* res->frame = KheFrameMakeNull(); */
  res->first_time = NULL;
  res->last_time = NULL;
  res->copy = NULL;
  res->time_group_info_count = res->cutoff_index;
  rtm = KheResourceInSolnTimetableMonitor(rs);
  for( i = 0;  i < res->time_group_info_count;  i++ )
  {
    tg = KheClusterBusyTimesConstraintTimeGroup(c, i, offset, &mi.polarity);
    mi.monitored_time_group = 
      KheResourceTimetableMonitorAddMonitoredTimeGroup(rtm, tg);
    mi.state = KHE_INACTIVE;
    mi.not_busy_state = KHE_OPEN;
    res->time_group_info[i] = mi;
  }
  qsort(res->time_group_info, res->time_group_info_count,
    sizeof(KHE_TIME_GROUP_INFO), &KheClusterMtgInfoCmp);
  KheResourceInSolnAddMonitor(rs, (KHE_MONITOR) res);
  if( DEBUG8 )
    fprintf(stderr, "  %s after making, cost is %.5f\n",
      KheMonitorId((KHE_MONITOR) res), KheCostShow(res->cost));
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_CLUSTER_BUSY_TIMES_MONITOR KheClusterBusyTimesMonitorCopyPhase1(     */
/*    KHE_CLUSTER_BUSY_TIMES_MONITOR m, HA_ARENA a)                          */
/*                                                                           */
/*  Carry out Phase 1 of the copying of m.                                   */
/*                                                                           */
/*****************************************************************************/

KHE_CLUSTER_BUSY_TIMES_MONITOR KheClusterBusyTimesMonitorCopyPhase1(
  KHE_CLUSTER_BUSY_TIMES_MONITOR m, HA_ARENA a)
{
  KHE_CLUSTER_BUSY_TIMES_MONITOR copy;  int i;  KHE_TIME_GROUP_INFO mi, mi2;
  if( m->copy == NULL )
  {
    copy = HaAlloc(a, sizeof(struct khe_cluster_busy_times_monitor_rec)
      + (m->time_group_info_count - 1) * sizeof(KHE_TIME_GROUP_INFO));
    /* ***
    copy = MMallocCount(sizeof(struct khe_cluster_busy_times_monitor_rec)
      + (m->time_group_info_count - 1) * sizeof(KHE_TIME_GROUP_INFO), 60);
    *** */
    m->copy = copy;
    KheMonitorCopyCommonFieldsPhase1((KHE_MONITOR) copy, (KHE_MONITOR) m, a);
    copy->allow_zero = m->allow_zero;
    copy->deviation = m->deviation;
    copy->resource_in_soln = KheResourceInSolnCopyPhase1(m->resource_in_soln,a);
    copy->constraint = m->constraint;
    copy->offset = m->offset;
    copy->minimum = m->minimum;
    copy->maximum = m->maximum;
    copy->multiplier = m->multiplier;
    /* ***
    copy->history_before = m->history_before;
    copy->history_after = m->history_after;
    copy->history = m->history;
    *** */
    copy->cutoff_index = m->cutoff_index;
    copy->cached_cutoff_index = m->cached_cutoff_index;
    copy->cached_cutoff_time = m->cached_cutoff_time;
    copy->active_group_count = m->active_group_count;
    copy->open_group_count = m->open_group_count;
    copy->history_adjustment = m->history_adjustment;
    copy->first_time = m->first_time;
    copy->last_time = m->last_time;
    /* copy->frame = KheFrameMakeNull(); */
    copy->copy = NULL;
    for( i = 0;  i < m->time_group_info_count;  i++ )
    {
      mi = m->time_group_info[i];
      mi2.monitored_time_group =
	KheMonitoredTimeGroupCopyPhase1(mi.monitored_time_group, a);
      mi2.polarity = mi.polarity;
      mi2.state = mi.state;
      mi2.not_busy_state = mi.not_busy_state;
      copy->time_group_info[i] = mi2;
    }
  }
  return m->copy;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheClusterBusyTimesMonitorCopyPhase2(                               */
/*    KHE_CLUSTER_BUSY_TIMES_MONITOR m)                                      */
/*                                                                           */
/*  Carry out Phase 2 of the copying of m.                                   */
/*                                                                           */
/*****************************************************************************/

void KheClusterBusyTimesMonitorCopyPhase2(KHE_CLUSTER_BUSY_TIMES_MONITOR m)
{
  int i;  KHE_TIME_GROUP_INFO mi;
  if( m->copy != NULL )
  {
    m->copy = NULL;
    KheMonitorCopyCommonFieldsPhase2((KHE_MONITOR) m);
    KheResourceInSolnCopyPhase2(m->resource_in_soln);
    for( i = 0;  i < m->time_group_info_count;  i++ )
    {
      mi = m->time_group_info[i];
      KheMonitoredTimeGroupCopyPhase2(mi.monitored_time_group);
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheClusterBusyTimesMonitorDelete(KHE_CLUSTER_BUSY_TIMES_MONITOR m)  */
/*                                                                           */
/*  Delete m.                                                                */
/*                                                                           */
/*****************************************************************************/

/* ***
void KheClusterBusyTimesMonitorDelete(KHE_CLUSTER_BUSY_TIMES_MONITOR m)
{
  if( !KheFrameIsNull(m->frame) )
    KheFrameDelete(m->frame);
  if( m->attached )
    KheClusterBusyTimesMonitorDetachFromSoln(m);
  KheMonitorDeleteAllParentMonitors((KHE_MONITOR) m);
  KheResourceInSolnDeleteMonitor(m->resource_in_soln, (KHE_MONITOR) m);
  KheSolnDeleteMonitor(m->soln, (KHE_MONITOR) m);
  MFree(m);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_RESOURCE_IN_SOLN KheClusterBusyTimesMonitorResourceInSoln(           */
/*    KHE_CLUSTER_BUSY_TIMES_MONITOR m)                                      */
/*                                                                           */
/*  Return the resource monitor holding m.                                   */
/*                                                                           */
/*****************************************************************************/

KHE_RESOURCE_IN_SOLN KheClusterBusyTimesMonitorResourceInSoln(
  KHE_CLUSTER_BUSY_TIMES_MONITOR m)
{
  return m->resource_in_soln;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_CLUSTER_BUSY_TIMES_CONSTRAINT KheClusterBusyTimesMonitorConstraint(  */
/*    KHE_CLUSTER_BUSY_TIMES_MONITOR m)                                      */
/*                                                                           */
/*  Return the contraint that m is monitoring.                               */
/*                                                                           */
/*****************************************************************************/

KHE_CLUSTER_BUSY_TIMES_CONSTRAINT KheClusterBusyTimesMonitorConstraint(
  KHE_CLUSTER_BUSY_TIMES_MONITOR m)
{
  return m->constraint;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_RESOURCE KheClusterBusyTimesMonitorResource(                         */
/*    KHE_CLUSTER_BUSY_TIMES_MONITOR m)                                      */
/*                                                                           */
/*  Return the resource that m is monitoring.                                */
/*                                                                           */
/*****************************************************************************/

KHE_RESOURCE KheClusterBusyTimesMonitorResource(
  KHE_CLUSTER_BUSY_TIMES_MONITOR m)
{
  return KheResourceInSolnResource(m->resource_in_soln);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheClusterBusyTimesMonitorHistoryBefore(                             */
/*    KHE_CLUSTER_BUSY_TIMES_MONITOR m)                                      */
/*                                                                           */
/*  Return the history before value (ai) stored in m.                        */
/*                                                                           */
/*****************************************************************************/

int KheClusterBusyTimesMonitorHistoryBefore(KHE_CLUSTER_BUSY_TIMES_MONITOR m)
{
  return KheClusterBusyTimesConstraintHistoryBefore(m->constraint);
  /* return m->history_before; */
}


/*****************************************************************************/
/*                                                                           */
/*  int KheClusterBusyTimesMonitorHistoryAfter(                              */
/*    KHE_CLUSTER_BUSY_TIMES_MONITOR m)                                      */
/*                                                                           */
/*  Return the history after value (ci) stored in m.                         */
/*                                                                           */
/*****************************************************************************/

int KheClusterBusyTimesMonitorHistoryAfter(KHE_CLUSTER_BUSY_TIMES_MONITOR m)
{
  return KheClusterBusyTimesConstraintHistoryAfter(m->constraint);
  /* return m->history_after; */
}


/*****************************************************************************/
/*                                                                           */
/*  int KheClusterBusyTimesMonitorHistory(KHE_CLUSTER_BUSY_TIMES_MONITOR m)  */
/*                                                                           */
/*  Return the history value (xi) stored in m.                               */
/*                                                                           */
/*****************************************************************************/

int KheClusterBusyTimesMonitorHistory(KHE_CLUSTER_BUSY_TIMES_MONITOR m)
{
  KHE_RESOURCE r;
  r = KheClusterBusyTimesMonitorResource(m);
  return KheClusterBusyTimesConstraintHistory(m->constraint, r);
  /* return m->history; */
}


/*****************************************************************************/
/*                                                                           */
/*  int KheClusterBusyTimesMonitorOffset(KHE_CLUSTER_BUSY_TIMES_MONITOR m)   */
/*                                                                           */
/*  Return the offset that m is monitoring.                                  */
/*                                                                           */
/*****************************************************************************/

int KheClusterBusyTimesMonitorOffset(KHE_CLUSTER_BUSY_TIMES_MONITOR m)
{
  return m->offset;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheClusterBusyTimesMonitorActiveTimeGroupCount(                     */
/*    KHE_CLUSTER_BUSY_TIMES_MONITOR m, int *active_group_count,             */
/*    int *open_group_count, int *minimum, int *maximum, bool *allow_zero)   */
/*                                                                           */
/*  Set *active_group_count to the current number of active time groups,     */
/*  *open_group_count to the current number of open time groups, and         */
/*  *minimum, *maximum, and *allow_zero to the Minimum, Maximum, and         */
/*  AllowZero attributes of the corresponding constraint.                    */
/*                                                                           */
/*****************************************************************************/

void KheClusterBusyTimesMonitorActiveTimeGroupCount(
  KHE_CLUSTER_BUSY_TIMES_MONITOR m, int *active_group_count,
  int *open_group_count, int *minimum, int *maximum, bool *allow_zero)
{
  *active_group_count = m->active_group_count;
  *open_group_count = m->open_group_count;
  *minimum = m->minimum;
  *maximum = m->maximum;
  *allow_zero = m->allow_zero;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "reporting state during the solve"                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  int KheClusterBusyTimesMonitorTimeGroupCount(                            */
/*    KHE_CLUSTER_BUSY_TIMES_MONITOR m)                                      */
/*                                                                           */
/*  Return the number of time groups being monitored by m.                   */
/*                                                                           */
/*****************************************************************************/

int KheClusterBusyTimesMonitorTimeGroupCount(KHE_CLUSTER_BUSY_TIMES_MONITOR m)
{
  return m->time_group_info_count;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_TIME_GROUP KheClusterBusyTimesMonitorTimeGroup(                      */
/*    KHE_CLUSTER_BUSY_TIMES_MONITOR m, int i, KHE_POLARITY *po)             */
/*                                                                           */
/*  Return the i'th time group being monitored by m, and its polarity.       */
/*                                                                           */
/*****************************************************************************/

KHE_TIME_GROUP KheClusterBusyTimesMonitorTimeGroup(
  KHE_CLUSTER_BUSY_TIMES_MONITOR m, int i, KHE_POLARITY *po)
{
  KHE_TIME_GROUP_INFO mi;
  mi = m->time_group_info[i];
  *po = mi.polarity;
  return KheMonitoredTimeGroupTimeGroup(mi.monitored_time_group);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheClusterBusyTimesMonitorRange(KHE_CLUSTER_BUSY_TIMES_MONITOR m,   */
/*    KHE_TIME *first_time, KHE_TIME *last_time)                             */
/*                                                                           */
/*  Return the time range of m.                                              */
/*                                                                           */
/*****************************************************************************/

void KheClusterBusyTimesMonitorRange(KHE_CLUSTER_BUSY_TIMES_MONITOR m,
  KHE_TIME *first_time, KHE_TIME *last_time)
{
  int i, first_index, last_index, index, count;  KHE_TIME_GROUP tg;
  KHE_INSTANCE ins;
  if( m->first_time == NULL )
  {
    first_index = INT_MAX;
    last_index = 0;
    for( i = 0;  i < m->time_group_info_count;  i++ )
    {
      tg = KheMonitoredTimeGroupTimeGroup(
	m->time_group_info[i].monitored_time_group);
      count = KheTimeGroupTimeCount(tg);
      if( count > 0 )
      {
	index = KheTimeIndex(KheTimeGroupTime(tg, 0));
	if( index < first_index )
          first_index = index;
	index = KheTimeIndex(KheTimeGroupTime(tg, count - 1));
	if( index > last_index )
          last_index = index;
      }
    }
    if( first_index < INT_MAX )
    {
      ins = KheSolnInstance(m->soln);
      m->first_time = KheInstanceTime(ins, first_index);
      m->last_time = KheInstanceTime(ins, last_index);
    }
  }
  *first_time = m->first_time;
  *last_time = m->last_time;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheClusterBusyTimesMonitorTimeGroupIsActive(                        */
/*    KHE_CLUSTER_BUSY_TIMES_MONITOR m, int i, KHE_TIME_GROUP *tg,           */
/*    KHE_POLARITY *po, int *busy_count)                                     */
/*                                                                           */
/*  Similar to KheClusterBusyTimesMonitorTimeGroupState, only returning      */
/*  true when tg is active.                                                  */
/*                                                                           */
/*****************************************************************************/

bool KheClusterBusyTimesMonitorTimeGroupIsActive(
  KHE_CLUSTER_BUSY_TIMES_MONITOR m, int i, KHE_TIME_GROUP *tg,
  KHE_POLARITY *po, int *busy_count)
{
  KHE_TIME_GROUP_INFO mi;
  mi = m->time_group_info[i];
  *tg = KheMonitoredTimeGroupTimeGroup(mi.monitored_time_group);
  *po = mi.polarity;
  *busy_count = KheMonitoredTimeGroupBusyCount(mi.monitored_time_group);
  return (*po == KHE_NEGATIVE) == (*busy_count == 0);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheClusterBusyTimesMonitorAtMaxLimitCount(                           */
/*    KHE_CLUSTER_BUSY_TIMES_MONITOR m)                                      */
/*                                                                           */
/*  Return true if m is not violating but is at the maximum limit.           */
/*                                                                           */
/*****************************************************************************/

int KheClusterBusyTimesMonitorAtMaxLimitCount(KHE_CLUSTER_BUSY_TIMES_MONITOR m)
{
  return m->deviation == 0 && m->active_group_count == m->maximum ? 1 : 0;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "cutoff index and cutoff time"                                 */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheRealPointChangeCutoffIndex(KHE_CLUSTER_BUSY_TIMES_MONITOR m,     */
/*    int index, int cutoff_index)                                           */
/*                                                                           */
/*  There is a new cutoff_index (the one passed), m is attached, and this    */
/*  could affect the state of the real point at index.  Update everything.   */
/*                                                                           */
/*****************************************************************************/

static void KheRealPointChangeCutoffIndex(KHE_CLUSTER_BUSY_TIMES_MONITOR m,
  int index, int cutoff_index)
{
  KHE_TIME_GROUP_INFO mi;  int busy;  KHE_TIME_GROUP_STATE new_state;
  mi = m->time_group_info[index];
  busy = KheMonitoredTimeGroupBusyCount(mi.monitored_time_group);
  new_state = KheRealAttachedPointNewState(m, index, busy, cutoff_index);
  KheRealPointChangeState(m, index, new_state);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheClusterBusyTimesMonitorSetCutoffIndex(                           */
/*    KHE_CLUSTER_BUSY_TIMES_MONITOR m, int cutoff_index)                    */
/*                                                                           */
/*  Set the cutoff index of m to cutoff_index.                               */
/*                                                                           */
/*  Return true when something really is being cut off.                      */
/*                                                                           */
/*****************************************************************************/

bool KheClusterBusyTimesMonitorSetCutoffIndex(
  KHE_CLUSTER_BUSY_TIMES_MONITOR m, int cutoff_index)
{
  int i;
  if( DEBUG3 )
    fprintf(stderr, "   KheClusterBusyTimesMonitorSetCutoffIndex(%s, %d)\n",
      KheMonitorId((KHE_MONITOR) m), cutoff_index);
  HnAssert(0 <= cutoff_index && cutoff_index <= m->time_group_info_count,
    "KheClusterBusyTimesMonitorSetCutoffIndex: cutoff index (%d) out of"
    " range (0 .. %d)", cutoff_index, m->time_group_info_count);
  if( m->attached )
  {
    if( cutoff_index < m->cutoff_index )
    {
      /* cut off more than is cut off now */
      for( i = m->cutoff_index - 1;  i >= cutoff_index;  i-- )
	KheRealPointChangeCutoffIndex(m, i, cutoff_index);
    }
    else
    {
      /* cut off less than is cut off now */
      for( i = m->cutoff_index;  i < cutoff_index;  i++ )
	KheRealPointChangeCutoffIndex(m, i, cutoff_index);
    }
  }
  m->cutoff_index = cutoff_index;
  return cutoff_index < m->time_group_info_count;
}


/*****************************************************************************/
/*                                                                           */
/*  int KheClusterBusyTimesMonitorCutoffIndex(                               */
/*    KHE_CLUSTER_BUSY_TIMES_MONITOR m)                                      */
/*                                                                           */
/*  Return the cutoff index of m.                                            */
/*                                                                           */
/*****************************************************************************/

int KheClusterBusyTimesMonitorCutoffIndex(KHE_CLUSTER_BUSY_TIMES_MONITOR m)
{
  return m->cutoff_index;
}


/*****************************************************************************/
/*                                                                           */
/*  int KheClusterBusyTimesMonitorConvertCutoffTimeToIndex(                  */
/*    KHE_CLUSTER_BUSY_TIMES_MONITOR m, KHE_TIME cutoff_time)                */
/*                                                                           */
/*  Convert a cutoff time to a cutoff index.                                 */
/*                                                                           */
/*  If there has been a previous call to this function with a non-NULL       */
/*  cutoff time, the result of that call is stored in m->cached_cutoff_time  */
/*  and m->cached_cutoff_index.  Since the cutoff times typically increase   */
/*  with each call, this can save time.                                      */
/*                                                                           */
/*****************************************************************************/

static int KheClusterBusyTimesMonitorConvertCutoffTimeToIndex(
  KHE_CLUSTER_BUSY_TIMES_MONITOR m, KHE_TIME cutoff_time)
{
  int i;  KHE_TIME_GROUP_INFO mi;  KHE_TIME_GROUP tg;  KHE_TIME t;
  if( cutoff_time == NULL )
    i = m->time_group_info_count;
  else
  {
    /* find an initial index to search from, either cached or 0 */
    if( m->cached_cutoff_time != NULL &&
        KheTimeIndex(m->cached_cutoff_time) <= KheTimeIndex(cutoff_time) )
      i = m->cached_cutoff_index;
    else
      i = 0;

    /* search for the index from i */
    for( ;  i < m->time_group_info_count;  i++ )
    {
      mi = m->time_group_info[i];
      tg = KheMonitoredTimeGroupTimeGroup(mi.monitored_time_group);
      if( KheTimeGroupTimeCount(tg) > 0 )
      {
	t = KheTimeGroupTime(tg, KheTimeGroupTimeCount(tg) - 1);
	if( KheTimeIndex(t) > KheTimeIndex(cutoff_time) )
	  break;
      }
    }

    /* save a new cache value */
    m->cached_cutoff_time = cutoff_time;
    m->cached_cutoff_index = i;
  }
  return i;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheClusterBusyTimesMonitorSetCutoffTime(                            */
/*    KHE_CLUSTER_BUSY_TIMES_MONITOR m, KHE_TIME cutoff_time)                */
/*                                                                           */
/*  Set the cutoff index of m to the largest index such that all the         */
/*  time groups up to and including that index have times that are not       */
/*  later than cutoff_time.                                                  */
/*                                                                           */
/*  Return true when something really is being cut off.                      */
/*                                                                           */
/*****************************************************************************/

bool KheClusterBusyTimesMonitorSetCutoffTime(
  KHE_CLUSTER_BUSY_TIMES_MONITOR m, KHE_TIME cutoff_time)
{
  return KheClusterBusyTimesMonitorSetCutoffIndex(m,
    KheClusterBusyTimesMonitorConvertCutoffTimeToIndex(m, cutoff_time));
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_TIME KheClusterBusyTimesMonitorInitialCutoffTime(                    */
/*    KHE_CLUSTER_BUSY_TIMES_MONITOR m)                                      */
/*                                                                           */
/*  Return the first time t such that cutting off at t is not the same as    */
/*  cutting off at index 0, or NULL if there is no such time.                */
/*                                                                           */
/*****************************************************************************/

KHE_TIME KheClusterBusyTimesMonitorInitialCutoffTime(
  KHE_CLUSTER_BUSY_TIMES_MONITOR m)
{
  KHE_INSTANCE ins;  KHE_TIME_GROUP_INFO mi;  KHE_TIME_GROUP tg;

  if( m->time_group_info_count == 0 )
  {
    /* no time groups, so no time can produce a different cutoff */
    return NULL;
  }
  else
  {
    mi = m->time_group_info[0];
    tg = KheMonitoredTimeGroupTimeGroup(mi.monitored_time_group);
    if( KheTimeGroupTimeCount(tg) == 0 )
    {
      /* even cutting off at the first time will be different */
      ins = KheSolnInstance(m->soln);
      if( KheInstanceTimeCount(ins) == 0 )
	return NULL;
      else
	return KheInstanceTime(ins, 0);
    }
    else
    {
      /* cutting off at the last time of tg will be different */
      return KheTimeGroupTime(tg, KheTimeGroupTimeCount(tg) - 1);
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheClusterBusyTimesMonitorSetNotBusyState(                          */
/*    KHE_CLUSTER_BUSY_TIMES_MONITOR m, int i, bool active)                  */
/*                                                                           */
/*  Inform m that time group i should be considered active (if active is     */
/*  true) or inactive (if active is false) when it is not busy and it lies   */
/*  beyond the cutoff index.                                                 */
/*                                                                           */
/*****************************************************************************/

void KheClusterBusyTimesMonitorSetNotBusyState(
  KHE_CLUSTER_BUSY_TIMES_MONITOR m, int i, bool active)
{
  KHE_TIME_GROUP_STATE new_state;  int busy_count;
  KHE_MONITORED_TIME_GROUP mtg;
  m->time_group_info[i].not_busy_state = (active ? KHE_ACTIVE : KHE_INACTIVE);
  if( m->attached )
  {
    mtg = m->time_group_info[i].monitored_time_group;
    busy_count = KheMonitoredTimeGroupBusyCount(mtg);
    new_state = KheRealAttachedPointNewState(m, i, busy_count, m->cutoff_index);
    KheRealPointChangeState(m, i, new_state);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheClusterBusyTimesMonitorClearNotBusyState(                        */
/*    KHE_CLUSTER_BUSY_TIMES_MONITOR m, int i)                               */
/*                                                                           */
/*  Undo the corresponding KheClusterBusyTimesMonitorSetNotBusyState.        */
/*                                                                           */
/*****************************************************************************/

void KheClusterBusyTimesMonitorClearNotBusyState(
  KHE_CLUSTER_BUSY_TIMES_MONITOR m, int i)
{
  KHE_TIME_GROUP_STATE new_state;  int busy_count;
  KHE_MONITORED_TIME_GROUP mtg;
  m->time_group_info[i].not_busy_state = KHE_OPEN;
  if( m->attached )
  {
    mtg = m->time_group_info[i].monitored_time_group;
    busy_count = KheMonitoredTimeGroupBusyCount(mtg);
    new_state = KheRealAttachedPointNewState(m, i, busy_count, m->cutoff_index);
    KheRealPointChangeState(m, i, new_state);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "frame"                                                        */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_FRAME KheClusterBusyTimesMonitorFrame(                               */
/*    KHE_CLUSTER_BUSY_TIMES_MONITOR m)                                      */
/*                                                                           */
/*  Return the frame corresponding to m.                                     */
/*                                                                           */
/*  This is the full frame, unaffected by any cutoff index.                  */
/*                                                                           */
/*****************************************************************************/

/* ***
KHE_FRAME KheClusterBusyTimesMonitorFrame(KHE_CLUSTER_BUSY_TIMES_MONITOR m)
{
  int i;  KHE_TIME_GROUP tg;  KHE_POLARITY po;  KHE_FRAME_MAKE fm;
  KHE_SOLN soln;
  if( KheFrameIsNull(m->frame) )
  {
    soln = KheMonitorSoln((KHE_MONITOR) m);
    if( DEBUG1 )
      fprintf(stderr,
	"  calling KheFrameMakeBegin from KheClusterBusyTimesMonitorFrame\n");
    fm = KheFrameMakeBegin(soln);
    for( i = 0;  i < m->time_group_info_count;  i++ )
    {
      tg = KheClusterBusyTimesMonitorTimeGroup(m, i, &po);
      KheFrameMakeAddTimeGroup(fm, tg, po);
    }
    m->frame = KheFrameMakeEnd(fm, false);
    HnAssert(!KheFrameIsNull(m->frame),
      "KheClusterBusyTimesMonitorFrame internal error 1");
  }
  return m->frame;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  Submodule "attach and detach"                                            */
/*                                                                           */
/*  When m is unattached, all time groups are considered to be inactive,     */
/*  so the deviation is 0, as is the cost.                                   */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheClusterBusyTimesMonitorAttachToSoln(                             */
/*    KHE_CLUSTER_BUSY_TIMES_MONITOR m)                                      */
/*                                                                           */
/*  Attach m.  It is known to be currently detached with cost 0.             */
/*                                                                           */
/*****************************************************************************/

void KheClusterBusyTimesMonitorAttachToSoln(KHE_CLUSTER_BUSY_TIMES_MONITOR m)
{
  KHE_TIME_GROUP_INFO mi;  int i;  KHE_RESOURCE r;

  if( DEBUG2 )
  {
    fprintf(stderr, "[ KheClusterBusyTimesMonitorAttachToSoln(m)\n");
    KheClusterBusyTimesMonitorDebug(m, 2, 2, stderr);
  }

  /* check that current state is as expected when unattached */
  HnAssert(m->cost == 0,
    "KheClusterBusyTimesMonitorAttachToSoln internal error 1");
  HnAssert(m->deviation == 0,
    "KheClusterBusyTimesMonitorAttachToSoln internal error 2");
  HnAssert(m->active_group_count == 0,
    "KheClusterBusyTimesMonitorAttachToSoln internal error 3");
  HnAssert(m->open_group_count == 0,
    "KheClusterBusyTimesMonitorAttachToSoln internal error 4");

  /* move from unattached state to unlinked state */
  m->attached = true;
  r = KheClusterBusyTimesMonitorResource(m);
  m->active_group_count +=
    KheClusterBusyTimesConstraintHistory(m->constraint, r);
  /* m->active_group_count += m->history; */
  m->open_group_count +=
    KheClusterBusyTimesConstraintHistoryAfter(m->constraint);
  /* m->open_group_count += m->history_after; */
  Flush(m, false);

  /* attach each monitored time group; it will call back with a busy count */
  for( i = 0;  i < m->time_group_info_count;  i++ )
  {
    mi = m->time_group_info[i];
    KheMonitoredTimeGroupAttachMonitor(mi.monitored_time_group,
      (KHE_MONITOR) m, i);
  }

  if( DEBUG2 )
  {
    fprintf(stderr, "  at end:\n");
    KheClusterBusyTimesMonitorDebug(m, 2, 2, stderr);
    fprintf(stderr, "] KheClusterBusyTimesMonitorAttachToSoln(m)\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheClusterBusyTimesMonitorDetachFromSoln(                           */
/*    KHE_CLUSTER_BUSY_TIMES_MONITOR m)                                      */
/*                                                                           */
/*  Detach m.  It is known to be currently attached.                         */
/*                                                                           */
/*****************************************************************************/

void KheClusterBusyTimesMonitorDetachFromSoln(KHE_CLUSTER_BUSY_TIMES_MONITOR m)
{
  KHE_TIME_GROUP_INFO mi;  int i;  KHE_RESOURCE r;
  if( DEBUG2 )
  {
    fprintf(stderr, "[ KheClusterBusyTimesMonitorDetachFromSoln(m)\n");
    KheClusterBusyTimesMonitorDebug(m, 2, 2, stderr);
  }

  /* detach each monitored time group; it will call back to change state */
  for( i = 0;  i < m->time_group_info_count;  i++ )
  {
    mi = m->time_group_info[i];
    KheMonitoredTimeGroupDetachMonitor(mi.monitored_time_group,
      (KHE_MONITOR) m, i);
  }

  /* move from unlinked state to unattached state */
  r = KheClusterBusyTimesMonitorResource(m);
  m->active_group_count -=
    KheClusterBusyTimesConstraintHistory(m->constraint, r);
  /* m->active_group_count -= m->history; */
  m->open_group_count -=
    KheClusterBusyTimesConstraintHistoryAfter(m->constraint);
  /* m->open_group_count -= m->history_after; */
  HnAssert(m->active_group_count == 0,
    "KheClusterBusyTimesMonitorAttachToSoln internal error 3");
  HnAssert(m->open_group_count == 0,
    "KheClusterBusyTimesMonitorAttachToSoln internal error 4");
  m->deviation = 0;
  if( m->cost != 0 )
  {
    KheMonitorChangeCost((KHE_MONITOR) m, 0);
    m->cost = 0;
  }
  m->attached = false;

  if( DEBUG2 )
  {
    fprintf(stderr, "  at end:\n");
    KheClusterBusyTimesMonitorDebug(m, 2, 2, stderr);
    fprintf(stderr, "] KheClusterBusyTimesMonitorDetachFromSoln(m)\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "monitoring calls"                                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheClusterBusyTimesMonitorAddBusyAndIdle(                           */
/*    KHE_CLUSTER_BUSY_TIMES_MONITOR m, int index,                           */
/*    int busy_count, int idle_count, float workload)                        */
/*                                                                           */
/*  Add a busy value for one monitored time group.                           */
/*                                                                           */
/*****************************************************************************/

void KheClusterBusyTimesMonitorAddBusyAndIdle(
  KHE_CLUSTER_BUSY_TIMES_MONITOR m, int index,
  int busy_count, int idle_count, float workload)
{
  KHE_TIME_GROUP_STATE new_state;
  HnAssert(m->attached,
    "KheClusterBusyTimesMonitorAddBusyAndIdle internal error 1");
  new_state = KheRealAttachedPointNewState(m, index, busy_count,
    m->cutoff_index);
  KheRealPointChangeState(m, index, new_state);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheClusterBusyTimesMonitorDeleteBusyAndIdle(                        */
/*    KHE_CLUSTER_BUSY_TIMES_MONITOR m, int index,                           */
/*    int busy_count, int idle_count, float workload)                        */
/*                                                                           */
/*  Remove the busy value of one monitored time group.                       */
/*                                                                           */
/*****************************************************************************/

void KheClusterBusyTimesMonitorDeleteBusyAndIdle(
  KHE_CLUSTER_BUSY_TIMES_MONITOR m, int index,
  int busy_count, int idle_count, float workload)
{
  HnAssert(m->attached,
    "KheClusterBusyTimesMonitorDeleteBusyAndIdle internal error 1");
  KheRealPointChangeState(m, index, KHE_INACTIVE);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheClusterBusyTimesMonitorChangeBusyAndIdle(                        */
/*    KHE_CLUSTER_BUSY_TIMES_MONITOR m, int index,                           */
/*    int old_busy_count, int new_busy_count,                                */
/*    int old_idle_count, int new_idle_count,                                */
/*    float old_workload, float new_workload)                                */
/*                                                                           */
/*  Change the value of one monitored time group.                            */
/*                                                                           */
/*****************************************************************************/

void KheClusterBusyTimesMonitorChangeBusyAndIdle(
  KHE_CLUSTER_BUSY_TIMES_MONITOR m, int index,
  int old_busy_count, int new_busy_count,
  int old_idle_count, int new_idle_count,
  float old_workload, float new_workload)
{
  KHE_TIME_GROUP_STATE new_state;
  HnAssert(m->attached,
    "KheClusterBusyTimesMonitorChangeBusyAndIdle internal error 1");
  new_state = KheRealAttachedPointNewState(m, index, new_busy_count,
    m->cutoff_index);
  KheRealPointChangeState(m, index, new_state);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "deviations"                                                   */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  int KheClusterBusyTimesMonitorDeviation(KHE_CLUSTER_BUSY_TIMES_MONITOR m)*/
/*                                                                           */
/*  Return the deviation of m.                                               */
/*                                                                           */
/*****************************************************************************/

int KheClusterBusyTimesMonitorDeviation(KHE_CLUSTER_BUSY_TIMES_MONITOR m)
{
  return m->deviation;
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheActiveTimeGroupName(KHE_TIME_GROUP tg)                          */
/*                                                                           */
/*  Return an adjusted form of the name of tg.                               */
/*                                                                           */
/*****************************************************************************/

static char *KheActiveTimeGroupName(KHE_TIME_GROUP tg)
{
  if( KheTimeGroupTimeCount(tg) == 1 )
    return KheTimeName(KheTimeGroupTime(tg, 0));
  else if( KheTimeGroupId(tg) != NULL )
    return KheTimeGroupId(tg);
  else
    return "?";
}


/*****************************************************************************/
/*                                                                           */
/*  void KheAddActiveTimeGroups(KHE_CLUSTER_BUSY_TIMES_MONITOR m,            */
/*    HA_ARRAY_NCHAR *ac, char sign, bool *not_first)                        */
/*                                                                           */
/*  Add string representations of the active time groups of m to *ac.        */
/*                                                                           */
/*****************************************************************************/

static void KheAddActiveTimeGroups(KHE_CLUSTER_BUSY_TIMES_MONITOR m,
  HA_ARRAY_NCHAR *ac, char sign, bool *not_first)
{
  int i, busy;  char buff[10];  KHE_TIME_GROUP tg;  KHE_POLARITY po;
  for( i = 0;  i < KheClusterBusyTimesMonitorTimeGroupCount(m);  i++ )
  {
    if( KheClusterBusyTimesMonitorTimeGroupIsActive(m, i, &tg, &po, &busy) )
    {
      if( *not_first )
	sprintf(buff, "  %c  ", sign);
      else
	sprintf(buff, "%s", "");
      HnStringAdd(ac, "%s%s%s", buff, po == KHE_NEGATIVE ? "*" : "",
	KheActiveTimeGroupName(tg));
      *not_first = true;
    }
  }
}

/* ***
static void KheAddActiveTimeGroups(KHE_CLUSTER_BUSY_TIMES_MONITOR m,
  ARRAY_CHAR *ac, char sign, bool *not_first)
{
  int i, busy_count;  KHE_TIME_GROUP_INFO mi;  char buff[10]; KHE_TIME_GROUP tg;
  MArrayForEach(m->time_group_info, &mi, &i)
  {
    busy_count = KheMonitoredTimeGroupBusyCount(mi.monitored_time_group);
    if( KheClusterBusyTimesTimeGroupIsActive(mi.polarity, busy_count) )
    {
      tg = KheMonitoredTimeGroupTimeGroup(mi.monitored_time_group);
      if( *not_first )
	sprintf(buff, "  %c  ", sign);
      else
	sprintf(buff, "%s", "");
      MStringPrintf(*ac, 100, "%s%s%s", buff,
	mi.polarity == KHE_NEGATIVE ? "*" : "", KheActiveTimeGroupName(tg));
      *not_first = true;
    }
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  char *KheClusterBusyTimesMonitorDeviationDescription(                    */
/*    KHE_CLUSTER_BUSY_TIMES_MONITOR m)                                      */
/*                                                                           */
/*  Return a description of the deviation of m in heap memory.               */
/*                                                                           */
/*****************************************************************************/

char *KheClusterBusyTimesMonitorDeviationDescription(
  KHE_CLUSTER_BUSY_TIMES_MONITOR m)
{
  HA_ARRAY_NCHAR ac;  int total, history, history_after;
  bool not_first;  HA_ARENA a;  KHE_RESOURCE r;
  r = KheClusterBusyTimesMonitorResource(m);
  history_after = KheClusterBusyTimesConstraintHistoryAfter(m->constraint);
  history = KheClusterBusyTimesConstraintHistory(m->constraint, r);
  total = m->active_group_count + history;
  if( m->allow_zero && total == 0 )
    return "0";
  else
  {
    a = KheSolnArena(m->soln);
    HnStringBegin(ac, a);
    if( total + history_after < m->minimum )
    {
      /* m->minimum - history - m->active_group_count - history_after */
      HnStringAdd(&ac, "%d", m->minimum);
      if( history > 0 )
	HnStringAdd(&ac, "  -  History %d", history);
      not_first = true;
      KheAddActiveTimeGroups(m, &ac, '-', &not_first);
      if( history_after > 0 )
	HnStringAdd(&ac, "  -  History %d", history_after);
    }
    else if( total > m->maximum )
    {
      /* history + m->active_group_count - m->maximum */
      not_first = false;
      if( history > 0 )
      {
	HnStringAdd(&ac, "History %d", history);
	not_first = true;
      }
      KheAddActiveTimeGroups(m, &ac, '+', &not_first);
      HnStringAdd(&ac, "  -  %d", m->maximum);
    }
    else
      HnStringAdd(&ac, "0");
    if( m->history_adjustment > 0 )
      HnStringAdd(&ac, " (adjust %.5f)",KheCostShow(m->history_adjustment));
    if( m->multiplier != 1 )
      HnStringAdd(&ac, " (multiplier %d)", m->multiplier);
    return HnStringEnd(ac);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheClusterBusyTimesMonitorPointOfApplication(                      */
/*    KHE_CLUSTER_BUSY_TIMES_MONITOR m)                                      */
/*                                                                           */
/*  Return a description of the point of application of m.                   */
/*                                                                           */
/*****************************************************************************/

char *KheClusterBusyTimesMonitorPointOfApplication(
  KHE_CLUSTER_BUSY_TIMES_MONITOR m)
{
  char *name;  HA_ARENA a;
  name = KheResourceName(KheClusterBusyTimesMonitorResource(m));
  if( KheClusterBusyTimesConstraintAppliesToTimeGroup(m->constraint) != NULL )
  {
    a = KheSolnArena(m->soln);
    return HnStringMake(a, "%s offset %d", name, m->offset);
  }
  else
    return name;
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheClusterBusyTimesMonitorId(KHE_CLUSTER_BUSY_TIMES_MONITOR m)     */
/*                                                                           */
/*  Return the Id of m.                                                      */
/*                                                                           */
/*****************************************************************************/

char *KheClusterBusyTimesMonitorId(KHE_CLUSTER_BUSY_TIMES_MONITOR m)
{
  char *constraint_id, *resource_id;  HA_ARENA a;
  if( m->id == NULL )
  {
    constraint_id = KheConstraintId((KHE_CONSTRAINT) m->constraint);
    resource_id = KheResourceId(KheClusterBusyTimesMonitorResource(m));
    a = KheSolnArena(m->soln);
    if( m->offset == 0 )
      m->id = HnStringMake(a, "%s/%s", constraint_id, resource_id);
    else
      m->id = HnStringMake(a, "%s/%s/%d", constraint_id, resource_id,m->offset);
  }
  return m->id;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "multiplier"                                                   */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheClusterBusyTimesMonitorSetMultiplier(                            */
/*    KHE_CLUSTER_BUSY_TIMES_MONITOR m, int val)                             */
/*                                                                           */
/*  Set m's multiplier.                                                      */
/*                                                                           */
/*****************************************************************************/

void KheClusterBusyTimesMonitorSetMultiplier(
  KHE_CLUSTER_BUSY_TIMES_MONITOR m, int val)
{
  HnAssert(val >= 0, "KheClusterBusyTimesMonitorSetMultiplier: val < 0");
  if( DEBUG7 )
    fprintf(stderr, "  KheClusterBusyTimesMonitorSetMultiplier(%s, %d -> %d)\n",
      KheMonitorId((KHE_MONITOR) m), m->multiplier, val);
  m->multiplier = val;
  if( m->attached )
    Flush(m, true);
}


/*****************************************************************************/
/*                                                                           */
/* int KheClusterBusyTimesMonitorMultiplier(KHE_CLUSTER_BUSY_TIMES_MONITOR m)*/
/*                                                                           */
/*  Return m's multiplier.                                                   */
/*                                                                           */
/*****************************************************************************/

int KheClusterBusyTimesMonitorMultiplier(KHE_CLUSTER_BUSY_TIMES_MONITOR m)
{
  return m->multiplier;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "debug"                                                        */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheClusterBusyTimesMonitorDebug(KHE_CLUSTER_BUSY_TIMES_MONITOR m,   */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of m onto fp with the given verbosity and indent.            */
/*                                                                           */
/*****************************************************************************/

void KheClusterBusyTimesMonitorDebug(KHE_CLUSTER_BUSY_TIMES_MONITOR m,
  int verbosity, int indent, FILE *fp)
{
  int history, history_after;  KHE_RESOURCE r;
  r = KheClusterBusyTimesMonitorResource(m);
  history_after = KheClusterBusyTimesConstraintHistoryAfter(m->constraint);
  history = KheClusterBusyTimesConstraintHistory(m->constraint, r);
  if( verbosity >= 1 )
  {
    KheMonitorDebugBegin((KHE_MONITOR) m, indent, fp);
    fprintf(fp, " %s", "CBTM ");
    if( m->cutoff_index < m->time_group_info_count )
      fprintf(fp, "cutoff %d, ", m->cutoff_index);
    if( m->minimum > 0 )
      fprintf(fp, "min %d, ", m->minimum);
    if( m->maximum < m->time_group_info_count )
      fprintf(fp, "max %d, ", m->maximum);
    if( m->multiplier != 1 )
      fprintf(fp, "mult %d, ", m->multiplier);
    if( m->allow_zero )
      fprintf(fp, "allow_zero, ");
    if( history > 0 )
      fprintf(fp, "history %d, ", history);
    if( history_after > 0 )
      fprintf(fp, "history_after %d, ", history_after);
    fprintf(fp, "active_count %d", m->active_group_count);
    if( m->open_group_count > 0 )
      fprintf(fp, ", open_count %d", m->open_group_count);
    if( m->history_adjustment > 0 )
      fprintf(fp, ", adjust %.5f", KheCostShow(m->history_adjustment));
    KheMonitorDebugEnd((KHE_MONITOR) m, true, indent, fp);
  }
}
