
/*****************************************************************************/
/*                                                                           */
/*  THE KHE HIGH SCHOOL TIMETABLING ENGINE                                   */
/*  COPYRIGHT (C) 2010 Jeffrey H. Kingston                                   */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 3, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         khe_monitored_time_group.c                                 */
/*  DESCRIPTION:  A monitored time group                                     */
/*                                                                           */
/*****************************************************************************/
#include "khe_interns.h"

#define DEBUG1(m) (false && strcmp(KheMonitorId((KHE_MONITOR) m), "S4/t24")==0)

/*****************************************************************************/
/*                                                                           */
/*  KHE_MONITORED_TIME_GROUP - one time group monitored by a timetable.      */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  KHE_MTG_STATE_UNATTACHED,		/* not attached to timetable monitor */
  KHE_MTG_STATE_ATTACHED_BUSY,		/* attached, monitor busy only       */
  KHE_MTG_STATE_ATTACHED_BUSY_AND_IDLE	/* attached, monitor busy and idle   */
} KHE_MTG_STATE;

struct khe_monitored_time_group_rec {
  KHE_TIME_GROUP		time_group;		/* monitored tg      */
  KHE_RESOURCE_TIMETABLE_MONITOR timetable_monitor;	/* enclosing tm      */
  ARRAY_KHE_MONITOR		monitors;		/* attached monitors */
  HA_ARRAY_INT			monitor_indexes;	/* id in monitors    */
  KHE_MTG_STATE			state;			/* attach state      */
  int				busy_count;		/* busy times count  */
  float				workload;		/* workload in tg    */
  int				idle_count;		/* idle times count  */
  SSET				busy_set;		/* busy times lset   */
  KHE_MONITORED_TIME_GROUP	copy;			/* used when copying */
};


/*****************************************************************************/
/*                                                                           */
/*  Submodule "construction and query"                                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_MONITORED_TIME_GROUP KheMonitoredTimeGroupMake(KHE_TIME_GROUP tg,    */
/*    KHE_RESOURCE_TIMETABLE_MONITOR rtm)                                    */
/*                                                                           */
/*  Make a new monitored time group object for rtm, monitoring tg.           */
/*                                                                           */
/*****************************************************************************/

KHE_MONITORED_TIME_GROUP KheMonitoredTimeGroupMake(KHE_TIME_GROUP tg,
  KHE_RESOURCE_TIMETABLE_MONITOR rtm)
{
  KHE_MONITORED_TIME_GROUP res;  HA_ARENA a;
  a = KheSolnArena(KheMonitorSoln((KHE_MONITOR) rtm));
  HaMake(res, a);
  res->time_group = tg;
  res->timetable_monitor = rtm;
  HaArrayInit(res->monitors, a);
  HaArrayInit(res->monitor_indexes, a);
  res->state = KHE_MTG_STATE_UNATTACHED;
  res->busy_count = 0;
  res->workload = 0.0;
  res->idle_count = 0;
  SSetInit(res->busy_set, a);
  res->copy = NULL;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_MONITORED_TIME_GROUP KheMonitoredTimeGroupCopyPhase1(                */
/*    KHE_MONITORED_TIME_GROUP mtg, HA_ARENA a)                              */
/*                                                                           */
/*  Carry out Phase 1 of the copying of mtg.                                 */
/*                                                                           */
/*****************************************************************************/

KHE_MONITORED_TIME_GROUP KheMonitoredTimeGroupCopyPhase1(
  KHE_MONITORED_TIME_GROUP mtg, HA_ARENA a)
{
  KHE_MONITORED_TIME_GROUP copy;  int i, mi;  KHE_MONITOR m;
  if( mtg->copy == NULL )
  {
    HaMake(copy, a);
    mtg->copy = copy;
    copy->time_group = mtg->time_group;
    copy->timetable_monitor =
      KheResourceTimetableMonitorCopyPhase1(mtg->timetable_monitor, a);
    HaArrayInit(copy->monitors, a);
    HaArrayForEach(mtg->monitors, m, i)
      HaArrayAddLast(copy->monitors, KheMonitorCopyPhase1(m, a));
    HaArrayInit(copy->monitor_indexes, a);
    HaArrayForEach(mtg->monitor_indexes, mi, i)
      HaArrayAddLast(copy->monitor_indexes, mi);
    copy->state = mtg->state;
    copy->busy_count = mtg->busy_count;
    copy->workload = mtg->workload;
    copy->idle_count = mtg->idle_count;
    SSetCopy(copy->busy_set, mtg->busy_set, a);
    /* ***
    copy->busy_set = mtg->busy_set == NULL ? NULL : LSetCopy(mtg->busy_set);
    *** */
    copy->copy = NULL;
  }
  return mtg->copy;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheTimeGroupMonitorCopyPhase2(KHE_TIME_GROUP_MONITOR tgm)           */
/*                                                                           */
/*  Carry out Phase 2 of the copying of tgm.                                 */
/*                                                                           */
/*****************************************************************************/

void KheMonitoredTimeGroupCopyPhase2(KHE_MONITORED_TIME_GROUP mtg)
{
  int i;  KHE_MONITOR m;
  if( mtg->copy != NULL )
  {
    mtg->copy = NULL;
    KheResourceTimetableMonitorCopyPhase2(mtg->timetable_monitor);
    HaArrayForEach(mtg->monitors, m, i)
      KheMonitorCopyPhase2(m);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheMonitoredTimeGroupDelete(KHE_MONITORED_TIME_GROUP mtg)           */
/*                                                                           */
/*  Delete mtg.                                                              */
/*                                                                           */
/*****************************************************************************/

/* ***
void KheMonitoredTimeGroupDelete(KHE_MONITORED_TIME_GROUP mtg)
{
  MArrayFree(mtg->monitors);
  MArrayFree(mtg->monitor_indexes);
  if( mtg->busy_set != NULL )
    LSetFree(mtg->busy_set);
  MFree(mtg);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_TIME_GROUP KheMonitoredTimeGroupTimeGroup(                           */
/*    KHE_MONITORED_TIME_GROUP mtg)                                          */
/*                                                                           */
/*  Return the time group being monitored by mtg.                            */
/*                                                                           */
/*****************************************************************************/

KHE_TIME_GROUP KheMonitoredTimeGroupTimeGroup(KHE_MONITORED_TIME_GROUP mtg)
{
  return mtg->time_group;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_RESOURCE_TIMETABLE_MONITOR KheMonitoredTimeGroupTimetableMonitor(    */
/*    KHE_MONITORED_TIME_GROUP tgm)                                          */
/*                                                                           */
/*  Return the enclosing timetable monitor of tgm.                           */
/*                                                                           */
/*****************************************************************************/

KHE_RESOURCE_TIMETABLE_MONITOR KheMonitoredTimeGroupTimetableMonitor(
  KHE_MONITORED_TIME_GROUP tgm)
{
  return tgm->timetable_monitor;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "state changes (private)"                                      */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheMonitoredTimeGroupUnattachedToBusy(KHE_MONITORED_TIME_GROUP mtg) */
/*                                                                           */
/*  Change state from unattached to monitoring busy only.                    */
/*                                                                           */
/*****************************************************************************/

static void KheMonitoredTimeGroupUnattachedToBusy(KHE_MONITORED_TIME_GROUP mtg)
{
  HnAssert(mtg->state == KHE_MTG_STATE_UNATTACHED,
    "KheMonitoredTimeGroupUnattachedToBusy internal error 1");
  HnAssert(mtg->busy_count == 0,
    "KheMonitoredTimeGroupUnattachedToBusy internal error 2");
  KheResourceTimetableMonitorAttachMonitoredTimeGroup(
    mtg->timetable_monitor, mtg, &mtg->busy_count, &mtg->workload);
  mtg->state = KHE_MTG_STATE_ATTACHED_BUSY;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheMonitoredTimeGroupBusyToUnattached(KHE_MONITORED_TIME_GROUP mtg) */
/*                                                                           */
/*  Change state from monitoring busy only to unattached.                    */
/*                                                                           */
/*****************************************************************************/

static void KheMonitoredTimeGroupBusyToUnattached(KHE_MONITORED_TIME_GROUP mtg)
{
  HnAssert(mtg->state == KHE_MTG_STATE_ATTACHED_BUSY,
    "KheMonitoredTimeGroupBusyToUnattached internal error 1");
  KheResourceTimetableMonitorDetachMonitoredTimeGroup(
    mtg->timetable_monitor, mtg);
  mtg->busy_count = 0;
  mtg->workload = 0.0;
  mtg->state = KHE_MTG_STATE_UNATTACHED;
}


/*****************************************************************************/
/*                                                                           */
/*  int KheIdleTimes(KHE_MONITORED_TIME_GROUP mtg)                           */
/*                                                                           */
/*  Return the number of idle times, assuming we are monitoring idle times.  */
/*                                                                           */
/*****************************************************************************/

static int KheIdleTimes(KHE_MONITORED_TIME_GROUP mtg)
{
  if( mtg->busy_count > 1 )
    return SSetMax(mtg->busy_set) - SSetMin(mtg->busy_set) + 1 -
      mtg->busy_count;
  else
    return 0;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheMonitoredTimeGroupBusyToBusyAndIdle(KHE_MONITORED_TIME_GROUP mtg)*/
/*                                                                           */
/*  Change state from monitoring busy only to monitoring busy and idle.      */
/*                                                                           */
/*****************************************************************************/

static void KheMonitoredTimeGroupBusyToBusyAndIdle(KHE_MONITORED_TIME_GROUP mtg)
{
  int i;  KHE_TIME t;  KHE_RESOURCE_TIMETABLE_MONITOR tm;
  HnAssert(mtg->state == KHE_MTG_STATE_ATTACHED_BUSY,
    "KheMonitoredTimeGroupBusyToBusyAndIdle internal error");
  /* ***
  HnAssert(mtg->busy_set == NULL,
    "KheMonitoredTimeGroupBusyToBusyAndIdle internal error 2");
  mtg->busy_set = LSet New();
  *** */
  HnAssert(mtg->idle_count == 0,
    "KheMonitoredTimeGroupBusyToBusyAndIdle internal error 2");
  SSetClear(mtg->busy_set);
  tm = mtg->timetable_monitor;
  for( i = 0;  i < KheTimeGroupTimeCount(mtg->time_group);  i++ )
  {
    t = KheTimeGroupTime(mtg->time_group, i);
    if( KheResourceTimetableMonitorTimeTaskCount(tm, t) > 0 )
      SSetInsert(mtg->busy_set, i);  /* NB *position* in time group */
  }
  mtg->idle_count = KheIdleTimes(mtg);
  mtg->state = KHE_MTG_STATE_ATTACHED_BUSY_AND_IDLE;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheMonitoredTimeGroupBusyAndIdleToBusy(KHE_MONITORED_TIME_GROUP mtg)*/
/*                                                                           */
/*  Finish monitoring idle times in mtg.                                     */
/*                                                                           */
/*****************************************************************************/

static void KheMonitoredTimeGroupBusyAndIdleToBusy(KHE_MONITORED_TIME_GROUP mtg)
{
  HnAssert(mtg->state == KHE_MTG_STATE_ATTACHED_BUSY_AND_IDLE,
    "KheMonitoredTimeGroupBusyAndIdleToBusy internal error");
  SSetClear(mtg->busy_set);
  /* ***
  HnAssert(mtg->busy_set != NULL,
    "KheMonitoredTimeGroupBusyAndIdleToBusy internal error");
  LSetFree(mtg->busy_set);
  *** */
  mtg->idle_count = 0;
  /* mtg->busy_set = NULL; */
  mtg->state = KHE_MTG_STATE_ATTACHED_BUSY;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "attach and detach"                                            */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheMonitoredTimeGroupAttachMonitor(KHE_MONITORED_TIME_GROUP mtg,    */
/*    KHE_MONITOR m, int index)                                              */
/*                                                                           */
/*  Attach m to mtg.  This means that m is becoming an attached monitor,     */
/*  and that mtg must be attached to tm and tm must be attached.             */
/*                                                                           */
/*****************************************************************************/

void KheMonitoredTimeGroupAttachMonitor(KHE_MONITORED_TIME_GROUP mtg,
  KHE_MONITOR m, int index)
{
  /* make sure mtg is monitoring busy times at least */
  if( mtg->state == KHE_MTG_STATE_UNATTACHED )
    KheMonitoredTimeGroupUnattachedToBusy(mtg);

  /* make sure mtg is monitoring idle times, if required */
  if( KheMonitorTag(m) == KHE_LIMIT_IDLE_TIMES_MONITOR_TAG && 
      mtg->state != KHE_MTG_STATE_ATTACHED_BUSY_AND_IDLE )
    KheMonitoredTimeGroupBusyToBusyAndIdle(mtg);

  /* add m and its index to mtg */
  HaArrayAddLast(mtg->monitors, m);
  HaArrayAddLast(mtg->monitor_indexes, index);

  /* report busy and idle back to m */
  KheMonitorAddBusyAndIdle(m, index, mtg->busy_count, mtg->idle_count,
    mtg->workload);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheMonitoredTimeGroupHasLimitIdleTimesMonitor(                      */
/*    KHE_MONITORED_TIME_GROUP mtg)                                          */
/*                                                                           */
/*  Return true when at least one of the monitors of mtg is a limit idle     */
/*  times monitor.                                                           */
/*                                                                           */
/*****************************************************************************/

static bool KheMonitoredTimeGroupHasLimitIdleTimesMonitor(
  KHE_MONITORED_TIME_GROUP mtg)
{
  KHE_MONITOR m;  int i;
  HaArrayForEach(mtg->monitors, m, i)
    if( KheMonitorTag(m) == KHE_LIMIT_IDLE_TIMES_MONITOR_TAG )
      return true;
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheMonitoredTimeGroupDetachMonitor(KHE_MONITORED_TIME_GROUP mtg,    */
/*    KHE_MONITOR m, int index)                                              */
/*                                                                           */
/*  Detach m from mtg.  This means that m is becoming a unattached monitor.  */
/*                                                                           */
/*  NB mtg must be attached, because it is attached whenever at least one    */
/*  monitor is attached to it, and m is such a monitor.                      */
/*                                                                           */
/*****************************************************************************/

void KheMonitoredTimeGroupDetachMonitor(KHE_MONITORED_TIME_GROUP mtg,
  KHE_MONITOR m, int index)
{
  int pos;

  /* delete busy and idle from m */
  HnAssert(mtg->state != KHE_MTG_STATE_UNATTACHED,
    "KheMonitoredTimeGroupDetachMonitor: internal error");
  KheMonitorDeleteBusyAndIdle(m, index, mtg->busy_count, mtg->idle_count,
    mtg->workload);

  /* remove m and its index from mtg */
  if( !HaArrayContains(mtg->monitors, m, &pos) )
    HnAbort("KheMonitoredTimeGroupDetachMonitor: m not present");
  HaArrayDeleteAndShift(mtg->monitors, pos);
  HaArrayDeleteAndShift(mtg->monitor_indexes, pos);

  /* make sure mtg is no longer monitoring idle times, if not required */
  if( mtg->state == KHE_MTG_STATE_ATTACHED_BUSY_AND_IDLE &&
      !KheMonitoredTimeGroupHasLimitIdleTimesMonitor(mtg) )
    KheMonitoredTimeGroupBusyAndIdleToBusy(mtg);

  /* make sure mtg is no longer attached, if not requred */
  if( HaArrayCount(mtg->monitors) == 0 )
    KheMonitoredTimeGroupBusyToUnattached(mtg);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "monitoring"                                                   */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheMonitoredTimeGroupFlush(KHE_MONITORED_TIME_GROUP mtg,            */
/*    int old_busy_count, int old_idle_count, float old_workload)            */
/*                                                                           */
/*  Report a change of state in mtg to its attached monitors.                */
/*                                                                           */
/*****************************************************************************/

static void KheMonitoredTimeGroupFlush(KHE_MONITORED_TIME_GROUP mtg,
  int old_busy_count, int old_idle_count, float old_workload)
{
  int i;  KHE_MONITOR m;
  HaArrayForEach(mtg->monitors, m, i)
  {
    if( DEBUG1(m) )
      fprintf(stderr, "KheMonitoredTimeGroupFlush(%s, %s, idle %d)\n",
	KheTimeGroupId(mtg->time_group), SSetShow(mtg->busy_set),
	mtg->idle_count);
    KheMonitorChangeBusyAndIdle(m, HaArray(mtg->monitor_indexes, i),
      old_busy_count, mtg->busy_count, old_idle_count, mtg->idle_count,
      old_workload, mtg->workload);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheMonitoredTimeGroupAssignNonClash(KHE_MONITORED_TIME_GROUP mtg,   */
/*    int assigned_time_index)                                               */
/*                                                                           */
/*  Inform mtg that in the enclosing timetable, the time with this index     */
/*  number (which lies in mtg's time group) has changed from being           */
/*  unoccupied to being occupied.                                            */
/*                                                                           */
/*****************************************************************************/

/* *** replaced by KheMonitoredTimeGroupAssign below
void KheMonitoredTimeGroupAssignNonClash(KHE_MONITORED_TIME_GROUP mtg,
  int assigned_time_index)
{
  int old_busy_count, old_idle_count;
  old_busy_count = mtg->busy_count;
  old_idle_count = mtg->idle_count;
  mtg->busy_count++;
  if( mtg->state == KHE_MTG_STATE_ATTACHED_BUSY_AND_IDLE )
  {
    SSetInsert(mtg->busy_set,
      KheTimeGroupTimePos(mtg->time_group, assigned_time_index));
    mtg->idle_count = KheIdleTimes(mtg);
  }
  KheMonitoredTimeGroupFlush(mtg, old_busy_count, old_idle_count);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheMonitoredTimeGroupUnAssignNonClash(KHE_MONITORED_TIME_GROUP mtg, */
/*    int assigned_time_index)                                               */
/*                                                                           */
/*  Inform mtg that in the enclosing timetable, the time with this index     */
/*  number (which lies in mtg's time group) has changed from being           */
/*  occupied to being unoccupied.                                            */
/*                                                                           */
/*****************************************************************************/

/* *** replaced by KheMonitoredTimeGroupUnAssign below
void KheMonitoredTimeGroupUnAssignNonClash(KHE_MONITORED_TIME_GROUP mtg,
  int assigned_time_index)
{
  int old_busy_count, old_idle_count;
  HnAssert(mtg->busy_count > 0,
    "KheMonitoredTimeGroupUnAssignNonClash internal error");
  old_busy_count = mtg->busy_count;
  old_idle_count = mtg->idle_count;
  mtg->busy_count--;
  if( mtg->state == KHE_MTG_STATE_ATTACHED_BUSY_AND_IDLE )
  {
    LSetDelete(mtg->busy_set,
      KheTimeGroupTimePos(mtg->time_group, assigned_time_index));
    mtg->idle_count = KheIdleTimes(mtg);
  }
  KheMonitoredTimeGroupFlush(mtg, old_busy_count, old_idle_count);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheMonitoredTimeGroupAssign(KHE_MONITORED_TIME_GROUP mtg,           */
/*    KHE_TASK task, int assigned_time_index, int busy_count_before)         */
/*                                                                           */
/*  Inform mtg that in the enclosing timetable, the time with this index     */
/*  number (which lies in mtg's time group) has been assigned task.  Here    */
/*  busy_count_before is the number of tasks that were running at this       */
/*  time before task was assigned.                                           */
/*                                                                           */
/*  Implementation note.  For attached monitors other than limit workload    */
/*  monitors, this change is only of interest when the time goes from        */
/*  unoccupied to occupied (that is, when busy_count_before == 0).           */
/*                                                                           */
/*****************************************************************************/

void KheMonitoredTimeGroupAssign(KHE_MONITORED_TIME_GROUP mtg,
  KHE_TASK task, int assigned_time_index, int busy_count_before)
{
  int old_busy_count, old_idle_count;  float old_workload;
  old_busy_count = mtg->busy_count;
  old_idle_count = mtg->idle_count;
  old_workload = mtg->workload;
  mtg->workload += KheTaskWorkloadPerTime(task);
  if( busy_count_before == 0 )
  {
    mtg->busy_count++;
    if( mtg->state == KHE_MTG_STATE_ATTACHED_BUSY_AND_IDLE )
    {
      SSetInsert(mtg->busy_set,
	KheTimeGroupTimePos(mtg->time_group, assigned_time_index));
      mtg->idle_count = KheIdleTimes(mtg);
    }
  }
  KheMonitoredTimeGroupFlush(mtg, old_busy_count, old_idle_count,
    old_workload);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheMonitoredTimeGroupUnAssign(KHE_MONITORED_TIME_GROUP mtg,         */
/*    KHE_TASK task, int assigned_time_index, int busy_count_after)          */
/*                                                                           */
/*  Inform mtg that in the enclosing timetable, the time with this index     */
/*  number (which lies in mtg's time group) has been unassigned task.  Here  */
/*  busy_count_after is the number of tasks that are running at this         */
/*  time after task is unassigned.                                           */
/*                                                                           */
/*  Implementation note.  For attached monitors other than limit workload    */
/*  monitors, this change is only of interest when the time goes from        */
/*  occupied to unoccupied (that is, when busy_count_after == 0).            */
/*                                                                           */
/*****************************************************************************/

void KheMonitoredTimeGroupUnAssign(KHE_MONITORED_TIME_GROUP mtg,
  KHE_TASK task, int assigned_time_index, int busy_count_after)
{
  int old_busy_count, old_idle_count;  float old_workload;
  old_busy_count = mtg->busy_count;
  old_idle_count = mtg->idle_count;
  old_workload = mtg->workload;
  mtg->workload -= KheTaskWorkloadPerTime(task);
  if( busy_count_after == 0 )
  {
    HnAssert(mtg->busy_count > 0,
      "KheMonitoredTimeGroupUnAssignNonClash internal error");
    mtg->busy_count--;
    if( mtg->state == KHE_MTG_STATE_ATTACHED_BUSY_AND_IDLE )
    {
      SSetDelete(mtg->busy_set,
	KheTimeGroupTimePos(mtg->time_group, assigned_time_index));
      mtg->idle_count = KheIdleTimes(mtg);
    }
  }
  KheMonitoredTimeGroupFlush(mtg, old_busy_count, old_idle_count,
    old_workload);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheMonitoredTimeGroupBusyCount(KHE_MONITORED_TIME_GROUP mtg)         */
/*                                                                           */
/*  Return the current number of busy times.                                 */
/*                                                                           */
/*****************************************************************************/

int KheMonitoredTimeGroupBusyCount(KHE_MONITORED_TIME_GROUP mtg)
{
  return mtg->busy_count;
}


/*****************************************************************************/
/*                                                                           */
/*  float KheMonitoredTimeGroupWorkload(KHE_MONITORED_TIME_GROUP mtg)        */
/*                                                                           */
/*  Return the current workload of mtg.                                      */
/*                                                                           */
/*****************************************************************************/

float KheMonitoredTimeGroupWorkload(KHE_MONITORED_TIME_GROUP mtg)
{
  return mtg->workload;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheMonitoredTimeGroupIdleState(KHE_MONITORED_TIME_GROUP mtg,        */
/*    int *busy_count, int *idle_count,                                      */
/*    KHE_TIME extreme_busy_times[2], int *extreme_busy_times_count)         */
/*                                                                           */
/*  Report the state of mtg when it is monitoring busy and idle:  the        */
/*  number of busy times and idle times, and the first and last busy times.  */
/*  This is only called when monitoring busy and idle.                       */
/*                                                                           */
/*****************************************************************************/

void KheMonitoredTimeGroupIdleState(KHE_MONITORED_TIME_GROUP mtg,
  int *busy_count, int *idle_count,
  KHE_TIME extreme_busy_times[2], int *extreme_busy_times_count)
{
  KHE_TIME first, last;
  HnAssert( mtg->state == KHE_MTG_STATE_ATTACHED_BUSY_AND_IDLE,
    "KheMonitoredTimeGroupIdleState: mtg not monitoring idle");
  *busy_count = mtg->busy_count;
  *idle_count = mtg->idle_count;
  *extreme_busy_times_count = 0;
  if( mtg->busy_count > 0 )
  {
    first = KheTimeGroupTime(mtg->time_group, SSetMin(mtg->busy_set));
    last  = KheTimeGroupTime(mtg->time_group, SSetMax(mtg->busy_set));
    extreme_busy_times[(*extreme_busy_times_count)++] = first;
    if( last != first )
      extreme_busy_times[(*extreme_busy_times_count)++] = last;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "at max limit count"                                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  int KheMonitoredTimeGroupAtMaxLimitCount(KHE_MONITORED_TIME_GROUP mtg)   */
/*                                                                           */
/*  Return the total at max limit count of the monitors of mtg.              */
/*                                                                           */
/*****************************************************************************/

int KheMonitoredTimeGroupAtMaxLimitCount(KHE_MONITORED_TIME_GROUP mtg)
{
  int res, i;  KHE_MONITOR m;
  res = 0;
  HaArrayForEach(mtg->monitors, m, i)
    switch( KheMonitorTag(m) )
    {
      case KHE_CLUSTER_BUSY_TIMES_MONITOR_TAG:

	res += KheClusterBusyTimesMonitorAtMaxLimitCount(
	  (KHE_CLUSTER_BUSY_TIMES_MONITOR) m);
	break;

      case KHE_LIMIT_ACTIVE_INTERVALS_MONITOR_TAG:

	res += KheLimitActiveIntervalsMonitorAtMaxLimitCount(
	  (KHE_LIMIT_ACTIVE_INTERVALS_MONITOR) m);
	break;

      default:

	break;
    }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "range"                                                        */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool InRange(KHE_TIME ft, KHE_TIME lt, int first_time_index,             */
/*    int last_time_index)                                                   */
/*                                                                           */
/*  Return true if [ft .. lt] lies in [first_time_index .. last_time_index]. */
/*                                                                           */
/*****************************************************************************/

static bool InRange(KHE_TIME ft, KHE_TIME lt, int first_time_index,
  int last_time_index)
{
  return first_time_index <= KheTimeIndex(ft) &&
    KheTimeIndex(lt) <= last_time_index;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheMonitoredTimeGroupAddRange(KHE_MONITORED_TIME_GROUP mtg,         */
/*    KHE_RESOURCE_TYPE rt, int first_time_index, int last_time_index,       */
/*    KHE_GROUP_MONITOR gm)                                                  */
/*                                                                           */
/*  Add as children to gm all limit busy times and cluster busy times        */
/*  monitors that are derived from constraints that monitor all resources    */
/*  of type rt, and monitor mtg at times limited to between first_time_index */
/*  and last_time_index.  Do not add any monitors that are already there.    */
/*                                                                           */
/*****************************************************************************/

void KheMonitoredTimeGroupAddRange(KHE_MONITORED_TIME_GROUP mtg,
  KHE_RESOURCE_TYPE rt, int first_time_index, int last_time_index,
  KHE_GROUP_MONITOR gm)
{
  int i, count;  KHE_MONITOR m;  KHE_TIME ft, lt;
  KHE_CLUSTER_BUSY_TIMES_CONSTRAINT cbtc;  KHE_CLUSTER_BUSY_TIMES_MONITOR cbtm;
  KHE_LIMIT_BUSY_TIMES_CONSTRAINT lbtc;  KHE_LIMIT_BUSY_TIMES_MONITOR lbtm;
  count = KheResourceTypeResourceCount(rt);
  HaArrayForEach(mtg->monitors, m, i)
    switch( KheMonitorTag(m) )
    {
      case KHE_CLUSTER_BUSY_TIMES_MONITOR_TAG:

	cbtm = (KHE_CLUSTER_BUSY_TIMES_MONITOR) m;
	cbtc = KheClusterBusyTimesMonitorConstraint(cbtm);
	if( KheClusterBusyTimesConstraintResourceOfTypeCount(cbtc,rt) >= count )
	{
	  KheClusterBusyTimesMonitorRange(cbtm, &ft, &lt);
	  if( InRange(ft, lt, first_time_index, last_time_index) &&
	      !KheGroupMonitorHasChildMonitor(gm, m) )
	    KheGroupMonitorAddChildMonitor(gm, m);
	}
	break;

      case KHE_LIMIT_BUSY_TIMES_MONITOR_TAG:

	lbtm = (KHE_LIMIT_BUSY_TIMES_MONITOR) m;
	lbtc = KheLimitBusyTimesMonitorConstraint(lbtm);
	if( KheLimitBusyTimesConstraintResourceOfTypeCount(lbtc, rt) >= count )
	{
	  KheLimitBusyTimesMonitorRange(lbtm, &ft, &lt);
	  if( InRange(ft, lt, first_time_index, last_time_index) &&
	      !KheGroupMonitorHasChildMonitor(gm, m) )
	    KheGroupMonitorAddChildMonitor(gm, m);
	}
	break;

      default:

	break;
    }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "debug"                                                        */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheMonitoredTimeGroupDebug(KHE_MONITORED_TIME_GROUP mtg,            */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of mtg onto fp with the given verbosity and indent.          */
/*                                                                           */
/*****************************************************************************/

void KheMonitoredTimeGroupDebug(KHE_MONITORED_TIME_GROUP mtg,
  int verbosity, int indent, FILE *fp)
{
  if( verbosity >= 1 )
  {
    fprintf(fp, "%*s[ ", indent, "");
    KheTimeGroupDebug(mtg->time_group, 1, -1, fp);
    fprintf(fp, " (%d busy", mtg->busy_count);
    if( mtg->state == KHE_MTG_STATE_ATTACHED_BUSY_AND_IDLE )
      fprintf(fp, ", %d idle", mtg->idle_count);
    fprintf(fp, ") ]");
  }
}
