
/*****************************************************************************/
/*                                                                           */
/*  THE KHE HIGH SCHOOL TIMETABLING ENGINE                                   */
/*  COPYRIGHT (C) 2010 Jeffrey H. Kingston                                   */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 3, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         khe_sm_timer.c                                             */
/*  DESCRIPTION:  Timers and timer sets                                      */
/*                                                                           */
/*****************************************************************************/
#include "khe_solvers.h"
#include "howard_a.h"
#include "howard_n.h"
#include <limits.h>
#if KHE_USE_TIMING
#include <time.h>
#include <sys/time.h>
#endif

#define DEBUG1	0
#define DEBUG2	0


/*****************************************************************************/
/*                                                                           */
/*  KHE_TIMER_LOG                                                            */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_timer_log_rec {
  char				*tag;		/* identifying tag           */
  HA_ARRAY_INT			log;		/* log of what happened      */
} *KHE_TIMER_LOG;

typedef HA_ARRAY(KHE_TIMER_LOG) ARRAY_KHE_TIMER_LOG;


/*****************************************************************************/
/*                                                                           */
/*  KHE_TIMER                                                                */
/*                                                                           */
/*****************************************************************************/

struct khe_timer_rec {
  char				*tag;		/* identifying tag           */
#if KHE_USE_TIMING
  struct timeval		start_time;	/* start time                */
#endif
  float				time_limit;	/* time limit                */
  /* KHE_TIMER_LOG		timer_log; */	/* the log                   */
};

typedef HA_ARRAY(KHE_TIMER) ARRAY_KHE_TIMER;


/*****************************************************************************/
/*                                                                           */
/*  KHE_TIMER_SET                                                            */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  TIMER_SET_LOG_NONE,
  TIMER_SET_LOG_READ,
  TIMER_SET_LOG_WRITE
} TIMER_SET_LOG_STATE;

struct khe_timer_set_rec {
  TIMER_SET_LOG_STATE		log_state;	/* timer set log state       */
  ARRAY_KHE_TIMER		timers;		/* the timers                */
  /* ARRAY_KHE_TIMER_LOG	timer_logs; */	/* the timer logs            */
};


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KheDateToday"                                                 */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  char *KheDateToday(void)                                                 */
/*                                                                           */
/*  Return today's date as a string in static memory.                        */
/*                                                                           */
/*****************************************************************************/

static char *KheMonthName(int tm_mon)
{
  char *months[] = { "January", "February", "March", "April", "May", "June",
    "July", "August", "September", "October", "November", "December" };
  HnAssert(tm_mon >= 0 && tm_mon <= 11, "KheMonthName internal error");
  return months[tm_mon];
}

char *KheDateToday(void)
{
#if KHE_USE_TIMING
  static char buff[100];
  time_t t;
  struct tm *time_now;
  time(&t);
  time_now = localtime(&t);
  snprintf(buff, 100, "%d %s %d", time_now->tm_mday,
    KheMonthName(time_now->tm_mon), 1900 + time_now->tm_year);
  return buff;
#else
  return "(no date)";
#endif
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "conversion"                                                   */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  float KheTimeFromString(char *str)                                       */
/*                                                                           */
/*  Convert *str into a number of seconds.  The format is secs, or           */
/*  mins:secs, or hrs:mins:secs.  Alternatively, "-" means KHE_NO_TIME.      */
/*                                                                           */
/*****************************************************************************/

float KheTimeFromString(char *str)
{
  char *p;  float hrs, mins, secs;

  /* "-" special case */
  if( strcmp(str, "-") == 0 )
    return KHE_NO_TIME;

  /* secs */
  p = strstr(str, ":");
  if( p == NULL )
  {
    if( sscanf(str, "%f", &secs) != 1 )
      HnAbort("KheTimeFromString:  format error in time limit %s", str);
    return secs;
  }

  /* mins:secs */
  p = strstr(p + 1, ":");
  if( p == NULL )
  {
    if( sscanf(str, "%f:%f", &mins, &secs) != 2 )
      HnAbort("KheTimeFromString:  format error in time limit %s", str);
    return mins * 60.0 + secs;
  }

  /* hrs:mins:secs */
  if( sscanf(str, "%f:%f:%f", &hrs, &mins, &secs) != 3 )
    HnAbort("KheTimeFromString:  format error in time limit %s", str);
  return hrs * 60.0 * 60.0 + mins * 60.0 + secs;
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheTimeShow(float secs, char buff[20])                             */
/*                                                                           */
/*  Show secs in the format "XX mins" or "XX secs".                          */
/*                                                                           */
/*****************************************************************************/

char *KheTimeShow(float secs, char buff[20])
{
  if( secs > 300.0 )
    snprintf(buff, 20, "%.1f mins", secs / 60.0);
  else
    snprintf(buff, 20, "%.1f secs", secs);
  return buff;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "timer logs"                                                   */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_TIMER_LOG KheTimerLogMake(char *tag, HA_ARENA a)                     */
/*                                                                           */
/*  Make a new, empty timer log with these attributes.                       */
/*                                                                           */
/*****************************************************************************/

/* ***
static KHE_TIMER_LOG KheTimerLogMake(char *tag, HA_ARENA a)
{
  KHE_TIMER_LOG res;
  HaMake(res, a);
  res->tag = tag;
  HaArrayInit(res->log, a);
  HaArrayAddLast(res->log, 0);
  return res;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheTimerLogUpdate(KHE_TIMER_LOG timer_log, bool reached)            */
/*                                                                           */
/*  Log the occurrence of a call to KheTimerTimeLimitReached whose result    */
/*  is reached.                                                              */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheTimerLogUpdate(KHE_TIMER_LOG timer_log, bool reached)
{
  if( reached )
    HaArrayAddLast(timer_log->log, 0);
  else
    HaArrayLast(timer_log->log)++;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheTimerLogSave(KHE_TIMER_LOG timer_log, FILE *fp)                  */
/*                                                                           */
/*  Save timer_log on fp.                                                    */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheTimerLogSave(KHE_TIMER_LOG timer_log, FILE *fp)
{
  int val, i;
  fprintf(fp, "  %s {", timer_log->tag);
  HaArrayForEach(timer_log->log, val, i)
    fprintf(fp, "%s%d", i % 8 == 0 ? "\n    " : " ", val);
  fprintf(fp, "\n  }\n");
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  Submodule "timers"                                                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_TIMER KheTimerMake(char *tag, float limit_in_seconds, HA_ARENA a)    */
/*                                                                           */
/*  Make and return a new timer in arena a with these attributes.            */
/*                                                                           */
/*****************************************************************************/

KHE_TIMER KheTimerMake(char *tag, float limit_in_seconds, HA_ARENA a)
{
  KHE_TIMER res;
  HaMake(res, a);
  res->tag = tag;
  KheTimerResetStartTime(res);
  res->time_limit = limit_in_seconds;
  /* res->timer_log = NULL; */
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_TIMER KheTimerCopy(KHE_TIMER timer, HA_ARENA a)                      */
/*                                                                           */
/*  Return a copy of timer in a.                                             */
/*                                                                           */
/*****************************************************************************/

KHE_TIMER KheTimerCopy(KHE_TIMER timer, HA_ARENA a)
{
  KHE_TIMER res;
  HaMake(res, a);
  res->tag = timer->tag;
#if KHE_USE_TIMING
  res->start_time = timer->start_time;
#endif
  res->time_limit = timer->time_limit;
  /* res->timer_log = NULL; */
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheTimerTag(KHE_TIMER timer)                                       */
/*                                                                           */
/*  Return the tag attribute of timer.                                       */
/*                                                                           */
/*****************************************************************************/

char *KheTimerTag(KHE_TIMER timer)
{
  return timer->tag;
}


/*****************************************************************************/
/*                                                                           */
/*  float KheTimerTimeLimit(KHE_TIMER timer)                                 */
/*                                                                           */
/*  Return the time limit attribute of timer.                                */
/*                                                                           */
/*****************************************************************************/

float KheTimerTimeLimit(KHE_TIMER timer)
{
  return timer->time_limit;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheTimerResetStartTime(KHE_TIMER timer)                             */
/*                                                                           */
/*  Reset the start time of timer to the current time.                       */
/*                                                                           */
/*****************************************************************************/

void KheTimerResetStartTime(KHE_TIMER timer)
{
#if KHE_USE_TIMING
  gettimeofday(&timer->start_time, NULL);
#endif
}


/*****************************************************************************/
/*                                                                           */
/*  void KheTimerResetTimeLimit(KHE_TIMER timer, float limit_in_seconds)     */
/*                                                                           */
/*  Reset the time limit of timer.                                           */
/*                                                                           */
/*****************************************************************************/

void KheTimerResetTimeLimit(KHE_TIMER timer, float limit_in_seconds)
{
  timer->time_limit = limit_in_seconds;
}


/*****************************************************************************/
/*                                                                           */
/*  float KheTimeDiff(struct timeval *earlier, struct timeval *later)        */
/*                                                                           */
/*  Return the time difference between earlier and later.                    */
/*                                                                           */
/*****************************************************************************/

#if KHE_USE_TIMING
static float KheTimeDiff(struct timeval *earlier, struct timeval *later)
{
  return (float) (later->tv_sec - earlier->tv_sec) +
    (float) (later->tv_usec - earlier->tv_usec) / 1000000.0;
}
#endif


/*****************************************************************************/
/*                                                                           */
/*  float KheTimerElapsedTime(KHE_TIMER timer)                               */
/*                                                                           */
/*  Return the elapsed time of timer.                                        */
/*                                                                           */
/*****************************************************************************/

float KheTimerElapsedTime(KHE_TIMER timer)
{
#if KHE_USE_TIMING
  struct timeval curr_time;
  gettimeofday(&curr_time, NULL);
  return KheTimeDiff(&timer->start_time, &curr_time);
#else
  return KHE_NO_TIME;
#endif
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheTimerReached(KHE_TIMER timer, bool *curr_time_known,             */
/*    struct timeval *curr_time)                                             */
/*                                                                           */
/*  Return true if timer's time limit has been reached; this will always     */
/*  be false if it has no time limit.                                        */
/*                                                                           */
/*  Parameter *curr_time_known is true when *curr_time contains the          */
/*  current time.  It is changed to true by this function if it finds        */
/*  the current time.                                                        */
/*                                                                           */
/*****************************************************************************/

static bool KheTimerReached(KHE_TIMER timer, bool *curr_time_known,
  struct timeval *curr_time)
{
  bool reached;
  if( timer->time_limit != KHE_NO_TIME )
  {
    /* get current time, if not done yet */
    if( !*curr_time_known )
    {
      gettimeofday(curr_time, NULL);
      *curr_time_known = true;
    }

    /* work out whether this timer's limit has been reached */
    reached = KheTimeDiff(&timer->start_time, curr_time) >= timer->time_limit;
    if( DEBUG2 )
      KheTimerDebug(timer, 2, 2, stderr);

    /* update the log and return */
    /* ***
    if( timer->timer_log != NULL )
      KheTimerLogUpdate(timer->timer_log, reached);
    *** */
    return reached;
  }
  else
    return false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheTimerTimeLimitReached(KHE_TIMER timer)                           */
/*                                                                           */
/*  Return true if timer has a time limit and it has been reached.           */
/*                                                                           */
/*****************************************************************************/

bool KheTimerTimeLimitReached(KHE_TIMER timer)
{
  struct timeval curr_time;  bool curr_time_known;
  curr_time_known = false;
  return KheTimerReached(timer, &curr_time_known, &curr_time);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheTimerDebug(KHE_TIMER timer, int verbosity, int indent, FILE *fp) */
/*                                                                           */
/*  Debug print of timer onto fp with the given verbosity and indent.        */
/*                                                                           */
/*****************************************************************************/

void KheTimerDebug(KHE_TIMER timer, int verbosity, int indent, FILE *fp)
{
  if( indent >= 0 )
    fprintf(fp, "%*s", indent, "");
  fprintf(fp, "%s:%.1f", timer->tag, KheTimerElapsedTime(timer));
  if( timer->time_limit != KHE_NO_TIME )
    fprintf(fp, ":%.1f", timer->time_limit);
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "timer sets"                                                   */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_TIMER_SET KheTimerSetMake(bool log, HA_ARENA a)                      */
/*                                                                           */
/*  Make a new, empty timer set.                                             */
/*                                                                           */
/*****************************************************************************/

KHE_TIMER_SET KheTimerSetMake(HA_ARENA a)
{
  KHE_TIMER_SET res;
  HaMake(res, a);
  res->log_state = TIMER_SET_LOG_NONE;
  HaArrayInit(res->timers, a);
  /* HaArrayInit(res->timer_logs, a); */
  if( DEBUG1 )
  {
    fprintf(stderr, "timer set at start:\n");
    KheTimerSetDebug(res, 2, 2, stderr);
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_TIMER_SET KheTimerSetCopy(KHE_TIMER_SET timer_set, HA_ARENA a)       */
/*                                                                           */
/*  Make a copy of timer_set in arena a.                                     */
/*                                                                           */
/*****************************************************************************/

KHE_TIMER_SET KheTimerSetCopy(KHE_TIMER_SET timer_set, HA_ARENA a)
{
  KHE_TIMER_SET res;  KHE_TIMER timer;  int i;
  HaMake(res, a);
  HaArrayInit(res->timers, a);
  HaArrayForEach(timer_set->timers, timer, i)
    HaArrayAddLast(timer_set->timers, KheTimerCopy(timer, a));
  /* HaArrayInit(res->timer_logs, a); */
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheTimerSetAddTimer(KHE_TIMER_SET timer_set, KHE_TIMER timer)       */
/*                                                                           */
/*  Add timer to timer_set.                                                  */
/*                                                                           */
/*****************************************************************************/
/* static KHE_TIMER_LOG KheTimerSetGetTimerLog(KHE_TIMER_SET ts, char *tag); */

void KheTimerSetAddTimer(KHE_TIMER_SET timer_set, KHE_TIMER timer)
{
  int pos;
  if( DEBUG1 )
  {
    fprintf(stderr, "[ KheTimerSetAddTimer(timer_set, timer):\n");
    KheTimerSetDebug(timer_set, 2, 2, stderr);
  }
  HnAssert(!HaArrayContains(timer_set->timers, timer, &pos),
    "KheTimerSetAddTimer: timer_set already contains timer");
  HaArrayAddLast(timer_set->timers, timer);
  /* timer->timer_log = KheTimerSetGetTimerLog(timer_set, timer->tag); */
  if( DEBUG1 )
  {
    KheTimerSetDebug(timer_set, 2, 2, stderr);
    fprintf(stderr, "] KheTimerSetAddTimer returning\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheTimerSetDeleteTimer(KHE_TIMER_SET timer_set, KHE_TIMER timer)    */
/*                                                                           */
/*  Delete timer from timer_set.                                             */
/*                                                                           */
/*****************************************************************************/

void KheTimerSetDeleteTimer(KHE_TIMER_SET timer_set, KHE_TIMER timer)
{
  int pos;
  if( !HaArrayContains(timer_set->timers, timer, &pos) )
    HnAbort("KheTimerSetDeleteTimer: time_set does not contain timer \"%s\"",
      timer->tag);
  HaArrayDeleteAndShift(timer_set->timers, pos);
  /* timer->timer_log = NULL; */
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheTimerSetContainsTimer(KHE_TIMER_SET timer_set, char *tag,        */
/*    KHE_TIMER *timer)                                                      */
/*                                                                           */
/*  If timer_set contains a timer with the given tag, set *timer to          */
/*  such a timer and return true.  Otherwise return false.                   */
/*                                                                           */
/*****************************************************************************/

bool KheTimerSetContainsTimer(KHE_TIMER_SET timer_set, char *tag,
  KHE_TIMER *timer)
{
  KHE_TIMER tm;  int i;
  HaArrayForEach(timer_set->timers, tm, i)
    if( strcmp(tag, tm->tag) == 0 )
      return *timer = tm, true;
  return *timer = NULL, false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheTimerSetTimeLimitReached(KHE_TIMER_SET timer_set)                */
/*                                                                           */
/*  Return true if any of the time limits of timer_set have been reached.    */
/*                                                                           */
/*  Implementation note.  This does not call KheTimerTimeLimitReached,       */
/*  because it it optimized to call gettimeofday only once.                  */
/*                                                                           */
/*****************************************************************************/

bool KheTimerSetTimeLimitReached(KHE_TIMER_SET timer_set)
{
#if KHE_USE_TIMING
  struct timeval curr_time;  KHE_TIMER timer;  int i;  bool curr_time_known;
  curr_time_known = false;
  HaArrayForEach(timer_set->timers, timer, i)
    if( KheTimerReached(timer, &curr_time_known, &curr_time) )
      return true;
#endif
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_OPTIONS_TIME_LIMIT KheOptionsTimeMake(char *tag, float time_limit,   */
/*    HA_ARENA a)                                                            */
/*                                                                           */
/*  Make an options time limit object with these attributes.                 */
/*                                                                           */
/*****************************************************************************/

/* ***
static KHE_OPTIONS_TIME_LIMIT KheOptionsTimeMake(char *tag, float time_limit,
  HA_ARENA a)
{
  KHE_OPTIONS_TIME_LIMIT res;
  HaMake(res, a);
  res->tag = tag;
  res->timer = KheStatsTimerMake(a);
  res->time_limit = time_limit;
  return res;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_TIMER_LOG KheTimerSetGetTimerLog(KHE_TIMER_SET ts, char *tag)        */
/*                                                                           */
/*  Find or create a timer log with this tag.                                */
/*                                                                           */
/*****************************************************************************/

/* ***
static KHE_TIMER_LOG KheTimerSetGetTimerLog(KHE_TIMER_SET ts, char *tag)
{
  KHE_TIMER_LOG timer_log;  int i;
  if( DEBUG1 )
  {
    fprintf(stderr, "[ KheTimerSetGetTimerLog(timer_set, \"%s\"):\n", tag);
    KheTimerSetDebug(ts, 2, 2, stderr);
  }
  HaArrayForEach(ts->timer_logs, timer_log, i)
    if( strcmp(timer_log->tag, tag) == 0 )
      return timer_log;
  timer_log = KheTimerLogMake(tag, HaArrayArena(ts->timers));
  HaArrayAddLast(ts->timer_logs, timer_log);
  if( DEBUG1 )
  {
    KheTimerSetDebug(ts, 2, 2, stderr);
    fprintf(stderr, "] KheTimerSetGetTimerLog returning\n");
  }
  return timer_log;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheTimerSetSaveLog(KHE_TIMER_SET ts, KHE_SOLN soln)                 */
/*                                                                           */
/*  Save a log of ts in a file whose name is based on soln's instance        */
/*  and diversifier.                                                         */
/*                                                                           */
/*****************************************************************************/

/* ***
void KheTimerSetSaveLog(KHE_TIMER_SET ts, KHE_SOLN soln)
{
  char buff[200];  FILE *fp;  KHE_TIMER_LOG timer_log;  int i;
  snprintf(buff, 200, "log-%s-%d.tsl", KheInstanceId(KheSolnInstance(soln)),
    KheSolnDiversifier(soln));
  fp = fopen(buff, "w");
  HnAssert(fp != NULL, "KheTimerSetSaveLog:  cannot write file \"%s\"", buff);
  HaArrayForEach(ts->timer_logs, timer_log, i)
    KheTimerLogSave(timer_log, fp);
  fclose(fp);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheTimerSetDebug(KHE_TIMER_SET timer_set, int verbosity,            */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of time limits.                                              */
/*                                                                           */
/*****************************************************************************/

void KheTimerSetDebug(KHE_TIMER_SET timer_set, int verbosity,
  int indent, FILE *fp)
{
  int i;  KHE_TIMER timer;  /* KHE_TIMER_LOG timer_log; */
  if( indent >= 0 )
    fprintf(fp, "%*s", indent, "");
  if( verbosity >= 2 )
    fprintf(fp, "%p ", (void *) timer_set);
  if( HaArrayCount(timer_set->timers) == 0 )
    fprintf(fp, "-");
  else
  {
    HaArrayForEach(timer_set->timers, timer, i)
    {
      if( i > 0 )
	fprintf(fp, " & ");
      KheTimerDebug(timer, verbosity, -1, fp);
    }
  }
  /* ***
  if( verbosity >= 2 )
    HaArrayForEach(timer_set->timer_logs, timer_log, i)
      KheTimerLogSave(timer_log, fp);
  *** */
  if( indent >= 0 )
    fprintf(fp, "\n");
}
