
/*****************************************************************************/
/*                                                                           */
/*  THE KHE HIGH SCHOOL TIMETABLING ENGINE                                   */
/*  COPYRIGHT (C) 2010 Jeffrey H. Kingston                                   */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 3, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         khe_solvers.h                                              */
/*  DESCRIPTION:  Include this file when you use KHE's platform and solvers  */
/*                                                                           */
/*  This file has two parts:  type declarations and function declarations.   */
/*  Each part is organized to parallel the User's Guide exactly:             */
/*                                                                           */
/*                    Part B:  Solvers                                       */
/*                                                                           */
/*    Chapter 8.   Introducing Solvers                                       */
/*    Chapter 9.   Time-Structural Solvers                                   */
/*    Chapter 10.  Time Solvers                                              */
/*    Chapter 11.  Resource-Structural Solvers                               */
/*    Chapter 12.  Resource Solvers                                          */
/*    Chapter 13.  Ejection Chains                                           */
/*                                                                           */
/*  This simplifies verifying that KHE offers what the User's Guide says.    */
/*                                                                           */
/*****************************************************************************/
#ifndef KHE_SOLVERS_HEADER_FILE
#define KHE_SOLVERS_HEADER_FILE

#define KHE_USE_PTHREAD 1
#define KHE_USE_TIMING 1
#include <stdio.h>
#include <stdint.h>
#include <stdbool.h>
#include "khe_platform.h"
#include "howard_a.h"
#include "howard_n.h"

#define KHE_ANY_DURATION 0
#define KHE_COST_SHOW_DIGITS 99999


/*****************************************************************************/
/*                                                                           */
/*                        TYPE DECLARATIONS                                  */
/*                                                                           */
/*****************************************************************************/


/*****************************************************************************/
/*                                                                           */
/*                    Part B:  Solvers                                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*    Chapter 8.  Introducing Solvers                                        */
/*                                                                           */
/*****************************************************************************/
 
#define KHE_NO_TIME -1.0
typedef struct khe_timer_rec *KHE_TIMER;
typedef struct khe_timer_set_rec *KHE_TIMER_SET;
typedef struct khe_options_rec *KHE_OPTIONS;

typedef KHE_SOLN (*KHE_GENERAL_SOLVER)(KHE_SOLN soln, KHE_OPTIONS options);

typedef struct khe_file_rec *KHE_FILE;
typedef struct khe_table_rec *KHE_TABLE;
typedef struct khe_row_rec *KHE_ROW;
typedef struct khe_graph_rec *KHE_GRAPH;
typedef struct khe_dataset_rec *KHE_DATASET;

typedef enum {
  KHE_FILE_PLAIN,
  KHE_FILE_LOUT,
  KHE_FILE_LOUT_STANDALONE,
  KHE_FILE_LATEX
} KHE_FILE_FORMAT;

typedef enum {
  KHE_DATASET_POINTS_NONE,
  KHE_DATASET_POINTS_CROSS,
  KHE_DATASET_POINTS_SQUARE,
  KHE_DATASET_POINTS_DIAMOND,
  KHE_DATASET_POINTS_CIRCLE,
  KHE_DATASET_POINTS_TRIANGLE,
  KHE_DATASET_POINTS_PLUS,
  KHE_DATASET_POINTS_FILLED_SQUARE,
  KHE_DATASET_POINTS_FILLED_DIAMOND,
  KHE_DATASET_POINTS_FILLED_CIRCLE,
  KHE_DATASET_POINTS_FILLED_TRIANGLE
} KHE_DATASET_POINTS_TYPE;

typedef enum {
  KHE_DATASET_PAIRS_NONE,
  KHE_DATASET_PAIRS_SOLID,
  KHE_DATASET_PAIRS_DASHED,
  KHE_DATASET_PAIRS_DOTTED,
  KHE_DATASET_PAIRS_DOT_DASHED,
  KHE_DATASET_PAIRS_DOT_DOT_DASHED,
  KHE_DATASET_PAIRS_DOT_DOT_DOT_DASHED,
  KHE_DATASET_PAIRS_YHISTO,
  KHE_DATASET_PAIRS_SURFACE_YHISTO,
  KHE_DATASET_PAIRS_FILLED_YHISTO,
  KHE_DATASET_PAIRS_XHISTO,
  KHE_DATASET_PAIRS_SURFACE_XHISTO,
  KHE_DATASET_PAIRS_FILLED_XHISTO
} KHE_DATASET_PAIRS_TYPE;

/* ***
typedef enum {
  KHE_STATS_TABLE_PLAIN,
  KHE_STATS_TABLE_LOUT,
  KHE_STATS_TABLE_LATEX
} KHE_STATS_TABLE_TYPE;

typedef enum {
  KHE_STATS_DATASET_DASHED,
  KHE_STATS_DATASET_HISTO
} KHE_STATS_DATASET_TYPE;
*** */

/* typedef struct khe_stats_timer_rec *KHE_STATS_TIMER; */

typedef enum {
  KHE_BACKOFF_NONE,
  KHE_BACKOFF_EXPONENTIAL
} KHE_BACKOFF_TYPE;

typedef struct khe_backoff_rec *KHE_BACKOFF;


/*****************************************************************************/
/*                                                                           */
/*    Chapter 9.   Time-Structural Solvers                                   */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_time_equiv_rec *KHE_TIME_EQUIV;

typedef enum {
  KHE_SUBTAG_SPLIT_EVENTS,	      /* "SplitEventsGroupMonitor"           */
  KHE_SUBTAG_DISTRIBUTE_SPLIT_EVENTS, /* "DistributeSplitEventsGroupMonitor" */
  KHE_SUBTAG_ASSIGN_TIME,	      /* "AssignTimeGroupMonitor"            */
  KHE_SUBTAG_PREFER_TIMES,	      /* "PreferTimesGroupMonitor"           */
  KHE_SUBTAG_SPREAD_EVENTS,	      /* "SpreadEventsGroupMonitor"          */
  KHE_SUBTAG_LINK_EVENTS,	      /* "LinkEventsGroupMonitor"            */
  KHE_SUBTAG_ORDER_EVENTS,	      /* "OrderEventsGroupMonitor"           */
  KHE_SUBTAG_ASSIGN_RESOURCE,	      /* "AssignResourceGroupMonitor"        */
  KHE_SUBTAG_PREFER_RESOURCES,	      /* "PreferResourcesGroupMonitor"       */
  KHE_SUBTAG_AVOID_SPLIT_ASSIGNMENTS, /* "AvoidSplitAssignmentsGroupMonitor" */
  KHE_SUBTAG_AVOID_CLASHES,	      /* "AvoidClashesGroupMonitor"          */
  KHE_SUBTAG_AVOID_UNAVAILABLE_TIMES, /* "AvoidUnavailableTimesGroupMonitor" */
  KHE_SUBTAG_LIMIT_IDLE_TIMES,	      /* "LimitIdleTimesGroupMonitor"        */
  KHE_SUBTAG_CLUSTER_BUSY_TIMES,      /* "ClusterBusyTimesGroupMonitor"      */
  KHE_SUBTAG_LIMIT_BUSY_TIMES,	      /* "LimitBusyTimesGroupMonitor"        */
  KHE_SUBTAG_LIMIT_WORKLOAD,	      /* "LimitWorkloadGroupMonitor"         */
  KHE_SUBTAG_LIMIT_ACTIVE_INTERVALS,  /* "LimitActiveIntervalsGroupMonitor"  */
  KHE_SUBTAG_LIMIT_RESOURCES,         /* "LimitResourcesGroupMonitor"        */
  KHE_SUBTAG_ORDINARY_DEMAND,	      /* "OrdinaryDemandGroupMonitor"        */
  KHE_SUBTAG_WORKLOAD_DEMAND,	      /* "WorkloadDemandGroupMonitor"        */
  KHE_SUBTAG_KEMPE_DEMAND,	      /* "KempeDemandGroupMonitor"           */
  KHE_SUBTAG_NODE_TIME_REPAIR,	      /* "NodeTimeRepairGroupMonitor"        */
  KHE_SUBTAG_LAYER_TIME_REPAIR,	      /* "LayerTimeRepairGroupMonitor"       */
  KHE_SUBTAG_TASKING,		      /* "TaskingGroupMonitor"               */
  KHE_SUBTAG_RESOURCE_REPAIR,	      /* "ResourceRepairGroupMonitor"        */
  KHE_SUBTAG_ALL_DEMAND		      /* "AllDemandGroupMonitor"             */
} KHE_SUBTAG_STANDARD_TYPE;

typedef struct khe_split_analyser_rec *KHE_SPLIT_ANALYSER;


/*****************************************************************************/
/*                                                                           */
/*    Chapter 10.   Time Solvers                                             */
/*                                                                           */
/*****************************************************************************/

typedef bool (*KHE_NODE_TIME_SOLVER)(KHE_NODE parent_node, KHE_OPTIONS options);
typedef bool (*KHE_LAYER_TIME_SOLVER)(KHE_LAYER layer, KHE_OPTIONS options);

typedef struct khe_kempe_stats_rec *KHE_KEMPE_STATS;
typedef struct khe_meet_bound_group_rec *KHE_MEET_BOUND_GROUP;

typedef struct khe_layer_asst_rec *KHE_LAYER_ASST;

typedef struct khe_meet_set_solver_rec *KHE_MEET_SET_SOLVER;


/*****************************************************************************/
/*                                                                           */
/*    Chapter 11.  Resource-Structural Solvers                               */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_task_bound_group_rec *KHE_TASK_BOUND_GROUP;

typedef struct khe_consec_solver_rec *KHE_CONSEC_SOLVER;

typedef struct khe_tasker_rec *KHE_TASKER;
typedef struct khe_tasker_time_rec *KHE_TASKER_TIME;
typedef struct khe_tasker_class_rec *KHE_TASKER_CLASS;
typedef struct khe_profile_time_group_rec *KHE_PROFILE_TIME_GROUP;
typedef struct khe_comb_solver_rec *KHE_COMB_SOLVER;

typedef enum {
  KHE_COMB_SOLVER_COVER_YES,
  KHE_COMB_SOLVER_COVER_NO,
  KHE_COMB_SOLVER_COVER_FREE,
  KHE_COMB_SOLVER_COVER_PREV
} KHE_COMB_SOLVER_COVER_TYPE;

typedef enum {
  KHE_COMB_SOLVER_COST_MIN,
  KHE_COMB_SOLVER_COST_ZERO,
  KHE_COMB_SOLVER_COST_SOLE_ZERO
} KHE_COMB_SOLVER_COST_TYPE;

/* ***
typedef enum {
  KHE_GROUP_SOLVER_COVER_YES,
  KHE_GROUP_SOLVER_COVER_NO,
  KHE_GROUP_SOLVER_COVER_PREV
} KHE_GROUP_SOLVER_COVER_TYPE;

typedef enum {
  KHE_GROUP_SOLVER_COST_MIN,
  KHE_GROUP_SOLVER_COST_ZERO,
  KHE_GROUP_SOLVER_COST_SOLE_ZERO
} KHE_GROUP_SOLVER_COST_TYPE;

typedef struct khe_group_solver_rec *KHE_GROUP_SOLVER;
*** */

typedef struct khe_task_grouper_rec *KHE_TASK_GROUPER;

typedef struct khe_task_finder_rec *KHE_TASK_FINDER;
typedef struct khe_widened_task_set_rec *KHE_WIDENED_TASK_SET;
typedef struct khe_daily_schedule_rec *KHE_DAILY_SCHEDULE;
/* ***
typedef HA_ARRAY(KHE_TIME) ARRAY_KHE_TIME;
typedef HA_ARRAY(KHE_TASK) ARRAY_KHE_TASK;
*** */

/* ***
typedef struct khe_interval_rec {
  int first;
  int last;
} KHE_INTERVAL;
*** */

typedef struct khe_task_group_rec *KHE_TASK_GROUP;
typedef struct khe_task_groups_rec *KHE_TASK_GROUPS;


/*****************************************************************************/
/*                                                                           */
/*    Chapter 12.  Resource Solvers                                          */
/*                                                                           */
/*****************************************************************************/

typedef bool (*KHE_TASKING_SOLVER)(KHE_TASKING tasking, KHE_OPTIONS options);

/* ***
typedef struct khe_frame_iterator_rec {
  KHE_FRAME		frame;		** the frame being iterated over     **
  int			count;		** 1 + KheFrameTimeGroupCount(frame) **
  int			extra;		** start point along frame           **
  int			curr;		** current position of iterator      **
  KHE_RESOURCE		resource;	** resource whose activity is wanted **
  KHE_RESOURCE_TIMETABLE_MONITOR rtm;	** timetable monitor of resource     **
} *KHE_FRAME_ITERATOR;
*** */

typedef struct khe_resource_matching_solver_rec *KHE_RESOURCE_MATCHING_SOLVER;
typedef struct khe_resource_matching_demand_set_rec
  *KHE_RESOURCE_MATCHING_DEMAND_SET;
typedef HA_ARRAY(KHE_RESOURCE_MATCHING_DEMAND_SET)
  ARRAY_KHE_RESOURCE_MATCHING_DEMAND_SET;

typedef enum {
  KHE_SRS_DOM_WEAK,
  KHE_SRS_DOM_MEDIUM,
  KHE_SRS_DOM_STRONG,
  KHE_SRS_DOM_TRIE
} KHE_SRS_DOM_KIND;

typedef struct khe_single_resource_solver_rec *KHE_SINGLE_RESOURCE_SOLVER;

typedef enum {
  KHE_MOVE_UNCHECKED,			/* just a basic unchecked move       */
  KHE_MOVE_CHECKED,			/* do nothing if would cause clashes */
  KHE_MOVE_EJECTING,			/* eject clashes                     */
  KHE_MOVE_KEMPE			/* fix clashes using Kempe chains    */
} KHE_MOVE_TYPE;

typedef struct khe_resource_run_solver_rec *KHE_RESOURCE_RUN_SOLVER;

typedef struct khe_reassign_solver_rec *KHE_REASSIGN_SOLVER;

typedef enum {
  KHE_REASSIGN_MINIMAL,
  KHE_REASSIGN_RUNS,
  KHE_REASSIGN_MAXIMAL
} KHE_REASSIGN_GROUPING;

typedef enum {
  KHE_REASSIGN_EXHAUSTIVE,
  KHE_REASSIGN_MATCHING
} KHE_REASSIGN_METHOD;


/*****************************************************************************/
/*                                                                           */
/*    Chapter 13.  Ejection Chains                                           */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_ejector_rec *KHE_EJECTOR;

/* ***
typedef struct khe_ejector_major_schedule_rec *KHE_EJECTOR_MAJOR_SCHEDULE;
typedef struct khe_ejector_minor_schedule_rec *KHE_EJECTOR_MINOR_SCHEDULE;
*** */

typedef bool (*KHE_EJECTOR_AUGMENT_FN)(KHE_EJECTOR ej, KHE_MONITOR d);


/*****************************************************************************/
/*                                                                           */
/*                     FUNCTION DECLARATIONS                                 */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*                    Part B:  Solvers                                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*    Chapter  8.  Introducing Solvers                                       */
/*                                                                           */
/*****************************************************************************/

/* 8.1 Keeping track of running time */
extern char *KheDateToday(void);
extern float KheTimeFromString(char *str);
extern char *KheTimeShow(float secs, char buff[20]);

extern KHE_TIMER KheTimerMake(char *tag, float limit_in_seconds, HA_ARENA a);
extern KHE_TIMER KheTimerCopy(KHE_TIMER timer, HA_ARENA a);
extern char *KheTimerTag(KHE_TIMER timer);
extern float KheTimerTimeLimit(KHE_TIMER timer);
extern void KheTimerResetStartTime(KHE_TIMER timer);
extern void KheTimerResetTimeLimit(KHE_TIMER timer, float limit_in_seconds);
extern float KheTimerElapsedTime(KHE_TIMER timer);
extern bool KheTimerTimeLimitReached(KHE_TIMER timer);
extern void KheTimerDebug(KHE_TIMER timer, int verbosity, int indent, FILE *fp);

extern KHE_TIMER_SET KheTimerSetMake(HA_ARENA a);
extern KHE_TIMER_SET KheTimerSetCopy(KHE_TIMER_SET timer_set, HA_ARENA a);
extern void KheTimerSetAddTimer(KHE_TIMER_SET timer_set, KHE_TIMER timer);
extern void KheTimerSetDeleteTimer(KHE_TIMER_SET timer_set, KHE_TIMER timer);
extern bool KheTimerSetContainsTimer(KHE_TIMER_SET timer_set, char *tag,
  KHE_TIMER *timer);
extern bool KheTimerSetTimeLimitReached(KHE_TIMER_SET timer_set);
/* extern void KheTimerSetSaveLog(KHE_TIMER_SET ts, KHE_SOLN soln); */
extern void KheTimerSetDebug(KHE_TIMER_SET timer_set, int verbosity,
  int indent, FILE *fp);

/* 8.2 Options, running time, and time limits */
extern KHE_OPTIONS KheOptionsMake(HA_ARENA a);
extern HA_ARENA KheOptionsArena(KHE_OPTIONS options);
/* extern void KheOptionsDelete(KHE_OPTIONS options); */
extern KHE_OPTIONS KheOptionsCopy(KHE_OPTIONS options, HA_ARENA a);

extern void KheOptionsSet(KHE_OPTIONS options, char *key, char *value);
extern char *KheOptionsGet(KHE_OPTIONS options, char *key, char *dft);

extern void KheOptionsSetBool(KHE_OPTIONS options, char *key, bool value);
extern bool KheOptionsGetBool(KHE_OPTIONS options, char *key, bool dft);

extern void KheOptionsSetInt(KHE_OPTIONS options, char *key, int value);
extern int KheOptionsGetInt(KHE_OPTIONS options, char *key, int dft);

extern void KheOptionsSetObject(KHE_OPTIONS options, char *key, void *value);
extern void *KheOptionsGetObject(KHE_OPTIONS options, char *key, void *dft);

extern KHE_TIMER KheOptionsAddTimer(KHE_OPTIONS options, char *tag,
  float limit_in_seconds);
extern void KheOptionsDeleteTimer(KHE_OPTIONS options, KHE_TIMER timer);
extern bool KheOptionsContainsTimer(KHE_OPTIONS options, char *tag,
  KHE_TIMER *timer);
extern bool KheOptionsTimeLimitReached(KHE_OPTIONS options);
/* ***
extern void KheOptionsTimerSetSaveLog(KHE_OPTIONS options, KHE_SOLN soln);
*** */
extern void KheOptionsTimerSetDebug(KHE_OPTIONS options, int verbosity,
  int indent, FILE *fp);

/* ***
extern void KheOptionsTimeLimitBegin(KHE_OPTIONS options, char *tag,
  float limit_in_secs);
extern float KheOptionsTimeLimitEnd(KHE_OPTIONS options, char *tag);
extern bool KheOptionsTimeLimitNow(KHE_OPTIONS options, char *tag,
  float *elapsed_time_in_secs);
extern bool KheOptionsTimeLimitReached(KHE_OPTIONS options);
extern void KheOptionsTimeLimitReset(KHE_OPTIONS options, char *tag,
  float limit_in_secs);
*** */

/* ***
extern float KheOptionsTimeNow(KHE_OPTIONS options);
extern void KheOptionsSetTimeLimit(KHE_OPTIONS options, float limit_in_secs);
extern float KheOptionsTimeLimit(KHE_OPTIONS options);
extern bool KheOptionsTimeLimitReached(KHE_OPTIONS options);
*** */

extern KHE_FRAME KheOptionsFrame(KHE_OPTIONS options, char *key, KHE_SOLN soln);

/* 8.3 General solving */
extern KHE_SOLN KheGeneralSolve2020(KHE_SOLN soln, KHE_OPTIONS options);
extern float KheConvertTimeLimit(char *str);
extern void KheSolveDebug(KHE_SOLN soln, KHE_OPTIONS options, char *fmt, ...);

/* 8.4 Parallel solving */
extern void KheArchiveParallelSolve(KHE_ARCHIVE archive,
  KHE_GENERAL_SOLVER solver, KHE_OPTIONS options,
  KHE_SOLN_TYPE soln_type, HA_ARENA_SET as);
extern KHE_SOLN KheInstanceParallelSolve(KHE_INSTANCE ins,
  KHE_GENERAL_SOLVER solver, KHE_OPTIONS options,
  KHE_SOLN_TYPE soln_type, HA_ARENA_SET as);

/* 8.5 Monitor adjustments */
extern void KheDetachLowCostMonitors(KHE_SOLN soln, KHE_COST min_weight,
  KHE_GROUP_MONITOR low_cost_gm);
extern void KheAttachLowCostMonitors(KHE_GROUP_MONITOR low_cost_gm);
extern void KheSetMonitorMultipliers(KHE_SOLN soln, char *str, int val);
extern bool KhePropagateUnavailableTimes(KHE_SOLN soln, KHE_RESOURCE_TYPE rt);

/* Benchmarking - obsolete */
/* ***
extern bool KheBenchmarkTryInstance(KHE_INSTANCE ins);
extern void KheBenchmark(KHE_ARCHIVE archive, KHE_GENERAL_SOLVER solver,
  char *solver_name, char *author_name, char test_label,
  KHE_STATS_TABLE_TYPE table_type);
*** */

/* 8.6 Gathering statistics */
/* ***
extern KHE_STATS_TIMER KheStatsTimerMake(HA_ARENA a);
extern void KheStatsTimerReset(KHE_STATS_TIMER st);
extern float KheStatsTimerNow(KHE_STATS_TIMER st);
** extern void KheStatsTimerDelete(KHE_STATS_TIMER st); **
extern KHE_STATS_TIMER KheStatsTimerCopy(KHE_STATS_TIMER st, HA_ARENA a);

extern char *KheStatsDateToday(void);

extern void KheStatsFileBegin(char *file_name);
extern void KheStatsFileEnd(char *file_name);

extern void KheStatsTableBegin(char *file_name, KHE_STATS_TABLE_TYPE table_type,
  int col_width, char *corner, bool with_average_row, bool with_total_row,
  bool highlight_cost_minima, bool highlight_time_minima,
  bool highlight_int_minima);
extern void KheStatsTableEnd(char *file_name);
extern void KheStatsCaptionAdd(char *file_name, char *fmt, ...);
extern void KheStatsRowAdd(char *file_name, char *row_label, bool rule_below);
extern void KheStatsColAdd(char *file_name, char *col_label, bool rule_after);

extern void KheStatsEntryAddString(char *file_name, char *row_label,
  char *col_label, char *str);
extern void KheStatsEntryAddCost(char *file_name, char *row_label,
  char *col_label, KHE_COST cost);
extern void KheStatsEntryAddTime(char *file_name, char *row_label,
  char *col_label, float time);
extern void KheStatsEntryAddInt(char *file_name, char *row_label,
  char *col_label, int val);

extern void KheStatsGraphBegin(char *file_name);
extern void KheStatsGraphEnd(char *file_name);
extern void KheStatsGraphSetWidth(char *file_name, float width);
extern void KheStatsGraphSetHeight(char *file_name, float height);
extern void KheStatsGraphSetXMax(char *file_name, float xmax);
extern void KheStatsGraphSetYMax(char *file_name, float ymax);
extern void KheStatsGraphSetAboveCaption(char *file_name, char *val);
extern void KheStatsGraphSetBelowCaption(char *file_name, char *val);
extern void KheStatsGraphSetLeftCaption(char *file_name, char *val);
extern void KheStatsGraphSetRightCaption(char *file_name, char *val);
extern void KheStatsDataSetAdd(char *file_name, char *dataset_label,
  KHE_STATS_DATASET_TYPE dataset_type);
extern void KheStatsPointAdd(char *file_name, char *dataset_label,
  float x, float y);
*** */

extern KHE_FILE KheFileBegin(char *file_name, KHE_FILE_FORMAT fmt);
extern void KheFileEnd(KHE_FILE kf);

extern KHE_TABLE KheTableBegin(KHE_FILE kf, int col_width, char *corner,
  bool with_average_row, bool with_total_row, bool highlight_cost_minima,
  bool highlight_time_minima, bool highlight_int_minima);
extern void KheTableEnd(KHE_TABLE kt);
extern void KheTableCaptionAdd(KHE_TABLE kt, char *fmt, ...);
extern void KheTableRowAdd(KHE_TABLE kt, char *row_label, bool rule_below);
extern void KheTableColAdd(KHE_TABLE kt, char *col_label, bool rule_after);

extern void KheTableEntryAddString(KHE_TABLE kt, char *row_label,
  char *col_label, char *str);
extern void KheTableEntryAddCost(KHE_TABLE kt, char *row_label,
  char *col_label, KHE_COST cost);
extern void KheTableEntryAddTime(KHE_TABLE kt, char *row_label,
  char *col_label, float time);
extern void KheTableEntryAddInt(KHE_TABLE kt, char *row_label,
  char *col_label, int val);

extern KHE_GRAPH KheGraphBegin(KHE_FILE kf);
extern void KheGraphEnd(KHE_GRAPH kg);
extern void KheGraphSetWidth(KHE_GRAPH kg, float width);
extern void KheGraphSetHeight(KHE_GRAPH kg, float height);
extern void KheGraphSetXMax(KHE_GRAPH kg, float xmax);
extern void KheGraphSetYMax(KHE_GRAPH kg, float ymax);
extern void KheGraphSetAboveCaption(KHE_GRAPH kg, char *val);
extern void KheGraphSetBelowCaption(KHE_GRAPH kg, char *val);
extern void KheGraphSetLeftCaptionAndGap(KHE_GRAPH kg, char *val, char *gap);
extern void KheGraphSetRightCaptionAndGap(KHE_GRAPH kg, char *val, char *gap);
extern KHE_DATASET KheDataSetAdd(KHE_GRAPH kg,
  KHE_DATASET_POINTS_TYPE points_type,
  KHE_DATASET_PAIRS_TYPE pairs_type, char *label);
extern void KhePointAdd(KHE_DATASET kd, float x, float y);

/* 8.7 Exponential backoff */
extern KHE_BACKOFF KheBackoffBegin(KHE_BACKOFF_TYPE backoff_type, HA_ARENA a);
extern bool KheBackoffAcceptOpportunity(KHE_BACKOFF bk);
extern void KheBackoffResult(KHE_BACKOFF bk, bool success);
/* extern void KheBackoffEnd(KHE_BACKOFF bk); */

extern char *KheBackoffShowNextDecision(KHE_BACKOFF bk);
extern void KheBackoffDebug(KHE_BACKOFF bk, int verbosity, int indent,FILE *fp);
extern void KheBackoffTest(FILE *fp);


/*****************************************************************************/
/*                                                                           */
/*    Chapter 9.   Structural Solvers                                        */
/*                                                                           */
/*****************************************************************************/

/* 9.1 Layer tree construction */
extern KHE_NODE KheLayerTreeMake(KHE_SOLN soln);

/* 9.2 Time-equivalence */
extern KHE_TIME_EQUIV KheTimeEquivMake(KHE_SOLN soln);
extern void KheTimeEquivDelete(KHE_TIME_EQUIV te);
extern void KheTimeEquivSolve(KHE_TIME_EQUIV te);

extern KHE_TIME_EQUIV KheTimeEquivOption(KHE_OPTIONS options,
  char *key, KHE_SOLN soln);

extern int KheTimeEquivEventGroupCount(KHE_TIME_EQUIV te);
extern KHE_EVENT_GROUP KheTimeEquivEventGroup(KHE_TIME_EQUIV te, int i);
extern KHE_EVENT_GROUP KheTimeEquivEventEventGroup(KHE_TIME_EQUIV te,
  KHE_EVENT e);
extern int KheTimeEquivEventEventGroupIndex(KHE_TIME_EQUIV te, KHE_EVENT e);

extern int KheTimeEquivResourceGroupCount(KHE_TIME_EQUIV te);
extern KHE_RESOURCE_GROUP KheTimeEquivResourceGroup(KHE_TIME_EQUIV te, int i);
extern KHE_RESOURCE_GROUP KheTimeEquivResourceResourceGroup(KHE_TIME_EQUIV te,
  KHE_RESOURCE r);
extern int KheTimeEquivResourceResourceGroupIndex(KHE_TIME_EQUIV te,
  KHE_RESOURCE r);

/* 9.3.1 Layer construction */
extern KHE_LAYER KheLayerMakeFromResource(KHE_NODE parent_node,
  KHE_RESOURCE r);
extern void KheNodeChildLayersMake(KHE_NODE parent_node);
extern void KheNodeChildLayersReduce(KHE_NODE parent_node);

/* 9.3.2 Layer coordination */
extern void KheCoordinateLayers(KHE_NODE parent_node, bool with_domination);

/* 9.4 Runarounds */
extern bool KheMinimumRunaroundDuration(KHE_NODE parent_node,
  KHE_NODE_TIME_SOLVER time_solver, KHE_OPTIONS options, int *duration);
extern void KheBuildRunarounds(KHE_NODE parent_node,
  KHE_NODE_TIME_SOLVER mrd_solver, KHE_OPTIONS mrd_options,
  KHE_NODE_TIME_SOLVER runaround_solver, KHE_OPTIONS runaround_options);

/* 9.5.1 Node merging */
extern bool KheNodeMergeCheck(KHE_NODE node1, KHE_NODE node2);
extern bool KheNodeMerge(KHE_NODE node1, KHE_NODE node2, KHE_NODE *res);

/* 9.5.2 Node meet splitting and merging */
extern void KheNodeMeetSplit(KHE_NODE node, bool recursive);
extern void KheNodeMeetMerge(KHE_NODE node, bool recursive);

/* 9.5.3 Node moving */
extern bool KheNodeMoveCheck(KHE_NODE child_node, KHE_NODE parent_node);
extern bool KheNodeMove(KHE_NODE child_node, KHE_NODE parent_node);

/* 9.5.4 Vizier nodes */
extern KHE_NODE KheNodeVizierMake(KHE_NODE parent_node);
extern void KheNodeVizierDelete(KHE_NODE parent_node);

/* 9.5.5 Flattening */
extern void KheNodeBypass(KHE_NODE node);
extern void KheNodeFlatten(KHE_NODE parent_node);

/* 9.6 Adding zones */
extern void KheLayerInstallZonesInParent(KHE_LAYER layer);
extern void KheNodeExtendZones(KHE_NODE node);

/* 9.7.1 Analysing split defects */
extern KHE_SPLIT_ANALYSER KheSplitAnalyserMake(KHE_SOLN soln);
extern void KheSplitAnalyserDelete(KHE_SPLIT_ANALYSER sa);
extern KHE_SPLIT_ANALYSER KheSplitAnalyserOption(KHE_OPTIONS options,
  char *key, KHE_SOLN soln);

extern void KheSplitAnalyserAnalyse(KHE_SPLIT_ANALYSER sa, KHE_EVENT e);
extern int KheSplitAnalyserSplitSuggestionCount(KHE_SPLIT_ANALYSER sa);
extern void KheSplitAnalyserSplitSuggestion(KHE_SPLIT_ANALYSER sa, int i,
  int *merged_durn, int *split1_durn);
extern int KheSplitAnalyserMergeSuggestionCount(KHE_SPLIT_ANALYSER sa);
extern void KheSplitAnalyserMergeSuggestion(KHE_SPLIT_ANALYSER sa, int i,
  int *split1_durn, int *split2_durn);
extern void KheSplitAnalyserDebug(KHE_SPLIT_ANALYSER sa, int verbosity,
  int indent, FILE *fp);

/* 9.7.2 Merging adjacent meets */
extern void KheMergeMeets(KHE_SOLN soln);

/* 9.8 Monitor attachment and grouping */
extern char *KheSubTagLabel(KHE_SUBTAG_STANDARD_TYPE sub_tag);
extern KHE_SUBTAG_STANDARD_TYPE KheSubTagFromTag(KHE_MONITOR_TAG tag);
extern bool KheMonitorHasParent(KHE_MONITOR m, int sub_tag,
  KHE_GROUP_MONITOR *res_gm);
extern void KheMonitorAddSelfOrParent(KHE_MONITOR m, int sub_tag,
  KHE_GROUP_MONITOR gm);
extern void KheMonitorDeleteAllParentsRecursive(KHE_MONITOR m);


/*****************************************************************************/
/*                                                                           */
/*    Chapter 10.   Time Solvers                                             */
/*                                                                           */
/*****************************************************************************/

/* 10.2.1 Node assignment functions */
extern bool KheNodeMeetSwapCheck(KHE_NODE node1, KHE_NODE node2);
extern bool KheNodeMeetSwap(KHE_NODE node1, KHE_NODE node2);
extern bool KheNodeMeetRegularAssignCheck(KHE_NODE node, KHE_NODE sibling_node);
extern bool KheNodeMeetRegularAssign(KHE_NODE node, KHE_NODE sibling_node);
extern void KheNodeMeetUnAssign(KHE_NODE node);

/* 10.2.2 Kempe and ejecting meet moves */
extern KHE_GROUP_MONITOR KheKempeDemandGroupMonitorMake(KHE_SOLN soln);
extern bool KheKempeMeetMove(KHE_MEET meet, KHE_MEET target_meet,
  int offset, bool preserve_regularity, int *demand, bool *basic,
  KHE_KEMPE_STATS kempe_stats);
extern bool KheKempeMeetMoveTime(KHE_MEET meet, KHE_TIME t,
  bool preserve_regularity, int *demand, bool *basic,
  KHE_KEMPE_STATS kempe_stats);

extern KHE_KEMPE_STATS KheKempeStatsMake(HA_ARENA a);
extern void KheKempeStatsDelete(KHE_KEMPE_STATS kempe_stats);
extern KHE_KEMPE_STATS KheKempeStatsOption(KHE_OPTIONS options, char *key);

extern int KheKempeStatsStepHistoMax(KHE_KEMPE_STATS kempe_stats);
extern int KheKempeStatsStepHistoFrequency(KHE_KEMPE_STATS kempe_stats,
  int step_count);
extern int KheKempeStatsStepHistoTotal(KHE_KEMPE_STATS kempe_stats);
extern float KheKempeStatsStepHistoAverage(KHE_KEMPE_STATS kempe_stats);

extern int KheKempeStatsPhaseHistoMax(KHE_KEMPE_STATS kempe_stats);
extern int KheKempeStatsPhaseHistoFrequency(KHE_KEMPE_STATS kempe_stats,
  int phase_count);
extern int KheKempeStatsPhaseHistoTotal(KHE_KEMPE_STATS kempe_stats);
extern float KheKempeStatsPhaseHistoAverage(KHE_KEMPE_STATS kempe_stats);

extern bool KheEjectingMeetMove(KHE_MEET meet, KHE_MEET target_meet, int offset,
  bool allow_eject, bool preserve_regularity, int *demand, bool *basic);
extern bool KheEjectingMeetMoveTime(KHE_MEET meet, KHE_TIME t,
  bool allow_eject, bool preserve_regularity, int *demand, bool *basic);
extern bool KheBasicMeetMove(KHE_MEET meet, KHE_MEET target_meet,
  int offset, bool preserve_regularity, int *demand);
extern bool KheBasicMeetMoveTime(KHE_MEET meet, KHE_TIME t,
  bool preserve_regularity, int *demand);
 
extern bool KheTypedMeetMove(KHE_MEET meet, KHE_MEET target_meet, int offset,
  KHE_MOVE_TYPE mt, bool preserve_regularity, int *demand, bool *basic,
  KHE_KEMPE_STATS kempe_stats);
extern bool KheTypedMeetMoveTime(KHE_MEET meet, KHE_TIME t,
  KHE_MOVE_TYPE mt, bool preserve_regularity, int *demand, bool *basic,
  KHE_KEMPE_STATS kempe_stats);

/* 10.3.1 Meet bound groups */
extern KHE_MEET_BOUND_GROUP KheMeetBoundGroupMake(KHE_SOLN soln);
extern void KheMeetBoundGroupAddMeetBound(KHE_MEET_BOUND_GROUP mbg,
  KHE_MEET_BOUND mb);
extern int KheMeetBoundGroupMeetBoundCount(KHE_MEET_BOUND_GROUP mbg);
extern KHE_MEET_BOUND KheMeetBoundGroupMeetBound(KHE_MEET_BOUND_GROUP mbg,
  int i);
extern bool KheMeetBoundGroupDelete(KHE_MEET_BOUND_GROUP mbg);

/* 10.3.2 Exposing resource unavailability */
extern void KheMeetAddUnavailableBound(KHE_MEET meet, KHE_COST min_weight,
  KHE_MEET_BOUND_GROUP mbg);
extern void KheSolnAddUnavailableBounds(KHE_SOLN soln, KHE_COST min_weight,
  KHE_MEET_BOUND_GROUP mbg);

/* 10.3.3 Preventing cluster busy times and limit idle times defects */
extern void KheSolnClusterAndLimitMeetDomains(KHE_SOLN soln,
  KHE_COST min_cluster_weight, KHE_COST min_idle_weight,
  float slack, KHE_MEET_BOUND_GROUP mbg, KHE_OPTIONS options);

/* 10.4 Some basic time solvers */
extern bool KheNodeSimpleAssignTimes(KHE_NODE parent_node, KHE_OPTIONS options);
extern bool KheLayerSimpleAssignTimes(KHE_LAYER layer, KHE_OPTIONS options);

extern bool KheNodeRecursiveAssignTimes(KHE_NODE parent_node,
  KHE_NODE_TIME_SOLVER solver, KHE_OPTIONS options);

extern bool KheNodeUnAssignTimes(KHE_NODE parent_node, KHE_OPTIONS options);
extern bool KheLayerUnAssignTimes(KHE_LAYER layer, KHE_OPTIONS options);

extern bool KheNodeAllChildMeetsAssigned(KHE_NODE parent_node);
extern bool KheLayerAllChildMeetsAssigned(KHE_LAYER layer);

extern bool KheNodePreassignedAssignTimes(KHE_NODE root_node,
  KHE_OPTIONS options);
extern bool KheLayerPreassignedAssignTimes(KHE_LAYER layer,
  KHE_OPTIONS options);

extern bool KheSolnTryMeetUnAssignments(KHE_SOLN soln);

/* 10.5 A time solver for runarounds */
extern bool KheRunaroundNodeAssignTimes(KHE_NODE parent_node,
  KHE_OPTIONS options);

/* 10.6 Extended layer matching with Elm */
extern bool KheElmLayerAssign(KHE_LAYER layer,
  KHE_SPREAD_EVENTS_CONSTRAINT sec, KHE_OPTIONS options);

/* 10.7.1 Node-regular time repair using layer node matching */
extern bool KheLayerNodeMatchingNodeRepairTimes(KHE_NODE parent_node,
  KHE_OPTIONS options);
extern bool KheLayerNodeMatchingLayerRepairTimes(KHE_LAYER layer,
  KHE_OPTIONS options);

/* 10.7.3 Tree search layer time repair */
extern bool KheTreeSearchLayerRepairTimes(KHE_SOLN soln, KHE_RESOURCE r);
extern bool KheTreeSearchRepairTimes(KHE_SOLN soln, KHE_RESOURCE_TYPE rt,
  bool with_defects);

/* 10.7.4 Meet set time repair */
extern KHE_MEET_SET_SOLVER KheMeetSetSolveBegin(KHE_SOLN soln, int max_meets);
extern void KheMeetSetSolveAddMeet(KHE_MEET_SET_SOLVER mss, KHE_MEET meet);
extern bool KheMeetSetSolveEnd(KHE_MEET_SET_SOLVER mss);

extern bool KheFuzzyMeetMove(KHE_MEET meet, KHE_MEET target_meet, int offset,
  int width, int depth, int max_meets);

/* 10.8.1 Layer assignments */
extern KHE_LAYER_ASST KheLayerAsstMake(KHE_SOLN soln);
extern void KheLayerAsstDelete(KHE_LAYER_ASST layer_asst);
extern void KheLayerAsstBegin(KHE_LAYER_ASST layer_asst, KHE_LAYER layer);
extern void KheLayerAsstEnd(KHE_LAYER_ASST layer_asst);
extern void KheLayerAsstUndo(KHE_LAYER_ASST layer_asst);
extern void KheLayerAsstRedo(KHE_LAYER_ASST layer_asst);
extern void KheLayerAsstDebug(KHE_LAYER_ASST layer_asst, int verbosity,
  int indent, FILE *fp);

/* 10.8.2 A solver for layered time assignment */
extern bool KheNodeLayeredAssignTimes(KHE_NODE parent_node,KHE_OPTIONS options);
extern int KheNodeLayeredLayerCmp(const void *t1, const void *t2);
extern bool KheLayerParallelAssignTimes(KHE_LAYER layer, KHE_OPTIONS options);

/* 10.8.3 A complete time solver */
extern bool KheCycleNodeAssignTimes(KHE_NODE cycle_node, KHE_OPTIONS options);


/*****************************************************************************/
/*                                                                           */
/*    Chapter 11.  Resource-Structural Solvers                               */
/*                                                                           */
/*****************************************************************************/

/* 11.1 Task bound groups */
extern KHE_TASK_BOUND_GROUP KheTaskBoundGroupMake(KHE_SOLN soln);
extern void KheTaskBoundGroupAddTaskBound(KHE_TASK_BOUND_GROUP tbg,
  KHE_TASK_BOUND tb);
extern int KheTaskBoundGroupTaskBoundCount(KHE_TASK_BOUND_GROUP tbg);
extern KHE_TASK_BOUND KheTaskBoundGroupTaskBound(KHE_TASK_BOUND_GROUP tbg,
  int i);
extern bool KheTaskBoundGroupDelete(KHE_TASK_BOUND_GROUP tbg);

/* 11.3 Task tree construction */
extern bool KheTaskTreeMake(KHE_SOLN soln, KHE_OPTIONS options);
extern KHE_TASKING KheTaskingMakeFromResourceType(KHE_SOLN soln,
  KHE_RESOURCE_TYPE rt);
extern bool KheTaskingMakeTaskTree(KHE_TASKING tasking,
  KHE_TASK_BOUND_GROUP tbg, KHE_OPTIONS options);

/* 11.4 Classifying resources by available workload */
extern bool KheClassifyResourcesByWorkload(KHE_SOLN soln, KHE_RESOURCE_GROUP rg,
  KHE_RESOURCE_GROUP *rg1, KHE_RESOURCE_GROUP *rg2);

/* 11.5 Limits on consecutive days */
extern KHE_CONSEC_SOLVER KheConsecSolverMake(KHE_SOLN soln, KHE_FRAME frame);
extern KHE_SOLN KheConsecSolverSoln(KHE_CONSEC_SOLVER cs);
extern KHE_FRAME KheConsecSolverFrame(KHE_CONSEC_SOLVER cs);
extern void KheConsecSolverDelete(KHE_CONSEC_SOLVER cs);

extern void KheConsecSolverFreeDaysLimits(KHE_CONSEC_SOLVER cs, KHE_RESOURCE r,
  int *history, int *min_limit, int *max_limit);
extern void KheConsecSolverBusyDaysLimits(KHE_CONSEC_SOLVER cs, KHE_RESOURCE r,
  int *history, int *min_limit, int *max_limit);
extern void KheConsecSolverBusyTimesLimits(KHE_CONSEC_SOLVER cs, KHE_RESOURCE r,
  int offset, int *history, int *min_limit, int *max_limit);

extern int KheConsecSolverMaxOffset(KHE_CONSEC_SOLVER cs);

extern void KheConsecSolverDebug(KHE_CONSEC_SOLVER cs, int verbosity,
  int indent, FILE *fp);

/* 11.6 Tighten to partition */
extern bool KheTaskingTightenToPartition(KHE_TASKING tasking,
  KHE_TASK_BOUND_GROUP tbg, KHE_OPTIONS options);

/* 11.7 Grouping by resource constraints */
extern bool KheGroupByResourceConstraints(KHE_SOLN soln,
  KHE_RESOURCE_TYPE rt, KHE_OPTIONS options, KHE_TASK_SET ts);

/* 11.7.1 Tasker objects */
extern int KheTaskerClassTaskCount(KHE_TASKER_CLASS c);
extern KHE_TASK KheTaskerClassTask(KHE_TASKER_CLASS c, int i);

extern KHE_RESOURCE_GROUP KheTaskerClassDomain(KHE_TASKER_CLASS c);
extern KHE_RESOURCE KheTaskerClassAsstResource(KHE_TASKER_CLASS c);
extern KHE_TIME_SET KheTaskerClassTimeSet(KHE_TASKER_CLASS c);

extern void KheTaskerClassDebug(KHE_TASKER_CLASS c, int verbosity,
  int indent, FILE *fp);

extern KHE_TIME KheTaskerTimeTime(KHE_TASKER_TIME t);

extern int KheTaskerTimeClassCount(KHE_TASKER_TIME t);
extern KHE_TASKER_CLASS KheTaskerTimeClass(KHE_TASKER_TIME t, int i);

extern void KheTaskerTimeDebug(KHE_TASKER_TIME t, int verbosity,
  int indent, FILE *fp);

extern KHE_TASKER KheTaskerMake(KHE_SOLN soln, KHE_RESOURCE_TYPE rt,
  KHE_TASK_SET task_set, HA_ARENA a);

extern KHE_SOLN KheTaskerSoln(KHE_TASKER tr);
extern KHE_RESOURCE_TYPE KheTaskerResourceType(KHE_TASKER tr);
extern HA_ARENA KheTaskerArena(KHE_TASKER tr);

extern int KheTaskerTimeCount(KHE_TASKER tr);
extern KHE_TASKER_TIME KheTaskerTime(KHE_TASKER tr, int i);

extern void KheTaskerDebug(KHE_TASKER tr, int verbosity, int indent, FILE *fp);

/* 11.7.2 Tasker support for grouping */
extern void KheTaskerGroupingClear(KHE_TASKER tr);
extern bool KheTaskerGroupingAddClass(KHE_TASKER tr, KHE_TASKER_CLASS c);
extern bool KheTaskerGroupingDeleteClass(KHE_TASKER tr, KHE_TASKER_CLASS c);
extern int KheTaskerGroupingBuild(KHE_TASKER tr, int max_num, char *debug_str);

extern int KheTaskerGroupingClassCount(KHE_TASKER tr);
extern KHE_TASKER_CLASS KheTaskerGroupingClass(KHE_TASKER tr, int i);

extern void KheTaskerAddOverlapFrame(KHE_TASKER tr, KHE_FRAME frame);
extern void KheTaskerDeleteOverlapFrame(KHE_TASKER tr);
/* ***
extern void KheTaskerAddOverlapTimeGroup(KHE_TASKER tr, KHE_TIME_GROUP tg);
extern void KheTaskerClearOverlapTimeGroups(KHE_TASKER tr);
*** */

extern bool KheTaskerTimeOverlapsGrouping(KHE_TASKER_TIME t);
extern bool KheTaskerClassOverlapsGrouping(KHE_TASKER_CLASS c);

/* ***
extern bool KheTaskerGroupingTestAsstBegin(KHE_TASKER tr, KHE_RESOURCE *r);
extern void KheTaskerGroupingTestAsstEnd(KHE_TASKER tr);
*** */

/* 11.7.3 Tasker support for profile grouping */
extern void KheTaskerSetProfileMaxLen(KHE_TASKER tr, int profile_max_len);
extern int KheTaskerProfileMaxLen(KHE_TASKER tr);

extern int KheTaskerProfileTimeGroupCount(KHE_TASKER tr);
extern KHE_PROFILE_TIME_GROUP KheTaskerProfileTimeGroup(KHE_TASKER tr, int i);

extern KHE_PROFILE_TIME_GROUP KheProfileTimeGroupMake(KHE_TASKER tr,
  KHE_TIME_GROUP tg);
extern void KheProfileTimeGroupDelete(KHE_PROFILE_TIME_GROUP ptg);
extern void KheTaskerDeleteProfileTimeGroups(KHE_TASKER tr);

extern KHE_TASKER KheProfileTimeGroupTasker(KHE_PROFILE_TIME_GROUP ptg);
extern KHE_TIME_GROUP KheProfileTimeGroupTimeGroup(KHE_PROFILE_TIME_GROUP ptg);
extern int KheProfileTimeGroupCover(KHE_PROFILE_TIME_GROUP ptg);
extern int KheProfileTimeGroupDomainCount(KHE_PROFILE_TIME_GROUP ptg);
extern KHE_RESOURCE_GROUP KheProfileTimeGroupDomain(KHE_PROFILE_TIME_GROUP ptg,
  int i, int *cover);
extern bool KheProfileTimeGroupContainsDomain(KHE_PROFILE_TIME_GROUP ptg,
  KHE_RESOURCE_GROUP domain, int *cover);
extern bool KheTaskerClassCoversProfileTimeGroup(KHE_TASKER_CLASS c,
  KHE_PROFILE_TIME_GROUP ptg);
extern int KheTaskerClassProfileTimeCount(KHE_TASKER_CLASS c);
extern KHE_TASKER_TIME KheTaskerClassProfileTime(KHE_TASKER_CLASS c, int i);

extern void KheTaskerProfileDebug(KHE_TASKER tr, int verbosity, int indent,
  FILE *fp);

/* 11.7.4 Combinatorial grouping */
extern KHE_COMB_SOLVER KheCombSolverMake(KHE_TASKER tr, KHE_FRAME days_frame);
extern KHE_TASKER KheCombSolverTasker(KHE_COMB_SOLVER cs);
extern KHE_FRAME KheCombSolverFrame(KHE_COMB_SOLVER cs);

extern void KheCombSolverClearRequirements(KHE_COMB_SOLVER cs);
extern void KheCombSolverAddTimeGroupRequirement(KHE_COMB_SOLVER cs,
  KHE_TIME_GROUP tg, KHE_COMB_SOLVER_COVER_TYPE cover);
extern void KheCombSolverDeleteTimeGroupRequirement(KHE_COMB_SOLVER cs,
  KHE_TIME_GROUP tg);
extern void KheCombSolverAddClassRequirement(KHE_COMB_SOLVER cs,
  KHE_TASKER_CLASS c, KHE_COMB_SOLVER_COVER_TYPE cover);
extern void KheCombSolverDeleteClassRequirement(KHE_COMB_SOLVER cs,
  KHE_TASKER_CLASS c);

extern void KheCombSolverAddProfileGroupRequirement(KHE_COMB_SOLVER cs,
  KHE_PROFILE_TIME_GROUP ptg, KHE_RESOURCE_GROUP domain);
extern void KheCombSolverDeleteProfileGroupRequirement(KHE_COMB_SOLVER cs,
  KHE_PROFILE_TIME_GROUP ptg);
extern void KheCombSolverAddProfileMaxLenRequirement(KHE_COMB_SOLVER cs);
extern void KheCombSolverDeleteProfileMaxLenRequirement(KHE_COMB_SOLVER cs);
extern void KheCombSolverAddNoSinglesRequirement(KHE_COMB_SOLVER cs);
extern void KheCombSolverDeleteNoSinglesRequirement(KHE_COMB_SOLVER cs);

extern int KheCombSolverSolve(KHE_COMB_SOLVER cs, int max_num,
  KHE_COMB_SOLVER_COST_TYPE ct, char *debug_str);
extern int KheCombSolverSingleTasks(KHE_COMB_SOLVER cs);

extern void KheCombSolverDebug(KHE_COMB_SOLVER cs, int verbosity,
  int indent, FILE *fp);

/* 11.7.5 Applying combinatorial grouping */
int KheCombGrouping(KHE_COMB_SOLVER cs, KHE_OPTIONS options);

/* 11.7.7 Applying profile grouping */
int KheProfileGrouping(KHE_COMB_SOLVER cs, bool extended);

/* group solvers */
/* ***
extern KHE_GROUP_SOLVER KheGroupSolverMake(KHE_SOLN soln, KHE_RESOURCE_TYPE rt,
  HA_ARENA a);
extern void KheGroupSolverClearTimeGroups(KHE_GROUP_SOLVER gs);
extern void KheGroupSolverAddTimeGroup(KHE_GROUP_SOLVER gs, KHE_TIME_GROUP tg,
  KHE_GROUP_SOLVER_COVER_TYPE cover);

extern bool KheGroupSolverTestSolve(KHE_GROUP_SOLVER gs,
  KHE_GROUP_SOLVER_COST_TYPE ct, KHE_RESOURCE_GROUP domain,
  bool allow_single, KHE_TASK *leader_task);
extern int KheGroupSolverSolve(KHE_GROUP_SOLVER gs, int max_num,
  KHE_GROUP_SOLVER_COST_TYPE ct, KHE_RESOURCE_GROUP domain,
  bool allow_single, KHE_TASK_SET ts, char *debug_str);
*** */

/* 11.8 Grouping by resource */
extern bool KheTaskingGroupByResource(KHE_TASKING tasking, KHE_OPTIONS options,
  KHE_TASK_SET ts);

/* 11.9 The task grouper */
extern KHE_TASK_GROUPER KheTaskGrouperMake(KHE_RESOURCE_TYPE rt, HA_ARENA a);
extern void KheTaskGrouperClear(KHE_TASK_GROUPER tg);
extern bool KheTaskGrouperAddTask(KHE_TASK_GROUPER tg, KHE_TASK t);
/* ***
extern KHE_TASK KheTaskGrouperLeaderTask(KHE_TASK_GROUPER tg, KHE_RESOURCE r);
*** */
extern void KheTaskGrouperGroup(KHE_TASK_GROUPER tg);
extern void KheTaskGrouperUnGroup(KHE_TASK_GROUPER tg);

/* 11.10 Task finder objects */
extern KHE_TASK_FINDER KheTaskFinderMake(KHE_SOLN soln, KHE_OPTIONS options,
  HA_ARENA a);
extern int KheTaskFinderLastIndex(KHE_TASK_FINDER tf);
extern KHE_FRAME KheTaskFinderFrame(KHE_TASK_FINDER tf);

extern void KheTaskFinderTaskInterval(KHE_TASK_FINDER tf,
  KHE_TASK task, int *first_index, int *last_index);
extern void KheTaskFinderTaskSetInterval(KHE_TASK_FINDER tf,
  KHE_TASK_SET ts, int *first_index, int *last_index);
extern void KheTaskFinderTimeGroupInterval(KHE_TASK_FINDER tf,
  KHE_TIME_GROUP tg, int *first_index, int *last_index);

extern KHE_DAILY_SCHEDULE KheTaskFinderTaskDailySchedule(
  KHE_TASK_FINDER tf, KHE_TASK task);
extern KHE_DAILY_SCHEDULE KheTaskFinderTaskSetDailySchedule(
  KHE_TASK_FINDER tf, KHE_TASK_SET ts);
extern KHE_DAILY_SCHEDULE KheTaskFinderTimeGroupDailySchedule(
  KHE_TASK_FINDER tf, KHE_TIME_GROUP tg);
extern KHE_DAILY_SCHEDULE KheTaskFinderNullDailySchedule(
  KHE_TASK_FINDER tf, int first_day_index, int last_day_index);

extern void KheDailyScheduleDelete(KHE_DAILY_SCHEDULE ds);

extern KHE_TASK_FINDER KheDailyScheduleTaskFinder(KHE_DAILY_SCHEDULE ds);
extern bool KheDailyScheduleNoOverlap(KHE_DAILY_SCHEDULE ds);
extern int KheDailyScheduleFirstDayIndex(KHE_DAILY_SCHEDULE ds);
extern int KheDailyScheduleLastDayIndex(KHE_DAILY_SCHEDULE ds);
extern KHE_TASK KheDailyScheduleTask(KHE_DAILY_SCHEDULE ds, int day_index);
extern KHE_TIME KheDailyScheduleTime(KHE_DAILY_SCHEDULE ds, int day_index);
/* ***
extern bool KheTaskFinderTaskIntervalAndTimes(KHE_TASK_FINDER tf,
  KHE_TASK task, int *first_index, int *last_index,
  ARRAY_KHE_TIME *times_by_day, ARRAY_KHE_TASK *tasks_by_day);
extern bool KheTaskFinderTaskSetIntervalAndTimes(KHE_TASK_FINDER tf,
  KHE_TASK_SET ts, int *first_index, int *last_index,
  ARRAY_KHE_TIME *times_by_day);
extern bool KheTaskFinderTimeGroupIntervalAndTimes(KHE_TASK_FINDER tf,
  KHE_TIME_GROUP tg, int *first_index, int *last_index,
  ARRAY_KHE_TIME *times_by_day);
*** */

/* ***
extern void KheTaskFinderTaskIntervalAndTimes(KHE_TASK_FINDER tf,
  KHE_TASK task, int *first_index, int *last_index,
  KHE_TIME *first_time, KHE_TIME *last_time);
extern void KheTaskFinderTaskSetIntervalAndTimes(KHE_TASK_FINDER tf,
  KHE_TASK_SET ts, int *first_index, int *last_index,
  KHE_TIME *first_time, KHE_TIME *last_time);
extern void KheTaskFinderTimeGroupIntervalAndTimes(KHE_TASK_FINDER tf,
  KHE_TIME_GROUP tg, int *first_index, int *last_index,
  KHE_TIME *first_time, KHE_TIME *last_time);
*** */

extern void KheFindTasksInInterval(KHE_TASK_FINDER tf, int first_index,
  int last_index, KHE_RESOURCE_TYPE rt, KHE_RESOURCE from_r,
  bool ignore_preassigned, bool ignore_partial,
  KHE_TASK_SET res_ts, int *res_first_index, int *res_last_index);
extern bool KheFindFirstRunInInterval(KHE_TASK_FINDER tf, int first_index,
  int last_index, KHE_RESOURCE_TYPE rt, KHE_RESOURCE from_r,
  bool ignore_preassigned, bool ignore_partial, bool sep_need_asst,
  KHE_TASK_SET res_ts, int *res_first_index, int *res_last_index);
extern bool KheFindLastRunInInterval(KHE_TASK_FINDER tf, int first_index,
  int last_index, KHE_RESOURCE_TYPE rt, KHE_RESOURCE from_r,
  bool ignore_preassigned, bool ignore_partial, bool sep_need_asst,
  KHE_TASK_SET res_ts, int *res_first_index, int *res_last_index);

/* 11.10.2 Widened task sets */
extern bool KheWidenedTaskSetMake(KHE_TASK_FINDER tf, KHE_RESOURCE from_r,
  KHE_TASK_SET from_r_ts, int max_left_wing_count,
  int max_right_wing_count, KHE_WIDENED_TASK_SET *wts);
extern bool KheWidenedTaskSetMakeFlexible(KHE_TASK_FINDER tf,
  KHE_RESOURCE from_r, KHE_TASK_SET from_r_ts,
  int max_wing_count, KHE_WIDENED_TASK_SET *wts);
extern void KheWidenedTaskSetDelete(KHE_WIDENED_TASK_SET wts);

extern void KheWidenedTaskSetInterval(KHE_WIDENED_TASK_SET wts,
  int left_count, int right_count, int *first_index, int *last_index);
extern void KheWidenedTaskSetFullInterval(KHE_WIDENED_TASK_SET wts,
  int *first_index, int *last_index);

extern bool KheFindMovableWidenedTaskSetRight(KHE_TASK_FINDER tf,
  KHE_RESOURCE from_r, KHE_RESOURCE to_r, int days_first_index,
  KHE_WIDENED_TASK_SET *res_wts);
extern bool KheFindMovableWidenedTaskSetLeft(KHE_TASK_FINDER tf,
  KHE_RESOURCE from_r, KHE_RESOURCE to_r, int days_last_index,
  KHE_WIDENED_TASK_SET *res_wts);

extern void KheWidenedTaskSetDebug(KHE_WIDENED_TASK_SET wts,
  int left_count, int right_count, int verbosity, int indent, FILE *fp);

/* 11.10.3 Widened task set moves */
extern bool KheWidenedTaskSetMoveCheck(KHE_WIDENED_TASK_SET wts,
  KHE_RESOURCE to_r, bool force, int *max_left_count, int *max_right_count);
extern bool KheWidenedTaskSetMove(KHE_WIDENED_TASK_SET wts,
  KHE_RESOURCE to_r, int left_count, int right_count,
  int *from_r_durn_change, int *to_r_durn_change);
extern void KheWidenedTaskSetMoveDebug(KHE_WIDENED_TASK_SET wts,
  KHE_RESOURCE to_r, int left_count, int right_count,
  int verbosity, int indent, FILE *fp);

extern bool KheWidenedTaskSetMovePartial(KHE_WIDENED_TASK_SET wts,
  KHE_RESOURCE to_r, int first_index, int last_index);
extern void KheWidenedTaskSetMovePartialDebug(KHE_WIDENED_TASK_SET wts,
  KHE_RESOURCE to_r, int first_index, int last_index,
  int verbosity, int indent, FILE *fp);

extern bool KheWidenedTaskSetFindInitial(KHE_WIDENED_TASK_SET wts,
  int wanted_durn, int *first_index, int *last_index);
extern bool KheWidenedTaskSetFindFinal(KHE_WIDENED_TASK_SET wts,
  int wanted_durn, int *first_index, int *last_index);

/* 11.10.4 Widened task set swaps */
extern bool KheWidenedTaskSetSwapCheck(KHE_WIDENED_TASK_SET wts,
  KHE_RESOURCE to_r, bool exact, KHE_TIME_GROUP blocking_tg,
  KHE_MONITOR blocking_m, int *max_left_count, int *max_right_count);
extern bool KheWidenedTaskSetSwap(KHE_WIDENED_TASK_SET wts,
  KHE_RESOURCE to_r, int left_count, int right_count,
  int *from_r_durn_change, int *to_r_durn_change);
extern void KheWidenedTaskSetSwapDebug(KHE_WIDENED_TASK_SET wts,
  KHE_RESOURCE to_r, int left_count, int right_count,
  int verbosity, int indent, FILE *fp);

/* 11.10.4 Widened task set optimal moves */
extern bool KheWidenedTaskSetOptimalMoveCheck(KHE_WIDENED_TASK_SET wts,
  KHE_RESOURCE to_r, KHE_TIME_GROUP blocking_tg, KHE_MONITOR blocking_m);
extern bool KheWidenedTaskSetOptimalMove(KHE_WIDENED_TASK_SET wts,
  KHE_RESOURCE to_r, int *from_r_durn_change, int *to_r_durn_change);
extern void KheWidenedTaskSetOptimalMoveDebug(KHE_WIDENED_TASK_SET wts,
  KHE_RESOURCE to_r, int verbosity, int indent, FILE *fp);

/* ***
extern bool KheFindTaskRunInitial(KHE_TASK_FINDER tf, KHE_TASK_SET ts,
  int wanted_durn, KHE_TASK_SET res_ts);
extern bool KheFindTaskRunFinal(KHE_TASK_FINDER tf, KHE_TASK_SET ts,
  int wanted_durn, KHE_TASK_SET res_ts);
*** */

/* 11.11 Other task tree solvers */
extern void KheTaskingAllowSplitAssignments(KHE_TASKING tasking,
  bool unassigned_only);
extern void KheTaskingEnlargeDomains(KHE_TASKING tasking, bool unassigned_only);

/* 11.12 Task groups */
extern KHE_TASK_GROUPS KheTaskGroupsMakeFromTasking(KHE_TASKING tasking);
extern void KheTaskGroupsDelete(KHE_TASK_GROUPS task_groups);
extern int KheTaskGroupsTaskGroupCount(KHE_TASK_GROUPS task_groups);
extern KHE_TASK_GROUP KheTaskGroupsTaskGroup(KHE_TASK_GROUPS task_groups,
  int i);
extern int KheTaskGroupTaskCount(KHE_TASK_GROUP task_group);
extern KHE_TASK KheTaskGroupTask(KHE_TASK_GROUP task_group, int i);

extern int KheTaskGroupTotalDuration(KHE_TASK_GROUP task_group);
extern float KheTaskGroupTotalWorkload(KHE_TASK_GROUP task_group);
extern KHE_RESOURCE_GROUP KheTaskGroupDomain(KHE_TASK_GROUP task_group);
extern int KheTaskGroupDecreasingDurationCmp(KHE_TASK_GROUP tg1,
  KHE_TASK_GROUP tg2);

extern int KheTaskGroupUnassignedTaskCount(KHE_TASK_GROUP task_group);
extern bool KheTaskGroupAssignCheck(KHE_TASK_GROUP task_group, KHE_RESOURCE r);
extern bool KheTaskGroupAssign(KHE_TASK_GROUP task_group, KHE_RESOURCE r);
extern void KheTaskGroupUnAssign(KHE_TASK_GROUP task_group, KHE_RESOURCE r);

extern void KheTaskGroupDebug(KHE_TASK_GROUP task_group, int verbosity,
  int indent, FILE *fp);
extern void KheTaskGroupsDebug(KHE_TASK_GROUPS task_groups, int verbosity,
  int indent, FILE *fp);


/*****************************************************************************/
/*                                                                           */
/*    Chapter 12.  Resource Solvers                                          */
/*                                                                           */
/*****************************************************************************/

/* 12.2 The resource assignment invariant */
extern void KheAtomicOperationBegin(KHE_SOLN soln, KHE_MARK *mark,
  int *init_count, bool resource_invariant);
extern bool KheAtomicOperationEnd(KHE_SOLN soln, KHE_MARK *mark,
  int *init_count, bool resource_invariant, bool success);
extern void KheDisconnectAllDemandMonitors(KHE_SOLN soln, KHE_RESOURCE_TYPE rt);

/* 12.3 Ordinary, ejecting, and Kempe task and task set moves */
/* extern bool KheTaskSetMove(KHE_TASK_SET ts, KHE_RESOURCE r); */
extern bool KheEjectingTaskMove(KHE_TASK task, KHE_RESOURCE r,
  bool allow_eject);
extern bool KheEjectingTaskMoveFrame(KHE_TASK task, KHE_RESOURCE r,
  bool allow_eject, KHE_FRAME frame);
extern bool KheEjectingTaskSetMove(KHE_TASK_SET ts, KHE_RESOURCE r,
  bool allow_eject);
extern bool KheEjectingTaskSetMoveFrame(KHE_TASK_SET ts, KHE_RESOURCE r,
  bool allow_eject, KHE_FRAME frame);

extern bool KheKempeTaskMove(KHE_TASK task, KHE_RESOURCE r);
extern bool KheKempeTaskMoveFrame(KHE_TASK task, KHE_RESOURCE r,
  KHE_FRAME frame);
extern bool KheKempeTaskSetMove(KHE_TASK_SET ts, KHE_RESOURCE r);
extern bool KheKempeTaskSetMoveFrame(KHE_TASK_SET ts, KHE_RESOURCE r,
  KHE_FRAME frame);

extern bool KheTypedTaskMove(KHE_TASK task, KHE_RESOURCE r, KHE_MOVE_TYPE mt);
extern bool KheTypedTaskMoveFrame(KHE_TASK task, KHE_RESOURCE r,
  KHE_MOVE_TYPE mt, KHE_FRAME frame);
extern bool KheTypedTaskSetMove(KHE_TASK_SET ts, KHE_RESOURCE r,
  KHE_MOVE_TYPE mt);
extern bool KheTypedTaskSetMoveFrame(KHE_TASK_SET ts, KHE_RESOURCE r,
  KHE_MOVE_TYPE mt, KHE_FRAME frame);
extern char *KheMoveTypeShow(KHE_MOVE_TYPE mt);

/* 12.4 Frame operations for resource solvers - obsolete */
/* ***
extern bool KheFrameTaskSet(KHE_FRAME frame, KHE_RESOURCE r, KHE_TASK_SET *ts);

extern void KheFrameSetVisitNum(KHE_FRAME frame, KHE_RESOURCE r, int num);
extern int KheFrameVisitNum(KHE_FRAME frame, KHE_RESOURCE r);
extern bool KheFrameVisited(KHE_FRAME frame, KHE_RESOURCE r, int slack);
extern void KheFrameVisit(KHE_FRAME frame, KHE_RESOURCE r);
extern void KheFrameUnVisit(KHE_FRAME frame, KHE_RESOURCE r);
*** */

/* ***
extern bool KheFrameIsActive(KHE_FRAME frame, KHE_RESOURCE r);
extern bool KheFrameIsInactive(KHE_FRAME frame, KHE_RESOURCE r);
*** */

/* ***
extern KHE_FRAME KheFrameActiveAtLeft(KHE_FRAME frame, KHE_RESOURCE r);
extern KHE_FRAME KheFrameInactiveAtLeft(KHE_FRAME frame, KHE_RESOURCE r);
extern KHE_FRAME KheFrameActiveAtRight(KHE_FRAME frame, KHE_RESOURCE r);
extern KHE_FRAME KheFrameInactiveAtRight(KHE_FRAME frame, KHE_RESOURCE r);
*** */

/* ***
extern void KheFrameIteratorInit(KHE_FRAME_ITERATOR fi, KHE_FRAME frame,
  KHE_RESOURCE r, int extra);
extern bool KheFrameIteratorNext(KHE_FRAME_ITERATOR fi, KHE_FRAME *res);
extern bool KheFrameIteratorNextPair(KHE_FRAME_ITERATOR fi, KHE_FRAME *res);
*** */

/* 12.4.1 Satisfying requested task assignments */
extern bool KheSolnAssignRequestedResources(KHE_SOLN soln,
  KHE_RESOURCE_TYPE rt, KHE_OPTIONS options);
extern bool KheMonitorRequestsSpecificBusyTimes(KHE_MONITOR m);

/* 12.4.2 Most-constrained-first assignment */
extern bool KheMostConstrainedFirstAssignResources(KHE_TASKING tasking,
  KHE_OPTIONS options);

/* 12.4.3 Resource packing */
extern bool KheResourcePackAssignResources(KHE_TASKING tasking,
  KHE_OPTIONS options);

/* 12.4.4 Consecutive packing */
/* ***
extern bool KheResourcePackConsecutive(KHE_TASKING tasking,
  KHE_OPTIONS options);
*** */

/* 12.4.4 Split assignments */
extern bool KheFindSplitResourceAssignments(KHE_TASKING tasking,
  KHE_OPTIONS options);

/* 12.6.1 A solver for resource matching */
extern KHE_RESOURCE_MATCHING_SOLVER KheResourceMatchingSolverMake(
  KHE_SOLN soln, KHE_RESOURCE_GROUP rg, HA_ARENA a);
extern void KheResourceMatchingSolverDelete(KHE_RESOURCE_MATCHING_SOLVER rms);

extern bool KheResourceMatchingSolverSolve(KHE_RESOURCE_MATCHING_SOLVER rms,
  KHE_RESOURCE_MATCHING_DEMAND_SET rmds, bool edge_adjust1_off,
  bool edge_adjust2_off, bool edge_adjust3_off, bool edge_adjust4_off,
  bool ejection_off, /* bool nocost_off, */ KHE_OPTIONS options);

extern KHE_RESOURCE_MATCHING_DEMAND_SET KheResourceMatchingDemandSetMake(
  KHE_RESOURCE_MATCHING_SOLVER rms, bool preserve_existing);
extern void KheResourceMatchingDemandSetAddTime(
  KHE_RESOURCE_MATCHING_DEMAND_SET rmds, KHE_TIME t);
extern void KheResourceMatchingDemandSetAddTimeGroup(
  KHE_RESOURCE_MATCHING_DEMAND_SET rmds, KHE_TIME_GROUP tg);
extern void KheResourceMatchingDemandSetAddFrame(
  KHE_RESOURCE_MATCHING_DEMAND_SET rmds, KHE_FRAME frame);
extern void KheResourceMatchingDemandSetDelete(
  KHE_RESOURCE_MATCHING_DEMAND_SET rmds);
extern void KheResourceMatchingDemandSetClear(
  KHE_RESOURCE_MATCHING_DEMAND_SET rmds);

/* 12.6.3 Time sweep resource assignment */
extern bool KheTimeSweepAssignResources(KHE_SOLN soln, KHE_RESOURCE_GROUP rg,
  KHE_OPTIONS options);

/* 12.6.4 Time sweep with lookahead */
extern bool KheResourceMatchingSolverSolveWithLookahead(
  KHE_RESOURCE_MATCHING_SOLVER rms,
  ARRAY_KHE_RESOURCE_MATCHING_DEMAND_SET *rmds_array,
  int first_index, int last_index, bool edge_adjust1_off,
  bool edge_adjust2_off, bool edge_adjust3_off, bool edge_adjust4_off,
  bool ejection_off, /* bool nocost_off, */ KHE_OPTIONS options);

/* 12.6.5 Resource rematching repair */
extern bool KheResourceRematch(KHE_SOLN soln, KHE_RESOURCE_GROUP rg,
  KHE_OPTIONS options, int variant);

/* 12.7 Single resource assignment */
extern KHE_SINGLE_RESOURCE_SOLVER KheSingleResourceSolverMake(KHE_SOLN soln,
  KHE_OPTIONS options);
extern void KheSingleResourceSolverDelete(KHE_SINGLE_RESOURCE_SOLVER srs);
extern void KheSingleResourceSolverSolve(KHE_SINGLE_RESOURCE_SOLVER srs,
  KHE_RESOURCE r, KHE_SRS_DOM_KIND dom_kind, int min_assts, int max_assts,
  KHE_COST cost_limit);
/* ***
extern void KheSingleResourceSolverAddCostCutoff(KHE_SINGLE_RESOURCE_SOLVER srs,
  KHE_COST cost_cutoff);
*** */
extern int KheSingleResourceSolverTimetableCount(
  KHE_SINGLE_RESOURCE_SOLVER srs);
extern void KheSingleResourceSolverTimetable(KHE_SINGLE_RESOURCE_SOLVER srs,
  int i, int *asst_count, KHE_COST *r_cost /* , KHE_COST *soln_cost */);
extern void KheSingleResourceSolverAdopt(KHE_SINGLE_RESOURCE_SOLVER srs,
  int i);
extern void KheSingleResourceSolverDebug(KHE_SINGLE_RESOURCE_SOLVER srs,
  int verbosity, int indent, FILE *fp);
extern void KheSingleResourceSolverTest(KHE_SOLN soln, KHE_OPTIONS options,
  KHE_RESOURCE r);

/* 12.8 Resource pair repair */
extern bool KheResourcePairReassign(KHE_SOLN soln, KHE_RESOURCE r1,
  KHE_RESOURCE r2, bool resource_invariant, bool fix_splits);
extern bool KheResourcePairRepair(KHE_TASKING tasking, KHE_OPTIONS options);

/* *** replaced by resource reassign repair
extern bool KheResourcePairSimpleReassign(KHE_SOLN soln,
  KHE_RESOURCE r1, KHE_RESOURCE r2, KHE_FRAME frame, int fi, int li,
  bool resource_invariant, int max_assignments);
extern bool KheResourcePairSimpleRepair(KHE_SOLN soln, KHE_OPTIONS options);
extern bool KheResourcePairSimpleBusyRepair(KHE_SOLN soln, KHE_OPTIONS options);

extern bool KheResourceTripleSimpleReassign(KHE_SOLN soln,
  KHE_RESOURCE r1, KHE_RESOURCE r2, KHE_RESOURCE r3, KHE_FRAME frame,
  int fi, int li, bool resource_invariant, int max_assignments);
extern bool KheResourceTripleSimpleRepair(KHE_SOLN soln, KHE_OPTIONS options);

extern bool KheResourcePairRunReassign(KHE_SOLN soln, KHE_OPTIONS options,
  KHE_RESOURCE r1, KHE_RESOURCE r2, int fi, int li,
  bool resource_invariant, int max_assignments);
extern bool KheResourcePairRunRepair(KHE_SOLN soln, KHE_OPTIONS options);

extern bool KheResourcePairSwap(KHE_SOLN soln, KHE_RESOURCE r1,
  KHE_RESOURCE r2, KHE_FRAME frame, int fi, int li, bool resource_invariant);
extern bool KheResourcePairSwapRepair(KHE_SOLN soln, KHE_OPTIONS options);

extern void KheResourcePairAllSwapRepair(KHE_SOLN soln, KHE_RESOURCE_TYPE rt,
  KHE_OPTIONS options);
*** */


/* 12.8 Resource run repair */
/* *** replaced by resource reassign repair
extern KHE_RESOURCE_RUN_SOLVER KheResourceRunSolverMake(KHE_SOLN soln,
  KHE_OPTIONS options, bool resource_invariant, int max_assignments);
extern void KheResourceRunSolverDelete(KHE_RESOURCE_RUN_SOLVER rrs);
extern bool KheResourceRunSolverAddResource(KHE_RESOURCE_RUN_SOLVER rrs,
  KHE_RESOURCE r);
extern bool KheResourceRunSolverDeleteResource(KHE_RESOURCE_RUN_SOLVER rrs,
  KHE_RESOURCE r);
extern void KheResourceRunSolverClearResources(KHE_RESOURCE_RUN_SOLVER rrs);
extern bool KheResourceRunSolverSolve(KHE_RESOURCE_RUN_SOLVER rrs,
  int fi, int li);
extern bool KheResourceRunRepair(KHE_SOLN soln, KHE_RESOURCE_TYPE rt,
  KHE_OPTIONS options);
*** */

/* 12.9 Resource reassign repair */
extern KHE_REASSIGN_SOLVER KheReassignSolverMake(KHE_SOLN soln,
  KHE_RESOURCE_TYPE rt, KHE_OPTIONS options);
extern void KheReassignSolverDelete(KHE_REASSIGN_SOLVER rs);
extern bool KheReassignSolverAddResource(KHE_REASSIGN_SOLVER rs,
  KHE_RESOURCE r);
extern bool KheReassignSolverDeleteResource(KHE_REASSIGN_SOLVER rs,
  KHE_RESOURCE r);
extern void KheReassignSolverClearResources(KHE_REASSIGN_SOLVER rs);
extern bool KheReassignSolverSolve(KHE_REASSIGN_SOLVER rs, int first_index,
  int last_index, KHE_REASSIGN_GROUPING grouping, bool ignore_partial,
  KHE_REASSIGN_METHOD method, int max_assignments);
extern void KheReassignSolverDebug(KHE_REASSIGN_SOLVER rs,
  int verbosity, int indent, FILE *fp);
extern bool KheReassignRepair(KHE_SOLN soln, KHE_RESOURCE_TYPE rt,
  KHE_OPTIONS options);
extern bool KheReassign2Repair(KHE_SOLN soln, KHE_RESOURCE_TYPE rt,
  KHE_OPTIONS options);
extern bool KheReassign3Repair(KHE_SOLN soln, KHE_RESOURCE_TYPE rt,
  KHE_OPTIONS options);

/* 12.10 Trying unassignments */
extern bool KheSolnTryTaskUnAssignments(KHE_SOLN soln, KHE_OPTIONS options);

/* 12.11 Putting it all together */
extern bool KheTaskingAssignResourcesStage1(KHE_TASKING tasking,
  KHE_OPTIONS options);
extern bool KheTaskingAssignResourcesStage2(KHE_TASKING tasking,
  KHE_OPTIONS options);
extern bool KheTaskingAssignResourcesStage3(KHE_TASKING tasking,
  KHE_OPTIONS options);
/* extern void KheAssignResourcesGlobalCheck(KHE_SOLN soln, char *pos); */


/*****************************************************************************/
/*                                                                           */
/*    Chapter 13.  Ejection Chains                                           */
/*                                                                           */
/*****************************************************************************/

/* 13.2 Ejector construction */
extern KHE_EJECTOR KheEjectorMakeBegin(char *schedules, HA_ARENA a);
extern void KheEjectorMakeEnd(KHE_EJECTOR ej);
extern char *KheEjectorSchedules(KHE_EJECTOR ej);
extern HA_ARENA KheEjectorArena(KHE_EJECTOR ej);

extern void KheEjectorAddAugment(KHE_EJECTOR ej, KHE_MONITOR_TAG tag,
  KHE_EJECTOR_AUGMENT_FN augment_fn, int augment_type);
extern void KheEjectorAddGroupAugment(KHE_EJECTOR ej, int sub_tag,
  KHE_EJECTOR_AUGMENT_FN augment_fn, int augment_type);

/* 13.2 Ejector construction - major schedules */
/* ***
extern int KheEjectorMajorScheduleCount(KHE_EJECTOR ej);
extern KHE_EJECTOR_MAJOR_SCHEDULE KheEjectorMajorSchedule(KHE_EJECTOR ej,
  int i);
*** */

/* ***
extern void KheEjectorMajorScheduleBegin(KHE_EJECTOR ej);
extern void KheEjectorMajorScheduleEnd(KHE_EJECTOR ej);
extern int KheEjectorMajorScheduleMinorScheduleCount(
  KHE_EJECTOR_MAJOR_SCHEDULE ejm);
extern KHE_EJECTOR_MINOR_SCHEDULE KheEjectorMajorScheduleMinorSchedule(
  KHE_EJECTOR_MAJOR_SCHEDULE ejm, int i);
*** */

/* 13.2 Ejector construction - minor schedules */
/* ***
extern void KheEjectorMinorScheduleAdd(KHE_EJECTOR ej,
  int max_depth, bool may_revisit);
extern int KheEjectorMinorScheduleMaxLength(KHE_EJECTOR_MINOR_SCHEDULE ejms);
extern bool KheEjectorMinorScheduleMayRevisit(KHE_EJECTOR_MINOR_SCHEDULE ejms);
*** */

/* 13.2 Ejector construction - other schedules */
/* ***
extern void KheEjectorAddDefaultSchedules(KHE_EJECTOR ej);
extern void KheEjectorSetSchedulesFromString(KHE_EJECTOR ej,
  char *ejector_schedules_string);
*** */

/* 13.3 Ejector solving */
extern bool KheEjectorSolve(KHE_EJECTOR ej, KHE_GROUP_MONITOR start_gm,
  KHE_GROUP_MONITOR continue_gm, KHE_OPTIONS options);
extern void KheEjectorSolveBegin(KHE_EJECTOR ej, KHE_GROUP_MONITOR start_gm,
  KHE_GROUP_MONITOR continue_gm, KHE_OPTIONS options);
extern bool KheEjectorSolveEnd(KHE_EJECTOR ej);

/* 13.3 Ejector solving - cost limits */
extern void KheEjectorAddMonitorCostLimit(KHE_EJECTOR ej,
  KHE_MONITOR m, KHE_COST cost_limit);
extern void KheEjectorAddMonitorCostLimitReducing(KHE_EJECTOR ej,
  KHE_MONITOR m);
extern int KheEjectorMonitorCostLimitCount(KHE_EJECTOR ej);
extern void KheEjectorMonitorCostLimit(KHE_EJECTOR ej, int i,
  KHE_MONITOR *m, KHE_COST *cost_limit, bool *reducing);

/* 13.3 Ejector solving - queries */
extern KHE_GROUP_MONITOR KheEjectorStartGroupMonitor(KHE_EJECTOR ej);
extern KHE_GROUP_MONITOR KheEjectorContinueGroupMonitor(KHE_EJECTOR ej);
extern KHE_OPTIONS KheEjectorOptions(KHE_EJECTOR ej);
extern KHE_FRAME KheEjectorFrame(KHE_EJECTOR ej);
extern KHE_EVENT_TIMETABLE_MONITOR KheEjectorEventTimetableMonitor(
  KHE_EJECTOR ej);
extern KHE_TASK_FINDER KheEjectorTaskFinder(KHE_EJECTOR ej);
extern KHE_SOLN KheEjectorSoln(KHE_EJECTOR ej);
extern KHE_COST KheEjectorTargetCost(KHE_EJECTOR ej);
/* ***
extern KHE_EJECTOR_MAJOR_SCHEDULE KheEjectorCurrMajorSchedule(KHE_EJECTOR ej);
extern KHE_EJECTOR_MINOR_SCHEDULE KheEjectorCurrMinorSchedule(KHE_EJECTOR ej);
*** */
extern bool KheEjectorCurrMayRevisit(KHE_EJECTOR ej);
extern int KheEjectorCurrLength(KHE_EJECTOR ej);
extern int KheEjectorCurrAugmentCount(KHE_EJECTOR ej);
extern bool KheEjectorCurrDebug(KHE_EJECTOR ej);
extern int KheEjectorCurrDebugIndent(KHE_EJECTOR ej);

/* 13.4 Ejector solving - repairs */
extern void KheEjectorRepairBegin(KHE_EJECTOR ej);
/* ***
extern bool KheEjectorRepairEndLong(KHE_EJECTOR ej, int repair_type,
  bool success,
  void (*on_success_fn)(void *on_success_val), void *on_success_val);
*** */
extern bool KheEjectorRepairEnd(KHE_EJECTOR ej, int repair_type, bool success);

/* 13.6.3 Statistics describing a single solve */
extern int KheEjectorImprovementCount(KHE_EJECTOR ej);
extern int KheEjectorImprovementNumberOfRepairs(KHE_EJECTOR ej, int i);
extern float KheEjectorImprovementTime(KHE_EJECTOR ej, int i);
extern KHE_COST KheEjectorImprovementCost(KHE_EJECTOR ej, int i);
extern int KheEjectorImprovementDefects(KHE_EJECTOR ej, int i);

extern KHE_COST KheEjectorInitCost(KHE_EJECTOR ej);
extern int KheEjectorInitDefects(KHE_EJECTOR ej);

/* 13.6.4 Statistics describing multiple solves */
extern int KheEjectorImprovementRepairHistoMax(KHE_EJECTOR ej);
extern int KheEjectorImprovementRepairHistoFrequency(KHE_EJECTOR ej,
  int repair_count);
extern int KheEjectorImprovementRepairHistoTotal(KHE_EJECTOR ej);
extern float KheEjectorImprovementRepairHistoAverage(KHE_EJECTOR ej);

extern int KheEjectorImprovementAugmentHistoMax(KHE_EJECTOR ej);
extern int KheEjectorImprovementAugmentHistoFrequency(KHE_EJECTOR ej,
  int augment_count);
extern int KheEjectorImprovementAugmentHistoTotal(KHE_EJECTOR ej);
extern float KheEjectorImprovementAugmentHistoAverage(KHE_EJECTOR ej);

extern int KheEjectorTotalRepairs(KHE_EJECTOR ej, int augment_type);
extern int KheEjectorTotalRepairsByType(KHE_EJECTOR ej, int augment_type,
  int repair_type);
extern int KheEjectorSuccessfulRepairs(KHE_EJECTOR ej, int augment_type);
extern int KheEjectorSuccessfulRepairsByType(KHE_EJECTOR ej, int augment_type,
  int repair_type);

extern void KheEjectorAddAugmentType(KHE_EJECTOR ej, int augment_type,
  char *augment_label);
extern int KheEjectorAugmentTypeCount(KHE_EJECTOR ej);
extern int KheEjectorAugmentType(KHE_EJECTOR ej, int i);
extern char *KheEjectorAugmentTypeLabel(KHE_EJECTOR ej, int augment_type);

extern void KheEjectorAddRepairType(KHE_EJECTOR ej, int repair_type,
  char *repair_label);
extern int KheEjectorRepairTypeCount(KHE_EJECTOR ej);
extern int KheEjectorRepairType(KHE_EJECTOR ej, int i);
extern char *KheEjectorRepairTypeLabel(KHE_EJECTOR ej, int repair_type);

/* 13.7 Ejection chain time and resource repair functions */
extern bool KheEjectionChainNodeRepairTimes(KHE_NODE parent_node,
  KHE_OPTIONS options);
extern bool KheEjectionChainLayerRepairTimes(KHE_LAYER layer,
  KHE_OPTIONS options);
extern bool KheEjectionChainRepairResources(KHE_TASKING tasking,
  KHE_OPTIONS options);
extern bool KheEjectionChainRepairInitialResourceAssignment(
  KHE_GROUP_MONITOR limit_resources_gm, KHE_OPTIONS options);

extern KHE_EJECTOR KheEjectionChainEjectorMake(KHE_OPTIONS options,
  HA_ARENA a);
extern KHE_EJECTOR KheEjectionChainEjectorOption(KHE_OPTIONS options,
  char *key);

extern void KheEjectionChainPrepareMonitors(KHE_SOLN soln, KHE_OPTIONS options);
extern void KheEjectionChainUnPrepareMonitors(KHE_SOLN soln);

extern KHE_GROUP_MONITOR KheNodeTimeRepairStartGroupMonitorMake(KHE_NODE node);
extern KHE_GROUP_MONITOR KheLayerTimeRepairStartGroupMonitorMake(
  KHE_LAYER layer);
extern KHE_GROUP_MONITOR KheLayerTimeRepairLongStartGroupMonitorMake(
  KHE_LAYER layer);
extern KHE_GROUP_MONITOR KheTaskingStartGroupMonitorMake(KHE_TASKING tasking);
extern KHE_GROUP_MONITOR KheGroupEventMonitors(KHE_SOLN soln,
  KHE_MONITOR_TAG tag, KHE_SUBTAG_STANDARD_TYPE sub_tag);
extern KHE_GROUP_MONITOR KheAllDemandGroupMonitorMake(KHE_SOLN soln);

#endif
