
/*****************************************************************************/
/*                                                                           */
/*  THE KHE HIGH SCHOOL TIMETABLING ENGINE                                   */
/*  COPYRIGHT (C) 2010 Jeffrey H. Kingston                                   */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 3, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         khe_sr_combined.c                                          */
/*  DESCRIPTION:  Combined resource solvers                                  */
/*                                                                           */
/*****************************************************************************/
#include "khe_solvers.h"
#include <limits.h>

#define DEBUG1 0	/* entry and exit of each main call */
#define DEBUG2 0	/* stages within each main call     */
#define DEBUG3 0	/* monitor debug within repair      */
#define DEBUG4 0	/* KheSolnCheckTasks                */
#define DEBUG5 0	/* repair steps                     */
#define DEBUG6 0
#define DEBUG7 0	/* print timetables for DEBUG7_ID   */
#define DEBUG7_ID "HN_2"
#define DEBUG8 0	/* clash checking */
/* ***
#define DEBUG7 1
#define DEBUG7_ID "DemandConstraint:18A/1Wed:E"
*** */
/* ***
#define DEBUG8_ID "A10"	** resource to print timetables for **
*** */

#define DEBUG9 0
#define DEBUG10 1


/*****************************************************************************/
/*                                                                           */
/*  Submodule "checks"                                                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheDebugTimetable(KHE_SOLN soln, char *header)                      */
/*                                                                           */
/*  If DEBUG7, print the timetable of resource DEBUG7_ID, plus header.       */
/*                                                                           */
/*****************************************************************************/

static void KheDebugTimetable(KHE_SOLN soln, char *header)
{
  KHE_RESOURCE_TIMETABLE_MONITOR rtm;  KHE_INSTANCE ins;  KHE_RESOURCE r;
  if( DEBUG7 )
  {
    ins = KheSolnInstance(soln);
    if( KheInstanceRetrieveResource(ins, DEBUG7_ID, &r) )
    {
      fprintf(stderr, "  [ %s timetable %s\n", KheResourceId(r), header);
      rtm = KheResourceTimetableMonitor(soln, r);
      KheResourceTimetableMonitorPrintTimetable(rtm, 10, 4, stderr);
      fprintf(stderr, "  ]\n");
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSolnCheckTasks(KHE_SOLN soln, char *label)                       */
/*                                                                           */
/*  Check soln for inconsistent tasks; if found, print an error message      */
/*  including label.                                                         */
/*                                                                           */
/*****************************************************************************/

static void KheSolnCheckTasks(KHE_SOLN soln, char *label)
{
  int i;  KHE_TASK task;  KHE_RESOURCE ass_r, pre_r;  KHE_EVENT e;
  char *role;  KHE_EVENT_RESOURCE er;  KHE_INSTANCE ins;
  for( i = 0;  i < KheSolnTaskCount(soln);  i++ )
  {
    task = KheSolnTask(soln, i);

    /* only interested in tasks derived from event resources */
    er = KheTaskEventResource(task);
    if( er == NULL )
      continue;

    /* get assigned and preassigned resources and check that they have Ids */
    role = KheEventResourceRole(er);
    e = KheEventResourceEvent(er);
    ass_r = KheTaskAsstResource(task);
    pre_r = KheEventResourcePreassignedResource(er);
    ins = KheEventInstance(e);
    HnAssert(ass_r == NULL || KheResourceId(ass_r) != NULL,
      "KheSolnCheckTasks %s: in instance %s, resource without Id assigned to "
      "event %s (role %s)", label, KheInstanceId(ins), KheEventId(e),
      role == NULL ? "(none)" : role);
    HnAssert(pre_r == NULL || KheResourceId(pre_r) != NULL,
      "KheSolnCheckTasks %s: in instance %s, resource without Id preassigned to"
      " event %s (role %s)", label, KheInstanceId(ins), KheEventId(e),
      role == NULL ? "(none)" : role);

    if( pre_r != NULL )
    {
      /* if preassigned, check that the assigned resource is equal to it */
      HnAssert(ass_r != NULL,
	"KheSolnCheckTasks %s: in event %s of instance %s, event resource with "
	"preassigned resource %s has task with missing resource assignment",
	label, KheEventId(e), KheInstanceId(ins), KheResourceId(pre_r));
      HnAssert(ass_r == pre_r,
	"KheSolnCheckTasks %s: in event %s of instance %s, event resource with "
	"preassigned resource %s has task with inconsistent assignment %s",
	label, KheEventId(e), KheInstanceId(ins), KheResourceId(pre_r),
	KheResourceId(ass_r));
    }
    else
    {
      /* if unpreassigned, must have a role */
      HnAssert(role != NULL, "KheSolnCheckTasks %s: in event %s of instance %s, "
	"unpreassigned event resource has no role", label, KheEventId(e),
	KheInstanceId(ins));
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheNoClashesCheck(KHE_SOLN soln, KHE_RESOURCE r)                    */
/*                                                                           */
/*  Make sure that r has no clashes.                                         */
/*                                                                           */
/*****************************************************************************/

/* *** withdrawn to save the cost of RTMClashingTimeCount
static void KheNoClashesCheck(KHE_SOLN soln, KHE_RESOURCE r)
{
  KHE_RESOURCE_TIMETABLE_MONITOR rtm;
  if( r != NULL )
  {
    rtm = KheResourceTimetableMonitor(soln, r);
    HnAssert(KheResourceTimetableMonitorClashingTimeCount(rtm) == 0,
      "KheNoClashesCheck failed on resource %s", KheResourceId(r));
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheSolnCheckForClashes(KHE_SOLN soln)                               */
/*                                                                           */
/*  Check for clashes.                                                       */
/*                                                                           */
/*****************************************************************************/

/* *** withdrawn to save the cost of RTMClashingTimeCount
static void KheSolnCheckForClashes(KHE_SOLN soln, char *message)
{
  int i;  KHE_RESOURCE r;  KHE_RESOURCE_TIMETABLE_MONITOR rtm;
  KHE_INSTANCE ins = KheSolnInstance(soln);
  for( i = 0;  i < KheInstanceResourceCount(ins);  i++ )
  {
    r = KheInstanceResource(ins, i);
    if( DEBUG8_ID != NULL && strcmp(KheResourceId(r), DEBUG8_ID) == 0 )
    {
      fprintf(stderr, "timetable for %s %s:\n", KheResourceId(r), message);
      rtm = KheResourceTimetableMonitor(soln, r);
      KheResourceTimetableMonitorPrintTimetable(rtm, 10, 0, stderr);
    }
    KheNoClashesCheck(soln, r);
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  Submodule "trying unassignments"                                         */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool TryUnassign(KHE_TASK_SET ts, int first_index, int last_index)       */
/*                                                                           */
/*  Try unassigning tasks ts[first_index .. last_index].                     */
/*                                                                           */
/*****************************************************************************/

static bool TryUnassign(KHE_SOLN soln, KHE_TASK_SET ts, int first_index,
  int last_index)
{
  KHE_MARK mark;  int i;  bool success;  KHE_TASK task;  KHE_COST init_cost;

  /* try the unassignments, using a mark to undo them if unsuccessful */
  mark = KheMarkBegin(soln);
  success = true;
  init_cost = KheSolnCost(soln);
  for( i = first_index;  success && i <= last_index;  i++ )
  {
    task = KheTaskSetTask(ts, i);
    success = KheTaskUnAssign(task);
  }
  success = success && KheSolnCost(soln) < init_cost;
  KheMarkEnd(mark, !success);

  /* debug print if successful, and return */
  if( DEBUG9 && success )
  {
    fprintf(stderr, "  KheSolnTryTaskUnAssignments unassign ");
    for( i = first_index;  i <= last_index;  i++ )
    {
      task = KheTaskSetTask(ts, i);
      if( i > first_index )
	fprintf(stderr, ", ");
      KheTaskDebug(task, 1, -1, stderr);
    }
    fprintf(stderr, " success: %.5f -> %.5f\n", KheCostShow(init_cost),
      KheCostShow(KheSolnCost(soln)));
  }
  return success;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheSolnTryTaskUnAssignments(KHE_SOLN soln, KHE_OPTIONS options)     */
/*                                                                           */
/*  Try unassigning each proper root task of soln, to see if that improves   */
/*  the cost.  Return true if any unassignments were kept.                   */
/*                                                                           */
/*****************************************************************************/

bool KheSolnTryTaskUnAssignments(KHE_SOLN soln, KHE_OPTIONS options)
{
  HA_ARENA a;  int i, j, unassign, max_unassign, x1, x2, ts_count;  bool res;
  KHE_TASK_SET ts;  KHE_INSTANCE ins;  KHE_RESOURCE r;  KHE_TASK_FINDER tf;
  a = KheSolnArenaBegin(soln, false);
  max_unassign = KheOptionsGetInt(options, "rs_max_unassign", 1);
  if( DEBUG9 )
    fprintf(stderr, "[ KheSolnTryTaskUnAssignments(soln, max_unassign %d)\n",
      max_unassign);
  tf = KheTaskFinderMake(soln, options, a);
  res = false;
  ins = KheSolnInstance(soln);
  ts = KheTaskSetMake(soln);
  for( i = 0;  i < KheInstanceResourceCount(ins);  i++ )
  {
    r = KheInstanceResource(ins, i);
    KheFindTasksInInterval(tf, 0, KheTaskFinderLastIndex(tf),
      KheResourceResourceType(r), r, true, false, ts, &x1, &x2);
    ts_count = KheTaskSetTaskCount(ts);
    for( unassign = 1;  unassign <= max_unassign;  unassign++ )
    {
      for( j = 0;  j <= ts_count - unassign;  j++ )
	if( TryUnassign(soln, ts, j, j + unassign - 1) )
	  res = true;
    }
  }
  KheTaskSetDelete(ts);
  KheSolnArenaEnd(soln, a);
  if( DEBUG9 )
    fprintf(stderr, "] KheSolnTryTaskUnAssignments returning %s\n",
      res ? "true" : "false");
  return res;
}


/* *** old version that does not consult rs_max_unassign option
bool KheSolnTryTaskUnAssignments(KHE_SOLN soln, KHE_OPTIONS options)
{
  KHE_TASK task, target_task;  int i, j;  KHE_COST init_cost;  bool ok, res;
  KHE_TRACE trace;  KHE_MONITOR m;  KHE_RESOURCE r;
  if( DEBUG9 )
    fprintf(stderr, "[ KheSolnTryTaskUnAssignments(soln)\n");
  res = false;
  for( i = 0;  i < KheSolnTaskCount(soln);  i++ )
  {
    task = KheSolnTask(soln, i);
    if( KheTaskProperRoot(task) == task )
    {
      target_task = KheTaskAsst(task);
      init_cost = KheSolnCost(soln);
      if( target_task != NULL && !KheTaskIsPreassigned(task, NULL) )
      {
	if( DEBUG10 )
	{
	  r = KheTaskAsstResource(task);
	  trace = KheTraceMake((KHE_GROUP_MONITOR) soln);
	  KheTraceBegin(trace);
	  ok = KheTaskUnAssign(task);
	  KheTraceEnd(trace);
	  fprintf(stderr, "  unassign %s from ",
	    r == NULL ? "?" : KheResourceId(r));
	  KheTaskDebug(task, 2, -1, stderr);
	  fprintf(stderr, ":  %.5f -> %.5f\n",
	    KheCostShow(init_cost), KheCostShow(KheSolnCost(soln)));
	  for( j = 0;  j < KheTraceMonitorCount(trace);  j++ )
	  {
	    m = KheTraceMonitor(trace, j);
	    fprintf(stderr, "      from %.5f to ",
	      KheCostShow(KheTraceMonitorInitCost(trace, j)));
	    KheMonitorDebug(m, 2, 0, stderr);
	  }
	  KheTraceDelete(trace);
	}
	else
	  ok = KheTaskUnAssign(task);
	if( ok )
	{
	  if( KheSolnCost(soln) < init_cost )
	  {
	    res = true;
	    if( DEBUG9 )
	    {
	      fprintf(stderr, "  %.5f -> %.5f for unassigning task ",
		KheCostShow(init_cost), KheCostShow(KheSolnCost(soln)));
	      KheTaskDebug(task, 2, 0, stderr);
	    }
	  }
	  else
	    KheTaskAssign(task, target_task);
	}
      }
    }
  }
  if( DEBUG9 )
    fprintf(stderr, "] KheSolnTryTaskUnAssignments returning %s\n",
      res ? "true" : "false");
  return res;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  Submodule "putting it all together"                                      */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  char *KheTaskingLabel(KHE_TASKING tasking)                               */
/*                                                                           */
/*  Return a string description of the scope of tasking.                     */
/*                                                                           */
/*****************************************************************************/

static char *KheTaskingLabel(KHE_TASKING tasking)
{
  return KheTaskingResourceType(tasking) == NULL ? "All Resources" :
    KheResourceTypeId(KheTaskingResourceType(tasking)) == NULL ? "?" :
    KheResourceTypeId(KheTaskingResourceType(tasking));
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDebugStage(KHE_TASKING tasking, char *stage, int verbosity)      */
/*                                                                           */
/*  Print a debug message showing the stage, and the solution cost.          */
/*                                                                           */
/*****************************************************************************/

static void KheDebugStage(KHE_TASKING tasking, char *stage, int verbosity)
{
  fprintf(stderr, "  %s:\n", stage);
  KheSolnDebug(KheTaskingSoln(tasking), verbosity, 2, stderr);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheResourceDebugMonitor(KHE_OPTIONS options, char *intro,           */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug the nominated monitor.                                             */
/*                                                                           */
/*****************************************************************************/

static void KheResourceDebugMonitor(KHE_OPTIONS options, char *intro,
  int indent, FILE *fp)
{
  KHE_MONITOR m;
  m = (KHE_MONITOR) KheOptionsGetObject(options, "gs_debug_monitor", NULL);
  if( m != NULL )
  {
    fprintf(fp, "%*s%s: ", indent, "", intro);
    KheMonitorDebug(m, 1, 0, fp);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void DebugStart(KHE_SOLN soln, KHE_OPTIONS options, char *fn_name)       */
/*                                                                           */
/*  Brief report of what is starting, for debugging.                         */
/*                                                                           */
/*****************************************************************************/

/* *** replaced by KheSolveDebug
static void DebugStart(KHE_SOLN soln, KHE_OPTIONS options, char *fn_name)
{
  KHE_INSTANCE ins;  KHE_TIMER timer;  char buff[20];
  if( DEBUG5 )
  {
    ins = KheSolnInstance(soln);
    fprintf(stderr, "%s div %d", KheInstanceId(ins), KheSolnD iversifier(soln));
    if( KheOptionsContainsTimer(options, "global", &timer) )
      fprintf(stderr, " after %s",
	KheTimeShow(KheTimerElapsedTime(timer), buff));
    fprintf(stderr, " calling %s\n", fn_name);
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheAssignResourcesGlobalCheck(KHE_SOLN soln, char *pos)             */
/*                                                                           */
/*  Global check of soln.                                                    */
/*                                                                           */
/*****************************************************************************/

/* ***
void KheAssignResourcesGlobalCheck(KHE_SOLN soln, char *pos)
{
  KHE_MONITOR m;
  if( DEBUG7 )
  {
    if( !KheSolnRetrieveMonitor(soln, DEBUG7_ID, &m) )
      HnAbort("DebugCheck (khe_sr_combined.c) internal error 1");
    HnAssert(KheMonitorTag(m) == KHE_LIMIT_RESOURCES_MONITOR_TAG,
      "DebugCheck (khe_sr_combined.c) internal error 2");
    KheLimitResourcesMonitorCheck((KHE_LIMIT_RESOURCES_MONITOR) m, pos);
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskingRepairResources(KHE_TASKING tasking, KHE_OPTIONS options, */
/*    float time_limit_multiplier, int variant)                              */
/*                                                                           */
/*  Repair the assignments of tasking using all the tricks available.        */
/*  (Not forgetting that each repair algorithm can be turned off by          */
/*  an option.)  Return true if any improvement.                             */
/*                                                                           */
/*  The time limit (if present) is multiplied by  time_limit_multiplier.     */
/*                                                                           */
/*  Parameter variant is passed to resource rematch to make it vary          */
/*  its behaviour on different calls on this function.                       */
/*                                                                           */
/*****************************************************************************/

static bool KheTaskingRepairResources(KHE_TASKING tasking, KHE_OPTIONS options,
  float time_limit_multiplier, int variant)
{
  KHE_COST init_cost;  KHE_SOLN soln;  KHE_INSTANCE ins;  KHE_RESOURCE_TYPE rt;
  float repair_time_limit;  KHE_TIMER repair_timer, rematch_timer;
  soln = KheTaskingSoln(tasking);
  if( DEBUG2 )
    KheDebugStage(tasking, "at start of KheTaskingRepairResources", 2);
  /* ***
  if( DEBUG8 )
    KheSolnCheckForClashes(soln, "at start of KheTaskingRepairResources");
  *** */

  /* set the repair and rematch timers */
  repair_time_limit = time_limit_multiplier * KheTimeFromString(
    KheOptionsGet(options, "rs_repair_time_limit", "-"));
  if( repair_time_limit >= 0.0 )
  {
    repair_timer = KheOptionsAddTimer(options, "repair", repair_time_limit);
    rematch_timer = KheOptionsAddTimer(options, "rematch",
      repair_time_limit / 2.0);
  }
  else
    repair_timer = rematch_timer = NULL;

  /* boilerplate */
  init_cost = KheSolnCost(soln);
  ins = KheSolnInstance(soln);
  rt = KheTaskingResourceType(tasking);

  /* rematch */
  if( DEBUG5 )
    KheSolveDebug(soln, options, "KheResourceRematch:");
  if( !KheOptionsGetBool(options, "rs_repair_rematch_off", false) )
    KheResourceRematch(soln, KheResourceTypeFullResourceGroup(rt), options,
      variant);
  /* ***
  if( DEBUG8 )
    KheSolnCheckForClashes(soln, "after KheResourceRematch");
  *** */
  if( rematch_timer != NULL )
    KheOptionsDeleteTimer(options, rematch_timer);
  if( DEBUG4 )
    KheSolnCheckTasks(KheTaskingSoln(tasking), "after KheResourceRematch");

  /* ejection chains */
  if( DEBUG5 )
    KheSolveDebug(soln, options, "KheEjectionChainRepairResources:");
  if( !KheOptionsGetBool(options, "rs_repair_ejection_off", false) )
    KheEjectionChainRepairResources(tasking, options);
  if( DEBUG4 )
    KheSolnCheckTasks(KheTaskingSoln(tasking),
      "after KheEjectionChainRepairResources");

  /* reassignment */
  if( KheInstanceModel(ins) == KHE_MODEL_EMPLOYEE_SCHEDULE )
  {
    /* KheReassignRepair */
    if( DEBUG5 )
      KheSolveDebug(soln, options, "KheReassignRepair:");
    KheReassignRepair(soln, rt, options);
    if( DEBUG4 )
      KheSolnCheckTasks(soln, "after KheReassignRepair");

    /* KheReassign2Repair */
    if( DEBUG5 )
      KheSolveDebug(soln, options, "KheReassign2Repair:");
    KheReassign2Repair(soln, rt, options);
    if( DEBUG4 )
      KheSolnCheckTasks(soln, "after KheReassign2Repair");

    /* KheReassign3Repair */
    if( DEBUG5 )
      KheSolveDebug(soln, options, "KheReassign3Repair:");
    KheReassign3Repair(soln, rt, options);
    if( DEBUG4 )
      KheSolnCheckTasks(soln, "after KheReassign3Repair");
  }

  /* wrapup */
  if( repair_timer != NULL )
    KheOptionsDeleteTimer(options, repair_timer);
  if( DEBUG6 && KheSolnCost(soln) < init_cost )
    fprintf(stderr, "KheTaskingRepairResources improvement (%.5f -> %.5f)\n",
      KheCostShow(init_cost), KheCostShow(KheSolnCost(soln)));
  return KheSolnCost(soln) < init_cost;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDoTimeSweep(KHE_SOLN soln, KHE_RESOURCE_TYPE rt,                 */
/*    KHE_OPTIONS options)                                                   */
/*                                                                           */
/*  Carry out a time sweep assignment, along with a few other odd things.    */
/*                                                                           */
/*****************************************************************************/

static void KheDoTimeSweep(KHE_SOLN soln, KHE_RESOURCE_TYPE rt,
  KHE_OPTIONS options)
{
  KHE_FRAME frame;
  frame = KheOptionsFrame(options, "gs_common_frame", soln);
  if( KheResourceTypeLimitResourcesCount(rt) > 0 )
    KheOptionsSetBool(options, "rs_rematch_matching_off", true);
  if( DEBUG5 )
    KheSolveDebug(soln, options, "KheSolnAssignRequestedResources:");
  if( DEBUG8 )
    KheFrameAssertNoClashes(frame);
  KheSolnAssignRequestedResources(soln, rt, options);
  KheDebugTimetable(soln, "after KheSolnAssignRequestedResources");
  if( DEBUG8 )
    KheFrameAssertNoClashes(frame);
  if( DEBUG4 )
    KheSolnCheckTasks(soln, "after Stage1 requested");
  if( DEBUG5 )
    KheSolveDebug(soln, options, "KheTimeSweepAssignResources:");
  KheTimeSweepAssignResources(soln, KheResourceTypeFullResourceGroup(rt),
    options);
  KheDebugTimetable(soln, "after KheTimeSweepAssignResources");
  if( DEBUG8 )
    KheFrameAssertNoClashes(frame);
  if( DEBUG4 )
    KheSolnCheckTasks(soln, "after Stage1 time sweep");
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskingAssignResourcesStage1(KHE_TASKING tasking,                */
/*    KHE_OPTIONS options)                                                   */
/*                                                                           */
/*  Assign resources to the leader tasks of tasking (Stage 1):  assign       */
/*  most tasks preserving the resource assignment invariant.                 */
/*                                                                           */
/*****************************************************************************/

bool KheTaskingAssignResourcesStage1(KHE_TASKING tasking, KHE_OPTIONS options)
{
  KHE_RESOURCE_TYPE rt;  KHE_INSTANCE ins;  char *resource_assign;
  KHE_SOLN soln;  KHE_TASK_SET group_by_rc_ts, group_by_resource_ts;
  char *rs_multiplier;  char mult_str[101];  int mult_val;  KHE_RESOURCE r;
  if( DEBUG1 )
    fprintf(stderr, "[ KheTaskingAssignResourcesStage1(%s)\n",
      KheTaskingLabel(tasking));
  soln = KheTaskingSoln(tasking);
  KheDebugTimetable(soln, "at the start of Stage 1");
  if( DEBUG4 )
    KheSolnCheckTasks(soln, "at start of Stage1");
  /* ***
  if( DEBUG8 )
    KheSolnCheckForClashes(soln, "at start of Stage1");
  *** */

  /* maintain the resource assignment invariant */
  KheOptionsSetBool(options, "rs_invariant", true);

  /* rs_multiplier option */
  rs_multiplier = KheOptionsGet(options, "rs_multiplier", NULL);
  if( rs_multiplier != NULL )
  {
    if( sscanf(rs_multiplier, "%d:%100s", &mult_val, mult_str) != 2 )
      HnAbort("syntax error in rs_multiplier value \"%s\"", rs_multiplier);
    KheSetMonitorMultipliers(soln, mult_str, mult_val);
  }

  /* grouping by resource constraints */
  if( DEBUG5 )
    KheSolveDebug(soln, options, "before KheTaskingGroupByResourceConstraints");
  /* DebugStart(soln, options, "KheTaskingGroupByResourceConstraints"); */
  group_by_rc_ts = KheTaskSetMake(soln);
  KheGroupByResourceConstraints(soln, KheTaskingResourceType(tasking),
    options, group_by_rc_ts);
  /* ***
  if( DEBUG8 )
    KheSolnCheckForClashes(soln, "after KheGroupByResourceConstraints");
  *** */

  /* enforcing work patterns */
  /* *** incomplete and not used
  if( DEBUG5 )
    KheSolveDebug(soln, options, "before KheEnforceWorkPatterns");
  KheEnforceWorkPatterns(soln, KheTaskingResourceType(tasking), options);
  *** */

  /* consec solver test */
  rt = KheTaskingResourceType(tasking);
  /* ***
  if( DEBUG10 )
  {
    KHE_FRAME frame;  KHE_CONSEC_SOLVER cs;
    frame = KheOptionsFrame(options, "gs_common_frame", soln);
    cs = KheConsecSolverMake(soln, frame);
    KheConsecSolverDebug(cs, 2, 2, stderr);
  }
  *** */

  /* sort out the resource_assign options */
  ins = KheSolnInstance(soln);
  resource_assign = KheOptionsGet(options, "rs_constructor", "auto");
  if( strcmp(resource_assign, "auto") == 0 )
  {
    if( KheInstanceModel(ins) == KHE_MODEL_EMPLOYEE_SCHEDULE )
      resource_assign = "time_sweep";
    else if( rt != NULL && KheResourceTypeAvoidSplitAssignmentsCount(rt) == 0 )
      resource_assign = "most_constrained";
    else
      resource_assign = "resource_packing";
  }

  /* construction */
  if( DEBUG5 )
    KheSolveDebug(soln, options, "before %s:", resource_assign);
  /* DebugStart(soln, options, resource_assign); */
  if( strcmp(resource_assign, "none") == 0 )
  {
    /* do nothing */
  }
  else if( strcmp(resource_assign, "most_constrained") == 0 )
    KheMostConstrainedFirstAssignResources(tasking, options);
  else if( strcmp(resource_assign, "resource_packing") == 0 )
    KheResourcePackAssignResources(tasking, options);
  else if( strcmp(resource_assign, "time_sweep") == 0 )
    KheDoTimeSweep(soln, rt, options);
  /* ***
  else if( strcmp(resource_assign, "consec_packing") == 0 )
    KheResourcePackConsecutive(tasking, options);
  *** */
  else if( strcmp(resource_assign, "requested_only") == 0 )
    KheSolnAssignRequestedResources(soln, rt, options);
  else if( strcmp(resource_assign, "single_test") == 0 )
  {
    if( KheResourceTypeResourceCount(rt) > 0 )
    {
      r = KheResourceTypeResource(rt, 0);
      KheSingleResourceSolverTest(soln, options, r);
    }
    KheOptionsSetBool(options, "rs_repair_off", true);
    return true;
  }
  else
    HnAbort("KheTaskingAssignResourcesStage1: "
      "invalid resource_assign option \"%s\"", resource_assign);
  /* ***
  if( DEBUG8 )
    KheSolnCheckForClashes(soln, "after Stage 1 construction");
  *** */
  if( DEBUG4 )
    KheSolnCheckTasks(KheTaskingSoln(tasking), "after Stage1 construction");

  /* repair */
  if( !KheOptionsGetBool(options, "rs_repair_off", false) &&
      !KheOptionsGetBool(options, "rs_repair1_off", false) )
  {
    /* repair without grouping by resource */
    if( DEBUG3 )
      KheResourceDebugMonitor(options, "before Stage1 repair", 2, stderr);
    KheTaskingRepairResources(tasking, options, 1.0, 0);

    /* optional repair with grouping by resource */
    if( KheOptionsGetBool(options, "rs_group_by_resource", false) )
    {
      group_by_resource_ts = KheTaskSetMake(soln);
      if( KheTaskingGroupByResource(tasking, options, group_by_resource_ts) )
	KheTaskingRepairResources(tasking, options, 0.5, 1);
      KheTaskSetUnGroup(group_by_resource_ts);
      KheTaskSetDelete(group_by_resource_ts);
    }
  }

  /* ungroup the grouping by resource constraints */
  if( !KheOptionsGetBool(options, "rs_ungroup_off", false) )
  {
    KheTaskSetUnGroup(group_by_rc_ts);
    KheTaskSetDelete(group_by_rc_ts);
  }

  /* undo the rs_multiplier option */
  if( rs_multiplier != NULL )
    KheSetMonitorMultipliers(soln, mult_str, 1);

  KheDebugTimetable(soln, "at the end of Stage 1");
  if( DEBUG1 )
  {
    KheDebugStage(tasking, "at end of Stage1", 2);
    fprintf(stderr, "] KheTaskingAssignResourcesStage1 returning\n");
  }
  if( DEBUG4 )
    KheSolnCheckTasks(KheTaskingSoln(tasking), "at end of Stage1");
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskingAssignResourcesStage2(KHE_TASKING tasking,                */
/*    KHE_OPTIONS options)                                                   */
/*                                                                           */
/*  Assign resources to the leader tasks of tasking (Stage 2):  find         */
/*  split assignments, still preserving the resource assignment invariant.   */
/*                                                                           */
/*****************************************************************************/

bool KheTaskingAssignResourcesStage2(KHE_TASKING tasking, KHE_OPTIONS options)
{
  KHE_RESOURCE_TYPE rt;  KHE_SOLN soln;

  rt = KheTaskingResourceType(tasking);
  if( KheResourceTypeAvoidSplitAssignmentsCount(rt) > 0 &&
      !KheOptionsGetBool(options, "rs_repair_off", false) &&
      !KheOptionsGetBool(options, "rs_repair2_off", false) )
  {
    if( DEBUG1 )
      fprintf(stderr, "[ KheTaskingAssignResourcesStage2(%s)\n",
	KheTaskingLabel(tasking));
    soln = KheTaskingSoln(tasking);
    KheDebugTimetable(soln, "at the start of Stage 2");
    KheFindSplitResourceAssignments(tasking, options);
    KheTaskingAllowSplitAssignments(tasking, false);
    if( DEBUG3 )
      KheResourceDebugMonitor(options, "monitor before Stage2 repair",2,stderr);
    KheTaskingRepairResources(tasking, options, 0.5, 0);
    if( DEBUG1 )
    {
      KheDebugStage(tasking, "at end of Stage2", 2);
      fprintf(stderr, "] KheTaskingAssignResourcesStage2 returning\n");
    }
  }
  if( DEBUG4 )
    KheSolnCheckTasks(soln, "at end of Stage2");
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskingAssignResourcesStage3(KHE_TASKING tasking,                */
/*    KHE_OPTIONS options)                                                   */
/*                                                                           */
/*  Assign resources to the leader tasks of tasking (Stage 2):  last-ditch   */
/*  assignments, without the resource assignment invariant.                  */
/*                                                                           */
/*****************************************************************************/

bool KheTaskingAssignResourcesStage3(KHE_TASKING tasking, KHE_OPTIONS options)
{
  KHE_SOLN soln;
  if( !KheOptionsGetBool(options, "rs_repair_off", false) &&
      !KheOptionsGetBool(options, "rs_repair3_off", false) )
  {
    if( DEBUG1 )
      fprintf(stderr, "[ KheTaskingAssignResourcesStage3(%s)\n",
	KheTaskingLabel(tasking));
    soln = KheTaskingSoln(tasking);
    KheDebugTimetable(soln, "at the start of Stage 3");
    /* ***
    if( DEBUG8 )
      KheSolnCheckForClashes(soln, "at start of Stage3");
    *** */
    KheOptionsSetBool(options, "rs_invariant", false);
    /* ***
    KheTaskingEnlargeDomains(tasking, false);
    *** */
    if( DEBUG3 )
      KheResourceDebugMonitor(options, "before Stage3 repair", 2, stderr);
    /* KheOptionsSetBool(options, "es_fresh_visits", true); */
    KheTaskingRepairResources(tasking, options, 0.5, 1);
    /* ***
    if( DEBUG4 )
      KheSolnCheckTasks(soln, "second repair in Stage3");
    KheTaskingRepairResources(tasking, options, 0.5, 1);
    *** */
    KheDebugTimetable(soln, "at the end of Stage 3");
    if( DEBUG4 )
      KheSolnCheckTasks(soln, "at end of Stage3");
    if( DEBUG1 )
    {
      KheDebugStage(tasking, "at end of Stage3", 2);
      fprintf(stderr, "] KheTaskingAssignResourcesStage3 returning\n");
    }
  }
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskingAssignResources(KHE_TASKING tasking, KHE_OPTIONS options) */
/*                                                                           */
/*  Assign resources to the leader tasks of tasking.                         */
/*                                                                           */
/*****************************************************************************/

/* *** not used; was not in khe.h
bool KheTaskingAssignResources(KHE_TASKING tasking, KHE_OPTIONS options)
{
  KheTaskingAssignResourcesStage1(tasking, options);
  KheTaskingAssignResourcesStage2(tasking, options);
  KheTaskingAssignResourcesStage3(tasking, options);
  return true;
}
*** */
