
/*****************************************************************************/
/*                                                                           */
/*  THE KHE HIGH SCHOOL TIMETABLING ENGINE                                   */
/*  COPYRIGHT (C) 2010 Jeffrey H. Kingston                                   */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 3, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         khe_sr_consec_resource.c                                   */
/*  DESCRIPTION:  Consecutive packing                                        */
/*                                                                           */
/*****************************************************************************/
#include "khe_solvers.h"
#include "khe_priqueue.h"

#define DEBUG1 1

/*****************************************************************************/
/*                                                                           */
/*  KHE_CONSEC_RESOURCE - a resource                                         */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_consec_resource_rec {
  int			max_workload;		/* max workload              */
  KHE_RESOURCE		resource;		/* the resource              */
} *KHE_CONSEC_RESOURCE;

typedef HA_ARRAY(KHE_CONSEC_RESOURCE) ARRAY_KHE_CONSEC_RESOURCE;


/*****************************************************************************/
/*                                                                           */
/*  KHE_CONSEC_RUN - a run of tasks                                          */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_consec_run_rec {
  KHE_TASK_SET		task_set;		/* the tasks                 */
  int			first_index;		/* first index               */
  int			last_index;		/* last index                */
  int			weight;			/* their weight              */
} *KHE_CONSEC_RUN;

typedef HA_ARRAY(KHE_CONSEC_RUN) ARRAY_KHE_CONSEC_RUN;


/*****************************************************************************/
/*                                                                           */
/*  KHE_CONSEC_GROUP - a group of tasks with the same domain and shift type  */
/*                                                                           */
/*****************************************************************************/

typedef HA_ARRAY(KHE_TASK_SET) ARRAY_KHE_TASK_SET;

typedef struct khe_consec_group_rec {
  int			shift_type;		/* shift type (index of)     */
  KHE_RESOURCE_GROUP	domain;			/* domain                    */
  ARRAY_KHE_TASK_SET	task_sets;		/* one for each day of frame */
} *KHE_CONSEC_GROUP;

typedef HA_ARRAY(KHE_CONSEC_GROUP) ARRAY_KHE_CONSEC_GROUP;


/*****************************************************************************/
/*                                                                           */
/*  KHE_CONSEC_SHIFT_TYPE - a group of tasks with the same shift type        */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_consec_shift_type_rec {
  int			shift_type;		/* shift type (index of)     */
  ARRAY_KHE_CONSEC_GROUP groups;		/* one group for each domain */
} *KHE_CONSEC_SHIFT_TYPE;

typedef HA_ARRAY(KHE_CONSEC_SHIFT_TYPE) ARRAY_KHE_CONSEC_SHIFT_TYPE;


/*****************************************************************************/
/*                                                                           */
/*  KHE_CONSEC_DOMAIN - a group of tasks with the same domain                */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_consec_domain_rec {
  KHE_RESOURCE_GROUP	domain;			/* domain                    */
  ARRAY_KHE_CONSEC_GROUP groups;		/* one for each shift type   */
} *KHE_CONSEC_DOMAIN;

typedef HA_ARRAY(KHE_CONSEC_DOMAIN) ARRAY_KHE_CONSEC_DOMAIN;


/*****************************************************************************/
/*                                                                           */
/*  KHE_CONSEC - solver object for this module                               */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_consec_rec {
  HA_ARENA			arena;			/* arena             */
  KHE_SOLN			soln;			/* solution          */
  KHE_FRAME			frame;			/* days frame        */
  KHE_TASK_FINDER		task_finder;		/* task finder       */
  KHE_CONSEC_SOLVER		consec_solver;		/* finds limits      */
  ARRAY_KHE_CONSEC_SHIFT_TYPE	tasks_by_shift_type;	/* by shift type     */
  ARRAY_KHE_CONSEC_DOMAIN	tasks_by_domain;	/* by domain         */
  ARRAY_KHE_CONSEC_RESOURCE	resources;		/* resources         */
  ARRAY_KHE_CONSEC_RUN		free_runs;		/* free runs         */
  ARRAY_KHE_CONSEC_RUN		curr_runs;		/* curr runs         */
  ARRAY_KHE_CONSEC_RUN		tmp_runs;		/* tmp runs          */
} *KHE_CONSEC;


/*****************************************************************************/
/*                                                                           */
/*  Submodule "consec resources"                                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_CONSEC_RESOURCE KheConsecResourceMake(KHE_RESOURCE r, KHE_CONSEC c)  */
/*                                                                           */
/*  Make a new consec resource object for r.                                 */
/*                                                                           */
/*****************************************************************************/

static KHE_CONSEC_RESOURCE KheConsecResourceMake(KHE_RESOURCE r, KHE_CONSEC c)
{
  KHE_CONSEC_RESOURCE res;
  HaMake(res, c->arena);
  if( !KheResourceAvailableBusyTimes(c->soln, r, &res->max_workload) )
     res->max_workload = KheFrameTimeGroupCount(c->frame);
  res->resource = r;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  int KheConsecResourceCmp(const void *t1, const void *t2)                 */
/*                                                                           */
/*  Comparison function for sorting an array of consec resource by           */
/*  decreasing max_workload.                                                 */
/*                                                                           */
/*****************************************************************************/

static int KheConsecResourceCmp(const void *t1, const void *t2)
{
  KHE_CONSEC_RESOURCE cr1 = * (KHE_CONSEC_RESOURCE *) t1;
  KHE_CONSEC_RESOURCE cr2 = * (KHE_CONSEC_RESOURCE *) t2;
  return cr2->max_workload - cr1->max_workload;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheConsecResourceDebug(KHE_CONSEC_RESOURCE cr, int indent, FILE *fp)*/
/*                                                                           */
/*  Debug print of cr onto fp with the given indent.                         */
/*                                                                           */
/*****************************************************************************/

static void KheConsecResourceDebug(KHE_CONSEC_RESOURCE cr, int indent, FILE *fp)
{
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  fprintf(fp, "%s(%d)", KheResourceId(cr->resource), cr->max_workload);
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "consec groups"                                                */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_CONSEC_GROUP KheConsecGroupMake(KHE_CONSEC c, int index,             */
/*    KHE_RESOURCE_GROUP domain, HA_ARENA a)                                 */
/*                                                                           */
/*  Make a new consec group object with these attributes.                    */
/*                                                                           */
/*****************************************************************************/

static KHE_CONSEC_GROUP KheConsecGroupMake(KHE_CONSEC c, int index,
  KHE_RESOURCE_GROUP domain, HA_ARENA a)
{
  KHE_CONSEC_GROUP res;  KHE_TASK_SET ts;  int i;
  HaMake(res, a);
  res->shift_type = index;
  res->domain = domain;
  HaArrayInit(res->task_sets, a);
  for( i = 0;  i < KheFrameTimeGroupCount(c->frame);  i++ )
  {
    ts = KheTaskSetMake(c->soln);
    HaArrayAddLast(res->task_sets, ts);
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheConsecGroupAddTask(KHE_CONSEC_GROUP cg, KHE_TASK task,           */
/*    KHE_CONSEC c)                                                          */
/*                                                                           */
/*  Add task to cg.                                                          */
/*                                                                           */
/*****************************************************************************/

static void KheConsecGroupAddTask(KHE_CONSEC_GROUP cg, KHE_TASK task,
  KHE_CONSEC c)
{
  int first_index, last_index;  KHE_TASK_SET ts;
  KheTaskFinderTaskInterval(c->task_finder, task, &first_index, &last_index);
  HnAssert(first_index >= 0 && first_index < HaArrayCount(cg->task_sets),
    "KheConsecGroupAddTask internal error:  first_index (%d) out of range "
    "(0.. %d)", first_index, HaArrayCount(cg->task_sets) - 1);
  ts = HaArray(cg->task_sets, first_index);
  KheTaskSetAddTask(ts, task);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheConsecGroupDebug(KHE_CONSEC_GROUP cg, int verbosity,             */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of cg with the given verbosity and indent.                   */
/*                                                                           */
/*****************************************************************************/

static void KheConsecGroupDebug(KHE_CONSEC_GROUP cg, int verbosity,
  int indent, FILE *fp)
{
  KHE_TASK_SET ts;  int i;
  fprintf(fp, "%*s[ ConsecGroup %d ", indent, "", cg->shift_type);
  KheResourceGroupDebug(cg->domain, 1, 0, fp);
  HaArrayForEach(cg->task_sets, ts, i)
    KheTaskSetDebug(ts, 2, indent + 2, fp);
  fprintf(fp, "%*s]\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "shift types"                                                  */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_CONSEC_SHIFT_TYPE KheConsecShiftTypeMake(int shift_type, HA_ARENA a) */
/*                                                                           */
/*  Make a new, empty shift type object.                                     */
/*                                                                           */
/*****************************************************************************/

static KHE_CONSEC_SHIFT_TYPE KheConsecShiftTypeMake(int shift_type, HA_ARENA a)
{
  KHE_CONSEC_SHIFT_TYPE res;
  HaMake(res, a);
  res->shift_type = shift_type;
  HaArrayInit(res->groups, a);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheConsecTaskHasShiftTypeIndex(KHE_TASK task, KHE_FRAME frame,      */
/*    int *index)                                                            */
/*                                                                           */
/*  If task has an assigned time in frame, return true and set *index to     */
/*  its index in its time group in frame.                                    */
/*                                                                           */
/*****************************************************************************/

static bool KheConsecTaskHasShiftTypeIndex(KHE_TASK task, KHE_FRAME frame,
  int *index)
{
  KHE_MEET meet;  KHE_TIME t;  KHE_TIME_GROUP tg;

  /* find the task's meet */
  meet = KheTaskMeet(KheTaskProperRoot(task));
  if( meet == NULL )
    return false;

  /* find the meet's assigned time */
  t = KheMeetAsstTime(meet);
  if( t == NULL )
    return false;

  /* find t's time group in frame and return its index in that time group */
  tg = KheFrameTimeTimeGroup(frame, t);
  return KheTimeGroupContains(tg, t, index);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "domains"                                                      */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_CONSEC_DOMAIN KheConsecDomainMake(KHE_RESOURCE_GROUP rg, HA_ARENA a) */
/*                                                                           */
/*  Make a new, empty domain object.                                         */
/*                                                                           */
/*****************************************************************************/

static KHE_CONSEC_DOMAIN KheConsecDomainMake(KHE_RESOURCE_GROUP rg, HA_ARENA a)
{
  KHE_CONSEC_DOMAIN res;
  HaMake(res, a);
  res->domain = rg;
  HaArrayInit(res->groups, a);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheConsecDomainContainsGroup(KHE_CONSEC_DOMAIN cd, int index,       */
/*    KHE_CONSEC_GROUP *cg)                                                  */
/*                                                                           */
/*  If cd contains a group with this shift type, return true and set *cg     */
/*  to that group.                                                           */
/*                                                                           */
/*****************************************************************************/

static bool KheConsecDomainContainsGroup(KHE_CONSEC_DOMAIN cd, int index,
  KHE_CONSEC_GROUP *cg)
{
  KHE_CONSEC_GROUP g;  int i;
  HaArrayForEach(cd->groups, g, i)
    if( g->shift_type == index )
      return *cg = g, true;
  return *cg = NULL, false;
}


/*****************************************************************************/
/*                                                                           */
/*  int KheConsecDomainCmp(const void *t1, const void *t2)                   */
/*                                                                           */
/*  Comparison function for sorting domain objects by increasing domain size.*/
/*                                                                           */
/*****************************************************************************/

static int KheConsecDomainCmp(const void *t1, const void *t2)
{
  KHE_CONSEC_DOMAIN cd1 = * (KHE_CONSEC_DOMAIN *) t1;
  KHE_CONSEC_DOMAIN cd2 = * (KHE_CONSEC_DOMAIN *) t2;
  return KheResourceGroupResourceCount(cd1->domain) -
    KheResourceGroupResourceCount(cd2->domain);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheConsecDomainDebug(KHE_CONSEC_DOMAIN cd, int verbosity,           */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of cd onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void KheConsecDomainDebug(KHE_CONSEC_DOMAIN cd, int verbosity,
  int indent, FILE *fp)
{
  KHE_CONSEC_GROUP cg;  int i;
  fprintf(fp, "%*s[ ConsecDomain (%d resources) ", indent, "",
    KheResourceGroupResourceCount(cd->domain));
  KheResourceGroupDebug(cd->domain, 1, 0, fp);
  HaArrayForEach(cd->groups, cg, i)
    KheConsecGroupDebug(cg, verbosity, indent + 2, fp);
  fprintf(fp, "%*s]\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "consec objects"                                               */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_CONSEC KheConsecMake(KHE_SOLN soln, KHE_OPTIONS options, HA_ARENA a) */
/*                                                                           */
/*  Make a new consec object in arena a.                                     */
/*                                                                           */
/*****************************************************************************/

static KHE_CONSEC KheConsecMake(KHE_SOLN soln, KHE_OPTIONS options, HA_ARENA a)
{
  KHE_CONSEC res;
  HaMake(res, a);
  res->arena = a;
  res->soln = soln;
  res->task_finder = KheTaskFinderMake(soln, options, a);
  res->frame = KheTaskFinderFrame(res->task_finder);
  res->consec_solver = KheConsecSolverMake(soln, res->frame);
  HaArrayInit(res->tasks_by_shift_type, a);
  HaArrayInit(res->tasks_by_domain, a);
  HaArrayInit(res->resources, a);
  HaArrayInit(res->free_runs, a);
  HaArrayInit(res->curr_runs, a);
  HaArrayInit(res->tmp_runs, a);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_CONSEC_DOMAIN KheConsecFindOrAddDomain(KHE_CONSEC c,                 */
/*    KHE_RESOURCE_GROUP domain)                                             */
/*                                                                           */
/*  Find or add a domain object with this domain to c.                       */
/*                                                                           */
/*****************************************************************************/

static KHE_CONSEC_DOMAIN KheConsecFindOrAddDomain(KHE_CONSEC c,
  KHE_RESOURCE_GROUP domain)
{
  KHE_CONSEC_DOMAIN cd;  int i;
  HaArrayForEach(c->tasks_by_domain, cd, i)
    if( KheResourceGroupEqual(cd->domain, domain) )
      return cd;
  cd = KheConsecDomainMake(domain, c->arena);
  HaArrayAddLast(c->tasks_by_domain, cd);
  return cd;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_CONSEC_SHIFT_TYPE KheConsecFindOrAddShiftType(KHE_CONSEC c,          */
/*    int index)                                                             */
/*                                                                           */
/*  Find or add a shift type object with this shift type to c.               */
/*                                                                           */
/*****************************************************************************/

static KHE_CONSEC_SHIFT_TYPE KheConsecFindOrAddShiftType(KHE_CONSEC c,
  int index)
{
  KHE_CONSEC_SHIFT_TYPE cs;  int i;
  HaArrayForEach(c->tasks_by_shift_type, cs, i)
    if( cs->shift_type == index )
      return cs;
  cs = KheConsecShiftTypeMake(index, c->arena);
  HaArrayAddLast(c->tasks_by_shift_type, cs);
  return cs;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheConsecAddTask(KHE_CONSEC c, KHE_TASK task)                       */
/*                                                                           */
/*  Add task to c.                                                           */
/*                                                                           */
/*****************************************************************************/

static void KheConsecAddTask(KHE_CONSEC c, KHE_TASK task)
{
  KHE_CONSEC_SHIFT_TYPE  cs;  KHE_CONSEC_DOMAIN  cd;
  int index;  KHE_RESOURCE_GROUP domain;  KHE_CONSEC_GROUP cg;
  if( KheConsecTaskHasShiftTypeIndex(task, c->frame, &index) )
  {
    /* make sure there are cs and cd objects available */
    domain = KheTaskDomain(task);
    cd = KheConsecFindOrAddDomain(c, domain);
    cs = KheConsecFindOrAddShiftType(c, index);

    /* make and add a cg object if there currently isn't one */
    if( !KheConsecDomainContainsGroup(cd, index, &cg) )
    {
      cg = KheConsecGroupMake(c, index, domain, c->arena);
      HaArrayAddLast(cd->groups, cg);
      HaArrayAddLast(cs->groups, cg);
    }

    /* add task to cg */
    KheConsecGroupAddTask(cg, task, c);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheConsecDebug(KHE_CONSEC c, int verbosity, int indent, FILE *fp)   */
/*                                                                           */
/*  Print c onto fp with the given verbosity and indent.                     */
/*                                                                           */
/*****************************************************************************/

static void KheConsecDebug(KHE_CONSEC c, int verbosity, int indent, FILE *fp)
{
  KHE_CONSEC_DOMAIN cd;  int i;  KHE_CONSEC_RESOURCE cr;
  fprintf(fp, "%*s[ Consec\n", indent, "");
  HaArrayForEach(c->tasks_by_domain, cd, i)
    KheConsecDomainDebug(cd, verbosity, indent + 2, fp);
  HaArrayForEach(c->resources, cr, i)
    KheConsecResourceDebug(cr, indent + 2, fp);
  fprintf(fp, "%*s]\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "runs"                                                         */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_CONSEC_RUN KheConsecRunMake(KHE_TASK_SET ts)                         */
/*                                                                           */
/*  Make a run from ts.                                                      */
/*                                                                           */
/*****************************************************************************/

static KHE_CONSEC_RUN KheConsecRunMake(KHE_CONSEC c, int weight)
{
  KHE_CONSEC_RUN res;
  if( HaArrayCount(c->free_runs) > 0 )
  {
    res = HaArrayLastAndDelete(c->free_runs);
    KheTaskSetClear(res->task_set);
  }
  else
  {
    HaMake(res, c->arena);
    res->task_set = KheTaskSetMake(c->soln);
  }
  res->weight = weight;
  res->first_index = -1;
  res->last_index = -1;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  int KheConsecRunCmp(const void *t1, const void *t2)                      */
/*                                                                           */
/*  Comparison function for sorting an array of runs by increasing weight.   */
/*                                                                           */
/*****************************************************************************/

static int KheConsecRunCmp(const void *t1, const void *t2)
{
  KHE_CONSEC_RUN run1 = * (KHE_CONSEC_RUN *) t1;
  KHE_CONSEC_RUN run2 = * (KHE_CONSEC_RUN *) t2;
  return run1->weight - run2->weight;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "main function"                                                */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheConsecAddResources(KHE_CONSEC c, KHE_RESOURCE_TYPE rt)           */
/*                                                                           */
/*  Add the resources of type rt to c.                                       */
/*                                                                           */
/*****************************************************************************/

static void KheConsecAddResources(KHE_CONSEC c, KHE_RESOURCE_TYPE rt)
{
  int i;  KHE_RESOURCE r;  KHE_CONSEC_RESOURCE cr;
  for( i = 0;  i < KheResourceTypeResourceCount(rt);  i++ )
  {
    r = KheResourceTypeResource(rt, i);
    cr = KheConsecResourceMake(r, c);
    HaArrayAddLast(c->resources, cr);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheConsecGatherRuns(KHE_CONSEC c, KHE_RESOURCE r,                   */
/*    int *start_index, int *stop_index)                                     */
/*                                                                           */
/*  Gather runs for r.  The first one is at *start_index, and *stop_index    */
/*  is just after the last one.                                              */
/*                                                                           */
/*****************************************************************************/

static void KheConsecGatherRuns(KHE_CONSEC c, KHE_RESOURCE r,
  int *start_index, int *stop_index)
{
  int i;

  /* gather the runs into c->tmp_runs */
  HaArrayClear(c->tmp_runs);
  /* remainder still to do */

  /* sort tmp_runs, append to curr_runs, and set *start_index and *stop_index */
  HaArraySort(c->tmp_runs, &KheConsecRunCmp);
  *start_index = HaArrayCount(c->curr_runs);
  HaArrayAppend(c->curr_runs, c->tmp_runs, i);
  *stop_index = HaArrayCount(c->curr_runs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheConsecFreeRuns(KHE_CONSEC c, int start_index, int stop_index)    */
/*                                                                           */
/*  Free the runs at positions start_index to stop_index.  Actually          */
/*  stop_index is off the end of the array.                                  */
/*                                                                           */
/*****************************************************************************/

static void KheConsecFreeRuns(KHE_CONSEC c, int start_index, int stop_index)
{
  KHE_CONSEC_RUN run;
  while( HaArrayCount(c->curr_runs) > start_index )
  {
    run = HaArrayLastAndDelete(c->curr_runs);
    HaArrayAddLast(c->free_runs, run);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheRunApply(KHE_CONSEC c, KHE_RESOURCE r, KHE_CONSEC_RUN run)       */
/*                                                                           */
/*  Apply run to r.                                                          */
/*                                                                           */
/*****************************************************************************/

static bool KheRunApply(KHE_CONSEC c, KHE_RESOURCE r, KHE_CONSEC_RUN run)
{
  /* still to do */
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheRunUnApply(KHE_CONSEC c, KHE_RESOURCE r, KHE_CONSEC_RUN run)     */
/*                                                                           */
/*  Unapply run to r.                                                        */
/*                                                                           */
/*****************************************************************************/

static void KheRunUnApply(KHE_CONSEC c, KHE_RESOURCE r, KHE_CONSEC_RUN run)
{
  /* still to do */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheConsecSolveForResource(KHE_CONSEC c, KHE_RESOURCE r,             */
/*    int day_index)                                                         */
/*                                                                           */
/*  Solve for r from day_index onwards.                                      */
/*                                                                           */
/*****************************************************************************/

static void KheConsecSolveForResource(KHE_CONSEC c, KHE_RESOURCE r,
  int day_index)
{
  int start_index, stop_index, i;  KHE_CONSEC_RUN run;
  if( day_index >= KheFrameTimeGroupCount(c->frame) )
  {
    /* at end of path */
  }
  else
  {
    /* need another run */
    KheConsecGatherRuns(c, r, &start_index, &stop_index);
    for( i = start_index;  i < stop_index;  i++ )
    {
      run = HaArray(c->curr_runs, i);
      if( KheRunApply(c, r, run) )
      {
        KheConsecSolveForResource(c, r, run->last_index + 1);
        KheRunUnApply(c, r, run);
      }
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheResourcePackConsecutive(KHE_TASKING tasking, KHE_OPTIONS options)*/
/*                                                                           */
/*  Carry out the pack consecutive algorithm.                                */
/*                                                                           */
/*****************************************************************************/

bool KheResourcePackConsecutive(KHE_TASKING tasking, KHE_OPTIONS options)
{
  KHE_CONSEC c;  HA_ARENA a;  KHE_SOLN soln;  int i;  KHE_TASK task;
  KHE_RESOURCE_TYPE rt;  KHE_INSTANCE ins;  KHE_CONSEC_RESOURCE cr;

  /* make a consec object and add the tasks of tasking to it */
  if( DEBUG1 )
    fprintf(stderr, "[ KheResourcePackConsecutive(tasking, options)\n");
  soln = KheTaskingSoln(tasking);
  a = KheSolnArenaBegin(soln, false);
  c = KheConsecMake(soln, options, a);
  for( i = 0;  i < KheTaskingTaskCount(tasking);  i++ )
  {
    task = KheTaskingTask(tasking, i);
    if( KheTaskProperRoot(task) == task && KheTaskNeedsAssignment(task) )
      KheConsecAddTask(c, task);
  }

  /* sort the consec domain objects by increasing domain size */
  HaArraySort(c->tasks_by_domain, &KheConsecDomainCmp);

  /* add the resources and sort them by decreasing workload */
  rt = KheTaskingResourceType(tasking);
  if( rt == NULL )
  {
    ins = KheSolnInstance(soln);
    for( i = 0;  i < KheInstanceResourceTypeCount(ins);  i++ )
    {
      rt = KheInstanceResourceType(ins, i);
      KheConsecAddResources(c, rt);
    }
  }
  else
    KheConsecAddResources(c, rt);
  HaArraySort(c->resources, &KheConsecResourceCmp);

  if( DEBUG1 )
    KheConsecDebug(c, 2, 2, stderr);

  /* do the solving */
  HaArrayForEach(c->resources, cr, i)
    KheConsecSolveForResource(c, cr->resource, 0);

  /* clean up and exit */
  KheSolnArenaEnd(soln, a);
  if( DEBUG1 )
    fprintf(stderr, "] KheResourcePackConsecutive returning\n");
  return false;
}
