
/*****************************************************************************/
/*                                                                           */
/*  THE KHE HIGH SCHOOL TIMETABLING ENGINE                                   */
/*  COPYRIGHT (C) 2010 Jeffrey H. Kingston                                   */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 3, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         khe_sr_group_solver.c                                      */
/*  DESCRIPTION:  Group solvers                                              */
/*                                                                           */
/*****************************************************************************/
#include "khe_solvers.h"
#include <limits.h>
#define bool_show(x) ((x) ? "true" : "false")
#define min(a, b) ((a) < (b) ? (a) : (b))

/*****************************************************************************/
/*                                                                           */
/*  KHE_TIME_INFO, KHE_ROOT_TASK_SET, KHE_GROUP_SOLVER                       */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_root_task_set_rec *KHE_ROOT_TASK_SET;
typedef HA_ARRAY(KHE_ROOT_TASK_SET) ARRAY_KHE_ROOT_TASK_SET;

typedef struct khe_time_group_info_rec *KHE_TIME_GROUP_INFO; 
typedef HA_ARRAY(KHE_TIME_GROUP_INFO) ARRAY_KHE_TIME_GROUP_INFO;

typedef struct khe_time_info_rec *KHE_TIME_INFO; 
typedef HA_ARRAY(KHE_TIME_INFO) ARRAY_KHE_TIME_INFO;

typedef struct khe_group_solver_rec *KHE_GROUP_SOLVER;


/*****************************************************************************/
/*                                                                           */
/*  KHE_TIME_GROUP_INFO                                                      */
/*                                                                           */
/*  Everything about what is going on at some time group.                    */
/*                                                                           */
/*****************************************************************************/

struct khe_time_group_info_rec {
  KHE_GROUP_SOLVER		solver;		/* enclosing solver          */
  int				index;		/* index of self in solver   */
  KHE_GROUP_SOLVER_COVER_TYPE	cover_type;	/* cover type                */
  KHE_TIME_GROUP		time_group;	/* the time group            */
  int				cover_count;	/* varies during solve       */
};


/*****************************************************************************/
/*                                                                           */
/*  KHE_TIME_INFO                                                            */
/*                                                                           */
/*  Everything about what is going on at one time.                           */
/*                                                                           */
/*  The data structure described in the documentation is an array of         */
/*  KHE_TIME_INFO objects, one for each time of the cycle.                   */
/*                                                                           */
/*  If pg_active is true, then profile grouping is currently active at this  */
/*  time, and pg_profile is the value called n_i in the documentation.  Its  */
/*  value is the sum, over all root task sets s in the root_task_sets array  */
/*  whose pg_status is KHE_PG_ACTIVE, of the number of tasks in s.           */
/*                                                                           */
/*****************************************************************************/

struct khe_time_info_rec {
  KHE_GROUP_SOLVER	solver;			/* enclosing solver         */
  KHE_TIME		time;			/* the time                 */
  ARRAY_KHE_ROOT_TASK_SET root_task_sets;	/* root tasks at time       */
  bool			pg_active;		/* pg active now            */
  int			pg_profile;		/* profile at this time     */
};


/*****************************************************************************/
/*                                                                           */
/*  KHE_ROOT_TASK_SET                                                        */
/*                                                                           */
/*  A set of equivalent root tasks.                                          */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  KHE_PG_INACTIVE_UNMONITORED,	/* inactive because unmonitored              */
  KHE_PG_INACTIVE_MAXIMAL,	/* inactive because maximal intersection     */
  KHE_PG_ACTIVE			/* active (contributes to profile)           */
} KHE_PG_STATUS;

struct khe_root_task_set_rec {
  KHE_GROUP_SOLVER	solver;			/* enclosing solver         */
  KHE_TASK_SET		tasks;			/* the tasks                */
  KHE_RESOURCE_GROUP	domain;			/* their common domain      */
  KHE_RESOURCE		init_asst;		/* if assigned initially    */
  KHE_TIME_SET		time_set;		/* their common times       */
  ARRAY_KHE_TIME_INFO	time_info;		/* parallels time_set       */
  bool			ignore_for_leader;	/* if ignoring for root_rts */
  KHE_PG_STATUS		pg_status;		/* contributes to profile   */
};


/*****************************************************************************/
/*                                                                           */
/*  KHE_GROUP_SOLVER - solver for group by resource constraint solving       */
/*                                                                           */
/*****************************************************************************/

typedef HA_ARRAY(KHE_TIME_SET) ARRAY_KHE_TIME_SET;
typedef HA_ARRAY(KHE_TASK_SET) ARRAY_KHE_TASK_SET;
typedef HA_ARRAY(KHE_RESOURCE_GROUP) ARRAY_KHE_RESOURCE_GROUP;

struct khe_group_solver_rec {

  /* general fields, more or less constant */
  HA_ARENA			arena;
  KHE_SOLN			soln;
  KHE_INSTANCE			instance;
  KHE_RESOURCE_TYPE		resource_type;
  bool				soln_changed;

  /* what's happening at each time */
  ARRAY_KHE_TIME_INFO		time_info;
  ARRAY_KHE_ROOT_TASK_SET	free_root_task_sets;

  /* time groups info */
  ARRAY_KHE_TIME_GROUP_INFO	time_group_info;

  /* parameters of KheGroupSolverSolve */
  int				max_num;	/* max number of groups     */
  KHE_GROUP_SOLVER_COST_TYPE	cost_type;	/* cost type                */
  KHE_RESOURCE_GROUP		domain;		/* domain of leader task    */
  bool				allow_single;	/* allow single tasks       */
  KHE_TASK_SET			task_set;	/* tasks with changed assts */
  char				*debug_str;	/* type of solve for debug  */

  /* time-based grouping */
  ARRAY_KHE_ROOT_TASK_SET	tbg_rts_set;
  ARRAY_KHE_ROOT_TASK_SET	tbg_ignore_for_leader_set;
  KHE_TIME_SET			tbg_in_ts;
  KHE_TIME_SET			tbg_out_ts;

  /* combinatorial grouping */
  ARRAY_KHE_CG_TIME_GROUP_INFO	cg_time_group_info;
  int				cg_max_days;
  int				cg_first_index;
  int				cg_last_index;
  KHE_TIME_SET			cg_in_ts;
  KHE_TIME_SET			cg_out_ts;
  int				cg_success_count;
  KHE_TIME_SET			cg_success_in_ts;
  KHE_TIME_SET			cg_success_out_ts;
  KHE_TRACE			cg_trace;
};


/*****************************************************************************/
/*                                                                           */
/*  Submodule "time group info objects"                                      */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  char *CoverTypeShow(KHE_GROUP_SOLVER_COVER_TYPE cover_type)              */
/*                                                                           */
/*  Display a cover type.                                                    */
/*                                                                           */
/*****************************************************************************/

static char *CoverTypeShow(KHE_GROUP_SOLVER_COVER_TYPE cover_type)
{
  switch( cover_type )
  {
    case KHE_GROUP_SOLVER_COVER_YES:	return "cover_yes";
    case KHE_GROUP_SOLVER_COVER_NO:	return "cover_no";
    case KHE_GROUP_SOLVER_COVER_PREV:	return "cover_prev";

    default:

      HnAbort("CoverTypeShow: unknown cover type (%d)", cover_type);
      return NULL;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_TIME_GROUP_INFO KheTimeGroupInfoMake(KHE_GROUP_SOLVER gs,            */
/*    int index, KHE_GROUP_SOLVER_COVER_TYPE cover_type, KHE_TIME_GROUP tg)  */
/*                                                                           */
/*  Make a new time group info object with these attributes.                 */
/*                                                                           */
/*****************************************************************************/

static KHE_TIME_GROUP_INFO KheTimeGroupInfoMake(KHE_GROUP_SOLVER gs,
  int index, KHE_GROUP_SOLVER_COVER_TYPE cover_type, KHE_TIME_GROUP tg)
{
  KHE_TIME_GROUP_INFO res;
  HaMake(res, gs->arena);
  res->solver = gs;
  res->index = index;
  res->cover_type = cover_type;
  res->time_group = tg;
  res->cover_count = 0;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheTimeGroupInfoDebug(KHE_TIME_GROUP_INFO tgi, int verbosity,       */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Make a debug print of tgi onto fp with the given verbosity and indent.   */
/*                                                                           */
/*****************************************************************************/

static void KheTimeGroupInfoDebug(KHE_TIME_GROUP_INFO tgi, int verbosity,
  int indent, FILE *fp)
{
  fprintf(fp, "%*sTimeGroup(%s, %s, curr %d)\n", indent, "",
    KheTimeGroupId(tgi->time_group), CoverTypeShow(tgi->cover_type),
    tgi->cover_count);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "time info objects"                                            */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_TIME_INFO KheTimeInfoMake(KHE_GROUP_SOLVER gs, KHE_TIME t)           */
/*                                                                           */
/*  Make a new time info object for t.                                       */
/*                                                                           */
/*****************************************************************************/

static KHE_TIME_INFO KheTimeInfoMake(KHE_GROUP_SOLVER gs, KHE_TIME t)
{
  KHE_TIME_INFO res;
  HaMake(res, gs->arena);
  res->solver = gs;
  res->time = t;
  HaArrayInit(res->root_task_sets, gs->arena);
  res->pg_active = false;
  res->pg_profile = -1;  /* undefined unless pg_active */
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheTimeInfoAddRootTaskSet(KHE_TIME_INFO ti, KHE_ROOT_TASK_SET rts)  */
/*                                                                           */
/*  Add rts to ti, at a suitable spot for its domain and assigned resource.  */
/*                                                                           */
/*  Implementation note.  This function is called only when rts has no       */
/*  tasks, and so rts has no implications for time info profiles.            */
/*                                                                           */
/*****************************************************************************/
static bool KheRootTaskSetLT(KHE_ROOT_TASK_SET rts1, KHE_ROOT_TASK_SET rts2);

static void KheTimeInfoAddRootTaskSet(KHE_TIME_INFO ti, KHE_ROOT_TASK_SET rts)
{
  int i;  KHE_ROOT_TASK_SET rts2;
  for( i = 0;  i < HaArrayCount(ti->root_task_sets);  i++ )
  {
    rts2 = HaArray(ti->root_task_sets, i);
    if( KheRootTaskSetLT(rts, rts2) )
      break;
  }
  HaArrayAdd(ti->root_task_sets, i, rts);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheTimeInfoDebug(KHE_TIME_INFO ti, int indent, FILE *fp)            */
/*                                                                           */
/*  Debug print of ti with the given indent.                                 */
/*                                                                           */
/*****************************************************************************/
static void KheRootTaskSetDebug(KHE_ROOT_TASK_SET rts, int indent, FILE *fp);

static void KheTimeInfoDebug(KHE_TIME_INFO ti, int indent, FILE *fp)
{
  KHE_ROOT_TASK_SET rts;  int i;
  fprintf(fp, "%*s[ TimeInfo(%s", indent, "", KheTimeId(ti->time));
  if( ti->pg_active )
    fprintf(fp, ", pg_profile %d", ti->pg_profile);
  fprintf(fp, ")\n");
  HaArrayForEach(ti->root_task_sets, rts, i)
    KheRootTaskSetDebug(rts, indent + 2, fp);
  fprintf(fp, "%*s]\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "root task sets"                                               */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_PG_STATUS KheRootTaskSetStatus(KHE_ROOT_TASK_SET rts)                */
/*                                                                           */
/*  Return the profile grouping status of rts.                               */
/*                                                                           */
/*  Implementation note.  When this function is called when gs->pg_active,   */
/*  it is only called on root task sets with a non-empty intersection with   */
/*  gs->pg_monitor_ts, hence the assertion to this effect.                   */
/*                                                                           */
/*****************************************************************************/

KHE_PG_STATUS KheRootTaskSetStatus(KHE_ROOT_TASK_SET rts)
{
  KHE_GROUP_SOLVER gs;  int count;
  gs = rts->solver;
  if( gs->pg_active )
  {
    count = KheTimeSetIntersectCount(gs->pg_monitor_ts, rts->time_set);
    HnAssert(count > 0, "KheRootTaskSetStatus internal error");
    if( count >= gs->pg_max_limit )
      return KHE_PG_INACTIVE_MAXIMAL;
    else
      return KHE_PG_ACTIVE;
  }
  else
    return KHE_PG_INACTIVE_UNMONITORED;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_ROOT_TASK_SET KheRootTaskSetMake(KHE_GROUP_SOLVER gs, KHE_TASK task) */
/*                                                                           */
/*  Make a new root task set suitable for holding task (but do not add task  */
/*  to it), and add it to the appropriate time info objects.                 */
/*                                                                           */
/*  Implementation note.  Because the new object contains no tasks when      */
/*  it returns, its presence has no implications for time info profiles.     */
/*                                                                           */
/*****************************************************************************/

static KHE_ROOT_TASK_SET KheRootTaskSetMake(KHE_GROUP_SOLVER gs, KHE_TASK task)
{
  KHE_ROOT_TASK_SET res;  int i;  KHE_TIME t;  KHE_TIME_INFO ti;  
  if( HaArrayCount(gs->free_root_task_sets) > 0 )
  {
    res = HaArrayLastAndDelete(gs->free_root_task_sets);
    KheTaskSetClear(res->tasks);
    KheTimeSetClear(res->time_set);
    HaArrayClear(res->time_info);
  }
  else
  {
    HaMake(res, gs->arena);
    res->tasks = KheTaskSetMake(gs->soln);
    res->time_set = KheTimeSetMake(KheSolnInstance(gs->soln), gs->arena);
    HaArrayInit(res->time_info, gs->arena);
  }
  res->solver = gs;
  res->domain = KheTaskDomain(task);
  res->init_asst = KheTaskAsstResource(task);
  KheTimeSetAddTaskTimes(res->time_set, task);
  for( i = 0;  i < KheTimeSetTimeCount(res->time_set);  i++ )
  {
    t = KheTimeSetTime(res->time_set, i);
    ti = HaArray(gs->time_info, KheTimeIndex(t));
    HaArrayAddLast(res->time_info, ti);
    KheTimeInfoAddRootTaskSet(ti, res);
  }
  res->ignore_for_leader = false;
  res->pg_status = KheRootTaskSetStatus(res);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheRootTaskSetDelete(KHE_ROOT_TASK_SET rts)                         */
/*                                                                           */
/*  Delete rts from its enclosing solver.                                    */
/*                                                                           */
/*  Implementation note.  Here rts must be empty before it is deleted,       */
/*  and so the deletion has no consequences for profile grouping.            */
/*                                                                           */
/*****************************************************************************/

static void KheRootTaskSetDelete(KHE_ROOT_TASK_SET rts)
{
  int i, j;  KHE_TIME_INFO ti;  KHE_ROOT_TASK_SET rts2;
  HnAssert(KheTaskSetTaskCount(rts->tasks) == 0,
    "KheRootTaskSetDelete internal error");
  HaArrayForEach(rts->time_info, ti, i)
  {
    HaArrayForEach(ti->root_task_sets, rts2, j)
      if( rts2 == rts )
	break;
    HnAssert(j < HaArrayCount(ti->root_task_sets),
      "KheRootTaskSetDelete internal error 1");
    HaArrayDeleteAndShift(ti->root_task_sets, j);
  }
  HaArrayAddLast(rts->solver->free_root_task_sets, rts);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheRootTaskSetLT(KHE_ROOT_TASK_SET rts1, KHE_ROOT_TASK_SET rts2)    */
/*                                                                           */
/*  Return true if rts1 comes earlier in sequence than rts2.                 */
/*                                                                           */
/*****************************************************************************/

static bool KheRootTaskSetLT(KHE_ROOT_TASK_SET rts1, KHE_ROOT_TASK_SET rts2)
{
  int rts1_val, rts2_val;
  rts1_val = (rts1->init_asst != NULL ? 0 : 1);
  rts2_val = (rts2->init_asst != NULL ? 0 : 1);
  return rts1_val != rts2_val ? rts1_val < rts2_val :
    KheResourceGroupResourceCount(rts1->domain) <
    KheResourceGroupResourceCount(rts2->domain);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheRootTaskSetAcceptsTask(KHE_ROOT_TASK_SET rts, KHE_TASK task)     */
/*                                                                           */
/*  Return true if task is equivalent to the tasks already in rts, and       */
/*  so could be added to it.                                                 */
/*                                                                           */
/*****************************************************************************/

static bool KheRootTaskSetAcceptsTask(KHE_ROOT_TASK_SET rts, KHE_TASK task)
{
  return KheTaskAsstResource(task) == rts->init_asst &&
    KheTaskEquivalent(KheTaskSetTask(rts->tasks, 0), task);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheRootTaskSetAddTask(KHE_ROOT_TASK_SET rts, KHE_TASK task)         */
/*                                                                           */
/*  Add task to rts, assuming KheRootTaskSetAcceptsTask(rts, task) holds.    */
/*                                                                           */
/*****************************************************************************/

static void KheRootTaskSetAddTask(KHE_ROOT_TASK_SET rts, KHE_TASK task)
{
  KHE_TIME_INFO ti;  int i;
  HnAssert(KheTaskAsst(task)==NULL, "KheRootTaskSetAddTask: task is assigned");
  KheTaskSetAddTask(rts->tasks, task);
  if( rts->pg_status == KHE_PG_ACTIVE )
    HaArrayForEach(rts->time_info, ti, i)
      if( ti->pg_active )
        ti->pg_profile++;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_TASK KheRootTaskSetLastAndDelete(KHE_ROOT_TASK_SET rts)              */
/*                                                                           */
/*  Delete and return the last task of rts.  If this causes rts to become    */
/*  empty, then delete rts from the enclosing solver.                        */
/*                                                                           */
/*****************************************************************************/

static KHE_TASK KheRootTaskSetLastAndDelete(KHE_ROOT_TASK_SET rts)
{
  int i;  KHE_TIME_INFO ti;  KHE_TASK res;
  res = KheTaskSetLastAndDelete(rts->tasks);
  if( rts->pg_status == KHE_PG_ACTIVE )
    HaArrayForEach(rts->time_info, ti, i)
      if( ti->pg_active )
	ti->pg_profile--;
  if( KheTaskSetTaskCount(rts->tasks) == 0 )
    KheRootTaskSetDelete(rts);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheRootTaskSetDebug(KHE_ROOT_TASK_SET rts, int indent, FILE *fp)    */
/*                                                                           */
/*  Debug print of rts with the given indent.                                */
/*                                                                           */
/*****************************************************************************/

static void KheRootTaskSetDebug(KHE_ROOT_TASK_SET rts, int indent, FILE *fp)
{
  fprintf(fp, "%*s[ RootTaskSet(%s", indent, "",
    rts->pg_status == KHE_PG_INACTIVE_UNMONITORED ? "" :
    rts->pg_status == KHE_PG_INACTIVE_MAXIMAL ? "maximal, " : "active, ");
  KheResourceGroupDebug(rts->domain, 1, -1, fp);
  if( rts->init_asst != NULL )
    fprintf(fp, " := %s", KheResourceId(rts->init_asst));
  fprintf(fp, ", ");
  KheTimeSetDebug(rts->time_set, 1, -1, fp);
  fprintf(fp, "%d tasks): ", KheTaskSetTaskCount(rts->tasks));
  KheTaskSetDebug(rts->tasks, 2, -1, fp);
  fprintf(fp, " ]\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "group solvers"                                                */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool KheGroupSolverAddTask(KHE_GROUP_SOLVER gs, KHE_TASK task)           */
/*                                                                           */
/*  Add task to gs and return true, or if task has no assigned time, do      */
/*  nothing.  Here task must be a proper root.                               */
/*                                                                           */
/*****************************************************************************/

static bool KheGroupSolverAddTask(KHE_GROUP_SOLVER gs, KHE_TASK task)
{
  KHE_MEET meet;  KHE_TIME t;  int i;  KHE_TIME_INFO ti;  KHE_ROOT_TASK_SET rts;
  HnAssert(KheTaskAsst(task) == NULL, "KheGroupSolverAddTask internal error");
  meet = KheTaskMeet(task);
  if( meet != NULL )
  {
    t = KheMeetAsstTime(meet);
    if( t != NULL )
    {
      /* search for a set of equivalent root tasks */
      ti = HaArray(gs->time_info, KheTimeIndex(t));
      rts = NULL;  /* keep compiler happy */
      HaArrayForEach(ti->root_task_sets, rts, i)
	if( KheRootTaskSetAcceptsTask(rts, task) )
	  break;

      /* if no equivalent tasks, make a new set (initially empty) */
      if( i >= HaArrayCount(ti->root_task_sets) )
	rts = KheRootTaskSetMake(gs, task);

      /* add task to the set (old or new) and return true */
      KheRootTaskSetAddTask(rts, task);
      return true;
    }
  }
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_GROUP_SOLVER KheGroupByResourceConstraintsSolverMake(                */
/*    KHE_SOLN soln, KHE_RESOURCE_TYPE rt, int max_days,                     */
/*    bool resource_invariant, KHE_FRAME common_frame,                       */
/*    KHE_EVENT_TIMETABLE_MONITOR etm, KHE_TASK_SET ts, HA_ARENA a)          */
/*                                                                           */
/*  Make a new group by resource constraints solver object with these        */
/*  attributes.  Add all tasks which are either unassigned or assigned       */
/*  directly to cycle tasks (that is, tasks which are proper roots).         */
/*                                                                           */
/*****************************************************************************/

static KHE_GROUP_SOLVER KheGroupByResourceConstraintsSolverMake(
  KHE_SOLN soln, KHE_RESOURCE_TYPE rt, int max_days,
  KHE_TASK_SET ts, HA_ARENA a)
{
  KHE_GROUP_SOLVER res;  KHE_TIME_GROUP tg;  KHE_TASK task;  KHE_TIME t;
  int i;  KHE_CG_TIME_GROUP_INFO tgi;  KHE_TIME_INFO ti;  KHE_EVENT_RESOURCE er;

  /* the solve as a whole */
  HaMake(res, a);
  res->arena = a;
  res->soln = soln;
  res->instance = KheSolnInstance(soln);
  /* res->resource_invariant = resource_invariant; */
  /* res->common_frame = common_frame; */
  /* res->etm = etm; */
  res->resource_type = rt;
  /* res->resource = NULL; */
  res->soln_changed = false;
  res->task_set = ts;

  /* time info */
  HaArrayInit(res->time_info, a);
  for( i = 0;  i < KheInstanceTimeCount(res->instance);  i++ )
  {
    t = KheInstanceTime(res->instance, i);
    ti = KheTimeInfoMake(res, t);
    HaArrayAddLast(res->time_info, ti);
  }
  HaArrayInit(res->free_root_task_sets, a);

  /* time-based grouping */
  HaArrayInit(res->tbg_rts_set, a);
  HaArrayInit(res->tbg_ignore_for_leader_set, a);
  res->tbg_in_ts = KheTimeSetMake(res->instance, a);
  res->tbg_out_ts = KheTimeSetMake(res->instance, a);

  /* add unassigned tasks of the right type to time group info */
  for( i = 0;  i < KheSolnTaskCount(soln);  i++ )
  {
    task = KheSolnTask(soln, i);
    if( !KheTaskIsCycleTask(task) && KheTaskResourceType(task) == rt &&
	(KheTaskAsst(task) == NULL || KheTaskIsCycleTask(KheTaskAsst(task))) )
    {
      er = KheTaskEventResource(task);
      if( er != NULL && KheEventResourceNeedsAssignment(er) == KHE_YES )
	KheGroupSolverAddTask(res, task);
    }
  }

  /* combinatorial grouping */
  res->cg_max_days = max_days;
  res->cg_first_index = -1;
  res->cg_last_index = -1;
  res->cg_in_ts = KheTimeSetMake(res->instance, a);
  res->cg_out_ts = KheTimeSetMake(res->instance, a);
  res->cg_success_count = 0;
  res->cg_success_in_ts = KheTimeSetMake(res->instance, a);
  res->cg_success_out_ts = KheTimeSetMake(res->instance, a);
  res->cg_trace = KheTraceMake((KHE_GROUP_MONITOR) soln);

  /* all done */
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheGroupSolverDebug(KHE_GROUP_SOLVER gs, int indent, FILE *fp)      */
/*                                                                           */
/*  Debug print of group solver gs onto fp with the given indent.            */
/*                                                                           */
/*****************************************************************************/

static void KheGroupSolverDebug(KHE_GROUP_SOLVER gs, int indent, FILE *fp)
{
  int i;  KHE_CG_TIME_GROUP_INFO tgi;  KHE_TIME_INFO ti;
  fprintf(fp, "%*s[ %s(%s, max_days %d, rs_invt %s)\n", 
    indent, "", "GroupByResourceConstraintsSolver", KheInstanceId(gs->instance),
    gs->cg_max_days, bool_show(gs->resource_invariant));
  fprintf(fp, "%*s[ time info:\n", indent + 2, "");
  HaArrayForEach(gs->time_info, ti, i)
    KheTimeInfoDebug(ti, indent + 4, fp);
  fprintf(fp, "%*s]\n", indent + 2, "");
  fprintf(fp, "%*s]\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "time-based grouping"                                          */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool KheFindLeaderRootTaskSet(KHE_GROUP_SOLVER gs,                       */
/*    KHE_RESOURCE_GROUP domain, bool allow_single,                          */
/*    KHE_ROOT_TASK_SET *res)                                                */
/*                                                                           */
/*  Find a leader root task set from the times of gs->tbg_in_ts.  This must  */
/*                                                                           */
/*    * Not be marked unsuitable for providing leader tasks                  */
/*    * Cover at least one of the times of gs->tbg_in_ts                     */
/*    * Have domain equal to domain, unless domain == NULL                   */
/*    * Not cover all of the times of gs->tbg_in_ts, unless allow_single     */
/*    * Not cover any of the times of gs->tbg_out_ts                         */
/*                                                                           */
/*  Among all such root task sets, give first priority to those with an      */
/*  initial assignment, and second priority to increasing domain size.       */
/*                                                                           */
/*  If a suitable leader root task set can be found, set *res to it and      */
/*  return true.  Otherwise set *res to NULL and return false.               */
/*                                                                           */
/*****************************************************************************/

static bool KheFindLeaderRootTaskSet(KHE_GROUP_SOLVER gs,
  KHE_RESOURCE_GROUP domain, bool allow_single, KHE_ROOT_TASK_SET *res)
{
  int i, j;  KHE_TIME t;  KHE_TIME_INFO ti;  KHE_ROOT_TASK_SET rts;
  if( DEBUG5 )
  {
    fprintf(stderr, "          [ KheFindLeaderRootTaskSet(in ");
    KheTimeSetDebug(gs->tbg_in_ts, 2, -1, stderr);
    fprintf(stderr, ", out ");
    KheTimeSetDebug(gs->tbg_out_ts, 2, -1, stderr);
    fprintf(stderr, ", %s, %s)\n",
      domain == NULL ? "-" : KheResourceGroupId(domain),
      bool_show(allow_single));
  }
  *res = NULL;
  for( i = 0;  i < KheTimeSetTimeCount(gs->tbg_in_ts);  i++ )
  {
    t = KheTimeSetTime(gs->tbg_in_ts, i);
    ti = HaArray(gs->time_info, KheTimeIndex(t));
    HaArrayForEach(ti->root_task_sets, rts, j)
      if( !rts->ignore_for_leader &&
	  (domain == NULL || KheResourceGroupEqual(domain, rts->domain)) &&
	  (allow_single || !KheTimeSetSubset(gs->tbg_in_ts, rts->time_set)) &&
	  KheTimeSetDisjoint(gs->tbg_out_ts, rts->time_set) &&
	  (*res == NULL || KheRootTaskSetLT(rts, *res)) )
	*res = rts;
  }
  if( DEBUG5 || DEBUG8 )
  {
    fprintf(stderr, "            leader:  ");
    if( *res != NULL )
      KheRootTaskSetDebug(*res, 0, stderr);
    else
      fprintf(stderr, "none\n");
    if( DEBUG5 )
      fprintf(stderr, "          ] KheFindLeaderRootTaskSet returning %s\n",
	bool_show(*res != NULL));
  }
  return *res != NULL;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheFindFollowerRootTaskSet(KHE_GROUP_SOLVER gs,                     */
/*    KHE_ROOT_TASK_SET root_rts, KHE_RESOURCE init_asst,                    */
/*    KHE_ROOT_TASK_SET *res, KHE_ROOT_TASK_SET *problem_rts)                */
/*                                                                           */
/*  Assuming that gs->tbg_in_ts contains at least one time, find a           */
/*  follower task set from the first time of gs->tbg_in_ts.  This must       */
/*                                                                           */
/*    * Have init_asst (possibly NULL) for its initial assignment            */
/*    * Cover at least one of the times of gs->tbg_in_ts                     */
/*    * Not cover any of the times of gs->tbg_out_ts                         */
/*    * Be movable to root_rts                                               */
/*                                                                           */
/*  If a suitable follower root task set can be found, set *res to it and    */
/*  return true.  Otherwise set *res to NULL and return false.               */
/*                                                                           */
/*  If we come upon a root task set with a non-NULL initial assignment       */
/*  that is equal to init_asst, we really want to choose it.  If we can't    */
/*  choose it, we set *problem_rts to it to tell the caller of the problem.  */
/*                                                                           */
/*****************************************************************************/

static bool KheFindFollowerRootTaskSet(KHE_GROUP_SOLVER gs,
  KHE_ROOT_TASK_SET root_rts, KHE_RESOURCE init_asst,
  KHE_ROOT_TASK_SET *res, KHE_ROOT_TASK_SET *problem_rts)
{
  int i;  KHE_TIME t;  KHE_TIME_INFO ti;  KHE_ROOT_TASK_SET rts;
  KHE_TASK root_task;
  t = KheTimeSetTime(gs->tbg_in_ts, 0);
  ti = HaArray(gs->time_info, KheTimeIndex(t));
  root_task =  KheTaskSetTask(root_rts->tasks, 0);
  HaArrayForEach(ti->root_task_sets, rts, i)
  {
    if( rts->init_asst == init_asst )
    {
      if( KheTimeSetDisjoint(gs->tbg_out_ts, rts->time_set) &&
	  KheTaskMoveCheck(KheTaskSetTask(rts->tasks, 0), root_task) )
      {
	if( DEBUG8 )
	{
	  fprintf(stderr, "            follower: ");
	  KheRootTaskSetDebug(rts, 0, stderr);
	}
	return *res = rts, true;
      }
      if( init_asst != NULL )
      {
	/* this is a problem, we really want to assign rts in this case */
	if( DEBUG8 )
	{
	  fprintf(stderr, "            problem: ");
	  KheRootTaskSetDebug(rts, 0, stderr);
	}
	*problem_rts = rts;
      }
    }
  }
  if( DEBUG8 )
    fprintf(stderr, "            follower: NULL\n");
  return *res = NULL, false;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheInitializeTimeSets(KHE_GROUP_SOLVER gs, KHE_TIME_SET in_ts,      */
/*    KHE_TIME_SET out_ts)                                                   */
/*                                                                           */
/*  Initialize gs->tbg_in_ts and gs->tbg_out_ts to in_ts and out_ts plus,    */
/*  for each time t in in_ts, the times of the common frame time group of    */
/*  t other than t itself.                                                   */
/*                                                                           */
/*****************************************************************************/

static void KheInitializeTimeSets(KHE_GROUP_SOLVER gs, KHE_TIME_SET in_ts,
  KHE_TIME_SET out_ts)
{
  int i;  KHE_TIME t;  KHE_TIME_GROUP tg;
  KheTimeSetCopyElements(gs->tbg_in_ts, in_ts);
  KheTimeSetCopyElements(gs->tbg_out_ts, out_ts);
  for( i = 0;  i < KheTimeSetTimeCount(in_ts);  i++ )
  {
    t = KheTimeSetTime(in_ts, i);
    tg = KheFrameTimeTimeGroup(gs->common_frame, t);
    KheTimeSetAddTimeGroup(gs->tbg_out_ts, tg);
    KheTimeSetDeleteTime(gs->tbg_out_ts, t);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheUpdateTimeSets(KHE_GROUP_SOLVER gs, KHE_ROOT_TASK_SET rts)       */
/*                                                                           */
/*  Update gs->tbg_in_ts and gs->tbg_out_ts to reflect the inclusion of      */
/*  rts->times.  That is, remove rts->times from gs->tbg_in_ts, and add      */
/*  every day covered by the times of rts->times from gs->tbg_out_ts.        */
/*                                                                           */
/*****************************************************************************/

static void KheUpdateTimeSets(KHE_GROUP_SOLVER gs, KHE_ROOT_TASK_SET rts)
{
  int i;  KHE_TIME t;  KHE_TIME_GROUP tg;
  KheTimeSetDifference(gs->tbg_in_ts, rts->time_set);
  for( i = 0;  i < KheTimeSetTimeCount(rts->time_set);  i++ )
  {
    t = KheTimeSetTime(rts->time_set, i);
    tg = KheFrameTimeTimeGroup(gs->common_frame, t);
    KheTimeSetAddTimeGroup(gs->tbg_out_ts, tg);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheTimeBasedGrouping(KHE_GROUP_SOLVER gs, KHE_TIME_SET in_ts,       */
/*    KHE_TIME_SET out_ts, KHE_RESOURCE_GROUP leader_domain, int max_num,    */
/*    bool allow_single, KHE_TASK *test_task, int *groups_made)              */
/*                                                                           */
/*  Try to ensure that there are up to max_num groups of tasks such that     */
/*  in each group, each time of in_ts is covered, and no time of out_ts is   */
/*  covered.  Existing groups with these properties count towards max_num.   */
/*  Return true if at least one group was made, and set *groups_made to      */
/*  the number of groups actually made.                                      */
/*                                                                           */
/*  If leader_domain != NULL, then only tasks whose domain is equal to       */
/*  leader_domain are considered for being leader tasks.                     */
/*                                                                           */
/*  If allow_single is true, it is permissible for a group to contain        */
/*  just a single task.                                                      */
/*                                                                           */
/*  If test_task is non-NULL, then if at least one group was made, set       */
/*  *test_task to its root task, otherwise set *test_task to NULL.  Also     */
/*  do not update the data structure and do not print debug output saying    */
/*  that a group was made, because this is just a test and it will be        */
/*  unmade shortly after this function returns.                              */
/*                                                                           */
/*  Time set in_ts will never contain two times from the same time group     */
/*  of the common frame, and time sets in_ts and out_ts will never overlap.  */
/*  It does not matter whether the grouped tasks run at times outside in_ts  */
/*  and out_ts or not.  However, no two of them may run at the same time,    */
/*  or at two times from the same time group of the common frame.            */
/*                                                                           */
/*  It is acceptable for in_ts to contain only one time.  This can produce   */
/*  no true grouping, but it is allowed because combinatorial grouping       */
/*  needs to know whether assigning a single time is cost-free.              */
/*                                                                           */
/*****************************************************************************/

static bool KheTimeBasedGrouping(KHE_GROUP_SOLVER gs, KHE_TIME_SET in_ts,
  KHE_TIME_SET out_ts, KHE_RESOURCE_GROUP leader_domain, int max_num,
  bool allow_single, KHE_TASK *test_task, int *groups_made, char *what)
{
  KHE_ROOT_TASK_SET root_rts, rts, problem_rts;  KHE_TIME t;  KHE_TIME_INFO ti;
  KHE_TASK root_task, task;  int i, j, count, groups_count;
  if( DEBUG5 || DEBUG8 )
  {
    fprintf(stderr, "        [ KheTimeBasedGrouping(gs, in ");
    KheTimeSetDebug(in_ts, 2, -1, stderr);
    fprintf(stderr, ", out ");
    KheTimeSetDebug(out_ts, 2, -1, stderr);
    fprintf(stderr, ", %s, max %d)\n",
      leader_domain == NULL ? "-" : KheResourceGroupId(leader_domain), max_num);
    if( DEBUG5 )
      for( i = 0;  i < KheTimeSetTimeCount(in_ts);  i++ )
      {
	t = KheTimeSetTime(in_ts, i);
	ti = HaArray(gs->time_info, KheTimeIndex(t));
	KheTimeInfoDebug(ti, 10, stderr);
      }
  }
  /* ***
  HnAssert(KheTimeSetTimeCount(in_ts) >= 2,
    "KheTimeBasedGrouping: in_ts has less than 2 elements");
  *** */
  root_task = NULL;
  *groups_made = 0;
  HaArrayClear(gs->tbg_ignore_for_leader_set);
  KheInitializeTimeSets(gs, in_ts, out_ts);
  root_rts = NULL;  /* keep compiler happy */
  while( max_num > 0 &&
    KheFindLeaderRootTaskSet(gs, leader_domain, allow_single, &root_rts) )
  {
    /* update time sets for root_rts */
    KheUpdateTimeSets(gs, root_rts);

    /* find the follower root task sets */
    HaArrayClear(gs->tbg_rts_set);
    problem_rts = NULL;
    if( root_rts->init_asst != NULL )
    {
      while( KheTimeSetTimeCount(gs->tbg_in_ts) > 0 &&
	  KheFindFollowerRootTaskSet(gs, root_rts, root_rts->init_asst, &rts,
	    &problem_rts) )
      {
	HaArrayAddLast(gs->tbg_rts_set, rts);
	KheUpdateTimeSets(gs, rts);
      }
    }
    while( KheTimeSetTimeCount(gs->tbg_in_ts) > 0 &&
	KheFindFollowerRootTaskSet(gs, root_rts, NULL, &rts, &problem_rts) )
    {
      HaArrayAddLast(gs->tbg_rts_set, rts);
      KheUpdateTimeSets(gs, rts);
    }

    if( problem_rts == NULL && KheTimeSetTimeCount(gs->tbg_in_ts) == 0 )
    {
      /* success; find the number of groups to make */
      groups_count = max_num;
      count = KheTaskSetTaskCount(root_rts->tasks);
      if( count < groups_count )
	groups_count = count;
      HaArrayForEach(gs->tbg_rts_set, rts, i)
      {
	count = KheTaskSetTaskCount(rts->tasks);
	if( count < groups_count )
	  groups_count = count;
      }
      HnAssert(groups_count >= 1, "KheTimeBasedGrouping internal error 1");

      /* make the groups */
      if( test_task != NULL )
      {
	/* grouping for testing only; don't update data structures */
	HnAssert(max_num == 1, "KheTimeBasedGrouping internal error 2");
	root_task = KheTaskSetLast(root_rts->tasks);
	HaArrayForEach(gs->tbg_rts_set, rts, j)
	{
	  task = KheTaskSetLast(rts->tasks);
	  if( !KheTaskMove(task, root_task) )
	    HnAbort("KheTimeBasedGrouping internal error 3");
	}
	*test_task = root_task;
	*groups_made += 1;
      }
      else
      {
	/* grouping for real, including updating data structures */
	for( i = 0;  i < groups_count;  i++ )
	{
	  root_task = KheRootTaskSetLastAndDelete(root_rts);
	  HaArrayForEach(gs->tbg_rts_set, rts, j)
	  {
	    task = KheRootTaskSetLastAndDelete(rts);
	    if( !KheTaskMove(task, root_task) )
	      HnAbort("KheTimeBasedGrouping internal error 4");
	    gs->soln_changed = true;
	    if( gs->task_set != NULL )
	      KheTaskSetAddTask(gs->task_set, task);
	  }
	  if( DEBUG3 )
	  {
	    fprintf(stderr, "  %s made grouped task (", what);
	    KheResourceGroupDebug(KheTaskDomain(root_task), 1, -1, stderr);
	    fprintf(stderr, ") ");
	    KheTaskDebug(root_task, 2, 0, stderr);
	  }
	  KheGroupSolverAddTask(gs, root_task);
	  *groups_made += 1;
	}
      }

      /* update max_num */
      max_num -= groups_count;
    }
    else
    {
      /* problem with root_rts, so ignore it and retry */
      root_rts->ignore_for_leader = true;
      HaArrayAddLast(gs->tbg_ignore_for_leader_set, root_rts);
    }
    KheInitializeTimeSets(gs, in_ts, out_ts);
    root_rts = NULL;  /* keep compiler happy */
  }

  /* all done */
  /* NB some of the root task sets visited here may be on the free list */
  HaArrayForEach(gs->tbg_ignore_for_leader_set, rts, i)
    rts->ignore_for_leader = false;
  if( DEBUG5 || DEBUG8 )
    fprintf(stderr, "        ] KheTimeBasedGrouping returning %s\n",
      root_task != NULL ? "true" : "false");
  return root_task != NULL;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "properties of constraints"                                    */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  int KheLimitBusyTimesConstraintResourceTypeCount(                        */
/*    KHE_LIMIT_BUSY_TIMES_CONSTRAINT c, KHE_RESOURCE_TYPE rt)               */
/*                                                                           */
/*  Return the number of resources of type rt monitored by c.                */
/*                                                                           */
/*****************************************************************************/

static int KheLimitBusyTimesConstraintResourceTypeCount(
  KHE_LIMIT_BUSY_TIMES_CONSTRAINT c, KHE_RESOURCE_TYPE rt)
{
  int i, count;  KHE_RESOURCE_GROUP rg;  KHE_RESOURCE r;
  count = 0;
  for( i = 0;  i < KheLimitBusyTimesConstraintResourceCount(c);  i++ )
  {
    r = KheLimitBusyTimesConstraintResource(c, i);
    if( KheResourceResourceType(r) == rt )
      count += 1;
  }
  for( i = 0;  i < KheLimitBusyTimesConstraintResourceGroupCount(c);  i++ )
  {
    rg = KheLimitBusyTimesConstraintResourceGroup(c, i);
    if( KheResourceGroupResourceType(rg) == rt )
      count += KheResourceGroupResourceCount(rg);
  }
  return count;
}


/*****************************************************************************/
/*                                                                           */
/*  int KheClusterBusyTimesConstraintResourceTypeCount(                      */
/*    KHE_CLUSTER_BUSY_TIMES_CONSTRAINT c, KHE_RESOURCE_TYPE rt)             */
/*                                                                           */
/*  Return the number of resources of type rt monitored by c.                */
/*                                                                           */
/*****************************************************************************/

static int KheClusterBusyTimesConstraintResourceTypeCount(
  KHE_CLUSTER_BUSY_TIMES_CONSTRAINT c, KHE_RESOURCE_TYPE rt)
{
  int i, count;  KHE_RESOURCE_GROUP rg;  KHE_RESOURCE r;
  count = 0;
  for( i = 0;  i < KheClusterBusyTimesConstraintResourceCount(c);  i++ )
  {
    r = KheClusterBusyTimesConstraintResource(c, i);
    if( KheResourceResourceType(r) == rt )
      count += 1;
  }
  for( i = 0;  i < KheClusterBusyTimesConstraintResourceGroupCount(c);  i++ )
  {
    rg = KheClusterBusyTimesConstraintResourceGroup(c, i);
    if( KheResourceGroupResourceType(rg) == rt )
      count += KheResourceGroupResourceCount(rg);
  }
  return count;
}


/*****************************************************************************/
/*                                                                           */
/*  int KheLimitActiveIntervalsConstraintResourceTypeCount(                  */
/*    KHE_LIMIT_ACTIVE_INTERVALS_CONSTRAINT c, KHE_RESOURCE_TYPE rt)         */
/*                                                                           */
/*  Return the number of resources of type rt monitored by c.                */
/*                                                                           */
/*****************************************************************************/

static int KheLimitActiveIntervalsConstraintResourceTypeCount(
  KHE_LIMIT_ACTIVE_INTERVALS_CONSTRAINT c, KHE_RESOURCE_TYPE rt)
{
  int i, count;  KHE_RESOURCE_GROUP rg;  KHE_RESOURCE r;
  count = 0;
  for( i = 0;  i < KheLimitActiveIntervalsConstraintResourceCount(c);  i++ )
  {
    r = KheLimitActiveIntervalsConstraintResource(c, i);
    if( KheResourceResourceType(r) == rt )
      count += 1;
  }
  for( i = 0;  i < KheLimitActiveIntervalsConstraintResourceGroupCount(c); i++ )
  {
    rg = KheLimitActiveIntervalsConstraintResourceGroup(c, i);
    if( KheResourceGroupResourceType(rg) == rt )
      count += KheResourceGroupResourceCount(rg);
  }
  return count;
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDemandAtTime(KHE_EVENT_TIMETABLE_MONITOR etm,                     */
/*    KHE_RESOURCE_TYPE rt, KHE_TIME t)                                      */
/*                                                                           */
/*  Return the total demand for resources of type rt at time t.              */
/*                                                                           */
/*****************************************************************************/

static int KheDemandAtTime(KHE_EVENT_TIMETABLE_MONITOR etm,
  KHE_RESOURCE_TYPE rt, KHE_TIME t)
{
  int j, k, res;  KHE_MEET meet;  KHE_TASK task;  KHE_RESOURCE r;
  KHE_EVENT_RESOURCE er;
  res = 0;
  for( j = 0;  j < KheEventTimetableMonitorTimeMeetCount(etm, t);  j++ )
  {
    meet = KheEventTimetableMonitorTimeMeet(etm, t, j);
    for( k = 0;  k < KheMeetTaskCount(meet);  k++ )
    {
      task = KheMeetTask(meet, k);
      if( KheTaskResourceType(task) == rt )
      {
	if( KheTaskIsPreassigned(task, &r) )
	{
	  if( DEBUG6 )
	  {
	    fprintf(stderr, "    demand (preassigned) ");
	    KheTaskDebug(task, 2, 0, stderr);
	  }
	  res++;
	}
	else
	{
	  er = KheTaskEventResource(task);
	  if( er != NULL && KheEventResourceNeedsAssignment(er) == KHE_YES )
	  {
	    if( DEBUG6 )
	    {
	      fprintf(stderr, "    demand ");
	      KheTaskDebug(task, 2, 0, stderr);
	    }
	    res++;
	  }
	}
      }
    }
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDemandAtTimeGroup(KHE_EVENT_TIMETABLE_MONITOR etm,                */
/*    KHE_RESOURCE_TYPE rt, KHE_TIME_GROUP tg)                               */
/*                                                                           */
/*  Return the total demand for resources of type rt in the times of tg.     */
/*                                                                           */
/*****************************************************************************/

static int KheDemandAtTimeGroup(KHE_EVENT_TIMETABLE_MONITOR etm,
  KHE_RESOURCE_TYPE rt, KHE_TIME_GROUP tg)
{
  int i, res;  KHE_TIME t;
  res = 0;
  for( i = 0;  i < KheTimeGroupTimeCount(tg);  i++ )
  {
    t = KheTimeGroupTime(tg, i);
    res += KheDemandAtTime(etm, rt, t);
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDemandAtClusterConstraint(KHE_EVENT_TIMETABLE_MONITOR etm,        */
/*    KHE_RESOURCE_TYPE rt, KHE_CLUSTER_BUSY_TIMES_CONSTRAINT c, int offset) */
/*                                                                           */
/*  Return the total demand for resources of type rt in the time groups      */
/*  of cluster busy times constraint c at offset.                            */
/*                                                                           */
/*****************************************************************************/

static int KheDemandAtClusterConstraint(KHE_EVENT_TIMETABLE_MONITOR etm,
  KHE_RESOURCE_TYPE rt, KHE_CLUSTER_BUSY_TIMES_CONSTRAINT c, int offset)
{
  int i, res;  KHE_TIME_GROUP tg;  KHE_POLARITY po;
  res = 0;
  for( i = 0;  i < KheClusterBusyTimesConstraintTimeGroupCount(c);  i++ )
  {
    tg = KheClusterBusyTimesConstraintTimeGroup(c, i, offset, &po);
    res += KheDemandAtTimeGroup(etm, rt, tg);
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheClusterFirstAndLastIndex(KHE_CLUSTER_BUSY_TIMES_CONSTRAINT c,    */
/*    int offset, KHE_FRAME frame, int *first_index, int *last_index)        */
/*                                                                           */
/*  Return the first and last indexes in frame where the time groups         */
/*  intersect with the time groups of c at offset.  If there is no           */
/*  intersection, return an empty interval (*first_index > *last_index).     */
/*                                                                           */
/*****************************************************************************/

static bool KheClusterFirstAndLastIndex(KHE_CLUSTER_BUSY_TIMES_CONSTRAINT c,
  int offset, KHE_FRAME frame, int *first_index, int *last_index)
{
  int i;  KHE_TIME_GROUP tg;  KHE_POLARITY po;
  KHE_TIME t, earliest_t, latest_t;

  /* find the earliest and latest times; if none, return false */
  earliest_t = latest_t = NULL;
  for( i = 0;  i < KheClusterBusyTimesConstraintTimeGroupCount(c);  i++ )
  {
    tg = KheClusterBusyTimesConstraintTimeGroup(c, i, offset, &po);
    if( KheTimeGroupTimeCount(tg) > 0 )
    {
      t = KheTimeGroupTime(tg, 0);
      if( earliest_t == NULL || KheTimeIndex(t) < KheTimeIndex(earliest_t) )
        earliest_t = t;
      t = KheTimeGroupTime(tg, KheTimeGroupTimeCount(tg) - 1);
      if( latest_t == NULL || KheTimeIndex(t) > KheTimeIndex(latest_t) )
        latest_t = t;
    }
  }
  if( earliest_t == NULL )
    return *first_index = 0, *last_index = -1, false;

  /* all good, return the indexes */
  return *first_index = KheFrameTimeIndex(frame, earliest_t),
    *last_index = KheFrameTimeIndex(frame, latest_t), true;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheLimitFirstAndLastIndex(KHE_LIMIT_BUSY_TIMES_CONSTRAINT c,        */
/*    int offset, KHE_FRAME frame, int *first_index, int *last_index)        */
/*                                                                           */
/*  Return the first and last indexes in frame where the time groups         */
/*  intersect with the time groups of c at offset.  If there is no           */
/*  intersection, return an empty interval (*first_index > *last_index).     */
/*                                                                           */
/*****************************************************************************/

static bool KheLimitFirstAndLastIndex(KHE_LIMIT_BUSY_TIMES_CONSTRAINT c,
  int offset, KHE_FRAME frame, int *first_index, int *last_index)
{
  int i;  KHE_TIME_GROUP tg;  KHE_TIME t, earliest_t, latest_t;

  /* find the earliest and latest times; if none, return false */
  earliest_t = latest_t = NULL;
  for( i = 0;  i < KheLimitBusyTimesConstraintTimeGroupCount(c);  i++ )
  {
    tg = KheLimitBusyTimesConstraintTimeGroup(c, i, offset);
    if( KheTimeGroupTimeCount(tg) > 0 )
    {
      t = KheTimeGroupTime(tg, 0);
      if( earliest_t == NULL || KheTimeIndex(t) < KheTimeIndex(earliest_t) )
        earliest_t = t;
      t = KheTimeGroupTime(tg, KheTimeGroupTimeCount(tg) - 1);
      if( latest_t == NULL || KheTimeIndex(t) > KheTimeIndex(latest_t) )
        latest_t = t;
    }
  }
  if( earliest_t == NULL )
    return *first_index = 0, *last_index = -1, false;

  /* all good, return the indexes */
  return *first_index = KheFrameTimeIndex(frame, earliest_t),
    *last_index = KheFrameTimeIndex(frame, latest_t), true;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheClusterConstraintHasNonSingletonTimeGroup(                       */
/*    KHE_CLUSTER_BUSY_TIMES_CONSTRAINT cbtc)                                */
/*                                                                           */
/*  Return true if cbtc has at least one time group with >=2 elements.       */
/*                                                                           */
/*****************************************************************************/

static bool KheClusterConstraintHasNonSingletonTimeGroup(
  KHE_CLUSTER_BUSY_TIMES_CONSTRAINT cbtc)
{
  int i;  KHE_TIME_GROUP tg;  KHE_POLARITY po;
  for( i = 0;  i < KheClusterBusyTimesConstraintTimeGroupCount(cbtc);  i++ )
  {
    tg = KheClusterBusyTimesConstraintTimeGroup(cbtc, i, 0, &po);
    if( KheTimeGroupTimeCount(tg) >= 2 )
      return true;
  }
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheClusterConstraintSuitsCombinationElimination(                    */
/*    KHE_CLUSTER_BUSY_TIMES_CONSTRAINT c, KHE_RESOURCE_TYPE rt, int *lim)   */
/*                                                                           */
/*  Return true if c is suited to combination elimination:  if its time      */
/*  groups are all positive and it has a non-trivial maximum limit, or its   */
/*  time groups are all negative and it has a non-trivial minimum limit.     */
/*                                                                           */
/*  If c is suitable, also set *lim to a maximum limit on the number of      */
/*  its busy (not active) time groups.  This will be c's maximum limit if    */
/*  its time groups are all positive, or the number of time groups minus     */
/*  the minimum limit if its time groups are all negative.                   */
/*                                                                           */
/*  Also check that c's time groups are disjoint, that it has at least one   */
/*  non-singleton time group, and that it covers all resources of type rt.   */
/*                                                                           */
/*****************************************************************************/

static bool KheClusterConstraintSuitsCombinationElimination(
  KHE_CLUSTER_BUSY_TIMES_CONSTRAINT c, KHE_RESOURCE_TYPE rt, int *lim)
{
  int max_lim, min_lim, count, rt_count;
  rt_count = KheResourceTypeResourceCount(rt);
  if( KheClusterBusyTimesConstraintTimeGroupsDisjoint(c) &&
      KheClusterConstraintHasNonSingletonTimeGroup(c) &&
      KheClusterBusyTimesConstraintResourceTypeCount(c, rt) == rt_count )
  {
    max_lim = KheClusterBusyTimesConstraintMaximum(c);
    min_lim = KheClusterBusyTimesConstraintMinimum(c);
    count = KheClusterBusyTimesConstraintTimeGroupCount(c);
    if( KheClusterBusyTimesConstraintAllPositive(c) && max_lim < count )
      return *lim = max_lim, true;
    else if( KheClusterBusyTimesConstraintAllNegative(c) && min_lim > 0 )
      return *lim = count - min_lim, true;
    else
      return *lim = -1, false;
  }
  else
    return *lim = -1, false;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "eliminating combinations based on supply and demand"          */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  int IntCmpIncreasing(const void *t1, const void *t2)                     */
/*                                                                           */
/*  Comparison fuunction for sorting an array of integers into increasing    */
/*  order.                                                                   */
/*                                                                           */
/*****************************************************************************/

static int IntCmpIncreasing(const void *t1, const void *t2)
{
  int val1 = * (int *) t1;
  int val2 = * (int *) t2;
  return val1 - val2;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheEliminateCombinationsForConstraint(KHE_GROUP_SOLVER gs,          */
/*    KHE_CLUSTER_BUSY_TIMES_CONSTRAINT c, int lim)                          */
/*                                                                           */
/*  Eliminate combinations based on supply and demand within the times of    */
/*  c, a suitable cluster busy times monitor with limit lim.                 */
/*                                                                           */
/*****************************************************************************/
#define offset_count KheClusterBusyTimesConstraintAppliesToOffsetCount

static void KheEliminateCombinationsForConstraint(KHE_GROUP_SOLVER gs,
  KHE_CLUSTER_BUSY_TIMES_CONSTRAINT c, int lim)
{
  KHE_TIME_GROUP c_tg, f_tg, prev_f_tg;
  int offset, i, j, count, sum, fi, li, supply, demand;  KHE_POLARITY po;
  bool f_tg_overlaps, prev_f_tg_overlaps;  KHE_CG_TIME_GROUP_INFO tgi;
  if( DEBUG6 )
  {
    fprintf(stderr, "[ KheEliminateCombinationsForConstraint(gs, c, %d), c:\n",
      lim);
    KheClusterBusyTimesConstraintDebug(c, 3, 2, stderr);
  }

  /* for each time group of c, find the number of frame time groups it */
  /* intersects with.  Also record in ec_time_group_info those time groups */
  /* which will be linked with previous time groups if demand allows */
  for( offset = 0;  offset < offset_count(c);  offset++ )
  {
    if( DEBUG6 )
      fprintf(stderr, "  [ testing offset %d:\n", offset);
    HaArrayClear(gs->ec_time_group_info);
    HaArrayClear(gs->ec_intersects);
    KheClusterFirstAndLastIndex(c, offset, gs->common_frame, &fi, &li);
    for( i = 0;  i < KheClusterBusyTimesConstraintTimeGroupCount(c);  i++ )
    {
      c_tg = KheClusterBusyTimesConstraintTimeGroup(c, i, offset, &po);
      count = 0;
      prev_f_tg = NULL;
      prev_f_tg_overlaps = false;
      for( j = fi;  j <= li;  j++ )
      {
	f_tg = KheFrameTimeGroup(gs->common_frame, j);
	f_tg_overlaps = !KheTimeGroupDisjoint(c_tg, f_tg);
	if( f_tg_overlaps )
	  count++;
	if( f_tg_overlaps && prev_f_tg_overlaps )
	{
	  /* if we later find that D(T) >= S(T), then if prev_f_tg is */
	  /* busy, then f_tg must be busy as well, because they both  */
	  /* overlap the same time group of the monitor */
	  if( DEBUG6 )
	  {
	    fprintf(stderr, "  provisionally linked time groups ");
	    KheTimeGroupDebug(prev_f_tg, 1, -1, stderr);
	    fprintf(stderr, " and ");
	    KheTimeGroupDebug(f_tg, 1, -1, stderr);
	    fprintf(stderr, "\n");
	  }

	  /* record the linkage provisionally */
	  tgi = HaArray(gs->cg_time_group_info, j);
	  HaArrayAddLast(gs->ec_time_group_info, tgi);
	}
	prev_f_tg = f_tg;
	prev_f_tg_overlaps = f_tg_overlaps;
      }
      HaArrayAddLast(gs->ec_intersects, count);
    }

    /* get the supply:  the largest lim intersect counts, times resources */
    HaArraySort(gs->ec_intersects, &IntCmpIncreasing);
    count = HaArrayCount(gs->ec_intersects);
    sum = 0;
    for( i = 1;  i <= lim;  i++ )
      sum += HaArray(gs->ec_intersects, count - i);
    supply = sum * KheResourceTypeResourceCount(gs->resource_type);

    /* get the demand */
    demand = KheDemandAtClusterConstraint(gs->etm, gs->resource_type,c,offset);

    /* if demand >= supply, provisional linkages become definite */
    if( demand >= supply )
      HaArrayForEach(gs->ec_time_group_info, tgi, i)
	tgi->linked_to_prev = c;
    if( DEBUG6 )
      fprintf(stderr, "  ] offset %d, demand %d, supply %d\n",
	offset, demand, supply);
  }
  if( DEBUG6 )
    fprintf(stderr, "] KheEliminateCombinationsForConstraint returning\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheEliminateCombinations(KHE_GROUP_SOLVER gs)                       */
/*                                                                           */
/*  Eliminate combinations based on supply and demand.                       */
/*                                                                           */
/*  NB the condition ClusterBusyTimesConstraintTimeGroupsDisjoint(cbtc) is   */
/*  not really required, but it always seems to hold, and it makes the       */
/*  calculation of demand much easier.                                       */
/*                                                                           */
/*****************************************************************************/
static void KheGroupSolverDebug(KHE_GROUP_SOLVER gs,
  int indent, FILE *fp);

static void KheEliminateCombinations(KHE_GROUP_SOLVER gs)
{
  int i, lim;  KHE_CONSTRAINT c;  KHE_RESOURCE r;  KHE_RESOURCE_TYPE rt;
  KHE_CLUSTER_BUSY_TIMES_CONSTRAINT cbtc;
  rt = gs->resource_type;
  if( DEBUG6 )
    fprintf(stderr, "[ KheEliminateCombinations(%s)\n", KheResourceTypeId(rt));
  r = KheResourceTypeResource(rt, 0);  /* must exist, see call on this fn */
  for( i = 0;  i < KheResourceConstraintCount(r);  i++ )
  {
    c = KheResourceConstraint(r, i);
    if( KheConstraintTag(c) == KHE_CLUSTER_BUSY_TIMES_CONSTRAINT_TAG )
    {
      cbtc = (KHE_CLUSTER_BUSY_TIMES_CONSTRAINT) c;
      if( KheClusterConstraintSuitsCombinationElimination(cbtc, rt, &lim) )
	KheEliminateCombinationsForConstraint(gs, cbtc, lim);
    }
  }
  if( DEBUG6 )
  {
    fprintf(stderr, "  at end of KheEliminateCombinations:\n");
    KheGroupSolverDebug(gs, 2, stderr);
    fprintf(stderr, "] KheEliminateCombinations returning\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "combinatorial grouping"                                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool KheMonitorSuitsCombinatorialGrouping(KHE_MONITOR m,                 */
/*    KHE_GROUP_SOLVER gs)                                                   */
/*                                                                           */
/*  Return true if m suits combinatorial grouping:  if it is a cluster busy  */
/*  times or limit busy times monitor whose times all lie in the current     */
/*  interval, derived from a constraint that applies to all resources.       */
/*                                                                           */
/*****************************************************************************/

static bool KheMonitorSuitsCombinatorialGrouping(KHE_MONITOR m,
  KHE_GROUP_SOLVER gs)
{
  KHE_CLUSTER_BUSY_TIMES_CONSTRAINT cbtc;  KHE_CLUSTER_BUSY_TIMES_MONITOR cbtm;
  KHE_LIMIT_BUSY_TIMES_CONSTRAINT lbtc;  KHE_LIMIT_BUSY_TIMES_MONITOR lbtm;
  int offset, fi, li, count;  KHE_RESOURCE_TYPE rt;  KHE_FRAME frame;
  rt = gs->resource_type;
  count = KheResourceTypeResourceCount(rt);
  frame = gs->common_frame;
  switch( KheMonitorTag(m) )
  {
    case KHE_CLUSTER_BUSY_TIMES_MONITOR_TAG:

      cbtm = (KHE_CLUSTER_BUSY_TIMES_MONITOR) m;
      cbtc = KheClusterBusyTimesMonitorConstraint(cbtm);
      offset = KheClusterBusyTimesMonitorOffset(cbtm);
      return KheClusterBusyTimesConstraintResourceTypeCount(cbtc, rt) == count
	&& KheClusterFirstAndLastIndex(cbtc, offset, frame, &fi, &li) &&
	fi >= gs->cg_first_index && li <= gs->cg_last_index;

    case KHE_LIMIT_BUSY_TIMES_MONITOR_TAG:

      lbtm = (KHE_LIMIT_BUSY_TIMES_MONITOR) m;
      lbtc = KheLimitBusyTimesMonitorConstraint(lbtm);
      offset = KheLimitBusyTimesMonitorOffset(lbtm);
      return KheLimitBusyTimesConstraintResourceTypeCount(lbtc, rt) == count
	&& KheLimitFirstAndLastIndex(lbtc, offset, frame, &fi, &li) &&
	fi >= gs->cg_first_index && li <= gs->cg_last_index;

    default:

      return false;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheTraceHasZeroCost(KHE_TRACE trace, KHE_GROUP_SOLVER gs)           */
/*                                                                           */
/*  Return true if the monitors that show up on trace all have zero cost.    */
/*  Actually we are only interested in monitors which are cluster busy       */
/*  times and limit busy times monitors derived from constraints that        */
/*  apply to all resources, and whose times lie between gs->cg_first_index   */
/*  and gs->cg_last_index in the common frame.                               */
/*                                                                           */
/*****************************************************************************/

static bool KheTraceHasZeroCost(KHE_GROUP_SOLVER gs, int indent)
{
  int i;  KHE_MONITOR m;  bool cost_increased, suits;
  for( i = 0;  i < KheTraceMonitorCount(gs->cg_trace);  i++ )
  {
    m = KheTraceMonitor(gs->cg_trace, i);
    cost_increased =
      (KheMonitorCost(m) > KheTraceMonitorInitCost(gs->cg_trace, i));
    suits = KheMonitorSuitsCombinatorialGrouping(m, gs);
    if( DEBUG4 )
    {
      fprintf(stderr, "%*s%c%c %.5f -> %.5f ", indent, "",
        cost_increased ? 'C' : ' ', suits ? 'S' : ' ',
	KheCostShow(KheTraceMonitorInitCost(gs->cg_trace, i)),
	KheCostShow(KheMonitorCost(m)));
      KheMonitorDebug(m, 1, 0, stderr);
    }
    if( cost_increased && suits )
      return false;
  }
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheCombinatorialGroupingAssign(KHE_GROUP_SOLVER gs, int curr_index, */
/*    bool prev_tg_assigned)                                                 */
/*                                                                           */
/*  Carry out the part of the solve that assigns all combinations of tasks   */
/*  and checks which ones are successful.                                    */
/*                                                                           */
/*  If prev_tg_assigned, then the previous time group (at curr_index - 1)    */
/*  has been assigned something, or is off the left end.                     */
/*                                                                           */
/*    cg_first_index          The first index of the area being searched     */
/*    cg_last_index           The last index of the area being searched      */
/*    cg_in_ts                The times that have so far been included       */
/*    cg_out_ts               The times that have so far been excluded       */
/*                                                                           */
/*    cg_success_count        Number of successes                            */
/*    cg_success_in_ts        If success, the times included                 */
/*    cg_success_out_ts       If success, the times excluded                 */
/*                                                                           */
/*    cg_trace                Temporary variable holding a trace object      */
/*                                                                           */
/*****************************************************************************/

static void KheCombinatorialGroupingAssign(KHE_GROUP_SOLVER gs, int curr_index,
  bool prev_tg_assigned, int indent)
{
  KHE_TIME_GROUP tg;  KHE_TIME t;  KHE_TASK root_task;  KHE_MARK mark;
  KHE_RESOURCE_GROUP rg;  KHE_RESOURCE r;  int i, groups_made;
  KHE_CG_TIME_GROUP_INFO tgi;  bool res;
  if( DEBUG4 )
    fprintf(stderr, "%*s[ KheCombinatorialGroupingAssign(-, %d, -, -)\n",
      indent, "", curr_index);
  if( curr_index > gs->cg_last_index )
  {
    /* time sets are complete; try for a grouped task now */
    mark = KheMarkBegin(gs->soln);
    KheTraceBegin(gs->cg_trace);
    if( DEBUG4 )
    {
      fprintf(stderr, "%*s  [ testing in_ts ", indent, "");
      KheTimeSetDebug(gs->cg_in_ts, 2, -1, stderr);
      fprintf(stderr, " and out_ts ");
      KheTimeSetDebug(gs->cg_out_ts, 2, -1, stderr);
      fprintf(stderr, ":\n");
    }
    if( KheTimeBasedGrouping(gs, gs->cg_in_ts, gs->cg_out_ts, NULL, 1, true,
	  &root_task, &groups_made, NULL) )
    {
      /* assign root_task and check the result of tracing it */
      if( DEBUG4 )
      {
	fprintf(stderr, "%*s    root_task ", indent, "");
	KheTaskDebug(root_task, 2, 0, stderr);
      }
      rg = KheTaskDomain(root_task);
      if( KheResourceGroupResourceCount(rg) > 0 )
      {
	r = KheTaskAsstResource(root_task);
	if( r != NULL )
	  res = KheTraceHasZeroCost(gs, indent + 2);
	else
	{
	  r = KheResourceGroupResource(rg, 0);
	  res = KheTaskMoveResource(root_task, r) &&
	    KheTraceHasZeroCost(gs, indent + 2);
	}
	if( res )
	{
	  gs->cg_success_count++;
	  KheTimeSetCopyElements(gs->cg_success_in_ts, gs->cg_in_ts);
	  KheTimeSetCopyElements(gs->cg_success_out_ts, gs->cg_out_ts);
	  if( DEBUG4 )
	    fprintf(stderr, "%*s  ] success\n", indent, "");
	}
	else
	{
	  if( DEBUG4 )
	    fprintf(stderr, "%*s  ] failure (no assign, or non-zero cost)\n",
	      indent, "");
	}
      }
      else
      {
	if( DEBUG4 )
	  fprintf(stderr, "%*s  ] failure (no resources)\n", indent, "");
      }
    }
    else
    {
      if( DEBUG4 )
	fprintf(stderr, "%*s  ] failure (no group)\n", indent, "");
    }
    KheTraceEnd(gs->cg_trace);
    KheMarkEnd(mark, true);
  }
  else
  {
    /* try assigning each time from the current time group */
    tg = KheFrameTimeGroup(gs->common_frame, curr_index);
    for( i = 0; gs->cg_success_count < 2 && i < KheTimeGroupTimeCount(tg); i++ )
    {
      t = KheTimeGroupTime(tg, i);
      KheTimeSetAddTime(gs->cg_in_ts, t);
      KheCombinatorialGroupingAssign(gs, curr_index + 1, true, indent + 2);
      KheTimeSetDeleteTime(gs->cg_in_ts, t);
    }

    /* try not assigning any time from the current time group */
    if( gs->cg_success_count < 2 )
    {
      tgi = HaArray(gs->cg_time_group_info, curr_index);
      if( prev_tg_assigned && tgi->linked_to_prev != NULL )
      {
	if( DEBUG6 )
	{
	  fprintf(stderr, "%*s  linkage by %s so omitting non-assignment of ",
	    indent, "", KheConstraintId((KHE_CONSTRAINT) tgi->linked_to_prev));
	  KheTimeGroupDebug(tg, 1, 0, stderr);
	}
      }
      else
      {
	KheTimeSetAddTimeGroup(gs->cg_out_ts, tg);
	KheCombinatorialGroupingAssign(gs, curr_index + 1, false, indent + 2);
	KheTimeSetDeleteTimeGroup(gs->cg_out_ts, tg);
      }
    }
  }
  if( DEBUG4 )
    fprintf(stderr, "%*s] KheCombinatorialGroupingAssign returning\n",
      indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDoCombinatorialGroupingForInterval(KHE_GROUP_SOLVER gs,          */
/*    int first_index, int last_index)                                       */
/*                                                                           */
/*  Solve gs for the interval first_index .. last_index, and if there was    */
/*  exactly one success, do time-based grouping for that set of times.       */
/*                                                                           */
/*****************************************************************************/

static void KheDoCombinatorialGroupingForInterval(KHE_GROUP_SOLVER gs,
  int first_index, int last_index)
{
  int i, groups_made;  bool success;  KHE_TIME_GROUP tg;  KHE_TIME t;
  if( DEBUG2 )
  {
    fprintf(stderr, "  [ KheDoCombinatorialGroupingForInterval(gs, ");
    tg = KheFrameTimeGroup(gs->common_frame, first_index);
    KheTimeGroupDebug(tg, 1, -1, stderr);
    fprintf(stderr, " .. ");
    tg = KheFrameTimeGroup(gs->common_frame, last_index);
    KheTimeGroupDebug(tg, 1, -1, stderr);
    fprintf(stderr, ")\n");
  }
  gs->cg_first_index = first_index;
  gs->cg_last_index = last_index;

  /* try one solve for each time in the first time group, but then */
  /* start all over again and have another go if any were successful */
  tg = KheFrameTimeGroup(gs->common_frame, first_index);
  do
  {
    success = false;
    for( i = 0;  i < KheTimeGroupTimeCount(tg);  i++ )
    {
      /* initialize the solve fields other than gm */
      KheTimeSetClear(gs->cg_in_ts);
      KheTimeSetClear(gs->cg_out_ts);
      gs->cg_success_count = 0;
      KheTimeSetClear(gs->cg_success_in_ts);
      KheTimeSetClear(gs->cg_success_out_ts);

      /* solve for t */
      t = KheTimeGroupTime(tg, i);
      if( DEBUG4 )
	fprintf(stderr, "    [ combinations starting with %s:\n", KheTimeId(t));
      KheTimeSetAddTime(gs->cg_in_ts, t);
      KheCombinatorialGroupingAssign(gs, first_index + 1, true, 6);
      KheTimeSetDeleteTime(gs->cg_in_ts, t);
      if( DEBUG4 )
	fprintf(stderr, "    ] (%d successes)\n", gs->cg_success_count);

      /* if exactly one success, we have succeeded */
      if( gs->cg_success_count == 1 )
      {
	/* have a single successful assignment, so group tasks */
	if( DEBUG2 )
	{
	  fprintf(stderr, "  KheDoCombinatorialGroupingForInterval found ");
	  KheTimeSetDebug(gs->cg_success_in_ts, 2, 0, stderr);
	}
	if( KheTimeBasedGrouping(gs, gs->cg_success_in_ts,
	      gs->cg_success_out_ts, NULL, INT_MAX, false, NULL,
	      &groups_made, "combinatorial grouping") )
	  success = true;
      }
    }
  } while( success );

  if( DEBUG2 )
    fprintf(stderr, "  ] KheDoCombinatorialGroupingForInterval returning\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDoCombinatorialGrouping(KHE_GROUP_SOLVER gs)                     */
/*                                                                           */
/*  Carry out combinatorial grouping for resource type gs->resource_type.    */
/*                                                                           */
/*****************************************************************************/

static void KheDoCombinatorialGrouping(KHE_GROUP_SOLVER gs)
{
  int len, first_index, days;

  /* solve for each subsequence of len days */
  /* ***
  if( DEBUG2 )
    KheGroupSolverDebug(gs, 2, stderr);
  *** */
  days = KheFrameTimeGroupCount(gs->common_frame);
  for( len = gs->cg_max_days;  len >= 2;  len-- )
    for( first_index = 0;  first_index <= days - len;  first_index++ )
      KheDoCombinatorialGroupingForInterval(gs, first_index,
	first_index + len - 1);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "profile grouping"                                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool KheMonitorSuitsProfileGrouping(KHE_LIMIT_ACTIVE_INTERVALS_MONITOR m,*/
/*    KHE_RESOURCE_TYPE rt, bool extended)                                   */
/*                                                                           */
/*  Return true if m suits profile grouping:  if its constraint applies to   */
/*  all resources of type rt, its time groups are positive, and its time     */
/*  groups are singletons (non-extended) or non-singletons (extended).       */
/*                                                                           */
/*****************************************************************************/

static bool KheMonitorSuitsProfileGrouping(KHE_LIMIT_ACTIVE_INTERVALS_MONITOR m,
  KHE_RESOURCE_TYPE rt, bool extended)
{
  int i, rt_count;  KHE_TIME_GROUP tg;  KHE_POLARITY po;
  KHE_LIMIT_ACTIVE_INTERVALS_CONSTRAINT c;

  /* there must be at least two time groups */
  if( KheLimitActiveIntervalsMonitorTimeGroupCount(m) < 2 )
    return false;

  /* its time groups must be positive, and either singletons or not */
  for( i = 0;  i < KheLimitActiveIntervalsMonitorTimeGroupCount(m);  i++ )
  {
    tg = KheLimitActiveIntervalsMonitorTimeGroup(m, i, &po);
    if( po != KHE_POSITIVE )
      return false;
    if( extended )
    {
      if( KheTimeGroupTimeCount(tg) == 1 )
	return false;
    }
    else
    {
      if( KheTimeGroupTimeCount(tg) != 1 )
	return false;
    }
  }

  /* it must monitor all resources of type rt */
  rt_count = KheResourceTypeResourceCount(rt);
  c = KheLimitActiveIntervalsMonitorConstraint(m);
  if( KheLimitActiveIntervalsConstraintResourceTypeCount(c, rt) < rt_count )
    return false;

  /* all OK */
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheLimitActiveIntervalsMonitorsCanGroupTogether(                    */
/*    KHE_LIMIT_ACTIVE_INTERVALS_MONITOR laim1,                              */
/*    KHE_LIMIT_ACTIVE_INTERVALS_MONITOR laim2)                              */
/*                                                                           */
/*  Return true if laim1 and laim2 can be handled together, because they     */
/*  have equal time groups, one has a maximum limit only, and the other      */
/*  has a minimum limit only.                                                */
/*                                                                           */
/*****************************************************************************/

static bool KheLimitActiveIntervalsMonitorsCanGroupTogether(
  KHE_LIMIT_ACTIVE_INTERVALS_MONITOR laim1,
  KHE_LIMIT_ACTIVE_INTERVALS_MONITOR laim2)
{
  int i;  KHE_TIME_GROUP tg1, tg2;  KHE_POLARITY po1, po2;
  bool has_min1, has_min2, has_max1, has_max2;

  /* time groups must be equal */
  if( KheLimitActiveIntervalsMonitorTimeGroupCount(laim1) !=
      KheLimitActiveIntervalsMonitorTimeGroupCount(laim2) )
    return false;
  for( i = 0;  i < KheLimitActiveIntervalsMonitorTimeGroupCount(laim1);  i++ )
  {
    tg1 = KheLimitActiveIntervalsMonitorTimeGroup(laim1, i, &po1);
    tg2 = KheLimitActiveIntervalsMonitorTimeGroup(laim2, i, &po2);
    if( !KheTimeGroupEqual(tg1, tg2) )
      return false;
  }

  /* one must have a min limit only, the other a max limit only */
  has_min1 = KheLimitActiveIntervalsMonitorMinimum(laim1) > 0;
  has_min2 = KheLimitActiveIntervalsMonitorMinimum(laim2) > 0;
  has_max1 = KheLimitActiveIntervalsMonitorMaximum(laim1) <
    KheLimitActiveIntervalsMonitorTimeGroupCount(laim1);
  has_max2 = KheLimitActiveIntervalsMonitorMaximum(laim2) <
    KheLimitActiveIntervalsMonitorTimeGroupCount(laim2);
  if( (has_min1 && !has_max1) && (!has_min2 && has_max2) )
    return true;
  if( (!has_min1 && has_max1) && (has_min2 && !has_max2) )
    return true;
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheLimitActiveIntervalsMonitorSetTimes(                             */
/*    KHE_LIMIT_ACTIVE_INTERVALS_MONITOR m, KHE_TIME_SET ts)                 */
/*                                                                           */
/*  Set the times of ts to the times of the time groups of m.                */
/*                                                                           */
/*****************************************************************************/

static void KheLimitActiveIntervalsMonitorSetTimes(
  KHE_LIMIT_ACTIVE_INTERVALS_MONITOR m, KHE_TIME_SET ts)
{
  int i, tg_count;  KHE_TIME_GROUP tg;  KHE_POLARITY po;
  KheTimeSetClear(ts);
  tg_count = KheLimitActiveIntervalsMonitorTimeGroupCount(m);
  for( i = 0;  i < tg_count;  i++ )
  {
    tg = KheLimitActiveIntervalsMonitorTimeGroup(m, i, &po);
    HnAssert(KheTimeGroupTimeCount(tg) == 1 && po == KHE_POSITIVE,
      "KheLimitActiveIntervalsMonitorSetTimes internal error");
    KheTimeSetAddTimeGroup(ts, tg);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheProfileGroupingBegin(KHE_GROUP_SOLVER gs)                        */
/*                                                                           */
/*  Begin profile grouping for gs->pg_monitor_ts.                            */
/*                                                                           */
/*****************************************************************************/

static void KheProfileGroupingBegin(KHE_GROUP_SOLVER gs)
{
  int i, j;  KHE_TIME t;  KHE_ROOT_TASK_SET rts;  KHE_TIME_INFO ti;
  gs->pg_active = true;
  for( i = 0;  i < KheTimeSetTimeCount(gs->pg_monitor_ts);  i++ )
  {
    /* initialize the profile grouping fields of ti and its root task sets */
    t = KheTimeSetTime(gs->pg_monitor_ts, i);
    ti = HaArray(gs->time_info, KheTimeIndex(t));
    ti->pg_active = true;
    ti->pg_profile = 0;
    HaArrayForEach(ti->root_task_sets, rts, j)
    {
      /* determine the profile grouping status of rts (may already be done) */
      if( rts->pg_status == KHE_PG_INACTIVE_UNMONITORED )
        rts->pg_status = KheRootTaskSetStatus(rts);

      /* add rts's contribution to ti's profile */
      switch( rts->pg_status )
      {
	case KHE_PG_INACTIVE_MAXIMAL:

	  /* maximal tasks do not contribute to the profile */
	  break;

	case KHE_PG_ACTIVE:

	  /* add to profile */
	  ti->pg_profile += KheTaskSetTaskCount(rts->tasks);
	  break;

	case KHE_PG_INACTIVE_UNMONITORED:
	default:

	  /* there are no unmonitored tasks at gs->pg_monitor_ts times */
	  HnAbort("KheProfileGroupingBegin internal error");
	  break;
      }
    }
    if( DEBUG9 )
      fprintf(stderr, "  profile at %s: %d\n", KheTimeId(t), ti->pg_profile);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheProfileGroupingEnd(KHE_GROUP_SOLVER gs)                          */
/*                                                                           */
/*  End profile grouping for gs->pg_monitor_ts.                              */
/*                                                                           */
/*****************************************************************************/

static void KheProfileGroupingEnd(KHE_GROUP_SOLVER gs)
{
  int i, j;  KHE_TIME t;  KHE_ROOT_TASK_SET rts;  KHE_TIME_INFO ti;
  for( i = 0;  i < KheTimeSetTimeCount(gs->pg_monitor_ts);  i++ )
  {
    /* clear out the profile grouping fields of ti and its root task sets */
    t = KheTimeSetTime(gs->pg_monitor_ts, i);
    ti = HaArray(gs->time_info, KheTimeIndex(t));
    ti->pg_active = false;
    ti->pg_profile = 0;
    HaArrayForEach(ti->root_task_sets, rts, j)
      rts->pg_status = KHE_PG_INACTIVE_UNMONITORED;
  }
  gs->pg_active = false;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheAddTime(KHE_GROUP_SOLVER gs, int index, KHE_TIME_SET ts)         */
/*                                                                           */
/*  Add the i'th element of gs->pg_monitor_ts to ts, but only if it exists.  */
/*                                                                           */
/*****************************************************************************/

static void KheAddTime(KHE_GROUP_SOLVER gs, int index, KHE_TIME_SET ts)
{
  KHE_TIME t;
  if( 0 <= index && index < KheTimeSetTimeCount(gs->pg_monitor_ts) )
  {
    t = KheTimeSetTime(gs->pg_monitor_ts, index);
    KheTimeSetAddTime(ts, t);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheProfileGroupingSetInTimeSet(KHE_GROUP_SOLVER gs,                 */
/*    int first_index, int last_index)                                       */
/*                                                                           */
/*  Set gs->pg_in_ps to include all the times of gs->pg_monitor_ts from      */
/*  first_index to last_index inclusive, except out-of-range ones.           */
/*                                                                           */
/*****************************************************************************/

static void KheProfileGroupingSetInTimeSet(KHE_GROUP_SOLVER gs,
  int first_index, int last_index)
{
  int i;
  KheTimeSetClear(gs->pg_in_ts);
  for( i = first_index;  i <= last_index;  i++ )
    KheAddTime(gs, i, gs->pg_in_ts);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheProfileGroupingSetOutTimeSet(KHE_GROUP_SOLVER gs, int index1,    */
/*   int index2)                                                             */
/*                                                                           */
/*  Set gs->pg_out_ts to exclude the times at index1 and index2 from         */
/*  gs->pg_monitor_ts.  Either index may be out of range, in which case      */
/*  it is ignored.                                                           */
/*                                                                           */
/*****************************************************************************/

static void KheProfileGroupingSetOutTimeSet(KHE_GROUP_SOLVER gs, int index1,
 int index2)
{
  KheTimeSetClear(gs->pg_out_ts);
  KheAddTime(gs, index1, gs->pg_out_ts);
  KheAddTime(gs, index2, gs->pg_out_ts);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheTasksSatisfyingInAndOut(KHE_GROUP_SOLVER gs,                      */
/*    KHE_TIME_SET in_ts, KHE_TIME_SET out_ts)                               */
/*                                                                           */
/*  Return the number of tasks that cover all of in_ts and none of out_ts.   */
/*                                                                           */
/*****************************************************************************/

static int KheTasksSatisfyingInAndOut(KHE_GROUP_SOLVER gs,
  KHE_TIME_SET in_ts, KHE_TIME_SET out_ts)
{
  KHE_TIME t;  KHE_TIME_INFO ti;  KHE_ROOT_TASK_SET rts;  int i, res;
  t = KheTimeSetTime(in_ts, 0);
  ti = HaArray(gs->time_info, KheTimeIndex(t));
  res = 0;
  HaArrayForEach(ti->root_task_sets, rts, i)
    if( KheTimeSetSubset(in_ts, rts->time_set) &&
	KheTimeSetDisjoint(out_ts, rts->time_set) )
      res += KheTaskSetTaskCount(rts->tasks);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  int KheActiveTasksWithDomain(KHE_TIME_INFO ti, KHE_RESOURCE_GROUP rg)    */
/*                                                                           */
/*  Return the number of pg-active tasks in ti's root task sets with         */
/*  domain rg.                                                               */
/*                                                                           */
/*****************************************************************************/

static int KheActiveTasksWithDomain(KHE_TIME_INFO ti, KHE_RESOURCE_GROUP rg)
{
  int i, res;  KHE_ROOT_TASK_SET rts;
  if( DEBUG11 )
    fprintf(stderr, "[ KheActiveTasksWithDomain(%s, %s)\n",
      KheTimeId(ti->time), KheResourceGroupId(rg));
  res = 0;
  HaArrayForEach(ti->root_task_sets, rts, i)
  {
    if( DEBUG11 )
      KheRootTaskSetDebug(rts, 2, stderr);
    if( rts->pg_status == KHE_PG_ACTIVE &&
	KheResourceGroupEqual(rts->domain, rg) )
      res += KheTaskSetTaskCount(rts->tasks);
  }
  if( DEBUG11 )
    fprintf(stderr, "] KheActiveTasksWithDomain returning %d\n", res);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDoProfile(KHE_GROUP_SOLVER gs, KHE_TIME_INFO ti,                 */
/*    KHE_TIME_INFO adj_ti, int max_num)                                     */
/*                                                                           */
/*  Do profile grouping starting or ending at ti.  The number of groups      */
/*  to make is max_num.                                                      */
/*                                                                           */
/*****************************************************************************/

static void KheDoProfile(KHE_GROUP_SOLVER gs, KHE_TIME_INFO ti,
  KHE_TIME_INFO adj_ti, int max_num)
{
  KHE_RESOURCE_GROUP domain;
  int i, pos, domain_count, groups_made;  KHE_ROOT_TASK_SET rts;

  /* build the list of domains and counts */
  if( DEBUG10 )
    fprintf(stderr, "[ KheDoProfile(gs, ti %s, adj_ti %s, max_num %d)\n",
      KheTimeId(ti->time), adj_ti == NULL ? "-" : KheTimeId(adj_ti->time),
      max_num);
  HaArrayClear(gs->pg_domains);
  HaArrayClear(gs->pg_domain_counts);
  HaArrayForEach(ti->root_task_sets, rts, i)
    if( !HaArrayContains(gs->pg_domains, rts->domain, &pos) )
    {
      /* NB if there is history we won't reach this point anyway */
      domain_count = KheActiveTasksWithDomain(ti, rts->domain) -
	(adj_ti == NULL ? 0 : KheActiveTasksWithDomain(adj_ti, rts->domain));
      HaArrayAddLast(gs->pg_domains, rts->domain);
      HaArrayAddLast(gs->pg_domain_counts, domain_count);
      if( DEBUG10 )
	fprintf(stderr, "  adding (%s, %d)\n",
	  KheResourceGroupId(rts->domain), domain_count);
    }

  /* traverse the list and do the grouping */
  for( i = 0;  i < HaArrayCount(gs->pg_domains) && max_num > 0;  i++ )
  {
    domain = HaArray(gs->pg_domains, i);
    domain_count = HaArray(gs->pg_domain_counts, i);
    if( domain_count > 0 )
    {
      KheTimeBasedGrouping(gs, gs->pg_in_ts, gs->pg_out_ts, domain,
	min(domain_count, max_num), false, NULL, &groups_made,
	"profile grouping");
      max_num -= groups_made;
      if( DEBUG10 )
	fprintf(stderr, "  made %d groups for %s", groups_made,
	  KheResourceGroupId(rts->domain));
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDoProfileGroupingForMonitors(KHE_GROUP_SOLVER gs,                */
/*    KHE_LIMIT_ACTIVE_INTERVALS_MONITOR laim1,                              */
/*    KHE_LIMIT_ACTIVE_INTERVALS_MONITOR laim2, bool extended)               */
/*                                                                           */
/*  Carry out profile grouping based on laim1 and (optionally) laim2.        */
/*                                                                           */
/*****************************************************************************/

static void KheDoProfileGroupingForMonitors(KHE_GROUP_SOLVER gs,
  KHE_LIMIT_ACTIVE_INTERVALS_MONITOR laim1,
  KHE_LIMIT_ACTIVE_INTERVALS_MONITOR laim2, bool extended)
{
  int tcount, prev_profile, next_profile, i, demand, supply;
  KHE_TIME t;  KHE_TIME_INFO ti, prev_ti, next_ti;

  if( DEBUG9 )
  {
    fprintf(stderr, "[ KheDoProfileGroupingForMonitors(gs, laim1, %s%s)\n",
      laim2 != NULL ? "laim2" : "-", extended ? ", extended" : "");
    KheLimitActiveIntervalsMonitorDebug(laim1, 1, 2, stderr);
    if( laim2 != NULL )
      KheLimitActiveIntervalsMonitorDebug(laim2, 1, 2, stderr);
  }

  /* get min_limit, max_limit, history_before, and history_after */
  gs->pg_min_limit = KheLimitActiveIntervalsMonitorMinimum(laim1);
  gs->pg_max_limit = KheLimitActiveIntervalsMonitorMaximum(laim1);
  gs->pg_history_before = KheLimitActiveIntervalsMonitorHistoryBefore(laim1)> 0;
  gs->pg_history_after  = KheLimitActiveIntervalsMonitorHistoryAfter(laim1) > 0;
  if( laim2 != NULL )
  {
    if( KheLimitActiveIntervalsMonitorMinimum(laim2) > gs->pg_min_limit )
      gs->pg_min_limit = KheLimitActiveIntervalsMonitorMinimum(laim2);
    if( KheLimitActiveIntervalsMonitorMaximum(laim2) < gs->pg_max_limit )
      gs->pg_max_limit = KheLimitActiveIntervalsMonitorMaximum(laim2);
    if( KheLimitActiveIntervalsMonitorHistoryBefore(laim2) > 0 )
      gs->pg_history_before = true;
    if( KheLimitActiveIntervalsMonitorHistoryAfter(laim2)  > 0 )
      gs->pg_history_after = true;
  }

  /* do it if non-trivial min limit and either unextended or limits equal */
  if( gs->pg_min_limit >= 2 &&
      (!extended || gs->pg_min_limit == gs->pg_max_limit) )
  {
    /* build the time set for laim1 and initialize profile grouping fields */
    KheLimitActiveIntervalsMonitorSetTimes(laim1, gs->pg_monitor_ts);
    tcount = KheTimeSetTimeCount(gs->pg_monitor_ts);
    KheProfileGroupingBegin(gs);

    /* look for places where sequences start */
    prev_ti = NULL;
    prev_profile = gs->pg_history_before ? INT_MAX : 0;
    for( i = 0;  i <= tcount - 2;  i++ )
    {
      /* find the time and time info for i */
      t = KheTimeSetTime(gs->pg_monitor_ts, i);
      ti = HaArray(gs->time_info, KheTimeIndex(t));

      if( ti->pg_profile > prev_profile )
      {
	/* ti->pg_profile - prev_profile sequences must start at i */
	demand = ti->pg_profile - prev_profile;
        KheProfileGroupingSetInTimeSet(gs, i, i + gs->pg_min_limit - 1);
	KheProfileGroupingSetOutTimeSet(gs, i - 1, i + gs->pg_max_limit);
	supply = KheTasksSatisfyingInAndOut(gs, gs->pg_in_ts, gs->pg_out_ts);
	if( supply < demand )
	{
	  if( DEBUG9 )
	    fprintf(stderr, "  grouping from %s (supply %d, demand %d - %d):\n",
	      KheTimeId(t), supply, ti->pg_profile, prev_profile);
          KheDoProfile(gs, ti, prev_ti, demand - supply);
	}
      }

      /* set prev_profile for next iteration */
      prev_ti = ti;
      prev_profile = ti->pg_profile;
    }

    /* look for places where sequences end */
    next_ti = NULL;
    next_profile = gs->pg_history_after ? INT_MAX : 0;
    for( i = tcount - 1;  i >= 1;  i-- )
    {
      /* find the time and time info for i */
      t = KheTimeSetTime(gs->pg_monitor_ts, i);
      ti = HaArray(gs->time_info, KheTimeIndex(t));

      if( ti->pg_profile > next_profile )
      {
	/* ti->pg_profile - next_profile sequences must end at i */
	demand = ti->pg_profile - next_profile;
        KheProfileGroupingSetInTimeSet(gs, i - gs->pg_min_limit + 1, i);
	KheProfileGroupingSetOutTimeSet(gs, i + 1, i - gs->pg_max_limit);
	supply = KheTasksSatisfyingInAndOut(gs, gs->pg_in_ts, gs->pg_out_ts);
	if( supply < demand )
	{
	  if( DEBUG9 )
	    fprintf(stderr, "  grouping to %s (supply %d, demand %d - %d):\n",
	      KheTimeId(t), supply, ti->pg_profile, next_profile);
          KheDoProfile(gs, ti, next_ti, demand - supply);
	}
      }

      /* set next_profile for next iteration */
      next_ti = ti;
      next_profile = ti->pg_profile;
    }
    KheProfileGroupingEnd(gs);
  }

  if( DEBUG9 )
    fprintf(stderr, "] KheDoProfileGroupingForMonitors returning\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDoProfileGrouping(KHE_GROUP_SOLVER gs, bool extended)            */
/*                                                                           */
/*  Carry out profile grouping for gs.  If extended is true, carry out       */
/*  extended profile grouping.                                               */
/*                                                                           */
/*****************************************************************************/

static void KheDoProfileGrouping(KHE_GROUP_SOLVER gs, bool extended)
{
  KHE_MONITOR m;  KHE_LIMIT_ACTIVE_INTERVALS_MONITOR laimi, laimj;
  int i, j;  KHE_RESOURCE r;  KHE_RESOURCE_TYPE rt;

  /* find suitable limit active intervals monitors */
  HaArrayClear(gs->pg_monitors);
  rt = gs->resource_type;
  r = KheResourceTypeResource(rt, 0);
  for( i = 0;  i < KheSolnResourceMonitorCount(gs->soln, r);  i++ )
  {
    m = KheSolnResourceMonitor(gs->soln, r, i);
    if( KheMonitorTag(m) == KHE_LIMIT_ACTIVE_INTERVALS_MONITOR_TAG )
    {
      laimi = (KHE_LIMIT_ACTIVE_INTERVALS_MONITOR) m;
      if( KheMonitorSuitsProfileGrouping(laimi, rt, extended) )
      {
	if( DEBUG7 )
	{
	  KHE_LIMIT_ACTIVE_INTERVALS_CONSTRAINT c;
	  c = KheLimitActiveIntervalsMonitorConstraint(laimi);
	  fprintf(stderr, "  found suitable monitor, constraint %s\n",
	    KheConstraintId((KHE_CONSTRAINT) c));
	}
	HaArrayAddLast(gs->pg_monitors, laimi);
      }
    }
  }

  /* find pairs of suitable monitors and do the job for each pair */
  for( i = 0;  i < HaArrayCount(gs->pg_monitors);  i++ )
  {
    laimi = HaArray(gs->pg_monitors, i);
    for( j = i + 1;  j < HaArrayCount(gs->pg_monitors);  j++ )
    {
      laimj = HaArray(gs->pg_monitors, j);
      if( KheLimitActiveIntervalsMonitorsCanGroupTogether(laimi, laimj) )
      {
	KheDoProfileGroupingForMonitors(gs, laimi, laimj, extended);
	HaArrayDeleteAndPlug(gs->pg_monitors, j);  /* order matters! */
	HaArrayDeleteAndPlug(gs->pg_monitors, i);  /* order matters! */
	i--;
	break;
      }
    }
  }

  /* find single suitable monitors and do the job for them */
  HaArrayForEach(gs->pg_monitors, laimi, i)
    KheDoProfileGroupingForMonitors(gs, laimi, NULL, extended);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "main function"                                                */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool KheGroupByResourceConstraintsForResourceType(KHE_SOLN soln,         */
/*    KHE_RESOURCE_TYPE rt, int max_days, bool resource_invariant,           */
/*    KHE_FRAME common_frame, KHE_EVENT_TIMETABLE_MONITOR etm,               */
/*    KHE_TASK_SET ts, HA_ARENA a)                                           */
/*                                                                           */
/*  Carry out group by resource constraints for resource type rt.  There     */
/*  is no point in doing anything if there are fewer than two resources.     */
/*                                                                           */
/*****************************************************************************/

static bool KheGroupByResourceConstraintsForResourceType(KHE_SOLN soln,
  KHE_RESOURCE_TYPE rt, int max_days, bool resource_invariant,
  KHE_FRAME common_frame, KHE_EVENT_TIMETABLE_MONITOR etm,
  KHE_TASK_SET ts, HA_ARENA a)
{
  KHE_GROUP_SOLVER gs;
  if( KheResourceTypeResourceCount(rt) > 1 )
  {
    gs = KheGroupByResourceConstraintsSolverMake(soln, rt, max_days,
      resource_invariant, common_frame, etm, ts, a);
    KheEliminateCombinations(gs);
    KheDoCombinatorialGrouping(gs);
    KheDoProfileGrouping(gs, false);
    KheDoProfileGrouping(gs, true);
    return gs->soln_changed;
  }
  else
    return false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheGroupByResourceConstraints(KHE_SOLN soln, KHE_RESOURCE_TYPE rt,  */
/*    KHE_OPTIONS options, KHE_TASK_SET ts)                                  */
/*                                                                           */
/*  Group the tasks of soln of type rt by resource constraints.  If rt is    */
/*  NULL, do this for each of the resource types of soln's instance.         */
/*                                                                           */
/*  If ts is non-NULL, add every task that is given an assignment to ts.     */
/*                                                                           */
/*****************************************************************************/

bool KheGroupByResourceConstraints(KHE_SOLN soln, KHE_RESOURCE_TYPE rt,
  KHE_OPTIONS options, KHE_TASK_SET ts)
{
  int max_days, i;  bool res, resource_invariant;  KHE_INSTANCE ins;
  HA_ARENA a;  KHE_FRAME common_frame;  KHE_EVENT_TIMETABLE_MONITOR etm;

  /* boilerplate */
  ins = KheSolnInstance(soln);
  if( DEBUG1 )
    fprintf(stderr, "[ KheGroupByResourceConstraints(%s %s, -)\n",
      KheInstanceId(ins), rt == NULL ? "all types" : KheResourceTypeId(rt));

  /* return immediately if option rs_group_by_frame_off is true */
  if( KheOptionsGetBool(options, "rs_group_by_rc_off", false) )
  {
    if( DEBUG1 )
      fprintf(stderr,"] KheGroupByResourceConstraints returning false (off)\n");
    return false;
  }

  /* get the other options used by this function */
  max_days = KheOptionsGetInt(options, "rs_group_by_rc_max_days", 3);
  resource_invariant = KheOptionsGetBool(options, "rs_invariant", false);
  common_frame = KheOptionsFrame(options, "gs_common_frame", soln);
  etm = (KHE_EVENT_TIMETABLE_MONITOR)
    KheOptionsGetObject(options, "gs_event_timetable_monitor", NULL);
  HnAssert(etm != NULL, "KheGroupByResourceConstraints: required "
    "option gs_event_timetable_monitor not present");

  /* do grouping by resource constraints for each resource type */
  a = KheSolnArenaBegin(soln);
  res = false;
  if( rt != NULL )
  {
    if( KheGroupByResourceConstraintsForResourceType(soln, rt, max_days,
	resource_invariant, common_frame, etm, ts, a) )
      res = true;
  }
  else for( i = 0;  i < KheInstanceResourceTypeCount(ins);  i++ )
  {
    rt = KheInstanceResourceType(ins, i);
    if( KheGroupByResourceConstraintsForResourceType(soln, rt, max_days,
	resource_invariant, common_frame, etm, ts, a) )
      res = true;
  }
  KheSolnArenaEnd(soln, a);

  /* all done */
  if( DEBUG1 )
    fprintf(stderr, "] KheGroupByResourceConstraints returning %s\n",
      bool_show(res));
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "public functions"                                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_GROUP_SOLVER KheGroupSolverMake(KHE_SOLN soln, KHE_RESOURCE_TYPE rt, */
/*    HA_ARENA a)                                                            */
/*                                                                           */
/*  Make a new group solver which works with the tasks of soln whose type    */
/*  is rt and which lie in meets whose times are assigned.                   */
/*                                                                           */
/*  All memory comes from arena a, and the solver is deleted when a is.      */
/*                                                                           */
/*****************************************************************************/

KHE_GROUP_SOLVER KheGroupSolverMake(KHE_SOLN soln, KHE_RESOURCE_TYPE rt,
  HA_ARENA a)
{
  KHE_GROUP_SOLVER res;  KHE_TIME_GROUP tg;  KHE_TASK task;  KHE_TIME t;
  int i;  KHE_CG_TIME_GROUP_INFO tgi;  KHE_TIME_INFO ti;  KHE_EVENT_RESOURCE er;

  /* general fields */
  HaMake(res, a);
  res->arena = a;
  res->soln = soln;
  res->instance = KheSolnInstance(soln);
  res->resource_type = rt;
  res->soln_changed = false;
  res->task_set = NULL;

  /* time groups and covers */
  HaArrayInit(res->time_groups);
  HaArrayInit(res->covers);

  /* time info */
  HaArrayInit(res->time_info, a);
  for( i = 0;  i < KheInstanceTimeCount(res->instance);  i++ )
  {
    t = KheInstanceTime(res->instance, i);
    ti = KheTimeInfoMake(res, t);
    HaArrayAddLast(res->time_info, ti);
  }
  HaArrayInit(res->free_root_task_sets, a);

  /* time-based grouping */
  HaArrayInit(res->tbg_rts_set, a);
  HaArrayInit(res->tbg_ignore_for_leader_set, a);
  res->tbg_in_ts = KheTimeSetMake(res->instance, a);
  res->tbg_out_ts = KheTimeSetMake(res->instance, a);

  /* add unassigned tasks of the right type to time group info */
  for( i = 0;  i < KheSolnTaskCount(soln);  i++ )
  {
    task = KheSolnTask(soln, i);
    if( !KheTaskIsCycleTask(task) && KheTaskResourceType(task) == rt &&
	(KheTaskAsst(task) == NULL || KheTaskIsCycleTask(KheTaskAsst(task))) )
    {
      er = KheTaskEventResource(task);
      if( er != NULL && KheEventResourceNeedsAssignment(er) == KHE_YES )
	KheGroupSolverAddTask(res, task);
    }
  }

  /* combinatorial grouping */
  res->cg_max_days = max_days;
  res->cg_first_index = -1;
  res->cg_last_index = -1;
  res->cg_in_ts = KheTimeSetMake(res->instance, a);
  res->cg_out_ts = KheTimeSetMake(res->instance, a);
  res->cg_success_count = 0;
  res->cg_success_in_ts = KheTimeSetMake(res->instance, a);
  res->cg_success_out_ts = KheTimeSetMake(res->instance, a);
  res->cg_trace = KheTraceMake((KHE_GROUP_MONITOR) soln);

  /* all done */
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheGroupSolverAddGroup(KHE_GROUP_SOLVER gs, KHE_TASK leader_task)   */
/*                                                                           */
/*  Inform gs that leader_task has just been grouped.                        */
/*                                                                           */
/*****************************************************************************/

/* *** too hard!
void KheGroupSolverAddGroup(KHE_GROUP_SOLVER gs, KHE_TASK leader_task)
{
  ** sti ll to do **
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheGroupSolverClearTimeGroups(KHE_GROUP_SOLVER gs)                  */
/*                                                                           */
/*  Clear out the time groups of gs ready for a fresh solve.                 */
/*                                                                           */
/*****************************************************************************/

void KheGroupSolverClearTimeGroups(KHE_GROUP_SOLVER gs)
{
  HaArrayClear(gs->time_groups);
  HaArrayClear(gs->covers);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheGroupSolverAddTimeGroup(KHE_GROUP_SOLVER gs, KHE_TIME_GROUP tg,  */
/*    KHE_GROUP_SOLVER_COVER_TYPE cover)                                     */
/*                                                                           */
/*  Add tg with the given cover type to gs.                                  */
/*                                                                           */
/*****************************************************************************/

void KheGroupSolverAddTimeGroup(KHE_GROUP_SOLVER gs, KHE_TIME_GROUP tg,
  KHE_GROUP_SOLVER_COVER_TYPE cover)
{
  if( HaArrayCount(gs->time_groups) == 0 )
    HnAssert(cover != KHE_GROUP_SOLVER_COVER_PREV, "KheGroupSolverAddTimeGroup:"
      " first time group has cover KHE_GROUP_SOLVER_COVER_PREV");
  HaArrayAddLast(gs->time_groups, tg);
  HaArrayAddLast(gs->covers, cover);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheGroupSolverTestSolve(KHE_GROUP_SOLVER gs,                        */
/*    KHE_GROUP_SOLVER_COST_TYPE ct, KHE_RESOURCE_GROUP domain,              */
/*    bool allow_single, KHE_TASK *leader_task)                              */
/*                                                                           */
/*  Carry out a test solve.  If successful, return true with *leader_task    */
/*  set to the leader task of the new group.  Otherwise return false with    */
/*  *leader_task set to NULL.                                                */
/*                                                                           */
/*****************************************************************************/

bool KheGroupSolverTestSolve(KHE_GROUP_SOLVER gs,
  KHE_GROUP_SOLVER_COST_TYPE ct, KHE_RESOURCE_GROUP domain,
  bool allow_single, KHE_TASK *leader_task)
{
  /* still to do */
  HnAbort("KheGroupSolverTestSolve still to do");
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*****************************************************************************/

static void SolveFrom(KHE_GROUP_SOLVER gs, int curr_index, bool prev_assigned)
{
  KHE_TIME_GROUP tg;  KHE_GROUP_SOLVER_COVER_TYPE cover;
  if( curr_index >= HaArrayCount(gs->time_groups) )
  {
    /* at the end of the time groups, do some time-based grouping */
  }
  else
  {
    /* try all acceptable alternatives at curr_index */
    tg = HaArray(gs->time_groups, curr_index);
    cover = HaArray(gs->covers, curr_index);
    switch( cost_type )
    {
      case KHE_GROUP_SOLVER_COVER_YES:

	break;

      case KHE_GROUP_SOLVER_COVER_NO:

	break;

      case KHE_GROUP_SOLVER_COVER_FREE:

	break;

      case KHE_GROUP_SOLVER_COVER_PREV:

	if( curr_index == 
	break;

      default:

	HnAbort("SolveFrom: unexpected cost_type (%d)", cost_type);
	break;
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  int KheGroupSolverSolve(KHE_GROUP_SOLVER gs, int max_num,                */
/*    KHE_GROUP_SOLVER_COST_TYPE ct, KHE_RESOURCE_GROUP domain,              */
/*    bool allow_single, KHE_TASK_SET ts, char *debug_str)                   */
/*                                                                           */
/*  Solve gs, adding up to max_num groups.                                   */
/*                                                                           */
/*****************************************************************************/

int KheGroupSolverSolve(KHE_GROUP_SOLVER gs, int max_num,
  KHE_GROUP_SOLVER_COST_TYPE ct, KHE_RESOURCE_GROUP domain,
  bool allow_single, KHE_TASK_SET ts, char *debug_str)
{
  /* save parameters in gs */
  gs->max_num = max_num;
  gs->cost_type = ct;
  gs->domain = domain;
  gs->allow_single = allow_single;
  gs->task_set = ts;
  gs->debug_str = debug_str;

  SolveFrom(gs, 0, false);
  /* still to do */
  HnAbort("KheGroupSolverSolve still to do");
}
