
/*****************************************************************************/
/*                                                                           */
/*  THE KHE HIGH SCHOOL TIMETABLING ENGINE                                   */
/*  COPYRIGHT (C) 2010 Jeffrey H. Kingston                                   */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 3, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         khe_sr_resource_pair_run.c                                 */
/*  DESCRIPTION:  Resource pair run repair                                   */
/*                                                                           */
/*****************************************************************************/
#include "khe_solvers.h"
#include <limits.h>
#define bool_show(x) ((x) ? "true" : "false")

#define SPLIT_RUNS 0
#define TRY_UNASSIGNMENTS false		/* slow and cost did not reduce      */

#define DEBUG1 1			/* public functions                  */
#define DEBUG2 0			/* solve pairs                       */
#define DEBUG3 0			/* SetRuns                           */
#define DEBUG4 0			/* RunMake and RunDelete             */
#define DEBUG5 0			/*                                   */
#define DEBUG6 1			/* brief display of improvements     */

#define	DEBUG5 0			/* solve pair details                */
#define DEBUG5_ID1  "CT_19"
#define DEBUG5_ID2  "CT_21"


/*****************************************************************************/
/*                                                                           */
/*  RPAIR_SELECT_TYPE - types of selection                                   */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  RPAIR_SELECT_NONE,
  RPAIR_SELECT_ADJACENT,
  RPAIR_SELECT_DEFECTIVE,
  RPAIR_SELECT_ALL
} RPAIR_SELECT_TYPE;


/*****************************************************************************/
/*                                                                           */
/*  KHE_RUN - one run of adjacent tasks, with the interval they occupy       */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_run_rec {
  KHE_TASK_SET		task_set;			/* the run's tasks   */
  /* bool		needs_asst; */			/* tasks need asst   */
  int			first_index;			/* start of interval */
  int			last_index;			/* end of interval   */
} *KHE_RUN;

typedef HA_ARRAY(KHE_RUN) ARRAY_KHE_RUN;


/*****************************************************************************/
/*                                                                           */
/*  KHE_RUN_RESOURCE - information about one resource                        */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_run_resource_rec {
  KHE_RESOURCE		resource;		/* the resource              */
  ARRAY_KHE_RUN		runs;			/* the resource's runs       */
  int			first_index;		/* interval runs are for     */
  int			last_index;		/* interval runs are for     */
  bool			defective;		/* if resource is defective  */
} *KHE_RUN_RESOURCE;

typedef HA_ARRAY(KHE_RUN_RESOURCE) ARRAY_KHE_RUN_RESOURCE;


/*****************************************************************************/
/*                                                                           */
/*  KHE_RUN_COMPONENT - one component of the clash graph                     */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_run_component_rec {
  ARRAY_KHE_RUN		part1;
  ARRAY_KHE_RUN		part2;
  bool			part1_needs_asst;		/* tasks need asst   */
  bool			part2_needs_asst;		/* tasks need asst   */
} *KHE_RUN_COMPONENT;

typedef HA_ARRAY(KHE_RUN_COMPONENT) ARRAY_KHE_RUN_COMPONENT;


/*****************************************************************************/
/*                                                                           */
/*  KHE_RUN_SOLVER - a solver for runs                                       */
/*                                                                           */
/*****************************************************************************/

typedef HA_ARRAY(KHE_TASK_SET) ARRAY_KHE_TASK_SET;

typedef struct khe_run_solver_rec {

  /* fixed attributes of the solve object */
  HA_ARENA			arena;
  KHE_SOLN			soln;
  KHE_TASK_FINDER		task_finder;
  bool				resource_invariant;
  int				max_assignments;

  /* free lists */
  ARRAY_KHE_RUN_RESOURCE	resources;
  ARRAY_KHE_RUN_RESOURCE	free_resources;
  ARRAY_KHE_RUN			free_runs;
  ARRAY_KHE_RUN_COMPONENT	free_run_components;

  /* defined during individual solves */
  ARRAY_KHE_RUN_COMPONENT	components;
  KHE_RESOURCE			resource1;
  KHE_RESOURCE			resource2;
  int				curr_assignments;
  KHE_MARK			init_mark;
  int				init_defect_count;
  bool				debug_on;
} *KHE_RUN_SOLVER;


/*****************************************************************************/
/*                                                                           */
/*  Submodule "select type"                                                  */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  RPAIR_SELECT_TYPE KheSelectTypeGet(KHE_OPTIONS options)                  */
/*                                                                           */
/*  Return an enum value representing the rs_rpair_select option.            */
/*                                                                           */
/*****************************************************************************/

static RPAIR_SELECT_TYPE KheSelectTypeGet(KHE_OPTIONS options)
{
  char *val;
  val = KheOptionsGet(options, "rs_rpair_select", "defective");
  if( strcmp(val, "defective") == 0 )
    return RPAIR_SELECT_DEFECTIVE;
  else if( strcmp(val, "none") == 0 )
    return RPAIR_SELECT_NONE;
  else if( strcmp(val, "adjacent") == 0 )
    return RPAIR_SELECT_ADJACENT;
  else if( strcmp(val, "all") == 0 )
    return RPAIR_SELECT_ALL;
  else
  {
    HnAbort("KheResourcePairRunRepair: unknown rs_rpair_select option"
      " value \"%s\"", val);
    return RPAIR_SELECT_ALL;  /* keep compiler happy */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "runs"                                                         */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_RUN KheRunMake(KHE_RUN_SOLVER rs)                                    */
/*                                                                           */
/*  Make or get a run object, whose task set is initially empty and          */
/*  whose first_index, last_index, and needs_asst fields are undefined.      */
/*                                                                           */
/*****************************************************************************/

static KHE_RUN KheRunMake(KHE_RUN_SOLVER rs)
{
  KHE_RUN res;
  if( HaArrayCount(rs->free_runs) > 0 )
  {
    /* get the run object from the free list */
    res = HaArrayLastAndDelete(rs->free_runs);
    KheTaskSetClear(res->task_set);
    if( DEBUG4 )
      fprintf(stderr, "KheRunMake returning %p from free list\n", (void *) res);
  }
  else
  {
    /* make a new run object */
    HaMake(res, rs->arena);
    res->task_set = KheTaskSetMake(rs->soln);
    if( DEBUG4 )
      fprintf(stderr, "KheRunMake returning %p from arena\n", (void *) res);
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheRunDelete(KHE_RUN run, KHE_RUN_SOLVER rs)                        */
/*                                                                           */
/*  Delete run; place it on rs's free list.  Its task set goes with it.      */
/*                                                                           */
/*****************************************************************************/

static void KheRunDelete(KHE_RUN run, KHE_RUN_SOLVER rs)
{
  if( DEBUG4 )
    fprintf(stderr, "KheRunDelete deleting %p\n", (void *) run);
  HaArrayAddLast(rs->free_runs, run);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheRunSplits(KHE_RUN run, KHE_RUN_SOLVER rs,                        */
/*    KHE_RUN *run1, KHE_RUN *run2)                                          */
/*                                                                           */
/*  If run can be split into two pieces, return true and set *run1 and       */
/*  *run2 to the two pieces, deleting run.  Otherwise return false,          */
/*  leaving run unchanged.                                                   */
/*                                                                           */
/*****************************************************************************/

static bool KheRunSplits(KHE_RUN run, KHE_RUN_SOLVER rs,
  KHE_RUN *run1, KHE_RUN *run2)
{
  int i, half_way;

  /* no good if less than four days covered */
  /* ***
  if( run->last_index - run->first_index + 1 < 4 )
    return false;
  *** */

  /* no good if less than two tasks in task set */
  if( KheTaskSetTaskCount(run->task_set) < 2 )
    return false;

  /* first half in *run1, second half in *run2 */
  half_way = KheTaskSetTaskCount(run->task_set) / 2;
  *run1 = KheRunMake(rs);
  *run2 = KheRunMake(rs);
  for( i = 0;  i < half_way;  i++ )
    KheTaskSetAddTask((*run1)->task_set, KheTaskSetTask(run->task_set, i));
  KheTaskFinderTaskSetInterval(rs->task_finder, (*run1)->task_set,
    &(*run1)->first_index, &(*run1)->last_index);
  for( i = half_way;  i < KheTaskSetTaskCount(run->task_set);  i++ )
    KheTaskSetAddTask((*run2)->task_set, KheTaskSetTask(run->task_set, i));
  KheTaskFinderTaskSetInterval(rs->task_finder, (*run2)->task_set,
    &(*run2)->first_index, &(*run2)->last_index);
  if( (*run1)->last_index >= (*run2)->first_index )
  {
    KheRunDelete(*run1, rs);
    KheRunDelete(*run2, rs);
    return false;
  }
  else
  {
    KheRunDelete(run, rs);
    return true;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheRunMoveCheck(KHE_RUN run, KHE_RESOURCE r)                        */
/*                                                                           */
/*  Check moving run to r, returning true if it would succeed.               */
/*                                                                           */
/*****************************************************************************/

static bool KheRunMoveCheck(KHE_RUN run, KHE_RESOURCE r)                       
{
  return KheTaskSetMoveResourceCheck(run->task_set, r);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheRunMove(KHE_RUN run, KHE_RESOURCE r)                             */
/*                                                                           */
/*  Move run to r, returning true if successful.                             */
/*                                                                           */
/*****************************************************************************/

static bool KheRunMove(KHE_RUN run, KHE_RESOURCE r)                       
{
  return KheTaskSetMoveResource(run->task_set, r);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheRunDebug(KHE_RUN run, int verbosity, int indent, FILE *fp)       */
/*                                                                           */
/*  Debug print of run onto fp with the given verbosity and indent.          */
/*                                                                           */
/*****************************************************************************/

static void KheRunDebug(KHE_RUN run, int verbosity, int indent, FILE *fp)
{
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  if( DEBUG4 )
    fprintf(fp, "%p", (void *) run);
  fprintf(fp, "(%02d-%02d)", run->first_index, run->last_index);
  KheTaskSetDebug(run->task_set, verbosity, -1, fp);
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "run resources"                                                */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_RUN_RESOURCE KheRunResourceMake(KHE_RUN_SOLVER rs, KHE_RESOURCE r)   */
/*                                                                           */
/*  Make a run resource object with these attributes.                        */
/*                                                                           */
/*****************************************************************************/

static KHE_RUN_RESOURCE KheRunResourceMake(KHE_RUN_SOLVER rs, KHE_RESOURCE r)
{
  KHE_RUN_RESOURCE res;
  if( HaArrayCount(rs->free_resources) > 0 )
  {
    /* get the run resource object from the free list */
    res = HaArrayLastAndDelete(rs->free_resources);
    HaArrayClear(res->runs);
  }
  else
  {
    /* make a new run resource object */
    HaMake(res, rs->arena);
    HaArrayInit(res->runs, rs->arena);
  }
  res->resource = r;
  res->first_index = -1;
  res->last_index = -1;
  res->defective = false;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheRunResourceDelete(KHE_RUN_RESOURCE rr, KHE_RUN_SOLVER rs)        */
/*                                                                           */
/*  Delete rr, including its runs.  Both rr and its run objects go onto      */
/*  free lists in rs.                                                        */
/*                                                                           */
/*****************************************************************************/

static void KheRunResourceDelete(KHE_RUN_RESOURCE rr, KHE_RUN_SOLVER rs)
{
  int i;  KHE_RUN run;
  HaArrayForEach(rr->runs, run, i)
    KheRunDelete(run, rs);
  HaArrayAddLast(rs->free_resources, rr);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskSetNeedsAsst(KHE_TASK_SET ts)                                */
/*                                                                           */
/*  Return true if the first task of ts needs assignment.                    */
/*                                                                           */
/*****************************************************************************/

static bool KheTaskSetNeedsAsst(KHE_TASK_SET ts)
{
  HnAssert(KheTaskSetTaskCount(ts) > 0, "KheTaskSetNeedsAsst internal error");
  return KheTaskNeedsAssignment(KheTaskSetFirst(ts));
}


/*****************************************************************************/
/*                                                                           */
/*  void KheRunResourceSetRuns(KHE_RUN_RESOURCE rr, KHE_RUN_SOLVER rs)       */
/*                                                                           */
/*  Set (or reset) rr->resource's runs.                                      */
/*                                                                           */
/*****************************************************************************/
static void KheRunResourceDebug(KHE_RUN_RESOURCE rr, int verbosity,
  int indent, FILE *fp);

static void KheRunResourceSetRuns(KHE_RUN_RESOURCE rr, KHE_RUN_SOLVER rs,
  int first_index, int last_index)
{
  int i;  KHE_RUN run, run1, run2;  KHE_RESOURCE_TYPE rt;
  if( DEBUG3 )
    fprintf(stderr, "[ KheRunResourceSetRuns(rr, rs, %d, %d)\n",
      first_index, last_index);

  if( rr->first_index != first_index || rr->last_index != last_index )
  {
    /* reset rr's interval */
    rr->first_index = first_index;
    rr->last_index = last_index;

    /* clear out any old runs */
    HaArrayForEach(rr->runs, run, i)
      KheRunDelete(run, rs);
    HaArrayClear(rr->runs);

    /* add the new runs */
    rt = KheResourceResourceType(rr->resource);
    run = KheRunMake(rs);
    while( KheFindTaskRunRight(rs->task_finder, rt, rr->resource,
	TRY_UNASSIGNMENTS, first_index, run->task_set,
	&run->first_index, &run->last_index) && run->last_index <= last_index )
    {
      HnAssert(run->first_index >= 0, "KheRunResourceSetRuns internal error 1");
      HnAssert(run->last_index >= 0,  "KheRunResourceSetRuns internal error 2");
      if( SPLIT_RUNS && KheRunSplits(run, rs, &run1, &run2) )
      {
	HaArrayAddLast(rr->runs, run1);
	HaArrayAddLast(rr->runs, run2);
	first_index = run2->last_index + 2;
      }
      else
      {
	/* run->needs_asst = KheTaskSetNeedsAsst(run->task_set); */
	HaArrayAddLast(rr->runs, run);
	first_index = run->last_index + 1;
      }
      run = KheRunMake(rs);
    }
    KheRunDelete(run, rs);
  }
  if( DEBUG3 )
  {
    KheRunResourceDebug(rr, 2, 2, stderr);
    fprintf(stderr, "] KheRunResourceSetRuns returning\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheRunResourceSetDefective(KHE_RUN_RESOURCE rr, KHE_RUN_SOLVER rs)  */
/*                                                                           */
/*  Set the defective flag of rr.                                            */
/*                                                                           */
/*****************************************************************************/

static void KheRunResourceSetDefective(KHE_RUN_RESOURCE rr, KHE_RUN_SOLVER rs)
{
  rr->defective = (KheSolnResourceCost(rs->soln, rr->resource) > 0);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheRunResourceDebug(KHE_RUN_RESOURCE rr, int verbosity,             */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of rr onto fp with the given verbosity and indent.           */
/*  The runs are only printed if indent >= 0.                                */
/*                                                                           */
/*****************************************************************************/

static void KheRunResourceDebug(KHE_RUN_RESOURCE rr, int verbosity,
  int indent, FILE *fp)
{
  KHE_RUN run;  int i;
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  if( rr->first_index >= 0 )
  {
    fprintf(fp, "[%s%s %d-%d:", KheResourceId(rr->resource),
      rr->defective ? "" : " defective", rr->first_index, rr->last_index);
    if( indent >= 0 )
    {
      fprintf(fp, "\n");
      HaArrayForEach(rr->runs, run, i)
	KheRunDebug(run, verbosity, indent + 2, fp);
      fprintf(fp, "%*s]", indent, "");
    }
    else
      fprintf(fp, " %d runs]", HaArrayCount(rr->runs));
  }
  else
    fprintf(fp, "[%s not up to date]", KheResourceId(rr->resource));
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "run components"                                               */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_RUN_COMPONENT KheRunComponentMake(KHE_RUN_SOLVER rs)                 */
/*                                                                           */
/*  Make a new, initially empty run component and add it to rs.              */
/*                                                                           */
/*****************************************************************************/

static KHE_RUN_COMPONENT KheRunComponentMake(KHE_RUN_SOLVER rs)
{
  KHE_RUN_COMPONENT res;
  if( HaArrayCount(rs->free_run_components) > 0 )
  {
    /* get an existing run component object from rs's free list */
    res = HaArrayLastAndDelete(rs->free_run_components);
    HaArrayClear(res->part1);
    HaArrayClear(res->part2);
  }
  else
  {
    /* make a new run component object using rs's arena */
    HaMake(res, rs->arena);
    HaArrayInit(res->part1, rs->arena);
    HaArrayInit(res->part2, rs->arena);
  }
  res->part1_needs_asst = !TRY_UNASSIGNMENTS;
  res->part2_needs_asst = !TRY_UNASSIGNMENTS;
  HaArrayAddLast(rs->components, res);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheRunComponentAddToPart1(KHE_RUN_COMPONENT rc, KHE_RUN run1)       */
/*  void KheRunComponentAddToPart2(KHE_RUN_COMPONENT rc, KHE_RUN run2)       */
/*                                                                           */
/*  Add run1 to part1 (or run2 to part2) of rc.  Keep track of whether       */
/*  the parts need assignment.                                               */
/*                                                                           */
/*****************************************************************************/

static void KheRunComponentAddToPart1(KHE_RUN_COMPONENT rc, KHE_RUN run1)
{
  HaArrayAddLast(rc->part1, run1);
  if( TRY_UNASSIGNMENTS && KheTaskSetNeedsAsst(run1->task_set) )
    rc->part1_needs_asst = true;
}

static void KheRunComponentAddToPart2(KHE_RUN_COMPONENT rc, KHE_RUN run2)
{
  HaArrayAddLast(rc->part2, run2);
  if( TRY_UNASSIGNMENTS && KheTaskSetNeedsAsst(run2->task_set) )
    rc->part2_needs_asst = true;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheRunComponentDelete(KHE_RUN_COMPONENT rc, KHE_RUN_SOLVER rs)      */
/*                                                                           */
/*  Delete rc, by returning it to the free lists in rs.                      */
/*                                                                           */
/*  Do not delete rc's runs; they are the property of the run resources      */
/*  that were used to make the component, and those are not being deleted.   */
/*                                                                           */
/*****************************************************************************/

static void KheRunComponentDelete(KHE_RUN_COMPONENT rc, KHE_RUN_SOLVER rs)
{
  HaArrayAddLast(rs->free_run_components, rc);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheRunComponentMoveCheck(KHE_RUN_COMPONENT rc, KHE_RESOURCE r1,     */
/*    KHE_RESOURCE r2)                                                       */
/*                                                                           */
/*  Check the move of rc->part1 to r1 and rc->part2 to r2, returning true    */
/*  if it would succeed.                                                     */
/*                                                                           */
/*****************************************************************************/

static bool KheRunComponentMoveCheck(KHE_RUN_COMPONENT rc, KHE_RESOURCE r1,
  KHE_RESOURCE r2)
{
  KHE_RUN run;  int i;
  HaArrayForEach(rc->part1, run, i)
    if( !KheRunMoveCheck(run, r1) )
      return false;
  HaArrayForEach(rc->part2, run, i)
    if( !KheRunMoveCheck(run, r2) )
      return false;
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheRunComponentMove(KHE_RUN_COMPONENT rc, KHE_RESOURCE r1,          */
/*    KHE_RESOURCE r2)                                                       */
/*                                                                           */
/*  Move rc->part1 to r1 and rc->part2 to r2, returning true if successful.  */
/*                                                                           */
/*****************************************************************************/

static bool KheRunComponentMove(KHE_RUN_COMPONENT rc, KHE_RESOURCE r1,
  KHE_RESOURCE r2)
{
  KHE_RUN run;  int i;
  HaArrayForEach(rc->part1, run, i)
    if( !KheRunMove(run, r1) )
      return false;
  HaArrayForEach(rc->part2, run, i)
    if( !KheRunMove(run, r2) )
      return false;
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheRunComponentDebug(KHE_RUN_COMPONENT rc, int verbosity,           */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of rc onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void KheRunComponentDebug(KHE_RUN_COMPONENT rc, int verbosity,
  int indent, FILE *fp)
{
  KHE_RUN run;  int i;
  if( indent >= 0 )
  {
    fprintf(fp, "%*sRunComponent(\n", indent, "");
    HaArrayForEach(rc->part1, run, i)
      KheRunDebug(run, verbosity, indent + 2, fp);
    fprintf(fp, "%*s--------------------------------------\n", indent + 2, "");
    HaArrayForEach(rc->part2, run, i)
      KheRunDebug(run, verbosity, indent + 2, fp);
    fprintf(fp, "%*s)\n", indent, "");
  }
  else
    fprintf(fp, "RunComponent(...)");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "using the solver to solve one pair"                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheGetRun(KHE_RUN_RESOURCE rr, int i, KHE_RUN *run, int *fi)        */
/*                                                                           */
/*  Find the run of rr indexed by i, and its first index; or NULL and        */
/*  INT_MAX if none.                                                         */
/*                                                                           */
/*****************************************************************************/

static void KheGetRun(KHE_RUN_RESOURCE rr, int i, KHE_RUN *run, int *fi)
{
  if( i < HaArrayCount(rr->runs) )
  {
    *run = HaArray(rr->runs, i), *fi = (*run)->first_index;
    if( DEBUG5 && *fi < 0 )
    {
      fprintf(stderr, "KheGetRun failing at run %d of rr:\n", i);
      KheRunResourceDebug(rr, 2, 2, stderr);
      HnAbort("KheGetRun internal error");
    }
  }
  else
    *run = NULL, *fi = INT_MAX;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheRunSolverFindComponents(KHE_RUN_SOLVER rs,                       */
/*    KHE_RUN_RESOURCE rr1, KHE_RUN_RESOURCE rr2)                            */
/*                                                                           */
/*  Find the components of rr1+rr2.                                          */
/*                                                                           */
/*****************************************************************************/

static void KheRunSolverFindComponents(KHE_RUN_SOLVER rs,
  KHE_RUN_RESOURCE rr1, KHE_RUN_RESOURCE rr2)
{
  int i1, i2, curr_comp_max, fi1, fi2;  KHE_RUN run1, run2;
  KHE_RUN_COMPONENT curr_comp;
  HaArrayClear(rs->components);
  curr_comp = NULL;
  curr_comp_max = -1;
  i1 = i2 = 0;
  KheGetRun(rr1, i1, &run1, &fi1);
  KheGetRun(rr2, i2, &run2, &fi2);
  while( run1 != NULL || run2 != NULL )
  {
    if( fi1 <= fi2 )
    {
      /* handle run1, then move to next run of rr1 */
      if( fi1 > curr_comp_max )
      {
	curr_comp = KheRunComponentMake(rs);
	HnAssert(curr_comp != NULL,
	  "KheRunSolverFindComponents internal error 0");
      }
      HnAssert(curr_comp != NULL,
	"KheRunSolverFindComponents internal error 1 (curr_comp_max == %d, "
	"run1 %s NULL, fi1 = %d, run2 %s NULL, fi2 = %d)", curr_comp_max,
	run1 == NULL ? "==" : "!=", fi1, run2 == NULL ? "==" : "!=", fi2);
      /* HaArrayAddLast(curr_comp->part1, run1); */
      KheRunComponentAddToPart1(curr_comp, run1);
      if( run1->last_index > curr_comp_max )
	curr_comp_max = run1->last_index;
      KheGetRun(rr1, ++i1, &run1, &fi1);
    }
    else
    {
      /* handle run2, then move to next run of rr2 */
      if( fi2 > curr_comp_max )
	curr_comp = KheRunComponentMake(rs);
      HnAssert(curr_comp != NULL,"KheRunSolverFindComponents internal error 2");
      /* HaArrayAddLast(curr_comp->part2, run2); */
      KheRunComponentAddToPart2(curr_comp, run2);
      if( run2->last_index > curr_comp_max )
	curr_comp_max = run2->last_index;
      KheGetRun(rr2, ++i2, &run2, &fi2);
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheRunSolverCullImmovableComponents(KHE_RUN_SOLVER rs)              */
/*                                                                           */
/*  Delete any components that will not move.                                */
/*                                                                           */
/*****************************************************************************/

static void KheRunSolverCullImmovableComponents(KHE_RUN_SOLVER rs,
  KHE_RUN_RESOURCE rr1, KHE_RUN_RESOURCE rr2)
{
  KHE_RUN_COMPONENT rc;  int i;
  HaArrayForEach(rs->components, rc, i)
    if( !KheRunComponentMoveCheck(rc, rr2->resource, rr1->resource) )
    {
      /* rc does not move, so get rid of it */
      HaArrayDeleteAndShift(rs->components, i);
      KheRunComponentDelete(rc, rs);
      i--;
    }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheSolnIsNewBest(KHE_RUN_SOLVER rs)                                 */
/*                                                                           */
/*  Return true if the current solution is a new best.                       */
/*                                                                           */
/*****************************************************************************/

static bool KheSolnIsNewBest(KHE_RUN_SOLVER rs)
{
  return KheSolnCost(rs->soln) < KhePathSolnCost(KheMarkPath(rs->init_mark, 0))
    && (!rs->resource_invariant ||
      KheSolnMatchingDefectCount(rs->soln) <= rs->init_defect_count);
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheResourceShow(KHE_RESOURCE r)                                    */
/*                                                                           */
/*  Return the Id of r, or "@" if r is NULL.                                 */
/*                                                                           */
/*****************************************************************************/

static char *KheResourceShow(KHE_RESOURCE r)
{
  if( r == NULL )
    return "@";
  else if( KheResourceId(r) == NULL )
    return "?";
  else
    return KheResourceId(r);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheRunSolverTry(KHE_RUN_SOLVER rs, KHE_RUN_COMPONENT rc,            */
/*    int curr_index, KHE_RESOURCE part1_r, KHE_RESOURCE part2_r)            */
/*                                                                           */
/*  Try one assignment of part1_r to rc's part1, and part2_r to rc's         */
/*  part2, and recurse.                                                      */
/*                                                                           */
/*****************************************************************************/
static void KheRunSolverDoSolvePair(KHE_RUN_SOLVER rs, int curr_index);

static void KheRunSolverTry(KHE_RUN_SOLVER rs, KHE_RUN_COMPONENT rc,
  int curr_index, KHE_RESOURCE part1_r, KHE_RESOURCE part2_r)
{
  KHE_MARK mark;
  mark = KheMarkBegin(rs->soln);
  if( DEBUG5 && rs->debug_on )
    fprintf(stderr, "%*s  at %d trying (%s, %s):\n", curr_index * 2, "",
      curr_index, KheResourceShow(part1_r), KheResourceShow(part2_r));
  if( KheRunComponentMove(rc, part1_r, part2_r) )
    KheRunSolverDoSolvePair(rs, curr_index + 1);
  KheMarkEnd(mark, true);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheRunSolverDoSolvePair(KHE_RUN_SOLVER rs, int curr_index)          */
/*                                                                           */
/*  Solve from curr_index onwards.  When this function is called, part1 is   */
/*  assigned rs->resource1 and part2 is assigned rs->resource2.              */
/*                                                                           */
/*****************************************************************************/

static void KheRunSolverDoSolvePair(KHE_RUN_SOLVER rs, int curr_index)
{
  KHE_RUN_COMPONENT rc;
  if( DEBUG5 && rs->debug_on )
    fprintf(stderr, "%*s[ DoSolvePair(r1, r2, %d)\n", curr_index * 2, "",
      curr_index);
  if( rs->curr_assignments < rs->max_assignments )
  {
    if( curr_index >= HaArrayCount(rs->components) )
    {
      /* time to stop; if new best, save path */
      if( DEBUG5 && rs->debug_on )
	fprintf(stderr, "%*s  off end: %s\n", curr_index * 2, "",
	   KheSolnIsNewBest(rs) ? "new best" : "no luck");
      if( KheSolnIsNewBest(rs) )
	KheMarkAddBestPath(rs->init_mark, 1);
      rs->curr_assignments++;
    }
    else
    {
      /* try the initial assignment of the component at curr_index */
      rc = HaArray(rs->components, curr_index);
      if( DEBUG5 && rs->debug_on )
	fprintf(stderr, "%*s  unswapped:\n", curr_index * 2, "");
      KheRunSolverDoSolvePair(rs, curr_index + 1);

      /* try the opposite assignment of the component at curr_index */
      KheRunSolverTry(rs, rc, curr_index, rs->resource2, rs->resource1);

      /* possibly try unassignments where permitted */
      if( TRY_UNASSIGNMENTS )
      {
	if( rc->part1_needs_asst )
	{
	  if( rc->part2_needs_asst )
	  {
	    /* try (r2, r1) only */
	  }
	  else
	  {
	    /* try (r2, r1) and (r2, NULL) */
	    KheRunSolverTry(rs, rc, curr_index, rs->resource2, NULL);
	  }
	}
	else
	{
	  if( rc->part2_needs_asst )
	  {
	    /* try (r2, r1) and (NULL, r1) */
	    KheRunSolverTry(rs, rc, curr_index, NULL, rs->resource1);
	  }
	  else
	  {
	    /* try (r2, r1), (NULL, r1), (r2, NULL), and (NULL, NULL) */
	    KheRunSolverTry(rs, rc, curr_index, NULL, rs->resource1);
	    KheRunSolverTry(rs, rc, curr_index, rs->resource2, NULL);
	    KheRunSolverTry(rs, rc, curr_index, NULL, NULL);
	  }
	}
      }

      /* *** old version that did not try unassignments
      mark = KheMarkBegin(rs->soln);
      if( DEBUG5 && rs->debug_on )
	fprintf(stderr, "%*s  swapped:\n", curr_index * 2, "");
      if( KheRunComponentMove(rc, rs->resource2, rs->resource1) )
	KheRunSolverDoSolvePair(rs, curr_index + 1);
      KheMarkEnd(mark, true);
      *** */
    }
  }
  else
  {
    if( DEBUG5 && rs->debug_on )
      fprintf(stderr, "%*s  limit %d reached\n", curr_index * 2, "",
	rs->max_assignments);
  }
  if( DEBUG5 && rs->debug_on )
    fprintf(stderr, "%*s]\n", curr_index * 2, "");
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheRunSolverSolvePair(KHE_RUN_SOLVER rs, KHE_RUN_RESOURCE rr1,      */
/*    KHE_RUN_RESOURCE rr2)                                                  */
/*                                                                           */
/*  Solve (rr1, rr2) and return true if any assignments changed.  At         */
/*  the start, make sure rr1 and rr2 are up to date, and at the end,         */
/*  mark them out of date if any assignments changed.                        */
/*                                                                           */
/*****************************************************************************/

static bool KheRunSolverSolvePair(KHE_RUN_SOLVER rs, KHE_RUN_RESOURCE rr1,
  KHE_RUN_RESOURCE rr2, int first_index, int last_index)
{
  KHE_RUN_COMPONENT rc;  int i, count;  KHE_COST init_cost;
  if( DEBUG2 )
    fprintf(stderr, "[ KheRunSolverSolvePair(rs, %s, %s)\n",
      KheResourceId(rr1->resource), KheResourceId(rr2->resource));

  /* make sure rr1 and rr2 are different, and bring their runs up to date */
  HnAssert(rr1 != rr2, "KheRunSolverSolvePair internal error (rr1 == rr2)");
  KheRunResourceSetRuns(rr1, rs, first_index, last_index);
  KheRunResourceSetRuns(rr2, rs, first_index, last_index);
  if( DEBUG2 )
  {
    KheRunResourceDebug(rr1, 2, 2, stderr);
    KheRunResourceDebug(rr2, 2, 2, stderr);
  }

  /* find components in rr1+rr2 */
  KheRunSolverFindComponents(rs, rr1, rr2);
  if( DEBUG2 )
    HaArrayForEach(rs->components, rc, i)
      KheRunComponentDebug(rc, 2, 2, stderr);
  count = HaArrayCount(rs->components);

  /* cull immovable components */
  KheRunSolverCullImmovableComponents(rs, rr1, rr2);
  if( DEBUG2 && HaArrayCount(rs->components) < count )
  {
    fprintf(stderr, "  after culling %d immovable components:\n",
      count - HaArrayCount(rs->components));
    HaArrayForEach(rs->components, rc, i)
      KheRunComponentDebug(rc, 2, 2, stderr);
  }

  /* perform the actual solve, ending by redoing the best path */
  init_cost = KheSolnCost(rs->soln);
  rs->resource1 = rr1->resource;
  rs->resource2 = rr2->resource;
  rs->curr_assignments = 0;
  rs->init_mark = KheMarkBegin(rs->soln);
  if( rs->resource_invariant )
    rs->init_defect_count = KheSolnMatchingDefectCount(rs->soln);
  KheMarkAddBestPath(rs->init_mark, 1);
  rs->debug_on = DEBUG5 &&
    strcmp(KheResourceId(rr1->resource), DEBUG5_ID1) == 0 &&
    strcmp(KheResourceId(rr2->resource), DEBUG5_ID2) == 0;
  KheRunSolverDoSolvePair(rs, 0);
  KhePathRedo(KheMarkPath(rs->init_mark, 0));
  KheMarkEnd(rs->init_mark, false);

  if( KheSolnCost(rs->soln) < init_cost )
  {
    /* success; assignments changed, so mark rr1 and rr2 as out of date */
    /* we also reset their runs, but that is basically for debugging    */
    /* efficiency is not a problem, this happens very rarely            */
    rr1->first_index = rr1->last_index = -1;
    rr2->first_index = rr2->last_index = -1;
    KheRunResourceSetRuns(rr1, rs, first_index, last_index);
    KheRunResourceSetRuns(rr2, rs, first_index, last_index);
    if( DEBUG2 )
    {
      KheRunResourceDebug(rr1, 2, 2, stderr);
      KheRunResourceDebug(rr2, 2, 2, stderr);
      fprintf(stderr, "] KheRunSolverSolvePair returning true (%.5f -> %.5f)\n",
       KheCostShow(init_cost), KheCostShow(KheSolnCost(rs->soln)));
    }
    if( DEBUG6 )
      fprintf(stderr, "  KheRunSolverSolvePair(rs, %s, %s, %d, %d) returning"
	" true (%.5f -> %.5f)\n", KheResourceId(rr1->resource),
	KheResourceId(rr2->resource), first_index, last_index,
	KheCostShow(init_cost), KheCostShow(KheSolnCost(rs->soln)));
    return true;
  }
  else
  {
    /* failure */
    if( DEBUG2 )
      fprintf(stderr, "] KheRunSolverSolvePair returning false\n");
    return false;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "public functions"                                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_RUN_SOLVER KheRunSolverMake(KHE_SOLN soln, KHE_FRAME frame,          */
/*    bool resource_invariant, int max_assignments)                          */
/*                                                                           */
/*  Make a new run solver with these attributes, using memory from an        */
/*  arena taken from soln.                                                   */
/*                                                                           */
/*****************************************************************************/

static KHE_RUN_SOLVER KheRunSolverMake(KHE_SOLN soln, KHE_OPTIONS options,
  bool resource_invariant, int max_assignments)
{
  KHE_RUN_SOLVER res;  HA_ARENA a;
  a = KheSolnArenaBegin(soln, false);
  HaMake(res, a);

  /* fixed attributes of the solve object */
  res->arena = a;
  res->soln = soln;
  res->task_finder = KheTaskFinderMake(soln, options, a);
  res->resource_invariant = resource_invariant;
  res->max_assignments = max_assignments;

  /* free lists */
  HaArrayInit(res->resources, a);
  HaArrayInit(res->free_resources, a);
  HaArrayInit(res->free_runs, a);
  HaArrayInit(res->free_run_components, a);

  /* defined during individual solves (really undefined here) */
  HaArrayInit(res->components, a);
  res->curr_assignments = 0;
  res->init_mark = NULL;
  res->init_defect_count = -1;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheRunSolverDelete(KHE_RUN_SOLVER rs)                               */
/*                                                                           */
/*  Delete rs.                                                               */
/*                                                                           */
/*****************************************************************************/

static void KheRunSolverDelete(KHE_RUN_SOLVER rs)
{
  KheSolnArenaEnd(rs->soln, rs->arena);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheRunSolverAddResource(KHE_RUN_SOLVER rs, KHE_RESOURCE r)          */
/*                                                                           */
/*  Add r to rs.                                                             */
/*                                                                           */
/*****************************************************************************/

static void KheRunSolverAddResource(KHE_RUN_SOLVER rs, KHE_RESOURCE r)
{
  KHE_RUN_RESOURCE rr;
  rr = KheRunResourceMake(rs, r);
  HaArrayAddLast(rs->resources, rr);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheRunSolverClearResources(KHE_RUN_SOLVER rs)                       */
/*                                                                           */
/*  Clear all the resources from rs.                                         */
/*                                                                           */
/*****************************************************************************/

static void KheRunSolverClearResources(KHE_RUN_SOLVER rs)
{
  while( HaArrayCount(rs->resources) > 0 )
    KheRunResourceDelete(HaArrayLastAndDelete(rs->resources), rs);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheRunSolverSolveAllPairs(KHE_RUN_SOLVER rs, int fi, int li)        */
/*                                                                           */
/*  Solve all pairs of resources and return true if any succeeded.           */
/*                                                                           */
/*****************************************************************************/

static bool KheRunSolverSolveAllPairs(KHE_RUN_SOLVER rs, int fi, int li)
{
  KHE_RUN_RESOURCE rr1, rr2;  int i, j;  bool res;
  res = false;
  for( i = 0;  i < HaArrayCount(rs->resources);  i++ )
  {
    rr1 = HaArray(rs->resources, i);
    for( j = i + 1;  j < HaArrayCount(rs->resources);  j++ )
    {
      rr2 = HaArray(rs->resources, j);
      if( KheRunSolverSolvePair(rs, rr1, rr2, fi, li) )
	res = true;
    }
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheRunSolverSolveDefectivePairs(KHE_RUN_SOLVER rs)                  */
/*                                                                           */
/*  Solve for all pairs of resources such that at least one is defective.    */
/*                                                                           */
/*****************************************************************************/

static bool KheRunSolverSolveDefectivePairs(KHE_RUN_SOLVER rs, int fi, int li)
{
  KHE_RUN_RESOURCE rr1, rr2;  int i, j;  bool res;
  res = false;
  for( i = 0;  i < HaArrayCount(rs->resources);  i++ )
  {
    rr1 = HaArray(rs->resources, i);
    for( j = i + 1;  j < HaArrayCount(rs->resources);  j++ )
    {
      rr2 = HaArray(rs->resources, j);
      if( rr1->defective || rr2->defective )
      {
	if( KheRunSolverSolvePair(rs, rr1, rr2, fi, li) )
	{
	  res = true;
	  KheRunResourceSetDefective(rr1, rs);  /* defective may have changed */
	  KheRunResourceSetDefective(rr2, rs);  /* defective may have changed */
	}
      }
    }
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheRunSolverSolveAdjacentPairs(KHE_RUN_SOLVER rs)                   */
/*                                                                           */
/*  Solve adjacent pairs of resources and return true if any succeeded.      */
/*                                                                           */
/*****************************************************************************/

static bool KheRunSolverSolveAdjacentPairs(KHE_RUN_SOLVER rs, int fi, int li)
{
  KHE_RUN_RESOURCE rr1, rr2;  int i;  bool res;
  res = false;
  for( i = 0;  i < HaArrayCount(rs->resources) - 1;  i++ )
  {
    rr1 = HaArray(rs->resources, i);
    rr2 = HaArray(rs->resources, i + 1);
    if( KheRunSolverSolvePair(rs, rr1, rr2, fi, li) )
      res = true;
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheResourcePairRunReassign(KHE_SOLN soln, KHE_RESOURCE r1,          */
/*    KHE_RESOURCE r2, KHE_FRAME frame, int fi, int li,                      */
/*    bool resource_invariant, int max_assignments)                          */
/*                                                                           */
/*  Resource pair run repair.                                                */
/*                                                                           */
/*****************************************************************************/

bool KheResourcePairRunReassign(KHE_SOLN soln, KHE_OPTIONS options,
  KHE_RESOURCE r1, KHE_RESOURCE r2, int fi, int li,
  bool resource_invariant, int max_assignments)
{
  KHE_RUN_SOLVER rs;  bool res;  KHE_COST init_cost;
  if( DEBUG5 )
    fprintf(stderr, "[ KheResourcePairRunReassign(%s, %s, %d-%d, %s, %d)\n",
      KheResourceId(r1), KheResourceId(r2), fi, li,
      bool_show(resource_invariant), max_assignments);
  HnAssert(r1 != NULL, "KheResourcePairRunReassign: r1 is NULL");
  HnAssert(r2 != NULL, "KheResourcePairRunReassign: r2 is NULL");
  HnAssert(r1 != r2, "KheResourcePairRunReassign: r1 == r2");
  init_cost = KheSolnCost(soln);
  rs = KheRunSolverMake(soln, options, resource_invariant, max_assignments);
  KheRunSolverAddResource(rs, r1);
  KheRunSolverAddResource(rs, r2);
  res = KheRunSolverSolveAdjacentPairs(rs, fi, li);
  HnAssert(res == (KheSolnCost(soln) < init_cost),
    "KheResourcePairRunReassign internal error");
  if( DEBUG5 )
  {
    fprintf(stderr, "] KheResourcePairRunReassign returning ");
    if( res )
      fprintf(stderr, "true (%.5f -> %.5f)\n",
	KheCostShow(init_cost), KheCostShow(KheSolnCost(soln)));
    else
      fprintf(stderr, "false\n");
  }
  KheRunSolverDelete(rs);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheRunSolverSolveForResources(KHE_RUN_SOLVER rs,                    */
/*    RPAIR_SELECT_TYPE rpair_select, int rpair_parts, int rpair_start,      */
/*    int rpair_increment)                                                   */
/*                                                                           */
/*  Solve with these parameters.                                             */
/*                                                                           */
/*****************************************************************************/

static bool KheRunSolverSolveForResources(KHE_RUN_SOLVER rs,
  RPAIR_SELECT_TYPE rpair_select, int rpair_parts, int rpair_start,
  int rpair_increment)
{
  bool res;  int fi, li, last_index;
  res = false;
  last_index = KheTaskFinderLastIndex(rs->task_finder);
  fi = rpair_start, li = fi + rpair_parts - 1;
  for( ;  li <= last_index;  fi += rpair_increment, li += rpair_increment )
  {
    switch( rpair_select )
    {
      case RPAIR_SELECT_DEFECTIVE:

	if( KheRunSolverSolveDefectivePairs(rs, fi, li) )
	  res = true;
	break;

      case RPAIR_SELECT_ADJACENT:

	if( KheRunSolverSolveAdjacentPairs(rs, fi, li) )
	  res = true;
	break;

      case RPAIR_SELECT_ALL:

	if( KheRunSolverSolveAllPairs(rs, fi, li) )
	  res = true;
	break;

      case RPAIR_SELECT_NONE:
      default:

	HnAbort("KheRunSolverSolveForResources internal error");
	break;
    }
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheResourcePairRunRepair(KHE_SOLN soln, KHE_OPTIONS options)        */
/*                                                                           */
/*  Call KheResourcePairRunReassign repeatedly, depending on options.        */
/*                                                                           */
/*****************************************************************************/

bool KheResourcePairRunRepair(KHE_SOLN soln, KHE_OPTIONS options)
{
  KHE_RUN_SOLVER rs;  bool resource_invariant, res, rpair_off;
  KHE_INSTANCE ins;  KHE_RESOURCE_TYPE rt;  KHE_RESOURCE r;
  int rpair_parts, rpair_start, rpair_increment, rpair_max, i, j;
  KHE_RUN_RESOURCE rr;  RPAIR_SELECT_TYPE rpair_select;  KHE_COST init_cost;

  /* quit now if rs_pair_off, or rs_pair_select is "none" */
  rpair_off = KheOptionsGetBool(options, "rs_rpair_off", false);
  rpair_select = KheSelectTypeGet(options);
  if( rpair_off || rpair_select == RPAIR_SELECT_NONE )
    return false;

  ins = KheSolnInstance(soln);
  if( DEBUG1 )
    fprintf(stderr, "[ KheResourcePairRunRepair(soln of %s, options)\n",
      KheInstanceId(ins));

  /* quit now if the time limit has been reached */
  if( KheOptionsTimeLimitReached(options) )
  {
    if( DEBUG1 )
      fprintf(stderr, "] KheResourcePairRunRepair ret. false (time limit)\n");
    return false;
  }

  /* get the other options */
  resource_invariant = KheOptionsGetBool(options, "rs_invariant", false);
  rpair_parts = KheOptionsGetInt(options, "rs_rpair_parts", 28);
  rpair_start = KheOptionsGetInt(options, "rs_rpair_start", 0);
  rpair_increment = KheOptionsGetInt(options, "rs_rpair_increment",rpair_parts);
  rpair_max = KheOptionsGetInt(options, "rs_rpair_max", 1000000);

  /* make a solver and solve for each resource type */
  res = false;
  init_cost = KheSolnCost(soln);
  rs = KheRunSolverMake(soln, options, resource_invariant, rpair_max);
  for( i = 0;  i < KheInstanceResourceTypeCount(ins);  i++ )
  {
    rt = KheInstanceResourceType(ins, i);

    /* clear the resources then add every resource of type rt */
    KheRunSolverClearResources(rs);
    for( j = 0;  j < KheResourceTypeResourceCount(rt);  j++ )
    {
      r = KheResourceTypeResource(rt, j);
      KheRunSolverAddResource(rs, r);
    }

    /* if "defective", we need to work out which resources are defective */
    if( rpair_select == RPAIR_SELECT_DEFECTIVE )
      HaArrayForEach(rs->resources, rr, j)
	KheRunResourceSetDefective(rr, rs);

    /* now do the solve */
    if( KheRunSolverSolveForResources(rs, rpair_select, rpair_parts,
	rpair_start, rpair_increment) )
      res = true;
  }

  /* all done, debug result and return */
  if( res )
  {
    if( DEBUG1 )
      fprintf(stderr, "] KheResourcePairRunRepair returning true (%.5f -> "
	"%.5f)\n", KheCostShow(init_cost), KheCostShow(KheSolnCost(soln)));
    return true;
  }
  else
  {
    if( DEBUG1 )
      fprintf(stderr, "] KheResourcePairRunRepair returning false\n");
    return false;
  }
  return res;
}
