
/*****************************************************************************/
/*                                                                           */
/*  THE KHE HIGH SCHOOL TIMETABLING ENGINE                                   */
/*  COPYRIGHT (C) 2010 Jeffrey H. Kingston                                   */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 3, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         khe_sr_resource_pair.c                                     */
/*  DESCRIPTION:  Resource pair resource repair                              */
/*                                                                           */
/*****************************************************************************/
#include "khe_solvers.h"
#include <limits.h>

#define bool_show(val)  ((val) ? "true" : "false")

#define DEBUG1 0
#define DEBUG2 0
#define DEBUG3 0
#define DEBUG4 0
#define DEBUG5 1		/* shows which swaps worked */
#define DEBUG6 1
#define DEBUG7 1
#define DEBUG8 0


/*****************************************************************************/
/*                                                                           */
/*  KHE_PAIR_SOLVER - a solver for simple pairs and triples                  */
/*                                                                           */
/*****************************************************************************/

typedef HA_ARRAY(KHE_TASK_SET) ARRAY_KHE_TASK_SET;

typedef struct khe_pair_solver_rec {
  KHE_SOLN			soln;
  /* ARRAY_KHE_TASK_SET		free_task_sets; */
  KHE_FRAME			frame;
  int				first_index;
  int				last_index;
  /* KHE_TIME_PARTITION		time_partition; */
  /* int			first_part_index; */
  /* int			last_part_index; */
  bool				resource_invariant;
  int				max_assignments;
  int				curr_assignments;
  KHE_MARK			init_mark;
  int				init_defect_count;
} *KHE_PAIR_SOLVER;


/*****************************************************************************/
/*                                                                           */
/*  void KhePairSolverInit(KHE_PAIR_SOLVER ps, KHE_SOLN soln,                */
/*    KHE_TIME_PARTITION tp, int first_part_index, int last_part_index,      */
/*    bool resource_invariant, int max_assignments)                          */
/*                                                                           */
/*  Initialize ps, not including memory allocation.                          */
/*                                                                           */
/*****************************************************************************/

static void KhePairSolverInit(KHE_PAIR_SOLVER ps, KHE_SOLN soln,
  KHE_FRAME frame, int fi, int li,
  /* KHE_TIME_PARTITION tp, int first_part_index,
  int last_part_index, */ bool resource_invariant, int max_assignments)
{
  ps->soln = soln;
  /* MArrayInit(ps->free_task_sets); */
  ps->frame = frame;
  ps->first_index = fi;
  ps->last_index = li;
  /* ps->time_partition = tp; */
  /* ps->first_part_index = first_part_index; */
  /* ps->last_part_index = last_part_index; */
  ps->resource_invariant = resource_invariant;
  ps->max_assignments = max_assignments;
  ps->curr_assignments = 0;	/* set properly later */
  ps->init_mark = NULL;		/* set properly later */
  ps->init_defect_count = -1;	/* set properly later */
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_TASK_SET KhePairSolverGetTaskSet(KHE_PAIR_SOLVER ps)                 */
/*                                                                           */
/*  Obtain an empty task set from ps.                                        */
/*                                                                           */
/*****************************************************************************/

/* *** obsolete, soln has a free list now
static KHE_TASK_SET KhePairSolverGetTaskSet(KHE_PAIR_SOLVER ps)
{
  KHE_TASK_SET res;
  if( HaArrayCount(ps->free_task_sets) > 0 )
  {
    res = MArrayRemoveLast(ps->free_task_sets);
    KheTaskSetClear(res);
  }
  else
    res = KheTaskSetMake(ps->soln);
  return res;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KhePairSolverPutTaskSet(KHE_PAIR_SOLVER ps, KHE_TASK_SET ts)        */
/*                                                                           */
/*  Return a task set to ps for re-use later.                                */
/*                                                                           */
/*****************************************************************************/

/* *** obsolete, soln has a free list now
static void KhePairSolverPutTaskSet(KHE_PAIR_SOLVER ps, KHE_TASK_SET ts)
{
  HaArrayAddLast(ps->free_task_sets, ts);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KhePairSolverFree(KHE_PAIR_SOLVER ps)                               */
/*                                                                           */
/*  Free the memory consumed by ps.  This does not include the ps record     */
/*  itself, because that will have been allocated on the heap.  It includes  */
/*  the free task sets and the array holding them.                           */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KhePairSolverFree(KHE_PAIR_SOLVER ps)
{
  ** ***
  KHE_TASK_SET ts;  int i;
  MArrayForEach(ps->free_task_sets, &ts, &i)
    KheTaskSetDelete(ts);
  MArrayFree(ps->free_task_sets);
  *** **
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheTryPermutation2(KHE_PAIR_SOLVER ps, int curr_index,              */
/*    KHE_RESOURCE r1, KHE_TASK_SET ts1, KHE_RESOURCE r2, KHE_TASK_SET ts2)  */
/*                                                                           */
/*  Helper function which permutes the assignments of two resources and      */
/*  recurses.  Except that it doesn't modify preassigned tasks.              */
/*                                                                           */
/*****************************************************************************/

static void KheDoSimpleAssign2(KHE_PAIR_SOLVER ps, int curr_index,
  KHE_RESOURCE r1, KHE_RESOURCE r2);

static void KheTryPermutation2(KHE_PAIR_SOLVER ps, int curr_index,
  KHE_RESOURCE r1, KHE_TASK_SET ts1, KHE_RESOURCE r2, KHE_TASK_SET ts2)
{
  int i;  bool ok;  KHE_MARK mark;
  mark = KheMarkBegin(ps->soln);
  for( i = 0;  i < KheTaskSetTaskCount(ts1);  i++ )
    KheTaskUnAssign(KheTaskSetTask(ts1, i));
  for( i = 0;  i < KheTaskSetTaskCount(ts2);  i++ )
    KheTaskUnAssign(KheTaskSetTask(ts2, i));
  ok = true;
  for( i = 0;  ok && i < KheTaskSetTaskCount(ts1);  i++ )
  {
    ok = ok && KheTaskAssignResource(KheTaskSetTask(ts1, i), r1);
    ps->curr_assignments += 1;
  }
  for( i = 0;  ok && i < KheTaskSetTaskCount(ts2);  i++ )
  {
    ok = ok && KheTaskAssignResource(KheTaskSetTask(ts2, i), r2);
    ps->curr_assignments += 1;
  }
  if( ok )
  {
    if( DEBUG2 )
    {
      fprintf(stderr, "%*s  swapping to %s",
	(curr_index /* - ps->first_part_index */ + 1) * 2, "",
	KheResourceId(r1));
      KheTaskSetDebug(ts1, 1, -1, stderr);
      fprintf(stderr, " and %s", KheResourceId(r2));
      KheTaskSetDebug(ts2, 1, -1, stderr);
      fprintf(stderr, ":\n");
    }
    KheDoSimpleAssign2(ps, curr_index + 1, r1, r2);
  }
  KheMarkEnd(mark, true);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheTryPermutation3(KHE_SOLN soln,                                   */
/*    KHE_RESOURCE r1, KHE_TASK_SET ts1,                                     */
/*    KHE_RESOURCE r2, KHE_TASK_SET ts2,                                     */
/*    KHE_RESOURCE r3, KHE_TASK_SET ts3,                                     */
/*    KHE_TIME_PARTITION tp, int first_part_index, int last_part_index,      */
/*    bool resource_invariant, int max_assignments,                          */
/*    int curr_index, int *curr_assignments, KHE_MARK init_mark)             */
/*                                                                           */
/*  Helper function which permutes the assignments of three resources and    */
/*  recurses.                                                                */
/*                                                                           */
/*****************************************************************************/

static void KheDoSimpleAssign3(KHE_PAIR_SOLVER ps, int curr_index,
  KHE_RESOURCE r1, KHE_RESOURCE r2, KHE_RESOURCE r3);

static void KheTryPermutation3(KHE_PAIR_SOLVER ps, int curr_index,
  KHE_RESOURCE r1, KHE_TASK_SET ts1,
  KHE_RESOURCE r2, KHE_TASK_SET ts2,
  KHE_RESOURCE r3, KHE_TASK_SET ts3)
{
  int i;  bool ok;  KHE_MARK mark;
  mark = KheMarkBegin(ps->soln);
  for( i = 0;  i < KheTaskSetTaskCount(ts1);  i++ )
    KheTaskUnAssign(KheTaskSetTask(ts1, i));
  for( i = 0;  i < KheTaskSetTaskCount(ts2);  i++ )
    KheTaskUnAssign(KheTaskSetTask(ts2, i));
  for( i = 0;  i < KheTaskSetTaskCount(ts3);  i++ )
    KheTaskUnAssign(KheTaskSetTask(ts3, i));
  ok = true;
  for( i = 0;  ok && i < KheTaskSetTaskCount(ts1);  i++ )
  {
    ok = ok && KheTaskAssignResource(KheTaskSetTask(ts1, i), r1);
    ps->curr_assignments += 1;
  }
  for( i = 0;  ok && i < KheTaskSetTaskCount(ts2);  i++ )
  {
    ok = ok && KheTaskAssignResource(KheTaskSetTask(ts2, i), r2);
    ps->curr_assignments += 1;
  }
  for( i = 0;  ok && i < KheTaskSetTaskCount(ts3);  i++ )
  {
    ok = ok && KheTaskAssignResource(KheTaskSetTask(ts3, i), r3);
    ps->curr_assignments += 1;
  }
  if( ok )
  {
    if( DEBUG2 )
    {
      fprintf(stderr, "%*s  swapping to %s",
	(curr_index /* - ps->first_part_index */ + 1) * 2, "",
	KheResourceId(r1));
      KheTaskSetDebug(ts1, 1, -1, stderr);
      fprintf(stderr, " and %s", KheResourceId(r2));
      KheTaskSetDebug(ts2, 1, -1, stderr);
      fprintf(stderr, " and %s", KheResourceId(r3));
      KheTaskSetDebug(ts3, 1, -1, stderr);
      fprintf(stderr, ":\n");
    }
    KheDoSimpleAssign3(ps, curr_index + 1, r1, r2, r3);
  }
  KheMarkEnd(mark, true);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDoSimpleAssign2(KHE_PAIR_SOLVER ps, int curr_index,              */
/*    KHE_RESOURCE r1, KHE_RESOURCE r2)                                      */
/*                                                                           */
/*  Do the actual work.                                                      */
/*                                                                           */
/*****************************************************************************/

static void KheDoSimpleAssign2(KHE_PAIR_SOLVER ps, int curr_index,
  KHE_RESOURCE r1, KHE_RESOURCE r2)
{
  KHE_RESOURCE_TIMETABLE_MONITOR rtm1, rtm2;  KHE_TIME_GROUP tg;
  KHE_TASK_SET ts1, ts2;
  if( DEBUG2 )
    fprintf(stderr, "%*s[ SimpleAssign2(frame, %d)\n",
      (curr_index + 1) * 2, "", curr_index);
    /* ***
    fprintf(stderr, "%*s[ SimpleAssign2(%d..%d, %d)\n",
      (curr_index - ps->first_part_index + 1) * 2, "",
      ps->first_part_index, ps->last_part_index, curr_index);
    *** */
  /* ***
  if( curr_index > KheFrameTimeGroupCount(ps->frame) ||
      ps->curr_assignments > ps->max_assignments )
  *** */
  if( curr_index > ps->last_index ||
      ps->curr_assignments > ps->max_assignments )
  {
    if( KheSolnCost(ps->soln) < KhePathSolnCost(KheMarkPath(ps->init_mark, 0))
	&& (!ps->resource_invariant ||
	 KheSolnMatchingDefectCount(ps->soln) <= ps->init_defect_count) )
    {
      /* new best path */
      KheMarkAddBestPath(ps->init_mark, 1);
    }
  }
  else
  {
    /* try existing assignments */
    KheDoSimpleAssign2(ps, curr_index + 1, r1, r2);

    /* get the two task sets for this part */
    tg = KheFrameTimeGroup(ps->frame, curr_index);
    /* tg = KheTimePartitionPart(ps->time_partition, curr_index); */
    ts1 = KheTaskSetMake(ps->soln);
    /* ts1 = KhePairSolverGetTaskSet(ps); */
    rtm1 = KheResourceTimetableMonitor(ps->soln, r1);
    KheResourceTimetableMonitorAddProperRootTasks(rtm1, tg, false, ts1);
    ts2 = KheTaskSetMake(ps->soln);
    /* ts2 = KhePairSolverGetTaskSet(ps); */
    rtm2 = KheResourceTimetableMonitor(ps->soln, r2);
    KheResourceTimetableMonitorAddProperRootTasks(rtm2, tg, false, ts2);

    /* try opposite assignments, if any */
    if( KheTaskSetTaskCount(ts1) + KheTaskSetTaskCount(ts2) > 0 )
      KheTryPermutation2(ps, curr_index, r1, ts1, r2, ts2);

    /* free the task sets */
    KheTaskSetDelete(ts1);
    KheTaskSetDelete(ts2);
    /* ***
    KhePairSolverPutTaskSet(ps, ts1);
    KhePairSolverPutTaskSet(ps, ts2);
    *** */
  }
  if( DEBUG2 )
    fprintf(stderr, "%*s] SimpleAssign2\n",
      (curr_index /* - ps->first_part_index */ + 1) * 2, "");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDoSimpleAssign3(KHE_SOLN soln, KHE_RESOURCE r1,                  */
/*    KHE_RESOURCE r2, KHE_RESOURCE r3, KHE_TIME_PARTITION tp,               */
/*    int first_part_index, int last_part_index, bool resource_invariant,    */
/*    int max_assignments, int curr_index, int *curr_assignments,            */
/*    KHE_MARK init_mark, int init_defect_count)                             */
/*                                                                           */
/*  Do the actual work.                                                      */
/*                                                                           */
/*****************************************************************************/

static void KheDoSimpleAssign3(KHE_PAIR_SOLVER ps, int curr_index,
  KHE_RESOURCE r1, KHE_RESOURCE r2, KHE_RESOURCE r3)
{
  KHE_RESOURCE_TIMETABLE_MONITOR rtm1, rtm2, rtm3;  KHE_TIME_GROUP tg;
  KHE_TASK_SET ts1, ts2, ts3;
  if( DEBUG2 )
    fprintf(stderr, "%*s[ SimpleAssign3(frame, %d)\n",
      (curr_index + 1) * 2, "", curr_index);
    /* ***
    fprintf(stderr, "%*s[ SimpleAssign3(%d..%d, %d)\n",
      (curr_index - ps->first_part_index + 1) * 2, "",
      ps->first_part_index, ps->last_part_index, curr_index);
    *** */
  /* ***
  if( curr_index > KheFrameTimeGroupCount(ps->frame) ||
      ps->curr_assignments > ps->max_assignments )
  *** */
  if( curr_index > ps->last_index ||
      ps->curr_assignments > ps->max_assignments )
  {
    if( KheSolnCost(ps->soln) < KhePathSolnCost(KheMarkPath(ps->init_mark, 0))
	&& (!ps->resource_invariant ||
	 KheSolnMatchingDefectCount(ps->soln) <= ps->init_defect_count) )
    {
      /* new best path */
      KheMarkAddBestPath(ps->init_mark, 1);
    }
  }
  else
  {
    /* try existing assignments */
    KheDoSimpleAssign3(ps, curr_index + 1, r1, r2, r3);

    /* get the three task sets for this part */
    tg = KheFrameTimeGroup(ps->frame, curr_index);
    /* tg = KheTimePartitionPart(ps->time_partition, curr_index); */
    ts1 = KheTaskSetMake(ps->soln);
    /* ts1 = KhePairSolverGetTaskSet(ps); */
    rtm1 = KheResourceTimetableMonitor(ps->soln, r1);
    KheResourceTimetableMonitorAddProperRootTasks(rtm1, tg, false, ts1);
    ts2 = KheTaskSetMake(ps->soln);
    /* ts2 = KhePairSolverGetTaskSet(ps); */
    rtm2 = KheResourceTimetableMonitor(ps->soln, r2);
    KheResourceTimetableMonitorAddProperRootTasks(rtm2, tg, false, ts2);
    ts3 = KheTaskSetMake(ps->soln);
    /* ts3 = KhePairSolverGetTaskSet(ps); */
    rtm3 = KheResourceTimetableMonitor(ps->soln, r3);
    KheResourceTimetableMonitorAddProperRootTasks(rtm3, tg, false, ts3);

    /* try the other five permutations, if needed */
    if( KheTaskSetTaskCount(ts1) + KheTaskSetTaskCount(ts2) 
	+ KheTaskSetTaskCount(ts3) > 0 )
    {
      KheTryPermutation3(ps, curr_index, r1, ts1, r2, ts3, r3, ts2);
      KheTryPermutation3(ps, curr_index, r1, ts2, r2, ts1, r3, ts3);
      KheTryPermutation3(ps, curr_index, r1, ts2, r2, ts3, r3, ts1);
      KheTryPermutation3(ps, curr_index, r1, ts3, r2, ts1, r3, ts2);
      KheTryPermutation3(ps, curr_index, r1, ts3, r2, ts2, r3, ts1);
    }

    /* free the task sets */
    KheTaskSetDelete(ts1);
    KheTaskSetDelete(ts2);
    KheTaskSetDelete(ts3);
    /* ***
    KhePairSolverPutTaskSet(ps, ts1);
    KhePairSolverPutTaskSet(ps, ts2);
    KhePairSolverPutTaskSet(ps, ts3);
    *** */
  }
  if( DEBUG2 )
    fprintf(stderr, "%*s] SimpleAssign3\n",
      (curr_index /* - ps->first_part_index */ + 1) * 2, "");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheResourcePairDoSimpleAssign(KHE_PAIR_SOLVER ps,                   */
/*    KHE_RESOURCE r1, KHE_RESOURCE r2)                                      */
/*                                                                           */
/*  Carry out the entire resource pair simple assign for r1 and r2.          */
/*                                                                           */
/*****************************************************************************/

static void KheResourcePairDoSimpleAssign(KHE_PAIR_SOLVER ps,
  KHE_RESOURCE r1, KHE_RESOURCE r2)
{
  ps->curr_assignments = 0;
  ps->init_mark = KheMarkBegin(ps->soln);
  ps->init_defect_count =
    (ps->resource_invariant ? KheSolnMatchingDefectCount(ps->soln) : INT_MAX);
  KheMarkAddBestPath(ps->init_mark, 1);
  KheDoSimpleAssign2(ps, ps->first_index, r1, r2);
  /* KheDoSimpleAssign2(ps, ps->first_part_index, r1, r2); */
  KhePathRedo(KheMarkPath(ps->init_mark, 0));
  KheMarkEnd(ps->init_mark, false);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheResourceTripleDoSimpleAssign(KHE_PAIR_SOLVER ps,                 */
/*    KHE_RESOURCE r1, KHE_RESOURCE r2, KHE_RESOURCE r3)                     */
/*                                                                           */
/*  Carry out the entire resource triple simple assign for r1, r2, and r3.   */
/*                                                                           */
/*****************************************************************************/

static void KheResourceTripleDoSimpleAssign(KHE_PAIR_SOLVER ps,
  KHE_RESOURCE r1, KHE_RESOURCE r2, KHE_RESOURCE r3)
{
  ps->curr_assignments = 0;
  ps->init_mark = KheMarkBegin(ps->soln);
  ps->init_defect_count =
    (ps->resource_invariant ? KheSolnMatchingDefectCount(ps->soln) : INT_MAX);
  KheMarkAddBestPath(ps->init_mark, 1);
  KheDoSimpleAssign3(ps, ps->first_index, r1, r2, r3);
  /* KheDoSimpleAssign3(ps, ps->first_part_index, r1, r2, r3); */
  KhePathRedo(KheMarkPath(ps->init_mark, 0));
  KheMarkEnd(ps->init_mark, false);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheResourcePairSimpleReassign(KHE_SOLN soln, KHE_RESOURCE r1,       */
/*    KHE_RESOURCE r2, KHE_FRAME frame, bool resource_invariant,             */
/*    int max_assignments)                                                   */
/*                                                                           */
/*  Simple resource pair repair.                                             */
/*                                                                           */
/*****************************************************************************/

bool KheResourcePairSimpleReassign(KHE_SOLN soln, KHE_RESOURCE r1,
  KHE_RESOURCE r2, KHE_FRAME frame, int fi, int li,
  bool resource_invariant, int max_assignments)
{
  KHE_COST init_cost;  struct khe_pair_solver_rec psr;
  if( DEBUG1 )
    fprintf(stderr, "[ KhePairSimpleReassign(%s, %s, frame, %s, %d)\n",
      KheResourceId(r1), KheResourceId(r2), bool_show(resource_invariant),
      max_assignments);
  KhePairSolverInit(&psr, soln, frame, fi, li, resource_invariant,
    max_assignments);
  init_cost = KheSolnCost(soln);
  KheResourcePairDoSimpleAssign(&psr, r1, r2);
  /* KhePairSolverFree(&psr); */
  if( DEBUG1 )
  {
    if( KheSolnCost(soln) < init_cost )
      fprintf(stderr, "] KhePairSimpleReassign returning true (%.5f -> %.5f)\n",
	KheCostShow(init_cost), KheCostShow(KheSolnCost(soln)));
    else
      fprintf(stderr, "] KhePairSimpleReassign returning false\n");
  }
  return KheSolnCost(soln) < init_cost;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheResourceTripleSimpleReassign(KHE_SOLN soln, KHE_RESOURCE r1,     */
/*    KHE_RESOURCE r2, KHE_RESOURCE r3, KHE_FRAME frame,                     */
/*    bool resource_invariant, int max_assignments)                          */
/*                                                                           */
/*  Simple resource triple repair.                                           */
/*                                                                           */
/*****************************************************************************/

bool KheResourceTripleSimpleReassign(KHE_SOLN soln, KHE_RESOURCE r1,
  KHE_RESOURCE r2, KHE_RESOURCE r3, KHE_FRAME frame, int fi, int li,
  bool resource_invariant, int max_assignments)
{
  KHE_COST init_cost;  struct khe_pair_solver_rec psr;
  if( DEBUG1 )
    fprintf(stderr,
      "[ KheTripleSimpleReassign(%s, %s, %s, tp, %s, %d)\n",
      KheResourceId(r1), KheResourceId(r2), KheResourceId(r3),
      bool_show(resource_invariant), max_assignments);
  KhePairSolverInit(&psr, soln, frame, fi, li, resource_invariant,
    max_assignments);
  init_cost = KheSolnCost(soln);
  KheResourceTripleDoSimpleAssign(&psr, r1, r2, r3);
  /* KhePairSolverFree(&psr); */
  if( DEBUG1 )
  {
    if( KheSolnCost(soln) < init_cost )
      fprintf(stderr,
	"] KheTripleSimpleReassign returning true (%.5f -> %.5f)\n",
	KheCostShow(init_cost), KheCostShow(KheSolnCost(soln)));
    else
      fprintf(stderr, "] KheTripleSimpleReassign returning false\n");
  }
  return KheSolnCost(soln) < init_cost;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheResourcePairDoSimpleRepair(KHE_SOLN soln, KHE_RESOURCE r1,       */
/*    KHE_RESOURCE r2, KHE_FRAME frame, int pair_parts, int pair_start,      */
/*    int pair_increment, bool resource_invariant, int pair_max)             */
/*                                                                           */
/*  Carry out multiple calls on KheResourcePairSimpleReassign.               */
/*                                                                           */
/*****************************************************************************/

static bool KheResourcePairDoSimpleRepair(KHE_SOLN soln, KHE_RESOURCE r1,
  KHE_RESOURCE r2, KHE_FRAME frame, int pair_parts, int pair_start,
  int pair_increment, bool resource_invariant, int pair_max)
{
  bool res;  int start_pi, stop_pi;
  res = false;
  for( start_pi = pair_start, stop_pi = start_pi + pair_parts;
       stop_pi <= KheFrameTimeGroupCount(frame);
       start_pi += pair_increment, stop_pi = start_pi + pair_parts)
    if( KheResourcePairSimpleReassign(soln, r1, r2,
	frame, start_pi, stop_pi - 1, resource_invariant, pair_max) )
      res = true;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheResourcePairSimpleRepair(KHE_SOLN soln, KHE_OPTIONS options)     */
/*                                                                           */
/*  Call KheResourcePairSimpleReassign repeatedly, depending on options.     */
/*                                                                           */
/*****************************************************************************/

bool KheResourcePairSimpleRepair(KHE_SOLN soln, KHE_OPTIONS options)
{
  bool resource_invariant, res;  KHE_FRAME frame;
  KHE_INSTANCE ins; KHE_RESOURCE_TYPE rt;  int i, j, k;  KHE_RESOURCE r1, r2;
  char *pair_select;  int pair_parts, pair_start, pair_increment, pair_max;
  HA_ARRAY_BOOL defective;  bool pair_off;  HA_ARENA a;

  /* quit now if rs_pair_off, or rs_pair_select is "none" */
  pair_off = KheOptionsGetBool(options, "rs_pair_off", false);
  pair_select = KheOptionsGet(options, "rs_pair_select", "defective");
  if( pair_off || strcmp(pair_select, "none") == 0 )
    return false;

  if( DEBUG7 )
    fprintf(stderr, "[ KheResourcePairSimpleRepair(soln, options)\n");

  /* quit now if the time limit has been reached */
  if( KheOptionsTimeLimitReached(options) )
  {
    if( DEBUG7 )
      fprintf(stderr, "] KheResourcePairSimpleRepair returning false (time)\n");
    return false;
  }

  /* get the options */
  ins = KheSolnInstance(soln);
  frame = KheOptionsFrame(options, "gs_common_frame", soln);
  /* tp = KheTimePartitionOption(options, "rs_time_partition", ins); */
  resource_invariant = KheOptionsGetBool(options, "rs_invariant", false);
  pair_parts = KheOptionsGetInt(options, "rs_pair_parts", 7);
  pair_start =  KheOptionsGetInt(options, "rs_pair_start", 0);
  pair_increment = KheOptionsGetInt(options, "rs_pair_increment", pair_parts);
  pair_max = KheOptionsGetInt(options, "rs_pair_max", 1000000);

  if( strcmp(pair_select, "adjacent") == 0 )
  {
    /* adjacent pairs of resources in each resource type */
    res = false;
    for( i = 0;  i < KheInstanceResourceTypeCount(ins);  i++ )
    {
      rt = KheInstanceResourceType(ins, i);
      for( j = 0;  j < KheResourceTypeResourceCount(rt) - 1;  j++ )
      {
	r1 = KheResourceTypeResource(rt, j);
	r2 = KheResourceTypeResource(rt, j + 1);
	if( KheResourcePairDoSimpleRepair(soln, r1, r2, frame, pair_parts,
	      pair_start, pair_increment, resource_invariant, pair_max) )
	  res = true;
      }
    }
  }
  else if( strcmp(pair_select, "all") == 0 )
  {
    /* all pairs of resources in each resource type */
    res = false;
    for( i = 0;  i < KheInstanceResourceTypeCount(ins);  i++ )
    {
      rt = KheInstanceResourceType(ins, i);
      for( j = 0;  j < KheResourceTypeResourceCount(rt) - 1;  j++ )
      {
	r1 = KheResourceTypeResource(rt, j);
	for( k = j + 1;  k < KheResourceTypeResourceCount(rt);  k++ )
	{
	  r2 = KheResourceTypeResource(rt, k);
	  if( KheResourcePairDoSimpleRepair(soln, r1, r2, frame, pair_parts,
		pair_start, pair_increment, resource_invariant, pair_max) )
	    res = true;
	}
      }
    }
  }
  else if( strcmp(pair_select, "defective") == 0 )
  {
    /* all pairs of resources for which at least one is defective */
    a = KheSolnArenaBegin(soln, false);
    HaArrayInit(defective, a);
    res = false;
    for( i = 0;  i < KheInstanceResourceTypeCount(ins);  i++ )
    {
      rt = KheInstanceResourceType(ins, i);

      /* work out which ones are defective */
      HaArrayClear(defective);
      for( j = 0;  j < KheResourceTypeResourceCount(rt);  j++ )
      {
	r1 = KheResourceTypeResource(rt, j);
	HaArrayAddLast(defective, KheSolnResourceCost(soln, r1) > 0);
      }

      /* visit all pairs, but only repair when at least one is defective */
      for( j = 0;  j < KheResourceTypeResourceCount(rt) - 1;  j++ )
      {
	r1 = KheResourceTypeResource(rt, j);
	for( k = j + 1;  k < KheResourceTypeResourceCount(rt);  k++ )
	{
	  r2 = KheResourceTypeResource(rt, k);
	  if( HaArray(defective, j) || HaArray(defective, k) )
	  {
	    if( KheResourcePairDoSimpleRepair(soln, r1, r2, frame, pair_parts,
		  pair_start, pair_increment, resource_invariant, pair_max) )
	      res = true;
	  }
	}
      }
    }
    KheSolnArenaEnd(soln, a);
  }
  else
  {
    HnAbort("KheResourcePairSimpleRepair: unknown pair_select value \"%s\"",
      pair_select);
    res = false;  /* keep compiler happy */
  }
  if( DEBUG7 )
    fprintf(stderr, "] KheResourcePairSimpleRepair returning %s\n",
      bool_show(res));
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KheResourcePairSimpleBusyRepair"                              */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_busy_resource_rec {
  KHE_RESOURCE		resource;		/* the resource              */
  int			curr_busy;		/* number of busy times      */
  int			max_busy;		/* limit on busy times       */
} KHE_BUSY_RESOURCE;

typedef HA_ARRAY(KHE_BUSY_RESOURCE) ARRAY_KHE_BUSY_RESOURCE;


/*****************************************************************************/
/*                                                                           */
/*  KHE_BUSY_RESOURCE KheBusyResourceMake(KHE_RESOURCE r,                    */
/*    int curr_busy, int max_busy)                                           */
/*                                                                           */
/*  Return a new busy resource object.                                       */
/*                                                                           */
/*****************************************************************************/

static KHE_BUSY_RESOURCE KheBusyResourceMake(KHE_RESOURCE r,
  int curr_busy, int max_busy)
{
  KHE_BUSY_RESOURCE res;
  res.resource = r;
  res.curr_busy = curr_busy;
  res.max_busy = max_busy;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  int KheBusyResourceIncreasingOverloadCmp(const void *t1, const void *t2) */
/*                                                                           */
/*  Comparison function for sorting an array of busy resources by            */
/*  increasing curr_busy - max_busy.                                         */
/*                                                                           */
/*****************************************************************************/

static int KheBusyResourceIncreasingOverloadCmp(const void *t1, const void *t2)
{
  KHE_BUSY_RESOURCE *bp1 = (KHE_BUSY_RESOURCE *) t1;
  KHE_BUSY_RESOURCE *bp2 = (KHE_BUSY_RESOURCE *) t2;
  return (bp1->curr_busy - bp1->max_busy) - (bp2->curr_busy - bp2->max_busy);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheResourcePairSimpleBusyRepair(KHE_SOLN soln, KHE_OPTIONS options) */
/*                                                                           */
/*  Find pairs of underloaded and overloaded assignments and do resource     */
/*  pair repairs on them.                                                    */
/*                                                                           */
/*  This function does not seem to help, despite trying some promising       */
/*  pairs.  Accordingly the default value of rs_bpair_off is true.           */
/*                                                                           */
/*****************************************************************************/

bool KheResourcePairSimpleBusyRepair(KHE_SOLN soln, KHE_OPTIONS options)
{
  bool resource_invariant, res, success, bpair_off;  KHE_FRAME frame;
  KHE_INSTANCE ins; KHE_RESOURCE_TYPE rt;  int curr_busy, max_busy;
  int bpair_parts, bpair_start, bpair_increment, bpair_max, i, j;
  KHE_RESOURCE r;  /* KHE_RESOURCE_TIMETABLE_MONITOR rtm; */  HA_ARENA a;
  ARRAY_KHE_BUSY_RESOURCE busy_resources;  KHE_BUSY_RESOURCE br1, br2;
  /* KHE_FRAME_WORKLOAD fw;  KHE_EVENT_TIMETABLE_MONITOR etm; */

  /* quit now if rs_bpair_off */
  bpair_off = KheOptionsGetBool(options, "rs_bpair_off", true);
  if( bpair_off )
    return false;

  /* quit now if the time limit has been reached */
  if( KheOptionsTimeLimitReached(options) )
    return false;

  /* get the options */
  ins = KheSolnInstance(soln);
  frame = KheOptionsFrame(options, "gs_common_frame", soln);
  resource_invariant = KheOptionsGetBool(options, "rs_invariant", false);
  bpair_parts = KheOptionsGetInt(options, "rs_bpair_parts", 14);
  bpair_start =  KheOptionsGetInt(options, "rs_bpair_start", 0);
  bpair_increment = KheOptionsGetInt(options, "rs_bpair_increment",
    bpair_parts / 2);
  bpair_max = KheOptionsGetInt(options, "rs_bpair_max", 1000000);
  if( DEBUG3 )
  {
    fprintf(stderr, "[ KheResourcePairSimpleBusyRepair(soln, options)\n");
    fprintf(stderr,
      "  bpair_parts %d, bpair_start %d, bpair_inc %d, bpair_max %d\n",
      bpair_parts, bpair_start, bpair_increment, bpair_max);
  }

  a = KheSolnArenaBegin(soln, false);
  /* a = HaAren aMake(); */
  HaArrayInit(busy_resources, a);
  res = false;
  for( i = 0;  i < KheInstanceResourceTypeCount(ins);  i++ )
  {
    rt = KheInstanceResourceType(ins, i);

    /* make the busy resources array for the resources of type rt */
    /* ***
    etm = (KHE_EVENT_TIMETABLE_MONITOR)
      KheOptionsGetObject(options, "gs_event_timetable_monitor", NULL);
    fw = KheFrame WorkloadMake(frame, rt, etm);
    *** */
    HaArrayClear(busy_resources);
    for( j = 0;  j < KheResourceTypeResourceCount(rt);  j++ )
    {
      r = KheResourceTypeResource(rt, j);
      /* rtm = KheResourceTimetableMonitor(soln, r); */
      curr_busy = KheResourceBusyTimes(soln, r);
      if( !KheResourceMaxBusyTimes(soln, r, &max_busy) )
	max_busy = KheInstanceTimeCount(KheSolnInstance(soln));
      /* ***
      curr_busy = KheResourceTimetable MonitorBusyTimes(rtm); 
      max_busy = KheFrameResourc eMaxBusyTimes(frame, fw, r);
      *** */
      br1 = KheBusyResourceMake(r, curr_busy, max_busy);
      HaArrayAddLast(busy_resources, br1);
    }
    /* KheFrameWorkloadDelete(fw); */

    /* sort the busy resources and pair worst with best etc. */
    HaArraySort(busy_resources, &KheBusyResourceIncreasingOverloadCmp);
    for( j = 0;  j < HaArrayCount(busy_resources) / 2;  j++ )
    {
      br1 = HaArray(busy_resources, j);
      br2 = HaArray(busy_resources, HaArrayCount(busy_resources) - j - 1);
      if( br2.curr_busy <= br2.max_busy )
	break;  /* br2 is not overloaded */
      if( br1.curr_busy >= br1.max_busy )
	break;  /* br1 is not underloaded */
      success = KheResourcePairDoSimpleRepair(soln, br1.resource, br2.resource,
	    frame, bpair_parts, bpair_start, bpair_increment,
	    resource_invariant, bpair_max);
      if( DEBUG3 )
	fprintf(stderr, "  tried (%s, %d, %d) and (%s, %d, %d): %s\n",
	  KheResourceId(br1.resource), br1.curr_busy, br1.max_busy,
	  KheResourceId(br2.resource), br2.curr_busy, br2.max_busy,
	  bool_show(success));
      if( success )
	res = true;
    }
  }
  KheSolnArenaEnd(soln, a);
  /* HaArenaD elete(a); */
  /* MArrayFree(busy_resources); */
  if( DEBUG3 )
    fprintf(stderr, "] KheResourcePairSimpleBusyRepair returning %s\n",
      bool_show(res));
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "resource triples"                                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool KheResourceTripleDoSimpleRepair(KHE_SOLN soln,                      */
/*    KHE_RESOURCE r1, KHE_RESOURCE r2, KHE_RESOURCE r3,                     */
/*    KHE_TIME_PARTITION tp, int pair_parts, int pair_start,                 */
/*    int pair_increment, bool resource_invariant, int pair_max)             */
/*                                                                           */
/*  Carry out multiple calls on KheResourceTripleSimpleReassign.             */
/*                                                                           */
/*****************************************************************************/

static bool KheResourceTripleDoSimpleRepair(KHE_SOLN soln,
  KHE_RESOURCE r1, KHE_RESOURCE r2, KHE_RESOURCE r3, KHE_FRAME frame,
  /* KHE_TIME_PARTITION tp, */ int pair_parts, int pair_start,
  int pair_increment, bool resource_invariant, int pair_max)
{
  bool res;  int start_pi, stop_pi;
  res = false;
  for( start_pi = pair_start, stop_pi = start_pi + pair_parts - 1;
       stop_pi < KheFrameTimeGroupCount(frame);
       start_pi += pair_increment, stop_pi = start_pi + pair_parts - 1)
    if( KheResourceTripleSimpleReassign(soln, r1, r2, r3,
	frame, start_pi, stop_pi - 1, resource_invariant, pair_max) )
      res = true;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  bool TripleSimpleRepairByConstraint(KHE_SOLN soln, char *triple_select,  */
/*    KHE_FRAME frame, int triple_parts, int triple_start,                   */
/*    int triple_increment, bool resource_invariant, int triple_max)         */
/*                                                                           */
/*  Solve triples of resources, selected by constraint.                      */
/*                                                                           */
/*****************************************************************************/

typedef HA_ARRAY(KHE_CLUSTER_BUSY_TIMES_CONSTRAINT)
  ARRAY_KHE_CLUSTER_BUSY_TIMES_CONSTRAINT;
typedef HA_ARRAY(KHE_RESOURCE) ARRAY_KHE_RESOURCE;

static bool TripleSimpleRepairByConstraint(KHE_SOLN soln, char *triple_select,
  KHE_FRAME frame, int triple_parts, int triple_start,
  int triple_increment, bool resource_invariant, int triple_max)
{
  HA_ARENA a;  KHE_INSTANCE ins;  int i, j, k, pos, n;  bool res;
  ARRAY_KHE_CLUSTER_BUSY_TIMES_CONSTRAINT constraints;
  ARRAY_KHE_RESOURCE violating_resources, slack_resources;
  KHE_CLUSTER_BUSY_TIMES_CONSTRAINT cbtc;  KHE_RESOURCE_TYPE rt;
  KHE_CONSTRAINT constr;  KHE_RESOURCE r, sr1, sr2;  KHE_MONITOR m;
  KHE_CLUSTER_BUSY_TIMES_MONITOR cbtm;  KHE_COST cost;

  if( DEBUG8 )
    fprintf(stderr, "[ TripleSimpleRepairByConstraint(soln, %s, parts %d, "
      "start %d, inc %d, max %d)\n", triple_select, triple_parts,
      triple_start, triple_increment, triple_max);

  a = KheSolnArenaBegin(soln, false);
  HaArrayInit(constraints, a);
  HaArrayInit(violating_resources, a);
  HaArrayInit(slack_resources, a);
  res = false;

  /* find the cluster busy times constraints that match triple_select */
  ins = KheSolnInstance(soln);
  for( i = 0;  i < KheInstanceConstraintCount(ins);  i++ )
  {
    constr = KheInstanceConstraint(ins, i);
    if( KheConstraintTag(constr) == KHE_CLUSTER_BUSY_TIMES_CONSTRAINT_TAG
	&& strstr(KheConstraintName(constr), triple_select) != NULL )
    {
      HaArrayAddLast(constraints, (KHE_CLUSTER_BUSY_TIMES_CONSTRAINT) constr);
      if( DEBUG8 )
	KheConstraintDebug(constr, 2, 2, stderr);
    }
  }

  /* do the job for each resource type */
  for( n = 0;  n < KheInstanceResourceTypeCount(ins);  n++ )
  {
    /* build arrays of violating and slack resources of type rt */
    rt = KheInstanceResourceType(ins, n);
    HaArrayClear(violating_resources);
    HaArrayClear(slack_resources);
    for( j = 0;  j < KheResourceTypeResourceCount(rt);  j++ )
    {
      r = KheResourceTypeResource(rt, j);
      for( k = 0;  k < KheSolnResourceMonitorCount(soln, r);  k++ )
      {
	m = KheSolnResourceMonitor(soln, r, k);
	if( KheMonitorTag(m) == KHE_CLUSTER_BUSY_TIMES_MONITOR_TAG )
	{
	  cbtm = (KHE_CLUSTER_BUSY_TIMES_MONITOR) m;
	  cbtc = KheClusterBusyTimesMonitorConstraint(cbtm);
	  if( HaArrayContains(constraints, cbtc, &pos) )
	  {
	    if( KheMonitorCost(m) > 0 )
	    {
	      HaArrayAddLast(violating_resources, r);
	      if( DEBUG8 )
		fprintf(stderr, "  violating resource %s\n", KheResourceId(r));
	      break;  /* next resource */
	    }
	    else if( !KheClusterBusyTimesMonitorAtMaxLimitCount(cbtm) )
	    {
	      HaArrayAddLast(slack_resources, r);
	      if( DEBUG8 )
		fprintf(stderr, "  slack resource %s\n", KheResourceId(r));
	      break;  /* next resource */
	    }
	  }
	}
      }
    }

    /* solve for each violating resource and pair of slack resources */
    HaArrayForEach(violating_resources, r, i)
      HaArrayForEach(slack_resources, sr1, j)
	for( k = j + 1;  k < HaArrayCount(slack_resources);  k++ )
	{
	  sr2 = HaArray(slack_resources, k);
	  if( DEBUG8 )
	    fprintf(stderr, "  solving for violation %s, slack %s and %s:\n",
	      KheResourceId(r), KheResourceId(sr1), KheResourceId(sr2));
	  cost = KheSolnCost(soln);
	  if( KheResourceTripleDoSimpleRepair(soln, r, sr1, sr2, frame,
	      triple_parts, triple_start, triple_increment,
	      resource_invariant, triple_max) )
	  {
	    if( DEBUG8 )
	      fprintf(stderr, "    success: %.5f -> %.5f\n",
		KheCostShow(cost), KheCostShow(KheSolnCost(soln)));
	    res = true;
	  }
	}
  }

  KheSolnArenaEnd(soln, a);
  if( DEBUG8 )
    fprintf(stderr, "] TripleSimpleRepairByConstraint returning %s\n",
      bool_show(res));
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheResourcePairSimpleRepair(KHE_SOLN soln, KHE_OPTIONS options)     */
/*                                                                           */
/*  Call KheResourcePairSimpleReassign repeatedly, depending on options.     */
/*                                                                           */
/*****************************************************************************/

bool KheResourceTripleSimpleRepair(KHE_SOLN soln, KHE_OPTIONS options)
{
  bool resource_invariant, res;  KHE_FRAME frame;  /* KHE_TIME_PARTITION tp; */
  KHE_INSTANCE ins; KHE_RESOURCE_TYPE rt;  int i, j, k, m;  HA_ARENA a;
  KHE_RESOURCE r1, r2, r3;  HA_ARRAY_BOOL defective;  char *triple_select;
  bool triple_off; int triple_parts, triple_start, triple_increment, triple_max;

  /* quit now if rs_triple_off, or triple_select is "none" */
  if( DEBUG8 )
    fprintf(stderr, "[ KheResourceTripleSimpleRepair(soln, options)\n");
  triple_off = KheOptionsGetBool(options, "rs_triple_off", false);
  triple_select = KheOptionsGet(options, "rs_triple_select", "none");
  if( DEBUG8 )
    fprintf(stderr, "  rs_triple_off=%s rs_triple_select=\"%s\"\n",
      bool_show(triple_off), triple_select);
  if( triple_off || strcmp(triple_select, "none") == 0 )
  {
    if( DEBUG8 )
      fprintf(stderr, "] KheResourceTripleSimpleRepair ret. false (options)\n");
    return false;
  }

  /* quit now if the time limit has been reached */
  if( KheOptionsTimeLimitReached(options) )
  {
    if( DEBUG8 )
      fprintf(stderr, "] KheResourceTripleSimpleRepair ret. false (time)\n");
    return false;
  }

  /* get the other options */
  ins = KheSolnInstance(soln);
  frame = KheOptionsFrame(options, "gs_common_frame", soln);
  /* tp = KheTimePartitionOption(options, "rs_time_partition", ins); */
  resource_invariant = KheOptionsGetBool(options, "rs_invariant", false);
  triple_parts = KheOptionsGetInt(options, "rs_triple_parts", 7);
  triple_start =  KheOptionsGetInt(options, "rs_triple_start", 0);
  triple_increment = KheOptionsGetInt(options, "rs_triple_increment",
    triple_parts);
  triple_max = KheOptionsGetInt(options, "rs_triple_max", 100000);

  res = false;
  if( strcmp(triple_select, "adjacent") == 0 )
  {
    /* adjacent triples of resources in each resource type */
    for( i = 0;  i < KheInstanceResourceTypeCount(ins);  i++ )
    {
      rt = KheInstanceResourceType(ins, i);
      for( j = 0;  j < KheResourceTypeResourceCount(rt) - 2;  j++ )
      {
	r1 = KheResourceTypeResource(rt, j);
	r2 = KheResourceTypeResource(rt, j + 1);
	r3 = KheResourceTypeResource(rt, j + 2);
	if( KheResourceTripleDoSimpleRepair(soln, r1, r2, r3, frame,
	      triple_parts, triple_start, triple_increment,
	      resource_invariant, triple_max) )
	  res = true;
      }
    }
  }
  else if( strcmp(triple_select, "all") == 0 )
  {
    /* all triples of resources in each resource type */
    for( i = 0;  i < KheInstanceResourceTypeCount(ins);  i++ )
    {
      rt = KheInstanceResourceType(ins, i);
      for( j = 0;  j < KheResourceTypeResourceCount(rt) - 1;  j++ )
      {
	r1 = KheResourceTypeResource(rt, j);
	for( k = j + 1;  k < KheResourceTypeResourceCount(rt);  k++ )
	{
	  r2 = KheResourceTypeResource(rt, k);
	  for( m = k + 1;  m < KheResourceTypeResourceCount(rt);  m++ )
	  {
	    r3 = KheResourceTypeResource(rt, m);
	    if( KheResourceTripleDoSimpleRepair(soln, r1, r2, r3, frame,
		triple_parts, triple_start, triple_increment,
		resource_invariant, triple_max) )
	      res = true;
	  }
	}
      }
    }
  }
  else if( strcmp(triple_select, "defective") == 0 )
  {
    /* all triples of resources for which at least one is defective */
    a = KheSolnArenaBegin(soln, false);
    /* a = HaAren aMake(); */
    HaArrayInit(defective, a);
    res = false;
    for( i = 0;  i < KheInstanceResourceTypeCount(ins);  i++ )
    {
      rt = KheInstanceResourceType(ins, i);

      /* work out which ones are defective */
      HaArrayClear(defective);
      for( j = 0;  j < KheResourceTypeResourceCount(rt);  j++ )
      {
	r1 = KheResourceTypeResource(rt, j);
	HaArrayAddLast(defective, KheSolnResourceCost(soln, r1) > 0);
      }

      /* visit all triples, but only repair when at least one is defective */
      for( j = 0;  j < KheResourceTypeResourceCount(rt) - 1;  j++ )
      {
	r1 = KheResourceTypeResource(rt, j);
	for( k = j + 1;  k < KheResourceTypeResourceCount(rt);  k++ )
	{
	  r2 = KheResourceTypeResource(rt, k);
          for( m = k + 1;  m < KheResourceTypeResourceCount(rt);  m++ )
	  {
	    r3 = KheResourceTypeResource(rt, m);
	    if( HaArray(defective, j) || HaArray(defective, k) ||
		HaArray(defective, m) )
	    {
	      if( KheResourceTripleDoSimpleRepair(soln, r1, r2, r3, frame,
		    triple_parts, triple_start, triple_increment,
		    resource_invariant, triple_max) )
		res = true;
	    }
	  }
	}
      }
    }
    KheSolnArenaEnd(soln, a);
    /* HaArenaDele te(a); */
    /* MArrayFree(defective); */
  }
  else if( strstr(triple_select, "constraint:") == triple_select )
  {
    /* selection by constraint */
    triple_select = &triple_select[strlen("constraint:")];
    res = TripleSimpleRepairByConstraint(soln, triple_select, frame,
      triple_parts, triple_start, triple_increment, resource_invariant,
      triple_max);
  }
  else
  {
    HnAbort("KheResourceTripleSimpleRepair: unknown triple_select value \"%s\"",
      triple_select);
    res = false;  /* keep compiler happy */
  }
  if( DEBUG8 )
    fprintf(stderr, "] KheResourceTripleSimpleRepair returning %s\n",
      bool_show(res));
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "resource swaps"                                               */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool KheResourcePairSwap(KHE_SOLN soln, KHE_RESOURCE r1, KHE_RESOURCE r2,*/
/*    KHE_FRAME frame, int fi, int li, bool resource_invariant)              */
/*                                                                           */
/*  Simple resource pair repair.                                             */
/*                                                                           */
/*****************************************************************************/

bool KheResourcePairSwap(KHE_SOLN soln, KHE_RESOURCE r1, KHE_RESOURCE r2,
  KHE_FRAME frame, int fi, int li, bool resource_invariant)
{
  KHE_COST init_cost;  int i, init_count;  KHE_TIME_GROUP tg, tg2;
  bool success;  KHE_RESOURCE_TIMETABLE_MONITOR rtm1, rtm2;  KHE_MARK mark;
  KHE_TASK_SET ts1, ts2;
  if( DEBUG4 )
    fprintf(stderr, "[ KheResourcePairSwap(%s, %s, frame[%d..%d], %s)\n",
      KheResourceId(r1), KheResourceId(r2), fi, li,
      bool_show(resource_invariant));

  /* get the sets of tasks assigned to each resource */
  rtm1 = KheResourceTimetableMonitor(soln, r1);
  ts1 = KheTaskSetMake(soln);
  rtm2 = KheResourceTimetableMonitor(soln, r2);
  ts2 = KheTaskSetMake(soln);
  for( i = fi;  i <= li;  i++ )
  {
    tg = KheFrameTimeGroup(frame, i);
    KheResourceTimetableMonitorAddProperRootTasks(rtm1, tg, true, ts1);
    KheResourceTimetableMonitorAddProperRootTasks(rtm2, tg, true, ts2);
  }
  
  /* try the move */
  init_cost = KheSolnCost(soln);
  KheAtomicOperationBegin(soln, &mark, &init_count, resource_invariant);
  success = KheTaskSetMoveResource(ts1, r2) && KheTaskSetMoveResource(ts2, r1);
  KheAtomicOperationEnd(soln, &mark, &init_count, resource_invariant,
    success && KheSolnCost(soln) < init_cost);
  KheTaskSetDelete(ts1);
  KheTaskSetDelete(ts2);
  if( DEBUG5 && KheSolnCost(soln) < init_cost )
  {
    tg = KheFrameTimeGroup(frame, fi);
    tg2 = KheFrameTimeGroup(frame, li);
    fprintf(stderr, "  KheResourcePairSwap(%s, %s, frame[%s..%s]) "
      "%.5f -> %.5f\n", KheResourceId(r1), KheResourceId(r2),
      KheTimeGroupId(tg), KheTimeGroupId(tg2), KheCostShow(init_cost),
      KheCostShow(KheSolnCost(soln)));
  }
  if( DEBUG4 )
    fprintf(stderr, "] KheResourcePairSwap returning %s (%.5f -> %.5f)\n",
      bool_show(KheSolnCost(soln) < init_cost), KheCostShow(init_cost),
      KheCostShow(KheSolnCost(soln)));
  return KheSolnCost(soln) < init_cost;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheResourcePairDoSwap(KHE_SOLN soln, KHE_RESOURCE r1,               */
/*    KHE_RESOURCE r2, KHE_FRAME frame, int swap_parts, int swap_start,      */
/*    int swap_increment, bool resource_invariant)                           */
/*                                                                           */
/*  Carry out multiple calls on KheResourcePairSwap.                         */
/*                                                                           */
/*****************************************************************************/

static bool KheResourcePairDoSwap(KHE_SOLN soln, KHE_RESOURCE r1,
  KHE_RESOURCE r2, KHE_FRAME frame, int swap_parts, int swap_start,
  int swap_increment, bool resource_invariant)
{
  bool res;  int start_pi, stop_pi;
  res = false;
  for( start_pi = swap_start, stop_pi = start_pi + swap_parts;
       stop_pi <= KheFrameTimeGroupCount(frame);
       start_pi += swap_increment, stop_pi = start_pi + swap_parts)
    if( KheResourcePairSwap(soln, r1, r2,
	frame, start_pi, stop_pi - 1, resource_invariant) )
      res = true;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheResourcePairSwapRepair(KHE_SOLN soln, KHE_OPTIONS options)       */
/*                                                                           */
/*  Call KheResourcePairSwap repeatedly, depending on options.               */
/*                                                                           */
/*****************************************************************************/

bool KheResourcePairSwapRepair(KHE_SOLN soln, KHE_OPTIONS options)
{
  bool resource_invariant, res;  KHE_FRAME frame;
  KHE_INSTANCE ins; KHE_RESOURCE_TYPE rt;  int i, j, k;  KHE_RESOURCE r1, r2;
  char *swap_select;  int swap_parts, swap_start, swap_increment;
  HA_ARRAY_BOOL defective;  bool swap_off;  HA_ARENA a;

  /* quit now if rs_swap_off, or rs_swap_select is "none" */
  swap_off = KheOptionsGetBool(options, "rs_swap_off", false);
  swap_select = KheOptionsGet(options, "rs_swap_select", "defective");
  if( swap_off || strcmp(swap_select, "none") == 0 )
    return false;

  /* quit now if the time limit has been reached */
  if( KheOptionsTimeLimitReached(options) )
    return false;

  /* get the options */
  ins = KheSolnInstance(soln);
  frame = KheOptionsFrame(options, "gs_common_frame", soln);
  resource_invariant = KheOptionsGetBool(options, "rs_invariant", false);
  swap_parts = KheOptionsGetInt(options, "rs_swap_parts",
    KheFrameTimeGroupCount(frame));
  swap_start =  KheOptionsGetInt(options, "rs_swap_start", 0);
  swap_increment = KheOptionsGetInt(options, "rs_swap_increment", swap_parts);

  if( strcmp(swap_select, "adjacent") == 0 )
  {
    /* adjacent pairs of resources in each resource type */
    res = false;
    for( i = 0;  i < KheInstanceResourceTypeCount(ins);  i++ )
    {
      rt = KheInstanceResourceType(ins, i);
      for( j = 0;  j < KheResourceTypeResourceCount(rt) - 1;  j++ )
      {
	r1 = KheResourceTypeResource(rt, j);
	r2 = KheResourceTypeResource(rt, j + 1);
	if( KheResourcePairDoSwap(soln, r1, r2, frame, swap_parts,
	      swap_start, swap_increment, resource_invariant) )
	  res = true;
      }
    }
    return res;
  }
  else if( strcmp(swap_select, "all") == 0 )
  {
    /* all pairs of resources in each resource type */
    res = false;
    for( i = 0;  i < KheInstanceResourceTypeCount(ins);  i++ )
    {
      rt = KheInstanceResourceType(ins, i);
      for( j = 0;  j < KheResourceTypeResourceCount(rt) - 1;  j++ )
      {
	r1 = KheResourceTypeResource(rt, j);
	for( k = j + 1;  k < KheResourceTypeResourceCount(rt);  k++ )
	{
	  r2 = KheResourceTypeResource(rt, k);
	  if( KheResourcePairDoSwap(soln, r1, r2, frame, swap_parts,
		swap_start, swap_increment, resource_invariant) )
	    res = true;
	}
      }
    }
    return res;
  }
  else if( strcmp(swap_select, "defective") == 0 )
  {
    /* all pairs of resources for which at least one is defective */
    a = KheSolnArenaBegin(soln, false);
    HaArrayInit(defective, a);
    res = false;
    for( i = 0;  i < KheInstanceResourceTypeCount(ins);  i++ )
    {
      rt = KheInstanceResourceType(ins, i);

      /* work out which ones are defective */
      HaArrayClear(defective);
      for( j = 0;  j < KheResourceTypeResourceCount(rt);  j++ )
      {
	r1 = KheResourceTypeResource(rt, j);
	HaArrayAddLast(defective, KheSolnResourceCost(soln, r1) > 0);
      }

      /* visit all pairs, but only repair when at least one is defective */
      for( j = 0;  j < KheResourceTypeResourceCount(rt) - 1;  j++ )
      {
	r1 = KheResourceTypeResource(rt, j);
	for( k = j + 1;  k < KheResourceTypeResourceCount(rt);  k++ )
	{
	  r2 = KheResourceTypeResource(rt, k);
	  if( HaArray(defective, j) || HaArray(defective, k) )
	  {
	    if( KheResourcePairDoSwap(soln, r1, r2, frame, swap_parts,
		  swap_start, swap_increment, resource_invariant) )
	      res = true;
	  }
	}
      }
    }
    KheSolnArenaEnd(soln, a);
    return res;
  }
  else
  {
    HnAbort("KheResourcePairSwap: unknown swap_select value \"%s\"",
      swap_select);
    return false;  /* keep compiler happy */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheResourcePairAllSwapRepair(KHE_SOLN soln, KHE_OPTIONS options)    */
/*                                                                           */
/*  Call KheResourcePairSwap repeatedly, depending on options.               */
/*                                                                           */
/*****************************************************************************/

void KheResourcePairAllSwapRepair(KHE_SOLN soln, KHE_RESOURCE_TYPE rt,
  KHE_OPTIONS options)
{
  ARRAY_KHE_TASK_SET task_sets;  HA_ARENA a;  KHE_RESOURCE r, r2;
  KHE_TASK_SET ts, ts2;  KHE_FRAME frame;  KHE_TASK_FINDER tf;
  int i, j, len, start, init_count;  KHE_COST init_cost;  KHE_MARK mark;
  bool success;

  /* make one task set for each resource of rt */
  if( DEBUG6 )
    fprintf(stderr, "[ KheResourcePairAllSwapRepair(soln, %s, options)\n",
      KheResourceTypeId(rt));
  a = KheSolnArenaBegin(soln, false);
  HaArrayInit(task_sets, a);
  for( i = 0;  i < KheResourceTypeResourceCount(rt);  i++ )
  {
    r = KheResourceTypeResource(rt, i);
    ts = KheTaskSetMake(soln);
    HaArrayAddLast(task_sets, ts);
  }

  /* iterate over all lengths */
  tf = KheTaskFinderMake(soln, options, a);
  frame = KheOptionsFrame(options, "gs_common_frame", soln);
  for( len = 1;  len <= KheFrameTimeGroupCount(frame);  len++ )
  {
    /* iterate over all starting positions */
    for( start = 0;  start <= KheFrameTimeGroupCount(frame) - len;  start++ )
    {
      /* build the task sets for this interval */
      RESTART:
      for( i = 0;  i < KheResourceTypeResourceCount(rt);  i++ )
      {
	r = KheResourceTypeResource(rt, i);
	ts = HaArray(task_sets, i);
	KheTaskSetClear(ts);  /* redundant, but anyway */
	KheFindTasksInInterval(tf, r, start, start + len - 1, NULL, ts);
      }

      /* try all swaps */
      for( i = 0;  i < KheResourceTypeResourceCount(rt);  i++ )
      {
	r = KheResourceTypeResource(rt, i);
	ts = HaArray(task_sets, i);
	for( j = i + 1;  j < KheResourceTypeResourceCount(rt);  j++ )
	{
	  r2 = KheResourceTypeResource(rt, j);
	  ts2 = HaArray(task_sets, j);
	  init_cost = KheSolnCost(soln);
	  if( DEBUG6 )
	    fprintf(stderr, "    try %s with %s in [%d, %d]:\n",
	      KheResourceId(r), KheResourceId(r2), start, start + len - 1);
	  KheAtomicOperationBegin(soln, &mark, &init_count, false);
	  success = KheTaskSetMoveResource(ts, r2) &&
	    KheTaskSetMoveResource(ts2, r);
	  if( KheAtomicOperationEnd(soln, &mark, &init_count, false,
		success && KheSolnCost(soln) < init_cost) )
	  {
	    if( DEBUG6 )
	      fprintf(stderr, "    swap %s with %s in [%d, %d]: %.5f -> %.5f\n",
		KheResourceId(r), KheResourceId(r2), start, start + len - 1,
		KheCostShow(init_cost), KheCostShow(KheSolnCost(soln)));
	    goto RESTART;
	  }
	}
      }
    }
  }

  /* tidy up and exit */
  KheSolnArenaEnd(soln, a);
  if( DEBUG6 )
    fprintf(stderr, "] KheResourcePairAllSwapRepair returning\n");
}
