
/*****************************************************************************/
/*                                                                           */
/*  THE KHE HIGH SCHOOL TIMETABLING ENGINE                                   */
/*  COPYRIGHT (C) 2010 Jeffrey H. Kingston                                   */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 3, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         khe_sr_resource_run.c                                      */
/*  DESCRIPTION:  Resource run repair                                        */
/*                                                                           */
/*****************************************************************************/
#include "khe_solvers.h"
#include <limits.h>

#define bool_show(val)  ((val) ? "true" : "false")

#define DEBUG1 1			/* public functions                  */
#define DEBUG2 0			/* individual solves                 */
#define DEBUG3 0			/* KheRunResourceEnsureRunsUpToDate  */
#define DEBUG4 0			/* RunMake and RunDelete             */
#define	DEBUG5 0			/* solve details                     */
#define DEBUG6 1			/* brief display of improvements     */
#define DEBUG8 0			/* SolveDefective                    */
#define DEBUG8_ID "HN_2"		/* SolveDefective                    */
#define DEBUG9 0			/*                                   */


/*****************************************************************************/
/*                                                                           */
/*  RUN_SELECT_TYPE - types of selection                                     */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  RUN_SELECT_NONE,
  RUN_SELECT_ADJACENT,
  RUN_SELECT_DEFECTIVE,
  RUN_SELECT_ALL
} RUN_SELECT_TYPE;


/*****************************************************************************/
/*                                                                           */
/*  RESOURCE_STATE - state of resource                                       */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  RESOURCE_STATE_UNKNOWN,
  RESOURCE_STATE_NOT_DEFECTIVE,
  RESOURCE_STATE_DEFECTIVE
} RESOURCE_STATE;


/*****************************************************************************/
/*                                                                           */
/*  KHE_RUN - one run of adjacent tasks, with their interval and domain      */
/*  KHE_RUN_RESOURCE - information about one resource                        */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_run_rec *KHE_RUN;
typedef HA_ARRAY(KHE_RUN) ARRAY_KHE_RUN;

typedef struct khe_run_resource_rec *KHE_RUN_RESOURCE;
typedef HA_ARRAY(KHE_RUN_RESOURCE) ARRAY_KHE_RUN_RESOURCE;

struct khe_run_rec {
  KHE_TASK_SET		task_set;		/* the run's tasks           */
  int			first_index;		/* start of interval         */
  int			last_index;		/* end of interval           */
  ARRAY_KHE_RUN_RESOURCE domain;		/* legal resources           */
};

struct khe_run_resource_rec {
  KHE_RESOURCE		resource;		/* the resource              */
  int			first_index;		/* interval runs are for     */
  int			last_index;		/* interval runs are for     */
  ARRAY_KHE_RUN		runs;			/* the resource's runs       */
  RESOURCE_STATE	state;			/* if resource is defective  */
  int			curr_last_index;	/* solve last assigned time  */
};


/*****************************************************************************/
/*                                                                           */
/*  KHE_RESOURCE_RUN_SOLVER - a solver for runs                              */
/*                                                                           */
/*****************************************************************************/

struct khe_resource_run_solver_rec {

  /* always defined */
  HA_ARENA			arena;
  KHE_SOLN			soln;
  KHE_FRAME			days_frame;
  KHE_TASK_FINDER		task_finder;
  bool				resource_invariant;
  int				max_assignments;
  ARRAY_KHE_RUN_RESOURCE	dormant_resources;
  ARRAY_KHE_RUN_RESOURCE	resources;
  ARRAY_KHE_RUN			free_runs;

  /* defined during solves */
  ARRAY_KHE_RUN			movable_runs;
  ARRAY_KHE_RUN			immovable_runs;
  KHE_MARK			init_mark;
  int				init_defect_count;
  bool				debug_on;
  int				curr_assignments;
};


/*****************************************************************************/
/*                                                                           */
/*  Submodule "resource state"                                               */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  char *ResourceStateShow(RESOURCE_STATE state)                            */
/*                                                                           */
/*  Display a resource state.                                                */
/*                                                                           */
/*****************************************************************************/

static char *ResourceStateShow(RESOURCE_STATE state)
{
  switch( state )
  {
    case RESOURCE_STATE_UNKNOWN:	return "state_unknown";
    case RESOURCE_STATE_NOT_DEFECTIVE:	return "state_not_defective";
    case RESOURCE_STATE_DEFECTIVE:	return "state_defective";
    default:				return "?";
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "select type"                                                  */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  RUN_SELECT_TYPE KheSelectTypeGet(KHE_OPTIONS options)                    */
/*                                                                           */
/*  Return an enum value representing the rs_run_select option.              */
/*                                                                           */
/*****************************************************************************/

static RUN_SELECT_TYPE KheSelectTypeGet(KHE_OPTIONS options)
{
  char *val;
  val = KheOptionsGet(options, "rs_run_select", "defective");
  if( strcmp(val, "defective") == 0 )
    return RUN_SELECT_DEFECTIVE;
  else if( strcmp(val, "none") == 0 )
    return RUN_SELECT_NONE;
  else if( strcmp(val, "adjacent") == 0 )
    return RUN_SELECT_ADJACENT;
  else if( strcmp(val, "all") == 0 )
    return RUN_SELECT_ALL;
  else
  {
    HnAbort("KheResourceRunRepair: unknown rs_run_select \"%s\"", val);
    return RUN_SELECT_ALL;  /* keep compiler happy */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheRunSelectTypeShow(RUN_SELECT_TYPE run_select)                   */
/*                                                                           */
/*  Display a run_select value.                                              */
/*                                                                           */
/*****************************************************************************/

static char *KheRunSelectTypeShow(RUN_SELECT_TYPE run_select)
{
  switch( run_select )
  {
    case RUN_SELECT_DEFECTIVE:	return "defective";
    case RUN_SELECT_NONE:	return "none";
    case RUN_SELECT_ADJACENT:	return "adjacent";
    case RUN_SELECT_ALL:	return "all";
    default:			return "?";
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "runs"                                                         */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_RUN KheRunMake(KHE_RESOURCE_RUN_SOLVER rrs)                          */
/*                                                                           */
/*  Make or get a run object, whose task set and domain are initially        */
/*  empty, and whose first_index and last_index fields are undefined.        */
/*                                                                           */
/*****************************************************************************/

static KHE_RUN KheRunMake(KHE_RESOURCE_RUN_SOLVER rrs)
{
  KHE_RUN res;
  if( HaArrayCount(rrs->free_runs) > 0 )
  {
    /* get the run object from the free list */
    res = HaArrayLastAndDelete(rrs->free_runs);
    KheTaskSetClear(res->task_set);
    HaArrayClear(res->domain);
    if( DEBUG4 )
      fprintf(stderr, "KheRunMake returning %p from free list\n", (void *) res);
  }
  else
  {
    /* make a new run object */
    HaMake(res, rrs->arena);
    res->task_set = KheTaskSetMake(rrs->soln);
    HaArrayInit(res->domain, rrs->arena);
    if( DEBUG4 )
      fprintf(stderr, "KheRunMake returning %p from arena\n", (void *) res);
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheRunDelete(KHE_RUN run, KHE_RESOURCE_RUN_SOLVER rrs)              */
/*                                                                           */
/*  Delete run; place it on rrs's free list.  Its task set and domain go     */
/*  with it.                                                                 */
/*                                                                           */
/*****************************************************************************/

static void KheRunDelete(KHE_RUN run, KHE_RESOURCE_RUN_SOLVER rrs)
{
  if( DEBUG4 )
    fprintf(stderr, "KheRunDelete deleting %p\n", (void *) run);
  HaArrayAddLast(rrs->free_runs, run);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheRunSplits(KHE_RUN run, KHE_RESOURCE_RUN_SOLVER rrs,              */
/*    KHE_RUN *run1, KHE_RUN *run2)                                          */
/*                                                                           */
/*  If run can be split into two pieces, return true and set *run1 and       */
/*  *run2 to the two pieces, deleting run.  Otherwise return false,          */
/*  leaving run unchanged.                                                   */
/*                                                                           */
/*****************************************************************************/

/* *** not used in this version of resource reassignment
static bool KheRunSplits(KHE_RUN run, KHE_RESOURCE_RUN_SOLVER rrs,
  KHE_RUN *run1, KHE_RUN *run2)
{
  int i, half_way;

  ** no good if less than four days covered */
  /* ***
  if( run->last_index - run->first_index + 1 < 4 )
    return false;
  *** **

  ** no good if less than two tasks in task set **
  if( KheTaskSetTaskCount(run->task_set) < 2 )
    return false;

  ** first half in *run1, second half in *run2 **
  half_way = KheTaskSetTaskCount(run->task_set) / 2;
  *run1 = KheRunMake(rrs);
  *run2 = KheRunMake(rrs);
  for( i = 0;  i < half_way;  i++ )
    KheTaskSetAddTask((*run1)->task_set, KheTaskSetTask(run->task_set, i));
  KheTaskFinderTaskSetInterval(rrs->task_finder, (*run1)->task_set,
    &(*run1)->first_index, &(*run1)->last_index);
  for( i = half_way;  i < KheTaskSetTaskCount(run->task_set);  i++ )
    KheTaskSetAddTask((*run2)->task_set, KheTaskSetTask(run->task_set, i));
  KheTaskFinderTaskSetInterval(rrs->task_finder, (*run2)->task_set,
    &(*run2)->first_index, &(*run2)->last_index);
  if( (*run1)->last_index >= (*run2)->first_index )
  {
    KheRunDelete(*run1, rrs);
    KheRunDelete(*run2, rrs);
    return false;
  }
  else
  {
    KheRunDelete(run, rrs);
    return true;
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheRunMoveCheck(KHE_RUN run, KHE_RESOURCE r)                        */
/*                                                                           */
/*  Check moving run to r, returning true if it would succeed.               */
/*                                                                           */
/*  If the run is already assigned r, do nothing but count this as           */
/*  successful too.                                                          */
/*                                                                           */
/*****************************************************************************/

static bool KheRunMoveCheck(KHE_RUN run, KHE_RESOURCE r)                       
{
  KHE_TASK first_task;
  first_task = KheTaskSetFirst(run->task_set);
  if( KheTaskAsstResource(first_task) == r )
    return true;
  else
    return KheTaskSetMoveResourceCheck(run->task_set, r);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheRunMove(KHE_RUN run, KHE_RESOURCE r)                             */
/*                                                                           */
/*  Move run to r, returning true if successful.                             */
/*                                                                           */
/*  If the run is already assigned r, do nothing but count this as           */
/*  successful too.                                                          */
/*                                                                           */
/*****************************************************************************/

static bool KheRunMove(KHE_RUN run, KHE_RESOURCE r)                       
{
  KHE_TASK first_task;
  first_task = KheTaskSetFirst(run->task_set);
  if( KheTaskAsstResource(first_task) == r )
    return true;
  else
    return KheTaskSetMoveResource(run->task_set, r);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheRunCmp(const void *t1, const void *t2)                            */
/*                                                                           */
/*  Comparison function for sorting an array of runs into increasing order.  */
/*                                                                           */
/*****************************************************************************/

static int KheRunCmp(const void *t1, const void *t2)
{
  KHE_RUN run1 = * (KHE_RUN *) t1;
  KHE_RUN run2 = * (KHE_RUN *) t2;
  if( run1->first_index != run2->first_index )
    return run1->first_index - run2->first_index;
  else
    return run1->last_index - run2->last_index;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheRunDisjoint(KHE_RUN run1, KHE_RUN run2)                          */
/*                                                                           */
/*  Return true if run1 and run2 are disjoint in time.                       */
/*                                                                           */
/*****************************************************************************/

static bool KheRunDisjoint(KHE_RUN run1, KHE_RUN run2)
{
  return run1->last_index < run2->first_index ||
    run2->last_index < run1->first_index;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheRunOverlap(KHE_RUN run1, KHE_RUN run2)                           */
/*                                                                           */
/*  Return true if run1 and run2 are not disjoint in time.                   */
/*                                                                           */
/*****************************************************************************/

static bool KheRunOverlap(KHE_RUN run1, KHE_RUN run2)
{
  return !KheRunDisjoint(run1, run2);
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheResourceShow(KHE_RESOURCE r)                                    */
/*                                                                           */
/*  Return the Id of r, or "@" if r is NULL.                                 */
/*                                                                           */
/*****************************************************************************/

static char *KheResourceShow(KHE_RESOURCE r)
{
  if( r == NULL )
    return "@";
  else if( KheResourceId(r) == NULL )
    return "?";
  else
    return KheResourceId(r);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheRunDebug(KHE_RUN run, int verbosity, int indent, FILE *fp)       */
/*                                                                           */
/*  Debug print of run onto fp with the given verbosity and indent.          */
/*                                                                           */
/*****************************************************************************/

static void KheRunDebug(KHE_RUN run, int verbosity, int indent, FILE *fp)
{
  KHE_RUN_RESOURCE rr;  int i;
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  if( DEBUG4 )
    fprintf(fp, "%p", (void *) run);
  fprintf(fp, "(%02d-%02d)", run->first_index, run->last_index);
  KheTaskSetDebug(run->task_set, verbosity, -1, fp);
  if( verbosity >= 2 && HaArrayCount(run->domain) > 0 )
  {
    fprintf(fp, "[");
    HaArrayForEach(run->domain, rr, i)
      fprintf(fp, "%s%s", i > 0 ? ", " : "", KheResourceShow(rr->resource));
    fprintf(fp, "]");
  }
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "run resources"                                                */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_RUN_RESOURCE KheRunResourceMake(KHE_RESOURCE_RUN_SOLVER rrs,         */
/*    KHE_RESOURCE r)                                                        */
/*                                                                           */
/*  Make a run resource object with these attributes.                        */
/*                                                                           */
/*****************************************************************************/

static KHE_RUN_RESOURCE KheRunResourceMake(KHE_RESOURCE_RUN_SOLVER rrs,
  KHE_RESOURCE r)
{
  KHE_RUN_RESOURCE res;
  HaMake(res, rrs->arena);
  res->resource = r;
  res->first_index = -1;
  res->last_index = -1;
  HaArrayInit(res->runs, rrs->arena);
  res->state = RESOURCE_STATE_UNKNOWN;
  res->curr_last_index = -1;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheRunResourceSetRunsOutOfDate(KHE_RUN_RESOURCE rr)                 */
/*                                                                           */
/*  Mark rr's runs as being out of date.                                     */
/*                                                                           */
/*****************************************************************************/

static void KheRunResourceSetRunsOutOfDate(KHE_RUN_RESOURCE rr)
{
  rr->first_index = -1;
  rr->last_index = -1;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheRunResourceEnsureRunsUpToDate(KHE_RUN_RESOURCE rr,               */
/*    KHE_RESOURCE_RUN_SOLVER rrs, int first_index, int last_index)          */
/*                                                                           */
/*  Ensure that rr is up to date for solving over first_index..last_index.   */
/*  This is mainly about finding rr's resource's runs in that interval, but  */
/*  it also sets the defective field.                                        */
/*                                                                           */
/*  If the stored first and last indexes are equal to the new ones, it is    */
/*  assumed that rr is already up to date.                                   */
/*                                                                           */
/*****************************************************************************/
static void KheRunResourceDebug(KHE_RUN_RESOURCE rr, int verbosity,
  int indent, FILE *fp);

static void KheRunResourceEnsureRunsUpToDate(KHE_RUN_RESOURCE rr,
  KHE_RESOURCE_RUN_SOLVER rrs, int first_index, int last_index, int indent)
{
  int i;  KHE_RUN run;  KHE_RESOURCE_TYPE rt;
  if( DEBUG3 )
    fprintf(stderr,"%*s[ KheRunResourceEnsureRunsUpToDate(rr, rrs, %d, %d)%s\n",
      indent, "", first_index, last_index,
      rr->first_index != first_index || rr->last_index != last_index ?
      " changed" : " unchanged");

  if( rr->first_index != first_index || rr->last_index != last_index )
  {
    /* reset rr's interval */
    rr->first_index = first_index;
    rr->last_index = last_index;

    /* clear out any old runs */
    HaArrayForEach(rr->runs, run, i)
      KheRunDelete(run, rrs);
    HaArrayClear(rr->runs);

    /* add the new runs */
    rt = KheResourceResourceType(rr->resource);
    run = KheRunMake(rrs);
    while( KheFindTaskRunRight(rrs->task_finder, rt, rr->resource, false,
      first_index, run->task_set, &run->first_index, &run->last_index) &&
      run->last_index <= last_index )
    {
      HnAssert(run->first_index >= 0,
	"KheRunResourceEnsureRunsUpToDate internal error 1");
      HnAssert(run->last_index >= 0,
	"KheRunResourceEnsureRunsUpToDate internal error 2");
      HaArrayAddLast(rr->runs, run);
      first_index = run->last_index + 1;
      run = KheRunMake(rrs);
    }
    KheRunDelete(run, rrs);

    /* reset defective field */
    /* *** doing this separately now
    rr->defective = (KheSolnResourceCost(rrs->soln, rr->resource) > 0);
    *** */
  }
  if( DEBUG3 )
  {
    KheRunResourceDebug(rr, 2, indent + 2, stderr);
    fprintf(stderr, "%*s] KheRunResourceEnsureRunsUpToDate returning\n",
      indent, "");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheRunResourceSetStateOutOfDate(KHE_RUN_RESOURCE rr)                */
/*                                                                           */
/*  Mark rr's runs as being out of date.                                     */
/*                                                                           */
/*****************************************************************************/

static void KheRunResourceSetStateOutOfDate(KHE_RUN_RESOURCE rr)
{
  rr->state = RESOURCE_STATE_UNKNOWN;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheRunResourceEnsureStateUpToDate(KHE_RUN_RESOURCE rr,              */
/*    KHE_RESOURCE_RUN_SOLVER rrs)                                           */
/*                                                                           */
/*  Ensure rr's defective field is up to date.                               */
/*                                                                           */
/*****************************************************************************/

static void KheRunResourceEnsureStateUpToDate(KHE_RUN_RESOURCE rr,
  KHE_RESOURCE_RUN_SOLVER rrs)
{
  if( rr->state == RESOURCE_STATE_UNKNOWN )
    rr->state = (KheSolnResourceCost(rrs->soln, rr->resource) > 0 ?
      RESOURCE_STATE_DEFECTIVE : RESOURCE_STATE_NOT_DEFECTIVE);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheRunResourceDebug(KHE_RUN_RESOURCE rr, int verbosity,             */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of rr onto fp with the given verbosity and indent.           */
/*  The runs are only printed if indent >= 0.                                */
/*                                                                           */
/*****************************************************************************/

static void KheRunResourceDebug(KHE_RUN_RESOURCE rr, int verbosity,
  int indent, FILE *fp)
{
  KHE_RUN run;  int i;
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  if( rr->first_index >= 0 )
  {
    fprintf(fp, "[%s %s %d-%d:", KheResourceId(rr->resource),
      ResourceStateShow(rr->state), rr->first_index, rr->last_index);
    if( indent >= 0 )
    {
      fprintf(fp, "\n");
      HaArrayForEach(rr->runs, run, i)
	KheRunDebug(run, verbosity, indent + 2, fp);
      fprintf(fp, "%*s]", indent, "");
    }
    else
      fprintf(fp, " %d runs]", HaArrayCount(rr->runs));
  }
  else
    fprintf(fp, "[%s not up to date]", KheResourceId(rr->resource));
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KheResourceRunSolverSolve"                                    */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool KheSolnIsNewBest(KHE_RESOURCE_RUN_SOLVER rrs)                       */
/*                                                                           */
/*  Return true if the current solution is a new best.                       */
/*                                                                           */
/*****************************************************************************/

static bool KheSolnIsNewBest(KHE_RESOURCE_RUN_SOLVER rrs)
{
  if( KheMarkPathCount(rrs->init_mark) == 0 )
    return true;
  return KheSolnCost(rrs->soln) < KhePathSolnCost(KheMarkPath(rrs->init_mark,0))
    && (!rrs->resource_invariant ||
      KheSolnMatchingDefectCount(rrs->soln) <= rrs->init_defect_count);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheRunSolverDoSolve(KHE_RESOURCE_RUN_SOLVER rrs, int curr_index)    */
/*                                                                           */
/*  Solve from curr_index onwards.                                           */
/*                                                                           */
/*****************************************************************************/

static void KheRunSolverDoSolve(KHE_RESOURCE_RUN_SOLVER rrs, int curr_index)
{
  KHE_RUN run;  KHE_RUN_RESOURCE rr;  int i, save_curr_last_index;
  KHE_MARK mark;
  if( DEBUG5 && rrs->debug_on )
    fprintf(stderr, "%*s[ DoSolve(rrs, %d)\n", curr_index * 2, "",
      curr_index);
  if( rrs->curr_assignments < rrs->max_assignments )
  {
    if( curr_index >= HaArrayCount(rrs->movable_runs) )
    {
      /* time to stop; if new best, save path */
      if( DEBUG5 && rrs->debug_on )
	fprintf(stderr, "%*s  off end: %s\n", curr_index * 2, "",
	   KheSolnIsNewBest(rrs) ? "new best" : "no luck");
      if( KheSolnIsNewBest(rrs) )
	KheMarkAddBestPath(rrs->init_mark, 1);
      rrs->curr_assignments++;
    }
    else
    {
      /* try each possible assignment of the run at curr_index */
      run = HaArray(rrs->movable_runs, curr_index);
      if( DEBUG5 && rrs->debug_on )
      {
	fprintf(stderr, "%*s  [ assigning ", curr_index * 2, "");
	KheRunDebug(run, 1, -1, stderr);
	fprintf(stderr, ":\n");
      }
      HaArrayForEach(run->domain, rr, i)
	if( rr->curr_last_index < run->first_index )
	{
	  /* rr->resource is in the domain and assignment would not clash */
	  mark = KheMarkBegin(rrs->soln);
	  if( KheRunMove(run, rr->resource) )
	  {
	    save_curr_last_index = rr->curr_last_index;
            rr->curr_last_index = run->last_index;
            KheRunSolverDoSolve(rrs, curr_index + 1);
            rr->curr_last_index = save_curr_last_index;
	  }
	  KheMarkEnd(mark, true);
	}
      if( DEBUG5 && rrs->debug_on )
	fprintf(stderr, "%*s  ]\n", curr_index * 2, "");
    }
  }
  else
  {
    if( DEBUG5 && rrs->debug_on )
      fprintf(stderr, "%*s  limit %d reached\n", curr_index * 2, "",
	rrs->max_assignments);
  }
  if( DEBUG5 && rrs->debug_on )
    fprintf(stderr, "%*s]\n", curr_index * 2, "");
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheResourceRunSolverDoSolve(KHE_RESOURCE_RUN_SOLVER rrs,            */
/*    int fi, int li, int indent)                                            */
/*                                                                           */
/*  Carry out KheResourceRunSolverSolve, but with an indent paramter for     */
/*  use during debugging.                                                    */
/*                                                                           */
/*****************************************************************************/

static bool KheResourceRunSolverDoSolve(KHE_RESOURCE_RUN_SOLVER rrs,
  int fi, int li, int indent)
{
  int i, j, k, pos;  KHE_COST init_cost;
  KHE_RUN_RESOURCE rr, rr2, imm_run_rr;  KHE_RUN run, imm_run;
  if( DEBUG2 )
  {
    fprintf(stderr, "%*s[ KheResourceRunSolverDoSolve(rrs, %s..%s) [",
      indent, "",
      KheTimeGroupId(KheFrameTimeGroup(rrs->days_frame, fi)),
      KheTimeGroupId(KheFrameTimeGroup(rrs->days_frame, li)));
    HaArrayForEach(rrs->resources, rr, i)
      fprintf(stderr, "%s%s", i > 0 ? ", " : "", KheResourceShow(rr->resource));
    fprintf(stderr, "]\n");
  }

  /* ensure that the run resources are up to date */
  HaArrayForEach(rrs->resources, rr, i)
  {
    KheRunResourceEnsureRunsUpToDate(rr, rrs, fi, li, indent + 2);
    if( DEBUG9 )
      KheRunResourceDebug(rr, 2, indent + 2, stderr);
  }

  /* set the domains of the runs, and set rrs->movable_runs to all runs */
  if( DEBUG9 )
    fprintf(stderr, "  setting domains\n");
  HaArrayClear(rrs->movable_runs);
  HaArrayClear(rrs->immovable_runs);
  HaArrayForEach(rrs->resources, rr2, i)
    HaArrayForEach(rr2->runs, run, j)
    {
      HaArrayClear(run->domain);
      HaArrayForEach(rrs->resources, rr, k)
	if( KheRunMoveCheck(run, rr->resource) )
	  HaArrayAddLast(run->domain, rr);
      if( HaArrayCount(run->domain) <= 1 )
      {
	/* immovable run */
	HnAssert(HaArrayCount(run->domain) == 1 &&
	  HaArrayFirst(run->domain) == rr2,
	  "KheResourceRunSolverSolve internal error 1");
	HaArrayAddLast(rrs->immovable_runs, run);
      }
      else
      {
	/* movable run */
	HaArrayAddLast(rrs->movable_runs, run);
      }
    }

  /* sort the movable and immovable runs */
  if( DEBUG9 )
    fprintf(stderr, "  sorting runs\n");
  HaArraySort(rrs->movable_runs, &KheRunCmp);
  HaArraySort(rrs->immovable_runs, &KheRunCmp);

  /* reduce the domains of movable runs that overlap immovable runs */
  if( DEBUG9 )
    fprintf(stderr, "  reducing domains\n");
  i = 0;
  HaArrayForEach(rrs->immovable_runs, imm_run, j)
  {
    /************************************************************************/
    /*                                                                      */
    /*  invariant: movable[0 .. i-1] strictly precede immovable[j-1..$]     */
    /*                                                                      */
    /************************************************************************/
    if( i > HaArrayCount(rrs->movable_runs) )
      break;
    run = HaArray(rrs->movable_runs, i);
    if( run->last_index < imm_run->first_index )
      i++;
    else
    {
      imm_run_rr = HaArrayFirst(imm_run->domain);
      for( k = i;  k < HaArrayCount(rrs->movable_runs);  k++ )
      {
	run = HaArray(rrs->movable_runs, k);
	if( run->first_index > imm_run_rr->last_index )
	  break;
	if( KheRunOverlap(imm_run, run) )
	{
	  if( HaArrayContains(run->domain, imm_run_rr, &pos) )
	    HaArrayDeleteAndPlug(run->domain, pos);
	}
      }
    }
  }

  /* perform the actual solve, ending by redoing the best path */
  if( DEBUG9 )
    fprintf(stderr, "  starting actual solve\n");
  HaArrayForEach(rrs->resources, rr, i)
    rr->curr_last_index = -1;
  rrs->init_mark = KheMarkBegin(rrs->soln);
  if( rrs->resource_invariant )
    rrs->init_defect_count = KheSolnMatchingDefectCount(rrs->soln);
  else
    rrs->init_defect_count = -1;
  rrs->debug_on = false;
  rrs->curr_assignments = 0;
  init_cost = KheSolnCost(rrs->soln);
  KheRunSolverDoSolve(rrs, 0);

  if( KheMarkPathCount(rrs->init_mark) > 0 &&
      KhePathSolnCost(KheMarkPath(rrs->init_mark, 0)) < init_cost )
  {
    /* success; redo best path and mark the resources as out of date */
    if( DEBUG9 )
      fprintf(stderr, "  wrapup (success)\n");
    HnAssert(KheMarkPathCount(rrs->init_mark) == 1,
      "KheResourceRunSolverSolve internal error 1");
    KhePathRedo(KheMarkPath(rrs->init_mark, 0));
    KheMarkEnd(rrs->init_mark, false);
    HaArrayForEach(rrs->resources, rr, i)
    {
      KheRunResourceSetRunsOutOfDate(rr);
      KheRunResourceSetStateOutOfDate(rr);
    }
    if( DEBUG2 )
      fprintf(stderr, "%*s] KheResourceRunSolverDoSolve returning true "
	"(%.5f -> %.5f)\n", indent, "", KheCostShow(init_cost),
	KheCostShow(KheSolnCost(rrs->soln)));
    if( DEBUG6 )
      fprintf(stderr, "  KheResourceRunSolverDoSolve(rrs, %d, %d) returning"
	" true (%.5f -> %.5f)\n", fi, li, KheCostShow(init_cost),
	KheCostShow(KheSolnCost(rrs->soln)));
    return true;
  }
  else
  {
    /* failure */
    if( DEBUG9 )
      fprintf(stderr, "  wrapup (failure)\n");
    KheMarkEnd(rrs->init_mark, true);
    if( DEBUG2 )
      fprintf(stderr, "%*s] KheResourceRunSolverDoSolve returning false\n",
	indent, "");
    return false;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheResourceRunSolverSolve(KHE_RESOURCE_RUN_SOLVER rrs,              */
/*    int fi, int li)                                                        */
/*                                                                           */
/*  Solve, and return true if any assignments changed.  At the start, make   */
/*  sure the resources are up to date, and at the end, mark them out of      */
/*  date if any assignments changed.                                         */
/*                                                                           */
/*****************************************************************************/

bool KheResourceRunSolverSolve(KHE_RESOURCE_RUN_SOLVER rrs, int fi, int li)
{
  return KheResourceRunSolverDoSolve(rrs, fi, li, 0);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "public functions"                                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_RESOURCE_RUN_SOLVER KheResourceRunSolverMake(KHE_SOLN soln,          */
/*    KHE_OPTIONS options, bool resource_invariant, int max_assignments)     */
/*                                                                           */
/*  Make a new run solver with these attributes, using memory from an        */
/*  arena taken from soln.                                                   */
/*                                                                           */
/*****************************************************************************/

KHE_RESOURCE_RUN_SOLVER KheResourceRunSolverMake(KHE_SOLN soln,
  KHE_OPTIONS options, bool resource_invariant, int max_assignments)
{
  KHE_RESOURCE_RUN_SOLVER res;  HA_ARENA a;
  a = KheSolnArenaBegin(soln, false);
  HaMake(res, a);

  /* always defined */
  res->arena = a;
  res->soln = soln;
  res->days_frame = KheOptionsFrame(options, "gs_common_frame", soln);
  res->task_finder = KheTaskFinderMake(soln, options, a);
  res->resource_invariant = resource_invariant;
  res->max_assignments = max_assignments;
  HaArrayInit(res->dormant_resources, a);
  HaArrayInit(res->resources, a);
  HaArrayInit(res->free_runs, a);

  /* defined during solves (so undefined here) */
  HaArrayInit(res->movable_runs, a);
  HaArrayInit(res->immovable_runs, a);
  res->init_mark = NULL;
  res->init_defect_count = -1;
  res->debug_on = false;
  res->curr_assignments = -1;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheRunSolverDelete(KHE_RESOURCE_RUN_SOLVER rrs)                     */
/*                                                                           */
/*  Delete rrs.                                                               */
/*                                                                           */
/*****************************************************************************/

void KheRunSolverDelete(KHE_RESOURCE_RUN_SOLVER rrs)
{
  KheSolnArenaEnd(rrs->soln, rrs->arena);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheSolverContainsResource(KHE_RESOURCE_RUN_SOLVER rrs,              */
/*    KHE_RESOURCE r, int *pos)                                              */
/*                                                                           */
/*  If rrs contains r, return true with *pos set to its position, else       */
/*  return false.                                                            */
/*                                                                           */
/*****************************************************************************/

static bool KheSolverContainsResource(KHE_RESOURCE_RUN_SOLVER rrs,
  KHE_RESOURCE r, int *pos)
{
  KHE_RUN_RESOURCE rr;  int i;
  for( i = HaArrayCount(rrs->resources) - 1;  i >= 0;  i-- )
  {
    rr = HaArray(rrs->resources, i);
    if( rr->resource == r )
      return *pos = i, true;
  }
  return *pos = -1, false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheResourceRunSolverAddResource(KHE_RESOURCE_RUN_SOLVER rrs,        */
/*    KHE_RESOURCE r)                                                        */
/*                                                                           */
/*  Add r to rrs and return true, or do nothing and return false if r is     */
/*  already present.                                                         */
/*                                                                           */
/*  It is an invariant that every resource in rrs->resources has an          */
/*  up-to-date state, so if this function adds a resource it also brings     */
/*  its state up to date.                                                    */
/*                                                                           */
/*****************************************************************************/

bool KheResourceRunSolverAddResource(KHE_RESOURCE_RUN_SOLVER rrs,
  KHE_RESOURCE r)
{
  int pos, index;  KHE_RUN_RESOURCE rr;

  /* if already present, do nothing and return false */
  if( KheSolverContainsResource(rrs, r, &pos) )
    return false;

  /* get rr from dormant resources if present, else make it */
  index = KheResourceInstanceIndex(r);
  HaArrayFill(rrs->dormant_resources, index + 1, NULL);
  rr = HaArray(rrs->dormant_resources, index);
  if( rr == NULL )
  {
    rr = KheRunResourceMake(rrs, r);
    HaArrayPut(rrs->dormant_resources, index, rr);
  }

  /* ensure rr's state is up to date */
  KheRunResourceEnsureStateUpToDate(rr, rrs);

  /* add rr to resources and return true */
  HaArrayAddLast(rrs->resources, rr);
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheResourceRunSolverDeleteResource(KHE_RESOURCE_RUN_SOLVER rrs,     */
/*    KHE_RESOURCE r)                                                        */
/*                                                                           */
/*  Delete r from rrs and return true, or do nothing and return false if r   */
/*  not present.                                                             */
/*                                                                           */
/*****************************************************************************/

bool KheResourceRunSolverDeleteResource(KHE_RESOURCE_RUN_SOLVER rrs,
  KHE_RESOURCE r)
{
  int pos;

  /* if not already present, do nothing and return false */
  if( !KheSolverContainsResource(rrs, r, &pos) )
    return false;

  /* delete from resources array (order does not matter) and return true */
  HaArrayDeleteAndPlug(rrs->resources, pos);
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheResourceRunSolverClearResources(KHE_RESOURCE_RUN_SOLVER rrs)     */
/*                                                                           */
/*  Clear all the resources from rrs.                                        */
/*                                                                           */
/*****************************************************************************/

void KheResourceRunSolverClearResources(KHE_RESOURCE_RUN_SOLVER rrs)
{
  HaArrayClear(rrs->resources);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheRunSolverSolveAll(KHE_RESOURCE_RUN_SOLVER rrs,                   */
/*    int fi, int li, bool resource_invariant, int max_assignments)          */
/*                                                                           */
/*  Solve all sets of resources and return true if any succeeded.            */
/*                                                                           */
/*****************************************************************************/

/* *** merged into KheRunSolverSolveDefective now
static bool KheRunSolverSolveAll(KHE_RESOURCE_RUN_SOLVER rrs,
  int fi, int li, bool resource_invariant, int max_assignments)
{
  KHE_RUN_RESOURCE rr1, rr2;  int i, j;  bool res;
  res = false;
  for( i = 0;  i < HaArrayCount(rrs->resources);  i++ )
  {
    rr1 = HaArray(rrs->resources, i);
    for( j = i + 1;  j < HaArrayCount(rrs->resources);  j++ )
    {
      rr2 = HaArray(rrs->resources, j);
      if( KheResourceRunSolverSolve(rrs, fi, li, resource_invariant,
	    max_assignments) )
	res = true;
    }
  }
  return res;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheRunSolverSolveDefective(KHE_RESOURCE_RUN_SOLVER rrs,             */
/*    KHE_RESOURCE_TYPE rt, int first_avail, int resource_count,             */
/*    int min_defective, int fi, int li, int indent)                         */
/*                                                                           */
/*  Solve for all sets of resources that satisfy these conditions:           */
/*                                                                           */
/*    * The resources are all those currently in the solver plus             */
/*      resource_count extra resources.                                      */
/*                                                                           */
/*    * The extra resources all come from rt[first_avail ..].                */
/*                                                                           */
/*    * There must be at least min_defective defective resources among       */
/*      the extra resources.                                                 */
/*                                                                           */
/*  Return true if the solution is improved.                                 */
/*                                                                           */
/*****************************************************************************/

static bool KheRunSolverSolveDefective(KHE_RESOURCE_RUN_SOLVER rrs,
  KHE_RESOURCE_TYPE rt, int first_avail, int resource_count,
  int min_defective, int fi, int li, int indent)
{
  KHE_RESOURCE r;  KHE_RUN_RESOURCE rr;  bool res;  int i;
  if( DEBUG2 )
    fprintf(stderr,
      "%*s[ SolveDefective(rrs, %s, fa %d, rc %d, md %d, %d..%d)\n",
      indent, "", KheResourceTypeId(rt), first_avail, resource_count,
      min_defective, fi,li);

  if( min_defective > resource_count )
  {
    /* no prospect of finding enough defective extra resources */
    res = false;
  }
  else if( resource_count == 0 )
  {
    /* resources are loaded, ready to solve */
    res = KheResourceRunSolverDoSolve(rrs, fi, li, indent + 2);
  }
  else
  {
    res = false;
    for( i = first_avail;  i < KheResourceTypeResourceCount(rt);  i++ )
    {
      r = KheResourceTypeResource(rt, i);
      KheResourceRunSolverAddResource(rrs, r);
      rr = HaArrayLast(rrs->resources);
      if( KheRunSolverSolveDefective(rrs, rt, i + 1, resource_count - 1,
	  rr->state==RESOURCE_STATE_DEFECTIVE ? min_defective-1 : min_defective,
	  fi, li, indent + 2) )
	res = true;
      KheResourceRunSolverDeleteResource(rrs, r);
    }
  }
  if( DEBUG2 )
    fprintf(stderr, "%*s] SolveDefective returning %s\n", indent, "",
      bool_show(res));
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheRunSolverSolveAdjacent(KHE_RESOURCE_RUN_SOLVER rrs,              */
/*    int fi, int li, bool resource_invariant, int max_assignments)          */
/*                                                                           */
/*  Solve adjacent sets of resources and return true if any succeeded.       */
/*                                                                           */
/*****************************************************************************/

static bool KheRunSolverSolveAdjacent(KHE_RESOURCE_RUN_SOLVER rrs,
  KHE_RESOURCE_TYPE rt, int resource_count, int fi, int li)
{
  int i, j, last_start;  bool res;  KHE_RESOURCE r;
  res = false;
  last_start = KheResourceTypeResourceCount(rt) - resource_count;
  for( i = 0;  i <= last_start;  i++ )
  {
    /* add resource_count resources to rrs */
    for( j = 0;  j < resource_count;  j++ )
    {
      r = KheResourceTypeResource(rt, i + j);
      KheResourceRunSolverAddResource(rrs, r);
    }

    /* solve */
    if( KheResourceRunSolverDoSolve(rrs, fi, li, 2) )
      res = true;

    /* clear the resources */
    KheResourceRunSolverClearResources(rrs);
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheRunSolverSolveForResources(KHE_RESOURCE_RUN_SOLVER rrs,          */
/*    KHE_RESOURCE_TYPE rt, int run_resources, RUN_SELECT_TYPE run_select,  */
/*    int run_parts, int run_start, int run_increment)                       */
/*                                                                           */
/*  Solve with these parameters.  Return true if any solves succeeded.       */
/*                                                                           */
/*****************************************************************************/

static bool KheRunSolverSolveForResources(KHE_RESOURCE_RUN_SOLVER rrs,
  KHE_RESOURCE_TYPE rt, int run_resources, RUN_SELECT_TYPE run_select,
  int run_parts, int run_start, int run_increment)
{
  bool res;  int fi, li, last_index;
  if( DEBUG2 )
    fprintf(stderr,
      "[ SolveForResources(rrs, %s, %d, sel %s, parts %d, start %d, inc %d)\n",
      KheResourceTypeId(rt), run_resources, KheRunSelectTypeShow(run_select),
      run_parts, run_start, run_increment);
  res = false;
  last_index = KheTaskFinderLastIndex(rrs->task_finder);
  fi = run_start, li = fi + run_parts - 1;
  for( ;  li <= last_index;  fi += run_increment, li += run_increment )
  {
    switch( run_select )
    {
      case RUN_SELECT_DEFECTIVE:

	if( KheRunSolverSolveDefective(rrs, rt, 0, run_resources, 1, fi,li,2) )
	  res = true;
	break;

      case RUN_SELECT_ADJACENT:

	if( KheRunSolverSolveAdjacent(rrs, rt, run_resources, fi, li) )
	  res = true;
	break;

      case RUN_SELECT_ALL:

	if( KheRunSolverSolveDefective(rrs, rt, 0, run_resources, 0, fi,li,2) )
	  res = true;
	break;

      case RUN_SELECT_NONE:
      default:

	HnAbort("KheRunSolverSolveForResources internal error");
	break;
    }
  }
  if( DEBUG2 )
    fprintf(stderr, "] SolveForResources returning %s\n", bool_show(res));
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheResourceRunRepair(KHE_SOLN soln, KHE_RESOURCE_TYPE rt,           */
/*    KHE_OPTIONS options)                                                   */
/*                                                                           */
/*  Call KheResourceRunSolverSolve repeatedly on the resources of type rt,   */
/*  depending on options.                                                    */
/*                                                                           */
/*****************************************************************************/

bool KheResourceRunRepair(KHE_SOLN soln, KHE_RESOURCE_TYPE rt,
  KHE_OPTIONS options)
{
  KHE_RESOURCE_RUN_SOLVER rrs;  bool run_off;  KHE_INSTANCE ins;
  int run_resources, run_parts, run_start, run_increment;
  RUN_SELECT_TYPE run_select;  KHE_COST init_cost;
  KHE_RESOURCE_TIMETABLE_MONITOR rtm;  KHE_RESOURCE r;

  /* quit now if rs_run_off, or rs_run_select is "none" */
  run_off = KheOptionsGetBool(options, "rs_run_off", false);
  run_select = KheSelectTypeGet(options);
  if( run_off || run_select == RUN_SELECT_NONE )
    return false;

  ins = KheSolnInstance(soln);
  if( DEBUG1 )
    fprintf(stderr, "[ KheResourceRunRepair(soln of %s, options)\n",
      KheInstanceId(ins));
  if( DEBUG1 && DEBUG8 && KheInstanceRetrieveResource(ins, DEBUG8_ID, &r) )
  {
    rtm = KheResourceTimetableMonitor(soln, r);
    KheResourceTimetableMonitorPrintTimetable(rtm, 10, 2, stderr);
  }

  /* quit now if the time limit has been reached */
  if( KheOptionsTimeLimitReached(options) )
  {
    if( DEBUG1 )
      fprintf(stderr, "] KheResourceRunRepair ret. false (time limit)\n");
    return false;
  }

  /* make a solver and get the other options */
  rrs = KheResourceRunSolverMake(soln, options,
    KheOptionsGetBool(options, "rs_invariant", false),
    KheOptionsGetInt(options, "rs_run_max", 1000000));
  run_resources = KheOptionsGetInt(options, "rs_run_resources", 2);
  run_parts = KheOptionsGetInt(options, "rs_run_parts", 28);
  run_start = KheOptionsGetInt(options, "rs_run_start", 0);
  run_increment = KheOptionsGetInt(options, "rs_run_increment", run_parts);
  init_cost = KheSolnCost(soln);

  /* solve for each resource type */
  /* *** just solving for one resource type now 
  res = false;
  for( i = 0;  i < KheInstanceResourceTypeCount(ins);  i++ )
  {
    rt = KheInstanceResourceType(ins, i);
    if( KheRunSolverSolveForResources(rrs, rt, resource_count,
	  run_select, run_parts, run_start, run_increment) )
      res = true;
  }
  *** */

  /* solve for rt, debug result and return */
  if( KheRunSolverSolveForResources(rrs, rt, run_resources,
	run_select, run_parts, run_start, run_increment) )
  {
    if( DEBUG1 )
      fprintf(stderr, "] KheResourceRunRepair returning true (%.5f -> "
	"%.5f)\n", KheCostShow(init_cost), KheCostShow(KheSolnCost(soln)));
    return true;
  }
  else
  {
    if( DEBUG1 )
      fprintf(stderr, "] KheResourceRunRepair returning false\n");
    return false;
  }
}
