
/*****************************************************************************/
/*                                                                           */
/*  THE KHE HIGH SCHOOL TIMETABLING ENGINE                                   */
/*  COPYRIGHT (C) 2010 Jeffrey H. Kingston                                   */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 3, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         khe_sr_single_resource.c                                   */
/*  DESCRIPTION:  Single resource solving                                    */
/*                                                                           */
/*****************************************************************************/
#include "khe_solvers.h"
#include "howard_p.h"

#define DEBUG1 1

/*****************************************************************************/
/*                                                                           */
/*  KHE_MONITOR_INFO - one monitor on one day                                */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_monitor_info_rec {
  KHE_MONITOR		monitor;		/* monitor this is about     */
  HA_ARRAY_INT		indexes;		/* optional signature indexes*/
} *KHE_MONITOR_INFO;

typedef HA_ARRAY(KHE_MONITOR_INFO) ARRAY_KHE_MONITOR_INFO;


/*****************************************************************************/
/*                                                                           */
/*  KHE_BEST_TASK - the best task with the given first time and last time    */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_best_task_rec {
  KHE_TASK			task;			/* the task          */
  KHE_TIME			last_time;		/* its last time     */
  int				last_index;		/* day index of l.t. */
  KHE_COST			cost_reduction;		/* its cost reduction*/
  int				domain_count;		/* its domain size   */
} *KHE_BEST_TASK;

typedef HA_ARRAY(KHE_BEST_TASK) ARRAY_KHE_BEST_TASK;


/*****************************************************************************/
/*                                                                           */
/*  KHE_BEST_TASKS_AT_TIME - the best tasks with a given first time          */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_best_tasks_at_time_rec {
  KHE_TIME			first_time;
  ARRAY_KHE_BEST_TASK		best_tasks;
} *KHE_BEST_TASKS_AT_TIME;

typedef HA_ARRAY(KHE_BEST_TASKS_AT_TIME) ARRAY_KHE_BEST_TASKS_AT_TIME;


/*****************************************************************************/
/*                                                                           */
/*  KHE_DAY - one day and everything about it                                */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_partial_soln_rec *KHE_PARTIAL_SOLN;

typedef HP_TABLE(KHE_PARTIAL_SOLN) TABLE_KHE_PARTIAL_SOLN;
typedef HA_ARRAY(KHE_PARTIAL_SOLN) ARRAY_KHE_PARTIAL_SOLN;

typedef struct khe_day_rec {
  KHE_TIME_GROUP		time_group;	/* the day's time group      */
  ARRAY_KHE_MONITOR_INFO	monitor_info;	/* the day's monitor info    */
  TABLE_KHE_PARTIAL_SOLN	partial_solns;	/* solns up to this day      */
  ARRAY_KHE_BEST_TASKS_AT_TIME	best_tasks_at_time;
} *KHE_DAY;

typedef HA_ARRAY(KHE_DAY) ARRAY_KHE_DAY;


/*****************************************************************************/
/*                                                                           */
/*  KHE_PARTIAL_SOLN - a partial solution                                    */
/*                                                                           */
/*****************************************************************************/

struct khe_partial_soln_rec {
  KHE_PARTIAL_SOLN	prev;			/* previous partial soln     */
  KHE_TASK		task;			/* asst leading to here      */
  int			best_of;		/* how many this is best of  */
  int			asst_count;		/* total task duration       */
  KHE_COST		r_cost;			/* cost of r's monitors      */
  KHE_COST		soln_cost;		/* cost of soln              */
  HA_ARRAY_INT		signature;		/* signature of this node    */
};


/*****************************************************************************/
/*                                                                           */
/*  KHE_SINGLE_RESOURCE_SOLVER - a single resource solver                    */
/*                                                                           */
/*****************************************************************************/

struct khe_single_resource_solver_rec {
  HA_ARENA			arena;
  KHE_SOLN			soln;
  KHE_OPTIONS			options;
  int				diversifier;
  KHE_FRAME			days_frame;
  KHE_EVENT_TIMETABLE_MONITOR	etm;
  KHE_TASK_FINDER		task_finder;
  KHE_RESOURCE			resource;	/* resource (if solving)     */
  KHE_GROUP_MONITOR		resource_gm;	/* its gm (if solving)       */
  ARRAY_KHE_DAY			days;
  ARRAY_KHE_PARTIAL_SOLN	final_solns;
  ARRAY_KHE_PARTIAL_SOLN	all_partial_solns;

  ARRAY_KHE_MONITOR_INFO	free_monitor_info;
  ARRAY_KHE_PARTIAL_SOLN	free_partial_solns;
  ARRAY_KHE_BEST_TASKS_AT_TIME	free_best_tasks_at_time;
  ARRAY_KHE_BEST_TASK		free_best_tasks;
};


/*****************************************************************************/
/*                                                                           */
/*  Submodule "monitor info"                                                 */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_MONITOR_INFO KheMonitorInfoMake(KHE_MONITOR m,                       */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Make a new monitor info object with these attributes.                    */
/*                                                                           */
/*****************************************************************************/

static KHE_MONITOR_INFO KheMonitorInfoMake(KHE_MONITOR m,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_MONITOR_INFO res;
  if( HaArrayCount(srs->free_monitor_info) > 0 )
  {
    res = HaArrayLastAndDelete(srs->free_monitor_info);
    HaArrayClear(res->indexes);
  }
  else
  {
    HaMake(res, srs->arena);
    HaArrayInit(res->indexes, srs->arena);
  }
  res->monitor = m;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheMonitorShowTag(KHE_MONITOR m)                                   */
/*                                                                           */
/*  Briefly show the tag of m.                                               */
/*                                                                           */
/*****************************************************************************/

static char *KheMonitorShowTag(KHE_MONITOR m)
{
  switch( KheMonitorTag(m) )
  {
    case KHE_AVOID_CLASHES_MONITOR_TAG:			return "clashes";
    case KHE_AVOID_UNAVAILABLE_TIMES_MONITOR_TAG:	return "unavail";
    case KHE_LIMIT_IDLE_TIMES_MONITOR_TAG:		return "idle";
    case KHE_CLUSTER_BUSY_TIMES_MONITOR_TAG:		return "cluster";
    case KHE_LIMIT_BUSY_TIMES_MONITOR_TAG:		return "limitbusy";
    case KHE_LIMIT_WORKLOAD_MONITOR_TAG:		return "workload";
    case KHE_LIMIT_ACTIVE_INTERVALS_MONITOR_TAG:	return "intervals";
    default:						return "?????????";
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheMonitorInfoDebug(KHE_MONITOR_INFO mi, int verbosity,             */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of mi onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void KheMonitorInfoDebug(KHE_MONITOR_INFO mi, int verbosity,
  int indent, FILE *fp)
{
  int i, index;
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  fprintf(fp, "%-9s %s", KheMonitorShowTag(mi->monitor),
    KheMonitorId(mi->monitor));
  if( HaArrayCount(mi->indexes) > 0 )
  {
    fprintf(fp, " [");
    HaArrayForEach(mi->indexes, index, i)
      fprintf(fp, "%s%d", i > 0 ? ", " : "", index);
    fprintf(fp, "]");
  }
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "best task"                                                    */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheBestTaskSet(KHE_BEST_TASK bt, KHE_TASK task, KHE_TIME last_time, */
/*    int last_index, KHE_COST cost_reduction, int domain_count)             */
/*                                                                           */
/*  Set bt's fields to these values.                                         */
/*                                                                           */
/*****************************************************************************/

static void KheBestTaskSet(KHE_BEST_TASK bt, KHE_TASK task, KHE_TIME last_time,
  int last_index, KHE_COST cost_reduction, int domain_count)
{
  bt->task = task;
  bt->last_time = last_time;
  bt->last_index = last_index;
  bt->cost_reduction = cost_reduction;
  bt->domain_count = domain_count;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_BEST_TASK KheBestTaskMake(KHE_TASK task, KHE_TIME last_time,         */
/*    int last_index, KHE_COST cost_reduction, int domain_count,             */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Make a new best task object with these attributes.                       */
/*                                                                           */
/*****************************************************************************/

static KHE_BEST_TASK KheBestTaskMake(KHE_TASK task, KHE_TIME last_time,
  int last_index, KHE_COST cost_reduction, int domain_count,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_BEST_TASK res;
  if( HaArrayCount(srs->free_best_tasks) > 0 )
    res = HaArrayLastAndDelete(srs->free_best_tasks);
  else
    HaMake(res, srs->arena);
  KheBestTaskSet(res, task, last_time, last_index, cost_reduction,
    domain_count);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheBestTaskDebug(KHE_BEST_TASK bt, int verbosity,                   */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of bt onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void KheBestTaskDebug(KHE_BEST_TASK bt, int verbosity,
  int indent, FILE *fp)
{
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  fprintf(fp, "(last %s, cost %.5f, domain %d) ", KheTimeId(bt->last_time),
    KheCostShow(bt->cost_reduction), bt->domain_count);
  KheTaskDebug(bt->task, verbosity, -1, fp);
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "best tasks at time"                                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_BEST_TASKS_AT_TIME KheBestTasksAtTimeMake(KHE_TIME first_time,       */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Return a new best tasks at time object with these attributes.            */
/*                                                                           */
/*****************************************************************************/

static KHE_BEST_TASKS_AT_TIME KheBestTasksAtTimeMake(KHE_TIME first_time,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_BEST_TASKS_AT_TIME res;
  if( HaArrayCount(srs->free_best_tasks_at_time) > 0 )
  {
    res = HaArrayLastAndDelete(srs->free_best_tasks_at_time);
    HaArrayClear(res->best_tasks);
  }
  else
  {
    HaMake(res, srs->arena);
    HaArrayInit(res->best_tasks, srs->arena);
  }
  res->first_time = first_time;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheBestTasksAtTimeContainsLastTime(KHE_BEST_TASKS_AT_TIME btt,      */
/*    KHE_TIME last_time, KHE_BEST_TASK *bt)                                 */
/*                                                                           */
/*  If btt contains a best time with the given last_time, return true with   */
/*  *bt set to that best time.  Otherwise return false with *bt set to NULL. */
/*                                                                           */
/*****************************************************************************/

static bool KheBestTasksAtTimeContainsLastTime(KHE_BEST_TASKS_AT_TIME btt,
  KHE_TIME last_time, KHE_BEST_TASK *bt)
{
  KHE_BEST_TASK bt2;  int i;
  HaArrayForEach(btt->best_tasks, bt2, i)
    if( bt2->last_time == last_time )
      return *bt = bt2, true;
  return *bt = NULL, false;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_BEST_TASK KheBestTasksAtTimeMakeAndAddBestTask(                      */
/*    KHE_BEST_TASKS_AT_TIME btt, KHE_TASK task, KHE_TIME last_time,         */
/*    int last_index, KHE_COST cr, int dc, KHE_SINGLE_RESOURCE_SOLVER srs)   */
/*                                                                           */
/*  Make, add to btt, and return a new best task object with these           */
/*  attributes.                                                              */
/*                                                                           */
/*****************************************************************************/

static KHE_BEST_TASK KheBestTasksAtTimeMakeAndAddBestTask(
  KHE_BEST_TASKS_AT_TIME btt, KHE_TASK task, KHE_TIME last_time,
  int last_index, KHE_COST cr, int dc, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_BEST_TASK res;
  res = KheBestTaskMake(task, last_time, last_index, cr, dc, srs);
  HaArrayAddLast(btt->best_tasks, res);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheBestTasksAtTimeDelete(KHE_BEST_TASKS_AT_TIME btt,                */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Delete btt, placing its objects onto free lists in srs.                  */
/*                                                                           */
/*****************************************************************************/

static void KheBestTasksAtTimeDelete(KHE_BEST_TASKS_AT_TIME btt,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  int i;
  HaArrayAppend(srs->free_best_tasks, btt->best_tasks, i);
  HaArrayAddLast(srs->free_best_tasks_at_time, btt);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheBestTasksAtTimeDebug(KHE_BEST_TASKS_AT_TIME btt, int verbosity,  */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of btt onto fp with the given verbosity and indent.          */
/*                                                                           */
/*****************************************************************************/

static void KheBestTasksAtTimeDebug(KHE_BEST_TASKS_AT_TIME btt, int verbosity,
  int indent, FILE *fp)
{
  KHE_BEST_TASK bt;  int i;
  fprintf(fp, "%*s[ Best tasks at %s:\n", indent, "",
    KheTimeId(btt->first_time));
  HaArrayForEach(btt->best_tasks, bt, i)
    KheBestTaskDebug(bt, verbosity, indent, fp);
  fprintf(fp, "%*s]\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "days"                                                         */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DAY KheDayMake(KHE_TIME_GROUP tg, HA_ARENA a)                        */
/*                                                                           */
/*  Make a new day object for a day with times tg.                           */
/*                                                                           */
/*****************************************************************************/
static int KhePartialSolnSignatureHashUntyped(void *p);
static bool KhePartialSolnSignaturesEqualUntyped(void *p1, void *p2);

static KHE_DAY KheDayMake(KHE_TIME_GROUP tg, HA_ARENA a)
{
  KHE_DAY res;
  HaMake(res, a);
  res->time_group = tg;
  HaArrayInit(res->monitor_info, a);
  HpTableInit(res->partial_solns, &KhePartialSolnSignatureHashUntyped,
    &KhePartialSolnSignaturesEqualUntyped, a);
  HaArrayInit(res->best_tasks_at_time, a);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDayClearBestTasksAtTime(KHE_DAY day,                             */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Clear day's best tasks at time array.                                    */
/*                                                                           */
/*****************************************************************************/

static void KheDayClearBestTasksAtTime(KHE_DAY day,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  int i;  KHE_BEST_TASKS_AT_TIME btt;
  HaArrayForEach(day->best_tasks_at_time, btt, i)
    KheBestTasksAtTimeDelete(btt, srs);
  HaArrayClear(day->best_tasks_at_time);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDayClear(KHE_DAY day, KHE_SINGLE_RESOURCE_SOLVER srs)            */
/*                                                                           */
/*  Clear day, ready for a fresh solve.                                      */
/*                                                                           */
/*****************************************************************************/

static void KheDayClear(KHE_DAY day, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  int i, pos;  KHE_PARTIAL_SOLN ps;
  HaArrayAppend(srs->free_monitor_info, day->monitor_info, i);
  HaArrayClear(srs->free_monitor_info);
  HpTableForEachValue(day->partial_solns, ps, pos)
    HaArrayAddLast(srs->free_partial_solns, ps);
  HpTableClear(day->partial_solns);
  KheDayClearBestTasksAtTime(day, srs);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_BEST_TASKS_AT_TIME KheDayMakeAndAddBestTasksAtTime(KHE_DAY day,      */
/*    KHE_TIME t, KHE_SINGLE_RESOURCE_SOLVER srs)                            */
/*                                                                           */
/*  Make, add to day, and return a new best tasks at times object with       */
/*  time t.                                                                  */
/*                                                                           */
/*****************************************************************************/

static KHE_BEST_TASKS_AT_TIME KheDayMakeAndAddBestTasksAtTime(KHE_DAY day,
  KHE_TIME t, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_BEST_TASKS_AT_TIME res;
  res = KheBestTasksAtTimeMake(t, srs);
  HaArrayAddLast(day->best_tasks_at_time, res);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDayAddTasks(KHE_DAY day, KHE_SINGLE_RESOURCE_SOLVER srs)         */
/*                                                                           */
/*  Add suitable tasks to day.                                               */
/*                                                                           */
/*****************************************************************************/

static void KheDayAddTasks(KHE_DAY day, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  int i, j, k, first_index, last_index, dc;  KHE_TIME t;  KHE_TASK task;
  KHE_BEST_TASKS_AT_TIME btt;  KHE_MEET meet;  KHE_TIME first_time, last_time;
  KHE_BEST_TASK bt;  KHE_COST cr;
  for( i = 0;  i < KheTimeGroupTimeCount(day->time_group);  i++ )
  {
    /* for each time of day */
    t = KheTimeGroupTime(day->time_group, i);
    btt = NULL;
    for( j = 0;  j < KheEventTimetableMonitorTimeMeetCount(srs->etm, t);  j++ )
    {
      /* for each meet running at time */
      meet = KheEventTimetableMonitorTimeMeet(srs->etm, t, j);
      for( k = 0;  k < KheMeetTaskCount(meet);  k++ )
      {
	/* for the proper root of each task of meet */
	task = KheTaskProperRoot(KheMeetTask(meet, k));
	KheTaskFinderTaskIntervalAndTimes(srs->task_finder, task,
	  &first_index, &last_index, &first_time, &last_time);
	if( first_time == t )
	{
	  /* the task begins at this time of day */
	  cr = KheTaskAssignmentCostReduction(task);
	  dc = KheResourceGroupResourceCount(KheTaskDomain(task));
	  if( KheTaskAsstResource(task) == srs->resource )
	  {
	    /* task is already assigned, no other choices */
	    KheDayClearBestTasksAtTime(day, srs);
            btt = KheDayMakeAndAddBestTasksAtTime(day, first_time, srs);
            KheBestTasksAtTimeMakeAndAddBestTask(btt, task, last_time,
	      last_index, cr, dc, srs);
	    return;
	  }
	  else
	  {
	    /* make a new btt object if none available yet for this time */
	    if( btt == NULL )
	      btt = KheDayMakeAndAddBestTasksAtTime(day, first_time, srs);

	    /* search for the same last_time in btt */
	    if( !KheBestTasksAtTimeContainsLastTime(btt, last_time, &bt) )
	    {
	      /* nothing comparable to task, add it to best_tasks */
              KheBestTasksAtTimeMakeAndAddBestTask(btt, task, last_time,
	        last_index, cr, dc, srs);
	    }
	    else if( cr > bt->cost_reduction ||
	      (cr == bt->cost_reduction && dc < bt->domain_count) )
	    {
	      /* task is better than bt->task, so overwrite bt */
	      KheBestTaskSet(bt, task, last_time, last_index, cr, dc);
	    }
	  }
	}
      }
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDayDebug(KHE_DAY day, int verbosity, int indent, FILE *fp)       */
/*                                                                           */
/*  Debug print of day onto fp.                                              */
/*                                                                           */
/*****************************************************************************/

static void KheDayDebug(KHE_DAY day, int verbosity, int indent, FILE *fp)
{
  int i;  KHE_MONITOR_INFO mi;  KHE_BEST_TASKS_AT_TIME btt;
  if( indent >= 0 )
  {
    fprintf(fp, "%*s[ ", indent, "");
    KheTimeGroupDebug(day->time_group, 1, 0, stderr);
    HaArrayForEach(day->monitor_info, mi, i)
      KheMonitorInfoDebug(mi, verbosity, indent + 2, fp);
    HaArrayForEach(day->best_tasks_at_time, btt, i)
      KheBestTasksAtTimeDebug(btt, verbosity, indent + 2, fp);
    fprintf(fp, "%*s]\n", indent, "");
  }
  else
    KheTimeGroupDebug(day->time_group, 1, -1, stderr);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "partial solutions"                                            */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KhePartialSolnAddAvoidClashesSignature(KHE_PARTIAL_SOLN ps,         */
/*    KHE_MONITOR_INFO mi, KHE_AVOID_CLASHES_MONITOR m)                      */
/*                                                                           */
/*  Add a summary of the current state of m to ps's signature.               */
/*                                                                           */
/*****************************************************************************/

static void KhePartialSolnAddAvoidClashesSignature(KHE_PARTIAL_SOLN ps,
  KHE_MONITOR_INFO mi, KHE_AVOID_CLASHES_MONITOR m)
{
  /* nothing to do here, in fact should never happen */
  HnAbort("KhePartialSolnAddAvoidClashesSignature internal error");
}


/*****************************************************************************/
/*                                                                           */
/*  void KhePartialSolnAddAvoidUnavailableTimesSignature(KHE_PARTIAL_SOLN ps,*/
/*    KHE_MONITOR_INFO mi, KHE_AVOID_UNAVAILABLE_TIMES_MONITOR m)            */
/*                                                                           */
/*  Add a summary of the current state of m to ps's signature.               */
/*                                                                           */
/*****************************************************************************/

static void KhePartialSolnAddAvoidUnavailableTimesSignature(KHE_PARTIAL_SOLN ps,
  KHE_MONITOR_INFO mi, KHE_AVOID_UNAVAILABLE_TIMES_MONITOR m)
{
  /* nothing to do here, in fact should never happen */
  HnAbort("KhePartialSolnAddAvoidUnavailableTimesSignature internal error");
}


/*****************************************************************************/
/*                                                                           */
/*  void KhePartialSolnAddLimitIdleTimesSignature(KHE_PARTIAL_SOLN ps,       */
/*    KHE_MONITOR_INFO mi, KHE_LIMIT_IDLE_TIMES_MONITOR m)                   */
/*                                                                           */
/*  Add a summary of the current state of m to ps's signature.               */
/*                                                                           */
/*****************************************************************************/

static void KhePartialSolnAddLimitIdleTimesSignature(KHE_PARTIAL_SOLN ps,
  KHE_MONITOR_INFO mi, KHE_LIMIT_IDLE_TIMES_MONITOR m)
{
  /* nothing to do here, in fact should never happen */
  HnAbort("KhePartialSolnAddLimitIdleTimesSignature internal error");
}


/*****************************************************************************/
/*                                                                           */
/*  void KhePartialSolnAddClusterBusyTimesSignature(KHE_PARTIAL_SOLN ps,     */
/*    KHE_MONITOR_INFO mi, KHE_CLUSTER_BUSY_TIMES_MONITOR m)                 */
/*                                                                           */
/*  Add a summary of the current state of m to ps's signature.               */
/*                                                                           */
/*****************************************************************************/

static void KhePartialSolnAddClusterBusyTimesSignature(KHE_PARTIAL_SOLN ps,
  KHE_MONITOR_INFO mi, KHE_CLUSTER_BUSY_TIMES_MONITOR m)
{
  int active_group_count, open_group_count, minimum, maximum, index, i, val;
  bool allow_zero;  KHE_TIME_GROUP tg;  KHE_POLARITY po;  int busy_count;

  /* add the number of active time groups to the signature */
  KheClusterBusyTimesMonitorActiveTimeGroupCount(m, &active_group_count,
    &open_group_count, &minimum, &maximum, &allow_zero);
  HaArrayAddLast(ps->signature, active_group_count);

  /* add one Boolean (0 or 1) for each indexed time group */
  HaArrayForEach(mi->indexes, index, i)
  {
    val = (KheClusterBusyTimesMonitorTimeGroupIsActive(m, index, &tg, &po,
      &busy_count) ? 1 : 0);
    HaArrayAddLast(ps->signature, val);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KhePartialSolnAddLimitBusyTimesSignature(KHE_PARTIAL_SOLN ps,       */
/*    KHE_MONITOR_INFO mi, KHE_LIMIT_BUSY_TIMES_MONITOR m)                   */
/*                                                                           */
/*  Add a summary of the current state of m to ps's signature.               */
/*                                                                           */
/*****************************************************************************/

static void KhePartialSolnAddLimitBusyTimesSignature(KHE_PARTIAL_SOLN ps,
  KHE_MONITOR_INFO mi, KHE_LIMIT_BUSY_TIMES_MONITOR m)
{
  int index, i, busy_count;

  /* add the deviation to the signature */
  HaArrayAddLast(ps->signature, KheLimitBusyTimesMonitorDeviation(m));

  /* add the busy count of each indexed time group to the signature */
  HaArrayForEach(mi->indexes, index, i)
  {
    KheLimitBusyTimesMonitorTimeGroup(m, index, &busy_count);
    HaArrayAddLast(ps->signature, busy_count);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KhePartialSolnAddLimitWorkloadSignature(KHE_PARTIAL_SOLN ps,        */
/*    KHE_MONITOR_INFO mi, KHE_LIMIT_WORKLOAD_MONITOR m)                     */
/*                                                                           */
/*  Add a summary of the current state of m to ps's signature.               */
/*                                                                           */
/*****************************************************************************/

static void KhePartialSolnAddLimitWorkloadSignature(KHE_PARTIAL_SOLN ps,
  KHE_MONITOR_INFO mi, KHE_LIMIT_WORKLOAD_MONITOR m)
{
  int index, i;  float workload;

  /* add the deviation to the signature */
  HaArrayAddLast(ps->signature, KheLimitWorkloadMonitorDeviation(m));

  /* add the approximate busy count of each indexed time group to signature */
  HaArrayForEach(mi->indexes, index, i)
  {
    KheLimitWorkloadMonitorTimeGroup(m, index, &workload);
    HaArrayAddLast(ps->signature, (int) (workload * 10.0));
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KhePartialSolnAddLimitActiveIntervalsSignature(KHE_PARTIAL_SOLN ps, */
/*    KHE_MONITOR_INFO mi, KHE_LIMIT_ACTIVE_INTERVALS_MONITOR m)             */
/*                                                                           */
/*  Add a summary of the current state of m to ps's signature.               */
/*                                                                           */
/*****************************************************************************/

static void KhePartialSolnAddLimitActiveIntervalsSignature(KHE_PARTIAL_SOLN ps,
  KHE_MONITOR_INFO mi, KHE_LIMIT_ACTIVE_INTERVALS_MONITOR m)
{
  int index, i, val, busy_count;  KHE_TIME_GROUP tg;  KHE_POLARITY po;

  /* add the number of adjacent active time groups to the signature */
  /* adjacent to what? still to do */

  /* add one Boolean (0 or 1) for each indexed time group */
  HaArrayForEach(mi->indexes, index, i)
  {
    val = (KheLimitActiveIntervalsMonitorTimeGroupIsActive(m, index, &tg, &po,
      &busy_count) ? 1 : 0);
    HaArrayAddLast(ps->signature, val);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KhePartialSolnSetSignature(KHE_PARTIAL_SOLN ps, KHE_DAY day)        */
/*                                                                           */
/*  Set the signature of ps.  It ends on day.                                */
/*                                                                           */
/*****************************************************************************/

static void KhePartialSolnSetSignature(KHE_PARTIAL_SOLN ps, KHE_DAY day)
{
  KHE_MONITOR_INFO mi;  int i;
  HaArrayForEach(day->monitor_info, mi, i)
    switch( KheMonitorTag(mi->monitor) )
    {
      case KHE_AVOID_CLASHES_MONITOR_TAG:

	KhePartialSolnAddAvoidClashesSignature(ps, mi,
	  (KHE_AVOID_CLASHES_MONITOR) mi->monitor);
	break;

      case KHE_AVOID_UNAVAILABLE_TIMES_MONITOR_TAG:

	KhePartialSolnAddAvoidUnavailableTimesSignature(ps, mi,
	  (KHE_AVOID_UNAVAILABLE_TIMES_MONITOR) mi->monitor);
	break;

      case KHE_LIMIT_IDLE_TIMES_MONITOR_TAG:

	KhePartialSolnAddLimitIdleTimesSignature(ps, mi,
	  (KHE_LIMIT_IDLE_TIMES_MONITOR) mi->monitor);
	break;

      case KHE_CLUSTER_BUSY_TIMES_MONITOR_TAG:

	KhePartialSolnAddClusterBusyTimesSignature(ps, mi,
	  (KHE_CLUSTER_BUSY_TIMES_MONITOR) mi->monitor);
	break;

      case KHE_LIMIT_BUSY_TIMES_MONITOR_TAG:

	KhePartialSolnAddLimitBusyTimesSignature(ps, mi,
	  (KHE_LIMIT_BUSY_TIMES_MONITOR) mi->monitor);
	break;

      case KHE_LIMIT_WORKLOAD_MONITOR_TAG:

	KhePartialSolnAddLimitWorkloadSignature(ps, mi,
	  (KHE_LIMIT_WORKLOAD_MONITOR) mi->monitor);
	break;

      case KHE_LIMIT_ACTIVE_INTERVALS_MONITOR_TAG:

	KhePartialSolnAddLimitActiveIntervalsSignature(ps, mi,
	  (KHE_LIMIT_ACTIVE_INTERVALS_MONITOR) mi->monitor);
	break;

      default:

	HnAbort("KheSingleResourceSolverAddMonitor internal error (tag %d)",
	  KheMonitorTag(mi->monitor));
    }
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_PARTIAL_SOLN KhePartialSolnMakeInit(KHE_SINGLE_RESOURCE_SOLVER srs)  */
/*                                                                           */
/*  Make the partial solution that the search grows from.  It does not       */
/*  end on any day.                                                          */
/*                                                                           */
/*****************************************************************************/

static KHE_PARTIAL_SOLN KhePartialSolnMakeInit(KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_PARTIAL_SOLN res;
  if( HaArrayCount(srs->free_partial_solns) > 0 )
  {
    res = HaArrayLastAndDelete(srs->free_partial_solns);
    HaArrayClear(res->signature);
  }
  else
  {
    HaMake(res, srs->arena);
    HaArrayInit(res->signature, srs->arena);
  }
  res->prev = NULL;
  res->task = NULL;
  res->best_of = 1;
  res->asst_count = 0;
  res->r_cost = 0;
  res->soln_cost = 0;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_PARTIAL_SOLN KhePartialSolnMake(KHE_DAY last_day,                    */
/*    KHE_PARTIAL_SOLN prev, KHE_TASK task, KHE_SINGLE_RESOURCE_SOLVER srs)  */
/*                                                                           */
/*  Make a new partial solution with these attributes.                       */
/*                                                                           */
/*****************************************************************************/

static KHE_PARTIAL_SOLN KhePartialSolnMake(KHE_DAY last_day,
  KHE_PARTIAL_SOLN prev, KHE_TASK task, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_PARTIAL_SOLN res;

  /* make the basic object */
  if( HaArrayCount(srs->free_partial_solns) > 0 )
  {
    res = HaArrayLastAndDelete(srs->free_partial_solns);
    HaArrayClear(res->signature);
  }
  else
  {
    HaMake(res, srs->arena);
    HaArrayInit(res->signature, srs->arena);
  }
  res->prev = prev;
  res->task = task;
  res->best_of = 1;
  res->asst_count = (prev == NULL ? 0 : prev->asst_count) +
    KheTaskTotalDuration(task);
  res->r_cost = KheMonitorCost((KHE_MONITOR) srs->resource_gm);
  res->soln_cost = KheSolnCost(srs->soln);

  /* set the signature */
  KhePartialSolnSetSignature(res, last_day);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  int KhePartialSolnSignatureHash(KHE_PARTIAL_SOLN ps)                     */
/*                                                                           */
/*  Hash function for hashing the signature of ps.                           */
/*                                                                           */
/*****************************************************************************/

static int KhePartialSolnSignatureHash(KHE_PARTIAL_SOLN ps)
{
  int res, val, i;
  res = HaArrayCount(ps->signature);
  HaArrayForEach(ps->signature, val, i)
    res = (res < 1) + val;
  return (res < 0 ? - res : res);
}


/*****************************************************************************/
/*                                                                           */
/*  int KhePartialSolnSignatureHashUntyped(void *p)                          */
/*                                                                           */
/*  Untyped version of KhePartialSolnSignatureHash.                          */
/*                                                                           */
/*****************************************************************************/

static int KhePartialSolnSignatureHashUntyped(void *p)
{
  return KhePartialSolnSignatureHash((KHE_PARTIAL_SOLN) p);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KhePartialSolnSignaturesEqual(KHE_PARTIAL_SOLN ps1,                 */
/*    KHE_PARTIAL_SOLN ps2)                                                  */
/*                                                                           */
/*  Return true if the signatures of ps1 and ps2 are equal.  The lengths     */
/*  of their signatures must be equal.                                       */
/*                                                                           */
/*****************************************************************************/

static bool KhePartialSolnSignaturesEqual(KHE_PARTIAL_SOLN ps1,
  KHE_PARTIAL_SOLN ps2)
{
  int i, val1, val2;
  HnAssert(HaArrayCount(ps1->signature) == HaArrayCount(ps2->signature),
    "KhePartialSolnSignaturesEqual: signatures have different lengths");
  for( i = 0;  i < HaArrayCount(ps1->signature);  i++ )
  {
    val1 = HaArray(ps1->signature, i);
    val2 = HaArray(ps2->signature, i);
    if( val1 != val2 )
      return false;
  }
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KhePartialSolnSignaturesEqualUntyped(void *p1, void *p2)            */
/*                                                                           */
/*  Untyped version of KhePartialSolnSignaturesEqual.                        */
/*                                                                           */
/*****************************************************************************/

static bool KhePartialSolnSignaturesEqualUntyped(void *p1, void *p2)
{
  return KhePartialSolnSignaturesEqual((KHE_PARTIAL_SOLN) p1,
    (KHE_PARTIAL_SOLN) p2);
}


/*****************************************************************************/
/*                                                                           */
/*  void KhePartialSolnOverWrite(KHE_PARTIAL_SOLN dest_ps,                   */
/*    KHE_PARTIAL_SOLN src_ps)                                               */
/*                                                                           */
/*  Overwrite src_ps onto dest_ps.  Their signatures are known to be         */
/*  equal, so those don't need to be touched.  Also the best_of field        */
/*  is not touched here; it gets updated separately.                         */
/*                                                                           */
/*****************************************************************************/

static void KhePartialSolnOverWrite(KHE_PARTIAL_SOLN dest_ps,
  KHE_PARTIAL_SOLN src_ps)
{
  dest_ps->prev = src_ps->prev;
  dest_ps->task = src_ps->task;
  dest_ps->asst_count = src_ps->asst_count;
  dest_ps->r_cost = src_ps->r_cost;
  dest_ps->soln_cost = src_ps->soln_cost;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KhePartialSolnCostLessThan(KHE_PARTIAL_SOLN ps1,                    */
/*    KHE_PARTIAL_SOLN ps2)                                                  */
/*                                                                           */
/*  Return true if ps1 is a better solution than ps2.                        */
/*                                                                           */
/*****************************************************************************/

static bool KhePartialSolnCostLessThan(KHE_PARTIAL_SOLN ps1,
  KHE_PARTIAL_SOLN ps2)
{
  return ps1->r_cost < ps2->r_cost ||
    (ps1->r_cost == ps2->r_cost && ps1->soln_cost < ps2->soln_cost);
}


/*****************************************************************************/
/*                                                                           */
/*  int KhePartialSolnIncreasingAsstCountCmp(const void *p1, const void *p2) */
/*                                                                           */
/*  Comparison function for sorting an array of partial solutions by         */
/*  increasing assignment count.                                             */
/*                                                                           */
/*****************************************************************************/

static int KhePartialSolnIncreasingAsstCountCmp(const void *p1, const void *p2)
{
  KHE_PARTIAL_SOLN ps1 = * (KHE_PARTIAL_SOLN *) p1;
  KHE_PARTIAL_SOLN ps2 = * (KHE_PARTIAL_SOLN *) p2;
  return ps1->asst_count - ps2->asst_count;
}


/*****************************************************************************/
/*                                                                           */
/*  void KhePartialSolnDebug(KHE_PARTIAL_SOLN ps, int verbosity,             */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of ps onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void KhePartialSolnDebug(KHE_PARTIAL_SOLN ps, int verbosity,
  int indent, FILE *fp)
{
  int val, i;
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  fprintf(fp, "[ Partial Soln ");
  if( ps->task != NULL )
    KheTaskDebug(ps->task, 1, -1, fp);
  else
    fprintf(fp, "(free day)");
  fprintf(fp, ": ");
  HaArrayForEach(ps->signature, val, i)
    fprintf(fp, "%s%d", i > 0 ? "," : "", val);
  fprintf(fp, " ]");
  if( indent > 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "monitor handling"                                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheSingleResourceSolverAddAvoidClashesMonitor(                      */
/*    KHE_SINGLE_RESOURCE_SOLVER srs, KHE_AVOID_CLASHES_MONITOR m)           */
/*                                                                           */
/*  Add avoid clashes monitor m to srs.                                      */
/*                                                                           */
/*****************************************************************************/

static void KheSingleResourceSolverAddAvoidClashesMonitor(
  KHE_SINGLE_RESOURCE_SOLVER srs, KHE_AVOID_CLASHES_MONITOR m)
{
  KHE_CONSTRAINT c;
  c = KheMonitorConstraint((KHE_MONITOR) m);
  if( KheConstraintCostFunction(c) == KHE_LINEAR_COST_FUNCTION )
  {
    /* no summary required; ignore m */
  }
  else
  {
    /* summary required */
    /* still to do */
    KheGroupMonitorAddChildMonitor(srs->resource_gm, (KHE_MONITOR) m);
    HnAbort("KheSingleResourceSolverAddAvoidClashesMonitor still to do");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSingleResourceSolverAddAvoidUnavailableTimesMonitor(             */
/*    KHE_SINGLE_RESOURCE_SOLVER srs, KHE_AVOID_UNAVAILABLE_TIMES_MONITOR m) */
/*                                                                           */
/*  Add avoid unavailable times monitor m to srs.                            */
/*                                                                           */
/*****************************************************************************/

static void KheSingleResourceSolverAddAvoidUnavailableTimesMonitor(
  KHE_SINGLE_RESOURCE_SOLVER srs, KHE_AVOID_UNAVAILABLE_TIMES_MONITOR m)
{
  KHE_CONSTRAINT c;
  c = KheMonitorConstraint((KHE_MONITOR) m);
  if( KheConstraintCostFunction(c) == KHE_LINEAR_COST_FUNCTION )
  {
    /* no summary required; ignore m */
  }
  else
  {
    /* summary required */
    /* still to do */
    KheGroupMonitorAddChildMonitor(srs->resource_gm, (KHE_MONITOR) m);
    HnAbort("KheSingleResourceSolverAddAvoidUnavailableTimesMonitor "
      "still to do");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSingleResourceSolverAddLimitIdleTimesMonitor(                    */
/*    KHE_SINGLE_RESOURCE_SOLVER srs, KHE_LIMIT_IDLE_TIMES_MONITOR m)        */
/*                                                                           */
/*  Add limit idle times monitor m to srs.                                   */
/*                                                                           */
/*****************************************************************************/

static void KheSingleResourceSolverAddLimitIdleTimesMonitor(
  KHE_SINGLE_RESOURCE_SOLVER srs, KHE_LIMIT_IDLE_TIMES_MONITOR m)
{
  /* not done, and that fact is documented */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSingleResourceSolverAddClusterBusyTimesMonitor(                  */
/*    KHE_SINGLE_RESOURCE_SOLVER srs, KHE_CLUSTER_BUSY_TIMES_MONITOR m)      */
/*                                                                           */
/*  Add cluster busy times monitor m to srs.                                 */
/*                                                                           */
/*****************************************************************************/

static void KheSingleResourceSolverAddClusterBusyTimesMonitor(
  KHE_SINGLE_RESOURCE_SOLVER srs, KHE_CLUSTER_BUSY_TIMES_MONITOR m)
{
  KHE_TIME first_time, last_time;  int first_index, last_index, i, j;
  KHE_DAY day;  KHE_MONITOR_INFO mi;  KHE_TIME_GROUP tg;  KHE_POLARITY po;

  /* find the interval of days that m covers */
  KheClusterBusyTimesMonitorRange(m, &first_time, &last_time);
  first_index = KheFrameTimeIndex(srs->days_frame, first_time);
  last_index = KheFrameTimeIndex(srs->days_frame, last_time);

  /* add a monitor info object to each of those days except the last */
  for( i = first_index;  i < last_index;  i++ )
  {
    day = HaArray(srs->days, i);
    mi = KheMonitorInfoMake((KHE_MONITOR) m, srs);
    HaArrayAddLast(day->monitor_info, mi);
  }

  /* add indexes to the monitor info objects as required */
  for( i = 0;  i < KheClusterBusyTimesMonitorTimeGroupCount(m);  i++ )
  {
    tg = KheClusterBusyTimesMonitorTimeGroup(m, i, &po);
    if( KheTimeGroupTimeCount(tg) > 0 )
    {
      first_time = KheTimeGroupTime(tg, 0);
      last_time = KheTimeGroupTime(tg, KheTimeGroupTimeCount(tg) - 1);
      first_index = KheFrameTimeIndex(srs->days_frame, first_time);
      last_index = KheFrameTimeIndex(srs->days_frame, last_time);
      for( j = first_index;  j < last_index;  j++ )
      {
	/* add i to the monitor info object at this index */
	day = HaArray(srs->days, j);
	mi = HaArrayLast(day->monitor_info);
	HaArrayAddLast(mi->indexes, i);
      }
    }
  }

  /* add m to srs_resource_gm */
  KheGroupMonitorAddChildMonitor(srs->resource_gm, (KHE_MONITOR) m);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSingleResourceSolverAddLimitBusyTimesMonitor(                    */
/*    KHE_SINGLE_RESOURCE_SOLVER srs, KHE_LIMIT_BUSY_TIMES_MONITOR m)        */
/*                                                                           */
/*  Add limit busy times monitor m to srs.                                   */
/*                                                                           */
/*****************************************************************************/

static void KheSingleResourceSolverAddLimitBusyTimesMonitor(
  KHE_SINGLE_RESOURCE_SOLVER srs, KHE_LIMIT_BUSY_TIMES_MONITOR m)
{
  KHE_TIME first_time, last_time;  int first_index, last_index, i, j, junk;
  KHE_DAY day;  KHE_MONITOR_INFO mi;  KHE_TIME_GROUP tg;

  /* find the interval of days that m covers */
  KheLimitBusyTimesMonitorRange(m, &first_time, &last_time);
  first_index = KheFrameTimeIndex(srs->days_frame, first_time);
  last_index = KheFrameTimeIndex(srs->days_frame, last_time);

  /* add a monitor info object to each of those days except the last */
  for( i = first_index;  i < last_index;  i++ )
  {
    day = HaArray(srs->days, i);
    mi = KheMonitorInfoMake((KHE_MONITOR) m, srs);
    HaArrayAddLast(day->monitor_info, mi);
  }

  /* add indexes to the monitor info objects as required */
  for( i = 0;  i < KheLimitBusyTimesMonitorTimeGroupCount(m);  i++ )
  {
    tg = KheLimitBusyTimesMonitorTimeGroup(m, i, &junk);
    if( KheTimeGroupTimeCount(tg) > 0 )
    {
      first_time = KheTimeGroupTime(tg, 0);
      last_time = KheTimeGroupTime(tg, KheTimeGroupTimeCount(tg) - 1);
      first_index = KheFrameTimeIndex(srs->days_frame, first_time);
      last_index = KheFrameTimeIndex(srs->days_frame, last_time);
      for( j = first_index;  j < last_index;  j++ )
      {
	/* add i to the monitor info object at this index */
	day = HaArray(srs->days, j);
	mi = HaArrayLast(day->monitor_info);
	HaArrayAddLast(mi->indexes, i);
      }
    }
  }

  /* add m to srs_resource_gm */
  KheGroupMonitorAddChildMonitor(srs->resource_gm, (KHE_MONITOR) m);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSingleResourceSolverAddLimitWorkloadMonitor(                     */
/*    KHE_SINGLE_RESOURCE_SOLVER srs, KHE_LIMIT_WORKLOAD_MONITOR m)          */
/*                                                                           */
/*  Add limit workload monitor m to srs.                                     */
/*                                                                           */
/*****************************************************************************/

static void KheSingleResourceSolverAddLimitWorkloadMonitor(
  KHE_SINGLE_RESOURCE_SOLVER srs, KHE_LIMIT_WORKLOAD_MONITOR m)
{
  KHE_TIME first_time, last_time;  int first_index, last_index, i, j;
  KHE_DAY day;  KHE_MONITOR_INFO mi;  KHE_TIME_GROUP tg;  float junk;

  /* find the interval of days that m covers */
  KheLimitWorkloadMonitorRange(m, &first_time, &last_time);
  first_index = KheFrameTimeIndex(srs->days_frame, first_time);
  last_index = KheFrameTimeIndex(srs->days_frame, last_time);

  /* add a monitor info object to each of those days except the last */
  for( i = first_index;  i < last_index;  i++ )
  {
    day = HaArray(srs->days, i);
    mi = KheMonitorInfoMake((KHE_MONITOR) m, srs);
    HaArrayAddLast(day->monitor_info, mi);
  }

  /* add indexes to the monitor info objects as required */
  for( i = 0;  i < KheLimitWorkloadMonitorTimeGroupCount(m);  i++ )
  {
    tg = KheLimitWorkloadMonitorTimeGroup(m, i, &junk);
    if( KheTimeGroupTimeCount(tg) > 0 )
    {
      first_time = KheTimeGroupTime(tg, 0);
      last_time = KheTimeGroupTime(tg, KheTimeGroupTimeCount(tg) - 1);
      first_index = KheFrameTimeIndex(srs->days_frame, first_time);
      last_index = KheFrameTimeIndex(srs->days_frame, last_time);
      for( j = first_index;  j < last_index;  j++ )
      {
	/* add i to the monitor info object at this index */
	day = HaArray(srs->days, j);
	mi = HaArrayLast(day->monitor_info);
	HaArrayAddLast(mi->indexes, i);
      }
    }
  }

  /* add m to srs_resource_gm */
  KheGroupMonitorAddChildMonitor(srs->resource_gm, (KHE_MONITOR) m);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSingleResourceSolverAddLimitActiveIntervalsMonitor(              */
/*    KHE_SINGLE_RESOURCE_SOLVER srs, KHE_LIMIT_ACTIVE_INTERVALS_MONITOR m)  */
/*                                                                           */
/*  Add limit active intervals monitor m to srs.                             */
/*                                                                           */
/*****************************************************************************/

static void KheSingleResourceSolverAddLimitActiveIntervalsMonitor(
  KHE_SINGLE_RESOURCE_SOLVER srs, KHE_LIMIT_ACTIVE_INTERVALS_MONITOR m)
{
  KHE_TIME first_time, last_time;  int first_index, last_index, i, j;
  KHE_DAY day;  KHE_MONITOR_INFO mi;  KHE_TIME_GROUP tg;  KHE_POLARITY po;

  /* find the interval of days that m covers */
  KheLimitActiveIntervalsMonitorRange(m, &first_time, &last_time);
  first_index = KheFrameTimeIndex(srs->days_frame, first_time);
  last_index = KheFrameTimeIndex(srs->days_frame, last_time);

  /* add a monitor info object to each of those days except the last */
  for( i = first_index;  i < last_index;  i++ )
  {
    day = HaArray(srs->days, i);
    mi = KheMonitorInfoMake((KHE_MONITOR) m, srs);
    HaArrayAddLast(day->monitor_info, mi);
  }

  /* add indexes to the monitor info objects as required */
  for( i = 0;  i < KheLimitActiveIntervalsMonitorTimeGroupCount(m);  i++ )
  {
    tg = KheLimitActiveIntervalsMonitorTimeGroup(m, i, &po);
    if( KheTimeGroupTimeCount(tg) > 0 )
    {
      first_time = KheTimeGroupTime(tg, 0);
      last_time = KheTimeGroupTime(tg, KheTimeGroupTimeCount(tg) - 1);
      first_index = KheFrameTimeIndex(srs->days_frame, first_time);
      last_index = KheFrameTimeIndex(srs->days_frame, last_time);
      for( j = first_index;  j < last_index;  j++ )
      {
	/* add i to the monitor info object at this index */
	day = HaArray(srs->days, j);
	mi = HaArrayLast(day->monitor_info);
	HaArrayAddLast(mi->indexes, i);
      }
    }
  }

  /* add m to srs_resource_gm */
  KheGroupMonitorAddChildMonitor(srs->resource_gm, (KHE_MONITOR) m);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSingleResourceSolverAddMonitor(KHE_SINGLE_RESOURCE_SOLVER srs,   */
/*    KHE_MONITOR m)                                                         */
/*                                                                           */
/*  Add m to srs.                                                            */
/*                                                                           */
/*****************************************************************************/

static void KheSingleResourceSolverAddMonitor(KHE_SINGLE_RESOURCE_SOLVER srs,
  KHE_MONITOR m)
{
  switch( KheMonitorTag(m) )
  {
    case KHE_AVOID_CLASHES_MONITOR_TAG:

      KheSingleResourceSolverAddAvoidClashesMonitor(srs,
        (KHE_AVOID_CLASHES_MONITOR) m);
      break;

    case KHE_AVOID_UNAVAILABLE_TIMES_MONITOR_TAG:

      KheSingleResourceSolverAddAvoidUnavailableTimesMonitor(srs,
        (KHE_AVOID_UNAVAILABLE_TIMES_MONITOR) m);
      break;

    case KHE_LIMIT_IDLE_TIMES_MONITOR_TAG:

      KheSingleResourceSolverAddLimitIdleTimesMonitor(srs,
        (KHE_LIMIT_IDLE_TIMES_MONITOR) m);
      break;

    case KHE_CLUSTER_BUSY_TIMES_MONITOR_TAG:

      KheSingleResourceSolverAddClusterBusyTimesMonitor(srs,
        (KHE_CLUSTER_BUSY_TIMES_MONITOR) m);
      break;

    case KHE_LIMIT_BUSY_TIMES_MONITOR_TAG:

      KheSingleResourceSolverAddLimitBusyTimesMonitor(srs,
        (KHE_LIMIT_BUSY_TIMES_MONITOR) m);
      break;

    case KHE_LIMIT_WORKLOAD_MONITOR_TAG:

      KheSingleResourceSolverAddLimitWorkloadMonitor(srs,
        (KHE_LIMIT_WORKLOAD_MONITOR) m);
      break;

    case KHE_LIMIT_ACTIVE_INTERVALS_MONITOR_TAG:

      KheSingleResourceSolverAddLimitActiveIntervalsMonitor(srs,
        (KHE_LIMIT_ACTIVE_INTERVALS_MONITOR) m);
      break;

    default:

      HnAbort("KheSingleResourceSolverAddMonitor internal error (tag %d)",
        KheMonitorTag(m));
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "public functions"                                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_SINGLE_RESOURCE_SOLVER KheSingleResourceSolverMake(KHE_SOLN soln,    */
/*    KHE_OPTIONS options)                                                   */
/*                                                                           */
/*  Make and return a new single resource solver with these attributes.      */
/*                                                                           */
/*****************************************************************************/

KHE_SINGLE_RESOURCE_SOLVER KheSingleResourceSolverMake(KHE_SOLN soln,
  KHE_OPTIONS options)
{
  KHE_SINGLE_RESOURCE_SOLVER res;  HA_ARENA a;  KHE_TIME_GROUP tg;  int i;
  KHE_FRAME days_frame;  KHE_EVENT_TIMETABLE_MONITOR etm;

  /* get the common frame and event timetable monitor; return NULL if can't */
  days_frame = KheOptionsFrame(options, "gs_common_frame", soln);
  etm = (KHE_EVENT_TIMETABLE_MONITOR)
    KheOptionsGetObject(options, "gs_event_timetable_monitor", NULL);
  if( days_frame == NULL || etm == NULL )
    return NULL;

  /* make the basic object */
  a = KheSolnArenaBegin(soln, false);
  HaMake(res, a);
  res->arena = a;
  res->soln = soln;
  res->options = options;
  res->diversifier = 0;
  res->days_frame = days_frame;
  res->etm = etm;
  res->task_finder = KheTaskFinderMake(soln, options, a);
  res->resource = NULL;
  res->resource_gm = NULL;
  HaArrayInit(res->days, a);
  HaArrayInit(res->final_solns, a);
  HaArrayInit(res->free_monitor_info, a);
  HaArrayInit(res->free_partial_solns, a);
  /* HaArrayInit(res->free_task_info, a); */
  HaArrayInit(res->free_best_tasks_at_time, a);
  HaArrayInit(res->free_best_tasks, a);

  /* add one day object for each time group of the common frame */
  for( i = 0;  i < KheFrameTimeGroupCount(res->days_frame);  i++ )
  {
    tg = KheFrameTimeGroup(res->days_frame, i);
    HaArrayAddLast(res->days, KheDayMake(tg, a));
  }

  /* all done */
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSingleResourceSolverDelete(KHE_SINGLE_RESOURCE_SOLVER srs)       */
/*                                                                           */
/*  Delete srs, reclaiming its memory.                                       */
/*                                                                           */
/*****************************************************************************/

void KheSingleResourceSolverDelete(KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KheSolnArenaEnd(srs->soln, srs->arena);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDayAddPartialSoln(KHE_DAY day, KHE_PARTIAL_SOLN ps,              */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Add ps to day.                                                           */
/*                                                                           */
/*  If day does not already contain a partial solution with the same         */
/*  signature as ps, just add ps to day and return.                          */
/*                                                                           */
/*  If day already contains a partial solution with the same signature       */
/*  as ps, called other_ps below, keep whichever one of ps and other_ps      */
/*  has the smallest cost, according to KhePartialSolnCostLessThan, by       */
/*  overwriting other_ps's fields with ps's fields, if required.             */
/*                                                                           */
/*  If ps and other_ps have equal cost, keep one chosen at random, with      */
/*  weights which ensure that if there are K partial solutions with the      */
/*  same signature and the same cost, then each has probability 1/K of       */
/*  being the one kept.                                                      */
/*                                                                           */
/*****************************************************************************/

static void KheDayAddPartialSoln(KHE_DAY day, KHE_PARTIAL_SOLN ps,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_PARTIAL_SOLN other_ps;
  if( !HpTableAddUnique(day->partial_solns, (void *) ps, ps, other_ps) )
  {
    /* other_ps will now be the best of one more partial solution than before */
    other_ps->best_of++;

    /* ensure that other_ps is the better of other_ps and ps, by overwriting */
    if( KhePartialSolnCostLessThan(ps, other_ps) ||
	(!KhePartialSolnCostLessThan(other_ps, ps) &&
	 srs->diversifier % other_ps->best_of == 0) )
    {
      /* ps is better than other_ps, so overwrite other_ps with ps */
      KhePartialSolnOverWrite(other_ps, ps);
    }

    /* free ps */
    HaArrayAddLast(srs->free_partial_solns, ps);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSingleResourceSolverExtend(KHE_SINGLE_RESOURCE_SOLVER srs,       */
/*    KHE_PARTIAL_SOLN ps, int di)                                           */
/*                                                                           */
/*  Extend the dynamic programming search by searching out of ps.  Each      */
/*  new task starts on day di.                                               */
/*                                                                           */
/*****************************************************************************/

static void KheSingleResourceSolverExtend(KHE_SINGLE_RESOURCE_SOLVER srs,
  KHE_PARTIAL_SOLN ps, int di)
{
  KHE_DAY first_day, last_day;  /* KHE_TASK_INFO ti; */  int i, j;
  KHE_PARTIAL_SOLN next_ps;  KHE_BEST_TASKS_AT_TIME btt;  KHE_BEST_TASK bt;
  first_day = HaArray(srs->days, di);
  HaArrayForEach(first_day->best_tasks_at_time, btt, i)
    HaArrayForEach(btt->best_tasks, bt, j)
    {
      last_day = HaArray(srs->days, bt->last_index);
      if( KheTaskAsstResource(bt->task) == srs->resource )
      {
	/* already assigned, add new partial soln without actually assigning */
	next_ps = KhePartialSolnMake(last_day, ps, bt->task, srs);
	KheDayAddPartialSoln(last_day, next_ps, srs);
      }
      else if( KheTaskAsstResource(bt->task) != NULL )
      {
	/* error, this task should not have been here */
	HnAbort("KheSingleResourceSolverExtend internal error 1");
      }
      else if( KheTaskAssignResource(bt->task, srs->resource) )
      {
	/* assign resource, add new partial soln, then unassign again */
	next_ps = KhePartialSolnMake(last_day, ps, bt->task, srs);
	KheDayAddPartialSoln(last_day, next_ps, srs);
	KheTaskUnAssignResource(bt->task);
      }
      else
      {
	/* error, this task failed to assign */
	HnAbort("KheSingleResourceSolverExtend internal error 2");
      }
      srs->diversifier++;
    }

  /* same thing but for taking the day off */
  last_day = first_day;
  next_ps = KhePartialSolnMake(last_day, ps, NULL, srs);
  KheDayAddPartialSoln(last_day, next_ps, srs);
  srs->diversifier++;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSingleResourceSolverSolve(KHE_SINGLE_RESOURCE_SOLVER srs,        */
/*    KHE_RESOURCE r)                                                        */
/*                                                                           */
/*  Solve for r.                                                             */
/*                                                                           */
/*****************************************************************************/

void KheSingleResourceSolverSolve(KHE_SINGLE_RESOURCE_SOLVER srs,
  KHE_RESOURCE r)
{
  int di, i, pos;  KHE_MONITOR m;  KHE_DAY day, prev_day;  KHE_PARTIAL_SOLN ps;

  /* if there was already a resource, clear out each day */
  if( srs->resource != NULL )
    HaArrayForEach(srs->days, day, di)
      KheDayClear(day, srs);

  /* add the resource and its monitors */
  srs->resource = r;
  srs->resource_gm = KheGroupMonitorMake(srs->soln, 0, "srs_group");
  for( i = 0;  i < KheSolnResourceMonitorCount(srs->soln, r);  i++ )
  {
    m = KheSolnResourceMonitor(srs->soln, r, i);
    KheSingleResourceSolverAddMonitor(srs, m);
  }

  /* add suitable tasks for each day */
  HaArrayForEach(srs->days, day, di)
    KheDayAddTasks(day, srs);

  /* build up the partial solutions for each day */
  prev_day = NULL;
  HaArrayForEach(srs->days, day, di)
  {
    if( prev_day == NULL )
    {
      /* virtual prev_day contains a single partial solution (an empty one) */
      ps = KhePartialSolnMakeInit(srs);
      KheSingleResourceSolverExtend(srs, ps, di);
    }
    else
    {
      /* extend each partial solution from prev_day with tasks from day */
      HpTableForEachValue(prev_day->partial_solns, ps, pos)
	KheSingleResourceSolverExtend(srs, ps, di);
    }
    prev_day = day;
  }

  /* extract final solutions */
  day = HaArrayLast(srs->days);
  HaArrayClear(srs->final_solns);
  HpTableForEachValue(prev_day->partial_solns, ps, pos)
    HaArrayAddLast(srs->final_solns, ps);
  HaArraySort(srs->final_solns, &KhePartialSolnIncreasingAsstCountCmp);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheSingleResourceSolverTimetableCount(KHE_SINGLE_RESOURCE_SOLVER srs)*/
/*                                                                           */
/*  Return the number of timetables found.                                   */
/*                                                                           */
/*****************************************************************************/

int KheSingleResourceSolverTimetableCount(KHE_SINGLE_RESOURCE_SOLVER srs)
{
  HnAssert(srs->resource != NULL,
    "KheSingleResourceSolverTimetableCount called out of order");
  return HaArrayCount(srs->final_solns);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSingleResourceSolverTimetable(KHE_SINGLE_RESOURCE_SOLVER srs,    */
/*    int i, int *asst_count, KHE_COST *r_cost, KHE_COST *soln_cost)         */
/*                                                                           */
/*  Report on the i'th timetable found by the solver.                        */
/*                                                                           */
/*****************************************************************************/

void KheSingleResourceSolverTimetable(KHE_SINGLE_RESOURCE_SOLVER srs,
  int i, int *asst_count, KHE_COST *r_cost, KHE_COST *soln_cost)
{
  /* make sure there is an i'th solution */
  KHE_PARTIAL_SOLN ps;
  HnAssert(srs->resource != NULL,
    "KheSingleResourceSolverTimetable called out of order");
  HnAssert(0 <= i && i < HaArrayCount(srs->final_solns),
    "KheSingleResourceSolverTimetable:  i (%d) out of range (0 .. %d)",
    i, HaArrayCount(srs->final_solns) - 1);

  /* return the relevant fields of the i'th solution */
  ps = HaArray(srs->final_solns, i);
  *asst_count = ps->asst_count;
  *r_cost = ps->r_cost;
  *soln_cost = ps->soln_cost;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSingleResourceSolverAdopt(KHE_SINGLE_RESOURCE_SOLVER srs, int i) */
/*                                                                           */
/*  Change srs's solution to include the i'th timetable.                     */
/*                                                                           */
/*****************************************************************************/

void KheSingleResourceSolverAdopt(KHE_SINGLE_RESOURCE_SOLVER srs, int i)
{
  /* make sure there is an i'th solution */
  KHE_PARTIAL_SOLN ps;
  HnAssert(srs->resource != NULL,
    "KheSingleResourceSolverAdopt called out of order");
  HnAssert(0 <= i && i < HaArrayCount(srs->final_solns),
    "KheSingleResourceSolverAdopt:  i (%d) out of range (0 .. %d)",
    i, HaArrayCount(srs->final_solns) - 1);

  /* assign the tasks of the i'th solution */
  for( ps = HaArray(srs->final_solns, i);  ps != NULL;  ps = ps->prev )
    if( ps->task != NULL && KheTaskAsstResource(ps->task) == NULL )
    {
      /* assign srs->resource to ps->task */
      if( !KheTaskAssignResource(ps->task, srs->resource) )
	HnAbort("KheSingleResourceSolverAdopt internal error");
    }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSingleResourceSolverDebug(KHE_SINGLE_RESOURCE_SOLVER srs,        */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug printf of srs onto fp with the given verbosity and indent.         */
/*                                                                           */
/*****************************************************************************/

void KheSingleResourceSolverDebug(KHE_SINGLE_RESOURCE_SOLVER srs,
  int verbosity, int indent, FILE *fp)
{
  KHE_DAY day;  int i;
  HnAssert(indent >= 0, "KheSingleResourceSolverDebug: indent < 0");
  fprintf(fp, "%*s[ SingleResourceSolver(%s)\n", indent, "",
    srs->resource == NULL ? "@" : KheResourceId(srs->resource));
  HaArrayForEach(srs->days, day, i)
    KheDayDebug(day, verbosity, indent + 2, fp);
  fprintf(fp, "%*s]\n", indent, "");
}
