
/*****************************************************************************/
/*                                                                           */
/*  THE KHE HIGH SCHOOL TIMETABLING ENGINE                                   */
/*  COPYRIGHT (C) 2010 Jeffrey H. Kingston                                   */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 3, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         khe_se_solvers.c                                           */
/*  DESCRIPTION:  Ejection chain solvers                                     */
/*                                                                           */
/*  Implementation note                                                      */
/*  -------------------                                                      */
/*                                                                           */
/*  Apart from helper functions, the functions in this file, or called       */
/*  from here, fall into five levels:                                        */
/*                                                                           */
/*  (1)  Full augment functions.  These are the functions passed to the      */
/*       ejector.  They usually call partial augment functions, although     */
/*       they may call multi-repair and repair functions.  Their names       */
/*       begin with the name of the defect they repair, and end with         */
/*       Augment.  Example: KheAssignTimeAugment.                            */
/*                                                                           */
/*  (2)  Partial augment functions.  These are functions that iterate over   */
/*       one or many entities, setting their visited flags and trying one    */
/*       or many repairs on each.  They call multi-repair and repair         */
/*       functions.  Their names end in Augment.  Example: KheMeetAugment.   */
/*                                                                           */
/*  (3)  Multi-repair functions.  These are functions that iterate over a    */
/*       set of alternative repairs to a single entity.  They call repair    */
/*       functions; occasionally they contain (in effect) inlined repair     */
/*       functions.  Their names end in MultiRepair.                         */
/*                                                                           */
/*  (4)  Repair functions.  These are functions that implement one change    */
/*       to the solution, recursively searching for a chain after each       */
/*       change.  They call KheEjectorRepairBegin at the start, and          */
/*       KheEjectorRepairEnd at the end, but they do not set the visited     */
/*       flags of any entities they touch, because it is expected that       */
/*       they will be called many times on the same entity.  Their names     */
/*       are based on the solution changing functions they call, with        */
/*       Repair added.  Example: KheMeetMoveRepair.                          */
/*                                                                           */
/*  (5)  Solution changing functions.  These are functions that change the   */
/*       solution, possibly in complex ways, but do not reference the        */
/*       ejector and do not change the visited status of any entity.  Most   */
/*       of them are defined elsewhere.  They have names that indicate       */
/*       what they do, like any function.  Example: KheTypedMeetMove.        */
/*                                                                           */
/*  There is one sub-module for each full augment function, and one          */
/*  sub-module containing the repair, multi-repair, and augment functions    */
/*  concerned with a single repair type, or a set of similar repair types.   */
/*                                                                           */
/*****************************************************************************/
#include "khe_solvers.h"
#include <limits.h>
#define min(a, b) ((a) < (b) ? (a) : (b))
#define max(a, b) ((a) > (b) ? (a) : (b))

#define IGNORE_NOCOST true

#define DEBUG1 0	/* main calls */
#define DEBUG2 0	/* initial and final defects */
#define DEBUG3 0	/* main augment functions */

#define DEBUG4 0	/* augment calls */
#define DEBUG5 0	/* repairs tried */
#define DEBUG6 0	/* demand repairs */
#define DEBUG7 0	/* cluster repairs */
#define DEBUG8 0	/* complex idle times repairs */
#define DEBUG9 0	/* complex idle times recursion */
#define DEBUG10 0	/* split repairs */

#define DEBUG11 0
#define DEBUG12 0
#define DEBUG13 0
#define DEBUG14 0
#define DEBUG15 0	/* avoid split assignments repairs */
#define DEBUG16 0	/* limit idle meet bound repairs */
#define DEBUG17 0	/* extra limit busy times repairs */
#define DEBUG18 0	/* new part of OverloadAugment */
#define DEBUG19 0	/* NurseOver */
#define DEBUG20 0	/* NurseOver */
#define DEBUG21 0	/* KheFrameMove */
#define DEBUG22 0	/* KheTaskSetRemovesUnderload */
#define DEBUG23 0	/* KheWorkloadLimitsAreTight */
#define DEBUG24 0	/* KheEjectionChainRepairInitialResourceAssignment */
#define DEBUG26 0	/* NoClashes check */
#define DEBUG27 0	/* visiting */
#define DEBUG28 0	/* cluster overload augment */
#define DEBUG29 0	/* KheResourceOverloadAugment */
#define DEBUG30 0
#define DEBUG31 0
#define DEBUG32 0
#define DEBUG33 0	/* double moves */
#define DEBUG34 0	/* swap to end repair */

#define MAX_MOVE 3	/* max number of days to left and right to move */

#define MAX_LIT_REPAIRS 1000	/* max number of complex limit idle repairs */

#define WITH_DEMAND_NODE_SWAPS true
#define WITH_SPREAD_EVENTS_NODE_SWAPS false
#define WITH_PREFER_TIMES_NODE_SWAPS false
#define WITH_OLD_AVOID_SPLIT_REPAIR 0
#define WITH_OLD_SPREAD_REPAIR 0

#define WITH_OLD_LIMIT_IDLE_REPAIR 1
#define WITH_NEW_LIMIT_IDLE_REPAIR_OVERLAP 0
#define WITH_NEW_LIMIT_IDLE_REPAIR_EXACT 0

/* #define WITH_EXTRA_LIMIT_BUSY_TIMES_REPAIRS 0 */

typedef enum {
  KHE_OPTIONS_KEMPE_FALSE,
  KHE_OPTIONS_KEMPE_LARGE_LAYERS,
  KHE_OPTIONS_KEMPE_TRUE
} KHE_OPTIONS_KEMPE;

typedef HA_ARRAY(KHE_MEET) ARRAY_KHE_MEET;
/* typedef HA_ARRAY(KHE_TASK) ARRAY_KHE_TASK; */
typedef HA_ARRAY(KHE_TIME_GROUP) ARRAY_KHE_TIME_GROUP;


/*****************************************************************************/
/*                                                                           */
/*  KHE_AUGMENT_TYPE - augment types (i.e. defect types)                     */
/*                                                                           */
/*  The order here follows the order of monitor types used in the Guide.     */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  KHE_AUGMENT_ORDINARY_DEMAND,
  KHE_AUGMENT_WORKLOAD_DEMAND,
  KHE_AUGMENT_SPLIT_EVENTS,
  KHE_AUGMENT_ASSIGN_TIME,
  KHE_AUGMENT_PREFER_TIMES,
  KHE_AUGMENT_SPREAD_EVENTS,
  KHE_AUGMENT_LINK_EVENTS,
  KHE_AUGMENT_ORDER_EVENTS,
  KHE_AUGMENT_ASSIGN_RESOURCE,
  KHE_AUGMENT_PREFER_RESOURCES,
  KHE_AUGMENT_AVOID_SPLIT_ASSIGNMENTS,
  KHE_AUGMENT_LIMIT_RESOURCES,
  KHE_AUGMENT_AVOID_CLASHES,
  KHE_AUGMENT_AVOID_UNAVAILABLE_TIMES,
  KHE_AUGMENT_LIMIT_IDLE_TIMES,
  KHE_AUGMENT_CLUSTER_BUSY_TIMES,
  KHE_AUGMENT_LIMIT_BUSY_TIMES,
  KHE_AUGMENT_LIMIT_WORKLOAD,
  KHE_AUGMENT_LIMIT_ACTIVE_INTERVALS 
} KHE_AUGMENT_TYPE;


/*****************************************************************************/
/*                                                                           */
/*  KHE_REPAIR_TYPE - repair types (i.e. repair operations)                  */
/*                                                                           */
/*  There doesn't seem to be any natural order for repair types.  These      */
/*  ones originally appeared in the order they appear in the text of my      */
/*  PATAT14 paper, but they have been fiddled with since then.               */
/*                                                                           */
/*  It's not clear that these categories are particularly useful.  In some   */
/*  cases they don't distinguish move from assignment and unassignment.      */
/*  In other cases the same repair is called from different contexts, and    */
/*  it would be better to measure its effectiveness in each context.         */
/*                                                                           */
/*****************************************************************************/

typedef enum {

  /* repairs applied to meets */
  KHE_REPAIR_MEET_MOVE_UNCHECKED,
  KHE_REPAIR_MEET_MOVE_CHECKED,
  KHE_REPAIR_MEET_MOVE_EJECTING,
  KHE_REPAIR_MEET_MOVE_KEMPE,

  /* old meet move types, mostly obsolete */
  /* KHE_REPAIR_BASIC_MEET_ASSIGN, */
  /* KHE_REPAIR_BASIC_MEET_MOVE, */
  /* KHE_REPAIR_EJECTING_MEET_ASSIGN, */
  /* KHE_REPAIR_EJECTING_MEET_MOVE, */
  /* KHE_REPAIR_KEMPE_MEET_MOVE, */
  KHE_REPAIR_FUZZY_MEET_MOVE,

  /* repairs applied to tasks and task sets */
  KHE_REPAIR_TASK_DOUBLE_MOVE,
  KHE_REPAIR_TASK_SET_MOVE,
  KHE_REPAIR_TASK_SET_DOUBLE_MOVE,
  KHE_REPAIR_TASK_SET_REPLACE,
  KHE_REPAIR_TASK_SET_DOUBLE_REPLACE,
  KHE_REPAIR_RESOURCE_MOVE,
  /* KHE_REPAIR_TASK_UNASSIGN, */
  /* KHE_REPAIR_EJECTING_TASK_ASSIGN, */
  /* KHE_REPAIR_TASK_MOVE_UNCHECKED, */
  /* KHE_REPAIR_TASK_MOVE_CHECKED, */
  /* KHE_REPAIR_TASK_MOVE_EJECTING, */
  /* KHE_REPAIR_TASK_MOVE_KEMPE, */
  /* KHE_REPAIR_TASK_SET_UNASSIGN, */
  /* KHE_REPAIR_TASK_SET_MOVE_UNCHECKED, */
  /* KHE_REPAIR_TASK_SET_MOVE_CHECKED, */
  /* KHE_REPAIR_TASK_SET_MOVE_EJECTING, */
  /* KHE_REPAIR_TASK_SET_MOVE_KEMPE, */
  /* KHE_REPAIR_TASK_SET_DOUBLE_MOVE, */
  /* KHE_REPAIR_EJECTING_TASK_SET_SWAP, */
  /* KHE_REPAIR_DAY_SET_SWAP, */
  /* KHE_REPAIR_DOUBLE_DAY_SET_SWAP, */
  /* KHE_REPAIR_DAY_SET_UNASSIGN, */
  /* KHE_REPAIR_DAY_SET_ASSIGN, */

  /* combined operations */
  KHE_REPAIR_NODE_MEET_SWAP,
  KHE_REPAIR_MEET_SPLIT,
  KHE_REPAIR_SPLIT_MOVE,
  KHE_REPAIR_MERGE_MOVE,
  KHE_REPAIR_SPLIT_TASKS_UNASSIGN,
  KHE_REPAIR_MEET_BOUND,
  KHE_REPAIR_KEMPE_MEET_MOVE_AND_EJECTING_TASK_ASSIGN,
  KHE_REPAIR_KEMPE_MEET_MOVE_AND_EJECTING_TASK_MOVE,
  KHE_REPAIR_COMPLEX_IDLE_MOVE
  /* KHE_REPAIR_LIMIT_IDLE_MEETS_UNASSIGN */
} KHE_REPAIR_TYPE;


/*****************************************************************************/
/*                                                                           */
/*  KHE_MOVE_REASON - why a task move is being undertaken                    */
/*                                                                           */
/*****************************************************************************/

/* ***
typedef enum {
  KHE_INCREASE_WORKLOAD,
  KHE_DECREASE_WORKLOAD,
  KHE_DECREASE_WORKLOAD_TO_ZERO,
  KHE_FIX_TASK
} KHE_MOVE_REASON;
*** */



/*****************************************************************************/
/*                                                                           */
/*  Submodule "options"                                                      */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool KheEjectorRepairTimes(KHE_EJECTOR ej)                               */
/*  KHE_NODE KheEjectorLimitNode(KHE_EJECTOR ej)                             */
/*  bool KheEjectorRepairResources(KHE_EJECTOR ej)                           */
/*  bool KheEjectorNodesBeforeMeets(KHE_EJECTOR ej)                          */
/*  bool KheEjectorUseKempeMoves(KHE_EJECTOR ej)                             */
/*  bool KheEjectorBasicNotEjecting(KHE_EJECTOR ej)                          */
/*  bool KheEjectorUseFuzzyMoves(KHE_EJECTOR ej)                             */
/*  bool KheEjectorUseSplitMoves(KHE_EJECTOR ej)                             */
/*  bool KheEjectorWideningOff(KHE_EJECTOR ej)                               */
/*  bool KheEjectorReversingOff(KHE_EJECTOR ej)                              */
/*  bool KheEjectorBalancingOff(KHE_EJECTOR ej)                              */
/*  bool KheEjectorNoCostOff(KHE_EJECTOR ej)                                 */
/*                                                                           */
/*  Return the indicated options.                                            */
/*                                                                           */
/*****************************************************************************/

static bool KheEjectorRepairTimes(KHE_EJECTOR ej)
{
  return KheOptionsGetBool(KheEjectorOptions(ej), "es_repair_times", false);
}

static KHE_NODE KheEjectorLimitNode(KHE_EJECTOR ej)
{
  return (KHE_NODE) KheOptionsGetObject(KheEjectorOptions(ej),
    "es_limit_node", NULL);
}

static bool KheEjectorRepairResources(KHE_EJECTOR ej)
{
  return KheOptionsGetBool(KheEjectorOptions(ej), "es_repair_resources", false);
}

static bool KheEjectorNodesBeforeMeets(KHE_EJECTOR ej)
{
  return KheOptionsGetBool(KheEjectorOptions(ej), "es_nodes_before_meets",
    false);
}

static KHE_OPTIONS_KEMPE KheEjectorUseKempeMoves(KHE_EJECTOR ej)
{
  char *opt = KheOptionsGet(KheEjectorOptions(ej), "es_kempe_moves",
    "large_layers");
  if( strcmp(opt, "false") == 0 )
    return KHE_OPTIONS_KEMPE_FALSE;
  else if( strcmp(opt, "large_layers") == 0 )
    return KHE_OPTIONS_KEMPE_LARGE_LAYERS;
  else if( strcmp(opt, "true") == 0 )
    return KHE_OPTIONS_KEMPE_TRUE;
  else
  {
    HnAbort("KheEjector: kempe_moves option has invalid value \"%s\"", opt);
    return KHE_OPTIONS_KEMPE_FALSE;
  }
}

static bool KheEjectorBasicNotEjecting(KHE_EJECTOR ej)
{
  return KheOptionsGetBool(KheEjectorOptions(ej), "es_no_ejecting_moves",false);
}

static bool KheEjectorUseFuzzyMoves(KHE_EJECTOR ej)
{
  return KheOptionsGetBool(KheEjectorOptions(ej), "es_fuzzy_moves", false);
}

static bool KheEjectorUseSplitMoves(KHE_EJECTOR ej)
{
  return KheOptionsGetBool(KheEjectorOptions(ej), "es_split_moves", false);
}

static KHE_KEMPE_STATS KheEjectorKempeStats(KHE_EJECTOR ej)
{
  return KheKempeStatsOption(KheEjectorOptions(ej), "ts_kempe_stats");
}

/* *** currently unused
static bool KheEjectorTaskSeqSwapsOff(KHE_EJECTOR ej)
{
  return KheOptionsGetBool(KheEjectorOptions(ej), "es_task_seq_swaps_off",
    false);
}
*** */

typedef struct khe_expansion_options_rec {
  bool	widening_off;
  int	widening_max;
  bool	reversing_off;
  bool	balancing_off;
  int	balancing_max;
  bool  full_widening_on;
} *KHE_EXPANSION_OPTIONS;

/* *** replaced by KheEjectorExpansionOptions
static bool KheEjectorWideningOff(KHE_EJECTOR ej)
{
  return KheOptionsGetBool(KheEjectorOptions(ej), "es_widening_off", false);
}

static bool KheEjectorReversingOff(KHE_EJECTOR ej)
{
  return KheOptionsGetBool(KheEjectorOptions(ej), "es_reversing_off", false);
}

static bool KheEjectorBalancingOff(KHE_EJECTOR ej)
{
  return KheOptionsGetBool(KheEjectorOptions(ej), "es_balancing_off", false);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheEjectorExpansionOptions(KHE_EJECTOR ej, KHE_EXPANSION_OPTIONS eo)*/
/*                                                                           */
/*  Set the fields of eo from ej's options.                                  */
/*                                                                           */
/*****************************************************************************/

static void KheEjectorExpansionOptions(KHE_EJECTOR ej, KHE_EXPANSION_OPTIONS eo)
{
  KHE_OPTIONS options;
  options = KheEjectorOptions(ej);
  eo->widening_off = KheOptionsGetBool(options, "es_widening_off", false);
  if( eo->widening_off )
    eo->widening_max = 0;
  else
    eo->widening_max = KheOptionsGetInt(options, "es_widening_max", 4);
  eo->reversing_off = KheOptionsGetBool(options, "es_reversing_off", false);
  eo->balancing_off = KheOptionsGetBool(options, "es_balancing_off", false);
  if( eo->balancing_off )
    eo->balancing_max = 0;
  else
    eo->balancing_max = KheOptionsGetInt(options, "es_balancing_max", 12);
  eo->full_widening_on = KheOptionsGetBool(options, "es_full_widening_on",
    false);
}

/* ***
static bool KheEjectorNoCostOff(KHE_EJECTOR ej)
{
  return KheOptionsGetBool(KheEjectorOptions(ej), "es_nocost_off", false);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  Submodule "clash checking"                                               */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheNoClashesCheck(KHE_EJECTOR ej, KHE_RESOURCE r)                   */
/*                                                                           */
/*  Make sure that r has no clashes.                                         */
/*                                                                           */
/*****************************************************************************/

static void KheNoClashesCheck(KHE_EJECTOR ej, KHE_RESOURCE r)
{
  if( DEBUG26 && r != NULL )
    KheFrameResourceAssertNoClashes(KheEjectorFrame(ej), r);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheClashes(KHE_EJECTOR ej, KHE_RESOURCE r)                          */
/*                                                                           */
/*  Return true if there are clashes.                                        */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused
static bool KheClashes(KHE_EJECTOR ej, KHE_RESOURCE r)
{
  return DEBUG26 && r != NULL &&
    KheFrameResourceHasClashes(KheEjectorFrame(ej), r);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  Submodule "resource iterator"                                            */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_SE_RESOURCE_SET - a set of resources, defined very flexibly          */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_se_resource_set_type {
  KHE_RESOURCE_GROUP		include_rg;
  KHE_RESOURCE_GROUP		exclude_rg;
  int				curr_index;
  int				stop_index;
  int				extra;
} *KHE_SE_RESOURCE_SET;


/*****************************************************************************/
/*                                                                           */
/*  void KheSeResourceSetInit(KHE_SE_RESOURCE_SET srs,                       */
/*    KHE_RESOURCE_GROUP include_rg, KHE_RESOURCE_GROUP exclude_rg,          */
/*    bool include_nonassign)                                                */
/*                                                                           */
/*  Initialize a resource set object.  It will iterate over the resources    */
/*  of include_rg, skipping the resources of exclude_rg if non-NULL.  If     */
/*  include_nonassign is true, non-assignment in the form of a NULL          */
/*  resource will be included in the iteration.  It comes first if present.  */
/*                                                                           */
/*****************************************************************************/

static void KheSeResourceSetInit(KHE_SE_RESOURCE_SET srs,
  KHE_RESOURCE_GROUP include_rg, KHE_RESOURCE_GROUP exclude_rg,
  bool include_nonassign, int extra)
{
  srs->include_rg = include_rg;
  srs->exclude_rg = exclude_rg;
  srs->curr_index = include_nonassign ? -1 : 0;
  srs->stop_index = KheResourceGroupResourceCount(include_rg);
  srs->extra = extra;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheSeResourceSetNext(KHE_SE_RESOURCE_SET srs, KHE_RESOURCE *r)      */
/*                                                                           */
/*  Return the next resource if there is one, or 0 otherwise.                */
/*                                                                           */
/*****************************************************************************/

static bool KheSeResourceSetNext(KHE_SE_RESOURCE_SET srs, KHE_RESOURCE *r)
{
  KHE_RESOURCE r2;
  if( srs->curr_index == -1 )
  {
    srs->curr_index++;
    return *r = NULL, true;
  }
  else
  {
    while( srs->curr_index < srs->stop_index )
    {
      r2 = KheResourceGroupResource(srs->include_rg,
	(srs->curr_index + srs->extra) % srs->stop_index);
      srs->curr_index++;
      if( srs->exclude_rg == NULL ||
	  !KheResourceGroupContains(srs->exclude_rg, r2) )
	return *r = r2, true;
    }
    return *r = NULL, false;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  KheForEachResource - iterator over resources                              */
/*                                                                           */
/*****************************************************************************/

#define KheForEachResource(srs_rec, irg, erg, in, extra, r)	\
for( KheSeResourceSetInit(&(srs_rec), irg, erg, in, extra);	\
     KheSeResourceSetNext(&(srs_rec), &r); )


/*****************************************************************************/
/*                                                                           */
/*  Submodule "task and task set visiting and equivalence"                   */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskSetFirstTasksEquivalent(KHE_TASK_SET ts1, KHE_TASK_SET ts2)  */
/*                                                                           */
/*  Return true if the first tasks of ts1 and ts2 are equivalent.            */
/*                                                                           */
/*****************************************************************************/

static bool KheTaskSetFirstTasksEquivalent(KHE_TASK_SET ts1, KHE_TASK_SET ts2)
{
  KHE_TASK task1, task2;
  if( KheTaskSetTaskCount(ts1) == 0 || KheTaskSetTaskCount(ts2) == 0 )
    return false;
  task1 = KheTaskSetTask(ts1, 0);
  task2 = KheTaskSetTask(ts2, 0);
  return KheTaskEquivalent(task1, task2);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheTaskVisitEquivalent(KHE_TASK task)                               */
/*                                                                           */
/*  If task is not visited, visit it and, if it is unassigned, also visit    */
/*  all equivalent unassigned tasks.                                         */
/*                                                                           */
/*****************************************************************************/

static void KheTaskVisitEquivalent(KHE_TASK task)
{
  KHE_MEET meet;  int i;  KHE_TASK task2;
  if( !KheTaskVisited(task, 0) )
  {
    KheTaskVisit(task);
    if( DEBUG27 )
    {
      fprintf(stderr, "  visiting task %p ", (void *) task);
      KheTaskDebug(task, 2, 0, stderr);
    }
    if( KheTaskAsstResource(task) == NULL )
    {
      meet = KheTaskMeet(task);
      if( meet != NULL )
      {
	for( i = 0;  i < KheMeetTaskCount(meet);  i++ )
	{
	  task2 = KheMeetTask(meet, i);
	  if( task2 != task && KheTaskAsstResource(task2) == NULL &&
	      KheTaskEquivalent(task2, task) )
	  {
	    if( DEBUG27 )
	    {
	      fprintf(stderr, "  visiting equivalent task %p ", (void *) task2);
	      KheTaskDebug(task2, 2, 0, stderr);
	    }
	    KheTaskVisit(task2);
	  }
	}
      }
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheTaskSetVisitEquivalent(KHE_TASK_SET ts)                          */
/*                                                                           */
/*  Visit the tasks of ts and all equivalent tasks.                          */
/*                                                                           */
/*****************************************************************************/

static void KheTaskSetVisitEquivalent(KHE_TASK_SET ts)
{
  KHE_TASK task;  int i;
  for( i = 0;  i < KheTaskSetTaskCount(ts);  i++ )
  {
    task = KheTaskSetTask(ts, i);
    KheTaskVisitEquivalent(task);
  }
  if( DEBUG32 )
  {
    fprintf(stderr, "  KheTaskSetVisitEquivalent(");
    KheTaskSetDebug(ts, 2, -1, stderr);
    fprintf(stderr, ")\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheTaskUnVisitEquivalent(KHE_TASK task)                             */
/*                                                                           */
/*  Unvisit task and, if unassigned, also unvisit all equivalent unassigned  */
/*  tasks.                                                                   */
/*                                                                           */
/*****************************************************************************/

static void KheTaskUnVisitEquivalent(KHE_TASK task)
{
  KHE_MEET meet;  int i;  KHE_TASK task2;
  KheTaskUnVisit(task);
  if( DEBUG27 )
  {
    fprintf(stderr, "  unvisiting task %p ", (void *) task);
    KheTaskDebug(task, 2, 0, stderr);
  }
  if( KheTaskAsstResource(task) == NULL )
  {
    meet = KheTaskMeet(task);
    if( meet != NULL )
    {
      for( i = 0;  i < KheMeetTaskCount(meet);  i++ )
      {
	task2 = KheMeetTask(meet, i);
	if( task2 != task && KheTaskAsstResource(task2) == NULL &&
	    KheTaskEquivalent(task2, task) )
	{
	  if( DEBUG27 )
	  {
	    fprintf(stderr, "  unvisiting equivalent task %p ", (void *) task2);
	    KheTaskDebug(task2, 2, 0, stderr);
	  }
	  KheTaskUnVisit(task2);
	}
      }
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheTaskSetUnVisitEquivalent(KHE_TASK_SET ts)                        */
/*                                                                           */
/*  Unvisit the tasks of ts and all equivalent tasks.                        */
/*                                                                           */
/*****************************************************************************/

static void KheTaskSetUnVisitEquivalent(KHE_TASK_SET ts)
{
  KHE_TASK task;  int i;
  for( i = 0;  i < KheTaskSetTaskCount(ts);  i++ )
  {
    task = KheTaskSetTask(ts, i);
    KheTaskUnVisitEquivalent(task);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheTasksDiffer(KHE_TASK task1, KHE_TASK task2)                      */
/*                                                                           */
/*  Return true when these two tasks are not equivalent, including when      */
/*  they have different assignments.                                         */
/*                                                                           */
/*****************************************************************************/

/* ***
static bool KheTasksDiffer(KHE_TASK task1, KHE_TASK task2)
{
  return KheTaskAsstResource(task1) != KheTaskAsstResource(task2) ||
    !KheTaskEquivalent(task1, task2);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskEquiv(KHE_TASK task1, KHE_TASK task2)                        */
/*                                                                           */
/*  Return true if task1 and task2 are equivalent and unassigned.            */
/*                                                                           */
/*****************************************************************************/

static bool KheTaskEquiv(KHE_TASK task1, KHE_TASK task2)
{
  return KheTaskAsstResource(task1) == KheTaskAsstResource(task2) &&
    KheTaskEquivalent(task1, task2);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "meet bound repairs and augments"                              */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheMeetBoundOnSuccess(void *on_success_val)                         */
/*                                                                           */
/*  On-success function used by meet bound repairs.                          */
/*                                                                           */
/*****************************************************************************/

static void KheMeetBoundOnSuccess(void *on_success_val)
{
  KHE_MEET_BOUND mb;  bool success;
  if( DEBUG13 )
    fprintf(stderr, "[ KheMeetBoundOnSuccess\n");
  mb = (KHE_MEET_BOUND) on_success_val;
  success = KheMeetBoundDelete(mb);
  if( DEBUG13 )
    fprintf(stderr, "] KheMeetBoundOnSuccess(success = %s)\n",
      success ? "true" : "false");
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheMeetBoundRepair(KHE_EJECTOR ej, KHE_RESOURCE r,                  */
/*    KHE_TIME_GROUP include_tg)                                             */
/*                                                                           */
/*  Reduce the domains of the meets currently assigned r, to contain a       */
/*  subset of the times of include_tg.  Meets currently assigned outside     */
/*  include_tg are unassigned first.  Return true if all successful.         */
/*                                                                           */
/*****************************************************************************/

static bool KheMeetBoundRepair(KHE_EJECTOR ej, KHE_RESOURCE r,
  KHE_TIME_GROUP include_tg)
{
  int i, junk, durn;  KHE_TASK task;  KHE_TIME time;  KHE_MEET meet;
  KHE_MEET_BOUND mb;  KHE_SOLN soln;  bool success;
  if( DEBUG13 )
  {
    fprintf(stderr, "[ KheMeetBoundRepair(ej, %s, ", KheResourceId(r));
    KheTimeGroupDebug(include_tg, 1, -1, stderr);
    fprintf(stderr, ")\n");
  }

  /* unassign r's meets outside include_tg; add time bounds to all r's meets */
  soln = KheEjectorSoln(ej);
  KheEjectorRepairBegin(ej);
  mb = KheMeetBoundMake(soln, true, include_tg);
  for( i = 0;  i < KheResourceAssignedTaskCount(soln, r);  i++ )
  {
    task = KheResourceAssignedTask(soln, r, i);
    meet = KheMeetFirstMovable(KheTaskMeet(task), &junk);
    if( meet == NULL || KheMeetIsPreassigned(meet, &time) )
    {
      if( DEBUG13 )
	fprintf(stderr, "] KheMeetBoundRepair returning false (no unfixed)");
      return KheEjectorRepairEnd(ej, KHE_REPAIR_MEET_BOUND, false);
    }
    durn = KheMeetDuration(meet);
    time = KheMeetAsstTime(meet);
    if( time != NULL && KheTimeGroupOverlap(include_tg, time, durn) != durn )
    {
      if( DEBUG13 )
      {
	fprintf(stderr, " unassigning ");
	KheMeetDebug(meet, 1, 0, stderr);
      }
      if( !KheMeetUnAssign(meet) )
      {
	if( DEBUG13 )
	{
	  fprintf(stderr, "] KheMeetBoundRepair returning false");
	  fprintf(stderr, " cannot unassign ");
	  KheMeetDebug(meet, 1, 0, stderr);
	}
	return KheEjectorRepairEnd(ej, KHE_REPAIR_MEET_BOUND, false);
      }
    }
    if( !KheMeetAddMeetBound(meet, mb) )
    {
      if( DEBUG13 )
      {
	fprintf(stderr, "] KheMeetBoundRepair returning false");
	fprintf(stderr, " cannot bound ");
	KheMeetDebug(meet, 1, 0, stderr);
      }
      return KheEjectorRepairEnd(ej, KHE_REPAIR_MEET_BOUND, false);
    }
  }
  success = KheEjectorRepairEndLong(ej, KHE_REPAIR_MEET_BOUND, true,
    INT_MAX, false, &KheMeetBoundOnSuccess, (void *) mb);
  if( DEBUG13 )
    fprintf(stderr, "] KheMeetBoundRepair returning %s\n",
      success ? "true" : "False");
  return success;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "meet move and swap repairs and augments"                      */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_TRY_MEET_MOVE_FN                                                     */
/*                                                                           */
/*  Functions of this type are called to decide whether to try meet moves    */
/*  of meet (or of an ancestor of meet) which give meet this starting time.  */
/*                                                                           */
/*  Parameter impl gives access to other information whose type is known     */
/*  to the function, which it can use in making its decision.                */
/*                                                                           */
/*****************************************************************************/

typedef bool (*KHE_TRY_MEET_MOVE_FN)(KHE_MEET meet, KHE_TIME time, void *impl);


/*****************************************************************************/
/*                                                                           */
/*  KHE_REPAIR_TYPE KheMoveTypeToMeetRepairType(KHE_MOVE_TYPE mt)            */
/*                                                                           */
/*  Convert move type mt into the corresponding task repair type.            */
/*                                                                           */
/*****************************************************************************/

static KHE_REPAIR_TYPE KheMoveTypeToMeetRepairType(KHE_MOVE_TYPE mt)
{
  switch( mt )
  {
    case KHE_MOVE_UNCHECKED:	return KHE_REPAIR_MEET_MOVE_UNCHECKED;
    case KHE_MOVE_CHECKED:	return KHE_REPAIR_MEET_MOVE_CHECKED;
    case KHE_MOVE_EJECTING:	return KHE_REPAIR_MEET_MOVE_EJECTING;
    case KHE_MOVE_KEMPE:	return KHE_REPAIR_MEET_MOVE_KEMPE;

    default:
      HnAbort("KheMoveTypeToMeetRepairType internal error");
      return KHE_REPAIR_MEET_MOVE_UNCHECKED;  /* keep compiler happy */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheBasicMeetAssignOrMoveRepair(KHE_EJECTOR ej, KHE_MEET meet,       */
/*    KHE_MEET target_meet, int offset, bool preserve_regularity)            */
/*                                                                           */
/*  Try a basic meet assignment (KHE_REPAIR_BASIC_MEET_ASSIGN) or a basic    */
/*  meet move (KHE_REPAIR_BASIC_MEET_MOVE) of meet to target_meet at offset. */
/*                                                                           */
/*****************************************************************************/

/* *** replaced by KheMeetMoveRepair with mt == KHE_MOVE_UNCHECKED
static bool KheBasicMeetAssignOrMoveRepair(KHE_EJECTOR ej, KHE_MEET meet,
  KHE_MEET target_meet, int offset, bool preserve_regularity)
{
  bool success, move;  int d;
  move = (KheMeetAsst(meet) != NULL);
  KheEjectorRepa irBegin(ej);
  success = KheBasicMeetMove(meet, target_meet, offset, preserve_regularity,&d);
  if( DEBUG5 || KheEjectorCurrDebug(ej) )
  {
    fprintf(stderr, "%cBasicMeetAssignOrMove(", success ? '+' : '-');
    KheMeetDebug(meet, 1, -1, stderr);
    fprintf(stderr, ", ");
    KheMeetDebug(target_meet, 1, -1, stderr);
    fprintf(stderr, ", %d, %s)", offset,
      preserve_regularity ? "true" : "false");
  }
  return KheEjectorRepa irEnd(ej, move ? KHE_REPAIR_BASIC_MEET_MOVE :
    KHE_REPAIR_BASIC_MEET_ASSIGN, success);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheEjectingMeetAssignOrMoveRepair(KHE_EJECTOR ej, KHE_MEET meet,    */
/*    KHE_MEET target_meet, int offset, bool preserve_regularity,bool *basic)*/
/*                                                                           */
/*  Try an ejecting meet assignment (KHE_REPAIR_EJECTING_MEET_ASSIGN) or     */
/*  an ejecting meet move (KHE_REPAIR_EJECTING_MEET_MOVE) of meet to         */
/*  target_meet at offset.                                                   */
/*                                                                           */
/*****************************************************************************/

/* *** replaced by KheMeetMoveRepair with mt == KHE_MOVE_EJECTING
static bool KheEjectingMeetAssignOrMoveRepair(KHE_EJECTOR ej, KHE_MEET meet,
  KHE_MEET target_meet, int offset, bool preserve_regularity, bool *basic)
{
  bool success, move;  int d;
  move = (KheMeetAsst(meet) != NULL);
  KheEjectorRep airBegin(ej);
  success = KheEjectingMeetMove(meet, target_meet, offset, true,
    preserve_regularity, &d, basic);
  if( DEBUG5 || KheEjectorCurrDebug(ej) )
  {
    fprintf(stderr, "%cEjectingMeetAssignOrMove(", success ? '+' : '-');
    KheMeetDebug(meet, 1, -1, stderr);
    fprintf(stderr, ", ");
    KheMeetDebug(target_meet, 1, -1, stderr);
    fprintf(stderr, ", %d, %s)", offset, preserve_regularity ? "true":"false");
  }
  return KheEjectorR epairEnd(ej, move ? KHE_REPAIR_EJECTING_MEET_MOVE :
    KHE_REPAIR_EJECTING_MEET_ASSIGN, success);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheKem peMeetMoveRepair(KHE_EJECTOR ej, KHE_MEET meet,              */
/*    KHE_MEET target_meet, int offset, bool preserve_regularity,            */
/*    bool *basic, KHE_KEMPE_STATS kempe_stats)                              */
/*                                                                           */
/*  Try a Ke mpe meet move (KHE_REPAIR_KEMPE_MEET_MOVE) of meet to           */
/*  target_meet at offset.                                                   */
/*                                                                           */
/*****************************************************************************/

/* *** replaced by KheMeetMoveRepair with mt == KHE_MOVE_KEMPE
static bool KheKemp eMeetMoveRepair(KHE_EJECTOR ej, KHE_MEET meet,
  KHE_MEET target_meet, int offset, bool preserve_regularity,
  bool *basic, KHE_KEMPE_STATS kempe_stats)
{
  bool success;  int d;
  KheEjectorRe pairBegin(ej);
  success = KheK empeMeetMove(meet, target_meet, offset,
    preserve_regularity, &d, basic, kempe_stats);
  if( DEBUG5 || KheEjectorCurrDebug(ej) )
  {
    fprintf(stderr, "%cK empeMeetMove(", success ? '+' : '-');
    KheMeetDebug(meet, 1, -1, stderr);
    fprintf(stderr, ", ");
    KheMeetDebug(target_meet, 1, -1, stderr);
    fprintf(stderr, ", %d, %s)", offset, preserve_regularity ? "true":"false");
  }
  return KheEjectorRe pairEnd(ej, KHE_REPAIR_KEMPE_MEET_MOVE, success);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheMeetMoveRepair(KHE_EJECTOR ej, KHE_MEET meet,                     */
/*    KHE_MEET target_meet, int offset, KHE_MOVE_TYPE mt,                    */
/*    bool preserve_regularity, bool *basic, KHE_KEMPE_STATS kempe_stats)    */
/*                                                                           */
/*  Try a typed meet move of meet to target_meet at offset.                  */
/*                                                                           */
/*****************************************************************************/

static bool KheMeetMoveRepair(KHE_EJECTOR ej, KHE_MEET meet,
  KHE_MEET target_meet, int offset, KHE_MOVE_TYPE mt,
  bool preserve_regularity, bool *basic, KHE_KEMPE_STATS kempe_stats)
{
  bool success;  int d;
  KheEjectorRepairBegin(ej);
  success = KheTypedMeetMove(meet, target_meet, offset, mt,
    preserve_regularity, &d, basic, kempe_stats);
  if( DEBUG5 || KheEjectorCurrDebug(ej) )
  {
    fprintf(stderr, "%cMeetMove(", success ? '+' : '-');
    KheMeetDebug(meet, 1, -1, stderr);
    fprintf(stderr, ", ");
    KheMeetDebug(target_meet, 1, -1, stderr);
    fprintf(stderr, ", %d, %s, %s)", offset, KheMoveTypeShow(mt),
      preserve_regularity ? "true":"false");
  }
  return KheEjectorRepairEnd(ej, KheMoveTypeToMeetRepairType(mt), success);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheFuzzyMeetMoveRepair(KHE_EJECTOR ej, KHE_MEET meet,               */
/*    KHE_MEET target_meet, int offset)                                      */
/*                                                                           */
/*  Try a fuzzy move (KHE_REPAIR_FUZZY_MEET_MOVE) of meet to target_meet     */
/*  at offset.                                                               */
/*                                                                           */
/*****************************************************************************/

static bool KheFuzzyMeetMoveRepair(KHE_EJECTOR ej, KHE_MEET meet,
  KHE_MEET target_meet, int offset)
{
  bool success;
  HnAssert(KheMeetAsst(meet) != NULL, "KheFuzzyMeetMoveRepair internal error");
  KheEjectorRepairBegin(ej);
  success = KheFuzzyMeetMove(meet, target_meet, offset, 4, 2, 12);
  if( DEBUG5 || KheEjectorCurrDebug(ej) )
  {
    fprintf(stderr, "%cFuzzyMeetMove(", success ? '+' : '-');
    KheMeetDebug(meet, 1, -1, stderr);
    fprintf(stderr, ", ");
    KheMeetDebug(target_meet, 1, -1, stderr);
    fprintf(stderr, ", %d)", offset);
  }
  return KheEjectorRepairEnd(ej, KHE_REPAIR_FUZZY_MEET_MOVE, success);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheNodeMeetSwapRepair(KHE_EJECTOR ej, KHE_NODE node1,KHE_NODE node2)*/
/*                                                                           */
/*  Try a node swap (repair KHE_REPAIR_NODE_MEET_SWAP) of node1 and node2.   */
/*                                                                           */
/*****************************************************************************/

static bool KheNodeMeetSwapRepair(KHE_EJECTOR ej, KHE_NODE node1,KHE_NODE node2)
{
  bool success;
  KheEjectorRepairBegin(ej);
  success = KheNodeMeetSwap(node1, node2);
  if( DEBUG5 || KheEjectorCurrDebug(ej) )
  {
    fprintf(stderr, "%cNodeMeetSwap(", success ? '+' : '-');
    KheNodeDebug(node1, 1, -1, stderr);
    fprintf(stderr, ", ");
    KheNodeDebug(node2, 1, -1, stderr);
    fprintf(stderr, ")");
  }
  return KheEjectorRepairEnd(ej, KHE_REPAIR_NODE_MEET_SWAP, success);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheNodeSwapToSimilarNodeAugment(KHE_EJECTOR ej, KHE_NODE node)      */
/*                                                                           */
/*  Try swapping node with other nodes that it shares a layer with.          */
/*  The node is known to be not visited.                                     */
/*                                                                           */
/*****************************************************************************/

static bool KheNodeSwapToSimilarNodeAugment(KHE_EJECTOR ej, KHE_NODE node)
{
  KHE_LAYER layer;  KHE_NODE node2;  int i, index, extra;
  if( KheNodeParentLayerCount(node) > 0 && !KheNodeVisited(node, 0) )
  {
    KheNodeVisit(node);
    layer = KheNodeParentLayer(node, 0);
    extra = KheEjectorCurrAugmentCount(ej);
    for( i = 0;  i < KheLayerChildNodeCount(layer);  i++ )
    {
      index = (extra + i) % KheLayerChildNodeCount(layer);
      node2 = KheLayerChildNode(layer, index);
      if( node2 != node && KheNodeSameParentLayers(node, node2) &&
	  KheNodeMeetSwapRepair(ej, node, node2) )
	return true;
      if( KheEjectorCurrMayRevisit(ej) )
	KheNodeUnVisit(node);
    }
  }
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheMeetSplitRepair(KHE_EJECTOR ej, KHE_MEET meet, int duration1)    */
/*                                                                           */
/*  Try a meet split (KHE_REPAIR_MEET_SPLIT) of meet.                        */
/*                                                                           */
/*****************************************************************************/

static bool KheMeetSplitRepair(KHE_EJECTOR ej, KHE_MEET meet, int duration1)
{
  bool success;  KHE_MEET meet2;
  KheEjectorRepairBegin(ej);
  success = KheMeetSplit(meet, duration1, true, &meet, &meet2);
  if( DEBUG5 || KheEjectorCurrDebug(ej) )
  {
    fprintf(stderr, "%cMeetSplit(", success ? '+' : '-');
    KheMeetDebug(meet, 1, -1, stderr);
    fprintf(stderr, ", %d)", duration1);
  }
  return KheEjectorRepairEnd(ej, KHE_REPAIR_MEET_SPLIT, success);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheSplitMoveMultiRepair(KHE_EJECTOR ej, KHE_MEET meet,              */
/*    KHE_MEET target_meet, int offset)                                      */
/*                                                                           */
/*  Try two split moves (KHE_REPAIR_SPLIT_MOVE) of meet to target_meet       */
/*  at offset.                                                               */
/*                                                                           */
/*****************************************************************************/

static bool KheSplitMoveMultiRepair(KHE_EJECTOR ej, KHE_MEET meet,
  KHE_MEET target_meet, int offset)
{
  bool success, /* split_success, */ basic;  int d, durn1;  KHE_MEET meet2;
  HnAssert(KheMeetAsst(meet) != NULL, "KheSplitMoveMultiRepair internal error");
  if( KheMeetDuration(meet) >= 2 )
  {
    /* try moving the first half of the split */
    durn1 = 1 + KheEjectorCurrAugmentCount(ej) % (KheMeetDuration(meet)-1);
    if( DEBUG11 )
    {
      fprintf(stderr, "[ KheSplitMoveMultiRepair splitting ");
      KheMeetDebug(meet, 1, -1, stderr);
      fprintf(stderr, " at %d\n", durn1);
    }
    KheEjectorRepairBegin(ej);
    success = KheMeetSplit(meet, durn1, true, &meet, &meet2) &&
      KheTypedMeetMove(meet, target_meet, offset, KHE_MOVE_KEMPE,
	false, &d, &basic, NULL);
    if( DEBUG11 || KheEjectorCurrDebug(ej) )
    {
      fprintf(stderr, "%cSplitMoveA(", success ? '+' : '-');
      KheMeetDebug(meet, 1, -1, stderr);
      fprintf(stderr, ", ");
      KheMeetDebug(target_meet, 1, -1, stderr);
      fprintf(stderr, ", %d)", offset);
    }
    /* ***
    if( DEBUG11 )
      fprintf(stderr, "  1 split_success: %s, success %s\n",
	split_success ? "true" : "false", success ? "true" : "false");
    *** */
    if( KheEjectorRepairEnd(ej, KHE_REPAIR_SPLIT_MOVE, success) )
    {
      if( DEBUG11 )
	fprintf(stderr, "] KheSplitMoveMultiRepair returning true\n");
      return true;
    }

    /* try moving the second half of the split */
    KheEjectorRepairBegin(ej);
    success = KheMeetSplit(meet, durn1, true, &meet, &meet2) &&
      KheTypedMeetMove(meet2, target_meet, offset, KHE_MOVE_KEMPE,
	false, &d, &basic, NULL);
    if( DEBUG11 || KheEjectorCurrDebug(ej) )
    {
      fprintf(stderr, "%cSplitMoveB(", success ? '+' : '-');
      KheMeetDebug(meet, 1, -1, stderr);
      fprintf(stderr, ", ");
      KheMeetDebug(target_meet, 1, -1, stderr);
      fprintf(stderr, ", %d)", offset);
    }
    /* ***
    if( DEBUG11 )
      fprintf(stderr, "  2 split_success: %s, success %s\n",
	split_success ? "true" : "false", success ? "true" : "false");
    *** */
    if( KheEjectorRepairEnd(ej, KHE_REPAIR_SPLIT_MOVE, success) )
    {
      if( DEBUG11 )
	fprintf(stderr, "] KheSplitMoveMultiRepair returning true\n");
      return true;
    }
    if( DEBUG11 )
      fprintf(stderr, "] KheSplitMoveMultiRepair returning false\n");
  }
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheMergeMoveRepair(KHE_EJECTOR ej, KHE_MEET meet1, KHE_MEET meet2,  */
/*    bool after)                                                            */
/*                                                                           */
/*  Try one merge move (KHE_REPAIR_MERGE_MOVE) which moves meet1 to          */
/*  before or after meet2, depending on the after parameter.  Both meets     */
/*  are currently assigned.                                                  */
/*                                                                           */
/*****************************************************************************/

static bool KheMergeMoveRepair(KHE_EJECTOR ej, KHE_MEET meet1, KHE_MEET meet2,
  bool after)
{
  bool success, b;  int d, offset;
  KheEjectorRepairBegin(ej);
  offset = (after ? KheMeetAsstOffset(meet2) + KheMeetDuration(meet2) :
    KheMeetAsstOffset(meet2) - KheMeetDuration(meet1));
  success = KheTypedMeetMove(meet1, KheMeetAsst(meet2), offset, KHE_MOVE_KEMPE,
      false, &d, &b, NULL) && KheMeetMerge(meet1, meet2, true, &meet1);
  if( DEBUG10 || KheEjectorCurrDebug(ej) )
  {
    fprintf(stderr, "%cMergeMove(", success ? '+' : '-');
    KheMeetDebug(meet1, 1, -1, stderr);
    fprintf(stderr, ", (meet2), %s)", after ? "after" : "before");
  }
  return KheEjectorRepairEnd(ej, KHE_REPAIR_MERGE_MOVE, success);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheNodeIsVizier(KHE_NODE node)                                      */
/*                                                                           */
/*  Return true if node is a vizier node.  At present we are deciding        */
/*  this by seeing whether it has a parent and is its parent's sole child.   */
/*                                                                           */
/*****************************************************************************/

static bool KheNodeIsVizier(KHE_NODE node)
{
  KHE_NODE parent_node;
  parent_node = KheNodeParent(node);
  return parent_node != NULL && KheNodeChildCount(parent_node) == 1;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheNodeIsSuitable(KHE_NODE node, bool try_vizier,                   */
/*    KHE_NODE limit_node)                                                   */
/*                                                                           */
/*  Return true if it is acceptable to change the assignments of the         */
/*  meets of node.                                                           */
/*                                                                           */
/*****************************************************************************/

static bool KheNodeIsSuitable(KHE_NODE node, bool try_vizier,
  KHE_NODE limit_node)
{
  /* not acceptable if there is no parent node */
  if( KheNodeParent(node) == NULL )
    return false;

  /* not acceptable if not trying viziers and this is a vizier */
  if( !try_vizier && KheNodeIsVizier(node) )
    return false;

  /* not acceptable if there is a node limit and we have reached it */
  if( node == limit_node )
    return false;

  /* otherwise OK */
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheTryKempe(KHE_MEET meet, KHE_OPTIONS_KEMPE kempe)                 */
/*                                                                           */
/*  Decide whether to try Kempe moves on meet.                               */
/*                                                                           */
/*****************************************************************************/

static bool KheTryKempe(KHE_MEET meet, KHE_OPTIONS_KEMPE kempe)
{
  int i, layer_durn, max_layer_durn, cycle_durn;  KHE_NODE node;
  switch( kempe )
  {
    case KHE_OPTIONS_KEMPE_FALSE:

      return false;

    case KHE_OPTIONS_KEMPE_LARGE_LAYERS:

      node = KheMeetNode(meet);
      if( node != NULL && KheNodeParentLayerCount(node) > 0 )
      {
	max_layer_durn = 0;
	for( i = 0;  i < KheNodeParentLayerCount(node);  i++ )
	{
	  layer_durn = KheLayerDuration(KheNodeParentLayer(node, i));
	  if( layer_durn > max_layer_durn )
	    max_layer_durn = layer_durn;
	}
	cycle_durn = KheInstanceTimeCount(KheSolnInstance(KheMeetSoln(meet)));
	return 10 * max_layer_durn >= 8 * cycle_durn;
      }
      else
	return true;
      break;

    case KHE_OPTIONS_KEMPE_TRUE:

      return true;

    default:

      HnAbort("KheTryKempe internal error");
      return false;  /* keep compiler happy */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheMeetAugment(KHE_EJECTOR ej, KHE_MEET meet,                       */
/*    bool try_vizier, KHE_TRY_MEET_MOVE_FN try_meet_move_fn, void *impl,    */
/*    KHE_OPTIONS_KEMPE try_kempe, bool try_ejecting, bool try_basic,        */
/*    bool try_node_swaps)                                                   */
/*                                                                           */
/*  This wonderfully general partial augment function tries to change the    */
/*  assignment of meet, or one of its ancestors, using meet assignments      */
/*  and moves of various kinds, and node swaps.  The other parameters        */
/*  offer detailed control over exactly what is tried, as follows:           */
/*                                                                           */
/*  try_vizier                                                               */
/*    If true, try assignments and moves of meets in vizier nodes as well    */
/*    as in other nodes.                                                     */
/*                                                                           */
/*  try_meet_move_fn                                                         */
/*    If this function returns true, try meet assignments and moves on       */
/*    meet, and on meet's proper ancestors if ancestors is true.   Exactly   */
/*    which meet assignments and moves depends on the next 3 parameters.     */
/*    If the function is NULL, no time is needed and we try everything.      */
/*                                                                           */
/*  try_kempe                                                                */
/*    If KHE_OPTIONS_KEMPE_FALSE, don't try Kempe moves.                     */
/*    If KHE_OPTIONS_KEMPE_LARGE_LAYERS, try Kempe moves when the meet lies  */
/*    in a layer of relatively large duration.                               */
/*    If KHE_OPTIONS_KEMPE_TRUE, try Kempe moves.                            */
/*                                                                           */
/*  try_ejecting                                                             */
/*    If true, try ejecting assignments or moves (depending on whether meet  */
/*    is unassigned or assigned) when try_meet_move_fn returns true.         */
/*                                                                           */
/*  try_basic                                                                */
/*    If true, try basic assignments or moves (depending on whether meet     */
/*    is unassigned or assigned) when try_meet_move_fn returns true.         */
/*                                                                           */
/*  try_node_swaps                                                           */
/*    If true, try node swaps on meet's node (if any), and on meet's         */
/*    proper ancestors' nodes (if any) if ancestors is true.                 */
/*                                                                           */
/*  Kempe meet moves are always tried before basic and ejecting moves,       */
/*  and basic and ejecting moves are skipped if a corresponding Kempe        */
/*  move was tried and indicated (via its "basic" parameter) that it         */
/*  did what a basic or ejecting move would do.                              */
/*                                                                           */
/*  This function also consults the time_limit_node option to ensure         */
/*  that any meets whose assignments it changes lie within the limit.        */
/*                                                                           */
/*****************************************************************************/

static bool KheMeetAugment(KHE_EJECTOR ej, KHE_MEET meet,
  bool try_vizier, KHE_TRY_MEET_MOVE_FN try_meet_move_fn, void *impl,
  KHE_OPTIONS_KEMPE try_kempe, bool try_ejecting, bool try_basic,
  bool try_node_swaps)
{
  int base, i, max_offset, offset, offs, index, extra;
  KHE_MEET anc_meet, target_meet;
  KHE_NODE node, limit_node, parent_node;  KHE_TIME time;
  bool basic, try_meet_moves;
  bool nodes_before_meets = KheEjectorNodesBeforeMeets(ej);
  bool try_fuzzy_moves = KheEjectorUseFuzzyMoves(ej);
  bool try_split_moves = KheEjectorUseSplitMoves(ej);
  KHE_KEMPE_STATS kempe_stats = KheEjectorKempeStats(ej);
  if( DEBUG4 )
  {
    fprintf(stderr, "%*s[ KheMeetAugment(",KheEjectorCurrDebugIndent(ej)+2, "");
    KheMeetDebug(meet, 1, -1, stderr);
    fprintf(stderr, ")\n");
  }

  /* follow the chain upwards to the first ancestor lying in a node */
  base = 0;
  anc_meet = meet;
  while( anc_meet != NULL && KheMeetNode(anc_meet) == NULL )
  {
    base += KheMeetAsstOffset(anc_meet);
    anc_meet = KheMeetAsst(anc_meet);
  }
  if( anc_meet == NULL )
  {
    if( DEBUG4 )
      fprintf(stderr,"%*s] KheMeetAugment ret false (no movable ancestor)\n",
	KheEjectorCurrDebugIndent(ej) + 2, "");
    return false;
  }

  /* make sure anc_meet's node is in scope */
  node = KheMeetNode(anc_meet);
  limit_node = KheEjectorLimitNode(ej);
  if( limit_node != NULL && !KheNodeIsProperDescendant(node, limit_node) )
  {
    if( DEBUG4 )
      fprintf(stderr,"%*s] KheMeetAugment ret false (not in node scope)\n",
	KheEjectorCurrDebugIndent(ej) + 2, "");
    return false;
  }

  /* try repairs at each ancestor of meet lying in a suitable node */
  try_meet_moves = (try_kempe != KHE_OPTIONS_KEMPE_FALSE) ||
    try_ejecting || try_basic;
  while( anc_meet != NULL )
  {
    node = KheMeetNode(anc_meet);
    if( node != NULL && KheNodeIsSuitable(node, try_vizier, limit_node) )
    {
      /* if nodes before meets, try node swaps of anc_meet's node */
      if( nodes_before_meets && try_node_swaps &&
	  KheNodeSwapToSimilarNodeAugment(ej, node) )
      {
	if( DEBUG4 )
	  fprintf(stderr, "%*s] KheMeetAugment ret true (node)\n",
	    KheEjectorCurrDebugIndent(ej) + 2, "");
	return true;
      }

      /* try meet moves of anc_meet */
      if( try_meet_moves && !KheMeetVisited(anc_meet, 0) )
      {
	KheMeetVisit(anc_meet);
	parent_node = KheNodeParent(node);
	extra = KheEjectorCurrAugmentCount(ej) +
	  5 * KheSolnDiversifier(KheEjectorSoln(ej));
	for( i = 0;  i < KheNodeMeetCount(parent_node);  i++ )
	{
	  index = (i + extra) % KheNodeMeetCount(parent_node);
	  target_meet = KheNodeMeet(parent_node, index);
	  time = KheMeetAsstTime(target_meet);
	  if( try_meet_move_fn == NULL || time != NULL )
	  {
	    max_offset = KheMeetDuration(target_meet)-KheMeetDuration(anc_meet);
	    for( offset = 0;  offset <= max_offset;  offset++ )
	    {
	      offs = (offset + extra) % (max_offset + 1);
	      if( try_meet_move_fn == NULL || try_meet_move_fn(meet,
		    KheTimeNeighbour(time, base + offs), impl) )
	      {
		if( KheMeetAsst(anc_meet) == NULL )
		{
		  /* meet assignments (not moves) are allowed; try ejecting */
		  basic = false;
		  if( try_ejecting && KheMeetMoveRepair(ej, anc_meet,
		      target_meet, offs, KHE_MOVE_EJECTING, true, &basic,NULL) )
		  {
		    if( DEBUG4 )
		      fprintf(stderr,
			"%*s] KheMeetAugment ret true (ejecting asst)\n",
			KheEjectorCurrDebugIndent(ej) + 2, "");
		    return true;
		  }

		  /* try basic, unless already tried by ejecting */
		  if( !basic && try_basic && KheMeetMoveRepair(ej, anc_meet,
		      target_meet, offs, KHE_MOVE_UNCHECKED, true,&basic,NULL) )
		  {
		    if( DEBUG4 )
		      fprintf(stderr,
			"%*s] KheMeetAugment ret true (basic asst)\n",
			KheEjectorCurrDebugIndent(ej) + 2, "");
		    return true;
		  }
		}
		else
		{
		  /* meet moves (not assignments) are allowed; try kempe */
		  basic = false;
		  if( KheTryKempe(anc_meet, try_kempe) &&
		      KheMeetMoveRepair(ej, anc_meet, target_meet,
			offs, KHE_MOVE_KEMPE, true, &basic, kempe_stats) )
		  {
		    if( DEBUG4 )
		      fprintf(stderr,"%*s] KheMeetAugment ret true (kempe)\n",
			KheEjectorCurrDebugIndent(ej) + 2, "");
		    return true;
		  }

		  /* try ejecting, unless already tried by kempe */
		  /* ***
		  if( KheEjectorCurrDepth(ej) == 1 )
		    KheSolnNew GlobalVisit(KheMeetSoln(meet));
		  *** */
		  if( !basic && try_ejecting &&
		      KheMeetMoveRepair(ej, anc_meet, target_meet, offs,
			KHE_MOVE_EJECTING, true, &basic, NULL) )
		  {
		    if( DEBUG4 )
		     fprintf(stderr,"%*s] KheMeetAugment ret true (ejecting)\n",
			KheEjectorCurrDebugIndent(ej) + 2, "");
		    return true;
		  }

		  /* try basic, unless already tried by kempe or ejecting */
		  if( !basic && try_basic && KheMeetMoveRepair(ej, anc_meet,
		      target_meet, offs, KHE_MOVE_UNCHECKED, true,&basic,NULL) )
		  {
		    if( DEBUG4 )
		      fprintf(stderr,"%*s] KheMeetAugment ret true (basic)\n",
			KheEjectorCurrDebugIndent(ej) + 2, "");
		    return true;
		  }

		  /* try fuzzy, if allowed and depth is 1 */
		  if( try_fuzzy_moves && KheEjectorCurrDepth(ej) == 1 &&
		      KheFuzzyMeetMoveRepair(ej, meet, target_meet, offs) )
		  {
		    if( DEBUG4 )
		      fprintf(stderr,"%*s] KheMeetAugment ret true (fuzzy)\n",
			KheEjectorCurrDebugIndent(ej) + 2, "");
		    return true;
		  }

		  /* try split, if allowed and depth is 1 */
		  if( try_split_moves && KheEjectorCurrDepth(ej) == 1 &&
		      KheSplitMoveMultiRepair(ej, meet, target_meet, offs) )
		  {
		    if( DEBUG4 )
		      fprintf(stderr,"%*s] KheMeetAugment ret true (split)\n",
			KheEjectorCurrDebugIndent(ej) + 2, "");
		    return true;
		  }
		}
	      }
	    }
	  }
	}
	if( KheEjectorCurrMayRevisit(ej) )
	  KheMeetUnVisit(anc_meet);
      }

      /* if nodes after meets, try node swaps of anc_meet's node */
      if( !nodes_before_meets && try_node_swaps &&
	  KheNodeSwapToSimilarNodeAugment(ej, node) )
      {
	if( DEBUG4 )
	  fprintf(stderr, "%*s] KheMeetAugment ret true (node)\n",
	    KheEjectorCurrDebugIndent(ej) + 2, "");
	return true;
      }
    }
    base += KheMeetAsstOffset(anc_meet);
    anc_meet = KheMeetAsst(anc_meet);
  } 
  if( DEBUG4 )
    fprintf(stderr,"%*s] KheMeetAugment ret false\n",
      KheEjectorCurrDebugIndent(ej) + 2, "");
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "task sequence moves"                                          */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskSequenceMoveRepairEnd(KHE_EJECTOR ej, KHE_TASK_SET ts_before,*/
/*    int before_len, KHE_TASK task, KHE_TASK_SET ts_after, int after_len,   */
/*    KHE_RESOURCE from_r, KHE_RESOURCE to_r, KHE_MOVE_TYPE mt, bool success)*/
/*                                                                           */
/*  Called to finish off KheTaskSequenceMoveRepair (just below).             */
/*                                                                           */
/*****************************************************************************/
/* static KHE_REPAIR_TYPE KheMoveTypeToTaskSetRepairType(KHE_MOVE_TYPE mt); */

/* ***
static bool KheTaskSequenceMoveRepairEnd(KHE_EJECTOR ej, KHE_TASK_SET ts_before,
  int before_len, KHE_TASK task, KHE_TASK_SET ts_after, int after_len,
  KHE_RESOURCE from_r, KHE_RESOURCE to_r, KHE_MOVE_TYPE mt, bool success)
{
  KHE_TASK task2;  int i;
  if( DEBUG18 || KheEjectorCurrDebug(ej) )
  {
    fprintf(stderr, "%cTaskSequenceMove([", success ? '+' : '-');
    for( i = before_len - 1;  i >= 0;  i-- )
    {
      task2 = KheTaskSetTask(ts_before, i);
      KheTaskDebug(task2, 1, -1, stderr);
      fprintf(stderr, ", ");
    }
    fprintf(stderr, "|");
    KheTaskDebug(task, 1, -1, stderr);
    fprintf(stderr, "|");
    for( i = 0;  i < after_len;  i++ )
    {
      task2 = KheTaskSetTask(ts_after, i);
      fprintf(stderr, ", ");
      KheTaskDebug(task2, 1, -1, stderr);
    }
    fprintf(stderr, "], %s -> %s, %s)",
      from_r == NULL ? "-" : KheResourceId(from_r),
      to_r == NULL ? "-" : KheResourceId(to_r), KheMoveTypeShow(mt));
  }
  return KheEjectorRepa irEnd(ej, KheMoveTypeToTaskSetRepairType(mt), success);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheTimeGroupCont ainsAssignedTask(KHE_TIME_GROUP tg,                */
/*    KHE_RESOURCE_TIMETABLE_MONITOR rtm, KHE_TASK *res)                     */
/*                                                                           */
/*  If tg contains an assigned task currently assigned rtm's resource, set   */
/*  *res to one such task and return true, otherwise return false.  Use rtm  */
/*  to find these tasks.                                                     */
/*                                                                           */
/*  Preassigned tasks are not included, because these tasks are being        */
/*  gathered so that they can be moved, and preassigned tasks should         */
/*  not be moved.                                                            */
/*                                                                           */
/*****************************************************************************/

/* *** no longer used
static bool KheTimeGroupCont ainsAssignedTask(KHE_TIME_GROUP tg,
  KHE_RESOURCE_TIMETABLE_MONITOR rtm, KHE_TASK *res)
{
  int i;  KHE_TIME t;  KHE_TASK task;  KHE_RESOURCE r;
  for( i = 0;  i < KheTimeGroupTimeCount(tg);  i++ )
  {
    t = KheTimeGroupTime(tg, i);
    if( KheResourceTimetableMonitorTimeTaskCount(rtm, t) > 0 )
    {
      task = KheResourceTimetableMonitorTimeTask(rtm, t, 0);
      if( KheTaskIsPreassigned(task, &r) )
	return *res = NULL, false;
      else
	return *res = task, true;
    }
  }
  return *res = NULL, false;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheFindAssi gnedTasks(KHE_EJECTOR ej, int index, int max_len,       */
/*    KHE_RESOURCE r, KHE_TASK_SET ts_before, KHE_TASK_SET ts_after)         */
/*                                                                           */
/*  Find one assigned task assigned to r in each of the up to max_len time   */
/*  groups before and after the time group at index in the days frame, and   */
/*  add them to ts_before and ts_after.                                      */
/*                                                                           */
/*****************************************************************************/

/* *** no longer used
static void KheFindAss ignedTasks(KHE_EJECTOR ej, int index, int max_before,
  int max_after, KHE_RESOURCE r, KHE_TASK_SET ts_before, KHE_TASK_SET ts_after)
{
  KHE_SOLN soln;  KHE_TIME_GROUP tg;  KHE_POLARITY po;  int i, count;
  KHE_TASK task;  KHE_FRAME days_frame;  KHE_RESOURCE_TIMETABLE_MONITOR rtm;

  ** boilerplate **
  soln = KheEjectorSoln(ej);
  days_frame = KheEjectorFrame(ej);
  rtm = KheResourceTimetableMonitor(soln, r);

  ** build ts_before **
  for( i = 1;  i <= max_before && index - i >= 0;  i++ )
  {
    tg = KheFrameTimeGroup(days_frame, index - i, &po);
    if( !KheTimeGroupCon tainsAssignedTask(tg, rtm, &task) )
      break;
    KheTaskSetAddTask(ts_before, task);
  }

  ** build ts_after **
  count = KheFrameTimeGroupCount(days_frame);
  for( i = 1;  i <= max_after && index + i < count;  i++ )
  {
    tg = KheFrameTimeGroup(days_frame, index + i, &po);
    if( !KheTimeGroupCont ainsAssignedTask(tg, rtm, &task) )
      break;
    KheTaskSetAddTask(ts_after, task);
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheFindUnassignedTasks(KHE_EJECTOR ej, int index, int max_len,      */
/*    KHE_RESOURCE_TYPE rt, KHE_TASK_SET ts_before, KHE_TASK_SET ts_after)   */
/*                                                                           */
/*  Find one unassigned task of type rt in each of the up to max_len time    */
/*  groups before and after the time group at index in the days frame, and   */
/*  return them in *ts_before and *ts_after.                                 */
/*                                                                           */
/*****************************************************************************/

/* *** no longer used
static void KheFindUnassignedTasks(KHE_EJECTOR ej, int index, int max_before,
  int max_after, KHE_RESOURCE_TYPE rt, KHE_TASK_SET ts_before,
  KHE_TASK_SET ts_after)
{
  KHE_TIME_GROUP tg;  KHE_POLARITY po;  int i, count;
  KHE_TASK task;  KHE_FRAME days_frame; KHE_EVENT_TIMETABLE_MONITOR etm;

  ** boilerplate **
  days_frame = KheEjectorFrame(ej);
  et m = KheEjectorEventTimetableMonitor(ej);

  ** build ts_before **
  for( i = 1;  i <= max_before && index - i >= 0;  i++ )
  {
    tg = KheFrameTimeGroup(days_frame, index - i, &po);
    if( !KheTimeGroup ContainsUnassignedTask(tg, rt, etm, &task) )
      break;
    KheTaskSetAddTask(ts_before, task);
  }

  ** build ts_after **
  count = KheFrameTimeGroupCount(days_frame);
  for( i = 1;  i <= max_after && index + i < count;  i++ )
  {
    tg = KheFrameTimeGroup(days_frame, index + i, &po);
    if( !KheTimeGroupConta insUnassignedTask(tg, rt, etm, &task) )
      break;
    KheTaskSetAddTask(ts_after, task);
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  int KheMaxFreeLen(KHE_TASK_SET ts, KHE_RESOURCE r)                       */
/*                                                                           */
/*  Return the length of the longest initial subsequence of the tasks        */
/*  of ts that can be assigned to r without causing a clash.  If r is        */
/*  NULL, this is the full length.                                           */
/*                                                                           */
/*****************************************************************************/

/* *** no longer used
static int KheMaxFreeLen(KHE_EJECTOR ej, KHE_TASK_SET ts, KHE_RESOURCE r)
{
  KHE_RESOURCE_TIMETABLE_MONITOR rtm;  int count, i;  KHE_TASK task;
  KHE_SOLN soln;  KHE_FRAME frame;
  count = KheTaskSetTaskCount(ts);
  if( r == NULL || count == 0 )
    return count;
  soln = KheTaskSoln(KheTaskSetTask(ts, 0));
  rtm = KheResourceTimetableMonitor(soln, r);
  frame = KheEjectorFrame(ej);
  for( i = 0;  i < count;  i++ )
  {
    task = KheTaskSetTask(ts, i);
    ** ***
    if( KheEjectorCurrDebug(ej) )
    {
      fprintf(stderr, "  KheMaxFreeLen(");
      KheTaskSetDebug(ts, 1, -1, stderr);
      fprintf(stderr, ", %s) avail at %s: %s\n", KheResourceId(r), KheTimeId(t),
        KheResourceTimetableMonitorTaskAvailableInFrame(rtm, task, frame) ?
	"true" : "false");
    }
    *** **
    if( !KheResourceTimetableMonitorTaskAvailableInFrame(rtm, task, frame) )
      return i;
  }
  return count;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  int KheMaxBusyLen(KHE_TASK_SET ts, KHE_RESOURCE r)                      */
/*                                                                           */
/*  Return the length of the longest initial subsequence of the tasks        */
/*  of ts that can be assigned to r without causing a clash.  If r is        */
/*  NULL, this is the full length.                                           */
/*                                                                           */
/*****************************************************************************/

/* *** no longer used
static int KheMaxBusyLen(KHE_EJECTOR ej, KHE_TASK_SET ts, KHE_RESOURCE r)
{
  KHE_RESOURCE_TIMETABLE_MONITOR rtm;  int count, i;  KHE_TASK task;
  KHE_SOLN soln;  KHE_FRAME frame;
  count = KheTaskSetTaskCount(ts);
  if( r == NULL || count == 0 )
    return count;
  soln = KheTaskSoln(KheTaskSetTask(ts, 0));
  rtm = KheResourceTimetableMonitor(soln, r);
  frame = KheEjectorFrame(ej);
  for( i = 0;  i < count;  i++ )
  {
    task = KheTaskSetTask(ts, i);
    if( KheResourceTimetableMonitorTaskAvailableInFrame(rtm, task, frame) )
      return i;
  }
  return count;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheFreeForTask(KHE_EJECTOR ej, KHE_TASK task, KHE_RESOURCE r)       */
/*                                                                           */
/*  Return true if r is available for assignment to task.  If r is NULL,     */
/*  it is considered to be available.                                        */
/*                                                                           */
/*****************************************************************************/

/* *** no longer used
static bool KheFreeForTask(KHE_EJECTOR ej, KHE_TASK task, KHE_RESOURCE r)
{
  KHE_RESOURCE_TIMETABLE_MONITOR rtm;  KHE_SOLN soln;  KHE_FRAME frame;
  if( r == NULL )
    return true;
  soln = KheTaskSoln(task);
  rtm = KheResourceTimetableMonitor(soln, r);
  frame = KheEjectorFrame(ej);
  return KheResourceTimetableMonitorTaskAvailableInFrame(rtm, task, frame);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheTaskSequenceDebug(KHE_TASK_SET ts_before, int before_len,        */
/*    KHE_TASK task, KHE_TASK_SET ts_after, int after_len, FILE *fp)         */
/*                                                                           */
/*  Debug print of task sequence.                                            */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheTaskSequenceDebug(KHE_TASK_SET ts_before, int before_len,
  KHE_TASK_SET ts, KHE_TASK_SET ts_after, int after_len, FILE *fp)
{
  int i;  KHE_TASK task;
  fprintf(fp, "[");
  for( i = before_len - 1;  i >= 0;  i-- )
  {
    task = KheTaskSetTask(ts_before, i);
    KheTaskDebug(task, 1, -1, fp);
    fprintf(fp, ", ");
  }
  for( i = 0;  i < KheTaskSetTaskCount(ts);  i++ )
  {
    task = KheTaskSetTask(ts, i);
    if( i > 0 )
      fprintf(fp, ", ");
    fprintf(fp, "@");
    KheTaskDebug(task, 1, -1, fp);
  }
  for( i = 0;  i < after_len;  i++ )
  {
    task = KheTaskSetTask(ts_after, i);
    fprintf(fp, ", ");
    KheTaskDebug(task, 1, -1, fp);
  }
  fprintf(fp, "]");
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskSequenceMove(KHE_EJECTOR ej, KHE_TASK_SET ts_before,         */
/*    int before_len, KHE_TASK task, KHE_TASK_SET ts_after, int after_len,   */
/*    KHE_RESOURCE to_r)                                                     */
/*                                                                           */
/*  Move to to_r the first before_len tasks of ts_before, task, and the      */
/*  first after_len tasks of ts_after.  Return true if successful.           */
/*                                                                           */
/*****************************************************************************/

/* ***
static bool KheTaskSequenceMove(KHE_EJECTOR ej, KHE_TASK_SET ts_before,
  int before_len, KHE_TASK_SET ts, KHE_TASK_SET ts_after, int after_len,
  KHE_RESOURCE to_r)
{
  int i;  KHE_TASK task;  KHE_FRAME frame;
  HnAssert(before_len <= KheTaskSetTaskCount(ts_before),
    "KheTaskSequenceMove internal error 1");
  frame = KheEjectorFrame(ej);
  for( i = before_len - 1;  i >= 0;  i-- )
  {
    task = KheTaskSetTask(ts_before, i);
    if( !KheTypedTaskMoveFrame(task, to_r, KHE_MOVE_CHECKED, frame) )
      return false;
  }

  ** move ts, return false if can't **
  for( i = 0;  i < KheTaskSetTaskCount(ts);  i++ )
  {
    task = KheTaskSetTask(ts, i);
    if( !KheTypedTaskMoveFrame(task, to_r, KHE_MOVE_CHECKED, frame) )
      return false;
  }

  ** move the after ones, return false if can't **
  HnAssert(after_len <= KheTaskSetTaskCount(ts_after),
    "KheTaskSequenceMove internal error 2");
  for( i = 0;  i < after_len;  i++ )
  {
    task = KheTaskSetTask(ts_after, i);
    if( !KheTypedTaskMoveFrame(task, to_r, KHE_MOVE_CHECKED, frame) )
      return false;
  }

  ** all good **
  return true;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskSequenceMoveRepair(KHE_EJECTOR ej,                           */
/*    KHE_TASK_SET ts_before, int before_len, KHE_TASK task,                 */
/*    KHE_TASK_SET ts_after, int after_len, KHE_RESOURCE from_r,             */
/*    KHE_RESOURCE to_r)                                                     */
/*                                                                           */
/*  Carry out one repair that moves tasks ts_before[0 .. before_len - 1],    */
/*  task, and ts_after[0 .. after_len - 1] to to_r.  At a minimum, just      */
/*  task is moved, although the current plan is that there will always be    */
/*  at least two tasks tried.  The tasks are currently assigned to from_r,   */
/*  which may be NULL.                                                       */
/*                                                                           */
/*****************************************************************************/

/* ***
static bool KheTaskSequenceMoveRepair(KHE_EJECTOR ej,
  KHE_TASK_SET ts_before, int before_len, KHE_TASK_SET ts,
  KHE_TASK_SET ts_after, int after_len, KHE_RESOURCE from_r,
  KHE_RESOURCE to_r)
{
  bool success;
  KheEjectorRepairB egin(ej);
  success = KheTaskSequenceMove(ej, ts_before, before_len, ts, ts_after,
    after_len, to_r);
  if( DEBUG18 || KheEjectorCurrDebug(ej) )
  {
    fprintf(stderr, "%cTaskSequenceMove(", success ? '+' : '-');
    KheTaskSequenceDebug(ts_before, before_len, ts, ts_after, after_len,
      stderr);
    fprintf(stderr, ", %s -> %s)",
      from_r == NULL ? "-" : KheResourceId(from_r),
      to_r == NULL ? "-" : KheResourceId(to_r));
  }
  return KheEjectorRe pairEnd(ej, KHE_REPAIR_TASK_SET_MOVE_EJECTING, success);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheTaskSetDoDebug(KHE_TASK_SET ts, FILE *fp)                        */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheTaskSetDoDebug(KHE_TASK_SET ts, FILE *fp)
{
  int i;  KHE_TASK task;
  for( i = 0;  i < KheTaskSetTaskCount(ts);  i++ )
  {
    task = KheTaskSetTask(ts, i);
    if( i == 0 )
      fprintf(fp, ", ");
    KheTaskDebug(task, 1, -1, fp);
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskSequenceSwapRepair(KHE_EJECTOR ej,                           */
/*    KHE_TASK_SET ts_before, int before_len, KHE_TASK task,                 */
/*    KHE_TASK_SET ts_after, int after_len, KHE_RESOURCE from_r,             */
/*    KHE_RESOURCE to_r, KHE_FRAME frame2)                                   */
/*                                                                           */
/*  Carry out one task sequence swap repair.  This moves task et al from     */
/*  from_r to to_r, and to_r's tasks in frame2 to from_r.                    */
/*                                                                           */
/*****************************************************************************/
/* ***
static bool KheFrameMove(KHE_FRAME frame, KHE_RESOURCE from_r,
  KHE_RESOURCE to_r, KHE_FRAME days_frame);
*** */

/* ***
static bool KheTaskSequenceSwapRepair(KHE_EJECTOR ej,
  KHE_TASK_SET ts_before, int before_len, KHE_TASK_SET ts,
  KHE_TASK_SET ts_after, int after_len, KHE_RESOURCE from_r,
  KHE_RESOURCE to_r, KHE_TASK_SET back_ts)
{
  bool success;  KHE_FRAME days_frame;
  KheEjectorRepai rBegin(ej);
  days_frame = KheEjectorFrame(ej);
  success = KheTaskSequenceMove(ej, ts_before, before_len, ts, ts_after,
    after_len, to_r) && KheTypedTaskSetMoveFrame(back_ts, from_r,
    KHE_MOVE_CHECKED, days_frame);
  if( DEBUG18 || KheEjectorCurrDebug(ej) )
  {
    fprintf(stderr, "%cTaskSequenceSwap(", success ? '+' : '-');
    KheTaskSequenceDebug(ts_before, before_len, ts, ts_after, after_len,
      stderr);
    fprintf(stderr, ", %s -> %s, ",
      from_r == NULL ? "-" : KheResourceId(from_r),
      to_r == NULL ? "-" : KheResourceId(to_r));
    KheTaskSetDoDebug(back_ts, stderr);
    fprintf(stderr, ")");
  }
  return KheEjectorR epairEnd(ej, KHE_REPAIR_TASK_SET_MOVE_EJECTING, success);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskSequenceDoMoveAndSwapMultiRepair(KHE_EJECTOR ej,             */
/*    KHE_TASK_SET ts_before, int before_len, KHE_TASK task,                 */
/*    KHE_TASK_SET ts_after, int after_len, KHE_RESOURCE from_r,             */
/*    KHE_RESOURCE to_r, KHE_MOVE_TYPE mt)                                   */
/*                                                                           */
/*  Carry out one repair that moves tasks ts_before[0 .. before_len - 1],    */
/*  task, and ts_after[0 .. after_len - 1] to to_r.  At a minimum, task      */
/*  alone is moved.  All tasks are currently assigned from_r, possibly NULL. */
/*                                                                           */
/*  Also try swaps when from_r and to_r are both non-NULL.                   */
/*                                                                           */
/*****************************************************************************/
/* ***
static bool KheFrameSwapAugment(KHE_EJECTOR ej, KHE_FRAME frame1,
  KHE_FRAME frame2, KHE_RESOURCE from_r, KHE_RESOURCE to_r, KHE_MOVE_TYPE mt);
*** */

/* *** no longer used
static bool KheTaskSequenceDoMoveAndSwapMultiRepair(KHE_EJECTOR ej,
  KHE_TASK_SET ts_before, int before_len, KHE_TASK task,
  KHE_TASK_SET ts_after, int after_len, KHE_RESOURCE from_r,
  KHE_RESOURCE to_r, bool with_swaps)
{
  int len, extra, count;  KHE_FRAME frame2, active_frame;
  struct khe_frame_iterator_rec fi_rec;

  ** try a move **
  if( KheTaskSequenceMoveRepair(ej, ts_before, before_len, task,
	ts_after, after_len, from_r, to_r) )
    return true;

  ** try swaps **
  if( from_r != NULL && to_r != NULL && with_swaps )
  {
    ** iterate over the frame, finding active intervals for to_r **
    len = before_len + 1 + after_len;
    extra =  KheEjectorCurrAugmentCount(ej) +
      5 * KheSolnDiversifier(KheEjectorSoln(ej));
    KheFrameIter atorInit(&fi_rec, KheEjectorFrame(ej), to_r, extra);
    count = 0;
    while( count < 20 && KheFrameIteratorNext(&fi_rec, &active_frame) )
    {
      if( KheFrameTimeGroupCount(active_frame) >= len )
      {
	frame2 = KheFrameInsideBeforeSlice(active_frame, len);
	if( KheFrameIsIna ctive(frame2, from_r) )
	{
	  count++;
	  if( KheTaskSequenceSwapRepair(ej, ts_before, before_len, task,
	      ts_after, after_len, from_r, to_r, frame2) )
	    return true;
	}

	if( KheFrameTimeGroupCount(active_frame) > len )
	{
	  frame2 = KheFrameInsideAfterSlice(active_frame, len);
	  if( KheFrameIsInac tive(frame2, from_r) )
	  {
	    count++;
	    if( KheTaskSequenceSwapRepair(ej, ts_before, before_len, task,
		ts_after, after_len, from_r, to_r, frame2) )
	      return true;
	  }
	}
      }
    }
  }
  return false;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskSequenceDoSwapRepair(KHE_EJECTOR ej, KHE_TASK_SET ts_before, */
/*    int move_before, KHE_TASK task, KHE_TASK_SET ts_after, int move_after, */
/*    KHE_RESOURCE from_r, KHE_RESOURCE to_r)                                */
/*                                                                           */
/*  A repair which swaps tasks ts_before[0 .. before_len - 1], task, and     */
/*  ts_after[0 .. after_len - 1] (which are all initially assigned from_r)   */
/*  with whatever tasks are assigned to_r on the same days.                  */
/*                                                                           */
/*****************************************************************************/

/* *** no longer used
static bool KheTaskSequenceDoSwapRepair(KHE_EJECTOR ej, KHE_TASK_SET ts_before,
  int move_before, KHE_TASK task, KHE_TASK_SET ts_after, int move_after,
  KHE_RESOURCE from_r, KHE_RESOURCE to_r)
{
  KHE_FRAME frame, frame2;  KHE_TIME t;  int index;
  t = KheMeetAsstTime(KheTaskMeet(task));
  frame = KheEjectorFrame(ej);
  if( t == NULL || KheFrameIsNull(frame) )
    return false;
  index = KheFrameTimeIndex(frame, t);
  frame2 = KheFrameSlice(frame, index - move_before, index + move_after);
  HnAssert(index - move_before >= 0,
    "KheTaskSequenceDoSwapRepair internal error 1");
  HnAssert(index + move_after < KheFrameTimeGroupCount(frame),
    "KheTaskSequenceDoSwapRepair internal error 2");
  return KheTaskSequenceSwapRepair(ej, ts_before, move_before, task,
    ts_after, move_after, from_r, to_r, frame2);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskSequenceMoveAndSwapMultiRepair(KHE_EJECTOR ej, KHE_TASK task,*/
/*    KHE_RESOURCE_GROUP rg, KHE_RESOURCE_GROUP not_rg, bool allow_unassign, */
/*    int max_before, int max_after, KHE_MOVE_TYPE mt)                       */
/*                                                                           */
/*  Try a set of task sequence moves.  Each move moves at least 1 task,      */
/*  including task, from adjacent time groups in the common frame, to a      */
/*  resource of rg or (if allow_unassign) to NULL.  At most max_before       */
/*  of the tasks come from before task, and at most max_after come from      */
/*  after task.                                                              */
/*                                                                           */
/*  If rg is a singleton resource group, the move is attempting to           */
/*  repair an underload of that resource at the time of task, and so         */
/*  all the tasks moved must not be assigned that task.                      */
/*                                                                           */
/*  When task is assigned a resource initially, it is trivial to identify    */
/*  other tasks that should be moved along with it:  they must all be        */
/*  assigned that resource.  When task is not assigned a resource initially  */
/*  the other tasks must also be unassigned, but in general there will be    */
/*  choices as to which unassigned tasks to try.  As a heuristic we          */
/*  prefer unassigned tasks at the same offset in their meets as task.       */
/*                                                                           */
/*****************************************************************************/

/* *** no longer used
static bool KheTaskSequenceMoveAndSwapMultiRepair(KHE_EJECTOR ej, KHE_TASK task,
  KHE_RESOURCE_GROUP rg, KHE_RESOURCE_GROUP not_rg, bool allow_unassign)
{
  struct khe_se_resource_set_type srs_rec;  KHE_RESOURCE from_r, to_r;
  int index, extra, r_max_len, len, before_len, after_len;
  int move_before, move_after, task_pos ** , days_count **;
  KHE_TASK_SET ts_before, ts_after;  KHE_SOLN soln;  KHE_TIME t;
  KHE_FRAME days_frame;  bool with_swaps, free_for_task;

  ** get assigned time and frame and exit early if not present **
  t = KheMeetAsstTime(KheTaskMeet(task));
  days_frame = KheEjectorFrame(ej);
  ** days_count = KheFrameTimeGroupCount(days_frame); **
  if( t == NULL || KheFrameIsNull(days_frame) )
    return false;
  with_swaps = !KheEjectorTaskSeqSwapsOff(ej);

  ** get task sets from before and after **
  soln = KheEjectorSoln(ej);
  ts_before = KheTaskSetMake(soln);
  ts_after = KheTaskSetMake(soln);
  index = KheFrameTimeIndex(days_frame, t);
  from_r = KheTaskAsstResource(task);
  if( from_r != NULL )
    KheFindAssign edTasks(ej, index, MAX_MOVE, MAX_MOVE, from_r,
      ts_before, ts_after);
  else
    KheFindUnassignedTasks(ej, index, MAX_MOVE, MAX_MOVE,
      KheTaskResourceType(task), ts_before, ts_after);

  ** iterate over resources, moving subsets of the task sets **
  extra = KheEjectorCurrAugmentCount(ej) +
    5 * KheSolnDiversifier(KheEjectorSoln(ej));
  KheForEachResource(srs_rec, rg, not_rg, allow_unassign, extra, to_r)
    if( to_r != from_r )
    {
      free_for_task = KheFreeForTask(ej, task, to_r);
      if( true || free_for_task )
      {
	before_len = KheMaxFreeLen(ej, ts_before, to_r);
	after_len = KheMaxFreeLen(ej, ts_after, to_r);
	r_max_len = before_len + 1 + after_len;
	for( len = 1;  len <= r_max_len;  len++ )
	{
	  ** now task can be at position 0, 1, ... , len-1 of the moved tasks **
	  for( task_pos = 0;  task_pos < len;  task_pos++ )
	  {
	    ** must be enough to move on each side **
	    move_before = task_pos;
	    move_after = len - task_pos - 1;
	    if( move_before <= before_len && move_after <= after_len &&
		KheTaskSequenceDoMoveAndSwapMultiRepair(ej, ts_before,
		  move_before, task, ts_after, move_after, from_r, to_r,
		  with_swaps) )
	    {
	      KheTaskSetDelete(ts_before);
	      KheTaskSetDelete(ts_after);
	      return true;
	    }
	  }
	}
      }

      if( !free_for_task && from_r != NULL && to_r != NULL )
      {
	** swap from_r and to_r  **
	before_len = KheMaxBusyLen(ej, ts_before, to_r);
	after_len = KheMaxBusyLen(ej, ts_after, to_r);
	r_max_len = before_len + 1 + after_len;
	for( len = 1;  len <= r_max_len;  len++ )
	{
	  ** now task can be at position 0, 1, ... , len-1 of the moved tasks **
	  for( task_pos = 0;  task_pos < len;  task_pos++ )
	  {
	    ** must be enough to move on each side **
	    move_before = task_pos;
	    move_after = len - task_pos - 1;
	    if( move_before <= before_len && move_after <= after_len &&
		KheTaskSequenceDoSwapRepair(ej, ts_before, move_before,
		  task, ts_after, move_after, from_r, to_r) )
	    {
	      KheTaskSetDelete(ts_before);
	      KheTaskSetDelete(ts_after);
	      return true;
	    }
	  }
	}
      }
    }
  KheTaskSetDelete(ts_before);
  KheTaskSetDelete(ts_after);
  return false;
}
*** */

/* *** very old version
static bool KheTaskSequenceMoveAndSwapMultiRepair(KHE_EJECTOR ej, KHE_TASK task,
  KHE_RESOURCE_GROUP rg, KHE_RESOURCE_GROUP not_rg, bool allow_unassign,
  int max_len, KHE_MOVE_TYPE mt)
{
  struct khe_se_resource_set_type srs_rec;  KHE_RESOURCE from_r, to_r;
  int index, extra, r_max_len, len, before_len, after_len;
  int move_before, move_after, task_pos;
  KHE_TASK_SET ts_before, ts_after;  KHE_SOLN soln;  KHE_TIME t;
  KHE_FRAME days_frame;

  ** get assigned time and frame and exit early if not present **
  t = KheMeetAsstTime(KheTaskMeet(task));
  days_frame = KheEjectorFrame(ej);
  if( t == NULL || KheFrameIsNull(days_frame) )
    return false;

  ** get task sets from before and after **
  soln = KheEjectorSoln(ej);
  ts_before = KheTaskSetMake(soln);
  ts_after = KheTaskSetMake(soln);
  index = KheFrameTimeIndex(days_frame, t);
  from_r = KheTaskAsstResource(task);
  if( from_r != NULL )
    KheFindAssigne dTasks(ej, index, max_len - 1, from_r, ts_before, ts_after);
  else
    KheFindUnassignedTasks(ej, index, max_len - 1, KheTaskResourceType(task),
      ts_before, ts_after);

  ** iterate over resources, moving subsets of the task sets **
  extra = KheEjectorCurrAugmentCount(ej) +
    5 * KheSolnDiversifier(KheEjectorSoln(ej));
  KheForEachResource(srs_rec, rg, not_rg, allow_unassign, extra, to_r)
    if( to_r != from_r )
    {
      before_len = KheMaxFreeLen(ts_before, to_r);
      after_len = KheMaxFreeLen(ts_after, to_r);
      r_max_len = min(max_len, before_len + 1 + after_len);
      for( len = 2;  len <= r_max_len;  len++ )
      {
	** now task can be at position 0, 1, ... , len-1 of the moved tasks **
	for( task_pos = 0;  task_pos < len;  task_pos++ )
	{
	  ** must be enough to move on each side **
	  move_before = task_pos;
	  move_after = len - task_pos - 1;
	  if( move_before <= before_len && move_after <= after_len &&
	      KheTaskSequenceDoMoveAndSwapMultiRepair(ej, ts_before, move_before, task,
		ts_after, move_after, from_r, to_r, mt) )
	  {
	    KheTaskSetDelete(ts_before);
	    KheTaskSetDelete(ts_after);
	    return true;
	  }
	}
      }
    }
  KheTaskSetDelete(ts_before);
  KheTaskSetDelete(ts_after);
  return false;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  Submodule "task move and swap repairs and augments"                      */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_REPAIR_TYPE KheMoveTypeToTaskRepairType(KHE_MOVE_TYPE mt)            */
/*                                                                           */
/*  Convert move type mt into the corresponding task repair type.            */
/*                                                                           */
/*****************************************************************************/

/* ***
static KHE_REPAIR_TYPE KheMoveTypeToTaskRepairType(KHE_MOVE_TYPE mt)
{
  switch( mt )
  {
    case KHE_MOVE_UNCHECKED:	return KHE_REPAIR_TASK_MOVE_UNCHECKED;
    case KHE_MOVE_CHECKED:	return KHE_REPAIR_TASK_MOVE_CHECKED;
    case KHE_MOVE_EJECTING:	return KHE_REPAIR_TASK_MOVE_EJECTING;
    case KHE_MOVE_KEMPE:	return KHE_REPAIR_TASK_MOVE_KEMPE;

    default:
      HnAbort("KheMoveTypeToTaskRepairType internal error");
      return KHE_REPAIR_TASK_MOVE_UNCHECKED;  ** keep compiler happy **
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskUnAssignRepair(KHE_EJECTOR ej, KHE_TASK task)                */
/*                                                                           */
/*  Try an ejecting task unassignment (KHE_REPAIR_TASK_UNASSIGN) of task,    */
/*  although not if the task is preassigned.                                 */
/*                                                                           */
/*****************************************************************************/

/* *** omitted, KheTaskMoveRepair(ej, task, NULL, -) does the same
static bool KheTaskUnAssignRepair(KHE_EJECTOR ej, KHE_TASK task)
{
  bool success;  KHE_RESOURCE r;
  if( KheTaskIsPreassigned(task, &r) )
    return false;
  KheEjectorR epairBegin(ej);
  success = KheTaskUnAssign(task);
  if( DEBUG4 || KheEjectorCurrDebug(ej) )
  {
    fprintf(stderr, "%cTaskUnAssign(", success ? '+' : '-');
    KheTaskDebug(task, 1, -1, stderr);
    fprintf(stderr, ")");
  }
  return KheEjectorR epairEnd(ej, KHE_REPAIR_TASK_UNASSIGN, success);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskUnAssignAugment(KHE_EJECTOR ej, KHE_TASK task)               */
/*                                                                           */
/*  Unassign task.                                                           */
/*                                                                           */
/*  If allow_e ject is true, the assignment or move may cause other tasks    */
/*  to become unassigned.  If allow_meet_move is true, a Kem pe meet move    */
/*  of the enclosing meet may be tried at the same time as the task move.    */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused
static bool KheTaskUnAssignAugment(KHE_EJECTOR ej, KHE_TASK task)
{
  HnAssert(KheTaskAsstResource(task) != NULL,
    "KheTaskUnAssignAugment internal error");
  if( !KheTas kVisited(task, 0) ) 
  {
    KheTa skVisit(task);
    if( KheTaskUnAssignRepair(ej, task) )
      return true;
    if( KheEjectorCurrMayRevisit(ej) )
      KheTas kUnVisit(task);
  }
  return false;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskMoveRepair(KHE_EJECTOR ej, KHE_TASK task, KHE_RESOURCE to_r) */
/*                                                                           */
/*  Try a task move repair of task to r, which may be NULL.                  */
/*                                                                           */
/*****************************************************************************/

/* *** no longer used
static bool KheTaskMoveRepair(KHE_EJECTOR ej, KHE_TASK task, KHE_RESOURCE to_r)
{
  bool success;  KHE_FRAME frame;  KHE_RESOURCE from_r;
  from_r = KheTaskAsstResource(task);
  KheEjectorRepairBe gin(ej);
  frame = KheEjectorFrame(ej);
  success = KheTypedTaskMoveFrame(task, to_r, KHE_MOVE_EJECTING, frame);
  if( DEBUG4 || KheEjectorCurrDebug(ej) )
  {
    fprintf(stderr, "%cTaskMove(", success ? '+' : '-');
    KheTaskDebug(task, 1, -1, stderr);
    fprintf(stderr, ", %s -> %s)",
      from_r == NULL ? "-" : KheResourceId(from_r),
      to_r == NULL ? "-" : KheResourceId(to_r));
  }
  return KheEjectorRep airEnd(ej, KHE_REPAIR_TASK_MOVE_EJECTING, success);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskDou bleMoveRepair(KHE_EJECTOR ej, KHE_TASK task1,            */
/*    KHE_RESOURCE r1, KHE_TASK task2, KHE_RESOURCE r2)                      */
/*                                                                           */
/*  Try a double task move repair of task1 to r1 and task2 to r2.  Either    */
/*  of both of r1 and r2 may be NULL.                                        */
/*                                                                           */
/*****************************************************************************/

/* *** no longer used
static bool KheTaskDou bleMoveRepair(KHE_EJECTOR ej, KHE_TASK task1,
  KHE_RESOURCE r1, KHE_TASK task2, KHE_RESOURCE r2)
{
  bool success;
  KheEjectorRe pairBegin(ej);
  success = KheTaskMoveResource(task1, r1) && KheTaskMoveResource(task2, r2);
  if( KheClashes(ej, r1) || KheClashes(ej, r2) ||
      DEBUG4 || KheEjectorCurrDebug(ej) )
  {
    fprintf(stderr, "%cTaskD oubleMove(", success ? '+' : '-');
    KheTaskDebug(task1, 1, -1, stderr);
    fprintf(stderr, " -> %s, ", r1 == NULL ? "-" : KheResourceId(r1));
    KheTaskDebug(task2, 1, -1, stderr);
    fprintf(stderr, " -> %s)", r2 == NULL ? "-" : KheResourceId(r2));
  }
  KheNoClashesCheck(ej, r1);
  KheNoClashesCheck(ej, r2);
  return KheEjectorR epairEnd(ej, KHE_REPAIR_TASK_DOUBLE_MOVE, success);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskMoveMultiRepair(KHE_EJECTOR ej, KHE_TASK task,               */
/*    KHE_RESOURCE_GROUP rg, KHE_RESOURCE_GROUP not_rg, bool allow_unassign, */
/*    KHE_MOVE_TYPE mt, bool allow_meet_move)                                */
/*                                                                           */
/*  Try a set of repairs that move task to the resources of rg but not       */
/*  not_rg (if non-NULL) and not what it is assigned to now.  This will      */
/*  include unassignment if allow_unassign is true and the task is           */
/*  currently assigned a resource.                                           */
/*                                                                           */
/*****************************************************************************/
/* ***
static bool KheMeetMoveAndTaskMoveMultiRepair(KHE_EJECTOR ej,
  KHE_MOVE_TYPE meet_mt, KHE_TASK task, KHE_RESOURCE r);
*** */

/* *** no longer used
static bool KheTaskMoveMultiRepair(KHE_EJECTOR ej, KHE_TASK task,
  KHE_RESOURCE_GROUP rg, KHE_RESOURCE_GROUP not_rg, bool allow_unassign,
  bool allow_meet_move)
{
  struct khe_se_resource_set_type srs_rec;  int extra;
  KHE_RESOURCE from_r, to_r;
  extra = KheEjectorCurrAugmentCount(ej);
  from_r = KheTaskAsstResource(task);
  KheForEachResource(srs_rec, rg, not_rg, allow_unassign, extra, to_r)
    if( to_r != from_r )
    {
      if( KheTaskMoveRepair(ej, task, to_r) )
	return true;
      if( to_r != NULL && allow_meet_move &&
	  KheMeetMoveAndTaskMoveMultiRepair(ej, KHE_MOVE_KEMPE, task, to_r) )
	return true;
    }
  return false;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheOldTaskMoveAugment(KHE_EJECTOR ej, KHE_TASK task,                */
/*    KHE_RESOURCE_GROUP rg, KHE_RESOURCE_GROUP not_rg, bool allow_unassign, */
/*    KHE_MOVE_TYPE mt, bool allow_meet_move)                                */
/*                                                                           */
/*  Assign or move task to any element of rg except (1) what it's assigned   */
/*  to now, and (2) if not_rg != NULL, the elements of not_rg.  Possibly     */
/*  do a meet move of the enclosing meet as well.                            */
/*                                                                           */
/*  If allow_unassign is true and task is currently assigned, try            */
/*  unassigning it before trying to move it.                                 */
/*                                                                           */
/*  If allow_resource_swaps is true, try repairs that swap the assignment    */
/*  to task with the assignments to other tasks on the same day.             */
/*                                                                           */
/*  If the model is employee scheduling, also try task sequence moves.       */
/*                                                                           */
/*****************************************************************************/

/* *** old version
static bool KheOldTaskMoveAugment(KHE_EJECTOR ej, KHE_TASK task,
  KHE_RESOURCE_GROUP rg, KHE_RESOURCE_GROUP not_rg, bool allow_unassign,
  bool allow_meet_move)
{
  KHE_INSTANCE ins;  KHE_RESOURCE r;
  if( !KheTask Visited(task, 0) && !KheTaskIsPreassigned(task, &r) )
  {
    KheTask Visit(task);
    ins = KheResourceGroupInstance(rg);
    if( KheInstanceModel(ins) == KHE_MODEL_EMPLOYEE_SCHEDULE )
    {
      ** nurse rostering version **
      if( KheTaskSequenceMoveAndSwapMultiRepair(ej, task, rg, not_rg,
	    allow_unassign) )
	return true;
    }
    else
    {
      ** high school version **
      if( KheTaskMoveMultiRepair(ej, task, rg, not_rg, allow_unassign,
	    allow_meet_move) )
	return true;
    }
    if( KheEjectorCurrMayRevisit(ej) )
      K heTaskUnVisit(task);
  }
  return false;
}
*** */

/* *** very old version
static bool KheOldTaskMoveAugment(KHE_EJECTOR ej, KHE_TASK task,
  KHE_RESOURCE_GROUP rg, KHE_RESOURCE_GROUP not_rg, bool allow_unassign,
  KHE_MOVE_TYPE mt, bool allow_meet_move)
{
  if( KheInstanceModel(KheResourceGroupInstance(rg)) ==
      KHE_MODEL_EMPLOYEE_SCHEDULE )
  {
    ** nurse rostering version **
    return KheNurseMoveAugment(ej, task, rg, not_rg,
      allow_unassign, mt, allow_meet_move);
  }
  else
  {
    ** high school timetabling version **
    if( !K heTaskVisited(task, 0) )
    {
      K heTaskVisit(task);
      if( KheTaskMoveMultiRepair(ej, task, rg, not_rg, allow_unassign, mt,
	    allow_meet_move) )
	return true;
      if( KheEjectorCurrMayRevisit(ej) )
	K heTaskUnVisit(task);
    }
  }
  return false;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskEquiv alent(KHE_TASK task1, KHE_TASK task2, bool incl_asst)  */ 
/*                                                                           */
/*  Return true if task1 and task2 are equivalent, that is, if assigning     */
/*  task1 to a resource must have the same effect on its timetable, and      */
/*  on solution cost, as assigning task2.                                    */
/*                                                                           */
/*  Either or both of task1 and task2 may be NULL.                           */
/*                                                                           */
/*  If incl_asst is true, the tasks must either be assigned the same         */
/*  resource, or assigned NULL.                                              */
/*                                                                           */
/*****************************************************************************/

/* *** moved to khe_task.c
static bool KheTaskEqui valent(KHE_TASK task1, KHE_TASK task2, bool incl_asst)
{
  KHE_EVENT_RESOURCE er1, er2;  KHE_MEET meet1, meet2;
  KHE_TASK child_task1, child_task2;  int i;

  ** if task1 is NULL, it depends on whether task2 is NULL **
  if( task1 == NULL )
    return task2 == NULL;

  ** if task2 is NULL, it depends on whether task1 is NULL (it isn't) **
  if( task2 == NULL )
    return false;

  ** must be assigned the same resource, or none **
  if( incl_asst && KheTaskAsstResource(task1) != KheTaskAsstResource(task2) )
    return false;

  ** must be derived from equivalent event resources **
  er1 = KheTaskEventResource(task1);
  er2 = KheTaskEventResource(task2);
  if( er1 == NULL || er2 == NULL || !KheEventResourceEquivalent(er1, er2) )
    return false;

  ** must have the same duration and the same starting time **
  meet1 = KheTaskMeet(task1);
  meet2 = KheTaskMeet(task2);
  if( KheMeetDuration(meet1) != KheMeetDuration(meet2) )
    return false;
  if( KheMeetAsstTime(meet1) != KheMeetAsstTime(meet2) )
    return false;

  ** must have the same domain **
  if( !KheResourceGroupEqual(KheTaskDomain(task1), KheTaskDomain(task2)) )
    return false;

  ** assigned tasks must be equivalent **
  ** we should really sort them first, but we don't **
  if( KheTaskAssignedToCount(task1) != KheTaskAssignedToCount(task2) )
    return false;
  for( i = 0;  i < KheTaskAssignedToCount(task1);  i++ )
  {
    child_task1 = KheTaskAssignedTo(task1, i);
    child_task2 = KheTaskAssignedTo(task2, i);
    if( !KheTaskEqui valent(child_task1, child_task2, incl_asst) )
      return false;
  }
  return true;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheMoveTaskToResourceAugment(KHE_EJECTOR ej, KHE_RESOURCE r,        */
/*    KHE_TIME_GROUP tg, bool assigned_tasks)                                */
/*                                                                           */
/*  Try repairs that move a task currently running during time tg to r.      */
/*  Here r must be free at the task's times.  If assigned_tasks is true,     */
/*  the moved task must be currently assigned, otherwise it must be          */
/*  currently unassigned.                                                    */
/*                                                                           */
/*  At present this is used to repair time groups where r is underloaded.    */
/*                                                                           */
/*****************************************************************************/

/* *** obsolete
static bool KheTaskMove Augment(KHE_EJECTOR ej, KHE_TASK task,
  KHE_RESOURCE_GROUP rg, KHE_RESOURCE_GROUP not_rg, bool allow_unassign
  ** , KHE_MOVE_REASON mr **)

static bool KheMoveTaskToResourceAugment(KHE_EJECTOR ej, KHE_RESOURCE r,
  KHE_TIME_GROUP tg, bool assigned_tasks ** , KHE_MOVE_REASON mr **)
{
  KHE_RESOURCE_GROUP r_rg;  int tg_time_count, i, j, k, extra;
  KHE_RESOURCE_TIMETABLE_MONITOR rtm;  KHE_SOLN soln;  KHE_MEET meet;
  KHE_TIME t, mt;  KHE_TASK task, prev_task;  KHE_EVENT_TIMETABLE_MONITOR etm;
  et m = KheEjectorEventTimetableMonitor(ej);
  if( etm == NULL )
    return false;
  soln = KheEjectorSoln(ej);
  rtm = KheResourceTimetableMonitor(soln, r);
  r_rg = KheResourceSingletonResourceGroup(r);
  tg_time_count = KheTimeGroupTimeCount(tg);
  extra = KheEjectorCurrAugmentCount(ej) + 5 * KheSolnDiversifier(soln);
  for( i = 0;  i < tg_time_count;  i++ )
  {
    t = KheTimeGroupTime(tg, (i + extra) % tg_time_count);
    if( KheResourceTimetableMonitorTimeTaskCount(rtm, t) == 0 )
    {
      for( j = 0;  j < KheEventTimetableMonitorTimeMeetCount(etm, t);  j++ )
      {
	meet = KheEventTimetableMonitorTimeMeet(etm, t, j);
	mt = KheMeetAsstTime(meet);
	if( KheResourceTimet ableMonitorTimeAvailable(rtm, meet, mt) )
	{
	  ** r is free at the times of meet **
	  if( assigned_tasks )
	  {
	    ** try moving currently assigned tasks to r **
	    for( k = 0;  k < KheMeetTaskCount(meet);  k++ )
	    {
	      task = KheMeetTask(meet, k);
	      if( KheTaskAsstResource(task) != NULL &&
		  KheResourceGroupContains(KheTaskDomain(task), r) &&
		  KheTaskMov eAugment(ej, task, r_rg, NULL, false ** , mr **) )
		return true;
	    }
	  }
	  else
	  {
	    ** try moving non-equivalent currently unassigned tasks to r **
	    prev_task = NULL;
	    for( k = 0;  k < KheMeetTaskCount(meet);  k++ )
	    {
	      task = KheMeetTask(meet, k);
	      if( KheTaskAsstResource(task) == NULL )
	      {
		if( KheResourceGroupContains(KheTaskDomain(task), r) &&
		    (prev_task == NULL || !Khe TaskEquivalent(prev_task, task))
		    && KheTaskMov eAugment(ej, task, r_rg, NULL, false ** , mr **) )
		  return true;
		prev_task = task;
	      }
	    }
	  }
	}
      }
    }
  }
  return false;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheResourceMoveRepair(KHE_EJECTOR ej, KHE_TASK old_task,            */
/*    KHE_TASK new_task, KHE_MOVE_TYPE mt)                                   */
/*                                                                           */
/*  Repair that moves a resource from old_task to new_task.                  */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused
static bool KheResourceMoveRepair(KHE_EJECTOR ej, KHE_TASK old_task,
  KHE_TASK new_task, KHE_MOVE_TYPE mt)
{
  bool success;  KHE_RESOURCE r;  KHE_FRAME frame;
  KheEjectorR epairBegin(ej);
  r = KheTaskAsstResource(old_task);
  HnAssert(r != NULL, "KheResourceMoveRepair internal error");
  frame = KheEjectorFrame(ej);
  success = KheTaskUnAssign(old_task) &&
    KheTypedTaskMoveFrame(new_task, r, mt, frame);
  if( KheClashes(ej, r) || DEBUG4 || KheEjectorCurrDebug(ej) )
  {
    fprintf(stderr, "%cResourceMove(", success ? '+' : '-');
    KheTaskDebug(old_task, 1, -1, stderr);
    fprintf(stderr, ", ");
    KheTaskDebug(new_task, 1, -1, stderr);
    fprintf(stderr, ")");
  }
  KheNoClashesCheck(ej, r);
  return KheEjectorR epairEnd(ej, KHE_REPAIR_RESOURCE_MOVE, success);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  Submodule "day set moves and swaps"                                      */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool KheAddAssignedTasks(KHE_FRAME frame, int first_index,               */
/*    int last_index, KHE_RESOURCE r, KHE_TASK_SET ts)                       */
/*                                                                           */
/*  Add all the tasks assigned r in frame[first_index .. last_index] to ts.  */
/*  Return true if all of the tasks are not preassigned, else false.         */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused
static bool KheAddAssignedTasks(KHE_FRAME frame, int first_index,
  int last_index, KHE_RESOURCE r, KHE_TASK_SET ts)
{
  KHE_RESOURCE_TIMETABLE_MONITOR rtm;  int i, j, k, pos, count;  bool res;
  KHE_TIME_GROUP tg;  KHE_TIME t;  KHE_TASK task;  KHE_RESOURCE r2;
  res = true;
  rtm = KheResourceTimetableMonitor(KheFrameSoln(frame), r);
  for( i = first_index;  i <= last_index;  i++ )
  {
    tg = KheFrameTimeGroup(frame, i);
    for( j = 0;  j < KheTimeGroupTimeCount(tg);  j++ )
    {
      t = KheTimeGroupTime(tg, j);
      count = KheResourceTimetableMonitorTimeTaskCount(rtm, t);
      for( k = 0;  k < count;  k++ )
      {
	task = KheResourceTimetableMonitorTimeTask(rtm, t, k);
	task = KheTaskProperRoot(task);
	if( task != NULL && !KheTaskSetContainsTask(ts, task, &pos) )
	{
	  KheTaskSetAddTask(ts, task);
	  if( KheTaskIsPreassigned(task, &r2) )
	    res = false;
	}
      }
    }
  }
  return res;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheTimeGroupContainsUna ssignedTask(KHE_TIME_GROUP tg,              */
/*    KHE_RESOURCE_TYPE rt, KHE_EVENT_TIMETABLE_MONITOR etm, KHE_TASK *res)  */
/*                                                                           */
/*  If tg contains an unassigned task of type rt, set *res to one such       */
/*  task and return true, otherwise return false.  Use etm to find the       */
/*  meets running at the times of tg.                                        */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused
static bool KheTimeGroupConta insUnassignedTask(KHE_TIME_GROUP tg,
  KHE_RESOURCE_TYPE rt, KHE_EVENT_TIMETABLE_MONITOR etm, KHE_TASK *res)
{
  int i, j, k;  KHE_TIME t;  KHE_MEET meet;  KHE_TASK task;
  for( i = 0;  i < KheTimeGroupTimeCount(tg);  i++ )
  {
    t = KheTimeGroupTime(tg, i);
    for( j = 0;  j < KheEventTimetableMonitorTimeMeetCount(etm, t);  j++ )
    {
      meet = KheEventTimetableMonitorTimeMeet(etm, t, j);
      for( k = 0;  k < KheMeetTaskCount(meet);  k++ )
      {
	task = KheMeetTask(meet, k);
	task = KheTaskProperRoot(task);
	if( KheTaskResourceType(task) == rt &&
	    KheTaskAsstResource(task) == NULL )
	  return *res = task, true;
      }
    }
  }
  return *res = NULL, false;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheAddUnassignedTasks(KHE_FRAME frame, int first_index,             */
/*    int last_index, KHE_RESOURCE r, KHE_EVENT_TIMETABLE_MONITOR etm,       */
/*    KHE_TASK_SET ts)                                                       */
/*                                                                           */
/*  Add unassigned, unpreassigned tasks in frame[first_index .. last_index]  */
/*  that could be assigned to r to ts, skipping days when (no: r is busy or) */
/*  there are no unassigned tasks.  Always return true.                      */
/*                                                                           */
/*****************************************************************************/

/* *** currently unassigned
static bool KheAddUnassignedTasks(KHE_FRAME frame, int first_index,
  int last_index, KHE_RESOURCE r, KHE_EVENT_TIMETABLE_MONITOR etm,
  KHE_TASK_SET ts)
{
  int i, j, k, m, pos;  KHE_MEET meet;
  KHE_TIME_GROUP tg;  KHE_TASK task;  KHE_TIME t;
  for( i = first_index;  i <= last_index;  i++ )
  {
    tg = KheFrameTimeGroup(frame, i);
    for( j = 0;  j < KheTimeGroupTimeCount(tg);  j++ )
    {
      t = KheTimeGroupTime(tg, j);
      for( k = 0;  k < KheEventTimetableMonitorTimeMeetCount(etm, t);  k++ )
      {
	meet = KheEventTimetableMonitorTimeMeet(etm, t, k);
	for( m = 0;  m < KheMeetTaskCount(meet);  m++ )
	{
	  task = KheMeetTask(meet, m);
	  task = KheTaskProperRoot(task);
	  if( task != NULL && KheTaskAssignResourceCheck(task, r) &&
	      !KheTaskSetContainsTask(ts, task, &pos) )
	    KheTaskSetAddTask(ts, task);
	}
      }
    }
  }
  return true;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskSetMoveResource(KHE_TASK_SET ts, KHE_RESOURCE r)             */
/*                                                                           */
/*  Move the tasks of ts to r, and return true if all successful.            */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused
static bool KheTaskSetMoveResource(KHE_TASK_SET ts, KHE_RESOURCE r)
{
  int i;  KHE_TASK task;
  for( i = 0;  i < KheTaskSetTaskCount(ts);  i++ )
  {
    task = KheTaskSetTask(ts, i);
    if( !KheTaskMoveResource(task, r) )
      return false;
  }
  return true;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheOpDebug(char *op_name, bool success, KHE_FRAME frame,            */
/*    int fi, int li, KHE_RESOURCE r1, KHE_TASK_SET r1_ts, KHE_RESOURCE r2,  */
/*    KHE_TASK_SET r2_ts)                                                    */
/*                                                                           */
/*  Write on stderr a debug print of a day set operation with these          */
/*  attributes.                                                              */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused
static void KheOpDebug(char *op_name, bool success, KHE_FRAME frame,
  int fi, int li, KHE_RESOURCE r1, KHE_TASK_SET r1_ts, KHE_RESOURCE r2,
  KHE_TASK_SET r2_ts)
{
  KHE_TIME_GROUP tg;

  ** initial stuff **
  fprintf(stderr, "%c%s(", success ? '+' : '-', op_name);

  ** time groups **
  tg = KheFrameTimeGroup(frame, fi);
  fprintf(stderr, "%s", KheTimeGroupId(tg));
  if( fi != li )
  {
    tg = KheFrameTimeGroup(frame, li);
    fprintf(stderr, "..%s", KheTimeGroupId(tg));
  }

  ** r1 and r1_ts **
  fprintf(stderr, ", %s", r1 == NULL ? "-" : KheResourceId(r1));
  if( KheTaskSetTaskCount(r1_ts) > 0 )
  {
    fprintf(stderr, " incl ");
    KheTaskSetDebug(r1_ts, 2, -1, stderr);
  }

  ** r2 and r2_ts **
  fprintf(stderr, ", %s", r2 == NULL ? "-" : KheResourceId(r2));
  if( r2_ts != NULL && KheTaskSetTaskCount(r2_ts) > 0 )
  {
    fprintf(stderr, " incl ");
    KheTaskSetDebug(r2_ts, 2, -1, stderr);
  }

  fprintf(stderr, ")");
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheOp2Debug(char *op_name, bool success, KHE_FRAME frame,           */
/*    int first_index, int last_index, int first_index2, int last_index2,    */
/*    KHE_RESOURCE r1, KHE_RESOURCE r2)                                      */
/*                                                                           */
/*  Write on stderr a debug print of a day set operation with these          */
/*  attributes.                                                              */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheOp2Debug(char *op_name, bool success, KHE_FRAME frame,
  int fi1, int li1, int fi2, int li2, KHE_RESOURCE r1,
  KHE_TASK_SET r1_ts, KHE_RESOURCE r2, KHE_TASK_SET r2_ts)
{
  KHE_TIME_GROUP tg;

  ** initial stuff **
  fprintf(stderr, "%c%s(", success ? '+' : '-', op_name);

  ** first lot of time groups **
  tg = KheFrameTimeGroup(frame, fi1);
  fprintf(stderr, "%s", KheTimeGroupId(tg));
  if( fi1 != li1 )
  {
    tg = KheFrameTimeGroup(frame, li1);
    fprintf(stderr, "..%s", KheTimeGroupId(tg));
  }

  ** second lot of time groups **
  tg = KheFrameTimeGroup(frame, fi2);
  fprintf(stderr, ", %s", KheTimeGroupId(tg));
  if( fi2 != li2 )
  {
    tg = KheFrameTimeGroup(frame, li2);
    fprintf(stderr, "..%s", KheTimeGroupId(tg));
  }

  ** r1 and r1_ts **
  fprintf(stderr, ", %s", r1 == NULL ? "-" : KheResourceId(r1));
  if( KheTaskSetTaskCount(r1_ts) > 0 )
  {
    fprintf(stderr, " incl ");
    KheTaskSetDebug(r1_ts, 2, -1, stderr);
  }

  ** r2 and r2_ts **
  fprintf(stderr, ", %s", r2 == NULL ? "-" : KheResourceId(r2));
  if( r2_ts != NULL && KheTaskSetTaskCount(r2_ts) > 0 )
  {
    fprintf(stderr, " incl ");
    KheTaskSetDebug(r2_ts, 2, -1, stderr);
  }

  fprintf(stderr, ")");
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskBounded(KHE_TASK task, KHE_FRAME frame, int fi, int li)      */
/*                                                                           */
/*  Make sure that task (including the tasks assigned to it) lies            */
/*  entirely in the range frame[fi .. li].                                   */
/*                                                                           */
/*****************************************************************************/

/* ***
static bool KheTaskBounded(KHE_TASK task, KHE_FRAME frame, int fi, int li)
{
  KHE_MEET meet;  int i, meet_fi, meet_li;
  KHE_TASK child_task;  KHE_TIME first_t, last_t;

  ** check task **
  meet = KheTaskMeet(task);
  first_t = KheMeetAsstTime(meet);
  if( first_t != NULL )
  {
    last_t = KheTimeNeighbour(first_t, KheMeetDuration(meet) - 1);
    meet_fi = KheFrameTimeIndex(frame, first_t);
    meet_li = KheFrameTimeIndex(frame, last_t);
    if( meet_fi < fi || meet_li > li )
      return false;
  }

  ** check task's child tasks **
  for( i = 0;  i < KheTaskAssignedToCount(task);  i++ )
  {
    child_task = KheTaskAssignedTo(task, i);
    if( !KheTaskBounded(child_task, frame, fi, li) )
      return false;
  }

  ** all good **
  return true;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskSetBounded(KHE_TASK_SET ts, KHE_FRAME frame, int fi, int li) */
/*                                                                           */
/*  Make sure that the tasks of ts (including the tasks assigned to them)    */
/*  lie entirely in the range frame[fi .. li].                               */
/*                                                                           */
/*****************************************************************************/

/* ***
static bool KheTaskSetBounded(KHE_TASK_SET ts, KHE_FRAME frame, int fi, int li)
{
  KHE_TASK task;  int i;
  for( i = 0;  i < KheTaskSetTaskCount(ts);  i++ )
  {
    task = KheTaskSetTask(ts, i);
    if( !KheTaskBounded(task, frame, fi, li) )
      return false;
  }
  return true;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDaySetSwap(KHE_FRAME frame, int fi, int li, KHE_RESOURCE r1,     */
/*    KHE_TASK_SET r1_ts, KHE_RESOURCE r2, KHE_TASK_SET r2_ts, int *r1_to_r2)*/
/*                                                                           */
/*  Swap the timetables of r1 and r2 on days fi ... li, and return true if   */
/*  this succeeds.  If it would not change the solution, or cannot be done   */
/*  without changing the timetable on other days as well (because of task    */
/*  grouping, the operation does not succeed.                                */
/*                                                                           */
/*  Here r1_ts is a task set to be used to hold the tasks initially          */
/*  assigned r1.  Some of these tasks may already be present in r1_ts.       */
/*  This task set may be changed by this operation, but if so it is          */
/*  reset to its initial value before exit.                                  */
/*                                                                           */
/*  Similarly, r2_ts is a task set to be used to hold the tasks initially    */
/*  assigned r2.  Some of these tasks may already be present in r2_ts.       */
/*  This task set may be changed by this operation, but if so it is          */
/*  reset to its initial value before exit.                                  */
/*                                                                           */
/*  Set *r1_to_r2 to the net number of tasks that moved from r1 to r2, that  */
/*  is, the number that moved from r1 to r2 minus the number that moved      */
/*  from r2 to r1.  This could be negative or zero.                          */
/*                                                                           */
/*****************************************************************************/

/* ***
static bool KheDaySetSwap(KHE_FRAME frame, int fi, int li, KHE_RESOURCE r1,
  KHE_TASK_SET r1_ts, KHE_RESOURCE r2, KHE_TASK_SET r2_ts, int *r1_to_r2)
{
  bool res;  int r1_ts_count, r2_ts_count;
  r1_ts_count = KheTaskSetTaskCount(r1_ts);
  r2_ts_count = KheTaskSetTaskCount(r2_ts);
  res = KheAddAssignedTasks(frame, fi, li, r1, r1_ts)		&&
        KheAddAssignedTasks(frame, fi, li, r2, r2_ts)		&&
	KheTaskSetTaskCount(r1_ts) + KheTaskSetTaskCount(r2_ts) > 0 &&
	KheTaskSetBounded(r1_ts, frame, fi, li)		&&
	KheTaskSetBounded(r2_ts, frame, fi, li)		&&
        KheTaskSetMoveResource(r1_ts, r2) && KheTaskSetMoveResource(r2_ts, r1);
  if( DEBUG25 && res )
  {
    fprintf(stderr, "DaySetSwap(%s: ", KheResourceId(r1));
    KheTaskSetDebug(r1_ts, 2, -1, stderr);
    fprintf(stderr, ", %s: ", KheResourceId(r2));
    KheTaskSetDebug(r2_ts, 2, -1, stderr);
    fprintf(stderr, ")\n");
  }
  *r1_to_r2 = KheTaskSetTaskCount(r1_ts) - KheTaskSetTaskCount(r2_ts);
  KheTaskSetDropFromEnd(r1_ts, KheTaskSetTaskCount(r1_ts) - r1_ts_count);
  KheTaskSetDropFromEnd(r2_ts, KheTaskSetTaskCount(r2_ts) - r2_ts_count);
  return res;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDaySetSwapRepair(KHE_EJECTOR ej, int fi, int li, KHE_RESOURCE r1,*/
/*    KHE_TASK_SET r1_ts, KHE_RESOURCE r2, KHE_TASK_SET r2_ts, int *r1_to_r2)*/
/*                                                                           */
/*  Carry out one repair, a day-set swap with these attributes.              */
/*                                                                           */
/*  Set *r1_to_r2 to the net number of tasks that moved from r1 to r2, that  */
/*  is, the number that moved from r1 to r2 minus the number that moved      */
/*  from r2 to r1.  This could be negative or zero.                          */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused
static bool KheDaySetSwapRepair(KHE_EJECTOR ej, int fi, int li, KHE_RESOURCE r1,
  KHE_TASK_SET r1_ts, KHE_RESOURCE r2, KHE_TASK_SET r2_ts, int *r1_to_r2)
{
  bool success;  KHE_FRAME frame;
  KheEjectorRe pairBegin(ej);
  frame = KheEjectorFrame(ej);
  success = KheDaySetSwap(frame, fi, li, r1, r1_ts, r2, r2_ts, r1_to_r2);
  if( DEBUG5 || KheEjectorCurrDebug(ej) )
    KheOpDebug("DaySetSwap", success, frame, fi, li, r1, r1_ts, r2, r2_ts);
  return KheEjectorR epairEnd(ej, KHE_REPAIR_DAY_SET_SWAP, success);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDaySetDoubleSwapRepair(KHE_EJECTOR ej, int fi1, int li1,         */
/*    int fi2, int li2, KHE_RESOURCE r1, KHE_TASK_SET r1_ts,                 */
/*    KHE_RESOURCE r2, KHE_TASK_SET r2_ts)                                   */
/*                                                                           */
/*  Like KheDaySetSwapRepair except that it does a double swap, one in       */
/*  fi1 .. li1, the other in fi2 .. li2.                                     */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused
static bool KheDaySetDoubleSwapRepair(KHE_EJECTOR ej, int fi1, int li1,
  int fi2, int li2, KHE_RESOURCE r1, KHE_TASK_SET r1_ts1, KHE_TASK_SET r1_ts2,
  KHE_RESOURCE r2, KHE_TASK_SET r2_ts1, KHE_TASK_SET r2_ts2)
{
  bool success;  KHE_FRAME frame;  int x;
  KheEjectorRe pairBegin(ej);
  frame = KheEjectorFrame(ej);
  success = KheDaySetSwap(frame, fi1, li1, r1, r1_ts1, r2, r2_ts1, &x)
         && KheDaySetSwap(frame, fi2, li2, r1, r1_ts2, r2, r2_ts2, &x);
  if( DEBUG5 || KheEjectorCurrDebug(ej) )
    KheOp2Debug("DaySetDoubleSwap", success, frame, fi1, li1,
      fi2, li2, r1, r1_ts1, r2, r2_ts2);
  return KheEjectorRep airEnd(ej, KHE_REPAIR_DOUBLE_DAY_SET_SWAP, success);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheResourceIsNotBusy(KHE_FRAME days_frame, int index,               */
/*    KHE_RESOURCE_TIMETABLE_MONITOR rtm)                                    */
/*                                                                           */
/*  Return true if the resource monitored by rtm is not busy at index,       */
/*  either because index is off the frame, or because it is free then.       */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused
static bool KheResourceIsNotBusy(KHE_FRAME days_frame, int index,
  KHE_RESOURCE_TIMETABLE_MONITOR rtm)
{
  KHE_TIME_GROUP tg;
  ** return true; **
  if( index < 0 || index >= KheFrameTimeGroupCount(days_frame) )
    return true;
  tg = KheFrameTimeGroup(days_frame, index);
  return KheResourceTim etableMonitorTimeGroupAvailable(rtm, tg);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheResourceIsBusy(KHE_FRAME days_frame, int index,                  */
/*    KHE_RESOURCE_TIMETABLE_MONITOR rtm)                                    */
/*                                                                           */
/*  Return true if the resource monitored by rtm is busy at index, because   */
/*  index is within the frame, and the resource is busy then.                */
/*                                                                           */
/*****************************************************************************/

/* ***
static bool KheResourceIsBusy(KHE_FRAME days_frame, int index,
  KHE_RESOURCE_TIMETABLE_MONITOR rtm)
{
  KHE_TIME_GROUP tg;
  if( index < 0 || index >= KheFrameTimeGroupCount(days_frame) )
    return false;
  tg = KheFrameTimeGroup(days_frame, index);
  return !KheResourceTimet ableMonitorTimeGroupAvailable(rtm, tg);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheIntervalIsBusy(KHE_FRAME frame, int fi, int li, KHE_RESOURCE r)  */
/*                                                                           */
/*  Return true if frame[fi .. li] is busy for resource r.                   */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused
static bool KheIntervalIsBusy(KHE_FRAME frame, int fi, int li, KHE_RESOURCE r)
{
  KHE_SOLN soln;  KHE_RESOURCE_TIMETABLE_MONITOR rtm;  int i;
  KHE_TIME_GROUP tg;
  soln = KheFrameSoln(frame);
  rtm = KheResourceTimetableMonitor(soln, r);
  for( i = fi;  i <= li;  i++ )
  {
    tg = KheFrameTimeGroup(frame, i);
    if( KheResourceTimeta bleMonitorTimeGroupAvailable(rtm, tg) )
      return false;
  }
  return true;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheIntervalIsFree(KHE_FRAME frame, int fi, int li, KHE_RESOURCE r)  */
/*                                                                           */
/*  Return true if frame[fi .. li] is free for resource r.                   */
/*                                                                           */
/*****************************************************************************/

/* ***
static bool KheIntervalIsFree(KHE_FRAME frame, int fi, int li, KHE_RESOURCE r)
{
  KHE_SOLN soln;  KHE_RESOURCE_TIMETABLE_MONITOR rtm;  int i;
  KHE_TIME_GROUP tg;  KHE_POLARITY po;
  soln = KheFrameSoln(frame);
  rtm = KheResourceTimetableMonitor(soln, r);
  for( i = fi;  i <= li;  i++ )
  {
    tg = KheFrameTimeGroup(frame, i, &po);
    if( !KheResourceTimetable MonitorTimeGroupAvailable(rtm, tg) )
      return false;
  }
  return true;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheIntervalIsFree(KHE_FRAME days_frame, int first_index,            */
/*    int last_index, KHE_RESOURCE_TIMETABLE_MONITOR rtm)                    */
/*                                                                           */
/*  Return true if days_frame[first_index, last_index] is free for rtm.      */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused
static bool KheIntervalIsFree(KHE_FRAME days_frame, int first_index,
  int last_index, KHE_RESOURCE_TIMETABLE_MONITOR rtm)
{
  int i;  KHE_TIME_GROUP tg;
  for( i = first_index;  i <= last_index;  i++ )
  {
    tg = KheFrameTimeGroup(days_frame, i);
    if( !KheResourceTimetableMon itorTimeGroupAvailable(rtm, tg) )
      return false;
  }
  return true;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheFindBusyInterval(KHE_FRAME days_frame, int base_pos,             */
/*    KHE_RESOURCE_TIMETABLE_MONITOR rtm, KHE_SEARCH_TYPE st,                */
/*    int *first_index, int *last_index)                                     */
/*                                                                           */
/*  Search for a busy interval for rtm in days_frame, near base_pos.         */
/*  If found, return true with *first_index and *last_index bracketing       */
/*  the busy interval, otherwise return false.                               */
/*                                                                           */
/*  Parameter st defines four ways to search near base_pos.                  */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused
typedef enum {
  KHE_SEARCH_ADJACENT_ON_RIGHT,
  KHE_SEARCH_ADJACENT_ON_LEFT,
  KHE_SEARCH_TO_RIGHT,
  KHE_SEARCH_TO_LEFT
} KHE_SEARCH_TYPE;

static bool KheFindBusyInterval(KHE_FRAME days_frame, int base_pos,
  KHE_RESOURCE_TIMETABLE_MONITOR rtm, KHE_SEARCH_TYPE st,
  int *first_index, int *last_index)
{
  int days_count, i;
  days_count = KheFrameTimeGroupCount(days_frame);
  switch( st )
  {
    case KHE_SEARCH_ADJACENT_ON_RIGHT:

      if( !KheResourceIsBusy(days_frame, base_pos + 1, rtm) )
	return false;
      *first_index = base_pos + 1;
      for( i = base_pos + 2;  KheResourceIsBusy(days_frame, i, rtm);  i++ );
      *last_index = i - 1;
      return true;

    case KHE_SEARCH_ADJACENT_ON_LEFT:

      if( !KheResourceIsBusy(days_frame, base_pos - 1, rtm) )
	return false;
      *last_index = base_pos - 1;
      for( i = base_pos - 2;  KheResourceIsBusy(days_frame, i, rtm);  i-- );
      *first_index = i + 1;
      return true;

    case KHE_SEARCH_TO_RIGHT:

      for( i = base_pos + 1;  i < days_count;  i++ )
	if( KheResourceIsBusy(days_frame, i, rtm) )
	  break;
      if( i >= days_count )
	return false;
      *first_index = i;
      for( i = i + 1;  i < days_count;  i++ )
	if( !KheResourceIsBusy(days_frame, i, rtm) )
	  break;
      *last_index = i - 1;
      return true;

    case KHE_SEARCH_TO_LEFT:

      for( i = base_pos - 1;  i >= 0;  i-- )
	if( KheResourceIsBusy(days_frame, i, rtm) )
	  break;
      if( i < 0 )
	return false;
      *last_index = i;
      for( i = i - 1;  i >= 0;  i-- )
	if( !KheResourceIsBusy(days_frame, i, rtm) )
	  break;
      *last_index = i + 1;
      return true;

    default:

      HnAbort("KheFindBusyInterval internal error");
      return false;  ** keep compiler happy **
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheMovable(KHE_TIME_GROUP tg, int index, int first_index,           */
/*    int last_index, KHE_RESOURCE_TIMETABLE_MONITOR rtm1,                   */
/*    KHE_RESOURCE_TIMETABLE_MONITOR rtm2)                                   */
/*                                                                           */
/*  Return true if a move is possible at index, whose time group is tg.      */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused
static bool KheMovable(KHE_TIME_GROUP tg, int index, int first_index,
  int last_index, KHE_RESOURCE_TIMETABLE_MONITOR rtm1,
  KHE_RESOURCE_TIMETABLE_MONITOR rtm2)
{
  ** indexes in the scope of the first swap are not movable **
  if( index >= first_index && index <= last_index )
    return false;

  ** other indexes are movable if r1 is free and r2 is busy **
  return KheResourceTimetable MonitorTimeGroupAvailable(rtm1, tg) &&
    !KheResourceTimet ableMonitorTimeGroupAvailable(rtm2, tg);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDaySetDoubleSwapMultiRepair(KHE_EJECTOR ej, int fi1, int li1,    */
/*    KHE_RESOURCE r1, KHE_TASK_SET r1_ts, KHE_RESOURCE r2,                  */
/*    KHE_TASK_SET r2_ts, int r1_to_r2)                                      */
/*                                                                           */
/*  Carry out a set of double day-set swaps, first swapping r1 and r2's      */
/*  tasks between fi1 and li1, and then swapping their tasks elsewhere in    */
/*  the days frame, moving r1_to_r2 tasks from r2 to r1.                     */
/*                                                                           */
/*  Implementation note.  The loop invariant is                              */
/*                                                                           */
/*    if( fi2 != -1 )                                                        */
/*      fi2 ... li2 - 1 is a run of movable positions                        */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused
#define swap(a, b, tmp) ((tmp) = (a), (a) = (b), (b) = (tmp))

typedef enum {
  KHE_SWAP_CLOSE,
  KHE_SWAP_FAR,
  KHE_SWAP_ADJACENT_AND_FAR
} KHE_SWAP_TYPE;

static bool KheDaySetDoubleSwapMultiRepair(KHE_EJECTOR ej, int fi1, int li1,
  KHE_RESOURCE r1, KHE_TASK_SET r1_ts, KHE_RESOURCE r2, KHE_TASK_SET r2_ts,
  int r1_to_r2)
{
  KHE_RESOURCE tmp;  KHE_SOLN soln;  KHE_TASK_SET r1_ts2, r2_ts2, tmp_ts;
  KHE_RESOURCE_TIMETABLE_MONITOR rtm1, rtm2;  KHE_FRAME days_frame;
  int i, extra, days_count, attempts, fi2, li2;  KHE_TIME_GROUP tg;

  ** get the resources around the right way **
  if( r1_to_r2 < 0 )
    swap(r1, r2, tmp), swap(r1_ts, r2_ts, tmp_ts), r1_to_r2 = - r1_to_r2;
  soln = KheEjectorSoln(ej);
  rtm1 = KheResourceTimetableMonitor(soln, r1);
  rtm2 = KheResourceTimetableMonitor(soln, r2);

  ** now we need to move r1_to_r2 tasks from r2 to r1, to even things up **
  days_frame = KheEjectorFrame(ej);
  days_count = KheFrameTimeGroupCount(days_frame);
  r1_ts2 = KheTaskSetMake(soln);
  r2_ts2 = KheTaskSetMake(soln);

  switch( KHE_SWAP_ADJACENT_AND_FAR )
  {
    case KHE_SWAP_CLOSE:

      attempts = 0;
      fi2 = -1;
      for( li2 = li1 + 1;  li2 < days_count && attempts <= 10;  li2++ )
      {
	tg = KheFrameTimeGroup(days_frame, li2);
	if( KheMovable(tg, li2, fi1, li1, rtm1, rtm2) )
	{
	  ** update fi2 **
	  if( fi2 == -1 )
	    fi2 = li2;

	  ** if we have enough movable spots for a repair, then try one **
	  if( li2 - fi2 + 1 == r1_to_r2 )
	  {
	    if( KheResourceIsNotBusy(days_frame, li2 + 1, rtm1) )
	    {
	      if( KheDaySetDoubleSwapRepair(ej, fi1, li1, fi2, li2,
		    r1, r1_ts, r1_ts2, r2, r2_ts, r2_ts2) )
	      {
		KheTaskSetDelete(r1_ts2);
		KheTaskSetDelete(r2_ts2);
		return true;
	      }
	      attempts++;
	      fi2 = -1;
	    }
	    else
	    {
	      fi2++;
	    }
	  }
	}
	else
	  fi2 = -1;
      }

      attempts = 0;
      li2 = -1;
      for( fi2 = fi1 - 1;  fi2 >= 0 && attempts <= 10;  fi2-- )
      {
	** get the next index and time group; disallow moves across end **
	tg = KheFrameTimeGroup(days_frame, fi2);
	** loop invariant holds here **

	if( KheMovable(tg, fi2, fi1, li1, rtm1, rtm2) )
	{
	  ** update li2 **
	  if( li2 == -1 )
	    li2 = fi2;

	  ** if we have enough movable spots for a repair, then try one **
	  if( li2 - fi2 + 1 == r1_to_r2 )
	  {
	    if( KheResourceIsNotBusy(days_frame, fi2 - 1, rtm1) )
	    {
	      if( KheDaySetDoubleSwapRepair(ej, fi1, li1, fi2, li2,
		    r1, r1_ts, r1_ts2, r2, r2_ts, r2_ts2) )
	      {
		KheTaskSetDelete(r1_ts2);
		KheTaskSetDelete(r2_ts2);
		return true;
	      }
	      attempts++;
	      li2 = -1;
	    }
	    else
	    {
	      li2--;
	    }
	  }
	}
	else
	  li2 = -1;
      }
      break;

    case KHE_SWAP_ADJACENT_AND_FAR:

      if( KheFindBusyInterval(days_frame, li1, rtm2,
	KHE_SEARCH_ADJACENT_ON_RIGHT, &fi2, &li2) && li2 - fi2 + 1 >= r1_to_r2 )
      {
        if( KheIntervalIsFree(days_frame, li2 - r1_to_r2 + 1, li2, rtm1) &&
	      KheDaySetDoubleSwapRepair(ej, fi1, li1, li2 - r1_to_r2 + 1, li2,
		r1, r1_ts, r1_ts2, r2, r2_ts, r2_ts2) )
	{
	  KheTaskSetDelete(r1_ts2);
	  KheTaskSetDelete(r2_ts2);
	  return true;
	}
      }
      if( KheFindBusyInterval(days_frame, fi1, rtm2,
	KHE_SEARCH_ADJACENT_ON_LEFT, &fi2, &li2) && li2 - fi2 + 1 >= r1_to_r2 )
      {
        if( KheIntervalIsFree(days_frame, fi2, fi2 + r1_to_r2 - 1, rtm1) &&
	    KheDaySetDoubleSwapRepair(ej, fi1, li1, fi2, fi2 + r1_to_r2 - 1,
	      r1, r1_ts, r1_ts2, r2, r2_ts, r2_ts2) )

	{
	  KheTaskSetDelete(r1_ts2);
	  KheTaskSetDelete(r2_ts2);
	  return true;
	}
      }
      ** NB NO BREAK **

    case KHE_SWAP_FAR:

      extra = 3 * KheEjectorCurrAugmentCount(ej) + 5 * KheSolnDiversifier(soln);
      attempts = 0;
      fi2 = -1;
      for( i = 0;  i < days_count && attempts <= 20;  i++ )
      {
	** get the next index and time group; disallow moves across end **
	li2 = (i + extra) % days_count;
	tg = KheFrameTimeGroup(days_frame, li2);
	if( li2 == 0 )
	  fi2 = -1;
	** loop invariant holds here **

	if( KheMovable(tg, li2, fi1, li1, rtm1, rtm2) )
	{
	  ** update fi2 **
	  if( fi2 == -1 )
	    fi2 = li2;

	  ** if we have enough movable spots for a repair, then try one **
	  if( li2 - fi2 + 1 == r1_to_r2 )
	  {
	    if( KheResourceIsNotBusy(days_frame, li2 + 1, rtm1) )
	    {
	      if( KheDaySetDoubleSwapRepair(ej, fi1, li1, fi2, li2,
		    r1, r1_ts, r1_ts2, r2, r2_ts, r2_ts2) )
	      {
		KheTaskSetDelete(r1_ts2);
		KheTaskSetDelete(r2_ts2);
		return true;
	      }
	      attempts++;
	      fi2 = -1;
	    }
	    else
	    {
	      fi2++;
	    }
	  }
	}
	else
	  fi2 = -1;
      }
      break;

    default:

      HnAbort("KheDaySetDoubleSwapMultiRepair internal error");
      break;
  }
  KheTaskSetDelete(r1_ts2);
  KheTaskSetDelete(r2_ts2);
  return false;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDaySetUnassign(KHE_FRAME frame, int fi1, int li1,                */
/*    KHE_TASK_SET ts, int fi2, int li2, KHE_RESOURCE r, int *r_to_NULL)     */
/*                                                                           */
/*  Unassign the tasks assigned r on days fi1 ... li1, in ts, and on days    */
/*  fi2 ... li2.  Return true if the operation succeeded and changed the     */
/*  solution.  Set *r_to_NULL to the number of tasks that were unassigned.   */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused
static bool KheDaySetUnassign(KHE_FRAME frame, int fi1, int li1,
  KHE_TASK_SET ts, int fi2, int li2, KHE_RESOURCE r, int *r_to_NULL)
{
  bool res;  int ts_count;
  ts_count = KheTaskSetTaskCount(ts);
  res = KheAddAssignedTasks(frame, fi1, li1, r, ts) &&
        KheAddAssignedTasks(frame, fi2, li2, r, ts) &&
	KheTaskSetBounded(ts, frame, fi1, li2)    &&
	KheTaskSetTaskCount(ts) > 0 && KheTaskSetMoveResource(ts, NULL);
  *r_to_NULL = KheTaskSetTaskCount(ts);
  KheTaskSetDropFromEnd(ts, KheTaskSetTaskCount(ts) - ts_count);
  return res;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDaySetUnassignRepair(KHE_EJECTOR ej, int fi1, int li1,           */
/*    KHE_TASK_SET ts, int fi2, int li2, KHE_RESOURCE r)                     */
/*                                                                           */
/*  Carry out one repair, a day-set unassign with these attributes.          */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused
static bool KheDaySetUnassignRepair(KHE_EJECTOR ej, int fi1, int li1,
  KHE_TASK_SET ts, int fi2, int li2, KHE_RESOURCE r)
{
  bool success;  KHE_FRAME frame;  int r_to_NULL;
  KheEjectorRepa irBegin(ej);
  frame = KheEjectorFrame(ej);
  success = KheDaySetUnassign(frame, fi1, li1, ts, fi2, li2, r, &r_to_NULL);
  if( DEBUG5 || KheEjectorCurrDebug(ej) )
    KheOpDebug("DaySetUnassign", success, frame, fi1, li2, r, ts, NULL, NULL);
  return KheEjectorRepa irEnd(ej, KHE_REPAIR_DAY_SET_UNASSIGN, success);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDaySetAssign(KHE_FRAME frame, int fi1, int li1,                  */
/*    KHE_TASK_SET ts, int fi2, int li2, KHE_RESOURCE r,                     */
/*    KHE_EVENT_TIMETABLE_MONITOR etm, int *NULL_to_r)                       */
/*                                                                           */
/*  Assign tasks to r on days fi1 ... li1, in ts, and on days fi2 ... li2,   */
/*  skipping days when r is already busy or there are no unassigned tasks.   */
/*  Return true if the operation succeeded and changed the solution.         */
/*  Also set *NULL_to_r to the number of tasks that were assigned.           */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused
static bool KheDaySetAssign(KHE_FRAME frame, int fi1, int li1,
  KHE_TASK_SET ts, int fi2, int li2, KHE_RESOURCE r,
  KHE_EVENT_TIMETABLE_MONITOR etm, int *NULL_to_r)
{
  bool res;  int ts_count;
  ts_count = KheTaskSetTaskCount(ts);
  res = KheAddUnassignedTasks(frame, fi1, li1, r, etm, ts) &&
        KheAddUnassignedTasks(frame, fi2, li2, r, etm, ts) &&
	KheTaskSetTaskCount(ts) > 0 &&
	KheTaskSetBounded(ts, frame, fi1, li2)    &&
	KheEjectingTaskSetMoveFrame(ts, r, true, frame);
  *NULL_to_r = KheTaskSetTaskCount(ts);
  KheTaskSetDropFromEnd(ts, KheTaskSetTaskCount(ts) - ts_count);
  return res;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDaySetAssignRepair(KHE_EJECTOR ej, int fi1, int li1,             */
/*    KHE_TASK_SET ts, int fi2, int li2, KHE_RESOURCE r)                     */
/*                                                                           */
/*  Carry out one repair, a day-set assign with these attributes.            */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused
static bool KheDaySetAssignRepair(KHE_EJECTOR ej, int fi1, int li1,
  KHE_TASK_SET ts, int fi2, int li2, KHE_RESOURCE r)
{
  bool success;  KHE_FRAME frame;  KHE_EVENT_TIMETABLE_MONITOR etm;
  int NULL_to_r;
  et m = KheEjectorEventTimetableMonitor(ej);
  frame = KheEjectorFrame(ej);
  KheEjectorRepairBe gin(ej);
  success = KheDaySetAssign(frame, fi1, li1, ts, fi2, li2, r, etm, &NULL_to_r);
  if( DEBUG5 || KheEjectorCurrDebug(ej) )
    KheOpDebug("DaySetAssign", success, frame, fi1, li2, r, ts, NULL, NULL);
  return KheEjectorRep airEnd(ej, KHE_REPAIR_DAY_SET_ASSIGN, success);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  Submodule "task set move and double move repairs and augments"           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_REPAIR_TYPE KheMoveTypeToTaskSetRepairType(KHE_MOVE_TYPE mt)         */
/*                                                                           */
/*  Convert move type mt into the corresponding task set repair type.        */
/*                                                                           */
/*****************************************************************************/

/* ***
static KHE_REPAIR_TYPE KheMoveTypeToTaskSetRepairType(KHE_MOVE_TYPE mt)
{
  switch( mt )
  {
    case KHE_MOVE_UNCHECKED:	return KHE_REPAIR_TASK_SET_MOVE_UNCHECKED;
    case KHE_MOVE_CHECKED:	return KHE_REPAIR_TASK_SET_MOVE_CHECKED;
    case KHE_MOVE_EJECTING:	return KHE_REPAIR_TASK_SET_MOVE_EJECTING;
    case KHE_MOVE_KEMPE:	return KHE_REPAIR_TASK_SET_MOVE_KEMPE;

    default:
      HnAbort("KheMoveTypeToTaskRepairType internal error");
      return KHE_REPAIR_TASK_SET_MOVE_UNCHECKED;  ** keep compiler happy **
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskSetUnAssignRepair(KHE_EJECTOR ej, KHE_TASK_SET ts)           */
/*                                                                           */
/*  Try unassigning the tasks of ts.                                         */
/*                                                                           */
/*****************************************************************************/

/* *** omitted since KheTaskSetMoveRepair(ej, ts, NULL, -) is the same
static bool KheTaskSetUnAssignRepair(KHE_EJECTOR ej, KHE_TASK_SET ts)
{
  KHE_TASK task;  int i;  bool success;
  if( KheTaskSetTaskCount(ts) > 0 )
  {
    KheEjectorRe pairBegin(ej);
    success = true;
    for( i = 0;  success && i < KheTaskSetTaskCount(ts);  i++ )
    {
      task = KheTaskSetTask(ts, i);
      success = KheTaskUnAssign(task);
    }
    if( DEBUG18 || KheEjectorCurrDebug(ej) )
    {
      fprintf(stderr, "%cTaskSetUnAssign(", success ? '+' : '-');
      KheTaskSetDebug(ts, 1, -1, stderr);
      fprintf(stderr, ")");
    }
    return KheEjectorR epairEnd(ej, KHE_REPAIR_TASK_SET_UNASSIGN, success);
  }
  else
    return false;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheTaskSetCheckProperRoots(KHE_TASK_SET ts)                         */
/*                                                                           */
/*  Check that the tasks of ts are proper roots.                             */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheTaskSetCheckProperRoots(KHE_TASK_SET ts)
{
  int i;  KHE_TASK task;
  for( i = 0;  i < KheTaskSetTaskCount(ts);  i++ )
  {
    task = KheTaskSetTask(ts, i);
    HnAssert(KheTaskProperRoot(task) == task,
      "KheTaskSetCheckProperRoots internal error");
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskSetMoveRepair(KHE_EJECTOR ej, KHE_TASK_SET ts,               */
/*    KHE_RESOURCE r, bool nocost_off)                                       */
/*                                                                           */
/*  Try a task-set move of the tasks of ts to r (which could be NULL).       */
/*                                                                           */
/*  Ordinarily, tasks for which non-assignment has no cost are considered    */
/*  to be the same as free time.  They may be present, needing to be         */
/*  ejected, so ejecting moves are used.  But if nocost_off is true,         */
/*  they are not considered to be free time and will not be present and      */
/*  so do not need to be ejected, and so ordinary task moves are used.       */
/*                                                                           */
/*****************************************************************************/
/* ***
static bool KheTaskSetAllTasksAtFreeTimes(KHE_TASK_SET ts, KHE_FRAME frame,
  KHE_RESOURCE r);
*** */

/* *** becoming obsolete
static bool KheTaskSetMoveRepair(KHE_EJECTOR ej, KHE_TASK_SET ts,
  KHE_RESOURCE r ** , bool nocost_off **)
{
  bool success;  KHE_COST init_cost;  KHE_SOLN soln;
  HnAssert(KheTaskSetTaskCount(ts) > 0, "KheTaskSetMoveRepair: ts is empty");
  ** ***
  if( !K heTaskSetVisited(ts, 0) )
  {
    fprintf(stderr, "  task set ");
    KheTaskSetDebug(ts, 2, -1, stderr);
    fprintf(stderr, "  is unvisited\n");
    HnAbort("KheTaskSetMoveRepair: ts is unvisited");
  }
  *** **
  KheEjectorRep airBegin(ej);
  KheNoClashesCheck(ej, r);
  soln = KheEjectorSoln(ej);
  init_cost = KheSolnCost(soln);
  ** ***
  if( nocost_off )
    success = KheTaskSetMove(ts, r);
  else
  *** **
    success = KheEjectingTaskSetMoveFrame(ts, r, true, KheEjectorFrame(ej));
  if( KheClashes(ej, r) || DEBUG18 || KheEjectorCurrDebug(ej) )
  {
    fprintf(stderr, "%cTaskSetMove(", success ? '+' : '-');
    KheTaskSetDebug(ts, 2, -1, stderr);
    fprintf(stderr, ", %s) %s", r == NULL ? "-" : KheResourceId(r),
      KheTaskSetAnyVisited(ts, 0) ? "visited" : "unvisited");
    ** ***
    fprintf(stderr, ", %s, no_cost %s) %s", r == NULL ? "-" : KheResourceId(r),
      nocost_off ? "off" : "on",
      KheTaskSetVisited(ts, 0) ? "visited" : "unvisited");
    *** **
    if( success )
      fprintf(stderr, " %.5f -> %.5f", KheCostShow(init_cost),
	KheCostShow(KheSolnCost(soln)));
    if( KheClashes(ej, r) )
    {
      KHE_RESOURCE_TIMETABLE_MONITOR rtm;
      rtm = KheResourceTimetableMonitor(KheEjectorSoln(ej), r);
      KheResourceTimetableMonitorPrintTimetable(rtm, 10, 2, stderr);
    }
  }
  ** KheTaskSetVisitEquivalent(ts); **
  KheNoClashesCheck(ej, r);
  return KheEjectorR epairEnd(ej, KHE_REPAIR_TASK_SET_MOVE, success);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskSetEjectingMoveRepair(KHE_EJECTOR ej, KHE_TASK_SET ts,       */
/*    KHE_RESOURCE r)                                                        */
/*                                                                           */
/*  Try an ejecting task-set move of the tasks of ts to r (possibly NULL).   */
/*                                                                           */
/*****************************************************************************/

/* ***  added "ejecting" flag to KheTaskSetMoveRepair now
static bool KheTaskSetEjectingMoveRepair(KHE_EJECTOR ej, KHE_TASK_SET ts,
  KHE_RESOURCE r)
{
  bool success;  KHE_FRAME days_frame;
  HnAssert(KheTaskSetTaskCount(ts) > 0,
    "KheTaskSetEjectingMoveRepair: ts is empty");
  if( !K heTaskSetVisited(ts, 0) )
  {
    fprintf(stderr, "  task set ");
    KheTaskSetDebug(ts, 2, -1, stderr);
    fprintf(stderr, "  is unvisited\n");
    HnAbort("KheTaskSetEjectingMoveRepair: ts is unvisited");
  }
  KheEjectorRep airBegin(ej);
  days_frame = KheEjectorFrame(ej);
  KheNoClashesCheck(ej, r);
  success = KheEjectingTaskSetMoveFrame(ts, r, true, days_frame);
  if( KheClashes(ej, r) || DEBUG18 || KheEjectorCurrDebug(ej) )
  {
    fprintf(stderr, "%cTaskSetEjectingMove(", success ? '+' : '-');
    KheTaskSetDebug(ts, 2, -1, stderr);
    fprintf(stderr, ", %s) %s", r == NULL ? "-" : KheResourceId(r),
      K heTaskSetVisited(ts, 0) ? "visited" : "unvisited");
    if( KheClashes(ej, r) )
    {
      KHE_RESOURCE_TIMETABLE_MONITOR rtm;
      rtm = KheResourceTimetableMonitor(KheEjectorSoln(ej), r);
      KheResourceTimetableMonitorPrintTimetable(rtm, 10, 2, stderr);
    }
  }
  KheNoClashesCheck(ej, r);
  return KheEjectorRe pairEnd(ej, KHE_REPAIR_TASK_SET_MOVE, success);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskSetD oubleMoveRepair(KHE_EJECTOR ej, KHE_TASK_SET ts1,       */
/*    KHE_RESOURCE r1, KHE_TASK_SET ts2, KHE_RESOURCE r2)                    */
/*                                                                           */
/*  Carry out a repair whick moves ts1 to r1 and ts2 to r2.                  */
/*                                                                           */
/*  Ordinarily, tasks for which non-assignment has no cost are considered    */
/*  to be the same as free time.  They may be present, needing to be         */
/*  ejected, so ejecting moves are used.  But if nocost_off is true,         */
/*  they are not considered to be free time and will not be present and      */
/*  so do not need to be ejected, and so ordinary task moves are used.       */
/*                                                                           */
/*****************************************************************************/

/* *** becoming obsolete
static bool KheTaskSetDoub leMoveRepair(KHE_EJECTOR ej, KHE_TASK_SET ts1,
  KHE_RESOURCE r1, KHE_TASK_SET ts2, KHE_RESOURCE r2 ** , bool nocost_off **)
{
  bool success, success2;  KHE_FRAME days_frame;  KHE_COST init_cost;
  KHE_SOLN soln;
  HnAssert(KheTaskSetTaskCount(ts1) > 0,
    "KheTaskSetDo ubleMoveRepair: ts1 is empty");
  ** KheTaskSetCheckProperRoots(ts1); **
  HnAssert(KheTaskSetTaskCount(ts2) > 0,
    "KheTaskSetD oubleMoveRepair: ts2 is empty");
  ** KheTaskSetCheckProperRoots(ts2); **
  HnAssert(r1 != r2, "KheTaskSetD oubleMoveRepair: r1 == r2");
  KheEjectorRepa irBegin(ej);
  KheNoClashesCheck(ej, r1);
  KheNoClashesCheck(ej, r2);
  soln = KheEjectorSoln(ej);
  init_cost = KheSolnCost(soln);
  ** ***
  if( nocost_off )
  {
    success = KheTaskSetMove(ts1, r1);
    if( success )
      success2 = KheTaskSetMove(ts2, r2);
    else
      success2 = false;
  }
  else
  {
  *** **
    days_frame = KheEjectorFrame(ej);
    success = KheEjectingTaskSetMoveFrame(ts1, r1, true, days_frame);
    if( success )
      success2 = KheEjectingTaskSetMoveFrame(ts2, r2, true, days_frame);
    else
      success2 = false;
  ** } **
  ** success = KheTypedTaskSetMoveFrame(ts, r, KHE_MOVE_EJECTING, frame); **
  if( (success && (KheClashes(ej, r1) || KheClashes(ej, r2))) ||
      DEBUG18 || KheEjectorCurrDebug(ej) )
  {
    fprintf(stderr, "%cTaskSetD oubleMove(", success ? '+' : '-');
    KheTaskSetDebug(ts1, 2, -1, stderr);
    fprintf(stderr, " -> %s, %s", r1 == NULL ? "-" : KheResourceId(r1),
      success2 ? "" : " failed: ");
    KheTaskSetDebug(ts2, 2, -1, stderr);
    fprintf(stderr, " -> %s)", r2 == NULL ? "-" : KheResourceId(r2));
    ** ***
    fprintf(stderr, " -> %s, nocost %s)", r2 == NULL ? "-" : KheResourceId(r2),
      nocost_off ? "off" : "on");
    *** **
    if( success )
      fprintf(stderr, " %.5f -> %.5f", KheCostShow(init_cost),
	KheCostShow(KheSolnCost(soln)));
  }
  if( success )
  {
    KheNoClashesCheck(ej, r1);
    KheNoClashesCheck(ej, r2);
  }
  return KheEjectorRe pairEnd(ej, KHE_REPAIR_TASK_SET_DOUBLE_MOVE, success);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskSetDoMove(KHE_TASK_SET ts, KHE_RESOURCE r)                   */
/*                                                                           */
/*  Move the tasks of ts to r (which could be NULL), returning false if any  */
/*  fail to move.                                                            */
/*                                                                           */
/*****************************************************************************/

static bool KheTaskSetDoMove(KHE_TASK_SET ts, KHE_RESOURCE r)
{
  int i;  KHE_TASK task;
  for( i = 0;  i < KheTaskSetTaskCount(ts);  i++ )
  {
    task = KheTaskSetTask(ts, i);
    if( !KheTaskMoveResource(task, r) )
      return false;
  }
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskSetDifferenceMove(KHE_TASK_SET ts, KHE_TASK_SET not_ts,      */
/*    KHE_RESOURCE r)                                                        */
/*                                                                           */
/*  Move the tasks of ts (which could be NULL) to r (which could be NULL),   */
/*  returning false if any fail to move.  Except don't move any tasks        */
/*  lying in not_ts (which cannot be NULL).                                  */
/*                                                                           */
/*****************************************************************************/

static bool KheTaskSetDifferenceMove(KHE_TASK_SET ts, KHE_TASK_SET not_ts,
  KHE_RESOURCE r)
{
  int i, pos;  KHE_TASK task;
  if( ts != NULL )
    for( i = 0;  i < KheTaskSetTaskCount(ts);  i++ )
    {
      task = KheTaskSetTask(ts, i);
      if( !KheTaskSetContainsTask(not_ts, task, &pos) )
      {
	if( !KheTaskMoveResource(task, r) )
	  return false;
      }
    }
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDebugOneMove(KHE_RESOURCE r, KHE_TASK_SET assign_r_ts,           */
/*    KHE_TASK_SET unassign_r_ts)                                            */
/*                                                                           */
/*  Debug r, assign_r_ts, and unassign_r_ts (which could be NULL).           */
/*                                                                           */
/*****************************************************************************/

static void KheDebugOneMove(KHE_RESOURCE r1, KHE_RESOURCE r2,
  KHE_TASK_SET assign_r2_ts, KHE_TASK_SET unassign_r2_ts)
{
  if( assign_r2_ts != NULL && KheTaskSetTaskCount(assign_r2_ts) > 0 )
  {
    fprintf(stderr, "%s -> %s ", r1 == NULL ? "@" : KheResourceId(r1),
      r2 == NULL ? "@" : KheResourceId(r2));
    KheTaskSetDebug(assign_r2_ts, 2, -1, stderr);
  }
  if( unassign_r2_ts != NULL && KheTaskSetTaskCount(unassign_r2_ts) > 0 )
  {
    fprintf(stderr, ", %s -> - ", r2 == NULL ? "@" : KheResourceId(r2));
    KheTaskSetDebug(unassign_r2_ts, 2, -1, stderr);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskSetMoveRepair(KHE_EJECTOR ej, KHE_RESOURCE r,                */
/*    KHE_TASK_SET assign_r_ts, KHE_TASK_SET unassign_r_ts)                  */
/*                                                                           */
/*  Carry out a repair which unassigns the tasks of unassign_r_ts from r     */
/*  and assigns the tasks of assign_r_ts to r.  The task sets are disjoint.  */
/*                                                                           */
/*  Here assign_ts1 must be non-NULL, but unassign_ts1 may be NULL.          */
/*                                                                           */
/*  If r == NULL, the tasks of assign_r_ts are to be unassigned, and         */
/*  unassign_r_ts must be empty.                                             */
/*                                                                           */
/*****************************************************************************/

static bool KheTaskSetMoveRepair(KHE_EJECTOR ej, KHE_RESOURCE r1,
  KHE_RESOURCE r2, KHE_TASK_SET assign_r2_ts, KHE_TASK_SET unassign_r2_ts)
{
  bool success;
  KheEjectorRepairBegin(ej);
  success = (unassign_r2_ts == NULL || KheTaskSetDoMove(unassign_r2_ts, NULL))
    && KheTaskSetDoMove(assign_r2_ts, r2);
  if( DEBUG18 || KheEjectorCurrDebug(ej) )
  {
    fprintf(stderr, "%cTaskSetMove(", success ? '+' : '-');
    KheDebugOneMove(r1, r2, assign_r2_ts, unassign_r2_ts);
    fprintf(stderr, ")");
  }
  if( success )
  {
    KheNoClashesCheck(ej, r1);
    KheNoClashesCheck(ej, r2);
  }
  return KheEjectorRepairEnd(ej, KHE_REPAIR_TASK_SET_REPLACE, success);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskSetDoubleMoveRepair(KHE_EJECTOR ej,                          */
/*    KHE_RESOURCE r1, KHE_TASK_SET assign_ts1, KHE_TASK_SET unassign_ts1,   */
/*    KHE_RESOURCE r2, KHE_TASK_SET assign_ts2, KHE_TASK_SET unassign_ts2)   */
/*                                                                           */
/*  Carry out a repair which unassigns the tasks of unassign_ts1 from r1,    */
/*  assigns the tasks of assign_ts1 to r1, unassigns the tasks of            */
/*  unassign_ts2 from r2, and assigns the tasks of assign_ts2 to r2.         */
/*                                                                           */
/*  Here assign_ts1 and assign_ts2 must be non-NULL, but unassign_ts1 and    */
/*  unassign_ts2 may be NULL.                                                */
/*                                                                           */
/*  If r1 == NULL, the tasks of assign_ts1 are to be unassigned, and         */
/*  unassign_ts1 must be empty.  If r2 == NULL, the tasks of assign_ts2      */
/*  are to be unassigned, and unassign_ts2 must be empty.                    */
/*                                                                           */
/*  The task sets must be disjoint, except that assign_ts1 and unassign_ts2  */
/*  may have common elements, in which case the assignment to r1 (possibly   */
/*  NULL) is done, and assign_ts2 and unassign_ts1 may have common           */
/*  elements, in which case the assignment to r2 (possibly NULL) is done.    */
/*                                                                           */
/*****************************************************************************/

static bool KheTaskSetDoubleMoveRepair(KHE_EJECTOR ej,
  KHE_RESOURCE r1, KHE_TASK_SET assign_ts1, KHE_TASK_SET unassign_ts1,
  KHE_RESOURCE r2, KHE_TASK_SET assign_ts2, KHE_TASK_SET unassign_ts2)
{
  bool success;
  KheEjectorRepairBegin(ej);
  success = KheTaskSetDifferenceMove(unassign_ts1, assign_ts2, NULL) &&
    KheTaskSetDifferenceMove(unassign_ts2, assign_ts1, NULL) &&
    KheTaskSetDoMove(assign_ts1, r1) && KheTaskSetDoMove(assign_ts2, r2);
  if( DEBUG18 || KheEjectorCurrDebug(ej) )
  {
    fprintf(stderr, "%cTaskSetDoubleMove(", success ? '+' : '-');
    KheDebugOneMove(r2, r1, assign_ts1, unassign_ts1);
    fprintf(stderr, ", ");
    KheDebugOneMove(r1, r2, assign_ts2, unassign_ts2);
    fprintf(stderr, ")");
  }
  if( success )
  {
    KheNoClashesCheck(ej, r1);
    KheNoClashesCheck(ej, r2);
  }
  return KheEjectorRepairEnd(ej, KHE_REPAIR_TASK_SET_DOUBLE_REPLACE, success);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskSetDoubleEjectingMoveRepair(KHE_EJECTOR ej,                  */
/*    KHE_TASK_SET ts1, KHE_RESOURCE r1, KHE_TASK_SET ts2, KHE_RESOURCE r2)  */
/*                                                                           */
/*  Carry out a repair whick ejecting-moves ts1 to r1 and ts2 to r2.         */
/*                                                                           */
/*****************************************************************************/

/* ***
static bool KheTaskSetDoubleEjectingMoveRepair(KHE_EJECTOR ej,
  KHE_TASK_SET ts1, KHE_RESOURCE r1, KHE_TASK_SET ts2, KHE_RESOURCE r2)
{
  bool success;  KHE_FRAME days_frame;
  HnAssert(KheTaskSetTaskCount(ts1) > 0,
    "KheTaskSetDoubleEjectingMoveRepair: ts1 is empty");
  ** KheTaskSetCheckProperRoots(ts1); **
  HnAssert(KheTaskSetTaskCount(ts2) > 0,
    "KheTaskSetDoubleEjectingMoveRepair: ts2 is empty");
  ** KheTaskSetCheckProperRoots(ts2); **
  KheEjectorRe pairBegin(ej);
  KheNoClashesCheck(ej, r1);
  KheNoClashesCheck(ej, r2);
  days_frame = KheEjectorFrame(ej);
  success = KheEjectingTaskSetMoveFrame(ts1, r1, true, days_frame) &&
    KheEjectingTaskSetMoveFrame(ts2, r2, true, days_frame);
  ** success = KheTypedTaskSetMoveFrame(ts, r, KHE_MOVE_EJECTING, frame); **
  if( (success && (KheClashes(ej, r1) || KheClashes(ej, r2))) ||
      DEBUG18 || KheEjectorCurrDebug(ej) )
  {
    fprintf(stderr, "%cTaskSetDoubleEjectingMove(", success ? '+' : '-');
    KheTaskSetDebug(ts1, 2, -1, stderr);
    fprintf(stderr, " -> %s, ", r1 == NULL ? "-" : KheResourceId(r1));
    KheTaskSetDebug(ts2, 2, -1, stderr);
    fprintf(stderr, " -> %s)", r2 == NULL ? "-" : KheResourceId(r2));
  }
  if( success )
  {
    KheNoClashesCheck(ej, r1);
    KheNoClashesCheck(ej, r2);
  }
  return KheEjecto rRepairEnd(ej, KHE_REPAIR_TASK_SET_DOUBLE_MOVE, success);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheTimeGroupConta insAssignedTask(KHE_TIME_GROUP tg,                */
/*    KHE_RESOURCE_TIMETABLE_MONITOR rtm, KHE_TASK *res)                     */
/*                                                                           */
/*  If tg contains an assigned task currently assigned rtm's resource, set   */
/*  *res to one such task and return true, otherwise return false.  Use rtm  */
/*  to find these tasks.                                                     */
/*                                                                           */
/*  Preassigned tasks are not included, because these tasks are being        */
/*  gathered so that they can be moved, and preassigned tasks should         */
/*  not be moved.                                                            */
/*                                                                           */
/*****************************************************************************/

/* *** already present elsewhere in this file
static bool KheTimeGroupCon tainsAssignedTask(KHE_TIME_GROUP tg,
  KHE_RESOURCE_TIMETABLE_MONITOR rtm, KHE_TASK *res)
{
  int i;  KHE_TIME t;  KHE_TASK task;  KHE_RESOURCE r;
  for( i = 0;  i < KheTimeGroupTimeCount(tg);  i++ )
  {
    t = KheTimeGroupTime(tg, i);
    if( KheResourceTimetableMonitorTimeTaskCount(rtm, t) > 0 )
    {
      task = KheResourceTimetableMonitorTimeTask(rtm, t, 0);
      if( KheTaskIsPreassigned(task, &r) )
	return *res = NULL, false;
      else
	return *res = task, true;
    }
  }
  return *res = NULL, false;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheAddAssignedTasksFromBefore(KHE_EJECTOR ej, int index,            */
/*    KHE_RESOURCE r, KHE_TASK_SET ts)                                       */
/*                                                                           */
/*  Find one assigned task assigned to r in each of the up to MAX_MOVE time  */
/*  groups before the time group at index in the days frame; add them to ts. */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused
static void KheAddAssignedTasksFromBefore(KHE_EJECTOR ej, int index,
  KHE_RESOURCE r, KHE_TASK_SET ts)
{
  KHE_SOLN soln;  KHE_TIME_GROUP tg;  KHE_POLARITY po;  int i;
  KHE_TASK task;  KHE_FRAME days_frame;  KHE_RESOURCE_TIMETABLE_MONITOR rtm;
  soln = KheEjectorSoln(ej);
  days_frame = KheEjectorFrame(ej);
  rtm = KheResourceTimetableMonitor(soln, r);
  for( i = 1;  i <= MAX_MOVE && index - i >= 0;  i++ )
  {
    tg = KheFrameTimeGroup(days_frame, index - i, &po);
    if( !KheTimeGroupContain sAssignedTask(tg, rtm, &task) )
      break;
    KheTaskSetAddTask(ts, task);
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheAddAssignedTasksFromAfter(KHE_EJECTOR ej, int index,             */
/*    KHE_RESOURCE r, KHE_TASK_SET ts)                                       */
/*                                                                           */
/*  Find one assigned task assigned to r in each of the up to MAX_MOVE time  */
/*  groups after the time group at index in the days frame; add them to ts.  */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused
static void KheAddAssignedTasksFromAfter(KHE_EJECTOR ej, int index,
  KHE_RESOURCE r, KHE_TASK_SET ts)
{
  KHE_SOLN soln;  KHE_TIME_GROUP tg;  KHE_POLARITY po;  int i, count;
  KHE_TASK task;  KHE_FRAME days_frame;  KHE_RESOURCE_TIMETABLE_MONITOR rtm;
  soln = KheEjectorSoln(ej);
  days_frame = KheEjectorFrame(ej);
  rtm = KheResourceTimetableMonitor(soln, r);
  count = KheFrameTimeGroupCount(days_frame);
  for( i = 1;  i <= MAX_MOVE && index + i < count;  i++ )
  {
    tg = KheFrameTimeGroup(days_frame, index + i, &po);
    if( !KheTimeGroupContain sAssignedTask(tg, rtm, &task) )
      break;
    KheTaskSetAddTask(ts, task);
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheAddUnassignedTasksFromBefore(KHE_EJECTOR ej, int index,          */
/*    KHE_RESOURCE_TYPE rt, KHE_TASK_SET ts)                                 */
/*                                                                           */
/*  Find one unassigned task of type rt in each of the up to MAX_MOVE time   */
/*  groups before the time group at index in the days frame; add them to ts. */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused
static void KheAddUnassignedTasksFromBefore(KHE_EJECTOR ej, int index,
  KHE_RESOURCE_TYPE rt, KHE_TASK_SET ts)
{
  KHE_TIME_GROUP tg;  KHE_POLARITY po;  int i;
  KHE_TASK task;  KHE_FRAME days_frame; KHE_EVENT_TIMETABLE_MONITOR etm;
  days_frame = KheEjectorFrame(ej);
  et m = KheEjectorEventTimetableMonitor(ej);
  for( i = 1;  i <= MAX_MOVE && index - i >= 0;  i++ )
  {
    tg = KheFrameTimeGroup(days_frame, index - i, &po);
    if( !KheTimeGroupContainsUn assignedTask(tg, rt, etm, &task) )
      break;
    KheTaskSetAddTask(ts, task);
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheAddUnassignedTasksFromAfter(KHE_EJECTOR ej, int index,           */
/*    KHE_RESOURCE_TYPE rt, KHE_TASK_SET ts)                                 */
/*                                                                           */
/*  Find one unassigned task of type rt in each of the up to MAX_MOVE time   */
/*  groups after the time group at index in the days frame; add them to ts.  */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused
static void KheAddUnassignedTasksFromAfter(KHE_EJECTOR ej, int index,
  KHE_RESOURCE_TYPE rt, KHE_TASK_SET ts)
{
  KHE_TIME_GROUP tg;  KHE_POLARITY po;  int i, count;
  KHE_TASK task;  KHE_FRAME days_frame; KHE_EVENT_TIMETABLE_MONITOR etm;
  days_frame = KheEjectorFrame(ej);
  et m = KheEjectorEventTimetableMonitor(ej);
  count = KheFrameTimeGroupCount(days_frame);
  for( i = 1;  i <= MAX_MOVE && index + i < count;  i++ )
  {
    tg = KheFrameTimeGroup(days_frame, index + i, &po);
    if( !KheTimeGroupContai nsUnassignedTask(tg, rt, etm, &task) )
      break;
    KheTaskSetAddTask(ts, task);
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskSetTimesFree(KHE_EJECTOR ej, KHE_TASK_SET ts, KHE_RESOURCE r)*/
/*                                                                           */
/*  Return true if the times that ts is running are free for r.              */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused
static bool KheTaskSetTimesFree(KHE_EJECTOR ej, KHE_TASK_SET ts, KHE_RESOURCE r)
{
  KHE_TASK task;  KHE_FRAME frame;  KHE_RESOURCE_TIMETABLE_MONITOR rtm;  int i;
  if( r == NULL )
    return true;
  rtm = KheResourceTimetableMonitor(KheEjectorSoln(ej), r);
  frame = KheEjectorFrame(ej);
  for( i = 0;  i < KheTaskSetTaskCount(ts);  i++ )
  {
    task = KheTaskSetTask(ts, i);
    if( !KheResourceTimetableMonitorTaskAvailableInFrame(rtm, task, frame) )
      return false;
  }
  return true;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*                                                                           */
/*  Adjust *first_legal and *last_legal depending on the move reason and     */
/*  whether r is busy or free.                                               */
/*                                                                           */
/*****************************************************************************/

/* *** hmmm
static void KheAdjustLegalToMoveReason(KHE_EJECTOR ej, KHE_RESOURCE r,
  KHE_MOVE_REASON mr, int first_index, int last_index,
  int *first_legal, int *last_legal)
{
  KHE_FRAME days_frame;  KHE_SOLN soln;  KHE_RESOURCE_TIMETABLE_MONITOR rtm;
  KHE_POLARITY po;
  if( from_r == NULL )
    return;
  days_frame = KheEjectorFrame(ej);
  soln = KheEjectorSoln(soln);
  rtm = KheResourceTimetableMonitor(soln, r);
  switch( mr )
  {
    case KHE_INCREASE_WORKLOAD:

      ** we don't want *first_legal to cover any busy indexes **
      ** we don't want *last_legal to cover any busy indexes **
      break;

    case KHE_DECREASE_WORKLOAD:

      ** we don't want *first_legal to cover any free indexes **
      for( index = first_index - 1;  index >= *first_legal;  index-- )
      {
	tg = KheFrameTimeGroup(days_frame, index, &po);
	if( KheResourceTimet ableMonitorTimeGroupAvailable(rtm, tg) )
	  break;
      }
      *first_legal = index;

      ** we don't want *last_legal to cover any free indexes **
      for( index = last_index + 1;  index <= *last_legal;  index++ )
      {
	tg = KheFrameTimeGroup(days_frame, index, &po);
	if( KheResourceTimeta bleMonitorTimeGroupAvailable(rtm, tg) )
	  break;
      }
      *last_legal = index;
      break;

    case KHE_FIX_TASK:

      ** nothing to do here **
      break;

    default:

      HnAbort("KheAdjustLegalToMoveReason internal error");
      break;
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  int KheTaskSetBusyCount(KHE_TASK_SET ts, KHE_RESOURCE r)                 */
/*                                                                           */
/*  Return the number of times during the times of ts that r is busy.        */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused
static int KheTaskSetBusyCount(KHE_EJECTOR ej, KHE_TASK_SET ts, KHE_RESOURCE r)
{
  KHE_RESOURCE_TIMETABLE_MONITOR rtm;  int res, i, index;  KHE_TIME t;
  KHE_SOLN soln;  KHE_FRAME days_frame;  KHE_TASK task;
  KHE_TIME_GROUP tg;  KHE_POLARITY po;
  days_frame = KheEjectorFrame(ej);
  soln = KheEjectorSoln(ej);
  rtm = KheResourceTimetableMonitor(soln, r);
  res = 0;
  for( i = 0;  i < KheTaskSetTaskCount(ts);  i++ )
  {
    task = KheTaskSetTask(ts, i);
    t = KheMeetAsstTime(KheTaskMeet(task));
    index = KheFrameTimeIndex(days_frame, t);
    tg = KheFrameTimeGroup(days_frame, index, &po);
    if( !KheResourceTimetable MonitorTimeGroupAvailable(rtm, tg) )
      res++;
  }
  return res;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheMoveSuitsReason(KHE_EJECTOR ej, KHE_RESOURCE from_r,             */
/*    KHE_RESOURCE to_r, KHE_TASK_SET ts, KHE_MOVE_REASON mr)                */
/*                                                                           */
/*  Return true if moving the tasks of ts from from_r to to_r will           */
/*  suit (that is, will carry out the intention of) reason mr; not           */
/*  forgetting that if both resources are non-NULL, we actually swap         */
/*  their timetables.                                                        */
/*                                                                           */
/*****************************************************************************/

/* ***
static bool KheMoveSuitsReason(KHE_EJECTOR ej, KHE_RESOURCE from_r,
  KHE_RESOURCE to_r, KHE_TASK_SET ts, KHE_MOVE_REASON mr)
{
  return true;
  ** *** made things worse, I don't know why
  switch( mr )
  {
    case KHE_INCREASE_WORKLOAD:

      ** the aim is to increase the workload of to_r during ts **
      HnAssert(to_r != NULL, "KheMoveSuitsReason internal error 1");
      return KheTaskSetBusyCount(ej, ts, to_r) < KheTaskSetTaskCount(ts);

    case KHE_DECREASE_WORKLOAD:

      ** the aim is to decrease the workload of from_r **
      if( to_r == NULL )
	return true;
      else
	return KheTaskSetBusyCount(ej, ts, to_r) < KheTaskSetTaskCount(ts);
      break;

    case KHE_DECREASE_WORKLOAD_TO_ZERO:

      ** the aim is to decrease the workload of from_r to zero **
      if( to_r == NULL )
	return true;
      else
	return KheTaskSetBusyCount(ej, ts, to_r) == 0;
      break;

    case KHE_FIX_TASK:

      ** the aim is to fix a problem with the task **
      ** the repairs tried in these cases are always suitable **
      return true;

    default:

      HnAbort("KheMoveSuitsReason internal error 2");
      return false;  ** keep compiler happy **
  }
  *** **
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheTaskDoInterval(KHE_TASK task, KHE_FRAME frame,                   */
/*    int *first_index, int *last_index)                                     */
/*                                                                           */
/*  Do that part of the work of KheTaskSetInterval below                     */
/*  that pertains to task and its descendants.                               */
/*                                                                           */
/*****************************************************************************/

static void KheTaskDoInterval(KHE_TASK task, KHE_FRAME frame,
  int *first_index, int *last_index)
{
  KHE_MEET meet;  KHE_TIME t;  KHE_TASK child_task;  int i, durn, fi, li;

  /* do it for task itself */
  meet = KheTaskMeet(task);
  if( meet != NULL )
  {
    t = KheMeetAsstTime(meet);
    if( t != NULL )
    {
      durn = KheMeetDuration(meet);
      if( durn == 1 )
	fi = li = KheFrameTimeIndex(frame, t);
      else
      {
	fi = KheFrameTimeIndex(frame, t);
	li = KheFrameTimeIndex(frame, KheTimeNeighbour(t, durn - 1));
      }
      if( fi < *first_index )
	*first_index = fi;
      if( li > *last_index )
	*last_index = li;
    }
  }

  /* do it for the children of task */
  for( i = 0;  i < KheTaskAssignedToCount(task);  i++ )
  {
    child_task = KheTaskAssignedTo(task, i);
    KheTaskDoInterval(child_task, frame, first_index, last_index);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheTaskInterval(KHE_TASK task, KHE_FRAME frame,                     */
/*    int *first_index, int *last_index)                                     */
/*                                                                           */
/*  Return the first and last indexes in frame occupied by task.             */
/*                                                                           */
/*****************************************************************************/

static void KheTaskInterval(KHE_TASK task, KHE_FRAME frame,
  int *first_index, int *last_index)
{
  *first_index = KheFrameTimeGroupCount(frame) - 1;
  *last_index = 0;
  KheTaskDoInterval(task, frame, first_index, last_index);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheTaskSetInterval(KHE_TASK_SET ts, KHE_FRAME frame,                */
/*    int *first_index, int *last_index)                                     */
/*                                                                           */
/*  Return the first and last indexes in frame occupied by ts.               */
/*                                                                           */
/*****************************************************************************/

static void KheTaskSetInterval(KHE_TASK_SET ts, KHE_FRAME frame,
  int *first_index, int *last_index)
{
  int i;  KHE_TASK task;  
  *first_index = KheFrameTimeGroupCount(frame) - 1;
  *last_index = 0;
  for( i = 0;  i < KheTaskSetTaskCount(ts);  i++ )
  {
    task = KheTaskSetTask(ts, i);
    KheTaskDoInterval(task, frame, first_index, last_index);
  }
}


/* ***
static void KheTaskSetsInterval(KHE_FRAME frame,
  KHE_TASK_SET ts_before, int before, KHE_TASK_SET ts,
  KHE_TASK_SET ts_after, int after, int *first_index, int *last_index)
{
  int i;  KHE_TASK task;  
  *first_index = INT_MAX - 1;
  *last_index = -1;
  for( i = before - 1;  i >= 0;  i-- )
  {
    task = KheTaskSetTask(ts_before, i);
    KheTaskDoInterval(task, frame, first_index, last_index);
  }
  for( i = 0;  i < KheTaskSetTaskCount(ts);  i++ )
  {
    task = KheTaskSetTask(ts, i);
    KheTaskDoInterval(task, frame, first_index, last_index);
  }
  for( i = 0;  i < after;  i++ )
  {
    task = KheTaskSetTask(ts_after, i);
    KheTaskDoInterval(task, frame, first_index, last_index);
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheIntervalSubset(int fi1, int li1, int fi2, int li2)               */
/*                                                                           */
/*  Return true if [fi1 .. li1] is a subset of [fi2 .. li2].                 */
/*                                                                           */
/*****************************************************************************/

static bool KheIntervalSubset(int fi1, int li1, int fi2, int li2)
{
  return fi2 <= fi1 && li1 <= li2;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheIntervalDisjoint(int fi1, int li1, int fi2, int li2)             */
/*                                                                           */
/*  Return true if intervals [fi1 .. li1] and [fi2 .. li2] are disjoint.     */
/*                                                                           */
/*****************************************************************************/

static bool KheIntervalDisjoint(int fi1, int li1, int fi2, int li2)
{
  return li1 < fi2 || li2 < fi1;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheIntervalUnion(int *fi1, int *li1, int fi2, int li2)              */
/*                                                                           */
/*  Union [fi2 .. li2] into [*fi1 .. *li1].                                  */
/*                                                                           */
/*****************************************************************************/

static void KheIntervalUnion(int *fi1, int *li1, int fi2, int li2)
{
  if( fi2 < *fi1 ) *fi1 = fi2;
  if( li2 > *li1 ) *li1 = li2;
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheIntervalShow(int fi, int li, KHE_FRAME days_frame)              */
/*                                                                           */
/*  Show interval fi .. li of days_frame in static memory.                   */
/*                                                                           */
/*****************************************************************************/

static char *KheIntervalShow(int fi, int li, KHE_FRAME days_frame)
{
  static char buff[500];
  sprintf(buff, "%s-%s", KheTimeGroupId(KheFrameTimeGroup(days_frame, fi)),
    KheTimeGroupId(KheFrameTimeGroup(days_frame, li)));
  return buff;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheTimeGroupContainsAssignedTask(KHE_TIME_GROUP tg,                 */
/*    KHE_RESOURCE_TIMETABLE_MONITOR rtm, KHE_TASK *res)                     */
/*                                                                           */
/*  If tg contains an assigned task currently assigned rtm's resource, set   */
/*  *res to one such task and return true, otherwise return false.  Use rtm  */
/*  to find these tasks.                                                     */
/*                                                                           */
/*  Preassigned tasks are not included, because these tasks are being        */
/*  gathered so that they can be moved, and preassigned tasks should         */
/*  not be moved.                                                            */
/*                                                                           */
/*****************************************************************************/

/* *** using KheTimeGroupAddAssignedTask now (see below)
static bool KheTimeGroupContainsAssignedTask(KHE_TIME_GROUP tg,
  KHE_RESOURCE_TIMETABLE_MONITOR rtm, KHE_TASK *res)
{
  int i, j;  KHE_TIME t;  KHE_TASK task;  KHE_RESOURCE r;
  for( i = 0;  i < KheTimeGroupTimeCount(tg);  i++ )
  {
    t = KheTimeGroupTime(tg, i);
    for( j = 0;  j < KheResourceTimetableMonitorTimeTaskCount(rtm, t);  j++ )
    {
      task = KheResourceTimetableMonitorTimeTask(rtm, t, j);
      task = KheTaskProperRoot(task);
      if( !KheTaskIsPreassigned(task, &r) )
	return *res = task, true;
    }
  }
  return *res = NULL, false;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDoTimeGroupContainsUnassignedTask(KHE_TIME_GROUP tg,             */
/*    bool cost, KHE_RESOURCE_TYPE rt, KHE_EVENT_TIMETABLE_MONITOR etm,      */
/*    KHE_RESOURCE_GROUP domain, int tg_index, KHE_TASK *res)                */
/*                                                                           */
/*  Like KheTimeGroupContainsUnassignedTask just below, except that if cost  */
/*  is true, we require a task for which non-assignment has a cost.          */
/*                                                                           */
/*****************************************************************************/

/* ***
static bool KheDoTimeGroupContainsUnassignedTask(KHE_TIME_GROUP tg,
  bool cost, KHE_EVENT_TIMETABLE_MONITOR etm, KHE_RESOURCE_GROUP domain,
  int tg_index, KHE_TASK *res)
{
  int i, j, k, count, dc;  KHE_TIME t;  KHE_MEET meet;  KHE_TASK task;
  count = KheTimeGroupTimeCount(tg);
  dc = KheResourceGroupResourceCount(domain) / 2;
  for( i = 0;  i < count;  i++ )
  {
    t = KheTimeGroupTime(tg, (i + tg_index) % count);
    for( j = 0;  j < KheEventTimetableMonitorTimeMeetCount(etm, t);  j++ )
    {
      meet = KheEventTimetableMonitorTimeMeet(etm, t, j);
      for( k = 0;  k < KheMeetTaskCount(meet);  k++ )
      {
	task = KheMeetTask(meet, k);
	task = KheTaskProperRoot(task);
	if( KheTaskResourceType(task) == KheResourceGroupResourceType(domain) &&
	    KheTaskAsstResource(task) == NULL &&
	    KheResourceGroupIntersectCount(domain, KheTaskDomain(task)) >= dc
	    && (cost ? KheTaskN eedsAssignment(task) : true) )
	  return *res = task, true;
      }
    }
  }
  return *res = NULL, false;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheTimeGroupContainsUnassignedTask(KHE_TIME_GROUP tg,               */
/*    KHE_RESOURCE_TYPE rt, KHE_EVENT_TIMETABLE_MONITOR etm,                 */
/*    KHE_RESOURCE_GROUP domain, int tg_index, KHE_TASK *res)                */
/*                                                                           */
/*  If tg contains an unassigned task of type rt, set *res to one such       */
/*  task and return true, otherwise return false.  Use etm to find the       */
/*  meets running at the times of tg; and start the search at index          */
/*  tg_index of tg.                                                          */
/*                                                                           */
/*****************************************************************************/

/* ***
static bool KheTimeGroupContainsUnassignedTask(KHE_TIME_GROUP tg,
  KHE_EVENT_TIMETABLE_MONITOR etm, KHE_RESOURCE_GROUP domain,
  int tg_index, KHE_TASK *res)
{
  if( KheDoTimeGroupContainsUnassignedTask(tg, true, etm, domain,
	tg_index, res) )
    return true;
  if( KheDoTimeGroupContainsUnassignedTask(tg, false, etm, domain,
	tg_index, res) )
    return true;
  return *res = NULL, false;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheTimeGroupAddAssignedTask(KHE_TIME_GROUP tg,                      */
/*    KHE_RESOURCE_TIMETABLE_MONITOR rtm, int *first_index, int *last_index, */
/*    KHE_TASK_SET ts)                                                       */
/*                                                                           */
/*  If rtm contains a task running during tg which does not overlap          */
/*  interval (*first_index, *last_index), add that task to ts, update        */
/*  *first_index and *last_index appropriately, and return true.  Else       */
/*  change nothing and return false.                                         */
/*                                                                           */
/*  Preassigned tasks are not included, because these tasks are being        */
/*  gathered so that they can be moved, and preassigned tasks should         */
/*  not be moved.                                                            */
/*                                                                           */
/*****************************************************************************/

static bool KheTimeGroupAddAssignedTask(KHE_TIME_GROUP tg,
  KHE_RESOURCE_TIMETABLE_MONITOR rtm, KHE_FRAME days_frame,
  int *first_index, int *last_index, KHE_TASK_SET ts)
{
  int i, j, fi, li;  KHE_TIME t;  KHE_TASK task;  KHE_RESOURCE r;
  for( i = 0;  i < KheTimeGroupTimeCount(tg);  i++ )
  {
    t = KheTimeGroupTime(tg, i);
    for( j = 0;  j < KheResourceTimetableMonitorTimeTaskCount(rtm, t);  j++ )
    {
      task = KheResourceTimetableMonitorTimeTask(rtm, t, j);
      task = KheTaskProperRoot(task);
      if( !KheTaskIsPreassigned(task, &r) )
      {
	KheTaskInterval(task, days_frame, &fi, &li);
	if( KheIntervalDisjoint(*first_index, *last_index, fi, li) )
	{
	  KheTaskSetAddTask(ts, task);
	  KheIntervalUnion(first_index, last_index, fi, li);
	  return true;
	}
      }
    }
  }
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheFindAssignedTasksBefore(KHE_EJECTOR ej, int count,               */
/*    KHE_RESOURCE r, int *first_index, int *last_index,                     */
/*    KHE_TASK_SET ts_before)                                                */
/*                                                                           */
/*  Add to ts_before up to count tasks from just before *first_index which   */
/*  do not overlap [*first_index .. *last_index] and are assigned r, but     */
/*  are not preassigned.  Update the interval as we go.                      */
/*                                                                           */
/*****************************************************************************/

static void KheFindAssignedTasksBefore(KHE_EJECTOR ej, int count,
  KHE_RESOURCE r, int *first_index, int *last_index, KHE_TASK_SET ts_before)
{
  KHE_TIME_GROUP tg;  KHE_FRAME days_frame;  KHE_RESOURCE_TIMETABLE_MONITOR rtm;
  days_frame = KheEjectorFrame(ej);
  rtm = KheResourceTimetableMonitor(KheEjectorSoln(ej), r);
  KheTaskSetClear(ts_before);
  while( *first_index > 0 && KheTaskSetTaskCount(ts_before) < count )
  {
    tg = KheFrameTimeGroup(days_frame, *first_index - 1);
    if( !KheTimeGroupAddAssignedTask(tg, rtm, days_frame, first_index,
	  last_index, ts_before) )
      break;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheFindAssignedTasksAfter(KHE_EJECTOR ej, int count,                */
/*    KHE_RESOURCE r, int *first_index, int *last_index,                     */
/*    KHE_TASK_SET ts_after)                                                 */
/*                                                                           */
/*  Add to ts up to count tasks from after [*first_index .. *last_index]     */
/*  which do not overlap that interval and are assigned r.  Update the       */
/*  interval as we go.                                                       */
/*                                                                           */
/*****************************************************************************/

static void KheFindAssignedTasksAfter(KHE_EJECTOR ej, int count,
  KHE_RESOURCE r, int *first_index, int *last_index, KHE_TASK_SET ts_after)
{
  KHE_TIME_GROUP tg;  int m;  KHE_FRAME days_frame;
  KHE_RESOURCE_TIMETABLE_MONITOR rtm;
  days_frame = KheEjectorFrame(ej);
  m = KheFrameTimeGroupCount(days_frame);
  rtm = KheResourceTimetableMonitor(KheEjectorSoln(ej), r);
  KheTaskSetClear(ts_after);
  while( *last_index + 1 < m && KheTaskSetTaskCount(ts_after) < count )
  {
    tg = KheFrameTimeGroup(days_frame, *last_index + 1);
    if( !KheTimeGroupAddAssignedTask(tg, rtm, days_frame, first_index,
	  last_index, ts_after) )
      break;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheTimeGroupAddUnassignedTask(KHE_TIME_GROUP tg,                    */
/*    KHE_EVENT_TIMETABLE_MONITOR etm, KHE_RESOURCE_GROUP domain,            */
/*    int tg_start_index, KHE_FRAME days_frame, int *first_index,            */
/*    int *last_index, KHE_TASK_SET ts)                                      */
/*                                                                           */
/*  If etm contains an unassigned task whose domain is compatible with       */
/*  domain, which runs during tg, and which does not overlap interval        */
/*  (*first_index, *last_index), then add that task to ts, update            */
/*  *first_index and *last_index appropriately, and return true.             */
/*  Else change nothing and return false.                                    */
/*                                                                           */
/*  Preassigned tasks are not included, because these tasks are being        */
/*  gathered so that they can be assigned, and preassigned tasks should      */
/*  not need to be assigned.                                                 */
/*                                                                           */
/*****************************************************************************/

static bool KheTimeGroupAddUnassignedTask(KHE_TIME_GROUP tg,
  KHE_EVENT_TIMETABLE_MONITOR etm, KHE_RESOURCE_GROUP domain,
  int tg_start_index, KHE_FRAME days_frame, int *first_index,
  int *last_index, KHE_TASK_SET ts)
{
  int i, j, k, count, dc, fi, li;  KHE_TIME t;  KHE_MEET meet;  KHE_TASK task;
  KHE_RESOURCE r;
  dc = KheResourceGroupResourceCount(domain) / 2;
  count = KheTimeGroupTimeCount(tg);
  for( i = 0;  i < count;  i++ )
  {
    t = KheTimeGroupTime(tg, (i + tg_start_index) % count);
    for( j = 0;  j < KheEventTimetableMonitorTimeMeetCount(etm, t);  j++ )
    {
      meet = KheEventTimetableMonitorTimeMeet(etm, t, j);
      for( k = 0;  k < KheMeetTaskCount(meet);  k++ )
      {
	task = KheMeetTask(meet, k);
	task = KheTaskProperRoot(task);
	if( KheTaskResourceType(task) == KheResourceGroupResourceType(domain) &&
	    KheTaskAsstResource(task) == NULL &&
	    !KheTaskIsPreassigned(task, &r) &&
	    KheResourceGroupIntersectCount(domain, KheTaskDomain(task)) >= dc )
	{
	  KheTaskInterval(task, days_frame, &fi, &li);
	  if( KheIntervalDisjoint(*first_index, *last_index, fi, li) )
	  {
	    KheTaskSetAddTask(ts, task);
	    KheIntervalUnion(first_index, last_index, fi, li);
	    return true;
	  }
	}
      }
    }
  }
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheFindUnassignedTasksBefore(KHE_EJECTOR ej, int count,             */
/*    KHE_RESOURCE_GROUP domain, int tg_start_index, int *first_index,       */
/*    int *last_index, KHE_TASK_SET ts_before)                               */
/*                                                                           */
/*  Add to ts up to count tasks from before [*first_index .. *last_index]    */
/*  which do not overlap that interval and are unassigned with type rt.      */
/*  Update the interval as we go.                                            */
/*                                                                           */
/*****************************************************************************/

static void KheFindUnassignedTasksBefore(KHE_EJECTOR ej, int count,
  KHE_RESOURCE_GROUP domain, int tg_start_index, int *first_index,
  int *last_index, KHE_TASK_SET ts_before)
{
  KHE_TIME_GROUP tg;  KHE_FRAME days_frame;  KHE_EVENT_TIMETABLE_MONITOR etm;
  days_frame = KheEjectorFrame(ej);
  etm = KheEjectorEventTimetableMonitor(ej);
  KheTaskSetClear(ts_before);
  while( *first_index > 0 && KheTaskSetTaskCount(ts_before) < count )
  {
    tg = KheFrameTimeGroup(days_frame, *first_index - 1);
    if( !KheTimeGroupAddUnassignedTask(tg, etm, domain, tg_start_index,
	  days_frame, first_index, last_index, ts_before) )
      break;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheFindUnassignedTasksAfter(KHE_EJECTOR ej, int count,              */
/*    KHE_RESOURCE_GROUP domain, int tg_start_index, int *first_index,       */
/*    int *last_index, KHE_TASK_SET ts_after)                                */
/*                                                                           */
/*  Add to ts up to count tasks from after [*first_index .. *last_index]     */
/*  which do not overlap that interval and are unassigned with type rt.      */
/*  Update the interval as we go.                                            */
/*                                                                           */
/*****************************************************************************/

static void KheFindUnassignedTasksAfter(KHE_EJECTOR ej, int count,
  KHE_RESOURCE_GROUP domain, int tg_start_index, int *first_index,
  int *last_index, KHE_TASK_SET ts_after)
{
  KHE_TIME_GROUP tg;  int m;  KHE_FRAME days_frame;
  KHE_EVENT_TIMETABLE_MONITOR etm;
  days_frame = KheEjectorFrame(ej);
  m = KheFrameTimeGroupCount(days_frame);
  etm = KheEjectorEventTimetableMonitor(ej);
  KheTaskSetClear(ts_after);
  while( *last_index + 1 < m && KheTaskSetTaskCount(ts_after) < count )
  {
    tg = KheFrameTimeGroup(days_frame, *last_index + 1);
    if( !KheTimeGroupAddUnassignedTask(tg, etm, domain, tg_start_index,
	  days_frame, first_index, last_index, ts_after) )
      break;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskRunsWhenResourceIsFree(KHE_TASK task, KHE_FRAME frame,       */
/*    KHE_RESOURCE_TIMETABLE_MONITOR rtm, bool nocost_off)                   */
/*                                                                           */
/*  Return true if task runs only at times when rtm's resource is free.      */
/*                                                                           */
/*  If nocost_off is false, times when r is assigned only to tasks for       */
/*  which non-assignment has no cost are considered to be free times.        */
/*                                                                           */
/*****************************************************************************/

static bool KheTaskRunsWhenResourceIsFree(KHE_TASK task, KHE_FRAME frame,
  KHE_RESOURCE_TIMETABLE_MONITOR rtm, bool nocost_off)
{
  int i, index;  KHE_TASK child_task;  KHE_MEET meet;  KHE_TIME t;
  KHE_TIME_GROUP tg;

  /* do the job for task itself */
  meet = KheTaskMeet(task);
  if( meet != NULL )
  {
    t = KheMeetAsstTime(meet);
    if( t != NULL )
    {
      index = KheFrameTimeIndex(frame, t);
      tg = KheFrameTimeGroup(frame, index);
      if( !KheResourceTimetableMonitorTimeGroupAvailable(rtm, tg, nocost_off) )
	return false;
    }
  }

  /* do the job for task's descendants */
  for( i = 0;  i < KheTaskAssignedToCount(task);  i++ )
  {
    child_task = KheTaskAssignedTo(task, i);
    if( !KheTaskRunsWhenResourceIsFree(child_task, frame, rtm, nocost_off) )
      return false;
  }

  /* all good */
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  int KheInitialTasksOnFreeDays(KHE_TASK_SET ts, KHE_FRAME frame,          */
/*    KHE_RESOURCE r, bool nocost_off)                                       */
/*                                                                           */
/*  Return the number of initial tasks in ts which are running at times      */
/*  that r is free.  Their descendant tasks must also be free.  If r is      */
/*  NULL, all tasks are included.                                            */
/*                                                                           */
/*  If nocost_off is false, times when r is assigned only to tasks for       */
/*  which non-assignment has no cost are considered to be free times.        */
/*                                                                           */
/*  This code assumes that all tasks in ts are proper root tasks.            */
/*                                                                           */
/*****************************************************************************/

static int KheInitialTasksOnFreeDays(KHE_TASK_SET ts, KHE_FRAME frame,
  KHE_RESOURCE r, bool nocost_off)
{
  KHE_TASK task;  int i;  KHE_RESOURCE_TIMETABLE_MONITOR rtm;  KHE_SOLN soln;
  if( r == NULL )
    return KheTaskSetTaskCount(ts);
  soln = KheTaskSetSoln(ts);
  rtm = KheResourceTimetableMonitor(soln, r);
  for( i = 0;  i < KheTaskSetTaskCount(ts);  i++ )
  {
    task = KheTaskSetTask(ts, i);
    HnAssert(task == KheTaskProperRoot(task),
      "KheInitialTasksOnFreeDays internal error");
    if( !KheTaskRunsWhenResourceIsFree(task, frame, rtm, nocost_off) )
      break;
  }
  return i;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheTaskAddDays(KHE_TASK task, KHE_FRAME days_frame,                 */
/*    KHE_TIME_GROUP days[MAX_DAYS], int *days_count)                        */
/*                                                                           */
/*  Add the days that task is running in days_frame to days, increasing      */
/*  *days_count appropriately.                                               */
/*                                                                           */
/*****************************************************************************/
#define MAX_DAYS 20

static void KheTaskAddDays(KHE_TASK task, KHE_FRAME days_frame,
  KHE_TIME_GROUP days[MAX_DAYS], int *days_count)
{
  KHE_MEET meet;  KHE_TIME t, t2;  int durn, i;  KHE_TIME_GROUP tg;
  KHE_TASK child_task;

  /* do the job for task itself */
  meet = KheTaskMeet(task);
  if( meet != NULL )
  {
    t = KheMeetAsstTime(meet);
    if( t != NULL )
    {
      durn = KheMeetDuration(meet);
      for( i = 0;  i < durn;  i++ )
      {
	t2 = KheTimeNeighbour(t, i);
	tg = KheFrameTimeTimeGroup(days_frame, t2);
	if( *days_count < MAX_DAYS )
	  days[(*days_count)++] = tg;
      }
    }
  }

  /* do the job for the children of task */
  for( i = 0;  i < KheTaskAssignedToCount(task);  i++ )
  {
    child_task = KheTaskAssignedTo(task, i);
    KheTaskAddDays(child_task, days_frame, days, days_count);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDaysEqual(KHE_TIME_GROUP days1[MAX_DAYS], int days1_count,       */
/*    KHE_TIME_GROUP days2[MAX_DAYS], int days2_count)                       */
/*                                                                           */
/*  Return true if days1[0 .. days1_count-1] and days2[0 .. days2_count-1]   */
/*  contain the same time groups.                                            */
/*                                                                           */
/*  To avoid memory allocation, the number of days can be at most MAX_DAYS   */
/*  minus one, otherwise the result is false.                                */
/*                                                                           */
/*****************************************************************************/

static bool KheDaysEqual(KHE_TIME_GROUP days1[MAX_DAYS], int days1_count,
  KHE_TIME_GROUP days2[MAX_DAYS], int days2_count)
{
  int i;
  if( days1_count != days2_count )
    return false;
  else if( days1_count == 1 )
    return days1[0] == days2[0];
  else if( days1_count == MAX_DAYS )
    return false;  /* see comment above */
  else
  {
    qsort(days1, days1_count, sizeof(KHE_TIME_GROUP), &KheTimeGroupCmp);
    qsort(days2, days2_count, sizeof(KHE_TIME_GROUP), &KheTimeGroupCmp);
    for( i = 0;  i < days1_count;  i++ )
      if( days1[i] != days2[i] )
	return false;
  }
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheTasksRunOnSameDays(KHE_TASK task1, KHE_TASK task2)               */
/*                                                                           */
/*  Return true if task1 and task2 run on the same days, taking their        */
/*  child tasks into account.                                                */
/*                                                                           */
/*  To avoid memory allocation, the number of days can be at most MAX_DAYS   */
/*  minus one, otherwise the result is false.                                */
/*                                                                           */
/*****************************************************************************/

static bool KheTasksRunOnSameDays(KHE_TASK task1, KHE_TASK task2,
  KHE_FRAME days_frame)
{
  KHE_TIME_GROUP days1[MAX_DAYS], days2[MAX_DAYS];
  int days1_count, days2_count;
  days1_count = days2_count = 0;
  KheTaskAddDays(task1, days_frame, days1, &days1_count);
  KheTaskAddDays(task2, days_frame, days2, &days2_count);
  return KheDaysEqual(days1, days1_count, days2, days2_count);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskSetsRunOnSameDays(KHE_TASK_SET ts1, KHE_TASK_SET ts2,        */
/*    KHE_FRAME days_frame)                                                  */
/*                                                                           */
/*  Return true if task1 and task2 run on the same days of days_frame,       */
/*  taking their child tasks into account.                                   */
/*                                                                           */
/*  To avoid memory allocation, the number of days can be at most MAX_DAYS   */
/*  minus one, otherwise the result is false.                                */
/*                                                                           */
/*****************************************************************************/

static bool KheTaskSetsRunOnSameDays(KHE_TASK_SET ts1, KHE_TASK_SET ts2,
  KHE_FRAME days_frame)
{
  KHE_TIME_GROUP days1[MAX_DAYS], days2[MAX_DAYS];
  int days1_count, days2_count, i;  KHE_TASK task;
  days1_count = days2_count = 0;
  for( i = 0;  i < KheTaskSetTaskCount(ts1);  i++ )
  {
    task = KheTaskSetTask(ts1, i);
    KheTaskAddDays(task, days_frame, days1, &days1_count);
  }
  for( i = 0;  i < KheTaskSetTaskCount(ts2);  i++ )
  {
    task = KheTaskSetTask(ts2, i);
    KheTaskAddDays(task, days_frame, days2, &days2_count);
  }
  return KheDaysEqual(days1, days1_count, days2, days2_count);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskAtBusyButNotPreassignedDays(KHE_TASK task, KHE_FRAME frame,  */
/*    KHE_RESOURCE_TIMETABLE_MONITOR rtm)                                    */
/*                                                                           */
/*  Return true if task runs only on days when rtm's resource is busy        */
/*  but not preassigned.                                                     */
/*                                                                           */
/*****************************************************************************/

/* *** no longer used
static bool KheTaskAtBusyButNotPreassignedDays(KHE_TASK task, KHE_FRAME frame,
  KHE_RESOURCE_TIMETABLE_MONITOR rtm)
{
  int i, j, k, index;  KHE_TASK child_task, task2;  KHE_MEET meet;
  KHE_TIME t, t2, t3;  KHE_TIME_GROUP tg;  KHE_RESOURCE r2;  bool tg_busy;

  ** do the job for task itself **
  meet = KheTaskMeet(task);
  if( meet != NULL )
  {
    t = KheMeetAsstTime(meet);
    if( t != NULL )
    {
      for( i = 0;  i < KheMeetDuration(meet);  i++ )
      {
	t2 = KheTimeNeighbour(t, i);
	index = KheFrameTimeIndex(frame, t2);
	tg = KheFrameTimeGroup(frame, index);
	tg_busy = false;
	for( j = 0;  j < KheTimeGroupTimeCount(tg);  j++ )
	{
	  t3 = KheTimeGroupTime(tg, j);
	  for( k=0; k < KheResourceTimetableMonitorTimeTaskCount(rtm, t3); k++ )
	  {
	    task2 = KheResourceTimetableMonitorTimeTask(rtm, t3, k);
	    task2 = KheTaskProperRoot(task2);
	    if( KheTaskIsPreassigned(task2, &r2) )
	      return false;
	    tg_busy = true;
	  }
	}
	if( tg_busy == false )
	  return false;
      }
    }
  }

  ** do the job for task's descendants **
  for( i = 0;  i < KheTaskAssignedToCount(task);  i++ )
  {
    child_task = KheTaskAssignedTo(task, i);
    if( !KheTaskAtBusyButNotPreassignedDays(child_task, frame, rtm) )
      return false;
  }

  ** all good **
  return true;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheUnpreassignedTaskOnSameDay(KHE_TASK task, KHE_FRAME days_frame,  */
/*    KHE_RESOURCE_TIMETABLE_MONITOR rtm, KHE_TASK *task2)                   */
/*                                                                           */
/*  If rtm contains an unpreassigned task that shares at least one day of    */
/*  days_frame with task, set *task2 to the proper root of one such task     */
/*  and return true.  Otherwise set *task2 to NULL and return false.         */
/*                                                                           */
/*****************************************************************************/

static bool KheUnpreassignedTaskOnSameDay(KHE_TASK task, KHE_FRAME days_frame,
  KHE_RESOURCE_TIMETABLE_MONITOR rtm, KHE_TASK *task2)
{
  int i, j, k, index, count;  KHE_TASK child_task;  KHE_MEET meet;
  KHE_TIME t, t2, t3;  KHE_TIME_GROUP tg;  KHE_RESOURCE r2;

  /* do the job for task itself */
  meet = KheTaskMeet(task);
  if( meet != NULL )
  {
    t = KheMeetAsstTime(meet);
    if( t != NULL )
    {
      for( i = 0;  i < KheMeetDuration(meet);  i++ )
      {
	t2 = KheTimeNeighbour(t, i);
	index = KheFrameTimeIndex(days_frame, t2);
	tg = KheFrameTimeGroup(days_frame, index);
	for( j = 0;  j < KheTimeGroupTimeCount(tg);  j++ )
	{
	  t3 = KheTimeGroupTime(tg, j);
	  count = KheResourceTimetableMonitorTimeTaskCount(rtm, t3);
	  for( k = 0;  k < count;  k++ )
	  {
	    *task2 = KheResourceTimetableMonitorTimeTask(rtm, t3, k);
	    *task2 = KheTaskProperRoot(*task2);
	    if( !KheTaskIsPreassigned(*task2, &r2) )
	      return true;
	  }
	}
      }
    }
  }

  /* do the job for task's descendants */
  for( i = 0;  i < KheTaskAssignedToCount(task);  i++ )
  {
    child_task = KheTaskAssignedTo(task, i);
    if( KheUnpreassignedTaskOnSameDay(child_task, days_frame, rtm, task2) )
      return true;
  }

  /* no luck */
  return *task2 = NULL, false;
}


/*****************************************************************************/
/*                                                                           */
/*  int KheInitialCorrespondingTasks(KHE_TASK_SET ts, KHE_FRAME days_frame,  */
/*    KHE_RESOURCE r, KHE_TASK_SET unassign_ts)                              */
/*                                                                           */
/*  Set unassign_ts and return the number of tasks it contains.  This is a   */
/*  maximal number such that for all i, the ith task of unassign_ts is not   */
/*  preassigned, is a proper root task, is assigned r (r must be non-NULL),  */
/*  and runs on the same days of days_frame as the ith task of ts.           */
/*                                                                           */
/*****************************************************************************/

static int KheInitialCorrespondingTasks(KHE_TASK_SET ts, KHE_FRAME days_frame,
  KHE_RESOURCE r, KHE_TASK_SET unassign_ts)
{
  KHE_TASK task, task2;  int i;  KHE_RESOURCE_TIMETABLE_MONITOR rtm;
  KHE_SOLN soln;
  HnAssert(r != NULL, "KheInitialCorrespondingTasks error");
  KheTaskSetClear(unassign_ts);
  soln = KheTaskSetSoln(ts);
  rtm = KheResourceTimetableMonitor(soln, r);
  for( i = 0;  i < KheTaskSetTaskCount(ts);  i++ )
  {
    task = KheTaskSetTask(ts, i);
    HnAssert(task == KheTaskProperRoot(task),
      "KheInitialCorrespondingTasks internal error");
    if( !KheUnpreassignedTaskOnSameDay(task, days_frame, rtm, &task2) ||
	!KheTasksRunOnSameDays(task, task2, days_frame) )
      break;
    KheTaskSetAddTask(unassign_ts, task2);
  }
  return i;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskSetAllTasksAtFreeTimes(KHE_TASK_SET ts, KHE_FRAME frame,     */
/*    KHE_RESOURCE r)                                                        */
/*                                                                           */
/*  Return true when all the tasks of ts are running when r is free.         */
/*                                                                           */
/*****************************************************************************/

/* *** no longer used
static bool KheTaskSetAllTasksAtFreeTimes(KHE_TASK_SET ts, KHE_FRAME frame,
  KHE_RESOURCE r)
{
  return KheInitialTasksOnFreeDays(ts, frame, r) ==
    KheTaskSetTaskCount(ts);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  int KheTaskSequenceTotalDuration(KHE_TASK_SET ts_before, int before,     */
/*    KHE_TASK_SET ts, KHE_TASK_SET ts_after, int after)                     */
/*                                                                           */
/*  Return the total duration of all these tasks.                            */
/*                                                                           */
/*****************************************************************************/

/* ***
static int KheTaskSequenceTotalDuration(KHE_TASK_SET ts_before, int before,
  KHE_TASK_SET ts, KHE_TASK_SET ts_after, int after)
{
  int i, res;  KHE_TASK task;
  res = 0;
  for( i = before - 1;  i >= 0;  i-- )
  {
    task = KheTaskSetTask(ts_before, i);
    res += KheTaskTotalDuration(task);
  }
  for( i = 0;  i < KheTaskSetTaskCount(ts);  i++ )
  {
    task = KheTaskSetTask(ts, i);
    res += KheTaskTotalDuration(task);
  }
  for( i = 0;  i < after;  i++ )
  {
    task = KheTaskSetTask(ts_after, i);
    res += KheTaskTotalDuration(task);
  }
  return res;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheResourceIsBusy(KHE_FRAME days_frame, int index,                  */
/*    KHE_RESOURCE_TIMETABLE_MONITOR rtm)                                    */
/*                                                                           */
/*  Return true if the resource monitored by rtm is busy at index.           */
/*                                                                           */
/*****************************************************************************/

/* *** no longer used (dangerous, does not take account of wider tasks
static bool KheResourceIsBusy(KHE_FRAME days_frame, int index,
  KHE_RESOURCE_TIMETABLE_MONITOR rtm)
{
  KHE_TIME_GROUP tg;
  tg = KheFrameTimeGroup(days_frame, index);
  return !KheResourceTime tableMonitorTimeGroupAvailable(rtm, tg);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheResourceIsBusy(KHE_FRAME days_frame, int index,                  */
/*    KHE_RESOURCE_TIMETABLE_MONITOR rtm)                                    */
/*                                                                           */
/*  Return true if the resource monitored by rtm is busy at index, but       */
/*  the task(s) involved are not preassigned.                                */
/*                                                                           */
/*****************************************************************************/

/* ***
static bool KheResourceIsBusyAndMovable(KHE_FRAME days_frame, int index,
  KHE_RESOURCE_TIMETABLE_MONITOR rtm)
{
  KHE_TIME_GROUP tg;  bool res;  KHE_TIME t;  int j, k, count;
  res = false;
  tg = KheFrameTimeGroup(days_frame, index);
  for( j = 0;  j < KheTimeGroupTimeCount(tg);  j++ )
  {
    t = KheTimeGroupTime(tg, j);
    count = KheResourceTimetableMonitorTimeTaskCount(rtm, t);
    for( k = 0;  k < count;  k++ )
    {
      task = KheResourceTimetableMonitorTimeTask(rtm, t, k);
      task = KheTaskProperRoot(task);
      if( task != NULL )
      {
	if( KheTaskIsPreassigned(task, &r2) )
	  return false;
	else
	  res = true;
      }
    }
  }
  return res;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheNextRun(KHE_FRAME days_frame, int *first_index, int *last_index, */
/*  KHE_RESOURCE_TIMETABLE_MONITOR rtm1, KHE_RESOURCE_TIMETABLE_MONITOR rtm2)*/
/*                                                                           */
/*  Find the first maximal run of indexes of days_frame after *last_index    */
/*  such that rtm1 is busy then and rtm2 is either free or NULL.  If found,  */
/*  return true and set (*first_index, *last_index) to this run, otherwise    */
/*  return false.                                                            */
/*                                                                           */
/*  When false is returned, *last_index is set to days_count, ensuring       */
/*  that all subsequent calls to KheNextRun will also return false.          */
/*                                                                           */
/*****************************************************************************/

/* *** obsolete version
static bool KheNe xtRun(KHE_FRAME days_frame, int *first_index, int *last_index,
  KHE_RESOURCE_TIMETABLE_MONITOR rtm1, KHE_RESOURCE_TIMETABLE_MONITOR rtm2)
{
  int days_count, i;
  days_count = KheFrameTimeGroupCount(days_frame);
  *first_index = -1;
  for( i = *last_index + 1;  i < days_count;  i++ )
  {
    if( KheResourceIsBusy(days_frame, i, rtm1) &&
      (rtm2 == NULL || !KheResourceIsBusy(days_frame, i, rtm2)) )
    {
      if( *first_index == -1 )
	*first_index = i;
    }
    else
    {
      if( *first_index != -1 )
	return *last_index = i - 1, true;
    }
  }
  if( *first_index == -1 )
    return *first_index = *last_index = days_count, false;
  else
    return *last_index = i - 1, true;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KhePosSuits(KHE_FRAME frame, int index,                             */
/*    KHE_RESOURCE_TIMETABLE_MONITOR rtm2,                                   */
/*    KHE_RESOURCE_TIMETABLE_MONITOR rtm1, bool nocost_off,                  */
/*    KHE_TASK_SET run_ts)                                                   */
/*                                                                           */
/*  Return true if position index in frame is busy for rtm2, and at least    */
/*  one of the tasks that make it so is free for rtm1 (always, if rtm1 is    */
/*  NULL).  Add the proper roots of all such tasks to run_ts, except where   */
/*  already present.                                                         */
/*                                                                           */
/*  If nocost_off is false, times when rtm1 is assigned only to tasks for    */
/*  which non-assignment has no cost are considered to be free times.        */
/*                                                                           */
/*****************************************************************************/

static bool KhePosSuits(KHE_FRAME frame, int index,
  KHE_RESOURCE_TIMETABLE_MONITOR rtm2,
  KHE_RESOURCE_TIMETABLE_MONITOR rtm1, bool nocost_off,
  KHE_TASK_SET run_ts)
{
  int i, j, pos;  KHE_TASK task;  KHE_TIME_GROUP tg;  KHE_TIME t;  bool res;
  res = false;
  tg = KheFrameTimeGroup(frame, index);
  for( i = 0;  i < KheTimeGroupTimeCount(tg);  i++ )
  {
    t = KheTimeGroupTime(tg, i);
    for( j = 0;  j < KheResourceTimetableMonitorTimeTaskCount(rtm2, t);  j++ )
    {
      task = KheResourceTimetableMonitorTimeTask(rtm2, t, j);
      task = KheTaskProperRoot(task);
      if( KheTaskSetContainsTask(run_ts, task, &pos) )
	res = true;
      else if( rtm1 == NULL ||
	KheTaskRunsWhenResourceIsFree(task, frame, rtm1, nocost_off) )
      {
	KheTaskSetAddTask(run_ts, task);
	res = true;
      }
    }
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheNextRun(KHE_FRAME days_frame, int *first_index,                  */
/*    int *last_index, KHE_RESOURCE_TIMETABLE_MONITOR rtm2,                  */
/*    KHE_RESOURCE_TIMETABLE_MONITOR rtm1, bool nocost_off,                  */
/*    KHE_TASK_SET run_ts)                                                   */
/*                                                                           */
/*  Find the first maximal run of indexes of days_frame after *last_index    */
/*  such that rtm2 is busy then and rtm1 is either free for all of the       */
/*  tasks involved, or NULL.  If found, return true, set run_ts to the       */
/*  tasks involved, and set (*first_index, *last_index) to this run.         */
/*  Otherwise return false.                                                  */
/*                                                                           */
/*  When false is returned, *last_index is set to days_count, ensuring       */
/*  that all subsequent calls to KheNextRun will also return false.          */
/*                                                                           */
/*  If nocost_off is false, times when rtm1 is assigned only to tasks for    */
/*  which non-assignment has no cost are considered to be free times.        */
/*                                                                           */
/*****************************************************************************/

static bool KheNextRun(KHE_FRAME days_frame, int *first_index,
  int *last_index, KHE_RESOURCE_TIMETABLE_MONITOR rtm2,
  KHE_RESOURCE_TIMETABLE_MONITOR rtm1, bool nocost_off,
  KHE_TASK_SET run_ts)
{
  int days_count, i;
  KheTaskSetClear(run_ts);
  days_count = KheFrameTimeGroupCount(days_frame);
  *first_index = -1;
  for( i = *last_index + 1;  i < days_count;  i++ )
  {
    if( KhePosSuits(days_frame, i, rtm2, rtm1, nocost_off, run_ts) )
    {
      if( *first_index == -1 )
	*first_index = i;
    }
    else
    {
      if( *first_index != -1 )
	return *last_index = i - 1, true;
    }
  }
  if( *first_index == -1 )
    return *first_index = *last_index = days_count, false;
  else
    return *last_index = i - 1, true;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KhePrevRun(KHE_FRAME days_frame, int *first_index,                  */
/*    int *last_index, KHE_RESOURCE_TIMETABLE_MONITOR rtm2,                  */
/*    KHE_RESOURCE_TIMETABLE_MONITOR rtm1, bool nocost_off,                  */
/*    KHE_TASK_SET run_ts)                                                   */
/*                                                                           */
/*  Like KheNextRun, except that it searches before *first_index, and        */
/*  when false is returned, *first_index is set to 0, ensuring that all      */
/*  subsequent calls to KhePrevRun will also return false.                   */
/*                                                                           */
/*****************************************************************************/

static bool KhePrevRun(KHE_FRAME days_frame, int *first_index,
  int *last_index, KHE_RESOURCE_TIMETABLE_MONITOR rtm2,
  KHE_RESOURCE_TIMETABLE_MONITOR rtm1, bool nocost_off,
  KHE_TASK_SET run_ts)
{
  int i;
  KheTaskSetClear(run_ts);
  *last_index = -1;
  for( i = *first_index - 1;  i >= 0;  i-- )
  {
    if( KhePosSuits(days_frame, i, rtm2, rtm1, nocost_off, run_ts) )
    {
      if( *last_index == -1 )
	*last_index = i;
    }
    else
    {
      if( *last_index != -1 )
	return *first_index = i + 1, true;
    }
  }
  if( *last_index == -1 )
    return *first_index = *last_index = 0, false;
  else
    return *first_index = i + 1, true;
}


/* *** obsolete version
static bool KhePrevRun(KHE_FRAME days_frame, int *first_index, int *last_index,
  KHE_RESOURCE_TIMETABLE_MONITOR rtm1, KHE_RESOURCE_TIMETABLE_MONITOR rtm2)
{
  int i;
  *last_index = -1;
  for( i = *first_index - 1;  i >= 0;  i-- )
  {
    if( KheResourceIsBusy(days_frame, i, rtm1) &&
      (rtm2 == NULL || !KheResourceIsBusy(days_frame, i, rtm2)) )
    {
      if( *last_index == -1 )
	*last_index = i;
    }
    else
    {
      if( *last_index != -1 )
	return *first_index = i + 1, true;
    }
  }
  if( *last_index == -1 )
    return *first_index = *last_index = 0, false;
  else
    return *first_index = i + 1, true;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheFrameAddAssignedTasksBefore(KHE_FRAME frame,                     */
/*    int first_index, int last_index, KHE_RESOURCE_TIMETABLE_MONITOR rtm,   */
/*    int max_total_durn, KHE_TASK_SET ts, bool *whole_run)                  */
/*                                                                           */
/*  Add rtm's tasks at the left end of frame[first_index .. last_index] to   */
/*  ts (after clearing ts initially), continuing until either a preassigned  */
/*  task is reached (it is not added), or the total duration of the tasks    */
/*  added to ts equals or exceeds max_total_duration, or the whole interval  */
/*  is added.                                                                */
/*                                                                           */
/*  If the whole interval is added, set *whole_run to true.  Return true     */
/*  if the total duration of the tasks added exactly equals max_total_durn.  */
/*                                                                           */
/*****************************************************************************/

/* *** no longer used
static bool KheFrameAddAssignedTasksBefore(KHE_FRAME frame,
  int first_index, int last_index, KHE_RESOURCE_TIMETABLE_MONITOR rtm,
  int max_total_durn, KHE_TASK_SET ts, bool *whole_run)
{
  int i, j, k, count, pos, total_durn;  KHE_TASK task;  KHE_TIME t;
  KHE_RESOURCE r;  KHE_TIME_GROUP tg;
  KheTaskSetClear(ts);
  total_durn = 0;
  for( i = first_index;  i <= last_index;  i++ )
  {
    tg = KheFrameTimeGroup(frame, i);
    for( j = 0;  j < KheTimeGroupTimeCount(tg);  j++ )
    {
      t = KheTimeGroupTime(tg, j);
      count = KheResourceTimetableMonitorTimeTaskCount(rtm, t);
      for( k = 0;  k < count;  k++ )
      {
	task = KheResourceTimetableMonitorTimeTask(rtm, t, k);
	task = KheTaskProperRoot(task);
	HnAssert(task != NULL, "KheFrameAddAssignedTasksBefore");
	if( !KheTaskSetContainsTask(ts, task, &pos) )
	{
	  if( KheTaskIsPreassigned(task, &r) )
	    return *whole_run = false, total_durn == max_total_durn;
	  KheTaskSetAddTask(ts, task);
	  total_durn += KheTaskTotalDuration(task);
	  if( total_durn >= max_total_durn )
	    return *whole_run = (i == last_index), total_durn == max_total_durn;
	}
      }
    }
  }
  return *whole_run = true, total_durn == max_total_durn;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheFrameAddAssignedTasksAfter(KHE_FRAME frame,                      */
/*    int first_index, int last_index, KHE_RESOURCE_TIMETABLE_MONITOR rtm,   */
/*    int max_total_durn, KHE_TASK_SET ts, bool *whole_run)                  */
/*                                                                           */
/*  Like KheFrameAddAssignedTasksBefore, only proceeding from right to left  */
/*  rather than left to right.                                               */
/*                                                                           */
/*****************************************************************************/

/* *** no longer used
static bool KheFrameAddAssignedTasksAfter(KHE_FRAME frame,
  int first_index, int last_index, KHE_RESOURCE_TIMETABLE_MONITOR rtm,
  int max_total_durn, KHE_TASK_SET ts, bool *whole_run)
{
  int i, j, k, count, pos, total_durn;  KHE_TASK task;  KHE_TIME t;
  KHE_RESOURCE r;  KHE_TIME_GROUP tg;
  KheTaskSetClear(ts);
  total_durn = 0;
  for( i = last_index;  i >= first_index;  i-- )
  {
    tg = KheFrameTimeGroup(frame, i);
    for( j = 0;  j < KheTimeGroupTimeCount(tg);  j++ )
    {
      t = KheTimeGroupTime(tg, j);
      count = KheResourceTimetableMonitorTimeTaskCount(rtm, t);
      for( k = 0;  k < count;  k++ )
      {
	task = KheResourceTimetableMonitorTimeTask(rtm, t, k);
	task = KheTaskProperRoot(task);
	HnAssert(task != NULL, "KheFrameAddAssignedTasksAfter");
	if( !KheTaskSetContainsTask(ts, task, &pos) )
	{
	  if( KheTaskIsPreassigned(task, &r) )
	    return *whole_run = false, total_durn == max_total_durn;
	  KheTaskSetAddTask(ts, task);
	  total_durn += KheTaskTotalDuration(task);
	  if( total_durn >= max_total_durn )
	    return *whole_run = (i==first_index), total_durn == max_total_durn;
	}
      }
    }
  }
  return *whole_run = true, total_durn == max_total_durn;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheAddInitialTasks(KHE_TASK_SET run_ts, int max_total_durn,         */
/*    KHE_TASK_SET res_ts, bool *all)                                        */
/*                                                                           */
/*  Set res_ts to an initial run of tasks from run_ts, stopping just before  */
/*  the first preassigned task, or the first task whose addition would       */
/*  cause the total duration of the tasks added to exceed max_total_durn.    */
/*                                                                           */
/*  If the whole of run_ts is added, set *all to true.  Return true if the   */
/*  total duration of the tasks added exactly equals max_total_durn.         */
/*                                                                           */
/*****************************************************************************/

static bool KheAddInitialTasks(KHE_TASK_SET run_ts, int max_total_durn,
  KHE_TASK_SET res_ts, bool *all)
{
  int i, total_durn, run_ts_count, d;  KHE_TASK task;  KHE_RESOURCE r;
  KheTaskSetClear(res_ts);
  total_durn = 0;
  run_ts_count = KheTaskSetTaskCount(run_ts);
  for( i = 0;  i < run_ts_count;  i++ )
  {
    task = KheTaskSetTask(run_ts, i);
    if( /* KheTaskVisited(task, 0) || */ KheTaskIsPreassigned(task, &r) )
      break;
    d = KheTaskTotalDuration(task);
    if( total_durn + d > max_total_durn )
      break;
    KheTaskSetAddTask(res_ts, task);
    total_durn += d;
  }
  *all = (i == run_ts_count);
  return total_durn == max_total_durn;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheAddFinalTasks(KHE_TASK_SET run_ts, int max_total_durn,           */
/*    KHE_TASK_SET res_ts, bool *all)                                        */
/*                                                                           */
/*  Like KheAddInitialTasks, only adding final tasks.                        */
/*                                                                           */
/*****************************************************************************/

static bool KheAddFinalTasks(KHE_TASK_SET run_ts, int max_total_durn,
  KHE_TASK_SET res_ts, bool *all)
{
  int i, total_durn, d;  KHE_TASK task;  KHE_RESOURCE r;
  KheTaskSetClear(res_ts);
  total_durn = 0;
  for( i = KheTaskSetTaskCount(run_ts) - 1;  i >= 0;  i-- )
  {
    task = KheTaskSetTask(run_ts, i);
    if( /* KheTaskVisited(task, 0) || */ KheTaskIsPreassigned(task, &r) )
      break;
    d = KheTaskTotalDuration(task);
    if( total_durn + d > max_total_durn )
      break;
    KheTaskSetAddTask(res_ts, task);
    total_durn += d;
  }
  *all = (i == -1);
  return total_durn == max_total_durn;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDoTaskSetMoveMultiRepair(KHE_EJECTOR ej, KHE_RESOURCE r2,        */
/*    KHE_TASK_SET assign_r2_ts, KHE_TASK_SET unassign_r2_ts,                */
/*    KHE_EXPANSION_OPTIONS eo, KHE_TASK_SET scratch_ts1,                    */
/*    KHE_TASK_SET scratch_ts2)                                              */
/*                                                                           */
/*  Try repairs that assign the tasks of assign_r2_ts to r2, first clearing  */
/*  the way by unassigning the tasks of unassign_r2_ts.  The first repair    */
/*  is just the move itself; the rest are double moves which also move       */
/*  tasks the other way, so as not to change the overall workload of r2.     */
/*                                                                           */
/*  If eo->balancing_off is true, or r2 is NULL, don't do any double moves.  */
/*  Parameters assign_r1_ts and run_ts are scratch task sets.                */
/*                                                                           */
/*  We make up to eo->balancing_max attempts to repair using double moves.   */
/*  The tasks in the double moves are assign_r2_ts plus the tasks at the     */
/*  left or right end of the runs following or preceding assign_r2_ts.       */
/*                                                                           */
/*****************************************************************************/

static bool KheDoTaskSetMoveMultiRepair(KHE_EJECTOR ej, KHE_RESOURCE r1,
  KHE_RESOURCE r2, KHE_TASK_SET assign_r2_ts, KHE_TASK_SET unassign_r2_ts,
  KHE_EXPANSION_OPTIONS eo, KHE_TASK_SET assign_r1_ts, KHE_TASK_SET run_ts)
{
  int total_durn, attempts, foll_fi, foll_li, prev_fi, prev_li;
  bool all, active;  KHE_SOLN soln;
  KHE_FRAME days_frame;  KHE_RESOURCE_TIMETABLE_MONITOR rtm1, rtm2;

  /* try a move */
  if( KheTaskSetMoveRepair(ej, r1, r2, assign_r2_ts, unassign_r2_ts) )
    return true;

  /* try double moves */
  if( eo->balancing_off )
    return false;

  /* try double moves */
  if( r2 != NULL )
  {
    /* try double moves for cases where r2 is non-NULL */
    soln = KheEjectorSoln(ej);
    rtm2 = KheResourceTimetableMonitor(soln, r2);
    total_durn = KheTaskSetTotalDuration(assign_r2_ts) -
      KheTaskSetTotalDuration(unassign_r2_ts);
    days_frame = KheEjectorFrame(ej);
    /* RETRY_WITH_R1_NULL: failed idea */
    rtm1 = (r1 == NULL ? NULL : KheResourceTimetableMonitor(soln, r1));
    KheTaskSetInterval(assign_r2_ts, days_frame, &foll_fi, &foll_li);
    if( DEBUG33 && KheEjectorCurrDebug(ej) )
    {
      fprintf(stderr, "%*s[ balancing %s:\n", KheEjectorCurrDebugIndent(ej),
	"", KheIntervalShow(foll_fi, foll_li, days_frame));
      fprintf(stderr, "%*s  r1 %s, r2 %s\n", KheEjectorCurrDebugIndent(ej), "",
	r1==NULL ? "@" : KheResourceId(r1), r2==NULL ? "@" : KheResourceId(r2));
      fprintf(stderr, "%*s  assign_r2_ts ", KheEjectorCurrDebugIndent(ej), "");
      KheTaskSetDebug(assign_r2_ts, 2, 0, stderr);
    }
    prev_fi = foll_fi, prev_li = foll_li;
    active = true;
    for( attempts = 0;  active && attempts < eo->balancing_max;  attempts++ )
    {
      active = false;
      if( KheNextRun(days_frame, &foll_fi, &foll_li, rtm2, rtm1, true, run_ts) )
      {
	if( DEBUG33 && KheEjectorCurrDebug(ej) )
	{
	  fprintf(stderr, "%*s  next %s ", KheEjectorCurrDebugIndent(ej), "",
	    KheIntervalShow(foll_fi, foll_li, days_frame));
	  KheTaskSetDebug(run_ts, 2, 0, stderr);
	}
	active = true;
	if( KheAddInitialTasks(run_ts, total_durn, assign_r1_ts, &all) )
	{
	  if( KheTaskSetDoubleMoveRepair(ej, r2, assign_r2_ts, unassign_r2_ts,
		r1, assign_r1_ts, NULL) )
	    return true;
	  attempts++;
	}
	if( !all && KheAddFinalTasks(run_ts, total_durn, assign_r1_ts, &all) )
	{
	  if( KheTaskSetDoubleMoveRepair(ej, r2, assign_r2_ts, unassign_r2_ts,
		r1, assign_r1_ts, NULL) )
	    return true;
	  attempts++;
	}
      }
      if( KhePrevRun(days_frame, &prev_fi, &prev_li, rtm2, rtm1, true, run_ts) )
      {
	if( DEBUG33 && KheEjectorCurrDebug(ej) )
	{
	  fprintf(stderr, "%*s  prev %s ", KheEjectorCurrDebugIndent(ej), "",
	    KheIntervalShow(prev_fi, prev_li, days_frame));
	  KheTaskSetDebug(run_ts, 2, 0, stderr);
	}
	active = true;
	if( KheAddFinalTasks(run_ts, total_durn, assign_r1_ts, &all) )
	{
	  if( KheTaskSetDoubleMoveRepair(ej, r2, assign_r2_ts, unassign_r2_ts,
		r1, assign_r1_ts, NULL) )
	    return true;
	  attempts++;
	}
	if( !all && KheAddInitialTasks(run_ts, total_durn, assign_r1_ts, &all) )
	{
	  if( KheTaskSetDoubleMoveRepair(ej, r2, assign_r2_ts, unassign_r2_ts,
		r1, assign_r1_ts, NULL) )
	    return true;
	  attempts++;
	}
      }
    }
    /* *** failed idea
    if( total_durn == 1 && r1 != NULL )
    {
      ** retry with r1 set to NULL **
      r1 = NULL;
      goto RETRY_WITH_R1_NULL;
    }
    *** */
    if( DEBUG33 && KheEjectorCurrDebug(ej) )
      fprintf(stderr, "%*s]\n", KheEjectorCurrDebugIndent(ej), "");
  }
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDoTaskS etMoveMultiRepair(KHE_EJECTOR ej, KHE_TASK_SET ts,       */
/*    KHE_RESOURCE from_r, KHE_RESOURCE to_r, bool balancing_off,            */
/*    KHE_TASK_SET ts2, KHE_TASK_SET ts3)                                    */
/*                                                                           */
/*  Try repairs that move ts from from_r to to_r.  The first is just the     */
/*  move itself; the rest are double moves which also move the other way.    */
/*                                                                           */
/*  If balancing_off, don't do any double moves.  Task sets ts2 and ts3      */
/*  are scratch task sets, they don't convey any information.                */
/*                                                                           */
/*  We make up to MAX_ATTEMPTS attempts to repair using a double move.  The  */
/*  tasks in the double moves are ts plus the tasks at the left or right     */
/*  end of the runs following or preceding ts.                               */
/*                                                                           */
/*****************************************************************************/

/* ***
#define MAX_ATTEMPTS 12
static bool KheDoTaskSetMo veMultiRepair(KHE_EJECTOR ej, KHE_TASK_SET ts,
  KHE_RESOURCE from_r, KHE_RESOURCE to_r, bool balancing_off,
  KHE_TASK_SET ts2, KHE_TASK_SET ts3 ** , bool nocost_off **)
{
  int total_durn, attempts, foll_fi, foll_li, prev_fi, prev_li;
  bool whole_run, active;  KHE_SOLN soln;
  KHE_FRAME days_frame;  KHE_RESOURCE_TIMETABLE_MONITOR from_rtm, to_rtm;

  ** try a move **
  if( KheTaskSetMoveRepair(ej, ts, to_r ** , nocost_off **) )
    return true;

  ** try double moves **
  if( !balancing_off && to_r != NULL )
  {
    soln = KheEjectorSoln(ej);
    from_rtm = (from_r==NULL ? NULL : KheResourceTimetableMonitor(soln,from_r));
    to_rtm = KheResourceTimetableMonitor(soln, to_r);
    total_durn = KheTaskSetTotalDuration(ts);
    days_frame = KheEjectorFrame(ej);
    KheTaskSetInterval(ts, days_frame, &foll_fi, &foll_li);
    prev_fi = foll_fi, prev_li = foll_li;
    KheTaskSetClear(ts2);
    KheTaskSetClear(ts3);
    active = true;
    for( attempts = 0;  active && attempts < 12;  attempts++ )
    {
      active = false;
      if( KheNextRun(days_frame, &foll_fi, &foll_li, to_rtm, from_rtm, ts2
	    ** , nocost_off **) )
      {
	active = true;
	if( KheAddInitialTasks(ts2, total_durn, ts3, &whole_run) )
	{
	  if( KheTaskSetDoubleMoveRepair(ej, ts, to_r, ts3, from_r
	      ** , nocost_off **) )
	    return true;
	  attempts++;
	}
	if( !whole_run && KheAddFinalTasks(ts2, total_durn, ts3, &whole_run) )
	{
	  if( KheTaskSetDoubleMoveRepair(ej, ts, to_r, ts3, from_r
	      ** , nocost_off **) )
	    return true;
	  attempts++;
	}
      }
      if( KhePrevRun(days_frame, &prev_fi, &prev_li, to_rtm, from_rtm, ts2
	    ** , nocost_off **) )
      {
	active = true;
	if( KheAddFinalTasks(ts2, total_durn, ts3, &whole_run) )
	{
	  if( KheTaskSetDoubleMoveRepair(ej, ts, to_r, ts3, from_r
	      ** , nocost_off **) )
	    return true;
	  attempts++;
	}
	if( !whole_run && KheAddInitialTasks(ts2, total_durn, ts3, &whole_run) )
	{
	  if( KheTaskSetDoubleMoveRepair(ej, ts, to_r, ts3, from_r
	      ** , nocost_off **) )
	    return true;
	  attempts++;
	}
      }
    }
  }
  return false;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDoTaskSetSwapMultiRepair(KHE_EJECTOR ej, KHE_TASK_SET ts,        */
/*    KHE_RESOURCE from_r, KHE_RESOURCE to_r)                                */
/*                                                                           */
/*  Try repairs that move ts from from_r to to_r, and also the tasks that    */
/*  to_r is busy on during the times of ts to from_r.  Just one repair.      */
/*                                                                           */
/*  Task set ts2 is a scratch task set, it doesn't convey any information.   */
/*                                                                           */
/*****************************************************************************/

/* *** obsolete
static bool KheDoTaskSetRep laceSwapMultiRepair(KHE_EJECTOR ej,
   KHE_RESOURCE r2, KHE_TASK_SET assign_r2_ts, KHE_TASK_SET unassign_r2_ts,
   KHE_TASK_SET scratch_ts1, KHE_TASK_SET scratch_ts2)
{
  KHE_RESOURCE r1;  KHE_TASK_SET assign_r1_ts, unassign_r1_ts;
  KHE_FRAME days_frame;  KHE_RESOURCE_TIMETABLE_MONITOR rtm2;  KHE_SOLN soln;
  assign_r1_ts = scratch_ts1;
  unassign_r1_ts = scratch_ts2;
  soln = KheEjectorSoln(ej);
  rtm2 = KheResourceTimetableMonitor(soln, r2);
  days_frame = KheEjectorFrame(ej);
  KheTaskSetClear(assign_r1_ts);
  KheTaskSetClear(assign_r2_ts);
  r1 = KheTaskAsstResource(KheTaskSetTask(assign_r2_ts, 0));
  return KheResourceTimetableMonitorTaskSetBusyType(rtm2, assign_r2_ts,
    days_frame, assign_r1_ts, false) == KHE_BUSY_ALL &&
    KheTaskSetTotalDuration(assign_r2_ts)==KheTaskSetTotalDuration(assign_r1_ts)
    && KheTaskSetDoubleMoveRepair(ej, r1, assign_r1_ts, unassign_r1_ts,
	r2, assign_r2_ts, unassign_r2_ts);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDoTaskSetSwapMultiRepair(KHE_EJECTOR ej, KHE_TASK_SET ts,        */
/*    KHE_RESOURCE from_r, KHE_RESOURCE to_r)                                */
/*                                                                           */
/*  Try repairs that move ts from from_r to to_r, and also the tasks that    */
/*  to_r is busy on during the times of ts to from_r.  Just one repair.      */
/*                                                                           */
/*  Task set ts2 is a scratch task set, it doesn't convey any information.   */
/*                                                                           */
/*****************************************************************************/

/* *** becoming obsolete
static bool KheDoTaskSetSwapMultiRepair(KHE_EJECTOR ej, KHE_TASK_SET ts,
  KHE_RESOURCE from_r, KHE_RESOURCE to_r, KHE_TASK_SET ts2
  ** , bool nocost_off **)
{
  KHE_FRAME days_frame;  KHE_RESOURCE_TIMETABLE_MONITOR to_rtm;  KHE_SOLN soln;
  soln = KheEjectorSoln(ej);
  to_rtm = KheResourceTimetableMonitor(soln, to_r);
  days_frame = KheEjectorFrame(ej);
  KheTaskSetClear(ts2);
  return KheResourceTimetableMonitorTaskSetBusyType(to_rtm, ts, days_frame,
      ts2, ** nocost_off, ** false) == KHE_BUSY_ALL &&
    KheTaskSetTotalDuration(ts) == KheTaskSetTotalDuration(ts2) &&
    KheTaskSetDoubleMoveRepair(ej, ts, to_r, ts2, from_r ** , nocost_off **);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheTaskSetExpand(KHE_TASK_SET ts, KHE_TASK_SET ts_before,           */
/*    int before_len, KHE_TASK_SET ts_after, int after_len)                  */
/*                                                                           */
/*  Add the first before_len tasks of ts_before, and the first after_len     */
/*  tasks of ts_after, to ts.                                                */
/*                                                                           */
/*****************************************************************************/

static void KheTaskSetExpand(KHE_TASK_SET ts, KHE_TASK_SET ts_before,
  int before_len, KHE_TASK_SET ts_after, int after_len)
{
  int i, pos;  KHE_TASK task;
  for( i = 0;  i < before_len;  i++ )
  {
    task = KheTaskSetTask(ts_before, i);
    HnAssert(!KheTaskSetContainsTask(ts, task, &pos),
      "KheTaskSetExpand internal error 1");
    KheTaskSetAddTask(ts, task);
  }
  for( i = 0;  i < after_len;  i++ )
  {
    task = KheTaskSetTask(ts_after, i);
    HnAssert(!KheTaskSetContainsTask(ts, task, &pos),
      "KheTaskSetExpand internal error 2");
    KheTaskSetAddTask(ts, task);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheResourceBusyOnDayOfTime(KHE_RESOURCE_TIMETABLE_MONITOR rtm,      */
/*    KHE_FRAME days_frame, KHE_TIME t, KHE_TASK *task)                      */
/*                                                                           */
/*  If rtm's resource is busy on the day containing time t, return true      */
/*  and set *task to the proper root of a task that proves this fact.        */
/*  Otherwise return false and set *task to NULL.                            */
/*                                                                           */
/*****************************************************************************/

static bool KheResourceBusyOnDayOfTime(KHE_RESOURCE_TIMETABLE_MONITOR rtm,
  KHE_FRAME days_frame, KHE_TIME t, KHE_TASK *task)
{
  int i, index;  KHE_TIME t2;  KHE_TIME_GROUP tg;
  index = KheFrameTimeIndex(days_frame, t);
  tg = KheFrameTimeGroup(days_frame, index);
  for( i = 0;  i < KheTimeGroupTimeCount(tg);  i++ )
  {
    t2 = KheTimeGroupTime(tg, i);
    if( KheResourceTimetableMonitorTimeTaskCount(rtm, t2) > 0 )
    {
      *task = KheResourceTimetableMonitorTimeTask(rtm, t2, 0);
      return *task = KheTaskProperRoot(*task), true;
    }
  }
  return *task = NULL, false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskDoubleMoveMultiRepair(KHE_EJECTOR ej, KHE_TASK task1,        */
/*    KHE_TASK task2)                                                        */
/*                                                                           */
/*  Try repairs including the task double move task1 := r2, task2 := r1      */
/*  where r2 is what task2 is assigned to initially, and r1 is what task1    */
/*  is assigned to initially.  Also try expansions of that move.             */
/*                                                                           */
/*****************************************************************************/

/* *** no longer used
static bool KheTaskDoubleMoveMultiRepair(KHE_EJECTOR ej, KHE_TASK task1,
  KHE_TASK task2)
{
  KHE_FRAME days_frame;  KHE_SOLN soln;
  int max_extra, max_before, max_after, before, after;
  int first_index1, last_index1;
  int first_index2, last_index2;
  KHE_TASK_SET ts1, ts1_before, ts1_after;
  KHE_TASK_SET ts2, ts2_before, ts2_after;
  KHE_RESOURCE r1, r2;

  ** try the basic double move **
  ** subsumed into the cases below 
  if( KheTaskDoubleMoveRepair(ej, task1, r1, task2, r2) )
    return true;
  *** **

  ** boilerplate **
  soln = KheEjectorSoln(ej);
  days_frame = KheEjectorFrame(ej);
  max_extra = 3;

  ** find r1, ts1, ts1_before, and ts1_after **
  r1 = KheTaskAsstResource(task1);
  HnAssert(r1 != NULL, "KheTaskDoubleMoveMultiRepair internal error 1");
  KheTaskInterval(task1, days_frame, &first_index1, &last_index1);
  ts1 = KheTaskSetMake(soln);
  KheTaskSetAddTask(ts1, task1);
  ts1_before = KheTaskSetMake(soln);
  KheFindAssignedTasksBefore(ej, max_extra, r1, &first_index1,
    &last_index1, ts1_before);
  ts1_after = KheTaskSetMake(soln);
  KheFindAssignedTasksAfter(ej, max_extra, r1, &first_index1,
    &last_index1, ts1_after);

  ** find r2, ts2, ts2_before, and ts2_after **
  r2 = KheTaskAsstResource(task2);
  HnAssert(r2 != NULL, "KheTaskDoubleMoveMultiRepair internal error 2");
  KheTaskInterval(task2, days_frame, &first_index2, &last_index2);
  ts2 = KheTaskSetMake(soln);
  KheTaskSetAddTask(ts2, task2);
  ts2_before = KheTaskSetMake(soln);
  KheFindAssignedTasksBefore(ej, max_extra, r2, &first_index2,
    &last_index2, ts2_before);
  ts2_after = KheTaskSetMake(soln);
  KheFindAssignedTasksAfter(ej, max_extra, r2, &first_index2,
    &last_index2, ts2_after);

  max_before =
    min(KheTaskSetTaskCount(ts1_before), KheTaskSetTaskCount(ts2_before));
  max_after =
    min(KheTaskSetTaskCount(ts1_after), KheTaskSetTaskCount(ts2_after));

  for( before = 0;  before <= max_before;  before++ )
  {
    for( after=0;  after <= max_after && before + after <= max_extra;  after++ )
    {
      KheTaskSetExpand(ts1, ts1_before, before, ts1_after, after);
      KheTaskSetExpand(ts2, ts2_before, before, ts2_after, after);
      KheTaskSetInterval(ts1, days_frame, &first_index1, &last_index1);
      KheTaskSetInterval(ts2, days_frame, &first_index2, &last_index2);
      if( first_index1 == first_index2 && last_index1 == last_index2 )
      {
	** ts1 and ts2 cover the same days, so safe to swap **
	if( KheTaskSetDoubleMoveRepair(ej, ts1, r2, ts2, r1) )
	{
	  KheTaskSetDelete(ts1);
	  KheTaskSetDelete(ts1_before);
	  KheTaskSetDelete(ts1_after);
	  KheTaskSetDelete(ts2);
	  KheTaskSetDelete(ts2_before);
	  KheTaskSetDelete(ts2_after);
	  return true;
	}
      }
      KheTaskSetClearFromEnd(ts1, 1);
      KheTaskSetClearFromEnd(ts2, 1);
    }
  }

  ** no luck **
  KheTaskSetDelete(ts1);
  KheTaskSetDelete(ts1_before);
  KheTaskSetDelete(ts1_after);
  KheTaskSetDelete(ts2);
  KheTaskSetDelete(ts2_before);
  KheTaskSetDelete(ts2_after);
  return false;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskBlockedByTimeGroup(KHE_TASK task, KHE_TIME_GROUP tg)         */
/*                                                                           */
/*  Return true if moving task is blocked because it has a time from tg.     */
/*                                                                           */
/*****************************************************************************/

static bool KheTaskBlockedByTimeGroup(KHE_TASK task, KHE_TIME_GROUP tg)
{
  KHE_MEET meet;  KHE_TIME t, t2;  int i, durn, pos;  KHE_TASK child_task;

  /* do the job for task itself */
  meet = KheTaskMeet(task);
  t = KheMeetAsstTime(meet);
  durn = KheMeetDuration(meet);
  for( i = 0;  i < durn;  i++ )
  {
    t2 = KheTimeNeighbour(t, i);
    if( KheTimeGroupContains(tg, t2, &pos) )
      return true;
  }

  /* do the job for task's children */
  for( i = 0;  i < KheTaskAssignedToCount(task);  i++ )
  {
    child_task = KheTaskAssignedTo(task, i);
    if( KheTaskBlockedByTimeGroup(child_task, tg) )
      return true;
  }

  /* no blockage anywhere */
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskSetBlockedByTimeGroup(KHE_TASK_SET ts, KHE_TIME_GROUP tg)    */
/*                                                                           */
/*  Return true if moving ts is blocked because some of the tasks have       */
/*  times from time group tg.  Here tg may be NULL, in which case false      */
/*  is returned.                                                             */
/*                                                                           */
/*****************************************************************************/

static bool KheTaskSetBlockedByTimeGroup(KHE_TASK_SET ts, KHE_TIME_GROUP tg)
{
  int i;  KHE_TASK task;
  if( tg != NULL )
  {
    for( i = 0;  i < KheTaskSetTaskCount(ts);  i++ )
    {
      task = KheTaskSetTask(ts, i);
      if( KheTaskBlockedByTimeGroup(task, tg) )
	return true;
    }
  }
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheMonitorMonitorsTask(KHE_MONITOR d, KHE_TASK task)                */
/*                                                                           */
/*  Return true if d monitors task.                                          */
/*                                                                           */
/*****************************************************************************/

/* *** renamed, see below
static bool KheMonitorMonitorsTask(KHE_MONITOR d, KHE_TASK task)
{
  KHE_EVENT_RESOURCE er;  int i;  KHE_SOLN soln;  KHE_MONITOR m;
  soln = KheTaskSoln(task);
  er = KheTaskEventResource(task);
  for( i = 0;  i < KheSolnEventResourceMonitorCount(soln, er);  i++ )
  {
    m = KheSolnEventResourceMonitor(soln, er, i);
    if( m == d )
      return true;
  }
  return false;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskBlockedByMonitor(KHE_TASK task, KHE_MONITOR d)               */
/*                                                                           */
/*  Return true if d monitors task.                                          */
/*                                                                           */
/*****************************************************************************/

static bool KheTaskBlockedByMonitor(KHE_TASK task, KHE_MONITOR d)
{
  KHE_EVENT_RESOURCE er;  int i;  KHE_SOLN soln;  KHE_TASK child_task;

  /* do the job for task itself */
  soln = KheTaskSoln(task);
  er = KheTaskEventResource(task);
  if( er != NULL )
    for( i = 0;  i < KheSolnEventResourceMonitorCount(soln, er);  i++ )
      if( KheSolnEventResourceMonitor(soln, er, i) == d )
	return true;

  /* do the job for the children of task */
  for( i = 0;  i < KheTaskAssignedToCount(task);  i++ )
  {
    child_task = KheTaskAssignedTo(task, i);
    if( KheTaskBlockedByMonitor(child_task, d) )
      return true;
  }
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskSetBlockedByMonitor(KHE_TASK_SET ts, KHE_MONITOR d)          */
/*                                                                           */
/*  Return true if moving ts is blocked because some of the tasks are        */
/*  monitored by d.  Here d may be NULL, in which case false is returned.    */
/*                                                                           */
/*****************************************************************************/

static bool KheTaskSetBlockedByMonitor(KHE_TASK_SET ts, KHE_MONITOR d)
{
  int i;  KHE_TASK task;
  if( d != NULL )
  {
    for( i = 0;  i < KheTaskSetTaskCount(ts);  i++ )
    {
      task = KheTaskSetTask(ts, i);
      if( KheTaskBlockedByMonitor(task, d) )
	return true;
    }
  }
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskSetContainsPreassignedTask(KHE_TASK_SET ts)                  */
/*                                                                           */
/*  Return true if one or more of the tasks of ts is preassigned.            */
/*                                                                           */
/*****************************************************************************/

static bool KheTaskSetContainsPreassignedTask(KHE_TASK_SET ts)
{
  int i;  KHE_TASK task;  KHE_RESOURCE r;
  for( i = 0;  i < KheTaskSetTaskCount(ts);  i++ )
  {
    task = KheTaskSetTask(ts, i);
    if( KheTaskIsPreassigned(task, &r) )
      return true;
  }
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_REPAIR_STATUS KheTaskSetRepairStatus(KHE_RESOURCE r2,                */
/*    KHE_TASK_SET assign_r2_ts, KHE_FRAME days_frame,                       */
/*    bool reversing_off, KHE_TIME_GROUP blocking_tg,                        */
/*    KHE_MONITOR blocking_d, KHE_TASK_SET unassign_r2_ts)                   */
/*                                                                           */
/*  Investigate moving to r2 the tasks of assign_r2_ts.  Set unassign_r2_ts  */
/*  to the set of distinct proper roots of the tasks that would have to be   */
/*  unassigned from r2 in order to avoid clashes in days_frame if those      */
/*  moves were made.  The return value is                                    */
/*                                                                           */
/*    KHE_REPAIR_AVAIL_FREE - the move may be carried out, because           */
/*      r2 is either NULL or available on all the days of assign_r2_ts.      */
/*      More precisely, unassign_r2_ts is empty or contains only tasks       */
/*      t for which KheTaskNeedsAssignment(t) is false.                      */
/*                                                                           */
/*    KHE_REPAIR_AVAIL_BUSY - the move may be carried out, because r2 is     */
/*      busy on all the days of assign_r2_ts, doing tasks unassign_r2_ts,    */
/*      and the intention is that these will be moved back the other way.    */
/*                                                                           */
/*    KHE_REPAIR_BLOCKED - the repair is blocked, that is, may not be        */
/*      carried out, because r2 is free on some days and busy on others.     */
/*                                                                           */
/*  There is one case where KHE_REPAIR_AVAIL_FREE applies, but               */
/*  KheTaskSetRepairStatus nevertheless returns KHE_REPAIR_BLOCKED instead:  */
/*                                                                           */
/*    * When any of the tasks of unassign_r2_ts are preassigned; in that     */
/*      case the move would have failed to unassign it anyway.               */
/*                                                                           */
/*  There are several cases where KHE_REPAIR_AVAIL_BUSY applies, but         */
/*  KheTaskSetRepairStatus nevertheless returns KHE_REPAIR_BLOCKED instead:  */
/*                                                                           */
/*    * When reversing_off is true.                                          */
/*                                                                           */
/*    * When the first tasks of assign_r2_ts and unassign_r2_ts are          */
/*      equivalent, so that swapping them would achieve nothing (and yes,    */
/*      swapping the whole sets might achieve something).                    */
/*                                                                           */
/*    * When blocking_tg != NULL and the reverse part of the repair would    */
/*      move a task of unassign_r2_ts into blocking_tg.  The point here      */
/*      is that this would undermine the intention of the repair, which      */
/*      when blocking_tg != NULL is to reduce workload in blocking_tg.       */
/*                                                                           */
/*    * When blocking_d != NULL and the reverse part of the repair would     */
/*      move a task of unassign_r2_ts which is monitored by blocking_d.      */
/*      The point here is that this would undermine the intention of the     */
/*      repair, which when blocking_d != NULL is to reduce the number of     */
/*      tasks which are assigned resources from a particular resource group. */
/*                                                                           */
/*    * When any of the tasks of unassign_r2_ts are preassigned; in that     */
/*      case the reverse move would have failed anyway.                      */
/*                                                                           */
/*    * When the tasks of unassign_r2_ts do not cover the same set of days   */
/*      as those of assign_r2_ts.  Allowing this would complicate the        */
/*      bookkeeping beyond what seems reasonable.                            */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  KHE_REPAIR_AVAIL_FREE,
  KHE_REPAIR_AVAIL_BUSY,
  KHE_REPAIR_BLOCKED
} KHE_REPAIR_STATUS;

static KHE_REPAIR_STATUS KheTaskSetRepairStatus(KHE_RESOURCE r2,
  KHE_TASK_SET assign_r2_ts, KHE_FRAME days_frame,
  bool reversing_off, KHE_TIME_GROUP blocking_tg,
  KHE_MONITOR blocking_d, KHE_TASK_SET unassign_r2_ts)
{
  KHE_RESOURCE_TIMETABLE_MONITOR rtm2;  KHE_SOLN soln;
  KheTaskSetClear(unassign_r2_ts);
  if( r2 == NULL )
    return KHE_REPAIR_AVAIL_FREE;
  soln = KheTaskSetSoln(assign_r2_ts);
  rtm2 = KheResourceTimetableMonitor(soln, r2);
  switch( KheResourceTimetableMonitorTaskSetBusyType(rtm2, assign_r2_ts,
    days_frame, unassign_r2_ts, false) )
  {
    case KHE_BUSY_NONE:

      /* r2 is busy on none of the days of the times of assign_r2_ts */
      if( KheTaskSetContainsPreassignedTask(unassign_r2_ts) )
	return KHE_REPAIR_BLOCKED;
      else
	return KHE_REPAIR_AVAIL_FREE;

    case KHE_BUSY_SOME:

      /* r2 is busy on some but not all days of the times of assign_r2_ts */
      return KHE_REPAIR_BLOCKED;

    case KHE_BUSY_ALL:

      /* r2 is busy on all days of the times of assign_r2_ts */
      if( reversing_off ||
	  KheTaskSetFirstTasksEquivalent(assign_r2_ts, unassign_r2_ts) ||
	  KheTaskSetBlockedByTimeGroup(unassign_r2_ts, blocking_tg) ||
	  KheTaskSetBlockedByMonitor(unassign_r2_ts, blocking_d) ||
	  KheTaskSetContainsPreassignedTask(unassign_r2_ts) ||
          !KheTaskSetsRunOnSameDays(assign_r2_ts, unassign_r2_ts, days_frame) )
	return KHE_REPAIR_BLOCKED;
      else
	return KHE_REPAIR_AVAIL_BUSY;

    default:

      HnAbort("KheTaskSetRepairStatus internal error");
      return KHE_REPAIR_BLOCKED;  /* keep compiler happy */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  int KheTaskIndexInFrameTimeGroup(KHE_TASK task, KHE_FRAME days_frame)    */
/*                                                                           */
/*  Return the index in its days_frame time group of the time that task      */
/*  is running.                                                              */
/*                                                                           */
/*****************************************************************************/

static int KheTaskIndexInFrameTimeGroup(KHE_TASK task, KHE_FRAME days_frame)
{
  KHE_MEET meet;  KHE_TIME t;  KHE_TIME_GROUP tg;  int pos;
  meet = KheTaskMeet(task);
  if( meet != NULL )
  {
    t = KheMeetAsstTime(meet);
    if( t != NULL )
    {
      tg = KheFrameTimeTimeGroup(days_frame, t);
      if( KheTimeGroupContains(tg, t, &pos) )
	return pos;
      HnAbort("KheTaskIndexInFrameTimeGroup internal error");
    }
  }
  return 0;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskSetMoveMultiRepair(KHE_EJECTOR ej, KHE_TASK_SET assign_r2_ts,*/
/*    KHE_RESOURCE_GROUP rg, KHE_RESOURCE_GROUP not_rg, bool allow_unassign, */
/*    KHE_EXPANSION_OPTIONS eo, KHE_TIME_GROUP blocking_tg,                  */
/*    KHE_MONITOR blocking_d)                                                */
/*                                                                           */
/*  Try repairs that move the tasks of assign_r2_ts from whatever they are   */
/*  assigned to now to a resource r2, which must lie in rg and not in        */
/*  not_rg (if not_rg != NULL).  If allow_unassign is true, the tasks        */
/*  may be unassigned as well as moved to a resource from rg.                */
/*                                                                           */
/*  Apply widening unless eo->widening_off is true, reversing unless         */
/*  eo->reversing_off is true, and balancing unless eo->balancing_off is     */
/*  true.                                                                    */
/*                                                                           */
/*  If blocking_tg != NULL, block reversals that move tasks to blocking_tg.  */
/*  If blocking_d != NULL, block reversals that move tasks monitored by      */
/*  blocking_d.                                                              */
/*                                                                           */
/*  If the ejector allows meet moves, try them too, but without widening,    */
/*  reversing, or balancing.                                                 */
/*                                                                           */
/*****************************************************************************/
static bool KheMeetMoveAndTaskMoveMultiRepair(KHE_EJECTOR ej,
  KHE_MOVE_TYPE meet_mt, KHE_TASK task, KHE_RESOURCE r);

static bool KheTaskSetMoveMultiRepair(KHE_EJECTOR ej, KHE_TASK_SET assign_r2_ts,
  KHE_RESOURCE_GROUP rg, KHE_RESOURCE_GROUP not_rg, bool allow_unassign,
  KHE_EXPANSION_OPTIONS eo, KHE_TIME_GROUP blocking_tg,
  KHE_MONITOR blocking_d)
{
  struct khe_se_resource_set_type srs_rec;  KHE_FRAME days_frame;
  KHE_RESOURCE r1, r2;  KHE_TASK first_task;  KHE_RESOURCE_GROUP domain;
  KHE_TASK_SET assign_r2_ts_before, assign_r2_ts_after;
  KHE_TASK_SET unassign_r2_ts_before, unassign_r2_ts_after;
  KHE_TASK_SET unassign_r2_ts;
  int assign_r2_ts_count, unassign_r2_ts_count, phase;
  int extra, first_index, last_index, tg_start_index;
  int max_extra, max_before, max_after, before, after;  KHE_SOLN soln;

  /* get days frame and exit if not present */
  soln = KheEjectorSoln(ej);
  days_frame = KheEjectorFrame(ej);
  if( days_frame == NULL )
    return false;

  /* make sure assign_r2_ts is non-empty and get r1 from its first task */
  assign_r2_ts_count = KheTaskSetTaskCount(assign_r2_ts);
  HnAssert(assign_r2_ts_count > 0,
    "KheTaskSetMoveMultiRepair internal error 1");
  /* ***
  HnAssert(!KheTaskSetContainsPreassignedTask(assign_r2_ts),
    "KheTaskSetMoveMultiRepair internal error 2");
  HnAssert(KheTaskSetAllVisited(assign_r2_ts, 0),
    "KheTaskSetMoveMultiRepair internal error 2");
  *** */
  first_task = KheTaskSetTask(assign_r2_ts, 0);
  r1 = KheTaskAsstResource(first_task);

  /* get first_index and last_index, bracketing everything in assign_r2_ts */
  KheTaskSetInterval(assign_r2_ts, days_frame, &first_index, &last_index);

  /* make the five task sets we need for this operation */
  assign_r2_ts_before = KheTaskSetMake(soln);
  assign_r2_ts_after = KheTaskSetMake(soln);
  unassign_r2_ts = KheTaskSetMake(soln);
  unassign_r2_ts_before = KheTaskSetMake(soln);
  unassign_r2_ts_after = KheTaskSetMake(soln);

  /* get max_extra, assign_r2_ts_before, and assign_r2_ts_after */
  if( eo->widening_off )
  {
    max_extra = 0;
  }
  else
  {
    max_extra = max(eo->widening_max, assign_r2_ts_count) - assign_r2_ts_count;
    if( r1 != NULL )
    {
      KheFindAssignedTasksBefore(ej, max_extra, r1, &first_index,
	&last_index, assign_r2_ts_before);
      KheFindAssignedTasksAfter(ej, max_extra, r1, &first_index,
	&last_index, assign_r2_ts_after);
    }
    else
    {
      tg_start_index = KheTaskIndexInFrameTimeGroup(first_task, days_frame);
      domain = KheTaskDomain(first_task);
      KheFindUnassignedTasksBefore(ej, max_extra, domain, tg_start_index,
	&first_index, &last_index, assign_r2_ts_before);
      KheFindUnassignedTasksAfter(ej, max_extra, domain, tg_start_index,
	&first_index, &last_index, assign_r2_ts_after);
    }
    /* ***
    HnAssert(!KheTaskSetContainsPreassignedTask(assign_r2_ts_before),
      "KheTaskSetMoveMultiRepair internal error 3");
    HnAssert(!KheTaskSetContainsPreassignedTask(assign_r2_ts_after),
      "KheTaskSetMoveMultiRepair internal error 4");
    *** */
  }

  /* for each resource r2 in rg, not in not_rg, plus possibly unassignment */
  if( DEBUG30 && KheEjectorCurrDebug(ej) )
  {
    fprintf(stderr, "%*sKheTaskSetMoveMultiRepair trying resource group:\n",
      KheEjectorCurrDebugIndent(ej), "");
    KheResourceGroupDebug(rg, 2, KheEjectorCurrDebugIndent(ej), stderr);
    if( not_rg != NULL )
    {
      fprintf(stderr, "%*somitting resource group:\n",
	KheEjectorCurrDebugIndent(ej), "");
      KheResourceGroupDebug(not_rg, 2, KheEjectorCurrDebugIndent(ej), stderr);
    }
  }
  extra = KheEjectorCurrAugmentCount(ej) + 5 * KheSolnDiversifier(soln);
  for( phase = 0;  phase < 2;  phase++ )
  {
  KheForEachResource(srs_rec, rg, not_rg, allow_unassign, extra, r2)
  {
    if( DEBUG30 && KheEjectorCurrDebug(ej) )
      fprintf(stderr, "%*sKheTaskSetMoveMultiRepair trying resource %s\n",
	KheEjectorCurrDebugIndent(ej), "", r2==NULL ? "@" : KheResourceId(r2));
    if( r2 != r1 ) switch( KheTaskSetRepairStatus(r2, assign_r2_ts, days_frame,
	  eo->reversing_off, blocking_tg, blocking_d, unassign_r2_ts) )
    {
      case KHE_REPAIR_AVAIL_FREE:

	if( phase != 0 )
	  break;
	/* find the max number of tasks from ts_before and ts_after that */
	/* r2 is free for; the nocost_off parameter is true here because */
	/* r2 may not be assigned tasks that do not need assignment -    */
	/* that would make it too tedious to update unassign_r2_ts    */
	max_before = KheInitialTasksOnFreeDays(assign_r2_ts_before,
	  days_frame, r2, true);
	max_after  = KheInitialTasksOnFreeDays(assign_r2_ts_after,
	  days_frame, r2, true);

	/* for each possible number of tasks before */
	for( before = 0;  before <= max_before;  before++ )
	{
	  /* for each possible number of tasks after */
	  for( after=0; after<=max_after && before+after<=max_extra; after++ )
	  {
	    KheTaskSetExpand(assign_r2_ts, assign_r2_ts_before, before,
	      assign_r2_ts_after, after);
	    if( KheDoTaskSetMoveMultiRepair(ej, r1, r2, assign_r2_ts,
		  unassign_r2_ts, eo,
		  unassign_r2_ts_before /* just a scratch ts here */,
		  unassign_r2_ts_after  /* just a scratch ts here */) )
	    {
	      KheTaskSetClearFromEnd(assign_r2_ts, assign_r2_ts_count);
	      KheTaskSetDelete(assign_r2_ts_before);
	      KheTaskSetDelete(assign_r2_ts_after);
	      KheTaskSetDelete(unassign_r2_ts);
	      KheTaskSetDelete(unassign_r2_ts_before);
	      KheTaskSetDelete(unassign_r2_ts_after);
	      return true;
	    }
	    KheTaskSetClearFromEnd(assign_r2_ts, assign_r2_ts_count);
	  }
	}
	break;

      case KHE_REPAIR_AVAIL_BUSY:

	if( phase != 1 )
	  break;
	/* find the max number of tasks that r2 is busy for that run */
	/* on the same days as the tasks of ts_before and ts_after   */
	/* NB r2 is non-NULL in this case                            */
	max_before = KheInitialCorrespondingTasks(assign_r2_ts_before,
	  days_frame, r2, unassign_r2_ts_before);
	max_after  = KheInitialCorrespondingTasks(assign_r2_ts_after,
	  days_frame, r2, unassign_r2_ts_after);

	/* for each possible number of tasks before */
        unassign_r2_ts_count = KheTaskSetTaskCount(unassign_r2_ts);
	for( before = 0;  before <= max_before;  before++ )
	{
	  /* for each possible number of tasks after */
	  for( after=0; after<=max_after && before+after<=max_extra; after++ )
	  {
	    /* expand assign_r2_ts and unassign_r2_ts and swap them */
	    KheTaskSetExpand(assign_r2_ts, assign_r2_ts_before, before,
	      assign_r2_ts_after, after);
	    KheTaskSetExpand(unassign_r2_ts, unassign_r2_ts_before, before,
	      unassign_r2_ts_after, after);
	    if( KheTaskSetDoubleMoveRepair(ej, r1, unassign_r2_ts, NULL,
		  r2, assign_r2_ts, NULL) )
	    {
	      KheTaskSetClearFromEnd(assign_r2_ts, assign_r2_ts_count);
	      KheTaskSetDelete(assign_r2_ts_before);
	      KheTaskSetDelete(assign_r2_ts_after);
	      KheTaskSetDelete(unassign_r2_ts);
	      KheTaskSetDelete(unassign_r2_ts_before);
	      KheTaskSetDelete(unassign_r2_ts_after);
	      return true;
	    }
	    KheTaskSetClearFromEnd(assign_r2_ts, assign_r2_ts_count);
	    KheTaskSetClearFromEnd(unassign_r2_ts, unassign_r2_ts_count);
	  }
	}
	break;

      case KHE_REPAIR_BLOCKED:

	/* repair is blocked; do nothing */
	break;

      default:

	HnAbort("KheTaskSetMoveMultiRepair internal error 9");
	break;
    }
  }
  }

  /* delete the five task sets we made for this operation */
  KheTaskSetDelete(assign_r2_ts_before);
  KheTaskSetDelete(assign_r2_ts_after);
  KheTaskSetDelete(unassign_r2_ts);
  KheTaskSetDelete(unassign_r2_ts_before);
  KheTaskSetDelete(unassign_r2_ts_after);

  /* try meet plus task moves, if permitted and just one task */
  if( KheEjectorRepairTimes(ej) && assign_r2_ts_count == 1 )
  {
    extra = KheEjectorCurrAugmentCount(ej) + 5 * KheSolnDiversifier(soln);
    KheForEachResource(srs_rec, rg, not_rg, allow_unassign, extra, r2)
      if( r2 != r1 )
      {
	if( KheMeetMoveAndTaskMoveMultiRepair(ej,KHE_MOVE_KEMPE,first_task,r2) )
	  return true;
      }
  }
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskSetMoveMultiRepair(KHE_EJECTOR ej, KHE_TASK_SET ts,          */
/*    KHE_RESOURCE_GROUP rg, KHE_RESOURCE_GROUP not_rg, bool allow_unassign, */
/*    bool wide ning_off, bool reversing_off, bool balancing_off,            */
/*    bool nocost_off, KHE_TIME_GROUP blocking_tg, KHE_MONITOR blocking_d)   */
/*                                                                           */
/*  Try repairs that move the tasks of ts from whatever they are assigned    */
/*  to now to a single resource, which must lie in rg and not in not_rg      */
/*  (if not_rg != NULL).  If allow_unassigned is true, the tasks may be      */
/*  unassigned as well as moved to a resource from rg.                       */
/*                                                                           */
/*  Apply widening unless wide ning_off is true, reversing unless            */
/*  reversing_off is true, and balancing unless balancing_off is true.       */
/*                                                                           */
/*  If blocking_tg != NULL, block reversals that move tasks to blocking_tg.  */
/*  If blocking_d != NULL, block reversals that move tasks monitored by      */
/*  blocking_d.                                                              */
/*                                                                           */
/*  If the ejector allows meet moves, try them too, but without widening,    */
/*  reversing, or balancing.                                                 */
/*                                                                           */
/*****************************************************************************/

/* *** becoming obsolete
static bool KheMeetMoveAndTaskMoveMultiRepair(KHE_EJECTOR ej,
  KHE_MOVE_TYPE meet_mt, KHE_TASK task, KHE_RESOURCE r);

static bool KheTaskSetMoveMultiRepair(KHE_EJECTOR ej, KHE_TASK_SET ts,
  KHE_RESOURCE_GROUP rg, KHE_RESOURCE_GROUP not_rg, bool allow_unassign,
  bool widen ing_off, bool reversing_off, bool balancing_off,
  ** bool nocost_off, ** KHE_TIME_GROUP blocking_tg, KHE_MONITOR blocking_d)
{
  struct khe_se_resource_set_type srs_rec;  KHE_FRAME days_frame;
  KHE_RESOURCE from_r, to_r;  KHE_TASK first_task;  KHE_RESOURCE_GROUP domain;
  KHE_SOLN soln;  KHE_TASK_SET ts_before, ts_after, ts2, ts3;
  int extra, first_index, last_index, ts_count, tg_start_index;
  int max_extra, max_before, max_after, before, after;

  ** get days frame and exit if not present **
  soln = KheEjectorSoln(ej);
  days_frame = KheEjectorFrame(ej);
  if( days_frame == NULL )
    return false;

  ** make sure ts is non-empty and get from_r from its first task **
  ts_count = KheTaskSetTaskCount(ts);
  HnAssert(ts_count > 0, "KheTaskSetMoveMultiRepair internal error 1");
  HnAssert(KheTaskSetAllVisited(ts, 0),
    "KheTaskSetMoveMultiRepair internal error 2");
  first_task = KheTaskSetTask(ts, 0);
  from_r = KheTaskAsstResource(first_task);

  ** get first_index and last_index, bracketing everything in ts **
  KheTaskSetInterval(ts, days_frame, &first_index, &last_index);

  ** get max_extra, the maximum number of tasks to include beyond ts **
  max_extra = max(4, ts_count) - ts_count;

  ** get ts_before and ts_after **
  ts_before = KheTaskSetMake(soln);
  ts_after = KheTaskSetMake(soln);
  if( !wid ening_off )
  {
    if( from_r != NULL )
    {
      KheFindAssignedTasksBefore(ej, max_extra, from_r, &first_index,
	&last_index, ts_before);
      KheFindAssignedTasksAfter(ej, max_extra, from_r, &first_index,
	&last_index, ts_after);
    }
    else
    {
      tg_start_index = KheTaskIndexInFrameTimeGroup(first_task, days_frame);
      domain = KheTaskDomain(first_task);
      KheFindUnassignedTasksBefore(ej, max_extra, domain, tg_start_index,
	&first_index, &last_index, ts_before);
      KheFindUnassignedTasksAfter(ej, max_extra, domain, tg_start_index,
	&first_index, &last_index, ts_after);
    }
    ** K heTaskSetVisitEquivalent(ts_before); **
    ** K heTaskSetVisitEquivalent(ts_after); **
  }

  ** for each resource to_r in rg, not in not_rg, plus possibly unassignment **
  extra = KheEjectorCurrAugmentCount(ej) + 5 * KheSolnDiversifier(soln);
  ts2 = KheTaskSetMake(soln);
  ts3 = KheTaskSetMake(soln);
  if( DEBUG30 && KheEjectorCurrDebug(ej) )
  {
    fprintf(stderr, "%*sKheTaskSetMoveMultiRepair trying resource group:\n",
      KheEjectorCurrDebugIndent(ej), "");
    KheResourceGroupDebug(rg, 2, KheEjectorCurrDebugIndent(ej), stderr);
    if( not_rg != NULL )
    {
      fprintf(stderr, "%*somitting resource group:\n",
	KheEjectorCurrDebugIndent(ej), "");
      KheResourceGroupDebug(not_rg, 2, KheEjectorCurrDebugIndent(ej), stderr);
    }
  }
  KheForEachResource(srs_rec, rg, not_rg, allow_unassign, extra, to_r)
  {
    if( KheEjectorCurrDepth(ej) == 1 )
    {
      KheSolnNewG lobalVisit(soln);
      KheTaskSetAllVisit(ts);
    }
    if( DEBUG30 && KheEjectorCurrDebug(ej) )
      fprintf(stderr, "%*sKheTaskSetMoveMultiRepair trying resource %s\n",
	KheEjectorCurrDebugIndent(ej), "",
	to_r == NULL ? "-" : KheResourceId(to_r));
    if( to_r != from_r )
      switch( KheTaskSetRepairStatus(ts, to_r, days_frame, ts2,
	reversing_off, ** nocost_off, ** blocking_tg, blocking_d) )
      {
	case KHE_REPAIR_AVAIL_FREE:

	  ** find the max number of tasks from ts_before and ts_after **
	  max_before = KheInitialTasksOnFreeDays(ts_before, days_frame, to_r
	    ** , nocost_off **);
	  max_after  = KheInitialTasksOnFreeDays(ts_after,  days_frame, to_r
	    ** , nocost_off **);

	  ** for each possible number of tasks before **
	  for( before = 0;  before <= max_before;  before++ )
	  {
	    ** for each possible number of tasks after **
	    for(after=0; after<=max_after && before+after<=max_extra; after++)
	    {
	      KheTaskSetExpand(ts, ts_before, before, ts_after, after);
	      ** *** not sure about ts_before and ts_after here
	      if( !K heTaskSetVisited(ts, 0) )
	      {
		fprintf(stderr, "  task set ");
		KheTaskSetDebug(ts, 2, -1, stderr);
		fprintf(stderr, " unvisited\n");
		HnAbort("KheTaskSetMoveMultiRepair internal error 3");
	      }
	      *** **
	      if( KheDoTaskSetMo veMultiRepair(ej, ts, from_r, to_r,
		    balancing_off, ts2, ts3 ** , nocost_off **) )
	      {
		KheTaskSetClearFromEnd(ts, ts_count);
		KheTaskSetDelete(ts_before);
		KheTaskSetDelete(ts_after);
		KheTaskSetDelete(ts2);
		KheTaskSetDelete(ts3);
		return true;
	      }
	      KheTaskSetClearFromEnd(ts, ts_count);
	    }
	  }
	  break;

	case KHE_REPAIR_AVAIL_BUSY:

	  ** find the max number of tasks from ts_before and ts_after **
	  max_before = KheInitialCorrespondingTasks(ts_before,
	    days_frame, to_r);
	  max_after  = KheInitialCorrespondingTasks(ts_after,
	    days_frame, to_r);

	  ** for each possible number of tasks before **
	  for( before = 0;  before <= max_before;  before++ )
	  {
	    ** for each possible number of tasks after **
	    for(after=0; after<=max_after && before+after<=max_extra; after++)
	    {
	      KheTaskSetExpand(ts, ts_before, before, ts_after, after);
	      if( KheEjectorCurrDepth(ej) == 1 )
	      {
		KheSolnNewGl obalVisit(soln);
		KheTaskSetAllVisit(ts);
	      }
	      if( KheDoTaskSetSwapMultiRepair(ej, ts, from_r, to_r, ts2
		    ** , nocost_off **) )
	      {
		KheTaskSetClearFromEnd(ts, ts_count);
		KheTaskSetDelete(ts_before);
		KheTaskSetDelete(ts_after);
		KheTaskSetDelete(ts2);
		KheTaskSetDelete(ts3);
		return true;
	      }
	      KheTaskSetClearFromEnd(ts, ts_count);
	    }
	  }
	  break;

	case KHE_REPAIR_BLOCKED:

	  ** repair is blocked; do nothing **
	  break;

	default:

	  HnAbort("KheTaskSetMoveMultiRepair internal error 9");
	  break;
      }
  }
  ** ***
  if( KheEjectorCurrMayRevisit(ej) )
  {
    KheTaskSetUnVisitEquivalent(ts_before);
    KheTaskSetUnVisitEquivalent(ts_after);
  }
  *** **
  KheTaskSetDelete(ts_before);
  KheTaskSetDelete(ts_after);
  KheTaskSetDelete(ts2);
  KheTaskSetDelete(ts3);

  ** try meet plus task moves, if permitted and just one task **
  if( KheEjectorRepairTimes(ej) && ts_count == 1 )
  {
    extra = KheEjectorCurrAugmentCount(ej) + 5 * KheSolnDiversifier(soln);
    KheForEachResource(srs_rec, rg, not_rg, allow_unassign, extra, to_r)
      if( to_r != from_r )
      {
	if( KheMeetMoveAndTaskMoveMultiRepair(ej, KHE_MOVE_KEMPE, first_task,
	      to_r) )
	  return true;
      }
  }
  return false;
}
*** */


/* *** old version
static bool KheTaskSetMoveMultiRepair(KHE_EJECTOR ej, KHE_TASK_SET ts,
  KHE_RESOURCE_GROUP rg, KHE_RESOURCE_GROUP not_rg, bool allow_unassign
  ** , KHE_MOVE_REASON mr **)
{
  struct khe_se_resource_set_type srs_rec;  KHE_FRAME days_frame;
  KHE_RESOURCE from_r, to_r ** , r2 **;  KHE_TASK first_task ** , task **;
  ** bool with_swaps; **  KHE_SOLN soln;  ** KHE_TIME t; **
  int extra, first_index, last_index, ** frame_count, days, ** ts_count;
  ** int min_days, max_days, first_legal, last_legal, fi, li, num; **
  int max_extra, max_before, max_after, before, after;
  KHE_TASK_SET ts_before, ts_after;  ** KHE_RESOURCE_TIMETABLE_MONITOR rtm; **

  ** get days frame and exit if not present **
  soln = KheEjectorSoln(ej);
  days_frame = KheEjectorFrame(ej);
  if( days_frame == NULL )
    return false;
  ** frame_count = KheFrameTimeGroupCount(days_frame); **

  ** make sure ts is non-empty and get from_r from its first task **
  ts_count = KheTaskSetTaskCount(ts);
  HnAssert(ts_count > 0, "KheTaskSetMoveMultiRepair internal error 1");
  first_task = KheTaskSetTask(ts, 0);
  from_r = KheTaskAsstResource(first_task);

  ** get first_index and last_index, bracketing everything in ts **
  KheTaskSetInterval(ts, days_frame, &first_index, &last_index);

  ** get max_extra, the maximum number of tasks to include beyond ts **
  max_extra = max(4, ts_count) - ts_count;

  ** get ts_before and ts_after **
  ts_before = KheTaskSetMake(soln);
  ts_after = KheTaskSetMake(soln);
  if( from_r != NULL )
  {
    KheFindAssignedTasksBefore(ej, max_extra, from_r,
      &first_index, &last_index, ts_before);
    KheFindAssignedTasksAfter(ej, max_extra, from_r,
      &first_index, &last_index, ts_after);
  }
  else
  {
    KheFindUnassignedTasksBefore(ej, max_extra,KheTaskResourceType(first_task),
      &first_index, &last_index, ts_before);
    KheFindUnassignedTasksAfter(ej, max_extra, KheTaskResourceType(first_task),
      &first_index, &last_index, ts_after);
  }

  ** get min_days and max_days **
  ** with_swaps = !KheEjectorTaskSeqSwapsOff(ej); **
  ** ***
  min_days = last_index - first_index + 1;
  max_days = max(4, min_days);
  max_days = min(max_days, frame_count);
  *** **

  ** for each resource to_r in rg, not in not_rg, plus possibly unassignment **
  ** ts2 = KheTaskSetMake(soln); **
  extra = KheEjectorCurrAugmentCount(ej) + 5 * KheSolnDiversifier(soln);
  KheForEachResource(srs_rec, rg, not_rg, allow_unassign, extra, to_r)
    if( to_r != from_r )
    {
      if( KheTaskSetAllTasksAtFreeTimes(ts, days_frame, to_r) )
      {
	** find the max number of tasks from ts_before and ts_after **
	max_before=KheInitialTasksOnFreeDays(ts_before,days_frame,to_r);
	max_after =KheInitialTasksOnFreeDays(ts_after, days_frame,to_r);

	** for each possible number of tasks before **
	for( before = 0;  before <= max_before;  before++ )
	{
	  ** for each possible number of tasks after **
	  for(after=0; after <= max_after && before+after <= max_extra; after++)
	  {
	    KheTaskSetExpand(ts, ts_before, before, ts_after, after);
	    ** ***
	    HnAssert(KheTaskSetAllTasksAtFreeTimes(ts, days_frame, to_r),
	      "KheTaskSetMoveMultiRepair internal error 1");
	    KheTaskSetCheckProperRoots(ts);
	    *** **
	    if( KheDoTaskS etMoveMultiRepair(ej, ts, from_r, to_r) )
	    {
	      KheTaskSetClearFromEnd(ts, ts_count);
	      KheTaskSetDelete(ts_before);
	      KheTaskSetDelete(ts_after);
	      return true;
	    }
	    KheTaskSetClearFromEnd(ts, ts_count);
	  }
	}

	** ***
	** for each possible number of days **
	for( days = min_days;  days <= max_days;  days++ )
	{
	  first_legal = max(0, last_index - days + 1);
	  last_legal = min(first_index + days - 1, frame_count - 1);
	  HnAssert(first_legal <= first_index,
	    "KheTaskSetMoveMultiRepair internal error 2");
	  HnAssert(last_legal >= last_index,
	    "KheTaskSetMoveMultiRepair internal error 3");
	  ** ***
	  KheAdjustLegalToMoveReason(from_r, mr, first_index, last_index,
	    &first_legal, &last_legal);
	  *** **
	  ** for each legal sequence of "days" days **
	  for( fi = first_legal, li = fi+days-1;  li <= last_legal;  fi++, li++ )
	  {
	    if( from_r == NULL )
	    {
	      KheNoClashesCheck(soln, from_r);
	      if( KheDaySetAssignRepair(ej, fi, first_index - 1, ts,
		    last_index + 1, li, to_r) )
	      {
		KheNoClashesCheck(soln, to_r);
		return true;
	      }
	    }
	    else if( to_r == NULL )
	    {
	      KheNoClashesCheck(soln, from_r);
	      if( KheIntervalIsBusy(days_frame, fi, li, from_r) )
	      {
		if( KheDaySetUnassignRepair(ej, fi, first_index - 1, ts,
		      last_index + 1, li, from_r) )
		{
		  KheNoClashesCheck(soln, from_r);
		  return true;
		}
	      }
	    }
	    else
	    {
	      KheNoClashesCheck(soln, from_r);
	      KheNoClashesCheck(soln, to_r);
	      if( KheIntervalIsBusy(days_frame, fi, li, from_r) )
	      {
		if( KheDaySetSwapRepair(ej, fi, li, from_r, ts, to_r, ts2, &num) )
		{
		  KheNoClashesCheck(soln, from_r);
		  KheNoClashesCheck(soln, to_r);
		  return true;
		}
		if( KheDaySetDoubleSwapMultiRepair(ej, fi, li, from_r, ts, to_r,
		      ts2, num) )
		{
		  KheNoClashesCheck(soln, from_r);
		  KheNoClashesCheck(soln, to_r);
		  return true;
		}
	      }
	      ** ***
	      if( KheDaySetSwapRepair(ej, fi, ** first_index - 1, ** ts,
		    ** last_index + 1, ** li, from_r, to_r, &num) )
	      {
		KheNoClashesCheck(soln, from_r);
		KheNoClashesCheck(soln, to_r);
		return true;
	      }
	      if( KheDaySetDoubleSwapMultiRepair(ej, fi, ** first_index - 1, **
		    ts, ** last_index + 1, ** li, from_r, to_r, num) )
	      {
		KheNoClashesCheck(soln, from_r);
		KheNoClashesCheck(soln, to_r);
		return true;
	      }
	      *** **
	    }
	  }
	}
	*** **
      }
      ** *** not really the right place for this
      else if( KheTaskSetTaskCount(ts) == 1 && from_r != NULL && to_r != NULL )
      {
	t = KheMeetAsstTime(KheTaskMeet(first_task));
	rtm = KheResourceTimetableMonitor(soln, to_r);
	if( t != NULL && KheResourceBusyOnDayOfTime(rtm, days_frame, t, &task) )
	{
	  if( ** !Khe TaskEquivalent(first_task, task) && **
	      !K heTaskVisited(task, 0) && !KheTaskIsPreassigned(task, &r2) &&
	      KheResourceGroupContains(KheTaskDomain(task), from_r) &&
	      KheResourceGroupContains(KheTaskDomain(first_task), to_r) )
	  {
	    ** try first_task := to_r, task := from_r (expanded) **
	    K heTaskVisit(task);
	    if( KheTaskDoubleMoveMultiRepair(ej, first_task, task) )
	      return true;
	    if( KheEjectorCurrMayRevisit(ej) )
	      K heTaskUnVisit(task);
	  }
	}
      }
      *** **
    }
  KheTaskSetDelete(ts_before);
  KheTaskSetDelete(ts_after);

  ** try meet plus task moves, if permitted and just one task **
  if( KheEjectorRepairTimes(ej) && ts_count == 1 )
  {
    extra = KheEjectorCurrAugmentCount(ej) + 5 * KheSolnDiversifier(soln);
    KheForEachResource(srs_rec, rg, not_rg, allow_unassign, extra, to_r)
      if( to_r != from_r )
      {
	if( KheMeetMoveAndTaskMoveMultiRepair(ej, KHE_MOVE_KEMPE, first_task,
	      to_r) )
	  return true;
      }
  }
  return false;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheOldTaskSetMoveAugment(KHE_EJECTOR ej, KHE_TASK_SET ts,           */
/*    KHE_RESOURCE not_r, KHE_MOVE_TYPE mt)                                  */
/*                                                                           */
/*  Carry out augments which move the tasks of ts to the resources of the    */
/*  domain of ts's first task.  If not_r != NULL, these tasks will be        */
/*  currently assigned to not_r, so don't try moving there.                  */
/*                                                                           */
/*  Also try unassignment of the tasks of ts.                                */
/*                                                                           */
/*****************************************************************************/

/* ***
static bool KheOldTaskSetMoveAugment(KHE_EJECTOR ej, KHE_TASK_SET ts,
  KHE_RESOURCE_GROUP rg, KHE_RESOURCE_GROUP not_rg, bool allow_unassign)
{
  if( K heTaskSetTaskCount(ts) > 0 && !K heTaskSetVisited(ts, 0) )
  {
    K heTaskSetVisit(ts);
    if( KheTaskSetMoveMultiRepair(ej, ts, rg, not_rg, allow_unassign) )
      return true;
    if( KheEjectorCurrMayRevisit(ej) )
      K heTaskSetUnVisit(ts);
  }
  return false;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheFindAssignedTasks(KHE_EJECTOR ej, KHE_RESOURCE r,                */
/*    int first_index, int last_index, KHE_TASK_SET ts)                      */
/*                                                                           */
/*  Set ts to all proper root tasks that overlap [first_index, last_index]   */
/*  and are assigned r, and return true.  But if any of these tasks extend   */
/*  outside [first_index, last_index], or are preassigned, return false      */
/*  instead; in that case, ts is well-defined but its value is unspecified.  */
/*  Here [first_index, last_index] is an interval in days_frame as usual.    */
/*                                                                           */
/*  This function does not check for clashes.  But all the tasks in ts are   */
/*  assigned r and r initially has no clashes, so there can be none.         */
/*                                                                           */
/*****************************************************************************/

static bool KheFindAssignedTasks(KHE_EJECTOR ej, KHE_RESOURCE r,
  int first_index, int last_index, KHE_TASK_SET ts)
{
  KHE_RESOURCE_TIMETABLE_MONITOR rtm;  KHE_FRAME days_frame;  KHE_TASK task;
  KHE_TIME_GROUP tg;  KHE_RESOURCE r2;  int i, j, k, fi, li;  KHE_TIME t;
  days_frame = KheEjectorFrame(ej);
  rtm = KheResourceTimetableMonitor(KheEjectorSoln(ej), r);
  KheTaskSetClear(ts);
  for( i = first_index;  i <= last_index;  i++ )
  {
    tg = KheFrameTimeGroup(days_frame, i);
    for( j = 0;  j < KheTimeGroupTimeCount(tg);  j++ )
    {
      t = KheTimeGroupTime(tg, j);
      for( k = 0;  k < KheResourceTimetableMonitorTimeTaskCount(rtm, t);  k++ )
      {
	task = KheResourceTimetableMonitorTimeTask(rtm, t, k);
	task = KheTaskProperRoot(task);
	if( KheTaskIsPreassigned(task, &r2) )
	{
	  if( DEBUG34 )
	    fprintf(stderr, "  KheFindAssignedTasks(ej, %s, %s .. %s) %s\n",
	      KheResourceId(r),
	      KheTimeGroupId(KheFrameTimeGroup(days_frame, first_index)),
	      KheTimeGroupId(KheFrameTimeGroup(days_frame, last_index)),
	      "returning false (preassigned task)");
	  return false;
	}
	KheTaskInterval(task, days_frame, &fi, &li);
	if( !KheIntervalSubset(fi, li, first_index, last_index) )
        {
	  if( DEBUG34 )
	    fprintf(stderr, "  KheFindAssignedTasks(ej, %s, %s .. %s) %s\n",
	      KheResourceId(r),
	      KheTimeGroupId(KheFrameTimeGroup(days_frame, first_index)),
	      KheTimeGroupId(KheFrameTimeGroup(days_frame, last_index)),
	      "returning false (task not subset)");
	  return false;
	}
	if( fi == i )  /* this ensures that no task is added twice */
	  KheTaskSetAddTask(ts, task);
      }
    }
  }
  if( DEBUG34 )
  {
    fprintf(stderr, "  KheFindAssignedTasks(ej, %s, %s .. %s) returning true: ",
      KheResourceId(r),
      KheTimeGroupId(KheFrameTimeGroup(days_frame, first_index)),
      KheTimeGroupId(KheFrameTimeGroup(days_frame, last_index)));
    KheTaskSetDebug(ts, 2, 0, stderr);
  }
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskSetSwapToEndRepair(KHE_EJECTOR ej, KHE_TASK_SET ts,          */
/*    KHE_RESOURCE_GROUP rg, KHE_RESOURCE_GROUP not_rg)                      */
/*                                                                           */
/*  Repair function which swaps the timetables of r1 and r2, where r1        */
/*  is currently assigned ts and r2 is a resource of rg but not not_rg,      */
/*  from ts back to the start of the cycle, and from ts forward to the       */
/*  end of the cycle.                                                        */
/*                                                                           */
/*  Do nothing if ts is currently unassigned.                                */
/*                                                                           */
/*****************************************************************************/

static bool KheTaskSetSwapToEndRepair(KHE_EJECTOR ej, KHE_TASK_SET ts,
  KHE_RESOURCE_GROUP rg, KHE_RESOURCE_GROUP not_rg)
{
  struct khe_se_resource_set_type srs_rec;  KHE_SOLN soln;
  KHE_RESOURCE r1, r2;  int ts_first_index, ts_last_index, extra, count;
  KHE_TASK_SET ts1, ts2;  KHE_FRAME days_frame;

  /* make sure that ts is not empty; do nothing if it's unassigned */
  HnAssert(KheTaskSetTaskCount(ts) > 0,
    "KheTaskSetSwapToEndRepair internal error 1");
  r1 = KheTaskAsstResource(KheTaskSetTask(ts, 0));

  if( r1 != NULL )
  {
    /* boilerplate */
    if( DEBUG34 )
    {
      fprintf(stderr, "[ KheTaskSetSwapToEndRepair(ej, ");
      KheTaskSetDebug(ts, 2, -1, stderr);
      fprintf(stderr, " (assigned %s), ", KheResourceId(r1));
      KheResourceGroupDebug(rg, 1, -1, stderr);
      fprintf(stderr, ", ");
      if( not_rg != NULL )
	KheResourceGroupDebug(not_rg, 1, -1, stderr);
      else
	fprintf(stderr, "-");
      fprintf(stderr, ")\n");
    }
    days_frame = KheEjectorFrame(ej);
    count = KheFrameTimeGroupCount(days_frame);
    soln = KheEjectorSoln(ej);
    ts1 = KheTaskSetMake(soln);
    ts2 = KheTaskSetMake(soln);
    KheTaskSetInterval(ts, days_frame, &ts_first_index, &ts_last_index);

    /* get the tasks for r1 from the start to ts */
    if( KheFindAssignedTasks(ej, r1, 0, ts_last_index, ts1) )
    {
      /* try swapping r1 and r2's timetables up to and including ts */
      extra = KheEjectorCurrAugmentCount(ej) + 5 * KheSolnDiversifier(soln);
      KheForEachResource(srs_rec, rg, not_rg, false, extra, r2)
      {
	/* get the tasks for r2 from the start to ts, and try the move */
	if( r2 != r1 &&
	    KheFindAssignedTasks(ej, r2, 0, ts_last_index, ts2) &&
	    KheTaskSetDoubleMoveRepair(ej, r1, ts2, NULL, r2, ts1, NULL) )
	{
	  KheTaskSetDelete(ts1);
	  KheTaskSetDelete(ts2);
	  if( DEBUG34 )
	    fprintf(stderr, "] KheTaskSetSwapToEndRepair returning true\n");
	  return true;
	}
      }
    }

    /* get the tasks for r1 from ts to the end */
    if( KheFindAssignedTasks(ej, r1, ts_first_index, count - 1, ts1) )
    {
      /* try swapping r1 and r2's timetables up to and including ts */
      extra = KheEjectorCurrAugmentCount(ej) + 5 * KheSolnDiversifier(soln);
      KheForEachResource(srs_rec, rg, not_rg, false, extra, r2)
      {
	/* get the tasks for r2 from ts to the end, and try the move */
	if( r2 != r1 &&
	    KheFindAssignedTasks(ej, r2, ts_first_index, count - 1, ts2) &&
	    KheTaskSetDoubleMoveRepair(ej, r1, ts2, NULL, r2, ts1, NULL) )
	{
	  KheTaskSetDelete(ts1);
	  KheTaskSetDelete(ts2);
	  if( DEBUG34 )
	    fprintf(stderr, "] KheTaskSetSwapToEndRepair returning true\n");
	  return true;
	}
      }
    }

    /* get the tasks for r1 from the start to the end */
    /* *** tried it but slow and results were not remarkable
    if( KheFindAssignedTasks(ej, r1, 0, count - 1, ts1) )
    {
      ** try swapping r1 and r2's timetables up to and including ts **
      extra = KheEjectorCurrAugmentCount(ej) + 5 * KheSolnDiversifier(soln);
      KheForEachResource(srs_rec, rg, not_rg, false, extra, r2)
      {
	** get the tasks for r2 from the start to the end, and try the move **
	if( KheFindAssignedTasks(ej, r2, 0, count - 1, ts2) &&
	    KheTaskSetDoubleMoveRepair(ej, r1, ts2, NULL, r2, ts1, NULL) )
	{
	  KheTaskSetDelete(ts1);
	  KheTaskSetDelete(ts2);
	  return true;
	}
      }
    }
    *** */

    /* no luck; tidy up and exit */
    KheTaskSetDelete(ts1);
    KheTaskSetDelete(ts2);
  }
  if( DEBUG34 )
    fprintf(stderr, "] KheTaskSetSwapToEndRepair returning false\n");
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskSetMoveAugment(KHE_EJECTOR ej, KHE_TASK_SET ts,              */
/*    KHE_RESOURCE_GROUP rg, KHE_RESOURCE_GROUP not_rg, bool allow_unassign, */
/*    KHE_EXPANSION_OPTIONS eo, KHE_TIME_GROUP blocking_tg,                  */
/*    KHE_MONITOR blocking_d)                                                */
/*                                                                           */
/*  Augment function which moves the tasks of ts from the resource they      */
/*  are all assigned to now to a different resource lying in rg, not         */
/*  lying in not_rg (if non-NULL).  The new resource may be NULL if          */
/*  allow_unassign is true.                                                  */
/*                                                                           */
/*****************************************************************************/

static bool KheTaskSetMoveAugment(KHE_EJECTOR ej, KHE_TASK_SET ts,
  KHE_RESOURCE_GROUP rg, KHE_RESOURCE_GROUP not_rg, bool allow_unassign,
  KHE_EXPANSION_OPTIONS eo, KHE_TIME_GROUP blocking_tg,
  KHE_MONITOR blocking_d)
{
  if( KheTaskSetTaskCount(ts) > 0 && !KheTaskSetAnyVisited(ts, 0) )
  {
    KheTaskSetVisitEquivalent(ts);
    if( KheTaskSetMoveMultiRepair(ej, ts, rg, not_rg, allow_unassign,
	  eo, blocking_tg, blocking_d) )
      return true;
    if( /* KheEjectorCurrDepth(ej) == 1 && */ eo->full_widening_on &&
	KheTaskSetSwapToEndRepair(ej, ts, rg, not_rg) )
      return true;
    if( KheEjectorCurrMayRevisit(ej) )
      KheTaskSetUnVisitEquivalent(ts);
  }
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskMoveAugment(KHE_EJECTOR ej, KHE_TASK task,                   */
/*    KHE_RESOURCE_GROUP rg, KHE_RESOURCE_GROUP not_rg, bool allow_unassign, */
/*    KHE_EXPANSION_OPTIONS eo, KHE_TIME_GROUP blocking_tg,                  */
/*    KHE_MONITOR blocking_d, KHE_TASK_SET scratch_ts)                       */
/*                                                                           */
/*  Like KheTaskSetMoveAugment but for a single task.                        */
/*                                                                           */
/*****************************************************************************/

static bool KheTaskMoveAugment(KHE_EJECTOR ej, KHE_TASK task,
  KHE_RESOURCE_GROUP rg, KHE_RESOURCE_GROUP not_rg, bool allow_unassign,
  KHE_EXPANSION_OPTIONS eo, KHE_TIME_GROUP blocking_tg,
  KHE_MONITOR blocking_d, KHE_TASK_SET scratch_ts)
{
  HnAssert(task == KheTaskProperRoot(task),
    "KheTaskMoveAugment internal error");
  if( !KheTaskVisited(task, 0) )
  {
    KheTaskVisitEquivalent(task);
    KheTaskSetClear(scratch_ts);
    KheTaskSetAddTask(scratch_ts, task);
    HnAssert(KheTaskSetAllVisited(scratch_ts, 0),
      "KheTaskMoveAugment internal error");
    if( KheTaskSetMoveMultiRepair(ej, scratch_ts, rg, not_rg, allow_unassign,
	  eo, blocking_tg, blocking_d) )
      return true;
    if( /* KheEjectorCurrDepth(ej) == 1 && */ eo->full_widening_on &&
	KheTaskSetSwapToEndRepair(ej, scratch_ts, rg, not_rg) )
      return true;
    if( KheEjectorCurrMayRevisit(ej) )
      KheTaskUnVisitEquivalent(task);
  }
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "resource frame move and swap repairs and augments"            */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_ACTIVE_AUGMENT_TYPE                                                  */
/*                                                                           */
/*  Interpreted value of the active_augment option.                          */
/*                                                                           */
/*****************************************************************************/

/* *** no longer used
typedef enum {
  KHE_ACTIVE_AUGMENT_MOVE,
  KHE_ACTIVE_AUGMENT_SWAP
} KHE_ACTIVE_AUGMENT_TYPE;
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheFrameMove(KHE_FRAME frame, KHE_RESOURCE from_r,                  */
/*    KHE_RESOURCE to_r, KHE_MOVE_TYPE mt, KHE_FRAME days_frame)             */
/*                                                                           */
/*  Move the tasks currently assigned to from_r in frame to to_r.            */
/*  Alternatively, if to_r is NULL, just unassign the tasks.                 */
/*  But don't unassign or move any preassigned tasks.                        */
/*                                                                           */
/*****************************************************************************/

/* *** no longer used
static bool KheFrameMove(KHE_FRAME frame, KHE_RESOURCE from_r,
  KHE_RESOURCE to_r, KHE_FRAME days_frame)
{
  bool res;  KHE_TASK_SET ts;

  if( DEBUG21 )
    fprintf(stderr, "[ KheFrameMove(frame, %s, %s, days_frame)\n",
      from_r != NULL ? KheResourceId(from_r) : "-",
      to_r != NULL ? KheResourceId(to_r) : "-");
  HnAssert(from_r != to_r, "KheFrameMove internal error 1");
  res = KheFrame TaskSet(frame, from_r, &ts) &&
    KheTypedTaskSetMoveFrame(ts, to_r, KHE_MOVE_EJECTING, days_frame);
  KheTaskSetDelete(ts);
  if( DEBUG21 )
    fprintf(stderr, "] KheFrameMove returning %s\n", res ? "true" : "false");
  return res;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_POLARITY KheFramePolarity(KHE_FRAME frame)                           */
/*                                                                           */
/*  Return the polarity of frame, assuming that it is non-empty and that     */
/*  all of its time groups have the same polarity.                           */
/*                                                                           */
/*****************************************************************************/

/* ***
static KHE_POLARITY KheFramePolarity(KHE_FRAME frame)
{
  KHE_POLARITY res;
  KheFrameTimeGroup(frame, 0, &res);
  return res;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheFrameMoveRepair(KHE_EJECTOR ej, KHE_FRAME frame,                 */
/*    KHE_RESOURCE from_r, KHE_RESOURCE to_r, KHE_MOVE_TYPE mt)              */
/*                                                                           */
/*  Repair operation that moves from_r's tasks in frame to to_r, or vice     */
/*  versa if frame is negative.                                              */
/*                                                                           */
/*  Alternatively, if to_r is NULL, this just unassigns the tasks.           */
/*                                                                           */
/*****************************************************************************/

/* ***
static bool KheFrameMoveRepair(KHE_EJECTOR ej, KHE_FRAME frame,
  KHE_RESOURCE from_r, KHE_RESOURCE to_r, KHE_MOVE_TYPE mt)
{
  KHE_FRAME days_frame;  bool ok;  KHE_RESOURCE tmp_r;
  KheEjectorRe pairBegin(ej);
  if( DEBUG20 )
  {
    fprintf(stderr, "%*sFrameMoveRepair(", KheEjectorCurrDebugIndent(ej), "");
    KheFrameDebug(frame, 1, -1, stderr);
    fprintf(stderr, ", %s, %s, %s)\n", KheResourceId(from_r),
      to_r == NULL ? "-" : KheResourceId(to_r), KheMoveTypeShow(mt));
  }
  if( KheFramePolarity(frame) == KHE_NEGATIVE )
    tmp_r = from_r, from_r = to_r, to_r = tmp_r;
  days_frame = KheEjectorFrame(ej);
  ok = KheFrameMove(frame, from_r, to_r, mt, days_frame);
  if( DEBUG18 || KheEjectorCurrDebug(ej) )
  {
    fprintf(stderr, "%cFrameMoveRepair(", ok ? '+' : '-');
    KheFrameDebug(frame, 1, -1, stderr);
    fprintf(stderr, ", %s, %s, %s)", KheResourceId(from_r),
      to_r == NULL ? "-" : KheResourceId(to_r), KheMoveTypeShow(mt));
  }
  return KheEjectorRepa irEnd(ej, KheMoveTypeToTaskSetRepairType(mt), ok);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheFrameSwapRepair(KHE_EJECTOR ej, KHE_FRAME frame1,                */
/*    KHE_FRAME frame2, KHE_RESOURCE from_r, KHE_RESOURCE to_r,              */
/*    KHE_MOVE_TYPE mt)                                                      */
/*                                                                           */
/*  Repair operation that moves from_r's tasks in frame1 to to_r, and        */
/*  to_r's tasks in frame2 to from_r, or vice versa if frame1 is negative.   */
/*                                                                           */
/*****************************************************************************/

/* ***
static bool KheFrameSwapRepair(KHE_EJECTOR ej, KHE_FRAME frame1,
  KHE_FRAME frame2, KHE_RESOURCE from_r, KHE_RESOURCE to_r,
  KHE_MOVE_TYPE mt)
{
  KHE_FRAME days_frame;  bool ok;  KHE_RESOURCE tmp_r;
  KheEjectorRep airBegin(ej);
  if( KheFramePolarity(frame1) == KHE_NEGATIVE )
    tmp_r = from_r, from_r = to_r, to_r = tmp_r;
  days_frame = KheEjectorFrame(ej);
  ok = KheFrameMove(frame1, from_r, to_r, mt, days_frame) &&
    KheFrameMove(frame2, to_r, from_r, mt, days_frame);
  if( DEBUG18 || KheEjectorCurrDebug(ej) )
  {
    fprintf(stderr, "%cFrameSwapRepair(%s", ok ? '+' : '-',
      KheResourceId(from_r));
    KheFrameDebug(frame1, 1, -1, stderr);
    fprintf(stderr, " <--> %s", KheResourceId(to_r));
    KheFrameDebug(frame2, 1, -1, stderr);
    fprintf(stderr, ", %s)", KheMoveTypeShow(mt));
  }
  return KheEjectorRep airEnd(ej, KHE_REPAIR_EJECTING_TASK_SET_SWAP, ok);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheFrameSwapAugment(KHE_EJECTOR ej, KHE_FRAME frame1,               */
/*    KHE_FRAME frame2, KHE_RESOURCE from_r, KHE_RESOURCE to_r,              */
/*    KHE_MOVE_TYPE mt)                                                      */
/*                                                                           */
/*  Augment operation that visits frame2 and calls a frame swap repair.      */
/*                                                                           */
/*****************************************************************************/

/* ***
static bool KheFrameSwapAugment(KHE_EJECTOR ej, KHE_FRAME frame1,
  KHE_FRAME frame2, KHE_RESOURCE from_r, KHE_RESOURCE to_r, KHE_MOVE_TYPE mt)
{
  if( !KheFrameVis ited(frame2, from_r, 0) )
  {
    KheFrameVisit(frame2, from_r);
    if( KheFrameSwapRepair(ej, frame1, frame2, from_r, to_r, mt) )
      return true;
    if( KheEjectorCurrMayRevisit(ej) )
      KheFrameVisit(frame2, from_r);
  }
  return false;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheFrameMoveOrSwapRepair(KHE_EJECTOR ej, KHE_FRAME frame,           */
/*    KHE_RESOURCE r1, KHE_RESOURCE r2, bool allow _eject,                   */
/*    KHE_ACTIVE_AUGMENT_TYPE aa)                                            */
/*                                                                           */
/*  Repair so that frame, which is active for r1 and inactive for r2,        */
/*  becomes active for r2 and inactive for r1.                               */
/*                                                                           */
/*  If aa is KHE_ACTIVE_AUGMENT_MOVE, that is all that is done.  But if it   */
/*  is KHE_ACTIVE_AUGMENT_SWAP, this function also looks for ways to move    */
/*  an active frame of r2's back to an inactive frame of r's.                */
/*                                                                           */
/*****************************************************************************/

/* ***
static bool KheFrameMoveOrSwapRepair(KHE_EJECTOR ej, KHE_FRAME frame,
  KHE_RESOURCE r1, KHE_RESOURCE r2, KHE_FRAME orig_frame, KHE_MOVE_TYPE mt,
  KHE_ACTIVE_AUGMENT_TYPE aa)
{
  int extra, len;  struct khe_frame_iterator_rec fi_rec;
  KHE_FRAME active_frame, frame2;
  if( aa == KHE_ACTIVE_AUGMENT_MOVE )
  {
    return KheFrameMoveRepair(ej, frame, r1, r2, mt);
  }
  else
  {
    ** try the move repair first **
    if( KheFrameMoveRepair(ej, frame, r1, r2, mt) )
      return true;

    ** iterate over the frame, finding active intervals for r2 **
    len = KheFrameTimeGroupCount(frame);
    extra =  KheEjectorCurrAugmentCount(ej) +
      5 * KheSolnDiversifier(KheEjectorSoln(ej));
    KheFrameI teratorInit(&fi_rec, orig_frame, r2, extra);
    while( KheFrameIteratorNext(&fi_rec, &active_frame) )
    {
      if( KheFrameTimeGroupCount(active_frame) >= len )
      {
	frame2 = KheFrameInsideBeforeSlice(active_frame, len);
	if( KheFrameIsInac tive(frame2, r1) &&
	    ** KheFrameAdjacentToActive(frame2, r1) && **
	    KheFrameSwapAugment(ej, frame, frame2, r1, r2, mt) )
	  return true;

	if( KheFrameTimeGroupCount(active_frame) > len )
	{
	  frame2 = KheFrameInsideAfterSlice(active_frame, len);
	  if( KheFrameIsIna ctive(frame2, r1) &&
	      ** KheFrameAdjacentToActive(frame2, r1) && **
	      KheFrameSwapAugment(ej, frame, frame2, r1, r2, mt) )
	    return true;
	}
      }
    }
    return false;
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  Submodule "combined meet and task move and swap repairs and augments"    */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool KheMeetMoveAndTaskMoveRepair(KHE_EJECTOR ej, KHE_MEET meet,         */
/*    KHE_MEET target_meet, int offset, KHE_MOVE_TYPE meet_mt,               */
/*    KHE_TASK task, KHE_RESOURCE r, KHE_MOVE_TYPE task_mt)                  */
/*                                                                           */
/*  Carry out a combined meet move and task move augment.                    */
/*                                                                           */
/*****************************************************************************/

static bool KheMeetMoveAndTaskMoveRepair(KHE_EJECTOR ej, KHE_MEET meet,
  KHE_MEET target_meet, int offset, KHE_MOVE_TYPE meet_mt,
  KHE_TASK task, KHE_RESOURCE r, KHE_MOVE_TYPE task_mt)
{
  bool success, basic, move;  int d;  KHE_RESOURCE r2;  KHE_FRAME frame;
  if( KheTaskIsPreassigned(task, &r2) )
    return false;
  HnAssert(KheMeetAsst(meet) != NULL,
    "KheMeetMoveAndTaskMoveRepair internal error 1");
  KheEjectorRepairBegin(ej);
  move = (KheTaskAsst(task) != NULL);
  frame = KheEjectorFrame(ej);
  success = (move ? KheTaskUnAssign(task) : true) &&
    KheTypedMeetMove(meet, target_meet, offset, meet_mt, false, &d,&basic,NULL)
    && KheTypedTaskMoveFrame(task, r, task_mt, frame);
  if( DEBUG5 || KheEjectorCurrDebug(ej) )
  {
    fprintf(stderr, "%cMeetMoveAndTaskMove(",
      success ? '+' : '-');
    KheMeetDebug(meet, 1, -1, stderr);
    fprintf(stderr, ", ");
    KheMeetDebug(target_meet, 1, -1, stderr);
    fprintf(stderr, ", %d, meet_mt %s", offset, KheMoveTypeShow(meet_mt));
    KheTaskDebug(task, 1, -1, stderr);
    fprintf(stderr, ", %s, task_mt %s)", KheResourceId(r),
      KheMoveTypeShow(task_mt));
  }
  return KheEjectorRepairEnd(ej,
    move ? KHE_REPAIR_KEMPE_MEET_MOVE_AND_EJECTING_TASK_MOVE :
    KHE_REPAIR_KEMPE_MEET_MOVE_AND_EJECTING_TASK_ASSIGN, success);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheMeetMoveAndTaskMoveMultiRepair(KHE_EJECTOR ej,                   */
/*    KHE_MOVE_TYPE meet_mt, KHE_TASK task, KHE_RESOURCE r,                  */
/*    KHE_MOVE_TYPE task_mt)                                                 */
/*                                                                           */
/*  Try a move of type meet_mt of task's meet to somewhere else, and at the  */
/*  same time a move of type task_mt of task to r.                           */
/*                                                                           */
/*****************************************************************************/

static bool KheMeetMoveAndTaskMoveMultiRepair(KHE_EJECTOR ej,
  KHE_MOVE_TYPE meet_mt, KHE_TASK task, KHE_RESOURCE r)
{
  int max_offset, offset, base, i, junk, index, extra;
  KHE_MEET anc_meet, target_meet;  KHE_NODE node, parent_node;
  if( DEBUG5 )
  {
    fprintf(stderr, "%*s[ KheMeetMoveAndTaskMoveMultiRepair(",
      KheEjectorCurrDebugIndent(ej) + 2, "");
    KheTaskDebug(task, 1, -1, stderr);
    fprintf(stderr, ", %s)\n", KheResourceId(r));
  }
  anc_meet = KheMeetFirstMovable(KheTaskMeet(task), &junk);
  base = 0;
  while( anc_meet != NULL )
  {
    node = KheMeetNode(anc_meet);
    if( node != NULL && KheNodeParent(node) != NULL && !KheNodeIsVizier(node) )
    {
      parent_node = KheNodeParent(node);
      extra = KheEjectorCurrAugmentCount(ej);
      for( i = 0;  i < KheNodeMeetCount(parent_node);  i++ )
      {
	index = (extra + i ) % KheNodeMeetCount(parent_node);
	target_meet = KheNodeMeet(parent_node, index);
	max_offset = KheMeetDuration(target_meet)-KheMeetDuration(anc_meet);
	for( offset = 0;  offset <= max_offset;  offset++ )
	{
	  if( KheMeetMoveAndTaskMoveRepair(ej, anc_meet, target_meet,
		offset, meet_mt, task, r, KHE_MOVE_EJECTING) )
	  {
	    if( DEBUG5 )
	      fprintf(stderr, "%*s] KheMeetMoveAndTaskMoveMultiRepair ret true\n",
		KheEjectorCurrDebugIndent(ej) + 2, "");
	    return true;
	  }
	}
      }
    }
    base += KheMeetAsstOffset(anc_meet);
    anc_meet = KheMeetAsst(anc_meet);
  }
  if( DEBUG5 )
    fprintf(stderr, "%*s] KheMeetMoveAndTaskMoveMultiRepair ret false\n",
      KheEjectorCurrDebugIndent(ej) + 2, "");
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "resource overload repair and augment"                         */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool KheOneTask(KHE_RESOURCE_TIMETABLE_MONITOR rtm, KHE_TIME_GROUP tg)   */
/*                                                                           */
/*  Return true if rtm contains exactly one task overlapping tg.             */
/*                                                                           */
/*****************************************************************************/

/* ***
static bool KheOneTask(KHE_RESOURCE_TIMETABLE_MONITOR rtm, KHE_TIME_GROUP tg)
{
  int i, j, count;  KHE_TIME t;  KHE_TASK first_task, task;
  first_task = NULL;
  for( i = 0;  i < KheTimeGroupTimeCount(tg);  i++ )
  {
    t = KheTimeGroupTime(tg, i);
    count = KheResourceTimetableMonitorTimeTaskCount(rtm, t);
    for( j = 0;  j < count;  j++ )
    {
      task = KheResourceTimetableMonitorTimeTask(rtm, t, j);
      if( first_task == NULL )
	first_task = task;
      else if( task != first_task )
	return false;
    }
  }
  return first_task != NULL;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDecreaseOverlapMeetMoveFn(KHE_MEET meet,                         */
/*    KHE_TIME time, void *impl)                                             */
/*                                                                           */
/*  Return true when moving meet to time would decrease the overlap with     */
/*  a time group, passed in impl.  This function assumes that meet is        */
/*  currently assigned a time.                                               */
/*                                                                           */
/*****************************************************************************/

static bool KheDecreaseOverlapMeetMoveFn(KHE_MEET meet,
  KHE_TIME time, void *impl)
{
  KHE_TIME_GROUP tg;  int curr_overlap, new_overlap, durn;
  tg = (KHE_TIME_GROUP) impl;
  durn = KheMeetDuration(meet);
  curr_overlap = KheTimeGroupOverlap(tg, KheMeetAsstTime(meet), durn);
  new_overlap = KheTimeGroupOverlap(tg, time, durn);
  return new_overlap < curr_overlap;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheResourceOverloadAugment(KHE_EJECTOR ej, KHE_RESOURCE r,          */
/*    KHE_TIME_GROUP tg, bool require_zero)                                  */
/*                                                                           */
/*  Carry out augments which try to decrease the number of times that r is   */
/*  busy within time group tg.  If require_zero is true, only try repairs    */
/*  that will reduce this number to zero.                                    */
/*                                                                           */
/*****************************************************************************/

static bool KheResourceOverloadAugment(KHE_EJECTOR ej, KHE_RESOURCE r,
  KHE_TIME_GROUP tg, bool require_zero)
{
  KHE_MEET meet;  KHE_TASK task;  KHE_RESOURCE_GROUP r_rg, domain;
  bool not_ejecting;  int i, tg_time_count, all_time_count, extra, ts_count;
  KHE_SOLN soln;  KHE_TASK_SET ts, scratch_ts;
  KHE_RESOURCE_TIMETABLE_MONITOR rtm;
  /* bool wid ening_off, reversing_off, balancing_off , nocost_off */
  struct khe_expansion_options_rec eor;

  /* find the set of tasks running during tg that are assigned r */
  soln = KheEjectorSoln(ej);
  r_rg = KheResourceSingletonResourceGroup(r);
  rtm = KheResourceTimetableMonitor(soln, r);
  ts = KheTaskSetMake(soln);
  KheResourceTimetableMonitorAddProperRootTasks(rtm, tg, false, ts);
  ts_count = KheTaskSetTaskCount(ts);
  if( DEBUG29 && KheEjectorCurrDebug(ej) )
    fprintf(stderr, "%*s[ KheResourceOverloadAugment(ej, %s, %s%s, %s)\n",
      KheEjectorCurrDebugIndent(ej), "", KheResourceId(r),
      KheTimeGroupId(tg), KheTaskSetAllVisited(ts, 0) ? " (visited)" : "",
      require_zero ? "true" : "false");
  if( ts_count == 0 )
  {
    KheTaskSetDelete(ts);
    if( DEBUG29 && KheEjectorCurrDebug(ej) )
      fprintf(stderr, "%*s] KheResourceOverloadAugment returning false (ts)\n",
	KheEjectorCurrDebugIndent(ej), "");
    return false;
  }
  /* HnAssert(ts_count > 0, "KheResourceOverloadAugment internal error 1"); */

  /* get expansion options */
  KheEjectorExpansionOptions(ej, &eor);

  /* different repairs depending on whether require_zero is true */
  /* nocost_off = KheEjectorNoCostOff(ej); */
  if( require_zero )
  {
    if( KheEjectorRepairResources(ej) &&
      !KheResourceTypeDemandIsAllPreassigned(KheResourceResourceType(r)) )
    {
      /* require_zero, so try to move the whole task set */
      domain = KheTaskDomain(KheTaskSetTask(ts, 0));
      if( KheTaskSetMoveAugment(ej, ts, domain, r_rg, true, &eor, tg, NULL) )
      {
	if( DEBUG29 && KheEjectorCurrDebug(ej) )
	  fprintf(stderr,"%*s] KheResourceOverloadAugment returning true (d)\n",
	    KheEjectorCurrDebugIndent(ej), "");
	KheTaskSetDelete(ts);
	return true;
      }
    }
  }
  else
  {
    /* try ejecting task moves of individual tasks in ts away from r */
    if( KheEjectorRepairResources(ej) &&
	!KheResourceTypeDemandIsAllPreassigned(KheResourceResourceType(r)) )
    {
      extra = KheEjectorCurrAugmentCount(ej) + 5 * KheSolnDiversifier(soln);
      scratch_ts = KheTaskSetMake(soln);
      for( i = 0;  i < ts_count;  i++ )
      {
	task = KheTaskSetTask(ts, (i + extra) % ts_count);
	domain = KheTaskDomain(task);
	if( KheTaskMoveAugment(ej, task, domain, r_rg, true, &eor, tg,
	      NULL, scratch_ts) )
	{
	  if( DEBUG24 )
	    fprintf(stderr, "%*s] KheResourceOverloadAugment ret. true (a)\n",
	      KheEjectorCurrDebugIndent(ej), "");
	  KheTaskSetDelete(scratch_ts);
	  KheTaskSetDelete(ts);
	  return true;
	}
      }
      KheTaskSetDelete(scratch_ts);
    }

    /* try ejecting meet moves of meets in tg away from tg */
    if( KheEjectorRepairTimes(ej) )
    {
      tg_time_count = KheTimeGroupTimeCount(tg);
      all_time_count = KheInstanceTimeCount(KheSolnInstance(soln));
      if( tg_time_count < all_time_count )  /* saves time with workload c's */
      {
	not_ejecting = KheEjectorBasicNotEjecting(ej);
	extra = KheEjectorCurrAugmentCount(ej) + 5 * KheSolnDiversifier(soln);
	for( i = 0;  i < ts_count;  i++ )
	{
	  task = KheTaskSetTask(ts, (i + extra) % ts_count);
	  meet = KheTaskMeet(task);
	  if( KheMeetAugment(ej, meet, true, &KheDecreaseOverlapMeetMoveFn,
	    (void *) tg, KHE_OPTIONS_KEMPE_FALSE, !not_ejecting, not_ejecting,
	    false) )
	  {
	    if( DEBUG29 && KheEjectorCurrDebug(ej) )
	      fprintf(stderr, "%*s] KheResourceOverloadAugment ret. true (c)\n",
		KheEjectorCurrDebugIndent(ej), "");
	    KheTaskSetDelete(ts);
	    return true;
	  }
	}
      }
    }
  }
  if( DEBUG29 && KheEjectorCurrDebug(ej) )
    fprintf(stderr, "%*s] KheResourceOverloadAugment returning false\n",
      KheEjectorCurrDebugIndent(ej), "");
  KheTaskSetDelete(ts);
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheNurseOverloadAugment(KHE_EJECTOR ej, KHE_RESOURCE r,             */
/*    KHE_TIME_GROUP tg, KHE_MONITOR d)                                      */
/*                                                                           */
/*  Try repairs that reduce the workload of time group tg for r.             */
/*  Structurally this parallels KheOverLoadAugment (require_zero = false).   */
/*  However it moves sequences of adjacent assignments, rather than          */
/*  individual assignments.                                                  */
/*                                                                           */
/*****************************************************************************/

/* *** obsolete
static bool KheNurseOverloadAugment(KHE_EJECTOR ej, KHE_RESOURCE r,
  KHE_TIME_GROUP tg, KHE_MONITOR d)
{
  struct khe_se_resource_set_type srs_rec;
  struct khe_frame_iterator_rec fi_rec;  KHE_FRAME frame, active_frame, frame2;
  int extra, len;  KHE_RESOURCE_GROUP rg, not_rg;  KHE_RESOURCE r2;
  if( DEBUG19 && KheEjectorCurrDebug(ej) )
    fprintf(stderr, "%*s[ KheNurseOverloadAugment(ej, %s, %s, %s)\n",
      KheEjectorCurrDebugIndent(ej), "", KheResourceId(r), KheTimeGroupId(tg),
      KheMonitorId(d));

  ** try moving one active frame **
  frame = KheEjectorFrame(ej);
  extra =  KheEjectorCurrAugmentCount(ej) +
    5 * KheSolnDiversifier(KheEjectorSoln(ej));
  rg = KheResourceTypeFullResourceGroup(KheResourceResourceType(r));
  not_rg = KheResource SingletonResourceGroup(r);
  KheFrameIt eratorInit(&fi_rec, frame, r, extra);
  while( KheFrameIteratorNext(&fi_rec, &active_frame) )
  {
    if( DEBUG19 && KheEjectorCurrDebug(ej) )
    {
      fprintf(stderr, "%*s  frame (%s) ", KheEjectorCurrDebugIndent(ej), "",
        KheFrameVi sited(active_frame, r, 0) ? "visited" : "unvisited");
      KheFrameDebug(active_frame, 1, 0, stderr);
    }
    if( !KheFrameVi sited(active_frame, r, 0) &&
	KheFrameIntersectsTimeGroup(active_frame, tg) )
    {
      KheFrameVisit(active_frame, r);
      KheForEachResource(srs_rec, rg, not_rg, true, extra, r2)
      {
	for( len = KheFrameTimeGroupCount(active_frame);  len >= 1;  len-- )
	{
	  frame2 = KheFrameInsideBeforeSlice(active_frame, len);
	  if( KheFrameMoveRepair(ej, frame2, r, r2, KHE_MOVE_KEMPE) )
	  {
	    if( DEBUG19 && KheEjectorCurrDebug(ej) )
	      fprintf(stderr, "%*s] KheNurseOverloadAugment returning true\n",
		KheEjectorCurrDebugIndent(ej), "");
	    return true;
	  }
	  if( len != KheFrameTimeGroupCount(active_frame) )
	  {
	    frame2 = KheFrameInsideAfterSlice(active_frame, len);
	    if( KheFrameMoveRepair(ej, frame2, r, r2, KHE_MOVE_KEMPE) )
	    {
	      if( DEBUG19 && KheEjectorCurrDebug(ej) )
		fprintf(stderr, "%*s] KheNurseOverloadAugment returning true\n",
		  KheEjectorCurrDebugIndent(ej), "");
	      return true;
	    }
	  }
	}
      }
      if( KheEjectorCurrMayRevisit(ej) )
	KheFrameVisit(active_frame, r);
    }
  }

  ** as a last resort, try rematching pairs of active frames **
  ** NB all these frames have already been visited; and anyway **
  ** this code cannot recurse, because rematching either reduces **
  ** cost or changes nothing **

  ** turned off, did not help with LH05 **
  if( false && KheEjectorCurrDepth(ej) == 1 )
  {
    KHE_RESOURCE_MATCHING_SOLVER rms;  bool success;  KHE_SOLN soln;
    KHE_RESOURCE_GROUP rg;  HA_ARENA a;
    ** ***
    if( DEBUG19 || KheEjectorCurrDebug(ej) )
      fprintf(stderr, "ejector rematching %s\n", KheResourceId(r));
    *** **
    soln = KheEjectorSoln(ej);
    a = KheSolnArenaBegin(soln);
    rg = KheResourceTypeFullResourceGroup(KheResourceResourceType(r));
    rms = KheResourceMatchin gSolverMake(soln, rg, a);
    extra =  KheEjectorCurrAugmentCount(ej) +
      5 * KheSolnDiversifier(KheEjectorSoln(ej));
    KheFrameIter atorInit(&fi_rec, frame, r, extra);
    while( KheFrameIteratorNextPair(&fi_rec, &active_frame) )
    {
      KHE_RESOURCE_MATCHING_DEMAND_SET rmds;
      KheResourceMatchingDemandSetMake(rms, false);
      KheResourceMatchingDemandSetAddFrame(rmds, active_frame);
      KheEjecto rRepairBegin(ej);
      if( DEBUG19 || KheEjectorCurrDebug(ej) )
      {
	fprintf(stderr, "rematching %s over ", KheResourceId(r));
	KheFrameDebug(active_frame, 1, 0, stderr);
      }
      success = KheResource MatchingSolverSolve(rms, rmds, ** true, ** false,
        KheEjectorOptions(ej));
      if( KheEjector RepairEnd(ej, 0, success) )
      {
	** KheResourceMatchingSolverDelete(rms); **
	KheSolnArenaEnd(soln, a);
	if( DEBUG19 && KheEjectorCurrDebug(ej) )
	  fprintf(stderr, "%*s] KheNurseOverloadAugment returning true (rematch)\n",
	    KheEjectorCurrDebugIndent(ej), "");
	return true;
      }
      KheResourceMatchingDemandSetDelete(rmds);
    }
    ** KheResourceMatchingSolverDelete(rms); **
    KheSolnArenaEnd(soln, a);
  }
  if( DEBUG19 && KheEjectorCurrDebug(ej) )
    fprintf(stderr, "%*s] KheNurseOverloadAugment returning false\n",
      KheEjectorCurrDebugIndent(ej), "");
  return false;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  Submodule "resource underload repair and augment"                        */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskSetRemovesUnderload(KHE_TASK_SET ts, KHE_MONITOR m,          */
/*    KHE_TIME_GROUP tg)                                                     */
/*                                                                           */
/*  Return true when ts removes the underload of tg in m.                    */
/*                                                                           */
/*  NB although tg is known to lie in m, we have to search for it, because   */
/*  its position may change as repairs are tried and fail.                   */
/*                                                                           */
/*  NBB we could perhaps pass the position of tg and obtain what we          */
/*  need from other queries on the monitor, based on position.               */
/*                                                                           */
/*****************************************************************************/

/* *** no longer called
static bool KheTaskSetRemovesUnderload(KHE_TASK_SET ts, KHE_MONITOR m,
  KHE_TIME_GROUP tg)
{
  KHE_LIMIT_BUSY_TIMES_MONITOR lbtm;  KHE_LIMIT_WORKLOAD_MONITOR lwm;
  KHE_TIME_GROUP tg2;  float ts_workload, workload;
  int i, ts_duration, busy_count, minimum, maximum;  bool allow_zero, res;
  res = false;
  switch( KheMonitorTag(m) )
  {
    case KHE_LIMIT_BUSY_TIMES_MONITOR_TAG:

      lbtm = (KHE_LIMIT_BUSY_TIMES_MONITOR) m;
      ts_duration = KheTaskSetTotalDuration(ts);
      for( i=0; i < KheLimitBus yTimesMonitorDefectiveTimeGroupCount(lbtm);i++ )
      {
	KheLimitBusyTimesMonitorDefect iveTimeGroup(lbtm, i, &tg2,
	  &busy_count, &minimum, &maximum, &allow_zero);
	if( tg2 == tg )
	{
	  res = (busy_count + ts_duration >= minimum);
	  break;
	}
      }
      HnAssert(i < KheLimitBusyTimesMonitor DefectiveTimeGroupCount(lbtm),
        "KheTaskSetRemovesUnderload internal error 1");
      break;

    case KHE_CLUSTER_BUSY_TIMES_MONITOR_TAG:

      ts_duration = KheTaskSetTotalDuration(ts);
      res = (ts_duration > 0);
      break;

    case KHE_LIMIT_WORKLOAD_MONITOR_TAG:

      ts_workload = KheTaskSetTotalWorkload(ts);
      lwm = (KHE_LIMIT_WORKLOAD_MONITOR) m;
      for( i = 0; i < KheLimitWorkloadMonitorDefectiveTimeGroupCount(lwm); i++ )
      {
	KheLimitWorkloadMonitorDefectiveTimeGroup(lwm, i, &tg2,
	  &workload, &minimum, &maximum, &allow_zero);
	if( tg2 == tg )
	{
	  res = (workload + ts_workload >= minimum);
	  break;
	}
      }
      HnAssert(i < KheLimitWorkloadMonitorDefectiveTimeGroupCount(lwm),
	"KheTaskSetRemovesUnderload internal error 3");
      break;

    default:

      res = true;  ** because we don't have anything special to say **
      break;
  }
  if( DEBUG22 )
  {
    fprintf(stderr, "  KheTaskSetRemovesUnderload(");
    KheTaskSetDebug(ts, 1, -1, stderr);
    fprintf(stderr, "%s, %s) = %s\n", KheMonitorId(m), KheTimeGroupId(tg),
      res ? "true" : "false");
  }
  return res;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheNurseUnderloadAugment(KHE_EJECTOR ej, KHE_RESOURCE r,            */
/*    KHE_TIME_GROUP tg, KHE_MONITOR d)                                      */
/*                                                                           */
/*  Try to remove an underload defect the nurse rostering way.               */
/*                                                                           */
/*****************************************************************************/

/* *** no longer called
static bool KheNurseUnderloadAugment(KHE_EJECTOR ej, KHE_RESOURCE r,
  KHE_TIME_GROUP tg, KHE_MONITOR d)
{
  KHE_FRAME frame;  int i, pos;  KHE_TIME t;  KHE_TIME_GROUP tg2;
  KHE_SOLN soln;  KHE_EVENT_TIMETABLE_MONITOR etm;  KHE_TASK task;
  KHE_RESOURCE_TIMETABLE_MONITOR rtm;  KHE_POLARITY po;  KHE_TASK_SET ts;

  soln = KheEjectorSoln(ej);
  et m = KheEjectorEventTimetableMonitor(ej);
  if( etm == NULL )  return false;
  rtm = KheResourceTimetableMonitor(soln, r);
  frame = KheEjectorFrame(ej);
  ts = KheTaskSetMake(soln);
  for( i = 0;  i < KheTimeGroupTimeCount(tg);  i++ )
  {
    ** build a task set of sufficient length, starting at time t **
    ** and with one task from succeeding positions in the frame **
    t = KheTimeGroupTime(tg, i);
    pos = KheFrameTimeIndex(frame, t);

    ** build a task set with a good chance of removing the underload **
    if( KheFindUnassignedTaskAtTime(etm, rtm, r, t, &task) )
    {
      KheTaskSetClear(ts);
      KheTaskSetAddTask(ts, task);
      while( KheTaskSetTaskCount(ts) <= 4 &&
	!KheTaskSetRemovesUnderload(ts, d, tg) )
      {
	pos++;
	if( pos >= KheFrameTimeGroupCount(frame) )
	  break;
	tg2 = KheFrameTimeGroup(frame, pos, &po);
        if( !KheFindUnassignedTaskAtTimeGroup(ej, etm, rtm, r, tg2, &task) )
	  break;
	KheTaskSetAddTask(ts, task);
      }

      ** try assigning the task set **
      if( !K heTaskSetVisited(ts, 0) )
      {
	K heTaskSetVisit(ts);
	if( KheTaskSetMoveRepair(ej, ts, r, KHE_MOVE_CHECKED) )
	{
	  KheTaskSetDelete(ts);
	  return true;
	}
	if( KheEjectorCurrMayRevisit(ej) )
	  K heTaskSetUnVisit(ts);
      }
    }
  }
  KheTaskSetDelete(ts);
  return false;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_TIME_GROUP KheSolnComplementTimeGroup(KHE_SOLN soln,                 */
/*    KHE_TIME_GROUP tg)                                                     */
/*                                                                           */
/*  Return a time group that is the cycle minus tg.                          */
/*                                                                           */
/*****************************************************************************/

static KHE_TIME_GROUP KheSolnComplementTimeGroup(KHE_SOLN soln,
  KHE_TIME_GROUP tg)
{
  KheSolnTimeGroupBegin(soln);
  KheSolnTimeGroupUnion(soln, KheInstanceFullTimeGroup(KheSolnInstance(soln)));
  KheSolnTimeGroupDifference(soln, tg);
  return KheSolnTimeGroupEnd(soln);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheIncreaseOverlapMeetMoveFn(KHE_MEET meet,                         */
/*    KHE_TIME time, void *impl)                                             */
/*                                                                           */
/*  Return true when moving meet to time would increase the overlap with     */
/*  a time group, passed in impl.  This function assumes that meet is        */
/*  currently assigned a time.                                               */
/*                                                                           */
/*****************************************************************************/

static bool KheIncreaseOverlapMeetMoveFn(KHE_MEET meet,
  KHE_TIME time, void *impl)
{
  KHE_TIME_GROUP tg;  int curr_overlap, new_overlap, durn;
  tg = (KHE_TIME_GROUP) impl;
  durn = KheMeetDuration(meet);
  curr_overlap = KheTimeGroupOverlap(tg, KheMeetAsstTime(meet), durn);
  new_overlap = KheTimeGroupOverlap(tg, time, durn);
  return new_overlap > curr_overlap;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheOneMeet(KHE_RESOURCE_TIMETABLE_MONITOR rtm, KHE_TIME_GROUP tg)   */
/*                                                                           */
/*  Return true if rtm contains exactly one meet overlapping tg.             */
/*                                                                           */
/*****************************************************************************/

static bool KheOneMeet(KHE_RESOURCE_TIMETABLE_MONITOR rtm, KHE_TIME_GROUP tg)
{
  int i, j, count;  KHE_TIME t;  KHE_MEET first_meet, meet;
  first_meet = NULL;
  for( i = 0;  i < KheTimeGroupTimeCount(tg);  i++ )
  {
    t = KheTimeGroupTime(tg, i);
    count = KheResourceTimetableMonitorTimeTaskCount(rtm, t);
    for( j = 0;  j < count;  j++ )
    {
      meet = KheTaskMeet(KheResourceTimetableMonitorTimeTask(rtm, t, j));
      if( first_meet == NULL )
	first_meet = meet;
      else if( meet != first_meet )
	return false;
    }
  }
  return first_meet != NULL;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheResourceBusyAtTimeGroup(KHE_RESOURCE_TIMETABLE_MONITOR rtm,      */
/*    KHE_TIME_GROUP tg, KHE_TASK *task)                                     */
/*                                                                           */
/*  If resource is busy during tg, return true and set *task to one task     */
/*  that proves it.  Otherwise return false.                                 */
/*                                                                           */
/*****************************************************************************/

/* ***
static bool KheResourceBusyAtTimeGroup(KHE_RESOURCE_TIMETABLE_MONITOR rtm,
  KHE_TIME_GROUP tg, KHE_TASK *task)
{
  int i;  KHE_TIME t;
  for( i = 0;  i < KheTimeGroupTimeCount(tg);  i++ )
  {
    t = KheTimeGroupTime(tg, i);
    if( KheResourceTimetableMonitorTimeTaskCount(rtm, t) > 0 )
      return *task = KheResourceTimetableMonitorTimeTask(rtm, t, 0), true;
  }
  return *task = NULL, false;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheResourceHasOneTaskOnDaysOfTask(                                  */
/*    KHE_RESOURCE_TIMETABLE_MONITOR rtm,                                    */
/*    KHE_FRAME days_frame, KHE_TASK r_task, KHE_TASK *task)                 */
/*                                                                           */
/*  If rtm's resource is busy doing exactly one task on the days that        */
/*  r_task is running, return true with *task set to that task.  Otherwise   */
/*  return false with *task set to NULL.                                     */
/*                                                                           */
/*****************************************************************************/

/* *** not currently used
static bool KheResourceHasOneTaskOnDaysOfTask(KHE_RESOURCE_TIMETABLE_MONITOR rtm,
  KHE_FRAME days_frame, KHE_TASK r_task, KHE_TASK *task)
{
  int first_index, last_index, i, j, k;  KHE_TIME_GROUP tg;  KHE_TIME t;
  KHE_TASK res_task, task2;
  res_task = NULL;
  KheTaskInterval(r_task, days_frame, &first_index, &last_index);
  for( i = first_index;  i <= last_index; i++ )
  {
    tg = KheFrameTimeGroup(days_frame, i);
    for( j = 0;  j < KheTimeGroupTimeCount(tg);  j++ )
    {
      t = KheTimeGroupTime(tg, j);
      for( k = 0;  k < KheResourceTimetableMonitorTimeTaskCount(rtm, t);  k++ )
      {
	task2 = KheResourceTimetableMonitorTimeTask(rtm, t, k);
        task2 = KheTaskProperRoot(task2);
	if( res_task == NULL )
	  res_task = task2;
	else if( res_task != task2 )
	  return *task = NULL, false;
      }
    }
  }
  return *task = res_task, (res_task != NULL);
  ** ***
  KHE_TIME t;  KHE_MEET meet;
  meet = KheTaskMeet(r_task);
  if( meet != NULL )
  {
    t = KheMeetAsstTime(meet);
    if( t != NULL )
      return KheResourceBusyOnDayOfTime(rtm, days_frame, t, task);
  }
  return *task = NULL, false;
  *** **
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskOverlapsTimeGroup(KHE_TASK task, KHE_TIME_GROUP tg)          */
/*                                                                           */
/*  Return true if task overlaps tg.                                         */
/*                                                                           */
/*****************************************************************************/

static bool KheTaskOverlapsTimeGroup(KHE_TASK task, KHE_TIME_GROUP tg)
{
  KHE_MEET meet;  KHE_TIME t, t2;  int i, pos;
  meet = KheTaskMeet(task);
  if( meet != NULL )
  {
    t = KheMeetAsstTime(meet);
    if( t != NULL )
    {
      for( i = 0;  i < KheMeetDuration(meet);  i++ )
      {
	t2 = KheTimeNeighbour(t, i);
	if( KheTimeGroupContains(tg, t2, &pos) )
	  return true;
      }
    }
  }
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheResourceGainTaskAugment(KHE_EJECTOR ej, KHE_RESOURCE r,          */
/*    KHE_TIME_GROUP tg, bool force, KHE_TASK_SET scratch_ts)                */
/*                                                                           */
/*  Carry out augments which move a task to r within time group tg.  If      */
/*  force is true, do this even at times when r is already busy.             */
/*                                                                           */
/*****************************************************************************/

static bool KheResourceGainTaskAugment(KHE_EJECTOR ej, KHE_RESOURCE r,
  KHE_TIME_GROUP tg, bool force, KHE_TASK_SET scratch_ts)
{
  int i, j, k, index, extra, tg_time_count;  KHE_RESOURCE_TYPE rt;
  KHE_RESOURCE_GROUP r_rg;  KHE_TIME t;  KHE_TASK r_task, task, prev_task;
  KHE_MEET meet;  KHE_RESOURCE r2;  KHE_RESOURCE_TIMETABLE_MONITOR rtm;
  KHE_EVENT_TIMETABLE_MONITOR etm;  KHE_FRAME days_frame;
  struct khe_expansion_options_rec eor;  KHE_SOLN soln;

  /* boilerplate */
  soln = KheEjectorSoln(ej);
  KheEjectorExpansionOptions(ej, &eor);
  r_rg = KheResourceSingletonResourceGroup(r);
  rt = KheResourceResourceType(r);
  rtm = KheResourceTimetableMonitor(soln, r);
  etm = KheEjectorEventTimetableMonitor(ej);
  days_frame = KheEjectorFrame(ej);

  /* try each time of tg */
  tg_time_count = KheTimeGroupTimeCount(tg);
  extra = KheEjectorCurrAugmentCount(ej);
  prev_task = NULL;
  for( i = 0;  i < tg_time_count;  i++ )
  {
    index = (extra + i) % tg_time_count;
    t = KheTimeGroupTime(tg, index);

    /* skip t if r is already busy on its day with a task overlapping tg */
    if( !force )
    {
      KheResourceBusyOnDayOfTime(rtm, days_frame, t, &r_task);
      if( r_task != NULL && (KheTaskIsPreassigned(r_task, &r2) ||
	  KheTaskOverlapsTimeGroup(r_task, tg)) )
	continue;
    }

    /* try to move a task to r at t */
    for( j = 0;  j < KheEventTimetableMonitorTimeMeetCount(etm, t);  j++ )
    {
      meet = KheEventTimetableMonitorTimeMeet(etm, t, j);
      for( k = 0;  k < KheMeetTaskCount(meet);  k++ )
      {
	task = KheMeetTask(meet, k);
	if( KheTaskResourceType(task) == rt )
	{
	  task = KheTaskProperRoot(task);
	  if( KheTaskMoveResourceCheck(task, r) &&
	      (prev_task == NULL || !KheTaskEquiv(prev_task, task)) )
	  {
	    if( KheTaskMoveAugment(ej, task, r_rg, NULL, false, &eor, NULL,
		  NULL, scratch_ts) )
	      return true;
	    prev_task = task;
	  }
	}
      }
    }
  }
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheResourceUnderloadAugment(KHE_EJECTOR ej, KHE_RESOURCE r,         */
/*    KHE_TIME_GROUP tg, bool allow_zero)                                    */
/*                                                                           */
/*  Carry out augments which try to either increase or (if allow_zero is     */
/*  true) reduce to zero the number of times that r is busy within tg.       */
/*                                                                           */
/*  Obsolete:                                                                */
/*  Here tg_index is the index of tg in d, if an index is needed to          */
/*  identify its position, and -1 otherwise.                                 */
/*                                                                           */
/*  Obsolete:                                                                */
/*  Parameter d is the monitor whose augment function called this function.  */
/*  It was traditionally used for debugging only, but now it is being used   */
/*  to determine how many tasks should be assigned within one repair.        */
/*                                                                           */
/*****************************************************************************/

static bool KheResourceUnderloadAugment(KHE_EJECTOR ej, KHE_RESOURCE r,
  KHE_TIME_GROUP tg, bool allow_zero)
{
  int i, tg_time_count, all_time_count, durn, extra, index;
  bool not_ejecting;  KHE_MEET meet;  KHE_TIME t;  KHE_SOLN soln;
  KHE_RESOURCE_TIMETABLE_MONITOR rtm;  KHE_TASK_SET scratch_ts;
  soln = KheEjectorSoln(ej);
  rtm = KheResourceTimetableMonitor(soln, r);
  tg_time_count = KheTimeGroupTimeCount(tg);

  if( KheEjectorRepairResources(ej) &&
      !KheResourceTypeDemandIsAllPreassigned(KheResourceResourceType(r)) )
  {
    /* try clearing out tg altogether, if allow_zero */
    if( allow_zero && KheResourceOverloadAugment(ej, r, tg, true) )
      return true;

    /* for each time t of tg, try to make r busy at t */
    scratch_ts = KheTaskSetMake(soln);
    if( KheResourceGainTaskAugment(ej, r, tg, false, scratch_ts) )
    {
      KheTaskSetDelete(scratch_ts);
      return true;
    }
    KheTaskSetDelete(scratch_ts);

    /* *** replaced by KheResourceGainTaskAugment
    KheEjectorExpansionOptions(ej, &eor);
    days_frame = KheEjectorFrame(ej);
    et m = KheEjectorEventTimetableMonitor(ej);
    r_rg = KheResourceSingletonResourceGroup(r);
    rt = KheResourceResourceType(r);
    extra = KheEjectorCurrAugmentCount(ej);
    prev_task = NULL;
    scratch_ts = KheTaskSetMake(soln);
    for( i = 0;  i < tg_time_count;  i++ )
    {
      index = (extra + i) % tg_time_count;
      t = KheTimeGroupTime(tg, index);

      ** skip t if r is already busy on its day with a task overlapping tg **
      KheResourceBusyOnDayOfTime(rtm, days_frame, t, &r_task);
      if( r_task != NULL && (KheTaskIsPreassigned(r_task, &r2) ||
	  KheTaskOverlapsTimeGroup(r_task, tg)) )
	continue;

      ** change r at t; it does not already contribute to tg there **
      for( j = 0;  j < KheEventTimetableMonitorTimeMeetCount(etm, t);  j++ )
      {
	meet = KheEventTimetableMonitorTimeMeet(etm, t, j);
	for( k = 0;  k < KheMeetTaskCount(meet);  k++ )
	{
	  task = KheMeetTask(meet, k);
	  task = KheTaskProperRoot(task);
	  if( KheTaskResourceType(task) == rt &&
	      KheTaskMoveResourceCheck(task, r) &&
              (prev_task == NULL || !KheTaskEquiv(prev_task, task)) )
	  {
	    if( KheTaskMoveAugment(ej, task, r_rg, NULL, false, &eor, NULL,
		  NULL, scratch_ts) )
	    {
	      KheTaskSetDelete(scratch_ts);
	      return true;
	    }
	    prev_task = task;
	  }
	}
      }
    }
    KheTaskSetDelete(scratch_ts);
    *** */
  }

  if( KheEjectorRepairTimes(ej) )
  {
    all_time_count = KheInstanceTimeCount(KheSolnInstance(soln));
    if( tg_time_count < all_time_count )
    {
      /* try ejecting meet moves into tg */
      /* NB finding tasks to move into tg, so might as well search whole list */
      extra = KheEjectorCurrAugmentCount(ej);
      not_ejecting = KheEjectorBasicNotEjecting(ej);
      for( i = 0;  i < KheResourceAssignedTaskCount(soln, r);  i++ )
      {
	index = (extra + i) % KheResourceAssignedTaskCount(soln, r);
	meet = KheTaskMeet(KheResourceAssignedTask(soln, r, index));
	t = KheMeetAsstTime(meet);
	durn = KheMeetDuration(meet);
	if( t != NULL && KheTimeGroupOverlap(tg, t, durn) < durn &&
	  KheMeetAugment(ej, meet, false, &KheIncreaseOverlapMeetMoveFn,
	    (void *) tg, KHE_OPTIONS_KEMPE_TRUE, !not_ejecting, not_ejecting,
	    false) )
	  return true;
      }

      /* try clearing out tg altogether, using a cluster-like repair */
      if( allow_zero && (KheEjectorCurrDepth(ej) == 1 || KheOneMeet(rtm, tg)) )
      {
	if( KheEjectorCurrDepth(ej) == 1 )
	  KheSolnNewGlobalVisit(soln);
	if( KheMeetBoundRepair(ej, r, KheSolnComplementTimeGroup(soln, tg)) )
	  return true;
      }
    }
  }
  return false;
}

/* *** old code now replaced by a single call to KheTaskMoveAugment
{
  r2 = KheTaskAsstResource(task);
  if( r_task == NULL )
  {
    ** r not busy on t's day, so try a regular task move augment **
    if( KheTaskMov eAugment(ej, task, r_rg, NULL, false, NULL,NULL) )
      return true;
  }
  else if( r2 != NULL )
  {
    ** try r_task := r2, task := r (expanded) **
    K heTaskVisit(task);
    if( KheTaskDoubleMoveMultiRepair(ej, r_task, task) )
      return true;
    if( KheEjectorCurrMayRevisit(ej) )
      K heTaskUnVisit(task);
  }
  else if( prev_task == NULL || !KheTas kEquivalent(prev_task,task) )
  {
    ** only try task if different from previous unassigned task **
    if( KheResourceTimetableMonitorTaskAvailableInFrame(rtm,
	  task, days_frame, r_task) )
    {
      K heTaskVisit(task);
      prev_task = task;
      ** try r_task := NULL, task := r **
      if( KheTaskDoubleMoveRepair(ej, r_task, NULL, task, r) )
	return true;
      if( KheEjectorCurrMayRevisit(ej) )
	K heTaskUnVisit(task);
    }
  }
}
*** */

/* *** old version
static bool KheResourceUnderloadAugment(KHE_EJECTOR ej, KHE_RESOURCE r,
  KHE_TIME_GROUP tg, bool allow_zero)
{
  int i, tg_time_count, all_time_count, index, durn, extra;
  bool not_ejecting;
  KHE_MEET meet;  KHE_TIME t;  KHE_SOLN soln;
  KHE_RESOURCE_TIMETABLE_MONITOR rtm;
  soln = KheEjectorSoln(ej);
  rtm = KheResourceTimetableMonitor(soln, r);

  if( KheEjectorRepairResources(ej) &&
      !KheResourceTypeDemandIsAllPreassigned(KheResourceResourceType(r)) )
  {
    ** try clearing out tg altogether, if allow_zero **
    if( allow_zero && KheResourceOverloadAugment(ej, r, tg, true) )
      return true;

    ** try ejecting task moves of r to tasks in tg, unassigned tasks first **
    if( KheMoveTaskToResourceAugment(ej, r, tg, false) )
      ** , KHE_INCREASE_WORKLOAD) ) **
      return true;
    if( KheMoveTaskToResourceAugment(ej, r, tg, true) )
      ** , KHE_INCREASE_WORKLOAD) ) **
      return true;

    ** try a new nurse rostering repair if that's the model **
    ** *** obsolete
    if( KheInstanceModel(KheSolnInstance(soln)) == KHE_MODEL_EMPLOYEE_SCHEDULE )
    {
      if( KheNurseUnderloadAugment(ej, r, tg, d) )
	return true;
    }
    else
    {
      ** try ejecting task moves of r to tasks in tg **
      if( KheMoveTaskToResourceAugment(ej, r, tg, false) )
	return true;
      if( KheMoveTaskToResourceAugment(ej, r, tg, true) )
	return true;
    }
    *** **
  }

  if( KheEjectorRepairTimes(ej) )
  {
    tg_time_count = KheTimeGroupTimeCount(tg);
    all_time_count = KheInstanceTimeCount(KheSolnInstance(soln));
    if( tg_time_count < all_time_count )
    {
      ** try ejecting meet moves into tg **
      ** NB finding tasks to move into tg, so might as well search whole list **
      extra = KheEjectorCurrAugmentCount(ej);
      not_ejecting = KheEjectorBasicNotEjecting(ej);
      for( i = 0;  i < KheResourceAssignedTaskCount(soln, r);  i++ )
      {
	index = (extra + i) % KheResourceAssignedTaskCount(soln, r);
	meet = KheTaskMeet(KheResourceAssignedTask(soln, r, index));
	t = KheMeetAsstTime(meet);
	durn = KheMeetDuration(meet);
	if( t != NULL && KheTimeGroupOverlap(tg, t, durn) < durn &&
	  KheMeetAugment(ej, meet, false, &KheIncreaseOverlapMeetMoveFn,
	    (void *) tg, KHE_OPTIONS_KEMPE_TRUE, !not_ejecting, not_ejecting,
	    false) )
	  return true;
      }

      ** try clearing out tg altogether, using a cluster-like repair **
      if( allow_zero && (KheEjectorCurrDepth(ej) == 1 || KheOneMeet(rtm, tg)) )
      {
	if( KheEjectorCurrDepth(ej) == 1 )
	  KheSolnNewGlo balVisit(soln);
	if( KheMeetBoundRepair(ej, r, KheSolnComplementTimeGroup(soln, tg)) )
	  return true;
      }
    }
  }
  return false;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheOverUnderAugment(KHE_EJECTOR ej, KHE_RESOURCE r,                 */
/*    KHE_TIME_GROUP tg, bool over, bool require_zero, bool allow_zero)      */
/*                                                                           */
/*  Call KheResourceOverloadAugment(ej, r, tg, require_zero) if over is true,*/
/*  and KheResourceUnderloadAugment(ej, r, tg, allow_zero) if over is false. */
/*                                                                           */
/*****************************************************************************/

static bool KheOverUnderAugment(KHE_EJECTOR ej, KHE_RESOURCE r,
  KHE_TIME_GROUP tg, bool over, bool require_zero, bool allow_zero)
{
  if( over )
    return KheResourceOverloadAugment(ej, r, tg, require_zero);
  else
    return KheResourceUnderloadAugment(ej, r, tg, allow_zero);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "split events augment functions"                               */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_EVENT KheSplitMonitorEvent(KHE_MONITOR d)                            */
/*                                                                           */
/*  Return the event monitored by split events or distribute split events    */
/*  monitor d.                                                               */
/*                                                                           */
/*****************************************************************************/

static KHE_EVENT KheSplitMonitorEvent(KHE_MONITOR d)
{
  switch( KheMonitorTag(d) )
  {
    case KHE_SPLIT_EVENTS_MONITOR_TAG:

      return KheSplitEventsMonitorEvent((KHE_SPLIT_EVENTS_MONITOR) d);

    case KHE_DISTRIBUTE_SPLIT_EVENTS_MONITOR_TAG:

      return KheDistributeSplitEventsMonitorEvent(
	(KHE_DISTRIBUTE_SPLIT_EVENTS_MONITOR) d);

    default:

      HnAbort("KheSplitMonitorEvent internal error");
      return NULL;  /* keep compiler happy */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheSolnEventMeetsVisited(KHE_SOLN soln, KHE_EVENT e, int slack)     */
/*                                                                           */
/*  Like KheMeetVisited(meet, slack) only applied to all the meets of soln   */
/*  derived from e; it returns true if any of them are visited.              */
/*                                                                           */
/*****************************************************************************/

static bool KheSolnEventMeetsVisited(KHE_SOLN soln, KHE_EVENT e, int slack)
{
  int i;
  for( i = 0;  i < KheEventMeetCount(soln, e);  i++ )
    if( KheMeetVisited(KheEventMeet(soln, e, i), slack) )
      return true;
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSolnEventMeetsVisit(KHE_SOLN soln, KHE_EVENT e)                  */
/*                                                                           */
/*  KheMeetVisit(meet) applied to all the meets of soln derived from e.      */
/*                                                                           */
/*****************************************************************************/

static void KheSolnEventMeetsVisit(KHE_SOLN soln, KHE_EVENT e)
{
  int i;
  for( i = 0;  i < KheEventMeetCount(soln, e);  i++ )
    KheMeetVisit(KheEventMeet(soln, e, i));
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSolnEventMeetsUnVisit(KHE_SOLN soln, KHE_EVENT e)                */
/*                                                                           */
/*  KheMeetUnVisit(meet) applied to all the meets of soln derived from e.    */
/*                                                                           */
/*****************************************************************************/

static void KheSolnEventMeetsUnVisit(KHE_SOLN soln, KHE_EVENT e)
{
  int i;
  for( i = 0;  i < KheEventMeetCount(soln, e);  i++ )
    KheMeetUnVisit(KheEventMeet(soln, e, i));
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheSplitEventsAugment(KHE_EJECTOR ej, KHE_MONITOR d)                */
/*                                                                           */
/*  Augment function for individual split events and distribute split        */
/*  events defects.                                                          */
/*                                                                           */
/*****************************************************************************/

static bool KheSplitEventsAugment(KHE_EJECTOR ej, KHE_MONITOR d)
{
  KHE_SPLIT_ANALYSER sa;  int i, j, k, merged_durn, split1_durn, split2_durn;
  KHE_SOLN soln;  KHE_EVENT e;  KHE_MEET meet1, meet2;
  if( DEBUG4 )
  {
    fprintf(stderr, "%*s[ KheSplitEventsAugment(",
      KheEjectorCurrDebugIndent(ej) + 2, "");
    KheMonitorDebug(d, 1, -1, stderr);
    fprintf(stderr, ")\n");
  }
  HnAssert(KheMonitorTag(d) == KHE_SPLIT_EVENTS_MONITOR_TAG ||
    KheMonitorTag(d) == KHE_DISTRIBUTE_SPLIT_EVENTS_MONITOR_TAG,
    "KheSplitEventsAugment internal error 1");
  if( KheEjectorUseSplitMoves(ej) )
  {
    soln = KheMonitorSoln(d);
    e = KheSplitMonitorEvent(d);
    if( !KheSolnEventMeetsVisited(soln, e, 0) )
    {
      /* get split and merge suggestions */
      sa = KheSplitAnalyserOption(KheEjectorOptions(ej), "ts_split_analyser",
	soln);
      /* ***
      sa = (KHE_SPLIT_ANALYSER) KheOptionsGetObject(KheEjectorOptions(ej),
	"ts_split_analyser", NULL);
      *** */
      /* sa = KheOptionsStructuralSplitAnalyser(KheEjectorOptions(ej)); */
      KheSolnEventMeetsVisit(soln, e);
      KheSplitAnalyserAnalyse(sa, e);
      if( DEBUG10 )
	KheSplitAnalyserDebug(sa, 1, KheEjectorCurrDebugIndent(ej) + 2, stderr);

      /* carry out sa's split suggestions */
      for( i = 0;  i < KheSplitAnalyserSplitSuggestionCount(sa);  i++ )
      {
	KheSplitAnalyserSplitSuggestion(sa, i, &merged_durn, &split1_durn);
	for( j = 0;  j < KheEventMeetCount(soln, e);  j++ )
	{
	  meet1 = KheEventMeet(soln, e, j);
	  if( KheMeetAsst(meet1)!=NULL && KheMeetDuration(meet1)==merged_durn )
	  {
            if( KheMeetSplitRepair(ej, meet1, split1_durn) )
	    {
	      if( DEBUG4 )
		fprintf(stderr, "%*s] KheSplitEventsAugment ret true (a)\n",
		  KheEjectorCurrDebugIndent(ej) + 2, "");
	      return true;
	    }
	  }
	}
      }

      /* carry out sa's merge suggestions */
      for( i = 0;  i < KheSplitAnalyserMergeSuggestionCount(sa);  i++ )
      {
	KheSplitAnalyserMergeSuggestion(sa, i, &split1_durn, &split2_durn);
	for( j = 0;  j < KheEventMeetCount(soln, e);  j++ )
	{
	  meet1 = KheEventMeet(soln, e, j);
	  if( KheMeetAsst(meet1)!=NULL && KheMeetDuration(meet1)==split1_durn )
	  {
	    for( k = j + 1;  k < KheEventMeetCount(soln, e);  k++ )
	    {
	      meet2 = KheEventMeet(soln, e, k);
	      if( KheMeetAsst(meet2) != NULL &&
		  KheMeetDuration(meet2) == split2_durn )
	      {
		if( KheMergeMoveRepair(ej, meet1, meet2, false) )
		{
		  if( DEBUG4 )
		    fprintf(stderr, "%*s] KheSplitEventsAugment ret true\n",
		      KheEjectorCurrDebugIndent(ej) + 2, "");
		  return true;
		}
		if( KheMergeMoveRepair(ej, meet1, meet2, true) )
		{
		  if( DEBUG4 )
		    fprintf(stderr, "%*s] KheSplitEventsAugment ret true\n",
		      KheEjectorCurrDebugIndent(ej) + 2, "");
		  return true;
		}
		if( KheMergeMoveRepair(ej, meet2, meet1, false) )
		{
		  if( DEBUG4 )
		    fprintf(stderr, "%*s] KheSplitEventsAugment ret true\n",
		      KheEjectorCurrDebugIndent(ej) + 2, "");
		  return true;
		}
		if( KheMergeMoveRepair(ej, meet2, meet1, true) )
		{
		  if( DEBUG4 )
		    fprintf(stderr, "%*s] KheSplitEventsAugment ret true\n",
		      KheEjectorCurrDebugIndent(ej) + 2, "");
		  return true;
		}
	      }
	    }
	  }
	}
      }
      if( KheEjectorCurrMayRevisit(ej) )
	KheSolnEventMeetsUnVisit(soln, e);
    }
  }
  if( DEBUG4 )
    fprintf(stderr, "%*s] KheSplitEventsAugment ret false\n",
      KheEjectorCurrDebugIndent(ej) + 2, "");
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheSplitEventsGroupAugment(KHE_EJECTOR ej, KHE_MONITOR d)           */
/*                                                                           */
/*  Augment function for groups of split events and distribute split events  */
/*  defects.                                                                 */
/*                                                                           */
/*****************************************************************************/

static bool KheSplitEventsGroupAugment(KHE_EJECTOR ej, KHE_MONITOR d)
{
  KHE_GROUP_MONITOR gm;
  HnAssert(KheMonitorTag(d) == KHE_GROUP_MONITOR_TAG,
    "KheSplitEventsGroupAugment internal error 1");
  gm = (KHE_GROUP_MONITOR) d;
  HnAssert(KheGroupMonitorDefectCount(gm) > 0,
    "KheSplitEventsGroupAugment internal error 2");
  return KheSplitEventsAugment(ej, KheGroupMonitorDefect(gm, 0));
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "assign time augment functions"                                */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheDebugAugmentBegin(bool debug, KHE_EJECTOR ej, KHE_MONITOR d,     */
/*    char *str)                                                             */
/*  void KheDebugAugmentEnd(bool debug, KHE_EJECTOR ej, KHE_MONITOR d,       */
/*    char *str, bool res)                                                   */
/*                                                                           */
/*  Debug functions to call at the start and end of each augment function.   */
/*                                                                           */
/*****************************************************************************/

static void KheDebugAugmentBegin(bool debug, KHE_EJECTOR ej, KHE_MONITOR d,
  char *str)
{
  if( debug )
  {
    fprintf(stderr, "%*s[ %s(", KheEjectorCurrDebugIndent(ej) + 2, "", str);
    KheMonitorDebug(d, 1, -1, stderr);
    fprintf(stderr, ")\n");
  }
}

static void KheDebugAugmentEnd(bool debug, KHE_EJECTOR ej, KHE_MONITOR d,
  char *str, bool res)
{
  if( debug )
    fprintf(stderr, "%*s] %s ret %s\n", KheEjectorCurrDebugIndent(ej) + 2, "",
      str, res ? "true" : "false");
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheInDomainMeetMoveFn(KHE_MEET meet, KHE_TIME time, void *impl)     */
/*                                                                           */
/*  Return true when assigning or moving meet to time would place its        */
/*  starting time in a given time group, passed in impl.                     */
/*                                                                           */
/*****************************************************************************/

static bool KheInDomainMeetMoveFn(KHE_MEET meet, KHE_TIME time, void *impl)
{
  int pos;
  return KheTimeGroupContains((KHE_TIME_GROUP) impl, time, &pos);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheNotInDomainMeetMoveFn(KHE_MEET meet, KHE_TIME time, void *impl)  */
/*                                                                           */
/*  Return true when assigning or moving meet to time would not place its    */
/*  starting time in a given time group, passed in impl.                     */
/*                                                                           */
/*****************************************************************************/

static bool KheNotInDomainMeetMoveFn(KHE_MEET meet, KHE_TIME time, void *impl)
{
  int pos;
  return !KheTimeGroupContains((KHE_TIME_GROUP) impl, time, &pos);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheAssignTimeAugment(KHE_EJECTOR ej, KHE_MONITOR d)                 */
/*                                                                           */
/*  Augment function for individual assign time defects.                     */
/*                                                                           */
/*  If KheEjectorRepairTimes(ej), then for each monitored unassigned meet,   */
/*  try all ejecting meet moves to a parent meet and offset that would       */
/*  assign the unassigned meet within its domain.                            */
/*                                                                           */
/*****************************************************************************/

static bool KheAssignTimeAugment(KHE_EJECTOR ej, KHE_MONITOR d)
{
  KHE_SOLN soln;  KHE_EVENT e;  KHE_MEET meet;  int i;
  bool not_ejecting;  KHE_ASSIGN_TIME_MONITOR atm;
  KheDebugAugmentBegin(DEBUG4, ej, d, "KheAssignTimeAugment");
  HnAssert(KheMonitorTag(d) == KHE_ASSIGN_TIME_MONITOR_TAG,
    "KheAssignTimeAugment internal error 1");
  if( KheEjectorRepairTimes(ej) )
  {
    atm = (KHE_ASSIGN_TIME_MONITOR) d;
    soln = KheEjectorSoln(ej);
    e = KheAssignTimeMonitorEvent(atm);
    not_ejecting = KheEjectorBasicNotEjecting(ej);
    for( i = 0;  i < KheEventMeetCount(soln, e);  i++ )
    {
      meet = KheEventMeet(soln, e, i);
      if( KheMeetAsstTime(meet) == NULL )
      {
	/* ***
	if( KheEjectorCurrDepth(ej) == 1 )
	  KheSolnNewG lobalVisit(soln);
	*** */
	if( KheMeetAugment(ej, meet, true, &KheInDomainMeetMoveFn,
	  (void *) KheMeetDomain(meet), KHE_OPTIONS_KEMPE_FALSE,
	  !not_ejecting, not_ejecting, false) )
	{
	  KheDebugAugmentEnd(DEBUG4, ej, d, "KheAssignTimeAugment", true);
	  return true;
	}
      }
    }
  }
  KheDebugAugmentEnd(DEBUG4, ej, d, "KheAssignTimeAugment", false);
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheAssignTimeGroupAugment(KHE_EJECTOR ej, KHE_MONITOR d)            */
/*                                                                           */
/*  Augment function for groups of assign time defects.                      */
/*                                                                           */
/*****************************************************************************/

static bool KheAssignTimeGroupAugment(KHE_EJECTOR ej, KHE_MONITOR d)
{
  KHE_GROUP_MONITOR gm;
  HnAssert(KheMonitorTag(d) == KHE_GROUP_MONITOR_TAG,
    "KheAssignTimeGroupAugment internal error 1");
  gm = (KHE_GROUP_MONITOR) d;
  HnAssert(KheGroupMonitorDefectCount(gm) > 0,
    "KheAssignTimeGroupAugment internal error 2");
  return KheAssignTimeAugment(ej, KheGroupMonitorDefect(gm, 0));
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "prefer times augment functions"                               */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool KhePreferTimesAugment(KHE_EJECTOR ej, KHE_MONITOR d)                */
/*                                                                           */
/*  Augment function for individual prefer times defects.                    */
/*                                                                           */
/*  If KheEjectorRepairTimes(ej), then for each monitored meet with an       */
/*  unpreferred assignment, try all meet moves to a parent meet and offset   */
/*  that give the meet a preferred assignment.                               */
/*                                                                           */
/*****************************************************************************/

static bool KhePreferTimesAugment(KHE_EJECTOR ej, KHE_MONITOR d)
{
  KHE_SOLN soln;  KHE_EVENT e;  KHE_MEET meet;  int i, durn, pos;
  KHE_PREFER_TIMES_CONSTRAINT ptc;  KHE_TIME_GROUP domain;  KHE_TIME t;
  bool not_ejecting;  KHE_OPTIONS_KEMPE kempe;  KHE_PREFER_TIMES_MONITOR ptm;
  KheDebugAugmentBegin(DEBUG4, ej, d, "KhePreferTimesAugment");
  HnAssert(KheMonitorTag(d) == KHE_PREFER_TIMES_MONITOR_TAG,
    "KhePreferTimesAugment internal error 1");
  if( KheEjectorRepairTimes(ej) )
  {
    soln = KheEjectorSoln(ej);
    ptm = (KHE_PREFER_TIMES_MONITOR) d;
    e = KhePreferTimesMonitorEvent(ptm);
    ptc = KhePreferTimesMonitorConstraint(ptm);
    domain = KhePreferTimesConstraintDomain(ptc);
    durn = KhePreferTimesConstraintDuration(ptc);
    not_ejecting = KheEjectorBasicNotEjecting(ej);
    kempe = KheEjectorUseKempeMoves(ej);
    for( i = 0;  i < KheEventMeetCount(soln, e);  i++ )
    {
      meet = KheEventMeet(soln, e, i);
      t = KheMeetAsstTime(meet);
      if( (durn == KHE_ANY_DURATION || KheMeetDuration(meet) == durn) &&
	  t != NULL && !KheTimeGroupContains(domain, t, &pos) )
      {
	/* ***
	if( KheEjectorCurrDepth(ej) == 1 )
	  KheSolnNewG lobalVisit(soln);
	*** */
	if( KheMeetAugment(ej, meet, true, &KheInDomainMeetMoveFn,
	    (void *) domain, kempe, !not_ejecting, not_ejecting,
	    WITH_PREFER_TIMES_NODE_SWAPS) )
	{
	  KheDebugAugmentEnd(DEBUG4, ej, d, "KhePreferTimesAugment", true);
	  return true;
	}
      }
    }
  }
  KheDebugAugmentEnd(DEBUG4, ej, d, "KhePreferTimesAugment", false);
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KhePreferTimesGroupAugment(KHE_EJECTOR ej, KHE_MONITOR d)           */
/*                                                                           */
/*  Augment function for groups of prefer times defects.                     */
/*                                                                           */
/*****************************************************************************/

static bool KhePreferTimesGroupAugment(KHE_EJECTOR ej, KHE_MONITOR d)
{
  KHE_GROUP_MONITOR gm;
  HnAssert(KheMonitorTag(d) == KHE_GROUP_MONITOR_TAG,
    "KhePreferTimesGroupAugment internal error 1");
  gm = (KHE_GROUP_MONITOR) d;
  HnAssert(KheGroupMonitorDefectCount(gm) > 0,
    "KhePreferTimesGroupAugment internal error 2");
  return KhePreferTimesAugment(ej, KheGroupMonitorDefect(gm, 0));
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "spread events augment functions"                              */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool KheSpreadEventsAugment(KHE_EJECTOR ej, KHE_MONITOR d)               */
/*                                                                           */
/*  Augment function for individual spread events defects.                   */
/*                                                                           */
/*  If KheEjectorRepairTimes(ej), try all meet moves of the relevant         */
/*  meets from outside each day to inside, or vice versa, depending on       */
/*  whether the problem is too few meets or too many.                        */
/*                                                                           */
/*****************************************************************************/

static bool KheSpreadEventsAugment(KHE_EJECTOR ej, KHE_MONITOR d)
{
  KHE_SOLN soln;  KHE_EVENT_GROUP eg;  KHE_EVENT e;  KHE_MEET meet;
  KHE_SPREAD_EVENTS_MONITOR sem;  KHE_TIME time;  KHE_TIME_GROUP tg;
  int i, j, k, minimum, maximum, inc, index, extra, pos;  bool not_ejecting;
  KHE_OPTIONS_KEMPE kempe;
  KheDebugAugmentBegin(DEBUG4, ej, d, "KheSpreadEventsAugment");
  HnAssert(KheMonitorTag(d) == KHE_SPREAD_EVENTS_MONITOR_TAG,
    "KheSpreadEventsAugment internal error 1");
  if( KheEjectorRepairTimes(ej) )
  {
    not_ejecting = KheEjectorBasicNotEjecting(ej);
    kempe = KheEjectorUseKempeMoves(ej);
    soln = KheEjectorSoln(ej);
    sem = (KHE_SPREAD_EVENTS_MONITOR) d;
    eg = KheSpreadEventsMonitorEventGroup(sem);
    extra = KheEjectorCurrAugmentCount(ej) +
      5 * KheSolnDiversifier(KheEjectorSoln(ej));
    for( i = 0;  i < KheSpreadEventsMonitorTimeGroupCount(sem);  i++ )
    {
      KheSpreadEventsMonitorTimeGroup(sem, i, &tg, &minimum, &maximum, &inc);
      if( inc < minimum )
      {
	/* try all meet moves from outside tg to inside tg */
	for( j = 0;  j < KheEventGroupEventCount(eg);  j++ )
	{
	  index = (j + extra) % KheEventGroupEventCount(eg);
	  e = KheEventGroupEvent(eg, index);
	  for( k = 0;  k < KheEventMeetCount(soln, e);  k++ )
	  {
	    index = (k + extra) % KheEventMeetCount(soln, e);
	    meet = KheEventMeet(soln, e, index);
	    time = KheMeetAsstTime(meet);
	    if( time != NULL && !KheTimeGroupContains(tg, time, &pos) &&
	        KheMeetAugment(ej, meet, true, &KheInDomainMeetMoveFn,
		  (void *) tg, kempe, !not_ejecting, not_ejecting,
		  WITH_SPREAD_EVENTS_NODE_SWAPS) )
	    {
	      KheDebugAugmentEnd(DEBUG4, ej, d, "KheSpreadEventsAugment (a)",
		true);
	      return true;
	    }
	  }
	}
      }
      else if( inc > maximum )
      {
	/* try all meet moves from inside tg to outside tg */
	for( j = 0;  j < KheEventGroupEventCount(eg);  j++ )
	{
	  index = (extra + j) % KheEventGroupEventCount(eg);
	  e = KheEventGroupEvent(eg, index);
	  for( k = 0;  k < KheEventMeetCount(soln, e);  k++ )
	  {
            index = (k + extra) % KheEventMeetCount(soln, e);
	    meet = KheEventMeet(soln, e, index);
	    time = KheMeetAsstTime(meet);
	    if( time != NULL && KheTimeGroupContains(tg, time, &pos) &&
	        KheMeetAugment(ej, meet, true, &KheNotInDomainMeetMoveFn,
		  (void *) tg, kempe, !not_ejecting, not_ejecting,
		  WITH_SPREAD_EVENTS_NODE_SWAPS) )
	    {
	      KheDebugAugmentEnd(DEBUG4, ej, d, "KheSpreadEventsAugment (b)",
		true);
	      return true;
	    }
	  }
	}
      }
    }
  }
  KheDebugAugmentEnd(DEBUG4, ej, d, "KheSpreadEventsAugment", false);
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheSpreadEventsGroupAugment(KHE_EJECTOR ej, KHE_MONITOR d)          */
/*                                                                           */
/*  Augment function for groups of spread events defects.                    */
/*                                                                           */
/*****************************************************************************/

static bool KheSpreadEventsGroupAugment(KHE_EJECTOR ej, KHE_MONITOR d)
{
  KHE_GROUP_MONITOR gm;
  HnAssert(KheMonitorTag(d) == KHE_GROUP_MONITOR_TAG,
    "KheSpreadEventsGroupAugment internal error 1");
  gm = (KHE_GROUP_MONITOR) d;
  HnAssert(KheGroupMonitorDefectCount(gm) > 0,
    "KheSpreadEventsGroupAugment internal error 2");
  return KheSpreadEventsAugment(ej, KheGroupMonitorDefect(gm, 0));
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "order events augment functions"                               */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool KheOrderEventsAugment(KHE_EJECTOR ej, KHE_MONITOR d)                */
/*                                                                           */
/*  Augment function for individual order events defects.                    */
/*                                                                           */
/*  Not implemented yet.                                                     */
/*                                                                           */
/*****************************************************************************/

static bool KheOrderEventsAugment(KHE_EJECTOR ej, KHE_MONITOR d)
{
  KheDebugAugmentBegin(DEBUG4, ej, d, "KheOrderEventsAugment");
  HnAssert(KheMonitorTag(d) == KHE_ORDER_EVENTS_MONITOR_TAG,
    "KheOrderEventsAugment internal error 1");
  if( KheEjectorRepairTimes(ej) )
  {
    /* still to do */
  }
  KheDebugAugmentEnd(DEBUG4, ej, d, "KheOrderEventsAugment", false);
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheOrderEventsGroupAugment(KHE_EJECTOR ej, KHE_MONITOR d)           */
/*                                                                           */
/*  Augment function for groups of order events defects.                     */
/*                                                                           */
/*****************************************************************************/

static bool KheOrderEventsGroupAugment(KHE_EJECTOR ej, KHE_MONITOR d)
{
  KHE_GROUP_MONITOR gm;
  HnAssert(KheMonitorTag(d) == KHE_GROUP_MONITOR_TAG,
    "KheOrderEventsGroupAugment internal error 1");
  gm = (KHE_GROUP_MONITOR) d;
  HnAssert(KheGroupMonitorDefectCount(gm) > 0,
    "KheOrderEventsGroupAugment internal error 2");
  return KheOrderEventsAugment(ej, KheGroupMonitorDefect(gm, 0));
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "assign resource augment functions"                            */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool KheEventResourceUnderloadAugment(KHE_EJECTOR ej,                    */
/*    KHE_EVENT_RESOURCE er, KHE_RESOURCE_GROUP rg, bool allow_unassign,     */
/*    KHE_MONITOR d)                                                         */
/*                                                                           */
/*  Try to repair defect d, by ensuring that the tasks of event resource er  */
/*  are assigned resources from rg, or are unassigned if allow_unassigned.   */
/*                                                                           */
/*****************************************************************************/

/* ***
static bool KheEventResourceUnderloadAugment(KHE_EJECTOR ej,
  KHE_EVENT_RESOURCE er, KHE_RESOURCE_GROUP rg, bool allow_unassign,
  KHE_MONITOR d)
{
  int i;  KHE_SOLN soln;  KHE_TASK task;  KHE_RESOURCE from_r;
  soln = KheEjectorSoln(ej);
  for( i = 0;  i < KheEventResourceTaskCount(soln, er);  i++ )
  {
    task = KheEventResourceTask(soln, er, i);
    task = KheTaskProperRoot(task);
    from_r = KheTaskAsstResource(task);
    if( from_r == NULL || !KheResourceGroupContains(rg, from_r) )
    {
      if( KheTaskM oveAugment(ej, task, rg, NULL, allow_unassign, NULL, d) )
	return true;
    }
  }
  return false;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheEventResourceOverloadAugment(KHE_EJECTOR ej,                     */
/*    KHE_EVENT_RESOURCE er, KHE_RESOURCE_GROUP not_rg, KHE_MONITOR d)       */
/*                                                                           */
/*  Try to repair defect d, by ensuring that the tasks of event resource er  */
/*  are not assigned resources from rg.  Unassigning them is one option.     */
/*                                                                           */
/*****************************************************************************/

/* ***
static bool KheEventResourceOverloadAugment(KHE_EJECTOR ej,
  KHE_EVENT_RESOURCE er, KHE_RESOURCE_GROUP not_rg, KHE_MONITOR d)
{
  int i;  KHE_SOLN soln;  KHE_TASK task;  KHE_RESOURCE from_r;
  KHE_RESOURCE_GROUP domain;
  soln = KheEjectorSoln(ej);
  for( i = 0;  i < KheEventResourceTaskCount(soln, er);  i++ )
  {
    ** here task is one of the tasks being monitored **
    task = KheEventResourceTask(soln, er, i);
    task = KheTaskProperRoot(task);
    from_r = KheTaskAsstResource(task);
    if( from_r != NULL && KheResourceGroupContains(not_rg, from_r) )
    {
      domain = KheTaskDomain(task);
      if( KheTaskMo veAugment(ej, task, domain, not_rg, true, NULL, d) )
	return true;
    }
  }
  return false;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskSatisfies(KHE_TASK task, KHE_RESOURCE_GROUP rg,              */
/*    KHE_RESOURCE_GROUP not_rg, bool allow_unassign)                        */
/*                                                                           */
/*  Return true if task is currently assigned a resource from rg but         */
/*  not from not_rg, or is unassigned if allow_unassign is true.             */
/*  When rg and not_rg are NULL, no condition is imposed by them.            */
/*                                                                           */
/*****************************************************************************/

static bool KheTaskSatisfies(KHE_TASK task, KHE_RESOURCE_GROUP rg,
  KHE_RESOURCE_GROUP not_rg, bool allow_unassign)
{
  KHE_RESOURCE r;
  r = KheTaskAsstResource(task);
  if( r == NULL )
   return allow_unassign;
  else
    return (rg == NULL || KheResourceGroupContains(rg, r)) &&
    (not_rg == NULL || !KheResourceGroupContains(not_rg, r));
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheEventResourceMoveAugment(KHE_EJECTOR ej, KHE_EVENT_RESOURCE er,  */
/*    KHE_RESOURCE_GROUP rg, KHE_RESOURCE_GROUP not_rg, bool allow_unassign, */
/*    KHE_MONITOR d, KHE_TASK *prev_task)                                    */
/*                                                                           */
/*  Try to repair defect d, by ensuring that the tasks of event resource er  */
/*  are assigned resources from rg (which must be present) but not not_rg    */
/*  (if present), and are unassigned or not depending on allow_unassign.     */
/*                                                                           */
/*  Here *prev_task is the task that this was tried on previously, or        */
/*  NULL if none.  This is used to avoid trying the same augment on          */
/*  equivalent tasks, and reassigned by this operation.                      */
/*                                                                           */
/*****************************************************************************/

static bool KheEventResourceMoveAugment(KHE_EJECTOR ej, KHE_EVENT_RESOURCE er,
  KHE_RESOURCE_GROUP rg, KHE_RESOURCE_GROUP not_rg, bool allow_unassign,
  KHE_MONITOR d, KHE_TASK *prev_task, KHE_TASK_SET scratch_ts)
{
  int i, count;  KHE_SOLN soln;  KHE_TASK task;  KHE_RESOURCE r;
  KHE_TIME t;  KHE_TIME_GROUP t_tg;  struct khe_expansion_options_rec eor;
  soln = KheEjectorSoln(ej);
  HnAssert(rg != NULL, "KheEventResourceMoveAugment internal error 1");
  KheEjectorExpansionOptions(ej, &eor);
  /* nocost_off = KheEjectorNoCostOff(ej); */
  count = KheEventResourceTaskCount(soln, er);
  for( i = 0;  i < count;  i++ )
  {
    /* here task is one of the tasks being monitored */
    task = KheEventResourceTask(soln, er, i);
    task = KheTaskProperRoot(task);
    if( (*prev_task == NULL || !KheTaskEquiv(*prev_task, task)) &&
        !KheTaskSatisfies(task, rg, not_rg, allow_unassign) )
    {
      /* try moving the task to a resource in its domain */
      if( KheTaskMoveAugment(ej, task, rg, not_rg, allow_unassign, &eor,
	  NULL, d, scratch_ts) )
	return true;

      /* if the domain is empty, try a double move of another task */
      r = KheTaskAsstResource(task);
      t = KheMeetAsstTime(KheTaskMeet(task));
      if( allow_unassign && t != NULL && r != NULL &&
	  KheResourceGroupResourceCount(rg) == 0 )
      {
	t_tg = KheTimeSingletonTimeGroup(t);
	if( KheResourceGainTaskAugment(ej, r, t_tg, true, scratch_ts) )
	  return true;

	/* *** replaced by KheResourceGainTaskAugment
 	rt = KheResourceResourceType(r);
	r_rg = KheResourceSingletonResourceGroup(r);
	et m = KheEjectorEventTimetableMonitor(ej);
	prev_task2 = NULL;
	for( j = 0;  j < KheEventTimetableMonitorTimeMeetCount(etm, t);  j++ )
	{
	  meet = KheEventTimetableMonitorTimeMeet(etm, t, j);
	  for( k = 0;  k < KheMeetTaskCount(meet);  k++ )
	  {
	    task2 = KheMeetTask(meet, k);
	    if( KheTaskResourceType(task2) == rt )
	    {
	      task2 = KheTaskProperRoot(task2);
	      if( KheTaskMoveResourceCheck(task2, r) &&
		  (prev_task2 == NULL || !KheTaskEquiv(prev_task2, task2)) )
	      {
		if( KheTaskMoveAugment(ej, task, r_rg, NULL, false, &eor, NULL,
		      NULL, scratch_ts) )
		  return true;
	      }
	      prev_task2 = task2;
	    }
	  }
	}
	*** */
      }
      *prev_task = task;
    }
  }
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheAssignResourceAugment(KHE_EJECTOR ej, KHE_MONITOR d)             */
/*                                                                           */
/*  Augment function for individual assign resource defects.                 */
/*                                                                           */
/*  This uses calls to KheTaskMoveAugment to try to move each monitored      */
/*  unassigned task to any resource in that task's domain.                   */
/*                                                                           */
/*****************************************************************************/

static bool KheAssignResourceAugment(KHE_EJECTOR ej, KHE_MONITOR d)
{
  KHE_ASSIGN_RESOURCE_MONITOR arm;  KHE_RESOURCE_GROUP domain;
  KHE_TASK pt;  KHE_EVENT_RESOURCE er;  KHE_TASK_SET scratch_ts;
  KheDebugAugmentBegin(DEBUG3, ej, d, "KheAssignResourceAugment");
  HnAssert(KheMonitorTag(d) == KHE_ASSIGN_RESOURCE_MONITOR_TAG,
    "KheAssignResourceAugment internal error 1");
  if( KheEjectorRepairResources(ej) )
  {
    arm = (KHE_ASSIGN_RESOURCE_MONITOR) d;
    er = KheAssignResourceMonitorEventResource(arm);
    domain = KheEventResourceHardDomain(er);
    scratch_ts = KheTaskSetMake(KheEjectorSoln(ej));
    pt = NULL;
    if( KheEventResourceMoveAugment(ej, er, domain, NULL, false, d, &pt,
	  scratch_ts) )
    {
      KheTaskSetDelete(scratch_ts);
      return true;
    }
    KheTaskSetDelete(scratch_ts);
  }
  KheDebugAugmentEnd(DEBUG3, ej, d, "KheAssignResourceAugment", false);
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheAssignResourceGroupAugment(KHE_EJECTOR ej, KHE_MONITOR d)        */
/*                                                                           */
/*  Augment function for groups of assign resource defects.                  */
/*                                                                           */
/*****************************************************************************/

static bool KheAssignResourceGroupAugment(KHE_EJECTOR ej, KHE_MONITOR d)
{
  KHE_GROUP_MONITOR gm;
  HnAssert(KheMonitorTag(d) == KHE_GROUP_MONITOR_TAG,
    "KheAssignResourceGroupAugment internal error 1");
  gm = (KHE_GROUP_MONITOR) d;
  HnAssert(KheGroupMonitorDefectCount(gm) > 0,
    "KheAssignResourceGroupAugment internal error 2");
  return KheAssignResourceAugment(ej, KheGroupMonitorDefect(gm, 0));
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "prefer resources augment functions"                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool KhePreferResourcesAugment(KHE_EJECTOR ej, KHE_MONITOR d)            */
/*                                                                           */
/*  Augment function for individual prefer resources defects.                */
/*                                                                           */
/*  This uses calls to KheTaskMoveAugment to try to move each monitored      */
/*  task assigned an unpreferred resource to a preferred resource.  There    */
/*  is also another kind of repair, which moves the unpreferred resource     */
/*  to elsewhere in the same meet.                                           */
/*                                                                           */
/*****************************************************************************/

static bool KhePreferResourcesAugment(KHE_EJECTOR ej, KHE_MONITOR d)
{
  KHE_PREFER_RESOURCES_CONSTRAINT prc;  KHE_PREFER_RESOURCES_MONITOR prm;
  KHE_EVENT_RESOURCE er;  KHE_TASK pt;  KHE_RESOURCE_GROUP domain;
  KHE_TASK_SET scratch_ts;
  KheDebugAugmentBegin(DEBUG3, ej, d, "KhePreferResourcesAugment");
  HnAssert(KheMonitorTag(d) == KHE_PREFER_RESOURCES_MONITOR_TAG,
    "KhePreferResourcesAugment internal error 1");
  if( KheEjectorRepairResources(ej) )
  {
    prm = (KHE_PREFER_RESOURCES_MONITOR) d;
    er = KhePreferResourcesMonitorEventResource(prm);
    prc = KhePreferResourcesMonitorConstraint(prm);
    domain = KhePreferResourcesConstraintDomain(prc);
    scratch_ts = KheTaskSetMake(KheEjectorSoln(ej));
    pt = NULL;
    if( KheEventResourceMoveAugment(ej, er, domain, NULL, true, d, &pt,
	  scratch_ts) )
    {
      KheTaskSetDelete(scratch_ts);
      return true;
    }
    KheTaskSetDelete(scratch_ts);
  }
  KheDebugAugmentEnd(DEBUG3, ej, d, "KhePreferResourcesAugment", false);
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KhePreferResourcesGroupAugment(KHE_EJECTOR ej, KHE_MONITOR d)       */
/*                                                                           */
/*  Augment function for groups of prefer resources defects.                 */
/*                                                                           */
/*****************************************************************************/

static bool KhePreferResourcesGroupAugment(KHE_EJECTOR ej, KHE_MONITOR d)
{
  KHE_GROUP_MONITOR gm;
  HnAssert(KheMonitorTag(d) == KHE_GROUP_MONITOR_TAG,
    "KhePreferResourcesGroupAugment internal error 1");
  gm = (KHE_GROUP_MONITOR) d;
  HnAssert(KheGroupMonitorDefectCount(gm) > 0,
    "KhePreferResourcesGroupAugment internal error 2");
  return KhePreferResourcesAugment(ej, KheGroupMonitorDefect(gm, 0));
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "avoid split assignments augment functions"                    */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool KheEject ingTaskSetMoveRepair(KHE_EJECTOR ej,                       */
/*    KHE_AVOID_SPLIT_ASSIGNMENTS_MONITOR asam, KHE_RESOURCE from_r1,        */
/*    KHE_RESOURCE from_r2, KHE_RESOURCE r)                                  */
/*                                                                           */
/*  Try an ejecting task-set move (KHE_REPAIR_TASK_SET_MOVE) to              */
/*  r of those tasks monitored by asam which are currently assigned to       */
/*  from_r1 and from_r2.                                                     */
/*                                                                           */
/*****************************************************************************/

#if WITH_OLD_AVOID_SPLIT_REPAIR
static bool KheEject ingTaskSetMoveRepair(KHE_EJECTOR ej,
  KHE_AVOID_SPLIT_ASSIGNMENTS_MONITOR asam, KHE_RESOURCE from_r1,
  KHE_RESOURCE from_r2, KHE_RESOURCE r)
{
  KHE_RESOURCE ar;  KHE_AVOID_SPLIT_ASSIGNMENTS_CONSTRAINT c;
  KHE_TASK task;  int egi, count, j, k, durn;  KHE_TIME_PARTITION tp;
  KHE_EVENT_RESOURCE er;  KHE_SOLN soln;  bool success;
  c = KheAvoidSplitAssignmentsMonitorConstraint(asam);
  egi = KheAvoidSplitAssignmentsMonitorEventGroupIndex(asam);
  count = KheAvoidSplitAssignmentsConstraintEventResourceCount(c, egi);
  soln = KheEjectorSoln(ej);
  KheEjecto rRepairBegin(ej);
  success = true;
  durn = 0;
  tp = KheEjectorTimePartition(ej);
  for( j = 0;  success && j < count;  j++ )
  {
    er = KheAvoidSplitAssignmentsConstraintEventResource(c, egi, j);
    for( k = 0;  success && k < KheEventResourceTaskCount(soln, er);  k++ )
    {
      task = KheEventResourceTask(soln, er, k);
      ar = KheTaskAsstResource(task);
      if( (ar == from_r1 || ar == from_r2) && ar != r )
      {
	success = success && KheEjectingTaskMove(task, r, tp);
	durn += KheTaskDuration(task);
      }
    }
  }
  return KheEjector RepairEnd(ej, KHE_REPAIR_TASK_SET_MOVE, success);
}
#endif


/*****************************************************************************/
/*                                                                           */
/*  KHE_RESOURCE_GROUP KheSplitTasksUnassignResourceGroup(                   */
/*    KHE_AVOID_SPLIT_ASSIGNMENTS_MONITOR asam, KHE_RESOURCE r)              */
/*                                                                           */
/*  Return a resource group consisting of the resources currently assigned   */
/*  to the tasks monitored by asam, excluding r.                             */
/*                                                                           */
/*****************************************************************************/

#if WITH_OLD_AVOID_SPLIT_REPAIR
#else
static KHE_RESOURCE_GROUP KheSplitTasksUnassignResourceGroup(
  KHE_AVOID_SPLIT_ASSIGNMENTS_MONITOR asam, KHE_RESOURCE r)
{
  KHE_SOLN soln;  int i;  KHE_RESOURCE r2;
  soln = KheMonitorSoln((KHE_MONITOR) asam);
  KheSolnResourceGroupBegin(soln, KheResourceResourceType(r));
  for( i = 0;  i < KheAvoidSplitAssignmentsMonitorResourceCount(asam);  i++ )
  {
    r2 = KheAvoidSplitAssignmentsMonitorResource(asam, i);
    if( r2 != r )
      KheSolnResourceGroupAddResource(soln, r2);
  }
  return KheSolnResourceGroupEnd(soln);
}
#endif


/*****************************************************************************/
/*                                                                           */
/*  void KheSplitTasksUnassignOnSuccess(void *on_success_val)                */
/*                                                                           */
/*  On-success function used when repairing avoid split assignments defects. */
/*                                                                           */
/*****************************************************************************/

#if WITH_OLD_AVOID_SPLIT_REPAIR
#else
static void KheSplitTasksUnassignOnSuccess(void *on_success_val)
{
  KHE_TASK_BOUND tb;  bool success;
  if( DEBUG13 )
    fprintf(stderr, "[ KheSplitTasksUnassignOnSuccess\n");
  tb = (KHE_TASK_BOUND) on_success_val;
  success = KheTaskBoundDelete(tb);
  if( DEBUG13 )
    fprintf(stderr, "] KheSplitTasksUnassignOnSuccess(success = %s)\n",
      success ? "true" : "false");
}
#endif


/*****************************************************************************/
/*                                                                           */
/*  void KheSplitTasksUnassignRepair(KHE_EJECTOR ej,                         */
/*    KHE_AVOID_SPLIT_ASSIGNMENTS_MONITOR asam, KHE_RESOURCE r)              */
/*                                                                           */
/*  Repair which tries unassigning all tasks monitored by asam that are      */
/*  assigned r, and reducing the domains of all tasks monitored by asam      */
/*  to exclude r.                                                            */
/*                                                                           */
/*****************************************************************************/

#if WITH_OLD_AVOID_SPLIT_REPAIR
#else
static bool KheSplitTasksUnassignRepair(KHE_EJECTOR ej,
  KHE_AVOID_SPLIT_ASSIGNMENTS_MONITOR asam, KHE_RESOURCE r)
{
  KHE_SOLN soln;  KHE_RESOURCE_GROUP avoid_splits_rg;  KHE_TASK task;
  int egi, count, i, j;  KHE_TASK_BOUND tb;
  KHE_AVOID_SPLIT_ASSIGNMENTS_CONSTRAINT c;  KHE_EVENT_RESOURCE er;
  KheEjectorRepairBegin(ej);
  soln = KheEjectorSoln(ej);
  avoid_splits_rg = KheSplitTasksUnassignResourceGroup(asam, r);
  tb = KheTaskBoundMake(soln, avoid_splits_rg);
  c = KheAvoidSplitAssignmentsMonitorConstraint(asam);
  egi = KheAvoidSplitAssignmentsMonitorEventGroupIndex(asam);
  count = KheAvoidSplitAssignmentsConstraintEventResourceCount(c, egi);
  for( i = 0;  i < count;  i++ )
  {
    er = KheAvoidSplitAssignmentsConstraintEventResource(c, egi, i);
    for( j = 0;  j < KheEventResourceTaskCount(soln, er);  j++ )
    {
      task = KheEventResourceTask(soln, er, j);
      if( KheTaskAsstResource(task) == r && !KheTaskUnAssign(task) )
	return KheEjectorRepairEnd(ej, KHE_REPAIR_SPLIT_TASKS_UNASSIGN, false);
      if( !KheTaskAddTaskBound(task, tb) )
	return KheEjectorRepairEnd(ej, KHE_REPAIR_SPLIT_TASKS_UNASSIGN, false);
    }
  }
  return KheEjectorRepairEndLong(ej, KHE_REPAIR_SPLIT_TASKS_UNASSIGN, true,
    INT_MAX, false, &KheSplitTasksUnassignOnSuccess, (void *) tb);
}
#endif


/*****************************************************************************/
/*                                                                           */
/*  bool KheAssignedTasksVisited(KHE_AVOID_SPLIT_ASSIGNMENTS_MONITOR asam,   */
/*    int slack)                                                             */
/*                                                                           */
/*  Like KheTaskVisited(task, slack) only applied to all the assigned        */
/*  tasks monitored by asam; it returns true if any of them are visited.     */
/*                                                                           */
/*****************************************************************************/

#if WITH_OLD_AVOID_SPLIT_REPAIR
static bool KheAssignedTasksVisited(KHE_AVOID_SPLIT_ASSIGNMENTS_MONITOR asam,
  int slack)
{
  KHE_AVOID_SPLIT_ASSIGNMENTS_CONSTRAINT c;  int egi, count, i, j;
  KHE_EVENT_RESOURCE er;  KHE_SOLN soln;  KHE_TASK task;
  c = KheAvoidSplitAssignmentsMonitorConstraint(asam);
  egi = KheAvoidSplitAssignmentsMonitorEventGroupIndex(asam);
  count = KheAvoidSplitAssignmentsConstraintEventResourceCount(c, egi);
  soln = KheMonitorSoln((KHE_MONITOR) asam);
  for( i = 0;  i < count;  i++ )
  {
    er = KheAvoidSplitAssignmentsConstraintEventResource(c, egi, i);
    for( j = 0;  j < KheEventResourceTaskCount(soln, er);  j++ )
    {
      task = KheTaskFirstUnFixed(KheEventResourceTask(soln, er, j));
      if( task == NULL )
	return false;
      if( KheTaskAsstResource(task) != NULL && KheTaskVisited(task, slack) )
	return true;
    }
  }
  return false;
}
#endif


/*****************************************************************************/
/*                                                                           */
/*  void KheAssignedTasksVisit(KHE_AVOID_SPLIT_ASSIGNMENTS_MONITOR asam)     */
/*                                                                           */
/*  Like KheTaskVisit(task) only applied to all the assigned tasks           */
/*  monitored by asam.                                                       */
/*                                                                           */
/*****************************************************************************/

#if WITH_OLD_AVOID_SPLIT_REPAIR
static void KheAssignedTasksVisit(KHE_AVOID_SPLIT_ASSIGNMENTS_MONITOR asam)
{
  KHE_AVOID_SPLIT_ASSIGNMENTS_CONSTRAINT c;  int egi, count, i, j;
  KHE_EVENT_RESOURCE er;  KHE_SOLN soln;  KHE_TASK task;
  c = KheAvoidSplitAssignmentsMonitorConstraint(asam);
  egi = KheAvoidSplitAssignmentsMonitorEventGroupIndex(asam);
  count = KheAvoidSplitAssignmentsConstraintEventResourceCount(c, egi);
  soln = KheMonitorSoln((KHE_MONITOR) asam);
  for( i = 0;  i < count;  i++ )
  {
    er = KheAvoidSplitAssignmentsConstraintEventResource(c, egi, i);
    for( j = 0;  j < KheEventResourceTaskCount(soln, er);  j++ )
    {
      task = KheTaskFirstUnFixed(KheEventResourceTask(soln, er, j));
      if( task != NULL && KheTaskAsstResource(task) != NULL )
	KheTaskVisit(task);
    }
  }
}
#endif


/*****************************************************************************/
/*                                                                           */
/*  void KheAssignedTasksUnVisit(KHE_AVOID_SPLIT_ASSIGNMENTS_MONITOR asam)   */
/*                                                                           */
/*  Like KheTaskUnVisit(task) only applied to all the assigned tasks         */
/*  monitored by asam.                                                       */
/*                                                                           */
/*****************************************************************************/

#if WITH_OLD_AVOID_SPLIT_REPAIR
static void KheAssignedTasksUnVisit(KHE_AVOID_SPLIT_ASSIGNMENTS_MONITOR asam)
{
  KHE_AVOID_SPLIT_ASSIGNMENTS_CONSTRAINT c;  int egi, count, i, j;
  KHE_EVENT_RESOURCE er;  KHE_SOLN soln;  KHE_TASK task;
  c = KheAvoidSplitAssignmentsMonitorConstraint(asam);
  egi = KheAvoidSplitAssignmentsMonitorEventGroupIndex(asam);
  count = KheAvoidSplitAssignmentsConstraintEventResourceCount(c, egi);
  soln = KheMonitorSoln((KHE_MONITOR) asam);
  for( i = 0;  i < count;  i++ )
  {
    er = KheAvoidSplitAssignmentsConstraintEventResource(c, egi, i);
    for( j = 0;  j < KheEventResourceTaskCount(soln, er);  j++ )
    {
      task = KheTaskFirstUnFixed(KheEventResourceTask(soln, er, j));
      if( task != NULL && KheTaskAsstResource(task) != NULL )
	KheTaskUnVisit(task);
    }
  }
}
#endif


/*****************************************************************************/
/*                                                                           */
/*  KHE_TASK KheAnyTaskAssignedToResource(KHE_RESOURCE r,                    */
/*    KHE_AVOID_SPLIT_ASSIGNMENTS_MONITOR asam)                              */
/*                                                                           */
/*  Return any task monitored by asam assigned r.  There must be one.        */
/*                                                                           */
/*****************************************************************************/

#if WITH_OLD_AVOID_SPLIT_REPAIR
static KHE_TASK KheAnyTaskAssignedToResource(KHE_RESOURCE r,
  KHE_AVOID_SPLIT_ASSIGNMENTS_MONITOR asam)
{
  KHE_AVOID_SPLIT_ASSIGNMENTS_CONSTRAINT c;  int egi, count, i, j;
  KHE_EVENT_RESOURCE er;  KHE_SOLN soln;  KHE_TASK task;
  c = KheAvoidSplitAssignmentsMonitorConstraint(asam);
  egi = KheAvoidSplitAssignmentsMonitorEventGroupIndex(asam);
  count = KheAvoidSplitAssignmentsConstraintEventResourceCount(c, egi);
  soln = KheMonitorSoln((KHE_MONITOR) asam);
  for( i = 0;  i < count;  i++ )
  {
    er = KheAvoidSplitAssignmentsConstraintEventResource(c, egi, i);
    for( j = 0;  j < KheEventResourceTaskCount(soln, er);  j++ )
    {
      task = KheTaskFirstUnFixed(KheEventResourceTask(soln, er, j));
      if( task != NULL && KheTaskAsstResource(task) == r )
	return task;
    }
  }
  HnAbort("KheAnyTaskAssignedToResource internal error");
  return NULL;  /* keep compiler happy */
}
#endif


/*****************************************************************************/
/*                                                                           */
/*  bool KheAvoidSplitAssignmentsAugment(KHE_EJECTOR ej, KHE_MONITOR d)      */
/*                                                                           */
/*  Augment function for individual avoid split assignments defects.         */
/*  There is no group version of this function.                              */
/*                                                                           */
/*  If KheEjectorRepairResources(ej), then for each pair of resources        */
/*  assigned to monitored tasks, and for each resource in the domain of      */
/*  one of those tasks, try a set of ejecting task moves of those tasks to   */
/*  that resource.                                                           */
/*                                                                           */
/*  Also, if KheEjectorRepairTimes(ej) as well, then for each resource       */
/*  assigned to exactly one of the monitored tasks, try moving that          */
/*  task's meet to a different time, and simultaneously moving the task      */
/*  to one of the other resources assigned to a monitored task.              */
/*                                                                           */
/*****************************************************************************/

#if WITH_OLD_AVOID_SPLIT_REPAIR
static bool KheAvoidSplitAssignmentsAugment(KHE_EJECTOR ej, KHE_MONITOR d)
{
  KHE_AVOID_SPLIT_ASSIGNMENTS_MONITOR asam;  KHE_TASK task;
  int i, j, k;  KHE_RESOURCE r1, r2, r;  KHE_RESOURCE_GROUP rg;
  KheDebugAugmentBegin(DEBUG5, ej, d, "KheAvoidSplitAssignmentsAugment");
  if( KheEjectorRepairResources(ej) )
  {
    asam = (KHE_AVOID_SPLIT_ASSIGNMENTS_MONITOR) d;
    if( !KheAssignedTasksVisited(asam, 0) )
    {
      KheAssignedTasksVisit(asam);

      /* try ejecting task-set moves */
      for( i = 0; i < KheAvoidSplitAssignmentsMonitorResourceCount(asam); i++ )
      {
	r1 = KheAvoidSplitAssignmentsMonitorResource(asam, i);
	rg = KheTaskDomain(KheAnyTaskAssignedToResource(r1, asam));
	for( j=i+1; j<KheAvoidSplitAssignmentsMonitorResourceCount(asam); j++ )
	{
	  r2 = KheAvoidSplitAssignmentsMonitorResource(asam, j);
	  for( k = 0;  k < KheResourceGroupResourceCount(rg);  k++ )
	  {
	    r = KheResourceGroupResource(rg, k);
	    if( DEBUG12 )
	      fprintf(stderr, "  trying %s\n",
		KheResourceId(r) == NULL ? "@" : KheResourceId(r));
	    if( KheEjectingTaskSe tMoveRepair(ej, asam, r1, r2, r) )
	    {
	      KheDebugAugmentEnd(DEBUG5, ej, d,
		"KheAvoidSplitAssignmentsAugment", true);
	      return true;
	    }
	  }
	}
      }

      /* try meet/task moves */
      if( KheEjectorRepairTimes(ej) )
      for( i = 0; i < KheAvoidSplitAssignmentsMonitorResourceCount(asam); i++ )
      {
	if( KheAvoidSplitAssignmentsMonitorResourceMultiplicity(asam, i) == 1 )
	{
	  /* r1 occurs in only one task, try moving it to any other resource */
	  r1 = KheAvoidSplitAssignmentsMonitorResource(asam, i);
	  task = KheAnyTaskAssignedToResource(r1, asam);
	  HnAssert(task!=NULL, "KheAvoidSplitAssignmentsAugment internal error");
	  for( j=0; j<KheAvoidSplitAssignmentsMonitorResourceCount(asam); j++ )
	  {
	    r2 = KheAvoidSplitAssignmentsMonitorResource(asam, j);
	    if(r2!=r1 && KheKem peMeetMoveAndTaskMoveAugment(ej,
		  task, r2, KHE_MOVE_EJECTING))
	    {
	      KheDebugAugmentEnd(DEBUG5, ej, d,
		"KheAvoidSplitAssignmentsAugment", true);
	      return true;
	    }
	  }
	}
      }

      if( KheEjectorCurrMayRevisit(ej) )
	KheAssignedTasksUnVisit(asam);
    }
  }
  KheDebugAugmentEnd(DEBUG5, ej, d, "KheAvoidSplitAssignmentsAugment", false);
  return false;
}
#else
static bool KheAvoidSplitAssignmentsAugment(KHE_EJECTOR ej, KHE_MONITOR d)
{
  KHE_AVOID_SPLIT_ASSIGNMENTS_MONITOR asam;  int i;  KHE_RESOURCE r;
  /* bool save_ejector_repair_times; */
  KheDebugAugmentBegin(DEBUG15, ej, d, "KheAvoidSplitAssignmentsAugment");
  if( KheEjectorRepairResources(ej) )
  {
    asam = (KHE_AVOID_SPLIT_ASSIGNMENTS_MONITOR) d;
    /* *** interesting idea, didn't help much
    save_ejector_repair_times = KheEjectorRepairTimes(ej);
    if( KheEjectorCurrDepth(ej) == 1 )
      KheOptionsSetEjectorRepairTimes(KheEjectorOptions(ej), true);
    *** */
    for( i = 0;  i < KheAvoidSplitAssignmentsMonitorResourceCount(asam);  i++ )
    {
      r = KheAvoidSplitAssignmentsMonitorResource(asam, i);
      if( KheEjectorCurrDepth(ej) == 1 ||
	  KheAvoidSplitAssignmentsMonitorResourceMultiplicity(asam, i) == 1 )
      {
	if( KheEjectorCurrDepth(ej) == 1 )
	  KheSolnNewGlobalVisit(KheEjectorSoln(ej));
	if( KheSplitTasksUnassignRepair(ej, asam, r) )
	{
	  if( DEBUG15 )
	  {
	    fprintf(stderr,
	      "%*s] KheAvoidSplitAssignmentsAugment ret true%s:\n",
	      KheEjectorCurrDebugIndent(ej) + 2, "",
	      KheEjectorCurrDepth(ej) == 1 ? " 1" : "");
	      KheMonitorDebug(d, 1, KheEjectorCurrDebugIndent(ej) + 4, stderr);
	  }
	  /* ***
	  KheOptionsSetEjectorRepairTimes(KheEjectorOptions(ej),
	    save_ejector_repair_times);
	  *** */
	  return true;
	}
      }
    }
    /* ***
    KheOptionsSetEjectorRepairTimes(KheEjectorOptions(ej),
      save_ejector_repair_times);
    *** */
  }
  KheDebugAugmentEnd(DEBUG15, ej, d, "KheAvoidSplitAssignmentsAugment", false);
  return false;
}
#endif


/*****************************************************************************/
/*                                                                           */
/*  Submodule "limit resources augment functions"                            */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool KheEventResourceEquiv(KHE_EVENT_RESOURCE er1,                       */
/*    KHE_EVENT_RESOURCE er2)                                                */
/*                                                                           */
/*  Return true if event resources er1 and er2 are equivalent, meaning       */
/*  that we don't have to try augments from both.                            */
/*                                                                           */
/*****************************************************************************/

/* *** obsolete, we are comparing tasks for equivalence now
static bool KheEventResourceEquiv(KHE_SOLN soln, KHE_EVENT_RESOURCE er1,
  KHE_EVENT_RESOURCE er2)
{
  KHE_TASK task1, task2;  KHE_RESOURCE r1, r2;

  ** make sure they are equivalent in the instance and both have one task **
  if( KheEventResourceEquivalent(er1, er2) &&
      KheEventResourceTaskCount(soln, er1) == 1 &&
      KheEventResourceTaskCount(soln, er2) == 1 )
  {
    task1 = KheEventResourceTask(soln, er1, 0);
    task2 = KheEventResourceTask(soln, er2, 0);
    r1 = KheTaskAsstResource(task1);
    r2 = KheTaskAsstResource(task2);
    if( DEBUG31 )
    {
      fprintf(stderr, "  KheEventResourceEquiv(");
      KheEventResourceDebug(er1, 1, -1, stderr);
      fprintf(stderr, ", ");
      KheEventResourceDebug(er2, 1, -1, stderr);
      if( r1 == NULL && r2 == NULL )
	fprintf(stderr, ") returning true\n");
      else
	fprintf(stderr, ") returning false (task1 := %s, task2 := %s)\n",
	  r1 == NULL ? "-" : KheResourceId(r1),
	  r2 == NULL ? "-" : KheResourceId(r2));
    }
    return r1 == NULL && r2 == NULL;
  }
  else
    return false;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheLimitResourcesAugment(KHE_EJECTOR ej, KHE_MONITOR d)             */
/*                                                                           */
/*  Augment function for individual limit resources defects.                 */
/*                                                                           */
/*  If the defect is an underload, use KheTaskMoveAugment to try to move     */
/*  each task which is either unassigned or contains an unpreferred          */
/*  resource to a resource in the domain of the constraint.  Unassigned      */
/*  tasks are tried first.                                                   */
/*                                                                           */
/*  If the defect is an overload, use KheTaskMoveAugment to try to move      */
/*  each task which is currently assigned a resource from the domain to a    */
/*  resource from outside the domain (including unassignment).               */
/*                                                                           */
/*****************************************************************************/

static bool KheLimitResourcesAugment(KHE_EJECTOR ej, KHE_MONITOR d)
{
  KHE_LIMIT_RESOURCES_CONSTRAINT c;  KHE_LIMIT_RESOURCES_MONITOR lrm;
  KHE_TASK pt;  int eg_index, i, count, maximum, minimum, active_durn;
  KHE_RESOURCE_GROUP domain, rg;  KHE_EVENT_RESOURCE er;
  KHE_TASK_SET scratch_ts;
  KheDebugAugmentBegin(DEBUG15, ej, d, "KheLimitResourcesAugment");
  HnAssert(KheMonitorTag(d) == KHE_LIMIT_RESOURCES_MONITOR_TAG,
    "KheLimitResourcesAugment internal error 1");
  lrm = (KHE_LIMIT_RESOURCES_MONITOR) d;
  c = KheLimitResourcesMonitorConstraint(lrm);
  eg_index = KheLimitResourcesMonitorEventGroupIndex(lrm);
  count = KheLimitResourcesConstraintEventResourceCount(c, eg_index);
  domain = KheLimitResourcesConstraintDomain(c);
  if( KheEjectorRepairResources(ej) )
  {
    KheLimitResourcesMonitorActiveDuration(lrm, &minimum,&maximum,&active_durn);
    if( active_durn < minimum )
    {
      /* too few resources from c's domain */
      scratch_ts = KheTaskSetMake(KheEjectorSoln(ej));
      pt = NULL;
      for( i = 0;  i < count;  i++ )
      {
	er = KheLimitResourcesConstraintEventResource(c, eg_index, i);
        if( KheEventResourceMoveAugment(ej, er, domain, NULL, false, d, &pt,
	      scratch_ts) )
	{
	  KheTaskSetDelete(scratch_ts);
	  KheDebugAugmentEnd(DEBUG15, ej, d, "KheLimitResourcesAugment",true);
	  return true;
	}
      }
      KheTaskSetDelete(scratch_ts);
    }
    else if( active_durn > maximum )
    {
      /* too many resources from c's domain */
      scratch_ts = KheTaskSetMake(KheEjectorSoln(ej));
      pt = NULL;
      for( i = 0;  i < count;  i++ )
      {
	er = KheLimitResourcesConstraintEventResource(c, eg_index, i);
	rg = KheEventResourceHardDomain(er);
        if( KheEventResourceMoveAugment(ej, er, rg, domain, true, d, &pt,
	      scratch_ts) )
	{
	  KheTaskSetDelete(scratch_ts);
	  KheDebugAugmentEnd(DEBUG15, ej, d, "KheLimitResourcesAugment", true);
	  return true;
	}
      }
      KheTaskSetDelete(scratch_ts);
    }
    else
      HnAbort("KheLimitResourcesAugment internal error 2");
  }
  KheDebugAugmentEnd(DEBUG15, ej, d, "KheLimitResourcesAugment", false);
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheLimitResourcesGroupAugment(KHE_EJECTOR ej, KHE_MONITOR d)        */
/*                                                                           */
/*  Augment function for groups of limit resources defects.                  */
/*                                                                           */
/*  These are not necessarily equivalent so we search for one with           */
/*  non-zero cost and repair that.                                           */
/*                                                                           */
/*****************************************************************************/

static bool KheLimitResourcesGroupAugment(KHE_EJECTOR ej, KHE_MONITOR d)
{
  KHE_GROUP_MONITOR gm;  KHE_MONITOR m;  int i;
  HnAssert(KheMonitorTag(d) == KHE_GROUP_MONITOR_TAG,
    "KheLimitResourcesGroupAugment internal error 1");
  gm = (KHE_GROUP_MONITOR) d;
  HnAssert(KheGroupMonitorDefectCount(gm) > 0,
    "KheLimitResourcesGroupAugment internal error 2");
  for( i = 0;  i < KheGroupMonitorChildMonitorCount(gm);  i++ )
  {
    m = KheGroupMonitorChildMonitor(gm, i);
    if( KheMonitorCost(m) > 0 && KheLimitResourcesAugment(ej, m) )
      return true;
  }
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "avoid clashes augment functions"                              */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool KheAvoidClashesAugment(KHE_EJECTOR ej, KHE_MONITOR d)               */
/*                                                                           */
/*  Augment function for individual avoid clashes defects.                   */
/*                                                                           */
/*  The current plan is to detach all avoid clashes monitors during          */
/*  repair, since their work is done (and done better) by demand monitors.   */
/*  So this function aborts if it is called.                                 */
/*                                                                           */
/*  Old specification                                                        */
/*  -----------------                                                        */
/*  For each clashing task, try moving it to any other resource in its       */
/*  domain.  This function expects to be called only during resource repair, */
/*  because during time repair avoid clashes monitors are usually detached.  */
/*  Even during resource repair it would probably be better to ignore these  */
/*  defects, since they are basically unrepairable then anyway.              */
/*                                                                           */
/*****************************************************************************/

static bool KheAvoidClashesAugment(KHE_EJECTOR ej, KHE_MONITOR d)
{
  KheDebugAugmentBegin(DEBUG3, ej, d, "KheAvoidClashesAugment");
  HnAbort("KheAvoidClashesAugment: unexpected call");
  KheDebugAugmentEnd(DEBUG3, ej, d, "KheAvoidClashesAugment", false);
  return false;  /* keep compiler happy */

  /* boilerplate */
  /* ***
  HnAssert(KheMonitorTag(d) == KHE_AVOID_CLASHES_MONITOR_TAG,
    "KheAvoidClashesAugment internal error 1");
  soln = KheEjectorSoln(ej);
  acm = (KHE_AVOID_CLASHES_MONITOR) d;
  r = KheAvoidClashesMonitorResource(acm);
  tm = KheResourceTimetableMonitor(soln, r);

  ** find each clashing task and reassign it in all possible ways **
  for( i = 0;  i < KheTimetableMonitorClashingTimeCount(tm);  i++ )
  {
    t = KheTimetableMonitorClashingTime(tm, i);
    for( j = 0;  j < KheTimetableMonitorTimeMeetCount(tm, t);  j++ )
    {
      meet = KheTimetableMonitorTimeMeet(tm, t, j);
      if(KheMeetEjectingTaskMoveAugment(ej, meet, r) )
      {
	KheDebugAugmentEnd(DEBUG3, ej, d, "KheAvoidClashesAugment", true);
	return;
      }
    }
  }

  ** wrapup **
  KheDebugAugmentEnd(DEBUG3, ej, d, "KheAvoidClashesAugment", false);
  *** */
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheAvoidClashesGroupAugment(KHE_EJECTOR ej, KHE_MONITOR d)          */
/*                                                                           */
/*  Augment function for groups of avoid clashes defects.                    */
/*                                                                           */
/*****************************************************************************/

static bool KheAvoidClashesGroupAugment(KHE_EJECTOR ej, KHE_MONITOR d)
{
  KHE_GROUP_MONITOR gm;
  HnAssert(KheMonitorTag(d) == KHE_GROUP_MONITOR_TAG,
    "KheAvoidClashesGroupAugment internal error 1");
  gm = (KHE_GROUP_MONITOR) d;
  HnAssert(KheGroupMonitorDefectCount(gm) > 0,
    "KheAvoidClashesGroupAugment internal error 2");
  return KheAvoidClashesAugment(ej, KheGroupMonitorDefect(gm, 0));
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "avoid unavailable times augment functions"                    */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool KheAvoidUnavailableTimesAugment(KHE_EJECTOR ej, KHE_MONITOR d)      */
/*                                                                           */
/*  Augment function for individual avoid unavailable times defects.         */
/*                                                                           */
/*  Try repairs documented in the header of KheResourceOverloadAugment,      */
/*  which in this case amount to task moves away from d's resource and       */
/*  meet moves away from unavailable times.                                  */
/*                                                                           */
/*****************************************************************************/
/* ***
static bool KheNurseOverloadAugment(KHE_EJECTOR ej, KHE_RESOURCE r,
  KHE_TIME_GROUP tg, KHE_MONITOR d);
*** */

static bool KheAvoidUnavailableTimesAugment(KHE_EJECTOR ej, KHE_MONITOR d)
{
  KHE_AVOID_UNAVAILABLE_TIMES_MONITOR autm;  KHE_TIME_GROUP domain;
  KHE_RESOURCE r; KHE_AVOID_UNAVAILABLE_TIMES_CONSTRAINT autc;  bool res;
  KheDebugAugmentBegin(DEBUG4, ej, d, "KheAvoidUnavailableTimesAugment");
  HnAssert(KheMonitorTag(d) == KHE_AVOID_UNAVAILABLE_TIMES_MONITOR_TAG,
    "KheAvoidUnavailableTimesAugment internal error 1");
  autm = (KHE_AVOID_UNAVAILABLE_TIMES_MONITOR) d;
  autc = KheAvoidUnavailableTimesMonitorConstraint(autm);
  domain = KheAvoidUnavailableTimesConstraintUnavailableTimes(autc);
  r = KheAvoidUnavailableTimesMonitorResource(autm);
  /* ***
  if( KheInstanceModel(KheResourceInstance(r)) == KHE_MODEL_EMPLOYEE_SCHEDULE )
    res = KheNurseOverloadAugment(ej, r, domain, d);
  else
  *** */
  res = KheResourceOverloadAugment(ej, r, domain, false);
  KheDebugAugmentEnd(DEBUG4, ej, d, "KheAvoidUnavailableTimesAugment", res);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheAvoidUnavailableTimesGroupAugment(KHE_EJECTOR ej, KHE_MONITOR d) */
/*                                                                           */
/*  Augment function for groups of avoid unavailable times defects.          */
/*                                                                           */
/*****************************************************************************/

static bool KheAvoidUnavailableTimesGroupAugment(KHE_EJECTOR ej, KHE_MONITOR d)
{
  KHE_GROUP_MONITOR gm;
  HnAssert(KheMonitorTag(d) == KHE_GROUP_MONITOR_TAG,
    "KheAvoidUnavailableTimesGroupAugment internal error 1");
  gm = (KHE_GROUP_MONITOR) d;
  HnAssert(KheGroupMonitorDefectCount(gm) > 0,
    "KheAvoidUnavailableTimesGroupAugment internal error 2");
  return KheAvoidUnavailableTimesAugment(ej, KheGroupMonitorDefect(gm, 0));
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "limit idle times augments (simple)"                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool KheLimitIdleTimeWillBeBusy(KHE_TIME time,                           */
/*    KHE_TIMETABLE_MONITOR tm, KHE_MEET meet, KHE_TIME new_time)            */
/*                                                                           */
/*  Return true if time will be busy after meet moves to new_time.           */
/*                                                                           */
/*****************************************************************************/

#if WITH_OLD_LIMIT_IDLE_REPAIR
static bool KheLimitIdleTimeWillBeBusy(KHE_TIME time,
  KHE_RESOURCE_TIMETABLE_MONITOR rtm, KHE_MEET meet, KHE_TIME new_time)
{
  int i;  KHE_MEET meet2;

  /* time will be busy if there is already some other meet at time */
  for( i = 0;  i < KheResourceTimetableMonitorTimeTaskCount(rtm, time);  i++ )
  {
    meet2 = KheTaskMeet(KheResourceTimetableMonitorTimeTask(rtm, time, i));
    if( meet2 != meet )
      return true;
  }

  /* alternatively, time will be busy if meet will overlap time */
  return KheTimeIntervalsOverlap(new_time, KheMeetDuration(meet), time, 1) > 0;
}
#endif


/*****************************************************************************/
/*                                                                           */
/*  int KheLimitNewIdle(KHE_TIME_GROUP tg, KHE_TIMETABLE_MONITOR tm,         */
/*    KHE_MEET meet, KHE_TIME new_time)                                      */
/*                                                                           */
/*  Return the number of idle times there will be in tg after meet           */
/*  moves from where it is now to new_time.                                  */
/*                                                                           */
/*****************************************************************************/

#if WITH_OLD_LIMIT_IDLE_REPAIR
static int KheLimitNewIdle(KHE_TIME_GROUP tg,
  KHE_RESOURCE_TIMETABLE_MONITOR rtm, KHE_MEET meet, KHE_TIME new_time)
{
  KHE_TIME time, first_busy, last_busy;  int i, busy_count;

  /* find the number of and first and last busy times after meet moves */
  first_busy = last_busy = NULL;
  busy_count = 0;
  for( i = 0;  i < KheTimeGroupTimeCount(tg);  i++ )
  {
    time = KheTimeGroupTime(tg, i);
    if( KheLimitIdleTimeWillBeBusy(time, rtm, meet, new_time) )
    {
      busy_count++;
      if( first_busy == NULL )
	first_busy = time;
      last_busy = time;
    }
  }

  /* use the usual formula to calculate the number of idle times */
  return busy_count == 0 ? 0 :
    (KheTimeIndex(last_busy) - KheTimeIndex(first_busy) + 1) - busy_count;
}
#endif


/*****************************************************************************/
/*                                                                           */
/*  bool KheLimitIdleMeetMoveFn(KHE_MEET meet, KHE_TIME time, void *impl)    */
/*                                                                           */
/*  Return true if moving meet from where it is now to new_time would        */
/*  decrease the number of idle times, according to litm (passed in impl).   */
/*                                                                           */
/*****************************************************************************/

#if WITH_OLD_LIMIT_IDLE_REPAIR
static bool KheLimitIdleMeetMoveFn(KHE_MEET meet, KHE_TIME time, void *impl)
{
  int tg_durn, durn, i, delta, busy_count, idle_count, count;
  KHE_LIMIT_IDLE_TIMES_MONITOR litm;  KHE_TIME times[2];
  KHE_TIME_GROUP tg;  KHE_RESOURCE_TIMETABLE_MONITOR rtm;
  KHE_TIME tg_time, old_time;
  litm = (KHE_LIMIT_IDLE_TIMES_MONITOR) impl;
  rtm = KheResourceTimetableMonitor(KheMeetSoln(meet),
    KheLimitIdleTimesMonitorResource(litm));
  durn = KheMeetDuration(meet);
  old_time = KheMeetAsstTime(meet);
  delta = 0;
  for( i = 0;  i < KheLimitIdleTimesMonitorTimeGroupCount(litm);  i++ )
  {
    tg = KheLimitIdleTimesMonitorTimeGroupState(litm, i, &busy_count,
      &idle_count, times, &count);
    tg_time = KheTimeGroupTime(tg, 0);
    tg_durn = KheTimeGroupTimeCount(tg);
    if( KheTimeIntervalsOverlap(old_time, durn, tg_time, tg_durn) ||
	KheTimeIntervalsOverlap(time, durn, tg_time, tg_durn) )
      delta += KheLimitNewIdle(tg, rtm, meet, time) - idle_count;
  }
  return delta < 0;
}
#endif


/*****************************************************************************/
/*                                                                           */
/*  Submodule "limit idle times augments (complex)"                          */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_LIT_SOLVER - solver for complex limit idle times repairs             */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_lit_solver_rec {
  HA_ARENA			arena;		/* arena                     */
  KHE_EJECTOR			ejector;	/* the ejector               */
  KHE_LIMIT_IDLE_TIMES_MONITOR	litm;		/* the defect                */
  KHE_TIME_GROUP		tg;		/* defective time group      */
  KHE_RESOURCE			resource;	/* litm's resource           */
  KHE_SOLN			soln;		/* litm's soln               */
  ARRAY_KHE_MEET		meets;		/* the meets of tgm          */
  int				total_durn;	/* total duration of meets   */
  int				asst_count;	/* number of meets assigned  */
  int				asst_problems;	/* number of bad assignments */
  int				repair_count;	/* number repairs            */
  KHE_TIME			curr_start_time; /* start time, currently    */
} *KHE_LIT_SOLVER;


/*****************************************************************************/
/*                                                                           */
/*  KHE_LIT_SOLVER KheLitSolverMake(KHE_EJECTOR ej,                          */
/*    KHE_LIMIT_IDLE_TIMES_MONITOR litm, KHE_TIME_GROUP_MONITOR tgm)         */
/*                                                                           */
/*  Make and return a new lit solver with these attributes.                  */
/*                                                                           */
/*****************************************************************************/

static KHE_LIT_SOLVER KheLitSolverMake(KHE_EJECTOR ej,
  KHE_LIMIT_IDLE_TIMES_MONITOR litm, KHE_TIME_GROUP tg)
{
  KHE_LIT_SOLVER res;  KHE_RESOURCE_TIMETABLE_MONITOR rtm;  KHE_SOLN soln;
  KHE_TIME time;  int i, j, pos;  KHE_MEET meet;  HA_ARENA a;

  /* make the basic object */
  soln = KheEjectorSoln(ej);
  a = KheSolnArenaBegin(soln);
  /* a = HaAre naMake(); */
  HaMake(res, a);
  res->arena = a;
  res->ejector = ej;
  res->litm = litm;
  res->tg = tg;
  res->resource = KheLimitIdleTimesMonitorResource(litm);
  res->soln = soln;
  HaArrayInit(res->meets, a);
  res->total_durn = 0;
  res->asst_count = 0;
  res->asst_problems = 0;
  res->repair_count = 0;
  res->curr_start_time = NULL;

  /* add the meets of tgm */
  rtm = KheResourceTimetableMonitor(res->soln, res->resource);
  for( i = 0;  i < KheTimeGroupTimeCount(res->tg);  i++ )
  {
    time = KheTimeGroupTime(res->tg, i);
    for( j = 0;  j < KheResourceTimetableMonitorTimeTaskCount(rtm, time);  j++ )
    {
      meet = KheTaskMeet(KheResourceTimetableMonitorTimeTask(rtm, time, j));
      if( !HaArrayContains(res->meets, meet, &pos) )
      {
	HaArrayAddLast(res->meets, meet);
	res->total_durn += KheMeetDuration(meet);
      }
    }
  }

  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheLitSolverDelete(KHE_LIT_SOLVER lits)                             */
/*                                                                           */
/*  Delete lits.                                                             */
/*                                                                           */
/*****************************************************************************/

static void KheLitSolverDelete(KHE_LIT_SOLVER lits)
{
  KheSolnArenaEnd(lits->soln, lits->arena);
  /* HaArenaD elete(lits->arena); */
  /* ***
  MArrayFree(lits->meets);
  MFree(lits);
  *** */
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheLitSolverAsstIsOpen(KHE_LIT_SOLVER lits,                         */
/*    KHE_MEET meet, KHE_TIME time, int pos)                                 */
/*                                                                           */
/*  Return true if the assignment of meet to time is open.                   */
/*                                                                           */
/*****************************************************************************/
#define KheIndent 2*(KheEjectorCurrDepth(lits->ejector) + pos + 1)

/*****************************************************************************/
/*                                                                           */
/*  bool KheLitSolverSolve(KHE_LIT_SOLVER lits, int pos, int prev_durn)      */
/*                                                                           */
/*  Solve lits from position pos onwards; prev_durn is the total duration    */
/*  previous to pos.                                                         */
/*                                                                           */
/*****************************************************************************/

static bool KheLitSolverSolve(KHE_LIT_SOLVER lits, int pos, int prev_durn)
{
  KHE_TIME time;  int i, save_problems;  KHE_MEET meet, tmp;  bool success;
  if( DEBUG9 )
    fprintf(stderr, "%*s[ KheLitSolverSolve(lits, pos %d, prev_durn %d)\n",
      KheIndent, "", pos, prev_durn);
  if( pos >= HaArrayCount(lits->meets) )
  {
    /* carry out the repair now; the ejector will undo it */
    lits->repair_count++;
    KheEjectorRepairBegin(lits->ejector);
    HaArrayForEach(lits->meets, meet, i)
      KheMeetUnAssignTime(meet);
    time = lits->curr_start_time;
    success = true;
    HaArrayForEach(lits->meets, meet, i)
    {
      HnAssert(time != NULL, "KheLitSolverSolve internal error 1");
      success = KheMeetAssignTime(meet, time);
      if( !success )
	break;
      time = KheTimeHasNeighbour(time, KheMeetDuration(meet)) ?
	KheTimeNeighbour(time, KheMeetDuration(meet)) : NULL;
    }
    success = KheEjectorRepairEnd(lits->ejector, KHE_REPAIR_COMPLEX_IDLE_MOVE,
      success);
    if( DEBUG9 )
      fprintf(stderr, "%*s] KheLitSolverSolve returning %s (repair)\n",
	KheIndent, "", success ? "true" : "false");
    return success;
  }
  else if( lits->repair_count < MAX_LIT_REPAIRS )
  {
    /* try every meet from pos onwards in position pos */
    for( i = pos;  i < HaArrayCount(lits->meets);  i++ )
    {
      HaArraySwap(lits->meets, pos, i, tmp);
      meet = HaArray(lits->meets, pos);
      time = KheTimeNeighbour(lits->curr_start_time, prev_durn);
      save_problems = lits->asst_problems;
      if( /* KheLitSolverAsstIsOpen(lits, meet, time, pos) && */
	  KheLitSolverSolve(lits, pos+1, prev_durn + KheMeetDuration(meet)) )
      {
	if( DEBUG9 )
	  fprintf(stderr, "%*s] KheLitSolverSolve returning true\n",
	    KheIndent, "");
	return true;
      }
      lits->asst_problems = save_problems;
      HaArraySwap(lits->meets, pos, i, tmp);
    }
    if( DEBUG9 )
      fprintf(stderr, "%*s] KheLitSolverSolve returning false\n",
	KheIndent, "");
    return false;
  }
  else
  {
    /* just try one repair, since limit is reached */
    meet = HaArray(lits->meets, pos);
    time = KheTimeNeighbour(lits->curr_start_time, prev_durn);
    save_problems = lits->asst_problems;
    if( /* KheLitSolverAsstIsOpen(lits, meet, time, pos) && */
	KheLitSolverSolve(lits, pos+1, prev_durn + KheMeetDuration(meet)) )
    {
      if( DEBUG9 )
	fprintf(stderr, "%*s] KheLitSolverSolve returning true (single)\n",
	  KheIndent, "");
      return true;
    }
    lits->asst_problems = save_problems;
    if( DEBUG9 )
      fprintf(stderr, "%*s] KheLitSolverSolve returning false (single)\n",
	KheIndent, "");
    return false;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheLimitIdleComplexAugment(KHE_EJECTOR ej,                          */
/*    KHE_LIMIT_IDLE_TIMES_MONITOR litm, KHE_TIME_GROUP_MONITOR tgm)         */
/*                                                                           */
/*  Augment function for limit idle times defects which tries complex        */
/*  reassignments of the meets assigned to the times of tg, which is         */
/*  known to contain at least one idle time.                                 */
/*                                                                           */
/*****************************************************************************/

static bool KheLimitIdleComplexAugment(KHE_EJECTOR ej,
  KHE_LIMIT_IDLE_TIMES_MONITOR litm, KHE_TIME_GROUP tg)
{
  KHE_LIT_SOLVER lits;  int stop_index, i;
  lits = KheLitSolverMake(ej, litm, tg);
  if( DEBUG8 )
    fprintf(stderr, "%*s[ KheLimitIdleComplexAugment(ej, %s, %s)\n",
      KheEjectorCurrDebugIndent(ej), "", KheResourceId(lits->resource),
      KheTimeGroupId(lits->tg));
  stop_index = KheTimeGroupTimeCount(lits->tg) - lits->total_durn;
  for( i = 0;  i < stop_index;  i++ )
  {
    lits->curr_start_time = KheTimeGroupTime(lits->tg, i);
    if( DEBUG8 )
      fprintf(stderr, "%*s  starting time %s:\n", KheEjectorCurrDebugIndent(ej), "",
	KheTimeId(lits->curr_start_time));
    if( KheLitSolverSolve(lits, 0, 0) )
    {
      KheLitSolverDelete(lits);
      if( DEBUG8 )
	fprintf(stderr,
	  "%*s] KheLimitIdleComplexAugment returning true (%d repairs)\n",
	  KheEjectorCurrDebugIndent(ej), "", lits->repair_count);
      return true;
    }
  }
  KheLitSolverDelete(lits);
  if( DEBUG8 )
    fprintf(stderr,
      "%*s] KheLimitIdleComplexAugment returning false (%d repairs)\n",
      KheEjectorCurrDebugIndent(ej), "", lits->repair_count);
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheLimitIdleMeetMoveFn(KHE_MEET meet, KHE_TIME time, void *impl)    */
/*                                                                           */
/*  Return true if moving meet from where it is now to time would cause      */
/*  it to begin at an idle time of litm (passed in impl).                    */
/*                                                                           */
/*****************************************************************************/

#if WITH_OLD_LIMIT_IDLE_REPAIR
#else
#if WITH_NEW_LIMIT_IDLE_REPAIR_EXACT
static bool KheLimitIdleMeetMoveFn(KHE_MEET meet, KHE_TIME time, void *impl)
{
  KHE_LIMIT_IDLE_TIMES_MONITOR litm;  KHE_TIME_GROUP_MONITOR tgm;
  int durn, xdurn, i, count;  KHE_TIME times[2], xt;
  KHE_TIMETABLE_MONITOR tm;
  litm = (KHE_LIMIT_IDLE_TIMES_MONITOR) impl;
  tm = KheResourceTimetableMonitor(KheMeetSoln(meet),
    KheLimitIdleTimesMonitorResource(litm));
  if( KheTimetableMonitorTimeMeetCount(tm, time) > 0 )
    return false;
  durn = KheMeetDuration(meet);
  for( i = 0;  i < KheLimitIdleTimesMonitorTimeGroupMonitorCount(litm);  i++ )
  {
    tgm = KheLimitIdleTimesMonitorTimeGroupMonitor(litm, i);
    if( KheTimeGroupMonitorIdleCount(tgm) > 0 )
    {
      /* tgm has an idle time to overlap with */
      KheTimeGroupMonitorFirstAndLastBusyTimes(tgm, times, &count);
      HnAssert(count == 2, "KheLimitIdleMeetMoveFn internal error");
      xt = KheTimeNeighbour(times[0], 1);
      xdurn = KheTimeIndex(times[1]) - KheTimeIndex(times[0]) - 1;
      if( KheTimeIntervalsOverlap(time, 1, xt, xdurn) > 0 )
	return true;
    }
  }
  return false;
}
#endif
#endif


/*****************************************************************************/
/*                                                                           */
/*  bool KheLimitIdleMeetMoveFn(KHE_MEET meet, KHE_TIME time, void *impl)    */
/*                                                                           */
/*  Return true if moving meet from where it is now to time would cause      */
/*  it to overlap an idle time of litm (passed in impl).                     */
/*                                                                           */
/*****************************************************************************/

#if WITH_OLD_LIMIT_IDLE_REPAIR
#else
#if WITH_NEW_LIMIT_IDLE_REPAIR_OVERLAP
static bool KheLimitIdleMeetMoveFn(KHE_MEET meet, KHE_TIME time, void *impl)
{
  KHE_LIMIT_IDLE_TIMES_MONITOR litm;  KHE_TIME_GROUP_MONITOR tgm;
  int durn, xdurn, ydurn, i, count, ti;  KHE_TIME times[2], xt, yt, time2;
  KHE_TIMETABLE_MONITOR tm;
  litm = (KHE_LIMIT_IDLE_TIMES_MONITOR) impl;
  tm = KheResourceTimetableMonitor(KheMeetSoln(meet),
    KheLimitIdleTimesMonitorResource(litm));
  durn = KheMeetDuration(meet);
  for( i = 0;  i < KheLimitIdleTimesMonitorTimeGroupMonitorCount(litm);  i++ )
  {
    tgm = KheLimitIdleTimesMonitorTimeGroupMonitor(litm, i);
    if( KheTimeGroupMonitorIdleCount(tgm) > 0 )
    {
      /* tgm has an idle time to overlap with */
      KheTimeGroupMonitorFirstAndLastBusyTimes(tgm, times, &count);
      HnAssert(count == 2, "KheLimitIdleMeetMoveFn internal error");
      xt = KheTimeNeighbour(times[0], 1);
      xdurn = KheTimeIndex(times[1]) - KheTimeIndex(times[0]) - 1;
      if( KheTimeIntervalsOverlapInterval(time, durn, xt, xdurn, &yt, &ydurn) )
	for( ti = 0;  ti < ydurn;  ti++ )
	{
	  time2 = KheTimeNeighbour(yt, ti);
	  if( KheTimetableMonitorTimeMeetCount(tm, time2) == 0 )
	    return true;
	}
    }
  }
  return false;
}
#endif
#endif


/*****************************************************************************/
/*                                                                           */
/*  bool KheLimitIdleTimesAugment(KHE_EJECTOR ej, KHE_MONITOR d)             */
/*                                                                           */
/*  Augment function for individual limit idle times defects.                */
/*                                                                           */
/*  This function tries each ejecting meet of a meet assigned the monitor's  */
/*  resource to a time that causes it to overlap with an idle time.          */
/*                                                                           */
/*****************************************************************************/
#if WITH_OLD_LIMIT_IDLE_REPAIR
#else
static void KheLimitIdleTimesAugment(KHE_EJECTOR ej, KHE_MONITOR d)
{
  KHE_LIMIT_IDLE_TIMES_MONITOR litm;  KHE_MEET meet;  KHE_RESOURCE r;
  KHE_OPTIONS_KEMPE kempe;  bool ejecting;  int i, index, extra, junk, count;
  KHE_SOLN soln;  KHE_TASK task;  KHE_TIME_GROUP_MONITOR tgm;
  KHE_LIMIT_IDLE_TIMES_CONSTRAINT litc;
  KHE_TIME extreme_busy_times[2]; int extreme_busy_times_count;

  KheDebugAugmentBegin(DEBUG4, ej, d, "KheLimitIdleTimesAugment");
  if( KheEjectorRepairTimes(ej) )
  {
    /* simple repairs */
    soln = KheEjectorSoln(ej);
    litm = (KHE_LIMIT_IDLE_TIMES_MONITOR) d;
    ejecting = KheEjectorEjectingNotBasic(ej);
    kempe = KheEjectorUseKempeMoves(ej);
    extra = KheEjectorCurrAugmentCount(ej);
    r = KheLimitIdleTimesMonitorResource(litm);
    for( i = 0;  i < KheResourceAssignedTaskCount(soln, r);  i++ )
    {
      index = (extra + i) % KheResourceAssignedTaskCount(soln, r);
      task = KheResourceAssignedTask(soln, r, index);
      meet = KheMeetFirstMovable(KheTaskMeet(task), &junk);
      if( meet!=NULL && KheMeetAugment(ej, meet, true, &KheLimitIdleMeetMoveFn,
	  (void *) litm, kempe, ejecting, !ejecting, false) )
      {
	KheDebugAugmentEnd(DEBUG4, ej, d, "KheLimitIdleTimesAugment", true);
	return true;
      }
    }

    /* complex repairs, only tried at depth 1 (currently not at all) */
    if( KheEjectorCurrDepth(ej) == 1 && false )
    {
      litc = KheLimitIdleTimesMonitorConstraint(litm);
      count = KheLimitIdleTimesConstraintTimeGroupCount(litc);
      for( i = 0;  i < count;  i++ )
      {
        index = (extra + i) % count;
	tg = KheLimitIdleTimesConstraintTimeGroup(litc, index);
	KheLimitIdleTimesMonitorTimeGroupState(litm, index, &busy_count,
	  &idle_count, extreme_busy_times, &extreme_busy_times_count);
	if( idle_count > 0 )
	{
	  /* ***
	  KheSolnNewG lobalVisit(KheMonitorSoln(d));
	  *** */
	  if( KheLimitIdleComplexAugment(ej, litm, tg) )
	  {
	    KheDebugAugmentEnd(DEBUG4, ej, d, "KheLimitIdleTimesAugment (c)",
	      true);
	    return true;
	  }
	}
      }
    }
  }
  KheDebugAugmentEnd(DEBUG4, ej, d, "KheLimitIdleTimesAugment", false);
  return false;
}
#endif


/*****************************************************************************/
/*                                                                           */
/*  bool KheLimitIdleMeetBoundAugment(KHE_EJECTOR ej, KHE_MEET last_meet,    */
/*    KHE_LIMIT_IDLE_TIMES_MONITOR litm, KHE_TIME_GROUP_MONITOR tgm)         */
/*                                                                           */
/*  Try unassigning last_meet and reducing the domains of all litm's meets.  */
/*                                                                           */
/*****************************************************************************/

#if WITH_OLD_LIMIT_IDLE_REPAIR
static bool KheLimitIdleMeetBoundAugment(KHE_EJECTOR ej,
  /* KHE_MEET last_meet, */ KHE_LIMIT_IDLE_TIMES_MONITOR litm, int index)
{
  KHE_SOLN soln;  int i, j, count, num, junk, busy_count, idle_count;
  KHE_MEET meet;  KHE_TASK task;  KHE_RESOURCE r;
  KHE_TIME times[2];  KHE_TIME_GROUP tg, include_tg;  bool success;
  r = KheLimitIdleTimesMonitorResource(litm);
  soln = KheEjectorSoln(ej);
  if( DEBUG16 )
  {
    fprintf(stderr, "[ KheLimitIdleMeetBoundAugment(ej, ");
    /* KheMeetDebug(last_meet, 1, -1, stderr); */
    fprintf(stderr, ", litm(%s), %d)\n", KheResourceId(r), index);
    for( i = 0;  i < KheResourceAssignedTaskCount(soln, r);  i++ )
    {
      task = KheResourceAssignedTask(soln, r, i);
      meet = KheMeetFirstMovable(KheTaskMeet(task), &junk);
      if( meet != NULL )
	KheMeetDebug(meet, 2, 2, stderr);
    }
  }

  /* build a time group containing all times not mentioned in litm */
  KheSolnTimeGroupBegin(soln);
  KheSolnTimeGroupUnion(soln, KheInstanceFullTimeGroup(KheSolnInstance(soln)));
  for( i = 0;  i < KheLimitIdleTimesMonitorTimeGroupCount(litm);  i++ )
  {
    tg = KheLimitIdleTimesMonitorTimeGroup(litm, i);
    KheSolnTimeGroupDifference(soln, tg);
  }

  /* add the busy spans of litm's time groups, except the last time of the */
  /* index'th time group */
  for( i = 0;  i < KheLimitIdleTimesMonitorTimeGroupCount(litm);  i++ )
  {
    tg = KheLimitIdleTimesMonitorTimeGroupState(litm, i, &busy_count,
      &idle_count, times, &count);
    if( count >= 1 )
    {
      if( DEBUG16 )
      {
	fprintf(stderr, "  within ");
	KheTimeGroupDebug(tg, 1, -1, stderr);
	fprintf(stderr, " busy span %s .. %s\n", KheTimeId(times[0]),
	  KheTimeId(times[count - 1]));
      }
      num = KheTimeIndex(times[count-1]) - KheTimeIndex(times[0]) + 1;
      if( i == index )
	num--;
      for( j = 0;  j < num;  j++ )
        KheSolnTimeGroupAddTime(soln, KheTimeNeighbour(times[0], j));
    }
  }
  include_tg = KheSolnTimeGroupEnd(soln);
  if( DEBUG16 )
  {
    fprintf(stderr, "  include_tg: ");
    KheTimeGroupDebug(include_tg, 1, 0, stderr);
  }

  /* apply the meet bound repair */
  success = KheMeetBoundRepair(ej, r, include_tg);
  if( DEBUG16 )
    fprintf(stderr, "] KheLimitIdleMeetBoundAugment ret %s (end)\n",
      success ? "true" : "false");
  return success;
}

/* *** old version that does not call KheMeetBoundRepair
static bool KheLimitIdleMeetBoundAugment(KHE_EJECTOR ej, KHE_MEET last_meet,
  KHE_LIMIT_IDLE_TIMES_MONITOR litm, int index)
{
  KHE_SOLN soln;  int i, j, count, num, durn, junk, busy_count, idle_count;
  KHE_MEET meet;  KHE_TASK task;  KHE_RESOURCE r;  KHE_MEET_BOUND mb;
  KHE_TIME times[2];  KHE_TIME_GROUP tg, busy_tg, starting_tg;  bool success;
  r = KheLimitIdleTimesMonitorResource(litm);
  soln = KheEjectorSoln(ej);
  if( DEBUG16 )
  {
    fprintf(stderr, "[ KheLimitIdleMeetBoundAugment(ej, ");
    KheMeetDebug(last_meet, 1, -1, stderr);
    fprintf(stderr, ", litm(%s), %d)\n", KheResourceId(r), index);
    for( i = 0;  i < KheResourceAssignedTaskCount(soln, r);  i++ )
    {
      task = KheResourceAssignedTask(soln, r, i);
      meet = KheMeetFirstMovable(KheTaskMeet(task), &junk);
      if( meet != NULL )
	KheMeetDebug(meet, 2, 2, stderr);
    }
  }

  ** build a time group containing all times not mentioned in litm **
  KheSolnTimeGr oupBegin(soln);
  KheSolnTimeGroupUnion(soln, KheInstanceFullTimeGroup(KheSolnInstance(soln)));
  for( i = 0;  i < KheLimitIdleTimesMonitorTimeGroupCount(litm);  i++ )
  {
    tg = KheLimitIdleTimesMonitorTimeGroup(litm, i);
    KheSolnTimeGroupDifference(soln, tg);
  }

  ** add the busy spans of litm's time groups, except the last time of tgm **
  for( i = 0;  i < KheLimitIdleTimesMonitorTimeGroupCount(litm);  i++ )
  {
    tg = KheLimitIdleTimesMonitorTimeGroupState(litm, i, &busy_count,
      &idle_count, times, &count);
    if( count >= 1 )
    {
      if( DEBUG16 )
      {
	fprintf(stderr, "  within ");
	KheTimeGroupDebug(tg, 1, -1, stderr);
	fprintf(stderr, " busy span %s .. %s\n", KheTimeId(times[0]),
	  KheTimeId(times[count - 1]));
      }
      num = KheTimeIndex(times[count-1]) - KheTimeIndex(times[0]) + 1;
      if( i == index )
	num--;
      for( j = 0;  j < num;  j++ )
        KheSolnTimeGroupAddTime(soln, KheTimeNeighbour(times[0], j));
    }
  }
  busy_tg = KheSolnTimeGroupEnd(soln);
  if( DEBUG16 )
  {
    fprintf(stderr, "  busy_tg: ");
    KheTimeGroupDebug(busy_tg, 1, 0, stderr);
  }

  ** unassign last_meet **
  KheEjector RepairBegin(ej);
  if( !KheMeetUnAssign(last_meet) )
  {
    if( DEBUG16 )
      fprintf(stderr, "] KheLimitIdleMeetBoundAugment ret false (last_meet)\n");
    return KheEject orRepairEnd(ej, KHE_REPAIR_LIMIT_IDLE_MEETS_UNASSIGN, false);
  }

  ** add a meet bound to each meet **
  mb = KheMeetBoundMake(soln, true, busy_tg);
  for( i = 0;  i < KheResourceAssignedTaskCount(soln, r);  i++ )
  {
    task = KheResourceAssignedTask(soln, r, i);
    meet = KheMeetFirstMovable(KheTaskMeet(task), &junk);
    if( meet == NULL || !KheMeetAddMeetBound(meet, mb) )
    {
      if( DEBUG16 )
      {
	fprintf(stderr, "] KheLimitIdleMeetBoundAugment ret false (");
	KheMeetDebug(meet, 1, -1, stderr);
	fprintf(stderr, ", %d, ", durn);
	KheTimeGroupDebug(starting_tg, 1, -1, stderr);
	fprintf(stderr, ")\n");
      }
      return KheEjecto rRepairEnd(ej, KHE_REPAIR_LIMIT_IDLE_MEETS_UNASSIGN,
	false);
    }
  }
  success = KheEjector RepairEndLong(ej, KHE_REPAIR_LIMIT_IDLE_MEETS_UNASSIGN,
    true, 1, false, &KheMeetBoundOnSuccess, (void *) mb);
  if( DEBUG16 )
    fprintf(stderr, "] KheLimitIdleMeetBoundAugment ret %s (end)\n",
      success ? "true" : "false");
  return success;
}
*** */
#endif


/*****************************************************************************/
/*                                                                           */
/*  bool KheTimeGroupMonitorHasSoleLastMeet(KHE_TIME_GROUP_MONITOR tgm,      */
/*    KHE_MEET *last_meet)                                                   */
/*                                                                           */
/*  If the last busy time of tgm is occupied by a single meet, set           */
/*  *last_meet to that meet and return true.  Otherwise return false.        */
/*                                                                           */
/*****************************************************************************/

static bool KheTimeGroupHasSoleLastMeet(KHE_LIMIT_IDLE_TIMES_MONITOR litm,
  KHE_TIME_GROUP tg, KHE_TIME last_busy_time, KHE_MEET *last_meet)
{
  KHE_RESOURCE_TIMETABLE_MONITOR rtm;  KHE_RESOURCE r;
  r = KheLimitIdleTimesMonitorResource(litm);
  rtm = KheResourceTimetableMonitor(KheMonitorSoln((KHE_MONITOR) litm), r);
  if( KheResourceTimetableMonitorTimeTaskCount(rtm, last_busy_time) == 1 )
  {
    *last_meet =
      KheTaskMeet(KheResourceTimetableMonitorTimeTask(rtm, last_busy_time, 0));
    return true;
  }
  else
    return false;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheLimitIdleTimesAugment(KHE_EJECTOR ej, KHE_MONITOR d) old version */
/*                                                                           */
/*  Augment function for individual limit idle times defects.                */
/*                                                                           */
/*  This function tries each meet move whose initial meet move moves a       */
/*  meet from the start or end of a day (actually, a time group of the       */
/*  monitor) in any way that reduces the number of idle times.               */
/*                                                                           */
/*****************************************************************************/

#if WITH_OLD_LIMIT_IDLE_REPAIR
static bool KheLimitIdleTimesAugment(KHE_EJECTOR ej, KHE_MONITOR d)
{
  KHE_LIMIT_IDLE_TIMES_MONITOR litm;  KHE_MEET meet, last_meet;
  KHE_RESOURCE_TIMETABLE_MONITOR rtm;  bool not_ejecting;  KHE_TIME_GROUP tg;
  KHE_TIME time;  int i, j, k, index, extra, durn;
  KHE_LIMIT_IDLE_TIMES_CONSTRAINT litc;  int tg_count;
  int busy_count, idle_count;
  KHE_TIME times[2]; int times_count;

  KheDebugAugmentBegin(DEBUG4, ej, d, "KheLimitIdleTimesAugment");
  if( KheEjectorRepairTimes(ej) )
  {
    /* simple repairs */
    litm = (KHE_LIMIT_IDLE_TIMES_MONITOR) d;
    litc = KheLimitIdleTimesMonitorConstraint(litm);
    tg_count = KheLimitIdleTimesConstraintTimeGroupCount(litc);
    rtm = KheResourceTimetableMonitor(KheEjectorSoln(ej),
      KheLimitIdleTimesMonitorResource(litm));
    not_ejecting = KheEjectorBasicNotEjecting(ej);
    extra = KheEjectorCurrAugmentCount(ej);
    for( durn = 1;  durn <= 4;  durn++ )
    {
      for( i = 0;  i < tg_count;  i++ )
      {
	index = (extra + i) % tg_count;
	KheLimitIdleTimesMonitorTimeGroupState(litm, index, &busy_count,
	  &idle_count, times, &times_count);
	/* ***
	if( KheEjectorCurrDepth(ej) == 1 )
	  KheSolnNewG lobalVisit(KheMonitorSoln(d));
	*** */
	for( j = 0;  j < times_count;  j++ )
	{
	  time = times[j];
	  for( k = 0;  k < KheResourceTimetableMonitorTimeTaskCount(rtm, time);  k++ )
	  {
	    meet = KheTaskMeet(KheResourceTimetableMonitorTimeTask(rtm,time,k));
	    if( KheMeetDuration(meet) == durn ||
		(durn == 4 && KheMeetDuration(meet) >= durn) )
	    {
	      if( KheMeetAugment(ej, meet, true, &KheLimitIdleMeetMoveFn,
		(void *) litm, KHE_OPTIONS_KEMPE_TRUE, !not_ejecting,
		not_ejecting, false) )
	      {
		KheDebugAugmentEnd(DEBUG4, ej, d, "KheLimitIdleTimesAugment",
		  true);
		return true;
	      }
	    }
	  }
	}
      }
    }

    /* meet bound repairs, only tried at depth 1 (currently not at all) */
    if( KheEjectorCurrDepth(ej) == 1 && false )
      for( i = 0;  i < tg_count;  i++ )
      {
        index = (extra + i) % tg_count;
	tg = KheLimitIdleTimesMonitorTimeGroupState(litm, index, &busy_count,
	  &idle_count, times, &times_count);
	if( idle_count > 0 && KheTimeGroupHasSoleLastMeet(litm, tg,
	      times[times_count - 1], &last_meet)
	    /* KheTimeGroupMonitorHasSoleLastMeet(tgm, &last_meet) */ )
	{
	  KheSolnNewGlobalVisit(KheEjectorSoln(ej));
	  if( KheLimitIdleMeetBoundAugment(ej, /* last_meet, */ litm, index) )
	  {
	    KheDebugAugmentEnd(DEBUG4, ej, d, "KheLimitIdleTimesAugment (d)",
	      true);
	    return true;
	  }
	}
      }

    /* complex repairs, only tried at depth 1 (currently not at all) */
    if( KheEjectorCurrDepth(ej) == 1 && false )
      for( i = 0;  i < tg_count;  i++ )
      {
        index = (extra + i) % tg_count;
	tg = KheLimitIdleTimesConstraintTimeGroup(litc, index);
	KheLimitIdleTimesMonitorTimeGroupState(litm, index, &busy_count,
	  &idle_count, times, &times_count);
	if( idle_count > 0 )
	{
	  /* ***
	  KheSolnNewGl obalVisit(KheEjectorSoln(ej));
	  *** */
	  if( KheLimitIdleComplexAugment(ej, litm, tg) )
	  {
	    KheDebugAugmentEnd(DEBUG4, ej, d, "KheLimitIdleTimesAugment (c)",
	      true);
	    return true;
	  }
	}
      }
  }
  KheDebugAugmentEnd(DEBUG4, ej, d, "KheLimitIdleTimesAugment", false);
  return false;
}
#endif


/*****************************************************************************/
/*                                                                           */
/*  bool KheLimitIdleTimesGroupAugment(KHE_EJECTOR ej, KHE_MONITOR d)        */
/*                                                                           */
/*  Augment function for groups of limit idle times defects.                 */
/*                                                                           */
/*****************************************************************************/

static bool KheLimitIdleTimesGroupAugment(KHE_EJECTOR ej, KHE_MONITOR d)
{
  KHE_GROUP_MONITOR gm;
  HnAssert(KheMonitorTag(d) == KHE_GROUP_MONITOR_TAG,
    "KheLimitIdleTimesGroupAugment internal error 1");
  gm = (KHE_GROUP_MONITOR) d;
  HnAssert(KheGroupMonitorDefectCount(gm) > 0,
    "KheLimitIdleTimesGroupAugment internal error 2");
  return KheLimitIdleTimesAugment(ej, KheGroupMonitorDefect(gm, 0));
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "cluster busy times augment functions"                         */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool KheClusterUnderloadAugment(KHE_EJECTOR ej,                          */
/*    KHE_CLUSTER_BUSY_TIMES_MONITOR cbtm, int active_count, bool allow_zero)*/
/*                                                                           */
/*  Repair an underload defect in cbtm: make an inactive time group active.  */
/*  Here active_count is the number of active time groups.                   */
/*  Or if allow_zero is true, also try reducing to zero.                     */
/*                                                                           */
/*****************************************************************************/

static bool KheClusterUnderloadAugment(KHE_EJECTOR ej,
  KHE_CLUSTER_BUSY_TIMES_MONITOR cbtm, int active_count, bool allow_zero)
{
  int i, ix, extra, count, busy;  KHE_RESOURCE r;  KHE_TIME_GROUP tg;
  KHE_POLARITY po;
  r = KheClusterBusyTimesMonitorResource(cbtm);
  if( DEBUG7 )
    fprintf(stderr, "%*s[ KheClusterUnderloadAugment(ej, %s)\n",
      KheEjectorCurrDebugIndent(ej), "", KheResourceId(r));
  extra = KheEjectorCurrAugmentCount(ej);
  count = KheClusterBusyTimesMonitorTimeGroupCount(cbtm);
  for( i = 0;  i < count;  i++ )
  {
    ix = (extra + i) % count;
    if( !KheClusterBusyTimesMonitorTimeGroupIsActive(cbtm, ix, &tg, &po,&busy) )
    {
      if( DEBUG7 )
      {
	fprintf(stderr, "  trying time group ");
	KheTimeGroupDebug(tg, 1, 0, stderr);
      }
      if( KheOverUnderAugment(ej, r, tg, po == KHE_NEGATIVE, true, false) )
      {
	if( DEBUG7 )
	  fprintf(stderr, "%*s] KheClusterUnderloadAugment ret true\n",
	    KheEjectorCurrDebugIndent(ej), "");
	return true;
      }
    }
    else if( allow_zero && active_count == 1 )
    {
      /* try to reduce the number of active time groups to zero */
      if( DEBUG7 )
      {
	fprintf(stderr, "  trying allow_zero time group ");
	KheTimeGroupDebug(tg, 1, 0, stderr);
      }
      if( KheOverUnderAugment(ej, r, tg, po == KHE_POSITIVE, true, false) )
      {
	if( DEBUG7 )
	  fprintf(stderr, "%*s] KheClusterUnderloadAugment ret true\n",
	    KheEjectorCurrDebugIndent(ej), "");
	return true;
      }
    }
  }
  if( DEBUG7 )
    fprintf(stderr, "%*s] KheClusterUnderloadAugment ret false\n",
      KheEjectorCurrDebugIndent(ej), "");
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_TIME_GROUP KheClusterMeetsUnassignTimeGroup(                         */
/*    KHE_CLUSTER_BUSY_TIMES_MONITOR cbtm, KHE_TIME_GROUP tg)                */
/*                                                                           */
/*  Return a time group that is the cycle minus every time group of cbtm     */
/*  that has no meets, and minus tg as well.                                 */
/*                                                                           */
/*****************************************************************************/

static KHE_TIME_GROUP KheClusterMeetsUnassignTimeGroup(
  KHE_CLUSTER_BUSY_TIMES_MONITOR cbtm, KHE_TIME_GROUP tg)
{
  KHE_SOLN soln;  int i, tg_count, busy_count;  KHE_POLARITY po;
  KHE_TIME_GROUP tg2;
  soln = KheMonitorSoln((KHE_MONITOR) cbtm);
  KheSolnTimeGroupBegin(soln);
  KheSolnTimeGroupUnion(soln, KheInstanceFullTimeGroup(KheSolnInstance(soln)));
  tg_count = KheClusterBusyTimesMonitorTimeGroupCount(cbtm);
  for( i = 0;  i < tg_count;  i++ )
  {
    KheClusterBusyTimesMonitorTimeGroupIsActive(cbtm, i, &tg2, &po,&busy_count);
    /* ***
    tg2 = KheClusterBusyTimesMonitorTimeGroupState(cbtm, i, &po, &busy_count);
    *** */
    if( busy_count == 0 )
      KheSolnTimeGroupDifference(soln, tg2);
  }
  KheSolnTimeGroupDifference(soln, tg);
  return KheSolnTimeGroupEnd(soln);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheClusterOverloadAugment(KHE_EJECTOR ej,                           */
/*    KHE_CLUSTER_BUSY_TIMES_MONITOR cbtm)                                   */
/*                                                                           */
/*  Repair an overload defect in cbtm, by making an active time group        */
/*  inactive.                                                                */
/*                                                                           */
/*****************************************************************************/

static bool KheClusterOverloadAugment(KHE_EJECTOR ej,
  KHE_CLUSTER_BUSY_TIMES_MONITOR cbtm)
{
  int i, busy, ix, tg_count, count, extra;
  KHE_TIME_GROUP tg, include_tg;  KHE_RESOURCE_TIMETABLE_MONITOR rtm;
  KHE_POLARITY po;  KHE_RESOURCE r;  KHE_SOLN soln;
  if( DEBUG28 && KheEjectorCurrDebug(ej) )
    fprintf(stderr, "%*s[ KheClusterOverloadAugment(times %s, depth %d)\n",
      KheEjectorCurrDebugIndent(ej), "",
      KheEjectorRepairTimes(ej) ? "true" : "false", KheEjectorCurrDepth(ej));
  r = KheClusterBusyTimesMonitorResource(cbtm);

  /* underload/overload repair similar to repairing underloads */
  extra = KheEjectorCurrAugmentCount(ej);
  count = KheClusterBusyTimesMonitorTimeGroupCount(cbtm);
  for( i = 0;  i < count;  i++ )
  {
    ix = (extra + i) % count;
    if( KheClusterBusyTimesMonitorTimeGroupIsActive(cbtm, ix, &tg, &po, &busy) )
    {
      /* this time group is active */
      if( DEBUG28 && KheEjectorCurrDebug(ej) )
	fprintf(stderr, "%*s  KheClusterOverloadAugment active %s\n",
	  KheEjectorCurrDebugIndent(ej), "", KheTimeGroupId(tg));
      if( KheOverUnderAugment(ej, r, tg, po == KHE_POSITIVE, true, false) )
      {
	if( DEBUG28 && KheEjectorCurrDebug(ej) )
	  fprintf(stderr, "%*s] KheClusterOverloadAugment ret true\n",
	    KheEjectorCurrDebugIndent(ej), "");
	return true;
      }
    }
  }

  /* if meet moves are allowed, try a meet bound repair as last resort */
  if( KheEjectorRepairTimes(ej) )
  {
    soln = KheEjectorSoln(ej);
    rtm = KheResourceTimetableMonitor(soln, r);
    tg_count = KheClusterBusyTimesMonitorTimeGroupCount(cbtm);
    extra = KheEjectorCurrAugmentCount(ej);
    for( i = 0;  i < tg_count;  i++ )
    {
      ix = (extra + i) % tg_count;
      KheClusterBusyTimesMonitorTimeGroupIsActive(cbtm, ix, &tg, &po, &busy);
      /* ***
      tg = KheClusterBusyTimesMonitorTimeGroupState(cbtm, ix, &po, &busy);
      *** */
      if( po == KHE_POSITIVE && busy > 0 )
      {
	if( KheEjectorCurrDepth(ej) == 1 || KheOneMeet(rtm, tg) )
	{
	  if( KheEjectorCurrDepth(ej) == 1 )
	    KheSolnNewGlobalVisit(soln);
	  include_tg = KheClusterMeetsUnassignTimeGroup(cbtm, tg);
	  if( KheMeetBoundRepair(ej, r, include_tg) )
	  {
	    if( DEBUG28 && KheEjectorCurrDebug(ej) )
	      fprintf(stderr, "%*s] KheClusterOverloadAugment ret true (1)\n",
		KheEjectorCurrDebugIndent(ej), "");
	    return true;
	  }
	}
      }
    }
  }
  if( DEBUG28 && KheEjectorCurrDebug(ej) )
    fprintf(stderr, "%*s] KheClusterOverloadAugment ret false\n",
      KheEjectorCurrDebugIndent(ej), "");
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheClusterBusyTimesAugment(KHE_EJECTOR ej, KHE_MONITOR d)           */
/*                                                                           */
/*  Augment function for individual cluster busy times defects.              */
/*                                                                           */
/*  Use KheOverUnderAugment to make inactive time groups active (if there    */
/*  is an underload) or active time groups inactive (if there is an          */
/*  overload).  If repairing times, also try a more complex ejection         */
/*  tree repair as a last resort.                                            */
/*                                                                           */
/*****************************************************************************/

static bool KheClusterBusyTimesAugment(KHE_EJECTOR ej, KHE_MONITOR d)
{
  KHE_CLUSTER_BUSY_TIMES_MONITOR cbtm;
  int count, ind_count, minimum, maximum;  bool allow_zero, success;
  KheDebugAugmentBegin(DEBUG14, ej, d, "KheClusterBusyTimesAugment");
  HnAssert(KheMonitorTag(d) == KHE_CLUSTER_BUSY_TIMES_MONITOR_TAG,
    "KheClusterBusyTimesAugment internal error 1");
  cbtm = (KHE_CLUSTER_BUSY_TIMES_MONITOR) d;
  KheClusterBusyTimesMonitorActiveTimeGroupCount(cbtm, &count,
    &ind_count, &minimum, &maximum, &allow_zero);
  success = false;
  if( count < minimum )
    success = KheClusterUnderloadAugment(ej, cbtm, count, allow_zero);
  else if( count > maximum )
    success = KheClusterOverloadAugment(ej, cbtm);
  else
    HnAbort("KheClusterBusyTimesAugment internal error 2");
  KheDebugAugmentEnd(DEBUG14, ej, d, "KheClusterBusyTimesAugment", success);
  return success;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheClusterBusyTimesGroupAugment(KHE_EJECTOR ej, KHE_MONITOR d)      */
/*                                                                           */
/*  Augment function for groups of cluster busy times defects.               */
/*                                                                           */
/*****************************************************************************/

static bool KheClusterBusyTimesGroupAugment(KHE_EJECTOR ej, KHE_MONITOR d)
{
  KHE_GROUP_MONITOR gm;
  HnAssert(KheMonitorTag(d) == KHE_GROUP_MONITOR_TAG,
    "KheClusterBusyTimesGroupAugment internal error 1");
  gm = (KHE_GROUP_MONITOR) d;
  HnAssert(KheGroupMonitorDefectCount(gm) > 0,
    "KheClusterBusyTimesGroupAugment internal error 2");
  return KheClusterBusyTimesAugment(ej, KheGroupMonitorDefect(gm, 0));
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "limit busy times augment functions"                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_TASKING KheGetTasking(KHE_SOLN soln, KHE_RESOURCE r)                 */
/*                                                                           */
/*  Get the tasking containing all the tasks whose type is the type of r.    */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused
static KHE_TASKING KheGetTasking(KHE_SOLN soln, KHE_RESOURCE r)
{
  KHE_RESOURCE_TYPE rt;  KHE_TASKING tasking;  int i;
  rt = KheResourceResourceType(r);
  for( i = 0;  i < KheSolnTaskingCount(soln);  i++ )
  {
    tasking = KheSolnTasking(soln, i);
    if( KheTaskingResourceType(tasking) == rt )
      return tasking;
  }
  HnAbort("KheGetTasking: cannot find tasking for %s resource %s",
    KheResourceTypeId(rt), KheResourceId(r));
  return NULL;  ** keep compiler happy **
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheLimitBusyTimesAugment(KHE_EJECTOR ej, KHE_MONITOR d)             */
/*                                                                           */
/*  Augment function for individual limit busy times defects.                */
/*                                                                           */
/*  Use KheOverUnderAugment to increase or decrease the number of busy       */
/*  times in each time group as required.                                    */
/*                                                                           */
/*****************************************************************************/

static bool KheLimitBusyTimesAugment(KHE_EJECTOR ej, KHE_MONITOR d)
{
  KHE_LIMIT_BUSY_TIMES_MONITOR lbtm;  KHE_TIME_GROUP tg;  KHE_RESOURCE r;
  bool allow_zero /* , no_limit_busy_sequences */;
  int i, ix, count, extra, busy_count, minimum, maximum;
  /* KHE_INSTANCE ins;  KHE_MODEL model; */
  KheDebugAugmentBegin(DEBUG4, ej, d, "KheLimitBusyTimesAugment");
  HnAssert(KheMonitorTag(d) == KHE_LIMIT_BUSY_TIMES_MONITOR_TAG,
    "KheLimitBusyTimesAugment internal error 1");
  /* ***
  no_limit_busy_sequences = KheOptionsGetBool(KheEjectorOptions(ej),
    "es_no_limit_busy_sequences", false);
  ins = KheSolnInstance(KheEjectorSoln(ej));
  model = KheInstanceModel(ins);
  *** */
  lbtm = (KHE_LIMIT_BUSY_TIMES_MONITOR) d;
  r = KheLimitBusyTimesMonitorResource(lbtm);
  count = KheLimitBusyTimesMonitorDefectiveTimeGroupCount(lbtm);
  extra = KheEjectorCurrAugmentCount(ej);
  for( i = 0;  i < count;  i++ )
  {
    ix = (extra + i) % count;
    KheLimitBusyTimesMonitorDefectiveTimeGroup(lbtm, ix, &tg, &busy_count,
      &minimum, &maximum, &allow_zero);
    if( KheOverUnderAugment(ej, r, tg, busy_count > maximum, false,allow_zero) )
    {
      KheDebugAugmentEnd(DEBUG4, ej, d, "KheLimitBusyTimesAugment", true);
      return true;
    }
    /* ***
    if( model == KHE_MODEL_EMPLOYEE_SCHEDULE && busy_count > maximum &&
	!no_limit_busy_sequences )
    {
      ** overload in employee scheduling model **
      if( KheNurseOverloadAugment(ej, r, tg, d) )
      {
	KheDebugAugmentEnd(DEBUG4, ej, d, "KheLimitBusyTimesAugment", true);
	return;
      }
    }
    else
    {
      ** overload in high school model, or underload in either model **
      if( KheOverUnderAugment(ej, r, tg, busy_count > maximum, false,
	    allow_zero, d) )
      {
	KheDebugAugmentEnd(DEBUG4, ej, d, "KheLimitBusyTimesAugment", true);
	return;
      }
    }
    *** */
  }
  KheDebugAugmentEnd(DEBUG4, ej, d, "KheLimitBusyTimesAugment", false);
  return false;
}

/* old version with now inapplicable WITH_EXTRA_LIMIT_BUSY_TIMES_REPAIRS ***
static void KheLimitBusyTimesAugment(KHE_EJECTOR ej, KHE_MONITOR d)
{
  KHE_LIMIT_BUSY_TIMES_MONITOR lbtm;  KHE_TIME_GROUP defective_tg, tg, tg2;
  KHE_RESOURCE r;  bool first, allow_zero;  ARRAY_KHE_TIME_GROUP time_groups;
  int i, ix, j, k, count, extra, busy_count, minimum, maximum, offset;
  KHE_MONITOR m;  KHE_SOLN soln;  KHE_LIMIT_BUSY_TIMES_CONSTRAINT lbtc;
  KheDebugAugmentBegin(DEBUG4, ej, d, "KheLimitBusyTimesAugment");
  HnAssert(KheMonitorTag(d) == KHE_LIMIT_BUSY_TIMES_MONITOR_TAG,
    "KheLimitBusyTimesAugment internal error 1");
  lbtm = (KHE_LIMIT_BUSY_TIMES_MONITOR) d;
  r = KheLimitBusyTimesMonitorResource(lbtm);
  first = true;
  count = KheLimitBusyTime sMonitorDefectiveTimeGroupCount(lbtm);
  extra = KheEjectorCurrAugmentCount(ej);
  for( i = 0;  i < count;  i++ )
  {
    ix = (extra + i) % count;
    KheLimitBusyTimesMonitorDefectiveTimeGroup(lbtm, ix, &defective_tg,
      &busy_count, &minimum, &maximum, &allow_zero);
    ** ***
    if( KheEjectorCurrDepth(ej) == 1 )
      KheSolnNe wGlobalVisit(KheMonitorSoln(d));
    *** **
    if( busy_count > maximum )
    {
      if( KheResourceOverloadAugment(ej, r, defective_tg, false) )
      {
	KheDebugAugmentEnd(DEBUG4, ej, d, "KheLimitBusyTimesAugment", true);
	return;
      }
    }
    else
    {
      if( KheResourceUnderloadAugment(ej, r, defective_tg, allow_zero) )
      {
	KheDebugAugmentEnd(DEBUG4, ej, d, "KheLimitBusyTimesAugment", true);
	return;
      }
      if( WITH_EXTRA_LIMIT_BUSY_TIMES_REPAIRS &&
	  first && KheEjectorRepairTimes(ej) && KheEjectorCurrDepth(ej) == 1 )
      {
	** find all the time groups of all r's limit busy times monitors **
	if( DEBUG17 )
	  fprintf(stderr, "[ extra limit busy times repairs:\n");
	first = false;
	soln = KheEjectorSoln(ej);
	MArrayInit(time_groups);
	for( i = 0;  i < KheSolnResourceMonitorCount(soln, r);  i++ )
	{
	  m = KheSolnResourceMonitor(soln, r, i);
	  if( KheMonitorTag(m) == KHE_LIMIT_BUSY_TIMES_MONITOR_TAG )
	  {
	    ** find the time groups of all of r's limit busy times monitors **
            lbtm = (KHE_LIMIT_BUSY_TIMES_MONITOR) m;
            lbtc = KheLimitBusyTimesMonitorConstraint(lbtm);
	    offset = KheLimitBusyTimesMonitorOffset(lbtm);
	    for( j=0; j < KheLimitBusyTimesConstraintTimeGroupCount(lbtc); j++ )
	    {
	      tg = KheLimitBusyTimesConstraintTimeGroup(lbtc, j, offset);
	      if( !KheTimeGroupEqual(tg, defective_tg) )
	      {
		HaArrayForEach(time_groups, tg2, k)
		  if( KheTimeGroupEqual(tg, tg2) )
		    break;
		if( k >= HaArrayCount(time_groups) )
		  HaArrayAddLast(time_groups, tg);
	      }
	    }
	  }
	}

	** try clearing out each of these time groups **
	HaArrayForEach(time_groups, tg, k)
	{
	  if( KheEjectorCurrDepth(ej) == 1 )
	    KheSolnNe wGlobalVisit(soln);
	  if( DEBUG17 )
	  {
	    fprintf(stderr, "  extra limit busy times repair: ");
	    KheTimeGroupDebug(tg, 1, 0, stderr);
	  }
	  if( KheMeetBoundRepair(ej, r, KheSolnComplementTimeGroup(soln, tg)) )
	  {
	    MArrayFree(time_groups);
	    KheDebugAugmentEnd(DEBUG4, ej, d, "KheLimitBusyTimesAugment", true);
	    return;
	  }
	}

	** cleanup and end **
	if( DEBUG17 )
	  fprintf(stderr, "] extra limit busy times repairs ret false\n");
	MArrayFree(time_groups);
      }
    }
  }
  KheDebugAugmentEnd(DEBUG4, ej, d, "KheLimitBusyTimesAugment", false);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheLimitBusyTimesGroupAugment(KHE_EJECTOR ej, KHE_MONITOR d)        */
/*                                                                           */
/*  Augment function for groups of limit busy times defects.                 */
/*                                                                           */
/*****************************************************************************/

static bool KheLimitBusyTimesGroupAugment(KHE_EJECTOR ej, KHE_MONITOR d)
{
  KHE_GROUP_MONITOR gm;
  HnAssert(KheMonitorTag(d) == KHE_GROUP_MONITOR_TAG,
    "KheLimitBusyTimesGroupAugment internal error 1");
  gm = (KHE_GROUP_MONITOR) d;
  HnAssert(KheGroupMonitorDefectCount(gm) > 0,
    "KheLimitBusyTimesGroupAugment internal error 2");
  return KheLimitBusyTimesAugment(ej, KheGroupMonitorDefect(gm, 0));
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "limit workload augment functions"                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool KheLimitWorkloadAugment(KHE_EJECTOR ej, KHE_MONITOR d)              */
/*                                                                           */
/*  Augment function for individual limit workload defects.                  */
/*                                                                           */
/*  Use KheOverUnderAugment to increase or decrease the number of busy       */
/*  times in each time group as required.                                    */
/*                                                                           */
/*****************************************************************************/

static bool KheLimitWorkloadAugment(KHE_EJECTOR ej, KHE_MONITOR d)
{
  KHE_LIMIT_WORKLOAD_MONITOR lwm;  KHE_RESOURCE r;  int i, ix, count, extra;
  KHE_TIME_GROUP tg;  float workload;  int minimum, maximum;  bool allow_zero;
  KheDebugAugmentBegin(DEBUG3, ej, d, "KheLimitWorkloadAugment");
  lwm = (KHE_LIMIT_WORKLOAD_MONITOR) d;
  r = KheLimitWorkloadMonitorResource(lwm);
  tg = KheInstanceFullTimeGroup(KheResourceInstance(r));
  count = KheLimitWorkloadMonitorDefectiveTimeGroupCount(lwm);
  extra = KheEjectorCurrAugmentCount(ej);
  for( i = 0;  i < count;  i++ )
  {
    ix = (extra + i) % count;
    KheLimitWorkloadMonitorDefectiveTimeGroup(lwm, ix, &tg, &workload,
      &minimum, &maximum, &allow_zero);
    if( KheOverUnderAugment(ej, r, tg, workload > maximum, false, false) )
    {
      KheDebugAugmentEnd(DEBUG3, ej, d, "KheLimitWorkloadAugment (overload)",
	true);
      return true;
    }
  }
  KheDebugAugmentEnd(DEBUG3, ej, d, "KheLimitWorkloadAugment", false);
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheLimitWorkloadGroupAugment(KHE_EJECTOR ej, KHE_MONITOR d)         */
/*                                                                           */
/*  Augment function for groups of limit workload defects.                   */
/*                                                                           */
/*****************************************************************************/

static bool KheLimitWorkloadGroupAugment(KHE_EJECTOR ej, KHE_MONITOR d)
{
  KHE_GROUP_MONITOR gm;
  HnAssert(KheMonitorTag(d) == KHE_GROUP_MONITOR_TAG,
    "KheLimitWorkloadGroupAugment internal error 1");
  gm = (KHE_GROUP_MONITOR) d;
  HnAssert(KheGroupMonitorDefectCount(gm) > 0,
    "KheLimitWorkloadGroupAugment internal error 2");
  return KheLimitWorkloadAugment(ej, KheGroupMonitorDefect(gm, 0));
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "limit active intervals augments"                              */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_ACTIVE_AUGMENT_TYPE KheGetActiveAugmentOptions(KHE_OPTIONS options)  */
/*                                                                           */
/*  Get the interpreted value of the active_augment option of options.       */
/*                                                                           */
/*****************************************************************************/

/* ***
static KHE_ACTIVE_AUGMENT_TYPE KheGetActiveAugmentOptions(KHE_OPTIONS options)
{
  char *str;
  str = KheOptionsGet(options, "es_active_augment", "swap");
  if( strcmp(str, "move") == 0 )
    return KHE_ACTIVE_AUGMENT_MOVE;
  else if( strcmp(str, "swap") == 0 )
    return KHE_ACTIVE_AUGMENT_SWAP;
  else
  {
    HnAbort("invalid value of active_augment option: \"%s\"", str);
    return KHE_ACTIVE_AUGMENT_MOVE;  ** keep compiler happy **
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheActiveIntervalTooLongAugment(KHE_EJECTOR ej, int history,        */
/*    KHE_FRAME frame, KHE_ACTIVE_AUGMENT_TYPE aa)                           */
/*                                                                           */
/*  Active interval frame is too long and needs to be shortened.             */
/*                                                                           */
/*****************************************************************************/

/* ***
static bool KheActiveIntervalTooLongAugment(KHE_EJECTOR ej, int history,
  KHE_FRAME frame, KHE_LIMIT_ACTIVE_INTERVALS_MONITOR laim,
  KHE_FRAME orig_frame, KHE_ACTIVE_AUGMENT_TYPE aa)
{
  struct khe_se_resource_set_type srs_rec;
  KHE_RESOURCE_GROUP rg, not_rg;  KHE_RESOURCE r, r2;
  KHE_FRAME left_active_frame, right_active_frame;
  KHE_FRAME left_inactive_frame, right_inactive_frame, sub_frame;
  int left_active_len, right_active_len, left_inactive_len, right_inactive_len;
  int extra, len, max_lim, move_len, deviation_plus_one;

  ** get r, the problem resource, and related stuff **
  r = KheLimitActiveIntervalsMonitorResource(laim);
  rg = KheResourceTypeFullResourceGroup(KheResourceResourceType(r));
  not_rg = KheResourceSingletonResourceGroup(r);
  extra = KheEjectorCurrAugmentCount(ej) +
    5 * KheSolnDiversifier(KheEjectorSoln(ej));

  ** get r's left and right active intervals, just inside frame **
  len = KheFrameTimeGroupCount(frame);  ** yes, we omit history here **
  max_lim = KheLimitActiveIntervalsMonitorMaximum(laim);
  deviation_plus_one = max(len - max_lim + 1, 1);
  left_active_len = min(deviation_plus_one, len / 2);
  left_active_frame = KheFrameInsideBeforeSlice(frame, left_active_len);
  right_active_len = min(deviation_plus_one, len - left_active_len);
  right_active_frame = KheFrameInsideAfterSlice(frame, right_active_len);
  if( KheEjectorCurrDebug(ej) )
  {
    fprintf(stderr, "%*s  too_long, %s left_active (%s) ",
      KheEjectorCurrDebugIndent(ej), "", KheResourceId(r),
      KheFrameVi sited(left_active_frame, r, 0) ? "visited" : "unvisited");
    KheFrameDebug(left_active_frame, 1, -1, stderr);
    fprintf(stderr, ", %s right_active (%s) ", KheResourceId(r),
      KheFrameVi sited(right_active_frame, r, 0) ? "visited" : "unvisited");
    KheFrameDebug(right_active_frame, 1, -1, stderr);
    fprintf(stderr, "\n");
  }

  ** try moving r's left active interval, or part of it anyway **
  if( left_active_len > 0 && !KheFrameVi sited(left_active_frame, r, 0) )
  {
    KheFrameVisit(left_active_frame, r);
    KheForEachResource(srs_rec, rg, not_rg, false, extra, r2)
    {
      left_inactive_frame = KheFrameInactiveAtBefore(left_active_frame, r2);
      left_inactive_len = KheFrameTimeGroupCount(left_inactive_frame);
      if( KheEjectorCurrDebug(ej) )
      {
	fprintf(stderr, "%*s  %s left_inactive (%s) ",
	  KheEjectorCurrDebugIndent(ej), "",  KheResourceId(r2),
	  KheFrameVi sited(left_inactive_frame,r2,0) ? "visited" : "unvisited");
	KheFrameDebug(left_inactive_frame, 1, -1, stderr);
	fprintf(stderr, "\n");
      }
      if( left_inactive_len > 0 && !KheFrameV isited(left_inactive_frame,r2,0) )
      {
        KheFrameVisit(left_inactive_frame, r2);
	for( move_len = left_inactive_len;  move_len >= 1;  move_len-- )
	{
	  sub_frame = KheFrameInsideBeforeSlice(left_inactive_frame, move_len);
	  if( KheFrameMoveOrSwapRepair(ej, sub_frame, r, r2, orig_frame,
	      KHE_MOVE_CHECKED, aa) )
	    return true;
	}
	if( KheEjectorCurrMayRevisit(ej) )
	  KheFrameUnVisit(left_inactive_frame, r2);
      }
    }
    if( KheFramePolarity(left_active_frame) == KHE_POSITIVE )
      for( move_len = left_active_len;  move_len >= 1;  move_len-- )
      {
	sub_frame = KheFrameInsideBeforeSlice(left_active_frame, move_len);
	if( KheFrameMoveRepair(ej, sub_frame, r, NULL, KHE_MOVE_CHECKED) )
	  return true;
      }
    if( KheEjectorCurrMayRevisit(ej) )
      KheFrameUnVisit(left_active_frame, r);
  }

  ** try moving the right active interval, or part of it anyway **
  if( right_active_len > 0 && !KheFrame Visited(right_active_frame, r, 0) )
  {
    KheFrameVisit(right_active_frame, r);
    KheForEachResource(srs_rec, rg, not_rg, false, extra, r2)
    {
      right_inactive_frame = KheFrameInactiveAtAfter(frame, r2);
      right_inactive_len = KheFrameTimeGroupCount(right_inactive_frame);
      if( right_inactive_len> 0 && !KheFram eVisited(right_inactive_frame,r2,0))
      {
	KheFrameVisit(right_inactive_frame, r2);
	for( move_len = right_inactive_len;  move_len >= 1;  move_len-- )
	{
	  sub_frame = KheFrameInsideAfterSlice(right_inactive_frame, move_len);
	  if( KheFrameMoveOrSwapRepair(ej, sub_frame, r, r2, orig_frame,
		KHE_MOVE_CHECKED, aa) )
	    return true;
	}
	if( KheEjectorCurrMayRevisit(ej) )
	  KheFrameUnVisit(right_inactive_frame, r2);
      }
    }
    if( KheFramePolarity(right_active_frame) == KHE_POSITIVE )
      for( move_len = right_active_len;  move_len >= 1;  move_len-- )
      {
	sub_frame = KheFrameInsideAfterSlice(right_active_frame, move_len);
	if( KheFrameMoveRepair(ej, sub_frame, r, NULL, KHE_MOVE_CHECKED) )
	  return true;
      }
    if( KheEjectorCurrMayRevisit(ej) )
      KheFrameUnVisit(right_active_frame, r);
  }
  return false;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheActiveIntervalTooShortAugment(KHE_EJECTOR ej, int history,       */
/*    KHE_FRAME frame, KHE_ACTIVE_AUGMENT_TYPE aa)                           */
/*                                                                           */
/*  Active interval in is too short and needs to be lengthened.              */
/*                                                                           */
/*****************************************************************************/

/* ***
static bool KheActiveIntervalTooShortAugment(KHE_EJECTOR ej, int history,
  KHE_FRAME frame, KHE_LIMIT_ACTIVE_INTERVALS_MONITOR laim,
  KHE_FRAME orig_frame, KHE_ACTIVE_AUGMENT_TYPE aa)
{
  struct khe_se_resource_set_type srs_rec;
  KHE_RESOURCE_GROUP rg, not_rg;  KHE_RESOURCE r, r2;
  KHE_FRAME left_inactive_frame, right_inactive_frame;
  KHE_FRAME left_active_frame, right_active_frame, sub_frame;
  int left_inactive_len, right_inactive_len, left_active_len, right_active_len;
  int extra, min_lim, len, move_len, deviation;

  ** get r and related stuff **
  r = KheLimitActiveIntervalsMonitorResource(laim);
  rg = KheResourceTypeFullResourceGroup(KheResourceResourceType(r));
  not_rg = KheResourceSingletonResourceGroup(r);
  extra = KheEjectorCurrAugmentCount(ej) +
    5 * KheSolnDiversifier(KheEjectorSoln(ej));

  ** get r's left and right inactive frames, just outside frame **
  len = KheFrameTimeGroupCount(frame);  ** yes, we omit history here **
  min_lim = KheLimitActiveIntervalsMonitorMinimum(laim);
  deviation = max(1, min_lim - len);
  left_inactive_frame = KheFrameOutsideBeforeSlice(frame, deviation + 1);
  left_inactive_len = KheFrameTimeGroupCount(left_inactive_frame);
  right_inactive_frame = KheFrameOutsideAfterSlice(frame, deviation + 1);
  right_inactive_len = KheFrameTimeGroupCount(right_inactive_frame);

  ** try moving r's left inactive frame, or part of it anyway **
  if( left_inactive_len > 0 && !KheFrameVis ited(left_inactive_frame, r, 0) )
  {
    KheFrameVisit(left_inactive_frame, r);
    KheForEachResource(srs_rec, rg, not_rg, false, extra, r2)
    {
      left_active_frame = KheFrameActiveAtAfter(left_inactive_frame, r2);
      left_active_len = KheFrameTimeGroupCount(left_active_frame);
      if( left_active_len > 0 && !KheFrameVisi ted(left_active_frame, r2, 0) )
      {
        KheFrameVisit(left_active_frame, r2);
	for( move_len = left_active_len;  move_len >= 1;  move_len-- )
	{
	  sub_frame = KheFrameInsideAfterSlice(left_active_frame, move_len);
	  if( KheFrameMoveOrSwapRepair(ej, sub_frame, r2, r, orig_frame,
	      KHE_MOVE_CHECKED, aa) )
	    return true;
	}
	if( KheEjectorCurrMayRevisit(ej) )
	  KheFrameUnVisit(left_active_frame, r2);
      }
    }
    if( KheEjectorCurrMayRevisit(ej) )
      KheFrameUnVisit(left_inactive_frame, r);
  }

  ** try moving r's right inactive interval, or part of it anyway **
  if( right_inactive_len > 0 && !KheFrameVisi ted(right_inactive_frame, r, 0) )
  {
    KheFrameVisit(right_inactive_frame, r);
    KheForEachResource(srs_rec, rg, not_rg, false, extra, r2)
    {
      right_active_frame = KheFrameActiveAtBefore(right_inactive_frame, r2);
      right_active_len = KheFrameTimeGroupCount(right_active_frame);
      if( right_active_len > 0 && !KheFrameVisi ted(right_active_frame, r2, 0) )
      {
        KheFrameVisit(right_active_frame, r2);
	for( move_len = right_active_len;  move_len >= 1;  move_len-- )
	{
	  sub_frame = KheFrameInsideBeforeSlice(right_active_frame, move_len);
	  if( KheFrameMoveOrSwapRepair(ej, sub_frame, r2, r, orig_frame,
	      KHE_MOVE_CHECKED, aa) )
	    return true;
	}
	if( KheEjectorCurrMayRevisit(ej) )
	  KheFrameUnVisit(right_active_frame, r2);
      }
    }
    if( KheEjectorCurrMayRevisit(ej) )
      KheFrameUnVisit(right_inactive_frame, r);
  }

  ** try moving the entire interval to another resource **
  if( !KheFrameVis ited(frame, r, 0) )
  {
    KheFrameVisit(frame, r);
    KheForEachResource(srs_rec, rg, not_rg, false, extra, r2)
    {
      if( !KheFrameVis ited(frame, r2, 0) )
      {
	KheFrameVisit(frame, r2);
	sub_frame = KheFrameInactiveAtBefore(frame, r2);
	if( KheFrameTimeGroupCount(sub_frame) == KheFrameTimeGroupCount(frame) )
	{
	  ** r2 is inactive right through frame **
	  if( KheFrameMoveOrSwapRepair(ej, sub_frame, r, r2, orig_frame,
	      KHE_MOVE_CHECKED, aa) )
	    return true;
	}
	if( KheEjectorCurrMayRevisit(ej) )
	  KheFrameUnVisit(frame, r2);
      }
    }
    if( KheEjectorCurrMayRevisit(ej) )
      KheFrameUnVisit(frame, r);
  }

  ** no luck **
  return false;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheLimitActiveIntervalsAugment(KHE_EJECTOR ej, KHE_MONITOR d,       */
/*    KHE_ACTIVE_AUGMENT_TYPE aa)                                            */
/*                                                                           */
/*  Augment function for individual limit active intervals defects.          */
/*  Parameter aa is a fiddle that allows a choice of augment functions.      */
/*                                                                           */
/*  This is the version that uses frames; I'm currently experimenting        */
/*  with another version (see below) that doesn't.                           */
/*                                                                           */
/*****************************************************************************/

/* ***
static void PrevKheLimitActiveIntervalsAugment(KHE_EJECTOR ej, KHE_MONITOR d,
  KHE_ACTIVE_AUGMENT_TYPE aa)
{
  KHE_LIMIT_ACTIVE_INTERVALS_MONITOR laim;
  KHE_LIMIT_ACTIVE_INTERVALS_CONSTRAINT laic;
  KHE_FRAME frame, orig_frame;
  KHE_POLARITY po;  KHE_TIME_GROUP tg;  int busy;  KHE_RESOURCE r;
  int min_lim, len;
  int i, j, history, first_index, last_index, history_after, extra, count;
  bool too_long, all_positive, all_negative;
  HnAssert(KheMonitorTag(d) == KHE_LIMIT_ACTIVE_INTERVALS_MONITOR_TAG,
    "KheLimitActiveIntervalsAugment internal error 1");
  KheDebugAugmentBegin(DEBUG3, ej, d, "KheLimitActiveIntervalsAugment");
  laim = (KHE_LIMIT_ACTIVE_INTERVALS_MONITOR) d;
  laic = KheLimitActiveIntervalsMonitorConstraint(laim);
  all_positive = KheLimitActiveIntervalsConstraintAllPositive(laic);
  all_negative = KheLimitActiveIntervalsConstraintAllNegative(laic);
  extra = KheEjectorCurrAugmentCount(ej);
  count = KheLimitActiveInt ervalsMonitorDefectiveIntervalCount(laim);
  orig_frame = KheLimitActiveI ntervalsMonitorFrame(laim);
  for( i = 0;  i < count;  i++ )
  {
    KheLimitActiveInterva lsMonitorDefectiveInterval(laim, (extra + i) % count,
      &history, &first_index, &last_index, &history_after, &too_long);
    frame = KheFrameSlice(orig_frame, first_index, last_index + 1);

    ** in = KheIntervalMake(laim, first_index, last_index); **
    if( all_positive || all_negative )
    {
      if( too_long )
      {
        if( KheActiveIntervalTooLongAugment(ej, history, frame,
	      laim, orig_frame, aa) )
	{
	  KheDebugAugmentEnd(DEBUG3, ej, d, "KheLimitActiveIntervalsAugment",
	    true);
	  return;
	}
      }
      else
      {
        if( KheActiveIntervalTooShortAugment(ej, history, frame,
	      laim, orig_frame, aa))
	{
	  KheDebugAugmentEnd(DEBUG3, ej, d, "KheLimitActiveIntervalsAugment",
	    true);
	  return;
	}
      }
    }
    else
    {
      ** neither all positive nor all negative **
      r = KheLimitActiveIntervalsMonitorResource(laim);
      if( too_long )
      {
	for( j = first_index;  j <= last_index;  j++ )
	{
	  ** tg must be active **
	  KheLimitActiveIntervalsMonitorTimeGroupIsActive(laim, j, &tg, &po,
	    &busy);
	  if( KheOverUnderAugment(ej, r, tg, po == KHE_POSITIVE, true, false,
		MAX_MOVE, MAX_MOVE) )
	  {
	    KheDebugAugmentEnd(DEBUG3, ej, d, "KheLimitActiveIntervalsAugment",
	      true);
	    return;
	  }
	}
      }
      else
      {
	** if len == 1, try to remove the interval altogether **
	min_lim = KheLimitActiveIntervalsMonitorMinimum(laim);
	len = history + (last_index - first_index + 1) + history_after;
	HnAssert(min_lim>len,"KheLimitActiveIntervalsAugment internal error 3");
	if( len == 1 )
	{
          KheLimitActiveIntervalsMonitorTimeGroupIsActive(laim, first_index,
	    &tg, &po, &busy);
	  if( KheOverUnderAugment(ej, r, tg, po==KHE_POSITIVE, true, false,
		0, 0) )
	  {
	    KheDebugAugmentEnd(DEBUG3, ej, d, "KheLimitActiveIntervalsAugment",
	      true);
	    return;
	  }
	}

	** else need to make the interval longer, by extending at either end **
	if( first_index > 0 )
	{
          KheLimitActiveIntervalsMonitorTimeGroupIsActive(laim, first_index - 1,
	    &tg, &po, &busy);
	  if( KheOverUnderAugment(ej, r, tg, po == KHE_NEGATIVE, true, false,
		MAX_MOVE, 0) )
	  {
	    KheDebugAugmentEnd(DEBUG3, ej, d, "KheLimitActiveIntervalsAugment",
	      true);
	    return;
	  }
	}
	if( last_index < KheLimitActiveIntervalsMonitorTimeGroupCount(laim)-1 )
	{
          KheLimitActiveIntervalsMonitorTimeGroupIsActive(laim, last_index + 1,
	    &tg, &po, &busy);
	  if( KheOverUnderAugment(ej, r, tg, po == KHE_NEGATIVE, true, false,
		0, MAX_MOVE) )
	  {
	    KheDebugAugmentEnd(DEBUG3, ej, d, "KheLimitActiveIntervalsAugment",
	      true);
	    return;
	  }
	}
      }
    }
  }
  KheDebugAugmentEnd(DEBUG3, ej, d, "KheLimitActiveIntervalsAugment", false);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheLimitActiveIntervalsAugment(KHE_EJECTOR ej, KHE_MONITOR d,       */
/*    KHE_ACTIVE_AUGMENT_TYPE aa)                                            */
/*                                                                           */
/*  Augment function for individual limit active intervals defects.          */
/*  Parameter aa is a fiddle that allows a choice of augment functions.      */
/*                                                                           */
/*  For each defective sequence, if it is too long, use KheOverUnderAugment  */
/*  to try to make one of its time groups inactive, preferably the first or  */
/*  last.  If it is too short, first use KheOverUnderAugment to try to       */
/*  remove it altogether, then use KheOverUnderAugment to try to lengthen    */
/*  it at each end.                                                          */
/*                                                                           */
/*****************************************************************************/

static bool KheLimitActiveIntervalsAugment(KHE_EJECTOR ej, KHE_MONITOR d
  /* , KHE_ACTIVE_AUGMENT_TYPE aa */)
{
  KHE_LIMIT_ACTIVE_INTERVALS_MONITOR laim;
  KHE_POLARITY po;  KHE_TIME_GROUP tg;  int busy;  KHE_RESOURCE r;
  int min_lim, len, j;
  int i, history, first_index, last_index, history_after, extra, count;
  bool too_long;
  HnAssert(KheMonitorTag(d) == KHE_LIMIT_ACTIVE_INTERVALS_MONITOR_TAG,
    "KheLimitActiveIntervalsAugment internal error 1");
  KheDebugAugmentBegin(DEBUG3, ej, d, "KheLimitActiveIntervalsAugment");
  laim = (KHE_LIMIT_ACTIVE_INTERVALS_MONITOR) d;
  extra = KheEjectorCurrAugmentCount(ej);
  count = KheLimitActiveIntervalsMonitorDefectiveIntervalCount(laim);
  r = KheLimitActiveIntervalsMonitorResource(laim);
  for( i = 0;  i < count;  i++ )
  {
    KheLimitActiveIntervalsMonitorDefectiveInterval(laim, (extra + i) % count,
      &history, &first_index, &last_index, &history_after, &too_long);
    if( last_index < 0 )
      continue;
    if( too_long )
    {
      /* try to make the first time group inactive */
      KheLimitActiveIntervalsMonitorTimeGroupIsActive(laim, first_index,
	&tg, &po, &busy);
      if( KheOverUnderAugment(ej, r, tg, po == KHE_POSITIVE, true, false) )
      {
	KheDebugAugmentEnd(DEBUG3, ej, d, "KheLimitActiveIntervalsAugment",
	  true);
	return true;
      }

      /* try to make the last time group inactive, if it's not the first */
      if( last_index != first_index )
      {
	KheLimitActiveIntervalsMonitorTimeGroupIsActive(laim, last_index,
	  &tg, &po, &busy);
	if( KheOverUnderAugment(ej, r, tg, po == KHE_POSITIVE, true, false) )
	{
	  KheDebugAugmentEnd(DEBUG3, ej, d, "KheLimitActiveIntervalsAugment",
	    true);
	  return true;
	}
      }

      /* as a last resort, try the time groups in between */
      for( j = first_index + 1;  j < last_index;  j++ )
      {
	KheLimitActiveIntervalsMonitorTimeGroupIsActive(laim, j,
	  &tg, &po, &busy);
	if( KheOverUnderAugment(ej, r, tg, po == KHE_POSITIVE, true, false) )
	{
	  KheDebugAugmentEnd(DEBUG3, ej, d, "KheLimitActiveIntervalsAugment",
	    true);
	  return true;
	}
      }
    }
    else
    {
      /* if len <= 2, try to remove the interval altogether */
      min_lim = KheLimitActiveIntervalsMonitorMinimum(laim);
      len = history + (last_index - first_index + 1) + history_after;
      HnAssert(min_lim>len,"KheLimitActiveIntervalsAugment internal error 3");
      if( 1 <= len && len <= 2 )
      {
	KheLimitActiveIntervalsMonitorTimeGroupIsActive(laim, first_index,
	  &tg, &po, &busy);
	if( KheOverUnderAugment(ej, r, tg, po == KHE_POSITIVE, true, false) )
	{
	  KheDebugAugmentEnd(DEBUG3, ej, d, "KheLimitActiveIntervalsAugment",
	    true);
	  return true;
	}
      }

      /* try to make the interval longer, by extending at the left end */
      if( first_index > 0 )
      {
	KheLimitActiveIntervalsMonitorTimeGroupIsActive(laim, first_index - 1,
	  &tg, &po, &busy);
	if( KheOverUnderAugment(ej, r, tg, po == KHE_NEGATIVE, true, false) )
	{
	  KheDebugAugmentEnd(DEBUG3, ej, d, "KheLimitActiveIntervalsAugment",
	    true);
	  return true;
	}
      }

      /* try to make the interval longer, by extending at the right end */
      if( last_index < KheLimitActiveIntervalsMonitorTimeGroupCount(laim) - 1 )
      {
	KheLimitActiveIntervalsMonitorTimeGroupIsActive(laim, last_index + 1,
	  &tg, &po, &busy);
	if( KheOverUnderAugment(ej, r, tg, po == KHE_NEGATIVE, true, false) )
	{
	  KheDebugAugmentEnd(DEBUG3, ej, d, "KheLimitActiveIntervalsAugment",
	    true);
	  return true;
	}
      }
    }
  }
  KheDebugAugmentEnd(DEBUG3, ej, d, "KheLimitActiveIntervalsAugment", false);
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheLimitActiveIntervalsMoveAugment(KHE_EJECTOR ej, KHE_MONITOR d)   */
/*                                                                           */
/*  Limit active intervals augment function when active_augment is false.    */
/*                                                                           */
/*****************************************************************************/

/* *** no longer used
static bool KheLimitActiveIntervalsMoveAugment(KHE_EJECTOR ej, KHE_MONITOR d)
{
  return KheLimitActiveIntervalsAugment(ej, d, KHE_ACTIVE_AUGMENT_MOVE);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheLimitActiveIntervalsSwapAugment(KHE_EJECTOR ej, KHE_MONITOR d)   */
/*                                                                           */
/*  Limit active intervals augment function when active_augment is false.    */
/*                                                                           */
/*****************************************************************************/

/* *** no longer used
static bool KheLimitActiveIntervalsSwapAugment(KHE_EJECTOR ej, KHE_MONITOR d)
{
  return KheLimitActiveIntervalsAugment(ej, d, KHE_ACTIVE_AUGMENT_SWAP);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheLimitActiveIntervalsGroupAugment(KHE_EJECTOR ej, KHE_MONITOR d)  */
/*                                                                           */
/*  Augment function for groups of limit active intervals defects.           */
/*                                                                           */
/*****************************************************************************/

static bool KheLimitActiveIntervalsGroupAugment(KHE_EJECTOR ej, KHE_MONITOR d)
{
  KHE_GROUP_MONITOR gm;
  HnAssert(KheMonitorTag(d) == KHE_GROUP_MONITOR_TAG,
    "KheLimitActiveIntervalsGroupAugment internal error 1");
  gm = (KHE_GROUP_MONITOR) d;
  HnAssert(KheGroupMonitorDefectCount(gm) > 0,
    "KheLimitActiveIntervalsGroupAugment internal error 2");
  return KheLimitActiveIntervalsAugment(ej, KheGroupMonitorDefect(gm, 0));
  /* ***
  return KheLimitActiveIntervalsMoveAugment(ej, KheGroupMonitorDefect(gm, 0));
  *** */
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "ordinary and workload demand augments"                        */
/*                                                                           */
/*  Implementation note.  Ordinary and workload demand augments are          */
/*  treated the same, because for them it is the set of competitors which    */
/*  is the true defect, and it can have demand monitors of both kinds.       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool KheDemandDefectIsWorkloadOverload(KHE_MONITOR d, KHE_RESOURCE *r,   */
/*    KHE_TIME_GROUP *tg)                                                    */
/*                                                                           */
/*  If defect d has a workload demand defect competitor, set *r to the       */
/*  originating monitor's resource and *tg to its time group, and return     */
/*  true.  Otherwise return false.                                           */
/*                                                                           */
/*****************************************************************************/

static bool KheDemandDefectIsWorkloadOverload(KHE_MONITOR d, KHE_RESOURCE *r,
  KHE_TIME_GROUP *tg)
{
  int i;  KHE_MONITOR m;  KHE_WORKLOAD_DEMAND_MONITOR wdm;  KHE_SOLN soln;
  soln = KheMonitorSoln(d);
  KheSolnMatchingSetCompetitors(soln, d);
  for( i = 0;  i < KheSolnMatchingCompetitorCount(soln);  i++ )
  {
    m = KheSolnMatchingCompetitor(soln, i);
    if( KheMonitorTag(m) == KHE_WORKLOAD_DEMAND_MONITOR_TAG )
    {
      wdm = (KHE_WORKLOAD_DEMAND_MONITOR) m;
      *r = KheWorkloadDemandMonitorResource(wdm);
      *tg = KheWorkloadDemandMonitorTimeGroup(wdm);
      return true;
    }
  }
  *r = NULL;
  *tg = NULL;
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDemandAugment(KHE_EJECTOR ej, KHE_MONITOR d)                     */
/*                                                                           */
/*  Augment function for individual ordinary or workload demand defects.     */
/*                                                                           */
/*  If the defect has a workload demand defect competitor, try repairs       */
/*  documented in the header of KheResourceLoadDefectAugment, which in this  */
/*  case means meet moves out of the domain of the workload monitor's        */
/*  originating monitor.                                                     */
/*                                                                           */
/*  If the defect has no workload demand defect competitor, try moving       */
/*  each of clashing meets away.                                             */
/*                                                                           */
/*****************************************************************************/

static bool KheDemandAugment(KHE_EJECTOR ej, KHE_MONITOR d)
{
  KHE_TIME_GROUP tg;  KHE_MONITOR m;  KHE_MEET meet;  bool not_ejecting;
  KHE_OPTIONS_KEMPE kempe;  int i;  KHE_SOLN soln;  KHE_RESOURCE r;
  ARRAY_KHE_MEET meets;  KHE_ORDINARY_DEMAND_MONITOR odm;

  /* boilerplate */
  KheDebugAugmentBegin(DEBUG6, ej, d, "KheDemandAugment");
  HnAssert(KheMonitorTag(d) == KHE_ORDINARY_DEMAND_MONITOR_TAG ||
          KheMonitorTag(d) == KHE_WORKLOAD_DEMAND_MONITOR_TAG,
    "KheDemandAugment internal error 1");
  soln = KheEjectorSoln(ej);
  not_ejecting = KheEjectorBasicNotEjecting(ej);
  kempe = KheEjectorUseKempeMoves(ej);

  if( KheDemandDefectIsWorkloadOverload(d, &r, &tg) )
  {
    /* defect is workload overload of r in tg, so handle appropriately */
    if( KheResourceOverloadAugment(ej, r, tg, false) )
    {
      if( DEBUG6 )
	fprintf(stderr, "%*s] KheDemandAugment ret true: %.5f\n",
	  KheEjectorCurrDebugIndent(ej) + 2, "", KheCostShow(KheSolnCost(soln)));
      return true;
    }
  }
  else if( KheEjectorRepairTimes(ej) )
  {
    /* defect is ordinary overload; get meets */
    /* KheDemandDefectIsWorkloadOverload calls KheSolnMatchingSetCompetitors */
    HaArrayInit(meets, KheEjectorArena(ej));
    for( i = 0;  i < KheSolnMatchingCompetitorCount(soln);  i++ )
    {
      m = KheSolnMatchingCompetitor(soln, i);
      HnAssert(KheMonitorTag(m) == KHE_ORDINARY_DEMAND_MONITOR_TAG,
	"KheDemandAugment internal error 1");
      odm = (KHE_ORDINARY_DEMAND_MONITOR) m;
      meet = KheTaskMeet(KheOrdinaryDemandMonitorTask(odm));
      HaArrayAddLast(meets, meet);
    }
    if( DEBUG6 )
      fprintf(stderr, "  KheDemandAugment: %d meets\n", HaArrayCount(meets));

    /* move meets */
    /* MArraySort(meets, &KheMeetIncreasingDemandCmp); */
    HaArrayForEach(meets, meet, i)
    {
      /* this call to KheMeetAugment excludes moves at vizier nodes */
      if( KheMeetAugment(ej, meet, false, NULL, NULL, kempe, !not_ejecting,
	  not_ejecting, WITH_DEMAND_NODE_SWAPS) )
      {
	if( DEBUG6 )
	  fprintf(stderr, "%*s] KheDemandAugment ret true: %.5f\n",
	    KheEjectorCurrDebugIndent(ej) + 2, "",
	    KheCostShow(KheSolnCost(soln)));
	HaArrayFree(meets);
	return true;
      }
    }
    HaArrayFree(meets);
  }

  /* no luck */
  KheDebugAugmentEnd(DEBUG6, ej, d, "KheDemandAugment", false);
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDemandGroupAugment(KHE_EJECTOR ej, KHE_MONITOR d)                */
/*                                                                           */
/*  Augment function for groups of ordinary or workload demand defects.      */
/*                                                                           */
/*****************************************************************************/

static bool KheDemandGroupAugment(KHE_EJECTOR ej, KHE_MONITOR d)
{
  KHE_GROUP_MONITOR gm;
  HnAssert(KheMonitorTag(d) == KHE_GROUP_MONITOR_TAG,
    "KheDemandGroupAugment internal error 1");
  gm = (KHE_GROUP_MONITOR) d;
  HnAssert(KheGroupMonitorDefectCount(gm) > 0,
    "KheDemandGroupAugment internal error 2");
  return KheDemandAugment(ej, KheGroupMonitorDefect(gm, 0));
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "main functions"                                               */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_EJECTOR KheEjectionChainEjectorMake(KHE_OPTIONS options)             */
/*                                                                           */
/*  Make an ejector object.                                                  */
/*                                                                           */
/*****************************************************************************/

KHE_EJECTOR KheEjectionChainEjectorMake(KHE_OPTIONS options, HA_ARENA a)
{
  KHE_EJECTOR res;  /* char *active_augment; */
  /* KHE_ACTIVE_AUGMENT_TYPE active_augment; */

  /* ejector and schedules */
  res = KheEjectorMakeBegin(a);
  KheEjectorSetSchedulesFromString(res,
    KheOptionsGet(options, "es_schedules", "1+,u-"));

  /* augment types */
  KheEjectorAddAugmentType(res, KHE_AUGMENT_ORDINARY_DEMAND,
    "Ordinary demand");
  KheEjectorAddAugmentType(res, KHE_AUGMENT_WORKLOAD_DEMAND,
    "Workload demand");
  KheEjectorAddAugmentType(res, KHE_AUGMENT_SPLIT_EVENTS,
    "Split events");
  KheEjectorAddAugmentType(res, KHE_AUGMENT_ASSIGN_TIME,
    "Assign time");
  KheEjectorAddAugmentType(res, KHE_AUGMENT_PREFER_TIMES,
    "Prefer times");
  KheEjectorAddAugmentType(res, KHE_AUGMENT_SPREAD_EVENTS,
    "Spread events");
  /* no function for link events defects */
  KheEjectorAddAugmentType(res, KHE_AUGMENT_ORDER_EVENTS,
    "Order events");
  KheEjectorAddAugmentType(res, KHE_AUGMENT_ASSIGN_RESOURCE,
    "Assign resource");
  KheEjectorAddAugmentType(res, KHE_AUGMENT_PREFER_RESOURCES,
    "Prefer resources");
  KheEjectorAddAugmentType(res, KHE_AUGMENT_AVOID_SPLIT_ASSIGNMENTS,
    "Avoid split assts");
  KheEjectorAddAugmentType(res, KHE_AUGMENT_LIMIT_RESOURCES,
    "Limit resources");
  KheEjectorAddAugmentType(res, KHE_AUGMENT_AVOID_CLASHES,
    "Avoid clashes");
  KheEjectorAddAugmentType(res, KHE_AUGMENT_AVOID_UNAVAILABLE_TIMES,
    "Avoid unavailable times");
  KheEjectorAddAugmentType(res, KHE_AUGMENT_LIMIT_IDLE_TIMES,
    "Limit idle times");
  KheEjectorAddAugmentType(res, KHE_AUGMENT_CLUSTER_BUSY_TIMES,
    "Cluster busy times");
  KheEjectorAddAugmentType(res, KHE_AUGMENT_LIMIT_BUSY_TIMES,
    "Limit busy times");
  KheEjectorAddAugmentType(res, KHE_AUGMENT_LIMIT_WORKLOAD,
    "Limit workload");
  KheEjectorAddAugmentType(res, KHE_AUGMENT_LIMIT_ACTIVE_INTERVALS,
    "Limit active intervals");

  /* repair types - meet moves and assignments */
  KheEjectorAddRepairType(res, KHE_REPAIR_MEET_MOVE_UNCHECKED,
    "Unchecked meet move");
  KheEjectorAddRepairType(res, KHE_REPAIR_MEET_MOVE_CHECKED,
    "Checked meet move");
  KheEjectorAddRepairType(res, KHE_REPAIR_MEET_MOVE_EJECTING,
    "Ejecting meet move");
  KheEjectorAddRepairType(res, KHE_REPAIR_MEET_MOVE_KEMPE,
    "Kempe meet move");
  /* ***
  KheEjectorAddRepairType(res, KHE_REPAIR_BASIC_MEET_ASSIGN,
    "Basic meet assignment");
  KheEjectorAddRepairType(res, KHE_REPAIR_BASIC_MEET_MOVE,
    "Basic meet move");
  KheEjectorAddRepairType(res, KHE_REPAIR_EJECTING_MEET_ASSIGN,
    "Ejecting meet assignment");
  KheEjectorAddRepairType(res, KHE_REPAIR_EJECTING_MEET_MOVE,
    "Ejecting meet move");
  KheEjectorAddRepairType(res, KHE_REPAIR_KEMPE_MEET_MOVE,
    "Kempe meet move");
  *** */
  KheEjectorAddRepairType(res, KHE_REPAIR_FUZZY_MEET_MOVE,
    "Fuzzy meet move");

  /* repair types - task moves and assignments */
  /* ***
  KheEjectorAddRepairType(res, KHE_REPAIR_TASK_UNASSIGN,
    "Task unassignment");
  KheEjectorAddRepairType(res, KHE_REPAIR_EJECTING_TASK_ASSIGN,
    "Ejecting task assignment");
  KheEjectorAddRepairType(res, KHE_REPAIR_TASK_MOVE_UNCHECKED,
    "Unchecked task move");
  KheEjectorAddRepairType(res, KHE_REPAIR_TASK_MOVE_CHECKED,
    "Checked task move");
  KheEjectorAddRepairType(res, KHE_REPAIR_TASK_MOVE_EJECTING,
    "Ejecting task move");
  KheEjectorAddRepairType(res, KHE_REPAIR_TASK_MOVE_KEMPE,
    "Kempe task move");
  KheEjectorAddRepairType(res, KHE_REPAIR_TASK_SET_UNASSIGN,
    "Task set unassign");
  KheEjectorAddRepairType(res, KHE_REPAIR_TASK_SET_MOVE_UNCHECKED,
    "Unchecked task set move");
  KheEjectorAddRepairType(res, KHE_REPAIR_TASK_SET_MOVE_CHECKED,
    "Checked task set move");
  KheEjectorAddRepairType(res, KHE_REPAIR_TASK_SET_MOVE_EJECTING,
    "Ejecting task set move");
  KheEjectorAddRepairType(res, KHE_REPAIR_TASK_SET_MOVE_KEMPE,
    "Kempe task set move");
  KheEjectorAddRepairType(res, KHE_REPAIR_EJECTING_TASK_SET_SWAP,
    "Ejecting task set swap");
  KheEjectorAddRepairType(res, KHE_REPAIR_DAY_SET_SWAP,
    "Day-set swap");
  KheEjectorAddRepairType(res, KHE_REPAIR_DOUBLE_DAY_SET_SWAP,
    "Double day-set swap");
  KheEjectorAddRepairType(res, KHE_REPAIR_DAY_SET_UNASSIGN,
    "Day-set unassign");
  KheEjectorAddRepairType(res, KHE_REPAIR_DAY_SET_ASSIGN,
    "Day-set assign");
  *** */
  KheEjectorAddRepairType(res, KHE_REPAIR_TASK_DOUBLE_MOVE,
    "Task double move");
  KheEjectorAddRepairType(res, KHE_REPAIR_TASK_SET_MOVE,
    "Task set move");
  KheEjectorAddRepairType(res, KHE_REPAIR_TASK_SET_DOUBLE_MOVE,
    "Task set double move");
  KheEjectorAddRepairType(res, KHE_REPAIR_TASK_SET_REPLACE,
    "Task set replace");
  KheEjectorAddRepairType(res, KHE_REPAIR_TASK_SET_DOUBLE_REPLACE,
    "Task set double replace");
  KheEjectorAddRepairType(res, KHE_REPAIR_RESOURCE_MOVE,
    "Resource move");

  /* repair types - combined operations */
  KheEjectorAddRepairType(res, KHE_REPAIR_NODE_MEET_SWAP,
    "Node meet swap");
  KheEjectorAddRepairType(res, KHE_REPAIR_MEET_SPLIT,
    "Meet split");
  KheEjectorAddRepairType(res, KHE_REPAIR_SPLIT_MOVE,
    "Split move");
  KheEjectorAddRepairType(res, KHE_REPAIR_MERGE_MOVE,
    "Merge move");
  KheEjectorAddRepairType(res, KHE_REPAIR_SPLIT_TASKS_UNASSIGN,
    "Split tasks unassign and reduce");
  KheEjectorAddRepairType(res, KHE_REPAIR_MEET_BOUND,
    "Meet bound");
  KheEjectorAddRepairType(res,
    KHE_REPAIR_KEMPE_MEET_MOVE_AND_EJECTING_TASK_ASSIGN,
    "Kempe meet move + ejecting task assignment");
  KheEjectorAddRepairType(res,
    KHE_REPAIR_KEMPE_MEET_MOVE_AND_EJECTING_TASK_MOVE,
    "Kempe meet move + ejecting task move");
  KheEjectorAddRepairType(res, KHE_REPAIR_COMPLEX_IDLE_MOVE,
    "Complex idle move");

  /* augment functions - demand monitors (and groups thereof) */
  KheEjectorAddAugment(res, KHE_ORDINARY_DEMAND_MONITOR_TAG, 
    &KheDemandAugment, KHE_AUGMENT_ORDINARY_DEMAND);
  KheEjectorAddGroupAugment(res, KHE_SUBTAG_ORDINARY_DEMAND,
    &KheDemandGroupAugment, KHE_AUGMENT_ORDINARY_DEMAND);

  KheEjectorAddAugment(res, KHE_WORKLOAD_DEMAND_MONITOR_TAG, 
    &KheDemandAugment, KHE_AUGMENT_WORKLOAD_DEMAND);
  KheEjectorAddGroupAugment(res, KHE_SUBTAG_WORKLOAD_DEMAND,
    &KheDemandGroupAugment, KHE_AUGMENT_WORKLOAD_DEMAND);

  /* augment functions - event monitors (and groups thereof) */
  KheEjectorAddAugment(res, KHE_SPLIT_EVENTS_MONITOR_TAG,
    &KheSplitEventsAugment, KHE_AUGMENT_SPLIT_EVENTS);
  KheEjectorAddAugment(res, KHE_DISTRIBUTE_SPLIT_EVENTS_MONITOR_TAG,
    &KheSplitEventsAugment, KHE_AUGMENT_SPLIT_EVENTS);
  KheEjectorAddGroupAugment(res, KHE_SUBTAG_SPLIT_EVENTS, 
    &KheSplitEventsGroupAugment, KHE_AUGMENT_SPLIT_EVENTS);

  KheEjectorAddAugment(res, KHE_ASSIGN_TIME_MONITOR_TAG,
    &KheAssignTimeAugment, KHE_AUGMENT_ASSIGN_TIME);
  KheEjectorAddGroupAugment(res, KHE_SUBTAG_ASSIGN_TIME, 
    &KheAssignTimeGroupAugment, KHE_AUGMENT_ASSIGN_TIME);

  KheEjectorAddAugment(res, KHE_PREFER_TIMES_MONITOR_TAG,
    &KhePreferTimesAugment, KHE_AUGMENT_PREFER_TIMES);
  KheEjectorAddGroupAugment(res, KHE_SUBTAG_PREFER_TIMES, 
    &KhePreferTimesGroupAugment, KHE_AUGMENT_PREFER_TIMES);

  KheEjectorAddAugment(res, KHE_SPREAD_EVENTS_MONITOR_TAG,
    &KheSpreadEventsAugment, KHE_AUGMENT_SPREAD_EVENTS);
  KheEjectorAddGroupAugment(res, KHE_SUBTAG_SPREAD_EVENTS, 
    &KheSpreadEventsGroupAugment, KHE_AUGMENT_SPREAD_EVENTS);

  KheEjectorAddAugment(res, KHE_ORDER_EVENTS_MONITOR_TAG,
    &KheOrderEventsAugment, KHE_AUGMENT_ORDER_EVENTS);
  KheEjectorAddGroupAugment(res, KHE_SUBTAG_ORDER_EVENTS,
    &KheOrderEventsGroupAugment, KHE_AUGMENT_ORDER_EVENTS);

  /* augment functions - event resource monitors (and groups thereof) */
  KheEjectorAddAugment(res, KHE_ASSIGN_RESOURCE_MONITOR_TAG,
    &KheAssignResourceAugment, KHE_AUGMENT_ASSIGN_RESOURCE);
  KheEjectorAddGroupAugment(res, KHE_SUBTAG_ASSIGN_RESOURCE,
    &KheAssignResourceGroupAugment, KHE_AUGMENT_ASSIGN_RESOURCE);

  KheEjectorAddAugment(res, KHE_PREFER_RESOURCES_MONITOR_TAG,
    &KhePreferResourcesAugment, KHE_AUGMENT_PREFER_RESOURCES);
  KheEjectorAddGroupAugment(res, KHE_SUBTAG_PREFER_RESOURCES,
    &KhePreferResourcesGroupAugment, KHE_AUGMENT_PREFER_RESOURCES);

  KheEjectorAddAugment(res, KHE_AVOID_SPLIT_ASSIGNMENTS_MONITOR_TAG,
    &KheAvoidSplitAssignmentsAugment, KHE_AUGMENT_AVOID_SPLIT_ASSIGNMENTS);

  KheEjectorAddAugment(res, KHE_LIMIT_RESOURCES_MONITOR_TAG,
    &KheLimitResourcesAugment, KHE_AUGMENT_LIMIT_RESOURCES);
  KheEjectorAddGroupAugment(res, KHE_SUBTAG_LIMIT_RESOURCES,
    &KheLimitResourcesGroupAugment, KHE_AUGMENT_LIMIT_RESOURCES);

  /* augment functions - resource monitors (and groups thereof) */
  KheEjectorAddAugment(res, KHE_AVOID_CLASHES_MONITOR_TAG,
    &KheAvoidClashesAugment, KHE_AUGMENT_AVOID_CLASHES);
  KheEjectorAddGroupAugment(res, KHE_SUBTAG_AVOID_CLASHES,
    &KheAvoidClashesGroupAugment, KHE_AUGMENT_AVOID_CLASHES);

  KheEjectorAddAugment(res, KHE_AVOID_UNAVAILABLE_TIMES_MONITOR_TAG,
    &KheAvoidUnavailableTimesAugment, KHE_AUGMENT_AVOID_UNAVAILABLE_TIMES);
  KheEjectorAddGroupAugment(res, KHE_SUBTAG_AVOID_UNAVAILABLE_TIMES,
    &KheAvoidUnavailableTimesGroupAugment, KHE_AUGMENT_AVOID_UNAVAILABLE_TIMES);

  KheEjectorAddAugment(res, KHE_LIMIT_IDLE_TIMES_MONITOR_TAG,
    &KheLimitIdleTimesAugment, KHE_AUGMENT_LIMIT_IDLE_TIMES);
  KheEjectorAddGroupAugment(res, KHE_SUBTAG_LIMIT_IDLE_TIMES,
    &KheLimitIdleTimesGroupAugment, KHE_AUGMENT_LIMIT_IDLE_TIMES);

  KheEjectorAddAugment(res, KHE_CLUSTER_BUSY_TIMES_MONITOR_TAG,
    &KheClusterBusyTimesAugment, KHE_AUGMENT_CLUSTER_BUSY_TIMES);
  KheEjectorAddGroupAugment(res, KHE_SUBTAG_CLUSTER_BUSY_TIMES,
    &KheClusterBusyTimesGroupAugment, KHE_AUGMENT_CLUSTER_BUSY_TIMES);

  KheEjectorAddAugment(res, KHE_LIMIT_BUSY_TIMES_MONITOR_TAG,
    &KheLimitBusyTimesAugment, KHE_AUGMENT_LIMIT_BUSY_TIMES);
  KheEjectorAddGroupAugment(res, KHE_SUBTAG_LIMIT_BUSY_TIMES,
    &KheLimitBusyTimesGroupAugment, KHE_AUGMENT_LIMIT_BUSY_TIMES);

  KheEjectorAddAugment(res, KHE_LIMIT_WORKLOAD_MONITOR_TAG,
    &KheLimitWorkloadAugment, KHE_AUGMENT_LIMIT_WORKLOAD);
  KheEjectorAddGroupAugment(res, KHE_SUBTAG_LIMIT_WORKLOAD,
    &KheLimitWorkloadGroupAugment, KHE_AUGMENT_LIMIT_WORKLOAD);

  KheEjectorAddAugment(res, KHE_LIMIT_ACTIVE_INTERVALS_MONITOR_TAG,
    &KheLimitActiveIntervalsAugment, KHE_AUGMENT_LIMIT_ACTIVE_INTERVALS);
  /* ***
  active_augment = KheGetActiveAugmentOptions(options);
  if( active_augment == KHE_ACTIVE_AUGMENT_MOVE )
    KheEjectorAddAugment(res, KHE_LIMIT_ACTIVE_INTERVALS_MONITOR_TAG,
      &KheLimitActiveIntervalsMoveAugment, KHE_AUGMENT_LIMIT_ACTIVE_INTERVALS);
  else
    KheEjectorAddAugment(res, KHE_LIMIT_ACTIVE_INTERVALS_MONITOR_TAG,
      &KheLimitActiveIntervalsSwapAugment, KHE_AUGMENT_LIMIT_ACTIVE_INTERVALS);
  *** */
  KheEjectorAddGroupAugment(res, KHE_SUBTAG_LIMIT_ACTIVE_INTERVALS,
    &KheLimitActiveIntervalsGroupAugment, KHE_AUGMENT_LIMIT_ACTIVE_INTERVALS);

  /* wrap up */
  KheEjectorMakeEnd(res);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_EJECTOR KheEjectionChainEjectorOption(KHE_OPTIONS options, char *key)*/
/*                                                                           */
/*  Obtain an ejector from options, or create one if not present.            */
/*                                                                           */
/*****************************************************************************/

KHE_EJECTOR KheEjectionChainEjectorOption(KHE_OPTIONS options, char *key)
{
  void *value;  KHE_EJECTOR res;
  value = KheOptionsGetObject(options, key, NULL);
  if( value != NULL )
    res = (KHE_EJECTOR) value;
  else
  {
    res = KheEjectionChainEjectorMake(options, KheOptionsArena(options));
    KheOptionsSetObject(options, key, (void *) res);
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheEjectionChainNodeRepairTimes(KHE_NODE parent_node,               */
/*    KHE_OPTIONS options)                                                   */
/*                                                                           */
/*  Use an ejection chain to repair the time assignments of the meets of     */
/*  the descendants of parent_node.  The repair operation are Kempe meet     */
/*  moves (required to preserve zones, where present) and node swaps if      */
/*  nodes lie in layers.  Return true if any progress was made.              */
/*                                                                           */
/*  The ejector used is es_ejector1 from options, which will be created      */
/*  if it does not already exist.                                            */
/*                                                                           */
/*  If the vizier_node option is true, insert and split a vizier node        */
/*  before starting work, and remove it afterwards.                          */
/*                                                                           */
/*****************************************************************************/

bool KheEjectionChainNodeRepairTimes(KHE_NODE parent_node, KHE_OPTIONS options)
{
  KHE_GROUP_MONITOR kempe_gm, start_gm, limit_gm;  KHE_NODE viz_node;
  KHE_SOLN soln;  KHE_EJECTOR ej;  bool res, vizier_node, no_node_regularity;

  if( DEBUG1 )
  {
    fprintf(stderr, "[ KheEjectionChainNodeRepairTimes(");
    KheNodeDebug(parent_node, 1, -1, stderr);
    fprintf(stderr, ")\n");
  }

  /* get some options used by this function */
  vizier_node = KheOptionsGetBool(options, "es_vizier_node", false);
  no_node_regularity = KheOptionsGetBool(options, "ts_no_node_regularity",
    false);
  /* bool time_vizier_node = KheOptionsEjectorVizierNode(options); */
  /* bool time_node_regularity = KheOptionsTimeNodeRegularity(options); */

  /* get an ejector */
  /* ***
  ej = (KHE_EJECTOR) KheOptionsGetObject(options, "es_ejector1", NULL);
  if( ej == NULL )
    ej = KheEjectionCh ainEjectorMake(options);
  *** */
  ej = KheEjectionChainEjectorOption(options, "es_ejector1");

  /* set the control options */
  KheOptionsSetBool(options, "es_repair_times", true);
  KheOptionsSetObject(options, "es_limit_node",
    KheNodeIsCycleNode(parent_node) ? NULL : parent_node);
  KheOptionsSetBool(options, "es_repair_resources", false);

  /* ***
  KheOptionsSetEjectorRepairTimes(options, true);
  KheOptionsSetEjectorLimitNode(options,
    KheNodeIsCycleNode(parent_node) ? NULL : parent_node);
  KheOptionsSetEjectorRepairResources(options, false);
  *** */

  /* insert and split a vizier node, if required */
  if( vizier_node )
  {
    viz_node = KheNodeVizierMake(parent_node);
    KheNodeMeetSplit(viz_node, false);
  }

  /* build the required group monitors and solve */
  soln = KheNodeSoln(parent_node);
  KheEjectionChainPrepareMonitors(soln, options);
  kempe_gm = KheKempeDemandGroupMonitorMake(soln);
  start_gm = KheNodeTimeRepairStartGroupMonitorMake(parent_node);
  limit_gm = KheGroupEventMonitors(soln, KHE_ASSIGN_TIME_MONITOR_TAG,
    KHE_SUBTAG_ASSIGN_TIME);
  if( DEBUG2 )
  {
    fprintf(stderr, "  initial defects:\n");
    KheGroupMonitorDefectDebug(start_gm, 2, 4, stderr);
  }
  KheEjectorSolveBegin(ej, start_gm, (KHE_GROUP_MONITOR) soln, options);
  KheEjectorAddMonitorCostLimitReducing(ej, (KHE_MONITOR) limit_gm);
  res = KheEjectorSolveEnd(ej);
  if( DEBUG2 )
  {
    fprintf(stderr, "  final defects:\n");
    KheGroupMonitorDefectDebug(start_gm, 2, 4, stderr);
  }

  /* remove any vizier node and make sure the parent is fully zoned */
  if( vizier_node )
  {
    KheNodeVizierDelete(parent_node);
    if( !no_node_regularity )
      KheNodeExtendZones(parent_node);
  }

  /* clean up, including deleting the group monitors, and return */
  /* ***
  if( KheOptionsGetObject(options, "es_ejector1", NULL) == NULL )
    KheEjecto rDelete(ej);
  *** */
  KheGroupMonitorDelete(kempe_gm);
  KheGroupMonitorDelete(start_gm);
  KheGroupMonitorDelete(limit_gm);
  KheEjectionChainUnPrepareMonitors(soln);
  if( DEBUG1 )
    fprintf(stderr, "] KheEjectionChainNodeRepairTimes returning %s\n",
      res ? "true" : "false");
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheEjectionChainLayerRepairTimes(KHE_LAYER layer,                   */
/*    KHE_OPTIONS options)                                                   */
/*                                                                           */
/*  Use an ejection chain to repair the time assignments of the meets of     */
/*  layer.  In other respects the same as KheEjectionChainNoderRepairTimes.  */
/*                                                                           */
/*  The ejector used is es_ejector1 from options, which will be created      */
/*  if it does not already exist.                                            */
/*                                                                           */
/*  If the time_vizier_node option is true, insert and split a vizier        */
/*  node before starting work, and remove it afterwards.                     */
/*                                                                           */
/*****************************************************************************/

bool KheEjectionChainLayerRepairTimes(KHE_LAYER layer, KHE_OPTIONS options)
{
  KHE_GROUP_MONITOR kempe_gm, start_gm, limit_gm;
  KHE_SOLN soln;  KHE_EJECTOR ej;  KHE_NODE parent_node, viz_node;
  bool res, vizier_node, no_node_regularity, layer_repair_long;

  if( DEBUG1 )
  {
    fprintf(stderr, "[ KheEjectionChainLayerRepairTimes(");
    KheLayerDebug(layer, 1, -1, stderr);
    fprintf(stderr, ")\n");
    KheLayerDebug(layer, 2, 2, stderr);
  }

  /* get some options used by this function */
  vizier_node = KheOptionsGetBool(options, "es_vizier_node", false);
  no_node_regularity = KheOptionsGetBool(options, "ts_no_node_regularity",
    false);
  layer_repair_long = KheOptionsGetBool(options, "es_layer_repair_long", false);
  /* bool time_vizier_node = KheOptionsEjectorVizierNode(options); */
  /* bool time_node_regularity = KheOptionsTimeNodeRegularity(options); */

  /* get an ejector */
  /* ***
  ej = (KHE_EJECTOR) KheOptionsGetObject(options, "es_ejector1", NULL);
  if( ej == NULL )
    ej = KheEjectionChainEjec torMake(options);
  *** */
  ej = KheEjectionChainEjectorOption(options, "es_ejector1");

  /* set the control options */
  parent_node = KheLayerParentNode(layer);
  KheOptionsSetBool(options, "es_repair_times", true);
  KheOptionsSetObject(options, "es_limit_node",
    KheNodeIsCycleNode(parent_node) ? NULL : parent_node);
  KheOptionsSetBool(options, "es_repair_resources", false);

  /* ***
  KheOptionsSetEjectorRepairTimes(options, true);
  KheOptionsSetEjectorLimitNode(options,
    KheNodeIsCycleNode(parent_node) ? NULL : parent_node);
  KheOptionsSetEjectorRepairResources(options, false);
  *** */

  /* insert and split a vizier node, if required */
  if( vizier_node )
  {
    viz_node = KheNodeVizierMake(parent_node);
    KheNodeMeetSplit(viz_node, false);
  }

  /* build the required group monitors and solve */
  soln = KheLayerSoln(layer);
  KheEjectionChainPrepareMonitors(soln, options);
  kempe_gm = KheKempeDemandGroupMonitorMake(soln);
  /* if( KheOptionsTimeLayerRepairLong(options) ) */
  if( layer_repair_long )
    start_gm = KheLayerTimeRepairLongStartGroupMonitorMake(layer);
  else
    start_gm = KheLayerTimeRepairStartGroupMonitorMake(layer);
  limit_gm = KheGroupEventMonitors(soln, KHE_ASSIGN_TIME_MONITOR_TAG,
    KHE_SUBTAG_ASSIGN_TIME);
  if( DEBUG2 )
  {
    fprintf(stderr, "  initial defects (soln cost %.5f):\n",
      KheCostShow(KheSolnCost(soln)));
    KheGroupMonitorDefectDebug(start_gm, 2, 4, stderr);
  }
  KheEjectorSolveBegin(ej, start_gm, (KHE_GROUP_MONITOR) soln, options);
  KheEjectorAddMonitorCostLimitReducing(ej, (KHE_MONITOR) limit_gm);
  res = KheEjectorSolveEnd(ej);
  if( DEBUG2 )
  {
    fprintf(stderr, "  final defects (soln cost %.5f):\n",
      KheCostShow(KheSolnCost(soln)));
    KheGroupMonitorDefectDebug(start_gm, 2, 4, stderr);
  }

  /* remove any vizier node and make sure the parent is fully zoned */
  if( vizier_node )
  {
    KheNodeVizierDelete(parent_node);
    if( !no_node_regularity )
      KheNodeExtendZones(parent_node);
  }

  /* clean up, including deleting the group monitors, and return */
  /* ***
  if( KheOptionsGetObject(options, "es_ejector1", NULL) == NULL )
    KheEjectorDele te(ej);
  *** */
  KheGroupMonitorDelete(kempe_gm);
  KheGroupMonitorDelete(start_gm);
  KheGroupMonitorDelete(limit_gm);
  KheEjectionChainUnPrepareMonitors(soln);
  if( DEBUG1 )
    fprintf(stderr, "] KheEjectionChainLayerRepairTimes returning %s\n",
      res ? "true" : "false");
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheWorkloadLimitsAreTight(KHE_EJECTOR ej, KHE_TASKING tasking)      */
/*                                                                           */
/*  Return true if workload limits are tight enough to justify using         */
/*  swaps during repair.                                                     */
/*                                                                           */
/*****************************************************************************/

/* ** decided to always swap now
static bool KheWorkloadLimitsAreTight(KHE_EJECTOR ej, KHE_TASKING tasking,
  float threshold, KHE_OPTIONS options)
{
  KHE_FRAME frame;  KHE_RESOURCE_GROUP rg;  KHE_RESOURCE_TYPE rt;
  int supply, demand, i;  KHE_SOLN soln;  KHE_EVENT_TIMETABLE_MONITOR etm;
  KHE_TASK task;  bool res;  KHE_RESOURCE r;  KHE_FRAME_WORKLOAD fw;

  ** temporary; sorting this out is sti ll to do **
  ** there is an option for this: es_task_seq_swaps_off **
  ** return true; **

  ** boilerplate **
  soln = KheTaskingSoln(tasking);
  if( DEBUG23 )
    fprintf(stderr, "  KheWorkloadLimitsAreTight(%s, %.1f)",
      KheInstanceId(KheSolnInstance(soln)), threshold);
  rt = KheTaskingResourceType(tasking);
  if( rt == NULL )
  {
    if( DEBUG23 )
      fprintf(stderr, " returning false (rt missing)\n");
    return false;
  }
  et m = (KHE_EVENT_TIMETABLE_MONITOR)
    KheOptionsGetObject(options, "gs_event_timetable_monitor", NULL);
  if( etm == NULL )
  {
    if( DEBUG23 )
      fprintf(stderr, " returning false (etm missing)\n");
    return false;
  }
  frame = KheOptionsFrame(options, "gs_common_frame", soln);
  if( KheFrameIsNull(frame) )
  {
    if( DEBUG23 )
      fprintf(stderr, " returning false (frame missing)\n");
    return false;
  }

  ** get total supply **
  rg = KheResourceTypeFullResourceGroup(rt);
  supply = 0;
  fw = KheFrame WorkloadMake(frame, rt, etm);
  for( i = 0;  i < KheResourceGroupResourceCount(rg);  i++ )
  {
    r = KheResourceGroupResource(rg, i);
    supply += KheFrameResource MaxBusyTimes(frame, fw, r);
  }
  KheFrameWorkloadDelete(fw);

  ** get total demand **
  demand = 0;
  for( i = 0;  i < KheTaskingTaskCount(tasking);  i++ )
  {
    task = KheTaskingTask(tasking, i);
    if( KheTaskNonAss ignmentHasCost(task, false) )
      demand += KheTaskDuration(task);
  }

  ** return true if demand is within the threshold of supply **
  res = (float) demand >= supply * threshold;
  if( DEBUG23 )
    fprintf(stderr, " demand %d, supply %d: %s\n",
      demand, supply, res ? "true" : "false");
  return res;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheEjectionChainRepairResources(KHE_TASKING tasking,                */
/*    KHE_OPTIONS options)                                                   */
/*                                                                           */
/*  Ejection chain local search for improving resource assts.  If the        */
/*  rs_invariant option is true, preserve the resource assignment            */
/*  invariant.                                                               */
/*                                                                           */
/*  The ejector used is es_ejector1 from options, which will be created      */
/*  if it does not already exist.                                            */
/*                                                                           */
/*****************************************************************************/

bool KheEjectionChainRepairResources(KHE_TASKING tasking, KHE_OPTIONS options)
{
  KHE_EJECTOR ej;  KHE_GROUP_MONITOR kempe_gm, start_gm, limit_gm;
  KHE_SOLN soln;  bool res, resource_invariant;  /* char *opt; */

  /* return immediately if not wanted */
  if( KheOptionsGetBool(options, "rs_eject_off", false) )
    return false;

  if( DEBUG1 )
    fprintf(stderr, "[ KheEjectionChainRepairResources(tasking)\n");

  /* get options used by this function */
  resource_invariant = KheOptionsGetBool(options, "rs_invariant", false);

  /* get an ejector */
  /* ***
  ej = (KHE_EJECTOR) KheOptionsGetObject(options, "es_ejector2", NULL);
  if( ej == NULL )
    ej = KheEjectionChainEject orMake(options);
  *** */
  ej = KheEjectionChainEjectorOption(options, "es_ejector2");

  /* set the control options */
  KheOptionsSetBool(options, "es_repair_times", false);
  KheOptionsSetObject(options, "es_limit_node", NULL);
  KheOptionsSetBool(options, "es_repair_resources", true);

  /* set the non-improvement time limit */
  /* KheOptionsTimeLimitBegin(options, "non-improvement", 10.0); */

  /* use task swaps only if asked or if workload is tight */
  /* ***
  opt = KheOptionsGet(options, "es_task_seq_swaps_off", "auto");
  if( strcmp(opt, "auto") == 0 )
    KheOptionsSetBool(options, "es_task_seq_swaps_off",
      !KheWorkloadLimitsAreTight(ej, tasking, 0.95, options));
  *** */
  /* *** will be set when first asked for
  if( KheInstanceModel(KheSolnInstance(soln)) == KHE_MODEL_EMPLOYEE_SCHEDULE )
    KheOptionsSetBool(options, "time_pa rtition", true);
  *** */
  /* *** promoted to general
  KheOptionsSetBool(options, "es_use_eve nt_timetable_monitor", true);
  *** */

  /* ***
  KheOptionsSetEjectorRepairTimes(options, false);
  KheOptionsSetEjectorLimitNode(options, NULL);
  KheOptionsSetEjectorRepairResources(options, true);
  soln = KheTaskingSoln(tasking);
  if( KheInstanceModel(KheSolnInstance(soln)) == KHE_MODEL_EMPLOYEE_SCHEDULE )
    KheOptionsSetResourceTimePartition(options, true);
  KheOptionsSetEjectorUseEventTimetableMonitor(options, true);
  *** */

  /* build the required group monitors and solve */
  soln = KheTaskingSoln(tasking);
  KheEjectionChainPrepareMonitors(soln, options);
  kempe_gm = KheKempeDemandGroupMonitorMake(soln);
  start_gm = KheTaskingStartGroupMonitorMake(tasking);
  if( DEBUG2 )
  {
    fprintf(stderr, "  initial defects:\n");
    KheGroupMonitorDefectDebug(start_gm, 2, 4, stderr);
  }
  KheEjectorSolveBegin(ej, start_gm, (KHE_GROUP_MONITOR) soln, options);
  limit_gm = NULL;
  if( resource_invariant )
  {
    limit_gm = KheAllDemandGroupMonitorMake(soln);
    KheEjectorAddMonitorCostLimitReducing(ej, (KHE_MONITOR) limit_gm);
  }
  res = KheEjectorSolveEnd(ej);
  if( DEBUG2 )
  {
    fprintf(stderr, "  final defects:\n");
    KheGroupMonitorDefectDebug(start_gm, 2, 4, stderr);
  }

  /* clean up, including deleting the group monitors, and return */
  /* KheOptionsTimeLimitEnd(options, "non-improvement"); */
  /* ***
  if( KheOptionsGetObject(options, "es_ejector2", NULL) == NULL )
    KheEjectorDe lete(ej);
  *** */
  KheGroupMonitorDelete(kempe_gm);
  if( resource_invariant )
    KheGroupMonitorDelete(limit_gm);
  KheGroupMonitorDelete(start_gm);
  KheEjectionChainUnPrepareMonitors(soln);
  if( DEBUG1 )
    fprintf(stderr, "] KheEjectionChainRepairResources returning %s\n",
      res ? "true" : "false");
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheEjectionChainRepairResources(KHE_TASKING tasking,                */
/*    KHE_OPTIONS options)                                                   */
/*                                                                           */
/*  Ejection chain local search for improving resource assts.  If the        */
/*  rs_invariant option is true, preserve the resource assignment            */
/*  invariant.                                                               */
/*                                                                           */
/*  The ejector used is es_ejector1 from options, which will be created      */
/*  if it does not already exist.                                            */
/*                                                                           */
/*****************************************************************************/

bool KheEjectionChainRepairInitialResourceAssignment(
  KHE_GROUP_MONITOR limit_resources_gm, KHE_OPTIONS options)
{
  KHE_EJECTOR ej;  KHE_SOLN soln;  bool res, save_opt;  /* char *opt; */

  /* return immediately if not wanted */
  if( KheOptionsGetBool(options, "rs_eject_off", false) )
    return false;

  if( DEBUG24 )
  {
    fprintf(stderr, "[ KheEjectionChainRepairInitialResourceAssignment()\n");
    KheGroupMonitorDebug(limit_resources_gm, 3, 2, stderr);
  }

  /* get an ejector */
  ej = KheEjectionChainEjectorOption(options, "es_ejector2");

  /* set the control options */
  KheOptionsSetBool(options, "es_repair_times", false);
  KheOptionsSetObject(options, "es_limit_node", NULL);
  KheOptionsSetBool(options, "es_repair_resources", true);
  save_opt = KheOptionsGetBool(options, "es_group_limit_resources_off", false);
  KheOptionsSetBool(options, "es_group_limit_resources_off", true);

  /* set the non-improvement time limit */
  /* KheOptionsTimeLimitBegin(options, "non-improvement", 0.5); */

  /* build the required group monitors and solve */
  soln = KheMonitorSoln((KHE_MONITOR) limit_resources_gm);
  /* *** this no longer works! because ej has not been set to soln yet
  if( DEBUG25 )
  {
    int i;  KHE_RESOURCE r;
    KHE_INSTANCE ins = KheSolnInstance(soln);
    for( i = 0;  i < KheInstanceResourceCount(ins);  i++ )
    {
      r = KheInstanceResource(ins, i);
      KheNoClashesCheck(ej, r);
    }
  }
  *** */
  KheEjectionChainPrepareMonitors(soln, options);
  if( DEBUG24 )
  {
    fprintf(stderr, "  initial defects:\n");
    KheGroupMonitorDefectDebug(limit_resources_gm, 2, 4, stderr);
  }
  /* ***
  KheEjectorSolveBegin(ej, limit_resources_gm, limit_resources_gm, options);
  *** */
  KheEjectorSolveBegin(ej, limit_resources_gm, (KHE_GROUP_MONITOR) soln,
    options);
  res = KheEjectorSolveEnd(ej);
  if( DEBUG2 )
  {
    fprintf(stderr, "  final defects:\n");
    KheGroupMonitorDefectDebug(limit_resources_gm, 2, 4, stderr);
  }

  /* clean up, including deleting the group monitors, and return */
  /* KheOptionsTimeLimitEnd(options, "non-improvement"); */
  KheOptionsSetBool(options, "es_group_limit_resources_off", save_opt);
  KheEjectionChainUnPrepareMonitors(soln);
  if( DEBUG1 )
   fprintf(stderr, "] %s returning %s\n",
     "KheEjectionChainRepairInitialResourceAssignment",
	res ? "true" : "false");
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskHasAssignResourceConstraint(KHE_TASK task)                   */
/*                                                                           */
/*  Return true if task is monitored by an assign resource constraint of     */
/*  non-zero weight.                                                         */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused
static bool KheTaskHasAssignResourceConstraint(KHE_TASK task)
{
  KHE_EVENT_RESOURCE er;  int i;  KHE_CONSTRAINT c;
  er = KheTaskEventResource(task);
  for( i = 0;  i < KheEventResourceConstraintCount(er);  i++ )
  {
    c = KheEventResourceConstraint(er, i);
    if( KheConstraintTag(c) == KHE_ASSIGN_RESOURCE_CONSTRAINT_TAG &&
	KheConstraintWeight(c) > 0 )
      return true;
  }
  return false;
}
*** */
