
/*****************************************************************************/
/*                                                                           */
/*  THE KHE HIGH SCHOOL TIMETABLING ENGINE                                   */
/*  COPYRIGHT (C) 2010 Jeffrey H. Kingston                                   */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 3, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         khe_sr_requested.c                                         */
/*  DESCRIPTION:  Satisfying requested task assignments                      */
/*                                                                           */
/*****************************************************************************/
#include "khe_solvers.h"
#include <limits.h>

#define DEBUG1 1
#define DEBUG2 1
#define DEBUG3 1
#define DEBUG4 0
#define DEBUG4_ID "A10"
#define DEBUG5 0	/* clash checking */


typedef HA_ARRAY(KHE_MONITOR) ARRAY_KHE_MONITOR;

/*****************************************************************************/
/*                                                                           */
/*  int KheMonitorDecreasingWeightCmp(const void *t1, const void *t2)        */
/*                                                                           */
/*  Comparison function for sorting an array of monitors by decreasing       */
/*  weight.                                                                  */
/*                                                                           */
/*****************************************************************************/

static int KheMonitorDecreasingWeightCmp(const void *t1, const void *t2)
{
  KHE_COST weight1, weight2;
  KHE_CONSTRAINT c1, c2;
  KHE_MONITOR m1 = * (KHE_MONITOR *) t1;
  KHE_MONITOR m2 = * (KHE_MONITOR *) t2;
  c1 = KheMonitorConstraint(m1);
  c2 = KheMonitorConstraint(m2);
  weight1 = KheConstraintCombinedWeight(c1);
  weight2 = KheConstraintCombinedWeight(c2);
  return KheCostCmp(weight2, weight1);  /* reverse order, decreasing weight */
}


/*****************************************************************************/
/*                                                                           */
/*  void TryTime(KHE_SOLN soln, KHE_FRAME days_frame,                        */
/*    KHE_RESOURCE r, KHE_RESOURCE_TIMETABLE_MONITOR rtm,                    */
/*    KHE_TIME t, KHE_EVENT_TIMETABLE_MONITOR etm,                           */
/*    KHE_TASK *best_task, KHE_COST *best_cost, int *attempts)               */
/*                                                                           */
/*  Try assigning r to a task at time t, and update *best_task, *best_cost,  */
/*  and *attempts appropriately.                                             */
/*                                                                           */
/*****************************************************************************/

static void TryTime(KHE_SOLN soln, KHE_FRAME days_frame, KHE_RESOURCE r,
  KHE_TIME t, KHE_EVENT_TIMETABLE_MONITOR etm,
  KHE_TASK *best_task, KHE_COST *best_cost, int *attempts)
{
  int i, j, extra, index, inc;  KHE_MEET meet;  KHE_TIME_GROUP tg;
  KHE_TASK task;  KHE_RESOURCE_TIMETABLE_MONITOR rtm;

  /* if r is already busy at time t, return immediately */
  rtm = KheResourceTimetableMonitor(soln, r);
  index = KheFrameTimeIndex(days_frame, t);
  tg = KheFrameTimeGroup(days_frame, index);
  if( !KheResourceTimetableMonitorTimeGroupAvailable(rtm, tg, false) )
    return;

  /* try a selection of tasks running at time t */
  extra = KheResourceInstanceIndex(r);
  for( i = 0;  i < KheEventTimetableMonitorTimeMeetCount(etm, t);  i++ )
  {
    meet = KheEventTimetableMonitorTimeMeet(etm, t, i);
    inc = (KheMeetTaskCount(meet) >= 10 ? 5 : 1);
    for( j = 0;  j < KheMeetTaskCount(meet);  j += inc )
    {
      index = (j + extra) % KheMeetTaskCount(meet);
      task = KheTaskProperRoot(KheMeetTask(meet, index));
      if( KheTaskAsst(task) == NULL &&
	  KheResourceTimetableMonitorTaskAvailableInFrame(rtm, task,
	    days_frame, NULL) )
      {
	(*attempts)++;
	if( KheTaskAssignResource(task, r) )
	{
	  if( *best_task == NULL || KheSolnCost(soln) < *best_cost )
	    *best_task = task, *best_cost = KheSolnCost(soln);
	  KheTaskUnAssignResource(task);
	}
      }
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheBeginTrial(KHE_TASK *best_task, KHE_COST *best_cost,             */
/*    int *attempts)                                                         */
/*                                                                           */
/*  Begin a trial which tries to find the task whose assignment gives        */
/*  the best cost.                                                           */
/*                                                                           */
/*****************************************************************************/

static void KheBeginTrial(KHE_TASK *best_task, KHE_COST *best_cost,
  int *attempts)
{
  *best_task = NULL;
  *best_cost = KheCost(INT_MAX, INT_MAX);
  *attempts = 0;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheEndTrial(KHE_SOLN soln, KHE_RESOURCE r, KHE_TASK best_task,      */
/*    KHE_COST best_cost, int attempts)                                      */
/*                                                                           */
/*  End a trial.                                                             */
/*                                                                           */
/*****************************************************************************/

static void KheEndTrial(KHE_SOLN soln, KHE_RESOURCE r, KHE_TASK best_task,
  KHE_COST best_cost, int attempts)
{
  if( best_task != NULL )
  {
    KheTaskAssignResource(best_task, r);
    if( DEBUG2 )
    {
      fprintf(stderr, " tried %d tasks: ", attempts);
      KheTaskDebug(best_task, 2, -1, stderr);
      fprintf(stderr, " := %s (soln cost now %.5f)\n", KheResourceId(r),
	KheCostShow(KheSolnCost(soln)));
    }
  }
  else
  {
    if( DEBUG2 )
      fprintf(stderr, " tried %d tasks: -\n", attempts);
  }
  if( DEBUG4 && strcmp(KheResourceId(r), DEBUG4_ID) == 0 )
  {
    KHE_RESOURCE_TIMETABLE_MONITOR rtm;
    rtm = KheResourceTimetableMonitor(soln, r);
    KheResourceTimetableMonitorPrintTimetable(rtm, 10, 6, stderr);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool MakeBestAsstAtTime(KHE_SOLN soln, KHE_FRAME days_frame,             */
/*    KHE_RESOURCE r, KHE_TIME t, KHE_EVENT_TIMETABLE_MONITOR etm)           */
/*                                                                           */
/*  Make the best possible assignment of a task running at time t to r,      */
/*  that is, if r is not already busy then.  Obtain the events running       */
/*  at time t from etm.  If an assignment is actually made, return true.     */
/*                                                                           */
/*  Actually, if the meet has more than 10 tasks, we only try every fifth    */
/*  task, cyclically starting from a random point.  This is because most     */
/*  tasks are symmetrical and it takes too long to try them all.             */
/*                                                                           */
/*****************************************************************************/

static bool MakeBestAsstAtTime(KHE_SOLN soln, KHE_FRAME days_frame,
  KHE_RESOURCE r, KHE_TIME t, KHE_EVENT_TIMETABLE_MONITOR etm)
{
  int attempts;  KHE_TASK best_task;  KHE_COST best_cost;
  if( DEBUG2 )
    fprintf(stderr, "      MakeBestAsstAtTime(soln, %s, %s)", KheResourceId(r),
      KheTimeId(t));
  KheBeginTrial(&best_task, &best_cost, &attempts);
  TryTime(soln, days_frame, r, t, etm, &best_task, &best_cost, &attempts);
  KheEndTrial(soln, r, best_task, best_cost, attempts);
  return best_task != NULL;
}


/*****************************************************************************/
/*                                                                           */
/*  bool MakeBestAsstAtTimeGroup(KHE_SOLN soln, KHE_FRAME days_frame,        */
/*    KHE_RESOURCE r, KHE_TIME_GROUP tg, KHE_EVENT_TIMETABLE_MONITOR etm)    */
/*                                                                           */
/*  Make the best possible assignment of a task running at time group tg to  */
/*  r, that is, if r is not already busy then.  Obtain the events running    */
/*  at tg from etm.  If an assignment is actually made, return true.         */
/*                                                                           */
/*****************************************************************************/

/* *** correct but no longer used
static bool MakeBestAsstAtTimeGroup(KHE_SOLN soln, KHE_FRAME days_frame,
  KHE_RESOURCE r, KHE_TIME_GROUP tg, KHE_EVENT_TIMETABLE_MONITOR etm)
{
  int attempts, i;  KHE_TASK best_task;  KHE_COST best_cost;  KHE_TIME t;
  if( DEBUG2 )
    fprintf(stderr, "      MakeBestAsstAtTimeGroup(soln, %s, %s)",
      KheResourceId(r), KheTimeGroupId(tg));
  KheBeginTrial(&best_task, &best_cost, &attempts);
  for( i = 0;  i < KheTimeGroupTimeCount(tg);  i++ )
  {
    t = KheTimeGroupTime(tg, i);
    TryTime(soln, days_frame, r, t, etm, &best_task, &best_cost, &attempts);
  }
  KheEndTrial(soln, r, best_task, best_cost, attempts);
  return best_task != NULL;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool MakeBestAsstAtClusterMonitor(KHE_SOLN soln, KHE_FRAME days_frame,   */
/*    KHE_RESOURCE r, KHE_CLUSTER_BUSY_TIMES_MONITOR cbtm,                   */
/*    KHE_EVENT_TIMETABLE_MONITOR etm)                                       */
/*                                                                           */
/*  Make the best possible assignment within the inactive time groups        */
/*  of cbtm.  If an assignment is actually made, return true.                */
/*                                                                           */
/*****************************************************************************/

static bool MakeBestAsstAtClusterMonitor(KHE_SOLN soln, KHE_FRAME days_frame,
  KHE_RESOURCE r, KHE_CLUSTER_BUSY_TIMES_MONITOR cbtm,
  KHE_EVENT_TIMETABLE_MONITOR etm)
{
  int attempts, i, j, x, index, count, extra, count2, extra2;  KHE_TIME t;
  KHE_TASK best_task;  KHE_COST best_cost;  KHE_TIME_GROUP tg;  KHE_POLARITY po;
  if( DEBUG2 )
    fprintf(stderr, "      MakeBestAsstAtClusterMonitor(soln, %s, cbtm)",
      KheResourceId(r));
  KheBeginTrial(&best_task, &best_cost, &attempts);
  count = KheClusterBusyTimesMonitorTimeGroupCount(cbtm);
  extra = 97 + KheSolnDiversifier(soln) * 5;
  for( i = 0;  i < count;  i++ )
  {
    index = (i + extra) % count;
    if( !KheClusterBusyTimesMonitorTimeGroupIsActive(cbtm, index, &tg, &po,&x) )
    {
      extra2 = 4 + KheSolnDiversifier(soln) * 7;
      count2 = KheTimeGroupTimeCount(tg);
      for( j = 0;  j < count2;  j++ )
      {
	t = KheTimeGroupTime(tg, (j + extra2) % count2);
	TryTime(soln, days_frame, r, t, etm, &best_task, &best_cost, &attempts);
      }
    }
  }
  KheEndTrial(soln, r, best_task, best_cost, attempts);
  return best_task != NULL;
}


/*****************************************************************************/
/*                                                                           */
/*  bool MakeBestAsst(KHE_SOLN soln, KHE_FRAME days_frame, KHE_RESOURCE r,   */
/*    KHE_RESOURCE_TIMETABLE_MONITOR rtm, KHE_TIME t,                        */
/*    KHE_EVENT_TIMETABLE_MONITOR etm, bool *soln_changed)                   */
/*                                                                           */
/*  Make the best possible assignment of a task to r (whose timetable        */
/*  monitor is rtm) at time t, that is, if r is not already busy then.       */
/*  Obtain the events running at time t from etm.  If an assignment is       */
/*  actually made, set *soln_changed to true and also return true.           */
/*                                                                           */
/*  Actually, if the meet has more than 10 tasks, we only try every          */
/*  fifth task, cyclically starting from a random point.  This is because    */
/*  most tasks are symmetrical and it takes an unreasonably long time        */
/*  to try them all.                                                         */
/*                                                                           */
/*****************************************************************************/

/* *** obsolete
static bool OldMakeBestAsst(KHE_SOLN soln, KHE_FRAME days_frame, KHE_RESOURCE r,
  KHE_RESOURCE_TIMETABLE_MONITOR rtm, KHE_TIME t,
  KHE_EVENT_TIMETABLE_MONITOR etm, bool *soln_changed)
{
  int i, j, count, extra, index, inc;  KHE_MEET meet;  KHE_TIME_GROUP tg;
  KHE_TASK task, best_task;  KHE_COST best_cost;

  index = KheFrameTimeIndex(days_frame, t);
  tg = KheFrameTimeGroup(days_frame, index);
  if( DEBUG2 )
  {
    fprintf(stderr, "      MakeBestAsst(soln, %s, %s in ", KheResourceId(r),
      KheTimeId(t));
    KheTimeGroupDebug(tg, 2, -1, stderr);
    fprintf(stderr, ")");
  }

  ** if r is already busy at time t, return immediately **
  if( !KheResourceTimetableMonitorTimeGroupAvailable(rtm, tg, true) )
  {
    if( DEBUG2 )
      fprintf(stderr, " - (already busy)\n");
    return false;
  }

  ** find the best task among all events running at time t **
  count = 0;
  best_task = NULL;  best_cost = KheCost(INT_MAX, INT_MAX);
  extra = KheResourceInstanceIndex(r);
  for( i = 0;  i < KheEventTimetableMonitorTimeMeetCount(etm, t);  i++ )
  {
    meet = KheEventTimetableMonitorTimeMeet(etm, t, i);
    inc = (KheMeetTaskCount(meet) >= 10 ? 5 : 1);
    for( j = 0;  j < KheMeetTaskCount(meet);  j += inc )
    {
      index = (j + extra) % KheMeetTaskCount(meet);
      task = KheTaskProperRoot(KheMeetTask(meet, index));
      if( KheTaskAsst(task) == NULL &&
	  KheResourceTimetableMonitorTaskAvailableInFrame(rtm,
	    task, days_frame, false) )
      {
	count++;
	if( KheTaskAssignResource(task, r) )
	{
	  if( KheSolnCost(soln) < best_cost )
	    best_task = task, best_cost = KheSolnCost(soln);
	  KheTaskUnAssignResource(task);
	}
      }
    }
  }

  ** if there is a best task, make that assignment and set *soln_changed **
  if( best_task != NULL )
  {
    KheTaskAssignResource(best_task, r);
    ** KheTaskAssignFix(best_task); doing this a better way now **
    if( DEBUG2 )
    {
      fprintf(stderr, " tried %d tasks: ", count);
      KheTaskDebug(best_task, 2, -1, stderr);
      fprintf(stderr, " := %s (soln cost now %.5f)\n", KheResourceId(r),
	KheCostShow(KheSolnCost(soln)));
    }
    *soln_changed = true;
  }
  else
  {
    if( DEBUG2 )
      fprintf(stderr, " tried %d tasks: -\n", count);
  }
  if( DEBUG4 && strcmp(KheResourceId(r), DEBUG4_ID) == 0 )
    KheResourceTimetableMonitorPrintTimetable(rtm, 10, 6, stderr);
  return best_task != NULL;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheSolnAssignRequestedResources(KHE_SOLN soln, KHE_OPTIONS options) */
/*                                                                           */
/*  Assign resources in soln as requested by its instance's limit busy       */
/*  times constraints.                                                       */
/*                                                                           */
/*  This function uses option gs_event_timetable_monitor.                    */
/*                                                                           */
/*****************************************************************************/

bool KheSolnAssignRequestedResources(KHE_SOLN soln, KHE_RESOURCE_TYPE rt,
  KHE_OPTIONS options)
{
  ARRAY_KHE_MONITOR monitors;  HA_ARENA a;  KHE_POLARITY po;
  KHE_MONITOR m;  bool soln_changed;  KHE_TIME_GROUP tg;  KHE_RESOURCE r;
  KHE_TIME t;  KHE_FRAME days_frame;  KHE_COST init_cost;
  int i, j, k, offset, min_limit, max_active, extra, count, successes;
  /* KHE_RESOURCE_TIMETABLE_MONITOR rtm; */  KHE_EVENT_TIMETABLE_MONITOR etm;
  KHE_CLUSTER_BUSY_TIMES_MONITOR cbtm;  KHE_CLUSTER_BUSY_TIMES_CONSTRAINT cbtc;
  KHE_LIMIT_BUSY_TIMES_MONITOR lbtm;  KHE_LIMIT_BUSY_TIMES_CONSTRAINT lbtc;

  /* do nothing if so instructed by the relevant option */
  if( KheOptionsGetBool(options, "rs_requested_off", false) )
    return false;
  days_frame = KheOptionsFrame(options, "gs_common_frame", soln);
  if( DEBUG5 )
    KheFrameAssertNoClashes(days_frame);

  /* get the event timetable monitor and abort if not available */
  if( DEBUG1 )
  {
    fprintf(stderr, "[ KheSolnAssignRequestedResources(soln, %s, options)\n",
      rt == NULL ? "-" : KheResourceTypeId(rt));
    KheSolnCostByTypeDebug(soln, 1, 2, stderr);
  }
  etm = (KHE_EVENT_TIMETABLE_MONITOR)
    KheOptionsGetObject(options, "gs_event_timetable_monitor", NULL);
  HnAssert(etm != NULL, "KheSolnAssignRequestedResources: "
    "no gs_event_timetable_monitor option in options");
  HnAssert(rt != NULL, "KheSolnAssignRequestedResources: rt == NULL");

  /* build and sort monitors, holding the relevant monitors */
  init_cost = KheSolnCost(soln);
  a = KheSolnArenaBegin(soln);
  HaArrayInit(monitors, a);
  for( i = 0;  i < KheResourceTypeResourceCount(rt);  i++ )
  {
    r = KheResourceTypeResource(rt, i);
    for( j = 0;  j < KheSolnResourceMonitorCount(soln, r);  j++ )
    {
      m = KheSolnResourceMonitor(soln, r, j);
      if( KheMonitorTag(m) == KHE_LIMIT_BUSY_TIMES_MONITOR_TAG )
      {
	lbtm = (KHE_LIMIT_BUSY_TIMES_MONITOR) m;
	lbtc = KheLimitBusyTimesMonitorConstraint(lbtm);
	if( KheLimitBusyTimesConstraintMinimum(lbtc) > 0 &&
	    !KheLimitBusyTimesConstraintAllowZero(lbtc) &&
	    KheConstraintWeight((KHE_CONSTRAINT) lbtc) > 0 )
	  HaArrayAddLast(monitors, m);
      }
      else if( KheMonitorTag(m) == KHE_CLUSTER_BUSY_TIMES_MONITOR_TAG )
      {
	cbtm = (KHE_CLUSTER_BUSY_TIMES_MONITOR) m;
	cbtc = KheClusterBusyTimesMonitorConstraint(cbtm);
	max_active = KheClusterBusyTimesMonitorHistory(cbtm) +
	  KheClusterBusyTimesConstraintTimeGroupCount(cbtc) +
	  KheClusterBusyTimesMonitorHistoryAfter(cbtm);
	if( (KheClusterBusyTimesConstraintLimitBusyRecode(cbtc) ||
	     max_active <= KheClusterBusyTimesConstraintMinimum(cbtc))
	    && !KheClusterBusyTimesConstraintAllowZero(cbtc) &&
	    KheConstraintWeight((KHE_CONSTRAINT) cbtc) > 0 )
	  HaArrayAddLast(monitors, m);
      }
    }
  }
  HaArraySort(monitors, &KheMonitorDecreasingWeightCmp);

  /* handle each monitor, for the cases where particular times are wanted */
  soln_changed = false;
  if( DEBUG1 )
    fprintf(stderr, "  %d monitors (first pass):\n", HaArrayCount(monitors));
  HaArrayForEach(monitors, m, i)
  {
    if( DEBUG3 )
      KheMonitorDebug(m, 2, 2, stderr);
    if( DEBUG5 )
      KheFrameAssertNoClashes(days_frame);
    if( KheMonitorCost(m) > 0 )
    {
      if( KheMonitorTag(m) == KHE_LIMIT_BUSY_TIMES_MONITOR_TAG )
      {
	lbtm = (KHE_LIMIT_BUSY_TIMES_MONITOR) m;
	lbtc = KheLimitBusyTimesMonitorConstraint(lbtm);
	r = KheLimitBusyTimesMonitorResource(lbtm);
	offset = KheLimitBusyTimesMonitorOffset(lbtm);
	min_limit = KheLimitBusyTimesConstraintMinimum(lbtc);
	for( j = 0;  j < KheLimitBusyTimesConstraintTimeGroupCount(lbtc);  j++ )
	{
	  tg = KheLimitBusyTimesConstraintTimeGroup(lbtc, j, offset);
	  if( KheTimeGroupTimeCount(tg) <= min_limit )
	  {
	    /* every time of tg is requested to be busy */
	    for( k = 0;  k < KheTimeGroupTimeCount(tg);  k++ )
	    {
	      t = KheTimeGroupTime(tg, k);
	      if( MakeBestAsstAtTime(soln, days_frame, r, t, etm) )
		soln_changed = true;
	    }
	  }
	}
      }
      else if( KheMonitorTag(m) == KHE_CLUSTER_BUSY_TIMES_MONITOR_TAG )
      {
	cbtm = (KHE_CLUSTER_BUSY_TIMES_MONITOR) m;
	cbtc = KheClusterBusyTimesMonitorConstraint(cbtm);
	r = KheClusterBusyTimesMonitorResource(cbtm);
	offset = KheClusterBusyTimesMonitorOffset(cbtm);
	max_active = KheClusterBusyTimesMonitorHistory(cbtm) +
	  KheClusterBusyTimesConstraintTimeGroupCount(cbtc) +
	  KheClusterBusyTimesMonitorHistoryAfter(cbtm);
	if( max_active <= KheClusterBusyTimesConstraintMinimum(cbtc) )
	{
	  for( j=0; j < KheClusterBusyTimesConstraintTimeGroupCount(cbtc); j++ )
	  {
	    tg = KheClusterBusyTimesConstraintTimeGroup(cbtc, j, offset, &po);
	    if( KheTimeGroupTimeCount(tg) == 1 && po == KHE_POSITIVE )
	    {
	      /* tg should be active, so the sole time of tg should be busy */
	      t = KheTimeGroupTime(tg, 0);
	      if( MakeBestAsstAtTime(soln, days_frame, r, t, etm) )
		soln_changed = true;
	    }
	  }
	}
      }
      else
	HnAbort("KheSolnAssignRequestedResources internal error");
    }
  }

  /* handle each monitor, for the cases where a choice of times is available */
  if( DEBUG1 )
    fprintf(stderr, "  %d monitors (second pass):\n", HaArrayCount(monitors));
  HaArrayForEach(monitors, m, i)
  {
    if( DEBUG3 )
      KheMonitorDebug(m, 2, 4, stderr);
    if( DEBUG5 )
      KheFrameAssertNoClashes(days_frame);
    if( KheMonitorCost(m) > 0 )
    {
      if( KheMonitorTag(m) == KHE_LIMIT_BUSY_TIMES_MONITOR_TAG )
      {
	lbtm = (KHE_LIMIT_BUSY_TIMES_MONITOR) m;
	lbtc = KheLimitBusyTimesMonitorConstraint(lbtm);
	r = KheLimitBusyTimesMonitorResource(lbtm);
	offset = KheLimitBusyTimesMonitorOffset(lbtm);
	min_limit = KheLimitBusyTimesConstraintMinimum(lbtc);
	for( j = 0;  j < KheLimitBusyTimesConstraintTimeGroupCount(lbtc);  j++ )
	{
	  tg = KheLimitBusyTimesConstraintTimeGroup(lbtc, j, offset);
	  if( KheTimeGroupTimeCount(tg) > min_limit )
	  {
	    /* min_limit times of tg are requested to be busy */
	    count = KheTimeGroupTimeCount(tg);
	    if( DEBUG3 )
	      fprintf(stderr, "      trying %s (%d times, min_limit %d)\n",
		KheTimeGroupId(tg), count, min_limit);
	    extra = KheResourceInstanceIndex(r);
	    successes = 0;
	    for( k = 0;  k < count && successes < min_limit;  k++ )
	    {
	      t = KheTimeGroupTime(tg, (k + extra) % count);
	      if( MakeBestAsstAtTime(soln, days_frame, r, t, etm) )
		soln_changed = true, successes++;
	    }
	  }
	}
      }
      else if( KheMonitorTag(m) == KHE_CLUSTER_BUSY_TIMES_MONITOR_TAG )
      {
	cbtm = (KHE_CLUSTER_BUSY_TIMES_MONITOR) m;
	cbtc = KheClusterBusyTimesMonitorConstraint(cbtm);
	r = KheClusterBusyTimesMonitorResource(cbtm);
	if( DEBUG4 && strcmp(KheResourceId(r), DEBUG4_ID) == 0 )
	{
	  KHE_RESOURCE_TIMETABLE_MONITOR rtm;
	  rtm = KheResourceTimetableMonitor(soln, r);
	  KheResourceTimetableMonitorPrintTimetable(rtm, 10, 6, stderr);
	}
	offset = KheClusterBusyTimesMonitorOffset(cbtm);
	max_active = KheClusterBusyTimesMonitorHistory(cbtm) +
	  KheClusterBusyTimesConstraintTimeGroupCount(cbtc) +
	  KheClusterBusyTimesMonitorHistoryAfter(cbtm);
	min_limit = KheClusterBusyTimesConstraintMinimum(cbtc);
	if( max_active > min_limit &&
	    KheClusterBusyTimesConstraintLimitBusyRecode(cbtc) )
	{
	  /* a recoded limit busy times constraint with non-forced assignments*/
	  successes = 0;
	  while( successes < min_limit &&
		 MakeBestAsstAtClusterMonitor(soln, days_frame, r, cbtm, etm) )
	    soln_changed = true, successes++;
	}
	/* *** obsolete
	{
	  ** a recoded limit busy times constraint with non-forced assignments**
	  count = KheClusterBusyTimesConstraintTimeGroupCount(cbtc);
	  extra = KheResourceInstanceIndex(r);
	  successes = 0;
	  for( j = 0;  j < count && successes < min_limit;  j++ )
	  {
	    tg = KheClusterBusyTimesConstraintTimeGroup(cbtc, (j+extra) % count,
	      offset, &po);
	    HnAssert(KheTimeGroupTimeCount(tg) == 1 && po == KHE_POSITIVE,
	      "KheSolnAssignRequestedResources internal error 1");
	    if( DEBUG3 )
	      fprintf(stderr, "      trying %s (%d times, want 1)\n",
		KheTimeGroupId(tg), count);
	    if( MakeBestAsstAtTimeGroup(soln, days_frame, r, tg, etm) )
	      soln_changed = true, successes++;
	    ** ***
	    t = KheTimeGroupTime(tg, 0);
	    if( MakeBestAsst(soln, days_frame, r, rtm, t, etm, &soln_changed) )
	      successes++;
	    *** **
	  }
	}
	*** */
      }
      else
	HnAbort("KheSolnAssignRequestedResources internal error");
    }
  }

  /* and return */
  KheSolnArenaEnd(soln, a);
  /* HaAre naDelete(a); */
  if( DEBUG5 )
    KheFrameAssertNoClashes(days_frame);
  if( DEBUG1 )
  {
    if( soln_changed )
      fprintf(stderr,
	"] KheSolnAssignRequestedResources returning true (%.5f -> %.5f)\n",
	KheCostShow(init_cost), KheCostShow(KheSolnCost(soln)));
    else
      fprintf(stderr, "] KheSolnAssignRequestedResources returning false\n");
  }
  return soln_changed;
}
