
/*****************************************************************************/
/*                                                                           */
/*  THE KHE HIGH SCHOOL TIMETABLING ENGINE                                   */
/*  COPYRIGHT (C) 2010 Jeffrey H. Kingston                                   */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 3, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         khe_sr_requested.c                                         */
/*  DESCRIPTION:  Satisfying requested task assignments                      */
/*                                                                           */
/*****************************************************************************/
#include "khe_solvers.h"
#include <limits.h>

#define DEBUG1 1
#define DEBUG2 0
#define DEBUG3 1
#define DEBUG4 0
#define DEBUG4_ID "A10"
#define DEBUG5 0	/* clash checking */

#define DEBUG6 1	/* trial for a specific nurse */
#define DEBUG6_ID "Nurse4"
#define DEBUG6_ON(r) (DEBUG6 && strcmp(KheResourceId(r), DEBUG6_ID) == 0)


typedef HA_ARRAY(KHE_MONITOR) ARRAY_KHE_MONITOR;

/*****************************************************************************/
/*                                                                           */
/*  int KheMonitorDecreasingWeightCmp(const void *t1, const void *t2)        */
/*                                                                           */
/*  Comparison function for sorting an array of monitors by decreasing       */
/*  weight.                                                                  */
/*                                                                           */
/*****************************************************************************/

static int KheMonitorDecreasingWeightCmp(const void *t1, const void *t2)
{
  KHE_COST weight1, weight2;
  KHE_CONSTRAINT c1, c2;
  KHE_MONITOR m1 = * (KHE_MONITOR *) t1;
  KHE_MONITOR m2 = * (KHE_MONITOR *) t2;
  c1 = KheMonitorConstraint(m1);
  c2 = KheMonitorConstraint(m2);
  weight1 = KheConstraintCombinedWeight(c1);
  weight2 = KheConstraintCombinedWeight(c2);
  return KheCostCmp(weight2, weight1);  /* reverse order, decreasing weight */
}


/*****************************************************************************/
/*                                                                           */
/*  void TryTime(KHE_SOLN soln, KHE_FRAME days_frame,                        */
/*    KHE_RESOURCE r, KHE_RESOURCE_TIMETABLE_MONITOR rtm,                    */
/*    KHE_TIME t, KHE_EVENT_TIMETABLE_MONITOR etm,                           */
/*    KHE_TASK *best_task, KHE_COST *best_cost, int *attempts)               */
/*                                                                           */
/*  Try assigning r to a task at time t, and update *best_task, *best_cost,  */
/*  and *attempts appropriately.                                             */
/*                                                                           */
/*****************************************************************************/

static void TryTime(KHE_SOLN soln, KHE_FRAME days_frame, KHE_RESOURCE r,
  KHE_TIME t, KHE_EVENT_TIMETABLE_MONITOR etm,
  KHE_TASK *best_task, KHE_COST *best_cost, int *attempts)
{
  int i, j, extra, index, inc;  KHE_MEET meet;  KHE_TIME_GROUP tg;
  KHE_TASK task;  KHE_RESOURCE_TIMETABLE_MONITOR rtm;

  /* if r is already busy at time t, return immediately */
  rtm = KheResourceTimetableMonitor(soln, r);
  index = KheFrameTimeIndex(days_frame, t);
  tg = KheFrameTimeGroup(days_frame, index);
  if( !KheResourceTimetableMonitorTimeGroupAvailable(rtm, tg, false) )
    return;

  /* try a selection of tasks running at time t */
  extra = KheResourceInstanceIndex(r);
  for( i = 0;  i < KheEventTimetableMonitorTimeMeetCount(etm, t);  i++ )
  {
    meet = KheEventTimetableMonitorTimeMeet(etm, t, i);
    inc = (KheMeetTaskCount(meet) >= 10 ? 5 : 1);
    for( j = 0;  j < KheMeetTaskCount(meet);  j += inc )
    {
      index = (j + extra) % KheMeetTaskCount(meet);
      task = KheTaskProperRoot(KheMeetTask(meet, index));
      if( KheTaskAsst(task) != NULL )
      {
	if( DEBUG6_ON(r) )
	{
	  fprintf(stderr, "    tried ");
	  KheTaskDebug(task, 2, -1, stderr);
	  fprintf(stderr, " (already assigned %s)\n",
	    KheResourceId(KheTaskAsstResource(task)));
	}
      }
      else if( !KheResourceTimetableMonitorTaskAvailableInFrame(rtm, task,
	    days_frame, NULL) )
      {
	if( DEBUG6_ON(r) )
	{
	  fprintf(stderr, "    tried ");
	  KheTaskDebug(task, 2, -1, stderr);
	  fprintf(stderr, " (times unavailable in frame)\n");
	}
      }
      else
      {
	(*attempts)++;
	if( KheTaskAssignResource(task, r) )
	{
	  if( *best_task == NULL || KheSolnCost(soln) < *best_cost )
	  {
	    *best_task = task, *best_cost = KheSolnCost(soln);
	    if( DEBUG6_ON(r) )
	    {
	      fprintf(stderr, "    tried ");
	      KheTaskDebug(task, 2, -1, stderr);
	      fprintf(stderr, " (cost %.5f, new best)\n",
		KheCostShow(KheSolnCost(soln)));
	    }
	  }
	  else
	  {
	    if( DEBUG6_ON(r) )
	    {
	      fprintf(stderr, "    tried ");
	      KheTaskDebug(task, 2, -1, stderr);
	      fprintf(stderr, " (cost %.5f, not new best)\n",
		KheCostShow(KheSolnCost(soln)));
	    }
	  }
	  KheTaskUnAssignResource(task);
	}
	else
	{
	  if( DEBUG6_ON(r) )
	  {
	    fprintf(stderr, "    tried ");
	    KheTaskDebug(task, 2, -1, stderr);
	    fprintf(stderr, " (did not assign)\n");
	  }
	}
      }
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheBeginTrial(KHE_SOLN soln, KHE_RESOURCE r, KHE_TASK *best_task,   */
/*    KHE_COST *best_cost, int *attempts)                                    */
/*                                                                           */
/*  Begin a trial which tries to find the task whose assignment gives        */
/*  the best cost.                                                           */
/*                                                                           */
/*****************************************************************************/

static void KheBeginTrial(KHE_SOLN soln, KHE_RESOURCE r, KHE_TASK *best_task,
  KHE_COST *best_cost, int *attempts)
{
  if( DEBUG6_ON(r) )
    fprintf(stderr, "  [ BeginTrial(soln cost %.5f, %s)\n",
      KheCostShow(KheSolnCost(soln)), KheResourceId(r));
  *best_task = NULL;
  *best_cost = KheCost(INT_MAX, INT_MAX);
  *attempts = 0;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheEndTrial(KHE_SOLN soln, KHE_RESOURCE r, KHE_TASK best_task,      */
/*    KHE_COST best_cost, int attempts)                                      */
/*                                                                           */
/*  End a trial, including making the best assignment, if there is one.      */
/*                                                                           */
/*****************************************************************************/

static bool KheEndTrial(KHE_SOLN soln, KHE_RESOURCE r, KHE_TASK best_task,
  KHE_COST best_cost, int attempts)
{
  if( best_task != NULL )
  {
    KheTaskAssignResource(best_task, r);
    if( DEBUG2 )
    {
      fprintf(stderr, " tried %d tasks: ", attempts);
      KheTaskDebug(best_task, 2, -1, stderr);
      fprintf(stderr, " := %s (soln cost now %.5f)\n", KheResourceId(r),
	KheCostShow(KheSolnCost(soln)));
    }
    if( DEBUG6_ON(r) )
    {
      fprintf(stderr, "  ] EndTrial ");
      KheTaskDebug(best_task, 2, -1, stderr);
      fprintf(stderr, " := %s (soln cost now %.5f)\n", KheResourceId(r),
	KheCostShow(KheSolnCost(soln)));
    }
  }
  else
  {
    if( DEBUG2 )
      fprintf(stderr, " tried %d tasks: -\n", attempts);
    if( DEBUG6_ON(r) )
      fprintf(stderr, "  ] EndTrial (unsuccessful)\n");
  }
  if( DEBUG4 && strcmp(KheResourceId(r), DEBUG4_ID) == 0 )
  {
    KHE_RESOURCE_TIMETABLE_MONITOR rtm;
    rtm = KheResourceTimetableMonitor(soln, r);
    KheResourceTimetableMonitorPrintTimetable(rtm, 10, 6, stderr);
  }
  return best_task != NULL;
}


/*****************************************************************************/
/*                                                                           */
/*  bool MakeBestAsstAtTime(KHE_SOLN soln, KHE_FRAME days_frame,             */
/*    KHE_RESOURCE r, KHE_TIME t, KHE_EVENT_TIMETABLE_MONITOR etm)           */
/*                                                                           */
/*  Make the best possible assignment of a task running at time t to r,      */
/*  that is, if r is not already busy then.  Obtain the events running       */
/*  at time t from etm.  If an assignment is actually made, return true.     */
/*                                                                           */
/*  Actually, if the meet has more than 10 tasks, we only try every fifth    */
/*  task, cyclically starting from a random point.  This is because most     */
/*  tasks are symmetrical and it takes too long to try them all.             */
/*                                                                           */
/*****************************************************************************/

static bool MakeBestAsstAtTime(KHE_SOLN soln, KHE_FRAME days_frame,
  KHE_RESOURCE r, KHE_TIME t, KHE_EVENT_TIMETABLE_MONITOR etm)
{
  int attempts;  KHE_TASK best_task;  KHE_COST best_cost;
  if( DEBUG2 )
    fprintf(stderr, "      MakeBestAsstAtTime(soln, %s, %s)", KheResourceId(r),
      KheTimeId(t));
  KheBeginTrial(soln, r, &best_task, &best_cost, &attempts);
  TryTime(soln, days_frame, r, t, etm, &best_task, &best_cost, &attempts);
  return KheEndTrial(soln, r, best_task, best_cost, attempts);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheHandleClusterForced(KHE_CLUSTER_BUSY_TIMES_MONITOR cbtm,         */
/*    KHE_SOLN soln, KHE_FRAME days_frame, KHE_EVENT_TIMETABLE_MONITOR etm)  */
/*                                                                           */
/*  Try to correct the defects of cbtm in places where the choices of times  */
/*  are forced, and return true if any changes were made.                    */
/*                                                                           */
/*  A time is forced when it is the only time in a positive time group, and  */
/*  every time group has to be active; or when it is the only time in a      */
/*  negative time group, and every time group has to be inactive.            */
/*                                                                           */
/*****************************************************************************/

static bool KheHandleClusterForced(KHE_CLUSTER_BUSY_TIMES_MONITOR cbtm,
  KHE_SOLN soln, KHE_FRAME days_frame, KHE_EVENT_TIMETABLE_MONITOR etm)
{
  int active_group_count, open_group_count, min_limit, max_limit, i;
  int busy_count, count;  KHE_RESOURCE r;  KHE_TIME t;
  bool allow_zero, active, res;  KHE_TIME_GROUP tg;  KHE_POLARITY po;
  KHE_CLUSTER_BUSY_TIMES_CONSTRAINT cbtc;
  cbtc = KheClusterBusyTimesMonitorConstraint(cbtm);
  r = KheClusterBusyTimesMonitorResource(cbtm);
  count = KheClusterBusyTimesMonitorTimeGroupCount(cbtm);
  KheClusterBusyTimesMonitorActiveTimeGroupCount(cbtm, &active_group_count,
    &open_group_count, &min_limit, &max_limit, &allow_zero);
  res = false;
  if( min_limit >= count && active_group_count + open_group_count < min_limit )
  {
    /* every time group has to be active, and not all of them are */
    if( !KheClusterBusyTimesConstraintAllNegative(cbtc) )
    {
      for( i = 0;  i < count;  i++ )
      {
	active = KheClusterBusyTimesMonitorTimeGroupIsActive(cbtm, i,
	  &tg, &po, &busy_count);
	if( !active && po == KHE_POSITIVE && KheTimeGroupTimeCount(tg) == 1 )
	{
	  /* try to make tg busy */
	  t = KheTimeGroupTime(tg, 0);
	  if( MakeBestAsstAtTime(soln, days_frame, r, t, etm) )
	    res = true;
	}
      }
    }
  }
  else if( max_limit == 0 && active_group_count > max_limit )
  {
    /* every time group has to be inactive, and not all of them are */
    if( !KheClusterBusyTimesConstraintAllPositive(cbtc) )
    {
      for( i = 0;  i < count;  i++ )
      {
	active = KheClusterBusyTimesMonitorTimeGroupIsActive(cbtm, i,
	  &tg, &po, &busy_count);
	if( active && po == KHE_NEGATIVE && KheTimeGroupTimeCount(tg) == 1 )
	{
	  /* try to make tg busy */
	  t = KheTimeGroupTime(tg, 0);
	  if( MakeBestAsstAtTime(soln, days_frame, r, t, etm) )
	    res = true;
	}
      }
    }
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheHandleClusterNotForced(KHE_CLUSTER_BUSY_TIMES_MONITOR cbtm,      */
/*    KHE_SOLN soln, KHE_FRAME days_frame, KHE_EVENT_TIMETABLE_MONITOR etm)  */
/*                                                                           */
/*  Try to correct the defects of cbtm in places where the choices of times  */
/*  are not forced, and return true if any changes were made.                */
/*                                                                           */
/*****************************************************************************/

static bool KheHandleClusterNotForced(KHE_CLUSTER_BUSY_TIMES_MONITOR cbtm,
  KHE_SOLN soln, KHE_FRAME days_frame, KHE_EVENT_TIMETABLE_MONITOR etm)
{
  int active_group_count, open_group_count, min_limit, max_limit, i, j;
  int count, extra, index, busy_count;  KHE_TIME_GROUP tg;  KHE_POLARITY po;
  bool allow_zero, active, progressing, res;  KHE_RESOURCE r;  KHE_TIME t;
  int attempts;  KHE_TASK best_task;  KHE_COST best_cost;
  KHE_CLUSTER_BUSY_TIMES_CONSTRAINT cbtc;
  cbtc = KheClusterBusyTimesMonitorConstraint(cbtm);
  r = KheClusterBusyTimesMonitorResource(cbtm);
  res = false;
  do
  {
    progressing = false;
    KheClusterBusyTimesMonitorActiveTimeGroupCount(cbtm, &active_group_count,
      &open_group_count, &min_limit, &max_limit, &allow_zero);
    if( active_group_count + open_group_count < min_limit )
    {
      if( !KheClusterBusyTimesConstraintAllNegative(cbtc) )
      {
	/* need to increase the number of active time groups */
	/* by assigning r to tasks in inactive positive time groups */
	KheBeginTrial(soln, r, &best_task, &best_cost, &attempts);
	count = KheClusterBusyTimesMonitorTimeGroupCount(cbtm);
	extra = KheMonitorSolnIndex((KHE_MONITOR) cbtm) * 13 +
	  KheSolnDiversifier(soln) * 7;
	for( i = 0;  i < KheClusterBusyTimesMonitorTimeGroupCount(cbtm);  i++ )
	{
	  index = (i + extra) % count;
	  active = KheClusterBusyTimesMonitorTimeGroupIsActive(cbtm, index,
	    &tg, &po, &busy_count);
	  if( !active && po == KHE_POSITIVE )
	  {
	    /* try to make tg busy */
	    for( j = 0;  j < KheTimeGroupTimeCount(tg);  j++ )
	    {
	      t = KheTimeGroupTime(tg, j);
	      TryTime(soln, days_frame, r, t, etm, &best_task, &best_cost,
		&attempts);
	    }
	  }
	}
	if( KheEndTrial(soln, r, best_task, best_cost, attempts) )
	  progressing = res = true;
      }
    }
    else if( active_group_count > max_limit )
    {
      /* need to decrease the number of active time groups */
      /* by assigning r to tasks in active negative time groups */
      if( !KheClusterBusyTimesConstraintAllPositive(cbtc) )
      {
	KheBeginTrial(soln, r, &best_task, &best_cost, &attempts);
	count = KheClusterBusyTimesMonitorTimeGroupCount(cbtm);
	extra = KheMonitorSolnIndex((KHE_MONITOR) cbtm) * 13 +
	  KheSolnDiversifier(soln) * 7;
	for( i = 0;  i < KheClusterBusyTimesMonitorTimeGroupCount(cbtm);  i++ )
	{
	  index = (i + extra) % count;
	  active = KheClusterBusyTimesMonitorTimeGroupIsActive(cbtm, index,
	    &tg, &po, &busy_count);
	  if( active && po == KHE_NEGATIVE )
	  {
	    /* try to make tg busy */
	    for( j = 0;  j < KheTimeGroupTimeCount(tg);  j++ )
	    {
	      t = KheTimeGroupTime(tg, j);
	      TryTime(soln, days_frame, r, t, etm, &best_task, &best_cost,
		&attempts);
	    }
	  }
	}
	if( KheEndTrial(soln, r, best_task, best_cost, attempts) )
	  progressing = res = true;
      }
    }
  } while( progressing );
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheHandleLimitForced(KHE_LIMIT_BUSY_TIMES_MONITOR lbtm,             */
/*    KHE_SOLN soln, KHE_FRAME days_frame, KHE_EVENT_TIMETABLE_MONITOR etm)  */
/*                                                                           */
/*  Try to correct the defects of lbtm in places where the choices of times  */
/*  are forced, and return true if any changes were made.                    */
/*                                                                           */
/*****************************************************************************/

static bool KheHandleLimitForced(KHE_LIMIT_BUSY_TIMES_MONITOR lbtm,
  KHE_SOLN soln, KHE_FRAME days_frame, KHE_EVENT_TIMETABLE_MONITOR etm)
{
  int min_limit, max_limit, i, k, busy_count;
  bool allow_zero, res;  KHE_TIME_GROUP tg;  KHE_TIME t;  KHE_RESOURCE r;
  r = KheLimitBusyTimesMonitorResource(lbtm);
  res = false;
  for( i = 0;  i < KheLimitBusyTimesMonitorDefectiveTimeGroupCount(lbtm);  i++ )
  {
    KheLimitBusyTimesMonitorDefectiveTimeGroup(lbtm, i, &tg,
      &busy_count, &min_limit, &max_limit, &allow_zero);
    if( busy_count < min_limit && KheTimeGroupTimeCount(tg) <= min_limit )
    {
      /* every time of tg is requested to be busy */
      for( k = 0;  k < KheTimeGroupTimeCount(tg);  k++ )
      {
	t = KheTimeGroupTime(tg, k);
	if( MakeBestAsstAtTime(soln, days_frame, r, t, etm) )
	  res = true;
      }
    }
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheHandleLimitNotForced(KHE_LIMIT_BUSY_TIMES_MONITOR lbtm,          */
/*    KHE_SOLN soln, KHE_FRAME days_frame, KHE_EVENT_TIMETABLE_MONITOR etm)  */
/*                                                                           */
/*  Try to correct the defects of cbtm in places where the choices of times  */
/*  are not forced, and return true if any changes were made.                */
/*                                                                           */
/*****************************************************************************/

static bool KheHandleLimitNotForced(KHE_LIMIT_BUSY_TIMES_MONITOR lbtm,
  KHE_SOLN soln, KHE_FRAME days_frame, KHE_EVENT_TIMETABLE_MONITOR etm)
{
  int min_limit, max_limit, i, k, busy_count;
  int attempts;  KHE_TASK best_task;  KHE_COST best_cost;
  bool allow_zero, res;  KHE_TIME_GROUP tg;  KHE_TIME t;  KHE_RESOURCE r;
  r = KheLimitBusyTimesMonitorResource(lbtm);
  res = false;
  for( i = 0;  i < KheLimitBusyTimesMonitorDefectiveTimeGroupCount(lbtm);  i++ )
  {
    KheLimitBusyTimesMonitorDefectiveTimeGroup(lbtm, i, &tg,
      &busy_count, &min_limit, &max_limit, &allow_zero);
    if( busy_count < min_limit && KheTimeGroupTimeCount(tg) > min_limit )
    {
      /* not every time of tg is requested to be busy */
      KheBeginTrial(soln, r, &best_task, &best_cost, &attempts);
      for( k = 0;  k < KheTimeGroupTimeCount(tg);  k++ )
      {
	t = KheTimeGroupTime(tg, k);
	TryTime(soln, days_frame, r, t, etm, &best_task, &best_cost,
	  &attempts);
      }
      if( KheEndTrial(soln, r, best_task, best_cost, attempts) )
      {
	res = true;
	i--;  /* works whether or not tg is deleted from defective list */
      }
    }
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheSolnAssignRequestedResources(KHE_SOLN soln, KHE_OPTIONS options) */
/*                                                                           */
/*  Assign resources in soln as requested by its instance's limit busy       */
/*  times constraints.  Return true if any assignments were made.            */
/*                                                                           */
/*  This function uses option gs_event_timetable_monitor.                    */
/*                                                                           */
/*****************************************************************************/

bool KheSolnAssignRequestedResources(KHE_SOLN soln, KHE_RESOURCE_TYPE rt,
  KHE_OPTIONS options)
{
  ARRAY_KHE_MONITOR monitors;  HA_ARENA a;  KHE_EVENT_TIMETABLE_MONITOR etm;
  KHE_MONITOR m;  bool soln_changed;  KHE_RESOURCE r;
  KHE_FRAME days_frame;  KHE_COST init_cost;
  int i, j;
  KHE_CLUSTER_BUSY_TIMES_MONITOR cbtm;  KHE_CLUSTER_BUSY_TIMES_CONSTRAINT cbtc;
  KHE_LIMIT_BUSY_TIMES_MONITOR lbtm;  KHE_LIMIT_BUSY_TIMES_CONSTRAINT lbtc;

  /* do nothing if so instructed by the relevant option */
  if( KheOptionsGetBool(options, "rs_requested_off", false) )
    return false;
  days_frame = KheOptionsFrame(options, "gs_common_frame", soln);
  if( DEBUG5 )
    KheFrameAssertNoClashes(days_frame);

  /* get the event timetable monitor and abort if not available */
  if( DEBUG1 )
  {
    fprintf(stderr, "[ KheSolnAssignRequestedResources(soln, %s, options)\n",
      rt == NULL ? "-" : KheResourceTypeId(rt));
    KheSolnCostByTypeDebug(soln, 1, 2, stderr);
  }
  etm = (KHE_EVENT_TIMETABLE_MONITOR)
    KheOptionsGetObject(options, "gs_event_timetable_monitor", NULL);
  HnAssert(etm != NULL, "KheSolnAssignRequestedResources: "
    "no gs_event_timetable_monitor option in options");
  HnAssert(rt != NULL, "KheSolnAssignRequestedResources: rt == NULL");

  /* build and sort the relevant monitors:  for all resources of type */
  /* rt, those limit busy times and cluster busy times monitors with  */
  /* non-zero cost whose allow_zero attributes are false              */
  init_cost = KheSolnCost(soln);
  a = KheSolnArenaBegin(soln);
  HaArrayInit(monitors, a);
  for( i = 0;  i < KheResourceTypeResourceCount(rt);  i++ )
  {
    r = KheResourceTypeResource(rt, i);
    for( j = 0;  j < KheSolnResourceMonitorCount(soln, r);  j++ )
    {
      m = KheSolnResourceMonitor(soln, r, j);
      if( KheMonitorCost(m) > 0 )
      {
	if( KheMonitorTag(m) == KHE_LIMIT_BUSY_TIMES_MONITOR_TAG )
	{
	  lbtm = (KHE_LIMIT_BUSY_TIMES_MONITOR) m;
	  lbtc = KheLimitBusyTimesMonitorConstraint(lbtm);
	  if( KheLimitBusyTimesConstraintMinimum(lbtc) > 0 &&
	      !KheLimitBusyTimesConstraintAllowZero(lbtc) )
	    HaArrayAddLast(monitors, m);
	}
	else if( KheMonitorTag(m) == KHE_CLUSTER_BUSY_TIMES_MONITOR_TAG )
	{
	  cbtm = (KHE_CLUSTER_BUSY_TIMES_MONITOR) m;
	  cbtc = KheClusterBusyTimesMonitorConstraint(cbtm);
	  if( (KheClusterBusyTimesConstraintMinimum(cbtc) > 0 &&
	      !KheClusterBusyTimesConstraintAllowZero(cbtc) &&
	      !KheClusterBusyTimesConstraintAllNegative(cbtc)) ||
	      (KheClusterBusyTimesConstraintMaximum(cbtc) < 
	      KheClusterBusyTimesConstraintTimeGroupCount(cbtc) &&
	      !KheClusterBusyTimesConstraintAllPositive(cbtc)) )
	    HaArrayAddLast(monitors, m);
	}
      }
    }
  }
  HaArraySort(monitors, &KheMonitorDecreasingWeightCmp);

  /* handle each monitor, for cases where the choices of times are forced */
  soln_changed = false;
  if( DEBUG1 )
    fprintf(stderr, "  %d monitors (forced pass):\n", HaArrayCount(monitors));
  HaArrayForEach(monitors, m, i)
  {
    if( DEBUG3 )
      KheMonitorDebug(m, 2, 2, stderr);
    if( DEBUG5 )
      KheFrameAssertNoClashes(days_frame);
    if( KheMonitorCost(m) > 0 )  /* not redundant, cost may have changed */
    {
      if( KheMonitorTag(m) == KHE_LIMIT_BUSY_TIMES_MONITOR_TAG )
      {
	lbtm = (KHE_LIMIT_BUSY_TIMES_MONITOR) m;
        if( KheHandleLimitForced(lbtm, soln, days_frame, etm) )
	  soln_changed = true;
      }
      else if( KheMonitorTag(m) == KHE_CLUSTER_BUSY_TIMES_MONITOR_TAG )
      {
	cbtm = (KHE_CLUSTER_BUSY_TIMES_MONITOR) m;
        if( KheHandleClusterForced(cbtm, soln, days_frame, etm) )
	  soln_changed = true;
      }
      else
	HnAbort("KheSolnAssignRequestedResources internal error");
    }
  }

  /* handle each monitor, for cases where the choices of times are not forced */
  if( !KheOptionsGetBool(options, "rs_requested_nonforced_off", false) )
  {
    HaArrayForEach(monitors, m, i)
    {
      if( DEBUG1 )
	fprintf(stderr, "  %d monitors (not-forced pass):\n",
	  HaArrayCount(monitors));
      if( DEBUG3 )
	KheMonitorDebug(m, 2, 4, stderr);
      if( DEBUG5 )
	KheFrameAssertNoClashes(days_frame);
      if( KheMonitorCost(m) > 0 )
      {
	if( KheMonitorTag(m) == KHE_LIMIT_BUSY_TIMES_MONITOR_TAG )
	{
	  lbtm = (KHE_LIMIT_BUSY_TIMES_MONITOR) m;
	  if( KheHandleLimitNotForced(lbtm, soln, days_frame, etm) )
	    soln_changed = true;
	}
	else if( KheMonitorTag(m) == KHE_CLUSTER_BUSY_TIMES_MONITOR_TAG )
	{
	  cbtm = (KHE_CLUSTER_BUSY_TIMES_MONITOR) m;
	  if( KheHandleClusterNotForced(cbtm, soln, days_frame, etm) )
	    soln_changed = true;
	}
	else
	  HnAbort("KheSolnAssignRequestedResources internal error");
      }
    }
  }

  /* and return */
  KheSolnArenaEnd(soln, a);
  if( DEBUG5 )
    KheFrameAssertNoClashes(days_frame);
  if( DEBUG1 )
  {
    if( soln_changed )
      fprintf(stderr,
	"] KheSolnAssignRequestedResources returning true (%.5f -> %.5f)\n",
	KheCostShow(init_cost), KheCostShow(KheSolnCost(soln)));
    else
      fprintf(stderr, "] KheSolnAssignRequestedResources returning false\n");
  }
  return soln_changed;
}
