
/*****************************************************************************/
/*                                                                           */
/*  THE KHE HIGH SCHOOL TIMETABLING ENGINE                                   */
/*  COPYRIGHT (C) 2010 Jeffrey H. Kingston                                   */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 3, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         khe_sr_single_resource.c                                   */
/*  DESCRIPTION:  Single resource solving                                    */
/*                                                                           */
/*****************************************************************************/
#include "khe_solvers.h"
#include "howard_p.h"

#define DEBUG1 1

/*****************************************************************************/
/*                                                                           */
/*  Submodule "type declarations" (private)                                  */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_ACTION - the type of action needed when updating one time group  */
/*                                                                           */
/*  In the documentation, UNDEFINED is state A, DEFINED is state B, and      */
/*  FINALIZED is state C.                                                    */
/*                                                                           */
/*****************************************************************************/

/* ***
typedef enum {
  KHE_SRS_ACTION_UNDEFINED_TO_DEFINED,
  KHE_SRS_ACTION_UNDEFINED_TO_FINALIZED,
  KHE_SRS_ACTION_DEFINED_TO_DEFINED,
  KHE_SRS_ACTION_DEFINED_TO_FINALIZED
} KHE_SRS_ACTION;
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_MEASURE - the measure of a time group (busy count, workload etc) */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  KHE_SRS_MEASURE_ACTIVE_BUSY,		/* active; polarity is positive      */
  KHE_SRS_MEASURE_ACTIVE_FREE,		/* active; polarity is negative      */
  KHE_SRS_MEASURE_COUNT_BUSY,		/* number of busy times              */
  KHE_SRS_MEASURE_WORKLOAD		/* total workload at busy times      */
} KHE_SRS_MEASURE;


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_TIME_GROUP - one time group being monitor by the solver          */
/*  KHE_SRS_TIME_GROUP_ON_DAY - one time group on one day                    */
/*  KHE_SRS_MONITOR - one monitor being monitored by the solver              */
/*  KHE_SRS_MONITOR_ON_DAY - one monitor on one day                          */
/*  KHE_SRS_TIME - one time                                                  */
/*  KHE_SRS_TASK - a task with associated attributes pre-calculated          */
/*  KHE_SRS_TASKS_AT_TIME - some tasks (the best) with a given first time    */
/*  KHE_SRS_DAY - one day and everything about it                            */
/*  KHE_SRS_PARTIAL_SOLN - a partial solution                                */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_srs_time_group_rec *KHE_SRS_TIME_GROUP;
typedef struct khe_srs_time_group_on_day_rec *KHE_SRS_TIME_GROUP_ON_DAY;
typedef struct khe_srs_monitor_rec *KHE_SRS_MONITOR;
typedef struct khe_srs_monitor_on_day_rec *KHE_SRS_MONITOR_ON_DAY;
typedef struct khe_srs_time_rec *KHE_SRS_TIME;
typedef struct khe_srs_task_rec *KHE_SRS_TASK;
typedef struct khe_srs_tasks_at_time_rec *KHE_SRS_TASKS_AT_TIME;
typedef struct khe_srs_day_rec *KHE_SRS_DAY;
typedef struct khe_srs_partial_soln_rec *KHE_SRS_PARTIAL_SOLN;


/*****************************************************************************/
/*                                                                           */
/*  Arrays and tables of the types just defined, as required                 */
/*                                                                           */
/*****************************************************************************/

typedef HA_ARRAY(KHE_SRS_TIME_GROUP) ARRAY_KHE_SRS_TIME_GROUP;
typedef HA_ARRAY(KHE_SRS_TIME_GROUP_ON_DAY) ARRAY_KHE_SRS_TIME_GROUP_ON_DAY;
typedef HA_ARRAY(KHE_SRS_MONITOR) ARRAY_KHE_SRS_MONITOR;
typedef HA_ARRAY(KHE_SRS_MONITOR_ON_DAY) ARRAY_KHE_SRS_MONITOR_ON_DAY;
typedef HA_ARRAY(KHE_SRS_TIME) ARRAY_KHE_SRS_TIME;
typedef HA_ARRAY(KHE_SRS_TASK) ARRAY_KHE_SRS_TASK;
typedef HA_ARRAY(KHE_SRS_TASKS_AT_TIME) ARRAY_KHE_SRS_TASKS_AT_TIME;
typedef HA_ARRAY(KHE_SRS_DAY) ARRAY_KHE_SRS_DAY;
typedef HA_ARRAY(KHE_SRS_PARTIAL_SOLN) ARRAY_KHE_SRS_PARTIAL_SOLN;
typedef HP_TABLE(KHE_SRS_PARTIAL_SOLN) TABLE_KHE_SRS_PARTIAL_SOLN;


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_TIME_GROUP - one time group being monitor by the solver          */
/*                                                                           */
/*****************************************************************************/

struct khe_srs_time_group_rec {
  KHE_SRS_MONITOR		monitor;	/* enclosing monitor         */
  KHE_TIME_GROUP		time_group;	/* the time group            */
  KHE_SRS_MEASURE		measure;	/* the measure of it we want */
  ARRAY_KHE_SRS_TIME_GROUP_ON_DAY time_group_on_days;
};


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_TIME_GROUP_ON_DAY - one time group on one day                    */
/*                                                                           */
/*****************************************************************************/

struct khe_srs_time_group_on_day_rec {
  KHE_SRS_TIME_GROUP		time_group;	/* the time group            */
  KHE_SRS_TIME_GROUP_ON_DAY	prev;		/* same tg on previous day   */
  /* KHE_SRS_TIME_GROUP_ON_DAY	next; */	/* same tg on next day       */
  /* KHE_SRS_ACTION		action; */	/* what to do on this day    */
  int				visit_num;	/* visit number              */
  int				sig_index;	/* if in signature           */
};


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_MONITOR - one monitor being monitored by the solver              */
/*                                                                           */
/*****************************************************************************/

struct khe_srs_monitor_rec {
  KHE_MONITOR			monitor;	/* the actual monitor        */
  ARRAY_KHE_SRS_TIME_GROUP	time_groups;	/* the monitor's time groups */
};


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_MONITOR_ON_DAY - one monitor on one day                          */
/*                                                                           */
/*****************************************************************************/

struct khe_srs_monitor_on_day_rec {
  KHE_SRS_MONITOR		monitor;	/* monitor this is about     */
  ARRAY_KHE_SRS_TIME_GROUP_ON_DAY time_group_on_days; /* time group on day   */
  KHE_SRS_MONITOR_ON_DAY	prev;		/* same monitor on prev day  */
  KHE_SRS_MONITOR_ON_DAY	next;		/* same monitor on next day  */
  /* KHE_SRS_ACTION		action; */	/* what to do on this day    */
  int				sig_index;	/* if in signature           */
};


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_TIME - one time                                                  */
/*                                                                           */
/*****************************************************************************/

struct khe_srs_time_rec {
  KHE_TIME			time;		/* the time represented      */
  ARRAY_KHE_SRS_TIME_GROUP_ON_DAY time_group_on_days; /* monitors this time  */
};


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_TASK - a task with associated attributes pre-calculated          */
/*                                                                           */
/*****************************************************************************/

struct khe_srs_task_rec {
  KHE_TASK			task;			/* the task          */
  int				first_index;		/* day index of f.t. */
  int				last_index;		/* day index of l.t. */
  ARRAY_KHE_TIME		times_by_day;		/* its running times */
  KHE_COST			cost_reduction;		/* its cost reduction*/
  int				domain_count;		/* its domain size   */
};


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_TASKS_AT_TIME - some tasks (the best) with a given first time    */
/*                                                                           */
/*****************************************************************************/

struct khe_srs_tasks_at_time_rec {
  KHE_TIME			first_time;
  ARRAY_KHE_SRS_TASK		tasks;
};


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_DAY - one day and everything about it                            */
/*                                                                           */
/*****************************************************************************/

struct khe_srs_day_rec {
  KHE_TIME_GROUP		time_group;		/* day's time group  */
  int				day_index;		/* index of this day */
  int				sig_len;		/* length of sig     */
  KHE_SRS_TASK			null_task;		/* free on this day  */
  ARRAY_KHE_SRS_MONITOR_ON_DAY	monitor_on_days;	/* day's monitors    */
  TABLE_KHE_SRS_PARTIAL_SOLN	partial_solns;		/* solns to day incl */
  ARRAY_KHE_SRS_TASKS_AT_TIME	tasks_at_time;		/* tasks to try      */
};


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_PARTIAL_SOLN - a partial solution                                */
/*                                                                           */
/*****************************************************************************/

struct khe_srs_partial_soln_rec {
  KHE_SRS_PARTIAL_SOLN	prev;			/* previous partial soln     */
  KHE_TASK		task;			/* asst leading to here      */
  int			best_of;		/* how many this is best of  */
  KHE_COST		cost;			/* resource cost so far      */
  HA_ARRAY_INT		signature;		/* signature of this node    */
};


/*****************************************************************************/
/*                                                                           */
/*  KHE_SINGLE_RESOURCE_SOLVER - a single resource solver                    */
/*                                                                           */
/*****************************************************************************/

struct khe_single_resource_solver_rec {
  HA_ARENA			arena;
  KHE_SOLN			soln;
  KHE_OPTIONS			options;
  int				visit_num;
  int				diversifier;
  KHE_FRAME			days_frame;
  KHE_EVENT_TIMETABLE_MONITOR	etm;
  KHE_TASK_FINDER		task_finder;
  KHE_RESOURCE			resource;	/* resource (if solving)     */
  /* KHE_GROUP_MONITOR		resource_gm; */	/* its gm (if solving)       */
  ARRAY_KHE_SRS_MONITOR		monitors;
  ARRAY_KHE_SRS_DAY		days;
  ARRAY_KHE_SRS_TIME		times;
  ARRAY_KHE_SRS_PARTIAL_SOLN	final_solns;
  /* ARRAY_KHE_SRS_PARTIAL_SOLN	all_partial_solns; what's this for? */

  ARRAY_KHE_SRS_TIME_GROUP	free_time_groups;
  ARRAY_KHE_SRS_TIME_GROUP_ON_DAY free_time_group_on_days;
  ARRAY_KHE_SRS_MONITOR		free_monitors;
  ARRAY_KHE_SRS_MONITOR_ON_DAY	free_monitor_on_days;
  ARRAY_KHE_SRS_TASK		free_tasks;
  ARRAY_KHE_SRS_TASKS_AT_TIME	free_tasks_at_time;
  ARRAY_KHE_SRS_PARTIAL_SOLN	free_partial_solns;
};


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_SRS_ACTION"                                               */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_ACTION KheSrsAction(int index, int first_index, int last_index)  */
/*                                                                           */
/*  Make the action for index, by comparing with first_index and last_index. */
/*                                                                           */
/*****************************************************************************/

/* ***
static KHE_SRS_ACTION KheSrsAction(int index, int first_index, int last_index)
{
  if( index == first_index )
  {
    if( index == last_index )
      return KHE_SRS_ACTION_UNDEFINED_TO_FINALIZED;
    else
      return KHE_SRS_ACTION_UNDEFINED_TO_DEFINED;
  }
  else
  {
    if( index == last_index )
      return KHE_SRS_ACTION_DEFINED_TO_FINALIZED;
    else
      return KHE_SRS_ACTION_DEFINED_TO_DEFINED;
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  char *KheSrsActionShow(KHE_SRS_ACTION action)                            */
/*                                                                           */
/*  Return a display of action as a static string.                           */
/*                                                                           */
/*****************************************************************************/

/* ***
static char *KheSrsActionShow(KHE_SRS_ACTION action)
{
  switch( action )
  {
    case KHE_SRS_ACTION_UNDEFINED_TO_DEFINED:	 return "undef->defin";
    case KHE_SRS_ACTION_UNDEFINED_TO_FINALIZED:	 return "undef->final";
    case KHE_SRS_ACTION_DEFINED_TO_DEFINED:	 return "defin->defin";
    case KHE_SRS_ACTION_DEFINED_TO_FINALIZED:	 return "defin->final";
    default:					 return "?";
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_SRS_MEASURE"                                              */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  char *KheSrsMeasureShow(KHE_SRS_MEASURE measure)                         */
/*                                                                           */
/*  Return a display of measure as a static string.                          */
/*                                                                           */
/*****************************************************************************/

static char *KheSrsMeasureShow(KHE_SRS_MEASURE measure)
{
  switch( measure )
  {
    case KHE_SRS_MEASURE_ACTIVE_BUSY:	 return "active_busy";
    case KHE_SRS_MEASURE_ACTIVE_FREE:	 return "active_free";
    case KHE_SRS_MEASURE_COUNT_BUSY:	 return "count_busy";
    case KHE_SRS_MEASURE_WORKLOAD:	 return "workload";
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_SRS_TIME_GROUP"                                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_TIME_GROUP KheSrsTimeGroupGet(KHE_SINGLE_RESOURCE_SOLVER srs)    */
/*                                                                           */
/*  Get a time group object from srs.  Only the time_group_on_days field     */
/*  is initialized.                                                          */
/*                                                                           */
/*****************************************************************************/

static KHE_SRS_TIME_GROUP KheSrsTimeGroupGet(KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_TIME_GROUP res;
  if( HaArrayCount(srs->free_time_groups) > 0 )
  {
    res = HaArrayLastAndDelete(srs->free_time_groups);
    HaArrayClear(res->time_group_on_days);
  }
  else
  {
    HaMake(res, srs->arena);
    HaArrayInit(res->time_group_on_days, srs->arena);
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_TIME_GROUP KheSrsTimeGroupMake(KHE_SRS_MONITOR sm,               */
/*    KHE_TIME_GROUP tg, KHE_SRS_MEASURE measure,                            */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Make a new time group object with these attributes.                      */
/*                                                                           */
/*****************************************************************************/

static KHE_SRS_TIME_GROUP KheSrsTimeGroupMake(KHE_SRS_MONITOR sm,
  KHE_TIME_GROUP tg, KHE_SRS_MEASURE measure,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_TIME_GROUP res;
  res = KheSrsTimeGroupGet(srs);
  res->monitor = sm;
  res->time_group = tg;
  res->measure = measure;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsTimeGroupDelete(KHE_SRS_TIME_GROUP stg,                       */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Delete stg and its time group on day objects.                            */
/*                                                                           */
/*****************************************************************************/

static void KheSrsTimeGroupDelete(KHE_SRS_TIME_GROUP stg,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  int i;
  HaArrayAppend(srs->free_time_group_on_days, stg->time_group_on_days, i);
  HaArrayAddLast(srs->free_time_groups, stg);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsTimeGroupDebug(KHE_SRS_TIME_GROUP stg, int verbosity,         */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of stg onto fp with the given verbosity and indent.          */
/*                                                                           */
/*****************************************************************************/

static void KheSrsTimeGroupDebug(KHE_SRS_TIME_GROUP stg, int verbosity,
  int indent, FILE *fp)
{
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  fprintf(fp, "%s ", KheSrsMeasureShow(stg->measure));
  KheTimeGroupDebug(stg->time_group, verbosity, indent, fp);
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_SRS_TIME_GROUP_ON_DAY"                                    */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_TIME_GROUP_ON_DAY KheSrsTimeGroupOnDayGet(                       */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Get a time group on day object from srs.  The fields are uninitialized.  */
/*                                                                           */
/*****************************************************************************/

static KHE_SRS_TIME_GROUP_ON_DAY KheSrsTimeGroupOnDayGet(
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_TIME_GROUP_ON_DAY res;
  if( HaArrayCount(srs->free_time_group_on_days) > 0 )
    res = HaArrayLastAndDelete(srs->free_time_group_on_days);
  else
    HaMake(res, srs->arena);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_TIME_GROUP_ON_DAY KheSrsTimeGroupOnDayMake(                      */
/*    KHE_SRS_TIME_GROUP stg, KHE_SRS_TIME_GROUP_ON_DAY prev,                */
/*    KHE_SRS_ACTION action, int sig_index, KHE_SINGLE_RESOURCE_SOLVER srs)  */
/*                                                                           */
/*  Make a new time group on day object with these attributes.               */
/*                                                                           */
/*****************************************************************************/

static KHE_SRS_TIME_GROUP_ON_DAY KheSrsTimeGroupOnDayMake(
  KHE_SRS_TIME_GROUP stg, KHE_SRS_TIME_GROUP_ON_DAY prev,
  /* KHE_SRS_TIME_GROUP_ON_DAY next, */ int sig_index,
  KHE_SINGLE_RESOURCE_SOLVER srs)
  /* KHE_SRS_ACTION action, */
{
  KHE_SRS_TIME_GROUP_ON_DAY res;
  res = KheSrsTimeGroupOnDayGet(srs);
  res->time_group = stg;
  res->prev = prev;
  /* res->next = next; */
  /* res->action = action; */
  res->sig_index = sig_index;
  res->visit_num = -1;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsTimeGroupOnDayVisit(KHE_SRS_TIME_GROUP_ON_DAY stgd,           */
/*    int visit_num)                                                         */
/*                                                                           */
/*  Mark stgd with this visit number, indicating that it is busy on its day. */
/*                                                                           */
/*****************************************************************************/

static void KheSrsTimeGroupOnDayVisit(KHE_SRS_TIME_GROUP_ON_DAY stgd,
  int visit_num)
{
  stgd->visit_num = visit_num;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsTimeGroupOnDayDelete(KHE_SRS_TIME_GROUP_ON_DAY stgd,          */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Delete stgd.                                                             */
/*                                                                           */
/*****************************************************************************/

static void KheSrsTimeGroupOnDayDelete(KHE_SRS_TIME_GROUP_ON_DAY stgd,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  HaArrayAddLast(srs->free_time_group_on_days, stgd);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsTimeGroupOnDayAddToSignature(KHE_SRS_TIME_GROUP_ON_DAY stgd,  */
/*    KHE_SRS_PARTIAL_SOLN ps, KHE_TIME time, int visit_num)                 */
/*                                                                           */
/*  Work out stgd's contribution to the signature of ps, and add it in.      */
/*                                                                           */
/*  Implementation note.  Before this function is called, stgd->visit_num    */
/*  will be set to visit_num if stgd's time group contains time, and it      */
/*  will have some older, different value if stgd's time group does not      */
/*  contain time, or if time is NULL.  So (stgd->visit_num == visit_num)     */
/*  is true when the task currently being assigned makes stgd busy.          */
/*                                                                           */
/*****************************************************************************/

static void KheSrsTimeGroupOnDayAddToSignature(KHE_SRS_TIME_GROUP_ON_DAY stgd,
  KHE_SRS_PARTIAL_SOLN ps, KHE_TIME time, int visit_num)
{
  bool busy;  int prev_val, curr_val;

  /* work out curr_val, the value of time group with this time added */
  busy = (stgd->visit_num == visit_num);
  if( stgd->prev == NULL )
  {
    /* no previous value, the time group starts here */
    switch( stgd->time_group->measure )
    {
      case KHE_SRS_MEASURE_ACTIVE_BUSY:

	curr_val = busy ? 1 : 0;
	break;

      case KHE_SRS_MEASURE_ACTIVE_FREE:

	curr_val = !busy ? 1 : 0;
	break;

      case KHE_SRS_MEASURE_COUNT_BUSY:

	curr_val = busy ? 1 : 0;
	break;

      case KHE_SRS_MEASURE_WORKLOAD:

	curr_val = (int) (KheTaskWorkloadPerTime(ps->task) * 100.0);
	break;

      default:

	HnAbort("KheSrsTimeGroupOnDayAddToSignature internal error 1");
	curr_val = 0;  /* keep compiler happy */
	break;
    }
  }
  else
  {
    /* there is a previous value, we need to include it */
    prev_val = HaArray(ps->signature, stgd->prev->sig_index);
    switch( stgd->time_group->measure )
    {
      case KHE_SRS_MEASURE_ACTIVE_BUSY:

	curr_val = (prev_val == 1 || busy) ? 1 : 0;
	break;

      case KHE_SRS_MEASURE_ACTIVE_FREE:

	curr_val = (prev_val == 1 && !busy) ? 1 : 0;
	break;

      case KHE_SRS_MEASURE_COUNT_BUSY:

	curr_val = prev_val + (busy ? 1 : 0);
	break;

      case KHE_SRS_MEASURE_WORKLOAD:

	curr_val = prev_val + (int) (KheTaskWorkloadPerTime(ps->task) * 100.0);
	break;

      default:

	HnAbort("KheSrsTimeGroupOnDayAddToSignature internal error 2");
	curr_val = 0;  /* keep compiler happy */
	break;
    }
  }

  /* do something with curr_val, either add it to signature or otherwise */
  if( stgd->sig_index != -1 )
  {
    /* add curr_val to signature, at sig_index */
    HnAssert(HaArrayCount(ps->signature) == stgd->sig_index,
      "KheSrsTimeGroupOnDayAddToSignature internal error 3");
    HaArrayAddLast(ps->signature, curr_val);
  }
  else
  {
    /* value of time group is finalized, it needs to be reported back now */
    /* still to do */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsTimeGroupOnDayDebug(KHE_SRS_TIME_GROUP_ON_DAY stgd,           */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Make a debug print of stgd onto fp with the given verbosity and indent.  */
/*                                                                           */
/*****************************************************************************/

static void KheSrsTimeGroupOnDayDebug(KHE_SRS_TIME_GROUP_ON_DAY stgd,
  int verbosity, int indent, FILE *fp)
{
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  if( stgd->prev == NULL )
    fprintf(fp, "first ");
  if( stgd->sig_index == -1 )
    fprintf(fp, "last ");
  /* fprintf(fp, "%s ", KheSrsActionShow(stgd->action)); */
  KheTimeGroupDebug(stgd->time_group->time_group, verbosity, indent, fp);
  fprintf(fp, " [%d]", stgd->sig_index);
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_SRS_MONITOR"                                              */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_MONITOR KheSrsMonitorGet(KHE_SINGLE_RESOURCE_SOLVER srs)         */
/*                                                                           */
/*  Get a new monitor object from srs.  Only the time_groups field is        */
/*  initialized.                                                             */
/*                                                                           */
/*****************************************************************************/

static KHE_SRS_MONITOR KheSrsMonitorGet(KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_MONITOR res;
  if( HaArrayCount(srs->free_monitors) > 0 )
  {
    res = HaArrayLastAndDelete(srs->free_monitors);
    HaArrayClear(res->time_groups);
  }
  else
  {
    HaMake(res, srs->arena);
    HaArrayInit(res->time_groups, srs->arena);
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_MONITOR KheSrsMonitorMake(KHE_MONITOR m,                         */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Make a new monitor object with these attributes.                         */
/*                                                                           */
/*****************************************************************************/

static KHE_SRS_MONITOR KheSrsMonitorMake(KHE_MONITOR m,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_MONITOR res;
  res = KheSrsMonitorGet(srs);
  res->monitor = m;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsMonitorDelete(KHE_SRS_MONITOR sm,                             */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Delete sm and its monitor on day objects.                                */
/*                                                                           */
/*****************************************************************************/

static void KheSrsMonitorDelete(KHE_SRS_MONITOR sm,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  int i;  KHE_SRS_TIME_GROUP stg;
  HaArrayForEach(sm->time_groups, stg, i)
    KheSrsTimeGroupDelete(stg, srs);
  HaArrayAddLast(srs->free_monitors, sm);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsMonitorDebug(KHE_SRS_MONITOR sm, int verbosity,               */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of sm onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void KheSrsMonitorDebug(KHE_SRS_MONITOR sm, int verbosity,
  int indent, FILE *fp)
{
  KHE_SRS_TIME_GROUP stg;  int i;
  if( indent >= 0 )
  {
    if( indent > 0 )
      fprintf(fp, "%*s", indent, "");
    fprintf(fp, "[ %s %s\n", KheMonitorTagShow(KheMonitorTag(sm->monitor)),
      KheMonitorId(sm->monitor));
    HaArrayForEach(sm->time_groups, stg, i)
      KheSrsTimeGroupDebug(stg, verbosity, indent, fp);
    fprintf(fp, "%*s]\n", indent, "");
  }
  else
  {
    fprintf(fp, "[ %s %s (%d time groups) ]",
      KheMonitorTagShow(KheMonitorTag(sm->monitor)),
      KheMonitorId(sm->monitor), HaArrayCount(sm->time_groups));
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_SRS_MONITOR_ON_DAY"                                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_MONITOR_ON_DAY KheSrsMonitorOnDayGet(                            */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Get a new monitor on day object from srs.  Only the time_group_on_days   */
/*  field is initialized.                                                    */
/*                                                                           */
/*****************************************************************************/

static KHE_SRS_MONITOR_ON_DAY KheSrsMonitorOnDayGet(
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_MONITOR_ON_DAY res;
  if( HaArrayCount(srs->free_monitor_on_days) > 0 )
  {
    res = HaArrayLastAndDelete(srs->free_monitor_on_days);
    HaArrayClear(res->time_group_on_days);
  }
  else
  {
    HaMake(res, srs->arena);
    HaArrayInit(res->time_group_on_days, srs->arena);
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_MONITOR_ON_DAY KheSrsMonitorOnDayMake(KHE_SRS_MONITOR sm,        */
/*    KHE_SRS_MONITOR_ON_DAY prev, KHE_SRS_MONITOR_ON_DAY next,              */
/*    int sig_index, KHE_SINGLE_RESOURCE_SOLVER srs)                         */
/*                                                                           */
/*  Make a new monitor on day object with these attributes.                  */
/*                                                                           */
/*****************************************************************************/

static KHE_SRS_MONITOR_ON_DAY KheSrsMonitorOnDayMake(KHE_SRS_MONITOR sm,
  KHE_SRS_MONITOR_ON_DAY prev, KHE_SRS_MONITOR_ON_DAY next,
  int sig_index, KHE_SINGLE_RESOURCE_SOLVER srs)
  /* KHE_SRS_ACTION action, */
{
  KHE_SRS_MONITOR_ON_DAY res;
  res = KheSrsMonitorOnDayGet(srs);
  res->monitor = sm;
  res->prev = prev;
  res->next = next;
  /* res->action = action; */
  res->sig_index = sig_index;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsMonitorOnDayDelete(KHE_SRS_MONITOR_ON_DAY smd,                */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Delete smd.  Do not delete its time group on day objects, because they   */
/*  will be deleted separately.                                              */
/*                                                                           */
/*****************************************************************************/

static void KheSrsMonitorOnDayDelete(KHE_SRS_MONITOR_ON_DAY smd,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  HaArrayAddLast(srs->free_monitor_on_days, smd);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsMonitorOnDayAddToSignature(KHE_SRS_MONITOR_ON_DAY smd,        */
/*    KHE_SRS_PARTIAL_SOLN ps, KHE_TIME time, int visit_num)                 */
/*                                                                           */
/*  Work out smd's contribution to the signature of ps and add it in.        */
/*  The task is assigned at time, or free if time is NULL.                   */
/*                                                                           */
/*****************************************************************************/

static void KheSrsMonitorOnDayAddToSignature(KHE_SRS_MONITOR_ON_DAY smd,
  KHE_SRS_PARTIAL_SOLN ps, KHE_TIME time, int visit_num)
{
  KHE_SRS_TIME_GROUP_ON_DAY stgd;  int i;
  HaArrayForEach(smd->time_group_on_days, stgd, i)
    KheSrsTimeGroupOnDayAddToSignature(stgd, ps, time, visit_num);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsMonitorOnDayDebug(KHE_SRS_MONITOR_ON_DAY smd,                 */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of smd onto fp with the given verbosity and indent.          */
/*                                                                           */
/*****************************************************************************/

static void KheSrsMonitorOnDayDebug(KHE_SRS_MONITOR_ON_DAY smd,
  int verbosity, int indent, FILE *fp)
{
  KHE_SRS_TIME_GROUP_ON_DAY stgd;  int i;
  if( indent >= 0 )
  {
    if( indent > 0 )
      fprintf(fp, "%*s", indent, "");
    fprintf(fp, "[ /%d/ %s %s\n", smd->sig_index,
      KheMonitorTagShow(KheMonitorTag(smd->monitor->monitor)),
      KheMonitorId(smd->monitor->monitor));
    HaArrayForEach(smd->time_group_on_days, stgd, i)
      KheSrsTimeGroupOnDayDebug(stgd, verbosity, indent, fp);
    fprintf(fp, "%*s]\n", indent, "");
  }
  else
  {
    fprintf(fp, "[ /%d/ %s %s (%d time groups) ]", smd->sig_index,
      KheMonitorTagShow(KheMonitorTag(smd->monitor->monitor)),
      KheMonitorId(smd->monitor->monitor),
      HaArrayCount(smd->time_group_on_days));
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_SRS_TIME"                                                 */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_TIME KheSrsTimeMake(KHE_TIME t, HA_ARENA a)                      */
/*                                                                           */
/*  Make and return a new srs time object representing t.                    */
/*                                                                           */
/*****************************************************************************/

static KHE_SRS_TIME KheSrsTimeMake(KHE_TIME t, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_TIME res;
  HaMake(res, srs->arena);
  res->time = t;
  HaArrayInit(res->time_group_on_days, srs->arena);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsTimeClear(KHE_SRS_TIME st)                                    */
/*                                                                           */
/*  Clear st, that is, clear out its monitors.                               */
/*                                                                           */
/*  There is no function to delete st because that is never wanted.          */
/*                                                                           */
/*****************************************************************************/

static void KheSrsTimeClear(KHE_SRS_TIME st)
{
  HaArrayClear(st->time_group_on_days);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsTimeDebug(KHE_SRS_TIME st, int verbosity,                     */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of st onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void KheSrsTimeDebug(KHE_SRS_TIME st, int verbosity,
  int indent, FILE *fp)
{
  KHE_SRS_TIME_GROUP_ON_DAY stgd;  int i;
  if( indent >= 0 )
  {
    fprintf(fp, "%*s[ Time %s:\n", indent, "", KheTimeId(st->time));
    HaArrayForEach(st->time_group_on_days, stgd, i)
      KheSrsTimeGroupOnDayDebug(stgd, verbosity, indent + 2, fp);
    fprintf(fp, "%*s]\n", indent, "");
  }
  else
    fprintf(fp, "[ Time %s (%d time groups) ]", KheTimeId(st->time),
      HaArrayCount(st->time_group_on_days));
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_SRS_TASK"                                                 */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_TASK KheSrsTaskGet(KHE_SINGLE_RESOURCE_SOLVER srs)               */
/*                                                                           */
/*  Get a new srs task object.  Only the times_by_day field is initialized.  */
/*                                                                           */
/*****************************************************************************/

static KHE_SRS_TASK KheSrsTaskGet(KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_TASK res;
  if( HaArrayCount(srs->free_tasks) > 0 )
  {
    res = HaArrayLastAndDelete(srs->free_tasks);
    HaArrayClear(res->times_by_day);
  }
  else
  {
    HaMake(res, srs->arena);
    HaArrayInit(res->times_by_day, srs->arena);
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_TASK KheSrsTaskMake(KHE_TASK task,                               */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Make a new srs task from task.                                           */
/*                                                                           */
/*  This code aborts if task has no assigned times.  That is safe given      */
/*  the context of the sole call to this function below.                     */
/*                                                                           */
/*****************************************************************************/
static void KheSrsTaskDelete(KHE_SRS_TASK st, KHE_SINGLE_RESOURCE_SOLVER srs);

static KHE_SRS_TASK KheSrsTaskMake(KHE_TASK task,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_TASK res;
  res = KheSrsTaskGet(srs);
  res->task = task;
  KheTaskFinderTaskIntervalAndTimes(srs->task_finder, task, &res->first_index,
    &res->last_index, &res->times_by_day);
  res->cost_reduction = KheTaskAssignmentCostReduction(task);
  res->domain_count = KheResourceGroupResourceCount(KheTaskDomain(task));
  HnAssert(HaArrayCount(res->times_by_day), "KheSrsTaskMake internal error 1");
  HnAssert(HaArrayFirst(res->times_by_day) != NULL,
    "KheSrsTaskMake internal error 2");
  HnAssert(HaArrayLast(res->times_by_day) != NULL,
    "KheSrsTaskMake internal error 3");
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_TASK KheSrsTaskMakeNull(int day_index,                           */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Make a new srs task which represents doing nothing on the day with       */
/*  this day_index.                                                          */
/*                                                                           */
/*****************************************************************************/

static KHE_SRS_TASK KheSrsTaskMakeNull(int day_index,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_TASK res;
  res = KheSrsTaskGet(srs);
  res->task = NULL;
  res->first_index = res->last_index = day_index;
  HaArrayAddLast(res->times_by_day, NULL);
  res->cost_reduction = 0;
  res->domain_count = 0;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsTaskDelete(KHE_SRS_TASK st, KHE_SINGLE_RESOURCE_SOLVER srs)   */
/*                                                                           */
/*  Delete st.                                                               */
/*                                                                           */
/*****************************************************************************/

static void KheSrsTaskDelete(KHE_SRS_TASK st, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  HaArrayAddLast(srs->free_tasks, st);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_TIME KheSrsTaskFirstTime(KHE_SRS_TASK st)                            */
/*                                                                           */
/*  Return the first time of st.                                             */
/*                                                                           */
/*****************************************************************************/

static KHE_TIME KheSrsTaskFirstTime(KHE_SRS_TASK st)
{
  return HaArrayFirst(st->times_by_day);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_TIME KheSrsTaskLastTime(KHE_SRS_TASK st)                             */
/*                                                                           */
/*  Return the last time of st.                                              */
/*                                                                           */
/*****************************************************************************/

static KHE_TIME KheSrsTaskLastTime(KHE_SRS_TASK st)
{
  return HaArrayLast(st->times_by_day);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheSrsTaskBetterThan(KHE_SRS_TASK st1, KHE_SRS_TASK st2)            */
/*                                                                           */
/*  Return true if st1 is a better choice than st2, because it has a larger  */
/*  cost reduction, or equal cost reduction and a smaller domain.            */
/*                                                                           */
/*****************************************************************************/

static bool KheSrsTaskBetterThan(KHE_SRS_TASK st1, KHE_SRS_TASK st2)
{
  return st1->cost_reduction > st2->cost_reduction ||
    (st1->cost_reduction == st2->cost_reduction &&
     st1->domain_count < st2->domain_count);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsTaskDebug(KHE_SRS_TASK st, int verbosity,                     */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of st onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void KheSrsTaskDebug(KHE_SRS_TASK st, int verbosity,
  int indent, FILE *fp)
{
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  fprintf(fp, "(%s-%s, cost %.5f, domain %d) ",
    KheTimeId(KheSrsTaskFirstTime(st)), KheTimeId(KheSrsTaskLastTime(st)),
    KheCostShow(st->cost_reduction), st->domain_count);
  KheTaskDebug(st->task, verbosity, -1, fp);
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_SRS_TASKS_AT_TIME"                                        */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_TASKS_AT_TIME KheSrsTasksAtTimeGet(                              */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Get a tasks at time object from srs.  Only the tasks field is            */
/*  initialized.                                                             */
/*                                                                           */
/*****************************************************************************/

static KHE_SRS_TASKS_AT_TIME KheSrsTasksAtTimeGet(
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_TASKS_AT_TIME res;
  if( HaArrayCount(srs->free_tasks_at_time) > 0 )
  {
    res = HaArrayLastAndDelete(srs->free_tasks_at_time);
    HaArrayClear(res->tasks);
  }
  else
  {
    HaMake(res, srs->arena);
    HaArrayInit(res->tasks, srs->arena);
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_TASKS_AT_TIME KheSrsTasksAtTimeMake(KHE_TIME first_time,         */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Return a new tasks at time object with these attributes.                 */
/*                                                                           */
/*****************************************************************************/

static KHE_SRS_TASKS_AT_TIME KheSrsTasksAtTimeMake(KHE_TIME first_time,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_TASKS_AT_TIME res;
  res = KheSrsTasksAtTimeGet(srs);
  res->first_time = first_time;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheSrsTasksAtTimeContainsLastTime(KHE_SRS_TASKS_AT_TIME stt,        */
/*    KHE_TIME last_time, KHE_SRS_TASK *st, int *pos)                        */
/*                                                                           */
/*  If stt contains a task with the given last_time, return true with *st    */
/*  set to that time and *pos set to its position.  Otherwise return false   */
/*  with *st set to NULL and *pos set to -1.                                 */
/*                                                                           */
/*****************************************************************************/

static bool KheSrsTasksAtTimeContainsLastTime(KHE_SRS_TASKS_AT_TIME stt,
  KHE_TIME last_time, KHE_SRS_TASK *st, int *pos)
{
  KHE_SRS_TASK st2;  int i;
  HaArrayForEach(stt->tasks, st2, i)
    if( KheSrsTaskLastTime(st2) == last_time )
      return *st = st2, *pos = i, true;
  return *st = NULL, *pos = -1, false;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsTasksAtTimeAddTask(KHE_SRS_TASKS_AT_TIME stt,                 */
/*    KHE_SRS_TASK st, KHE_SINGLE_RESOURCE_SOLVER srs)                       */
/*                                                                           */
/*  Add st to stt.  More precisely, if it's better than what stt already     */
/*  has, then add it, otherwise delete it.                                   */
/*                                                                           */
/*****************************************************************************/

static void KheSrsTasksAtTimeAddTask(KHE_SRS_TASKS_AT_TIME stt,
  KHE_SRS_TASK st, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_TASK st2;  int x;
  if( !KheSrsTasksAtTimeContainsLastTime(stt, KheSrsTaskLastTime(st), &st2,&x) )
  {
    /* stt has no task with the same last time as st, so add st to stt */
    HaArrayAddLast(stt->tasks, st);
  }
  else if( KheSrsTaskBetterThan(st, st2) )
  {
    /* st is better than the similar task st2, so replace st2 by st */
    KheSrsTaskDelete(st2, srs);
    HaArrayPut(stt->tasks, x, st);
  }
  else
  {
    /* st is worse than the similar task st2, so delete st */
    KheSrsTaskDelete(st, srs);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsTasksAtTimeDelete(KHE_SRS_TASKS_AT_TIME stt,                  */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Delete stt, placing its objects onto free lists in srs.                  */
/*                                                                           */
/*****************************************************************************/

static void KheSrsTasksAtTimeDelete(KHE_SRS_TASKS_AT_TIME stt,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  int i;
  HaArrayAppend(srs->free_tasks, stt->tasks, i);
  HaArrayAddLast(srs->free_tasks_at_time, stt);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsTasksAtTimeDebug(KHE_SRS_TASKS_AT_TIME stt,                   */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of stt onto fp with the given verbosity and indent.          */
/*                                                                           */
/*****************************************************************************/

static void KheSrsTasksAtTimeDebug(KHE_SRS_TASKS_AT_TIME stt,
  int verbosity, int indent, FILE *fp)
{
  KHE_SRS_TASK st;  int i;
  fprintf(fp, "%*s[ Tasks at %s:\n", indent, "",
    KheTimeId(stt->first_time));
  HaArrayForEach(stt->tasks, st, i)
    KheSrsTaskDebug(st, verbosity, indent, fp);
  fprintf(fp, "%*s]\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_SRS_DAY"                                                  */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_DAY KheSrsDayMake(KHE_TIME_GROUP tg, int day_index, HA_ARENA a)  */
/*                                                                           */
/*  Make a new day object for a day with times tg.                           */
/*                                                                           */
/*****************************************************************************/
static int KheSrsPartialSolnSignatureHashUntyped(void *p);
static bool KheSrsPartialSolnSignaturesEqualUntyped(void *p1, void *p2);

static KHE_SRS_DAY KheSrsDayMake(KHE_TIME_GROUP tg, int day_index,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_DAY res;
  HaMake(res, srs->arena);
  res->time_group = tg;
  res->day_index = day_index;
  res->sig_len = 1;  /* assignment count is always at start of signature */
  res->null_task = KheSrsTaskMakeNull(day_index, srs);
  HaArrayInit(res->monitor_on_days, srs->arena);
  HpTableInit(res->partial_solns, &KheSrsPartialSolnSignatureHashUntyped,
    &KheSrsPartialSolnSignaturesEqualUntyped, srs->arena);
  HaArrayInit(res->tasks_at_time, srs->arena);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsDayClearTasksAtTime(KHE_SRS_DAY day,                          */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Clear day's tasks at time array.                                         */
/*                                                                           */
/*****************************************************************************/

static void KheSrsDayClearTasksAtTime(KHE_SRS_DAY day,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  int i;  KHE_SRS_TASKS_AT_TIME stt;
  HaArrayForEach(day->tasks_at_time, stt, i)
    KheSrsTasksAtTimeDelete(stt, srs);
  HaArrayClear(day->tasks_at_time);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsDayClear(KHE_SRS_DAY day, KHE_SINGLE_RESOURCE_SOLVER srs)     */
/*                                                                           */
/*  Clear day, ready for a fresh solve.                                      */
/*                                                                           */
/*****************************************************************************/
static void KheSrsPartialSolnDelete(KHE_SRS_PARTIAL_SOLN ps,
  KHE_SINGLE_RESOURCE_SOLVER srs);

static void KheSrsDayClear(KHE_SRS_DAY day, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  int i, pos;  KHE_SRS_PARTIAL_SOLN ps;
  day->sig_len = 1;
  HaArrayAppend(srs->free_monitor_on_days, day->monitor_on_days, i);
  HaArrayClear(day->monitor_on_days);
  HpTableForEachValue(day->partial_solns, ps, pos)
    KheSrsPartialSolnDelete(ps, srs);
  HpTableClear(day->partial_solns);
  KheSrsDayClearTasksAtTime(day, srs);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_TASKS_AT_TIME KheSrsDayMakeAndAddTasksAtTime(KHE_SRS_DAY day,    */
/*    KHE_TIME t, KHE_SINGLE_RESOURCE_SOLVER srs)                            */
/*                                                                           */
/*  Make, add to day, and return a new best tasks at times object with       */
/*  time t.                                                                  */
/*                                                                           */
/*****************************************************************************/

static KHE_SRS_TASKS_AT_TIME KheSrsDayMakeAndAddTasksAtTime(KHE_SRS_DAY day,
  KHE_TIME t, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_TASKS_AT_TIME res;
  res = KheSrsTasksAtTimeMake(t, srs);
  HaArrayAddLast(day->tasks_at_time, res);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsDayAddTasks(KHE_SRS_DAY day, KHE_SINGLE_RESOURCE_SOLVER srs)  */
/*                                                                           */
/*  Add suitable tasks to day.                                               */
/*                                                                           */
/*****************************************************************************/

static void KheSrsDayAddTasks(KHE_SRS_DAY day, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  int i, j, k;  KHE_TIME t;  KHE_TASK task;
  KHE_SRS_TASKS_AT_TIME stt;  KHE_SRS_TASK st;  KHE_MEET meet;
  for( i = 0;  i < KheTimeGroupTimeCount(day->time_group);  i++ )
  {
    /* for each time of day */
    t = KheTimeGroupTime(day->time_group, i);
    stt = KheSrsDayMakeAndAddTasksAtTime(day, t, srs);
    for( j = 0;  j < KheEventTimetableMonitorTimeMeetCount(srs->etm, t);  j++ )
    {
      /* for each meet running at time */
      meet = KheEventTimetableMonitorTimeMeet(srs->etm, t, j);
      for( k = 0;  k < KheMeetTaskCount(meet);  k++ )
      {
	/* for the proper root task of each task of meet */
	task = KheTaskProperRoot(KheMeetTask(meet, k));
	if( KheTaskAsstResource(task) == srs->resource )
	{
	  /* task is already assigned, so it is the only legal choice */
	  KheSrsDayClearTasksAtTime(day, srs);
	  stt = KheSrsDayMakeAndAddTasksAtTime(day, t, srs);
	  st = KheSrsTaskMake(task, srs);
	  HaArrayAddLast(stt->tasks, st);
	  return;
	}
	else if( KheTaskAssignResourceCheck(task, srs->resource) )
	{
	  st = KheSrsTaskMake(task, srs);
	  if( KheSrsTaskFirstTime(st) == t )
	  {
	    /* add task to stt, possibly replacing an existing task */
	    KheSrsTasksAtTimeAddTask(stt, st, srs);
	  }
	  else
	  {
	    /* st starts at the wrong time, so ignore it */
	    KheSrsTaskDelete(st, srs);
	  }
	}
      }
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsDayDebug(KHE_SRS_DAY day, int verbosity, int indent, FILE *fp)*/
/*                                                                           */
/*  Debug print of day onto fp.                                              */
/*                                                                           */
/*****************************************************************************/

static void KheSrsDayDebug(KHE_SRS_DAY day, int verbosity, int indent, FILE *fp)
{
  int i;  KHE_SRS_MONITOR_ON_DAY md;  KHE_SRS_TASKS_AT_TIME stt;
  if( indent >= 0 )
  {
    fprintf(fp, "%*s[ ", indent, "");
    KheTimeGroupDebug(day->time_group, 1, 0, stderr);
    HaArrayForEach(day->monitor_on_days, md, i)
      KheSrsMonitorOnDayDebug(md, verbosity, indent + 2, fp);
    HaArrayForEach(day->tasks_at_time, stt, i)
      KheSrsTasksAtTimeDebug(stt, verbosity, indent + 2, fp);
    fprintf(fp, "%*s]\n", indent, "");
  }
  else
    KheTimeGroupDebug(day->time_group, 1, -1, stderr);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_SRS_PARTIAL_SOLN"                                         */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheSrsPartialSolnAddAvoidClashesSignature(KHE_SRS_PARTIAL_SOLN ps,     */
/*    KHE_MONITOR_INFO mi, KHE_AVOID_CLASHES_MONITOR m)                      */
/*                                                                           */
/*  Add a summary of the current state of m to ps's signature.               */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheSrsPartialSolnAddAvoidClashesSignature(KHE_SRS_PARTIAL_SOLN ps,
  KHE_MONITOR_INFO mi, KHE_AVOID_CLASHES_MONITOR m)
{
  ** nothing to do here, in fact should never happen **
  HnAbort("KheSrsPartialSolnAddAvoidClashesSignature internal error");
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsPartialSolnAddAvoidUnavailableTimesSignature(                 */
/*    KHE_SRS_PARTIAL_SOLN ps, KHE_MONITOR_INFO mi,                          */
/*    KHE_AVOID_UNAVAILABLE_TIMES_MONITOR m)                                 */
/*                                                                           */
/*  Add a summary of the current state of m to ps's signature.               */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheSrsPartialSolnAddAvoidUnavailableTimesSignature(
  KHE_SRS_PARTIAL_SOLN ps, KHE_MONITOR_INFO mi,
  KHE_AVOID_UNAVAILABLE_TIMES_MONITOR m)
{
  ** nothing to do here, in fact should never happen **
  HnAbort("KheSrsPartialSolnAddAvoidUnavailableTimesSignature internal error");
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsPartialSolnAddLimitIdleTimesSignature(KHE_SRS_PARTIAL_SOLN ps,*/
/*    KHE_MONITOR_INFO mi, KHE_LIMIT_IDLE_TIMES_MONITOR m)                   */
/*                                                                           */
/*  Add a summary of the current state of m to ps's signature.               */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheSrsPartialSolnAddLimitIdleTimesSignature(KHE_SRS_PARTIAL_SOLN ps,
  KHE_MONITOR_INFO mi, KHE_LIMIT_IDLE_TIMES_MONITOR m)
{
  ** nothing to do here, in fact should never happen **
  HnAbort("KheSrsPartialSolnAddLimitIdleTimesSignature internal error");
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsPartialSolnAddClusterBusyTimesSignature(KHE_SRS_PARTIAL_SOLN ps,*/
/*    KHE_MONITOR_INFO mi, KHE_CLUSTER_BUSY_TIMES_MONITOR m)                 */
/*                                                                           */
/*  Add a summary of the current state of m to ps's signature.               */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheSrsPartialSolnAddClusterBusyTimesSignature(KHE_SRS_PARTIAL_SOLN ps,
  KHE_MONITOR_INFO mi, KHE_CLUSTER_BUSY_TIMES_MONITOR m)
{
  int active_group_count, open_group_count, minimum, maximum, index, i, val;
  bool allow_zero;  KHE_TIME_GROUP tg;  KHE_POLARITY po;  int busy_count;

  ** add the number of active time groups to the signature **
  KheClusterBusyTimesMonitorActiveTimeGroupCount(m, &active_group_count,
    &open_group_count, &minimum, &maximum, &allow_zero);
  HaArrayAddLast(ps->signature, active_group_count);

  ** add one Boolean (0 or 1) for each indexed time group **
  HaArrayForEach(mi->indexes, index, i)
  {
    val = (KheClusterBusyTimesMonitorTimeGroupIsActive(m, index, &tg, &po,
      &busy_count) ? 1 : 0);
    HaArrayAddLast(ps->signature, val);
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsPartialSolnAddLimitBusyTimesSignature(KHE_SRS_PARTIAL_SOLN ps,*/
/*    KHE_MONITOR_INFO mi, KHE_LIMIT_BUSY_TIMES_MONITOR m)                   */
/*                                                                           */
/*  Add a summary of the current state of m to ps's signature.               */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheSrsPartialSolnAddLimitBusyTimesSignature(KHE_SRS_PARTIAL_SOLN ps,
  KHE_MONITOR_INFO mi, KHE_LIMIT_BUSY_TIMES_MONITOR m)
{
  int index, i, busy_count;

  ** add the deviation to the signature **
  HaArrayAddLast(ps->signature, KheLimitBusyTimesMonitorDeviation(m));

  ** add the busy count of each indexed time group to the signature **
  HaArrayForEach(mi->indexes, index, i)
  {
    KheLimitBusyTimesMonitorTimeGroup(m, index, &busy_count);
    HaArrayAddLast(ps->signature, busy_count);
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsPartialSolnAddLimitWorkloadSignature(KHE_SRS_PARTIAL_SOLN ps, */
/*    KHE_MONITOR_INFO mi, KHE_LIMIT_WORKLOAD_MONITOR m)                     */
/*                                                                           */
/*  Add a summary of the current state of m to ps's signature.               */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheSrsPartialSolnAddLimitWorkloadSignature(KHE_SRS_PARTIAL_SOLN ps,
  KHE_MONITOR_INFO mi, KHE_LIMIT_WORKLOAD_MONITOR m)
{
  int index, i;  float workload;

  ** add the deviation to the signature **
  HaArrayAddLast(ps->signature, KheLimitWorkloadMonitorDeviation(m));

  ** add the approximate busy count of each indexed time group to signature **
  HaArrayForEach(mi->indexes, index, i)
  {
    KheLimitWorkloadMonitorTimeGroup(m, index, &workload);
    HaArrayAddLast(ps->signature, (int) (workload * 10.0));
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsPartialSolnAddLimitActiveIntervalsSignature(KHE_SRS_PARTIAL_SOLN ps,*/
/*    KHE_MONITOR_INFO mi, KHE_LIMIT_ACTIVE_INTERVALS_MONITOR m)             */
/*                                                                           */
/*  Add a summary of the current state of m to ps's signature.               */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheSrsPartialSolnAddLimitActiveIntervalsSignature(KHE_SRS_PARTIAL_SOLN ps,
  KHE_MONITOR_INFO mi, KHE_LIMIT_ACTIVE_INTERVALS_MONITOR m)
{
  int index, i, val, busy_count;  KHE_TIME_GROUP tg;  KHE_POLARITY po;

  ** add the number of adjacent active time groups to the signature **
  ** adjacent to what? still to do **

  ** add one Boolean (0 or 1) for each indexed time group **
  HaArrayForEach(mi->indexes, index, i)
  {
    val = (KheLimitActiveIntervalsMonitorTimeGroupIsActive(m, index, &tg, &po,
      &busy_count) ? 1 : 0);
    HaArrayAddLast(ps->signature, val);
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsPartialSolnSetSignature(KHE_SRS_PARTIAL_SOLN ps, KHE_SRS_DAY day)        */
/*                                                                           */
/*  Set the signature of ps.  It ends on day.                                */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheSrsPartialSolnSetSignature(KHE_SRS_PARTIAL_SOLN ps, KHE_SRS_DAY day)
{
  KHE_MONITOR_INFO mi;  int i;
  HaArrayForEach(day->monitor_info, mi, i)
    switch( KheMonitorTag(mi->monitor) )
    {
      case KHE_AVOID_CLASHES_MONITOR_TAG:

	KheSrsPartialSolnAddAvoidClashesSignature(ps, mi,
	  (KHE_AVOID_CLASHES_MONITOR) mi->monitor);
	break;

      case KHE_AVOID_UNAVAILABLE_TIMES_MONITOR_TAG:

	KheSrsPartialSolnAddAvoidUnavailableTimesSignature(ps, mi,
	  (KHE_AVOID_UNAVAILABLE_TIMES_MONITOR) mi->monitor);
	break;

      case KHE_LIMIT_IDLE_TIMES_MONITOR_TAG:

	KheSrsPartialSolnAddLimitIdleTimesSignature(ps, mi,
	  (KHE_LIMIT_IDLE_TIMES_MONITOR) mi->monitor);
	break;

      case KHE_CLUSTER_BUSY_TIMES_MONITOR_TAG:

	KheSrsPartialSolnAddClusterBusyTimesSignature(ps, mi,
	  (KHE_CLUSTER_BUSY_TIMES_MONITOR) mi->monitor);
	break;

      case KHE_LIMIT_BUSY_TIMES_MONITOR_TAG:

	KheSrsPartialSolnAddLimitBusyTimesSignature(ps, mi,
	  (KHE_LIMIT_BUSY_TIMES_MONITOR) mi->monitor);
	break;

      case KHE_LIMIT_WORKLOAD_MONITOR_TAG:

	KheSrsPartialSolnAddLimitWorkloadSignature(ps, mi,
	  (KHE_LIMIT_WORKLOAD_MONITOR) mi->monitor);
	break;

      case KHE_LIMIT_ACTIVE_INTERVALS_MONITOR_TAG:

	KheSrsPartialSolnAddLimitActiveIntervalsSignature(ps, mi,
	  (KHE_LIMIT_ACTIVE_INTERVALS_MONITOR) mi->monitor);
	break;

      default:

	HnAbort("KheSingleResourceSolverAddMonitor internal error (tag %d)",
	  KheMonitorTag(mi->monitor));
    }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_PARTIAL_SOLN KheSrsPartialSolnGet(KHE_SINGLE_RESOURCE_SOLVER srs)*/
/*                                                                           */
/*  Get a partial soln object.  Only the signature field is initialized.     */
/*                                                                           */
/*****************************************************************************/

static KHE_SRS_PARTIAL_SOLN KheSrsPartialSolnGet(KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_PARTIAL_SOLN res;
  if( HaArrayCount(srs->free_partial_solns) > 0 )
  {
    res = HaArrayLastAndDelete(srs->free_partial_solns);
    HaArrayClear(res->signature);
  }
  else
  {
    HaMake(res, srs->arena);
    HaArrayInit(res->signature, srs->arena);
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_PARTIAL_SOLN KheSrsPartialSolnMakeInit(                          */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Make the partial solution that the search grows from.  It does not       */
/*  end on any day.                                                          */
/*                                                                           */
/*****************************************************************************/

static KHE_SRS_PARTIAL_SOLN KheSrsPartialSolnMakeInit(
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_PARTIAL_SOLN res;
  res = KheSrsPartialSolnGet(srs);
  res->prev = NULL;
  res->task = NULL;
  res->best_of = 1;
  res->cost = 0;
  HaArrayAddLast(res->signature, 0);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_PARTIAL_SOLN KheSrsPartialSolnMake(KHE_SRS_PARTIAL_SOLN prev,    */
/*    KHE_SRS_TASK st, int day_index, KHE_SRS_TIME time,                     */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Make a new partial solution with these attributes.                       */
/*                                                                           */
/*****************************************************************************/

static KHE_SRS_PARTIAL_SOLN KheSrsPartialSolnMake(KHE_SRS_PARTIAL_SOLN prev,
  KHE_SRS_TASK st, int day_index, KHE_SRS_TIME time,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_PARTIAL_SOLN res;  int durn, i, visit_num;  KHE_SRS_DAY day;
  KHE_SRS_MONITOR_ON_DAY smd;  KHE_SRS_TIME_GROUP_ON_DAY stgd;

  /* make the basic object */
  res = KheSrsPartialSolnGet(srs);
  res->prev = prev;
  res->task = st->task;  /* in the free case, st will be a null srs task */
  res->best_of = 1;

  /* get a new visit number and pass it to the monitors of time */
  visit_num = srs->visit_num++;
  if( time != NULL )
    HaArrayForEach(time->time_group_on_days, stgd, i)
      KheSrsTimeGroupOnDayVisit(stgd, visit_num);

  /* set the cost and signature */
  res->cost = 0;
  durn = HaArrayFirst(prev->signature);
  if( st->task != NULL )
    durn += (st->last_index - st->first_index + 1);
  HaArrayAddLast(res->signature, durn);
  day = HaArray(srs->days, day_index);
  HaArrayForEach(day->monitor_on_days, smd, i)
    KheSrsMonitorOnDayAddToSignature(smd, res,
      time == NULL ? NULL : time->time, visit_num);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsPartialSolnDelete(KHE_SRS_PARTIAL_SOLN ps,                    */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Delete ps.                                                               */
/*                                                                           */
/*****************************************************************************/

static void KheSrsPartialSolnDelete(KHE_SRS_PARTIAL_SOLN ps,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  HaArrayAddLast(srs->free_partial_solns, ps);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheSrsPartialSolnSignatureHash(KHE_SRS_PARTIAL_SOLN ps)              */
/*                                                                           */
/*  Hash function for hashing the signature of ps.                           */
/*                                                                           */
/*****************************************************************************/

static int KheSrsPartialSolnSignatureHash(KHE_SRS_PARTIAL_SOLN ps)
{
  int res, val, i;
  res = HaArrayCount(ps->signature);
  HaArrayForEach(ps->signature, val, i)
    res = (res < 1) + val;
  return (res < 0 ? - res : res);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheSrsPartialSolnSignatureHashUntyped(void *p)                       */
/*                                                                           */
/*  Untyped version of KheSrsPartialSolnSignatureHash.                       */
/*                                                                           */
/*****************************************************************************/

static int KheSrsPartialSolnSignatureHashUntyped(void *p)
{
  return KheSrsPartialSolnSignatureHash((KHE_SRS_PARTIAL_SOLN) p);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheSrsPartialSolnSignaturesEqual(KHE_SRS_PARTIAL_SOLN ps1,          */
/*    KHE_SRS_PARTIAL_SOLN ps2)                                              */
/*                                                                           */
/*  Return true if the signatures of ps1 and ps2 are equal.  The lengths     */
/*  of their signatures must be equal.                                       */
/*                                                                           */
/*****************************************************************************/

static bool KheSrsPartialSolnSignaturesEqual(KHE_SRS_PARTIAL_SOLN ps1,
  KHE_SRS_PARTIAL_SOLN ps2)
{
  int i;
  HnAssert(HaArrayCount(ps1->signature) == HaArrayCount(ps2->signature),
    "KheSrsPartialSolnSignaturesEqual: signatures have different lengths");
  for( i = 0;  i < HaArrayCount(ps1->signature);  i++ )
    if( HaArray(ps1->signature, i) != HaArray(ps2->signature, i) )
      return false;
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheSrsPartialSolnSignaturesEqualUntyped(void *p1, void *p2)         */
/*                                                                           */
/*  Untyped version of KheSrsPartialSolnSignaturesEqual.                     */
/*                                                                           */
/*****************************************************************************/

static bool KheSrsPartialSolnSignaturesEqualUntyped(void *p1, void *p2)
{
  return KheSrsPartialSolnSignaturesEqual((KHE_SRS_PARTIAL_SOLN) p1,
    (KHE_SRS_PARTIAL_SOLN) p2);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsPartialSolnOverWrite(KHE_SRS_PARTIAL_SOLN dest_ps,            */
/*    KHE_SRS_PARTIAL_SOLN src_ps)                                           */
/*                                                                           */
/*  Overwrite src_ps onto dest_ps.  Their signatures are known to be         */
/*  equal, so those don't need to be touched.  Also the best_of field        */
/*  is not touched here; it gets updated separately.                         */
/*                                                                           */
/*****************************************************************************/

static void KheSrsPartialSolnOverWrite(KHE_SRS_PARTIAL_SOLN dest_ps,
  KHE_SRS_PARTIAL_SOLN src_ps)
{
  dest_ps->prev = src_ps->prev;
  dest_ps->task = src_ps->task;
  dest_ps->cost = src_ps->cost;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheSrsPartialSolnCostLessThan(KHE_SRS_PARTIAL_SOLN ps1,             */
/*    KHE_SRS_PARTIAL_SOLN ps2)                                              */
/*                                                                           */
/*  Return true if ps1 is a better solution than ps2.                        */
/*                                                                           */
/*****************************************************************************/

static bool KheSrsPartialSolnCostLessThan(KHE_SRS_PARTIAL_SOLN ps1,
  KHE_SRS_PARTIAL_SOLN ps2)
{
  return ps1->cost < ps2->cost;
    /* || (ps1->r_cost == ps2->r_cost && ps1->soln_cost < ps2->soln_cost); */
}


/*****************************************************************************/
/*                                                                           */
/*  int KheSrsPartialSolnIncreasingAsstCountCmp(const void *p1, void *p2)    */
/*                                                                           */
/*  Comparison function for sorting an array of partial solutions by         */
/*  increasing assignment count.                                             */
/*                                                                           */
/*****************************************************************************/

static int KheSrsPartialSolnIncreasingAsstCountCmp(const void *p1,
  const void *p2)
{
  KHE_SRS_PARTIAL_SOLN ps1 = * (KHE_SRS_PARTIAL_SOLN *) p1;
  KHE_SRS_PARTIAL_SOLN ps2 = * (KHE_SRS_PARTIAL_SOLN *) p2;
  return HaArrayFirst(ps1->signature) - HaArrayFirst(ps2->signature);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsPartialSolnDebug(KHE_SRS_PARTIAL_SOLN ps, int verbosity,      */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of ps onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void KheSrsPartialSolnDebug(KHE_SRS_PARTIAL_SOLN ps, int verbosity,
  int indent, FILE *fp)
{
  int val, i;
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  fprintf(fp, "[ Partial Soln ");
  if( ps->task != NULL )
    KheTaskDebug(ps->task, 1, -1, fp);
  else
    fprintf(fp, "(free day)");
  fprintf(fp, ": %.5f;", KheCostShow(ps->cost));
  HaArrayForEach(ps->signature, val, i)
    fprintf(fp, "%s%d", i > 0 ? "," : "", val);
  fprintf(fp, " ]");
  if( indent > 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "monitor handling"                                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheSingleResourceSolverAddAvoidClashesMonitor(                      */
/*    KHE_SINGLE_RESOURCE_SOLVER srs, KHE_AVOID_CLASHES_MONITOR m)           */
/*                                                                           */
/*  Add avoid clashes monitor m to srs.                                      */
/*                                                                           */
/*****************************************************************************/

static void KheSingleResourceSolverAddAvoidClashesMonitor(
  KHE_SINGLE_RESOURCE_SOLVER srs, KHE_AVOID_CLASHES_MONITOR m)
{
  KHE_CONSTRAINT c;
  c = KheMonitorConstraint((KHE_MONITOR) m);
  if( KheConstraintCostFunction(c) == KHE_LINEAR_COST_FUNCTION )
  {
    /* no summary required; ignore m */
  }
  else
  {
    /* summary required */
    /* still to do */
    /* KheGroupMonitorAddChildMonitor(srs->resource_gm, (KHE_MONITOR) m); */
    HnAbort("KheSingleResourceSolverAddAvoidClashesMonitor still to do");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSingleResourceSolverAddAvoidUnavailableTimesMonitor(             */
/*    KHE_SINGLE_RESOURCE_SOLVER srs, KHE_AVOID_UNAVAILABLE_TIMES_MONITOR m) */
/*                                                                           */
/*  Add avoid unavailable times monitor m to srs.                            */
/*                                                                           */
/*****************************************************************************/

static void KheSingleResourceSolverAddAvoidUnavailableTimesMonitor(
  KHE_SINGLE_RESOURCE_SOLVER srs, KHE_AVOID_UNAVAILABLE_TIMES_MONITOR m)
{
  KHE_CONSTRAINT c;
  c = KheMonitorConstraint((KHE_MONITOR) m);
  if( KheConstraintCostFunction(c) == KHE_LINEAR_COST_FUNCTION )
  {
    /* no summary required; ignore m */
  }
  else
  {
    /* summary required */
    /* still to do */
    /* KheGroupMonitorAddChildMonitor(srs->resource_gm, (KHE_MONITOR) m); */
    HnAbort("KheSingleResourceSolverAddAvoidUnavailableTimesMonitor "
      "still to do");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSingleResourceSolverAddLimitIdleTimesMonitor(                    */
/*    KHE_SINGLE_RESOURCE_SOLVER srs, KHE_LIMIT_IDLE_TIMES_MONITOR m)        */
/*                                                                           */
/*  Add limit idle times monitor m to srs.                                   */
/*                                                                           */
/*****************************************************************************/

static void KheSingleResourceSolverAddLimitIdleTimesMonitor(
  KHE_SINGLE_RESOURCE_SOLVER srs, KHE_LIMIT_IDLE_TIMES_MONITOR m)
{
  /* not done, and that fact is documented */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSingleResourceSolverAddClusterBusyTimesMonitor(                  */
/*    KHE_SINGLE_RESOURCE_SOLVER srs, KHE_CLUSTER_BUSY_TIMES_MONITOR m)      */
/*                                                                           */
/*  Add cluster busy times monitor m to srs.                                 */
/*                                                                           */
/*****************************************************************************/

static void KheSingleResourceSolverAddClusterBusyTimesMonitor(
  KHE_SINGLE_RESOURCE_SOLVER srs, KHE_CLUSTER_BUSY_TIMES_MONITOR m)
{
  KHE_TIME first_time, last_time, t;  KHE_TIME_GROUP tg;  KHE_POLARITY po;
  int first_index, last_index, i, j, k, junk;  KHE_SRS_DAY day;
  KHE_SRS_MONITOR sm;  KHE_SRS_MONITOR_ON_DAY smd, prev_smd;  KHE_SRS_TIME st;
  KHE_SRS_TIME_GROUP_ON_DAY stgd, prev_stgd;  KHE_SRS_TIME_GROUP stg;
  KHE_SRS_MEASURE measure;  /* KHE_SRS_ACTION action; */

  /* make an srs monitor object for m */
  sm = KheSrsMonitorMake((KHE_MONITOR) m, srs);
  HaArrayAddLast(srs->monitors, sm);

  /* find the interval of days that m covers */
  KheClusterBusyTimesMonitorRange(m, &first_time, &last_time);
  first_index = KheFrameTimeIndex(srs->days_frame, first_time);
  last_index = KheFrameTimeIndex(srs->days_frame, last_time);

  /* add a monitor on day object to each of those days */
  prev_smd = NULL;
  for( i = first_index;  i <= last_index;  i++ )
  {
    day = HaArray(srs->days, i);
    /* action = KheSrsAction(i, first_index, last_index); */
    smd = KheSrsMonitorOnDayMake(sm, /* action, */ prev_smd, NULL,
      i == last_index ? -1 : day->sig_len++, srs);
    if( prev_smd != NULL )
      prev_smd->next = smd;
    HaArrayAddLast(day->monitor_on_days, smd);
    prev_smd = smd;
  }

  /* add time group and time group on day objects as required */
  for( i = 0;  i < KheClusterBusyTimesMonitorTimeGroupCount(m);  i++ )
  {
    tg = KheClusterBusyTimesMonitorTimeGroup(m, i, &po);
    if( KheTimeGroupTimeCount(tg) > 0 )
    {
      /* make a time group and add it to the monitor */
      measure = (po == KHE_POSITIVE ? KHE_SRS_MEASURE_ACTIVE_BUSY :
	KHE_SRS_MEASURE_ACTIVE_FREE);
      stg = KheSrsTimeGroupMake(sm, tg, measure, srs);
      HaArrayAddLast(sm->time_groups, stg);

      /* make time group on day objects and add them in */
      first_time = KheTimeGroupTime(tg, 0);
      last_time = KheTimeGroupTime(tg, KheTimeGroupTimeCount(tg) - 1);
      first_index = KheFrameTimeIndex(srs->days_frame, first_time);
      last_index = KheFrameTimeIndex(srs->days_frame, last_time);
      prev_stgd = NULL;
      for( j = first_index;  j <= last_index;  j++ )
      {
	day = HaArray(srs->days, j);
	/* action = KheSrsAction(j, first_index, last_index); */
        stgd = KheSrsTimeGroupOnDayMake(stg, prev_stgd,
	  j == last_index ? -1 : day->sig_len++, srs);
	HaArrayAddLast(stg->time_group_on_days, stgd);
	smd = HaArrayLast(day->monitor_on_days);
	HaArrayAddLast(smd->time_group_on_days, stgd);
	for( k = 0;  k < KheTimeGroupTimeCount(tg);  k++ )
	{
	  t = KheTimeGroupTime(tg, k);
	  if( KheTimeGroupContains(day->time_group, t, &junk) )
	  {
	    st = HaArray(srs->times, KheTimeIndex(t));
	    HaArrayAddLast(st->time_group_on_days, stgd);
	  }
	}
	prev_stgd = stgd;
      }
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSingleResourceSolverAddLimitBusyTimesMonitor(                    */
/*    KHE_SINGLE_RESOURCE_SOLVER srs, KHE_LIMIT_BUSY_TIMES_MONITOR m)        */
/*                                                                           */
/*  Add limit busy times monitor m to srs.                                   */
/*                                                                           */
/*****************************************************************************/

static void KheSingleResourceSolverAddLimitBusyTimesMonitor(
  KHE_SINGLE_RESOURCE_SOLVER srs, KHE_LIMIT_BUSY_TIMES_MONITOR m)
{
  /* *** still to do
  KHE_TIME first_time, last_time;  int first_index, last_index, i, j, junk;
  KHE_SRS_DAY day;  KHE_MONITOR_INFO mi;  KHE_TIME_GROUP tg;

  ** find the interval of days that m covers **
  KheLimitBusyTimesMonitorRange(m, &first_time, &last_time);
  first_index = KheFrameTimeIndex(srs->days_frame, first_time);
  last_index = KheFrameTimeIndex(srs->days_frame, last_time);

  ** add a monitor info object to each of those days except the last **
  for( i = first_index;  i < last_index;  i++ )
  {
    day = HaArray(srs->days, i);
    mi = KheMonitorInfoMake((KHE_MONITOR) m, srs);
    HaArrayAddLast(day->monitor_info, mi);
  }

  ** add indexes to the monitor info objects as required **
  for( i = 0;  i < KheLimitBusyTimesMonitorTimeGroupCount(m);  i++ )
  {
    tg = KheLimitBusyTimesMonitorTimeGroup(m, i, &junk);
    if( KheTimeGroupTimeCount(tg) > 0 )
    {
      first_time = KheTimeGroupTime(tg, 0);
      last_time = KheTimeGroupTime(tg, KheTimeGroupTimeCount(tg) - 1);
      first_index = KheFrameTimeIndex(srs->days_frame, first_time);
      last_index = KheFrameTimeIndex(srs->days_frame, last_time);
      for( j = first_index;  j < last_index;  j++ )
      {
	** add i to the monitor info object at this index **
	day = HaArray(srs->days, j);
	mi = HaArrayLast(day->monitor_info);
	HaArrayAddLast(mi->indexes, i);
      }
    }
  }

  ** add m to srs_resource_gm **
  ** KheGroupMonitorAddChildMonitor(srs->resource_gm, (KHE_MONITOR) m); **
  *** */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSingleResourceSolverAddLimitWorkloadMonitor(                     */
/*    KHE_SINGLE_RESOURCE_SOLVER srs, KHE_LIMIT_WORKLOAD_MONITOR m)          */
/*                                                                           */
/*  Add limit workload monitor m to srs.                                     */
/*                                                                           */
/*****************************************************************************/

static void KheSingleResourceSolverAddLimitWorkloadMonitor(
  KHE_SINGLE_RESOURCE_SOLVER srs, KHE_LIMIT_WORKLOAD_MONITOR m)
{
  /* *** still to do
  KHE_TIME first_time, last_time;  int first_index, last_index, i, j;
  KHE_SRS_DAY day;  KHE_MONITOR_INFO mi;  KHE_TIME_GROUP tg;  float junk;

  ** find the interval of days that m covers **
  KheLimitWorkloadMonitorRange(m, &first_time, &last_time);
  first_index = KheFrameTimeIndex(srs->days_frame, first_time);
  last_index = KheFrameTimeIndex(srs->days_frame, last_time);

  ** add a monitor info object to each of those days except the last **
  for( i = first_index;  i < last_index;  i++ )
  {
    day = HaArray(srs->days, i);
    mi = KheMonitorInfoMake((KHE_MONITOR) m, srs);
    HaArrayAddLast(day->monitor_info, mi);
  }

  ** add indexes to the monitor info objects as required **
  for( i = 0;  i < KheLimitWorkloadMonitorTimeGroupCount(m);  i++ )
  {
    tg = KheLimitWorkloadMonitorTimeGroup(m, i, &junk);
    if( KheTimeGroupTimeCount(tg) > 0 )
    {
      first_time = KheTimeGroupTime(tg, 0);
      last_time = KheTimeGroupTime(tg, KheTimeGroupTimeCount(tg) - 1);
      first_index = KheFrameTimeIndex(srs->days_frame, first_time);
      last_index = KheFrameTimeIndex(srs->days_frame, last_time);
      for( j = first_index;  j < last_index;  j++ )
      {
	** add i to the monitor info object at this index **
	day = HaArray(srs->days, j);
	mi = HaArrayLast(day->monitor_info);
	HaArrayAddLast(mi->indexes, i);
      }
    }
  }

  ** add m to srs_resource_gm **
  ** KheGroupMonitorAddChildMonitor(srs->resource_gm, (KHE_MONITOR) m); **
  *** */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSingleResourceSolverAddLimitActiveIntervalsMonitor(              */
/*    KHE_SINGLE_RESOURCE_SOLVER srs, KHE_LIMIT_ACTIVE_INTERVALS_MONITOR m)  */
/*                                                                           */
/*  Add limit active intervals monitor m to srs.                             */
/*                                                                           */
/*****************************************************************************/

static void KheSingleResourceSolverAddLimitActiveIntervalsMonitor(
  KHE_SINGLE_RESOURCE_SOLVER srs, KHE_LIMIT_ACTIVE_INTERVALS_MONITOR m)
{
  /* *** still to do
  KHE_TIME first_time, last_time;  int first_index, last_index, i, j;
  KHE_SRS_DAY day;  KHE_MONITOR_INFO mi;  KHE_TIME_GROUP tg;  KHE_POLARITY po;

  ** find the interval of days that m covers **
  KheLimitActiveIntervalsMonitorRange(m, &first_time, &last_time);
  first_index = KheFrameTimeIndex(srs->days_frame, first_time);
  last_index = KheFrameTimeIndex(srs->days_frame, last_time);

  ** add a monitor info object to each of those days except the last **
  for( i = first_index;  i < last_index;  i++ )
  {
    day = HaArray(srs->days, i);
    mi = KheMonitorInfoMake((KHE_MONITOR) m, srs);
    HaArrayAddLast(day->monitor_info, mi);
  }

  ** add indexes to the monitor info objects as required **
  for( i = 0;  i < KheLimitActiveIntervalsMonitorTimeGroupCount(m);  i++ )
  {
    tg = KheLimitActiveIntervalsMonitorTimeGroup(m, i, &po);
    if( KheTimeGroupTimeCount(tg) > 0 )
    {
      first_time = KheTimeGroupTime(tg, 0);
      last_time = KheTimeGroupTime(tg, KheTimeGroupTimeCount(tg) - 1);
      first_index = KheFrameTimeIndex(srs->days_frame, first_time);
      last_index = KheFrameTimeIndex(srs->days_frame, last_time);
      for( j = first_index;  j < last_index;  j++ )
      {
	** add i to the monitor info object at this index **
	day = HaArray(srs->days, j);
	mi = HaArrayLast(day->monitor_info);
	HaArrayAddLast(mi->indexes, i);
      }
    }
  }

  ** add m to srs_resource_gm **
  ** KheGroupMonitorAddChildMonitor(srs->resource_gm, (KHE_MONITOR) m); **
  *** */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSingleResourceSolverAddMonitor(KHE_SINGLE_RESOURCE_SOLVER srs,   */
/*    KHE_MONITOR m)                                                         */
/*                                                                           */
/*  Add m to srs.                                                            */
/*                                                                           */
/*****************************************************************************/

static void KheSingleResourceSolverAddMonitor(KHE_SINGLE_RESOURCE_SOLVER srs,
  KHE_MONITOR m)
{
  switch( KheMonitorTag(m) )
  {
    case KHE_AVOID_CLASHES_MONITOR_TAG:

      KheSingleResourceSolverAddAvoidClashesMonitor(srs,
        (KHE_AVOID_CLASHES_MONITOR) m);
      break;

    case KHE_AVOID_UNAVAILABLE_TIMES_MONITOR_TAG:

      KheSingleResourceSolverAddAvoidUnavailableTimesMonitor(srs,
        (KHE_AVOID_UNAVAILABLE_TIMES_MONITOR) m);
      break;

    case KHE_LIMIT_IDLE_TIMES_MONITOR_TAG:

      KheSingleResourceSolverAddLimitIdleTimesMonitor(srs,
        (KHE_LIMIT_IDLE_TIMES_MONITOR) m);
      break;

    case KHE_CLUSTER_BUSY_TIMES_MONITOR_TAG:

      KheSingleResourceSolverAddClusterBusyTimesMonitor(srs,
        (KHE_CLUSTER_BUSY_TIMES_MONITOR) m);
      break;

    case KHE_LIMIT_BUSY_TIMES_MONITOR_TAG:

      KheSingleResourceSolverAddLimitBusyTimesMonitor(srs,
        (KHE_LIMIT_BUSY_TIMES_MONITOR) m);
      break;

    case KHE_LIMIT_WORKLOAD_MONITOR_TAG:

      KheSingleResourceSolverAddLimitWorkloadMonitor(srs,
        (KHE_LIMIT_WORKLOAD_MONITOR) m);
      break;

    case KHE_LIMIT_ACTIVE_INTERVALS_MONITOR_TAG:

      KheSingleResourceSolverAddLimitActiveIntervalsMonitor(srs,
        (KHE_LIMIT_ACTIVE_INTERVALS_MONITOR) m);
      break;

    default:

      HnAbort("KheSingleResourceSolverAddMonitor internal error (tag %d)",
        KheMonitorTag(m));
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "public functions"                                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_SINGLE_RESOURCE_SOLVER KheSingleResourceSolverMake(KHE_SOLN soln,    */
/*    KHE_OPTIONS options)                                                   */
/*                                                                           */
/*  Make and return a new single resource solver with these attributes.      */
/*                                                                           */
/*****************************************************************************/

KHE_SINGLE_RESOURCE_SOLVER KheSingleResourceSolverMake(KHE_SOLN soln,
  KHE_OPTIONS options)
{
  KHE_SINGLE_RESOURCE_SOLVER res;  HA_ARENA a;  KHE_TIME_GROUP tg;  int i;
  KHE_FRAME days_frame;  KHE_EVENT_TIMETABLE_MONITOR etm;  KHE_INSTANCE ins;
  KHE_TIME t;

  /* get the common frame and event timetable monitor; return NULL if can't */
  days_frame = KheOptionsFrame(options, "gs_common_frame", soln);
  etm = (KHE_EVENT_TIMETABLE_MONITOR)
    KheOptionsGetObject(options, "gs_event_timetable_monitor", NULL);
  if( days_frame == NULL || etm == NULL )
    return NULL;

  /* make the basic object */
  a = KheSolnArenaBegin(soln, false);
  HaMake(res, a);
  res->arena = a;
  res->soln = soln;
  res->options = options;
  res->diversifier = 0;
  res->visit_num = 0;
  res->days_frame = days_frame;
  res->etm = etm;
  res->task_finder = KheTaskFinderMake(soln, options, a);
  res->resource = NULL;
  HaArrayInit(res->monitors, a);
  HaArrayInit(res->days, a);
  HaArrayInit(res->times, a);
  HaArrayInit(res->final_solns, a);
  /* HaArrayInit(res->all_partial_solns, a); */

  HaArrayInit(res->free_time_groups, a);
  HaArrayInit(res->free_time_group_on_days, a);
  HaArrayInit(res->free_monitors, a);
  HaArrayInit(res->free_monitor_on_days, a);
  HaArrayInit(res->free_tasks, a);
  HaArrayInit(res->free_tasks_at_time, a);
  HaArrayInit(res->free_partial_solns, a);

  /* add one day object for each time group of the common frame */
  for( i = 0;  i < KheFrameTimeGroupCount(res->days_frame);  i++ )
  {
    tg = KheFrameTimeGroup(res->days_frame, i);
    HaArrayAddLast(res->days, KheSrsDayMake(tg, i, res));
  }

  /* add one time object for each time of the instance */
  ins = KheSolnInstance(soln);
  for( i = 0;  i < KheInstanceTimeCount(ins);  i++ )
  {
    t = KheInstanceTime(ins, i);
    HaArrayAddLast(res->times, KheSrsTimeMake(t, res));
  }

  /* all done */
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSingleResourceSolverDelete(KHE_SINGLE_RESOURCE_SOLVER srs)       */
/*                                                                           */
/*  Delete srs, reclaiming its memory.                                       */
/*                                                                           */
/*****************************************************************************/

void KheSingleResourceSolverDelete(KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KheSolnArenaEnd(srs->soln, srs->arena);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsDayAddPartialSoln(KHE_SRS_DAY day, KHE_SRS_PARTIAL_SOLN ps,   */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Add ps to day.                                                           */
/*                                                                           */
/*  If day does not already contain a partial solution with the same         */
/*  signature as ps, just add ps to day and return.                          */
/*                                                                           */
/*  If day already contains a partial solution with the same signature       */
/*  as ps, called other_ps below, keep whichever one of ps and other_ps      */
/*  has the smallest cost, according to KheSrsPartialSolnCostLessThan, by    */
/*  overwriting other_ps's fields with ps's fields, if required.             */
/*                                                                           */
/*  If ps and other_ps have equal cost, keep one chosen at random, with      */
/*  weights which ensure that if there are K partial solutions with the      */
/*  same signature and the same cost, then each has probability 1/K of       */
/*  being the one kept.                                                      */
/*                                                                           */
/*****************************************************************************/

static void KheSrsDayAddPartialSoln(KHE_SRS_DAY day, KHE_SRS_PARTIAL_SOLN ps,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_PARTIAL_SOLN other_ps;
  if( !HpTableAddUnique(day->partial_solns, (void *) ps, ps, other_ps) )
  {
    /* other_ps will now be the best of one more partial solution than before */
    other_ps->best_of++;

    /* ensure that other_ps is the better of other_ps and ps, by overwriting */
    if( KheSrsPartialSolnCostLessThan(ps, other_ps) ||
	(!KheSrsPartialSolnCostLessThan(other_ps, ps) &&
	 srs->diversifier % other_ps->best_of == 0) )
    {
      /* ps is better than other_ps, so overwrite other_ps with ps */
      KheSrsPartialSolnOverWrite(other_ps, ps);
    }

    /* free ps */
    KheSrsPartialSolnDelete(ps, srs);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void DoExtend(KHE_SINGLE_RESOURCE_SOLVER srs,                            */
/*    KHE_SRS_PARTIAL_SOLN ps, KHE_SRS_TASK st)                              */
/*                                                                           */
/*  Extend ps by adding st to it.  Add the resulting new partial solution    */
/*  to the day object representing st's last day.                            */
/*                                                                           */
/*****************************************************************************/

static void DoExtend(KHE_SINGLE_RESOURCE_SOLVER srs,
  KHE_SRS_PARTIAL_SOLN ps, KHE_SRS_TASK st)
{
  KHE_SRS_PARTIAL_SOLN curr_ps, last_ps;  int day_index;  KHE_SRS_DAY last_day;
  KHE_TIME time;  KHE_SRS_TIME stime;

  /* make one partial solution for each day of st, ending at last_ps */
  last_ps = ps;
  for( day_index = st->first_index;  day_index <= st->last_index;  day_index++ )
  {
    time = HaArray(st->times_by_day, day_index - st->first_index);
    stime = HaArray(srs->times, KheTimeIndex(time));
    last_ps = KheSrsPartialSolnMake(last_ps, st, day_index, stime, srs);
  }
  HnAssert(last_ps != ps, "DoExtend internal error");

  /* delete all partial solutions between ps and last_ps exclusive */
  /* NB field curr_ps->prev is usable after calling KhePartialSolnDelete */
  for( curr_ps = last_ps->prev;  curr_ps != ps;  curr_ps = curr_ps->prev )
    KheSrsPartialSolnDelete(curr_ps, srs);
  last_ps->prev = ps;

  /* add last_ps to last_day, and increment the diversifier */
  last_day = HaArray(srs->days, st->last_index);
  KheSrsDayAddPartialSoln(last_day, last_ps, srs);
  srs->diversifier++;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSingleResourceSolverExtend(KHE_SINGLE_RESOURCE_SOLVER srs,       */
/*    KHE_SRS_PARTIAL_SOLN ps, KHE_SRS_DAY day)                              */
/*                                                                           */
/*  Extend the dynamic programming search by searching out of ps.  Each      */
/*  new task starts on day.                                                  */
/*                                                                           */
/*****************************************************************************/

static void KheSingleResourceSolverExtend(KHE_SINGLE_RESOURCE_SOLVER srs,
  KHE_SRS_PARTIAL_SOLN ps, KHE_SRS_DAY day)
{
  int i, j;  KHE_SRS_TASK st;  KHE_SRS_TASKS_AT_TIME stt;

  /* extend for each task beginning on day */
  HaArrayForEach(day->tasks_at_time, stt, i)
    HaArrayForEach(stt->tasks, st, j)
      DoExtend(srs, ps, st);

  /* extend by taking day off */
  DoExtend(srs, ps, day->null_task);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSingleResourceSolverSolve(KHE_SINGLE_RESOURCE_SOLVER srs,        */
/*    KHE_RESOURCE r)                                                        */
/*                                                                           */
/*  Solve for r.                                                             */
/*                                                                           */
/*****************************************************************************/

void KheSingleResourceSolverSolve(KHE_SINGLE_RESOURCE_SOLVER srs,
  KHE_RESOURCE r)
{
  int di, i, pos;  KHE_MONITOR m;  KHE_SRS_DAY day, prev_day;
  KHE_SRS_PARTIAL_SOLN ps;  KHE_SRS_MONITOR sm;

  /* if there was already a resource, clear out old stuff */
  if( srs->resource != NULL )
  {
    HaArrayForEach(srs->days, day, di)
      KheSrsDayClear(day, srs);
    HaArrayForEach(srs->monitors, sm, i)
      KheSrsMonitorDelete(sm, srs);
    HaArrayClear(srs->monitors);
    HaArrayClear(srs->final_solns);
  }

  /* add the resource and its monitors */
  srs->resource = r;
  for( i = 0;  i < KheSolnResourceMonitorCount(srs->soln, r);  i++ )
  {
    m = KheSolnResourceMonitor(srs->soln, r, i);
    KheSingleResourceSolverAddMonitor(srs, m);
  }

  /* add suitable tasks for each day */
  HaArrayForEach(srs->days, day, di)
    KheSrsDayAddTasks(day, srs);

  /* build up the partial solutions for each day */
  prev_day = NULL;
  HaArrayForEach(srs->days, day, di)
  {
    if( prev_day == NULL )
    {
      /* virtual prev_day contains a single partial solution (an empty one) */
      ps = KheSrsPartialSolnMakeInit(srs);
      KheSingleResourceSolverExtend(srs, ps, day);
    }
    else
    {
      /* extend each partial solution from prev_day with tasks from day */
      HpTableForEachValue(prev_day->partial_solns, ps, pos)
	KheSingleResourceSolverExtend(srs, ps, day);
    }
    prev_day = day;
  }

  /* extract final solutions */
  day = HaArrayLast(srs->days);
  HpTableForEachValue(prev_day->partial_solns, ps, pos)
    HaArrayAddLast(srs->final_solns, ps);
  HaArraySort(srs->final_solns, &KheSrsPartialSolnIncreasingAsstCountCmp);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheSingleResourceSolverTimetableCount(KHE_SINGLE_RESOURCE_SOLVER srs)*/
/*                                                                           */
/*  Return the number of timetables found.                                   */
/*                                                                           */
/*****************************************************************************/

int KheSingleResourceSolverTimetableCount(KHE_SINGLE_RESOURCE_SOLVER srs)
{
  HnAssert(srs->resource != NULL,
    "KheSingleResourceSolverTimetableCount called out of order");
  return HaArrayCount(srs->final_solns);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSingleResourceSolverTimetable(KHE_SINGLE_RESOURCE_SOLVER srs,    */
/*    int i, int *asst_count, KHE_COST *r_cost, KHE_COST *soln_cost)         */
/*                                                                           */
/*  Report on the i'th timetable found by the solver.                        */
/*                                                                           */
/*****************************************************************************/

void KheSingleResourceSolverTimetable(KHE_SINGLE_RESOURCE_SOLVER srs,
  int i, int *asst_count, KHE_COST *cost /* , KHE_COST *soln_cost */)
{
  /* make sure there is an i'th solution */
  KHE_SRS_PARTIAL_SOLN ps;
  HnAssert(srs->resource != NULL,
    "KheSingleResourceSolverTimetable called out of order");
  HnAssert(0 <= i && i < HaArrayCount(srs->final_solns),
    "KheSingleResourceSolverTimetable:  i (%d) out of range (0 .. %d)",
    i, HaArrayCount(srs->final_solns) - 1);

  /* return the relevant fields of the i'th solution */
  ps = HaArray(srs->final_solns, i);
  *asst_count = HaArrayFirst(ps->signature);
  *cost = ps->cost;
  /* *soln_cost = ps->soln_cost; */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSingleResourceSolverAdopt(KHE_SINGLE_RESOURCE_SOLVER srs, int i) */
/*                                                                           */
/*  Change srs's solution to include the i'th timetable.                     */
/*                                                                           */
/*****************************************************************************/

void KheSingleResourceSolverAdopt(KHE_SINGLE_RESOURCE_SOLVER srs, int i)
{
  /* make sure there is an i'th solution */
  KHE_SRS_PARTIAL_SOLN ps;
  HnAssert(srs->resource != NULL,
    "KheSingleResourceSolverAdopt called out of order");
  HnAssert(0 <= i && i < HaArrayCount(srs->final_solns),
    "KheSingleResourceSolverAdopt:  i (%d) out of range (0 .. %d)",
    i, HaArrayCount(srs->final_solns) - 1);

  /* assign the tasks of the i'th solution */
  for( ps = HaArray(srs->final_solns, i);  ps != NULL;  ps = ps->prev )
    if( ps->task != NULL && KheTaskAsstResource(ps->task) == NULL )
    {
      /* assign srs->resource to ps->task */
      if( !KheTaskAssignResource(ps->task, srs->resource) )
	HnAbort("KheSingleResourceSolverAdopt internal error");
    }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSingleResourceSolverDebug(KHE_SINGLE_RESOURCE_SOLVER srs,        */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug printf of srs onto fp with the given verbosity and indent.         */
/*                                                                           */
/*****************************************************************************/

void KheSingleResourceSolverDebug(KHE_SINGLE_RESOURCE_SOLVER srs,
  int verbosity, int indent, FILE *fp)
{
  KHE_SRS_DAY day;  int i;
  HnAssert(indent >= 0, "KheSingleResourceSolverDebug: indent < 0");
  fprintf(fp, "%*s[ SingleResourceSolver(%s)\n", indent, "",
    srs->resource == NULL ? "@" : KheResourceId(srs->resource));
  HaArrayForEach(srs->days, day, i)
    KheSrsDayDebug(day, verbosity, indent + 2, fp);
  fprintf(fp, "%*s]\n", indent, "");
}
