
/*****************************************************************************/
/*                                                                           */
/*  THE KHE HIGH SCHOOL TIMETABLING ENGINE                                   */
/*  COPYRIGHT (C) 2010 Jeffrey H. Kingston                                   */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 3, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         khe_sr_task_finder.c                                       */
/*  DESCRIPTION:  Task finder                                                */
/*                                                                           */
/*****************************************************************************/
#include "khe_solvers.h"
#include "limits.h"

#define DEBUG1 0
#define DEBUG2 0


/*****************************************************************************/
/*                                                                           */
/*  KHE_INTERVAL - an interval (private)                                     */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_interval_rec {
  int first;
  int last;
} KHE_INTERVAL;


/*****************************************************************************/
/*                                                                           */
/*  PART_FROM - what we need to known about what from_r is doing             */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_part_from_rec {
  KHE_TASK_SET			ts;
  KHE_INTERVAL			in;
} PART_FROM;


/*****************************************************************************/
/*                                                                           */
/*  PART_TO - what we need to known about what to_r is doing                 */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_part_to_rec {
  KHE_TASK_SET			ts;
  int				durn;
  bool				effectively_free;
} PART_TO;


/*****************************************************************************/
/*                                                                           */
/*  PART                                                                     */
/*                                                                           */
/*****************************************************************************/



/*****************************************************************************/
/*                                                                           */
/*  KHE_CORE - the core of a widened task set                                */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_core_rec {
  KHE_TASK_SET			from_r_ts;
  KHE_INTERVAL			from_r_in;
  KHE_TASK_SET			to_r_ts;
  int				to_r_ts_durn;
  bool				to_r_ts_effectively_free;
} KHE_CORE;


/*****************************************************************************/
/*                                                                           */
/*  KHE_WING - one element of the wing of a widened task set                 */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_wing_rec {
  KHE_TASK			from_r_task;
  KHE_INTERVAL			from_r_in;
  KHE_TASK			to_r_task;
  int				to_r_task_durn;
  bool				to_r_task_effectively_free;
} *KHE_WING;

typedef HA_ARRAY(KHE_WING) ARRAY_KHE_WING;


/*****************************************************************************/
/*                                                                           */
/*  KHE_WIDENED_TASK_SET - a widened task set, with a core and wings         */
/*                                                                           */
/*****************************************************************************/

struct khe_widened_task_set_rec {
  KHE_TASK_FINDER			tf;
  KHE_TASK_SET				scratch_ts;
  KHE_RESOURCE				from_r;
  KHE_RESOURCE_TIMETABLE_MONITOR	from_r_rtm;
  KHE_RESOURCE				to_r;
  KHE_RESOURCE_TIMETABLE_MONITOR	to_r_rtm;
  KHE_CORE				core;
  ARRAY_KHE_WING			left_wings;
  ARRAY_KHE_WING			right_wings;
  int					to_r_max_left_wing_count;
  int					to_r_max_right_wing_count;
};

typedef HA_ARRAY(KHE_WIDENED_TASK_SET) ARRAY_KHE_WIDENED_TASK_SET;


/*****************************************************************************/
/*                                                                           */
/*  KHE_TASK_FINDER                                                          */
/*                                                                           */
/*****************************************************************************/

struct khe_task_finder_rec {
  HA_ARENA				arena;
  KHE_SOLN				soln;
  KHE_FRAME				days_frame;
  KHE_EVENT_TIMETABLE_MONITOR		etm;
  ARRAY_KHE_WIDENED_TASK_SET		free_widened_task_sets;
  ARRAY_KHE_WING			free_wings;
};


/*****************************************************************************/
/*                                                                           */
/*  Submodule "intervals"                                                    */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_INTERVAL KheInterval(int first, int last)                            */
/*                                                                           */
/*  Return a new interval with these attributes.                             */
/*                                                                           */
/*****************************************************************************/

KHE_INTERVAL KheInterval(int first, int last)
{
  KHE_INTERVAL res;
  res.first = first;
  res.last = last;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  int KheIntervalLength(KHE_INTERVAL in)                                   */
/*                                                                           */
/*  Return the length of in.                                                 */
/*                                                                           */
/*****************************************************************************/

int KheIntervalLength(KHE_INTERVAL in)
{
  return in.last - in.first + 1;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheIntervalEmpty(KHE_INTERVAL in)                                   */
/*                                                                           */
/*  Return true if in is empty.                                              */
/*                                                                           */
/*****************************************************************************/

bool KheIntervalEmpty(KHE_INTERVAL in)
{
  return in.first > in.last;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheIntervalSubset(KHE_INTERVAL in1, KHE_INTERVAL in2)               */
/*                                                                           */
/*  Return true if in1 is a subset of in2.                                   */
/*                                                                           */
/*****************************************************************************/

bool KheIntervalSubset(KHE_INTERVAL in1, KHE_INTERVAL in2)
{
  return KheIntervalEmpty(in1) ||
    (in2.first <= in1.first && in1.last <= in2.last);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheIntervalEqual(KHE_INTERVAL in1, KHE_INTERVAL in2)                */
/*                                                                           */
/*  Return true if in1 and in2 are equal.                                    */
/*                                                                           */
/*****************************************************************************/

bool KheIntervalEqual(KHE_INTERVAL in1, KHE_INTERVAL in2)
{
  if( KheIntervalEmpty(in1) )
    return KheIntervalEmpty(in2);
  else
    return in1.first == in2.first && in1.last == in2.last;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheIntervalDisjoint(KHE_INTERVAL in1, KHE_INTERVAL in2)             */
/*                                                                           */
/*  Return true if in1 and in2 are disjoint.                                 */
/*                                                                           */
/*****************************************************************************/

bool KheIntervalDisjoint(KHE_INTERVAL in1, KHE_INTERVAL in2)
{
  return KheIntervalEmpty(in1) || KheIntervalEmpty(in2) ||
    in1.last < in2.first || in2.last < in1.first;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheIntervalUnion(KHE_INTERVAL *in1, KHE_INTERVAL in2)               */
/*                                                                           */
/*  Overwrite *in1 with its union with in2.                                  */
/*                                                                           */
/*****************************************************************************/

void KheIntervalUnion(KHE_INTERVAL *in1, KHE_INTERVAL in2)
{
  if( KheIntervalEmpty(*in1) )
    *in1 = in2;
  else if( !KheIntervalEmpty(in2) )
  {
    if( in2.first < in1->first ) in1->first = in2.first;
    if( in2.last > in1->last ) in1->last = in2.last;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheTaskFinderIntervalShow(KHE_TASK_FINDER tf, KHE_INTERVAL in)     */
/*                                                                           */
/*  Show in in static memory.                                                */
/*                                                                           */
/*****************************************************************************/

/* *** currently unusued, which seems odd
char *KheTaskFinderIntervalShow(KHE_TASK_FINDER tf, KHE_INTERVAL in)
{
  static char buff[500];
  if( KheIntervalEmpty(in) )
    sprintf(buff, "-");
  else
    sprintf(buff, "%s-%s",
      KheTimeGroupId(KheFrameTimeGroup(tf->days_frame, in.first)),
      KheTimeGroupId(KheFrameTimeGroup(tf->days_frame, in.last)));
  return buff;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheTaskDoInterval(KHE_TASK task, KHE_FRAME frame, KHE_INTERVAL *in) */
/*                                                                           */
/*  Do that part of the work of KheTaskSetInterval below that pertains to    */
/*  task and its descendants.                                                */
/*                                                                           */
/*****************************************************************************/

static void KheTaskDoInterval(KHE_TASK task, KHE_FRAME frame, KHE_INTERVAL *in)
{
  KHE_MEET meet;  KHE_TIME t;  KHE_TASK child_task;  int i, durn, fi, li;

  /* do it for task itself */
  meet = KheTaskMeet(task);
  if( meet != NULL )
  {
    t = KheMeetAsstTime(meet);
    if( t != NULL )
    {
      durn = KheMeetDuration(meet);
      if( durn == 1 )
	fi = li = KheFrameTimeIndex(frame, t);
      else
      {
	fi = KheFrameTimeIndex(frame, t);
	li = KheFrameTimeIndex(frame, KheTimeNeighbour(t, durn - 1));
      }
      if( fi < in->first )
	in->first = fi;
      if( li > in->last )
	in->last = li;
    }
  }

  /* do it for the children of task */
  for( i = 0;  i < KheTaskAssignedToCount(task);  i++ )
  {
    child_task = KheTaskAssignedTo(task, i);
    KheTaskDoInterval(child_task, frame, in);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_INTERVAL KheTaskInterval(KHE_TASK_FINDER tf, KHE_TASK task)          */
/*                                                                           */
/*  Return the interval covered by task.                                     */
/*                                                                           */
/*****************************************************************************/

static KHE_INTERVAL KheTaskInterval(KHE_TASK_FINDER tf, KHE_TASK task)
{
  KHE_INTERVAL res;
  res.first = KheFrameTimeGroupCount(tf->days_frame) - 1;
  res.last = 0;
  KheTaskDoInterval(task, tf->days_frame, &res);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_INTERVAL KheTaskSetInterval(KHE_TASK_FINDER tf,  KHE_TASK_SET ts)    */
/*                                                                           */
/*  Return the interval covered by ts.                                       */
/*                                                                           */
/*****************************************************************************/

static KHE_INTERVAL KheTaskSetInterval(KHE_TASK_FINDER tf,  KHE_TASK_SET ts)
{
  int i;  KHE_TASK task;  KHE_INTERVAL res;
  res.first = KheFrameTimeGroupCount(tf->days_frame) - 1;
  res.last = 0;
  for( i = 0;  i < KheTaskSetTaskCount(ts);  i++ )
  {
    task = KheTaskSetTask(ts, i);
    KheTaskDoInterval(task, tf->days_frame, &res);
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "task finders - construction and query"                        */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_TASK_FINDER KheTaskFinderMake(KHE_SOLN soln, KHE_OPTIONS options,    */
/*    KHE_ARENA a)                                                           */
/*                                                                           */
/*  Return a new task finder object, or NULL is there is no frame or etm.    */
/*                                                                           */
/*****************************************************************************/

KHE_TASK_FINDER KheTaskFinderMake(KHE_SOLN soln, KHE_OPTIONS options,
  HA_ARENA a)
{ 
  KHE_TASK_FINDER res;  KHE_FRAME days_frame;  KHE_EVENT_TIMETABLE_MONITOR etm;

  /* get days_frame and etm and return NULL if can't */
  days_frame = KheOptionsFrame(options, "gs_common_frame", soln);
  etm = (KHE_EVENT_TIMETABLE_MONITOR)
    KheOptionsGetObject(options, "gs_event_timetable_monitor", NULL);
  if( days_frame == NULL || etm == NULL )
    return NULL;

  /* create, initialize, and return the result */
  HaMake(res, a);
  res->arena = a;
  res->soln = soln;
  res->days_frame = days_frame;
  res->etm = etm;
  HaArrayInit(res->free_widened_task_sets, a);
  HaArrayInit(res->free_wings, a);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  int KheTaskFinderLastIndex(KHE_TASK_FINDER tf)                           */
/*                                                                           */
/*  Return the last legal day index.                                         */
/*                                                                           */
/*****************************************************************************/

int KheTaskFinderLastIndex(KHE_TASK_FINDER tf)
{
  return KheFrameTimeGroupCount(tf->days_frame) - 1;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "widened task sets - core"                                     */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheCoreInit(KHE_CORE *core, KHE_TASK_SET from_r_ts,                 */
/*    KHE_INTERVAL from_r_in)                                                */
/*                                                                           */
/*  Initialize *core with these attributes.  The tasks of from_r_ts are      */
/*  copied into core->from_r_ts; the task set itself is not stored.          */
/*                                                                           */
/*****************************************************************************/

static void KheCoreInit(KHE_CORE *core, KHE_TASK_SET from_r_ts,
  KHE_INTERVAL from_r_in)
{
  int i;
  KheTaskSetClear(core->from_r_ts);
  for( i = 0;  i < KheTaskSetTaskCount(from_r_ts);  i++ )
    KheTaskSetAddTask(core->from_r_ts, KheTaskSetTask(from_r_ts, i));
  core->from_r_in = from_r_in;
  /* core->to_r_ts = NULL; - no, this will already be set to an empty ts */
  core->to_r_ts_durn = 0;
  core->to_r_ts_effectively_free = true;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheCoreMoveCheck(KHE_CORE *core, bool force, KHE_RESOURCE to_r)     */
/*                                                                           */
/*  Return true if *core can move to to_r, assuming it has been prepared.    */
/*                                                                           */
/*****************************************************************************/

static bool KheCoreMoveCheck(KHE_CORE *core, bool force, KHE_RESOURCE to_r)
{
  return (force || core->to_r_ts_effectively_free) &&
    KheTaskSetUnAssignResourceCheck(core->to_r_ts) &&
    KheTaskSetMoveResourceCheck(core->from_r_ts, to_r);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheCoreMove(KHE_CORE *core, KHE_RESOURCE to_r,                      */
/*    int *from_r_durn_change, int *to_r_durn_change)                        */
/*                                                                           */
/*  Move core to to_r, updating *from_r_durn_change and *to_r_durn_change    */
/*  if successful.                                                           */
/*                                                                           */
/*****************************************************************************/

static bool KheCoreMove(KHE_CORE *core, KHE_RESOURCE to_r,
  int *from_r_durn_change, int *to_r_durn_change)
{
  int len;
  if( KheTaskSetUnAssignResource(core->to_r_ts) &&
      KheTaskSetMoveResource(core->from_r_ts, to_r) )
  {
    len = KheIntervalLength(core->from_r_in);
    *from_r_durn_change -= len;
    *to_r_durn_change += (len - core->to_r_ts_durn);
    return true;
  }
  else
    return false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskSetFirstTasksEquivalent(KHE_TASK_SET ts1, KHE_TASK_SET ts2)  */
/*                                                                           */
/*  Return true if ts1 and ts2 are non-empty and their first tasks are       */
/*  equivalent.                                                              */
/*                                                                           */
/*****************************************************************************/

static bool KheTaskSetFirstTasksEquivalent(KHE_TASK_SET ts1, KHE_TASK_SET ts2)
{
  KHE_TASK task1, task2;
  if( KheTaskSetTaskCount(ts1) == 0 || KheTaskSetTaskCount(ts2) == 0 )
    return false;
  task1 = KheTaskSetTask(ts1, 0);
  task2 = KheTaskSetTask(ts2, 0);
  return KheTaskEquivalent(task1, task2);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskBlockedByTimeGroup(KHE_TASK task, KHE_TIME_GROUP tg)         */
/*                                                                           */
/*  Return true if moving task is blocked because it has a time from tg.     */
/*                                                                           */
/*****************************************************************************/

static bool KheTaskBlockedByTimeGroup(KHE_TASK task, KHE_TIME_GROUP tg)
{
  KHE_MEET meet;  KHE_TIME t, t2;  int i, durn, pos;  KHE_TASK child_task;

  /* do the job for task itself */
  meet = KheTaskMeet(task);
  t = KheMeetAsstTime(meet);
  durn = KheMeetDuration(meet);
  for( i = 0;  i < durn;  i++ )
  {
    t2 = KheTimeNeighbour(t, i);
    if( KheTimeGroupContains(tg, t2, &pos) )
      return true;
  }

  /* do the job for task's children */
  for( i = 0;  i < KheTaskAssignedToCount(task);  i++ )
  {
    child_task = KheTaskAssignedTo(task, i);
    if( KheTaskBlockedByTimeGroup(child_task, tg) )
      return true;
  }

  /* no blockage anywhere */
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskSetBlockedByTimeGroup(KHE_TASK_SET ts, KHE_TIME_GROUP tg)    */
/*                                                                           */
/*  Return true if moving ts is blocked because some of the tasks have       */
/*  times from time group tg.  Here tg may be NULL, in which case false      */
/*  is returned.                                                             */
/*                                                                           */
/*****************************************************************************/

static bool KheTaskSetBlockedByTimeGroup(KHE_TASK_SET ts, KHE_TIME_GROUP tg)
{
  int i;  KHE_TASK task;
  if( tg != NULL )
  {
    for( i = 0;  i < KheTaskSetTaskCount(ts);  i++ )
    {
      task = KheTaskSetTask(ts, i);
      if( KheTaskBlockedByTimeGroup(task, tg) )
	return true;
    }
  }
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskBlockedByMonitor(KHE_TASK task, KHE_MONITOR m)               */
/*                                                                           */
/*  Return true if m monitors task.                                          */
/*                                                                           */
/*****************************************************************************/

static bool KheTaskBlockedByMonitor(KHE_TASK task, KHE_MONITOR m)
{
  KHE_EVENT_RESOURCE er;  int i;  KHE_SOLN soln;  KHE_TASK child_task;

  /* do the job for task itself */
  soln = KheTaskSoln(task);
  er = KheTaskEventResource(task);
  if( er != NULL )
    for( i = 0;  i < KheSolnEventResourceMonitorCount(soln, er);  i++ )
      if( KheSolnEventResourceMonitor(soln, er, i) == m )
	return true;

  /* do the job for the children of task */
  for( i = 0;  i < KheTaskAssignedToCount(task);  i++ )
  {
    child_task = KheTaskAssignedTo(task, i);
    if( KheTaskBlockedByMonitor(child_task, m) )
      return true;
  }
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskSetBlockedByMonitor(KHE_TASK_SET ts, KHE_MONITOR m)          */
/*                                                                           */
/*  Return true if moving ts is blocked because some of the tasks are        */
/*  monitored by m.  Here m may be NULL, in which case false is returned.    */
/*                                                                           */
/*****************************************************************************/

static bool KheTaskSetBlockedByMonitor(KHE_TASK_SET ts, KHE_MONITOR m)
{
  int i;  KHE_TASK task;
  if( m != NULL )
  {
    for( i = 0;  i < KheTaskSetTaskCount(ts);  i++ )
    {
      task = KheTaskSetTask(ts, i);
      if( KheTaskBlockedByMonitor(task, m) )
	return true;
    }
  }
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheCoreSwapCheck(KHE_CORE *core, KHE_RESOURCE from_r,               */
/*    KHE_RESOURCE to_r, bool exact, KHE_TIME_GROUP blocking_tg,             */
/*    KHE_MONITOR blocking_m)                                                */
/*                                                                           */
/*  Return true if *core can swap to to_r.                                   */
/*                                                                           */
/*****************************************************************************/

static bool KheCoreSwapCheck(KHE_CORE *core, KHE_RESOURCE from_r,
  KHE_RESOURCE to_r, bool exact, KHE_TIME_GROUP blocking_tg,
  KHE_MONITOR blocking_m)
{
  /* check equivalent tasks, blocking_tg, and blocking_m */
  if( KheTaskSetFirstTasksEquivalent(core->from_r_ts, core->to_r_ts) )
  {
    if( DEBUG1 )
      fprintf(stderr, "KheCoreSwapCheck returning false (equivalent tasks)\n");
    return false;
  }

  if( KheTaskSetBlockedByTimeGroup(core->to_r_ts, blocking_tg) )
  {
    if( DEBUG1 )
      fprintf(stderr, "KheCoreSwapCheck returning false (time group)\n");
    return false;
  }

  if( KheTaskSetBlockedByMonitor(core->to_r_ts, blocking_m) )
  {
    if( DEBUG1 )
      fprintf(stderr, "KheCoreSwapCheck returning false (monitor)\n");
    return false;
  }

  /* check exact */
  if( exact && core->to_r_ts_durn != KheIntervalLength(core->from_r_in))
  {
    if( DEBUG1 )
      fprintf(stderr, "KheCoreSwapCheck returning false (exact)\n");
    return false;
  }

  /* check move from from_r to to_r */
  if( !KheTaskSetMoveResourceCheck(core->from_r_ts, to_r) )
  {
    if( DEBUG1 )
      fprintf(stderr, "KheCoreSwapCheck returning false (from_r to to_r)\n");
    return false;
  }

  /* check move from to_r to from_r */
  if( !KheTaskSetMoveResourceCheck(core->to_r_ts, from_r) )
  {
    if( DEBUG1 )
      fprintf(stderr, "KheCoreSwapCheck returning false (to_r to from_r)\n");
    return false;
  }

  /* all good */
  if( DEBUG1 )
    fprintf(stderr, "KheCoreSwapCheck returning true\n");
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheCoreSwap(KHE_CORE *core, KHE_RESOURCE from_r, KHE_RESOURCE to_r, */
/*    int *from_r_durn_change, int *to_r_durn_change)                        */
/*                                                                           */
/*  Swap core to to_r, updating *from_r_durn_change and *to_r_durn_change    */
/*  if successful.                                                           */
/*                                                                           */
/*****************************************************************************/

static bool KheCoreSwap(KHE_CORE *core, KHE_RESOURCE from_r, KHE_RESOURCE to_r,
  int *from_r_durn_change, int *to_r_durn_change)
{
  int len;
  if( KheTaskSetMoveResource(core->from_r_ts, to_r) &&
      KheTaskSetMoveResource(core->to_r_ts, from_r) )
  {
    len = KheIntervalLength(core->from_r_in) - core->to_r_ts_durn;
    *from_r_durn_change -= len;
    *to_r_durn_change += len;
    return true;
  }
  else
    return false;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "widened task sets - wings"                                    */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_WING KheWingMake(KHE_WIDENED_TASK_SET wts, KHE_TASK from_r_task)     */
/*                                                                           */
/*  Make a new wing object holding from_r_task.                              */
/*                                                                           */
/*****************************************************************************/

static KHE_WING KheWingMake(KHE_WIDENED_TASK_SET wts, KHE_TASK from_r_task,
  KHE_INTERVAL from_r_in)
{
  KHE_WING res;  KHE_TASK_FINDER tf;

  /* get a wing object from wts's free list or arena */
  tf = wts->tf;
  if( HaArrayCount(tf->free_wings) > 0 )
    res = HaArrayLastAndDelete(tf->free_wings);
  else
    HaMake(res, tf->arena);

  /* initialize its fields (the to_r ones are really undefined) and return it */
  res->from_r_task = from_r_task;
  res->from_r_in = from_r_in;
  res->to_r_task = NULL;
  res->to_r_task_durn = 0;
  res->to_r_task_effectively_free = false;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheWingMoveCheck(KHE_WING wing, bool force, KHE_RESOURCE to_r)      */
/*                                                                           */
/*  Return true if wing can move to to_r.                                    */
/*                                                                           */
/*****************************************************************************/

static bool KheWingMoveCheck(KHE_WING wing, bool force, KHE_RESOURCE to_r)
{
  return (force || wing->to_r_task_effectively_free) &&
    (wing->to_r_task == NULL || KheTaskUnAssignCheck(wing->to_r_task)) &&
    KheTaskMoveResourceCheck(wing->from_r_task, to_r);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheWingMove(KHE_WING wing, KHE_RESOURCE to_r,                       */
/*    int *from_r_durn_change, int *to_r_durn_change)                        */
/*                                                                           */
/*  Move wing to to_r.                                                       */
/*                                                                           */
/*****************************************************************************/

static bool KheWingMove(KHE_WING wing, KHE_RESOURCE to_r,
  int *from_r_durn_change, int *to_r_durn_change)
{
  int len;
  if( (wing->to_r_task == NULL || KheTaskUnAssign(wing->to_r_task)) &&
      KheTaskMoveResource(wing->from_r_task, to_r) )
  {
    len = KheIntervalLength(wing->from_r_in);
    *from_r_durn_change -= len;
    *to_r_durn_change += (len - wing->to_r_task_durn);
    return true;
  }
  else
    return false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheWingSwapCheck(KHE_WING wing, KHE_RESOURCE to_r, bool exact)      */
/*                                                                           */
/*  Return true if wing can swap to to_r, exactly if exact.                  */
/*                                                                           */
/*****************************************************************************/

static bool KheWingSwapCheck(KHE_WING wing, KHE_RESOURCE from_r,
  KHE_RESOURCE to_r, bool exact)
{
  /* check exact */
  if( exact && wing->to_r_task_durn != KheIntervalLength(wing->from_r_in) )
    return false;

  /* check move from from_r to to_r */
  if( !KheTaskMoveResourceCheck(wing->from_r_task, to_r) )
    return false;

  /* check move from to_r to from_r, if any */
  if( wing->to_r_task != NULL &&
      !KheTaskMoveResourceCheck(wing->to_r_task, from_r) )
    return false;

  /* all good */
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheWingSwap(KHE_WING wing, KHE_RESOURCE from_r, KHE_RESOURCE to_r,  */
/*    int *from_r_durn_change, int *to_r_durn_change)                        */
/*                                                                           */
/*  Swap wing to to_r.                                                       */
/*                                                                           */
/*****************************************************************************/

static bool KheWingSwap(KHE_WING wing, KHE_RESOURCE from_r, KHE_RESOURCE to_r,
  int *from_r_durn_change, int *to_r_durn_change)
{
  int len;

  /* move from from_r to to_r */
  if( !KheTaskMoveResource(wing->from_r_task, to_r) )
    return false;

  /* move from to_r to from_r, if any */
  if( wing->to_r_task != NULL &&
      !KheTaskMoveResource(wing->to_r_task, from_r) )
    return false;

  /* all good */
  len = KheIntervalLength(wing->from_r_in) - wing->to_r_task_durn;
  *from_r_durn_change -= len;
  *to_r_durn_change += len;
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KheFindTask" (private)                                        */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool KheFindAssignedTask(KHE_TASK_FINDER tf, int day_index,              */
/*    KHE_INTERVAL *in, KHE_RESOURCE_TIMETABLE_MONITOR rtm,                  */
/*    KHE_TASK *res_task, KHE_INTERVAL *res_task_in)                         */
/*                                                                           */
/*  Like KheFindTask below, but for finding assigned tasks only.             */
/*                                                                           */
/*****************************************************************************/

static bool KheFindAssignedTask(KHE_TASK_FINDER tf, int day_index,
  KHE_INTERVAL *in, KHE_RESOURCE_TIMETABLE_MONITOR rtm,
  KHE_TASK *res_task, KHE_INTERVAL *res_task_in)
{
  KHE_TIME_GROUP tg;  KHE_TIME t;  int i, j, count;
  KHE_RESOURCE r;
  tg = KheFrameTimeGroup(tf->days_frame, day_index);
  for( i = 0;  i < KheTimeGroupTimeCount(tg);  i++ )
  {
    t = KheTimeGroupTime(tg, i);
    count = KheResourceTimetableMonitorTimeTaskCount(rtm, t);
    for( j = 0;  j < count;  j++ )
    {
      *res_task = KheResourceTimetableMonitorTimeTask(rtm, t, j);
      *res_task = KheTaskProperRoot(*res_task);
      if( !KheTaskIsPreassigned(*res_task, &r) )
      {
	*res_task_in = KheTaskInterval(tf, *res_task);
	if( KheIntervalDisjoint(*res_task_in, *in) )
	{
	  KheIntervalUnion(in, *res_task_in);
	  return true;
	}
      }
    }
  }
  *res_task = NULL;
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheResourceGroupCompatible(KHE_TASK task, KHE_RESOURCE_GROUP rg,    */
/*    int count)                                                             */
/*                                                                           */
/*  Return true if task's domain is compatible with rg.                      */
/*                                                                           */
/*****************************************************************************/

static bool KheResourceGroupCompatible(KHE_TASK task, KHE_RESOURCE_GROUP rg,
  int count)
{
  return rg == NULL ||
    KheResourceGroupIntersectCount(rg, KheTaskDomain(task)) >= count;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheFindUnassignedTask(KHE_TASK_FINDER tf, int day_index,            */
/*    KHE_INTERVAL *in, KHE_RESOURCE_TYPE rt, KHE_RESOURCE_GROUP rg,         */
/*    int tg_start_index, KHE_TASK *res_task, KHE_INTERVAL *res_task_in)     */
/*                                                                           */
/*  Like KheFindTask below, but for finding unassigned tasks only.           */
/*                                                                           */
/*****************************************************************************/

static bool KheFindUnassignedTask(KHE_TASK_FINDER tf, int day_index,
  KHE_INTERVAL *in, KHE_RESOURCE_TYPE rt, KHE_RESOURCE_GROUP rg,
  int tg_start_index, KHE_TASK *res_task, KHE_INTERVAL *res_task_in)
{
  KHE_TIME_GROUP tg;  KHE_TIME t;  int i, j, k, count, dc;  KHE_MEET meet;
  KHE_RESOURCE r;
  tg = KheFrameTimeGroup(tf->days_frame, day_index);
  if( rg != NULL )
    dc = KheResourceGroupResourceCount(rg) / 2;
  else
    dc = tg_start_index = 0;
  count = KheTimeGroupTimeCount(tg);
  for( i = 0;  i < count;  i++ )
  {
    t = KheTimeGroupTime(tg, (i + tg_start_index) % count);
    for( j = 0;  j < KheEventTimetableMonitorTimeMeetCount(tf->etm, t);  j++ )
    {
      meet = KheEventTimetableMonitorTimeMeet(tf->etm, t, j);
      for( k = 0;  k < KheMeetTaskCount(meet);  k++ )
      {
	*res_task = KheMeetTask(meet, k);
	*res_task = KheTaskProperRoot(*res_task);
	if( KheTaskAsstResource(*res_task) == NULL &&
	    KheTaskResourceType(*res_task) == rt &&
	    !KheTaskIsPreassigned(*res_task, &r) &&
	    KheTaskNeedsAssignment(*res_task) &&
	    KheResourceGroupCompatible(*res_task, rg, dc) )
	{
	  *res_task_in = KheTaskInterval(tf, *res_task);
	  if( KheIntervalDisjoint(*res_task_in, *in) )
	  {
	    KheIntervalUnion(in, *res_task_in);
	    return true;
	  }
	}
      }
    }
  }
  *res_task = NULL;
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheFindTask(KHE_TASK_FINDER tf, int day_index, KHE_INTERVAL *in,    */
/*    KHE_RESOURCE_TIMETABLE_MONITOR rtm, KHE_RESOURCE_TYPE rt,              */
/*    KHE_RESOURCE_GROUP rg, int tg_start_index, KHE_TASK *res_task,         */
/*    KHE_INTERVAL *res_task_in)                                             */
/*                                                                           */
/*  This function is the heart of task finding; it finds one task.           */
/*                                                                           */
/*  If possible, set *res_task to an unpreassigned proper root task running  */
/*  on day_index but not during *in, set *res_task_in to its interval, and   */
/*  replace *in by its union with *res_task_in.  Otherwise return false.     */
/*                                                                           */
/*  If rtm != NULL, the task is to come from rtm and only the requirements   */
/*  just given apply.  If rtm == NULL, the task is to be an unassigned       */
/*  task of type rt.  In that case, the task must need assignment, and if    */
/*  rg != NULL, the task's domain must be compatible with rg, and it should  */
/*  preferably be running at index tg_start_index in the times of its day.   */
/*                                                                           */
/*****************************************************************************/

static bool KheFindTask(KHE_TASK_FINDER tf, int day_index, KHE_INTERVAL *in,
  KHE_RESOURCE_TIMETABLE_MONITOR rtm, KHE_RESOURCE_TYPE rt,
  KHE_RESOURCE_GROUP rg, int tg_start_index, KHE_TASK *res_task,
  KHE_INTERVAL *res_task_in)
{
  if( rtm != NULL )
    return KheFindAssignedTask(tf, day_index, in, rtm, res_task, res_task_in);
  else
    return KheFindUnassignedTask(tf, day_index, in, rt, rg, tg_start_index,
      res_task, res_task_in);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheGatherTasks(KHE_TASK_FINDER tf, KHE_INTERVAL in, bool exact,     */
/*    KHE_RESOURCE_TIMETABLE_MONITOR rtm, KHE_TASK_SET res_ts,               */
/*    int *res_durn, bool *res_effectively_free)                             */
/*                                                                           */
/*  Find all tasks from rtm running within in.  Add them to *res_ts, add     */
/*  their durations to *res_durn, and set *res_effectively_free to false     */
/*  if any of them need assignment.  Also return false if any of them        */
/*  are preassigned, or (when exact is true) running outside in.             */
/*                                                                           */
/*  Note.  One step of this function is similar to KheFindAssignedTask.      */
/*  But on close inspection there are differences as well as similarities.   */
/*                                                                           */
/*****************************************************************************/

static bool KheGatherTasks(KHE_TASK_FINDER tf, KHE_INTERVAL in, bool exact,
  KHE_RESOURCE_TIMETABLE_MONITOR rtm, KHE_TASK_SET res_ts,
  int *res_durn, bool *res_effectively_free)
{
  int day_index, i, j, count, pos;  KHE_TIME_GROUP tg;  KHE_TIME t;
  KHE_TASK task;  KHE_RESOURCE r;  KHE_INTERVAL task_in;
  /* ***
  KheTaskSetClear(res_ts);
  *res_durn = 0;
  *res_effectively_free = true;
  *** */
  if( rtm == NULL )
    return true;
  for( day_index = in.first;  day_index <= in.last;  day_index++ )
  {
    tg = KheFrameTimeGroup(tf->days_frame, day_index);
    for( i = 0;  i < KheTimeGroupTimeCount(tg);  i++ )
    {
      t = KheTimeGroupTime(tg, i);
      count = KheResourceTimetableMonitorTimeTaskCount(rtm, t);
      for( j = 0;  j < count;  j++ )
      {
	task = KheResourceTimetableMonitorTimeTask(rtm, t, j);
	task = KheTaskProperRoot(task);
	if( !KheTaskSetContainsTask(res_ts, task, &pos) )
	{
	  if( KheTaskIsPreassigned(task, &r) )
	    return false;
	  task_in = KheTaskInterval(tf, task);
	  if( exact && !KheIntervalSubset(task_in, in) )
	    return false;
	  KheTaskSetAddTask(res_ts, task);
	  *res_durn += KheIntervalLength(task_in);
	  if( KheTaskNeedsAssignment(task) )
	    *res_effectively_free = false;
	}
      }
    }
  }
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheResourceEffectivelyFree(KHE_RESOURCE_TIMETABLE_MONITOR rtm,      */
/*    KHE_INTERVAL in, KHE_TASK_SET unass_ts)                                */
/*                                                                           */
/*  If rtm is effectively free during in, return true and add any tasks      */
/*  that would need to be unassigned to unass_ts (unless already there).     */
/*                                                                           */
/*****************************************************************************/

/* *** unified with KheGatherTasks
static bool KheResourceEffectivelyFree(KHE_TASK_FINDER tf,
  KHE_RESOURCE_TIMETABLE_MONITOR rtm, KHE_INTERVAL in,
  KHE_TASK_SET res_ts, int *res_durn)
{
  int day_index, i, j, count, pos;  KHE_TIME_GROUP tg;  KHE_TIME t;
  KHE_TASK task;  KHE_RESOURCE r;
  if( rtm == NULL )
    return true;
  for( day_index = in.first;  day_index <= in.last;  day_index++ )
  {
    tg = KheFrameTimeGroup(tf->days_frame, day_index);
    for( i = 0;  i < KheTimeGroupTimeCount(tg);  i++ )
    {
      t = KheTimeGroupTime(tg, i);
      count = KheResourceTimetableMonitorTimeTaskCount(rtm, t);
      for( j = 0;  j < count;  j++ )
      {
	task = KheResourceTimetableMonitorTimeTask(rtm, t, j);
	task = KheTaskProperRoot(task);
	if( KheTaskIsPreassigned(task, &r) || KheTaskNeedsAssignment(task) )
	  return false;
	if( !KheTaskSetContainsTask(res_ts, task, &pos) )
	{
          KheTaskSetAddTask(res_ts, task);
	  *res_durn += KheIntervalLength(KheTaskInterval(tf, task));
	}
      }
    }
  }
  return true;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  Submodule "widened task sets - construction"                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheWidenedTaskSetBuildLeftWing(KHE_WIDENED_TASK_SET wts,            */
/*    KHE_RESOURCE_TIMETABLE_MONITOR rtm, KHE_RESOURCE_TYPE rt,              */
/*    KHE_RESOURCE_GROUP rg, int tg_start_index, int max_left_wing_count)    */
/*                                                                           */
/*  Build a left wing for wts containing up to max_left_wing_count tasks.    */
/*                                                                           */
/*****************************************************************************/

static void KheWidenedTaskSetBuildLeftWing(KHE_WIDENED_TASK_SET wts,
  KHE_RESOURCE_TIMETABLE_MONITOR rtm, KHE_RESOURCE_TYPE rt,
  KHE_RESOURCE_GROUP rg, int tg_start_index, int max_left_wing_count)
{
  KHE_INTERVAL in, task_in;  int i, day_index;  KHE_TASK task;
  KHE_WING wing;
  in = wts->core.from_r_in;
  for( i = 0;  i < max_left_wing_count;  i++ )
  {
    day_index = in.first - 1;
    if( day_index < 0 || !KheFindTask(wts->tf, day_index, &in, rtm, rt,
	  rg, tg_start_index, &task, &task_in) )
      break;
    wing = KheWingMake(wts, task, task_in);
    HaArrayAddLast(wts->left_wings, wing);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheWidenedTaskSetBuildRightWing(KHE_WIDENED_TASK_SET wts,           */
/*    KHE_RESOURCE_TIMETABLE_MONITOR rtm, KHE_RESOURCE_TYPE rt,              */
/*    KHE_RESOURCE_GROUP rg, int tg_start_index, int max_right_wing_count)   */
/*                                                                           */
/*  Build a right wing for wts containing up to max_right_wing_count tasks.  */
/*                                                                           */
/*****************************************************************************/

static void KheWidenedTaskSetBuildRightWing(KHE_WIDENED_TASK_SET wts,
  KHE_RESOURCE_TIMETABLE_MONITOR rtm, KHE_RESOURCE_TYPE rt,
  KHE_RESOURCE_GROUP rg, int tg_start_index, int max_right_wing_count)
{
  KHE_INTERVAL in, task_in;  int i, day_index, lim;  KHE_TASK task;
  KHE_WING wing;
  lim = KheFrameTimeGroupCount(wts->tf->days_frame);
  in = wts->core.from_r_in;
  for( i = 0;  i < max_right_wing_count;  i++ )
  {
    day_index = in.last + 1;
    if( day_index >= lim || !KheFindTask(wts->tf, day_index, &in, rtm, rt,
	  rg, tg_start_index, &task, &task_in) )
      break;
    wing = KheWingMake(wts, task, task_in);
    HaArrayAddLast(wts->right_wings, wing);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  int KheTaskIndexInFrameTimeGroup(KHE_TASK task, KHE_FRAME days_frame)    */
/*                                                                           */
/*  Return the index in its days_frame time group of the time that task      */
/*  is running.                                                              */
/*                                                                           */
/*****************************************************************************/

static int KheTaskIndexInFrameTimeGroup(KHE_TASK task, KHE_FRAME days_frame)
{
  KHE_MEET meet;  KHE_TIME t;  KHE_TIME_GROUP tg;  int pos;
  meet = KheTaskMeet(task);
  if( meet != NULL )
  {
    t = KheMeetAsstTime(meet);
    if( t != NULL )
    {
      tg = KheFrameTimeTimeGroup(days_frame, t);
      if( KheTimeGroupContains(tg, t, &pos) )
	return pos;
      HnAbort("KheTaskIndexInFrameTimeGroup internal error");
    }
  }
  return 0;
}


/*****************************************************************************/
/*                                                                           */
/*****************************************************************************/

static KHE_WIDENED_TASK_SET KheWidenedTaskSetGet(KHE_TASK_FINDER tf)
{
  KHE_WIDENED_TASK_SET res;  HA_ARENA a;
  if( HaArrayCount(tf->free_widened_task_sets) > 0 )
  {
    /* get res from tf's free list */
    res = HaArrayLastAndDelete(tf->free_widened_task_sets);
    KheTaskSetClear(res->scratch_ts);
    KheTaskSetClear(res->core.from_r_ts);
    KheTaskSetClear(res->core.to_r_ts);
    HaArrayClear(res->left_wings);
    HaArrayClear(res->right_wings);
  }
  else
  {
    /* get res from tf's arena */
    a = tf->arena;
    HaMake(res, a);
    res->scratch_ts = KheTaskSetMake(tf->soln);
    res->core.from_r_ts = KheTaskSetMake(tf->soln);
    res->core.to_r_ts = KheTaskSetMake(tf->soln);
    HaArrayInit(res->left_wings, a);
    HaArrayInit(res->right_wings, a);
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheWidenedTaskSetMake(KHE_TASK_FINDER tf, KHE_RESOURCE from_r,      */
/*    KHE_TASK_SET from_r_ts, int max_left_wing_count,                       */
/*    int max_right_wing_count, KHE_WIDENED_TASK_SET *wts)                   */
/*                                                                           */
/*  Check the basic conditions for from_r_ts and make a wts for it.          */
/*                                                                           */
/*****************************************************************************/

bool KheWidenedTaskSetMake(KHE_TASK_FINDER tf, KHE_RESOURCE from_r,
  KHE_TASK_SET from_r_ts, int max_left_wing_count,
  int max_right_wing_count, KHE_WIDENED_TASK_SET *wts)
{
  KHE_WIDENED_TASK_SET res;  KHE_TASK task;  KHE_RESOURCE_GROUP rg;
  KHE_INTERVAL from_r_in, task_in;  int i, tg_start_index;  KHE_RESOURCE r;
  KHE_RESOURCE_TYPE rt;

  /* check the basic conditions for from_r_ts and find its bounding interval */
  if( KheTaskSetTaskCount(from_r_ts) == 0 )
    return *wts = NULL, false;
  from_r_in = KheInterval(1, 0);
  for( i = 0;  i < KheTaskSetTaskCount(from_r_ts);  i++ )
  {
    task = KheTaskSetTask(from_r_ts, i);
    HnAssert(KheTaskAsstResource(task) == from_r,
      "KheWidenedTaskSetMake: from_r_ts has a task not assigned from_r");
    if( KheTaskProperRoot(task) != task || KheTaskIsPreassigned(task, &r) )
      return *wts = NULL, false;
    task_in = KheTaskInterval(tf, task);
    if( !KheIntervalDisjoint(task_in, from_r_in) )
      return *wts = NULL, false;
    KheIntervalUnion(&from_r_in, task_in);
  }

  /* get res, the new widened task set object */
  res = KheWidenedTaskSetGet(tf);

  /* get the first task and its resource type */
  task = KheTaskSetFirst(from_r_ts);
  rt = KheResourceGroupResourceType(KheTaskDomain(task));

  /* set the fields */
  res->tf = tf;
  res->from_r = from_r;
  if( from_r != NULL )
  {
    res->from_r_rtm = KheResourceTimetableMonitor(tf->soln, from_r);
    rg = NULL;
    tg_start_index = -1;
  }
  else
  {
    res->from_r_rtm = NULL;
    rg = KheTaskDomain(task);
    tg_start_index = KheTaskIndexInFrameTimeGroup(task, tf->days_frame);
  }
  res->to_r = (KHE_RESOURCE) 0x1;  /* not equal to any actual resource */
  res->to_r_rtm = NULL;
  KheCoreInit(&res->core, from_r_ts, from_r_in);
  KheWidenedTaskSetBuildLeftWing(res, res->from_r_rtm, rt, rg, tg_start_index,
    max_left_wing_count);
  KheWidenedTaskSetBuildRightWing(res, res->from_r_rtm, rt, rg, tg_start_index,
    max_right_wing_count);
  res->to_r_max_left_wing_count = -1;	/* undefined until to_r arrives */
  res->to_r_max_right_wing_count = -1;	/* undefined until to_r arrives */
  return *wts = res, true;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheWidenedTaskSetDelete(KHE_WIDENED_TASK_SET wts)                   */
/*                                                                           */
/*  Delete wts by cycling it through a free list in tf.                      */
/*                                                                           */
/*  Implementation note.  The number of wings will change next time, so      */
/*  we move them to a free list too.                                         */
/*                                                                           */
/*****************************************************************************/

void KheWidenedTaskSetDelete(KHE_WIDENED_TASK_SET wts)
{
  int i;  KHE_TASK_FINDER tf;
  tf = wts->tf;
  HaArrayAppend(tf->free_wings, wts->left_wings, i);
  HaArrayAppend(tf->free_wings, wts->right_wings, i);
  HaArrayAddLast(tf->free_widened_task_sets, wts);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "widened task sets - moves and swaps"                          */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool KhePrepareResource(KHE_WIDENED_TASK_SET wts, KHE_RESOURCE to_r)     */
/*                                                                           */
/*  Get wts ready for operations involving moves from from_r to to_r, or     */
/*  return false if that can't be done.  Do nothing if already set for to_r. */
/*                                                                           */
/*****************************************************************************/

static bool KhePrepareResource(KHE_WIDENED_TASK_SET wts, KHE_RESOURCE to_r)
{
  bool ef;  int i, x;  KHE_WING wing;  KHE_TASK_SET ts;
  if( to_r != wts->to_r )
  {
    if( to_r != NULL )
    {
      wts->to_r = to_r;
      wts->to_r_rtm = KheResourceTimetableMonitor(wts->tf->soln, to_r);
      ts = wts->scratch_ts;

      /* core - fail if it doesn't work */
      KheTaskSetClear(wts->core.to_r_ts);
      wts->core.to_r_ts_durn = 0;
      wts->core.to_r_ts_effectively_free = true;
      if( !KheGatherTasks(wts->tf, wts->core.from_r_in, true,
	    wts->to_r_rtm, wts->core.to_r_ts, &wts->core.to_r_ts_durn,
	    &wts->core.to_r_ts_effectively_free) )
	return wts->to_r = (KHE_RESOURCE) 0x1, false;

      /* left wing - stop at first that doesn't work */
      HaArrayForEach(wts->left_wings, wing, i)
      {
	KheTaskSetClear(ts);
	x = 0;
	ef = true;
	if( !KheGatherTasks(wts->tf, wing->from_r_in, true,
	      wts->to_r_rtm, ts, &x, &ef) || KheTaskSetTaskCount(ts) >= 2 )
	  break;
        wing->to_r_task = (KheTaskSetTaskCount(ts) == 0 ? NULL :
	  KheTaskSetFirst(ts));
	wing->to_r_task_durn = x;
	wing->to_r_task_effectively_free = ef;
      }
      wts->to_r_max_left_wing_count = i;

      /* right wing - stop at first that doesn't work */
      HaArrayForEach(wts->right_wings, wing, i)
      {
	KheTaskSetClear(ts);
	x = 0;
	ef = true;
	if( !KheGatherTasks(wts->tf, wing->from_r_in, true,
	      wts->to_r_rtm, ts, &x, &ef) || KheTaskSetTaskCount(ts) >= 2 )
	  break;
        wing->to_r_task = (KheTaskSetTaskCount(ts) == 0 ? NULL :
	  KheTaskSetFirst(ts));
	wing->to_r_task_durn = x;
	wing->to_r_task_effectively_free = ef;
      }
      wts->to_r_max_right_wing_count = i;
    }
    else
    {
      /* free during the core */
      wts->to_r = to_r;
      wts->to_r_rtm = NULL;
      KheTaskSetClear(wts->core.to_r_ts);
      wts->core.to_r_ts_durn = 0;
      wts->core.to_r_ts_effectively_free = true;

      /* free during the left wing */
      HaArrayForEach(wts->left_wings, wing, i)
      {
	wing->to_r_task = NULL;
	wing->to_r_task_durn = 0;
	wing->to_r_task_effectively_free = true;
      }
      wts->to_r_max_left_wing_count = i;

      /* free during the right wing */
      HaArrayForEach(wts->right_wings, wing, i)
      {
	wing->to_r_task = NULL;
	wing->to_r_task_durn = 0;
	wing->to_r_task_effectively_free = true;
      }
      wts->to_r_max_right_wing_count = i;
    }
  }
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheWidenedTaskSetMoveCheck(KHE_WIDENED_TASK_SET wts,                */
/*    KHE_RESOURCE to_r, bool force, int *max_left_count,                    */
/*    int *max_right_count)                                                  */
/*                                                                           */
/*  Return true if the core of wts can move to to_r; set *max_left_count     */
/*  and *max_right_count to the number of left and right wings that can too. */
/*                                                                           */
/*****************************************************************************/

bool KheWidenedTaskSetMoveCheck(KHE_WIDENED_TASK_SET wts,
  KHE_RESOURCE to_r, bool force, int *max_left_count, int *max_right_count)
{
  int i, j;  KHE_WING wing;

  /* give up now if to_r == from_r or can't prepare to_r */
  if( to_r == wts->from_r || !KhePrepareResource(wts, to_r) )
    return *max_left_count = *max_right_count = -1, false;

  /* core must be movable, else fail */
  if( !KheCoreMoveCheck(&wts->core, force, to_r) )
    return *max_left_count = *max_right_count = -1, false;

  /* find i, the number of left wings that are movable */
  for( i = 0;  i < wts->to_r_max_left_wing_count;  i++ )
  {
    wing = HaArray(wts->left_wings, i);
    if( !KheWingMoveCheck(wing, force, to_r) )
      break;
  }

  /* find j, the number of right wings that are movable */
  for( j = 0;  j < wts->to_r_max_right_wing_count;  j++ )
  {
    wing = HaArray(wts->right_wings, j);
    if( !KheWingMoveCheck(wing, force, to_r) )
      break;
  }

  /* all good, set *max_left_count and *max_right_count and return */
  return *max_left_count = i, *max_right_count = j, true;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheWidenedTaskSetMove(KHE_WIDENED_TASK_SET wts,                     */
/*    KHE_RESOURCE to_r, int left_count, int right_count,                    */
/*    int *from_r_durn_change, int *to_r_durn_change)                        */
/*                                                                           */
/*  Move the core of wts, its first left_count left wings, and its first     */
/*  right_count right wings, to to_r.  If successful return true and set     */
/*  *from_r_durn_change and *to_r_durn_change to the change in duration.     */
/*                                                                           */
/*****************************************************************************/

bool KheWidenedTaskSetMove(KHE_WIDENED_TASK_SET wts,
  KHE_RESOURCE to_r, int left_count, int right_count,
  int *from_r_durn_change, int *to_r_durn_change)
{
  int i;  KHE_WING wing;

  /* give up now if to_r == from_r or can't prepare to_r */
  *from_r_durn_change = *to_r_durn_change = 0;
  if( to_r == wts->from_r || !KhePrepareResource(wts, to_r) )
    return false;

  /* check that left_count and right_count are in range */
  HnAssert(0 <= left_count && left_count <= wts->to_r_max_left_wing_count,
    "KheWidenedTaskSetMove: left_count (%d) out of range (0 .. %d)",
    left_count, wts->to_r_max_left_wing_count);
  HnAssert(0 <= right_count && right_count <= wts->to_r_max_right_wing_count,
    "KheWidenedTaskSetMove: right_count (%d) out of range (0 .. %d)",
    right_count, wts->to_r_max_right_wing_count);

  /* move the core */
  if( !KheCoreMove(&wts->core, to_r, from_r_durn_change, to_r_durn_change) )
    return false;

  /* move the first left_count left wings */
  for( i = 0;  i < left_count;  i++ )
  {
    wing = HaArray(wts->left_wings, i);
    if( !KheWingMove(wing, to_r, from_r_durn_change, to_r_durn_change) )
      return false;
  }

  /* move the first right_count right wings */
  for( i = 0;  i < right_count;  i++ )
  {
    wing = HaArray(wts->right_wings, i);
    if( !KheWingMove(wing, to_r, from_r_durn_change, to_r_durn_change) )
      return false;
  }

  /* all good */
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheResourceShow(KHE_RESOURCE r)                                    */
/*                                                                           */
/*  Return the Id of r, or "@" if NULL.                                      */
/*                                                                           */
/*****************************************************************************/

static char *KheResourceShow(KHE_RESOURCE r)
{
  return r == NULL ? "@" : KheResourceId(r);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheWidenedTaskSetSwapCheck(KHE_WIDENED_TASK_SET wts,                */
/*    KHE_RESOURCE to_r, bool exact, KHE_TIME_GROUP blocking_tg,             */
/*    KHE_MONITOR blocking_m, int *max_left_count, int *max_right_count)     */
/*                                                                           */
/*  Check whether wts is swappable with to_r.                                */
/*                                                                           */
/*****************************************************************************/

bool KheWidenedTaskSetSwapCheck(KHE_WIDENED_TASK_SET wts,
  KHE_RESOURCE to_r, bool exact, KHE_TIME_GROUP blocking_tg,
  KHE_MONITOR blocking_m, int *max_left_count, int *max_right_count)
{
  int i, j;  KHE_WING wing;

  /* give up now if to_r == from_r or can't prepare to_r */
  if( to_r == wts->from_r || !KhePrepareResource(wts, to_r) )
  {
    if( DEBUG2 )
      fprintf(stderr, "KheWidenedTaskSetSwapCheck(%s) returning false (prep)\n",
	KheResourceShow(to_r));
    return *max_left_count = *max_right_count = -1, false;
  }

  /* check core swappable */
  if( !KheCoreSwapCheck(&wts->core, wts->from_r, to_r, exact,
	blocking_tg, blocking_m) )
  {
    if( DEBUG2 )
      fprintf(stderr, "KheWidenedTaskSetSwapCheck(%s) returning false (core)\n",
	KheResourceShow(to_r));
    return *max_left_count = *max_right_count = -1, false;
  }

  /* find swappable left wings */
  for( i = 0;  i < wts->to_r_max_left_wing_count;  i++ )
  {
    wing = HaArray(wts->left_wings, i);
    if( !KheWingSwapCheck(wing, wts->from_r, to_r, exact) )
      break;
  }

  /* find swappable right wings */
  for( j = 0;  j < wts->to_r_max_right_wing_count;  j++ )
  {
    wing = HaArray(wts->right_wings, j);
    if( !KheWingSwapCheck(wing, wts->from_r, to_r, exact) )
      break;
  }

  /* all good, set *max_left_count and *max_right_count and return */
  if( DEBUG2 )
    fprintf(stderr, "KheWidenedTaskSetSwapCheck(%s) returning true (%d, %d)\n",
      KheResourceShow(to_r), i, j);
  return *max_left_count = i, *max_right_count = j, true;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheWidenedTaskSetSwap(KHE_WIDENED_TASK_SET wts,                     */
/*    KHE_RESOURCE to_r, int left_count, int right_count,                    */
/*    int *from_r_durn_change, int *to_r_durn_change)                        */
/*                                                                           */
/*  Swap wts to to_r.                                                        */
/*                                                                           */
/*****************************************************************************/

bool KheWidenedTaskSetSwap(KHE_WIDENED_TASK_SET wts,
  KHE_RESOURCE to_r, int left_count, int right_count,
  int *from_r_durn_change, int *to_r_durn_change)
{
  int i;  KHE_WING wing;

  /* give up now if to_r == from_r or can't prepare to_r */
  *from_r_durn_change = *to_r_durn_change = 0;
  if( to_r == wts->from_r || !KhePrepareResource(wts, to_r) )
    return false;

  /* check that left_count and right_count are in range */
  HnAssert(0 <= left_count && left_count <= wts->to_r_max_left_wing_count,
    "KheWidenedTaskSetSwap: left_count (%d) out of range (0 .. %d)",
    left_count, wts->to_r_max_left_wing_count);
  HnAssert(0 <= right_count && right_count <= wts->to_r_max_right_wing_count,
    "KheWidenedTaskSetSwap: right_count (%d) out of range (0 .. %d)",
    right_count, wts->to_r_max_right_wing_count);

  /* swap the core */
  if( !KheCoreSwap(&wts->core, wts->from_r, to_r,
	from_r_durn_change, to_r_durn_change) )
    return false;

  /* swap the first left_count left wings */
  for( i = 0;  i < left_count;  i++ )
  {
    wing = HaArray(wts->left_wings, i);
    if( !KheWingSwap(wing, wts->from_r, to_r,
	  from_r_durn_change, to_r_durn_change) )
      return false;
  }

  /* swap the first right_count right wings */
  for( i = 0;  i < right_count;  i++ )
  {
    wing = HaArray(wts->right_wings, i);
    if( !KheWingSwap(wing, wts->from_r, to_r,
	  from_r_durn_change, to_r_durn_change) )
      return false;
  }

  /* all good */
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheWidenedTaskSetInterval(KHE_WIDENED_TASK_SET wts,                 */
/*    int left_count, int right_count, int *first_index, int *last_index)    */
/*                                                                           */
/*  Return the interval covered by wts's core, its first left_count left     */
/*  wings, and its first right_count right wings.                            */
/*                                                                           */
/*****************************************************************************/

void KheWidenedTaskSetInterval(KHE_WIDENED_TASK_SET wts,
  int left_count, int right_count, int *first_index, int *last_index)
{
  KHE_INTERVAL res;  KHE_WING wing;  int i;
  res = wts->core.from_r_in;
  for( i = 0;  i < left_count;  i++ )
  {
    wing = HaArray(wts->left_wings, i);
    KheIntervalUnion(&res, wing->from_r_in);
  }
  for( i = 0;  i < right_count;  i++ )
  {
    wing = HaArray(wts->right_wings, i);
    KheIntervalUnion(&res, wing->from_r_in);
  }
  *first_index = res.first;
  *last_index = res.last;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheWidenedTaskSetDebug(KHE_WIDENED_TASK_SET wts, int left_count,    */
/*    int right_count, int verbosity, int indent, FILE *fp)                  */
/*                                                                           */
/*  Debug print of widened task set wts (but only the first left_count       */
/*  tasks of the left wing, and only the first right_count tasks of the      */
/*  right wing) onto fp with the given verbosity and indent.                 */
/*                                                                           */
/*****************************************************************************/

void KheWidenedTaskSetDebug(KHE_WIDENED_TASK_SET wts, int left_count,
  int right_count, int verbosity, int indent, FILE *fp)
{
  KHE_WING wing;  KHE_TASK task;  int i;
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  fprintf(fp, "{");
  if( left_count > HaArrayCount(wts->left_wings) )
    left_count = HaArrayCount(wts->left_wings);
  if( right_count > HaArrayCount(wts->right_wings) )
    right_count = HaArrayCount(wts->right_wings);
  for( i = left_count - 1;  i >= 0;  i-- )
  {
    wing = HaArray(wts->left_wings, i);
    if( i < left_count - 1 )
      fprintf(fp, ", ");
    KheTaskDebug(wing->from_r_task, verbosity, -1, fp);
  }
  if( left_count > 0 || right_count > 0 )
  {
    if( left_count > 0 )
      fprintf(fp, " ");
    fprintf(fp, "[");
  }
  for( i = 0;  i < KheTaskSetTaskCount(wts->core.from_r_ts);  i++ )
  {
    task = KheTaskSetTask(wts->core.from_r_ts, i);
    if( i > 0 )
      fprintf(fp, ", ");
    KheTaskDebug(task, verbosity, -1, fp);
  }
  if( left_count > 0 || right_count > 0 )
  {
    fprintf(fp, "]");
    if( right_count > 0 )
      fprintf(fp, " ");
  }
  for( i = 0;  i < right_count;  i++ )
  {
    wing = HaArray(wts->right_wings, i);
    if( i > 0 )
      fprintf(fp, ", ");
    KheTaskDebug(wing->from_r_task, verbosity, -1, fp);
  }
  fprintf(fp, "}");
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheWidenedTaskSetToResourceDebug(KHE_WIDENED_TASK_SET wts,          */
/*    KHE_RESOURCE to_r, int left_count, int right_count,                    */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of the to_r parts of widened task set wts (but only the      */
/*  first left_count tasks of the left wing, and only the first right_count  */
/*  tasks of the right wing) onto fp with the given verbosity and indent.    */
/*                                                                           */
/*****************************************************************************/

static void KheWidenedTaskSetToResourceDebug(KHE_WIDENED_TASK_SET wts,
  KHE_RESOURCE to_r, int left_count, int right_count,
  int verbosity, int indent, FILE *fp)
{
  KHE_WING wing;  KHE_TASK task;  int i;
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  if( to_r == wts->from_r )
    fprintf(fp, "failed (from_r == to_r)");
  else if ( to_r != wts->to_r )
    fprintf(fp, "failed (not prepared for %s)", KheResourceShow(to_r));
  else
  {
    fprintf(fp, "{");
    if( left_count > HaArrayCount(wts->left_wings) )
      left_count = HaArrayCount(wts->left_wings);
    if( right_count > HaArrayCount(wts->right_wings) )
      right_count = HaArrayCount(wts->right_wings);
    for( i = left_count - 1;  i >= 0;  i-- )
    {
      wing = HaArray(wts->left_wings, i);
      if( i < left_count - 1 )
	fprintf(fp, ", ");
      if( wing->to_r_task == NULL )
	fprintf(fp, "-");
      else
	KheTaskDebug(wing->to_r_task, verbosity, -1, fp);
    }
    if( left_count > 0 || right_count > 0 )
    {
      if( left_count > 0 )
	fprintf(fp, " ");
      fprintf(fp, "[");
    }
    for( i = 0;  i < KheTaskSetTaskCount(wts->core.to_r_ts);  i++ )
    {
      task = KheTaskSetTask(wts->core.to_r_ts, i);
      if( i > 0 )
	fprintf(fp, ", ");
      KheTaskDebug(task, verbosity, -1, fp);
    }
    if( left_count > 0 || right_count > 0 )
    {
      fprintf(fp, "]");
      if( right_count > 0 )
	fprintf(fp, " ");
    }
    for( i = 0;  i < right_count;  i++ )
    {
      wing = HaArray(wts->right_wings, i);
      if( i > 0 )
	fprintf(fp, ", ");
      if( wing->to_r_task == NULL )
	fprintf(fp, "-");
      else
	KheTaskDebug(wing->to_r_task, verbosity, -1, fp);
    }
    fprintf(fp, "}");
  }
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheWidenedTaskSetOpDebug(KHE_WIDENED_TASK_SET wts, char *op,        */
/*    KHE_RESOURCE to_r, int left_count, int right_count, int verbosity,     */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of an operation.                                             */
/*                                                                           */
/*****************************************************************************/

static void KheWidenedTaskSetOpDebug(KHE_WIDENED_TASK_SET wts, char *op,
  KHE_RESOURCE to_r, int left_count, int right_count, int verbosity,
  int indent, FILE *fp)
{
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  fprintf(fp, "%s ", KheResourceShow(wts->from_r));
  KheWidenedTaskSetDebug(wts, left_count, right_count, verbosity, indent, fp);
  fprintf(fp, " %s ", op);
  fprintf(fp, "%s ", KheResourceShow(to_r));
  KheWidenedTaskSetToResourceDebug(wts, to_r, left_count, right_count,
    verbosity, indent, fp);
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheWidenedTaskSetMoveDebug(KHE_WIDENED_TASK_SET wts,                */
/*    KHE_RESOURCE to_r, int left_count, int right_count,                    */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of a move.                                                   */
/*                                                                           */
/*****************************************************************************/

void KheWidenedTaskSetMoveDebug(KHE_WIDENED_TASK_SET wts,
  KHE_RESOURCE to_r, int left_count, int right_count,
  int verbosity, int indent, FILE *fp)
{
  KheWidenedTaskSetOpDebug(wts, "--->", to_r, left_count, right_count,
    verbosity, indent, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheWidenedTaskSetSwapDebug(KHE_WIDENED_TASK_SET wts,                */
/*    KHE_RESOURCE to_r, int left_count, int right_count,                    */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of a swap.                                                   */
/*                                                                           */
/*****************************************************************************/

void KheWidenedTaskSetSwapDebug(KHE_WIDENED_TASK_SET wts,
  KHE_RESOURCE to_r, int left_count, int right_count,
  int verbosity, int indent, FILE *fp)
{
  KheWidenedTaskSetOpDebug(wts, "<-->", to_r, left_count, right_count,
    verbosity, indent, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "other task finding operatons"                                 */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheTaskFinderTaskInterval(KHE_TASK_FINDER tf,                       */
/*    KHE_TASK task, int *first_index, int *last_index)                      */
/*                                                                           */
/*  Return the interval covered by task.                                     */
/*                                                                           */
/*****************************************************************************/

void KheTaskFinderTaskInterval(KHE_TASK_FINDER tf,
  KHE_TASK task, int *first_index, int *last_index)
{
  KHE_INTERVAL res;
  res = KheTaskInterval(tf, task);
  *first_index = res.first;
  *last_index = res.last;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheTaskFinderTaskSetInterval(KHE_TASK_FINDER tf,                    */
/*    KHE_TASK_SET ts, int *first_index, int *last_index)                    */
/*                                                                           */
/*  Return the interval covered by ts.                                       */
/*                                                                           */
/*****************************************************************************/

void KheTaskFinderTaskSetInterval(KHE_TASK_FINDER tf,
  KHE_TASK_SET ts, int *first_index, int *last_index)
{
  KHE_INTERVAL res;
  res = KheTaskSetInterval(tf, ts);
  *first_index = res.first;
  *last_index = res.last;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheFindTaskRunRight(KHE_TASK_FINDER tf, KHE_RESOURCE_TYPE rt,       */
/*    KHE_RESOURCE from_r, int days_first_index, KHE_TASK_SET res_ts,        */
/*    int *res_first_index, int *res_last_index)                             */
/*                                                                           */
/*  Find the leftmost run for from_r beginning at or after days_first_index. */
/*                                                                           */
/*****************************************************************************/

bool KheFindTaskRunRight(KHE_TASK_FINDER tf, KHE_RESOURCE_TYPE rt,
  KHE_RESOURCE from_r, int days_first_index, KHE_TASK_SET res_ts,
  int *res_first_index, int *res_last_index)
{
  int tg_count, tg_start_index, i;  KHE_TASK task;  KHE_RESOURCE_GROUP rg;
  KHE_INTERVAL task_in, in, res_in;  KHE_RESOURCE_TIMETABLE_MONITOR rtm;

  rtm = (from_r == NULL ? NULL : KheResourceTimetableMonitor(tf->soln, from_r));
  tg_count = KheFrameTimeGroupCount(tf->days_frame);
  if( days_first_index < 0 || days_first_index >= tg_count )
    return false;
  in = KheInterval(0, days_first_index - 1); 
  rg = NULL;
  tg_start_index = 0;
  KheTaskSetClear(res_ts);
  res_in = KheInterval(1, 0);
  i = days_first_index;
  while( i < tg_count )
  {
    if( KheFindTask(tf, i, &in, rtm, rt, rg, tg_start_index, &task, &task_in) )
    {
      /* if not in the run, start it now */
      if( KheTaskSetTaskCount(res_ts) == 0 )
      {
	if( rtm == NULL )
	{
	  rg = KheTaskDomain(task);
	  tg_start_index = KheTaskIndexInFrameTimeGroup(task, tf->days_frame);
	}
      }

      /* add task to the run and skip to after it */
      KheTaskSetAddTask(res_ts, task);
      KheIntervalUnion(&res_in, task_in);
      i = in.last + 1;
    }
    else
    {
      if( KheTaskSetTaskCount(res_ts) == 0 )
	i += 1;
      else
      {
	/* run has ended */
	break;
      }
    }
  }
  *res_first_index = res_in.first;
  *res_last_index = res_in.last;
  return KheTaskSetTaskCount(res_ts) > 0;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheFindTaskRunLeft(KHE_TASK_FINDER tf, KHE_RESOURCE_TYPE rt,        */
/*    KHE_RESOURCE from_r, int days_last_index, KHE_TASK_SET res_ts,         */
/*    KHE_INTERVAL *res_in)                                                  */
/*                                                                           */
/*  Find the rightmost run for from_r ending at or before days_last_index.   */
/*                                                                           */
/*****************************************************************************/

bool KheFindTaskRunLeft(KHE_TASK_FINDER tf, KHE_RESOURCE_TYPE rt,
  KHE_RESOURCE from_r, int days_last_index, KHE_TASK_SET res_ts,
  int *res_first_index, int *res_last_index)
{
  int tg_count, tg_start_index, i;  KHE_TASK task;  KHE_RESOURCE_GROUP rg;
  KHE_INTERVAL task_in, in, res_in;  KHE_RESOURCE_TIMETABLE_MONITOR rtm;

  rtm = (from_r == NULL ? NULL : KheResourceTimetableMonitor(tf->soln, from_r));
  tg_count = KheFrameTimeGroupCount(tf->days_frame);
  if( days_last_index < 0 || days_last_index >= tg_count )
    return false;
  in = KheInterval(days_last_index + 1, tg_count - 1); 
  rg = NULL;
  tg_start_index = 0;
  KheTaskSetClear(res_ts);
  res_in = KheInterval(1, 0);
  i = days_last_index;
  while( i >= 0 )
  {
    if( KheFindTask(tf, i, &in, rtm, rt, rg, tg_start_index, &task, &task_in) )
    {
      /* if not in the run, start it now */
      if( KheTaskSetTaskCount(res_ts) == 0 )
      {
	if( rtm == NULL )
	{
	  rg = KheTaskDomain(task);
	  tg_start_index = KheTaskIndexInFrameTimeGroup(task, tf->days_frame);
	}
      }

      /* add task to the run and skip to before it */
      KheTaskSetAddTask(res_ts, task);
      KheIntervalUnion(&res_in, task_in);
      i = in.first - 1;
    }
    else
    {
      if( KheTaskSetTaskCount(res_ts) == 0 )
	i -= 1;
      else
      {
	/* run has ended */
	break;
      }
    }
  }
  *res_first_index = res_in.first;
  *res_last_index = res_in.last;
  return KheTaskSetTaskCount(res_ts) > 0;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheFindTaskRunInitial(KHE_TASK_FINDER tf, KHE_TASK_SET ts,          */
/*    int wanted_durn, KHE_TASK_SET res_ts)                                  */
/*                                                                           */
/*  Find an initial sequence of ts's tasks whose duration equals wanted_durn.*/
/*                                                                           */
/*****************************************************************************/

bool KheFindTaskRunInitial(KHE_TASK_FINDER tf, KHE_TASK_SET ts,
  int wanted_durn, KHE_TASK_SET res_ts)
{
  int durn, i;  KHE_TASK task;
  durn = 0;
  KheTaskSetClear(res_ts);
  for( i = 0;  i < KheTaskSetTaskCount(ts) && durn < wanted_durn;  i++ )
  {
    task = KheTaskSetTask(ts, i);
    durn += KheIntervalLength(KheTaskInterval(tf, task));
    KheTaskSetAddTask(res_ts, task);
  }
  return durn == wanted_durn;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheFindTaskRunFinal(KHE_TASK_FINDER tf, KHE_TASK_SET ts,            */
/*    int wanted_durn, KHE_TASK_SET res_ts)                                  */
/*                                                                           */
/*  Find a final sequence of ts's tasks whose duration equals wanted_durn.   */
/*                                                                           */
/*****************************************************************************/

bool KheFindTaskRunFinal(KHE_TASK_FINDER tf, KHE_TASK_SET ts,
  int wanted_durn, KHE_TASK_SET res_ts)
{
  int durn, i;  KHE_TASK task;
  durn = 0;
  KheTaskSetClear(res_ts);
  for( i = KheTaskSetTaskCount(ts) - 1;  i >= 0 && durn < wanted_durn;  i-- )
  {
    task = KheTaskSetTask(ts, i);
    durn += KheIntervalLength(KheTaskInterval(tf, task));
    KheTaskSetAddTask(res_ts, task);
  }
  return durn == wanted_durn;
}


/*****************************************************************************/
/*                                                                           */
/*****************************************************************************/

bool KheFindMovableWidenedTaskSetRight(KHE_TASK_FINDER tf,
  KHE_RESOURCE_TYPE rt, KHE_RESOURCE from_r, KHE_RESOURCE to_r,
  int days_first_index, KHE_WIDENED_TASK_SET *res_wts)
{
  int tg_count, tg_start_index, i;  KHE_TASK task;  KHE_RESOURCE_GROUP rg;
  KHE_INTERVAL task_in, in;  KHE_WIDENED_TASK_SET res;  bool effectively_free;

  /* exit early if days_first_index is out of range */
  tg_count = KheFrameTimeGroupCount(tf->days_frame);
  if( days_first_index < 0 || days_first_index >= tg_count )
    return *res_wts = NULL, false;

  /* get a widened task set and set from_r and to_r */
  res = KheWidenedTaskSetGet(tf);
  res->from_r = from_r;
  res->from_r_rtm = (from_r == NULL ? NULL :
    KheResourceTimetableMonitor(tf->soln, from_r));
  res->to_r = to_r;
  res->to_r_rtm = (to_r == NULL ? NULL :
    KheResourceTimetableMonitor(tf->soln, to_r));

  /* search for a suitable from_r_ts and to_r_ts */
  in = KheInterval(0, days_first_index - 1); 
  rg = NULL;
  tg_start_index = 0;
  res->core.from_r_in = KheInterval(1, 0);
  KheTaskSetClear(res->core.to_r_ts);
  res->core.to_r_ts_durn = 0;
  res->core.to_r_ts_effectively_free = true;
  i = days_first_index;
  while( i < tg_count )
  {
    if( KheFindTask(tf, i, &in, res->from_r_rtm, rt, rg, tg_start_index,
	  &task, &task_in) && KheGatherTasks(tf, task_in, false,
	    res->to_r_rtm, res->core.to_r_ts, &res->core.to_r_ts_durn,
	    &effectively_free) && effectively_free )
    {
      /* if not in the run, start it now */
      if( KheTaskSetTaskCount(res->core.from_r_ts) == 0 )
      {
	if( res->from_r_rtm == NULL )
	{
	  rg = KheTaskDomain(task);
	  tg_start_index = KheTaskIndexInFrameTimeGroup(task, tf->days_frame);
	}
      }

      /* add task to the run and skip to after it */
      KheTaskSetAddTask(res->core.from_r_ts, task);
      KheIntervalUnion(&res->core.from_r_in, task_in);
      i = in.last + 1;
    }
    else
    {
      if( KheTaskSetTaskCount(res->core.from_r_ts) == 0 )
	i += 1;
      else
      {
	/* run has ended */
	break;
      }
    }
  }
  if( KheTaskSetTaskCount(res->core.from_r_ts) > 0 )
    return *res_wts = res, true;
  else
  {
    KheWidenedTaskSetDelete(res);
    return *res_wts = NULL, false;
  }
}


/*****************************************************************************/
/*                                                                           */
/*****************************************************************************/

bool KheFindMovableWidenedTaskSetLeft(KHE_TASK_FINDER tf,
  KHE_RESOURCE_TYPE rt, KHE_RESOURCE from_r, KHE_RESOURCE to_r,
  int days_last_index, KHE_WIDENED_TASK_SET *res_wts)
{
  /* still to do */
  HnAbort("KheFindMovableWidenedTaskSetLeft still to do");
  return false;
}
