
/*****************************************************************************/
/*                                                                           */
/*  THE KHE HIGH SCHOOL TIMETABLING ENGINE                                   */
/*  COPYRIGHT (C) 2010 Jeffrey H. Kingston                                   */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 3, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         khe_sr_single_resource.c                                   */
/*  DESCRIPTION:  Single resource solving                                    */
/*                                                                           */
/*****************************************************************************/
#include "khe_solvers.h"
#include "howard_p.h"
#include <limits.h>
#include <float.h>
#include <math.h>
#define bool_show(x) ((x) ? "true" : "false")

#define UNDEF_INT -1
#define UNDEF_FLOAT -1.0

#define DEBUG1 1		/* main functions */
#define DEBUG2 0
#define DEBUG3 0		/* adopted solutions */
#define DEBUG4 0
#define DEBUG5 1
#define DEBUG6 0
#define DEBUG7 0
#define DEBUG8 0
#define DEBUG9 0

#define TESTING 1

/*****************************************************************************/
/*                                                                           */
/*  Submodule "type declarations" (private)                                  */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_DOM_TYPE - the type of a dominance test                          */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  KHE_SRS_DOM_UNUSED,		/* "-"      should never be used      */
  KHE_SRS_DOM_GE,		/* ">="     min limit only            */
  KHE_SRS_DOM_LE,		/* "<="     max limit only            */
  KHE_SRS_DOM_EQ,		/* "==",    for uncertain cases       */
  KHE_SRS_DOM_GE_LOOSE,		/* ">=*"    min limit with loosening  */
  KHE_SRS_DOM_EQ_LOOSE		/* "==*"    max + min with loosening  */
} KHE_SRS_DOM_TYPE;


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_DOM_TEST - a dominance test                                      */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_srs_dom_test_rec {
  KHE_SRS_DOM_TYPE	type;			/* type of test              */
  int			min_limit;		/* if MIN_ONLY & MIN_AND_MAX */
} KHE_SRS_DOM_TEST;

typedef HA_ARRAY(KHE_SRS_DOM_TEST) ARRAY_KHE_SRS_DOM_TEST;


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_EXPR_TAG                                                         */
/*                                                                           */
/*  Tags for expression types.                                               */
/*                                                                           */
/*****************************************************************************/

typedef enum {				/* type of children */
  KHE_SRS_EXPR_BUSY_TAG,		/* no children      */
  KHE_SRS_EXPR_FREE_TAG,		/* no children      */
  KHE_SRS_EXPR_WORK_TAG,		/* no children      */
  KHE_SRS_EXPR_OR_TAG,			/* int              */
  KHE_SRS_EXPR_AND_TAG,			/* int              */
  KHE_SRS_EXPR_INT_SUM_TAG,		/* int              */
  /* KHE_SRS_EXPR_INT_SEQ_TAG, */	/* int              */
  KHE_SRS_EXPR_FLOAT_SUM_TAG,		/* float            */
  KHE_SRS_EXPR_INT_DEV_TAG,		/* int		    */
  KHE_SRS_EXPR_FLOAT_DEV_TAG,		/* float            */
  KHE_SRS_EXPR_COST_TAG,		/* int              */
  KHE_SRS_EXPR_COST_SUM_TAG,		/* KHE_COST         */
  KHE_SRS_EXPR_INT_SUM_COMB_TAG,	/* int              */
  KHE_SRS_EXPR_INT_SEQ_COMB_TAG		/* int              */
} KHE_SRS_EXPR_TAG;


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_COMB_TYPE - types of comb nodes                                  */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  KHE_SRS_COMB_ORDINARY,
  KHE_SRS_COMB_NO_MAX,
  KHE_SRS_COMB_LINEAR,
  KHE_SRS_COMB_STEP
} KHE_SRS_COMB_TYPE;


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_EXPR - abstract supertype of all expression types                */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_srs_expr_rec *KHE_SRS_EXPR;
typedef HA_ARRAY(KHE_SRS_EXPR) ARRAY_KHE_SRS_EXPR;

#define INHERIT_KHE_SRS_EXPR						\
  KHE_SRS_EXPR_TAG		tag;		/* tag field      */	\
  KHE_SRS_EXPR			parent;		/* parent expr    */	\
  ARRAY_KHE_SRS_EXPR		children;	/* child exprs    */	\
  int				first_day_index; /* first day     */	\
  int				last_day_index;	/* last day       */	\
  HA_ARRAY_INT			sig_indexes;	/* for each day   */	\
  KHE_SRS_DOM_TEST		dom_test;	/* dom test       */

struct khe_srs_expr_rec {
  INHERIT_KHE_SRS_EXPR
};


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_EXPR_BUSY                                                        */
/*                                                                           */
/*  An expression whose value is 1 when the resource is busy at a given      */
/*  time, and 0 when the resource is free then.  The expression stores the   */
/*  time, but only for debugging.                                            */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_srs_expr_busy_rec {
  INHERIT_KHE_SRS_EXPR
  KHE_TIME			time;		/* the time we're busy at    */
  int				tmp_value;
} *KHE_SRS_EXPR_BUSY;

typedef HA_ARRAY(KHE_SRS_EXPR_BUSY) ARRAY_KHE_SRS_EXPR_BUSY;


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_EXPR_FREE                                                        */
/*                                                                           */
/*  An expression whose value is 1 when the resource is free at a given      */
/*  time, and 0 when the resource is busy then.  The expression stores the   */
/*  time, but only for debugging.                                            */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_srs_expr_free_rec {
  INHERIT_KHE_SRS_EXPR
  KHE_TIME			time;		/* the time we're free at    */
  int				tmp_value;
} *KHE_SRS_EXPR_FREE;

typedef HA_ARRAY(KHE_SRS_EXPR_FREE) ARRAY_KHE_SRS_EXPR_FREE;


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_EXPR_WORK                                                        */
/*                                                                           */
/*  An expression whose value is the workload incurred by the resource at    */
/*  a given time.  This will be 0.0 if the resource is free then.  The       */
/*  expression stores the time, but only for debugging.                      */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_srs_expr_work_rec {
  INHERIT_KHE_SRS_EXPR
  KHE_TIME			time;		/* the time of this workload */
  float				tmp_value;
} *KHE_SRS_EXPR_WORK;

typedef HA_ARRAY(KHE_SRS_EXPR_WORK) ARRAY_KHE_SRS_EXPR_WORK;


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_EXPR_OR                                                          */
/*                                                                           */
/*  An expression whose value is the logical `or' of its children's values.  */
/*  Here Booleans are represented by integers (0 for false, 1 for true).     */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_srs_expr_or_rec {
  INHERIT_KHE_SRS_EXPR
  int				tmp_value;	/* defined only within day   */
} *KHE_SRS_EXPR_OR;

typedef HA_ARRAY(KHE_SRS_EXPR_OR) ARRAY_KHE_SRS_EXPR_OR;


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_EXPR_AND                                                         */
/*                                                                           */
/*  An expression whose value is the logical `and' of its children's values. */
/*  Here Booleans are represented by integers (0 for false, 1 for true).     */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_srs_expr_and_rec {
  INHERIT_KHE_SRS_EXPR
  int				tmp_value;	/* defined only within day   */
} *KHE_SRS_EXPR_AND;

typedef HA_ARRAY(KHE_SRS_EXPR_AND) ARRAY_KHE_SRS_EXPR_AND;


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_EXPR_INT_SUM                                                     */
/*                                                                           */
/*  An expression whose value is the sum of its children's values.           */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_srs_expr_int_sum_rec {
  INHERIT_KHE_SRS_EXPR
  /* int			history_before; */ /* history value          */
  /* int			history_after; */  /* history value          */
  int				tmp_value;	/* defined only within day   */
} *KHE_SRS_EXPR_INT_SUM;

typedef HA_ARRAY(KHE_SRS_EXPR_INT_SUM) ARRAY_KHE_SRS_EXPR_INT_SUM;


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_EXPR_INT_SEQ                                                     */
/*                                                                           */
/*  An expression whose value is the sequence of its children's values,      */
/*  although it does not remember all that; it remembers the length of       */
/*  the current sequence, and passes each length up as its sequence ends.    */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused; only KHE_SRS_EXPR_INT_SEQ_COMB is used
typedef struct khe_srs_expr_int_seq_rec {
  INHERIT_KHE_SRS_EXPR
  int				history_before;	** history value             **
  int				history_after;	** history value             **
  int				tmp_value;	** defined only within day   **
} *KHE_SRS_EXPR_INT_SEQ;

typedef HA_ARRAY(KHE_SRS_EXPR_INT_SEQ) ARRAY_KHE_SRS_EXPR_INT_SEQ;
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_EXPR_FLOAT_SUM                                                   */
/*                                                                           */
/*  An expression whose value is the sum of its children's values.  The      */
/*  expression's value, and its children's values, have type float.          */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_srs_expr_float_sum_rec {
  INHERIT_KHE_SRS_EXPR
  float				tmp_value;	/* defined only within day   */
} *KHE_SRS_EXPR_FLOAT_SUM;

typedef HA_ARRAY(KHE_SRS_EXPR_FLOAT_SUM) ARRAY_KHE_SRS_EXPR_FLOAT_SUM;


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_EXPR_INT_DEV                                                     */
/*                                                                           */
/*  An expression whose value is the deviation of its integer-valued child.  */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_srs_expr_int_dev_rec {
  INHERIT_KHE_SRS_EXPR
  int				min_limit;	/* minimum limit             */
  int				max_limit;	/* maximum limit             */
  bool				allow_zero;	/* true if 0 has deviation 0 */
} *KHE_SRS_EXPR_INT_DEV;

typedef HA_ARRAY(KHE_SRS_EXPR_INT_DEV) ARRAY_KHE_SRS_EXPR_INT_DEV;


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_EXPR_FLOAT_DEV                                                   */
/*                                                                           */
/*  An expression whose value is the deviation of its float-valued child.    */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_srs_expr_float_dev_rec {
  INHERIT_KHE_SRS_EXPR
  int				min_limit;	/* minimum limit             */
  int				max_limit;	/* maximum limit             */
  bool				allow_zero;	/* true if 0 has deviation 0 */
} *KHE_SRS_EXPR_FLOAT_DEV;

typedef HA_ARRAY(KHE_SRS_EXPR_FLOAT_DEV) ARRAY_KHE_SRS_EXPR_FLOAT_DEV;


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_EXPR_COST                                                        */
/*                                                                           */
/*  An expression whose value is a cost based on its sole child's deviation. */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_srs_expr_cost_rec {
  INHERIT_KHE_SRS_EXPR
  KHE_MONITOR			monitor;	/* the monitor (debug only)  */
  KHE_COST_FUNCTION		cost_fn;	/* the cost function         */
  KHE_COST			combined_weight; /* the combined weight      */
} *KHE_SRS_EXPR_COST;

typedef HA_ARRAY(KHE_SRS_EXPR_COST) ARRAY_KHE_SRS_EXPR_COST;


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_EXPR_COST_SUM                                                    */
/*                                                                           */
/*  An expression whose value is the sum of the values of its children.      */
/*  All these values are costs.                                              */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_srs_expr_cost_sum_rec {
  INHERIT_KHE_SRS_EXPR
  KHE_COST			tmp_value;	/* total cost                */
} *KHE_SRS_EXPR_COST_SUM;

typedef HA_ARRAY(KHE_SRS_EXPR_COST_SUM) ARRAY_KHE_SRS_EXPR_COST_SUM;


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_EXPR_INT_SUM_COMB                                                */
/*                                                                           */
/*  Combines INT_SUM, INT_DEV, and COST into a single node.  This is not     */
/*  only for convenience; there are important optimizations in some cases,   */
/*  depending on comb_type.                                                  */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_srs_expr_int_sum_comb_rec {
  INHERIT_KHE_SRS_EXPR
  KHE_MONITOR			monitor;	/* the monitor (debug only)  */
  KHE_COST_FUNCTION		cost_fn;	/* the cost function         */
  KHE_COST			combined_weight; /* the combined weight      */
  int				min_limit;	/* minimum limit             */
  int				max_limit;	/* maximum limit             */
  bool				allow_zero;	/* true if 0 has deviation 0 */
  int				history_before;	/* history value before      */
  int				history_after;	/* history value after       */
  KHE_SRS_COMB_TYPE		comb_type;	/* type of combined value    */
  int				tmp_value;	/* defined only within day   */
} *KHE_SRS_EXPR_INT_SUM_COMB;

typedef HA_ARRAY(KHE_SRS_EXPR_INT_SUM_COMB) ARRAY_KHE_SRS_EXPR_INT_SUM_COMB;


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_EXPR_INT_SEQ_COMB                                                */
/*                                                                           */
/*  Combines INT_SEQ, INT_DEV, and COST into a single node.  This is not     */
/*  only for convenience; there are important optimizations in some cases,   */
/*  depending on comb_type.                                                  */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_srs_expr_int_seq_comb_rec {
  INHERIT_KHE_SRS_EXPR
  KHE_MONITOR			monitor;	/* the monitor (debug only)  */
  KHE_COST_FUNCTION		cost_fn;	/* the cost function         */
  KHE_COST			combined_weight; /* the combined weight      */
  int				min_limit;	/* minimum limit             */
  int				max_limit;	/* maximum limit             */
  int				history_before;	/* history value before      */
  int				history_after;	/* history value after       */
  KHE_SRS_COMB_TYPE		comb_type;	/* type of combined value    */
  int				tmp_value;	/* defined only within day   */
} *KHE_SRS_EXPR_INT_SEQ_COMB;

typedef HA_ARRAY(KHE_SRS_EXPR_INT_SEQ_COMB) ARRAY_KHE_SRS_EXPR_INT_SEQ_COMB;


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_STEP - one step, either a task or some free time                 */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_srs_step_rec {
  KHE_TASK			task;			/* the task, or NULL */
  KHE_DAILY_SCHEDULE		daily_schedule;		/* its daily schedule*/
  KHE_COST			cost_reduction;		/* its cost reduction*/
  int				domain_count;		/* its domain size   */
} *KHE_SRS_STEP;

typedef HA_ARRAY(KHE_SRS_STEP) ARRAY_KHE_SRS_STEP;


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_TIME - one time                                                  */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_srs_time_rec {
  KHE_TIME			time;		/* the time represented      */
  ARRAY_KHE_SRS_EXPR		leaf_exprs;	/* leaf exprs for this time  */
  ARRAY_KHE_SRS_STEP		best_steps;	/* best tasks starting here  */
} *KHE_SRS_TIME;

typedef HA_ARRAY(KHE_SRS_TIME) ARRAY_KHE_SRS_TIME;


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_PARTIAL_SOLN - a partial solution                                */
/*                                                                           */
/*  The task field is a bit tricky.  It is NULL only in the special          */
/*  initial partial solution, the one representing an empty solution.        */
/*  In all other partial solutions it is non-NULL, but it may have a         */
/*  NULL task field, in which case it represents a free day.                 */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_srs_partial_soln_rec *KHE_SRS_PARTIAL_SOLN;

struct khe_srs_partial_soln_rec {
  KHE_SRS_PARTIAL_SOLN	prev_ps;		/* previous partial soln     */
  KHE_SRS_STEP		step;			/* the step leading to here  */
  int			day_index;		/* day containing this ps    */
  int			best_of;		/* how many this is best of  */
  KHE_COST		cost;			/* resource cost so far      */
  HA_ARRAY_INT		signature;		/* signature of this node    */
};

typedef HA_ARRAY(KHE_SRS_PARTIAL_SOLN) ARRAY_KHE_SRS_PARTIAL_SOLN;
typedef HP_TABLE(KHE_SRS_PARTIAL_SOLN) TABLE_KHE_SRS_PARTIAL_SOLN;


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_DAY - one day and everything about it                            */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_srs_day_rec {
  KHE_TIME_GROUP		time_group;		/* day's time group  */
  ARRAY_KHE_SRS_TIME		times;			/* times of this day */
  int				partial_solns_count;	/* table count       */
  int				sig_len;		/* length of sig     */
  ARRAY_KHE_SRS_DOM_TEST	dom_tests;		/* for each sig pos  */
  ARRAY_KHE_SRS_EXPR		active_exprs;		/* active on this day*/
  KHE_SRS_STEP			free_step;		/* free on this day  */
  TABLE_KHE_SRS_PARTIAL_SOLN	partial_solns_table;	/* if equivalence    */
  ARRAY_KHE_SRS_PARTIAL_SOLN	partial_solns_array;	/* if full dominance */
} *KHE_SRS_DAY;

typedef HA_ARRAY(KHE_SRS_DAY) ARRAY_KHE_SRS_DAY;


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_FULL_SOLN                                                        */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_srs_full_item_rec {
  KHE_SRS_STEP			step;
  bool				previously_assigned;
} KHE_SRS_FULL_ITEM;

typedef HA_ARRAY(KHE_SRS_FULL_ITEM) ARRAY_KHE_SRS_FULL_ITEM;

typedef struct khe_srs_full_soln_rec {
  KHE_RESOURCE			resource;
  int				asst_count;
  ARRAY_KHE_SRS_FULL_ITEM	items;
  bool				adopted;
} *KHE_SRS_FULL_SOLN;


/*****************************************************************************/
/*                                                                           */
/*  KHE_SINGLE_RESOURCE_SOLVER - a single resource solver                    */
/*                                                                           */
/*****************************************************************************/
typedef HA_ARRAY(KHE_TASK) ARRAY_KHE_TASK;

struct khe_single_resource_solver_rec {

  /* boilerplate fields */
  HA_ARENA			arena;
  KHE_SOLN			soln;
  KHE_OPTIONS			options;
  KHE_FRAME			days_frame;
  KHE_EVENT_TIMETABLE_MONITOR	etm;
  KHE_TASK_FINDER		task_finder;

  /* fields that vary with the resource and solve */
  int				diversifier;
  KHE_RESOURCE			resource;	/* resource (if solving)     */
  bool				full_dominance;
  int				min_assts;
  int				max_assts;
  KHE_SRS_EXPR_COST_SUM		root_expr;
  ARRAY_KHE_SRS_DAY		days;
  ARRAY_KHE_SRS_PARTIAL_SOLN	final_solns;
  KHE_SRS_FULL_SOLN		full_soln;
  bool				rerun;
  /* ***
  int				adopted_index;
  ARRAY_KHE_TASK		adopted_tasks;
  *** */

  /* free lists */
  ARRAY_KHE_SRS_EXPR_BUSY	free_expr_busy;
  ARRAY_KHE_SRS_EXPR_FREE	free_expr_free;
  ARRAY_KHE_SRS_EXPR_WORK	free_expr_work;
  ARRAY_KHE_SRS_EXPR_OR		free_expr_or;
  ARRAY_KHE_SRS_EXPR_AND	free_expr_and;
  ARRAY_KHE_SRS_EXPR_INT_SUM	free_expr_int_sum;
  /* ARRAY_KHE_SRS_EXPR_INT_SEQ	free_expr_int_seq; */
  ARRAY_KHE_SRS_EXPR_FLOAT_SUM	free_expr_float_sum;
  ARRAY_KHE_SRS_EXPR_INT_DEV	free_expr_int_dev;
  ARRAY_KHE_SRS_EXPR_FLOAT_DEV	free_expr_float_dev;
  ARRAY_KHE_SRS_EXPR_COST	free_expr_cost;
  ARRAY_KHE_SRS_EXPR_COST_SUM	free_expr_cost_sum;
  ARRAY_KHE_SRS_EXPR_INT_SUM_COMB free_expr_int_sum_comb;
  ARRAY_KHE_SRS_EXPR_INT_SEQ_COMB free_expr_int_seq_comb;
  ARRAY_KHE_SRS_STEP		free_steps;
  ARRAY_KHE_SRS_PARTIAL_SOLN	free_partial_solns;

#if TESTING
  KHE_TIMER			timer;
  HA_ARRAY_FLOAT		running_time_per_day;
#endif
};


/*****************************************************************************/
/*                                                                           */
/*  Submodule "deviations and costs"                                         */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  int KheSrsSumDev(int min_limit, int max_limit, bool allow_zero,          */
/*    int history_after, int val)                                            */
/*                                                                           */
/*  Deviation function to use with sum nodes; it has allow_zero.             */
/*                                                                           */
/*****************************************************************************/

static int KheSrsSumDev(int min_limit, int max_limit, bool allow_zero,
  int history_after, int val)
{
  if( val == 0 && allow_zero )
    return 0;
  else if( val > max_limit )
    return val - max_limit;
  else if( val + history_after < min_limit )
    return min_limit - (val + history_after);
  else
    return 0;
}


/*****************************************************************************/
/*                                                                           */
/*  int KheSrsSeqDev(int min_limit, int max_limit,                           */
/*    int history_after, int val)                                            */
/*                                                                           */
/*  Deviation function to use with sequence nodes; it has no allow_zero.     */
/*                                                                           */
/*****************************************************************************/

static int KheSrsSeqDev(int min_limit, int max_limit,
  int history_after, int val)
{
  if( val > max_limit )
    return val - max_limit;
  else if( val + history_after < min_limit )
    return min_limit - (val + history_after);
  else
    return 0;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_COST KheSrsCost(KHE_COST_FUNCTION cost_fn, KHE_COST combined_weight, */
/*    int dev)                                                               */
/*                                                                           */
/*****************************************************************************/

static KHE_COST KheSrsCost(KHE_COST_FUNCTION cost_fn, KHE_COST combined_weight,
  int dev)
{
  switch( cost_fn )
  {
    case KHE_STEP_COST_FUNCTION:

        return dev > 0 ? combined_weight : 0;

    case KHE_LINEAR_COST_FUNCTION:

        return dev * combined_weight;

    case KHE_QUADRATIC_COST_FUNCTION:

	return dev * dev * combined_weight;

    default:

	HnAbort("KheSrsCost internal error");
	return KheCost(0, 0);  /* keep compiler happy */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_SRS_DOM_TEST"                                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_DOM_TEST KheSrsDomTestMake(KHE_SRS_DOM_TYPE type)                */
/*                                                                           */
/*  Make and return a new dominance test object with no min limit attribute. */
/*                                                                           */
/*****************************************************************************/

static KHE_SRS_DOM_TEST KheSrsDomTestMake(KHE_SRS_DOM_TYPE type)
{
  KHE_SRS_DOM_TEST res;
  res.type = type;
  res.min_limit = -1;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_DOM_TEST KheSrsDomTestMakeLoose(KHE_SRS_DOM_TYPE type, int val)  */
/*                                                                           */
/*  Make and return a new dominance test object with a min limit attribute.  */
/*                                                                           */
/*****************************************************************************/

static KHE_SRS_DOM_TEST KheSrsDomTestMakeLoose(KHE_SRS_DOM_TYPE type, int val)
{
  KHE_SRS_DOM_TEST res;
  res.type = type;
  res.min_limit = val;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheSrsDomTestDominates(KHE_SRS_DOM_TEST dt, int val1, int val2)     */
/*                                                                           */
/*  Return true if val1 dominates val2, according to dt.                     */
/*                                                                           */
/*****************************************************************************/

static bool KheSrsDomTestDominates(KHE_SRS_DOM_TEST dt, int val1, int val2)
{
  switch( dt.type )
  {
    case KHE_SRS_DOM_UNUSED:

      /* should never be called */
      HnAbort("KheSrsDomTestDominates internal error (KHE_SRS_DOM_UNUSED)");
      return false;  /* keep compiler happy */

    case KHE_SRS_DOM_GE:

      /* tight min limit */
      return val1 >= val2;

    case KHE_SRS_DOM_LE:

      /* max limit */
      return val1 <= val2;

    case KHE_SRS_DOM_EQ:

      /* values must be equal */
      return val1 == val2;

    case KHE_SRS_DOM_GE_LOOSE:

      /* loose min limit */
      return val1 >= val2 || val1 >= dt.min_limit;

    case KHE_SRS_DOM_EQ_LOOSE:

      /* max and loose min; both conditions must hold (basically equality) */
      return (val1 <= val2) && (val1 >= val2 || val1 >= dt.min_limit);

    default:

      HnAbort("KheSrsDomTestDominates internal error (%d)", dt.type);
      return false;  /* keep compiler happy */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheSrsDomTestShow(KHE_SRS_DOM_TEST dt, char buff[50])              */
/*                                                                           */
/*  Show dt, using buff for scratch storage.                                 */
/*                                                                           */
/*****************************************************************************/

static char *KheSrsDomTestShow(KHE_SRS_DOM_TEST dt, char buff[50])
{
  switch( dt.type )
  {
    case KHE_SRS_DOM_UNUSED:

      sprintf(buff, "UNUSED");
      break;

    case KHE_SRS_DOM_GE:

      sprintf(buff, ">=");
      break;

    case KHE_SRS_DOM_LE:

      sprintf(buff, "<=");
      break;

    case KHE_SRS_DOM_EQ:

      sprintf(buff, "==");
      break;

    case KHE_SRS_DOM_GE_LOOSE:

      sprintf(buff, ">=(%d)", dt.min_limit);
      break;

    case KHE_SRS_DOM_EQ_LOOSE:

      sprintf(buff, "=(%d)", dt.min_limit);
      break;

    default:

      sprintf(buff, "?");
      break;
  }
  return buff;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "Helper code for expression types"                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KheSrsExprInit(res, tg, dt, srs, free_list)                              */
/*                                                                           */
/*  Shared generic code for obtaining an expression object from a free list  */
/*  in srs and initializing those fields common to all expressions.          */
/*                                                                           */
/*****************************************************************************/

#define KheSrsExprInit(res, tg, dt, srs, free_list)			\
  if( HaArrayCount(srs->free_list) > 0 )				\
  {									\
    res = HaArrayLastAndDelete(srs->free_list);				\
    HaArrayClear(res->children);					\
    HaArrayClear(res->sig_indexes);					\
  }									\
  else									\
  {									\
    HaMake(res, srs->arena);						\
    HaArrayInit(res->children, srs->arena);				\
    HaArrayInit(res->sig_indexes, srs->arena);				\
  }									\
  res->tag = tg;							\
  res->parent = NULL;							\
  res->first_day_index = res->last_day_index = UNDEF_INT;		\
  res->dom_test = dt


/*****************************************************************************/
/*                                                                           */
/*  Forward declarations of expression functions.                            */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprReceiveValueLeaf(KHE_SRS_EXPR e, KHE_TASK task,
  KHE_SINGLE_RESOURCE_SOLVER srs);
static void KheSrsExprReceiveValueInt(KHE_SRS_EXPR e, int value,
  KHE_SINGLE_RESOURCE_SOLVER srs);
static void KheSrsExprReceiveValueFloat(KHE_SRS_EXPR e, float value,
  KHE_SINGLE_RESOURCE_SOLVER srs);
static void KheSrsExprReceiveValueCost(KHE_SRS_EXPR e, KHE_COST value,
  KHE_SINGLE_RESOURCE_SOLVER srs);
static void KheSrsExprDelete(KHE_SRS_EXPR e, KHE_SINGLE_RESOURCE_SOLVER srs);
static void KheSrsExprDebug(KHE_SRS_EXPR e, KHE_SINGLE_RESOURCE_SOLVER srs,
  int verbosity, int indent, FILE *fp);


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsAddExprToTime(KHE_SINGLE_RESOURCE_SOLVER srs,                 */
/*    KHE_SRS_EXPR e, KHE_TIME time)                                         */
/*                                                                           */
/*  Add e to the list of expressions monitored by time.                      */
/*                                                                           */
/*****************************************************************************/

static void KheSrsAddExprToTime(KHE_SINGLE_RESOURCE_SOLVER srs,
  KHE_SRS_EXPR e, KHE_TIME time)
{
  KHE_SRS_TIME t;  KHE_SRS_DAY day;  int i;

  /* find the day */
  day = HaArray(srs->days, KheFrameTimeIndex(srs->days_frame, time));

  /* find the time, add e to it, and return */
  HaArrayForEach(day->times, t, i)
    if( t->time == time )
    {
      HaArrayAddLast(t->leaf_exprs, e);
      return;
    }

  /* error if we get to this point */
  HnAbort("KheSrsAddExprToTime internal error (time %s)", KheTimeId(time));
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_DOM_TEST KheExprMonotoneDomTest(KHE_SRS_EXPR e)                  */
/*                                                                           */
/*  Return a dom test suitable to a node whose function is monotone          */
/*  increasing in its arguments, and which has a parent.                     */
/*                                                                           */
/*****************************************************************************/

/* *** decided agains this kind of general case analysis
static KHE_SRS_DOM_TEST KheExprMonotoneDomTest(KHE_SRS_EXPR e)
{
  ** still to do: is this right? (probably not) **
  HnAssert(e->parent != NULL, "KheExprMonotoneDomTest internal error");
  return KheSrsDomTestMake(e->parent->dom_test.type, 0);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  int KheSrsExprIntBeginDay(KHE_SRS_EXPR e, int day_index, int init_val,   */
/*    KHE_SRS_PARTIAL_SOLN prev_ps, KHE_SINGLE_RESOURCE_SOLVER srs)          */
/*                                                                           */
/*  For expressions e whose value is an integer, return the value at the     */
/*  beginning of day day_index.  If this is e's first day, the value         */
/*  returned is init_val.  Otherwise the value returned is the value         */
/*  stored in prev_ps at the end of the previous day.                        */
/*                                                                           */
/*****************************************************************************/

static int KheSrsExprIntBeginDay(KHE_SRS_EXPR e, int day_index, int init_val,
  KHE_SRS_PARTIAL_SOLN prev_ps, KHE_SINGLE_RESOURCE_SOLVER srs, char *label)
{
  int sig_index;
  HnAssert(e->first_day_index <= day_index && day_index <= e->last_day_index,
    "KheSrsExprIntBeginDay %s internal error (%d not in range %d..%d)",
    label, day_index, e->first_day_index, e->last_day_index);
  if( day_index == e->first_day_index )
  {
    /* this is the start of e's first day */
    return init_val;
  }
  else
  {
    /* this is not e's first day; get previous day's value from prev_ps */
    sig_index = HaArray(e->sig_indexes, day_index - 1 - e->first_day_index);
    return HaArray(prev_ps->signature, sig_index);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  float KheSrsExprFloatBeginDay(KHE_SRS_EXPR e, int day_index,             */
/*    float init_val, KHE_SRS_PARTIAL_SOLN prev_ps,                          */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  For expressions e whose value is a float, return the value at the        */
/*  beginning of day day_index.  If this is e's first day, the value         */
/*  returned is init_val.  Otherwise the value returned is the value         */
/*  stored in prev_ps at the end of the previous day.                        */
/*                                                                           */
/*****************************************************************************/

static float KheSrsExprFloatBeginDay(KHE_SRS_EXPR e, int day_index,
  float init_val, KHE_SRS_PARTIAL_SOLN prev_ps, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  int sig_index;
  HnAssert(e->first_day_index <= day_index && day_index <= e->last_day_index,
    "KheSrsExprFloatBeginDay internal error");
  if( day_index == e->first_day_index )
  {
    /* this is the start of e's first day */
    return init_val;
  }
  else
  {
    /* this is not e's first day; get previous day's value from prev_ps */
    sig_index = HaArray(e->sig_indexes, day_index - 1 - e->first_day_index);
    return (float) HaArray(prev_ps->signature, sig_index) / 100.0;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  int KheSrsExprIntEndDay(KHE_SRS_EXPR e, int day_index, int final_val,    */
/*    KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs)               */
/*                                                                           */
/*  For expressions e whose value is an integer, final_val is the final      */
/*  value of e on day day_index, which is now ending.  If this is e's last   */
/*  day, report final_val to the parent.  If this is not e's last day,       */
/*  store final_val in ps.                                                   */
/*                                                                           */
/*  The return value is always UNDEF_INT.  This is for the convenience of    */
/*  the callers.  The UNDEF_INT is used to replace final_val in e.           */
/*                                                                           */
/*****************************************************************************/

static int KheSrsExprIntEndDay(KHE_SRS_EXPR e, int day_index, int final_val,
  KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  int sig_index;
  HnAssert(e->first_day_index <= day_index && day_index <= e->last_day_index,
    "KheSrsExprIntEndDay internal error 1");
  HnAssert(final_val >= 0, "KheSrsExprIntEndDay internal error 2");
  if( day_index == e->last_day_index )
  {
    /* this is the end of e's last day; report final_val to parent */
    KheSrsExprReceiveValueInt(e->parent, final_val, srs);
  }
  else
  {
    /* this is the end of a day other than e's last; store final_val in ps */
    sig_index = HaArray(e->sig_indexes, day_index - e->first_day_index);
    HnAssert(HaArrayCount(ps->signature) == sig_index,
      "KheSrsExprIntEndDay internal error 3");
    HaArrayAddLast(ps->signature, final_val);
  }
  return UNDEF_INT;
}


/*****************************************************************************/
/*                                                                           */
/*  int KheSrsExprFloatEndDay(KHE_SRS_EXPR e, int day_index,                 */
/*    int final_val, KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs)*/
/*                                                                           */
/*  For expressions e whose value is an integer, final_val is the final      */
/*  value of e on day day_index, which is now ending.  If this is e's last   */
/*  day, report final_val to the parent if report is true.  If this is not   */
/*  e's last day, store final_val in ps.                                     */
/*                                                                           */
/*  The return value is always UNDEF_FLOAT.  This is for the convenience of  */
/*  the callers.  The UNDEF_FLOAT is used to replace final_val in e.         */
/*                                                                           */
/*****************************************************************************/

static float KheSrsExprFloatEndDay(KHE_SRS_EXPR e, int day_index,
  float final_val, KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  int sig_index;
  HnAssert(e->first_day_index <= day_index && day_index <= e->last_day_index,
    "KheSrsExprFloatEndDay internal error 1");
  HnAssert(final_val >= 0.0, "KheSrsExprFloatEndDay internal error 2");
  if( day_index == e->last_day_index )
  {
    /* this is the end of e's last day; report final_val to parent */
    KheSrsExprReceiveValueFloat(e->parent, final_val, srs);
  }
  else
  {
    /* this is the end of a day other than e's last; store final_val in ps */
    sig_index = HaArray(e->sig_indexes, day_index - e->first_day_index);
    HnAssert(HaArrayCount(ps->signature) == sig_index,
      "KheSrsExprFloatEndDay internal error 3");
    HaArrayAddLast(ps->signature, (int) (final_val * 100.0));
  }
  return -1;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprIntHeaderDebug(KHE_SRS_EXPR e, char *label,               */
/*    KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs,               */
/*    int verbosity, FILE *fp)                                               */
/*                                                                           */
/*  Debug print of the header part of e in ps onto fp.                       */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprIntHeaderDebug(KHE_SRS_EXPR e, char *label,
  int tmp_value, KHE_SRS_PARTIAL_SOLN ps,
  KHE_SINGLE_RESOURCE_SOLVER srs, int verbosity, FILE *fp)
{
  int sig_index;
  sig_index = HaArray(e->sig_indexes, ps->day_index - e->first_day_index);
  fprintf(fp, "%s %d", label, HaArray(ps->signature, sig_index));
  if( tmp_value != -1 )
    fprintf(fp, "$%d", tmp_value);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprDebugDayRange(KHE_SRS_EXPR e,                             */
/*    KHE_SINGLE_RESOURCE_SOLVER srs, FILE *fp)                              */
/*                                                                           */
/*  Debug print of the day range of e onto fp.                               */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprDebugDayRange(KHE_SRS_EXPR e,
  KHE_SINGLE_RESOURCE_SOLVER srs, FILE *fp)
{
  KHE_TIME_GROUP tg1, tg2;
  if( e->first_day_index == e->last_day_index )
  {
    tg1 = KheFrameTimeGroup(srs->days_frame, e->first_day_index);
    fprintf(fp, "%s", KheTimeGroupId(tg1));
  }
  else
  {
    tg1 = KheFrameTimeGroup(srs->days_frame, e->first_day_index);
    tg2 = KheFrameTimeGroup(srs->days_frame, e->last_day_index);
    fprintf(fp, "%s-%s", KheTimeGroupId(tg1), KheTimeGroupId(tg2));
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprIntDebug(KHE_SRS_EXPR e, char *label, int val,            */
/*    KHE_SINGLE_RESOURCE_SOLVER srs, int verbosity, int indent, FILE *fp)   */
/*                                                                           */
/*  Debug e, for expression types that store an integer value.  The value    */
/*  is passed to this function in val.                                       */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprIntDebug(KHE_SRS_EXPR e, char *label, int val,
  KHE_SINGLE_RESOURCE_SOLVER srs, int verbosity, int indent, FILE *fp)
{
  KHE_SRS_EXPR child_e;  int i;  char buff[50];
  if( indent >= 0 )
  {
    fprintf(fp, "%*s[ ", indent, "");
    KheSrsExprDebugDayRange(e, srs, fp);
    fprintf(fp, " %s", KheSrsDomTestShow(e->dom_test, buff));
    fprintf(fp, " %s", label);
    if( val != UNDEF_INT )
      fprintf(fp, " (val %d)", val);
    fprintf(fp, "\n");
    HaArrayForEach(e->children, child_e, i)
      KheSrsExprDebug(child_e, srs, verbosity, indent + 2, fp);
    fprintf(fp, "%*s]\n", indent, "");
  }
  else
    fprintf(fp, "%s(%d children)", label, HaArrayCount(e->children));
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprFloatHeaderDebug(KHE_SRS_EXPR e, char *label,             */
/*    float tmp_value, KHE_SRS_PARTIAL_SOLN ps,                              */
/*    KHE_SINGLE_RESOURCE_SOLVER srs, int verbosity, FILE *fp)               */
/*                                                                           */
/*  Debug print of the header part of e in ps onto fp.                       */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprFloatHeaderDebug(KHE_SRS_EXPR e, char *label,
  float tmp_value, KHE_SRS_PARTIAL_SOLN ps,
  KHE_SINGLE_RESOURCE_SOLVER srs, int verbosity, FILE *fp)
{
  int sig_index;
  sig_index = HaArray(e->sig_indexes, ps->day_index - e->first_day_index);
  fprintf(fp, "%s %.2f", label,
    (float) HaArray(ps->signature, sig_index) / 100.0);
  if( tmp_value != -1.0 )
    fprintf(fp, "$%.2f", tmp_value);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprFloatDebug(KHE_SRS_EXPR e, char *label, float val,        */
/*    KHE_SINGLE_RESOURCE_SOLVER srs, int verbosity, int indent, FILE *fp)   */
/*                                                                           */
/*  Debug e, for expression types that store a float value.  The value is    */
/*  passed to this function in val.                                          */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprFloatDebug(KHE_SRS_EXPR e, char *label, float val,
  KHE_SINGLE_RESOURCE_SOLVER srs, int verbosity, int indent, FILE *fp)
{
  KHE_SRS_EXPR child_e;  int i;
  if( indent >= 0 )
  {
    fprintf(fp, "%*s[ ", indent, "");
    KheSrsExprDebugDayRange(e, srs, fp);
    fprintf(fp, " %s", label);
    if( val != UNDEF_FLOAT )
      fprintf(fp, " (val %.2f)", val);
    fprintf(fp, "\n");
    HaArrayForEach(e->children, child_e, i)
      KheSrsExprDebug(child_e, srs, verbosity, indent + 2, fp);
    fprintf(fp, "%*s]\n", indent, "");
  }
  else
    fprintf(fp, "%s(%d children)", label, HaArrayCount(e->children));
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_SRS_EXPR_BUSY"                                            */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_EXPR_BUSY KheSrsExprBusyMake(KHE_TIME time,                      */
/*    KHE_SRS_DOM_TEST dom_test, KHE_SINGLE_RESOURCE_SOLVER srs)             */
/*                                                                           */
/*  Make a new KHE_SRS_EXPR_BUSY object with these attributes.               */
/*                                                                           */
/*****************************************************************************/

static KHE_SRS_EXPR_BUSY KheSrsExprBusyMake(KHE_TIME time,
  KHE_SRS_DOM_TEST dom_test, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_EXPR_BUSY res;
  KheSrsExprInit(res, KHE_SRS_EXPR_BUSY_TAG, dom_test, srs, free_expr_busy);
  res->time = time;
  res->tmp_value = UNDEF_INT;
  KheSrsAddExprToTime(srs, (KHE_SRS_EXPR) res, time);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprBusyDelete(KHE_SRS_EXPR_BUSY eb,                          */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Delete eb and its descendants (there are no descendants here).           */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprBusyDelete(KHE_SRS_EXPR_BUSY eb,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  HaArrayAddLast(srs->free_expr_busy, eb);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_DOM_TEST KheSrsExprBusyDomTest(KHE_SRS_EXPR_BUSY eb)             */
/*                                                                           */
/*  Return a suitable dom test for eb.                                       */
/*                                                                           */
/*****************************************************************************/

/* ***
static KHE_SRS_DOM_TEST KheSrsExprBusyDomTest(KHE_SRS_EXPR_BUSY eb)
{
  return KheExprMonotoneDomTest((KHE_SRS_EXPR) eb);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprBusyBeginDay(KHE_SRS_EXPR_BUSY eb, int day_index,         */
/*    KHE_SRS_PARTIAL_SOLN prev_ps, KHE_SINGLE_RESOURCE_SOLVER srs)          */
/*                                                                           */
/*  Inform eb that day day_index is beginning.                               */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprBusyBeginDay(KHE_SRS_EXPR_BUSY eb, int day_index,
  KHE_SRS_PARTIAL_SOLN prev_ps, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  if( DEBUG8 )
    fprintf(stderr, "ExprBusyBeginDay(time %s)\n", KheTimeId(eb->time));
  eb->tmp_value = KheSrsExprIntBeginDay((KHE_SRS_EXPR) eb, day_index,
    0, prev_ps, srs, "Busy");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprBusyReceiveValueLeaf(KHE_SRS_EXPR_BUSY eb,                */
/*    KHE_TASK task, KHE_SINGLE_RESOURCE_SOLVER srs)                         */
/*                                                                           */
/*  Inform eb that the resource is busy, assigned to task, at its time.      */
/*  Or if task == NULL, the resource is free at its time.                    */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprBusyReceiveValueLeaf(KHE_SRS_EXPR_BUSY eb,
  KHE_TASK task, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  eb->tmp_value = (task != NULL ? 1 : 0);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprBusyEndDay(KHE_SRS_EXPR_BUSY eb, int day_index,           */
/*    KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs)               */
/*                                                                           */
/*  Inform eb that day day_index is ending.                                  */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprBusyEndDay(KHE_SRS_EXPR_BUSY eb, int day_index,
  KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  eb->tmp_value = KheSrsExprIntEndDay((KHE_SRS_EXPR) eb, day_index,
    eb->tmp_value, ps, srs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprBusyHeaderDebug(KHE_SRS_EXPR_BUSY eb,                     */
/*    KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs,               */
/*    int verbosity, FILE *fp)                                               */
/*                                                                           */
/*  Debug print of the header part of eb in ps onto fp.                      */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprBusyHeaderDebug(KHE_SRS_EXPR_BUSY eb,
  KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs,
  int verbosity, FILE *fp)
{
  KheSrsExprIntHeaderDebug((KHE_SRS_EXPR) eb, "BUSY", eb->tmp_value,
    ps, srs, verbosity, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprBusyDebug(KHE_SRS_EXPR_BUSY eb,                           */
/*    KHE_SINGLE_RESOURCE_SOLVER srs, int verbosity, int indent, FILE *fp)   */
/*                                                                           */
/*  Debug print of eb onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprBusyDebug(KHE_SRS_EXPR_BUSY eb,
  KHE_SINGLE_RESOURCE_SOLVER srs, int verbosity, int indent, FILE *fp)
{
  char buff[50];
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  fprintf(fp, "[ BUSY %s(%s) ]", KheSrsDomTestShow(eb->dom_test, buff),
      KheTimeId(eb->time));
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_SRS_EXPR_FREE"                                            */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_EXPR_FREE KheSrsExprFreeMake(KHE_TIME time,                      */
/*    KHE_SRS_DOM_TEST dom_test, KHE_SINGLE_RESOURCE_SOLVER srs)             */
/*                                                                           */
/*  Make a new KHE_SRS_EXPR_FREE object with these attributes.               */
/*                                                                           */
/*****************************************************************************/

static KHE_SRS_EXPR_FREE KheSrsExprFreeMake(KHE_TIME time,
  KHE_SRS_DOM_TEST dom_test, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_EXPR_FREE res;
  KheSrsExprInit(res, KHE_SRS_EXPR_FREE_TAG, dom_test, srs, free_expr_free);
  res->time = time;
  res->tmp_value = UNDEF_INT;
  KheSrsAddExprToTime(srs, (KHE_SRS_EXPR) res, time);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprFreeDelete(KHE_SRS_EXPR_FREE ef,                          */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Delete ef and its descendants (there are no descendants here).           */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprFreeDelete(KHE_SRS_EXPR_FREE ef,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  HaArrayAddLast(srs->free_expr_free, ef);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_DOM_TEST KheSrsExprFreeDomTest(KHE_SRS_EXPR_FREE ef)             */
/*                                                                           */
/*  Return a suitable dom test for ef.                                       */
/*                                                                           */
/*****************************************************************************/

/* *** no longer doing it this way
static KHE_SRS_DOM_TEST KheSrsExprFreeDomTest(KHE_SRS_EXPR_FREE ef)
{
  return KheExprMonotoneDomTest((KHE_SRS_EXPR) ef);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprFreeBeginDay(KHE_SRS_EXPR_FREE ef, int day_index,         */
/*    KHE_SRS_PARTIAL_SOLN prev_ps, KHE_SINGLE_RESOURCE_SOLVER srs)          */
/*                                                                           */
/*  Inform ef that day day_index is beginning.                               */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprFreeBeginDay(KHE_SRS_EXPR_FREE ef, int day_index,
  KHE_SRS_PARTIAL_SOLN prev_ps, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  if( DEBUG8 )
    fprintf(stderr, "ExprFreeBeginDay(time %s)\n", KheTimeId(ef->time));
  ef->tmp_value = KheSrsExprIntBeginDay((KHE_SRS_EXPR) ef, day_index,
    0, prev_ps, srs, "Free");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprFreeReceiveValueLeaf(KHE_SRS_EXPR_FREE ef,                */
/*    KHE_TASK task, KHE_SINGLE_RESOURCE_SOLVER srs)                         */
/*                                                                           */
/*  Inform ef that the resource is busy, assigned to task, at its time.      */
/*  Or if task == NULL, the resource is free at its time.                    */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprFreeReceiveValueLeaf(KHE_SRS_EXPR_FREE ef,
  KHE_TASK task, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  ef->tmp_value = (task == NULL ? 1 : 0);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprFreeEndDay(KHE_SRS_EXPR_FREE ef, int day_index,           */
/*    KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs)               */
/*                                                                           */
/*  Inform ef that day day_index is ending.                                  */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprFreeEndDay(KHE_SRS_EXPR_FREE ef, int day_index,
  KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  ef->tmp_value = KheSrsExprIntEndDay((KHE_SRS_EXPR) ef, day_index,
    ef->tmp_value, ps, srs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprFreeHeaderDebug(KHE_SRS_EXPR_FREE ef,                     */
/*    KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs,               */
/*    int verbosity, FILE *fp)                                               */
/*                                                                           */
/*  Debug print of the header part of ef in ps onto fp.                      */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprFreeHeaderDebug(KHE_SRS_EXPR_FREE ef,
  KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs,
  int verbosity, FILE *fp)
{
  KheSrsExprIntHeaderDebug((KHE_SRS_EXPR) ef, "FREE", ef->tmp_value,
    ps, srs, verbosity, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprFreeDebug(KHE_SRS_EXPR_FREE ef,                           */
/*    KHE_SINGLE_RESOURCE_SOLVER srs, int verbosity, int indent, FILE *fp)   */
/*                                                                           */
/*  Debug print of ef onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprFreeDebug(KHE_SRS_EXPR_FREE ef,
  KHE_SINGLE_RESOURCE_SOLVER srs, int verbosity, int indent, FILE *fp)
{
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  fprintf(fp, "[ FREE(%s) ]", KheTimeId(ef->time));
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_SRS_EXPR_WORK"                                            */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_EXPR_WORK KheSrsExprWorkMake(KHE_TIME time,                      */
/*    KHE_SRS_DOM_TEST dom_test, KHE_SINGLE_RESOURCE_SOLVER srs)             */
/*                                                                           */
/*  Make a new KHE_SRS_EXPR_WORK object with these attributes.               */
/*                                                                           */
/*****************************************************************************/

static KHE_SRS_EXPR_WORK KheSrsExprWorkMake(KHE_TIME time,
  KHE_SRS_DOM_TEST dom_test, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_EXPR_WORK res;
  KheSrsExprInit(res, KHE_SRS_EXPR_WORK_TAG, dom_test, srs, free_expr_work);
  res->time = time;
  res->tmp_value = UNDEF_FLOAT;
  KheSrsAddExprToTime(srs, (KHE_SRS_EXPR) res, time);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprWorkDelete(KHE_SRS_EXPR_WORK ew,                          */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Delete ew and its descendants (there are no descendants here).           */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprWorkDelete(KHE_SRS_EXPR_WORK ew,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  HaArrayAddLast(srs->free_expr_work, ew);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_DOM_TEST KheSrsExprWorkDomTest(KHE_SRS_EXPR_WORK ew)             */
/*                                                                           */
/*  Return a suitable dom test for ew.                                       */
/*                                                                           */
/*****************************************************************************/

/* *** no longer doing it this way
static KHE_SRS_DOM_TEST KheSrsExprWorkDomTest(KHE_SRS_EXPR_WORK ew)
{
  return KheExprMonotoneDomTest((KHE_SRS_EXPR) ew);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprWorkBeginDay(KHE_SRS_EXPR_WORK ew, int day_index,         */
/*    KHE_SRS_PARTIAL_SOLN prev_ps, KHE_SINGLE_RESOURCE_SOLVER srs)          */
/*                                                                           */
/*  Inform ew that day day_index is beginning.                               */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprWorkBeginDay(KHE_SRS_EXPR_WORK ew, int day_index,
  KHE_SRS_PARTIAL_SOLN prev_ps, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  ew->tmp_value = KheSrsExprFloatBeginDay((KHE_SRS_EXPR) ew, day_index,
    UNDEF_FLOAT, prev_ps, srs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprWorkReceiveValueLeaf(KHE_SRS_EXPR_WORK ew,                */
/*    KHE_TASK task, KHE_SINGLE_RESOURCE_SOLVER srs)                         */
/*                                                                           */
/*  Inform ew that the resource is busy, assigned to task, at its time.      */
/*  Or if task == NULL, the resource is free at its time.                    */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprWorkReceiveValueLeaf(KHE_SRS_EXPR_WORK ew,
  KHE_TASK task, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  ew->tmp_value = KheTaskWorkloadPerTime(task);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprWorkEndDay(KHE_SRS_EXPR_WORK ew, int day_index,           */
/*    KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs)               */
/*                                                                           */
/*  Inform ew that day day_index is ending.                                  */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprWorkEndDay(KHE_SRS_EXPR_WORK ew, int day_index,
  KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  ew->tmp_value = KheSrsExprFloatEndDay((KHE_SRS_EXPR) ew, day_index,
    ew->tmp_value, ps, srs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprWorkHeaderDebug(KHE_SRS_EXPR_WORK ew,                     */
/*    KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs,               */
/*    int verbosity, FILE *fp)                                               */
/*                                                                           */
/*  Debug print of the header part of ew in ps onto fp.                      */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprWorkHeaderDebug(KHE_SRS_EXPR_WORK ew,
  KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs,
  int verbosity, FILE *fp)
{
  KheSrsExprFloatHeaderDebug((KHE_SRS_EXPR) ew, "WORK", ew->tmp_value,
    ps, srs, verbosity, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprWorkDebug(KHE_SRS_EXPR_WORK ew,                           */
/*    KHE_SINGLE_RESOURCE_SOLVER srs, int verbosity, int indent, FILE *fp)   */
/*                                                                           */
/*  Debug print of ew onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprWorkDebug(KHE_SRS_EXPR_WORK ew,
  KHE_SINGLE_RESOURCE_SOLVER srs, int verbosity, int indent, FILE *fp)
{
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  fprintf(fp, "[ WORK(%s) ]", KheTimeId(ew->time));
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_SRS_EXPR_OR"                                              */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_EXPR_OR KheSrsExprOrMake(KHE_SRS_DOM_TEST dom_test,              */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Make a new KHE_SRS_EXPR_OR object with these attributes.                 */
/*                                                                           */
/*****************************************************************************/

static KHE_SRS_EXPR_OR KheSrsExprOrMake(KHE_SRS_DOM_TEST dom_test,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_EXPR_OR res;
  KheSrsExprInit(res, KHE_SRS_EXPR_OR_TAG, dom_test, srs, free_expr_or);
  res->tmp_value = UNDEF_INT;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprOrDelete(KHE_SRS_EXPR_OR eo,                              */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Delete eo and its descendants.                                           */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprOrDelete(KHE_SRS_EXPR_OR eo,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_EXPR e;  int i;
  HaArrayForEach(eo->children, e, i)
    KheSrsExprDelete(e, srs);
  HaArrayAddLast(srs->free_expr_or, eo);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_DOM_TEST KheSrsExprOrDomTest(KHE_SRS_EXPR_OR eo)                 */
/*                                                                           */
/*  Return a suitable dom test for eo.                                       */
/*                                                                           */
/*****************************************************************************/

/* *** no longer doing it this way
static KHE_SRS_DOM_TEST KheSrsExprOrDomTest(KHE_SRS_EXPR_OR eo)
{
  return KheExprMonotoneDomTest((KHE_SRS_EXPR) eo);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprOrBeginDay(KHE_SRS_EXPR_OR eo, int day_index,             */
/*    KHE_SRS_PARTIAL_SOLN prev_ps, KHE_SINGLE_RESOURCE_SOLVER srs)          */
/*                                                                           */
/*  Inform eo that day day_index is beginning.                               */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprOrBeginDay(KHE_SRS_EXPR_OR eo, int day_index,
  KHE_SRS_PARTIAL_SOLN prev_ps, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  eo->tmp_value = KheSrsExprIntBeginDay((KHE_SRS_EXPR) eo, day_index, 0,
    prev_ps, srs, "Or");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprOrReceiveValueInt(KHE_SRS_EXPR_OR eo, int value,          */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Inform eo that one of its children has just finalized its value.         */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprOrReceiveValueInt(KHE_SRS_EXPR_OR eo, int value,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  HnAssert(eo->tmp_value != UNDEF_INT, "KheSrsExprOrReceiveValueInt internal error");
  if( value == 1 )
    eo->tmp_value = 1;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprOrEndDay(KHE_SRS_EXPR_OR eo, int day_index,               */
/*    KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs)               */
/*                                                                           */
/*  Inform eo that day day_index is ending.                                  */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprOrEndDay(KHE_SRS_EXPR_OR eo, int day_index,
  KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  eo->tmp_value = KheSrsExprIntEndDay((KHE_SRS_EXPR) eo, day_index,
    eo->tmp_value, ps, srs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprOrHeaderDebug(KHE_SRS_EXPR_OR eo,                         */
/*    KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs,               */
/*    int verbosity, FILE *fp)                                               */
/*                                                                           */
/*  Debug print of the header part of eo in ps onto fp.                      */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprOrHeaderDebug(KHE_SRS_EXPR_OR eo,
  KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs,
  int verbosity, FILE *fp)
{
  KheSrsExprIntHeaderDebug((KHE_SRS_EXPR) eo, "OR", eo->tmp_value,
    ps, srs, verbosity, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprOrDebug(KHE_SRS_EXPR_OR eo,                               */
/*    KHE_SINGLE_RESOURCE_SOLVER srs, int verbosity, int indent, FILE *fp)   */
/*                                                                           */
/*  Debug print of eo onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprOrDebug(KHE_SRS_EXPR_OR eo,
  KHE_SINGLE_RESOURCE_SOLVER srs, int verbosity, int indent, FILE *fp)
{
  KheSrsExprIntDebug((KHE_SRS_EXPR) eo, "OR", eo->tmp_value, srs,
    verbosity, indent, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_SRS_EXPR_AND"                                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_EXPR_AND KheSrsExprAndMake(KHE_SRS_DOM_TEST dom_test,            */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Make a new KHE_SRS_EXPR_AND object with these attributes.                */
/*                                                                           */
/*****************************************************************************/

static KHE_SRS_EXPR_AND KheSrsExprAndMake(KHE_SRS_DOM_TEST dom_test,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_EXPR_AND res;
  KheSrsExprInit(res, KHE_SRS_EXPR_AND_TAG, dom_test, srs, free_expr_and);
  res->tmp_value = UNDEF_INT;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprAndDelete(KHE_SRS_EXPR_AND ea,                            */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Delete ea and its descendants.                                           */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprAndDelete(KHE_SRS_EXPR_AND ea,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_EXPR e;  int i;
  HaArrayForEach(ea->children, e, i)
    KheSrsExprDelete(e, srs);
  HaArrayAddLast(srs->free_expr_and, ea);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_DOM_TEST KheSrsExprAndDomTest(KHE_SRS_EXPR_AND ea)               */
/*                                                                           */
/*  Return a suitable dom test for ea.                                       */
/*                                                                           */
/*****************************************************************************/

/* *** no longer doing it this way
static KHE_SRS_DOM_TEST KheSrsExprAndDomTest(KHE_SRS_EXPR_AND ea)
{
  return KheExprMonotoneDomTest((KHE_SRS_EXPR) ea);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprAndBeginDay(KHE_SRS_EXPR_AND ea, int day_index,           */
/*    KHE_SRS_PARTIAL_SOLN prev_ps, KHE_SINGLE_RESOURCE_SOLVER srs)          */
/*                                                                           */
/*  Inform ea that day day_index is beginning.                               */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprAndBeginDay(KHE_SRS_EXPR_AND ea, int day_index,
  KHE_SRS_PARTIAL_SOLN prev_ps, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  ea->tmp_value = KheSrsExprIntBeginDay((KHE_SRS_EXPR) ea, day_index, 1,
    prev_ps, srs, "And");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprAndReceiveValueInt(KHE_SRS_EXPR_AND ea, int value,        */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Inform ea that one of its children has just finalized its value.         */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprAndReceiveValueInt(KHE_SRS_EXPR_AND ea, int value,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  HnAssert(ea->tmp_value != UNDEF_INT, "KheSrsExprOrReceiveValueInt internal error");
  if( value == 0 )
    ea->tmp_value = 0;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprAndEndDay(KHE_SRS_EXPR_AND ea, int day_index,             */
/*    KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs)               */
/*                                                                           */
/*  Inform ea that day day_index is ending.                                  */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprAndEndDay(KHE_SRS_EXPR_AND ea, int day_index,
  KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  ea->tmp_value = KheSrsExprIntEndDay((KHE_SRS_EXPR) ea, day_index,
    ea->tmp_value, ps, srs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprAndHeaderDebug(KHE_SRS_EXPR_AND ea,                       */
/*    KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs,               */
/*    int verbosity, FILE *fp)                                               */
/*                                                                           */
/*  Debug print of the header part of ea in ps onto fp.                      */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprAndHeaderDebug(KHE_SRS_EXPR_AND ea,
  KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs,
  int verbosity, FILE *fp)
{
  KheSrsExprIntHeaderDebug((KHE_SRS_EXPR) ea, "AND", ea->tmp_value,
    ps, srs, verbosity, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprAndDebug(KHE_SRS_EXPR_AND ea,                             */
/*    KHE_SINGLE_RESOURCE_SOLVER srs, int verbosity, int indent, FILE *fp)   */
/*                                                                           */
/*  Debug print of ea onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprAndDebug(KHE_SRS_EXPR_AND ea,
  KHE_SINGLE_RESOURCE_SOLVER srs, int verbosity, int indent, FILE *fp)
{
  KheSrsExprIntDebug((KHE_SRS_EXPR) ea, "AND", ea->tmp_value, srs,
    verbosity, indent, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_SRS_EXPR_INT_SUM"                                         */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_EXPR_INT_SUM KheSrsExprIntSumMake(KHE_SRS_DOM_TEST dom_test,     */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Make a new KHE_SRS_EXPR_INT_SUM object with these attributes.            */
/*                                                                           */
/*****************************************************************************/

static KHE_SRS_EXPR_INT_SUM KheSrsExprIntSumMake(KHE_SRS_DOM_TEST dom_test,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_EXPR_INT_SUM res;
  KheSrsExprInit(res, KHE_SRS_EXPR_INT_SUM_TAG, dom_test, srs,
    free_expr_int_sum);
  /* res->history_before = history_before; */
  /* res->history_after = history_after; */
  res->tmp_value = UNDEF_INT;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprIntSumDelete(KHE_SRS_EXPR_INT_SUM eis,                    */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Delete eis and its descendants.                                          */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprIntSumDelete(KHE_SRS_EXPR_INT_SUM eis,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_EXPR e;  int i;
  HaArrayForEach(eis->children, e, i)
    KheSrsExprDelete(e, srs);
  HaArrayAddLast(srs->free_expr_int_sum, eis);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_DOM_TEST KheSrsExprIntSumDomTest(KHE_SRS_EXPR_INT_SUM eis)       */
/*                                                                           */
/*  Return a suitable dom test for eis.                                      */
/*                                                                           */
/*****************************************************************************/

/* *** no longer doing it this way
static KHE_SRS_DOM_TEST KheSrsExprIntSumDomTest(KHE_SRS_EXPR_INT_SUM eis)
{
  return KheExprMonotoneDomTest((KHE_SRS_EXPR) eis);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprIntSumBeginDay(KHE_SRS_EXPR_INT_SUM eis, int day_index,   */
/*    KHE_SRS_PARTIAL_SOLN prev_ps, KHE_SINGLE_RESOURCE_SOLVER srs)          */
/*                                                                           */
/*  Inform eis that day day_index is beginning.                              */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprIntSumBeginDay(KHE_SRS_EXPR_INT_SUM eis, int day_index,
  KHE_SRS_PARTIAL_SOLN prev_ps, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  eis->tmp_value = KheSrsExprIntBeginDay((KHE_SRS_EXPR) eis, day_index,
    /* eis->history_before */ 0, prev_ps, srs, "IntSum");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprIntSumReceiveValueInt(KHE_SRS_EXPR_INT_SUM eis,           */
/*    int value, KHE_SINGLE_RESOURCE_SOLVER srs)                             */
/*                                                                           */
/*  Inform eis that one of its children has just finalized its value.        */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprIntSumReceiveValueInt(KHE_SRS_EXPR_INT_SUM eis,
  int value, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  eis->tmp_value += value;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprIntSumEndDay(KHE_SRS_EXPR_INT_SUM eis, int day_index,     */
/*    KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs)               */
/*                                                                           */
/*  Inform eis that day day_index is ending.                                 */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprIntSumEndDay(KHE_SRS_EXPR_INT_SUM eis, int day_index,
  KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  eis->tmp_value = KheSrsExprIntEndDay((KHE_SRS_EXPR) eis, day_index,
    eis->tmp_value, ps, srs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprIntSumHeaderDebug(KHE_SRS_EXPR_INT_SUM eis,               */
/*    KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs,               */
/*    int verbosity, FILE *fp)                                               */
/*                                                                           */
/*  Debug print of the header part of eis in ps onto fp.                     */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprIntSumHeaderDebug(KHE_SRS_EXPR_INT_SUM eis,
  KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs,
  int verbosity, FILE *fp)
{
  KheSrsExprIntHeaderDebug((KHE_SRS_EXPR) eis, "INT_SUM", eis->tmp_value,
    ps, srs, verbosity, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprIntSumDebug(KHE_SRS_EXPR_INT_SUM eis,                     */
/*    KHE_SINGLE_RESOURCE_SOLVER srs, int verbosity, int indent, FILE *fp)   */
/*                                                                           */
/*  Debug print of eis onto fp with the given verbosity and indent.          */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprIntSumDebug(KHE_SRS_EXPR_INT_SUM eis,
  KHE_SINGLE_RESOURCE_SOLVER srs, int verbosity, int indent, FILE *fp)
{
  KheSrsExprIntDebug((KHE_SRS_EXPR) eis, "INT_SUM", eis->tmp_value,
    srs, verbosity, indent, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_SRS_EXPR_INT_SEQ"                                         */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_EXPR_INT_SEQ KheSrsExprIntSeqMake(int history_before,            */
/*    int history_after, KHE_SRS_DOM_TEST dom_test,                          */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Make a new KHE_SRS_EXPR_INT_SEQ object with these attributes.            */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused; only KHE_SRS_EXPR_INT_SEQ_COMB_TAG is used
static KHE_SRS_EXPR_INT_SEQ KheSrsExprIntSeqMake(int history_before,
  int history_after, KHE_SRS_DOM_TEST dom_test, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_EXPR_INT_SEQ res;
  KheSrsExprInit(res, KHE_SRS_EXPR_INT_SEQ_TAG, dom_test, srs,
    free_expr_int_seq);
  res->history_before = history_before;
  res->history_after = history_after;
  res->tmp_value = UNDEF_INT;
  return res;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprIntSeqDelete(KHE_SRS_EXPR_INT_SEQ eis,                    */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Delete eis and its descendants.                                          */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused; only KHE_SRS_EXPR_INT_SEQ_COMB_TAG is used
static void KheSrsExprIntSeqDelete(KHE_SRS_EXPR_INT_SEQ eis,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_EXPR e;  int i;
  HaArrayForEach(eis->children, e, i)
    KheSrsExprDelete(e, srs);
  HaArrayAddLast(srs->free_expr_int_seq, eis);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_DOM_TEST KheSrsExprIntSeqDomTest(KHE_SRS_EXPR_INT_SEQ eis)       */
/*                                                                           */
/*  Return a suitable dom test for eis.                                      */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused; only KHE_SRS_EXPR_INT_SEQ_COMB_TAG is used
static KHE_SRS_DOM_TEST KheSrsExprIntSeqDomTest(KHE_SRS_EXPR_INT_SEQ eis)
{
  return KheExprMonotoneDomTest((KHE_SRS_EXPR) eis);  ** is this right? **
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprIntSeqBeginDay(KHE_SRS_EXPR_INT_SEQ eis, int day_index,   */
/*    KHE_SRS_PARTIAL_SOLN prev_ps, KHE_SINGLE_RESOURCE_SOLVER srs)          */
/*                                                                           */
/*  Inform eis that day day_index is beginning.                              */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused; only KHE_SRS_EXPR_INT_SEQ_COMB_TAG is used
static void KheSrsExprIntSeqBeginDay(KHE_SRS_EXPR_INT_SEQ eis, int day_index,
  KHE_SRS_PARTIAL_SOLN prev_ps, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  eis->tmp_value = KheSrsExprIntBeginDay((KHE_SRS_EXPR) eis, day_index,
    eis->history_before, prev_ps, srs, "IntSeq");
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprIntSeqReceiveValueInt(KHE_SRS_EXPR_INT_SEQ eis,           */
/*    int value, KHE_SINGLE_RESOURCE_SOLVER srs)                             */
/*                                                                           */
/*  Inform eis that one of its children has just finalized its value.        */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused; only KHE_SRS_EXPR_INT_SEQ_COMB_TAG is used
static void KheSrsExprIntSeqReceiveValueInt(KHE_SRS_EXPR_INT_SEQ eis,
  int value, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  HnAssert(value == 0 || value == 1,
    "KheSrsExprIntSeqReceiveValueInt internal error");
  if( value == 1 )
  {
    ** a sequence is beginning or continuing **
    eis->tmp_value++;
  }
  else if( eis->tmp_value > 0 )
  {
    ** a non-empty sequence has just ended **
    KheSrsExprReceiveValueInt(eis->parent, eis->tmp_value, srs);
    eis->tmp_value = 0;
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprIntSeqEndDay(KHE_SRS_EXPR_INT_SEQ eis, int day_index,     */
/*    KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs)               */
/*                                                                           */
/*  Inform eis that day day_index is ending.                                 */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused; only KHE_SRS_EXPR_INT_SEQ_COMB_TAG is used
static void KheSrsExprIntSeqEndDay(KHE_SRS_EXPR_INT_SEQ eis, int day_index,
  KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  eis->tmp_value = KheSrsExprIntEndDay((KHE_SRS_EXPR) eis, day_index,
    ** eis->tmp_value > 0, ** eis->tmp_value, ps, srs);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprIntSeqDebug(KHE_SRS_EXPR_INT_SEQ eis,                     */
/*    KHE_SINGLE_RESOURCE_SOLVER srs, int verbosity, int indent, FILE *fp)   */
/*                                                                           */
/*  Debug print of eis onto fp with the given verbosity and indent.          */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused; only KHE_SRS_EXPR_INT_SEQ_COMB_TAG is used
static void KheSrsExprIntSeqDebug(KHE_SRS_EXPR_INT_SEQ eis,
  KHE_SINGLE_RESOURCE_SOLVER srs, int verbosity, int indent, FILE *fp)
{
  KheSrsExprIntDebug((KHE_SRS_EXPR) eis, "INT_SEQ", eis->tmp_value,
    srs, verbosity, indent, fp);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_SRS_EXPR_FLOAT_SUM"                                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_EXPR_FLOAT_SUM KheSrsExprFloatSumMake(                           */
/*    KHE_SRS_DOM_TEST dom_test, KHE_SINGLE_RESOURCE_SOLVER srs)             */
/*                                                                           */
/*  Make a new KHE_SRS_EXPR_FLOAT_SUM object with these attributes.          */
/*                                                                           */
/*****************************************************************************/

static KHE_SRS_EXPR_FLOAT_SUM KheSrsExprFloatSumMake(
  KHE_SRS_DOM_TEST dom_test, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_EXPR_FLOAT_SUM res;
  KheSrsExprInit(res, KHE_SRS_EXPR_FLOAT_SUM_TAG, dom_test, srs,
    free_expr_float_sum);
  res->tmp_value = UNDEF_FLOAT;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprFloatSumDelete(KHE_SRS_EXPR_FLOAT_SUM efs,                */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Delete efs and its descendants.                                          */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprFloatSumDelete(KHE_SRS_EXPR_FLOAT_SUM efs,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_EXPR e;  int i;
  HaArrayForEach(efs->children, e, i)
    KheSrsExprDelete(e, srs);
  HaArrayAddLast(srs->free_expr_float_sum, efs);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_DOM_TEST KheSrsExprFloatSumDomTest(KHE_SRS_EXPR_FLOAT_SUM efs)   */
/*                                                                           */
/*  Return a suitable dom test for efs.                                      */
/*                                                                           */
/*****************************************************************************/

/* *** no longer doing it this way
static KHE_SRS_DOM_TEST KheSrsExprFloatSumDomTest(KHE_SRS_EXPR_FLOAT_SUM efs)
{
  return KheExprMonotoneDomTest((KHE_SRS_EXPR) efs);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprFloatSumBeginDay(KHE_SRS_EXPR_FLOAT_SUM efs,              */
/*    int day_index, KHE_SRS_PARTIAL_SOLN prev_ps,                           */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Inform efs that day day_index is beginning.                              */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprFloatSumBeginDay(KHE_SRS_EXPR_FLOAT_SUM efs,
  int day_index, KHE_SRS_PARTIAL_SOLN prev_ps, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  efs->tmp_value = KheSrsExprFloatBeginDay((KHE_SRS_EXPR) efs, day_index,
    0.0, prev_ps, srs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprFloatSumReceiveValueFloat(KHE_SRS_EXPR_FLOAT_SUM efs,     */
/*    float value, KHE_SINGLE_RESOURCE_SOLVER srs)                           */
/*                                                                           */
/*  Inform efs that one of its children has just finalized its value.        */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprFloatSumReceiveValueFloat(KHE_SRS_EXPR_FLOAT_SUM efs,
  float value, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  HnAssert(efs->tmp_value >= 0.0,
    "KheSrsExprFloatSumReceiveValueFloat internal error");
  efs->tmp_value += value;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprFloatSumEndDay(KHE_SRS_EXPR_FLOAT_SUM efs,                */
/*    int day_index, KHE_SRS_PARTIAL_SOLN ps,                                */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Inform efs that day day_index is ending.                                 */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprFloatSumEndDay(KHE_SRS_EXPR_FLOAT_SUM efs,
  int day_index, KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  efs->tmp_value = KheSrsExprFloatEndDay((KHE_SRS_EXPR) efs, day_index,
    efs->tmp_value, ps, srs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprFloatSumHeaderDebug(KHE_SRS_EXPR_FLOAT_SUM efs,           */
/*    KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs,               */
/*    int verbosity, FILE *fp)                                               */
/*                                                                           */
/*  Debug print of the header part of efs in ps onto fp.                     */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprFloatSumHeaderDebug(KHE_SRS_EXPR_FLOAT_SUM efs,
  KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs,
  int verbosity, FILE *fp)
{
  KheSrsExprIntHeaderDebug((KHE_SRS_EXPR) efs, "FLOAT_SUM", efs->tmp_value,
    ps, srs, verbosity, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprFloatSumDebug(KHE_SRS_EXPR_FLOAT_SUM efs,                 */
/*    KHE_SINGLE_RESOURCE_SOLVER srs, int verbosity, int indent, FILE *fp)   */
/*                                                                           */
/*  Debug print of efs onto fp with the given verbosity and indent.          */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprFloatSumDebug(KHE_SRS_EXPR_FLOAT_SUM efs,
  KHE_SINGLE_RESOURCE_SOLVER srs, int verbosity, int indent, FILE *fp)
{
  KheSrsExprFloatDebug((KHE_SRS_EXPR) efs, "FLOAT_SUM", efs->tmp_value,
    srs, verbosity, indent, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_SRS_EXPR_INT_DEV"                                         */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_EXPR_INT_DEV KheSrsExprIntDevMake(int min_limit,                 */
/*    int max_limit, bool allow_zero, KHE_SINGLE_RESOURCE_SOLVER srs)        */
/*                                                                           */
/*  Make a new KHE_SRS_EXPR_INT_DEV object with these attributes.            */
/*                                                                           */
/*  There is no dom_test because its type is always KHE_SRS_DOM_UNUSED.      */
/*                                                                           */
/*****************************************************************************/

static KHE_SRS_EXPR_INT_DEV KheSrsExprIntDevMake(int min_limit,
  int max_limit, bool allow_zero, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_EXPR_INT_DEV res;  KHE_SRS_DOM_TEST dom_test;
  dom_test = KheSrsDomTestMake(KHE_SRS_DOM_UNUSED);
  KheSrsExprInit(res, KHE_SRS_EXPR_INT_DEV_TAG, dom_test, srs,
    free_expr_int_dev);
  res->min_limit = min_limit;
  res->max_limit = max_limit;
  res->allow_zero = allow_zero;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprIntDevDelete(KHE_SRS_EXPR_INT_DEV eid,                    */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Delete eid and its descendants.                                          */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprIntDevDelete(KHE_SRS_EXPR_INT_DEV eid,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_EXPR e;  int i;
  HaArrayForEach(eid->children, e, i)
    KheSrsExprDelete(e, srs);
  HaArrayAddLast(srs->free_expr_int_dev, eid);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_DOM_TEST KheSrsExprIntDevDomTest(KHE_SRS_EXPR_INT_DEV eid)       */
/*                                                                           */
/*  Return a suitable dom test for eid.                                      */
/*                                                                           */
/*****************************************************************************/

/* *** no longer doing it this way
static KHE_SRS_DOM_TEST KheSrsExprIntDevDomTest(KHE_SRS_EXPR_INT_DEV eid)
{
  if( eid->parent == NULL || eid->parent->dom_test.type == KHE_SRS_DOM_NONE )
  {
    ** no inherited dom test, so base the dom test on the limits in eid **
    return KheDomTestForLimits(eid->min_limit, eid->max_limit, eid->allow_zero);
  }
  else
    return KheSrsDomTestMake(KHE_SRS_DOM_MIN_AND_MAX, 0);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprIntDevBeginDay(KHE_SRS_EXPR_INT_DEV eid, int day_index,   */
/*    KHE_SRS_PARTIAL_SOLN prev_ps, KHE_SINGLE_RESOURCE_SOLVER srs)          */
/*                                                                           */
/*  Inform eid that day day_index is beginning.                              */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprIntDevBeginDay(KHE_SRS_EXPR_INT_DEV eid, int day_index,
  KHE_SRS_PARTIAL_SOLN prev_ps, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  /* not in active_exprs, should never be called */
  HnAbort("KheSrsExprIntDevBeginDay internal error");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprIntDevReceiveValueInt(KHE_SRS_EXPR_INT_DEV eid,           */
/*    int value, KHE_SINGLE_RESOURCE_SOLVER srs)                             */
/*                                                                           */
/*  Inform eid that one of its children has just finalized its value.        */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprIntDevReceiveValueInt(KHE_SRS_EXPR_INT_DEV eid,
  int value, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  int dev;
  dev = KheSrsSumDev(eid->min_limit, eid->max_limit, eid->allow_zero, 0, value);
  KheSrsExprReceiveValueInt(eid->parent, dev, srs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprIntDevEndDay(KHE_SRS_EXPR_INT_DEV eid, int day_index,     */
/*    KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs)               */
/*                                                                           */
/*  Inform eid that day day_index is ending.                                 */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprIntDevEndDay(KHE_SRS_EXPR_INT_DEV eid, int day_index,
  KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  /* not in active_exprs, should never be called */
  HnAbort("KheSrsExprIntDevEndDay internal error");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprIntDevHeaderDebug(KHE_SRS_EXPR_INT_DEV eid,               */
/*    KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs,               */
/*    int verbosity, FILE *fp)                                               */
/*                                                                           */
/*  Debug print of the header part of eid in ps onto fp.                     */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprIntDevHeaderDebug(KHE_SRS_EXPR_INT_DEV eid,
  KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs,
  int verbosity, FILE *fp)
{
  KheSrsExprIntHeaderDebug((KHE_SRS_EXPR) eid, "INT_DEV", -1,
    ps, srs, verbosity, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprIntDevDebug(KHE_SRS_EXPR_INT_DEV eid,                     */
/*    KHE_SINGLE_RESOURCE_SOLVER srs, int verbosity, int indent, FILE *fp)   */
/*                                                                           */
/*  Debug print of eid onto fp with the given verbosity and indent.          */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprIntDevDebug(KHE_SRS_EXPR_INT_DEV eid,
  KHE_SINGLE_RESOURCE_SOLVER srs, int verbosity, int indent, FILE *fp)
{
  KHE_SRS_EXPR e;  int i;
  if( indent >= 0 )
  {
    fprintf(fp, "%*s[ INT_DEV(min %d, max %d, allow_zero %s)\n", indent, "",
      eid->min_limit, eid->max_limit, bool_show(eid->allow_zero));
    HaArrayForEach(eid->children, e, i)
      KheSrsExprDebug(e, srs, verbosity, indent + 2, fp);
    fprintf(fp, "%*s]\n", indent, "");
  }
  else
    fprintf(fp, "INT_DEV(min %d, max %d, allow_zero %s)\n",
      eid->min_limit, eid->max_limit, bool_show(eid->allow_zero));
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_SRS_EXPR_FLOAT_DEV"                                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_EXPR_FLOAT_DEV KheSrsExprFloatDevMake(int min_limit,             */
/*    int max_limit, bool allow_zero, KHE_SINGLE_RESOURCE_SOLVER srs)        */
/*                                                                           */
/*  Make a new KHE_SRS_EXPR_FLOAT_DEV object with these attributes.          */
/*                                                                           */
/*  There is no dom_test because its type is always KHE_SRS_DOM_UNUSED.      */
/*                                                                           */
/*****************************************************************************/

static KHE_SRS_EXPR_FLOAT_DEV KheSrsExprFloatDevMake(int min_limit,
  int max_limit, bool allow_zero, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_EXPR_FLOAT_DEV res;  KHE_SRS_DOM_TEST dom_test;
  dom_test = KheSrsDomTestMake(KHE_SRS_DOM_UNUSED);
  KheSrsExprInit(res, KHE_SRS_EXPR_FLOAT_DEV_TAG, dom_test,
    srs, free_expr_float_dev);
  res->min_limit = min_limit;
  res->max_limit = max_limit;
  res->allow_zero = allow_zero;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprFloatDevDelete(KHE_SRS_EXPR_FLOAT_DEV efd,                */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Delete efd and its descendants.                                          */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprFloatDevDelete(KHE_SRS_EXPR_FLOAT_DEV efd,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_EXPR e;  int i;
  HaArrayForEach(efd->children, e, i)
    KheSrsExprDelete(e, srs);
  HaArrayAddLast(srs->free_expr_float_dev, efd);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_DOM_TEST KheSrsExprFloatDevDomTest(KHE_SRS_EXPR_FLOAT_DEV efd)   */
/*                                                                           */
/*  Return a suitable dom test for efd.                                      */
/*                                                                           */
/*****************************************************************************/

/* *** no longer doing it this way
static KHE_SRS_DOM_TEST KheSrsExprFloatDevDomTest(KHE_SRS_EXPR_FLOAT_DEV efd)
{
  if( efd->parent == NULL )
    return KheSrsDomTestMake(KHE_SRS_DOM_NONE, 0);
  else
    return efd->parent->dom_test;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprFloatDevBeginDay(KHE_SRS_EXPR_FLOAT_DEV efd,              */
/*    int day_index, KHE_SRS_PARTIAL_SOLN prev_ps,                           */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Inform efd that day day_index is beginning.                              */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprFloatDevBeginDay(KHE_SRS_EXPR_FLOAT_DEV efd,
  int day_index, KHE_SRS_PARTIAL_SOLN prev_ps, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  /* not in active_exprs, should never be called */
  HnAbort("KheSrsExprFloatDevBeginDay internal error");
}


/*****************************************************************************/
/*                                                                           */
/*  int KheLimitWorkloadMonitorDev(KHE_SRS_EXPR_FLOAT_DEV efd,               */
/*    float workload)                                                        */
/*                                                                           */
/*  Return the deviation of workload wrt efd.                                */
/*                                                                           */
/*****************************************************************************/

static int KheLimitWorkloadMonitorDev(KHE_SRS_EXPR_FLOAT_DEV efd,
  float workload)
{
  if( efd->allow_zero && workload < 0.001 )
    return 0;
  else if( workload < efd->min_limit - 0.001 )
    return (int) ceil(efd->min_limit - 0.001 - workload);
  else if( workload > efd->max_limit + 0.001 )
    return (int) ceil(workload - efd->max_limit - 0.001);
  else
    return 0;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprFloatDevReceiveValueFloat(KHE_SRS_EXPR_FLOAT_DEV efd,     */
/*    float value, KHE_SINGLE_RESOURCE_SOLVER srs)                           */
/*                                                                           */
/*  Inform efd that one of its children has just finalized its value.        */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprFloatDevReceiveValueFloat(KHE_SRS_EXPR_FLOAT_DEV efd,
  float value, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  int dev;
  dev = KheLimitWorkloadMonitorDev(efd, value);
  KheSrsExprReceiveValueInt(efd->parent, dev, srs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprFloatDevEndDay(KHE_SRS_EXPR_FLOAT_DEV efd,                */
/*    int day_index, KHE_SRS_PARTIAL_SOLN ps,                                */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Inform efd that day day_index is ending.                                 */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprFloatDevEndDay(KHE_SRS_EXPR_FLOAT_DEV efd,
  int day_index, KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  /* not in active_exprs, should never be called */
  HnAbort("KheSrsExprFloatDevEndDay internal error");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprFloatDevHeaderDebug(KHE_SRS_EXPR_FLOAT_DEV efd,           */
/*    KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs,               */
/*    int verbosity, FILE *fp)                                               */
/*                                                                           */
/*  Debug print of the header part of efd in ps onto fp.                     */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprFloatDevHeaderDebug(KHE_SRS_EXPR_FLOAT_DEV efd,
  KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs,
  int verbosity, FILE *fp)
{
  KheSrsExprIntHeaderDebug((KHE_SRS_EXPR) efd, "FLOAT_DEV", -1.0,
    ps, srs, verbosity, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprFloatDevDebug(KHE_SRS_EXPR_FLOAT_DEV efd,                 */
/*    KHE_SINGLE_RESOURCE_SOLVER srs, int verbosity, int indent, FILE *fp)   */
/*                                                                           */
/*  Debug print of efd onto fp with the given verbosity and indent.          */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprFloatDevDebug(KHE_SRS_EXPR_FLOAT_DEV efd,
  KHE_SINGLE_RESOURCE_SOLVER srs, int verbosity, int indent, FILE *fp)
{
  KHE_SRS_EXPR e;  int i;
  if( indent >= 0 )
  {
    fprintf(fp, "%*s[ FLOAT_DEV(min %d, max %d, allow_zero %s)\n", indent, "",
      efd->min_limit, efd->max_limit, bool_show(efd->allow_zero));
    HaArrayForEach(efd->children, e, i)
      KheSrsExprDebug(e, srs, verbosity, indent + 2, fp);
    fprintf(fp, "%*s]\n", indent, "");
  }
  else
    fprintf(fp, "FLOAT_DEV(min %d, max %d, allow_zero %s)\n",
      efd->min_limit, efd->max_limit, bool_show(efd->allow_zero));
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_SRS_EXPR_COST"                                            */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_EXPR_COST KheSrsExprCostMake(KHE_MONITOR m,                      */
/*    KHE_COST_FUNCTION cost_fn, KHE_COST combined_weight,                   */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Make a new KHE_SRS_EXPR_COST object with these attributes.               */
/*                                                                           */
/*  This function has no dom_test parameter, because there is never any      */
/*  dominance testing at this node.                                          */
/*                                                                           */
/*****************************************************************************/

static KHE_SRS_EXPR_COST KheSrsExprCostMake(KHE_MONITOR m,
  KHE_COST_FUNCTION cost_fn, KHE_COST combined_weight,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_EXPR_COST res;  KHE_SRS_DOM_TEST dom_test;
  dom_test = KheSrsDomTestMake(KHE_SRS_DOM_UNUSED);
  KheSrsExprInit(res, KHE_SRS_EXPR_COST_TAG, dom_test, srs, free_expr_cost);
  res->monitor = m;
  res->cost_fn = cost_fn;
  res->combined_weight = combined_weight;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprCostDelete(KHE_SRS_EXPR_COST ec,                          */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Delete ec and its descendants.                                           */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprCostDelete(KHE_SRS_EXPR_COST ec,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_EXPR e;  int i;
  HaArrayForEach(ec->children, e, i)
    KheSrsExprDelete(e, srs);
  HaArrayAddLast(srs->free_expr_cost, ec);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_DOM_TEST KheSrsExprCostDomTest(KHE_SRS_EXPR_COST ec)             */
/*                                                                           */
/*  Return a suitable dom test for ec.                                       */
/*                                                                           */
/*****************************************************************************/

/* *** no longer doing it this way
static KHE_SRS_DOM_TEST KheSrsExprCostDomTest(KHE_SRS_EXPR_COST ec)
{
  return KheSrsDomTestMake(KHE_SRS_DOM_NONE, 0);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprCostBeginDay(KHE_SRS_EXPR_COST ec, int day_index,         */
/*    KHE_SRS_PARTIAL_SOLN prev_ps, KHE_SINGLE_RESOURCE_SOLVER srs)          */
/*                                                                           */
/*  Inform ec that day day_index is beginning.                               */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprCostBeginDay(KHE_SRS_EXPR_COST ec, int day_index,
  KHE_SRS_PARTIAL_SOLN prev_ps, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  /* not in active_exprs, should never be called */
  HnAbort("KheSrsExprCostBeginDay internal error");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprCostReceiveValueInt(KHE_SRS_EXPR_COST ec,                 */
/*    int value, KHE_SINGLE_RESOURCE_SOLVER srs)                             */
/*                                                                           */
/*  Inform ec that one of its children has just finalized its value.         */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprCostReceiveValueInt(KHE_SRS_EXPR_COST ec,
  int value, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_COST cost;
  cost = KheSrsCost(ec->cost_fn, ec->combined_weight, value);
  if( cost > 0 )
  {
    if( srs->rerun )
      fprintf(stderr, "  %s reporting cost %.5f\n",
	KheMonitorId(ec->monitor), KheCostShow(cost));
    KheSrsExprReceiveValueCost(ec->parent, cost, srs);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprCostEndDay(KHE_SRS_EXPR_COST ec, int day_index,           */
/*    KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs)               */
/*                                                                           */
/*  Inform ec that day day_index is ending.                                  */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprCostEndDay(KHE_SRS_EXPR_COST ec, int day_index,
  KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  /* not in active_exprs, should never be called */
  HnAbort("KheSrsExprCostEndDay internal error");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprCostHeaderDebug(KHE_SRS_EXPR_COST ec,                     */
/*    KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs,               */
/*    int verbosity, FILE *fp)                                               */
/*                                                                           */
/*  Debug print of the header part of ec in ps onto fp.                      */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprCostHeaderDebug(KHE_SRS_EXPR_COST ec,
  KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs,
  int verbosity, FILE *fp)
{
  fprintf(fp, "COST %s", KheMonitorId(ec->monitor));
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprCostDebug(KHE_SRS_EXPR_COST ec,                           */
/*    KHE_SINGLE_RESOURCE_SOLVER srs, int verbosity, int indent, FILE *fp)   */
/*                                                                           */
/*  Debug print of ec onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprCostDebug(KHE_SRS_EXPR_COST ec,
  KHE_SINGLE_RESOURCE_SOLVER srs, int verbosity, int indent, FILE *fp)
{
  KHE_SRS_EXPR e;  int i;
  if( indent >= 0 )
  {
    fprintf(fp, "%*s[ COST(%s, cost_fn %s, combined_weight %.5f)\n", indent, "",
      ec->monitor != NULL ? KheMonitorId(ec->monitor) : "-",
      KheCostFunctionShow(ec->cost_fn), KheCostShow(ec->combined_weight));
    HaArrayForEach(ec->children, e, i)
      KheSrsExprDebug(e, srs, verbosity, indent + 2, fp);
    fprintf(fp, "%*s]\n", indent, "");
  }
  else
    fprintf(fp, "COST(%s, cost_fn %s, combined_weight %.5f)\n",
      ec->monitor != NULL ? KheMonitorId(ec->monitor) : "-",
      KheCostFunctionShow(ec->cost_fn), KheCostShow(ec->combined_weight));
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_SRS_EXPR_COST_SUM"                                        */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_EXPR_COST_SUM KheSrsExprCostSumMake(                             */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Make a new KHE_SRS_EXPR_COST_SUM object with these attributes.           */
/*                                                                           */
/*  This function has no dom_test parameter, because there is never any      */
/*  dominance testing at this node.                                          */
/*                                                                           */
/*****************************************************************************/

static KHE_SRS_EXPR_COST_SUM KheSrsExprCostSumMake(
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_EXPR_COST_SUM res;  KHE_SRS_DOM_TEST unused_dom_test;
  unused_dom_test = KheSrsDomTestMake(KHE_SRS_DOM_UNUSED);
  KheSrsExprInit(res, KHE_SRS_EXPR_COST_SUM_TAG, unused_dom_test, srs,
    free_expr_cost_sum);
  res->tmp_value = KheCost(-1, -1);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprCostSumDelete(KHE_SRS_EXPR_COST_SUM ecs,                  */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Delete ecs and its descendants.                                          */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprCostSumDelete(KHE_SRS_EXPR_COST_SUM ecs,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_EXPR e;  int i;
  HaArrayForEach(ecs->children, e, i)
    KheSrsExprDelete(e, srs);
  HaArrayAddLast(srs->free_expr_cost_sum, ecs);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_DOM_TEST KheSrsExprCostSumDomTest(KHE_SRS_EXPR_COST_SUM ecs)     */
/*                                                                           */
/*  Return a suitable dom test for ecs.                                      */
/*                                                                           */
/*****************************************************************************/

/* *** no longer doing it this way
static KHE_SRS_DOM_TEST KheSrsExprCostSumDomTest(KHE_SRS_EXPR_COST_SUM ecs)
{
  ** still to do **
  HnAbort("KheSrsExprCostSumDomTest still to do");
  return KheSrsDomTestMake(KHE_SRS_DOM_NONE, 0);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprCostSumBeginDay(KHE_SRS_EXPR_COST_SUM ecs, int day_index, */
/*    KHE_SRS_PARTIAL_SOLN prev_ps, KHE_SINGLE_RESOURCE_SOLVER srs)          */
/*                                                                           */
/*  Inform ecs that day day_index is beginning.                              */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprCostSumBeginDay(KHE_SRS_EXPR_COST_SUM ecs, int day_index,
  KHE_SRS_PARTIAL_SOLN prev_ps, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  if( day_index == ecs->first_day_index )
  {
    /* this is the start of the first day covered by this expression */
    ecs->tmp_value = KheCost(0, 0);
  }
  else
  {
    /* this is not the first day; get previous day's value */
    ecs->tmp_value = prev_ps->cost;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprCostSumReceiveValueCost(KHE_SRS_EXPR_COST_SUM ecs,        */
/*    KHE_COST value, KHE_SINGLE_RESOURCE_SOLVER srs)                        */
/*                                                                           */
/*  Inform ecs that one of its children has just finalized its value.        */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprCostSumReceiveValueCost(KHE_SRS_EXPR_COST_SUM ecs,
  KHE_COST value, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  ecs->tmp_value += value;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprCostSumEndDay(KHE_SRS_EXPR_COST_SUM ecs, int day_index,   */
/*    KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs)               */
/*                                                                           */
/*  Inform ecs that day day_index is ending.                                 */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprCostSumEndDay(KHE_SRS_EXPR_COST_SUM ecs, int day_index,
  KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  HnAssert(ecs->first_day_index <= day_index && day_index <=ecs->last_day_index,
    "KheSrsExprCostSumEndDay internal error");
  if( day_index == ecs->last_day_index )
  {
    /* this is the end of the last day covered by this expression */
    /* but there is nowhere else to report tmp_value to */
    ps->cost = ecs->tmp_value;
  }
  else
  {
    /* this is not the last day; store value in signature */
    ps->cost = ecs->tmp_value;
  }
  ecs->tmp_value = KheCost(-1, -1);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprCostSumHeaderDebug(KHE_SRS_EXPR_COST_SUM ecs,             */
/*    KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs,               */
/*    int verbosity, FILE *fp)                                               */
/*                                                                           */
/*  Debug print of the header part of ecs in ps onto fp.                     */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprCostSumHeaderDebug(KHE_SRS_EXPR_COST_SUM ecs,
  KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs,
  int verbosity, FILE *fp)
{
  HnAbort("KheSrsExprCostSumHeaderDebug internal error (unexpected call)");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprCostSumDebug(KHE_SRS_EXPR_COST_SUM ecs,                   */
/*    KHE_SINGLE_RESOURCE_SOLVER srs, int verbosity, int indent, FILE *fp)   */
/*                                                                           */
/*  Debug print of ecs onto fp with the given verbosity and indent.          */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprCostSumDebug(KHE_SRS_EXPR_COST_SUM ecs,
  KHE_SINGLE_RESOURCE_SOLVER srs, int verbosity, int indent, FILE *fp)
{
  KHE_SRS_EXPR e;  int i;
  if( indent >= 0 )
  {
    fprintf(fp, "%*s[ COST_SUM", indent, "");
    if( ecs->tmp_value != KheCost(-1, -1) )
      fprintf(fp, "(val %.5f)", KheCostShow(ecs->tmp_value));
    fprintf(fp, "\n");
    HaArrayForEach(ecs->children, e, i)
      KheSrsExprDebug(e, srs, verbosity, indent + 2, fp);
    fprintf(fp, "%*s]\n", indent, "");
  }
  else
    fprintf(fp, "COST_SUM(%d children)", HaArrayCount(ecs->children));
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_SRS_EXPR_INT_SUM_COMB"                                    */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_COMB_TYPE KheSrsCombType(KHE_COST_FUNCTION cost_fn,              */
/*    int max_limit)                                                         */
/*                                                                           */
/*  Return the comb type, as determined by the cost function and max limit.  */
/*                                                                           */
/*****************************************************************************/

static KHE_SRS_COMB_TYPE KheSrsCombType(KHE_COST_FUNCTION cost_fn,
  int max_limit)
{
  if( max_limit == INT_MAX )
    return KHE_SRS_COMB_NO_MAX;
  else if( cost_fn == KHE_LINEAR_COST_FUNCTION )
    return KHE_SRS_COMB_LINEAR;
  else if( cost_fn == KHE_STEP_COST_FUNCTION )
    return KHE_SRS_COMB_STEP;
  else
    return KHE_SRS_COMB_ORDINARY;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_EXPR_INT_SUM_COMB KheSrsExprIntSumCombMake(KHE_MONITOR m,        */
/*    KHE_COST_FUNCTION cost_fn, KHE_COST combined_weight,                   */
/*    int min_limit, int max_limit, bool allow_zero, int history_before,     */
/*    int history_after, KHE_SRS_DOM_TEST dom_test,                          */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Make a new KHE_SRS_EXPR_INT_SUM_COMB object with these attributes.       */
/*                                                                           */
/*****************************************************************************/

static KHE_SRS_EXPR_INT_SUM_COMB KheSrsExprIntSumCombMake(KHE_MONITOR m,
  KHE_COST_FUNCTION cost_fn, KHE_COST combined_weight,
  int min_limit, int max_limit, bool allow_zero, int history_before,
  int history_after, KHE_SRS_DOM_TEST dom_test, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_EXPR_INT_SUM_COMB res;
  KheSrsExprInit(res, KHE_SRS_EXPR_INT_SUM_COMB_TAG, dom_test, srs,
    free_expr_int_sum_comb);
  res->monitor = m;
  res->cost_fn = cost_fn;
  res->combined_weight = combined_weight;
  res->min_limit = min_limit;
  res->max_limit = max_limit;
  res->allow_zero = allow_zero;
  res->history_before = history_before;
  res->history_after = history_after;
  res->comb_type = KheSrsCombType(cost_fn, max_limit);
  res->tmp_value = UNDEF_INT;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprIntSumCombDelete(KHE_SRS_EXPR_INT_SUM_COMB eisc,          */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Delete eisc and its descendants.                                         */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprIntSumCombDelete(KHE_SRS_EXPR_INT_SUM_COMB eisc,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_EXPR e;  int i;
  HaArrayForEach(eisc->children, e, i)
    KheSrsExprDelete(e, srs);
  HaArrayAddLast(srs->free_expr_int_sum_comb, eisc);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_DOM_TEST KheSrsExprIntSumCombDomTest(                            */
/*    KHE_SRS_EXPR_INT_SUM_COMB eisc)                                        */
/*                                                                           */
/*  Return a suitable dom test for eisc.                                     */
/*                                                                           */
/*****************************************************************************/

/* *** no longer doing it this way
static KHE_SRS_DOM_TEST KheSrsExprIntSumCombDomTest(
  KHE_SRS_EXPR_INT_SUM_COMB eisc)
{
  ** still to do **
  HnAbort("KheSrsExprIntSumCombDomTest still to do");
  return KheSrsDomTestMake(KHE_SRS_DOM_NONE, 0);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprIntSumCombBeginDay(KHE_SRS_EXPR_INT_SUM_COMB eisc,        */
/*    int day_index, KHE_SRS_PARTIAL_SOLN prev_ps,                           */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Inform eisc that day day_index is beginning.                             */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprIntSumCombBeginDay(KHE_SRS_EXPR_INT_SUM_COMB eisc,
  int day_index, KHE_SRS_PARTIAL_SOLN prev_ps, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  /* KHE_SRS_DAY_INFO di; */  int sig_index;
  HnAssert(eisc->first_day_index <= day_index &&
    day_index <= eisc->last_day_index,
    "KheSrsExprIntSumBeginDay internal error");
  if( day_index == eisc->first_day_index )
  {
    /* this is the start of the first day covered by this expression */
    eisc->tmp_value = eisc->history_before;
  }
  else
  {
    /* this is not the first day; get previous day's value */
    sig_index = HaArray(eisc->sig_indexes, day_index -1 -eisc->first_day_index);
    eisc->tmp_value = HaArray(prev_ps->signature, sig_index);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprIntSumCombReceiveValueInt(KHE_SRS_EXPR_INT_SUM_COMB eisc, */
/*    int value, KHE_SINGLE_RESOURCE_SOLVER srs)                             */
/*                                                                           */
/*  Inform eisc that one of its children has just finalized its value.       */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprIntSumCombReceiveValueInt(KHE_SRS_EXPR_INT_SUM_COMB eisc,
  int value, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_COST cost;

  /* add value to tmp_value */
  eisc->tmp_value += value;

  /* adjust in various special cases */
  switch( eisc->comb_type )
  {
    case KHE_SRS_COMB_ORDINARY:

      /* no adjustment here */
      break;

    case KHE_SRS_COMB_NO_MAX:

      /* no maximum limit; store minimum limit at most */
      if( eisc->tmp_value > eisc->min_limit )
        eisc->tmp_value = eisc->min_limit;
      break;

    case KHE_SRS_COMB_LINEAR:

      /* linear cost; if over maximum, report excess immediately */
      if( eisc->tmp_value > eisc->max_limit )
      {
	cost = (eisc->tmp_value - eisc->max_limit) * eisc->combined_weight;
	if( cost > 0 )
	{
	  if( srs->rerun )
	    fprintf(stderr, "  %s reporting cost %.5f\n",
	      KheMonitorId(eisc->monitor), KheCostShow(cost));
	  KheSrsExprReceiveValueCost(eisc->parent, cost, srs);
	}
	eisc->tmp_value = eisc->max_limit;
      }
      break;

    case KHE_SRS_COMB_STEP:

      /* step cost; store maximum plus one at most */
      if( eisc->tmp_value > eisc->max_limit + 1 )
        eisc->tmp_value = eisc->max_limit + 1;
      break;

    default:

      HnAbort("KheSrsExprIntSumCombReceiveValueInt internal error");
      break;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprIntSumCombEndDay(KHE_SRS_EXPR_INT_SUM_COMB eisc,          */
/*    int day_index, KHE_SRS_PARTIAL_SOLN ps,                                */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Inform eisc that day day_index is ending.                                */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprIntSumCombEndDay(KHE_SRS_EXPR_INT_SUM_COMB eisc,
  int day_index, KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  int dev, sig_index;  KHE_COST cost;
  HnAssert(eisc->first_day_index <= day_index &&
    day_index <= eisc->last_day_index,
    "KheSrsExprIntSumCombEndDay internal error");
  if( day_index == eisc->last_day_index )
  {
    /* this is the end of the last day covered by this expression */
    dev = KheSrsSumDev(eisc->min_limit, eisc->max_limit, eisc->allow_zero,
      eisc->history_after, eisc->tmp_value);
    cost = KheSrsCost(eisc->cost_fn, eisc->combined_weight, dev);
    if( cost > 0 )
    {
      if( srs->rerun )
	fprintf(stderr, "  %s reporting cost %.5f\n",
	  KheMonitorId(eisc->monitor), KheCostShow(cost));
      KheSrsExprReceiveValueCost(eisc->parent, cost, srs);
    }
  }
  else
  {
    /* this is not the last day; store value in signature */
    sig_index = HaArray(eisc->sig_indexes, day_index - eisc->first_day_index);
    HnAssert(HaArrayCount(ps->signature) == sig_index,
      "KheSrsExprIntSumEndDay internal error");
    HaArrayAddLast(ps->signature, eisc->tmp_value);
  }
  eisc->tmp_value = UNDEF_INT;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprIntSumCombHeaderDebug(KHE_SRS_EXPR_INT_SUM_COMB eisc,     */
/*    KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs,               */
/*    int verbosity, FILE *fp)                                               */
/*                                                                           */
/*  Debug print of the header part of eisc in ps onto fp.                    */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprIntSumCombHeaderDebug(KHE_SRS_EXPR_INT_SUM_COMB eisc,
  KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs,
  int verbosity, FILE *fp)
{
  KheSrsExprIntHeaderDebug((KHE_SRS_EXPR) eisc, "INT_SUM_COMB", -1,
    ps, srs, verbosity, fp);
  fprintf(fp, " %s", KheMonitorId(eisc->monitor));
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprIntSumCombDebug(KHE_SRS_EXPR_INT_SUM_COMB eisc,           */
/*    KHE_SINGLE_RESOURCE_SOLVER srs, int verbosity, int indent, FILE *fp)   */
/*                                                                           */
/*  Debug print of eisc onto fp with the given verbosity and indent.         */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprIntSumCombDebug(KHE_SRS_EXPR_INT_SUM_COMB eisc,
  KHE_SINGLE_RESOURCE_SOLVER srs, int verbosity, int indent, FILE *fp)
{
  KHE_SRS_EXPR e;  int i;
  if( indent >= 0 )
  {
    fprintf(fp, "%*s[ INT_SUM_COMB(%s, %s %.5f, lim %d-%d, az %s, h %d,%d)\n",
      indent, "", eisc->monitor != NULL ? KheMonitorId(eisc->monitor) : "-",
      KheCostFunctionShow(eisc->cost_fn), KheCostShow(eisc->combined_weight),
      eisc->min_limit, eisc->max_limit, bool_show(eisc->allow_zero),
      eisc->history_before, eisc->history_after);
    HaArrayForEach(eisc->children, e, i)
      KheSrsExprDebug(e, srs, verbosity, indent + 2, fp);
    fprintf(fp, "%*s]\n", indent, "");
  }
  else
    fprintf(fp, "INT_SUM_COMB(%s, h %d,%d, %d children)",
      eisc->monitor != NULL ? KheMonitorId(eisc->monitor) : "-",
      eisc->history_before, eisc->history_after, HaArrayCount(eisc->children));
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_SRS_EXPR_INT_SEQ_COMB"                                    */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_EXPR_INT_SEQ_COMB KheSrsExprIntSeqCombMake(KHE_MONITOR m,        */
/*    KHE_COST_FUNCTION cost_fn, KHE_COST combined_weight,                   */
/*    KHE_COST_FUNCTION cost_fn, KHE_COST combined_weight, int min_limit,    */
/*    int max_limit, int history_before, int history_after,                  */
/*    KHE_SRS_DOM_TEST dom_test, KHE_SINGLE_RESOURCE_SOLVER srs)             */
/*                                                                           */
/*  Make a new KHE_SRS_EXPR_INT_SEQ_COMB object with these attributes.       */
/*                                                                           */
/*****************************************************************************/

static KHE_SRS_EXPR_INT_SEQ_COMB KheSrsExprIntSeqCombMake(KHE_MONITOR m,
  KHE_COST_FUNCTION cost_fn, KHE_COST combined_weight, int min_limit,
  int max_limit, int history_before, int history_after,
  KHE_SRS_DOM_TEST dom_test, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_EXPR_INT_SEQ_COMB res;
  KheSrsExprInit(res, KHE_SRS_EXPR_INT_SEQ_COMB_TAG, dom_test, srs,
    free_expr_int_seq_comb);
  res->monitor = m;
  res->cost_fn = cost_fn;
  res->combined_weight = combined_weight;
  res->min_limit = min_limit;
  res->max_limit = max_limit;
  res->history_before = history_before;
  res->history_after = history_after;
  res->comb_type = KheSrsCombType(cost_fn, max_limit);
  res->tmp_value = UNDEF_INT;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprIntSeqCombDelete(KHE_SRS_EXPR_INT_SEQ_COMB eisc,          */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Delete eisc and its descendants.                                         */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprIntSeqCombDelete(KHE_SRS_EXPR_INT_SEQ_COMB eisc,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_EXPR e;  int i;
  HaArrayForEach(eisc->children, e, i)
    KheSrsExprDelete(e, srs);
  HaArrayAddLast(srs->free_expr_int_seq_comb, eisc);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_DOM_TEST KheSrsExprIntSeqCombDomTest(                            */
/*    KHE_SRS_EXPR_INT_SEQ_COMB eisc)                                        */
/*                                                                           */
/*  Return a suitable dom test for eisc.                                     */
/*                                                                           */
/*****************************************************************************/

/* *** no longer doing it this way
static KHE_SRS_DOM_TEST KheSrsExprIntSeqCombDomTest(
  KHE_SRS_EXPR_INT_SEQ_COMB eisc)
{
  ** still to do **
  HnAbort("KheSrsExprIntSeqCombDomTest still to do");
  return KheSrsDomTestMake(KHE_SRS_DOM_NONE, 0);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprIntSeqCombBeginDay(KHE_SRS_EXPR_INT_SEQ_COMB eisc,        */
/*    int day_index, KHE_SRS_PARTIAL_SOLN prev_ps,                           */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Inform eisc that day day_index is beginning.                             */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprIntSeqCombBeginDay(KHE_SRS_EXPR_INT_SEQ_COMB eisc,
  int day_index, KHE_SRS_PARTIAL_SOLN prev_ps, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  int sig_index;
  HnAssert(eisc->first_day_index <= day_index &&
    day_index <= eisc->last_day_index,
    "KheSrsExprIntSeqBeginDay internal error");
  if( day_index == eisc->first_day_index )
  {
    /* this is the start of the first day covered by this expression */
    eisc->tmp_value = eisc->history_before;
  }
  else
  {
    /* this is not the first day; get previous day's value */
    sig_index = HaArray(eisc->sig_indexes, day_index -1 -eisc->first_day_index);
    eisc->tmp_value = HaArray(prev_ps->signature, sig_index);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprIntSeqCombReceiveValueInt(KHE_SRS_EXPR_INT_SEQ_COMB eisc, */
/*    int value, KHE_SINGLE_RESOURCE_SOLVER srs)                             */
/*                                                                           */
/*  Inform eisc that one of its children has just finalized its value.       */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprIntSeqCombReceiveValueInt(KHE_SRS_EXPR_INT_SEQ_COMB eisc,
  int value, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_COST cost;  int dev;

  HnAssert(value == 0 || value == 1,
    "KheSrsExprIntSeqReceiveValueInt internal error");
  if( value == 1 )
  {
    /* a sequence is beginning or continuing */
    eisc->tmp_value++;

    /* adjust in various special cases */
    switch( eisc->comb_type )
    {
      case KHE_SRS_COMB_ORDINARY:

	/* no adjustment here */
	break;

      case KHE_SRS_COMB_NO_MAX:

	/* no maximum limit; store minimum limit at most */
	if( eisc->tmp_value > eisc->min_limit )
	  eisc->tmp_value = eisc->min_limit;
	break;

      case KHE_SRS_COMB_LINEAR:

	/* linear cost; if over maximum, report excess immediately */
	if( eisc->tmp_value > eisc->max_limit )
	{
	  dev = KheSrsSeqDev(eisc->min_limit, eisc->max_limit, 0,
	    eisc->tmp_value);
	  cost = KheSrsCost(eisc->cost_fn, eisc->combined_weight, dev);
	  if( cost > 0 )
	  {
	    if( srs->rerun )
	      fprintf(stderr, "  %s reporting cost %.5f\n",
		KheMonitorId(eisc->monitor), KheCostShow(cost));
	    KheSrsExprReceiveValueCost(eisc->parent, cost, srs);
	  }
	  eisc->tmp_value = eisc->max_limit;
	}
	break;

      case KHE_SRS_COMB_STEP:

	/* step cost; store maximum plus one at most */
	if( eisc->tmp_value > eisc->max_limit + 1 )
	  eisc->tmp_value = eisc->max_limit + 1;
	break;

      default:

	HnAbort("KheSrsExprIntSeqCombReceiveValueInt internal error");
	break;
    }
  }
  else if( eisc->tmp_value > 0 )
  {
    /* a non-empty sequence not at the end of the cycle has just ended */
    dev = KheSrsSeqDev(eisc->min_limit, eisc->max_limit, 0, eisc->tmp_value);
    if( dev > 0 )
    {
      cost = KheSrsCost(eisc->cost_fn, eisc->combined_weight, dev);
      if( cost > 0 )
      {
	if( srs->rerun )
	  fprintf(stderr, "  %s reporting cost %.5f\n",
	    KheMonitorId(eisc->monitor), KheCostShow(cost));
	KheSrsExprReceiveValueCost(eisc->parent, cost, srs);
      }
    }
    eisc->tmp_value = 0;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprIntSeqCombEndDay(KHE_SRS_EXPR_INT_SEQ_COMB eisc,          */
/*    int day_index, KHE_SRS_PARTIAL_SOLN ps,                                */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Inform eisc that day day_index is ending.                                */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprIntSeqCombEndDay(KHE_SRS_EXPR_INT_SEQ_COMB eisc,
  int day_index, KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  int dev, sig_index;  KHE_COST cost;
  HnAssert(eisc->first_day_index <= day_index &&
    day_index <= eisc->last_day_index,
    "KheSrsExprIntSeqCombEndDay internal error");
  if( day_index == eisc->last_day_index )
  {
    /* this is the end of the last day covered by this expression */
    if( eisc->tmp_value > 0 )
    {
      dev = KheSrsSeqDev(eisc->min_limit, eisc->max_limit,
	eisc->history_after, eisc->tmp_value);
      if( dev > 0 )
      {
	cost = KheSrsCost(eisc->cost_fn, eisc->combined_weight, dev);
	if( cost > 0 )
	{
	  if( srs->rerun )
	    fprintf(stderr, "  %s reporting cost %.5f\n",
	      KheMonitorId(eisc->monitor), KheCostShow(cost));
	  KheSrsExprReceiveValueCost(eisc->parent, cost, srs);
	}
      }
    }
  }
  else
  {
    /* this is not the last day; store value in signature */
    sig_index = HaArray(eisc->sig_indexes, day_index - eisc->first_day_index);
    HnAssert(HaArrayCount(ps->signature) == sig_index,
      "KheSrsExprIntSeqEndDay internal error");
    HaArrayAddLast(ps->signature, eisc->tmp_value);
  }
  eisc->tmp_value = UNDEF_INT;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprIntSeqCombHeaderDebug(KHE_SRS_EXPR_INT_SEQ_COMB eisc,     */
/*    KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs,               */
/*    int verbosity, FILE *fp)                                               */
/*                                                                           */
/*  Debug print of the header part of eisc in ps onto fp.                    */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprIntSeqCombHeaderDebug(KHE_SRS_EXPR_INT_SEQ_COMB eisc,
  KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs,
  int verbosity, FILE *fp)
{
  KheSrsExprIntHeaderDebug((KHE_SRS_EXPR) eisc, "INT_SEQ_COMB", -1,
    ps, srs, verbosity, fp);
  fprintf(fp, " %s", KheMonitorId(eisc->monitor));
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprIntSeqCombDebug(KHE_SRS_EXPR_INT_SEQ_COMB eisc,           */
/*    KHE_SINGLE_RESOURCE_SOLVER srs, int verbosity, int indent, FILE *fp)   */
/*                                                                           */
/*  Debug print of eisc onto fp with the given verbosity and indent.         */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprIntSeqCombDebug(KHE_SRS_EXPR_INT_SEQ_COMB eisc,
  KHE_SINGLE_RESOURCE_SOLVER srs, int verbosity, int indent, FILE *fp)
{
  KHE_SRS_EXPR e;  int i;
  if( indent >= 0 )
  {
    fprintf(fp, "%*s[ INT_SEQ_COMB(%s, %s %.5f, lim %d-%d, h %d,%d)\n",
      indent, "", eisc->monitor != NULL ? KheMonitorId(eisc->monitor) : "-",
      KheCostFunctionShow(eisc->cost_fn), KheCostShow(eisc->combined_weight),
      eisc->min_limit, eisc->max_limit, eisc->history_before,
      eisc->history_after);
    HaArrayForEach(eisc->children, e, i)
      KheSrsExprDebug(e, srs, verbosity, indent + 2, fp);
    fprintf(fp, "%*s]\n", indent, "");
  }
  else
    fprintf(fp, "INT_SEQ_COMB(%s, h %d,%d, %d children)",
      eisc->monitor != NULL ? KheMonitorId(eisc->monitor) : "-",
      eisc->history_before, eisc->history_after, HaArrayCount(eisc->children));
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_SRS_EXPR"                                                 */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprAddChild(KHE_SRS_EXPR parent, KHE_SRS_EXPR child)         */
/*                                                                           */
/*  Make child a child of parent.                                            */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprAddChild(KHE_SRS_EXPR parent, KHE_SRS_EXPR child)
{
  HnAssert(child->parent == NULL, "KheSrsExprAddChild internal error");
  HaArrayAddLast(parent->children, child);
  child->parent = parent;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprEnsureDayRangeCovers(KHE_SRS_EXPR e,                      */
/*    int first_day_index, int last_day_index)                               */
/*                                                                           */
/*  Ensure that e's day range covers first_day_index .. last_day_index.      */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprEnsureDayRangeCovers(KHE_SRS_EXPR e,
  int first_day_index, int last_day_index)
{
  if( first_day_index < e->first_day_index )
    e->first_day_index = first_day_index;
  if( last_day_index > e->last_day_index )
    e->last_day_index = last_day_index;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprSetDayRange(KHE_SRS_EXPR e, int min_last_day_index,       */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Set the day range (first_day_index and last_day_index) of e.             */
/*                                                                           */
/*  Ensure that e->last_day_index is at least min_last_day_index.  This      */
/*  is a non-trivial value for the children of INT_SEQ and INT_SEQ_COMB      */
/*  nodes; it ensures that those children report in the order they appear.   */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprSetDayRange(KHE_SRS_EXPR e, int min_last_day_index,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_EXPR child_e;  int i, index, prev_last_day_index;
  KHE_SRS_EXPR_BUSY eb;  KHE_SRS_EXPR_FREE ef;  KHE_SRS_EXPR_WORK ew;

  /* initialize range to empty */
  e->first_day_index = INT_MAX;
  e->last_day_index = min_last_day_index;

  /* increase range depending on the expression type */
  switch( e->tag )
  {
    case KHE_SRS_EXPR_BUSY_TAG:

      eb = (KHE_SRS_EXPR_BUSY) e;
      index = KheFrameTimeIndex(srs->days_frame, eb->time);
      KheSrsExprEnsureDayRangeCovers(e, index, index);
      break;

    case KHE_SRS_EXPR_FREE_TAG:

      ef = (KHE_SRS_EXPR_FREE) e;
      index = KheFrameTimeIndex(srs->days_frame, ef->time);
      KheSrsExprEnsureDayRangeCovers(e, index, index);
      break;

    case KHE_SRS_EXPR_WORK_TAG:

      ew = (KHE_SRS_EXPR_WORK) e;
      index = KheFrameTimeIndex(srs->days_frame, ew->time);
      KheSrsExprEnsureDayRangeCovers(e, index, index);
      break;

    /* case KHE_SRS_EXPR_INT_SEQ_TAG: */
    case KHE_SRS_EXPR_INT_SEQ_COMB_TAG:

      prev_last_day_index = 0;
      HaArrayForEach(e->children, child_e, i)
      {
        KheSrsExprSetDayRange(child_e, prev_last_day_index, srs);
	KheSrsExprEnsureDayRangeCovers(e, child_e->first_day_index,
	  child_e->last_day_index);
	prev_last_day_index = child_e->last_day_index;
      }

      break;

    default:  /* all other nodes */

      /* set the day information for each child node and find day range here */
      HaArrayForEach(e->children, child_e, i)
      {
        KheSrsExprSetDayRange(child_e, 0, srs);
	KheSrsExprEnsureDayRangeCovers(e, child_e->first_day_index,
	  child_e->last_day_index);
      }
      break;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_DOM_TEST KheSrsExprChildDomTest(KHE_SRS_EXPR e,                  */
/*    KHE_SRS_DOM_TEST prnt_dom_test)                                        */
/*                                                                           */
/*  Return e's dom test, as a function of the type of e, any limits in e,    */
/*  and e's parent's dom test.                                               */
/*                                                                           */
/*****************************************************************************/

/* *** no longer doing it this way
KHE_SRS_DOM_TEST KheSrsExprDomTest(KHE_SRS_EXPR e)
{
  switch( e->tag )
  {
    case KHE_SRS_EXPR_BUSY_TAG:

      return KheSrsExprBusyDomTest((KHE_SRS_EXPR_BUSY) e);

    case KHE_SRS_EXPR_FREE_TAG:

      return KheSrsExprFreeDomTest((KHE_SRS_EXPR_FREE) e);

    case KHE_SRS_EXPR_WORK_TAG:

      return KheSrsExprWorkDomTest((KHE_SRS_EXPR_WORK) e);

    case KHE_SRS_EXPR_OR_TAG:

      return KheSrsExprOrDomTest((KHE_SRS_EXPR_OR) e);

    case KHE_SRS_EXPR_AND_TAG:

      return KheSrsExprAndDomTest((KHE_SRS_EXPR_AND) e);

    case KHE_SRS_EXPR_INT_SUM_TAG:

      return KheSrsExprIntSumDomTest((KHE_SRS_EXPR_INT_SUM) e);

    ** *** currently unused
    case KHE_SRS_EXPR_INT_SEQ_TAG:

      return KheSrsExprIntSeqDomTest((KHE_SRS_EXPR_INT_SEQ) e);
    *** **

    case KHE_SRS_EXPR_FLOAT_SUM_TAG:

      return KheSrsExprFloatSumDomTest((KHE_SRS_EXPR_FLOAT_SUM) e);

    case KHE_SRS_EXPR_INT_DEV_TAG:

      return KheSrsExprIntDevDomTest((KHE_SRS_EXPR_INT_DEV) e);

    case KHE_SRS_EXPR_FLOAT_DEV_TAG:

      return KheSrsExprFloatDevDomTest((KHE_SRS_EXPR_FLOAT_DEV) e);

    case KHE_SRS_EXPR_COST_TAG:

      return KheSrsExprCostDomTest((KHE_SRS_EXPR_COST) e);

    case KHE_SRS_EXPR_COST_SUM_TAG:

      return KheSrsExprCostSumDomTest((KHE_SRS_EXPR_COST_SUM) e);

    case KHE_SRS_EXPR_INT_SUM_COMB_TAG:

      return KheSrsExprIntSumCombDomTest((KHE_SRS_EXPR_INT_SUM_COMB) e);

    case KHE_SRS_EXPR_INT_SEQ_COMB_TAG:

      return KheSrsExprIntSeqCombDomTest((KHE_SRS_EXPR_INT_SEQ_COMB) e);

    default:

      HnAbort("KheSrsExprDomTest internal error: unknown tag %d", e->tag);
      return KheSrsDomTestMake(KHE_SRS_DOM_NONE, 0);  ** keep compiler happy **
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprSetSigIndexes(KHE_SRS_EXPR e,                             */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Set the signature indexes in e and its descendants.                      */
/*                                                                           */
/*  These are assigned in postorder, because that is the order that          */
/*  KheSrsExprEndDay visits the nodes.                                       */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprSetSigIndexes(KHE_SRS_EXPR e,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_EXPR child_e;  int i;  KHE_SRS_DAY day;

  /* find e's dom test, as a function of its type and its parent's dom test */
  /* e->dom_test = KheSrsExprDomTest(e); */

  /* set the signature indexes in e's descendants */
  HaArrayForEach(e->children, child_e, i)
    KheSrsExprSetSigIndexes(child_e, srs);

  /* set the signature indexes in e (NB < is correct, not <= here) */
  if( e->dom_test.type != KHE_SRS_DOM_UNUSED )
    for( i = e->first_day_index;  i < e->last_day_index;  i++ )
    {
      day = HaArray(srs->days, i);
      HaArrayAddLast(e->sig_indexes, day->sig_len++);
      HaArrayAddLast(day->dom_tests, e->dom_test);
    }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheSrsExprHasActiveDays(KHE_SRS_EXPR e)                             */
/*                                                                           */
/*  Return true if e is of a type that has active days, and hence needs      */
/*  to be called by BeginDay and EndDay.                                     */
/*                                                                           */
/*****************************************************************************/

static bool KheSrsExprHasActiveDays(KHE_SRS_EXPR e)
{
  switch( e->tag )
  {
    case KHE_SRS_EXPR_INT_DEV_TAG:
    case KHE_SRS_EXPR_FLOAT_DEV_TAG:
    case KHE_SRS_EXPR_COST_TAG:

      return false;

    default:

      return true;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprAddExpressionsToDays(KHE_SRS_EXPR e,                      */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Add e and its descendants to the days they are active on.  Most          */
/*  importantly, do it in postorder.                                         */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprAddExpressionsToDays(KHE_SRS_EXPR e,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_EXPR child_e;  int i;  KHE_SRS_DAY day;

  /* do it for e's proper descendants (first!) */
  HaArrayForEach(e->children, child_e, i)
    KheSrsExprAddExpressionsToDays(child_e, srs);

  /* do it for e (NB <= is correct, not < here) */
  if( KheSrsExprHasActiveDays(e) )
    for( i = e->first_day_index;  i <= e->last_day_index;  i++ )
    {
      day = HaArray(srs->days, i);
      HaArrayAddLast(day->active_exprs, e);
    }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprDelete(KHE_SRS_EXPR e, KHE_SINGLE_RESOURCE_SOLVER srs)    */
/*                                                                           */
/*  Delete e and its descendants.                                            */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprDelete(KHE_SRS_EXPR e, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  switch( e->tag )
  {
    case KHE_SRS_EXPR_BUSY_TAG:

      KheSrsExprBusyDelete((KHE_SRS_EXPR_BUSY) e, srs);
      break;

    case KHE_SRS_EXPR_FREE_TAG:

      KheSrsExprFreeDelete((KHE_SRS_EXPR_FREE) e, srs);
      break;

    case KHE_SRS_EXPR_WORK_TAG:

      KheSrsExprWorkDelete((KHE_SRS_EXPR_WORK) e, srs);
      break;

    case KHE_SRS_EXPR_OR_TAG:

      KheSrsExprOrDelete((KHE_SRS_EXPR_OR) e, srs);
      break;

    case KHE_SRS_EXPR_AND_TAG:

      KheSrsExprAndDelete((KHE_SRS_EXPR_AND) e, srs);
      break;

    case KHE_SRS_EXPR_INT_SUM_TAG:

      KheSrsExprIntSumDelete((KHE_SRS_EXPR_INT_SUM) e, srs);
      break;

    /* *** currently unused
    case KHE_SRS_EXPR_INT_SEQ_TAG:

      KheSrsExprIntSeqDelete((KHE_SRS_EXPR_INT_SEQ) e, srs);
      break;
    *** */

    case KHE_SRS_EXPR_FLOAT_SUM_TAG:

      KheSrsExprFloatSumDelete((KHE_SRS_EXPR_FLOAT_SUM) e, srs);
      break;

    case KHE_SRS_EXPR_INT_DEV_TAG:

      KheSrsExprIntDevDelete((KHE_SRS_EXPR_INT_DEV) e, srs);
      break;

    case KHE_SRS_EXPR_FLOAT_DEV_TAG:

      KheSrsExprFloatDevDelete((KHE_SRS_EXPR_FLOAT_DEV) e, srs);
      break;

    case KHE_SRS_EXPR_COST_TAG:

      KheSrsExprCostDelete((KHE_SRS_EXPR_COST) e, srs);
      break;

    case KHE_SRS_EXPR_COST_SUM_TAG:

      KheSrsExprCostSumDelete((KHE_SRS_EXPR_COST_SUM) e, srs);
      break;

    case KHE_SRS_EXPR_INT_SUM_COMB_TAG:

      KheSrsExprIntSumCombDelete((KHE_SRS_EXPR_INT_SUM_COMB) e, srs);
      break;

    case KHE_SRS_EXPR_INT_SEQ_COMB_TAG:

      KheSrsExprIntSeqCombDelete((KHE_SRS_EXPR_INT_SEQ_COMB) e, srs);
      break;

    default:

      HnAbort("KheSrsExprDelete internal error: unknown tag %d", e->tag);
      break;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprBeginDay(KHE_SRS_EXPR e, int day_index,                   */
/*    KHE_SRS_PARTIAL_SOLN prev_ps, KHE_SINGLE_RESOURCE_SOLVER srs)          */
/*                                                                           */
/*  Inform e that day day_index is beginning.                                */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprBeginDay(KHE_SRS_EXPR e, int day_index,
  KHE_SRS_PARTIAL_SOLN prev_ps, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  switch( e->tag )
  {
    case KHE_SRS_EXPR_BUSY_TAG:

      KheSrsExprBusyBeginDay((KHE_SRS_EXPR_BUSY) e, day_index, prev_ps, srs);
      break;

    case KHE_SRS_EXPR_FREE_TAG:

      KheSrsExprFreeBeginDay((KHE_SRS_EXPR_FREE) e, day_index, prev_ps, srs);
      break;

    case KHE_SRS_EXPR_WORK_TAG:

      KheSrsExprWorkBeginDay((KHE_SRS_EXPR_WORK) e, day_index, prev_ps, srs);
      break;

    case KHE_SRS_EXPR_OR_TAG:

      KheSrsExprOrBeginDay((KHE_SRS_EXPR_OR) e, day_index, prev_ps, srs);
      break;

    case KHE_SRS_EXPR_AND_TAG:

      KheSrsExprAndBeginDay((KHE_SRS_EXPR_AND) e, day_index, prev_ps, srs);
      break;

    case KHE_SRS_EXPR_INT_SUM_TAG:

      KheSrsExprIntSumBeginDay((KHE_SRS_EXPR_INT_SUM) e, day_index,
	prev_ps, srs);
      break;

    /* *** currently unused
    case KHE_SRS_EXPR_INT_SEQ_TAG:

      KheSrsExprIntSeqBeginDay((KHE_SRS_EXPR_INT_SEQ) e, day_index,
	prev_ps, srs);
      break;
    *** */

    case KHE_SRS_EXPR_FLOAT_SUM_TAG:

      KheSrsExprFloatSumBeginDay((KHE_SRS_EXPR_FLOAT_SUM) e, day_index,
	prev_ps, srs);
      break;

    case KHE_SRS_EXPR_INT_DEV_TAG:

      KheSrsExprIntDevBeginDay((KHE_SRS_EXPR_INT_DEV) e, day_index,
	prev_ps, srs);
      break;

    case KHE_SRS_EXPR_FLOAT_DEV_TAG:

      KheSrsExprFloatDevBeginDay((KHE_SRS_EXPR_FLOAT_DEV) e, day_index,
	prev_ps, srs);
      break;

    case KHE_SRS_EXPR_COST_TAG:

      KheSrsExprCostBeginDay((KHE_SRS_EXPR_COST) e, day_index, prev_ps, srs);
      break;

    case KHE_SRS_EXPR_COST_SUM_TAG:

      KheSrsExprCostSumBeginDay((KHE_SRS_EXPR_COST_SUM) e, day_index,
	prev_ps, srs);
      break;

    case KHE_SRS_EXPR_INT_SUM_COMB_TAG:

      KheSrsExprIntSumCombBeginDay((KHE_SRS_EXPR_INT_SUM_COMB) e, day_index,
	prev_ps, srs);
      break;

    case KHE_SRS_EXPR_INT_SEQ_COMB_TAG:

      KheSrsExprIntSeqCombBeginDay((KHE_SRS_EXPR_INT_SEQ_COMB) e, day_index,
	prev_ps, srs);
      break;

    default:

      HnAbort("KheSrsExprBeginDay internal error");
      break;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprReceiveValueLeaf(KHE_SRS_EXPR e,                          */
/*    KHE_TASK task, KHE_SINGLE_RESOURCE_SOLVER srs)                         */
/*                                                                           */
/*  Inform leaf e that task (possibly NULL) is running at its time.          */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprReceiveValueLeaf(KHE_SRS_EXPR e,
  KHE_TASK task, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  switch( e->tag )
  {
    case KHE_SRS_EXPR_BUSY_TAG:

      KheSrsExprBusyReceiveValueLeaf((KHE_SRS_EXPR_BUSY) e, task, srs);
      break;

    case KHE_SRS_EXPR_FREE_TAG:

      KheSrsExprFreeReceiveValueLeaf((KHE_SRS_EXPR_FREE) e, task, srs);
      break;

    case KHE_SRS_EXPR_WORK_TAG:

      KheSrsExprWorkReceiveValueLeaf((KHE_SRS_EXPR_WORK) e, task, srs);
      break;

    default:

      HnAbort("KheSrsExprReceiveValueLeaf internal error (tag %d)", e->tag);
      break;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprReceiveValueInt(KHE_SRS_EXPR e, int value,                */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Inform e that one of its children has just finalized its integer value.  */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprReceiveValueInt(KHE_SRS_EXPR e, int value,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  switch( e->tag )
  {
    case KHE_SRS_EXPR_OR_TAG:

      KheSrsExprOrReceiveValueInt((KHE_SRS_EXPR_OR) e, value, srs);
      break;

    case KHE_SRS_EXPR_AND_TAG:

      KheSrsExprAndReceiveValueInt((KHE_SRS_EXPR_AND) e, value, srs);
      break;

    case KHE_SRS_EXPR_INT_SUM_TAG:

      KheSrsExprIntSumReceiveValueInt((KHE_SRS_EXPR_INT_SUM) e, value, srs);
      break;

    /* *** currently unused
    case KHE_SRS_EXPR_INT_SEQ_TAG:

      KheSrsExprIntSeqReceiveValueInt((KHE_SRS_EXPR_INT_SEQ) e, value, srs);
      break;
    *** */

    case KHE_SRS_EXPR_INT_DEV_TAG:

      KheSrsExprIntDevReceiveValueInt((KHE_SRS_EXPR_INT_DEV) e, value, srs);
      break;

    case KHE_SRS_EXPR_COST_TAG:

      KheSrsExprCostReceiveValueInt((KHE_SRS_EXPR_COST) e, value, srs);
      break;

    case KHE_SRS_EXPR_INT_SUM_COMB_TAG:

      KheSrsExprIntSumCombReceiveValueInt((KHE_SRS_EXPR_INT_SUM_COMB) e,
	value, srs);
      break;

    case KHE_SRS_EXPR_INT_SEQ_COMB_TAG:

      KheSrsExprIntSeqCombReceiveValueInt((KHE_SRS_EXPR_INT_SEQ_COMB) e,
	value, srs);
      break;

    default:

      HnAbort("KheSrsExprReceiveValueInt internal error (tag %d)", e->tag);
      break;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprReceiveValueFloat(KHE_SRS_EXPR e, float value,            */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Inform e that one of its children has just finalized its float value.    */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprReceiveValueFloat(KHE_SRS_EXPR e, float value,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  switch( e->tag )
  {
    case KHE_SRS_EXPR_FLOAT_SUM_TAG:

      KheSrsExprFloatSumReceiveValueFloat((KHE_SRS_EXPR_FLOAT_SUM) e,
	value, srs);
      break;

    case KHE_SRS_EXPR_FLOAT_DEV_TAG:

      KheSrsExprFloatDevReceiveValueFloat((KHE_SRS_EXPR_FLOAT_DEV) e,
	value, srs);
      break;

    default:

      HnAbort("KheSrsExprReceiveValueFloat internal error (tag %d)", e->tag);
      break;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprReceiveValueCost(KHE_SRS_EXPR e, KHE_COST value,          */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Inform e that one of its children has just finalized its cost value.     */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprReceiveValueCost(KHE_SRS_EXPR e, KHE_COST value,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  switch( e->tag )
  {
    case KHE_SRS_EXPR_COST_SUM_TAG:

      KheSrsExprCostSumReceiveValueCost((KHE_SRS_EXPR_COST_SUM) e, value, srs);
      break;

    default:

      HnAbort("KheSrsExprReceiveValueCost internal error (tag %d)", e->tag);
      break;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprEndDay(KHE_SRS_EXPR e, int day_index,                     */
/*    KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs)               */
/*                                                                           */
/*  Inform e that day day_index is ending.                                   */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprEndDay(KHE_SRS_EXPR e, int day_index,
  KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  switch( e->tag )
  {
    case KHE_SRS_EXPR_BUSY_TAG:

      KheSrsExprBusyEndDay((KHE_SRS_EXPR_BUSY) e, day_index, ps, srs);
      break;

    case KHE_SRS_EXPR_FREE_TAG:

      KheSrsExprFreeEndDay((KHE_SRS_EXPR_FREE) e, day_index, ps, srs);
      break;

    case KHE_SRS_EXPR_WORK_TAG:

      KheSrsExprWorkEndDay((KHE_SRS_EXPR_WORK) e, day_index, ps, srs);
      break;

    case KHE_SRS_EXPR_OR_TAG:

      KheSrsExprOrEndDay((KHE_SRS_EXPR_OR) e, day_index, ps, srs);
      break;

    case KHE_SRS_EXPR_AND_TAG:

      KheSrsExprAndEndDay((KHE_SRS_EXPR_AND) e, day_index, ps, srs);
      break;

    case KHE_SRS_EXPR_INT_SUM_TAG:

      KheSrsExprIntSumEndDay((KHE_SRS_EXPR_INT_SUM) e, day_index, ps, srs);
      break;

    /* *** currently unused
    case KHE_SRS_EXPR_INT_SEQ_TAG:

      KheSrsExprIntSeqEndDay((KHE_SRS_EXPR_INT_SEQ) e, day_index, ps, srs);
      break;
    *** */

    case KHE_SRS_EXPR_FLOAT_SUM_TAG:

      KheSrsExprFloatSumEndDay((KHE_SRS_EXPR_FLOAT_SUM) e, day_index, ps, srs);
      break;

    case KHE_SRS_EXPR_INT_DEV_TAG:

      KheSrsExprIntDevEndDay((KHE_SRS_EXPR_INT_DEV) e, day_index, ps, srs);
      break;

    case KHE_SRS_EXPR_FLOAT_DEV_TAG:

      KheSrsExprFloatDevEndDay((KHE_SRS_EXPR_FLOAT_DEV) e, day_index, ps, srs);
      break;

    case KHE_SRS_EXPR_COST_TAG:

      KheSrsExprCostEndDay((KHE_SRS_EXPR_COST) e, day_index, ps, srs);
      break;

    case KHE_SRS_EXPR_COST_SUM_TAG:

      KheSrsExprCostSumEndDay((KHE_SRS_EXPR_COST_SUM) e, day_index, ps, srs);
      break;

    case KHE_SRS_EXPR_INT_SUM_COMB_TAG:

      KheSrsExprIntSumCombEndDay((KHE_SRS_EXPR_INT_SUM_COMB) e, day_index,
	ps, srs);
      break;

    case KHE_SRS_EXPR_INT_SEQ_COMB_TAG:

      KheSrsExprIntSeqCombEndDay((KHE_SRS_EXPR_INT_SEQ_COMB) e, day_index,
	ps, srs);
      break;

    default:

      HnAbort("KheSrsExprEndDay internal error");
      break;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprHeaderDebug(KHE_SRS_EXPR e,                               */
/*    KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs,               */
/*    int verbosity, FILE *fp)                                               */
/*                                                                           */
/*  I forget.                                                                */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprHeaderDebug(KHE_SRS_EXPR e,
  KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs,
  int verbosity, FILE *fp)
{
  switch( e->tag )
  {
    case KHE_SRS_EXPR_BUSY_TAG:

      KheSrsExprBusyHeaderDebug((KHE_SRS_EXPR_BUSY) e, ps,
	srs, verbosity, fp);
      break;

    case KHE_SRS_EXPR_FREE_TAG:

      KheSrsExprFreeHeaderDebug((KHE_SRS_EXPR_FREE) e, ps,
	srs, verbosity, fp);
      break;

    case KHE_SRS_EXPR_WORK_TAG:

      KheSrsExprWorkHeaderDebug((KHE_SRS_EXPR_WORK) e, ps,
	srs, verbosity, fp);
      break;

    case KHE_SRS_EXPR_OR_TAG:

      KheSrsExprOrHeaderDebug((KHE_SRS_EXPR_OR) e, ps,
	srs, verbosity, fp);
      break;

    case KHE_SRS_EXPR_AND_TAG:

      KheSrsExprAndHeaderDebug((KHE_SRS_EXPR_AND) e, ps,
	srs, verbosity, fp);
      break;

    case KHE_SRS_EXPR_INT_SUM_TAG:

      KheSrsExprIntSumHeaderDebug((KHE_SRS_EXPR_INT_SUM) e, ps,
	srs, verbosity, fp);
      break;

    /* *** currently unused
    case KHE_SRS_EXPR_INT_SEQ_TAG:

      KheSrsExprIntSeqHeaderDebug((KHE_SRS_EXPR_INT_SEQ) e, ps,
    srs, verbosity, fp);
      break;
    *** */

    case KHE_SRS_EXPR_FLOAT_SUM_TAG:

      KheSrsExprFloatSumHeaderDebug((KHE_SRS_EXPR_FLOAT_SUM) e, ps,
	srs, verbosity, fp);
      break;

    case KHE_SRS_EXPR_INT_DEV_TAG:

      KheSrsExprIntDevHeaderDebug((KHE_SRS_EXPR_INT_DEV) e, ps,
	srs, verbosity, fp);
      break;

    case KHE_SRS_EXPR_FLOAT_DEV_TAG:

      KheSrsExprFloatDevHeaderDebug((KHE_SRS_EXPR_FLOAT_DEV) e, ps,
	srs, verbosity, fp);
      break;

    case KHE_SRS_EXPR_COST_TAG:

      KheSrsExprCostHeaderDebug((KHE_SRS_EXPR_COST) e, ps,
	srs, verbosity, fp);
      break;

    case KHE_SRS_EXPR_COST_SUM_TAG:

      KheSrsExprCostSumHeaderDebug((KHE_SRS_EXPR_COST_SUM) e, ps,
	srs, verbosity, fp);
      break;

    case KHE_SRS_EXPR_INT_SUM_COMB_TAG:

      KheSrsExprIntSumCombHeaderDebug((KHE_SRS_EXPR_INT_SUM_COMB) e, ps,
	srs, verbosity, fp);
      break;

    case KHE_SRS_EXPR_INT_SEQ_COMB_TAG:

      KheSrsExprIntSeqCombHeaderDebug((KHE_SRS_EXPR_INT_SEQ_COMB) e, ps,
	srs, verbosity, fp);
      break;

    default:

      HnAbort("KheSrsExprHeaderDebug internal error");
      break;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprDebugPartialSoln(KHE_SRS_EXPR e, KHE_SRS_PARTIAL_SOLN ps, */
/*    KHE_SINGLE_RESOURCE_SOLVER srs, int verbosity, int indent, FILE *fp)   */
/*                                                                           */
/*  Debug print of e's state under ps, if active.                            */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprDebugPartialSoln(KHE_SRS_EXPR e, KHE_SRS_PARTIAL_SOLN ps,
  KHE_SINGLE_RESOURCE_SOLVER srs, int verbosity, FILE *fp)
{
  KHE_SRS_EXPR child_e;  int i;  bool child_printed;
  KheSrsExprHeaderDebug(e, ps, srs, verbosity, fp);
  if( HaArrayCount(e->children) > 0 )
  {
    fprintf(fp, "(");
    child_printed = false;
    HaArrayForEach(e->children, child_e, i)
      if( child_e->first_day_index <= ps->day_index &&
	  ps->day_index < child_e->last_day_index )
      {
	if( child_printed )
	  fprintf(fp, ", ");
        KheSrsExprDebugPartialSoln(child_e, ps, srs, verbosity, fp);
	child_printed = true;
      }
    fprintf(fp, ")");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprDebug(KHE_SRS_EXPR e, KHE_SINGLE_RESOURCE_SOLVER srs,     */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of e onto fp with the given verbosity and indent.            */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprDebug(KHE_SRS_EXPR e, KHE_SINGLE_RESOURCE_SOLVER srs,
  int verbosity, int indent, FILE *fp)
{
  switch( e->tag )
  {
    case KHE_SRS_EXPR_BUSY_TAG:

      KheSrsExprBusyDebug((KHE_SRS_EXPR_BUSY) e, srs, verbosity, indent, fp);
      break;

    case KHE_SRS_EXPR_FREE_TAG:

      KheSrsExprFreeDebug((KHE_SRS_EXPR_FREE) e, srs, verbosity, indent, fp);
      break;

    case KHE_SRS_EXPR_WORK_TAG:

      KheSrsExprWorkDebug((KHE_SRS_EXPR_WORK) e, srs, verbosity, indent, fp);
      break;

    case KHE_SRS_EXPR_OR_TAG:

      KheSrsExprOrDebug((KHE_SRS_EXPR_OR) e, srs, verbosity, indent, fp);
      break;

    case KHE_SRS_EXPR_AND_TAG:

      KheSrsExprAndDebug((KHE_SRS_EXPR_AND) e, srs, verbosity, indent, fp);
      break;

    case KHE_SRS_EXPR_INT_SUM_TAG:

      KheSrsExprIntSumDebug((KHE_SRS_EXPR_INT_SUM) e, srs, verbosity,
	indent, fp);
      break;

    /* *** currently unused
    case KHE_SRS_EXPR_INT_SEQ_TAG:

      KheSrsExprIntSeqDebug((KHE_SRS_EXPR_INT_SEQ) e, srs, verbosity,
	indent, fp);
      break;
    *** */

    case KHE_SRS_EXPR_FLOAT_SUM_TAG:

      KheSrsExprFloatSumDebug((KHE_SRS_EXPR_FLOAT_SUM) e, srs, verbosity,
	indent,fp);
      break;

    case KHE_SRS_EXPR_INT_DEV_TAG:

      KheSrsExprIntDevDebug((KHE_SRS_EXPR_INT_DEV) e, srs, verbosity,
	indent, fp);
      break;

    case KHE_SRS_EXPR_FLOAT_DEV_TAG:

      KheSrsExprFloatDevDebug((KHE_SRS_EXPR_FLOAT_DEV) e, srs, verbosity,
	indent,fp);
      break;

    case KHE_SRS_EXPR_COST_TAG:

      KheSrsExprCostDebug((KHE_SRS_EXPR_COST) e, srs, verbosity, indent, fp);
      break;

    case KHE_SRS_EXPR_COST_SUM_TAG:

      KheSrsExprCostSumDebug((KHE_SRS_EXPR_COST_SUM) e, srs, verbosity,
	indent, fp);
      break;

    case KHE_SRS_EXPR_INT_SUM_COMB_TAG:

      KheSrsExprIntSumCombDebug((KHE_SRS_EXPR_INT_SUM_COMB) e, srs, verbosity,
	indent, fp);
      break;

    case KHE_SRS_EXPR_INT_SEQ_COMB_TAG:

      KheSrsExprIntSeqCombDebug((KHE_SRS_EXPR_INT_SEQ_COMB) e, srs, verbosity,
	indent, fp);
      break;

    default:

      HnAbort("KheSrsExprDebug internal error: unknown tag %d", e->tag);
      break;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_SRS_STEP"                                                 */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_STEP KheSrsStepGet(KHE_SINGLE_RESOURCE_SOLVER srs)               */
/*                                                                           */
/*  Get a new srs task object.  Only the times_by_day and tasks_by day       */
/*  fields are initialized.                                                  */
/*                                                                           */
/*****************************************************************************/

static KHE_SRS_STEP KheSrsStepGet(KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_STEP res;
  if( HaArrayCount(srs->free_steps) > 0 )
    res = HaArrayLastAndDelete(srs->free_steps);
  else
    HaMake(res, srs->arena);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_STEP KheSrsStepMake(KHE_TASK task,                               */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Make a new srs step from task.                                           */
/*                                                                           */
/*  This code aborts if task has no assigned times.  That is safe given      */
/*  the context of the sole call to this function below.                     */
/*                                                                           */
/*****************************************************************************/

static KHE_SRS_STEP KheSrsStepMake(KHE_TASK task,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_STEP res;
  res = KheSrsStepGet(srs);
  res->task = task;
  res->daily_schedule = KheTaskFinderTaskDailySchedule(srs->task_finder, task);
  res->cost_reduction = KheTaskAssignmentCostReduction(task);
  res->domain_count = KheResourceGroupResourceCount(KheTaskDomain(task));
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_STEP KheSrsStepMakeFree(int day_index,                           */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Make a new srs step which represents doing nothing on the day with       */
/*  this day_index.                                                          */
/*                                                                           */
/*****************************************************************************/

static KHE_SRS_STEP KheSrsStepMakeFree(int day_index,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_STEP res;
  res = KheSrsStepGet(srs);
  res->task = NULL;
  res->daily_schedule = KheTaskFinderNullDailySchedule(srs->task_finder,
    day_index, day_index);
  res->cost_reduction = 0;
  res->domain_count = 0;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsStepDelete(KHE_SRS_STEP step, KHE_SINGLE_RESOURCE_SOLVER srs) */
/*                                                                           */
/*  Delete step.                                                             */
/*                                                                           */
/*****************************************************************************/

static void KheSrsStepDelete(KHE_SRS_STEP step, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KheDailyScheduleDelete(step->daily_schedule);
  HaArrayAddLast(srs->free_steps, step);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_TIME KheSrsStepFirstTime(KHE_SRS_STEP step)                          */
/*                                                                           */
/*  Return the first time of step.                                           */
/*                                                                           */
/*****************************************************************************/

static KHE_TIME KheSrsStepFirstTime(KHE_SRS_STEP step)
{
  int index;
  index = KheDailyScheduleFirstDayIndex(step->daily_schedule);
  return KheDailyScheduleTime(step->daily_schedule, index);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_TIME KheSrsStepLastTime(KHE_SRS_STEP step)                           */
/*                                                                           */
/*  Return the last time of step.                                            */
/*                                                                           */
/*****************************************************************************/

static KHE_TIME KheSrsStepLastTime(KHE_SRS_STEP step)
{
  int index;
  index = KheDailyScheduleLastDayIndex(step->daily_schedule);
  return KheDailyScheduleTime(step->daily_schedule, index);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheSrsStepBetterThan(KHE_SRS_STEP step1, KHE_SRS_STEP step2)        */
/*                                                                           */
/*  Return true if step1 is a better choice than step2, because it has a     */
/*  larger cost reduction, or equal cost reduction and a smaller domain.     */
/*                                                                           */
/*****************************************************************************/

static bool KheSrsStepBetterThan(KHE_SRS_STEP step1, KHE_SRS_STEP step2)
{
  return step1->cost_reduction > step2->cost_reduction ||
    (step1->cost_reduction == step2->cost_reduction &&
     step1->domain_count < step2->domain_count);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsStepDebug(KHE_SRS_STEP step, int verbosity,                   */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of step onto fp with the given verbosity and indent.         */
/*                                                                           */
/*****************************************************************************/

static void KheSrsStepDebug(KHE_SRS_STEP step, int verbosity,
  int indent, FILE *fp)
{
  KHE_TIME first_time, last_time;  int first_day_index, last_day_index;
  KHE_TASK_FINDER tf;  KHE_TIME_GROUP tg1, tg2;  KHE_FRAME frame;
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  if( step->task != NULL )
  {
    first_time = KheSrsStepFirstTime(step);
    last_time = KheSrsStepLastTime(step);
    KheTaskDebug(step->task, verbosity, -1, fp);
    if( first_time != last_time )
      fprintf(fp, " (%s-%s", KheTimeId(first_time), KheTimeId(last_time));
    else
      fprintf(fp, " (%s", KheTimeId(first_time));
    fprintf(fp, ", cost %.5f, domain %d)",
      KheCostShow(step->cost_reduction), step->domain_count);
  }
  else
  {
    tf = KheDailyScheduleTaskFinder(step->daily_schedule);
    frame = KheTaskFinderFrame(tf);
    first_day_index = KheDailyScheduleFirstDayIndex(step->daily_schedule);
    last_day_index = KheDailyScheduleLastDayIndex(step->daily_schedule);
    tg1 = KheFrameTimeGroup(frame, first_day_index);
    tg2 = KheFrameTimeGroup(frame, last_day_index);
    if( tg1 != tg2 )
      fprintf(fp, "Free (%s-%s)", KheTimeGroupId(tg1), KheTimeGroupId(tg2));
    else
      fprintf(fp, "Free (%s)", KheTimeGroupId(tg1));
  }
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_SRS_TIME"                                                 */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_TIME KheSrsTimeMake(KHE_TIME t, HA_ARENA a)                      */
/*                                                                           */
/*  Make and return a new srs time object representing t.                    */
/*                                                                           */
/*****************************************************************************/

static KHE_SRS_TIME KheSrsTimeMake(KHE_TIME t, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_TIME res;
  HaMake(res, srs->arena);
  res->time = t;
  HaArrayInit(res->leaf_exprs, srs->arena);
  HaArrayInit(res->best_steps, srs->arena);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsTimeClearTasks(KHE_SRS_TIME st,                               */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Clear the tasks of st, but not the leaf expressions.                     */
/*                                                                           */
/*****************************************************************************/

static void KheSrsTimeClearTasks(KHE_SRS_TIME st,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  int i;  KHE_SRS_STEP step;
  HaArrayForEach(st->best_steps, step, i)
    KheSrsStepDelete(step, srs);
  HaArrayClear(st->best_steps);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsTimeClear(KHE_SRS_TIME st, KHE_SINGLE_RESOURCE_SOLVER srs)    */
/*                                                                           */
/*  Clear st, that is, clear out its expressions and tasks.                  */
/*                                                                           */
/*  There is no function to delete st because that is never wanted.  And     */
/*  the expressions are deleted elsewhere, so they are not deleted here.     */
/*  But the tasks are not deleted elsewhere, so they are deleted here.       */
/*                                                                           */
/*****************************************************************************/

static void KheSrsTimeClear(KHE_SRS_TIME st, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  HaArrayClear(st->leaf_exprs);
  KheSrsTimeClearTasks(st, srs);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheSrsTimeContainsLastTime(KHE_SRS_TIME stime,                      */
/*    KHE_TIME last_time, KHE_SRS_STEP *step, int *pos)                      */
/*                                                                           */
/*  If stime contains a task with the given last_time, return true with      */
/*  *step set to that time and *pos set to its position.  Otherwise return   */
/*  false with *step set to NULL and *pos set to -1.                         */
/*                                                                           */
/*****************************************************************************/

static bool KheSrsTimeContainsLastTime(KHE_SRS_TIME stime,
  KHE_TIME last_time, KHE_SRS_STEP *step, int *pos)
{
  KHE_SRS_STEP step2;  int i;
  HaArrayForEach(stime->best_steps, step2, i)
    if( KheSrsStepLastTime(step2) == last_time )
      return *step = step2, *pos = i, true;
  return *step = NULL, *pos = -1, false;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsTimeAddTask(KHE_SRS_TIME st, KHE_SRS_STEP step,               */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Add step to st.  More precisely, if it's better than what st already     */
/*  has, then add it, otherwise delete it.                                   */
/*                                                                           */
/*****************************************************************************/

static void KheSrsTimeAddTask(KHE_SRS_TIME st, KHE_SRS_STEP step,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_STEP step2;  int pos;
  if( !KheSrsTimeContainsLastTime(st, KheSrsStepLastTime(step), &step2, &pos) )
  {
    /* st has no task with the same last time as step, so add step to st */
    HaArrayAddLast(st->best_steps, step);
  }
  else if( KheSrsStepBetterThan(step, step2) )
  {
    /* step is better than the similar task step2, so replace step2 by step */
    KheSrsStepDelete(step2, srs);
    HaArrayPut(st->best_steps, pos, step);
  }
  else
  {
    /* step is worse than the similar step2, so delete step */
    KheSrsStepDelete(step, srs);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsTimeDebug(KHE_SRS_TIME st, KHE_SINGLE_RESOURCE_SOLVER srs,    */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of st onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void KheSrsTimeDebug(KHE_SRS_TIME st, KHE_SINGLE_RESOURCE_SOLVER srs,
  int verbosity, int indent, FILE *fp)
{
  int i;  KHE_SRS_EXPR e;  KHE_SRS_STEP step;
  if( indent >= 0 )
  {
    fprintf(fp, "%*s[ Time %s:\n", indent, "", KheTimeId(st->time));
    HaArrayForEach(st->leaf_exprs, e, i)
      KheSrsExprDebug(e, srs, verbosity, indent + 2, fp);
    HaArrayForEach(st->best_steps, step, i)
      KheSrsStepDebug(step, verbosity, indent + 2, fp);
    fprintf(fp, "%*s]\n", indent, "");
  }
  else
    fprintf(fp, "[ Time %s (%d expressions) ]", KheTimeId(st->time),
      HaArrayCount(st->leaf_exprs));
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_SRS_PARTIAL_SOLN"                                         */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_PARTIAL_SOLN KheSrsPartialSolnGet(KHE_SINGLE_RESOURCE_SOLVER srs)*/
/*                                                                           */
/*  Get a partial soln object.  Only the signature field is initialized.     */
/*                                                                           */
/*****************************************************************************/

static KHE_SRS_PARTIAL_SOLN KheSrsPartialSolnGet(KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_PARTIAL_SOLN res;
  if( HaArrayCount(srs->free_partial_solns) > 0 )
  {
    res = HaArrayLastAndDelete(srs->free_partial_solns);
    HaArrayClear(res->signature);
  }
  else
  {
    HaMake(res, srs->arena);
    HaArrayInit(res->signature, srs->arena);
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_PARTIAL_SOLN KheSrsPartialSolnMakeInit(                          */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Make the partial solution that the search grows from.  It does not       */
/*  end on any day.                                                          */
/*                                                                           */
/*****************************************************************************/

static KHE_SRS_PARTIAL_SOLN KheSrsPartialSolnMakeInit(
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_PARTIAL_SOLN res;
  res = KheSrsPartialSolnGet(srs);
  res->prev_ps = NULL;
  res->step = NULL;
  res->best_of = 1;
  res->day_index = -1;
  res->cost = 0;
  HaArrayAddLast(res->signature, 0);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_PARTIAL_SOLN KheSrsPartialSolnMake(KHE_SRS_PARTIAL_SOLN prev_ps, */
/*    KHE_SRS_STEP step, int day_index, KHE_SINGLE_RESOURCE_SOLVER srs)      */
/*                                                                           */
/*  Make a new partial solution with these attributes.                       */
/*                                                                           */
/*****************************************************************************/
static void KheSrsPartialSolnDebug(KHE_SRS_PARTIAL_SOLN ps,
  KHE_SINGLE_RESOURCE_SOLVER srs, int verbosity, int indent, FILE *fp);

static KHE_SRS_PARTIAL_SOLN KheSrsPartialSolnMake(KHE_SRS_PARTIAL_SOLN prev_ps,
  KHE_SRS_STEP step, int day_index, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_PARTIAL_SOLN res;  int durn, i, j;  KHE_TIME time;
  KHE_SRS_DAY day;  KHE_SRS_EXPR e;  KHE_SRS_TIME stime;

  /* make the basic object */
  HnAssert(step != NULL, "KheSrsPartialSolnMake internal error (step)");
  if( DEBUG4 )
    fprintf(stderr, "[ KheSrsPartialSolnMake(prev_ps, %s, %s, srs)\n",
      step->task == NULL ? "-" : "task",
      KheTimeGroupId(KheFrameTimeGroup(srs->days_frame, day_index)));
  res = KheSrsPartialSolnGet(srs);
  res->prev_ps = prev_ps;
  res->step = step;  /* in the free case, st's task will be null */
  res->best_of = 1;
  res->day_index = day_index;

  /* set the cost and signature */
  res->cost = 0;
  durn = HaArrayFirst(prev_ps->signature) + (step->task != NULL ? 1 : 0);
  HaArrayAddLast(res->signature, durn);
  day = HaArray(srs->days, day_index);
  HaArrayForEach(day->active_exprs, e, i)
    KheSrsExprBeginDay(e, day_index, prev_ps, srs);
  time = KheDailyScheduleTime(step->daily_schedule, day_index);
  HaArrayForEach(day->times, stime, i)
  {
    if( stime->time == time )
      HaArrayForEach(stime->leaf_exprs, e, j)
	KheSrsExprReceiveValueLeaf(e, step->task, srs);
    else
      HaArrayForEach(stime->leaf_exprs, e, j)
	KheSrsExprReceiveValueLeaf(e, NULL, srs);
  }
  HaArrayForEach(day->active_exprs, e, i)
    KheSrsExprEndDay(e, day_index, res, srs);
  if( DEBUG4 )
  {
    KheSrsPartialSolnDebug(res, srs, 1, 2, stderr);
    fprintf(stderr, "] PartialSolnMake returning\n");
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsPartialSolnDelete(KHE_SRS_PARTIAL_SOLN ps,                    */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Delete ps.                                                               */
/*                                                                           */
/*****************************************************************************/

static void KheSrsPartialSolnDelete(KHE_SRS_PARTIAL_SOLN ps,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  HaArrayAddLast(srs->free_partial_solns, ps);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheSrsPartialSolnSignatureHash(KHE_SRS_PARTIAL_SOLN ps)              */
/*                                                                           */
/*  Hash function for hashing the signature of ps.                           */
/*                                                                           */
/*****************************************************************************/

static int KheSrsPartialSolnSignatureHash(KHE_SRS_PARTIAL_SOLN ps)
{
  int res, val, i;
  res = 0;
  HaArrayForEach(ps->signature, val, i)
  {
    res = (res << 3) + val;
    if( i > 8 )
      break;
  }
  return (res < 0 ? - res : res);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheSrsPartialSolnSignatureHashUntyped(void *p)                       */
/*                                                                           */
/*  Untyped version of KheSrsPartialSolnSignatureHash.                       */
/*                                                                           */
/*****************************************************************************/

static int KheSrsPartialSolnSignatureHashUntyped(void *p)
{
  return KheSrsPartialSolnSignatureHash((KHE_SRS_PARTIAL_SOLN) p);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheSrsPartialSolnSignaturesEqual(KHE_SRS_PARTIAL_SOLN ps1,          */
/*    KHE_SRS_PARTIAL_SOLN ps2)                                              */
/*                                                                           */
/*  Return true if the signatures of ps1 and ps2 are equal.  The lengths     */
/*  of their signatures must be equal.                                       */
/*                                                                           */
/*****************************************************************************/

static bool KheSrsPartialSolnSignaturesEqual(KHE_SRS_PARTIAL_SOLN ps1,
  KHE_SRS_PARTIAL_SOLN ps2)
{
  int i;
  HnAssert(HaArrayCount(ps1->signature) == HaArrayCount(ps2->signature),
    "KheSrsPartialSolnSignaturesEqual: signatures have different lengths");
  for( i = 0;  i < HaArrayCount(ps1->signature);  i++ )
    if( HaArray(ps1->signature, i) != HaArray(ps2->signature, i) )
      return false;
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheSrsPartialSolnSignaturesEqualUntyped(void *p1, void *p2)         */
/*                                                                           */
/*  Untyped version of KheSrsPartialSolnSignaturesEqual.                     */
/*                                                                           */
/*****************************************************************************/

static bool KheSrsPartialSolnSignaturesEqualUntyped(void *p1, void *p2)
{
  return KheSrsPartialSolnSignaturesEqual((KHE_SRS_PARTIAL_SOLN) p1,
    (KHE_SRS_PARTIAL_SOLN) p2);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheSrsPartialSolnDominates(KHE_SRS_PARTIAL_SOLN ps1,                */
/*    KHE_SRS_PARTIAL_SOLN ps2, KHE_SRS_DAY day)                             */
/*                                                                           */
/*  Return true if ps1 dominates ps2.  Both solutions come from day.         */
/*                                                                           */
/*****************************************************************************/

static bool KheSrsPartialSolnDominates(KHE_SRS_PARTIAL_SOLN ps1,
  KHE_SRS_PARTIAL_SOLN ps2, KHE_SRS_DAY day)
{
  KHE_SRS_DOM_TEST dt;  int i, val1, val2;
  if( ps2->cost < ps1->cost )
    return false;
  for( i = 0;  i < day->sig_len;  i++ )
  {
    dt = HaArray(day->dom_tests, i);
    val1 = HaArray(ps1->signature, i);
    val2 = HaArray(ps2->signature, i);
    if( !KheSrsDomTestDominates(dt, val1, val2) )
      return false;
  }
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsPartialSolnOverWrite(KHE_SRS_PARTIAL_SOLN dest_ps,            */
/*    KHE_SRS_PARTIAL_SOLN src_ps)                                           */
/*                                                                           */
/*  Overwrite src_ps onto dest_ps.  Their signatures are known to be         */
/*  equal, so those don't need to be touched.  Also the best_of field        */
/*  is not touched here; it gets updated separately.                         */
/*                                                                           */
/*****************************************************************************/

static void KheSrsPartialSolnOverWrite(KHE_SRS_PARTIAL_SOLN dest_ps,
  KHE_SRS_PARTIAL_SOLN src_ps)
{
  dest_ps->prev_ps = src_ps->prev_ps;
  dest_ps->step = src_ps->step;
  dest_ps->cost = src_ps->cost;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheSrsPartialSolnCostLessThan(KHE_SRS_PARTIAL_SOLN ps1,             */
/*    KHE_SRS_PARTIAL_SOLN ps2)                                              */
/*                                                                           */
/*  Return true if ps1 is a better solution than ps2.                        */
/*                                                                           */
/*****************************************************************************/

static bool KheSrsPartialSolnCostLessThan(KHE_SRS_PARTIAL_SOLN ps1,
  KHE_SRS_PARTIAL_SOLN ps2)
{
  return ps1->cost < ps2->cost;
    /* || (ps1->r_cost == ps2->r_cost && ps1->soln_cost < ps2->soln_cost); */
}


/*****************************************************************************/
/*                                                                           */
/*  int KheSrsPartialSolnIncreasingAsstCountCmp(const void *p1, void *p2)    */
/*                                                                           */
/*  Comparison function for sorting an array of partial solutions by         */
/*  increasing assignment count.                                             */
/*                                                                           */
/*****************************************************************************/

static int KheSrsPartialSolnIncreasingAsstCountCmp(const void *p1,
  const void *p2)
{
  KHE_SRS_PARTIAL_SOLN ps1 = * (KHE_SRS_PARTIAL_SOLN *) p1;
  KHE_SRS_PARTIAL_SOLN ps2 = * (KHE_SRS_PARTIAL_SOLN *) p2;
  return HaArrayFirst(ps1->signature) - HaArrayFirst(ps2->signature);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsPartialSolnDoDebug(KHE_SRS_PARTIAL_SOLN ps,                   */
/*    KHE_SINGLE_RESOURCE_SOLVER srs, int verbosity, int indent, FILE *fp)   */
/*                                                                           */
/*  Debug print of task and expression of ps.                                */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheSrsPartialSolnDoDebug(KHE_SRS_PARTIAL_SOLN ps,
  KHE_SINGLE_RESOURCE_SOLVER srs, int verbosity, int indent, FILE *fp)
{
  KHE_TIME_GROUP tg;
  if( ps->task != NULL )
    KheTaskDebug(ps->task, 1, -1, fp);
  else
    fprintf(fp, "-");
  tg = KheFrameTimeGroup(srs->days_frame, ps->day_index);
  fprintf(fp, " ending on %s (cost %.5f):\n", KheTimeGroupId(tg),
    KheCostShow(ps->cost));
  KheSrsExprDebugPartialSoln((KHE_SRS_EXPR) srs->root_expr, ps, srs,
    verbosity, indent, fp);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsPartialSolnDebugHeader(KHE_SRS_PARTIAL_SOLN ps,               */
/*    KHE_SINGLE_RESOURCE_SOLVER srs, int verbosity, FILE *fp)               */
/*                                                                           */
/*  Debug print of the header part of ps.                                    */
/*                                                                           */
/*****************************************************************************/

static void KheSrsPartialSolnDebugHeader(KHE_SRS_PARTIAL_SOLN ps,
  KHE_SINGLE_RESOURCE_SOLVER srs, int verbosity, FILE *fp)
{
  KHE_TIME_GROUP tg;
  fprintf(fp, "PS(task ");
  if( ps->step->task != NULL )
    KheTaskDebug(ps->step->task, 1, -1, fp);
  else
    fprintf(fp, "-");
  if( ps->day_index < 0 )
    fprintf(fp, ", ending on -");
  else
  {
    tg = KheFrameTimeGroup(srs->days_frame, ps->day_index);
    fprintf(fp, ", ending on %s", KheTimeGroupId(tg));
  }
  fprintf(fp, ", assts %d, best of %d, cost %.5f)", HaArray(ps->signature, 0),
    ps->best_of, KheCostShow(ps->cost));
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsExprCostSumDebugPartialSoln(KHE_SRS_EXPR_COST_SUM ecs,        */
/*    KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs,               */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug the ecs part of ps.                                                */
/*                                                                           */
/*****************************************************************************/

static void KheSrsExprCostSumDebugPartialSoln(KHE_SRS_EXPR_COST_SUM ecs,
  KHE_SRS_PARTIAL_SOLN ps, KHE_SINGLE_RESOURCE_SOLVER srs,
  int verbosity, int indent, FILE *fp)
{
  KHE_SRS_EXPR e;  int i;
  fprintf(fp, "%*s[ COST_SUM\n", indent, "");
  HaArrayForEach(ecs->children, e, i)
    if( e->first_day_index <= ps->day_index &&
	ps->day_index < e->last_day_index )
    {
      fprintf(fp, "%*s", indent + 2, "");
      KheSrsExprDebugPartialSoln(e, ps, srs, verbosity, fp);
      fprintf(fp, "\n");
    }
  fprintf(fp, "%*s]\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsPartialSolnDebug(KHE_SRS_PARTIAL_SOLN ps,                     */
/*    KHE_SINGLE_RESOURCE_SOLVER srs, int verbosity, int indent, FILE *fp)   */
/*                                                                           */
/*  Debug print of ps onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void KheSrsPartialSolnDebug(KHE_SRS_PARTIAL_SOLN ps,
  KHE_SINGLE_RESOURCE_SOLVER srs, int verbosity, int indent, FILE *fp)
{
  if( indent < 0 )
    KheSrsPartialSolnDebugHeader(ps, srs, verbosity, fp);
  else
  {
    fprintf(fp, "%*s[ Partial Soln ", indent, "");
    KheSrsPartialSolnDebugHeader(ps, srs, verbosity, fp);
    fprintf(fp, "\n");
    KheSrsExprCostSumDebugPartialSoln(srs->root_expr, ps, srs,
      verbosity, indent + 2, fp);
    if( ps->prev_ps != NULL )
    {
      fprintf(fp, "%*sprev_ps ", indent + 2, "");
      KheSrsPartialSolnDebugHeader(ps->prev_ps, srs, verbosity, fp);
      fprintf(fp, "\n");
      KheSrsExprCostSumDebugPartialSoln(srs->root_expr, ps->prev_ps,
	srs, verbosity, indent + 2, fp);
    }
    fprintf(fp, "%*s]\n", indent, "");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_SRS_DAY"                                                  */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_DAY KheSrsDayMake(KHE_TIME_GROUP tg, int day_index, HA_ARENA a)  */
/*                                                                           */
/*  Make a new day object for a day with times tg.                           */
/*                                                                           */
/*****************************************************************************/

static KHE_SRS_DAY KheSrsDayMake(KHE_TIME_GROUP tg, int day_index,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_DAY res;  KHE_TIME t;  int i;  KHE_SRS_TIME stime;
  HaMake(res, srs->arena);
  res->time_group = tg;
  HaArrayInit(res->times, srs->arena);
  for( i = 0;  i < KheTimeGroupTimeCount(tg);  i++ )
  {
    t = KheTimeGroupTime(tg, i);
    stime = KheSrsTimeMake(t, srs);
    HaArrayAddLast(res->times, stime);
  }
  res->partial_solns_count = 0;
  res->sig_len = 1;  /* assignment count is always at start of signature */
  HaArrayInit(res->dom_tests, srs->arena);
  HaArrayAddLast(res->dom_tests, KheSrsDomTestMake(KHE_SRS_DOM_EQ));
  HaArrayInit(res->active_exprs, srs->arena);
  res->free_step = KheSrsStepMakeFree(day_index, srs);
  HpTableInit(res->partial_solns_table, &KheSrsPartialSolnSignatureHashUntyped,
    &KheSrsPartialSolnSignaturesEqualUntyped, srs->arena);
  HaArrayInit(res->partial_solns_array, srs->arena);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsDayClearTasks(KHE_SRS_DAY day, KHE_SINGLE_RESOURCE_SOLVER srs)*/
/*                                                                           */
/*  Clear out all the tasks running on day.                                  */
/*                                                                           */
/*****************************************************************************/

static void KheSrsDayClearTasks(KHE_SRS_DAY day, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  int i;  KHE_SRS_TIME t;
  HaArrayForEach(day->times, t, i)
    KheSrsTimeClearTasks(t, srs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsDayClear(KHE_SRS_DAY day, KHE_SINGLE_RESOURCE_SOLVER srs)     */
/*                                                                           */
/*  Clear day, ready for a fresh solve.                                      */
/*                                                                           */
/*****************************************************************************/

static void KheSrsDayClear(KHE_SRS_DAY day, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  int i, pos;  KHE_SRS_PARTIAL_SOLN ps;  KHE_SRS_TIME t;
  HpTableForEachValue(day->partial_solns_table, ps, pos)
    KheSrsPartialSolnDelete(ps, srs);
  HpTableClear(day->partial_solns_table);
  day->partial_solns_count = 0;
  day->sig_len = 1;
  HaArrayClear(day->dom_tests);
  HaArrayAddLast(day->dom_tests, KheSrsDomTestMake(KHE_SRS_DOM_EQ));
  HaArrayClear(day->active_exprs);
  HaArrayForEach(day->times, t, i)
    KheSrsTimeClear(t, srs);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheSrsDayAddTasks(KHE_SRS_DAY day, KHE_SINGLE_RESOURCE_SOLVER srs,  */
/*    KHE_SRS_STEP *preassigned_step)                                        */
/*                                                                           */
/*  Add suitable steps to day.                                               */
/*                                                                           */
/*  If true is returned, it means that a preassigned task was found, so      */
/*  that will be the only task added.  It is returned in *preassigned_step.  */
/*                                                                           */
/*****************************************************************************/

static bool KheSrsDayAddTasks(KHE_SRS_DAY day, KHE_SINGLE_RESOURCE_SOLVER srs,
  KHE_SRS_STEP *preassigned_step)
{
  int i, j, k;  KHE_TIME t;  KHE_TASK task;  KHE_SRS_TIME stime;
  KHE_SRS_STEP step;  KHE_MEET meet;
  HaArrayForEach(day->times, stime, i)
  {
    /* for each time of day */
    t = stime->time;
    for( j = 0;  j < KheEventTimetableMonitorTimeMeetCount(srs->etm, t);  j++ )
    {
      /* for each meet running at time */
      meet = KheEventTimetableMonitorTimeMeet(srs->etm, stime->time, j);
      for( k = 0;  k < KheMeetTaskCount(meet);  k++ )
      {
	/* for the proper root task of each task of meet */
	task = KheTaskProperRoot(KheMeetTask(meet, k));
	if( KheTaskAsstResource(task) == srs->resource )
	{
	  /* task is already assigned, so it is the only legal choice */
	  KheSrsDayClearTasks(day, srs);
	  step = KheSrsStepMake(task, srs);
	  HaArrayAddLast(stime->best_steps, step);
	  *preassigned_step = step;
	  return true;
	}
	else if( KheTaskAssignResourceCheck(task, srs->resource) )
	{
	  step = KheSrsStepMake(task, srs);
	  if( KheSrsStepFirstTime(step) == t )
	  {
	    /* add task to stime, possibly replacing an existing task */
	    KheSrsTimeAddTask(stime, step, srs);
	  }
	  else
	  {
	    /* step starts at the wrong time, so ignore it */
	    KheSrsStepDelete(step, srs);
	  }
	}
      }
    }
  }
  *preassigned_step = NULL;
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsDayDebug(KHE_SRS_DAY day, KHE_SINGLE_RESOURCE_SOLVER srs,     */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of day onto fp.                                              */
/*                                                                           */
/*****************************************************************************/

static void KheSrsDayDebug(KHE_SRS_DAY day, KHE_SINGLE_RESOURCE_SOLVER srs,
  int verbosity, int indent, FILE *fp)
{
  /* int i;  KHE_SRS_TIME stime; */
  if( indent >= 0 )
  {
    fprintf(fp, "%*s[ Day ", indent, "");
    KheTimeGroupDebug(day->time_group, 1, -1, stderr);
    fprintf(fp, " (times %d, sig_len %d, active_exprs %d)\n",
      HaArrayCount(day->times), day->sig_len, HaArrayCount(day->active_exprs));
    /* ***
    HaArrayForEach(day->times, stime, i)
      KheSrsTimeDebug(stime, srs, verbosity, indent + 2, fp);
    *** */
    fprintf(fp, "%*s]\n", indent, "");
  }
  else
    KheTimeGroupDebug(day->time_group, 1, -1, stderr);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_SRS_FULL_SOLN"                                            */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_FULL_ITEM KheSrsFullItemMake(KHE_SRS_STEP step,                  */
/*    bool previously_assigned)                                              */
/*                                                                           */
/*  Make a new full item object with these attributes.                       */
/*                                                                           */
/*****************************************************************************/

static KHE_SRS_FULL_ITEM KheSrsFullItemMake(KHE_SRS_STEP step,
  bool previously_assigned)
{
  KHE_SRS_FULL_ITEM res;
  res.step = step;
  res.previously_assigned = previously_assigned;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_FULL_SOLN KheSrsFullSolnMake(KHE_SINGLE_RESOURCE_SOLVER srs)     */
/*                                                                           */
/*  Make a new, empty full soln object.                                      */
/*                                                                           */
/*****************************************************************************/

static KHE_SRS_FULL_SOLN KheSrsFullSolnMake(KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_FULL_SOLN res;
  HaMake(res, srs->arena);
  HaArrayInit(res->items, srs->arena);
  res->resource = NULL;
  res->adopted = false;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsFullSolnLoad(KHE_SRS_FULL_SOLN fs, KHE_SRS_PARTIAL_SOLN ps)   */
/*                                                                           */
/*  Load ps into fs.                                                         */
/*                                                                           */
/*****************************************************************************/

static void KheSrsFullSolnLoad(KHE_SRS_FULL_SOLN fs, KHE_SRS_PARTIAL_SOLN ps,
  KHE_RESOURCE r)
{
  KHE_SRS_FULL_ITEM item;
  HaArrayClear(fs->items);
  fs->resource = r;
  fs->asst_count = HaArrayFirst(ps->signature);
  while( ps != NULL && ps->step != NULL )
  {
    item = KheSrsFullItemMake(ps->step,
      ps->step->task != NULL && KheTaskAsstResource(ps->step->task) != NULL);
    HaArrayAddLast(fs->items, item);
    ps = ps->prev_ps;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsFullSolnAdopt(KHE_SRS_FULL_SOLN fs)                           */
/*                                                                           */
/*  Adopt fs.                                                                */
/*                                                                           */
/*****************************************************************************/

static void KheSrsFullSolnAdopt(KHE_SRS_FULL_SOLN fs)
{
  KHE_SRS_FULL_ITEM item;  int i;
  HnAssert(!fs->adopted, "KheSrsFullSolnAdopt: already adopted");
  HaArrayForEach(fs->items, item, i)
    if( item.step != NULL && item.step->task != NULL &&
	!item.previously_assigned )
    {
      if( !KheTaskAssignResource(item.step->task, fs->resource) )
	HnAbort("KheSrsFullSolnAdopt internal error");
      if( DEBUG6 )
      {
	fprintf(stderr, "  KheSrsFullSolnAdopt assigning %s to ",
	  KheResourceId(fs->resource));
	KheTaskDebug(item.step->task, 1, 0, stderr);
      }
    }
  fs->adopted = true;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsFullSolnUnAdopt(KHE_SRS_FULL_SOLN fs)                         */
/*                                                                           */
/*  Undo the matching call to KheSrsFullSolnAdopt.                           */
/*                                                                           */
/*****************************************************************************/

static void KheSrsFullSolnUnAdopt(KHE_SRS_FULL_SOLN fs)
{
  KHE_SRS_FULL_ITEM item;  int i;
  HnAssert(fs->adopted, "KheSrsFullSolnUnAdopt: not adopted");
  HaArrayForEach(fs->items, item, i)
    if( item.step != NULL && item.step->task != NULL &&
	!item.previously_assigned )
    {
      if( !KheTaskUnAssignResource(item.step->task) )
	HnAbort("KheSrsFullSolnAdopt internal error");
    }
  fs->adopted = false;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_SRS_PARTIAL_SOLN KheFullSolnReRun(KHE_SRS_FULL_SOLN fs,              */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Rerun fs, returning the resulting partial solution.                      */
/*                                                                           */
/*****************************************************************************/

static KHE_SRS_PARTIAL_SOLN KheFullSolnReRun(KHE_SRS_FULL_SOLN fs,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_FULL_ITEM item;  int i, di, first_day_index, last_day_index;
  KHE_SRS_PARTIAL_SOLN ps;  KHE_TIME_GROUP tg;
  HnAssert(!fs->adopted, "KheFullSolnReRun: already adopted");
  srs->rerun = true;
  ps = KheSrsPartialSolnMakeInit(srs);
  HaArrayForEachReverse(fs->items, item, i)
  {
    first_day_index = KheDailyScheduleFirstDayIndex(item.step->daily_schedule);
    last_day_index = KheDailyScheduleLastDayIndex(item.step->daily_schedule);
    for( di = first_day_index;  di <= last_day_index;  di++ )
    {
      tg = KheFrameTimeGroup(srs->days_frame, di);
      fprintf(stderr, "  starting day %s\n", KheTimeGroupId(tg));
      ps = KheSrsPartialSolnMake(ps, item.step, di, srs);
    }
  }
  srs->rerun = false;
  return ps;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsFullSolnDebug(KHE_SRS_FULL_SOLN fs, int verbosity,            */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of fs onto fp.                                               */
/*                                                                           */
/*****************************************************************************/

static void KheSrsFullSolnDebug(KHE_SRS_FULL_SOLN fs, int verbosity,
  int indent, FILE *fp)
{
  KHE_SRS_FULL_ITEM item;  int i;
  fprintf(fp, "%*s[ Full Soln (%d assts) for %s%s:\n", indent, "",
    fs->asst_count, KheResourceId(fs->resource),
    fs->adopted ? " (currently adopted)" : "");
  HaArrayForEachReverse(fs->items, item, i)
    KheSrsStepDebug(item.step, verbosity, indent + 2, fp);
  fprintf(fp, "%*s]\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "monitor handling"                                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheSingleResourceSolverAddAvoidClashesMonitor(                      */
/*    KHE_SINGLE_RESOURCE_SOLVER srs, KHE_AVOID_CLASHES_MONITOR m)           */
/*                                                                           */
/*  Add avoid clashes monitor m to srs.                                      */
/*                                                                           */
/*****************************************************************************/

static void KheSingleResourceSolverAddAvoidClashesMonitor(
  KHE_SINGLE_RESOURCE_SOLVER srs, KHE_AVOID_CLASHES_MONITOR m)
{
  /* ignoring this; there will be no clashes */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSingleResourceSolverAddAvoidUnavailableTimesMonitor(             */
/*    KHE_SINGLE_RESOURCE_SOLVER srs, KHE_AVOID_UNAVAILABLE_TIMES_MONITOR m) */
/*                                                                           */
/*  Add avoid unavailable times monitor m to srs.                            */
/*                                                                           */
/*****************************************************************************/

static void KheSingleResourceSolverAddAvoidUnavailableTimesMonitor(
  KHE_SINGLE_RESOURCE_SOLVER srs, KHE_AVOID_UNAVAILABLE_TIMES_MONITOR m)
{
  KHE_AVOID_UNAVAILABLE_TIMES_CONSTRAINT c;  KHE_TIME_GROUP tg;
  int i;  KHE_TIME t;  KHE_SRS_EXPR_COST ec;  KHE_COST_FUNCTION cost_fn;
  KHE_SRS_EXPR_BUSY eb;  KHE_SRS_EXPR_INT_SUM eis;  KHE_COST combined_weight;
  KHE_SRS_DOM_TEST unused_dom_test, max_dom_test;
  c = KheAvoidUnavailableTimesMonitorConstraint(m);
  tg = KheAvoidUnavailableTimesConstraintUnavailableTimes(c);
  unused_dom_test = KheSrsDomTestMake(KHE_SRS_DOM_UNUSED);
  combined_weight = KheConstraintCombinedWeight((KHE_CONSTRAINT) c);
  if( combined_weight > 0 && KheTimeGroupTimeCount(tg) > 0 )
  {
    cost_fn = KheConstraintCostFunction((KHE_CONSTRAINT) c);
    if( cost_fn == KHE_LINEAR_COST_FUNCTION || KheTimeGroupTimeCount(tg) == 1 )
    {
      /* a separate expression tree for each time */
      for( i = 0;  i < KheTimeGroupTimeCount(tg);  i++ )
      {
	t = KheTimeGroupTime(tg, i);
	ec = KheSrsExprCostMake((KHE_MONITOR) m, cost_fn, combined_weight, srs);
	KheSrsExprAddChild((KHE_SRS_EXPR) srs->root_expr, (KHE_SRS_EXPR) ec);
	eb = KheSrsExprBusyMake(t, unused_dom_test, srs);
	KheSrsExprAddChild((KHE_SRS_EXPR) ec, (KHE_SRS_EXPR) eb);
      }
    }
    else
    {
      /* a single expression tree (max limit of 0, in effect) */
      ec = KheSrsExprCostMake((KHE_MONITOR) m, cost_fn, combined_weight, srs);
      KheSrsExprAddChild((KHE_SRS_EXPR) srs->root_expr, (KHE_SRS_EXPR) ec);
      max_dom_test = KheSrsDomTestMake(KHE_SRS_DOM_LE);
      eis = KheSrsExprIntSumMake(max_dom_test, srs);
      KheSrsExprAddChild((KHE_SRS_EXPR) ec, (KHE_SRS_EXPR) eis);
      for( i = 0;  i < KheTimeGroupTimeCount(tg);  i++ )
      {
	t = KheTimeGroupTime(tg, i);
	eb = KheSrsExprBusyMake(t, unused_dom_test, srs);
	KheSrsExprAddChild((KHE_SRS_EXPR) eis, (KHE_SRS_EXPR) eb);
      }
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSingleResourceSolverAddLimitIdleTimesMonitor(                    */
/*    KHE_SINGLE_RESOURCE_SOLVER srs, KHE_LIMIT_IDLE_TIMES_MONITOR m)        */
/*                                                                           */
/*  Add limit idle times monitor m to srs.                                   */
/*                                                                           */
/*****************************************************************************/

static void KheSingleResourceSolverAddLimitIdleTimesMonitor(
  KHE_SINGLE_RESOURCE_SOLVER srs, KHE_LIMIT_IDLE_TIMES_MONITOR m)
{
  /* not included in the solve, and that fact is documented */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheMakeOrNodeAndChildren(KHE_SRS_EXPR prnt, KHE_TIME_GROUP tg,      */
/*    KHE_SRS_DOM_TEST dom_test, KHE_SINGLE_RESOURCE_SOLVER srs)             */
/*                                                                           */
/*  Make an OR node with one BUSY child for each time of tg, and add the     */
/*  OR node as a child of prnt.  If tg has exactly one time, optimize by     */
/*  adding just the BUSY node for that time as a child of prnt.              */
/*                                                                           */
/*  Either way, add dom_test to all of these vertices.                       */
/*                                                                           */
/*****************************************************************************/

static void KheMakeOrNodeAndChildren(KHE_SRS_EXPR prnt, KHE_TIME_GROUP tg,
  KHE_SRS_DOM_TEST dom_test, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_EXPR_OR eo;  KHE_SRS_EXPR_BUSY eb;  int i;  KHE_TIME t;
  KHE_SRS_DOM_TEST unused_dom_test;
  if( KheTimeGroupTimeCount(tg) == 1 )
  {
    /* just a single BUSY node */
    t = KheTimeGroupTime(tg, 0);
    eb = KheSrsExprBusyMake(t, dom_test, srs);
    KheSrsExprAddChild(prnt, (KHE_SRS_EXPR) eb);
  }
  else
  {
    /* an OR node with BUSY children */
    eo = KheSrsExprOrMake(dom_test, srs);
    KheSrsExprAddChild(prnt, (KHE_SRS_EXPR) eo);
    unused_dom_test = KheSrsDomTestMake(KHE_SRS_DOM_UNUSED);
    for( i = 0;  i < KheTimeGroupTimeCount(tg);  i++ )
    {
      t = KheTimeGroupTime(tg, i);
      eb = KheSrsExprBusyMake(t, unused_dom_test, srs);
      KheSrsExprAddChild((KHE_SRS_EXPR) eo, (KHE_SRS_EXPR) eb);
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheMakeAndNodeAndChildren(KHE_SRS_EXPR prnt, KHE_TIME_GROUP tg,     */
/*    KHE_SRS_DOM_TEST dom_test, KHE_SINGLE_RESOURCE_SOLVER srs)             */
/*                                                                           */
/*  Make an AND node with one FREE child for each time of tg, and add the    */
/*  AND node as a child of prnt.  If tg has exactly one time, optimize by    */
/*  adding just the FREE node for that time as a child of prnt.              */
/*                                                                           */
/*  Either way, add dom_test to all of these vertices.                       */
/*                                                                           */
/*****************************************************************************/

static void KheMakeAndNodeAndChildren(KHE_SRS_EXPR prnt, KHE_TIME_GROUP tg,
  KHE_SRS_DOM_TEST dom_test, KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_EXPR_AND ea;  KHE_SRS_EXPR_FREE ef;  int i;  KHE_TIME t;
  KHE_SRS_DOM_TEST unused_dom_test;
  if( KheTimeGroupTimeCount(tg) == 1 )
  {
    /* just a single FREE node */
    t = KheTimeGroupTime(tg, 0);
    ef = KheSrsExprFreeMake(t, dom_test, srs);
    KheSrsExprAddChild(prnt, (KHE_SRS_EXPR) ef);
  }
  else
  {
    /* an AND node with FREE children */
    ea = KheSrsExprAndMake(dom_test, srs);
    KheSrsExprAddChild(prnt, (KHE_SRS_EXPR) ea);
    unused_dom_test = KheSrsDomTestMake(KHE_SRS_DOM_UNUSED);
    for( i = 0;  i < KheTimeGroupTimeCount(tg);  i++ )
    {
      t = KheTimeGroupTime(tg, i);
      ef = KheSrsExprFreeMake(t, unused_dom_test, srs);
      KheSrsExprAddChild((KHE_SRS_EXPR) ea, (KHE_SRS_EXPR) ef);
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheLimitsToTests(int min_limit, int max_limit, int max_max_limit,   */
/*    bool allow_zero, KHE_SRS_DOM_TEST *alpha_dom_test,                     */
/*    KHE_SRS_DOM_TEST *beta_dom_test)                                       */
/*                                                                           */
/*  Convert limits (min_limit, max_limit, allow_zero) into dominance tests.  */
/*  if min_limit <= 0 there is no min limit; if max_limit >= max_max_limit   */
/*  there is no maximum limit.                                               */
/*                                                                           */
/*  The values returned in *alpha_dom_test and *beta_dom_test are referred   */
/*  to as alpha and beta in the User's Guide.                                */
/*                                                                           */
/*  Note.  If there is a trivial allow_zero flag (if min_limit <= 1), an     */
/*  adjustment will have already been made by now to remove it.  So this     */
/*  call may assume that allow_zero implies min_limit > 1.                   */
/*                                                                           */
/*****************************************************************************/

static void KheLimitsToTests(int min_limit, int max_limit, int max_max_limit,
  bool allow_zero, KHE_SRS_DOM_TEST *alpha_dom_test,
  KHE_SRS_DOM_TEST *beta_dom_test)
{
  if( allow_zero /* && min_limit > 1 */ )
  {
    /* fall back on equality when there is a non-trivial allow_zero flag */
    *alpha_dom_test = *beta_dom_test = KheSrsDomTestMake(KHE_SRS_DOM_EQ);
  }
  else if( max_limit >= max_max_limit )
  {
    if( min_limit <= 0 )
    {
      /* no max_limit, and no min_limit (should never happen) */
      HnAbort("KheLimitsToTests: no max limit, no min limit");
      *alpha_dom_test = *beta_dom_test = KheSrsDomTestMake(KHE_SRS_DOM_UNUSED);
    }
    else
    {
      /* no max_limit, and min_limit */
      *alpha_dom_test = KheSrsDomTestMakeLoose(KHE_SRS_DOM_GE_LOOSE, min_limit);
      *beta_dom_test = KheSrsDomTestMake(KHE_SRS_DOM_GE);
    }
  }
  else
  {
    if( min_limit <= 0 )
    {
      /* max_limit, and no min_limit */
      *alpha_dom_test = *beta_dom_test = KheSrsDomTestMake(KHE_SRS_DOM_LE);
    }
    else
    {
      /* max_limit, and min_limit */
      *alpha_dom_test = KheSrsDomTestMakeLoose(KHE_SRS_DOM_EQ_LOOSE, min_limit);
      *beta_dom_test = KheSrsDomTestMake(KHE_SRS_DOM_EQ);
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSingleResourceSolverAddClusterBusyTimesMonitor(                  */
/*    KHE_SINGLE_RESOURCE_SOLVER srs, KHE_CLUSTER_BUSY_TIMES_MONITOR m)      */
/*                                                                           */
/*  Add cluster busy times monitor m to srs.                                 */
/*                                                                           */
/*****************************************************************************/

static void KheSingleResourceSolverAddClusterBusyTimesMonitor(
  KHE_SINGLE_RESOURCE_SOLVER srs, KHE_CLUSTER_BUSY_TIMES_MONITOR m)
{
  KHE_CLUSTER_BUSY_TIMES_CONSTRAINT c;  KHE_TIME_GROUP tg;  KHE_POLARITY po;
  KHE_COST_FUNCTION cost_fn;  KHE_COST combined_weight;  bool allow_zero;
  int i, min_limit, max_limit, history_before, history_after, tg_count;
  KHE_SRS_EXPR_INT_SUM_COMB eisc;
  KHE_SRS_DOM_TEST alpha_dom_test, beta_dom_test;

  /* boilerplate, and return if constraint does nothing */
  c = KheClusterBusyTimesMonitorConstraint(m);
  combined_weight = KheConstraintCombinedWeight((KHE_CONSTRAINT) c);
  cost_fn = KheConstraintCostFunction((KHE_CONSTRAINT) c);
  min_limit = KheClusterBusyTimesConstraintMinimum(c);
  max_limit = KheClusterBusyTimesConstraintMaximum(c);
  allow_zero = KheClusterBusyTimesConstraintAllowZero(c);
  history_before = KheClusterBusyTimesMonitorHistory(m);
  history_after = KheClusterBusyTimesMonitorHistoryAfter(m);
  if( allow_zero && min_limit <= 1 )
    min_limit = 0, allow_zero = false;
  tg_count = KheClusterBusyTimesConstraintTimeGroupCount(c);
  if( (min_limit <= 0 && max_limit >= tg_count) || combined_weight == 0 )
    return;

  /* work out the various dom_tests */
  KheLimitsToTests(min_limit, max_limit, tg_count, allow_zero,
    &alpha_dom_test, &beta_dom_test);

  /* one int_cost_comb node at top of tree */
  eisc = KheSrsExprIntSumCombMake((KHE_MONITOR) m, cost_fn, combined_weight,
    min_limit, max_limit, allow_zero, history_before, history_after,
    alpha_dom_test, srs);
  KheSrsExprAddChild((KHE_SRS_EXPR) srs->root_expr, (KHE_SRS_EXPR) eisc);

  /* one OR or AND node for each time group */
  for( i = 0;  i < tg_count;  i++ )
  {
    tg = KheClusterBusyTimesMonitorTimeGroup(m, i, &po);
    if( po == KHE_POSITIVE )
    {
      /* make an OR node with one BUSY node for each time of tg */
      KheMakeOrNodeAndChildren((KHE_SRS_EXPR) eisc, tg, beta_dom_test, srs);
    }
    else
    {
      /* make an AND node with one FREE node for each time of tg */
      KheMakeAndNodeAndChildren((KHE_SRS_EXPR) eisc, tg, beta_dom_test, srs);
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSingleResourceSolverAddLimitBusyTimesMonitor(                    */
/*    KHE_SINGLE_RESOURCE_SOLVER srs, KHE_LIMIT_BUSY_TIMES_MONITOR m)        */
/*                                                                           */
/*  Add limit busy times monitor m to srs.                                   */
/*                                                                           */
/*****************************************************************************/

static void KheSingleResourceSolverAddLimitBusyTimesMonitor(
  KHE_SINGLE_RESOURCE_SOLVER srs, KHE_LIMIT_BUSY_TIMES_MONITOR m)
{
  KHE_LIMIT_BUSY_TIMES_CONSTRAINT c;
  KHE_COST_FUNCTION cost_fn;  KHE_COST combined_weight;
  int i, j, junk, min_limit, max_limit;  bool allow_zero;  KHE_TIME_GROUP tg;
  KHE_SRS_EXPR_INT_SUM_COMB eisc;  KHE_TIME t;
  KHE_SRS_EXPR_INT_SUM eis, eis2;  KHE_SRS_EXPR_COST ec;
  KHE_SRS_EXPR_INT_DEV eid;
  KHE_SRS_EXPR_BUSY eb;
  KHE_SRS_DOM_TEST unused_dom_test, le_dom_test, alpha_dom_test, beta_dom_test;

  c = KheLimitBusyTimesMonitorConstraint(m);
  cost_fn = KheConstraintCostFunction((KHE_CONSTRAINT) c);
  combined_weight = KheConstraintCombinedWeight((KHE_CONSTRAINT) c);
  min_limit = KheLimitBusyTimesConstraintMinimum(c);
  max_limit = KheLimitBusyTimesConstraintMaximum(c);
  allow_zero = KheLimitBusyTimesConstraintAllowZero(c);
  if( allow_zero && min_limit <= 1 )
    min_limit = 0, allow_zero = false;
  unused_dom_test = KheSrsDomTestMake(KHE_SRS_DOM_UNUSED);
  le_dom_test = KheSrsDomTestMake(KHE_SRS_DOM_LE);
  if( combined_weight == 0 )
    return;

  if( cost_fn == KHE_LINEAR_COST_FUNCTION ||
      KheLimitBusyTimesMonitorTimeGroupCount(m) == 1 )
  {
    /* one INT_SUM_COMB for each time group */
    for( i = 0;  i < KheLimitBusyTimesMonitorTimeGroupCount(m);  i++ )
    {
      tg = KheLimitBusyTimesMonitorTimeGroup(m, i, &junk);
      if( min_limit > 0 || max_limit < KheTimeGroupTimeCount(tg) )
      {
	KheLimitsToTests(min_limit, max_limit, KheTimeGroupTimeCount(tg),
	  allow_zero, &alpha_dom_test, &beta_dom_test);
	eisc = KheSrsExprIntSumCombMake((KHE_MONITOR) m, cost_fn,
	  combined_weight, min_limit, max_limit, allow_zero, 0, 0,
	  alpha_dom_test, srs);
	KheSrsExprAddChild((KHE_SRS_EXPR) srs->root_expr, (KHE_SRS_EXPR) eisc);
	for( j = 0;  j < KheTimeGroupTimeCount(tg);  j++ )
	{
	  t = KheTimeGroupTime(tg, j);
	  eb = KheSrsExprBusyMake(t, unused_dom_test, srs);
	  KheSrsExprAddChild((KHE_SRS_EXPR) eisc, (KHE_SRS_EXPR) eb);
	}
      }
    }
  }
  else
  {
    /* one COST and INT_SUM at root of tree */
    ec = KheSrsExprCostMake((KHE_MONITOR) m, cost_fn, combined_weight, srs);
    KheSrsExprAddChild((KHE_SRS_EXPR) srs->root_expr, (KHE_SRS_EXPR) ec);
    eis = KheSrsExprIntSumMake(le_dom_test, srs);
    KheSrsExprAddChild((KHE_SRS_EXPR) ec, (KHE_SRS_EXPR) eis);

    /* one INT_DEV and INT_SUM for each time group */
    for( i = 0;  i < KheLimitBusyTimesMonitorTimeGroupCount(m);  i++ )
    {
      tg = KheLimitBusyTimesMonitorTimeGroup(m, i, &junk);
      eid = KheSrsExprIntDevMake(min_limit, max_limit, allow_zero, srs);
      KheSrsExprAddChild((KHE_SRS_EXPR) eis, (KHE_SRS_EXPR) eid);
      KheLimitsToTests(min_limit, max_limit, KheTimeGroupTimeCount(tg),
	allow_zero, &alpha_dom_test, &beta_dom_test);
      eis2 = KheSrsExprIntSumMake(alpha_dom_test, srs);
      KheSrsExprAddChild((KHE_SRS_EXPR) eid, (KHE_SRS_EXPR) eis2);
      for( j = 0;  j < KheTimeGroupTimeCount(tg);  j++ )
      {
	t = KheTimeGroupTime(tg, j);
	eb = KheSrsExprBusyMake(t, unused_dom_test, srs);
	KheSrsExprAddChild((KHE_SRS_EXPR) eis2, (KHE_SRS_EXPR) eb);
      }
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSingleResourceSolverAddLimitWorkloadMonitor(                     */
/*    KHE_SINGLE_RESOURCE_SOLVER srs, KHE_LIMIT_WORKLOAD_MONITOR m)          */
/*                                                                           */
/*  Add limit workload monitor m to srs.                                     */
/*                                                                           */
/*****************************************************************************/

static void KheSingleResourceSolverAddLimitWorkloadMonitor(
  KHE_SINGLE_RESOURCE_SOLVER srs, KHE_LIMIT_WORKLOAD_MONITOR m)
{
  KHE_LIMIT_WORKLOAD_CONSTRAINT c;
  KHE_COST_FUNCTION cost_fn;  KHE_COST combined_weight;
  int i, j, min_limit, max_limit;  bool allow_zero;  float junk;
  KHE_TIME_GROUP tg;  KHE_TIME t;  KHE_SRS_EXPR_WORK ew;
  KHE_SRS_EXPR_INT_SUM eis;  KHE_SRS_EXPR_COST ec;
  KHE_SRS_EXPR_FLOAT_SUM efs;  KHE_SRS_EXPR_FLOAT_DEV efd;
  KHE_SRS_DOM_TEST unused_dom_test, le_dom_test, alpha_dom_test, beta_dom_test;

  /* boilerplate */
  c = KheLimitWorkloadMonitorConstraint(m);
  cost_fn = KheConstraintCostFunction((KHE_CONSTRAINT) c);
  combined_weight = KheConstraintCombinedWeight((KHE_CONSTRAINT) c);
  min_limit = KheLimitWorkloadConstraintMinimum(c);
  max_limit = KheLimitWorkloadConstraintMaximum(c);
  allow_zero = KheLimitWorkloadConstraintAllowZero(c);
  if( allow_zero && min_limit <= 1 )
    min_limit = 0, allow_zero = false;

  le_dom_test = KheSrsDomTestMake(KHE_SRS_DOM_EQ);
  unused_dom_test = KheSrsDomTestMake(KHE_SRS_DOM_UNUSED);

  /* one COST and INT_SUM at the root */
  ec = KheSrsExprCostMake((KHE_MONITOR) m, cost_fn, combined_weight, srs);
  KheSrsExprAddChild((KHE_SRS_EXPR) srs->root_expr, (KHE_SRS_EXPR) ec);
  eis = KheSrsExprIntSumMake(le_dom_test, srs);
  KheSrsExprAddChild((KHE_SRS_EXPR) ec, (KHE_SRS_EXPR) eis);

  /* one FLOAT_DEV and FLOAT_SUM for each time group */
  KheLimitsToTests(min_limit, max_limit, INT_MAX, allow_zero,
    &alpha_dom_test, &beta_dom_test);
  for( i = 0;  i < KheLimitWorkloadMonitorTimeGroupCount(m);  i++ )
  {
    tg = KheLimitWorkloadMonitorTimeGroup(m, i, &junk);
    efd = KheSrsExprFloatDevMake(min_limit, max_limit, allow_zero, srs);
    KheSrsExprAddChild((KHE_SRS_EXPR) eis, (KHE_SRS_EXPR) efd);
    efs = KheSrsExprFloatSumMake(alpha_dom_test, srs);
    KheSrsExprAddChild((KHE_SRS_EXPR) efd, (KHE_SRS_EXPR) efs);
    for( j = 0;  j < KheTimeGroupTimeCount(tg);  j++ )
    {
      t = KheTimeGroupTime(tg, j);
      ew = KheSrsExprWorkMake(t, unused_dom_test, srs);
      KheSrsExprAddChild((KHE_SRS_EXPR) efs, (KHE_SRS_EXPR) ew);
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSingleResourceSolverAddLimitActiveIntervalsMonitor(              */
/*    KHE_SINGLE_RESOURCE_SOLVER srs, KHE_LIMIT_ACTIVE_INTERVALS_MONITOR m)  */
/*                                                                           */
/*  Add limit active intervals monitor m to srs.                             */
/*                                                                           */
/*****************************************************************************/

static void KheSingleResourceSolverAddLimitActiveIntervalsMonitor(
  KHE_SINGLE_RESOURCE_SOLVER srs, KHE_LIMIT_ACTIVE_INTERVALS_MONITOR m)
{
  KHE_LIMIT_ACTIVE_INTERVALS_CONSTRAINT c;  KHE_TIME_GROUP tg;  KHE_POLARITY po;
  KHE_COST_FUNCTION cost_fn;  KHE_COST combined_weight;
  int i, min_limit, max_limit, history_before, history_after, tg_count;
  KHE_SRS_EXPR_INT_SEQ_COMB eisc;
  KHE_SRS_DOM_TEST alpha_dom_test, beta_dom_test;

  /* boilerplate, and return if does nothing */
  c = KheLimitActiveIntervalsMonitorConstraint(m);
  combined_weight = KheConstraintCombinedWeight((KHE_CONSTRAINT) c);
  cost_fn = KheConstraintCostFunction((KHE_CONSTRAINT) c);
  min_limit = KheLimitActiveIntervalsConstraintMinimum(c);
  max_limit = KheLimitActiveIntervalsConstraintMaximum(c);
  history_before = KheLimitActiveIntervalsMonitorHistory(m);
  history_after = KheLimitActiveIntervalsMonitorHistoryAfter(m);
  tg_count = KheLimitActiveIntervalsConstraintTimeGroupCount(c);
  if( combined_weight == 0 )
    return;

  /* work out the various dom_tests */
  KheLimitsToTests(min_limit, max_limit, tg_count, false,
    &alpha_dom_test, &beta_dom_test);

  /* one int_seq_comb node at top of tree */
  eisc = KheSrsExprIntSeqCombMake((KHE_MONITOR) m, cost_fn, combined_weight,
    min_limit, max_limit, history_before, history_after, alpha_dom_test, srs);
  KheSrsExprAddChild((KHE_SRS_EXPR) srs->root_expr, (KHE_SRS_EXPR) eisc);

  /* one OR or AND node for each time group */
  for( i = 0;  i < KheLimitActiveIntervalsMonitorTimeGroupCount(m);  i++ )
  {
    tg = KheLimitActiveIntervalsMonitorTimeGroup(m, i, &po);
    if( po == KHE_POSITIVE )
    {
      /* make an OR node with one BUSY node for each time of tg */
      KheMakeOrNodeAndChildren((KHE_SRS_EXPR) eisc, tg, beta_dom_test, srs);
    }
    else
    {
      /* make an AND node with one FREE node for each time of tg */
      KheMakeAndNodeAndChildren((KHE_SRS_EXPR) eisc, tg, beta_dom_test, srs);
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSingleResourceSolverAddMonitor(KHE_SINGLE_RESOURCE_SOLVER srs,   */
/*    KHE_MONITOR m)                                                         */
/*                                                                           */
/*  Add m to srs.                                                            */
/*                                                                           */
/*****************************************************************************/

static void KheSingleResourceSolverAddMonitor(KHE_SINGLE_RESOURCE_SOLVER srs,
  KHE_MONITOR m)
{
  switch( KheMonitorTag(m) )
  {
    case KHE_AVOID_CLASHES_MONITOR_TAG:

      KheSingleResourceSolverAddAvoidClashesMonitor(srs,
        (KHE_AVOID_CLASHES_MONITOR) m);
      break;

    case KHE_AVOID_UNAVAILABLE_TIMES_MONITOR_TAG:

      KheSingleResourceSolverAddAvoidUnavailableTimesMonitor(srs,
        (KHE_AVOID_UNAVAILABLE_TIMES_MONITOR) m);
      break;

    case KHE_LIMIT_IDLE_TIMES_MONITOR_TAG:

      KheSingleResourceSolverAddLimitIdleTimesMonitor(srs,
        (KHE_LIMIT_IDLE_TIMES_MONITOR) m);
      break;

    case KHE_CLUSTER_BUSY_TIMES_MONITOR_TAG:

      KheSingleResourceSolverAddClusterBusyTimesMonitor(srs,
        (KHE_CLUSTER_BUSY_TIMES_MONITOR) m);
      break;

    case KHE_LIMIT_BUSY_TIMES_MONITOR_TAG:

      KheSingleResourceSolverAddLimitBusyTimesMonitor(srs,
        (KHE_LIMIT_BUSY_TIMES_MONITOR) m);
      break;

    case KHE_LIMIT_WORKLOAD_MONITOR_TAG:

      KheSingleResourceSolverAddLimitWorkloadMonitor(srs,
        (KHE_LIMIT_WORKLOAD_MONITOR) m);
      break;

    case KHE_LIMIT_ACTIVE_INTERVALS_MONITOR_TAG:

      KheSingleResourceSolverAddLimitActiveIntervalsMonitor(srs,
        (KHE_LIMIT_ACTIVE_INTERVALS_MONITOR) m);
      break;

    default:

      HnAbort("KheSingleResourceSolverAddMonitor internal error (tag %d)",
        KheMonitorTag(m));
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "public functions"                                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_SINGLE_RESOURCE_SOLVER KheSingleResourceSolverMake(KHE_SOLN soln,    */
/*    KHE_OPTIONS options)                                                   */
/*                                                                           */
/*  Make and return a new single resource solver with these attributes.      */
/*                                                                           */
/*****************************************************************************/

KHE_SINGLE_RESOURCE_SOLVER KheSingleResourceSolverMake(KHE_SOLN soln,
  KHE_OPTIONS options)
{
  KHE_SINGLE_RESOURCE_SOLVER res;  HA_ARENA a;  KHE_TIME_GROUP tg;
  int i;  KHE_FRAME days_frame;  KHE_EVENT_TIMETABLE_MONITOR etm;

  /* get the common frame and event timetable monitor; return NULL if can't */
  days_frame = KheOptionsFrame(options, "gs_common_frame", soln);
  etm = (KHE_EVENT_TIMETABLE_MONITOR)
    KheOptionsGetObject(options, "gs_event_timetable_monitor", NULL);
  if( days_frame == NULL || etm == NULL )
    return NULL;

  /* make the basic object and initialize the boilerplate fields */
  a = KheSolnArenaBegin(soln, false);
  HaMake(res, a);
  res->arena = a;
  res->soln = soln;
  res->options = options;
  res->days_frame = days_frame;
  res->etm = etm;
  res->task_finder = KheTaskFinderMake(soln, options, a);

  /* initialize the fields that vary with the resource and solve */
  res->diversifier = 0;
  res->resource = NULL;
  res->root_expr = KheSrsExprCostSumMake(res);
  HaArrayInit(res->days, a);
  for( i = 0;  i < KheFrameTimeGroupCount(days_frame);  i++ )
  {
    tg = KheFrameTimeGroup(days_frame, i);
    HaArrayAddLast(res->days, KheSrsDayMake(tg, i, res));
  }
  HaArrayInit(res->final_solns, a);
  res->full_soln = KheSrsFullSolnMake(res);
  res->rerun = false;
  /* ***
  res->adopted_index = -1;
  HaArrayInit(res->adopted_tasks, a);
  *** */

  /* initialize the free lists */
  HaArrayInit(res->free_expr_busy, a);
  HaArrayInit(res->free_expr_free, a);
  HaArrayInit(res->free_expr_work, a);
  HaArrayInit(res->free_expr_or, a);
  HaArrayInit(res->free_expr_and, a);
  HaArrayInit(res->free_expr_int_sum, a);
  /* HaArrayInit(res->free_expr_int_seq, a); */
  HaArrayInit(res->free_expr_float_sum, a);
  HaArrayInit(res->free_expr_int_dev, a);
  HaArrayInit(res->free_expr_float_dev, a);
  HaArrayInit(res->free_expr_cost, a);
  HaArrayInit(res->free_expr_cost_sum, a);
  HaArrayInit(res->free_expr_int_sum_comb, a);
  HaArrayInit(res->free_expr_int_seq_comb, a);
  HaArrayInit(res->free_steps, a);
  HaArrayInit(res->free_partial_solns, a);

#if TESTING
  res->timer = KheTimerMake("single", KHE_NO_TIME, a);
  HaArrayInit(res->running_time_per_day, a);
#endif

  /* all done */
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSingleResourceSolverDelete(KHE_SINGLE_RESOURCE_SOLVER srs)       */
/*                                                                           */
/*  Delete srs, reclaiming its memory.                                       */
/*                                                                           */
/*****************************************************************************/

void KheSingleResourceSolverDelete(KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KheSolnArenaEnd(srs->soln, srs->arena);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsDayAddPartialSoln(KHE_SRS_DAY day, KHE_SRS_PARTIAL_SOLN ps,   */
/*    KHE_SINGLE_RESOURCE_SOLVER srs)                                        */
/*                                                                           */
/*  Add ps to day, or do nothing if ps has a hard constraint violation.      */
/*                                                                           */
/*  If day does not already contain a partial solution with the same         */
/*  signature as ps, just add ps to day and return.                          */
/*                                                                           */
/*  If day already contains a partial solution with the same signature       */
/*  as ps, called other_ps below, keep whichever one of ps and other_ps      */
/*  has the smallest cost, according to KheSrsPartialSolnCostLessThan, by    */
/*  overwriting other_ps's fields with ps's fields, if required.             */
/*                                                                           */
/*  If ps and other_ps have equal cost, keep one chosen at random, with      */
/*  weights which ensure that if there are K partial solutions with the      */
/*  same signature and the same cost, then each has probability 1/K of       */
/*  being the one kept.                                                      */
/*                                                                           */
/*****************************************************************************/

static void KheSrsDayAddPartialSoln(KHE_SRS_DAY day, KHE_SRS_PARTIAL_SOLN ps,
  KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KHE_SRS_PARTIAL_SOLN other_ps;  int asst_count, i;
  asst_count = HaArrayFirst(ps->signature);
  if( asst_count > srs->max_assts || ps->cost >= KheCost(1, 0) )
  {
    /* do nothing except free ps; its cost is too great, or too many assts */
    KheSrsPartialSolnDelete(ps, srs);
  }
  else if( srs->full_dominance )
  {
    /* if ps is dominated by anything else, delete ps and return */
    HaArrayForEach(day->partial_solns_array, other_ps, i)
      if( KheSrsPartialSolnDominates(other_ps, ps, day) )
      {
	KheSrsPartialSolnDelete(ps, srs);
	return;
      }

    /* remove other partial solutions that ps dominates */
    HaArrayForEach(day->partial_solns_array, other_ps, i)
      if( KheSrsPartialSolnDominates(ps, other_ps, day) )
      {
	KheSrsPartialSolnDelete(other_ps, srs);
	HaArrayDeleteAndPlug(day->partial_solns_array, i);
	day->partial_solns_count--;
	i--;
      }

    /* add ps to the array */
    HaArrayAddLast(day->partial_solns_array, ps);
    day->partial_solns_count++;
  }
  else
  {
    /* do equivalence testing */
    if( HpTableAddUnique(day->partial_solns_table, (void *) ps, ps, other_ps) )
      day->partial_solns_count++;
    else
    {
      /* other_ps will be the best of one more partial solution than before */
      other_ps->best_of++;

      /* ensure other_ps is the better of other_ps and ps, by overwriting */
      if( KheSrsPartialSolnCostLessThan(ps, other_ps) ||
	  (!KheSrsPartialSolnCostLessThan(other_ps, ps) &&
	   srs->diversifier % other_ps->best_of == 0) )
      {
	/* ps is better than other_ps, so overwrite other_ps with ps */
	KheSrsPartialSolnOverWrite(other_ps, ps);
      }

      /* free ps */
      KheSrsPartialSolnDelete(ps, srs);
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void DoExtend(KHE_SINGLE_RESOURCE_SOLVER srs,                            */
/*    KHE_SRS_PARTIAL_SOLN ps, KHE_SRS_STEP step)                            */
/*                                                                           */
/*  Extend ps by adding step to it.  Add the resulting new partial solution  */
/*  to the day object representing step's last day.                          */
/*                                                                           */
/*****************************************************************************/

static void DoExtend(KHE_SINGLE_RESOURCE_SOLVER srs,
  KHE_SRS_PARTIAL_SOLN ps, KHE_SRS_STEP step)
{
  KHE_SRS_PARTIAL_SOLN curr_ps, last_ps;
  int di, first_day_index, last_day_index;  KHE_SRS_DAY last_day;
  /* KHE_TIME time;  KHE_SRS_TIME stime;  KHE_TASK task; */

  /* make one partial solution for each day of step, ending at last_ps */
  last_ps = ps;
  first_day_index = KheDailyScheduleFirstDayIndex(step->daily_schedule);
  last_day_index = KheDailyScheduleLastDayIndex(step->daily_schedule);
  for( di = first_day_index;  di <= last_day_index;  di++ )
    last_ps = KheSrsPartialSolnMake(last_ps, step, di, srs);
  /* ***
  {
    time = KheDailyScheduleTime(step->daily_schedule, di);
    task = KheDailyScheduleTask(step->daily_schedule, di);
    last_ps = KheSrsPartialSolnMake(last_ps, step, di, srs);
  }
  *** */
  HnAssert(last_ps != ps, "DoExtend internal error");

  /* delete all partial solutions between ps and last_ps exclusive */
  /* NB field curr_ps->prev is usable after calling KhePartialSolnDelete */
  for( curr_ps = last_ps->prev_ps;  curr_ps != ps;  curr_ps = curr_ps->prev_ps )
    KheSrsPartialSolnDelete(curr_ps, srs);
  last_ps->prev_ps = ps;

  /* add last_ps to last_day, and increment the diversifier */
  last_day = HaArray(srs->days, last_day_index);
  KheSrsDayAddPartialSoln(last_day, last_ps, srs);
  srs->diversifier++;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSingleResourceSolverExtend(KHE_SINGLE_RESOURCE_SOLVER srs,       */
/*    KHE_SRS_PARTIAL_SOLN ps, KHE_SRS_DAY day)                              */
/*                                                                           */
/*  Extend the dynamic programming search by searching out of ps.  Each      */
/*  new task starts on day.                                                  */
/*                                                                           */
/*****************************************************************************/

static void KheSingleResourceSolverExtend(KHE_SINGLE_RESOURCE_SOLVER srs,
  KHE_SRS_PARTIAL_SOLN ps, KHE_SRS_DAY day)
{
  int i, j;  KHE_SRS_STEP step;  KHE_SRS_TIME stime;

  /* extend for each task beginning on day */
  HaArrayForEach(day->times, stime, i)
    HaArrayForEach(stime->best_steps, step, j)
      DoExtend(srs, ps, step);

  /* extend by taking day off */
  DoExtend(srs, ps, day->free_step);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSingleResourceSolverClear(KHE_SINGLE_RESOURCE_SOLVER srs)        */
/*                                                                           */
/*  Clear srs, ready for a fresh solve.                                      */
/*                                                                           */
/*****************************************************************************/

static void KheSingleResourceSolverClear(KHE_SINGLE_RESOURCE_SOLVER srs)
{
  int di;  KHE_SRS_DAY day;
  srs->diversifier = 0;
  HaArrayForEach(srs->days, day, di)
    KheSrsDayClear(day, srs);
  HaArrayClear(srs->final_solns);
  KheSrsExprCostSumDelete(srs->root_expr, srs);
  srs->root_expr = KheSrsExprCostSumMake(srs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSingleResourceSolverSolve(KHE_SINGLE_RESOURCE_SOLVER srs,        */
/*    KHE_RESOURCE r, bool full_dominance, int min_assts, int max_assts)     */
/*                                                                           */
/*  Solve for r.                                                             */
/*                                                                           */
/*****************************************************************************/

void KheSingleResourceSolverSolve(KHE_SINGLE_RESOURCE_SOLVER srs,
  KHE_RESOURCE r, bool full_dominance, int min_assts, int max_assts)
{
  int di, i, pos, asst_count;  KHE_MONITOR m;  KHE_SRS_DAY day, prev_day;
  KHE_SRS_PARTIAL_SOLN ps;  KHE_SRS_STEP preassigned_step;

  if( DEBUG1 )
    fprintf(stderr, "[ KheSingleResourceSolverSolve(srs, %s)\n",
      KheResourceId(r));

  /* if there was already a resource, clear out old stuff */
  if( srs->resource != NULL )
    KheSingleResourceSolverClear(srs);

  /* add the resource, and the expression tree representing the monitors */
  srs->resource = r;
  srs->full_dominance = full_dominance;
  srs->min_assts = min_assts;
  srs->max_assts = max_assts;
  for( i = 0;  i < KheSolnResourceMonitorCount(srs->soln, r);  i++ )
  {
    m = KheSolnResourceMonitor(srs->soln, r, i);
    KheSingleResourceSolverAddMonitor(srs, m);
  }

  /* sort out the day info for the expression tree */
  KheSrsExprSetDayRange((KHE_SRS_EXPR) srs->root_expr, 0, srs);
  KheSrsExprSetSigIndexes((KHE_SRS_EXPR) srs->root_expr, srs);
  KheSrsExprAddExpressionsToDays((KHE_SRS_EXPR) srs->root_expr, srs);

  /* add suitable tasks for each day, skipping days with preassigned tasks */
  HaArrayForEach(srs->days, day, di)
    if( KheSrsDayAddTasks(day, srs, &preassigned_step) )
      di = KheDailyScheduleLastDayIndex(preassigned_step->daily_schedule);
           /* continue from after here */

  /* debug print of assembled mass */
  if( DEBUG5 )
    KheSingleResourceSolverDebug(srs, 1, 2, stderr);

  /* build up the partial solutions for each day */
  prev_day = NULL;
  HaArrayForEach(srs->days, day, di)
  {
    if( prev_day == NULL )
    {
      /* virtual prev_day contains a single partial solution (an empty one) */
      if( DEBUG2 )
	fprintf(stderr, "  extending from day 0 (1 partial soln)\n");
      ps = KheSrsPartialSolnMakeInit(srs);
      KheSingleResourceSolverExtend(srs, ps, day);
    }
    else
    {
      /* extend each partial solution from prev_day with tasks from day */
#if TESTING
      if( DEBUG7 )
	fprintf(stderr, "  running_time_per_day = %d, days = %d, di = %d\n",
	  HaArrayCount(srs->running_time_per_day), HaArrayCount(srs->days), di);
      HaArray(srs->running_time_per_day, di) = KheTimerElapsedTime(srs->timer);
#endif
      if( DEBUG2 )
	fprintf(stderr, "  extending from day %s (%d partial solns)\n",
	  KheTimeGroupId(prev_day->time_group), prev_day->partial_solns_count);
      if( full_dominance )
	HaArrayForEach(prev_day->partial_solns_array, ps, pos)
	  KheSingleResourceSolverExtend(srs, ps, day);
      else
	HpTableForEachValue(prev_day->partial_solns_table, ps, pos)
	  KheSingleResourceSolverExtend(srs, ps, day);
    }
    prev_day = day;
  }

  /* extract final solutions whose assignment count is in the desired range */
  day = HaArrayLast(srs->days);
  if( srs->full_dominance )
  {
    HaArrayForEach(day->partial_solns_array, ps, pos)
    {
      asst_count = HaArrayFirst(ps->signature);
      if( asst_count >= srs->min_assts && asst_count <= srs->max_assts )
	HaArrayAddLast(srs->final_solns, ps);
    }
  }
  else
  {
    HpTableForEachValue(day->partial_solns_table, ps, pos)
    {
      asst_count = HaArrayFirst(ps->signature);
      if( asst_count >= srs->min_assts && asst_count <= srs->max_assts )
	HaArrayAddLast(srs->final_solns, ps);
    }
  }
  HaArraySort(srs->final_solns, &KheSrsPartialSolnIncreasingAsstCountCmp);
  if( DEBUG2 )
  {
    fprintf(stderr, "  final solutions:\n");
    HaArrayForEach(srs->final_solns, ps, i)
      KheSrsPartialSolnDebug(ps, srs, 1, 2, stderr);
  }

  if( DEBUG1 )
    fprintf(stderr, "] KheSingleResourceSolverSolve returning\n");
}


/*****************************************************************************/
/*                                                                           */
/*  int KheSingleResourceSolverTimetableCount(KHE_SINGLE_RESOURCE_SOLVER srs)*/
/*                                                                           */
/*  Return the number of timetables found.                                   */
/*                                                                           */
/*****************************************************************************/

int KheSingleResourceSolverTimetableCount(KHE_SINGLE_RESOURCE_SOLVER srs)
{
  HnAssert(srs->resource != NULL,
    "KheSingleResourceSolverTimetableCount called out of order");
  return HaArrayCount(srs->final_solns);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSingleResourceSolverTimetable(KHE_SINGLE_RESOURCE_SOLVER srs,    */
/*    int i, int *asst_count, KHE_COST *r_cost, KHE_COST *soln_cost)         */
/*                                                                           */
/*  Report on the i'th timetable found by the solver.                        */
/*                                                                           */
/*****************************************************************************/

void KheSingleResourceSolverTimetable(KHE_SINGLE_RESOURCE_SOLVER srs,
  int i, int *asst_count, KHE_COST *cost)
{
  /* make sure there is an i'th solution */
  KHE_SRS_PARTIAL_SOLN ps;
  HnAssert(srs->resource != NULL,
    "KheSingleResourceSolverTimetable called out of order");
  HnAssert(0 <= i && i < HaArrayCount(srs->final_solns),
    "KheSingleResourceSolverTimetable:  i (%d) out of range (0 .. %d)",
    i, HaArrayCount(srs->final_solns) - 1);

  /* return the relevant fields of the i'th solution */
  ps = HaArray(srs->final_solns, i);
  *asst_count = HaArrayFirst(ps->signature);
  *cost = ps->cost;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSingleResourceSolverAdopt(KHE_SINGLE_RESOURCE_SOLVER srs, int i) */
/*                                                                           */
/*  Change srs's solution to include the i'th timetable.                     */
/*                                                                           */
/*****************************************************************************/

void KheSingleResourceSolverAdopt(KHE_SINGLE_RESOURCE_SOLVER srs, int i)
{
  /* make sure there is an i'th solution */
  KHE_SRS_PARTIAL_SOLN final_ps, rerun_ps;  int j;  KHE_COST cost, m_cost;
  KHE_MONITOR m;  KHE_RESOURCE_TIMETABLE_MONITOR rtm;
  HnAssert(srs->resource != NULL,
    "KheSingleResourceSolverAdopt called out of order");
  HnAssert(0 <= i && i < HaArrayCount(srs->final_solns),
    "KheSingleResourceSolverAdopt:  i (%d) out of range (0 .. %d)",
    i, HaArrayCount(srs->final_solns) - 1);

  /* load the i'th solution, and adopt it */
  final_ps = HaArray(srs->final_solns, i);
  KheSrsFullSolnLoad(srs->full_soln, final_ps, srs->resource);
  KheSrsFullSolnAdopt(srs->full_soln);

  /* assign the tasks of the i'th solution */
  /* ***
  srs->adopted_index = i;
  HaArrayClear(srs->adopted_tasks);
  final_ps = HaArray(srs->final_solns, i);
  for( ps = final_ps;  ps != NULL;  ps = ps->prev_ps )
    if( ps->task != NULL && KheTaskAsstResource(ps->task) == NULL )
    {
      ** assign srs->resource to ps->task **
      if( !KheTaskAssignResource(ps->task, srs->resource) )
	HnAbort("KheSingleResourceSolverAdopt internal error");
      HaArrayAddLast(srs->adopted_tasks, ps->task);
    }
  *** */

  /* check that the resulting resource costs agree with the final soln */
  if( DEBUG3 )
  {
    fprintf(stderr, "[ KheSingleResourceSolverAdopt(srs, %d)\n", i);
    rtm = KheResourceTimetableMonitor(srs->soln, srs->resource);
    KheResourceTimetableMonitorPrintTimetable(rtm, 6, 2, stderr);
    cost = KheCost(0, 0);
    for( j = 0; j < KheSolnResourceMonitorCount(srs->soln, srs->resource); j++ )
    {
      m = KheSolnResourceMonitor(srs->soln, srs->resource, j);
      switch( KheMonitorTag(m) )
      {
	case KHE_AVOID_UNAVAILABLE_TIMES_MONITOR_TAG:
	case KHE_CLUSTER_BUSY_TIMES_MONITOR_TAG:
	case KHE_LIMIT_BUSY_TIMES_MONITOR_TAG:
	case KHE_LIMIT_WORKLOAD_MONITOR_TAG:
	case KHE_LIMIT_ACTIVE_INTERVALS_MONITOR_TAG:

	  m_cost = KheMonitorCost(m);
	  if( m_cost > 0 )
	  {
	    fprintf(stderr, "  true cost of %s is %.5f\n", KheMonitorId(m),
	      KheCostShow(m_cost));
	    cost += m_cost;
	  }
	  break;

	default:

	  /* ignore monitors of other types */
	  break;
      }
    }
    if( cost != final_ps->cost )
    {
      fprintf(stderr, "KheSingleResourceSolverAdopt: cost discrepancy "
	"(true %.5f != single %.5f):\n", KheCostShow(cost),
	KheCostShow(final_ps->cost));
      KheSrsFullSolnDebug(srs->full_soln, 1, 2, stderr);
      KheSrsFullSolnUnAdopt(srs->full_soln);
      rerun_ps = KheFullSolnReRun(srs->full_soln, srs);
      fprintf(stderr, "  rerun_ps cost is %.5f\n", KheCostShow(rerun_ps->cost));
      HnAbort("aborting now (cost discrepancy)");
    }
    fprintf(stderr, "  %s assts %d, cost %.5f\n", KheResourceId(srs->resource),
      HaArrayFirst(final_ps->signature), KheCostShow(cost));
    fprintf(stderr, "] KheSingleResourceSolverAdopt returning\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSingleResourceSolverUnAdopt(KHE_SINGLE_RESOURCE_SOLVER srs)      */
/*                                                                           */
/*  Unadopt the most recently adopted full solution.                         */
/*                                                                           */
/*****************************************************************************/

void KheSingleResourceSolverUnAdopt(KHE_SINGLE_RESOURCE_SOLVER srs)
{
  KheSrsFullSolnUnAdopt(srs->full_soln);
  /* ***
  KHE_TASK task;  int j;

  ** make sure the call is legal **
  HnAssert(srs->resource != NULL,
    "KheSingleResourceSolverUnAdopt called out of order");
  HnAssert(0 <= i && i < HaArrayCount(srs->final_solns),
    "KheSingleResourceSolverUnAdopt:  i (%d) out of range (0 .. %d)",
    i, HaArrayCount(srs->final_solns) - 1);
  HnAssert(srs->adopted_index != -1, "KheSingleResourceSolverUnAdopt:  "
    "no previous matching call to KheSingleResourceSolverAdopt");
  HnAssert(i == srs->adopted_index, "KheSingleResourceSolverUnAdopt:  "
    "i (%d) not most recently adopted (%d)", i, srs->adopted_index);

  ** unassign the tasks that were assigned by KheSingleResourceSolverAdopt **
  HaArrayForEach(srs->adopted_tasks, task, j)
    if( !KheTaskUnAssignResource(task) )
      HnAbort("KheSingleResourceSolverUnAdopt internal error");

  ** reset adopted fields **
  HaArrayClear(srs->adopted_tasks);
  srs->adopted_index = -1;
  *** */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheSingleResourceSolverDebug(KHE_SINGLE_RESOURCE_SOLVER srs,        */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug printf of srs onto fp with the given verbosity and indent.         */
/*                                                                           */
/*****************************************************************************/

void KheSingleResourceSolverDebug(KHE_SINGLE_RESOURCE_SOLVER srs,
  int verbosity, int indent, FILE *fp)
{
  KHE_SRS_DAY day;  int i;
  HnAssert(indent >= 0, "KheSingleResourceSolverDebug: indent < 0");
  fprintf(fp, "%*s[ SingleResourceSolver(%s)\n", indent, "",
    srs->resource == NULL ? "@" : KheResourceId(srs->resource));
  HaArrayForEach(srs->days, day, i)
    KheSrsDayDebug(day, srs, verbosity, indent + 2, fp);
  KheSrsExprDebug((KHE_SRS_EXPR) srs->root_expr, srs, verbosity, indent+2, fp);
  fprintf(fp, "%*s]\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "testing"                                                      */
/*                                                                           */
/*****************************************************************************/

#if TESTING
typedef enum {
  KHE_SRS_TEST_WEAK,
  KHE_SRS_TEST_STRONG,
  KHE_SRS_TEST_WEAK_AND_STRONG
} KHE_SRS_TEST_TYPE;
#endif


/*****************************************************************************/
/*                                                                           */
/*  char *KheSrsTestTypeShow(KHE_SRS_TEST_TYPE type)                         */
/*                                                                           */
/*  Show a test type.                                                        */
/*                                                                           */
/*****************************************************************************/

#if TESTING
static char *KheSrsTestTypeShow(KHE_SRS_TEST_TYPE type)
{
  switch( type )
  {
    case KHE_SRS_TEST_WEAK:		return "w";
    case KHE_SRS_TEST_STRONG:		return "s";
    case KHE_SRS_TEST_WEAK_AND_STRONG:	return "ws";
    default:				return "?";
  }
}
#endif


/*****************************************************************************/
/*                                                                           */
/*  void TestName(KHE_INSTANCE ins, KHE_RESOURCE r, KHE_SRS_TEST_TYPE type,  */
/*    int min_assts, int max_assts, char *suffix, char buff[200])            */
/*                                                                           */
/*  Set buff to the name of a test with these attributes.                    */
/*                                                                           */
/*****************************************************************************/

#if TESTING
static void TestName(KHE_INSTANCE ins, KHE_RESOURCE r, KHE_SRS_TEST_TYPE type,
  int min_assts, int max_assts, char *suffix, char buff[200])
{
  if( min_assts == 0 )
  {
    if( max_assts == INT_MAX )
      snprintf(buff, 200, "%s-%s-%s-%s", KheInstanceId(ins),
	KheResourceId(r), KheSrsTestTypeShow(type), suffix);
    else
      snprintf(buff, 200, "%s-%s-%s-%d-%s", KheInstanceId(ins),
	KheResourceId(r), KheSrsTestTypeShow(type), min_assts, suffix);
  }
  else
  {
    if( max_assts == INT_MAX )
      snprintf(buff, 200, "%s-%s-%s--%d-%s", KheInstanceId(ins),
	KheResourceId(r), KheSrsTestTypeShow(type), max_assts, suffix);
    else
      snprintf(buff, 200, "%s-%s-%s-%d-%d-%s", KheInstanceId(ins),
	KheResourceId(r), KheSrsTestTypeShow(type), min_assts, max_assts,
	suffix);
  }
}
#endif


/*****************************************************************************/
/*                                                                           */
/*  void KheSrsSolveTest(KHE_SINGLE_RESOURCE_SOLVER srs, KHE_RESOURCE r,     */
/*    bool strong_dom, int min_assts, int max_assts, KHE_GRAPH time_graph,   */
/*    KHE_GRAPH size_graph, KHE_GRAPH cost_graph)                            */
/*                                                                           */
/*  Carry out a test with these attributes, and add a dataset to time_graph, */
/*  size_graph, and cost_graph recording the results.                        */
/*                                                                           */
/*****************************************************************************/

#if TESTING
static void KheSrsSolveTest(KHE_SINGLE_RESOURCE_SOLVER srs, KHE_RESOURCE r,
  bool strong_dom, int min_assts, int max_assts, KHE_GRAPH time_graph,
  KHE_GRAPH size_graph, KHE_GRAPH cost_graph)
{
  int i;  float val;  KHE_DATASET kd;  KHE_DATASET_TYPE kdt;  KHE_SRS_DAY day;
  int asst_count;  KHE_COST r_cost;

  /* solve, measuring running time */
  HaArrayClear(srs->running_time_per_day);
  HaArrayFill(srs->running_time_per_day, HaArrayCount(srs->days), 0.0);
  KheTimerResetStartTime(srs->timer);
  KheSingleResourceSolverSolve(srs, r, strong_dom, min_assts, max_assts);

  /* verify costs */
  for( i = 0;  i < KheSingleResourceSolverTimetableCount(srs);  i++ )
  {
    KheSingleResourceSolverAdopt(srs, i);
    KheSingleResourceSolverUnAdopt(srs);
  }

  /* add one data set to the running time graph */
  kdt = strong_dom ? KHE_DATASET_DOTTED : KHE_DATASET_DASHED;
  kd = KheDataSetAdd(time_graph, kdt);
  HaArrayForEach(srs->running_time_per_day, val, i)
    KhePointAdd(kd, (float) i + 1, val);

  /* add one data set to the table size graph */
  kd = KheDataSetAdd(size_graph, kdt);
  KhePointAdd(kd, 0.0, 0.0);
  HaArrayForEach(srs->days, day, i)
    KhePointAdd(kd, (float) i + 1, day->partial_solns_count);

  /* add one data set to the cost graph */
  kd = KheDataSetAdd(cost_graph, kdt);
  for( i = 0;  i < KheSingleResourceSolverTimetableCount(srs);  i++ )
  {
    KheSingleResourceSolverTimetable(srs, i, &asst_count, &r_cost);
    KhePointAdd(kd, (float) asst_count, (float) KheSoftCost(r_cost));
  }
}
#endif


/*****************************************************************************/
/*                                                                           */
/*  void KheSingleResourceSolverTest(KHE_SOLN soln, KHE_OPTIONS options,     */
/*    KHE_RESOURCE r)                                                        */
/*                                                                           */
/*  Test single resource solving on r.                                       */
/*                                                                           */
/*****************************************************************************/

void KheSingleResourceSolverTest(KHE_SOLN soln, KHE_OPTIONS options,
  KHE_RESOURCE r)
{
#if TESTING
  KHE_SINGLE_RESOURCE_SOLVER srs;
  char time_file_name[200], size_file_name[200], cost_file_name[200];
  KHE_FILE time_file, size_file, cost_file;
  KHE_GRAPH time_graph, size_graph, cost_graph;

  /* make a solver */
  srs = KheSingleResourceSolverMake(soln, options);

  /* initialize the time file and graph */
  TestName(KheSolnInstance(soln), r, KHE_SRS_TEST_WEAK_AND_STRONG,
    0, INT_MAX, "time", time_file_name);
  time_file = KheFileBegin(time_file_name, KHE_FILE_LOUT_STANDALONE);
  time_graph = KheGraphBegin(time_file);
  KheGraphSetBelowCaption(time_graph, "Days");
  KheGraphSetLeftCaptionAndGap(time_graph, "Secs", "1.0c");

  /* initialize the table size file and graph */
  TestName(KheSolnInstance(soln), r, KHE_SRS_TEST_WEAK_AND_STRONG,
    0, INT_MAX, "size", size_file_name);
  size_file = KheFileBegin(size_file_name,KHE_FILE_LOUT_STANDALONE);
  size_graph = KheGraphBegin(size_file);
  KheGraphSetBelowCaption(size_graph, "Days");
  KheGraphSetLeftCaptionAndGap(size_graph, "Size", "1.0c");

  /* initialize the cost file and graph */
  TestName(KheSolnInstance(soln), r, KHE_SRS_TEST_WEAK_AND_STRONG,
    0, INT_MAX, "cost", cost_file_name);
  cost_file = KheFileBegin(cost_file_name,KHE_FILE_LOUT_STANDALONE);
  cost_graph = KheGraphBegin(cost_file);
  KheGraphSetBelowCaption(cost_graph, "Assignments");
  KheGraphSetLeftCaptionAndGap(cost_graph, "Cost", "1.0c");

  /* test with strong and weak dominance */
  KheSrsSolveTest(srs, r, true,  0, INT_MAX, time_graph, size_graph,cost_graph);
  KheSrsSolveTest(srs, r, false, 0, INT_MAX, time_graph, size_graph,cost_graph);

  /* tidy up and return */
  KheSingleResourceSolverDelete(srs);
  KheGraphEnd(time_graph);
  KheFileEnd(time_file);
  KheGraphEnd(size_graph);
  KheFileEnd(size_file);
  KheGraphEnd(cost_graph);
  KheFileEnd(cost_file);
#else
  fprintf(stderr, "to test, set TESTING to 1 in khe_sr_single_resource.c"\n);
#endif
}
