/*****************************************************************************/
/*                                                                           */
/*  THE KHE HIGH SCHOOL TIMETABLING ENGINE                                   */
/*  COPYRIGHT (C) 2010 Jeffrey H. Kingston                                   */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 3, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         khe_sr_dynamic_resource.c                                  */
/*  DESCRIPTION:  Dynamic resource solving                                   */
/*                                                                           */
/*  This file defines many types with no obvious best ordering, so we've     */
/*  followed the traditional XHSTT ordering:  times, resources, events       */
/*  (just tasks here), constraints (called expressions here) and solutions;  */
/*  and after those, sets of solutions and solvers.  It is all fully         */
/*  documented in the Appendix to the User's Guide, in the same order.       */
/*                                                                           */
/*****************************************************************************/
#include "khe_solvers.h"
#include "khe_priqueue.h"
#include "howard_p.h"
#include <limits.h>
#include <float.h>
#include <math.h>

#define TESTING 1
#define COL_WIDTH  12		/* when debugging solutions */
#define COL_COUNT   7		/* when debugging solutions */

#define bool_show(x) ((x) ? "true" : "false")

#define UNDEF_INT -1
#define UNDEF_FLOAT -1.0

#define DEBUG_MONITOR_ID "Constraint:2/TR_28/2Fri3"
#define DEBUG_MONITOR(m) (false &&					\
  strcmp(KheMonitorId((KHE_MONITOR) m), DEBUG_MONITOR_ID) == 0)

#define	RERUN  1		/* rerun                        */
#define RERUN_MONITOR_ID ""

#define DEBUG1 1		/* KheDynamicResourceSolverDoSolve */
#define DEBUG3 0		/* task classes                 */
#define DEBUG4 0		/* KheDynamicResourceSolverMake */
#define DEBUG7 0		/* ResourceMake                 */
#define DEBUG8 0		/* expression construction      */
#define DEBUG9 0		/* KheDrsSolveOpen (costs)      */
#define DEBUG10 0		/* KheDrsExprOpen               */
#define DEBUG11 0		/* KheDrsResourceOpen           */
#define DEBUG12 0		/* KheDrsSolnExpandMakeAndMeld  */
#define DEBUG14 0		/* int seq cost child has opened*/
#define DEBUG15 0		/* KheDrsTaskAssign             */
#define DEBUG16 0		/* KheTaskUnAssignResource      */
#define DEBUG17 1		/* Search                       */
#define DEBUG18 0		/* KheDrsDayOpen                */
#define DEBUG19 0		/* KheDrsTaskClassOpen          */
#define DEBUG20 0		/* KheDrsAUIntervalFindRight    */
#define DEBUG21 0		/* Search statistics            */
#define DEBUG22 0		/* SolnExpand and SolnDoExpand  */
#define DEBUG24 0		/* rerun costs                  */
#define DEBUG26 0		/* trie stats                   */
#define DEBUG27 0		/* trie stats                   */
#define DEBUG28 0		/* ancestor and dominates freqs */
#define DEBUG30 0		/* FirstAndLast dominance debug */
#define DEBUG31 0		/* priqueue insert and delete   */
#define DEBUG32 0		/* DistanceToAncestor           */
#define DEBUG34 0		/* prune cost                   */
#define DEBUG35 0		/* two extra selection          */
#define DEBUG36 0		/* dom tables                   */
#define DEBUG37 0		/* dom tables                   */
#define DEBUG41 0		/* task class min cost          */
#define DEBUG42 0		/* dom tables                   */
#define DEBUG43 0		/* one extra dominates          */

#define DEBUG44 0		/* sub get                      */
#define DEBUG44_MONITOR_ID "Constraint:22/HN_1"
#define DEBUG44_MONITOR(m) (DEBUG44 && (m) != NULL && 			\
  strcmp(KheMonitorId((KHE_MONITOR) (m)), DEBUG44_MONITOR_ID) == 0)

#define DEBUG45(day) (0 && ((day) == NULL || (day)->solve_expand_count == 1))
#define DEBUG46 0		/* various                      */
#define DEBUG47(day) (0 && ((day) == NULL || (day)->solve_expand_count == 1))

#define DEBUG49	0
#define DEBUG50	0
#define DEBUG51(day) (0 && ((day) == NULL || (day)->solve_expand_count == 1))
#define DEBUG52(day, ds) (0 &&						\
  ((day) == NULL || (day)->solve_expand_count == 1) &&			\
  (ds)->open_shift_index == 27)

#define DEBUG54 0
#define DEBUG55 0
#define DEBUG56 0
#define DEBUG57 0

#define DEBUG58(mi)							\
  (false && strcmp(KheDrsMonitorInfoId(mi), "Constraint:9/HN_1") == 0 )

#define DEBUG59 0
#define DEBUG60 0
#define DEBUG61 0
#define DEBUG63 0
#define DEBUG64 0
#define DEBUG64_ID "1Tue"
#define DEBUG65 0
#define DEBUG66 0
#define DEBUG67 0

#define USE_DOM_CACHING 0

#define DEBUG_LAST_EDGE (false && (drs)->solve_daily_expand_limit == 0)


/*****************************************************************************/
/*                                                                           */
/*  Submodule "forward type declarations"                                    */
/*                                                                           */
/*  This submodule contains forward type declarations, but only if needed.   */
/*                                                                           */
/*****************************************************************************/

/* times */
typedef struct khe_drs_day_rec *KHE_DRS_DAY;
typedef HA_ARRAY(KHE_DRS_DAY) ARRAY_KHE_DRS_DAY;

/* resources */
typedef struct khe_drs_resource_rec *KHE_DRS_RESOURCE;
typedef HA_ARRAY(KHE_DRS_RESOURCE) ARRAY_KHE_DRS_RESOURCE;

typedef struct khe_drs_resource_set_rec *KHE_DRS_RESOURCE_SET;
typedef HA_ARRAY(KHE_DRS_RESOURCE_SET) ARRAY_KHE_DRS_RESOURCE_SET;

typedef struct khe_drs_resource_on_day_rec *KHE_DRS_RESOURCE_ON_DAY;
typedef HA_ARRAY(KHE_DRS_RESOURCE_ON_DAY) ARRAY_KHE_DRS_RESOURCE_ON_DAY;

/* events */
typedef struct khe_drs_task_rec *KHE_DRS_TASK;
typedef HA_ARRAY(KHE_DRS_TASK) ARRAY_KHE_DRS_TASK;

typedef struct khe_drs_task_on_day_rec *KHE_DRS_TASK_ON_DAY;
typedef HA_ARRAY(KHE_DRS_TASK_ON_DAY) ARRAY_KHE_DRS_TASK_ON_DAY;

typedef struct khe_drs_task_class_rec *KHE_DRS_TASK_CLASS;
typedef HA_ARRAY(KHE_DRS_TASK_CLASS) ARRAY_KHE_DRS_TASK_CLASS;

typedef struct khe_drs_shift_rec *KHE_DRS_SHIFT;
typedef HA_ARRAY(KHE_DRS_SHIFT) ARRAY_KHE_DRS_SHIFT;

typedef struct khe_drs_asst_to_task_set_rec *KHE_DRS_ASST_TO_TASK_SET;

typedef struct khe_drs_shift_asst_trie_rec *KHE_DRS_SHIFT_ASST_TRIE;
typedef HA_ARRAY(KHE_DRS_SHIFT_ASST_TRIE) ARRAY_KHE_DRS_SHIFT_ASST_TRIE;

/* signatures */
typedef struct khe_drs_signature_rec *KHE_DRS_SIGNATURE;
typedef HA_ARRAY(KHE_DRS_SIGNATURE) ARRAY_KHE_DRS_SIGNATURE;

typedef struct khe_drs_signature_set_rec *KHE_DRS_SIGNATURE_SET;
typedef HA_ARRAY(KHE_DRS_SIGNATURE_SET) ARRAY_KHE_DRS_SIGNATURE_SET;

typedef struct khe_drs_correlator_rec *KHE_DRS_CORRELATOR;

typedef struct khe_drs_signer_rec *KHE_DRS_SIGNER;
typedef HA_ARRAY(KHE_DRS_SIGNER) ARRAY_KHE_DRS_SIGNER;

typedef struct khe_drs_signer_set_rec *KHE_DRS_SIGNER_SET;

typedef struct khe_drs_dom_test_rec *KHE_DRS_DOM_TEST;
typedef HA_ARRAY(KHE_DRS_DOM_TEST) ARRAY_KHE_DRS_DOM_TEST;

/* constraints i.e. expressions */
typedef struct khe_drs_expr_rec *KHE_DRS_EXPR;
typedef HA_ARRAY(KHE_DRS_EXPR) ARRAY_KHE_DRS_EXPR;

/* assignments */
typedef struct khe_drs_asst_to_task_rec KHE_DRS_ASST_TO_TASK;
typedef HA_ARRAY(KHE_DRS_ASST_TO_TASK) ARRAY_KHE_DRS_ASST_TO_TASK;

typedef struct khe_drs_asst_to_task_class_rec *KHE_DRS_ASST_TO_TASK_CLASS;
typedef HA_ARRAY(KHE_DRS_ASST_TO_TASK_CLASS) ARRAY_KHE_DRS_ASST_TO_TASK_CLASS;

typedef struct khe_drs_asst_to_shift_rec *KHE_DRS_ASST_TO_SHIFT;
typedef HA_ARRAY(KHE_DRS_ASST_TO_SHIFT) ARRAY_KHE_DRS_ASST_TO_SHIFT;

/* solutions */
typedef struct khe_drs_soln_rec *KHE_DRS_SOLN;
typedef HA_ARRAY(KHE_DRS_SOLN) ARRAY_KHE_DRS_SOLN;
typedef HP_TABLE(KHE_DRS_SOLN) TABLE_KHE_DRS_SOLN;

/* sets of solutions */
typedef struct khe_drs_soln_list_rec *KHE_DRS_SOLN_LIST;
typedef HA_ARRAY(KHE_DRS_SOLN_LIST) ARRAY_KHE_DRS_SOLN_LIST;
typedef HP_TABLE(KHE_DRS_SOLN_LIST) TABLE_KHE_DRS_SOLN_LIST;

typedef struct khe_drs_soln_set_rec *KHE_DRS_SOLN_SET;
typedef HA_ARRAY(KHE_DRS_SOLN_SET) ARRAY_KHE_DRS_SOLN_SET;


/*****************************************************************************/
/*                                                                           */
/*  Submodule "type definitions - times"                                     */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_DAY_RANGE - a range of days                                 */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_day_range_rec {
  int			first;
  int			last;
} KHE_DRS_DAY_RANGE;

typedef HA_ARRAY(KHE_DRS_DAY_RANGE) ARRAY_KHE_DRS_DAY_RANGE;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_DAY - one day                                               */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_day_rec {
  int				frame_index;		/* index in frame    */
  int				open_day_index;		/* index when open   */
  KHE_TIME_GROUP		time_group;		/* day's time group  */
  ARRAY_KHE_DRS_SHIFT		shifts;			/* shifts            */
  KHE_DRS_SIGNER_SET		signer_set;		/* dom tests etc.    */
  KHE_DRS_SOLN_SET		soln_set;		/* today's solns     */
  KHE_DRS_SOLN_LIST		soln_list;		/* sorted soln_set   */
  int				soln_made_count;	/* solns made today  */
  int				solve_expand_count;	/* expanded today    */
};


/*****************************************************************************/
/*                                                                           */
/*  Submodule "type definitions - resources"                                 */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_RESOURCE_EXPAND_ROLE - resource's role in current expansion */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  KHE_DRS_RESOURCE_EXPAND_NO,
  KHE_DRS_RESOURCE_EXPAND_FIXED,
  KHE_DRS_RESOURCE_EXPAND_FREE
} KHE_DRS_RESOURCE_EXPAND_ROLE;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_RESOURCE - one resource                                     */
/*                                                                           */
/*****************************************************************************/
typedef struct khe_drs_dim2_table_rec *KHE_DRS_DIM2_TABLE;

struct khe_drs_resource_rec {
  KHE_RESOURCE			resource;		/* the resource      */
  int				open_resource_index;	/* index when open   */
  ARRAY_KHE_DRS_RESOURCE_ON_DAY	days;			/* doing each day    */
  KHE_DRS_RESOURCE_EXPAND_ROLE	expand_role;		/* current expand    */
  ARRAY_KHE_DRS_ASST_TO_SHIFT	expand_assts_to_shifts;	/* current expand    */
  ARRAY_KHE_DRS_ASST_TO_TASK_CLASS expand_assts;	/* current expand    */
  KHE_DRS_ASST_TO_TASK_CLASS	expand_free_day_asst;	/* current expand    */
  KHE_DRS_DIM2_TABLE		expand_dom_test_cache;	/* current expand    */
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_RESOURCE_SET - a set of resources                           */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_resource_set_rec {
  ARRAY_KHE_DRS_RESOURCE	resources;		/* the resources     */
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_RESOURCE_ON_DAY - what one resource is doing on one day     */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_resource_on_day_rec {
  KHE_DRS_RESOURCE		encl_dr;		/* the resource      */
  KHE_DRS_DAY			day;			/* the day           */
  KHE_DRS_TASK_ON_DAY		closed_asst;		/* asst when closed  */
  ARRAY_KHE_DRS_EXPR		external_today;		/* external today    */
  KHE_DRS_SIGNER		signer;			/* signer         */
};


/*****************************************************************************/
/*                                                                           */
/*  Submodule "type definitions - events"                                    */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_TASK_EXPAND_ROLE - task's role in current expansion              */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  KHE_DRS_TASK_EXPAND_NO,
  KHE_DRS_TASK_EXPAND_FIXED,
  KHE_DRS_TASK_EXPAND_MUST,
  KHE_DRS_TASK_EXPAND_FREE
} KHE_DRS_TASK_EXPAND_ROLE;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_TASK - one proper root task                                 */
/*                                                                           */
/*  The days are sorted chronologically, and this is used to test whether    */
/*  a given task on day is the first of its task.                            */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_task_rec {
  KHE_DRS_TASK_CLASS		encl_dtc;
  int				index_in_encl_dtc;
  KHE_DRS_TASK_EXPAND_ROLE	expand_role;		/* used by SolnExpand*/
  bool				open;			/* true when open    */
  KHE_TASK			task;			/* the KHE task      */
  KHE_DRS_RESOURCE		closed_asst;		/* asst when closed  */
  KHE_COST			asst_cost;		/* assignment cost   */
  KHE_COST			non_asst_cost;		/* non-asst cost     */
  ARRAY_KHE_DRS_TASK_ON_DAY	days;
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_TASK_ON_DAY - what one proper root task is doing on one day */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_task_on_day_rec {
  KHE_DRS_TASK			encl_dt;		/* enclosing task    */
  KHE_DRS_DAY			day;			/* the day           */
  KHE_TASK			task;			/* atomic task today */
  KHE_TIME			time;			/* time on day       */
  KHE_DRS_RESOURCE_ON_DAY	closed_asst;		/* asst when closed  */
  ARRAY_KHE_DRS_EXPR		external_today;		/* leaf today        */
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_TASK_CLASS - a set of tasks, taken from one KHE_TASK_CLASS  */
/*                                                                           */
/*  For efficiency, unassigned_tasks is managed lazily.  Every unassigned    */
/*  task is in it, but not in order, and there may be assigned tasks in it.  */
/*  These defects are corrected when the task class is opened for solving.   */
/*                                                                           */
/*  Field expand_prev_unfixed is the index in unassigned_tasks of the        */
/*  unfixed task most recently given away to expansion, or -1 if no tasks    */
/*  are currently given away, or a large number if there are none to give    */
/*  away.                                                                    */
/*                                                                           */
/*  Obsolete:                                                                */
/*  Field expand_next_unfixed is the index in unassigned_tasks of the next   */
/*  available unfixed unassigned task, unless it points beyond the end of    */
/*  unassigned_tasks, in which case there is no such task.                   */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_task_class_rec {
  KHE_TASK_CLASS		orig_task_class;	/* origin of this    */
  KHE_DRS_SHIFT			encl_shift;		/* enclosing shift   */
  KHE_DRS_DAY_RANGE		day_range;		/* days running      */
  ARRAY_KHE_DRS_TASK		all_tasks;		/* all tasks in class*/
  ARRAY_KHE_DRS_TASK		unassigned_tasks;	/* unassigned tasks  */
  int				expand_prev_unfixed;	/* used by SolnExpand*/
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_SHIFT - the set of all task classes with the same times     */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_shift_rec {
  KHE_DRS_DAY			encl_day;		/* enclosing day     */
  int				open_shift_index;	/* set when open     */
  int				expand_must_assign_count;
  int				expand_max_included_free_resource_count;
  ARRAY_KHE_DRS_TASK_CLASS	task_classes;		/* the classes       */
  KHE_DRS_SIGNER		signer;			/* the signer        */
  KHE_DRS_SHIFT_ASST_TRIE	asst_trie;
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_SHIFT_ASST - an assignment of resources to one shift        */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_shift_asst_rec {
  KHE_DRS_SIGNATURE		sig;
  KHE_DRS_ASST_TO_TASK_SET	assts_to_tasks;
} *KHE_DRS_SHIFT_ASST;

typedef HA_ARRAY(KHE_DRS_SHIFT_ASST) ARRAY_KHE_DRS_SHIFT_ASST;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_SHIFT_ASST_TRIE - shift assignment lists indexed by r-set   */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_shift_asst_trie_rec {
  ARRAY_KHE_DRS_SHIFT_ASST		assts;		/* assts at this pt  */
  ARRAY_KHE_DRS_SHIFT_ASST_TRIE		children;	/* children */
};


/*****************************************************************************/
/*                                                                           */
/*  Submodule "type definitions - signatures"                                */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_DOM_KIND - the kind of dominance testing to use             */
/*                                                                           */
/*****************************************************************************/

/* *** defined in khe_solvers.h
typedef enum {
  KHE_DRS_DOM_LIST_NONE,
  KHE_DRS_DOM_LIST_STRONG,
  KHE_DRS_DOM_LIST_TRADEOFF,
  KHE_DRS_DOM_LIST_UNIFORM,
  KHE_DRS_DOM_HASH_WEAK,
  KHE_DRS_DOM_HASH_MEDIUM,
  KHE_DRS_DOM_TRIE_STRONG,
  KHE_DRS_DOM_TRIE_TRADEOFF,
  KHE_DRS_DOM_INDEXED_TRADEOFF,
  KHE_DRS_DOM_INDEXED_UNIFORM
} KHE_DRS_DOM_KIND;
*** */


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_COST_TUPLE - a pair of costs stored in a dominance table     */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_cost_tuple_rec {
  KHE_COST			psi;
  KHE_COST			psi0;
  bool				psi_plus_defined;
  KHE_COST			psi_plus;
} KHE_DRS_COST_TUPLE;

typedef HA_ARRAY(KHE_DRS_COST_TUPLE) ARRAY_KHE_DRS_COST_TUPLE;


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DIM1_TABLE - a one-dimensional table of tuples                   */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_dim1_table_rec {
  int				offset;
  ARRAY_KHE_DRS_COST_TUPLE	children;
} *KHE_DRS_DIM1_TABLE;

typedef HA_ARRAY(KHE_DRS_DIM1_TABLE) ARRAY_KHE_DRS_DIM1_TABLE;


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DIM2_TABLE - a two-dimensional table of tuples                   */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_dim2_table_rec {
  int				offset;
  ARRAY_KHE_DRS_DIM1_TABLE	children;
  char				*debug_str;
};

typedef HA_ARRAY(KHE_DRS_DIM2_TABLE) ARRAY_KHE_DRS_DIM2_TABLE;


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DIM3_TABLE - a three-dimensional table of tuples                 */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_dim3_table_rec {
  int				offset;
  ARRAY_KHE_DRS_DIM2_TABLE	children;
  char				*debug_str;
} *KHE_DRS_DIM3_TABLE;

typedef HA_ARRAY(KHE_DRS_DIM3_TABLE) ARRAY_KHE_DRS_DIM3_TABLE;


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DIM4_TABLE - a four-dimensional table of tuples                  */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_dim4_table_rec {
  int				offset;
  ARRAY_KHE_DRS_DIM3_TABLE	children;
  char				*debug_str;
} *KHE_DRS_DIM4_TABLE;

typedef HA_ARRAY(KHE_DRS_DIM4_TABLE) ARRAY_KHE_DRS_DIM4_TABLE;


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DIM5_TABLE - a five-dimensional table of tuples                  */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_dim5_table_rec {
  int				offset;
  ARRAY_KHE_DRS_DIM4_TABLE	children;
  char				*debug_str;
} *KHE_DRS_DIM5_TABLE;

typedef HA_ARRAY(KHE_DRS_DIM5_TABLE) ARRAY_KHE_DRS_DIM5_TABLE;


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST_TYPE - type of dom test                                 */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  KHE_DRS_DOM_TEST_UNUSED,			/* unused                    */
  KHE_DRS_DOM_TEST_STRONG,			/* strong                    */
  KHE_DRS_DOM_TEST_TRADEOFF,			/* tradeoff                  */
  KHE_DRS_DOM_TEST_UNIFORM,			/* uniform                   */
  KHE_DRS_DOM_TEST_CORR1_PARENT,		/* parent corr1              */
  KHE_DRS_DOM_TEST_CORR1_CHILD,			/* child corr1               */
  KHE_DRS_DOM_TEST_CORR2_CHILD,			/* child corr2               */
  KHE_DRS_DOM_TEST_CORR3_FIRST,			/* first corr3               */
  KHE_DRS_DOM_TEST_CORR3_MID,			/* in the middle             */
  KHE_DRS_DOM_TEST_CORR3_LAST,			/* last corr3                */
  KHE_DRS_DOM_TEST_CORR4_FIRST,			/* first corr4               */
  KHE_DRS_DOM_TEST_CORR4_MID,			/* in the middle             */
  KHE_DRS_DOM_TEST_CORR4_LAST			/* last corr4                */
} KHE_DRS_DOM_TEST_TYPE;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_SIGNATURE - a signature                                     */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_signature_rec {
  int			reference_count;	/* reference count           */
  int			asst_to_shift_index;	/* index of enclosing a_to_s */
  KHE_COST		cost;			/* cost                      */
  HA_ARRAY_INT		states;			/* expr states               */
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_SIGNATURE_SET - a set of signatures                         */
/*                                                                           */
/*  KHE_DRS_SIGNATURE_SET is a pointer type as usual, but in KHE_DRS_SOLN    */
/*  it is expanded ("struct khe_drs_signature_set_rec") to save memory.      */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_signature_set_rec {
  KHE_COST			cost;
  ARRAY_KHE_DRS_SIGNATURE	signatures;
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_CORRELATOR - an expression correlator                       */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_correlator_rec {
  bool				make_correlated;
  HA_ARRAY_INT			children;
  HA_ARRAY_INT			positives;
  HA_ARRAY_INT			negatives;
  HA_ARRAY_INT			singles;
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_SIGNER_TYPE - the type of a signer                          */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  KHE_DRS_SIGNER_DAY,
  KHE_DRS_SIGNER_RESOURCE_ON_DAY,
  KHE_DRS_SIGNER_SHIFT
} KHE_DRS_SIGNER_TYPE;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_SIGNER - a signature controller                             */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_signer_rec {
  ARRAY_KHE_DRS_EXPR		internal_exprs;		/* that affect this  */
  ARRAY_KHE_DRS_DOM_TEST	dom_tests;
  HA_ARRAY_INT			eq_dom_test_indexes;	/* where = tests are */
  int				last_hard_cost_index;
  bool				debug_eval_if_rerun;	/* debug evals       */
  KHE_DRS_CORRELATOR		correlator;		/* always present    */
  KHE_DRS_SIGNER_TYPE		type;
  union {
    KHE_DRS_DAY			day;
    KHE_DRS_RESOURCE_ON_DAY	resource_on_day;
    KHE_DRS_SHIFT		shift;
  } u;
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_SIGNER_TEST - signer tests to do (hard, soft, or all)       */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  KHE_DRS_SIGNER_HARD,
  KHE_DRS_SIGNER_SOFT,
  KHE_DRS_SIGNER_ALL
} KHE_DRS_SIGNER_TEST;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_SIGNER_SET - a set of signers                               */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_signer_set_rec {
  KHE_DRS_DAY			day;
  ARRAY_KHE_DRS_SIGNER		signers;
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_DOM_TEST - a dominance test                                 */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_dom_test_rec {
  KHE_DRS_DOM_TEST_TYPE	type;			/* type of test              */
  int			correlated_delta;	/* if correlated             */
  KHE_DRS_EXPR		expr;			/* originating expression    */
  bool			allow_zero;		/* allow zero ("Z" in doc)   */
  int			min_limit;              /* min limit (0 if none)     */
  int			max_limit;              /* max limit (INT_MAX if ne) */
  int			a;			/* "a" in doc                */
  int			b;			/* "b" in doc                */
  KHE_COST		tradeoff_cost;		/* tradeoff cost, if tradeoff*/
  KHE_DRS_DIM2_TABLE	main_dom_table2;	/* uniform dominance table   */
  KHE_DRS_DIM4_TABLE	corr_dom_table4;	/* uniform dominance table   */
  KHE_MONITOR		monitor;		/* monitor, if any           */
};


/*****************************************************************************/
/*                                                                           */
/*  Submodule "type definitions - constraints (i.e. expressions)"            */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_MONITOR_INFO - information about one monitor                */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_monitor_info_rec {
  KHE_MONITOR		monitor;
  KHE_COST		rerun_open_and_search_cost;
  KHE_COST		rerun_open_and_close_cost;
} *KHE_DRS_MONITOR_INFO;

typedef HA_ARRAY(KHE_DRS_MONITOR_INFO) ARRAY_KHE_DRS_MONITOR_INFO;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_OP - an open, search, or close operation type               */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  KHE_DRS_OPEN,
  KHE_DRS_SEARCH,
  KHE_DRS_CLOSE
} KHE_DRS_OP;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_TAG                                                    */
/*                                                                           */
/*  Tags for expression types.                                               */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  KHE_DRS_EXPR_ASSIGNED_TASK_TAG,
  KHE_DRS_EXPR_BUSY_TIME_TAG,
  KHE_DRS_EXPR_FREE_TIME_TAG,
  KHE_DRS_EXPR_WORK_TIME_TAG,
  KHE_DRS_EXPR_BUSY_DAY_TAG,
  KHE_DRS_EXPR_FREE_DAY_TAG,
  KHE_DRS_EXPR_WORK_DAY_TAG,
  KHE_DRS_EXPR_OR_TAG,
  KHE_DRS_EXPR_AND_TAG,	
  KHE_DRS_EXPR_INT_SUM_TAG,
  KHE_DRS_EXPR_FLOAT_SUM_TAG,
  KHE_DRS_EXPR_INT_DEV_TAG,
  KHE_DRS_EXPR_FLOAT_DEV_TAG,
  KHE_DRS_EXPR_INT_SUM_COST_TAG,
  KHE_DRS_EXPR_INT_SEQ_COST_TAG
} KHE_DRS_EXPR_TAG;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_ADJUST_TYPE - types of signature value adjustments          */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  KHE_DRS_ADJUST_ORDINARY,
  KHE_DRS_ADJUST_NO_MAX,
  KHE_DRS_ADJUST_LINEAR,
  KHE_DRS_ADJUST_STEP
} KHE_DRS_ADJUST_TYPE;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_VALUE - the value of an expression; could be int or float   */
/*                                                                           */
/*****************************************************************************/

typedef union {
  int			int_val;		/* value, if int   */
  float			float_val;		/* value, if float */
} KHE_DRS_VALUE;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_PARENT - a expression's parent                              */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_parent_rec {
  KHE_DRS_EXPR		expr;			/* the parent expression */
  int			index;			/* index in parent       */
} KHE_DRS_PARENT;

typedef HA_ARRAY(KHE_DRS_PARENT) ARRAY_KHE_DRS_PARENT;


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_OPEN_CHILDREN - a sequence of open children                      */
/*                                                                           */
/*  Field oc->children is a sequence of children sorted by an "open          */
/*  index" which could be either the last open day index of the child's      */
/*  open day range, or the open shift index of the shift that the child      */
/*  is affected by.                                                          */
/*                                                                           */
/*  Field oc->range is the range of open indexes of the children.  If there  */
/*  is at least one child this range must be non-empty; if there are no      */
/*  children it must be empty.                                               */
/*                                                                           */
/*  Field oc->child_indexes is an array of indexes into the children array,  */
/*  such that for each open index oi of a child in the array,                */
/*                                                                           */
/*    oc->child_indexes[oi - oc->range.first]                                */
/*                                                                           */
/*  is the index in oc->children of the first open child x such that the     */
/*  open index of x is oi or more.  Equivalently, it is the number of        */
/*  children x whose open index is strictly less than oi, as returned by     */
/*                                                                           */
/*    int KheDrsOpenChildrenBefore(KHE_DRS_OPEN_CHILDREN oc, int index)      */
/*                                                                           */
/*  There is also a sentinel at the end of oc->child_indexes holding the     */
/*  total number of children, so that for any open index oi, the iterator    */
/*                                                                           */
/*    #define KheDrsOpenChildrenForEach(oc, oi, x, i)                   \    */
/*      i1 = KheDrsOpenChildrenBefore((oc), (oi));                      \    */
/*      i2 = KheDrsOpenChildrenBefore((oc), (oi) + 1);                  \    */
/*      for( (i) = i1;                                                  \    */
/*           (i) < i2 ? ((x) = HaArray((oc)->children, (i)), true) : false;\ */
/*           (i)++ )                                                         */
/*                                                                           */
/*  sets x to each child in turn whose open index is oi.                     */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_open_children_rec {
  ARRAY_KHE_DRS_EXPR	children;
  KHE_DRS_DAY_RANGE	range;
  HA_ARRAY_INT		child_indexes;
} *KHE_DRS_OPEN_CHILDREN;


/*****************************************************************************/
/*                                                                           */
/*  INHERIT_KHE_DRS_EXPR - fields inherited by all expression types          */
/*                                                                           */
/*****************************************************************************/

#define INHERIT_KHE_DRS_EXPR						\
  KHE_DRS_EXPR_TAG	tag;			/* tag field      */	\
  bool			gathered;		/* for opening    */	\
  int			postorder_index;	/* expr index     */	\
  KHE_DRS_RESOURCE	resource;		/* if resource    */	\
  KHE_DRS_VALUE		u;			/* value          */	\
  ARRAY_KHE_DRS_PARENT	parents;		/* parent exprs   */	\
  ARRAY_KHE_DRS_EXPR	children;		/* child exprs    */	\
  struct khe_drs_open_children_rec open_children_by_day; /* sorted by day  */ \
  HA_ARRAY_INT		sig_indexes;		/* for each day   */


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR - abstract supertype of expressions                    */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_expr_rec {
  INHERIT_KHE_DRS_EXPR
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_ASSIGNED_TASK                                          */
/*                                                                           */
/*  An expression derived from a task on day, whose value is 1 if the task   */
/*  is assigned a resource from the given resource group, and 0 otherwise.   */
/*                                                                           */
/*  We include the task on day because we merge equal expressions of this    */
/*  kind, and that is only possible when the task is in the expression.      */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expr_assigned_task_rec {
  INHERIT_KHE_DRS_EXPR
  KHE_DRS_TASK_ON_DAY		task_on_day;
  KHE_RESOURCE_GROUP		resource_group;
} *KHE_DRS_EXPR_ASSIGNED_TASK;

typedef HA_ARRAY(KHE_DRS_EXPR_ASSIGNED_TASK) ARRAY_KHE_DRS_EXPR_ASSIGNED_TASK;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_BUSY_TIME                                              */
/*                                                                           */
/*  An expression whose value is 1 when the resource is busy at a given      */
/*  time, and 0 when the resource is free then.  The expression stores the   */
/*  time, but only for debugging.                                            */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expr_busy_time_rec {
  INHERIT_KHE_DRS_EXPR
  KHE_DRS_RESOURCE_ON_DAY	resource_on_day;  /* the resource and day    */
  KHE_TIME			time;		/* the time we're busy at    */
} *KHE_DRS_EXPR_BUSY_TIME;

typedef HA_ARRAY(KHE_DRS_EXPR_BUSY_TIME) ARRAY_KHE_DRS_EXPR_BUSY_TIME;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_FREE_TIME                                              */
/*                                                                           */
/*  An expression whose value is 1 when the resource is free at a given      */
/*  time, and 0 when the resource is busy then.  The expression stores the   */
/*  time, but only for debugging.                                            */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expr_free_time_rec {
  INHERIT_KHE_DRS_EXPR
  KHE_DRS_RESOURCE_ON_DAY	resource_on_day;  /* the resource and day    */
  KHE_TIME			time;		/* the time we're free at    */
} *KHE_DRS_EXPR_FREE_TIME;

typedef HA_ARRAY(KHE_DRS_EXPR_FREE_TIME) ARRAY_KHE_DRS_EXPR_FREE_TIME;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_WORK_TIME                                              */
/*                                                                           */
/*  An expression whose value is the workload incurred by the resource at    */
/*  a given time.  This will be 0.0 if the resource is free then.  The       */
/*  expression stores the time, but only for debugging.                      */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expr_work_time_rec {
  INHERIT_KHE_DRS_EXPR
  KHE_DRS_RESOURCE_ON_DAY	resource_on_day;  /* the resource and day    */
  KHE_TIME			time;		/* the time of this workload */
} *KHE_DRS_EXPR_WORK_TIME;

typedef HA_ARRAY(KHE_DRS_EXPR_WORK_TIME) ARRAY_KHE_DRS_EXPR_WORK_TIME;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_BUSY_DAY                                               */
/*                                                                           */
/*  An expression whose value is 1 when the resource is busy on a given      */
/*  day, and 0 when the resource is free then.  The expression stores the    */
/*  day (as a time group), and not just for debugging.                       */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expr_busy_day_rec {
  INHERIT_KHE_DRS_EXPR
  KHE_DRS_RESOURCE_ON_DAY	resource_on_day;  /* the resource and day    */
  KHE_TIME_GROUP		time_group;	/* the day we're busy at     */
} *KHE_DRS_EXPR_BUSY_DAY;

typedef HA_ARRAY(KHE_DRS_EXPR_BUSY_DAY) ARRAY_KHE_DRS_EXPR_BUSY_DAY;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_FREE_DAY                                               */
/*                                                                           */
/*  An expression whose value is 1 when the resource is free on a given      */
/*  day, and 0 when the resource is busy then.  The expression stores the    */
/*  day (as a time group), and not just for debugging.                       */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expr_free_day_rec {
  INHERIT_KHE_DRS_EXPR
  KHE_DRS_RESOURCE_ON_DAY	resource_on_day;  /* the resource and day    */
  KHE_TIME_GROUP		time_group;	/* the day we're free at     */
} *KHE_DRS_EXPR_FREE_DAY;

typedef HA_ARRAY(KHE_DRS_EXPR_FREE_DAY) ARRAY_KHE_DRS_EXPR_FREE_DAY;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_WORK_DAY                                               */
/*                                                                           */
/*  An expression whose value is the workload incurred by the resource on    */
/*  a given day.  This will be 0.0 if the resource is free then.  The        */
/*  expression stores the day (as a time group), but only for debugging.     */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expr_work_day_rec {
  INHERIT_KHE_DRS_EXPR
  KHE_DRS_RESOURCE_ON_DAY	resource_on_day;  /* the resource and day    */
  KHE_TIME_GROUP		time_group;	/* the day we're free at     */
} *KHE_DRS_EXPR_WORK_DAY;

typedef HA_ARRAY(KHE_DRS_EXPR_WORK_DAY) ARRAY_KHE_DRS_EXPR_WORK_DAY;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_OR                                                     */
/*                                                                           */
/*  An expression whose value is the logical `or' of its children's values.  */
/*  Here Booleans are represented by integers (0 for false, 1 for true).     */
/*                                                                           */
/*  The closed state here is the number of closed children with value 1.     */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expr_or_rec {
  INHERIT_KHE_DRS_EXPR
  int				closed_state;
} *KHE_DRS_EXPR_OR;

typedef HA_ARRAY(KHE_DRS_EXPR_OR) ARRAY_KHE_DRS_EXPR_OR;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_AND                                                    */
/*                                                                           */
/*  An expression whose value is the logical `and' of its children's values. */
/*  Here Booleans are represented by integers (0 for false, 1 for true).     */
/*                                                                           */
/*  The closed state here is the number of closed children with value 0.     */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expr_and_rec {
  INHERIT_KHE_DRS_EXPR
  int				closed_state;
} *KHE_DRS_EXPR_AND;

typedef HA_ARRAY(KHE_DRS_EXPR_AND) ARRAY_KHE_DRS_EXPR_AND;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_INT_SUM                                                */
/*                                                                           */
/*  An expression whose value is the sum of its children's values.           */
/*                                                                           */
/*  The closed state here is the sum of the values of the closed children.   */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expr_int_sum_rec {
  INHERIT_KHE_DRS_EXPR
  int				closed_state;
} *KHE_DRS_EXPR_INT_SUM;

typedef HA_ARRAY(KHE_DRS_EXPR_INT_SUM) ARRAY_KHE_DRS_EXPR_INT_SUM;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_FLOAT_SUM                                              */
/*                                                                           */
/*  An expression whose value is the sum of its children's values.  The      */
/*  expression's value, and its children's values, have type float.          */
/*                                                                           */
/*  The closed state here is the sum of the values of the closed children.   */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expr_float_sum_rec {
  INHERIT_KHE_DRS_EXPR
  float				closed_state;
} *KHE_DRS_EXPR_FLOAT_SUM;

typedef HA_ARRAY(KHE_DRS_EXPR_FLOAT_SUM) ARRAY_KHE_DRS_EXPR_FLOAT_SUM;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_INT_DEV                                                */
/*                                                                           */
/*  An expression whose value is the deviation of its integer-valued child.  */
/*                                                                           */
/*  The closed state here is the value of the closed child.  It is not       */
/*  stored here explicitly; it is retrieved from the child when required.    */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expr_int_dev_rec {
  INHERIT_KHE_DRS_EXPR
  int				min_limit;	/* minimum limit             */
  int				max_limit;	/* maximum limit             */
  bool				allow_zero;	/* true if 0 has deviation 0 */
} *KHE_DRS_EXPR_INT_DEV;

typedef HA_ARRAY(KHE_DRS_EXPR_INT_DEV) ARRAY_KHE_DRS_EXPR_INT_DEV;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_FLOAT_DEV                                              */
/*                                                                           */
/*  An expression whose value is the integer-valued deviation of its         */
/*  float-valued child.                                                      */
/*                                                                           */
/*  The closed state here is the value of the closed child.  It is not       */
/*  stored here explicitly; it is retrieved from the child when required.    */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expr_float_dev_rec {
  INHERIT_KHE_DRS_EXPR
  int				min_limit;	/* minimum limit             */
  int				max_limit;	/* maximum limit             */
  bool				allow_zero;	/* true if 0 has deviation 0 */
} *KHE_DRS_EXPR_FLOAT_DEV;

typedef HA_ARRAY(KHE_DRS_EXPR_FLOAT_DEV) ARRAY_KHE_DRS_EXPR_FLOAT_DEV;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_COST                                                   */
/*                                                                           */
/*  This type is the abstract supertype of KHE_DRS_EXPR_INT_SUM_COST and     */
/*  KHE_DRS_EXPR_INT_SEQ_COST.  It is used, but not much.                    */
/*                                                                           */
/*****************************************************************************/

#define INHERIT_KHE_DRS_EXPR_COST					   \
  INHERIT_KHE_DRS_EXPR							   \
  KHE_COST_FUNCTION	cost_fn;		/* cost function        */ \
  KHE_COST		combined_weight;	/* combined weight      */

typedef struct khe_drs_expr_cost_rec {
  INHERIT_KHE_DRS_EXPR_COST
} *KHE_DRS_EXPR_COST;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_INT_SUM_COST                                           */
/*                                                                           */
/*  Combines INT_SUM, INT_DEV, and COST into a single expression.  This      */
/*  allows important optimizations in some cases, depending on adjust_type.  */
/*                                                                           */
/*  The closed state here is the sum of the values of the closed children.   */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expr_int_sum_cost_rec {
  INHERIT_KHE_DRS_EXPR_COST
  KHE_DRS_MONITOR_INFO		monitor_info;		/* monitor (debug)   */
  int				min_limit;		/* minimum limit     */
  int				max_limit;		/* maximum limit     */
  KHE_DRS_ADJUST_TYPE		adjust_type;		/* type of adjustment*/
  int				history_before;		/* history value befo*/
  int				history_after;		/* history value afte*/
  int				history;		/* history value     */
  int				closed_state;		/* closed state      */
  bool				allow_zero;		/* allow zero flag   */
  KHE_DRS_DIM3_TABLE		main_dom_table3;	/* dominance testing */
  KHE_DRS_DIM5_TABLE		corr_dom_table5;	/* dominance testing */
  struct khe_drs_open_children_rec open_children_by_shift; /* sorted by shift*/
} *KHE_DRS_EXPR_INT_SUM_COST;

typedef HA_ARRAY(KHE_DRS_EXPR_INT_SUM_COST) ARRAY_KHE_DRS_EXPR_INT_SUM_COST;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_CLOSED_SEQ                                                  */
/*                                                                           */
/*  A sequence of consecutive closed children, summarized.  The fields are   */
/*                                                                           */
/*    start_index                                                            */
/*      The index in the sequence of all children where the sequence starts. */
/*                                                                           */
/*    stop_index                                                             */
/*      The index in the sequence of all children where the sequence         */
/*      stops; that is, the first child index not included in it.  So        */
/*      start_index <= stop_index, and when they are equal, the sequence     */
/*      is empty.                                                            */
/*                                                                           */
/*    active_at_left                                                         */
/*      The length of the a-interval at the extreme left, or 0 if none.      */
/*                                                                           */
/*    active_at_right                                                        */
/*      The length of the a-interval at the extreme right, or 0 if none.     */
/*                                                                           */
/*  If there is a single a-interval running the full length of the           */
/*  closed sequence (if all its children are active), then                   */
/*                                                                           */
/*    active_at_left == active_at_right == stop_index - start_index          */
/*                                                                           */
/*****************************************************************************/

typedef struct {
  int		start_index;
  int		stop_index;
  int		active_at_left;
  int		active_at_right;
} *KHE_DRS_CLOSED_SEQ;

typedef HA_ARRAY(KHE_DRS_CLOSED_SEQ) ARRAY_KHE_DRS_CLOSED_SEQ;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_A_INTERVAL - one a-interval                                 */
/*                                                                           */
/*  This may include indexes from history.                                   */
/*                                                                           */
/*  Note.  If the length of ai is 0, it does not matter whether an           */
/*  unassigned child precedes it or not, because its cost will be 0          */
/*  anyway.  However we must keep the value correct because there may        */
/*  be a merge later that makes it relevant.                                 */
/*                                                                           */
/*****************************************************************************/

typedef struct {
  int		start_index;		/* index of first child              */
  int		stop_index;		/* index after last child            */
  bool		unassigned_precedes;	/* true if unassigned child precedes */
} KHE_DRS_A_INTERVAL;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_AU_INTERVAL - one au-interval                               */
/*                                                                           */
/*  This may include indexes from history and history_after.                 */
/*                                                                           */
/*****************************************************************************/

typedef struct {
  int		start_index;		/* index of first child              */
  int		stop_index;		/* index after last child            */
  bool		has_active_child;	/* true if contains an active child  */
} KHE_DRS_AU_INTERVAL;


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SEQ_TYPE - whether int seq type echoes whole frame               */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  KHE_DRS_SEQ_NONE,				/* none of what follows      */
  KHE_DRS_SEQ_DAYS_POSITIVE,			/* all days, all positive    */
  KHE_DRS_SEQ_DAYS_NEGATIVE,			/* all days, all negative    */
  KHE_DRS_SEQ_SINGLE_POSITIVE			/* single times, all days    */
} KHE_DRS_SEQ_TYPE;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_INT_SEQ_COST                                           */
/*                                                                           */
/*  Combines INT_SEQ, INT_DEV (without allow_zero), and COST into a single   */
/*  expression, for the same optimizations as KHE_DRS_EXPR_INT_SUM_COST.     */
/*                                                                           */
/*  The first_open_child_index contains the index in open_children of the    */
/*  first open child.  This will usually be 0, but during closing the        */
/*  children are closed in increasing index order and this field increases.  */
/*                                                                           */
/*  Field e->closed_seqs contains the closed sequences of e.  When e is      */
/*  closed there is just one of these, containing all the children of e.     */
/*  When e is open there is one for each gap between adjacent open children  */
/*  of e, plus one for before the first open child and one for after the     */
/*  last open child.  This is one more closed sequence than the number of    */
/*  open children.  History children are not included in closed sequences.   */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expr_int_seq_cost_rec {
  INHERIT_KHE_DRS_EXPR_COST
  KHE_DRS_MONITOR_INFO		monitor_info;		/* monitor (debug)   */
  KHE_DRS_SEQ_TYPE		seq_type;		/* echoes frame      */
  int				seq_index;		/* if seq_type       */
  int				min_limit;		/* minimum limit     */
  int				max_limit;		/* maximum limit     */
  KHE_DRS_ADJUST_TYPE		adjust_type;		/* type of adjustment*/
  int				history_before;		/* history value befo*/
  int				history_after;		/* history value afte*/
  int				history;		/* history value     */
  KHE_DRS_DIM5_TABLE		dom_table;		/* dominance tests   */
  int				first_open_child_index;	/* temporary         */
  ARRAY_KHE_DRS_CLOSED_SEQ	closed_seqs;		/* closed sequences  */
} *KHE_DRS_EXPR_INT_SEQ_COST;

typedef HA_ARRAY(KHE_DRS_EXPR_INT_SEQ_COST) ARRAY_KHE_DRS_EXPR_INT_SEQ_COST;


/*****************************************************************************/
/*                                                                           */
/*  Submodule "type definitions - assignments"                               */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_ASST_TO_TASK - assignment of resource to task               */
/*                                                                           */
/*  NB this is not a pointer type.                                           */
/*                                                                           */
/*  This is not the same as asst->fixed_task_on_day; that represents an      */
/*  assignment that is fixed from the start.  This is an assignment that     */
/*  initially may be free but subsequently gets fixed.                       */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_asst_to_task_rec {
  KHE_DRS_ASST_TO_TASK_CLASS	asst;			/* the assignment */
  KHE_DRS_TASK_ON_DAY		fixed_dtd;		/* what fixed to  */
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_ASST_TO_TASK_SET - a set of assignment to task objects      */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_asst_to_task_set_rec {
  ARRAY_KHE_DRS_ASST_TO_TASK	assts_to_tasks;
};

typedef HA_ARRAY(KHE_DRS_ASST_TO_TASK_SET) ARRAY_KHE_DRS_ASST_TO_TASK_SET;

#define KheDrsAsstToTaskSetForEach(dats, dat, i)			\
  HaArrayForEach(dats->assts_to_tasks, dat, i)


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_ASST_TO_TASK_CLASS - assignment of resource to task class   */
/*                                                                           */
/*  An object of this type is always non-NULL.  It represents one            */
/*  assignment of a given resource on day to a given task class or task      */
/*  on that day.  The asst_to_shift field is always non-NULL, and says       */
/*  which resource is being assigned by this assignment, and what the        */
/*  resource monitor signature is for that assignment.                       */
/*                                                                           */
/*  If task_class != NULL, then the assignment is free to be to any          */
/*  task of that task class.  Someone will have to decide which of the       */
/*  task class's tasks to use before the assignment can actually be made.    */
/*  In this case fixed_task_on_day is not used.  Its value will be NULL.     */
/*                                                                           */
/*  Otherwise, task_class == NULL.  The decision about which task to         */
/*  use has already been made, and fixed_task_on_day holds that decision.    */
/*  It could be NULL, in which case the decision is to assign a free day.    */
/*                                                                           */
/*  The skip_assts array holds assignments that can be skipped (owing to     */
/*  two-extra selection) if this assignment is used.  A non-zero skip_count  */
/*  value indicates that this assignment may be skipped.                     */
/*                                                                           */
/*  These next paragraphs are obsolete:                                      */
/*  The cost and signature are for the resource's resource constraints       */
/*  only.  They depend on the previous solution, evidently.                  */
/*                                                                           */
/*  The resource on day is given.  The task on day is deducible as follows:  */
/*                                                                           */
/*    Case 1: planned_task_class != NULL.                                    */
/*      In this case, planned_task_on_day is irrelevant and will be NULL.    */
/*      The task on day is obtained from planned_task_class.                 */
/*                                                                           */
/*    Case 2: planned_task_class == NULL && planned_task_on_day != NULL.     */
/*      In this case, the task on day is just planned_task_on_day.           */
/*                                                                           */
/*    Case 3: planned_task_class == NULL && planned_task_on_day == NULL.     */
/*      In this case, the task on day is NULL (a free day).                  */
/*                                                                           */
/*  After the choice is actually made, task_on_day holds that choice.  It    */
/*  changes as the solve proceeds.                                           */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_asst_to_task_class_rec {
  KHE_DRS_ASST_TO_SHIFT			asst_to_shift;
  KHE_DRS_TASK_CLASS			task_class;
  KHE_DRS_TASK_ON_DAY			fixed_task_on_day;
  ARRAY_KHE_DRS_ASST_TO_TASK_CLASS	skip_assts;
  int					skip_count;
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_ASST_TO_SHIFT - assignment of a resource to a shift         */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_asst_to_shift_rec {
  KHE_DRS_RESOURCE_ON_DAY		resource_on_day;
  KHE_DRS_SIGNATURE			sig;
  bool					used;
};


/*****************************************************************************/
/*                                                                           */
/*  Submodule "type definitions - solutions"                                 */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_ASST_OP - an assignment operation                                */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  KHE_DRS_ASST_OP_UNASSIGN,
  KHE_DRS_ASST_OP_ASSIGN,
  KHE_DRS_ASST_OP_REPLACE
} KHE_DRS_ASST_OP;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_SOLN - a partial solution, part of the search tree          */
/*                                                                           */
/*  All valid values are non-NULL.  In the root solution the prev_soln and   */
/*  day fields are NULL, and prev_tasks is empty.                            */
/*                                                                           */
/*  Field priqueue_index of solution S is a back pointer into the priority   */
/*  queue, when that is in use.  Actually its meaning is more complicated:   */
/*                                                                           */
/*    * If S->priqueue_index == -1, S is not in the priority queue.  It has  */
/*      been expanded, which means that it cannot be freed until the end of  */
/*      the solve, since that would invalidate its descendants.              */
/*                                                                           */
/*    * If S->priqueue_index == 0, S is not in the priority queue, either    */
/*      because there is no priority queue, or because it has not been       */
/*      inserted yet, or because it was previously inserted but then it      */
/*      was deleted, owing to being found to be dominated by some other      */
/*      solution.  It has not been expanded, so it can be freed safely.      */
/*                                                                           */
/*  These first two cases apply whether there is a priority queue or not.    */
/*  When there is no priority queue the condition they record does not       */
/*  matter, but it is simpler to record it anyway.                           */
/*                                                                           */
/*    * If priqueue_index >= 1, this solution is currently in the priority   */
/*      queue, at position priqueue_index.  It has not been expanded.        */
/*                                                                           */
/*  Ordinarily, when a solution S lying in a solution set is found to be     */
/*  dominated by some other solution, it is removed from the solution set    */
/*  and from the priority queue and freed.  However, this is not done when   */
/*  S is expanded (when S->priqueue_index == -1).  Instead, S is left as     */
/*  is, just as though it was not dominated at all.  This has two benefits.  */
/*  First, every solution can be sure that its ancestors have not been       */
/*  freed; if any had been freed, that would make the prev_soln fields of    */
/*  solutions invalid.  Second, every unfreed solution lies in some          */
/*  solution set, so it will eventually be freed (at the end of the solve).  */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_soln_rec {
  struct khe_drs_signature_set_rec sig_set;	/* expanded to save space    */
  KHE_DRS_SOLN			prev_soln;	/* edge from prev_soln       */
  ARRAY_KHE_DRS_TASK_ON_DAY	prev_tasks;	/* indexed by open resources */
  int				priqueue_index;	/* priqueue index, if any    */
#if TESTING
  int				sorted_rank;	/* rank when solns sorted    */
#endif
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_PACKED_SOLN_DAY - one day of a packed solution              */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_packed_soln_day_rec {
  KHE_DRS_DAY			day;			/* day of this obj   */
  ARRAY_KHE_DRS_TASK_ON_DAY	prev_tasks;		/* tasks             */
} *KHE_DRS_PACKED_SOLN_DAY;

typedef HA_ARRAY(KHE_DRS_PACKED_SOLN_DAY) ARRAY_KHE_DRS_PACKED_SOLN_DAY;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_PACKED_SOLN - a packed solution object                      */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_packed_soln_rec {
  KHE_COST			cost;			/* soln cost         */
  ARRAY_KHE_DRS_PACKED_SOLN_DAY	days;			/* days of rerun     */
} *KHE_DRS_PACKED_SOLN;

typedef HA_ARRAY(KHE_DRS_PACKED_SOLN) ARRAY_KHE_DRS_PACKED_SOLN;


/*****************************************************************************/
/*                                                                           */
/*  Submodule "type definitions - expansion"                                 */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPANDER - global values that vary during expansion         */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expander_rec *KHE_DRS_EXPANDER;
typedef HA_ARRAY(KHE_DRS_EXPANDER) ARRAY_KHE_DRS_EXPANDER;

struct khe_drs_expander_rec {
  KHE_DYNAMIC_RESOURCE_SOLVER	solver;
  ARRAY_KHE_DRS_ASST_TO_TASK	assts_to_tasks;
  ARRAY_KHE_DRS_ASST_TO_TASK	tmp_assts_to_tasks;
  bool				whole_tasks;
  bool				open;
  KHE_COST			cost;
  KHE_COST			cost_limit;
  int				free_resource_count;
  int				must_assign_count;
  HA_ARRAY_INT			marks;
};


/*****************************************************************************/
/*                                                                           */
/*  Submodule "type definitions - sets of solutions"                         */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_SOLN_LIST - a list of solns; has several uses               */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_soln_list_rec {
  KHE_DYNAMIC_RESOURCE_SOLVER	solver;
  ARRAY_KHE_DRS_SOLN		solns;
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_SOLN_TRIE - a trie of solns                                 */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_soln_trie_rec *KHE_DRS_SOLN_TRIE;
typedef HA_ARRAY(KHE_DRS_SOLN_TRIE) ARRAY_KHE_DRS_SOLN_TRIE;

struct khe_drs_soln_trie_rec {
  KHE_DRS_SOLN			soln;
  ARRAY_KHE_DRS_SOLN_TRIE	children;
  int				base;		/* first index of children */
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_INDEXED_SOLN_SET - an array of soln lists, indexed by cost  */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_indexed_soln_set_rec {
  KHE_COST			base;
  KHE_COST			increment;
  ARRAY_KHE_DRS_SOLN_LIST	soln_lists;
  int				count;
} *KHE_DRS_INDEXED_SOLN_SET;

typedef HA_ARRAY(KHE_DRS_INDEXED_SOLN_SET) ARRAY_KHE_DRS_INDEXED_SOLN_SET;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_SOLN_SET_PART - one part of a solution set                  */
/*                                                                           */
/*****************************************************************************/

#define INHERIT_KHE_DRS_SOLN_SET_PART					\
  KHE_DRS_DOM_KIND		dom_kind;		/* kind of dominance */

typedef struct khe_drs_soln_set_part_rec {
  INHERIT_KHE_DRS_SOLN_SET_PART
} *KHE_DRS_SOLN_SET_PART;

typedef struct khe_drs_soln_set_part_dom_none_rec {
  INHERIT_KHE_DRS_SOLN_SET_PART
  KHE_DRS_SOLN_LIST		soln_list;
} *KHE_DRS_SOLN_SET_PART_DOM_NONE;

typedef HA_ARRAY(KHE_DRS_SOLN_SET_PART_DOM_NONE)
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_NONE;

typedef struct khe_drs_soln_set_part_dom_weak_rec {
  INHERIT_KHE_DRS_SOLN_SET_PART
  TABLE_KHE_DRS_SOLN		soln_table;
} *KHE_DRS_SOLN_SET_PART_DOM_WEAK;

typedef HA_ARRAY(KHE_DRS_SOLN_SET_PART_DOM_WEAK)
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_WEAK;

typedef struct khe_drs_soln_set_part_dom_medium_rec {
  INHERIT_KHE_DRS_SOLN_SET_PART
  TABLE_KHE_DRS_SOLN_LIST	soln_list_table;
} *KHE_DRS_SOLN_SET_PART_DOM_MEDIUM;

typedef HA_ARRAY(KHE_DRS_SOLN_SET_PART_DOM_MEDIUM)
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_MEDIUM;

typedef struct khe_drs_soln_set_part_dom_strong_rec {
  INHERIT_KHE_DRS_SOLN_SET_PART
  KHE_DRS_SOLN_LIST		soln_list;
} *KHE_DRS_SOLN_SET_PART_DOM_STRONG;

typedef HA_ARRAY(KHE_DRS_SOLN_SET_PART_DOM_STRONG)
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_STRONG;

typedef struct khe_drs_soln_set_part_dom_trie_rec {
  INHERIT_KHE_DRS_SOLN_SET_PART
  KHE_DRS_SOLN_TRIE		soln_trie;
} *KHE_DRS_SOLN_SET_PART_DOM_TRIE;

typedef HA_ARRAY(KHE_DRS_SOLN_SET_PART_DOM_TRIE)
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_TRIE;

typedef struct khe_drs_soln_set_part_dom_indexed_rec {
  INHERIT_KHE_DRS_SOLN_SET_PART
  KHE_DRS_INDEXED_SOLN_SET	indexed_solns;
} *KHE_DRS_SOLN_SET_PART_DOM_INDEXED;

typedef HA_ARRAY(KHE_DRS_SOLN_SET_PART_DOM_INDEXED)
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_INDEXED;

typedef struct khe_drs_soln_set_part_dom_uniform_rec {
  INHERIT_KHE_DRS_SOLN_SET_PART
  KHE_DRS_SOLN_LIST		soln_list;
} *KHE_DRS_SOLN_SET_PART_DOM_UNIFORM;

typedef HA_ARRAY(KHE_DRS_SOLN_SET_PART_DOM_UNIFORM)
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_UNIFORM;

typedef struct khe_drs_soln_set_part_dom_indexed_uniform_rec {
  INHERIT_KHE_DRS_SOLN_SET_PART
  KHE_DRS_INDEXED_SOLN_SET	indexed_solns;
} *KHE_DRS_SOLN_SET_PART_DOM_INDEXED_UNIFORM;

typedef HA_ARRAY(KHE_DRS_SOLN_SET_PART_DOM_INDEXED_UNIFORM)
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_INDEXED_UNIFORM;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_SOLN_SET - a set of solns, with a choice of implementations */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_soln_set_rec {
  KHE_DRS_SOLN_SET_PART		cache;			/* cache (optional)  */
  KHE_DRS_SOLN_SET_PART		main;			/* main part         */
};


/*****************************************************************************/
/*                                                                           */
/*  Submodule "type definitions - solving"                                   */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DYNAMIC_RESOURCE_SOLVER - a dynamic resource solver             */
/*                                                                           */
/*****************************************************************************/

struct khe_dynamic_resource_solver_rec {

  /* fields which are constant throughout the lifetime of the solver */
  HA_ARENA			arena;
  KHE_SOLN			soln;
  KHE_RESOURCE_TYPE		resource_type;
  KHE_OPTIONS			options;
  KHE_FRAME			days_frame;
  KHE_EVENT_TIMETABLE_MONITOR	etm;
  KHE_TASK_CLASS_SOLVER		task_class_solver;
  ARRAY_KHE_DRS_RESOURCE	all_resources;
  ARRAY_KHE_DRS_DAY		all_days;
  ARRAY_KHE_DRS_TASK		all_root_tasks;
  ARRAY_KHE_DRS_MONITOR_INFO	all_monitor_infos;
  int				postorder_count;
  KHE_COST			increment;		/* gcd of weights */

  /* free list fields (and one used list) */
  ARRAY_KHE_DRS_RESOURCE_SET		   resource_set_free_list;
  ARRAY_KHE_DRS_EXPANDER		   expander_free_list;
  ARRAY_KHE_DRS_SHIFT_ASST		   shift_asst_free_list;
  ARRAY_KHE_DRS_SHIFT_ASST_TRIE		   shift_asst_trie_free_list;
  ARRAY_KHE_DRS_CLOSED_SEQ		   closed_seq_free_list;
  ARRAY_KHE_DRS_DIM1_TABLE		   table1_free_list;
  ARRAY_KHE_DRS_DOM_TEST		   dom_test_free_list;
  ARRAY_KHE_DRS_SIGNATURE		   signature_free_list;
  ARRAY_KHE_DRS_SIGNER			   signer_free_list;
  ARRAY_KHE_DRS_ASST_TO_TASK_SET	   asst_to_task_set_free_list;
  ARRAY_KHE_DRS_ASST_TO_TASK_CLASS	   asst_to_task_class_free_list;
  ARRAY_KHE_DRS_ASST_TO_SHIFT		   asst_to_shift_free_list;
  ARRAY_KHE_DRS_SOLN			   soln_free_list;
  ARRAY_KHE_DRS_SOLN_LIST		   soln_list_free_list;
  ARRAY_KHE_DRS_SOLN_TRIE		   soln_trie_free_list;
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_NONE	   soln_set_part_dom_none_free_list;
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_WEAK	   soln_set_part_dom_weak_free_list;
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_MEDIUM   soln_set_part_dom_medium_free_list;
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_STRONG   soln_set_part_dom_strong_free_list;
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_TRIE	   soln_set_part_dom_trie_free_list;
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_INDEXED  soln_set_part_dom_indexed_free_list;
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_UNIFORM  soln_set_part_dom_uniform_free_list;
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_INDEXED_UNIFORM
				   soln_set_part_dom_indexed_uniform_free_list;
  ARRAY_KHE_DRS_INDEXED_SOLN_SET	   indexed_soln_set_free_list;
  ARRAY_KHE_DRS_SOLN_SET		   soln_set_free_list;
  ARRAY_KHE_DRS_PACKED_SOLN_DAY		   packed_soln_day_free_list;
  ARRAY_KHE_DRS_PACKED_SOLN		   packed_soln_free_list;

  /* priority queue (always initialized, but actual use is optional) */
  KHE_PRIQUEUE			priqueue;

  /* fields that vary with the solve */
  ARRAY_KHE_DRS_DAY_RANGE	selected_day_ranges;	/* chosen for solve  */
  KHE_RESOURCE_SET		selected_resource_set;	/* chosen for solve  */
  bool				solve_priqueue;		/* use priqueue      */
  bool				solve_extra_selection;	/* use int. d. */
  bool				solve_expand_by_shifts;	/* expand by shifts  */
  bool				solve_correlated_exprs;
  int				solve_daily_expand_limit;
  int				solve_daily_prune_trigger;
  int				solve_resource_expand_limit;
  int				solve_dom_approx;
  KHE_DRS_DOM_KIND		solve_main_dom_kind;
  bool				solve_cache;
  KHE_DRS_DOM_KIND		solve_cache_dom_kind;
  KHE_DRS_DOM_TEST_TYPE		solve_dom_test_type;	/* from main_dom_kind*/
  KHE_COST			solve_init_cost;	/* before solving    */
  KHE_COST			solve_start_cost;	/* before first day  */
  ARRAY_KHE_DRS_DAY		open_days;
  ARRAY_KHE_DRS_SHIFT		open_shifts;
  KHE_DRS_RESOURCE_SET		open_resources;
  ARRAY_KHE_DRS_EXPR		open_exprs;
  KHE_DRS_PACKED_SOLN		rerun;			/* non-NULL if rerun */

  /* fields for recording running time and other statistics */
#if TESTING
  KHE_TIMER			timer;
  HA_ARRAY_INT			solns_made_per_day;
  HA_ARRAY_INT			table_size_per_day;
  HA_ARRAY_FLOAT		running_time_per_day;
  HA_ARRAY_INT			ancestor_freq;
  HA_ARRAY_INT			dominates_freq;
  int				max_open_day_index;
#endif
};


/*****************************************************************************/
/*                                                                           */
/*  Submodule " miscellaneous functions"                                     */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  int min(int a, int b)                                                    */
/*                                                                           */
/*  Return the smaller of a and b, or either if they are equal.              */
/*                                                                           */
/*****************************************************************************/

static int min(int a, int b)
{
  return a < b ? a : b;
}


/*****************************************************************************/
/*                                                                           */
/*  int max(int a, int b)                                                    */
/*                                                                           */
/*  Return the larger of a and b, or either if they are equal.               */
/*                                                                           */
/*****************************************************************************/

static int max(int a, int b)
{
  return a > b ? a : b;
}


/*****************************************************************************/
/*                                                                           */
/*  Major category "times"                                                   */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_DAY_RANGE"                                            */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DAY_RANGE KheDrsDayRangeMake(int first, int last)                */
/*                                                                           */
/*  Return a new interval with these endpoints.                              */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DAY_RANGE KheDrsDayRangeMake(int first, int last)
{
  KHE_DRS_DAY_RANGE res;
  res.first = first;
  res.last = last;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDayRangeAdd(KHE_DRS_DAY_RANGE *ddr, int index)                */
/*                                                                           */
/*  Update *ddr to include index.                                            */
/*                                                                           */
/*  Implementation note.  Both cases are needed here, as proved by a trace   */
/*  of Add([1, 0], 4).  The second case wrongly produces [1, 4].             */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDayRangeAdd(KHE_DRS_DAY_RANGE *ddr, int index)
{
  if( ddr->first > ddr->last )
  {
    /* ddr is empty, so make it cover just index */
    ddr->first = ddr->last = index;
  }
  else
  {
    /* ddr is non-empty, so expand it to include index */
    if( index < ddr->first )
      ddr->first = index;
    if( index > ddr->last )
      ddr->last = index;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDayRangeMerge(ARRAY_KHE_DRS_DAY_RANGE *day_ranges,            */
/*    KHE_DRS_DAY_RANGE ddr)                                                 */
/*                                                                           */
/*  Merge ddr into *day_ranges, a sorted array of day ranges.                */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDayRangeMerge(ARRAY_KHE_DRS_DAY_RANGE *day_ranges,
  KHE_DRS_DAY_RANGE ddr)
{
  KHE_DRS_DAY_RANGE ddr2;  int i;

  /* if ddr is empty, it changes nothing (this can't happen, actually) */
  if( ddr.first > ddr.last )
    return;

  HaArrayForEach(*day_ranges, ddr2, i)
  {
    /***********************************************************************/
    /*                                                                     */
    /*  Invariant: day_ranges and ddr may have changed, but it remains     */
    /*  true that merging ddr with day_ranges gives the desired result.    */
    /*  Also, day_ranges[0..i-1] precede and do not interact with ddr.     */
    /*                                                                     */
    /***********************************************************************/

    if( ddr2.last + 1 < ddr.first )
    {
      /* ddr2 precedes ddr and does not interact with it; continue */
      continue;
    }

    if( ddr.last + 1 < ddr2.first )
    {
      /* ddr precedes ddr2 and does not interact with it, or with day_ranges */
      HaArrayAdd(*day_ranges, i, ddr);
      return;
    }

    /* ddr2 interacts with ddr, so expand ddr to include ddr2, then */
    /* delete ddr2 and continue */
    if( ddr2.first < ddr.first )
      ddr.first = ddr2.first;
    if( ddr2.last > ddr.last )
      ddr.last = ddr2.last;
    HaArrayDeleteAndShift(*day_ranges, i);
    i--;
  }

  /* ddr does not interact with any day_ranges, and it follows them all */
  HaArrayAddLast(*day_ranges, ddr);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsDayRangeSubset(KHE_DRS_DAY_RANGE ddr1, KHE_DRS_DAY_RANGE ddr2)*/
/*                                                                           */
/*  Return true if ddr1 is a subset of ddr2.                                 */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsDayRangeSubset(KHE_DRS_DAY_RANGE ddr1, KHE_DRS_DAY_RANGE ddr2)
{
  return ddr1.first >= ddr2.first && ddr1.last <= ddr2.last;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDayRangeDebug(KHE_DRS_DAY_RANGE ddr,                          */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)  */
/*                                                                           */
/*  Debug print of ddr onto fp with the given verbosity and indent.          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDayRangeDebug(KHE_DRS_DAY_RANGE ddr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)
{
  KHE_TIME_GROUP tg1, tg2;
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  tg1 = KheFrameTimeGroup(drs->days_frame, ddr.first);
  tg2 = KheFrameTimeGroup(drs->days_frame, ddr.last);
  if( tg1 == tg2 )
    fprintf(fp, "%s", KheTimeGroupId(tg1));
  else
    fprintf(fp, "%s-%s", KheTimeGroupId(tg1), KheTimeGroupId(tg2));
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_DAY"                                                  */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DAY KheDrsDayMake(int frame_index,                               */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new day object with the given attributes.  Its task classes,      */
/*  expressions, and dominance tests will be added later.                    */
/*                                                                           */
/*  Here frame_index is the index of the new day in the common frame.        */
/*                                                                           */
/*****************************************************************************/
static KHE_DRS_SIGNER_SET KheDrsSignerSetMake(KHE_DRS_DAY day,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);

static KHE_DRS_DAY KheDrsDayMake(int frame_index,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DAY res;
  HaMake(res, drs->arena);
  res->frame_index = frame_index;
  res->open_day_index = -1;  /* signals not currently open for solving */
  res->time_group = KheFrameTimeGroup(drs->days_frame, frame_index);
  HaArrayInit(res->shifts, drs->arena);
  res->signer_set = KheDrsSignerSetMake(res, drs);
  res->soln_set = NULL;   /* initialized at the start of each solve */
  res->soln_list = NULL;  /* initialized at the start of each solve */
  res->soln_made_count = 0;
  res->solve_expand_count = 0;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheDrsDayId(KHE_DRS_DAY day)                                       */
/*                                                                           */
/*  Return the Id of day.  This is just the Id of its time group.  A NULL    */
/*  day is allowed and "-" is returned in that case.                         */
/*                                                                           */
/*****************************************************************************/

static char *KheDrsDayId(KHE_DRS_DAY day)
{
  return day == NULL ? "-" : KheTimeGroupId(day->time_group);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDayAddTaskClass(KHE_DRS_DAY day, KHE_DRS_TASK_CLASS dtc,      */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Add dtc to the day's task classes and shifts.                            */
/*                                                                           */
/*****************************************************************************/
static KHE_DRS_SHIFT KheDrsShiftMake(KHE_DRS_DAY day, KHE_DRS_TASK_CLASS dtc,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);
static bool KheDrsShiftAcceptsClass(KHE_DRS_SHIFT ds, KHE_DRS_TASK_CLASS dtc);

static void KheDrsDayAddTaskClass(KHE_DRS_DAY day, KHE_DRS_TASK_CLASS dtc,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SHIFT ds;  int i;

  /* add dtc to day->shifts */
  HaArrayForEach(day->shifts, ds, i)
    if( KheDrsShiftAcceptsClass(ds, dtc) )
    {
      dtc->encl_shift = ds;
      return;
    }

  /* need a new shift */
  ds = KheDrsShiftMake(day, dtc, drs);
  dtc->encl_shift = ds;
  HaArrayAddLast(day->shifts, ds);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDayOpen(KHE_DRS_DAY day, KHE_DRS_DAY_RANGE ddr,               */
/*    int open_day_index, KHE_DRS_DOM_KIND dom_kind,                         */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Open day for solving.  It lies within day range ddr, its index in the    */
/*  the sequence of all open days is open_day_index, and dom_kind is the     */
/*  kind of dominance testing to use on this day.                            */
/*                                                                           */
/*  The signer is clear initially and cleared each time the day is closed,   */
/*  so there is no need to clear it here, and we don't.                      */
/*                                                                           */
/*  NB We open the day's task classes separately, because they may contain   */
/*  tasks for later days and opening assumes that those days are open.       */
/*                                                                           */
/*  NBB The assignment "day->solve_prune_cost = drs->solve_init_cost" is     */
/*  safe here because KheDrsDayOpen is called after drs->solve_init_cost     */
/*  is initialized.  (This comment is obsolete.)                             */
/*                                                                           */
/*****************************************************************************/
static void KheDrsTaskClassOpen(KHE_DRS_TASK_CLASS dtc, KHE_DRS_DAY_RANGE ddr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);
static KHE_DRS_SOLN_SET KheDrsSolnSetMake(KHE_DRS_DOM_KIND main_dom_kind,
  bool cache, KHE_DRS_DOM_KIND cache_dom_kind, KHE_DYNAMIC_RESOURCE_SOLVER drs);
static void KheDrsSignerSetOpen(KHE_DRS_SIGNER_SET signer_set,
  KHE_DRS_DAY day, KHE_DYNAMIC_RESOURCE_SOLVER drs);
static void KheDrsSignerSetDebug(KHE_DRS_SIGNER_SET signer_set,
  int verbosity, int indent, FILE *fp);

static void KheDrsDayOpen(KHE_DRS_DAY day, KHE_DRS_DAY_RANGE ddr,
  int open_day_index, KHE_DRS_DOM_KIND main_dom_kind, bool cache,
  KHE_DRS_DOM_KIND cache_dom_kind, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  if( DEBUG18 )
    fprintf(stderr, "[ KheDrsDayOpen(%s, %d-%d, %d)\n", KheDrsDayId(day),
      ddr.first, ddr.last, open_day_index);
  day->open_day_index = open_day_index;
  KheDrsSignerSetOpen(day->signer_set, day, drs);
  day->soln_set = KheDrsSolnSetMake(main_dom_kind, cache, cache_dom_kind, drs);
  day->soln_made_count = 0;
  day->solve_expand_count = 0;
  if( DEBUG18 )
    fprintf(stderr, "] KheDrsDayOpen\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDayOpenShifts(KHE_DRS_DAY day, KHE_DRS_DAY_RANGE ddr,         */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Open the shifts of day, including opening their task classes.            */
/*                                                                           */
/*****************************************************************************/
static void KheDrsShiftOpen(KHE_DRS_SHIFT ds, KHE_DRS_DAY_RANGE ddr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsDayOpenShifts(KHE_DRS_DAY day, KHE_DRS_DAY_RANGE ddr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SHIFT shift;  int i;
  if( DEBUG18 )
    fprintf(stderr, "[ KheDrsDayOpenShifts(%s, %d-%d)\n",
      KheDrsDayId(day), ddr.first, ddr.last);
  HaArrayForEach(day->shifts, shift, i)
    KheDrsShiftOpen(shift, ddr, drs);
  if( DEBUG18 )
    fprintf(stderr, "] KheDrsDayOpenShifts\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDayAddOpenExpr(KHE_DRS_DAY day, KHE_DRS_EXPR e)               */
/*                                                                           */
/*  Add open expression e to day.                                            */
/*                                                                           */
/*****************************************************************************/
static void KheDrsSignerAddOpenExpr(KHE_DRS_SIGNER dsg, KHE_DRS_EXPR e);

static void KheDrsDayAddOpenExpr(KHE_DRS_DAY day, KHE_DRS_EXPR e)
{
  KHE_DRS_SIGNER dsg;
  HnAssert(e->resource == NULL, "KheDrsDayAddOpenExpr internal error");
  HnAssert(HaArrayCount(day->signer_set->signers) > 0,
    "KheDrsDayAddOpenExpr internal error");
  dsg = HaArrayLast(day->signer_set->signers);
  KheDrsSignerAddOpenExpr(dsg, e);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsDayAddDomTest(KHE_DRS_DAY day, KHE_DRS_DOM_TEST dom_test)      */
/*                                                                           */
/*  Add dom_test to day and return its index in day->dom_tests.              */
/*                                                                           */
/*****************************************************************************/
static int KheDrsSignerAddDomTest(KHE_DRS_SIGNER dsg,
  KHE_DRS_DOM_TEST dom_test, KHE_DYNAMIC_RESOURCE_SOLVER drs);

static int KheDrsDayAddDomTest(KHE_DRS_DAY day, KHE_DRS_DOM_TEST dom_test,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SIGNER dsg;
  HnAssert(HaArrayCount(day->signer_set->signers) > 0,
    "KheDrsDayAddOpenExpr internal error");
  dsg = HaArrayLast(day->signer_set->signers);
  return KheDrsSignerAddDomTest(dsg, dom_test, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDayClose(KHE_DRS_DAY day, KHE_DYNAMIC_RESOURCE_SOLVER drs)    */
/*                                                                           */
/*  Close day.                                                               */
/*                                                                           */
/*****************************************************************************/
static void KheDrsShiftClose(KHE_DRS_SHIFT ds, KHE_DYNAMIC_RESOURCE_SOLVER drs);
static void KheDrsSolnListFreeSolns(KHE_DRS_SOLN_LIST soln_list,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);
static void KheDrsSolnListFree(KHE_DRS_SOLN_LIST soln_list,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);
static KHE_DRS_SOLN_LIST KheDrsDayGatherSolns(KHE_DRS_DAY next_day,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);
static void KheDrsSignerSetClose(KHE_DRS_SIGNER_SET signer_set,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsDayClose(KHE_DRS_DAY day, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int i;  KHE_DRS_SHIFT shift;
  day->open_day_index = -1;  /* signals no longer open */
  HaArrayForEach(day->shifts, shift, i)
    KheDrsShiftClose(shift, drs);
  KheDrsSignerSetClose(day->signer_set, drs);

  /* delete solution sets, solution lists, and solutions as required */
  if( day->soln_list != NULL )
  {
    KheDrsSolnListFreeSolns(day->soln_list, drs);
    KheDrsSolnListFree(day->soln_list, drs);
    day->soln_list = NULL;
  }
  if( day->soln_set != NULL )
  {
    KheDrsDayGatherSolns(day, drs);
    KheDrsSolnListFreeSolns(day->soln_list, drs);
    KheDrsSolnListFree(day->soln_list, drs);
    day->soln_list = NULL;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDayDebug(KHE_DRS_DAY day, KHE_DYNAMIC_RESOURCE_SOLVER drs,    */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of day onto fp.                                              */
/*                                                                           */
/*****************************************************************************/
static void KheDrsShiftDebug(KHE_DRS_SHIFT ds,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp);
static void KheDrsTaskClassDebug(KHE_DRS_TASK_CLASS dtc,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp);
static void KheDrsSolnSetDebug(KHE_DRS_SOLN_SET dns,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp);

static void KheDrsDayDebug(KHE_DRS_DAY day, KHE_DYNAMIC_RESOURCE_SOLVER drs,
  int verbosity, int indent, FILE *fp)
{
  int i;  KHE_DRS_SHIFT ds;
  if( indent >= 0 )
  {
    fprintf(fp, "%*s[ Day %s (frame %d, ", indent, "",
      KheDrsDayId(day), day->frame_index);
    if( day->open_day_index >= 0 )
      fprintf(fp, "open %d)\n", day->open_day_index);
    else
      fprintf(fp, "closed)\n");
    if( verbosity >= 2 )
    {
      HaArrayForEach(day->shifts, ds, i)
	KheDrsShiftDebug(ds, drs, verbosity, indent + 2, fp);
    }
    if( verbosity >= 3 && day->soln_set != NULL )
      KheDrsSolnSetDebug(day->soln_set, drs, verbosity, indent + 2, fp);
    fprintf(fp, "%*s]\n", indent, "");
  }
  else
    fprintf(fp, "%s", KheDrsDayId(day));
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_DAY - expansion"                                      */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDayExpandBegin(KHE_DRS_DAY next_day, KHE_DRS_SOLN prev_soln,  */
/*    KHE_DRS_EXPANDER de, int *must_assign_count)                           */
/*                                                                           */
/*  Begin the expansion of prev_soln into next_day.                          */
/*                                                                           */
/*  Implementation note - setting ds->expand_max_included_free_resource_count*/
/*  -----------------------------------------------------------------        */
/*  This field contains the maximum number of free resources that can be     */
/*  assigned to ds without making it impossible to fulfil the minimum        */
/*  requirements of other shifts.  First we calculate excess, the number of  */
/*  free resources that are not already committed to minimum requirements.   */
/*  Then ds->expand_max_included_free_resource_count is this number added to */
/*  the minimum for ds.  This can never exceed the number of free resources. */
/*                                                                           */
/*****************************************************************************/
static void KheDrsSolnSetBeginCacheSegment(KHE_DRS_SOLN_SET soln_set,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);
static void KheDrsShiftExpandBegin(KHE_DRS_SHIFT ds, KHE_DRS_SOLN prev_soln,
  KHE_DRS_EXPANDER de);
static int KheDrsExpanderExcessFreeResourceCount(KHE_DRS_EXPANDER de);

static void KheDrsDayExpandBegin(KHE_DRS_DAY next_day, KHE_DRS_SOLN prev_soln,
  KHE_DRS_EXPANDER de)
{
  int i, excess;  KHE_DRS_SHIFT ds;

  /* begin caching expansions */
  KheDrsSolnSetBeginCacheSegment(next_day->soln_set, de->solver);

  /* begin expansion in each shift */
  HaArrayForEach(next_day->shifts, ds, i)
    KheDrsShiftExpandBegin(ds, prev_soln, de);

  /* set expand_max_included_free_resource_count in each shift */
  /* excess = de ->free_resource_count - de ->must_assign_count; */
  excess = KheDrsExpanderExcessFreeResourceCount(de);
  HaArrayForEach(next_day->shifts, ds, i)
    ds->expand_max_included_free_resource_count =
      ds->expand_must_assign_count + excess;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDayExpandEnd(KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de)       */
/*                                                                           */
/*  End the previously begun expansion of prev_soln into next_day.           */
/*                                                                           */
/*****************************************************************************/
static void KheDrsShiftExpandEnd(KHE_DRS_SHIFT ds, KHE_DRS_EXPANDER de);
static void KheDrsSolnSetEndCacheSegment(KHE_DRS_SOLN_SET soln_set,
  KHE_DRS_DAY soln_day, KHE_DRS_EXPANDER de, KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsDayExpandEnd(KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de)
{
  int i;  KHE_DRS_SHIFT ds;

  /* end expansion in each shift */
  HaArrayForEach(next_day->shifts, ds, i)
    KheDrsShiftExpandEnd(ds, de);

  /* end caching expansions */
  KheDrsSolnSetEndCacheSegment(next_day->soln_set, next_day, de, de->solver);
}


/*****************************************************************************/
/*                                                                           */
/*  Major category "resources"                                               */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_RESOURCE_EXPAND_ROLE"                                 */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  char *KheDrsResourceExpandRoleShow(KHE_DRS_RESOURCE_EXPAND_ROLE role)    */
/*                                                                           */
/*  Return a readable string representation of role.                         */
/*                                                                           */
/*****************************************************************************/

static char *KheDrsResourceExpandRoleShow(KHE_DRS_RESOURCE_EXPAND_ROLE role)
{
  switch( role )
  {
    case KHE_DRS_RESOURCE_EXPAND_NO:	return "expand_no";
    case KHE_DRS_RESOURCE_EXPAND_FIXED:	return "expand_fixed";
    case KHE_DRS_RESOURCE_EXPAND_FREE:	return "expand_free";
    default:				return "?";
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_RESOURCE"                                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  char *KheDrsResourceId(KHE_DRS_RESOURCE dr)                              */
/*                                                                           */
/*  Return the Id of dr.                                                     */
/*                                                                           */
/*****************************************************************************/

static char *KheDrsResourceId(KHE_DRS_RESOURCE dr)
{
  return KheResourceId(dr->resource);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_RESOURCE KheDrsResourceMake(KHE_RESOURCE r,                      */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new resource object with these attributes.                        */
/*  Return NULL if unsuccessful.                                             */
/*                                                                           */
/*****************************************************************************/
static KHE_DRS_RESOURCE_ON_DAY KheDrsResourceOnDayMake(KHE_DRS_RESOURCE dr,
  KHE_DRS_DAY day, KHE_DYNAMIC_RESOURCE_SOLVER drs);
static KHE_DRS_DIM2_TABLE KheDrsDim2TableMake(char *debug_str, HA_ARENA a);

static KHE_DRS_RESOURCE KheDrsResourceMake(KHE_RESOURCE r,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_RESOURCE res;  int i;  KHE_DRS_RESOURCE_ON_DAY drd;  KHE_DRS_DAY day;

  /* initialize res, including one resource on day object for each day */
  if( DEBUG7 )
    fprintf(stderr, "[ KheDrsResourceMake(%s, drs)\n", KheResourceId(r));
  HaMake(res, drs->arena);
  res->resource = r;
  res->open_resource_index = -1;
  HaArrayInit(res->days, drs->arena);
  HaArrayForEach(drs->all_days, day, i)
  {
    drd = KheDrsResourceOnDayMake(res, day, drs);
    HaArrayAddLast(res->days, drd);
  }
  res->expand_role = KHE_DRS_RESOURCE_EXPAND_NO;
  HaArrayInit(res->expand_assts_to_shifts, drs->arena);
  HaArrayInit(res->expand_assts, drs->arena);
  res->expand_free_day_asst = NULL;
  res->expand_dom_test_cache = KheDrsDim2TableMake(KheDrsResourceId(res),
    drs->arena);
  if( DEBUG7 )
    fprintf(stderr, "] KheDrsResourceMake returning\n");
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_RESOURCE KheDrsResource(KHE_RESOURCE r,                          */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Assuming that r has the correct resource type, return its corresponding  */
/*  drs resource.                                                            */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_RESOURCE KheDrsResource(KHE_RESOURCE r,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  HnAssert(KheResourceResourceTypeIndex(r) < HaArrayCount(drs->all_resources),
    "KheDrsResource internal error");
  return HaArray(drs->all_resources, KheResourceResourceTypeIndex(r));
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_RESOURCE_ON_DAY KheDrsResourceOnDay(KHE_DRS_RESOURCE dr,         */
/*    KHE_DRS_DAY day)                                                       */
/*                                                                           */
/*  Return the resource on day object representing dr on day.                */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_RESOURCE_ON_DAY KheDrsResourceOnDay(KHE_DRS_RESOURCE dr,
  KHE_DRS_DAY day)
{
  return HaArray(dr->days, day->frame_index);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_RESOURCE_ON_DAY KheDrsResourceOnDayAtTime(KHE_DRS_RESOURCE dr,   */
/*    KHE_TIME time, KHE_DYNAMIC_RESOURCE_SOLVER drs)                        */
/*                                                                           */
/*  Return the resource on day object whose resource is dr and whose day     */
/*  is time's day.                                                           */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_RESOURCE_ON_DAY KheDrsResourceOnDayAtTime(KHE_DRS_RESOURCE dr,
  KHE_TIME time, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int day_index;
  day_index = KheFrameTimeIndex(drs->days_frame, time);
  return HaArray(dr->days, day_index);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceOpen(KHE_DRS_RESOURCE dr, int open_resource_index,    */
/*    KHE_DRS_PACKED_SOLN init_soln, KHE_DYNAMIC_RESOURCE_SOLVER drs)        */
/*                                                                           */
/*  Open dr.  This involves unassigning all the tasks assigned dr which      */
/*  lie in the selected day ranges of drs, and gathering for opening the     */
/*  expressions that depend on what dr is doing on those days, including     */
/*  on days where dr remains assigned.                                       */
/*                                                                           */
/*  Also, if init_soln != NULL, included the unassignments as assignments    */
/*  in init_soln, so that they can be undone later if required.              */
/*                                                                           */
/*  The signer is clear initially and cleared each time the resource is   */
/*  closed, so there is no need to clear it here, and we don't.              */
/*                                                                           */
/*****************************************************************************/
static void KheDrsResourceOnDayOpen(KHE_DRS_RESOURCE_ON_DAY drd,
  int open_resource_index, int open_day_index, KHE_DRS_DAY_RANGE ddr,
  KHE_DRS_PACKED_SOLN init_soln, KHE_DYNAMIC_RESOURCE_SOLVER drs);
static void KheDrsResourceDebug(KHE_DRS_RESOURCE dr, int verbosity,
  int indent, FILE *fp);

static void KheDrsResourceOpen(KHE_DRS_RESOURCE dr, int open_resource_index,
  KHE_DRS_PACKED_SOLN init_soln, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DAY_RANGE ddr;  int i, j, open_day_index; KHE_DRS_RESOURCE_ON_DAY drd;
  if( DEBUG11 )
  {
    fprintf(stderr, "[ KheDrsResourceOpen(%s, %d, drs)\n", KheDrsResourceId(dr),
      open_resource_index);
    KheDrsResourceDebug(dr, 2, 2, stderr);
  }
  dr->open_resource_index = open_resource_index;
  open_day_index = 0;
  HaArrayForEach(drs->selected_day_ranges, ddr, i)
    for( j = ddr.first;  j <= ddr.last;  j++ )
    {
      /* unassign any task assigned on drd, if it lies entirely in ddr */
      drd = HaArray(dr->days, j);
      KheDrsResourceOnDayOpen(drd, open_resource_index, open_day_index,
	ddr, init_soln, drs);

      /* increase open_day_index */
      open_day_index++;
    }
  if( DEBUG11 )
  {
    fprintf(stderr, "  at end of KheDrsResourceOpen(%s, %d, drs):\n",
      KheDrsResourceId(dr), open_resource_index);
    KheDrsResourceDebug(dr, 2, 2, stderr);
    fprintf(stderr, "] KheDrsResourceOpen\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceClose(KHE_DRS_RESOURCE dr)                            */
/*                                                                           */
/*  Close dr.                                                                */
/*                                                                           */
/*****************************************************************************/
static void KheDrsResourceOnDayClose(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsResourceClose(KHE_DRS_RESOURCE dr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DAY_RANGE ddr;  int i, j;  KHE_DRS_RESOURCE_ON_DAY drd;
  HaArrayForEach(drs->selected_day_ranges, ddr, i)
    for( j = ddr.first;  j <= ddr.last;  j++ )
    {
      drd = HaArray(dr->days, j);
      KheDrsResourceOnDayClose(drd, drs);
    }
  dr->open_resource_index = -1;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceDebug(KHE_DRS_RESOURCE dr, int verbosity,             */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of dr onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/
static void KheDrsResourceOnDayDebug(KHE_DRS_RESOURCE_ON_DAY drd,
  int verbosity, int indent, FILE *fp);

static void KheDrsResourceDebug(KHE_DRS_RESOURCE dr, int verbosity,
  int indent, FILE *fp)
{
  KHE_DRS_RESOURCE_ON_DAY drd;  int i;
  if( verbosity > 1 && indent >= 0 )
  {
    /* show timetable in full */
    fprintf(fp, "%*s[ %s", indent, "", KheDrsResourceId(dr));
    if( dr->open_resource_index >= 0 )
      fprintf(fp, " (open %d)", dr->open_resource_index);
    fprintf(fp, "\n");
    HaArrayForEach(dr->days, drd, i)
      KheDrsResourceOnDayDebug(drd, verbosity, indent + 2, fp);
    fprintf(fp, "%*s]\n", indent, "");
  }
  else
  {
    if( indent > 0 )
      fprintf(fp, "%*s", indent, "");
    fprintf(fp, "%s", KheDrsResourceId(dr));
    if( indent >= 0 )
      fprintf(fp, "\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_RESOURCE - expansion"                                 */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceSetSignature(KHE_DRS_SIGNATURE next_sig,              */
/*    KHE_DRS_RESOURCE_ON_DAY drd, KHE_DRS_TASK_ON_DAY dtd,                  */
/*    KHE_DRS_SIGNATURE prev_sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)           */
/*                                                                           */
/*  Build a signature object for the effect on resource monitors when        */
/*  drd is assigned to dtd.  Only extra costs are included, not the          */
/*  cost of prev_soln.                                                       */
/*                                                                           */
/*****************************************************************************/
static void KheDrsResourceOnDayLeafSet(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_DRS_TASK_ON_DAY dtd, KHE_DYNAMIC_RESOURCE_SOLVER drs);
static void KheDrsResourceOnDayLeafClear(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_DRS_TASK_ON_DAY dtd, KHE_DYNAMIC_RESOURCE_SOLVER drs);
static void KheDrsSignerEvalSignature(KHE_DRS_SIGNER dsg,
  bool debug_if_rerun, KHE_DRS_SIGNATURE prev_sig, int next_di,
  KHE_DRS_SIGNATURE sig, KHE_DYNAMIC_RESOURCE_SOLVER drs, bool debug);

static void KheDrsResourceSetSignature(KHE_DRS_SIGNATURE next_sig,
  KHE_DRS_RESOURCE_ON_DAY drd, KHE_DRS_TASK_ON_DAY dtd,
  KHE_DRS_SIGNATURE prev_sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  /* set leaf expressions of the resource on day */
  KheDrsResourceOnDayLeafSet(drd, dtd, drs);

  /* build the signature */
  KheDrsSignerEvalSignature(drd->signer, true, prev_sig,
    drd->day->open_day_index, next_sig, drs, false);

  /* clear leaf expressions of the resource on day */
  KheDrsResourceOnDayLeafClear(drd, dtd, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_ASST_TO_TASK_CLASS KheDrsResourceAddAsst(KHE_DRS_RESOURCE dr,    */
/*    KHE_DRS_ASST_TO_SHIFT sasst, KHE_DRS_TASK_CLASS dtc,                   */
/*    KHE_DRS_TASK_ON_DAY fixed_dtd, KHE_DRS_EXPANDER de)                    */
/*                                                                           */
/*  Build an assignment object.  Add it to drd->encl_dr->expand_assts        */
/*  if it is competitive, and delete it if not.                              */
/*                                                                           */
/*****************************************************************************/
static KHE_DRS_ASST_TO_TASK_CLASS KheDrsAsstToTaskClassMake(
  KHE_DRS_ASST_TO_SHIFT sasst, KHE_DRS_TASK_CLASS dtc,
  KHE_DRS_TASK_ON_DAY fixed_dtd, KHE_DYNAMIC_RESOURCE_SOLVER drs);
static bool KheDrsExpanderOpenToExtraCost(KHE_DRS_EXPANDER de,
  KHE_COST extra_cost);

static KHE_DRS_ASST_TO_TASK_CLASS KheDrsResourceAddAsst(KHE_DRS_RESOURCE dr,
  KHE_DRS_ASST_TO_SHIFT sasst, KHE_DRS_TASK_CLASS dtc,
  KHE_DRS_TASK_ON_DAY fixed_dtd, bool force, KHE_DRS_EXPANDER de)
{
  KHE_DRS_ASST_TO_TASK_CLASS res;
  if( force || KheDrsExpanderOpenToExtraCost(de, sasst->sig->cost) )
  {
    res = KheDrsAsstToTaskClassMake(sasst, dtc, fixed_dtd, de->solver);
    HaArrayAddLast(dr->expand_assts, res);
    sasst->used = true;
    return res;
  }
  else
    return NULL;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceAsstToShiftAddOrFree(KHE_DRS_RESOURCE dr,             */
/*    KHE_DRS_ASST_TO_SHIFT sasst, KHE_DYNAMIC_RESOURCE_SOLVER drs)          */
/*                                                                           */
/*  Either store sasst in dr->expand_assts_to_shifts or free it, depending   */
/*  on sasst->used.                                                          */
/*                                                                           */
/*****************************************************************************/
static void KheDrsAsstToShiftFree(KHE_DRS_ASST_TO_SHIFT sasst,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsResourceAsstToShiftAddOrFree(KHE_DRS_RESOURCE dr,
  KHE_DRS_ASST_TO_SHIFT sasst, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  if( sasst->used )
    HaArrayAddLast(dr->expand_assts_to_shifts, sasst);
  else
    KheDrsAsstToShiftFree(sasst, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSortAndReduceAssignments(KHE_DRS_RESOURCE dr,                 */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Sort the signatures of dr, and keep the best dr->resource_expand_limit   */
/*  of them, or more if there are ties.                                      */
/*                                                                           */
/*****************************************************************************/
static void KheDrsAsstToTaskClassFree(KHE_DRS_ASST_TO_TASK_CLASS asst,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);
static KHE_DRS_SIGNATURE KheDrsAsstToTaskClassSignature(
  KHE_DRS_ASST_TO_TASK_CLASS asst);
static int KheDrsAsstToTaskClassCmp(const void *t1, const void *t2);

static void KheDrsSortAndReduceAssignments(KHE_DRS_RESOURCE dr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_ASST_TO_TASK_CLASS asst, asst2;  KHE_DRS_SIGNATURE sig, sig2;

  /* sort the assignments by increasing cost */
  HaArraySort(dr->expand_assts, &KheDrsAsstToTaskClassCmp);

  /* if there is a resource expand limit and it will make a difference here */
  if( drs->solve_resource_expand_limit > 0 &&
      drs->solve_resource_expand_limit < HaArrayCount(dr->expand_assts) )
  {
    /* find asst, a signature with the largest cost that we want to keep */
    asst = HaArray(dr->expand_assts, drs->solve_resource_expand_limit-1);
    sig = KheDrsAsstToTaskClassSignature(asst);

    /* delete and free all signatures whose cost exceeds asst's */
    asst2 = HaArrayLast(dr->expand_assts);
    sig2 = KheDrsAsstToTaskClassSignature(asst2);
    while( sig2->cost > sig->cost )
    {
      HaArrayDeleteLast(dr->expand_assts);
      if( asst2 == dr->expand_free_day_asst )
        dr->expand_free_day_asst = NULL;
      KheDrsAsstToTaskClassFree(asst2, drs);
      asst2 = HaArrayLast(dr->expand_assts);
      sig2 = KheDrsAsstToTaskClassSignature(asst2);
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceAdjustSignatureCosts(KHE_DRS_RESOURCE dr,             */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Adjust the signature costs in the resource assignments of dr, by         */
/*  finding the minimum cost (that must be incurred whatever dr is           */
/*  assigned to) and moving it out of the resource assignments and           */
/*  into drs->expand_cost.                                                   */
/*                                                                           */
/*****************************************************************************/
static void KheDrsExpanderAddCost(KHE_DRS_EXPANDER de, KHE_COST cost);

static void KheDrsResourceAdjustSignatureCosts(KHE_DRS_RESOURCE dr,
  KHE_DRS_EXPANDER de)
{
  KHE_COST movable_cost;  int i;  KHE_DRS_ASST_TO_SHIFT sasst;
  KHE_DRS_ASST_TO_TASK_CLASS asst;
  if( HaArrayCount(dr->expand_assts) > 0 )
  {
    asst = HaArrayFirst(dr->expand_assts);
    movable_cost = asst->asst_to_shift->sig->cost;
    KheDrsExpanderAddCost(de, movable_cost);
    HaArrayForEach(dr->expand_assts_to_shifts, sasst, i)
      sasst->sig->cost -= movable_cost;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsResourceOnDayIsFixed(KHE_DRS_RESOURCE_ON_DAY drd,             */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs,                    */
/*    KHE_DRS_TASK_ON_DAY *dtd)                                              */
/*                                                                           */
/*  If the assignment of drd is fixed, return true and set *dtd to the task  */
/*  on day that it is fixed to.                                              */
/*                                                                           */
/*  There are three reasons why drd may be fixed:                            */
/*                                                                           */
/*    * this run is a rerun so we take the assignment from a packed soln;    */
/*                                                                           */
/*    * drd may have a closed assignment;                                    */
/*                                                                           */
/*    * drd's resource may have been assigned (in soln) to some task on the  */
/*      previous day, and that task may still be running.                    */
/*                                                                           */
/*  We can find out whether drd's resource has been assigned to some task    */
/*  on the previous day by consulting the edge leading into soln.            */
/*                                                                           */
/*****************************************************************************/
static KHE_DRS_TASK_ON_DAY KheDrsPackedSolnTaskOnDay(
  KHE_DRS_PACKED_SOLN packed_soln, KHE_DRS_DAY open_day,
  KHE_DRS_RESOURCE open_resource);
static bool KheDrsSolnResourceIsAssigned(KHE_DRS_SOLN soln,
  KHE_DRS_RESOURCE dr, KHE_DRS_TASK_ON_DAY *dtd);
static bool KheDrsTaskBusyOnDay(KHE_DRS_TASK dt, KHE_DRS_DAY day,
  KHE_DRS_TASK_ON_DAY *dtd);

static bool KheDrsResourceOnDayIsFixed(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs,
  KHE_DRS_TASK_ON_DAY *dtd)
{
  KHE_DRS_TASK_ON_DAY dtd1, dtd2;

  /* if this is a rerun, that fixes the assignment */
  if( RERUN && drs->rerun != NULL )
    return *dtd = KheDrsPackedSolnTaskOnDay(drs->rerun, drd->day, drd->encl_dr),
      true;

  /* if drd has a closed assignment, it's fixed to that */
  if( drd->closed_asst != NULL )
    return *dtd = drd->closed_asst, true;

  /* if drd's resource is assigned to a task in soln which is still running, */
  /* then drd is fixed to that */
  if( KheDrsSolnResourceIsAssigned(soln, drd->encl_dr, &dtd1) &&
	KheDrsTaskBusyOnDay(dtd1->encl_dt, drd->day, &dtd2) )
    return *dtd = dtd2, true;

  /* otherwise drd has no fixed assignment */
  return *dtd = NULL, false;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceDominanceTestCacheDebug(KHE_DRS_RESOURCE dr,          */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of dr's dominance test cache.                                */
/*                                                                           */
/*****************************************************************************/
static void KheDrsDim2TableDebug(KHE_DRS_DIM2_TABLE d2, int verbosity,
  int indent, FILE *fp);

static void KheDrsResourceDominanceTestCacheDebug(KHE_DRS_RESOURCE dr,
  int verbosity, int indent, FILE *fp)
{
  fprintf(fp, "%*s[ Dominance test cache for %s:\n", indent, "",
    KheDrsResourceId(dr));
  KheDrsDim2TableDebug(dr->expand_dom_test_cache, verbosity, indent + 2, fp);
  fprintf(fp, "%*s]\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceBuildDominanceTestCache(KHE_DRS_RESOURCE dr,          */
/*    KHE_DRS_RESOURCE_ON_DAY drd, KHE_DYNAMIC_RESOURCE_SOLVER drs)          */
/*                                                                           */
/*  Build dr's dominance test cache, assuming that its asst to shifts        */
/*  are built.                                                               */
/*                                                                           */
/*****************************************************************************/
static bool KheDrsSignerDoDominates(KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2,
  KHE_DRS_SIGNER_TEST test, int trie_start_depth, bool stop_on_neg,
  KHE_COST *avail_cost, int verbosity, int indent, FILE *fp);
static KHE_DRS_COST_TUPLE KheDrsCostTupleMake(KHE_COST psi, KHE_COST psi0,
  bool psi_plus_defined, KHE_COST psi_plus);
static void KheDrsDim2TablePut(KHE_DRS_DIM2_TABLE d2, int index2,
  int index1, KHE_DRS_COST_TUPLE ct, KHE_DYNAMIC_RESOURCE_SOLVER drs);
static void KheDrsSignatureAddAsstToShiftIndex(KHE_DRS_SIGNATURE sig, int val);

static void KheDrsResourceBuildDominanceTestCache(KHE_DRS_RESOURCE dr,
  KHE_DRS_RESOURCE_ON_DAY drd, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int i, j;  KHE_DRS_ASST_TO_SHIFT sasst1, sasst2;  KHE_COST cost;
  KHE_DRS_COST_TUPLE ct;
  if( USE_DOM_CACHING )
  {
    HaArrayForEach(dr->expand_assts_to_shifts, sasst1, i)
    {
      KheDrsSignatureAddAsstToShiftIndex(sasst1->sig, i);
      HaArrayForEach(dr->expand_assts_to_shifts, sasst2, j)
      {
	cost = 0;
	KheDrsSignerDoDominates(drd->signer, sasst1->sig, sasst2->sig,
	  KHE_DRS_SIGNER_ALL, 0, false, &cost, 0, 0, NULL);
	ct = KheDrsCostTupleMake(cost, 0, false, 0);
	KheDrsDim2TablePut(dr->expand_dom_test_cache, i, j, ct, drs);
      }
    }
    if( DEBUG67 )
      KheDrsResourceDominanceTestCacheDebug(dr, 2, 2, stderr);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceExpandBegin(KHE_DRS_RESOURCE dr,                      */
/*    KHE_DRS_SOLN prev_soln, KHE_DRS_DAY next_day,                          */
/*    KHE_DRS_RESOURCE_SET free_resources,                                   */
/*    KHE_DRS_ASST_TO_TASK_SET fixed_assts, KHE_DRS_EXPANDER de)             */
/*                                                                           */
/*  Carry out the first part of beginning an expansion using open resource   */
/*  dr, which is to work out whether dr is involved in a fixed assignment,   */
/*  and then either add that assignment to fixed_assts (if it is) or add     */
/*  the resource to free_resources (if it isn't).                            */
/*                                                                           */
/*****************************************************************************/
static void KheDrsExpanderDeleteFreeResource(KHE_DRS_EXPANDER de);
static KHE_DRS_ASST_TO_SHIFT KheDrsAsstToShiftMake(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_SIGNATURE prev_sig,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);
static KHE_DRS_ASST_TO_TASK KheDrsAsstToTaskMake(
  KHE_DRS_ASST_TO_TASK_CLASS asst, KHE_DRS_TASK_ON_DAY fixed_dtd);
static void KheDrsAsstToTaskSetAddLast(KHE_DRS_ASST_TO_TASK_SET dats,
  KHE_DRS_ASST_TO_TASK dat);
static void KheDrsResourceSetAddLast(KHE_DRS_RESOURCE_SET rs,
  KHE_DRS_RESOURCE dr);
static char *KheDrsTaskExpandRoleShow(KHE_DRS_TASK_EXPAND_ROLE dter);

static void KheDrsResourceExpandBegin(KHE_DRS_RESOURCE dr,
  KHE_DRS_SOLN prev_soln, KHE_DRS_DAY next_day,
  KHE_DRS_RESOURCE_SET free_resources,
  KHE_DRS_ASST_TO_TASK_SET fixed_assts, KHE_DRS_EXPANDER de)
{
  KHE_DRS_TASK_ON_DAY fixed_dtd;
  KHE_DRS_RESOURCE_ON_DAY drd;  KHE_DRS_ASST_TO_SHIFT sasst;
  KHE_DRS_ASST_TO_TASK tasst;  KHE_DRS_ASST_TO_TASK_CLASS asst;
  KHE_DRS_TASK dt;  KHE_DRS_SIGNATURE prev_sig;

  /* there should be no old assts to shifts or assignments to classes */
  HnAssert(HaArrayCount(dr->expand_assts_to_shifts) == 0,
    "KheDrsResourceExpandBegin internal error 1");
  HnAssert(HaArrayCount(dr->expand_assts) == 0,
    "KheDrsResourceExpandBegin internal error 2");
  HnAssert(dr->expand_free_day_asst == NULL,
    "KheDrsResourceExpandBegin internal error 3");

  /* add assignments to shifts and assignments to classes */
  drd = KheDrsResourceOnDay(dr, next_day);
  if( KheDrsResourceOnDayIsFixed(drd, prev_soln, de->solver, &fixed_dtd) )
  {
    /* mark the resource as fixed, and inform the expander */
    dr->expand_role = KHE_DRS_RESOURCE_EXPAND_FIXED;
    KheDrsExpanderDeleteFreeResource(de);

    /* fixed assignment to fixed_dtd; make and add sasst and asst */
    prev_sig = HaArray(prev_soln->sig_set.signatures, dr->open_resource_index);
    sasst = KheDrsAsstToShiftMake(drd, fixed_dtd, prev_sig, de->solver);
    asst = KheDrsResourceAddAsst(dr, sasst, NULL, fixed_dtd, true, de);
    KheDrsResourceAsstToShiftAddOrFree(dr, sasst, de->solver);

    /* mark the task as fixed */
    if( fixed_dtd != NULL )
    {
      dt = fixed_dtd->encl_dt;
      HnAssert(dt->expand_role == KHE_DRS_TASK_EXPAND_NO,
	"KheDrsResourceExpandBegin internal error 4 (role %s)",
	KheDrsTaskExpandRoleShow(dt->expand_role));
      dt->expand_role = KHE_DRS_TASK_EXPAND_FIXED;
    }

    /* make tasst and add to expander and fixed_assts */
    tasst = KheDrsAsstToTaskMake(asst, fixed_dtd);
    KheDrsAsstToTaskSetAddLast(fixed_assts, tasst);

    /* and build dominance cache */
    KheDrsResourceBuildDominanceTestCache(dr, drd, de->solver);
  }
  else
  {
    /* mark the resource as free */
    dr->expand_role = KHE_DRS_RESOURCE_EXPAND_FREE;

    /* add dr to free_resources */
    KheDrsResourceSetAddLast(free_resources, dr);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceExpandBeginFree(KHE_DRS_RESOURCE dr,                  */
/*    KHE_DRS_SOLN prev_soln, KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de)     */
/*                                                                           */
/*  Carry out the second part of beginning an expansion using open resource  */
/*  dr, which is now known to be free.  So add its assignments in.           */
/*                                                                           */
/*  Implementation note.  This code is separated from the previous function  */
/*  because there are cases where, by adding fixed assignments to the        */
/*  expander, the first function might allow more pruning in the second.     */
/*                                                                           */
/*****************************************************************************/
static bool KheDrsTaskClassAcceptResourceBegin(KHE_DRS_TASK_CLASS dtc,
  KHE_DRS_RESOURCE_ON_DAY drd, KHE_DRS_TASK_ON_DAY *dtd);
static void KheDrsTaskClassAcceptResourceEnd(KHE_DRS_TASK_CLASS dtc,
  KHE_DRS_TASK_ON_DAY dtd);

static void KheDrsResourceExpandBeginFree(KHE_DRS_RESOURCE dr,
  KHE_DRS_SOLN prev_soln, KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de)
{
  int i, j;  KHE_DRS_RESOURCE_ON_DAY drd;  KHE_DRS_TASK_ON_DAY dtd;
  KHE_DRS_TASK_CLASS dtc;  KHE_DRS_SHIFT ds;  KHE_DRS_ASST_TO_SHIFT sasst;
  KHE_DRS_SIGNATURE prev_sig;

  /* there should be no old assts to shifts or assignments to classes */
  HnAssert(dr->expand_role == KHE_DRS_RESOURCE_EXPAND_FREE,
    "KheDrsResourceExpandBeginFree internal error");

  /* unfixed assignment; make sassts and assts as required */
  drd = KheDrsResourceOnDay(dr, next_day);
  prev_sig = HaArray(prev_soln->sig_set.signatures, dr->open_resource_index);
  HaArrayForEach(next_day->shifts, ds, i)
  {
    sasst = NULL;
    HaArrayForEach(ds->task_classes, dtc, j)
      if( KheDrsTaskClassAcceptResourceBegin(dtc, drd, &dtd) )
      {
	if( sasst == NULL )
	  sasst = KheDrsAsstToShiftMake(drd, dtd, prev_sig, de->solver);
	KheDrsResourceAddAsst(dr, sasst, dtc, NULL, false, de);
	KheDrsTaskClassAcceptResourceEnd(dtc, dtd);
      }
    if( sasst != NULL )
      KheDrsResourceAsstToShiftAddOrFree(dr, sasst, de->solver);
  }

  /* one sasst and asst for a free day */
  sasst = KheDrsAsstToShiftMake(drd, NULL, prev_sig, de->solver);
  dr->expand_free_day_asst = KheDrsResourceAddAsst(dr, sasst, NULL, NULL,
    false, de);
  KheDrsResourceAsstToShiftAddOrFree(dr, sasst, de->solver);

  /* sort expand_assts by increasing cost and keep only the best */
  KheDrsSortAndReduceAssignments(dr, de->solver);

  /* move any unavoidable cost into the expander */
  KheDrsResourceAdjustSignatureCosts(dr, de);

  /* add a dominance test cache (still to do) */
  KheDrsResourceBuildDominanceTestCache(dr, drd, de->solver);
  /* dr->expand_dom_test_cache is clear here, need to build it now */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceExpandEnd(KHE_DRS_RESOURCE dr, KHE_DRS_EXPANDER de)   */
/*                                                                           */
/*  End dr's part of the expansion of one solution.                          */
/*                                                                           */
/*****************************************************************************/
static void KheDrsAsstToShiftFree(KHE_DRS_ASST_TO_SHIFT sasst,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);
static void KheDrsDim2TableClear(KHE_DRS_DIM2_TABLE d2,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);
static void KheDrsSignatureClearAsstToShiftIndex(KHE_DRS_SIGNATURE sig);

static void KheDrsResourceExpandEnd(KHE_DRS_RESOURCE dr, KHE_DRS_EXPANDER de)
{
  int i;  KHE_DRS_ASST_TO_SHIFT sasst;

  /* mark the resource as not involved in any expansion */
  dr->expand_role = KHE_DRS_RESOURCE_EXPAND_NO;

  /* clear out the dominance test cache */
  if( USE_DOM_CACHING )
    KheDrsDim2TableClear(dr->expand_dom_test_cache, de->solver);

  /* clear assignments to shifts */
  HaArrayForEach(dr->expand_assts_to_shifts, sasst, i)
  {
    /* if( sasst != NULL ) */
    if( USE_DOM_CACHING )
      KheDrsSignatureClearAsstToShiftIndex(sasst->sig);
    KheDrsAsstToShiftFree(sasst, de->solver);
  }
  HaArrayClear(dr->expand_assts_to_shifts);

  /* clear assignments to task classes */
  HaArrayAppend(de->solver->asst_to_task_class_free_list, dr->expand_assts, i);
  HaArrayClear(dr->expand_assts);
  dr->expand_free_day_asst = NULL;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_RESOURCE_SET"                                         */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_RESOURCE_SET KheDrsResourceSetMake(                              */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make or get a new, empty resource set.                                   */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_RESOURCE_SET KheDrsResourceSetMake(
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_RESOURCE_SET res;
  if( HaArrayCount(drs->resource_set_free_list) > 0 )
  {
    res = HaArrayLastAndDelete(drs->resource_set_free_list);
    HaArrayClear(res->resources);
  }
  else
  {
    HaMake(res, drs->arena);
    HaArrayInit(res->resources, drs->arena);
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceSetFree(KHE_DRS_RESOURCE_SET rs,                      */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Free rs.                                                                 */
/*                                                                           */
/*****************************************************************************/

static void KheDrsResourceSetFree(KHE_DRS_RESOURCE_SET rs,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  HaArrayAddLast(drs->resource_set_free_list, rs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceSetClear(KHE_DRS_RESOURCE_SET rs)                     */
/*                                                                           */
/*  Clear rs.                                                                */
/*                                                                           */
/*****************************************************************************/

static void KheDrsResourceSetClear(KHE_DRS_RESOURCE_SET rs)
{
  HaArrayClear(rs->resources);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsResourceSetCount(KHE_DRS_RESOURCE_SET rs)                      */
/*                                                                           */
/*  Return the number of resources in rs.                                    */
/*                                                                           */
/*****************************************************************************/

static int KheDrsResourceSetCount(KHE_DRS_RESOURCE_SET rs)
{
  return HaArrayCount(rs->resources);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_RESOURCE KheDrsResourceSetResource(KHE_DRS_RESOURCE_SET rs,      */
/*    int i)                                                                 */
/*                                                                           */
/*  Return the i'th resource of rs.                                          */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_RESOURCE KheDrsResourceSetResource(KHE_DRS_RESOURCE_SET rs,
  int i)
{
  return HaArray(rs->resources, i);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceSetAddLast(KHE_DRS_RESOURCE_SET rs,                   */
/*    KHE_DRS_RESOURCE dr)                                                   */
/*                                                                           */
/*  Add dr to the end of rs.                                                 */
/*                                                                           */
/*****************************************************************************/

static void KheDrsResourceSetAddLast(KHE_DRS_RESOURCE_SET rs,
  KHE_DRS_RESOURCE dr)
{
  HaArrayAddLast(rs->resources, dr);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceSetDeleteLast(KHE_DRS_RESOURCE_SET rs)                */
/*                                                                           */
/*  Delete the last element of rs.                                           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsResourceSetDeleteLast(KHE_DRS_RESOURCE_SET rs)
{
  HnAssert(KheDrsResourceSetCount(rs) > 0,
    "KheDrsResourceSetDeleteLast internal error");
  HaArrayDeleteLast(rs->resources);
}


/*****************************************************************************/
/*                                                                           */
/*  KheDrsResourceSetForEach(rs, dr, i)                                      */
/*                                                                           */
/*  Iterate over the resources dr of rs; like HaArrayForEach.                */
/*                                                                           */
/*****************************************************************************/

#define KheDrsResourceSetForEach(rs, dr, i)				\
  HaArrayForEach((rs)->resources, (dr), (i))


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceSetDebug(KHE_DRS_RESOURCE_SET rs,                     */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug printf of rs onto fp with the given verbosity and indent.          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsResourceSetDebug(KHE_DRS_RESOURCE_SET rs,
  int verbosity, int indent, FILE *fp)
{
  KHE_DRS_RESOURCE dr;  int i;
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  fprintf(fp, "{");
  KheDrsResourceSetForEach(rs, dr, i)
  {
    if( i > 0 )
      fprintf(fp, ", ");
    fprintf(fp, "%s", KheDrsResourceId(dr));
  }
  fprintf(fp, "}");
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_RESOURCE_ON_DAY"                                      */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_RESOURCE_ON_DAY KheDrsResourceOnDayMake(KHE_DRS_RESOURCE dr,     */
/*    KHE_DRS_DAY day, KHE_DYNAMIC_RESOURCE_SOLVER drs)                      */
/*                                                                           */
/*  Make a new resource on day object with these attributes.                 */
/*  This function assumes that dr is not assigned any task on day.           */
/*                                                                           */
/*****************************************************************************/
static KHE_DRS_SIGNER KheDrsSignerMake(KHE_DRS_DAY day,
  KHE_DRS_RESOURCE_ON_DAY drd, KHE_DRS_SHIFT ds,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);

static KHE_DRS_RESOURCE_ON_DAY KheDrsResourceOnDayMake(KHE_DRS_RESOURCE dr,
  KHE_DRS_DAY day, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_RESOURCE_ON_DAY res;
  HaMake(res, drs->arena);
  res->encl_dr = dr;
  res->day = day;
  res->closed_asst = NULL;
  HaArrayInit(res->external_today, drs->arena);
  res->signer = KheDrsSignerMake(NULL, res, NULL, drs);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsResourceOnDayIndex(KHE_DRS_RESOURCE_ON_DAY drd)                */
/*                                                                           */
/*  Return the open resource index of drd.  This will be -1 if the resource  */
/*  is not open                                                              */
/*                                                                           */
/*****************************************************************************/

static int KheDrsResourceOnDayIndex(KHE_DRS_RESOURCE_ON_DAY drd)
{
  return drd->encl_dr->open_resource_index;
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheDrsResourceOnDayId(KHE_DRS_RESOURCE_ON_DAY drd)                 */
/*                                                                           */
/*  Return an Id for drd.  The result is held in a static array and is       */
/*  best used for debugging only.                                            */
/*                                                                           */
/*****************************************************************************/

static char *KheDrsResourceOnDayId(KHE_DRS_RESOURCE_ON_DAY drd)
{
  static char buff[200];
  snprintf(buff, 200, "%s:%s", KheDrsResourceId(drd->encl_dr),
    KheDrsDayId(drd->day));
  return buff;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsResourceOnDayHasTimeExpr(KHE_DRS_RESOURCE_ON_DAY drd,         */
/*    KHE_DRS_EXPR_TAG tag, KHE_TIME time, KHE_DYNAMIC_RESOURCE_SOLVER drs,  */
/*    KHE_DRS_EXPR *res_e)                                                   */
/*                                                                           */
/*  If drd has a BUSY_TIME, FREE_TIME, or WORK_TIME expression with the      */
/*  given tag and time, then return true and set *res_e to it.  Otherwise    */
/*  return false with *res_e set to NULL.                                    */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsResourceOnDayHasTimeExpr(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_DRS_EXPR_TAG tag, KHE_TIME time, KHE_DYNAMIC_RESOURCE_SOLVER drs,
  KHE_DRS_EXPR *res_e)
{
  int i;  KHE_DRS_EXPR e;  KHE_DRS_EXPR_BUSY_TIME ebt;
  KHE_DRS_EXPR_FREE_TIME eft;  KHE_DRS_EXPR_WORK_TIME ewt;
  HaArrayForEach(drd->external_today, e, i)
    if( e->tag == tag )
    {
      switch( e->tag )
      {
	case KHE_DRS_EXPR_BUSY_TIME_TAG:

	  ebt = (KHE_DRS_EXPR_BUSY_TIME) e;
	  if( ebt->time == time )
	    return *res_e = e, true;
	  break;

	case KHE_DRS_EXPR_FREE_TIME_TAG:

	  eft = (KHE_DRS_EXPR_FREE_TIME) e;
	  if( eft->time == time )
	    return *res_e = e, true;
	  break;

	case KHE_DRS_EXPR_WORK_TIME_TAG:

	  ewt = (KHE_DRS_EXPR_WORK_TIME) e;
	  if( ewt->time == time )
	    return *res_e = e, true;
	  break;

	default:

	  HnAbort("KheDrsResourceOnDayHasTimeExpr: invalid tag (%d)", e->tag);
      }
    }
  return *res_e = NULL, false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsResourceOnDayHasDayExpr(KHE_DRS_RESOURCE_ON_DAY drd,          */
/*    KHE_DRS_EXPR_TAG tag, KHE_TIME_GROUP tg,                               */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, KHE_DRS_EXPR *res_e)                  */
/*                                                                           */
/*  If drd has a BUSY_DAY, FREE_DAY, or WORK_DAY expression with the given   */
/*  tag and time group, then return true and set *res_e to it.  Otherwise    */
/*  return false with *res_e set to NULL.                                    */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsResourceOnDayHasDayExpr(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_DRS_EXPR_TAG tag, KHE_TIME_GROUP tg,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, KHE_DRS_EXPR *res_e)
{
  int i;  KHE_DRS_EXPR e;  KHE_DRS_EXPR_BUSY_DAY ebd;
  KHE_DRS_EXPR_FREE_DAY efd;  KHE_DRS_EXPR_WORK_DAY ewd;
  HaArrayForEach(drd->external_today, e, i)
    if( e->tag == tag )
    {
      switch( e->tag )
      {
	case KHE_DRS_EXPR_BUSY_DAY_TAG:

	  ebd = (KHE_DRS_EXPR_BUSY_DAY) e;
	  if( KheTimeGroupEqual(ebd->time_group, tg) )
	    return *res_e = e, true;
	  break;

	case KHE_DRS_EXPR_FREE_DAY_TAG:

	  efd = (KHE_DRS_EXPR_FREE_DAY) e;
	  if( KheTimeGroupEqual(efd->time_group, tg) )
	    return *res_e = e, true;
	  break;

	case KHE_DRS_EXPR_WORK_DAY_TAG:

	  ewd = (KHE_DRS_EXPR_WORK_DAY) e;
	  if( KheTimeGroupEqual(ewd->time_group, tg) )
	    return *res_e = e, true;
	  break;

	default:

	  HnAbort("KheDrsResourceHasDayExpr: invalid tag (%d)", e->tag);
      }
    }
  return *res_e = NULL, false;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceOnDayOpen(KHE_DRS_RESOURCE_ON_DAY drd,                */
/*    int open_resource_index, int open_day_index, KHE_DRS_DAY_RANGE ddr,    */
/*    KHE_DRS_PACKED_SOLN init_soln, KHE_DYNAMIC_RESOURCE_SOLVER drs)        */
/*                                                                           */
/*  Open drd.                                                                */
/*                                                                           */
/*****************************************************************************/
static void KheDrsTaskUnAssign(KHE_DRS_TASK dt, bool task);
static void KheDrsExprGatherForOpening(KHE_DRS_EXPR e,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);
static void KheDrsPackedSolnSetTaskOnDay(KHE_DRS_PACKED_SOLN packed_soln,
  int open_day_index, int open_resource_index, KHE_DRS_TASK_ON_DAY dtd);
static void KheDrsSignerMakeCorrelated(KHE_DRS_SIGNER dsg);

static void KheDrsResourceOnDayOpen(KHE_DRS_RESOURCE_ON_DAY drd,
  int open_resource_index, int open_day_index, KHE_DRS_DAY_RANGE ddr,
  KHE_DRS_PACKED_SOLN init_soln, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_TASK_ON_DAY dtd;  KHE_DRS_TASK dt;  KHE_DRS_EXPR e;  int i;
  KHE_DRS_DAY_RANGE open_day_range;

  /* unassign affected tasks and possibly add the assignments to init_soln */
  dtd = drd->closed_asst;
  if( dtd != NULL )
  {
    dt = dtd->encl_dt;
    if( KheDrsDayRangeSubset(dt->encl_dtc->day_range, ddr) )
    {
      KheDrsTaskUnAssign(dt, true);
      if( init_soln != NULL )
	KheDrsPackedSolnSetTaskOnDay(init_soln, open_day_index,
	  open_resource_index, dtd);
    }
  }

  /* set make_correlated field of signer */
  if( drs->solve_correlated_exprs )
    KheDrsSignerMakeCorrelated(drd->signer);

  /* gather for opening expressions depending on drd (unassigned or not) */
  open_day_range = KheDrsDayRangeMake(open_day_index, open_day_index);
  HaArrayForEach(drd->external_today, e, i)
  {
    e->open_children_by_day.range = open_day_range;
    KheDrsExprGatherForOpening(e, drs);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceOnDayClose(KHE_DRS_RESOURCE_ON_DAY drd,               */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Close drd.                                                               */
/*                                                                           */
/*****************************************************************************/
static void KheDrsSignerClear(KHE_DRS_SIGNER dsg,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsResourceOnDayClose(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KheDrsSignerClear(drd->signer, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceOnDayAddExpr(KHE_DRS_RESOURCE_ON_DAY drd,             */
/*    KHE_DRS_EXPR e)                                                        */
/*                                                                           */
/*  Add e, known to be new, to drd.                                          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsResourceOnDayAddExpr(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_DRS_EXPR e)
{
  HaArrayAddLast(drd->external_today, e);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceOnDayAddOpenExpr(KHE_DRS_RESOURCE_ON_DAY drd,         */
/*    KHE_DRS_EXPR e)                                                        */
/*                                                                           */
/*  Add internal expression e to drd.                                        */
/*                                                                           */
/*****************************************************************************/

static void KheDrsResourceOnDayAddOpenExpr(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_DRS_EXPR e)
{
  KheDrsSignerAddOpenExpr(drd->signer, e);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsResourceOnDayAddDomTest(KHE_DRS_RESOURCE_ON_DAY drd,           */
/*    KHE_DRS_DOM_TEST dom_test)                                             */
/*                                                                           */
/*  Add dom_test to drd and return its sig index.                            */
/*                                                                           */
/*****************************************************************************/

static int KheDrsResourceOnDayAddDomTest(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_DRS_DOM_TEST dom_test, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  return KheDrsSignerAddDomTest(drd->signer, dom_test, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceOnDayLeafSet(KHE_DRS_RESOURCE_ON_DAY drd,             */
/*    KHE_DRS_TASK_ON_DAY dtd, KHE_DYNAMIC_RESOURCE_SOLVER drs)              */
/*                                                                           */
/*  Call KheDrsExprLeafSet for each external expression affected by drd.     */
/*                                                                           */
/*****************************************************************************/
static void KheDrsExprLeafSet(KHE_DRS_EXPR e, KHE_DRS_TASK_ON_DAY dtd,
  KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsResourceOnDayLeafSet(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_DRS_TASK_ON_DAY dtd, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR e;  int i;
  if( dtd != NULL )
    HaArrayForEach(drd->external_today, e, i)
      KheDrsExprLeafSet(e, dtd, drd->encl_dr, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceOnDayLeafClear(KHE_DRS_RESOURCE_ON_DAY drd,           */
/*    KHE_DRS_TASK_ON_DAY dtd, KHE_DYNAMIC_RESOURCE_SOLVER drs)              */
/*                                                                           */
/*  Call KheDrsExprLeafClear for each external expression affected by drd.   */
/*                                                                           */
/*****************************************************************************/
static void KheDrsExprLeafClear(KHE_DRS_EXPR e,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsResourceOnDayLeafClear(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_DRS_TASK_ON_DAY dtd, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR e;  int i;
  if( dtd != NULL )
    HaArrayForEach(drd->external_today, e, i)
      KheDrsExprLeafClear(e, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceOnDayDebug(KHE_DRS_RESOURCE_ON_DAY drd,               */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of drd onto fp with the given verbosity and indent.          */
/*                                                                           */
/*****************************************************************************/
static void KheDrsTaskOnDayDebug(KHE_DRS_TASK_ON_DAY dtd,
  int verbosity, int indent, FILE *fp);

static void KheDrsResourceOnDayDebug(KHE_DRS_RESOURCE_ON_DAY drd,
  int verbosity, int indent, FILE *fp)
{
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  fprintf(fp, "%s: ", KheDrsDayId(drd->day));
  if( drd->closed_asst != NULL )
    KheDrsTaskOnDayDebug(drd->closed_asst, 1, -1, fp);
  else
    fprintf(fp, "-");
  fprintf(fp, " (external_today %d)", HaArrayCount(drd->external_today));
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Major category "events"                                                  */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_TASK_EXPAND_ROLE"                                     */
/*                                                                           */
/*****************************************************************************/

static char *KheDrsTaskExpandRoleShow(KHE_DRS_TASK_EXPAND_ROLE dter)
{
  switch( dter )
  {
    case KHE_DRS_TASK_EXPAND_NO:	return "expand_no";
    case KHE_DRS_TASK_EXPAND_FIXED:	return "expand_fixed";
    case KHE_DRS_TASK_EXPAND_MUST:	return "expand_must";
    case KHE_DRS_TASK_EXPAND_FREE:	return "expand_free";
    default:				return "?";
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_TASK"                                                 */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheTaskAddDays(KHE_TASK task, KHE_DRS_TASK dt,                      */
/*    int first_solve_index, int last_solve_index,                           */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Add the days covered by task and its descendants to dt.                  */
/*                                                                           */
/*****************************************************************************/
static KHE_DRS_TASK_ON_DAY KheDrsTaskOnDayMake(KHE_DRS_TASK encl_dt,
  KHE_DRS_DAY day, KHE_TASK task, KHE_TIME time,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheTaskAddDays(KHE_TASK task, KHE_DRS_TASK dt,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_TASK child_task;  int i, durn, day_index;  KHE_DRS_DAY day;
  KHE_MEET meet;  KHE_TIME start_time, time;  KHE_DRS_TASK_ON_DAY dtd;

  /* do it for task */
  meet = KheTaskMeet(task);
  HnAssert(meet != NULL, "KheTaskAddDays internal error (no meet)");
  start_time = KheMeetAsstTime(meet);
  HnAssert(start_time != NULL,  "KheTaskAddDays internal error (no time)");
  durn = KheMeetDuration(meet);
  for( i = 0;  i < durn;  i++ )
  {
    time = KheTimeNeighbour(start_time, i);
    day_index = KheFrameTimeIndex(drs->days_frame, time);
    day = HaArray(drs->all_days, day_index);
    dtd = KheDrsTaskOnDayMake(dt, day, task, time, drs);
    HaArrayAddLast(dt->days, dtd);
  }

  /* do it for task's descendants */
  for( i = 0;  i < KheTaskAssignedToCount(task);  i++ )
  {
    child_task = KheTaskAssignedTo(task, i);
    KheTaskAddDays(child_task, dt, drs);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsTaskMake(KHE_DRS_TASK_CLASS encl_dtc,                         */
/*    int index_in_encl_dtc, KHE_TASK task, KHE_COST asst_cost,              */
/*    KHE_COST non_asst_cost, KHE_DYNAMIC_RESOURCE_SOLVER drs,               */
/*    KHE_DRS_TASK *dt)                                                      */
/*                                                                           */
/*  Make a new drs task with these attributes, add it to drs->all_root_tasks */
/*  at the same index as it has in the solution, and return it in *dt.       */
/*                                                                           */
/*  Return true if all this has been done successfully.  The only reason     */
/*  why false could be returned is that the call to KheDrsTaskAssign         */
/*  uncovered a case of a resource assigned to two tasks on the same day.    */
/*                                                                           */
/*****************************************************************************/
static int KheDrsTaskOnDayCmp(const void *t1, const void *t2);
static bool KheDrsTaskAssign(KHE_DRS_TASK dt, KHE_DRS_RESOURCE dr, bool task);

static bool KheDrsTaskMake(KHE_DRS_TASK_CLASS encl_dtc,
  int index_in_encl_dtc, KHE_TASK task, KHE_COST asst_cost,
  KHE_COST non_asst_cost, KHE_DYNAMIC_RESOURCE_SOLVER drs,
  KHE_DRS_TASK *dt)
{
  KHE_DRS_TASK res;  int task_index;  KHE_RESOURCE r;

  /* get the basic object and initialize its fields */
  HnAssert(KheTaskIsProperRoot(task), "KheDrsTaskMake internal error");
  HaMake(res, drs->arena);
  res->encl_dtc = encl_dtc;
  res->index_in_encl_dtc = index_in_encl_dtc;
  res->expand_role = KHE_DRS_TASK_EXPAND_NO;
  res->open = false;
  res->task = task;
  res->closed_asst = NULL;
  res->asst_cost = asst_cost;
  res->non_asst_cost = non_asst_cost;
  HaArrayInit(res->days, drs->arena);

  /* add, sort, and organize the task on day objects */
  KheTaskAddDays(task, res, drs);
  HaArraySort(res->days, &KheDrsTaskOnDayCmp);

  /* add res to drs->all_root_tasks */
  task_index = KheTaskSolnIndex(task);
  HaArrayFill(drs->all_root_tasks, task_index + 1, NULL);
  HnAssert(HaArray(drs->all_root_tasks, task_index) == NULL,
    "KheDrsTaskMake internal error");
  HaArrayPut(drs->all_root_tasks, task_index, res);

  /* if task is assigned, make that assignment in res */
  r = KheTaskAsstResource(task);
  if( r != NULL )
  {
    if( !KheDrsTaskAssign(res, KheDrsResource(r, drs), false) )
      return *dt = false, NULL;
  }

  /* and also return res */
  return *dt = res, true;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsTaskAssign(KHE_DRS_TASK dt, KHE_DRS_RESOURCE dr, bool task)   */
/*                                                                           */
/*  Assign dr (which must be non-NULL) to dt, by setting the closed_asst     */
/*  fields in dt, dt->days, and dr.  If task is true, also assign dt's KHE   */
/*  task to dr's KHE resource, aborting if that does not succeed.            */
/*                                                                           */
/*  The enclosing task class has a list of unassigned tasks; but do not      */
/*  remove dt from it, because we are being lazy:  the unassigned tasks      */
/*  will be sorted out later, by KheDrsTaskClassOrganizeUnassignedTasks.     */
/*                                                                           */
/*  Return true if the assignment was successful.  The only reason for       */
/*  failure is if the assignment gives dr two tasks on one day.              */
/*                                                                           */
/*****************************************************************************/
static void KheDrsTaskDebug(KHE_DRS_TASK dt, int verbosity,
  int indent, FILE *fp);

static bool KheDrsTaskAssign(KHE_DRS_TASK dt, KHE_DRS_RESOURCE dr, bool task)
{
  KHE_DRS_TASK_ON_DAY dtd;  int i;  KHE_DRS_RESOURCE_ON_DAY drd;
  if( DEBUG15 )
    fprintf(stderr, "[ KheDrsTaskAssign(%s (%d days), %s, %s)\n",
      KheTaskId(dt->task), HaArrayCount(dt->days), KheDrsResourceId(dr),
      bool_show(task));
  HnAssert(dt->closed_asst == NULL, "KheDrsTaskAssign internal error 1");
  HnAssert(dr != NULL, "KheDrsTaskAssign internal error 2");
  if( task && !KheTaskAssignResource(dt->task, dr->resource) )
    HnAbort("KheDrsTaskAssign internal error 3 (cannot assign %s to %s)",
      KheDrsResourceId(dr), KheTaskId(dt->task));
  dt->closed_asst = dr;
  HaArrayForEach(dt->days, dtd, i)
  {
    drd = KheDrsResourceOnDay(dr, dtd->day);
    HnAssert(dtd->closed_asst == NULL, "KheDrsTaskAssign internal error 4");
    if( drd->closed_asst != NULL )
    {
      if( DEBUG15 )
	fprintf(stderr, "] KheDrsTaskAssign returning false\n");
      return false;
    }
    dtd->closed_asst = drd;
    drd->closed_asst = dtd;
  }
  if( DEBUG15 )
    fprintf(stderr, "] KheDrsTaskAssign returning true\n");
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsTaskUnAssign(KHE_DRS_TASK dt, bool task)                      */
/*                                                                           */
/*  Unassign dt (which must be assigned some drs resource dr), by clearing   */
/*  the closed_asst fields in dt, dt->days, and dr.  If task is true, also   */
/*  unassign dt's KHE task, or do nothing if the task will not unassign.     */
/*                                                                           */
/*  The enclosing task class has a list of unassigned tasks; add dt to it.   */
/*                                                                           */
/*****************************************************************************/

static void KheDrsTaskUnAssign(KHE_DRS_TASK dt, bool task)
{
  KHE_DRS_TASK_ON_DAY dtd;  int i;
  KHE_DRS_RESOURCE dr;  KHE_DRS_RESOURCE_ON_DAY drd;
  dr = dt->closed_asst;
  HnAssert(dr != NULL, "KheDrsTaskUnAssign internal error 1");
  if( DEBUG15 )
    fprintf(stderr, "[ KheDrsTaskUnAssign(%s, %s, %s)\n", KheTaskId(dt->task),
      KheDrsResourceId(dr), bool_show(task));
  if( task )
  {
    HnAssert(KheTaskAsstResource(dt->task) == dr->resource,
      "KheDrsTaskUnAssign internal error 2");
    if( !KheTaskUnAssignResource(dt->task) )
    {
      if( DEBUG16 )
	fprintf(stderr, "  failed to unassign %s\n", KheTaskId(dt->task));
      return;
    }
    if( DEBUG9 )
      fprintf(stderr, "  after unassigning %s, cost = %.5f\n",
	KheTaskId(dt->task), KheCostShow(KheSolnCost(KheTaskSoln(dt->task))));
  }
  HaArrayForEach(dt->days, dtd, i)
  {
    drd = KheDrsResourceOnDay(dr, dtd->day);
    HnAssert(dtd->closed_asst == drd, "KheDrsTaskUnAssign internal error 3");
    HnAssert(drd->closed_asst == dtd, "KheDrsTaskUnAssign internal error 4");
    dtd->closed_asst = NULL;
    drd->closed_asst = NULL;
  }
  dt->closed_asst = NULL;
  HaArrayAddLast(dt->encl_dtc->unassigned_tasks, dt);
  if( DEBUG15 )
    fprintf(stderr, "] KheDrsTaskUnAssign returning\n");
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsTaskIndexInEnclCmp(const void *t1, const void *t2)             */
/*                                                                           */
/*  Comparison function for sorting an array of drs tasks into increasing    */
/*  order of their index in their enclosing task class.                      */
/*                                                                           */
/*  In addition, it moves any assigned tasks to the end.                     */
/*                                                                           */
/*****************************************************************************/

static int KheDrsTaskIndexInEnclCmp(const void *t1, const void *t2)
{
  KHE_DRS_TASK dt1 = * (KHE_DRS_TASK *) t1;
  KHE_DRS_TASK dt2 = * (KHE_DRS_TASK *) t2;
  if( dt1->closed_asst != NULL )
  {
    if( dt2->closed_asst != NULL )
    {
      /* dt1 is assigned, dt2 is assigned */
      return 0;
    }
    else
    {
      /* dt1 is assigned, dt2 is not assigned */
      return 1;
    }
  }
  else
  {
    if( dt2->closed_asst != NULL )
    {
      /* dt1 is not assigned, dt2 is assigned */
      return -1;
    }
    else
    {
      /* dt1 is not assigned, dt2 is not assigned */
      return dt1->index_in_encl_dtc - dt2->index_in_encl_dtc;
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsTaskOpen(KHE_DRS_TASK dt, KHE_DYNAMIC_RESOURCE_SOLVER drs)    */
/*                                                                           */
/*  Open task for solving.  We'll have checked previously that its days      */
/*  all lie in the current open interval, and it must currently be closed    */
/*  but unassigned; so opening dt basically just gathers for opening all     */
/*  expressions that depend on dt's assignment.                              */
/*                                                                           */
/*****************************************************************************/

static void KheDrsTaskOpen(KHE_DRS_TASK dt, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_TASK_ON_DAY dtd;  KHE_DRS_EXPR e;  int i, j, di;
  KHE_DRS_DAY_RANGE open_day_range;

  /* open dt */
  if( DEBUG19 )
    fprintf(stderr, "[ KheDrsTaskOpen(%s, drs)\n", KheTaskId(dt->task));
  HnAssert(!dt->open, "KheDrsTaskOpen internal error 1");
  HnAssert(dt->closed_asst == NULL, "KheDrsTaskOpen internal error 2");
  dt->open = true;

  /* gather external expressions for opening */
  HaArrayForEach(dt->days, dtd, i)
  {
    di = dtd->day->open_day_index;
    HnAssert(di >= 0, "KheDrsTaskOpen internal error 3");
    open_day_range = KheDrsDayRangeMake(di, di);
    HaArrayForEach(dtd->external_today, e, j)
    {
      e->open_children_by_day.range = open_day_range;
      KheDrsExprGatherForOpening(e, drs);
    }
  }
  if( DEBUG19 )
    fprintf(stderr, "] KheDrsTaskOpen\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsTaskClose(KHE_DRS_TASK dt, KHE_DRS_RESOURCE dr)               */
/*                                                                           */
/*  If dt is open, close it, with a call to KheDrsTaskAssign if dr != NULL.  */
/*                                                                           */
/*****************************************************************************/

static void KheDrsTaskClose(KHE_DRS_TASK dt, KHE_DRS_RESOURCE dr)
{
  if( dt->open )
  {
    if( DEBUG19 )
      fprintf(stderr, "[ KheDrsTaskClose(%s, drs)\n", KheTaskId(dt->task));
    dt->open = false;
    if( dr != NULL )
    {
      if( !KheDrsTaskAssign(dt, dr, true) )
	HnAbort("KheDrsTaskClose internal error");
    }
    if( DEBUG19 )
      fprintf(stderr, "] KheDrsTaskClose\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsTaskBusyOnDay(KHE_DRS_TASK dt, KHE_DRS_DAY day,               */
/*    KHE_DRS_TASK_ON_DAY *dtd)                                              */
/*                                                                           */
/*  If dt is running on day, return true with *dtd set to its task on day    */
/*  object for that day.  Otherwise return false with *dtd set to NULL.      */
/*                                                                           */
/*  Implementation note.  The task on days are sorted chronologically,       */
/*  but we don't bother to use that property here.                           */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsTaskBusyOnDay(KHE_DRS_TASK dt, KHE_DRS_DAY day,
  KHE_DRS_TASK_ON_DAY *dtd)
{
  KHE_DRS_TASK_ON_DAY dtd2;  int i;
  HaArrayForEach(dt->days, dtd2, i)
    if( dtd2->day == day )
      return *dtd = dtd2, true;
  return *dtd = NULL, false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskDoMinCost(KHE_TASK task, KHE_DRS_ASST_OP op,                 */
/*    KHE_RESOURCE r, KHE_COST *cost)                                        */
/*                                                                           */
/*  Do the actual work of KheTaskMinCost, defined just below.                */
/*                                                                           */
/*  Implementation note.  For the step cost function there can always        */
/*  be no change in cost, when we are beyond the upper limit.  So the        */
/*  minimum cost change is never positive.                                   */
/*                                                                           */
/*****************************************************************************/

static bool KheTaskDoMinCost(KHE_TASK task, KHE_DRS_ASST_OP op,
  KHE_RESOURCE ra, KHE_RESOURCE rb, KHE_COST *cost)
{
  KHE_EVENT_RESOURCE er;  KHE_SOLN soln;  int i, durn;  KHE_MONITOR m;
  KHE_TASK child_task;  KHE_CONSTRAINT c;  KHE_RESOURCE_GROUP rg;
  KHE_PREFER_RESOURCES_MONITOR prm;  KHE_PREFER_RESOURCES_CONSTRAINT prc;
  KHE_LIMIT_RESOURCES_MONITOR lrm;  KHE_LIMIT_RESOURCES_CONSTRAINT lrc;
  KHE_COST weight;

  /* do it for task */
  er = KheTaskEventResource(task);
  if( er != NULL )
  {
    soln = KheTaskSoln(task);
    for( i = 0;  i < KheSolnEventResourceMonitorCount(soln, er);  i++ )
    {
      m = KheSolnEventResourceMonitor(soln, er, i);
      c = KheMonitorConstraint(m);
      weight = KheConstraintCombinedWeight(c);
      durn = KheTaskDuration(task);
      if( KheMonitorAttachedToSoln(m) && weight > 0 )
	switch( KheConstraintCostFunction(c) )
	{
	  case KHE_STEP_COST_FUNCTION:

	    switch( KheMonitorTag(m) )
	    {
	      case KHE_ASSIGN_RESOURCE_MONITOR_TAG:

		switch( op )
		{
		  case KHE_DRS_ASST_OP_UNASSIGN:

		    /* no change */
		    break;

		  case KHE_DRS_ASST_OP_ASSIGN:

		    *cost -= weight * durn;
		    break;

		  case KHE_DRS_ASST_OP_REPLACE:

		    /* no change */
		    break;

		  default:

		    HnAbort("KheTaskDoMinCost internal error (op %d)", op);
		    break;
		}
		break;

	      case KHE_PREFER_RESOURCES_MONITOR_TAG:

		prm = (KHE_PREFER_RESOURCES_MONITOR) m;
		prc = KhePreferResourcesMonitorConstraint(prm);
		rg = KhePreferResourcesConstraintDomain(prc);
		switch( op )
		{
		  case KHE_DRS_ASST_OP_UNASSIGN:

		    if( !KheResourceGroupContains(rg, ra) )
		      *cost -= weight * durn;
		    break;

		  case KHE_DRS_ASST_OP_ASSIGN:

		    break;

		  case KHE_DRS_ASST_OP_REPLACE:

		    if( !KheResourceGroupContains(rg, ra) )
		      *cost -= weight * durn;
		    break;

		  default:

		    HnAbort("KheTaskDoMinCost internal error (op %d)", op);
		    break;
		}
		break;

	      case KHE_AVOID_SPLIT_ASSIGNMENTS_MONITOR_TAG:

		/* not handling this one */
		return false;

	      case KHE_LIMIT_RESOURCES_MONITOR_TAG:

		lrm = (KHE_LIMIT_RESOURCES_MONITOR) m;
		lrc = KheLimitResourcesMonitorConstraint(lrm);
		rg = KheLimitResourcesConstraintDomain(lrc);
		switch( op )
		{
		  case KHE_DRS_ASST_OP_UNASSIGN:
		  case KHE_DRS_ASST_OP_ASSIGN:

		    if( KheResourceGroupContains(rg, ra) )
		      *cost -= weight * durn;
		    break;

		  case KHE_DRS_ASST_OP_REPLACE:

		    if( KheResourceGroupContains(rg, ra) !=
			KheResourceGroupContains(rg, rb) )
		      *cost -= weight * durn;
		    break;

		  default:

		    HnAbort("KheTaskDoMinCost internal error (op %d)", op);
		    break;
		}
		break;

	      default:

		HnAbort("KheDrsTaskDoMinCost: unexpected monitor tag (%s)",
		  KheMonitorTagShow(KheMonitorTag(m)));
	    }
	    break;

	  case KHE_LINEAR_COST_FUNCTION:

	    switch( KheMonitorTag(m) )
	    {
	      case KHE_ASSIGN_RESOURCE_MONITOR_TAG:

		switch( op )
		{
		  case KHE_DRS_ASST_OP_UNASSIGN:

		    *cost += weight * durn;
		    break;

		  case KHE_DRS_ASST_OP_ASSIGN:

		    *cost -= weight * durn;
		    break;

		  case KHE_DRS_ASST_OP_REPLACE:

		    /* no change */
		    break;

		  default:

		    HnAbort("KheTaskDoMinCost internal error (op %d)", op);
		    break;
		}
		break;

	      case KHE_PREFER_RESOURCES_MONITOR_TAG:

		prm = (KHE_PREFER_RESOURCES_MONITOR) m;
		prc = KhePreferResourcesMonitorConstraint(prm);
		rg = KhePreferResourcesConstraintDomain(prc);
		switch( op )
		{
		  case KHE_DRS_ASST_OP_UNASSIGN:

		    if( !KheResourceGroupContains(rg, ra) )
		      *cost -= weight * durn;
		    break;

		  case KHE_DRS_ASST_OP_ASSIGN:

		    if( !KheResourceGroupContains(rg, ra) )
		      *cost += weight * durn;
		    break;

		  case KHE_DRS_ASST_OP_REPLACE:

		    if( !KheResourceGroupContains(rg, ra) )
		      *cost -= weight * durn;
		    if( !KheResourceGroupContains(rg, rb) )
		      *cost += weight * durn;
		    break;

		  default:

		    HnAbort("KheTaskDoMinCost internal error (op %d)", op);
		    break;
		}
		break;

	      case KHE_AVOID_SPLIT_ASSIGNMENTS_MONITOR_TAG:

		/* not handling this one */
		return false;

	      case KHE_LIMIT_RESOURCES_MONITOR_TAG:

		lrm = (KHE_LIMIT_RESOURCES_MONITOR) m;
		lrc = KheLimitResourcesMonitorConstraint(lrm);
		rg = KheLimitResourcesConstraintDomain(lrc);
		switch( op )
		{
		  case KHE_DRS_ASST_OP_UNASSIGN:
		  case KHE_DRS_ASST_OP_ASSIGN:

		    if( KheResourceGroupContains(rg, ra) )
		      *cost -= weight * durn;
		    break;

		  case KHE_DRS_ASST_OP_REPLACE:

		    if( KheResourceGroupContains(rg, ra) !=
			KheResourceGroupContains(rg, rb) )
		      *cost -= weight * durn;
		    break;

		  default:

		    HnAbort("KheTaskDoMinCost internal error (op %d)", op);
		    break;
		}
		break;

	      default:

		HnAbort("KheDrsTaskDoMinCost: unexpected monitor tag (%s)",
		  KheMonitorTagShow(KheMonitorTag(m)));
	    }
	    break;

	  case KHE_QUADRATIC_COST_FUNCTION:

	    /* not handling this one */
	    return false;
	}
    }
  }

  /* do it for task's children */
  for( i = 0;  i < KheTaskAssignedToCount(task);  i++ )
  {
    child_task = KheTaskAssignedTo(task, i);
    if( !KheTaskDoMinCost(child_task, op, ra, rb, cost) )
      return false;
  }
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsTaskMinCost(KHE_DRS_TASK dt, KHE_DRS_ASST_OP op,              */
/*    KHE_DRS_RESOURCE dra, KHE_DRS_RESOURCE drb, KHE_COST *cost)             */
/*                                                                           */
/*  This function evaluates three expressions from the documentation:        */
/*                                                                           */
/*    op                           function                                  */
/*    ---------------------------------------------------------------        */
/*    KHE_DRS_ASST_OP_UNASSIGN     c-(M, ra)                                 */
/*    KHE_DRS_ASST_OP_ASSIGN       c+(M, ra)                                 */
/*    KHE_DRS_ASST_OP_REPLACE      c-+(M, ra, rb)                            */
/*    ---------------------------------------------------------------        */
/*                                                                           */
/*  where M is the set of monitors that monitor dt.                          */
/*                                                                           */
/*  Instead of returning the value it sets *cost to it.  The return value    */
/*  is true if the function can calculate this cost, and false if not.       */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsTaskMinCost(KHE_DRS_TASK dt, KHE_DRS_ASST_OP op,
  KHE_DRS_RESOURCE dra, KHE_DRS_RESOURCE drb, KHE_COST *cost)
{
  bool res;
  if( DEBUG41 )
    fprintf(stderr, "[ KheDrsTaskMinCost\n");
  *cost = 0;
  res = KheTaskDoMinCost(dt->task, op, dra->resource,
    drb == NULL ? NULL : drb->resource, cost);
  if( DEBUG41 )
    fprintf(stderr, "] KheDrsTaskMinCost returning %s\n", bool_show(res));
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsTaskDebug(KHE_DRS_TASK dt, int verbosity,                     */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of dt onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsTaskDebug(KHE_DRS_TASK dt, int verbosity,
  int indent, FILE *fp)
{
  KHE_DRS_TASK_ON_DAY dtd;  int i;
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  KheTaskDebug(dt->task, 1, -1, fp);
  if( verbosity > 1 )
  {
    fprintf(fp, " asst_cost %.5f, non_asst_cost %.5f ",
      KheCostShow(dt->asst_cost), KheCostShow(dt->non_asst_cost));
    fprintf(fp, "(%d, %s, %s; ", dt->index_in_encl_dtc,
      dt->open ? "open" : "closed",
      dt->closed_asst == NULL ? "-" : KheDrsResourceId(dt->closed_asst));
    HaArrayForEach(dt->days, dtd, i)
      KheDrsTaskOnDayDebug(dtd, verbosity, -1, fp);
    fprintf(fp, ")");
  }
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_TASK_ON_DAY"                                          */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_TASK_ON_DAY KheDrsTaskOnDayMake(KHE_DRS_TASK encl_dt,            */
/*    KHE_DRS_DAY day, KHE_TASK task, KHE_TIME time,                         */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new task on day object with these attributes.                     */
/*  This function assumes that the task is not assigned anything.            */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_TASK_ON_DAY KheDrsTaskOnDayMake(KHE_DRS_TASK encl_dt,
  KHE_DRS_DAY day, KHE_TASK task, KHE_TIME time,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_TASK_ON_DAY res;
  HnAssert(task != NULL && time != NULL, "KheDrsTaskOnDayMake internal error");
  HaMake(res, drs->arena);
  res->encl_dt = encl_dt;
  res->day = day;
  res->task = task;
  res->time = time;
  res->closed_asst = NULL;
  HaArrayInit(res->external_today, drs->arena);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsTaskOnDayCmp(const void *t1, const void *t2)                   */
/*                                                                           */
/*  Comparison function for sorting an array of task on day objects by       */
/*  increasing day index.                                                    */
/*                                                                           */
/*****************************************************************************/

static int KheDrsTaskOnDayCmp(const void *t1, const void *t2)
{
  KHE_DRS_TASK_ON_DAY dtd1, dtd2;
  dtd1 = * (KHE_DRS_TASK_ON_DAY *) t1;
  dtd2 = * (KHE_DRS_TASK_ON_DAY *) t2;
  return dtd1->day->frame_index - dtd2->day->frame_index;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsOptionalResourceGroupEqual(KHE_RESOURCE_GROUP rg1,            */
/*    KHE_RESOURCE_GROUP rg2)                                                */
/*                                                                           */
/*  Return true if rg1 and rg2 are both NULL, or both non-NULL and equal.    */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsOptionalResourceGroupEqual(KHE_RESOURCE_GROUP rg1,
  KHE_RESOURCE_GROUP rg2)
{
  if( rg1 == NULL )
    return rg2 == NULL;
  else
    return rg2 != NULL && KheResourceGroupEqual(rg1, rg2);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsTaskOnDayHasAssignedTaskExpr(KHE_DRS_TASK_ON_DAY dtd,         */
/*    KHE_RESOURCE_GROUP rg, KHE_DRS_EXPR_ASSIGNED_TASK *res)                */
/*                                                                           */
/*  If dtd has an assigned task expression with resource group attribute     */
/*  rg, return true with *res set to that expression; else return false.     */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsTaskOnDayHasAssignedTaskExpr(KHE_DRS_TASK_ON_DAY dtd,
  KHE_RESOURCE_GROUP rg, KHE_DRS_EXPR_ASSIGNED_TASK *res)
{
  KHE_DRS_EXPR e;  int i;  KHE_DRS_EXPR_ASSIGNED_TASK eat;
  HaArrayForEach(dtd->external_today, e, i)
  {
    eat = (KHE_DRS_EXPR_ASSIGNED_TASK) e;
    if( KheDrsOptionalResourceGroupEqual(rg, eat->resource_group) )
      return *res = eat, true;
  }
  return *res = NULL, false;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsTaskOnDayAddAssignedTaskExpr(KHE_DRS_TASK_ON_DAY dtd,         */
/*    KHE_DRS_EXPR_ASSIGNED_TASK eat)                                        */
/*                                                                           */
/*  Add eat (assumed new) to dtd.                                            */
/*                                                                           */
/*****************************************************************************/

static void KheDrsTaskOnDayAddAssignedTaskExpr(KHE_DRS_TASK_ON_DAY dtd,
  KHE_DRS_EXPR_ASSIGNED_TASK eat)
{
  HaArrayAddLast(dtd->external_today, (KHE_DRS_EXPR) eat);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsTaskOnDayIsLast(KHE_DRS_TASK_ON_DAY dtd)                      */
/*                                                                           */
/*  Return true if dtd is the last task on day of its task.                  */
/*  This function is used only when debugging.                               */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsTaskOnDayIsLast(KHE_DRS_TASK_ON_DAY dtd)
{
  return dtd == HaArrayLast(dtd->encl_dt->days);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsTaskOnDayLeafSet(KHE_DRS_TASK_ON_DAY dtd,                     */
/*    KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs)                  */
/*                                                                           */
/*  Call KheDrsExprLeafSet for each external expression affected by dtd.     */
/*  Or do nothing if dtd is NULL.                                            */
/*                                                                           */
/*****************************************************************************/

static void KheDrsTaskOnDayLeafSet(KHE_DRS_TASK_ON_DAY dtd,
  KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR e;  int i;
  if( dtd != NULL )
    HaArrayForEach(dtd->external_today, e, i)
      KheDrsExprLeafSet(e, dtd, dr, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsTaskOnDayLeafClear(KHE_DRS_TASK_ON_DAY dtd,                   */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Call KheDrsExprLeafClear for each external expression affected by dtd.   */
/*  Or do nothing if dtd is NULL.                                            */
/*                                                                           */
/*****************************************************************************/

static void KheDrsTaskOnDayLeafClear(KHE_DRS_TASK_ON_DAY dtd,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR e;  int i;
  if( dtd != NULL )
    HaArrayForEach(dtd->external_today, e, i)
      KheDrsExprLeafClear(e, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsTaskOnDayDebug(KHE_DRS_TASK_ON_DAY dtd,                       */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of dtd onto fp with the given verbosity and indent.          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsTaskOnDayDebug(KHE_DRS_TASK_ON_DAY dtd,
  int verbosity, int indent, FILE *fp)
{
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  KheTaskDebug(dtd->task, 1, -1, fp);
  if( verbosity > 1 )
    fprintf(fp, "(time %s, day %s, closed_asst %s)", KheTimeId(dtd->time),
      KheDrsDayId(dtd->day), dtd->closed_asst == NULL ? "-" :
      KheDrsResourceId(dtd->closed_asst->encl_dr));
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_TASK_CLASS"                                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsTaskClassMake(KHE_TASK_CLASS tc,                              */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new drs task class based on tc and add it to its first day.       */
/*  Also make drs tasks corresponding to tc's tasks and add them to the      */
/*  drs class task and to drs->all_root_tasks.                               */
/*                                                                           */
/*  Return true if all this is done successfully.  The only reason why       */
/*  false is returned is that creating one of the tasks uncovered a case     */
/*  where a resource is assigned two tasks on one day.                       */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsTaskClassMake(KHE_TASK_CLASS tc,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_TASK_CLASS res;  int i;  KHE_TASK task;  KHE_DRS_TASK dt;
  KHE_DRS_DAY day;  KHE_COST asst_cost, non_asst_cost;

  /* get or make a drs task class object */
  HaMake(res, drs->arena);
  res->orig_task_class = tc;
  res->encl_shift = NULL;  /* reset later */
  res->day_range = KheDrsDayRangeMake(KheTaskClassFirstDayIndex(tc),
    KheTaskClassLastDayIndex(tc));
  HaArrayInit(res->all_tasks, drs->arena);
  HaArrayInit(res->unassigned_tasks, drs->arena);
  res->expand_prev_unfixed = -1;

  /* make drs tasks for its khe tasks and add them */
  for( i = 0;  i < KheTaskClassTaskCount(tc);  i++ )
  {
    task = KheTaskClassTask(tc, i, &asst_cost, &non_asst_cost);
    if( !KheDrsTaskMake(res, i, task, asst_cost, non_asst_cost, drs, &dt) )
      return false;
    HaArrayAddLast(res->all_tasks, dt);
    if( dt->closed_asst == NULL )
      HaArrayAddLast(res->unassigned_tasks, dt);
  }

  /* make res one of the task classes of its first day */
  day = HaArray(drs->all_days, res->day_range.first);
  KheDrsDayAddTaskClass(day, res, drs);
  if( DEBUG50 && res->day_range.first < res->day_range.last )
    KheDrsTaskClassDebug(res, drs, 2, 2, stderr);
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsTaskClassOrganizeUnassignedTasks(KHE_DRS_TASK_CLASS dtc)      */
/*                                                                           */
/*  Organize the unassigned tasks of dtc for solving.                        */
/*                                                                           */
/*  When this function is called, unassigned_tasks may be out of order,      */
/*  and, because unassigned_tasks is handled lazily, some of them may in     */
/*  fact be assigned.  This function fixes both of those problems.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsTaskClassOrganizeUnassignedTasks(KHE_DRS_TASK_CLASS dtc)
{
  KHE_DRS_TASK dt;

  /* sort unassigned_tasks into order with assigned tasks at the end */
  HaArraySortUnique(dtc->unassigned_tasks, &KheDrsTaskIndexInEnclCmp);

  /* remove assigned tasks from the end */
  while( HaArrayCount(dtc->unassigned_tasks) > 0 )
  {
    dt = HaArrayLast(dtc->unassigned_tasks);
    if( dt->closed_asst == NULL )
      break;
    HaArrayDeleteLast(dtc->unassigned_tasks);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsTaskClassOpen(KHE_DRS_TASK_CLASS dtc, KHE_DRS_DAY_RANGE ddr,  */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  If dtc is suitable (if its day range lies within ddr, and its resource   */
/*  domain includes at least one open resource), then open it.  Opening dtc  */
/*  basically means organizing its unassigned tasks into decreasing          */
/*  desirability order and opening them.                                     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsTaskClassOpen(KHE_DRS_TASK_CLASS dtc, KHE_DRS_DAY_RANGE ddr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_TASK dt;  int i;
  if( DEBUG19 )
    fprintf(stderr, "[ KheDrsTaskClassOpen(dtc, %d-%d)\n", ddr.first, ddr.last);
  if( KheDrsDayRangeSubset(dtc->day_range, ddr) &&
      !KheResourceSetDisjointGroup(drs->selected_resource_set,
	KheTaskClassDomain(dtc->orig_task_class)) )
  {
    /* dtc can open; organize and open unassigned_tasks; say none are used */
    KheDrsTaskClassOrganizeUnassignedTasks(dtc);
    HaArrayForEach(dtc->unassigned_tasks, dt, i)
      KheDrsTaskOpen(dt, drs);
    dtc->expand_prev_unfixed = -1;
  }
  else
  {
    /* dtc can't open; set dtc->expand_used to make that clear */
    dtc->expand_prev_unfixed = HaArrayCount(dtc->unassigned_tasks) - 1;
  }
  if( DEBUG19 )
    fprintf(stderr, "] KheDrsTaskClassOpen\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsTaskClassClose(KHE_DRS_TASK_CLASS dtc,                        */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Close dtc.  Some of its tasks may be already closed; that's OK.          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsTaskClassClose(KHE_DRS_TASK_CLASS dtc)
{
  KHE_DRS_TASK dt;  int i;
  HaArrayForEach(dtc->unassigned_tasks, dt, i)
    KheDrsTaskClose(dt, NULL);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsTaskClassDebug(KHE_DRS_TASK_CLASS dtc, int verbosity,         */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)  */
/*                                                                           */
/*  Debug print of dtc onto fp with the given verbosity and indent.          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsTaskClassDebug(KHE_DRS_TASK_CLASS dtc,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)
{
  KHE_DRS_TASK dt;  int i, pos;
  if( indent >= 0 )
  {
    fprintf(fp, "%*s[ Task class ", indent, "");
    if( dtc == NULL )
      fprintf(fp, "(free) ]\n");
    else
    {
      if( drs != NULL )
	KheDrsDayRangeDebug(dtc->day_range, drs, 1, 0, fp);
      HaArrayForEach(dtc->all_tasks, dt, i)
      {
	fprintf(fp, "%*s  ", indent, "");
	KheDrsTaskDebug(dt, verbosity, -1, fp);
	fprintf(fp, " %s%s\n", HaArrayContains(dtc->unassigned_tasks, dt,&pos) ?
	  "" : "assigned ", KheDrsTaskExpandRoleShow(dt->expand_role));
      }
      fprintf(fp, "%*s]\n", indent, "");
    }
  }
  else
  {
    if( dtc == NULL )
      fprintf(fp, "{free}");
    else
    {
      fprintf(fp, "{");
      if( HaArrayCount(dtc->all_tasks) >= 2 )
      {
	KheDrsTaskDebug(HaArrayFirst(dtc->all_tasks), 1, -1, fp);
	fprintf(fp, "...(%d)", HaArrayCount(dtc->all_tasks));
	/* KheDrsTaskDebug(HaArrayLast(dtc->all_tasks), 1, -1, fp); */
      }
      else
	KheDrsTaskDebug(HaArrayFirst(dtc->all_tasks), 1, -1, fp);
      fprintf(fp, "}");
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_TASK_CLASS - expansion"                               */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheDrsTaskClassExpandBegin(KHE_DRS_TASK_CLASS dtc,                  */
/*    KHE_DRS_SOLN prev_soln, KHE_DRS_EXPANDER de)                           */
/*                                                                           */
/*  Begin an expansion involving task class dtc.                             */
/*                                                                           */
/*****************************************************************************/
static void KheDrsExpanderAddMustAssign(KHE_DRS_EXPANDER de);

static void KheDrsTaskClassExpandBegin(KHE_DRS_TASK_CLASS dtc,
  KHE_DRS_SOLN prev_soln, KHE_DRS_EXPANDER de)
{
  KHE_DRS_TASK dt;  int i;
  HaArrayForEach(dtc->unassigned_tasks, dt, i)
    if( dt->expand_role != KHE_DRS_TASK_EXPAND_FIXED )
    {
      if( KheDrsExpanderOpenToExtraCost(de, dt->non_asst_cost) )
	dt->expand_role = KHE_DRS_TASK_EXPAND_FREE;
      else
      {
	/* dt must be assigned, otherwise cost will be too high */
	dt->expand_role = KHE_DRS_TASK_EXPAND_MUST;
	dtc->encl_shift->expand_must_assign_count++;
	KheDrsExpanderAddMustAssign(de);
      }
    }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsTaskClassExpandEnd(KHE_DRS_TASK_CLASS dtc,                    */
/*    KHE_DRS_EXPANDER de)                                                   */
/*                                                                           */
/*  End an expansion involving task class dtc.                               */
/*                                                                           */
/*****************************************************************************/

static void KheDrsTaskClassExpandEnd(KHE_DRS_TASK_CLASS dtc,
  KHE_DRS_EXPANDER de)
{
  KHE_DRS_TASK dt;  int i;
  HaArrayForEach(dtc->unassigned_tasks, dt, i)
    dt->expand_role = KHE_DRS_TASK_EXPAND_NO;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsTaskClassAcceptResourceBegin(KHE_DRS_TASK_CLASS dtc,          */
/*    KHE_DRS_RESOURCE_ON_DAY drd, KHE_DRS_TASK_ON_DAY *dtd)                 */
/*                                                                           */
/*  If dtc accepts drd (because drd lies in its tasks' domains, and not all  */
/*  the tasks are assigned), then return true and set *dtd to the relevant   */
/*  task on day of one of dtc's tasks that drd can be assigned to.  Else     */
/*  return false.                                                            */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsTaskClassAcceptResourceBegin(KHE_DRS_TASK_CLASS dtc,
  KHE_DRS_RESOURCE_ON_DAY drd, KHE_DRS_TASK_ON_DAY *dtd)
{
  KHE_DRS_TASK dt;  int i, count;
  if( KheResourceGroupContains(KheTaskClassDomain(dtc->orig_task_class),
    drd->encl_dr->resource) )
  {
    /* search forwards for next unfixed task */
    count = HaArrayCount(dtc->unassigned_tasks);
    for( i = dtc->expand_prev_unfixed + 1;  i < count;  i++ )
    {
      dt = HaArray(dtc->unassigned_tasks, i);
      if( dt->expand_role != KHE_DRS_TASK_EXPAND_FIXED )
      {
	if( !KheDrsTaskBusyOnDay(dt, drd->day, dtd) )
	  HnAbort("KheDrsTaskClassAcceptResourceBegin internal error");
        dtc->expand_prev_unfixed = i;
	return true;
      }
    }
  }

  /* if we get here we've failed to identify a suitable task */
  return *dtd = NULL, false;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsTaskClassAcceptResourceEnd(KHE_DRS_TASK_CLASS dtc,            */
/*    KHE_DRS_TASK_ON_DAY dtd)                                               */
/*                                                                           */
/*  Undo the effect of a corresponding KheDrsTaskClassAcceptResourceBegin    */
/*  which returned true.                                                     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsTaskClassAcceptResourceEnd(KHE_DRS_TASK_CLASS dtc,
  KHE_DRS_TASK_ON_DAY dtd)
{
  KHE_DRS_TASK dt;  int i;
  HnAssert(dtc->expand_prev_unfixed < HaArrayCount(dtc->unassigned_tasks) &&
    dtd->encl_dt == HaArray(dtc->unassigned_tasks, dtc->expand_prev_unfixed),
    "KheDrsTaskClassAcceptResourceEnd internal error");

  /* search backwards for next unfixed task, or start of array */
  for( i = dtc->expand_prev_unfixed - 1;  i >= 0;  i-- )
  {
    dt = HaArray(dtc->unassigned_tasks, dtc->expand_prev_unfixed);
    if( dt->expand_role != KHE_DRS_TASK_EXPAND_FIXED )
      break;
  }
  dtc->expand_prev_unfixed = i;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_TASK_CLASS - one-extra and two-extra selection"       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsTaskClassOneExtraAvailable(KHE_DRS_TASK_CLASS dtc,            */
/*    int open_resource_count)                                               */
/*                                                                           */
/*  Return true if dtc is available for participating in a one-extra         */
/*  dominance test.  Parameter open_resource_count is the number of open     */
/*  resources.                                                               */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsTaskClassOneExtraAvailable(KHE_DRS_TASK_CLASS dtc,
  int open_resource_count)
{
  return dtc == NULL ||
    open_resource_count - 1 < HaArrayCount(dtc->unassigned_tasks);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsTaskClassMinCost(KHE_DRS_TASK_CLASS dtc, KHE_DRS_ASST_OP op,  */
/*    KHE_DRS_RESOURCE dra, KHE_DRS_RESOURCE drb, int open_resource_count,   */
/*    KHE_COST *avail_cost, int verbosity, int indent, FILE *fp)             */
/*                                                                           */
/*  This function evaluates three expressions from the documentation:        */
/*                                                                           */
/*    op                           function                                  */
/*    ---------------------------------------------------------------        */
/*    KHE_DRS_ASST_OP_UNASSIGN     min(p) c-(M, ra)                          */
/*    KHE_DRS_ASST_OP_ASSIGN       min(q) c+(M, ra)                          */
/*    KHE_DRS_ASST_OP_REPLACE      min(p) c-+(M, ra, rb)                     */
/*    ---------------------------------------------------------------        */
/*                                                                           */
/*  But instead of returning the value it adds it to *avail_cost.  The       */
/*  return value is true if the function is able to calculate this cost,     */
/*  and false if not.                                                        */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsTaskClassMinCost(KHE_DRS_TASK_CLASS dtc, KHE_DRS_ASST_OP op,
  KHE_DRS_RESOURCE dra, KHE_DRS_RESOURCE drb, int open_resource_count,
  KHE_COST *avail_cost, int verbosity, int indent, FILE *fp)
{
  int p, max_p;  KHE_DRS_TASK dt;  KHE_COST cost, min_cost;

  if( fp != NULL )
  {
    fprintf(stderr, "[ KheDrsTaskClassMinCost(");
    if( dtc == NULL )
      fprintf(stderr, "(free)");
    else
      KheDrsTaskClassDebug(dtc, NULL, 1, -1, stderr);
    fprintf(stderr, ", op %d, dra %s, drb %s, count %d, &avail_cost)\n",
      (int) op, dra == NULL ? "-" : KheDrsResourceId(dra),
      drb == NULL ? "-" : KheDrsResourceId(drb), open_resource_count);
  }

  /* if dtc is NULL (represents a free day) there is no change to *avail_cost */
  if( dtc == NULL )
  {
    if( fp != NULL )
      fprintf(stderr, "] KheDrsTaskClassMinCost returning true (free)\n");
    return true;
  }

  /* find the minimum, over unassigned tasks 0 ... max_p, of the cost */
  max_p = min(open_resource_count - 1, HaArrayCount(dtc->unassigned_tasks) - 1);
  min_cost = KheCost(INT_MAX, INT_MAX);
  for( p = 0;  p <= max_p;  p++ )
  {
    if( fp != NULL )
      fprintf(stderr, "  p = %d\n", p);
    dt = HaArray(dtc->unassigned_tasks, p);   /* the (p+1)st unassigned task */
    if( !KheDrsTaskMinCost(dt, op, dra, drb, &cost) )
    {
      if( fp != NULL )
	fprintf(stderr, "] KheDrsTaskClassMinCost returning false\n");
      return false;
    }
    if( cost < min_cost )
      min_cost = cost;
  }

  /* wrapup */
  HnAssert(min_cost < KheCost(INT_MAX, INT_MAX),
    "KheDrsTaskClassMinCost internal error");
  if( fp != NULL )
    fprintf(stderr, "] KheDrsTaskClassMinCost returning true (min_cost %.5f)\n",
      KheCostShow(min_cost));
  *avail_cost += min_cost;
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_SHIFT"                                                */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SHIFT KheDrsShiftMake(KHE_DRS_DAY day, KHE_DRS_TASK_CLASS dtc,   */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new shift containing just dtc.                                    */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_SHIFT KheDrsShiftMake(KHE_DRS_DAY day, KHE_DRS_TASK_CLASS dtc,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SHIFT res;
  HaMake(res, drs->arena);
  res->encl_day = day;
  res->open_shift_index = -1;
  res->expand_must_assign_count = 0;
  res->expand_max_included_free_resource_count = 0;
  HaArrayInit(res->task_classes, drs->arena);
  res->signer = KheDrsSignerMake(NULL, NULL, res, drs);
  HaArrayAddLast(res->task_classes, dtc);
  res->asst_trie = NULL;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheDrsShiftId(KHE_DRS_SHIFT ds)                                    */
/*                                                                           */
/*  Return an Id for shift.                                                  */
/*                                                                           */
/*****************************************************************************/

static char *KheDrsShiftId(KHE_DRS_SHIFT ds)
{
  static char buff[200];
  if( ds->open_shift_index >= 0 )
    snprintf(buff, 200, "%s:%d", KheDrsDayId(ds->encl_day),
      ds->open_shift_index);
  else
    snprintf(buff, 200, "%s:?", KheDrsDayId(ds->encl_day));
  return buff;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsShiftAcceptsClass(KHE_DRS_SHIFT ds, KHE_DRS_TASK_CLASS dtc)   */
/*                                                                           */
/*  If ds can accept dtc, because ds's classes have the same times and       */
/*  the same total workload as dtc, then add dtc to ds and return true.      */
/*  Otherwise change nothing and return false.                               */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsShiftAcceptsClass(KHE_DRS_SHIFT ds, KHE_DRS_TASK_CLASS dtc)
{
  KHE_DRS_TASK_CLASS dtc2;  int first_day_index, last_day_index, day_index;
  KHE_TASK_CLASS tc1, tc2;  KHE_TIME time1, time2;  float workload1, workload2;

  /* get the two task classes, tc1 and tc2, that have to be compared */
  dtc2 = HaArrayFirst(ds->task_classes);
  tc1 = dtc->orig_task_class;
  tc2 = dtc2->orig_task_class;

  /* first_day_index and last_day_index must be the same */
  first_day_index = KheTaskClassFirstDayIndex(tc1);
  if( KheTaskClassFirstDayIndex(tc2) != first_day_index )
    return false;
  last_day_index = KheTaskClassLastDayIndex(tc1);
  if( KheTaskClassLastDayIndex(tc2) != last_day_index )
    return false;

  /* busy times and workloads on each day must be the same */
  for( day_index = first_day_index;  day_index <= last_day_index;  day_index++ )
  {
    time1 = KheTaskClassDayTime(tc1, day_index, &workload1);
    time2 = KheTaskClassDayTime(tc2, day_index, &workload2);
    if( time1 != time2 || workload1 != workload2 )
      return false;
  }

  /* all good, add dtc to ds and return true */
  HaArrayAddLast(ds->task_classes, dtc);
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftOpen(KHE_DRS_SHIFT ds, KHE_DRS_DAY_RANGE ddr,            */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Open ds and its task classes.                                            */
/*                                                                           */
/*****************************************************************************/

static void KheDrsShiftOpen(KHE_DRS_SHIFT ds, KHE_DRS_DAY_RANGE ddr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_TASK_CLASS dtc;  int i;
  ds->open_shift_index = HaArrayCount(drs->open_shifts);
  HaArrayAddLast(drs->open_shifts, ds);
  HaArrayForEach(ds->task_classes, dtc, i)
    KheDrsTaskClassOpen(dtc, ddr, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftClose(KHE_DRS_SHIFT ds, KHE_DYNAMIC_RESOURCE_SOLVER drs) */
/*                                                                           */
/*  Close ds and its task classes.                                           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsShiftClose(KHE_DRS_SHIFT ds, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_TASK_CLASS dtc;  int i;
  HaArrayForEach(ds->task_classes, dtc, i)
    KheDrsTaskClassClose(dtc);
  ds->open_shift_index = -1;
  KheDrsSignerClear(ds->signer, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftAddOpenExpr(KHE_DRS_SHIFT ds, KHE_DRS_EXPR e)            */
/*                                                                           */
/*  Add open expression e to ds.                                             */
/*                                                                           */
/*****************************************************************************/

static void KheDrsShiftAddOpenExpr(KHE_DRS_SHIFT ds, KHE_DRS_EXPR e)
{
  KheDrsSignerAddOpenExpr(ds->signer, e);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftAddDomTest(KHE_DRS_SHIFT ds, KHE_DRS_DOM_TEST dom_test)  */
/*                                                                           */
/*  Add dom_test to ds.                                                      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsShiftAddDomTest(KHE_DRS_SHIFT ds, KHE_DRS_DOM_TEST dom_test,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KheDrsSignerAddDomTest(ds->signer, dom_test, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_SHIFT - expansion"                                    */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftExpandBegin(KHE_DRS_SHIFT ds, KHE_DRS_SOLN prev_soln,    */
/*    KHE_DRS_EXPANDER de)                                                   */
/*                                                                           */
/*  Initialize ds and its task classes and tasks for expansion of            */
/*  prev_soln.  Only don't initialize the assignment trie yet.               */
/*                                                                           */
/*****************************************************************************/

static void KheDrsShiftExpandBegin(KHE_DRS_SHIFT ds, KHE_DRS_SOLN prev_soln,
  KHE_DRS_EXPANDER de)
{
  KHE_DRS_TASK_CLASS dtc;  int i;
  ds->expand_must_assign_count = 0;
  ds->expand_max_included_free_resource_count = 0;
  HaArrayForEach(ds->task_classes, dtc, i)
    KheDrsTaskClassExpandBegin(dtc, prev_soln, de);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftExpandEnd(KHE_DRS_SHIFT ds, KHE_DRS_EXPANDER de)         */
/*                                                                           */
/*  An expansion involving this shift is ending; clear it out.               */
/*                                                                           */
/*****************************************************************************/
static void KheDrsShiftAsstTrieFree(KHE_DRS_SHIFT_ASST_TRIE dsat,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsShiftExpandEnd(KHE_DRS_SHIFT ds, KHE_DRS_EXPANDER de)
{
  KHE_DRS_TASK_CLASS dtc;  int i;
  HaArrayForEach(ds->task_classes, dtc, i)
    KheDrsTaskClassExpandEnd(dtc, de);
  KheDrsShiftAsstTrieFree(ds->asst_trie, de->solver);
  ds->asst_trie = NULL;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftBuildShiftAsstTrie(KHE_DRS_SHIFT ds,                     */
/*    KHE_DRS_SOLN prev_soln, KHE_DRS_DAY next_day,                          */
/*    KHE_DRS_RESOURCE_SET free_resources, KHE_DYNAMIC_RESOURCE_SOLVER drs)  */
/*                                                                           */
/*  Initialize the assignment trie of ds.                                    */
/*                                                                           */
/*****************************************************************************/
static KHE_DRS_SHIFT_ASST_TRIE KheDrsShiftAsstTrieBuild(KHE_DRS_SHIFT ds,
  KHE_DRS_SOLN prev_soln, KHE_DRS_DAY prev_day, KHE_DRS_DAY next_day,
  KHE_DRS_RESOURCE_SET all_free_resources, int all_free_resources_index,
  KHE_DRS_RESOURCE_SET included_free_resources,
  KHE_DRS_ASST_TO_TASK_SET included_fixed_assts, KHE_DRS_EXPANDER de);

static void KheDrsShiftBuildShiftAsstTrie(KHE_DRS_SHIFT ds,
  KHE_DRS_SOLN prev_soln, KHE_DRS_DAY prev_day, KHE_DRS_DAY next_day,
  KHE_DRS_RESOURCE_SET all_free_resources,
  KHE_DRS_ASST_TO_TASK_SET included_fixed_assts,
  KHE_DRS_EXPANDER de, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_RESOURCE_SET included_free_resources;
  HnAssert(ds->asst_trie == NULL, "KheDrsShiftInitAsstTrie internal error");
  included_free_resources = KheDrsResourceSetMake(drs);
  ds->asst_trie = KheDrsShiftAsstTrieBuild(ds, prev_soln, prev_day, next_day,
    all_free_resources, 0, included_free_resources, included_fixed_assts, de);
  KheDrsResourceSetFree(included_free_resources, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftExpandByShifts(KHE_DRS_SHIFT ds, int shift_index,        */
/*    KHE_DRS_SOLN prev_soln, KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de,     */
/*    KHE_DRS_RESOURCE_SET free_resources)                                   */
/*                                                                           */
/*  Expand shift ds and beyond using free_resources.                         */
/*                                                                           */
/*****************************************************************************/
static void KheDrsShiftAsstTrieExpandByShifts(KHE_DRS_SHIFT_ASST_TRIE dsat,
  KHE_DRS_SHIFT ds, int shift_index, KHE_DRS_SOLN prev_soln,
  KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de,
  KHE_DRS_RESOURCE_SET free_resources, int avail_index,
  int selected_resource_count, KHE_DRS_RESOURCE_SET omitted_resources);

static void KheDrsShiftExpandByShifts(KHE_DRS_SHIFT ds, int shift_index,
  KHE_DRS_SOLN prev_soln, KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de,
  KHE_DRS_RESOURCE_SET free_resources)
{
  KHE_DRS_RESOURCE_SET omitted_resources;
  if( ds->asst_trie != NULL )
  {
    omitted_resources = KheDrsResourceSetMake(de->solver);
    KheDrsShiftAsstTrieExpandByShifts(ds->asst_trie, ds, shift_index,
      prev_soln, next_day, de, free_resources, 0, 0, omitted_resources);
    KheDrsResourceSetFree(omitted_resources, de->solver);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftDebug(KHE_DRS_SHIFT ds, KHE_DYNAMIC_RESOURCE_SOLVER drs, */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of shift ds onto fp with the given verbosity and indent.     */
/*  Here drs may be NULL, although the result will be more readable if it    */
/*  is non-NULL.                                                             */
/*                                                                           */
/*****************************************************************************/
static bool KheDrsShiftAsstTrieResourceIsForced(KHE_DRS_SHIFT_ASST_TRIE dsat,
  KHE_DRS_RESOURCE dr);
static void KheDrsShiftAsstTrieDebug(KHE_DRS_SHIFT_ASST_TRIE dsat,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp);

static void KheDrsShiftDebug(KHE_DRS_SHIFT ds, KHE_DYNAMIC_RESOURCE_SOLVER drs,
  int verbosity, int indent, FILE *fp)
{
  KHE_DRS_TASK_CLASS dtc;  KHE_DRS_RESOURCE dr;  int i;
  if( indent >= 0 )
  {
    fprintf(fp, "%*s[ Shift %d (expand_min %d, expand_max %d)\n", indent, "",
      ds->open_shift_index, ds->expand_must_assign_count,
      ds->expand_max_included_free_resource_count);
    HaArrayForEach(ds->task_classes, dtc, i)
      KheDrsTaskClassDebug(dtc, drs, verbosity, indent + 2, fp);
    if( verbosity >= 2 && ds->asst_trie != NULL )
    {
      KheDrsShiftAsstTrieDebug(ds->asst_trie, drs, verbosity-1, indent+2, fp);
      KheDrsResourceSetForEach(drs->open_resources, dr, i)
	if( KheDrsShiftAsstTrieResourceIsForced(ds->asst_trie, dr) )
	  fprintf(fp, "%*s  forced resource %s\n", indent, "",
	    KheDrsResourceId(dr));
    }
    fprintf(fp, "%*s]\n", indent, "");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_SHIFT_ASST"                                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SHIFT_ASST KheDrsShiftAsstMake(KHE_DYNAMIC_RESOURCE_SOLVER drs)  */
/*                                                                           */
/*  Make a new shift assignment object.                                      */
/*                                                                           */
/*****************************************************************************/
static KHE_DRS_SIGNATURE KheDrsSignatureMake(KHE_DYNAMIC_RESOURCE_SOLVER drs);
static void KheDrsAsstToTaskSetClear(KHE_DRS_ASST_TO_TASK_SET dats);

static KHE_DRS_ASST_TO_TASK_SET KheDrsAsstToTaskSetMake(
  KHE_DYNAMIC_RESOURCE_SOLVER drs);

static KHE_DRS_SHIFT_ASST KheDrsShiftAsstMake(KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SHIFT_ASST res;

  /* get or make the object */
  if( HaArrayCount(drs->shift_asst_free_list) > 0 )
  {
    res = HaArrayLastAndDelete(drs->shift_asst_free_list);
    KheDrsAsstToTaskSetClear(res->assts_to_tasks);
  }
  else
  {
    HaMake(res, drs->arena);
    res->assts_to_tasks = KheDrsAsstToTaskSetMake(drs);
  }

  /* all good, make sig and return */
  res->sig = KheDrsSignatureMake(drs);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftAsstFree(KHE_DRS_SHIFT_ASST dsa,                         */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Free shift assignment dsa.                                               */
/*                                                                           */
/*****************************************************************************/
static void KheDrsSignatureUnRefer(KHE_DRS_SIGNATURE sig,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsShiftAsstFree(KHE_DRS_SHIFT_ASST dsa,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KheDrsSignatureUnRefer(dsa->sig, drs);
  dsa->sig = NULL;
  HaArrayAddLast(drs->shift_asst_free_list, dsa);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsShiftAsstDominates(KHE_DRS_SHIFT_ASST dsa1,                   */
/*    KHE_DRS_SHIFT_ASST dsa2, KHE_DRS_SIGNER dsg,                           */
/*    KHE_DRS_DOM_TEST_TYPE dom_test_type, KHE_DYNAMIC_RESOURCE_SOLVER drs)  */
/*                                                                           */
/*  Return true if dsa1 dominates dsa2.                                      */
/*                                                                           */
/*****************************************************************************/
static bool KheDrsSignerDominates(KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE sig1,
  KHE_DRS_SIGNATURE sig2, KHE_COST *avail_cost,
  int verbosity, int indent, FILE *fp);

static bool KheDrsShiftAsstDominates(KHE_DRS_SHIFT_ASST dsa1,
  KHE_DRS_SHIFT_ASST dsa2, KHE_DRS_SIGNER dsg,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_COST avail_cost;
  avail_cost = 0;
  return KheDrsSignerDominates(dsg, dsa1->sig, dsa2->sig, &avail_cost,
    0, 0, NULL);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftAsstDebug(KHE_DRS_SHIFT_ASST dsa, int verbosity,         */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of dsa onto fp with the given verbosity and indent.          */
/*                                                                           */
/*****************************************************************************/
static void KheDrsSignatureDebug(KHE_DRS_SIGNATURE sig, int verbosity,
  int indent, FILE *fp);

static void KheDrsShiftAsstDebug(KHE_DRS_SHIFT_ASST dsa, int verbosity,
  int indent, FILE *fp)
{
  int i;  KHE_DRS_ASST_TO_TASK tasst; /* KHE_DRS_TASK_ON_DAY dtd; */
  if( indent >= 0 )
    fprintf(fp, "%*s", indent, "");
  fprintf(fp, "(");
  KheDrsSignatureDebug(dsa->sig, 1, -1, fp);
  /* HaArrayForEach(dsa->task_on_days, dtd, i) */
  KheDrsAsstToTaskSetForEach(dsa->assts_to_tasks, tasst, i)
  {
    if( i > 0 )
      fprintf(fp, ", ");
    else
      fprintf(fp, " ");
    KheDrsTaskDebug(tasst.fixed_dtd->encl_dt, 1, -1, fp);
  }
  fprintf(fp, ")");
  if( indent > 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_SHIFT_ASST - expansion"                               */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftAsstExpandByShifts(KHE_DRS_SHIFT_ASST dsa,               */
/*    KHE_DRS_SHIFT ds, int shift_index, KHE_DRS_SOLN prev_soln,             */
/*    KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de,                             */
/*    KHE_DRS_RESOURCE_SET omitted_resources)                                */
/*                                                                           */
/*  Expand dsa.                                                              */
/*                                                                           */
/*****************************************************************************/
static void KheDrsSolnExpandByShifts(KHE_DRS_SOLN prev_soln,
  KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de,
  KHE_DRS_RESOURCE_SET avail_resources, int shift_index);
static void KheDrsExpanderMarkBegin(KHE_DRS_EXPANDER de);
static void KheDrsExpanderMarkEnd(KHE_DRS_EXPANDER de);
static bool KheDrsExpanderIsOpen(KHE_DRS_EXPANDER de);
static void KheDrsExpanderAddAsstToTaskSet(KHE_DRS_EXPANDER de,
  KHE_DRS_ASST_TO_TASK_SET dats, KHE_DRS_SHIFT ds);

static void KheDrsShiftAsstExpandByShifts(KHE_DRS_SHIFT_ASST dsa,
  KHE_DRS_SHIFT ds, int shift_index, KHE_DRS_SOLN prev_soln,
  KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de,
  KHE_DRS_RESOURCE_SET omitted_resources)
{
  /* save the expander */
  KheDrsExpanderMarkBegin(de);

  /* add the assignments stored in dsa to the expander */
  KheDrsExpanderAddAsstToTaskSet(de, dsa->assts_to_tasks, NULL);

  /* if the expander is still open, recurse */
  if( KheDrsExpanderIsOpen(de) )
    KheDrsSolnExpandByShifts(prev_soln, next_day, de, omitted_resources,
      shift_index + 1);

  /* restore the expander */
  KheDrsExpanderMarkEnd(de);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsShiftAsstContainsResource(KHE_DRS_SHIFT_ASST dsa,             */
/*    KHE_DRS_RESOURCE dr)                                                   */
/*                                                                           */
/*  Return true if dsa contains an assignment which assigns dr.              */
/*                                                                           */
/*****************************************************************************/
static KHE_DRS_RESOURCE KheDrsAsstToTaskResource(KHE_DRS_ASST_TO_TASK dat);

static bool KheDrsShiftAsstContainsResource(KHE_DRS_SHIFT_ASST dsa,
  KHE_DRS_RESOURCE dr)
{
  KHE_DRS_ASST_TO_TASK dat;  int i;
  KheDrsAsstToTaskSetForEach(dsa->assts_to_tasks, dat, i)
    if( KheDrsAsstToTaskResource(dat) == dr )
      return true;
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_SHIFT_ASST_TRIE" - this whole submodule is expansion  */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SHIFT_ASST_TRIE KheDrsShiftAsstTrieMake(                         */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new, empty shift asst trie.                                       */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_SHIFT_ASST_TRIE KheDrsShiftAsstTrieMake(
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SHIFT_ASST_TRIE res;
  if( HaArrayCount(drs->shift_asst_trie_free_list) > 0 )
  {
    res = HaArrayLastAndDelete(drs->shift_asst_trie_free_list);
    HaArrayClear(res->assts);
    HaArrayClear(res->children);
  }
  else
  {
    HaMake(res, drs->arena);
    HaArrayInit(res->assts, drs->arena);
    HaArrayInit(res->children, drs->arena);
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftAsstTrieFree(KHE_DRS_SHIFT_ASST_TRIE dsat,               */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Free dsat, including freeing its shift assignment objects and its        */
/*  proper descendant trie objects.                                          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsShiftAsstTrieFree(KHE_DRS_SHIFT_ASST_TRIE dsat,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SHIFT_ASST dsa;  int i;  KHE_DRS_SHIFT_ASST_TRIE child_dsat;

  if( dsat != NULL )
  {
    /* free the shift assignment objects */
    HaArrayForEach(dsat->assts, dsa, i)
      KheDrsShiftAsstFree(dsa, drs);

    /* free the proper descendant trie objects */
    HaArrayForEach(dsat->children, child_dsat, i)
      KheDrsShiftAsstTrieFree(child_dsat, drs);

    /* free dsat itself */
    HaArrayAddLast(drs->shift_asst_trie_free_list, dsat);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsShiftAsstTrieDominates(KHE_DRS_SHIFT_ASST_TRIE dsat,          */
/*    KHE_DRS_SHIFT_ASST dsa, KHE_DRS_SIGNER dsg,                            */
/*    KHE_DRS_DOM_TEST_TYPE dom_test_type, KHE_DYNAMIC_RESOURCE_SOLVER drs)  */
/*                                                                           */
/*  Return true if dsat->assts contains a shift assignment that dominates    */
/*  dsa.                                                                     */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsShiftAsstTrieDominates(KHE_DRS_SHIFT_ASST_TRIE dsat,
  KHE_DRS_SHIFT_ASST dsa, KHE_DRS_SIGNER dsg,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SHIFT_ASST other_dsa;  int i;
  HaArrayForEach(dsat->assts, other_dsa, i)
    if( KheDrsShiftAsstDominates(other_dsa, dsa, dsg, drs) )
      return true;
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftAsstTrieRemoveDominated(KHE_DRS_SHIFT_ASST_TRIE dsat,    */
/*    KHE_DRS_SHIFT_ASST dsa, KHE_DRS_SIGNER dsg,                            */
/*    KHE_DRS_DOM_TEST_TYPE dom_test_type, KHE_DYNAMIC_RESOURCE_SOLVER drs)  */
/*                                                                           */
/*  Remove from dsat->assts those shift assignments dominated by dsa.        */
/*                                                                           */
/*****************************************************************************/

static void KheDrsShiftAsstTrieRemoveDominated(KHE_DRS_SHIFT_ASST_TRIE dsat,
  KHE_DRS_SHIFT_ASST dsa, KHE_DRS_SIGNER dsg,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SHIFT_ASST other_dsa;  int i;
  HaArrayForEach(dsat->assts, other_dsa, i)
    if( KheDrsShiftAsstDominates(dsa, other_dsa, dsg, drs) )
    {
      HaArrayDeleteAndPlug(dsat->assts, i);
      KheDrsShiftAsstFree(other_dsa, drs);
      i--;
    }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftAsstTrieAddShiftAsst(KHE_DRS_SHIFT_ASST_TRIE dsat,       */
/*    KHE_DRS_SHIFT_ASST dsa)                                                */
/*                                                                           */
/*  Add dsa to dsat, without worrying about dominance testing.               */
/*                                                                           */
/*****************************************************************************/

static void KheDrsShiftAsstTrieAddShiftAsst(KHE_DRS_SHIFT_ASST_TRIE dsat,
  KHE_DRS_SHIFT_ASST dsa)
{
  HaArrayAddLast(dsat->assts, dsa);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftAsstTrieMeldShiftAsst(KHE_DRS_SHIFT_ASST_TRIE dsat,      */
/*    KHE_DRS_SHIFT_ASST dsa, KHE_DRS_SIGNER dsg,                            */
/*    KHE_DRS_DOM_TEST_TYPE dom_test_type, KHE_DYNAMIC_RESOURCE_SOLVER drs)  */
/*                                                                           */
/*  Add dsa to dsat->assts, including dominance testing.                     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsShiftAsstTrieMeldShiftAsst(KHE_DRS_SHIFT_ASST_TRIE dsat,
  KHE_DRS_SHIFT_ASST dsa, KHE_DRS_SIGNER dsg,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  if( KheDrsShiftAsstTrieDominates(dsat, dsa, dsg, drs) )
  {
    /* dsa is dominated, so free dsa */
    KheDrsShiftAsstFree(dsa, drs);
  }
  else
  {
    /* remove other assts that dsa dominates, then add dsa to dsat */
    KheDrsShiftAsstTrieRemoveDominated(dsat, dsa, dsg, drs);
    KheDrsShiftAsstTrieAddShiftAsst(dsat, dsa);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftAsstTrieBuildAssts(KHE_DRS_SHIFT_ASST_TRIE dsat,         */
/*    KHE_DRS_SHIFT ds, KHE_DRS_RESOURCE_SET included_free_resources,        */
/*    int included_free_resources_index, KHE_DRS_SOLN prev_soln,             */
/*    KHE_DRS_DAY prev_day, KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de)       */
/*                                                                           */
/*  Build all assignments for included_free_resources to the task classes    */
/*  of dsat's shift, and do dominance testing among them.  The next          */
/*  element of included_free_resources to assign is the one with index       */
/*  included_free_resources_index.                                           */
/*                                                                           */
/*****************************************************************************/
static void KheDrsExpanderMakeAndMeldShiftAsst(KHE_DRS_EXPANDER de,
  KHE_DRS_SHIFT_ASST_TRIE dsat, KHE_DRS_SHIFT ds,
  KHE_DRS_SOLN prev_soln, KHE_DRS_DAY prev_day, KHE_DRS_DAY next_day);
static void KheDrsAsstToTaskClassShiftAsstTrieBuildAssts(
  KHE_DRS_ASST_TO_TASK_CLASS asst, KHE_DRS_SHIFT_ASST_TRIE dsat,
  KHE_DRS_SHIFT ds, KHE_DRS_RESOURCE_SET included_free_resources,
  int included_free_resources_index, KHE_DRS_SOLN prev_soln,
  KHE_DRS_DAY prev_day, KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de);
static KHE_DRS_SHIFT KheDrsAsstToTaskClassShift(
  KHE_DRS_ASST_TO_TASK_CLASS asst);

static void KheDrsShiftAsstTrieBuildAssts(KHE_DRS_SHIFT_ASST_TRIE dsat,
  KHE_DRS_SHIFT ds, KHE_DRS_RESOURCE_SET included_free_resources,
  int included_free_resources_index, KHE_DRS_SOLN prev_soln,
  KHE_DRS_DAY prev_day, KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de)
{
  KHE_DRS_RESOURCE dr;  int i, indent;  KHE_DRS_ASST_TO_TASK_CLASS asst;
  if( DEBUG52(prev_day, ds) )
  {
    indent = KheDrsResourceSetCount(included_free_resources) +
      included_free_resources_index + 1;
    fprintf(stderr, "%*s[ KheDrsShiftAsstTrieBuildAssts(...)\n", indent*2, "");
  }

  if( included_free_resources_index >=
      KheDrsResourceSetCount(included_free_resources) )
  {
    /* all included resources assigned, so build soln and add it now */
    if( DEBUG52(prev_day, ds) )
    {
      indent = KheDrsResourceSetCount(included_free_resources) +
	included_free_resources_index + 1;
      fprintf(stderr, "%*s  calling KheDrsExpanderMakeAndMeldShiftAsst\n",
	indent * 2, "");
    }
    KheDrsExpanderMakeAndMeldShiftAsst(de, dsat, ds, prev_soln, prev_day,
      next_day);
  }
  else
  {
    /* try all assignments of the next included resource */
    dr = KheDrsResourceSetResource(included_free_resources,
      included_free_resources_index);
    HaArrayForEach(dr->expand_assts, asst, i)
      if( KheDrsAsstToTaskClassShift(asst) == ds )
	KheDrsAsstToTaskClassShiftAsstTrieBuildAssts(asst, dsat, ds,
	  included_free_resources, included_free_resources_index + 1,
	  prev_soln, prev_day, next_day, de);
  }
  if( DEBUG52(prev_day, ds) )
  {
    indent = KheDrsResourceSetCount(included_free_resources) +
      included_free_resources_index + 1;
    fprintf(stderr, "%*s] KheDrsShiftAsstTrieBuildAssts returning\n",
      indent * 2, "");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SHIFT_ASST_TRIE KheDrsShiftAsstTrieBuild(KHE_DRS_SHIFT ds,       */
/*    KHE_DRS_SOLN prev_soln, KHE_DRS_DAY next_day,                          */
/*    KHE_DRS_RESOURCE_SET all_free_resources, int all_free_resources_index, */
/*    KHE_DRS_RESOURCE_SET included_free_resources,                          */
/*    KHE_DRS_ASST_TO_TASK_SET included_fixed_assts, KHE_DRS_EXPANDER de)    */
/*                                                                           */
/*  Build a shift assignment trie for all_free_resources starting at         */
/*  all_free_resources_index.  Here included_free_resources contains the     */
/*  resources included so far, and included_fixed_assts holds some fixed     */
/*  assignments that are also to be included.  Return NULL if there are      */
/*  no shift assignment objects in the resulting trie.                       */
/*                                                                           */
/*  Here de is a scratch expander which is reset before each use.            */
/*                                                                           */
/*****************************************************************************/
static void KheDrsExpanderReset(KHE_DRS_EXPANDER de, bool whole_tasks,
  KHE_COST cost, KHE_COST cost_limit, int free_resource_count,
  int must_assign_count);
static void KheDrsAsstToTaskSetDebug(KHE_DRS_ASST_TO_TASK_SET dats,
  int verbosity, int indent, FILE *fp);
static KHE_COST KheDrsSolnCost(KHE_DRS_SOLN soln);

static KHE_DRS_SHIFT_ASST_TRIE KheDrsShiftAsstTrieBuild(KHE_DRS_SHIFT ds,
  KHE_DRS_SOLN prev_soln, KHE_DRS_DAY prev_day, KHE_DRS_DAY next_day,
  KHE_DRS_RESOURCE_SET all_free_resources, int all_free_resources_index,
  KHE_DRS_RESOURCE_SET included_free_resources,
  KHE_DRS_ASST_TO_TASK_SET included_fixed_assts, KHE_DRS_EXPANDER de)
{
  KHE_DRS_SHIFT_ASST_TRIE res, child_dsat;  bool no_non_null_children;
  int i, count, included_free_resource_count;  KHE_DRS_RESOURCE dr;

  included_free_resource_count =
    KheDrsResourceSetCount(included_free_resources);
  if( DEBUG52(prev_day, ds) )
  {
    fprintf(stderr, "%*s[ KheDrsShiftAsstTrieBuild(Shift %d, included_free ",
      2 * included_free_resource_count, "", ds->open_shift_index);
    KheDrsResourceSetDebug(included_free_resources, 1, -1, stderr);
    fprintf(stderr, ", included_fixed ");
    KheDrsAsstToTaskSetDebug(included_fixed_assts, 1, -1, stderr);
    fprintf(stderr, "\n");
  }

  /* return NULL immediately if too many included free resources */
  if(included_free_resource_count > ds->expand_max_included_free_resource_count)
  {
    if( DEBUG52(prev_day, ds) )
      fprintf(stderr, "%*s] KheDrsShiftAsstTrieBuild returning NULL\n",
	2 * included_free_resource_count, "");
    return NULL;
  }

  /* make res and add assignments for included_free_resources to res */
  res = KheDrsShiftAsstTrieMake(de->solver);
  if( included_free_resource_count >= ds->expand_must_assign_count )
  {
    KheDrsExpanderReset(de, true, KheDrsSolnCost(prev_soln),
      de->solver->solve_init_cost, included_free_resource_count,
      ds->expand_must_assign_count);
    if( DEBUG52(prev_day, ds) )
      fprintf(stderr, "%*sDrsExpanderReset(de, true, %.5f, %.5f, %d, %d) %s\n",
	2 * included_free_resource_count + 2, "",
	KheCostShow(KheDrsSolnCost(prev_soln)),
        KheCostShow(de->solver->solve_init_cost),
        included_free_resource_count, ds->expand_must_assign_count,
	KheDrsExpanderIsOpen(de) ? "open" : "closed");
    KheDrsExpanderMarkBegin(de);
    KheDrsExpanderAddAsstToTaskSet(de, included_fixed_assts, ds);
    if( KheDrsExpanderIsOpen(de) )
      KheDrsShiftAsstTrieBuildAssts(res, ds, included_free_resources,
	0, prev_soln, prev_day, next_day, de);
    KheDrsExpanderMarkEnd(de);
  }

  /* add a NULL child for every open resource */
  count = KheDrsResourceSetCount(de->solver->open_resources);
  HaArrayFill(res->children, count, NULL);

  /* add a potentially non-NULL child for each possible next resource */
  no_non_null_children = true;
  count = KheDrsResourceSetCount(all_free_resources);
  for( i = all_free_resources_index;  i < count;  i++ )
  {
    dr = KheDrsResourceSetResource(all_free_resources, i);
    KheDrsResourceSetAddLast(included_free_resources, dr);
    child_dsat = KheDrsShiftAsstTrieBuild(ds, prev_soln, prev_day,
      next_day, all_free_resources, i + 1, included_free_resources,
      included_fixed_assts, de);
    if( child_dsat != NULL )
    {
      HaArrayPut(res->children, dr->open_resource_index, child_dsat);
      no_non_null_children = false;
    }
    KheDrsResourceSetDeleteLast(included_free_resources);
  }

  /* replace by NULL if the tree contains no assignments */
  if( no_non_null_children && HaArrayCount(res->assts) == 0 )
  {
    KheDrsShiftAsstTrieFree(res, de->solver);
    if( DEBUG52(prev_day, ds) )
      fprintf(stderr, "%*s] KheDrsShiftAsstTrieBuild returning NULL (2)\n",
	2 * included_free_resource_count, "");
    return NULL;
  }
  else
  {
    if( DEBUG52(prev_day, ds) )
      fprintf(stderr, "%*s] KheDrsShiftAsstTrieBuild returning non-NULL\n",
	2 * included_free_resource_count, "");
    return res;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftAsstTrieExpandByShifts(KHE_DRS_SHIFT_ASST_TRIE dsat,     */
/*    KHE_DRS_SHIFT ds, int shift_index, KHE_DRS_SOLN prev_soln,             */
/*    KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de,                             */
/*    KHE_DRS_RESOURCE_SET free_resources, int free_index,                   */
/*    KHE_DRS_RESOURCE_SET omitted_resources)                                */
/*                                                                           */
/*  Expand dsat, which is non-NULL, assigning free_resources (starting       */
/*  from free_index) in all ways.  Here free_resources is not changed,       */
/*  but omitted_resources does change as the recursion proceeds; it holds    */
/*  the unused resources.                                                    */
/*                                                                           */
/*****************************************************************************/

static void KheDrsShiftAsstTrieExpandByShifts(KHE_DRS_SHIFT_ASST_TRIE dsat,
  KHE_DRS_SHIFT ds, int shift_index, KHE_DRS_SOLN prev_soln,
  KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de,
  KHE_DRS_RESOURCE_SET free_resources, int free_index,
  int selected_resource_count, KHE_DRS_RESOURCE_SET omitted_resources)
{
  int i, avail_resource_count;  KHE_DRS_SHIFT_ASST dsa;
  KHE_DRS_SHIFT_ASST_TRIE child_dsat;  KHE_DRS_RESOURCE dr;
  avail_resource_count = KheDrsResourceSetCount(free_resources) - free_index;
  if( avail_resource_count <= 0 )
  {
    /* resources all done, so try each assignment in dsat */
    HaArrayForEach(dsat->assts, dsa, i)
      KheDrsShiftAsstExpandByShifts(dsa, ds, shift_index, prev_soln,
	next_day, de, omitted_resources);
  }
  else
  {
    /* try assignments that select dr, the next available resource */
    dr = KheDrsResourceSetResource(free_resources, free_index);
    child_dsat = HaArray(dsat->children, dr->open_resource_index);
    if( child_dsat != NULL )
      KheDrsShiftAsstTrieExpandByShifts(child_dsat, ds, shift_index, prev_soln,
	next_day, de, free_resources, free_index + 1,
	selected_resource_count + 1, omitted_resources);

    /* try assignments that do not select dr, staying in dsat */
    if( selected_resource_count + avail_resource_count >
	  ds->expand_must_assign_count )
    {
      KheDrsResourceSetAddLast(omitted_resources, dr);
      KheDrsShiftAsstTrieExpandByShifts(dsat, ds, shift_index, prev_soln,
	next_day, de, free_resources, free_index + 1, selected_resource_count,
	omitted_resources);
      KheDrsResourceSetDeleteLast(omitted_resources);
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsShiftAsstTrieResourceIsForced(KHE_DRS_SHIFT_ASST_TRIE dsat,   */
/*    KHE_DRS_RESOURCE dr)                                                   */
/*                                                                           */
/*  Return true if assignment of dr to dsat is forced, that is, if every     */
/*  assignment of dsat contains dr.                                          */
/*                                                                           */
/*  Logically, the function should return true when dsat is empty, and       */
/*  indeed it does.  But in practice we are not interested in that case.     */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsShiftAsstTrieResourceIsForced(KHE_DRS_SHIFT_ASST_TRIE dsat,
  KHE_DRS_RESOURCE dr)
{
  int i;  KHE_DRS_SHIFT_ASST dsa;  KHE_DRS_SHIFT_ASST_TRIE child_dsat;

  if( dsat != NULL )
  {
    /* try the assignments */
    HaArrayForEach(dsat->assts, dsa, i)
      if( !KheDrsShiftAsstContainsResource(dsa, dr) )
	return false;

    /* try the children */
    HaArrayForEach(dsat->children, child_dsat, i)
      if( !KheDrsShiftAsstTrieResourceIsForced(child_dsat, dr) )
	return false;
  }

  /* all forced */
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftAsstTriePruneForced(KHE_DRS_SHIFT_ASST_TRIE dsat,        */
/*    KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs)                  */
/*                                                                           */
/*  Prune shift assignments that assign dr from dsat.                        */
/*                                                                           */
/*****************************************************************************/

static void KheDrsShiftAsstTriePruneForced(KHE_DRS_SHIFT_ASST_TRIE dsat,
  KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int i;  KHE_DRS_SHIFT_ASST dsa;  KHE_DRS_SHIFT_ASST_TRIE child_dsat;

  if( dsat != NULL )
  {
    /* prune all of these assignments or none */
    if( HaArrayCount(dsat->assts) > 0 )
    {
      dsa = HaArrayFirst(dsat->assts);
      if( KheDrsShiftAsstContainsResource(dsa, dr) )
      {
	/* prune them all */
	HaArrayForEach(dsat->assts, dsa, i)
	  KheDrsShiftAsstFree(dsa, drs);
	HaArrayClear(dsat->assts);
      }
    }

    /* try the children */
    HaArrayForEach(dsat->children, child_dsat, i)
      KheDrsShiftAsstTriePruneForced(child_dsat, dr, drs);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftAsstTrieDoDebug(KHE_DRS_SHIFT_ASST_TRIE dsat,            */
/*    KHE_DRS_RESOURCE_SET included_resources,                               */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)  */
/*                                                                           */
/*  Carry out the real work of KheDrsShiftAsstTrieDebug, assuming that       */
/*  indent >= 0.                                                             */
/*                                                                           */
/*****************************************************************************/

static void KheDrsShiftAsstTrieDoDebug(KHE_DRS_SHIFT_ASST_TRIE dsat,
  KHE_DRS_RESOURCE_SET included_resources,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)
{
  KHE_DRS_SHIFT_ASST dsa;  int i;  KHE_DRS_SHIFT_ASST_TRIE child_dsat;
  KHE_DRS_RESOURCE dr;
  if( dsat != NULL )
  {
    fprintf(fp, "%*s[ ", indent, "");
    KheDrsResourceSetDebug(included_resources, 1, -1, fp);
    fprintf(fp, ": {");
    HaArrayForEach(dsat->assts, dsa, i)
    {
      if( i > 0 )
	fprintf(fp, ", ");
      KheDrsShiftAsstDebug(dsa, 1, -1, fp);
    }
    fprintf(fp, "}\n");
    HaArrayForEach(dsat->children, child_dsat, i)
    {
      dr = KheDrsResourceSetResource(drs->open_resources, i);
      KheDrsResourceSetAddLast(included_resources, dr);
      KheDrsShiftAsstTrieDoDebug(child_dsat, included_resources, drs,
	verbosity, indent + 2, fp);
      KheDrsResourceSetDeleteLast(included_resources);
    }
    fprintf(fp, "%*s]\n", indent, "");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftAsstTrieDebug(KHE_DRS_SHIFT_ASST_TRIE dsat,              */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)  */
/*                                                                           */
/*  Debug print of shift asst trie dsat.                                     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsShiftAsstTrieDebug(KHE_DRS_SHIFT_ASST_TRIE dsat,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)
{
  KHE_DRS_RESOURCE_SET included_resources;
  if( indent >= 0 )
  {
    included_resources = KheDrsResourceSetMake(drs);
    fprintf(fp, "%*s[ ShiftAsstTrie\n", indent, "");
    KheDrsShiftAsstTrieDoDebug(dsat, included_resources, drs, verbosity,
      indent + 2, fp);
    fprintf(fp, "%*s]\n", indent, "");
    KheDrsResourceSetFree(included_resources, drs);
  }
  else
    fprintf(fp, "ShiftAsstTrie");
}


/*****************************************************************************/
/*                                                                           */
/*  Major category "signatures"                                              */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_SIGNATURE"                                            */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SIGNATURE KheDrsSignatureMake(KHE_DYNAMIC_RESOURCE_SOLVER drs)   */
/*                                                                           */
/*  Return a new signature object with zero cost and empty states.           */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_SIGNATURE KheDrsSignatureMake(KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SIGNATURE res;
  if( HaArrayCount(drs->signature_free_list) > 0 )
  {
    res = HaArrayLastAndDelete(drs->signature_free_list);
    HaArrayClear(res->states);
  }
  else
  {
    HaMake(res, drs->arena);
    HaArrayInit(res->states, drs->arena);
  }
  res->reference_count = 1;
  res->asst_to_shift_index = -1;  /* means none; may be reset later */
  res->cost = 0;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignatureRefer(KHE_DRS_SIGNATURE sig)                         */
/*                                                                           */
/*  Add a reference to sig.                                                  */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignatureRefer(KHE_DRS_SIGNATURE sig)
{
  sig->reference_count++;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignatureUnRefer(KHE_DRS_SIGNATURE sig,                       */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Delete a reference to sig; free it if the reference count drops to 0.    */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignatureUnRefer(KHE_DRS_SIGNATURE sig,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  HnAssert(sig->reference_count > 0, "KheDrsSignatureUnRefer internal error");
  if( --sig->reference_count == 0 )
    HaArrayAddLast(drs->signature_free_list, sig);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignatureAddAsstToShiftIndex(KHE_DRS_SIGNATURE sig, int val)  */
/*                                                                           */
/*  Add an assignment to shift index with value val to sig.                  */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignatureAddAsstToShiftIndex(KHE_DRS_SIGNATURE sig, int val)
{
  HnAssert(sig->asst_to_shift_index == -1,
    "KheDrsSignatureAddAsstToShiftIndex internal error");
  sig->asst_to_shift_index = val;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignatureClearAsstToShiftIndex(KHE_DRS_SIGNATURE sig)         */
/*                                                                           */
/*  Clear out the assignment to shift index of sig.                          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignatureClearAsstToShiftIndex(KHE_DRS_SIGNATURE sig)
{
  HnAssert(sig->asst_to_shift_index >= 0,
    "KheDrsSignatureClearAsstToShiftIndex internal error");
  sig->asst_to_shift_index = -1;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignatureAddCost(KHE_DRS_SIGNATURE sig, KHE_COST cost)        */
/*                                                                           */
/*  Add cost to sig.                                                         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignatureAddCost(KHE_DRS_SIGNATURE sig, KHE_COST cost)
{
  sig->cost += cost;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignatureAddState(KHE_DRS_SIGNATURE sig, int state,           */
/*    KHE_DRS_SIGNER dsg, KHE_DRS_EXPR e)                                    */
/*                                                                           */
/*  Add state to the end of sig.  Here dsg and e are just for checking.      */
/*                                                                           */
/*****************************************************************************/
static void KheDrsExprDebug(KHE_DRS_EXPR e, KHE_DYNAMIC_RESOURCE_SOLVER drs,
  int verbosity, int indent, FILE *fp);

static void KheDrsSignatureAddState(KHE_DRS_SIGNATURE sig, int state,
  KHE_DRS_SIGNER dsg, KHE_DRS_EXPR e)
{
  KHE_DRS_DOM_TEST dt;
  dt = HaArray(dsg->dom_tests, HaArrayCount(sig->states));
  if( dt->monitor != NULL && DEBUG44_MONITOR(dt->monitor) )
  {
    fprintf(stderr, "[ KheDrsSignatureAddState(%p, pos %d, state %d), e =\n",
      (void *) sig, HaArrayCount(sig->states), state);
    KheDrsExprDebug(e, NULL, 2, 2, stderr);
    fprintf(stderr, "]\n");
  }
  HaArrayAddLast(sig->states, state);
  HnAssert(HaArrayCount(sig->states) <= HaArrayCount(dsg->dom_tests),
    "KheDrsSignatureAddState internal error 1:  %d states > %d dom_tests",
    HaArrayCount(sig->states), HaArrayCount(dsg->dom_tests));
  if( DEBUG63 && dt->expr != e )
  {
    fprintf(stderr, "[ KheDrsSignatureAddState(%p) failing on expressions\n",
      (void *) sig);
    fprintf(stderr, "  calling expression:\n");
    KheDrsExprDebug(e, NULL, 2, 2, stderr);
    fprintf(stderr, "  dom_test expression:\n");
    KheDrsExprDebug(dt->expr, NULL, 2, 2, stderr);
    fprintf(stderr, "]\n");
  }
  if( DEBUG63 && state == 4 )
  {
    fprintf(stderr, "[ KheDrsSignatureAddState(%p) reporting state == 4\n",
      (void *) sig);
    KheDrsExprDebug(e, NULL, 2, 2, stderr);
    fprintf(stderr, "]\n");
  }
  HnAssert(dt->expr == e, "KheDrsSignatureAddState internal error 2");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignatureDebug(KHE_DRS_SIGNATURE sig, int verbosity,          */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of sig onto fp with the given verbosity and indent.          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignatureDebug(KHE_DRS_SIGNATURE sig, int verbosity,
  int indent, FILE *fp)
{
  int val, i;
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  if( HaArrayCount(sig->states) > 0 )
  {
    fprintf(fp, "<%.5f:", KheCostShow(sig->cost));
    HaArrayForEach(sig->states, val, i)
      fprintf(fp, " %d", val);
    fprintf(fp, ">");
  }
  else
    fprintf(fp, "<%.5f>", KheCostShow(sig->cost));
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_SIGNATURE_SET"                                        */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignatureSetInit(KHE_DRS_SIGNATURE_SET sig_set,               */
/*    KHE_COST cost, HA_ARENA a)                                             */
/*                                                                           */
/*  Initialize sig_set to have the given cost and no signatures.             */
/*                                                                           */
/*  There is no KheDrsSignatureSetMake, because signature sets are used      */
/*  only in solution objects, and those signature sets are expanded, hence   */
/*  "Init" (which assumes that memory is already allocated), not "Make".     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignatureSetInit(KHE_DRS_SIGNATURE_SET sig_set,
  KHE_COST cost, HA_ARENA a)
{
  sig_set->cost = cost;
  HaArrayInit(sig_set->signatures, a);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignatureSetClear(KHE_DRS_SIGNATURE_SET sig_set,              */
/*    KHE_COST cost, KHE_DYNAMIC_RESOURCE_SOLVER drs)                        */
/*                                                                           */
/*  Clear sig_set back to the given cost and no signatures.                  */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignatureSetClear(KHE_DRS_SIGNATURE_SET sig_set,
  KHE_COST cost, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SIGNATURE sig;
  sig_set->cost = cost;
  while( HaArrayCount(sig_set->signatures) > 0 )
  {
    sig = HaArrayLastAndDelete(sig_set->signatures);
    KheDrsSignatureUnRefer(sig, drs);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsSignatureSetFullHash(KHE_DRS_SIGNATURE_SET sig_set)            */
/*                                                                           */
/*  Hash function for hashing all of sig_set.                                */
/*                                                                           */
/*****************************************************************************/

static int KheDrsSignatureSetFullHash(KHE_DRS_SIGNATURE_SET sig_set)
{
  int res, val, i, j;  KHE_DRS_SIGNATURE sig;
  res = 0;
  HaArrayForEach(sig_set->signatures, sig, i)
    HaArrayForEach(sig->states, val, j)
      res = (res + val) * 3;
  return (res < 0 ? - res : res);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSignatureFullEqual(KHE_DRS_SIGNATURE sig1,                    */
/*    KHE_DRS_SIGNATURE sig2)                                                */
/*                                                                           */
/*  Return true if sig1 and sig2 are all equal.                              */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsSignatureSetFullEqual(KHE_DRS_SIGNATURE_SET sig_set1,
  KHE_DRS_SIGNATURE_SET sig_set2)
{
  int i, j;  KHE_DRS_SIGNATURE sig1, sig2;
  HnAssert(HaArrayCount(sig_set1->signatures) ==
    HaArrayCount(sig_set2->signatures),
    "KheDrsSignatureFullEqual internal error (1)");
  for( i = 0;  i < HaArrayCount(sig_set1->signatures);  i++ )
  {
    sig1 = HaArray(sig_set1->signatures, i);
    sig2 = HaArray(sig_set2->signatures, i);
    HnAssert(HaArrayCount(sig1->states) == HaArrayCount(sig2->states),
      "KheDrsSignatureFullEqual: signatures have different lengths");
    for( j = 0;  j < HaArrayCount(sig1->states);  j++ )
      if( HaArray(sig1->states, j) != HaArray(sig2->states, j) )
	return false;
  }
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignatureSetAddSignature(KHE_DRS_SIGNATURE_SET sig_set,       */
/*    KHE_DRS_SIGNATURE sig, bool with_cost)                                 */
/*                                                                           */
/*  Add sig to sig_set, but only include its cost if with_cost is true.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignatureSetAddSignature(KHE_DRS_SIGNATURE_SET sig_set,
  KHE_DRS_SIGNATURE sig, bool with_cost)
{
  if( with_cost )
    sig_set->cost += sig->cost;
  HaArrayAddLast(sig_set->signatures, sig);
  KheDrsSignatureRefer(sig);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignatureSetDebug(KHE_DRS_SIGNATURE_SET sig_set,              */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of sig_set onto fp with the given verbosity and indent.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignatureSetDebug(KHE_DRS_SIGNATURE_SET sig_set,
  int verbosity, int indent, FILE *fp)
{
  int i;  KHE_DRS_SIGNATURE sig;
  if( indent >= 0 )
  {
    fprintf(fp, "%*s<%.5f:\n", indent, "", KheCostShow(sig_set->cost));
    HaArrayForEach(sig_set->signatures, sig, i)
      KheDrsSignatureDebug(sig, verbosity, indent + 2, fp);
    fprintf(fp, "%*s>\n", indent, "");
  }
  else
    fprintf(fp, "<sig_set %.5f>", KheCostShow(sig_set->cost));
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_CORRELATOR" - handle correlated expressions           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_CORRELATOR KheDrsCorrelatorMake(KHE_DYNAMIC_RESOURCE_SOLVER drs) */
/*                                                                           */
/*  Make a new, empty correlator object.                                     */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_CORRELATOR KheDrsCorrelatorMake(KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_CORRELATOR res;  HA_ARENA a;
  a = drs->arena;
  HaMake(res, a);
  res->make_correlated = false;		/* reset when solving */
  HaArrayInit(res->children, a);
  HaArrayInit(res->positives, a);
  HaArrayInit(res->negatives, a);
  HaArrayInit(res->singles, a);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsCorrelatorClear(KHE_DRS_CORRELATOR dc)                        */
/*                                                                           */
/*  Clear dc.                                                                */
/*                                                                           */
/*****************************************************************************/

static void KheDrsCorrelatorClear(KHE_DRS_CORRELATOR dc)
{
  dc->make_correlated = false;
  HaArrayClear(dc->children);
  HaArrayClear(dc->positives);
  HaArrayClear(dc->negatives);
  HaArrayClear(dc->singles);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsCorrelatorMakeCorrelated(KHE_DRS_CORRELATOR dc)               */
/*                                                                           */
/*  Inform dc that correlated expressions are wanted.                        */
/*                                                                           */
/*****************************************************************************/

static void KheDrsCorrelatorMakeCorrelated(KHE_DRS_CORRELATOR dc)
{
  dc->make_correlated = true;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsCorr1Debug(KHE_DRS_DOM_TEST prnt_dt,                          */
/*    KHE_DRS_DOM_TEST child_dt, KHE_DRS_SIGNER dsg,                         */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)  */
/*                                                                           */
/*  Debug print of corr1 (prnt_dt, child_dt).                                */
/*                                                                           */
/*****************************************************************************/
static char *KheDrsSignerId(KHE_DRS_SIGNER dsg);

static void KheDrsCorr1Debug(KHE_DRS_DOM_TEST prnt_dt,
  KHE_DRS_DOM_TEST child_dt, KHE_DRS_SIGNER dsg,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)
{
  if( DEBUG55 )
  {
    fprintf(fp, "%*s[ signer %s found corr1 (delta %d):\n", indent, "",
      KheDrsSignerId(dsg), prnt_dt->correlated_delta);
    KheDrsExprDebug(prnt_dt->expr, drs, verbosity, indent + 2, fp);
    KheDrsExprDebug(child_dt->expr, drs, verbosity, indent + 2, fp);
    fprintf(fp, "%*s]\n", indent, "");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsCorr3Debug(KHE_DRS_DOM_TEST dt, int dt_index,                 */
/*    KHE_DRS_SIGNER dsg, KHE_DYNAMIC_RESOURCE_SOLVER drs,                   */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of corr3, beginning at dt.                                   */
/*                                                                           */
/*****************************************************************************/

static void KheDrsCorr3Debug(KHE_DRS_DOM_TEST dt, int dt_index,
  KHE_DRS_SIGNER dsg, KHE_DYNAMIC_RESOURCE_SOLVER drs,
  int verbosity, int indent, FILE *fp)
{
  if( DEBUG55 )
  {
    fprintf(fp, "%*s[ signer %s found corr3:\n", indent, "",
      KheDrsSignerId(dsg));
    do
    {
      KheDrsExprDebug(dt->expr, drs, verbosity, indent + 2, fp);
      dt_index += dt->correlated_delta;
      dt = HaArray(dsg->dom_tests, dt_index);
    } while( dt->type != KHE_DRS_DOM_TEST_CORR3_LAST );
    KheDrsExprDebug(dt->expr, drs, verbosity, indent + 2, fp);
    fprintf(fp, "%*s]\n", indent, "");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsCorr4Debug(KHE_DRS_DOM_TEST dt, int dt_index,                 */
/*    KHE_DRS_SIGNER dsg, KHE_DYNAMIC_RESOURCE_SOLVER drs,                   */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of corr4, beginning at dt.                                   */
/*                                                                           */
/*****************************************************************************/

static void KheDrsCorr4Debug(KHE_DRS_DOM_TEST dt, int dt_index,
  KHE_DRS_SIGNER dsg, KHE_DYNAMIC_RESOURCE_SOLVER drs,
  int verbosity, int indent, FILE *fp)
{
  if( DEBUG55 )
  {
    fprintf(fp, "%*s[ signer %s found corr4:\n", indent, "",
      KheDrsSignerId(dsg));
    do
    {
      KheDrsExprDebug(dt->expr, drs, verbosity, indent + 2, fp);
      dt_index += dt->correlated_delta;
      dt = HaArray(dsg->dom_tests, dt_index);
    } while( dt->type != KHE_DRS_DOM_TEST_CORR4_LAST );
    KheDrsExprDebug(dt->expr, drs, verbosity, indent + 2, fp);
    fprintf(fp, "%*s]\n", indent, "");
  }
}


/*****************************************************************************/
/*                                                                           */
/*   void KheDrsCorrelatorAddChildDomTest(KHE_DRS_CORRELATOR dc,             */
/*     KHE_DRS_DOM_TEST child_dt, int child_dt_index, KHE_DRS_SIGNER dsg,    */
/*     KHE_DYNAMIC_RESOURCE_SOLVER drs)                                      */
/*                                                                           */
/*  Carry out KheDrsCorrelatorAddDomTest for the case where child_dt->expr   */
/*  is an OR expression which is the child of a KHE_DRS_EXPR_INT_SUM_COST    */
/*  expression.  The child always arrives before its parent.                 */
/*                                                                           */
/*****************************************************************************/

static void KheDrsCorrelatorAddChildDomTest(KHE_DRS_CORRELATOR dc,
  KHE_DRS_DOM_TEST child_dt, int child_dt_index, KHE_DRS_SIGNER dsg,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  HnAssert(child_dt->corr_dom_table4 != NULL,
    "KheDrsCorrelatorAddChildDomTest internal error");
  child_dt->type = KHE_DRS_DOM_TEST_CORR2_CHILD;
  HaArrayAddLast(dc->children, child_dt_index);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsCorrelatorAddParentDomTest(KHE_DRS_CORRELATOR dc,             */
/*    KHE_DRS_DOM_TEST prnt_dt, int prnt_dt_index, KHE_DRS_SIGNER dsg,       */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Carry out the work of KheDrsCorrelatorAddDomTest for the case where      */
/*  prnt_dt->expr is an KHE_DRS_EXPR_INT_SUM_COST:  try to match it with     */
/*  a child dt, which must have arrived previously.                          */
/*                                                                           */
/*****************************************************************************/
static bool KheDrsParentIsFirstIntSumCostParent(KHE_DRS_EXPR child,
  KHE_DRS_EXPR parent);

static void KheDrsCorrelatorAddParentDomTest(KHE_DRS_CORRELATOR dc,
  KHE_DRS_DOM_TEST prnt_dt, int prnt_dt_index, KHE_DRS_SIGNER dsg,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DOM_TEST child_dt;  int child_dt_index, i;
  HaArrayForEach(dc->children, child_dt_index, i)
  {
    child_dt = HaArray(dsg->dom_tests, child_dt_index);
    if( KheDrsParentIsFirstIntSumCostParent(child_dt->expr, prnt_dt->expr) )
    {
      prnt_dt->type = KHE_DRS_DOM_TEST_CORR1_PARENT;
      child_dt->type = KHE_DRS_DOM_TEST_CORR1_CHILD;
      child_dt->correlated_delta = prnt_dt_index - child_dt_index;
      HaArrayDeleteAndPlug(dc->children, i);
      KheDrsCorr1Debug(prnt_dt, child_dt, dsg, drs, 1, 0, stderr);
      return;
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsCorrelatorAddDaysPositiveDomTest(KHE_DRS_CORRELATOR dc,       */
/*    KHE_DRS_DOM_TEST pos_dt, int pos_dt_index, KHE_DRS_SIGNER dsg,         */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Carry out the work of KheDrsCorrelatorAddDomTest for the case where      */
/*  dt->expr is a days-positive KHE_DRS_EXPR_INT_SEQ_COST.                   */
/*                                                                           */
/*****************************************************************************/

static void KheDrsCorrelatorAddDaysPositiveDomTest(KHE_DRS_CORRELATOR dc,
  KHE_DRS_DOM_TEST pos_dt, int pos_dt_index, KHE_DRS_SIGNER dsg,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DOM_TEST neg_dt;  int neg_dt_index;
  if( HaArrayCount(dc->negatives) > 0 )
  {
    /* make a matching pair with any existing negative */
    neg_dt_index = HaArrayLastAndDelete(dc->negatives);
    neg_dt = HaArray(dsg->dom_tests, neg_dt_index);
    pos_dt->type = KHE_DRS_DOM_TEST_CORR4_FIRST;
    pos_dt->correlated_delta = neg_dt_index - pos_dt_index;
    neg_dt->type = KHE_DRS_DOM_TEST_CORR4_LAST;
    KheDrsCorr4Debug(pos_dt, pos_dt_index, dsg, drs, 1, 2, stderr);
  }
  else
  {
    /* store for a potential future match */
    HaArrayAddLast(dc->positives, pos_dt_index);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsCorrelatorAddDaysNegativeDomTest(KHE_DRS_CORRELATOR dc,       */
/*    KHE_DRS_DOM_TEST neg_dt, int neg_dt_index, KHE_DRS_SIGNER dsg,         */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Carry out the work of KheDrsCorrelatorAddDomTest for the case where      */
/*  dt->expr is a days-negative KHE_DRS_EXPR_INT_SEQ_COST.                   */
/*                                                                           */
/*****************************************************************************/

static void KheDrsCorrelatorAddDaysNegativeDomTest(KHE_DRS_CORRELATOR dc,
  KHE_DRS_DOM_TEST neg_dt, int neg_dt_index, KHE_DRS_SIGNER dsg,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DOM_TEST pos_dt;  int pos_dt_index;
  if( HaArrayCount(dc->positives) > 0 )
  {
    /* make a matching pair with any existing positive */
    pos_dt_index = HaArrayLastAndDelete(dc->positives);
    pos_dt = HaArray(dsg->dom_tests, pos_dt_index);
    pos_dt->type = KHE_DRS_DOM_TEST_CORR4_FIRST;
    pos_dt->correlated_delta = neg_dt_index - pos_dt_index;
    neg_dt->type = KHE_DRS_DOM_TEST_CORR4_LAST;
    KheDrsCorr4Debug(pos_dt, pos_dt_index, dsg, drs, 1, 2, stderr);
  }
  else
  {
    /* store for a potential future match */
    HaArrayAddLast(dc->negatives, neg_dt_index);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsDomTestSingleSeqIndexesEqual(KHE_DRS_DOM_TEST dt1,            */
/*    KHE_DRS_DOM_TEST dt2)                                                  */
/*                                                                           */
/*  Return true if the indexes of singles dt1 and dt2 are equal.             */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsDomTestSingleSeqIndexesEqual(KHE_DRS_DOM_TEST dt1,
  KHE_DRS_DOM_TEST dt2)
{
  KHE_DRS_EXPR_INT_SEQ_COST eisc1, eisc2;
  eisc1 = (KHE_DRS_EXPR_INT_SEQ_COST) dt1->expr;
  eisc2 = (KHE_DRS_EXPR_INT_SEQ_COST) dt2->expr;
  return eisc1->seq_index == eisc2->seq_index;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsCorrelatorSingleAdd(KHE_DRS_DOM_TEST single_dt,               */
/*    int single_dt_index, KHE_DRS_DOM_TEST new_single_dt,                   */
/*    int new_single_dt_index, KHE_DRS_SIGNER dsg,                           */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Try to add new_single_dt (at new_single_dt_index) to the existing        */
/*  singles sequence beginning with single_dt (at single_dt_index).  If      */
/*  successful, update the existing singles sequence and return true.        */
/*  Otherwise change nothing and return false.                               */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsCorrelatorSingleAdd(KHE_DRS_DOM_TEST single_dt,
  int single_dt_index, KHE_DRS_DOM_TEST new_single_dt,
  int new_single_dt_index, KHE_DRS_SIGNER dsg,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DOM_TEST save_single_dt;  int save_single_dt_index;
  switch( single_dt->type )
  {
    case KHE_DRS_DOM_TEST_UNUSED:
    case KHE_DRS_DOM_TEST_CORR1_PARENT:
    case KHE_DRS_DOM_TEST_CORR1_CHILD:
    case KHE_DRS_DOM_TEST_CORR2_CHILD:

      /* should never happen */
      HnAbort("KheDrsCorrelatorSingleAdd internal error 1");
      return false;  /* keep compiler happy */

    case KHE_DRS_DOM_TEST_STRONG:
    case KHE_DRS_DOM_TEST_TRADEOFF:
    case KHE_DRS_DOM_TEST_UNIFORM:

      /* test and stop; if successful, result has two elements */
      if( KheDrsDomTestSingleSeqIndexesEqual(single_dt, new_single_dt) )
	return false;
      else
      {
        single_dt->type = KHE_DRS_DOM_TEST_CORR3_FIRST;
        single_dt->correlated_delta = new_single_dt_index - single_dt_index;
	new_single_dt->type = KHE_DRS_DOM_TEST_CORR3_LAST;
	KheDrsCorr3Debug(single_dt, single_dt_index, dsg, drs, 1, 2, stderr);
	return true;
      }

    case KHE_DRS_DOM_TEST_CORR3_FIRST:

      save_single_dt = single_dt;
      save_single_dt_index = single_dt_index;
      do
      {
	/* check new_single_dt against single_dt */
	if( KheDrsDomTestSingleSeqIndexesEqual(single_dt, new_single_dt) )
	  return false;

	/* move on to the next single_dt */
	single_dt_index += single_dt->correlated_delta;
	single_dt = HaArray(dsg->dom_tests, single_dt_index);
      } while( single_dt->type == KHE_DRS_DOM_TEST_CORR3_MID );

      /* check last, and if successful, add new_single_dt to the end */
      HnAssert(single_dt->type == KHE_DRS_DOM_TEST_CORR3_LAST,
        "KheDrsCorrelatorSingleAdd internal error 2");
      if( KheDrsDomTestSingleSeqIndexesEqual(single_dt, new_single_dt) )
	return false;
      else
      {
	single_dt->type = KHE_DRS_DOM_TEST_CORR3_MID;
	single_dt->correlated_delta = new_single_dt_index - single_dt_index;
	new_single_dt->type = KHE_DRS_DOM_TEST_CORR3_LAST;
	KheDrsCorr3Debug(save_single_dt, save_single_dt_index, dsg, drs,
	  1, 2, stderr);
	return true;
      }

    default:

      HnAbort("KheDrsCorrelatorSingleAdd internal error 2");
      return false;  /* keep compiler happy */
  }

  /* unreachable here */
  HnAbort("KheDrsCorrelatorSingleAdd internal error 3");
  return false;  /* keep compiler happy */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsCorrelatorAddSinglePositiveDomTest(KHE_DRS_CORRELATOR dc,     */
/*    KHE_DRS_DOM_TEST new_single_dt, int new_single_dt_index,               */
/*    KHE_DRS_SIGNER dsg, KHE_DYNAMIC_RESOURCE_SOLVER drs)                   */
/*                                                                           */
/*  Carry out the work of KheDrsCorrelatorAddDomTest for the case where      */
/*  new_single_dt->expr is a single-time positive KHE_DRS_EXPR_INT_SEQ_COST. */
/*                                                                           */
/*****************************************************************************/

static void KheDrsCorrelatorAddSinglePositiveDomTest(KHE_DRS_CORRELATOR dc,
  KHE_DRS_DOM_TEST new_single_dt, int new_single_dt_index,
  KHE_DRS_SIGNER dsg, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int single_dt_index, i;  KHE_DRS_DOM_TEST single_dt;

  /* try adding to each singles sequence, and return if successful */
  HaArrayForEach(dc->singles, single_dt_index, i)
  {
    single_dt = HaArray(dsg->dom_tests, single_dt_index);
    if( KheDrsCorrelatorSingleAdd(single_dt, single_dt_index,
	  new_single_dt, new_single_dt_index, dsg, drs) )
      return;
  }

  /* no match so save new_single_dt for later */
  HaArrayAddLast(dc->singles, new_single_dt_index);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsCorrelatorAddDomTest(KHE_DRS_CORRELATOR dc,                   */
/*    KHE_DRS_DOM_TEST dt, int dt_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)    */
/*                                                                           */
/*  Add dt to dc.                                                            */
/*                                                                           */
/*****************************************************************************/
static bool KheDrsExprHasParent(KHE_DRS_EXPR child, KHE_DRS_EXPR_TAG tag);

static void KheDrsCorrelatorAddDomTest(KHE_DRS_CORRELATOR dc,
  KHE_DRS_DOM_TEST dt, int dt_index, KHE_DRS_SIGNER dsg,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_INT_SEQ_COST eisc;
  if( dc->make_correlated )
  {
    if( DEBUG55 )
      fprintf(stderr, "[ KheDrsCorrelatorAddDomTest(dc, dt, %d, %s, drs)\n",
	dt_index, KheDrsSignerId(dsg));
    switch( dt->expr->tag )
    {
      case KHE_DRS_EXPR_OR_TAG:

        if( KheDrsExprHasParent(dt->expr, KHE_DRS_EXPR_INT_SUM_COST_TAG) )
	  KheDrsCorrelatorAddChildDomTest(dc, dt, dt_index, dsg, drs);
	break;

      case KHE_DRS_EXPR_INT_SUM_COST_TAG:

        KheDrsCorrelatorAddParentDomTest(dc, dt, dt_index, dsg, drs);
	break;

      case KHE_DRS_EXPR_INT_SEQ_COST_TAG:

	eisc = (KHE_DRS_EXPR_INT_SEQ_COST) dt->expr;
	if( eisc->seq_type == KHE_DRS_SEQ_DAYS_POSITIVE )
	  KheDrsCorrelatorAddDaysPositiveDomTest(dc, dt, dt_index, dsg, drs);
	else if( eisc->seq_type == KHE_DRS_SEQ_DAYS_NEGATIVE )
	  KheDrsCorrelatorAddDaysNegativeDomTest(dc, dt, dt_index, dsg, drs);
	else if( eisc->seq_type == KHE_DRS_SEQ_SINGLE_POSITIVE )
	  KheDrsCorrelatorAddSinglePositiveDomTest(dc, dt, dt_index, dsg, drs);
	break;

      default:

	/* ignore others */
	break;
    }
    if( DEBUG55 )
      fprintf(stderr, "] KheDrsCorrelatorAddDomTest returning\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_SIGNER"                                               */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SIGNER KheDrsSignerMake(KHE_DRS_DAY day,                         */
/*    KHE_DRS_RESOURCE_ON_DAY drd, KHE_DRS_SHIFT ds,                         */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new, empty signer object for day, drd, or ds.                     */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_SIGNER KheDrsSignerMake(KHE_DRS_DAY day,
  KHE_DRS_RESOURCE_ON_DAY drd, KHE_DRS_SHIFT ds,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SIGNER res;

  /* get res and make sure its arrays are initialized and empty */
  if( HaArrayCount(drs->signer_free_list) > 0 )
  {
    res = HaArrayLastAndDelete(drs->signer_free_list);
    HaArrayClear(res->internal_exprs);
    HaArrayClear(res->dom_tests);
    HaArrayClear(res->eq_dom_test_indexes);
  }
  else
  {
    HaMake(res, drs->arena);
    HaArrayInit(res->internal_exprs, drs->arena);
    HaArrayInit(res->dom_tests, drs->arena);
    HaArrayInit(res->eq_dom_test_indexes, drs->arena);
  }

  /* get a correlator, but only if drd type (never freed, by the way) */
  res->correlator = (drd != NULL ? KheDrsCorrelatorMake(drs) : NULL);

  /* set the other fields */
  res->last_hard_cost_index = -1;
  res->debug_eval_if_rerun = false;  /* placeholder */
  if( day != NULL )
  {
    HnAssert(drd == NULL && ds == NULL, "KheDrsSignerMake internal error 1");
    res->type = KHE_DRS_SIGNER_DAY;
    res->u.day = day;
  }
  else if( drd != NULL )
  {
    HnAssert(day == NULL && ds == NULL, "KheDrsSignerMake internal error 2");
    res->type = KHE_DRS_SIGNER_RESOURCE_ON_DAY;
    res->u.resource_on_day = drd;
  }
  else if( ds != NULL )
  {
    HnAssert(day == NULL && drd == NULL,"KheDrsSignerMake internal error 3");
    res->type = KHE_DRS_SIGNER_SHIFT;
    res->u.shift = ds;
  }
  else
    HnAbort("KheDrsSignerMake internal error 4");

  /* all good, return */
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignerMakeCorrelated(KHE_DRS_SIGNER dsg)                      */
/*                                                                           */
/*  Inform dsg that correlated expressions are wanted.                       */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignerMakeCorrelated(KHE_DRS_SIGNER dsg)
{
  HnAssert(dsg->correlator != NULL,
    "KheDrsSignerMakeCorrelated internal error");
  KheDrsCorrelatorMakeCorrelated(dsg->correlator);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignerFree(KHE_DRS_SIGNER dsg,                                */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Free dsg.                                                                */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignerFree(KHE_DRS_SIGNER dsg, 
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  HnAssert(dsg->correlator == NULL, "KheDrsSignerFree internal error");
  HaArrayAddLast(drs->signer_free_list, dsg);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignerClear(KHE_DRS_SIGNER dsg, bool free_dom_tests,          */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Clear dsg ready for a new solve, without changing its type.  This        */
/*  includes freeing any dom tests left over from the previous solve.        */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignerClear(KHE_DRS_SIGNER dsg,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int i;
  HaArrayClear(dsg->internal_exprs);
  HaArrayAppend(drs->dom_test_free_list, dsg->dom_tests, i);
  HaArrayClear(dsg->dom_tests);
  HaArrayClear(dsg->eq_dom_test_indexes);
  dsg->last_hard_cost_index = -1;
  dsg->debug_eval_if_rerun = false;
  if( dsg->correlator != NULL )
    KheDrsCorrelatorClear(dsg->correlator);
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheDrsSignerId(KHE_DRS_SIGNER dsg)                                 */
/*                                                                           */
/*  Return the Id if dsg.                                                    */
/*                                                                           */
/*****************************************************************************/

static char *KheDrsSignerId(KHE_DRS_SIGNER dsg)
{
  switch( dsg->type )
  {
    case KHE_DRS_SIGNER_DAY:

      return KheDrsDayId(dsg->u.day);

    case KHE_DRS_SIGNER_RESOURCE_ON_DAY:

      return KheDrsResourceOnDayId(dsg->u.resource_on_day);

    case KHE_DRS_SIGNER_SHIFT:

      return KheDrsShiftId(dsg->u.shift);

    default:

      return "?";
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignerAddOpenExpr(KHE_DRS_SIGNER dsg, KHE_DRS_EXPR e)         */
/*                                                                           */
/*  Add e to dsg.                                                            */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignerAddOpenExpr(KHE_DRS_SIGNER dsg, KHE_DRS_EXPR e)
{
  if( dsg->type == KHE_DRS_SIGNER_SHIFT )
    HnAssert(e->tag == KHE_DRS_EXPR_INT_SUM_COST_TAG,
      "KheDrsSignerAddOpenExpr internal error");
  HaArrayAddLast(dsg->internal_exprs, e);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsSignerAddDomTest(KHE_DRS_SIGNER dsg,                           */
/*    KHE_DRS_DOM_TEST dom_test)                                             */
/*                                                                           */
/*  Add dom_test to dsg and return its index in dsg.                         */
/*                                                                           */
/*  If dsg->make_correlated is set, identify pairs of correlated dom         */
/*  tests and adjust them appropriately.                                     */
/*                                                                           */
/*****************************************************************************/
static bool KheDrsDomTestIsStrictEquality(KHE_DRS_DOM_TEST dom_test);
static KHE_COST KheDrsMonitorWeight(KHE_MONITOR m);

static int KheDrsSignerAddDomTest(KHE_DRS_SIGNER dsg,
  KHE_DRS_DOM_TEST dt, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int dt_index;

  /* add dt */
  dt_index = HaArrayCount(dsg->dom_tests);
  HaArrayAddLast(dsg->dom_tests, dt);

  /* if dt is equality, record it in eq_dom_test_indexes */
  if( KheDrsDomTestIsStrictEquality(dt) )
    HaArrayAddLast(dsg->eq_dom_test_indexes, dt_index);

  /* handle correlated expressions, if wanted */
  if( dsg->correlator != NULL )
    KheDrsCorrelatorAddDomTest(dsg->correlator, dt, dt_index, dsg, drs);
    /* KheDrsSignerHandleCorrelatedTests(dsg, dt, dt_index, drs); */

  /* keep track of where hard constraints end */
  if( KheDrsMonitorWeight(dt->monitor) >= KheCost(1, 0) )
    dsg->last_hard_cost_index = dt_index;

  /* return dt's index */
  return dt_index;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignerEvalSignature(KHE_DRS_SIGNER dsg,                       */
/*    bool debug_if_rerun, KHE_DRS_SIGNATURE prev_sig, int next_di,          */
/*    KHE_DRS_SIGNATURE next_sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)           */
/*                                                                           */
/*  Using dsg, build next_sig, not necessarily from scratch.                 */
/*                                                                           */
/*****************************************************************************/
static void KheDrsExprEvalSignature(KHE_DRS_EXPR e, KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE prev_sig, int next_di, KHE_DRS_SIGNATURE next_sig,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, bool debug);

static void KheDrsSignerEvalSignature(KHE_DRS_SIGNER dsg,
  bool debug_if_rerun, KHE_DRS_SIGNATURE prev_sig, int next_di,
  KHE_DRS_SIGNATURE next_sig, KHE_DYNAMIC_RESOURCE_SOLVER drs, bool debug)
{
  KHE_DRS_EXPR e;  int i;
  dsg->debug_eval_if_rerun = debug_if_rerun;
  HaArrayForEach(dsg->internal_exprs, e, i)
    KheDrsExprEvalSignature(e, dsg, prev_sig, next_di, next_sig, drs, debug);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDominanceDebugSpacer(int indent, FILE *fp)                    */
/*                                                                           */
/*  Print a spacer line in a debug print of a dominance test.                */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDominanceDebugSpacer(int indent, FILE *fp)
{
  if( fp != NULL )
    fprintf(fp, "%*s------------------------------------------\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsPrintLimits(char *tag, bool allow_zero, int min_lim,          */
/*    int max_lim, KHE_COST combined_wt, KHE_COST_FUNCTION cost_fn, FILE *fp)*/
/*                                                                           */
/*  Print all this stuff in an intuitive format.                             */
/*                                                                           */
/*****************************************************************************/

static void KheDrsPrintLimits(char *tag, bool allow_zero, int min_lim,
  int max_lim, KHE_COST combined_wt, KHE_COST_FUNCTION cost_fn, FILE *fp)
{
  char buff[200], *buff2;  int soft_wt, hard_wt;

  /* print the initial tag */
  fprintf(fp, "%-4s ", tag);

  /* print allow_zero, min_lim, and max_lim if required */
  if( min_lim >= 0 )
  {
    /* sort out allow_zero flag */
    if( allow_zero && min_lim <= 1 )
      allow_zero = false, min_lim = 0;

    /* print min_lim and max_lim, where present */
    if( min_lim > 0 )
    {
      if( max_lim < INT_MAX )
      {
	/* have min_lim and max_lim */
	if( min_lim == max_lim )
	  sprintf(buff, "%s%d", allow_zero ? "0," : "", min_lim);
	else
	  sprintf(buff, "%s%d-%d", allow_zero ? "0," : "", min_lim, max_lim);
      }
      else
      {
	/* have min_lim only */
	sprintf(buff, "%s%d-inf", allow_zero ? "0," : "", min_lim);
      }
    }
    else
    {
      if( max_lim < INT_MAX )
      {
	/* have max_lim only */
	sprintf(buff, "%s0-%d", allow_zero ? "0," : "", max_lim);
      }
      else
      {
	/* have no limits at all */
	sprintf(buff, "%s-", allow_zero ? "0," : "");
      }
    }
  }
  else
    sprintf(buff, "%s", "");

  /* print bar and the the soft and hard weight */
  buff2 = &buff[strlen(buff)];
  soft_wt = KheSoftCost(combined_wt);
  hard_wt = KheHardCost(combined_wt);
  if( soft_wt > 0 )
  {
    if( hard_wt > 0 )
    {
      /* soft and hard weight */
      sprintf(buff2, "|h%d+s%d", hard_wt, soft_wt);
    }
    else
    {
      /* soft weight only */
      sprintf(buff2, "|s%d", soft_wt);
    }
  }
  else
  {
    if( hard_wt > 0 )
    {
      /* hard weight only */
      sprintf(buff2, "|h%d", hard_wt);
    }
    else
    {
      /* no weight at all */
      sprintf(buff2, "|0");
    }
  }

  /* print the cost function */
  buff2 = &buff[strlen(buff)];
  switch( cost_fn )
  {
    case KHE_STEP_COST_FUNCTION:

      sprintf(buff2, "s");
      break;

    case KHE_LINEAR_COST_FUNCTION:

      /* default, don't print anything */
      break;

    case KHE_QUADRATIC_COST_FUNCTION:

      sprintf(buff2, "q");
      break;

    default:

      sprintf(buff2, "?");
  }

  /* print it all out, fixed width */
  fprintf(fp, "%-8s", buff);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsMonitorDebug(KHE_MONITOR m, FILE *fp)                         */
/*                                                                           */
/*  Debug print of m with a few key attributes.  Alternatively, if m is      */
/*  NULL, print nothing.                                                     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsMonitorDebug(KHE_MONITOR m, FILE *fp)
{
  KHE_CONSTRAINT c;  KHE_COST wt;  KHE_COST_FUNCTION cost_fn;
  int min_lim, max_lim;  bool allow_zero;
  KHE_CLUSTER_BUSY_TIMES_CONSTRAINT cbtc;
  KHE_LIMIT_BUSY_TIMES_CONSTRAINT lbtc;
  KHE_LIMIT_WORKLOAD_CONSTRAINT lwc;
  KHE_LIMIT_ACTIVE_INTERVALS_CONSTRAINT laic;
  KHE_LIMIT_RESOURCES_CONSTRAINT lrc;
  if( m != NULL )
  {
    c = KheMonitorConstraint(m);
    wt = KheConstraintCombinedWeight(c);
    cost_fn = KheConstraintCostFunction(c);
    switch( KheConstraintTag(c) )
    {
      case KHE_ASSIGN_RESOURCE_CONSTRAINT_TAG:

	KheDrsPrintLimits("arc", false, -1, -1, wt, cost_fn, fp);
	break;

      case KHE_PREFER_RESOURCES_CONSTRAINT_TAG:

	KheDrsPrintLimits("prc", false, -1, -1, wt, cost_fn, fp);
	break;

      case KHE_AVOID_SPLIT_ASSIGNMENTS_CONSTRAINT_TAG:

	KheDrsPrintLimits("asac", false, -1, -1, wt, cost_fn, fp);
	break;

      case KHE_AVOID_CLASHES_CONSTRAINT_TAG:

	KheDrsPrintLimits("acc", false, -1, -1, wt, cost_fn, fp);
	break;

      case KHE_AVOID_UNAVAILABLE_TIMES_CONSTRAINT_TAG:

	KheDrsPrintLimits("autc", false, -1, -1, wt, cost_fn, fp);
	break;

      case KHE_LIMIT_IDLE_TIMES_CONSTRAINT_TAG:

	KheDrsPrintLimits("litc", false, -1, -1, wt, cost_fn, fp);
	break;

      case KHE_CLUSTER_BUSY_TIMES_CONSTRAINT_TAG:

	cbtc = (KHE_CLUSTER_BUSY_TIMES_CONSTRAINT) c;
	min_lim = KheClusterBusyTimesConstraintMinimum(cbtc);
	max_lim = KheClusterBusyTimesConstraintMaximum(cbtc);
	allow_zero = KheClusterBusyTimesConstraintAllowZero(cbtc);
	KheDrsPrintLimits("cbtc", allow_zero, min_lim, max_lim, wt, cost_fn,fp);
	break;

      case KHE_LIMIT_BUSY_TIMES_CONSTRAINT_TAG:

	lbtc = (KHE_LIMIT_BUSY_TIMES_CONSTRAINT) c;
	min_lim = KheLimitBusyTimesConstraintMinimum(lbtc);
	max_lim = KheLimitBusyTimesConstraintMaximum(lbtc);
	allow_zero = KheLimitBusyTimesConstraintAllowZero(lbtc);
	KheDrsPrintLimits("lbtc", allow_zero, min_lim, max_lim, wt, cost_fn,fp);
	break;

      case KHE_LIMIT_WORKLOAD_CONSTRAINT_TAG:

	lwc = (KHE_LIMIT_WORKLOAD_CONSTRAINT) c;
	min_lim = KheLimitWorkloadConstraintMinimum(lwc);
	max_lim = KheLimitWorkloadConstraintMaximum(lwc);
	allow_zero = KheLimitWorkloadConstraintAllowZero(lwc);
	KheDrsPrintLimits("lwc", allow_zero, min_lim, max_lim, wt, cost_fn, fp);
	break;

      case KHE_LIMIT_ACTIVE_INTERVALS_CONSTRAINT_TAG:

	laic = (KHE_LIMIT_ACTIVE_INTERVALS_CONSTRAINT) c;
	min_lim = KheLimitActiveIntervalsConstraintMinimum(laic);
	max_lim = KheLimitActiveIntervalsConstraintMaximum(laic);
	KheDrsPrintLimits("laic", false, min_lim, max_lim, wt, cost_fn, fp);
	break;

      case KHE_LIMIT_RESOURCES_CONSTRAINT_TAG:

	lrc = (KHE_LIMIT_RESOURCES_CONSTRAINT) c;
	min_lim = KheLimitResourcesConstraintMinimum(lrc);
	max_lim = KheLimitResourcesConstraintMaximum(lrc);
	KheDrsPrintLimits("lrc", false, min_lim, max_lim, wt, cost_fn, fp);
	break;

      default:

	fprintf(fp, "? %.5f \n", KheCostShow(wt));
	break;
    }
    fprintf(fp, " %s", KheMonitorId(m));
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDominanceDebugCmpLine(bool res, KHE_COST avail_cost,          */
/*    KHE_DRS_DOM_TEST_TYPE type, int i, int v1, int v2, KHE_MONITOR m,      */
/*    char *details, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Print of one line of a dominance debug, but only if fp != NULL.          */
/*                                                                           */
/*****************************************************************************/
static char *KheDrsDomTestTypeShow(KHE_DRS_DOM_TEST_TYPE type);

static void KheDrsDominanceDebugCmpLine(KHE_COST avail_cost,
  KHE_DRS_DOM_TEST_TYPE type, int i, int v1, int v2, KHE_MONITOR m,
  int indent, FILE *fp)
{
  if( fp != NULL )
  {
    fprintf(fp, "%*s%8.5f  %2s[%2d] %2d %2d", indent, "",
      KheCostShow(avail_cost), KheDrsDomTestTypeShow(type), i, v1, v2);
    if( m != NULL )
    {
      fprintf(fp, "  ");
      KheDrsMonitorDebug(m, fp);
    }
    fprintf(fp, "\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSignerCorr1Dominates(KHE_DRS_SIGNER dsg,                      */
/*    KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2, KHE_DRS_DOM_TEST dt,   */
/*    int dt_index, int v1, int v2, KHE_COST *avail_cost, int verbosity,     */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Carry out a CORR1 dominance test whose first (child) element is dt.      */
/*                                                                           */
/*****************************************************************************/
static KHE_DRS_COST_TUPLE KheDrsDim4TableGet4(KHE_DRS_DIM4_TABLE d4,
  int index4, int index3, int index2, int index1);

static void KheDrsSignerCorr1Dominates(KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2, KHE_DRS_DOM_TEST dt,
  int dt_index, int a1, int a2, KHE_COST *avail_cost, int verbosity,
  int indent, FILE *fp)
{
  int other_dt_index, l1, l2;  KHE_DRS_DOM_TEST other_dt;  KHE_COST psi;
  other_dt_index = dt_index + dt->correlated_delta;
  other_dt = HaArray(dsg->dom_tests, other_dt_index);
  HnAssert(other_dt->type == KHE_DRS_DOM_TEST_CORR1_PARENT,
    "KheDrsSignerCorr1Dominates internal error (other_dt_index = %d)",
    other_dt_index);
  l1 = HaArray(sig1->states, other_dt_index);
  l2 = HaArray(sig2->states, other_dt_index);
  psi = KheDrsDim4TableGet4(dt->corr_dom_table4, a1, a2, l1, l2).psi;
  if( fp != NULL )
    fprintf(fp, "\n%*s[ Corr1(a1 %d, a2 %d, l1 %d, l2 %d) -> %.5f ]\n\n",
      indent, "", a1, a2, l1, l2, KheCostShow(psi));
  *avail_cost += psi;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignerCorr2Dominates(KHE_DRS_SIGNER dsg,                      */
/*    KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2, KHE_DRS_DOM_TEST dt,   */
/*    int dt_index, int v1, int v2, KHE_COST *avail_cost, int verbosity,     */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Carry out a CORR2 dominance test whose sole (child) element is dt.       */
/*                                                                           */
/*****************************************************************************/
static int KheDrsExprIntSumCostInitialValue(KHE_DRS_EXPR_INT_SUM_COST eisc);

static void KheDrsSignerCorr2Dominates(KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2, KHE_DRS_DOM_TEST dt,
  int dt_index, int a1, int a2, KHE_COST *avail_cost, int verbosity,
  int indent, FILE *fp)
{
  int l1, l2;  KHE_DRS_EXPR_INT_SUM_COST eisc;  KHE_DRS_PARENT prnt;
  KHE_COST psi;
  HnAssert(HaArrayCount(dt->expr->parents) > 0,
    "KheDrsSignerCorr2Dominates internal error (no parents)");
  prnt = HaArrayFirst(dt->expr->parents);
  HnAssert(prnt.expr->tag == KHE_DRS_EXPR_INT_SUM_COST_TAG,
    "KheDrsSignerCorr2Dominates internal error (prnt.expr->tag = %d)",
    prnt.expr->tag);
  eisc = (KHE_DRS_EXPR_INT_SUM_COST) prnt.expr;
  l1 = l2 = KheDrsExprIntSumCostInitialValue(eisc);
  psi = KheDrsDim4TableGet4(dt->corr_dom_table4, a1, a2, l1, l2).psi;
  if( fp != NULL )
    fprintf(fp, "\n%*s[ Corr2(a1 %d, a2 %d, l1 %d, l2 %d) -> %.5f ]\n\n",
      indent, "", a1, a2, l1, l2, KheCostShow(psi));
  *avail_cost += psi;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDebugCorr34Header(char *header_str, int verbosity,            */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of the header of a Corr3 or Corr4 test.                      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDebugCorr34Header(char *header_str, int verbosity,
  int indent, FILE *fp)
{
  if( fp != NULL )
    fprintf(fp, "\n%*s[ %-6s %2s %2s  %8s %8s %8s %8s %8s\n", indent, "",
      header_str, "v1", "v2", "Psi", "Psi0", "diff", "min_diff", "sum_Psi0");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDebugCorr34Line(KHE_DRS_DOM_TEST dt, int dt_index,            */
/*    int v1, int v2, KHE_COST psi, KHE_COST psi0, KHE_COST diff,            */
/*    KHE_COST min_diff, KHE_COST sum_psi0, int indent, FILE *fp)            */
/*                                                                           */
/*  Print one line of a corr3 or corr4 debug, but only if fp != NULL.        */
/*                                                                           */
/*****************************************************************************/
static void KheDrsMonitorDebug(KHE_MONITOR m, FILE *fp);

static void KheDrsDebugCorr34Line(KHE_DRS_DOM_TEST dt, int dt_index,
  int v1, int v2, KHE_COST psi, KHE_COST psi0, KHE_COST diff,
  KHE_COST min_diff, KHE_COST sum_psi0, int indent, FILE *fp)
{
  if( fp != NULL )
  {
    fprintf(fp, "%*s  %2s[%2d] %2d %2d  %8.5f %8.5f %8.5f %8.5f %8.5f ",
      indent, "", KheDrsDomTestTypeShow(dt->type), dt_index, v1, v2,
      KheCostShow(psi), KheCostShow(psi0), KheCostShow(diff),
      KheCostShow(min_diff), KheCostShow(sum_psi0));
    KheDrsMonitorDebug(dt->monitor, fp);
    fprintf(fp, "\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignerCorr3Dominates(KHE_DRS_SIGNER dsg,                      */
/*    KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2, KHE_DRS_DOM_TEST dt,   */
/*    int dt_index, int v1, int v2, KHE_COST *avail_cost,                    */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Carry out a CORR3 dominance test whose first element is dt.              */
/*                                                                           */
/*****************************************************************************/
static KHE_DRS_COST_TUPLE KheDrsDim2TableGet2(KHE_DRS_DIM2_TABLE d2,
  int index2, int index1);

static void KheDrsSignerCorr3Dominates(KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2, KHE_DRS_DOM_TEST dt,
  int dt_index, int v1, int v2, KHE_COST *avail_cost,
  int verbosity, int indent, FILE *fp)
{
  KHE_COST psi_diff, min_psi_diff, sum_psi0, sum_psi;  KHE_DRS_COST_TUPLE mp;
  bool differ3;
  KheDrsDebugCorr34Header("Corr3", verbosity, indent, fp);
  mp = KheDrsDim2TableGet2(dt->main_dom_table2, v1, v2);
  min_psi_diff = mp.psi - mp.psi0;
  sum_psi0 = mp.psi0;
  sum_psi = mp.psi;
  KheDrsDebugCorr34Line(dt, dt_index, v1, v2, mp.psi, mp.psi0, min_psi_diff,
    min_psi_diff, sum_psi0, indent, fp);
  do
  {
    /* move to next dt, v1, and v2 */
    dt_index += dt->correlated_delta;
    dt = HaArray(dsg->dom_tests, dt_index);
    v1 = HaArray(sig1->states, dt_index);
    v2 = HaArray(sig2->states, dt_index);

    /* accumulate min_psi_diff and sum_psi0 */
    mp = KheDrsDim2TableGet2(dt->main_dom_table2, v1, v2);
    psi_diff = mp.psi - mp.psi0;
    if( psi_diff < min_psi_diff )
      min_psi_diff = psi_diff;
    sum_psi0 += mp.psi0;
    sum_psi  += mp.psi;
    KheDrsDebugCorr34Line(dt, dt_index, v1, v2, mp.psi, mp.psi0, psi_diff,
      min_psi_diff, sum_psi0, indent, fp);
  } while( dt->type != KHE_DRS_DOM_TEST_CORR3_LAST );

  /* and there we are */
  *avail_cost += (min_psi_diff + sum_psi0);
  if( fp != NULL )
  {
    differ3 = (min_psi_diff + sum_psi0 != sum_psi);
    fprintf(fp, "%*s] corr3 %.5f, uncorrelated %.5f%s\n\n", indent, "",
      KheCostShow(min_psi_diff + sum_psi0), KheCostShow(sum_psi),
      differ3 ? " (corr3 differs from uncorrelated)" : "");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignerCorr4Dominates(KHE_DRS_SIGNER dsg,                      */
/*    KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2, KHE_DRS_DOM_TEST dt,   */
/*    int dt_index, int v1, int v2, KHE_COST *avail_cost,                    */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Like KheDrsSignerCorr4Dominates, only with debug output.                 */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignerCorr4Dominates(KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2, KHE_DRS_DOM_TEST dt,
  int dt_index, int v1, int v2, KHE_COST *avail_cost,
  int verbosity, int indent, FILE *fp)
{
  KHE_COST psi_diff, min_psi_diff, psi_plus_diff, min_psi_plus_diff;
  KHE_COST sum_psi0, sum_psi;  KHE_DRS_COST_TUPLE mp;  bool differ3, differ4;
  KheDrsDebugCorr34Header("Corr4", verbosity, indent, fp);
  mp = KheDrsDim2TableGet2(dt->main_dom_table2, v1, v2);
  HnAssert(mp.psi_plus_defined, "KheDrsSignerCorr4Dominates internal error 1");
  psi_plus_diff = mp.psi_plus - mp.psi0;
  min_psi_plus_diff = psi_plus_diff;
  psi_diff = mp.psi - mp.psi0;
  min_psi_diff = psi_diff;
  sum_psi0 = mp.psi0;
  sum_psi = mp.psi;
  KheDrsDebugCorr34Line(dt, dt_index, v1, v2, mp.psi_plus, mp.psi0,
    psi_plus_diff, min_psi_plus_diff, sum_psi0, indent, fp);
  do
  {
    /* move to next dt, v1, and v2 */
    dt_index += dt->correlated_delta;
    dt = HaArray(dsg->dom_tests, dt_index);
    v1 = HaArray(sig1->states, dt_index);
    v2 = HaArray(sig2->states, dt_index);

    /* accumulate min_psi_plus_diff and sum_psi0 */
    mp = KheDrsDim2TableGet2(dt->main_dom_table2, v1, v2);
    HnAssert(mp.psi_plus_defined >= 0,
      "KheDrsSignerCorr4Dominatesinternal error 2");
    psi_plus_diff = mp.psi_plus - mp.psi0;
    if( psi_plus_diff < min_psi_plus_diff )
      min_psi_plus_diff = psi_plus_diff;
    psi_diff = mp.psi - mp.psi0;
    if( psi_diff < min_psi_diff )
      min_psi_diff = psi_diff;
    sum_psi0 += mp.psi0;
    sum_psi  += mp.psi;
    KheDrsDebugCorr34Line(dt, dt_index, v1, v2, mp.psi_plus, mp.psi0,
      psi_plus_diff, min_psi_plus_diff, sum_psi0, indent, fp);
  } while( dt->type != KHE_DRS_DOM_TEST_CORR4_LAST );

  /* and there we are */
  *avail_cost += (min_psi_plus_diff + sum_psi0);
  if( fp != NULL )
  {
    differ3 = (min_psi_diff + sum_psi0 != sum_psi);
    differ4 = (min_psi_plus_diff != min_psi_diff);
    fprintf(fp, "%*s] corr4 %.5f, corr3 %.5f, uncorrelated %.5f%s%s\n\n",
      indent, "", KheCostShow(min_psi_plus_diff + sum_psi0),
      KheCostShow(min_psi_diff + sum_psi0), KheCostShow(sum_psi),
      differ3 ? " (corr3 differs from uncorrelated)" : "",
      differ4 ? " (corr4 differs from corr3)" : "");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDominanceDebugHeaderLine(char *desc, KHE_DRS_SIGNER dsg,      */
/*    KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2,                        */
/*    KHE_DRS_SIGNER_TEST test, KHE_COST avail_cost, int indent, FILE *fp)   */
/*                                                                           */
/*  Debug print one header line of a dominance test, after a spacer line.    */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDominanceDebugHeaderLine(KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2,
  KHE_DRS_SIGNER_TEST test, KHE_COST avail_cost, int indent, FILE *fp)
{
  char *descr[3] = {"Hard", "Soft", "All"};
  if( fp != NULL )
  {
    KheDrsDominanceDebugSpacer(indent, fp);
    fprintf(fp, "%*s%s %-11s v1 v2 (avail %.5f)\n", indent, "",
      descr[(int) test], KheDrsSignerId(dsg), KheCostShow(avail_cost));
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSignerDoDominates(KHE_DRS_SIGNER dsg,                         */
/*    KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2,                        */
/*    KHE_DRS_SIGNER_TEST test, int trie_start_depth, bool stop_on_neg,      */
/*    KHE_COST *avail_cost, int verbosity, int indent, FILE *fp)             */
/*                                                                           */
/*  This function does most of the work of deciding whether sig1             */
/*  dominates sig2.  It has several extra parameters, allowing it            */
/*  to handle several variants of the idea, as follows.                      */
/*                                                                           */
/*  Parameter test says whether to compare only hard constraints, or only    */
/*  soft constraints, or all.  The individual tests can be done in any       */
/*  order, and doing all the hard constraints of several signers before      */
/*  all the soft constraints is known to save time.                          */
/*                                                                           */
/*  Parameter trie_start_depth is non-zero only when this function is        */
/*  called from within the trie data structure.  It specifies that some      */
/*  positions have already been tested and testing is to start after         */
/*  those.  Must be testing all, not hard or soft, else abort.               */
/*                                                                           */
/*  Parameter stop_on_neg, when true, specifies that the testing is to       */
/*  stop as soon as *avail_cost goes negative.                               */
/*                                                                           */
/*  Parameter *avail_cost is the available cost on entry and the             */
/*  remaining available cost on exit.  KheDrsSignerDoDominates does not      */
/*  consult the cost fields of sig1 and sig2; it assumes that they have      */
/*  already given rise to a suitable initial value of *avail_cost.           */
/*                                                                           */
/*  If fp != NULL, produce debug output while doing this.                    */
/*                                                                           */
/*****************************************************************************/
static bool KheDrsDomTestDominatesStrong(KHE_DRS_DOM_TEST dt, int val1,
  int val2);
static bool KheDrsDomTestDominatesTradeoff(KHE_DRS_DOM_TEST dt,
  int val1, int val2, KHE_COST *avail_cost);
static void KheDrsDomTestDominatesUniform(KHE_DRS_DOM_TEST dt,
  int val1, int val2, KHE_COST *avail_cost);

static bool KheDrsSignerDoDominates(KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2,
  KHE_DRS_SIGNER_TEST test, int trie_start_depth, bool stop_on_neg,
  KHE_COST *avail_cost, int verbosity, int indent, FILE *fp)
{
  KHE_DRS_DOM_TEST dt;
  int start_index, stop_index, dt_index, v1, v2, sig_len;

  /* consistency checks */
  sig_len = HaArrayCount(dsg->dom_tests);
  HnAssert(HaArrayCount(sig1->states) == sig_len,
    "KheDrsSignerDoDominates internal error 1 (count %d != count %d)\n",
    HaArrayCount(sig1->states), sig_len);
  HnAssert(HaArrayCount(sig2->states) == sig_len,
    "KheDrsSignerDoDominates internal error 2 (count %d != count %d)\n",
    HaArrayCount(sig2->states), sig_len);

  /* work out start_index and stop_index */
  switch( test )
  {
    case KHE_DRS_SIGNER_HARD:

      HnAssert(trie_start_depth == 0,
	"KheDrsSignerDoDominates internal error 1");
      start_index = 0;
      stop_index = dsg->last_hard_cost_index + 1;
      break;

    case KHE_DRS_SIGNER_SOFT:

      HnAssert(trie_start_depth == 0,
	"KheDrsSignerDoDominates internal error 2");
      start_index = dsg->last_hard_cost_index + 1;
      stop_index = sig_len;
      break;

    case KHE_DRS_SIGNER_ALL:

      start_index = trie_start_depth;
      stop_index = sig_len;
      break;

    default:

      HnAbort("KheDrsSignerDoDominates internal error 3");
      start_index = 0, stop_index = 0;  /* keep compiler happy */
  }

  /* print spacer and header line */
  KheDrsDominanceDebugHeaderLine(dsg, sig1, sig2, test, *avail_cost,
    indent, fp);

  /* quit immediately if no available cost */
  if( stop_on_neg && *avail_cost < 0 )
    return false;

  /* do the actual test from start_index inclusive to stop_index exclusive */
  for( dt_index = start_index;  dt_index < stop_index;  dt_index++ )
  {
    dt = HaArray(dsg->dom_tests, dt_index);
    v1 = HaArray(sig1->states, dt_index);
    v2 = HaArray(sig2->states, dt_index);
    switch( dt->type )
    {
      case KHE_DRS_DOM_TEST_UNUSED:

	/* unused test, should never happen */
	HnAbort("internal error in KheDrsSignerDominates (UNUSED)");
	break;

      case KHE_DRS_DOM_TEST_STRONG:

	/* strong dominance */
	if( !KheDrsDomTestDominatesStrong(dt, v1, v2) )
	  *avail_cost -= KheCost(1, 0);
	break;

      case KHE_DRS_DOM_TEST_TRADEOFF:

	/* tradeoff dominance */
	if( !KheDrsDomTestDominatesTradeoff(dt, v1, v2, avail_cost) )
	  *avail_cost -= KheCost(1, 0);
	break;

      case KHE_DRS_DOM_TEST_UNIFORM:

	/* uniform dominance */
	KheDrsDomTestDominatesUniform(dt, v1, v2, avail_cost);
	break;

      case KHE_DRS_DOM_TEST_CORR1_PARENT:

	/* parent of correlated parent/child pair */
	/* nothing to do here (the child does it all) */
	break;

      case KHE_DRS_DOM_TEST_CORR1_CHILD:

	/* child of correlated parent/child pair */
	KheDrsSignerCorr1Dominates(dsg, sig1, sig2, dt, dt_index,
	  v1, v2, avail_cost, verbosity, indent + 4, fp);
	break;

      case KHE_DRS_DOM_TEST_CORR2_CHILD:

	/* child of correlated parent/child pair (parent not in signature) */
	KheDrsSignerCorr2Dominates(dsg, sig1, sig2, dt, dt_index,
	  v1, v2, avail_cost, verbosity, indent + 4, fp);
	break;

      case KHE_DRS_DOM_TEST_CORR3_FIRST:

	/* first element of a set of two or more corr3 dom tests */
	KheDrsSignerCorr3Dominates(dsg, sig1, sig2, dt, dt_index,
	  v1, v2, avail_cost, verbosity, indent + 4, fp);
	break;

      case KHE_DRS_DOM_TEST_CORR3_MID:
      case KHE_DRS_DOM_TEST_CORR3_LAST:

	/* non-first element of a set of two or more corr3 dom tests */
	/* nothing to do here (the first element does it all) */
	break;

      case KHE_DRS_DOM_TEST_CORR4_FIRST:

	/* first element of a set of two or more corr4 dom tests */
	KheDrsSignerCorr4Dominates(dsg, sig1, sig2, dt, dt_index,
	  v1, v2, avail_cost, verbosity, indent + 4, fp);
	break;

      case KHE_DRS_DOM_TEST_CORR4_MID:
      case KHE_DRS_DOM_TEST_CORR4_LAST:

	/* non-first element of a set of two or more corr4 dom tests */
	/* nothing to do here (the first element does it all) */
	break;

      default:

	HnAbort("KheDrsSignerDominates internal error 5 (%d)", dt->type);
	break;
    }

    /* print debug line, and quit early if *avail_cost is now negative */
    KheDrsDominanceDebugCmpLine(*avail_cost, dt->type, dt_index, v1, v2,
      dt->monitor, indent, fp);
    if( stop_on_neg && *avail_cost < 0 )
      return false;
  }
  return *avail_cost >= 0;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSignerDominates(KHE_DRS_SIGNER dsg,                           */
/*    KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2,                        */
/*    KHE_COST *avail_cost, int verbosity, int indent, FILE *fp)             */
/*                                                                           */
/*  Return true if sig1 dominates sig2, including comparing their costs.     */
/*  To allow this call to be integrated with others that affect *avail_cost, */
/*  this function assumes that *avail_cost is already initialized, and it    */
/*  carries on based on that initial value.                                  */
/*                                                                           */
/*  If fp != NULL, produce debug output while doing this.                    */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsSignerDominates(KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2,
  KHE_COST *avail_cost, int verbosity, int indent, FILE *fp)
{
  *avail_cost += (sig2->cost - sig1->cost);
  return KheDrsSignerDoDominates(dsg, sig1, sig2, KHE_DRS_SIGNER_ALL,
    0, true, avail_cost, verbosity, indent, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignerDebug(KHE_DRS_SIGNER dsg,                               */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of dsg onto fp with the given verbosity and indent.          */
/*                                                                           */
/*****************************************************************************/
static void KheDrsDomTestDebug(KHE_DRS_DOM_TEST dt, int verbosity,
  int indent, FILE *fp);

static void KheDrsSignerDebug(KHE_DRS_SIGNER dsg,
  int verbosity, int indent, FILE *fp)
{
  KHE_DRS_DOM_TEST dt;  int i;
  fprintf(fp, "%*s[ Signer %s\n", indent, "", KheDrsSignerId(dsg));
  HaArrayForEach(dsg->dom_tests, dt, i)
    KheDrsDomTestDebug(dt, verbosity, indent + 2, fp);
  fprintf(fp, "%*s]\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_SIGNER_SET"                                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SIGNER_SET KheDrsSignerSetMake(KHE_DRS_DAY day,                  */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new signer set.                                                   */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_SIGNER_SET KheDrsSignerSetMake(KHE_DRS_DAY day,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SIGNER_SET res;
  HaMake(res, drs->arena);
  res->day = day;
  HaArrayInit(res->signers, drs->arena);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignerSetOpen(KHE_DRS_SIGNER_SET signer_set,                  */
/*    KHE_DRS_DAY day, KHE_DYNAMIC_RESOURCE_SOLVER drs)                      */
/*                                                                           */
/*  Open signer_set for use on day.                                          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignerSetOpen(KHE_DRS_SIGNER_SET signer_set,
  KHE_DRS_DAY day, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_RESOURCE dr;  int i;  KHE_DRS_RESOURCE_ON_DAY drd;
  KHE_DRS_SIGNER dsg;

  /* one signer for each open resource */
  HnAssert(HaArrayCount(signer_set->signers) == 0,
    "KheDrsSignerSetOpen internal error");
  for( i = 0;  i < KheDrsResourceSetCount(drs->open_resources);  i++ )
  {
    dr = KheDrsResourceSetResource(drs->open_resources, i);
    drd = KheDrsResourceOnDay(dr, day);
    HaArrayAddLast(signer_set->signers, drd->signer);
  }

  /* one additional signer for event resource expressions */
  dsg = KheDrsSignerMake(day, NULL, NULL, drs);
  HaArrayAddLast(signer_set->signers, dsg);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignerSetClose(KHE_DRS_SIGNER_SET signer_set,                 */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Close signer_set.                                                        */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignerSetClose(KHE_DRS_SIGNER_SET signer_set,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SIGNER dsg;

  /* delete and free the last signer (the others were not made here) */
  HnAssert(HaArrayCount(signer_set->signers) > 0,
    "KheDrsSignerSetClose internal error");
  dsg = HaArrayLast(signer_set->signers);
  KheDrsSignerFree(dsg, drs);

  /* clear the signers */
  HaArrayClear(signer_set->signers);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsSignerSetPartialHashSignature(KHE_DRS_SIGNER_SET signer_set,   */
/*    KHE_DRS_SIGNATURE_SET sig_set)                                         */
/*                                                                           */
/*  Hash sig_set, but only the positions that have equality dom tests.       */
/*                                                                           */
/*****************************************************************************/

static int KheDrsSignerSetPartialHashSignature(KHE_DRS_SIGNER_SET signer_set,
  KHE_DRS_SIGNATURE_SET sig_set)
{
  int res, index, val, i, j;  KHE_DRS_SIGNER dsg;  KHE_DRS_SIGNATURE sig;
  HnAssert(HaArrayCount(signer_set->signers)==HaArrayCount(sig_set->signatures),
    "KheDrsSignerSetPartialHashSignature internal error");
  res = 0;
  HaArrayForEach(signer_set->signers, dsg, i)
  {
    sig = HaArray(sig_set->signatures, i);
    HaArrayForEach(dsg->eq_dom_test_indexes, index, j)
    {
      val = HaArray(sig->states, index);
      res = (res + val) * 3;
    }
  }
  if( res < 0 )
    res = - res;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSignerSetPartialEqualSignature(KHE_DRS_SIGNER_SET signer_set, */
/*    KHE_DRS_SIGNATURE_SET sig_set1, KHE_DRS_SIGNATURE_SET sig_set2)        */
/*                                                                           */
/*  Return true if sig1 and sig2 are equal at the positions that have        */
/*  equality dom tests.                                                      */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsSignerSetPartialEqualSignature(KHE_DRS_SIGNER_SET signer_set,
  KHE_DRS_SIGNATURE_SET sig_set1, KHE_DRS_SIGNATURE_SET sig_set2)
{
  int i, j, index;  KHE_DRS_SIGNATURE sig1, sig2;  KHE_DRS_SIGNER dsg;

  HnAssert(HaArrayCount(signer_set->signers) ==
    HaArrayCount(sig_set1->signatures),
    "KheDrsSignerSetPartialEqualSignature internal error 1");
  HnAssert(HaArrayCount(signer_set->signers) ==
    HaArrayCount(sig_set2->signatures),
    "KheDrsSignerSetPartialEqualSignature internal error 2");
  for( i = 0;  i < HaArrayCount(signer_set->signers);  i++ )
  {
    dsg = HaArray(signer_set->signers, i);
    sig1 = HaArray(sig_set1->signatures, i);
    sig2 = HaArray(sig_set2->signatures, i);
    HnAssert(HaArrayCount(sig1->states) == HaArrayCount(sig2->states),
     "KheDrsSignerPartialEqualSignature: signatures have different lengths");
    HaArrayForEach(dsg->eq_dom_test_indexes, index, j)
      if( HaArray(sig1->states, index) != HaArray(sig2->states, index) )
	return false;
  }
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSignerSetDominates(KHE_DRS_SIGNER_SET signer_set,             */
/*    KHE_DRS_SIGNATURE_SET sig_set1, KHE_DRS_SIGNATURE_SET sig_set2,        */
/*    KHE_COST trie_extra_cost, int trie_start_depth, bool use_caching,      */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)  */
/*                                                                           */
/*  Return true if sig_set1 dominates sig_set2, including comparing their    */
/*  costs.  If fp != NULL, produce debug output while doing this.            */
/*                                                                           */
/*  When this function is called from the trie data structure, some of its   */
/*  work may already be done.  It starts comparing from the states at        */
/*  position trie_start_depth (counting across any number of signatures).    */
/*                                                                           */
/*  If trie_start_depth is 0 (as it usually will be) and use_caching is      */
/*  true, then dominance test caching is used to speed up the test.          */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsSignerSetDominates(KHE_DRS_SIGNER_SET signer_set,
  KHE_DRS_SIGNATURE_SET sig_set1, KHE_DRS_SIGNATURE_SET sig_set2,
  KHE_COST trie_extra_cost, int trie_start_depth, bool use_caching,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)
{
  int i, count, last_cache;  KHE_DRS_SIGNER dsg;  KHE_COST avail_cost;
  KHE_DRS_SIGNATURE sig1, sig2;  KHE_DRS_RESOURCE dr;  KHE_DRS_COST_TUPLE ct;
  count = HaArrayCount(signer_set->signers);
  HnAssert(HaArrayCount(sig_set1->signatures) == count,
    "KheDrsSignerSetDominates internal error 1");
  HnAssert(HaArrayCount(sig_set2->signatures) == count,
    "KheDrsSignerSetDominates internal error 1");
  avail_cost = sig_set2->cost - sig_set1->cost - trie_extra_cost;
  if( avail_cost < 0 )
    return false;
  if( drs->solve_dom_approx > 0 )
    avail_cost += (avail_cost * drs->solve_dom_approx) / 10;
  if( trie_start_depth > 0 )
  {
    /* won't happen anyway but do it the easy way if it does */
    HaArrayForEach(signer_set->signers, dsg, i)
    {
      if( trie_start_depth < HaArrayCount(dsg->dom_tests) )
      {
	if( fp != NULL && i > 0 )
	  KheDrsDominanceDebugSpacer(indent, fp);
	sig1 = HaArray(sig_set1->signatures, i);
	sig2 = HaArray(sig_set2->signatures, i);
	if( !KheDrsSignerDoDominates(dsg, sig1, sig2, KHE_DRS_SIGNER_ALL,
	      trie_start_depth, true, &avail_cost, verbosity,indent,fp) )
	  return false;
	trie_start_depth = 0;
      }
      else
	trie_start_depth -= HaArrayCount(dsg->dom_tests);
    }
  }
  else if( USE_DOM_CACHING && use_caching )
  {
    /* used a cached value for all positions except the last */
    last_cache = HaArrayCount(signer_set->signers) - 2;
    for( i = 0;  i <= last_cache;  i++ )
    {
      if( fp != NULL && i > 0 )
	KheDrsDominanceDebugSpacer(indent, fp);
      dr = KheDrsResourceSetResource(drs->open_resources, i);
      dsg = HaArray(signer_set->signers, i);
      sig1 = HaArray(sig_set1->signatures, i);
      HnAssert(sig1->asst_to_shift_index >= 0,
	"KheDrsSignerSetDominates internal error 3");
      sig2 = HaArray(sig_set2->signatures, i);
      HnAssert(sig2->asst_to_shift_index >= 0,
	"KheDrsSignerSetDominates internal error 4");
      ct = KheDrsDim2TableGet2(dr->expand_dom_test_cache,
	sig1->asst_to_shift_index, sig2->asst_to_shift_index);
      avail_cost += ct.psi;
      if( fp != NULL )
	fprintf(fp, "%*s%8.5f  (cache[%d, %d] = %.5f)\n", indent, "",
	  KheCostShow(avail_cost), sig1->asst_to_shift_index,
	  sig2->asst_to_shift_index, KheCostShow(ct.psi));
      if( avail_cost < 0 )
	return false;
    }

    /* regular test for the last position */
    if( fp != NULL && i > 0 )
      KheDrsDominanceDebugSpacer(indent, fp);
    dsg = HaArrayLast(signer_set->signers);
    sig1 = HaArrayLast(sig_set1->signatures);
    sig2 = HaArrayLast(sig_set2->signatures);
    if( !KheDrsSignerDoDominates(dsg, sig1, sig2, KHE_DRS_SIGNER_ALL,
	  0, true, &avail_cost, verbosity, indent, fp) )
      return false;
  }
  else
  {
    /* visit hard constraints first; they often end the test quickly */
    HaArrayForEach(signer_set->signers, dsg, i)
    {
      sig1 = HaArray(sig_set1->signatures, i);
      sig2 = HaArray(sig_set2->signatures, i);
      if( !KheDrsSignerDoDominates(dsg, sig1, sig2, KHE_DRS_SIGNER_HARD,
	    0, true, &avail_cost, verbosity, indent, fp) )
	return false;
    }

    /* visit soft constraints */
    HaArrayForEach(signer_set->signers, dsg, i)
    {
      sig1 = HaArray(sig_set1->signatures, i);
      sig2 = HaArray(sig_set2->signatures, i);
      if( !KheDrsSignerDoDominates(dsg, sig1, sig2, KHE_DRS_SIGNER_SOFT,
	  0, true, &avail_cost, verbosity, indent, fp) )
	return false;
    }
  }
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignerSetDebug(KHE_DRS_SIGNER_SET signer_set,                 */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of signer_set onto fp with the given verbosity and indent.   */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignerSetDebug(KHE_DRS_SIGNER_SET signer_set,
  int verbosity, int indent, FILE *fp)
{
  KHE_DRS_SIGNER dsg;  int i;
  fprintf(fp, "%*s[ SignerSet %s\n", indent, "", KheDrsDayId(signer_set->day));
  HaArrayForEach(signer_set->signers, dsg, i)
  {
    if( i > 0 )
      KheDrsDominanceDebugSpacer(indent + 2, fp);
    KheDrsSignerDebug(dsg, verbosity, indent + 2, fp);
  }
  fprintf(fp, "%*s]\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_DOM_KIND"                                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST_TYPE KheDomKindToDomTestType(KHE_DRS_DOM_KIND dom_kind) */
/*                                                                           */
/*  Return the dom test type of dom_kind.                                    */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DOM_TEST_TYPE KheDomKindToDomTestType(KHE_DRS_DOM_KIND dom_kind)
{
  switch( dom_kind )
  {
    case KHE_DRS_DOM_LIST_NONE:		return KHE_DRS_DOM_TEST_UNUSED;
    case KHE_DRS_DOM_LIST_STRONG:	return KHE_DRS_DOM_TEST_STRONG;
    case KHE_DRS_DOM_LIST_TRADEOFF:	return KHE_DRS_DOM_TEST_TRADEOFF;
    case KHE_DRS_DOM_LIST_UNIFORM:	return KHE_DRS_DOM_TEST_UNIFORM;
    case KHE_DRS_DOM_HASH_WEAK:		return KHE_DRS_DOM_TEST_UNUSED;
    case KHE_DRS_DOM_HASH_MEDIUM:	return KHE_DRS_DOM_TEST_STRONG;
    case KHE_DRS_DOM_TRIE_STRONG:	return KHE_DRS_DOM_TEST_STRONG;
    case KHE_DRS_DOM_TRIE_TRADEOFF:	return KHE_DRS_DOM_TEST_TRADEOFF;
    case KHE_DRS_DOM_INDEXED_TRADEOFF:	return KHE_DRS_DOM_TEST_TRADEOFF;
    case KHE_DRS_DOM_INDEXED_UNIFORM:	return KHE_DRS_DOM_TEST_UNIFORM;

    default:

      HnAbort("KheDomKindToDomTestType: unknown dom_kind (%d)", dom_kind);
      return 0;  /* keep compiler happy */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST_TYPE KheDomKindCheckConsistency(                        */
/*    KHE_DRS_DOM_KIND main_dom_kind, bool cache,                            */
/*    KHE_DRS_DOM_KIND cache_dom_kind)                                       */
/*                                                                           */
/*  Check that main_dom_kind, cache, and cache_dom_kind are consistent,      */
/*  and return the consistent dom test type.                                 */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DOM_TEST_TYPE KheDomKindCheckConsistency(
  KHE_DRS_DOM_KIND main_dom_kind, bool cache, KHE_DRS_DOM_KIND cache_dom_kind)
{
  KHE_DRS_DOM_TEST_TYPE main_dom_test_type, cache_dom_test_type;
  main_dom_test_type = KheDomKindToDomTestType(main_dom_kind);
  if( cache )
  {
    cache_dom_test_type = KheDomKindToDomTestType(cache_dom_kind);
    HnAssert(main_dom_test_type == KHE_DRS_DOM_TEST_UNUSED ||
      cache_dom_test_type == KHE_DRS_DOM_TEST_UNUSED ||
      main_dom_test_type == cache_dom_test_type, "KheDomKindCheckConsistency:"
      "inconsistent main_dom_kind and cache_dom_kind arguments");
  }
  return main_dom_test_type;
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheDomKindShow(KHE_DRS_DOM_KIND dom_kind)                          */
/*                                                                           */
/*  Return a suitable label for dom_kind.                                    */
/*                                                                           */
/*****************************************************************************/

static char *KheDomKindShow(KHE_DRS_DOM_KIND dom_kind)
{
  switch( dom_kind )
  {
    case KHE_DRS_DOM_LIST_NONE:		return "ListNone";
    case KHE_DRS_DOM_LIST_STRONG:	return "ListStrong";
    case KHE_DRS_DOM_LIST_TRADEOFF:	return "ListTradeoff";
    case KHE_DRS_DOM_LIST_UNIFORM:	return "ListUniform";
    case KHE_DRS_DOM_HASH_WEAK:		return "HashWeak";
    case KHE_DRS_DOM_HASH_MEDIUM:	return "HashMedium";
    case KHE_DRS_DOM_TRIE_STRONG:	return "TrieStrong";
    case KHE_DRS_DOM_TRIE_TRADEOFF:	return "TrieTradeoff";
    case KHE_DRS_DOM_INDEXED_TRADEOFF:	return "IndexedTradeoff";
    case KHE_DRS_DOM_INDEXED_UNIFORM:	return "IndexedUniform";

    default:

      HnAbort("KheDomKindShow: unknown dom_kind (%d)", dom_kind);
      return NULL;  /* keep compiler happy */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_COST_TUPLE"                                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_COST_TUPLE KheDrsCostTupleMake(KHE_COST psi, KHE_COST psi0,      */
/*    bool psi_plus_defined, KHE_COST psi_plus)                              */
/*                                                                           */
/*  Make a cost tuple object with these attributes.                          */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_COST_TUPLE KheDrsCostTupleMake(KHE_COST psi, KHE_COST psi0,
  bool psi_plus_defined, KHE_COST psi_plus)
{
  KHE_DRS_COST_TUPLE res;
  res.psi = psi;
  res.psi0 = psi0;
  res.psi_plus_defined = psi_plus_defined;
  res.psi_plus = psi_plus;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsCostTupleDebug(KHE_DRS_COST_TUPLE ct, int verbosity,          */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of ct onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsCostTupleDebug(KHE_DRS_COST_TUPLE ct, int verbosity,
  int indent, FILE *fp)
{
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  if( ct.psi_plus_defined )
    fprintf(fp, "(%.5f, %.5f, %.5f)", KheCostShow(ct.psi),
      KheCostShow(ct.psi0), KheCostShow(ct.psi_plus));
  else
    fprintf(fp, "(%.5f, %.5f)", KheCostShow(ct.psi), KheCostShow(ct.psi0));
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_DIM1_TABLE"                                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DIM1_TABLE KheDrsDim1TableMake(KHE_DYNAMIC_RESOURCE_SOLVER drs)  */
/*                                                                           */
/*  Make a new, empty dim1 table.  It will work out its offset later.        */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DIM1_TABLE KheDrsDim1TableMake(KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DIM1_TABLE res;
  if( HaArrayCount(drs->table1_free_list) > 0 )
  {
    res = HaArrayLastAndDelete(drs->table1_free_list);
    HaArrayClear(res->children);
  }
  else
  {
    HaMake(res, drs->arena);
    HaArrayInit(res->children, drs->arena);
  }
  res->offset = 0;  /* actually undefined when the child array is empty */
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDim1TablePut(KHE_DRS_DIM1_TABLE d1, int index,                */
/*    KHE_DRS_COST_TUPLE val)                                                */
/*                                                                           */
/*  Add val to d1 at index.  If this is the first child, set the offset.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDim1TablePut(KHE_DRS_DIM1_TABLE d1, int index,
  KHE_DRS_COST_TUPLE val)
{
  if( HaArrayCount(d1->children) == 0 )
  {
    /* first child, set offset */
    d1->offset = index;
  }
  else
  {
    /* not first child, must be next though */
    HnAssert(index == d1->offset + HaArrayCount(d1->children),
      "KheDrsDim1TablePut: index is %d when %d expected", index,
      d1->offset + HaArrayCount(d1->children));
  }
  HaArrayAddLast(d1->children, val);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_COST_TUPLE KheDrsDim1TableGet(KHE_DRS_DIM1_TABLE d1, int index)  */
/*                                                                           */
/*  Get the value at this index.  There must be one.                         */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_COST_TUPLE KheDrsDim1TableGet(KHE_DRS_DIM1_TABLE d1, int index)
{
  int pos;
  pos = index - d1->offset;
  HnAssert(0 <= pos && pos < HaArrayCount(d1->children),
    "KheDrsDim1TableGet: index %d out of range %d .. %d", index,
    d1->offset, HaArrayCount(d1->children) + d1->offset - 1);
  return HaArray(d1->children, pos);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDim1TableFree(KHE_DRS_DIM1_TABLE d1,                          */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Free d1.                                                                 */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDim1TableFree(KHE_DRS_DIM1_TABLE d1,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  HaArrayAddLast(drs->table1_free_list, d1);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDim1TableDebug(KHE_DRS_DIM1_TABLE d1, int verbosity,          */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of d1 onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDim1TableDebug(KHE_DRS_DIM1_TABLE d1, int verbosity,
  int indent, FILE *fp)
{
  KHE_DRS_COST_TUPLE ct;  int i;
  fprintf(fp, "%*s[ Dim1 (%d .. %d)\n", indent, "", d1->offset,
    d1->offset + HaArrayCount(d1->children) - 1);
  HaArrayForEach(d1->children, ct, i)
  {
    if( i == 0 )
      fprintf(fp, "%*s", indent + 2, "");
    else if( i % 4 == 0 )
      fprintf(fp, "\n%*s", indent + 2, "");
    else
      fprintf(fp, ", ");
    KheDrsCostTupleDebug(ct, verbosity, -1, fp);
  }
  fprintf(fp, "\n%*s]\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_DIM2_TABLE"                                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DIM2_TABLE KheDrsDim2TableMake(char *debug_str, HA_ARENA a)      */
/*                                                                           */
/*  Make a new, empty dim1 table.  It will work out its offset later.        */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DIM2_TABLE KheDrsDim2TableMake(char *debug_str, HA_ARENA a)
{
  KHE_DRS_DIM2_TABLE res;
  HaMake(res, a);
  res->offset = 0;  /* actually undefined when the child array is empty */
  HaArrayInit(res->children, a);
  res->debug_str = debug_str;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDim2TablePut(KHE_DRS_DIM2_TABLE d2, int index,                */
/*    KHE_DRS_DIM1_TABLE val)                                                */
/*                                                                           */
/*  Add val to d2 at index.  If this is the first child, set the offset.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDim2TablePut(KHE_DRS_DIM2_TABLE d2, int index2,
  int index1, KHE_DRS_COST_TUPLE ct, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DIM1_TABLE d1;  int pos;

  /* set offset if this is the first insertion */
  if( HaArrayCount(d2->children) == 0 )
    d2->offset = index2;

  /* make sure index2 is within, or one step beyond, the current range */
  pos = index2 - d2->offset;
  HnAssert(0 <= pos && pos <= HaArrayCount(d2->children),
    "KheDrsDim2TablePut: index2 %d is out of range %d .. %d", index2,
    d2->offset, d2->offset + HaArrayCount(d2->children));

  /* find or make-and-add d1, the sub-array to insert into */
  if( pos < HaArrayCount(d2->children) )
    d1 = HaArray(d2->children, pos);
  else
  {
    d1 = KheDrsDim1TableMake(drs);
    HaArrayAddLast(d2->children, d1);
  }
  
  /* do the insertion */
  KheDrsDim1TablePut(d1, index1, ct);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DIM1_TABLE KheDrsDim2TableGet(KHE_DRS_DIM2_TABLE d2, int index2) */
/*                                                                           */
/*  Get the value at this index.  There must be one.                         */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DIM1_TABLE KheDrsDim2TableGet(KHE_DRS_DIM2_TABLE d2, int index2)
{
  int pos;
  pos = index2 - d2->offset;
  HnAssert(0 <= pos && pos < HaArrayCount(d2->children),
    "KheDrsDim2TableGet: index2 %d out of range %d .. %d", index2,
    d2->offset, HaArrayCount(d2->children) + d2->offset - 1);
  return HaArray(d2->children, pos);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_COST_TUPLE KheDrsDim2TableGet2(KHE_DRS_DIM2_TABLE d2,            */
/*    int index2, int index1)                                                */
/*                                                                           */
/*  Get the cost tuple at d2[index2, index1].                                */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_COST_TUPLE KheDrsDim2TableGet2(KHE_DRS_DIM2_TABLE d2,
  int index2, int index1)
{
  return KheDrsDim1TableGet(KheDrsDim2TableGet(d2, index2), index1);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDim2TableClear(KHE_DRS_DIM2_TABLE d2,                         */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Clear d2, placing all its d1's onto a free list.                         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDim2TableClear(KHE_DRS_DIM2_TABLE d2,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DIM1_TABLE d1;  int i;
  HaArrayForEach(d2->children, d1, i)
    KheDrsDim1TableFree(d1, drs);
  HaArrayClear(d2->children);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDim2TableDebug(KHE_DRS_DIM2_TABLE d2, int verbosity,          */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of d2 onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDim2TableDebug(KHE_DRS_DIM2_TABLE d2, int verbosity,
  int indent, FILE *fp)
{
  KHE_DRS_DIM1_TABLE d1;  int i;
  fprintf(fp, "%*s[ Dim2 %s(%d .. %d)\n", indent, "",
    d2->debug_str == NULL ? "" : d2->debug_str,
    d2->offset, d2->offset + HaArrayCount(d2->children) - 1);
  HaArrayForEach(d2->children, d1, i)
    KheDrsDim1TableDebug(d1, verbosity, indent + 2, fp);
  fprintf(fp, "%*s]\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_DIM3_TABLE"                                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DIM3_TABLE KheDrsDim3TableMake(char *debug_str, HA_ARENA a)      */
/*                                                                           */
/*  Make a new, empty dim3 table.  It will work out its offset later.        */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DIM3_TABLE KheDrsDim3TableMake(char *debug_str, HA_ARENA a)
{
  KHE_DRS_DIM3_TABLE res;
  HaMake(res, a);
  res->offset = 0;  /* actually undefined when the child array is empty */
  HaArrayInit(res->children, a);
  res->debug_str = debug_str;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDim3TablePut(KHE_DRS_DIM3_TABLE d3, int index3, int index2,   */
/*    int index1, KHE_DRS_COST_TUPLE ct, KHE_DYNAMIC_RESOURCE_SOLVER drs)    */
/*                                                                           */
/*  Add c3 to d3.  If this is the first insertion, set the offset.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDim3TablePut(KHE_DRS_DIM3_TABLE d3, int index3, int index2,
  int index1, KHE_DRS_COST_TUPLE ct, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DIM2_TABLE d2;  int pos;  HA_ARENA a;  char *debug_str;

  /* set offset if this is the first insertion */
  if( HaArrayCount(d3->children) == 0 )
    d3->offset = index3;

  /* make sure index3 is within, or one step beyond, the current range */
  pos = index3 - d3->offset;
  HnAssert(0 <= pos && pos <= HaArrayCount(d3->children),
    "KheDrsDim3TablePut: index3 %d is out of range %d .. %d", index3,
    d3->offset, d3->offset + HaArrayCount(d3->children));

  /* find or make-and-add d2, the sub-array to insert into */
  if( pos < HaArrayCount(d3->children) )
    d2 = HaArray(d3->children, pos);
  else
  {
    a = HaArrayArena(d3->children);
    debug_str = HnStringMake(a, "%s:%d", d3->debug_str, index3);
    d2 = KheDrsDim2TableMake(debug_str, a);
    HaArrayAddLast(d3->children, d2);
  }
  
  /* do the insertion */
  KheDrsDim2TablePut(d2, index2, index1, ct, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DIM2_TABLE KheDrsDim3TableGet(KHE_DRS_DIM3_TABLE d3, int index3) */
/*                                                                           */
/*  Get the value at this index.  There must be one.                         */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DIM2_TABLE KheDrsDim3TableGet(KHE_DRS_DIM3_TABLE d3, int index3)
{
  int pos;
  pos = index3 - d3->offset;
  HnAssert(0 <= pos && pos < HaArrayCount(d3->children),
    "KheDrsDim3TableGet: index %d out of range %d .. %d", index3,
    d3->offset, HaArrayCount(d3->children) + d3->offset - 1);
  return HaArray(d3->children, pos);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDim3TableDebug(KHE_DRS_DIM3_TABLE d3, int verbosity,          */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of d3 onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDim3TableDebug(KHE_DRS_DIM3_TABLE d3, int verbosity,
  int indent, FILE *fp)
{
  KHE_DRS_DIM2_TABLE d2;  int i;
  fprintf(fp, "%*s[ Dim3 %s(%d .. %d)\n", indent, "",
    d3->debug_str == NULL ? "" : d3->debug_str,
    d3->offset, d3->offset + HaArrayCount(d3->children) - 1);
  HaArrayForEach(d3->children, d2, i)
    KheDrsDim2TableDebug(d2, verbosity, indent + 2, fp);
  fprintf(fp, "%*s]\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_DIM4_TABLE"                                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DIM4_TABLE KheDrsDim4TableMake(char *debug_str, HA_ARENA a)      */
/*                                                                           */
/*  Make a new, empty Dim4 table.  It will work out its offset later.        */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DIM4_TABLE KheDrsDim4TableMake(char *debug_str, HA_ARENA a)
{
  KHE_DRS_DIM4_TABLE res;
  HaMake(res, a);
  res->offset = 0;  /* actually undefined when the child array is empty */
  HaArrayInit(res->children, a);
  res->debug_str = debug_str;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDim4TablePut(KHE_DRS_DIM4_TABLE d4, int index4, int index3,   */
/*    int index2, int index1, KHE_DRS_COST_TUPLE ct,                         */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Add ct to d4.  If this is the first insertion, set the offset.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDim4TablePut(KHE_DRS_DIM4_TABLE d4, int index4, int index3,
  int index2, int index1, KHE_DRS_COST_TUPLE ct,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DIM3_TABLE d3;  int pos;  HA_ARENA a;  char *debug_str;

  /* set offset if this is the first insertion */
  if( HaArrayCount(d4->children) == 0 )
    d4->offset = index4;

  /* make sure index4 is within, or one step beyond, the current range */
  pos = index4 - d4->offset;
  HnAssert(0 <= pos && pos <= HaArrayCount(d4->children),
    "KheDrsDim4TablePut: index4 %d is out of range %d .. %d", index4,
    d4->offset, d4->offset + HaArrayCount(d4->children));

  /* find or make-and-add d3, the sub-array to insert into */
  if( pos < HaArrayCount(d4->children) )
    d3 = HaArray(d4->children, pos);
  else
  {
    a = HaArrayArena(d4->children);
    debug_str = HnStringMake(a, "%s:%d", d4->debug_str, index4);
    d3 = KheDrsDim3TableMake(debug_str, a);
    HaArrayAddLast(d4->children, d3);
  }
  
  /* do the insertion */
  KheDrsDim3TablePut(d3, index3, index2, index1, ct, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DIM3_TABLE KheDrsDim4TableGet(KHE_DRS_DIM4_TABLE d4, int index4) */
/*                                                                           */
/*  Get the value at this index.  There must be one.                         */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DIM3_TABLE KheDrsDim4TableGet(KHE_DRS_DIM4_TABLE d4, int index4)
{
  int pos;
  pos = index4 - d4->offset;
  HnAssert(0 <= pos && pos < HaArrayCount(d4->children),
    "KheDrsDim4TableGet: index %d out of range %d .. %d", index4,
    d4->offset, HaArrayCount(d4->children) + d4->offset - 1);
  return HaArray(d4->children, pos);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_COST_TUPLE KheDrsDim4TableGet4(KHE_DRS_DIM4_TABLE d4,            */
/*    int index4, int index3, int index2, int index1)                        */
/*                                                                           */
/*  Get the cost tuple at d4[index4, index3, index2, index1].                */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_COST_TUPLE KheDrsDim4TableGet4(KHE_DRS_DIM4_TABLE d4,
  int index4, int index3, int index2, int index1)
{
  return KheDrsDim2TableGet2(KheDrsDim3TableGet(KheDrsDim4TableGet(d4, index4),
    index3), index2, index1);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDim4TableDebug(KHE_DRS_DIM4_TABLE d4, int verbosity,          */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of d4 onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDim4TableDebug(KHE_DRS_DIM4_TABLE d4, int verbosity,
  int indent, FILE *fp)
{
  KHE_DRS_DIM3_TABLE d3;  int i;
  fprintf(fp, "%*s[ Dim4 %s(%d .. %d)\n", indent, "",
    d4->debug_str == NULL ? "" : d4->debug_str,
    d4->offset, d4->offset + HaArrayCount(d4->children) - 1);
  HaArrayForEach(d4->children, d3, i)
    KheDrsDim3TableDebug(d3, verbosity, indent + 2, fp);
  fprintf(fp, "%*s]\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_DIM5_TABLE"                                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DIM5_TABLE KheDrsDim5TableMake(char *debug_str, HA_ARENA a)      */
/*                                                                           */
/*  Make a new, empty Dim5 table.  It will work out its offset later.        */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DIM5_TABLE KheDrsDim5TableMake(char *debug_str, HA_ARENA a)
{
  KHE_DRS_DIM5_TABLE res;
  HaMake(res, a);
  res->offset = 0;  /* actually undefined when the child array is empty */
  HaArrayInit(res->children, a);
  res->debug_str = debug_str;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDim5TablePut(KHE_DRS_DIM5_TABLE d5, int index5, int index4,   */
/*    int index3, int index2, int index1, KHE_DRS_COST_TUPLE ct,             */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Add ct to d5.  If this is the first insertion, set the offset.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDim5TablePut(KHE_DRS_DIM5_TABLE d5, int index5, int index4,
  int index3, int index2, int index1, KHE_DRS_COST_TUPLE ct,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DIM4_TABLE d4;  int pos;  char *debug_str;  HA_ARENA a;

  /* set offset if this is the first insertion */
  if( HaArrayCount(d5->children) == 0 )
    d5->offset = index5;

  /* make sure index5 is within, or one step beyond, the current range */
  pos = index5 - d5->offset;
  HnAssert(0 <= pos && pos <= HaArrayCount(d5->children),
    "KheDrsDim5TablePut: index5 %d is out of range %d .. %d", index5,
    d5->offset, d5->offset + HaArrayCount(d5->children));

  /* find or make-and-add d4, the sub-array to insert into */
  if( pos < HaArrayCount(d5->children) )
    d4 = HaArray(d5->children, pos);
  else
  {
    a = HaArrayArena(d5->children);
    debug_str = HnStringMake(a, "%s:%d", d5->debug_str, index5);
    d4 = KheDrsDim4TableMake(debug_str, a);
    HaArrayAddLast(d5->children, d4);
  }
  
  /* do the insertion */
  KheDrsDim4TablePut(d4, index4, index3, index2, index1, ct, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DIM4_TABLE KheDrsDim5TableGet(KHE_DRS_DIM5_TABLE d5, int index5) */
/*                                                                           */
/*  Get the value at this index.  There must be one.                         */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DIM4_TABLE KheDrsDim5TableGet(KHE_DRS_DIM5_TABLE d5, int index5)
{
  int pos;
  pos = index5 - d5->offset;
  HnAssert(0 <= pos && pos < HaArrayCount(d5->children),
    "KheDrsDim5TableGet: index %d out of range %d .. %d", index5,
    d5->offset, HaArrayCount(d5->children) + d5->offset - 1);
  return HaArray(d5->children, pos);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DIM2_TABLE KheDrsDim5TableGet3(KHE_DRS_DIM5_TABLE d5,            */
/*    int index5, int index4, int index3)                                    */
/*                                                                           */
/*  Get the two-dimensional table that results from applying three indexes   */
/*  to five-dimensional table d5.                                            */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DIM2_TABLE KheDrsDim5TableGet3(KHE_DRS_DIM5_TABLE d5,
  int index5, int index4, int index3)
{
  return KheDrsDim3TableGet(KheDrsDim4TableGet(KheDrsDim5TableGet(d5, index5),
    index4), index3);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDim5TableDebug(KHE_DRS_DIM5_TABLE d5, int verbosity,          */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of d5 onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDim5TableDebug(KHE_DRS_DIM5_TABLE d5, int verbosity,
  int indent, FILE *fp)
{
  KHE_DRS_DIM4_TABLE d4;  int i;
  fprintf(fp, "%*s[ Dim5 %s(%d .. %d)\n", indent, "",
    d5->debug_str == NULL ? "" : d5->debug_str,
    d5->offset, d5->offset + HaArrayCount(d5->children) - 1);
  HaArrayForEach(d5->children, d4, i)
    KheDrsDim4TableDebug(d4, verbosity, indent + 2, fp);
  fprintf(fp, "%*s]\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_DOM_TEST_TYPE"                                        */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  char *KheDrsDomTestTypeShow(KHE_DRS_DOM_TEST_TYPE type)                  */
/*                                                                           */
/*  Return a one-character string representation of type.                    */
/*                                                                           */
/*****************************************************************************/

static char *KheDrsDomTestTypeShow(KHE_DRS_DOM_TEST_TYPE type)
{
  switch( type )
  {
    case KHE_DRS_DOM_TEST_UNUSED:		return "!!";
    case KHE_DRS_DOM_TEST_STRONG:		return "ST";
    case KHE_DRS_DOM_TEST_TRADEOFF:		return "TR";
    case KHE_DRS_DOM_TEST_UNIFORM:		return "UN";
    case KHE_DRS_DOM_TEST_CORR1_PARENT:		return "1P";
    case KHE_DRS_DOM_TEST_CORR1_CHILD:		return "1C";
    case KHE_DRS_DOM_TEST_CORR2_CHILD:		return "2C";
    case KHE_DRS_DOM_TEST_CORR3_FIRST:		return "3F";
    case KHE_DRS_DOM_TEST_CORR3_MID:		return "3M";
    case KHE_DRS_DOM_TEST_CORR3_LAST:		return "3L";
    case KHE_DRS_DOM_TEST_CORR4_FIRST:		return "4F";
    case KHE_DRS_DOM_TEST_CORR4_MID:		return "4M";
    case KHE_DRS_DOM_TEST_CORR4_LAST:		return "4L";
    default:					return "??";
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_DOM_TEST"                                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsDomTestMake(KHE_DRS_DOM_TEST_TYPE type,           */
/*    KHE_DRS_EXPR e, bool allow_zero, int min_limit, int max_limit,         */
/*    int a, int b, KHE_COST tradeoff_cost,                                  */
/*    KHE_DRS_DIM2_TABLE main_dom_table2, KHE_DRS_DIM4_TABLE corr_dom_table4,*/
/*    KHE_MONITOR m, KHE_DYNAMIC_RESOURCE_SOLVER drs)                        */
/*                                                                           */
/*  Make and return a new dominance test object with these attributes.       */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DOM_TEST KheDrsDomTestMake(KHE_DRS_DOM_TEST_TYPE type,
  KHE_DRS_EXPR e, bool allow_zero, int min_limit, int max_limit,
  int a, int b, KHE_COST tradeoff_cost,
  KHE_DRS_DIM2_TABLE main_dom_table2, KHE_DRS_DIM4_TABLE corr_dom_table4,
  KHE_MONITOR m, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DOM_TEST res;
  if( HaArrayCount(drs->dom_test_free_list) > 0 )
    res = HaArrayLastAndDelete(drs->dom_test_free_list);
  else
    HaMake(res, drs->arena);
  res->type = type;
  res->correlated_delta = 0;
  res->expr = e;
  res->allow_zero = allow_zero;
  res->min_limit = min_limit;
  res->max_limit = max_limit;
  res->a = a;
  res->b = b;
  res->tradeoff_cost = tradeoff_cost;
  res->main_dom_table2 = main_dom_table2;
  res->corr_dom_table4 = corr_dom_table4;
  res->monitor = m;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsDomTestDominatesStrong(KHE_DRS_DOM_TEST dt, int val1,int val2)*/
/*                                                                           */
/*  Return true if val1 dominates val2, according to dt.                     */
/*                                                                           */
/*  Implementation note.  This formula is derived in the Appendix to the     */
/*  User's Guide, where it is called dom(val1, val2).                        */
/*                                                                           */
/*****************************************************************************/

static bool tilde(KHE_DRS_DOM_TEST dt, int val1, int val2)
{
  return (dt->allow_zero ? (val1 == 0) == (val2 == 0) : true);
}

static bool KheDrsDomMax(KHE_DRS_DOM_TEST dt, int val1, int val2)
{
  return val1 <= dt->a || val1 <= val2;
}

static bool KheDrsDomMin(KHE_DRS_DOM_TEST dt, int val1, int val2)
{
  return val1 >= dt->b || (tilde(dt, val1, val2) && val1 >= val2);
}

static bool KheDrsDomTestDominatesStrong(KHE_DRS_DOM_TEST dt, int val1,int val2)
{
  return KheDrsDomMax(dt, val1, val2) && KheDrsDomMin(dt, val1, val2);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsTryTradeoff(KHE_DRS_DOM_TEST dt, int delta_val,               */
/*    KHE_COST *cost1, KHE_COST cost2)                                       */
/*                                                                           */
/*  If dt->tradeoff_allowed, carry out tradeoff dominance with value change  */
/*  delta_val.  Otherwise return false.                                      */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsTryTradeoff(KHE_DRS_DOM_TEST dt, int delta_val,
  KHE_COST *avail_cost)
{
  HnAssert(delta_val >= 0, "KheDrsTryTradeoff internal error");
  *avail_cost -= dt->tradeoff_cost * delta_val;
  return *avail_cost >= 0;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsDomTestDominatesTradeoff(KHE_DRS_DOM_TEST dt,                 */
/*    int val1, KHE_COST *cost1, int val2, KHE_COST cost2)                   */
/*                                                                           */
/*  Return true if val1 dominates val2, according to dt, allowing for a      */
/*  tradeoff.                                                                */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsDomTestDominatesTradeoff(KHE_DRS_DOM_TEST dt,
  int val1, int val2, KHE_COST *avail_cost)
{
  if( KheDrsDomMax(dt, val1, val2) )
  {
    if( KheDrsDomMin(dt, val1, val2) )
      return true;
    else if( dt->allow_zero )
      return false;
    else
      return KheDrsTryTradeoff(dt, val2 - val1, avail_cost);
  }
  else
  {
    if( KheDrsDomMin(dt, val1, val2) )
      return KheDrsTryTradeoff(dt, val1 - val2, avail_cost);
    else
      return false;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDomTestDominatesUniform(KHE_DRS_DOM_TEST dt,                  */
/*    int val1, int val2, KHE_COST *avail_cost)                              */
/*                                                                           */
/*  Check whether val1 dominates val2 using uniform dominance, updating      */
/*  *avail_cost by the change in available cost.                             */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDomTestDominatesUniform(KHE_DRS_DOM_TEST dt,
  int val1, int val2, KHE_COST *avail_cost)
{
  if( DEBUG60 || DEBUG44_MONITOR(dt->monitor) )
    fprintf(stderr, "  [ KheDrsDomTestDominatesUniform(%s, %d, %d)\n",
      dt->monitor != NULL ? KheMonitorId(dt->monitor) : "-", val1, val2);
  *avail_cost += KheDrsDim2TableGet2(dt->main_dom_table2, val1, val2).psi;
  if( DEBUG60 || DEBUG44_MONITOR(dt->monitor) )
    fprintf(stderr, "  ]\n");
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsDomTestIsStrictEquality(KHE_DRS_DOM_TEST dom_test)            */
/*                                                                           */
/*  Return true if dom_test amounts to a strict equality test.  As shown     */
/*  in the documentation, the allow_zero flag does not affect the result.    */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsDomTestIsStrictEquality(KHE_DRS_DOM_TEST dom_test)
{
  return dom_test->a < 0 && dom_test->b >= INT_MAX;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDomTestDominatingSet(KHE_DRS_DOM_TEST dt, int x,              */
/*    int *from_val, int *to_val, bool *also_zero)                           */
/*                                                                           */
/*  Set [*from_val, *to_val] and *also_zero to the set of non-negative       */
/*  values that dominate x under dt.  That is, set it to                     */
/*                                                                           */
/*    { v | v >= 0 && KheDrsDomTestDominatesStrong(dt, v, x) }               */
/*                                                                           */
/*  The formula for this is derived in the User's Guide.                     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDomTestDominatingSet(KHE_DRS_DOM_TEST dt, int x,
  int *from_val, int *to_val, bool *also_zero)
{
  if( dt->allow_zero && x == 0 )
  {
    *from_val = max(0, dt->b);
    *to_val = max(dt->a, 0);
    *also_zero = (*from_val > 0);
  }
  else
  {
    *from_val = min(dt->b, x);
    if( *from_val < 0 )  *from_val = 0;
    *to_val = max(dt->a, x);
    *also_zero = false;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDomTestDominatedSet(KHE_DRS_DOM_TEST dt, int x,               */
/*    int *from_val, int *to_val, bool *also_zero)                           */
/*                                                                           */
/*  Set [*from_val, *to_val] and *also_zero to the set of non-negative       */
/*  values that x dominates under dt.  That is, set it to                    */
/*                                                                           */
/*    { v | v >= 0 && KheDrsDomTestDominatesStrong(dt, x, val) }             */
/*                                                                           */
/*  This formula is derived in the Appendix to the User's Guide.             */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDomTestDominatedSet(KHE_DRS_DOM_TEST dt, int x,
  int *from_val, int *to_val, bool *also_zero)
{
  if( dt->allow_zero && x >= 1 )
  {
    *from_val = (x <= dt->a ? 1 : max(1, x));
    *to_val = (x >= dt->b ? INT_MAX : x);
    if( x <= max(dt->a, 0) && x >= dt->b )
    {
      /* make sure 0 is in the set */
      if( *from_val == 1 )
	*from_val = 0;
      *also_zero = (*from_val > 0);
    }
    else
      *also_zero = false;
  }
  else
  {
    *from_val = (x <= dt->a ? 0 : x);
    *to_val = (x >= dt->b ? INT_MAX : x);
    *also_zero = false;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDomTestDebug(KHE_DRS_DOM_TEST dt, int verbosity,              */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of dt onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDomTestDebug(KHE_DRS_DOM_TEST dt, int verbosity,
  int indent, FILE *fp)
{
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  fprintf(fp, "DomTest(%s %s, main %s, corr %s)",
    KheDrsDomTestTypeShow(dt->type),
    dt->monitor != NULL ? KheMonitorId(dt->monitor) : "-",
    dt->main_dom_table2 != NULL ? dt->main_dom_table2->debug_str : "-",
    dt->corr_dom_table4 != NULL ? dt->corr_dom_table4->debug_str : "-");
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Major category "constraints (i.e. expressions)"                          */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_MONITOR_INFO"                                         */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_MONITOR_INFO KheDrsMonitorInfoMakeAndAdd(KHE_MONITOR m,          */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new monitor info object for m, add it to drs->all_monitor_infos,  */
/*  and return it.                                                           */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_MONITOR_INFO KheDrsMonitorInfoMakeAndAdd(KHE_MONITOR m,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_MONITOR_INFO res;
  HaMake(res, drs->arena);
  res->monitor = m;
  res->rerun_open_and_search_cost = 0;
  res->rerun_open_and_close_cost = 0;
  HaArrayAddLast(drs->all_monitor_infos, res);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheDrsMonitorInfoId(KHE_DRS_MONITOR_INFO mi)                       */
/*                                                                           */
/*  Monitor info id.                                                         */
/*                                                                           */
/*****************************************************************************/

static char *KheDrsMonitorInfoId(KHE_DRS_MONITOR_INFO mi)
{
  return KheMonitorId(mi->monitor);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsMonitorInfoDoUpdateRerunCost(KHE_DRS_MONITOR_INFO mi,         */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, char *field, char *operation,         */
/*    int child_index, KHE_COST *cost_field, char *fmt, va_list ap)          */
/*                                                                           */
/*  Do the actual work of KheDrsMonitorInfoUpdateRerunCost.                  */
/*                                                                           */
/*****************************************************************************/

static void KheDrsMonitorInfoDoUpdateRerunCost(KHE_DRS_MONITOR_INFO mi,
  char *field, char *operation, int child_index, KHE_COST *cost_field,
  char *fmt, va_list ap)
{
  char buff[13];  int count;  KHE_COST cost;  bool print;
  print = (strcmp(KheDrsMonitorInfoId(mi), RERUN_MONITOR_ID) == 0);
  if( print )
  {
    if( child_index >= 0 )
      snprintf(buff, 12, "%s %s %d", field, operation, child_index);
    else
      snprintf(buff, 12, "%s %s", field, operation);
    fprintf(stderr, "  %12s: %.5f", buff, KheCostShow(*cost_field));
  }
  for( count = 0;  fmt[count] != '\0';  count++ )
  {
    cost = va_arg(ap, KHE_COST);
    if( print )
      fprintf(stderr, " %c %.5f", fmt[count], KheCostShow(cost));
    if( fmt[count] == '+' )
      *cost_field += cost;
    else
      *cost_field -= cost;
  }
  if( print )
  {
    while( count < 4 )
    {
      fprintf(stderr, "          ");
      count++;
    }
    fprintf(stderr, "  (%s)\n", KheMonitorId(mi->monitor));
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsMonitorInfoUpdateRerunCost(KHE_DRS_MONITOR_INFO mi,           */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, KHE_DRS_SIGNER dsg,                   */
/*    KHE_DRS_OP op, char *operation, int child_index, char *fmt, ...)       */
/*                                                                           */
/*  Update mi.  Also produce a debug print about it, if mi's monitor is      */
/*  RERUN_MONITOR_ID.                                                        */
/*                                                                           */
/*****************************************************************************/

static void KheDrsMonitorInfoUpdateRerunCost(KHE_DRS_MONITOR_INFO mi,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, KHE_DRS_SIGNER dsg,
  KHE_DRS_OP op, char *operation, int child_index, char *fmt, ...)
{
  va_list ap;
  if( RERUN && drs->rerun != NULL && (dsg == NULL || dsg->debug_eval_if_rerun) )
  {
    if( op == KHE_DRS_OPEN || op == KHE_DRS_SEARCH )
    {
      va_start(ap, fmt);
      KheDrsMonitorInfoDoUpdateRerunCost(mi, "os", operation,
	child_index, &mi->rerun_open_and_search_cost, fmt, ap);
      va_end(ap);
    }
    if( op == KHE_DRS_OPEN || op == KHE_DRS_CLOSE )
    {
      va_start(ap, fmt);
      KheDrsMonitorInfoDoUpdateRerunCost(mi, "oc", operation,
	child_index, &mi->rerun_open_and_close_cost, fmt, ap);
      va_end(ap);
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsMonitorInfoInitRerunCost(KHE_DRS_MONITOR_INFO mi)             */
/*                                                                           */
/*  Initialize the rerun cost fields of mi to its monitor's cost.            */
/*                                                                           */
/*****************************************************************************/

static void KheDrsMonitorInfoInitRerunCost(KHE_DRS_MONITOR_INFO mi,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  mi->rerun_open_and_search_cost = mi->rerun_open_and_close_cost = 0;
  KheDrsMonitorInfoUpdateRerunCost(mi, drs, NULL, KHE_DRS_OPEN, "init", -1,
    "+", KheMonitorCost(mi->monitor));
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsMonitorInfoCheckRerunCost(KHE_DRS_MONITOR_INFO mi)            */
/*                                                                           */
/*  Check the rerun costs of mi, and do some printing if there is a problem. */
/*                                                                           */
/*****************************************************************************/

static void KheDrsMonitorInfoCheckRerunCost(KHE_DRS_MONITOR_INFO mi)
{
  if( mi->rerun_open_and_search_cost != KheMonitorCost(mi->monitor) ||
      mi->rerun_open_and_close_cost != KheMonitorCost(mi->monitor) )
  {
    fprintf(stderr, "  [ KheDrsRerun inconsistent monitor\n");
    fprintf(stderr, "    %s\n", KheMonitorId(mi->monitor));
    fprintf(stderr, "    monitor cost         %.5f\n",
      KheCostShow(KheMonitorCost(mi->monitor)));
    if( mi->rerun_open_and_search_cost != KheMonitorCost(mi->monitor) )
      fprintf(stderr, "    open_and_search_cost %.5f\n",
	KheCostShow(mi->rerun_open_and_search_cost));
    if( mi->rerun_open_and_close_cost != KheMonitorCost(mi->monitor) )
      fprintf(stderr, "    open_and_close_cost  %.5f\n",
	KheCostShow(mi->rerun_open_and_close_cost));
    fprintf(stderr, "  ]\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_OPEN_CHILDREN"                                        */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheDrsOpenChildrenInit(KHE_DRS_OPEN_CHILDREN oc,                    */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Initialize oc.                                                           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsOpenChildrenInit(KHE_DRS_OPEN_CHILDREN oc,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  HaArrayInit(oc->children, drs->arena);
  oc->range = KheDrsDayRangeMake(1, 0);
  HaArrayInit(oc->child_indexes, drs->arena);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsOpenChildrenClear(KHE_DRS_OPEN_CHILDREN oc)                   */
/*                                                                           */
/*  Clear out oc to the state it takes when it is closed.                    */
/*                                                                           */
/*****************************************************************************/

static void KheDrsOpenChildrenClear(KHE_DRS_OPEN_CHILDREN oc)
{
  HaArrayClear(oc->children);
  oc->range = KheDrsDayRangeMake(1, 0);
  HaArrayClear(oc->child_indexes);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsOpenChildrenCount(KHE_DRS_OPEN_CHILDREN oc)                    */
/*                                                                           */
/*  Return the number of open children in oc.                                */
/*                                                                           */
/*****************************************************************************/

static int KheDrsOpenChildrenCount(KHE_DRS_OPEN_CHILDREN oc)
{
  return HaArrayCount(oc->children);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsOpenChildrenAddChildInDayOrder(KHE_DRS_OPEN_CHILDREN oc,      */
/*    KHE_DRS_EXPR child_e)                                                  */
/*                                                                           */
/*  Add child_e to oc's list of open children, at a position that ensures    */
/*  that the children's day indexes are monotone non-decreasing.             */
/*                                                                           */
/*****************************************************************************/

static void KheDrsOpenChildrenAddChildInDayOrder(KHE_DRS_OPEN_CHILDREN oc,
  KHE_DRS_EXPR child_e)
{
  KHE_DRS_EXPR e;  int i;

  /* add child_e to oc->children in sorted position */
  HaArrayAddLast(oc->children, NULL);
  for( i = HaArrayCount(oc->children) - 2;  i >= 0;  i-- )
  {
    e = HaArray(oc->children, i);
    if( e->open_children_by_day.range.last <=
	child_e->open_children_by_day.range.last )
      break;
    HaArrayPut(oc->children, i + 1, e);
  }
  HaArrayPut(oc->children, i + 1, child_e);

  /* add child_e's last open day to oc->range */
  KheDrsDayRangeAdd(&oc->range, child_e->open_children_by_day.range.last);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsOpenChildrenAddChildWithDayUpdate(KHE_DRS_OPEN_CHILDREN oc,   */
/*    KHE_DRS_EXPR child_e)                                                  */
/*                                                                           */
/*  Add child_e to the end of oc's list of open children, increasing         */
/*  child_e's open day index if required.                                    */
/*                                                                           */
/*****************************************************************************/

static void KheDrsOpenChildrenAddChildWithDayUpdate(KHE_DRS_OPEN_CHILDREN oc,
  KHE_DRS_EXPR child_e)
{
  KHE_DRS_EXPR e;

  /* increase child_e's last open day if required */
  if( HaArrayCount(oc->children) > 0 )
  {
    e = HaArrayLast(oc->children);
    if( e->open_children_by_day.range.last >
	child_e->open_children_by_day.range.last )
      child_e->open_children_by_day.range.last =
	e->open_children_by_day.range.last;
  }

  /* add child_e to the end of oc->children */
  HaArrayAddLast(oc->children, child_e);

  /* add child_e's last open day to e's open_day_range */
  KheDrsDayRangeAdd(&oc->range, child_e->open_children_by_day.range.last);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprOpenShiftIndex(KHE_DRS_EXPR e)                             */
/*                                                                           */
/*  Return the open shift index of the shift containing the task monitored   */
/*  by e, which is known to be a KHE_DRS_EXPR_ASSIGNED_TASK expression.      */
/*                                                                           */
/*****************************************************************************/

static int KheDrsExprOpenShiftIndex(KHE_DRS_EXPR e)
{
  KHE_DRS_EXPR_ASSIGNED_TASK eat;  int res;
  eat = (KHE_DRS_EXPR_ASSIGNED_TASK) e;
  res = eat->task_on_day->encl_dt->encl_dtc->encl_shift->open_shift_index;
  HnAssert(res >= 0, "KheDrsExprOpenShiftIndex internal error 2");
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsOpenChildrenAddChildInShiftOrder(KHE_DRS_OPEN_CHILDREN oc,    */
/*    KHE_DRS_EXPR child_e)                                                  */
/*                                                                           */
/*  Add child_e to oc's list of open children, at a position that ensures    */
/*  that the children's open shift indexes are monotone non-decreasing.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsOpenChildrenAddChildInShiftOrder(KHE_DRS_OPEN_CHILDREN oc,
  KHE_DRS_EXPR child_e)
{
  KHE_DRS_EXPR e;  int i, child_e_index;

  /* make sure child_e is an assigned task child */
  HnAssert(child_e->tag == KHE_DRS_EXPR_ASSIGNED_TASK_TAG,
    "KheDrsOpenChildrenAddChildInShiftOrder internal error (tag %d)",
    child_e->tag);

  /* add child_e to oc->children in sorted position */
  child_e_index = KheDrsExprOpenShiftIndex(child_e);
  HaArrayAddLast(oc->children, NULL);
  for( i = HaArrayCount(oc->children) - 2;  i >= 0;  i-- )
  {
    e = HaArray(oc->children, i);
    if( KheDrsExprOpenShiftIndex(e) <= child_e_index )
      break;
    HaArrayPut(oc->children, i + 1, e);
  }
  HaArrayPut(oc->children, i + 1, child_e);

  /* add child_e's shift index to oc->range */
  KheDrsDayRangeAdd(&oc->range, child_e_index);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsOpenChildrenDeleteLast(KHE_DRS_OPEN_CHILDREN oc)              */
/*                                                                           */
/*  Delete the last child of oc.                                             */
/*                                                                           */
/*****************************************************************************/

static void KheDrsOpenChildrenDeleteLast(KHE_DRS_OPEN_CHILDREN oc)
{
  HaArrayDeleteLast(oc->children);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsOpenChildrenBuildDayChildIndexes(KHE_DRS_OPEN_CHILDREN oc)    */
/*                                                                           */
/*  Build the child indexes of oc based on the day ranges of the children.   */
/*                                                                           */
/*****************************************************************************/

static void KheDrsOpenChildrenBuildDayChildIndexes(KHE_DRS_OPEN_CHILDREN oc)
{
  int index, i;  KHE_DRS_EXPR child_e;
  HaArrayClear(oc->child_indexes);
  index = oc->range.first - 1;
  HaArrayForEach(oc->children, child_e, i)
  {
    while( child_e->open_children_by_day.range.last > index )
    {
      index++;
      HaArrayAddLast(oc->child_indexes, i);

      /******************************************************************/
      /*                                                                */
      /*  Here we have just added i at position index of child_indexes, */
      /*  and the child at index i is the first to have a larger last   */
      /*  index than index.                                             */
      /*                                                                */
      /******************************************************************/
    }
  }
  HaArrayAddLast(oc->child_indexes, i);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsOpenChildrenBuildShiftChildIndexes(KHE_DRS_OPEN_CHILDREN oc)  */
/*                                                                           */
/*  Build the child indexes of oc based on the children's shift indexes.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsOpenChildrenBuildShiftChildIndexes(KHE_DRS_OPEN_CHILDREN oc)
{
  int index, i;  KHE_DRS_EXPR child_e;
  HaArrayClear(oc->child_indexes);
  index = oc->range.first - 1;
  HaArrayForEach(oc->children, child_e, i)
  {
    while( KheDrsExprOpenShiftIndex(child_e) > index )
    {
      index++;
      HaArrayAddLast(oc->child_indexes, i);

      /******************************************************************/
      /*                                                                */
      /*  Here we have just added i at position index of child_indexes, */
      /*  and the child at index i is the first to have a larger last   */
      /*  index than index.                                             */
      /*                                                                */
      /******************************************************************/
    }
  }
  HaArrayAddLast(oc->child_indexes, i);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsOpenChildrenBefore(KHE_DRS_OPEN_CHILDREN oc, int index)        */
/*                                                                           */
/*  Return the number of open children of oc whose index is less than the    */
/*  given index.  Here index is expected to satisfy                          */
/*                                                                           */
/*    oc->range.first <= index <= oc->range.last + 1                         */
/*                                                                           */
/*  If index = oc->range.last + 1, the number of open children is returned.  */
/*                                                                           */
/*  Implementation note.  We've removed the limits on index to help with     */
/*  evaluating shift assignment objects.  This is really work in progress.   */
/*                                                                           */
/*****************************************************************************/

static int KheDrsOpenChildrenBefore(KHE_DRS_OPEN_CHILDREN oc, int index)
{
  if( index < oc->range.first )
    return 0;
  else if( index > oc->range.last )
    return HaArrayCount(oc->children);
  else
    return HaArray(oc->child_indexes, index - oc->range.first);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsOpenChildrenAtOrAfter(KHE_DRS_OPEN_CHILDREN oc, int index)     */
/*                                                                           */
/*  Return the number of open children of e whose index is index or more.    */
/*  Again, index is expected to satisfy                                      */
/*                                                                           */
/*    oc->range.first <= index <= oc->range.last + 1                         */
/*                                                                           */
/*  If index = oc->range.last + 1, 0 is returned.                            */
/*                                                                           */
/*****************************************************************************/

static int KheDrsOpenChildrenAtOrAfter(KHE_DRS_OPEN_CHILDREN oc, int index)
{
  return HaArrayCount(oc->children) - KheDrsOpenChildrenBefore(oc, index);
}


/*****************************************************************************/
/*                                                                           */
/*  KheDrsOpenChildrenForEachIndex(oc, index)                                */
/*                                                                           */
/*  Iterator for visiting each index of oc.                                  */
/*                                                                           */
/*****************************************************************************/

#define KheDrsOpenChildrenForEachIndex(oc, index)			\
  for( index = (oc)->range.first;  index <= (oc)->range.last;  index++ )


/*****************************************************************************/
/*                                                                           */
/*  KheDrsOpenChildrenForEach(oc, index, x, i)                               */
/*                                                                           */
/*  Iterator over all the open children x of oc with the given index.        */
/*                                                                           */
/*****************************************************************************/

#define KheDrsOpenChildrenForEach(oc, index, x, i)			\
  i1 = KheDrsOpenChildrenBefore((oc), (index));				\
  i2 = KheDrsOpenChildrenBefore((oc), (index) + 1);			\
  for( (i) = i1;							\
       (i) < i2 ? ((x) = HaArray((oc)->children, (i)), true) : false;	\
       (i)++ )


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsOpenChildrenIndexIsFirst(KHE_DRS_OPEN_CHILDREN oc, int index) */
/*                                                                           */
/*  Return true if index is the first open index of oc.                      */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsOpenChildrenIndexIsFirst(KHE_DRS_OPEN_CHILDREN oc, int index)
{
  return index == oc->range.first;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsOpenChildrenIndexIsFirstOrLess(KHE_DRS_OPEN_CHILDREN oc,      */
/*    int index)                                                             */
/*                                                                           */
/*  Return true if index is the first open index of oc, or less.             */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsOpenChildrenIndexIsFirstOrLess(KHE_DRS_OPEN_CHILDREN oc,
  int index)
{
  return index <= oc->range.first;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsOpenChildrenIndexIsLast(KHE_DRS_OPEN_CHILDREN oc, int index)  */
/*                                                                           */
/*  Return true if index is the last open index of oc.                       */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsOpenChildrenIndexIsLast(KHE_DRS_OPEN_CHILDREN oc, int index)
{
  return index == oc->range.last;
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsOpenChildrenWithIndex(KHE_DRS_OPEN_CHILDREN oc, int index)     */
/*                                                                           */
/*  Return the number of children in oc with this exact index.               */
/*                                                                           */
/*****************************************************************************/

static int KheDrsOpenChildrenWithIndex(KHE_DRS_OPEN_CHILDREN oc, int index)
{
  return KheDrsOpenChildrenBefore(oc, index + 1) - 
    KheDrsOpenChildrenBefore(oc, index);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsOpenChildrenDebug(KHE_DRS_OPEN_CHILDREN oc,                   */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of oc.                                                       */
/*                                                                           */
/*****************************************************************************/

static void KheDrsOpenChildrenDebugIndexes(KHE_DRS_OPEN_CHILDREN oc, FILE *fp)
{
  int val, i;
  fprintf(fp, "[%d open children, open range %d-%d: ",
    HaArrayCount(oc->children), oc->range.first, oc->range.last);
  HaArrayForEach(oc->child_indexes, val, i)
  {
    if( i > 0 )
      fprintf(fp, ",");
    fprintf(fp, "%d", val);
  }
  fprintf(fp, "]");
}

static void KheDrsOpenChildrenDebug(KHE_DRS_OPEN_CHILDREN oc,
  int verbosity, int indent, FILE *fp)
{
  int i;  KHE_DRS_EXPR child_e;
  if( indent >= 0 )
  {
    if( indent > 0 )
      fprintf(fp, "%*s", indent, "");
    fprintf(fp, "[ ");
    KheDrsOpenChildrenDebugIndexes(oc, fp);
    fprintf(fp, "\n");
    HaArrayForEach(oc->children, child_e, i)
      KheDrsExprDebug(child_e, NULL, verbosity, indent + 2, fp);
    if( indent > 0 )
      fprintf(fp, "%*s", indent, "");
    fprintf(fp, "]\n");
  }
  else
    KheDrsOpenChildrenDebugIndexes(oc, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR - construction"                                  */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsExprDebugWanted(KHE_DRS_EXPR e)                               */
/*                                                                           */
/*  Return true if e is wanted for debugging, according to DEBUG58, or       */
/*  is a child of a wanted expression.                                       */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsExprDebugWanted(KHE_DRS_EXPR e)
{
  KHE_DRS_PARENT prnt;  int i;  KHE_DRS_EXPR_INT_SUM_COST eisc;
  if( DEBUG10 )
    return true;
  if( e->tag == KHE_DRS_EXPR_INT_SUM_COST_TAG )
  {
    eisc = (KHE_DRS_EXPR_INT_SUM_COST) e;
    return DEBUG58(eisc->monitor_info);
  }
  else
  {
    HaArrayForEach(e->parents, prnt, i)
      if( KheDrsExprDebugWanted(prnt.expr) )
	return true;
  }
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprInitBegin(KHE_DRS_EXPR e, KHE_DRS_EXPR_TAG tag,           */
/*    KHE_DRS_DOM_TEST dom_test, KHE_DRS_RESOURCE dr,                        */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Initialize those fields of e common to all expressions.                  */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprInitBegin(KHE_DRS_EXPR e, KHE_DRS_EXPR_TAG tag,
  KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  e->tag = tag;
  e->gathered = false;
  e->postorder_index = -1;
  e->resource = dr;
  HaArrayInit(e->parents, drs->arena);
  HaArrayInit(e->children, drs->arena);
  KheDrsOpenChildrenInit(&e->open_children_by_day, drs);
  HaArrayInit(e->sig_indexes, drs->arena);
}


/*****************************************************************************/
/*                                                                           */
/*  KheDrsExprInitEnd(res, drs)                                              */
/*                                                                           */
/*  Finalize the initialization of e by setting its postorder index and      */
/*  closed value.                                                            */
/*                                                                           */
/*****************************************************************************/
static void KheDrsExprSetClosedValue(KHE_DRS_EXPR e,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsExprInitEnd(KHE_DRS_EXPR e, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR child_e;  int i;
  e->postorder_index = drs->postorder_count++;
  KheDrsExprSetClosedValue(e, drs);
  if( KheDrsExprDebugWanted(e) )
  {
    fprintf(stderr, "  KheDrsExprInitEnd(e), e =\n");
    KheDrsExprDebug(e, drs, 2, 2, stderr);
  }
  if( DEBUG59 )
  {
    HaArrayForEach(e->children, child_e, i)
      if( e->postorder_index <= child_e->postorder_index )
      {
	fprintf(stderr, "  KheDrsExprInitEnd aborting on e:\n");
	KheDrsExprDebug(e, drs, 2, 2, stderr);
	HnAbort("KheDrsExprInitEnd: e pi %d <= child_e pi %d",
	  e->postorder_index, child_e->postorder_index);
      }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprAddChild(KHE_DRS_EXPR parent, KHE_DRS_EXPR child,         */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make child a child of parent.                                            */
/*                                                                           */
/*  This function includes a switch on the expression type and then calls a  */
/*  version specific to that type.  Like all version-specific expression     */
/*  functions, its implementation is deferred to after the subtypes.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprAddChild(KHE_DRS_EXPR parent, KHE_DRS_EXPR child,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsParentIsFirstIntSumCostParent(KHE_DRS_EXPR child,             */
/*    KHE_DRS_EXPR parent)                                                   */
/*                                                                           */
/*  Return true if parent is the first int sum cost parent of child.         */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsParentIsFirstIntSumCostParent(KHE_DRS_EXPR child,
  KHE_DRS_EXPR parent)
{
  KHE_DRS_PARENT prnt;  int i;
  HaArrayForEach(child->parents, prnt, i)
    if( prnt.expr->tag == KHE_DRS_EXPR_INT_SUM_COST_TAG )
      return prnt.expr == parent;
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsExprHasParent(KHE_DRS_EXPR child, KHE_DRS_EXPR_TAG tag)       */
/*                                                                           */
/*  Return true if child has a parent of the type given by tag.              */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsExprHasParent(KHE_DRS_EXPR child, KHE_DRS_EXPR_TAG tag)
{
  KHE_DRS_PARENT prnt;  int i;
  HaArrayForEach(child->parents, prnt, i)
    if( prnt.expr->tag == tag )
      return true;
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR - opening"                                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprGatherForOpening(KHE_DRS_EXPR e,                          */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Gather e into drs->open_exprs, the list of expressions that are to be    */
/*  opened.  This does not actually open e; that comes later.                */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprGatherForOpening(KHE_DRS_EXPR e,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_PARENT prnt;  int i;
  if( !e->gathered )
  {
    e->gathered = true;
    HaArrayAddLast(drs->open_exprs, e);
    HaArrayForEach(e->parents, prnt, i)
      KheDrsExprGatherForOpening(prnt.expr, drs);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprPostorderCmp(const void *t1, const void *t2)               */
/*                                                                           */
/*  Comparison function for sorting an array of expressions by increasing    */
/*  postorder index.                                                         */
/*                                                                           */
/*****************************************************************************/

static int KheDrsExprPostorderCmp(const void *t1, const void *t2)
{
  KHE_DRS_EXPR e1 = * (KHE_DRS_EXPR *) t1;
  KHE_DRS_EXPR e2 = * (KHE_DRS_EXPR *) t2;
  return e1->postorder_index - e2->postorder_index;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprChildHasOpened(KHE_DRS_EXPR e, KHE_DRS_EXPR child_e,      */
/*    int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)                      */
/*                                                                           */
/*  As part of opening e for solving, inform e that its child_index'th       */
/*  child, child_e, has opened.                                              */
/*                                                                           */
/*  This function switches on the expression type and then calls a           */
/*  version specific to that type.  Like all version-specific expression     */
/*  functions, its implementation is deferred to after the subtypes.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprChildHasOpened(KHE_DRS_EXPR e, KHE_DRS_EXPR child_e,
  int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs);


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprChildDomTest(KHE_DRS_EXPR e,                  */
/*    KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  Return a dom test suited to a child of e.                                */
/*                                                                           */
/*  This function switches on the expression type and then calls a           */
/*  version specific to that type.  Like all version-specific expression     */
/*  functions, its implementation is deferred to after the subtypes.         */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DOM_TEST KheDrsExprChildDomTest(KHE_DRS_EXPR e,
  int open_day_index, KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs);


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprDayDomTest(KHE_DRS_EXPR e,                    */
/*    int open_child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  Return a dom test suited to e on its open_child_index'th open day.       */
/*                                                                           */
/*  This function switches on the expression type and then calls a           */
/*  version specific to that type.  Like all version-specific expression     */
/*  functions, its implementation is deferred to after the subtypes.         */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DOM_TEST KheDrsExprDayDomTest(KHE_DRS_EXPR e,
  int open_child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs);


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprOpen(KHE_DRS_EXPR e, KHE_DYNAMIC_RESOURCE_SOLVER drs)     */
/*                                                                           */
/*  Open e for solving.                                                      */
/*                                                                           */
/*  By the time this is called, all of e's open children will have           */
/*  called KheDrsExprChildHasOpened.                                         */
/*                                                                           */
/*  Implementation note - how e->open_day_range is set                       */
/*  --------------------------------------------------                       */
/*  In external expressions, e->open_day_range receives its final value      */
/*  when KheDrsExprGatherForOpening is called.  In internal expressions,     */
/*  its value is initially empty and is updated by each call to              */
/*  KheDrsExprChildHasOpened until it reaches its final value.               */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprOpen(KHE_DRS_EXPR e, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_PARENT prnt;  int i;  bool debug;

  debug = KheDrsExprDebugWanted(e);
  if( debug )
  {
    fprintf(stderr, "[ KheDrsExprOpen\n");
    KheDrsExprDebug(e, drs, 2, 2, stderr);
  }

  /* inform e's parents that e is now open */
  e->gathered = false;
  HaArrayForEach(e->parents, prnt, i)
    KheDrsExprChildHasOpened(prnt.expr, e, prnt.index, drs);

  /* what to do now depends on whether e is external or internal, and if */
  /* internal whether from an event resource or resource constraint */
  if( e->tag <= KHE_DRS_EXPR_WORK_DAY_TAG )
  {
    /* external expression; clear its value */
    KheDrsExprLeafClear(e, drs);
  }
  else
  {
    /* internal expression, build day child indexes */
    KheDrsOpenChildrenBuildDayChildIndexes(&e->open_children_by_day);
  }

  if( debug )
  {
    fprintf(stderr, "  KheDrsExprOpen returning; e is now:\n");
    KheDrsExprDebug(e, drs, 2, 2, stderr);
    fprintf(stderr, "] KheDrsExprOpen\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprBuildDomTests(KHE_DRS_EXPR e,                             */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Build e's dom tests.                                                     */
/*                                                                           */
/*  By the time this is called, every expression that is going to open       */
/*  will be open, including knowing what all of its open children are.       */
/*                                                                           */
/*****************************************************************************/
static bool KheDrsIntSumCostExprRequiresShiftDomTest(
  KHE_DRS_EXPR_INT_SUM_COST eisc, int si,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int *unassigned_child_count);
static KHE_DRS_DOM_TEST KheDrsExprIntSumCostGeneralDomTest(
  KHE_DRS_EXPR_INT_SUM_COST eisc, int unassigned_child_count,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsExprBuildDomTests(KHE_DRS_EXPR e,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DAY day;  KHE_DRS_RESOURCE_ON_DAY drd;  int di, si, val;
  KHE_DRS_DOM_TEST dom_test;  bool debug; KHE_DRS_SHIFT ds;
  KHE_DRS_EXPR_INT_SUM_COST eisc;

  debug = KheDrsExprDebugWanted(e);
  if( debug )
  {
    fprintf(stderr, "[ KheDrsExprBuildDomTests\n");
    KheDrsExprDebug(e, drs, 2, 2, stderr);
  }

  if( e->tag <= KHE_DRS_EXPR_WORK_DAY_TAG )
  {
    /* external expression; nothing to do */
  }
  else if( e->resource != NULL )
  {
    /* internal expression from a resource constraint */
    /* add e to its resource on day signers */
    HaArrayClear(e->sig_indexes);
    KheDrsOpenChildrenForEachIndex(&e->open_children_by_day, di)
    {
      day = HaArray(drs->open_days, di);
      drd = KheDrsResourceOnDay(e->resource, day);
      KheDrsResourceOnDayAddOpenExpr(drd, e);
      if( !KheDrsOpenChildrenIndexIsLast(&e->open_children_by_day, di) )
      {
	dom_test = KheDrsExprDayDomTest(e, di, drs);
	HaArrayAddLast(e->sig_indexes,
	  KheDrsResourceOnDayAddDomTest(drd, dom_test, drs));
      }
    }
  }
  else
  {
    /* internal expression from an event resource constraint */
    /* add e to its day signers */
    HaArrayClear(e->sig_indexes);
    KheDrsOpenChildrenForEachIndex(&e->open_children_by_day, di)
    {
      day = HaArray(drs->open_days, di);
      KheDrsDayAddOpenExpr(day, e);
      if( !KheDrsOpenChildrenIndexIsLast(&e->open_children_by_day, di) )
      {
	dom_test = KheDrsExprDayDomTest(e, di, drs);
	HaArrayAddLast(e->sig_indexes,
	  KheDrsDayAddDomTest(day, dom_test, drs));
      }
    }

    /* add e to its shift signers */
    HnAssert(e->tag == KHE_DRS_EXPR_INT_SUM_COST_TAG,
      "KheDrsExprBuildDomTests: internal error");
    eisc = (KHE_DRS_EXPR_INT_SUM_COST) e;
    KheDrsOpenChildrenBuildShiftChildIndexes(&eisc->open_children_by_shift);
    KheDrsOpenChildrenForEachIndex(&eisc->open_children_by_shift, si)
      if( KheDrsOpenChildrenWithIndex(&eisc->open_children_by_shift, si) > 0 )
      {
	ds = HaArray(drs->open_shifts, si);
	KheDrsShiftAddOpenExpr(ds, e);
        if( KheDrsIntSumCostExprRequiresShiftDomTest(eisc, si, drs, &val) )
	{
	  dom_test = KheDrsExprIntSumCostGeneralDomTest(eisc, val, drs);
	  KheDrsShiftAddDomTest(ds, dom_test, drs);
	}
      }
  }
  if( debug )
  {
    fprintf(stderr, "  KheDrsExprBuildDomTests returning; e is now:\n");
    KheDrsExprDebug(e, drs, 2, 2, stderr);
    fprintf(stderr, "] KheDrsExprBuildDomTests\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR - accessing signatures"                          */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprDaySigVal(KHE_DRS_EXPR e, int open_day_index,              */
/*    KHE_DRS_SOLN soln)                                                     */
/*                                                                           */
/*  Retrieve from soln the signature of e on open day open_day_index.        */
/*                                                                           */
/*****************************************************************************/

static int KheDrsExprDaySigVal(KHE_DRS_EXPR e, int open_day_index,
  KHE_DRS_SIGNATURE sig)
{
  int pos;
  pos = HaArray(e->sig_indexes, open_day_index -
    e->open_children_by_day.range.first);
  return HaArray(sig->states, pos);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR - closing"                                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprChildHasClosed(KHE_DRS_EXPR e,                            */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  Inform internal expression e that its child_index'th child, child_e,     */
/*  has closed.  The child will have its correct closed_state and value      */
/*  fields when this is called.                                              */
/*                                                                           */
/*  Although KheDrsExprChildHasOpened adds child_e to e->open_children,      */
/*  KheDrsExprChildHasClosed does not remove child_e from e->open_children.  */
/*  This is because children are closed in the same order they were opened,  */
/*  but there is no efficient way to remove them in this order, short of     */
/*  something too complicated such as a circular list.                       */
/*                                                                           */
/*  This function switches on the expression type and then calls a           */
/*  version specific to that type.  Like all version-specific expression     */
/*  functions, its implementation is deferred to after the subtypes.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprChildHasClosed(KHE_DRS_EXPR e,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs);


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprClose(KHE_DRS_EXPR e, KHE_DYNAMIC_RESOURCE_SOLVER drs)    */
/*                                                                           */
/*  Close e, which may be internal or external.  Whatever it needs to work   */
/*  out its closed value (an assignment, closed children, or closed state)   */
/*  is up to date.  At the end, inform e's parents that e has closed.        */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprClose(KHE_DRS_EXPR e, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_PARENT prnt;  int i;

  /* set e's closed value */
  KheDrsExprSetClosedValue(e, drs);

  /* clear fields that are used only when e is open */
  KheDrsOpenChildrenClear(&e->open_children_by_day);
  HaArrayClear(e->sig_indexes);

  /* inform e's parents that e has closed */
  HaArrayForEach(e->parents, prnt, i)
    KheDrsExprChildHasClosed(prnt.expr, e, prnt.index, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprSetClosedValue(KHE_DRS_EXPR e,                            */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing e, set its value suitably for its closed state.       */
/*                                                                           */
/*  This function switches on the expression type and then calls a           */
/*  version specific to that type.  Like all version-specific expression     */
/*  functions, its implementation is deferred to after the subtypes.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprSetClosedValue(KHE_DRS_EXPR e,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR - searching"                                     */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprLeafSet(KHE_DRS_EXPR e, KHE_DRS_TASK_ON_DAY dtd,          */
/*    KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs)                  */
/*                                                                           */
/*  Inform leaf expression e that dtd has been assigned drd.                 */
/*                                                                           */
/*  This function switches on the expression type and then calls a           */
/*  version specific to that type.  Like all version-specific expression     */
/*  functions, its implementation is deferred to after the subtypes.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprLeafSet(KHE_DRS_EXPR e, KHE_DRS_TASK_ON_DAY dtd,
  KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs);


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprLeafClear(KHE_DRS_EXPR e)                                 */
/*                                                                           */
/*  Inform leaf expression e that the assignment of drd to dtd carried out   */
/*  by KheDrsExprLeafSet above has been removed.                             */
/*                                                                           */
/*  This function switches on the expression type and then calls a           */
/*  version specific to that type.  Like all version-specific expression     */
/*  functions, its implementation is deferred to after the subtypes.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprLeafClear(KHE_DRS_EXPR e,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprEvalSignature(KHE_DRS_EXPR e, KHE_DRS_SIGNER dsg,         */
/*    KHE_DRS_SIGNATURE prev_sig, int next_di, KHE_DRS_SIGNATURE sig,        */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, bool debug)                           */
/*                                                                           */
/*  Evaluate expression e on day next_di, which is one of e's open days.     */
/*  The results go into sig.                                                 */
/*                                                                           */
/*  This function switches on the expression type and then calls a           */
/*  version specific to that type.  Like all version-specific expression     */
/*  functions, its implementation is deferred to after the subtypes.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprEvalSignature(KHE_DRS_EXPR e, KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE prev_sig, int next_di, KHE_DRS_SIGNATURE sig,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, bool debug);


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR - debug"                                         */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprDoDebug(KHE_DRS_EXPR e, KHE_DRS_SOLN soln,                */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp);                            */
/*                                                                           */
/*  Unindented debug print of that part of e which is specific to it.        */
/*  If soln != NULL, also print e's signature (if any) in soln.              */
/*                                                                           */
/*  This function switches on the expression type and then calls a           */
/*  version specific to that type.  Like all version-specific expression     */
/*  functions, its implementation is deferred to after the subtypes.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprDoDebug(KHE_DRS_EXPR e, KHE_DRS_SOLN soln,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp);


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprDebug(KHE_DRS_EXPR e, KHE_DYNAMIC_RESOURCE_SOLVER drs,    */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of e onto fp with the given verbosity and indent.            */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprDebug(KHE_DRS_EXPR e, KHE_DYNAMIC_RESOURCE_SOLVER drs,
  int verbosity, int indent, FILE *fp)
{
  KHE_DRS_EXPR child_e;  int i;
  if( indent >= 0 )
  {
    fprintf(fp, "%*s[ ", indent, "");
    KheDrsExprDoDebug(e, NULL, drs, fp);
    fprintf(fp, " p %p pi %d oc %d", (void *) e, e->postorder_index,
      KheDrsOpenChildrenCount(&e->open_children_by_day));
    if( verbosity >= 2 && HaArrayCount(e->children) > 0 )
    {
      fprintf(fp, "\n");
      HaArrayForEach(e->children, child_e, i)
	KheDrsExprDebug(child_e, drs, verbosity, indent + 2, fp);
      fprintf(fp, "%*s]\n", indent, "");
    }
    else
      fprintf(fp, " ]\n");
  }
  else
    KheDrsExprDoDebug(e, NULL, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprDebugSignature(KHE_DRS_EXPR e, KHE_DRS_SOLN soln,         */
/*    bool is_float, FILE *fp)                                               */
/*                                                                           */
/*  If soln != NULL and e has a signature entry in soln, debug that entry.   */
/*                                                                           */
/*  NB soln == NULL does not mean that there is a valid solution whose       */
/*  value is NULL, it just means that we are doing this debug without the    */
/*  benefit of a current solution, and so this function has nothing to do.   */
/*                                                                           */
/*****************************************************************************/
static KHE_DRS_DAY KheDrsSolnDay(KHE_DRS_SOLN soln,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsExprDebugSignature(KHE_DRS_EXPR e, KHE_DRS_SOLN soln,
  bool is_float, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  if( soln != NULL )
  {
    fprintf(fp, "  KheDrsExprDebugSignature decommissioned");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_ASSIGNED_TASK"                                   */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_ASSIGNED_TASK KheDrsExprAssignedTaskMake(                   */
/*    KHE_DRS_TASK_ON_DAY dtd, KHE_RESOURCE_GROUP rg,                        */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make an assigned task expression object with these attributes.           */
/*  Or if there is already such an expression in dtd's list, use that.       */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_ASSIGNED_TASK KheDrsExprAssignedTaskMake(
  KHE_DRS_TASK_ON_DAY dtd, KHE_RESOURCE_GROUP rg,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_ASSIGNED_TASK res;
  if( KheDrsTaskOnDayHasAssignedTaskExpr(dtd, rg, &res) )
    return res;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_ASSIGNED_TASK_TAG,
    NULL, drs);
  res->task_on_day = dtd;
  res->resource_group = rg;
  KheDrsTaskOnDayAddAssignedTaskExpr(dtd, res);
  KheDrsExprInitEnd((KHE_DRS_EXPR) res, drs);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprAssignedTaskValue(KHE_DRS_EXPR_ASSIGNED_TASK eat,          */
/*    KHE_RESOURCE r)                                                        */
/*                                                                           */
/*  Return the value of eat when its task is assigned r (possibly NULL).     */
/*  This will be 1 when r is non-NULL and lies in eat's resource group.      */
/*  For efficiency, eat's resource group may be NULL, meaning all resources. */
/*                                                                           */
/*****************************************************************************/

static int KheDrsExprAssignedTaskValue(KHE_DRS_EXPR_ASSIGNED_TASK eat,
  KHE_RESOURCE r)
{
  KHE_RESOURCE_GROUP rg;
  if( r == NULL )
    return 0;
  rg = eat->resource_group;
  return (rg == NULL || KheResourceGroupContains(rg, r)) ? 1 : 0;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprAssignedTaskLeafSet(KHE_DRS_EXPR_ASSIGNED_TASK eat,       */
/*    KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_RESOURCE dr)                          */
/*                                                                           */
/*  Update the value of eat to reflect the assignment of dr to dtd.  Here    */
/*  dtd is known to be the relevant task (which is why it is unused), and    */
/*  dr is known to represent a non-NULL resource, not a non-assignment.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprAssignedTaskLeafSet(KHE_DRS_EXPR_ASSIGNED_TASK eat,
  KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_RESOURCE dr)
{
  eat->u.int_val = KheDrsExprAssignedTaskValue(eat, dr->resource);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprAssignedTaskLeafClear(KHE_DRS_EXPR_ASSIGNED_TASK eat)     */
/*                                                                           */
/*  Inform leaf expression ebt that the assignment of drd to dtd carried     */
/*  out by KheDrsExprAssignedTaskLeafSet above has been removed.             */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprAssignedTaskLeafClear(KHE_DRS_EXPR_ASSIGNED_TASK eat)
{
  eat->u.int_val = 0;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprAssignedTaskSetClosedValue(KHE_DRS_EXPR_ASSIGNED_TASK eat,*/
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing eat, set its value suitably for its closed state.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprAssignedTaskSetClosedValue(KHE_DRS_EXPR_ASSIGNED_TASK eat,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_RESOURCE_ON_DAY drd;  KHE_RESOURCE r;
  drd = eat->task_on_day->closed_asst;
  r = (drd == NULL ? NULL : drd->encl_dr->resource);
  eat->u.int_val = KheDrsExprAssignedTaskValue(eat, r);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprAssignedTaskDoDebug(KHE_DRS_EXPR_ASSIGNED_TASK eat,       */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of eat which is specific to it.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprAssignedTaskDoDebug(KHE_DRS_EXPR_ASSIGNED_TASK eat,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "ASSIGNED_TASK(");
  HnAssert(eat->task_on_day->task != NULL,
    "KheDrsExprAssignedTaskDoDebug internal error");
  KheTaskDebug(eat->task_on_day->task, 1, -1, fp);
  fprintf(fp, ", %s, %d)", eat->resource_group == NULL ? "-" :
    KheResourceGroupId(eat->resource_group), eat->u.int_val);
  KheDrsExprDebugSignature((KHE_DRS_EXPR) eat, soln, false, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_BUSY_TIME"                                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_BUSY_TIME KheDrsExprBusyTimeMake(                           */
/*    KHE_DRS_RESOURCE_ON_DAY drd, KHE_TIME time,                            */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new KHE_DRS_EXPR_BUSY_TIME object with these attributes.          */
/*  Or retrieve an equivalent existing expression if there is one.           */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_BUSY_TIME KheDrsExprBusyTimeMake(
  KHE_DRS_RESOURCE_ON_DAY drd, KHE_TIME time, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_BUSY_TIME res;  KHE_DRS_EXPR e;
  if( KheDrsResourceOnDayHasTimeExpr(drd, KHE_DRS_EXPR_BUSY_TIME_TAG,
	time, drs, &e) )
    return (KHE_DRS_EXPR_BUSY_TIME) e;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_BUSY_TIME_TAG,
    drd->encl_dr, drs);
  res->resource_on_day = drd;
  res->time = time;
  KheDrsResourceOnDayAddExpr(drd, (KHE_DRS_EXPR) res);
  KheDrsExprInitEnd((KHE_DRS_EXPR) res, drs);  /* will set u.int_val */
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprBusyTimeLeafSet(KHE_DRS_EXPR_BUSY_TIME ebt,               */
/*    KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_RESOURCE dr)                          */
/*                                                                           */
/*  Set the value of ebt, given that dtd has just been assigned dr.          */
/*  Here dtd is known to be an actual task, not a non-assignment, and dr     */
/*  is known to be concerned with the same resource as ebt is.               */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprBusyTimeLeafSet(KHE_DRS_EXPR_BUSY_TIME ebt,
  KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_RESOURCE dr)
{
  ebt->u.int_val = (dtd->time == ebt->time ? 1 : 0);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprBusyTimeLeafClear(KHE_DRS_EXPR_BUSY_TIME ebt)             */
/*                                                                           */
/*  Inform leaf expression ebt that the assignment of drd to dtd carried     */
/*  out by KheDrsExprBusyTimeLeafSet above has been removed.                 */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprBusyTimeLeafClear(KHE_DRS_EXPR_BUSY_TIME ebt)
{
  ebt->u.int_val = 0;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprBusyTimeSetClosedValue(KHE_DRS_EXPR_BUSY_TIME ebt,        */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing ebt, set its value suitably for its closed state.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprBusyTimeSetClosedValue(KHE_DRS_EXPR_BUSY_TIME ebt,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_TASK_ON_DAY dtd;
  dtd = ebt->resource_on_day->closed_asst;
  ebt->u.int_val = (dtd != NULL && dtd->time == ebt->time ? 1 : 0);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprBusyTimeDoDebug(KHE_DRS_EXPR_BUSY_TIME ebt,               */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of ebt which is specific to it.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprBusyTimeDoDebug(KHE_DRS_EXPR_BUSY_TIME ebt,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "BUSY_TIME(%s, %s, iv %d)",
    KheDrsResourceId(ebt->resource_on_day->encl_dr),
    KheTimeId(ebt->time), ebt->u.int_val);
  KheDrsExprDebugSignature((KHE_DRS_EXPR) ebt, soln, false, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_FREE_TIME"                                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_FREE_TIME KheDrsExprFreeTimeMake(                           */
/*    KHE_DRS_RESOURCE_ON_DAY drd, KHE_TIME time,                            */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new KHE_DRS_EXPR_FREE_TIME object with these attributes.          */
/*  Or retrieve an equivalent existing expression if there is one.           */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_FREE_TIME KheDrsExprFreeTimeMake(
  KHE_DRS_RESOURCE_ON_DAY drd, KHE_TIME time, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_FREE_TIME res;  KHE_DRS_EXPR e;
  if( KheDrsResourceOnDayHasTimeExpr(drd, KHE_DRS_EXPR_FREE_TIME_TAG,
      time, drs, &e) )
    return (KHE_DRS_EXPR_FREE_TIME) e;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_FREE_TIME_TAG,
    drd->encl_dr, drs);
  res->resource_on_day = drd;
  res->time = time;
  KheDrsResourceOnDayAddExpr(drd, (KHE_DRS_EXPR) res);
  KheDrsExprInitEnd((KHE_DRS_EXPR) res, drs);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFreeTimeLeafSet(KHE_DRS_EXPR_FREE_TIME eft,               */
/*    KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_RESOURCE dr)                          */
/*                                                                           */
/*  Set the value of eft, given that dtd has just been assigned dr.          */
/*  Here dtd is known to be an actual task, not a non-assignment, and dr     */
/*  is known to be concerned with the same resource as eft is.               */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFreeTimeLeafSet(KHE_DRS_EXPR_FREE_TIME eft,
  KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_RESOURCE dr)
{
  eft->u.int_val = (dtd->time == eft->time ? 0 : 1);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFreeTimeLeafClear(KHE_DRS_EXPR_FREE_TIME eft)             */
/*                                                                           */
/*  Inform leaf expression eft that the assignment of drd to dtd carried     */
/*  out by KheDrsExprFreeTimeLeafSet above has been removed.                 */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFreeTimeLeafClear(KHE_DRS_EXPR_FREE_TIME eft)
{
  eft->u.int_val = 1;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFreeTimeSetClosedValue(KHE_DRS_EXPR_FREE_TIME eft,        */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing eft, set its value suitably for its closed state.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFreeTimeSetClosedValue(KHE_DRS_EXPR_FREE_TIME eft,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_TASK_ON_DAY dtd;
  dtd = eft->resource_on_day->closed_asst;
  eft->u.int_val = (dtd != NULL && dtd->time == eft->time ? 0 : 1);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFreeTimeDoDebug(KHE_DRS_EXPR_FREE_TIME eft,               */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of eft which is specific to it.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFreeTimeDoDebug(KHE_DRS_EXPR_FREE_TIME eft,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "FREE_TIME(%s, %s, iv %d)",
    KheDrsResourceId(eft->resource_on_day->encl_dr),
    KheTimeId(eft->time), eft->u.int_val);
  KheDrsExprDebugSignature((KHE_DRS_EXPR) eft, soln, false, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_WORK_TIME"                                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_WORK_TIME KheDrsExprWorkTimeMake(                           */
/*    KHE_DRS_RESOURCE_ON_DAY drd, KHE_TIME time,                            */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new KHE_DRS_EXPR_WORK_TIME object with these attributes.          */
/*  Or retrieve an equivalent existing expression if there is one.           */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_WORK_TIME KheDrsExprWorkTimeMake(
  KHE_DRS_RESOURCE_ON_DAY drd, KHE_TIME time,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_WORK_TIME res;  KHE_DRS_EXPR e;
  if( KheDrsResourceOnDayHasTimeExpr(drd, KHE_DRS_EXPR_WORK_TIME_TAG,
	time, drs, &e) )
    return (KHE_DRS_EXPR_WORK_TIME) e;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_WORK_TIME_TAG,
    drd->encl_dr, drs);
  res->resource_on_day = drd;
  res->time = time;
  KheDrsResourceOnDayAddExpr(drd, (KHE_DRS_EXPR) res);
  KheDrsExprInitEnd((KHE_DRS_EXPR) res, drs);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprWorkTimeLeafSet(KHE_DRS_EXPR_WORK_TIME ewt,               */
/*    KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_RESOURCE dr)                          */
/*                                                                           */
/*  Set the value of ewt, given that dtd has just been assigned dr.          */
/*  Here dtd is known to be an actual task, not a non-assignment, and dr     */
/*  is known to be concerned with the same resource as ewt is.               */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprWorkTimeLeafSet(KHE_DRS_EXPR_WORK_TIME ewt,
  KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_RESOURCE dr)
{
  ewt->u.float_val =
    (dtd->time == ewt->time ? KheTaskWorkloadPerTime(dtd->task) : 0.0);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprWorkTimeLeafClear(KHE_DRS_EXPR_WORK_TIME ewt)             */
/*                                                                           */
/*  Inform leaf expression ebt that the assignment of drd to dtd carried     */
/*  out by KheDrsExprWorkTimeLeafSet above has been removed.                 */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprWorkTimeLeafClear(KHE_DRS_EXPR_WORK_TIME ewt)
{
  ewt->u.float_val = 0.0;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprWorkTimeSetClosedValue(KHE_DRS_EXPR_WORK_TIME ewt,        */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing ewt, set its value suitably for its closed state.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprWorkTimeSetClosedValue(KHE_DRS_EXPR_WORK_TIME ewt,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_TASK_ON_DAY dtd;
  dtd = ewt->resource_on_day->closed_asst;
  ewt->u.float_val = (dtd != NULL && dtd->time == ewt->time ?
    KheTaskWorkloadPerTime(dtd->task) : 0.0);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprWorkTimeDoDebug(KHE_DRS_EXPR_WORK_TIME ewt,               */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of ewt which is specific to it.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprWorkTimeDoDebug(KHE_DRS_EXPR_WORK_TIME ewt,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "WORK_TIME(%s, %s)",
    KheDrsResourceId(ewt->resource_on_day->encl_dr), KheTimeId(ewt->time));
  KheDrsExprDebugSignature((KHE_DRS_EXPR) ewt, soln, true, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_BUSY_DAY"                                        */
/*                                                                           */
/*****************************************************************************/

#define BUSY_DAY_DEBUG 0
#define BUSY_DAY_TIME_GROUP_ID "1Sat"
#define BUSY_DAY_RESOURCE "TR_25"


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsBusyDayDebug(KHE_DRS_EXPR_BUSY_DAY ebd)                       */
/*                                                                           */
/*  Return true if we want to debug ebd.                                     */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsBusyDayDebug(KHE_DRS_EXPR_BUSY_DAY ebd,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_RESOURCE r;  KHE_TIME_GROUP tg;
  if( RERUN && drs->rerun != NULL && BUSY_DAY_DEBUG )
  {
    r = ebd->resource_on_day->encl_dr->resource;
    tg = ebd->time_group;
    return strcmp(KheResourceId(r), BUSY_DAY_RESOURCE) == 0 &&
      strcmp(KheTimeGroupId(tg), BUSY_DAY_TIME_GROUP_ID) == 0;
  }
  else
    return false;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_BUSY_DAY KheDrsExprBusyDayMake(KHE_DRS_RESOURCE_ON_DAY drd, */
/*    KHE_TIME_GROUP tg, KHE_DYNAMIC_RESOURCE_SOLVER drs)                    */
/*                                                                           */
/*  Make a new KHE_DRS_EXPR_BUSY_DAY object with these attributes.           */
/*  Or retrieve an equivalent existing expression if there is one.           */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_BUSY_DAY KheDrsExprBusyDayMake(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_TIME_GROUP tg, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_BUSY_DAY res;  KHE_DRS_EXPR e;
  if( KheDrsResourceOnDayHasDayExpr(drd, KHE_DRS_EXPR_BUSY_DAY_TAG, tg,drs,&e) )
    return (KHE_DRS_EXPR_BUSY_DAY) e;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_BUSY_DAY_TAG,
    drd->encl_dr, drs);
  res->resource_on_day = drd;
  res->time_group = tg;
  KheDrsResourceOnDayAddExpr(drd, (KHE_DRS_EXPR) res);
  KheDrsExprInitEnd((KHE_DRS_EXPR) res, drs);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprBusyDayLeafSet(KHE_DRS_EXPR_BUSY_DAY ebd,                 */
/*    KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_RESOURCE dr,                          */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Set the value of ebd, given that dtd has just been assigned dr.          */
/*  Here dtd is known to be an actual task, not a non-assignment, and dr     */
/*  is known to be concerned with the same resource as ebt is.               */
/*                                                                           */
/*****************************************************************************/
static void KheDrsExprBusyDayDoDebug(KHE_DRS_EXPR_BUSY_DAY ebd,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp);

static void KheDrsExprBusyDayLeafSet(KHE_DRS_EXPR_BUSY_DAY ebd,
  KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_RESOURCE dr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  ebd->u.int_val = 1;
  if( KheDrsBusyDayDebug(ebd, drs) )
  {
    fprintf(stderr, "  KheDrsExprBusyDayLeafSet(");
    KheDrsExprBusyDayDoDebug(ebd, NULL, drs, stderr);
    fprintf(stderr, ")\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprBusyDayLeafClear(KHE_DRS_EXPR_BUSY_DAY ebd)               */
/*                                                                           */
/*  Inform leaf soln ebd that the assignment of drd to dtd carried out by    */
/*  KheDrsExprBusyDayLeafSet above has been removed.                         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprBusyDayLeafClear(KHE_DRS_EXPR_BUSY_DAY ebd,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  ebd->u.int_val = 0;
  if( KheDrsBusyDayDebug(ebd, drs) )
  {
    fprintf(stderr, "  KheDrsExprBusyDayLeafClear(");
    KheDrsExprBusyDayDoDebug(ebd, NULL, drs, stderr);
    fprintf(stderr, ")\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprBusyDaySetClosedValue(KHE_DRS_EXPR_BUSY_DAY ebd,          */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing ebd, set its value suitably for its closed state.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprBusyDaySetClosedValue(KHE_DRS_EXPR_BUSY_DAY ebd,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_TASK_ON_DAY dtd;
  dtd = ebd->resource_on_day->closed_asst;
  ebd->u.int_val = (dtd != NULL ? 1 : 0);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprBusyDayDoDebug(KHE_DRS_EXPR_BUSY_DAY ebd,                 */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of ebd which is specific to it.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprBusyDayDoDebug(KHE_DRS_EXPR_BUSY_DAY ebd,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "BUSY_DAY(%s, %s, iv %d)",
    KheDrsResourceId(ebd->resource_on_day->encl_dr),
    KheTimeGroupId(ebd->time_group), ebd->u.int_val);
  KheDrsExprDebugSignature((KHE_DRS_EXPR) ebd, soln, false, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_FREE_DAY"                                        */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_FREE_DAY KheDrsExprFreeDayMake(KHE_DRS_RESOURCE_ON_DAY drd, */
/*    KHE_TIME_GROUP tg, KHE_DYNAMIC_RESOURCE_SOLVER drs)                    */
/*                                                                           */
/*  Make a new KHE_DRS_EXPR_FREE_DAY object with these attributes.           */
/*  Or retrieve an equivalent existing expression if there is one.           */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_FREE_DAY KheDrsExprFreeDayMake(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_TIME_GROUP tg, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_FREE_DAY res;  KHE_DRS_EXPR e;
  if( KheDrsResourceOnDayHasDayExpr(drd, KHE_DRS_EXPR_FREE_DAY_TAG, tg,drs,&e) )
    return (KHE_DRS_EXPR_FREE_DAY) e;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_FREE_DAY_TAG,
    drd->encl_dr, drs);
  res->resource_on_day = drd;
  res->time_group = tg;
  KheDrsResourceOnDayAddExpr(drd, (KHE_DRS_EXPR) res);
  KheDrsExprInitEnd((KHE_DRS_EXPR) res, drs);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFreeDayLeafSet(KHE_DRS_EXPR_FREE_DAY efd,                 */
/*    KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_RESOURCE dr)                          */
/*                                                                           */
/*  Set the value of efd, given that dtd has just been assigned dr.          */
/*  Here dtd is known to be an actual task, not a non-assignment, and dr     */
/*  is known to be concerned with the same resource as efd is.               */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFreeDayLeafSet(KHE_DRS_EXPR_FREE_DAY efd,
  KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_RESOURCE dr)
{
  efd->u.int_val = 0;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFreeDayLeafClear(KHE_DRS_EXPR_FREE_DAY efd)               */
/*                                                                           */
/*  Inform leaf expression eft that the assignment of drd to dtd carried     */
/*  out by KheDrsExprFreeDayLeafSet above has been removed.                  */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFreeDayLeafClear(KHE_DRS_EXPR_FREE_DAY efd)
{
  efd->u.int_val = 1;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFreeDaySetClosedValue(KHE_DRS_EXPR_FREE_DAY efd,          */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing efd, set its value suitably for its closed state.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFreeDaySetClosedValue(KHE_DRS_EXPR_FREE_DAY efd,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_TASK_ON_DAY dtd;
  dtd = efd->resource_on_day->closed_asst;
  efd->u.int_val = (dtd != NULL ? 0 : 1);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFreeDayDoDebug(KHE_DRS_EXPR_FREE_DAY efd,                 */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of efd which is specific to it.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFreeDayDoDebug(KHE_DRS_EXPR_FREE_DAY efd,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "FREE_DAY(%s, %s, iv %d)",
    KheDrsResourceId(efd->resource_on_day->encl_dr),
    KheTimeGroupId(efd->time_group), efd->u.int_val);
  KheDrsExprDebugSignature((KHE_DRS_EXPR) efd, soln, false, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_WORK_DAY"                                        */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_WORK_DAY KheDrsExprWorkDayMake(KHE_TIME_GROUP tg,           */
/*    KHE_DRS_DOM_TEST dom_test, KHE_DYNAMIC_RESOURCE_SOLVER drs)            */
/*                                                                           */
/*  Make a new KHE_DRS_EXPR_WORK_DAY object with these attributes.           */
/*  Or retrieve an equivalent existing expression if there is one.           */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused, although I really should be using it
static KHE_DRS_EXPR_WORK_DAY KheDrsExprWorkDayMake(KHE_TIME_GROUP tg,
  KHE_DRS_DOM_TEST dom_test, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_WORK_DAY res;  KHE_DRS_EXPR e;
  if( KheDrsHasDayExpr(drs, KHE_DRS_EXPR_WORK_DAY_TAG, dom_test, tg, &e) )
    return (KHE_DRS_EXPR_WORK_DAY) e;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_WORK_DAY_TAG,
    ** dom_test, ** drd->encl_dr, drs);
  res->u.float_val = 0.0;  ** default value, used sometimes **
  res->time_group = tg;
  KheDrsAddDayExpr(drs, tg, (KHE_DRS_EXPR) res);
  KheDrsExprInitEnd((KHE_DRS_EXPR) res, drs);
  return res;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprWorkDayLeafSet(KHE_DRS_EXPR_WORK_DAY ewd,                 */
/*    KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_RESOURCE dr)                          */
/*                                                                           */
/*  Set the value of ewd, given that dtd has just been assigned dr.          */
/*  Here dtd is known to be an actual task, not a non-assignment, and dr     */
/*  is known to be concerned with the same resource as ewd is.               */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprWorkDayLeafSet(KHE_DRS_EXPR_WORK_DAY ewd,
  KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_RESOURCE dr)
{
  ewd->u.float_val = KheTaskWorkloadPerTime(dtd->task);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprWorkDayLeafClear(KHE_DRS_EXPR_WORK_DAY ewd)               */
/*                                                                           */
/*  Inform leaf expression ewt that the assignment of drd to dtd carried     */
/*  out by KheDrsExprWorkDayLeafSet above has been removed.                  */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprWorkDayLeafClear(KHE_DRS_EXPR_WORK_DAY ewd)
{
  ewd->u.float_val = 0.0;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprWorkDaySetClosedValue(KHE_DRS_EXPR_WORK_DAY ewd,          */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing ewd, set its value suitably for its closed state.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprWorkDaySetClosedValue(KHE_DRS_EXPR_WORK_DAY ewd,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_TASK_ON_DAY dtd;
  dtd = ewd->resource_on_day->closed_asst;
  ewd->u.int_val = (dtd != NULL ? KheTaskWorkloadPerTime(dtd->task) : 0.0);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprWorkDayDoDebug(KHE_DRS_EXPR_WORK_DAY ewd,                 */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of ewd which is specific to it.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprWorkDayDoDebug(KHE_DRS_EXPR_WORK_DAY ewd,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "WORK_DAY(%s, %s)",
    KheDrsResourceId(ewd->resource_on_day->encl_dr),
    KheTimeGroupId(ewd->time_group));
  KheDrsExprDebugSignature((KHE_DRS_EXPR) ewd, soln, true, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_OR"                                              */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_OR KheDrsExprOrMake(KHE_DRS_RESOURCE dr,                    */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new KHE_DRS_EXPR_OR object with these attributes.                 */
/*  The caller must call KheDrsExprInitEnd after adding all the children.    */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_OR KheDrsExprOrMake(KHE_DRS_RESOURCE dr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_OR res;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_OR_TAG, dr, drs);
  res->closed_state = 0;
  /* later call to KheDrsExprInitEnd will set u.int_val */
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprOrAddChild(KHE_DRS_EXPR_OR eo, KHE_DRS_EXPR child_e,      */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  New child child_e has just been added to eo.  Update eo's closed         */
/*  state appropriately.  This happens to be the same as closing a child.    */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprOrAddChild(KHE_DRS_EXPR_OR eo, KHE_DRS_EXPR child_e,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  if( child_e->u.int_val == 1 )
    eo->closed_state += 1;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprOrChildHasOpened(KHE_DRS_EXPR_OR eo,                      */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of opening eo for solving, eo's child_index'th child, child_e,   */
/*  has been opened.  Update eo's closed state to reflect this.              */
/*                                                                           */
/*  The closed state of eo is the number of eo's closed children whose       */
/*  value is 1, so this update subtracts 1 from the closed state if          */
/*  child_e's value (which preserves its value before opening) is 1.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprOrChildHasOpened(KHE_DRS_EXPR_OR eo,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  if( child_e->u.int_val == 1 )
    eo->closed_state -= 1;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprOrChildHasClosed(KHE_DRS_EXPR_OR eo,                      */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of closing eo, eo's child_index'th child, child_e, has been      */
/*  closed.  Update eo's closed state to reflect this.                       */
/*                                                                           */
/*  The closed state of eo is the number of eo's closed children whose       */
/*  value is 1, so this update adds 1 to the closed state if child_e's       */
/*  value (which is up to date because it is closed first) is 1.             */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprOrChildHasClosed(KHE_DRS_EXPR_OR eo,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  if( child_e->u.int_val == 1 )
    eo->closed_state += 1;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprOrSetClosedValue(KHE_DRS_EXPR_OR eo,                      */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing eo, set its value suitably for its closed state.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprOrSetClosedValue(KHE_DRS_EXPR_OR eo,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  eo->u.int_val = (eo->closed_state > 0 ? 1 : 0);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprOrEvalSignature(KHE_DRS_EXPR_OR eo,                       */
/*    KHE_DRS_SOLN prev_soln, int next_di, KHE_DRS_SIGNATURE sig,            */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Evaluate expression eo within sig.                                       */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprOrEvalSignature(KHE_DRS_EXPR_OR eo, KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE prev_sig, int next_di, KHE_DRS_SIGNATURE next_sig,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int i, i1, i2, val;  KHE_DRS_EXPR child_e;

  if( KheDrsOpenChildrenIndexIsFirst(&eo->open_children_by_day, next_di) )
  {
    /* first day, so we have a 0 (false) value here */
    val = 0;
  }
  else
  {
    /* not first day, so retrieve a previous value */
    val = KheDrsExprDaySigVal((KHE_DRS_EXPR) eo, next_di - 1, prev_sig);
  }

  /* accumulate the values of the children of eo that finalized today */
  KheDrsOpenChildrenForEach(&eo->open_children_by_day, next_di, child_e, i)
    if( child_e->u.int_val == 1 )
      val = 1;

  if( KheDrsOpenChildrenIndexIsLast(&eo->open_children_by_day, next_di) )
  {
    /* last day; incorporate closed state and set value */
    if( eo->closed_state > 0 )
      val = 1;
    eo->u.int_val = val;
  }
  else
  {
    /* not last day; store val in next_soln */
    KheDrsSignatureAddState(next_sig, val, dsg, (KHE_DRS_EXPR) eo);
    /* KheDrsExprPutSigVal((KHE_DRS_EXPR) eo, next_di, sig, val); */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprOrDoDebug(KHE_DRS_EXPR_OR eo,                             */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of eo which is specific to it.       */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprOrDoDebug(KHE_DRS_EXPR_OR eo,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "OR(cs %d, iv %d)", eo->closed_state, eo->u.int_val);
  KheDrsExprDebugSignature((KHE_DRS_EXPR) eo, soln, false, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_AND"                                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_AND KheDrsExprAndMake(KHE_DRS_RESOURCE dr,                  */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new KHE_DRS_EXPR_AND object with these attributes.                */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_AND KheDrsExprAndMake(KHE_DRS_RESOURCE dr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_AND res;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_AND_TAG, dr, drs);
  res->u.int_val = UNDEF_INT;
  res->closed_state = 0;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprAndAddChild(KHE_DRS_EXPR_AND ea, KHE_DRS_EXPR child_e,    */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  New child child_e has just been added to ea.  Update ea's closed         */
/*  state appropriately.  This happens to be the same as closing a child.    */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprAndAddChild(KHE_DRS_EXPR_AND ea, KHE_DRS_EXPR child_e,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  if( child_e->u.int_val == 0 )
    ea->closed_state += 1;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprAndChildHasOpened(KHE_DRS_EXPR_AND ea,                    */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of opening ea for solving, ea's child_index'th child, child_e,   */
/*  has been opened.  Update ea's closed state to reflect this.              */
/*                                                                           */
/*  The closed state of ea is the number of ea's closed children whose       */
/*  value is 0, so this update subtracts 1 from the closed state if          */
/*  child_e's value (which preserves its value before opening) is 0.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprAndChildHasOpened(KHE_DRS_EXPR_AND ea,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  if( child_e->u.int_val == 0 )
    ea->closed_state -= 1;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprAndChildHasClosed(KHE_DRS_EXPR_AND ea,                    */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of closing ea, ea's child_index'th child, child_e, has been      */
/*  closed.  Update ea's closed state to reflect this.                       */
/*                                                                           */
/*  The closed state of ea is the number of ea's closed children whose       */
/*  value is 0, so this update adds 1 to the closed state if child_e's       */
/*  value (which is up to date because it is closed first) is 0.             */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprAndChildHasClosed(KHE_DRS_EXPR_AND ea,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  if( child_e->u.int_val == 0 )
    ea->closed_state += 1;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprAndSetClosedValue(KHE_DRS_EXPR_AND ea,                    */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing ea, set its value suitably for its closed state.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprAndSetClosedValue(KHE_DRS_EXPR_AND ea,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  ea->u.int_val = (ea->closed_state > 0 ? 0 : 1);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprAndEvalSignature(KHE_DRS_EXPR_AND ea,                     */
/*    KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, int next_di,           */
/*    KHE_DRS_SIGNATURE sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)                */
/*                                                                           */
/*  Evaluate expression ea within sig.                                       */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprAndEvalSignature(KHE_DRS_EXPR_AND ea,
  KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, int next_di,
  KHE_DRS_SIGNATURE next_sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int i, i1, i2, val;  KHE_DRS_EXPR child_e;

  if( KheDrsOpenChildrenIndexIsFirst(&ea->open_children_by_day, next_di) )
  {
    /* first day, so we have a 1 (true) value here */
    val = 1;
  }
  else
  {
    /* not first day, so retrieve a previous value */
    val = KheDrsExprDaySigVal((KHE_DRS_EXPR) ea, next_di - 1, prev_sig);
  }

  /* accumulate the values of the children of ea that finalized today */
  KheDrsOpenChildrenForEach(&ea->open_children_by_day, next_di, child_e, i)
    if( child_e->u.int_val == 0 )
      val = 0;

  if( KheDrsOpenChildrenIndexIsLast(&ea->open_children_by_day, next_di) )
  {
    /* last day; incorporate closed_state and set value */
    if( ea->closed_state > 0 )
      val = 0;
    ea->u.int_val = val;
  }
  else
  {
    /* not last day; store value in next_soln */
    KheDrsSignatureAddState(next_sig, val, dsg, (KHE_DRS_EXPR) ea);
    /* KheDrsExprPutSigVal((KHE_DRS_EXPR) ea, next_di, sig, val); */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprAndDoDebug(KHE_DRS_EXPR_AND ea,                           */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of ea which is specific to it.       */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprAndDoDebug(KHE_DRS_EXPR_AND ea,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "AND(cs %d, iv %d)", ea->closed_state, ea->u.int_val);
  KheDrsExprDebugSignature((KHE_DRS_EXPR) ea, soln, false, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_INT_SUM"                                         */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_INT_SUM KheDrsExprIntSumMake(KHE_DRS_RESOURCE dr,           */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new KHE_DRS_EXPR_INT_SUM object with these attributes.            */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_INT_SUM KheDrsExprIntSumMake(KHE_DRS_RESOURCE dr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_INT_SUM res;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_INT_SUM_TAG, dr, drs);
  res->u.int_val = UNDEF_INT;
  res->closed_state = 0;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSumAddChild(KHE_DRS_EXPR_INT_SUM eis,                  */
/*    KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  New child child_e has just been added to eis.  Update eis's closed       */
/*  state appropriately.  This happens to be the same as closing a child.    */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntSumAddChild(KHE_DRS_EXPR_INT_SUM eis,
  KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  eis->closed_state += child_e->u.int_val;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSumChildHasOpened(KHE_DRS_EXPR_INT_SUM eis,            */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of opening eis for solving, eis's child_index'th child, child_e, */
/*  has been opened.  Update eis's closed state to reflect this.             */
/*                                                                           */
/*  The closed state of eis is the sum of the values of eis's s closed       */
/*  children, so this update subtracts child_e's value (which preserves      */
/*  its value before opening) from the closed state.                         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntSumChildHasOpened(KHE_DRS_EXPR_INT_SUM eis,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  eis->closed_state -= child_e->u.int_val;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSumChildHasClosed(KHE_DRS_EXPR_INT_SUM eis,            */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of closing eis, eis's child_index'th child, child_e, has been    */
/*  closed.  Update eis's closed state to reflect this.                      */
/*                                                                           */
/*  The closed state of eis is the sum of the values of eis's s closed       */
/*  children, so this update adds child_e's value (which preserves           */
/*  its value before opening) to the closed state.                           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntSumChildHasClosed(KHE_DRS_EXPR_INT_SUM eis,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  eis->closed_state += child_e->u.int_val;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSumSetClosedValue(KHE_DRS_EXPR_INT_SUM eis,            */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing eis, set its value suitably for its closed state.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntSumSetClosedValue(KHE_DRS_EXPR_INT_SUM eis,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  eis->u.int_val = eis->closed_state;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSumEvalSignature(KHE_DRS_EXPR_INT_SUM eis,             */
/*    KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, int next_di,           */
/*    KHE_DRS_SIGNATURE sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)                */
/*                                                                           */
/*  Evaluate expression eis within sig.                                      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntSumEvalSignature(KHE_DRS_EXPR_INT_SUM eis,
  KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, int next_di,
  KHE_DRS_SIGNATURE next_sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int i, i1, i2, val;  KHE_DRS_EXPR child_e;

  if( KheDrsOpenChildrenIndexIsFirst(&eis->open_children_by_day, next_di) )
  {
    /* first day, so (ignoring closed_state) we have a 0 value here */
    val = 0;
  }
  else
  {
    /* not first day, so retrieve a previous value */
    val = KheDrsExprDaySigVal((KHE_DRS_EXPR) eis, next_di - 1, prev_sig);
  }

  /* accumulate the open values of the children of eis that finalized today */
  KheDrsOpenChildrenForEach(&eis->open_children_by_day, next_di, child_e, i)
    val += child_e->u.int_val;

  if( KheDrsOpenChildrenIndexIsLast(&eis->open_children_by_day, next_di) )
  {
    /* last day; add closed_state to the value */
    eis->u.int_val = val + eis->closed_state;
  }
  else
  {
    /* this not eis's last day; store value in next_soln */
    KheDrsSignatureAddState(next_sig, val, dsg, (KHE_DRS_EXPR) eis);
    /* KheDrsExprPutSigVal((KHE_DRS_EXPR) eis, next_di, sig, val); */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSumDoDebug(KHE_DRS_EXPR_INT_SUM eis,                   */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of eis which is specific to it.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntSumDoDebug(KHE_DRS_EXPR_INT_SUM eis,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "INT_SUM(%d)", eis->closed_state);
  KheDrsExprDebugSignature((KHE_DRS_EXPR) eis, soln, false, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_FLOAT_SUM"                                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_FLOAT_SUM KheDrsExprFloatSumMake(KHE_DRS_RESOURCE dr,       */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new KHE_DRS_EXPR_FLOAT_SUM object with these attributes.          */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_FLOAT_SUM KheDrsExprFloatSumMake(KHE_DRS_RESOURCE dr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_FLOAT_SUM res;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_FLOAT_SUM_TAG, dr, drs);
  res->u.float_val = UNDEF_FLOAT;
  res->closed_state = 0.0;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFloatSumAddChild(KHE_DRS_EXPR_FLOAT_SUM efs,              */
/*    KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  New child child_e has just been added to efs.  Update efs's closed       */
/*  state appropriately.  This happens to be the same as closing a child.    */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFloatSumAddChild(KHE_DRS_EXPR_FLOAT_SUM efs,
  KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  efs->closed_state += child_e->u.float_val;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFloatSumChildHasOpened(KHE_DRS_EXPR_FLOAT_SUM efs,        */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of opening efs for solving, efs's child_index'th child, child_e, */
/*  has been opened.  Update efs's closed state to reflect this.             */
/*                                                                           */
/*  The closed state of efs is the sum of the values of efs's closed         */
/*  children, so this update subtracts child_e's value (which preserves      */
/*  its value before opening) from the closed state.                         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFloatSumChildHasOpened(KHE_DRS_EXPR_FLOAT_SUM efs,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  efs->closed_state -= child_e->u.float_val;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFloatSumChildHasClosed(KHE_DRS_EXPR_FLOAT_SUM efs,        */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of opening efs for solving, efs's child_index'th child, child_e, */
/*  has been closed.  Update efs's closed state to reflect this.             */
/*                                                                           */
/*  The closed state of efs is the sum of the values of efs's closed         */
/*  children, so this update adds child_e's value (which preserves           */
/*  its value before opening) to the closed state.                           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFloatSumChildHasClosed(KHE_DRS_EXPR_FLOAT_SUM efs,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  efs->closed_state += child_e->u.float_val;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFloatSumSetClosedValue(KHE_DRS_EXPR_FLOAT_SUM efs,        */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing efs, set its value suitably for its closed state.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFloatSumSetClosedValue(KHE_DRS_EXPR_FLOAT_SUM efs,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  efs->u.float_val = efs->closed_state;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFloatSumEvalSignature(KHE_DRS_EXPR_FLOAT_SUM efs,         */
/*    KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, int next_di,           */
/*    KHE_DRS_SIGNATURE sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)                */
/*                                                                           */
/*  Evaluate expression efs in sig.                                          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFloatSumEvalSignature(KHE_DRS_EXPR_FLOAT_SUM efs,
  KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, int next_di,
  KHE_DRS_SIGNATURE next_sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int i, i1, i2;  float val;  KHE_DRS_EXPR child_e;

  if( KheDrsOpenChildrenIndexIsFirst(&efs->open_children_by_day, next_di) )
  {
    /* first day, so we have a 0.0 value here */
    val = 0.0;
  }
  else
  {
    /* not first day, so retrieve a previous value */
    val = (float) KheDrsExprDaySigVal((KHE_DRS_EXPR) efs, next_di-1, prev_sig)
      / 100.0;
  }

  /* accumulate the open values of the children of efs that finalized today */
  KheDrsOpenChildrenForEach(&efs->open_children_by_day, next_di, child_e, i)
    val += child_e->u.float_val;

  if( KheDrsOpenChildrenIndexIsLast(&efs->open_children_by_day, next_di) )
  {
    /* last day; add closed_state to the value */
    efs->u.float_val = val + efs->closed_state;
  }
  else
  {
    /* this not efs's last day; store value in next_soln */
    KheDrsSignatureAddState(next_sig, (int) (val * 100.0), dsg,
      (KHE_DRS_EXPR) efs);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFloatSumDoDebug(KHE_DRS_EXPR_FLOAT_SUM efs,               */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of efs which is specific to it.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFloatSumDoDebug(KHE_DRS_EXPR_FLOAT_SUM efs,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "FLOAT_SUM(%.2f)", efs->closed_state);
  KheDrsExprDebugSignature((KHE_DRS_EXPR) efs, soln, true, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_INT_DEV"                                         */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprIntDevDev(KHE_DRS_EXPR_INT_DEV eid, int val)               */
/*                                                                           */
/*  Deviation function to use with these expressions.                        */
/*                                                                           */
/*****************************************************************************/

static int KheDrsExprIntDevDev(KHE_DRS_EXPR_INT_DEV eid, int val)
{
  if( val == 0 && eid->allow_zero )
    return 0;
  else if( val > eid->max_limit )
    return val - eid->max_limit;
  else if( val < eid->min_limit )
    return eid->min_limit - val;
  else
    return 0;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_INT_DEV KheDrsExprIntDevMake(int min_limit, int max_limit,  */
/*    bool allow_zero, KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs) */
/*                                                                           */
/*  Make a new KHE_DRS_EXPR_INT_DEV object with these attributes.            */
/*                                                                           */
/*  There is no dom_test because its type is always KHE_DRS_DOM_UNUSED.      */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_INT_DEV KheDrsExprIntDevMake(int min_limit, int max_limit,
  bool allow_zero, KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_INT_DEV res;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_INT_DEV_TAG, dr, drs);
  res->u.int_val = UNDEF_INT;
  res->min_limit = min_limit;
  res->max_limit = max_limit;
  res->allow_zero = allow_zero;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntDevAddChild(KHE_DRS_EXPR_INT_DEV eid,                  */
/*    KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  New child child_e has just been added to eid.  Update eid's closed       */
/*  state appropriately.  This happens to be the same as closing a child.    */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntDevAddChild(KHE_DRS_EXPR_INT_DEV eid,
  KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  /* nothing to do here */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntDevChildHasOpened(KHE_DRS_EXPR_INT_DEV eid,            */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of opening eid for solving, eid's child_index'th child, child_e, */
/*  has been opened.  Update eid's closed state to reflect this.             */
/*                                                                           */
/*  The closed state of eid is the value of eid's single closed child.       */
/*  This is not stored explicitly, so this update does nothing.              */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntDevChildHasOpened(KHE_DRS_EXPR_INT_DEV eid,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  /* nothing to do here */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntDevChildHasClosed(KHE_DRS_EXPR_INT_DEV eid,            */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of opening eid for solving, eid's child_index'th child, child_e, */
/*  has been closed.  Update eid's closed state to reflect this.             */
/*                                                                           */
/*  The closed state of eid is the value of eid's single closed child.       */
/*  This is not stored explicitly, so this update does nothing.              */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntDevChildHasClosed(KHE_DRS_EXPR_INT_DEV eid,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  /* nothing to do here */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntDevSetClosedValue(KHE_DRS_EXPR_INT_DEV eid,            */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing eid, set its value suitably for its closed state.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntDevSetClosedValue(KHE_DRS_EXPR_INT_DEV eid,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR child_e;
  child_e = HaArrayFirst(eid->children);
  eid->u.int_val = KheDrsExprIntDevDev(eid, child_e->u.int_val);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntDevEvalSignature(KHE_DRS_EXPR_INT_DEV eid,             */
/*    KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, int next_di,           */
/*    KHE_DRS_SIGNATURE sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)                */
/*                                                                           */
/*  Evaluate expression eid within sig.                                      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntDevEvalSignature(KHE_DRS_EXPR_INT_DEV eid,
  KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, int next_di,
  KHE_DRS_SIGNATURE next_sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int i, i1, i2, val;  KHE_DRS_EXPR child_e;

  if( KheDrsOpenChildrenIndexIsFirst(&eid->open_children_by_day, next_di) )
  {
    /* first day */
    val = 0;
  }
  else
  {
    /* not first day, so retrieve a previous value */
    HnAbort("KheDrsExprIntDevEvalSignature internal error (non-first day)");
  }

  /* accumulate the open values of the children of eid that finalized today */
  KheDrsOpenChildrenForEach(&eid->open_children_by_day, next_di, child_e, i)
    val += KheDrsExprIntDevDev(eid, child_e->u.int_val);

  if( KheDrsOpenChildrenIndexIsLast(&eid->open_children_by_day, next_di) )
  {
    /* last day; set value */
    eid->u.int_val = val;
  }
  else
  {
    /* not last day; store val in next_soln */
    HnAbort("KheDrsExprIntDevEvalSignature internal error (non-last day)");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntDevDoDebug(KHE_DRS_EXPR_INT_DEV eid,                   */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of eid which is specific to it.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntDevDoDebug(KHE_DRS_EXPR_INT_DEV eid,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "INT_DEV(%d, %d, %s)", eid->min_limit, eid->max_limit,
    bool_show(eid->allow_zero));
  KheDrsExprDebugSignature((KHE_DRS_EXPR) eid, soln, false, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_FLOAT_DEV"                                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_FLOAT_DEV KheDrsExprFloatDevMake(int min_limit,             */
/*    int max_limit, bool allow_zero, KHE_DRS_RESOURCE dr,                   */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new KHE_DRS_EXPR_FLOAT_DEV object with these attributes.          */
/*                                                                           */
/*  There is no dom_test because its type is always KHE_DRS_DOM_UNUSED.      */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_FLOAT_DEV KheDrsExprFloatDevMake(int min_limit,
  int max_limit, bool allow_zero, KHE_DRS_RESOURCE dr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_FLOAT_DEV res;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_FLOAT_DEV_TAG, dr, drs);
  res->u.int_val = UNDEF_INT;
  res->min_limit = min_limit;
  res->max_limit = max_limit;
  res->allow_zero = allow_zero;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFloatDevAddChild(KHE_DRS_EXPR_FLOAT_DEV efd,              */
/*    KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  New child child_e has just been added to efd.  Update efd's closed       */
/*  state appropriately.  This happens to be the same as closing a child.    */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFloatDevAddChild(KHE_DRS_EXPR_FLOAT_DEV efd,
  KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  /* nothing to do here */
}


/*****************************************************************************/
/*                                                                           */
/*  int KheLimitWorkloadMonitorDev(KHE_DRS_EXPR_FLOAT_DEV efd,               */
/*    float workload)                                                        */
/*                                                                           */
/*  Return the deviation of workload wrt efd.                                */
/*                                                                           */
/*****************************************************************************/

static int KheLimitWorkloadMonitorDev(KHE_DRS_EXPR_FLOAT_DEV efd,
  float workload)
{
  if( efd->allow_zero && workload < 0.001 )
    return 0;
  else if( workload < efd->min_limit - 0.001 )
    return (int) ceil(efd->min_limit - 0.001 - workload);
  else if( workload > efd->max_limit + 0.001 )
    return (int) ceil(workload - efd->max_limit - 0.001);
  else
    return 0;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFloatDevChildHasOpened(KHE_DRS_EXPR_FLOAT_DEV efd,        */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of opening efd for solving, efd's child_index'th child, child_e, */
/*  has been opened.  Update efd's closed state to reflect this.             */
/*                                                                           */
/*  The closed state of efd is the value of efd's single closed child,       */
/*  so this update needs to do nothing.                                      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFloatDevChildHasOpened(KHE_DRS_EXPR_FLOAT_DEV efd,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  /* nothing to do here */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFloatDevChildHasClosed(KHE_DRS_EXPR_FLOAT_DEV efd,        */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of closing efd, efd's child_index'th child, child_e,             */
/*  has been closed.  Update efd's closed state to reflect this.             */
/*                                                                           */
/*  The closed state of efd is the value of efd's single closed child,       */
/*  so this update needs to do nothing.                                      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFloatDevChildHasClosed(KHE_DRS_EXPR_FLOAT_DEV efd,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  /* nothing to do here */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFloatDevSetClosedValue(KHE_DRS_EXPR_FLOAT_DEV efd,        */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing efd, set its value suitably for its closed state.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFloatDevSetClosedValue(KHE_DRS_EXPR_FLOAT_DEV efd,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR child_e;
  child_e = HaArrayFirst(efd->children);
  efd->u.int_val = KheLimitWorkloadMonitorDev(efd, child_e->u.float_val);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFloatDevEvalSignature(KHE_DRS_EXPR_FLOAT_DEV efd,         */
/*    KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, int next_di,           */
/*    KHE_DRS_SIGNATURE sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)                */
/*                                                                           */
/*  Evaluate expression efd within sig.                                      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFloatDevEvalSignature(KHE_DRS_EXPR_FLOAT_DEV efd,
  KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, int next_di,
  KHE_DRS_SIGNATURE sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int i, i1, i2, val;  KHE_DRS_EXPR child_e;

  if( KheDrsOpenChildrenIndexIsFirst(&efd->open_children_by_day, next_di) )
  {
    /* first day */
    val = 0;
  }
  else
  {
    /* not first day, so retrieve a previous value */
    HnAbort("KheDrsExprFloatDevEvalSignature internal error (non-first day)");
    val = 0;  /* keep compiler happy */
  }

  /* accumulate the open values of the children of efd that finalized today */
  KheDrsOpenChildrenForEach(&efd->open_children_by_day, next_di, child_e, i)
    val += KheLimitWorkloadMonitorDev(efd, child_e->u.float_val);

  if( KheDrsOpenChildrenIndexIsLast(&efd->open_children_by_day, next_di) )
  {
    /* last day; set value */
    efd->u.int_val = val;
  }
  else
  {
    /* not last day; store value in next_soln */
    HnAbort("KheDrsExprFloatDevEvalSignature internal error (non-last day)");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFloatDevDoDebug(KHE_DRS_EXPR_FLOAT_DEV efd,               */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of efd which is specific to it.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFloatDevDoDebug(KHE_DRS_EXPR_FLOAT_DEV efd,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "FLOAT_DEV(%d, %d, %s)", efd->min_limit, efd->max_limit,
    bool_show(efd->allow_zero));
  KheDrsExprDebugSignature((KHE_DRS_EXPR) efd, soln, false, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_INT_SUM_COST - dom tables"                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_COST KheDrsExprIntSumCostExtraCost(KHE_DRS_EXPR_INT_SUM_COST eisc,   */
/*    int e, int l, int x, bool debug)                                       */
/*                                                                           */
/*  Return the extra cost in the dominance testing formula with these        */
/*  attributes.  This is function Gamma in the documentation.                */
/*                                                                           */
/*****************************************************************************/
static int KheDrsExprIntSumCostDelta(KHE_DRS_EXPR_INT_SUM_COST eisc,
  int lower_det, int upper_det);
static KHE_COST KheDrsExprCostCost(KHE_DRS_EXPR_COST eisc, int dev);
#define f(e, d) KheDrsExprCostCost((KHE_DRS_EXPR_COST) (e), (d))

static KHE_COST KheDrsExprIntSumCostExtraCost(KHE_DRS_EXPR_INT_SUM_COST eisc,
  int e, int l, int y, bool debug)
{
  int delta1, delta2;  KHE_COST res;
  delta1 = KheDrsExprIntSumCostDelta(eisc, l + y, l + y + eisc->history_after);
  delta2 = KheDrsExprIntSumCostDelta(eisc, l, l + e + eisc->history_after);
  res = f(eisc, delta1) - f(eisc, delta2);
  if( debug )
    fprintf(stderr, "    Gamma(e %d, l %d, y %d) = f(%d) %.5f - f(%d) "
      "%.5f = %.5f\n", e, l, y, delta1, KheCostShow(f(eisc, delta1)),
      delta2, KheCostShow(f(eisc, delta2)), KheCostShow(res));
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_COST KheDrsExprIntSumCostFindAvailCost(                              */
/*    KHE_DRS_EXPR_INT_SUM_COST eisc, int e, int l1, int l2)                 */
/*                                                                           */
/*  Find the available cost for eisc given e, l1, and l2.                    */
/*  This implements the outer "min" of the formula in the documentation.     */
/*  KheDrsExprIntSumCostExtraCost is Gamma.                                  */
/*                                                                           */
/*****************************************************************************/
#define DEBUG_INT_SUM_DOM(e, d1, d2)					\
  (DEBUG37 && (e) == 10 && (d1) == 10 && (d2) == 5)

static KHE_COST KheDrsExprIntSumCostFindAvailCost(
  KHE_DRS_EXPR_INT_SUM_COST eisc, int e, int l1, int l2)
{
  int y;  KHE_COST res, extra_cost1, extra_cost2, cost;  bool debug;
  debug = DEBUG_INT_SUM_DOM(e, l1, l2);
  if( debug )
    fprintf(stderr, "[ KheDrsExprIntSumCostFindAvailCost(%s, e %d, l1 %d,"
      " l2 %d)\n", KheDrsMonitorInfoId(eisc->monitor_info), e, l1, l2);
  res = KheCost(INT_MAX, INT_MAX);
  for( y = 0;  y <= e;  y++ )
  {
    extra_cost2 = KheDrsExprIntSumCostExtraCost(eisc, e, l2, y, debug);
    extra_cost1 = KheDrsExprIntSumCostExtraCost(eisc, e, l1, y, debug);
    cost = extra_cost2 - extra_cost1;
    if( debug )
      fprintf(stderr, "  y = %2d: %8.5f - %8.5f = %8.5f\n", y,
        KheCostShow(extra_cost2), KheCostShow(extra_cost1), KheCostShow(cost));
    if( cost < res )
      res = cost;
  }
  HnAssert(res <= 0, "KheDrsExprIntSumCostFindAvailCost: res > 0");
  if( debug )
    fprintf(stderr, "] KheDrsExprIntSumCostFindAvailCost returning %.5f\n",
      KheCostShow(res));
  return res;
}

/*****************************************************************************/
/*                                                                           */
/*  KHE_COST KheDrsExprIntSumCostFindCorrAvailCost(                          */
/*    KHE_DRS_EXPR_INT_SUM_COST eisc, int e, int a1, int a2, int l1, int l2) */
/*                                                                           */
/*  Find the correlated available cost for eisc given e, a1, a2, l1, and l2. */
/*                                                                           */
/*****************************************************************************/
#define DEBUG_INT_SUM_CORR(e, a1, a2, l1, l2)				\
  (DEBUG56 && (e) == 1 && (a1) == 0 && (a2) == 1 && (l1) == 0 && (l2) == 2)

static KHE_COST KheDrsExprIntSumCostFindCorrAvailCost(
  KHE_DRS_EXPR_INT_SUM_COST eisc, int e, int a1, int a2, int l1, int l2)
{
  KHE_COST res, extra_cost1, extra_cost2, cost;
  int y1, y2, D_first, D_last, i;  bool debug;
  debug = DEBUG_INT_SUM_CORR(e, a1, a2, l1, l2);
  if( debug )
    fprintf(stderr, "[ KheDrsExprIntSumCostFindCorrAvailCost(%s, e %d, a1 %d,"
      " a2 %d, l1 %d, l2 %d)\n", KheDrsMonitorInfoId(eisc->monitor_info),
      e, a1, a2, l1, l2);

  /* work out D, in the form D_first .. D_last */
  if( a1 == 0 )
  {
    if( a2 == 0 )
      D_first = D_last = 0;
    else
      D_first = -1, D_last = 0;
  }
  else
  {
    if( a2 == 0 )
      D_first = 0, D_last = 1;
    else
      D_first = D_last = 0;
  }

  /* do the rest */
  res = KheCost(INT_MAX, INT_MAX);
  for( y1 = a1;  y1 <= e;  y1++ )
    for( i = D_first;  i <= D_last;  i++ )
    {
      y2 = y1 - i;
      if( a2 <= y2 && y2 <= e )
      {
	extra_cost2 = KheDrsExprIntSumCostExtraCost(eisc, e, l2, y2, debug);
	extra_cost1 = KheDrsExprIntSumCostExtraCost(eisc, e, l1, y1, debug);
	cost = extra_cost2 - extra_cost1;
	if( debug )
	  fprintf(stderr, "  y1 = %2d, y2 = %2d: %8.5f - %8.5f = %8.5f\n", y1,
	    y2, KheCostShow(extra_cost2), KheCostShow(extra_cost1),
	    KheCostShow(cost));
	if( cost < res )
	  res = cost;
      }
    }
  /* *** there are cases where the result is genuinely positive
  HnAssert(res <= 0, "KheDrsExprIntSumCostFindCorrAvailCost: res > 0");
  *** */
  if( debug )
    fprintf(stderr, "] KheDrsExprIntSumCostFindCorrAvailCost returning %.5f\n",
      KheCostShow(res));
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsExprIsSingleDay(KHE_DRS_EXPR e, KHE_FRAME days_frame,         */
/*    KHE_TIME_GROUP *time_group)                                            */
/*                                                                           */
/*  If e covers a single day of days_frame, return true and set *time_group  */
/*  to the time group of that day in days_frame.  Otherwise return false     */
/*  with *time_group set to NULL.                                            */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsExprIsSingleDay(KHE_DRS_EXPR e, KHE_FRAME days_frame,
  KHE_TIME_GROUP *time_group)
{
  KHE_TIME_GROUP tg;  KHE_DRS_EXPR child_e;  int i;
  switch( e->tag )
  {
    case KHE_DRS_EXPR_ASSIGNED_TASK_TAG:

      /* shortcut; we aren't expecting this case to ever occur */
      return *time_group = NULL, false;

    case KHE_DRS_EXPR_BUSY_TIME_TAG:

      { KHE_DRS_EXPR_BUSY_TIME ebt;
	ebt = (KHE_DRS_EXPR_BUSY_TIME) e;
	*time_group = KheFrameTimeTimeGroup(days_frame, ebt->time);
	return true;
      }

    case KHE_DRS_EXPR_FREE_TIME_TAG:

      { KHE_DRS_EXPR_FREE_TIME eft;
	eft = (KHE_DRS_EXPR_FREE_TIME) e;
	*time_group = KheFrameTimeTimeGroup(days_frame, eft->time);
	return true;
      }

    case KHE_DRS_EXPR_WORK_TIME_TAG:

      { KHE_DRS_EXPR_WORK_TIME ewt;
	ewt = (KHE_DRS_EXPR_WORK_TIME) e;
	*time_group = KheFrameTimeTimeGroup(days_frame, ewt->time);
	return true;
      }

    case KHE_DRS_EXPR_BUSY_DAY_TAG:
      
      { KHE_DRS_EXPR_BUSY_DAY ebd;
	ebd = (KHE_DRS_EXPR_BUSY_DAY) e;
        *time_group = ebd->time_group;
	return true;
      }

    case KHE_DRS_EXPR_FREE_DAY_TAG:

      { KHE_DRS_EXPR_FREE_DAY efd;
	efd = (KHE_DRS_EXPR_FREE_DAY) e;
        *time_group = efd->time_group;
	return true;
      }

    case KHE_DRS_EXPR_WORK_DAY_TAG:

      { KHE_DRS_EXPR_WORK_DAY ewd;
	ewd = (KHE_DRS_EXPR_WORK_DAY) e;
        *time_group = ewd->time_group;
	return true;
      }

    case KHE_DRS_EXPR_OR_TAG:
    case KHE_DRS_EXPR_AND_TAG:	

      *time_group = NULL;
      HaArrayForEach(e->children, child_e, i)
      {
	if( !KheDrsExprIsSingleDay(child_e, days_frame, &tg) )
	  return false;
	else if( *time_group == NULL )
	  *time_group = tg;
	else if( !KheTimeGroupEqual(*time_group, tg) )
	  return false;
      }
      return (*time_group != NULL);
      
    case KHE_DRS_EXPR_INT_SUM_TAG:
    case KHE_DRS_EXPR_FLOAT_SUM_TAG:
    case KHE_DRS_EXPR_INT_DEV_TAG:
    case KHE_DRS_EXPR_FLOAT_DEV_TAG:
    case KHE_DRS_EXPR_INT_SUM_COST_TAG:
    case KHE_DRS_EXPR_INT_SEQ_COST_TAG:

      /* shortcut; we aren't expecting these cases to ever occur */
      return *time_group = NULL, false;

    default:

      HnAbort("KheDrsExprMightSpanMoreThanOneDay internal error (tag %d)",
	e->tag);
      return *time_group = NULL, false;  /* keep compiler happy */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsExprIntSumCostNeedsCorrTable(KHE_DRS_EXPR_INT_SUM_COST eisc,  */
/*    KHE_FRAME days_frame)                                                  */
/*                                                                           */
/*  Return true if eisc needs a corr table - if any of its children span     */
/*  more than one day of days_frame.                                         */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsExprIntSumCostNeedsCorrTable(KHE_DRS_EXPR_INT_SUM_COST eisc,
  KHE_FRAME days_frame)
{
  KHE_DRS_EXPR child_e;  KHE_TIME_GROUP tg;  int i;
  HaArrayForEach(eisc->children, child_e, i)
    if( !KheDrsExprIsSingleDay(child_e, days_frame, &tg) )
      return true;
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSumCostSetDomTables(KHE_DRS_EXPR_INT_SUM_COST eisc,    */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Set the main_dom_table3 and corr_dom_table5 fields of eisc.              */
/*                                                                           */
/*****************************************************************************/
static void KheDrsExprIntSumCostDebugDomTables(KHE_DRS_EXPR_INT_SUM_COST eisc,
  int verbosity, int indent, FILE *fp);

static void KheDrsExprIntSumCostSetDomTables(KHE_DRS_EXPR_INT_SUM_COST eisc,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int e, l1, l2, a1, a2, child_count, lim;  KHE_COST cost;
  KHE_DRS_COST_TUPLE ct;  char *debug_str;

  /* build eisc->dom_table */
  HnAssert(eisc->main_dom_table3 == NULL,
    "KheDrsExprIntSumCostSetDomTable: main_dom_table");
  child_count = HaArrayCount(eisc->children);
  debug_str = KheDrsMonitorInfoId(eisc->monitor_info);
  eisc->main_dom_table3 = KheDrsDim3TableMake(debug_str, drs->arena);
  for( e = 0;  e <= child_count;  e++ )
    for( l1 = eisc->history;  l1 <= child_count + eisc->history - e;  l1++ )
      for( l2 = eisc->history;  l2 <= child_count + eisc->history - e;  l2++ )
      {
	cost = KheDrsExprIntSumCostFindAvailCost(eisc, e, l1, l2);
	ct = KheDrsCostTupleMake(cost, 0, false, 0);
	KheDrsDim3TablePut(eisc->main_dom_table3, e, l1, l2, ct, drs);
      }

  /* build eisc->corr_dom_table */
  if( KheDrsExprIntSumCostNeedsCorrTable(eisc, drs->days_frame) )
  {
    eisc->corr_dom_table5 = KheDrsDim5TableMake(
      HnStringMake(drs->arena, "Corr %s", debug_str), drs->arena);
    for( e = 1;  e <= child_count;  e++ )
    {
      lim = child_count + eisc->history - e;
      for( a1 = 0;  a1 <= 1;  a1++ )
	for( a2 = 0;  a2 <= 1;  a2++ )
	  for( l1 = eisc->history;  l1 <= lim;  l1++ )
	    for( l2 = eisc->history;  l2 <= lim;  l2++ )
	    {
	      cost = KheDrsExprIntSumCostFindCorrAvailCost(eisc, e,a1,a2,l1,l2);
	      ct = KheDrsCostTupleMake(cost, 0, false, 0);
	      KheDrsDim5TablePut(eisc->corr_dom_table5, e, a1, a2, l1, l2, ct,
		drs);
	    }
    }
  }
  if( DEBUG36 )
    KheDrsExprIntSumCostDebugDomTables(eisc, 1, 0, stderr);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsExprIntSumCostWantDebugDomTable(                              */
/*    KHE_DRS_EXPR_INT_SUM_COST eisc)                                        */
/*                                                                           */
/*  Return true if we want to debug eisc's dom table.                        */
/*                                                                           */
/*  At present we are doing this is ec's monitor is a resource monitor       */
/*  and the resource it monitors is the first resource of its type.          */
/*  This is to limit the debug output to a manageable amount.                */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsExprIntSumCostWantDebugDomTable(
  KHE_DRS_EXPR_INT_SUM_COST eisc)
{
  KHE_MONITOR m;  KHE_RESOURCE r;  KHE_RESOURCE_TYPE rt;

  /* find the resource eisc monitors; return false if not resource monitor */
  m = eisc->monitor_info->monitor;
  switch( KheMonitorTag(m) )
  {
    case KHE_AVOID_CLASHES_MONITOR_TAG:

      r = KheAvoidClashesMonitorResource((KHE_AVOID_CLASHES_MONITOR) m);
      break;

    case KHE_AVOID_UNAVAILABLE_TIMES_MONITOR_TAG:

      r = KheAvoidUnavailableTimesMonitorResource(
	(KHE_AVOID_UNAVAILABLE_TIMES_MONITOR) m);
      break;

    case KHE_LIMIT_IDLE_TIMES_MONITOR_TAG:

      r = KheLimitIdleTimesMonitorResource((KHE_LIMIT_IDLE_TIMES_MONITOR) m);
      break;

    case KHE_CLUSTER_BUSY_TIMES_MONITOR_TAG:

      r = KheClusterBusyTimesMonitorResource(
	(KHE_CLUSTER_BUSY_TIMES_MONITOR) m);
      break;

    case KHE_LIMIT_BUSY_TIMES_MONITOR_TAG:

      r = KheLimitBusyTimesMonitorResource((KHE_LIMIT_BUSY_TIMES_MONITOR) m);
      break;

    case KHE_LIMIT_WORKLOAD_MONITOR_TAG:

      r = KheLimitWorkloadMonitorResource((KHE_LIMIT_WORKLOAD_MONITOR) m);
      break;

    default:

      return false;
  }

  /* return true if r is the first resource of its type */
  rt = KheResourceResourceType(r);
  return KheResourceTypeResourceCount(rt) > 0 &&
    KheResourceTypeResource(rt, 0) == r;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSumCostDebugDomTables(KHE_DRS_EXPR_INT_SUM_COST eisc,  */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of eisc's dom table.                                         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntSumCostDebugDomTables(KHE_DRS_EXPR_INT_SUM_COST eisc,
  int verbosity, int indent, FILE *fp)
{
  if( KheDrsExprIntSumCostWantDebugDomTable(eisc) )
  {
    fprintf(fp, "%*s[ ExprCost for %s\n", indent, "",
      KheDrsMonitorInfoId(eisc->monitor_info));
    fprintf(fp, "%*s  cost_fn %s, combined_weight %.5f, min_limit %d,"
      " max_limit %d, allow_zero %s\n", indent, "",
      KheCostFunctionShow(eisc->cost_fn), KheCostShow(eisc->combined_weight),
      eisc->min_limit, eisc->max_limit, bool_show(eisc->allow_zero));
    fprintf(fp, "%*s  history_before %d, history_after %d, history %d\n",
      indent, "", eisc->history_before, eisc->history_after, eisc->history);
    KheDrsDim3TableDebug(eisc->main_dom_table3, verbosity, indent + 2, fp);
    KheDrsDim5TableDebug(eisc->corr_dom_table5, verbosity, indent + 2, fp);
    fprintf(fp, "%*s]\n", indent, "");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_COST"                                            */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_COST KheDrsExprCostCost(KHE_DRS_EXPR_COST ec, int dev)               */
/*                                                                           */
/*  Return f(dev) for ec.                                                    */
/*                                                                           */
/*****************************************************************************/

static KHE_COST KheDrsExprCostCost(KHE_DRS_EXPR_COST ec, int dev)
{
  switch( ec->cost_fn )
  {
    case KHE_STEP_COST_FUNCTION:

      return dev > 0 ? ec->combined_weight : 0;

    case KHE_LINEAR_COST_FUNCTION:

      return dev * ec->combined_weight;

    case KHE_QUADRATIC_COST_FUNCTION:

      return dev * dev * ec->combined_weight;

    default:

      HnAbort("KheDrsExprCostCost internal error");
      return KheCost(0, 0);  /* keep compiler happy */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_INT_SUM_COST"                                    */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprIntSumCostDelta(KHE_DRS_EXPR_INT_SUM_COST eisc,            */
/*    int lower_det, int upper_det)                                          */
/*                                                                           */
/*  Return the deviation of eisc for the given lower and upper determinants. */
/*  This is function delta(l, u) from the documentation.                     */
/*                                                                           */
/*****************************************************************************/

static int KheDrsExprIntSumCostDelta(KHE_DRS_EXPR_INT_SUM_COST eisc,
  int lower_det, int upper_det)
{
  if( eisc->allow_zero && lower_det == 0 )
    return 0;
  else if( lower_det > eisc->max_limit )
    return lower_det - eisc->max_limit;
  else if( upper_det < eisc->min_limit )
    return eisc->min_limit - upper_det;
  else
    return 0;
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprIntSumCostDev(KHE_DRS_EXPR_INT_SUM_COST eisc,              */
/*    int active_children, int unassigned_children)                          */
/*                                                                           */
/*  Return the deviation of eisc, assuming that eisc has the given number    */
/*  of active children, and the given number of unassigned children.  This   */
/*  is closely related to delta but more convenient at times.                */
/*                                                                           */
/*****************************************************************************/

static int KheDrsExprIntSumCostDev(KHE_DRS_EXPR_INT_SUM_COST eisc,
  int active_children, int unassigned_children)
{
  return KheDrsExprIntSumCostDelta(eisc, active_children,
    active_children + unassigned_children);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_ADJUST_TYPE KheDrsAdjustType(KHE_COST_FUNCTION cost_fn,          */
/*    int max_limit)                                                         */
/*                                                                           */
/*  Return the adjust type, determined by the cost function and max limit.   */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_ADJUST_TYPE KheDrsAdjustType(KHE_COST_FUNCTION cost_fn,
  int max_limit)
{
  if( max_limit == INT_MAX )
    return KHE_DRS_ADJUST_NO_MAX;
  else if( cost_fn == KHE_LINEAR_COST_FUNCTION )
    return KHE_DRS_ADJUST_LINEAR;
  else if( cost_fn == KHE_STEP_COST_FUNCTION )
    return KHE_DRS_ADJUST_STEP;
  else
    return KHE_DRS_ADJUST_ORDINARY;
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheDrsAdjustTypeShow(KHE_DRS_ADJUST_TYPE dct)                      */
/*                                                                           */
/*  Return a string representation of dct.                                   */
/*                                                                           */
/*****************************************************************************/

static char *KheDrsAdjustTypeShow(KHE_DRS_ADJUST_TYPE dct)
{
  switch( dct )
  {
    case KHE_DRS_ADJUST_NO_MAX:	  return "NO_MAX";
    case KHE_DRS_ADJUST_LINEAR:	  return "LINEAR";
    case KHE_DRS_ADJUST_STEP:	  return "STEP";
    case KHE_DRS_ADJUST_ORDINARY: return "ORDINARY";
    default:			  return "??";
  }
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_INT_SUM_COST KheDrsExprIntSumCostMake(                      */
/*    KHE_DRS_MONITOR_INFO mi, KHE_COST_FUNCTION cost_fn,                    */
/*    KHE_COST combined_weight, int min_limit, int max_limit,                */
/*    bool allow_zero, int history_before, int history_after,                */
/*    int history, KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs)     */
/*                                                                           */
/*  Make a new KHE_DRS_EXPR_INT_SUM_COST object with these attributes.       */
/*                                                                           */
/*****************************************************************************/
static void KheDrsAddWeight(KHE_DYNAMIC_RESOURCE_SOLVER drs, KHE_COST weight);

static KHE_DRS_EXPR_INT_SUM_COST KheDrsExprIntSumCostMake(
  KHE_DRS_MONITOR_INFO mi, KHE_COST_FUNCTION cost_fn,
  KHE_COST combined_weight, int min_limit, int max_limit,
  bool allow_zero, int history_before, int history_after,
  int history, KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  /* build the object */
  KHE_DRS_EXPR_INT_SUM_COST res;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_INT_SUM_COST_TAG,dr,drs);
  res->u.int_val = UNDEF_INT;
  res->monitor_info = mi;
  res->cost_fn = cost_fn;
  res->combined_weight = combined_weight;
  res->min_limit = min_limit;
  res->max_limit = max_limit;
  res->allow_zero = allow_zero;
  res->adjust_type = KheDrsAdjustType(cost_fn, max_limit);
  res->history_before = history_before;
  res->history_after = history_after;
  res->history = history;
  res->main_dom_table3 = NULL;  /* reset later */
  res->corr_dom_table5 = NULL;  /* possibly reset later */
  res->closed_state = 0;
  KheDrsOpenChildrenInit(&res->open_children_by_shift, drs);
  KheDrsAddWeight(drs, combined_weight);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_INT_SUM_COST KheDrsExprIntSumCostZeroMaxMake(               */
/*    KHE_DRS_MONITOR_INFO mi, KHE_COST_FUNCTION cost_fn,                    */
/*    KHE_COST combined_weight, KHE_DRS_RESOURCE dr,                         */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make an INT_SUM_COST object for the case where there is a maximum limit  */
/*  of 0 (so the dominance test is <=) and no history.  This is the usual    */
/*  (but not the only) replacement for the now omitted EXPR_COST.            */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_INT_SUM_COST KheDrsExprIntSumCostZeroMaxMake(
  KHE_DRS_MONITOR_INFO mi, KHE_COST_FUNCTION cost_fn,
  KHE_COST combined_weight, KHE_DRS_RESOURCE dr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  return KheDrsExprIntSumCostMake(mi, cost_fn, combined_weight, 0, 0, false,
    0, 0, 0, dr, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSumCostAddChild(KHE_DRS_EXPR_INT_SUM_COST eisc,        */
/*    KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  New child child_e has just been added to eisc.  Update eisc's state      */
/*  appropriately.  This is similar to closing a child, but not identical.   */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntSumCostAddChild(KHE_DRS_EXPR_INT_SUM_COST eisc,
  KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  eisc->closed_state += child_e->u.int_val;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsIntSumCostAdjustLimits(KHE_DRS_EXPR_INT_SUM_COST eisc)        */
/*                                                                           */
/*  Adjust the limits of eisc.  To be called after all children are added.   */
/*                                                                           */
/*****************************************************************************/

static void KheDrsIntSumCostAdjustLimits(KHE_DRS_EXPR_INT_SUM_COST eisc)
{
  int max_max_limit;

  /* adjust the lower limit if required */
  if( eisc->min_limit < eisc->history )
    eisc->min_limit = eisc->history;

  /* adjust eisc->max_limit if required */
  max_max_limit = HaArrayCount(eisc->children) + eisc->history +
    eisc->history_after;
  if( eisc->max_limit > max_max_limit )
    eisc->max_limit = max_max_limit;

  /* make sure limits are consistent */
  HnAssert(eisc->min_limit <= eisc->max_limit, "KheDrsIntSumCostAdjustLimits: "
    "after adjustment, inconsistent limits (min %d, max %d) for monitor %s",
    eisc->min_limit, eisc->max_limit, KheDrsMonitorInfoId(eisc->monitor_info));
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSumCostChildHasOpened(KHE_DRS_EXPR_INT_SUM_COST eisc,  */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of opening eisc for solving, eisc's child_index'th child,        */
/*  child_e, has been opened.  Update eisc to reflect this, and also         */
/*  drs->solve_start_cost.                                                   */
/*                                                                           */
/*  This function is called after child_e is added to eisc->open_children.   */
/*  The closed state of eisc is the sum of the values of eisc's closed       */
/*  children, so this update subtracts child_e's value (which preserves      */
/*  its value before opening) from the closed state.                         */
/*                                                                           */
/*  Obsolete:                                                                */
/*  So when KheDrsExprIntSumCostUpdateStartCost retrieves the number of      */
/*  open children, it gets the correct (that is, the changed) value.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntSumCostChildHasOpened(KHE_DRS_EXPR_INT_SUM_COST eisc,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int old_dev, new_dev;  KHE_COST old_cost, new_cost;
  if( DEBUG58(eisc->monitor_info) )
  {
    fprintf(stderr, "[ KheDrsExprIntSumCostChildHasOpened(%s, child_e, %d):\n",
      KheDrsMonitorInfoId(eisc->monitor_info), child_index);
    KheDrsExprDebug(child_e, drs, 2, 2, stderr);
  }
  old_dev = KheDrsExprIntSumCostDev(eisc, eisc->history + eisc->closed_state,
    KheDrsOpenChildrenCount(&eisc->open_children_by_day) - 1 +
    eisc->history_after);
  eisc->closed_state -= child_e->u.int_val;
  new_dev = KheDrsExprIntSumCostDev(eisc, eisc->history + eisc->closed_state,
     KheDrsOpenChildrenCount(&eisc->open_children_by_day) +eisc->history_after);
  if( old_dev != new_dev )
  {
    old_cost = f(eisc, old_dev);
    new_cost = f(eisc, new_dev);
    drs->solve_start_cost += new_cost - old_cost;
    KheDrsMonitorInfoUpdateRerunCost(eisc->monitor_info, drs, NULL,
      KHE_DRS_OPEN, "open", child_index, "+-", new_cost, old_cost);
  }

  /* update open_children_by_shift, if required */
  if( eisc->resource == NULL )
    KheDrsOpenChildrenAddChildInShiftOrder(&eisc->open_children_by_shift,
      child_e);

  /* all done */
  if( DEBUG58(eisc->monitor_info) )
    fprintf(stderr, "] KheDrsExprIntSumCostChildHasOpened\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSumCostChildHasClosed(KHE_DRS_EXPR_INT_SUM_COST eisc,  */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of closing eisc, eisc's child_index'th child, child_e, has       */
/*  been closed.  Update eisc's closed state to reflect this, and also       */
/*  drs->solve_start_cost.                                                   */
/*                                                                           */
/*  Although KheDrsExprChildHasOpened adds the newly opened child to         */
/*  e->open_children, KheDrsExprChildHasClosed does not remove it.  This     */
/*  is because the removal cannot be done efficiently while preserving       */
/*  the order of the open children.  However that does not matter here,      */
/*  and we need the correct number of open children, so we remove any one.   */
/*                                                                           */
/*  The closed state of eisc is the sum of the values of eisc's closed       */
/*  children, so this update adds child_e's value to the closed state.       */
/*                                                                           */
/*  Obsolete:                                                                */
/*  This is called after child_e is removed from eisc->open_children.        */
/*  So when KheDrsExprIntSumCostUpdateStartCost retrieves the number of      */
/*  open children, it gets the correct (that is, the changed) value.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntSumCostChildHasClosed(KHE_DRS_EXPR_INT_SUM_COST eisc,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int old_dev, new_dev;  KHE_COST old_cost, new_cost;
  old_dev = KheDrsExprIntSumCostDev(eisc, eisc->history + eisc->closed_state,
    KheDrsOpenChildrenCount(&eisc->open_children_by_day) + eisc->history_after);
  KheDrsOpenChildrenDeleteLast(&eisc->open_children_by_day);
  eisc->closed_state += child_e->u.int_val;
  new_dev = KheDrsExprIntSumCostDev(eisc, eisc->history + eisc->closed_state,
    KheDrsOpenChildrenCount(&eisc->open_children_by_day) + eisc->history_after);
  if( old_dev != new_dev )
  {
    new_cost = f(eisc, new_dev);
    old_cost = f(eisc, old_dev);
    drs->solve_start_cost += new_cost - old_cost;
    KheDrsMonitorInfoUpdateRerunCost(eisc->monitor_info, drs, NULL,
      KHE_DRS_CLOSE, "close", child_index, "+-", new_cost, old_cost);
  }
  if( eisc->resource == NULL )
    KheDrsOpenChildrenDeleteLast(&eisc->open_children_by_shift);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSumCostSetClosedValue(KHE_DRS_EXPR_INT_SUM_COST eisc,  */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing eisc, set its value suitably for its closed state.    */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntSumCostSetClosedValue(KHE_DRS_EXPR_INT_SUM_COST eisc,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  /* nothing to do here, since eisc does not store a closed value */
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprIntSumCostChildDomTest(                       */
/*    KHE_DRS_EXPR_INT_SUM_COST eisc, KHE_DYNAMIC_RESOURCE_SOLVER drs)       */
/*                                                                           */
/*  Return the dom test for a child of eisc on its open_day_index'th         */
/*  open day.                                                                */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DOM_TEST KheDrsExprIntSumCostChildDomTest(
  KHE_DRS_EXPR_INT_SUM_COST eisc, int open_day_index, KHE_DRS_EXPR child_e,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int a, b, unassigned_child_count;  KHE_DRS_DIM4_TABLE corr_dom_table4;

  /* calculate a */
  a = (eisc->max_limit < INT_MAX ? -1 : INT_MAX);

  /* calculate b */
  b = (eisc->min_limit > 0 ? INT_MAX : 0);

  /* make and return the dom test */
  if( child_e->tag == KHE_DRS_EXPR_OR_TAG )
  {
    unassigned_child_count = KheDrsOpenChildrenAtOrAfter(
      &eisc->open_children_by_day, open_day_index + 1);
    if( DEBUG57 )
    {
      fprintf(stderr, "[ calling KheDrsDim5TableGet(%p, %d), open_day_index "
	"%d, eisc:\n", (void *) eisc->corr_dom_table5, unassigned_child_count,
	open_day_index);
      KheDrsExprDebug((KHE_DRS_EXPR) eisc, drs, 2, 2, stderr);
      KheDrsOpenChildrenDebug(&eisc->open_children_by_day, 2, 2, stderr);
      fprintf(stderr, "]\n");
    }
    corr_dom_table4 = KheDrsDim5TableGet(eisc->corr_dom_table5,
      unassigned_child_count);
  }
  else
    corr_dom_table4 = NULL;
  return KheDrsDomTestMake(KHE_DRS_DOM_TEST_STRONG, child_e, false, 0,
    INT_MAX, a, b, 0, NULL, corr_dom_table4, NULL, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsIntSumCostExprRequiresShiftDomTest(                           */
/*    KHE_DRS_EXPR_INT_SUM_COST eisc, int si,                                */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, int *val)                             */
/*                                                                           */
/*  Find the number of unassigned children of eisc there will be when the    */
/*  days before ds (the shift with open shift index si) begins are all       */
/*  assigned, and ds itself is assigned.  Return this number in *val, and    */
/*  return true if *val > 0.  This will mean that shift assignments for ds   */
/*  need a dom test, because assigning ds does not completely assign eisc.   */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsIntSumCostExprRequiresShiftDomTest(
  KHE_DRS_EXPR_INT_SUM_COST eisc, int si,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int *val)
{
  int di, assigned;  KHE_DRS_SHIFT ds;
  ds = HaArray(drs->open_shifts, si);
  di = ds->encl_day->open_day_index;
  assigned = KheDrsOpenChildrenBefore(&eisc->open_children_by_day, di) +
    KheDrsOpenChildrenWithIndex(&eisc->open_children_by_shift, si);
  *val = KheDrsOpenChildrenCount(&eisc->open_children_by_day) - assigned;
  return *val > 0;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST_TYPE KheDomTestTypeActual(KHE_DRS_DOM_TEST_TYPE dt,     */
/*    bool tradeoff_allowed)                                                 */
/*                                                                           */
/*  Return the dom test type to actually use, given than the requested       */
/*  type is dt and tradoff_allowed says whether tradeoff dominance is        */
/*  allowed.                                                                 */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DOM_TEST_TYPE KheDomTestTypeActual(KHE_DRS_DOM_TEST_TYPE dt,
  bool tradeoff_allowed)
{
  if( dt == KHE_DRS_DOM_TEST_TRADEOFF && !tradeoff_allowed )
    return KHE_DRS_DOM_TEST_STRONG;
  else
    return dt;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprIntSumCostGeneralDomTest(                     */
/*    KHE_DRS_EXPR_INT_SUM_COST eisc, int unassigned_child_count,            */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make and return a dominance test for eisc, assuming that there are       */
/*  unassigned_child_count unassigned children (not including history).      */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DOM_TEST KheDrsExprIntSumCostGeneralDomTest(
  KHE_DRS_EXPR_INT_SUM_COST eisc, int unassigned_child_count,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int a, b;  KHE_DRS_DOM_TEST_TYPE dt_type;
  KHE_DRS_DIM2_TABLE main_dom_table2;

  /* calculate a */
  if( eisc->max_limit < INT_MAX )
    a = eisc->max_limit - unassigned_child_count;
  else
    a = INT_MAX;

  /* calculate b */
  if( eisc->min_limit > 0 )
    b = eisc->min_limit;
  else
    b = 0;

  /* make and return the dom test */
  main_dom_table2 = KheDrsDim3TableGet(eisc->main_dom_table3,
    unassigned_child_count);
  dt_type = KheDomTestTypeActual(drs->solve_dom_test_type,
    eisc->cost_fn != KHE_QUADRATIC_COST_FUNCTION);
  return KheDrsDomTestMake(dt_type, (KHE_DRS_EXPR) eisc, eisc->allow_zero,
    eisc->min_limit, eisc->max_limit, a, b, eisc->combined_weight,
    main_dom_table2, NULL, eisc->monitor_info->monitor, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprIntSumCostDayDomTest(                         */
/*    KHE_DRS_EXPR_INT_SUM_COST eisc, int open_day_index,                    */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Return the dom test for eisc on the open day with index open_day_index.  */
/*                                                                           */
/*  When two solutions are compared using this dom test, the open children   */
/*  up to and including those on the day with index open_day_index will be   */
/*  be assigned, leaving                                                     */
/*                                                                           */
/*    KheDrsOpenChildrenAtOrAfter(eisc, open_day_index + 1)                  */
/*                                                                           */
/*  unassigned children.                                                     */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DOM_TEST KheDrsExprIntSumCostDayDomTest(
  KHE_DRS_EXPR_INT_SUM_COST eisc, int open_day_index,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int unassigned_child_count;
  unassigned_child_count = KheDrsOpenChildrenAtOrAfter(
    &eisc->open_children_by_day, open_day_index + 1);
  return KheDrsExprIntSumCostGeneralDomTest(eisc, unassigned_child_count, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsAdjustedSigVal(int val, KHE_DRS_ADJUST_TYPE adjust_type,       */
/*    int min_limit, int max_limit, int pc, int qc)                          */
/*                                                                           */
/*  Return the adjusted signature value.                                     */
/*                                                                           */
/*****************************************************************************/

static int KheDrsAdjustedSigVal(int val, KHE_DRS_ADJUST_TYPE adjust_type,
  int min_limit, int max_limit, int history_after)
{
  int lbar;
  switch( adjust_type )
  {
    case KHE_DRS_ADJUST_ORDINARY:

      return val;

    case KHE_DRS_ADJUST_NO_MAX:

      lbar = max(0, min_limit - history_after);
      return min(lbar, val);

    case KHE_DRS_ADJUST_LINEAR:

      lbar = max_limit;
      return min(lbar, val);

    case KHE_DRS_ADJUST_STEP:

      lbar = max_limit + 1;
      return min(lbar, val);

    default:

      HnAbort("KheDrsAdjustedSigVal internal error (adjusttype)");
      return val;  /* keep compiler happy */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprIntSumCostInitialValue(KHE_DRS_EXPR_INT_SUM_COST eisc)     */
/*                                                                           */
/*  Return the initial value of eisc.  This is used just below, but it is    */
/*  also used for dominance testing of correlated expressions.               */
/*                                                                           */
/*****************************************************************************/

static int KheDrsExprIntSumCostInitialValue(KHE_DRS_EXPR_INT_SUM_COST eisc)
{
  return eisc->history + eisc->closed_state;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSumCostEvalSignature(KHE_DRS_EXPR_INT_SUM_COST eisc,   */
/*    KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, int next_di,           */
/*    KHE_DRS_SIGNATURE next_sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)           */
/*                                                                           */
/*  Evaluate expression eisc, assuming that next_di has just been assigned,  */
/*  and update the cost and states of sig with the result.                   */
/*                                                                           */
/*  Terminology used here        Terminology used in doc                     */
/*  --------------------------------------------------------                 */
/*  ld1                          l(m, S)                                     */
/*  ud1                          u(m, S)                                     */
/*  dev1                         delta(l(m, S), u(m, S)                      */
/*  cost1                        f(delta(l(m, S), u(m, S))                   */
/*  --------------------------------------------------------                 */
/*  ld2                          l(m, S')                                    */
/*  ud2                          u(m, S')                                    */
/*  dev2                         delta(l(m, S'), u(m, S')                    */
/*  cost2                        f(delta(l(m, S'), u(m, S'))                 */
/*  --------------------------------------------------------                 */
/*                                                                           */
/*****************************************************************************/
static void KheDrsSolnSignatureSetDebug(KHE_DRS_SOLN soln, FILE *fp);

static void KheDrsExprIntSumCostEvalSignature(KHE_DRS_EXPR_INT_SUM_COST eisc,
  KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, int next_di,
  KHE_DRS_SIGNATURE next_sig, KHE_DYNAMIC_RESOURCE_SOLVER drs, bool debug)
{
  int ld1, ld2, ud1, ud2, i, i1, i2, dev1, dev2, adjusted_ld2, si;
  KHE_DRS_EXPR child_e;

  /* get ld1, ud1, and dev1 (for all days before next_di) */
  if( debug )
  {
    fprintf(stderr, "  [ eval %s, next_di %d, open_children_by_day:\n",
      KheDrsMonitorInfoId(eisc->monitor_info), next_di);
    KheDrsOpenChildrenDebug(&eisc->open_children_by_day, 1, 4, stderr);
  }
  if( KheDrsOpenChildrenIndexIsFirstOrLess(&eisc->open_children_by_day,next_di))
  {
    ld1 = KheDrsExprIntSumCostInitialValue(eisc);
    if( debug )
      fprintf(stderr, "    ld1 %d + %d", eisc->history, eisc->closed_state);
  }
  else
  {
    ld1 = KheDrsExprDaySigVal((KHE_DRS_EXPR) eisc, next_di - 1, prev_sig);
    if( debug )
      fprintf(stderr, "    ld1 %d", ld1);
  }
  ud1 = ld1 + eisc->history_after +
    KheDrsOpenChildrenAtOrAfter(&eisc->open_children_by_day, next_di);
  dev1 = KheDrsExprIntSumCostDelta(eisc, ld1, ud1);
  if( debug )
    fprintf(stderr, " ud1 %d + %d + %d dev1 %d\n", ld1, eisc->history_after,
      KheDrsOpenChildrenAtOrAfter(&eisc->open_children_by_day, next_di), dev1);

  /* get ld2, ud2, and dev2 for one more shift or one more day */
  ld2 = ld1;
  ud2 = ud1;
  if( dsg->type == KHE_DRS_SIGNER_SHIFT )
  {
    si = dsg->u.shift->open_shift_index;
    if( debug )
    {
      fprintf(stderr, "    open_children_by_shift:\n");
      KheDrsOpenChildrenDebug(&eisc->open_children_by_shift, 1, 4, stderr);
      fprintf(stderr, "    si %d ", si);
    }
    KheDrsOpenChildrenForEach(&eisc->open_children_by_shift, si, child_e, i)
      if( child_e->u.int_val == 1 )
      {
	ld2 += 1;  /* unassigned to active */
	if( debug )
	  fprintf(stderr, "%d+", i);
      }
      else
      {
	ud2 -= 1;  /* unassigned to inactive */
	if( debug )
	  fprintf(stderr, "%d-", i);
      }
  }
  else
  {
    if( debug )
      fprintf(stderr, " next_di %d ", next_di);
    KheDrsOpenChildrenForEach(&eisc->open_children_by_day, next_di, child_e, i)
      if( child_e->u.int_val == 1 )
      {
	ld2 += 1;  /* unassigned to active */
	if( debug )
	  fprintf(stderr, "+");
      }
      else
      {
	ud2 -= 1;  /* unassigned to inactive */
	if( debug )
	  fprintf(stderr, "-");
      }
  }
  dev2 = KheDrsExprIntSumCostDelta(eisc, ld2, ud2);
  if( debug )
    fprintf(stderr, " ld2 %d ud2 %d dev2 %d", ld2, ud2, dev2);

  /* if more to do, store ld2 (adjusted) in next_sig */
  if( ud2 - ld2 > eisc->history_after )
  {
    adjusted_ld2 = KheDrsAdjustedSigVal(ld2, eisc->adjust_type,
      eisc->min_limit, eisc->max_limit, eisc->history_after);
    KheDrsSignatureAddState(next_sig, adjusted_ld2, dsg, (KHE_DRS_EXPR) eisc);
    if( debug )
      fprintf(stderr, " S%d", adjusted_ld2);
  }

  /* report the extra cost, if any */
  if( dev2 != dev1 )
  {
    KheDrsSignatureAddCost(next_sig, f(eisc, dev2) - f(eisc, dev1));
    KheDrsMonitorInfoUpdateRerunCost(eisc->monitor_info, drs, dsg,
      KHE_DRS_SEARCH, "search", -1, "+-", f(eisc, dev2), f(eisc, dev1));
    if( debug )
      fprintf(stderr, " C%.5f", KheCostShow(f(eisc, dev2) - f(eisc, dev1)));
  }
  if( debug )
    fprintf(stderr, "\n  ]\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSumCostDoDebug(KHE_DRS_EXPR_INT_SUM_COST eisc,         */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of eisc which is specific to it.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntSumCostDoDebug(KHE_DRS_EXPR_INT_SUM_COST eisc,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "INT_SUM_COST(%s, %s, %d)",
    KheDrsMonitorInfoId(eisc->monitor_info),
    KheDrsAdjustTypeShow(eisc->adjust_type), eisc->closed_state);
  KheDrsExprDebugSignature((KHE_DRS_EXPR) eisc, soln, false, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_CLOSED_SEQ"                                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_CLOSED_SEQ KheDrsClosedSeqMake(KHE_DYNAMIC_RESOURCE_SOLVER drs,  */
/*    int start_index, int stop_index)                                       */
/*                                                                           */
/*  Make a new KHE_DRS_CLOSED_SEQ object with these attributes.              */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_CLOSED_SEQ KheDrsClosedSeqMake(KHE_DYNAMIC_RESOURCE_SOLVER drs,
  int start_index, int stop_index)
{
  KHE_DRS_CLOSED_SEQ res;
  if( HaArrayCount(drs->closed_seq_free_list) > 0 )
    res = HaArrayLastAndDelete(drs->closed_seq_free_list);
  else
    HaMake(res, drs->arena);
  res->start_index = start_index;
  res->stop_index = stop_index;
  res->active_at_left = 0;		/* placeholder (unless empty) */
  res->active_at_right = 0;		/* placeholder (unless empty) */
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_CLOSED_SEQ KheDrsClosedSeqMakeInit(                              */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make an initial empty closed sequence.                                   */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_CLOSED_SEQ KheDrsClosedSeqMakeInit(
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  return KheDrsClosedSeqMake(drs, 0, 0);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsClosedSeqDelete(KHE_DRS_CLOSED_SEQ dcs,                       */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Delete dcs.                                                              */
/*                                                                           */
/*****************************************************************************/

static void KheDrsClosedSeqDelete(KHE_DRS_CLOSED_SEQ dcs,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  HaArrayAddLast(drs->closed_seq_free_list, dcs);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsClosedSeqLength(KHE_DRS_CLOSED_SEQ dcs)                        */
/*                                                                           */
/*  Return the length of dcs.                                                */
/*                                                                           */
/*****************************************************************************/

static int KheDrsClosedSeqLength(KHE_DRS_CLOSED_SEQ dcs)
{
  return dcs->stop_index - dcs->start_index;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsClosedSeqAllActive(KHE_DRS_CLOSED_SEQ dcs)                    */
/*                                                                           */
/*  Return true if every child of dcs is active.                             */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsClosedSeqAllActive(KHE_DRS_CLOSED_SEQ dcs)
{
  return KheDrsClosedSeqLength(dcs) == dcs->active_at_left;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsClosedSeqExtendToRight(KHE_DRS_CLOSED_SEQ dcs, int val)       */
/*                                                                           */
/*  Extend dcs one place to the right, with a child with value val.          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsClosedSeqExtendToRight(KHE_DRS_CLOSED_SEQ dcs, int val)
{
  if( val == 0 )
    dcs->active_at_right = 0;
  else
  {
    if( KheDrsClosedSeqAllActive(dcs) )
      dcs->active_at_left++;
    dcs->active_at_right++;
  }
  dcs->stop_index++;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsClosedSeqSetActiveLengths(KHE_DRS_CLOSED_SEQ dcs,             */
/*    KHE_DRS_EXPR_INT_SEQ_COST eisc)                                        */
/*                                                                           */
/*  Assuming dcs->start_index and dcs->stop_index are up to date, bring      */
/*  dcs->active_at_left and dcs-> active_at_right up to date.                */
/*                                                                           */
/*****************************************************************************/

static void KheDrsClosedSeqSetActiveLengths(KHE_DRS_CLOSED_SEQ dcs,
  KHE_DRS_EXPR_INT_SEQ_COST eisc)
{
  KHE_DRS_EXPR child_e;  int i;

  /* bring dcs->active_at_left up to date */
  dcs->active_at_left = 0;
  for( i = dcs->start_index;  i < dcs->stop_index;  i++ )
  {
    child_e = HaArray(eisc->children, i);
    if( child_e->u.int_val != 1 )
      break;
    dcs->active_at_left++;
  }

  /* bring dcs->active_at_right up to date */
  dcs->active_at_right = 0;
  for( i = dcs->stop_index - 1;  i >= dcs->start_index;  i-- )
  {
    child_e = HaArray(eisc->children, i);
    if( child_e->u.int_val != 1 )
      break;
    dcs->active_at_right++;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_CLOSED_SEQ KheDrsClosedSeqSplit(KHE_DRS_CLOSED_SEQ dcs,          */
/*    int child_index)                                                       */
/*                                                                           */
/*  Split dcs at child_index.  The child at child_index will be in           */
/*  neither of the two fragments.                                            */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_CLOSED_SEQ KheDrsClosedSeqSplit(KHE_DRS_CLOSED_SEQ dcs,
  int child_index, KHE_DRS_EXPR_INT_SEQ_COST eisc,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_CLOSED_SEQ res;

  /* make the new fragment and reset the stop index of the old one */
  HnAssert(dcs->start_index <= child_index,
    "KheDrsClosedSeqSplit internal error 1");
  HnAssert(child_index < dcs->stop_index,
    "KheDrsClosedSeqSplit internal error 2 (child_index %d, dcs->stop_index %d",
    child_index, dcs->stop_index);
  res = KheDrsClosedSeqMake(drs, child_index + 1, dcs->stop_index);
  dcs->stop_index = child_index;

  /* update the active lengths of the two fragments, and return the new one */
  KheDrsClosedSeqSetActiveLengths(dcs, eisc);
  KheDrsClosedSeqSetActiveLengths(res, eisc);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsClosedSeqMerge(KHE_DRS_CLOSED_SEQ dcs1,                       */
/*    KHE_DRS_CLOSED_SEQ dcs2, KHE_DRS_EXPR_INT_SEQ_COST eisc,               */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Merge adjacent closed sequences dcs1 and dcs2.  The formerly open        */
/*  child separating them has closed, and its closed value has been set.     */
/*                                                                           */
/*  Place the merged closed sequence in dcs1, and delete dcs2.               */
/*                                                                           */
/*****************************************************************************/

static void KheDrsClosedSeqMerge(KHE_DRS_CLOSED_SEQ dcs1,
  KHE_DRS_CLOSED_SEQ dcs2, KHE_DRS_EXPR_INT_SEQ_COST eisc,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  /* update dcs1 to hold the merged sequence */
  HnAssert(dcs1->stop_index + 1 == dcs2->start_index,
    "KheDrsClosedSeqMerge internal error 1");
  dcs1->stop_index = dcs2->stop_index;
  KheDrsClosedSeqSetActiveLengths(dcs1, eisc);

  /* delete dcs2 */
  KheDrsClosedSeqDelete(dcs2, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsClosedSeqDebug(KHE_DRS_CLOSED_SEQ dcs, int verbosity,         */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of dcs onto fp with the given verbosity and indent.          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsClosedSeqDebug(KHE_DRS_CLOSED_SEQ dcs, int verbosity,
  int indent, FILE *fp)
{
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  fprintf(fp, "[CLOSED_SEQ %d-%d, active_at_left %d, active_at_right %d]",
    dcs->start_index, dcs->stop_index, dcs->active_at_left,
    dcs->active_at_right);
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_A_INTERVAL"                                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_A_INTERVAL KheDrsAIntervalMake(int start_index, int stop_index,  */
/*    bool unassigned_precedes)                                              */
/*                                                                           */
/*  Make and return a new a-interval with these attributes.                  */
/*                                                                           */
/*****************************************************************************/

KHE_DRS_A_INTERVAL KheDrsAIntervalMake(int start_index, int stop_index,
  bool unassigned_precedes)
{
  KHE_DRS_A_INTERVAL res;
  res.start_index = start_index;
  res.stop_index = stop_index;
  res.unassigned_precedes = unassigned_precedes;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_A_INTERVAL KheDrsAIntervalMakeLeft(                              */
/*    KHE_DRS_EXPR_INT_SEQ_COST eisc, int open_index, int active_len)        */
/*                                                                           */
/*  Make an a-interval of length active_len which ends just before           */
/*  the open child with index open_index.                                    */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_A_INTERVAL KheDrsAIntervalMakeLeft(
  KHE_DRS_EXPR_INT_SEQ_COST eisc, int open_index, int active_len)
{
  KHE_DRS_CLOSED_SEQ dcs;
  dcs = HaArray(eisc->closed_seqs, open_index);
  return KheDrsAIntervalMake(dcs->stop_index - active_len,
    dcs->stop_index, false);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_A_INTERVAL KheDrsAIntervalFindLeft(                              */
/*    KHE_DRS_EXPR_INT_SEQ_COST eisc, int open_index)                        */
/*                                                                           */
/*  Find the (possibly empty) a-interval immediately to the left of the      */
/*  child with the given open_index.  Include history if necessary.          */
/*                                                                           */
/*  Implementation note.  This interval consists entirely of the last        */
/*  active_at_right children of the preceding KHE_DRS_CLOSED_SEQ             */
/*  (possibly plus history), even if their number is zero; because they      */
/*  are preceded by another open child, or an inactive child, or nothing.    */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_A_INTERVAL KheDrsAIntervalFindLeft(
  KHE_DRS_EXPR_INT_SEQ_COST eisc, int open_index)
{
  KHE_DRS_CLOSED_SEQ dcs;
  dcs = HaArray(eisc->closed_seqs, open_index);
  if( !KheDrsClosedSeqAllActive(dcs) )
  {
    /* an inactive child precedes the active_at_right active children */
    return KheDrsAIntervalMake(dcs->stop_index - dcs->active_at_right,
      dcs->stop_index, false);
  }
  else if( open_index > 0 )
  {
    /* an unassigned child precedes the active_at_right active children */
    return KheDrsAIntervalMake(dcs->stop_index - dcs->active_at_right,
      dcs->stop_index, true);
  }
  else
  {
    /* nothing but history precedes the active_at_right active children */
    return KheDrsAIntervalMake(dcs->stop_index - dcs->active_at_right
      - eisc->history, dcs->stop_index, false);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_A_INTERVAL KheDrsAIntervalFindRight(                             */
/*    KHE_DRS_EXPR_INT_SEQ_COST eisc, int open_index, bool child_unassigned) */
/*                                                                           */
/*  Find the (possibly empty) a-interval immediately to the right of the     */
/*  child with the given open_index.  That child is assumed to be unassigned */
/*  when child_unassigned is true.                                           */
/*                                                                           */
/*  Implementation note.  This interval consists entirely of the first       */
/*  active_at_left children of the following KHE_DRS_CLOSED_SEQ,             */
/*  even if their number is zero; because they are followed by another       */
/*  open child, or an inactive child, or history_after, or nothing.          */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_A_INTERVAL KheDrsAIntervalFindRight(
  KHE_DRS_EXPR_INT_SEQ_COST eisc, int open_index, bool child_unassigned)
{
  KHE_DRS_CLOSED_SEQ dcs;
  dcs = HaArray(eisc->closed_seqs, open_index + 1);
  return KheDrsAIntervalMake(dcs->start_index,
    dcs->start_index + dcs->active_at_left, child_unassigned);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_A_INTERVAL KheDrsAIntervalMerge(KHE_DRS_A_INTERVAL ai_left,      */
/*    KHE_DRS_A_INTERVAL ai_right)                                           */
/*                                                                           */
/*  Merge ai_left with ai_right.  These intervals are separated by one       */
/*  child (this is checked).  That child is assumed to be active, but that   */
/*  is not checked, and indeed may not be true - we might be building this   */
/*  interval to reflect the old state of that child.                         */
/*                                                                           */
/*  It is safe to set the unassigned_precedes value of the result interval   */
/*  to ai_left.unassigned_precedes.  Even when ai_left is empty, its place   */
/*  is well defined, and ai_left.unassigned_precedes is correct for there.   */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_A_INTERVAL KheDrsAIntervalMerge(KHE_DRS_A_INTERVAL ai_left,
  KHE_DRS_A_INTERVAL ai_right)
{
  HnAssert(ai_left.stop_index + 1 == ai_right.start_index,
    "KheDrsAIntervalMerge internal error");
  return KheDrsAIntervalMake(ai_left.start_index, ai_right.stop_index,
    ai_left.unassigned_precedes);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsAIntervalDev(KHE_DRS_A_INTERVAL ai,                            */
/*    KHE_DRS_EXPR_INT_SEQ_COST eisc)                                        */
/*                                                                           */
/*  Return the deviation of a-interval ai.  If ai.unassigned_precedes is     */
/*  true, there is an unassigned child immediately before ai.first_index,    */
/*  so return 0 if the cost function is KHE_STEP_COST_FUNCTION.              */
/*                                                                           */
/*****************************************************************************/

static int KheDrsAIntervalDev(KHE_DRS_A_INTERVAL ai,
  KHE_DRS_EXPR_INT_SEQ_COST eisc)
{
  int len;

  /* return 0 if an unassigned child immediately precedes this interval */
  /* and the cost function is step                                      */
  if( ai.unassigned_precedes && eisc->cost_fn == KHE_STEP_COST_FUNCTION )
    return 0;

  /* calculate the length and deviation */
  len = ai.stop_index - ai.start_index;
  return len > eisc->max_limit ? len - eisc->max_limit : 0;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_COST KheDrsAIntervalCost(KHE_DRS_A_INTERVAL ai,                      */
/*    KHE_DRS_EXPR_INT_SEQ_COST eisc)                                        */
/*                                                                           */
/*  Return the cost of ai.                                                   */
/*                                                                           */
/*****************************************************************************/

static KHE_COST KheDrsAIntervalCost(KHE_DRS_A_INTERVAL ai,
  KHE_DRS_EXPR_INT_SEQ_COST eisc)
{
  int dev;
  dev = KheDrsAIntervalDev(ai, eisc);
  return dev == 0 ? 0 : f(eisc, dev);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_AU_INTERVAL"                                          */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_AU_INTERVAL KheDrsAUIntervalMake(int start_index,                */
/*    int stop_index, bool has_active_child)                                 */
/*                                                                           */
/*  Make and return an au-interval with these attributes.  However, the      */
/*  has_active_child attribute is ignored when the interval is empty:  an    */
/*  empty interval cannot contain an active child.                           */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_AU_INTERVAL KheDrsAUIntervalMake(int start_index,
  int stop_index, bool has_active_child)
{
  KHE_DRS_AU_INTERVAL res;
  res.start_index = start_index;
  res.stop_index = stop_index;
  res.has_active_child = (start_index < stop_index && has_active_child);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsAUIntervalLength(KHE_DRS_AU_INTERVAL aui)                      */
/*                                                                           */
/*  Return the length of aui.                                                */
/*                                                                           */
/*****************************************************************************/

static int KheDrsAUIntervalLength(KHE_DRS_AU_INTERVAL aui)
{
  return aui.stop_index - aui.start_index;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsAUIntervalExtendToLeft(KHE_DRS_AU_INTERVAL *aui,              */
/*    int extra_len, bool has_active_child)                                  */
/*                                                                           */
/*  Extend *aui extra_len places to the left.  If extra_len > 0,             */
/*  has_active_child says whether the extra elements are active or not.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsAUIntervalExtendToLeft(KHE_DRS_AU_INTERVAL *aui,
  int extra_len, bool has_active_child)
{
  if( extra_len > 0 )
  {
    aui->start_index -= extra_len;
    if( has_active_child )
      aui->has_active_child = true;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsAUIntervalExtendToRight(KHE_DRS_AU_INTERVAL *aui,             */
/*    int extra_len, bool has_active_child)                                  */
/*                                                                           */
/*  Extend *aui extra_len places to the right.  If extra_len > 0,            */
/*  has_active_child says whether the extra elements are active or not.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsAUIntervalExtendToRight(KHE_DRS_AU_INTERVAL *aui,
  int extra_len, bool has_active_child)
{
  if( extra_len > 0 )
  {
    aui->stop_index += extra_len;
    if( has_active_child )
      aui->has_active_child = true;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_AU_INTERVAL KheDrsAUIntervalMakeLeft(                            */
/*    KHE_DRS_EXPR_INT_SEQ_COST eisc, int open_index, int active_len)        */
/*                                                                           */
/*  Make an au-interval of length active_len which ends just before          */
/*  the open child with index open_index.                                    */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_AU_INTERVAL KheDrsAUIntervalMakeLeft(
  KHE_DRS_EXPR_INT_SEQ_COST eisc, int open_index, int active_len)
{
  KHE_DRS_CLOSED_SEQ dcs;
  dcs = HaArray(eisc->closed_seqs, open_index);
  return KheDrsAUIntervalMake(dcs->stop_index - active_len,
    dcs->stop_index, true);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_AU_INTERVAL KheDrsAUIntervalFindLeft(                            */
/*    KHE_DRS_EXPR_INT_SEQ_COST eisc, int open_index)                        */
/*                                                                           */
/*  Find the au-interval ending just before the child whose open index       */
/*  is open_index.                                                           */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_AU_INTERVAL KheDrsAUIntervalFindLeft(
  KHE_DRS_EXPR_INT_SEQ_COST eisc, int open_index)
{
  KHE_DRS_CLOSED_SEQ dcs;  KHE_DRS_AU_INTERVAL res;  int i;

  /* initialize res to the interval of active children at the right of dcs */
  dcs = HaArray(eisc->closed_seqs, open_index);
  res = KheDrsAUIntervalMake(dcs->stop_index - dcs->active_at_right,
    dcs->stop_index, true);
  if( !KheDrsClosedSeqAllActive(dcs) )
    return res;

  /* now keep looking to the left of there */
  for( i = open_index - 1;  i >= 0;  i-- )
  {
    /* return early if eisc->min_limit reached */
    if( KheDrsAUIntervalLength(res) >= eisc->min_limit )
      return res;

    /* res includes the open unassigned child before the previous dcs */
    KheDrsAUIntervalExtendToLeft(&res, 1, false);

    /* res includes the active children at the right end of the next dcs */
    dcs = HaArray(eisc->closed_seqs, i);
    KheDrsAUIntervalExtendToLeft(&res, dcs->active_at_right, true);
    if( !KheDrsClosedSeqAllActive(dcs) )
      return res;
  }

  /* at the start, so res includes history */
  KheDrsAUIntervalExtendToLeft(&res, eisc->history, true);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_AU_INTERVAL KheDrsAUIntervalFindRight(                           */
/*    KHE_DRS_EXPR_INT_SEQ_COST eisc, int open_index)                        */
/*                                                                           */
/*  Find the au-interval beginning just after the child whose open index     */
/*  is open_index.                                                           */
/*                                                                           */
/*****************************************************************************/
static char *DrsAUIntervalShow(KHE_DRS_AU_INTERVAL aui);

static KHE_DRS_AU_INTERVAL KheDrsAUIntervalFindRight(
  KHE_DRS_EXPR_INT_SEQ_COST eisc, int open_index,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_CLOSED_SEQ dcs;  KHE_DRS_AU_INTERVAL res;  int i;

  /* initialize res to the interval of active children at the left of dcs */
  if( DEBUG20 && drs->rerun != NULL &&
      strcmp(KheDrsMonitorInfoId(eisc->monitor_info), RERUN_MONITOR_ID) == 0 )
  {
    fprintf(stderr, "[ KheDrsAUIntervalFindRight(%s, %d, drs)\n",
      KheDrsMonitorInfoId(eisc->monitor_info), open_index);
    for( i = open_index + 1;  i < HaArrayCount(eisc->closed_seqs);  i++ )
    {
      dcs = HaArray(eisc->closed_seqs, i);
      fprintf(stderr, "  dcs %d ", i);
      KheDrsClosedSeqDebug(dcs, 2, 0, stderr);
    }
  }
  dcs = HaArray(eisc->closed_seqs, open_index + 1);
  res = KheDrsAUIntervalMake(dcs->start_index,
    dcs->start_index + dcs->active_at_left, true);
  if( !KheDrsClosedSeqAllActive(dcs) )
  {
    if( DEBUG20 && drs->rerun != NULL &&
	strcmp(KheDrsMonitorInfoId(eisc->monitor_info), RERUN_MONITOR_ID) == 0 )
      fprintf(stderr, "] KheDrsAUIntervalFindRight(eisc, %d) 1= %s\n",
	open_index, DrsAUIntervalShow(res));
    return res;
  }

  /* now keep looking to the right of there */
  for( i = open_index + 2;  i < HaArrayCount(eisc->closed_seqs);  i++ )
  {
    /* return early if eisc->min_limit reached */
    if( KheDrsAUIntervalLength(res) >= eisc->min_limit )
    {
      if( DEBUG20 && drs->rerun != NULL &&
	  strcmp(KheDrsMonitorInfoId(eisc->monitor_info), RERUN_MONITOR_ID)==0 )
	fprintf(stderr, "] KheDrsAUIntervalFindRight(eisc, %d) 2= %s\n",
	  open_index, DrsAUIntervalShow(res));
      return res;
    }

    /* res includes the open unassigned child after the previous dcs */
    KheDrsAUIntervalExtendToRight(&res, 1, false);

    /* res includes the active children at the left end of the next dcs */
    dcs = HaArray(eisc->closed_seqs, i);
    KheDrsAUIntervalExtendToRight(&res, dcs->active_at_left, true);
    if( !KheDrsClosedSeqAllActive(dcs) )
    {
      if( DEBUG20 && drs->rerun != NULL &&
	  strcmp(KheDrsMonitorInfoId(eisc->monitor_info), RERUN_MONITOR_ID)==0 )
	fprintf(stderr, "] KheDrsAUIntervalFindRight(eisc, %d) 3= %s\n",
	  open_index, DrsAUIntervalShow(res));
      return res;
    }
  }

  /* at the end, so res includes history_after */
  KheDrsAUIntervalExtendToRight(&res, eisc->history_after, false);
  if( DEBUG20 && drs->rerun != NULL &&
      strcmp(KheDrsMonitorInfoId(eisc->monitor_info), RERUN_MONITOR_ID) == 0 )
    fprintf(stderr, "] KheDrsAUIntervalFindRight(eisc, %d) 4= %s\n",
      open_index, DrsAUIntervalShow(res));
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsAUIntervalFindRightLengths(KHE_DRS_EXPR_INT_SEQ_COST eisc,    */
/*    int open_index, int *q, int *r)                                        */
/*                                                                           */
/*  Similar to KheDrsAUIntervalFindRight, only the interval includes the     */
/*  child whose open index is open_index, and instead of returning the       */
/*  interval, we return *q, the number of unassigned children in it, and     */
/*  *r, the number of active children in it.                                 */
/*                                                                           */
/*****************************************************************************/

static void KheDrsAUIntervalFindRightLengths(KHE_DRS_EXPR_INT_SEQ_COST eisc,
  int open_index, int *q, int *r)
{
  KHE_DRS_CLOSED_SEQ dcs;  int i;

  /* handle each unassigned child and its *following* closed interval */
  *q = *r = 0;
  for( i = open_index + 1;  i < HaArrayCount(eisc->closed_seqs);  i++ )
  {
    dcs = HaArray(eisc->closed_seqs, i);

    /* include the open (unassigned) child just before dcs */
    *q += 1;

    /* include the active children at the left end of dcs */
    *r += dcs->active_at_left;

    /* if there is an inactive child in dcs, it's time to stop */
    if( !KheDrsClosedSeqAllActive(dcs) )
      return;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_AU_INTERVAL KheDrsAUIntervalMerge(KHE_DRS_AU_INTERVAL aui_left,  */
/*    KHE_DRS_AU_INTERVAL aui_right, bool sep_is_active)                     */
/*                                                                           */
/*  Return the merge of aui_left and aui_right.  They are separated by one   */
/*  child, and that child is active if sep_is_active.                        */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_AU_INTERVAL KheDrsAUIntervalMerge(KHE_DRS_AU_INTERVAL aui_left,
  KHE_DRS_AU_INTERVAL aui_right, bool sep_is_active)
{
  HnAssert(aui_left.stop_index + 1 == aui_right.start_index,
    "KheDrsAUIntervalMerge internal error");
  return KheDrsAUIntervalMake(aui_left.start_index, aui_right.stop_index,
    aui_left.has_active_child || aui_right.has_active_child || sep_is_active);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsAUIntervalDev(KHE_DRS_AU_INTERVAL aui,                         */
/*    KHE_DRS_EXPR_INT_SEQ_COST eisc)                                        */
/*                                                                           */
/*  Return the deviation of au-interval aui.  If aui.has_active_child is     */
/*  false, the interval has no active child so return 0.  This should        */
/*  also handle the special case of len == 0, where the result is also 0.    */
/*                                                                           */
/*****************************************************************************/

static int KheDrsAUIntervalDev(KHE_DRS_AU_INTERVAL aui,
  KHE_DRS_EXPR_INT_SEQ_COST eisc)
{
  int len;

  /* return 0 if the interval has no active child (including len == 0) */
  if( !aui.has_active_child )
    return 0;

  /* calculate the length and deviation */
  len = aui.stop_index - aui.start_index;
  return len < eisc->min_limit ? eisc->min_limit - len : 0;
}


/*****************************************************************************/
/*                                                                           */
/*  char *DrsAUIntervalShow(KHE_DRS_AU_INTERVAL aui)                         */
/*                                                                           */
/*  Return a display of aui in static memory.                                */
/*                                                                           */
/*****************************************************************************/

static char *DrsAUIntervalShow(KHE_DRS_AU_INTERVAL aui)
{
  static char buff[20];
  snprintf(buff, 20, "[%d, %d%s]", aui.start_index, aui.stop_index,
    aui.has_active_child ? "a" : "");
  return buff;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_COST KheDrsAUIntervalCost(KHE_DRS_AU_INTERVAL aui,                   */
/*    KHE_DRS_EXPR_INT_SEQ_COST eisc)                                        */
/*                                                                           */
/*  Return the cost of aui.                                                  */
/*                                                                           */
/*****************************************************************************/

static KHE_COST KheDrsAUIntervalCost(KHE_DRS_AU_INTERVAL aui,
  KHE_DRS_EXPR_INT_SEQ_COST eisc)
{
  int dev;
  dev = KheDrsAUIntervalDev(aui, eisc);
  return dev == 0 ? 0 : f(eisc, dev);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_INT_SEQ_COST - dominance testing and cost"       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprIntSeqCostDelta(KHE_DRS_EXPR_INT_SEQ_COST eisc,            */
/*    int lower_det, int upper_det)                                          */
/*                                                                           */
/*  Return the deviation of eisc for the given lower and upper determinants. */
/*  This is function delta(l, u) from the documentation.                     */
/*                                                                           */
/*****************************************************************************/

static int KheDrsExprIntSeqCostDelta(KHE_DRS_EXPR_INT_SEQ_COST eisc,
  int z, int lower_det, int upper_det)
{
  if( z == 0 )
    return 0;
  else if( lower_det > eisc->max_limit )
    return lower_det - eisc->max_limit;
  else if( upper_det < eisc->min_limit )
    return eisc->min_limit - upper_det;
  else
    return 0;
}


/*****************************************************************************/
/*                                                                           */
/*  int ci_bar(KHE_DRS_EXPR_INT_SEQ_COST eisc, int p, int x)                 */
/*                                                                           */
/*  Return the value of ci overbar as defined in the documentation.          */
/*                                                                           */
/*****************************************************************************/

static int ci_bar(KHE_DRS_EXPR_INT_SEQ_COST eisc, int p, int x)
{
  return x < p ? 0 : eisc->history_after;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_COST KheDrsExprIntSeqCostExtraCost(KHE_DRS_EXPR_INT_SEQ_COST eisc,   */
/*    int p, int q, int r, int l, int y, bool debug)                         */
/*                                                                           */
/*  Return the extra cost in the dominance testing formula with these        */
/*  attributes.                                                              */
/*                                                                           */
/*****************************************************************************/

static KHE_COST KheDrsExprIntSeqCostExtraCost(KHE_DRS_EXPR_INT_SEQ_COST eisc,
  int p, int q, int r, int l, int y, bool debug)
{
  int d1, d2;  KHE_COST res;
  d1 = KheDrsExprIntSeqCostDelta(eisc, l+y, l+y, l+y + ci_bar(eisc, p, y));
  d2 = KheDrsExprIntSeqCostDelta(eisc, l+r, l, l+q+r + ci_bar(eisc, p, q+r));
  res = f(eisc, d1) - f(eisc, d2);
  if( debug )
    fprintf(stderr, "    extra(p %d, q %d, r %d, l %d, y %d) = f(%d) %8.5f - "
      "f(%d) %8.5f = %8.5f\n", p, q, r, l, y, d1, KheCostShow(f(eisc, d1)),
      d2, KheCostShow(f(eisc, d2)), KheCostShow(res));
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_COST KheDrsExprIntSeqCostFindPsi(KHE_DRS_EXPR_INT_SEQ_COST eisc,     */
/*    int p, int q, int r, int l1, int l2)                                   */
/*                                                                           */
/*  Find psi, the available cost for eisc given p, q, r, l1, and l2.         */
/*  This implements the outer "min" of the formula in the documentation.     */
/*                                                                           */
/*****************************************************************************/
#define DEBUG_INT_SEQ_DOM(p, q, r, l1, l2)				\
  (DEBUG37 && (p) == 10 && (l1) == 10 && (l2) == 5)

static KHE_COST KheDrsExprIntSeqCostFindPsi(KHE_DRS_EXPR_INT_SEQ_COST eisc,
  int lower, int p, int q, int r, int l1, int l2)
{
  int y;  KHE_COST res, extra_cost1, extra_cost2, cost;  bool debug;
  HnAssert(lower <= p + q, "KheDrsExprIntSeqCostFindPsi internal error");
  debug = DEBUG_INT_SEQ_DOM(p, q, r, l1, l2);
  if( debug )
    fprintf(stderr, "[ KheDrsExprIntSeqCostFindPsi(%s, lower %d, p %d, q %d,"
      " r %d, l1 %d, l2 %d)\n", KheDrsMonitorInfoId(eisc->monitor_info),
      lower, p, q, r, l1, l2);
  res = KheCost(INT_MAX, INT_MAX);
  for( y = lower;  y <= q + r;  y++ )
  {
    extra_cost2 = KheDrsExprIntSeqCostExtraCost(eisc, p, q, r, l2, y, debug);
    extra_cost1 = KheDrsExprIntSeqCostExtraCost(eisc, p, q, r, l1, y, debug);
    cost = extra_cost2 - extra_cost1;
    if( debug )
      fprintf(stderr, "  y = %2d: %8.5f - %8.5f = %8.5f\n", y,
        KheCostShow(extra_cost2), KheCostShow(extra_cost1), KheCostShow(cost));
    if( cost < res )
      res = cost;
  }
  if( debug )
    fprintf(stderr, "] KheDrsExprIntSeqCostFindPsi returning %.5f\n",
      KheCostShow(res));
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_COST KheDrsExprIntSeqCostFindPsi0(KHE_DRS_EXPR_INT_SEQ_COST eisc,    */
/*    int p, int q, int r, int l1, int l2)                                   */
/*                                                                           */
/*  Find psi0, part of the available cost for eisc given p, q, r, l1, and    */
/*  This implements the outer "min" of the formula in the documentation.     */
/*                                                                           */
/*****************************************************************************/
#define DEBUG_INT_SEQ_DOM(p, q, r, l1, l2)				\
  (DEBUG37 && (p) == 10 && (l1) == 10 && (l2) == 5)

static KHE_COST KheDrsExprIntSeqCostFindPsi0(KHE_DRS_EXPR_INT_SEQ_COST eisc,
  int p, int q, int r, int l1, int l2)
{
  int y;  KHE_COST res, extra_cost1, extra_cost2;  bool debug;
  debug = DEBUG_INT_SEQ_DOM(p, q, r, l1, l2);
  if( debug )
    fprintf(stderr, "[ KheDrsExprIntSeqCostFindPsi0(%s, p %d, q %d,"
      " r %d, l1 %d, l2 %d)\n", KheDrsMonitorInfoId(eisc->monitor_info),
      p, q, r, l1, l2);
  res = KheCost(INT_MAX, INT_MAX);
  y = 0;
  extra_cost2 = KheDrsExprIntSeqCostExtraCost(eisc, p, q, r, l2, y, debug);
  extra_cost1 = KheDrsExprIntSeqCostExtraCost(eisc, p, q, r, l1, y, debug);
  res = extra_cost2 - extra_cost1;
  if( debug )
  {
    fprintf(stderr, "  y = %2d: %8.5f - %8.5f = %8.5f\n", y,
      KheCostShow(extra_cost2), KheCostShow(extra_cost1), KheCostShow(res));
    fprintf(stderr, "] KheDrsExprIntSeqCostFindPsi0 returning %.5f\n",
      KheCostShow(res));
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSeqCostSetDomTable(KHE_DRS_EXPR_INT_SEQ_COST eisc,     */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Set the dom_table field of eisc.                                         */
/*                                                                           */
/*****************************************************************************/
static void KheDrsExprIntSeqCostDebugDomTable(KHE_DRS_EXPR_INT_SEQ_COST eisc,
  int verbosity, int indent, FILE *fp);

static void KheDrsExprIntSeqCostSetDomTable(KHE_DRS_EXPR_INT_SEQ_COST eisc,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int p, q, r, l1, l2, child_count;  KHE_COST psi, psi0, psi_plus;
  KHE_DRS_COST_TUPLE ct;  char *debug_str;  HA_ARENA a;
  debug_str = KheDrsMonitorInfoId(eisc->monitor_info);
  if( DEBUG42 )
    fprintf(stderr, "[ KheDrsExprIntSeqCostSetDomTable(%s)\n", debug_str);
  HnAssert(eisc->dom_table == NULL,
    "KheDrsExprIntSeqCostSetDomTable: dom_table");
  child_count = HaArrayCount(eisc->children);
  a = drs->arena;
  eisc->dom_table = KheDrsDim5TableMake(HnStringMake(a, "Seq %s",debug_str), a);
  for( p = 0;  p <= child_count;  p++ )
    for( q = 0;  q <= p;  q++ )
      for( r = 0;  r <= p - q;  r++ )
	for( l1 = 0;  l1 <= child_count - p + eisc->history;  l1++ )
	  for( l2 = 0;  l2 <= child_count - p + eisc->history;  l2++ )
	  {
	    psi = KheDrsExprIntSeqCostFindPsi(eisc, 0, p, q, r, l1, l2);
	    psi0 = KheDrsExprIntSeqCostFindPsi0(eisc, p, q, r, l1, l2);
	    if( q == 0 )
	      ct = KheDrsCostTupleMake(psi, psi0, false, 0);
	    else
	    {
	      psi_plus = KheDrsExprIntSeqCostFindPsi(eisc, 1, p, q, r, l1, l2);
	      ct = KheDrsCostTupleMake(psi, psi0, true, psi_plus);
	    }
	    KheDrsDim5TablePut(eisc->dom_table, p, q, r, l1, l2, ct, drs);
	  }
  if( DEBUG36 || DEBUG44_MONITOR(eisc->monitor_info->monitor) )
    KheDrsExprIntSeqCostDebugDomTable(eisc, 1, 0, stderr);
  if( DEBUG42 )
    fprintf(stderr, "] KheDrsExprIntSeqCostSetDomTable\n");
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsExprIntSeqCostWantDebugDomTable(                              */
/*    KHE_DRS_EXPR_INT_SEQ_COST eisc)                                        */
/*                                                                           */
/*  Return true if we want to debug eisc's dom table.                        */
/*                                                                           */
/*  At present we are doing this is ec's monitor is a resource monitor       */
/*  and the resource it monitors is the first resource of its type.          */
/*  This is to limit the debug output to a manageable amount.                */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsExprIntSeqCostWantDebugDomTable(
  KHE_DRS_EXPR_INT_SEQ_COST eisc)
{
  KHE_MONITOR m;  KHE_RESOURCE r;  KHE_RESOURCE_TYPE rt;

  /* find the resource */
  m = eisc->monitor_info->monitor;
  HnAssert(KheMonitorTag(m) == KHE_LIMIT_ACTIVE_INTERVALS_MONITOR_TAG,
    "KheDrsExprIntSeqCostWantDebugDomTable internal error");
  r = KheLimitActiveIntervalsMonitorResource(
    (KHE_LIMIT_ACTIVE_INTERVALS_MONITOR) m);

  /* return true if r is the first resource of its type */
  rt = KheResourceResourceType(r);
  return KheResourceTypeResourceCount(rt) > 0 &&
    KheResourceTypeResource(rt, 0) == r;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSeqCostDebugDomTable(KHE_DRS_EXPR_INT_SEQ_COST eisc,   */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of eisc's dom table.                                         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntSeqCostDebugDomTable(KHE_DRS_EXPR_INT_SEQ_COST eisc,
  int verbosity, int indent, FILE *fp)
{
  if( KheDrsExprIntSeqCostWantDebugDomTable(eisc) )
  {
    fprintf(fp, "%*s[ ExprCost for %s\n", indent, "",
      KheDrsMonitorInfoId(eisc->monitor_info));
    fprintf(fp, "%*s  cost_fn %s, combined_weight %.5f, min_limit %d,"
      " max_limit %d\n", indent, "",
      KheCostFunctionShow(eisc->cost_fn), KheCostShow(eisc->combined_weight),
      eisc->min_limit, eisc->max_limit);
    fprintf(fp, "%*s  history_before %d, history_after %d, history %d\n",
      indent, "", eisc->history_before, eisc->history_after, eisc->history);
    KheDrsDim5TableDebug(eisc->dom_table, verbosity, indent + 2, fp);
    /* KheDrsDomTableSeqDebug(eisc->dom_table, verbosity, indent + 2, fp); */
    fprintf(fp, "%*s]\n", indent, "");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_INT_SEQ_COST"                                    */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSeqCostSetSeqType(KHE_DRS_EXPR_INT_SEQ_COST eisc,      */
/*    KHE_FRAME days_frame)                                                  */
/*                                                                           */
/*  Set eisc->seq_type and eisc->seq_index if eisc matches days_frame.       */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntSeqCostSetSeqType(KHE_DRS_EXPR_INT_SEQ_COST eisc,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  bool found_positive, found_negative, found_single;
  KHE_DRS_EXPR child_e, child_child_e;  int i, index, pos;
  KHE_DRS_EXPR_BUSY_DAY busy_day_e;  KHE_DRS_EXPR_FREE_DAY free_day_e;
  KHE_DRS_EXPR_BUSY_TIME busy_time_e;  KHE_TIME_GROUP frame_tg;

  /* fail immediately if the number of children is wrong */
  if( HaArrayCount(eisc->children) != KheFrameTimeGroupCount(drs->days_frame) )
    return;

  /* compare time groups and check all positive, negative, or positive single */
  found_positive = found_negative = found_single = false;
  index = -1;
  HaArrayForEach(eisc->children, child_e, i)
  {
    frame_tg = KheFrameTimeGroup(drs->days_frame, i);

    if( child_e->tag == KHE_DRS_EXPR_OR_TAG )
    {
      /* if not a single child, fail */
      if( HaArrayCount(child_e->children) != 1 )
	return;

      child_child_e = HaArrayFirst(child_e->children);
      if( child_child_e->tag == KHE_DRS_EXPR_BUSY_DAY_TAG )
      {
	/* this is a positive, so if not all positive, fail */
	if( found_negative || found_single )
	  return;
	found_positive = true;

	/* if time groups don't match, fail */
	busy_day_e = (KHE_DRS_EXPR_BUSY_DAY) child_child_e;
	if( !KheTimeGroupEqual(busy_day_e->time_group, frame_tg) )
	  return;
      }
      else if( child_child_e->tag == KHE_DRS_EXPR_BUSY_TIME_TAG )
      {
	/* this is a single positive, so if not all single, fail */
	if( found_negative || found_positive )
	  return;
	found_single = true;

	/* its time has to be an element of frame_tg */
	busy_time_e = (KHE_DRS_EXPR_BUSY_TIME) child_child_e;
	if( !KheTimeGroupContains(frame_tg, busy_time_e->time, &pos) )
	  return;

	/* its time has to have the same index as the other singles */
	if( index == -1 )
	  index = pos;
	else if( pos != index )
	  return;
      }
      else
	return;
    }
    else if( child_e->tag == KHE_DRS_EXPR_AND_TAG )
    {
      /* this is negative, so if not all negative, fail */
      if( found_positive || found_single )
	return;
      found_negative = true;

      /* if not a single child, fail */
      if( HaArrayCount(child_e->children) != 1 )
	return;

      /* if not a free day child, fail */
      child_child_e = HaArrayFirst(child_e->children);
      if( child_child_e->tag != KHE_DRS_EXPR_FREE_DAY_TAG )
	return;

      /* if time groups don't match, fail */
      free_day_e = (KHE_DRS_EXPR_FREE_DAY) child_child_e;
      if( !KheTimeGroupEqual(free_day_e->time_group, frame_tg) )
	return;
    }
    else
    {
      HnAbort("KheDrsExprIntSeqCostSetSeqType internal error 1");
      return;  /* keep compiler happy */
    }
  }

  /* all good, so wrap up */
  if( found_positive )
    eisc->seq_type = KHE_DRS_SEQ_DAYS_POSITIVE, eisc->seq_index = -1;
  else if( found_negative )
    eisc->seq_type = KHE_DRS_SEQ_DAYS_NEGATIVE, eisc->seq_index = -1;
  else if( found_single )
    eisc->seq_type = KHE_DRS_SEQ_SINGLE_POSITIVE, eisc->seq_index = index;
  else
    return;

  /* if we get to this point, success */
  if( DEBUG54 )
  {
    fprintf(stderr, "[ %s matches frame:\n",
      KheDrsMonitorInfoId(eisc->monitor_info));
    KheDrsExprDebug((KHE_DRS_EXPR) eisc, drs, 1, 2, stderr);
    fprintf(stderr, "]\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_INT_SEQ_COST KheDrsExprIntSeqCostMake(                      */
/*    KHE_DRS_MONITOR_INFO mi, KHE_COST_FUNCTION cost_fn,                    */
/*    KHE_COST combined_weight, int min_limit, int max_limit,                */
/*    int history_before, int history_after, int history,                    */
/*    KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs)                  */
/*                                                                           */
/*  Make a new KHE_DRS_EXPR_INT_SEQ_COST object with these attributes.       */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_INT_SEQ_COST KheDrsExprIntSeqCostMake(
  KHE_DRS_MONITOR_INFO mi, KHE_COST_FUNCTION cost_fn,
  KHE_COST combined_weight, int min_limit, int max_limit,
  int history_before, int history_after, int history,
  KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_INT_SEQ_COST res;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_INT_SEQ_COST_TAG,
    dr, drs);
  res->u.int_val = UNDEF_INT;
  res->monitor_info = mi;
  res->seq_type = KHE_DRS_SEQ_NONE;  /* may change later */
  res->seq_index = 0;
  res->cost_fn = cost_fn;
  res->combined_weight = combined_weight;
  res->min_limit = min_limit;
  res->max_limit = max_limit;
  res->adjust_type = KheDrsAdjustType(cost_fn, max_limit);
  res->history_before = history_before;
  res->history_after = history_after;
  res->history = history;
  res->dom_table = NULL;  /* reset later */
  res->first_open_child_index = 0;  /* ready for when we open */
  HaArrayInit(res->closed_seqs, drs->arena);
  HaArrayAddLast(res->closed_seqs, KheDrsClosedSeqMakeInit(drs));
  KheDrsAddWeight(drs, combined_weight);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSeqCostAddChild(KHE_DRS_EXPR_INT_SEQ_COST eisc,        */
/*    KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  New child child_e has just been added to eisc.  Update eisc's state      */
/*  appropriately.  That is, extend the sole closed sequence one place.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntSeqCostAddChild(KHE_DRS_EXPR_INT_SEQ_COST eisc,
  KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_CLOSED_SEQ dcs;
  HnAssert(HaArrayCount(eisc->closed_seqs) == 1,
    "KheDrsExprIntSeqCostAddChild internal error (%d closed seqs)",
    HaArrayCount(eisc->closed_seqs));
  dcs = HaArrayFirst(eisc->closed_seqs);
  KheDrsClosedSeqExtendToRight(dcs, child_e->u.int_val);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSeqCostClosedSeqDebug(KHE_DRS_EXPR_INT_SEQ_COST eisc,  */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug printf of eisc's closed sequences.                                 */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntSeqCostClosedSeqDebug(KHE_DRS_EXPR_INT_SEQ_COST eisc,
  int verbosity, int indent, FILE *fp)
{
  KHE_DRS_CLOSED_SEQ dcs;  int i;
  HaArrayForEach(eisc->closed_seqs, dcs, i)
    KheDrsClosedSeqDebug(dcs, verbosity, indent, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSeqCostChildHasOpened(KHE_DRS_EXPR_INT_SEQ_COST eisc,  */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  Inform internal expression eisc that its child_index'th child, child_e,  */
/*  has opened.  The child will have its correct closed_state and value      */
/*  fields when this is called, and it will have been added to the end of    */
/*  eisc->open_children.                                                     */
/*                                                                           */
/*****************************************************************************/

#define DEBUG_RERUN(eisc) (DEBUG14 &&					  \
  strcmp(KheDrsMonitorInfoId(eisc->monitor_info), RERUN_MONITOR_ID) == 0)

static void KheDrsExprIntSeqCostChildHasOpened(KHE_DRS_EXPR_INT_SEQ_COST eisc,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_CLOSED_SEQ dcs, new_dcs;  KHE_DRS_EXPR e;  int i, open_index;
  KHE_DRS_AU_INTERVAL aui_left, aui_right, aui_merged, aui_before, aui_after;
  KHE_DRS_A_INTERVAL ai_left, ai_right, ai_right_before, ai_merged;

  /* find the index of child_e in eisc->open_children (easy: at the end) */
  open_index = KheDrsOpenChildrenCount(&eisc->open_children_by_day) - 1;
  if( DEBUG_RERUN(eisc) )
    fprintf(stderr, "[ KheDrsExprIntSeqCostChildHasOpened(%s, child_e, %d)\n",
      KheDrsMonitorInfoId(eisc->monitor_info), child_index);

  /* split the last closed seq at child_index */
  dcs = HaArrayLast(eisc->closed_seqs);
  new_dcs = KheDrsClosedSeqSplit(dcs, child_index, eisc, drs);
  HaArrayAddLast(eisc->closed_seqs, new_dcs);
  if( DEBUG_RERUN(eisc) )
  {
    KheDrsExprIntSeqCostClosedSeqDebug(eisc, 2, 2, stderr);
    HaArrayForEach(eisc->children, e, i)
      KheDrsExprDebug(e, drs, 2, 4, stderr);
  }

  /* update the au-intervals and a-intervals to take account of the change */
  if( child_e->u.int_val == 0 )
  {
    /* the opened child moves from inactive to unassigned */
    /* the au-intervals on each side merge */
    aui_left = KheDrsAUIntervalFindLeft(eisc, open_index);
    aui_right = KheDrsAUIntervalFindRight(eisc, open_index, drs);
    aui_merged = KheDrsAUIntervalMerge(aui_left, aui_right, false);
    drs->solve_start_cost += KheDrsAUIntervalCost(aui_merged, eisc)
      - KheDrsAUIntervalCost(aui_left, eisc)
      - KheDrsAUIntervalCost(aui_right, eisc);
    KheDrsMonitorInfoUpdateRerunCost(eisc->monitor_info, drs, NULL,
      KHE_DRS_OPEN, "open0a", child_index, "+--",
      KheDrsAUIntervalCost(aui_merged, eisc),
      KheDrsAUIntervalCost(aui_left, eisc),
      KheDrsAUIntervalCost(aui_right, eisc));

    /* the a-interval to the right changes its unassigned_precedes */
    if( eisc->cost_fn == KHE_STEP_COST_FUNCTION )
    {
      ai_right_before = KheDrsAIntervalFindRight(eisc, open_index, false);
      ai_right        = KheDrsAIntervalFindRight(eisc, open_index, true);
      drs->solve_start_cost += KheDrsAIntervalCost(ai_right, eisc)
	- KheDrsAIntervalCost(ai_right_before, eisc);
      KheDrsMonitorInfoUpdateRerunCost(eisc->monitor_info, drs, NULL,
	KHE_DRS_OPEN, "open0b", child_index, "+-",
	KheDrsAIntervalCost(ai_right, eisc),
	KheDrsAIntervalCost(ai_right_before, eisc));
    }
  }
  else
  {
    /* the opened child moves from active to unassigned */
    /* the enclosing au-interval is unchanged, but its cost may change */
    aui_left = KheDrsAUIntervalFindLeft(eisc, open_index);
    aui_right = KheDrsAUIntervalFindRight(eisc, open_index, drs);
    aui_before = KheDrsAUIntervalMerge(aui_left, aui_right, true);
    aui_after = KheDrsAUIntervalMerge(aui_left, aui_right, false);
    drs->solve_start_cost += KheDrsAUIntervalCost(aui_after, eisc)
      - KheDrsAUIntervalCost(aui_before, eisc);
    KheDrsMonitorInfoUpdateRerunCost(eisc->monitor_info, drs, NULL,
      KHE_DRS_OPEN, "open1a", child_index, "+-",
      KheDrsAUIntervalCost(aui_after, eisc),
      KheDrsAUIntervalCost(aui_before, eisc));

    /* the enclosing a-interval splits */
    /* NB unassigned_precedes is correct when ai_left or ai_right is empty */
    ai_left = KheDrsAIntervalFindLeft(eisc, open_index);
    ai_right = KheDrsAIntervalFindRight(eisc, open_index, true);
    ai_merged = KheDrsAIntervalMerge(ai_left, ai_right);
    drs->solve_start_cost += KheDrsAIntervalCost(ai_left, eisc)
      + KheDrsAIntervalCost(ai_right, eisc)
      - KheDrsAIntervalCost(ai_merged, eisc);
    KheDrsMonitorInfoUpdateRerunCost(eisc->monitor_info, drs, NULL,
      KHE_DRS_OPEN, "open1b", child_index, "++-",
      KheDrsAIntervalCost(ai_left, eisc),
      KheDrsAIntervalCost(ai_right, eisc),
      KheDrsAIntervalCost(ai_merged, eisc));
  }
  if( DEBUG_RERUN(eisc) )
    fprintf(stderr, "] KheDrsExprIntSeqCostChildHasOpened\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSeqCostChildHasClosed(KHE_DRS_EXPR_INT_SEQ_COST eisc,  */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  Inform internal expression eisc that its child_index'th child, child_e,  */
/*  has closed.  The child will have its correct closed_state and value      */
/*  fields when this is called; but it will not have been removed from       */
/*  e->open_children.                                                        */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntSeqCostChildHasClosed(KHE_DRS_EXPR_INT_SEQ_COST eisc,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int open_index;  KHE_DRS_CLOSED_SEQ dcs1, dcs2;
  KHE_DRS_AU_INTERVAL aui_left, aui_right, aui_merged, aui_before, aui_after;
  KHE_DRS_A_INTERVAL ai_left, ai_right, ai_right_before, ai_merged;

  /* find the index of child_e in eisc->open_children */
  open_index = eisc->first_open_child_index;
  HnAssert(HaArray(eisc->open_children_by_day.children, open_index)
    == child_e, "KheDrsExprIntSeqCostChildHasClosed internal error 1");
  eisc->first_open_child_index++;

  /* update the au-intervals and a-intervals to take account of the change */
  if( child_e->u.int_val == 0 )
  {
    /* the closed child moves from unassigned to inactive */
    /* the enclosing au-interval splits */
    aui_left = KheDrsAUIntervalFindLeft(eisc, 0 /* not open_index */);
    aui_right = KheDrsAUIntervalFindRight(eisc, open_index, drs);
    aui_merged = KheDrsAUIntervalMerge(aui_left, aui_right, false);
    drs->solve_start_cost += KheDrsAUIntervalCost(aui_left, eisc)
      + KheDrsAUIntervalCost(aui_right, eisc)
      - KheDrsAUIntervalCost(aui_merged, eisc);
    KheDrsMonitorInfoUpdateRerunCost(eisc->monitor_info, drs, NULL,
      KHE_DRS_CLOSE, "close0a", child_index, "++-",
      KheDrsAUIntervalCost(aui_left, eisc),
      KheDrsAUIntervalCost(aui_right, eisc),
      KheDrsAUIntervalCost(aui_merged, eisc));

    /* the a-interval to the right changes its unassigned_precedes */
    ai_right_before = KheDrsAIntervalFindRight(eisc, open_index, true);
    ai_right        = KheDrsAIntervalFindRight(eisc, open_index, false);
    drs->solve_start_cost += KheDrsAIntervalCost(ai_right, eisc)
      - KheDrsAIntervalCost(ai_right_before, eisc);
    KheDrsMonitorInfoUpdateRerunCost(eisc->monitor_info, drs, NULL,
      KHE_DRS_CLOSE, "close0b", child_index, "+-",
      KheDrsAIntervalCost(ai_right, eisc),
      KheDrsAIntervalCost(ai_right_before, eisc));
  }
  else
  {
    /* the closed child moves from unassigned to active */
    /* the enclosing au-interval is unchanged, but its cost may change */
    aui_left = KheDrsAUIntervalFindLeft(eisc, 0 /* not open_index */);
    aui_right = KheDrsAUIntervalFindRight(eisc, open_index, drs);
    aui_before = KheDrsAUIntervalMerge(aui_left, aui_right, false);
    aui_after = KheDrsAUIntervalMerge(aui_left, aui_right, true);
    drs->solve_start_cost += KheDrsAUIntervalCost(aui_after, eisc)
      - KheDrsAUIntervalCost(aui_before, eisc);
    KheDrsMonitorInfoUpdateRerunCost(eisc->monitor_info, drs, NULL,
      KHE_DRS_CLOSE, "close1a", child_index, "+-",
      KheDrsAUIntervalCost(aui_after, eisc),
      KheDrsAUIntervalCost(aui_before, eisc));

    /* the a-intervals on each side merge */
    ai_left = KheDrsAIntervalFindLeft(eisc, 0 /* not open_index */);
    ai_right = KheDrsAIntervalFindRight(eisc, open_index, true);
    ai_merged = KheDrsAIntervalMerge(ai_left, ai_right);
    drs->solve_start_cost += KheDrsAIntervalCost(ai_merged, eisc)
      - KheDrsAIntervalCost(ai_left, eisc)
      - KheDrsAIntervalCost(ai_right, eisc);
    KheDrsMonitorInfoUpdateRerunCost(eisc->monitor_info, drs, NULL,
      KHE_DRS_CLOSE, "close1b", child_index, "+--",
      KheDrsAIntervalCost(ai_merged, eisc), KheDrsAIntervalCost(ai_left, eisc),
      KheDrsAIntervalCost(ai_right, eisc));
  }

  /* merge the two relevant closed seqs */
  dcs1 = HaArray(eisc->closed_seqs, 0);
  dcs2 = HaArray(eisc->closed_seqs, open_index + 1);
  KheDrsClosedSeqMerge(dcs1, dcs2, eisc, drs);
  HaArrayPut(eisc->closed_seqs, open_index + 1, NULL);  /* defensive */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSeqCostSetClosedValue(KHE_DRS_EXPR_INT_SEQ_COST eisc,  */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing eisc, set its value suitably for its closed state.    */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntSeqCostSetClosedValue(KHE_DRS_EXPR_INT_SEQ_COST eisc,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  /* clear out eisc->first_open_child_index and eisc->closed_seqs */
  eisc->first_open_child_index = 0;
  while( HaArrayCount(eisc->closed_seqs) > 0 &&
      HaArrayLast(eisc->closed_seqs) == NULL )
    HaArrayDeleteLast(eisc->closed_seqs);
  HnAssert(HaArrayCount(eisc->closed_seqs) == 1,
    "KheDrsExprIntSeqCostSetClosedValue internal error 1");

  /* nothing else to do here, since eisc has a cost but no value */
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprIntSeqCostChildDomTest(                       */
/*    KHE_DRS_EXPR_INT_SEQ_COST eisc, int open_day_index,                    */
/*    KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  Return a dom test suited to a child of eisc.                             */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DOM_TEST KheDrsExprIntSeqCostChildDomTest(
  KHE_DRS_EXPR_INT_SEQ_COST eisc, int open_day_index,
  KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int a, b;

  /* calculate a */
  a = (eisc->max_limit < INT_MAX ? -1 : INT_MAX);

  /* calculate b */
  b = (eisc->min_limit > 0 ? INT_MAX : 0);

  /* make and return the dom test */
  return KheDrsDomTestMake(KHE_DRS_DOM_TEST_STRONG, child_e, false, 0,
    INT_MAX, a, b, /* false, false, */ 0, NULL, NULL, NULL, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprIntSeqCostOpenDayIndexToChildIndex(                        */
/*    KHE_DRS_EXPR_INT_SEQ_COST eisc, int open_day_index)                    */
/*                                                                           */
/*  Convert open_day_index to the index of the first open child whose        */
/*  last open day is that day, and from there to the index in the            */
/*  full list of children of that open child.                                */
/*                                                                           */
/*****************************************************************************/

static int KheDrsExprIntSeqCostOpenDayIndexToChildIndex(
  KHE_DRS_EXPR_INT_SEQ_COST eisc, int open_day_index)
{
  int open_child_index;  KHE_DRS_CLOSED_SEQ dcs;
  open_child_index = KheDrsOpenChildrenBefore(&eisc->open_children_by_day,
    open_day_index);
  HnAssert(open_child_index <
    KheDrsOpenChildrenCount(&eisc->open_children_by_day),
    "KheDrsExprIntSeqCostOpenDayIndexToChildIndex internal error");
  dcs = HaArray(eisc->closed_seqs, open_child_index);
  return dcs->stop_index;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprIntSeqCostDayDomTest(                         */
/*    KHE_DRS_EXPR_INT_SEQ_COST eisc, int open_day_index,                    */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Return a dom test suited to eisc on the open day with this index.        */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DOM_TEST KheDrsExprIntSeqCostDayDomTest(
  KHE_DRS_EXPR_INT_SEQ_COST eisc, int open_day_index,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int a, b, p, q, r, t, index;
  KHE_DRS_DOM_TEST_TYPE dom_test_type;  KHE_DRS_DIM2_TABLE d2;

  /* calculate a */
  if( eisc->max_limit < INT_MAX )
  {
    t = KheDrsOpenChildrenAtOrAfter(&eisc->open_children_by_day,open_day_index);
    a = eisc->max_limit - t;
  }
  else
    a = INT_MAX;

  /* calculate b */
  if( eisc->min_limit > 0 )
    b = eisc->min_limit;
  else
    b = 0;

  /* sort out uniform dominance test */
  index = KheDrsExprIntSeqCostOpenDayIndexToChildIndex(eisc, open_day_index+1);
  p = HaArrayCount(eisc->children) - index;
  KheDrsAUIntervalFindRightLengths(eisc, open_day_index + 1, &q, &r);

  /* make and return the dom test */
  d2 = KheDrsDim5TableGet3(eisc->dom_table, p, q, r);
  if( q < 0 || DEBUG44_MONITOR(eisc->monitor_info->monitor) )
  {
    fprintf(stderr, "[ KheDrsExprIntSeqCostDomTest(%s, %s, drs)\n",
      KheMonitorId(eisc->monitor_info->monitor),
      KheDrsDayId(HaArray(drs->open_days, open_day_index)));
    fprintf(stderr, "  p = %d, q = %d, r = %d, dts:\n", p, q, r);
    KheDrsDim2TableDebug(d2, 2, 2, stderr);
    fprintf(stderr, "]\n");
  }
  HnAssert(q >= 0, "KheDrsExprIntSeqCostDayDomTest internal error (q = %d)", q);
  dom_test_type = KheDomTestTypeActual(drs->solve_dom_test_type,
    eisc->cost_fn != KHE_QUADRATIC_COST_FUNCTION);
  return KheDrsDomTestMake(dom_test_type, (KHE_DRS_EXPR) eisc,
    false, eisc->min_limit, eisc->max_limit, a, b, eisc->combined_weight,
    d2, NULL, eisc->monitor_info->monitor, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDebugActiveLen(KHE_DRS_EXPR_INT_SEQ_COST eisc, char *label,   */
/*    KHE_DRS_CLOSED_SEQ dcs, int active_len,KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  Debug these values if wanted.                                            */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDebugActiveLen(KHE_DRS_EXPR_INT_SEQ_COST eisc, char *label,
  int open_index, KHE_DRS_EXPR child_e, KHE_DRS_SIGNATURE prev_sig,
  KHE_DRS_CLOSED_SEQ dcs, int active_len, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  if( RERUN && drs->rerun != NULL &&
      strcmp(KheDrsMonitorInfoId(eisc->monitor_info), RERUN_MONITOR_ID) == 0 )
  {
    fprintf(stderr, "  [ active_len %s: ", label);
    KheDrsClosedSeqDebug(dcs, 1, -1, stderr);
    fprintf(stderr, " --> %d\n", active_len);
    fprintf(stderr, "    child at open_index %d:", open_index);
    KheDrsExprDebug(child_e, drs, 1, 2, stderr);
    fprintf(stderr, "  ]\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSeqCostEvalSignature(KHE_DRS_EXPR_INT_SEQ_COST eisc,   */
/*    KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, int next_di,           */
/*    KHE_DRS_SIGNATURE next_sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)           */
/*                                                                           */
/*  Evaluate expression eisc in sig.                                         */
/*                                                                           */
/*  Implementation note.  Much of this code is the same as the code for      */
/*  closing each child_e, except that the a-interval and au-interval at      */
/*  left both have length active_len, and the cost is updated in             */
/*  next_soln->cost rather than in drs->solve_start_cost.                    */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntSeqCostEvalSignature(KHE_DRS_EXPR_INT_SEQ_COST eisc,
  KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, int next_di,
  KHE_DRS_SIGNATURE next_sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int open_index, i1, i2, active_len, adjusted_len;  KHE_DRS_EXPR child_e;
  KHE_DRS_AU_INTERVAL aui_left, aui_right, aui_merged;
  KHE_DRS_AU_INTERVAL aui_before, aui_after;
  KHE_DRS_A_INTERVAL ai_right_before, ai_right_after;
  KHE_DRS_A_INTERVAL ai_left, ai_right, ai_merged;
  KHE_DRS_CLOSED_SEQ dcs;

  /* initialize active_len */
  if( KheDrsOpenChildrenIndexIsFirst(&eisc->open_children_by_day, next_di) )
  {
    /* first day */
    dcs = HaArrayFirst(eisc->closed_seqs);
    active_len = dcs->active_at_right;
    if( KheDrsClosedSeqAllActive(dcs) )
      active_len += eisc->history;
    if( DEBUG_LAST_EDGE )
      fprintf(stderr, "  _%d", active_len);
  }
  else
  {
    /* not first day, so retrieve previous active_len */
    active_len = KheDrsExprDaySigVal((KHE_DRS_EXPR) eisc, next_di-1, prev_sig);
    if( DEBUG_LAST_EDGE )
      fprintf(stderr, "  $%d", active_len);
  }

  /* handle each child_e whose last open day is next_di */
  KheDrsOpenChildrenForEach(&eisc->open_children_by_day, next_di,
    child_e, open_index)
  {
    if( child_e->u.int_val == 0 )
    {
      /* child_e moves from unassigned to inactive: update cost */
      /* the enclosing au-interval splits */
      aui_left = KheDrsAUIntervalMakeLeft(eisc, open_index, active_len);
      aui_right = KheDrsAUIntervalFindRight(eisc, open_index, drs);
      aui_merged = KheDrsAUIntervalMerge(aui_left, aui_right, false);
      KheDrsSignatureAddCost(next_sig, KheDrsAUIntervalCost(aui_left, eisc)
	+ KheDrsAUIntervalCost(aui_right, eisc)
	- KheDrsAUIntervalCost(aui_merged, eisc));
      KheDrsMonitorInfoUpdateRerunCost(eisc->monitor_info, drs, dsg,
	KHE_DRS_SEARCH, "search", -1, "++-",
	KheDrsAUIntervalCost(aui_left, eisc),
	KheDrsAUIntervalCost(aui_right, eisc),
	KheDrsAUIntervalCost(aui_merged, eisc));
      if( DEBUG_LAST_EDGE )
	fprintf(stderr, " 0aui +%.5f +%.5f -%.5f",
	  KheCostShow(KheDrsAUIntervalCost(aui_left, eisc)),
          KheCostShow(KheDrsAUIntervalCost(aui_right, eisc)),
          KheCostShow(KheDrsAUIntervalCost(aui_merged, eisc)));

      /* the a-interval to the right changes its unassigned_precedes */
      ai_right_before = KheDrsAIntervalFindRight(eisc, open_index, true);
      ai_right_after  = KheDrsAIntervalFindRight(eisc, open_index, false);
      KheDrsSignatureAddCost(next_sig, KheDrsAIntervalCost(ai_right_after, eisc)
	- KheDrsAIntervalCost(ai_right_before, eisc));
      KheDrsMonitorInfoUpdateRerunCost(eisc->monitor_info, drs, dsg,
	KHE_DRS_SEARCH, "search2", -1, "+-",
	KheDrsAIntervalCost(ai_right_after, eisc),
        KheDrsAIntervalCost(ai_right_before, eisc));
      if( DEBUG_LAST_EDGE )
	fprintf(stderr, " 0ai +%.5f -%.5f",
	  KheCostShow(KheDrsAIntervalCost(ai_right_after, eisc)),
          KheCostShow(KheDrsAIntervalCost(ai_right_before, eisc)));

      /* set active_len for next iteration (child_e is now inactive) */
      dcs = HaArray(eisc->closed_seqs, open_index + 1);
      active_len = dcs->active_at_right;
      KheDrsDebugActiveLen(eisc, "inactive", open_index, child_e, prev_sig,
	dcs, active_len, drs);
      if( DEBUG_LAST_EDGE )
	fprintf(stderr, " %d", active_len);
    }
    else
    {
      /* child_e moves from unassigned to active: update cost */
      /* the enclosing au-interval is unchanged, but its cost may change */
      aui_left = KheDrsAUIntervalMakeLeft(eisc, open_index, active_len);
      aui_right = KheDrsAUIntervalFindRight(eisc, open_index, drs);
      aui_before = KheDrsAUIntervalMerge(aui_left, aui_right, false);
      aui_after = KheDrsAUIntervalMerge(aui_left, aui_right, true);
      KheDrsSignatureAddCost(next_sig, KheDrsAUIntervalCost(aui_after, eisc)
	- KheDrsAUIntervalCost(aui_before, eisc));
      KheDrsMonitorInfoUpdateRerunCost(eisc->monitor_info, drs, dsg,
	KHE_DRS_SEARCH, "search", -1, "+-",
	KheDrsAUIntervalCost(aui_after, eisc),
	KheDrsAUIntervalCost(aui_before, eisc));
      if( DEBUG_LAST_EDGE )
	fprintf(stderr, " 1aui +%.5f -%.5f",
	  KheCostShow(KheDrsAUIntervalCost(aui_after, eisc)),
          KheCostShow(KheDrsAUIntervalCost(aui_before, eisc)));

      /* the a-intervals on each side merge */
      ai_left = KheDrsAIntervalMakeLeft(eisc, open_index, active_len);
      ai_right = KheDrsAIntervalFindRight(eisc, open_index, true);
      ai_merged = KheDrsAIntervalMerge(ai_left, ai_right);
      KheDrsSignatureAddCost(next_sig, KheDrsAIntervalCost(ai_merged, eisc)
	- KheDrsAIntervalCost(ai_left, eisc)
	- KheDrsAIntervalCost(ai_right, eisc));
      KheDrsMonitorInfoUpdateRerunCost(eisc->monitor_info, drs, dsg,
	KHE_DRS_SEARCH, "search2", -1, "+--",
	KheDrsAIntervalCost(ai_merged, eisc),
	KheDrsAIntervalCost(ai_left, eisc),
	KheDrsAIntervalCost(ai_right, eisc));
      if( DEBUG_LAST_EDGE )
	fprintf(stderr, " 1ai +%.5f -%.5f -%.5f",
	  KheCostShow(KheDrsAIntervalCost(ai_merged, eisc)),
          KheCostShow(KheDrsAIntervalCost(ai_left, eisc)),
          KheCostShow(KheDrsAIntervalCost(ai_right, eisc)));

      /* set active_len for next iteration (child_e is now active) */
      dcs = HaArray(eisc->closed_seqs, open_index + 1);
      if( KheDrsClosedSeqAllActive(dcs) )
	active_len += 1 + dcs->active_at_right;
      else
        active_len = dcs->active_at_right;
      KheDrsDebugActiveLen(eisc, "active", open_index, child_e, prev_sig,
	dcs, active_len, drs);
      if( DEBUG_LAST_EDGE )
	fprintf(stderr, " %d", active_len);
    }
  }

  if( !KheDrsOpenChildrenIndexIsLast(&eisc->open_children_by_day, next_di) )
  {
    /* not last day; store active_len (adjusted) in sig */
    adjusted_len = KheDrsAdjustedSigVal(active_len,
      eisc->adjust_type, eisc->min_limit, eisc->max_limit, 0);
    if( DEBUG44_MONITOR(eisc->monitor_info->monitor) && adjusted_len >= 4 )
      fprintf(stderr, "  EvalSig(%s): day %d, adjusted_len = %d\n",
	KheDrsMonitorInfoId(eisc->monitor_info), next_di, adjusted_len);
    KheDrsSignatureAddState(next_sig, adjusted_len, dsg, (KHE_DRS_EXPR) eisc);
    if( DEBUG_LAST_EDGE )
      fprintf(stderr, ":%d$ %s\n", adjusted_len,
	KheDrsMonitorInfoId(eisc->monitor_info));
  }
  else
    if( DEBUG_LAST_EDGE )
      fprintf(stderr, "_ %s\n", KheDrsMonitorInfoId(eisc->monitor_info));
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheDrsExprIntSeqCostShowSeqType(KHE_DRS_EXPR_INT_SEQ_COST eisc)    */
/*                                                                           */
/*  Show the seq_type and seq_index fields of eisc.                          */
/*                                                                           */
/*****************************************************************************/

static char *KheDrsExprIntSeqCostShowSeqType(KHE_DRS_EXPR_INT_SEQ_COST eisc)
{
  static char buff[200];
  switch( eisc->seq_type )
  {
    case KHE_DRS_SEQ_NONE:

      sprintf(buff, "-");
      break;

    case KHE_DRS_SEQ_DAYS_POSITIVE:

      sprintf(buff, "+ve");
      break;

    case KHE_DRS_SEQ_DAYS_NEGATIVE:

      sprintf(buff, "-ve");
      break;

    case KHE_DRS_SEQ_SINGLE_POSITIVE:

      sprintf(buff, "+%d", eisc->seq_index);
      break;

    default:

      sprintf(buff, "?");
      break;
  }
  return buff;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSeqCostDoDebug(KHE_DRS_EXPR_INT_SEQ_COST eisc,         */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of eisc which is specific to it.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntSeqCostDoDebug(KHE_DRS_EXPR_INT_SEQ_COST eisc,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "INT_SEQ_COST(%s, %s) %s",
    KheDrsMonitorInfoId(eisc->monitor_info),
    KheDrsAdjustTypeShow(eisc->adjust_type),
    KheDrsExprIntSeqCostShowSeqType(eisc));
  KheDrsExprDebugSignature((KHE_DRS_EXPR) eisc, soln, false, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR - deferred functions"                            */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprAddChild(KHE_DRS_EXPR parent, KHE_DRS_EXPR child,         */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make child a child of parent.                                            */
/*                                                                           */
/*  Implementation note.  During expression construction, all expressions    */
/*  are closed by the time they are made children of their parents (by       */
/*  calls on this function).  Hence KheDrsExprChildHasClosed below.          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprAddChild(KHE_DRS_EXPR parent, KHE_DRS_EXPR child,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_PARENT prnt;

  /* link parent and child */
  HnAssert(HaArrayCount(parent->parents) == 0,
    "KheDrsExprAddChild internal error:  too late to add child (1)");
  HnAssert(parent->postorder_index == -1,
    "KheDrsExprAddChild internal error:  too late to add child (2)");
  prnt.expr = parent;
  prnt.index = HaArrayCount(parent->children);
  HaArrayAddLast(child->parents, prnt);
  HaArrayAddLast(parent->children, child);

  /* update state in each parent */
  switch( parent->tag )
  {
    case KHE_DRS_EXPR_OR_TAG:

      KheDrsExprOrAddChild((KHE_DRS_EXPR_OR) parent, child, drs);
      break;

    case KHE_DRS_EXPR_AND_TAG:

      KheDrsExprAndAddChild((KHE_DRS_EXPR_AND) parent, child, drs);
      break;

    case KHE_DRS_EXPR_INT_SUM_TAG:

      KheDrsExprIntSumAddChild((KHE_DRS_EXPR_INT_SUM) parent, child, drs);
      break;

    case KHE_DRS_EXPR_FLOAT_SUM_TAG:

      KheDrsExprFloatSumAddChild((KHE_DRS_EXPR_FLOAT_SUM) parent, child, drs);
      break;

    case KHE_DRS_EXPR_INT_DEV_TAG:

      KheDrsExprIntDevAddChild((KHE_DRS_EXPR_INT_DEV) parent, child, drs);
      break;

    case KHE_DRS_EXPR_FLOAT_DEV_TAG:

      KheDrsExprFloatDevAddChild((KHE_DRS_EXPR_FLOAT_DEV) parent, child, drs);
      break;

    case KHE_DRS_EXPR_INT_SUM_COST_TAG:

      KheDrsExprIntSumCostAddChild((KHE_DRS_EXPR_INT_SUM_COST) parent,
	child, drs);
      break;

    case KHE_DRS_EXPR_INT_SEQ_COST_TAG:

      KheDrsExprIntSeqCostAddChild((KHE_DRS_EXPR_INT_SEQ_COST) parent,
	child, drs);
      break;

    default:

      HnAbort("KheDrsExprAddChild internal error (tag %d)", parent->tag);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprChildHasOpened(KHE_DRS_EXPR e, KHE_DRS_EXPR child_e,      */
/*    int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)                      */
/*                                                                           */
/*  As part of opening e for solving, inform e that its child_index'th       */
/*  child, child_e, has opened.                                              */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprChildHasOpened(KHE_DRS_EXPR e, KHE_DRS_EXPR child_e,
  int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  /* int i, e2_index, child_e_index;  KHE_DRS_EXPR e2; */

  /* add child_e to e->open_children */
  if( e->tag == KHE_DRS_EXPR_INT_SEQ_COST_TAG )
    KheDrsOpenChildrenAddChildWithDayUpdate(&e->open_children_by_day, child_e);
  else
    KheDrsOpenChildrenAddChildInDayOrder(&e->open_children_by_day, child_e);

  /* alter the state of e to take account of child_e opening */
  switch( e->tag )
  {
    case KHE_DRS_EXPR_OR_TAG:

      KheDrsExprOrChildHasOpened((KHE_DRS_EXPR_OR) e,
	child_e, child_index, drs);
      break;

    case KHE_DRS_EXPR_AND_TAG:

      KheDrsExprAndChildHasOpened((KHE_DRS_EXPR_AND) e,
	child_e, child_index, drs);
      break;

    case KHE_DRS_EXPR_INT_SUM_TAG:

      KheDrsExprIntSumChildHasOpened((KHE_DRS_EXPR_INT_SUM) e,
	child_e, child_index, drs);
      break;

    case KHE_DRS_EXPR_FLOAT_SUM_TAG:

      KheDrsExprFloatSumChildHasOpened((KHE_DRS_EXPR_FLOAT_SUM) e,
	child_e, child_index, drs);
      break;

    case KHE_DRS_EXPR_INT_DEV_TAG:

      KheDrsExprIntDevChildHasOpened((KHE_DRS_EXPR_INT_DEV) e,
	child_e, child_index, drs);
      break;

    case KHE_DRS_EXPR_FLOAT_DEV_TAG:

      KheDrsExprFloatDevChildHasOpened((KHE_DRS_EXPR_FLOAT_DEV) e,
	child_e, child_index, drs);
      break;

    case KHE_DRS_EXPR_INT_SUM_COST_TAG:

      KheDrsExprIntSumCostChildHasOpened((KHE_DRS_EXPR_INT_SUM_COST) e,
	child_e, child_index, drs);
      break;

    case KHE_DRS_EXPR_INT_SEQ_COST_TAG:

      KheDrsExprIntSeqCostChildHasOpened((KHE_DRS_EXPR_INT_SEQ_COST) e,
	child_e, child_index, drs);
      break;

    default:

      HnAbort("KheDrsExprChildHasOpened internal error (tag %d)", e->tag);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprChildHasClosed(KHE_DRS_EXPR e,                            */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  Inform internal expression e that its child_index'th child, child_e,     */
/*  has closed.  The child will have its correct closed_state and value      */
/*  fields when this is called.                                              */
/*                                                                           */
/*  Although KheDrsExprChildHasOpened adds child_e to e->open_children,      */
/*  KheDrsExprChildHasClosed does not remove child_e from e->open_children.  */
/*  This is because children are closed in the same order they were opened,  */
/*  but there is no efficient way to remove them in this order, short of     */
/*  something too complicated such as a circular list.                       */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprChildHasClosed(KHE_DRS_EXPR e,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  /* do the type-specific part of this operation */
  switch( e->tag )
  {
    case KHE_DRS_EXPR_OR_TAG:

      KheDrsExprOrChildHasClosed((KHE_DRS_EXPR_OR) e,
	child_e, child_index, drs);
      break;

    case KHE_DRS_EXPR_AND_TAG:

      KheDrsExprAndChildHasClosed((KHE_DRS_EXPR_AND) e,
	child_e, child_index, drs);
      break;

    case KHE_DRS_EXPR_INT_SUM_TAG:

      KheDrsExprIntSumChildHasClosed((KHE_DRS_EXPR_INT_SUM) e,
	child_e, child_index, drs);
      break;

    case KHE_DRS_EXPR_FLOAT_SUM_TAG:

      KheDrsExprFloatSumChildHasClosed((KHE_DRS_EXPR_FLOAT_SUM) e,
	child_e, child_index, drs);
      break;

    case KHE_DRS_EXPR_INT_DEV_TAG:

      KheDrsExprIntDevChildHasClosed((KHE_DRS_EXPR_INT_DEV) e,
	child_e, child_index, drs);
      break;

    case KHE_DRS_EXPR_FLOAT_DEV_TAG:

      KheDrsExprFloatDevChildHasClosed((KHE_DRS_EXPR_FLOAT_DEV) e,
	child_e, child_index, drs);
      break;

    case KHE_DRS_EXPR_INT_SUM_COST_TAG:

      KheDrsExprIntSumCostChildHasClosed((KHE_DRS_EXPR_INT_SUM_COST) e,
	child_e, child_index, drs);
      break;

    case KHE_DRS_EXPR_INT_SEQ_COST_TAG:

      KheDrsExprIntSeqCostChildHasClosed((KHE_DRS_EXPR_INT_SEQ_COST) e,
	child_e, child_index, drs);
      break;

    default:

      HnAbort("KheDrsExprChildHasClosed internal error (tag %d)", e->tag);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprSetClosedValue(KHE_DRS_EXPR e,                            */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing e, set its value suitably for its closed state.       */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprSetClosedValue(KHE_DRS_EXPR e,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  switch( e->tag )
  {
    case KHE_DRS_EXPR_ASSIGNED_TASK_TAG:

      KheDrsExprAssignedTaskSetClosedValue((KHE_DRS_EXPR_ASSIGNED_TASK) e, drs);
      break;

    case KHE_DRS_EXPR_BUSY_TIME_TAG:

      KheDrsExprBusyTimeSetClosedValue((KHE_DRS_EXPR_BUSY_TIME) e, drs);
      break;

    case KHE_DRS_EXPR_FREE_TIME_TAG:

      KheDrsExprFreeTimeSetClosedValue((KHE_DRS_EXPR_FREE_TIME) e, drs);
      break;

    case KHE_DRS_EXPR_WORK_TIME_TAG:

      KheDrsExprWorkTimeSetClosedValue((KHE_DRS_EXPR_WORK_TIME) e, drs);
      break;

    case KHE_DRS_EXPR_BUSY_DAY_TAG:

      KheDrsExprBusyDaySetClosedValue((KHE_DRS_EXPR_BUSY_DAY) e, drs);
      break;

    case KHE_DRS_EXPR_FREE_DAY_TAG:

      KheDrsExprFreeDaySetClosedValue((KHE_DRS_EXPR_FREE_DAY) e, drs);
      break;

    case KHE_DRS_EXPR_WORK_DAY_TAG:

      KheDrsExprWorkDaySetClosedValue((KHE_DRS_EXPR_WORK_DAY) e, drs);
      break;

    case KHE_DRS_EXPR_OR_TAG:

      KheDrsExprOrSetClosedValue((KHE_DRS_EXPR_OR) e, drs);
      break;

    case KHE_DRS_EXPR_AND_TAG:

      KheDrsExprAndSetClosedValue((KHE_DRS_EXPR_AND) e, drs);
      break;

    case KHE_DRS_EXPR_INT_SUM_TAG:

      KheDrsExprIntSumSetClosedValue((KHE_DRS_EXPR_INT_SUM) e, drs);
      break;

    case KHE_DRS_EXPR_FLOAT_SUM_TAG:

      KheDrsExprFloatSumSetClosedValue((KHE_DRS_EXPR_FLOAT_SUM) e,drs);
      break;

    case KHE_DRS_EXPR_INT_DEV_TAG:

      KheDrsExprIntDevSetClosedValue((KHE_DRS_EXPR_INT_DEV) e, drs);
      break;

    case KHE_DRS_EXPR_FLOAT_DEV_TAG:

      KheDrsExprFloatDevSetClosedValue((KHE_DRS_EXPR_FLOAT_DEV) e,drs);
      break;

    case KHE_DRS_EXPR_INT_SUM_COST_TAG:

      KheDrsExprIntSumCostSetClosedValue((KHE_DRS_EXPR_INT_SUM_COST) e, drs);
      break;

    case KHE_DRS_EXPR_INT_SEQ_COST_TAG:

      KheDrsExprIntSeqCostSetClosedValue((KHE_DRS_EXPR_INT_SEQ_COST) e, drs);
      break;

    default:

      HnAbort("KheDrsExprSetClosedValue internal error (tag %d)", e->tag);
      break;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprChildDomTest(KHE_DRS_EXPR e,                  */
/*    KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  Return a dom test suited to child_e, a child of e.                       */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DOM_TEST KheDrsExprChildDomTest(KHE_DRS_EXPR e,
  int open_day_index, KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_PARENT parent;
  switch( e->tag )
  {
    case KHE_DRS_EXPR_ASSIGNED_TASK_TAG:
    case KHE_DRS_EXPR_BUSY_TIME_TAG:
    case KHE_DRS_EXPR_FREE_TIME_TAG:
    case KHE_DRS_EXPR_WORK_TIME_TAG:
    case KHE_DRS_EXPR_BUSY_DAY_TAG:
    case KHE_DRS_EXPR_FREE_DAY_TAG:
    case KHE_DRS_EXPR_WORK_DAY_TAG:

      HnAbort("KheDrsExprChildDomTest: unexpected tag %d", e->tag);
      /* no break, to keep compiler happy */

    case KHE_DRS_EXPR_OR_TAG:
    case KHE_DRS_EXPR_AND_TAG:
    case KHE_DRS_EXPR_INT_SUM_TAG:
    case KHE_DRS_EXPR_FLOAT_SUM_TAG:
    case KHE_DRS_EXPR_INT_DEV_TAG:
    case KHE_DRS_EXPR_FLOAT_DEV_TAG:

      HnAssert(HaArrayCount(e->parents) > 0,
	"KheDrsExprChildDomTest internal error");
      parent = HaArrayFirst(e->parents);
      return KheDrsExprChildDomTest(parent.expr, open_day_index, child_e, drs);

    case KHE_DRS_EXPR_INT_SUM_COST_TAG:

      return KheDrsExprIntSumCostChildDomTest((KHE_DRS_EXPR_INT_SUM_COST) e,
	open_day_index, child_e, drs);

    case KHE_DRS_EXPR_INT_SEQ_COST_TAG:

      return KheDrsExprIntSeqCostChildDomTest((KHE_DRS_EXPR_INT_SEQ_COST) e,
	open_day_index, child_e, drs);

    default:

      HnAbort("KheDrsExprChildDomTest internal error (tag %d)", e->tag);
      return KheDrsExprIntSeqCostChildDomTest((KHE_DRS_EXPR_INT_SEQ_COST) e,
	0, NULL, drs);  /* keep compiler happy */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprDayDomTest(KHE_DRS_EXPR e,                    */
/*    int open_day_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)                   */
/*                                                                           */
/*  Return a dom test suited to e on the open day with this open day index.  */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DOM_TEST KheDrsExprDayDomTest(KHE_DRS_EXPR e,
  int open_day_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_PARENT prnt;
  switch( e->tag )
  {
    case KHE_DRS_EXPR_ASSIGNED_TASK_TAG:
    case KHE_DRS_EXPR_BUSY_TIME_TAG:
    case KHE_DRS_EXPR_FREE_TIME_TAG:
    case KHE_DRS_EXPR_WORK_TIME_TAG:
    case KHE_DRS_EXPR_BUSY_DAY_TAG:
    case KHE_DRS_EXPR_FREE_DAY_TAG:
    case KHE_DRS_EXPR_WORK_DAY_TAG:

      HnAbort("KheDrsExprDayDomTest: unexpected tag %d", e->tag);
      /* no break, to keep compiler happy */

    case KHE_DRS_EXPR_OR_TAG:
    case KHE_DRS_EXPR_AND_TAG:
    case KHE_DRS_EXPR_INT_SUM_TAG:
    case KHE_DRS_EXPR_FLOAT_SUM_TAG:
    case KHE_DRS_EXPR_INT_DEV_TAG:
    case KHE_DRS_EXPR_FLOAT_DEV_TAG:

      HnAssert(HaArrayCount(e->parents) > 0,
	"KheDrsExprDayDomTest internal error");
      prnt = HaArrayFirst(e->parents);
      return KheDrsExprChildDomTest(prnt.expr, open_day_index, e, drs);

    case KHE_DRS_EXPR_INT_SUM_COST_TAG:

      return KheDrsExprIntSumCostDayDomTest((KHE_DRS_EXPR_INT_SUM_COST) e,
	open_day_index, drs);

    case KHE_DRS_EXPR_INT_SEQ_COST_TAG:

      return KheDrsExprIntSeqCostDayDomTest((KHE_DRS_EXPR_INT_SEQ_COST) e,
	open_day_index, drs);

    default:

      HnAbort("KheDrsExprDayDomTest internal error (tag %d)", e->tag);
      return KheDrsExprIntSeqCostDayDomTest((KHE_DRS_EXPR_INT_SEQ_COST) e,
	open_day_index, drs);  /* keep compiler happy */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprLeafSet(KHE_DRS_EXPR e, KHE_DRS_TASK_ON_DAY dtd,          */
/*    KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs)                  */
/*                                                                           */
/*  Inform leaf expression e that dtd has been assigned dr.                  */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprLeafSet(KHE_DRS_EXPR e, KHE_DRS_TASK_ON_DAY dtd,
  KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  switch( e->tag )
  {
    case KHE_DRS_EXPR_ASSIGNED_TASK_TAG:

      KheDrsExprAssignedTaskLeafSet((KHE_DRS_EXPR_ASSIGNED_TASK) e, dtd, dr);
      break;

    case KHE_DRS_EXPR_BUSY_TIME_TAG:

      KheDrsExprBusyTimeLeafSet((KHE_DRS_EXPR_BUSY_TIME) e, dtd, dr);
      break;

    case KHE_DRS_EXPR_FREE_TIME_TAG:

      KheDrsExprFreeTimeLeafSet((KHE_DRS_EXPR_FREE_TIME) e, dtd, dr);
      break;

    case KHE_DRS_EXPR_WORK_TIME_TAG:

      KheDrsExprWorkTimeLeafSet((KHE_DRS_EXPR_WORK_TIME) e, dtd, dr);
      break;

    case KHE_DRS_EXPR_BUSY_DAY_TAG:

      KheDrsExprBusyDayLeafSet((KHE_DRS_EXPR_BUSY_DAY) e, dtd, dr, drs);
      break;

    case KHE_DRS_EXPR_FREE_DAY_TAG:

      KheDrsExprFreeDayLeafSet((KHE_DRS_EXPR_FREE_DAY) e, dtd, dr);
      break;

    case KHE_DRS_EXPR_WORK_DAY_TAG:

      KheDrsExprWorkDayLeafSet((KHE_DRS_EXPR_WORK_DAY) e, dtd, dr);
      break;

    default:

      HnAbort("KheDrsExprLeafSet internal error: tag %d", e->tag);
      break;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprLeafClear(KHE_DRS_EXPR e)                                 */
/*                                                                           */
/*  Inform leaf expression e that the assignment of drd to dtd carried out   */
/*  by KheDrsExprLeafSet above has been removed.                             */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprLeafClear(KHE_DRS_EXPR e, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  switch( e->tag )
  {
    case KHE_DRS_EXPR_ASSIGNED_TASK_TAG:

      KheDrsExprAssignedTaskLeafClear((KHE_DRS_EXPR_ASSIGNED_TASK) e);
      break;

    case KHE_DRS_EXPR_BUSY_TIME_TAG:

      KheDrsExprBusyTimeLeafClear((KHE_DRS_EXPR_BUSY_TIME) e);
      break;

    case KHE_DRS_EXPR_FREE_TIME_TAG:

      KheDrsExprFreeTimeLeafClear((KHE_DRS_EXPR_FREE_TIME) e);
      break;

    case KHE_DRS_EXPR_WORK_TIME_TAG:

      KheDrsExprWorkTimeLeafClear((KHE_DRS_EXPR_WORK_TIME) e);
      break;

    case KHE_DRS_EXPR_BUSY_DAY_TAG:

      KheDrsExprBusyDayLeafClear((KHE_DRS_EXPR_BUSY_DAY) e, drs);
      break;

    case KHE_DRS_EXPR_FREE_DAY_TAG:

      KheDrsExprFreeDayLeafClear((KHE_DRS_EXPR_FREE_DAY) e);
      break;

    case KHE_DRS_EXPR_WORK_DAY_TAG:

      KheDrsExprWorkDayLeafClear((KHE_DRS_EXPR_WORK_DAY) e);
      break;

    default:

      HnAbort("KheDrsExprLeafClear internal error: tag %d");
      break;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprEvalSignature(KHE_DRS_EXPR e, KHE_DRS_SIGNER dsg,         */
/*    KHE_DRS_SIGNATURE prev_sig, int next_di, KHE_DRS_SIGNATURE sig,        */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, bool debug)                           */
/*                                                                           */
/*  Like KheDrsExprEvalDay except that the results go into ss rather         */
/*  than into next_soln.                                                     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprEvalSignature(KHE_DRS_EXPR e, KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE prev_sig, int next_di, KHE_DRS_SIGNATURE next_sig,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, bool debug)
{
  switch( e->tag )
  {
    case KHE_DRS_EXPR_ASSIGNED_TASK_TAG:
    case KHE_DRS_EXPR_BUSY_TIME_TAG:
    case KHE_DRS_EXPR_FREE_TIME_TAG:
    case KHE_DRS_EXPR_WORK_TIME_TAG:
    case KHE_DRS_EXPR_BUSY_DAY_TAG:
    case KHE_DRS_EXPR_FREE_DAY_TAG:
    case KHE_DRS_EXPR_WORK_DAY_TAG:

      HnAbort("KheDrsExprEvalSignature: external tag (%d)", e->tag);
      break;

    case KHE_DRS_EXPR_OR_TAG:

      KheDrsExprOrEvalSignature((KHE_DRS_EXPR_OR) e, dsg,
	prev_sig, next_di, next_sig, drs);
      break;

    case KHE_DRS_EXPR_AND_TAG:

      KheDrsExprAndEvalSignature((KHE_DRS_EXPR_AND) e, dsg,
	prev_sig, next_di, next_sig, drs);
      break;

    case KHE_DRS_EXPR_INT_SUM_TAG:

      KheDrsExprIntSumEvalSignature((KHE_DRS_EXPR_INT_SUM) e, dsg,
	prev_sig, next_di, next_sig, drs);
      break;

    case KHE_DRS_EXPR_FLOAT_SUM_TAG:

      KheDrsExprFloatSumEvalSignature((KHE_DRS_EXPR_FLOAT_SUM) e, dsg,
	prev_sig, next_di, next_sig, drs);
      break;

    case KHE_DRS_EXPR_INT_DEV_TAG:

      KheDrsExprIntDevEvalSignature((KHE_DRS_EXPR_INT_DEV) e, dsg,
	prev_sig, next_di, next_sig, drs);
      break;

    case KHE_DRS_EXPR_FLOAT_DEV_TAG:

      KheDrsExprFloatDevEvalSignature((KHE_DRS_EXPR_FLOAT_DEV) e, dsg,
	prev_sig, next_di, next_sig, drs);
      break;

    case KHE_DRS_EXPR_INT_SUM_COST_TAG:

      KheDrsExprIntSumCostEvalSignature((KHE_DRS_EXPR_INT_SUM_COST) e, dsg,
	prev_sig, next_di, next_sig, drs, debug);
      break;

    case KHE_DRS_EXPR_INT_SEQ_COST_TAG:

      KheDrsExprIntSeqCostEvalSignature((KHE_DRS_EXPR_INT_SEQ_COST) e, dsg,
	prev_sig, next_di, next_sig, drs);
      break;

    default:

      HnAbort("KheDrsExprEvalSignature internal error");
      break;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprDoDebug(KHE_DRS_EXPR e, KHE_DRS_SOLN soln,                */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)                             */
/*                                                                           */
/*  Unindented debug print of that part of e which is specific to it.        */
/*  If soln != NULL, also print e's signature (if any) in soln.              */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprDoDebug(KHE_DRS_EXPR e, KHE_DRS_SOLN soln,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  switch( e->tag )
  {
    case KHE_DRS_EXPR_ASSIGNED_TASK_TAG:

      KheDrsExprAssignedTaskDoDebug((KHE_DRS_EXPR_ASSIGNED_TASK) e,
	soln, drs, fp);
      break;

    case KHE_DRS_EXPR_BUSY_TIME_TAG:

      KheDrsExprBusyTimeDoDebug((KHE_DRS_EXPR_BUSY_TIME) e, soln, drs, fp);
      break;

    case KHE_DRS_EXPR_FREE_TIME_TAG:

      KheDrsExprFreeTimeDoDebug((KHE_DRS_EXPR_FREE_TIME) e, soln, drs, fp);
      break;

    case KHE_DRS_EXPR_WORK_TIME_TAG:

      KheDrsExprWorkTimeDoDebug((KHE_DRS_EXPR_WORK_TIME) e, soln, drs, fp);
      break;

    case KHE_DRS_EXPR_BUSY_DAY_TAG:

      KheDrsExprBusyDayDoDebug((KHE_DRS_EXPR_BUSY_DAY) e, soln, drs, fp);
      break;

    case KHE_DRS_EXPR_FREE_DAY_TAG:

      KheDrsExprFreeDayDoDebug((KHE_DRS_EXPR_FREE_DAY) e, soln, drs, fp);
      break;

    case KHE_DRS_EXPR_WORK_DAY_TAG:

      KheDrsExprWorkDayDoDebug((KHE_DRS_EXPR_WORK_DAY) e, soln, drs, fp);
      break;

    case KHE_DRS_EXPR_OR_TAG:

      KheDrsExprOrDoDebug((KHE_DRS_EXPR_OR) e, soln, drs, fp);
      break;

    case KHE_DRS_EXPR_AND_TAG:

      KheDrsExprAndDoDebug((KHE_DRS_EXPR_AND) e, soln, drs, fp);
      break;

    case KHE_DRS_EXPR_INT_SUM_TAG:

      KheDrsExprIntSumDoDebug((KHE_DRS_EXPR_INT_SUM) e, soln, drs, fp);
      break;

    case KHE_DRS_EXPR_FLOAT_SUM_TAG:

      KheDrsExprFloatSumDoDebug((KHE_DRS_EXPR_FLOAT_SUM) e, soln, drs, fp);
      break;

    case KHE_DRS_EXPR_INT_DEV_TAG:

      KheDrsExprIntDevDoDebug((KHE_DRS_EXPR_INT_DEV) e, soln, drs, fp);
      break;

    case KHE_DRS_EXPR_FLOAT_DEV_TAG:

      KheDrsExprFloatDevDoDebug((KHE_DRS_EXPR_FLOAT_DEV) e, soln, drs, fp);
      break;

    case KHE_DRS_EXPR_INT_SUM_COST_TAG:

      KheDrsExprIntSumCostDoDebug((KHE_DRS_EXPR_INT_SUM_COST) e, soln, drs, fp);
      break;

    case KHE_DRS_EXPR_INT_SEQ_COST_TAG:

      KheDrsExprIntSeqCostDoDebug((KHE_DRS_EXPR_INT_SEQ_COST) e, soln, drs, fp);
      break;

    default:

      HnAbort("KheDrsExprDoDebug internal error: unknown tag %d", e->tag);
      break;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Major category "assignments"                                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_ASST_TO_TASK"                                         */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_ASST_TO_TASK KheDrsAsstToTaskMake(                               */
/*    KHE_DRS_ASST_TO_TASK_CLASS asst, KHE_DRS_TASK_ON_DAY fixed_dtd)        */
/*                                                                           */
/*  Make a new fixed resource assignment object with these attributes.       */
/*  NB there is no memory allocation.                                        */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_ASST_TO_TASK KheDrsAsstToTaskMake(
  KHE_DRS_ASST_TO_TASK_CLASS asst, KHE_DRS_TASK_ON_DAY fixed_dtd)
{
  KHE_DRS_ASST_TO_TASK res;
  res.asst = asst;
  res.fixed_dtd = fixed_dtd;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SHIFT KheDrsAsstToTaskShift(KHE_DRS_ASST_TO_TASK dat)            */
/*                                                                           */
/*  Return the shift that dat is for, or NULL if none.                       */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_SHIFT KheDrsAsstToTaskShift(KHE_DRS_ASST_TO_TASK dat)
{
  return KheDrsAsstToTaskClassShift(dat.asst);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_RESOURCE KheDrsAsstToTaskResource(KHE_DRS_ASST_TO_TASK dat)      */
/*                                                                           */
/*  Return the resource that dat assigns.                                    */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_RESOURCE KheDrsAsstToTaskResource(KHE_DRS_ASST_TO_TASK dat)
{
  return dat.asst->asst_to_shift->resource_on_day->encl_dr;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsAsstToTaskIsFixed(KHE_DRS_ASST_TO_TASK dat)                   */
/*                                                                           */
/*  Return true if dat is a fixed assignment.                                */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsAsstToTaskIsFixed(KHE_DRS_ASST_TO_TASK dat)
{
  KHE_DRS_RESOURCE dr;
  dr = KheDrsAsstToTaskResource(dat);
  return dr->expand_role == KHE_DRS_RESOURCE_EXPAND_FIXED;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsAsstToTaskDebug(KHE_DRS_ASST_TO_TASK dat, int verbosity,      */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of dat onto fp with the given verbosity and indent.          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsAsstToTaskDebug(KHE_DRS_ASST_TO_TASK dat, int verbosity,
  int indent, FILE *fp)
{
  KHE_DRS_RESOURCE dr;
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  dr = KheDrsAsstToTaskResource(dat);
  if( verbosity == 1 )
    fprintf(fp, "<%s, ", KheDrsResourceId(dr));
  else
    fprintf(fp, "<%s %s, ", KheDrsResourceExpandRoleShow(dr->expand_role),
      KheDrsResourceId(dr));
  if( dat.fixed_dtd != NULL )
    KheDrsTaskOnDayDebug(dat.fixed_dtd, 1, -1, fp);
  else
    fprintf(fp, "(free)");
  fprintf(fp, ">");
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_ASST_TO_TASK - expansion"                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheDrsAsstToTaskShiftAsstTrieBuildAssts(                            */
/*    KHE_DRS_ASST_TO_TASK tasst, KHE_DRS_SHIFT_ASST_TRIE dsat,              */
/*    KHE_DRS_SHIFT ds, KHE_DRS_RESOURCE_SET included_resources,             */
/*    int resource_index, KHE_DRS_SOLN prev_soln, KHE_DRS_DAY next_day,      */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Build shift assignments including tasst and later included_resources.    */
/*                                                                           */
/*****************************************************************************/
static void KheDrsExpanderAddAsstToTask(KHE_DRS_EXPANDER de,
  KHE_DRS_ASST_TO_TASK tasst);
static void KheDrsShiftAsstTrieBuildAssts(KHE_DRS_SHIFT_ASST_TRIE dsat,
  KHE_DRS_SHIFT ds, KHE_DRS_RESOURCE_SET included_free_resources,
  int resource_index, KHE_DRS_SOLN prev_soln, KHE_DRS_DAY prev_day,
  KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de);

static void KheDrsAsstToTaskShiftAsstTrieBuildAssts(
  KHE_DRS_ASST_TO_TASK tasst, KHE_DRS_SHIFT_ASST_TRIE dsat,
  KHE_DRS_SHIFT ds, KHE_DRS_RESOURCE_SET included_free_resources,
  int included_free_resources_index, KHE_DRS_SOLN prev_soln,
  KHE_DRS_DAY prev_day, KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de)
{
  /* save the expander so it can be restored later */
  KheDrsExpanderMarkBegin(de);

  /* add tasst to the expander */
  KheDrsExpanderAddAsstToTask(de, tasst);

  /* if the expander is still open, recurse */
  if( KheDrsExpanderIsOpen(de) )
    KheDrsShiftAsstTrieBuildAssts(dsat, ds, included_free_resources,
      included_free_resources_index, prev_soln, prev_day, next_day, de);

  /* restore the expander */
  KheDrsExpanderMarkEnd(de);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsAsstToTaskExpandByResources(KHE_DRS_ASST_TO_TASK tasst,       */
/*    KHE_DRS_SOLN prev_soln, KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de,     */
/*    KHE_DRS_RESOURCE_SET free_resources, int free_resources_index)         */
/*                                                                           */
/*  Add tasst, recurse, then unassign.                                       */
/*                                                                           */
/*****************************************************************************/
static void KheDrsSolnExpandByResources(KHE_DRS_SOLN prev_soln,
  KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de,
  KHE_DRS_RESOURCE_SET free_resources, int free_resources_index);

static void KheDrsAsstToTaskExpandByResources(KHE_DRS_ASST_TO_TASK tasst,
  KHE_DRS_SOLN prev_soln, KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de,
  KHE_DRS_RESOURCE_SET free_resources, int free_resources_index)
{
  /* save the expander so it can be restored later */
  KheDrsExpanderMarkBegin(de);

  /* add tasst to the expander */
  KheDrsExpanderAddAsstToTask(de, tasst);

  /* if the expander is still open, recurse */
  if( KheDrsExpanderIsOpen(de) )
    KheDrsSolnExpandByResources(prev_soln, next_day, de,
      free_resources, free_resources_index + 1);

  /* restore the expander */
  KheDrsExpanderMarkEnd(de);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_ASST_TO_TASK_SET"                                     */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_ASST_TO_TASK_SET KheDrsAsstToTaskSetMake(                        */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new, empty asst to tast set object.                               */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_ASST_TO_TASK_SET KheDrsAsstToTaskSetMake(
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_ASST_TO_TASK_SET res;
  if( HaArrayCount(drs->asst_to_task_set_free_list) > 0 )
  {
    res = HaArrayLastAndDelete(drs->asst_to_task_set_free_list);
    HaArrayClear(res->assts_to_tasks);
  }
  else
  {
    HaMake(res, drs->arena);
    HaArrayInit(res->assts_to_tasks, drs->arena);
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsAsstToTaskSetFree(KHE_DRS_ASST_TO_TASK_SET dats,              */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Free dats.                                                               */
/*                                                                           */
/*****************************************************************************/

static void KheDrsAsstToTaskSetFree(KHE_DRS_ASST_TO_TASK_SET dats,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  HaArrayAddLast(drs->asst_to_task_set_free_list, dats);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsAsstToTaskSetClear(KHE_DRS_ASST_TO_TASK_SET dats)             */
/*                                                                           */
/*  Clear dats back to the empty set of assignments to tasks.                */
/*                                                                           */
/*****************************************************************************/

static void KheDrsAsstToTaskSetClear(KHE_DRS_ASST_TO_TASK_SET dats)
{
  HaArrayClear(dats->assts_to_tasks);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsAsstToTaskSetCount(KHE_DRS_ASST_TO_TASK_SET dats)              */
/*                                                                           */
/*  Return the number of elements of dats.                                   */
/*                                                                           */
/*****************************************************************************/

static int KheDrsAsstToTaskSetCount(KHE_DRS_ASST_TO_TASK_SET dats)
{
  return HaArrayCount(dats->assts_to_tasks);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsAsstToTaskSetAddLast(KHE_DRS_ASST_TO_TASK_SET dats,           */
/*    KHE_DRS_ASST_TO_TASK dat)                                              */
/*                                                                           */
/*  Add dat to dats.                                                         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsAsstToTaskSetAddLast(KHE_DRS_ASST_TO_TASK_SET dats,
  KHE_DRS_ASST_TO_TASK dat)
{
  HaArrayAddLast(dats->assts_to_tasks, dat);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsAsstToTaskSetDebug(KHE_DRS_ASST_TO_TASK_SET dats,             */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of dats onto fp with the given verbosity and indent.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsAsstToTaskSetDebug(KHE_DRS_ASST_TO_TASK_SET dats,
  int verbosity, int indent, FILE *fp)
{
  KHE_DRS_ASST_TO_TASK dat;  int i;
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  fprintf(fp, "{");
  HaArrayForEach(dats->assts_to_tasks, dat, i)
  {
    if( i > 0 )
      fprintf(fp, ", ");
    KheDrsAsstToTaskDebug(dat, verbosity, -1, fp);
  }
  fprintf(fp, "}");
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_ASST_TO_TASK_CLASS"                                   */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_ASST_TO_TASK_CLASS KheDrsAsstToTaskClassMake(                    */
/*    KHE_DRS_ASST_TO_SHIFT sasst, KHE_DRS_TASK_CLASS dtc,                   */
/*    KHE_DRS_TASK_ON_DAY fixed_dtd, KHE_DYNAMIC_RESOURCE_SOLVER drs)        */
/*                                                                           */
/*  Make a new, empty assignment object with these attributes.               */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_ASST_TO_TASK_CLASS KheDrsAsstToTaskClassMake(
  KHE_DRS_ASST_TO_SHIFT sasst, KHE_DRS_TASK_CLASS dtc,
  KHE_DRS_TASK_ON_DAY fixed_dtd, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_ASST_TO_TASK_CLASS res;
  if( HaArrayCount(drs->asst_to_task_class_free_list) > 0 )
  {
    res = HaArrayLastAndDelete(drs->asst_to_task_class_free_list);
    HaArrayClear(res->skip_assts);
  }
  else
  {
    HaMake(res, drs->arena);
    HaArrayInit(res->skip_assts, drs->arena);
  }
  res->asst_to_shift = sasst;
  /* res->resource_on_day = drd; */
  res->task_class = dtc;
  res->fixed_task_on_day = fixed_dtd;
  res->skip_count = 0;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsAsstToTaskClassFree(KHE_DRS_ASST_TO_TASK_CLASS asst,          */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Free asst.                                                               */
/*                                                                           */
/*****************************************************************************/

static void KheDrsAsstToTaskClassFree(KHE_DRS_ASST_TO_TASK_CLASS asst,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  HaArrayAddLast(drs->asst_to_task_class_free_list, asst);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SHIFT KheDrsAsstToTaskClassShift(KHE_DRS_ASST_TO_TASK_CLASS asst)*/
/*                                                                           */
/*  Return the shift that asst assigns, or NULL if asst is for a free day.   */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_SHIFT KheDrsAsstToTaskClassShift(KHE_DRS_ASST_TO_TASK_CLASS asst)
{
  if( asst->task_class != NULL )
    return asst->task_class->encl_shift;
  else if( asst->fixed_task_on_day != NULL )
    return asst->fixed_task_on_day->encl_dt->encl_dtc->encl_shift;
  else
    return NULL;
}


/*****************************************************************************/
/*                                                                           */ 
/*  KHE_DRS_SIGNATURE KheDrsAsstToTaskClassSignature(                        */
/*    KHE_DRS_ASST_TO_TASK_CLASS asst)                                       */
/*                                                                           */
/*  Return the signature of asst.                                            */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_SIGNATURE KheDrsAsstToTaskClassSignature(
  KHE_DRS_ASST_TO_TASK_CLASS asst)
{
  return asst->asst_to_shift->sig;
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsAsstToTaskClassCmp(const void *t1, const void *t2)             */
/*                                                                           */
/*  Comparison function for sorting an array of assignment objects by        */
/*  increasing cost.                                                         */
/*                                                                           */
/*****************************************************************************/

static int KheDrsAsstToTaskClassCmp(const void *t1, const void *t2)
{
  KHE_DRS_ASST_TO_TASK_CLASS asst1 = * (KHE_DRS_ASST_TO_TASK_CLASS *) t1;
  KHE_DRS_ASST_TO_TASK_CLASS asst2 = * (KHE_DRS_ASST_TO_TASK_CLASS *) t2;
  KHE_DRS_SIGNATURE sig1, sig2;
  sig1 = KheDrsAsstToTaskClassSignature(asst1);
  sig2 = KheDrsAsstToTaskClassSignature(asst2);
  return KheCostCmp(sig1->cost, sig2->cost);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsAsstToTaskClassDebug(KHE_DRS_ASST_TO_TASK_CLASS asst,         */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of asst with the given verbosity and indent.                 */
/*                                                                           */
/*****************************************************************************/

static void KheDrsAsstToTaskClassDebug(KHE_DRS_ASST_TO_TASK_CLASS asst,
  int verbosity, int indent, FILE *fp)
{
  KHE_DRS_SIGNATURE sig;
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  fprintf(fp, "<%s on %s: ",
    KheDrsResourceId(asst->asst_to_shift->resource_on_day->encl_dr),
    KheDrsDayId(asst->asst_to_shift->resource_on_day->day));
  if( asst->task_class != NULL )
    KheDrsTaskClassDebug(asst->task_class, NULL, 1, -1, fp);
  else if( asst->fixed_task_on_day != NULL )
    KheDrsTaskOnDayDebug(asst->fixed_task_on_day, 1, -1, fp);
  else
    fprintf(fp, "(free)");
  if( verbosity > 1 )
  {
    fprintf(fp, ", ");
    sig = KheDrsAsstToTaskClassSignature(asst);
    KheDrsSignatureDebug(sig, verbosity, -1, fp);
  }
  fprintf(fp, ">");
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_ASST_TO_TASK_CLASS - expansion"                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheDrsAsstToTaskClassExpandByResources(                             */
/*    KHE_DRS_ASST_TO_TASK_CLASS asst, KHE_DRS_SOLN prev_soln,               */
/*    KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de,                             */
/*    KHE_DRS_RESOURCE_SET free_resources, int free_resources_index)         */
/*                                                                           */
/*  Carry on the expansion of prev_soln at asst.  This works out which       */
/*  task to assign drd, and then tries that assignment.                      */
/*                                                                           */
/*****************************************************************************/
static KHE_DRS_ASST_TO_TASK KheDrsAsstToTaskMake(
  KHE_DRS_ASST_TO_TASK_CLASS asst, KHE_DRS_TASK_ON_DAY fixed_dtd);

static void KheDrsAsstToTaskClassExpandByResources(
  KHE_DRS_ASST_TO_TASK_CLASS asst, KHE_DRS_SOLN prev_soln,
  KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de,
  KHE_DRS_RESOURCE_SET free_resources, int free_resources_index)
{
  KHE_DRS_TASK_ON_DAY dtd;  KHE_DRS_TASK_CLASS dtc;
  KHE_DRS_ASST_TO_TASK tasst;
  dtc = asst->task_class;
  if( dtc != NULL )
  {
    /* select a task from asst->task_class and assign it */
    if( KheDrsTaskClassAcceptResourceBegin(dtc,
	asst->asst_to_shift->resource_on_day, &dtd) )
    {
      tasst = KheDrsAsstToTaskMake(asst, dtd);
      KheDrsAsstToTaskExpandByResources(tasst, prev_soln, next_day,
	de, free_resources, free_resources_index);
      KheDrsTaskClassAcceptResourceEnd(dtc, dtd);
    }
  }
  else
  {
    /* use asst->fixed_task_on_day, possibly NULL meaning a free day */
    tasst = KheDrsAsstToTaskMake(asst, asst->fixed_task_on_day);
    KheDrsAsstToTaskExpandByResources(tasst, prev_soln, next_day,
      de, free_resources, free_resources_index);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsAsstToTaskClassShiftAsstTrieBuildAssts(                       */
/*    KHE_DRS_ASST_TO_TASK_CLASS asst, KHE_DRS_SHIFT_ASST_TRIE dsat,         */
/*    KHE_DRS_SIGNER dsg, KHE_DRS_RESOURCE_SET included_resources,           */
/*    int resource_index, KHE_DRS_SOLN prev_soln, KHE_DRS_DAY next_day,      */
/*    KHE_DRS_EXPANDER de)                                                   */
/*                                                                           */
/*  Build a shift assignment including asst.                                 */
/*                                                                           */
/*****************************************************************************/

static void KheDrsAsstToTaskClassShiftAsstTrieBuildAssts(
  KHE_DRS_ASST_TO_TASK_CLASS asst, KHE_DRS_SHIFT_ASST_TRIE dsat,
  KHE_DRS_SHIFT ds, KHE_DRS_RESOURCE_SET included_free_resources,
  int included_free_resources_index, KHE_DRS_SOLN prev_soln,
  KHE_DRS_DAY prev_day, KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de)
{
  KHE_DRS_TASK_ON_DAY dtd;  KHE_DRS_TASK_CLASS dtc;
  KHE_DRS_ASST_TO_TASK tasst;  int indent;
  if( DEBUG47(prev_day) )
  {
    indent = KheDrsResourceSetCount(included_free_resources) +
      included_free_resources_index + 1;
    fprintf(stderr, "%*s[ KheDrsAsstToTaskClassShiftAsstTrieBuildAssts(...)\n",
      indent * 2, "");
  }
  dtc = asst->task_class;
  if( dtc != NULL )
  {
    /* select a task from dtc and assign it */
    if( KheDrsTaskClassAcceptResourceBegin(dtc,
	asst->asst_to_shift->resource_on_day, &dtd) )
    {
      tasst = KheDrsAsstToTaskMake(asst, dtd);
      KheDrsAsstToTaskShiftAsstTrieBuildAssts(tasst, dsat, ds,
	included_free_resources, included_free_resources_index,
	prev_soln, prev_day, next_day, de);
      KheDrsTaskClassAcceptResourceEnd(dtc, dtd);
    }
  }
  else
  {
    /* use asst->fixed_task_on_day, possibly NULL meaning a free day */
    tasst = KheDrsAsstToTaskMake(asst, asst->fixed_task_on_day);
    KheDrsAsstToTaskShiftAsstTrieBuildAssts(tasst, dsat, ds,
      included_free_resources, included_free_resources_index,
      prev_soln, prev_day, next_day, de);
  }
  if( DEBUG47(prev_day) )
  {
    indent = KheDrsResourceSetCount(included_free_resources) +
      included_free_resources_index + 1;
    fprintf(stderr, "%*s] KheDrsAsstToTaskClassShiftAsstTrieBuildAssts ret.\n",
      indent * 2, "");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_ASST_TO_SHIFT"                                        */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_ASST_TO_SHIFT KheDrsAsstToShiftMake(KHE_DRS_RESOURCE_ON_DAY drd, */
/*    KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_SIGNATURE prev_sig,                   */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new asst to shift object, with the signature set.                 */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_ASST_TO_SHIFT KheDrsAsstToShiftMake(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_SIGNATURE prev_sig,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_ASST_TO_SHIFT res;
  if( HaArrayCount(drs->asst_to_shift_free_list) > 0 )
    res = HaArrayLastAndDelete(drs->asst_to_shift_free_list);
  else
    HaMake(res, drs->arena);
  res->resource_on_day = drd;
  res->used = false;
  res->sig = KheDrsSignatureMake(drs);
  KheDrsResourceSetSignature(res->sig, drd, dtd, prev_sig, drs);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsAsstToShiftFree(KHE_DRS_ASST_TO_SHIFT sasst,                  */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Free sasst.  This reduces the reference count of its signature by one,   */
/*  which may or may not cause it to be freed.                               */
/*                                                                           */
/*****************************************************************************/

static void KheDrsAsstToShiftFree(KHE_DRS_ASST_TO_SHIFT sasst,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KheDrsSignatureUnRefer(sasst->sig, drs);
  sasst->sig = NULL;
  HaArrayAddLast(drs->asst_to_shift_free_list, sasst);
}


/*****************************************************************************/
/*                                                                           */
/*  Major category "solutions"                                               */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_ASST_OP" - the type of an operation (unassign, etc.)  */
/*                                                                           */
/*****************************************************************************/

/* *** should use this but currently unused
static char *KheDrsAsstOpShow(KHE_DRS_ASST_OP op)
{
  switch( op )
  {
    case KHE_DRS_ASST_OP_UNASSIGN:	return "-";
    case KHE_DRS_ASST_OP_ASSIGN:	return "+";
    case KHE_DRS_ASST_OP_REPLACE:	return "-+";
    default:				return "?";
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_SOLN"                                                 */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SOLN KheDrsSolnMake(KHE_DRS_SOLN prev_soln,                      */
/*    KHE_COST cost, KHE_DYNAMIC_RESOURCE_SOLVER drs)                        */
/*                                                                           */
/*  Make a new soln with these attributes.                                   */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_SOLN KheDrsSolnMake(KHE_DRS_SOLN prev_soln,
  KHE_COST cost, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SOLN res;

  /* get the basic object */
  if( HaArrayCount(drs->soln_free_list) > 0 )
  {
    res = HaArrayLastAndDelete(drs->soln_free_list);
    KheDrsSignatureSetClear(&res->sig_set, cost, drs);  /* just sets cost */
    HaArrayClear(res->prev_tasks);
  }
  else
  {
    HaMake(res, drs->arena);
    HaArrayInit(res->prev_tasks, drs->arena);
    KheDrsSignatureSetInit(&res->sig_set, cost, drs->arena);
  }

  /* initialize the fields */
  res->prev_soln = prev_soln;
  res->priqueue_index = 0;  /* means never yet been in priqueue */
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnFree(KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs)  */
/*                                                                           */
/*  Free soln.  Don't worry about whether it is in the priqueue, or          */
/*  indeed in any container.  If it is, the caller will handle it.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnFree(KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  /* unrefer to all signatures; this may or may not free them */
  KheDrsSignatureSetClear(&soln->sig_set, 0, drs);

  /* put soln on the free list */
  HaArrayAddLast(drs->soln_free_list, soln);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnMarkExpanded(KHE_DRS_SOLN soln)                           */
/*                                                                           */
/*  Mark solution as expanded.                                               */
/*                                                                           */
/*  Note.  All solutions are created unexpanded.  At some point they may     */
/*  be expanded (so this function is called), after which they remain        */
/*  expanded until they are freed.                                           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnMarkExpanded(KHE_DRS_SOLN soln)
{
  soln->priqueue_index = -1;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSolnNotExpanded(KHE_DRS_SOLN soln)                            */
/*                                                                           */
/*  Return true if soln is not expanded.                                     */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsSolnNotExpanded(KHE_DRS_SOLN soln)
{
  return soln->priqueue_index != -1;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_COST KheDrsSolnCost(KHE_DRS_SOLN soln)                               */
/*                                                                           */
/*  Return the cost of soln.                                                 */
/*                                                                           */
/*****************************************************************************/

static KHE_COST KheDrsSolnCost(KHE_DRS_SOLN soln)
{
  return soln->sig_set.cost;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DAY KheDrsSolnDay(KHE_DRS_SOLN soln,                             */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Return the day that soln is for, or NULL if soln is the root solution.   */
/*                                                                           */
/*  Note.  Formerly we stored this as an attribute of soln.  We are now      */
/*  avoiding storing it, to save memory.  But searching for it, as we do     */
/*  here, is expensive and is only done when there is no better way.         */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DAY KheDrsSolnDay(KHE_DRS_SOLN soln,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_TASK_ON_DAY dtd;  int i;  KHE_DRS_DAY prev_day;

  /* return NULL if this is the root solution, not on any day */
  if( soln->prev_soln == NULL )
    return NULL;

  /* if prev_tasks has a non-NULL task on day, return its day */
  HaArrayForEach(soln->prev_tasks, dtd, i)
    if( dtd != NULL )
      return dtd->day;

  /* else have to recurse back */
  prev_day = KheDrsSolnDay(soln->prev_soln, drs);
  if( prev_day == NULL )
    return HaArray(drs->open_days, 0);
  else
    return HaArray(drs->open_days, prev_day->open_day_index + 1);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsSolnSignatureSetFullHash(KHE_DRS_SOLN soln)                    */
/*                                                                           */
/*  Hash function for hashing the full signature of soln.                    */
/*                                                                           */
/*****************************************************************************/

static int KheDrsSolnSignatureSetFullHash(KHE_DRS_SOLN soln)
{
  return KheDrsSignatureSetFullHash(&soln->sig_set);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsSolnSignatureSetFullHashUntyped(void *p)                       */
/*                                                                           */
/*  Untyped version of KheDrsSolnSignatureSetHash.                           */
/*                                                                           */
/*****************************************************************************/

static int KheDrsSolnSignatureSetFullHashUntyped(void *p)
{
  return KheDrsSolnSignatureSetFullHash((KHE_DRS_SOLN) p);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsSolnSignatureSetPartialHash(KHE_DRS_SOLN soln)                 */
/*                                                                           */
/*  Hash function for hashing the partial signature of soln, that is, only   */
/*  the parts of it whose dominance tests are equality (or loose equality).  */
/*                                                                           */
/*****************************************************************************/

static int KheDrsSolnSignatureSetPartialHash(KHE_DRS_SOLN soln,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DAY day;
  day = KheDrsSolnDay(soln, drs);
  return KheDrsSignerSetPartialHashSignature(day->signer_set, &soln->sig_set);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSolnSignatureSetFullEqual(KHE_DRS_SOLN soln1,                 */
/*    KHE_DRS_SOLN soln2)                                                    */
/*                                                                           */
/*  Return true if the signatures of soln1 and soln2 are equal.  The lengths */
/*  of their signatures must be equal.                                       */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsSolnSignatureSetFullEqual(KHE_DRS_SOLN soln1,
  KHE_DRS_SOLN soln2)
{
  return KheDrsSignatureSetFullEqual(&soln1->sig_set, &soln2->sig_set);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSolnSignatureSetFullEqualUntyped(void *p1, void *p2)          */
/*                                                                           */
/*  Untyped version of KheDrsSolnAssignmentsEqual.                           */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsSolnSignatureSetFullEqualUntyped(void *p1, void *p2)
{
  return KheDrsSolnSignatureSetFullEqual((KHE_DRS_SOLN) p1, (KHE_DRS_SOLN) p2);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSolnSignatureSetPartialEqual(KHE_DRS_SOLN soln1,              */
/*    KHE_DRS_SOLN soln2, KHE_DYNAMIC_RESOURCE_SOLVER drs)                   */
/*                                                                           */
/*  Return true if the signatures of soln1 and soln2 are partially equal,    */
/*  that is, the entries whose dominance tests are equality (or loose        */
/*  equality) are equal.                                                     */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsSolnSignatureSetPartialEqual(KHE_DRS_SOLN soln1,
  KHE_DRS_SOLN soln2, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DAY day;
  day = KheDrsSolnDay(soln1, drs);
  return KheDrsSignerSetPartialEqualSignature(day->signer_set,
    &soln1->sig_set, &soln2->sig_set);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSignatureSetDebug(KHE_DRS_SOLN soln, FILE *fp)            */
/*                                                                           */
/*  Debug print of the signature of soln onto fp (no verbosity or indent).   */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSignatureSetDebug(KHE_DRS_SOLN soln, FILE *fp)
{
  KheDrsSignatureSetDebug(&soln->sig_set, 1, -1, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSignatureSetDebugUntyped(void *p, FILE *fp)               */
/*                                                                           */
/*  Untyped version of KheDrsSolnSignatureSetDebug.                          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSignatureSetDebugUntyped(void *p, FILE *fp)
{
  KheDrsSolnSignatureSetDebug((KHE_DRS_SOLN) p, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsDistanceToAncestor(KHE_DRS_SOLN soln1, KHE_DRS_SOLN soln2)     */
/*                                                                           */
/*  Return the distance to the first common ancestor.  If soln1 == soln2     */
/*  this will be 0; otherwise, if they have the same parent solution, it     */
/*  will be 1; and so on.  The distance must be the same from each soln.     */
/*                                                                           */
/*****************************************************************************/

#if TESTING
static int KheDrsDistanceToAncestor(KHE_DRS_SOLN soln1, KHE_DRS_SOLN soln2,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int res;
  KHE_DRS_DAY day1, day2;
  day1 = KheDrsSolnDay(soln1, drs);
  day2 = KheDrsSolnDay(soln2, drs);
  if( DEBUG32 )
    fprintf(stderr, "  KheDrsDistanceToAncestor(%p day %s, %p day %s)\n",
      (void *) soln1, KheDrsDayId(day1), (void *) soln2, KheDrsDayId(day2));
  res = 0;
  while( soln1 != soln2 )
    soln1 = soln1->prev_soln, soln2 = soln2->prev_soln, res++;
  return res;
}
#endif


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSolnDoDominates(KHE_DRS_SOLN soln1, KHE_DRS_SOLN soln2,       */
/*    KHE_DRS_SIGNER_SET signer_set, KHE_COST trie_extra_cost,               */
/*    int trie_start_depth, int *dom_test_count,                             */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)  */
/*                                                                           */
/*  Return true if soln1 dominates soln2, compared using signer_set.         */
/*                                                                           */
/*  If the trie data structure is in use, trie_extra_cost has already been   */
/*  traded for soln1, so available cost has to be reduced by that much, and  */
/*  and we are already a position trie_start_depth.                          */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsSolnDoDominates(KHE_DRS_SOLN soln1, KHE_DRS_SOLN soln2,
  KHE_DRS_SIGNER_SET signer_set, KHE_COST trie_extra_cost,
  int trie_start_depth, int *dom_test_count,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)
{
  bool res;
  if( fp != NULL )
    fprintf(fp, "%*s[ KheDrsSolnDoDominates(soln1 %.5f, soln2 %.5f, day %s)\n",
      indent, "", KheCostShow(KheDrsSolnCost(soln1)),
      KheCostShow(KheDrsSolnCost(soln2)), KheDrsDayId(signer_set->day));
  else
    *dom_test_count += 1;
  res = KheDrsSignerSetDominates(signer_set, &soln1->sig_set,
    &soln2->sig_set, trie_extra_cost, trie_start_depth,
    soln1->prev_soln == soln2->prev_soln, drs, verbosity, indent, fp);
  if( fp != NULL )
  {
    KheDrsDominanceDebugSpacer(indent + 2, fp);
    fprintf(fp, "%*s] KheDrsSolnDoDominates returning %s\n", indent, "",
      bool_show(res));
  }
#if TESTING
  else if( res )
  {
    int dist = KheDrsDistanceToAncestor(soln1, soln2, drs);
    HaArrayFill(drs->ancestor_freq, dist + 1, 0);
    HaArray(drs->ancestor_freq, dist)++;
  }
#endif
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSolnDominates(KHE_DRS_SOLN soln1, KHE_DRS_SOLN soln2,         */
/*    KHE_DRS_SIGNER_SET signer_set, int *dom_test_count,                    */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Return true if soln1 dominates soln2.                                    */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsSolnDominates(KHE_DRS_SOLN soln1, KHE_DRS_SOLN soln2,
  KHE_DRS_SIGNER_SET signer_set, int *dom_test_count,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)
{
  if( DEBUG64 && strcmp(KheDrsDayId(signer_set->day), DEBUG64_ID) == 0 )
    verbosity = 2, indent = 2, fp = stderr;
  return KheDrsSolnDoDominates(soln1, soln2, signer_set, 0, 0,
    dom_test_count, drs, verbosity, indent, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnOverWrite(KHE_DRS_SOLN dest_soln, KHE_DRS_SOLN src_soln)  */
/*                                                                           */
/*  Overwrite src_soln onto dest_soln.  Their days and signatures are known  */
/*  to be equal, so those don't need to be touched; and priqueue_index has   */
/*  to be left unchanged.                                                    */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnOverWrite(KHE_DRS_SOLN dest_soln, KHE_DRS_SOLN src_soln)
{
  int i;
  dest_soln->prev_soln = src_soln->prev_soln;
  HaArrayClear(dest_soln->prev_tasks);
  HaArrayAppend(dest_soln->prev_tasks, src_soln->prev_tasks, i);
  dest_soln->sig_set.cost = KheDrsSolnCost(src_soln);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSolnResourceIsAssigned(KHE_DRS_SOLN soln,                     */
/*    KHE_DRS_RESOURCE dr, KHE_DRS_TASK_ON_DAY *dtd)                         */
/*                                                                           */
/*  If dr is assigned in soln->prev_tasks, return true and set *dtd to what  */
/*  it is assigned to; otherwise return false and set *dtd to NULL.          */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsSolnResourceIsAssigned(KHE_DRS_SOLN soln,
  KHE_DRS_RESOURCE dr, KHE_DRS_TASK_ON_DAY *dtd)
{
  if( soln->prev_soln == NULL )
  {
    /* this is the root solution, so there can be no assignment */
    return *dtd = NULL, false;
  }
  else
  {
    /* non-root solution, get assignment from soln->prev_tasks */
    *dtd = HaArray(soln->prev_tasks, dr->open_resource_index);
    return *dtd != NULL;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KhePrintCol(char *str, bool rule_at_right, FILE *fp)                */
/*                                                                           */
/*  Print str (truncated if required) onto fp in a column.  If rule_at_right */
/*  is true, include a vertical rule at the right.                           */
/*                                                                           */
/*****************************************************************************/

static void KhePrintCol(char *str, bool rule_at_right, FILE *fp)
{
  int i;  char *p;

  /* print initial margin */
  fprintf(fp, " ");

  /* print exactly COL_WIDTH - 3 characters, starting with str */
  for( i = 0, p = str;  i < COL_WIDTH - 3;  i++ )
  {
    if( *p != '\0' )
      fprintf(fp, "%c", *p), p++;
    else
      fprintf(fp, " ");
  }

  /* print final margin, optionally including a rule */
  fprintf(fp, rule_at_right ? " |" : "  ");
}


/*****************************************************************************/
/*                                                                           */
/*  void KhePrintRule(int col_count, int indent, FILE *fp)                   */
/*                                                                           */
/*  Print a rule, wide enough for col_count columns.                         */
/*                                                                           */
/*****************************************************************************/

static void KhePrintRule(int col_count, int indent, FILE *fp)
{
  int i, j;

  /* indent */
  fprintf(fp, "%*s", indent, "");

  /* header column */
  fprintf(fp, "%*s+", COL_WIDTH - 1, "");

  /* ordinary columns */
  for( i = 0;  i < col_count;  i++ )
  {
    for( j = 1;  j < COL_WIDTH;  j++ )
      fprintf(fp, "-");
    fprintf(fp, "+");
  }

  /* final newline */
  fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  char *ShortName(char *name, char *prefix)                                */
/*                                                                           */
/*  Return name with prefix removed from the front, if present.              */
/*                                                                           */
/*****************************************************************************/

static char *ShortName(char *name, char *prefix)
{
  if( strstr(name, prefix) != name )
    return name;
  else
  {
    name = &name[strlen(prefix)];
    if( name[0] == ':' )
      return &name[1];
    else
      return name;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnDebugResourceTimetable(KHE_DRS_RESOURCE dr,               */
/*    KHE_DRS_SOLN soln, int first_day_index, int last_day_index,            */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)  */
/*                                                                           */
/*  Debug print of the full timetable of dr from first_day_index to          */
/*  last_day_index inclusive.                                                */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnDebugResourceTimetable(KHE_DRS_RESOURCE dr,
  KHE_DRS_SOLN soln, int first_day_index, int last_day_index,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)
{
  KHE_DRS_SOLN soln2;  KHE_DRS_TASK_ON_DAY dtd;  KHE_DRS_RESOURCE_ON_DAY drd;
  int i;  KHE_DRS_DAY soln_day, day;

  /* indent */
  fprintf(fp, "%*s", indent, "");

  /* header column */
  KhePrintCol(KheDrsResourceId(dr), true, fp);

  /* ordinary columns */
  soln_day = KheDrsSolnDay(soln, drs);
  for( i = first_day_index;  i <= last_day_index;  i++ )
  {
    day = HaArray(drs->all_days, i);
    if( day->open_day_index == -1 )
    {
      /* a closed day, get timetable from dr */
      drd = HaArray(dr->days, i);
      dtd = drd->closed_asst;
      if( dtd == NULL /* || dtd->task == NULL */ )
        KhePrintCol("", true, fp);
      else
	KhePrintCol(ShortName(KheTaskId(dtd->task), KheDrsDayId(day)),
	  KheDrsTaskOnDayIsLast(dtd), fp);
    }
    else if( soln_day==NULL || day->open_day_index > soln_day->open_day_index )
    {
      /* an open day whose assignment is not yet known */
      KhePrintCol("?", true, fp);
    }
    else
    {
      /* an open day whose assignment is known */
      for( soln2 = soln;  soln2 != NULL;  soln2 = soln2->prev_soln )
	if( KheDrsSolnDay(soln2, drs) == day )
	  break;
      HnAssert(soln2 != NULL,
	"KheDrsSolnDebugResourceTimetable internal error 1");
      HnAssert(0 <= dr->open_resource_index &&
	dr->open_resource_index < HaArrayCount(soln2->prev_tasks),
	"KheDrsSolnDebugResourceTimetable internal error 2");
      dtd = HaArray(soln2->prev_tasks, dr->open_resource_index);
      if( dtd == NULL /* || dtd->task == NULL */ )
        KhePrintCol("", true, fp);
      else
	KhePrintCol(ShortName(KheTaskId(dtd->task), KheDrsDayId(day)),
	  KheDrsTaskOnDayIsLast(dtd), fp);
    }
  }

  /* final newline */
  fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnDebugTimetable(KHE_DRS_SOLN soln, int first_day_index,    */
/*    int last_day_index, KHE_DYNAMIC_RESOURCE_SOLVER drs,                   */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of the timetables of the open resources from first_day_index */
/*  to last_day_index inclusive.                                             */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnDebugTimetable(KHE_DRS_SOLN soln, int first_day_index,
  int last_day_index, KHE_DYNAMIC_RESOURCE_SOLVER drs,
  int verbosity, int indent, FILE *fp)
{
  int i;  KHE_DRS_DAY day;  KHE_DRS_RESOURCE dr;  char buff[200];

  /* header row and first rule */
  fprintf(fp, "%*s", indent + COL_WIDTH, "");
  for( i = first_day_index;  i <= last_day_index;  i++ )
  {
    day = HaArray(drs->all_days, i);
    if( day->open_day_index != -1 )
    {
      snprintf(buff, 200, "*%s*", KheDrsDayId(day));
      KhePrintCol(buff, false, fp);
    }
    else
      KhePrintCol(KheDrsDayId(day), false, fp);
  }
  fprintf(fp, "\n");
  KhePrintRule(last_day_index - first_day_index + 1, indent, fp);

  /* resource rows and other rules */
  KheDrsResourceSetForEach(drs->open_resources, dr, i)
  {
    KheDrsSolnDebugResourceTimetable(dr, soln, first_day_index, last_day_index,
      drs, verbosity, indent, fp);
    KhePrintRule(last_day_index - first_day_index + 1, indent, fp);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnDebug(KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of soln onto fp with the given verbosity and indent.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnDebug(KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs,
  int verbosity, int indent, FILE *fp)
{
  int i, j;
  for( i = 0;  i < HaArrayCount(drs->all_days) - 1;  i += COL_COUNT )
  {
    fprintf(fp, "\n");  /* even the first block has a preceding blank line */
    j = min(i + COL_COUNT - 1, HaArrayCount(drs->all_days) - 1);
    KheDrsSolnDebugTimetable(soln, i, j, drs, verbosity, indent, fp);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "One-extra and two-extra selection"                            */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsOneExtraSelectionDominates(                                   */
/*    KHE_DRS_ASST_TO_TASK_CLASS asst_ra_ci,                                 */
/*    KHE_DRS_ASST_TO_TASK_CLASS asst_ra_cj,                                 */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)  */
/*                                                                           */
/*  Return true if S + asst_ra_ci dominates S + asst_ra_cj.                  */
/*                                                                           */
/*  If fp != NULL, produce debug output while doing this.                    */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsOneExtraSelectionDominates(
  KHE_DRS_ASST_TO_TASK_CLASS asst_ra_ci,
  KHE_DRS_ASST_TO_TASK_CLASS asst_ra_cj,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)
{
  KHE_COST avail_cost;  int m;  bool res;
  KHE_DRS_RESOURCE ra;  KHE_DRS_TASK_CLASS dtc_ra_ci, dtc_ra_cj;
  ra = asst_ra_ci->asst_to_shift->resource_on_day->encl_dr;
  m = KheDrsResourceSetCount(drs->open_resources);
  avail_cost = 0;
  if( fp != NULL )
  {
    fprintf(fp, "%*s[ KheDrsOneExtraSelectionDominates(", indent, "");
    KheDrsAsstToTaskClassDebug(asst_ra_ci, 2, -1, fp);
    fprintf(fp, ", ");
    KheDrsAsstToTaskClassDebug(asst_ra_cj, 2, -1, fp);
    fprintf(fp, ", drs)\n");
  }
  dtc_ra_ci = asst_ra_ci->task_class;
  dtc_ra_cj = asst_ra_cj->task_class;
  res = KheDrsTaskClassOneExtraAvailable(dtc_ra_ci, m) &&
    KheDrsTaskClassMinCost(dtc_ra_ci, KHE_DRS_ASST_OP_UNASSIGN, ra,
      NULL, m, &avail_cost, verbosity, indent, fp) &&
    KheDrsTaskClassMinCost(dtc_ra_cj, KHE_DRS_ASST_OP_ASSIGN, ra,
	NULL, m, &avail_cost, verbosity, indent, fp) &&
    KheDrsSignerDominates(asst_ra_ci->asst_to_shift->resource_on_day->signer,
      KheDrsAsstToTaskClassSignature(asst_ra_ci),
      KheDrsAsstToTaskClassSignature(asst_ra_cj),
      &avail_cost, verbosity, indent, fp);
  if( fp != NULL )
    fprintf(stderr, "] KheDrsOneExtraSelectionDominates returning %s "
      "(avail_cost %.5f)\n", bool_show(res), KheCostShow(avail_cost));
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsOneExtraSelectionInit(KHE_DYNAMIC_RESOURCE_SOLVER drs)        */
/*                                                                           */
/*  Assuming that the open resources of drs have their assignments, set up   */
/*  those assignments for one-extra selection.  This involves deleting       */
/*  assignments which are dominated by other assignments.                    */
/*                                                                           */
/*****************************************************************************/

static void KheDrsOneExtraSelectionInit(KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int ri, i, j;  KHE_DRS_RESOURCE dra;  int verbosity, indent;  FILE *fp;
  KHE_DRS_ASST_TO_TASK_CLASS asst_ra_ci, asst_ra_cj;
  if( DEBUG43 )
    verbosity = 1, indent = 2, fp = stderr;
  else
    verbosity = 0, indent = 0, fp = NULL;
  if( fp != NULL )
    fprintf(fp, "[ KheDrsOneExtraSelectionInit(drs)\n");
  KheDrsResourceSetForEach(drs->open_resources, dra, ri)
    if( HaArrayCount(dra->expand_assts) >= 2 )
      HaArrayForEach(dra->expand_assts, asst_ra_ci, i)
      {
	for( j = i + 1;  j < HaArrayCount(dra->expand_assts);  j++ )
	{
	  /* for each distinct pair of assignments (asst_ra_ci, asst_ra_cj) */
	  asst_ra_cj = HaArray(dra->expand_assts, j);
	  if( KheDrsOneExtraSelectionDominates(asst_ra_ci, asst_ra_cj,
		drs, verbosity, indent, fp))
	  {
	    /* asst_ra_ci dominates asst_ra_cj, so delete asst_ra_cj */
	    HaArrayDeleteAndShift(dra->expand_assts, j);
	    KheDrsAsstToTaskClassFree(asst_ra_cj, drs);
	    j--;  /* and try the next asst_ra_cj */
	  }
	  else if( KheDrsOneExtraSelectionDominates(asst_ra_cj, asst_ra_ci,
		    drs, verbosity, indent, fp) )
	  {
	    /* asst_ra_cj dominates asst_ra_ci, so delete asst_ra_ci */
	    HaArrayDeleteAndShift(dra->expand_assts, i);
	    KheDrsAsstToTaskClassFree(asst_ra_ci, drs);
	    i--;
	    break;  /* and try the next asst_ra_ci */
	  }
	}
      }
  if( fp != NULL )
    fprintf(fp, "] KheDrsOneExtraSelectionInit\n");
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsResourceHasAssignment(KHE_DRS_RESOURCE dr,                    */
/*    KHE_DRS_ASST_TO_TASK_CLASS asst, KHE_DRS_ASST_TO_TASK_CLASS *res)      */
/*                                                                           */
/*  If dr contains an assignment which corresponds to asst, set *res to      */
/*  that assignment and return true, otherwise false.                        */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsResourceHasAssignment(KHE_DRS_RESOURCE dr,
  KHE_DRS_ASST_TO_TASK_CLASS asst, KHE_DRS_ASST_TO_TASK_CLASS *res)
{
  KHE_DRS_ASST_TO_TASK_CLASS a;  int i;  /* KHE_DRS_TASK_CLASS dtc1, dtc2; */

  if( asst->task_class != NULL )
  {
    /* Case 1: task_class != NULL */
    HaArrayForEach(dr->expand_assts, a, i)
      if( a->task_class == asst->task_class )
	return *res = a, true;
    return *res = NULL, false;
  }
  else if( asst->fixed_task_on_day != NULL )
  {
    /* Case 2: task_class == NULL && fixed_task_on_day != NULL */
    return *res = NULL, false;
  }
  else
  {
    /* Case 3: task_class == NULL && fixed_task_on_day == NULL */
    HaArrayForEach(dr->expand_assts, a, i)
      if( a->task_class == NULL && a->fixed_task_on_day == NULL )
	return *res = a, true;
    return *res = NULL, false;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsTwoExtraSelectionDominates(                                   */
/*    KHE_DRS_ASST_TO_TASK_CLASS asst_ra_ci,                                 */
/*    KHE_DRS_ASST_TO_TASK_CLASS asst_rb_cj,                                 */
/*    KHE_DRS_ASST_TO_TASK_CLASS asst_ra_cj,                                 */
/*    KHE_DRS_ASST_TO_TASK_CLASS asst_rb_ci,                                 */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)  */
/*                                                                           */
/*  Test S + asst_ra_ci + asst_rb_cj dominates S + asst_ra_cj + asst_rb_ci.  */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsTwoExtraSelectionDominates(
  KHE_DRS_ASST_TO_TASK_CLASS asst_ra_ci,
  KHE_DRS_ASST_TO_TASK_CLASS asst_rb_cj,
  KHE_DRS_ASST_TO_TASK_CLASS asst_ra_cj,
  KHE_DRS_ASST_TO_TASK_CLASS asst_rb_ci,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)
{
  KHE_COST avail_cost;  KHE_DRS_RESOURCE dra, drb;  int m;
  KHE_DRS_TASK_CLASS dtc_ra_ci, dtc_rb_cj;  bool res;

  if( fp != NULL )
  {
    fprintf(fp, "%*s[ KheDrsTwoExtraSelectionDominates(...)\n", indent, "");
    fprintf(fp, "%*s  asst_ra_ci: ", indent, "");
    KheDrsAsstToTaskClassDebug(asst_ra_ci, 2, 0, fp);
    fprintf(fp, "%*s  asst_rb_cj: ", indent, "");
    KheDrsAsstToTaskClassDebug(asst_rb_cj, 2, 0, fp);
    fprintf(fp, "%*s  asst_ra_cj: ", indent, "");
    KheDrsAsstToTaskClassDebug(asst_ra_cj, 2, 0, fp);
    fprintf(fp, "%*s  asst_rb_ci: ", indent, "");
    KheDrsAsstToTaskClassDebug(asst_rb_ci, 2, 0, fp);
  }

  dra = asst_ra_ci->asst_to_shift->resource_on_day->encl_dr;
  drb = asst_rb_cj->asst_to_shift->resource_on_day->encl_dr;
  m = KheDrsResourceSetCount(drs->open_resources);
  avail_cost = 0;
  dtc_ra_ci = asst_ra_ci->task_class;
  dtc_rb_cj = asst_rb_cj->task_class;
  res =
    KheDrsTaskClassMinCost(dtc_ra_ci, KHE_DRS_ASST_OP_REPLACE,
      dra, drb, m, &avail_cost, verbosity, indent, fp) &&
    KheDrsTaskClassMinCost(dtc_rb_cj, KHE_DRS_ASST_OP_REPLACE,
      drb, dra, m, &avail_cost, verbosity, indent, fp) &&
    KheDrsSignerDominates(asst_ra_ci->asst_to_shift->resource_on_day->signer,
      KheDrsAsstToTaskClassSignature(asst_ra_ci),
      KheDrsAsstToTaskClassSignature(asst_ra_cj),
      &avail_cost, verbosity, indent, fp) &&
    KheDrsSignerDominates(asst_rb_ci->asst_to_shift->resource_on_day->signer,
      KheDrsAsstToTaskClassSignature(asst_rb_cj),
      KheDrsAsstToTaskClassSignature(asst_rb_ci),
      &avail_cost, verbosity, indent, fp);

  if( fp != NULL )
    fprintf(fp, "%*s] KheDrsTwoExtraSelectionDominatesDebug returning %s\n",
      indent, "", bool_show(res));
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsTwoExtraSelectionInit(KHE_DYNAMIC_RESOURCE_SOLVER drs)        */
/*                                                                           */
/*  Assuming that the open resources of drs have their assignments, set      */
/*  up those assignments for two-extra dominance.                            */
/*                                                                           */
/*****************************************************************************/

static void KheDrsTwoExtraSelectionInit(KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int a, b, i, j, verbosity, indent;  KHE_DRS_RESOURCE dra, drb;  FILE *fp;
  KHE_DRS_ASST_TO_TASK_CLASS asst_ra_ci, asst_ra_cj, asst_rb_ci, asst_rb_cj;
  if( DEBUG35 )
    verbosity = 1, indent = 2, fp = stderr;
  else
    verbosity = 0, indent = 0, fp = NULL;
  if( fp != NULL )
    fprintf(fp, "%*s[ KheDrsTwoExtraSelectionInit(drs)\n", indent, "");
  KheDrsResourceSetForEach(drs->open_resources, dra, a)
    if( HaArrayCount(dra->expand_assts) >= 2 )
      for( b = a + 1;  b < KheDrsResourceSetCount(drs->open_resources);  b++ )
      {
	drb = KheDrsResourceSetResource(drs->open_resources, b);
	if( HaArrayCount(drb->expand_assts) >= 2 )
	  HaArrayForEach(dra->expand_assts, asst_ra_ci, i)
	  {
	    if( KheDrsResourceHasAssignment(drb, asst_ra_ci, &asst_rb_ci) )
	    {
	      for( j = i + 1;  j < HaArrayCount(dra->expand_assts);  j++ )
	      {
		asst_ra_cj = HaArray(dra->expand_assts, j);
		if( KheDrsResourceHasAssignment(drb, asst_ra_cj, &asst_rb_cj) )
		{
		  if( KheDrsTwoExtraSelectionDominates(asst_ra_ci, asst_rb_cj,
			asst_ra_cj, asst_rb_ci, drs, verbosity, indent, fp) )
		  {
		    /* S + asst_ra_cj + asst_rb_ci is dominated */
		    HaArrayAddLast(asst_ra_cj->skip_assts, asst_rb_ci);
		    HaArrayAddLast(asst_rb_ci->skip_assts, asst_ra_cj);
		  }
		  else if( KheDrsTwoExtraSelectionDominates(asst_ra_cj,
		    asst_rb_ci, asst_ra_ci,asst_rb_cj,drs,verbosity,indent,fp))
		  {
		    /* S + asst_ra_ci + asst_rb_cj is dominated */
		    HaArrayAddLast(asst_ra_ci->skip_assts, asst_rb_cj);
		    HaArrayAddLast(asst_rb_cj->skip_assts, asst_ra_ci);
		  }
		}
	      }
	    }
	  }
      }
  if( fp != NULL )
    fprintf(stderr, "%*s] KheDrsTwoExtraSelectionInit\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_PACKED_SOLN" (solutions in an easier format)          */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_PACKED_SOLN_DAY KhePackedSolnDayMakeEmpty(KHE_DRS_DAY day,       */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make an empty packed solution day object for day.                        */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_PACKED_SOLN_DAY KhePackedSolnDayMakeEmpty(KHE_DRS_DAY day,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_PACKED_SOLN_DAY res;
  if( HaArrayCount(drs->packed_soln_day_free_list) > 0 )
  {
    res = HaArrayLastAndDelete(drs->packed_soln_day_free_list);
    HaArrayClear(res->prev_tasks);
  }
  else
  {
    HaMake(res, drs->arena);
    HaArrayInit(res->prev_tasks, drs->arena);
  }
  res->day = day;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_PACKED_SOLN_DAY KheDrsPackedSolnDayMakeFromSoln(                 */
/*    KHE_DRS_SOLN soln, KHE_DRS_DAY day, KHE_DYNAMIC_RESOURCE_SOLVER drs)   */
/*                                                                           */
/*  Make a packed solution day object from soln.                             */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_PACKED_SOLN_DAY KheDrsPackedSolnDayMakeFromSoln(
  KHE_DRS_SOLN soln, KHE_DRS_DAY day, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int i;  KHE_DRS_PACKED_SOLN_DAY res;
  res = KhePackedSolnDayMakeEmpty(day, drs);
  HaArrayAppend(res->prev_tasks, soln->prev_tasks, i);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_PACKED_SOLN KheDrsPackedSolnMake(KHE_COST cost,                  */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new, empty packed solution object with this cost.                 */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_PACKED_SOLN KheDrsPackedSolnMake(KHE_COST cost,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_PACKED_SOLN res;
  if( DEBUG66 )
    fprintf(stderr, "KheDrsPackedSolnMake(%.5f, drs)\n", KheCostShow(cost));
  if( HaArrayCount(drs->packed_soln_free_list) > 0 )
  {
    res = HaArrayLastAndDelete(drs->packed_soln_free_list);
    HaArrayClear(res->days);
  }
  else
  {
    HaMake(res, drs->arena);
    HaArrayInit(res->days, drs->arena);
  }
  res->cost = cost;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsPackedSolnDelete(KHE_DRS_PACKED_SOLN packed_soln,             */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Delete packed_soln.                                                      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsPackedSolnDelete(KHE_DRS_PACKED_SOLN packed_soln,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int i;
  HaArrayAppend(drs->packed_soln_day_free_list, packed_soln->days, i);
  HaArrayAddLast(drs->packed_soln_free_list, packed_soln);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_PACKED_SOLN KheDrsPackedSolnBuildEmpty(                          */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Build an empty packed soln object for the open days and resources of     */
/*  drs, and cost KheSolnCost(drs->soln).                                    */
/*                                                                           */
/*  Implementation note.  This function is called early during opening,      */
/*  when only selected_resource_set and selected_day_ranges are defined.     */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_PACKED_SOLN KheDrsPackedSolnBuildEmpty(
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_PACKED_SOLN res;  KHE_DRS_PACKED_SOLN_DAY rd;
  KHE_DRS_DAY_RANGE ddr;  KHE_DRS_DAY day;  int i, j, count;

  res = KheDrsPackedSolnMake(KheSolnCost(drs->soln), drs);
  count = KheResourceSetResourceCount(drs->selected_resource_set);
  HaArrayForEach(drs->selected_day_ranges, ddr, i)
    for( j = ddr.first;  j <= ddr.last;  j++ )
    {
      /* make a day for j */
      day = HaArray(drs->all_days, j);
      rd = KhePackedSolnDayMakeEmpty(day, drs);
      HaArrayFill(rd->prev_tasks, count, NULL);
      HaArrayAddLast(res->days, rd);
    }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_PACKED_SOLN KheDrsPackedSolnBuildFromSoln(KHE_DRS_SOLN soln,     */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Build a packed soln object from soln.                                    */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_PACKED_SOLN KheDrsPackedSolnBuildFromSoln(KHE_DRS_SOLN soln,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_PACKED_SOLN res;  KHE_DRS_SOLN soln2;  KHE_DRS_PACKED_SOLN_DAY rd;
  int i, j, open_day_index;  KHE_DRS_DAY day;

  /* build the packed soln but with the days in reverse order */
  res = KheDrsPackedSolnMake(KheDrsSolnCost(soln), drs);
  open_day_index = HaArrayCount(drs->open_days) - 1;
  for( soln2 = soln;  soln2->prev_soln != NULL;  soln2 = soln2->prev_soln )
  {
    day = HaArray(drs->open_days, open_day_index);
    rd = KheDrsPackedSolnDayMakeFromSoln(soln2, day, drs);
    HaArrayAddLast(res->days, rd);
    open_day_index--;
  }

  /* reverse the order of the days and return */
  for( i = 0, j = HaArrayCount(res->days) - 1;  i < j;  i++, j-- )
    HaArraySwap(res->days, i, j, rd);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_TASK_ON_DAY KheDrsPackedSolnTaskOnDay(                           */
/*    KHE_DRS_PACKED_SOLN packed_soln, int open_day_index,                   */
/*    int open_resource_index)                                               */
/*                                                                           */
/*  Return the task on day object (possibly NULL) held in packed_soln for    */
/*  the given open day index and open resource index.                        */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_TASK_ON_DAY KheDrsPackedSolnTaskOnDay(
  KHE_DRS_PACKED_SOLN packed_soln, KHE_DRS_DAY open_day,
  KHE_DRS_RESOURCE open_resource)
{
  KHE_DRS_PACKED_SOLN_DAY rd;  int open_day_index, open_resource_index;
  open_day_index = open_day->open_day_index;
  HnAssert(0 <= open_day_index &&
    open_day_index < HaArrayCount(packed_soln->days),
    "KheDrsPackedSolnTaskOnDay internal error (day %d out of range 0 .. %d)",
    open_day_index, HaArrayCount(packed_soln->days) - 1);
  rd = HaArray(packed_soln->days, open_day_index);
  open_resource_index = open_resource->open_resource_index;
  HnAssert(0 <= open_resource_index &&
      open_resource_index < HaArrayCount(rd->prev_tasks),
   "KheDrsPackedSolnTaskOnDay internal error (resource %d out of range 0 .. %d",
    open_resource_index, HaArrayCount(rd->prev_tasks) - 1);
  return HaArray(rd->prev_tasks, open_resource_index);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsPackedSolnSetTaskOnDay(KHE_DRS_PACKED_SOLN packed_soln,       */
/*    int open_day_index, int open_resource_index, KHE_DRS_TASK_ON_DAY dtd)  */
/*                                                                           */
/*  Set the task on day object (possibly NULL) held in packed_soln for       */
/*  the given open day index and open resource index.                        */
/*                                                                           */
/*****************************************************************************/

static void KheDrsPackedSolnSetTaskOnDay(KHE_DRS_PACKED_SOLN packed_soln,
  int open_day_index, int open_resource_index, KHE_DRS_TASK_ON_DAY dtd)
{
  KHE_DRS_PACKED_SOLN_DAY rd;
  HnAssert(0 <= open_day_index &&
    open_day_index < HaArrayCount(packed_soln->days),
    "KheDrsPackedSolnSetTaskOnDay internal error (day %d out of range 0 .. %d)",
    open_day_index, HaArrayCount(packed_soln->days) - 1);
  rd = HaArray(packed_soln->days, open_day_index);
  HnAssert(0 <= open_resource_index && open_resource_index <
    HaArrayCount(rd->prev_tasks), "KheDrsPackedSolnSetTaskOnDay "
    "internal error (resource %d out of range 0 .. %d)",
    open_resource_index, HaArrayCount(rd->prev_tasks) - 1);
  HaArrayPut(rd->prev_tasks, open_resource_index, dtd);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsPackedSolnDebugResourceTimetable(KHE_DRS_RESOURCE dr,         */
/*    KHE_DRS_PACKED_SOLN packed_soln, int first_day_index, last_day_index,  */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)  */
/*                                                                           */
/*  Debug print of the full timetable of dr from first_day_index to          */
/*  last_day_index inclusive.                                                */
/*                                                                           */
/*****************************************************************************/

static void KheDrsPackedSolnDebugResourceTimetable(KHE_DRS_RESOURCE dr,
  KHE_DRS_PACKED_SOLN packed_soln, int first_day_index, int last_day_index,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)
{
  KHE_DRS_TASK_ON_DAY dtd;  KHE_DRS_RESOURCE_ON_DAY drd;
  int i;  KHE_DRS_DAY day;  KHE_DRS_PACKED_SOLN_DAY rd;

  /* indent */
  fprintf(fp, "%*s", indent, "");

  /* header column */
  KhePrintCol(KheDrsResourceId(dr), true, fp);

  /* ordinary columns */
  for( i = first_day_index;  i <= last_day_index;  i++ )
  {
    day = HaArray(drs->all_days, i);
    if( day->open_day_index == -1 )
    {
      /* a closed day, get timetable from dr */
      drd = HaArray(dr->days, i);
      dtd = drd->closed_asst;
      if( dtd == NULL /* || dtd->task == NULL */ )
        KhePrintCol("", true, fp);
      else
	KhePrintCol(ShortName(KheTaskId(dtd->task), KheDrsDayId(day)),
	  KheDrsTaskOnDayIsLast(dtd), fp);
    }
    else
    {
      /* an open day whose assignment is known */
      rd = HaArray(packed_soln->days, day->open_day_index);
      HnAssert(0 <= dr->open_resource_index &&
	dr->open_resource_index < HaArrayCount(rd->prev_tasks),
	"KheDrsPackedSolnDebugResourceTimetable internal error 2");
      dtd = HaArray(rd->prev_tasks, dr->open_resource_index);
      if( dtd == NULL /* || dtd->task == NULL */ )
        KhePrintCol("", true, fp);
      else
	KhePrintCol(ShortName(KheTaskId(dtd->task), KheDrsDayId(day)),
	  KheDrsTaskOnDayIsLast(dtd), fp);
    }
  }

  /* final newline */
  fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsPackedSolnDebugTimetable(KHE_DRS_PACKED_SOLN packed_soln,     */
/*    int first_day_index, int last_day_index,                               */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)  */
/*                                                                           */
/*  Debug print of the timetables of the open resources from first_day_index */
/*  to last_day_index inclusive.                                             */
/*                                                                           */
/*****************************************************************************/

static void KheDrsPackedSolnDebugTimetable(KHE_DRS_PACKED_SOLN packed_soln,
  int first_day_index, int last_day_index,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)
{
  int i;  KHE_DRS_DAY day;  KHE_DRS_RESOURCE dr;  char buff[200];

  /* header row and first rule */
  fprintf(fp, "%*s", indent, "");
  if( first_day_index == 0 )
    fprintf(fp, "%*.5f  ", COL_WIDTH - 2, KheCostShow(packed_soln->cost));
  else
    fprintf(fp, "%*s", COL_WIDTH, "");
  for( i = first_day_index;  i <= last_day_index;  i++ )
  {
    day = HaArray(drs->all_days, i);
    if( day->open_day_index != -1 )
    {
      snprintf(buff, 200, "*%s*", KheDrsDayId(day));
      KhePrintCol(buff, false, fp);
    }
    else
      KhePrintCol(KheDrsDayId(day), false, fp);
  }
  fprintf(fp, "\n");
  KhePrintRule(last_day_index - first_day_index + 1, indent, fp);

  /* resource rows and other rules */
  KheDrsResourceSetForEach(drs->open_resources, dr, i)
  {
    KheDrsPackedSolnDebugResourceTimetable(dr, packed_soln,
      first_day_index, last_day_index, drs, verbosity, indent, fp);
    KhePrintRule(last_day_index - first_day_index + 1, indent, fp);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsPackedSolnDebug(KHE_DRS_PACKED_SOLN packed_soln, char *str,   */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)  */
/*                                                                           */
/*  Debug print of packed_soln in timetable format.                          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsPackedSolnDebug(KHE_DRS_PACKED_SOLN packed_soln, char *str,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)
{
  int i, j;
  if( str != NULL && str[0] != '\0' )
    fprintf(fp, "%*s%s:\n", indent, "", str);
  for( i = 0;  i < HaArrayCount(drs->all_days) - 1;  i += COL_COUNT )
  {
    fprintf(fp, "\n");  /* even the first block has a preceding blank line */
    j = min(i + COL_COUNT - 1, HaArrayCount(drs->all_days) - 1);
    KheDrsPackedSolnDebugTimetable(packed_soln, i, j, drs, verbosity,
      indent, fp);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Major category "expansion"                                               */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPANDER"                                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExpanderSetOpen(KHE_DRS_EXPANDER de)                          */
/*                                                                           */
/*  Set de's open flag.  It's a function of the other attributes.            */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExpanderSetOpen(KHE_DRS_EXPANDER de)
{
  de->open = de->cost < de->cost_limit &&
    de->free_resource_count >= de->must_assign_count;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExpanderReset(KHE_DRS_EXPANDER de, bool whole_tasks,          */
/*    KHE_COST cost, KHE_COST cost_limit, int free_resource_count,           */
/*    int must_assign_count)                                                 */
/*                                                                           */
/*  Reset de.  This is like creating a new expander, only it recycles an     */
/*  existing one.                                                            */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExpanderReset(KHE_DRS_EXPANDER de, bool whole_tasks,
  KHE_COST cost, KHE_COST cost_limit, int free_resource_count,
  int must_assign_count)
{
  de->whole_tasks = whole_tasks;
  de->cost = cost;
  de->cost_limit = cost_limit;
  de->free_resource_count = free_resource_count;
  de->must_assign_count = must_assign_count;
  KheDrsExpanderSetOpen(de);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPANDER KheDrsExpanderMake(bool whole_tasks, KHE_COST cost,     */
/*    KHE_COST cost_limit, int free_resource_count, int must_assign_count,   */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make and return a new expander object with these attributes.             */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPANDER KheDrsExpanderMake(bool whole_tasks, KHE_COST cost,
  KHE_COST cost_limit, int free_resource_count, int must_assign_count,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPANDER res;

  /* get an expander from scratch or from the free list */
  if( HaArrayCount(drs->expander_free_list) > 0 )
  {
    res = HaArrayLastAndDelete(drs->expander_free_list);
    HaArrayClear(res->assts_to_tasks);
    HaArrayClear(res->tmp_assts_to_tasks);
    HaArrayClear(res->marks);
  }
  else
  {
    HaMake(res, drs->arena);
    HaArrayInit(res->assts_to_tasks, drs->arena);
    HaArrayInit(res->tmp_assts_to_tasks, drs->arena);
    HaArrayInit(res->marks, drs->arena);
  }

  /* initialize its fields and return it */
  res->solver = drs;
  KheDrsExpanderReset(res, whole_tasks, cost, cost_limit,
    free_resource_count, must_assign_count);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExpanderFree(KHE_DRS_EXPANDER de)                             */
/*                                                                           */
/*  Free de.                                                                 */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExpanderFree(KHE_DRS_EXPANDER de)
{
  HnAssert(HaArrayCount(de->assts_to_tasks) == 0,
    "KheDrsExpanderFree internal error 1");
  HnAssert(HaArrayCount(de->marks) == 0,
    "KheDrsExpanderFree internal error 2");
  HaArrayAddLast(de->solver->expander_free_list, de);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExpanderAddCost(KHE_DRS_EXPANDER de, KHE_COST cost)           */
/*                                                                           */
/*  Add cost to expander's cost.                                             */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExpanderAddCost(KHE_DRS_EXPANDER de, KHE_COST cost)
{
  de->cost += cost;
  KheDrsExpanderSetOpen(de);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExpanderReduceCostLimit(KHE_DRS_EXPANDER de,                  */
/*    KHE_COST cost_limit)                                                   */
/*                                                                           */
/*  Change de's cost limit to cost_limit, but only if that would be a        */
/*  reduction.  This could cause the expander to close, but only after       */
/*  the solution that led to this cost limit reduction has been finalized,   */
/*  so there is no danger that that solution will be excluded by this call.  */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExpanderReduceCostLimit(KHE_DRS_EXPANDER de,
  KHE_COST cost_limit)
{
  if( cost_limit < de->cost_limit )
  {
    de->cost_limit = cost_limit;
    KheDrsExpanderSetOpen(de);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsExpanderOpenToExtraCost(KHE_DRS_EXPANDER de,                  */
/*    KHE_COST extra_cost)                                                   */
/*                                                                           */
/*  Return true if increasing de's cost by extra_cost would not reach        */
/*  the limit.  Don't actually change de->cost.                              */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsExpanderOpenToExtraCost(KHE_DRS_EXPANDER de,
  KHE_COST extra_cost)
{
  return de->cost + extra_cost < de->cost_limit;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExpanderAddMustAssign(KHE_DRS_EXPANDER de)                    */
/*                                                                           */
/*  Inform de that one must-assign task has been found.                      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExpanderAddMustAssign(KHE_DRS_EXPANDER de)
{
  de->must_assign_count++;
  KheDrsExpanderSetOpen(de);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExpanderDeleteFreeResource(KHE_DRS_EXPANDER de)               */
/*                                                                           */
/*  Inform de that one of its free resources has become fixed.               */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExpanderDeleteFreeResource(KHE_DRS_EXPANDER de)
{
  de->free_resource_count--;
  KheDrsExpanderSetOpen(de);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExpanderExcessFreeResourceCount(KHE_DRS_EXPANDER de)           */
/*                                                                           */
/*  Return the amount by which the number of free resources exceeds          */
/*  the number of must-assign tasks.                                         */
/*                                                                           */
/*****************************************************************************/

static int KheDrsExpanderExcessFreeResourceCount(KHE_DRS_EXPANDER de)
{
  return de->free_resource_count - de->must_assign_count;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExpanderAddAsstToTask(KHE_DRS_EXPANDER de,                    */
/*    KHE_DRS_ASST_TO_TASK tasst)                                            */
/*                                                                           */
/*  Add tasst to the expansion recorded in de, or change nothing if there    */
/*  is a problem.                                                            */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExpanderAddAsstToTask(KHE_DRS_EXPANDER de,
  KHE_DRS_ASST_TO_TASK tasst)
{
  KHE_DRS_SIGNATURE sig;  KHE_DRS_TASK dt;  KHE_DRS_ASST_TO_TASK_CLASS asst;
  int i;  KHE_COST cost;  int must_assign_count, free_resource_count;
  KHE_DRS_RESOURCE dr;  KHE_DRS_TASK_ON_DAY dtd;

  /* if not open, do nothing */
  if( !de->open )
    return;

  /* if there is a skip count problem, close and do nothing */
  if( tasst.asst->skip_count > 0 )
  {
    de->open = false;
    return;
  }

  /* find new must_assign_count, free_resource_count, and cost values */
  must_assign_count = de->must_assign_count;
  free_resource_count = de->free_resource_count;
  cost = de->cost;
  if( tasst.fixed_dtd != NULL )
  {
    dt = tasst.fixed_dtd->encl_dt;
    if( tasst.fixed_dtd == HaArrayFirst(dt->days) )
      cost += dt->asst_cost;
    if( dt->expand_role == KHE_DRS_TASK_EXPAND_MUST )
      must_assign_count--;
  }
  sig = KheDrsAsstToTaskClassSignature(tasst.asst);
  cost += sig->cost;
  dr = KheDrsAsstToTaskResource(tasst);
  if( dr->expand_role == KHE_DRS_RESOURCE_EXPAND_FREE )
    free_resource_count--;

  /* if there is a problem with the values just found, close and return */
  if( cost >= de->cost_limit || free_resource_count < must_assign_count )
  {
    de->open = false;
    return;
  }

  /* no problems with the addition; change the state of de */
  de->must_assign_count = must_assign_count;
  de->free_resource_count = free_resource_count;
  de->cost = cost;
  HaArrayAddLast(de->assts_to_tasks, tasst);

  /* increment the skip counts of the skip_assts */
  HaArrayForEach(tasst.asst->skip_assts, asst, i)
    asst->skip_count++;

  /* update dtd's leaf expressions */
  if( tasst.fixed_dtd != NULL )
  {
    dr = tasst.asst->asst_to_shift->resource_on_day->encl_dr;
    if( de->whole_tasks )
    {
      dt = tasst.fixed_dtd->encl_dt;
      HaArrayForEach(dt->days, dtd, i)
	KheDrsTaskOnDayLeafSet(dtd, dr, de->solver);
    }
    else
      KheDrsTaskOnDayLeafSet(tasst.fixed_dtd, dr, de->solver);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExpanderAddAsstToTaskSet(KHE_DRS_EXPANDER de,                 */
/*    KHE_DRS_ASST_TO_TASK_SET dats, KHE_DRS_SHIFT ds)                       */
/*                                                                           */
/*  Add the assignment to task objects of dats to de.                        */
/*                                                                           */
/*  If ds != NULL, just add those for shift ds.                              */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExpanderAddAsstToTaskSet(KHE_DRS_EXPANDER de,
  KHE_DRS_ASST_TO_TASK_SET dats, KHE_DRS_SHIFT ds)
{
  KHE_DRS_ASST_TO_TASK dat;  int i;
  if( ds != NULL )
  {
    HaArrayForEach(dats->assts_to_tasks, dat, i)
      if( KheDrsAsstToTaskShift(dat) == ds )
	KheDrsExpanderAddAsstToTask(de, dat);
  }
  else
    HaArrayForEach(dats->assts_to_tasks, dat, i)
      KheDrsExpanderAddAsstToTask(de, dat);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExpanderDoDeleteAsstToTask(KHE_DRS_EXPANDER de,               */
/*    KHE_DRS_ASST_TO_TASK tasst)                                            */
/*                                                                           */
/*  Do the deletion of tasst from de, assuming that tasst has already        */
/*  been removed from de->assts_to_tasks.                                    */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExpanderDoDeleteAsstToTask(KHE_DRS_EXPANDER de,
  KHE_DRS_ASST_TO_TASK tasst)
{
  KHE_DRS_SIGNATURE sig;  KHE_DRS_TASK dt;  KHE_DRS_RESOURCE dr;
  int i;  KHE_DRS_TASK_ON_DAY dtd;  KHE_DRS_ASST_TO_TASK_CLASS asst;

  /* update dtd's leaf expressions */
  if( tasst.fixed_dtd != NULL )
  {
    if( de->whole_tasks )
    {
      dt = tasst.fixed_dtd->encl_dt;
      HaArrayForEach(dt->days, dtd, i)
	KheDrsTaskOnDayLeafClear(dtd, de->solver);
    }
    else
      KheDrsTaskOnDayLeafClear(tasst.fixed_dtd, de->solver);
  }

  /* decrement the skip counts of the skip_assts */
  HaArrayForEach(tasst.asst->skip_assts, asst, i)
    asst->skip_count--;

  /* update cost, must_assign_count, and free_resource_count fields */
  if( tasst.fixed_dtd != NULL )
  {
    dt = tasst.fixed_dtd->encl_dt;
    if( tasst.fixed_dtd == HaArrayFirst(dt->days) )
      de->cost -= dt->asst_cost;
    if( dt->expand_role == KHE_DRS_TASK_EXPAND_MUST )
      de->must_assign_count++;
  }
  sig = KheDrsAsstToTaskClassSignature(tasst.asst);
  de->cost -= sig->cost;
  dr = KheDrsAsstToTaskResource(tasst);
  if( dr->expand_role == KHE_DRS_RESOURCE_EXPAND_FREE )
    de->free_resource_count++;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExpanderMarkBegin(KHE_DRS_EXPANDER de)                        */
/*                                                                           */
/*  Begin a mark.                                                            */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExpanderMarkBegin(KHE_DRS_EXPANDER de)
{
  HaArrayAddLast(de->marks, HaArrayCount(de->assts_to_tasks));
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExpanderMarkEnd(KHE_DRS_EXPANDER de)                          */
/*                                                                           */
/*  End a mark.                                                              */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExpanderMarkEnd(KHE_DRS_EXPANDER de)
{
  int prev_count;  KHE_DRS_ASST_TO_TASK tasst;
  HnAssert(HaArrayCount(de->marks) > 0, "KheDrsExpanderMarkEnd internal error");
  prev_count = HaArrayLastAndDelete(de->marks);
  while( HaArrayCount(de->assts_to_tasks) > prev_count )
  {
    tasst = HaArrayLastAndDelete(de->assts_to_tasks);
    KheDrsExpanderDoDeleteAsstToTask(de, tasst);
  }
  KheDrsExpanderSetOpen(de);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsExpanderIsOpen(KHE_DRS_EXPANDER de)                           */
/*                                                                           */
/*  Return true if de is open, that is, if carrying on from here could       */
/*  produce something useful.                                                */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsExpanderIsOpen(KHE_DRS_EXPANDER de)
{
  return de->open;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExpanderMakeAndMeldSoln(KHE_DRS_EXPANDER de,                  */
/*    KHE_DRS_SOLN prev_soln, KHE_DRS_DAY next_day)                          */
/*                                                                           */
/*  Use the assignments in de to make a new solution object, then, if it     */
/*  is competitive, meld it into next_day's solution set.                    */
/*                                                                           */
/*****************************************************************************/
static void KheDrsSolnSetMeldSoln(KHE_DRS_SOLN_SET soln_set, KHE_DRS_SOLN soln,
  KHE_DRS_DAY soln_day, KHE_DRS_EXPANDER de, KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsExpanderMakeAndMeldSoln(KHE_DRS_EXPANDER de,
  KHE_DRS_SOLN prev_soln, KHE_DRS_DAY next_day)
{
  KHE_DRS_SOLN next_soln;  KHE_DRS_ASST_TO_TASK tasst, junk;  int i, ri;
  KHE_DRS_SIGNATURE sig, prev_sig;  KHE_DRS_SIGNER dsg;

  /* consistency checks */
  HnAssert(prev_soln != NULL, "KheDrsExpanderMakeAndMeldSoln internal error 1");
  HnAssert(de->open, "KheDrsExpanderMakeAndMeldSoln internal error 2");
  if( DEBUG49 && HaArrayCount(de->assts_to_tasks) !=
      KheDrsResourceSetCount(de->solver->open_resources) )
  {
    fprintf(stderr, "  [ KheDrsExpanderMakeAndMeldSoln failing (%d assts, "
      "%d resources)\n", HaArrayCount(de->assts_to_tasks),
      KheDrsResourceSetCount(de->solver->open_resources));
    HaArrayForEach(de->assts_to_tasks, tasst, i)
      KheDrsAsstToTaskDebug(tasst, 2, 4, stderr);
    fprintf(stderr, "  ]\n");
  }
  HnAssert(HaArrayCount(de->assts_to_tasks) ==
    KheDrsResourceSetCount(de->solver->open_resources),
    "KheDrsExpanderMakeAndMeldSoln internal error 3 (%d assts, %d resources)",
    HaArrayCount(de->assts_to_tasks),
    KheDrsResourceSetCount(de->solver->open_resources));

  /* make a soln object */
  next_day->soln_made_count++;
  next_soln = KheDrsSolnMake(prev_soln, de->cost, de->solver);

  /* make sure de->tmp_assts_to_tasks has the right length */
  if( HaArrayCount(de->tmp_assts_to_tasks) != HaArrayCount(de->assts_to_tasks) )
  {
    HaArrayClear(de->tmp_assts_to_tasks);
    junk = KheDrsAsstToTaskMake(NULL, NULL);
    HaArrayFill(de->tmp_assts_to_tasks, HaArrayCount(de->assts_to_tasks), junk);
  }

  /* reorder de->assts_to_tasks into de->tmp_assts_to_tasks */
  HaArrayForEach(de->assts_to_tasks, tasst, i)
  {
    ri = KheDrsResourceOnDayIndex(tasst.asst->asst_to_shift->resource_on_day);
    HaArrayPut(de->tmp_assts_to_tasks, ri, tasst);
  }

  /* add each tasst's task and signature (but not its cost) to soln */
  HaArrayClear(next_soln->prev_tasks);
  HaArrayForEach(de->tmp_assts_to_tasks, tasst, i)
  {
    HaArrayAddLast(next_soln->prev_tasks, tasst.fixed_dtd);
    KheDrsSignatureSetAddSignature(&next_soln->sig_set,
      tasst.asst->asst_to_shift->sig, false);
  }

  /* set the event resource monitor part of next_soln's signature set */
  /* this last signature will be freed when next_soln is freed */
  dsg = HaArrayLast(next_day->signer_set->signers);
  if( HaArrayCount(prev_soln->sig_set.signatures) > 0 )
    prev_sig = HaArrayLast(prev_soln->sig_set.signatures);
  else
    prev_sig = NULL;  /* prev_soln is root, so prev_sig won't be accessed */
  sig = KheDrsSignatureMake(de->solver /* , 0 */);
  if( DEBUG65 )
    fprintf(stderr,
      "  KheDrsExpanderMakeAndMeldSoln: %p = KheDrsSignatureMake(drs)\n",
      (void *) sig);
  KheDrsSignerEvalSignature(dsg, true, prev_sig, next_day->open_day_index,
    sig, de->solver, false);
  KheDrsSignatureSetAddSignature(&next_soln->sig_set, sig, true);

  /* depending on cost, either add next_soln to next_day or free it */
  if( DEBUG12 )
  {
    fprintf(stderr, "  KheDrsExpanderMakeAndMeldSoln adding soln:\n");
    KheDrsSolnDebug(next_soln, de->solver, 2, 2, stderr);
  }
  if( KheDrsSolnCost(next_soln) < de->cost_limit )
    KheDrsSolnSetMeldSoln(next_day->soln_set, next_soln, next_day, de,
      de->solver);
  else
    KheDrsSolnFree(next_soln, de->solver);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExpanderMakeAndMeldShiftAsst(KHE_DRS_EXPANDER de,             */
/*    KHE_DRS_SHIFT_ASST_TRIE dsat, KHE_DRS_SIGNER dsg,                      */
/*    KHE_DRS_SOLN prev_soln, KHE_DRS_DAY next_day)                          */
/*                                                                           */
/*  Like KheDrsExpanderMakeAndMeldSoln except that it makes a shift          */
/*  assignment object and melds it into dsat's list of them.                 */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExpanderMakeAndMeldShiftAsst(KHE_DRS_EXPANDER de,
  KHE_DRS_SHIFT_ASST_TRIE dsat, KHE_DRS_SHIFT ds,
  KHE_DRS_SOLN prev_soln, KHE_DRS_DAY prev_day, KHE_DRS_DAY next_day)
{
  KHE_DRS_SHIFT_ASST dsa;  int i;  KHE_DRS_ASST_TO_TASK tasst;
  KHE_DRS_SIGNATURE prev_sig;

  /* make dsa and add de's non-fixed assignments to tasks to it */
  if( DEBUG52(prev_day, ds) )
    fprintf(stderr, "[ KheDrsExpanderMakeAndMeldShiftAsst(de, dsat, ...)\n");
  dsa = KheDrsShiftAsstMake(de->solver);
  HaArrayForEach(de->assts_to_tasks, tasst, i)
    if( !KheDrsAsstToTaskIsFixed(tasst) )
      KheDrsAsstToTaskSetAddLast(dsa->assts_to_tasks, tasst);

  /* set dsa's signature and cost */
  if( prev_soln != NULL )
    prev_sig = HaArrayLast(prev_soln->sig_set.signatures);
  else
    prev_sig = NULL;  /* prev_soln is root, so prev_sig won't be accessed */
  KheDrsSignerEvalSignature(ds->signer, false, prev_sig,
    next_day->open_day_index, dsa->sig, de->solver, DEBUG52(prev_day, ds));

  /* depending on cost, either add dsa to dsat or free it */
  if( DEBUG52(prev_day, ds) )
    fprintf(stderr, "  prev %.5f + dsa %.5f :: lim %.5f\n",
      KheCostShow(KheDrsSolnCost(prev_soln)), KheCostShow(dsa->sig->cost),
      KheCostShow(de->cost_limit));
  if( KheDrsSolnCost(prev_soln) + dsa->sig->cost < de->cost_limit )
  {
    KheDrsShiftAsstTrieMeldShiftAsst(dsat, dsa, ds->signer, de->solver);
    if( DEBUG52(prev_day, ds) )
      fprintf(stderr, "] KheDrsExpanderMakeAndMeldShiftAsst returning (yes)\n");
  }
  else
  {
    KheDrsShiftAsstFree(dsa, de->solver);
    if( DEBUG52(prev_day, ds) )
      fprintf(stderr, "] KheDrsExpanderMakeAndMeldShiftAsst returning (no)\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "Solution expansion"                                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnExpandByResources(KHE_DRS_SOLN prev_soln,                 */
/*    KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de,                             */
/*    KHE_DRS_RESOURCE_SET free_resources, int resource_index)               */
/*                                                                           */
/*  Carry out the main part of KheDrsSolnExpand, expanding by resources,     */
/*  starting with the resource whose index in drs->open_resources is         */
/*  resource_index.                                                          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnExpandByResources(KHE_DRS_SOLN prev_soln,
  KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de,
  KHE_DRS_RESOURCE_SET free_resources, int free_resources_index)
{
  KHE_DRS_RESOURCE dr;  int i;  KHE_DRS_ASST_TO_TASK_CLASS asst;

  if( DEBUG22 )
    fprintf(stderr, "%*s[ KheDrsSolnExpandByResources(-, %s, drs)\n",
      2 * free_resources_index, "", KheDrsDayId(next_day));
  if( free_resources_index >= KheDrsResourceSetCount(free_resources) )
  {
    /* de has enough assts to make into a soln and evaluate */
    KheDrsExpanderMakeAndMeldSoln(de, prev_soln, next_day);
  }
  else
  {
    dr = KheDrsResourceSetResource(free_resources, free_resources_index);
    HaArrayForEach(dr->expand_assts, asst, i)
      KheDrsAsstToTaskClassExpandByResources(asst, prev_soln,
	next_day, de, free_resources, free_resources_index);
  }
  if( DEBUG22 )
    fprintf(stderr, "%*s] KheDrsSolnExpandByResources returning\n",
      2 * free_resources_index, "");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnExpandByShifts(KHE_DRS_SOLN prev_soln,                    */
/*    KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de,                             */
/*    KHE_DRS_RESOURCE_SET free_resources, int shift_index)                  */
/*                                                                           */
/*  Like KheDrsSolnExpandByResources except that expansion is by shifts.     */
/*  The shifts to expand are those from next_day->shifts[shift_index]        */
/*  onwards, and the resources available for them are free_resources.        */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnExpandByShifts(KHE_DRS_SOLN prev_soln,
  KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de,
  KHE_DRS_RESOURCE_SET free_resources, int shift_index)
{
  KHE_DRS_SHIFT ds;  int i;  KHE_DRS_RESOURCE dr;  KHE_DRS_ASST_TO_TASK tasst;
  if( shift_index >= HaArrayCount(next_day->shifts) )
  {
    /* assign a free day to each remaining free resource; first, */
    /* abandon this path if any free resource has no free day asst */
    KheDrsResourceSetForEach(free_resources, dr, i)
      if( dr->expand_free_day_asst == NULL )
	return;

    /* save the expander */
    KheDrsExpanderMarkBegin(de);

    /* add free day assignments to the expander */
    KheDrsResourceSetForEach(free_resources, dr, i)
    {
      tasst = KheDrsAsstToTaskMake(dr->expand_free_day_asst, NULL);
      KheDrsExpanderAddAsstToTask(de, tasst);
    }

    /* if the expander is still open then make the solution */
    if( KheDrsExpanderIsOpen(de) )
      KheDrsExpanderMakeAndMeldSoln(de, prev_soln, next_day);

    /* restore the expander */
    KheDrsExpanderMarkEnd(de);
  }
  else
  {
    ds = HaArray(next_day->shifts, shift_index);
    KheDrsShiftExpandByShifts(ds, shift_index, prev_soln, next_day,
      de, free_resources);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnExpand(KHE_DRS_SOLN prev_soln, KHE_DRS_DAY next_day,      */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Build drs->expand_prev_assts up to full length in all possible ways,     */
/*  then for each way, use it to expand prev_soln to a solution for new_day. */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnExpand(KHE_DRS_SOLN prev_soln, KHE_DRS_DAY prev_day,
  KHE_DRS_DAY next_day, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int i, j, k;  KHE_DRS_RESOURCE dr;  KHE_DRS_SHIFT ds, ds2;
  KHE_DRS_EXPANDER de, shift_de;  KHE_DRS_RESOURCE_SET free_resources;
  KHE_DRS_ASST_TO_TASK_SET fixed_assts;  KHE_DRS_ASST_TO_TASK asst;

  /* check on and update the number of expansions from prev_day */
  if( prev_day != NULL )
  {
    if( drs->solve_daily_expand_limit > 0 &&
	prev_day->solve_expand_count >= drs->solve_daily_expand_limit )
    {
      if( DEBUG22 )
	fprintf(stderr, "  KheDrsSolnExpand returning early\n");
      return;
    }
    prev_day->solve_expand_count += 1;
  }

  /* detailed debug print if first expand out of prev_day */
  if( DEBUG45(prev_day) || DEBUG47(prev_day) || DEBUG51(prev_day) || DEBUG22 )
    fprintf(stderr, "[ KheDrsSolnExpand(prev_soln %d, %s, %s, drs)\n",
      prev_day == NULL ? -1 : prev_day->solve_expand_count,
      KheDrsDayId(prev_day), KheDrsDayId(next_day));

  /* mark prev_soln as expanded */
  KheDrsSolnMarkExpanded(prev_soln);

  if( DEBUG22 )
  {
    fprintf(stderr, "[ KheDrsSolnExpand(-, %s, %s, drs)\n",
      KheDrsDayId(prev_day), KheDrsDayId(next_day));
    fprintf(stderr, "  solve_init_cost %.5f, solve_start_cost %.5f\n",
      KheCostShow(drs->solve_init_cost), KheCostShow(drs->solve_start_cost));
    fprintf(stderr, "  prev_soln->cost %.5f, solve_daily_expand_limit %d\n",
      KheCostShow(KheDrsSolnCost(prev_soln)), drs->solve_daily_expand_limit);
    fprintf(stderr, "  solve_daily_prune_trigger %d\n",
      drs->solve_daily_prune_trigger);
    fprintf(stderr, "  solve_resource_expand_limit %d\n",
      drs->solve_resource_expand_limit);
    fprintf(stderr, "  solve_dom_approx %d\n",
      drs->solve_dom_approx);
  }

  /* make the main expander */
  de = KheDrsExpanderMake(false, KheDrsSolnCost(prev_soln),
    drs->solve_init_cost, KheDrsResourceSetCount(drs->open_resources), 0, drs);

  /* begin expansion in each open resource */
  free_resources = KheDrsResourceSetMake(drs);
  fixed_assts = KheDrsAsstToTaskSetMake(drs);
  KheDrsResourceSetForEach(drs->open_resources, dr, i)
    KheDrsResourceExpandBegin(dr, prev_soln, next_day, free_resources,
      fixed_assts, de);
  KheDrsResourceSetForEach(free_resources, dr, i)
    KheDrsResourceExpandBeginFree(dr, prev_soln, next_day, de);
  HnAssert(KheDrsAsstToTaskSetCount(fixed_assts) +
    KheDrsResourceSetCount(free_resources) ==
    KheDrsResourceSetCount(drs->open_resources),
    "KheDrsSolnExpand internal error 1 (fixed %d + free %d != open %d)",
    KheDrsAsstToTaskSetCount(fixed_assts),
    KheDrsResourceSetCount(free_resources),
    KheDrsResourceSetCount(drs->open_resources));
  if( DEBUG51(prev_day) )
  {
    fprintf(stderr, "  fixed: ");
    KheDrsAsstToTaskSetDebug(fixed_assts, 1, 0, stderr);
    fprintf(stderr, "  free:  ");
    KheDrsResourceSetDebug(free_resources, 2, 0, stderr);
  }

  /* begin expansion in next_day and its shifts */
  KheDrsDayExpandBegin(next_day, prev_soln, de);

  /* set up for one-extra and two-extra selection, if requested */
  if( drs->solve_extra_selection )
  {
    KheDrsOneExtraSelectionInit(drs);
    KheDrsTwoExtraSelectionInit(drs);
  }

  /* initialize assignment tries in each shift */
  if( drs->solve_expand_by_shifts )
  {
    /* initialize shift assignment tries */
    shift_de = KheDrsExpanderMake(true, 0, 0, 0, 0, drs);
    HaArrayForEach(next_day->shifts, ds, i)
    {
      KheDrsShiftBuildShiftAsstTrie(ds, prev_soln, prev_day, next_day,
	free_resources, fixed_assts, shift_de, drs);
      if( DEBUG47(prev_day) || DEBUG51(prev_day) )
	KheDrsShiftDebug(ds, drs, 2, 2, stderr);
    }
    KheDrsExpanderFree(shift_de);

    /* find forced assignments, prune shift assignments invalidated by them */
    HaArrayForEach(next_day->shifts, ds, i)
      KheDrsResourceSetForEach(drs->open_resources, dr, j)
	if( KheDrsShiftAsstTrieResourceIsForced(ds->asst_trie, dr) )
	{
	  /* dr is forced in ds, so prune it from the others */
	  HaArrayForEach(next_day->shifts, ds2, k)
	    if( ds2 != ds )
	      KheDrsShiftAsstTriePruneForced(ds2->asst_trie, dr, drs);
	}
  }

  /* carry out the main part of the expansion */
  if( DEBUG49 && KheDrsAsstToTaskSetCount(fixed_assts) > 0 )
    fprintf(stderr, " [ KheDrsSolnExpand %d free, %d fixed:\n",
      KheDrsResourceSetCount(free_resources),
      KheDrsAsstToTaskSetCount(fixed_assts));
  KheDrsExpanderMarkBegin(de);
  KheDrsExpanderAddAsstToTaskSet(de, fixed_assts, NULL);
  if( KheDrsExpanderIsOpen(de) )
  {
    if( drs->solve_expand_by_shifts )
      KheDrsSolnExpandByShifts(prev_soln, next_day, de, free_resources, 0);
    else
      KheDrsSolnExpandByResources(prev_soln, next_day, de, free_resources, 0);
  }
  KheDrsExpanderMarkEnd(de);
  if( DEBUG49 && KheDrsAsstToTaskSetCount(fixed_assts) > 0 )
    fprintf(stderr, " ] KheDrsSolnExpand\n");

  /* end expansion in next_day and its shifts */
  KheDrsDayExpandEnd(next_day, de);

  /* end expansion in each open resource */
  KheDrsResourceSetForEach(drs->open_resources, dr, i)
    KheDrsResourceExpandEnd(dr, de);
  KheDrsResourceSetFree(free_resources, drs);

  /* free fixed_assts and the expander and return */
  KheDrsAsstToTaskSetForEach(fixed_assts, asst, i)
    if( asst.fixed_dtd != NULL )
      asst.fixed_dtd->encl_dt->expand_role = KHE_DRS_TASK_EXPAND_NO;
  KheDrsAsstToTaskSetFree(fixed_assts, drs);
  KheDrsExpanderFree(de);
  if( DEBUG45(prev_day) || DEBUG47(prev_day) || DEBUG51(prev_day) || DEBUG22 )
    fprintf(stderr, "] KheDrsSolnExpand returning\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Major category "solution sets"                                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Submodule "priority queue of solutions"                                  */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  int64_t KhePriQueueKeyFn(void *entry)                                    */
/*  int KhePriQueueIndexGetFn(void *entry)                                   */
/*  void KhePriQueueIndexSetFn(void *entry, int index)                       */
/*                                                                           */
/*  Functions passed to the priority queue object.                           */
/*                                                                           */
/*****************************************************************************/

static int64_t KhePriQueueKeyFn(void *entry)
{
  KHE_DRS_SOLN soln = (KHE_DRS_SOLN) entry;
  return (int64_t) KheDrsSolnCost(soln);
}

static int KhePriQueueIndexGetFn(void *entry)
{
  KHE_DRS_SOLN soln = (KHE_DRS_SOLN) entry;
  return soln->priqueue_index;
}

static void KhePriQueueIndexSetFn(void *entry, int index)
{
  KHE_DRS_SOLN soln = (KHE_DRS_SOLN) entry;
  soln->priqueue_index = index;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsPriQueueAddSoln(KHE_DYNAMIC_RESOURCE_SOLVER drs,              */
/*    KHE_DRS_SOLN soln)                                                     */
/*                                                                           */
/*  If the priqueue is in use, add soln to it.                               */
/*                                                                           */
/*  It should be impossible to add the same solution twice, and this         */
/*  code checks this condition.                                              */
/*                                                                           */
/*****************************************************************************/

static void KheDrsPriQueueAddSoln(KHE_DYNAMIC_RESOURCE_SOLVER drs,
  KHE_DRS_SOLN soln)
{
  if( drs->solve_priqueue )
  {
    if( DEBUG31 )
      fprintf(stderr, "  KheDrsPriQueueAddSoln(drs->priqueue, %p)\n",
	(void *) soln);
    HnAssert(soln->priqueue_index == 0, "KheDrsPriQueueAddSoln internal error");
    KhePriQueueInsert(drs->priqueue, (void *) soln);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsPriQueueDeleteSoln(KHE_DYNAMIC_RESOURCE_SOLVER drs,           */
/*    KHE_DRS_SOLN soln)                                                     */
/*                                                                           */
/*  If the priqueue is in use, delete soln from it.                          */
/*                                                                           */
/*  It should be impossible to delete a solution that is not currently       */
/*  in the priqueue, but actually it can happen when deleting everything     */
/*  at the end of a solve, so it is allowed.                                 */
/*                                                                           */
/*  Obsolete:                                                                */
/*  It is possible that this deletion might occur after this solution        */
/*  has been deleted from the priqueue by a delete min operation.  So        */
/*  this code checks whether soln is in the priqueue, and if not, it is      */
/*  not considered to be an error.  Instead, the code does nothing.          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsPriQueueDeleteSoln(KHE_DYNAMIC_RESOURCE_SOLVER drs,
  KHE_DRS_SOLN soln)
{
  if( drs->solve_priqueue && soln->priqueue_index > 0 )
  {
    if( DEBUG31 )
      fprintf(stderr, "  KheDrsPriQueueDeleteSoln(drs->priqueue, %p)\n",
	(void *) soln);
    KhePriQueueDeleteEntry(drs->priqueue, (void *) soln);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsPriQueueNotifyKeyChangeSoln(KHE_DYNAMIC_RESOURCE_SOLVER drs,  */
/*    KHE_DRS_SOLN soln)                                                     */
/*                                                                           */
/*  If the priqueue is in use and soln is in it, notify a key change.        */
/*                                                                           */
/*****************************************************************************/

static void KheDrsPriQueueNotifyKeyChangeSoln(KHE_DYNAMIC_RESOURCE_SOLVER drs,
  KHE_DRS_SOLN soln)
{
  if( drs->solve_priqueue )
  {
    HnAssert(soln->priqueue_index > 0,
      "KheDrsPriQueueNotifyKeyChangeSoln internal error");
    KhePriQueueNotifyKeyChange(drs->priqueue, (void *) soln);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_SOLN_LIST"                                            */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SOLN_LIST KheDrsSolnListMake(KHE_DYNAMIC_RESOURCE_SOLVER drs)    */
/*                                                                           */
/*  Return a new, empty KHE_DRS_SOLN_LIST object.                            */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_SOLN_LIST KheDrsSolnListMake(KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SOLN_LIST res;
  if( HaArrayCount(drs->soln_list_free_list) > 0 )
  {
    res = HaArrayLastAndDelete(drs->soln_list_free_list);
    HaArrayClear(res->solns);
  }
  else
  {
    HaMake(res, drs->arena);
    HaArrayInit(res->solns, drs->arena);
  }
  res->solver = drs;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnListFree(KHE_DRS_SOLN_LIST soln_list,                     */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Free soln_list, but not the solutions it contains.                       */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnListFree(KHE_DRS_SOLN_LIST soln_list,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  HaArrayAddLast(drs->soln_list_free_list, soln_list);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnListFreeSolns(KHE_DRS_SOLN_LIST soln_list,                */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Free the solutions of soln_list.                                         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnListFreeSolns(KHE_DRS_SOLN_LIST soln_list,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SOLN soln;  int i;
  HaArrayForEach(soln_list->solns, soln, i)
  {
    KheDrsPriQueueDeleteSoln(drs, soln);
    KheDrsSolnFree(soln, drs);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsSolnListCount(KHE_DRS_SOLN_LIST soln_list)                     */
/*                                                                           */
/*  Return the number of solutions in soln_list.                             */
/*                                                                           */
/*****************************************************************************/

static int KheDrsSolnListCount(KHE_DRS_SOLN_LIST soln_list)
{
  return HaArrayCount(soln_list->solns);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnListGather(KHE_DRS_SOLN_LIST soln_list,                   */
/*    KHE_DRS_SOLN_LIST res)                                                 */
/*                                                                           */
/*  Gather the solutions of soln_list into res.                              */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnListGather(KHE_DRS_SOLN_LIST soln_list,
  KHE_DRS_SOLN_LIST res)
{
  int i;
  HaArrayAppend(res->solns, soln_list->solns, i);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSolnListDominates(KHE_DRS_SOLN_LIST soln_list,                */
/*    KHE_DRS_SOLN soln, KHE_DRS_SIGNER_SET signer_set,                      */
/*    int *dom_test_count, KHE_DYNAMIC_RESOURCE_SOLVER drs)                  */
/*                                                                           */
/*  Return true if soln_list contains a solution that dominates soln.        */
/*  If with_tradeoff is true, use tradeoff dominance.  If with_uniform       */
/*  is true, use uniform dominance.                                          */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsSolnListDominates(KHE_DRS_SOLN_LIST soln_list,
  KHE_DRS_SOLN soln, KHE_DRS_SIGNER_SET signer_set,
  int *dom_test_count, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SOLN other_soln;  int i;
  HaArrayForEach(soln_list->solns, other_soln, i)
    if( KheDrsSolnDominates(other_soln, soln, signer_set, dom_test_count, drs,
	0, 0, NULL) )
      return true;
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnListRemoveDominated(KHE_DRS_SOLN_LIST soln_list,          */
/*    KHE_DRS_SOLN soln, KHE_DRS_SIGNER_SET signer_set,                      */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Delete and free the unexpanded solutions of soln_list that are           */
/*  dominated by soln, including deleting them from drs->priqueue.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnListRemoveDominated(KHE_DRS_SOLN_LIST soln_list,
  KHE_DRS_SOLN soln, KHE_DRS_SIGNER_SET signer_set,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SOLN other_soln;  int i, dom_test_count;
  dom_test_count = 0;
  HaArrayForEach(soln_list->solns, other_soln, i)
    if( KheDrsSolnNotExpanded(other_soln) && KheDrsSolnDominates(soln,
	  other_soln, signer_set, &dom_test_count, drs, 0, 0, NULL) )
    {
      KheDrsPriQueueDeleteSoln(drs, other_soln);
      KheDrsSolnFree(other_soln, drs);
      HaArrayDeleteAndPlug(soln_list->solns, i);
      i--;
    }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnListAddSoln(KHE_DRS_SOLN_LIST soln_list,                  */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs)                    */
/*                                                                           */
/*  Add soln to soln_list, assuming that all the dominance testing is done.  */
/*  This includes adding soln to drs->priqueue, if required.                 */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnListAddSoln(KHE_DRS_SOLN_LIST soln_list,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  HaArrayAddLast(soln_list->solns, soln);
  KheDrsPriQueueAddSoln(drs, soln);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnListSortAndReduce(KHE_DRS_SOLN_LIST soln_list,            */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Sort the solutions of soln_list into increasing cost order, then         */
/*  reduce their number to drs->daily_solve_limit, if non-zero.  Also        */
/*  assign a rank in the sorted list to each remaining solution.             */
/*                                                                           */
/*****************************************************************************/

static int KheDrsSolnCmp(const void *t1, const void *t2)
{
  KHE_DRS_SOLN soln1 = * (KHE_DRS_SOLN *) t1;
  KHE_DRS_SOLN soln2 = * (KHE_DRS_SOLN *) t2;
  return KheCostCmp(KheDrsSolnCost(soln1), KheDrsSolnCost(soln2));
}

static void KheDrsSolnListSortAndReduce(KHE_DRS_SOLN_LIST soln_list,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SOLN soln;  int i;
  HaArraySort(soln_list->solns, &KheDrsSolnCmp);
  if( drs->solve_daily_expand_limit > 0 )
    while( HaArrayCount(soln_list->solns) > drs->solve_daily_expand_limit )
    {
      soln = HaArrayLastAndDelete(soln_list->solns);
      KheDrsPriQueueDeleteSoln(drs, soln);
      KheDrsSolnFree(soln, drs);
    }
#if TESTING
  HaArrayForEach(soln_list->solns, soln, i)
    soln->sorted_rank = i + 1;
#endif
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsSolnListSignatureSetPartialHashUntyped(void *p)                */
/*                                                                           */
/*  Hash function for soln lists, being the partial hash of the list's       */
/*  first element.                                                           */
/*                                                                           */
/*****************************************************************************/

static int KheDrsSolnListSignatureSetPartialHashUntyped(void *p)
{
  KHE_DRS_SOLN_LIST soln_list;  KHE_DRS_SOLN soln;
  soln_list = (KHE_DRS_SOLN_LIST) p;
  soln = HaArrayFirst(soln_list->solns);
  return KheDrsSolnSignatureSetPartialHash(soln, soln_list->solver);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSolnListSignatureSetPartialEqualUntyped(void *p1, void *p2)   */
/*                                                                           */
/*  Equality function for soln lists, being the partial equality of the      */
/*  lists' first elements.                                                   */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsSolnListSignatureSetPartialEqualUntyped(void *p1, void *p2)
{
  KHE_DRS_SOLN_LIST soln_list1, soln_list2;  KHE_DRS_SOLN soln1, soln2;
  soln_list1 = (KHE_DRS_SOLN_LIST) p1;
  soln_list2 = (KHE_DRS_SOLN_LIST) p2;
  soln1 = HaArrayFirst(soln_list1->solns);
  soln2 = HaArrayFirst(soln_list2->solns);
  return KheDrsSolnSignatureSetPartialEqual(soln1, soln2, soln_list1->solver);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnListExpand(KHE_DRS_SOLN_LIST soln_list,                   */
/*    KHE_DRS_DAY prev_day, KHE_DRS_DAY next_day,                            */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Expand soln_list.                                                        */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnListExpand(KHE_DRS_SOLN_LIST soln_list,
  KHE_DRS_DAY prev_day, KHE_DRS_DAY next_day,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SOLN prev_soln;  int i;
  if( DEBUG46 )
    fprintf(stderr, "[ KheDrsSolnListExpand(soln_list, %s, %s, drs)\n",
      KheDrsDayId(prev_day), KheDrsDayId(next_day));
  HaArrayForEach(soln_list->solns, prev_soln, i)
  {
    if( DEBUG46 )
      fprintf(stderr, "  KheDrsSolnListExpand calling KheDrsSolnExpand\n");
    KheDrsSolnExpand(prev_soln, prev_day, next_day, drs);
  }
  if( DEBUG46 )
    fprintf(stderr, "] KheDrsSolnListExpand returning\n");
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SOLN KheDrsSolnListFirstSoln(KHE_DRS_SOLN_LIST soln_list)        */
/*                                                                           */
/*  Return the first soln of soln_list.  This assumes there is one.          */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_SOLN KheDrsSolnListFirstSoln(KHE_DRS_SOLN_LIST soln_list)
{
  return HaArrayFirst(soln_list->solns);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsSolnListViableSolns(KHE_DRS_SOLN_LIST soln_list,               */
/*    KHE_COST max_viable_cost)                                              */
/*                                                                           */
/*  Return the number of solutions in soln_list whose cost is at most        */
/*  max_viable_cost.  This is assuming that the solutions are sorted         */
/*  by increasing cost.  This function is used only when testing.            */
/*                                                                           */
/*****************************************************************************/

static int KheDrsSolnListViableSolns(KHE_DRS_SOLN_LIST soln_list,
  KHE_COST max_viable_cost)
{
  KHE_DRS_SOLN soln;  int i;
  HaArrayForEach(soln_list->solns, soln, i)
    if( KheDrsSolnCost(soln) > max_viable_cost )
      break;
  return i;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnListCostDistributionDebug(KHE_DRS_SOLN_LIST soln_list,    */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Print onto fp with the given verbosity and indent a histogram of         */
/*  the distribution of costs in soln_list.  This is assuming that the       */
/*  solutions are sorted to bring solutions with equal cost together.        */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnListCostDistributionDebug(KHE_DRS_SOLN_LIST soln_list,
  int verbosity, int indent, FILE *fp)
{
  KHE_DRS_SOLN solni, solnj;  int i, j;
  for( i = 0;  i < HaArrayCount(soln_list->solns);  i = j )
  {
    solni = HaArray(soln_list->solns, i);
    for( j = i + 1;  j < HaArrayCount(soln_list->solns);  j++ )
    {
      solnj = HaArray(soln_list->solns, j);
      if( KheDrsSolnCost(solni) != KheDrsSolnCost(solnj) )
	break;
    }
    fprintf(fp, "%*s%.5f: %d\n", indent, "",
      KheCostShow(KheDrsSolnCost(solni)), j - i);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnListSignatureSetDebugUntyped(void *p, FILE *fp)           */
/*                                                                           */
/*  Debug print of the signature of soln set p.                              */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnListSignatureSetDebugUntyped(void *p, FILE *fp)
{
  KHE_DRS_SOLN_LIST soln_list;  KHE_DRS_SOLN soln;
  soln_list = (KHE_DRS_SOLN_LIST) p;
  soln = HaArrayFirst(soln_list->solns);
  KheDrsSolnSignatureSetDebug(soln, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnListDebugFirstAndLast(KHE_DRS_SOLN_LIST soln_list,        */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)  */
/*                                                                           */
/*  If they exist and are distinct, do a detailed debug of the dominance     */
/*  test between the first and last elements of soln_list.                   */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnListDebugFirstAndLast(KHE_DRS_SOLN_LIST soln_list,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)
{
  KHE_DRS_SOLN soln1, soln2;  KHE_DRS_DAY day;  int junk;
  if( HaArrayCount(soln_list->solns) >= 2 )
  {
    fprintf(fp, "%*s[ FirstAndLast:\n", indent, "");
    soln1 = HaArrayFirst(soln_list->solns);
    soln2 = HaArrayLast(soln_list->solns);
    day = KheDrsSolnDay(soln1, drs);
    KheDrsSolnDominates(soln1, soln2, day->signer_set, &junk, drs,
      verbosity, indent + 2, fp);
    fprintf(fp, "%*s]\n", indent, "");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnListDebug(KHE_DRS_SOLN_LIST soln_list,                    */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)  */
/*                                                                           */
/*  Debug print of soln_list onto fp with the given verbosity and indent.    */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnListDebug(KHE_DRS_SOLN_LIST soln_list,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)
{
  KHE_DRS_SOLN soln;  int i;
  if( indent >= 0 )
  {
    fprintf(fp, "%*s[ SolnList(%d solns)\n", indent, "",
      KheDrsSolnListCount(soln_list));
    HaArrayForEach(soln_list->solns, soln, i)
    {
      KheDrsSolnDebug(soln, drs, verbosity, indent + 2, fp);
      if( i > 50 )
      {
	fprintf(fp, "%*s... omitting the rest\n", indent + 2, "");
	break;
      }
    }
    fprintf(fp, "%*s]\n", indent, "");
  }
  else
    fprintf(fp, "SolnList(%d solns)", KheDrsSolnListCount(soln_list));
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_SOLN_TRIE and trie dominance"                         */
/*                                                                           */
/*  Abstraction function                                                     */
/*  --------------------                                                     */
/*                                                                           */
/*  A soln_trie represents a set of solns.  If soln_trie == NULL, the set    */
/*  is empty.  Otherwise the set consists of soln_trie->soln (if non-NULL)   */
/*  plus the sets stored in the children.                                    */
/*                                                                           */
/*  Invariant                                                                */
/*  ---------                                                                */
/*                                                                           */
/*  Each soln appears in the trie in a place determined by the sequence      */
/*  of values in its signature, in the usual trie manner.  Exactly one       */
/*  of these three cases applies:                                            */
/*                                                                           */
/*    (1)  soln_trie == NULL                                                 */
/*                                                                           */
/*    (2)  soln_trie != NULL && soln_trie->soln != NULL &&                   */
/*           HaArrayCount(soln_trie->children) == 0                          */
/*                                                                           */
/*    (3)  soln_trie != NULL && soln_trie->soln == NULL &&                   */
/*           HaArrayCount(soln_trie->children) > 0                           */
/*                                                                           */
/*  In case (3), the first child of soln_trie->children is non-NULL, and     */
/*  the last child of soln_trie->children is non-NULL.  These may be the     */
/*  same child.  The concrete index of the first child is 0 as usual,        */
/*  but it represents a child with abstract index soln_trie->base, with      */
/*  subsequent children also having abstract indexes increased by this.      */
/*  This allows the trie to handle indexes which are all far above zero      */
/*  without becoming burdened by many initial NULL children.                 */
/*                                                                           */
/*  There is no requirement that two or more solns be present in case (3).   */
/*  If there are no solns, case (1) always applies; if there are two or      */
/*  more solns, case (3) always applies; but if there is exactly one soln,   */
/*  then either of cases (2) or (3) applies.  This is because it would be    */
/*  too slow and tedious to detect situations where a case (3) soln could    */
/*  be simplified to a case (2) soln.                                        */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SOLN_TRIE KheDrsSolnTrieMake(KHE_DRS_SOLN soln,                  */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new trie object containing soln.  It satisfies case (2) above.    */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_SOLN_TRIE KheDrsSolnTrieMake(KHE_DRS_SOLN soln,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SOLN_TRIE res;
  HnAssert(soln != NULL, "KheDrsSolnTrieMake internal error (soln == NULL)");
  if( HaArrayCount(drs->soln_trie_free_list) > 0 )
  {
    res = HaArrayLastAndDelete(drs->soln_trie_free_list);
    HaArrayClear(res->children);
  }
  else
  {
    HaMake(res, drs->arena);
    HaArrayInit(res->children, drs->arena);
  }
  res->soln = soln;
  res->base = 0;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnTrieFree(KHE_DRS_SOLN_TRIE *soln_trie,                    */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Free *soln_trie and all its descendants, setting *soln_trie to NULL;     */
/*  but do not free the solutions contained in *soln_trie.                   */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnTrieFree(KHE_DRS_SOLN_TRIE *soln_trie,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int i;
  if( *soln_trie == NULL )
  {
    /* case (1), do nothing */
  }
  else if( (*soln_trie)->soln != NULL )
  {
    /* case (2), delete *soln_trie but not (*soln_trie)->soln */
    HaArrayAddLast(drs->soln_trie_free_list, *soln_trie);
    *soln_trie = NULL;
  }
  else
  {
    /* case (3), delete children and *soln_trie */
    for( i = 0;  i < HaArrayCount((*soln_trie)->children);  i++ )
      KheDrsSolnTrieFree(&HaArray((*soln_trie)->children, i), drs);
    HaArrayAddLast(drs->soln_trie_free_list, *soln_trie);
    *soln_trie = NULL;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsSolnTrieCount(KHE_DRS_SOLN_TRIE soln_trie)                     */
/*                                                                           */
/*  Return the number of solutions in soln_trie.                             */
/*                                                                           */
/*****************************************************************************/

static int KheDrsSolnTrieCount(KHE_DRS_SOLN_TRIE soln_trie)
{
  int i, res;  KHE_DRS_SOLN_TRIE child_trie;
  if( soln_trie == NULL )
  {
    /* case (1), empty */
    return 0;
  }
  else if( soln_trie->soln != NULL )
  {
    /* case (2), just soln_trie->soln */
    return 1;
  }
  else
  {
    /* case (3), search children */
    res = 0;
    HaArrayForEach(soln_trie->children, child_trie, i)
      res += KheDrsSolnTrieCount(child_trie);
    return res;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnTrieClear(KHE_DRS_SOLN_TRIE *soln_trie,                   */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Clear soln_trie back to empty.  For this particular data structure,      */
/*  clearing and freeing are necessarily the same.                           */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsSolnTrieClear(KHE_DRS_SOLN_TRIE *soln_trie,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KheDrsSolnTrieFree(soln_trie, drs);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnTrieGather(KHE_DRS_SOLN_TRIE soln_trie,                   */
/*    KHE_DRS_SOLN_LIST soln_list)                                           */
/*                                                                           */
/*  Gather the solutions of soln_trie into soln_list, without disturbing     */
/*  soln_trie.                                                               */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnTrieGather(KHE_DRS_SOLN_TRIE soln_trie,
  KHE_DRS_SOLN_LIST soln_list)
{
  KHE_DRS_SOLN_TRIE child_trie;  int i;
  if( soln_trie == NULL )
  {
    /* case (1); do nothing */
  }
  else if( soln_trie->soln != NULL )
  {
    /* case (2); add soln_trie->soln to soln_List->solns */
    HaArrayAddLast(soln_list->solns, soln_trie->soln);
  }
  else
  {
    /* case (3); handle children */
    HaArrayForEach(soln_trie->children, child_trie, i)
      KheDrsSolnTrieGather(child_trie, soln_list);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheSetHasExtraValueAtLeft(int val, int from_val, int to_val,        */
/*    bool also_zero, int *extra_val)                                        */
/*                                                                           */
/*  If the interval [from_val .. to_val] has val at the left end, return     */
/*  true and set *extra_val to the adjacent value outside the interval.      */
/*                                                                           */
/*****************************************************************************/

static bool KheSetHasExtraValueAtLeft(int val, int from_val, int to_val,
  bool also_zero, int *extra_val)
{
  if( from_val <= to_val && from_val == val && from_val >= (also_zero ? 2 : 1) )
    return *extra_val = from_val - 1, true;
  else
    return *extra_val = -1, false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheSetHasExtraValueAtRight(int val, int from_val, int to_val,       */
/*    bool also_zero, int *extra_val)                                        */
/*                                                                           */
/*  If the interval [from_val .. to_val] has val at the right end, return    */
/*  true and set *extra_val to the adjacent value outside the interval.      */
/*                                                                           */
/*****************************************************************************/

static bool KheSetHasExtraValueAtRight(int val, int from_val, int to_val,
  bool also_zero, int *extra_val)
{
  if( from_val <= to_val && to_val == val )
    return *extra_val = to_val + 1, true;
  else
    return *extra_val = -1, false;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDomTestAndValueAtDepth(KHE_DRS_SIGNER_SET signer_set,         */
/*    KHE_DRS_SIGNATURE_SET sig_set, int depth, KHE_DRS_DOM_TEST *dom_test,  */
/*    int *value)                                                            */
/*                                                                           */
/*  Find the dom test of signer_set and the value of sig_set at depth.       */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDomTestAndValueAtDepth(KHE_DRS_SIGNER_SET signer_set,
  KHE_DRS_SIGNATURE_SET sig_set, int depth, KHE_DRS_DOM_TEST *dom_test,
  int *value)
{
  KHE_DRS_SIGNER dsg;  KHE_DRS_SIGNATURE sig;  int i;
  HnAssert(HaArrayCount(signer_set->signers)==HaArrayCount(sig_set->signatures),
    "KheDrsDomTestAndValueAtDepth internal error 1");
  for( i = 0;  i < HaArrayCount(signer_set->signers);  i++ )
  {
    dsg = HaArray(signer_set->signers, i);
    if( depth < HaArrayCount(dsg->dom_tests) )
    {
      sig = HaArray(sig_set->signatures, i);
      *dom_test = HaArray(dsg->dom_tests, depth);
      *value = HaArray(sig->states, depth);
      return;
    }
    depth -= HaArrayCount(dsg->dom_tests);
  }
  HnAbort("KheDrsDomTestAndValueAtDepth internal error 2");
  *dom_test = NULL;   /* keep compiler happy */
  *value = 0;         /* keep compiler happy */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsValueAtDepth(KHE_DRS_SIGNATURE_SET sig_set, int depth,        */
/*    int *value)                                                            */
/*                                                                           */
/*  Like KheDrsDomTestAndValueAtDepth but returning just the value.          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsValueAtDepth(KHE_DRS_SIGNATURE_SET sig_set, int depth,
  int *value)
{
  KHE_DRS_SIGNATURE sig;  int i;
  HaArrayForEach(sig_set->signatures, sig, i)
  {
    if( depth < HaArrayCount(sig->states) )
    {
      sig = HaArray(sig_set->signatures, i);
      *value = HaArray(sig->states, depth);
      return;
    }
    depth -= HaArrayCount(sig->states);
  }
  HnAbort("KheDrsValueAtDepth internal error");
  *value = 0;         /* keep compiler happy */
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSolnTrieDoDominates(KHE_DRS_SOLN_TRIE soln_trie,              */
/*    KHE_DRS_SOLN soln, KHE_DRS_DAY soln_day, KHE_COST extra_cost,          */
/*    int depth, int *dom_test_count, KHE_DYNAMIC_RESOURCE_SOLVER drs)       */
/*                                                                           */
/*  Return true if there is a soln in soln_trie that dominates soln, given   */
/*  that we have already paid extra_cost for the solutions in soln_trie.     */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsSolnTrieDoDominates(KHE_DRS_SOLN_TRIE soln_trie,
  KHE_DRS_SOLN soln, KHE_DRS_DAY soln_day, KHE_COST extra_cost,
  int depth, int *dom_test_count, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int val, from_val, to_val, i;  KHE_DRS_DOM_TEST dom_test;  bool also_zero;
  if( soln_trie == NULL )
  {
    /* case (1); no solns in soln_trie, so none dominate */
    return false;
  }
  else if( soln_trie->soln != NULL )
  {
    /* case (2); the only possible signer is soln_trie->soln */
    return KheDrsSolnDoDominates(soln_trie->soln, soln, soln_day->signer_set,
      extra_cost, depth, dom_test_count, drs, 0, 0, NULL);
  }
  else
  {
    /* case (3); find the dominating set for the current signature value */
    KheDrsDomTestAndValueAtDepth(soln_day->signer_set, &soln->sig_set,
      depth, &dom_test, &val);
    KheDrsDomTestDominatingSet(dom_test, val, &from_val, &to_val, &also_zero);

    /* convert dominating set to dominating indexes actually present */
    from_val -= soln_trie->base;
    to_val -= soln_trie->base;
    if( from_val < 0 )
      from_val = 0;
    if( to_val >= HaArrayCount(soln_trie->children) )
      to_val = HaArrayCount(soln_trie->children) - 1;
    also_zero = also_zero && soln_trie->base == 0;

    /* search for dominating solutions at the dominating indexes */
    for( i = from_val;  i <= to_val;  i++ )
      if( KheDrsSolnTrieDoDominates(HaArray(soln_trie->children, i),
	    soln, soln_day, extra_cost, depth + 1, dom_test_count, drs) )
	return true;
    if( also_zero && KheDrsSolnTrieDoDominates(HaArray(soln_trie->children, 0),
	    soln, soln_day, extra_cost, depth + 1, dom_test_count, drs) )
	return true;

    /* if( with_tradeoff && dom_test->tradeoff_allowed ) */
    if( drs->solve_dom_test_type == KHE_DRS_DOM_TEST_TRADEOFF )
    {
      /* if there is an extra value at right, convert that to a dominating */
      /* index and search for dominating solutions at that dominating index */
      if( KheSetHasExtraValueAtRight(val, from_val, to_val, also_zero, &i))
      {
	i -= soln_trie->base;
	if( i >= 0 && i < HaArrayCount(soln_trie->children) &&
	    KheDrsSolnTrieDoDominates(HaArray(soln_trie->children, i), soln,
	      soln_day, extra_cost + dom_test->tradeoff_cost, depth + 1,
	      dom_test_count, drs) )
	  return true;
      }

      /* if there is an extra value at left, convert that to a dominating */
      /* index and search for dominating solutions at that dominating index */
      if( KheSetHasExtraValueAtLeft(val, from_val, to_val, also_zero, &i))
      {
	/* extra value at left */
	i -= soln_trie->base;
	if( i >= 0 && i < HaArrayCount(soln_trie->children) &&
	    KheDrsSolnTrieDoDominates(HaArray(soln_trie->children, i), soln,
	      soln_day, extra_cost + dom_test->tradeoff_cost, depth + 1,
	      dom_test_count, drs) )
	  return true;
      }
    }

    /* all done, no luck */
    return false;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnTrieRemoveDominated(KHE_DRS_SOLN_TRIE *soln_trie,         */
/*    KHE_DRS_SOLN soln, KHE_DRS_DAY soln_day, KHE_COST extra_cost,          */
/*    int depth, KHE_DYNAMIC_RESOURCE_SOLVER drs)                            */
/*                                                                           */
/*  Delete and free the unexpanded solutions of *soln_trie that are          */
/*  dominated by soln, including deleting them from drs->priqueue.  If       */
/*  with_tradeoff is true, we have already paid extra_cost for soln.  This   */
/*  can change *soln_trie.                                                   */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnTrieRemoveDominated(KHE_DRS_SOLN_TRIE *soln_trie,
  KHE_DRS_SOLN soln, KHE_DRS_DAY soln_day, KHE_COST extra_cost,
  int depth, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int val, from_val, to_val, i, dom_test_count;  KHE_DRS_DOM_TEST dom_test;
  bool also_zero;  KHE_DRS_SOLN_TRIE child_trie;
  dom_test_count = 0;
  if( *soln_trie == NULL )
  {
    /* case (1); there is nothing to remove */
  }
  else if( (*soln_trie)->soln != NULL )
  {
    /* case (2); if (*soln_trie)->soln is dominated, remove it and its trie */
    if( KheDrsSolnNotExpanded((*soln_trie)->soln) &&
        KheDrsSolnDoDominates(soln, (*soln_trie)->soln, soln_day->signer_set,
	  extra_cost, depth, &dom_test_count, drs, 0, 0, NULL) )
    {
      KheDrsPriQueueDeleteSoln(drs, (*soln_trie)->soln);
      KheDrsSolnFree((*soln_trie)->soln, drs);
      KheDrsSolnTrieFree(soln_trie, drs);
    }
  }
  else
  {
    /* case (3); remove dominated solns among the children */
    KheDrsDomTestAndValueAtDepth(soln_day->signer_set, &soln->sig_set,
      depth, &dom_test, &val);
    KheDrsDomTestDominatedSet(dom_test, val, &from_val, &to_val, &also_zero);

    /* convert dominated set to dominated indexes */
    from_val -= (*soln_trie)->base;
    to_val -= (*soln_trie)->base;
    if( from_val < 0 )
      from_val = 0;
    if( to_val >= HaArrayCount((*soln_trie)->children) )
      to_val = HaArrayCount((*soln_trie)->children) - 1;
    also_zero = also_zero && (*soln_trie)->base == 0;

    /* remove dominated solutions at the dominated indexes */
    for( i = from_val;  i <= to_val;  i++ )
      KheDrsSolnTrieRemoveDominated(&HaArray((*soln_trie)->children, i),
	soln, soln_day, extra_cost, depth + 1, drs);
    if( also_zero )
      KheDrsSolnTrieRemoveDominated(&HaArray((*soln_trie)->children, 0),
	soln, soln_day, extra_cost, depth + 1, drs);

    /* if( with_tradeoff && dom_test->tradeoff_allowed ) */
    if( drs->solve_dom_test_type == KHE_DRS_DOM_TEST_TRADEOFF )
    {
      /* if there is an extra value at right, convert that to a dominated */
      /* index and search for dominated solutions at that dominated index */
      if( KheSetHasExtraValueAtRight(val, from_val, to_val, also_zero, &i) )
      {
	i -= (*soln_trie)->base;
	if( i >= 0 && i < HaArrayCount((*soln_trie)->children) )
	  KheDrsSolnTrieRemoveDominated(&HaArray((*soln_trie)->children, i),
	    soln, soln_day, extra_cost + dom_test->tradeoff_cost, depth+1, drs);
      }

      /* if there is an extra value at left, convert that to a dominated */
      /* index and search for dominated solutions at that dominated index */
      if( KheSetHasExtraValueAtLeft(val, from_val, to_val, also_zero, &i) )
      {
	i -= (*soln_trie)->base;
	if( i >= 0 && i < HaArrayCount((*soln_trie)->children) )
	  KheDrsSolnTrieRemoveDominated(&HaArray((*soln_trie)->children, i),
	    soln, soln_day, extra_cost + dom_test->tradeoff_cost, depth+1, drs);
      }
    }

    /* remove trailing NULL children */
    while( HaArrayCount((*soln_trie)->children) > 0 &&
	HaArrayLast((*soln_trie)->children) == NULL )
      HaArrayDeleteLast((*soln_trie)->children);

    if( HaArrayCount((*soln_trie)->children) == 0 )
    {
      /* the children array is empty so the trie is empty; delete it */
      KheDrsSolnTrieFree(soln_trie, drs);
    }
    else
    {
      /* the children array is non-empty, so remove initial NULL children */
      HaArrayForEach((*soln_trie)->children, child_trie, i)
	if( child_trie != NULL )
	  break;
      if( i > 0 )
      {
	HaArrayShiftLeft((*soln_trie)->children, i);
        (*soln_trie)->base += i;
      }
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnTrieAddChild(KHE_DRS_SOLN_TRIE soln_trie,                 */
/*    KHE_DRS_SOLN soln, int depth, KHE_DYNAMIC_RESOURCE_SOLVER drs)         */
/*                                                                           */
/*  Insert soln in its appropriate place among the children of soln_trie.    */
/*                                                                           */
/*  Implementation note.  This is a helper function; it neither assumes      */
/*  that the invariant is true before nor guarantees that it is true after.  */
/*                                                                           */
/*****************************************************************************/
static void KheDrsSolnTrieAddSoln(KHE_DRS_SOLN_TRIE *soln_trie,
  KHE_DRS_SOLN soln, int depth, KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsSolnTrieAddChild(KHE_DRS_SOLN_TRIE soln_trie,
  KHE_DRS_SOLN soln, int depth, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int index;
  KheDrsValueAtDepth(&soln->sig_set, depth, &index);
  if( HaArrayCount(soln_trie->children) == 0 )
  {
    /* array is empty, set base to index and insert at 0 */
    HaArrayAddLast(soln_trie->children, NULL);
    soln_trie->base = index;
    KheDrsSolnTrieAddSoln(&HaArray(soln_trie->children, 0), soln, depth+1, drs);
  }
  else if( index < soln_trie->base )
  {
    /* shift the children right, decrease the base to index, and insert at 0 */
    HaArrayShiftRight(soln_trie->children, soln_trie->base - index, NULL);
    soln_trie->base = index;
    KheDrsSolnTrieAddSoln(&HaArray(soln_trie->children, 0), soln, depth+1, drs);
  }
  else
  {
    /* make sure there is a position for soln, and insert it there */
    index -= soln_trie->base;
    HaArrayFill(soln_trie->children, index + 1, NULL);
    KheDrsSolnTrieAddSoln(&HaArray(soln_trie->children, index), soln,
      depth+1, drs);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnTrieAddSoln(KHE_DRS_SOLN_TRIE *soln_trie,                 */
/*    KHE_DRS_SOLN soln, int depth, KHE_DYNAMIC_RESOURCE_SOLVER drs)         */
/*                                                                           */
/*  Insert soln into *soln_trie, assuming all the dominance testing is done. */
/*  This includes adding soln to drs->priqueue, if required.                 */
/*                                                                           */
/*  The value of *soln_trie may be changed by this function.                 */
/*  The depth parameter indicates how far along the signature we are.        */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnTrieAddSoln(KHE_DRS_SOLN_TRIE *soln_trie,
  KHE_DRS_SOLN soln, int depth, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  if( *soln_trie == NULL )
  {
    /* case (1), make a new soln containing just soln, -> case (2) */
    *soln_trie = KheDrsSolnTrieMake(soln, drs);
    KheDrsPriQueueAddSoln(drs, soln);
  }
  else if( (*soln_trie)->soln != NULL )
  {
    /* case (2), add soln and (*soln_trie)->soln to children, -> case (3) */
    KheDrsSolnTrieAddChild(*soln_trie, soln, depth, drs);
    KheDrsPriQueueAddSoln(drs, soln);
    soln = (*soln_trie)->soln;
    (*soln_trie)->soln = NULL;
    KheDrsSolnTrieAddChild(*soln_trie, soln, depth, drs);
  }
  else
  {
    /* case (3), add soln to children, -> case (3) */
    KheDrsSolnTrieAddChild(*soln_trie, soln, depth, drs);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnTrieInitNullStats(KHE_DRS_SOLN_TRIE soln_trie,            */
/*    int *node_count, int *init_null_count)                                 */
/*                                                                           */
/*  Find the number of initial NULL children.                                */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnTrieInitNullStats(KHE_DRS_SOLN_TRIE soln_trie,
  int *node_count, int *init_null_count)
{
  KHE_DRS_SOLN_TRIE child_trie;  int i;
  if( soln_trie == NULL )
  {
    /* case (1); do nothing */
  }
  else if( soln_trie->soln != NULL )
  {
    /* case (2); just a node, no initial NULL children */
    *node_count += 1;
  }
  else
  {
    /* case (3); count initial NULL children here in soln_trie */
    *node_count += 1;
    HaArrayForEach(soln_trie->children, child_trie, i)
    {
      if( child_trie != NULL )
	break;
      *init_null_count += 1;
    }

    /* do the job in the children */
    HaArrayForEach(soln_trie->children, child_trie, i)
      KheDrsSolnTrieInitNullStats(child_trie, node_count, init_null_count);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnTrieDebugNullInitialChildren(KHE_DRS_SOLN_TRIE soln_trie) */
/*                                                                           */
/*  Debug print of the number of NULL initial children in soln_trie.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnTrieDebugNullInitialChildren(KHE_DRS_SOLN_TRIE soln_trie)
{
  int node_count, init_null_count;
  node_count = init_null_count = 0;
  KheDrsSolnTrieInitNullStats(soln_trie, &node_count, &init_null_count);
  fprintf(stderr, "  initial NULL children: %d nodes, %d count, average %.1f\n",
    node_count, init_null_count, node_count == 0 ? 0.0 :
    (float) init_null_count / (float) node_count);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnTrieDoDebug(KHE_DRS_SOLN_TRIE soln_trie, int index,       */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)  */
/*                                                                           */
/*  Carry out the recursive part of KheDrsSolnTrieDebug.                     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnTrieDoDebug(KHE_DRS_SOLN_TRIE soln_trie, int index,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)
{
  KHE_DRS_SOLN_TRIE child_trie;  int i;
  if( soln_trie == NULL )
  {
    /* case (1); do nothing */
  }
  else if( soln_trie->soln != NULL )
  {
    /* case (2); soln_trie->soln */
    fprintf(fp, "%*s%d: ", indent, "", index);
    KheDrsSolnDebug(soln_trie->soln, drs, verbosity, indent, fp);
  }
  else
  {
    /* case (3); handle children */
    fprintf(fp, "%*s[ %d (base %d):\n", indent, "", index, soln_trie->base);
    HaArrayForEach(soln_trie->children, child_trie, i)
      KheDrsSolnTrieDoDebug(child_trie, i, drs, verbosity, indent + 2, fp);
    fprintf(fp, "%*s]\n", indent, "");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnTrieDebug(KHE_DRS_SOLN_TRIE soln_trie,                    */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)  */
/*                                                                           */
/*  Debug print of soln_trie onto fp with the given verbosity and indent.    */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnTrieDebug(KHE_DRS_SOLN_TRIE soln_trie,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)
{
  KHE_DRS_SOLN_TRIE child_trie;  int i;
  if( indent >= 0 )
  {
    fprintf(fp, "%*s[ SolnTrie\n", indent, "");
    if( soln_trie == NULL )
    {
      /* case (1); do nothing */
    }
    else if( soln_trie->soln != NULL )
    {
      /* case (2); soln_trie->soln */
      KheDrsSolnDebug(soln_trie->soln, drs, verbosity, indent + 2, fp);
    }
    else
    {
      /* case (3); handle children */
      fprintf(fp, "%*s  (base %d):\n", indent, "", soln_trie->base);
      HaArrayForEach(soln_trie->children, child_trie, i)
	KheDrsSolnTrieDoDebug(child_trie, i, drs, verbosity, indent + 2, fp);
    }
    fprintf(fp, "%*s]\n", indent, "");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_INDEXED_SOLN_SET and indexed dominance"               */
/*                                                                           */
/*  Invariant - base is undefined when soln_lists is empty; increment is     */
/*  always well-defined (and constant).                                      */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_INDEXED_SOLN_SET KheDrsIndexedSolnSetMake(                       */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new, empty indexed soln set.                                      */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_INDEXED_SOLN_SET KheDrsIndexedSolnSetMake(
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_INDEXED_SOLN_SET res;

  /* make the object */
  if( HaArrayCount(drs->indexed_soln_set_free_list) > 0 )
  {
    res = HaArrayLastAndDelete(drs->indexed_soln_set_free_list);
    HaArrayClear(res->soln_lists);
  }
  else
  {
    HaMake(res, drs->arena);
    HaArrayInit(res->soln_lists, drs->arena);
  }

  /* initialize the base and increment */
  res->base = 0;
  HnAssert(drs->increment > 0, "KheDrsIndexedSolnSetMake internal error");
  res->increment = drs->increment;
  res->count = 0;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsIndexedDrsSolnSetFree(KHE_DRS_INDEXED_SOLN_SET iss,           */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Free iss, but not the solutions it contains.                             */
/*                                                                           */
/*****************************************************************************/

static void KheDrsIndexedSolnSetFree(KHE_DRS_INDEXED_SOLN_SET iss,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SOLN_LIST soln_list;  int i;
  HaArrayForEach(iss->soln_lists, soln_list, i)
    if( soln_list != NULL )
      KheDrsSolnListFree(soln_list, drs);
  HaArrayAddLast(drs->indexed_soln_set_free_list, iss);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsIndexedSolnSetCount(KHE_DRS_INDEXED_SOLN_SET iss)              */
/*                                                                           */
/*  Return the number of solutions in iss.                                   */
/*                                                                           */
/*****************************************************************************/

static int KheDrsIndexedSolnSetCount(KHE_DRS_INDEXED_SOLN_SET iss)
{
  return iss->count;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsIndexedSolnSetGather(KHE_DRS_INDEXED_SOLN_SET iss,            */
/*    KHE_DRS_SOLN_LIST soln_list)                                           */
/*                                                                           */
/*  Gather the solutions of iss into soln_list.                              */
/*                                                                           */
/*****************************************************************************/

static void KheDrsIndexedSolnSetGather(KHE_DRS_INDEXED_SOLN_SET iss,
  KHE_DRS_SOLN_LIST soln_list)
{
  KHE_DRS_SOLN_LIST soln_list2;  int i;
  HaArrayForEach(iss->soln_lists, soln_list2, i)
    if( soln_list2 != NULL )
      KheDrsSolnListGather(soln_list2, soln_list);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsIndexedSolnSetDominates(KHE_DRS_INDEXED_SOLN_SET iss,         */
/*    KHE_DRS_SOLN soln, KHE_DRS_DAY soln_day,                               */
/*    KHE_DRS_DOM_TEST_TYPE dom_test_type, KHE_DYNAMIC_RESOURCE_SOLVER drs)  */
/*                                                                           */
/*  Return true if iss contains a solution that dominates soln.              */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsIndexedSolnSetDominates(KHE_DRS_INDEXED_SOLN_SET iss,
  KHE_DRS_SOLN soln, KHE_DRS_DAY soln_day, int *dom_test_count,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int i, pos;  KHE_DRS_SOLN_LIST soln_list;
  *dom_test_count = 0;
  if( HaArrayCount(iss->soln_lists) == 0 )
    return false;
  else
  {
    pos = (KheDrsSolnCost(soln) - iss->base) / iss->increment;
    if( pos >= HaArrayCount(iss->soln_lists) )
      pos = HaArrayCount(iss->soln_lists) - 1;
    for( i = 0;  i <= pos;  i++ )
    {
      soln_list = HaArray(iss->soln_lists, i);
      if( soln_list != NULL && KheDrsSolnListDominates(soln_list, soln,
	  soln_day->signer_set, dom_test_count, drs) )
	return true;
    }
  }
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsIndexedSolnSetRemoveDominated(KHE_DRS_INDEXED_SOLN_SET iss,   */
/*    KHE_DRS_SOLN soln, KHE_DRS_DAY soln_day,                               */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Delete and free the unexpanded solutions of iss that are                 */
/*  dominated by soln, including deleting them from drs->priqueue.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsIndexedSolnSetRemoveDominated(KHE_DRS_INDEXED_SOLN_SET iss,
  KHE_DRS_SOLN soln, KHE_DRS_DAY soln_day, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int i, pos, before, after;  KHE_DRS_SOLN_LIST soln_list;
  if( HaArrayCount(iss->soln_lists) > 0 )
  {
    pos = (KheDrsSolnCost(soln) - iss->base) / iss->increment;
    if( pos < 0 )
      pos = 0;
    for( i = pos;  i < HaArrayCount(iss->soln_lists);  i++ )
    {
      soln_list = HaArray(iss->soln_lists, i);
      if( soln_list != NULL )
      {
	before = KheDrsSolnListCount(soln_list);
	KheDrsSolnListRemoveDominated(soln_list, soln, soln_day->signer_set,
	  drs);
	after = KheDrsSolnListCount(soln_list);
	iss->count += (after - before);
      }
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsIndexedSolnSetHasLastSoln(KHE_DRS_INDEXED_SOLN_SET iss,       */
/*    KHE_DRS_SOLN *res)                                                     */
/*                                                                           */
/*  If iss is non-empty, return true and set *res to a solution of maximum   */
/*  cost.  Otherwise return false with *res set to NULL.                     */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsIndexedSolnSetHasLastSoln(KHE_DRS_INDEXED_SOLN_SET iss,
  KHE_DRS_SOLN *res)
{
  KHE_DRS_SOLN_LIST soln_list;  int i;
  HaArrayForEachReverse(iss->soln_lists, soln_list, i)
    if( soln_list != NULL )
    {
      *res = KheDrsSolnListFirstSoln(soln_list);
      if( *res != NULL )
	return true;
    }
  return *res = NULL, false;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsIndexedSolnSetAddSoln(KHE_DRS_INDEXED_SOLN_SET iss,           */
/*    KHE_DRS_SOLN soln, KHE_DRS_DAY soln_day, KHE_DRS_EXPANDER de,          */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Add soln to iss, assuming all the dominance testing is done.             */
/*  This includes adding soln to drs->priqueue, if required.                 */
/*                                                                           */
/*****************************************************************************/

static void KheDrsIndexedSolnSetAddSoln(KHE_DRS_INDEXED_SOLN_SET iss,
  KHE_DRS_SOLN soln, KHE_DRS_DAY soln_day, KHE_DRS_EXPANDER de,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SOLN_LIST soln_list;  int shift, pos;

  /* make sure the array contains soln_list, ready to receive soln */
  if( HaArrayCount(iss->soln_lists) == 0 )
  {
    /* array is empty, so set the base and add a soln list for soln */
    iss->base = KheDrsSolnCost(soln);
    soln_list = KheDrsSolnListMake(drs);
    HaArrayAddLast(iss->soln_lists, soln_list);
  }
  else if( iss->base > KheDrsSolnCost(soln) )
  {
    /* shift the array right, reset the base, and add a soln list for soln */
    shift = (iss->base - KheDrsSolnCost(soln)) / iss->increment;
    HnAssert(shift > 0, "KheDrsIndexedSolnSetAdd internal error");
    HaArrayShiftRight(iss->soln_lists, shift, NULL);
    iss->base = KheDrsSolnCost(soln);
    soln_list = KheDrsSolnListMake(drs);
    HaArrayPut(iss->soln_lists, 0, soln_list);
  }
  else
  {
    /* no shift or base reset, but may need to extend the array to the right */
    pos = (KheDrsSolnCost(soln) - iss->base) / iss->increment;
    HaArrayFill(iss->soln_lists, pos + 1, NULL);
    soln_list = HaArray(iss->soln_lists, pos);
    if( soln_list == NULL )
    {
      soln_list = KheDrsSolnListMake(drs);
      HaArrayPut(iss->soln_lists, pos, soln_list);
    }
  }

  /* add soln to soln_list and optionally reduce the cost limit */
  KheDrsSolnListAddSoln(soln_list, soln, drs);
  iss->count++;
  if( iss->count >= drs->solve_daily_prune_trigger &&
      KheDrsIndexedSolnSetHasLastSoln(iss, &soln) )
    KheDrsExpanderReduceCostLimit(de, KheDrsSolnCost(soln));
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsIndexedSolnSetDebug(KHE_DRS_INDEXED_SOLN_SET iss,             */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)  */
/*                                                                           */
/*  Debug print of iss onto fp with the given verbosity and index.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsIndexedSolnSetDebug(KHE_DRS_INDEXED_SOLN_SET iss,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)
{
  KHE_DRS_SOLN_LIST soln_list;  int i;  KHE_DRS_SOLN soln;
  if( indent >= 0 )
  {
    fprintf(fp, "%*s[ IndexSolnSet(base %.5f, increment %.5f, count %d)\n",
      indent, "", KheCostShow(iss->base), KheCostShow(iss->increment),
      iss->count);
    HaArrayForEach(iss->soln_lists, soln_list, i)
    {
      fprintf(fp, "%*s  [%.5f] ", indent, "",
	KheCostShow(iss->base + i * iss->increment));
      if( soln_list == NULL )
	fprintf(fp, "%3s", "-");
      else if( KheDrsSolnListCount(soln_list) == 0 )
	fprintf(fp, "%3d", 0);
      else
      {
	fprintf(fp, "%3d: ", KheDrsSolnListCount(soln_list));
	soln = HaArrayFirst(soln_list->solns);
	KheDrsSolnSignatureSetDebug(soln, fp);
      }
      fprintf(fp, "\n");
    }
    fprintf(fp, "%*s]\n", indent, "");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_SOLN_SET_PART_DOM_NONE"                               */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SOLN_SET_PART_DOM_NONE KheDrsSolnSetPartDomNoneMake(             */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new ssp dom none object.                                          */
/*                                                                           */
/*  Implementation note.  We could keep the soln list and not have to        */
/*  remake it every time, but we don't, for consistency with other types.    */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_SOLN_SET_PART_DOM_NONE KheDrsSolnSetPartDomNoneMake(
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SOLN_SET_PART_DOM_NONE res;
  if( HaArrayCount(drs->soln_set_part_dom_none_free_list) > 0 )
    res = HaArrayLastAndDelete(drs->soln_set_part_dom_none_free_list);
  else
    HaMake(res, drs->arena);
  res->soln_list = KheDrsSolnListMake(drs);
  res->dom_kind = KHE_DRS_DOM_LIST_NONE;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSetPartDomNoneFree(KHE_DRS_SOLN_SET_PART_DOM_NONE sspn,   */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Free sspn, but not the solutions it contains.                            */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSetPartDomNoneFree(KHE_DRS_SOLN_SET_PART_DOM_NONE sspn,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KheDrsSolnListFree(sspn->soln_list, drs);
  HaArrayAddLast(drs->soln_set_part_dom_none_free_list, sspn);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsSolnSetPartDomNoneCount(KHE_DRS_SOLN_SET_PART_DOM_NONE sspn)   */
/*                                                                           */
/*  Return the number of solutions in sspn.                                  */
/*                                                                           */
/*****************************************************************************/

static int KheDrsSolnSetPartDomNoneCount(KHE_DRS_SOLN_SET_PART_DOM_NONE sspn)
{
  return KheDrsSolnListCount(sspn->soln_list);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSetPartDomNoneGather(KHE_DRS_SOLN_SET_PART_DOM_NONE sspn, */
/*    KHE_DRS_SOLN_LIST soln_list)                                           */
/*                                                                           */
/*  Gather the solutions of sspn into soln_list.                             */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSetPartDomNoneGather(KHE_DRS_SOLN_SET_PART_DOM_NONE sspn,
  KHE_DRS_SOLN_LIST soln_list)
{
  KheDrsSolnListGather(sspn->soln_list, soln_list);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSolnSetPartDomNoneDominates(                                  */
/*    KHE_DRS_SOLN_SET_PART_DOM_NONE sspn, KHE_DRS_SOLN soln,                */
/*    KHE_DRS_DAY soln_day, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  Return true if sspn dominates soln.                                      */
/*                                                                           */
/*  Implementation note.  In this case we have no dominance checking, and    */
/*  so we return false here.                                                 */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsSolnSetPartDomNoneDominates(
  KHE_DRS_SOLN_SET_PART_DOM_NONE sspn, KHE_DRS_SOLN soln,
  KHE_DRS_DAY soln_day, int *dom_test_count, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSetPartDomNoneRemoveDominated(                            */
/*    KHE_DRS_SOLN_SET_PART_DOM_NONE sspn, KHE_DRS_SOLN soln,                */
/*    KHE_DRS_DAY soln_day, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  Remove the solutions of sspn that are dominated by soln.                 */
/*                                                                           */
/*  Implementation note.  In this case we have no dominance checking, and    */
/*  so there is nothing to do here.                                          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSetPartDomNoneRemoveDominated(
  KHE_DRS_SOLN_SET_PART_DOM_NONE sspn, KHE_DRS_SOLN soln,
  KHE_DRS_DAY soln_day, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  /* nothing to do here */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSetPartDomNoneAddSoln(KHE_DRS_SOLN_SET_PART_DOM_NONE sspn,*/
/*    KHE_DRS_SOLN soln, KHE_DRS_DAY soln_day,                               */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Add soln to sspn, assuming all the dominance testing is done.            */
/*  This includes adding soln to drs->priqueue, if required.                 */
/*                                                                           */
/*  Implementation note.  In this case we have no dominance checking.        */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSetPartDomNoneAddSoln(KHE_DRS_SOLN_SET_PART_DOM_NONE sspn,
  KHE_DRS_SOLN soln, KHE_DRS_DAY soln_day, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KheDrsSolnListAddSoln(sspn->soln_list, soln, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSetPartDomNoneDebug(KHE_DRS_SOLN_SET_PART_DOM_NONE sspn,  */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)  */
/*                                                                           */
/*  Debug print of sspn onto fp with the given verbosity and indent.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSetPartDomNoneDebug(KHE_DRS_SOLN_SET_PART_DOM_NONE sspn,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)
{
  KheDrsSolnListDebug(sspn->soln_list, drs, verbosity, indent, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_SOLN_SET_PART_DOM_WEAK"                               */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SOLN_SET_PART_DOM_WEAK KheDrsSolnSetPartDomWeakMake(             */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new ssp dom weak object.                                          */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_SOLN_SET_PART_DOM_WEAK KheDrsSolnSetPartDomWeakMake(
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SOLN_SET_PART_DOM_WEAK res;
  if( HaArrayCount(drs->soln_set_part_dom_weak_free_list) > 0 )
  {
    res = HaArrayLastAndDelete(drs->soln_set_part_dom_weak_free_list);
    HpTableClear(res->soln_table);
  }
  else
  {
    HaMake(res, drs->arena);
    HpTableInit(res->soln_table, &KheDrsSolnSignatureSetFullHashUntyped,
      &KheDrsSolnSignatureSetFullEqualUntyped,
      &KheDrsSolnSignatureSetDebugUntyped, drs->arena);
  }
  res->dom_kind = KHE_DRS_DOM_HASH_WEAK;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSetPartDomWeakFree(KHE_DRS_SOLN_SET_PART_DOM_WEAK sspw,   */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Free sspw, but not the solutions it contains.                            */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSetPartDomWeakFree(KHE_DRS_SOLN_SET_PART_DOM_WEAK sspw,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  HaArrayAddLast(drs->soln_set_part_dom_weak_free_list, sspw);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsSolnSetPartDomWeakCount(KHE_DRS_SOLN_SET_PART_DOM_WEAK sspw)   */
/*                                                                           */
/*  Return the number of solutions in sspw.                                  */
/*                                                                           */
/*****************************************************************************/

static int KheDrsSolnSetPartDomWeakCount(KHE_DRS_SOLN_SET_PART_DOM_WEAK sspw)
{
  KHE_DRS_SOLN soln;  int pos, res;
  res = 0;
  HpTableForEachValue(sspw->soln_table, soln, pos)
    res++;
  soln = soln;  /* avoid stupid compiler warning */
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSetPartDomWeakGather(KHE_DRS_SOLN_SET_PART_DOM_WEAK sspw, */
/*    KHE_DRS_SOLN_LIST soln_list)                                           */
/*    KHE_DRS_SOLN_SET soln_set)                                             */
/*                                                                           */
/*  Gather the solutions of sspw into soln_list.                             */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSetPartDomWeakGather(KHE_DRS_SOLN_SET_PART_DOM_WEAK sspw,
  KHE_DRS_SOLN_LIST soln_list)
{
  KHE_DRS_SOLN soln;  int i;
  HpTableForEachValue(sspw->soln_table, soln, i)
    HaArrayAddLast(soln_list->solns, soln);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSolnSetPartDomWeakDominates(                                  */
/*    KHE_DRS_SOLN_SET_PART_DOM_WEAK sspw, KHE_DRS_SOLN soln,                */
/*    KHE_DRS_DAY soln_day, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  Return true if sspw dominates soln.                                      */
/*                                                                           */
/*  Implementation note.  Weak dominance does not really fit into the        */
/*  usual pattern when it comes to inserting solutions, so we do nothing     */
/*  here, leaving the whole task for later.                                  */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsSolnSetPartDomWeakDominates(
  KHE_DRS_SOLN_SET_PART_DOM_WEAK sspn, KHE_DRS_SOLN soln,
  KHE_DRS_DAY soln_day, int *dom_test_count, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSetPartDomWeakRemoveDominated(                            */
/*    KHE_DRS_SOLN_SET_PART_DOM_WEAK sspw, KHE_DRS_SOLN soln,                */
/*    KHE_DRS_DAY soln_day, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  Remove the solutions of sspw that are dominated by soln.                 */
/*                                                                           */
/*  Implementation note.  Weak dominance does not really fit into the        */
/*  usual pattern when it comes to inserting solutions, so we do nothing     */
/*  here, leaving the whole task for later.                                  */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSetPartDomWeakRemoveDominated(
  KHE_DRS_SOLN_SET_PART_DOM_WEAK sspw, KHE_DRS_SOLN soln,
  KHE_DRS_DAY soln_day, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  /* nothing to do here */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSetPartDomWeakAddSoln(KHE_DRS_SOLN_SET_PART_DOM_WEAK sspw,*/
/*    KHE_DRS_SOLN soln, KHE_DRS_DAY soln_day,                               */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Add soln to sspw, assuming all the dominance testing is done.            */
/*  This includes adding soln to drs->priqueue, if required.                 */
/*                                                                           */
/*  Implementation note 1.  Weak dominance does not really fit into the      */
/*  usual pattern when it comes to inserting solutions, so we do nothing     */
/*  earlier, and carry out the whole task here (including dominance).        */
/*                                                                           */
/*  Implementation note 2.  Weak dominance can change an existing solution,  */
/*  as opposed to deleting an old one and inserting a new one.  So we use    */
/*  KhePriQueueNotifyKeyChange in that case to notify the priority queue     */
/*  that the soln's cost has changed.                                        */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSetPartDomWeakAddSoln(KHE_DRS_SOLN_SET_PART_DOM_WEAK sspw,
  KHE_DRS_SOLN soln, KHE_DRS_DAY soln_day, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SOLN other_soln;
  if( HpTableAddUnique(sspw->soln_table, (void *) soln, soln, other_soln))
    KheDrsPriQueueAddSoln(drs, soln);
  else
  {
    /* ensure other_soln is the better of other_soln and soln */
    if( KheDrsSolnCost(soln) < KheDrsSolnCost(other_soln) )
    {
      KheDrsSolnOverWrite(other_soln, soln);
      KheDrsPriQueueNotifyKeyChangeSoln(drs, soln);
    }

    /* free soln */
    KheDrsSolnFree(soln, drs);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSetPartDomWeakDebug(KHE_DRS_SOLN_SET_PART_DOM_WEAK sspw,  */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)  */
/*                                                                           */
/*  Debug print of sspw onto fp with the given verbosity and indent.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSetPartDomWeakDebug(KHE_DRS_SOLN_SET_PART_DOM_WEAK sspw,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)
{
  KHE_DRS_SOLN soln;  int i;
  fprintf(fp, "%*s[ WeakTable\n", indent, "");
  HpTableForEachValue(sspw->soln_table, soln, i)
    KheDrsSolnDebug(soln, drs, verbosity, indent + 2, fp);
  fprintf(fp, "%*s]\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_SOLN_SET_PART_DOM_MEDIUM"                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SOLN_SET_PART_DOM_MEDIUM KheDrsSolnSetPartDomMediumMake(         */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new ssp dom medium object.                                        */
/*                                                                           */
/*  Implementation note.  When we free an sspm, we free its solution         */
/*  lists but not its table.                                                 */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_SOLN_SET_PART_DOM_MEDIUM KheDrsSolnSetPartDomMediumMake(
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SOLN_SET_PART_DOM_MEDIUM res;
  if( HaArrayCount(drs->soln_set_part_dom_medium_free_list) > 0 )
  {
    res = HaArrayLastAndDelete(drs->soln_set_part_dom_medium_free_list);
    HpTableClear(res->soln_list_table);
  }
  else
  {
    HaMake(res, drs->arena);
    HpTableInit(res->soln_list_table,
      &KheDrsSolnListSignatureSetPartialHashUntyped,
      &KheDrsSolnListSignatureSetPartialEqualUntyped,
      &KheDrsSolnListSignatureSetDebugUntyped, drs->arena);
  }
  res->dom_kind = KHE_DRS_DOM_HASH_MEDIUM;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSetPartDomMediumFree(                                     */
/*    KHE_DRS_SOLN_SET_PART_DOM_MEDIUM sspm, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  Free sspm, but not the solutions it contains.                            */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSetPartDomMediumFree(
  KHE_DRS_SOLN_SET_PART_DOM_MEDIUM sspm, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SOLN_LIST soln_list;  int pos;
  HpTableForEachValue(sspm->soln_list_table, soln_list, pos)
    KheDrsSolnListFree(soln_list, drs);
  HaArrayAddLast(drs->soln_set_part_dom_medium_free_list, sspm);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsSolnSetPartDomMediumCount(                                     */
/*    KHE_DRS_SOLN_SET_PART_DOM_MEDIUM sspm)                                 */
/*                                                                           */
/*  Return the number of solutions in sspm.                                  */
/*                                                                           */
/*****************************************************************************/

static int KheDrsSolnSetPartDomMediumCount(
  KHE_DRS_SOLN_SET_PART_DOM_MEDIUM sspm)
{
  KHE_DRS_SOLN_LIST soln_list;  int res, pos;
  res = 0;
  HpTableForEachValue(sspm->soln_list_table, soln_list, pos)
    res += KheDrsSolnListCount(soln_list);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSetPartDomMediumGather(                                   */
/*    KHE_DRS_SOLN_SET_PART_DOM_MEDIUM sspm, KHE_DYNAMIC_RESOURCE_SOLVER drs,*/
/*    KHE_DRS_SOLN_LIST soln_list)                                           */
/*                                                                           */
/*  Gather the solutions of sspm into soln_list, then free sspm.             */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSetPartDomMediumGather(
  KHE_DRS_SOLN_SET_PART_DOM_MEDIUM sspm, KHE_DRS_SOLN_LIST soln_list)
{
  KHE_DRS_SOLN_LIST soln_list2;  int i;
  HpTableForEachValue(sspm->soln_list_table, soln_list2, i)
    KheDrsSolnListGather(soln_list2, soln_list);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSolnSetPartDomMediumDominates(                                */
/*    KHE_DRS_SOLN_SET_PART_DOM_MEDIUM sspm, KHE_DRS_SOLN soln,              */
/*    KHE_DRS_DAY soln_day, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  Return true if sspm dominates soln.                                      */
/*                                                                           */
/*  Implementation note.  Medium dominance does not really fit into the      */
/*  usual pattern when it comes to inserting solutions, so we do nothing     */
/*  here, leaving the whole task for later.                                  */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsSolnSetPartDomMediumDominates(
  KHE_DRS_SOLN_SET_PART_DOM_MEDIUM sspn, KHE_DRS_SOLN soln,
  KHE_DRS_DAY soln_day, int *dom_test_count, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSetPartDomMediumRemoveDominated(                          */
/*    KHE_DRS_SOLN_SET_PART_DOM_MEDIUM sspm, KHE_DRS_SOLN soln,              */
/*    KHE_DRS_DAY soln_day, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  Remove the solutions of sspm that are dominated by soln.                 */
/*                                                                           */
/*  Implementation note.  Medium dominance does not really fit into the      */
/*  usual pattern when it comes to inserting solutions, so we do nothing     */
/*  here, leaving the whole task for later.                                  */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSetPartDomMediumRemoveDominated(
  KHE_DRS_SOLN_SET_PART_DOM_MEDIUM sspm, KHE_DRS_SOLN soln,
  KHE_DRS_DAY soln_day, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  /* nothing to do here */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSetPartDomMediumAddSoln(                                  */
/*    KHE_DRS_SOLN_SET_PART_DOM_MEDIUM sspm,                                 */
/*    KHE_DRS_SOLN soln, KHE_DRS_DAY soln_day,                               */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Add soln to sspm, assuming all the dominance testing is done.            */
/*  This includes adding soln to drs->priqueue, if required.                 */
/*                                                                           */
/*  Implementation note.  Medium dominance does not really fit into the      */
/*  usual pattern when it comes to inserting solutions, so we do nothing     */
/*  earlier, and carry out the whole task here (including dominance).        */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSetPartDomMediumAddSoln(
  KHE_DRS_SOLN_SET_PART_DOM_MEDIUM sspm,
  KHE_DRS_SOLN soln, KHE_DRS_DAY soln_day, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SOLN_LIST soln_list, other_soln_list;  int dom_test_count;

  /* make soln_list holding just soln and try adding it to soln_list_table */
  dom_test_count = 0;
  soln_list = KheDrsSolnListMake(drs);
  HaArrayAddLast(soln_list->solns, soln);
  if( HpTableAddUnique(sspm->soln_list_table, (void *) soln_list, soln_list,
	other_soln_list) )
  {
    /* soln_list added, so that's all that's needed */
    KheDrsPriQueueAddSoln(drs, soln);
  }
  else
  {
    /* free soln_list and do dominance test of soln with other_soln_list */
    KheDrsSolnListFree(soln_list, drs);
    if( KheDrsSolnListDominates(other_soln_list, soln, soln_day->signer_set,
	  &dom_test_count, drs) )
    {
      /* soln is dominated by some other soln, so delete soln */
      KheDrsSolnFree(soln, drs);
    }
    else
    {
      /* remove other solns that soln dominates, and add soln to soln_list */
      KheDrsSolnListRemoveDominated(other_soln_list, soln,
	soln_day->signer_set, drs);
      KheDrsSolnListAddSoln(other_soln_list, soln, drs);
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSetPartDomMediumDebug(                                    */
/*    KHE_DRS_SOLN_SET_PART_DOM_MEDIUM sspm,                                 */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)  */
/*                                                                           */
/*  Debug print of sspm onto fp with the given verbosity and indent.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSetPartDomMediumDebug(
  KHE_DRS_SOLN_SET_PART_DOM_MEDIUM sspm,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)
{
  KHE_DRS_SOLN_LIST soln_list;  int i;
  fprintf(fp, "%*s[ MediumTable\n", indent, "");
  HpTableForEachValue(sspm->soln_list_table, soln_list, i)
    KheDrsSolnListDebug(soln_list, drs, verbosity, indent, fp);
  fprintf(fp, "%*s]\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_SOLN_SET_PART_DOM_STRONG"                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SOLN_SET_PART_DOM_STRONG KheDrsSolnSetPartDomStrongMake(         */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new ssp dom strong object.                                        */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_SOLN_SET_PART_DOM_STRONG KheDrsSolnSetPartDomStrongMake(
  KHE_DRS_DOM_KIND dom_kind, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SOLN_SET_PART_DOM_STRONG res;
  if( HaArrayCount(drs->soln_set_part_dom_strong_free_list) > 0 )
    res = HaArrayLastAndDelete(drs->soln_set_part_dom_strong_free_list);
  else
    HaMake(res, drs->arena);
  res->soln_list = KheDrsSolnListMake(drs);
  res->dom_kind = dom_kind;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSetPartDomStrongFree(                                     */
/*    KHE_DRS_SOLN_SET_PART_DOM_STRONG ssps, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  Free sspt, but not the solutions it contains.                            */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSetPartDomStrongFree(
  KHE_DRS_SOLN_SET_PART_DOM_STRONG ssps, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KheDrsSolnListFree(ssps->soln_list, drs);
  HaArrayAddLast(drs->soln_set_part_dom_strong_free_list, ssps);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsSolnSetPartDomStrongCount(                                     */
/*    KHE_DRS_SOLN_SET_PART_DOM_STRONG ssps)                                 */
/*                                                                           */
/*  Return the number of solutions in ssps.                                  */
/*                                                                           */
/*****************************************************************************/

static int KheDrsSolnSetPartDomStrongCount(
  KHE_DRS_SOLN_SET_PART_DOM_STRONG ssps)
{
  return KheDrsSolnListCount(ssps->soln_list);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSetPartDomStrongGather(                                   */
/*    KHE_DRS_SOLN_SET_PART_DOM_STRONG ssps, KHE_DRS_SOLN_LIST soln_list)    */
/*                                                                           */
/*  Gather the solutions of ssps into soln_list.                             */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSetPartDomStrongGather(
  KHE_DRS_SOLN_SET_PART_DOM_STRONG ssps, KHE_DRS_SOLN_LIST soln_list)
{
  KheDrsSolnListGather(ssps->soln_list, soln_list);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSolnSetPartDomStrongDominates(                                */
/*    KHE_DRS_SOLN_SET_PART_DOM_STRONG ssps, KHE_DRS_SOLN soln,              */
/*    KHE_DRS_DAY soln_day, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  Return true if ssps dominates soln.                                      */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsSolnSetPartDomStrongDominates(
  KHE_DRS_SOLN_SET_PART_DOM_STRONG ssps, KHE_DRS_SOLN soln,
  KHE_DRS_DAY soln_day, int *dom_test_count, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  return KheDrsSolnListDominates(ssps->soln_list, soln, soln_day->signer_set,
    dom_test_count, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSetPartDomStrongRemoveDominated(                          */
/*    KHE_DRS_SOLN_SET_PART_DOM_STRONG ssps, KHE_DRS_SOLN soln,              */
/*    KHE_DRS_DAY soln_day, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  Delete and free the unexpanded solutions of ssps that are                */
/*  dominated by soln, including deleting them from drs->priqueue.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSetPartDomStrongRemoveDominated(
  KHE_DRS_SOLN_SET_PART_DOM_STRONG ssps, KHE_DRS_SOLN soln,
  KHE_DRS_DAY soln_day, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KheDrsSolnListRemoveDominated(ssps->soln_list, soln, soln_day->signer_set,
    drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSetPartDomStrongAddSoln(                                  */
/*    KHE_DRS_SOLN_SET_PART_DOM_STRONG ssps, KHE_DRS_SOLN soln,              */
/*    KHE_DRS_DAY soln_day, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  Add soln to ssps, assuming all the dominance testing is done.            */
/*  This includes adding soln to drs->priqueue, if required.                 */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSetPartDomStrongAddSoln(
  KHE_DRS_SOLN_SET_PART_DOM_STRONG ssps, KHE_DRS_SOLN soln,
  KHE_DRS_DAY soln_day, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KheDrsSolnListAddSoln(ssps->soln_list, soln, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSetPartDomStrongDebug(                                    */
/*    KHE_DRS_SOLN_SET_PART_DOM_STRONG ssps,                                 */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)  */
/*                                                                           */
/*  Debug print of ssps onto fp with the given verbosity and indent.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSetPartDomStrongDebug(
  KHE_DRS_SOLN_SET_PART_DOM_STRONG ssps,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)
{
  KheDrsSolnListDebug(ssps->soln_list, drs, verbosity, indent, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_SOLN_SET_PART_DOM_TRIE"                               */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SOLN_SET_PART_DOM_TRIE KheDrsSolnSetPartDomTrieMake(             */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new ssp dom trie object.                                         */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_SOLN_SET_PART_DOM_TRIE KheDrsSolnSetPartDomTrieMake(
  KHE_DRS_DOM_KIND dom_kind, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SOLN_SET_PART_DOM_TRIE res;
  if( HaArrayCount(drs->soln_set_part_dom_trie_free_list) > 0 )
    res = HaArrayLastAndDelete(drs->soln_set_part_dom_trie_free_list);
  else
    HaMake(res, drs->arena);
  res->dom_kind = dom_kind;
  res->soln_trie = NULL;  /* represents an empty trie */
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSetPartDomTrieFree(KHE_DRS_SOLN_SET_PART_DOM_TRIE sspt,   */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Free sspt but not its elements.                                          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSetPartDomTrieFree(KHE_DRS_SOLN_SET_PART_DOM_TRIE sspt,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  if( DEBUG26 )
    KheDrsSolnTrieDebugNullInitialChildren(sspt->soln_trie);
  KheDrsSolnTrieFree(&sspt->soln_trie, drs);
  HnAssert(sspt->soln_trie == NULL,
    "KheDrsSolnSetPartDomTrieFree internal error");
  HaArrayAddLast(drs->soln_set_part_dom_trie_free_list, sspt);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsSolnSetPartDomTrieCount(KHE_DRS_SOLN_SET_PART_DOM_TRIE sspt)   */
/*                                                                           */
/*  Return the number of solutions in sspt.                                  */
/*                                                                           */
/*****************************************************************************/

static int KheDrsSolnSetPartDomTrieCount(KHE_DRS_SOLN_SET_PART_DOM_TRIE sspt)
{
  return KheDrsSolnTrieCount(sspt->soln_trie);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSetPartDomTrieGather(                                     */
/*    KHE_DRS_SOLN_SET_PART_DOM_TRIE sspt, KHE_DRS_SOLN_LIST soln_list)      */
/*                                                                           */
/*  Gather the solutions of sspt into soln_list.                             */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSetPartDomTrieGather(
  KHE_DRS_SOLN_SET_PART_DOM_TRIE sspt, KHE_DRS_SOLN_LIST soln_list)
{
  KheDrsSolnTrieGather(sspt->soln_trie, soln_list);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSolnSetPartDomTrieDominates(                                  */
/*    KHE_DRS_SOLN_SET_PART_DOM_TRIE sspt, KHE_DRS_SOLN soln,                */
/*    KHE_DRS_DAY soln_day, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  Return true if sspt dominates soln.                                      */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsSolnSetPartDomTrieDominates(
  KHE_DRS_SOLN_SET_PART_DOM_TRIE sspt, KHE_DRS_SOLN soln,
  KHE_DRS_DAY soln_day, int *dom_test_count, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  return KheDrsSolnTrieDoDominates(sspt->soln_trie, soln, soln_day,
    0, 0, dom_test_count, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSetPartDomTrieRemoveDominated(                            */
/*    KHE_DRS_SOLN_SET_PART_DOM_TRIE sspt, KHE_DRS_SOLN soln,                */
/*    KHE_DRS_DAY soln_day, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  Delete and free the unexpanded solutions of sspt that are                */
/*  dominated by soln, including deleting them from drs->priqueue.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSetPartDomTrieRemoveDominated(
  KHE_DRS_SOLN_SET_PART_DOM_TRIE sspt, KHE_DRS_SOLN soln,
  KHE_DRS_DAY soln_day, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KheDrsSolnTrieRemoveDominated(&sspt->soln_trie, soln, soln_day, 0, 0, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSetPartDomTrieAddSoln(KHE_DRS_SOLN_SET_PART_DOM_TRIE sspt,*/
/*    KHE_DRS_SOLN soln, KHE_DRS_DAY soln_day,                               */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Add soln to sspt, assuming all the dominance testing is done.            */
/*  This includes adding soln to drs->priqueue, if required.                 */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSetPartDomTrieAddSoln(KHE_DRS_SOLN_SET_PART_DOM_TRIE sspt,
  KHE_DRS_SOLN soln, KHE_DRS_DAY soln_day, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KheDrsSolnTrieAddSoln(&sspt->soln_trie, soln, 0, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSetPartDomTrieDebug(KHE_DRS_SOLN_SET_PART_DOM_TRIE sspt,  */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)  */
/*                                                                           */
/*  Debug print of sspt onto fp with the given verbosity and indent.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSetPartDomTrieDebug(KHE_DRS_SOLN_SET_PART_DOM_TRIE sspt,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)
{
  KheDrsSolnTrieDebug(sspt->soln_trie, drs, verbosity, indent, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_SOLN_SET_PART_DOM_INDEXED"                            */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SOLN_SET_PART_DOM_INDEXED KheDrsSolnSetPartDomIndexedMake(       */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new ssp dom indexed object.                                       */
/*                                                                           */
/*  Implementation note.  Even when we retrieve an existing object from      */
/*  drs->soln_set_part_dom_indexed_free_list, still we need to create a      */
/*  fresh indexed set.                                                       */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_SOLN_SET_PART_DOM_INDEXED KheDrsSolnSetPartDomIndexedMake(
  KHE_DRS_DOM_KIND dom_kind, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SOLN_SET_PART_DOM_INDEXED res;
  if( HaArrayCount(drs->soln_set_part_dom_indexed_free_list) > 0 )
    res = HaArrayLastAndDelete(drs->soln_set_part_dom_indexed_free_list);
  else
    HaMake(res, drs->arena);
  res->dom_kind = dom_kind;
  res->indexed_solns = KheDrsIndexedSolnSetMake(drs);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSetPartDomIndexedFree(                                    */
/*    KHE_DRS_SOLN_SET_PART_DOM_INDEXED sspi,                                */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Free sspi, but not the solutions it contains.                            */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSetPartDomIndexedFree(
  KHE_DRS_SOLN_SET_PART_DOM_INDEXED sspi,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KheDrsIndexedSolnSetFree(sspi->indexed_solns, drs);
  HaArrayAddLast(drs->soln_set_part_dom_indexed_free_list, sspi);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsSolnSetPartDomIndexedCount(                                    */
/*    KHE_DRS_SOLN_SET_PART_DOM_INDEXED sspi)                                */
/*                                                                           */
/*  Return the number of solutions in sspi.                                  */
/*                                                                           */
/*****************************************************************************/

static int KheDrsSolnSetPartDomIndexedCount(
  KHE_DRS_SOLN_SET_PART_DOM_INDEXED sspi)
{
  return KheDrsIndexedSolnSetCount(sspi->indexed_solns);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSetPartDomIndexedGather(                                  */
/*    KHE_DRS_SOLN_SET_PART_DOM_INDEXED sspi, KHE_DRS_SOLN_LIST soln_list)   */
/*                                                                           */
/*  Gather the solutions of sspi into soln_list.                             */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSetPartDomIndexedGather(
  KHE_DRS_SOLN_SET_PART_DOM_INDEXED sspi, KHE_DRS_SOLN_LIST soln_list)
{
  KheDrsIndexedSolnSetGather(sspi->indexed_solns, soln_list);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSolnSetPartDomIndexedDominates(                               */
/*    KHE_DRS_SOLN_SET_PART_DOM_INDEXED sspi, KHE_DRS_SOLN soln,             */
/*    KHE_DRS_DAY soln_day, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  Return true if sspi dominates soln.                                      */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsSolnSetPartDomIndexedDominates(
  KHE_DRS_SOLN_SET_PART_DOM_INDEXED sspi, KHE_DRS_SOLN soln,
  KHE_DRS_DAY soln_day, int *dom_test_count, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  return KheDrsIndexedSolnSetDominates(sspi->indexed_solns, soln, soln_day,
    dom_test_count, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSetPartDomIndexedRemoveDominated(                         */
/*    KHE_DRS_SOLN_SET_PART_DOM_INDEXED sspi, KHE_DRS_SOLN soln,             */
/*    KHE_DRS_DAY soln_day, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  Delete and free the unexpanded solutions of sspi that are                */
/*  dominated by soln, including deleting them from drs->priqueue.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSetPartDomIndexedRemoveDominated(
  KHE_DRS_SOLN_SET_PART_DOM_INDEXED sspi, KHE_DRS_SOLN soln,
  KHE_DRS_DAY soln_day, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KheDrsIndexedSolnSetRemoveDominated(sspi->indexed_solns, soln, soln_day, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSetPartDomIndexedAddSoln(                                 */
/*    KHE_DRS_SOLN_SET_PART_DOM_INDEXED sspi, KHE_DRS_SOLN soln,             */
/*    KHE_DRS_DAY soln_day, KHE_DRS_EXPANDER de,                             */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Add soln to sspi, assuming all the dominance testing is done.            */
/*  This includes adding soln to drs->priqueue, if required.                 */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSetPartDomIndexedAddSoln(
  KHE_DRS_SOLN_SET_PART_DOM_INDEXED sspi, KHE_DRS_SOLN soln,
  KHE_DRS_DAY soln_day, KHE_DRS_EXPANDER de,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KheDrsIndexedSolnSetAddSoln(sspi->indexed_solns, soln, soln_day, de, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSetPartDomIndexedDebug(                                   */
/*    KHE_DRS_SOLN_SET_PART_DOM_INDEXED sspi,                                */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)  */
/*                                                                           */
/*  Debug print of sspi onto fp with the given verbosity and indent.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSetPartDomIndexedDebug(
  KHE_DRS_SOLN_SET_PART_DOM_INDEXED sspi,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)
{
  KheDrsIndexedSolnSetDebug(sspi->indexed_solns, drs, verbosity, indent, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_SOLN_SET_PART_DOM_UNIFORM"                            */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SOLN_SET_PART_DOM_UNIFORM KheDrsSolnSetPartDomUniformMake(       */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new ssp dom strong object.                                        */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_SOLN_SET_PART_DOM_UNIFORM KheDrsSolnSetPartDomUniformMake(
  KHE_DRS_DOM_KIND dom_kind, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SOLN_SET_PART_DOM_UNIFORM res;
  if( HaArrayCount(drs->soln_set_part_dom_uniform_free_list) > 0 )
    res = HaArrayLastAndDelete(drs->soln_set_part_dom_uniform_free_list);
  else
    HaMake(res, drs->arena);
  res->soln_list = KheDrsSolnListMake(drs);
  res->dom_kind = dom_kind;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSetPartDomUniformFree(                                    */
/*    KHE_DRS_SOLN_SET_PART_DOM_UNIFORM sspu,                                */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Free sspt, but not the solutions it contains.                            */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSetPartDomUniformFree(
  KHE_DRS_SOLN_SET_PART_DOM_UNIFORM sspu, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KheDrsSolnListFree(sspu->soln_list, drs);
  HaArrayAddLast(drs->soln_set_part_dom_uniform_free_list, sspu);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsSolnSetPartDomUniformCount(                                    */
/*    KHE_DRS_SOLN_SET_PART_DOM_UNIFORM sspu)                                */
/*                                                                           */
/*  Return the number of solutions in sspu.                                  */
/*                                                                           */
/*****************************************************************************/

static int KheDrsSolnSetPartDomUniformCount(
  KHE_DRS_SOLN_SET_PART_DOM_UNIFORM sspu)
{
  return KheDrsSolnListCount(sspu->soln_list);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSetPartDomUniformGather(                                  */
/*    KHE_DRS_SOLN_SET_PART_DOM_UNIFORM sspu, KHE_DRS_SOLN_LIST soln_list)   */
/*                                                                           */
/*  Gather the solutions of sspu into soln_list.                             */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSetPartDomUniformGather(
  KHE_DRS_SOLN_SET_PART_DOM_UNIFORM sspu, KHE_DRS_SOLN_LIST soln_list)
{
  KheDrsSolnListGather(sspu->soln_list, soln_list);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSolnSetPartDomUniformDominates(                               */
/*    KHE_DRS_SOLN_SET_PART_DOM_UNIFORM sspu, KHE_DRS_SOLN soln,             */
/*    KHE_DRS_DAY soln_day, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  Return true if sspu dominates soln.                                      */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsSolnSetPartDomUniformDominates(
  KHE_DRS_SOLN_SET_PART_DOM_UNIFORM sspu, KHE_DRS_SOLN soln,
  KHE_DRS_DAY soln_day, int *dom_test_count, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  return KheDrsSolnListDominates(sspu->soln_list, soln, soln_day->signer_set,
    /* KHE_DRS_DOM_TEST_UNIFORM, */ dom_test_count, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSetPartDomUniformRemoveDominated(                         */
/*    KHE_DRS_SOLN_SET_PART_DOM_UNIFORM sspu, KHE_DRS_SOLN soln,             */
/*    KHE_DRS_DAY soln_day, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  Delete and free the unexpanded solutions of sspu that are                */
/*  dominated by soln, including deleting them from drs->priqueue.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSetPartDomUniformRemoveDominated(
  KHE_DRS_SOLN_SET_PART_DOM_UNIFORM sspu, KHE_DRS_SOLN soln,
  KHE_DRS_DAY soln_day, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KheDrsSolnListRemoveDominated(sspu->soln_list, soln, soln_day->signer_set,
    drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSetPartDomUniformAddSoln(                                 */
/*    KHE_DRS_SOLN_SET_PART_DOM_UNIFORM sspu, KHE_DRS_SOLN soln,             */
/*    KHE_DRS_DAY soln_day, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  Add soln to sspu, assuming all the dominance testing is done.            */
/*  This includes adding soln to drs->priqueue, if required.                 */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSetPartDomUniformAddSoln(
  KHE_DRS_SOLN_SET_PART_DOM_UNIFORM sspu, KHE_DRS_SOLN soln,
  KHE_DRS_DAY soln_day, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KheDrsSolnListAddSoln(sspu->soln_list, soln, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSetPartDomUniformDebug(                                    */
/*    KHE_DRS_SOLN_SET_PART_DOM_UNIFORM sspu,                                 */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)  */
/*                                                                           */
/*  Debug print of sspu onto fp with the given verbosity and indent.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSetPartDomUniformDebug(
  KHE_DRS_SOLN_SET_PART_DOM_UNIFORM sspu,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)
{
  KheDrsSolnListDebug(sspu->soln_list, drs, verbosity, indent, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_SOLN_SET_PART_DOM_INDEXED_UNIFORM"                    */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SOLN_SET_PART_DOM_INDEXED_UNIFORM                                */
/*    KheDrsSolnSetPartDomIndexedUniformMake(KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  Make a new ssp dom indexed uniform object.                               */
/*                                                                           */
/*  Implementation note.  Even when we retrieve an existing object from      */
/*  drs->soln_set_part_dom_indexed_uniform_free_list, still we need to       */
/*  create a fresh indexed set.                                              */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_SOLN_SET_PART_DOM_INDEXED_UNIFORM
  KheDrsSolnSetPartDomIndexedUniformMake(
  KHE_DRS_DOM_KIND dom_kind, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SOLN_SET_PART_DOM_INDEXED_UNIFORM res;
  if( HaArrayCount(drs->soln_set_part_dom_indexed_uniform_free_list) > 0 )
   res = HaArrayLastAndDelete(drs->soln_set_part_dom_indexed_uniform_free_list);
  else
    HaMake(res, drs->arena);
  res->dom_kind = dom_kind;
  res->indexed_solns = KheDrsIndexedSolnSetMake(drs);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSetPartDomIndexedUniformFree(                             */
/*    KHE_DRS_SOLN_SET_PART_DOM_INDEXED_UNIFORM sspiu,                       */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Free sspiu, but not the solutions it contains.                           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSetPartDomIndexedUniformFree(
  KHE_DRS_SOLN_SET_PART_DOM_INDEXED_UNIFORM sspiu,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KheDrsIndexedSolnSetFree(sspiu->indexed_solns, drs);
  HaArrayAddLast(drs->soln_set_part_dom_indexed_uniform_free_list, sspiu);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsSolnSetPartDomIndexedUniformCount(                             */
/*    KHE_DRS_SOLN_SET_PART_DOM_INDEXED_UNIFORM sspiu)                       */
/*                                                                           */
/*  Return the number of solutions in sspiu.                                 */
/*                                                                           */
/*****************************************************************************/

static int KheDrsSolnSetPartDomIndexedUniformCount(
  KHE_DRS_SOLN_SET_PART_DOM_INDEXED_UNIFORM sspiu)
{
  return KheDrsIndexedSolnSetCount(sspiu->indexed_solns);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSetPartDomIndexedUniformGather(                           */
/*    KHE_DRS_SOLN_SET_PART_DOM_INDEXED_UNIFORM sspiu,                       */
/*    KHE_DRS_SOLN_LIST soln_list)                                           */
/*                                                                           */
/*  Gather the solutions of sspiu into soln_list.                            */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSetPartDomIndexedUniformGather(
  KHE_DRS_SOLN_SET_PART_DOM_INDEXED_UNIFORM sspiu, KHE_DRS_SOLN_LIST soln_list)
{
  KheDrsIndexedSolnSetGather(sspiu->indexed_solns, soln_list);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSolnSetPartDomIndexedUniformDominates(                        */
/*    KHE_DRS_SOLN_SET_PART_DOM_INDEXED_UNIFORM sspiu, KHE_DRS_SOLN soln,    */
/*    KHE_DRS_DAY soln_day, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  Return true if sspiu dominates soln.                                     */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsSolnSetPartDomIndexedUniformDominates(
  KHE_DRS_SOLN_SET_PART_DOM_INDEXED_UNIFORM sspiu, KHE_DRS_SOLN soln,
  KHE_DRS_DAY soln_day, int *dom_test_count, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  return KheDrsIndexedSolnSetDominates(sspiu->indexed_solns,
    soln, soln_day, dom_test_count, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSetPartDomIndexedUniformRemoveDominated(                  */
/*    KHE_DRS_SOLN_SET_PART_DOM_INDEXED_UNIFORM sspiu, KHE_DRS_SOLN soln,    */
/*    KHE_DRS_DAY soln_day, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  Delete and free the unexpanded solutions of sspiu that are               */
/*  dominated by soln, including deleting them from drs->priqueue.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSetPartDomIndexedUniformRemoveDominated(
  KHE_DRS_SOLN_SET_PART_DOM_INDEXED_UNIFORM sspiu, KHE_DRS_SOLN soln,
  KHE_DRS_DAY soln_day, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KheDrsIndexedSolnSetRemoveDominated(sspiu->indexed_solns,
    soln, soln_day, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSetPartDomIndexedUniformAddSoln(                          */
/*    KHE_DRS_SOLN_SET_PART_DOM_INDEXED_UNIFORM sspiu, KHE_DRS_SOLN soln,    */
/*    KHE_DRS_DAY soln_day, KHE_DRS_EXPANDER de,                             */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Add soln to sspiu, assuming all the dominance testing is done.           */
/*  This includes adding soln to drs->priqueue, if required.                 */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSetPartDomIndexedUniformAddSoln(
  KHE_DRS_SOLN_SET_PART_DOM_INDEXED_UNIFORM sspiu, KHE_DRS_SOLN soln,
  KHE_DRS_DAY soln_day, KHE_DRS_EXPANDER de,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KheDrsIndexedSolnSetAddSoln(sspiu->indexed_solns, soln, soln_day, de, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSetPartDomIndexedUniformDebug(                            */
/*    KHE_DRS_SOLN_SET_PART_DOM_INDEXED_UNIFORM sspiu,                       */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)  */
/*                                                                           */
/*  Debug print of sspiu onto fp with the given verbosity and indent.        */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSetPartDomIndexedUniformDebug(
  KHE_DRS_SOLN_SET_PART_DOM_INDEXED_UNIFORM sspiu,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)
{
  KheDrsIndexedSolnSetDebug(sspiu->indexed_solns, drs, verbosity, indent, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_SOLN_SET_PART"                                        */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SOLN_SET_PART KheDrsSolnSetPartMake(KHE_DRS_DOM_KIND dom_kind,   */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Return a new KHE_DRS_SOLN_SET_PART object.                               */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_SOLN_SET_PART KheDrsSolnSetPartMake(KHE_DRS_DOM_KIND dom_kind,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  switch( dom_kind )
  {
    case KHE_DRS_DOM_LIST_NONE:

      return (KHE_DRS_SOLN_SET_PART) KheDrsSolnSetPartDomNoneMake(drs);

    case KHE_DRS_DOM_LIST_STRONG:
    case KHE_DRS_DOM_LIST_TRADEOFF:

      return (KHE_DRS_SOLN_SET_PART)
	KheDrsSolnSetPartDomStrongMake(dom_kind, drs);

    case KHE_DRS_DOM_LIST_UNIFORM:

      return (KHE_DRS_SOLN_SET_PART)
	KheDrsSolnSetPartDomUniformMake(dom_kind, drs);

    case KHE_DRS_DOM_HASH_WEAK:

      return (KHE_DRS_SOLN_SET_PART) KheDrsSolnSetPartDomWeakMake(drs);

    case KHE_DRS_DOM_HASH_MEDIUM:

      return (KHE_DRS_SOLN_SET_PART) KheDrsSolnSetPartDomMediumMake(drs);

    case KHE_DRS_DOM_TRIE_STRONG:
    case KHE_DRS_DOM_TRIE_TRADEOFF:

      return (KHE_DRS_SOLN_SET_PART)
	KheDrsSolnSetPartDomTrieMake(dom_kind, drs);

    case KHE_DRS_DOM_INDEXED_TRADEOFF:

      return (KHE_DRS_SOLN_SET_PART)
	KheDrsSolnSetPartDomIndexedMake(dom_kind, drs);

    case KHE_DRS_DOM_INDEXED_UNIFORM:

      return (KHE_DRS_SOLN_SET_PART)
	KheDrsSolnSetPartDomIndexedUniformMake(dom_kind, drs);

    default:

      HnAbort("KheDrsSolnSetPartMake internal error: dom_kind %d\n", dom_kind);
      return NULL;  /* keep compiler happy */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSetPartFree(KHE_DRS_SOLN_SET_PART ssp,                    */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Free ssp, but not the solutions it contains.                             */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSetPartFree(KHE_DRS_SOLN_SET_PART ssp,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  switch( ssp->dom_kind )
  {
    case KHE_DRS_DOM_LIST_NONE:

      KheDrsSolnSetPartDomNoneFree((KHE_DRS_SOLN_SET_PART_DOM_NONE) ssp, drs);
      break;

    case KHE_DRS_DOM_LIST_STRONG:
    case KHE_DRS_DOM_LIST_TRADEOFF:

      KheDrsSolnSetPartDomStrongFree((KHE_DRS_SOLN_SET_PART_DOM_STRONG) ssp,
	drs);
      break;

    case KHE_DRS_DOM_LIST_UNIFORM:

      KheDrsSolnSetPartDomUniformFree((KHE_DRS_SOLN_SET_PART_DOM_UNIFORM) ssp,
	drs);
      break;

    case KHE_DRS_DOM_HASH_WEAK:

      KheDrsSolnSetPartDomWeakFree((KHE_DRS_SOLN_SET_PART_DOM_WEAK) ssp, drs);
      break;

    case KHE_DRS_DOM_HASH_MEDIUM:

      KheDrsSolnSetPartDomMediumFree((KHE_DRS_SOLN_SET_PART_DOM_MEDIUM) ssp,
	drs);
      break;

    case KHE_DRS_DOM_TRIE_STRONG:
    case KHE_DRS_DOM_TRIE_TRADEOFF:

      KheDrsSolnSetPartDomTrieFree((KHE_DRS_SOLN_SET_PART_DOM_TRIE) ssp, drs);
      break;

    case KHE_DRS_DOM_INDEXED_TRADEOFF:

      KheDrsSolnSetPartDomIndexedFree((KHE_DRS_SOLN_SET_PART_DOM_INDEXED) ssp,
	drs);
      break;

    case KHE_DRS_DOM_INDEXED_UNIFORM:

      KheDrsSolnSetPartDomIndexedUniformFree(
	(KHE_DRS_SOLN_SET_PART_DOM_INDEXED_UNIFORM) ssp, drs);
      break;

    default:

      HnAbort("KheDrsSolnSetPartFree internal error: unknown dom_kind (%d)\n",
	ssp->dom_kind);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsSolnSetPartCount(KHE_DRS_SOLN_SET_PART ssp)                    */
/*                                                                           */
/*  Return the number of solutions in ssp.                                   */
/*                                                                           */
/*****************************************************************************/

static int KheDrsSolnSetPartCount(KHE_DRS_SOLN_SET_PART ssp)
{
  switch( ssp->dom_kind )
  {
    case KHE_DRS_DOM_LIST_NONE:

      return KheDrsSolnSetPartDomNoneCount(
	(KHE_DRS_SOLN_SET_PART_DOM_NONE) ssp);

    case KHE_DRS_DOM_LIST_STRONG:
    case KHE_DRS_DOM_LIST_TRADEOFF:

      return KheDrsSolnSetPartDomStrongCount(
	(KHE_DRS_SOLN_SET_PART_DOM_STRONG) ssp);

    case KHE_DRS_DOM_LIST_UNIFORM:

      return KheDrsSolnSetPartDomUniformCount(
	(KHE_DRS_SOLN_SET_PART_DOM_UNIFORM) ssp);

    case KHE_DRS_DOM_HASH_WEAK:

      return KheDrsSolnSetPartDomWeakCount(
	(KHE_DRS_SOLN_SET_PART_DOM_WEAK) ssp);

    case KHE_DRS_DOM_HASH_MEDIUM:

      return KheDrsSolnSetPartDomMediumCount(
	(KHE_DRS_SOLN_SET_PART_DOM_MEDIUM) ssp);

    case KHE_DRS_DOM_TRIE_STRONG:
    case KHE_DRS_DOM_TRIE_TRADEOFF:

      return KheDrsSolnSetPartDomTrieCount(
	(KHE_DRS_SOLN_SET_PART_DOM_TRIE) ssp);

    case KHE_DRS_DOM_INDEXED_TRADEOFF:

      return KheDrsSolnSetPartDomIndexedCount(
	(KHE_DRS_SOLN_SET_PART_DOM_INDEXED) ssp);

    case KHE_DRS_DOM_INDEXED_UNIFORM:

      return KheDrsSolnSetPartDomIndexedUniformCount(
	(KHE_DRS_SOLN_SET_PART_DOM_INDEXED_UNIFORM) ssp);

    default:

      HnAbort("KheDrsSolnSetPartCount internal error: unknown "
	"dom_kind (%d)\n", ssp->dom_kind);
      return 0;  /* keep compiler happy */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSetPartGather(KHE_DRS_SOLN_SET_PART ssp,                  */
/*    KHE_DRS_SOLN_LIST soln_list)                                           */
/*    KHE_DRS_SOLN_SET soln_set)                                             */
/*                                                                           */
/*  Gather the solutions of ssp into soln_list, without disturbing ssp.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSetPartGather(KHE_DRS_SOLN_SET_PART ssp,
  KHE_DRS_SOLN_LIST soln_list)
{
  switch( ssp->dom_kind )
  {
    case KHE_DRS_DOM_LIST_NONE:

      KheDrsSolnSetPartDomNoneGather(
	(KHE_DRS_SOLN_SET_PART_DOM_NONE) ssp, soln_list);
      break;

    case KHE_DRS_DOM_LIST_STRONG:
    case KHE_DRS_DOM_LIST_TRADEOFF:

      KheDrsSolnSetPartDomStrongGather(
	(KHE_DRS_SOLN_SET_PART_DOM_STRONG) ssp, soln_list);
      break;

    case KHE_DRS_DOM_LIST_UNIFORM:

      KheDrsSolnSetPartDomUniformGather(
	(KHE_DRS_SOLN_SET_PART_DOM_UNIFORM) ssp, soln_list);
      break;

    case KHE_DRS_DOM_HASH_WEAK:

      KheDrsSolnSetPartDomWeakGather(
	(KHE_DRS_SOLN_SET_PART_DOM_WEAK) ssp, soln_list);
      break;

    case KHE_DRS_DOM_HASH_MEDIUM:

      KheDrsSolnSetPartDomMediumGather(
	(KHE_DRS_SOLN_SET_PART_DOM_MEDIUM) ssp, soln_list);
      break;

    case KHE_DRS_DOM_TRIE_STRONG:
    case KHE_DRS_DOM_TRIE_TRADEOFF:

      KheDrsSolnSetPartDomTrieGather(
	(KHE_DRS_SOLN_SET_PART_DOM_TRIE) ssp, soln_list);
      break;

    case KHE_DRS_DOM_INDEXED_TRADEOFF:

      KheDrsSolnSetPartDomIndexedGather(
	(KHE_DRS_SOLN_SET_PART_DOM_INDEXED) ssp, soln_list);
      break;

    case KHE_DRS_DOM_INDEXED_UNIFORM:

      KheDrsSolnSetPartDomIndexedUniformGather(
	(KHE_DRS_SOLN_SET_PART_DOM_INDEXED_UNIFORM) ssp, soln_list);
      break;

    default:

      HnAbort("KheDrsSolnSetPartGather internal error: unknown "
	"dom_kind (%d)\n", ssp->dom_kind);
      break;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSolnSetPartDominates(KHE_DRS_SOLN_SET_PART ssp,               */
/*    KHE_DRS_SOLN soln, KHE_DRS_DAY soln_day,                               */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Return true if ssp dominates soln.                                       */
/*                                                                           */
/*  Add 1 to *dom_test_count for each dominance test done along the way.     */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsSolnSetPartDominates(KHE_DRS_SOLN_SET_PART ssp,
  KHE_DRS_SOLN soln, KHE_DRS_DAY soln_day, int *dom_test_count,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  switch( ssp->dom_kind )
  {
    case KHE_DRS_DOM_LIST_NONE:

      return KheDrsSolnSetPartDomNoneDominates(
	(KHE_DRS_SOLN_SET_PART_DOM_NONE) ssp, soln, soln_day,
	dom_test_count, drs);

    case KHE_DRS_DOM_LIST_STRONG:
    case KHE_DRS_DOM_LIST_TRADEOFF:

      return KheDrsSolnSetPartDomStrongDominates(
	(KHE_DRS_SOLN_SET_PART_DOM_STRONG) ssp, soln, soln_day,
	dom_test_count, drs);

    case KHE_DRS_DOM_LIST_UNIFORM:

      return KheDrsSolnSetPartDomUniformDominates(
	(KHE_DRS_SOLN_SET_PART_DOM_UNIFORM) ssp, soln, soln_day,
	dom_test_count, drs);

    case KHE_DRS_DOM_HASH_WEAK:

      return KheDrsSolnSetPartDomWeakDominates(
	(KHE_DRS_SOLN_SET_PART_DOM_WEAK) ssp, soln, soln_day,
	dom_test_count, drs);

    case KHE_DRS_DOM_HASH_MEDIUM:

      return KheDrsSolnSetPartDomMediumDominates(
	(KHE_DRS_SOLN_SET_PART_DOM_MEDIUM) ssp, soln, soln_day,
	dom_test_count, drs);

    case KHE_DRS_DOM_TRIE_STRONG:
    case KHE_DRS_DOM_TRIE_TRADEOFF:

      return KheDrsSolnSetPartDomTrieDominates(
	(KHE_DRS_SOLN_SET_PART_DOM_TRIE) ssp, soln, soln_day,
	dom_test_count, drs);

    case KHE_DRS_DOM_INDEXED_TRADEOFF:

      return KheDrsSolnSetPartDomIndexedDominates(
	(KHE_DRS_SOLN_SET_PART_DOM_INDEXED) ssp, soln, soln_day,
	dom_test_count, drs);

    case KHE_DRS_DOM_INDEXED_UNIFORM:

      return KheDrsSolnSetPartDomIndexedUniformDominates(
	(KHE_DRS_SOLN_SET_PART_DOM_INDEXED_UNIFORM) ssp, soln, soln_day,
	dom_test_count, drs);

    default:

      HnAbort("KheDrsSolnSetPartDominates internal error: unknown dom_kind "
	"(%d)\n", ssp->dom_kind);
      return false;  /* keep compiler happy */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSetPartRemoveDominated(KHE_DRS_SOLN_SET_PART ssp,         */
/*    KHE_DRS_SOLN soln, KHE_DRS_DAY soln_day,                               */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Delete and free the unexpanded solutions of ssp that are                 */
/*  dominated by soln, including deleting them from drs->priqueue.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSolnSetPartRemoveDominated(KHE_DRS_SOLN_SET_PART ssp,
  KHE_DRS_SOLN soln, KHE_DRS_DAY soln_day, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  switch( ssp->dom_kind )
  {
    case KHE_DRS_DOM_LIST_NONE:

      KheDrsSolnSetPartDomNoneRemoveDominated(
	(KHE_DRS_SOLN_SET_PART_DOM_NONE) ssp, soln, soln_day, drs);
      break;

    case KHE_DRS_DOM_LIST_STRONG:
    case KHE_DRS_DOM_LIST_TRADEOFF:

      KheDrsSolnSetPartDomStrongRemoveDominated(
	(KHE_DRS_SOLN_SET_PART_DOM_STRONG) ssp, soln, soln_day, drs);
      break;

    case KHE_DRS_DOM_LIST_UNIFORM:

      KheDrsSolnSetPartDomUniformRemoveDominated(
	(KHE_DRS_SOLN_SET_PART_DOM_UNIFORM) ssp, soln, soln_day, drs);
      break;

    case KHE_DRS_DOM_HASH_WEAK:

      KheDrsSolnSetPartDomWeakRemoveDominated(
	(KHE_DRS_SOLN_SET_PART_DOM_WEAK) ssp, soln, soln_day, drs);
      break;

    case KHE_DRS_DOM_HASH_MEDIUM:

      KheDrsSolnSetPartDomMediumRemoveDominated(
	(KHE_DRS_SOLN_SET_PART_DOM_MEDIUM) ssp, soln, soln_day, drs);
      break;

    case KHE_DRS_DOM_TRIE_STRONG:
    case KHE_DRS_DOM_TRIE_TRADEOFF:

      KheDrsSolnSetPartDomTrieRemoveDominated(
	(KHE_DRS_SOLN_SET_PART_DOM_TRIE) ssp, soln, soln_day, drs);
      break;

    case KHE_DRS_DOM_INDEXED_TRADEOFF:

      KheDrsSolnSetPartDomIndexedRemoveDominated(
	(KHE_DRS_SOLN_SET_PART_DOM_INDEXED) ssp, soln, soln_day, drs);
      break;

    case KHE_DRS_DOM_INDEXED_UNIFORM:

      KheDrsSolnSetPartDomIndexedUniformRemoveDominated(
	(KHE_DRS_SOLN_SET_PART_DOM_INDEXED_UNIFORM) ssp, soln, soln_day, drs);
      break;

    default:

      HnAbort("KheDrsSolnSetPartRemoveDominated internal error: unknown "
	"dom_kind (%d)\n", ssp->dom_kind);
      break;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSolnSetPartAddSoln(KHE_DRS_SOLN_SET_PART ssp,                 */
/*    KHE_DRS_SOLN soln, KHE_DRS_DAY soln_day, KHE_DRS_EXPANDER de,          */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */