/*****************************************************************************/
/*                                                                           */
/*  THE KHE HIGH SCHOOL TIMETABLING ENGINE                                   */
/*  COPYRIGHT (C) 2010 Jeffrey H. Kingston                                   */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 3, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         khe_sr_dynamic_resource.c                                  */
/*  DESCRIPTION:  Dynamic resource solving                                   */
/*                                                                           */
/*  This file defines many types with no obvious best ordering, so we've     */
/*  followed the traditional XHSTT ordering:  times, resources, events       */
/*  (just tasks here), constraints (called expressions here) and solutions;  */
/*  and after those, sets of solutions and solvers.  It is all fully         */
/*  documented in the Appendix to the User's Guide, in the same order.       */
/*                                                                           */
/*****************************************************************************/
#include "khe_solvers.h"
#include "khe_priqueue.h"
#include "howard_p.h"
#include <limits.h>
#include <float.h>
#include <math.h>

#define TESTING 1
#define COL_WIDTH  12		/* when debugging solutions */
#define COL_COUNT   7		/* when debugging solutions */

#define bool_show(x) ((x) ? "true" : "false")

#define UNDEF_INT -1
#define UNDEF_FLOAT -1.0

#define DEBUG_MONITOR_ID "Constraint:2/TR_28/2Fri3"
#define DEBUG_MONITOR(m) (false &&					\
  strcmp(KheMonitorId((KHE_MONITOR) m), DEBUG_MONITOR_ID) == 0)

#define	RERUN  1		/* rerun                        */
#define RERUN_MONITOR_ID ""

#define DEBUG1 1		/* KheDynamicResourceSolverDoSolve */
#define DEBUG2 0		/* final solutions		*/
#define DEBUG3 0		/* task classes                 */
#define DEBUG4 0		/* KheDynamicResourceSolverMake */
#define DEBUG5 0		/* hash table efficiency      	*/
#define DEBUG6 0		/* trie dominance tests 	*/
#define DEBUG7 0		/* ResourceMake                 */
#define DEBUG8 0		/* expression construction      */
#define DEBUG9 0		/* KheDrsSolveOpen (costs)      */
#define DEBUG10 0		/* KheDrsExprOpen               */
#define DEBUG11 0		/* KheDrsResourceOpen           */
#define DEBUG12 0		/* KheDrsSolnExpandMakeAndMeld  */
#define DEBUG14 0		/* int seq cost child has opened*/
#define DEBUG15 0		/* KheDrsTaskAssign             */
#define DEBUG16 0		/* KheTaskUnAssignResource      */
#define DEBUG17 1		/* Search                       */
#define DEBUG18 0		/* KheDrsDayOpen                */
#define DEBUG19 0		/* KheDrsTaskClassOpen          */
#define DEBUG20 0		/* KheDrsAUIntervalFindRight    */
#define DEBUG21 0		/* Search statistics            */
#define DEBUG22 0		/* SolnExpand and SolnDoExpand  */
#define DEBUG23 0		/* set signatures               */
#define DEBUG24 0		/* rerun costs                  */
#define DEBUG25 0		/* dominating and dominated     */
#define DEBUG26 0		/* trie stats                   */
#define DEBUG27 0		/* trie stats                   */
#define DEBUG28 0		/* ancestor and dominates freqs */
#define DEBUG29 0		/* soln set on each day         */
#define DEBUG30 1		/* dominance debug              */
#define DEBUG31 0		/* priqueue insert and delete   */
#define DEBUG32 0		/* DistanceToAncestor           */
#define DEBUG33 0		/* prune cost                   */
#define DEBUG34 0		/* prune cost                   */
#define DEBUG35 0		/* two extra selection          */
#define DEBUG36 0		/* dom tables                   */
#define DEBUG37 0		/* dom tables                   */
#define DEBUG39 0		/* dom tables (positive entry)  */
#define DEBUG40 0		/* one extra dominates          */
#define DEBUG41 0		/* task class min cost          */
#define DEBUG42 0		/* dom tables                   */
#define DEBUG43 0		/* one extra dominates          */

#define DEBUG44 0		/* sub get                      */
#define DEBUG44_MONITOR_ID "Constraint:22/HN_1"
#define DEBUG44_MONITOR(m) (DEBUG44 && (m) != NULL && 			\
  strcmp(KheMonitorId((KHE_MONITOR) (m)), DEBUG44_MONITOR_ID) == 0)

#define DEBUG45(day) (0 && ((day) == NULL || (day)->solve_expand_count == 1))
#define DEBUG46 0		/* various                      */
#define DEBUG47(day) (0 && ((day) == NULL || (day)->solve_expand_count == 1))

#define DEBUG48	0
#define DEBUG49	0
#define DEBUG50	0
#define DEBUG51(day) (0 && ((day) == NULL || (day)->solve_expand_count == 1))
#define DEBUG52(day, ds) (0 &&						\
  ((day) == NULL || (day)->solve_expand_count == 1) &&			\
  (ds)->open_shift_index == 27)

#define DEBUG54 0
#define DEBUG55 0
#define DEBUG56 0
#define DEBUG57 0

#define DEBUG58(mi)							\
  (false && strcmp(KheDrsMonitorInfoId(mi), "Constraint:9/HN_1") == 0 )

#define DEBUG59 0
#define DEBUG60 0
#define DEBUG61 0
#define DEBUG62 0
#define DEBUG63 0
#define DEBUG64 0
#define DEBUG64_ID "1Tue"
#define DEBUG65 0
#define DEBUG66 0
#define DEBUG67 0

#define USE_DOM_CACHING 0

#define DEBUG_LAST_EDGE (false && (drs)->solve_daily_expand_limit == 0)


/*****************************************************************************/
/*                                                                           */
/*  Submodule "type declarations"                                            */
/*                                                                           */
/*  This submodule contains forward type declarations, but only if needed.   */
/*                                                                           */
/*****************************************************************************/

/* type declarations (times) */
typedef struct khe_drs_day_rec *KHE_DRS_DAY;
typedef HA_ARRAY(KHE_DRS_DAY) ARRAY_KHE_DRS_DAY;

/* type declarations (resources) */
typedef struct khe_drs_resource_rec *KHE_DRS_RESOURCE;
typedef HA_ARRAY(KHE_DRS_RESOURCE) ARRAY_KHE_DRS_RESOURCE;

typedef struct khe_drs_resource_set_rec *KHE_DRS_RESOURCE_SET;
typedef HA_ARRAY(KHE_DRS_RESOURCE_SET) ARRAY_KHE_DRS_RESOURCE_SET;

typedef struct khe_drs_resource_on_day_rec *KHE_DRS_RESOURCE_ON_DAY;
typedef HA_ARRAY(KHE_DRS_RESOURCE_ON_DAY) ARRAY_KHE_DRS_RESOURCE_ON_DAY;

/* ***
typedef struct khe_drs_resource_on_day_set_rec *KHE_DRS_RESOURCE_ON_DAY_SET;
typedef HA_ARRAY(KHE_DRS_RESOURCE_ON_DAY_SET) ARRAY_KHE_DRS_RESOURCE_ON_DAY_SET;
*** */

/* type declarations (events) */
typedef struct khe_drs_task_rec *KHE_DRS_TASK;
typedef HA_ARRAY(KHE_DRS_TASK) ARRAY_KHE_DRS_TASK;

typedef struct khe_drs_task_on_day_rec *KHE_DRS_TASK_ON_DAY;
typedef HA_ARRAY(KHE_DRS_TASK_ON_DAY) ARRAY_KHE_DRS_TASK_ON_DAY;

typedef struct khe_drs_task_class_rec *KHE_DRS_TASK_CLASS;
typedef HA_ARRAY(KHE_DRS_TASK_CLASS) ARRAY_KHE_DRS_TASK_CLASS;

typedef struct khe_drs_shift_rec *KHE_DRS_SHIFT;
typedef HA_ARRAY(KHE_DRS_SHIFT) ARRAY_KHE_DRS_SHIFT;

typedef struct khe_drs_asst_to_task_set_rec *KHE_DRS_ASST_TO_TASK_SET;

typedef struct khe_drs_shift_asst_trie_rec *KHE_DRS_SHIFT_ASST_TRIE;
typedef HA_ARRAY(KHE_DRS_SHIFT_ASST_TRIE) ARRAY_KHE_DRS_SHIFT_ASST_TRIE;

/* type declarations (dominance) */
/* ***
typedef struct khe_drs_dom_table_sum_rec *KHE_DRS_DOM_TABLE_SUM;
typedef struct khe_drs_dom_table_seq_rec *KHE_DRS_DOM_TABLE_SEQ;
*** */

typedef struct khe_drs_dom_test_rec *KHE_DRS_DOM_TEST;
typedef HA_ARRAY(KHE_DRS_DOM_TEST) ARRAY_KHE_DRS_DOM_TEST;

/* ***
typedef struct khe_drs_dom_test_set_rec *KHE_DRS_DOM_TEST_SET;

#define INHERIT_KHE_DRS_DOM_TEST_SET				\
  ARRAY_KHE_DRS_DOM_TEST	dom_tests;
*** */

typedef struct khe_drs_signature_rec *KHE_DRS_SIGNATURE;
typedef HA_ARRAY(KHE_DRS_SIGNATURE) ARRAY_KHE_DRS_SIGNATURE;

typedef struct khe_drs_signature_set_rec *KHE_DRS_SIGNATURE_SET;
typedef HA_ARRAY(KHE_DRS_SIGNATURE_SET) ARRAY_KHE_DRS_SIGNATURE_SET;

typedef struct khe_drs_correlator_rec *KHE_DRS_CORRELATOR;

typedef struct khe_drs_signer_rec *KHE_DRS_SIGNER;
typedef HA_ARRAY(KHE_DRS_SIGNER) ARRAY_KHE_DRS_SIGNER;

typedef struct khe_drs_signer_set_rec *KHE_DRS_SIGNER_SET;

/* type declarations (constraints i.e. expressions) */
typedef struct khe_drs_expr_rec *KHE_DRS_EXPR;
typedef HA_ARRAY(KHE_DRS_EXPR) ARRAY_KHE_DRS_EXPR;

/* typedef struct khe_drs_expr_cost_rec *KHE_DRS_EXPR_COST; */

/* type declarations (assignments) */
typedef struct khe_drs_asst_to_task_rec KHE_DRS_ASST_TO_TASK;
typedef HA_ARRAY(KHE_DRS_ASST_TO_TASK) ARRAY_KHE_DRS_ASST_TO_TASK;

typedef struct khe_drs_asst_to_task_class_rec *KHE_DRS_ASST_TO_TASK_CLASS;
typedef HA_ARRAY(KHE_DRS_ASST_TO_TASK_CLASS) ARRAY_KHE_DRS_ASST_TO_TASK_CLASS;

typedef struct khe_drs_asst_to_shift_rec *KHE_DRS_ASST_TO_SHIFT;
typedef HA_ARRAY(KHE_DRS_ASST_TO_SHIFT) ARRAY_KHE_DRS_ASST_TO_SHIFT;

/* type declarations (solutions) */
typedef struct khe_drs_soln_rec *KHE_DRS_SOLN;
typedef HA_ARRAY(KHE_DRS_SOLN) ARRAY_KHE_DRS_SOLN;
typedef HP_TABLE(KHE_DRS_SOLN) TABLE_KHE_DRS_SOLN;

/* type declarations (sets of solutions) */
typedef struct khe_drs_soln_list_rec *KHE_DRS_SOLN_LIST;
typedef HA_ARRAY(KHE_DRS_SOLN_LIST) ARRAY_KHE_DRS_SOLN_LIST;
typedef HP_TABLE(KHE_DRS_SOLN_LIST) TABLE_KHE_DRS_SOLN_LIST;

typedef struct khe_drs_soln_set_rec *KHE_DRS_SOLN_SET;
typedef HA_ARRAY(KHE_DRS_SOLN_SET) ARRAY_KHE_DRS_SOLN_SET;


/*****************************************************************************/
/*                                                                           */
/*  Submodule "type definitions - times"                                     */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_DAY_RANGE - a range of days                                 */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_day_range_rec {
  int			first;
  int			last;
} KHE_DRS_DAY_RANGE;

typedef HA_ARRAY(KHE_DRS_DAY_RANGE) ARRAY_KHE_DRS_DAY_RANGE;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_DAY - one day                                               */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_day_rec {
  /* INHERIT_KHE_DRS_DOM_TEST_SET */
  int				frame_index;		/* index in frame    */
  int				open_day_index;		/* index when open   */
  KHE_TIME_GROUP		time_group;		/* day's time group  */
  /* ARRAY_KHE_DRS_TASK_CLASS	task_classes; */	/* today's tasks     */
  ARRAY_KHE_DRS_SHIFT		shifts;			/* shifts            */
  KHE_DRS_SIGNER_SET		signer_set;		/* dom tests etc.    */
  /* ARRAY_KHE_DRS_EXPR		nr_internal_today; */	/* open today        */
  /* HA_ARRAY_INT		eq_dom_test_indexes; */	/* where = tests are */
  KHE_DRS_SOLN_SET		soln_set;		/* today's solns     */
  KHE_DRS_SOLN_LIST		soln_list;		/* sorted soln_set   */
  int				soln_made_count;	/* solns made today  */
  int				solve_expand_count;	/* expanded today    */
  /* KHE_COST			solve_prune_cost; */	/* prune from here   */
};


/*****************************************************************************/
/*                                                                           */
/*  Submodule "type definitions - resources"                                 */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_RESOURCE_EXPAND_ROLE - resource's role in current expansion */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  KHE_DRS_RESOURCE_EXPAND_NO,
  KHE_DRS_RESOURCE_EXPAND_FIXED,
  KHE_DRS_RESOURCE_EXPAND_FREE
} KHE_DRS_RESOURCE_EXPAND_ROLE;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_RESOURCE - one resource                                     */
/*                                                                           */
/*****************************************************************************/
typedef struct khe_drs_dim2_table_rec *KHE_DRS_DIM2_TABLE;

struct khe_drs_resource_rec {
  KHE_RESOURCE			resource;		/* the resource      */
  int				open_resource_index;	/* index when open   */
  ARRAY_KHE_DRS_RESOURCE_ON_DAY	days;			/* doing each day    */
  KHE_DRS_RESOURCE_EXPAND_ROLE	expand_role;		/* current expand    */
  ARRAY_KHE_DRS_ASST_TO_SHIFT	expand_assts_to_shifts;	/* current expand    */
  ARRAY_KHE_DRS_ASST_TO_TASK_CLASS expand_assts;	/* current expand    */
  KHE_DRS_ASST_TO_TASK_CLASS	expand_free_day_asst;	/* current expand    */
  KHE_DRS_DIM2_TABLE		expand_dom_test_cache;	/* current expand    */
  /* ***
   KHE_DRS_ASST_TO_SHIFT expand_free_asst_to_shift;     ** curr. expand      **
  ARRAY_KHE_DRS_SIGNATURE	expand_shift_sigs;	** current expand    **
  KHE_DRS_SIGNATURE		expand_free_shift_sig;	** current expand    **
  *** */
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_RESOURCE_SET - a set of resources                           */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_resource_set_rec {
  ARRAY_KHE_DRS_RESOURCE	resources;		/* the resources     */
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_RESOURCE_ON_DAY - what one resource is doing on one day     */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_resource_on_day_rec {
  /* INHERIT_KHE_DRS_DOM_TEST_SET */
  KHE_DRS_RESOURCE		encl_dr;		/* the resource      */
  KHE_DRS_DAY			day;			/* the day           */
  KHE_DRS_TASK_ON_DAY		closed_asst;		/* asst when closed  */
  ARRAY_KHE_DRS_EXPR		external_today;		/* external today    */
  /* ARRAY_KHE_DRS_EXPR		internal_today; */	/* internal today    */
  KHE_DRS_SIGNER		signer;			/* signer         */
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_RESOURCE_ON_DAY_SET - a set of resource on day objects      */
/*                                                                           */
/*****************************************************************************/

/* *** not currently used
struct khe_drs_resource_on_day_set_rec {
  ARRAY_KHE_DRS_RESOURCE_ON_DAY	resource_on_days;	** the resources     **
};
*** */


/*****************************************************************************/
/*                                                                           */
/*  Submodule "type definitions - events"                                    */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_TASK_EXPAND_ROLE - task's role in current expansion              */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  KHE_DRS_TASK_EXPAND_NO,
  KHE_DRS_TASK_EXPAND_FIXED,
  KHE_DRS_TASK_EXPAND_MUST,
  KHE_DRS_TASK_EXPAND_FREE
} KHE_DRS_TASK_EXPAND_ROLE;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_TASK - one proper root task                                 */
/*                                                                           */
/*  The days are sorted chronologically, and this is used to test whether    */
/*  a given task on day is the first of its task.                            */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_task_rec {
  KHE_DRS_TASK_CLASS		encl_dtc;
  int				index_in_encl_dtc;
  KHE_DRS_TASK_EXPAND_ROLE	expand_role;		/* used by SolnExpand*/
  bool				open;			/* true when open    */
  KHE_TASK			task;			/* the KHE task      */
  KHE_DRS_RESOURCE		closed_asst;		/* asst when closed  */
  KHE_COST			asst_cost;		/* assignment cost   */
  KHE_COST			non_asst_cost;		/* non-asst cost     */
  ARRAY_KHE_DRS_TASK_ON_DAY	days;
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_TASK_ON_DAY - what one proper root task is doing on one day */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_task_on_day_rec {
  KHE_DRS_TASK			encl_dt;		/* enclosing task    */
  KHE_DRS_DAY			day;			/* the day           */
  KHE_TASK			task;			/* atomic task today */
  KHE_TIME			time;			/* time on day       */
  KHE_DRS_RESOURCE_ON_DAY	closed_asst;		/* asst when closed  */
  ARRAY_KHE_DRS_EXPR		external_today;		/* leaf today        */
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_TASK_CLASS - a set of tasks, taken from one KHE_TASK_CLASS  */
/*                                                                           */
/*  For efficiency, unassigned_tasks is managed lazily.  Every unassigned    */
/*  task is in it, but not in order, and there may be assigned tasks in it.  */
/*  These defects are corrected when the task class is opened for solving.   */
/*                                                                           */
/*  Field expand_prev_unfixed is the index in unassigned_tasks of the        */
/*  unfixed task most recently given away to expansion, or -1 if no tasks    */
/*  are currently given away, or a large number if there are none to give    */
/*  away.                                                                    */
/*                                                                           */
/*  Obsolete:                                                                */
/*  Field expand_next_unfixed is the index in unassigned_tasks of the next   */
/*  available unfixed unassigned task, unless it points beyond the end of    */
/*  unassigned_tasks, in which case there is no such task.                   */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_task_class_rec {
  KHE_TASK_CLASS		orig_task_class;	/* origin of this    */
  KHE_DRS_SHIFT			encl_shift;		/* enclosing shift   */
  KHE_DRS_DAY_RANGE		day_range;		/* days running      */
  ARRAY_KHE_DRS_TASK		all_tasks;		/* all tasks in class*/
  ARRAY_KHE_DRS_TASK		unassigned_tasks;	/* unassigned tasks  */
  /* int			expand_used; */		/* used by SolnExpand*/
  int				expand_prev_unfixed;	/* used by SolnExpand*/
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_SHIFT - the set of all task classes with the same times     */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_shift_rec {
  KHE_DRS_DAY			encl_day;		/* enclosing day     */
  int				open_shift_index;	/* set when open     */
  /* int			index_in_cycle; */	/* overall index     */
  /* int			index_in_day; */	/* index in day      */
  int				expand_must_assign_count;
  int				expand_max_included_free_resource_count;
  ARRAY_KHE_DRS_TASK_CLASS	task_classes;		/* the classes       */
  KHE_DRS_SIGNER		signer;			/* signer         */
  KHE_DRS_SHIFT_ASST_TRIE	asst_trie;
  /* INHERIT_KHE_DRS_DOM_TEST_SET */
  /* ARRAY_KHE_DRS_EXPR		nr_internal_today; */	/* open today        */
  /* ARRAY_KHE_DRS_SIGNATURE	resource_signatures; */	/* for each resource */
  /* ARRAY_KHE_DRS_ASST_TO_TASK_CLASS assts; */		/* for each resource */
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_SHIFT_ASST - an assignment of resources to one shift        */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_shift_asst_rec {
  KHE_DRS_SIGNATURE		sig;
  /* struct khe_drs_signature_rec	sig; */
  KHE_DRS_ASST_TO_TASK_SET	assts_to_tasks;
  /* ARRAY_KHE_DRS_ASST_TO_TASK	assts_to_tasks; */
  /* KHE_DRS_SIGNATURE		sig; */
  /* ARRAY_KHE_DRS_TASK_ON_DAY	task_on_days; */
  /* ARRAY_KHE_DRS_ASST_TO_TASK_CLASS assts; */
  /* INHERIT_KHE_DRS_SIGNATURE */
} *KHE_DRS_SHIFT_ASST;

typedef HA_ARRAY(KHE_DRS_SHIFT_ASST) ARRAY_KHE_DRS_SHIFT_ASST;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_SHIFT_ASST_TRIE - shift assignment lists indexed by r-set   */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_shift_asst_trie_rec {
  ARRAY_KHE_DRS_SHIFT_ASST		assts;		/* assts at this pt  */
  ARRAY_KHE_DRS_SHIFT_ASST_TRIE		children;	/* children */
};


/*****************************************************************************/
/*                                                                           */
/*  Submodule "type definitions - dominance"                                 */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_DOM_KIND - the kind of dominance testing to use             */
/*                                                                           */
/*****************************************************************************/

/* *** defined in khe_solvers.h
typedef enum {
  KHE_DRS_DOM_LIST_NONE,
  KHE_DRS_DOM_LIST_STRONG,
  KHE_DRS_DOM_LIST_TRADEOFF,
  KHE_DRS_DOM_LIST_UNIFORM,
  KHE_DRS_DOM_HASH_WEAK,
  KHE_DRS_DOM_HASH_MEDIUM,
  KHE_DRS_DOM_TRIE_STRONG,
  KHE_DRS_DOM_TRIE_TRADEOFF,
  KHE_DRS_DOM_INDEXED_TRADEOFF,
  KHE_DRS_DOM_INDEXED_UNIFORM
} KHE_DRS_DOM_KIND;
*** */


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_DOM_TYPE - the type of a dominance test                     */
/*                                                                           */
/*****************************************************************************/

/* *** no longer used
typedef enum {
  KHE_DRS_DOM_UNUSED,		** "-"      should never be used      **
  KHE_DRS_DOM_GE,		** ">="     min limit only            **
  KHE_DRS_DOM_LE,		** "<="     max limit only            **
  KHE_DRS_DOM_EQ,		** "==",    for uncertain cases       **
  KHE_DRS_DOM_GE_LOOSE,		** ">=*"    min limit with loosening  **
  KHE_DRS_DOM_EQ_LOOSE		** "==*"    max + min with loosening  **
} KHE_DRS_DOM_TYPE;
*** */


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_COST_TUPLE - a pair of costs stored in a dominance table     */
/*                                                                           */
/*****************************************************************************/

/* typedef HA_ARRAY(KHE_COST) ARRAY_KHE_COST; */

typedef struct khe_drs_cost_tuple_rec {
  KHE_COST			psi;
  KHE_COST			psi0;
  bool				psi_plus_defined;
  KHE_COST			psi_plus;
} KHE_DRS_COST_TUPLE;

typedef HA_ARRAY(KHE_DRS_COST_TUPLE) ARRAY_KHE_DRS_COST_TUPLE;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_DOM_TABLE_SUB_SUB - a dominance sub-sub-table               */
/*                                                                           */
/*****************************************************************************/

/* ***
typedef struct khe_drs_dom_table_sub_sub_rec {
  int				offset;
  ARRAY_KHE_DRS_COST_TUPLE	indexed_by_l2;
  ** ARRAY_KHE_COST		indexed_by_l2; **
} *KHE_DRS_DOM_TABLE_SUB_SUB;

typedef HA_ARRAY(KHE_DRS_DOM_TABLE_SUB_SUB) ARRAY_KHE_DRS_DOM_TABLE_SUB_SUB;
*** */


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_DOM_TABLE_SUB - a dominance sub-table                       */
/*                                                                           */
/*****************************************************************************/

/* ***
typedef struct khe_drs_dom_table_sub_rec {
  char					*debug_str;
  int					offset;
  ARRAY_KHE_DRS_DOM_TABLE_SUB_SUB	indexed_by_l1;
  KHE_COST				max_entry;
} *KHE_DRS_DOM_TABLE_SUB;

typedef HA_ARRAY(KHE_DRS_DOM_TABLE_SUB) ARRAY_KHE_DRS_DOM_TABLE_SUB;
*** */


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_DOM_TABLE_SUM - a dominance table, holding available costs  */
/*                                                                           */
/*****************************************************************************/

/* ***
struct khe_drs_dom_table_sum_rec {
  ARRAY_KHE_DRS_DOM_TABLE_SUB	indexed_by_e;
};
*** */


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_DOM_TABLE_SEQ - a dominance table, holding available costs  */
/*                                                                           */
/*****************************************************************************/

/* ***
typedef struct khe_drs_dom_table_sub_r {
  ARRAY_KHE_DRS_DOM_TABLE_SUB	indexed_by_r;
} *KHE_DRS_DOM_TABLE_SUB_R;

typedef HA_ARRAY(KHE_DRS_DOM_TABLE_SUB_R) ARRAY_KHE_DRS_DOM_TABLE_SUB_R;

typedef struct khe_drs_dom_table_sub_qr {
  ARRAY_KHE_DRS_DOM_TABLE_SUB_R indexed_by_q;
} *KHE_DRS_DOM_TABLE_SUB_QR;

typedef HA_ARRAY(KHE_DRS_DOM_TABLE_SUB_QR) ARRAY_KHE_DRS_DOM_TABLE_SUB_QR;

struct khe_drs_dom_table_seq_rec {
  ARRAY_KHE_DRS_DOM_TABLE_SUB_QR indexed_by_p;
};
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DIM1_TABLE - a one-dimensional table of tuples                   */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_dim1_table_rec {
  int				offset;
  ARRAY_KHE_DRS_COST_TUPLE	children;
} *KHE_DRS_DIM1_TABLE;

typedef HA_ARRAY(KHE_DRS_DIM1_TABLE) ARRAY_KHE_DRS_DIM1_TABLE;


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DIM2_TABLE - a two-dimensional table of tuples                   */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_dim2_table_rec {
  int				offset;
  ARRAY_KHE_DRS_DIM1_TABLE	children;
  char				*debug_str;
};

typedef HA_ARRAY(KHE_DRS_DIM2_TABLE) ARRAY_KHE_DRS_DIM2_TABLE;


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DIM3_TABLE - a three-dimensional table of tuples                 */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_dim3_table_rec {
  int				offset;
  ARRAY_KHE_DRS_DIM2_TABLE	children;
  char				*debug_str;
} *KHE_DRS_DIM3_TABLE;

typedef HA_ARRAY(KHE_DRS_DIM3_TABLE) ARRAY_KHE_DRS_DIM3_TABLE;


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DIM4_TABLE - a four-dimensional table of tuples                  */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_dim4_table_rec {
  int				offset;
  ARRAY_KHE_DRS_DIM3_TABLE	children;
  char				*debug_str;
} *KHE_DRS_DIM4_TABLE;

typedef HA_ARRAY(KHE_DRS_DIM4_TABLE) ARRAY_KHE_DRS_DIM4_TABLE;


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DIM5_TABLE - a five-dimensional table of tuples                  */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_dim5_table_rec {
  int				offset;
  ARRAY_KHE_DRS_DIM4_TABLE	children;
  char				*debug_str;
} *KHE_DRS_DIM5_TABLE;

typedef HA_ARRAY(KHE_DRS_DIM5_TABLE) ARRAY_KHE_DRS_DIM5_TABLE;


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST_TYPE - type of dom test                                 */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  KHE_DRS_DOM_TEST_UNUSED,			/* unused                    */
  KHE_DRS_DOM_TEST_STRONG,			/* strong                    */
  KHE_DRS_DOM_TEST_TRADEOFF,			/* tradeoff                  */
  KHE_DRS_DOM_TEST_UNIFORM,			/* uniform                   */
  KHE_DRS_DOM_TEST_CORR1_PARENT,		/* parent corr1              */
  KHE_DRS_DOM_TEST_CORR1_CHILD,			/* child corr1               */
  KHE_DRS_DOM_TEST_CORR2_CHILD,			/* child corr2               */
  KHE_DRS_DOM_TEST_CORR3_FIRST,			/* first corr3               */
  KHE_DRS_DOM_TEST_CORR3_MID,			/* in the middle             */
  KHE_DRS_DOM_TEST_CORR3_LAST,			/* last corr3                */
  KHE_DRS_DOM_TEST_CORR4_FIRST,			/* first corr4               */
  KHE_DRS_DOM_TEST_CORR4_MID,			/* in the middle             */
  KHE_DRS_DOM_TEST_CORR4_LAST			/* last corr4                */
} KHE_DRS_DOM_TEST_TYPE;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_DOM_TEST - a dominance test                                 */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_dom_test_rec {
  KHE_DRS_DOM_TEST_TYPE	type;			/* type of test              */
  int			correlated_delta;	/* if correlated             */
  KHE_DRS_EXPR		expr;			/* originating expression    */
  bool			allow_zero;		/* allow zero ("Z" in doc)   */
  int			min_limit;              /* min limit (0 if none)     */
  int			max_limit;              /* max limit (INT_MAX if ne) */
  /* bool		tradeoff_allowed; */	/* true if tradeoff allowed  */
  /* bool		uniform_allowed; */	/* true if uniform allowed   */
  int			a;			/* "a" in doc                */
  int			b;			/* "b" in doc                */
  KHE_COST		tradeoff_cost;		/* tradeoff cost, if tradeoff*/
  /* KHE_DRS_EXPR_COST	expr_cost; */		/* cost expr, if any (debug) */
  KHE_DRS_DIM2_TABLE	main_dom_table2;	/* uniform dominance table   */
  KHE_DRS_DIM4_TABLE	corr_dom_table4;	/* uniform dominance table   */
  KHE_MONITOR		monitor;		/* monitor, if any           */
};

/* *** obsolete version
struct khe_drs_dom_test_rec {
  KHE_DRS_DOM_TYPE	type;			** type of test              **
  bool			tradeoff;		** true if accepts tradeoff  **
  int			min_limit;		** GE_LOOSE and EQ_LOOSE     **
  KHE_COST		tradeoff_cost;		** tradeoff cost, if tradeoff**
};
*** */


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_DOM_TEST_SET - a set (actually sequence) of dominance tests */
/*                                                                           */
/*****************************************************************************/

/* *** replaced by KHE_DRS_SIGNER
struct khe_drs_dom_test_set_rec {
  INHERIT_KHE_DRS_DOM_TEST_SET
};
*** */


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_SIGNATURE - a signature                                     */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_signature_rec {
  int			reference_count;	/* reference count           */
  int			asst_to_shift_index;	/* index of enclosing a_to_s */
  KHE_COST		cost;			/* cost                      */
  HA_ARRAY_INT		states;			/* expr states               */
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_SIGNATURE_SET - a set of signatures                         */
/*                                                                           */
/*  KHE_DRS_SIGNATURE_SET is a pointer type as usual, but in KHE_DRS_SOLN    */
/*  it is expanded ("struct khe_drs_signature_set_rec") to save memory.      */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_signature_set_rec {
  KHE_COST			cost;
  ARRAY_KHE_DRS_SIGNATURE	signatures;
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_CORRELATED_SET - a set of correlated dom tests              */
/*                                                                           */
/*****************************************************************************/

/* ***
typedef struct khe_drs_correlated_set {
  HA_ARRAY_INT			dt_indexes;
} *KHE_DRS_CORRELATED_SET;

typedef HA_ARRAY(KHE_DRS_CORRELATED_SET) ARRAY_KHE_DRS_CORRELATED_SET;
*** */


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_CORRELATOR - an expression correlator                       */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_correlator_rec {
  bool				make_correlated;
  HA_ARRAY_INT			children;
  HA_ARRAY_INT			positives;
  HA_ARRAY_INT			negatives;
  HA_ARRAY_INT			singles;
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_SIGNER - a signature controller                             */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  KHE_DRS_SIGNER_DAY,
  KHE_DRS_SIGNER_RESOURCE_ON_DAY,
  KHE_DRS_SIGNER_SHIFT
} KHE_DRS_SIGNER_TYPE;

struct khe_drs_signer_rec {
  ARRAY_KHE_DRS_EXPR		internal_exprs;		/* that affect this  */
  ARRAY_KHE_DRS_DOM_TEST	dom_tests;
  HA_ARRAY_INT			eq_dom_test_indexes;	/* where = tests are */
  /* HA_ARRAY_INT		potential_matches; */	/* matches might be  */
  int				last_hard_cost_index;
  bool				debug_eval_if_rerun;	/* debug evals       */
  KHE_DRS_CORRELATOR		correlator;		/* always present    */
  KHE_DRS_SIGNER_TYPE		type;
  union {
    KHE_DRS_DAY			day;
    KHE_DRS_RESOURCE_ON_DAY	resource_on_day;
    KHE_DRS_SHIFT		shift;
  } u;
};


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SIGNER_TEST - the signer tests to carry out (hard, soft, or all) */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  KHE_DRS_SIGNER_HARD,
  KHE_DRS_SIGNER_SOFT,
  KHE_DRS_SIGNER_ALL
} KHE_DRS_SIGNER_TEST;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_SIGNER_SET - a set of signers                               */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_signer_set_rec {
  KHE_DRS_DAY			day;
  ARRAY_KHE_DRS_SIGNER		signers;
};


/*****************************************************************************/
/*                                                                           */
/*  Submodule "type definitions - constraints (i.e. expressions)"            */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_MONITOR_INFO - information about one monitor                */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_monitor_info_rec {
  KHE_MONITOR		monitor;
  KHE_COST		rerun_open_and_search_cost;
  KHE_COST		rerun_open_and_close_cost;
} *KHE_DRS_MONITOR_INFO;

typedef HA_ARRAY(KHE_DRS_MONITOR_INFO) ARRAY_KHE_DRS_MONITOR_INFO;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_OP - an open, search, or close operation type               */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  KHE_DRS_OPEN,
  KHE_DRS_SEARCH,
  KHE_DRS_CLOSE
} KHE_DRS_OP;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_TAG                                                    */
/*                                                                           */
/*  Tags for expression types.                                               */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  KHE_DRS_EXPR_ASSIGNED_TASK_TAG,
  KHE_DRS_EXPR_BUSY_TIME_TAG,
  KHE_DRS_EXPR_FREE_TIME_TAG,
  KHE_DRS_EXPR_WORK_TIME_TAG,
  KHE_DRS_EXPR_BUSY_DAY_TAG,
  KHE_DRS_EXPR_FREE_DAY_TAG,
  KHE_DRS_EXPR_WORK_DAY_TAG,
  KHE_DRS_EXPR_OR_TAG,
  KHE_DRS_EXPR_AND_TAG,	
  KHE_DRS_EXPR_INT_SUM_TAG,
  KHE_DRS_EXPR_FLOAT_SUM_TAG,
  KHE_DRS_EXPR_INT_DEV_TAG,
  KHE_DRS_EXPR_FLOAT_DEV_TAG,
  KHE_DRS_EXPR_INT_SUM_COST_TAG,
  KHE_DRS_EXPR_INT_SEQ_COST_TAG
} KHE_DRS_EXPR_TAG;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_ADJUST_TYPE - types of signature value adjustments          */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  KHE_DRS_ADJUST_ORDINARY,
  KHE_DRS_ADJUST_NO_MAX,
  KHE_DRS_ADJUST_LINEAR,
  KHE_DRS_ADJUST_STEP
} KHE_DRS_ADJUST_TYPE;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_VALUE - the value of an expression; could be int or float   */
/*                                                                           */
/*****************************************************************************/

typedef union {
  int			int_val;		/* value, if int   */
  float			float_val;		/* value, if float */
} KHE_DRS_VALUE;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_PARENT - a expression's parent                              */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_parent_rec {
  KHE_DRS_EXPR		expr;			/* the parent expression */
  int			index;			/* index in parent       */
} KHE_DRS_PARENT;

typedef HA_ARRAY(KHE_DRS_PARENT) ARRAY_KHE_DRS_PARENT;


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_OPEN_CHILDREN - a sequence of open children                      */
/*                                                                           */
/*  Field oc->children is a sequence of children sorted by an "open          */
/*  index" which could be either the last open day index of the child's      */
/*  open day range, or the open shift index of the shift that the child      */
/*  is affected by.                                                          */
/*                                                                           */
/*  Field oc->range is the range of open indexes of the children.  If there  */
/*  is at least one child this range must be non-empty; if there are no      */
/*  children it must be empty.                                               */
/*                                                                           */
/*  Field oc->child_indexes is an array of indexes into the children array,  */
/*  such that for each open index oi of a child in the array,                */
/*                                                                           */
/*    oc->child_indexes[oi - oc->range.first]                                */
/*                                                                           */
/*  is the index in oc->children of the first open child x such that the     */
/*  open index of x is oi or more.  Equivalently, it is the number of        */
/*  children x whose open index is strictly less than oi, as returned by     */
/*                                                                           */
/*    int KheDrsOpenChildrenBefore(KHE_DRS_OPEN_CHILDREN oc, int index)      */
/*                                                                           */
/*  There is also a sentinel at the end of oc->child_indexes holding the     */
/*  total number of children, so that for any open index oi, the iterator    */
/*                                                                           */
/*    #define KheDrsOpenChildrenForEach(oc, oi, x, i)                   \    */
/*      i1 = KheDrsOpenChildrenBefore((oc), (oi));                      \    */
/*      i2 = KheDrsOpenChildrenBefore((oc), (oi) + 1);                  \    */
/*      for( (i) = i1;                                                  \    */
/*           (i) < i2 ? ((x) = HaArray((oc)->children, (i)), true) : false;\ */
/*           (i)++ )                                                         */
/*                                                                           */
/*  sets x to each child in turn whose open index is oi.                     */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_open_children_rec {
  ARRAY_KHE_DRS_EXPR	children;
  KHE_DRS_DAY_RANGE	range;
  HA_ARRAY_INT		child_indexes;
} *KHE_DRS_OPEN_CHILDREN;


/*****************************************************************************/
/*                                                                           */
/*  INHERIT_KHE_DRS_EXPR - fields inherited by all expression types          */
/*                                                                           */
/*****************************************************************************/

#define INHERIT_KHE_DRS_EXPR						\
  KHE_DRS_EXPR_TAG	tag;			/* tag field      */	\
  bool			gathered;		/* for opening    */	\
  int			postorder_index;	/* expr index     */	\
  KHE_DRS_RESOURCE	resource;		/* if resource    */	\
  KHE_DRS_VALUE		u;			/* value          */	\
  ARRAY_KHE_DRS_PARENT	parents;		/* parent exprs   */	\
  ARRAY_KHE_DRS_EXPR	children;		/* child exprs    */	\
  struct khe_drs_open_children_rec open_children_by_day; /* sorted by day  */ \
  /* KHE_DRS_DAY_RANGE	open_day_range; */	/* open day range */	\
  /* ARRAY_KHE_DRS_EXPR	open_children; */	/* open children  */	\
  /* HA_ARRAY_INT	open_child_indexes; */	/* for each day   */	\
  HA_ARRAY_INT		sig_indexes;		/* for each day   */


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR - abstract supertype of expressions                    */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_expr_rec {
  INHERIT_KHE_DRS_EXPR
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_ASSIGNED_TASK                                          */
/*                                                                           */
/*  An expression derived from a task on day, whose value is 1 if the task   */
/*  is assigned a resource from the given resource group, and 0 otherwise.   */
/*                                                                           */
/*  We include the task on day because we merge equal expressions of this    */
/*  kind, and that is only possible when the task is in the expression.      */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expr_assigned_task_rec {
  INHERIT_KHE_DRS_EXPR
  KHE_DRS_TASK_ON_DAY		task_on_day;
  KHE_RESOURCE_GROUP		resource_group;
} *KHE_DRS_EXPR_ASSIGNED_TASK;

typedef HA_ARRAY(KHE_DRS_EXPR_ASSIGNED_TASK) ARRAY_KHE_DRS_EXPR_ASSIGNED_TASK;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_BUSY_TIME                                              */
/*                                                                           */
/*  An expression whose value is 1 when the resource is busy at a given      */
/*  time, and 0 when the resource is free then.  The expression stores the   */
/*  time, but only for debugging.                                            */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expr_busy_time_rec {
  INHERIT_KHE_DRS_EXPR
  KHE_DRS_RESOURCE_ON_DAY	resource_on_day;  /* the resource and day    */
  KHE_TIME			time;		/* the time we're busy at    */
} *KHE_DRS_EXPR_BUSY_TIME;

typedef HA_ARRAY(KHE_DRS_EXPR_BUSY_TIME) ARRAY_KHE_DRS_EXPR_BUSY_TIME;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_FREE_TIME                                              */
/*                                                                           */
/*  An expression whose value is 1 when the resource is free at a given      */
/*  time, and 0 when the resource is busy then.  The expression stores the   */
/*  time, but only for debugging.                                            */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expr_free_time_rec {
  INHERIT_KHE_DRS_EXPR
  KHE_DRS_RESOURCE_ON_DAY	resource_on_day;  /* the resource and day    */
  KHE_TIME			time;		/* the time we're free at    */
} *KHE_DRS_EXPR_FREE_TIME;

typedef HA_ARRAY(KHE_DRS_EXPR_FREE_TIME) ARRAY_KHE_DRS_EXPR_FREE_TIME;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_WORK_TIME                                              */
/*                                                                           */
/*  An expression whose value is the workload incurred by the resource at    */
/*  a given time.  This will be 0.0 if the resource is free then.  The       */
/*  expression stores the time, but only for debugging.                      */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expr_work_time_rec {
  INHERIT_KHE_DRS_EXPR
  KHE_DRS_RESOURCE_ON_DAY	resource_on_day;  /* the resource and day    */
  KHE_TIME			time;		/* the time of this workload */
} *KHE_DRS_EXPR_WORK_TIME;

typedef HA_ARRAY(KHE_DRS_EXPR_WORK_TIME) ARRAY_KHE_DRS_EXPR_WORK_TIME;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_BUSY_DAY                                               */
/*                                                                           */
/*  An expression whose value is 1 when the resource is busy on a given      */
/*  day, and 0 when the resource is free then.  The expression stores the    */
/*  day (as a time group), and not just for debugging.                       */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expr_busy_day_rec {
  INHERIT_KHE_DRS_EXPR
  KHE_DRS_RESOURCE_ON_DAY	resource_on_day;  /* the resource and day    */
  KHE_TIME_GROUP		time_group;	/* the day we're busy at     */
} *KHE_DRS_EXPR_BUSY_DAY;

typedef HA_ARRAY(KHE_DRS_EXPR_BUSY_DAY) ARRAY_KHE_DRS_EXPR_BUSY_DAY;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_FREE_DAY                                               */
/*                                                                           */
/*  An expression whose value is 1 when the resource is free on a given      */
/*  day, and 0 when the resource is busy then.  The expression stores the    */
/*  day (as a time group), and not just for debugging.                       */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expr_free_day_rec {
  INHERIT_KHE_DRS_EXPR
  KHE_DRS_RESOURCE_ON_DAY	resource_on_day;  /* the resource and day    */
  KHE_TIME_GROUP		time_group;	/* the day we're free at     */
} *KHE_DRS_EXPR_FREE_DAY;

typedef HA_ARRAY(KHE_DRS_EXPR_FREE_DAY) ARRAY_KHE_DRS_EXPR_FREE_DAY;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_WORK_DAY                                               */
/*                                                                           */
/*  An expression whose value is the workload incurred by the resource on    */
/*  a given day.  This will be 0.0 if the resource is free then.  The        */
/*  expression stores the day (as a time group), but only for debugging.     */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expr_work_day_rec {
  INHERIT_KHE_DRS_EXPR
  KHE_DRS_RESOURCE_ON_DAY	resource_on_day;  /* the resource and day    */
  KHE_TIME_GROUP		time_group;	/* the day we're free at     */
} *KHE_DRS_EXPR_WORK_DAY;

typedef HA_ARRAY(KHE_DRS_EXPR_WORK_DAY) ARRAY_KHE_DRS_EXPR_WORK_DAY;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_OR                                                     */
/*                                                                           */
/*  An expression whose value is the logical `or' of its children's values.  */
/*  Here Booleans are represented by integers (0 for false, 1 for true).     */
/*                                                                           */
/*  The closed state here is the number of closed children with value 1.     */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expr_or_rec {
  INHERIT_KHE_DRS_EXPR
  int				closed_state;
} *KHE_DRS_EXPR_OR;

typedef HA_ARRAY(KHE_DRS_EXPR_OR) ARRAY_KHE_DRS_EXPR_OR;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_AND                                                    */
/*                                                                           */
/*  An expression whose value is the logical `and' of its children's values. */
/*  Here Booleans are represented by integers (0 for false, 1 for true).     */
/*                                                                           */
/*  The closed state here is the number of closed children with value 0.     */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expr_and_rec {
  INHERIT_KHE_DRS_EXPR
  int				closed_state;
} *KHE_DRS_EXPR_AND;

typedef HA_ARRAY(KHE_DRS_EXPR_AND) ARRAY_KHE_DRS_EXPR_AND;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_INT_SUM                                                */
/*                                                                           */
/*  An expression whose value is the sum of its children's values.           */
/*                                                                           */
/*  The closed state here is the sum of the values of the closed children.   */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expr_int_sum_rec {
  INHERIT_KHE_DRS_EXPR
  int				closed_state;
} *KHE_DRS_EXPR_INT_SUM;

typedef HA_ARRAY(KHE_DRS_EXPR_INT_SUM) ARRAY_KHE_DRS_EXPR_INT_SUM;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_FLOAT_SUM                                              */
/*                                                                           */
/*  An expression whose value is the sum of its children's values.  The      */
/*  expression's value, and its children's values, have type float.          */
/*                                                                           */
/*  The closed state here is the sum of the values of the closed children.   */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expr_float_sum_rec {
  INHERIT_KHE_DRS_EXPR
  float				closed_state;
} *KHE_DRS_EXPR_FLOAT_SUM;

typedef HA_ARRAY(KHE_DRS_EXPR_FLOAT_SUM) ARRAY_KHE_DRS_EXPR_FLOAT_SUM;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_INT_DEV                                                */
/*                                                                           */
/*  An expression whose value is the deviation of its integer-valued child.  */
/*                                                                           */
/*  The closed state here is the value of the closed child.  It is not       */
/*  stored here explicitly; it is retrieved from the child when required.    */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expr_int_dev_rec {
  INHERIT_KHE_DRS_EXPR
  int				min_limit;	/* minimum limit             */
  int				max_limit;	/* maximum limit             */
  bool				allow_zero;	/* true if 0 has deviation 0 */
} *KHE_DRS_EXPR_INT_DEV;

typedef HA_ARRAY(KHE_DRS_EXPR_INT_DEV) ARRAY_KHE_DRS_EXPR_INT_DEV;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_FLOAT_DEV                                              */
/*                                                                           */
/*  An expression whose value is the integer-valued deviation of its         */
/*  float-valued child.                                                      */
/*                                                                           */
/*  The closed state here is the value of the closed child.  It is not       */
/*  stored here explicitly; it is retrieved from the child when required.    */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expr_float_dev_rec {
  INHERIT_KHE_DRS_EXPR
  int				min_limit;	/* minimum limit             */
  int				max_limit;	/* maximum limit             */
  bool				allow_zero;	/* true if 0 has deviation 0 */
} *KHE_DRS_EXPR_FLOAT_DEV;

typedef HA_ARRAY(KHE_DRS_EXPR_FLOAT_DEV) ARRAY_KHE_DRS_EXPR_FLOAT_DEV;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_COST                                                   */
/*                                                                           */
/*  This type is the abstract supertype of KHE_DRS_EXPR_INT_SUM_COST and     */
/*  KHE_DRS_EXPR_INT_SEQ_COST.  It is used, but not much.                    */
/*                                                                           */
/*****************************************************************************/

#define INHERIT_KHE_DRS_EXPR_COST					   \
  INHERIT_KHE_DRS_EXPR							   \
  KHE_COST_FUNCTION	cost_fn;		/* cost function        */ \
  KHE_COST		combined_weight;	/* combined weight      */

typedef struct khe_drs_expr_cost_rec {
  INHERIT_KHE_DRS_EXPR_COST
} *KHE_DRS_EXPR_COST;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_INT_SUM_COST                                           */
/*                                                                           */
/*  Combines INT_SUM, INT_DEV, and COST into a single expression.  This      */
/*  allows important optimizations in some cases, depending on adjust_type.  */
/*                                                                           */
/*  The closed state here is the sum of the values of the closed children.   */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expr_int_sum_cost_rec {
  INHERIT_KHE_DRS_EXPR_COST
  KHE_DRS_MONITOR_INFO		monitor_info;		/* monitor (debug)   */
  int				min_limit;		/* minimum limit     */
  int				max_limit;		/* maximum limit     */
  KHE_DRS_ADJUST_TYPE		adjust_type;		/* type of adjustment*/
  int				history_before;		/* history value befo*/
  int				history_after;		/* history value afte*/
  int				history;		/* history value     */
  int				closed_state;		/* closed state      */
  bool				allow_zero;		/* allow zero flag   */
  KHE_DRS_DIM3_TABLE		main_dom_table3;	/* dominance testing */
  KHE_DRS_DIM5_TABLE		corr_dom_table5;	/* dominance testing */
  struct khe_drs_open_children_rec open_children_by_shift; /* sorted by shift*/
} *KHE_DRS_EXPR_INT_SUM_COST;

typedef HA_ARRAY(KHE_DRS_EXPR_INT_SUM_COST) ARRAY_KHE_DRS_EXPR_INT_SUM_COST;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_CLOSED_SEQ                                                  */
/*                                                                           */
/*  A sequence of consecutive closed children, summarized.  The fields are   */
/*                                                                           */
/*    start_index                                                            */
/*      The index in the sequence of all children where the sequence starts. */
/*                                                                           */
/*    stop_index                                                             */
/*      The index in the sequence of all children where the sequence         */
/*      stops; that is, the first child index not included in it.  So        */
/*      start_index <= stop_index, and when they are equal, the sequence     */
/*      is empty.                                                            */
/*                                                                           */
/*    active_at_left                                                         */
/*      The length of the a-interval at the extreme left, or 0 if none.      */
/*                                                                           */
/*    active_at_right                                                        */
/*      The length of the a-interval at the extreme right, or 0 if none.     */
/*                                                                           */
/*  If there is a single a-interval running the full length of the           */
/*  closed sequence (if all its children are active), then                   */
/*                                                                           */
/*    active_at_left == active_at_right == stop_index - start_index          */
/*                                                                           */
/*****************************************************************************/

typedef struct {
  int		start_index;
  int		stop_index;
  int		active_at_left;
  int		active_at_right;
} *KHE_DRS_CLOSED_SEQ;

typedef HA_ARRAY(KHE_DRS_CLOSED_SEQ) ARRAY_KHE_DRS_CLOSED_SEQ;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_A_INTERVAL - one a-interval                                 */
/*                                                                           */
/*  This may include indexes from history.                                   */
/*                                                                           */
/*  Note.  If the length of ai is 0, it does not matter whether an           */
/*  unassigned child precedes it or not, because its cost will be 0          */
/*  anyway.  However we must keep the value correct because there may        */
/*  be a merge later that makes it relevant.                                 */
/*                                                                           */
/*****************************************************************************/

typedef struct {
  int		start_index;		/* index of first child              */
  int		stop_index;		/* index after last child            */
  bool		unassigned_precedes;	/* true if unassigned child precedes */
} KHE_DRS_A_INTERVAL;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_AU_INTERVAL - one au-interval                               */
/*                                                                           */
/*  This may include indexes from history and history_after.                 */
/*                                                                           */
/*****************************************************************************/

typedef struct {
  int		start_index;		/* index of first child              */
  int		stop_index;		/* index after last child            */
  bool		has_active_child;	/* true if contains an active child  */
} KHE_DRS_AU_INTERVAL;


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SEQ_TYPE - whether int seq type echoes whole frame               */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  KHE_DRS_SEQ_NONE,				/* none of what follows      */
  KHE_DRS_SEQ_DAYS_POSITIVE,			/* all days, all positive    */
  KHE_DRS_SEQ_DAYS_NEGATIVE,			/* all days, all negative    */
  KHE_DRS_SEQ_SINGLE_POSITIVE			/* single times, all days    */
} KHE_DRS_SEQ_TYPE;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPR_INT_SEQ_COST                                           */
/*                                                                           */
/*  Combines INT_SEQ, INT_DEV (without allow_zero), and COST into a single   */
/*  expression, for the same optimizations as KHE_DRS_EXPR_INT_SUM_COST.     */
/*                                                                           */
/*  The first_open_child_index contains the index in open_children of the    */
/*  first open child.  This will usually be 0, but during closing the        */
/*  children are closed in increasing index order and this field increases.  */
/*                                                                           */
/*  Field e->closed_seqs contains the closed sequences of e.  When e is      */
/*  closed there is just one of these, containing all the children of e.     */
/*  When e is open there is one for each gap between adjacent open children  */
/*  of e, plus one for before the first open child and one for after the     */
/*  last open child.  This is one more closed sequence than the number of    */
/*  open children.  History children are not included in closed sequences.   */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expr_int_seq_cost_rec {
  INHERIT_KHE_DRS_EXPR_COST
  KHE_DRS_MONITOR_INFO		monitor_info;		/* monitor (debug)   */
  KHE_DRS_SEQ_TYPE		seq_type;		/* echoes frame      */
  int				seq_index;		/* if seq_type       */
  int				min_limit;		/* minimum limit     */
  int				max_limit;		/* maximum limit     */
  KHE_DRS_ADJUST_TYPE		adjust_type;		/* type of adjustment*/
  int				history_before;		/* history value befo*/
  int				history_after;		/* history value afte*/
  int				history;		/* history value     */
  KHE_DRS_DIM5_TABLE		dom_table;		/* dominance tests   */
  /* KHE_DRS_DOM_TABLE_SEQ	dom_table; */		/* dominance tests   */
  int				first_open_child_index;	/* temporary         */
  ARRAY_KHE_DRS_CLOSED_SEQ	closed_seqs;		/* closed sequences  */
} *KHE_DRS_EXPR_INT_SEQ_COST;

typedef HA_ARRAY(KHE_DRS_EXPR_INT_SEQ_COST) ARRAY_KHE_DRS_EXPR_INT_SEQ_COST;


/*****************************************************************************/
/*                                                                           */
/*  Submodule "type definitions - assignments"                               */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_ASST_TO_TASK - assignment of resource to task               */
/*                                                                           */
/*  NB this is not a pointer type.                                           */
/*                                                                           */
/*  This is not the same as asst->fixed_task_on_day; that represents an      */
/*  assignment that is fixed from the start.  This is an assignment that     */
/*  initially may be free but subsequently gets fixed.                       */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_asst_to_task_rec {
  KHE_DRS_ASST_TO_TASK_CLASS	asst;			/* the assignment */
  KHE_DRS_TASK_ON_DAY		fixed_dtd;		/* what fixed to  */
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_ASST_TO_TASK_SET - a set of assignment to task objects      */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_asst_to_task_set_rec {
  ARRAY_KHE_DRS_ASST_TO_TASK	assts_to_tasks;
};

typedef HA_ARRAY(KHE_DRS_ASST_TO_TASK_SET) ARRAY_KHE_DRS_ASST_TO_TASK_SET;

#define KheDrsAsstToTaskSetForEach(dats, dat, i)			\
  HaArrayForEach(dats->assts_to_tasks, dat, i)


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_ASST_TO_TASK_CLASS - assignment of resource to task class   */
/*                                                                           */
/*  An object of this type is always non-NULL.  It represents one            */
/*  assignment of a given resource on day to a given task class or task      */
/*  on that day.  The asst_to_shift field is always non-NULL, and says       */
/*  which resource is being assigned by this assignment, and what the        */
/*  resource monitor signature is for that assignment.                       */
/*                                                                           */
/*  If task_class != NULL, then the assignment is free to be to any          */
/*  task of that task class.  Someone will have to decide which of the       */
/*  task class's tasks to use before the assignment can actually be made.    */
/*  In this case fixed_task_on_day is not used.  Its value will be NULL.     */
/*                                                                           */
/*  Otherwise, task_class == NULL.  The decision about which task to         */
/*  use has already been made, and fixed_task_on_day holds that decision.    */
/*  It could be NULL, in which case the decision is to assign a free day.    */
/*                                                                           */
/*  The skip_assts array holds assignments that can be skipped (owing to     */
/*  two-extra selection) if this assignment is used.  A non-zero skip_count  */
/*  value indicates that this assignment may be skipped.                     */
/*                                                                           */
/*  These next paragraphs are obsolete:                                      */
/*  The cost and signature are for the resource's resource constraints       */
/*  only.  They depend on the previous solution, evidently.                  */
/*                                                                           */
/*  The resource on day is given.  The task on day is deducible as follows:  */
/*                                                                           */
/*    Case 1: planned_task_class != NULL.                                    */
/*      In this case, planned_task_on_day is irrelevant and will be NULL.    */
/*      The task on day is obtained from planned_task_class.                 */
/*                                                                           */
/*    Case 2: planned_task_class == NULL && planned_task_on_day != NULL.     */
/*      In this case, the task on day is just planned_task_on_day.           */
/*                                                                           */
/*    Case 3: planned_task_class == NULL && planned_task_on_day == NULL.     */
/*      In this case, the task on day is NULL (a free day).                  */
/*                                                                           */
/*  After the choice is actually made, task_on_day holds that choice.  It    */
/*  changes as the solve proceeds.                                           */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_asst_to_task_class_rec {
  /* INHERIT_KHE_DRS_SIGNATURE */
  /* KHE_DRS_RESOURCE_ON_DAY		resource_on_day; */
  KHE_DRS_ASST_TO_SHIFT			asst_to_shift;
  KHE_DRS_TASK_CLASS			task_class;
  KHE_DRS_TASK_ON_DAY			fixed_task_on_day;
  /* KHE_DRS_TASK_CLASS			planned_task_class;  */
  /* KHE_DRS_TASK_ON_DAY		planned_task_on_day; */
  ARRAY_KHE_DRS_ASST_TO_TASK_CLASS	skip_assts;
  int					skip_count;
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_ASST_TO_SHIFT - assignment of a resource to a shift         */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_asst_to_shift_rec {
  KHE_DRS_RESOURCE_ON_DAY		resource_on_day;
  KHE_DRS_SIGNATURE			sig;
  /* struct khe_drs_signature_rec	sig; */
  bool					used;
};


/*****************************************************************************/
/*                                                                           */
/*  Submodule "type definitions - solutions"                                 */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_ASST_OP - an assignment operation                                */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  KHE_DRS_ASST_OP_UNASSIGN,
  KHE_DRS_ASST_OP_ASSIGN,
  KHE_DRS_ASST_OP_REPLACE
} KHE_DRS_ASST_OP;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_SOLN - a partial solution, part of the search tree          */
/*                                                                           */
/*  All valid values are non-NULL.  In the root solution the prev_soln and   */
/*  day fields are NULL, and prev_tasks is empty.                            */
/*                                                                           */
/*  Field priqueue_index of solution S is a back pointer into the priority   */
/*  queue, when that is in use.  Actually its meaning is more complicated:   */
/*                                                                           */
/*    * If S->priqueue_index == -1, S is not in the priority queue.  It has  */
/*      been expanded, which means that it cannot be freed until the end of  */
/*      the solve, since that would invalidate its descendants.              */
/*                                                                           */
/*    * If S->priqueue_index == 0, S is not in the priority queue, either    */
/*      because there is no priority queue, or because it has not been       */
/*      inserted yet, or because it was previously inserted but then it      */
/*      was deleted, owing to being found to be dominated by some other      */
/*      solution.  It has not been expanded, so it can be freed safely.      */
/*                                                                           */
/*  These first two cases apply whether there is a priority queue or not.    */
/*  When there is no priority queue the condition they record does not       */
/*  matter, but it is simpler to record it anyway.                           */
/*                                                                           */
/*    * If priqueue_index >= 1, this solution is currently in the priority   */
/*      queue, at position priqueue_index.  It has not been expanded.        */
/*                                                                           */
/*  Ordinarily, when a solution S lying in a solution set is found to be     */
/*  dominated by some other solution, it is removed from the solution set    */
/*  and from the priority queue and freed.  However, this is not done when   */
/*  S is expanded (when S->priqueue_index == -1).  Instead, S is left as     */
/*  is, just as though it was not dominated at all.  This has two benefits.  */
/*  First, every solution can be sure that its ancestors have not been       */
/*  freed; if any had been freed, that would make the prev_soln fields of    */
/*  solutions invalid.  Second, every unfreed solution lies in some          */
/*  solution set, so it will eventually be freed (at the end of the solve).  */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_soln_rec {
  struct khe_drs_signature_set_rec sig_set;	/* expanded to save space    */
  /* struct khe_drs_signature_rec sig; */	/* expanded to save space    */
  KHE_DRS_SOLN			prev_soln;	/* edge from prev_soln       */
  ARRAY_KHE_DRS_TASK_ON_DAY	prev_tasks;	/* indexed by open resources */
  int				priqueue_index;	/* priqueue index, if any    */
#if TESTING
  int				sorted_rank;	/* rank when solns sorted    */
#endif
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_PACKED_SOLN_DAY - one day of a packed solution              */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_packed_soln_day_rec {
  KHE_DRS_DAY			day;			/* day of this obj   */
  ARRAY_KHE_DRS_TASK_ON_DAY	prev_tasks;		/* tasks             */
} *KHE_DRS_PACKED_SOLN_DAY;

typedef HA_ARRAY(KHE_DRS_PACKED_SOLN_DAY) ARRAY_KHE_DRS_PACKED_SOLN_DAY;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_PACKED_SOLN - a packed solution object                      */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_packed_soln_rec {
  KHE_COST			cost;			/* soln cost         */
  ARRAY_KHE_DRS_PACKED_SOLN_DAY	days;			/* days of rerun     */
} *KHE_DRS_PACKED_SOLN;

typedef HA_ARRAY(KHE_DRS_PACKED_SOLN) ARRAY_KHE_DRS_PACKED_SOLN;


/*****************************************************************************/
/*                                                                           */
/*  Submodule "type definitions - expansion"                                 */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_EXPANDER - global values that vary during expansion         */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_expander_rec *KHE_DRS_EXPANDER;
typedef HA_ARRAY(KHE_DRS_EXPANDER) ARRAY_KHE_DRS_EXPANDER;

struct khe_drs_expander_rec {
  KHE_DYNAMIC_RESOURCE_SOLVER	solver;
  ARRAY_KHE_DRS_ASST_TO_TASK	assts_to_tasks;
  ARRAY_KHE_DRS_ASST_TO_TASK	tmp_assts_to_tasks;
  bool				whole_tasks;
  bool				open;
  KHE_COST			cost;
  KHE_COST			cost_limit;
  int				free_resource_count;
  int				must_assign_count;
  /* bool			expanding; */
  HA_ARRAY_INT			marks;
};


/*****************************************************************************/
/*                                                                           */
/*  Submodule "type definitions - sets of solutions"                         */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_SOLN_LIST - a list of solns; has several uses               */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_soln_list_rec {
  KHE_DYNAMIC_RESOURCE_SOLVER	solver;
  ARRAY_KHE_DRS_SOLN		solns;
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_SOLN_TRIE - a trie of solns                                 */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_soln_trie_rec *KHE_DRS_SOLN_TRIE;
typedef HA_ARRAY(KHE_DRS_SOLN_TRIE) ARRAY_KHE_DRS_SOLN_TRIE;

struct khe_drs_soln_trie_rec {
  KHE_DRS_SOLN			soln;
  ARRAY_KHE_DRS_SOLN_TRIE	children;
  int				base;		/* first index of children */
};


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_INDEXED_SOLN_SET - an array of soln lists, indexed by cost  */
/*                                                                           */
/*****************************************************************************/

typedef struct khe_drs_indexed_soln_set_rec {
  KHE_COST			base;
  KHE_COST			increment;
  ARRAY_KHE_DRS_SOLN_LIST	soln_lists;
  int				count;
} *KHE_DRS_INDEXED_SOLN_SET;

typedef HA_ARRAY(KHE_DRS_INDEXED_SOLN_SET) ARRAY_KHE_DRS_INDEXED_SOLN_SET;


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_SOLN_SET_PART - one part of a solution set                  */
/*                                                                           */
/*****************************************************************************/

#define INHERIT_KHE_DRS_SOLN_SET_PART					\
  KHE_DRS_DOM_KIND		dom_kind;		/* kind of dominance */

typedef struct khe_drs_soln_set_part_rec {
  INHERIT_KHE_DRS_SOLN_SET_PART
} *KHE_DRS_SOLN_SET_PART;

typedef struct khe_drs_soln_set_part_dom_none_rec {
  INHERIT_KHE_DRS_SOLN_SET_PART
  KHE_DRS_SOLN_LIST		soln_list;
} *KHE_DRS_SOLN_SET_PART_DOM_NONE;

typedef HA_ARRAY(KHE_DRS_SOLN_SET_PART_DOM_NONE)
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_NONE;

typedef struct khe_drs_soln_set_part_dom_weak_rec {
  INHERIT_KHE_DRS_SOLN_SET_PART
  TABLE_KHE_DRS_SOLN		soln_table;
} *KHE_DRS_SOLN_SET_PART_DOM_WEAK;

typedef HA_ARRAY(KHE_DRS_SOLN_SET_PART_DOM_WEAK)
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_WEAK;

typedef struct khe_drs_soln_set_part_dom_medium_rec {
  INHERIT_KHE_DRS_SOLN_SET_PART
  TABLE_KHE_DRS_SOLN_LIST	soln_list_table;
} *KHE_DRS_SOLN_SET_PART_DOM_MEDIUM;

typedef HA_ARRAY(KHE_DRS_SOLN_SET_PART_DOM_MEDIUM)
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_MEDIUM;

typedef struct khe_drs_soln_set_part_dom_strong_rec {
  INHERIT_KHE_DRS_SOLN_SET_PART
  KHE_DRS_SOLN_LIST		soln_list;
} *KHE_DRS_SOLN_SET_PART_DOM_STRONG;

typedef HA_ARRAY(KHE_DRS_SOLN_SET_PART_DOM_STRONG)
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_STRONG;

typedef struct khe_drs_soln_set_part_dom_trie_rec {
  INHERIT_KHE_DRS_SOLN_SET_PART
  KHE_DRS_SOLN_TRIE		soln_trie;
} *KHE_DRS_SOLN_SET_PART_DOM_TRIE;

typedef HA_ARRAY(KHE_DRS_SOLN_SET_PART_DOM_TRIE)
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_TRIE;

typedef struct khe_drs_soln_set_part_dom_indexed_rec {
  INHERIT_KHE_DRS_SOLN_SET_PART
  KHE_DRS_INDEXED_SOLN_SET	indexed_solns;
} *KHE_DRS_SOLN_SET_PART_DOM_INDEXED;

typedef HA_ARRAY(KHE_DRS_SOLN_SET_PART_DOM_INDEXED)
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_INDEXED;

typedef struct khe_drs_soln_set_part_dom_uniform_rec {
  INHERIT_KHE_DRS_SOLN_SET_PART
  KHE_DRS_SOLN_LIST		soln_list;
} *KHE_DRS_SOLN_SET_PART_DOM_UNIFORM;

typedef HA_ARRAY(KHE_DRS_SOLN_SET_PART_DOM_UNIFORM)
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_UNIFORM;

typedef struct khe_drs_soln_set_part_dom_indexed_uniform_rec {
  INHERIT_KHE_DRS_SOLN_SET_PART
  KHE_DRS_INDEXED_SOLN_SET	indexed_solns;
} *KHE_DRS_SOLN_SET_PART_DOM_INDEXED_UNIFORM;

typedef HA_ARRAY(KHE_DRS_SOLN_SET_PART_DOM_INDEXED_UNIFORM)
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_INDEXED_UNIFORM;

/* typedef HA_ARRAY(KHE_DRS_SOLN_SET_PART) ARRAY_KHE_DRS_SOLN_SET_PART; */


/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DRS_SOLN_SET - a set of solns, with a choice of implementations */
/*                                                                           */
/*****************************************************************************/

struct khe_drs_soln_set_rec {
  /* int			made_count; */		/* solns made        */
  KHE_DRS_SOLN_SET_PART		cache;			/* cache (optional)  */
  KHE_DRS_SOLN_SET_PART		main;			/* main part         */
};


/*****************************************************************************/
/*                                                                           */
/*  Submodule "type definitions - solving"                                   */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Type KHE_DYNAMIC_RESOURCE_SOLVER - a dynamic resource solver             */
/*                                                                           */
/*****************************************************************************/

struct khe_dynamic_resource_solver_rec {

  /* fields which are constant throughout the lifetime of the solver */
  HA_ARENA			arena;
  KHE_SOLN			soln;
  KHE_RESOURCE_TYPE		resource_type;
  KHE_OPTIONS			options;
  KHE_FRAME			days_frame;
  KHE_EVENT_TIMETABLE_MONITOR	etm;
  KHE_TASK_CLASS_SOLVER		task_class_solver;
  ARRAY_KHE_DRS_RESOURCE	all_resources;
  ARRAY_KHE_DRS_DAY		all_days;
  ARRAY_KHE_DRS_TASK		all_root_tasks;
  ARRAY_KHE_DRS_MONITOR_INFO	all_monitor_infos;
  int				postorder_count;
  /* int			shift_index_in_cycle; */
  KHE_COST			increment;		/* gcd of weights */

  /* free list fields (and one used list) */
  ARRAY_KHE_DRS_RESOURCE_SET		   resource_set_free_list;
  ARRAY_KHE_DRS_EXPANDER		   expander_free_list;
  /* ARRAY_KHE_DRS_RESOURCE_ON_DAY_SET	   resource_on_day_set_free_list; */
  ARRAY_KHE_DRS_SHIFT_ASST		   shift_asst_free_list;
  ARRAY_KHE_DRS_SHIFT_ASST_TRIE		   shift_asst_trie_free_list;
  ARRAY_KHE_DRS_CLOSED_SEQ		   closed_seq_free_list;
  ARRAY_KHE_DRS_DIM1_TABLE		   table1_free_list;
  ARRAY_KHE_DRS_DOM_TEST		   dom_test_free_list;
  ARRAY_KHE_DRS_SIGNATURE		   signature_free_list;
  ARRAY_KHE_DRS_SIGNER			   signer_free_list;
  /* ARRAY_KHE_DRS_SIGNATURE		   signature_used_list; */
  ARRAY_KHE_DRS_ASST_TO_TASK_SET	   asst_to_task_set_free_list;
  ARRAY_KHE_DRS_ASST_TO_TASK_CLASS	   asst_to_task_class_free_list;
  ARRAY_KHE_DRS_ASST_TO_SHIFT		   asst_to_shift_free_list;
  ARRAY_KHE_DRS_SOLN			   soln_free_list;
  ARRAY_KHE_DRS_SOLN_LIST		   soln_list_free_list;
  ARRAY_KHE_DRS_SOLN_TRIE		   soln_trie_free_list;
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_NONE	   soln_set_part_dom_none_free_list;
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_WEAK	   soln_set_part_dom_weak_free_list;
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_MEDIUM   soln_set_part_dom_medium_free_list;
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_STRONG   soln_set_part_dom_strong_free_list;
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_TRIE	   soln_set_part_dom_trie_free_list;
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_INDEXED  soln_set_part_dom_indexed_free_list;
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_UNIFORM  soln_set_part_dom_uniform_free_list;
  ARRAY_KHE_DRS_SOLN_SET_PART_DOM_INDEXED_UNIFORM
				   soln_set_part_dom_indexed_uniform_free_list;
  ARRAY_KHE_DRS_INDEXED_SOLN_SET	   indexed_soln_set_free_list;
  ARRAY_KHE_DRS_SOLN_SET		   soln_set_free_list;
  ARRAY_KHE_DRS_PACKED_SOLN_DAY		   packed_soln_day_free_list;
  ARRAY_KHE_DRS_PACKED_SOLN		   packed_soln_free_list;

  /* priority queue (always initialized, but actual use is optional) */
  KHE_PRIQUEUE			priqueue;

  /* fields that vary with the solve */
  ARRAY_KHE_DRS_DAY_RANGE	selected_day_ranges;	/* chosen for solve  */
  KHE_RESOURCE_SET		selected_resource_set;	/* chosen for solve  */
  bool				solve_priqueue;		/* use priqueue      */
  bool				solve_extra_selection;	/* use int. d. */
  bool				solve_expand_by_shifts;	/* expand by shifts  */
  bool				solve_correlated_exprs;
  int				solve_daily_expand_limit;
  int				solve_daily_prune_trigger;
  int				solve_resource_expand_limit;
  int				solve_dom_approx;
  KHE_DRS_DOM_KIND		solve_main_dom_kind;
  bool				solve_cache;
  KHE_DRS_DOM_KIND		solve_cache_dom_kind;
  KHE_DRS_DOM_TEST_TYPE		solve_dom_test_type;	/* from main_dom_kind*/
  KHE_COST			solve_init_cost;	/* before solving    */
  KHE_COST			solve_start_cost;	/* before first day  */
  ARRAY_KHE_DRS_DAY		open_days;
  ARRAY_KHE_DRS_SHIFT		open_shifts;
  /* int			open_shift_count; */
  /* ARRAY_KHE_DRS_RESOURCE	open_resources; */
  KHE_DRS_RESOURCE_SET		open_resources;
  ARRAY_KHE_DRS_EXPR		open_exprs;
  KHE_DRS_PACKED_SOLN		rerun;			/* non-NULL if rerun */

  /* fields used by expansion */
  /* KHE_DRS_RESOURCE_ON_DAY_SET	expand_drds; */
  /* int			expand_must_assign_count; */
  /* KHE_DRS_EXPANDER		expander; */
  /* ARRAY_KHE_DRS_ASST_TO_TASK_CLASS expand_prev_assts; */
  /* ARRAY_KHE_DRS_TASK_ON_DAY	expand_prev_task_on_days; */
  /* ARRAY_KHE_DRS_SIGNATURE	expand_prev_signatures; */
  /* ARRAY_KHE_DRS_ASST_TO_TASK	expand_assts_to_tasks; */
  /* ARRAY_KHE_DRS_TASK_ON_DAY	expand_dtds; */
  /* ARRAY_KHE_DRS_ASST_TO_TASK_CLASS expand_assts; */

  /* fields for recording running time and other statistics */
#if TESTING
  KHE_TIMER			timer;
  HA_ARRAY_INT			solns_made_per_day;
  HA_ARRAY_INT			table_size_per_day;
  HA_ARRAY_FLOAT		running_time_per_day;
  HA_ARRAY_INT			ancestor_freq;
  HA_ARRAY_INT			dominates_freq;
  int				max_open_day_index;
#endif
};


/*****************************************************************************/
/*                                                                           */
/*  Submodule " miscellaneous functions"                                     */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  int min(int a, int b)                                                    */
/*                                                                           */
/*  Return the smaller of a and b, or either if they are equal.              */
/*                                                                           */
/*****************************************************************************/

static int min(int a, int b)
{
  return a < b ? a : b;
}


/*****************************************************************************/
/*                                                                           */
/*  int max(int a, int b)                                                    */
/*                                                                           */
/*  Return the larger of a and b, or either if they are equal.               */
/*                                                                           */
/*****************************************************************************/

static int max(int a, int b)
{
  return a > b ? a : b;
}


/*****************************************************************************/
/*                                                                           */
/*  Major category "times"                                                   */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_DAY_RANGE"                                            */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DAY_RANGE KheDrsDayRangeMake(int first, int last)                */
/*                                                                           */
/*  Return a new interval with these endpoints.                              */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DAY_RANGE KheDrsDayRangeMake(int first, int last)
{
  KHE_DRS_DAY_RANGE res;
  res.first = first;
  res.last = last;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDayRangeAdd(KHE_DRS_DAY_RANGE *ddr, int index)                */
/*                                                                           */
/*  Update *ddr to include index.                                            */
/*                                                                           */
/*  Implementation note.  Both cases are needed here, as proved by a trace   */
/*  of Add([1, 0], 4).  The second case wrongly produces [1, 4].             */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDayRangeAdd(KHE_DRS_DAY_RANGE *ddr, int index)
{
  if( ddr->first > ddr->last )
  {
    /* ddr is empty, so make it cover just index */
    ddr->first = ddr->last = index;
  }
  else
  {
    /* ddr is non-empty, so expand it to include index */
    if( index < ddr->first )
      ddr->first = index;
    if( index > ddr->last )
      ddr->last = index;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDayRangeMerge(ARRAY_KHE_DRS_DAY_RANGE *day_ranges,            */
/*    KHE_DRS_DAY_RANGE ddr)                                                 */
/*                                                                           */
/*  Merge ddr into *day_ranges, a sorted array of day ranges.                */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDayRangeMerge(ARRAY_KHE_DRS_DAY_RANGE *day_ranges,
  KHE_DRS_DAY_RANGE ddr)
{
  KHE_DRS_DAY_RANGE ddr2;  int i;

  /* if ddr is empty, it changes nothing (this can't happen, actually) */
  if( ddr.first > ddr.last )
    return;

  HaArrayForEach(*day_ranges, ddr2, i)
  {
    /***********************************************************************/
    /*                                                                     */
    /*  Invariant: day_ranges and ddr may have changed, but it remains     */
    /*  true that merging ddr with day_ranges gives the desired result.    */
    /*  Also, day_ranges[0..i-1] precede and do not interact with ddr.     */
    /*                                                                     */
    /***********************************************************************/

    if( ddr2.last + 1 < ddr.first )
    {
      /* ddr2 precedes ddr and does not interact with it; continue */
      continue;
    }

    if( ddr.last + 1 < ddr2.first )
    {
      /* ddr precedes ddr2 and does not interact with it, or with day_ranges */
      HaArrayAdd(*day_ranges, i, ddr);
      return;
    }

    /* ddr2 interacts with ddr, so expand ddr to include ddr2, then */
    /* delete ddr2 and continue */
    if( ddr2.first < ddr.first )
      ddr.first = ddr2.first;
    if( ddr2.last > ddr.last )
      ddr.last = ddr2.last;
    HaArrayDeleteAndShift(*day_ranges, i);
    i--;
  }

  /* ddr does not interact with any day_ranges, and it follows them all */
  HaArrayAddLast(*day_ranges, ddr);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsDayRangeContains(KHE_DRS_DAY_RANGE ddr, int i)                */
/*                                                                           */
/*  Return true if ddr contains i.                                           */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused
static bool KheDrsDayRangeContains(KHE_DRS_DAY_RANGE ddr, int i)
{
  return ddr.first <= i && i <= ddr.last;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsDayRangeSubset(KHE_DRS_DAY_RANGE ddr1, KHE_DRS_DAY_RANGE ddr2)*/
/*                                                                           */
/*  Return true if ddr1 is a subset of ddr2.                                 */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsDayRangeSubset(KHE_DRS_DAY_RANGE ddr1, KHE_DRS_DAY_RANGE ddr2)
{
  return ddr1.first >= ddr2.first && ddr1.last <= ddr2.last;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDayRangeDebug(KHE_DRS_DAY_RANGE ddr,                          */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)  */
/*                                                                           */
/*  Debug print of ddr onto fp with the given verbosity and indent.          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDayRangeDebug(KHE_DRS_DAY_RANGE ddr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)
{
  KHE_TIME_GROUP tg1, tg2;
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  tg1 = KheFrameTimeGroup(drs->days_frame, ddr.first);
  tg2 = KheFrameTimeGroup(drs->days_frame, ddr.last);
  if( tg1 == tg2 )
    fprintf(fp, "%s", KheTimeGroupId(tg1));
  else
    fprintf(fp, "%s-%s", KheTimeGroupId(tg1), KheTimeGroupId(tg2));
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDayRangesDebug(ARRAY_KHE_DRS_DAY_RANGE *day_ranges,           */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of *day_ranges onto fp with the given verbosity and indent.  */
/*                                                                           */
/*****************************************************************************/

/* *** good but currently unused
static void KheDrsDayRangesDebug(ARRAY_KHE_DRS_DAY_RANGE *day_ranges,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)
{
  KHE_DRS_DAY_RANGE ddr;  int i;
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  fprintf(fp, "[");
  HaArrayForEach(*day_ranges, ddr, i)
  {
    if( i > 0 )
      fprintf(fp, ", ");
    KheDrsDayRangeDebug(ddr, drs, 1, -1, fp);
  }
  fprintf(fp, "]");
  if( indent >= 0 )
    fprintf(fp, "\n");
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_DAY"                                                  */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DAY KheDrsDayMake(int frame_index,                               */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new day object with the given attributes.  Its task classes,      */
/*  expressions, and dominance tests will be added later.                    */
/*                                                                           */
/*  Here frame_index is the index of the new day in the common frame.        */
/*                                                                           */
/*****************************************************************************/
static KHE_DRS_SIGNER_SET KheDrsSignerSetMake(KHE_DRS_DAY day,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);

static KHE_DRS_DAY KheDrsDayMake(int frame_index,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DAY res;
  HaMake(res, drs->arena);
  res->frame_index = frame_index;
  res->open_day_index = -1;  /* signals not currently open for solving */
  res->time_group = KheFrameTimeGroup(drs->days_frame, frame_index);
  /* HaArrayInit(res->task_classes, drs->arena); */
  HaArrayInit(res->shifts, drs->arena);
  /* HaArrayInit(res->internal_today, drs->arena); */
  res->signer_set = KheDrsSignerSetMake(res, drs);
  /* HaArrayInit(res->nr_internal_today, drs->arena); */
  /* HaArrayInit(res->dom_tests, drs->arena); */
  /* HaArrayInit(res->eq_dom_test_indexes, drs->arena); */
  res->soln_set = NULL;   /* initialized at the start of each solve */
  res->soln_list = NULL;  /* initialized at the start of each solve */
  /* res->single_day_curr_cost = res->single_day_min_cost = 0; */
  res->soln_made_count = 0;
  res->solve_expand_count = 0;
  /* res->solve_prune_cost = 0; */
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheDrsDayId(KHE_DRS_DAY day)                                       */
/*                                                                           */
/*  Return the Id of day.  This is just the Id of its time group.  A NULL    */
/*  day is allowed and "-" is returned in that case.                         */
/*                                                                           */
/*****************************************************************************/

static char *KheDrsDayId(KHE_DRS_DAY day)
{
  return day == NULL ? "-" : KheTimeGroupId(day->time_group);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDayAddTaskClass(KHE_DRS_DAY day, KHE_DRS_TASK_CLASS dtc,      */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Add dtc to the day's task classes and shifts.                            */
/*                                                                           */
/*****************************************************************************/
static KHE_DRS_SHIFT KheDrsShiftMake(KHE_DRS_DAY day, KHE_DRS_TASK_CLASS dtc,
  /* int index_in_day, */ KHE_DYNAMIC_RESOURCE_SOLVER drs);
static bool KheDrsShiftAcceptsClass(KHE_DRS_SHIFT ds, KHE_DRS_TASK_CLASS dtc);

static void KheDrsDayAddTaskClass(KHE_DRS_DAY day, KHE_DRS_TASK_CLASS dtc,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SHIFT ds;  int i;

  /* add dtc to day->task_classes */
  /* HaArrayAddLast(day->task_classes, dtc); */

  /* add dtc to day->shifts */
  HaArrayForEach(day->shifts, ds, i)
    if( KheDrsShiftAcceptsClass(ds, dtc) )
    {
      dtc->encl_shift = ds;
      return;
    }

  /* need a new shift */
  ds = KheDrsShiftMake(day, dtc, /* HaArrayCount(day->shifts), */ drs);
  dtc->encl_shift = ds;
  HaArrayAddLast(day->shifts, ds);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDayOpen(KHE_DRS_DAY day, KHE_DRS_DAY_RANGE ddr,               */
/*    int open_day_index, KHE_DRS_DOM_KIND dom_kind,                         */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Open day for solving.  It lies within day range ddr, its index in the    */
/*  the sequence of all open days is open_day_index, and dom_kind is the     */
/*  kind of dominance testing to use on this day.                            */
/*                                                                           */
/*  The signer is clear initially and cleared each time the day is closed,   */
/*  so there is no need to clear it here, and we don't.                      */
/*                                                                           */
/*  NB We open the day's task classes separately, because they may contain   */
/*  tasks for later days and opening assumes that those days are open.       */
/*                                                                           */
/*  NBB The assignment "day->solve_prune_cost = drs->solve_init_cost" is     */
/*  safe here because KheDrsDayOpen is called after drs->solve_init_cost     */
/*  is initialized.  (This comment is obsolete.)                             */
/*                                                                           */
/*****************************************************************************/
static void KheDrsTaskClassOpen(KHE_DRS_TASK_CLASS dtc, KHE_DRS_DAY_RANGE ddr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);
static KHE_DRS_SOLN_SET KheDrsSolnSetMake(KHE_DRS_DOM_KIND main_dom_kind,
  bool cache, KHE_DRS_DOM_KIND cache_dom_kind, KHE_DYNAMIC_RESOURCE_SOLVER drs);
static void KheDrsSignerSetOpen(KHE_DRS_SIGNER_SET signer_set,
  KHE_DRS_DAY day, KHE_DYNAMIC_RESOURCE_SOLVER drs);
static void KheDrsSignerSetDebug(KHE_DRS_SIGNER_SET signer_set,
  int verbosity, int indent, FILE *fp);

static void KheDrsDayOpen(KHE_DRS_DAY day, KHE_DRS_DAY_RANGE ddr,
  int open_day_index, KHE_DRS_DOM_KIND main_dom_kind, bool cache,
  KHE_DRS_DOM_KIND cache_dom_kind, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  /* KHE_DRS_TASK_CLASS dtc;  int i; */
  if( DEBUG18 )
    fprintf(stderr, "[ KheDrsDayOpen(%s, %d-%d, %d)\n", KheDrsDayId(day),
      ddr.first, ddr.last, open_day_index);
  day->open_day_index = open_day_index;
  KheDrsSignerSetOpen(day->signer_set, day, drs);
  /* *** doing this separately now
  HaArrayForEach(day->task_classes, dtc, i)
    KheDrsTaskClassOpen(dtc, ddr, drs);
  *** */
  /* HaArrayClear(day->internal_today); */
  /* KheDrsSignerClear(day->signer, drs); */
  /* HaArrayClear(day->nr_internal_today); */
  /* HaArrayClear(day->dom_tests); */
  /* HaArrayClear(day->eq_dom_test_indexes); */
  day->soln_set = KheDrsSolnSetMake(main_dom_kind, cache, cache_dom_kind, drs);
  day->soln_made_count = 0;
  day->solve_expand_count = 0;
  /* day->solve_prune_cost = drs->solve_init_cost; */
  if( DEBUG18 )
    fprintf(stderr, "] KheDrsDayOpen\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDayOpenShifts(KHE_DRS_DAY day, KHE_DRS_DAY_RANGE ddr,         */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Open the shifts of day, including opening their task classes.            */
/*                                                                           */
/*****************************************************************************/
static void KheDrsShiftOpen(KHE_DRS_SHIFT ds, KHE_DRS_DAY_RANGE ddr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsDayOpenShifts(KHE_DRS_DAY day, KHE_DRS_DAY_RANGE ddr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SHIFT shift;  int i;
  if( DEBUG18 )
    fprintf(stderr, "[ KheDrsDayOpenShifts(%s, %d-%d)\n",
      KheDrsDayId(day), ddr.first, ddr.last);
  HaArrayForEach(day->shifts, shift, i)
    KheDrsShiftOpen(shift, ddr, drs);
  /* ***
  HaArrayForEach(day->task_classes, dtc, i)
    KheDrsTaskClassOpen(dtc, ddr, drs);
  *** */
  if( DEBUG18 )
    fprintf(stderr, "] KheDrsDayOpenShifts\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDayOpenTaskClasses(KHE_DRS_DAY day, KHE_DRS_DAY_RANGE ddr,    */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Open the task classes of day.                                            */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsDayOpenTaskClasses(KHE_DRS_DAY day, KHE_DRS_DAY_RANGE ddr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_TASK_CLASS dtc;  int i;
  if( DEBUG18 )
    fprintf(stderr, "[ KheDrsDayOpenTaskClasses(%s, %d-%d)\n",
      KheDrsDayId(day), ddr.first, ddr.last);
  HaArrayForEach(day->task_classes, dtc, i)
    KheDrsTaskClassOpen(dtc, ddr, drs);
  if( DEBUG18 )
    fprintf(stderr, "] KheDrsDayOpenTaskClasses\n");
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDayAddOpenExpr(KHE_DRS_DAY day, KHE_DRS_EXPR e)               */
/*                                                                           */
/*  Add open expression e to day.                                            */
/*                                                                           */
/*****************************************************************************/
static void KheDrsSignerAddOpenExpr(KHE_DRS_SIGNER dsg, KHE_DRS_EXPR e);

static void KheDrsDayAddOpenExpr(KHE_DRS_DAY day, KHE_DRS_EXPR e)
{
  KHE_DRS_SIGNER dsg;
  HnAssert(e->resource == NULL, "KheDrsDayAddOpenExpr internal error");
  HnAssert(HaArrayCount(day->signer_set->signers) > 0,
    "KheDrsDayAddOpenExpr internal error");
  dsg = HaArrayLast(day->signer_set->signers);
  KheDrsSignerAddOpenExpr(dsg, e);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsDayAddDomTest(KHE_DRS_DAY day, KHE_DRS_DOM_TEST dom_test)      */
/*                                                                           */
/*  Add dom_test to day and return its index in day->dom_tests.              */
/*                                                                           */
/*****************************************************************************/
/* static bool KheDrsDomTestIsStrictEquality(KHE_DRS_DOM_TEST dom_test); */
static int KheDrsSignerAddDomTest(KHE_DRS_SIGNER dsg,
  KHE_DRS_DOM_TEST dom_test, KHE_DYNAMIC_RESOURCE_SOLVER drs);

static int KheDrsDayAddDomTest(KHE_DRS_DAY day, KHE_DRS_DOM_TEST dom_test,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SIGNER dsg;
  HnAssert(HaArrayCount(day->signer_set->signers) > 0,
    "KheDrsDayAddOpenExpr internal error");
  dsg = HaArrayLast(day->signer_set->signers);
  return KheDrsSignerAddDomTest(dsg, dom_test, drs);

  /* ***
  int res;

  ** add dom_test to day **
  res = HaArrayCount(day->dom_tests);
  HaArrayAddLast(day->dom_tests, dom_test);

  ** if dom_test is equality, record it in eq_dom_test_indexes **
  if( KheDrsDomTestIsStrictEquality(dom_test) )
    HaArrayAddLast(day->eq_dom_test_indexes, res);

  ** return dom_test's index **
  return res;
  *** */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDayOpeningHasEnded(KHE_DRS_DAY day)                           */
/*                                                                           */
/*  Inform day that opening has ended.                                       */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsSignerOpeningHasEnded(KHE_DRS_SIGNER dsg);

static void KheDrsDayOpeningHasEnded(KHE_DRS_DAY day)
{
  KheDrsSignerOpeningHasEnded(day->signer);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDayClose(KHE_DRS_DAY day, KHE_DYNAMIC_RESOURCE_SOLVER drs)    */
/*                                                                           */
/*  Close day.                                                               */
/*                                                                           */
/*****************************************************************************/
static void KheDrsShiftClose(KHE_DRS_SHIFT ds, KHE_DYNAMIC_RESOURCE_SOLVER drs);
static void KheDrsSolnListFreeSolns(KHE_DRS_SOLN_LIST soln_list,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);
static void KheDrsSolnListFree(KHE_DRS_SOLN_LIST soln_list,
  /* bool free_solns, */ KHE_DYNAMIC_RESOURCE_SOLVER drs);
static KHE_DRS_SOLN_LIST KheDrsDayGatherSolns(KHE_DRS_DAY next_day,
  KHE_DYNAMIC_RESOURCE_SOLVER drs /* , int *made_count, int *kept_count */);
static void KheDrsSignerSetClose(KHE_DRS_SIGNER_SET signer_set,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsDayClose(KHE_DRS_DAY day, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int i /* , made_count, kept_count */;  KHE_DRS_SHIFT shift;
  day->open_day_index = -1;  /* signals no longer open */
  HaArrayForEach(day->shifts, shift, i)
    KheDrsShiftClose(shift, drs);
  KheDrsSignerSetClose(day->signer_set, drs);

  /* delete solution sets, solution lists, and solutions as required */
  if( day->soln_list != NULL )
  {
    KheDrsSolnListFreeSolns(day->soln_list, drs);
    KheDrsSolnListFree(day->soln_list, drs);
    day->soln_list = NULL;
  }
  if( day->soln_set != NULL )
  {
    KheDrsDayGatherSolns(day, drs /* , &made_count, &kept_count */);
    KheDrsSolnListFreeSolns(day->soln_list, drs);
    KheDrsSolnListFree(day->soln_list, drs);
    day->soln_list = NULL;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDaySingleCostBegin(KHE_DRS_DAY day)                           */
/*                                                                           */
/*  Begin keeping track of the single cost on day.                           */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsDaySingleCostBegin(KHE_DRS_DAY day)
{
  if( DEBUG22 )
    day->single_day_min_cost = KheCost(INT_MAX, INT_MAX);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDaySingleCostEnd(KHE_DRS_DAY day)                             */
/*                                                                           */
/*  End keeping track of the single cost on day.                             */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsDaySingleCostEnd(KHE_DRS_DAY day)
{
  if( DEBUG22 )
  {
    if( day->single_day_min_cost < KheCost(INT_MAX, INT_MAX) )
      fprintf(stderr, "  at end of day %s, single_day_min_cost = %.5f\n",
	KheTimeGroupId(day->time_group), KheCostShow(day->single_day_min_cost));
    else
      fprintf(stderr, "  at end of day %s, single_day_min_cost = inf\n",
	KheTimeGroupId(day->time_group));
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDaySingleCostSolnBegin(KHE_DRS_DAY day)                       */
/*                                                                           */
/*  Begin recording the single cost of one solution on day.                  */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsDaySingleCostSolnBegin(KHE_DRS_DAY day)
{
  if( DEBUG22 )
    day->single_day_curr_cost = 0;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDaySingleCostSolnEnd(KHE_DRS_DAY day)                         */
/*                                                                           */
/*  End recording the single cost of one solution on day.                    */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsDaySingleCostSolnEnd(KHE_DRS_DAY day)
{
  if( DEBUG22 )
  {
    if( day->single_day_curr_cost < day->single_day_min_cost )
      day->single_day_min_cost = day->single_day_curr_cost;
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDaySingleCostAddCost(KHE_DRS_DAY day, KHE_COST cost)          */
/*                                                                           */
/*  Add cost to the current single cost on day.                              */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsDaySingleCostAddCost(KHE_DRS_DAY day, KHE_COST cost)
{
  if( DEBUG22 )
    day->single_day_curr_cost += cost;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDayDomTestsDebug(KHE_DRS_DAY day, int verbosity,              */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of the dom tests of day.                                     */
/*                                                                           */
/*****************************************************************************/
/* ***
static char *KheDrsDomTestShow(KHE_DRS_DOM_TEST dt, char buff[100]);
static KHE_DRS_DOM_TEST KheDrsDomTestMake(KHE_DRS_DOM_TYPE type,
  bool tradeoff, KHE_COST tradeoff_cost);
*** */

/* *** can't conveniently do this any more
static void KheDrsDayDomTestsDebug(KHE_DRS_DAY day, int verbosity,
  int indent, FILE *fp)
{
  int dom_test_freq[KHE_DRS_DOM_EQ_LOOSE + 1];  int i;
  KHE_DRS_DOM_TEST dom_test;  char buff[100];

  ** initialize dom_test_freq **
  for( i = KHE_DRS_DOM_UNUSED;  i <= KHE_DRS_DOM_EQ_LOOSE;  i++ )
    dom_test_freq[i] = 0;

  ** accumulate dom_test_freq **
  HaArrayForEach(day->dom_tests, dom_test, i)
    dom_test_freq[dom_test.type]++;

  ** display frequencies **
  if( indent >= 0 )
    fprintf(fp, "%*s", indent, "");
  fprintf(fp, "[ Day %s dom tests: ", KheDrsDayId(day));
  for( i = 0;  i <= KHE_DRS_DOM_EQ_LOOSE;  i++ )
  {
    if( i > 0 )
      fprintf(fp, ", ");
    fprintf(fp, "%d %s", dom_test_freq[i],
      KheDrsDomTestShow(KheDrsDomTestMake(i, false, 0), buff));
  }
  fprintf(fp, " ]");
  if( indent >= 0 )
    fprintf(fp, "\n");
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDayDebug(KHE_DRS_DAY day, KHE_DYNAMIC_RESOURCE_SOLVER drs,    */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of day onto fp.                                              */
/*                                                                           */
/*****************************************************************************/
static void KheDrsShiftDebug(KHE_DRS_SHIFT ds,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp);
static void KheDrsTaskClassDebug(KHE_DRS_TASK_CLASS dtc,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp);
static void KheDrsSolnSetDebug(KHE_DRS_SOLN_SET dns,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp);

static void KheDrsDayDebug(KHE_DRS_DAY day, KHE_DYNAMIC_RESOURCE_SOLVER drs,
  int verbosity, int indent, FILE *fp)
{
  int i;  KHE_DRS_SHIFT ds;
  if( indent >= 0 )
  {
    fprintf(fp, "%*s[ Day %s (frame %d, ", indent, "",
      KheDrsDayId(day), day->frame_index);
    if( day->open_day_index >= 0 )
      fprintf(fp, "open %d)\n", day->open_day_index);
    else
      fprintf(fp, "closed)\n");
    if( verbosity >= 2 )
    {
      HaArrayForEach(day->shifts, ds, i)
	KheDrsShiftDebug(ds, drs, verbosity, indent + 2, fp);
      /* ***
      if( day->open_day_index >= 0 )
	KheDrsDayDomTestsDebug(day, verbosity, indent + 2, fp);
      *** */
    }
    if( verbosity >= 3 && day->soln_set != NULL )
      KheDrsSolnSetDebug(day->soln_set, drs, verbosity, indent + 2, fp);
    fprintf(fp, "%*s]\n", indent, "");
  }
  else
    fprintf(fp, "%s", KheDrsDayId(day));
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDayDebugSignatureKey(KHE_DRS_DAY day,                         */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of a key to the signature on open day day.                   */
/*                                                                           */
/*****************************************************************************/
/* ***
static bool KheDrsExprInSignature(KHE_DRS_EXPR e, KHE_DRS_DAY day, int *si);
*** */
static void KheDrsExprDebug(KHE_DRS_EXPR e, KHE_DYNAMIC_RESOURCE_SOLVER drs,
  int verbosity, int indent, FILE *fp);

/* *** perfectly good, but currently unused
static void KheDrsDayDebugSignatureKey(KHE_DRS_DAY day,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)
{
  KHE_DRS_EXPR e;  int i, si;  bool print;
  KHE_DRS_EXPR_INT_SUM_COST e1;
  KHE_DRS_EXPR_INT_SEQ_COST e2;
  HnAssert(day->open_day_index >= 0,
    "KheDrsDayDebugSignatureKey internal error");
  if( verbosity == 1 )
  {
    ** print only the signature of the expression we are tracking **
    HaArrayForEach(day->internal_today, e, i)
      if( KheDrsExprInSignature(e, day, &si) )
      {
	switch( e->tag )
	{
	  case KHE_DRS_EXPR_INT_SUM_COST_TAG:

	    e1 = (KHE_DRS_EXPR_INT_SUM_COST) e;
	    print = (strcmp(KheMonitorId(e1->monitor), RERUN_MONITOR_ID)==0);
	    break;

	  case KHE_DRS_EXPR_INT_SEQ_COST_TAG:

	    e2 = (KHE_DRS_EXPR_INT_SEQ_COST) e;
	    print = (strcmp(KheMonitorId(e2->monitor), RERUN_MONITOR_ID)==0);
	    break;

	  default:

	    print = false;
	    break;
	}

	** e has an entry in the signature for this day **
	if( print )
	{
	  fprintf(fp, "%*s  %2d: ", indent, "", si);
	  KheDrsExprDebug(e, drs, 1, -1, fp);
	  fprintf(fp, "\n");
	}
      }
  }
  else
  {
    ** print the whole thing **
    fprintf(fp, "%*s[ Signature of %s:\n", indent, "", KheDrsDayId(day));
    HaArrayForEach(day->internal_today, e, i)
      if( KheDrsExprInSignature(e, day, &si) )
      {
	** e has an entry in the signature for this day **
	fprintf(fp, "%*s  %2d: ", indent, "", si);
	KheDrsExprDebug(e, drs, 1, -1, fp);
	fprintf(fp, "\n");
      }
    fprintf(fp, "%*s]\n", indent, "");
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_DAY - expansion"                                      */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDayExpandBegin(KHE_DRS_DAY next_day, KHE_DRS_SOLN prev_soln,  */
/*    KHE_DRS_EXPANDER de, int *must_assign_count)                           */
/*                                                                           */
/*  Begin the expansion of prev_soln into next_day.                          */
/*                                                                           */
/*  Implementation note - setting ds->expand_max_included_free_resource_count*/
/*  -----------------------------------------------------------------        */
/*  This field contains the maximum number of free resources that can be     */
/*  assigned to ds without making it impossible to fulfil the minimum        */
/*  requirements of other shifts.  First we calculate excess, the number of  */
/*  free resources that are not already committed to minimum requirements.   */
/*  Then ds->expand_max_included_free_resource_count is this number added to */
/*  the minimum for ds.  This can never exceed the number of free resources. */
/*                                                                           */
/*****************************************************************************/
static void KheDrsSolnSetBeginCacheSegment(KHE_DRS_SOLN_SET soln_set,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);
static void KheDrsShiftExpandBegin(KHE_DRS_SHIFT ds, KHE_DRS_SOLN prev_soln,
  KHE_DRS_EXPANDER de);
static int KheDrsExpanderExcessFreeResourceCount(KHE_DRS_EXPANDER de);

static void KheDrsDayExpandBegin(KHE_DRS_DAY next_day, KHE_DRS_SOLN prev_soln,
  KHE_DRS_EXPANDER de)
{
  int i, excess;  KHE_DRS_SHIFT ds;

  /* begin caching expansions */
  KheDrsSolnSetBeginCacheSegment(next_day->soln_set, de->solver);

  /* begin expansion in each shift */
  HaArrayForEach(next_day->shifts, ds, i)
    KheDrsShiftExpandBegin(ds, prev_soln, de);

  /* set expand_max_included_free_resource_count in each shift */
  /* excess = de ->free_resource_count - de ->must_assign_count; */
  excess = KheDrsExpanderExcessFreeResourceCount(de);
  HaArrayForEach(next_day->shifts, ds, i)
    ds->expand_max_included_free_resource_count =
      ds->expand_must_assign_count + excess;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDayExpandEnd(KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de)       */
/*                                                                           */
/*  End the previously begun expansion of prev_soln into next_day.           */
/*                                                                           */
/*****************************************************************************/
static void KheDrsShiftExpandEnd(KHE_DRS_SHIFT ds, KHE_DRS_EXPANDER de);
static void KheDrsSolnSetEndCacheSegment(KHE_DRS_SOLN_SET soln_set,
  KHE_DRS_DAY soln_day, KHE_DRS_EXPANDER de, KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsDayExpandEnd(KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de)
{
  int i;  KHE_DRS_SHIFT ds;

  /* end expansion in each shift */
  HaArrayForEach(next_day->shifts, ds, i)
    KheDrsShiftExpandEnd(ds, de);

  /* end caching expansions */
  KheDrsSolnSetEndCacheSegment(next_day->soln_set, next_day, de, de->solver);
}


/*****************************************************************************/
/*                                                                           */
/*  Major category "resources"                                               */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_RESOURCE_EXPAND_ROLE"                                 */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  char *KheDrsResourceExpandRoleShow(KHE_DRS_RESOURCE_EXPAND_ROLE role)    */
/*                                                                           */
/*  Return a readable string representation of role.                         */
/*                                                                           */
/*****************************************************************************/

static char *KheDrsResourceExpandRoleShow(KHE_DRS_RESOURCE_EXPAND_ROLE role)
{
  switch( role )
  {
    case KHE_DRS_RESOURCE_EXPAND_NO:	return "expand_no";
    case KHE_DRS_RESOURCE_EXPAND_FIXED:	return "expand_fixed";
    case KHE_DRS_RESOURCE_EXPAND_FREE:	return "expand_free";
    default:				return "?";
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_RESOURCE"                                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  char *KheDrsResourceId(KHE_DRS_RESOURCE dr)                              */
/*                                                                           */
/*  Return the Id of dr.                                                     */
/*                                                                           */
/*****************************************************************************/

static char *KheDrsResourceId(KHE_DRS_RESOURCE dr)
{
  return KheResourceId(dr->resource);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_RESOURCE KheDrsResourceMake(KHE_RESOURCE r,                      */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new resource object with these attributes.                        */
/*  Return NULL if unsuccessful.                                             */
/*                                                                           */
/*****************************************************************************/
static KHE_DRS_RESOURCE_ON_DAY KheDrsResourceOnDayMake(KHE_DRS_RESOURCE dr,
  KHE_DRS_DAY day, KHE_DYNAMIC_RESOURCE_SOLVER drs);
static KHE_DRS_DIM2_TABLE KheDrsDim2TableMake(char *debug_str, HA_ARENA a);

static KHE_DRS_RESOURCE KheDrsResourceMake(KHE_RESOURCE r,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_RESOURCE res;  int i;  KHE_DRS_RESOURCE_ON_DAY drd;
  KHE_DRS_DAY day;  /* KHE_MONITOR m; */

  /* initialize res, including one resource on day object for each day */
  if( DEBUG7 )
    fprintf(stderr, "[ KheDrsResourceMake(%s, drs)\n", KheResourceId(r));
  HaMake(res, drs->arena);
  res->resource = r;
  res->open_resource_index = -1;
  HaArrayInit(res->days, drs->arena);
  HaArrayForEach(drs->all_days, day, i)
  {
    drd = KheDrsResourceOnDayMake(res, day, drs);
    HaArrayAddLast(res->days, drd);
  }
  res->expand_role = KHE_DRS_RESOURCE_EXPAND_NO;
  HaArrayInit(res->expand_assts_to_shifts, drs->arena);
  /* res->expand_free_asst_to_shift = NULL; */
  HaArrayInit(res->expand_assts, drs->arena);
  res->expand_free_day_asst = NULL;
  res->expand_dom_test_cache = KheDrsDim2TableMake(KheDrsResourceId(res),
    drs->arena);

  /* add expressions based on the resource's monitors */
  /* *** moved to KheDrsResourceAddMonitors to fix sequencing problem
  KheSolnResourceMonitorSort(drs->soln, r, &KheMonitorLargestLimitFirstCmp);
  for( i = 0;  i < KheSolnResourceMonitorCount(drs->soln, r);  i++ )
  {
    m = KheSolnResourceMonitor(drs->soln, r, i);
    if( DEBUG7 )
      fprintf(stderr, "  %s monitor %s\n", KheResourceId(r), KheMonitorId(m));
    KheDrsAddResourceMonitor(drs, m, res);
  }
  *** */
  if( DEBUG7 )
    fprintf(stderr, "] KheDrsResourceMake returning\n");
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_RESOURCE KheDrsResource(KHE_RESOURCE r,                          */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Assuming that r has the correct resource type, return its corresponding  */
/*  drs resource.                                                            */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_RESOURCE KheDrsResource(KHE_RESOURCE r,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  HnAssert(KheResourceResourceTypeIndex(r) < HaArrayCount(drs->all_resources),
    "KheDrsResource internal error");
  return HaArray(drs->all_resources, KheResourceResourceTypeIndex(r));
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_RESOURCE_ON_DAY KheDrsResourceOnDay(KHE_DRS_RESOURCE dr,         */
/*    KHE_DRS_DAY day)                                                       */
/*                                                                           */
/*  Return the resource on day object representing dr on day.                */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_RESOURCE_ON_DAY KheDrsResourceOnDay(KHE_DRS_RESOURCE dr,
  KHE_DRS_DAY day)
{
  return HaArray(dr->days, day->frame_index);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_RESOURCE_ON_DAY KheDrsResourceOnDayAtTime(KHE_DRS_RESOURCE dr,   */
/*    KHE_TIME time, KHE_DYNAMIC_RESOURCE_SOLVER drs)                        */
/*                                                                           */
/*  Return the resource on day object whose resource is dr and whose day     */
/*  is time's day.                                                           */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_RESOURCE_ON_DAY KheDrsResourceOnDayAtTime(KHE_DRS_RESOURCE dr,
  KHE_TIME time, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int day_index;
  day_index = KheFrameTimeIndex(drs->days_frame, time);
  /* ***
  if( DEBU G4 )
    fprintf(stderr, "  in KheDrsResourceOnDayAtTime(dr %p %s, %s, drs)"
      " day_index = %d, dr->days = %d\n",
      (void *) dr, KheResourceId(dr->resource),
      KheTimeId(time), day_index, HaArrayCount(dr->days));
  *** */
  return HaArray(dr->days, day_index);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceOpen(KHE_DRS_RESOURCE dr, int open_resource_index,    */
/*    KHE_DRS_PACKED_SOLN init_soln, KHE_DYNAMIC_RESOURCE_SOLVER drs)        */
/*                                                                           */
/*  Open dr.  This involves unassigning all the tasks assigned dr which      */
/*  lie in the selected day ranges of drs, and gathering for opening the     */
/*  expressions that depend on what dr is doing on those days, including     */
/*  on days where dr remains assigned.                                       */
/*                                                                           */
/*  Also, if init_soln != NULL, included the unassignments as assignments    */
/*  in init_soln, so that they can be undone later if required.              */
/*                                                                           */
/*  The signer is clear initially and cleared each time the resource is   */
/*  closed, so there is no need to clear it here, and we don't.              */
/*                                                                           */
/*****************************************************************************/
static void KheDrsResourceOnDayOpen(KHE_DRS_RESOURCE_ON_DAY drd,
  int open_resource_index, int open_day_index, KHE_DRS_DAY_RANGE ddr,
  KHE_DRS_PACKED_SOLN init_soln, KHE_DYNAMIC_RESOURCE_SOLVER drs);
static void KheDrsResourceDebug(KHE_DRS_RESOURCE dr, int verbosity,
  int indent, FILE *fp);

static void KheDrsResourceOpen(KHE_DRS_RESOURCE dr, int open_resource_index,
  KHE_DRS_PACKED_SOLN init_soln, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DAY_RANGE ddr;  int i, j, open_day_index; KHE_DRS_RESOURCE_ON_DAY drd;
  if( DEBUG11 )
  {
    fprintf(stderr, "[ KheDrsResourceOpen(%s, %d, drs)\n", KheDrsResourceId(dr),
      open_resource_index);
    KheDrsResourceDebug(dr, 2, 2, stderr);
  }
  dr->open_resource_index = open_resource_index;
  open_day_index = 0;
  HaArrayForEach(drs->selected_day_ranges, ddr, i)
    for( j = ddr.first;  j <= ddr.last;  j++ )
    {
      /* unassign any task assigned on drd, if it lies entirely in ddr */
      drd = HaArray(dr->days, j);
      KheDrsResourceOnDayOpen(drd, open_resource_index, open_day_index,
	ddr, init_soln, drs);

      /* increase open_day_index */
      open_day_index++;
    }
  if( DEBUG11 )
  {
    fprintf(stderr, "  at end of KheDrsResourceOpen(%s, %d, drs):\n",
      KheDrsResourceId(dr), open_resource_index);
    KheDrsResourceDebug(dr, 2, 2, stderr);
    fprintf(stderr, "] KheDrsResourceOpen\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceClose(KHE_DRS_RESOURCE dr)                            */
/*                                                                           */
/*  Close dr.                                                                */
/*                                                                           */
/*****************************************************************************/
static void KheDrsResourceOnDayClose(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsResourceClose(KHE_DRS_RESOURCE dr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DAY_RANGE ddr;  int i, j;  KHE_DRS_RESOURCE_ON_DAY drd;
  HaArrayForEach(drs->selected_day_ranges, ddr, i)
    for( j = ddr.first;  j <= ddr.last;  j++ )
    {
      drd = HaArray(dr->days, j);
      KheDrsResourceOnDayClose(drd, drs);
    }
  dr->open_resource_index = -1;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceDebug(KHE_DRS_RESOURCE dr, int verbosity,             */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of dr onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/
static void KheDrsResourceOnDayDebug(KHE_DRS_RESOURCE_ON_DAY drd,
  int verbosity, int indent, FILE *fp);

static void KheDrsResourceDebug(KHE_DRS_RESOURCE dr, int verbosity,
  int indent, FILE *fp)
{
  KHE_DRS_RESOURCE_ON_DAY drd;  int i;
  if( verbosity > 1 && indent >= 0 )
  {
    /* show timetable in full */
    fprintf(fp, "%*s[ %s", indent, "", KheDrsResourceId(dr));
    if( dr->open_resource_index >= 0 )
      fprintf(fp, " (open %d)", dr->open_resource_index);
    fprintf(fp, "\n");
    HaArrayForEach(dr->days, drd, i)
      KheDrsResourceOnDayDebug(drd, verbosity, indent + 2, fp);
    fprintf(fp, "%*s]\n", indent, "");
  }
  else
  {
    if( indent > 0 )
      fprintf(fp, "%*s", indent, "");
    fprintf(fp, "%s", KheDrsResourceId(dr));
    if( indent >= 0 )
      fprintf(fp, "\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_RESOURCE - expansion"                                 */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceSetSignature(KHE_DRS_SIGNATURE next_sig,              */
/*    KHE_DRS_RESOURCE_ON_DAY drd, KHE_DRS_TASK_ON_DAY dtd,                  */
/*    KHE_DRS_SIGNATURE prev_sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)           */
/*                                                                           */
/*  Build a signature object for the effect on resource monitors when        */
/*  drd is assigned to dtd.  Only extra costs are included, not the          */
/*  cost of prev_soln.                                                       */
/*                                                                           */
/*****************************************************************************/
static void KheDrsResourceOnDayLeafSet(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_DRS_TASK_ON_DAY dtd, KHE_DYNAMIC_RESOURCE_SOLVER drs);
static void KheDrsResourceOnDayLeafClear(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_DRS_TASK_ON_DAY dtd, KHE_DYNAMIC_RESOURCE_SOLVER drs);
static void KheDrsSignerEvalSignature(KHE_DRS_SIGNER dsg,
  bool debug_if_rerun, KHE_DRS_SIGNATURE prev_sig, int next_di,
  KHE_DRS_SIGNATURE sig, KHE_DYNAMIC_RESOURCE_SOLVER drs, bool debug);

static void KheDrsResourceSetSignature(KHE_DRS_SIGNATURE next_sig,
  KHE_DRS_RESOURCE_ON_DAY drd, KHE_DRS_TASK_ON_DAY dtd,
  KHE_DRS_SIGNATURE prev_sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  /* set leaf expressions of the resource on day */
  KheDrsResourceOnDayLeafSet(drd, dtd, drs);

  /* build the signature */
  KheDrsSignerEvalSignature(drd->signer, true, prev_sig,
    drd->day->open_day_index, next_sig, drs, false);

  /* clear leaf expressions of the resource on day */
  KheDrsResourceOnDayLeafClear(drd, dtd, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_ASST_TO_TASK_CLASS KheDrsResourceAddAsst(KHE_DRS_RESOURCE dr,    */
/*    KHE_DRS_ASST_TO_SHIFT sasst, KHE_DRS_TASK_CLASS dtc,                   */
/*    KHE_DRS_TASK_ON_DAY fixed_dtd, KHE_DRS_EXPANDER de)                    */
/*                                                                           */
/*  Build an assignment object.  Add it to drd->encl_dr->expand_assts        */
/*  if it is competitive, and delete it if not.                              */
/*                                                                           */
/*****************************************************************************/
static KHE_DRS_ASST_TO_TASK_CLASS KheDrsAsstToTaskClassMake(
  KHE_DRS_ASST_TO_SHIFT sasst, KHE_DRS_TASK_CLASS dtc,
  KHE_DRS_TASK_ON_DAY fixed_dtd, KHE_DYNAMIC_RESOURCE_SOLVER drs);
static bool KheDrsExpanderOpenToExtraCost(KHE_DRS_EXPANDER de,
  KHE_COST extra_cost);

static KHE_DRS_ASST_TO_TASK_CLASS KheDrsResourceAddAsst(KHE_DRS_RESOURCE dr,
  KHE_DRS_ASST_TO_SHIFT sasst, KHE_DRS_TASK_CLASS dtc,
  KHE_DRS_TASK_ON_DAY fixed_dtd, bool force, KHE_DRS_EXPANDER de)
{
  KHE_DRS_ASST_TO_TASK_CLASS res;
  if( force || KheDrsExpanderOpenToExtraCost(de, sasst->sig->cost) )
  {
    res = KheDrsAsstToTaskClassMake(sasst, dtc, fixed_dtd, de->solver);
    HaArrayAddLast(dr->expand_assts, res);
    sasst->used = true;
    return res;
  }
  else
    return NULL;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceAsstToShiftAddOrFree(KHE_DRS_RESOURCE dr,             */
/*    KHE_DRS_ASST_TO_SHIFT sasst, KHE_DYNAMIC_RESOURCE_SOLVER drs)          */
/*                                                                           */
/*  Either store sasst in dr->expand_assts_to_shifts or free it, depending   */
/*  on sasst->used.                                                          */
/*                                                                           */
/*****************************************************************************/
static void KheDrsAsstToShiftFree(KHE_DRS_ASST_TO_SHIFT sasst,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsResourceAsstToShiftAddOrFree(KHE_DRS_RESOURCE dr,
  KHE_DRS_ASST_TO_SHIFT sasst, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  if( sasst->used )
    HaArrayAddLast(dr->expand_assts_to_shifts, sasst);
  else
    KheDrsAsstToShiftFree(sasst, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSortAndReduceAssignments(KHE_DRS_RESOURCE dr,                 */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Sort the signatures of dr, and keep the best dr->resource_expand_limit   */
/*  of them, or more if there are ties.                                      */
/*                                                                           */
/*****************************************************************************/
static void KheDrsAsstToTaskClassFree(KHE_DRS_ASST_TO_TASK_CLASS asst,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);
static KHE_DRS_SIGNATURE KheDrsAsstToTaskClassSignature(
  KHE_DRS_ASST_TO_TASK_CLASS asst);
static int KheDrsAsstToTaskClassCmp(const void *t1, const void *t2);

static void KheDrsSortAndReduceAssignments(KHE_DRS_RESOURCE dr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_ASST_TO_TASK_CLASS asst, asst2;  KHE_DRS_SIGNATURE sig, sig2;

  /* sort the assignments by increasing cost */
  HaArraySort(dr->expand_assts, &KheDrsAsstToTaskClassCmp);

  /* if there is a resource expand limit and it will make a difference here */
  if( drs->solve_resource_expand_limit > 0 &&
      drs->solve_resource_expand_limit < HaArrayCount(dr->expand_assts) )
  {
    /* find asst, a signature with the largest cost that we want to keep */
    asst = HaArray(dr->expand_assts, drs->solve_resource_expand_limit-1);
    sig = KheDrsAsstToTaskClassSignature(asst);

    /* delete and free all signatures whose cost exceeds asst's */
    asst2 = HaArrayLast(dr->expand_assts);
    sig2 = KheDrsAsstToTaskClassSignature(asst2);
    while( sig2->cost > sig->cost )
    {
      HaArrayDeleteLast(dr->expand_assts);
      if( asst2 == dr->expand_free_day_asst )
        dr->expand_free_day_asst = NULL;
      KheDrsAsstToTaskClassFree(asst2, drs);
      asst2 = HaArrayLast(dr->expand_assts);
      sig2 = KheDrsAsstToTaskClassSignature(asst2);
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceAdjustSignatureCosts(KHE_DRS_RESOURCE dr,             */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Adjust the signature costs in the resource assignments of dr, by         */
/*  finding the minimum cost (that must be incurred whatever dr is           */
/*  assigned to) and moving it out of the resource assignments and           */
/*  into drs->expand_cost.                                                   */
/*                                                                           */
/*****************************************************************************/
static void KheDrsExpanderAddCost(KHE_DRS_EXPANDER de, KHE_COST cost);

static void KheDrsResourceAdjustSignatureCosts(KHE_DRS_RESOURCE dr,
  KHE_DRS_EXPANDER de)
{
  KHE_COST movable_cost;  int i;  KHE_DRS_ASST_TO_SHIFT sasst;
  KHE_DRS_ASST_TO_TASK_CLASS asst;
  if( HaArrayCount(dr->expand_assts) > 0 )
  {
    asst = HaArrayFirst(dr->expand_assts);
    movable_cost = asst->asst_to_shift->sig->cost;
    KheDrsExpanderAddCost(de, movable_cost);
    HaArrayForEach(dr->expand_assts_to_shifts, sasst, i)
      sasst->sig->cost -= movable_cost;
  }
}

/* ***
static void KheDrsResourceAdjustSignatureCosts(KHE_DRS_RESOURCE dr,
  KHE_DRS_EXPANDER de)
{
  KHE_COST movable_cost;  int i;  KHE_DRS_ASST_TO_SHIFT sasst;

  ** find the minimum cost over all dr's signatures **
  movable_cost = KheCost(INT_MAX, INT_MAX);
  HaArrayForEach(dr->expand_assts_to_shifts, sasst, i)
    if( sasst->sig.cost < movable_cost )
      movable_cost = sasst->sig.cost;

  ** if there is a movable cost, then move it to the expander **
  if( 0 < movable_cost && movable_cost < KheCost(INT_MAX, INT_MAX) )
  {
    KheDrsExpanderAddCost(de, movable_cost);
    HaArrayForEach(dr->expand_assts_to_shifts, sasst, i)
      sasst->sig.cost -= movable_cost;
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsResourceOnDayIsFixed(KHE_DRS_RESOURCE_ON_DAY drd,             */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs,                    */
/*    KHE_DRS_TASK_ON_DAY *dtd)                                              */
/*                                                                           */
/*  If the assignment of drd is fixed, return true and set *dtd to the task  */
/*  on day that it is fixed to.                                              */
/*                                                                           */
/*  There are three reasons why drd may be fixed:                            */
/*                                                                           */
/*    * this run is a rerun so we take the assignment from a packed soln;    */
/*                                                                           */
/*    * drd may have a closed assignment;                                    */
/*                                                                           */
/*    * drd's resource may have been assigned (in soln) to some task on the  */
/*      previous day, and that task may still be running.                    */
/*                                                                           */
/*  We can find out whether drd's resource has been assigned to some task    */
/*  on the previous day by consulting the edge leading into soln.            */
/*                                                                           */
/*****************************************************************************/
static KHE_DRS_TASK_ON_DAY KheDrsPackedSolnTaskOnDay(
  KHE_DRS_PACKED_SOLN packed_soln, KHE_DRS_DAY open_day,
  KHE_DRS_RESOURCE open_resource);
static bool KheDrsSolnResourceIsAssigned(KHE_DRS_SOLN soln,
  KHE_DRS_RESOURCE dr, KHE_DRS_TASK_ON_DAY *dtd);
static bool KheDrsTaskBusyOnDay(KHE_DRS_TASK dt, KHE_DRS_DAY day,
  KHE_DRS_TASK_ON_DAY *dtd);

static bool KheDrsResourceOnDayIsFixed(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs,
  KHE_DRS_TASK_ON_DAY *dtd)
{
  KHE_DRS_TASK_ON_DAY dtd1, dtd2;

  /* if this is a rerun, that fixes the assignment */
  if( RERUN && drs->rerun != NULL )
    return *dtd = KheDrsPackedSolnTaskOnDay(drs->rerun, drd->day, drd->encl_dr),
      true;

  /* if drd has a closed assignment, it's fixed to that */
  if( drd->closed_asst != NULL )
    return *dtd = drd->closed_asst, true;

  /* if drd's resource is assigned to a task in soln which is still running, */
  /* then drd is fixed to that */
  if( KheDrsSolnResourceIsAssigned(soln, drd->encl_dr, &dtd1) &&
	KheDrsTaskBusyOnDay(dtd1->encl_dt, drd->day, &dtd2) )
    return *dtd = dtd2, true;

  /* otherwise drd has no fixed assignment */
  return *dtd = NULL, false;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceDominanceTestCacheDebug(KHE_DRS_RESOURCE dr,          */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of dr's dominance test cache.                                */
/*                                                                           */
/*****************************************************************************/
static void KheDrsDim2TableDebug(KHE_DRS_DIM2_TABLE d2, int verbosity,
  int indent, FILE *fp);

static void KheDrsResourceDominanceTestCacheDebug(KHE_DRS_RESOURCE dr,
  int verbosity, int indent, FILE *fp)
{
  fprintf(fp, "%*s[ Dominance test cache for %s:\n", indent, "",
    KheDrsResourceId(dr));
  KheDrsDim2TableDebug(dr->expand_dom_test_cache, verbosity, indent + 2, fp);
  fprintf(fp, "%*s]\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceBuildDominanceTestCache(KHE_DRS_RESOURCE dr,          */
/*    KHE_DRS_RESOURCE_ON_DAY drd, KHE_DYNAMIC_RESOURCE_SOLVER drs)          */
/*                                                                           */
/*  Build dr's dominance test cache, assuming that its asst to shifts        */
/*  are built.                                                               */
/*                                                                           */
/*****************************************************************************/
/* ***
static bool KheDrsSignerDominatesStatesOnlyAll(KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2, int trie_start_depth,
  bool stop_on_neg, KHE_COST *avail_cost, int verbosity, int indent, FILE *fp);
*** */
static bool KheDrsSignerDoDominates(KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2,
  KHE_DRS_SIGNER_TEST test, int trie_start_depth, bool stop_on_neg,
  KHE_COST *avail_cost, int verbosity, int indent, FILE *fp);
static KHE_DRS_COST_TUPLE KheDrsCostTupleMake(KHE_COST psi, KHE_COST psi0,
  bool psi_plus_defined, KHE_COST psi_plus);
static void KheDrsDim2TablePut(KHE_DRS_DIM2_TABLE d2, int index2,
  int index1, KHE_DRS_COST_TUPLE ct, KHE_DYNAMIC_RESOURCE_SOLVER drs);
static void KheDrsSignatureAddAsstToShiftIndex(KHE_DRS_SIGNATURE sig, int val);

static void KheDrsResourceBuildDominanceTestCache(KHE_DRS_RESOURCE dr,
  KHE_DRS_RESOURCE_ON_DAY drd, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int i, j;  KHE_DRS_ASST_TO_SHIFT sasst1, sasst2;  KHE_COST cost;
  KHE_DRS_COST_TUPLE ct;
  if( USE_DOM_CACHING )
  {
    HaArrayForEach(dr->expand_assts_to_shifts, sasst1, i)
    {
      KheDrsSignatureAddAsstToShiftIndex(sasst1->sig, i);
      HaArrayForEach(dr->expand_assts_to_shifts, sasst2, j)
      {
	cost = 0;
	KheDrsSignerDoDominates(drd->signer, sasst1->sig, sasst2->sig,
	  KHE_DRS_SIGNER_ALL, 0, false, &cost, 0, 0, NULL);
	/* ***
	KheDrsSignerDominatesStatesOnlyAll(drd->signer, sasst1->sig,
	  sasst2->sig, 0, false, &cost, 0, 0, NULL);
	*** */
	ct = KheDrsCostTupleMake(cost, 0, false, 0);
	KheDrsDim2TablePut(dr->expand_dom_test_cache, i, j, ct, drs);
      }
    }
    if( DEBUG67 )
      KheDrsResourceDominanceTestCacheDebug(dr, 2, 2, stderr);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceExpandBegin(KHE_DRS_RESOURCE dr,                      */
/*    KHE_DRS_SOLN prev_soln, KHE_DRS_DAY next_day,                          */
/*    KHE_DRS_RESOURCE_SET free_resources,                                   */
/*    KHE_DRS_ASST_TO_TASK_SET fixed_assts, KHE_DRS_EXPANDER de)             */
/*                                                                           */
/*  Carry out the first part of beginning an expansion using open resource   */
/*  dr, which is to work out whether dr is involved in a fixed assignment,   */
/*  and then either add that assignment to fixed_assts (if it is) or add     */
/*  the resource to free_resources (if it isn't).                            */
/*                                                                           */
/*****************************************************************************/
static void KheDrsExpanderDeleteFreeResource(KHE_DRS_EXPANDER de);
static KHE_DRS_ASST_TO_SHIFT KheDrsAsstToShiftMake(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_SIGNATURE prev_sig,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);
static KHE_DRS_ASST_TO_TASK KheDrsAsstToTaskMake(
  KHE_DRS_ASST_TO_TASK_CLASS asst, KHE_DRS_TASK_ON_DAY fixed_dtd);
static void KheDrsAsstToTaskSetAddLast(KHE_DRS_ASST_TO_TASK_SET dats,
  KHE_DRS_ASST_TO_TASK dat);
static void KheDrsResourceSetAddLast(KHE_DRS_RESOURCE_SET rs,
  KHE_DRS_RESOURCE dr);
static char *KheDrsTaskExpandRoleShow(KHE_DRS_TASK_EXPAND_ROLE dter);

static void KheDrsResourceExpandBegin(KHE_DRS_RESOURCE dr,
  KHE_DRS_SOLN prev_soln, KHE_DRS_DAY next_day,
  KHE_DRS_RESOURCE_SET free_resources,
  KHE_DRS_ASST_TO_TASK_SET fixed_assts, KHE_DRS_EXPANDER de)
{
  KHE_DRS_TASK_ON_DAY fixed_dtd;
  KHE_DRS_RESOURCE_ON_DAY drd;  KHE_DRS_ASST_TO_SHIFT sasst;
  KHE_DRS_ASST_TO_TASK tasst;  KHE_DRS_ASST_TO_TASK_CLASS asst;
  KHE_DRS_TASK dt;  KHE_DRS_SIGNATURE prev_sig;

  /* there should be no old assts to shifts or assignments to classes */
  HnAssert(HaArrayCount(dr->expand_assts_to_shifts) == 0,
    "KheDrsResourceExpandBegin internal error 1");
  HnAssert(HaArrayCount(dr->expand_assts) == 0,
    "KheDrsResourceExpandBegin internal error 2");
  HnAssert(dr->expand_free_day_asst == NULL,
    "KheDrsResourceExpandBegin internal error 3");

  /* add assignments to shifts and assignments to classes */
  drd = KheDrsResourceOnDay(dr, next_day);
  if( KheDrsResourceOnDayIsFixed(drd, prev_soln, de->solver, &fixed_dtd) )
  {
    /* mark the resource as fixed, and inform the expander */
    dr->expand_role = KHE_DRS_RESOURCE_EXPAND_FIXED;
    KheDrsExpanderDeleteFreeResource(de);

    /* fixed assignment to fixed_dtd; make and add sasst and asst */
    prev_sig = HaArray(prev_soln->sig_set.signatures, dr->open_resource_index);
    sasst = KheDrsAsstToShiftMake(drd, fixed_dtd, prev_sig, de->solver);
    asst = KheDrsResourceAddAsst(dr, sasst, NULL, fixed_dtd, true, de);
    KheDrsResourceAsstToShiftAddOrFree(dr, sasst, de->solver);

    /* mark the task as fixed */
    if( fixed_dtd != NULL )
    {
      dt = fixed_dtd->encl_dt;
      HnAssert(dt->expand_role == KHE_DRS_TASK_EXPAND_NO,
	"KheDrsResourceExpandBegin internal error 4 (role %s)",
	KheDrsTaskExpandRoleShow(dt->expand_role));
      dt->expand_role = KHE_DRS_TASK_EXPAND_FIXED;
    }

    /* make tasst and add to expander and fixed_assts */
    tasst = KheDrsAsstToTaskMake(asst, fixed_dtd);
    KheDrsAsstToTaskSetAddLast(fixed_assts, tasst);

    /* and build dominance cache */
    KheDrsResourceBuildDominanceTestCache(dr, drd, de->solver);
  }
  else
  {
    /* mark the resource as free */
    dr->expand_role = KHE_DRS_RESOURCE_EXPAND_FREE;

    /* add dr to free_resources */
    KheDrsResourceSetAddLast(free_resources, dr);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceExpandBeginFree(KHE_DRS_RESOURCE dr,                  */
/*    KHE_DRS_SOLN prev_soln, KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de)     */
/*                                                                           */
/*  Carry out the second part of beginning an expansion using open resource  */
/*  dr, which is now known to be free.  So add its assignments in.           */
/*                                                                           */
/*  Implementation note.  This code is separated from the previous function  */
/*  because there are cases where, by adding fixed assignments to the        */
/*  expander, the first function might allow more pruning in the second.     */
/*                                                                           */
/*****************************************************************************/
static bool KheDrsTaskClassAcceptResourceBegin(KHE_DRS_TASK_CLASS dtc,
  KHE_DRS_RESOURCE_ON_DAY drd, KHE_DRS_TASK_ON_DAY *dtd);
static void KheDrsTaskClassAcceptResourceEnd(KHE_DRS_TASK_CLASS dtc,
  KHE_DRS_TASK_ON_DAY dtd);

static void KheDrsResourceExpandBeginFree(KHE_DRS_RESOURCE dr,
  KHE_DRS_SOLN prev_soln, KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de)
{
  int i, j;  KHE_DRS_RESOURCE_ON_DAY drd;  KHE_DRS_TASK_ON_DAY dtd;
  KHE_DRS_TASK_CLASS dtc;  KHE_DRS_SHIFT ds;  KHE_DRS_ASST_TO_SHIFT sasst;
  KHE_DRS_SIGNATURE prev_sig;

  /* there should be no old assts to shifts or assignments to classes */
  HnAssert(dr->expand_role == KHE_DRS_RESOURCE_EXPAND_FREE,
    "KheDrsResourceExpandBeginFree internal error");

  /* unfixed assignment; make sassts and assts as required */
  drd = KheDrsResourceOnDay(dr, next_day);
  prev_sig = HaArray(prev_soln->sig_set.signatures, dr->open_resource_index);
  HaArrayForEach(next_day->shifts, ds, i)
  {
    sasst = NULL;
    HaArrayForEach(ds->task_classes, dtc, j)
      if( KheDrsTaskClassAcceptResourceBegin(dtc, drd, &dtd) )
      {
	if( sasst == NULL )
	  sasst = KheDrsAsstToShiftMake(drd, dtd, prev_sig, de->solver);
	KheDrsResourceAddAsst(dr, sasst, dtc, NULL, false, de);
	KheDrsTaskClassAcceptResourceEnd(dtc, dtd);
      }
    if( sasst != NULL )
      KheDrsResourceAsstToShiftAddOrFree(dr, sasst, de->solver);
  }

  /* one sasst and asst for a free day */
  sasst = KheDrsAsstToShiftMake(drd, NULL, prev_sig, de->solver);
  dr->expand_free_day_asst = KheDrsResourceAddAsst(dr, sasst, NULL, NULL,
    false, de);
  KheDrsResourceAsstToShiftAddOrFree(dr, sasst, de->solver);

  /* sort expand_assts by increasing cost and keep only the best */
  KheDrsSortAndReduceAssignments(dr, de->solver);

  /* move any unavoidable cost into the expander */
  KheDrsResourceAdjustSignatureCosts(dr, de);

  /* add a dominance test cache (still to do) */
  KheDrsResourceBuildDominanceTestCache(dr, drd, de->solver);
  /* dr->expand_dom_test_cache is clear here, need to build it now */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceExpandBegin(KHE_DRS_RESOURCE dr,                      */
/*    KHE_DRS_SOLN prev_soln, KHE_DRS_DAY day,                               */
/*    KHE_DRS_RESOURCE_SET free_resources, KHE_DRS_EXPANDER de)              */
/*                                                                           */
/*  Begin dr's part of the expansion of one solution, including adding       */
/*  dr to free_resources if it is not subject to a fixed assignment, and     */
/*  adding that fixed assignment to the expander if it is.                   */
/*                                                                           */
/*****************************************************************************/
/* ***
static void KheDrsAsstToTaskClassDebug(KHE_DRS_ASST_TO_TASK_CLASS asst,
  int verbosity, int indent, FILE *fp);
static bool KheDrsExprInSignature(KHE_DRS_EXPR e, KHE_DRS_DAY day, int *si);
static void KheDrsExprDoDebug(KHE_DRS_EXPR e, KHE_DRS_SOLN soln,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp);
*** */

/* *** old version that does it all at once
static void KheDrsResourceExpandBegin(KHE_DRS_RESOURCE dr,
  KHE_DRS_SOLN prev_soln, KHE_DRS_DAY next_day,
  KHE_DRS_RESOURCE_SET free_resources,
  KHE_DRS_ASST_TO_TASK_SET fixed_assts, KHE_DRS_EXPANDER de)
{
  KHE_DRS_TASK_ON_DAY dtd, fixed_dtd;  KHE_DRS_TASK_CLASS dtc;
  KHE_DRS_SHIFT ds;  KHE_DRS_RESOURCE_ON_DAY drd;  KHE_DRS_ASST_TO_SHIFT sasst;
  int i, j;  KHE_DRS_ASST_TO_TASK tasst;  KHE_DRS_ASST_TO_TASK_CLASS asst;
  KHE_DRS_TASK dt;

  ** there should be no old assts to shifts or assignments to classes **
  HnAssert(HaArrayCount(dr->expand_assts_to_shifts) == 0,
    "KheDrsResourceExpandBegin internal error 1");
  HnAssert(HaArrayCount(dr->expand_assts) == 0,
    "KheDrsResourceExpandBegin internal error 3");
  HnAssert(dr->expand_free_day_asst == NULL,
    "KheDrsResourceExpandBegin internal error 4");

  ** add assignments to shifts and assignments to classes **
  drd = KheDrsResourceOnDay(dr, next_day);
  if( KheDrsResourceOnDayIsFixed(drd, prev_soln, de->solver, &fixed_dtd) )
  {
    ** mark the resource as fixed, and inform the expander **
    dr->expand_role = KHE_DRS_RESOURCE_EXPAND_FIXED;
    KheDrsExpanderDeleteFreeResource(de);

    ** fixed assignment to fixed_dtd; make and add sasst and asst **
    sasst = KheDrsAsstToShiftMake(drd, fixed_dtd, prev_soln, de->solver);
    asst = KheDrsResourceAddAsst(dr, sasst, NULL, fixed_dtd, true, de);
    KheDrsResourceAsstToShiftAddOrFree(dr, sasst, de->solver);

    ** mark the task as fixed **
    dt = fixed_dtd->encl_dt;
    HnAssert(dt->expand_role == KHE_DRS_TASK_EXPAND_NO,
      "KheDrsResourceExpandBegin internal error 5");
    dt->expand_role = KHE_DRS_TASK_EXPAND_FIXED;

    ** make tasst and add to expander and fixed_assts **
    tasst = KheDrsAsstToTaskMake(asst, fixed_dtd);
    KheDrsAsstToTaskSetAddLast(fixed_assts, tasst);
    ** KheDrsExpanderAddAsstToTask(de, tasst); **
  }
  else
  {
    ** mark the resource as free **
    dr->expand_role = KHE_DRS_RESOURCE_EXPAND_FREE;

    ** unfixed assignment; make sassts and assts as required **
    HaArrayForEach(next_day->shifts, ds, i)
    {
      sasst = NULL;
      HaArrayForEach(ds->task_classes, dtc, j)
	if( KheDrsTaskClassAcceptResourceBegin(dtc, drd, &dtd) )
	{
	  if( sasst == NULL )
	    sasst = KheDrsAsstToShiftMake(drd, dtd, prev_soln, de->solver);
	  KheDrsResourceAddAsst(dr, sasst, dtc, NULL, false, de);
	  KheDrsTaskClassAcceptResourceEnd(dtc);
	}
      if( sasst != NULL )
	KheDrsResourceAsstToShiftAddOrFree(dr, sasst, de->solver);
    }

    ** one sasst and asst for a free day **
    sasst = KheDrsAsstToShiftMake(drd, NULL, prev_soln, de->solver);
    dr->expand_free_day_asst = KheDrsResourceAddAsst(dr, sasst, NULL, NULL,
      false, de);
    KheDrsResourceAsstToShiftAddOrFree(dr, sasst, de->solver);

    ** add dr to free_resources **
    KheDrsResourceSetAddLast(free_resources, dr);

    ** sort expand_assts by increasing cost and keep only the best **
    KheDrsSortAndReduceAssignments(dr, de->solver);

    ** move any unavoidable cost into the expander **
    KheDrsResourceAdjustSignatureCosts(dr, de);
  }

  if( DEBUG23 )
  {
    KHE_DRS_EXPR e;  int si;  KHE_DRS_ASST_TO_TASK_CLASS asst;
    fprintf(stderr, "  [ KheDrsResourceExpandBegin(%s, -, %s, drs)\n",
      KheDrsResourceId(dr), KheDrsDayId(next_day));
    HaArrayForEach(drd->signer->internal_exprs, e, i)
      if( KheDrsExprInSignature(e, next_day, &si) )
      {
	fprintf(stderr, "    %d: ", si);
	KheDrsExprDoDebug(e, NULL, de->solver, stderr);
	fprintf(stderr, "\n");
      }
    HaArrayForEach(dr->expand_assts, asst, i)
      KheDrsAsstToTaskClassDebug(asst, 1, 4, stderr);
    fprintf(stderr, "  ] KheDrsResourceExpandBegin\n");
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceExpandEnd(KHE_DRS_RESOURCE dr, KHE_DRS_EXPANDER de)   */
/*                                                                           */
/*  End dr's part of the expansion of one solution.                          */
/*                                                                           */
/*****************************************************************************/
/* ***
static void KheDrsSignatureFree(KHE_DRS_SIGNATURE sig,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);
*** */
static void KheDrsAsstToShiftFree(KHE_DRS_ASST_TO_SHIFT sasst,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);
static void KheDrsDim2TableClear(KHE_DRS_DIM2_TABLE d2,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);
static void KheDrsSignatureClearAsstToShiftIndex(KHE_DRS_SIGNATURE sig);

static void KheDrsResourceExpandEnd(KHE_DRS_RESOURCE dr, KHE_DRS_EXPANDER de)
{
  int i;  KHE_DRS_ASST_TO_SHIFT sasst;

  /* mark the resource as not involved in any expansion */
  dr->expand_role = KHE_DRS_RESOURCE_EXPAND_NO;

  /* clear out the dominance test cache */
  if( USE_DOM_CACHING )
    KheDrsDim2TableClear(dr->expand_dom_test_cache, de->solver);

  /* clear assignments to shifts */
  HaArrayForEach(dr->expand_assts_to_shifts, sasst, i)
  {
    /* if( sasst != NULL ) */
    if( USE_DOM_CACHING )
      KheDrsSignatureClearAsstToShiftIndex(sasst->sig);
    KheDrsAsstToShiftFree(sasst, de->solver);
  }
  HaArrayClear(dr->expand_assts_to_shifts);
  /* ***
  if( dr->expand_free_asst_to_shift != NULL )
  {
    KheDrsAsstToShiftFree(dr->expand_free_asst_to_shift, drs);
    dr->expand_free_asst_to_shift = NULL;
  }
  *** */

  /* clear assignments to task classes */
  HaArrayAppend(de->solver->asst_to_task_class_free_list, dr->expand_assts, i);
  HaArrayClear(dr->expand_assts);
  dr->expand_free_day_asst = NULL;

  /* ***
  if( dr->expand_fixed_signature != NULL )
  {
    HaArrayAddLast(drs->asst_to_task_class_free_list, dr->expand_fixed_signature);
    dr->expand_fixed_signature = NULL;
  }
  *** */
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsAsstMatches(KHE_DRS_ASST_TO_TASK_CLASS asst,                  */
/*    KHE_DRS_TASK_ON_DAY dtd)                                               */
/*                                                                           */
/*  Return true if asst and dtd match.                                       */
/*                                                                           */
/*****************************************************************************/

/* *** no longer needed
static bool KheDrsAsstMatches(KHE_DRS_ASST_TO_TASK_CLASS asst,
  KHE_DRS_TASK_ON_DAY dtd)
{
  if( asst->task_class != NULL )
    return dtd != NULL && dtd->encl_dt->encl_dtc == asst->task_class;
  else
    return asst->fixed_task_on_day == dtd;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsResourceExpandFindResourceAsst(KHE_DRS_RESOURCE dr,           */
/*    KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_ASST_TO_TASK *tasst)                  */
/*                                                                           */
/*  If there is a resource assignment object in dr suited to dtd, package    */
/*  it and dtd into *tasst and return true.  Otherwise return false.         */
/*                                                                           */
/*****************************************************************************/

/* *** no longer needed
static KHE_DRS_ASST_TO_TASK KheDrsAsstToTaskMake(
  KHE_DRS_ASST_TO_TASK_CLASS asst, KHE_DRS_TASK_ON_DAY fixed_dtd);

static bool KheDrsResourceExpandFindResourceAsst(KHE_DRS_RESOURCE dr,
  KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_ASST_TO_TASK *tasst)
{
  KHE_DRS_ASST_TO_TASK_CLASS asst2;  int i;
  HaArrayForEach(dr->expand_assts, asst2, i)
    if( KheDrsAsstMatches(asst2, dtd) )
      return *tasst = KheDrsAsstToTaskMake(asst2, dtd), true;
  return *tasst = KheDrsAsstToTaskMake(NULL, NULL), false;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_RESOURCE_SET"                                         */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_RESOURCE_SET KheDrsResourceSetMake(                              */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make or get a new, empty resource set.                                   */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_RESOURCE_SET KheDrsResourceSetMake(
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_RESOURCE_SET res;
  if( HaArrayCount(drs->resource_set_free_list) > 0 )
  {
    res = HaArrayLastAndDelete(drs->resource_set_free_list);
    HaArrayClear(res->resources);
  }
  else
  {
    HaMake(res, drs->arena);
    HaArrayInit(res->resources, drs->arena);
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceSetFree(KHE_DRS_RESOURCE_SET rs,                      */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Free rs.                                                                 */
/*                                                                           */
/*****************************************************************************/

static void KheDrsResourceSetFree(KHE_DRS_RESOURCE_SET rs,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  HaArrayAddLast(drs->resource_set_free_list, rs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceSetClear(KHE_DRS_RESOURCE_SET rs)                     */
/*                                                                           */
/*  Clear rs.                                                                */
/*                                                                           */
/*****************************************************************************/

static void KheDrsResourceSetClear(KHE_DRS_RESOURCE_SET rs)
{
  HaArrayClear(rs->resources);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsResourceSetCount(KHE_DRS_RESOURCE_SET rs)                      */
/*                                                                           */
/*  Return the number of resources in rs.                                    */
/*                                                                           */
/*****************************************************************************/

static int KheDrsResourceSetCount(KHE_DRS_RESOURCE_SET rs)
{
  return HaArrayCount(rs->resources);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_RESOURCE KheDrsResourceSetResource(KHE_DRS_RESOURCE_SET rs,      */
/*    int i)                                                                 */
/*                                                                           */
/*  Return the i'th resource of rs.                                          */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_RESOURCE KheDrsResourceSetResource(KHE_DRS_RESOURCE_SET rs,
  int i)
{
  return HaArray(rs->resources, i);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceSetAddLast(KHE_DRS_RESOURCE_SET rs,                   */
/*    KHE_DRS_RESOURCE dr)                                                   */
/*                                                                           */
/*  Add dr to the end of rs.                                                 */
/*                                                                           */
/*****************************************************************************/

static void KheDrsResourceSetAddLast(KHE_DRS_RESOURCE_SET rs,
  KHE_DRS_RESOURCE dr)
{
  HaArrayAddLast(rs->resources, dr);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceSetDeleteLast(KHE_DRS_RESOURCE_SET rs)                */
/*                                                                           */
/*  Delete the last element of rs.                                           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsResourceSetDeleteLast(KHE_DRS_RESOURCE_SET rs)
{
  HnAssert(KheDrsResourceSetCount(rs) > 0,
    "KheDrsResourceSetDeleteLast internal error");
  HaArrayDeleteLast(rs->resources);
}


/*****************************************************************************/
/*                                                                           */
/*  KheDrsResourceSetForEach(rs, dr, i)                                      */
/*                                                                           */
/*  Iterate over the resources dr of rs; like HaArrayForEach.                */
/*                                                                           */
/*****************************************************************************/

#define KheDrsResourceSetForEach(rs, dr, i)				\
  HaArrayForEach((rs)->resources, (dr), (i))


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceOnDaySetDebug(KHE_DRS_RESOURCE_ON_DAY_SET drds,       */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug printf of rs onto fp with the given verbosity and indent.          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsResourceSetDebug(KHE_DRS_RESOURCE_SET rs,
  int verbosity, int indent, FILE *fp)
{
  KHE_DRS_RESOURCE dr;  int i;
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  fprintf(fp, "{");
  KheDrsResourceSetForEach(rs, dr, i)
  {
    if( i > 0 )
      fprintf(fp, ", ");
    fprintf(fp, "%s", KheDrsResourceId(dr));
  }
  fprintf(fp, "}");
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_RESOURCE_ON_DAY"                                      */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_RESOURCE_ON_DAY KheDrsResourceOnDayMake(KHE_DRS_RESOURCE dr,     */
/*    KHE_DRS_DAY day, KHE_DYNAMIC_RESOURCE_SOLVER drs)                      */
/*                                                                           */
/*  Make a new resource on day object with these attributes.                 */
/*  This function assumes that dr is not assigned any task on day.           */
/*                                                                           */
/*****************************************************************************/
static KHE_DRS_SIGNER KheDrsSignerMake(KHE_DRS_DAY day,
  KHE_DRS_RESOURCE_ON_DAY drd, KHE_DRS_SHIFT ds,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);

static KHE_DRS_RESOURCE_ON_DAY KheDrsResourceOnDayMake(KHE_DRS_RESOURCE dr,
  KHE_DRS_DAY day, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_RESOURCE_ON_DAY res;
  HaMake(res, drs->arena);
  /* HaArrayInit(res->dom_tests, drs->arena); */
  res->encl_dr = dr;
  res->day = day;
  res->closed_asst = NULL;
  HaArrayInit(res->external_today, drs->arena);
  /* HaArrayInit(res->internal_today, drs->arena); */
  /* HaArrayInit(res->dom_tests, drs->arena); */
  /* HaArrayInit(res->eq_dom_test_indexes, drs->arena); */
  res->signer = KheDrsSignerMake(NULL, res, NULL, drs);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsResourceOnDayIndex(KHE_DRS_RESOURCE_ON_DAY drd)                */
/*                                                                           */
/*  Return the open resource index of drd.  This will be -1 if the resource  */
/*  is not open                                                              */
/*                                                                           */
/*****************************************************************************/

static int KheDrsResourceOnDayIndex(KHE_DRS_RESOURCE_ON_DAY drd)
{
  return drd->encl_dr->open_resource_index;
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheDrsResourceOnDayId(KHE_DRS_RESOURCE_ON_DAY drd)                 */
/*                                                                           */
/*  Return an Id for drd.  The result is held in a static array and is       */
/*  best used for debugging only.                                            */
/*                                                                           */
/*****************************************************************************/

static char *KheDrsResourceOnDayId(KHE_DRS_RESOURCE_ON_DAY drd)
{
  static char buff[200];
  snprintf(buff, 200, "%s:%s", KheDrsResourceId(drd->encl_dr),
    KheDrsDayId(drd->day));
  return buff;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsResourceOnDayHasTimeExpr(KHE_DRS_RESOURCE_ON_DAY drd,         */
/*    KHE_DRS_EXPR_TAG tag, KHE_DRS_DOM_TEST dom_test, KHE_TIME time,        */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, KHE_DRS_EXPR *res_e)                  */
/*                                                                           */
/*  If drd has a BUSY_TIME, FREE_TIME, or WORK_TIME expression with the      */
/*  given tag, dom_test, and time, then return true and set *res_e to it.    */
/*  Otherwise return false with *res_e set to NULL.                          */
/*                                                                           */
/*****************************************************************************/
/* ***
static bool KheDrsDomTestEqual(KHE_DRS_DOM_TEST dt1, KHE_DRS_DOM_TEST dt2);
*** */

static bool KheDrsResourceOnDayHasTimeExpr(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_DRS_EXPR_TAG tag, /* KHE_DRS_DOM_TEST dom_test, */ KHE_TIME time,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, KHE_DRS_EXPR *res_e)
{
  int i;  KHE_DRS_EXPR e;  KHE_DRS_EXPR_BUSY_TIME ebt;
  KHE_DRS_EXPR_FREE_TIME eft;  KHE_DRS_EXPR_WORK_TIME ewt;
  HaArrayForEach(drd->external_today, e, i)
    if( e->tag == tag /* && KheDrsDomTestEqual(e->dom_test, dom_test) */ )
    {
      switch( e->tag )
      {
	case KHE_DRS_EXPR_BUSY_TIME_TAG:

	  ebt = (KHE_DRS_EXPR_BUSY_TIME) e;
	  if( ebt->time == time )
	    return *res_e = e, true;
	  break;

	case KHE_DRS_EXPR_FREE_TIME_TAG:

	  eft = (KHE_DRS_EXPR_FREE_TIME) e;
	  if( eft->time == time )
	    return *res_e = e, true;
	  break;

	case KHE_DRS_EXPR_WORK_TIME_TAG:

	  ewt = (KHE_DRS_EXPR_WORK_TIME) e;
	  if( ewt->time == time )
	    return *res_e = e, true;
	  break;

	default:

	  HnAbort("KheDrsResourceOnDayHasTimeExpr: invalid tag (%d)", e->tag);
      }
    }
  return *res_e = NULL, false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsResourceOnDayHasDayExpr(KHE_DRS_RESOURCE_ON_DAY drd,          */
/*    KHE_DRS_EXPR_TAG tag, KHE_DRS_DOM_TEST dom_test, KHE_TIME_GROUP tg,    */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, KHE_DRS_EXPR *res_e)                  */
/*                                                                           */
/*  If drd has a BUSY_DAY, FREE_DAY, or WORK_DAY expression with the given   */
/*  tag, dom_test, and time group, then return true and set *res_e to it.    */
/*  Otherwise return false with *res_e set to NULL.                          */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsResourceOnDayHasDayExpr(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_DRS_EXPR_TAG tag, /* KHE_DRS_DOM_TEST dom_test, */ KHE_TIME_GROUP tg,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, KHE_DRS_EXPR *res_e)
{
  int i;  KHE_DRS_EXPR e;  KHE_DRS_EXPR_BUSY_DAY ebd;
  KHE_DRS_EXPR_FREE_DAY efd;  KHE_DRS_EXPR_WORK_DAY ewd;
  HaArrayForEach(drd->external_today, e, i)
    if( e->tag == tag /* && KheDrsDomTestEqual(e->dom_test, dom_test) */ )
    {
      switch( e->tag )
      {
	case KHE_DRS_EXPR_BUSY_DAY_TAG:

	  ebd = (KHE_DRS_EXPR_BUSY_DAY) e;
	  if( KheTimeGroupEqual(ebd->time_group, tg) )
	    return *res_e = e, true;
	  break;

	case KHE_DRS_EXPR_FREE_DAY_TAG:

	  efd = (KHE_DRS_EXPR_FREE_DAY) e;
	  if( KheTimeGroupEqual(efd->time_group, tg) )
	    return *res_e = e, true;
	  break;

	case KHE_DRS_EXPR_WORK_DAY_TAG:

	  ewd = (KHE_DRS_EXPR_WORK_DAY) e;
	  if( KheTimeGroupEqual(ewd->time_group, tg) )
	    return *res_e = e, true;
	  break;

	default:

	  HnAbort("KheDrsResourceHasDayExpr: invalid tag (%d)", e->tag);
      }
    }
  return *res_e = NULL, false;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceOnDayOpen(KHE_DRS_RESOURCE_ON_DAY drd,                */
/*    int open_resource_index, int open_day_index, KHE_DRS_DAY_RANGE ddr,    */
/*    KHE_DRS_PACKED_SOLN init_soln, KHE_DYNAMIC_RESOURCE_SOLVER drs)        */
/*                                                                           */
/*  Open drd.                                                                */
/*                                                                           */
/*****************************************************************************/
static void KheDrsTaskUnAssign(KHE_DRS_TASK dt, bool task);
static void KheDrsExprGatherForOpening(KHE_DRS_EXPR e,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);
static void KheDrsPackedSolnSetTaskOnDay(KHE_DRS_PACKED_SOLN packed_soln,
  int open_day_index, int open_resource_index, KHE_DRS_TASK_ON_DAY dtd);
static void KheDrsSignerMakeCorrelated(KHE_DRS_SIGNER dsg);

static void KheDrsResourceOnDayOpen(KHE_DRS_RESOURCE_ON_DAY drd,
  int open_resource_index, int open_day_index, KHE_DRS_DAY_RANGE ddr,
  KHE_DRS_PACKED_SOLN init_soln, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_TASK_ON_DAY dtd;  KHE_DRS_TASK dt;  KHE_DRS_EXPR e;  int i;
  KHE_DRS_DAY_RANGE open_day_range;

  /* unassign affected tasks and possibly add the assignments to init_soln */
  dtd = drd->closed_asst;
  if( dtd != NULL )
  {
    dt = dtd->encl_dt;
    if( KheDrsDayRangeSubset(dt->encl_dtc->day_range, ddr) )
    {
      KheDrsTaskUnAssign(dt, true);
      if( init_soln != NULL )
	KheDrsPackedSolnSetTaskOnDay(init_soln, open_day_index,
	  open_resource_index, dtd);
    }
  }

  /* set make_correlated field of signer */
  if( drs->solve_correlated_exprs )
    KheDrsSignerMakeCorrelated(drd->signer);

  /* gather for opening expressions depending on drd (unassigned or not) */
  open_day_range = KheDrsDayRangeMake(open_day_index, open_day_index);
  HaArrayForEach(drd->external_today, e, i)
  {
    e->open_children_by_day.range = open_day_range;
    KheDrsExprGatherForOpening(e, drs);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceOnDayClose(KHE_DRS_RESOURCE_ON_DAY drd,               */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Close drd.                                                               */
/*                                                                           */
/*****************************************************************************/
static void KheDrsSignerClear(KHE_DRS_SIGNER dsg,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsResourceOnDayClose(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KheDrsSignerClear(drd->signer, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceOnDayAddExpr(KHE_DRS_RESOURCE_ON_DAY drd,             */
/*    KHE_DRS_EXPR e)                                                        */
/*                                                                           */
/*  Add e, known to be new, to drd.                                          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsResourceOnDayAddExpr(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_DRS_EXPR e)
{
  HaArrayAddLast(drd->external_today, e);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceOnDayAddOpenExpr(KHE_DRS_RESOURCE_ON_DAY drd,         */
/*    KHE_DRS_EXPR e)                                                        */
/*                                                                           */
/*  Add internal expression e to drd.                                        */
/*                                                                           */
/*****************************************************************************/

static void KheDrsResourceOnDayAddOpenExpr(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_DRS_EXPR e)
{
  KheDrsSignerAddOpenExpr(drd->signer, e);
  /* HaArrayAddLast(drd->internal_today, e); */
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsResourceOnDayAddDomTest(KHE_DRS_RESOURCE_ON_DAY drd,           */
/*    KHE_DRS_DOM_TEST dom_test)                                             */
/*                                                                           */
/*  Add dom_test to drd and return its sig index.                            */
/*                                                                           */
/*****************************************************************************/

static int KheDrsResourceOnDayAddDomTest(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_DRS_DOM_TEST dom_test, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  return KheDrsSignerAddDomTest(drd->signer, dom_test, drs);
}

/* ***
static int KheDrsResourceOnDayAddDomTest(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_DRS_DOM_TEST dom_test)
{
  int res;

  ** add dom_test to drd **
  res = HaArrayCount(drd->dom_tests);
  HnAssert(dom_test.type != KHE_DRS_DOM_UNUSED,
    "KheDrsDayAddDomTest internal error");
  HaArrayAddLast(drd->dom_tests, dom_test);

  ** if dom_test is equality, record it in eq_dom_test_indexes **
  if( dom_test.type == KHE_DRS_DOM_EQ || dom_test.type == KHE_DRS_DOM_EQ_LOOSE )
    HaArrayAddLast(drd->eq_dom_test_indexes, res);

  ** return dom_test's index **
  return res;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SIGNATURE KheDrsResourceOnDaySignature(                          */
/*    KHE_DRS_RESOURCE_ON_DAY drd, KHE_DRS_TASK_ON_DAY dtd)                  */
/*                                                                           */
/*  Return the resource signature of drd's resource in the shift of dtd.     */
/*                                                                           */
/*****************************************************************************/

/* *** unused, and expand_assts_to_shifts has changed so it can't be used now
static KHE_DRS_SIGNATURE KheDrsResourceOnDaySignature(
  KHE_DRS_RESOURCE_ON_DAY drd, KHE_DRS_TASK_ON_DAY dtd)
{
  KHE_DRS_SHIFT ds;  KHE_DRS_RESOURCE dr;  KHE_DRS_ASST_TO_SHIFT sasst;
  dr = drd->encl_dr;
  if( dtd != NULL )
  {
    ds = KheDrsTaskOnDayShift(dtd);
    sasst = HaArray(dr->expand_assts_to_shifts, ds->index_in_day);
  }
  else
    sasst = dr->expand_free_asst_to_shift;
  return &sasst->sig;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceOnDayLeafSet(KHE_DRS_RESOURCE_ON_DAY drd,             */
/*    KHE_DRS_TASK_ON_DAY dtd, KHE_DYNAMIC_RESOURCE_SOLVER drs)              */
/*                                                                           */
/*  Call KheDrsExprLeafSet for each external expression affected by drd.     */
/*                                                                           */
/*****************************************************************************/
static void KheDrsExprLeafSet(KHE_DRS_EXPR e, KHE_DRS_TASK_ON_DAY dtd,
  KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsResourceOnDayLeafSet(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_DRS_TASK_ON_DAY dtd, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR e;  int i;
  if( dtd != NULL )
    HaArrayForEach(drd->external_today, e, i)
      KheDrsExprLeafSet(e, dtd, drd->encl_dr, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceOnDayLeafClear(KHE_DRS_RESOURCE_ON_DAY drd,           */
/*    KHE_DRS_TASK_ON_DAY dtd, KHE_DYNAMIC_RESOURCE_SOLVER drs)              */
/*                                                                           */
/*  Call KheDrsExprLeafClear for each external expression affected by drd.   */
/*                                                                           */
/*****************************************************************************/
static void KheDrsExprLeafClear(KHE_DRS_EXPR e,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsResourceOnDayLeafClear(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_DRS_TASK_ON_DAY dtd, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR e;  int i;
  if( dtd != NULL )
    HaArrayForEach(drd->external_today, e, i)
      KheDrsExprLeafClear(e, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceOnDayDebug(KHE_DRS_RESOURCE_ON_DAY drd,               */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of drd onto fp with the given verbosity and indent.          */
/*                                                                           */
/*****************************************************************************/
static void KheDrsTaskOnDayDebug(KHE_DRS_TASK_ON_DAY dtd,
  int verbosity, int indent, FILE *fp);
/* static char *KheDrsDayId(KHE_DRS_DAY day); */

static void KheDrsResourceOnDayDebug(KHE_DRS_RESOURCE_ON_DAY drd,
  int verbosity, int indent, FILE *fp)
{
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  fprintf(fp, "%s: ", KheDrsDayId(drd->day));
  if( drd->closed_asst != NULL )
    KheDrsTaskOnDayDebug(drd->closed_asst, 1, -1, fp);
  else
    fprintf(fp, "-");
  fprintf(fp, " (external_today %d)", HaArrayCount(drd->external_today));
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_RESOURCE_ON_DAY_SET"                                  */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_RESOURCE_ON_DAY_SET KheDrsResourceOnDaySetMake(                  */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make or get a new, empty resource on day set.                            */
/*                                                                           */
/*****************************************************************************/

/* *** not currently used
static KHE_DRS_RESOURCE_ON_DAY_SET KheDrsResourceOnDaySetMake(
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_RESOURCE_ON_DAY_SET res;
  if( HaArrayCount(drs->resource_on_day_set_free_list) > 0 )
  {
    res = HaArrayLastAndDelete(drs->resource_on_day_set_free_list);
    HaArrayClear(res->resource_on_days);
  }
  else
  {
    HaMake(res, drs->arena);
    HaArrayInit(res->resource_on_days, drs->arena);
  }
  return res;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceOnDaySetFree(KHE_DRS_RESOURCE_ON_DAY_SET drds,        */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Free drds.                                                               */
/*                                                                           */
/*****************************************************************************/

/* *** not currently used
static void KheDrsResourceOnDaySetFree(KHE_DRS_RESOURCE_ON_DAY_SET drds,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  HaArrayAddLast(drs->resource_on_day_set_free_list, drds);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceOnDaySetClear(KHE_DRS_RESOURCE_ON_DAY_SET drds)       */
/*                                                                           */
/*  Clear drds back to the empty set.                                        */
/*                                                                           */
/*****************************************************************************/

/* *** not currently used
static void KheDrsResourceOnDaySetClear(KHE_DRS_RESOURCE_ON_DAY_SET drds)
{
  HaArrayClear(drds->resource_on_days);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsResourceOnDaySetCount(KHE_DRS_RESOURCE_ON_DAY_SET drds)        */
/*                                                                           */
/*  Return the number of resource on days in drds.                           */
/*                                                                           */
/*****************************************************************************/

/* *** not currently used
static int KheDrsResourceOnDaySetCount(KHE_DRS_RESOURCE_ON_DAY_SET drds)
{
  return HaArrayCount(drds->resource_on_days);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_RESOURCE_ON_DAY KheDrsResourceOnDaySetResourceOnDay(             */
/*    KHE_DRS_RESOURCE_ON_DAY_SET drds, int i)                               */
/*                                                                           */
/*  Return the i'th resource on day in drds.                                 */
/*                                                                           */
/*****************************************************************************/

/* *** not currently used
static KHE_DRS_RESOURCE_ON_DAY KheDrsResourceOnDaySetResourceOnDay(
  KHE_DRS_RESOURCE_ON_DAY_SET drds, int i)
{
  return HaArray(drds->resource_on_days, i);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceSetAddLast(KHE_DRS_RESOURCE_ON_DAY_SET drds,          */
/*    KHE_DRS_RESOURCE_ON_DAY dr)                                            */
/*                                                                           */
/*  Add drd to the end of drds.                                              */
/*                                                                           */
/*****************************************************************************/

/* *** not currently used
static void KheDrsResourceOnDaySetAddLast(KHE_DRS_RESOURCE_ON_DAY_SET drds,
  KHE_DRS_RESOURCE_ON_DAY drd)
{
  HaArrayAddLast(drds->resource_on_days, drd);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceSetDeleteLast(KHE_DRS_RESOURCE_ON_DAY_SET drds)       */
/*                                                                           */
/*  Delete the last element of drds.                                         */
/*                                                                           */
/*****************************************************************************/

/* *** not currently used
static void KheDrsResourceOnDaySetDeleteLast(KHE_DRS_RESOURCE_ON_DAY_SET drds)
{
  HnAssert(KheDrsResourceOnDaySetCount(drds) > 0,
    "KheDrsResourceOnDaySetDeleteLast internal error");
  HaArrayDeleteLast(drds->resource_on_days);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsResourceOnDaySetDebug(KHE_DRS_RESOURCE_ON_DAY_SET drds,       */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug printf of drds onto fp with the given verbosity and indent.        */
/*                                                                           */
/*****************************************************************************/

/* *** not currently used
static void KheDrsResourceOnDaySetDebug(KHE_DRS_RESOURCE_ON_DAY_SET drds,
  int verbosity, int indent, FILE *fp)
{
  KHE_DRS_RESOURCE_ON_DAY drd;  int i;
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  fprintf(fp, "{");
  HaArrayForEach(drds->resource_on_days, drd, i)
  {
    if( i > 0 )
      fprintf(fp, ", ");
    fprintf(fp, "%s", KheDrsResourceId(drd->encl_dr));
  }
  fprintf(fp, "}");
  if( indent >= 0 )
    fprintf(fp, "\n");
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  Major category "events"                                                  */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_TASK_EXPAND_ROLE"                                     */
/*                                                                           */
/*****************************************************************************/

static char *KheDrsTaskExpandRoleShow(KHE_DRS_TASK_EXPAND_ROLE dter)
{
  switch( dter )
  {
    case KHE_DRS_TASK_EXPAND_NO:	return "expand_no";
    case KHE_DRS_TASK_EXPAND_FIXED:	return "expand_fixed";
    case KHE_DRS_TASK_EXPAND_MUST:	return "expand_must";
    case KHE_DRS_TASK_EXPAND_FREE:	return "expand_free";
    default:				return "?";
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_TASK"                                                 */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheTaskAddDays(KHE_TASK task, KHE_DRS_TASK dt,                      */
/*    int first_solve_index, int last_solve_index,                           */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Add the days covered by task and its descendants to dt.                  */
/*                                                                           */
/*****************************************************************************/
static KHE_DRS_TASK_ON_DAY KheDrsTaskOnDayMake(KHE_DRS_TASK encl_dt,
  KHE_DRS_DAY day, KHE_TASK task, KHE_TIME time,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheTaskAddDays(KHE_TASK task, KHE_DRS_TASK dt,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_TASK child_task;  int i, durn, day_index;  KHE_DRS_DAY day;
  KHE_MEET meet;  KHE_TIME start_time, time;  KHE_DRS_TASK_ON_DAY dtd;

  /* do it for task */
  meet = KheTaskMeet(task);
  HnAssert(meet != NULL, "KheTaskAddDays internal error (no meet)");
  start_time = KheMeetAsstTime(meet);
  HnAssert(start_time != NULL,  "KheTaskAddDays internal error (no time)");
  durn = KheMeetDuration(meet);
  for( i = 0;  i < durn;  i++ )
  {
    time = KheTimeNeighbour(start_time, i);
    day_index = KheFrameTimeIndex(drs->days_frame, time);
    day = HaArray(drs->all_days, day_index);
    dtd = KheDrsTaskOnDayMake(dt, day, task, time, drs);
    HaArrayAddLast(dt->days, dtd);
  }

  /* do it for task's descendants */
  for( i = 0;  i < KheTaskAssignedToCount(task);  i++ )
  {
    child_task = KheTaskAssignedTo(task, i);
    KheTaskAddDays(child_task, dt, drs);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsTaskOrganizeDays(KHE_DRS_TASK dt)                             */
/*                                                                           */
/*  Assuming that dt->days are sorted by increasing day index, organize      */
/*  them so that there is one per day, by inserting NULL entries where       */
/*  there are gaps.  There can be no cases of two on one day, thanks to an   */
/*  earlier call to KheTaskClassNoOverlap.                                   */
/*                                                                           */
/*****************************************************************************/

/* *** obsolete; there are now no gaps
static void KheDrsTaskOrganizeDays(KHE_DRS_TASK dt)
{
  int init_len, reqd_len, i, final_pos;
  KHE_DRS_TASK_ON_DAY first_dtd, last_dtd, dtd;

  ** quit early if the number of days is fine as is **
  first_dtd = HaArrayFirst(dt->days);
  last_dtd = HaArrayLast(dt->days);
  reqd_len = last_dtd->day->frame_index - first_dtd->day->frame_index + 1;
  init_len = HaArrayCount(dt->days);
  if( reqd_len == init_len )
    return;

  ** add the required number of NULL entries, initially at the array's end **
  HaArrayFill(dt->days, reqd_len, NULL);

  ** move the entries to their final positions, working right to left **
  for( i = init_len - 1;  i >= 0;  i-- )
  {
    dtd = HaArray(dt->days, i);
    final_pos = dtd->day->frame_index - first_dtd->day->frame_index;
    if( final_pos != i )
    {
      HaArrayPut(dt->days, final_pos, dtd);
      HaArrayPut(dt->days, i, NULL);
    }
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsTaskMake(KHE_DRS_TASK_CLASS encl_dtc,                         */
/*    int index_in_encl_dtc, KHE_TASK task, KHE_COST asst_cost,              */
/*    KHE_COST non_asst_cost, KHE_DYNAMIC_RESOURCE_SOLVER drs,               */
/*    KHE_DRS_TASK *dt)                                                      */
/*                                                                           */
/*  Make a new drs task with these attributes, add it to drs->all_root_tasks */
/*  at the same index as it has in the solution, and return it in *dt.       */
/*                                                                           */
/*  Return true if all this has been done successfully.  The only reason     */
/*  why false could be returned is that the call to KheDrsTaskAssign         */
/*  uncovered a case of a resource assigned to two tasks on the same day.    */
/*                                                                           */
/*****************************************************************************/
static int KheDrsTaskOnDayCmp(const void *t1, const void *t2);
static bool KheDrsTaskAssign(KHE_DRS_TASK dt, KHE_DRS_RESOURCE dr, bool task);

static bool KheDrsTaskMake(KHE_DRS_TASK_CLASS encl_dtc,
  int index_in_encl_dtc, KHE_TASK task, KHE_COST asst_cost,
  KHE_COST non_asst_cost, KHE_DYNAMIC_RESOURCE_SOLVER drs,
  KHE_DRS_TASK *dt)
{
  KHE_DRS_TASK res;  int task_index;  KHE_RESOURCE r;

  /* get the basic object and initialize its fields */
  HnAssert(KheTaskIsProperRoot(task), "KheDrsTaskMake internal error");
  HaMake(res, drs->arena);
  res->encl_dtc = encl_dtc;
  res->index_in_encl_dtc = index_in_encl_dtc;
  res->expand_role = KHE_DRS_TASK_EXPAND_NO;
  res->open = false;
  res->task = task;
  res->closed_asst = NULL;
  res->asst_cost = asst_cost;
  res->non_asst_cost = non_asst_cost;
  HaArrayInit(res->days, drs->arena);

  /* add, sort, and organize the task on day objects */
  KheTaskAddDays(task, res, drs);
  HaArraySort(res->days, &KheDrsTaskOnDayCmp);
  /* KheDrsTaskOrganizeDays(res); */

  /* add res to drs->all_root_tasks */
  task_index = KheTaskSolnIndex(task);
  HaArrayFill(drs->all_root_tasks, task_index + 1, NULL);
  HnAssert(HaArray(drs->all_root_tasks, task_index) == NULL,
    "KheDrsTaskMake internal error");
  HaArrayPut(drs->all_root_tasks, task_index, res);

  /* if task is assigned, make that assignment in res */
  r = KheTaskAsstResource(task);
  if( r != NULL )
  {
    if( !KheDrsTaskAssign(res, KheDrsResource(r, drs), false) )
      return *dt = false, NULL;
  }

  /* and also return res */
  return *dt = res, true;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsTaskAssign(KHE_DRS_TASK dt, KHE_DRS_RESOURCE dr, bool task)   */
/*                                                                           */
/*  Assign dr (which must be non-NULL) to dt, by setting the closed_asst     */
/*  fields in dt, dt->days, and dr.  If task is true, also assign dt's KHE   */
/*  task to dr's KHE resource, aborting if that does not succeed.            */
/*                                                                           */
/*  The enclosing task class has a list of unassigned tasks; but do not      */
/*  remove dt from it, because we are being lazy:  the unassigned tasks      */
/*  will be sorted out later, by KheDrsTaskClassOrganizeUnassignedTasks.     */
/*                                                                           */
/*  Return true if the assignment was successful.  The only reason for       */
/*  failure is if the assignment gives dr two tasks on one day.              */
/*                                                                           */
/*****************************************************************************/
static void KheDrsTaskDebug(KHE_DRS_TASK dt, int verbosity,
  int indent, FILE *fp);

static bool KheDrsTaskAssign(KHE_DRS_TASK dt, KHE_DRS_RESOURCE dr, bool task)
{
  KHE_DRS_TASK_ON_DAY dtd;  int i;  KHE_DRS_RESOURCE_ON_DAY drd;
  if( DEBUG15 )
    fprintf(stderr, "[ KheDrsTaskAssign(%s (%d days), %s, %s)\n",
      KheTaskId(dt->task), HaArrayCount(dt->days), KheDrsResourceId(dr),
      bool_show(task));
  HnAssert(dt->closed_asst == NULL, "KheDrsTaskAssign internal error 1");
  HnAssert(dr != NULL, "KheDrsTaskAssign internal error 2");
  if( task && !KheTaskAssignResource(dt->task, dr->resource) )
    HnAbort("KheDrsTaskAssign internal error 3 (cannot assign %s to %s)",
      KheDrsResourceId(dr), KheTaskId(dt->task));
  dt->closed_asst = dr;
  HaArrayForEach(dt->days, dtd, i)
  {
    drd = KheDrsResourceOnDay(dr, dtd->day);
    HnAssert(dtd->closed_asst == NULL, "KheDrsTaskAssign internal error 4");
    if( drd->closed_asst != NULL )
    {
      if( DEBUG15 )
	fprintf(stderr, "] KheDrsTaskAssign returning false\n");
      return false;
    }
    /* ***
    HnAssert(drd->closed_asst == NULL, "KheDrsTaskAssign internal error 5");
    *** */
    dtd->closed_asst = drd;
    drd->closed_asst = dtd;
  }
  if( DEBUG15 )
    fprintf(stderr, "] KheDrsTaskAssign returning true\n");
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsTaskUnAssign(KHE_DRS_TASK dt, bool task)                      */
/*                                                                           */
/*  Unassign dt (which must be assigned some drs resource dr), by clearing   */
/*  the closed_asst fields in dt, dt->days, and dr.  If task is true, also   */
/*  unassign dt's KHE task, or do nothing if the task will not unassign.     */
/*                                                                           */
/*  The enclosing task class has a list of unassigned tasks; add dt to it.   */
/*                                                                           */
/*****************************************************************************/

static void KheDrsTaskUnAssign(KHE_DRS_TASK dt, bool task)
{
  KHE_DRS_TASK_ON_DAY dtd;  int i;
  KHE_DRS_RESOURCE dr;  KHE_DRS_RESOURCE_ON_DAY drd;
  dr = dt->closed_asst;
  HnAssert(dr != NULL, "KheDrsTaskUnAssign internal error 1");
  if( DEBUG15 )
    fprintf(stderr, "[ KheDrsTaskUnAssign(%s, %s, %s)\n", KheTaskId(dt->task),
      KheDrsResourceId(dr), bool_show(task));
  if( task )
  {
    HnAssert(KheTaskAsstResource(dt->task) == dr->resource,
      "KheDrsTaskUnAssign internal error 2");
    if( !KheTaskUnAssignResource(dt->task) )
    {
      if( DEBUG16 )
	fprintf(stderr, "  failed to unassign %s\n", KheTaskId(dt->task));
      return;
    }
    if( DEBUG9 )
      fprintf(stderr, "  after unassigning %s, cost = %.5f\n",
	KheTaskId(dt->task), KheCostShow(KheSolnCost(KheTaskSoln(dt->task))));
  }
  HaArrayForEach(dt->days, dtd, i)
  {
    drd = KheDrsResourceOnDay(dr, dtd->day);
    HnAssert(dtd->closed_asst == drd, "KheDrsTaskUnAssign internal error 3");
    HnAssert(drd->closed_asst == dtd, "KheDrsTaskUnAssign internal error 4");
    dtd->closed_asst = NULL;
    drd->closed_asst = NULL;
  }
  dt->closed_asst = NULL;
  HaArrayAddLast(dt->encl_dtc->unassigned_tasks, dt);
  if( DEBUG15 )
    fprintf(stderr, "] KheDrsTaskUnAssign returning\n");
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsTaskIndexInEnclCmp(const void *t1, const void *t2)             */
/*                                                                           */
/*  Comparison function for sorting an array of drs tasks into increasing    */
/*  order of their index in their enclosing task class.                      */
/*                                                                           */
/*  In addition, it moves any assigned tasks to the end.                     */
/*                                                                           */
/*****************************************************************************/

static int KheDrsTaskIndexInEnclCmp(const void *t1, const void *t2)
{
  KHE_DRS_TASK dt1 = * (KHE_DRS_TASK *) t1;
  KHE_DRS_TASK dt2 = * (KHE_DRS_TASK *) t2;
  if( dt1->closed_asst != NULL )
  {
    if( dt2->closed_asst != NULL )
    {
      /* dt1 is assigned, dt2 is assigned */
      return 0;
    }
    else
    {
      /* dt1 is assigned, dt2 is not assigned */
      return 1;
    }
  }
  else
  {
    if( dt2->closed_asst != NULL )
    {
      /* dt1 is not assigned, dt2 is assigned */
      return -1;
    }
    else
    {
      /* dt1 is not assigned, dt2 is not assigned */
      return dt1->index_in_encl_dtc - dt2->index_in_encl_dtc;
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsTaskOpen(KHE_DRS_TASK dt, KHE_DYNAMIC_RESOURCE_SOLVER drs)    */
/*                                                                           */
/*  Open task for solving.  We'll have checked previously that its days      */
/*  all lie in the current open interval, and it must currently be closed    */
/*  but unassigned; so opening dt basically just gathers for opening all     */
/*  expressions that depend on dt's assignment.                              */
/*                                                                           */
/*****************************************************************************/

static void KheDrsTaskOpen(KHE_DRS_TASK dt, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_TASK_ON_DAY dtd;  KHE_DRS_EXPR e;  int i, j, di;
  KHE_DRS_DAY_RANGE open_day_range;

  /* open dt */
  if( DEBUG19 )
    fprintf(stderr, "[ KheDrsTaskOpen(%s, drs)\n", KheTaskId(dt->task));
  HnAssert(!dt->open, "KheDrsTaskOpen internal error 1");
  HnAssert(dt->closed_asst == NULL, "KheDrsTaskOpen internal error 2");
  dt->open = true;

  /* gather external expressions for opening */
  HaArrayForEach(dt->days, dtd, i)
  {
    di = dtd->day->open_day_index;
    HnAssert(di >= 0, "KheDrsTaskOpen internal error 3");
    open_day_range = KheDrsDayRangeMake(di, di);
    HaArrayForEach(dtd->external_today, e, j)
    {
      e->open_children_by_day.range = open_day_range;
      /* e->open_day_range = open_day_range; */
      KheDrsExprGatherForOpening(e, drs);
    }
  }
  if( DEBUG19 )
    fprintf(stderr, "] KheDrsTaskOpen\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsTaskClose(KHE_DRS_TASK dt, KHE_DRS_RESOURCE dr)               */
/*                                                                           */
/*  If dt is open, close it, with a call to KheDrsTaskAssign if dr != NULL.  */
/*                                                                           */
/*****************************************************************************/

static void KheDrsTaskClose(KHE_DRS_TASK dt, KHE_DRS_RESOURCE dr)
{
  if( dt->open )
  {
    if( DEBUG19 )
      fprintf(stderr, "[ KheDrsTaskClose(%s, drs)\n", KheTaskId(dt->task));
    dt->open = false;
    if( dr != NULL )
    {
      if( !KheDrsTaskAssign(dt, dr, true) )
	HnAbort("KheDrsTaskClose internal error");
    }
    if( DEBUG19 )
      fprintf(stderr, "] KheDrsTaskClose\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsTaskBusyOnDay(KHE_DRS_TASK dt, KHE_DRS_DAY day,               */
/*    KHE_DRS_TASK_ON_DAY *dtd)                                              */
/*                                                                           */
/*  If dt is running on day, return true with *dtd set to its task on day    */
/*  object for that day.  Otherwise return false with *dtd set to NULL.      */
/*                                                                           */
/*  Implementation note.  The task on days are sorted chronologically,       */
/*  but we don't bother to use that property here.                           */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsTaskBusyOnDay(KHE_DRS_TASK dt, KHE_DRS_DAY day,
  KHE_DRS_TASK_ON_DAY *dtd)
{
  KHE_DRS_TASK_ON_DAY dtd2;  int i;
  HaArrayForEach(dt->days, dtd2, i)
    if( dtd2->day == day )
      return *dtd = dtd2, true;
  return *dtd = NULL, false;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_COST KheFindCost(KHE_CONSTRAINT c, int dev)                          */
/*                                                                           */
/*  Find the cost of c when the deviation is dev.                            */
/*                                                                           */
/*****************************************************************************/

/* *** OK but currently unused
static KHE_COST KheFindCost(KHE_CONSTRAINT c, int dev)
{
  switch( KheConstraintCostFunction(c) )
  {
    case KHE_STEP_COST_FUNCTION:

      return dev > 0 ? KheConstraintCombinedWeight(c) : 0;

    case KHE_LINEAR_COST_FUNCTION:

      return KheConstraintCombinedWeight(c) * dev;

    case KHE_QUADRATIC_COST_FUNCTION:

      return KheConstraintCombinedWeight(c) * dev * dev;

    default:

      HnAbort("KheFindCost: unknown cost function (%d)",
	KheConstraintCostFunction(c));
      return 0;  ** keep compiler happy **
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskDoMinCost(KHE_TASK task, KHE_DRS_ASST_OP op,                 */
/*    KHE_RESOURCE r, KHE_COST *cost)                                        */
/*                                                                           */
/*  Do the actual work of KheTaskMinCost, defined just below.                */
/*                                                                           */
/*  Implementation note.  For the step cost function there can always        */
/*  be no change in cost, when we are beyond the upper limit.  So the        */
/*  minimum cost change is never positive.                                   */
/*                                                                           */
/*****************************************************************************/

static bool KheTaskDoMinCost(KHE_TASK task, KHE_DRS_ASST_OP op,
  KHE_RESOURCE ra, KHE_RESOURCE rb, KHE_COST *cost)
{
  KHE_EVENT_RESOURCE er;  KHE_SOLN soln;  int i, durn;  KHE_MONITOR m;
  KHE_TASK child_task;  KHE_CONSTRAINT c;  KHE_RESOURCE_GROUP rg;
  KHE_PREFER_RESOURCES_MONITOR prm;  KHE_PREFER_RESOURCES_CONSTRAINT prc;
  KHE_LIMIT_RESOURCES_MONITOR lrm;  KHE_LIMIT_RESOURCES_CONSTRAINT lrc;
  KHE_COST weight;

  /* do it for task */
  er = KheTaskEventResource(task);
  if( er != NULL )
  {
    soln = KheTaskSoln(task);
    for( i = 0;  i < KheSolnEventResourceMonitorCount(soln, er);  i++ )
    {
      m = KheSolnEventResourceMonitor(soln, er, i);
      c = KheMonitorConstraint(m);
      weight = KheConstraintCombinedWeight(c);
      durn = KheTaskDuration(task);
      if( KheMonitorAttachedToSoln(m) && weight > 0 )
	switch( KheConstraintCostFunction(c) )
	{
	  case KHE_STEP_COST_FUNCTION:

	    switch( KheMonitorTag(m) )
	    {
	      case KHE_ASSIGN_RESOURCE_MONITOR_TAG:

		switch( op )
		{
		  case KHE_DRS_ASST_OP_UNASSIGN:

		    /* no change */
		    break;

		  case KHE_DRS_ASST_OP_ASSIGN:

		    *cost -= weight * durn;
		    break;

		  case KHE_DRS_ASST_OP_REPLACE:

		    /* no change */
		    break;

		  default:

		    HnAbort("KheTaskDoMinCost internal error (op %d)", op);
		    break;
		}
		break;

	      case KHE_PREFER_RESOURCES_MONITOR_TAG:

		prm = (KHE_PREFER_RESOURCES_MONITOR) m;
		prc = KhePreferResourcesMonitorConstraint(prm);
		rg = KhePreferResourcesConstraintDomain(prc);
		switch( op )
		{
		  case KHE_DRS_ASST_OP_UNASSIGN:

		    if( !KheResourceGroupContains(rg, ra) )
		      *cost -= weight * durn;
		    break;

		  case KHE_DRS_ASST_OP_ASSIGN:

		    break;

		  case KHE_DRS_ASST_OP_REPLACE:

		    if( !KheResourceGroupContains(rg, ra) )
		      *cost -= weight * durn;
		    break;

		  default:

		    HnAbort("KheTaskDoMinCost internal error (op %d)", op);
		    break;
		}
		break;

	      case KHE_AVOID_SPLIT_ASSIGNMENTS_MONITOR_TAG:

		/* not handling this one */
		return false;

	      case KHE_LIMIT_RESOURCES_MONITOR_TAG:

		lrm = (KHE_LIMIT_RESOURCES_MONITOR) m;
		lrc = KheLimitResourcesMonitorConstraint(lrm);
		rg = KheLimitResourcesConstraintDomain(lrc);
		switch( op )
		{
		  case KHE_DRS_ASST_OP_UNASSIGN:
		  case KHE_DRS_ASST_OP_ASSIGN:

		    if( KheResourceGroupContains(rg, ra) )
		      *cost -= weight * durn;
		    break;

		  case KHE_DRS_ASST_OP_REPLACE:

		    if( KheResourceGroupContains(rg, ra) !=
			KheResourceGroupContains(rg, rb) )
		      *cost -= weight * durn;
		    break;

		  default:

		    HnAbort("KheTaskDoMinCost internal error (op %d)", op);
		    break;
		}
		break;

	      default:

		HnAbort("KheDrsTaskDoMinCost: unexpected monitor tag (%s)",
		  KheMonitorTagShow(KheMonitorTag(m)));
	    }
	    break;

	  case KHE_LINEAR_COST_FUNCTION:

	    switch( KheMonitorTag(m) )
	    {
	      case KHE_ASSIGN_RESOURCE_MONITOR_TAG:

		switch( op )
		{
		  case KHE_DRS_ASST_OP_UNASSIGN:

		    *cost += weight * durn;
		    break;

		  case KHE_DRS_ASST_OP_ASSIGN:

		    *cost -= weight * durn;
		    break;

		  case KHE_DRS_ASST_OP_REPLACE:

		    /* no change */
		    break;

		  default:

		    HnAbort("KheTaskDoMinCost internal error (op %d)", op);
		    break;
		}
		break;

	      case KHE_PREFER_RESOURCES_MONITOR_TAG:

		prm = (KHE_PREFER_RESOURCES_MONITOR) m;
		prc = KhePreferResourcesMonitorConstraint(prm);
		rg = KhePreferResourcesConstraintDomain(prc);
		switch( op )
		{
		  case KHE_DRS_ASST_OP_UNASSIGN:

		    if( !KheResourceGroupContains(rg, ra) )
		      *cost -= weight * durn;
		    break;

		  case KHE_DRS_ASST_OP_ASSIGN:

		    if( !KheResourceGroupContains(rg, ra) )
		      *cost += weight * durn;
		    break;

		  case KHE_DRS_ASST_OP_REPLACE:

		    if( !KheResourceGroupContains(rg, ra) )
		      *cost -= weight * durn;
		    if( !KheResourceGroupContains(rg, rb) )
		      *cost += weight * durn;
		    break;

		  default:

		    HnAbort("KheTaskDoMinCost internal error (op %d)", op);
		    break;
		}
		break;

	      case KHE_AVOID_SPLIT_ASSIGNMENTS_MONITOR_TAG:

		/* not handling this one */
		return false;

	      case KHE_LIMIT_RESOURCES_MONITOR_TAG:

		lrm = (KHE_LIMIT_RESOURCES_MONITOR) m;
		lrc = KheLimitResourcesMonitorConstraint(lrm);
		rg = KheLimitResourcesConstraintDomain(lrc);
		switch( op )
		{
		  case KHE_DRS_ASST_OP_UNASSIGN:
		  case KHE_DRS_ASST_OP_ASSIGN:

		    if( KheResourceGroupContains(rg, ra) )
		      *cost -= weight * durn;
		    break;

		  case KHE_DRS_ASST_OP_REPLACE:

		    if( KheResourceGroupContains(rg, ra) !=
			KheResourceGroupContains(rg, rb) )
		      *cost -= weight * durn;
		    break;

		  default:

		    HnAbort("KheTaskDoMinCost internal error (op %d)", op);
		    break;
		}
		break;

	      default:

		HnAbort("KheDrsTaskDoMinCost: unexpected monitor tag (%s)",
		  KheMonitorTagShow(KheMonitorTag(m)));
	    }
	    break;

	  case KHE_QUADRATIC_COST_FUNCTION:

	    /* not handling this one */
	    return false;
	}
    }
  }

  /* do it for task's children */
  for( i = 0;  i < KheTaskAssignedToCount(task);  i++ )
  {
    child_task = KheTaskAssignedTo(task, i);
    if( !KheTaskDoMinCost(child_task, op, ra, rb, cost) )
      return false;
  }
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsTaskMinCost(KHE_DRS_TASK dt, KHE_DRS_ASST_OP op,              */
/*    KHE_DRS_RESOURCE dra, KHE_DRS_RESOURCE drb, KHE_COST *cost)             */
/*                                                                           */
/*  This function evaluates three expressions from the documentation:        */
/*                                                                           */
/*    op                           function                                  */
/*    ---------------------------------------------------------------        */
/*    KHE_DRS_ASST_OP_UNASSIGN     c-(M, ra)                                 */
/*    KHE_DRS_ASST_OP_ASSIGN       c+(M, ra)                                 */
/*    KHE_DRS_ASST_OP_REPLACE      c-+(M, ra, rb)                            */
/*    ---------------------------------------------------------------        */
/*                                                                           */
/*  where M is the set of monitors that monitor dt.                          */
/*                                                                           */
/*  Instead of returning the value it sets *cost to it.  The return value    */
/*  is true if the function can calculate this cost, and false if not.       */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsTaskMinCost(KHE_DRS_TASK dt, KHE_DRS_ASST_OP op,
  KHE_DRS_RESOURCE dra, KHE_DRS_RESOURCE drb, KHE_COST *cost)
{
  bool res;
  if( DEBUG41 )
    fprintf(stderr, "[ KheDrsTaskMinCost\n");
  *cost = 0;
  res = KheTaskDoMinCost(dt->task, op, dra->resource,
    drb == NULL ? NULL : drb->resource, cost);
  if( DEBUG41 )
    fprintf(stderr, "] KheDrsTaskMinCost returning %s\n", bool_show(res));
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskDoMinAssignCost(KHE_TASK task, KHE_RESOURCE r,               */
/*    KHE_COST *cost)                                                        */
/*                                                                           */
/*  Do the actual work of KheTaskMinAssignCost, defined just below.          */
/*                                                                           */
/*****************************************************************************/

/* ***
static bool KheTaskDoMinAssignCost(KHE_TASK task, KHE_RESOURCE r,
  KHE_COST *cost)
{
  KHE_EVENT_RESOURCE er;  KHE_SOLN soln;  int i, durn;  KHE_MONITOR m;
  KHE_TASK child_task;  KHE_CONSTRAINT c;  KHE_RESOURCE_GROUP rg;
  KHE_PREFER_RESOURCES_MONITOR prm;  KHE_PREFER_RESOURCES_CONSTRAINT prc;
  KHE_LIMIT_RESOURCES_MONITOR lrm;  KHE_LIMIT_RESOURCES_CONSTRAINT lrc;
  KHE_COST weight;

  ** do it for task **
  er = KheTaskEventResource(task);
  if( er != NULL )
  {
    soln = KheTaskSoln(task);
    for( i = 0;  i < KheSolnEventResourceMonitorCount(soln, er);  i++ )
    {
      m = KheSolnEventResourceMonitor(soln, er, i);
      c = KheMonitorConstraint(m);
      weight = KheConstraintCombinedWeight(c);
      durn = KheTaskDuration(task);
      if( KheMonitorAttachedToSoln(m) && weight > 0 )
      {
	if( KheConstraintCostFunction(c) == KHE_QUADRATIC_COST_FUNCTION )
	  return false;
	switch( KheMonitorTag(m) )
	{
	  case KHE_ASSIGN_RESOURCE_MONITOR_TAG:

	    *cost -= weight * durn;
	    break;

	  case KHE_PREFER_RESOURCES_MONITOR_TAG:

	    prm = (KHE_PREFER_RESOURCES_MONITOR) m;
	    prc = KhePreferResourcesMonitorConstraint(prm);
	    rg = KhePreferResourcesConstraintDomain(prc);
	    if( !KheResourceGroupContains(rg, r) )
	      *cost += weight * durn;
	    break;

	  case KHE_AVOID_SPLIT_ASSIGNMENTS_MONITOR_TAG:

	    ** not handling this one **
	    return false;

	  case KHE_LIMIT_RESOURCES_MONITOR_TAG:

	    lrm = (KHE_LIMIT_RESOURCES_MONITOR) m;
	    lrc = KheLimitResourcesMonitorConstraint(lrm);
	    rg = KheLimitResourcesConstraintDomain(lrc);
	    if( KheResourceGroupContains(rg, r) )
	      *cost -= weight * durn;
	    break;

	  default:

	    HnAbort("KheDrsTaskMinAssignCost: unexpected monitor tag (%s)",
	      KheMonitorTagShow(KheMonitorTag(m)));
	}
      }
    }
  }

  ** do it for task's children **
  for( i = 0;  i < KheTaskAssignedToCount(task);  i++ )
  {
    child_task = KheTaskAssignedTo(task, i);
    if( !KheTaskDoMinAssignCost(child_task, r, cost) )
      return false;
  }
  return true;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsTaskMinAssignCost(KHE_DRS_TASK dt, KHE_DRS_RESOURCE dr,       */
/*    KHE_COST *cost)                                                        */
/*                                                                           */
/*  If we can reasonably calculate a lower limit on the cost of assigning    */
/*  r to task, return true and add this lower limit to *cost.  The amount    */
/*  added may be negative.  If we can't reasonably do this, perhaps because  */
/*  the cost function is quadratic, return false with *cost indeterminate.   */
/*                                                                           */
/*  This function is called c+(M, S) in the documentation, where M is the    */
/*  set of all event resource monitors that monitor a particular task.       */
/*                                                                           */
/*****************************************************************************/

/* ***
static bool KheDrsTaskMinAssignCost(KHE_DRS_TASK dt, KHE_DRS_RESOURCE dr,
  KHE_COST *cost)
{
  *cost = 0;
  return KheTaskDoMinAssignCost(dt->task, dr->resource, cost);
}
*** */


/* ***
static bool KheTaskAssignCost(KHE_TASK task, KHE_RESOURCE r, KHE_COST *cost)
{
  KHE_EVENT_RESOURCE er;  KHE_SOLN soln;  int i;  KHE_MONITOR m;
  KHE_TASK child_task;  KHE_CONSTRAINT c;  KHE_RESOURCE_GROUP rg;
  KHE_PREFER_RESOURCES_MONITOR prm;  KHE_PREFER_RESOURCES_CONSTRAINT prc;
  KHE_LIMIT_RESOURCES_MONITOR lrm;  KHE_LIMIT_RESOURCES_CONSTRAINT lrc;

  ** do it for task **
  er = KheTaskEventResource(task);
  if( er != NULL )
  {
    soln = KheTaskSoln(task);
    for( i = 0;  i < KheSolnEventResourceMonitorCount(soln, er);  i++ )
    {
      m = KheSolnEventResourceMonitor(soln, er, i);
      if( KheMonitorAttachedToSoln(m) &&
	  KheConstraintCombinedWeight(KheMonitorConstraint(m)) > 0 )
      {
	c = KheMonitorConstraint(m);
	if( KheConstraintCostFunction(c) == KHE_QUADRATIC_COST_FUNCTION )
	  return false;
	switch( KheMonitorTag(m) )
	{
	  case KHE_ASSIGN_RESOURCE_MONITOR_TAG:

	    *cost -= KheFindCost(c, KheTaskDuration(task));
	    break;

	  case KHE_PREFER_RESOURCES_MONITOR_TAG:

	    prm = (KHE_PREFER_RESOURCES_MONITOR) m;
	    prc = KhePreferResourcesMonitorConstraint(prm);
	    rg = KhePreferResourcesConstraintDomain(prc);
	    if( !KheResourceGroupContains(rg, r) )
	      *cost += KheFindCost(c, KheTaskDuration(task));
	    break;

	  case KHE_AVOID_SPLIT_ASSIGNMENTS_MONITOR_TAG:

	    ** upper limit here only **
	    *cost += KheFindCost(c, KheTaskDuration(task));
	    break;

	  case KHE_LIMIT_RESOURCES_MONITOR_TAG:

	    ** upper limit here only **
	    lrm = (KHE_LIMIT_RESOURCES_MONITOR) m;
	    lrc = KheLimitResourcesMonitorConstraint(lrm);
	    rg = KheLimitResourcesConstraintDomain(lrc);
	    if( KheResourceGroupContains(rg, r) )
	      *cost += KheFindCost(c, KheTaskDuration(task));
	    break;

	  default:

	    HnAbort("KheDrsTaskAssignCost: unexpected monitor tag (%s)",
	      KheMonitorTagShow(KheMonitorTag(m)));
	}
      }
    }
  }

  ** do it for task's children **
  for( i = 0;  i < KheTaskAssignedToCount(task);  i++ )
  {
    child_task = KheTaskAssignedTo(task, i);
    if( !KheTaskAssignCost(child_task, r, cost) )
      return false;
  }
  return true;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheTaskDoMinUnassignCost(KHE_TASK task, KHE_RESOURCE r,             */
/*    KHE_COST *cost)                                                        */
/*                                                                           */
/*  Like KheTaskDoMinAssignCost only for unassignment instead of assignment. */
/*                                                                           */
/*****************************************************************************/

/* ***
static bool KheTaskDoMinUnassignCost(KHE_TASK task, KHE_RESOURCE r,
  KHE_COST *cost)
{
  KHE_EVENT_RESOURCE er;  KHE_SOLN soln;  int i, durn;  KHE_MONITOR m;
  KHE_TASK child_task;  KHE_CONSTRAINT c;  KHE_RESOURCE_GROUP rg;
  KHE_PREFER_RESOURCES_MONITOR prm;  KHE_PREFER_RESOURCES_CONSTRAINT prc;
  KHE_LIMIT_RESOURCES_MONITOR lrm;  KHE_LIMIT_RESOURCES_CONSTRAINT lrc;
  KHE_COST weight;

  ** do it for task **
  er = KheTaskEventResource(task);
  if( er != NULL )
  {
    soln = KheTaskSoln(task);
    for( i = 0;  i < KheSolnEventResourceMonitorCount(soln, er);  i++ )
    {
      m = KheSolnEventResourceMonitor(soln, er, i);
      c = KheMonitorConstraint(m);
      weight = KheConstraintCombinedWeight(c);
      durn = KheTaskDuration(task);
      if( KheMonitorAttachedToSoln(m) && weight > 0 )
      {
	if( KheConstraintCostFunction(c) == KHE_QUADRATIC_COST_FUNCTION )
	  return false;
	switch( KheMonitorTag(m) )
	{
	  case KHE_ASSIGN_RESOURCE_MONITOR_TAG:

	    *cost += weight * durn;
	    break;

	  case KHE_PREFER_RESOURCES_MONITOR_TAG:

	    prm = (KHE_PREFER_RESOURCES_MONITOR) m;
	    prc = KhePreferResourcesMonitorConstraint(prm);
	    rg = KhePreferResourcesConstraintDomain(prc);
	    if( !KheResourceGroupContains(rg, r) )
	      *cost -= weight * durn;
	    break;

	  case KHE_AVOID_SPLIT_ASSIGNMENTS_MONITOR_TAG:

	    ** not handling this one **
	    return false;

	  case KHE_LIMIT_RESOURCES_MONITOR_TAG:

	    lrm = (KHE_LIMIT_RESOURCES_MONITOR) m;
	    lrc = KheLimitResourcesMonitorConstraint(lrm);
	    rg = KheLimitResourcesConstraintDomain(lrc);
	    if( KheResourceGroupContains(rg, r) )
	      *cost -= weight * durn;
	    break;

	  default:

	    HnAbort("KheDrsTaskMinUnassignCost: unexpected monitor tag (%s)",
	      KheMonitorTagShow(KheMonitorTag(m)));
	}
      }
    }
  }

  ** do it for task's children **
  for( i = 0;  i < KheTaskAssignedToCount(task);  i++ )
  {
    child_task = KheTaskAssignedTo(task, i);
    if( !KheTaskDoMinUnassignCost(child_task, r, cost) )
      return false;
  }
  return true;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsTaskMinUnassignCost(KHE_DRS_TASK dt, KHE_DRS_RESOURCE dr,     */
/*    KHE_COST *cost)                                                        */
/*                                                                           */
/*  Like KheTaskMinAssignCost only for unassignment instead of assignment.   */
/*  This function is called c-(M, S) in the documentation.                   */
/*                                                                           */
/*****************************************************************************/

/* ***
static bool KheDrsTaskMinUnassignCost(KHE_DRS_TASK dt, KHE_DRS_RESOURCE dr,
  KHE_COST *cost)
{
  *cost = 0;
  return KheTaskDoMinUnassignCost(dt->task, dr->resource, cost);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsTaskAssignCost(KHE_DRS_TASK dt, KHE_DRS_RESOURCE dr,          */
/*    KHE_COST *cost)                                                        */
/*                                                                           */
/*  If we can reasonably calculate the cost of assigning dr to dt, return    */
/*  true and set *cost to the cost of changing from unassigned to assigned.  */
/*  This change may be negative.  Return an upper bound on the cost if it    */
/*  cannot be calculated exactly.  If we can't calculate even an upper       */
/*  bound (a quadratic cost function is the only reason for this), return    */
/*  false with *cost indeterminate.                                          */
/*                                                                           */
/*****************************************************************************/

/* ***
static bool KheDrsTaskAssignCost(KHE_DRS_TASK dt, KHE_DRS_RESOURCE dr,
  KHE_COST *cost)
{
  *cost = 0;
  return KheTaskAssignCost(dt->task, dr->resource, cost);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsTaskUnAssignCost(KHE_DRS_TASK dt, KHE_DRS_RESOURCE dr,        */
/*    KHE_COST *cost)                                                        */
/*                                                                           */
/*  If we can reasonably calculate the cost of unassigning dr from dt, return*/
/*  true and set *cost to the cost of changing from assigned to unassigned.  */
/*  This change may be negative.  Return an upper bound on the cost if it    */
/*  cannot be calculated exactly.  If we can't calculate even an upper       */
/*  bound (a quadratic cost function is the only reason for this), return    */
/*  false with *cost indeterminate.                                          */
/*                                                                           */
/*  Implementation note.  The cost here is just the negative of the cost     */
/*  returned by KheDrsTaskAssignCost.                                        */
/*                                                                           */
/*****************************************************************************/

/* ***
static bool KheDrsTaskUnAssignCost(KHE_DRS_TASK dt, KHE_DRS_RESOURCE dr,
  KHE_COST *cost)
{
  bool res;  KHE_COST add_cost;
  res = KheDrsTaskAssignCost(dt, dr, &add_cost);
  return *cost = - add_cost, res;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  char *KheDrsTaskId(KHE_DRS_TASK dt)                                      */
/*                                                                           */
/*  Return the Id of dt's task, or "-" if dt is NULL.                        */
/*                                                                           */
/*****************************************************************************/

/* *** good but currently unused
static char *KheDrsTaskId(KHE_DRS_TASK dt)
{
  return dt == NULL ? "-" : KheTaskId(dt->task);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsTaskDebug(KHE_DRS_TASK dt, int verbosity,                     */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of dt onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsTaskDebug(KHE_DRS_TASK dt, int verbosity,
  int indent, FILE *fp)
{
  KHE_DRS_TASK_ON_DAY dtd;  int i;
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  KheTaskDebug(dt->task, 1, -1, fp);
  if( verbosity > 1 )
  {
    fprintf(fp, " asst_cost %.5f, non_asst_cost %.5f ",
      KheCostShow(dt->asst_cost), KheCostShow(dt->non_asst_cost));
    fprintf(fp, "(%d, %s, %s; ", dt->index_in_encl_dtc,
      dt->open ? "open" : "closed",
      dt->closed_asst == NULL ? "-" : KheDrsResourceId(dt->closed_asst));
    HaArrayForEach(dt->days, dtd, i)
      KheDrsTaskOnDayDebug(dtd, verbosity, -1, fp);
    fprintf(fp, ")");
  }
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_TASK_ON_DAY"                                          */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_TASK_ON_DAY KheDrsTaskOnDayMake(KHE_DRS_TASK encl_dt,            */
/*    KHE_DRS_DAY day, KHE_TASK task, KHE_TIME time,                         */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new task on day object with these attributes.                     */
/*  This function assumes that the task is not assigned anything.            */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_TASK_ON_DAY KheDrsTaskOnDayMake(KHE_DRS_TASK encl_dt,
  KHE_DRS_DAY day, KHE_TASK task, KHE_TIME time,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_TASK_ON_DAY res;
  HnAssert(task != NULL && time != NULL, "KheDrsTaskOnDayMake internal error");
  HaMake(res, drs->arena);
  res->encl_dt = encl_dt;
  res->day = day;
  res->task = task;
  res->time = time;
  res->closed_asst = NULL;
  HaArrayInit(res->external_today, drs->arena);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SHIFT KheDrsTaskOnDayShift(KHE_DRS_TASK_ON_DAY dtd)              */
/*                                                                           */
/*  Return the shift containing dtd.                                         */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused
static KHE_DRS_SHIFT KheDrsTaskOnDayShift(KHE_DRS_TASK_ON_DAY dtd)
{
  return dtd->encl_dt->encl_dtc->encl_shift;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsTaskOnDayCmp(const void *t1, const void *t2)                   */
/*                                                                           */
/*  Comparison function for sorting an array of task on day objects by       */
/*  increasing day index.                                                    */
/*                                                                           */
/*****************************************************************************/

static int KheDrsTaskOnDayCmp(const void *t1, const void *t2)
{
  KHE_DRS_TASK_ON_DAY dtd1, dtd2;
  dtd1 = * (KHE_DRS_TASK_ON_DAY *) t1;
  dtd2 = * (KHE_DRS_TASK_ON_DAY *) t2;
  return dtd1->day->frame_index - dtd2->day->frame_index;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsOptionalResourceGroupEqual(KHE_RESOURCE_GROUP rg1,            */
/*    KHE_RESOURCE_GROUP rg2)                                                */
/*                                                                           */
/*  Return true if rg1 and rg2 are both NULL, or both non-NULL and equal.    */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsOptionalResourceGroupEqual(KHE_RESOURCE_GROUP rg1,
  KHE_RESOURCE_GROUP rg2)
{
  if( rg1 == NULL )
    return rg2 == NULL;
  else
    return rg2 != NULL && KheResourceGroupEqual(rg1, rg2);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsTaskOnDayHasAssignedTaskExpr(KHE_DRS_TASK_ON_DAY dtd,         */
/*    KHE_RESOURCE_GROUP rg, KHE_DRS_EXPR_ASSIGNED_TASK *res)                */
/*                                                                           */
/*  If dtd has an assigned task expression with resource group attribute     */
/*  rg, return true with *res set to that expression; else return false.     */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsTaskOnDayHasAssignedTaskExpr(KHE_DRS_TASK_ON_DAY dtd,
  KHE_RESOURCE_GROUP rg, KHE_DRS_EXPR_ASSIGNED_TASK *res)
{
  KHE_DRS_EXPR e;  int i;  KHE_DRS_EXPR_ASSIGNED_TASK eat;
  HaArrayForEach(dtd->external_today, e, i)
  {
    eat = (KHE_DRS_EXPR_ASSIGNED_TASK) e;
    if( KheDrsOptionalResourceGroupEqual(rg, eat->resource_group) )
      return *res = eat, true;
  }
  return *res = NULL, false;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsTaskOnDayAddAssignedTaskExpr(KHE_DRS_TASK_ON_DAY dtd,         */
/*    KHE_DRS_EXPR_ASSIGNED_TASK eat)                                        */
/*                                                                           */
/*  Add eat (assumed new) to dtd.                                            */
/*                                                                           */
/*****************************************************************************/

static void KheDrsTaskOnDayAddAssignedTaskExpr(KHE_DRS_TASK_ON_DAY dtd,
  KHE_DRS_EXPR_ASSIGNED_TASK eat)
{
  HaArrayAddLast(dtd->external_today, (KHE_DRS_EXPR) eat);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsTaskOnDayIsLast(KHE_DRS_TASK_ON_DAY dtd)                      */
/*                                                                           */
/*  Return true if dtd is the last task on day of its task.                  */
/*  This function is used only when debugging.                               */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsTaskOnDayIsLast(KHE_DRS_TASK_ON_DAY dtd)
{
  return dtd == HaArrayLast(dtd->encl_dt->days);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsTaskOnDayLeafSet(KHE_DRS_TASK_ON_DAY dtd,                     */
/*    KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs)                  */
/*                                                                           */
/*  Call KheDrsExprLeafSet for each external expression affected by dtd.     */
/*  Or do nothing if dtd is NULL.                                            */
/*                                                                           */
/*****************************************************************************/

static void KheDrsTaskOnDayLeafSet(KHE_DRS_TASK_ON_DAY dtd,
  KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR e;  int i;
  if( dtd != NULL )
    HaArrayForEach(dtd->external_today, e, i)
      KheDrsExprLeafSet(e, dtd, dr, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsTaskOnDayLeafClear(KHE_DRS_TASK_ON_DAY dtd,                   */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Call KheDrsExprLeafClear for each external expression affected by dtd.   */
/*  Or do nothing if dtd is NULL.                                            */
/*                                                                           */
/*****************************************************************************/

static void KheDrsTaskOnDayLeafClear(KHE_DRS_TASK_ON_DAY dtd,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR e;  int i;
  if( dtd != NULL )
    HaArrayForEach(dtd->external_today, e, i)
      KheDrsExprLeafClear(e, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheDrsTaskOnDayId(KHE_DRS_TASK_ON_DAY dtd)                         */
/*                                                                           */
/*  Return the Id of dtd's task, or "-" if dtd is NULL.                      */
/*                                                                           */
/*****************************************************************************/

/* *** good but currently unused
static char *KheDrsTaskOnDayId(KHE_DRS_TASK_ON_DAY dtd)
{
  return dtd == NULL ? "-" : KheDrsTaskId(dtd->encl_dt);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsTaskOnDayDebug(KHE_DRS_TASK_ON_DAY dtd,                       */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of dtd onto fp with the given verbosity and indent.          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsTaskOnDayDebug(KHE_DRS_TASK_ON_DAY dtd,
  int verbosity, int indent, FILE *fp)
{
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  KheTaskDebug(dtd->task, 1, -1, fp);
  if( verbosity > 1 )
    fprintf(fp, "(time %s, day %s, closed_asst %s)", KheTimeId(dtd->time),
      KheDrsDayId(dtd->day), dtd->closed_asst == NULL ? "-" :
      KheDrsResourceId(dtd->closed_asst->encl_dr));
  /* ***
  if( dtd->task != NULL )
  {
    KheTaskDebug(dtd->task, 1, -1, fp);
    fprintf(fp, "(%s:%s:%s)", KheTimeId(dtd->time), KheDrsDayId(dtd->day),
      dtd->closed_asst == NULL ? "-" :
      KheDrsResourceId(dtd->closed_asst->encl_dr));
  }
  else
  {
    fprintf(fp, "-(%s)", KheDrsDayId(dtd->day));
  }
  *** */
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_TASK_CLASS"                                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsTaskClassMake(KHE_TASK_CLASS tc,                              */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new drs task class based on tc and add it to its first day.       */
/*  Also make drs tasks corresponding to tc's tasks and add them to the      */
/*  drs class task and to drs->all_root_tasks.                               */
/*                                                                           */
/*  Return true if all this is done successfully.  The only reason why       */
/*  false is returned is that creating one of the tasks uncovered a case     */
/*  where a resource is assigned two tasks on one day.                       */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsTaskClassMake(KHE_TASK_CLASS tc,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_TASK_CLASS res;  int i;  KHE_TASK task;  KHE_DRS_TASK dt;
  KHE_DRS_DAY day;  KHE_COST asst_cost, non_asst_cost;

  /* get or make a drs task class object */
  HaMake(res, drs->arena);
  res->orig_task_class = tc;
  res->encl_shift = NULL;  /* reset later */
  res->day_range = KheDrsDayRangeMake(KheTaskClassFirstDayIndex(tc),
    KheTaskClassLastDayIndex(tc));
  HaArrayInit(res->all_tasks, drs->arena);
  HaArrayInit(res->unassigned_tasks, drs->arena);
  /* res->expand_used = 0; */
  res->expand_prev_unfixed = -1;

  /* make drs tasks for its khe tasks and add them */
  for( i = 0;  i < KheTaskClassTaskCount(tc);  i++ )
  {
    task = KheTaskClassTask(tc, i, &asst_cost, &non_asst_cost);
    if( !KheDrsTaskMake(res, i, task, asst_cost, non_asst_cost, drs, &dt) )
      return false;
    HaArrayAddLast(res->all_tasks, dt);
    if( dt->closed_asst == NULL )
      HaArrayAddLast(res->unassigned_tasks, dt);
  }

  /* make res one of the task classes of its first day */
  day = HaArray(drs->all_days, res->day_range.first);
  KheDrsDayAddTaskClass(day, res, drs);
  if( DEBUG50 && res->day_range.first < res->day_range.last )
    KheDrsTaskClassDebug(res, drs, 2, 2, stderr);
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsTaskClassOrganizeUnassignedTasks(KHE_DRS_TASK_CLASS dtc)      */
/*                                                                           */
/*  Organize the unassigned tasks of dtc for solving.                        */
/*                                                                           */
/*  When this function is called, unassigned_tasks may be out of order,      */
/*  and, because unassigned_tasks is handled lazily, some of them may in     */
/*  fact be assigned.  This function fixes both of those problems.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsTaskClassOrganizeUnassignedTasks(KHE_DRS_TASK_CLASS dtc)
{
  KHE_DRS_TASK dt;

  /* sort unassigned_tasks into order with assigned tasks at the end */
  HaArraySortUnique(dtc->unassigned_tasks, &KheDrsTaskIndexInEnclCmp);

  /* remove assigned tasks from the end */
  while( HaArrayCount(dtc->unassigned_tasks) > 0 )
  {
    dt = HaArrayLast(dtc->unassigned_tasks);
    if( dt->closed_asst == NULL )
      break;
    HaArrayDeleteLast(dtc->unassigned_tasks);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsTaskClassOpen(KHE_DRS_TASK_CLASS dtc, KHE_DRS_DAY_RANGE ddr,  */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  If dtc is suitable (if its day range lies within ddr, and its resource   */
/*  domain includes at least one open resource), then open it.  Opening dtc  */
/*  basically means organizing its unassigned tasks into decreasing          */
/*  desirability order and opening them.                                     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsTaskClassOpen(KHE_DRS_TASK_CLASS dtc, KHE_DRS_DAY_RANGE ddr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_TASK dt;  int i;
  if( DEBUG19 )
    fprintf(stderr, "[ KheDrsTaskClassOpen(dtc, %d-%d)\n", ddr.first, ddr.last);
  if( KheDrsDayRangeSubset(dtc->day_range, ddr) &&
      !KheResourceSetDisjointGroup(drs->selected_resource_set,
	KheTaskClassDomain(dtc->orig_task_class)) )
  {
    /* dtc can open; organize and open unassigned_tasks; say none are used */
    KheDrsTaskClassOrganizeUnassignedTasks(dtc);
    HaArrayForEach(dtc->unassigned_tasks, dt, i)
      KheDrsTaskOpen(dt, drs);
    dtc->expand_prev_unfixed = -1;
    /* dtc->expand_used = 0; */
  }
  else
  {
    /* dtc can't open; set dtc->expand_used to make that clear */
    dtc->expand_prev_unfixed = HaArrayCount(dtc->unassigned_tasks) - 1;
    /* dtc->expand_used = INT_MAX; */
  }
  if( DEBUG19 )
    fprintf(stderr, "] KheDrsTaskClassOpen\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsTaskClassClose(KHE_DRS_TASK_CLASS dtc,                        */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Close dtc.  Some of its tasks may be already closed; that's OK.          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsTaskClassClose(KHE_DRS_TASK_CLASS dtc)
{
  KHE_DRS_TASK dt;  int i;
  HaArrayForEach(dtc->unassigned_tasks, dt, i)
    KheDrsTaskClose(dt, NULL);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsTaskClassDebug(KHE_DRS_TASK_CLASS dtc, int verbosity,         */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)  */
/*                                                                           */
/*  Debug print of dtc onto fp with the given verbosity and indent.          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsTaskClassDebug(KHE_DRS_TASK_CLASS dtc,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)
{
  KHE_DRS_TASK dt;  int i, pos;
  if( indent >= 0 )
  {
    fprintf(fp, "%*s[ Task class ", indent, "");
    if( dtc == NULL )
      fprintf(fp, "(free) ]\n");
    else
    {
      if( drs != NULL )
	KheDrsDayRangeDebug(dtc->day_range, drs, 1, 0, fp);
      HaArrayForEach(dtc->all_tasks, dt, i)
      {
	fprintf(fp, "%*s  ", indent, "");
	KheDrsTaskDebug(dt, verbosity, -1, fp);
	fprintf(fp, " %s%s\n", HaArrayContains(dtc->unassigned_tasks, dt,&pos) ?
	  "" : "assigned ", KheDrsTaskExpandRoleShow(dt->expand_role));
      }
      fprintf(fp, "%*s]\n", indent, "");
    }
  }
  else
  {
    if( dtc == NULL )
      fprintf(fp, "{free}");
    else
    {
      fprintf(fp, "{");
      if( HaArrayCount(dtc->all_tasks) >= 2 )
      {
	KheDrsTaskDebug(HaArrayFirst(dtc->all_tasks), 1, -1, fp);
	fprintf(fp, "...(%d)", HaArrayCount(dtc->all_tasks));
	/* KheDrsTaskDebug(HaArrayLast(dtc->all_tasks), 1, -1, fp); */
      }
      else
	KheDrsTaskDebug(HaArrayFirst(dtc->all_tasks), 1, -1, fp);
      fprintf(fp, "}");
      /* ***
      fprintf(fp, "TaskClass(");
      KheDrsDayRangeDebug(dtc->day_range, drs, 1, -1, fp);
      fprintf(fp, ", %d tasks)", HaArrayCount(dtc->all_tasks));
      *** */
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_TASK_CLASS - expansion"                               */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheDrsTaskClassExpandBegin(KHE_DRS_TASK_CLASS dtc,                  */
/*    KHE_DRS_SOLN prev_soln, KHE_DRS_EXPANDER de)                           */
/*                                                                           */
/*  Begin an expansion involving task class dtc.                             */
/*                                                                           */
/*****************************************************************************/
static void KheDrsExpanderAddMustAssign(KHE_DRS_EXPANDER de);

static void KheDrsTaskClassExpandBegin(KHE_DRS_TASK_CLASS dtc,
  KHE_DRS_SOLN prev_soln, KHE_DRS_EXPANDER de)
{
  KHE_DRS_TASK dt;  int i;
  HaArrayForEach(dtc->unassigned_tasks, dt, i)
    if( dt->expand_role != KHE_DRS_TASK_EXPAND_FIXED )
    {
      if( KheDrsExpanderOpenToExtraCost(de, dt->non_asst_cost) )
	dt->expand_role = KHE_DRS_TASK_EXPAND_FREE;
      else
      {
	/* dt must be assigned, otherwise cost will be too high */
	dt->expand_role = KHE_DRS_TASK_EXPAND_MUST;
	dtc->encl_shift->expand_must_assign_count++;
	KheDrsExpanderAddMustAssign(de);
      }
    }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsTaskClassExpandEnd(KHE_DRS_TASK_CLASS dtc,                    */
/*    KHE_DRS_EXPANDER de)                                                   */
/*                                                                           */
/*  End an expansion involving task class dtc.                               */
/*                                                                           */
/*****************************************************************************/

static void KheDrsTaskClassExpandEnd(KHE_DRS_TASK_CLASS dtc,
  KHE_DRS_EXPANDER de)
{
  KHE_DRS_TASK dt;  int i;
  HaArrayForEach(dtc->unassigned_tasks, dt, i)
    dt->expand_role = KHE_DRS_TASK_EXPAND_NO;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsTaskClassAcceptResourceBegin(KHE_DRS_TASK_CLASS dtc,          */
/*    KHE_DRS_RESOURCE_ON_DAY drd, KHE_DRS_TASK_ON_DAY *dtd)                 */
/*                                                                           */
/*  If dtc accepts drd (because drd lies in its tasks' domains, and not all  */
/*  the tasks are assigned), then return true and set *dtd to the relevant   */
/*  task on day of one of dtc's tasks that drd can be assigned to.  Else     */
/*  return false.                                                            */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsTaskClassAcceptResourceBegin(KHE_DRS_TASK_CLASS dtc,
  KHE_DRS_RESOURCE_ON_DAY drd, KHE_DRS_TASK_ON_DAY *dtd)
{
  KHE_DRS_TASK dt;  int i, count;
  if( KheResourceGroupContains(KheTaskClassDomain(dtc->orig_task_class),
    drd->encl_dr->resource) )
  {
    /* search forwards for next unfixed task */
    count = HaArrayCount(dtc->unassigned_tasks);
    for( i = dtc->expand_prev_unfixed + 1;  i < count;  i++ )
    {
      dt = HaArray(dtc->unassigned_tasks, i);
      if( dt->expand_role != KHE_DRS_TASK_EXPAND_FIXED )
      {
	if( !KheDrsTaskBusyOnDay(dt, drd->day, dtd) )
	  HnAbort("KheDrsTaskClassAcceptResourceBegin internal error");
        dtc->expand_prev_unfixed = i;
	return true;
      }
    }
  }

  /* if we get here we've failed to identify a suitable task */
  return *dtd = NULL, false;
}


/* *** old version
static bool KheDrsTaskClassAcceptResourceBegin(KHE_DRS_TASK_CLASS dtc,
  KHE_DRS_RESOURCE_ON_DAY drd, KHE_DRS_TASK_ON_DAY *dtd)
{
  KHE_DRS_TASK dt;
  if( dtc->expand_used < HaArrayCount(dtc->unassigned_tasks) &&
      KheResourceGroupContains(KheTaskClassDomain(dtc->orig_task_class),
	drd->encl_dr->resource) )
  {
    dt = HaArray(dtc->unassigned_tasks, dtc->expand_used);
    if( !KheDrsTaskBusyOnDay(dt, drd->day, dtd) )
      HnAbort("KheDrsTaskClassAcceptResourceBegin internal error");
    dtc->expand_used++;
    return true;
  }
  else
    return *dtd = NULL, false;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsTaskClassAcceptResourceEnd(KHE_DRS_TASK_CLASS dtc,            */
/*    KHE_DRS_TASK_ON_DAY dtd)                                               */
/*                                                                           */
/*  Undo the effect of a corresponding KheDrsTaskClassAcceptResourceBegin    */
/*  which returned true.                                                     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsTaskClassAcceptResourceEnd(KHE_DRS_TASK_CLASS dtc,
  KHE_DRS_TASK_ON_DAY dtd)
{
  KHE_DRS_TASK dt;  int i;
  HnAssert(dtc->expand_prev_unfixed < HaArrayCount(dtc->unassigned_tasks) &&
    dtd->encl_dt == HaArray(dtc->unassigned_tasks, dtc->expand_prev_unfixed),
    "KheDrsTaskClassAcceptResourceEnd internal error");

  /* search backwards for next unfixed task, or start of array */
  for( i = dtc->expand_prev_unfixed - 1;  i >= 0;  i-- )
  {
    dt = HaArray(dtc->unassigned_tasks, dtc->expand_prev_unfixed);
    if( dt->expand_role != KHE_DRS_TASK_EXPAND_FIXED )
      break;
  }
  dtc->expand_prev_unfixed = i;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_TASK_CLASS - one-extra and two-extra selection"       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsTaskClassOneExtraAvailable(KHE_DRS_TASK_CLASS dtc,            */
/*    int open_resource_count)                                               */
/*                                                                           */
/*  Return true if dtc is available for participating in a one-extra         */
/*  dominance test.  Parameter open_resource_count is the number of open     */
/*  resources.                                                               */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsTaskClassOneExtraAvailable(KHE_DRS_TASK_CLASS dtc,
  int open_resource_count)
{
  return dtc == NULL ||
    open_resource_count - 1 < HaArrayCount(dtc->unassigned_tasks);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsTaskClassMinCost(KHE_DRS_TASK_CLASS dtc, KHE_DRS_ASST_OP op,  */
/*    KHE_DRS_RESOURCE dra, KHE_DRS_RESOURCE drb, int open_resource_count,   */
/*    KHE_COST *avail_cost, int verbosity, int indent, FILE *fp)             */
/*                                                                           */
/*  This function evaluates three expressions from the documentation:        */
/*                                                                           */
/*    op                           function                                  */
/*    ---------------------------------------------------------------        */
/*    KHE_DRS_ASST_OP_UNASSIGN     min(p) c-(M, ra)                          */
/*    KHE_DRS_ASST_OP_ASSIGN       min(q) c+(M, ra)                          */
/*    KHE_DRS_ASST_OP_REPLACE      min(p) c-+(M, ra, rb)                     */
/*    ---------------------------------------------------------------        */
/*                                                                           */
/*  But instead of returning the value it adds it to *avail_cost.  The       */
/*  return value is true if the function is able to calculate this cost,     */
/*  and false if not.                                                        */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsTaskClassMinCost(KHE_DRS_TASK_CLASS dtc, KHE_DRS_ASST_OP op,
  KHE_DRS_RESOURCE dra, KHE_DRS_RESOURCE drb, int open_resource_count,
  KHE_COST *avail_cost, int verbosity, int indent, FILE *fp)
{
  int p, max_p;  KHE_DRS_TASK dt;  KHE_COST cost, min_cost;

  if( fp != NULL )
  {
    fprintf(stderr, "[ KheDrsTaskClassMinCost(");
    if( dtc == NULL )
      fprintf(stderr, "(free)");
    else
      KheDrsTaskClassDebug(dtc, NULL, 1, -1, stderr);
    fprintf(stderr, ", op %d, dra %s, drb %s, count %d, &avail_cost)\n",
      (int) op, dra == NULL ? "-" : KheDrsResourceId(dra),
      drb == NULL ? "-" : KheDrsResourceId(drb), open_resource_count);
  }

  /* if dtc is NULL (represents a free day) there is no change to *avail_cost */
  if( dtc == NULL )
  {
    if( fp != NULL )
      fprintf(stderr, "] KheDrsTaskClassMinCost returning true (free)\n");
    return true;
  }

  /* find the minimum, over unassigned tasks 0 ... max_p, of the cost */
  max_p = min(open_resource_count - 1, HaArrayCount(dtc->unassigned_tasks) - 1);
  min_cost = KheCost(INT_MAX, INT_MAX);
  for( p = 0;  p <= max_p;  p++ )
  {
    if( fp != NULL )
      fprintf(stderr, "  p = %d\n", p);
    dt = HaArray(dtc->unassigned_tasks, p);   /* the (p+1)st unassigned task */
    if( !KheDrsTaskMinCost(dt, op, dra, drb, &cost) )
    {
      if( fp != NULL )
	fprintf(stderr, "] KheDrsTaskClassMinCost returning false\n");
      return false;
    }
    if( cost < min_cost )
      min_cost = cost;
  }

  /* wrapup */
  HnAssert(min_cost < KheCost(INT_MAX, INT_MAX),
    "KheDrsTaskClassMinCost internal error");
  if( fp != NULL )
    fprintf(stderr, "] KheDrsTaskClassMinCost returning true (min_cost %.5f)\n",
      KheCostShow(min_cost));
  *avail_cost += min_cost;
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsTaskClassMinCostDebug(KHE_DRS_TASK_CLASS dtc,                 */
/*    KHE_DRS_ASST_OP op, KHE_DRS_RESOURCE dra, KHE_DRS_RESOURCE drb,        */
/*    int open_resource_count, int verbosity, int indent, FILE *fp)          */
/*                                                                           */
/*  Like KheDrsTaskClassMinCost but with debug output.                       */
/*                                                                           */
/*****************************************************************************/

/* ***
static char *KheDrsAsstOpShow(KHE_DRS_ASST_OP op);

static void KheDrsTaskClassMinCostDebug(KHE_DRS_TASK_CLASS dtc,
  KHE_DRS_ASST_OP op, KHE_DRS_RESOURCE dra, KHE_DRS_RESOURCE drb,
  int open_resource_count, int verbosity, int indent, FILE *fp)
{
  int p, max_p;  KHE_DRS_TASK dt;  KHE_COST cost, min_cost;  bool res;

  fprintf(fp, "%*s[ KheDrsTaskClassMinCostDebug(", indent, "");
  if( dtc == NULL )
    fprintf(fp, "(free)");
  else
    KheDrsTaskClassDebug(dtc, NULL, 1, -1, fp);
  fprintf(fp, ", op %s, dra %s, drb %s, count %d)\n",
    KheDrsAsstOpShow(op), dra == NULL ? "-" : KheDrsResourceId(dra),
    drb == NULL ? "-" : KheDrsResourceId(drb), open_resource_count);

  ** if dtc is NULL (represents a free day) there is no change to *avail_cost **
  if( dtc == NULL )
  {
    fprintf(fp, "%*s] KheDrsTaskClassMinCostDebug returning true (free)\n",
      indent, "");
    return;
  }

  ** find the minimum, over unassigned tasks 0 ... max_p, of the cost **
  max_p = min(open_resource_count - 1, HaArrayCount(dtc->unassigned_tasks) - 1);
  min_cost = KheCost(INT_MAX, INT_MAX);
  for( p = 0;  p <= max_p;  p++ )
  {
    fprintf(fp, "%*sp %d: ", indent + 2, "", p);
    dt = HaArray(dtc->unassigned_tasks, p);   ** the (p+1)st unassigned task **
    res = KheDrsTaskMinCost(dt, op, dra, drb, &cost);
    KheTaskDebug(dt->task, 1, -1, fp);
    if( !res )
    {
      fprintf(fp, ", res false\n");
      fprintf(fp, "%*s] KheDrsTaskClassMinCostDebug returning false\n",
	indent, "");
      return;
    }
    if( cost < min_cost )
      min_cost = cost;
    fprintf(fp, ", res true, cost %.5f, min_cost %.5f\n",
      KheCostShow(cost), KheCostShow(min_cost));
  }

  ** wrapup **
  HnAssert(min_cost < KheCost(INT_MAX, INT_MAX),
    "KheDrsTaskClassMinCost internal error");
  fprintf(fp, "%*s] KheDrsTaskClassMinCost returning true (min_cost %.5f)\n",
    indent, "", KheCostShow(min_cost));
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsTaskClassMinAssignCost(KHE_DRS_TASK_CLASS dtc,                */
/*    KHE_DRS_RESOURCE dr, int q, KHE_COST *avail_cost)                      */
/*                                                                           */
/*  This function is denoted min(q)c+(M, r) in the documentation, although   */
/*  it adds its result to *avail_cost rather than returning it.              */
/*                                                                           */
/*****************************************************************************/

/* ***
static bool KheDrsTaskClassMinAssignCost(KHE_DRS_TASK_CLASS dtc,
  KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs, KHE_COST *avail_cost)
{
  int q, max_q;  bool min_cost_defined;  KHE_DRS_TASK dt;
  KHE_COST cost, min_cost;

  ** if dtc is NULL (represents a free day) there is no change to *avail_cost **
  if( dtc == NULL )
    return true;

  max_q = min(KheDrsResourceSetCount(drs->open_resources) - 1,
    HaArrayCount(dtc->unassigned_tasks) - 1);
  ** return false if there are not enough unassigned tasks for all resources **
  ** does not occur here
  if( q >= HaArrayCount(dtc->unassigned_tasks) )
    return false;
  *** **

  ** find the maximum, over unassigned tasks 0 ... max_q, of the asst cost **
  min_cost_defined = false;
  min_cost = 0;  ** actually undefined until min_cost_defined is true **
  for( q = 0;  q <= max_q;  q++ )
  {
    dt = HaArray(dtc->unassigned_tasks, q);
    if( !KheDrsTaskMinCost(dt, true, dr, &cost) )
      return false;
    if( !min_cost_defined || cost < min_cost )
    {
      min_cost = cost;
      min_cost_defined = true;
    }
  }

  ** wrapup **
  HnAssert(min_cost_defined, "KheDrsTaskClassMinAssignCost internal error");
  *avail_cost += min_cost;
  return true;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsTaskClassMinUnassignCost(KHE_DRS_TASK_CLASS dtc,              */
/*    KHE_DRS_RESOURCE dr, int q, KHE_COST *avail_cost)                      */
/*                                                                           */
/*  This function is denoted min(q)c-(M, r) in the documentation, although   */
/*  it adds its result to *avail_cost rather than returning it.              */
/*                                                                           */
/*****************************************************************************/

/* ***
static bool KheDrsTaskClassMinUnassignCost(KHE_DRS_TASK_CLASS dtc,
  KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs, KHE_COST *avail_cost)
{
  int p, max_p;  bool min_cost_defined;  KHE_DRS_TASK dt;
  KHE_COST cost, min_cost;

  ** if dtc is NULL (represents a free day) there is no change to *avail_cost **
  if( dtc == NULL )
    return true;

  ** return false if there are not enough unassigned tasks for all resources **
  if( KheDrsResourceSetCount(drs->open_resources) - 1 >=
        HaArrayCount(dtc->unassigned_tasks) )
    return false;
  max_p = min(KheDrsResourceSetCount(drs->open_resources) - 1,
    HaArrayCount(dtc->unassigned_tasks) - 1);

  ** find the maximum, over unassigned tasks 0 ... max_p, of the unasst cost **
  min_cost_defined = false;
  min_cost = 0;  ** actually undefined until min_cost_defined is true **
  for( p = 0;  p <= max_p;  p++ )
  {
    dt = HaArray(dtc->unassigned_tasks, p);
    if( !KheDrsTaskMinCost(dt, false, dr, &cost) )
      return false;
    if( !min_cost_defined || cost < min_cost )
    {
      min_cost = cost;
      min_cost_defined = true;
    }
  }

  ** wrapup **
  HnAssert(min_cost_defined, "KheDrsTaskClassMinAssignCost internal error");
  *avail_cost += min_cost;
  return true;
}
*** */


/* ***
static bool KheDrsTaskClassMaxAssignCost(KHE_DRS_TASK_CLASS dtc,
  KHE_DRS_RESOURCE dr, int q, KHE_COST *avail_cost)
{
  int i;  bool max_cost_defined;  KHE_DRS_TASK dt;  KHE_COST cost, max_cost;

  ** if dtc is NULL (represents a free day) there is no change to *avail_cost **
  if( dtc == NULL )
    return true;

  ** return false if there are not enough unassigned tasks for all resources **
  if( q >= HaArrayCount(dtc->unassigned_tasks) )
    return false;

  ** find the maximum, over unassigned tasks 0 ... q, of the assignment cost **
  max_cost_defined = false;
  max_cost = 0;  ** actually undefined until max_cost_defined is true **
  for( i = 0;  i <= q;  i++ )
  {
    dt = HaArray(dtc->unassigned_tasks, i);
    if( !KheDrsTaskAssignCost(dt, dr, &cost) )
      return false;
    if( !max_cost_defined || cost > max_cost )
    {
      max_cost = cost;
      max_cost_defined = true;
    }
  }

  ** wrapup **
  HnAssert(max_cost_defined, "KheDrsTaskClassMaxAssignCost internal error");
  *avail_cost -= max_cost;
  return true;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsTaskClassMinUnAssignCost(KHE_DRS_TASK_CLASS dtc,              */
/*    KHE_DRS_RESOURCE dr, int q, KHE_COST *avail_cost)                      */
/*                                                                           */
/*  Calculate a lower limit on the cost of not assigning dr to dtc, assuming */
/*  any number from 0 up to q other assignments are present.  If this can    */
/*  be done, return true and subtract the cost of doing it from *avail_cost. */
/*  If it can't be done, return false.                                       */
/*                                                                           */
/*  The minimum cost may be negative, in which case *avail_cost increases.   */
/*                                                                           */
/*****************************************************************************/

/* ***
static bool KheDrsTaskClassMinUnAssignCost(KHE_DRS_TASK_CLASS dtc,
  KHE_DRS_RESOURCE dr, int q, KHE_COST *avail_cost)
{
  int i;  bool min_cost_defined;  KHE_DRS_TASK dt;  KHE_COST cost, min_cost;

  ** if dtc is NULL (represents a free day) there is no change to avail_cost **
  if( dtc == NULL )
    return true;

  ** find the minimum, over unassigned tasks 0 ... q, of the unassign cost **
  min_cost_defined = false;
  min_cost = 0;  ** actually undefined until min_cost_defined is true **
  for( i = 0;  i <= q;  i++ )
  {
    dt = HaArray(dtc->unassigned_tasks, i);
    if( !KheDrsTaskMinUnAssignCost(dt, dr, &cost) )
      return false;
    if( !min_cost_defined || cost < min_cost )
    {
      min_cost = cost;
      min_cost_defined = true;
    }
  }

  ** wrapup **
  HnAssert(min_cost_defined, "KheDrsTaskClassMinUnAssignCost internal error");
  *avail_cost -= min_cost;
  return true;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_SHIFT"                                                */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SHIFT KheDrsShiftMake(KHE_DRS_DAY day, KHE_DRS_TASK_CLASS dtc,   */
/*    int index_in_day, KHE_DYNAMIC_RESOURCE_SOLVER drs)                     */
/*                                                                           */
/*  Make a new shift containing just dtc.                                    */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_SHIFT KheDrsShiftMake(KHE_DRS_DAY day, KHE_DRS_TASK_CLASS dtc,
  /* int index_in_day, */ KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SHIFT res;
  HaMake(res, drs->arena);
  res->encl_day = day;
  res->open_shift_index = -1;
  /* res->index_in_cycle = drs->shift_index_in_cycle++; */
  /* res->index_in_day = index_in_day; */
  res->expand_must_assign_count = 0;
  res->expand_max_included_free_resource_count = 0;
  HaArrayInit(res->task_classes, drs->arena);
  res->signer = KheDrsSignerMake(NULL, NULL, res, drs);
  HaArrayAddLast(res->task_classes, dtc);
  /* HaArrayInit(res->dom_tests, a); */
  /* HaArrayInit(res->nr_internal_today, a); */
  /* HaArrayInit(res->resource_signatures, a); */
  /* HaArrayInit(res->assts, a); */
  res->asst_trie = NULL;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheDrsShiftId(KHE_DRS_SHIFT ds)                                    */
/*                                                                           */
/*  Return an Id for shift.                                                  */
/*                                                                           */
/*****************************************************************************/

static char *KheDrsShiftId(KHE_DRS_SHIFT ds)
{
  static char buff[200];
  if( ds->open_shift_index >= 0 )
    snprintf(buff, 200, "%s:%d", KheDrsDayId(ds->encl_day),
      ds->open_shift_index);
  else
    snprintf(buff, 200, "%s:?", KheDrsDayId(ds->encl_day));
  return buff;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsShiftAcceptsClass(KHE_DRS_SHIFT ds, KHE_DRS_TASK_CLASS dtc)   */
/*                                                                           */
/*  If ds can accept dtc, because ds's classes have the same times and       */
/*  the same total workload as dtc, then add dtc to ds and return true.      */
/*  Otherwise change nothing and return false.                               */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsShiftAcceptsClass(KHE_DRS_SHIFT ds, KHE_DRS_TASK_CLASS dtc)
{
  KHE_DRS_TASK_CLASS dtc2;  int first_day_index, last_day_index, day_index;
  KHE_TASK_CLASS tc1, tc2;  KHE_TIME time1, time2;  float workload1, workload2;

  /* get the two task classes, tc1 and tc2, that have to be compared */
  dtc2 = HaArrayFirst(ds->task_classes);
  tc1 = dtc->orig_task_class;
  tc2 = dtc2->orig_task_class;

  /* first_day_index and last_day_index must be the same */
  first_day_index = KheTaskClassFirstDayIndex(tc1);
  if( KheTaskClassFirstDayIndex(tc2) != first_day_index )
    return false;
  last_day_index = KheTaskClassLastDayIndex(tc1);
  if( KheTaskClassLastDayIndex(tc2) != last_day_index )
    return false;

  /* busy times and workloads on each day must be the same */
  for( day_index = first_day_index;  day_index <= last_day_index;  day_index++ )
  {
    time1 = KheTaskClassDayTime(tc1, day_index, &workload1);
    time2 = KheTaskClassDayTime(tc2, day_index, &workload2);
    if( time1 != time2 || workload1 != workload2 )
      return false;
  }

  /* all good, add dtc to ds and return true */
  HaArrayAddLast(ds->task_classes, dtc);
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftOpen(KHE_DRS_SHIFT ds, KHE_DRS_DAY_RANGE ddr,            */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Open ds and its task classes.                                            */
/*                                                                           */
/*****************************************************************************/

static void KheDrsShiftOpen(KHE_DRS_SHIFT ds, KHE_DRS_DAY_RANGE ddr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_TASK_CLASS dtc;  int i;
  ds->open_shift_index = HaArrayCount(drs->open_shifts);
  HaArrayAddLast(drs->open_shifts, ds);
  HaArrayForEach(ds->task_classes, dtc, i)
    KheDrsTaskClassOpen(dtc, ddr, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftClose(KHE_DRS_SHIFT ds, KHE_DYNAMIC_RESOURCE_SOLVER drs) */
/*                                                                           */
/*  Close ds and its task classes.                                           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsShiftClose(KHE_DRS_SHIFT ds, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_TASK_CLASS dtc;  int i;
  HaArrayForEach(ds->task_classes, dtc, i)
    KheDrsTaskClassClose(dtc);
  ds->open_shift_index = -1;
  /* drs->open_shift_count--; */
  KheDrsSignerClear(ds->signer, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftAddResourceAsst(KHE_DRS_SHIFT ds, KHE_DRS_RESOURCE dr,   */
/*    KHE_DRS_ASST_TO_TASK_CLASS asst)                                       */
/*                                                                           */
/*  Add an assignment for dr to ds.                                          */
/*                                                                           */
/*****************************************************************************/

/* *** no longer used, saving signatures instead
static void KheDrsShiftAddResourceAsst(KHE_DRS_SHIFT ds, KHE_DRS_RESOURCE dr,
  KHE_DRS_ASST_TO_TASK_CLASS asst)
{
  HaArrayFill(ds->assts, dr->open_resource_index + 1, NULL);
  HnAssert(HaArray(ds->assts, dr->open_resource_index) == NULL,
    "KheDrsShiftAddResourceAsst internal error");
  HaArrayPut(ds->assts, dr->open_resource_index, NULL);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftAddResourceSignature(KHE_DRS_SHIFT ds,                   */
/*    KHE_DRS_RESOURCE dr, KHE_DRS_SIGNATURE sig)                            */
/*                                                                           */
/*  Add a signature for dr to ds.                                            */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsShiftAddResourceSignature(KHE_DRS_SHIFT ds,
  KHE_DRS_RESOURCE dr, KHE_DRS_SIGNATURE sig)
{
  HaArrayFill(ds->resource_signatures, dr->open_resource_index + 1, NULL);
  HnAssert(HaArray(ds->resource_signatures, dr->open_resource_index) == NULL,
    "KheDrsShiftAddResourceSignature internal error");
  HaArrayPut(ds->resource_signatures, dr->open_resource_index, sig);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SIGNATURE KheDrsShiftResourceSignature(KHE_DRS_SHIFT ds,         */
/*    KHE_DRS_RESOURCE dr)                                                   */
/*                                                                           */
/*  Return the resource signature associated with dr in ds, or NULL if none. */
/*                                                                           */
/*****************************************************************************/

/* ***
static KHE_DRS_SIGNATURE KheDrsShiftResourceSignature(KHE_DRS_SHIFT ds,
  KHE_DRS_RESOURCE dr)
{
  return dr->open_resource_index >= HaArrayCount(ds->resource_signatures) ?
    NULL : HaArray(ds->resource_signatures, dr->open_resource_index);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftAddOpenExpr(KHE_DRS_SHIFT ds, KHE_DRS_EXPR e)            */
/*                                                                           */
/*  Add open expression e to ds.                                             */
/*                                                                           */
/*****************************************************************************/

static void KheDrsShiftAddOpenExpr(KHE_DRS_SHIFT ds, KHE_DRS_EXPR e)
{
  /* *** all rubbish here it seems
  KHE_DRS_DAY day;
  day = ds->encl_day;
  HnAssert(KheDrsDayRangeContains(e->open_children_by_day.range,
    day->open_day_index),
    "KheDrsShiftAddOpenExpr internal error:  day range %d - %d, day %d",
    e->open_children_by_day.range.first,
    e->open_children_by_day.range.last, day->open_day_index);
  *** */
  KheDrsSignerAddOpenExpr(ds->signer, e);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftAddDomTest(KHE_DRS_SHIFT ds, KHE_DRS_DOM_TEST dom_test)  */
/*                                                                           */
/*  Add dom_test to ds.                                                      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsShiftAddDomTest(KHE_DRS_SHIFT ds, KHE_DRS_DOM_TEST dom_test,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KheDrsSignerAddDomTest(ds->signer, dom_test, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_SHIFT - expansion"                                    */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftExpandBegin(KHE_DRS_SHIFT ds, KHE_DRS_SOLN prev_soln,    */
/*    KHE_DRS_EXPANDER de)                                                   */
/*                                                                           */
/*  Initialize ds and its task classes and tasks for expansion of            */
/*  prev_soln.  Only don't initialize the assignment trie yet.               */
/*                                                                           */
/*****************************************************************************/
/* ***
static bool KheDrsExpanderContainsAsstToTask(KHE_DRS_EXPANDER de,
  KHE_DRS_TASK dt);
*** */

static void KheDrsShiftExpandBegin(KHE_DRS_SHIFT ds, KHE_DRS_SOLN prev_soln,
  KHE_DRS_EXPANDER de)
{
  KHE_DRS_TASK_CLASS dtc;  int i;
  ds->expand_must_assign_count = 0;
  ds->expand_max_included_free_resource_count = 0;
  HaArrayForEach(ds->task_classes, dtc, i)
    KheDrsTaskClassExpandBegin(dtc, prev_soln, de);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftExpandEnd(KHE_DRS_SHIFT ds, KHE_DRS_EXPANDER de)         */
/*                                                                           */
/*  An expansion involving this shift is ending; clear it out.               */
/*                                                                           */
/*****************************************************************************/
static void KheDrsShiftAsstTrieFree(KHE_DRS_SHIFT_ASST_TRIE dsat,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsShiftExpandEnd(KHE_DRS_SHIFT ds, KHE_DRS_EXPANDER de)
{
  KHE_DRS_TASK_CLASS dtc;  int i;
  HaArrayForEach(ds->task_classes, dtc, i)
    KheDrsTaskClassExpandEnd(dtc, de);
  KheDrsShiftAsstTrieFree(ds->asst_trie, de->solver);
  ds->asst_trie = NULL;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftBuildShiftAsstTrie(KHE_DRS_SHIFT ds,                     */
/*    KHE_DRS_SOLN prev_soln, KHE_DRS_DAY next_day,                          */
/*    KHE_DRS_RESOURCE_SET free_resources, KHE_DYNAMIC_RESOURCE_SOLVER drs)  */
/*                                                                           */
/*  Initialize the assignment trie of ds.                                    */
/*                                                                           */
/*****************************************************************************/
static KHE_DRS_SHIFT_ASST_TRIE KheDrsShiftAsstTrieBuild(KHE_DRS_SHIFT ds,
  KHE_DRS_SOLN prev_soln, KHE_DRS_DAY prev_day, KHE_DRS_DAY next_day,
  KHE_DRS_RESOURCE_SET all_free_resources, int all_free_resources_index,
  KHE_DRS_RESOURCE_SET included_free_resources,
  KHE_DRS_ASST_TO_TASK_SET included_fixed_assts, KHE_DRS_EXPANDER de);

static void KheDrsShiftBuildShiftAsstTrie(KHE_DRS_SHIFT ds,
  KHE_DRS_SOLN prev_soln, KHE_DRS_DAY prev_day, KHE_DRS_DAY next_day,
  KHE_DRS_RESOURCE_SET all_free_resources,
  KHE_DRS_ASST_TO_TASK_SET included_fixed_assts,
  KHE_DRS_EXPANDER de, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_RESOURCE_SET included_free_resources;
  HnAssert(ds->asst_trie == NULL, "KheDrsShiftInitAsstTrie internal error");
  included_free_resources = KheDrsResourceSetMake(drs);
  ds->asst_trie = KheDrsShiftAsstTrieBuild(ds, prev_soln, prev_day, next_day,
    all_free_resources, 0, included_free_resources, included_fixed_assts, de);
  KheDrsResourceSetFree(included_free_resources, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftExpandByShifts(KHE_DRS_SHIFT ds, int shift_index,        */
/*    KHE_DRS_SOLN prev_soln, KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de,     */
/*    KHE_DRS_RESOURCE_SET free_resources)                                   */
/*                                                                           */
/*  Expand shift ds and beyond using free_resources.                         */
/*                                                                           */
/*****************************************************************************/
static void KheDrsShiftAsstTrieExpandByShifts(KHE_DRS_SHIFT_ASST_TRIE dsat,
  KHE_DRS_SHIFT ds, int shift_index, KHE_DRS_SOLN prev_soln,
  KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de,
  KHE_DRS_RESOURCE_SET free_resources, int avail_index,
  int selected_resource_count, KHE_DRS_RESOURCE_SET omitted_resources);

static void KheDrsShiftExpandByShifts(KHE_DRS_SHIFT ds, int shift_index,
  KHE_DRS_SOLN prev_soln, KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de,
  KHE_DRS_RESOURCE_SET free_resources)
{
  KHE_DRS_RESOURCE_SET omitted_resources;
  if( ds->asst_trie != NULL )
  {
    omitted_resources = KheDrsResourceSetMake(de->solver);
    KheDrsShiftAsstTrieExpandByShifts(ds->asst_trie, ds, shift_index,
      prev_soln, next_day, de, free_resources, 0, 0, omitted_resources);
    KheDrsResourceSetFree(omitted_resources, de->solver);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftDebug(KHE_DRS_SHIFT ds, KHE_DYNAMIC_RESOURCE_SOLVER drs, */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of shift ds onto fp with the given verbosity and indent.     */
/*  Here drs may be NULL, although the result will be more readable if it    */
/*  is non-NULL.                                                             */
/*                                                                           */
/*****************************************************************************/
static bool KheDrsShiftAsstTrieResourceIsForced(KHE_DRS_SHIFT_ASST_TRIE dsat,
  KHE_DRS_RESOURCE dr);
static void KheDrsShiftAsstTrieDebug(KHE_DRS_SHIFT_ASST_TRIE dsat,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp);

static void KheDrsShiftDebug(KHE_DRS_SHIFT ds, KHE_DYNAMIC_RESOURCE_SOLVER drs,
  int verbosity, int indent, FILE *fp)
{
  KHE_DRS_TASK_CLASS dtc;  KHE_DRS_RESOURCE dr;  int i;
  if( indent >= 0 )
  {
    fprintf(fp, "%*s[ Shift %d (expand_min %d, expand_max %d)\n", indent, "",
      ds->open_shift_index, ds->expand_must_assign_count,
      ds->expand_max_included_free_resource_count);
    HaArrayForEach(ds->task_classes, dtc, i)
      KheDrsTaskClassDebug(dtc, drs, verbosity, indent + 2, fp);
    if( verbosity >= 2 && ds->asst_trie != NULL )
    {
      KheDrsShiftAsstTrieDebug(ds->asst_trie, drs, verbosity-1, indent+2, fp);
      KheDrsResourceSetForEach(drs->open_resources, dr, i)
	if( KheDrsShiftAsstTrieResourceIsForced(ds->asst_trie, dr) )
	  fprintf(fp, "%*s  forced resource %s\n", indent, "",
	    KheDrsResourceId(dr));
    }
    fprintf(fp, "%*s]\n", indent, "");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_SHIFT_ASST"                                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SHIFT_ASST KheDrsShiftAsstMake(KHE_DYNAMIC_RESOURCE_SOLVER drs)  */
/*                                                                           */
/*  Make a new shift assignment object.                                      */
/*                                                                           */
/*****************************************************************************/
static KHE_DRS_SIGNATURE KheDrsSignatureMake(KHE_DYNAMIC_RESOURCE_SOLVER drs);
static void KheDrsAsstToTaskSetClear(KHE_DRS_ASST_TO_TASK_SET dats);

static KHE_DRS_ASST_TO_TASK_SET KheDrsAsstToTaskSetMake(
  KHE_DYNAMIC_RESOURCE_SOLVER drs);

static KHE_DRS_SHIFT_ASST KheDrsShiftAsstMake(KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SHIFT_ASST res;

  /* get or make the object */
  if( HaArrayCount(drs->shift_asst_free_list) > 0 )
  {
    res = HaArrayLastAndDelete(drs->shift_asst_free_list);
    KheDrsAsstToTaskSetClear(res->assts_to_tasks);
  }
  else
  {
    HaMake(res, drs->arena);
    res->assts_to_tasks = KheDrsAsstToTaskSetMake(drs);
  }

  /* all good, make sig and return */
  res->sig = KheDrsSignatureMake(drs);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftAsstFree(KHE_DRS_SHIFT_ASST dsa,                         */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Free shift assignment dsa.                                               */
/*                                                                           */
/*****************************************************************************/
static void KheDrsSignatureUnRefer(KHE_DRS_SIGNATURE sig,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsShiftAsstFree(KHE_DRS_SHIFT_ASST dsa,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KheDrsSignatureUnRefer(dsa->sig, drs);
  dsa->sig = NULL;
  HaArrayAddLast(drs->shift_asst_free_list, dsa);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsShiftAsstDominates(KHE_DRS_SHIFT_ASST dsa1,                   */
/*    KHE_DRS_SHIFT_ASST dsa2, KHE_DRS_SIGNER dsg,                           */
/*    KHE_DRS_DOM_TEST_TYPE dom_test_type, KHE_DYNAMIC_RESOURCE_SOLVER drs)  */
/*                                                                           */
/*  Return true if dsa1 dominates dsa2.                                      */
/*                                                                           */
/*****************************************************************************/
static bool KheDrsSignerDominates(KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE sig1,
  KHE_DRS_SIGNATURE sig2, KHE_COST *avail_cost,
  int verbosity, int indent, FILE *fp);

static bool KheDrsShiftAsstDominates(KHE_DRS_SHIFT_ASST dsa1,
  KHE_DRS_SHIFT_ASST dsa2, KHE_DRS_SIGNER dsg,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_COST avail_cost;
  avail_cost = 0;
  return KheDrsSignerDominates(dsg, dsa1->sig, dsa2->sig, &avail_cost,
    0, 0, NULL);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftAsstDebug(KHE_DRS_SHIFT_ASST dsa, int verbosity,         */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of dsa onto fp with the given verbosity and indent.          */
/*                                                                           */
/*****************************************************************************/
static void KheDrsSignatureDebug(KHE_DRS_SIGNATURE sig, int verbosity,
  int indent, FILE *fp);

static void KheDrsShiftAsstDebug(KHE_DRS_SHIFT_ASST dsa, int verbosity,
  int indent, FILE *fp)
{
  int i;  KHE_DRS_ASST_TO_TASK tasst; /* KHE_DRS_TASK_ON_DAY dtd; */
  if( indent >= 0 )
    fprintf(fp, "%*s", indent, "");
  fprintf(fp, "(");
  KheDrsSignatureDebug(dsa->sig, 1, -1, fp);
  /* HaArrayForEach(dsa->task_on_days, dtd, i) */
  KheDrsAsstToTaskSetForEach(dsa->assts_to_tasks, tasst, i)
  {
    if( i > 0 )
      fprintf(fp, ", ");
    else
      fprintf(fp, " ");
    KheDrsTaskDebug(tasst.fixed_dtd->encl_dt, 1, -1, fp);
  }
  fprintf(fp, ")");
  if( indent > 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_SHIFT_ASST - expansion"                               */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftAsstExpandByShifts(KHE_DRS_SHIFT_ASST dsa,               */
/*    KHE_DRS_SHIFT ds, int shift_index, KHE_DRS_SOLN prev_soln,             */
/*    KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de,                             */
/*    KHE_DRS_RESOURCE_SET omitted_resources)                                */
/*                                                                           */
/*  Expand dsa.                                                              */
/*                                                                           */
/*****************************************************************************/
static void KheDrsSolnExpandByShifts(KHE_DRS_SOLN prev_soln,
  KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de,
  KHE_DRS_RESOURCE_SET avail_resources, int shift_index);
static void KheDrsExpanderMarkBegin(KHE_DRS_EXPANDER de);
static void KheDrsExpanderMarkEnd(KHE_DRS_EXPANDER de);
static bool KheDrsExpanderIsOpen(KHE_DRS_EXPANDER de);
static void KheDrsExpanderAddAsstToTaskSet(KHE_DRS_EXPANDER de,
  KHE_DRS_ASST_TO_TASK_SET dats, KHE_DRS_SHIFT ds);

static void KheDrsShiftAsstExpandByShifts(KHE_DRS_SHIFT_ASST dsa,
  KHE_DRS_SHIFT ds, int shift_index, KHE_DRS_SOLN prev_soln,
  KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de,
  KHE_DRS_RESOURCE_SET omitted_resources)
{
  /* save the expander */
  KheDrsExpanderMarkBegin(de);

  /* add the assignments stored in dsa to the expander */
  KheDrsExpanderAddAsstToTaskSet(de, dsa->assts_to_tasks, NULL);
  /* ***
  KheDrsAsstToTaskSetForEach(dsa->assts_to_tasks, tasst, i)
    KheDrsExpanderAddAsstToTask(de, tasst);
  *** */

  /* if the expander is still open, recurse */
  if( KheDrsExpanderIsOpen(de) )
    KheDrsSolnExpandByShifts(prev_soln, next_day, de, omitted_resources,
      shift_index + 1);

  /* restore the expander */
  KheDrsExpanderMarkEnd(de);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsShiftAsstContainsResource(KHE_DRS_SHIFT_ASST dsa,             */
/*    KHE_DRS_RESOURCE dr)                                                   */
/*                                                                           */
/*  Return true if dsa contains an assignment which assigns dr.              */
/*                                                                           */
/*****************************************************************************/
static KHE_DRS_RESOURCE KheDrsAsstToTaskResource(KHE_DRS_ASST_TO_TASK dat);

static bool KheDrsShiftAsstContainsResource(KHE_DRS_SHIFT_ASST dsa,
  KHE_DRS_RESOURCE dr)
{
  KHE_DRS_ASST_TO_TASK dat;  int i;
  KheDrsAsstToTaskSetForEach(dsa->assts_to_tasks, dat, i)
    if( KheDrsAsstToTaskResource(dat) == dr )
      return true;
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_SHIFT_ASST_TRIE" - this whole submodule is expansion  */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SHIFT_ASST_TRIE KheDrsShiftAsstTrieMake(                         */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new, empty shift asst trie.                                       */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_SHIFT_ASST_TRIE KheDrsShiftAsstTrieMake(
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SHIFT_ASST_TRIE res;
  if( HaArrayCount(drs->shift_asst_trie_free_list) > 0 )
  {
    res = HaArrayLastAndDelete(drs->shift_asst_trie_free_list);
    HaArrayClear(res->assts);
    HaArrayClear(res->children);
  }
  else
  {
    HaMake(res, drs->arena);
    HaArrayInit(res->assts, drs->arena);
    HaArrayInit(res->children, drs->arena);
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftAsstTrieFree(KHE_DRS_SHIFT_ASST_TRIE dsat,               */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Free dsat, including freeing its shift assignment objects and its        */
/*  proper descendant trie objects.                                          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsShiftAsstTrieFree(KHE_DRS_SHIFT_ASST_TRIE dsat,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SHIFT_ASST dsa;  int i;  KHE_DRS_SHIFT_ASST_TRIE child_dsat;

  if( dsat != NULL )
  {
    /* free the shift assignment objects */
    HaArrayForEach(dsat->assts, dsa, i)
      KheDrsShiftAsstFree(dsa, drs);

    /* free the proper descendant trie objects */
    HaArrayForEach(dsat->children, child_dsat, i)
      KheDrsShiftAsstTrieFree(child_dsat, drs);

    /* free dsat itself */
    HaArrayAddLast(drs->shift_asst_trie_free_list, dsat);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsShiftAsstTrieDominates(KHE_DRS_SHIFT_ASST_TRIE dsat,          */
/*    KHE_DRS_SHIFT_ASST dsa, KHE_DRS_SIGNER dsg,                            */
/*    KHE_DRS_DOM_TEST_TYPE dom_test_type, KHE_DYNAMIC_RESOURCE_SOLVER drs)  */
/*                                                                           */
/*  Return true if dsat->assts contains a shift assignment that dominates    */
/*  dsa.                                                                     */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsShiftAsstTrieDominates(KHE_DRS_SHIFT_ASST_TRIE dsat,
  KHE_DRS_SHIFT_ASST dsa, KHE_DRS_SIGNER dsg,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SHIFT_ASST other_dsa;  int i;
  HaArrayForEach(dsat->assts, other_dsa, i)
    if( KheDrsShiftAsstDominates(other_dsa, dsa, dsg, drs) )
      return true;
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftAsstTrieRemoveDominated(KHE_DRS_SHIFT_ASST_TRIE dsat,    */
/*    KHE_DRS_SHIFT_ASST dsa, KHE_DRS_SIGNER dsg,                            */
/*    KHE_DRS_DOM_TEST_TYPE dom_test_type, KHE_DYNAMIC_RESOURCE_SOLVER drs)  */
/*                                                                           */
/*  Remove from dsat->assts those shift assignments dominated by dsa.        */
/*                                                                           */
/*****************************************************************************/

static void KheDrsShiftAsstTrieRemoveDominated(KHE_DRS_SHIFT_ASST_TRIE dsat,
  KHE_DRS_SHIFT_ASST dsa, KHE_DRS_SIGNER dsg,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SHIFT_ASST other_dsa;  int i;
  HaArrayForEach(dsat->assts, other_dsa, i)
    if( KheDrsShiftAsstDominates(dsa, other_dsa, dsg, drs) )
    {
      HaArrayDeleteAndPlug(dsat->assts, i);
      KheDrsShiftAsstFree(other_dsa, drs);
      i--;
    }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftAsstTrieAddShiftAsst(KHE_DRS_SHIFT_ASST_TRIE dsat,       */
/*    KHE_DRS_SHIFT_ASST dsa)                                                */
/*                                                                           */
/*  Add dsa to dsat, without worrying about dominance testing.               */
/*                                                                           */
/*****************************************************************************/

static void KheDrsShiftAsstTrieAddShiftAsst(KHE_DRS_SHIFT_ASST_TRIE dsat,
  KHE_DRS_SHIFT_ASST dsa)
{
  HaArrayAddLast(dsat->assts, dsa);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftAsstTrieMeldShiftAsst(KHE_DRS_SHIFT_ASST_TRIE dsat,      */
/*    KHE_DRS_SHIFT_ASST dsa, KHE_DRS_SIGNER dsg,                            */
/*    KHE_DRS_DOM_TEST_TYPE dom_test_type, KHE_DYNAMIC_RESOURCE_SOLVER drs)  */
/*                                                                           */
/*  Add dsa to dsat->assts, including dominance testing.                     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsShiftAsstTrieMeldShiftAsst(KHE_DRS_SHIFT_ASST_TRIE dsat,
  KHE_DRS_SHIFT_ASST dsa, KHE_DRS_SIGNER dsg,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  if( KheDrsShiftAsstTrieDominates(dsat, dsa, dsg, drs) )
  {
    /* dsa is dominated, so free dsa */
    KheDrsShiftAsstFree(dsa, drs);
  }
  else
  {
    /* remove other assts that dsa dominates, then add dsa to dsat */
    KheDrsShiftAsstTrieRemoveDominated(dsat, dsa, dsg, drs);
    KheDrsShiftAsstTrieAddShiftAsst(dsat, dsa);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftMakeEvaluateAndMeldSoln(KHE_DRS_SHIFT_ASST_TRIE dsat,    */
/*    KHE_DRS_SIGNER dsg, KHE_DRS_RESOURCE_SET included_resources,        */
/*    KHE_DRS_SOLN prev_soln, KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de)     */
/*                                                                           */
/*  Use drs->expand_dtds to make a new shift assignment object, then         */
/*  evaluate it and meld it into dsat->assts, including dominance testing.   */
/*                                                                           */
/*****************************************************************************/

/* *** replaced by KheDrrsExpanderMakeAndMeldShiftAsst
static void KheDrsExprEvalShiftSignature(KHE_DRS_EXPR e,
  KHE_DRS_SOLN prev_soln, int next_di, KHE_DRS_SHIFT ds,
  KHE_DRS_SIGNATURE sig, KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsShiftMakeEvaluateAndMeldSoln(KHE_DRS_SHIFT_ASST_TRIE dsat,
  KHE_DRS_SIGNER dsg, KHE_DRS_RESOURCE_SET included_resources,
  KHE_DRS_SOLN prev_soln, KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de)
  ** KHE_DYNAMIC_RESOURCE_SOLVER drs) **
{
  KHE_DRS_SHIFT_ASST dsa;  int i ** , j **;  ** KHE_DRS_EXPR e; **
  ** KHE_DRS_RESOURCE dr; **  ** KHE_DRS_ASST_TO_TASK tasst; **
  ** KHE_DRS_TASK_ON_DAY dtd;  KHE_DRS_RESOURCE_ON_DAY drd; **

  ** make shift asst from de->assts_to_tasks **
  dsa = KheDrsShiftAsstMake(de->solver);
  HaArrayAppend(dsa->assts_to_tasks, de->assts_to_tasks, i);
  ** HaArrayAppend(dsa->assts_to_tasks, drs->expand_assts_to_tasks, i); **
  ** HaArrayAppend(dsa->task_on_days, drs->expand_dtds, i); **

  ** push assignments **
  ** HaArrayForEach(dsa->task_on_days, dtd, i) **
  ** *** no, already done
  HaArrayForEach(dsa->assts_to_tasks, tasst, i)
    KheDrsTaskOnDayLeafSet(tasst.fixed_dtd, tasst.asst->resource_on_day, drs);
  *** **
    ** ***
    if( tasst.fixed_dtd != NULL )
    {
      ** drd = KheDrsResourceOnDaySetResourceOnDay(included_resources, i); **
      ** drd = KheDrsResourceOnDay(dr, next_day); **
      KheDrsTaskOnDayLeafSet(tasst.fixed_dtd, tasst.asst->resource_on_day, drs);
      ** ***
      HaArrayForEach(dtd->external_today, e, j)
	KheDrsExprLeafSet(e, dtd, drd, drs);
      *** **
    }
    *** **

  ** set signature and update cost **
  KheDrsSignerEvalSignature(dsg, false, prev_soln,
    next_day->open_day_index, &dsa->sig, de->solver);
  KheDrsExpanderAddCost(de, dsa->sig.cost);
  ** ***
  HaArrayForEach(ds->nr_internal_today, e, i)
    KheDrsExprEvalShiftSignature(e, prev_soln, next_day->open_day_index,
      ds, dsa->sig, drs);
  *** **

  ** pop assignments **
  ** HaArrayForEach(dsa->task_on_days, dtd, i) **
  ** *** no, done on the way out
  HaArrayForEach(dsa->assts_to_tasks, tasst, i)
    KheDrsTaskOnDayLeafClear(tasst.fixed_dtd, drs);
  *** **
      ** ***
      if( tasst.fixed_dtd != NULL )
      HaArrayForEach(dtd->external_today, e, j)
	KheDrsExprLeafClear(e, drs);
      *** **

  ** depending on cost, either add to dsat or free soln **
  ** ***
  if( prev_soln->sig.cost + dsa->sig.cost < de->solver->solve_init_cost )
  *** **
  if( KheDrsExpanderIsOpen(de) )
    KheDrsShiftAsstTrieMeldShiftAsst(dsat, dsa, dsg, KHE_DRS_DOM_TEST_UNIFORM,
      de->solver);
  else
    KheDrsShiftAsstFree(dsa, de->solver);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftAsstTrieBuildAssts(KHE_DRS_SHIFT_ASST_TRIE dsat,         */
/*    KHE_DRS_SIGNER dsg, KHE_DRS_RESOURCE_SET included_free_resources,      */
/*    int included_free_resources_index, KHE_DRS_SOLN prev_soln,             */
/*    KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de)                             */
/*                                                                           */
/*  Build all assignments for included_free_resources to the task classes    */
/*  of dsat's shift, and do dominance testing among them.  The next          */
/*  element of included_free_resources to assign is the one with index       */
/*  included_free_resources_index.                                           */
/*                                                                           */
/*****************************************************************************/
static void KheDrsExpanderMakeAndMeldShiftAsst(KHE_DRS_EXPANDER de,
  KHE_DRS_SHIFT_ASST_TRIE dsat, KHE_DRS_SHIFT ds,
  KHE_DRS_SOLN prev_soln, KHE_DRS_DAY prev_day, KHE_DRS_DAY next_day);
static void KheDrsAsstToTaskClassShiftAsstTrieBuildAssts(
  KHE_DRS_ASST_TO_TASK_CLASS asst, KHE_DRS_SHIFT_ASST_TRIE dsat,
  KHE_DRS_SHIFT ds, KHE_DRS_RESOURCE_SET included_free_resources,
  int included_free_resources_index, KHE_DRS_SOLN prev_soln,
  KHE_DRS_DAY prev_day, KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de);
static KHE_DRS_SHIFT KheDrsAsstToTaskClassShift(
  KHE_DRS_ASST_TO_TASK_CLASS asst);

static void KheDrsShiftAsstTrieBuildAssts(KHE_DRS_SHIFT_ASST_TRIE dsat,
  KHE_DRS_SHIFT ds, KHE_DRS_RESOURCE_SET included_free_resources,
  int included_free_resources_index, KHE_DRS_SOLN prev_soln,
  KHE_DRS_DAY prev_day, KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de)
  /* KHE_DYNAMIC_RESOURCE_SOLVER drs) */
{
  KHE_DRS_RESOURCE dr;  /* KHE_DRS_RESOURCE_ON_DAY drd; */  int i, indent;
  KHE_DRS_ASST_TO_TASK_CLASS asst;
  if( DEBUG52(prev_day, ds) )
  {
    indent = KheDrsResourceSetCount(included_free_resources) +
      included_free_resources_index + 1;
    fprintf(stderr, "%*s[ KheDrsShiftAsstTrieBuildAssts(...)\n", indent*2, "");
  }

  if( included_free_resources_index >=
      KheDrsResourceSetCount(included_free_resources) )
  {
    /* all included resources assigned, so build soln and add it now */
    if( DEBUG52(prev_day, ds) )
    {
      indent = KheDrsResourceSetCount(included_free_resources) +
	included_free_resources_index + 1;
      fprintf(stderr, "%*s  calling KheDrsExpanderMakeAndMeldShiftAsst\n",
	indent * 2, "");
    }
    KheDrsExpanderMakeAndMeldShiftAsst(de, dsat, ds, prev_soln, prev_day,
      next_day);
  }
  else
  {
    /* try all assignments of the next included resource */
    dr = KheDrsResourceSetResource(included_free_resources,
      included_free_resources_index);
    HaArrayForEach(dr->expand_assts, asst, i)
      if( KheDrsAsstToTaskClassShift(asst) == ds )
	KheDrsAsstToTaskClassShiftAsstTrieBuildAssts(asst, dsat, ds,
	  included_free_resources, included_free_resources_index + 1,
	  prev_soln, prev_day, next_day, de);
  }
  if( DEBUG52(prev_day, ds) )
  {
    indent = KheDrsResourceSetCount(included_free_resources) +
      included_free_resources_index + 1;
    fprintf(stderr, "%*s] KheDrsShiftAsstTrieBuildAssts returning\n",
      indent * 2, "");
  }
}

/* *** old version that calls KheDrsResourceOnDayIsFixed unnecessarily
static void KheDrsShiftAsstTrieBuildAssts(KHE_DRS_SHIFT_ASST_TRIE dsat,
  KHE_DRS_SHIFT ds, KHE_DRS_RESOURCE_SET included_resources,
  int resource_index, KHE_DRS_SOLN prev_soln, KHE_DRS_DAY next_day,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  ** KHE_DRS_RESOURCE dr; **  KHE_DRS_TASK_CLASS dtc;  int i;
  ** KHE_DRS_TASK dt; **  KHE_DRS_TASK_ON_DAY dtd;  KHE_DRS_RESOURCE_ON_DAY drd;
  ** KHE_DRS_ASST_TO_TASK_CLASS asst; **  KHE_DRS_ASST_TO_TASK tasst;
  if( resource_index >= KheDrsResourceOnDaySetCount(included_resources) )
  {
    ** all resources assigned to classes, so build soln and add it now **
    KheDrsShiftMakeEvaluateAndMeldSoln(dsat, ds, included_resources,
      prev_soln, next_day, drs);
  }
  else
  {
    ** try all task classes **
    drd = KheDrsResourceOnDaySetResourceOnDay(included_resources,
      resource_index);
    ** drd = KheDrsResourceOnDay(dr, next_day); **
    if( KheDrsResourceOnDayIsFixed(drd, prev_soln, drs, &dtd) )
    {
      if( dtd != NULL && KheDrsTaskOnDayShift(dtd) == ds &&
	  KheDrsResourceExpandFindResourceAsst(drd->encl_dr, dtd, &tasst) &&
	  KheDrsAsstToTaskIsCompatible(tasst, drs) )
      {
	** HaArrayAddLast(drs->expand_dtds, dtd); **
	** HaArrayAddLast(drs->expand_assts, asst); **
        HaArrayAddLast(drs->expand_assts_to_tasks, tasst);
	KheDrsShiftAsstTrieBuildAssts(dsat, ds, included_resources,
	  resource_index + 1, prev_soln, next_day, drs);
	HaArrayDeleteLast(drs->expand_assts_to_tasks);
	** ***
	HaArrayDeleteLast(drs->expand_dtds);
	HaArrayDeleteLast(drs->expand_assts);
	*** **
      }
    }
    ** *** moved to KheDrsResourceOnDayIsFixed
    else if( RERUN && drs->rerun != NULL )
    {
      dtd = KheDrsPackedSolnTaskOnDay(drs->rerun, next_day, dr);
      HaArrayAddLast(drs->expand_dtds, dtd);
      KheDrsShiftAsstTrieBuildAssts(dsat, ds, included_resources,
	resource_index + 1, prev_soln, next_day, drs);
      HaArrayDeleteLast(drs->expand_dtds);
    }
    *** **
    else
    {
      HaArrayForEach(ds->task_classes, dtc, i)
	if( KheDrsTaskClassAcceptResourceBegin(dtc, drd, &dtd) )
	{
	  if( KheDrsResourceExpandFindResourceAsst(drd->encl_dr, dtd, &tasst)
	      && KheDrsAsstToTaskIsCompatible(tasst, drs) )
	  {
	    ** *** moved into KheDrsTaskClassAcceptResourceBegin
	    if( !KheDrsTaskBusyOnDay(dt, next_day, &dtd) )
	      HnAbort("KheDrsShiftAsstTrieBuildAssts internal error");
	    *** **
	    ** ***
	    HaArrayAddLast(drs->expand_dtds, dtd);
	    HaArrayAddLast(drs->expand_assts, asst);
	    *** **
	    HaArrayAddLast(drs->expand_assts_to_tasks, tasst);
	    KheDrsShiftAsstTrieBuildAssts(dsat, ds, included_resources,
	      resource_index + 1, prev_soln, next_day, drs);
	    HaArrayDeleteLast(drs->expand_assts_to_tasks);
	    ** ***
	    HaArrayDeleteLast(drs->expand_dtds);
	    HaArrayDeleteLast(drs->expand_assts);
	    *** **
	  }
	  KheDrsTaskClassAcceptResourceEnd(dtc);
	}
    }
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SHIFT_ASST_TRIE KheDrsShiftAsstTrieBuild(KHE_DRS_SHIFT ds,       */
/*    KHE_DRS_SOLN prev_soln, KHE_DRS_DAY next_day,                          */
/*    KHE_DRS_RESOURCE_SET all_free_resources, int all_free_resources_index, */
/*    KHE_DRS_RESOURCE_SET included_free_resources,                          */
/*    KHE_DRS_ASST_TO_TASK_SET included_fixed_assts, KHE_DRS_EXPANDER de)    */
/*                                                                           */
/*  Build a shift assignment trie for all_free_resources starting at         */
/*  all_free_resources_index.  Here included_free_resources contains the     */
/*  resources included so far, and included_fixed_assts holds some fixed     */
/*  assignments that are also to be included.  Return NULL if there are      */
/*  no shift assignment objects in the resulting trie.                       */
/*                                                                           */
/*  Here de is a scratch expander which is reset before each use.            */
/*                                                                           */
/*****************************************************************************/
/* ***
static void KheDrsExpanderExpandBegin(KHE_DRS_EXPANDER de, KHE_COST cost_limit,
  int must_assign_count, int free_resource_count, KHE_COST cost);
static void KheDrsExpanderExpandEnd(KHE_DRS_EXPANDER de);
static KHE_DRS_EXPANDER KheDrsExpanderMake(bool whole_tasks, KHE_COST cost,
  KHE_COST cost_limit, int free_resource_count, int must_assign_count,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);
static void KheDrsExpanderFree(KHE_DRS_EXPANDER de);
*** */
static void KheDrsExpanderReset(KHE_DRS_EXPANDER de, bool whole_tasks,
  KHE_COST cost, KHE_COST cost_limit, int free_resource_count,
  int must_assign_count);
static void KheDrsAsstToTaskSetDebug(KHE_DRS_ASST_TO_TASK_SET dats,
  int verbosity, int indent, FILE *fp);
static KHE_COST KheDrsSolnCost(KHE_DRS_SOLN soln);

static KHE_DRS_SHIFT_ASST_TRIE KheDrsShiftAsstTrieBuild(KHE_DRS_SHIFT ds,
  KHE_DRS_SOLN prev_soln, KHE_DRS_DAY prev_day, KHE_DRS_DAY next_day,
  KHE_DRS_RESOURCE_SET all_free_resources, int all_free_resources_index,
  KHE_DRS_RESOURCE_SET included_free_resources,
  KHE_DRS_ASST_TO_TASK_SET included_fixed_assts, KHE_DRS_EXPANDER de)
{
  KHE_DRS_SHIFT_ASST_TRIE res, child_dsat;  bool no_non_null_children;
  int i, count, included_free_resource_count;  KHE_DRS_RESOURCE dr;

  included_free_resource_count =
    KheDrsResourceSetCount(included_free_resources);
  if( DEBUG52(prev_day, ds) )
  {
    fprintf(stderr, "%*s[ KheDrsShiftAsstTrieBuild(Shift %d, included_free ",
      2 * included_free_resource_count, "", ds->open_shift_index);
    KheDrsResourceSetDebug(included_free_resources, 1, -1, stderr);
    fprintf(stderr, ", included_fixed ");
    KheDrsAsstToTaskSetDebug(included_fixed_assts, 1, -1, stderr);
    fprintf(stderr, "\n");
  }

  /* return NULL immediately if too many included free resources */
  if(included_free_resource_count > ds->expand_max_included_free_resource_count)
  {
    if( DEBUG52(prev_day, ds) )
      fprintf(stderr, "%*s] KheDrsShiftAsstTrieBuild returning NULL\n",
	2 * included_free_resource_count, "");
    return NULL;
  }

  /* make res and add assignments for included_free_resources to res */
  res = KheDrsShiftAsstTrieMake(de->solver);
  if( included_free_resource_count >= ds->expand_must_assign_count )
  {
    KheDrsExpanderReset(de, true, KheDrsSolnCost(prev_soln),
      de->solver->solve_init_cost, included_free_resource_count,
      ds->expand_must_assign_count);
    if( DEBUG52(prev_day, ds) )
      fprintf(stderr, "%*sDrsExpanderReset(de, true, %.5f, %.5f, %d, %d) %s\n",
	2 * included_free_resource_count + 2, "",
	KheCostShow(KheDrsSolnCost(prev_soln)),
        KheCostShow(de->solver->solve_init_cost),
        included_free_resource_count, ds->expand_must_assign_count,
	KheDrsExpanderIsOpen(de) ? "open" : "closed");
    KheDrsExpanderMarkBegin(de);
    KheDrsExpanderAddAsstToTaskSet(de, included_fixed_assts, ds);
    if( KheDrsExpanderIsOpen(de) )
      KheDrsShiftAsstTrieBuildAssts(res, ds, included_free_resources,
	0, prev_soln, prev_day, next_day, de);
    KheDrsExpanderMarkEnd(de);
    /* ***
    de = KheDrsExpanderMake(true, prev_soln->sig.cost, drs->solve_init_cost,
      included_free_resource_count, ds->expand_must_assign_count, drs);
    KheDrsExpanderAddAsstToTaskSet(de, included_fixed_assts, ds);
    KheDrsShiftAsstTrieBuildAssts(res, ds->signer, included_free_resources,
      0, prev_soln, next_day, de);
    KheDrsExpanderFree(de);
    *** */
  }

  /* add a NULL child for every open resource */
  count = KheDrsResourceSetCount(de->solver->open_resources);
  HaArrayFill(res->children, count, NULL);

  /* add a potentially non-NULL child for each possible next resource */
  no_non_null_children = true;
  count = KheDrsResourceSetCount(all_free_resources);
  for( i = all_free_resources_index;  i < count;  i++ )
  {
    dr = KheDrsResourceSetResource(all_free_resources, i);
    KheDrsResourceSetAddLast(included_free_resources, dr);
    child_dsat = KheDrsShiftAsstTrieBuild(ds, prev_soln, prev_day,
      next_day, all_free_resources, i + 1, included_free_resources,
      included_fixed_assts, de);
    if( child_dsat != NULL )
    {
      HaArrayPut(res->children, dr->open_resource_index, child_dsat);
      no_non_null_children = false;
    }
    KheDrsResourceSetDeleteLast(included_free_resources);
  }

  /* replace by NULL if the tree contains no assignments */
  if( no_non_null_children && HaArrayCount(res->assts) == 0 )
  {
    KheDrsShiftAsstTrieFree(res, de->solver);
    if( DEBUG52(prev_day, ds) )
      fprintf(stderr, "%*s] KheDrsShiftAsstTrieBuild returning NULL (2)\n",
	2 * included_free_resource_count, "");
    return NULL;
  }
  else
  {
    if( DEBUG52(prev_day, ds) )
      fprintf(stderr, "%*s] KheDrsShiftAsstTrieBuild returning non-NULL\n",
	2 * included_free_resource_count, "");
    return res;
  }
}


/* ** old version, has a bug
static KHE_DRS_SHIFT_ASST_TRIE KheDrsShiftAsstTrieBuild(KHE_DRS_SHIFT ds,
  KHE_DRS_SOLN prev_soln, KHE_DRS_DAY next_day,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int resource_index,
  KHE_DRS_RESOURCE_SET included_resources)
{
  KHE_DRS_SHIFT_ASST_TRIE res, child_dsat;  int i, count, icount;
  ** KHE_DRS_RESOURCE dr; **  bool no_non_null_children;
  KHE_DRS_RESOURCE_ON_DAY drd;
  count = KheDrsResourceSetCount(drs->open_resources);
  if( resource_index >= count )
    res = NULL;
  else
  {
    res = KheDrsShiftAsstTrieMake(drs);

    ** add assignments for included_resources to res **
    icount = KheDrsResourceSetCount(included_resources);
    if( icount >= ds->expand_must_assign_count )
      KheDrsShiftAsstTrieBuildAssts(res, ds, included_resources, 0, prev_soln,
	next_day, drs);

    ** add children to res **
    no_non_null_children = true;
    for( i = 0;  i < resource_index;  i++ )
      HaArrayAddLast(res->children, NULL);
    for( i = resource_index;  i < count;  i++ )
    {
      drd = KheDrsResourceSetResource(drs->expand_drds, i);
      ** dr = KheDrsResourceSetResource(drs->open_resources, i); **
      KheDrsResourceSetAddLast(included_resources, drd);
      child_dsat = KheDrsShiftAsstTrieBuild(ds, prev_soln, next_day, drs,
	i + 1, included_resources);
      HaArrayAddLast(res->children, child_dsat);
      if( child_dsat != NULL )
	no_non_null_children = false;
      KheDrsResourceSetDeleteLast(included_resources);
    }

    ** replace by NULL if the tree contains no assignments **
    if( no_non_null_children && HaArrayCount(res->assts) == 0 )
    {
      KheDrsShiftAsstTrieFree(res, drs);
      res = NULL;
    }
  }
  return res;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SHIFT_ASST_TRIE KheDrsShiftAsstTrieGetChild(                     */
/*    KHE_DRS_SHIFT_ASST_TRIE dsat, KHE_DRS_RESOURCE dr)                     */
/*                                                                           */
/*  Get the child of dsat associated with dr.  The result may be NULL.       */
/*                                                                           */
/*****************************************************************************/

/* ***
static KHE_DRS_SHIFT_ASST_TRIE KheDrsShiftAsstTrieGetChild(
  KHE_DRS_SHIFT_ASST_TRIE dsat, KHE_DRS_RESOURCE dr)
{
  HnAssert(dsat != NULL, "KheDrsShiftAsstTrieGetChild internal error 1");
  HnAssert(dr->open_resource_index >= 0,
    "KheDrsShiftAsstTrieGetChild internal error 2");
  if( dr->open_resource_index < HaArrayCount(dsat->children) )
    return HaArray(dsat->children, dr->open_resource_index);
  else
    return NULL;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftAsstTrieExpandByShifts(KHE_DRS_SHIFT_ASST_TRIE dsat,     */
/*    KHE_DRS_SHIFT ds, int shift_index, KHE_DRS_SOLN prev_soln,             */
/*    KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de,                             */
/*    KHE_DRS_RESOURCE_SET free_resources, int free_index,                   */
/*    KHE_DRS_RESOURCE_SET omitted_resources)                                */
/*                                                                           */
/*  Expand dsat, which is non-NULL, assigning free_resources (starting       */
/*  from free_index) in all ways.  Here free_resources is not changed,       */
/*  but omitted_resources does change as the recursion proceeds; it holds    */
/*  the unused resources.                                                    */
/*                                                                           */
/*****************************************************************************/

static void KheDrsShiftAsstTrieExpandByShifts(KHE_DRS_SHIFT_ASST_TRIE dsat,
  KHE_DRS_SHIFT ds, int shift_index, KHE_DRS_SOLN prev_soln,
  KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de,
  KHE_DRS_RESOURCE_SET free_resources, int free_index,
  int selected_resource_count, KHE_DRS_RESOURCE_SET omitted_resources)
{
  int i, avail_resource_count;  KHE_DRS_SHIFT_ASST dsa;
  KHE_DRS_SHIFT_ASST_TRIE child_dsat;  KHE_DRS_RESOURCE dr;
  avail_resource_count = KheDrsResourceSetCount(free_resources) - free_index;
  if( avail_resource_count <= 0 )
  {
    /* resources all done, so try each assignment in dsat */
    HaArrayForEach(dsat->assts, dsa, i)
      KheDrsShiftAsstExpandByShifts(dsa, ds, shift_index, prev_soln,
	next_day, de, omitted_resources);
  }
  else
  {
    /* try assignments that select dr, the next available resource */
    dr = KheDrsResourceSetResource(free_resources, free_index);
    child_dsat = HaArray(dsat->children, dr->open_resource_index);
    if( child_dsat != NULL )
      KheDrsShiftAsstTrieExpandByShifts(child_dsat, ds, shift_index, prev_soln,
	next_day, de, free_resources, free_index + 1,
	selected_resource_count + 1, omitted_resources);

    /* try assignments that do not select dr, staying in dsat */
    if( selected_resource_count + avail_resource_count >
	  ds->expand_must_assign_count )
    {
      KheDrsResourceSetAddLast(omitted_resources, dr);
      KheDrsShiftAsstTrieExpandByShifts(dsat, ds, shift_index, prev_soln,
	next_day, de, free_resources, free_index + 1, selected_resource_count,
	omitted_resources);
      KheDrsResourceSetDeleteLast(omitted_resources);
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsShiftAsstTrieResourceIsForced(KHE_DRS_SHIFT_ASST_TRIE dsat,   */
/*    KHE_DRS_RESOURCE dr)                                                   */
/*                                                                           */
/*  Return true if assignment of dr to dsat is forced, that is, if every     */
/*  assignment of dsat contains dr.                                          */
/*                                                                           */
/*  Logically, the function should return true when dsat is empty, and       */
/*  indeed it does.  But in practice we are not interested in that case.     */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsShiftAsstTrieResourceIsForced(KHE_DRS_SHIFT_ASST_TRIE dsat,
  KHE_DRS_RESOURCE dr)
{
  int i;  KHE_DRS_SHIFT_ASST dsa;  KHE_DRS_SHIFT_ASST_TRIE child_dsat;

  if( dsat != NULL )
  {
    /* try the assignments */
    HaArrayForEach(dsat->assts, dsa, i)
      if( !KheDrsShiftAsstContainsResource(dsa, dr) )
	return false;

    /* try the children */
    HaArrayForEach(dsat->children, child_dsat, i)
      if( !KheDrsShiftAsstTrieResourceIsForced(child_dsat, dr) )
	return false;
  }

  /* all forced */
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftAsstTriePruneForced(KHE_DRS_SHIFT_ASST_TRIE dsat,        */
/*    KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs)                  */
/*                                                                           */
/*  Prune shift assignments that assign dr from dsat.                        */
/*                                                                           */
/*****************************************************************************/

static void KheDrsShiftAsstTriePruneForced(KHE_DRS_SHIFT_ASST_TRIE dsat,
  KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int i;  KHE_DRS_SHIFT_ASST dsa;  KHE_DRS_SHIFT_ASST_TRIE child_dsat;

  if( dsat != NULL )
  {
    /* prune all of these assignments or none */
    if( HaArrayCount(dsat->assts) > 0 )
    {
      dsa = HaArrayFirst(dsat->assts);
      if( KheDrsShiftAsstContainsResource(dsa, dr) )
      {
	/* prune them all */
	HaArrayForEach(dsat->assts, dsa, i)
	  KheDrsShiftAsstFree(dsa, drs);
	HaArrayClear(dsat->assts);
      }
    }

    /* try the children */
    HaArrayForEach(dsat->children, child_dsat, i)
      KheDrsShiftAsstTriePruneForced(child_dsat, dr, drs);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftAsstTrieDoDebug(KHE_DRS_SHIFT_ASST_TRIE dsat,            */
/*    KHE_DRS_RESOURCE_SET included_resources,                               */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)  */
/*                                                                           */
/*  Carry out the real work of KheDrsShiftAsstTrieDebug, assuming that       */
/*  indent >= 0.                                                             */
/*                                                                           */
/*****************************************************************************/

static void KheDrsShiftAsstTrieDoDebug(KHE_DRS_SHIFT_ASST_TRIE dsat,
  KHE_DRS_RESOURCE_SET included_resources,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)
{
  KHE_DRS_SHIFT_ASST dsa;  int i;  KHE_DRS_SHIFT_ASST_TRIE child_dsat;
  /* KHE_DRS_RESOURCE_ON_DAY drd; */  KHE_DRS_RESOURCE dr;
  if( dsat != NULL )
  {
    fprintf(fp, "%*s[ ", indent, "");
    KheDrsResourceSetDebug(included_resources, 1, -1, fp);
    fprintf(fp, ": {");
    HaArrayForEach(dsat->assts, dsa, i)
    {
      if( i > 0 )
	fprintf(fp, ", ");
      KheDrsShiftAsstDebug(dsa, 1, -1, fp);
    }
    fprintf(fp, "}\n");
    HaArrayForEach(dsat->children, child_dsat, i)
    {
      dr = KheDrsResourceSetResource(drs->open_resources, i);
      /* drd = KheDrsResourceSetResource(drs->expand_drds, i); */
      KheDrsResourceSetAddLast(included_resources, dr);
      KheDrsShiftAsstTrieDoDebug(child_dsat, included_resources, drs,
	verbosity, indent + 2, fp);
      KheDrsResourceSetDeleteLast(included_resources);
    }
    fprintf(fp, "%*s]\n", indent, "");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsShiftAsstTrieDebug(KHE_DRS_SHIFT_ASST_TRIE dsat,              */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)  */
/*                                                                           */
/*  Debug print of shift asst trie dsat.                                     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsShiftAsstTrieDebug(KHE_DRS_SHIFT_ASST_TRIE dsat,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)
{
  KHE_DRS_RESOURCE_SET included_resources;
  if( indent >= 0 )
  {
    included_resources = KheDrsResourceSetMake(drs);
    fprintf(fp, "%*s[ ShiftAsstTrie\n", indent, "");
    KheDrsShiftAsstTrieDoDebug(dsat, included_resources, drs, verbosity,
      indent + 2, fp);
    fprintf(fp, "%*s]\n", indent, "");
    KheDrsResourceSetFree(included_resources, drs);
  }
  else
    fprintf(fp, "ShiftAsstTrie");
}


/*****************************************************************************/
/*                                                                           */
/*  Major category "signatures and dominance"                                */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_SIGNATURE"                                            */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SIGNATURE KheDrsSignatureMake(KHE_DYNAMIC_RESOURCE_SOLVER drs,   */
/*    KHE_COST cost)                                                         */
/*                                                                           */
/*  Return a new signature object with the given cost and empty states.      */
/*  This is assumed to include a reference to the result.                    */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_SIGNATURE KheDrsSignatureMake(KHE_DYNAMIC_RESOURCE_SOLVER drs
  /* , KHE_COST cost */)
{
  KHE_DRS_SIGNATURE res;
  if( HaArrayCount(drs->signature_free_list) > 0 )
  {
    res = HaArrayLastAndDelete(drs->signature_free_list);
    HaArrayClear(res->states);
  }
  else
  {
    HaMake(res, drs->arena);
    HaArrayInit(res->states, drs->arena);
  }
  res->reference_count = 1;
  res->asst_to_shift_index = -1;  /* means none; may be reset later */
  res->cost = /* cost */ 0;
  /* HaArrayAddLast(drs->signature_used_list, res); */
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignatureInit(KHE_DRS_SIGNATURE sig, KHE_COST cost,           */
/*    HA_ARENA a)                                                            */
/*                                                                           */
/*  Initialize sig to have the given cost and empty states.  This is like    */
/*  KheDrsSignatureMake except that sig's memory, in *sig, is allocated but  */
/*  not initialized.                                                         */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsSignatureInit(KHE_DRS_SIGNATURE sig, KHE_COST cost,
  HA_ARENA a)
{
  sig->cost = cost;
  HaArrayInit(sig->states, a);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignatureClear(KHE_DRS_SIGNATURE sig, KHE_COST cost)          */
/*                                                                           */
/*  Clear sig back to the given cost and an empty array of states.           */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsSignatureClear(KHE_DRS_SIGNATURE sig ** , KHE_COST cost **)
{
  sig->cost = ** cost ** 0;
  HaArrayClear(sig->states);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignatureFree(KHE_DRS_SIGNATURE sig,                          */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Free sig.                                                                */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsSignatureFree(KHE_DRS_SIGNATURE sig,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  HaArrayAddLast(drs->signature_free_list, sig);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignatureRefer(KHE_DRS_SIGNATURE sig)                         */
/*                                                                           */
/*  Add a reference to sig.                                                  */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignatureRefer(KHE_DRS_SIGNATURE sig)
{
  sig->reference_count++;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignatureUnRefer(KHE_DRS_SIGNATURE sig,                       */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Delete a reference to sig; free it if the reference count drops to 0.    */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignatureUnRefer(KHE_DRS_SIGNATURE sig,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  HnAssert(sig->reference_count > 0, "KheDrsSignatureUnRefer internal error");
  if( --sig->reference_count == 0 )
    HaArrayAddLast(drs->signature_free_list, sig);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignatureAddAsstToShiftIndex(KHE_DRS_SIGNATURE sig, int val)  */
/*                                                                           */
/*  Add an assignment to shift index with value val to sig.                  */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignatureAddAsstToShiftIndex(KHE_DRS_SIGNATURE sig, int val)
{
  HnAssert(sig->asst_to_shift_index == -1,
    "KheDrsSignatureAddAsstToShiftIndex internal error");
  sig->asst_to_shift_index = val;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignatureClearAsstToShiftIndex(KHE_DRS_SIGNATURE sig)         */
/*                                                                           */
/*  Clear out the assignment to shift index of sig.                          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignatureClearAsstToShiftIndex(KHE_DRS_SIGNATURE sig)
{
  HnAssert(sig->asst_to_shift_index >= 0,
    "KheDrsSignatureClearAsstToShiftIndex internal error");
  sig->asst_to_shift_index = -1;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignatureFreeAll(KHE_DYNAMIC_RESOURCE_SOLVER drs)             */
/*                                                                           */
/*  Free all signatures, by moving them from the used list to the free list. */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsSignatureFreeAll(KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int i;
  HaArrayAppend(drs->signature_free_list, drs->signature_used_list, i);
  HaArrayClear(drs->signature_used_list);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignatureAddCost(KHE_DRS_SIGNATURE sig, KHE_COST cost)        */
/*                                                                           */
/*  Add cost to sig.                                                         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignatureAddCost(KHE_DRS_SIGNATURE sig, KHE_COST cost)
{
  sig->cost += cost;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignatureAddState(KHE_DRS_SIGNATURE sig, int state,           */
/*    KHE_DRS_SIGNER dsg, KHE_DRS_EXPR e)                                    */
/*                                                                           */
/*  Add state to the end of sig.  Here dsg and e are just for checking.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignatureAddState(KHE_DRS_SIGNATURE sig, int state,
  KHE_DRS_SIGNER dsg, KHE_DRS_EXPR e)
{
  KHE_DRS_DOM_TEST dt;
  dt = HaArray(dsg->dom_tests, HaArrayCount(sig->states));
  if( dt->monitor != NULL && DEBUG44_MONITOR(dt->monitor) )
  {
    fprintf(stderr, "[ KheDrsSignatureAddState(%p, pos %d, state %d), e =\n",
      (void *) sig, HaArrayCount(sig->states), state);
    KheDrsExprDebug(e, NULL, 2, 2, stderr);
    fprintf(stderr, "]\n");
  }
  HaArrayAddLast(sig->states, state);
  HnAssert(HaArrayCount(sig->states) <= HaArrayCount(dsg->dom_tests),
    "KheDrsSignatureAddState internal error 1:  %d states > %d dom_tests",
    HaArrayCount(sig->states), HaArrayCount(dsg->dom_tests));
  if( DEBUG63 && dt->expr != e )
  {
    fprintf(stderr, "[ KheDrsSignatureAddState(%p) failing on expressions\n",
      (void *) sig);
    fprintf(stderr, "  calling expression:\n");
    KheDrsExprDebug(e, NULL, 2, 2, stderr);
    fprintf(stderr, "  dom_test expression:\n");
    KheDrsExprDebug(dt->expr, NULL, 2, 2, stderr);
    fprintf(stderr, "]\n");
  }
  if( DEBUG63 && state == 4 )
  {
    fprintf(stderr, "[ KheDrsSignatureAddState(%p) reporting state == 4\n",
      (void *) sig);
    KheDrsExprDebug(e, NULL, 2, 2, stderr);
    fprintf(stderr, "]\n");
  }
  HnAssert(dt->expr == e, "KheDrsSignatureAddState internal error 2");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignatureAppendStates(KHE_DRS_SIGNATURE to_sig,               */
/*    KHE_DRS_SIGNATURE from_sig)                                            */
/*                                                                           */
/*  Append the states of from_sig to the end of to_sig.                      */
/*                                                                           */
/*****************************************************************************/

/* *** no longer used, which is good
static void KheDrsSignatureAppendStates(KHE_DRS_SIGNATURE to_sig,
  KHE_DRS_SIGNATURE from_sig)
{
  int i;
  HaArrayAppend(to_sig->states, from_sig->states, i);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSignerDominates(KHE_DRS_SIGNATURE sig1,                       */
/*    KHE_DRS_SIGNATURE sig2, KHE_DRS_DAY day, int start,                    */
/*    KHE_COST *avail_cost)                                                  */
/*                                                                           */
/*  Return true if sig1 dominates sig2 on day.  If a cost tradeoff is        */
/*  used to justify this, reduce *avail_cost by the amount used.             */
/*                                                                           */
/*****************************************************************************/

/* *** obsolete
static bool KheDrsSignerDominates(KHE_DRS_SIGNATURE sig1,
  KHE_DRS_SIGNATURE sig2, KHE_DRS_DAY day, int start, KHE_COST *avail_cost)
{
  KHE_DRS_DOM_TEST dt;  int i, v1, v2, sig_len;
  if( *avail_cost < 0 )
    return false;
  sig_len = HaArrayCount(sig1->sig);
  for( i = start;  i < sig_len;  i++ )
  {
    dt = HaArray(day->dom_tests, i);
    v1 = HaArray(sig1->sig, i);
    v2 = HaArray(sig2->sig, i);
    if( !KheDrsDomTestDominatesNew(dt, v1, v2, avail_cost) )
      return false;
  }
  return true;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsSignatureFullHash(KHE_DRS_SIGNATURE sig)                       */
/*                                                                           */
/*  Hash function for hashing all of sig.                                    */
/*                                                                           */
/*****************************************************************************/

/* *** replaced by KheDrsSignatureSetFullHash
static int KheDrsSignatureFullHash(KHE_DRS_SIGNATURE sig)
{
  int res, val, i;
  res = 0;
  HaArrayForEach(sig->states, val, i)
    res = (res + val) * 3;
  return (res < 0 ? - res : res);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSignatureFullEqual(KHE_DRS_SIGNATURE sig1,                    */
/*    KHE_DRS_SIGNATURE sig2)                                                */
/*                                                                           */
/*  Return true if sig1 and sig2 are all equal.                              */
/*                                                                           */
/*****************************************************************************/

/* *** replaced by KheDrsSignatureSetFullEqual
static bool KheDrsSignatureFullEqual(KHE_DRS_SIGNATURE sig1,
  KHE_DRS_SIGNATURE sig2)
{
  int i;
  HnAssert(HaArrayCount(sig1->states) == HaArrayCount(sig2->states),
    "KheDrsSignatureFullEqual: signatures have different lengths");
  for( i = 0;  i < HaArrayCount(sig1->states);  i++ )
    if( HaArray(sig1->states, i) != HaArray(sig2->states, i) )
      return false;
  return true;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignatureDebug(KHE_DRS_SIGNATURE sig, int verbosity,          */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of sig onto fp with the given verbosity and indent.          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignatureDebug(KHE_DRS_SIGNATURE sig, int verbosity,
  int indent, FILE *fp)
{
  int val, i;
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  if( HaArrayCount(sig->states) > 0 )
  {
    fprintf(fp, "<%.5f:", KheCostShow(sig->cost));
    HaArrayForEach(sig->states, val, i)
      fprintf(fp, " %d", val);
    fprintf(fp, ">");
  }
  else
    fprintf(fp, "<%.5f>", KheCostShow(sig->cost));
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_SIGNATURE_SET"                                        */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignatureSetInit(KHE_DRS_SIGNATURE_SET sig_set,               */
/*    KHE_COST cost, HA_ARENA a)                                             */
/*                                                                           */
/*  Initialize sig_set to have the given cost and no signatures.             */
/*                                                                           */
/*  There is no KheDrsSignatureSetMake, because signature sets are used      */
/*  only in solution objects, and those signature sets are expanded, hence   */
/*  "Init" (which assumes that memory is already allocated), not "Make".     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignatureSetInit(KHE_DRS_SIGNATURE_SET sig_set,
  KHE_COST cost, HA_ARENA a)
{
  sig_set->cost = cost;
  HaArrayInit(sig_set->signatures, a);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignatureSetClear(KHE_DRS_SIGNATURE_SET sig_set,              */
/*    KHE_COST cost, KHE_DYNAMIC_RESOURCE_SOLVER drs)                        */
/*                                                                           */
/*  Clear sig_set back to the given cost and no signatures.                  */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignatureSetClear(KHE_DRS_SIGNATURE_SET sig_set,
  KHE_COST cost, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SIGNATURE sig;
  sig_set->cost = cost;
  while( HaArrayCount(sig_set->signatures) > 0 )
  {
    sig = HaArrayLastAndDelete(sig_set->signatures);
    KheDrsSignatureUnRefer(sig, drs);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsSignatureSetFullHash(KHE_DRS_SIGNATURE_SET sig_set)            */
/*                                                                           */
/*  Hash function for hashing all of sig_set.                                */
/*                                                                           */
/*****************************************************************************/

static int KheDrsSignatureSetFullHash(KHE_DRS_SIGNATURE_SET sig_set)
{
  int res, val, i, j;  KHE_DRS_SIGNATURE sig;
  res = 0;
  HaArrayForEach(sig_set->signatures, sig, i)
    HaArrayForEach(sig->states, val, j)
      res = (res + val) * 3;
  return (res < 0 ? - res : res);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSignatureFullEqual(KHE_DRS_SIGNATURE sig1,                    */
/*    KHE_DRS_SIGNATURE sig2)                                                */
/*                                                                           */
/*  Return true if sig1 and sig2 are all equal.                              */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsSignatureSetFullEqual(KHE_DRS_SIGNATURE_SET sig_set1,
  KHE_DRS_SIGNATURE_SET sig_set2)
{
  int i, j;  KHE_DRS_SIGNATURE sig1, sig2;
  HnAssert(HaArrayCount(sig_set1->signatures) ==
    HaArrayCount(sig_set2->signatures),
    "KheDrsSignatureFullEqual internal error (1)");
  for( i = 0;  i < HaArrayCount(sig_set1->signatures);  i++ )
  {
    sig1 = HaArray(sig_set1->signatures, i);
    sig2 = HaArray(sig_set2->signatures, i);
    HnAssert(HaArrayCount(sig1->states) == HaArrayCount(sig2->states),
      "KheDrsSignatureFullEqual: signatures have different lengths");
    for( j = 0;  j < HaArrayCount(sig1->states);  j++ )
      if( HaArray(sig1->states, j) != HaArray(sig2->states, j) )
	return false;
  }
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignatureSetAddSignature(KHE_DRS_SIGNATURE_SET sig_set,       */
/*    KHE_DRS_SIGNATURE sig, bool with_cost)                                 */
/*                                                                           */
/*  Add sig to sig_set, but only include its cost if with_cost is true.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignatureSetAddSignature(KHE_DRS_SIGNATURE_SET sig_set,
  KHE_DRS_SIGNATURE sig, bool with_cost)
{
  if( with_cost )
    sig_set->cost += sig->cost;
  HaArrayAddLast(sig_set->signatures, sig);
  KheDrsSignatureRefer(sig);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignatureSetDebug(KHE_DRS_SIGNATURE_SET sig_set,              */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of sig_set onto fp with the given verbosity and indent.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignatureSetDebug(KHE_DRS_SIGNATURE_SET sig_set,
  int verbosity, int indent, FILE *fp)
{
  int i;  KHE_DRS_SIGNATURE sig;
  if( indent >= 0 )
  {
    fprintf(fp, "%*s<%.5f:\n", indent, "", KheCostShow(sig_set->cost));
    HaArrayForEach(sig_set->signatures, sig, i)
      KheDrsSignatureDebug(sig, verbosity, indent + 2, fp);
    fprintf(fp, "%*s>\n", indent, "");
  }
  else
    fprintf(fp, "<sig_set %.5f>", KheCostShow(sig_set->cost));
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_CORRELATOR" - handle correlated expressions           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_CORRELATOR KheDrsCorrelatorMake(KHE_DYNAMIC_RESOURCE_SOLVER drs) */
/*                                                                           */
/*  Make a new, empty correlator object.                                     */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_CORRELATOR KheDrsCorrelatorMake(KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_CORRELATOR res;  HA_ARENA a;
  a = drs->arena;
  HaMake(res, a);
  res->make_correlated = false;		/* reset when solving */
  HaArrayInit(res->children, a);
  HaArrayInit(res->positives, a);
  HaArrayInit(res->negatives, a);
  HaArrayInit(res->singles, a);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsCorrelatorClear(KHE_DRS_CORRELATOR dc)                        */
/*                                                                           */
/*  Clear dc.                                                                */
/*                                                                           */
/*****************************************************************************/

static void KheDrsCorrelatorClear(KHE_DRS_CORRELATOR dc)
{
  dc->make_correlated = false;
  HaArrayClear(dc->children);
  HaArrayClear(dc->positives);
  HaArrayClear(dc->negatives);
  HaArrayClear(dc->singles);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsCorrelatorMakeCorrelated(KHE_DRS_CORRELATOR dc)               */
/*                                                                           */
/*  Inform dc that correlated expressions are wanted.                        */
/*                                                                           */
/*****************************************************************************/

static void KheDrsCorrelatorMakeCorrelated(KHE_DRS_CORRELATOR dc)
{
  dc->make_correlated = true;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsCorr1Debug(KHE_DRS_DOM_TEST prnt_dt,                          */
/*    KHE_DRS_DOM_TEST child_dt, KHE_DRS_SIGNER dsg,                         */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)  */
/*                                                                           */
/*  Debug print of corr1 (prnt_dt, child_dt).                                */
/*                                                                           */
/*****************************************************************************/
static char *KheDrsSignerId(KHE_DRS_SIGNER dsg);

static void KheDrsCorr1Debug(KHE_DRS_DOM_TEST prnt_dt,
  KHE_DRS_DOM_TEST child_dt, KHE_DRS_SIGNER dsg,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)
{
  if( DEBUG55 )
  {
    fprintf(fp, "%*s[ signer %s found corr1 (delta %d):\n", indent, "",
      KheDrsSignerId(dsg), prnt_dt->correlated_delta);
    KheDrsExprDebug(prnt_dt->expr, drs, verbosity, indent + 2, fp);
    KheDrsExprDebug(child_dt->expr, drs, verbosity, indent + 2, fp);
    fprintf(fp, "%*s]\n", indent, "");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsCorr3Debug(KHE_DRS_DOM_TEST dt, int dt_index,                 */
/*    KHE_DRS_SIGNER dsg, KHE_DYNAMIC_RESOURCE_SOLVER drs,                   */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of corr3, beginning at dt.                                   */
/*                                                                           */
/*****************************************************************************/

static void KheDrsCorr3Debug(KHE_DRS_DOM_TEST dt, int dt_index,
  KHE_DRS_SIGNER dsg, KHE_DYNAMIC_RESOURCE_SOLVER drs,
  int verbosity, int indent, FILE *fp)
{
  if( DEBUG55 )
  {
    fprintf(fp, "%*s[ signer %s found corr3:\n", indent, "",
      KheDrsSignerId(dsg));
    do
    {
      KheDrsExprDebug(dt->expr, drs, verbosity, indent + 2, fp);
      dt_index += dt->correlated_delta;
      dt = HaArray(dsg->dom_tests, dt_index);
    } while( dt->type != KHE_DRS_DOM_TEST_CORR3_LAST );
    KheDrsExprDebug(dt->expr, drs, verbosity, indent + 2, fp);
    fprintf(fp, "%*s]\n", indent, "");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsCorr4Debug(KHE_DRS_DOM_TEST dt, int dt_index,                 */
/*    KHE_DRS_SIGNER dsg, KHE_DYNAMIC_RESOURCE_SOLVER drs,                   */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of corr4, beginning at dt.                                   */
/*                                                                           */
/*****************************************************************************/

static void KheDrsCorr4Debug(KHE_DRS_DOM_TEST dt, int dt_index,
  KHE_DRS_SIGNER dsg, KHE_DYNAMIC_RESOURCE_SOLVER drs,
  int verbosity, int indent, FILE *fp)
{
  if( DEBUG55 )
  {
    fprintf(fp, "%*s[ signer %s found corr4:\n", indent, "",
      KheDrsSignerId(dsg));
    do
    {
      KheDrsExprDebug(dt->expr, drs, verbosity, indent + 2, fp);
      dt_index += dt->correlated_delta;
      dt = HaArray(dsg->dom_tests, dt_index);
    } while( dt->type != KHE_DRS_DOM_TEST_CORR4_LAST );
    KheDrsExprDebug(dt->expr, drs, verbosity, indent + 2, fp);
    fprintf(fp, "%*s]\n", indent, "");
  }
}


/*****************************************************************************/
/*                                                                           */
/*   void KheDrsCorrelatorAddChildDomTest(KHE_DRS_CORRELATOR dc,             */
/*     KHE_DRS_DOM_TEST child_dt, int child_dt_index, KHE_DRS_SIGNER dsg,    */
/*     KHE_DYNAMIC_RESOURCE_SOLVER drs)                                      */
/*                                                                           */
/*  Carry out KheDrsCorrelatorAddDomTest for the case where child_dt->expr   */
/*  is an OR expression which is the child of a KHE_DRS_EXPR_INT_SUM_COST    */
/*  expression.  The child always arrives before its parent.                 */
/*                                                                           */
/*****************************************************************************/

static void KheDrsCorrelatorAddChildDomTest(KHE_DRS_CORRELATOR dc,
  KHE_DRS_DOM_TEST child_dt, int child_dt_index, KHE_DRS_SIGNER dsg,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  HnAssert(child_dt->corr_dom_table4 != NULL,
    "KheDrsCorrelatorAddChildDomTest internal error");
  child_dt->type = KHE_DRS_DOM_TEST_CORR2_CHILD;
  HaArrayAddLast(dc->children, child_dt_index);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsCorrelatorAddParentDomTest(KHE_DRS_CORRELATOR dc,             */
/*    KHE_DRS_DOM_TEST prnt_dt, int prnt_dt_index, KHE_DRS_SIGNER dsg,       */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Carry out the work of KheDrsCorrelatorAddDomTest for the case where      */
/*  prnt_dt->expr is an KHE_DRS_EXPR_INT_SUM_COST:  try to match it with     */
/*  a child dt, which must have arrived previously.                          */
/*                                                                           */
/*****************************************************************************/
static bool KheDrsParentIsFirstIntSumCostParent(KHE_DRS_EXPR child,
  KHE_DRS_EXPR parent);

static void KheDrsCorrelatorAddParentDomTest(KHE_DRS_CORRELATOR dc,
  KHE_DRS_DOM_TEST prnt_dt, int prnt_dt_index, KHE_DRS_SIGNER dsg,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DOM_TEST child_dt;  int child_dt_index, i;
  HaArrayForEach(dc->children, child_dt_index, i)
  {
    child_dt = HaArray(dsg->dom_tests, child_dt_index);
    if( KheDrsParentIsFirstIntSumCostParent(child_dt->expr, prnt_dt->expr) )
    {
      prnt_dt->type = KHE_DRS_DOM_TEST_CORR1_PARENT;
      child_dt->type = KHE_DRS_DOM_TEST_CORR1_CHILD;
      child_dt->correlated_delta = prnt_dt_index - child_dt_index;
      HaArrayDeleteAndPlug(dc->children, i);
      KheDrsCorr1Debug(prnt_dt, child_dt, dsg, drs, 1, 0, stderr);
      return;
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsCorrelatorAddDaysPositiveDomTest(KHE_DRS_CORRELATOR dc,       */
/*    KHE_DRS_DOM_TEST pos_dt, int pos_dt_index, KHE_DRS_SIGNER dsg,         */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Carry out the work of KheDrsCorrelatorAddDomTest for the case where      */
/*  dt->expr is a days-positive KHE_DRS_EXPR_INT_SEQ_COST.                   */
/*                                                                           */
/*****************************************************************************/

static void KheDrsCorrelatorAddDaysPositiveDomTest(KHE_DRS_CORRELATOR dc,
  KHE_DRS_DOM_TEST pos_dt, int pos_dt_index, KHE_DRS_SIGNER dsg,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DOM_TEST neg_dt;  int neg_dt_index;
  if( HaArrayCount(dc->negatives) > 0 )
  {
    /* make a matching pair with any existing negative */
    neg_dt_index = HaArrayLastAndDelete(dc->negatives);
    neg_dt = HaArray(dsg->dom_tests, neg_dt_index);
    pos_dt->type = KHE_DRS_DOM_TEST_CORR4_FIRST;
    pos_dt->correlated_delta = neg_dt_index - pos_dt_index;
    neg_dt->type = KHE_DRS_DOM_TEST_CORR4_LAST;
    KheDrsCorr4Debug(pos_dt, pos_dt_index, dsg, drs, 1, 2, stderr);
  }
  else
  {
    /* store for a potential future match */
    HaArrayAddLast(dc->positives, pos_dt_index);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsCorrelatorAddDaysNegativeDomTest(KHE_DRS_CORRELATOR dc,       */
/*    KHE_DRS_DOM_TEST neg_dt, int neg_dt_index, KHE_DRS_SIGNER dsg,         */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Carry out the work of KheDrsCorrelatorAddDomTest for the case where      */
/*  dt->expr is a days-negative KHE_DRS_EXPR_INT_SEQ_COST.                   */
/*                                                                           */
/*****************************************************************************/

static void KheDrsCorrelatorAddDaysNegativeDomTest(KHE_DRS_CORRELATOR dc,
  KHE_DRS_DOM_TEST neg_dt, int neg_dt_index, KHE_DRS_SIGNER dsg,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DOM_TEST pos_dt;  int pos_dt_index;
  if( HaArrayCount(dc->positives) > 0 )
  {
    /* make a matching pair with any existing positive */
    pos_dt_index = HaArrayLastAndDelete(dc->positives);
    pos_dt = HaArray(dsg->dom_tests, pos_dt_index);
    pos_dt->type = KHE_DRS_DOM_TEST_CORR4_FIRST;
    pos_dt->correlated_delta = neg_dt_index - pos_dt_index;
    neg_dt->type = KHE_DRS_DOM_TEST_CORR4_LAST;
    KheDrsCorr4Debug(pos_dt, pos_dt_index, dsg, drs, 1, 2, stderr);
  }
  else
  {
    /* store for a potential future match */
    HaArrayAddLast(dc->negatives, neg_dt_index);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsDomTestSingleSeqIndexesEqual(KHE_DRS_DOM_TEST dt1,            */
/*    KHE_DRS_DOM_TEST dt2)                                                  */
/*                                                                           */
/*  Return true if the indexes of singles dt1 and dt2 are equal.             */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsDomTestSingleSeqIndexesEqual(KHE_DRS_DOM_TEST dt1,
  KHE_DRS_DOM_TEST dt2)
{
  KHE_DRS_EXPR_INT_SEQ_COST eisc1, eisc2;
  eisc1 = (KHE_DRS_EXPR_INT_SEQ_COST) dt1->expr;
  eisc2 = (KHE_DRS_EXPR_INT_SEQ_COST) dt2->expr;
  return eisc1->seq_index == eisc2->seq_index;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsCorrelatorSingleAdd(KHE_DRS_DOM_TEST single_dt,               */
/*    int single_dt_index, KHE_DRS_DOM_TEST new_single_dt,                   */
/*    int new_single_dt_index, KHE_DRS_SIGNER dsg,                           */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Try to add new_single_dt (at new_single_dt_index) to the existing        */
/*  singles sequence beginning with single_dt (at single_dt_index).  If      */
/*  successful, update the existing singles sequence and return true.        */
/*  Otherwise change nothing and return false.                               */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsCorrelatorSingleAdd(KHE_DRS_DOM_TEST single_dt,
  int single_dt_index, KHE_DRS_DOM_TEST new_single_dt,
  int new_single_dt_index, KHE_DRS_SIGNER dsg,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DOM_TEST save_single_dt;  int save_single_dt_index;
  switch( single_dt->type )
  {
    case KHE_DRS_DOM_TEST_UNUSED:
    case KHE_DRS_DOM_TEST_CORR1_PARENT:
    case KHE_DRS_DOM_TEST_CORR1_CHILD:
    case KHE_DRS_DOM_TEST_CORR2_CHILD:

      /* should never happen */
      HnAbort("KheDrsCorrelatorSingleAdd internal error 1");
      return false;  /* keep compiler happy */

    case KHE_DRS_DOM_TEST_STRONG:
    case KHE_DRS_DOM_TEST_TRADEOFF:
    case KHE_DRS_DOM_TEST_UNIFORM:

      /* test and stop; if successful, result has two elements */
      if( KheDrsDomTestSingleSeqIndexesEqual(single_dt, new_single_dt) )
	return false;
      else
      {
        single_dt->type = KHE_DRS_DOM_TEST_CORR3_FIRST;
        single_dt->correlated_delta = new_single_dt_index - single_dt_index;
	new_single_dt->type = KHE_DRS_DOM_TEST_CORR3_LAST;
	KheDrsCorr3Debug(single_dt, single_dt_index, dsg, drs, 1, 2, stderr);
	return true;
      }

    case KHE_DRS_DOM_TEST_CORR3_FIRST:

      save_single_dt = single_dt;
      save_single_dt_index = single_dt_index;
      do
      {
	/* check new_single_dt against single_dt */
	if( KheDrsDomTestSingleSeqIndexesEqual(single_dt, new_single_dt) )
	  return false;

	/* move on to the next single_dt */
	single_dt_index += single_dt->correlated_delta;
	single_dt = HaArray(dsg->dom_tests, single_dt_index);
      } while( single_dt->type == KHE_DRS_DOM_TEST_CORR3_MID );

      /* check last, and if successful, add new_single_dt to the end */
      HnAssert(single_dt->type == KHE_DRS_DOM_TEST_CORR3_LAST,
        "KheDrsCorrelatorSingleAdd internal error 2");
      if( KheDrsDomTestSingleSeqIndexesEqual(single_dt, new_single_dt) )
	return false;
      else
      {
	single_dt->type = KHE_DRS_DOM_TEST_CORR3_MID;
	single_dt->correlated_delta = new_single_dt_index - single_dt_index;
	new_single_dt->type = KHE_DRS_DOM_TEST_CORR3_LAST;
	KheDrsCorr3Debug(save_single_dt, save_single_dt_index, dsg, drs,
	  1, 2, stderr);
	return true;
      }

    default:

      HnAbort("KheDrsCorrelatorSingleAdd internal error 2");
      return false;  /* keep compiler happy */
  }

  /* unreachable here */
  HnAbort("KheDrsCorrelatorSingleAdd internal error 3");
  return false;  /* keep compiler happy */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsCorrelatorAddSinglePositiveDomTest(KHE_DRS_CORRELATOR dc,     */
/*    KHE_DRS_DOM_TEST new_single_dt, int new_single_dt_index,               */
/*    KHE_DRS_SIGNER dsg, KHE_DYNAMIC_RESOURCE_SOLVER drs)                   */
/*                                                                           */
/*  Carry out the work of KheDrsCorrelatorAddDomTest for the case where      */
/*  new_single_dt->expr is a single-time positive KHE_DRS_EXPR_INT_SEQ_COST. */
/*                                                                           */
/*****************************************************************************/

static void KheDrsCorrelatorAddSinglePositiveDomTest(KHE_DRS_CORRELATOR dc,
  KHE_DRS_DOM_TEST new_single_dt, int new_single_dt_index,
  KHE_DRS_SIGNER dsg, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int single_dt_index, i;  KHE_DRS_DOM_TEST single_dt;

  /* try adding to each singles sequence, and return if successful */
  HaArrayForEach(dc->singles, single_dt_index, i)
  {
    single_dt = HaArray(dsg->dom_tests, single_dt_index);
    if( KheDrsCorrelatorSingleAdd(single_dt, single_dt_index,
	  new_single_dt, new_single_dt_index, dsg, drs) )
      return;
  }

  /* no match so save new_single_dt for later */
  HaArrayAddLast(dc->singles, new_single_dt_index);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsCorrelatorAddDomTest(KHE_DRS_CORRELATOR dc,                   */
/*    KHE_DRS_DOM_TEST dt, int dt_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)    */
/*                                                                           */
/*  Add dt to dc.                                                            */
/*                                                                           */
/*****************************************************************************/
static bool KheDrsExprHasParent(KHE_DRS_EXPR child, KHE_DRS_EXPR_TAG tag);

static void KheDrsCorrelatorAddDomTest(KHE_DRS_CORRELATOR dc,
  KHE_DRS_DOM_TEST dt, int dt_index, KHE_DRS_SIGNER dsg,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_INT_SEQ_COST eisc;
  if( dc->make_correlated )
  {
    if( DEBUG55 )
      fprintf(stderr, "[ KheDrsCorrelatorAddDomTest(dc, dt, %d, %s, drs)\n",
	dt_index, KheDrsSignerId(dsg));
    switch( dt->expr->tag )
    {
      case KHE_DRS_EXPR_OR_TAG:

        if( KheDrsExprHasParent(dt->expr, KHE_DRS_EXPR_INT_SUM_COST_TAG) )
	  KheDrsCorrelatorAddChildDomTest(dc, dt, dt_index, dsg, drs);
	break;

      case KHE_DRS_EXPR_INT_SUM_COST_TAG:

        KheDrsCorrelatorAddParentDomTest(dc, dt, dt_index, dsg, drs);
	break;

      case KHE_DRS_EXPR_INT_SEQ_COST_TAG:

	eisc = (KHE_DRS_EXPR_INT_SEQ_COST) dt->expr;
	if( eisc->seq_type == KHE_DRS_SEQ_DAYS_POSITIVE )
	  KheDrsCorrelatorAddDaysPositiveDomTest(dc, dt, dt_index, dsg, drs);
	else if( eisc->seq_type == KHE_DRS_SEQ_DAYS_NEGATIVE )
	  KheDrsCorrelatorAddDaysNegativeDomTest(dc, dt, dt_index, dsg, drs);
	else if( eisc->seq_type == KHE_DRS_SEQ_SINGLE_POSITIVE )
	  KheDrsCorrelatorAddSinglePositiveDomTest(dc, dt, dt_index, dsg, drs);
	break;

      default:

	/* ignore others */
	break;
    }
    if( DEBUG55 )
      fprintf(stderr, "] KheDrsCorrelatorAddDomTest returning\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsCorrelatorOpeningHasEnded(KHE_DRS_CORRELATOR dc,              */
/*    KHE_DRS_SIGNER dsg)                                                    */
/*                                                                           */
/*  Inform dc that opening has ended.  It is time therefore to carry out     */
/*  the correlations recorded in dc on the dom tests of dsg.                 */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsCorrelatorOpeningHasEnded(KHE_DRS_CORRELATOR dc,
  KHE_DRS_SIGNER dsg)
{
  if( dc->make_correlated )
  {
    ** carry out the correlations **
    ** st ill to do **

    ** clear dc ready for next time **
    KheDrsCorrelatorClear(dc);
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_SIGNER"                                               */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SIGNER KheDrsSignerMake(KHE_DRS_DAY day,                         */
/*    KHE_DRS_RESOURCE_ON_DAY drd, KHE_DRS_SHIFT ds,                         */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new, empty signer object for day, drd, or ds.                     */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_SIGNER KheDrsSignerMake(KHE_DRS_DAY day,
  KHE_DRS_RESOURCE_ON_DAY drd, KHE_DRS_SHIFT ds,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SIGNER res;

  /* get res and make sure its arrays are initialized and empty */
  if( HaArrayCount(drs->signer_free_list) > 0 )
  {
    res = HaArrayLastAndDelete(drs->signer_free_list);
    HaArrayClear(res->internal_exprs);
    HaArrayClear(res->dom_tests);
    HaArrayClear(res->eq_dom_test_indexes);
  }
  else
  {
    HaMake(res, drs->arena);
    HaArrayInit(res->internal_exprs, drs->arena);
    HaArrayInit(res->dom_tests, drs->arena);
    HaArrayInit(res->eq_dom_test_indexes, drs->arena);
  }

  /* get a correlator, but only if drd type (never freed, by the way) */
  res->correlator = (drd != NULL ? KheDrsCorrelatorMake(drs) : NULL);

  /* set the other fields */
  res->last_hard_cost_index = -1;
  res->debug_eval_if_rerun = false;  /* placeholder */
  if( day != NULL )
  {
    HnAssert(drd == NULL && ds == NULL, "KheDrsSignerMake internal error 1");
    res->type = KHE_DRS_SIGNER_DAY;
    res->u.day = day;
  }
  else if( drd != NULL )
  {
    HnAssert(day == NULL && ds == NULL, "KheDrsSignerMake internal error 2");
    res->type = KHE_DRS_SIGNER_RESOURCE_ON_DAY;
    res->u.resource_on_day = drd;
  }
  else if( ds != NULL )
  {
    HnAssert(day == NULL && drd == NULL,"KheDrsSignerMake internal error 3");
    res->type = KHE_DRS_SIGNER_SHIFT;
    res->u.shift = ds;
  }
  else
    HnAbort("KheDrsSignerMake internal error 4");

  /* all good, return */
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignerMakeCorrelated(KHE_DRS_SIGNER dsg)                      */
/*                                                                           */
/*  Inform dsg that correlated expressions are wanted.                       */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignerMakeCorrelated(KHE_DRS_SIGNER dsg)
{
  HnAssert(dsg->correlator != NULL,
    "KheDrsSignerMakeCorrelated internal error");
  KheDrsCorrelatorMakeCorrelated(dsg->correlator);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignerFree(KHE_DRS_SIGNER dsg,                                */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Free dsg.                                                                */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignerFree(KHE_DRS_SIGNER dsg, 
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  HnAssert(dsg->correlator == NULL, "KheDrsSignerFree internal error");
  HaArrayAddLast(drs->signer_free_list, dsg);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignerClear(KHE_DRS_SIGNER dsg, bool free_dom_tests,          */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Clear dsg ready for a new solve, without changing its type.  This        */
/*  includes freeing any dom tests left over from the previous solve.        */
/*                                                                           */
/*  Obsolete:  Also free                                                     */
/*  dsg's dom tests, unless dsg->type == KHE_DRS_SIGNER_RESOURCE_ON_DAY, in  */
/*  which case the dom tests are shared with the enclosing day signer, and   */
/*  that signer will free them.                                              */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignerClear(KHE_DRS_SIGNER dsg,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int i;
  /* if( dsg->type != KHE_DRS_SIGNER_RESOURCE_ON_DAY ) */
  HaArrayClear(dsg->internal_exprs);
  HaArrayAppend(drs->dom_test_free_list, dsg->dom_tests, i);
  HaArrayClear(dsg->dom_tests);
  HaArrayClear(dsg->eq_dom_test_indexes);
  dsg->last_hard_cost_index = -1;
  dsg->debug_eval_if_rerun = false;
  if( dsg->correlator != NULL )
    KheDrsCorrelatorClear(dsg->correlator);
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheDrsSignerId(KHE_DRS_SIGNER dsg)                                 */
/*                                                                           */
/*  Return the Id if dsg.                                                    */
/*                                                                           */
/*****************************************************************************/

static char *KheDrsSignerId(KHE_DRS_SIGNER dsg)
{
  switch( dsg->type )
  {
    case KHE_DRS_SIGNER_DAY:

      return KheDrsDayId(dsg->u.day);

    case KHE_DRS_SIGNER_RESOURCE_ON_DAY:

      return KheDrsResourceOnDayId(dsg->u.resource_on_day);

    case KHE_DRS_SIGNER_SHIFT:

      return KheDrsShiftId(dsg->u.shift);

    default:

      return "?";
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignerAddOpenExpr(KHE_DRS_SIGNER dsg, KHE_DRS_EXPR e)         */
/*                                                                           */
/*  Add e to dsg.                                                            */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignerAddOpenExpr(KHE_DRS_SIGNER dsg, KHE_DRS_EXPR e)
{
  if( dsg->type == KHE_DRS_SIGNER_SHIFT )
    HnAssert(e->tag == KHE_DRS_EXPR_INT_SUM_COST_TAG,
      "KheDrsSignerAddOpenExpr internal error");
  HaArrayAddLast(dsg->internal_exprs, e);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignerDebugCorrelatedPair(KHE_DRS_DOM_TEST dt1,               */
/*    KHE_DRS_DOM_TEST dt2, KHE_DRS_SIGNER dsg,                              */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)  */
/*                                                                           */
/*  Debug print of correlated pair (dt1, dt2).                               */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsSignerDebugCorrelatedPair(KHE_DRS_DOM_TEST dt1,
  KHE_DRS_DOM_TEST dt2, KHE_DRS_SIGNER dsg,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)
{
  if( DEBUG53 )
  {
    fprintf(fp, "%*s[ signer %s found correlated pair (delta %d):\n",
      indent, "", KheDrsSignerId(dsg), dt1->correlated_delta);
    KheDrsExprDebug(dt1->expr, drs, verbosity, indent + 2, fp);
    KheDrsExprDebug(dt2->expr, drs, verbosity, indent + 2, fp);
    fprintf(fp, "%*s]\n", indent, "");
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignerFoundCorrelatedPair(KHE_DRS_SIGNER dsg,                 */
/*    KHE_DRS_DOM_TEST dt1, int dt1_index, KHE_DRS_DOM_TEST dt2,             */
/*    int dt2_index, KHE_DRS_DOM_TEST_TYPE dt1_new_type, int i,              */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Do this when a correlated pair (dt1, dt2) is found.  Set dt1->type to    */
/*  dt1_new_type, dt2->type to dt2_new_type, and dt1->correlated_delta to    */
/*  the amount to add to dt1's index to get dt2's index.  Delete element i   */
/*  of dsg->potential_matches.                                               */
/*                                                                           */
/*****************************************************************************/

/* *** no longer used, because pairs are no longer special
static void KheDrsSignerFoundCorrelatedPair(KHE_DRS_SIGNER dsg,
  KHE_DRS_DOM_TEST dt1, int dt1_index, KHE_DRS_DOM_TEST dt2,
  int dt2_index, KHE_DRS_DOM_TEST_TYPE dt1_new_type,
  KHE_DRS_DOM_TEST_TYPE dt2_new_type, int i,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  dt1->type = dt1_new_type;
  dt2->type = dt2_new_type;
  dt1->correlated_delta = dt2_index - dt1_index;
  HaArrayDeleteAndPlug(dsg->potential_matches, i);
  KheDrsSignerDebugCorrelatedPair(dt1, dt2, dsg, drs, 2, 0, stderr);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignerHandleCorrelatedTests(KHE_DRS_SIGNER dsg,               */
/*    KHE_DRS_DOM_TEST dt2, int dt2_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)  */
/*                                                                           */
/*  Dominance test dt2 (with index dt2_index) has just arrived.  Decide      */
/*  whether it is the second element of a correlated pair, or else could     */
/*  be the first element, and update things appropriately.                   */
/*                                                                           */
/*  If dt2 is the second element of a correlated pair, change the types of   */
/*  the two dom tests, and set correlated_delta in the non-SKIP dom test     */
/*  to the amount to add to its index to get to the SKIP dom test.  Also,    */
/*  delete the dom test that matches with dt2 from dsg->potential_matches.   */
/*                                                                           */
/*  Otherwise, if dt2 could be the first element of a correlated pair, add   */
/*  it to dsg->potential_matches where its second element can find it later. */
/*                                                                           */
/*****************************************************************************/

/* *** replaced by KheDrsCorrelatorAddDomTest
static void KheDrsSignerHandleCorrelatedTests(KHE_DRS_SIGNER dsg,
  KHE_DRS_DOM_TEST dt2, int dt2_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int dt1_index, i;  KHE_DRS_DOM_TEST dt1;
  KHE_DRS_EXPR_INT_SEQ_COST eisc1, eisc2;
  if( DEBUG55 )
    fprintf(stderr, "[ KheDrsSignerHandleCorrelatedTests(dsg, dt2, %d, drs)\n",
      dt2_index);
  if( dt2->expr->tag == KHE_DRS_EXPR_INT_SUM_COST_TAG )
  {
    ** dt2 is INT_SUM_COST, dt1 must be its child **
    HaArrayForEach(dsg->potential_matches, dt1_index, i)
    {
      dt1 = HaArray(dsg->dom_tests, dt1_index);
      if( KheDrsParentIsFirstIntSumCostParent(dt1->expr, dt2->expr) )
      {
	KheDrsSignerFoundCorrelatedPair(dsg, dt2, dt2_index, dt1, dt1_index,
	  KHE_DRS_DOM_TEST_CORR1_FIRST, KHE_DRS_DOM_TEST_CORR1_LAST, i, drs);
	if( DEBUG55 )
	  fprintf(stderr, "] KheDrsSignerHandleCorrelatedTests success\n");
	return;
      }
    }

    ** dt2 is not the second element of a pair, and it cannot be the first **
  }
  else if( dt2->expr->tag == KHE_DRS_EXPR_INT_SEQ_COST_TAG )
  {
    eisc2 = (KHE_DRS_EXPR_INT_SEQ_COST) dt2->expr;
    if( eisc2->seq_type == KHE_DRS_SEQ_DAYS_POSITIVE )
    {
      ** dt2 is positive INT_SEQ_COST, dt1 must be negative INT_SEQ_COST **
      HaArrayForEach(dsg->potential_matches, dt1_index, i)
      {
	dt1 = HaArray(dsg->dom_tests, dt1_index);
	if( dt1->expr->tag == KHE_DRS_EXPR_INT_SEQ_COST_TAG )
	{
	  eisc1 = (KHE_DRS_EXPR_INT_SEQ_COST) dt1->expr;
	  if( eisc1->seq_type == KHE_DRS_SEQ_DAYS_NEGATIVE )
	  {
	    KheDrsSignerFoundCorrelatedPair(dsg, dt2, dt2_index, 
	      dt1, dt1_index, KHE_DRS_DOM_TEST_CORR3_FIRST, 
	      KHE_DRS_DOM_TEST_CORR3_LAST, i, drs);
	    if( DEBUG55 )
	      fprintf(stderr, "] KheDrsSignerHandleCorrelatedTests success\n");
	    return;
	  }
	}
      }

      ** dt2 is not the second element of a pair, but it could be the first **
      HaArrayAddLast(dsg->potential_matches, dt2_index);
    }
    else if( eisc2->seq_type == KHE_DRS_SEQ_DAYS_NEGATIVE )
    {
      ** dt2 is negative INT_SEQ_COST, dt1 must be positive INT_SEQ_COST **
      HaArrayForEach(dsg->potential_matches, dt1_index, i)
      {
	dt1 = HaArray(dsg->dom_tests, dt1_index);
	if( dt1->expr->tag == KHE_DRS_EXPR_INT_SEQ_COST_TAG )
	{
	  eisc1 = (KHE_DRS_EXPR_INT_SEQ_COST) dt1->expr;
	  if( eisc1->seq_type == KHE_DRS_SEQ_DAYS_POSITIVE )
	  {
	    KheDrsSignerFoundCorrelatedPair(dsg, dt1, dt1_index, 
	      dt2, dt2_index, KHE_DRS_DOM_TEST_CORR3_FIRST,
	      KHE_DRS_DOM_TEST_CORR3_LAST, i, drs);
	    if( DEBUG55 )
	      fprintf(stderr, "] KheDrsSignerHandleCorrelatedTests success\n");
	    return;
	  }
	}
      }

      ** dt2 is not the second element of a pair, but it could be the first **
      HaArrayAddLast(dsg->potential_matches, dt2_index);
    }
  }
  else if( KheDrsExprHasParent(dt2->expr, KHE_DRS_EXPR_INT_SUM_COST_TAG) )
  {
    ** dt2 is not the second element of a pair, but it could be the first **
    HaArrayAddLast(dsg->potential_matches, dt2_index);
  }
  if( DEBUG55 )
    fprintf(stderr, "] KheDrsSignerHandleCorrelatedTests fail\n");
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsSignerAddDomTest(KHE_DRS_SIGNER dsg,                           */
/*    KHE_DRS_DOM_TEST dom_test)                                             */
/*                                                                           */
/*  Add dom_test to dsg and return its index in dsg.                         */
/*                                                                           */
/*  If dsg->make_correlated is set, identify pairs of correlated dom         */
/*  tests and adjust them appropriately.                                     */
/*                                                                           */
/*****************************************************************************/
static bool KheDrsDomTestIsStrictEquality(KHE_DRS_DOM_TEST dom_test);
static KHE_COST KheDrsMonitorWeight(KHE_MONITOR m);

static int KheDrsSignerAddDomTest(KHE_DRS_SIGNER dsg,
  KHE_DRS_DOM_TEST dt, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int dt_index;

  /* add dt */
  dt_index = HaArrayCount(dsg->dom_tests);
  HaArrayAddLast(dsg->dom_tests, dt);

  /* if dt is equality, record it in eq_dom_test_indexes */
  if( KheDrsDomTestIsStrictEquality(dt) )
    HaArrayAddLast(dsg->eq_dom_test_indexes, dt_index);

  /* handle correlated expressions, if wanted */
  if( dsg->correlator != NULL )
    KheDrsCorrelatorAddDomTest(dsg->correlator, dt, dt_index, dsg, drs);
    /* KheDrsSignerHandleCorrelatedTests(dsg, dt, dt_index, drs); */

  /* keep track of where hard constraints end */
  if( KheDrsMonitorWeight(dt->monitor) >= KheCost(1, 0) )
    dsg->last_hard_cost_index = dt_index;

  /* return dt's index */
  return dt_index;

  /* *** old version of correlated expressions
  if( dsg->make_correlated )
  {
    count = HaArrayCount(dsg->dom_tests);
    if( count >= 2 )
    {
      dt1 = HaArray(dsg->dom_tests, count - 2);
      dt2 = HaArray(dsg->dom_tests, count - 1);
      if( dt2->expr->tag == KHE_DRS_EXPR_INT_SUM_COST_TAG )
      {
	if( KheDrsParentIsFirstIntSumCostParent(dt1->expr, dt2->expr) )
	{
	  if( DEBUG53 )
	  {
	    fprintf(stderr, "[ signer found correlated (1) expressions:\n");
	    KheDrsExprDebug(dt1->expr, NULL, 1, 2, stderr);
	    KheDrsExprDebug(dt2->expr, NULL, 1, 2, stderr);
	    fprintf(stderr, "]\n");
	  }
	  dt1->type = KHE_DRS_DOM_TEST_CORR1_LAST;
	  dt2->type = KHE_DRS_DOM_TEST_CORR1_FIRST;
	}
      }
      else if( dt1->expr->tag == KHE_DRS_EXPR_INT_SEQ_COST_TAG )
      {
	if( dt2->expr->tag == KHE_DRS_EXPR_INT_SEQ_COST_TAG )
	{
	  eisc1 = (KHE_DRS_EXPR_INT_SEQ_COST) dt1->expr;
	  eisc2 = (KHE_DRS_EXPR_INT_SEQ_COST) dt2->expr;
	  if( eisc1->seq_type == KHE_DRS_SEQ_DAYS_POSITIVE &&
	      eisc2->seq_type == KHE_DRS_SEQ_DAYS_NEGATIVE )
	  {
	    if( DEBUG53 )
	    {
	      fprintf(stderr, "[ signer found correlated (2) expressions:\n");
	      KheDrsExprDebug(dt1->expr, NULL, 1, 2, stderr);
	      KheDrsExprDebug(dt2->expr, NULL, 1, 2, stderr);
	      fprintf(stderr, "]\n");
	    }
	    ** *** not doing this until we can make use of it
	    dt1->type = KHE_DRS_DOM_TEST_CORR3_LAST;
	    dt2->type = KHE_DRS_DOM_TEST_CORR3_FIRST;
	    *** **
	  }
	}
      }
    }
  }
  *** */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignerOpeningHasEnded(KHE_DRS_SIGNER dsg)                     */
/*                                                                           */
/*  Inform dsg that opening has ended.                                       */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsSignerOpeningHasEnded(KHE_DRS_SIGNER dsg)
{
  if( dsg->correlator != NULL )
    KheDrsCorrelatorOpeningHasEnded(dsg->correlator, dsg);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsSignerPartialHashSignature(KHE_DRS_SIGNER dsg,                 */
/*    KHE_DRS_SIGNATURE sig)                                                 */
/*                                                                           */
/*  Hash sig, but only the positions that have equality dom tests.           */
/*                                                                           */
/*****************************************************************************/

/* *** replaced by KheDrsSignerSetPartialHashSignature
static int KheDrsSignerPartialHashSignature(KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE sig)
{
  int res, index, val, i;
  res = 0;
  HaArrayForEach(dsg->eq_dom_test_indexes, index, i)
  {
    val = HaArray(sig->states, index);
    res = (res + val) * 3;
  }
  if( res < 0 )
    res = - res;
  return res;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSignerPartialEqualSignature(KHE_DRS_SIGNER dsg,               */
/*    KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2)                        */
/*                                                                           */
/*  Return true if sig1 and sig2 are equal at the positions that have        */
/*  equality dom tests.                                                      */
/*                                                                           */
/*****************************************************************************/

/* *** replaced by KheDrsSignerSetPartialEqualSignature
static bool KheDrsSignerPartialEqualSignature(KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2)
{
  int i, index;
  HnAssert(HaArrayCount(sig1->states) == HaArrayCount(sig2->states),
   "KheDrsSignerPartialEqualSignature: signatures have different lengths");
  HaArrayForEach(dsg->eq_dom_test_indexes, index, i)
    if( HaArray(sig1->states, index) != HaArray(sig2->states, index) )
      return false;
  return true;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignerEvalSignature(KHE_DRS_SIGNER dsg,                       */
/*    bool debug_if_rerun, KHE_DRS_SIGNATURE prev_sig, int next_di,          */
/*    KHE_DRS_SIGNATURE next_sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)           */
/*                                                                           */
/*  Using dsg, build next_sig, not necessarily from scratch.                 */
/*                                                                           */
/*****************************************************************************/
static void KheDrsExprEvalSignature(KHE_DRS_EXPR e, KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE prev_sig, int next_di, KHE_DRS_SIGNATURE next_sig,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, bool debug);

static void KheDrsSignerEvalSignature(KHE_DRS_SIGNER dsg,
  bool debug_if_rerun, KHE_DRS_SIGNATURE prev_sig, int next_di,
  KHE_DRS_SIGNATURE next_sig, KHE_DYNAMIC_RESOURCE_SOLVER drs, bool debug)
{
  KHE_DRS_EXPR e;  int i;
  dsg->debug_eval_if_rerun = debug_if_rerun;
  HaArrayForEach(dsg->internal_exprs, e, i)
    KheDrsExprEvalSignature(e, dsg, prev_sig, next_di, next_sig, drs, debug);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSignerDominates(KHE_DRS_SIGNER dsg,                           */
/*    KHE_DRS_DOM_TEST_TYPE dom_test_type, KHE_DRS_SIGNATURE sig1,           */
/*    KHE_DRS_SIGNATURE sig2, KHE_COST tradeoff_extra_cost,                  */
/*    int start_depth, KHE_COST *uniform_avail_cost)                         */
/*                                                                           */
/*  Return true if sig1 dominates sig2, including comparing their costs.     */
/*  Parameter dom_test_type says what kind of dominance testing to use:      */
/*                                                                           */
/*    KHE_DRS_DOM_TEST_STRONG - strong dominance                             */
/*                                                                           */
/*    KHE_DRS_DOM_TEST_TRADEOFF - tradeoff dominance.  In this case,         */
/*      tradeoff_extra_cost is the extra cost already paid by sig1.          */
/*                                                                           */
/*    KHE_DRS_DOM_TEST_UNIFORM - uniform dominance.  In this case,           */
/*      *uniform_avail_cost is the available cost before testing these       */
/*      signatures, and it is reset to the available cost afterwards.        */
/*                                                                           */
/*  Parameter start_depth says how far along to start.                       */
/*                                                                           */
/*****************************************************************************/
/* ***
static bool KheDrsDomTestDominatesStrong(KHE_DRS_DOM_TEST dt, int val1,
  int val2);
static bool KheDrsDomTestDominatesTradeoff(KHE_DRS_DOM_TEST dt,
  int val1, int val2, KHE_COST *avail_cost);
static bool KheDrsDomTestDominatesUniform(KHE_DRS_DOM_TEST dt,
  int val1, int val2, KHE_COST *avail_cost);
*** */
/* ***
static bool KheDrsDomTestDominatesCorr3(KHE_DRS_DOM_TEST dt,
  int v1, int v2, int other_v1, int other_v2, KHE_COST *avail_cost);
*** */

/* *** replaced by KheDrsSignerDominatesWithDebug;  I got sick of coordinating them
static bool KheDrsSignerDominates(KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2,
  int start_depth, KHE_COST *avail_cost)
{
  KHE_DRS_DOM_TEST dt;  int dt_index, v1, v2, sig_len;

  *avail_cost += sig2->cost - sig1->cost;
  if( *avail_cost < 0 )
    return false;
  sig_len = HaArrayCount(dsg->dom_tests);
  HnAssert(HaArrayCount(sig1->states) == sig_len,
    "KheDrsSignerDominates internal error 1 (count %d != count %d)\n",
    HaArrayCount(sig1->states), sig_len);
  HnAssert(HaArrayCount(sig2->states) == sig_len,
    "KheDrsSignerDominates internal error 2 (count %d != count %d)\n",
    HaArrayCount(sig2->states), sig_len);
  for( dt_index = start_depth;  dt_index < sig_len;  dt_index++ )
  {
    dt = HaArray(dsg->dom_tests, dt_index);
    v1 = HaArray(sig1->states, dt_index);
    v2 = HaArray(sig2->states, dt_index);
    switch( dt->type )
    {
      case KHE_DRS_DOM_TEST_UNUSED:

	HnAbort("internal error in KheDrsSignerDominates (UNUSED)");
	break;

      case KHE_DRS_DOM_TEST_STRONG:

	if( !KheDrsDomTestDominatesStrong(dt, v1, v2) )
	  return false;
	break;

      case KHE_DRS_DOM_TEST_TRADEOFF:

	if( !KheDrsDomTestDominatesTradeoff(dt, v1, v2, avail_cost) )
	  return false;
	break;

      case KHE_DRS_DOM_TEST_UNIFORM:

	if( !KheDrsDomTestDominatesUniform(dt, v1, v2, avail_cost) )
	  return false;
	break;

      case KHE_DRS_DOM_TEST_CORR1_PARENT:

	** do nothing here, KHE_DRS_DOM_TEST_CORR1_CHILD does it all **
	break;

	** parent element of a parent/child pair; do it here **
	** wrong day, surely?  fixing this is sti ll to do **
	** ***
	if( !KheDrsSignerCorr1Dominates(dsg, sig1, sig2, dt, dt_index, v1, v2,
	      avail_cost) )
	  return false;
	*** **
	** ***
	other_i = i + dt->correlated_delta;
	other_dt = HaArray(dsg->dom_tests, other_i);
        HnAssert(other_dt->type == KHE_DRS_DOM_TEST_CORR1_CHILD,
          "KheDrsSignerDominates internal error (other_i = %d)", other_i);
        a1 = HaArray(sig1->states, other_i);
	a2 = HaArray(sig2->states, other_i);
	if( !KheDrsDomTestDominatesCorr1(dt, a1, v1, a2, v2, avail_cost) )
	  return false;
	break;
	*** **

      case KHE_DRS_DOM_TEST_CORR1_CHILD:

	if( !KheDrsSignerCorr1Dominates(dsg, sig1, sig2, dt, dt_index, v1, v2,
	      avail_cost) )
	break;

      case KHE_DRS_DOM_TEST_CORR1_CHILD:

	if( !KheDrsSignerCorr2Dominates(dsg, sig1, sig2, dt, dt_index, v1, v2,
	      avail_cost) )
	break;

      case KHE_DRS_DOM_TEST_CORR3_FIRST:

	** first element of a set of two or more corr3 dom tests **
        if( !KheDrsSignerCorr3Dominates(dsg, sig1, sig2, dt, dt_index, v1, v2,
	    avail_cost) )
	  return false;
	break;

      case KHE_DRS_DOM_TEST_CORR3_MID:
      case KHE_DRS_DOM_TEST_CORR3_LAST:

	** non-first element of a set of two or more corr3 dom tests **
	** do nothing here, KHE_DRS_DOM_TEST_CORR3_FIRST does it all **
	break;

      case KHE_DRS_DOM_TEST_CORR4_FIRST:

	** first element of a set of two or more corr4 dom tests **
        if( !KheDrsSignerCorr4Dominates(dsg, sig1, sig2, dt, dt_index, v1, v2,
	    avail_cost) )
	  return false;
	break;

      case KHE_DRS_DOM_TEST_CORR4_MID:
      case KHE_DRS_DOM_TEST_CORR4_LAST:

	** non-first element of a set of two or more corr4 dom tests **
	** do nothing here, KHE_DRS_DOM_TEST_CORR4_FIRST does it all **
	break;

      default:

	HnAbort("internal error in KheDrsSignerDominates (%d)", dt->type);
	break;
    }
  }
  return true;
}
*** */


/* *** old version
static bool KheDrsSignerDominates(KHE_DRS_SIGNER dsg,
  KHE_DRS_DOM_TEST_TYPE dom_test_type, KHE_DRS_SIGNATURE sig1,
  KHE_DRS_SIGNATURE sig2, KHE_COST tradeoff_extra_cost,
  int start_depth, KHE_COST *uniform_avail_cost)
{
  KHE_DRS_DOM_TEST dt;  int i, v1, v2, sig_len;  KHE_COST cost1;
  switch( dom_test_type )
  {
    case KHE_DRS_DOM_TEST_STRONG:

      if( sig2->cost < sig1->cost )
	return false;
      sig_len = HaArrayCount(sig1->states);
      for( i = start_depth;  i < sig_len;  i++ )
      {
	dt = HaArray(dsg->dom_tests, i);
	v1 = HaArray(sig1->states, i);
	v2 = HaArray(sig2->states, i);
	if( !KheDrsDomTestDominatesStrong(dt, v1, v2) )
	  return false;
      }
      break;

    case KHE_DRS_DOM_TEST_TRADEOFF:
 
      cost1 = sig1->cost + tradeoff_extra_cost;
      if( sig2->cost < cost1 )
	return false;
      sig_len = HaArrayCount(sig1->states);
      for( i = start_depth;  i < sig_len;  i++ )
      {
	dt = HaArray(dsg->dom_tests, i);
	v1 = HaArray(sig1->states, i);
	v2 = HaArray(sig2->states, i);
	if( !KheDrsDomTestDominatesTradeoff(dt, v1, &cost1, v2, sig2->cost))
	  return false;
      }
      break;

    case KHE_DRS_DOM_TEST_UNIFORM:

      *uniform_avail_cost += sig2->cost - sig1->cost;
      if( *uniform_avail_cost < 0 )
	return false;
      sig_len = HaArrayCount(sig1->states);
      HnAssert(HaArrayCount(dsg->dom_tests) == HaArrayCount(sig1->states),
	"KheDrsSignerDominates internal error 1 (count %d != count %d)\n",
        HaArrayCount(dsg->dom_tests), HaArrayCount(sig1->states));
      HnAssert(HaArrayCount(sig1->states) == HaArrayCount(sig2->states),
	"KheDrsSignerDominates internal error 2 (count %d != count %d)\n",
        HaArrayCount(sig1->states), HaArrayCount(sig2->states));
      for( i = start_depth;  i < sig_len;  i++ )
      {
	dt = HaArray(dsg->dom_tests, i);
	v1 = HaArray(sig1->states, i);
	v2 = HaArray(sig2->states, i);
        if( !KheDrsDomTestDominatesUniform(dt, v1, v2, uniform_avail_cost) )
	  return false;
      }
      break;

    default:

      HnAbort("KheDrsSignerDominates internal error (dom_test_type %d)",
        dom_test_type);
      break;
  }
  return true;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDominanceDebugBegin(int indent, FILE *fp)                     */
/*                                                                           */
/*  Begin a dominance check debug.                                           */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsDominanceDebugBegin(int indent, FILE *fp)
{
  ** fprintf(fp, "%*s%5s %8s  %s\n", indent, "", "Res", "Avail", "Details"); **
  fprintf(fp, "%*s%5s-%8s--%s\n", indent, "", "-----", "--------",
    "------------------------------------------------");
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDominanceDebugEnd(int indent, FILE *fp)                       */
/*                                                                           */
/*  End a dominance check debug.                                             */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsDominanceDebugEnd(int indent, FILE *fp)
{
  fprintf(fp, "%*s%5s-%8s--%s\n", indent, "", "-----", "--------",
    "------------------------------------------------");
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDominanceDebugSpacer(int indent, FILE *fp)                    */
/*                                                                           */
/*  Print a spacer line in a debug print of a dominance test.                */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDominanceDebugSpacer(int indent, FILE *fp)
{
  if( fp != NULL )
    fprintf(fp, "%*s------------------------------------------\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDominanceDebugLine(KHE_COST avail_cost, char *details,        */
/*    KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2, int indent, FILE *fp)  */
/*                                                                           */
/*  Print of one line of a dominance debug, but only if fp != NULL.          */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsDominanceDebugLine(KHE_COST avail_cost, char *details,
  KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2, int indent, FILE *fp)
{
  if( fp != NULL )
    fprintf(fp, "%*s%8.5f  %s (sig1 = %p, sig2 = %p)\n", indent, "",
      KheCostShow(avail_cost), details, (void *) sig1, (void *) sig2);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsPrintLimits(char *tag, bool allow_zero, int min_lim,          */
/*    int max_lim, KHE_COST combined_wt, KHE_COST_FUNCTION cost_fn, FILE *fp)*/
/*                                                                           */
/*  Print all this stuff in an intuitive format.                             */
/*                                                                           */
/*****************************************************************************/

static void KheDrsPrintLimits(char *tag, bool allow_zero, int min_lim,
  int max_lim, KHE_COST combined_wt, KHE_COST_FUNCTION cost_fn, FILE *fp)
{
  char buff[200], *buff2;  int soft_wt, hard_wt;

  /* print the initial tag */
  fprintf(fp, "%-4s ", tag);

  /* print allow_zero, min_lim, and max_lim if required */
  if( min_lim >= 0 )
  {
    /* sort out allow_zero flag */
    if( allow_zero && min_lim <= 1 )
      allow_zero = false, min_lim = 0;

    /* print min_lim and max_lim, where present */
    if( min_lim > 0 )
    {
      if( max_lim < INT_MAX )
      {
	/* have min_lim and max_lim */
	if( min_lim == max_lim )
	  sprintf(buff, "%s%d", allow_zero ? "0," : "", min_lim);
	else
	  sprintf(buff, "%s%d-%d", allow_zero ? "0," : "", min_lim, max_lim);
      }
      else
      {
	/* have min_lim only */
	sprintf(buff, "%s%d-inf", allow_zero ? "0," : "", min_lim);
      }
    }
    else
    {
      if( max_lim < INT_MAX )
      {
	/* have max_lim only */
	sprintf(buff, "%s0-%d", allow_zero ? "0," : "", max_lim);
      }
      else
      {
	/* have no limits at all */
	sprintf(buff, "%s-", allow_zero ? "0," : "");
      }
    }
  }
  else
    sprintf(buff, "%s", "");

  /* print bar and the the soft and hard weight */
  buff2 = &buff[strlen(buff)];
  soft_wt = KheSoftCost(combined_wt);
  hard_wt = KheHardCost(combined_wt);
  if( soft_wt > 0 )
  {
    if( hard_wt > 0 )
    {
      /* soft and hard weight */
      sprintf(buff2, "|h%d+s%d", hard_wt, soft_wt);
    }
    else
    {
      /* soft weight only */
      sprintf(buff2, "|s%d", soft_wt);
    }
  }
  else
  {
    if( hard_wt > 0 )
    {
      /* hard weight only */
      sprintf(buff2, "|h%d", hard_wt);
    }
    else
    {
      /* no weight at all */
      sprintf(buff2, "|0");
    }
  }

  /* print the cost function */
  buff2 = &buff[strlen(buff)];
  switch( cost_fn )
  {
    case KHE_STEP_COST_FUNCTION:

      sprintf(buff2, "s");
      break;

    case KHE_LINEAR_COST_FUNCTION:

      /* default, don't print anything */
      break;

    case KHE_QUADRATIC_COST_FUNCTION:

      sprintf(buff2, "q");
      break;

    default:

      sprintf(buff2, "?");
  }

  /* print it all out, fixed width */
  fprintf(fp, "%-8s", buff);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsMonitorDebug(KHE_MONITOR m, FILE *fp)                         */
/*                                                                           */
/*  Debug print of m with a few key attributes.  Alternatively, if m is      */
/*  NULL, print nothing.                                                     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsMonitorDebug(KHE_MONITOR m, FILE *fp)
{
  KHE_CONSTRAINT c;  KHE_COST wt;  KHE_COST_FUNCTION cost_fn;
  int min_lim, max_lim;  bool allow_zero;
  KHE_CLUSTER_BUSY_TIMES_CONSTRAINT cbtc;
  KHE_LIMIT_BUSY_TIMES_CONSTRAINT lbtc;
  KHE_LIMIT_WORKLOAD_CONSTRAINT lwc;
  KHE_LIMIT_ACTIVE_INTERVALS_CONSTRAINT laic;
  KHE_LIMIT_RESOURCES_CONSTRAINT lrc;
  if( m != NULL )
  {
    c = KheMonitorConstraint(m);
    wt = KheConstraintCombinedWeight(c);
    cost_fn = KheConstraintCostFunction(c);
    switch( KheConstraintTag(c) )
    {
      case KHE_ASSIGN_RESOURCE_CONSTRAINT_TAG:

	KheDrsPrintLimits("arc", false, -1, -1, wt, cost_fn, fp);
	break;

      case KHE_PREFER_RESOURCES_CONSTRAINT_TAG:

	KheDrsPrintLimits("prc", false, -1, -1, wt, cost_fn, fp);
	break;

      case KHE_AVOID_SPLIT_ASSIGNMENTS_CONSTRAINT_TAG:

	KheDrsPrintLimits("asac", false, -1, -1, wt, cost_fn, fp);
	break;

      case KHE_AVOID_CLASHES_CONSTRAINT_TAG:

	KheDrsPrintLimits("acc", false, -1, -1, wt, cost_fn, fp);
	break;

      case KHE_AVOID_UNAVAILABLE_TIMES_CONSTRAINT_TAG:

	KheDrsPrintLimits("autc", false, -1, -1, wt, cost_fn, fp);
	break;

      case KHE_LIMIT_IDLE_TIMES_CONSTRAINT_TAG:

	KheDrsPrintLimits("litc", false, -1, -1, wt, cost_fn, fp);
	break;

      case KHE_CLUSTER_BUSY_TIMES_CONSTRAINT_TAG:

	cbtc = (KHE_CLUSTER_BUSY_TIMES_CONSTRAINT) c;
	min_lim = KheClusterBusyTimesConstraintMinimum(cbtc);
	max_lim = KheClusterBusyTimesConstraintMaximum(cbtc);
	allow_zero = KheClusterBusyTimesConstraintAllowZero(cbtc);
	KheDrsPrintLimits("cbtc", allow_zero, min_lim, max_lim, wt, cost_fn,fp);
	break;

      case KHE_LIMIT_BUSY_TIMES_CONSTRAINT_TAG:

	lbtc = (KHE_LIMIT_BUSY_TIMES_CONSTRAINT) c;
	min_lim = KheLimitBusyTimesConstraintMinimum(lbtc);
	max_lim = KheLimitBusyTimesConstraintMaximum(lbtc);
	allow_zero = KheLimitBusyTimesConstraintAllowZero(lbtc);
	KheDrsPrintLimits("lbtc", allow_zero, min_lim, max_lim, wt, cost_fn,fp);
	break;

      case KHE_LIMIT_WORKLOAD_CONSTRAINT_TAG:

	lwc = (KHE_LIMIT_WORKLOAD_CONSTRAINT) c;
	min_lim = KheLimitWorkloadConstraintMinimum(lwc);
	max_lim = KheLimitWorkloadConstraintMaximum(lwc);
	allow_zero = KheLimitWorkloadConstraintAllowZero(lwc);
	KheDrsPrintLimits("lwc", allow_zero, min_lim, max_lim, wt, cost_fn, fp);
	break;

      case KHE_LIMIT_ACTIVE_INTERVALS_CONSTRAINT_TAG:

	laic = (KHE_LIMIT_ACTIVE_INTERVALS_CONSTRAINT) c;
	min_lim = KheLimitActiveIntervalsConstraintMinimum(laic);
	max_lim = KheLimitActiveIntervalsConstraintMaximum(laic);
	KheDrsPrintLimits("laic", false, min_lim, max_lim, wt, cost_fn, fp);
	break;

      case KHE_LIMIT_RESOURCES_CONSTRAINT_TAG:

	lrc = (KHE_LIMIT_RESOURCES_CONSTRAINT) c;
	min_lim = KheLimitResourcesConstraintMinimum(lrc);
	max_lim = KheLimitResourcesConstraintMaximum(lrc);
	KheDrsPrintLimits("lrc", false, min_lim, max_lim, wt, cost_fn, fp);
	break;

      default:

	fprintf(fp, "? %.5f \n", KheCostShow(wt));
	break;
    }
    fprintf(fp, " %s", KheMonitorId(m));
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDominanceDebugCmpLine(bool res, KHE_COST avail_cost,          */
/*    KHE_DRS_DOM_TEST_TYPE type, int i, int v1, int v2, KHE_MONITOR m,      */
/*    char *details, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Print of one line of a dominance debug, but only if fp != NULL.          */
/*                                                                           */
/*****************************************************************************/
static char *KheDrsDomTestTypeShow(KHE_DRS_DOM_TEST_TYPE type);

static void KheDrsDominanceDebugCmpLine(KHE_COST avail_cost,
  KHE_DRS_DOM_TEST_TYPE type, int i, int v1, int v2, KHE_MONITOR m,
  int indent, FILE *fp)
{
  if( fp != NULL )
  {
    fprintf(fp, "%*s%8.5f  %2s[%2d] %2d %2d", indent, "",
      KheCostShow(avail_cost), KheDrsDomTestTypeShow(type), i, v1, v2);
    if( m != NULL )
    {
      fprintf(fp, "  ");
      KheDrsMonitorDebug(m, fp);
    }
    fprintf(fp, "\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSignerCorr1Dominates(KHE_DRS_SIGNER dsg,                      */
/*    KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2, KHE_DRS_DOM_TEST dt,   */
/*    int dt_index, int v1, int v2, KHE_COST *avail_cost)                    */
/*                                                                           */
/*  Carry out a CORR1 dominance test whose child element is dt.              */
/*                                                                           */
/*****************************************************************************/
/* ***
static bool KheDrsDomTestDominatesCorr1(KHE_DRS_DOM_TEST dt,
  int a1, int b1, int a2, int b2, KHE_COST *avail_cost);

static bool KheDrsSignerCorr1Dominates(KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2, KHE_DRS_DOM_TEST dt,
  int dt_index, int v1, int v2, KHE_COST *avail_cost)
{
  int other_dt_index, a1, a2;  KHE_DRS_DOM_TEST other_dt;
  other_dt_index = dt_index + dt->correlated_delta;
  other_dt = HaArray(dsg->dom_tests, other_dt_index);
  HnAssert(other_dt->type == KHE_DRS_DOM_TEST_CORR1_CHILD,
    "KheDrsSignerCorr1Dominates internal error (other_dt_index = %d)",
    other_dt_index);
  a1 = HaArray(sig1->states, other_dt_index);
  a2 = HaArray(sig2->states, other_dt_index);
  return KheDrsDomTestDominatesCorr1(dt, a1, v1, a2, v2, avail_cost);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSignerCorr1Dominates(KHE_DRS_SIGNER dsg,                      */
/*    KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2, KHE_DRS_DOM_TEST dt,   */
/*    int dt_index, int v1, int v2, KHE_COST *avail_cost, int verbosity,     */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Carry out a CORR1 dominance test whose first (child) element is dt.      */
/*                                                                           */
/*****************************************************************************/
/* ***
static bool KheDrsDomTestDominatesCorr1(KHE_DRS_DOM_TEST dt,
  int a1, int b1, int a2, int b2, KHE_COST *avail_cost);
*** */
static KHE_DRS_COST_TUPLE KheDrsDim4TableGet4(KHE_DRS_DIM4_TABLE d4,
  int index4, int index3, int index2, int index1);

static void KheDrsSignerCorr1Dominates(KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2, KHE_DRS_DOM_TEST dt,
  int dt_index, int a1, int a2, KHE_COST *avail_cost, int verbosity,
  int indent, FILE *fp)
{
  int other_dt_index, l1, l2;  KHE_DRS_DOM_TEST other_dt;  KHE_COST psi;
  other_dt_index = dt_index + dt->correlated_delta;
  other_dt = HaArray(dsg->dom_tests, other_dt_index);
  HnAssert(other_dt->type == KHE_DRS_DOM_TEST_CORR1_PARENT,
    "KheDrsSignerCorr1Dominates internal error (other_dt_index = %d)",
    other_dt_index);
  l1 = HaArray(sig1->states, other_dt_index);
  l2 = HaArray(sig2->states, other_dt_index);
  psi = KheDrsDim4TableGet4(dt->corr_dom_table4, a1, a2, l1, l2).psi;
  if( fp != NULL )
    fprintf(fp, "\n%*s[ Corr1(a1 %d, a2 %d, l1 %d, l2 %d) -> %.5f ]\n\n",
      indent, "", a1, a2, l1, l2, KheCostShow(psi));
  *avail_cost += psi;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignerCorr2Dominates(KHE_DRS_SIGNER dsg,                      */
/*    KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2, KHE_DRS_DOM_TEST dt,   */
/*    int dt_index, int v1, int v2, KHE_COST *avail_cost, int verbosity,     */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Carry out a CORR2 dominance test whose sole (child) element is dt.       */
/*                                                                           */
/*****************************************************************************/
static int KheDrsExprIntSumCostInitialValue(KHE_DRS_EXPR_INT_SUM_COST eisc);

static void KheDrsSignerCorr2Dominates(KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2, KHE_DRS_DOM_TEST dt,
  int dt_index, int a1, int a2, KHE_COST *avail_cost, int verbosity,
  int indent, FILE *fp)
{
  int l1, l2;  KHE_DRS_EXPR_INT_SUM_COST eisc;  KHE_DRS_PARENT prnt;
  KHE_COST psi;
  HnAssert(HaArrayCount(dt->expr->parents) > 0,
    "KheDrsSignerCorr2Dominates internal error (no parents)");
  prnt = HaArrayFirst(dt->expr->parents);
  HnAssert(prnt.expr->tag == KHE_DRS_EXPR_INT_SUM_COST_TAG,
    "KheDrsSignerCorr2Dominates internal error (prnt.expr->tag = %d)",
    prnt.expr->tag);
  eisc = (KHE_DRS_EXPR_INT_SUM_COST) prnt.expr;
  l1 = l2 = KheDrsExprIntSumCostInitialValue(eisc);
  psi = KheDrsDim4TableGet4(dt->corr_dom_table4, a1, a2, l1, l2).psi;
  if( fp != NULL )
    fprintf(fp, "\n%*s[ Corr2(a1 %d, a2 %d, l1 %d, l2 %d) -> %.5f ]\n\n",
      indent, "", a1, a2, l1, l2, KheCostShow(psi));
  *avail_cost += psi;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSignerCorr3Dominates(KHE_DRS_SIGNER dsg,                      */
/*    KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2, KHE_DRS_DOM_TEST dt,   */
/*    int dt_index, int v1, int v2, KHE_COST *avail_cost)                    */
/*                                                                           */
/*  Carry out one corr3 dominance test.                                      */
/*                                                                           */
/*  Here dt is the first element of a set of two or more corr3 dom tests.    */
/*                                                                           */
/*****************************************************************************/
/* ***
static KHE_DRS_COST_TUPLE KheDrsDomTableSubGet(KHE_DRS_DOM_TABLE_SUB dts,
  int l1, int l2, KHE_DRS_DOM_TEST dt);
static KHE_DRS_COST_TUPLE KheDrsDim2TableGet2(KHE_DRS_DIM2_TABLE d2,
  int index2, int index1);

static bool KheDrsSignerCorr3Dominates(KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2, KHE_DRS_DOM_TEST dt,
  int dt_index, int v1, int v2, KHE_COST *avail_cost)
{
  KHE_COST psi_diff, min_psi_diff, sum_psi0;  KHE_DRS_COST_TUPLE mp;
  ** mp = KheDrsDomTableSubGet(dt->main_dom_table2, v1, v2, dt); **
  mp = KheDrsDim2TableGet2(dt->main_dom_table2, v1, v2);
  min_psi_diff = mp.psi - mp.psi0;
  sum_psi0 = mp.psi0;
  do
  {
    ** move to next dt, v1, and v2 **
    dt_index += dt->correlated_delta;
    dt = HaArray(dsg->dom_tests, dt_index);
    v1 = HaArray(sig1->states, dt_index);
    v2 = HaArray(sig2->states, dt_index);

    ** accumulate min_psi_diff and sum_psi0 **
    ** mp = KheDrsDomTableSubGet(dt->main_dom_table2, v1, v2, dt); **
    mp = KheDrsDim2TableGet2(dt->main_dom_table2, v1, v2);
    psi_diff = mp.psi - mp.psi0;
    if( psi_diff < min_psi_diff )
      min_psi_diff = psi_diff;
    sum_psi0 += mp.psi0;
  } while( dt->type != KHE_DRS_DOM_TEST_CORR3_LAST );

  ** and there we are **
  *avail_cost += (min_psi_diff + sum_psi0);
  return *avail_cost >= 0;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDebugCorr34Header(char *header_str, int verbosity,            */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of the header of a Corr3 or Corr4 test.                      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDebugCorr34Header(char *header_str, int verbosity,
  int indent, FILE *fp)
{
  if( fp != NULL )
    fprintf(fp, "\n%*s[ %-6s %2s %2s  %8s %8s %8s %8s %8s\n", indent, "",
      header_str, "v1", "v2", "Psi", "Psi0", "diff", "min_diff", "sum_Psi0");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDebugCorr34Line(KHE_DRS_DOM_TEST dt, int dt_index,            */
/*    int v1, int v2, KHE_COST psi, KHE_COST psi0, KHE_COST diff,            */
/*    KHE_COST min_diff, KHE_COST sum_psi0, int indent, FILE *fp)            */
/*                                                                           */
/*  Print one line of a corr3 or corr4 debug, but only if fp != NULL.        */
/*                                                                           */
/*****************************************************************************/
static void KheDrsMonitorDebug(KHE_MONITOR m, FILE *fp);

static void KheDrsDebugCorr34Line(KHE_DRS_DOM_TEST dt, int dt_index,
  int v1, int v2, KHE_COST psi, KHE_COST psi0, KHE_COST diff,
  KHE_COST min_diff, KHE_COST sum_psi0, int indent, FILE *fp)
{
  if( fp != NULL )
  {
    fprintf(fp, "%*s  %2s[%2d] %2d %2d  %8.5f %8.5f %8.5f %8.5f %8.5f ",
      indent, "", KheDrsDomTestTypeShow(dt->type), dt_index, v1, v2,
      KheCostShow(psi), KheCostShow(psi0), KheCostShow(diff),
      KheCostShow(min_diff), KheCostShow(sum_psi0));
    KheDrsMonitorDebug(dt->monitor, fp);
    fprintf(fp, "\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignerCorr3Dominates(KHE_DRS_SIGNER dsg,                      */
/*    KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2, KHE_DRS_DOM_TEST dt,   */
/*    int dt_index, int v1, int v2, KHE_COST *avail_cost,                    */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Carry out a CORR3 dominance test whose first element is dt.              */
/*                                                                           */
/*****************************************************************************/
static KHE_DRS_COST_TUPLE KheDrsDim2TableGet2(KHE_DRS_DIM2_TABLE d2,
  int index2, int index1);

static void KheDrsSignerCorr3Dominates(KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2, KHE_DRS_DOM_TEST dt,
  int dt_index, int v1, int v2, KHE_COST *avail_cost,
  int verbosity, int indent, FILE *fp)
{
  KHE_COST psi_diff, min_psi_diff, sum_psi0, sum_psi;  KHE_DRS_COST_TUPLE mp;
  bool differ3;
  KheDrsDebugCorr34Header("Corr3", verbosity, indent, fp);
  mp = KheDrsDim2TableGet2(dt->main_dom_table2, v1, v2);
  min_psi_diff = mp.psi - mp.psi0;
  sum_psi0 = mp.psi0;
  sum_psi = mp.psi;
  KheDrsDebugCorr34Line(dt, dt_index, v1, v2, mp.psi, mp.psi0, min_psi_diff,
    min_psi_diff, sum_psi0, indent, fp);
  do
  {
    /* move to next dt, v1, and v2 */
    dt_index += dt->correlated_delta;
    dt = HaArray(dsg->dom_tests, dt_index);
    v1 = HaArray(sig1->states, dt_index);
    v2 = HaArray(sig2->states, dt_index);

    /* accumulate min_psi_diff and sum_psi0 */
    mp = KheDrsDim2TableGet2(dt->main_dom_table2, v1, v2);
    psi_diff = mp.psi - mp.psi0;
    if( psi_diff < min_psi_diff )
      min_psi_diff = psi_diff;
    sum_psi0 += mp.psi0;
    sum_psi  += mp.psi;
    KheDrsDebugCorr34Line(dt, dt_index, v1, v2, mp.psi, mp.psi0, psi_diff,
      min_psi_diff, sum_psi0, indent, fp);
  } while( dt->type != KHE_DRS_DOM_TEST_CORR3_LAST );

  /* and there we are */
  *avail_cost += (min_psi_diff + sum_psi0);
  if( fp != NULL )
  {
    differ3 = (min_psi_diff + sum_psi0 != sum_psi);
    fprintf(fp, "%*s] corr3 %.5f, uncorrelated %.5f%s\n\n", indent, "",
      KheCostShow(min_psi_diff + sum_psi0), KheCostShow(sum_psi),
      differ3 ? " (corr3 differs from uncorrelated)" : "");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSignerCorr4Dominates(KHE_DRS_SIGNER dsg,                      */
/*    KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2, KHE_DRS_DOM_TEST dt,   */
/*    int dt_index, int v1, int v2, KHE_COST *avail_cost)                    */
/*                                                                           */
/*  Carry out one corr4 dominance test.  This is identical to a corr3        */
/*  test except that it uses psi_plus instead of psi.                        */
/*                                                                           */
/*  Here dt is the first element of a set of two or more corr4 dom tests.    */
/*                                                                           */
/*****************************************************************************/
/* ***
static KHE_DRS_COST_TUPLE KheDrsDomTableSubGet(KHE_DRS_DOM_TABLE_SUB dts,
  int l1, int l2, KHE_DRS_DOM_TEST dt);
*** */

/* ***
static bool KheDrsSignerCorr4Dominates(KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2, KHE_DRS_DOM_TEST dt,
  int dt_index, int v1, int v2, KHE_COST *avail_cost)
{
  KHE_COST psi_diff, min_psi_diff, sum_psi0;  KHE_DRS_COST_TUPLE mp;
  ** mp = KheDrsDomTableSubGet(dt->main_dom_table2, v1, v2, dt); **
  mp = KheDrsDim2TableGet2(dt->main_dom_table2, v1, v2);
  HnAssert(mp.psi_plus_defined, "KheDrsSignerCorr4Dominates internal error 1");
  min_psi_diff = mp.psi_plus - mp.psi0;
  sum_psi0 = mp.psi0;
  do
  {
    ** move to next dt, v1, and v2 **
    dt_index += dt->correlated_delta;
    dt = HaArray(dsg->dom_tests, dt_index);
    v1 = HaArray(sig1->states, dt_index);
    v2 = HaArray(sig2->states, dt_index);

    ** accumulate min_psi_diff and sum_psi0 **
    ** mp = KheDrsDomTableSubGet(dt->main_dom_table2, v1, v2, dt); **
    mp = KheDrsDim2TableGet2(dt->main_dom_table2, v1, v2);
    HnAssert(mp.psi_plus_defined,"KheDrsSignerCorr4Dominates internal error 2");
    psi_diff = mp.psi_plus - mp.psi0;
    if( psi_diff < min_psi_diff )
      min_psi_diff = psi_diff;
    sum_psi0 += mp.psi0;
  } while( dt->type != KHE_DRS_DOM_TEST_CORR4_LAST );

  ** and there we are **
  *avail_cost += (min_psi_diff + sum_psi0);
  return *avail_cost >= 0;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignerCorr4Dominates(KHE_DRS_SIGNER dsg,                      */
/*    KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2, KHE_DRS_DOM_TEST dt,   */
/*    int dt_index, int v1, int v2, KHE_COST *avail_cost,                    */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Like KheDrsSignerCorr4Dominates, only with debug output.                 */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignerCorr4Dominates(KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2, KHE_DRS_DOM_TEST dt,
  int dt_index, int v1, int v2, KHE_COST *avail_cost,
  int verbosity, int indent, FILE *fp)
{
  KHE_COST psi_diff, min_psi_diff, psi_plus_diff, min_psi_plus_diff;
  KHE_COST sum_psi0, sum_psi;  KHE_DRS_COST_TUPLE mp;  bool differ3, differ4;
  KheDrsDebugCorr34Header("Corr4", verbosity, indent, fp);
  mp = KheDrsDim2TableGet2(dt->main_dom_table2, v1, v2);
  HnAssert(mp.psi_plus_defined, "KheDrsSignerCorr4Dominates internal error 1");
  psi_plus_diff = mp.psi_plus - mp.psi0;
  min_psi_plus_diff = psi_plus_diff;
  psi_diff = mp.psi - mp.psi0;
  min_psi_diff = psi_diff;
  sum_psi0 = mp.psi0;
  sum_psi = mp.psi;
  KheDrsDebugCorr34Line(dt, dt_index, v1, v2, mp.psi_plus, mp.psi0,
    psi_plus_diff, min_psi_plus_diff, sum_psi0, indent, fp);
  do
  {
    /* move to next dt, v1, and v2 */
    dt_index += dt->correlated_delta;
    dt = HaArray(dsg->dom_tests, dt_index);
    v1 = HaArray(sig1->states, dt_index);
    v2 = HaArray(sig2->states, dt_index);

    /* accumulate min_psi_plus_diff and sum_psi0 */
    mp = KheDrsDim2TableGet2(dt->main_dom_table2, v1, v2);
    HnAssert(mp.psi_plus_defined >= 0,
      "KheDrsSignerCorr4Dominatesinternal error 2");
    psi_plus_diff = mp.psi_plus - mp.psi0;
    if( psi_plus_diff < min_psi_plus_diff )
      min_psi_plus_diff = psi_plus_diff;
    psi_diff = mp.psi - mp.psi0;
    if( psi_diff < min_psi_diff )
      min_psi_diff = psi_diff;
    sum_psi0 += mp.psi0;
    sum_psi  += mp.psi;
    KheDrsDebugCorr34Line(dt, dt_index, v1, v2, mp.psi_plus, mp.psi0,
      psi_plus_diff, min_psi_plus_diff, sum_psi0, indent, fp);
  } while( dt->type != KHE_DRS_DOM_TEST_CORR4_LAST );

  /* and there we are */
  *avail_cost += (min_psi_plus_diff + sum_psi0);
  if( fp != NULL )
  {
    differ3 = (min_psi_diff + sum_psi0 != sum_psi);
    differ4 = (min_psi_plus_diff != min_psi_diff);
    fprintf(fp, "%*s] corr4 %.5f, corr3 %.5f, uncorrelated %.5f%s%s\n\n",
      indent, "", KheCostShow(min_psi_plus_diff + sum_psi0),
      KheCostShow(min_psi_diff + sum_psi0), KheCostShow(sum_psi),
      differ3 ? " (corr3 differs from uncorrelated)" : "",
      differ4 ? " (corr4 differs from corr3)" : "");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDominanceDebugHeaderLine(char *desc, KHE_DRS_SIGNER dsg,      */
/*    KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2,                        */
/*    KHE_DRS_SIGNER_TEST test, KHE_COST avail_cost, int indent, FILE *fp)   */
/*                                                                           */
/*  Debug print one header line of a dominance test, after a spacer line.    */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDominanceDebugHeaderLine(KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2,
  KHE_DRS_SIGNER_TEST test, KHE_COST avail_cost, int indent, FILE *fp)
{
  char *descr[3] = {"Hard", "Soft", "All"};
  if( fp != NULL )
  {
    KheDrsDominanceDebugSpacer(indent, fp);
    fprintf(fp, "%*s%s %-11s v1 v2 (avail %.5f)\n", indent, "",
      descr[(int) test], KheDrsSignerId(dsg), KheCostShow(avail_cost));
    /* ***
    fprintf(fp, "%*sHard %-11s v1 v2 (sig1 %p, sig2 %p, avail %.5f)\n",
      indent, "", KheDrsSignerId(dsg), (void *) sig1, (void *) sig2,
      KheCostShow(avail_cost));
    *** */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSignerDoDominates(KHE_DRS_SIGNER dsg,                         */
/*    KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2,                        */
/*    KHE_DRS_SIGNER_TEST test, int trie_start_depth, bool stop_on_neg,      */
/*    KHE_COST *avail_cost, int verbosity, int indent, FILE *fp)             */
/*                                                                           */
/*  This function does most of the work of deciding whether sig1             */
/*  dominates sig2.  It has several extra parameters, allowing it            */
/*  to handle several variants of the idea, as follows.                      */
/*                                                                           */
/*  Parameter test says whether to compare only hard constraints, or only    */
/*  soft constraints, or all.  The individual tests can be done in any       */
/*  order, and doing all the hard constraints of several signers before      */
/*  all the soft constraints is known to save time.                          */
/*                                                                           */
/*  Parameter trie_start_depth is non-zero only when this function is        */
/*  called from within the trie data structure.  It specifies that some      */
/*  positions have already been tested and testing is to start after         */
/*  those.  Must be testing all, not hard or soft, else abort.               */
/*                                                                           */
/*  Parameter stop_on_neg, when true, specifies that the testing is to       */
/*  stop as soon as *avail_cost goes negative.                               */
/*                                                                           */
/*  Parameter *avail_cost is the available cost on entry and the             */
/*  remaining available cost on exit.  KheDrsSignerDoDominates does not      */
/*  consult the cost fields of sig1 and sig2; it assumes that they have      */
/*  already given rise to a suitable initial value of *avail_cost.           */
/*                                                                           */
/*  If fp != NULL, produce debug output while doing this.                    */
/*                                                                           */
/*****************************************************************************/
static bool KheDrsDomTestDominatesStrong(KHE_DRS_DOM_TEST dt, int val1,
  int val2);
static bool KheDrsDomTestDominatesTradeoff(KHE_DRS_DOM_TEST dt,
  int val1, int val2, KHE_COST *avail_cost);
static void KheDrsDomTestDominatesUniform(KHE_DRS_DOM_TEST dt,
  int val1, int val2, KHE_COST *avail_cost);

static bool KheDrsSignerDoDominates(KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2,
  KHE_DRS_SIGNER_TEST test, int trie_start_depth, bool stop_on_neg,
  KHE_COST *avail_cost, int verbosity, int indent, FILE *fp)
{
  KHE_DRS_DOM_TEST dt;
  int start_index, stop_index, dt_index, v1, v2, sig_len;

  /* consistency checks */
  sig_len = HaArrayCount(dsg->dom_tests);
  HnAssert(HaArrayCount(sig1->states) == sig_len,
    "KheDrsSignerDoDominates internal error 1 (count %d != count %d)\n",
    HaArrayCount(sig1->states), sig_len);
  HnAssert(HaArrayCount(sig2->states) == sig_len,
    "KheDrsSignerDoDominates internal error 2 (count %d != count %d)\n",
    HaArrayCount(sig2->states), sig_len);

  /* work out start_index and stop_index */
  switch( test )
  {
    case KHE_DRS_SIGNER_HARD:

      HnAssert(trie_start_depth == 0,
	"KheDrsSignerDoDominates internal error 1");
      start_index = 0;
      stop_index = dsg->last_hard_cost_index + 1;
      break;

    case KHE_DRS_SIGNER_SOFT:

      HnAssert(trie_start_depth == 0,
	"KheDrsSignerDoDominates internal error 2");
      start_index = dsg->last_hard_cost_index + 1;
      stop_index = sig_len;
      break;

    case KHE_DRS_SIGNER_ALL:

      start_index = trie_start_depth;
      stop_index = sig_len;
      break;

    default:

      HnAbort("KheDrsSignerDoDominates internal error 3");
      start_index = 0, stop_index = 0;  /* keep compiler happy */
  }

  /* print spacer and header line */
  KheDrsDominanceDebugHeaderLine(dsg, sig1, sig2, test, *avail_cost,
    indent, fp);

  /* quit immediately if no available cost */
  if( stop_on_neg && *avail_cost < 0 )
    return false;

  /* do the actual test from start_index inclusive to stop_index exclusive */
  for( dt_index = start_index;  dt_index < stop_index;  dt_index++ )
  {
    dt = HaArray(dsg->dom_tests, dt_index);
    v1 = HaArray(sig1->states, dt_index);
    v2 = HaArray(sig2->states, dt_index);
    switch( dt->type )
    {
      case KHE_DRS_DOM_TEST_UNUSED:

	/* unused test, should never happen */
	HnAbort("internal error in KheDrsSignerDominates (UNUSED)");
	break;

      case KHE_DRS_DOM_TEST_STRONG:

	/* strong dominance */
	if( !KheDrsDomTestDominatesStrong(dt, v1, v2) )
	  *avail_cost -= KheCost(1, 0);
	break;

      case KHE_DRS_DOM_TEST_TRADEOFF:

	/* tradeoff dominance */
	if( !KheDrsDomTestDominatesTradeoff(dt, v1, v2, avail_cost) )
	  *avail_cost -= KheCost(1, 0);
	break;

      case KHE_DRS_DOM_TEST_UNIFORM:

	/* uniform dominance */
	KheDrsDomTestDominatesUniform(dt, v1, v2, avail_cost);
	break;

      case KHE_DRS_DOM_TEST_CORR1_PARENT:

	/* parent of correlated parent/child pair */
	/* nothing to do here (the child does it all) */
	break;

      case KHE_DRS_DOM_TEST_CORR1_CHILD:

	/* child of correlated parent/child pair */
	KheDrsSignerCorr1Dominates(dsg, sig1, sig2, dt, dt_index,
	  v1, v2, avail_cost, verbosity, indent + 4, fp);
	break;

      case KHE_DRS_DOM_TEST_CORR2_CHILD:

	/* child of correlated parent/child pair (parent not in signature) */
	KheDrsSignerCorr2Dominates(dsg, sig1, sig2, dt, dt_index,
	  v1, v2, avail_cost, verbosity, indent + 4, fp);
	break;

      case KHE_DRS_DOM_TEST_CORR3_FIRST:

	/* first element of a set of two or more corr3 dom tests */
	KheDrsSignerCorr3Dominates(dsg, sig1, sig2, dt, dt_index,
	  v1, v2, avail_cost, verbosity, indent + 4, fp);
	break;

      case KHE_DRS_DOM_TEST_CORR3_MID:
      case KHE_DRS_DOM_TEST_CORR3_LAST:

	/* non-first element of a set of two or more corr3 dom tests */
	/* nothing to do here (the first element does it all) */
	break;

      case KHE_DRS_DOM_TEST_CORR4_FIRST:

	/* first element of a set of two or more corr4 dom tests */
	KheDrsSignerCorr4Dominates(dsg, sig1, sig2, dt, dt_index,
	  v1, v2, avail_cost, verbosity, indent + 4, fp);
	break;

      case KHE_DRS_DOM_TEST_CORR4_MID:
      case KHE_DRS_DOM_TEST_CORR4_LAST:

	/* non-first element of a set of two or more corr4 dom tests */
	/* nothing to do here (the first element does it all) */
	break;

      default:

	HnAbort("KheDrsSignerDominates internal error 5 (%d)", dt->type);
	break;
    }

    /* print debug line, and quit early if *avail_cost is now negative */
    KheDrsDominanceDebugCmpLine(*avail_cost, dt->type, dt_index, v1, v2,
      dt->monitor, indent, fp);
    if( stop_on_neg && *avail_cost < 0 )
      return false;
  }
  return *avail_cost >= 0;
}


/*****************************************************************************/
/*                                                                           */
/*****************************************************************************/

/* ***
static bool KheDrsSignerDominatesStatesOnlyAll(KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2, int trie_start_depth,
  bool stop_on_neg, KHE_COST *avail_cost, int verbosity, int indent, FILE *fp)
{
  return KheDrsSignerDoDominates(dsg, sig1, sig2, trie_start_depth,
    HaArrayCount(dsg->dom_tests), stop_on_neg, avail_cost,
    verbosity, indent, fp);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*****************************************************************************/

/* ***
static bool KheDrsSignerDominatesStatesOnlyHard(KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2,
  KHE_COST *avail_cost, int verbosity, int indent, FILE *fp)
{
  return KheDrsSignerDoDominates(dsg, sig1, sig2, 0,
    dsg->last_hard_cost_index + 1, true, avail_cost, verbosity, indent, fp);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*****************************************************************************/

/* ***
static bool KheDrsSignerDominatesStatesOnlySoft(KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2,
  KHE_COST *avail_cost, int verbosity, int indent, FILE *fp)
{
  return KheDrsSignerDoDominates(dsg, sig1, sig2,
    dsg->last_hard_cost_index + 1, HaArrayCount(dsg->dom_tests),
    true, avail_cost, verbosity, indent, fp);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSignerDominates(KHE_DRS_SIGNER dsg,                           */
/*    KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2,                        */
/*    KHE_COST *avail_cost, int verbosity, int indent, FILE *fp)             */
/*                                                                           */
/*  Return true if sig1 dominates sig2, including comparing their costs.     */
/*  To allow this call to be integrated with others that affect *avail_cost, */
/*  this function assumes that *avail_cost is already initialized, and it    */
/*  carries on based on that initial value.                                  */
/*                                                                           */
/*  If fp != NULL, produce debug output while doing this.                    */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsSignerDominates(KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE sig1, KHE_DRS_SIGNATURE sig2,
  KHE_COST *avail_cost, int verbosity, int indent, FILE *fp)
{
  *avail_cost += (sig2->cost - sig1->cost);
  return KheDrsSignerDoDominates(dsg, sig1, sig2, KHE_DRS_SIGNER_ALL,
    0, true, avail_cost, verbosity, indent, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignerDebug(KHE_DRS_SIGNER dsg,                               */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of dsg onto fp with the given verbosity and indent.          */
/*                                                                           */
/*****************************************************************************/
static void KheDrsDomTestDebug(KHE_DRS_DOM_TEST dt, int verbosity,
  int indent, FILE *fp);

static void KheDrsSignerDebug(KHE_DRS_SIGNER dsg,
  int verbosity, int indent, FILE *fp)
{
  KHE_DRS_DOM_TEST dt;  int i;
  fprintf(fp, "%*s[ Signer %s\n", indent, "", KheDrsSignerId(dsg));
  HaArrayForEach(dsg->dom_tests, dt, i)
    KheDrsDomTestDebug(dt, verbosity, indent + 2, fp);
  fprintf(fp, "%*s]\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_SIGNER_SET"                                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SIGNER_SET KheDrsSignerSetMake(KHE_DRS_DAY day,                  */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new signer set.                                                   */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_SIGNER_SET KheDrsSignerSetMake(KHE_DRS_DAY day,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SIGNER_SET res;
  HaMake(res, drs->arena);
  res->day = day;
  HaArrayInit(res->signers, drs->arena);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignerSetOpen(KHE_DRS_SIGNER_SET signer_set,                  */
/*    KHE_DRS_DAY day, KHE_DYNAMIC_RESOURCE_SOLVER drs)                      */
/*                                                                           */
/*  Open signer_set for use on day.                                          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignerSetOpen(KHE_DRS_SIGNER_SET signer_set,
  KHE_DRS_DAY day, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_RESOURCE dr;  int i;  KHE_DRS_RESOURCE_ON_DAY drd;
  KHE_DRS_SIGNER dsg;

  /* one signer for each open resource */
  HnAssert(HaArrayCount(signer_set->signers) == 0,
    "KheDrsSignerSetOpen internal error");
  for( i = 0;  i < KheDrsResourceSetCount(drs->open_resources);  i++ )
  {
    dr = KheDrsResourceSetResource(drs->open_resources, i);
    drd = KheDrsResourceOnDay(dr, day);
    HaArrayAddLast(signer_set->signers, drd->signer);
  }

  /* one additional signer for event resource expressions */
  dsg = KheDrsSignerMake(day, NULL, NULL, drs);
  HaArrayAddLast(signer_set->signers, dsg);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignerSetClose(KHE_DRS_SIGNER_SET signer_set,                 */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Close signer_set.                                                        */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignerSetClose(KHE_DRS_SIGNER_SET signer_set,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_SIGNER dsg;

  /* delete and free the last signer (the others were not made here) */
  HnAssert(HaArrayCount(signer_set->signers) > 0,
    "KheDrsSignerSetClose internal error");
  dsg = HaArrayLast(signer_set->signers);
  KheDrsSignerFree(dsg, drs);

  /* clear the signers */
  HaArrayClear(signer_set->signers);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsSignerSetPartialHashSignature(KHE_DRS_SIGNER_SET signer_set,   */
/*    KHE_DRS_SIGNATURE_SET sig_set)                                         */
/*                                                                           */
/*  Hash sig_set, but only the positions that have equality dom tests.       */
/*                                                                           */
/*****************************************************************************/

static int KheDrsSignerSetPartialHashSignature(KHE_DRS_SIGNER_SET signer_set,
  KHE_DRS_SIGNATURE_SET sig_set)
{
  int res, index, val, i, j;  KHE_DRS_SIGNER dsg;  KHE_DRS_SIGNATURE sig;
  HnAssert(HaArrayCount(signer_set->signers)==HaArrayCount(sig_set->signatures),
    "KheDrsSignerSetPartialHashSignature internal error");
  res = 0;
  HaArrayForEach(signer_set->signers, dsg, i)
  {
    sig = HaArray(sig_set->signatures, i);
    HaArrayForEach(dsg->eq_dom_test_indexes, index, j)
    {
      val = HaArray(sig->states, index);
      res = (res + val) * 3;
    }
  }
  if( res < 0 )
    res = - res;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSignerSetPartialEqualSignature(KHE_DRS_SIGNER_SET signer_set, */
/*    KHE_DRS_SIGNATURE_SET sig_set1, KHE_DRS_SIGNATURE_SET sig_set2)        */
/*                                                                           */
/*  Return true if sig1 and sig2 are equal at the positions that have        */
/*  equality dom tests.                                                      */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsSignerSetPartialEqualSignature(KHE_DRS_SIGNER_SET signer_set,
  KHE_DRS_SIGNATURE_SET sig_set1, KHE_DRS_SIGNATURE_SET sig_set2)
{
  int i, j, index;  KHE_DRS_SIGNATURE sig1, sig2;  KHE_DRS_SIGNER dsg;

  HnAssert(HaArrayCount(signer_set->signers) ==
    HaArrayCount(sig_set1->signatures),
    "KheDrsSignerSetPartialEqualSignature internal error 1");
  HnAssert(HaArrayCount(signer_set->signers) ==
    HaArrayCount(sig_set2->signatures),
    "KheDrsSignerSetPartialEqualSignature internal error 2");
  for( i = 0;  i < HaArrayCount(signer_set->signers);  i++ )
  {
    dsg = HaArray(signer_set->signers, i);
    sig1 = HaArray(sig_set1->signatures, i);
    sig2 = HaArray(sig_set2->signatures, i);
    HnAssert(HaArrayCount(sig1->states) == HaArrayCount(sig2->states),
     "KheDrsSignerPartialEqualSignature: signatures have different lengths");
    HaArrayForEach(dsg->eq_dom_test_indexes, index, j)
      if( HaArray(sig1->states, index) != HaArray(sig2->states, index) )
	return false;
  }
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignerEvalSignature(KHE_DRS_SIGNER dsg,                       */
/*    bool debug_if_rerun, KHE_DRS_SIGNATURE prev_sig, int next_di,          */
/*    KHE_DRS_SIGNATURE sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)                */
/*                                                                           */
/*  Using dsg, build sig, not necessarily from scratch.                      */
/*                                                                           */
/*****************************************************************************/

/* *** misconceived
static void KheDrsExprEvalSignature(KHE_DRS_EXPR e, KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE prev_sig, int next_di, KHE_DRS_SIGNATURE sig,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, bool debug);

static void KheDrsSignerSetEvalSignatureSet(KHE_DRS_SIGNER_SET signer_set,
  bool debug_if_rerun, KHE_DRS_SIGNATURE_SET prev_sig_set, int next_di,
  KHE_DRS_SIGNATURE_SET sig_set, KHE_DYNAMIC_RESOURCE_SOLVER drs, bool debug)
{
  KHE_DRS_SIGNER dsg;  int i;  KHE_DRS_SIGNATURE sig, prev_sig;
  HnAssert(HaArrayCount(sig_set->signatures) == 0,
    "KheDrsSignerSetEvalSignatureSet internal error 1");
  HnAssert(HaArrayCount(signer_set->signers)==HaArrayCount(sig_set->signatures),
    "KheDrsSignerSetEvalSignatureSet internal error 2");
  HaArrayForEach(signer_set->signers, dsg, i)
  {
    sig = KheDrsSignatureMake(drs, 0);
    prev_sig = HaArray(prev_sig_set->signatures, i);
    KheDrsSignerEvalSignature(dsg, debug_if_rerun, prev_sig, next_di,
      sig, drs, debug);
    KheDrsSignatur eSetAddSignature(sig_set, sig);
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  int KheFirstIndex(KHE_DRS_SIGNER_SET signer_set, int *start_depth)       */
/*                                                                           */
/*  Find the index of the first signer and adjust start_depth.               */
/*                                                                           */
/*****************************************************************************/

/* *** decided that inlining was better 
static int KheFirstIndex(KHE_DRS_SIGNER_SET signer_set, int *start_depth)
{
  HE_DRS_SIGNER dsg;  KHE_DRS_SIGNATURE sig;  int i;
  HaArrayForEach(signer_set->signers, dsg, i)
  {
    if( *start_depth < HaArrayCount(dsg->dom_tests) )
      return i;
    *start_depth -= HaArrayCount(dsg->dom_tests);
  }
  HnAbort("KheFirstIndex internal error 2");
  return 0;  ** keep compiler happy **
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSignerSetDominates(KHE_DRS_SIGNER_SET signer_set,             */
/*    KHE_DRS_SIGNATURE_SET sig_set1, KHE_DRS_SIGNATURE_SET sig_set2,        */
/*    KHE_COST trie_extra_cost, int trie_start_depth, bool use_caching,      */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)  */
/*                                                                           */
/*  Return true if sig_set1 dominates sig_set2, including comparing their    */
/*  costs.  If fp != NULL, produce debug output while doing this.            */
/*                                                                           */
/*  When this function is called from the trie data structure, some of its   */
/*  work may already be done.  It starts comparing from the states at        */
/*  position trie_start_depth (counting across any number of signatures).    */
/*                                                                           */
/*  If trie_start_depth is 0 (as it usually will be) and use_caching is      */
/*  true, then dominance test caching is used to speed up the test.          */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsSignerSetDominates(KHE_DRS_SIGNER_SET signer_set,
  KHE_DRS_SIGNATURE_SET sig_set1, KHE_DRS_SIGNATURE_SET sig_set2,
  KHE_COST trie_extra_cost, int trie_start_depth, bool use_caching,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)
{
  int i, count, last_cache;  KHE_DRS_SIGNER dsg;  KHE_COST avail_cost;
  KHE_DRS_SIGNATURE sig1, sig2;  KHE_DRS_RESOURCE dr;  KHE_DRS_COST_TUPLE ct;
  count = HaArrayCount(signer_set->signers);
  HnAssert(HaArrayCount(sig_set1->signatures) == count,
    "KheDrsSignerSetDominates internal error 1");
  HnAssert(HaArrayCount(sig_set2->signatures) == count,
    "KheDrsSignerSetDominates internal error 1");
  avail_cost = sig_set2->cost - sig_set1->cost - trie_extra_cost;
  if( avail_cost < 0 )
    return false;
  if( drs->solve_dom_approx > 0 )
    avail_cost += (avail_cost * drs->solve_dom_approx) / 10;
  if( trie_start_depth > 0 )
  {
    /* won't happen anyway but do it the easy way if it does */
    HaArrayForEach(signer_set->signers, dsg, i)
    {
      if( trie_start_depth < HaArrayCount(dsg->dom_tests) )
      {
	if( fp != NULL && i > 0 )
	  KheDrsDominanceDebugSpacer(indent, fp);
	sig1 = HaArray(sig_set1->signatures, i);
	sig2 = HaArray(sig_set2->signatures, i);
	if( !KheDrsSignerDoDominates(dsg, sig1, sig2, KHE_DRS_SIGNER_ALL,
	      trie_start_depth, true, &avail_cost, verbosity,indent,fp) )
	  return false;
	trie_start_depth = 0;
      }
      else
	trie_start_depth -= HaArrayCount(dsg->dom_tests);
    }
  }
  else if( USE_DOM_CACHING && use_caching )
  {
    /* used a cached value for all positions except the last */
    last_cache = HaArrayCount(signer_set->signers) - 2;
    for( i = 0;  i <= last_cache;  i++ )
    {
      if( fp != NULL && i > 0 )
	KheDrsDominanceDebugSpacer(indent, fp);
      dr = KheDrsResourceSetResource(drs->open_resources, i);
      dsg = HaArray(signer_set->signers, i);
      sig1 = HaArray(sig_set1->signatures, i);
      HnAssert(sig1->asst_to_shift_index >= 0,
	"KheDrsSignerSetDominates internal error 3");
      sig2 = HaArray(sig_set2->signatures, i);
      HnAssert(sig2->asst_to_shift_index >= 0,
	"KheDrsSignerSetDominates internal error 4");
      ct = KheDrsDim2TableGet2(dr->expand_dom_test_cache,
	sig1->asst_to_shift_index, sig2->asst_to_shift_index);
      avail_cost += ct.psi;
      if( fp != NULL )
	fprintf(fp, "%*s%8.5f  (cache[%d, %d] = %.5f)\n", indent, "",
	  KheCostShow(avail_cost), sig1->asst_to_shift_index,
	  sig2->asst_to_shift_index, KheCostShow(ct.psi));
      if( avail_cost < 0 )
	return false;
    }

    /* regular test for the last position */
    if( fp != NULL && i > 0 )
      KheDrsDominanceDebugSpacer(indent, fp);
    dsg = HaArrayLast(signer_set->signers);
    sig1 = HaArrayLast(sig_set1->signatures);
    sig2 = HaArrayLast(sig_set2->signatures);
    if( !KheDrsSignerDoDominates(dsg, sig1, sig2, KHE_DRS_SIGNER_ALL,
	  0, true, &avail_cost, verbosity, indent, fp) )
      return false;
  }
  else
  {
    /* visit hard constraints first; they often end the test quickly */
    HaArrayForEach(signer_set->signers, dsg, i)
    {
      sig1 = HaArray(sig_set1->signatures, i);
      sig2 = HaArray(sig_set2->signatures, i);
      /* ***
      KheDrsDominanceDebugHeaderLine("Hard", dsg, sig1, sig2, avail_cost,
	indent, fp);
      *** */
      if( !KheDrsSignerDoDominates(dsg, sig1, sig2, KHE_DRS_SIGNER_HARD,
	    0, true, &avail_cost, verbosity, indent, fp) )
	return false;
    }

    /* visit soft constraints */
    HaArrayForEach(signer_set->signers, dsg, i)
    {
      sig1 = HaArray(sig_set1->signatures, i);
      sig2 = HaArray(sig_set2->signatures, i);
      /* ***
      KheDrsDominanceDebugHeaderLine("Soft", dsg, sig1, sig2, avail_cost,
	indent, fp);
      *** */
      if( !KheDrsSignerDoDominates(dsg, sig1, sig2, KHE_DRS_SIGNER_SOFT,
	  0, true, &avail_cost, verbosity, indent, fp) )
	return false;
    }
  }
  return true;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsSignerSetDominates(KHE_DRS_SIGNER_SET signer_set,             */
/*    KHE_DRS_SIGNATURE_SET sig_set1, KHE_DRS_SIGNATURE_SET sig_set2,        */
/*    int start_depth, KHE_COST *avail_cost)                                 */
/*                                                                           */
/*  Dominance test without debug.                                            */
/*                                                                           */
/*****************************************************************************/

/* ***
static bool KheDrsSignerSetDominates(KHE_DRS_SIGNER_SET signer_set,
  KHE_DRS_SIGNATURE_SET sig_set1, KHE_DRS_SIGNATURE_SET sig_set2,
  KHE_COST trie_extra_cost, int trie_start_depth)
{
  return KheDrsSignerSetDominates(signer_set, sig_set1, sig_set2,
    trie_extra_cost, trie_start_depth, 0, 0, NULL);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSignerSetDebug(KHE_DRS_SIGNER_SET signer_set,                 */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of signer_set onto fp with the given verbosity and indent.   */
/*                                                                           */
/*****************************************************************************/

static void KheDrsSignerSetDebug(KHE_DRS_SIGNER_SET signer_set,
  int verbosity, int indent, FILE *fp)
{
  KHE_DRS_SIGNER dsg;  int i;
  fprintf(fp, "%*s[ SignerSet %s\n", indent, "", KheDrsDayId(signer_set->day));
  HaArrayForEach(signer_set->signers, dsg, i)
  {
    if( i > 0 )
      KheDrsDominanceDebugSpacer(indent + 2, fp);
    KheDrsSignerDebug(dsg, verbosity, indent + 2, fp);
  }
  fprintf(fp, "%*s]\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_DOM_KIND"                                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST_TYPE KheDrsDomKindToDomTestType(                        */
/*    KHE_DRS_DOM_KIND dom_kind)                                             */
/*                                                                           */
/*  Convert a dom kind into a dom test type.  That is, return the dom        */
/*  test type that is included in dom_kind.                                  */
/*                                                                           */
/*****************************************************************************/

/* *** good but currently unused
static KHE_DRS_DOM_TEST_TYPE KheDrsDomKindToDomTestType(
  KHE_DRS_DOM_KIND dom_kind)
{
  switch( dom_kind )
  {
    case KHE_DRS_DOM_LIST_NONE:		return KHE_DRS_DOM_TEST_STRONG;
    case KHE_DRS_DOM_LIST_STRONG:	return KHE_DRS_DOM_TEST_STRONG;
    case KHE_DRS_DOM_LIST_TRADEOFF:	return KHE_DRS_DOM_TEST_TRADEOFF;
    case KHE_DRS_DOM_LIST_UNIFORM:	return KHE_DRS_DOM_TEST_UNIFORM;
    case KHE_DRS_DOM_HASH_WEAK:		return KHE_DRS_DOM_TEST_STRONG;
    case KHE_DRS_DOM_HASH_MEDIUM:	return KHE_DRS_DOM_TEST_STRONG;
    case KHE_DRS_DOM_TRIE_STRONG:	return KHE_DRS_DOM_TEST_STRONG;
    case KHE_DRS_DOM_TRIE_TRADEOFF:	return KHE_DRS_DOM_TEST_TRADEOFF;
    case KHE_DRS_DOM_INDEXED_TRADEOFF:	return KHE_DRS_DOM_TEST_TRADEOFF;
    case KHE_DRS_DOM_INDEXED_UNIFORM:	return KHE_DRS_DOM_TEST_UNIFORM;

    default:

      HnAbort("KheDrsDomKindToDomTestType: unknown dom_kind (%d)", dom_kind);
      return KHE_DRS_DOM_TEST_STRONG;  ** keep compiler happy **
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST_TYPE KheDomKindToDomTestType(KHE_DRS_DOM_KIND dom_kind) */
/*                                                                           */
/*  Return the dom test type of dom_kind.                                    */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DOM_TEST_TYPE KheDomKindToDomTestType(KHE_DRS_DOM_KIND dom_kind)
{
  switch( dom_kind )
  {
    case KHE_DRS_DOM_LIST_NONE:		return KHE_DRS_DOM_TEST_UNUSED;
    case KHE_DRS_DOM_LIST_STRONG:	return KHE_DRS_DOM_TEST_STRONG;
    case KHE_DRS_DOM_LIST_TRADEOFF:	return KHE_DRS_DOM_TEST_TRADEOFF;
    case KHE_DRS_DOM_LIST_UNIFORM:	return KHE_DRS_DOM_TEST_UNIFORM;
    case KHE_DRS_DOM_HASH_WEAK:		return KHE_DRS_DOM_TEST_UNUSED;
    case KHE_DRS_DOM_HASH_MEDIUM:	return KHE_DRS_DOM_TEST_STRONG;
    case KHE_DRS_DOM_TRIE_STRONG:	return KHE_DRS_DOM_TEST_STRONG;
    case KHE_DRS_DOM_TRIE_TRADEOFF:	return KHE_DRS_DOM_TEST_TRADEOFF;
    case KHE_DRS_DOM_INDEXED_TRADEOFF:	return KHE_DRS_DOM_TEST_TRADEOFF;
    case KHE_DRS_DOM_INDEXED_UNIFORM:	return KHE_DRS_DOM_TEST_UNIFORM;

    default:

      HnAbort("KheDomKindToDomTestType: unknown dom_kind (%d)", dom_kind);
      return 0;  /* keep compiler happy */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST_TYPE KheDomKindCheckConsistency(                        */
/*    KHE_DRS_DOM_KIND main_dom_kind, bool cache,                            */
/*    KHE_DRS_DOM_KIND cache_dom_kind)                                       */
/*                                                                           */
/*  Check that main_dom_kind, cache, and cache_dom_kind are consistent,      */
/*  and return the consistent dom test type.                                 */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DOM_TEST_TYPE KheDomKindCheckConsistency(
  KHE_DRS_DOM_KIND main_dom_kind, bool cache, KHE_DRS_DOM_KIND cache_dom_kind)
{
  KHE_DRS_DOM_TEST_TYPE main_dom_test_type, cache_dom_test_type;
  main_dom_test_type = KheDomKindToDomTestType(main_dom_kind);
  if( cache )
  {
    cache_dom_test_type = KheDomKindToDomTestType(cache_dom_kind);
    HnAssert(main_dom_test_type == KHE_DRS_DOM_TEST_UNUSED ||
      cache_dom_test_type == KHE_DRS_DOM_TEST_UNUSED ||
      main_dom_test_type == cache_dom_test_type, "KheDomKindCheckConsistency:"
      "inconsistent main_dom_kind and cache_dom_kind arguments");
  }
  return main_dom_test_type;
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheDomKindShow(KHE_DRS_DOM_KIND dom_kind)                          */
/*                                                                           */
/*  Return a suitable label for dom_kind.                                    */
/*                                                                           */
/*****************************************************************************/

static char *KheDomKindShow(KHE_DRS_DOM_KIND dom_kind)
{
  switch( dom_kind )
  {
    case KHE_DRS_DOM_LIST_NONE:		return "ListNone";
    case KHE_DRS_DOM_LIST_STRONG:	return "ListStrong";
    case KHE_DRS_DOM_LIST_TRADEOFF:	return "ListTradeoff";
    case KHE_DRS_DOM_LIST_UNIFORM:	return "ListUniform";
    case KHE_DRS_DOM_HASH_WEAK:		return "HashWeak";
    case KHE_DRS_DOM_HASH_MEDIUM:	return "HashMedium";
    case KHE_DRS_DOM_TRIE_STRONG:	return "TrieStrong";
    case KHE_DRS_DOM_TRIE_TRADEOFF:	return "TrieTradeoff";
    case KHE_DRS_DOM_INDEXED_TRADEOFF:	return "IndexedTradeoff";
    case KHE_DRS_DOM_INDEXED_UNIFORM:	return "IndexedUniform";

    default:

      HnAbort("KheDomKindShow: unknown dom_kind (%d)", dom_kind);
      return NULL;  /* keep compiler happy */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_COST_TUPLE"                                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_COST_TUPLE KheDrsCostTupleMake(KHE_COST psi, KHE_COST psi0,      */
/*    bool psi_plus_defined, KHE_COST psi_plus)                              */
/*                                                                           */
/*  Make a cost tuple object with these attributes.                          */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_COST_TUPLE KheDrsCostTupleMake(KHE_COST psi, KHE_COST psi0,
  bool psi_plus_defined, KHE_COST psi_plus)
{
  KHE_DRS_COST_TUPLE res;
  res.psi = psi;
  res.psi0 = psi0;
  res.psi_plus_defined = psi_plus_defined;
  res.psi_plus = psi_plus;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsCostTupleDebug(KHE_DRS_COST_TUPLE ct, int verbosity,          */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of ct onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsCostTupleDebug(KHE_DRS_COST_TUPLE ct, int verbosity,
  int indent, FILE *fp)
{
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  if( ct.psi_plus_defined )
    fprintf(fp, "(%.5f, %.5f, %.5f)", KheCostShow(ct.psi),
      KheCostShow(ct.psi0), KheCostShow(ct.psi_plus));
  else
    fprintf(fp, "(%.5f, %.5f)", KheCostShow(ct.psi), KheCostShow(ct.psi0));
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_DOM_TABLE_SUB_SUB"                                    */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TABLE_SUB_SUB KheDrsDomTableSubSubMake(int offset,           */
/*    HA_ARENA a)                                                            */
/*                                                                           */
/*  Make a new, empty dominance sub-sub-table.                               */
/*                                                                           */
/*****************************************************************************/

/* ***
static KHE_DRS_DOM_TABLE_SUB_SUB KheDrsDomTableSubSubMake(int offset,
  HA_ARENA a)
{
  KHE_DRS_DOM_TABLE_SUB_SUB res;
  HaMake(res, a);
  res->offset = offset;
  HaArrayInit(res->indexed_by_l2, a);
  ** ***
  if( DEBUG44 )
    fprintf(stderr, "KheDrsDomTableSubSubMake(%d) = %p\n", offset,
      (void *) res);
  *** **
  return res;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDomTableSubSubPut(KHE_DRS_DOM_TABLE_SUB_SUB dtss,             */
/*    int l2, KHE_COST psi, KHE_COST psi0, bool psi_plus_defined,            */
/*    KHE_COST psi_plus)                                                     */
/*                                                                           */
/*  Add (psi, psi0, psi_plus) to dtss at position l2.  This must be the      */
/*  next available position.                                                 */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsDomTableSubSubPut(KHE_DRS_DOM_TABLE_SUB_SUB dtss,
  int l2, KHE_COST psi, KHE_COST psi0, bool psi_plus_defined,
  KHE_COST psi_plus)
{
  HnAssert(l2 == HaArrayCount(dtss->indexed_by_l2) + dtss->offset,
    "KheDrsDomTableSubSubPut:" " l2 = %d, expected %d\n", l2,
    HaArrayCount(dtss->indexed_by_l2) + dtss->offset);
  HaArrayAddLast(dtss->indexed_by_l2,
    KheDrsCostTupleMake(psi, psi0, psi_plus_defined, psi_plus));
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_COST_TUPLE KheDrsDomTableSubSubGet(                              */
/*    KHE_DRS_DOM_TABLE_SUB_SUB dtss, int l2, KHE_DRS_DOM_TEST dt)           */
/*                                                                           */
/*  Retrieve the cost tuple stored at position l2 of dtss, which must exist.  */
/*  Parameter dt is for debug output only.                                   */
/*                                                                           */
/*****************************************************************************/

/* ***
static KHE_DRS_COST_TUPLE KheDrsDomTableSubSubGet(
  KHE_DRS_DOM_TABLE_SUB_SUB dtss, int l2, KHE_DRS_DOM_TEST dt)
{
  int index;
  index = l2 - dtss->offset;
  if( DEBUG44 && !(0 <= index && index < HaArrayCount(dtss->indexed_by_l2)) )
  {
    ** about to crash **
    fprintf(stderr, "KheDrsDomTableSubSubGet(%p) about to crash\n",
      (void *) dtss);
    fprintf(stderr, "  offset = %d, HaArrayCount(dtss->indexed_by_l2) = %d\n",
      dtss->offset, HaArrayCount(dtss->indexed_by_l2));
    fprintf(stderr, "  monitor %s\n", dt->monitor == NULL ? "-" :
      KheMonitorId(dt->monitor));
  }
  HnAssert(0 <= index && index < HaArrayCount(dtss->indexed_by_l2),
    "KheDrsDomTableSubSubGet(%p): l2 (%d) out of range (%d .. %d)",
    (void *) dtss, l2, dtss->offset,
    HaArrayCount(dtss->indexed_by_l2) - 1 + dtss->offset);
  return HaArray(dtss->indexed_by_l2, index);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDomTableSubSubDebug(KHE_DRS_DOM_TABLE_SUB_SUB dtss, int e,    */
/*    int l1, int verbosity, int indent, FILE *fp)                           */
/*                                                                           */
/*  Debug print of dtss onto fp with the given verbosity and indent.  Its    */
/*  index in the enclosing dom table is (e, l1).                             */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsDomTableSubSubDebug(KHE_DRS_DOM_TABLE_SUB_SUB dtss, int e,
  int l1, int verbosity, int indent, FILE *fp)
{
  int l2;  KHE_COST cost;
  fprintf(fp, "%*s[ e = %d, l1 = %d:\n", indent, "", e, l1);
  HaArrayForEach(dtss->indexed_by_l2, cost, l2)
  {
    if( l2 == 0 )
      fprintf(fp, "%*s", indent + 2, "");
    else if( l2 % 8 == 0 )
      fprintf(fp, ",\n%*s", indent + 2, "");
    else
      fprintf(fp, ", ");
    fprintf(fp, "%d: %.5f", l2, KheCostShow(cost));
  }
  fprintf(fp, "%*s]\n", indent, "");
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_DOM_TABLE_SUB"                                        */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TABLE_SUB KheDrsDomTableSubMake(int offset, HA_ARENA a)      */
/*                                                                           */
/*  Make a new, empty dominance sub-table.                                   */
/*                                                                           */
/*****************************************************************************/

/* ***
static KHE_DRS_DOM_TABLE_SUB KheDrsDomTableSubMake(int offset,
  char *debug_str, HA_ARENA a)
{
  KHE_DRS_DOM_TABLE_SUB res;
  HaMake(res, a);
  res->debug_str = debug_str;
  res->offset = offset;
  HaArrayInit(res->indexed_by_l1, a);
  res->max_entry = INT64_MIN;
  return res;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDomTableSubPut(KHE_DRS_DOM_TABLE_SUB dts, int l1, int l2,     */
/*    KHE_COST psi, KHE_COST psi0, bool psi_plus_defined, KHE_COST psi_plus, */
/*    HA_ARENA a)                                                            */
/*                                                                           */
/*  Add cost to dts at position (l1, l2).  This must be the next available.  */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsDomTableSubPut(KHE_DRS_DOM_TABLE_SUB dts, int l1, int l2,
  KHE_COST psi, KHE_COST psi0, bool psi_plus_defined, KHE_COST psi_plus,
  HA_ARENA a)
{
  KHE_DRS_DOM_TABLE_SUB_SUB dtss;  int index;
  index = l1 - dts->offset;
  HnAssert(index >= 0, "KheDrsDomTableSubPut: index (%d - %d) is negative",
    l1, dts->offset);
  if( index < HaArrayCount(dts->indexed_by_l1) )
    dtss = HaArray(dts->indexed_by_l1, index);
  else
  {
    HnAssert(index == HaArrayCount(dts->indexed_by_l1),
      "KheDrsDomTableSubPut: index (%d) unexpectedly large", index);
    dtss = KheDrsDomTableSubSubMake(dts->offset, a);
    HaArrayAddLast(dts->indexed_by_l1, dtss);
  }
  KheDrsDomTableSubSubPut(dtss, l2, psi, psi0, psi_plus_defined, psi_plus);
  if( psi > dts->max_entry )
    dts->max_entry = psi;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_COST_TUPLE KheDrsDomTableSubGet(KHE_DRS_DOM_TABLE_SUB dts,        */
/*    int l1, int l2, KHE_DRS_DOM_TEST dt)                                   */
/*                                                                           */
/*  Retrieve the cost pair stored at position (l1, l2) of dts, which must    */
/*  exist.  Parameter dt is for debugging only.                              */
/*                                                                           */
/*****************************************************************************/

/* ***
static KHE_DRS_COST_TUPLE KheDrsDomTableSubGet(KHE_DRS_DOM_TABLE_SUB dts,
  int l1, int l2, KHE_DRS_DOM_TEST dt)
{
  KHE_DRS_DOM_TABLE_SUB_SUB dtss;  int index;
  index = l1 - dts->offset;
  HnAssert(0 <= index && index < HaArrayCount(dts->indexed_by_l1),
    "KheDrsDomTableSubGet: l1 (%d) out of range (%d .. %d)",
    l1, dts->offset, HaArrayCount(dts->indexed_by_l1) - 1 + dts->offset);
  dtss = HaArray(dts->indexed_by_l1, index);
  return KheDrsDomTableSubSubGet(dtss, l2, dt);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDomTableSubDebugPart(KHE_DRS_DOM_TABLE_SUB dts,               */
/*    int first_l2, int last_l2, int verbosity, int indent, FILE *fp)        */
/*                                                                           */
/*  Debug print of one part of the table for dts/e.                          */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsDomTableSubDebugPart(KHE_DRS_DOM_TABLE_SUB dts,
  int first_l2, int last_l2, int verbosity, int indent, FILE *fp)
{
  KHE_DRS_DOM_TABLE_SUB_SUB dtss;  int l1, l2;  KHE_COST cost;

  ** print header line **
  fprintf(fp, "%*s[          ", indent, "");
  for( l2 = first_l2;  l2 <= last_l2;  l2++ )
    fprintf(fp, "  l2 = %2d", l2 + dts->offset);
  fprintf(fp, "\n");

  ** print header spacer line **
  fprintf(fp, "%*s          +", indent, "");
  for( l2 = first_l2;  l2 <= last_l2;  l2++ )
    fprintf(fp, "---------");
  fprintf(fp, "\n");

  ** print each l1 line for first_l2 .. last_l2 **
  HaArrayForEach(dts->indexed_by_l1, dtss, l1)
  {
    fprintf(fp, "%*s  l1 = %2d |", indent, "", l1 + dts->offset);
    for( l2 = first_l2;  l2 <= last_l2;  l2++ )
    {
      if( l2 < HaArrayCount(dtss->indexed_by_l2) )
      {
	cost = HaArray(dtss->indexed_by_l2, l2).psi;
	fprintf(fp, " %8.5f", KheCostShow(cost));
      }
    }
    fprintf(fp, "\n");
  }

  ** print footer spacer line **
  fprintf(fp, "%*s          +", indent, "");
  for( l2 = first_l2;  l2 <= last_l2;  l2++ )
    fprintf(fp, "---------");
  fprintf(fp, "\n%*s] max %.5f\n", indent, "", KheCostShow(dts->max_entry));
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDomTableSubDebug(KHE_DRS_DOM_TABLE_SUB dts,                   */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of dts onto fp with the given verbosity and indent.          */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsDomTableSubDebug(KHE_DRS_DOM_TABLE_SUB dts,
  int verbosity, int indent, FILE *fp)
{
  KHE_DRS_DOM_TABLE_SUB_SUB dtss;  int l1, max_l2, l2, first_l2, last_l2;

  ** find the maximum value of l2 **
  max_l2 = 0;
  HaArrayForEach(dts->indexed_by_l1, dtss, l1)
  {
    l2 = HaArrayCount(dtss->indexed_by_l2) - 1;
    if( l2 > max_l2 )
      max_l2 = l2;
  }

  ** seven content columns per table **
  fprintf(fp, "%*s[ DomTableSub(%s)\n", indent, "",
    dts->debug_str == NULL ? "-" : dts->debug_str);
  for( first_l2 = 0;  first_l2 <= max_l2;  first_l2 += 6 )
  {
    if( first_l2 > 0 )
      fprintf(fp, "\n");
    last_l2 = min(first_l2 + 5, max_l2);
    KheDrsDomTableSubDebugPart(dts, first_l2, last_l2, verbosity, indent+2, fp);
  }
  fprintf(fp, "%*s]\n", indent, "");
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_DOM_TABLE_SUM"                                        */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TABLE_SUM KheDrsDomTableSumMake(HA_ARENA a)                  */
/*                                                                           */
/*  Make a new, empty dominance table.                                       */
/*                                                                           */
/*****************************************************************************/

/* ***
static KHE_DRS_DOM_TABLE_SUM KheDrsDomTableSumMake(HA_ARENA a)
{
  KHE_DRS_DOM_TABLE_SUM res;
  HaMake(res, a);
  HaArrayInit(res->indexed_by_e, a);
  return res;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDomTableSumPut(KHE_DRS_DOM_TABLE_SUM dt, int offset,          */
/*    int e, int l1, int l2, KHE_COST cost, HA_ARENA a)                      */
/*                                                                           */
/*  Add cost to dt at position (e, l1, l2).  This must be next available.    */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsDomTableSumPut(KHE_DRS_DOM_TABLE_SUM dt, int offset,
  int e, int l1, int l2, KHE_COST cost, HA_ARENA a)
{
  KHE_DRS_DOM_TABLE_SUB dts;  char *debug_str;
  if( DEBUG39 && cost > 0 )
    fprintf(stderr, "  KheDrsDomTableSumPut(dt, offset %d, e %d, l1 %d, l2 %d,"
      " cost %.5f) cost is positive\n", offset, e, l1, l2, KheCostShow(cost));
  ** ***
  HnAssert(cost <= 0, "KheDrsDomTablePut: cost (%.5f) is positive",
    KheCostShow(cost));
  *** **
  HnAssert(e >= 0, "KheDrsDomTablePut: e (%d) is negative", e);
  if( e < HaArrayCount(dt->indexed_by_e) )
    dts = HaArray(dt->indexed_by_e, e);
  else
  {
    HnAssert(e == HaArrayCount(dt->indexed_by_e),
      "KheDrsDomTablePut: e (%d) unexpectedly large", e);
    debug_str = NULL;
    if( DEBUG44 )
      debug_str = HnStringMake(a, "e = %d", e);
    dts = KheDrsDomTableSubMake(offset, debug_str, a);
    HaArrayAddLast(dt->indexed_by_e, dts);
  }
  KheDrsDomTableSubPut(dts, l1, l2, cost, 0, false, 0, a);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TABLE_SUB KheDrsDomTableSumGet(KHE_DRS_DOM_TABLE_SUM dt,     */
/*    int e)                                                                 */
/*                                                                           */
/*  Retrieve the dom table at position e of dt; it must exist.               */
/*                                                                           */
/*****************************************************************************/

/* ***
static KHE_DRS_DOM_TABLE_SUB KheDrsDomTableSumGet(KHE_DRS_DOM_TABLE_SUM dt,
  int e)
{
  HnAssert(0 <= e && e < HaArrayCount(dt->indexed_by_e),
    "KheDrsDomTableSumGet: e (%d) out of range (0 .. %d)",
    e, HaArrayCount(dt->indexed_by_e) - 1);
  return HaArray(dt->indexed_by_e, e);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDomTableSumDebug(KHE_DRS_DOM_TABLE_SUM dt, int verbosity,     */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of dt onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsDomTableSumDebug(KHE_DRS_DOM_TABLE_SUM dt, int verbosity,
  int indent, FILE *fp)
{
  int e;
  KHE_DRS_DOM_TABLE_SUB dts;
  fprintf(fp, "%*s[ DomTableSum\n", indent, "");
  HaArrayForEach(dt->indexed_by_e, dts, e)
    KheDrsDomTableSubDebug(dts, verbosity, indent + 2, fp);
  fprintf(fp, "%*s]\n", indent, "");
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_DOM_TABLE_SEQ"                                        */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TABLE_SEQ KheDrsDomTableSeqMake(HA_ARENA a)                  */
/*                                                                           */
/*  Make a dom table for seq objects.                                        */
/*                                                                           */
/*****************************************************************************/

/* ***
static KHE_DRS_DOM_TABLE_SEQ KheDrsDomTableSeqMake(HA_ARENA a)
{
  KHE_DRS_DOM_TABLE_SEQ res;
  HaMake(res, a);
  HaArrayInit(res->indexed_by_p, a);
  return res;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TABLE_SUB_QR KheDrsDomTableSubQRMake(HA_ARENA a)             */
/*                                                                           */
/*  Make a new, empty KHE_DRS_DOM_TABLE_SUB_QR object.                       */
/*                                                                           */
/*****************************************************************************/

/* ***
static KHE_DRS_DOM_TABLE_SUB_QR KheDrsDomTableSubQRMake(HA_ARENA a)
{
  KHE_DRS_DOM_TABLE_SUB_QR res;
  HaMake(res, a);
  HaArrayInit(res->indexed_by_q, a);
  return res;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TABLE_SUB_R KheDrsDomTableSubRMake(HA_ARENA a)               */
/*                                                                           */
/*  Make a new, empty KHE_DRS_DOM_TABLE_SUB_R object.                        */
/*                                                                           */
/*****************************************************************************/

/* ***
static KHE_DRS_DOM_TABLE_SUB_R KheDrsDomTableSubRMake(HA_ARENA a)
{
  KHE_DRS_DOM_TABLE_SUB_R res;
  HaMake(res, a);
  HaArrayInit(res->indexed_by_r, a);
  return res;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDomTableSeqPut(KHE_DRS_DOM_TABLE_SEQ dt, int p, int q,        */
/*    int r, int l1, int l2, KHE_COST psi, KHE_COST psi0,                    */
/*    bool psi_plus_defined, KHE_COST psi_plus, HA_ARENA a)                  */
/*                                                                           */
/*  Add (psi, psi0, psi_plus) to dt at position (p, q, r, l1, l2).  This is  */
/*  the next available position.  Unlike KheDrsDomTableSumPut, there is no   */
/*  offset parameter here; it would just be 0 anyway.                        */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsDomTableSeqPut(KHE_DRS_DOM_TABLE_SEQ dt, int p, int q,
  int r, int l1, int l2, KHE_COST psi, KHE_COST psi0,
  bool psi_plus_defined, KHE_COST psi_plus, HA_ARENA a)
{
  KHE_DRS_DOM_TABLE_SUB_QR dts_qr;  KHE_DRS_DOM_TABLE_SUB_R dts_r;
  KHE_DRS_DOM_TABLE_SUB dts;  char *debug_str;
  if( DEBUG39 && psi > 0 )
    fprintf(stderr, "  KheDrsDomTableSeqPut(dt, p %d, q %d, r %d, l1 %d, l2 %d,"
      " psi %.5f) cost is positive\n", p, q, r, l1, l2, KheCostShow(psi));
  ** ***
  HnAssert(cost <= 0, "KheDrsDomTableSeqPut: cost (%.5f) is positive",
    KheCostShow(cost));
  *** **

  ** do the p indexing **
  HnAssert(p >= 0, "KheDrsDomTableSeqPut: p (%d) is negative", p);
  if( p < HaArrayCount(dt->indexed_by_p) )
    dts_qr = HaArray(dt->indexed_by_p, p);
  else
  {
    HnAssert(p == HaArrayCount(dt->indexed_by_p),
      "KheDrsDomTablePut: p (%d) unexpectedly large", p);
    dts_qr = KheDrsDomTableSubQRMake(a);
    HaArrayAddLast(dt->indexed_by_p, dts_qr);
  }

  ** do the q indexing **
  HnAssert(q >= 0, "KheDrsDomTableSeqPut: q (%d) is negative", q);
  if( q < HaArrayCount(dts_qr->indexed_by_q) )
    dts_r = HaArray(dts_qr->indexed_by_q, q);
  else
  {
    HnAssert(q == HaArrayCount(dts_qr->indexed_by_q),
      "KheDrsDomTablePut: q (%d) unexpectedly large", q);
    dts_r = KheDrsDomTableSubRMake(a);
    HaArrayAddLast(dts_qr->indexed_by_q, dts_r);
  }

  ** do the r indexing **
  HnAssert(r >= 0, "KheDrsDomTableSeqPut: r (%d) is negative", r);
  if( r < HaArrayCount(dts_r->indexed_by_r) )
    dts = HaArray(dts_r->indexed_by_r, r);
  else
  {
    HnAssert(r == HaArrayCount(dts_r->indexed_by_r),
      "KheDrsDomTablePut: r (%d) unexpectedly large", r);
    debug_str = NULL;
    if( DEBUG44 )
      debug_str = HnStringMake(a, "p = %d, q = %d, r = %d", p, q, r);
    dts = KheDrsDomTableSubMake(0, debug_str, a);
    HaArrayAddLast(dts_r->indexed_by_r, dts);
  }

  ** final put **
  KheDrsDomTableSubPut(dts, l1, l2, psi, psi0, psi_plus_defined, psi_plus, a);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TABLE_SUB KheDrsDomTableSeqGet(KHE_DRS_DOM_TABLE_SEQ dt,     */
/*    int p, int q, int r)                                                   */
/*                                                                           */
/*  Retrieve the dom table at position (p, q, r) of dt; it must exist.       */
/*                                                                           */
/*****************************************************************************/

/* ***
static KHE_DRS_DOM_TABLE_SUB KheDrsDomTableSeqGet(KHE_DRS_DOM_TABLE_SEQ dt,
  int p, int q, int r)
{
  KHE_DRS_DOM_TABLE_SUB_QR dts_qr;
  KHE_DRS_DOM_TABLE_SUB_R dts_r;

  ** do the p indexing **
  HnAssert(0 <= p && p < HaArrayCount(dt->indexed_by_p),
    "KheDrsDomTableSeqGet: p (%d) out of range (0 .. %d)", p,
    HaArrayCount(dt->indexed_by_p) - 1);
  dts_qr = HaArray(dt->indexed_by_p, p);

  ** do the q indexing **
  HnAssert(0 <= q && q < HaArrayCount(dts_qr->indexed_by_q),
    "KheDrsDomTableSeqGet: p = %d, q (%d) out of range (0 .. %d)", p, q,
    HaArrayCount(dts_qr->indexed_by_q) - 1);
  dts_r = HaArray(dts_qr->indexed_by_q, q);

  ** do the r indexing **
  HnAssert(0 <= r && r < HaArrayCount(dts_r->indexed_by_r),
    "KheDrsDomTableSeqGet: p = %d, q = %d, r (%d) out of range (0 .. %d)",
    p, q, r, HaArrayCount(dts_r->indexed_by_r) - 1);
  return HaArray(dts_r->indexed_by_r, r);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDomTableSeqDebug(KHE_DRS_DOM_TABLE_SEQ dt, int verbosity,     */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of dt onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsDomTableSeqDebug(KHE_DRS_DOM_TABLE_SEQ dt, int verbosity,
  int indent, FILE *fp)
{
  int p, q, r;
  KHE_DRS_DOM_TABLE_SUB_QR dts_qr;
  KHE_DRS_DOM_TABLE_SUB_R dts_r;
  KHE_DRS_DOM_TABLE_SUB dts;
  fprintf(fp, "%*s[ DomTableSeq\n", indent, "");
  HaArrayForEach(dt->indexed_by_p, dts_qr, p)
    HaArrayForEach(dts_qr->indexed_by_q, dts_r, q)
    {
      fprintf(fp, "%*s  p = %d, q = %d:\n", indent, "", p, q);
      HaArrayForEach(dts_r->indexed_by_r, dts, r)
	KheDrsDomTableSubDebug(dts, verbosity, indent + 2, fp);
    }
  fprintf(fp, "%*s]\n", indent, "");
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_DIM1_TABLE"                                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DIM1_TABLE KheDrsDim1TableMake(KHE_DYNAMIC_RESOURCE_SOLVER drs)  */
/*                                                                           */
/*  Make a new, empty dim1 table.  It will work out its offset later.        */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DIM1_TABLE KheDrsDim1TableMake(KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DIM1_TABLE res;
  if( HaArrayCount(drs->table1_free_list) > 0 )
  {
    res = HaArrayLastAndDelete(drs->table1_free_list);
    HaArrayClear(res->children);
  }
  else
  {
    HaMake(res, drs->arena);
    HaArrayInit(res->children, drs->arena);
  }
  res->offset = 0;  /* actually undefined when the child array is empty */
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDim1TablePut(KHE_DRS_DIM1_TABLE d1, int index,                */
/*    KHE_DRS_COST_TUPLE val)                                                */
/*                                                                           */
/*  Add val to d1 at index.  If this is the first child, set the offset.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDim1TablePut(KHE_DRS_DIM1_TABLE d1, int index,
  KHE_DRS_COST_TUPLE val)
{
  if( HaArrayCount(d1->children) == 0 )
  {
    /* first child, set offset */
    d1->offset = index;
  }
  else
  {
    /* not first child, must be next though */
    HnAssert(index == d1->offset + HaArrayCount(d1->children),
      "KheDrsDim1TablePut: index is %d when %d expected", index,
      d1->offset + HaArrayCount(d1->children));
  }
  HaArrayAddLast(d1->children, val);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_COST_TUPLE KheDrsDim1TableGet(KHE_DRS_DIM1_TABLE d1, int index)  */
/*                                                                           */
/*  Get the value at this index.  There must be one.                         */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_COST_TUPLE KheDrsDim1TableGet(KHE_DRS_DIM1_TABLE d1, int index)
{
  int pos;
  pos = index - d1->offset;
  HnAssert(0 <= pos && pos < HaArrayCount(d1->children),
    "KheDrsDim1TableGet: index %d out of range %d .. %d", index,
    d1->offset, HaArrayCount(d1->children) + d1->offset - 1);
  return HaArray(d1->children, pos);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDim1TableFree(KHE_DRS_DIM1_TABLE d1,                          */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Free d1.                                                                 */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDim1TableFree(KHE_DRS_DIM1_TABLE d1,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  HaArrayAddLast(drs->table1_free_list, d1);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDim1TableDebug(KHE_DRS_DIM1_TABLE d1, int verbosity,          */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of d1 onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDim1TableDebug(KHE_DRS_DIM1_TABLE d1, int verbosity,
  int indent, FILE *fp)
{
  KHE_DRS_COST_TUPLE ct;  int i;
  fprintf(fp, "%*s[ Dim1 (%d .. %d)\n", indent, "", d1->offset,
    d1->offset + HaArrayCount(d1->children) - 1);
  HaArrayForEach(d1->children, ct, i)
  {
    if( i == 0 )
      fprintf(fp, "%*s", indent + 2, "");
    else if( i % 4 == 0 )
      fprintf(fp, "\n%*s", indent + 2, "");
    else
      fprintf(fp, ", ");
    KheDrsCostTupleDebug(ct, verbosity, -1, fp);
  }
  fprintf(fp, "\n%*s]\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_DIM2_TABLE"                                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DIM2_TABLE KheDrsDim2TableMake(char *debug_str, HA_ARENA a)      */
/*                                                                           */
/*  Make a new, empty dim1 table.  It will work out its offset later.        */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DIM2_TABLE KheDrsDim2TableMake(char *debug_str, HA_ARENA a)
{
  KHE_DRS_DIM2_TABLE res;
  HaMake(res, a);
  res->offset = 0;  /* actually undefined when the child array is empty */
  HaArrayInit(res->children, a);
  res->debug_str = debug_str;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDim2TablePut(KHE_DRS_DIM2_TABLE d2, int index,                */
/*    KHE_DRS_DIM1_TABLE val)                                                */
/*                                                                           */
/*  Add val to d2 at index.  If this is the first child, set the offset.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDim2TablePut(KHE_DRS_DIM2_TABLE d2, int index2,
  int index1, KHE_DRS_COST_TUPLE ct, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DIM1_TABLE d1;  int pos;

  /* set offset if this is the first insertion */
  if( HaArrayCount(d2->children) == 0 )
    d2->offset = index2;

  /* make sure index2 is within, or one step beyond, the current range */
  pos = index2 - d2->offset;
  HnAssert(0 <= pos && pos <= HaArrayCount(d2->children),
    "KheDrsDim2TablePut: index2 %d is out of range %d .. %d", index2,
    d2->offset, d2->offset + HaArrayCount(d2->children));

  /* find or make-and-add d1, the sub-array to insert into */
  if( pos < HaArrayCount(d2->children) )
    d1 = HaArray(d2->children, pos);
  else
  {
    d1 = KheDrsDim1TableMake(drs);
    HaArrayAddLast(d2->children, d1);
  }
  
  /* do the insertion */
  KheDrsDim1TablePut(d1, index1, ct);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DIM1_TABLE KheDrsDim2TableGet(KHE_DRS_DIM2_TABLE d2, int index2) */
/*                                                                           */
/*  Get the value at this index.  There must be one.                         */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DIM1_TABLE KheDrsDim2TableGet(KHE_DRS_DIM2_TABLE d2, int index2)
{
  int pos;
  pos = index2 - d2->offset;
  HnAssert(0 <= pos && pos < HaArrayCount(d2->children),
    "KheDrsDim2TableGet: index2 %d out of range %d .. %d", index2,
    d2->offset, HaArrayCount(d2->children) + d2->offset - 1);
  return HaArray(d2->children, pos);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_COST_TUPLE KheDrsDim2TableGet2(KHE_DRS_DIM2_TABLE d2,            */
/*    int index2, int index1)                                                */
/*                                                                           */
/*  Get the cost tuple at d2[index2, index1].                                */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_COST_TUPLE KheDrsDim2TableGet2(KHE_DRS_DIM2_TABLE d2,
  int index2, int index1)
{
  return KheDrsDim1TableGet(KheDrsDim2TableGet(d2, index2), index1);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDim2TableClear(KHE_DRS_DIM2_TABLE d2,                         */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Clear d2, placing all its d1's onto a free list.                         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDim2TableClear(KHE_DRS_DIM2_TABLE d2,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DIM1_TABLE d1;  int i;
  HaArrayForEach(d2->children, d1, i)
    KheDrsDim1TableFree(d1, drs);
  HaArrayClear(d2->children);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDim2TableDebug(KHE_DRS_DIM2_TABLE d2, int verbosity,          */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of d2 onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDim2TableDebug(KHE_DRS_DIM2_TABLE d2, int verbosity,
  int indent, FILE *fp)
{
  KHE_DRS_DIM1_TABLE d1;  int i;
  fprintf(fp, "%*s[ Dim2 %s(%d .. %d)\n", indent, "",
    d2->debug_str == NULL ? "" : d2->debug_str,
    d2->offset, d2->offset + HaArrayCount(d2->children) - 1);
  HaArrayForEach(d2->children, d1, i)
    KheDrsDim1TableDebug(d1, verbosity, indent + 2, fp);
  fprintf(fp, "%*s]\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_DIM3_TABLE"                                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DIM3_TABLE KheDrsDim3TableMake(char *debug_str, HA_ARENA a)      */
/*                                                                           */
/*  Make a new, empty dim3 table.  It will work out its offset later.        */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DIM3_TABLE KheDrsDim3TableMake(char *debug_str, HA_ARENA a)
{
  KHE_DRS_DIM3_TABLE res;
  HaMake(res, a);
  res->offset = 0;  /* actually undefined when the child array is empty */
  HaArrayInit(res->children, a);
  res->debug_str = debug_str;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDim3TablePut(KHE_DRS_DIM3_TABLE d3, int index3, int index2,   */
/*    int index1, KHE_DRS_COST_TUPLE ct, KHE_DYNAMIC_RESOURCE_SOLVER drs)    */
/*                                                                           */
/*  Add c3 to d3.  If this is the first insertion, set the offset.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDim3TablePut(KHE_DRS_DIM3_TABLE d3, int index3, int index2,
  int index1, KHE_DRS_COST_TUPLE ct, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DIM2_TABLE d2;  int pos;  HA_ARENA a;  char *debug_str;

  /* set offset if this is the first insertion */
  if( HaArrayCount(d3->children) == 0 )
    d3->offset = index3;

  /* make sure index3 is within, or one step beyond, the current range */
  pos = index3 - d3->offset;
  HnAssert(0 <= pos && pos <= HaArrayCount(d3->children),
    "KheDrsDim3TablePut: index3 %d is out of range %d .. %d", index3,
    d3->offset, d3->offset + HaArrayCount(d3->children));

  /* find or make-and-add d2, the sub-array to insert into */
  if( pos < HaArrayCount(d3->children) )
    d2 = HaArray(d3->children, pos);
  else
  {
    a = HaArrayArena(d3->children);
    debug_str = HnStringMake(a, "%s:%d", d3->debug_str, index3);
    d2 = KheDrsDim2TableMake(debug_str, a);
    HaArrayAddLast(d3->children, d2);
  }
  
  /* do the insertion */
  KheDrsDim2TablePut(d2, index2, index1, ct, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DIM2_TABLE KheDrsDim3TableGet(KHE_DRS_DIM3_TABLE d3, int index3) */
/*                                                                           */
/*  Get the value at this index.  There must be one.                         */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DIM2_TABLE KheDrsDim3TableGet(KHE_DRS_DIM3_TABLE d3, int index3)
{
  int pos;
  pos = index3 - d3->offset;
  HnAssert(0 <= pos && pos < HaArrayCount(d3->children),
    "KheDrsDim3TableGet: index %d out of range %d .. %d", index3,
    d3->offset, HaArrayCount(d3->children) + d3->offset - 1);
  return HaArray(d3->children, pos);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDim3TableDebug(KHE_DRS_DIM3_TABLE d3, int verbosity,          */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of d3 onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDim3TableDebug(KHE_DRS_DIM3_TABLE d3, int verbosity,
  int indent, FILE *fp)
{
  KHE_DRS_DIM2_TABLE d2;  int i;
  fprintf(fp, "%*s[ Dim3 %s(%d .. %d)\n", indent, "",
    d3->debug_str == NULL ? "" : d3->debug_str,
    d3->offset, d3->offset + HaArrayCount(d3->children) - 1);
  HaArrayForEach(d3->children, d2, i)
    KheDrsDim2TableDebug(d2, verbosity, indent + 2, fp);
  fprintf(fp, "%*s]\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_DIM4_TABLE"                                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DIM4_TABLE KheDrsDim4TableMake(char *debug_str, HA_ARENA a)      */
/*                                                                           */
/*  Make a new, empty Dim4 table.  It will work out its offset later.        */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DIM4_TABLE KheDrsDim4TableMake(char *debug_str, HA_ARENA a)
{
  KHE_DRS_DIM4_TABLE res;
  HaMake(res, a);
  res->offset = 0;  /* actually undefined when the child array is empty */
  HaArrayInit(res->children, a);
  res->debug_str = debug_str;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDim4TablePut(KHE_DRS_DIM4_TABLE d4, int index4, int index3,   */
/*    int index2, int index1, KHE_DRS_COST_TUPLE ct,                         */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Add ct to d4.  If this is the first insertion, set the offset.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDim4TablePut(KHE_DRS_DIM4_TABLE d4, int index4, int index3,
  int index2, int index1, KHE_DRS_COST_TUPLE ct,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DIM3_TABLE d3;  int pos;  HA_ARENA a;  char *debug_str;

  /* set offset if this is the first insertion */
  if( HaArrayCount(d4->children) == 0 )
    d4->offset = index4;

  /* make sure index4 is within, or one step beyond, the current range */
  pos = index4 - d4->offset;
  HnAssert(0 <= pos && pos <= HaArrayCount(d4->children),
    "KheDrsDim4TablePut: index4 %d is out of range %d .. %d", index4,
    d4->offset, d4->offset + HaArrayCount(d4->children));

  /* find or make-and-add d3, the sub-array to insert into */
  if( pos < HaArrayCount(d4->children) )
    d3 = HaArray(d4->children, pos);
  else
  {
    a = HaArrayArena(d4->children);
    debug_str = HnStringMake(a, "%s:%d", d4->debug_str, index4);
    d3 = KheDrsDim3TableMake(debug_str, a);
    HaArrayAddLast(d4->children, d3);
  }
  
  /* do the insertion */
  KheDrsDim3TablePut(d3, index3, index2, index1, ct, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DIM3_TABLE KheDrsDim4TableGet(KHE_DRS_DIM4_TABLE d4, int index4) */
/*                                                                           */
/*  Get the value at this index.  There must be one.                         */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DIM3_TABLE KheDrsDim4TableGet(KHE_DRS_DIM4_TABLE d4, int index4)
{
  int pos;
  pos = index4 - d4->offset;
  HnAssert(0 <= pos && pos < HaArrayCount(d4->children),
    "KheDrsDim4TableGet: index %d out of range %d .. %d", index4,
    d4->offset, HaArrayCount(d4->children) + d4->offset - 1);
  return HaArray(d4->children, pos);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_COST_TUPLE KheDrsDim4TableGet4(KHE_DRS_DIM4_TABLE d4,            */
/*    int index4, int index3, int index2, int index1)                        */
/*                                                                           */
/*  Get the cost tuple at d4[index4, index3, index2, index1].                */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_COST_TUPLE KheDrsDim4TableGet4(KHE_DRS_DIM4_TABLE d4,
  int index4, int index3, int index2, int index1)
{
  return KheDrsDim2TableGet2(KheDrsDim3TableGet(KheDrsDim4TableGet(d4, index4),
    index3), index2, index1);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDim4TableDebug(KHE_DRS_DIM4_TABLE d4, int verbosity,          */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of d4 onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDim4TableDebug(KHE_DRS_DIM4_TABLE d4, int verbosity,
  int indent, FILE *fp)
{
  KHE_DRS_DIM3_TABLE d3;  int i;
  fprintf(fp, "%*s[ Dim4 %s(%d .. %d)\n", indent, "",
    d4->debug_str == NULL ? "" : d4->debug_str,
    d4->offset, d4->offset + HaArrayCount(d4->children) - 1);
  HaArrayForEach(d4->children, d3, i)
    KheDrsDim3TableDebug(d3, verbosity, indent + 2, fp);
  fprintf(fp, "%*s]\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_DIM5_TABLE"                                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DIM5_TABLE KheDrsDim5TableMake(char *debug_str, HA_ARENA a)      */
/*                                                                           */
/*  Make a new, empty Dim5 table.  It will work out its offset later.        */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DIM5_TABLE KheDrsDim5TableMake(char *debug_str, HA_ARENA a)
{
  KHE_DRS_DIM5_TABLE res;
  HaMake(res, a);
  res->offset = 0;  /* actually undefined when the child array is empty */
  HaArrayInit(res->children, a);
  res->debug_str = debug_str;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDim5TablePut(KHE_DRS_DIM5_TABLE d5, int index5, int index4,   */
/*    int index3, int index2, int index1, KHE_DRS_COST_TUPLE ct,             */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Add ct to d5.  If this is the first insertion, set the offset.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDim5TablePut(KHE_DRS_DIM5_TABLE d5, int index5, int index4,
  int index3, int index2, int index1, KHE_DRS_COST_TUPLE ct,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DIM4_TABLE d4;  int pos;  char *debug_str;  HA_ARENA a;

  /* set offset if this is the first insertion */
  if( HaArrayCount(d5->children) == 0 )
    d5->offset = index5;

  /* make sure index5 is within, or one step beyond, the current range */
  pos = index5 - d5->offset;
  HnAssert(0 <= pos && pos <= HaArrayCount(d5->children),
    "KheDrsDim5TablePut: index5 %d is out of range %d .. %d", index5,
    d5->offset, d5->offset + HaArrayCount(d5->children));

  /* find or make-and-add d4, the sub-array to insert into */
  if( pos < HaArrayCount(d5->children) )
    d4 = HaArray(d5->children, pos);
  else
  {
    a = HaArrayArena(d5->children);
    debug_str = HnStringMake(a, "%s:%d", d5->debug_str, index5);
    d4 = KheDrsDim4TableMake(debug_str, a);
    HaArrayAddLast(d5->children, d4);
  }
  
  /* do the insertion */
  KheDrsDim4TablePut(d4, index4, index3, index2, index1, ct, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DIM4_TABLE KheDrsDim5TableGet(KHE_DRS_DIM5_TABLE d5, int index5) */
/*                                                                           */
/*  Get the value at this index.  There must be one.                         */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DIM4_TABLE KheDrsDim5TableGet(KHE_DRS_DIM5_TABLE d5, int index5)
{
  int pos;
  pos = index5 - d5->offset;
  HnAssert(0 <= pos && pos < HaArrayCount(d5->children),
    "KheDrsDim5TableGet: index %d out of range %d .. %d", index5,
    d5->offset, HaArrayCount(d5->children) + d5->offset - 1);
  return HaArray(d5->children, pos);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DIM2_TABLE KheDrsDim5TableGet3(KHE_DRS_DIM5_TABLE d5,            */
/*    int index5, int index4, int index3)                                    */
/*                                                                           */
/*  Get the two-dimensional table that results from applying three indexes   */
/*  to five-dimensional table d5.                                            */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DIM2_TABLE KheDrsDim5TableGet3(KHE_DRS_DIM5_TABLE d5,
  int index5, int index4, int index3)
{
  return KheDrsDim3TableGet(KheDrsDim4TableGet(KheDrsDim5TableGet(d5, index5),
    index4), index3);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDim5TableDebug(KHE_DRS_DIM5_TABLE d5, int verbosity,          */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of d5 onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDim5TableDebug(KHE_DRS_DIM5_TABLE d5, int verbosity,
  int indent, FILE *fp)
{
  KHE_DRS_DIM4_TABLE d4;  int i;
  fprintf(fp, "%*s[ Dim5 %s(%d .. %d)\n", indent, "",
    d5->debug_str == NULL ? "" : d5->debug_str,
    d5->offset, d5->offset + HaArrayCount(d5->children) - 1);
  HaArrayForEach(d5->children, d4, i)
    KheDrsDim4TableDebug(d4, verbosity, indent + 2, fp);
  fprintf(fp, "%*s]\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_DOM_TEST_TYPE"                                        */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  char *KheDrsDomTestTypeShow(KHE_DRS_DOM_TEST_TYPE type)                  */
/*                                                                           */
/*  Return a one-character string representation of type.                    */
/*                                                                           */
/*****************************************************************************/

static char *KheDrsDomTestTypeShow(KHE_DRS_DOM_TEST_TYPE type)
{
  switch( type )
  {
    case KHE_DRS_DOM_TEST_UNUSED:		return "!!";
    case KHE_DRS_DOM_TEST_STRONG:		return "ST";
    case KHE_DRS_DOM_TEST_TRADEOFF:		return "TR";
    case KHE_DRS_DOM_TEST_UNIFORM:		return "UN";
    case KHE_DRS_DOM_TEST_CORR1_PARENT:		return "1P";
    case KHE_DRS_DOM_TEST_CORR1_CHILD:		return "1C";
    case KHE_DRS_DOM_TEST_CORR2_CHILD:		return "2C";
    case KHE_DRS_DOM_TEST_CORR3_FIRST:		return "3F";
    case KHE_DRS_DOM_TEST_CORR3_MID:		return "3M";
    case KHE_DRS_DOM_TEST_CORR3_LAST:		return "3L";
    case KHE_DRS_DOM_TEST_CORR4_FIRST:		return "4F";
    case KHE_DRS_DOM_TEST_CORR4_MID:		return "4M";
    case KHE_DRS_DOM_TEST_CORR4_LAST:		return "4L";
    default:					return "??";
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_DOM_TEST"                                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsDomTestMake(KHE_DRS_DOM_TEST_TYPE type,           */
/*    KHE_DRS_EXPR e, bool allow_zero, int min_limit, int max_limit,         */
/*    int a, int b, KHE_COST tradeoff_cost,                                  */
/*    KHE_DRS_DIM2_TABLE main_dom_table2, KHE_DRS_DIM4_TABLE corr_dom_table4,*/
/*    KHE_MONITOR m, KHE_DYNAMIC_RESOURCE_SOLVER drs)                        */
/*                                                                           */
/*  Make and return a new dominance test object with these attributes.       */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DOM_TEST KheDrsDomTestMake(KHE_DRS_DOM_TEST_TYPE type,
  KHE_DRS_EXPR e, bool allow_zero, int min_limit, int max_limit,
  int a, int b, KHE_COST tradeoff_cost,
  KHE_DRS_DIM2_TABLE main_dom_table2, KHE_DRS_DIM4_TABLE corr_dom_table4,
  KHE_MONITOR m, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DOM_TEST res;
  if( HaArrayCount(drs->dom_test_free_list) > 0 )
    res = HaArrayLastAndDelete(drs->dom_test_free_list);
  else
    HaMake(res, drs->arena);
  res->type = type;
  res->correlated_delta = 0;
  res->expr = e;
  res->allow_zero = allow_zero;
  res->min_limit = min_limit;
  res->max_limit = max_limit;
  res->a = a;
  res->b = b;
  res->tradeoff_cost = tradeoff_cost;
  res->main_dom_table2 = main_dom_table2;
  res->corr_dom_table4 = corr_dom_table4;
  res->monitor = m;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsDomTestDominatesStrong(KHE_DRS_DOM_TEST dt, int val1,int val2)*/
/*                                                                           */
/*  Return true if val1 dominates val2, according to dt.                     */
/*                                                                           */
/*  Implementation note.  This formula is derived in the Appendix to the     */
/*  User's Guide, where it is called dom(val1, val2).                        */
/*                                                                           */
/*****************************************************************************/

static bool tilde(KHE_DRS_DOM_TEST dt, int val1, int val2)
{
  return (dt->allow_zero ? (val1 == 0) == (val2 == 0) : true);
}

static bool KheDrsDomMax(KHE_DRS_DOM_TEST dt, int val1, int val2)
{
  return val1 <= dt->a || val1 <= val2;
}

static bool KheDrsDomMin(KHE_DRS_DOM_TEST dt, int val1, int val2)
{
  return val1 >= dt->b || (tilde(dt, val1, val2) && val1 >= val2);
}

static bool KheDrsDomTestDominatesStrong(KHE_DRS_DOM_TEST dt, int val1,int val2)
{
  return KheDrsDomMax(dt, val1, val2) && KheDrsDomMin(dt, val1, val2);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsTryTradeoff(KHE_DRS_DOM_TEST dt, int delta_val,               */
/*    KHE_COST *cost1, KHE_COST cost2)                                       */
/*                                                                           */
/*  If dt->tradeoff_allowed, carry out tradeoff dominance with value change  */
/*  delta_val.  Otherwise return false.                                      */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsTryTradeoff(KHE_DRS_DOM_TEST dt, int delta_val,
  KHE_COST *avail_cost)
{
  HnAssert(delta_val >= 0, "KheDrsTryTradeoff internal error");
  *avail_cost -= dt->tradeoff_cost * delta_val;
  return *avail_cost >= 0;
}

/* ***
static bool KheDrsTryTradeoff(KHE_DRS_DOM_TEST dt, int delta_val,
  KHE_COST *cost1, KHE_COST cost2)
{
  KHE_COST cost;
  cost = *cost1 + dt->tradeoff_cost * delta_val;
  return cost <= cost2 ? (*cost1 = cost, true) : false;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsDomTestDominatesTradeoff(KHE_DRS_DOM_TEST dt,                 */
/*    int val1, KHE_COST *cost1, int val2, KHE_COST cost2)                   */
/*                                                                           */
/*  Return true if val1 dominates val2, according to dt, allowing for a      */
/*  tradeoff.                                                                */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsDomTestDominatesTradeoff(KHE_DRS_DOM_TEST dt,
  int val1, int val2, KHE_COST *avail_cost)
{
  if( KheDrsDomMax(dt, val1, val2) )
  {
    if( KheDrsDomMin(dt, val1, val2) )
      return true;
    else if( dt->allow_zero )
      return false;
    else
      return KheDrsTryTradeoff(dt, val2 - val1, avail_cost);
  }
  else
  {
    if( KheDrsDomMin(dt, val1, val2) )
      return KheDrsTryTradeoff(dt, val1 - val2, avail_cost);
    else
      return false;
  }
}


/* *** old version
static bool KheDrsDomTestDominatesTradeoff(KHE_DRS_DOM_TEST dt,
  int val1, KHE_COST *cost1, int val2, KHE_COST cost2)
{
  if( dt->tradeoff_allowed )
  {
    ** try it with tradeoff **
    if( KheDrsDomMax(dt, val1, val2) )
    {
      if( KheDrsDomMin(dt, val1, val2) )
	return true;
      else if( dt->allow_zero )
	return false;
      else
	return KheDrsTryTradeoff(dt, val2 - val1, cost1, cost2);
    }
    else
    {
      if( KheDrsDomMin(dt, val1, val2) )
	return KheDrsTryTradeoff(dt, val1 - val2, cost1, cost2);
      else
	return false;
    }
  }
  else
  {
    ** try it without tradeoff **
    return KheDrsDomTestDominatesStrong(dt, val1, val2);
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsTryTradeoffNew(KHE_DRS_DOM_TEST dt, int delta_val,            */
/*    KHE_COST *avail_tradeoff_cost)                                         */
/*                                                                           */
/*  If change in value delta_val is not too great for dominance, return      */
/*  true and reduce *avail_tradeoff_cost to the revised amount of available  */
/*  cost.  Otherwise return false.                                           */
/*                                                                           */
/*****************************************************************************/

/* *** unused
static bool KheDrsTryTradeoffNew(KHE_DRS_DOM_TEST dt, int delta_val,
  KHE_COST *avail_tradeoff_cost)
{
  KHE_COST new_avail;
  new_avail = *avail_tradeoff_cost - dt->tradeoff_cost * delta_val;
  return new_avail >= 0 ? (*avail_tradeoff_cost = new_avail, true) : false;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsDomTestDominatesNew(KHE_DRS_DOM_TEST dt,                      */
/*    int val1, int val2, KHE_COST *avail_tradeoff_cost)                     */
/*                                                                           */
/*  Return true if val1 dominates val2.  If tradeoff is used to justify      */
/*  this, reduce *avail_tradeoff_cost to the revised amount of available     */
/*  cost.                                                                    */
/*                                                                           */
/*****************************************************************************/
/* ***
static bool KheDrsDomMax(KHE_DRS_DOM_TEST dt, int val1, int val2);
static bool KheDrsDomMin(KHE_DRS_DOM_TEST dt, int val1, int val2);
*** */

/* *** unused
static bool KheDrsDomTestDominatesNew(KHE_DRS_DOM_TEST dt,
  int val1, int val2, KHE_COST *avail_tradeoff_cost)
{
  if( dt->tradeoff_allowed )
  {
    ** try it with tradeoff **
    if( KheDrsDomMax(dt, val1, val2) )
    {
      if( KheDrsDomMin(dt, val1, val2) )
	return true;
      else if( dt->allow_zero )
	return false;
      else
	return KheDrsTryTradeoffNew(dt, val2 - val1, avail_tradeoff_cost);
    }
    else
    {
      if( KheDrsDomMin(dt, val1, val2) )
	return KheDrsTryTradeoffNew(dt, val1 - val2, avail_tradeoff_cost);
      else
	return false;
    }
  }
  else
  {
    ** try it without tradeoff **
    return KheDrsDomTestDominatesStrong(dt, val1, val2);
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDomTestDominatesUniform(KHE_DRS_DOM_TEST dt,                  */
/*    int val1, int val2, KHE_COST *avail_cost)                              */
/*                                                                           */
/*  Check whether val1 dominates val2 using uniform dominance, updating      */
/*  *avail_cost by the change in available cost.                             */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDomTestDominatesUniform(KHE_DRS_DOM_TEST dt,
  int val1, int val2, KHE_COST *avail_cost)
{
  if( DEBUG60 || DEBUG44_MONITOR(dt->monitor) )
    fprintf(stderr, "  [ KheDrsDomTestDominatesUniform(%s, %d, %d)\n",
      dt->monitor != NULL ? KheMonitorId(dt->monitor) : "-", val1, val2);
  *avail_cost += KheDrsDim2TableGet2(dt->main_dom_table2, val1, val2).psi;
  if( DEBUG60 || DEBUG44_MONITOR(dt->monitor) )
    fprintf(stderr, "  ]\n");
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_COST KheCostMin(KHE_COST cost1, KHE_COST cost2)                      */
/*                                                                           */
/*  Return the minimum of cost1 and cost2.                                   */
/*                                                                           */
/*****************************************************************************/

/* *** no longer used
static KHE_COST KheCostMin(KHE_COST cost1, KHE_COST cost2)
{
  return cost1 <= cost2 ? cost1 : cost2;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsCorrMin(KHE_DRS_DOM_TEST dt, int u1, int u2,                  */
/*    int v1, int v2, KHE_COST *avail_cost)                                  */
/*                                                                           */
/*  Helper function for KheDrsDomTestDominatesCorr1 below.  It tries two      */
/*  tests, one for u1 vs u2 and one for v1 vs v2, and sets *avail_cost to    */
/*  the smaller of the two results.                                          */
/*                                                                           */
/*****************************************************************************/

/* *** no longer used
static bool KheDrsCorrMin(KHE_DRS_DOM_TEST dt, int u1, int u2,
  int v1, int v2, KHE_COST *avail_cost)
{
  KHE_COST u_cost, v_cost;
  u_cost = KheDrsDim2TableGet2(dt->main_dom_table2, u1, u2).psi;
  v_cost = KheDrsDim2TableGet2(dt->main_dom_table2, v1, v2).psi;
  ** u_cost = KheDrsDomTableSubGet(dt->main_dom_table2, u1, u2, dt).psi; **
  ** v_cost = KheDrsDomTableSubGet(dt->main_dom_table2, v1, v2, dt).psi; **
  *avail_cost += KheCostMin(u_cost, v_cost);
  return *avail_cost >= 0;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsDomTestDominatesCorr1(KHE_DRS_DOM_TEST dt,                    */
/*    int a1, int v1, int a2, int v2, KHE_COST *avail_cost)                  */
/*                                                                           */
/*  Return true if (a1, v1) dominates (a2, v2).                              */
/*                                                                           */
/*****************************************************************************/

/* *** this is rubbish! what was I thinking?
static bool KheDrsDomTestDominatesCorr1(KHE_DRS_DOM_TEST dt,
  int a1, int v1, int a2, int v2, KHE_COST *avail_cost)
{
  if( a1 == 0 )
  {
    if( a2 == 0 )
      return KheDrsCorrMin(dt, v1, v2, v1 + 1, v2 + 1, avail_cost);
    else if( a2 == 1 )
      return KheDrsCorrMin(dt, v1, v2 + 1, v1 + 1, v2 + 1, avail_cost);
  }
  else if( a1 == 1 )
  {
    if( a2 == 0 )
      return KheDrsCorrMin(dt, v1 + 1, v2, v1 + 1, v2 + 1, avail_cost);
    else if( a2 == 1 )
      return KheDrsDomTestDominatesUniform(dt, v1 + 1, v2 + 1, avail_cost);
  }

  ** if we get to this point we are in trouble **
  HnAbort("KheDrsDomTestDominatesCorr1 internal error: a1 %d, a2 %d", a1, a2);
  return false;  ** keep compiler happy **
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsDomTestDominatesCorr3(KHE_DRS_DOM_TEST dt,                    */
/*    int v1, int v2, int other_v1, int other_v2, KHE_COST *avail_cost)      */
/*    int a1, int v1, int a2, int v2, KHE_COST *avail_cost)                  */
/*                                                                           */
/*  Return true if this is a CORR3_FIRST domination.                         */
/*                                                                           */
/*****************************************************************************/

/* *** no longer used
static bool KheDrsDomTestDominatesCorr3(KHE_DRS_DOM_TEST dt,
  int v1, int v2, int other_v1, int other_v2, KHE_COST *avail_cost)
{
  KHE_DRS_COST_TUPLE mp, other_mp;
  mp = KheDrsDomTableSubGet(dt->main_dom_table2, v1, v2, dt);
  other_mp = KheDrsDomTableSubGet(dt->main_dom_table2, other_v1, other_v2, dt);
  *avail_cost += KheCostMin(mp.psi + other_mp.psi0, mp.psi0 + other_mp.psi);
  return *avail_cost >= 0;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsDomTestIsStrictEquality(KHE_DRS_DOM_TEST dom_test)            */
/*                                                                           */
/*  Return true if dom_test amounts to a strict equality test.  As shown     */
/*  in the documentation, the allow_zero flag does not affect the result.    */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsDomTestIsStrictEquality(KHE_DRS_DOM_TEST dom_test)
{
  return dom_test->a < 0 && dom_test->b >= INT_MAX;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDomTestDominatingSet(KHE_DRS_DOM_TEST dt, int x,              */
/*    int *from_val, int *to_val, bool *also_zero)                           */
/*                                                                           */
/*  Set [*from_val, *to_val] and *also_zero to the set of non-negative       */
/*  values that dominate x under dt.  That is, set it to                     */
/*                                                                           */
/*    { v | v >= 0 && KheDrsDomTestDominatesStrong(dt, v, x) }               */
/*                                                                           */
/*  The formula for this is derived in the User's Guide.                     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDomTestDominatingSet(KHE_DRS_DOM_TEST dt, int x,
  int *from_val, int *to_val, bool *also_zero)
{
  if( dt->allow_zero && x == 0 )
  {
    *from_val = max(0, dt->b);
    *to_val = max(dt->a, 0);
    *also_zero = (*from_val > 0);
  }
  else
  {
    *from_val = min(dt->b, x);
    if( *from_val < 0 )  *from_val = 0;
    *to_val = max(dt->a, x);
    *also_zero = false;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDomTestDominatedSet(KHE_DRS_DOM_TEST dt, int x,               */
/*    int *from_val, int *to_val, bool *also_zero)                           */
/*                                                                           */
/*  Set [*from_val, *to_val] and *also_zero to the set of non-negative       */
/*  values that x dominates under dt.  That is, set it to                    */
/*                                                                           */
/*    { v | v >= 0 && KheDrsDomTestDominatesStrong(dt, x, val) }             */
/*                                                                           */
/*  This formula is derived in the Appendix to the User's Guide.             */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDomTestDominatedSet(KHE_DRS_DOM_TEST dt, int x,
  int *from_val, int *to_val, bool *also_zero)
{
  if( dt->allow_zero && x >= 1 )
  {
    *from_val = (x <= dt->a ? 1 : max(1, x));
    *to_val = (x >= dt->b ? INT_MAX : x);
    if( x <= max(dt->a, 0) && x >= dt->b )
    {
      /* make sure 0 is in the set */
      if( *from_val == 1 )
	*from_val = 0;
      *also_zero = (*from_val > 0);
    }
    else
      *also_zero = false;
  }
  else
  {
    *from_val = (x <= dt->a ? 0 : x);
    *to_val = (x >= dt->b ? INT_MAX : x);
    *also_zero = false;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheDrsDomTestShow(KHE_DRS_DOM_TEST dt, char buff[100])             */
/*                                                                           */
/*  Show dt, using buff for scratch storage.                                 */
/*                                                                           */
/*****************************************************************************/

/* *** perfectly good, but currently unused
static char *KheDrsDomTestShow(KHE_DRS_DOM_TEST dt, char buff[100])
{
  snprintf(buff, 100, "[Z %s, a %d, b %d]", bool_show(dt->allow_zero),
    dt->a, dt->b);
  return buff;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsSet(char *label, int x, int from_val, int to_val,             */
/*    bool also_zero, int indent, FILE *fp)                                  */
/*                                                                           */
/*  Print label(x) = {set} onto fp with the given indent.                    */
/*                                                                           */
/*****************************************************************************/

/* *** perfectly good, but currently unused
static void KhePrintInt(int val, FILE *fp)
{
  if( val == INT_MAX )
    fprintf(fp, "inf");
  else
    fprintf(fp, "%d", val);
}

static void KheDrsSet(char *label, int x, int from_val, int to_val,
  bool also_zero, int indent, FILE *fp)
{
  fprintf(fp, "%*s%s(%d) = {%s", indent, "", label, x, also_zero ? "0, " : "");
  KhePrintInt(from_val, fp);
  fprintf(fp, " .. ");
  KhePrintInt(to_val, fp);
  fprintf(fp, "}\n");
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDomTestDebug(KHE_DRS_DOM_TEST dom_test, int verbosity,        */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of dom_test, including dominating and dominated.             */
/*                                                                           */
/*****************************************************************************/

/* *** perfectly good, but currently unused
static void KheDrsDomTestDebug(KHE_DRS_DOM_TEST dom_test, int verbosity,
  int indent, FILE *fp)
{
  char buff[100];  int x, from_val, to_val;  bool also_zero;
  if( indent < 0 )
    fprintf(fp, "%s", KheDrsDomTestShow(dom_test, buff));
  else
  {
    fprintf(fp, "%*s[ %s\n", indent, "", KheDrsDomTestShow(dom_test, buff));
    if( verbosity >= 3 )
    {
      for( x = 0;  x < 10;  x++ )
      {
	KheDrsDomTestDominatingSet(dom_test, x, &from_val, &to_val, &also_zero);
        KheDrsSet("dominating", x, from_val, to_val, also_zero, indent + 2, fp);
      }
      for( x = 0;  x < 10;  x++ )
      {
	KheDrsDomTestDominatedSet(dom_test, x, &from_val, &to_val, &also_zero);
        KheDrsSet("dominated", x, from_val, to_val, also_zero, indent + 2, fp);
      }
    }
    fprintf(fp, "%*s]\n", indent, "");
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDomTestDebug(KHE_DRS_DOM_TEST dt, int verbosity,              */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of dt onto fp with the given verbosity and indent.           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsDomTestDebug(KHE_DRS_DOM_TEST dt, int verbosity,
  int indent, FILE *fp)
{
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  fprintf(fp, "DomTest(%s %s, main %s, corr %s)",
    KheDrsDomTestTypeShow(dt->type),
    dt->monitor != NULL ? KheMonitorId(dt->monitor) : "-",
    dt->main_dom_table2 != NULL ? dt->main_dom_table2->debug_str : "-",
    dt->corr_dom_table4 != NULL ? dt->corr_dom_table4->debug_str : "-");
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Major category "constraints (i.e. expressions)"                          */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_MONITOR_INFO"                                         */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_MONITOR_INFO KheDrsMonitorInfoMakeAndAdd(KHE_MONITOR m,          */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new monitor info object for m, add it to drs->all_monitor_infos,  */
/*  and return it.                                                           */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_MONITOR_INFO KheDrsMonitorInfoMakeAndAdd(KHE_MONITOR m,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_MONITOR_INFO res;
  HaMake(res, drs->arena);
  res->monitor = m;
  res->rerun_open_and_search_cost = 0;
  res->rerun_open_and_close_cost = 0;
  HaArrayAddLast(drs->all_monitor_infos, res);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheDrsMonitorInfoId(KHE_DRS_MONITOR_INFO mi)                       */
/*                                                                           */
/*  Monitor info id.                                                         */
/*                                                                           */
/*****************************************************************************/

static char *KheDrsMonitorInfoId(KHE_DRS_MONITOR_INFO mi)
{
  return KheMonitorId(mi->monitor);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsMonitorInfoDoUpdateRerunCost(KHE_DRS_MONITOR_INFO mi,         */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, char *field, char *operation,         */
/*    int child_index, KHE_COST *cost_field, char *fmt, va_list ap)          */
/*                                                                           */
/*  Do the actual work of KheDrsMonitorInfoUpdateRerunCost.                  */
/*                                                                           */
/*****************************************************************************/

static void KheDrsMonitorInfoDoUpdateRerunCost(KHE_DRS_MONITOR_INFO mi,
  char *field, char *operation, int child_index, KHE_COST *cost_field,
  char *fmt, va_list ap)
{
  char buff[13];  int count;  KHE_COST cost;  bool print;
  print = (strcmp(KheDrsMonitorInfoId(mi), RERUN_MONITOR_ID) == 0);
  if( print )
  {
    if( child_index >= 0 )
      snprintf(buff, 12, "%s %s %d", field, operation, child_index);
    else
      snprintf(buff, 12, "%s %s", field, operation);
    fprintf(stderr, "  %12s: %.5f", buff, KheCostShow(*cost_field));
  }
  for( count = 0;  fmt[count] != '\0';  count++ )
  {
    cost = va_arg(ap, KHE_COST);
    if( print )
      fprintf(stderr, " %c %.5f", fmt[count], KheCostShow(cost));
    if( fmt[count] == '+' )
      *cost_field += cost;
    else
      *cost_field -= cost;
  }
  if( print )
  {
    while( count < 4 )
    {
      fprintf(stderr, "          ");
      count++;
    }
    fprintf(stderr, "  (%s)\n", KheMonitorId(mi->monitor));
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsMonitorInfoUpdateRerunCost(KHE_DRS_MONITOR_INFO mi,           */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, KHE_DRS_SIGNER dsg,                   */
/*    KHE_DRS_OP op, char *operation, int child_index, char *fmt, ...)       */
/*                                                                           */
/*  Update mi.  Also produce a debug print about it, if mi's monitor is      */
/*  RERUN_MONITOR_ID.                                                        */
/*                                                                           */
/*****************************************************************************/

static void KheDrsMonitorInfoUpdateRerunCost(KHE_DRS_MONITOR_INFO mi,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, KHE_DRS_SIGNER dsg,
  KHE_DRS_OP op, char *operation, int child_index, char *fmt, ...)
{
  va_list ap;
  if( RERUN && drs->rerun != NULL && (dsg == NULL || dsg->debug_eval_if_rerun) )
  {
    if( op == KHE_DRS_OPEN || op == KHE_DRS_SEARCH )
    {
      va_start(ap, fmt);
      KheDrsMonitorInfoDoUpdateRerunCost(mi, "os", operation,
	child_index, &mi->rerun_open_and_search_cost, fmt, ap);
      va_end(ap);
    }
    if( op == KHE_DRS_OPEN || op == KHE_DRS_CLOSE )
    {
      va_start(ap, fmt);
      KheDrsMonitorInfoDoUpdateRerunCost(mi, "oc", operation,
	child_index, &mi->rerun_open_and_close_cost, fmt, ap);
      va_end(ap);
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsMonitorInfoInitRerunCost(KHE_DRS_MONITOR_INFO mi)             */
/*                                                                           */
/*  Initialize the rerun cost fields of mi to its monitor's cost.            */
/*                                                                           */
/*****************************************************************************/

static void KheDrsMonitorInfoInitRerunCost(KHE_DRS_MONITOR_INFO mi,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  mi->rerun_open_and_search_cost = mi->rerun_open_and_close_cost = 0;
  KheDrsMonitorInfoUpdateRerunCost(mi, drs, NULL, KHE_DRS_OPEN, "init", -1,
    "+", KheMonitorCost(mi->monitor));
  /* ***
  mi->rerun_open_and_search_cost = mi->rerun_open_and_close_cost =
    KheMonitorCost(mi->monitor);
  *** */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsMonitorInfoCheckRerunCost(KHE_DRS_MONITOR_INFO mi)            */
/*                                                                           */
/*  Check the rerun costs of mi, and do some printing if there is a problem. */
/*                                                                           */
/*****************************************************************************/

static void KheDrsMonitorInfoCheckRerunCost(KHE_DRS_MONITOR_INFO mi)
{
  if( mi->rerun_open_and_search_cost != KheMonitorCost(mi->monitor) ||
      mi->rerun_open_and_close_cost != KheMonitorCost(mi->monitor) )
  {
    fprintf(stderr, "  [ KheDrsRerun inconsistent monitor\n");
    fprintf(stderr, "    %s\n", KheMonitorId(mi->monitor));
    fprintf(stderr, "    monitor cost         %.5f\n",
      KheCostShow(KheMonitorCost(mi->monitor)));
    if( mi->rerun_open_and_search_cost != KheMonitorCost(mi->monitor) )
      fprintf(stderr, "    open_and_search_cost %.5f\n",
	KheCostShow(mi->rerun_open_and_search_cost));
    if( mi->rerun_open_and_close_cost != KheMonitorCost(mi->monitor) )
      fprintf(stderr, "    open_and_close_cost  %.5f\n",
	KheCostShow(mi->rerun_open_and_close_cost));
    fprintf(stderr, "  ]\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_OPEN_CHILDREN"                                        */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheDrsOpenChildrenInit(KHE_DRS_OPEN_CHILDREN oc,                    */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Initialize oc.                                                           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsOpenChildrenInit(KHE_DRS_OPEN_CHILDREN oc,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  HaArrayInit(oc->children, drs->arena);
  oc->range = KheDrsDayRangeMake(1, 0);
  HaArrayInit(oc->child_indexes, drs->arena);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsOpenChildrenClear(KHE_DRS_OPEN_CHILDREN oc)                   */
/*                                                                           */
/*  Clear out oc to the state it takes when it is closed.                    */
/*                                                                           */
/*****************************************************************************/

static void KheDrsOpenChildrenClear(KHE_DRS_OPEN_CHILDREN oc)
{
  HaArrayClear(oc->children);
  oc->range = KheDrsDayRangeMake(1, 0);
  HaArrayClear(oc->child_indexes);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsOpenChildrenCount(KHE_DRS_OPEN_CHILDREN oc)                    */
/*                                                                           */
/*  Return the number of open children in oc.                                */
/*                                                                           */
/*****************************************************************************/

static int KheDrsOpenChildrenCount(KHE_DRS_OPEN_CHILDREN oc)
{
  return HaArrayCount(oc->children);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsOpenChildrenAddChildInDayOrder(KHE_DRS_OPEN_CHILDREN oc,      */
/*    KHE_DRS_EXPR child_e)                                                  */
/*                                                                           */
/*  Add child_e to oc's list of open children, at a position that ensures    */
/*  that the children's day indexes are monotone non-decreasing.             */
/*                                                                           */
/*****************************************************************************/

static void KheDrsOpenChildrenAddChildInDayOrder(KHE_DRS_OPEN_CHILDREN oc,
  KHE_DRS_EXPR child_e)
{
  KHE_DRS_EXPR e;  int i;

  /* add child_e to oc->children in sorted position */
  HaArrayAddLast(oc->children, NULL);
  for( i = HaArrayCount(oc->children) - 2;  i >= 0;  i-- )
  {
    e = HaArray(oc->children, i);
    if( e->open_children_by_day.range.last <=
	child_e->open_children_by_day.range.last )
      break;
    HaArrayPut(oc->children, i + 1, e);
  }
  HaArrayPut(oc->children, i + 1, child_e);

  /* add child_e's last open day to oc->range */
  KheDrsDayRangeAdd(&oc->range, child_e->open_children_by_day.range.last);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsOpenChildrenAddChildWithDayUpdate(KHE_DRS_OPEN_CHILDREN oc,   */
/*    KHE_DRS_EXPR child_e)                                                  */
/*                                                                           */
/*  Add child_e to the end of oc's list of open children, increasing         */
/*  child_e's open day index if required.                                    */
/*                                                                           */
/*****************************************************************************/

static void KheDrsOpenChildrenAddChildWithDayUpdate(KHE_DRS_OPEN_CHILDREN oc,
  KHE_DRS_EXPR child_e)
{
  KHE_DRS_EXPR e;

  /* increase child_e's last open day if required */
  if( HaArrayCount(oc->children) > 0 )
  {
    e = HaArrayLast(oc->children);
    if( e->open_children_by_day.range.last >
	child_e->open_children_by_day.range.last )
      child_e->open_children_by_day.range.last =
	e->open_children_by_day.range.last;
  }

  /* add child_e to the end of oc->children */
  HaArrayAddLast(oc->children, child_e);

  /* add child_e's last open day to e's open_day_range */
  KheDrsDayRangeAdd(&oc->range, child_e->open_children_by_day.range.last);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprOpenShiftIndex(KHE_DRS_EXPR e)                             */
/*                                                                           */
/*  Return the open shift index of the shift containing the task monitored   */
/*  by e, which is known to be a KHE_DRS_EXPR_ASSIGNED_TASK expression.      */
/*                                                                           */
/*****************************************************************************/

static int KheDrsExprOpenShiftIndex(KHE_DRS_EXPR e)
{
  KHE_DRS_EXPR_ASSIGNED_TASK eat;  int res;
  eat = (KHE_DRS_EXPR_ASSIGNED_TASK) e;
  res = eat->task_on_day->encl_dt->encl_dtc->encl_shift->open_shift_index;
  HnAssert(res >= 0, "KheDrsExprOpenShiftIndex internal error 2");
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsOpenChildrenAddChildInShiftOrder(KHE_DRS_OPEN_CHILDREN oc,    */
/*    KHE_DRS_EXPR child_e)                                                  */
/*                                                                           */
/*  Add child_e to oc's list of open children, at a position that ensures    */
/*  that the children's open shift indexes are monotone non-decreasing.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsOpenChildrenAddChildInShiftOrder(KHE_DRS_OPEN_CHILDREN oc,
  KHE_DRS_EXPR child_e)
{
  KHE_DRS_EXPR e;  int i, child_e_index;

  /* make sure child_e is an assigned task child */
  HnAssert(child_e->tag == KHE_DRS_EXPR_ASSIGNED_TASK_TAG,
    "KheDrsOpenChildrenAddChildInShiftOrder internal error (tag %d)",
    child_e->tag);

  /* add child_e to oc->children in sorted position */
  child_e_index = KheDrsExprOpenShiftIndex(child_e);
  HaArrayAddLast(oc->children, NULL);
  for( i = HaArrayCount(oc->children) - 2;  i >= 0;  i-- )
  {
    e = HaArray(oc->children, i);
    if( KheDrsExprOpenShiftIndex(e) <= child_e_index )
      break;
    HaArrayPut(oc->children, i + 1, e);
  }
  HaArrayPut(oc->children, i + 1, child_e);

  /* add child_e's shift index to oc->range */
  KheDrsDayRangeAdd(&oc->range, child_e_index);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsOpenChildrenDeleteLast(KHE_DRS_OPEN_CHILDREN oc)              */
/*                                                                           */
/*  Delete the last child of oc.                                             */
/*                                                                           */
/*****************************************************************************/

static void KheDrsOpenChildrenDeleteLast(KHE_DRS_OPEN_CHILDREN oc)
{
  HaArrayDeleteLast(oc->children);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsOpenChildrenBuildDayChildIndexes(KHE_DRS_OPEN_CHILDREN oc)    */
/*                                                                           */
/*  Build the child indexes of oc based on the day ranges of the children.   */
/*                                                                           */
/*****************************************************************************/

static void KheDrsOpenChildrenBuildDayChildIndexes(KHE_DRS_OPEN_CHILDREN oc)
{
  int index, i;  KHE_DRS_EXPR child_e;
  HaArrayClear(oc->child_indexes);
  index = oc->range.first - 1;
  HaArrayForEach(oc->children, child_e, i)
  {
    while( child_e->open_children_by_day.range.last > index )
    {
      index++;
      HaArrayAddLast(oc->child_indexes, i);

      /******************************************************************/
      /*                                                                */
      /*  Here we have just added i at position index of child_indexes, */
      /*  and the child at index i is the first to have a larger last   */
      /*  index than index.                                             */
      /*                                                                */
      /******************************************************************/
    }
  }
  HaArrayAddLast(oc->child_indexes, i);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsOpenChildrenBuildShiftChildIndexes(KHE_DRS_OPEN_CHILDREN oc)  */
/*                                                                           */
/*  Build the child indexes of oc based on the children's shift indexes.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsOpenChildrenBuildShiftChildIndexes(KHE_DRS_OPEN_CHILDREN oc)
{
  int index, i;  KHE_DRS_EXPR child_e;
  HaArrayClear(oc->child_indexes);
  index = oc->range.first - 1;
  HaArrayForEach(oc->children, child_e, i)
  {
    while( KheDrsExprOpenShiftIndex(child_e) > index )
    {
      index++;
      HaArrayAddLast(oc->child_indexes, i);

      /******************************************************************/
      /*                                                                */
      /*  Here we have just added i at position index of child_indexes, */
      /*  and the child at index i is the first to have a larger last   */
      /*  index than index.                                             */
      /*                                                                */
      /******************************************************************/
    }
  }
  HaArrayAddLast(oc->child_indexes, i);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsOpenChildrenBefore(KHE_DRS_OPEN_CHILDREN oc, int index)        */
/*                                                                           */
/*  Return the number of open children of oc whose index is less than the    */
/*  given index.  Here index is expected to satisfy                          */
/*                                                                           */
/*    oc->range.first <= index <= oc->range.last + 1                         */
/*                                                                           */
/*  If index = oc->range.last + 1, the number of open children is returned.  */
/*                                                                           */
/*  Implementation note.  We've removed the limits on index to help with     */
/*  evaluating shift assignment objects.  This is really work in progress.   */
/*                                                                           */
/*****************************************************************************/

static int KheDrsOpenChildrenBefore(KHE_DRS_OPEN_CHILDREN oc, int index)
{
  if( index < oc->range.first )
    return 0;
  else if( index > oc->range.last )
    return HaArrayCount(oc->children);
  else
    return HaArray(oc->child_indexes, index - oc->range.first);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsOpenChildrenAtOrAfter(KHE_DRS_OPEN_CHILDREN oc, int index)     */
/*                                                                           */
/*  Return the number of open children of e whose index is index or more.    */
/*  Again, index is expected to satisfy                                      */
/*                                                                           */
/*    oc->range.first <= index <= oc->range.last + 1                         */
/*                                                                           */
/*  If index = oc->range.last + 1, 0 is returned.                            */
/*                                                                           */
/*****************************************************************************/

static int KheDrsOpenChildrenAtOrAfter(KHE_DRS_OPEN_CHILDREN oc, int index)
{
  return HaArrayCount(oc->children) - KheDrsOpenChildrenBefore(oc, index);
}


/*****************************************************************************/
/*                                                                           */
/*  KheDrsOpenChildrenForEachIndex(oc, index)                                */
/*                                                                           */
/*  Iterator for visiting each index of oc.                                  */
/*                                                                           */
/*****************************************************************************/

#define KheDrsOpenChildrenForEachIndex(oc, index)			\
  for( index = (oc)->range.first;  index <= (oc)->range.last;  index++ )


/*****************************************************************************/
/*                                                                           */
/*  KheDrsOpenChildrenForEach(oc, index, x, i)                               */
/*                                                                           */
/*  Iterator over all the open children x of oc with the given index.        */
/*                                                                           */
/*****************************************************************************/

#define KheDrsOpenChildrenForEach(oc, index, x, i)			\
  i1 = KheDrsOpenChildrenBefore((oc), (index));				\
  i2 = KheDrsOpenChildrenBefore((oc), (index) + 1);			\
  for( (i) = i1;							\
       (i) < i2 ? ((x) = HaArray((oc)->children, (i)), true) : false;	\
       (i)++ )


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsOpenChildrenIndexIsFirst(KHE_DRS_OPEN_CHILDREN oc, int index) */
/*                                                                           */
/*  Return true if index is the first open index of oc.                      */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsOpenChildrenIndexIsFirst(KHE_DRS_OPEN_CHILDREN oc, int index)
{
  return index == oc->range.first;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsOpenChildrenIndexIsFirstOrLess(KHE_DRS_OPEN_CHILDREN oc,      */
/*    int index)                                                             */
/*                                                                           */
/*  Return true if index is the first open index of oc, or less.             */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsOpenChildrenIndexIsFirstOrLess(KHE_DRS_OPEN_CHILDREN oc,
  int index)
{
  return index <= oc->range.first;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsOpenChildrenIndexIsLast(KHE_DRS_OPEN_CHILDREN oc, int index)  */
/*                                                                           */
/*  Return true if index is the last open index of oc.                       */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsOpenChildrenIndexIsLast(KHE_DRS_OPEN_CHILDREN oc, int index)
{
  return index == oc->range.last;
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsOpenChildrenWithIndex(KHE_DRS_OPEN_CHILDREN oc, int index)     */
/*                                                                           */
/*  Return the number of children in oc with this exact index.               */
/*                                                                           */
/*****************************************************************************/

static int KheDrsOpenChildrenWithIndex(KHE_DRS_OPEN_CHILDREN oc, int index)
{
  return KheDrsOpenChildrenBefore(oc, index + 1) - 
    KheDrsOpenChildrenBefore(oc, index);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsOpenChildrenDebug(KHE_DRS_OPEN_CHILDREN oc,                   */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of oc.                                                       */
/*                                                                           */
/*****************************************************************************/

static void KheDrsOpenChildrenDebugIndexes(KHE_DRS_OPEN_CHILDREN oc, FILE *fp)
{
  int val, i;
  fprintf(fp, "[%d open children, open range %d-%d: ",
    HaArrayCount(oc->children), oc->range.first, oc->range.last);
  HaArrayForEach(oc->child_indexes, val, i)
  {
    if( i > 0 )
      fprintf(fp, ",");
    fprintf(fp, "%d", val);
  }
  fprintf(fp, "]");
}

static void KheDrsOpenChildrenDebug(KHE_DRS_OPEN_CHILDREN oc,
  int verbosity, int indent, FILE *fp)
{
  int i;  KHE_DRS_EXPR child_e;
  if( indent >= 0 )
  {
    if( indent > 0 )
      fprintf(fp, "%*s", indent, "");
    fprintf(fp, "[ ");
    KheDrsOpenChildrenDebugIndexes(oc, fp);
    fprintf(fp, "\n");
    HaArrayForEach(oc->children, child_e, i)
      KheDrsExprDebug(child_e, NULL, verbosity, indent + 2, fp);
    if( indent > 0 )
      fprintf(fp, "%*s", indent, "");
    fprintf(fp, "]\n");
  }
  else
    KheDrsOpenChildrenDebugIndexes(oc, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR - construction"                                  */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsExprDebugWanted(KHE_DRS_EXPR e)                               */
/*                                                                           */
/*  Return true if e is wanted for debugging, according to DEBUG58, or       */
/*  is a child of a wanted expression.                                       */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsExprDebugWanted(KHE_DRS_EXPR e)
{
  KHE_DRS_PARENT prnt;  int i;  KHE_DRS_EXPR_INT_SUM_COST eisc;
  if( DEBUG10 )
    return true;
  if( e->tag == KHE_DRS_EXPR_INT_SUM_COST_TAG )
  {
    eisc = (KHE_DRS_EXPR_INT_SUM_COST) e;
    return DEBUG58(eisc->monitor_info);
  }
  else
  {
    HaArrayForEach(e->parents, prnt, i)
      if( KheDrsExprDebugWanted(prnt.expr) )
	return true;
  }
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprInitBegin(KHE_DRS_EXPR e, KHE_DRS_EXPR_TAG tag,           */
/*    KHE_DRS_DOM_TEST dom_test, KHE_DRS_RESOURCE dr,                        */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Initialize those fields of e common to all expressions.                  */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprInitBegin(KHE_DRS_EXPR e, KHE_DRS_EXPR_TAG tag,
  KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  e->tag = tag;
  e->gathered = false;
  e->postorder_index = -1;
  e->resource = dr;
  HaArrayInit(e->parents, drs->arena);
  HaArrayInit(e->children, drs->arena);
  KheDrsOpenChildrenInit(&e->open_children_by_day, drs);
  /* e->open_day_range = KheDrsDayRangeMake(1, 0); */
  /* HaArrayInit(e->open_children, drs->arena); */
  /* HaArrayInit(e->open_child_indexes, drs->arena); */
  HaArrayInit(e->sig_indexes, drs->arena);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsExprIsSingleDay(KHE_DRS_EXPR e, KHE_DRS_DAY *day)             */
/*                                                                           */
/*  Return true if e is derived from and event resource constraint on a      */
/*  single day.  Initially *day is NULL, but if an ASSIGNED_TASK is found,   */
/*  *day is set to its day.                                                  */
/*                                                                           */
/*****************************************************************************/

/* ***
static bool KheDrsExprIsSingleDay(KHE_DRS_EXPR e, KHE_DRS_DAY *day)
{
  KHE_DRS_EXPR child_e;  int i;  KHE_DRS_EXPR_ASSIGNED_TASK eat;
  KHE_DRS_EXPR_BUSY_TIME ebt;  KHE_DRS_EXPR_FREE_TIME eft;
  KHE_DRS_EXPR_WORK_TIME ewt;  KHE_DRS_EXPR_BUSY_DAY ebd;
  KHE_DRS_EXPR_FREE_DAY efd;  KHE_DRS_EXPR_WORK_DAY ewd;
  KHE_DRS_DAY e_day;

  ** check the children **
  HaArrayForEach(e->children, child_e, i)
    if( !KheDrsExprIsSingleDay(child_e, day) )
      return false;

  ** check e itself **
  switch( e->tag )
  {
    case KHE_DRS_EXPR_ASSIGNED_TASK_TAG:

      eat = (KHE_DRS_EXPR_ASSIGNED_TASK) e;
      e_day = eat->task_on_day->day;
      return (*day == NULL ? *day = e_day, true : *day == e_day);

    case KHE_DRS_EXPR_BUSY_TIME_TAG:

      ebt = (KHE_DRS_EXPR_BUSY_TIME) e;
      e_day = ebt->resource_on_day->day;
      return (*day == NULL ? *day = e_day, true : *day == e_day);

    case KHE_DRS_EXPR_FREE_TIME_TAG:

      eft = (KHE_DRS_EXPR_FREE_TIME) e;
      e_day = eft->resource_on_day->day;
      return (*day == NULL ? *day = e_day, true : *day == e_day);

    case KHE_DRS_EXPR_WORK_TIME_TAG:

      ewt = (KHE_DRS_EXPR_WORK_TIME) e;
      e_day = ewt->resource_on_day->day;
      return (*day == NULL ? *day = e_day, true : *day == e_day);

    case KHE_DRS_EXPR_BUSY_DAY_TAG:

      ebd = (KHE_DRS_EXPR_BUSY_DAY) e;
      e_day = ebd->resource_on_day->day;
      return (*day == NULL ? *day = e_day, true : *day == e_day);

    case KHE_DRS_EXPR_FREE_DAY_TAG:

      efd = (KHE_DRS_EXPR_FREE_DAY) e;
      e_day = efd->resource_on_day->day;
      return (*day == NULL ? *day = e_day, true : *day == e_day);

    case KHE_DRS_EXPR_WORK_DAY_TAG:

      ewd = (KHE_DRS_EXPR_WORK_DAY) e;
      e_day = ewd->resource_on_day->day;
      return (*day == NULL ? *day = e_day, true : *day == e_day);

    case KHE_DRS_EXPR_OR_TAG:
    case KHE_DRS_EXPR_AND_TAG:	
    case KHE_DRS_EXPR_INT_SUM_TAG:
    case KHE_DRS_EXPR_FLOAT_SUM_TAG:
    case KHE_DRS_EXPR_INT_DEV_TAG:
    case KHE_DRS_EXPR_FLOAT_DEV_TAG:
    case KHE_DRS_EXPR_INT_SUM_COST_TAG:
    case KHE_DRS_EXPR_INT_SEQ_COST_TAG:

      ** these don't prevent anything **
      return true;

    default:

      HnAbort("KheDrsExprIsSingleDay: internal error");
      return false;  ** keep compiler happy **
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KheDrsExprInitEnd(res, drs)                                              */
/*                                                                           */
/*  Finalize the initialization of e by setting its postorder index and      */
/*  closed value.                                                            */
/*                                                                           */
/*****************************************************************************/
static void KheDrsExprSetClosedValue(KHE_DRS_EXPR e,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsExprInitEnd(KHE_DRS_EXPR e, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR child_e;  int i;
  e->postorder_index = drs->postorder_count++;
  KheDrsExprSetClosedValue(e, drs);
  if( KheDrsExprDebugWanted(e) )
  {
    fprintf(stderr, "  KheDrsExprInitEnd(e), e =\n");
    KheDrsExprDebug(e, drs, 2, 2, stderr);
  }
  if( DEBUG59 )
  {
    HaArrayForEach(e->children, child_e, i)
      if( e->postorder_index <= child_e->postorder_index )
      {
	fprintf(stderr, "  KheDrsExprInitEnd aborting on e:\n");
	KheDrsExprDebug(e, drs, 2, 2, stderr);
	HnAbort("KheDrsExprInitEnd: e pi %d <= child_e pi %d",
	  e->postorder_index, child_e->postorder_index);
      }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprAddChild(KHE_DRS_EXPR parent, KHE_DRS_EXPR child,         */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make child a child of parent.                                            */
/*                                                                           */
/*  This function includes a switch on the expression type and then calls a  */
/*  version specific to that type.  Like all version-specific expression     */
/*  functions, its implementation is deferred to after the subtypes.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprAddChild(KHE_DRS_EXPR parent, KHE_DRS_EXPR child,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsParentIsFirstIntSumCostParent(KHE_DRS_EXPR child,             */
/*    KHE_DRS_EXPR parent)                                                   */
/*                                                                           */
/*  Return true if parent is the first int sum cost parent of child.         */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsParentIsFirstIntSumCostParent(KHE_DRS_EXPR child,
  KHE_DRS_EXPR parent)
{
  KHE_DRS_PARENT prnt;  int i;
  HaArrayForEach(child->parents, prnt, i)
    if( prnt.expr->tag == KHE_DRS_EXPR_INT_SUM_COST_TAG )
      return prnt.expr == parent;
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsExprHasParent(KHE_DRS_EXPR child, KHE_DRS_EXPR_TAG tag)       */
/*                                                                           */
/*  Return true if child has a parent of the type given by tag.              */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsExprHasParent(KHE_DRS_EXPR child, KHE_DRS_EXPR_TAG tag)
{
  KHE_DRS_PARENT prnt;  int i;
  HaArrayForEach(child->parents, prnt, i)
    if( prnt.expr->tag == tag )
      return true;
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR - open day indexes and shifts"                   */
/*                                                                           */
/*  This submodule has been replaced by submodule KHE_DRS_OPEN_CHILDREN.     */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprForEachOpenDayChild(KHE_DRS_EXPR e, int di,               */
/*    KHE_DRS_EXPR x, int i)                                                 */
/*                                                                           */
/*  Iterator which visits every open child of e whose last open day is di,   */
/*  setting x to each child in turn and i to its index in e->open_children.  */
/*                                                                           */
/*****************************************************************************/

/* *** replaced by KheDrsOpenChildrenForEach
#define KheDrsExprForEachOpenDayChild(e, di, x, i)			\
  i1 = KheDrsOpenChildrenBefore((KHE_DRS_EXPR) (e), (di));		\
  i2 = KheDrsOpenChildrenBefore((KHE_DRS_EXPR) (e), (di) + 1);	\
  for( (i) = i1;							\
       (i) < i2 ? ((x) = HaArray((e)->open_children_by_day.children,	\
	(i)), true) : false;						\
       (i)++ )
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprForEachOpenDayShiftChild(KHE_DRS_EXPR e, int di, int si,  */
/*    KHE_DRS_EXPR x, int i)                                                 */
/*                                                                           */
/*  Iterator which visits every open child of e whose last open day is di    */
/*  and whose shift index is si, setting x to each such child in turn and    */
/*  i to its index in e->open_children.                                      */
/*                                                                           */
/*  Implementation note.  This is not very fast, but that does not matter.   */
/*                                                                           */
/*****************************************************************************/

/* *** no longer used
#define KheDrsExprForEachOpenDayShiftChild(e, di, si, x, i)		  \
  KheDrsExprForEachOpenDayChild(e, di, x, i)				  \
    if( KheDrsExprShiftIndex(x) == (si) )
*** */


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprShift(KHE_DRS_EXPR e)                                      */
/*                                                                           */
/*  Return the shift that determines the value of e.  Abort if there is no   */
/*  no such shift.                                                           */
/*                                                                           */
/*****************************************************************************/

/* *** unused
static KHE_DRS_SHIFT KheDrsExprAssignedTaskShift(
  KHE_DRS_EXPR_ASSIGNED_TASK eat);

static KHE_DRS_SHIFT KheDrsExprShift(KHE_DRS_EXPR e)
{
  HnAssert(e->tag == KHE_DRS_EXPR_ASSIGNED_TASK_TAG,
    "KheDrsExprShift internal error (e->tag == %d)", e->tag);
  return KheDrsExprAssignedTaskShift((KHE_DRS_EXPR_ASSIGNED_TASK) e);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprShiftIndex(KHE_DRS_EXPR e)                                 */
/*                                                                           */
/*  Return the index of the shift that determines the value of e.  Abort     */
/*  if there is no such shift.                                               */
/*                                                                           */
/*****************************************************************************/

/* *** no longer used
static int KheDrsExprShiftIndex(KHE_DRS_EXPR e)
{
  KHE_DRS_SHIFT shift;
  shift = KheDrsExprShift(e);
  return shift->index_in_day;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsExprOpenDayIsFirst(KHE_DRS_EXPR e, int open_day_index)        */
/*                                                                           */
/*  Return true if open_day_index is the index of the first open day of e.   */
/*                                                                           */
/*****************************************************************************/

/* *** replaced by KheDrsOpenChildrenIndexIsFirst
static bool KheDrsExprOpenDayIsFirst(KHE_DRS_EXPR e, int open_day_index)
{
  return open_day_index == e->open_children_by_day.range.first;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsExprOpenDayIsLast(KHE_DRS_EXPR e, int open_day_index)         */
/*                                                                           */
/*  Return true if open_day_index is the index of the last open day of e.    */
/*                                                                           */
/*****************************************************************************/

/* *** replaced by KheDrsOpenChildrenIndexIsLast
static bool KheDrsExprOpenDayIsLast(KHE_DRS_EXPR e, int open_day_index)
{
  return open_day_index == e->open_children_by_day.range.last;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsExprOpenDayIsLastWithSingleShift(KHE_DRS_EXPR e,              */
/*    int open_day_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)                   */
/*                                                                           */
/*  Return true if open_day_index is e's last open day, and also that day    */
/*  has a single shift.  This means that we'll be finalizing e if we         */
/*  assign resources to the tasks of that shift.                             */
/*                                                                           */
/*****************************************************************************/

/* *** no longer used (in fact logically wrong)
static bool KheDrsExprOpenDayIsLastWithSingleShift(KHE_DRS_EXPR e,
  int open_day_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DAY day;
  if( !KheDrsOpenChildrenIndexIsLast(&e->open_children_by_day, open_day_index) )
    return false;
  day = HaArray(drs->open_days, open_day_index);
  return HaArrayCount(day->shifts) == 1;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprOpenChildrenBefore(KHE_DRS_EXPR e, int di)                 */
/*                                                                           */
/*  Return the number of open children of e whose last open day precedes     */
/*  the day with open day index di.  Or di could be one plus the index of    */
/*  the last open day, in which case return the number of open children.     */
/*                                                                           */
/*****************************************************************************/

/* *** replaced by KheDrsOpenChildrenBefore
static int KheDrsExprOpenChildrenBefore(KHE_DRS_EXPR e, int di)
{
  return HaArray(e->open_children_by_day.child_indexes,
    di - e->open_children_by_day.range.first);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprOpenChildrenAtOrAfter(KHE_DRS_EXPR e, int di)              */
/*                                                                           */
/*  Return the number of open children of e whose last open day is at or     */
/*  after the day with open day index di.  Or di could be one plus the       */
/*  index of the last open day, in which case return 0.                      */
/*                                                                           */
/*****************************************************************************/

/* *** replaced by KheDrsOpenChildrenAtOrAfter
static int KheDrsExprOpenChildrenAtOrAfter(KHE_DRS_EXPR e, int di)
{
  return HaArrayCount(e->open_children_by_day.children) -
    KheDrsExprOpenChildrenBefore(e, di);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprOpenChildrenForShiftOnDay(KHE_DRS_EXPR e, int di, int si)  */
/*                                                                           */
/*  Return the number of children of e whose last open day is di and         */
/*  which are affected by shift si.                                          */
/*                                                                           */
/*****************************************************************************/

/* *** no longer used
static int KheDrsExprOpenChildrenForShiftOnDay(KHE_DRS_EXPR e, int di, int si)
{
  int res, i, i1, i2;  KHE_DRS_EXPR child_e;
  res = 0;
  KheDrsExprForEachOpenDayShiftChild(e, di, si, child_e, i)
    res++;
  return res;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR - opening"                                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprGatherForOpening(KHE_DRS_EXPR e,                          */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Gather e into drs->open_exprs, the list of expressions that are to be    */
/*  opened.  This does not actually open e; that comes later.                */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprGatherForOpening(KHE_DRS_EXPR e,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_PARENT prnt;  int i;
  if( !e->gathered )
  {
    e->gathered = true;
    HaArrayAddLast(drs->open_exprs, e);
    HaArrayForEach(e->parents, prnt, i)
      KheDrsExprGatherForOpening(prnt.expr, drs);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprPostorderCmp(const void *t1, const void *t2)               */
/*                                                                           */
/*  Comparison function for sorting an array of expressions by increasing    */
/*  postorder index.                                                         */
/*                                                                           */
/*****************************************************************************/

static int KheDrsExprPostorderCmp(const void *t1, const void *t2)
{
  KHE_DRS_EXPR e1 = * (KHE_DRS_EXPR *) t1;
  KHE_DRS_EXPR e2 = * (KHE_DRS_EXPR *) t2;
  return e1->postorder_index - e2->postorder_index;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprChildHasOpened(KHE_DRS_EXPR e, KHE_DRS_EXPR child_e,      */
/*    int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)                      */
/*                                                                           */
/*  As part of opening e for solving, inform e that its child_index'th       */
/*  child, child_e, has opened.                                              */
/*                                                                           */
/*  This function switches on the expression type and then calls a           */
/*  version specific to that type.  Like all version-specific expression     */
/*  functions, its implementation is deferred to after the subtypes.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprChildHasOpened(KHE_DRS_EXPR e, KHE_DRS_EXPR child_e,
  int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs);


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprChildDomTest(KHE_DRS_EXPR e,                  */
/*    KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  Return a dom test suited to a child of e.                                */
/*                                                                           */
/*  This function switches on the expression type and then calls a           */
/*  version specific to that type.  Like all version-specific expression     */
/*  functions, its implementation is deferred to after the subtypes.         */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DOM_TEST KheDrsExprChildDomTest(KHE_DRS_EXPR e,
  int open_day_index, KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs);


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprDayDomTest(KHE_DRS_EXPR e,                    */
/*    int open_child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  Return a dom test suited to e on its open_child_index'th open day.       */
/*                                                                           */
/*  This function switches on the expression type and then calls a           */
/*  version specific to that type.  Like all version-specific expression     */
/*  functions, its implementation is deferred to after the subtypes.         */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DOM_TEST KheDrsExprDayDomTest(KHE_DRS_EXPR e,
  int open_child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs);


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprInitOpenChildIndexes(KHE_DRS_EXPR e)                      */
/*                                                                           */
/*  Initialize the open child index array of e.                              */
/*                                                                           */
/*****************************************************************************/

/* *** inlined into KheDrsExprOpen now
static void KheDrsExprInitOpenChildIndexes(KHE_DRS_EXPR e)
{
  int i, di;  KHE_DRS_EXPR child_e;
  HaArrayClear(e->open_child_indexes);
  di = e->open_day_range.first - 1;
  HaArrayForEach(e->open_children, child_e, i)
  {
    while( child_e->open_day_range.last > di )
    {
      di++;
      HaArrayAddLast(e->open_child_indexes, i);

      *******************************************************************
      **                                                               **
      **  Here we have just added i to index di of open_child_indexes, **
      **  and the child at index i is the first to have a larger last  **
      **  open day index than di.                                      **
      **                                                               **
      *******************************************************************
    }
  }
  HaArrayAddLast(e->open_child_indexes, i);  ** number of open children **
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprAddToShift(KHE_DRS_EXPR e, int di, KHE_DRS_SHIFT curr_ds, */
/*    int curr_count, bool only_shift, KHE_DYNAMIC_RESOURCE_SOLVER drs)      */
/*                                                                           */
/*  Add open expression e to shift curr_ds, optionally with a dom test.      */
/*                                                                           */
/*****************************************************************************/

/* *** trying to get rid of this
static KHE_DRS_DOM_TEST KheDrsExprShiftDomTest(KHE_DRS_EXPR e,
  int open_day_index, int curr_count, KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsExprAddToShift(KHE_DRS_EXPR e, int di, KHE_DRS_SHIFT curr_ds,
  int curr_count, bool only_shift, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DOM_TEST dom_test;

  ** bug fixer **
  HnAssert(KheDrsDayRangeContains(e->open_children_by_day.range, di),
    "KheDrsExprAddToShift internal error 1:  day range %d - %d, day %d\n",
    e->open_children_by_day.range.first, e->open_children_by_day.range.last,di);
  ** HnAssert(e->open_day_range.first <= di && di <= e->open_day_range.last, **
  if( DEBUG48 && di != curr_ds->encl_day->open_day_index )
  {
    fprintf(stderr, "  KheDrsExprAddToShift failing; expr:\n");
    KheDrsExprDebug(e, drs, 2, 4, stderr);
    fprintf(stderr, "  shift:\n");
    KheDrsShiftDebug(curr_ds, drs, 2, 4, stderr);
    HnAbort("KheDrsExprAddToShift internal error 2:  di = %d, shift day = %d\n",
      di, curr_ds->encl_day->open_day_index);
  }
  ** add e to curr_ds **
  KheDrsShiftAddOpenExpr(curr_ds, e);

  ** add a dom test if required **
  if( di < e->open_children_by_day.range.last || !only_shift )
  {
    dom_test = KheDrsExprShiftDomTest(e, di, curr_count, drs);
    KheDrsShiftAddDomTest(curr_ds, dom_test);
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprOpen(KHE_DRS_EXPR e, KHE_DYNAMIC_RESOURCE_SOLVER drs)     */
/*                                                                           */
/*  Open e for solving.                                                      */
/*                                                                           */
/*  By the time this is called, all of e's open children will have           */
/*  called KheDrsExprChildHasOpened.                                         */
/*                                                                           */
/*  Implementation note - how e->open_day_range is set                       */
/*  --------------------------------------------------                       */
/*  In external expressions, e->open_day_range receives its final value      */
/*  when KheDrsExprGatherForOpening is called.  In internal expressions,     */
/*  its value is initially empty and is updated by each call to              */
/*  KheDrsExprChildHasOpened until it reaches its final value.               */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprOpen(KHE_DRS_EXPR e, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_PARENT prnt;  int i;  bool debug;

  debug = KheDrsExprDebugWanted(e);
  if( debug )
  {
    fprintf(stderr, "[ KheDrsExprOpen\n");
    KheDrsExprDebug(e, drs, 2, 2, stderr);
  }

  /* inform e's parents that e is now open */
  e->gathered = false;
  HaArrayForEach(e->parents, prnt, i)
    KheDrsExprChildHasOpened(prnt.expr, e, prnt.index, drs);

  /* what to do now depends on whether e is external or internal, and if */
  /* internal whether from an event resource or resource constraint */
  if( e->tag <= KHE_DRS_EXPR_WORK_DAY_TAG )
  {
    /* external expression; clear its value */
    KheDrsExprLeafClear(e, drs);
  }
  else
  {
    /* internal expression, build day child indexes */
    KheDrsOpenChildrenBuildDayChildIndexes(&e->open_children_by_day);
  }

  /* *** moved to KheDrsExprBuildDomTests
  else if( e->resource != NULL )
  {
    ** internal expression from a resource constraint **
    ** add e to its day and resource on day signers **
    KheDrsOpenChildrenBuildDayChildIndexes(&e->open_children_by_day);
    HaArrayClear(e->sig_indexes);
    KheDrsOpenChildrenForEachIndex(&e->open_children_by_day, di)
    {
      day = HaArray(drs->open_days, di);
      drd = KheDrsResourceOnDay(e->resource, day);

      ** add e to its drd signer, but not to its day signer **
      KheDrsResourceOnDayAddOpenExpr(drd, e);

      ** add a dom test for e to its day and drd signers, if not e's last day **
      if( !KheDrsOpenChildrenIndexIsLast(&e->open_children_by_day, di) )
      {
	dom_test = KheDrsExprDayDomTest(e, di, drs);
	HaArrayAddLast(e->sig_indexes, KheDrsDayAddDomTest(day, dom_test, drs));
	KheDrsResourceOnDayAddDomTest(drd, dom_test, drs);
      }
    }
  }
  else
  {
    ** internal expression from an event resource constraint **
    ** add e to its day signers **
    KheDrsOpenChildrenBuildDayChildIndexes(&e->open_children_by_day);
    HaArrayClear(e->sig_indexes);
    KheDrsOpenChildrenForEachIndex(&e->open_children_by_day, di)
    {
      day = HaArray(drs->open_days, di);
      KheDrsDayAddOpenExpr(day, e);
      if( !KheDrsOpenChildrenIndexIsLast(&e->open_children_by_day, di) )
      {
	dom_test = KheDrsExprDayDomTest(e, di, drs);
	HaArrayAddLast(e->sig_indexes, KheDrsDayAddDomTest(day, dom_test, drs));
      }
    }

    ** add e to its shift signers **
    HnAssert(e->tag == KHE_DRS_EXPR_INT_SUM_COST_TAG,
      "KheDrsExprOpen: internal error");
    eisc = (KHE_DRS_EXPR_INT_SUM_COST) e;
    KheDrsOpenChildrenBuildShiftChildIndexes(&eisc->open_children_by_shift);
    KheDrsOpenChildrenForEachIndex(&eisc->open_children_by_shift, si)
      if( KheDrsOpenChildrenWithIndex(&eisc->open_children_by_shift, si) > 0 )
      {
	ds = HaArray(drs->open_shifts, si);
	KheDrsShiftAddOpenExpr(ds, e);
        if( KheDrsIntSumCostExprRequiresShiftDomTest(eisc, si, drs, &val) )
	{
	  dom_test = KheDrsExprIntSumCostGeneralDomTest(eisc, val, drs);
	  KheDrsShiftAddDomTest(ds, dom_test, drs);
	}
      }
  }
  *** */
  if( debug )
  {
    fprintf(stderr, "  KheDrsExprOpen returning; e is now:\n");
    KheDrsExprDebug(e, drs, 2, 2, stderr);
    fprintf(stderr, "] KheDrsExprOpen\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprBuildDomTests(KHE_DRS_EXPR e,                             */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Build e's dom tests.                                                     */
/*                                                                           */
/*  By the time this is called, every expression that is going to open       */
/*  will be open, including knowing what all of its open children are.       */
/*                                                                           */
/*****************************************************************************/
static bool KheDrsIntSumCostExprRequiresShiftDomTest(
  KHE_DRS_EXPR_INT_SUM_COST eisc, int si,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int *unassigned_child_count);
static KHE_DRS_DOM_TEST KheDrsExprIntSumCostGeneralDomTest(
  KHE_DRS_EXPR_INT_SUM_COST eisc, int unassigned_child_count,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsExprBuildDomTests(KHE_DRS_EXPR e,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_DAY day;  KHE_DRS_RESOURCE_ON_DAY drd;  int di, si, val;
  KHE_DRS_DOM_TEST dom_test;  bool debug; KHE_DRS_SHIFT ds;
  KHE_DRS_EXPR_INT_SUM_COST eisc;

  debug = KheDrsExprDebugWanted(e);
  if( debug )
  {
    fprintf(stderr, "[ KheDrsExprBuildDomTests\n");
    KheDrsExprDebug(e, drs, 2, 2, stderr);
  }

  if( e->tag <= KHE_DRS_EXPR_WORK_DAY_TAG )
  {
    /* external expression; nothing to do */
  }
  else if( e->resource != NULL )
  {
    /* internal expression from a resource constraint */
    /* add e to its resource on day signers */
    HaArrayClear(e->sig_indexes);
    KheDrsOpenChildrenForEachIndex(&e->open_children_by_day, di)
    {
      day = HaArray(drs->open_days, di);
      drd = KheDrsResourceOnDay(e->resource, day);
      KheDrsResourceOnDayAddOpenExpr(drd, e);
      if( !KheDrsOpenChildrenIndexIsLast(&e->open_children_by_day, di) )
      {
	dom_test = KheDrsExprDayDomTest(e, di, drs);
	HaArrayAddLast(e->sig_indexes,
	  KheDrsResourceOnDayAddDomTest(drd, dom_test, drs));
        /* ***
	HaArrayAddLast(e->sig_indexes, KheDrsDayAddDomTest(day, dom_test, drs));
	KheDrsResourceOnDayAddDomTest(drd, dom_test, drs);
	*** */
      }
    }
  }
  else
  {
    /* internal expression from an event resource constraint */
    /* add e to its day signers */
    HaArrayClear(e->sig_indexes);
    KheDrsOpenChildrenForEachIndex(&e->open_children_by_day, di)
    {
      day = HaArray(drs->open_days, di);
      KheDrsDayAddOpenExpr(day, e);
      if( !KheDrsOpenChildrenIndexIsLast(&e->open_children_by_day, di) )
      {
	dom_test = KheDrsExprDayDomTest(e, di, drs);
	HaArrayAddLast(e->sig_indexes,
	  KheDrsDayAddDomTest(day, dom_test, drs));
      }
    }

    /* add e to its shift signers */
    HnAssert(e->tag == KHE_DRS_EXPR_INT_SUM_COST_TAG,
      "KheDrsExprBuildDomTests: internal error");
    eisc = (KHE_DRS_EXPR_INT_SUM_COST) e;
    KheDrsOpenChildrenBuildShiftChildIndexes(&eisc->open_children_by_shift);
    KheDrsOpenChildrenForEachIndex(&eisc->open_children_by_shift, si)
      if( KheDrsOpenChildrenWithIndex(&eisc->open_children_by_shift, si) > 0 )
      {
	ds = HaArray(drs->open_shifts, si);
	KheDrsShiftAddOpenExpr(ds, e);
        if( KheDrsIntSumCostExprRequiresShiftDomTest(eisc, si, drs, &val) )
	{
	  dom_test = KheDrsExprIntSumCostGeneralDomTest(eisc, val, drs);
	  KheDrsShiftAddDomTest(ds, dom_test, drs);
	}
      }
  }
  if( debug )
  {
    fprintf(stderr, "  KheDrsExprBuildDomTests returning; e is now:\n");
    KheDrsExprDebug(e, drs, 2, 2, stderr);
    fprintf(stderr, "] KheDrsExprBuildDomTests\n");
  }
}


/* *** old version
static void KheDrsExprOpen(KHE_DRS_EXPR e, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_PARENT prnt;  KHE_DRS_DAY day;  int i, di, i1, i2, curr_count;
  KHE_DRS_EXPR child_e;  KHE_DRS_RESOURCE_ON_DAY drd;
  KHE_DRS_DOM_TEST dom_test;  KHE_DRS_SHIFT ds, curr_ds;  bool only_shift;

  ** inform e's parents that e is open **
  if( DEBUG10 )
  {
    fprintf(stderr, "[ KheDrsExprOpen\n");
    KheDrsExprDebug(e, drs, 2, 2, stderr);
  }
  e->gathered = false;
  HaArrayForEach(e->parents, prnt, i)
    KheDrsExprChildHasOpened(prnt.expr, e, prnt.index, drs);

  ** what to do now depends on whether e is external or internal **
  if( e->tag <= KHE_DRS_EXPR_WORK_DAY_TAG )
  {
    ** external expression; clear its value **
    KheDrsExprLeafClear(e, drs);
  }
  else
  {
    ** internal expression; build open_child_indexes into e's open children **
    KheDrsOpenChildrenBuildDayChildIndexes(&e->open_children_by_day);

    ** inform e's days and shifts that e is open **
    HaArrayClear(e->sig_indexes);
    KheDrsOpenChildrenForEachIndex(&e->open_children_by_day, di)
    {
      ** add e to either its day or its resource on day **
      HnAssert(0 <= di && di < HaArrayCount(drs->open_days),
	"KheDrsExprOpen: open day index (%d) out of range (0 .. %d)",
	di, HaArrayCount(drs->open_days) - 1);
      day = HaArray(drs->open_days, di);
      if( e->resource == NULL )
      {
	** internal expression with no resource can only be int sum cost **
	HnAssert(e->tag == KHE_DRS_EXPR_INT_SUM_COST_TAG,
          "KheDrsExprOpen: internal error");

	** add e to day di, optionally with a corresponding dom test **
	KheDrsDayAddOpenExpr(day, e);
	if( !KheDrsOpenChildrenIndexIsLast(&e->open_children_by_day, di) )
	{
	  dom_test = KheDrsExprDayDomTest(e, di, drs);
	  HaArrayAddLast(e->sig_indexes, KheDrsDayAddDomTest(day, dom_test));
	}

	** add e to its day di shifts (NB same shift children are together) **
	curr_ds = NULL;
	curr_count = 0;
	only_shift = true;
	KheDrsOpenChildrenForEach(&e->open_children_by_day, di, child_e, i)
	{
	  ds = KheDrsExprShift(child_e);

	  if( ds != curr_ds )
	  {
	    ** finish off curr_ds, if non-NULL **
	    if( curr_ds != NULL )
	    {
	      only_shift = false;
	      KheDrsExprAddToShift(e, di, curr_ds, curr_count, only_shift, drs);
	    }

	    ** start off with ds **
	    curr_ds = ds;
	    curr_count = 0;
	  }

	  ** carry on with the current shift **
	  curr_count++;
	}

	** finish off curr_ds, if non-NULL **
	if( curr_ds != NULL )
	  KheDrsExprAddToShift(e, di, curr_ds, curr_count, only_shift, drs);

      }
      else
      {
	** add e to its drd, but not to its day **
	** *** No! we don't add e to day, but we do add its dom test
	KheDrsDayAddOpenExpr(day, e);
	*** **
	drd = KheDrsResourceOnDay(e->resource, day);
	KheDrsResourceOnDayAddOpenExpr(drd, e);

	** add a dom test (the same dom test) to both if required **
	** if( di < e->open_children_by_day.range.last ) **
	if( !KheDrsOpenChildrenIndexIsLast(&e->open_children_by_day, di) )
	{
	  dom_test = KheDrsExprDayDomTest(e, di, drs);
	  HaArrayAddLast(e->sig_indexes, KheDrsDayAddDomTest(day, dom_test));
	  KheDrsResourceOnDayAddDomTest(drd, dom_test);
	}
      }
    }

    ** ***
    if( DEBUG25 && e->tag == KHE_DRS_EXPR_INT_SUM_COST_TAG )
    {
      KHE_DRS_EXPR_INT_SUM_COST eisc = (KHE_DRS_EXPR_INT_SUM_COST) e;
      fprintf(stderr, "[ dom_test for %s on day %s:\n",
	KheDrsMonitorInfoId(eisc->monitor_info), KheDrsDayId(day));
      KheDrsDomTestDebug(dom_test, 3, 2, stderr);
      fprintf(stderr, "]\n");
    }
    *** **

    ** ***
    ** if e is a resource expression, inform its drd's that it is open **
    if( e->resource != NULL )
    {
      ** HaArrayClear(e->signature_indexes); **
      for( i = e->open_day_range.first;  i <= e->open_day_range.last;  i++ )
      {
	** add e to drd's list of open internal expressions **
	HnAssert( 0 <= i && i < HaArrayCount(drs->open_days),
	  "KheDrsExprOpen: open day index (%d) out of range (0 .. %d)",
	  i, HaArrayCount(drs->open_days) - 1);
	day = HaArray(drs->open_days, i);
	drd = KheDrsResourceOnDay(e->resource, day);
	KheDrsResourceOnDayAddOpenExpr(drd, e);

	** signatures and dom tests (needed on each day except the last) **
	** sorting this out sti ll to do; we don't need a dom test, but  **
	** we do need to add to e->signature_indexes, surely?            **
	** ***
	if( i < e->open_day_range.last )
	  HaArrayAddLast(e->signature_indexes,
	    KheDrsResourceOnDayAddDomTest(drd, e->dom_test));
	*** **
      }
    }
    *** **
  }
  if( DEBUG10 )
  {
    KheDrsExprDebug(e, drs, 2, 2, stderr);
    fprintf(stderr, "] KheDrsExprOpen\n");
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR - accessing signatures"                          */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprDaySigVal(KHE_DRS_EXPR e, int open_day_index,              */
/*    KHE_DRS_SOLN soln)                                                     */
/*                                                                           */
/*  Retrieve from soln the signature of e on open day open_day_index.        */
/*                                                                           */
/*****************************************************************************/

static int KheDrsExprDaySigVal(KHE_DRS_EXPR e, int open_day_index,
  KHE_DRS_SIGNATURE sig)
{
  int pos;
  pos = HaArray(e->sig_indexes, open_day_index -
    e->open_children_by_day.range.first);
  return HaArray(sig->states, pos);
}

/* *** old version before signature sets
static int KheDrsExprDaySigVal(KHE_DRS_EXPR e, int open_day_index,
  KHE_DRS_SOLN soln)
{
  int pos;
  pos = HaArray(e->sig_indexes, open_day_index -
    e->open_children_by_day.range.first);
  return HaArray(soln->sig.states, pos);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprPutSigVal(KHE_DRS_EXPR e, int open_day_index,             */
/*    KHE_DRS_SOLN soln, int sig_val)                                        */
/*                                                                           */
/*  Add sig_val, the signature of e on open_day_index, to soln.              */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsExprPutSigVal(KHE_DRS_EXPR e, int open_day_index,
  KHE_DRS_SOLN soln, int sig_val)
{
  int pos;
  pos = HaArray(e->sig_indexes, open_day_index - e->open_day_range.first);
  HnAssert(HaArrayCount(soln->sig) == pos,
    "KheDrsExprPutSigVal internal error"),
  HaArrayAddLast(soln->sig, sig_val);
}
*** */

/* ***
#define KheDrsExprPutSigVal(e, open_day_index, soln, sig_val)		     \
  HnAssert(HaArrayCount((soln)->sig) ==				     \
    HaArray((e)->sig_indexes, (open_day_index) - (e)->open_day_range.first), \
    "KheDrsExprPutSigVal internal error"),				     \
  HaArrayAddLast((soln)->sig, (sig_val));
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprPutSigVal(KHE_DRS_EXPR e, int open_day_index,             */
/*    KHE_DRS_SIGNATURE sig, int sig_val)                                    */
/*                                                                           */
/*  Add sig_val, the signature of e on open_day_index, to sig.               */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsExprPutSigVal(KHE_DRS_EXPR e, int open_day_index,
  KHE_DRS_SIGNATURE sig, int sig_val)
{
  ** ***
  int pos;
  pos = HaArray(e->signature_indexes, open_day_index - e->open_day_range.first);
  HnAssert(HaArrayCount(sig->sig) == pos,
    "KheDrsExprPutSigVal internal error"),
  *** **
  HaArrayAddLast(sig->states, sig_val);
}
*** */

/* ***
#define KheDrsExprPutSigVal(e, open_day_index, soln, sig_val)		     \
  HnAssert(HaArrayCount((soln)->sig) ==				     \
    HaArray((e)->sig_indexes, (open_day_index) - (e)->open_day_range.first), \
    "KheDrsExprPutSigVal internal error"),				     \
  HaArrayAddLast((soln)->sig, (sig_val));
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsExprInSignature(KHE_DRS_EXPR e, KHE_DRS_DAY day, int *si)     */
/*                                                                           */
/*  This function is used only during debugging.  If e is in the signature   */
/*  on day, return true and set *si to its position in the signature.        */
/*  Otherwise return false.                                                  */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused; will need to return a signer as well if reinstated
static bool KheDrsExprInSignature(KHE_DRS_EXPR e, KHE_DRS_DAY day, int *si)
{
  int i = day->open_day_index;
  if( e->open_children_by_day.range.first <= i &&
      i < e->open_children_by_day.range.last )
    return *si = HaArray(e->sig_indexes,
      i - e->open_children_by_day.range.first), true;
  else
    return *si = -1, false;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR - closing"                                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprChildHasClosed(KHE_DRS_EXPR e,                            */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  Inform internal expression e that its child_index'th child, child_e,     */
/*  has closed.  The child will have its correct closed_state and value      */
/*  fields when this is called.                                              */
/*                                                                           */
/*  Although KheDrsExprChildHasOpened adds child_e to e->open_children,      */
/*  KheDrsExprChildHasClosed does not remove child_e from e->open_children.  */
/*  This is because children are closed in the same order they were opened,  */
/*  but there is no efficient way to remove them in this order, short of     */
/*  something too complicated such as a circular list.                       */
/*                                                                           */
/*  This function switches on the expression type and then calls a           */
/*  version specific to that type.  Like all version-specific expression     */
/*  functions, its implementation is deferred to after the subtypes.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprChildHasClosed(KHE_DRS_EXPR e,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs);


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprClose(KHE_DRS_EXPR e, KHE_DYNAMIC_RESOURCE_SOLVER drs)    */
/*                                                                           */
/*  Close e, which may be internal or external.  Whatever it needs to work   */
/*  out its closed value (an assignment, closed children, or closed state)   */
/*  is up to date.  At the end, inform e's parents that e has closed.        */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprClose(KHE_DRS_EXPR e, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_PARENT prnt;  int i;

  /* set e's closed value */
  KheDrsExprSetClosedValue(e, drs);

  /* clear fields that are used only when e is open */
  KheDrsOpenChildrenClear(&e->open_children_by_day);
  /* ***
  e->open_children_by_day.range = KheDrsDayRangeMake(1, 0);    ** empty **
  HaArrayClear(e->open_children_by_day.children);
  HaArrayClear(e->open_children_by_day.child_indexes);
  *** */
  HaArrayClear(e->sig_indexes);

  /* inform e's parents that e has closed */
  HaArrayForEach(e->parents, prnt, i)
    KheDrsExprChildHasClosed(prnt.expr, e, prnt.index, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprSetClosedValue(KHE_DRS_EXPR e,                            */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing e, set its value suitably for its closed state.       */
/*                                                                           */
/*  This function switches on the expression type and then calls a           */
/*  version specific to that type.  Like all version-specific expression     */
/*  functions, its implementation is deferred to after the subtypes.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprSetClosedValue(KHE_DRS_EXPR e,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR - searching"                                     */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprLeafSet(KHE_DRS_EXPR e, KHE_DRS_TASK_ON_DAY dtd,          */
/*    KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs)                  */
/*                                                                           */
/*  Inform leaf expression e that dtd has been assigned drd.                 */
/*                                                                           */
/*  This function switches on the expression type and then calls a           */
/*  version specific to that type.  Like all version-specific expression     */
/*  functions, its implementation is deferred to after the subtypes.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprLeafSet(KHE_DRS_EXPR e, KHE_DRS_TASK_ON_DAY dtd,
  KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs);


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprLeafClear(KHE_DRS_EXPR e)                                 */
/*                                                                           */
/*  Inform leaf expression e that the assignment of drd to dtd carried out   */
/*  by KheDrsExprLeafSet above has been removed.                             */
/*                                                                           */
/*  This function switches on the expression type and then calls a           */
/*  version specific to that type.  Like all version-specific expression     */
/*  functions, its implementation is deferred to after the subtypes.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprLeafClear(KHE_DRS_EXPR e,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprEvalDay(KHE_DRS_EXPR e, KHE_DRS_SOLN prev_soln,           */
/*    KHE_DRS_SOLN next_soln, KHE_DYNAMIC_RESOURCE_SOLVER drs)               */
/*                                                                           */
/*  Evaluate expression e in next_soln, whose day is one of e's open days.   */
/*                                                                           */
/*  This function switches on the expression type and then calls a           */
/*  version specific to that type.  Like all version-specific expression     */
/*  functions, its implementation is deferred to after the subtypes.         */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsExprEvalDay(KHE_DRS_EXPR e, KHE_DRS_SOLN prev_soln,
  KHE_DRS_SOLN next_soln, KHE_DYNAMIC_RESOURCE_SOLVER drs);
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprEvalSignature(KHE_DRS_EXPR e, KHE_DRS_SIGNER dsg,         */
/*    KHE_DRS_SIGNATURE prev_sig, int next_di, KHE_DRS_SIGNATURE sig,        */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, bool debug)                           */
/*                                                                           */
/*  Evaluate expression e on day next_di, which is one of e's open days.     */
/*  The results go into sig.                                                 */
/*                                                                           */
/*  This function switches on the expression type and then calls a           */
/*  version specific to that type.  Like all version-specific expression     */
/*  functions, its implementation is deferred to after the subtypes.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprEvalSignature(KHE_DRS_EXPR e, KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE prev_sig, int next_di, KHE_DRS_SIGNATURE sig,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, bool debug);


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR - debug"                                         */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprDoDebug(KHE_DRS_EXPR e, KHE_DRS_SOLN soln,                */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp);                            */
/*                                                                           */
/*  Unindented debug print of that part of e which is specific to it.        */
/*  If soln != NULL, also print e's signature (if any) in soln.              */
/*                                                                           */
/*  This function switches on the expression type and then calls a           */
/*  version specific to that type.  Like all version-specific expression     */
/*  functions, its implementation is deferred to after the subtypes.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprDoDebug(KHE_DRS_EXPR e, KHE_DRS_SOLN soln,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp);


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprDebug(KHE_DRS_EXPR e, KHE_DYNAMIC_RESOURCE_SOLVER drs,    */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of e onto fp with the given verbosity and indent.            */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprDebug(KHE_DRS_EXPR e, KHE_DYNAMIC_RESOURCE_SOLVER drs,
  int verbosity, int indent, FILE *fp)
{
  /* char buff2[100]; */  KHE_DRS_EXPR child_e;  int i;
  if( indent >= 0 )
  {
    fprintf(fp, "%*s[ ", indent, "");
    KheDrsExprDoDebug(e, NULL, drs, fp);
    fprintf(fp, " p %p pi %d oc %d", (void *) e, e->postorder_index,
      KheDrsOpenChildrenCount(&e->open_children_by_day));
      /* HaArrayCount(e->open_children_by_day.children)); */
    /* , KheDrsDomTestShow(e->dom_test, buff2)); */
    /* ***
    fprintf(fp, " pi %d sd %s oc %d dt %s", e->postorder_index,
      bool_show(e->single_day), HaArrayCount(e->open_children),
      KheDrsDomTestShow(e->dom_test, buff2));
    *** */
    if( verbosity >= 2 && HaArrayCount(e->children) > 0 )
    {
      fprintf(fp, "\n");
      HaArrayForEach(e->children, child_e, i)
	KheDrsExprDebug(child_e, drs, verbosity, indent + 2, fp);
      fprintf(fp, "%*s]\n", indent, "");
    }
    else
      fprintf(fp, " ]\n");
  }
  else
    KheDrsExprDoDebug(e, NULL, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprDebugSoln(KHE_DRS_EXPR e, KHE_DRS_SOLN soln,              */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, int verbosity, int indent, FILE *fp)  */
/*                                                                           */
/*  Debug print of e onto fp with the given verbosity and indent.  Here      */
/*  e is known to be active in soln.                                         */
/*                                                                           */
/*****************************************************************************/

/* *** not currently used
static void KheDrsExprDebugSoln(KHE_DRS_EXPR e, KHE_DRS_SOLN soln,
  int verbosity, int indent, FILE *fp)
{
  if( indent >= 0 )
    fprintf(fp, "%*s", indent, "");
  KheDrsExprDoDebug(e, soln, drs, fp);
  if( indent > 0 )
    fprintf(fp, "\n");
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_COST KheDrsCost(KHE_COST_FUNCTION cost_fn, KHE_COST combined_weight, */
/*    int dev)                                                               */
/*                                                                           */
/*****************************************************************************/

/* ***
static KHE_COST KheDrsCost(KHE_COST_FUNCTION cost_fn, KHE_COST combined_weight,
  int dev)
{
  switch( cost_fn )
  {
    case KHE_STEP_COST_FUNCTION:

        return dev > 0 ? combined_weight : 0;

    case KHE_LINEAR_COST_FUNCTION:

        return dev * combined_weight;

    case KHE_QUADRATIC_COST_FUNCTION:

	return dev * dev * combined_weight;

    default:

	HnAbort("KheDrsCost internal error");
	return KheCost(0, 0);  ** keep compiler happy **
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprDebugSignature(KHE_DRS_EXPR e, KHE_DRS_SOLN soln,         */
/*    bool is_float, FILE *fp)                                               */
/*                                                                           */
/*  If soln != NULL and e has a signature entry in soln, debug that entry.   */
/*                                                                           */
/*  NB soln == NULL does not mean that there is a valid solution whose       */
/*  value is NULL, it just means that we are doing this debug without the    */
/*  benefit of a current solution, and so this function has nothing to do.   */
/*                                                                           */
/*****************************************************************************/
static KHE_DRS_DAY KheDrsSolnDay(KHE_DRS_SOLN soln,
  KHE_DYNAMIC_RESOURCE_SOLVER drs);

static void KheDrsExprDebugSignature(KHE_DRS_EXPR e, KHE_DRS_SOLN soln,
  bool is_float, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  /* int si;  KHE_DRS_DAY day; */
  if( soln != NULL )
  {
    fprintf(fp, "  KheDrsExprDebugSignature decommissioned");
    /* *** not sure how to upgrade this, and anyway soln == NULL in all calls
    day = KheDrsSolnDay(soln, drs);
    if( day != NULL && KheDrsExprInSignature(e, day, &si) )
    {
      if( is_float )
	fprintf(fp, " $%d:%.2f$", si,
	  (float) HaArray(soln->sig.states, si) / 100.0);
      else
	fprintf(fp, " $%d:%d$", si, HaArray(soln->sig.states, si));
    }
    *** */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_ASSIGNED_TASK"                                   */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SHIFT KheDrsExprAssignedTaskShift(KHE_DRS_EXPR_ASSIGNED_TASK eat)*/
/*                                                                           */
/*  Return the shift that determines the value of eat.                       */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused
static KHE_DRS_SHIFT KheDrsExprAssignedTaskShift(KHE_DRS_EXPR_ASSIGNED_TASK eat)
{
  return eat->task_on_day->encl_dt->encl_dtc->encl_shift;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprAssignedTaskShiftIndexInCycle(                             */
/*    KHE_DRS_EXPR_ASSIGNED_TASK eat)                                        */
/*                                                                           */
/*  Return the index in the cycle of eat's shift.                            */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused
static int KheDrsExprAssignedTaskShiftIndexInCycle(
  KHE_DRS_EXPR_ASSIGNED_TASK eat)
{
  return eat->task_on_day->encl_dt->encl_dtc->encl_shift->index_in_cycle;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_ASSIGNED_TASK KheDrsExprAssignedTaskMake(                   */
/*    KHE_DRS_TASK_ON_DAY dtd, KHE_RESOURCE_GROUP rg,                        */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make an assigned task expression object with these attributes.           */
/*  Or if there is already such an expression in dtd's list, use that.       */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_ASSIGNED_TASK KheDrsExprAssignedTaskMake(
  KHE_DRS_TASK_ON_DAY dtd, KHE_RESOURCE_GROUP rg,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_ASSIGNED_TASK res;
  /* ***
  HnAssert(dtd->task != NULL, "KheDrsExprAssignedTaskMake internal error");
  *** */
  if( KheDrsTaskOnDayHasAssignedTaskExpr(dtd, rg, &res) )
    return res;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_ASSIGNED_TASK_TAG,
    NULL, drs);
  res->task_on_day = dtd;
  res->resource_group = rg;
  KheDrsTaskOnDayAddAssignedTaskExpr(dtd, res);
  KheDrsExprInitEnd((KHE_DRS_EXPR) res, drs);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprAssignedTaskValue(KHE_DRS_EXPR_ASSIGNED_TASK eat,          */
/*    KHE_RESOURCE r)                                                        */
/*                                                                           */
/*  Return the value of eat when its task is assigned r (possibly NULL).     */
/*  This will be 1 when r is non-NULL and lies in eat's resource group.      */
/*  For efficiency, eat's resource group may be NULL, meaning all resources. */
/*                                                                           */
/*****************************************************************************/

static int KheDrsExprAssignedTaskValue(KHE_DRS_EXPR_ASSIGNED_TASK eat,
  KHE_RESOURCE r)
{
  KHE_RESOURCE_GROUP rg;
  if( r == NULL )
    return 0;
  rg = eat->resource_group;
  return (rg == NULL || KheResourceGroupContains(rg, r)) ? 1 : 0;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprAssignedTaskLeafSet(KHE_DRS_EXPR_ASSIGNED_TASK eat,       */
/*    KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_RESOURCE dr)                          */
/*                                                                           */
/*  Update the value of eat to reflect the assignment of dr to dtd.  Here    */
/*  dtd is known to be the relevant task (which is why it is unused), and    */
/*  dr is known to represent a non-NULL resource, not a non-assignment.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprAssignedTaskLeafSet(KHE_DRS_EXPR_ASSIGNED_TASK eat,
  KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_RESOURCE dr)
{
  eat->u.int_val = KheDrsExprAssignedTaskValue(eat, dr->resource);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprAssignedTaskLeafClear(KHE_DRS_EXPR_ASSIGNED_TASK eat)     */
/*                                                                           */
/*  Inform leaf expression ebt that the assignment of drd to dtd carried     */
/*  out by KheDrsExprAssignedTaskLeafSet above has been removed.             */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprAssignedTaskLeafClear(KHE_DRS_EXPR_ASSIGNED_TASK eat)
{
  eat->u.int_val = 0;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprAssignedTaskSetClosedValue(KHE_DRS_EXPR_ASSIGNED_TASK eat,*/
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing eat, set its value suitably for its closed state.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprAssignedTaskSetClosedValue(KHE_DRS_EXPR_ASSIGNED_TASK eat,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_RESOURCE_ON_DAY drd;  KHE_RESOURCE r;
  drd = eat->task_on_day->closed_asst;
  r = (drd == NULL ? NULL : drd->encl_dr->resource);
  eat->u.int_val = KheDrsExprAssignedTaskValue(eat, r);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprAssignedTaskDoDebug(KHE_DRS_EXPR_ASSIGNED_TASK eat,       */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of eat which is specific to it.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprAssignedTaskDoDebug(KHE_DRS_EXPR_ASSIGNED_TASK eat,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "ASSIGNED_TASK(");
  HnAssert(eat->task_on_day->task != NULL,
    "KheDrsExprAssignedTaskDoDebug internal error");
  KheTaskDebug(eat->task_on_day->task, 1, -1, fp);
  fprintf(fp, ", %s, %d)", eat->resource_group == NULL ? "-" :
    KheResourceGroupId(eat->resource_group), eat->u.int_val);
  KheDrsExprDebugSignature((KHE_DRS_EXPR) eat, soln, false, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_BUSY_TIME"                                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_BUSY_TIME KheDrsExprBusyTimeMake(                           */
/*    KHE_DRS_RESOURCE_ON_DAY drd, KHE_TIME time,                            */
/*    KHE_DRS_DOM_TEST dom_test, KHE_DYNAMIC_RESOURCE_SOLVER drs)            */
/*                                                                           */
/*  Make a new KHE_DRS_EXPR_BUSY_TIME object with these attributes.          */
/*  Or retrieve an equivalent existing expression if there is one.           */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_BUSY_TIME KheDrsExprBusyTimeMake(
  KHE_DRS_RESOURCE_ON_DAY drd, KHE_TIME time,
  /* KHE_DRS_DOM_TEST dom_test, */ KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_BUSY_TIME res;  KHE_DRS_EXPR e;
  if( KheDrsResourceOnDayHasTimeExpr(drd, KHE_DRS_EXPR_BUSY_TIME_TAG,
	/* dom_test, */ time, drs, &e) )
    return (KHE_DRS_EXPR_BUSY_TIME) e;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_BUSY_TIME_TAG,
    /* dom_test, */ drd->encl_dr, drs);
  res->resource_on_day = drd;
  res->time = time;
  KheDrsResourceOnDayAddExpr(drd, (KHE_DRS_EXPR) res);
  KheDrsExprInitEnd((KHE_DRS_EXPR) res, drs);  /* will set u.int_val */
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprBusyTimeLeafSet(KHE_DRS_EXPR_BUSY_TIME ebt,               */
/*    KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_RESOURCE dr)                          */
/*                                                                           */
/*  Set the value of ebt, given that dtd has just been assigned dr.          */
/*  Here dtd is known to be an actual task, not a non-assignment, and dr     */
/*  is known to be concerned with the same resource as ebt is.               */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprBusyTimeLeafSet(KHE_DRS_EXPR_BUSY_TIME ebt,
  KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_RESOURCE dr)
{
  ebt->u.int_val = (dtd->time == ebt->time ? 1 : 0);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprBusyTimeLeafClear(KHE_DRS_EXPR_BUSY_TIME ebt)             */
/*                                                                           */
/*  Inform leaf expression ebt that the assignment of drd to dtd carried     */
/*  out by KheDrsExprBusyTimeLeafSet above has been removed.                 */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprBusyTimeLeafClear(KHE_DRS_EXPR_BUSY_TIME ebt)
{
  ebt->u.int_val = 0;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprBusyTimeSetClosedValue(KHE_DRS_EXPR_BUSY_TIME ebt,        */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing ebt, set its value suitably for its closed state.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprBusyTimeSetClosedValue(KHE_DRS_EXPR_BUSY_TIME ebt,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_TASK_ON_DAY dtd;
  dtd = ebt->resource_on_day->closed_asst;
  ebt->u.int_val = (dtd != NULL && dtd->time == ebt->time ? 1 : 0);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprBusyTimeDoDebug(KHE_DRS_EXPR_BUSY_TIME ebt,               */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of ebt which is specific to it.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprBusyTimeDoDebug(KHE_DRS_EXPR_BUSY_TIME ebt,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "BUSY_TIME(%s, %s, iv %d)",
    KheDrsResourceId(ebt->resource_on_day->encl_dr),
    KheTimeId(ebt->time), ebt->u.int_val);
  KheDrsExprDebugSignature((KHE_DRS_EXPR) ebt, soln, false, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_FREE_TIME"                                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_FREE_TIME KheDrsExprFreeTimeMake(                           */
/*    KHE_DRS_RESOURCE_ON_DAY drd, KHE_TIME time,                            */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new KHE_DRS_EXPR_FREE_TIME object with these attributes.          */
/*  Or retrieve an equivalent existing expression if there is one.           */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_FREE_TIME KheDrsExprFreeTimeMake(
  KHE_DRS_RESOURCE_ON_DAY drd, KHE_TIME time, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_FREE_TIME res;  KHE_DRS_EXPR e;
  if( KheDrsResourceOnDayHasTimeExpr(drd, KHE_DRS_EXPR_FREE_TIME_TAG,
      time, drs, &e) )
    return (KHE_DRS_EXPR_FREE_TIME) e;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_FREE_TIME_TAG,
    drd->encl_dr, drs);
  res->resource_on_day = drd;
  res->time = time;
  KheDrsResourceOnDayAddExpr(drd, (KHE_DRS_EXPR) res);
  KheDrsExprInitEnd((KHE_DRS_EXPR) res, drs);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFreeTimeLeafSet(KHE_DRS_EXPR_FREE_TIME eft,               */
/*    KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_RESOURCE dr)                          */
/*                                                                           */
/*  Set the value of eft, given that dtd has just been assigned dr.          */
/*  Here dtd is known to be an actual task, not a non-assignment, and dr     */
/*  is known to be concerned with the same resource as eft is.               */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFreeTimeLeafSet(KHE_DRS_EXPR_FREE_TIME eft,
  KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_RESOURCE dr)
{
  eft->u.int_val = (dtd->time == eft->time ? 0 : 1);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFreeTimeLeafClear(KHE_DRS_EXPR_FREE_TIME eft)             */
/*                                                                           */
/*  Inform leaf expression eft that the assignment of drd to dtd carried     */
/*  out by KheDrsExprFreeTimeLeafSet above has been removed.                 */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFreeTimeLeafClear(KHE_DRS_EXPR_FREE_TIME eft)
{
  eft->u.int_val = 1;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFreeTimeSetClosedValue(KHE_DRS_EXPR_FREE_TIME eft,        */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing eft, set its value suitably for its closed state.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFreeTimeSetClosedValue(KHE_DRS_EXPR_FREE_TIME eft,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_TASK_ON_DAY dtd;
  dtd = eft->resource_on_day->closed_asst;
  eft->u.int_val = (dtd != NULL && dtd->time == eft->time ? 0 : 1);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFreeTimeDoDebug(KHE_DRS_EXPR_FREE_TIME eft,               */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of eft which is specific to it.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFreeTimeDoDebug(KHE_DRS_EXPR_FREE_TIME eft,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "FREE_TIME(%s, %s, iv %d)",
    KheDrsResourceId(eft->resource_on_day->encl_dr),
    KheTimeId(eft->time), eft->u.int_val);
  KheDrsExprDebugSignature((KHE_DRS_EXPR) eft, soln, false, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_WORK_TIME"                                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_WORK_TIME KheDrsExprWorkTimeMake(                           */
/*    KHE_DRS_RESOURCE_ON_DAY drd, KHE_TIME time,                            */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new KHE_DRS_EXPR_WORK_TIME object with these attributes.          */
/*  Or retrieve an equivalent existing expression if there is one.           */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_WORK_TIME KheDrsExprWorkTimeMake(
  KHE_DRS_RESOURCE_ON_DAY drd, KHE_TIME time,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_WORK_TIME res;  KHE_DRS_EXPR e;
  if( KheDrsResourceOnDayHasTimeExpr(drd, KHE_DRS_EXPR_WORK_TIME_TAG,
	time, drs, &e) )
    return (KHE_DRS_EXPR_WORK_TIME) e;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_WORK_TIME_TAG,
    drd->encl_dr, drs);
  res->resource_on_day = drd;
  res->time = time;
  KheDrsResourceOnDayAddExpr(drd, (KHE_DRS_EXPR) res);
  KheDrsExprInitEnd((KHE_DRS_EXPR) res, drs);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprWorkTimeLeafSet(KHE_DRS_EXPR_WORK_TIME ewt,               */
/*    KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_RESOURCE dr)                          */
/*                                                                           */
/*  Set the value of ewt, given that dtd has just been assigned dr.          */
/*  Here dtd is known to be an actual task, not a non-assignment, and dr     */
/*  is known to be concerned with the same resource as ewt is.               */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprWorkTimeLeafSet(KHE_DRS_EXPR_WORK_TIME ewt,
  KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_RESOURCE dr)
{
  ewt->u.float_val =
    (dtd->time == ewt->time ? KheTaskWorkloadPerTime(dtd->task) : 0.0);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprWorkTimeLeafClear(KHE_DRS_EXPR_WORK_TIME ewt)             */
/*                                                                           */
/*  Inform leaf expression ebt that the assignment of drd to dtd carried     */
/*  out by KheDrsExprWorkTimeLeafSet above has been removed.                 */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprWorkTimeLeafClear(KHE_DRS_EXPR_WORK_TIME ewt)
{
  ewt->u.float_val = 0.0;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprWorkTimeSetClosedValue(KHE_DRS_EXPR_WORK_TIME ewt,        */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing ewt, set its value suitably for its closed state.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprWorkTimeSetClosedValue(KHE_DRS_EXPR_WORK_TIME ewt,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_TASK_ON_DAY dtd;
  dtd = ewt->resource_on_day->closed_asst;
  ewt->u.float_val = (dtd != NULL && dtd->time == ewt->time ?
    KheTaskWorkloadPerTime(dtd->task) : 0.0);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprWorkTimeDoDebug(KHE_DRS_EXPR_WORK_TIME ewt,               */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of ewt which is specific to it.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprWorkTimeDoDebug(KHE_DRS_EXPR_WORK_TIME ewt,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "WORK_TIME(%s, %s)",
    KheDrsResourceId(ewt->resource_on_day->encl_dr), KheTimeId(ewt->time));
  KheDrsExprDebugSignature((KHE_DRS_EXPR) ewt, soln, true, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_BUSY_DAY"                                        */
/*                                                                           */
/*****************************************************************************/

#define BUSY_DAY_DEBUG 0
#define BUSY_DAY_TIME_GROUP_ID "1Sat"
#define BUSY_DAY_RESOURCE "TR_25"


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsBusyDayDebug(KHE_DRS_EXPR_BUSY_DAY ebd)                       */
/*                                                                           */
/*  Return true if we want to debug ebd.                                     */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsBusyDayDebug(KHE_DRS_EXPR_BUSY_DAY ebd,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_RESOURCE r;  KHE_TIME_GROUP tg;
  if( RERUN && drs->rerun != NULL && BUSY_DAY_DEBUG )
  {
    r = ebd->resource_on_day->encl_dr->resource;
    tg = ebd->time_group;
    return strcmp(KheResourceId(r), BUSY_DAY_RESOURCE) == 0 &&
      strcmp(KheTimeGroupId(tg), BUSY_DAY_TIME_GROUP_ID) == 0;
  }
  else
    return false;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_BUSY_DAY KheDrsExprBusyDayMake(KHE_DRS_RESOURCE_ON_DAY drd, */
/*    KHE_TIME_GROUP tg, KHE_DYNAMIC_RESOURCE_SOLVER drs)                    */
/*                                                                           */
/*  Make a new KHE_DRS_EXPR_BUSY_DAY object with these attributes.           */
/*  Or retrieve an equivalent existing expression if there is one.           */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_BUSY_DAY KheDrsExprBusyDayMake(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_TIME_GROUP tg, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_BUSY_DAY res;  KHE_DRS_EXPR e;
  if( KheDrsResourceOnDayHasDayExpr(drd, KHE_DRS_EXPR_BUSY_DAY_TAG, tg,drs,&e) )
    return (KHE_DRS_EXPR_BUSY_DAY) e;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_BUSY_DAY_TAG,
    drd->encl_dr, drs);
  res->resource_on_day = drd;
  res->time_group = tg;
  KheDrsResourceOnDayAddExpr(drd, (KHE_DRS_EXPR) res);
  KheDrsExprInitEnd((KHE_DRS_EXPR) res, drs);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprBusyDayLeafSet(KHE_DRS_EXPR_BUSY_DAY ebd,                 */
/*    KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_RESOURCE dr,                          */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Set the value of ebd, given that dtd has just been assigned dr.          */
/*  Here dtd is known to be an actual task, not a non-assignment, and dr     */
/*  is known to be concerned with the same resource as ebt is.               */
/*                                                                           */
/*****************************************************************************/
static void KheDrsExprBusyDayDoDebug(KHE_DRS_EXPR_BUSY_DAY ebd,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp);

static void KheDrsExprBusyDayLeafSet(KHE_DRS_EXPR_BUSY_DAY ebd,
  KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_RESOURCE dr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  ebd->u.int_val = 1;
  if( KheDrsBusyDayDebug(ebd, drs) )
  {
    fprintf(stderr, "  KheDrsExprBusyDayLeafSet(");
    KheDrsExprBusyDayDoDebug(ebd, NULL, drs, stderr);
    fprintf(stderr, ")\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprBusyDayLeafClear(KHE_DRS_EXPR_BUSY_DAY ebd)               */
/*                                                                           */
/*  Inform leaf soln ebd that the assignment of drd to dtd carried out by    */
/*  KheDrsExprBusyDayLeafSet above has been removed.                         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprBusyDayLeafClear(KHE_DRS_EXPR_BUSY_DAY ebd,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  ebd->u.int_val = 0;
  if( KheDrsBusyDayDebug(ebd, drs) )
  {
    fprintf(stderr, "  KheDrsExprBusyDayLeafClear(");
    KheDrsExprBusyDayDoDebug(ebd, NULL, drs, stderr);
    fprintf(stderr, ")\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprBusyDaySetClosedValue(KHE_DRS_EXPR_BUSY_DAY ebd,          */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing ebd, set its value suitably for its closed state.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprBusyDaySetClosedValue(KHE_DRS_EXPR_BUSY_DAY ebd,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_TASK_ON_DAY dtd;
  dtd = ebd->resource_on_day->closed_asst;
  ebd->u.int_val = (dtd != NULL ? 1 : 0);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprBusyDayDoDebug(KHE_DRS_EXPR_BUSY_DAY ebd,                 */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of ebd which is specific to it.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprBusyDayDoDebug(KHE_DRS_EXPR_BUSY_DAY ebd,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "BUSY_DAY(%s, %s, iv %d)",
    KheDrsResourceId(ebd->resource_on_day->encl_dr),
    KheTimeGroupId(ebd->time_group), ebd->u.int_val);
  KheDrsExprDebugSignature((KHE_DRS_EXPR) ebd, soln, false, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_FREE_DAY"                                        */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_FREE_DAY KheDrsExprFreeDayMake(KHE_DRS_RESOURCE_ON_DAY drd, */
/*    KHE_TIME_GROUP tg, KHE_DYNAMIC_RESOURCE_SOLVER drs)                    */
/*                                                                           */
/*  Make a new KHE_DRS_EXPR_FREE_DAY object with these attributes.           */
/*  Or retrieve an equivalent existing expression if there is one.           */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_FREE_DAY KheDrsExprFreeDayMake(KHE_DRS_RESOURCE_ON_DAY drd,
  KHE_TIME_GROUP tg, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_FREE_DAY res;  KHE_DRS_EXPR e;
  if( KheDrsResourceOnDayHasDayExpr(drd, KHE_DRS_EXPR_FREE_DAY_TAG, tg,drs,&e) )
    return (KHE_DRS_EXPR_FREE_DAY) e;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_FREE_DAY_TAG,
    drd->encl_dr, drs);
  res->resource_on_day = drd;
  res->time_group = tg;
  KheDrsResourceOnDayAddExpr(drd, (KHE_DRS_EXPR) res);
  KheDrsExprInitEnd((KHE_DRS_EXPR) res, drs);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFreeDayLeafSet(KHE_DRS_EXPR_FREE_DAY efd,                 */
/*    KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_RESOURCE dr)                          */
/*                                                                           */
/*  Set the value of efd, given that dtd has just been assigned dr.          */
/*  Here dtd is known to be an actual task, not a non-assignment, and dr     */
/*  is known to be concerned with the same resource as efd is.               */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFreeDayLeafSet(KHE_DRS_EXPR_FREE_DAY efd,
  KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_RESOURCE dr)
{
  efd->u.int_val = 0;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFreeDayLeafClear(KHE_DRS_EXPR_FREE_DAY efd)               */
/*                                                                           */
/*  Inform leaf expression eft that the assignment of drd to dtd carried     */
/*  out by KheDrsExprFreeDayLeafSet above has been removed.                  */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFreeDayLeafClear(KHE_DRS_EXPR_FREE_DAY efd)
{
  efd->u.int_val = 1;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFreeDaySetClosedValue(KHE_DRS_EXPR_FREE_DAY efd,          */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing efd, set its value suitably for its closed state.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFreeDaySetClosedValue(KHE_DRS_EXPR_FREE_DAY efd,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_TASK_ON_DAY dtd;
  dtd = efd->resource_on_day->closed_asst;
  efd->u.int_val = (dtd != NULL ? 0 : 1);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFreeDayDoDebug(KHE_DRS_EXPR_FREE_DAY efd,                 */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of efd which is specific to it.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFreeDayDoDebug(KHE_DRS_EXPR_FREE_DAY efd,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "FREE_DAY(%s, %s, iv %d)",
    KheDrsResourceId(efd->resource_on_day->encl_dr),
    KheTimeGroupId(efd->time_group), efd->u.int_val);
  KheDrsExprDebugSignature((KHE_DRS_EXPR) efd, soln, false, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_WORK_DAY"                                        */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_WORK_DAY KheDrsExprWorkDayMake(KHE_TIME_GROUP tg,           */
/*    KHE_DRS_DOM_TEST dom_test, KHE_DYNAMIC_RESOURCE_SOLVER drs)            */
/*                                                                           */
/*  Make a new KHE_DRS_EXPR_WORK_DAY object with these attributes.           */
/*  Or retrieve an equivalent existing expression if there is one.           */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused, although I really should be using it
static KHE_DRS_EXPR_WORK_DAY KheDrsExprWorkDayMake(KHE_TIME_GROUP tg,
  KHE_DRS_DOM_TEST dom_test, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_WORK_DAY res;  KHE_DRS_EXPR e;
  if( KheDrsHasDayExpr(drs, KHE_DRS_EXPR_WORK_DAY_TAG, dom_test, tg, &e) )
    return (KHE_DRS_EXPR_WORK_DAY) e;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_WORK_DAY_TAG,
    ** dom_test, ** drd->encl_dr, drs);
  res->u.float_val = 0.0;  ** default value, used sometimes **
  res->time_group = tg;
  KheDrsAddDayExpr(drs, tg, (KHE_DRS_EXPR) res);
  KheDrsExprInitEnd((KHE_DRS_EXPR) res, drs);
  return res;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprWorkDayLeafSet(KHE_DRS_EXPR_WORK_DAY ewd,                 */
/*    KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_RESOURCE dr)                          */
/*                                                                           */
/*  Set the value of ewd, given that dtd has just been assigned dr.          */
/*  Here dtd is known to be an actual task, not a non-assignment, and dr     */
/*  is known to be concerned with the same resource as ewd is.               */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprWorkDayLeafSet(KHE_DRS_EXPR_WORK_DAY ewd,
  KHE_DRS_TASK_ON_DAY dtd, KHE_DRS_RESOURCE dr)
{
  ewd->u.float_val = KheTaskWorkloadPerTime(dtd->task);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprWorkDayLeafClear(KHE_DRS_EXPR_WORK_DAY ewd)               */
/*                                                                           */
/*  Inform leaf expression ewt that the assignment of drd to dtd carried     */
/*  out by KheDrsExprWorkDayLeafSet above has been removed.                  */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprWorkDayLeafClear(KHE_DRS_EXPR_WORK_DAY ewd)
{
  ewd->u.float_val = 0.0;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprWorkDaySetClosedValue(KHE_DRS_EXPR_WORK_DAY ewd,          */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing ewd, set its value suitably for its closed state.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprWorkDaySetClosedValue(KHE_DRS_EXPR_WORK_DAY ewd,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_TASK_ON_DAY dtd;
  dtd = ewd->resource_on_day->closed_asst;
  ewd->u.int_val = (dtd != NULL ? KheTaskWorkloadPerTime(dtd->task) : 0.0);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprWorkDayDoDebug(KHE_DRS_EXPR_WORK_DAY ewd,                 */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of ewd which is specific to it.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprWorkDayDoDebug(KHE_DRS_EXPR_WORK_DAY ewd,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "WORK_DAY(%s, %s)",
    KheDrsResourceId(ewd->resource_on_day->encl_dr),
    KheTimeGroupId(ewd->time_group));
  KheDrsExprDebugSignature((KHE_DRS_EXPR) ewd, soln, true, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_OR"                                              */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_OR KheDrsExprOrMake(KHE_DRS_RESOURCE dr,                    */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new KHE_DRS_EXPR_OR object with these attributes.                 */
/*  The caller must call KheDrsExprInitEnd after adding all the children.    */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_OR KheDrsExprOrMake(KHE_DRS_RESOURCE dr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_OR res;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_OR_TAG, dr, drs);
  res->closed_state = 0;
  /* later call to KheDrsExprInitEnd will set u.int_val */
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprOrAddChild(KHE_DRS_EXPR_OR eo, KHE_DRS_EXPR child_e,      */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  New child child_e has just been added to eo.  Update eo's closed         */
/*  state appropriately.  This happens to be the same as closing a child.    */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprOrAddChild(KHE_DRS_EXPR_OR eo, KHE_DRS_EXPR child_e,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  if( child_e->u.int_val == 1 )
    eo->closed_state += 1;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprOrChildHasOpened(KHE_DRS_EXPR_OR eo,                      */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of opening eo for solving, eo's child_index'th child, child_e,   */
/*  has been opened.  Update eo's closed state to reflect this.              */
/*                                                                           */
/*  The closed state of eo is the number of eo's closed children whose       */
/*  value is 1, so this update subtracts 1 from the closed state if          */
/*  child_e's value (which preserves its value before opening) is 1.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprOrChildHasOpened(KHE_DRS_EXPR_OR eo,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  if( child_e->u.int_val == 1 )
    eo->closed_state -= 1;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprOrChildHasClosed(KHE_DRS_EXPR_OR eo,                      */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of closing eo, eo's child_index'th child, child_e, has been      */
/*  closed.  Update eo's closed state to reflect this.                       */
/*                                                                           */
/*  The closed state of eo is the number of eo's closed children whose       */
/*  value is 1, so this update adds 1 to the closed state if child_e's       */
/*  value (which is up to date because it is closed first) is 1.             */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprOrChildHasClosed(KHE_DRS_EXPR_OR eo,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  if( child_e->u.int_val == 1 )
    eo->closed_state += 1;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprOrSetClosedValue(KHE_DRS_EXPR_OR eo,                      */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing eo, set its value suitably for its closed state.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprOrSetClosedValue(KHE_DRS_EXPR_OR eo,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  eo->u.int_val = (eo->closed_state > 0 ? 1 : 0);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprOrDomTest(KHE_DRS_EXPR_OR eo,                 */
/*    int open_child_index)                                                  */
/*                                                                           */
/*  Return a dom test suited to eo on its open_child_index'th open day.      */
/*                                                                           */
/*****************************************************************************/

/* *** unused
static KHE_DRS_DOM_TEST KheDrsExprOrDomTest(KHE_DRS_EXPR_OR eo,
  int open_child_index)
{
  ** placeholder; sti ll to do **
  return KheDrsDomTestMake(false, INT_MAX, 0, false, 0);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsSolnOpenDayIndex(KHE_DRS_SOLN soln)                            */
/*                                                                           */
/*  Return the open day index of soln's day.  This day is assumed to exist,  */
/*  which means that KheDrsSolnOpenDayIndex cannot be called on the root     */
/*  solution since it has no day.                                            */
/*                                                                           */
/*****************************************************************************/

/* ***
static int KheDrsSolnOpenDayIndex(KHE_DRS_SOLN soln)
{
  return soln->day->open_day_index;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprOrEvalDay(KHE_DRS_EXPR_OR eo, KHE_DRS_SOLN prev_soln,     */
/*    KHE_DRS_SOLN next_soln, KHE_DYNAMIC_RESOURCE_SOLVER drs)               */
/*                                                                           */
/*  Evaluate expression eo within next_soln.                                 */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsExprOrEvalDay(KHE_DRS_EXPR_OR eo, KHE_DRS_SOLN prev_soln,
  KHE_DRS_SOLN next_soln, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int i, i1, i2, val, di;  KHE_DRS_EXPR child_e;

  di = KheDrsSolnOpenDayIndex(next_soln);
  if( KheDrsOpenChildrenIndexIsFirst(&eo->open_children_by_day, di) )
  {
    ** first day, so we have a 0 (false) value here **
    val = 0;
  }
  else
  {
    ** not first day, so retrieve a previous value **
    val = KheDrsExprDaySigVal((KHE_DRS_EXPR) eo, di - 1, prev_soln);
  }

  ** accumulate the values of the children of eo that finalized today **
  KheDrsOpenChildrenForEach(&eo->open_children_by_day, di, child_e, i)
    if( child_e->u.int_val == 1 )
      val = 1;

  if( KheDrsOpenChildrenIndexIsLast(&eo->open_children_by_day, di) )
  {
    ** last day; incorporate closed state and set value **
    if( eo->closed_state > 0 )
      val = 1;
    eo->u.int_val = val;
  }
  else
  {
    ** not last day; store val in next_soln **
    KheDrsExprPutSigVal((KHE_DRS_EXPR) eo, di, next_soln, val);
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprOrEvalSignature(KHE_DRS_EXPR_OR eo,                       */
/*    KHE_DRS_SOLN prev_soln, int next_di, KHE_DRS_SIGNATURE sig,            */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Evaluate expression eo within sig.                                       */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprOrEvalSignature(KHE_DRS_EXPR_OR eo, KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE prev_sig, int next_di, KHE_DRS_SIGNATURE next_sig,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int i, i1, i2, val;  KHE_DRS_EXPR child_e;

  if( KheDrsOpenChildrenIndexIsFirst(&eo->open_children_by_day, next_di) )
  {
    /* first day, so we have a 0 (false) value here */
    val = 0;
  }
  else
  {
    /* not first day, so retrieve a previous value */
    val = KheDrsExprDaySigVal((KHE_DRS_EXPR) eo, next_di - 1, prev_sig);
  }

  /* accumulate the values of the children of eo that finalized today */
  KheDrsOpenChildrenForEach(&eo->open_children_by_day, next_di, child_e, i)
    if( child_e->u.int_val == 1 )
      val = 1;

  if( KheDrsOpenChildrenIndexIsLast(&eo->open_children_by_day, next_di) )
  {
    /* last day; incorporate closed state and set value */
    if( eo->closed_state > 0 )
      val = 1;
    eo->u.int_val = val;
  }
  else
  {
    /* not last day; store val in next_soln */
    KheDrsSignatureAddState(next_sig, val, dsg, (KHE_DRS_EXPR) eo);
    /* KheDrsExprPutSigVal((KHE_DRS_EXPR) eo, next_di, sig, val); */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprOrDoDebug(KHE_DRS_EXPR_OR eo,                             */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of eo which is specific to it.       */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprOrDoDebug(KHE_DRS_EXPR_OR eo,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "OR(cs %d, iv %d)", eo->closed_state, eo->u.int_val);
  KheDrsExprDebugSignature((KHE_DRS_EXPR) eo, soln, false, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_AND"                                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_AND KheDrsExprAndMake(KHE_DRS_RESOURCE dr,                  */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new KHE_DRS_EXPR_AND object with these attributes.                */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_AND KheDrsExprAndMake(KHE_DRS_RESOURCE dr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_AND res;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_AND_TAG, dr, drs);
  res->u.int_val = UNDEF_INT;
  res->closed_state = 0;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprAndAddChild(KHE_DRS_EXPR_AND ea, KHE_DRS_EXPR child_e,    */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  New child child_e has just been added to ea.  Update ea's closed         */
/*  state appropriately.  This happens to be the same as closing a child.    */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprAndAddChild(KHE_DRS_EXPR_AND ea, KHE_DRS_EXPR child_e,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  if( child_e->u.int_val == 0 )
    ea->closed_state += 1;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprAndChildHasOpened(KHE_DRS_EXPR_AND ea,                    */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of opening ea for solving, ea's child_index'th child, child_e,   */
/*  has been opened.  Update ea's closed state to reflect this.              */
/*                                                                           */
/*  The closed state of ea is the number of ea's closed children whose       */
/*  value is 0, so this update subtracts 1 from the closed state if          */
/*  child_e's value (which preserves its value before opening) is 0.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprAndChildHasOpened(KHE_DRS_EXPR_AND ea,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  if( child_e->u.int_val == 0 )
    ea->closed_state -= 1;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprAndChildHasClosed(KHE_DRS_EXPR_AND ea,                    */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of closing ea, ea's child_index'th child, child_e, has been      */
/*  closed.  Update ea's closed state to reflect this.                       */
/*                                                                           */
/*  The closed state of ea is the number of ea's closed children whose       */
/*  value is 0, so this update adds 1 to the closed state if child_e's       */
/*  value (which is up to date because it is closed first) is 0.             */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprAndChildHasClosed(KHE_DRS_EXPR_AND ea,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  if( child_e->u.int_val == 0 )
    ea->closed_state += 1;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprAndSetClosedValue(KHE_DRS_EXPR_AND ea,                    */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing ea, set its value suitably for its closed state.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprAndSetClosedValue(KHE_DRS_EXPR_AND ea,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  ea->u.int_val = (ea->closed_state > 0 ? 0 : 1);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprAndDomTest(KHE_DRS_EXPR_AND ea,               */
/*    int open_child_index)                                                  */
/*                                                                           */
/*  Return a dom test suited to ea on its open_child_index'th open day.      */
/*                                                                           */
/*****************************************************************************/

/* *** unused
static KHE_DRS_DOM_TEST KheDrsExprAndDomTest(KHE_DRS_EXPR_AND ea,
  int open_child_index)
{
  ** placeholder; sti ll to do **
  return KheDrsDomTestMake(false, INT_MAX, 0, false, 0);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprAndEvalDay(KHE_DRS_EXPR_AND ea, KHE_DRS_SOLN prev_soln,   */
/*    KHE_DRS_SOLN next_soln, KHE_DYNAMIC_RESOURCE_SOLVER drs)               */
/*                                                                           */
/*  Evaluate expression ea within next_soln.                                 */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsExprAndEvalDay(KHE_DRS_EXPR_AND ea, KHE_DRS_SOLN prev_soln,
  KHE_DRS_SOLN next_soln, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int i, i1, i2, val, di;  KHE_DRS_EXPR child_e;

  di = KheDrsSolnOpenDayIndex(next_soln);
  if( KheDrsOpenChildrenIndexIsFirst(&ea->open_children_by_day, di) )
  {
    ** first day, so we have a 1 (true) value here **
    val = 1;
  }
  else
  {
    ** not first day, so retrieve a previous value **
    val = KheDrsExprDaySigVal((KHE_DRS_EXPR) ea, di - 1, prev_soln);
  }

  ** accumulate the values of the children of ea that finalized today **
  KheDrsOpenChildrenForEach(&ea->open_children_by_day, di, child_e, i)
    if( child_e->u.int_val == 0 )
      val = 0;

  if( KheDrsOpenChildrenIndexIsLast(&ea->open_children_by_day, di) )
  {
    ** last day; incorporate closed_state and set value **
    if( ea->closed_state > 0 )
      val = 0;
    ea->u.int_val = val;
  }
  else
  {
    ** not last day; store value in next_soln **
    KheDrsExprPutSigVal((KHE_DRS_EXPR) ea, di, next_soln, val);
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprAndEvalSignature(KHE_DRS_EXPR_AND ea,                     */
/*    KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, int next_di,           */
/*    KHE_DRS_SIGNATURE sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)                */
/*                                                                           */
/*  Evaluate expression ea within sig.                                       */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprAndEvalSignature(KHE_DRS_EXPR_AND ea,
  KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, int next_di,
  KHE_DRS_SIGNATURE next_sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int i, i1, i2, val;  KHE_DRS_EXPR child_e;

  if( KheDrsOpenChildrenIndexIsFirst(&ea->open_children_by_day, next_di) )
  {
    /* first day, so we have a 1 (true) value here */
    val = 1;
  }
  else
  {
    /* not first day, so retrieve a previous value */
    val = KheDrsExprDaySigVal((KHE_DRS_EXPR) ea, next_di - 1, prev_sig);
  }

  /* accumulate the values of the children of ea that finalized today */
  KheDrsOpenChildrenForEach(&ea->open_children_by_day, next_di, child_e, i)
    if( child_e->u.int_val == 0 )
      val = 0;

  if( KheDrsOpenChildrenIndexIsLast(&ea->open_children_by_day, next_di) )
  {
    /* last day; incorporate closed_state and set value */
    if( ea->closed_state > 0 )
      val = 0;
    ea->u.int_val = val;
  }
  else
  {
    /* not last day; store value in next_soln */
    KheDrsSignatureAddState(next_sig, val, dsg, (KHE_DRS_EXPR) ea);
    /* KheDrsExprPutSigVal((KHE_DRS_EXPR) ea, next_di, sig, val); */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprAndDoDebug(KHE_DRS_EXPR_AND ea,                           */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of ea which is specific to it.       */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprAndDoDebug(KHE_DRS_EXPR_AND ea,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "AND(cs %d, iv %d)", ea->closed_state, ea->u.int_val);
  KheDrsExprDebugSignature((KHE_DRS_EXPR) ea, soln, false, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_INT_SUM"                                         */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_INT_SUM KheDrsExprIntSumMake(KHE_DRS_RESOURCE dr,           */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new KHE_DRS_EXPR_INT_SUM object with these attributes.            */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_INT_SUM KheDrsExprIntSumMake(KHE_DRS_RESOURCE dr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_INT_SUM res;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_INT_SUM_TAG, dr, drs);
  res->u.int_val = UNDEF_INT;
  res->closed_state = 0;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSumAddChild(KHE_DRS_EXPR_INT_SUM eis,                  */
/*    KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  New child child_e has just been added to eis.  Update eis's closed       */
/*  state appropriately.  This happens to be the same as closing a child.    */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntSumAddChild(KHE_DRS_EXPR_INT_SUM eis,
  KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  eis->closed_state += child_e->u.int_val;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSumChildHasOpened(KHE_DRS_EXPR_INT_SUM eis,            */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of opening eis for solving, eis's child_index'th child, child_e, */
/*  has been opened.  Update eis's closed state to reflect this.             */
/*                                                                           */
/*  The closed state of eis is the sum of the values of eis's s closed       */
/*  children, so this update subtracts child_e's value (which preserves      */
/*  its value before opening) from the closed state.                         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntSumChildHasOpened(KHE_DRS_EXPR_INT_SUM eis,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  eis->closed_state -= child_e->u.int_val;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSumChildHasClosed(KHE_DRS_EXPR_INT_SUM eis,            */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of closing eis, eis's child_index'th child, child_e, has been    */
/*  closed.  Update eis's closed state to reflect this.                      */
/*                                                                           */
/*  The closed state of eis is the sum of the values of eis's s closed       */
/*  children, so this update adds child_e's value (which preserves           */
/*  its value before opening) to the closed state.                           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntSumChildHasClosed(KHE_DRS_EXPR_INT_SUM eis,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  eis->closed_state += child_e->u.int_val;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSumSetClosedValue(KHE_DRS_EXPR_INT_SUM eis,            */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing eis, set its value suitably for its closed state.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntSumSetClosedValue(KHE_DRS_EXPR_INT_SUM eis,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  eis->u.int_val = eis->closed_state;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprIntSumDomTest(KHE_DRS_EXPR_INT_SUM eis,       */
/*    int open_child_index)                                                  */
/*                                                                           */
/*  Return a dom test suited to eis on its open_child_index'th open day.     */
/*                                                                           */
/*****************************************************************************/

/* *** unused
static KHE_DRS_DOM_TEST KheDrsExprIntSumDomTest(KHE_DRS_EXPR_INT_SUM eis,
  int open_child_index)
{
  ** placeholder; sti ll to do **
  return KheDrsDomTestMake(false, INT_MAX, 0, false, 0);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSumEvalDay(KHE_DRS_EXPR_INT_SUM eis,                   */
/*    KHE_DRS_SOLN prev_soln, KHE_DRS_SOLN next_soln,                        */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Evaluate expression eis on day open_day_index.                           */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsExprIntSumEvalDay(KHE_DRS_EXPR_INT_SUM eis,
  KHE_DRS_SOLN prev_soln, KHE_DRS_SOLN next_soln,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int i, i1, i2, val, di;  KHE_DRS_EXPR child_e;

  di = KheDrsSolnOpenDayIndex(next_soln);
  if( KheDrsOpenChildrenIndexIsFirst(&eis->open_children_by_day, di) )
  {
    ** first day, so (ignoring closed_state) we have a 0 value here **
    val = 0;
  }
  else
  {
    ** not first day, so retrieve a previous value **
    val = KheDrsExprDaySigVal((KHE_DRS_EXPR) eis, di - 1, prev_soln);
  }

  ** accumulate the open values of the children of eis that finalized today **
  KheDrsOpenChildrenForEach(&eis->open_children_by_day, di, child_e, i)
    val += child_e->u.int_val;

  if( KheDrsOpenChildrenIndexIsLast(&eis->open_children_by_day, di) )
  {
    ** last day; add closed_state to the value **
    eis->u.int_val = val + eis->closed_state;
  }
  else
  {
    ** this not eis's last day; store value in next_soln **
    KheDrsExprPutSigVal((KHE_DRS_EXPR) eis, di, next_soln, val);
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSumEvalSignature(KHE_DRS_EXPR_INT_SUM eis,             */
/*    KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, int next_di,           */
/*    KHE_DRS_SIGNATURE sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)                */
/*                                                                           */
/*  Evaluate expression eis within sig.                                      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntSumEvalSignature(KHE_DRS_EXPR_INT_SUM eis,
  KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, int next_di,
  KHE_DRS_SIGNATURE next_sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int i, i1, i2, val;  KHE_DRS_EXPR child_e;

  if( KheDrsOpenChildrenIndexIsFirst(&eis->open_children_by_day, next_di) )
  {
    /* first day, so (ignoring closed_state) we have a 0 value here */
    val = 0;
  }
  else
  {
    /* not first day, so retrieve a previous value */
    val = KheDrsExprDaySigVal((KHE_DRS_EXPR) eis, next_di - 1, prev_sig);
  }

  /* accumulate the open values of the children of eis that finalized today */
  KheDrsOpenChildrenForEach(&eis->open_children_by_day, next_di, child_e, i)
    val += child_e->u.int_val;

  if( KheDrsOpenChildrenIndexIsLast(&eis->open_children_by_day, next_di) )
  {
    /* last day; add closed_state to the value */
    eis->u.int_val = val + eis->closed_state;
  }
  else
  {
    /* this not eis's last day; store value in next_soln */
    KheDrsSignatureAddState(next_sig, val, dsg, (KHE_DRS_EXPR) eis);
    /* KheDrsExprPutSigVal((KHE_DRS_EXPR) eis, next_di, sig, val); */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSumDoDebug(KHE_DRS_EXPR_INT_SUM eis,                   */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of eis which is specific to it.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntSumDoDebug(KHE_DRS_EXPR_INT_SUM eis,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "INT_SUM(%d)", eis->closed_state);
  KheDrsExprDebugSignature((KHE_DRS_EXPR) eis, soln, false, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_FLOAT_SUM"                                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_FLOAT_SUM KheDrsExprFloatSumMake(KHE_DRS_RESOURCE dr,       */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new KHE_DRS_EXPR_FLOAT_SUM object with these attributes.          */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_FLOAT_SUM KheDrsExprFloatSumMake(KHE_DRS_RESOURCE dr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_FLOAT_SUM res;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_FLOAT_SUM_TAG, dr, drs);
  res->u.float_val = UNDEF_FLOAT;
  res->closed_state = 0.0;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFloatSumAddChild(KHE_DRS_EXPR_FLOAT_SUM efs,              */
/*    KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  New child child_e has just been added to efs.  Update efs's closed       */
/*  state appropriately.  This happens to be the same as closing a child.    */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFloatSumAddChild(KHE_DRS_EXPR_FLOAT_SUM efs,
  KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  efs->closed_state += child_e->u.float_val;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFloatSumChildHasOpened(KHE_DRS_EXPR_FLOAT_SUM efs,        */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of opening efs for solving, efs's child_index'th child, child_e, */
/*  has been opened.  Update efs's closed state to reflect this.             */
/*                                                                           */
/*  The closed state of efs is the sum of the values of efs's closed         */
/*  children, so this update subtracts child_e's value (which preserves      */
/*  its value before opening) from the closed state.                         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFloatSumChildHasOpened(KHE_DRS_EXPR_FLOAT_SUM efs,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  efs->closed_state -= child_e->u.float_val;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFloatSumChildHasClosed(KHE_DRS_EXPR_FLOAT_SUM efs,        */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of opening efs for solving, efs's child_index'th child, child_e, */
/*  has been closed.  Update efs's closed state to reflect this.             */
/*                                                                           */
/*  The closed state of efs is the sum of the values of efs's closed         */
/*  children, so this update adds child_e's value (which preserves           */
/*  its value before opening) to the closed state.                           */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFloatSumChildHasClosed(KHE_DRS_EXPR_FLOAT_SUM efs,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  efs->closed_state += child_e->u.float_val;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFloatSumSetClosedValue(KHE_DRS_EXPR_FLOAT_SUM efs,        */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing efs, set its value suitably for its closed state.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFloatSumSetClosedValue(KHE_DRS_EXPR_FLOAT_SUM efs,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  efs->u.float_val = efs->closed_state;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprFloatSumDomTest(KHE_DRS_EXPR_FLOAT_SUM efs,   */
/*    int open_child_index)                                                  */
/*                                                                           */
/*  Return a dom test suited to efs on its open_child_index'th open day.     */
/*                                                                           */
/*****************************************************************************/

/* *** unused
static KHE_DRS_DOM_TEST KheDrsExprFloatSumDomTest(KHE_DRS_EXPR_FLOAT_SUM efs,
  int open_child_index)
{
  ** placeholder; sti ll to do **
  return KheDrsDomTestMake(false, INT_MAX, 0, false, 0);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFloatSumEvalDay(KHE_DRS_EXPR_FLOAT_SUM efs,               */
/*    KHE_DRS_SOLN prev_soln, KHE_DRS_SOLN next_soln,                        */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Evaluate expression efs on day open_day_index.                           */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsExprFloatSumEvalDay(KHE_DRS_EXPR_FLOAT_SUM efs,
  KHE_DRS_SOLN prev_soln, KHE_DRS_SOLN next_soln,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int i, i1, i2, di;  float val;  KHE_DRS_EXPR child_e;

  di = KheDrsSolnOpenDayIndex(next_soln);
  if( KheDrsOpenChildrenIndexIsFirst(&efs->open_children_by_day, di) )
  {
    ** first day, so we have a 0.0 value here **
    val = 0.0;
  }
  else
  {
    ** not first day, so retrieve a previous value **
    val = (float) KheDrsExprDaySigVal((KHE_DRS_EXPR) efs, di-1, prev_soln) / 100.0;
  }

  ** accumulate the open values of the children of efs that finalized today **
  KheDrsOpenChildrenForEach(&efs->open_children_by_day, di, child_e, i)
    val += child_e->u.float_val;

  if( KheDrsOpenChildrenIndexIsLast(&efs->open_children_by_day, di) )
  {
    ** last day; add closed_state to the value **
    efs->u.float_val = val + efs->closed_state;
  }
  else
  {
    ** this not efs's last day; store value in next_soln **
    KheDrsExprPutSigVal((KHE_DRS_EXPR) efs, di, next_soln, (int) (val * 100.0));
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFloatSumEvalSignature(KHE_DRS_EXPR_FLOAT_SUM efs,         */
/*    KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, int next_di,           */
/*    KHE_DRS_SIGNATURE sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)                */
/*                                                                           */
/*  Evaluate expression efs in sig.                                          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFloatSumEvalSignature(KHE_DRS_EXPR_FLOAT_SUM efs,
  KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, int next_di,
  KHE_DRS_SIGNATURE next_sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int i, i1, i2;  float val;  KHE_DRS_EXPR child_e;

  if( KheDrsOpenChildrenIndexIsFirst(&efs->open_children_by_day, next_di) )
  {
    /* first day, so we have a 0.0 value here */
    val = 0.0;
  }
  else
  {
    /* not first day, so retrieve a previous value */
    val = (float) KheDrsExprDaySigVal((KHE_DRS_EXPR) efs, next_di-1, prev_sig)
      / 100.0;
  }

  /* accumulate the open values of the children of efs that finalized today */
  KheDrsOpenChildrenForEach(&efs->open_children_by_day, next_di, child_e, i)
    val += child_e->u.float_val;

  if( KheDrsOpenChildrenIndexIsLast(&efs->open_children_by_day, next_di) )
  {
    /* last day; add closed_state to the value */
    efs->u.float_val = val + efs->closed_state;
  }
  else
  {
    /* this not efs's last day; store value in next_soln */
    KheDrsSignatureAddState(next_sig, (int) (val * 100.0), dsg,
      (KHE_DRS_EXPR) efs);
    /* KheDrsExprPutSigVal((KHE_DRS_EXPR) efs, next_di,sig,(int)(val*100.0)); */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFloatSumDoDebug(KHE_DRS_EXPR_FLOAT_SUM efs,               */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of efs which is specific to it.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFloatSumDoDebug(KHE_DRS_EXPR_FLOAT_SUM efs,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "FLOAT_SUM(%.2f)", efs->closed_state);
  KheDrsExprDebugSignature((KHE_DRS_EXPR) efs, soln, true, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_INT_DEV"                                         */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprIntDevDev(KHE_DRS_EXPR_INT_DEV eid, int val)               */
/*                                                                           */
/*  Deviation function to use with these expressions.                        */
/*                                                                           */
/*****************************************************************************/

static int KheDrsExprIntDevDev(KHE_DRS_EXPR_INT_DEV eid, int val)
{
  if( val == 0 && eid->allow_zero )
    return 0;
  else if( val > eid->max_limit )
    return val - eid->max_limit;
  else if( val < eid->min_limit )
    return eid->min_limit - val;
  else
    return 0;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_INT_DEV KheDrsExprIntDevMake(int min_limit, int max_limit,  */
/*    bool allow_zero, KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs) */
/*                                                                           */
/*  Make a new KHE_DRS_EXPR_INT_DEV object with these attributes.            */
/*                                                                           */
/*  There is no dom_test because its type is always KHE_DRS_DOM_UNUSED.      */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_INT_DEV KheDrsExprIntDevMake(int min_limit, int max_limit,
  bool allow_zero, KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_INT_DEV res;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_INT_DEV_TAG, dr, drs);
  res->u.int_val = UNDEF_INT;
  res->min_limit = min_limit;
  res->max_limit = max_limit;
  res->allow_zero = allow_zero;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntDevAddChild(KHE_DRS_EXPR_INT_DEV eid,                  */
/*    KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  New child child_e has just been added to eid.  Update eid's closed       */
/*  state appropriately.  This happens to be the same as closing a child.    */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntDevAddChild(KHE_DRS_EXPR_INT_DEV eid,
  KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  /* nothing to do here */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntDevChildHasOpened(KHE_DRS_EXPR_INT_DEV eid,            */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of opening eid for solving, eid's child_index'th child, child_e, */
/*  has been opened.  Update eid's closed state to reflect this.             */
/*                                                                           */
/*  The closed state of eid is the value of eid's single closed child.       */
/*  This is not stored explicitly, so this update does nothing.              */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntDevChildHasOpened(KHE_DRS_EXPR_INT_DEV eid,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  /* nothing to do here */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntDevChildHasClosed(KHE_DRS_EXPR_INT_DEV eid,            */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of opening eid for solving, eid's child_index'th child, child_e, */
/*  has been closed.  Update eid's closed state to reflect this.             */
/*                                                                           */
/*  The closed state of eid is the value of eid's single closed child.       */
/*  This is not stored explicitly, so this update does nothing.              */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntDevChildHasClosed(KHE_DRS_EXPR_INT_DEV eid,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  /* nothing to do here */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntDevSetClosedValue(KHE_DRS_EXPR_INT_DEV eid,            */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing eid, set its value suitably for its closed state.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntDevSetClosedValue(KHE_DRS_EXPR_INT_DEV eid,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR child_e;
  child_e = HaArrayFirst(eid->children);
  eid->u.int_val = KheDrsExprIntDevDev(eid, child_e->u.int_val);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprIntDevDomTest(KHE_DRS_EXPR_INT_DEV eid,       */
/*    int open_child_index)                                                  */
/*                                                                           */
/*  Return a dom test suited to eid on its open_child_index'th open day.     */
/*                                                                           */
/*****************************************************************************/

/* *** unused
static KHE_DRS_DOM_TEST KheDrsExprIntDevDomTest(KHE_DRS_EXPR_INT_DEV eid,
  int open_child_index)
{
  ** placeholder; sti ll to do **
  return KheDrsDomTestMake(false, INT_MAX, 0, false, 0);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntDevEvalDay(KHE_DRS_EXPR_INT_DEV eid,                   */
/*    KHE_DRS_SOLN prev_soln, KHE_DRS_SOLN next_soln,                        */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Evaluate expression eid within next_soln.                                */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsExprIntDevEvalDay(KHE_DRS_EXPR_INT_DEV eid,
  KHE_DRS_SOLN prev_soln, KHE_DRS_SOLN next_soln,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int i, i1, i2, val, di;  KHE_DRS_EXPR child_e;

  di = KheDrsSolnOpenDayIndex(next_soln);
  if( KheDrsOpenChildrenIndexIsFirst(&eid->open_children_by_day, di) )
  {
    ** first day **
    val = 0;
  }
  else
  {
    ** not first day, so retrieve a previous value **
    HnAbort("KheDrsExprIntDevEvalDay internal error (non-first day)");
  }

  ** accumulate the open values of the children of eid that finalized today **
  KheDrsOpenChildrenForEach(&eid->open_children_by_day, di, child_e, i)
    val += KheDrsExprIntDevDev(eid, child_e->u.int_val);

  if( KheDrsOpenChildrenIndexIsLast(&eid->open_children_by_day, di) )
  {
    ** last day; set value **
    eid->u.int_val = val;
  }
  else
  {
    ** not last day; store val in next_soln **
    HnAbort("KheDrsExprIntDevEvalDay internal error (non-last day)");
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntDevEvalSignature(KHE_DRS_EXPR_INT_DEV eid,             */
/*    KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, int next_di,           */
/*    KHE_DRS_SIGNATURE sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)                */
/*                                                                           */
/*  Evaluate expression eid within sig.                                      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntDevEvalSignature(KHE_DRS_EXPR_INT_DEV eid,
  KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, int next_di,
  KHE_DRS_SIGNATURE next_sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int i, i1, i2, val;  KHE_DRS_EXPR child_e;

  if( KheDrsOpenChildrenIndexIsFirst(&eid->open_children_by_day, next_di) )
  {
    /* first day */
    val = 0;
  }
  else
  {
    /* not first day, so retrieve a previous value */
    HnAbort("KheDrsExprIntDevEvalSignature internal error (non-first day)");
  }

  /* accumulate the open values of the children of eid that finalized today */
  KheDrsOpenChildrenForEach(&eid->open_children_by_day, next_di, child_e, i)
    val += KheDrsExprIntDevDev(eid, child_e->u.int_val);

  if( KheDrsOpenChildrenIndexIsLast(&eid->open_children_by_day, next_di) )
  {
    /* last day; set value */
    eid->u.int_val = val;
  }
  else
  {
    /* not last day; store val in next_soln */
    HnAbort("KheDrsExprIntDevEvalSignature internal error (non-last day)");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntDevDoDebug(KHE_DRS_EXPR_INT_DEV eid,                   */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of eid which is specific to it.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntDevDoDebug(KHE_DRS_EXPR_INT_DEV eid,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "INT_DEV(%d, %d, %s)", eid->min_limit, eid->max_limit,
    bool_show(eid->allow_zero));
  KheDrsExprDebugSignature((KHE_DRS_EXPR) eid, soln, false, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_FLOAT_DEV"                                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_FLOAT_DEV KheDrsExprFloatDevMake(int min_limit,             */
/*    int max_limit, bool allow_zero, KHE_DRS_RESOURCE dr,                   */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new KHE_DRS_EXPR_FLOAT_DEV object with these attributes.          */
/*                                                                           */
/*  There is no dom_test because its type is always KHE_DRS_DOM_UNUSED.      */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_FLOAT_DEV KheDrsExprFloatDevMake(int min_limit,
  int max_limit, bool allow_zero, KHE_DRS_RESOURCE dr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_FLOAT_DEV res;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_FLOAT_DEV_TAG, dr, drs);
  res->u.int_val = UNDEF_INT;
  res->min_limit = min_limit;
  res->max_limit = max_limit;
  res->allow_zero = allow_zero;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFloatDevAddChild(KHE_DRS_EXPR_FLOAT_DEV efd,              */
/*    KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  New child child_e has just been added to efd.  Update efd's closed       */
/*  state appropriately.  This happens to be the same as closing a child.    */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFloatDevAddChild(KHE_DRS_EXPR_FLOAT_DEV efd,
  KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  /* nothing to do here */
}


/*****************************************************************************/
/*                                                                           */
/*  int KheLimitWorkloadMonitorDev(KHE_DRS_EXPR_FLOAT_DEV efd,               */
/*    float workload)                                                        */
/*                                                                           */
/*  Return the deviation of workload wrt efd.                                */
/*                                                                           */
/*****************************************************************************/

static int KheLimitWorkloadMonitorDev(KHE_DRS_EXPR_FLOAT_DEV efd,
  float workload)
{
  if( efd->allow_zero && workload < 0.001 )
    return 0;
  else if( workload < efd->min_limit - 0.001 )
    return (int) ceil(efd->min_limit - 0.001 - workload);
  else if( workload > efd->max_limit + 0.001 )
    return (int) ceil(workload - efd->max_limit - 0.001);
  else
    return 0;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFloatDevChildHasOpened(KHE_DRS_EXPR_FLOAT_DEV efd,        */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of opening efd for solving, efd's child_index'th child, child_e, */
/*  has been opened.  Update efd's closed state to reflect this.             */
/*                                                                           */
/*  The closed state of efd is the value of efd's single closed child,       */
/*  so this update needs to do nothing.                                      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFloatDevChildHasOpened(KHE_DRS_EXPR_FLOAT_DEV efd,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  /* nothing to do here */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFloatDevChildHasClosed(KHE_DRS_EXPR_FLOAT_DEV efd,        */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of closing efd, efd's child_index'th child, child_e,             */
/*  has been closed.  Update efd's closed state to reflect this.             */
/*                                                                           */
/*  The closed state of efd is the value of efd's single closed child,       */
/*  so this update needs to do nothing.                                      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFloatDevChildHasClosed(KHE_DRS_EXPR_FLOAT_DEV efd,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  /* nothing to do here */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFloatDevSetClosedValue(KHE_DRS_EXPR_FLOAT_DEV efd,        */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing efd, set its value suitably for its closed state.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFloatDevSetClosedValue(KHE_DRS_EXPR_FLOAT_DEV efd,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR child_e;
  child_e = HaArrayFirst(efd->children);
  efd->u.int_val = KheLimitWorkloadMonitorDev(efd, child_e->u.float_val);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprFloatDevDomTest(KHE_DRS_EXPR_FLOAT_DEV efd,   */
/*    int open_child_index)                                                  */
/*                                                                           */
/*  Return a dom test suited to efd on its open_child_index'th open day.     */
/*                                                                           */
/*****************************************************************************/

/* *** unused
static KHE_DRS_DOM_TEST KheDrsExprFloatDevDomTest(KHE_DRS_EXPR_FLOAT_DEV efd,
  int open_child_index)
{
  ** placeholder; sti ll to do **
  return KheDrsDomTestMake(false, INT_MAX, 0, false, 0);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFloatDevEvalDay(KHE_DRS_EXPR_FLOAT_DEV efd,               */
/*    KHE_DRS_SOLN prev_soln, KHE_DRS_SOLN next_soln,                        */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Evaluate expression efd on day open_day_index.                           */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsExprFloatDevEvalDay(KHE_DRS_EXPR_FLOAT_DEV efd,
  KHE_DRS_SOLN prev_soln, KHE_DRS_SOLN next_soln,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int i, i1, i2, val, di;  KHE_DRS_EXPR child_e;

  di = KheDrsSolnOpenDayIndex(next_soln);
  if( KheDrsOpenChildrenIndexIsFirst(&efd->open_children_by_day, di) )
  {
    ** first day **
    val = 0;
  }
  else
  {
    ** not first day, so retrieve a previous value **
    HnAbort("KheDrsExprFloatDevEvalDay internal error (non-first day)");
    val = 0;  ** keep compiler happy **
  }

  ** accumulate the open values of the children of efd that finalized today **
  KheDrsOpenChildrenForEach(&efd->open_children_by_day, di, child_e, i)
    val += KheLimitWorkloadMonitorDev(efd, child_e->u.float_val);

  if( KheDrsOpenChildrenIndexIsLast(&efd->open_children_by_day, di) )
  {
    ** last day; set value **
    efd->u.int_val = val;
  }
  else
  {
    ** not last day; store value in next_soln **
    HnAbort("KheDrsExprFloatDevEvalDay internal error (non-last day)");
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFloatDevEvalSignature(KHE_DRS_EXPR_FLOAT_DEV efd,         */
/*    KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, int next_di,           */
/*    KHE_DRS_SIGNATURE sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)                */
/*                                                                           */
/*  Evaluate expression efd within sig.                                      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFloatDevEvalSignature(KHE_DRS_EXPR_FLOAT_DEV efd,
  KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, int next_di,
  KHE_DRS_SIGNATURE sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int i, i1, i2, val;  KHE_DRS_EXPR child_e;

  if( KheDrsOpenChildrenIndexIsFirst(&efd->open_children_by_day, next_di) )
  {
    /* first day */
    val = 0;
  }
  else
  {
    /* not first day, so retrieve a previous value */
    HnAbort("KheDrsExprFloatDevEvalSignature internal error (non-first day)");
    val = 0;  /* keep compiler happy */
  }

  /* accumulate the open values of the children of efd that finalized today */
  KheDrsOpenChildrenForEach(&efd->open_children_by_day, next_di, child_e, i)
    val += KheLimitWorkloadMonitorDev(efd, child_e->u.float_val);

  if( KheDrsOpenChildrenIndexIsLast(&efd->open_children_by_day, next_di) )
  {
    /* last day; set value */
    efd->u.int_val = val;
  }
  else
  {
    /* not last day; store value in next_soln */
    HnAbort("KheDrsExprFloatDevEvalSignature internal error (non-last day)");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprFloatDevDoDebug(KHE_DRS_EXPR_FLOAT_DEV efd,               */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of efd which is specific to it.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprFloatDevDoDebug(KHE_DRS_EXPR_FLOAT_DEV efd,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "FLOAT_DEV(%d, %d, %s)", efd->min_limit, efd->max_limit,
    bool_show(efd->allow_zero));
  KheDrsExprDebugSignature((KHE_DRS_EXPR) efd, soln, false, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_INT_SUM_COST - dom tables"                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_COST KheDrsExprIntSumCostExtraCost(KHE_DRS_EXPR_INT_SUM_COST eisc,   */
/*    int e, int l, int x, bool debug)                                       */
/*                                                                           */
/*  Return the extra cost in the dominance testing formula with these        */
/*  attributes.  This is function Gamma in the documentation.                */
/*                                                                           */
/*****************************************************************************/
static int KheDrsExprIntSumCostDelta(KHE_DRS_EXPR_INT_SUM_COST eisc,
  int lower_det, int upper_det);
static KHE_COST KheDrsExprCostCost(KHE_DRS_EXPR_COST eisc, int dev);
#define f(e, d) KheDrsExprCostCost((KHE_DRS_EXPR_COST) (e), (d))

static KHE_COST KheDrsExprIntSumCostExtraCost(KHE_DRS_EXPR_INT_SUM_COST eisc,
  int e, int l, int y, bool debug)
{
  int delta1, delta2;  KHE_COST res;
  delta1 = KheDrsExprIntSumCostDelta(eisc, l + y, l + y + eisc->history_after);
  delta2 = KheDrsExprIntSumCostDelta(eisc, l, l + e + eisc->history_after);
  res = f(eisc, delta1) - f(eisc, delta2);
  if( debug )
    fprintf(stderr, "    Gamma(e %d, l %d, y %d) = f(%d) %.5f - f(%d) "
      "%.5f = %.5f\n", e, l, y, delta1, KheCostShow(f(eisc, delta1)),
      delta2, KheCostShow(f(eisc, delta2)), KheCostShow(res));
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_COST KheDrsExprIntSumCostFindAvailCost(                              */
/*    KHE_DRS_EXPR_INT_SUM_COST eisc, int e, int l1, int l2)                 */
/*                                                                           */
/*  Find the available cost for eisc given e, l1, and l2.                    */
/*  This implements the outer "min" of the formula in the documentation.     */
/*  KheDrsExprIntSumCostExtraCost is Gamma.                                  */
/*                                                                           */
/*****************************************************************************/
#define DEBUG_INT_SUM_DOM(e, d1, d2)					\
  (DEBUG37 && (e) == 10 && (d1) == 10 && (d2) == 5)

static KHE_COST KheDrsExprIntSumCostFindAvailCost(
  KHE_DRS_EXPR_INT_SUM_COST eisc, int e, int l1, int l2)
{
  int y;  KHE_COST res, extra_cost1, extra_cost2, cost;  bool debug;
  debug = DEBUG_INT_SUM_DOM(e, l1, l2);
  if( debug )
    fprintf(stderr, "[ KheDrsExprIntSumCostFindAvailCost(%s, e %d, l1 %d,"
      " l2 %d)\n", KheDrsMonitorInfoId(eisc->monitor_info), e, l1, l2);
  res = KheCost(INT_MAX, INT_MAX);
  for( y = 0;  y <= e;  y++ )
  {
    extra_cost2 = KheDrsExprIntSumCostExtraCost(eisc, e, l2, y, debug);
    extra_cost1 = KheDrsExprIntSumCostExtraCost(eisc, e, l1, y, debug);
    cost = extra_cost2 - extra_cost1;
    if( debug )
      fprintf(stderr, "  y = %2d: %8.5f - %8.5f = %8.5f\n", y,
        KheCostShow(extra_cost2), KheCostShow(extra_cost1), KheCostShow(cost));
    if( cost < res )
      res = cost;
  }
  HnAssert(res <= 0, "KheDrsExprIntSumCostFindAvailCost: res > 0");
  if( debug )
    fprintf(stderr, "] KheDrsExprIntSumCostFindAvailCost returning %.5f\n",
      KheCostShow(res));
  return res;
}

/*****************************************************************************/
/*                                                                           */
/*  KHE_COST KheDrsExprIntSumCostFindCorrAvailCost(                          */
/*    KHE_DRS_EXPR_INT_SUM_COST eisc, int e, int a1, int a2, int l1, int l2) */
/*                                                                           */
/*  Find the correlated available cost for eisc given e, a1, a2, l1, and l2. */
/*                                                                           */
/*****************************************************************************/
#define DEBUG_INT_SUM_CORR(e, a1, a2, l1, l2)				\
  (DEBUG56 && (e) == 1 && (a1) == 0 && (a2) == 1 && (l1) == 0 && (l2) == 2)

static KHE_COST KheDrsExprIntSumCostFindCorrAvailCost(
  KHE_DRS_EXPR_INT_SUM_COST eisc, int e, int a1, int a2, int l1, int l2)
{
  KHE_COST res, extra_cost1, extra_cost2, cost;
  int y1, y2, D_first, D_last, i;  bool debug;
  debug = DEBUG_INT_SUM_CORR(e, a1, a2, l1, l2);
  if( debug )
    fprintf(stderr, "[ KheDrsExprIntSumCostFindCorrAvailCost(%s, e %d, a1 %d,"
      " a2 %d, l1 %d, l2 %d)\n", KheDrsMonitorInfoId(eisc->monitor_info),
      e, a1, a2, l1, l2);

  /* work out D, in the form D_first .. D_last */
  if( a1 == 0 )
  {
    if( a2 == 0 )
      D_first = D_last = 0;
    else
      D_first = -1, D_last = 0;
  }
  else
  {
    if( a2 == 0 )
      D_first = 0, D_last = 1;
    else
      D_first = D_last = 0;
  }

  /* do the rest */
  res = KheCost(INT_MAX, INT_MAX);
  for( y1 = a1;  y1 <= e;  y1++ )
    for( i = D_first;  i <= D_last;  i++ )
    {
      y2 = y1 - i;
      if( a2 <= y2 && y2 <= e )
      {
	extra_cost2 = KheDrsExprIntSumCostExtraCost(eisc, e, l2, y2, debug);
	extra_cost1 = KheDrsExprIntSumCostExtraCost(eisc, e, l1, y1, debug);
	cost = extra_cost2 - extra_cost1;
	if( debug )
	  fprintf(stderr, "  y1 = %2d, y2 = %2d: %8.5f - %8.5f = %8.5f\n", y1,
	    y2, KheCostShow(extra_cost2), KheCostShow(extra_cost1),
	    KheCostShow(cost));
	if( cost < res )
	  res = cost;
      }
    }
  /* *** there are cases where the result is genuinely positive
  HnAssert(res <= 0, "KheDrsExprIntSumCostFindCorrAvailCost: res > 0");
  *** */
  if( debug )
    fprintf(stderr, "] KheDrsExprIntSumCostFindCorrAvailCost returning %.5f\n",
      KheCostShow(res));
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsExprIsSingleDay(KHE_DRS_EXPR e, KHE_FRAME days_frame,         */
/*    KHE_TIME_GROUP *time_group)                                            */
/*                                                                           */
/*  If e covers a single day of days_frame, return true and set *time_group  */
/*  to the time group of that day in days_frame.  Otherwise return false     */
/*  with *time_group set to NULL.                                            */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsExprIsSingleDay(KHE_DRS_EXPR e, KHE_FRAME days_frame,
  KHE_TIME_GROUP *time_group)
{
  KHE_TIME_GROUP tg;  KHE_DRS_EXPR child_e;  int i;
  switch( e->tag )
  {
    case KHE_DRS_EXPR_ASSIGNED_TASK_TAG:

      /* shortcut; we aren't expecting this case to ever occur */
      return *time_group = NULL, false;

    case KHE_DRS_EXPR_BUSY_TIME_TAG:

      { KHE_DRS_EXPR_BUSY_TIME ebt;
	ebt = (KHE_DRS_EXPR_BUSY_TIME) e;
	*time_group = KheFrameTimeTimeGroup(days_frame, ebt->time);
	return true;
      }

    case KHE_DRS_EXPR_FREE_TIME_TAG:

      { KHE_DRS_EXPR_FREE_TIME eft;
	eft = (KHE_DRS_EXPR_FREE_TIME) e;
	*time_group = KheFrameTimeTimeGroup(days_frame, eft->time);
	return true;
      }

    case KHE_DRS_EXPR_WORK_TIME_TAG:

      { KHE_DRS_EXPR_WORK_TIME ewt;
	ewt = (KHE_DRS_EXPR_WORK_TIME) e;
	*time_group = KheFrameTimeTimeGroup(days_frame, ewt->time);
	return true;
      }

    case KHE_DRS_EXPR_BUSY_DAY_TAG:
      
      { KHE_DRS_EXPR_BUSY_DAY ebd;
	ebd = (KHE_DRS_EXPR_BUSY_DAY) e;
        *time_group = ebd->time_group;
	return true;
      }

    case KHE_DRS_EXPR_FREE_DAY_TAG:

      { KHE_DRS_EXPR_FREE_DAY efd;
	efd = (KHE_DRS_EXPR_FREE_DAY) e;
        *time_group = efd->time_group;
	return true;
      }

    case KHE_DRS_EXPR_WORK_DAY_TAG:

      { KHE_DRS_EXPR_WORK_DAY ewd;
	ewd = (KHE_DRS_EXPR_WORK_DAY) e;
        *time_group = ewd->time_group;
	return true;
      }

    case KHE_DRS_EXPR_OR_TAG:
    case KHE_DRS_EXPR_AND_TAG:	

      *time_group = NULL;
      HaArrayForEach(e->children, child_e, i)
      {
	if( !KheDrsExprIsSingleDay(child_e, days_frame, &tg) )
	  return false;
	else if( *time_group == NULL )
	  *time_group = tg;
	else if( !KheTimeGroupEqual(*time_group, tg) )
	  return false;
      }
      return (*time_group != NULL);
      
    case KHE_DRS_EXPR_INT_SUM_TAG:
    case KHE_DRS_EXPR_FLOAT_SUM_TAG:
    case KHE_DRS_EXPR_INT_DEV_TAG:
    case KHE_DRS_EXPR_FLOAT_DEV_TAG:
    case KHE_DRS_EXPR_INT_SUM_COST_TAG:
    case KHE_DRS_EXPR_INT_SEQ_COST_TAG:

      /* shortcut; we aren't expecting these cases to ever occur */
      return *time_group = NULL, false;

    default:

      HnAbort("KheDrsExprMightSpanMoreThanOneDay internal error (tag %d)",
	e->tag);
      return *time_group = NULL, false;  /* keep compiler happy */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsExprIntSumCostNeedsCorrTable(KHE_DRS_EXPR_INT_SUM_COST eisc,  */
/*    KHE_FRAME days_frame)                                                  */
/*                                                                           */
/*  Return true if eisc needs a corr table - if any of its children span     */
/*  more than one day of days_frame.                                         */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsExprIntSumCostNeedsCorrTable(KHE_DRS_EXPR_INT_SUM_COST eisc,
  KHE_FRAME days_frame)
{
  KHE_DRS_EXPR child_e;  KHE_TIME_GROUP tg;  int i;
  HaArrayForEach(eisc->children, child_e, i)
    if( !KheDrsExprIsSingleDay(child_e, days_frame, &tg) )
      return true;
  return false;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSumCostSetDomTables(KHE_DRS_EXPR_INT_SUM_COST eisc,    */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Set the main_dom_table3 and corr_dom_table5 fields of eisc.              */
/*                                                                           */
/*****************************************************************************/
/* ***
static KHE_DRS_DOM_TABLE_SUM KheDrsDomTableSumMake(HA_ARENA a);
static void KheDrsDomTableSumPut(KHE_DRS_DOM_TABLE_SUM dt, int offset,
  int e, int l1, int l2, KHE_COST cost, HA_ARENA a);
*** */
static void KheDrsExprIntSumCostDebugDomTables(KHE_DRS_EXPR_INT_SUM_COST eisc,
  int verbosity, int indent, FILE *fp);

static void KheDrsExprIntSumCostSetDomTables(KHE_DRS_EXPR_INT_SUM_COST eisc,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int e, l1, l2, a1, a2, child_count, lim;  KHE_COST cost;
  KHE_DRS_COST_TUPLE ct;  char *debug_str;

  /* build eisc->dom_table */
  HnAssert(eisc->main_dom_table3 == NULL,
    "KheDrsExprIntSumCostSetDomTable: main_dom_table");
  child_count = HaArrayCount(eisc->children);
  debug_str = KheDrsMonitorInfoId(eisc->monitor_info);
  eisc->main_dom_table3 = KheDrsDim3TableMake(debug_str, drs->arena);
  /* eisc->dom_table = KheDrsDomTableSumMake(a); */
  for( e = 0;  e <= child_count;  e++ )
    for( l1 = eisc->history;  l1 <= child_count + eisc->history - e;  l1++ )
      for( l2 = eisc->history;  l2 <= child_count + eisc->history - e;  l2++ )
      {
	cost = KheDrsExprIntSumCostFindAvailCost(eisc, e, l1, l2);
	ct = KheDrsCostTupleMake(cost, 0, false, 0);
	KheDrsDim3TablePut(eisc->main_dom_table3, e, l1, l2, ct, drs);
	/* ***
	KheDrsDomTableSumPut(eisc->dom_table, eisc->history, e, l1, l2, cost,a);
	*** */
      }

  /* build eisc->corr_dom_table */
  if( KheDrsExprIntSumCostNeedsCorrTable(eisc, drs->days_frame) )
  {
    eisc->corr_dom_table5 = KheDrsDim5TableMake(
      HnStringMake(drs->arena, "Corr %s", debug_str), drs->arena);
    for( e = 1;  e <= child_count;  e++ )
    {
      lim = child_count + eisc->history - e;
      for( a1 = 0;  a1 <= 1;  a1++ )
	for( a2 = 0;  a2 <= 1;  a2++ )
	  for( l1 = eisc->history;  l1 <= lim;  l1++ )
	    for( l2 = eisc->history;  l2 <= lim;  l2++ )
	    {
	      cost = KheDrsExprIntSumCostFindCorrAvailCost(eisc, e,a1,a2,l1,l2);
	      ct = KheDrsCostTupleMake(cost, 0, false, 0);
	      KheDrsDim5TablePut(eisc->corr_dom_table5, e, a1, a2, l1, l2, ct,
		drs);
	    }
    }
  }
  if( DEBUG36 )
    KheDrsExprIntSumCostDebugDomTables(eisc, 1, 0, stderr);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsExprIntSumCostWantDebugDomTable(                              */
/*    KHE_DRS_EXPR_INT_SUM_COST eisc)                                        */
/*                                                                           */
/*  Return true if we want to debug eisc's dom table.                        */
/*                                                                           */
/*  At present we are doing this is ec's monitor is a resource monitor       */
/*  and the resource it monitors is the first resource of its type.          */
/*  This is to limit the debug output to a manageable amount.                */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsExprIntSumCostWantDebugDomTable(
  KHE_DRS_EXPR_INT_SUM_COST eisc)
{
  KHE_MONITOR m;  KHE_RESOURCE r;  KHE_RESOURCE_TYPE rt;

  /* find the resource eisc monitors; return false if not resource monitor */
  m = eisc->monitor_info->monitor;
  switch( KheMonitorTag(m) )
  {
    case KHE_AVOID_CLASHES_MONITOR_TAG:

      r = KheAvoidClashesMonitorResource((KHE_AVOID_CLASHES_MONITOR) m);
      break;

    case KHE_AVOID_UNAVAILABLE_TIMES_MONITOR_TAG:

      r = KheAvoidUnavailableTimesMonitorResource(
	(KHE_AVOID_UNAVAILABLE_TIMES_MONITOR) m);
      break;

    case KHE_LIMIT_IDLE_TIMES_MONITOR_TAG:

      r = KheLimitIdleTimesMonitorResource((KHE_LIMIT_IDLE_TIMES_MONITOR) m);
      break;

    case KHE_CLUSTER_BUSY_TIMES_MONITOR_TAG:

      r = KheClusterBusyTimesMonitorResource(
	(KHE_CLUSTER_BUSY_TIMES_MONITOR) m);
      break;

    case KHE_LIMIT_BUSY_TIMES_MONITOR_TAG:

      r = KheLimitBusyTimesMonitorResource((KHE_LIMIT_BUSY_TIMES_MONITOR) m);
      break;

    case KHE_LIMIT_WORKLOAD_MONITOR_TAG:

      r = KheLimitWorkloadMonitorResource((KHE_LIMIT_WORKLOAD_MONITOR) m);
      break;

    /* ***
    case KHE_LIMIT_ACTIVE_INTERVALS_MONITOR_TAG:

      r = KheLimitActiveIntervalsMonitorResource(
	(KHE_LIMIT_ACTIVE_INTERVALS_MONITOR) m);
      break;
    *** */

    default:

      return false;
  }

  /* return true if r is the first resource of its type */
  rt = KheResourceResourceType(r);
  return KheResourceTypeResourceCount(rt) > 0 &&
    KheResourceTypeResource(rt, 0) == r;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSumCostDebugDomTables(KHE_DRS_EXPR_INT_SUM_COST eisc,  */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of eisc's dom table.                                         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntSumCostDebugDomTables(KHE_DRS_EXPR_INT_SUM_COST eisc,
  int verbosity, int indent, FILE *fp)
{
  if( KheDrsExprIntSumCostWantDebugDomTable(eisc) )
  {
    fprintf(fp, "%*s[ ExprCost for %s\n", indent, "",
      KheDrsMonitorInfoId(eisc->monitor_info));
    fprintf(fp, "%*s  cost_fn %s, combined_weight %.5f, min_limit %d,"
      " max_limit %d, allow_zero %s\n", indent, "",
      KheCostFunctionShow(eisc->cost_fn), KheCostShow(eisc->combined_weight),
      eisc->min_limit, eisc->max_limit, bool_show(eisc->allow_zero));
    fprintf(fp, "%*s  history_before %d, history_after %d, history %d\n",
      indent, "", eisc->history_before, eisc->history_after, eisc->history);
    KheDrsDim3TableDebug(eisc->main_dom_table3, verbosity, indent + 2, fp);
    KheDrsDim5TableDebug(eisc->corr_dom_table5, verbosity, indent + 2, fp);
    /* KheDrsDomTableSumDebug(eisc->dom_table, verbosity, indent + 2, fp); */
    fprintf(fp, "%*s]\n", indent, "");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_COST"                                            */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_COST KheDrsExprCostCost(KHE_DRS_EXPR_COST ec, int dev)               */
/*                                                                           */
/*  Return f(dev) for ec.                                                    */
/*                                                                           */
/*****************************************************************************/

static KHE_COST KheDrsExprCostCost(KHE_DRS_EXPR_COST ec, int dev)
{
  switch( ec->cost_fn )
  {
    case KHE_STEP_COST_FUNCTION:

      return dev > 0 ? ec->combined_weight : 0;

    case KHE_LINEAR_COST_FUNCTION:

      return dev * ec->combined_weight;

    case KHE_QUADRATIC_COST_FUNCTION:

      return dev * dev * ec->combined_weight;

    default:

      HnAbort("KheDrsExprCostCost internal error");
      return KheCost(0, 0);  /* keep compiler happy */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_INT_SUM_COST"                                    */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprIntSumCostDelta(KHE_DRS_EXPR_INT_SUM_COST eisc,            */
/*    int lower_det, int upper_det)                                          */
/*                                                                           */
/*  Return the deviation of eisc for the given lower and upper determinants. */
/*  This is function delta(l, u) from the documentation.                     */
/*                                                                           */
/*****************************************************************************/

static int KheDrsExprIntSumCostDelta(KHE_DRS_EXPR_INT_SUM_COST eisc,
  int lower_det, int upper_det)
{
  if( eisc->allow_zero && lower_det == 0 )
    return 0;
  else if( lower_det > eisc->max_limit )
    return lower_det - eisc->max_limit;
  else if( upper_det < eisc->min_limit )
    return eisc->min_limit - upper_det;
  else
    return 0;
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprIntSumCostDev(KHE_DRS_EXPR_INT_SUM_COST eisc,              */
/*    int active_children, int unassigned_children)                          */
/*                                                                           */
/*  Return the deviation of eisc, assuming that eisc has the given number    */
/*  of active children, and the given number of unassigned children.  This   */
/*  is closely related to delta but more convenient at times.                */
/*                                                                           */
/*****************************************************************************/

static int KheDrsExprIntSumCostDev(KHE_DRS_EXPR_INT_SUM_COST eisc,
  int active_children, int unassigned_children)
{
  return KheDrsExprIntSumCostDelta(eisc, active_children,
    active_children + unassigned_children);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_ADJUST_TYPE KheDrsAdjustType(KHE_COST_FUNCTION cost_fn,          */
/*    int max_limit)                                                         */
/*                                                                           */
/*  Return the adjust type, determined by the cost function and max limit.   */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_ADJUST_TYPE KheDrsAdjustType(KHE_COST_FUNCTION cost_fn,
  int max_limit)
{
  if( max_limit == INT_MAX )
    return KHE_DRS_ADJUST_NO_MAX;
  else if( cost_fn == KHE_LINEAR_COST_FUNCTION )
    return KHE_DRS_ADJUST_LINEAR;
  else if( cost_fn == KHE_STEP_COST_FUNCTION )
    return KHE_DRS_ADJUST_STEP;
  else
    return KHE_DRS_ADJUST_ORDINARY;
}


/*****************************************************************************/
/*                                                                           */
/*  char *KheDrsAdjustTypeShow(KHE_DRS_ADJUST_TYPE dct)                      */
/*                                                                           */
/*  Return a string representation of dct.                                   */
/*                                                                           */
/*****************************************************************************/

static char *KheDrsAdjustTypeShow(KHE_DRS_ADJUST_TYPE dct)
{
  switch( dct )
  {
    case KHE_DRS_ADJUST_NO_MAX:	  return "NO_MAX";
    case KHE_DRS_ADJUST_LINEAR:	  return "LINEAR";
    case KHE_DRS_ADJUST_STEP:	  return "STEP";
    case KHE_DRS_ADJUST_ORDINARY: return "ORDINARY";
    default:			  return "??";
  }
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_INT_SUM_COST KheDrsExprIntSumCostMake(                      */
/*    KHE_DRS_MONITOR_INFO mi, KHE_COST_FUNCTION cost_fn,                    */
/*    KHE_COST combined_weight, int min_limit, int max_limit,                */
/*    bool allow_zero, int history_before, int history_after,                */
/*    int history, KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs)     */
/*                                                                           */
/*  Make a new KHE_DRS_EXPR_INT_SUM_COST object with these attributes.       */
/*                                                                           */
/*****************************************************************************/
static void KheDrsAddWeight(KHE_DYNAMIC_RESOURCE_SOLVER drs, KHE_COST weight);

static KHE_DRS_EXPR_INT_SUM_COST KheDrsExprIntSumCostMake(
  KHE_DRS_MONITOR_INFO mi, KHE_COST_FUNCTION cost_fn,
  KHE_COST combined_weight, int min_limit, int max_limit,
  bool allow_zero, int history_before, int history_after,
  int history, KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  /* build the object */
  KHE_DRS_EXPR_INT_SUM_COST res;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_INT_SUM_COST_TAG,dr,drs);
  res->u.int_val = UNDEF_INT;
  res->monitor_info = mi;
  res->cost_fn = cost_fn;
  res->combined_weight = combined_weight;
  res->min_limit = min_limit;
  res->max_limit = max_limit;
  res->allow_zero = allow_zero;
  res->adjust_type = KheDrsAdjustType(cost_fn, max_limit);
  res->history_before = history_before;
  res->history_after = history_after;
  res->history = history;
  res->main_dom_table3 = NULL;  /* reset later */
  res->corr_dom_table5 = NULL;  /* possibly reset later */
  res->closed_state = 0;
  KheDrsOpenChildrenInit(&res->open_children_by_shift, drs);
  /* HaArrayInit(res->open_children_by_shift, drs->arena); */
  KheDrsAddWeight(drs, combined_weight);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_INT_SUM_COST KheDrsExprIntSumCostZeroMaxMake(               */
/*    KHE_DRS_MONITOR_INFO mi, KHE_COST_FUNCTION cost_fn,                    */
/*    KHE_COST combined_weight, KHE_DRS_RESOURCE dr,                         */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make an INT_SUM_COST object for the case where there is a maximum limit  */
/*  of 0 (so the dominance test is <=) and no history.  This is the usual    */
/*  (but not the only) replacement for the now omitted EXPR_COST.            */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_INT_SUM_COST KheDrsExprIntSumCostZeroMaxMake(
  KHE_DRS_MONITOR_INFO mi, KHE_COST_FUNCTION cost_fn,
  KHE_COST combined_weight, KHE_DRS_RESOURCE dr,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  return KheDrsExprIntSumCostMake(mi, cost_fn, combined_weight, 0, 0, false,
    0, 0, 0, dr, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSumCostAddChild(KHE_DRS_EXPR_INT_SUM_COST eisc,        */
/*    KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  New child child_e has just been added to eisc.  Update eisc's state      */
/*  appropriately.  This is similar to closing a child, but not identical.   */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntSumCostAddChild(KHE_DRS_EXPR_INT_SUM_COST eisc,
  KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  eisc->closed_state += child_e->u.int_val;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsIntSumCostAdjustLimits(KHE_DRS_EXPR_INT_SUM_COST eisc)        */
/*                                                                           */
/*  Adjust the limits of eisc.  To be called after all children are added.   */
/*                                                                           */
/*****************************************************************************/

static void KheDrsIntSumCostAdjustLimits(KHE_DRS_EXPR_INT_SUM_COST eisc)
{
  int max_max_limit;

  /* adjust the lower limit if required */
  if( eisc->min_limit < eisc->history )
    eisc->min_limit = eisc->history;

  /* adjust eisc->max_limit if required */
  max_max_limit = HaArrayCount(eisc->children) + eisc->history +
    eisc->history_after;
  if( eisc->max_limit > max_max_limit )
    eisc->max_limit = max_max_limit;

  /* make sure limits are consistent */
  HnAssert(eisc->min_limit <= eisc->max_limit, "KheDrsIntSumCostAdjustLimits: "
    "after adjustment, inconsistent limits (min %d, max %d) for monitor %s",
    eisc->min_limit, eisc->max_limit, KheDrsMonitorInfoId(eisc->monitor_info));
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsUpdateRerunCost(KHE_DYNAMIC_RESOURCE_SOLVER drs,              */
/*    KHE_COST *rerun_cost1, KHE_COST *rerun_cost2, KHE_MONITOR m,           */
/*    char *operation, int child_index, char *fmt, ...)                      */
/*                                                                           */
/*  Update *rerun_cost1, and *rerun_cost2 if rerun_cost2 != NULL.  Also      */
/*  produce a debug print about it, if m is RERUN_MONITOR_ID.                */
/*                                                                           */
/*****************************************************************************/

/* *** obsolete
static void KheDrsUpdateRerunCost(KHE_DYNAMIC_RESOURCE_SOLVER drs,
  KHE_COST *rerun_cost1, KHE_COST *rerun_cost2, KHE_MONITOR m,
  char *operation, int child_index, char *fmt, ...)
{
  if( RERUN && drs->rerun != NULL )
  {
    va_list ap;  char buff[10];  int count;  KHE_COST cost;  bool print;
    print = (strcmp(KheMonitorId(m), RERUN_MONITOR_ID) == 0);
    if( print )
    {
      if( child_index >= 0 )
	snprintf(buff, 9, "%s %d", operation, child_index);
      else
	snprintf(buff, 9, "%s", operation);
      fprintf(stderr, "  %9s: %.5f", buff, KheCostShow(*rerun_cost1));
    }
    va_start(ap, fmt);
    for( count = 0;  fmt[count] != '\0';  count++ )
    {
      cost = va_arg(ap, KHE_COST);
      if( print )
	fprintf(stderr, " %c %.5f", fmt[count], KheCostShow(cost));
      if( fmt[count] == '+' )
	*rerun_cost1 += cost;
      else
	*rerun_cost1 -= cost;
      if( rerun_cost2 != NULL )
      {
	if( fmt[count] == '+' )
	  *rerun_cost2 += cost;
	else
	  *rerun_cost2 -= cost;
      }
    }
    va_end(ap);
    if( print )
    {
      while( count < 4 )
      {
	fprintf(stderr, "          ");
	count++;
      }
      fprintf(stderr, "  (%s)\n", KheMonitorId(m));
    }
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSumCostChildHasOpened(KHE_DRS_EXPR_INT_SUM_COST eisc,  */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of opening eisc for solving, eisc's child_index'th child,        */
/*  child_e, has been opened.  Update eisc to reflect this, and also         */
/*  drs->solve_start_cost.                                                   */
/*                                                                           */
/*  This function is called after child_e is added to eisc->open_children.   */
/*  The closed state of eisc is the sum of the values of eisc's closed       */
/*  children, so this update subtracts child_e's value (which preserves      */
/*  its value before opening) from the closed state.                         */
/*                                                                           */
/*  Obsolete:                                                                */
/*  So when KheDrsExprIntSumCostUpdateStartCost retrieves the number of      */
/*  open children, it gets the correct (that is, the changed) value.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntSumCostChildHasOpened(KHE_DRS_EXPR_INT_SUM_COST eisc,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int old_dev, new_dev;  KHE_COST old_cost, new_cost;
  if( DEBUG58(eisc->monitor_info) )
  {
    fprintf(stderr, "[ KheDrsExprIntSumCostChildHasOpened(%s, child_e, %d):\n",
      KheDrsMonitorInfoId(eisc->monitor_info), child_index);
    KheDrsExprDebug(child_e, drs, 2, 2, stderr);
    /* ***
    HaArrayForEach(eisc->children, e, i)
      KheDrsExprDebug(e, drs, 2, 4, stderr);
    *** */
  }
  old_dev = KheDrsExprIntSumCostDev(eisc, eisc->history + eisc->closed_state,
    KheDrsOpenChildrenCount(&eisc->open_children_by_day) - 1 +
    eisc->history_after);
  eisc->closed_state -= child_e->u.int_val;
  new_dev = KheDrsExprIntSumCostDev(eisc, eisc->history + eisc->closed_state,
     KheDrsOpenChildrenCount(&eisc->open_children_by_day) +eisc->history_after);
  if( old_dev != new_dev )
  {
    old_cost = f(eisc, old_dev);
    new_cost = f(eisc, new_dev);
    drs->solve_start_cost += new_cost - old_cost;
    KheDrsMonitorInfoUpdateRerunCost(eisc->monitor_info, drs, NULL,
      KHE_DRS_OPEN, "open", child_index, "+-", new_cost, old_cost);
  }

  /* update open_children_by_shift, if required */
  if( eisc->resource == NULL )
    KheDrsOpenChildrenAddChildInShiftOrder(&eisc->open_children_by_shift,
      child_e);

  /* all done */
  if( DEBUG58(eisc->monitor_info) )
    fprintf(stderr, "] KheDrsExprIntSumCostChildHasOpened\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSumCostChildHasClosed(KHE_DRS_EXPR_INT_SUM_COST eisc,  */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  As part of closing eisc, eisc's child_index'th child, child_e, has       */
/*  been closed.  Update eisc's closed state to reflect this, and also       */
/*  drs->solve_start_cost.                                                   */
/*                                                                           */
/*  Although KheDrsExprChildHasOpened adds the newly opened child to         */
/*  e->open_children, KheDrsExprChildHasClosed does not remove it.  This     */
/*  is because the removal cannot be done efficiently while preserving       */
/*  the order of the open children.  However that does not matter here,      */
/*  and we need the correct number of open children, so we remove any one.   */
/*                                                                           */
/*  The closed state of eisc is the sum of the values of eisc's closed       */
/*  children, so this update adds child_e's value to the closed state.       */
/*                                                                           */
/*  Obsolete:                                                                */
/*  This is called after child_e is removed from eisc->open_children.        */
/*  So when KheDrsExprIntSumCostUpdateStartCost retrieves the number of      */
/*  open children, it gets the correct (that is, the changed) value.         */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntSumCostChildHasClosed(KHE_DRS_EXPR_INT_SUM_COST eisc,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int old_dev, new_dev;  KHE_COST old_cost, new_cost;
  old_dev = KheDrsExprIntSumCostDev(eisc, eisc->history + eisc->closed_state,
    KheDrsOpenChildrenCount(&eisc->open_children_by_day) + eisc->history_after);
  KheDrsOpenChildrenDeleteLast(&eisc->open_children_by_day);
  /* HaArrayDeleteLast(eisc->open_children_by_day.children); */
  eisc->closed_state += child_e->u.int_val;
  new_dev = KheDrsExprIntSumCostDev(eisc, eisc->history + eisc->closed_state,
    KheDrsOpenChildrenCount(&eisc->open_children_by_day) + eisc->history_after);
  if( old_dev != new_dev )
  {
    new_cost = f(eisc, new_dev);
    old_cost = f(eisc, old_dev);
    drs->solve_start_cost += new_cost - old_cost;
    KheDrsMonitorInfoUpdateRerunCost(eisc->monitor_info, drs, NULL,
      KHE_DRS_CLOSE, "close", child_index, "+-", new_cost, old_cost);
  }
  if( eisc->resource == NULL )
    KheDrsOpenChildrenDeleteLast(&eisc->open_children_by_shift);
    /* HaArrayDeleteLast(eisc->open_children_by_shift.children); */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSumCostSetClosedValue(KHE_DRS_EXPR_INT_SUM_COST eisc,  */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing eisc, set its value suitably for its closed state.    */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntSumCostSetClosedValue(KHE_DRS_EXPR_INT_SUM_COST eisc,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  /* nothing to do here, since eisc does not store a closed value */
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprIntSumCostChildDomTest(                       */
/*    KHE_DRS_EXPR_INT_SUM_COST eisc, KHE_DYNAMIC_RESOURCE_SOLVER drs)       */
/*                                                                           */
/*  Return the dom test for a child of eisc on its open_day_index'th         */
/*  open day.                                                                */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DOM_TEST KheDrsExprIntSumCostChildDomTest(
  KHE_DRS_EXPR_INT_SUM_COST eisc, int open_day_index, KHE_DRS_EXPR child_e,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int a, b, unassigned_child_count;  KHE_DRS_DIM4_TABLE corr_dom_table4;

  /* calculate a */
  a = (eisc->max_limit < INT_MAX ? -1 : INT_MAX);

  /* calculate b */
  b = (eisc->min_limit > 0 ? INT_MAX : 0);

  /* make and return the dom test */
  if( child_e->tag == KHE_DRS_EXPR_OR_TAG )
  {
    unassigned_child_count = KheDrsOpenChildrenAtOrAfter(
      &eisc->open_children_by_day, open_day_index + 1);
    if( DEBUG57 )
    {
      fprintf(stderr, "[ calling KheDrsDim5TableGet(%p, %d), open_day_index "
	"%d, eisc:\n", (void *) eisc->corr_dom_table5, unassigned_child_count,
	open_day_index);
      KheDrsExprDebug((KHE_DRS_EXPR) eisc, drs, 2, 2, stderr);
      KheDrsOpenChildrenDebug(&eisc->open_children_by_day, 2, 2, stderr);
      fprintf(stderr, "]\n");
    }
    corr_dom_table4 = KheDrsDim5TableGet(eisc->corr_dom_table5,
      unassigned_child_count);
  }
  else
    corr_dom_table4 = NULL;
  return KheDrsDomTestMake(KHE_DRS_DOM_TEST_STRONG, child_e, false, 0,
    INT_MAX, a, b, 0, NULL, corr_dom_table4, NULL, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsIntSumCostExprRequiresShiftDomTest(                           */
/*    KHE_DRS_EXPR_INT_SUM_COST eisc, int si,                                */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, int *val)                             */
/*                                                                           */
/*  Find the number of unassigned children of eisc there will be when the    */
/*  days before ds (the shift with open shift index si) begins are all       */
/*  assigned, and ds itself is assigned.  Return this number in *val, and    */
/*  return true if *val > 0.  This will mean that shift assignments for ds   */
/*  need a dom test, because assigning ds does not completely assign eisc.   */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsIntSumCostExprRequiresShiftDomTest(
  KHE_DRS_EXPR_INT_SUM_COST eisc, int si,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, int *val)
{
  int di, assigned;  KHE_DRS_SHIFT ds;
  ds = HaArray(drs->open_shifts, si);
  di = ds->encl_day->open_day_index;
  assigned = KheDrsOpenChildrenBefore(&eisc->open_children_by_day, di) +
    KheDrsOpenChildrenWithIndex(&eisc->open_children_by_shift, si);
  *val = KheDrsOpenChildrenCount(&eisc->open_children_by_day) - assigned;
  return *val > 0;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST_TYPE KheDomTestTypeActual(KHE_DRS_DOM_TEST_TYPE dt,     */
/*    bool tradeoff_allowed)                                                 */
/*                                                                           */
/*  Return the dom test type to actually use, given than the requested       */
/*  type is dt and tradoff_allowed says whether tradeoff dominance is        */
/*  allowed.                                                                 */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DOM_TEST_TYPE KheDomTestTypeActual(KHE_DRS_DOM_TEST_TYPE dt,
  bool tradeoff_allowed)
{
  if( dt == KHE_DRS_DOM_TEST_TRADEOFF && !tradeoff_allowed )
    return KHE_DRS_DOM_TEST_STRONG;
  else
    return dt;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprIntSumCostGeneralDomTest(                     */
/*    KHE_DRS_EXPR_INT_SUM_COST eisc, int unassigned_child_count,            */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make and return a dominance test for eisc, assuming that there are       */
/*  unassigned_child_count unassigned children (not including history).      */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DOM_TEST KheDrsExprIntSumCostGeneralDomTest(
  KHE_DRS_EXPR_INT_SUM_COST eisc, int unassigned_child_count,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int a, b;  KHE_DRS_DOM_TEST_TYPE dt_type;
  KHE_DRS_DIM2_TABLE main_dom_table2;

  /* calculate a */
  if( eisc->max_limit < INT_MAX )
    a = eisc->max_limit - unassigned_child_count;
  else
    a = INT_MAX;

  /* calculate b */
  if( eisc->min_limit > 0 )
    b = eisc->min_limit;
  else
    b = 0;

  /* make and return the dom test */
  main_dom_table2 = KheDrsDim3TableGet(eisc->main_dom_table3,
    unassigned_child_count);
  dt_type = KheDomTestTypeActual(drs->solve_dom_test_type,
    eisc->cost_fn != KHE_QUADRATIC_COST_FUNCTION);
  return KheDrsDomTestMake(dt_type, (KHE_DRS_EXPR) eisc, eisc->allow_zero,
    eisc->min_limit, eisc->max_limit, a, b, eisc->combined_weight,
    main_dom_table2, NULL, eisc->monitor_info->monitor, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprIntSumCostDayDomTest(                         */
/*    KHE_DRS_EXPR_INT_SUM_COST eisc, int open_day_index,                    */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Return the dom test for eisc on the open day with index open_day_index.  */
/*                                                                           */
/*  When two solutions are compared using this dom test, the open children   */
/*  up to and including those on the day with index open_day_index will be   */
/*  be assigned, leaving                                                     */
/*                                                                           */
/*    KheDrsOpenChildrenAtOrAfter(eisc, open_day_index + 1)                  */
/*                                                                           */
/*  unassigned children.                                                     */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DOM_TEST KheDrsExprIntSumCostDayDomTest(
  KHE_DRS_EXPR_INT_SUM_COST eisc, int open_day_index,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int unassigned_child_count;
  unassigned_child_count = KheDrsOpenChildrenAtOrAfter(
    &eisc->open_children_by_day, open_day_index + 1);
  return KheDrsExprIntSumCostGeneralDomTest(eisc, unassigned_child_count, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprIntSumCostShiftDomTest(                       */
/*    KHE_DRS_EXPR_INT_SUM_COST eisc, int open_day_index, int curr_count,    */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Return the shift dom test for eisc on the open day with index            */
/*  open_day_index and curr_count children affected by shift.                */
/*                                                                           */
/*  When two solutions are compared using this dom test, the open children   */
/*  up to and including those on the day with index open_day_index - 1 will  */
/*  all be assigned, plus curr_count children on day open_day_index, making  */
/*                                                                           */
/*    KheDrsOpenChildrenAtOrAfter(eisc, open_day_index) - curr_count         */
/*                                                                           */
/*  unassigned children.                                                     */
/*                                                                           */
/*****************************************************************************/

/* *** unused
static KHE_DRS_DOM_TEST KheDrsExprIntSumCostShiftDomTest(
  KHE_DRS_EXPR_INT_SUM_COST eisc, int open_day_index, int curr_count,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int unassigned_child_count;
  unassigned_child_count = KheDrsOpenChildrenAtOrAfter(
    &eisc->open_children_by_day, open_day_index) - curr_count;
  return KheDrsExprIntSumCostGeneralDomTest(eisc, unassigned_child_count, drs);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsAdjustedSigVal(int val, KHE_DRS_ADJUST_TYPE adjust_type,       */
/*    int min_limit, int max_limit, int pc, int qc)                          */
/*                                                                           */
/*  Return the adjusted signature value.                                     */
/*                                                                           */
/*****************************************************************************/

static int KheDrsAdjustedSigVal(int val, KHE_DRS_ADJUST_TYPE adjust_type,
  int min_limit, int max_limit, int history_after)
{
  int lbar;
  switch( adjust_type )
  {
    case KHE_DRS_ADJUST_ORDINARY:

      return val;

    case KHE_DRS_ADJUST_NO_MAX:

      lbar = max(0, min_limit - history_after);
      return min(lbar, val);

    case KHE_DRS_ADJUST_LINEAR:

      lbar = max_limit;
      return min(lbar, val);

    case KHE_DRS_ADJUST_STEP:

      lbar = max_limit + 1;
      return min(lbar, val);

    default:

      HnAbort("KheDrsAdjustedSigVal internal error (adjusttype)");
      return val;  /* keep compiler happy */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSumCostEvalDay(KHE_DRS_EXPR_INT_SUM_COST eisc,         */
/*    KHE_DRS_SOLN prev_soln, KHE_DRS_SOLN next_soln,                        */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Evaluate expression eisc in next_soln.                                   */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsExprIntSumCostEvalDay(KHE_DRS_EXPR_INT_SUM_COST eisc,
  KHE_DRS_SOLN prev_soln, KHE_DRS_SOLN next_soln,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int di, pc, p1, p2, qc, q1, q2, i, i1, i2, dev1, dev2, adjusted_val;
  KHE_DRS_EXPR child_e;  KHE_COST old_cost, new_cost;

  ** get di, the open day index of next_soln **
  di = KheDrsSolnOpenDayIndex(next_soln);

  ** get pc, p1, q1, and dev1 **
  p1 = pc = eisc->history + eisc->closed_state;
  if( KheDrsOpenChildrenIndexIsFirst(&eisc->open_children_by_day, di) )
  {
    if( DEBUG_LAST_EDGE )
      fprintf(stderr, "  _%d", p1);
  }
  else
  {
    p1 += KheDrsExprDaySigVal((KHE_DRS_EXPR) eisc, di - 1, prev_soln);
    if( DEBUG_LAST_EDGE )
      fprintf(stderr, "  $%d:%d",
	KheDrsExprDaySigVal((KHE_DRS_EXPR) eisc, di-1, prev_soln), p1);
  }
  qc = eisc->history_after;
  q1 = qc + KheDrsOpenChildrenAtOrAfter(&eisc->open_children_by_day, di);
  dev1 = KheDrsExprIntSumCostDev(eisc, p1, q1);
  if( DEBUG_LAST_EDGE )
    fprintf(stderr, " dev1 %d = g(%d, %d)", dev1, p1, q1);

  ** get qc, p2, q2, and dev2 **
  p2 = p1;
  KheDrsOpenChildrenForEach(&eisc->open_children_by_day, di, child_e, i)
    p2 += child_e->u.int_val;
  q2 = qc + KheDrsOpenChildrenAtOrAfter(&eisc->open_children_by_day, di + 1);
  dev2 = KheDrsExprIntSumCostDev(eisc, p2, q2);
  if( DEBUG_LAST_EDGE )
    fprintf(stderr, " dev2 %d = g(%d, %d)", dev2, p2, q2);

  ** report the change in cost, if any **
  if( dev2 != dev1 )
  {
    new_cost = KheDrsCost(eisc->cost_fn, eisc->combined_weight, dev2);
    old_cost = KheDrsCost(eisc->cost_fn, eisc->combined_weight, dev1);
    next_soln->cost += new_cost - old_cost;
    ** KheDrsDaySingleCostAddCost(next_soln->day, new_cost - old_cost); **
    KheDrsUpdateRerunCost(drs, &eisc->rerun_open_and_search_cost, NULL,
      eisc->monitor, "search", -1, "+-", new_cost, old_cost);
    if( DEBUG_LAST_EDGE )
      fprintf(stderr, " +%.5f -%.5f", KheCostShow(new_cost),
	KheCostShow(old_cost));
  }

  ** if not eisc's last day, store p2 - pc (adjusted) in next_soln **
  if( !KheDrsOpenChildrenIndexIsLast(&eisc->open_children_by_day, di) )
  {
    adjusted_val = KheDrsAdjustedSigVal(p2 - pc, eisc->adjust_type,
      eisc->min_limit, eisc->max_limit, pc, qc);
    KheDrsExprPutSigVal((KHE_DRS_EXPR) eisc, di, next_soln, adjusted_val);
    if( DEBUG_LAST_EDGE )
      fprintf(stderr, "  %d:%d$", p2 - pc, adjusted_val);
  }
  if( DEBUG_LAST_EDGE )
    fprintf(stderr, " %s\n", KheMonitorId(eisc->monitor));
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprIntSumCostInitialValue(KHE_DRS_EXPR_INT_SUM_COST eisc)     */
/*                                                                           */
/*  Return the initial value of eisc.  This is used just below, but it is    */
/*  also used for dominance testing of correlated expressions.               */
/*                                                                           */
/*****************************************************************************/

static int KheDrsExprIntSumCostInitialValue(KHE_DRS_EXPR_INT_SUM_COST eisc)
{
  return eisc->history + eisc->closed_state;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSumCostEvalSignature(KHE_DRS_EXPR_INT_SUM_COST eisc,   */
/*    KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, int next_di,           */
/*    KHE_DRS_SIGNATURE next_sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)           */
/*                                                                           */
/*  Evaluate expression eisc, assuming that next_di has just been assigned,  */
/*  and update the cost and states of sig with the result.                   */
/*                                                                           */
/*  Terminology used here        Terminology used in doc                     */
/*  --------------------------------------------------------                 */
/*  ld1                          l(m, S)                                     */
/*  ud1                          u(m, S)                                     */
/*  dev1                         delta(l(m, S), u(m, S)                      */
/*  cost1                        f(delta(l(m, S), u(m, S))                   */
/*  --------------------------------------------------------                 */
/*  ld2                          l(m, S')                                    */
/*  ud2                          u(m, S')                                    */
/*  dev2                         delta(l(m, S'), u(m, S')                    */
/*  cost2                        f(delta(l(m, S'), u(m, S'))                 */
/*  --------------------------------------------------------                 */
/*                                                                           */
/*****************************************************************************/
static void KheDrsSolnSignatureSetDebug(KHE_DRS_SOLN soln, FILE *fp);
/* ***
#define DEBUG_ME (false &&						\
  strcmp(KheDrsMonitorInfoId(eisc->monitor_info), "Constraint:13/NU_6") == 0)
*** */

static void KheDrsExprIntSumCostEvalSignature(KHE_DRS_EXPR_INT_SUM_COST eisc,
  KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, int next_di,
  KHE_DRS_SIGNATURE next_sig, KHE_DYNAMIC_RESOURCE_SOLVER drs, bool debug)
{
  int ld1, ld2, ud1, ud2, i, i1, i2, dev1, dev2, adjusted_ld2, si;
  KHE_DRS_EXPR child_e;

  /* get ld1, ud1, and dev1 (for all days before next_di) */
  if( debug )
  {
    fprintf(stderr, "  [ eval %s, next_di %d, open_children_by_day:\n",
      KheDrsMonitorInfoId(eisc->monitor_info), next_di);
    KheDrsOpenChildrenDebug(&eisc->open_children_by_day, 1, 4, stderr);
  }
  if( KheDrsOpenChildrenIndexIsFirstOrLess(&eisc->open_children_by_day,next_di))
  {
    ld1 = KheDrsExprIntSumCostInitialValue(eisc);
    if( debug )
      fprintf(stderr, "    ld1 %d + %d", eisc->history, eisc->closed_state);
  }
  else
  {
    ld1 = KheDrsExprDaySigVal((KHE_DRS_EXPR) eisc, next_di - 1, prev_sig);
    if( debug )
      fprintf(stderr, "    ld1 %d", ld1);
  }
  ud1 = ld1 + eisc->history_after +
    KheDrsOpenChildrenAtOrAfter(&eisc->open_children_by_day, next_di);
  dev1 = KheDrsExprIntSumCostDelta(eisc, ld1, ud1);
  if( debug )
    fprintf(stderr, " ud1 %d + %d + %d dev1 %d\n", ld1, eisc->history_after,
      KheDrsOpenChildrenAtOrAfter(&eisc->open_children_by_day, next_di), dev1);

  /* get ld2, ud2, and dev2 for one more shift or one more day */
  ld2 = ld1;
  ud2 = ud1;
  if( dsg->type == KHE_DRS_SIGNER_SHIFT )
  {
    si = dsg->u.shift->open_shift_index;
    if( debug )
    {
      fprintf(stderr, "    open_children_by_shift:\n");
      KheDrsOpenChildrenDebug(&eisc->open_children_by_shift, 1, 4, stderr);
      fprintf(stderr, "    si %d ", si);
    }
    KheDrsOpenChildrenForEach(&eisc->open_children_by_shift, si, child_e, i)
      if( child_e->u.int_val == 1 )
      {
	ld2 += 1;  /* unassigned to active */
	if( debug )
	  fprintf(stderr, "%d+", i);
      }
      else
      {
	ud2 -= 1;  /* unassigned to inactive */
	if( debug )
	  fprintf(stderr, "%d-", i);
      }
  }
  else
  {
    if( debug )
      fprintf(stderr, " next_di %d ", next_di);
    KheDrsOpenChildrenForEach(&eisc->open_children_by_day, next_di, child_e, i)
      if( child_e->u.int_val == 1 )
      {
	ld2 += 1;  /* unassigned to active */
	if( debug )
	  fprintf(stderr, "+");
      }
      else
      {
	ud2 -= 1;  /* unassigned to inactive */
	if( debug )
	  fprintf(stderr, "-");
      }
  }
  dev2 = KheDrsExprIntSumCostDelta(eisc, ld2, ud2);
  if( debug )
    fprintf(stderr, " ld2 %d ud2 %d dev2 %d", ld2, ud2, dev2);

  /* if more to do, store ld2 (adjusted) in next_sig */
  if( ud2 - ld2 > eisc->history_after )
  {
    adjusted_ld2 = KheDrsAdjustedSigVal(ld2, eisc->adjust_type,
      eisc->min_limit, eisc->max_limit, eisc->history_after);
    KheDrsSignatureAddState(next_sig, adjusted_ld2, dsg, (KHE_DRS_EXPR) eisc);
    if( debug )
      fprintf(stderr, " S%d", adjusted_ld2);
  }

  /* report the extra cost, if any */
  if( dev2 != dev1 )
  {
    KheDrsSignatureAddCost(next_sig, f(eisc, dev2) - f(eisc, dev1));
    KheDrsMonitorInfoUpdateRerunCost(eisc->monitor_info, drs, dsg,
      KHE_DRS_SEARCH, "search", -1, "+-", f(eisc, dev2), f(eisc, dev1));
    if( debug )
      fprintf(stderr, " C%.5f", KheCostShow(f(eisc, dev2) - f(eisc, dev1)));
  }
  if( debug )
    fprintf(stderr, "\n  ]\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSumCostEvalShiftSignature(                             */
/*    KHE_DRS_EXPR_INT_SUM_COST eisc, KHE_DRS_SIGNER dsg,                    */
/*    KHE_DRS_SOLN prev_soln, int next_di, KHE_DRS_SIGNATURE sig,            */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Evaluate expression eisc in sig, for shifts not days.                    */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsExprIntSumCostEvalShiftSignature(
  KHE_DRS_EXPR_INT_SUM_COST eisc, KHE_DRS_SIGNER dsg,
  KHE_DRS_SOLN prev_soln, int next_di, KHE_DRS_SIGNATURE sig,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int ld1, ld2, ud1, ud2, i, i1, i2, dev1, dev2, adjusted_ld2, si;
  KHE_DRS_EXPR child_e;

  ** get ld1, ud1, and dev1 (for all days before next_di) **
  if( KheDrsOpenChildrenIndexIsFirst(&eisc->open_children_by_day, next_di) )
    ld1 = eisc->history + eisc->closed_state;
  else
    ld1 = KheDrsExprDaySigVal((KHE_DRS_EXPR) eisc, next_di - 1, prev_soln);
  ud1 = ld1 + eisc->history_after +
    KheDrsOpenChildrenAtOrAfter(&eisc->open_children_by_day, next_di);
  dev1 = KheDrsExprIntSumCostDelta(eisc, ld1, ud1);

  ** get ld2, ud2, and dev2 (for one extra shift) **
  ld2 = ld1;
  ud2 = ud1;
  si = dsg->u.shift->open_shift_index;
  KheDrsOpenChildrenForEach(&eisc->open_children_by_shift, si, child_e, i)
    if( child_e->u.int_val == 1 )
      ld2 += 1;  ** unassigned to active **
    else
      ud2 -= 1;  ** unassigned to inactive **
  dev2 = KheDrsExprIntSumCostDelta(eisc, ld2, ud2);

  ** if not complete, store ld2 (adjusted) in sig **
  if( ud2 - ld2 > eisc->history_after )
  {
    adjusted_ld2 = KheDrsAdjustedSigVal(ld2, eisc->adjust_type,
      eisc->min_limit, eisc->max_limit, eisc->history_after);
    KheDrsSignatureAddState(sig, adjusted_ld2, dsg);
  }

  ** report the change in cost, if any **
  if( dev2 != dev1 )
    KheDrsSignatureAddCost(sig, f(eisc, dev2) - f(eisc, dev1));
}
*** */


/* *** old version, old terminology
static void KheDrsExprIntSumCostEvalSignature(KHE_DRS_EXPR_INT_SUM_COST eisc,
  KHE_DRS_SIGNER dsg, KHE_DRS_SOLN prev_soln, int next_di,
  KHE_DRS_SIGNATURE sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int p1, p2, qc, q1, q2, i, i1, i2, dev1, dev2, adjusted_val;
  KHE_DRS_EXPR child_e;  KHE_COST old_cost, new_cost;  bool only_shift;

  ** get p1, q1, and dev1 **
  if( KheDrsOpenChildrenIndexIsFirst(&eisc->open_children_by_day, next_di) )
  {
    p1 = eisc->history + eisc->closed_state;
  }
  else
  {
    p1 = KheDrsExprDaySigVal((KHE_DRS_EXPR) eisc, next_di - 1, prev_soln);
    if( DEBUG_LAST_EDGE || DEBUG_ME )
    {
      fprintf(stderr, "  prev_soln %.5f ", KheCostShow(prev_soln->sig.cost));
      KheDrsSolnSignaturSeteDebug(prev_soln, stderr);
      fprintf(stderr, "\n  index %d, pos %d, ",
        next_di - 1 - eisc->open_children_by_day.range.first,
	HaArray(eisc->sig_indexes, next_di - 1 -
	eisc->open_children_by_day.range.first));
    }
  }
  qc = eisc->history_after;
  q1 = qc + KheDrsOpenChildrenAtOrAfter(&eisc->open_children_by_day, next_di);
  dev1 = KheDrsExprIntSumCostDev(eisc, p1, q1);
  if( DEBUG_LAST_EDGE || DEBUG_ME )
    fprintf(stderr, "  hi %d cs %d, p1 %d, qc %d, q1 %d, dev1 %d, ",
      eisc->history, eisc->closed_state, p1, qc, q1, dev1);

  ** get qc, p2, q2, and dev2 **
  if( dsg->type == KHE_DRS_SIGNER_SHIFT )
  {
    ** doing this for a shift signer **
    p2 = p1;
    q2 = q1;
    only_shift = true;
    KheDrsOpenChildrenForEach(&eisc->open_children_by_day, next_di,
      child_e, i)
    {
      if( KheDrsExprShift(child_e) == dsg->u.shift )
	p2 += child_e->u.int_val, q2 -= 1;
      else
	only_shift = false;
    }

    ** if not eisc's last day and only shift, store p2 (adjusted) in sig **
    if( !(KheDrsOpenChildrenIndexIsLast(&eisc->open_children_by_day, next_di) && only_shift) )
    {
      ** store p2 **
      adjusted_val = KheDrsAdjustedSigVal(p2, eisc->adjust_type,
	eisc->min_limit, eisc->max_limit, qc);
      if( DEBUG48 && HaArrayCount(sig->states) >= HaArrayCount(dsg->dom_tests) )
      {
	fprintf(stderr, "[ KheDrsExprIntSumCostEvalSignature(%s, day %d)\n",
	  KheDrsMonitorInfoId(eisc->monitor_info), next_di);
	fprintf(stderr, "  KheDrsExprOpenDayIsLast %s, only_shift %s\n",
          bool_show(KheDrsOpenChildrenIndexIsLast(&eisc->open_children_by_day, next_di)),
          bool_show(only_shift));
	fprintf(stderr, "  open_day_range %d - %d, next_di %d\n",
          eisc->open_children_by_day.range.first,
	  eisc->open_children_by_day.range.last, next_di);
	fprintf(stderr, "]\n");
      }
      KheDrsSignatureAddState(sig, adjusted_val, dsg);
    }
  }
  else
  {
    ** doing this for a day or resource on day signer **
    p2 = p1;
    KheDrsOpenChildrenForEach(&eisc->open_children_by_day, next_di,
      child_e, i)
      p2 += child_e->u.int_val;
    q2 = qc +
      KheDrsOpenChildrenAtOrAfter(&eisc->open_children_by_day, next_di + 1);
    if( DEBUG_LAST_EDGE || DEBUG_ME )
      fprintf(stderr, "p2 %d, q2 %d, dev2 %d, ", p2, q2, dev2);

    ** if not eisc's last day, store p2 (adjusted) in next_soln **
    if( !KheDrsOpenChildrenIndexIsLast(&eisc->open_children_by_day, next_di) )
    {
      ** store p2 **
      adjusted_val = KheDrsAdjustedSigVal(p2, eisc->adjust_type,
	eisc->min_limit, eisc->max_limit, qc);
      KheDrsSignatureAddState(sig, adjusted_val, dsg);
      if( DEBUG_LAST_EDGE || DEBUG_ME )
	fprintf(stderr, "  p2 %d, adj %d, ", p2, adjusted_val);
    }
    if( DEBUG_LAST_EDGE || DEBUG_ME )
      fprintf(stderr, " %s\n", KheDrsMonitorInfoId(eisc->monitor_info));
  }

  ** report the change in cost, if any **
  dev2 = KheDrsExprIntSumCostDev(eisc, p2, q2);
  if( dev2 != dev1 )
  {
    new_cost = f(eisc, dev2);
    old_cost = f(eisc, dev1);
    sig->cost += new_cost - old_cost;
    KheDrsMonitorInfoUpdateRerunCost(eisc->monitor_info, drs, dsg,
      KHE_DRS_SEARCH, "search", -1, "+-", new_cost, old_cost);
    if( DEBUG_LAST_EDGE || DEBUG_ME )
      fprintf(stderr, " +%.5f -%.5f, ", KheCostShow(new_cost),
	KheCostShow(old_cost));
  }
}
*** */


/* *** correct but uses old terminology
static void KheDrsExprIntSumCostEvalDay(KHE_DRS_EXPR_INT_SUM_COST eisc,
  KHE_DRS_SOLN prev_soln, KHE_DRS_SOLN next_soln,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int i1, i2, ai, bi, ci, xi, prev_dev, yi, dev, di;  KHE_DRS_EXPR child_e;

  ** get ai, bi, and ci **
  di = KheDrsSolnOpenDayIndex(next_soln);
  ai = KheDrsExprNumberOfOpenChildrenBefore(eisc, di);
  bi = KheDrsExprNumberOfOpenChildrenBefore(eisc, di + 1) - ai;
  ci = HaArrayCount(eisc->open_children) - ai - bi;

  ** get xi and prev_dev **
  if( KheDrsOpenChildrenIndexIsFirst(&eisc->open_children_by_day, di) )
  {
    xi = 0;
    prev_dev = KheDrsExprIntSumCostStartDev(eisc);
  }
  else
  {
    xi = KheDrsExprDaySigVal((KHE_DRS_EXPR) eisc, di - 1, prev_soln);
    prev_dev = KheDrsExprIntSumCostDev(eisc, eisc->history +
      eisc->closed_state + xi, bi + ci + eisc->history_after);
  }

  ** get yi and dev **
  yi = 0;
  KheDrsOpenChildrenForEach(&eisc->open_children_by_day, di, child_e, i1)
    yi += child_e->u.int_val;
  dev = KheDrsExprIntSumCostDev(eisc,
    eisc->history + eisc->closed_state + xi + yi, ci + eisc->history_after);

  ** if not eisc's last day, store xi + yi (or less) in next_soln **
  if( !KheDrsOpenChildrenIndexIsLast(&eisc->open_children_by_day, di) )
    KheDrsExprPutSigVal((KHE_DRS_EXPR) eisc, di, next_soln,
      KheDrsExprIntSumCostAdjustedSigVal(eisc, xi + yi));

  ** report the change in cost c(Ci) - c(C{i-1}) **
  if( dev != prev_dev )
    next_soln->cost += (KheDrsCost(eisc->cost_fn, eisc->combined_weight, dev)
      - KheDrsCost(eisc->cost_fn, eisc->combined_weight, prev_dev));
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSumCostEvalShiftSignature(                             */
/*    KHE_DRS_EXPR_INT_SUM_COST eisc, KHE_DRS_SOLN prev_soln,                */
/*    int next_di, KHE_DRS_SHIFT ds, KHE_DRS_SIGNATURE sig,                  */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Evaluate expression eisc in sig after assigning one shift.               */
/*                                                                           */
/*****************************************************************************/

/* *** now folded into KheDrsExprIntSumCostEvalSignature
static void KheDrsSolnSignatureSetDebug(KHE_DRS_SOLN soln, FILE *fp);

#define DEBUG_ME (false &&						\
  strcmp(KheDrsMonitorInfoId(eisc->monitor_info), "Constraint:13/NU_6") == 0)

static void KheDrsExprIntSumCostEvalShiftSignature(
  KHE_DRS_EXPR_INT_SUM_COST eisc, KHE_DRS_SOLN prev_soln,
  int next_di, KHE_DRS_SHIFT ds, KHE_DRS_SIGNATURE sig,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int p1, p2, qc, q1, q2, i, i1, i2, dev1, dev2, adjusted_val;
  KHE_DRS_EXPR child_e;  KHE_COST old_cost, new_cost;  bool only_shift;

  ** get p1, q1, and dev1 **
  if( KheDrsOpenChildrenIndexIsFirst(&eisc->open_children_by_day, next_di) )
    p1 = eisc->history + eisc->closed_state;
  else
    p1 = KheDrsExprDaySigVal((KHE_DRS_EXPR) eisc, next_di - 1, prev_soln);
  qc = eisc->history_after;
  q1 = qc + KheDrsOpenChildrenAtOrAfter(&eisc->open_children_by_day, next_di);
  dev1 = KheDrsExprIntSumCostDev(eisc, p1, q1);

  ** get qc, p2, q2, and dev2 for adding shift si **
  p2 = p1;
  q2 = q1;
  only_shift = true;
  KheDrsOpenChildrenForEach(&eisc->open_children_by_day, next_di, child_e, i)
  {
    if( KheDrsExprShift(child_e) == ds )
      p2 += child_e->u.int_val, q2 -= 1;
    else
      only_shift = false;
  }
  ** ***
  q2 = qc + KheDrsExprOpenChildrenAtOrAfterExceptShift((KHE_DRS_EXPR) eisc,
    next_di, shift_index);
  *** **
  dev2 = KheDrsExprIntSumCostDev(eisc, p2, q2);

  ** report the change in cost, if any **
  if( dev2 != dev1 )
  {
    new_cost = f(eisc, dev2);
    old_cost = f(eisc, dev1);
    sig->cost += new_cost - old_cost;
  }

  ** if not eisc's last day and only shift, store p2 (adjusted) in next_soln **
  if( !(KheDrsOpenChildrenIndexIsLast(&eisc->open_children_by_day, next_di) && only_shift) )
  {
    ** store p2 **
    adjusted_val = KheDrsAdjustedSigVal(p2, eisc->adjust_type,
      eisc->min_limit, eisc->max_limit, qc);
    KheDrsExprPutSigVal((KHE_DRS_EXPR) eisc, next_di, sig, adjusted_val);
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSumCostInitRerunCost(KHE_DRS_EXPR_INT_SUM_COST eisc)   */
/*                                                                           */
/*  Initialize the rerun cost fields of eisc.                                */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsExprIntSumCostInitRerunCost(KHE_DRS_EXPR_INT_SUM_COST eisc)
{
  eisc->rerun_open_and_search_cost = eisc->rerun_open_and_close_cost =
    KheMonitorCost(eisc->monitor);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSumCostCheckRerunCost(KHE_DRS_EXPR_INT_SUM_COST eisc)  */
/*                                                                           */
/*  Check the rerun cost fields of eisc.                                     */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsExprIntSumCostCheckRerunCost(KHE_DRS_EXPR_INT_SUM_COST eisc)
{
  if( eisc->rerun_open_and_search_cost != KheMonitorCost(eisc->monitor) )
    fprintf(stderr, "  open_and_search_cost %.5f, monitor cost %.5f (%s)\n",
      KheCostShow(eisc->rerun_open_and_search_cost),
      KheCostShow(KheMonitorCost(eisc->monitor)), KheMonitorId(eisc->monitor));
  if( eisc->rerun_open_and_close_cost != KheMonitorCost(eisc->monitor) )
    fprintf(stderr, "  open_and_close_cost  %.5f, monitor cost %.5f (%s)\n",
      KheCostShow(eisc->rerun_open_and_close_cost),
      KheCostShow(KheMonitorCost(eisc->monitor)), KheMonitorId(eisc->monitor));
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSumCostDoDebug(KHE_DRS_EXPR_INT_SUM_COST eisc,         */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of eisc which is specific to it.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntSumCostDoDebug(KHE_DRS_EXPR_INT_SUM_COST eisc,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "INT_SUM_COST(%s, %s, %d)",
    KheDrsMonitorInfoId(eisc->monitor_info),
    KheDrsAdjustTypeShow(eisc->adjust_type), eisc->closed_state);
  KheDrsExprDebugSignature((KHE_DRS_EXPR) eisc, soln, false, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_CLOSED_SEQ"                                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_CLOSED_SEQ KheDrsClosedSeqMake(KHE_DYNAMIC_RESOURCE_SOLVER drs,  */
/*    int start_index, int stop_index)                                       */
/*                                                                           */
/*  Make a new KHE_DRS_CLOSED_SEQ object with these attributes.              */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_CLOSED_SEQ KheDrsClosedSeqMake(KHE_DYNAMIC_RESOURCE_SOLVER drs,
  int start_index, int stop_index)
{
  KHE_DRS_CLOSED_SEQ res;
  if( HaArrayCount(drs->closed_seq_free_list) > 0 )
    res = HaArrayLastAndDelete(drs->closed_seq_free_list);
  else
    HaMake(res, drs->arena);
  res->start_index = start_index;
  res->stop_index = stop_index;
  res->active_at_left = 0;		/* placeholder (unless empty) */
  res->active_at_right = 0;		/* placeholder (unless empty) */
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_CLOSED_SEQ KheDrsClosedSeqMakeInit(                              */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make an initial empty closed sequence.                                   */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_CLOSED_SEQ KheDrsClosedSeqMakeInit(
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  return KheDrsClosedSeqMake(drs, 0, 0);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsClosedSeqDelete(KHE_DRS_CLOSED_SEQ dcs,                       */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Delete dcs.                                                              */
/*                                                                           */
/*****************************************************************************/

static void KheDrsClosedSeqDelete(KHE_DRS_CLOSED_SEQ dcs,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  HaArrayAddLast(drs->closed_seq_free_list, dcs);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsClosedSeqLength(KHE_DRS_CLOSED_SEQ dcs)                        */
/*                                                                           */
/*  Return the length of dcs.                                                */
/*                                                                           */
/*****************************************************************************/

static int KheDrsClosedSeqLength(KHE_DRS_CLOSED_SEQ dcs)
{
  return dcs->stop_index - dcs->start_index;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsClosedSeqAllActive(KHE_DRS_CLOSED_SEQ dcs)                    */
/*                                                                           */
/*  Return true if every child of dcs is active.                             */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsClosedSeqAllActive(KHE_DRS_CLOSED_SEQ dcs)
{
  return KheDrsClosedSeqLength(dcs) == dcs->active_at_left;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsClosedSeqExtendToRight(KHE_DRS_CLOSED_SEQ dcs, int val)       */
/*                                                                           */
/*  Extend dcs one place to the right, with a child with value val.          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsClosedSeqExtendToRight(KHE_DRS_CLOSED_SEQ dcs, int val)
{
  if( val == 0 )
    dcs->active_at_right = 0;
  else
  {
    if( KheDrsClosedSeqAllActive(dcs) )
      dcs->active_at_left++;
    dcs->active_at_right++;
  }
  dcs->stop_index++;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsClosedSeqSetActiveLengths(KHE_DRS_CLOSED_SEQ dcs,             */
/*    KHE_DRS_EXPR_INT_SEQ_COST eisc)                                        */
/*                                                                           */
/*  Assuming dcs->start_index and dcs->stop_index are up to date, bring      */
/*  dcs->active_at_left and dcs-> active_at_right up to date.                */
/*                                                                           */
/*****************************************************************************/

static void KheDrsClosedSeqSetActiveLengths(KHE_DRS_CLOSED_SEQ dcs,
  KHE_DRS_EXPR_INT_SEQ_COST eisc)
{
  KHE_DRS_EXPR child_e;  int i;

  /* bring dcs->active_at_left up to date */
  dcs->active_at_left = 0;
  for( i = dcs->start_index;  i < dcs->stop_index;  i++ )
  {
    child_e = HaArray(eisc->children, i);
    if( child_e->u.int_val != 1 )
      break;
    dcs->active_at_left++;
  }

  /* bring dcs->active_at_right up to date */
  dcs->active_at_right = 0;
  for( i = dcs->stop_index - 1;  i >= dcs->start_index;  i-- )
  {
    child_e = HaArray(eisc->children, i);
    if( child_e->u.int_val != 1 )
      break;
    dcs->active_at_right++;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_CLOSED_SEQ KheDrsClosedSeqSplit(KHE_DRS_CLOSED_SEQ dcs,          */
/*    int child_index)                                                       */
/*                                                                           */
/*  Split dcs at child_index.  The child at child_index will be in           */
/*  neither of the two fragments.                                            */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_CLOSED_SEQ KheDrsClosedSeqSplit(KHE_DRS_CLOSED_SEQ dcs,
  int child_index, KHE_DRS_EXPR_INT_SEQ_COST eisc,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_CLOSED_SEQ res;

  /* make the new fragment and reset the stop index of the old one */
  HnAssert(dcs->start_index <= child_index,
    "KheDrsClosedSeqSplit internal error 1");
  HnAssert(child_index < dcs->stop_index,
    "KheDrsClosedSeqSplit internal error 2 (child_index %d, dcs->stop_index %d",
    child_index, dcs->stop_index);
  res = KheDrsClosedSeqMake(drs, child_index + 1, dcs->stop_index);
  dcs->stop_index = child_index;

  /* update the active lengths of the two fragments, and return the new one */
  KheDrsClosedSeqSetActiveLengths(dcs, eisc);
  KheDrsClosedSeqSetActiveLengths(res, eisc);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsClosedSeqMerge(KHE_DRS_CLOSED_SEQ dcs1,                       */
/*    KHE_DRS_CLOSED_SEQ dcs2, KHE_DRS_EXPR_INT_SEQ_COST eisc,               */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Merge adjacent closed sequences dcs1 and dcs2.  The formerly open        */
/*  child separating them has closed, and its closed value has been set.     */
/*                                                                           */
/*  Place the merged closed sequence in dcs1, and delete dcs2.               */
/*                                                                           */
/*****************************************************************************/

static void KheDrsClosedSeqMerge(KHE_DRS_CLOSED_SEQ dcs1,
  KHE_DRS_CLOSED_SEQ dcs2, KHE_DRS_EXPR_INT_SEQ_COST eisc,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  /* update dcs1 to hold the merged sequence */
  HnAssert(dcs1->stop_index + 1 == dcs2->start_index,
    "KheDrsClosedSeqMerge internal error 1");
  dcs1->stop_index = dcs2->stop_index;
  KheDrsClosedSeqSetActiveLengths(dcs1, eisc);

  /* delete dcs2 */
  KheDrsClosedSeqDelete(dcs2, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsClosedSeqDebug(KHE_DRS_CLOSED_SEQ dcs, int verbosity,         */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of dcs onto fp with the given verbosity and indent.          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsClosedSeqDebug(KHE_DRS_CLOSED_SEQ dcs, int verbosity,
  int indent, FILE *fp)
{
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  fprintf(fp, "[CLOSED_SEQ %d-%d, active_at_left %d, active_at_right %d]",
    dcs->start_index, dcs->stop_index, dcs->active_at_left,
    dcs->active_at_right);
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_A_INTERVAL"                                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_A_INTERVAL KheDrsAIntervalMake(int start_index, int stop_index,  */
/*    bool unassigned_precedes)                                              */
/*                                                                           */
/*  Make and return a new a-interval with these attributes.                  */
/*                                                                           */
/*****************************************************************************/

KHE_DRS_A_INTERVAL KheDrsAIntervalMake(int start_index, int stop_index,
  bool unassigned_precedes)
{
  KHE_DRS_A_INTERVAL res;
  res.start_index = start_index;
  res.stop_index = stop_index;
  res.unassigned_precedes = unassigned_precedes;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_A_INTERVAL KheDrsAIntervalMakeLeft(                              */
/*    KHE_DRS_EXPR_INT_SEQ_COST eisc, int open_index, int active_len)        */
/*                                                                           */
/*  Make an a-interval of length active_len which ends just before           */
/*  the open child with index open_index.                                    */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_A_INTERVAL KheDrsAIntervalMakeLeft(
  KHE_DRS_EXPR_INT_SEQ_COST eisc, int open_index, int active_len)
{
  KHE_DRS_CLOSED_SEQ dcs;
  dcs = HaArray(eisc->closed_seqs, open_index);
  return KheDrsAIntervalMake(dcs->stop_index - active_len,
    dcs->stop_index, false);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_A_INTERVAL KheDrsAIntervalFindLeft(                              */
/*    KHE_DRS_EXPR_INT_SEQ_COST eisc, int open_index)                        */
/*                                                                           */
/*  Find the (possibly empty) a-interval immediately to the left of the      */
/*  child with the given open_index.  Include history if necessary.          */
/*                                                                           */
/*  Implementation note.  This interval consists entirely of the last        */
/*  active_at_right children of the preceding KHE_DRS_CLOSED_SEQ             */
/*  (possibly plus history), even if their number is zero; because they      */
/*  are preceded by another open child, or an inactive child, or nothing.    */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_A_INTERVAL KheDrsAIntervalFindLeft(
  KHE_DRS_EXPR_INT_SEQ_COST eisc, int open_index)
{
  KHE_DRS_CLOSED_SEQ dcs;
  dcs = HaArray(eisc->closed_seqs, open_index);
  if( !KheDrsClosedSeqAllActive(dcs) )
  {
    /* an inactive child precedes the active_at_right active children */
    return KheDrsAIntervalMake(dcs->stop_index - dcs->active_at_right,
      dcs->stop_index, false);
  }
  else if( open_index > 0 )
  {
    /* an unassigned child precedes the active_at_right active children */
    return KheDrsAIntervalMake(dcs->stop_index - dcs->active_at_right,
      dcs->stop_index, true);
  }
  else
  {
    /* nothing but history precedes the active_at_right active children */
    return KheDrsAIntervalMake(dcs->stop_index - dcs->active_at_right
      - eisc->history, dcs->stop_index, false);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_A_INTERVAL KheDrsAIntervalFindRight(                             */
/*    KHE_DRS_EXPR_INT_SEQ_COST eisc, int open_index, bool child_unassigned) */
/*                                                                           */
/*  Find the (possibly empty) a-interval immediately to the right of the     */
/*  child with the given open_index.  That child is assumed to be unassigned */
/*  when child_unassigned is true.                                           */
/*                                                                           */
/*  Implementation note.  This interval consists entirely of the first       */
/*  active_at_left children of the following KHE_DRS_CLOSED_SEQ,             */
/*  even if their number is zero; because they are followed by another       */
/*  open child, or an inactive child, or history_after, or nothing.          */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_A_INTERVAL KheDrsAIntervalFindRight(
  KHE_DRS_EXPR_INT_SEQ_COST eisc, int open_index, bool child_unassigned)
{
  KHE_DRS_CLOSED_SEQ dcs;
  dcs = HaArray(eisc->closed_seqs, open_index + 1);
  return KheDrsAIntervalMake(dcs->start_index,
    dcs->start_index + dcs->active_at_left, child_unassigned);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_A_INTERVAL KheDrsAIntervalMerge(KHE_DRS_A_INTERVAL ai_left,      */
/*    KHE_DRS_A_INTERVAL ai_right)                                           */
/*                                                                           */
/*  Merge ai_left with ai_right.  These intervals are separated by one       */
/*  child (this is checked).  That child is assumed to be active, but that   */
/*  is not checked, and indeed may not be true - we might be building this   */
/*  interval to reflect the old state of that child.                         */
/*                                                                           */
/*  It is safe to set the unassigned_precedes value of the result interval   */
/*  to ai_left.unassigned_precedes.  Even when ai_left is empty, its place   */
/*  is well defined, and ai_left.unassigned_precedes is correct for there.   */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_A_INTERVAL KheDrsAIntervalMerge(KHE_DRS_A_INTERVAL ai_left,
  KHE_DRS_A_INTERVAL ai_right)
{
  HnAssert(ai_left.stop_index + 1 == ai_right.start_index,
    "KheDrsAIntervalMerge internal error");
  return KheDrsAIntervalMake(ai_left.start_index, ai_right.stop_index,
    ai_left.unassigned_precedes);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsAIntervalDev(KHE_DRS_A_INTERVAL ai,                            */
/*    KHE_DRS_EXPR_INT_SEQ_COST eisc)                                        */
/*                                                                           */
/*  Return the deviation of a-interval ai.  If ai.unassigned_precedes is     */
/*  true, there is an unassigned child immediately before ai.first_index,    */
/*  so return 0 if the cost function is KHE_STEP_COST_FUNCTION.              */
/*                                                                           */
/*****************************************************************************/

static int KheDrsAIntervalDev(KHE_DRS_A_INTERVAL ai,
  KHE_DRS_EXPR_INT_SEQ_COST eisc)
{
  int len;

  /* return 0 if an unassigned child immediately precedes this interval */
  /* and the cost function is step                                      */
  if( ai.unassigned_precedes && eisc->cost_fn == KHE_STEP_COST_FUNCTION )
    return 0;

  /* include history if we are at the start */
  /* *** already included where relevant
  if( ai.start_index == 0 )
    ai.start_index -= eisc->history;
  *** */

  /* calculate the length and deviation */
  len = ai.stop_index - ai.start_index;
  return len > eisc->max_limit ? len - eisc->max_limit : 0;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_COST KheDrsAIntervalCost(KHE_DRS_A_INTERVAL ai,                      */
/*    KHE_DRS_EXPR_INT_SEQ_COST eisc)                                        */
/*                                                                           */
/*  Return the cost of ai.                                                   */
/*                                                                           */
/*****************************************************************************/

static KHE_COST KheDrsAIntervalCost(KHE_DRS_A_INTERVAL ai,
  KHE_DRS_EXPR_INT_SEQ_COST eisc)
{
  int dev;
  dev = KheDrsAIntervalDev(ai, eisc);
  return dev == 0 ? 0 : f(eisc, dev);
  /* ***
  return dev == 0 ? 0 : KheDrsCost(eisc->cost_fn, eisc->combined_weight, dev);
  *** */
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_AU_INTERVAL"                                          */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_AU_INTERVAL KheDrsAUIntervalMake(int start_index,                */
/*    int stop_index, bool has_active_child)                                 */
/*                                                                           */
/*  Make and return an au-interval with these attributes.  However, the      */
/*  has_active_child attribute is ignored when the interval is empty:  an    */
/*  empty interval cannot contain an active child.                           */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_AU_INTERVAL KheDrsAUIntervalMake(int start_index,
  int stop_index, bool has_active_child)
{
  KHE_DRS_AU_INTERVAL res;
  res.start_index = start_index;
  res.stop_index = stop_index;
  res.has_active_child = (start_index < stop_index && has_active_child);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsAUIntervalLength(KHE_DRS_AU_INTERVAL aui)                      */
/*                                                                           */
/*  Return the length of aui.                                                */
/*                                                                           */
/*****************************************************************************/

static int KheDrsAUIntervalLength(KHE_DRS_AU_INTERVAL aui)
{
  return aui.stop_index - aui.start_index;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsAUIntervalExtendToLeft(KHE_DRS_AU_INTERVAL *aui,              */
/*    int extra_len, bool has_active_child)                                  */
/*                                                                           */
/*  Extend *aui extra_len places to the left.  If extra_len > 0,             */
/*  has_active_child says whether the extra elements are active or not.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsAUIntervalExtendToLeft(KHE_DRS_AU_INTERVAL *aui,
  int extra_len, bool has_active_child)
{
  if( extra_len > 0 )
  {
    aui->start_index -= extra_len;
    if( has_active_child )
      aui->has_active_child = true;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsAUIntervalExtendToRight(KHE_DRS_AU_INTERVAL *aui,             */
/*    int extra_len, bool has_active_child)                                  */
/*                                                                           */
/*  Extend *aui extra_len places to the right.  If extra_len > 0,            */
/*  has_active_child says whether the extra elements are active or not.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsAUIntervalExtendToRight(KHE_DRS_AU_INTERVAL *aui,
  int extra_len, bool has_active_child)
{
  if( extra_len > 0 )
  {
    aui->stop_index += extra_len;
    if( has_active_child )
      aui->has_active_child = true;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_AU_INTERVAL KheDrsAUIntervalMakeLeft(                            */
/*    KHE_DRS_EXPR_INT_SEQ_COST eisc, int open_index, int active_len)        */
/*                                                                           */
/*  Make an au-interval of length active_len which ends just before          */
/*  the open child with index open_index.                                    */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_AU_INTERVAL KheDrsAUIntervalMakeLeft(
  KHE_DRS_EXPR_INT_SEQ_COST eisc, int open_index, int active_len)
{
  KHE_DRS_CLOSED_SEQ dcs;
  dcs = HaArray(eisc->closed_seqs, open_index);
  return KheDrsAUIntervalMake(dcs->stop_index - active_len,
    dcs->stop_index, true);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_AU_INTERVAL KheDrsAUIntervalFindLeft(                            */
/*    KHE_DRS_EXPR_INT_SEQ_COST eisc, int open_index)                        */
/*                                                                           */
/*  Find the au-interval ending just before the child whose open index       */
/*  is open_index.                                                           */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_AU_INTERVAL KheDrsAUIntervalFindLeft(
  KHE_DRS_EXPR_INT_SEQ_COST eisc, int open_index)
{
  KHE_DRS_CLOSED_SEQ dcs;  KHE_DRS_AU_INTERVAL res;  int i;

  /* initialize res to the interval of active children at the right of dcs */
  dcs = HaArray(eisc->closed_seqs, open_index);
  res = KheDrsAUIntervalMake(dcs->stop_index - dcs->active_at_right,
    dcs->stop_index, true);
  if( !KheDrsClosedSeqAllActive(dcs) )
    return res;

  /* now keep looking to the left of there */
  for( i = open_index - 1;  i >= 0;  i-- )
  {
    /* return early if eisc->min_limit reached */
    if( KheDrsAUIntervalLength(res) >= eisc->min_limit )
      return res;

    /* res includes the open unassigned child before the previous dcs */
    KheDrsAUIntervalExtendToLeft(&res, 1, false);

    /* res includes the active children at the right end of the next dcs */
    dcs = HaArray(eisc->closed_seqs, i);
    KheDrsAUIntervalExtendToLeft(&res, dcs->active_at_right, true);
    if( !KheDrsClosedSeqAllActive(dcs) )
      return res;
  }

  /* at the start, so res includes history */
  KheDrsAUIntervalExtendToLeft(&res, eisc->history, true);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_AU_INTERVAL KheDrsAUIntervalFindRight(                           */
/*    KHE_DRS_EXPR_INT_SEQ_COST eisc, int open_index)                        */
/*                                                                           */
/*  Find the au-interval beginning just after the child whose open index     */
/*  is open_index.                                                           */
/*                                                                           */
/*****************************************************************************/
static char *DrsAUIntervalShow(KHE_DRS_AU_INTERVAL aui);

static KHE_DRS_AU_INTERVAL KheDrsAUIntervalFindRight(
  KHE_DRS_EXPR_INT_SEQ_COST eisc, int open_index,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_CLOSED_SEQ dcs;  KHE_DRS_AU_INTERVAL res;  int i;

  /* initialize res to the interval of active children at the left of dcs */
  if( DEBUG20 && drs->rerun != NULL &&
      strcmp(KheDrsMonitorInfoId(eisc->monitor_info), RERUN_MONITOR_ID) == 0 )
  {
    fprintf(stderr, "[ KheDrsAUIntervalFindRight(%s, %d, drs)\n",
      KheDrsMonitorInfoId(eisc->monitor_info), open_index);
    for( i = open_index + 1;  i < HaArrayCount(eisc->closed_seqs);  i++ )
    {
      dcs = HaArray(eisc->closed_seqs, i);
      fprintf(stderr, "  dcs %d ", i);
      KheDrsClosedSeqDebug(dcs, 2, 0, stderr);
    }
  }
  dcs = HaArray(eisc->closed_seqs, open_index + 1);
  res = KheDrsAUIntervalMake(dcs->start_index,
    dcs->start_index + dcs->active_at_left, true);
  if( !KheDrsClosedSeqAllActive(dcs) )
  {
    if( DEBUG20 && drs->rerun != NULL &&
	strcmp(KheDrsMonitorInfoId(eisc->monitor_info), RERUN_MONITOR_ID) == 0 )
      fprintf(stderr, "] KheDrsAUIntervalFindRight(eisc, %d) 1= %s\n",
	open_index, DrsAUIntervalShow(res));
    return res;
  }

  /* now keep looking to the right of there */
  for( i = open_index + 2;  i < HaArrayCount(eisc->closed_seqs);  i++ )
  {
    /* return early if eisc->min_limit reached */
    if( KheDrsAUIntervalLength(res) >= eisc->min_limit )
    {
      if( DEBUG20 && drs->rerun != NULL &&
	  strcmp(KheDrsMonitorInfoId(eisc->monitor_info), RERUN_MONITOR_ID)==0 )
	fprintf(stderr, "] KheDrsAUIntervalFindRight(eisc, %d) 2= %s\n",
	  open_index, DrsAUIntervalShow(res));
      return res;
    }

    /* res includes the open unassigned child after the previous dcs */
    KheDrsAUIntervalExtendToRight(&res, 1, false);

    /* res includes the active children at the left end of the next dcs */
    dcs = HaArray(eisc->closed_seqs, i);
    KheDrsAUIntervalExtendToRight(&res, dcs->active_at_left, true);
    if( !KheDrsClosedSeqAllActive(dcs) )
    {
      if( DEBUG20 && drs->rerun != NULL &&
	  strcmp(KheDrsMonitorInfoId(eisc->monitor_info), RERUN_MONITOR_ID)==0 )
	fprintf(stderr, "] KheDrsAUIntervalFindRight(eisc, %d) 3= %s\n",
	  open_index, DrsAUIntervalShow(res));
      return res;
    }
  }

  /* at the end, so res includes history_after */
  KheDrsAUIntervalExtendToRight(&res, eisc->history_after, false);
  if( DEBUG20 && drs->rerun != NULL &&
      strcmp(KheDrsMonitorInfoId(eisc->monitor_info), RERUN_MONITOR_ID) == 0 )
    fprintf(stderr, "] KheDrsAUIntervalFindRight(eisc, %d) 4= %s\n",
      open_index, DrsAUIntervalShow(res));
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsAUIntervalFindRightLengths(KHE_DRS_EXPR_INT_SEQ_COST eisc,    */
/*    int open_index, int *q, int *r)                                        */
/*                                                                           */
/*  Similar to KheDrsAUIntervalFindRight, only the interval includes the     */
/*  child whose open index is open_index, and instead of returning the       */
/*  interval, we return *q, the number of unassigned children in it, and     */
/*  *r, the number of active children in it.                                 */
/*                                                                           */
/*****************************************************************************/

static void KheDrsAUIntervalFindRightLengths(KHE_DRS_EXPR_INT_SEQ_COST eisc,
  int open_index, int *q, int *r)
{
  KHE_DRS_CLOSED_SEQ dcs;  int i;

  /* handle each unassigned child and its *following* closed interval */
  *q = *r = 0;
  for( i = open_index + 1;  i < HaArrayCount(eisc->closed_seqs);  i++ )
  {
    dcs = HaArray(eisc->closed_seqs, i);

    /* include the open (unassigned) child just before dcs */
    *q += 1;

    /* include the active children at the left end of dcs */
    *r += dcs->active_at_left;

    /* if there is an inactive child in dcs, it's time to stop */
    if( !KheDrsClosedSeqAllActive(dcs) )
      return;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_AU_INTERVAL KheDrsAUIntervalMerge(KHE_DRS_AU_INTERVAL aui_left,  */
/*    KHE_DRS_AU_INTERVAL aui_right, bool sep_is_active)                     */
/*                                                                           */
/*  Return the merge of aui_left and aui_right.  They are separated by one   */
/*  child, and that child is active if sep_is_active.                        */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_AU_INTERVAL KheDrsAUIntervalMerge(KHE_DRS_AU_INTERVAL aui_left,
  KHE_DRS_AU_INTERVAL aui_right, bool sep_is_active)
{
  HnAssert(aui_left.stop_index + 1 == aui_right.start_index,
    "KheDrsAUIntervalMerge internal error");
  return KheDrsAUIntervalMake(aui_left.start_index, aui_right.stop_index,
    aui_left.has_active_child || aui_right.has_active_child || sep_is_active);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsAUIntervalDev(KHE_DRS_AU_INTERVAL aui,                         */
/*    KHE_DRS_EXPR_INT_SEQ_COST eisc)                                        */
/*                                                                           */
/*  Return the deviation of au-interval aui.  If aui.has_active_child is     */
/*  false, the interval has no active child so return 0.  This should        */
/*  also handle the special case of len == 0, where the result is also 0.    */
/*                                                                           */
/*****************************************************************************/

static int KheDrsAUIntervalDev(KHE_DRS_AU_INTERVAL aui,
  KHE_DRS_EXPR_INT_SEQ_COST eisc)
{
  int len;

  /* include history if we are at the start */
  /* *** already included where relevant
  if( aui.start_index == 0 && eisc->history > 0 )
  {
    aui.start_index -= eisc->history;
    aui.has_active_child = true;
  }
  *** */

  /* return 0 if the interval has no active child (including len == 0) */
  if( !aui.has_active_child )
    return 0;

  /* include history_after if we are at the end */
  /* *** already included where relevant
  if( aui.stop_index == HaArrayCount(eisc->children) )
    aui.stop_index += eisc->history_after;
  *** */

  /* calculate the length and deviation */
  len = aui.stop_index - aui.start_index;
  return len < eisc->min_limit ? eisc->min_limit - len : 0;
}


/*****************************************************************************/
/*                                                                           */
/*  char *DrsAUIntervalShow(KHE_DRS_AU_INTERVAL aui)                         */
/*                                                                           */
/*  Return a display of aui in static memory.                                */
/*                                                                           */
/*****************************************************************************/

static char *DrsAUIntervalShow(KHE_DRS_AU_INTERVAL aui)
{
  static char buff[20];
  snprintf(buff, 20, "[%d, %d%s]", aui.start_index, aui.stop_index,
    aui.has_active_child ? "a" : "");
  return buff;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_COST KheDrsAUIntervalCost(KHE_DRS_AU_INTERVAL aui,                   */
/*    KHE_DRS_EXPR_INT_SEQ_COST eisc)                                        */
/*                                                                           */
/*  Return the cost of aui.                                                  */
/*                                                                           */
/*****************************************************************************/

static KHE_COST KheDrsAUIntervalCost(KHE_DRS_AU_INTERVAL aui,
  KHE_DRS_EXPR_INT_SEQ_COST eisc)
{
  int dev;
  dev = KheDrsAUIntervalDev(aui, eisc);
  return dev == 0 ? 0 : f(eisc, dev);
  /* ***
  return dev == 0 ? 0 : KheDrsCost(eisc->cost_fn, eisc->combined_weight, dev);
  *** */
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_INT_SEQ_COST - dominance testing and cost"       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprIntSeqCostDelta(KHE_DRS_EXPR_INT_SEQ_COST eisc,            */
/*    int lower_det, int upper_det)                                          */
/*                                                                           */
/*  Return the deviation of eisc for the given lower and upper determinants. */
/*  This is function delta(l, u) from the documentation.                     */
/*                                                                           */
/*****************************************************************************/

static int KheDrsExprIntSeqCostDelta(KHE_DRS_EXPR_INT_SEQ_COST eisc,
  int z, int lower_det, int upper_det)
{
  if( z == 0 )
    return 0;
  else if( lower_det > eisc->max_limit )
    return lower_det - eisc->max_limit;
  else if( upper_det < eisc->min_limit )
    return eisc->min_limit - upper_det;
  else
    return 0;
}


/*****************************************************************************/
/*                                                                           */
/*  int ci_bar(KHE_DRS_EXPR_INT_SEQ_COST eisc, int p, int x)                 */
/*                                                                           */
/*  Return the value of ci overbar as defined in the documentation.          */
/*                                                                           */
/*****************************************************************************/

static int ci_bar(KHE_DRS_EXPR_INT_SEQ_COST eisc, int p, int x)
{
  return x < p ? 0 : eisc->history_after;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_COST KheDrsExprIntSeqCostExtraCost(KHE_DRS_EXPR_INT_SEQ_COST eisc,   */
/*    int p, int q, int r, int l, int y, bool debug)                         */
/*                                                                           */
/*  Return the extra cost in the dominance testing formula with these        */
/*  attributes.                                                              */
/*                                                                           */
/*****************************************************************************/

static KHE_COST KheDrsExprIntSeqCostExtraCost(KHE_DRS_EXPR_INT_SEQ_COST eisc,
  int p, int q, int r, int l, int y, bool debug)
{
  int d1, d2;  KHE_COST res;
  d1 = KheDrsExprIntSeqCostDelta(eisc, l+y, l+y, l+y + ci_bar(eisc, p, y));
  d2 = KheDrsExprIntSeqCostDelta(eisc, l+r, l, l+q+r + ci_bar(eisc, p, q+r));
  res = f(eisc, d1) - f(eisc, d2);
  if( debug )
    fprintf(stderr, "    extra(p %d, q %d, r %d, l %d, y %d) = f(%d) %8.5f - "
      "f(%d) %8.5f = %8.5f\n", p, q, r, l, y, d1, KheCostShow(f(eisc, d1)),
      d2, KheCostShow(f(eisc, d2)), KheCostShow(res));
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_COST KheDrsExprIntSeqCostFindPsi(KHE_DRS_EXPR_INT_SEQ_COST eisc,     */
/*    int p, int q, int r, int l1, int l2)                                   */
/*                                                                           */
/*  Find psi, the available cost for eisc given p, q, r, l1, and l2.         */
/*  This implements the outer "min" of the formula in the documentation.     */
/*                                                                           */
/*****************************************************************************/
#define DEBUG_INT_SEQ_DOM(p, q, r, l1, l2)				\
  (DEBUG37 && (p) == 10 && (l1) == 10 && (l2) == 5)

static KHE_COST KheDrsExprIntSeqCostFindPsi(KHE_DRS_EXPR_INT_SEQ_COST eisc,
  int lower, int p, int q, int r, int l1, int l2)
{
  int y;  KHE_COST res, extra_cost1, extra_cost2, cost;  bool debug;
  HnAssert(lower <= p + q, "KheDrsExprIntSeqCostFindPsi internal error");
  debug = DEBUG_INT_SEQ_DOM(p, q, r, l1, l2);
  if( debug )
    fprintf(stderr, "[ KheDrsExprIntSeqCostFindPsi(%s, lower %d, p %d, q %d,"
      " r %d, l1 %d, l2 %d)\n", KheDrsMonitorInfoId(eisc->monitor_info),
      lower, p, q, r, l1, l2);
  res = KheCost(INT_MAX, INT_MAX);
  for( y = lower;  y <= q + r;  y++ )
  {
    extra_cost2 = KheDrsExprIntSeqCostExtraCost(eisc, p, q, r, l2, y, debug);
    extra_cost1 = KheDrsExprIntSeqCostExtraCost(eisc, p, q, r, l1, y, debug);
    cost = extra_cost2 - extra_cost1;
    if( debug )
      fprintf(stderr, "  y = %2d: %8.5f - %8.5f = %8.5f\n", y,
        KheCostShow(extra_cost2), KheCostShow(extra_cost1), KheCostShow(cost));
    if( cost < res )
      res = cost;
  }
  if( debug )
    fprintf(stderr, "] KheDrsExprIntSeqCostFindPsi returning %.5f\n",
      KheCostShow(res));
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_COST KheDrsExprIntSeqCostFindPsi0(KHE_DRS_EXPR_INT_SEQ_COST eisc,    */
/*    int p, int q, int r, int l1, int l2)                                   */
/*                                                                           */
/*  Find psi0, part of the available cost for eisc given p, q, r, l1, and    */
/*  This implements the outer "min" of the formula in the documentation.     */
/*                                                                           */
/*****************************************************************************/
#define DEBUG_INT_SEQ_DOM(p, q, r, l1, l2)				\
  (DEBUG37 && (p) == 10 && (l1) == 10 && (l2) == 5)

static KHE_COST KheDrsExprIntSeqCostFindPsi0(KHE_DRS_EXPR_INT_SEQ_COST eisc,
  int p, int q, int r, int l1, int l2)
{
  int y;  KHE_COST res, extra_cost1, extra_cost2;  bool debug;
  debug = DEBUG_INT_SEQ_DOM(p, q, r, l1, l2);
  if( debug )
    fprintf(stderr, "[ KheDrsExprIntSeqCostFindPsi0(%s, p %d, q %d,"
      " r %d, l1 %d, l2 %d)\n", KheDrsMonitorInfoId(eisc->monitor_info),
      p, q, r, l1, l2);
  res = KheCost(INT_MAX, INT_MAX);
  y = 0;
  extra_cost2 = KheDrsExprIntSeqCostExtraCost(eisc, p, q, r, l2, y, debug);
  extra_cost1 = KheDrsExprIntSeqCostExtraCost(eisc, p, q, r, l1, y, debug);
  res = extra_cost2 - extra_cost1;
  if( debug )
  {
    fprintf(stderr, "  y = %2d: %8.5f - %8.5f = %8.5f\n", y,
      KheCostShow(extra_cost2), KheCostShow(extra_cost1), KheCostShow(res));
    fprintf(stderr, "] KheDrsExprIntSeqCostFindPsi0 returning %.5f\n",
      KheCostShow(res));
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSeqCostSetDomTable(KHE_DRS_EXPR_INT_SEQ_COST eisc,     */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Set the dom_table field of eisc.                                         */
/*                                                                           */
/*****************************************************************************/
/* ***
 static KHE_DRS_DOM_TABLE_SEQ KheDrsDomTableSeqMake(HA_ARENA a);
static void KheDrsDomTableSeqPut(KHE_DRS_DOM_TABLE_SEQ dt, int p, int q,
  int r, int l1, int l2, KHE_COST psi, KHE_COST psi0,
  bool psi_plus_defined, KHE_COST psi_plus, HA_ARENA a);
*** */
static void KheDrsExprIntSeqCostDebugDomTable(KHE_DRS_EXPR_INT_SEQ_COST eisc,
  int verbosity, int indent, FILE *fp);

static void KheDrsExprIntSeqCostSetDomTable(KHE_DRS_EXPR_INT_SEQ_COST eisc,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int p, q, r, l1, l2, child_count;  KHE_COST psi, psi0, psi_plus;
  KHE_DRS_COST_TUPLE ct;  char *debug_str;  HA_ARENA a;
  debug_str = KheDrsMonitorInfoId(eisc->monitor_info);
  if( DEBUG42 )
    fprintf(stderr, "[ KheDrsExprIntSeqCostSetDomTable(%s)\n", debug_str);
  HnAssert(eisc->dom_table == NULL,
    "KheDrsExprIntSeqCostSetDomTable: dom_table");
  child_count = HaArrayCount(eisc->children);
  a = drs->arena;
  eisc->dom_table = KheDrsDim5TableMake(HnStringMake(a, "Seq %s",debug_str), a);
  /* eisc->dom_table = KheDrsDomTableSeqMake(a); */
  for( p = 0;  p <= child_count;  p++ )
    for( q = 0;  q <= p;  q++ )
      for( r = 0;  r <= p - q;  r++ )
	for( l1 = 0;  l1 <= child_count - p + eisc->history;  l1++ )
	  for( l2 = 0;  l2 <= child_count - p + eisc->history;  l2++ )
	  {
	    psi = KheDrsExprIntSeqCostFindPsi(eisc, 0, p, q, r, l1, l2);
	    psi0 = KheDrsExprIntSeqCostFindPsi0(eisc, p, q, r, l1, l2);
	    if( q == 0 )
	    {
	      ct = KheDrsCostTupleMake(psi, psi0, false, 0);
	      /* ***
	      KheDrsDomTableSeqPut(eisc->dom_table, p, q, r, l1, l2,
		psi, psi0, false, 0, a);
	      *** */
	    }
	    else
	    {
	      psi_plus = KheDrsExprIntSeqCostFindPsi(eisc, 1, p, q, r, l1, l2);
	      ct = KheDrsCostTupleMake(psi, psi0, true, psi_plus);
	      /* ***
	      KheDrsDomTableSeqPut(eisc->dom_table, p, q, r, l1, l2,
		psi, psi0, true, psi_plus, a);
	      *** */
	    }
	    KheDrsDim5TablePut(eisc->dom_table, p, q, r, l1, l2, ct, drs);
	  }
  if( DEBUG36 || DEBUG44_MONITOR(eisc->monitor_info->monitor) )
    KheDrsExprIntSeqCostDebugDomTable(eisc, 1, 0, stderr);
  if( DEBUG42 )
    fprintf(stderr, "] KheDrsExprIntSeqCostSetDomTable\n");
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsExprIntSeqCostWantDebugDomTable(                              */
/*    KHE_DRS_EXPR_INT_SEQ_COST eisc)                                        */
/*                                                                           */
/*  Return true if we want to debug eisc's dom table.                        */
/*                                                                           */
/*  At present we are doing this is ec's monitor is a resource monitor       */
/*  and the resource it monitors is the first resource of its type.          */
/*  This is to limit the debug output to a manageable amount.                */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsExprIntSeqCostWantDebugDomTable(
  KHE_DRS_EXPR_INT_SEQ_COST eisc)
{
  KHE_MONITOR m;  KHE_RESOURCE r;  KHE_RESOURCE_TYPE rt;

  /* find the resource */
  m = eisc->monitor_info->monitor;
  HnAssert(KheMonitorTag(m) == KHE_LIMIT_ACTIVE_INTERVALS_MONITOR_TAG,
    "KheDrsExprIntSeqCostWantDebugDomTable internal error");
  r = KheLimitActiveIntervalsMonitorResource(
    (KHE_LIMIT_ACTIVE_INTERVALS_MONITOR) m);

  /* return true if r is the first resource of its type */
  rt = KheResourceResourceType(r);
  return KheResourceTypeResourceCount(rt) > 0 &&
    KheResourceTypeResource(rt, 0) == r;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSeqCostDebugDomTable(KHE_DRS_EXPR_INT_SEQ_COST eisc,   */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug print of eisc's dom table.                                         */
/*                                                                           */
/*****************************************************************************/
/* ***
static void KheDrsDomTableSeqDebug(KHE_DRS_DOM_TABLE_SEQ dt, int verbosity,
  int indent, FILE *fp);
*** */

static void KheDrsExprIntSeqCostDebugDomTable(KHE_DRS_EXPR_INT_SEQ_COST eisc,
  int verbosity, int indent, FILE *fp)
{
  if( KheDrsExprIntSeqCostWantDebugDomTable(eisc) )
  {
    fprintf(fp, "%*s[ ExprCost for %s\n", indent, "",
      KheDrsMonitorInfoId(eisc->monitor_info));
    fprintf(fp, "%*s  cost_fn %s, combined_weight %.5f, min_limit %d,"
      " max_limit %d\n", indent, "",
      KheCostFunctionShow(eisc->cost_fn), KheCostShow(eisc->combined_weight),
      eisc->min_limit, eisc->max_limit);
    fprintf(fp, "%*s  history_before %d, history_after %d, history %d\n",
      indent, "", eisc->history_before, eisc->history_after, eisc->history);
    KheDrsDim5TableDebug(eisc->dom_table, verbosity, indent + 2, fp);
    /* KheDrsDomTableSeqDebug(eisc->dom_table, verbosity, indent + 2, fp); */
    fprintf(fp, "%*s]\n", indent, "");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR_INT_SEQ_COST"                                    */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSeqCostSetSeqType(KHE_DRS_EXPR_INT_SEQ_COST eisc,      */
/*    KHE_FRAME days_frame)                                                  */
/*                                                                           */
/*  Set eisc->seq_type and eisc->seq_index if eisc matches days_frame.       */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntSeqCostSetSeqType(KHE_DRS_EXPR_INT_SEQ_COST eisc,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  bool found_positive, found_negative, found_single;
  KHE_DRS_EXPR child_e, child_child_e;  int i, index, pos;
  KHE_DRS_EXPR_BUSY_DAY busy_day_e;  KHE_DRS_EXPR_FREE_DAY free_day_e;
  KHE_DRS_EXPR_BUSY_TIME busy_time_e;  KHE_TIME_GROUP frame_tg;

  /* fail immediately if the number of children is wrong */
  if( HaArrayCount(eisc->children) != KheFrameTimeGroupCount(drs->days_frame) )
    return;

  /* compare time groups and check all positive, negative, or positive single */
  found_positive = found_negative = found_single = false;
  index = -1;
  HaArrayForEach(eisc->children, child_e, i)
  {
    frame_tg = KheFrameTimeGroup(drs->days_frame, i);

    if( child_e->tag == KHE_DRS_EXPR_OR_TAG )
    {
      /* if not a single child, fail */
      if( HaArrayCount(child_e->children) != 1 )
	return;

      child_child_e = HaArrayFirst(child_e->children);
      if( child_child_e->tag == KHE_DRS_EXPR_BUSY_DAY_TAG )
      {
	/* this is a positive, so if not all positive, fail */
	if( found_negative || found_single )
	  return;
	found_positive = true;

	/* if time groups don't match, fail */
	busy_day_e = (KHE_DRS_EXPR_BUSY_DAY) child_child_e;
	if( !KheTimeGroupEqual(busy_day_e->time_group, frame_tg) )
	  return;
      }
      else if( child_child_e->tag == KHE_DRS_EXPR_BUSY_TIME_TAG )
      {
	/* this is a single positive, so if not all single, fail */
	if( found_negative || found_positive )
	  return;
	found_single = true;

	/* its time has to be an element of frame_tg */
	busy_time_e = (KHE_DRS_EXPR_BUSY_TIME) child_child_e;
	if( !KheTimeGroupContains(frame_tg, busy_time_e->time, &pos) )
	  return;

	/* its time has to have the same index as the other singles */
	if( index == -1 )
	  index = pos;
	else if( pos != index )
	  return;
      }
      else
	return;
    }
    else if( child_e->tag == KHE_DRS_EXPR_AND_TAG )
    {
      /* this is negative, so if not all negative, fail */
      if( found_positive || found_single )
	return;
      found_negative = true;

      /* if not a single child, fail */
      if( HaArrayCount(child_e->children) != 1 )
	return;

      /* if not a free day child, fail */
      child_child_e = HaArrayFirst(child_e->children);
      if( child_child_e->tag != KHE_DRS_EXPR_FREE_DAY_TAG )
	return;

      /* if time groups don't match, fail */
      free_day_e = (KHE_DRS_EXPR_FREE_DAY) child_child_e;
      if( !KheTimeGroupEqual(free_day_e->time_group, frame_tg) )
	return;
    }
    else
    {
      HnAbort("KheDrsExprIntSeqCostSetSeqType internal error 1");
      return;  /* keep compiler happy */
    }
  }

  /* all good, so wrap up */
  if( found_positive )
    eisc->seq_type = KHE_DRS_SEQ_DAYS_POSITIVE, eisc->seq_index = -1;
  else if( found_negative )
    eisc->seq_type = KHE_DRS_SEQ_DAYS_NEGATIVE, eisc->seq_index = -1;
  else if( found_single )
    eisc->seq_type = KHE_DRS_SEQ_SINGLE_POSITIVE, eisc->seq_index = index;
  else
    return;

  /* if we get to this point, success */
  if( DEBUG54 )
  {
    fprintf(stderr, "[ %s matches frame:\n",
      KheDrsMonitorInfoId(eisc->monitor_info));
    KheDrsExprDebug((KHE_DRS_EXPR) eisc, drs, 1, 2, stderr);
    fprintf(stderr, "]\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_EXPR_INT_SEQ_COST KheDrsExprIntSeqCostMake(                      */
/*    KHE_DRS_MONITOR_INFO mi, KHE_COST_FUNCTION cost_fn,                    */
/*    KHE_COST combined_weight, int min_limit, int max_limit,                */
/*    int history_before, int history_after, int history,                    */
/*    KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs)                  */
/*                                                                           */
/*  Make a new KHE_DRS_EXPR_INT_SEQ_COST object with these attributes.       */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_EXPR_INT_SEQ_COST KheDrsExprIntSeqCostMake(
  KHE_DRS_MONITOR_INFO mi, KHE_COST_FUNCTION cost_fn,
  KHE_COST combined_weight, int min_limit, int max_limit,
  int history_before, int history_after, int history,
  KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_EXPR_INT_SEQ_COST res;
  HaMake(res, drs->arena);
  KheDrsExprInitBegin((KHE_DRS_EXPR) res, KHE_DRS_EXPR_INT_SEQ_COST_TAG,
    dr, drs);
  res->u.int_val = UNDEF_INT;
  res->monitor_info = mi;
  res->seq_type = KHE_DRS_SEQ_NONE;  /* may change later */
  res->seq_index = 0;
  res->cost_fn = cost_fn;
  res->combined_weight = combined_weight;
  res->min_limit = min_limit;
  res->max_limit = max_limit;
  res->adjust_type = KheDrsAdjustType(cost_fn, max_limit);
  res->history_before = history_before;
  res->history_after = history_after;
  res->history = history;
  res->dom_table = NULL;  /* reset later */
  res->first_open_child_index = 0;  /* ready for when we open */
  HaArrayInit(res->closed_seqs, drs->arena);
  HaArrayAddLast(res->closed_seqs, KheDrsClosedSeqMakeInit(drs));
  KheDrsAddWeight(drs, combined_weight);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSeqCostAddChild(KHE_DRS_EXPR_INT_SEQ_COST eisc,        */
/*    KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  New child child_e has just been added to eisc.  Update eisc's state      */
/*  appropriately.  That is, extend the sole closed sequence one place.      */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntSeqCostAddChild(KHE_DRS_EXPR_INT_SEQ_COST eisc,
  KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_CLOSED_SEQ dcs;
  HnAssert(HaArrayCount(eisc->closed_seqs) == 1,
    "KheDrsExprIntSeqCostAddChild internal error (%d closed seqs)",
    HaArrayCount(eisc->closed_seqs));
  dcs = HaArrayFirst(eisc->closed_seqs);
  KheDrsClosedSeqExtendToRight(dcs, child_e->u.int_val);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSeqCostClosedSeqDebug(KHE_DRS_EXPR_INT_SEQ_COST eisc,  */
/*    int verbosity, int indent, FILE *fp)                                   */
/*                                                                           */
/*  Debug printf of eisc's closed sequences.                                 */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntSeqCostClosedSeqDebug(KHE_DRS_EXPR_INT_SEQ_COST eisc,
  int verbosity, int indent, FILE *fp)
{
  KHE_DRS_CLOSED_SEQ dcs;  int i;
  HaArrayForEach(eisc->closed_seqs, dcs, i)
    KheDrsClosedSeqDebug(dcs, verbosity, indent, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSeqCostChildHasOpened(KHE_DRS_EXPR_INT_SEQ_COST eisc,  */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  Inform internal expression eisc that its child_index'th child, child_e,  */
/*  has opened.  The child will have its correct closed_state and value      */
/*  fields when this is called, and it will have been added to the end of    */
/*  eisc->open_children.                                                     */
/*                                                                           */
/*****************************************************************************/

#define DEBUG_RERUN(eisc) (DEBUG14 &&					  \
  strcmp(KheDrsMonitorInfoId(eisc->monitor_info), RERUN_MONITOR_ID) == 0)

static void KheDrsExprIntSeqCostChildHasOpened(KHE_DRS_EXPR_INT_SEQ_COST eisc,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_CLOSED_SEQ dcs, new_dcs;  KHE_DRS_EXPR e;  int i, open_index;
  KHE_DRS_AU_INTERVAL aui_left, aui_right, aui_merged, aui_before, aui_after;
  KHE_DRS_A_INTERVAL ai_left, ai_right, ai_right_before, ai_merged;

  /* find the index of child_e in eisc->open_children (easy: at the end) */
  open_index = KheDrsOpenChildrenCount(&eisc->open_children_by_day) - 1;
  if( DEBUG_RERUN(eisc) )
    fprintf(stderr, "[ KheDrsExprIntSeqCostChildHasOpened(%s, child_e, %d)\n",
      KheDrsMonitorInfoId(eisc->monitor_info), child_index);

  /* split the last closed seq at child_index */
  dcs = HaArrayLast(eisc->closed_seqs);
  new_dcs = KheDrsClosedSeqSplit(dcs, child_index, eisc, drs);
  HaArrayAddLast(eisc->closed_seqs, new_dcs);
  if( DEBUG_RERUN(eisc) )
  {
    KheDrsExprIntSeqCostClosedSeqDebug(eisc, 2, 2, stderr);
    HaArrayForEach(eisc->children, e, i)
      KheDrsExprDebug(e, drs, 2, 4, stderr);
  }

  /* update the au-intervals and a-intervals to take account of the change */
  if( child_e->u.int_val == 0 )
  {
    /* the opened child moves from inactive to unassigned */
    /* the au-intervals on each side merge */
    aui_left = KheDrsAUIntervalFindLeft(eisc, open_index);
    aui_right = KheDrsAUIntervalFindRight(eisc, open_index, drs);
    aui_merged = KheDrsAUIntervalMerge(aui_left, aui_right, false);
    drs->solve_start_cost += KheDrsAUIntervalCost(aui_merged, eisc)
      - KheDrsAUIntervalCost(aui_left, eisc)
      - KheDrsAUIntervalCost(aui_right, eisc);
    KheDrsMonitorInfoUpdateRerunCost(eisc->monitor_info, drs, NULL,
      KHE_DRS_OPEN, "open0a", child_index, "+--",
      KheDrsAUIntervalCost(aui_merged, eisc),
      KheDrsAUIntervalCost(aui_left, eisc),
      KheDrsAUIntervalCost(aui_right, eisc));
    /* ***
    KheDrsUpdateRerunCost(drs, &eisc->rerun_open_and_search_cost,
      &eisc->rerun_open_and_close_cost, eisc->monitor, "open0a", child_index,
      "+--", KheDrsAUIntervalCost(aui_merged, eisc),
      KheDrsAUIntervalCost(aui_left, eisc),
      KheDrsAUIntervalCost(aui_right, eisc));
    *** */

    /* the a-interval to the right changes its unassigned_precedes */
    if( eisc->cost_fn == KHE_STEP_COST_FUNCTION )
    {
      ai_right_before = KheDrsAIntervalFindRight(eisc, open_index, false);
      ai_right        = KheDrsAIntervalFindRight(eisc, open_index, true);
      drs->solve_start_cost += KheDrsAIntervalCost(ai_right, eisc)
	- KheDrsAIntervalCost(ai_right_before, eisc);
      KheDrsMonitorInfoUpdateRerunCost(eisc->monitor_info, drs, NULL,
	KHE_DRS_OPEN, "open0b", child_index, "+-",
	KheDrsAIntervalCost(ai_right, eisc),
	KheDrsAIntervalCost(ai_right_before, eisc));
      /* ***
      KheDrsUpdateRerunCost(drs, &eisc->rerun_open_and_search_cost,
        &eisc->rerun_open_and_close_cost, eisc->monitor, "open0b", child_index,
	"+-", KheDrsAIntervalCost(ai_right, eisc),
        KheDrsAIntervalCost(ai_right_before, eisc));
      *** */
    }
  }
  else
  {
    /* the opened child moves from active to unassigned */
    /* the enclosing au-interval is unchanged, but its cost may change */
    aui_left = KheDrsAUIntervalFindLeft(eisc, open_index);
    aui_right = KheDrsAUIntervalFindRight(eisc, open_index, drs);
    aui_before = KheDrsAUIntervalMerge(aui_left, aui_right, true);
    aui_after = KheDrsAUIntervalMerge(aui_left, aui_right, false);
    drs->solve_start_cost += KheDrsAUIntervalCost(aui_after, eisc)
      - KheDrsAUIntervalCost(aui_before, eisc);
    KheDrsMonitorInfoUpdateRerunCost(eisc->monitor_info, drs, NULL,
      KHE_DRS_OPEN, "open1a", child_index, "+-",
      KheDrsAUIntervalCost(aui_after, eisc),
      KheDrsAUIntervalCost(aui_before, eisc));
    /* ***
    KheDrsUpdateRerunCost(drs, &eisc->rerun_open_and_search_cost,
      &eisc->rerun_open_and_close_cost, eisc->monitor, "open1a", child_index,
      "+-", KheDrsAUIntervalCost(aui_after, eisc),
      KheDrsAUIntervalCost(aui_before, eisc));
    *** */

    /* the enclosing a-interval splits */
    /* NB unassigned_precedes is correct when ai_left or ai_right is empty */
    ai_left = KheDrsAIntervalFindLeft(eisc, open_index);
    ai_right = KheDrsAIntervalFindRight(eisc, open_index, true);
    ai_merged = KheDrsAIntervalMerge(ai_left, ai_right);
    drs->solve_start_cost += KheDrsAIntervalCost(ai_left, eisc)
      + KheDrsAIntervalCost(ai_right, eisc)
      - KheDrsAIntervalCost(ai_merged, eisc);
    KheDrsMonitorInfoUpdateRerunCost(eisc->monitor_info, drs, NULL,
      KHE_DRS_OPEN, "open1b", child_index, "++-",
      KheDrsAIntervalCost(ai_left, eisc),
      KheDrsAIntervalCost(ai_right, eisc),
      KheDrsAIntervalCost(ai_merged, eisc));
    /* ***
    KheDrsUpdateRerunCost(drs, &eisc->rerun_open_and_search_cost,
      &eisc->rerun_open_and_close_cost, eisc->monitor, "open1b", child_index,
      "++-", KheDrsAIntervalCost(ai_left, eisc),
      KheDrsAIntervalCost(ai_right, eisc),
      KheDrsAIntervalCost(ai_merged, eisc));
    *** */
  }
  if( DEBUG_RERUN(eisc) )
    fprintf(stderr, "] KheDrsExprIntSeqCostChildHasOpened\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSeqCostChildHasClosed(KHE_DRS_EXPR_INT_SEQ_COST eisc,  */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  Inform internal expression eisc that its child_index'th child, child_e,  */
/*  has closed.  The child will have its correct closed_state and value      */
/*  fields when this is called; but it will not have been removed from       */
/*  e->open_children.                                                        */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntSeqCostChildHasClosed(KHE_DRS_EXPR_INT_SEQ_COST eisc,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int open_index;  KHE_DRS_CLOSED_SEQ dcs1, dcs2;
  KHE_DRS_AU_INTERVAL aui_left, aui_right, aui_merged, aui_before, aui_after;
  KHE_DRS_A_INTERVAL ai_left, ai_right, ai_right_before, ai_merged;

  /* find the index of child_e in eisc->open_children */
  open_index = eisc->first_open_child_index;
  HnAssert(HaArray(eisc->open_children_by_day.children, open_index)
    == child_e, "KheDrsExprIntSeqCostChildHasClosed internal error 1");
  eisc->first_open_child_index++;

  /* update the au-intervals and a-intervals to take account of the change */
  if( child_e->u.int_val == 0 )
  {
    /* the closed child moves from unassigned to inactive */
    /* the enclosing au-interval splits */
    aui_left = KheDrsAUIntervalFindLeft(eisc, 0 /* not open_index */);
    aui_right = KheDrsAUIntervalFindRight(eisc, open_index, drs);
    aui_merged = KheDrsAUIntervalMerge(aui_left, aui_right, false);
    drs->solve_start_cost += KheDrsAUIntervalCost(aui_left, eisc)
      + KheDrsAUIntervalCost(aui_right, eisc)
      - KheDrsAUIntervalCost(aui_merged, eisc);
    KheDrsMonitorInfoUpdateRerunCost(eisc->monitor_info, drs, NULL,
      KHE_DRS_CLOSE, "close0a", child_index, "++-",
      KheDrsAUIntervalCost(aui_left, eisc),
      KheDrsAUIntervalCost(aui_right, eisc),
      KheDrsAUIntervalCost(aui_merged, eisc));
    /***
    KheDrsUpdateRerunCost(drs, &eisc->rerun_open_and_close_cost, NULL,
      eisc->monitor, "close0a", child_index, "++-",
      KheDrsAUIntervalCost(aui_left, eisc),
      KheDrsAUIntervalCost(aui_right, eisc),
      KheDrsAUIntervalCost(aui_merged, eisc));
    *** */

    /* the a-interval to the right changes its unassigned_precedes */
    ai_right_before = KheDrsAIntervalFindRight(eisc, open_index, true);
    ai_right        = KheDrsAIntervalFindRight(eisc, open_index, false);
    drs->solve_start_cost += KheDrsAIntervalCost(ai_right, eisc)
      - KheDrsAIntervalCost(ai_right_before, eisc);
    KheDrsMonitorInfoUpdateRerunCost(eisc->monitor_info, drs, NULL,
      KHE_DRS_CLOSE, "close0b", child_index, "+-",
      KheDrsAIntervalCost(ai_right, eisc),
      KheDrsAIntervalCost(ai_right_before, eisc));
    /* ***
    KheDrsUpdateRerunCost(drs, &eisc->rerun_open_and_close_cost, NULL,
      eisc->monitor, "close0b", child_index, "+-",
      KheDrsAIntervalCost(ai_right, eisc),
      KheDrsAIntervalCost(ai_right_before, eisc));
    *** */
  }
  else
  {
    /* the closed child moves from unassigned to active */
    /* the enclosing au-interval is unchanged, but its cost may change */
    aui_left = KheDrsAUIntervalFindLeft(eisc, 0 /* not open_index */);
    aui_right = KheDrsAUIntervalFindRight(eisc, open_index, drs);
    aui_before = KheDrsAUIntervalMerge(aui_left, aui_right, false);
    aui_after = KheDrsAUIntervalMerge(aui_left, aui_right, true);
    drs->solve_start_cost += KheDrsAUIntervalCost(aui_after, eisc)
      - KheDrsAUIntervalCost(aui_before, eisc);
    KheDrsMonitorInfoUpdateRerunCost(eisc->monitor_info, drs, NULL,
      KHE_DRS_CLOSE, "close1a", child_index, "+-",
      KheDrsAUIntervalCost(aui_after, eisc),
      KheDrsAUIntervalCost(aui_before, eisc));
    /* ***
    KheDrsUpdateRerunCost(drs, &eisc->rerun_open_and_close_cost, NULL,
      eisc->monitor, "close1a", child_index, "+-",
      KheDrsAUIntervalCost(aui_after, eisc),
      KheDrsAUIntervalCost(aui_before, eisc));
    *** */

    /* the a-intervals on each side merge */
    ai_left = KheDrsAIntervalFindLeft(eisc, 0 /* not open_index */);
    ai_right = KheDrsAIntervalFindRight(eisc, open_index, true);
    ai_merged = KheDrsAIntervalMerge(ai_left, ai_right);
    drs->solve_start_cost += KheDrsAIntervalCost(ai_merged, eisc)
      - KheDrsAIntervalCost(ai_left, eisc)
      - KheDrsAIntervalCost(ai_right, eisc);
    KheDrsMonitorInfoUpdateRerunCost(eisc->monitor_info, drs, NULL,
      KHE_DRS_CLOSE, "close1b", child_index, "+--",
      KheDrsAIntervalCost(ai_merged, eisc), KheDrsAIntervalCost(ai_left, eisc),
      KheDrsAIntervalCost(ai_right, eisc));
    /* ***
    KheDrsUpdateRerunCost(drs, &eisc->rerun_open_and_close_cost, NULL,
      eisc->monitor, "close1b", child_index, "+--",
      KheDrsAIntervalCost(ai_merged, eisc),
      KheDrsAIntervalCost(ai_left, eisc), KheDrsAIntervalCost(ai_right, eisc));
    *** */
  }

  /* merge the two relevant closed seqs */
  dcs1 = HaArray(eisc->closed_seqs, 0);
  dcs2 = HaArray(eisc->closed_seqs, open_index + 1);
  KheDrsClosedSeqMerge(dcs1, dcs2, eisc, drs);
  HaArrayPut(eisc->closed_seqs, open_index + 1, NULL);  /* defensive */
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSeqCostSetClosedValue(KHE_DRS_EXPR_INT_SEQ_COST eisc,  */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing eisc, set its value suitably for its closed state.    */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntSeqCostSetClosedValue(KHE_DRS_EXPR_INT_SEQ_COST eisc,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  /* clear out eisc->first_open_child_index and eisc->closed_seqs */
  eisc->first_open_child_index = 0;
  while( HaArrayCount(eisc->closed_seqs) > 0 &&
      HaArrayLast(eisc->closed_seqs) == NULL )
    HaArrayDeleteLast(eisc->closed_seqs);
  HnAssert(HaArrayCount(eisc->closed_seqs) == 1,
    "KheDrsExprIntSeqCostSetClosedValue internal error 1");

  /* nothing else to do here, since eisc has a cost but no value */
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprIntSeqCostChildDomTest(                       */
/*    KHE_DRS_EXPR_INT_SEQ_COST eisc, int open_day_index,                    */
/*    KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  Return a dom test suited to a child of eisc.                             */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DOM_TEST KheDrsExprIntSeqCostChildDomTest(
  KHE_DRS_EXPR_INT_SEQ_COST eisc, int open_day_index,
  KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int a, b;

  /* calculate a */
  a = (eisc->max_limit < INT_MAX ? -1 : INT_MAX);

  /* calculate b */
  b = (eisc->min_limit > 0 ? INT_MAX : 0);

  /* make and return the dom test */
  return KheDrsDomTestMake(KHE_DRS_DOM_TEST_STRONG, child_e, false, 0,
    INT_MAX, a, b, /* false, false, */ 0, NULL, NULL, NULL, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  int KheDrsExprIntSeqCostOpenDayIndexToChildIndex(                        */
/*    KHE_DRS_EXPR_INT_SEQ_COST eisc, int open_day_index)                    */
/*                                                                           */
/*  Convert open_day_index to the index of the first open child whose        */
/*  last open day is that day, and from there to the index in the            */
/*  full list of children of that open child.                                */
/*                                                                           */
/*****************************************************************************/

static int KheDrsExprIntSeqCostOpenDayIndexToChildIndex(
  KHE_DRS_EXPR_INT_SEQ_COST eisc, int open_day_index)
{
  int open_child_index;  KHE_DRS_CLOSED_SEQ dcs;
  open_child_index = KheDrsOpenChildrenBefore(&eisc->open_children_by_day,
    open_day_index);
  HnAssert(open_child_index <
    KheDrsOpenChildrenCount(&eisc->open_children_by_day),
    "KheDrsExprIntSeqCostOpenDayIndexToChildIndex internal error");
  dcs = HaArray(eisc->closed_seqs, open_child_index);
  return dcs->stop_index;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprIntSeqCostDayDomTest(                         */
/*    KHE_DRS_EXPR_INT_SEQ_COST eisc, int open_day_index,                    */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Return a dom test suited to eisc on the open day with this index.        */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DOM_TEST KheDrsExprIntSeqCostDayDomTest(
  KHE_DRS_EXPR_INT_SEQ_COST eisc, int open_day_index,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int a, b, p, q, r, t, index;  /* KHE_DRS_DOM_TABLE_SUB dts; */
  KHE_DRS_DOM_TEST_TYPE dom_test_type;  KHE_DRS_DIM2_TABLE d2;

  /* calculate a */
  if( eisc->max_limit < INT_MAX )
  {
    t = KheDrsOpenChildrenAtOrAfter(&eisc->open_children_by_day,open_day_index);
    a = eisc->max_limit - t;
  }
  else
    a = INT_MAX;

  /* calculate b */
  if( eisc->min_limit > 0 )
    b = eisc->min_limit;
  else
    b = 0;

  /* sort out uniform dominance test */
  index = KheDrsExprIntSeqCostOpenDayIndexToChildIndex(eisc, open_day_index+1);
  p = HaArrayCount(eisc->children) - index;
  KheDrsAUIntervalFindRightLengths(eisc, open_day_index + 1, &q, &r);

  /* sort out uniform dominance test (old) */
  /* ***
  t = KheDrsOpenChildrenBefore(&eisc->open_children_by_day, open_child_index+1);
  if( t == HaArrayCount(eisc->open_children) )
  {
    ** right part of cut is empty **
    HnAbort("KheDrsExprIntSeqCostDomTest: unexpected open_child_index (%d)",
      open_child_index);
    p = q = r = 0;  ** keep compiler happy **
  }
  else
  {
    ** find x, the first unassigned open child after open_child_index **
    ** and index, its index in eisc's list of children **
    x = HaArray(eisc->open_children, t);
    index = -1;
    HaArrayForEach(x->parents, prnt, i)
      if( prnt.expr == (KHE_DRS_EXPR) eisc )
      {
	index = prnt.index;
	break;
      }
    HnAssert(index >= 0, "KheDrsExprIntSeqCostDomTest internal error");

    ** find p, the number of current interval children at or after x **
    p = HaArrayCount(eisc->children) - index;

    ** find q and r **
    KheDrsAUIntervalFindRightLengths(eisc, open_child_index + 1, &q, &r);
  }
  *** */

  /* make and return the dom test */
  d2 = KheDrsDim5TableGet3(eisc->dom_table, p, q, r);
  /* dts = KheDrsDomTableSeqGet(eisc->dom_table, p, q, r); */
  if( q < 0 || DEBUG44_MONITOR(eisc->monitor_info->monitor) )
  {
    fprintf(stderr, "[ KheDrsExprIntSeqCostDomTest(%s, %s, drs)\n",
      KheMonitorId(eisc->monitor_info->monitor),
      KheDrsDayId(HaArray(drs->open_days, open_day_index)));
    fprintf(stderr, "  p = %d, q = %d, r = %d, dts:\n", p, q, r);
    /* KheDrsDomTableSubDebug(dts, 2, 2, stderr); */
    KheDrsDim2TableDebug(d2, 2, 2, stderr);
    fprintf(stderr, "]\n");
  }
  HnAssert(q >= 0, "KheDrsExprIntSeqCostDayDomTest internal error (q = %d)", q);
  dom_test_type = KheDomTestTypeActual(drs->solve_dom_test_type,
    eisc->cost_fn != KHE_QUADRATIC_COST_FUNCTION);
  return KheDrsDomTestMake(dom_test_type, (KHE_DRS_EXPR) eisc,
    false, eisc->min_limit, eisc->max_limit, a, b, eisc->combined_weight,
    /* eisc->cost_fn != KHE_QUADRATIC_COST_FUNCTION, true, */
    /* dts, */ d2, NULL, eisc->monitor_info->monitor, drs);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsDebugActiveLen(KHE_DRS_EXPR_INT_SEQ_COST eisc, char *label,   */
/*    KHE_DRS_CLOSED_SEQ dcs, int active_len,KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  Debug these values if wanted.                                            */
/*                                                                           */
/*****************************************************************************/
/* ***
static void KheDrsExprDebugSoln(KHE_DRS_EXPR e, KHE_DRS_SOLN soln,
  int verbosity, int indent, FILE *fp);
*** */

static void KheDrsDebugActiveLen(KHE_DRS_EXPR_INT_SEQ_COST eisc, char *label,
  int open_index, KHE_DRS_EXPR child_e, KHE_DRS_SIGNATURE prev_sig,
  KHE_DRS_CLOSED_SEQ dcs, int active_len, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  if( RERUN && drs->rerun != NULL &&
      strcmp(KheDrsMonitorInfoId(eisc->monitor_info), RERUN_MONITOR_ID) == 0 )
  {
    fprintf(stderr, "  [ active_len %s: ", label);
    KheDrsClosedSeqDebug(dcs, 1, -1, stderr);
    fprintf(stderr, " --> %d\n", active_len);
    fprintf(stderr, "    child at open_index %d:", open_index);
    /* KheDrsExprDebugSoln(child_e, prev_soln, 1, 2, stderr); */
    KheDrsExprDebug(child_e, drs, 1, 2, stderr);
    fprintf(stderr, "  ]\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSeqCostEvalDay(KHE_DRS_EXPR_INT_SEQ_COST eisc,         */
/*    KHE_DRS_SOLN prev_soln, KHE_DRS_SOLN next_soln,                        */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Evaluate expression eisc in next_soln.                                   */
/*                                                                           */
/*  Implementation note.  Much of this code is the same as the code for      */
/*  closing each child_e, except that the a-interval and au-interval at      */
/*  left both have length active_len, and the cost is updated in             */
/*  next_soln->cost rather than in drs->solve_start_cost.                    */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsExprIntSeqCostEvalDay(KHE_DRS_EXPR_INT_SEQ_COST eisc,
  KHE_DRS_SOLN prev_soln, KHE_DRS_SOLN next_soln,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int open_index, i1, i2, active_len, di, adjusted_len;  KHE_DRS_EXPR child_e;
  KHE_DRS_AU_INTERVAL aui_left, aui_right, aui_merged;
  KHE_DRS_AU_INTERVAL aui_before, aui_after;
  KHE_DRS_A_INTERVAL ai_right_before, ai_right_after;
  KHE_DRS_A_INTERVAL ai_left, ai_right, ai_merged;
  KHE_DRS_CLOSED_SEQ dcs;

  ** get di, the open day index of next_soln **
  di = KheDrsSolnOpenDayIndex(next_soln);

  ** initialize active_len **
  if( KheDrsOpenChildrenIndexIsFirst(&eisc->open_children_by_day, di) )
  {
    ** first day **
    dcs = HaArrayFirst(eisc->closed_seqs);
    active_len = dcs->active_at_right;
    if( KheDrsClosedSeqAllActive(dcs) )
      active_len += eisc->history;
    if( DEBUG_LAST_EDGE )
      fprintf(stderr, "  _%d", active_len);
  }
  else
  {
    ** not first day, so retrieve previous active_len **
    active_len = KheDrsExprDaySigVal((KHE_DRS_EXPR) eisc, di - 1, prev_soln);
    if( DEBUG_LAST_EDGE )
      fprintf(stderr, "  $%d", active_len);
  }

  ** handle each child_e whose last open day is di **
  KheDrsOpenChildrenForEach(&eisc->open_children_by_day, di,
    child_e, open_index)
  {
    if( child_e->u.int_val == 0 )
    {
      ** child_e moves from unassigned to inactive: update cost **
      ** the enclosing au-interval splits **
      aui_left = KheDrsAUIntervalMakeLeft(eisc, open_index, active_len);
      aui_right = KheDrsAUIntervalFindRight(eisc, open_index, drs);
      aui_merged = KheDrsAUIntervalMerge(aui_left, aui_right, false);
      next_soln->cost += KheDrsAUIntervalCost(aui_left, eisc)
	+ KheDrsAUIntervalCost(aui_right, eisc)
	- KheDrsAUIntervalCost(aui_merged, eisc);
      KheDrsUpdateRerunCost(drs, &eisc->rerun_open_and_search_cost, NULL,
	eisc->monitor, "search", -1, "++-",
	KheDrsAUIntervalCost(aui_left, eisc),
	KheDrsAUIntervalCost(aui_right, eisc),
	KheDrsAUIntervalCost(aui_merged, eisc));
      if( DEBUG_LAST_EDGE )
	fprintf(stderr, " 0aui +%.5f +%.5f -%.5f",
	  KheCostShow(KheDrsAUIntervalCost(aui_left, eisc)),
          KheCostShow(KheDrsAUIntervalCost(aui_right, eisc)),
          KheCostShow(KheDrsAUIntervalCost(aui_merged, eisc)));

      ** the a-interval to the right changes its unassigned_precedes **
      ai_right_before = KheDrsAIntervalFindRight(eisc, open_index, true);
      ai_right_after  = KheDrsAIntervalFindRight(eisc, open_index, false);
      next_soln->cost += KheDrsAIntervalCost(ai_right_after, eisc)
	- KheDrsAIntervalCost(ai_right_before, eisc);
      KheDrsUpdateRerunCost(drs, &eisc->rerun_open_and_search_cost, NULL,
	eisc->monitor, "search2", -1, "+-",
        KheDrsAIntervalCost(ai_right_after, eisc),
        KheDrsAIntervalCost(ai_right_before, eisc));
      if( DEBUG_LAST_EDGE )
	fprintf(stderr, " 0ai +%.5f -%.5f",
	  KheCostShow(KheDrsAIntervalCost(ai_right_after, eisc)),
          KheCostShow(KheDrsAIntervalCost(ai_right_before, eisc)));

      ** set active_len for next iteration (child_e is now inactive) **
      dcs = HaArray(eisc->closed_seqs, open_index + 1);
      active_len = dcs->active_at_right;
      KheDrsDebugActiveLen(eisc, "inactive", open_index, child_e, prev_soln,
	dcs, active_len, drs);
      if( DEBUG_LAST_EDGE )
	fprintf(stderr, " %d", active_len);
    }
    else
    {
      ** child_e moves from unassigned to active: update cost **
      ** the enclosing au-interval is unchanged, but its cost may change **
      aui_left = KheDrsAUIntervalMakeLeft(eisc, open_index, active_len);
      aui_right = KheDrsAUIntervalFindRight(eisc, open_index, drs);
      aui_before = KheDrsAUIntervalMerge(aui_left, aui_right, false);
      aui_after = KheDrsAUIntervalMerge(aui_left, aui_right, true);
      next_soln->cost += KheDrsAUIntervalCost(aui_after, eisc)
	- KheDrsAUIntervalCost(aui_before, eisc);
      KheDrsUpdateRerunCost(drs, &eisc->rerun_open_and_search_cost, NULL,
	eisc->monitor, "search", -1, "+-",
        KheDrsAUIntervalCost(aui_after, eisc),
	KheDrsAUIntervalCost(aui_before, eisc));
      if( DEBUG_LAST_EDGE )
	fprintf(stderr, " 1aui +%.5f -%.5f",
	  KheCostShow(KheDrsAUIntervalCost(aui_after, eisc)),
          KheCostShow(KheDrsAUIntervalCost(aui_before, eisc)));

      ** the a-intervals on each side merge **
      ai_left = KheDrsAIntervalMakeLeft(eisc, open_index, active_len);
      ai_right = KheDrsAIntervalFindRight(eisc, open_index, true);
      ai_merged = KheDrsAIntervalMerge(ai_left, ai_right);
      next_soln->cost += KheDrsAIntervalCost(ai_merged, eisc)
	- KheDrsAIntervalCost(ai_left, eisc)
	- KheDrsAIntervalCost(ai_right, eisc);
      KheDrsUpdateRerunCost(drs, &eisc->rerun_open_and_search_cost, NULL,
	eisc->monitor, "search2", -1, "+--",
        KheDrsAIntervalCost(ai_merged, eisc),
	KheDrsAIntervalCost(ai_left, eisc),
	KheDrsAIntervalCost(ai_right, eisc));
      if( DEBUG_LAST_EDGE )
	fprintf(stderr, " 1ai +%.5f -%.5f -%.5f",
	  KheCostShow(KheDrsAIntervalCost(ai_merged, eisc)),
          KheCostShow(KheDrsAIntervalCost(ai_left, eisc)),
          KheCostShow(KheDrsAIntervalCost(ai_right, eisc)));

      ** set active_len for next iteration (child_e is now active) **
      dcs = HaArray(eisc->closed_seqs, open_index + 1);
      if( KheDrsClosedSeqAllActive(dcs) )
	active_len += 1 + dcs->active_at_right;
      else
        active_len = dcs->active_at_right;
      KheDrsDebugActiveLen(eisc, "active", open_index, child_e, prev_soln,
	dcs, active_len, drs);
      if( DEBUG_LAST_EDGE )
	fprintf(stderr, " %d", active_len);
    }
  }

  if( !KheDrsOpenChildrenIndexIsLast(&eisc->open_children_by_day, di) )
  {
    ** not last day; store active_len (adjusted) in next_soln **
    adjusted_len = KheDrsAdjustedSigVal(active_len,
      eisc->adjust_type, eisc->min_limit, eisc->max_limit, 0, 0);
    KheDrsExprPutSigVal((KHE_DRS_EXPR) eisc, di, next_soln, adjusted_len);
    if( DEBUG_LAST_EDGE )
      fprintf(stderr, ":%d$ %s\n", adjusted_len, KheMonitorId(eisc->monitor));
  }
  else
    if( DEBUG_LAST_EDGE )
      fprintf(stderr, "_ %s\n", KheMonitorId(eisc->monitor));
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSeqCostEvalSignature(KHE_DRS_EXPR_INT_SEQ_COST eisc,   */
/*    KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, int next_di,           */
/*    KHE_DRS_SIGNATURE next_sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)           */
/*                                                                           */
/*  Evaluate expression eisc in sig.                                         */
/*                                                                           */
/*  Implementation note.  Much of this code is the same as the code for      */
/*  closing each child_e, except that the a-interval and au-interval at      */
/*  left both have length active_len, and the cost is updated in             */
/*  next_soln->cost rather than in drs->solve_start_cost.                    */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntSeqCostEvalSignature(KHE_DRS_EXPR_INT_SEQ_COST eisc,
  KHE_DRS_SIGNER dsg, KHE_DRS_SIGNATURE prev_sig, int next_di,
  KHE_DRS_SIGNATURE next_sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  int open_index, i1, i2, active_len, adjusted_len;  KHE_DRS_EXPR child_e;
  KHE_DRS_AU_INTERVAL aui_left, aui_right, aui_merged;
  KHE_DRS_AU_INTERVAL aui_before, aui_after;
  KHE_DRS_A_INTERVAL ai_right_before, ai_right_after;
  KHE_DRS_A_INTERVAL ai_left, ai_right, ai_merged;
  KHE_DRS_CLOSED_SEQ dcs;

  /* initialize active_len */
  if( KheDrsOpenChildrenIndexIsFirst(&eisc->open_children_by_day, next_di) )
  {
    /* first day */
    dcs = HaArrayFirst(eisc->closed_seqs);
    active_len = dcs->active_at_right;
    if( KheDrsClosedSeqAllActive(dcs) )
      active_len += eisc->history;
    if( DEBUG_LAST_EDGE )
      fprintf(stderr, "  _%d", active_len);
  }
  else
  {
    /* not first day, so retrieve previous active_len */
    active_len = KheDrsExprDaySigVal((KHE_DRS_EXPR) eisc, next_di-1, prev_sig);
    if( DEBUG_LAST_EDGE )
      fprintf(stderr, "  $%d", active_len);
  }

  /* handle each child_e whose last open day is next_di */
  KheDrsOpenChildrenForEach(&eisc->open_children_by_day, next_di,
    child_e, open_index)
  {
    if( child_e->u.int_val == 0 )
    {
      /* child_e moves from unassigned to inactive: update cost */
      /* the enclosing au-interval splits */
      aui_left = KheDrsAUIntervalMakeLeft(eisc, open_index, active_len);
      aui_right = KheDrsAUIntervalFindRight(eisc, open_index, drs);
      aui_merged = KheDrsAUIntervalMerge(aui_left, aui_right, false);
      KheDrsSignatureAddCost(next_sig, KheDrsAUIntervalCost(aui_left, eisc)
	+ KheDrsAUIntervalCost(aui_right, eisc)
	- KheDrsAUIntervalCost(aui_merged, eisc));
      KheDrsMonitorInfoUpdateRerunCost(eisc->monitor_info, drs, dsg,
	KHE_DRS_SEARCH, "search", -1, "++-",
	KheDrsAUIntervalCost(aui_left, eisc),
	KheDrsAUIntervalCost(aui_right, eisc),
	KheDrsAUIntervalCost(aui_merged, eisc));
      /* ***
      KheDrsUpdateRerunCost(drs, &eisc->rerun_open_and_search_cost, NULL,
	eisc->monitor, "search", -1, "++-",
	KheDrsAUIntervalCost(aui_left, eisc),
	KheDrsAUIntervalCost(aui_right, eisc),
	KheDrsAUIntervalCost(aui_merged, eisc));
      *** */
      if( DEBUG_LAST_EDGE )
	fprintf(stderr, " 0aui +%.5f +%.5f -%.5f",
	  KheCostShow(KheDrsAUIntervalCost(aui_left, eisc)),
          KheCostShow(KheDrsAUIntervalCost(aui_right, eisc)),
          KheCostShow(KheDrsAUIntervalCost(aui_merged, eisc)));

      /* the a-interval to the right changes its unassigned_precedes */
      ai_right_before = KheDrsAIntervalFindRight(eisc, open_index, true);
      ai_right_after  = KheDrsAIntervalFindRight(eisc, open_index, false);
      KheDrsSignatureAddCost(next_sig, KheDrsAIntervalCost(ai_right_after, eisc)
	- KheDrsAIntervalCost(ai_right_before, eisc));
      KheDrsMonitorInfoUpdateRerunCost(eisc->monitor_info, drs, dsg,
	KHE_DRS_SEARCH, "search2", -1, "+-",
	KheDrsAIntervalCost(ai_right_after, eisc),
        KheDrsAIntervalCost(ai_right_before, eisc));
      /* ***
      KheDrsUpdateRerunCost(drs, &eisc->rerun_open_and_search_cost, NULL,
	eisc->monitor, "search2", -1, "+-",
        KheDrsAIntervalCost(ai_right_after, eisc),
        KheDrsAIntervalCost(ai_right_before, eisc));
      *** */
      if( DEBUG_LAST_EDGE )
	fprintf(stderr, " 0ai +%.5f -%.5f",
	  KheCostShow(KheDrsAIntervalCost(ai_right_after, eisc)),
          KheCostShow(KheDrsAIntervalCost(ai_right_before, eisc)));

      /* set active_len for next iteration (child_e is now inactive) */
      dcs = HaArray(eisc->closed_seqs, open_index + 1);
      active_len = dcs->active_at_right;
      KheDrsDebugActiveLen(eisc, "inactive", open_index, child_e, prev_sig,
	dcs, active_len, drs);
      if( DEBUG_LAST_EDGE )
	fprintf(stderr, " %d", active_len);
    }
    else
    {
      /* child_e moves from unassigned to active: update cost */
      /* the enclosing au-interval is unchanged, but its cost may change */
      aui_left = KheDrsAUIntervalMakeLeft(eisc, open_index, active_len);
      aui_right = KheDrsAUIntervalFindRight(eisc, open_index, drs);
      aui_before = KheDrsAUIntervalMerge(aui_left, aui_right, false);
      aui_after = KheDrsAUIntervalMerge(aui_left, aui_right, true);
      KheDrsSignatureAddCost(next_sig, KheDrsAUIntervalCost(aui_after, eisc)
	- KheDrsAUIntervalCost(aui_before, eisc));
      KheDrsMonitorInfoUpdateRerunCost(eisc->monitor_info, drs, dsg,
	KHE_DRS_SEARCH, "search", -1, "+-",
	KheDrsAUIntervalCost(aui_after, eisc),
	KheDrsAUIntervalCost(aui_before, eisc));
      /* ***
      KheDrsUpdateRerunCost(drs, &eisc->rerun_open_and_search_cost, NULL,
	eisc->monitor, "search", -1, "+-",
        KheDrsAUIntervalCost(aui_after, eisc),
	KheDrsAUIntervalCost(aui_before, eisc));
      *** */
      if( DEBUG_LAST_EDGE )
	fprintf(stderr, " 1aui +%.5f -%.5f",
	  KheCostShow(KheDrsAUIntervalCost(aui_after, eisc)),
          KheCostShow(KheDrsAUIntervalCost(aui_before, eisc)));

      /* the a-intervals on each side merge */
      ai_left = KheDrsAIntervalMakeLeft(eisc, open_index, active_len);
      ai_right = KheDrsAIntervalFindRight(eisc, open_index, true);
      ai_merged = KheDrsAIntervalMerge(ai_left, ai_right);
      KheDrsSignatureAddCost(next_sig, KheDrsAIntervalCost(ai_merged, eisc)
	- KheDrsAIntervalCost(ai_left, eisc)
	- KheDrsAIntervalCost(ai_right, eisc));
      KheDrsMonitorInfoUpdateRerunCost(eisc->monitor_info, drs, dsg,
	KHE_DRS_SEARCH, "search2", -1, "+--",
	KheDrsAIntervalCost(ai_merged, eisc),
	KheDrsAIntervalCost(ai_left, eisc),
	KheDrsAIntervalCost(ai_right, eisc));
      /* ***
      KheDrsUpdateRerunCost(drs, &eisc->rerun_open_and_search_cost, NULL,
	eisc->monitor, "search2", -1, "+--",
        KheDrsAIntervalCost(ai_merged, eisc),
	KheDrsAIntervalCost(ai_left, eisc),
	KheDrsAIntervalCost(ai_right, eisc));
      *** */
      if( DEBUG_LAST_EDGE )
	fprintf(stderr, " 1ai +%.5f -%.5f -%.5f",
	  KheCostShow(KheDrsAIntervalCost(ai_merged, eisc)),
          KheCostShow(KheDrsAIntervalCost(ai_left, eisc)),
          KheCostShow(KheDrsAIntervalCost(ai_right, eisc)));

      /* set active_len for next iteration (child_e is now active) */
      dcs = HaArray(eisc->closed_seqs, open_index + 1);
      if( KheDrsClosedSeqAllActive(dcs) )
	active_len += 1 + dcs->active_at_right;
      else
        active_len = dcs->active_at_right;
      KheDrsDebugActiveLen(eisc, "active", open_index, child_e, prev_sig,
	dcs, active_len, drs);
      if( DEBUG_LAST_EDGE )
	fprintf(stderr, " %d", active_len);
    }
  }

  if( !KheDrsOpenChildrenIndexIsLast(&eisc->open_children_by_day, next_di) )
  {
    /* not last day; store active_len (adjusted) in sig */
    adjusted_len = KheDrsAdjustedSigVal(active_len,
      eisc->adjust_type, eisc->min_limit, eisc->max_limit, 0);
    if( DEBUG44_MONITOR(eisc->monitor_info->monitor) && adjusted_len >= 4 )
      fprintf(stderr, "  EvalSig(%s): day %d, adjusted_len = %d\n",
	KheDrsMonitorInfoId(eisc->monitor_info), next_di, adjusted_len);
    KheDrsSignatureAddState(next_sig, adjusted_len, dsg, (KHE_DRS_EXPR) eisc);
    /* KheDrsExprPutSigVal((KHE_DRS_EXPR) eisc, next_di, sig, adjusted_len); */
    if( DEBUG_LAST_EDGE )
      fprintf(stderr, ":%d$ %s\n", adjusted_len,
	KheDrsMonitorInfoId(eisc->monitor_info));
  }
  else
    if( DEBUG_LAST_EDGE )
      fprintf(stderr, "_ %s\n", KheDrsMonitorInfoId(eisc->monitor_info));
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSeqCostInitRerunCost(KHE_DRS_EXPR_INT_SEQ_COST eisc)   */
/*                                                                           */
/*  Initialize the rerun cost fields of eisc.                                */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsExprIntSeqCostInitRerunCost(KHE_DRS_EXPR_INT_SEQ_COST eisc)
{
  eisc->rerun_open_and_search_cost = eisc->rerun_open_and_close_cost =
    KheMonitorCost(eisc->monitor);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSeqCostCheckRerunCost(KHE_DRS_EXPR_INT_SEQ_COST eisc)  */
/*                                                                           */
/*  Check the rerun cost fields of eisc.                                     */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsExprIntSeqCostCheckRerunCost(KHE_DRS_EXPR_INT_SEQ_COST eisc)
{
  if( eisc->rerun_open_and_search_cost != KheMonitorCost(eisc->monitor) )
    fprintf(stderr, "  open_and_search_cost %.5f, monitor cost %.5f (%s)\n",
      KheCostShow(eisc->rerun_open_and_search_cost),
      KheCostShow(KheMonitorCost(eisc->monitor)), KheMonitorId(eisc->monitor));
  if( eisc->rerun_open_and_close_cost != KheMonitorCost(eisc->monitor) )
    fprintf(stderr, "  open_and_close_cost  %.5f, monitor cost %.5f (%s)\n",
      KheCostShow(eisc->rerun_open_and_close_cost),
      KheCostShow(KheMonitorCost(eisc->monitor)), KheMonitorId(eisc->monitor));
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  char *KheDrsExprIntSeqCostShowSeqType(KHE_DRS_EXPR_INT_SEQ_COST eisc)    */
/*                                                                           */
/*  Show the seq_type and seq_index fields of eisc.                          */
/*                                                                           */
/*****************************************************************************/

static char *KheDrsExprIntSeqCostShowSeqType(KHE_DRS_EXPR_INT_SEQ_COST eisc)
{
  static char buff[200];
  switch( eisc->seq_type )
  {
    case KHE_DRS_SEQ_NONE:

      sprintf(buff, "-");
      break;

    case KHE_DRS_SEQ_DAYS_POSITIVE:

      sprintf(buff, "+ve");
      break;

    case KHE_DRS_SEQ_DAYS_NEGATIVE:

      sprintf(buff, "-ve");
      break;

    case KHE_DRS_SEQ_SINGLE_POSITIVE:

      sprintf(buff, "+%d", eisc->seq_index);
      break;

    default:

      sprintf(buff, "?");
      break;
  }
  return buff;
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprIntSeqCostDoDebug(KHE_DRS_EXPR_INT_SEQ_COST eisc,         */
/*    KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)          */
/*                                                                           */
/*  Unindented debug print of that part of eisc which is specific to it.     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprIntSeqCostDoDebug(KHE_DRS_EXPR_INT_SEQ_COST eisc,
  KHE_DRS_SOLN soln, KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  fprintf(fp, "INT_SEQ_COST(%s, %s) %s",
    KheDrsMonitorInfoId(eisc->monitor_info),
    KheDrsAdjustTypeShow(eisc->adjust_type),
    KheDrsExprIntSeqCostShowSeqType(eisc));
  KheDrsExprDebugSignature((KHE_DRS_EXPR) eisc, soln, false, drs, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_EXPR - deferred functions"                            */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprAddChild(KHE_DRS_EXPR parent, KHE_DRS_EXPR child,         */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make child a child of parent.                                            */
/*                                                                           */
/*  Implementation note.  During expression construction, all expressions    */
/*  are closed by the time they are made children of their parents (by       */
/*  calls on this function).  Hence KheDrsExprChildHasClosed below.          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprAddChild(KHE_DRS_EXPR parent, KHE_DRS_EXPR child,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_PARENT prnt;

  /* link parent and child */
  HnAssert(HaArrayCount(parent->parents) == 0,
    "KheDrsExprAddChild internal error:  too late to add child (1)");
  HnAssert(parent->postorder_index == -1,
    "KheDrsExprAddChild internal error:  too late to add child (2)");
  /* *** does not work; parent->postorder_index not set yet
  if( DEBUG59 )
    HnAssert(parent->postorder_index > child->postorder_index,
      "KheDrsExprAddChild internal error:  parent pi %d, child pi %d",
      parent->postorder_index, child->postorder_index);
  *** */
  prnt.expr = parent;
  prnt.index = HaArrayCount(parent->children);
  HaArrayAddLast(child->parents, prnt);
  HaArrayAddLast(parent->children, child);

  /* update state in each parent */
  switch( parent->tag )
  {
    case KHE_DRS_EXPR_OR_TAG:

      KheDrsExprOrAddChild((KHE_DRS_EXPR_OR) parent, child, drs);
      break;

    case KHE_DRS_EXPR_AND_TAG:

      KheDrsExprAndAddChild((KHE_DRS_EXPR_AND) parent, child, drs);
      break;

    case KHE_DRS_EXPR_INT_SUM_TAG:

      KheDrsExprIntSumAddChild((KHE_DRS_EXPR_INT_SUM) parent, child, drs);
      break;

    case KHE_DRS_EXPR_FLOAT_SUM_TAG:

      KheDrsExprFloatSumAddChild((KHE_DRS_EXPR_FLOAT_SUM) parent, child, drs);
      break;

    case KHE_DRS_EXPR_INT_DEV_TAG:

      KheDrsExprIntDevAddChild((KHE_DRS_EXPR_INT_DEV) parent, child, drs);
      break;

    case KHE_DRS_EXPR_FLOAT_DEV_TAG:

      KheDrsExprFloatDevAddChild((KHE_DRS_EXPR_FLOAT_DEV) parent, child, drs);
      break;

    case KHE_DRS_EXPR_INT_SUM_COST_TAG:

      KheDrsExprIntSumCostAddChild((KHE_DRS_EXPR_INT_SUM_COST) parent,
	child, drs);
      break;

    case KHE_DRS_EXPR_INT_SEQ_COST_TAG:

      KheDrsExprIntSeqCostAddChild((KHE_DRS_EXPR_INT_SEQ_COST) parent,
	child, drs);
      break;

    default:

      HnAbort("KheDrsExprAddChild internal error (tag %d)", parent->tag);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsExprExpressesEventResourceConstraint(KHE_DRS_EXPR e)          */
/*                                                                           */
/*  Return true if e expresses an event resource constraint, that is, if     */
/*  it is an INT_SUM_COST expression whose resource attribute is NULL.       */
/*                                                                           */
/*****************************************************************************/

/* *** no longer used
static bool KheDrsExprExpressesEventResourceConstraint(KHE_DRS_EXPR e)
{
  KHE_DRS_EXPR_INT_SUM_COST eisc;
  if( e->tag != KHE_DRS_EXPR_INT_SUM_COST_TAG )
    return false;
  eisc = (KHE_DRS_EXPR_INT_SUM_COST) e;
  return eisc->resource == NULL;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprChildHasOpened(KHE_DRS_EXPR e, KHE_DRS_EXPR child_e,      */
/*    int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)                      */
/*                                                                           */
/*  As part of opening e for solving, inform e that its child_index'th       */
/*  child, child_e, has opened.                                              */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprChildHasOpened(KHE_DRS_EXPR e, KHE_DRS_EXPR child_e,
  int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  /* int i, e2_index, child_e_index;  KHE_DRS_EXPR e2; */

  /* add child_e to e->open_children */
  if( e->tag == KHE_DRS_EXPR_INT_SEQ_COST_TAG )
    KheDrsOpenChildrenAddChildWithDayUpdate(&e->open_children_by_day, child_e);
  else
    KheDrsOpenChildrenAddChildInDayOrder(&e->open_children_by_day, child_e);

  /* alter the state of e to take account of child_e opening */
  switch( e->tag )
  {
    case KHE_DRS_EXPR_OR_TAG:

      KheDrsExprOrChildHasOpened((KHE_DRS_EXPR_OR) e,
	child_e, child_index, drs);
      break;

    case KHE_DRS_EXPR_AND_TAG:

      KheDrsExprAndChildHasOpened((KHE_DRS_EXPR_AND) e,
	child_e, child_index, drs);
      break;

    case KHE_DRS_EXPR_INT_SUM_TAG:

      KheDrsExprIntSumChildHasOpened((KHE_DRS_EXPR_INT_SUM) e,
	child_e, child_index, drs);
      break;

    case KHE_DRS_EXPR_FLOAT_SUM_TAG:

      KheDrsExprFloatSumChildHasOpened((KHE_DRS_EXPR_FLOAT_SUM) e,
	child_e, child_index, drs);
      break;

    case KHE_DRS_EXPR_INT_DEV_TAG:

      KheDrsExprIntDevChildHasOpened((KHE_DRS_EXPR_INT_DEV) e,
	child_e, child_index, drs);
      break;

    case KHE_DRS_EXPR_FLOAT_DEV_TAG:

      KheDrsExprFloatDevChildHasOpened((KHE_DRS_EXPR_FLOAT_DEV) e,
	child_e, child_index, drs);
      break;

    case KHE_DRS_EXPR_INT_SUM_COST_TAG:

      KheDrsExprIntSumCostChildHasOpened((KHE_DRS_EXPR_INT_SUM_COST) e,
	child_e, child_index, drs);
      break;

    case KHE_DRS_EXPR_INT_SEQ_COST_TAG:

      KheDrsExprIntSeqCostChildHasOpened((KHE_DRS_EXPR_INT_SEQ_COST) e,
	child_e, child_index, drs);
      break;

    default:

      HnAbort("KheDrsExprChildHasOpened internal error (tag %d)", e->tag);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprChildHasClosed(KHE_DRS_EXPR e,                            */
/*    KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)*/
/*                                                                           */
/*  Inform internal expression e that its child_index'th child, child_e,     */
/*  has closed.  The child will have its correct closed_state and value      */
/*  fields when this is called.                                              */
/*                                                                           */
/*  Although KheDrsExprChildHasOpened adds child_e to e->open_children,      */
/*  KheDrsExprChildHasClosed does not remove child_e from e->open_children.  */
/*  This is because children are closed in the same order they were opened,  */
/*  but there is no efficient way to remove them in this order, short of     */
/*  something too complicated such as a circular list.                       */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprChildHasClosed(KHE_DRS_EXPR e,
  KHE_DRS_EXPR child_e, int child_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  /* remove child_e from e->open_children */
  /* *** no longer doing this
  int pos;
  if( !HaArrayContains(e->open_children, child_e, &pos) )
    HnAbort("KheDrsExprChildHasClosed internal error");
  HaArrayDeleteAndPlug(e->open_children, pos);
  *** */

  /* do the type-specific part of this operation */
  switch( e->tag )
  {
    case KHE_DRS_EXPR_OR_TAG:

      KheDrsExprOrChildHasClosed((KHE_DRS_EXPR_OR) e,
	child_e, child_index, drs);
      break;

    case KHE_DRS_EXPR_AND_TAG:

      KheDrsExprAndChildHasClosed((KHE_DRS_EXPR_AND) e,
	child_e, child_index, drs);
      break;

    case KHE_DRS_EXPR_INT_SUM_TAG:

      KheDrsExprIntSumChildHasClosed((KHE_DRS_EXPR_INT_SUM) e,
	child_e, child_index, drs);
      break;

    case KHE_DRS_EXPR_FLOAT_SUM_TAG:

      KheDrsExprFloatSumChildHasClosed((KHE_DRS_EXPR_FLOAT_SUM) e,
	child_e, child_index, drs);
      break;

    case KHE_DRS_EXPR_INT_DEV_TAG:

      KheDrsExprIntDevChildHasClosed((KHE_DRS_EXPR_INT_DEV) e,
	child_e, child_index, drs);
      break;

    case KHE_DRS_EXPR_FLOAT_DEV_TAG:

      KheDrsExprFloatDevChildHasClosed((KHE_DRS_EXPR_FLOAT_DEV) e,
	child_e, child_index, drs);
      break;

    case KHE_DRS_EXPR_INT_SUM_COST_TAG:

      KheDrsExprIntSumCostChildHasClosed((KHE_DRS_EXPR_INT_SUM_COST) e,
	child_e, child_index, drs);
      break;

    case KHE_DRS_EXPR_INT_SEQ_COST_TAG:

      KheDrsExprIntSeqCostChildHasClosed((KHE_DRS_EXPR_INT_SEQ_COST) e,
	child_e, child_index, drs);
      break;

    default:

      HnAbort("KheDrsExprChildHasClosed internal error (tag %d)", e->tag);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprSetClosedValue(KHE_DRS_EXPR e,                            */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  As part of closing e, set its value suitably for its closed state.       */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprSetClosedValue(KHE_DRS_EXPR e,
  KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  switch( e->tag )
  {
    case KHE_DRS_EXPR_ASSIGNED_TASK_TAG:

      KheDrsExprAssignedTaskSetClosedValue((KHE_DRS_EXPR_ASSIGNED_TASK) e, drs);
      break;

    case KHE_DRS_EXPR_BUSY_TIME_TAG:

      KheDrsExprBusyTimeSetClosedValue((KHE_DRS_EXPR_BUSY_TIME) e, drs);
      break;

    case KHE_DRS_EXPR_FREE_TIME_TAG:

      KheDrsExprFreeTimeSetClosedValue((KHE_DRS_EXPR_FREE_TIME) e, drs);
      break;

    case KHE_DRS_EXPR_WORK_TIME_TAG:

      KheDrsExprWorkTimeSetClosedValue((KHE_DRS_EXPR_WORK_TIME) e, drs);
      break;

    case KHE_DRS_EXPR_BUSY_DAY_TAG:

      KheDrsExprBusyDaySetClosedValue((KHE_DRS_EXPR_BUSY_DAY) e, drs);
      break;

    case KHE_DRS_EXPR_FREE_DAY_TAG:

      KheDrsExprFreeDaySetClosedValue((KHE_DRS_EXPR_FREE_DAY) e, drs);
      break;

    case KHE_DRS_EXPR_WORK_DAY_TAG:

      KheDrsExprWorkDaySetClosedValue((KHE_DRS_EXPR_WORK_DAY) e, drs);
      break;

    case KHE_DRS_EXPR_OR_TAG:

      KheDrsExprOrSetClosedValue((KHE_DRS_EXPR_OR) e, drs);
      break;

    case KHE_DRS_EXPR_AND_TAG:

      KheDrsExprAndSetClosedValue((KHE_DRS_EXPR_AND) e, drs);
      break;

    case KHE_DRS_EXPR_INT_SUM_TAG:

      KheDrsExprIntSumSetClosedValue((KHE_DRS_EXPR_INT_SUM) e, drs);
      break;

    case KHE_DRS_EXPR_FLOAT_SUM_TAG:

      KheDrsExprFloatSumSetClosedValue((KHE_DRS_EXPR_FLOAT_SUM) e,drs);
      break;

    case KHE_DRS_EXPR_INT_DEV_TAG:

      KheDrsExprIntDevSetClosedValue((KHE_DRS_EXPR_INT_DEV) e, drs);
      break;

    case KHE_DRS_EXPR_FLOAT_DEV_TAG:

      KheDrsExprFloatDevSetClosedValue((KHE_DRS_EXPR_FLOAT_DEV) e,drs);
      break;

    case KHE_DRS_EXPR_INT_SUM_COST_TAG:

      KheDrsExprIntSumCostSetClosedValue((KHE_DRS_EXPR_INT_SUM_COST) e, drs);
      break;

    case KHE_DRS_EXPR_INT_SEQ_COST_TAG:

      KheDrsExprIntSeqCostSetClosedValue((KHE_DRS_EXPR_INT_SEQ_COST) e, drs);
      break;

    default:

      HnAbort("KheDrsExprSetClosedValue internal error (tag %d)", e->tag);
      break;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprChildDomTest(KHE_DRS_EXPR e,                  */
/*    KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)                 */
/*                                                                           */
/*  Return a dom test suited to child_e, a child of e.                       */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DOM_TEST KheDrsExprChildDomTest(KHE_DRS_EXPR e,
  int open_day_index, KHE_DRS_EXPR child_e, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_PARENT parent;
  switch( e->tag )
  {
    case KHE_DRS_EXPR_ASSIGNED_TASK_TAG:
    case KHE_DRS_EXPR_BUSY_TIME_TAG:
    case KHE_DRS_EXPR_FREE_TIME_TAG:
    case KHE_DRS_EXPR_WORK_TIME_TAG:
    case KHE_DRS_EXPR_BUSY_DAY_TAG:
    case KHE_DRS_EXPR_FREE_DAY_TAG:
    case KHE_DRS_EXPR_WORK_DAY_TAG:

      HnAbort("KheDrsExprChildDomTest: unexpected tag %d", e->tag);
      /* no break, to keep compiler happy */

    case KHE_DRS_EXPR_OR_TAG:
    case KHE_DRS_EXPR_AND_TAG:
    case KHE_DRS_EXPR_INT_SUM_TAG:
    case KHE_DRS_EXPR_FLOAT_SUM_TAG:
    case KHE_DRS_EXPR_INT_DEV_TAG:
    case KHE_DRS_EXPR_FLOAT_DEV_TAG:

      HnAssert(HaArrayCount(e->parents) > 0,
	"KheDrsExprChildDomTest internal error");
      parent = HaArrayFirst(e->parents);
      return KheDrsExprChildDomTest(parent.expr, open_day_index, child_e, drs);

    case KHE_DRS_EXPR_INT_SUM_COST_TAG:

      return KheDrsExprIntSumCostChildDomTest((KHE_DRS_EXPR_INT_SUM_COST) e,
	open_day_index, child_e, drs);

    case KHE_DRS_EXPR_INT_SEQ_COST_TAG:

      return KheDrsExprIntSeqCostChildDomTest((KHE_DRS_EXPR_INT_SEQ_COST) e,
	open_day_index, child_e, drs);

    default:

      HnAbort("KheDrsExprChildDomTest internal error (tag %d)", e->tag);
      return KheDrsExprIntSeqCostChildDomTest((KHE_DRS_EXPR_INT_SEQ_COST) e,
	0, NULL, drs);  /* keep compiler happy */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprDayDomTest(KHE_DRS_EXPR e,                    */
/*    int open_day_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)                   */
/*                                                                           */
/*  Return a dom test suited to e on the open day with this open day index.  */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_DOM_TEST KheDrsExprDayDomTest(KHE_DRS_EXPR e,
  int open_day_index, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_PARENT prnt;
  switch( e->tag )
  {
    case KHE_DRS_EXPR_ASSIGNED_TASK_TAG:
    case KHE_DRS_EXPR_BUSY_TIME_TAG:
    case KHE_DRS_EXPR_FREE_TIME_TAG:
    case KHE_DRS_EXPR_WORK_TIME_TAG:
    case KHE_DRS_EXPR_BUSY_DAY_TAG:
    case KHE_DRS_EXPR_FREE_DAY_TAG:
    case KHE_DRS_EXPR_WORK_DAY_TAG:

      HnAbort("KheDrsExprDayDomTest: unexpected tag %d", e->tag);
      /* no break, to keep compiler happy */

    case KHE_DRS_EXPR_OR_TAG:
    case KHE_DRS_EXPR_AND_TAG:
    case KHE_DRS_EXPR_INT_SUM_TAG:
    case KHE_DRS_EXPR_FLOAT_SUM_TAG:
    case KHE_DRS_EXPR_INT_DEV_TAG:
    case KHE_DRS_EXPR_FLOAT_DEV_TAG:

      HnAssert(HaArrayCount(e->parents) > 0,
	"KheDrsExprDayDomTest internal error");
      prnt = HaArrayFirst(e->parents);
      return KheDrsExprChildDomTest(prnt.expr, open_day_index, e, drs);

    case KHE_DRS_EXPR_INT_SUM_COST_TAG:

      return KheDrsExprIntSumCostDayDomTest((KHE_DRS_EXPR_INT_SUM_COST) e,
	open_day_index, drs);

    case KHE_DRS_EXPR_INT_SEQ_COST_TAG:

      return KheDrsExprIntSeqCostDayDomTest((KHE_DRS_EXPR_INT_SEQ_COST) e,
	open_day_index, drs);

    default:

      HnAbort("KheDrsExprDayDomTest internal error (tag %d)", e->tag);
      return KheDrsExprIntSeqCostDayDomTest((KHE_DRS_EXPR_INT_SEQ_COST) e,
	open_day_index, drs);  /* keep compiler happy */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_DOM_TEST KheDrsExprDayDomTest(KHE_DRS_EXPR e, int open_day_index)*/
/*                                                                           */
/*  Return a dom test suited to e on the open day with this open day index.  */
/*                                                                           */
/*****************************************************************************/

/* *** unused
static KHE_DRS_DOM_TEST KheDrsExprShiftDomTest(KHE_DRS_EXPR e,
  int open_day_index, int curr_count, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_PARENT parent;
  switch( e->tag )
  {
    case KHE_DRS_EXPR_ASSIGNED_TASK_TAG:
    case KHE_DRS_EXPR_BUSY_TIME_TAG:
    case KHE_DRS_EXPR_FREE_TIME_TAG:
    case KHE_DRS_EXPR_WORK_TIME_TAG:
    case KHE_DRS_EXPR_BUSY_DAY_TAG:
    case KHE_DRS_EXPR_FREE_DAY_TAG:
    case KHE_DRS_EXPR_WORK_DAY_TAG:
    case KHE_DRS_EXPR_INT_SEQ_COST_TAG:

      HnAbort("KheDrsExprShiftDomTest: unexpected tag %d", e->tag);
      ** no break, to keep compiler happy **

    case KHE_DRS_EXPR_OR_TAG:
    case KHE_DRS_EXPR_AND_TAG:
    case KHE_DRS_EXPR_INT_SUM_TAG:
    case KHE_DRS_EXPR_FLOAT_SUM_TAG:
    case KHE_DRS_EXPR_INT_DEV_TAG:
    case KHE_DRS_EXPR_FLOAT_DEV_TAG:

      HnAssert(HaArrayCount(e->parents) > 0,
	"KheDrsExprDayDomTest internal error");
      parent = HaArrayFirst(e->parents);
      return KheDrsExprChildDomTest(parent.expr, drs);

    case KHE_DRS_EXPR_INT_SUM_COST_TAG:

      return KheDrsExprIntSumCostShiftDomTest((KHE_DRS_EXPR_INT_SUM_COST) e,
	open_day_index, curr_count, drs);

    default:

      HnAbort("KheDrsExprShiftDomTest internal error (tag %d)", e->tag);
      return KheDrsExprIntSeqCostDayDomTest((KHE_DRS_EXPR_INT_SEQ_COST) e,
	open_day_index, drs);  ** keep compiler happy **
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprLeafSet(KHE_DRS_EXPR e, KHE_DRS_TASK_ON_DAY dtd,          */
/*    KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs)                  */
/*                                                                           */
/*  Inform leaf expression e that dtd has been assigned dr.                  */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprLeafSet(KHE_DRS_EXPR e, KHE_DRS_TASK_ON_DAY dtd,
  KHE_DRS_RESOURCE dr, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  switch( e->tag )
  {
    case KHE_DRS_EXPR_ASSIGNED_TASK_TAG:

      KheDrsExprAssignedTaskLeafSet((KHE_DRS_EXPR_ASSIGNED_TASK) e, dtd, dr);
      break;

    case KHE_DRS_EXPR_BUSY_TIME_TAG:

      KheDrsExprBusyTimeLeafSet((KHE_DRS_EXPR_BUSY_TIME) e, dtd, dr);
      break;

    case KHE_DRS_EXPR_FREE_TIME_TAG:

      KheDrsExprFreeTimeLeafSet((KHE_DRS_EXPR_FREE_TIME) e, dtd, dr);
      break;

    case KHE_DRS_EXPR_WORK_TIME_TAG:

      KheDrsExprWorkTimeLeafSet((KHE_DRS_EXPR_WORK_TIME) e, dtd, dr);
      break;

    case KHE_DRS_EXPR_BUSY_DAY_TAG:

      KheDrsExprBusyDayLeafSet((KHE_DRS_EXPR_BUSY_DAY) e, dtd, dr, drs);
      break;

    case KHE_DRS_EXPR_FREE_DAY_TAG:

      KheDrsExprFreeDayLeafSet((KHE_DRS_EXPR_FREE_DAY) e, dtd, dr);
      break;

    case KHE_DRS_EXPR_WORK_DAY_TAG:

      KheDrsExprWorkDayLeafSet((KHE_DRS_EXPR_WORK_DAY) e, dtd, dr);
      break;

    default:

      HnAbort("KheDrsExprLeafSet internal error: tag %d", e->tag);
      break;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprLeafClear(KHE_DRS_EXPR e)                                 */
/*                                                                           */
/*  Inform leaf expression e that the assignment of drd to dtd carried out   */
/*  by KheDrsExprLeafSet above has been removed.                             */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprLeafClear(KHE_DRS_EXPR e, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  switch( e->tag )
  {
    case KHE_DRS_EXPR_ASSIGNED_TASK_TAG:

      KheDrsExprAssignedTaskLeafClear((KHE_DRS_EXPR_ASSIGNED_TASK) e);
      break;

    case KHE_DRS_EXPR_BUSY_TIME_TAG:

      KheDrsExprBusyTimeLeafClear((KHE_DRS_EXPR_BUSY_TIME) e);
      break;

    case KHE_DRS_EXPR_FREE_TIME_TAG:

      KheDrsExprFreeTimeLeafClear((KHE_DRS_EXPR_FREE_TIME) e);
      break;

    case KHE_DRS_EXPR_WORK_TIME_TAG:

      KheDrsExprWorkTimeLeafClear((KHE_DRS_EXPR_WORK_TIME) e);
      break;

    case KHE_DRS_EXPR_BUSY_DAY_TAG:

      KheDrsExprBusyDayLeafClear((KHE_DRS_EXPR_BUSY_DAY) e, drs);
      break;

    case KHE_DRS_EXPR_FREE_DAY_TAG:

      KheDrsExprFreeDayLeafClear((KHE_DRS_EXPR_FREE_DAY) e);
      break;

    case KHE_DRS_EXPR_WORK_DAY_TAG:

      KheDrsExprWorkDayLeafClear((KHE_DRS_EXPR_WORK_DAY) e);
      break;

    default:

      HnAbort("KheDrsExprLeafClear internal error: tag %d");
      break;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprEvalDay(KHE_DRS_EXPR e, KHE_DRS_SOLN prev_soln,           */
/*    KHE_DRS_SOLN next_soln, KHE_DYNAMIC_RESOURCE_SOLVER drs)               */
/*                                                                           */
/*  Evaluate expression e in next_soln, whose day is one of e's open days.   */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsExprEvalDay(KHE_DRS_EXPR e, KHE_DRS_SOLN prev_soln,
  KHE_DRS_SOLN next_soln, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  switch( e->tag )
  {
    case KHE_DRS_EXPR_ASSIGNED_TASK_TAG:
    case KHE_DRS_EXPR_BUSY_TIME_TAG:
    case KHE_DRS_EXPR_FREE_TIME_TAG:
    case KHE_DRS_EXPR_WORK_TIME_TAG:
    case KHE_DRS_EXPR_BUSY_DAY_TAG:
    case KHE_DRS_EXPR_FREE_DAY_TAG:
    case KHE_DRS_EXPR_WORK_DAY_TAG:

      HnAbort("KheDrsExprEvalDay: external tag (%d)", e->tag);
      break;

    case KHE_DRS_EXPR_OR_TAG:

      KheDrsExprOrEvalDay((KHE_DRS_EXPR_OR) e, prev_soln, next_soln, drs);
      break;

    case KHE_DRS_EXPR_AND_TAG:

      KheDrsExprAndEvalDay((KHE_DRS_EXPR_AND) e, prev_soln, next_soln, drs);
      break;

    case KHE_DRS_EXPR_INT_SUM_TAG:

      KheDrsExprIntSumEvalDay((KHE_DRS_EXPR_INT_SUM) e,
	prev_soln, next_soln, drs);
      break;

    case KHE_DRS_EXPR_FLOAT_SUM_TAG:

      KheDrsExprFloatSumEvalDay((KHE_DRS_EXPR_FLOAT_SUM) e,
	prev_soln, next_soln, drs);
      break;

    case KHE_DRS_EXPR_INT_DEV_TAG:

      KheDrsExprIntDevEvalDay((KHE_DRS_EXPR_INT_DEV) e,
	prev_soln, next_soln, drs);
      break;

    case KHE_DRS_EXPR_FLOAT_DEV_TAG:

      KheDrsExprFloatDevEvalDay((KHE_DRS_EXPR_FLOAT_DEV) e,
	prev_soln, next_soln, drs);
      break;

    case KHE_DRS_EXPR_INT_SUM_COST_TAG:

      KheDrsExprIntSumCostEvalDay((KHE_DRS_EXPR_INT_SUM_COST) e,
	prev_soln, next_soln, drs);
      break;

    case KHE_DRS_EXPR_INT_SEQ_COST_TAG:

      KheDrsExprIntSeqCostEvalDay((KHE_DRS_EXPR_INT_SEQ_COST) e,
	prev_soln, next_soln, drs);
      break;

    default:

      HnAbort("KheDrsExprEvalDay internal error");
      break;
  }
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprEvalSignature(KHE_DRS_EXPR e, KHE_DRS_SIGNER dsg,         */
/*    KHE_DRS_SIGNATURE prev_sig, int next_di, KHE_DRS_SIGNATURE sig,        */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, bool debug)                           */
/*                                                                           */
/*  Like KheDrsExprEvalDay except that the results go into ss rather         */
/*  than into next_soln.                                                     */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprEvalSignature(KHE_DRS_EXPR e, KHE_DRS_SIGNER dsg,
  KHE_DRS_SIGNATURE prev_sig, int next_di, KHE_DRS_SIGNATURE next_sig,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, bool debug)
{
  switch( e->tag )
  {
    case KHE_DRS_EXPR_ASSIGNED_TASK_TAG:
    case KHE_DRS_EXPR_BUSY_TIME_TAG:
    case KHE_DRS_EXPR_FREE_TIME_TAG:
    case KHE_DRS_EXPR_WORK_TIME_TAG:
    case KHE_DRS_EXPR_BUSY_DAY_TAG:
    case KHE_DRS_EXPR_FREE_DAY_TAG:
    case KHE_DRS_EXPR_WORK_DAY_TAG:

      HnAbort("KheDrsExprEvalSignature: external tag (%d)", e->tag);
      break;

    case KHE_DRS_EXPR_OR_TAG:

      KheDrsExprOrEvalSignature((KHE_DRS_EXPR_OR) e, dsg,
	prev_sig, next_di, next_sig, drs);
      break;

    case KHE_DRS_EXPR_AND_TAG:

      KheDrsExprAndEvalSignature((KHE_DRS_EXPR_AND) e, dsg,
	prev_sig, next_di, next_sig, drs);
      break;

    case KHE_DRS_EXPR_INT_SUM_TAG:

      KheDrsExprIntSumEvalSignature((KHE_DRS_EXPR_INT_SUM) e, dsg,
	prev_sig, next_di, next_sig, drs);
      break;

    case KHE_DRS_EXPR_FLOAT_SUM_TAG:

      KheDrsExprFloatSumEvalSignature((KHE_DRS_EXPR_FLOAT_SUM) e, dsg,
	prev_sig, next_di, next_sig, drs);
      break;

    case KHE_DRS_EXPR_INT_DEV_TAG:

      KheDrsExprIntDevEvalSignature((KHE_DRS_EXPR_INT_DEV) e, dsg,
	prev_sig, next_di, next_sig, drs);
      break;

    case KHE_DRS_EXPR_FLOAT_DEV_TAG:

      KheDrsExprFloatDevEvalSignature((KHE_DRS_EXPR_FLOAT_DEV) e, dsg,
	prev_sig, next_di, next_sig, drs);
      break;

    case KHE_DRS_EXPR_INT_SUM_COST_TAG:

      KheDrsExprIntSumCostEvalSignature((KHE_DRS_EXPR_INT_SUM_COST) e, dsg,
	prev_sig, next_di, next_sig, drs, debug);
      break;

    case KHE_DRS_EXPR_INT_SEQ_COST_TAG:

      KheDrsExprIntSeqCostEvalSignature((KHE_DRS_EXPR_INT_SEQ_COST) e, dsg,
	prev_sig, next_di, next_sig, drs);
      break;

    default:

      HnAbort("KheDrsExprEvalSignature internal error");
      break;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprEvalShiftSignature(KHE_DRS_EXPR e,                        */
/*    KHE_DRS_SOLN prev_soln, int next_di, KHE_DRS_SHIFT ds,                 */
/*    KHE_DRS_SIGNATURE sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)                */
/*                                                                           */
/*  Like KheDrsExprEvalSignature except for evaluating a solution that       */
/*  consists of a d_k-complete solution plus an assignment to one shift.     */
/*  Only event resource constraints are evaluated, so this function only     */
/*  has to evaluate one kind of internal expression.                         */
/*                                                                           */
/*****************************************************************************/

/* *** now folded back into KheDrsExprEvalSignature
static void KheDrsExprEvalShiftSignature(KHE_DRS_EXPR e,
  KHE_DRS_SOLN prev_soln, int next_di, KHE_DRS_SHIFT ds,
  KHE_DRS_SIGNATURE sig, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  HnAssert(e->tag == KHE_DRS_EXPR_INT_SUM_COST_TAG,
    "KheDrsExprEvalShiftSignature internal error");
  KheDrsExprIntSumCostEvalShiftSignature((KHE_DRS_EXPR_INT_SUM_COST) e,
    prev_soln, next_di, ds, sig, drs);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprCostInitRerunCost(KHE_DRS_EXPR_COST ec)                   */
/*                                                                           */
/*  Initialize the rerun cost of e, if it has a rerun cost field.            */
/*                                                                           */
/*****************************************************************************/

/* ***
static void KheDrsExprCostInitRerunCost(KHE_DRS_EXPR_COST ec)
{
  ec->rerun_open_and_search_cost = ec->rerun_open_and_close_cost =
    KheMonitorCost(ec->monitor);
  ** ***
  switch( e->tag )
  {
    case KHE_DRS_EXPR_INT_SUM_COST_TAG:

      KheDrsExprIntSumCostInitRerunCost((KHE_DRS_EXPR_INT_SUM_COST) e);
      break;

    case KHE_DRS_EXPR_INT_SEQ_COST_TAG:

      KheDrsExprIntSeqCostInitRerunCost((KHE_DRS_EXPR_INT_SEQ_COST) e);
      break;

    default:

      ** do nothing **
      break;
  }
  *** **
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsExprDoDebug(KHE_DRS_EXPR e, KHE_DRS_SOLN soln,                */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)                             */
/*                                                                           */
/*  Unindented debug print of that part of e which is specific to it.        */
/*  If soln != NULL, also print e's signature (if any) in soln.              */
/*                                                                           */
/*****************************************************************************/

static void KheDrsExprDoDebug(KHE_DRS_EXPR e, KHE_DRS_SOLN soln,
  KHE_DYNAMIC_RESOURCE_SOLVER drs, FILE *fp)
{
  switch( e->tag )
  {
    case KHE_DRS_EXPR_ASSIGNED_TASK_TAG:

      KheDrsExprAssignedTaskDoDebug((KHE_DRS_EXPR_ASSIGNED_TASK) e,
	soln, drs, fp);
      break;

    case KHE_DRS_EXPR_BUSY_TIME_TAG:

      KheDrsExprBusyTimeDoDebug((KHE_DRS_EXPR_BUSY_TIME) e, soln, drs, fp);
      break;

    case KHE_DRS_EXPR_FREE_TIME_TAG:

      KheDrsExprFreeTimeDoDebug((KHE_DRS_EXPR_FREE_TIME) e, soln, drs, fp);
      break;

    case KHE_DRS_EXPR_WORK_TIME_TAG:

      KheDrsExprWorkTimeDoDebug((KHE_DRS_EXPR_WORK_TIME) e, soln, drs, fp);
      break;

    case KHE_DRS_EXPR_BUSY_DAY_TAG:

      KheDrsExprBusyDayDoDebug((KHE_DRS_EXPR_BUSY_DAY) e, soln, drs, fp);
      break;

    case KHE_DRS_EXPR_FREE_DAY_TAG:

      KheDrsExprFreeDayDoDebug((KHE_DRS_EXPR_FREE_DAY) e, soln, drs, fp);
      break;

    case KHE_DRS_EXPR_WORK_DAY_TAG:

      KheDrsExprWorkDayDoDebug((KHE_DRS_EXPR_WORK_DAY) e, soln, drs, fp);
      break;

    case KHE_DRS_EXPR_OR_TAG:

      KheDrsExprOrDoDebug((KHE_DRS_EXPR_OR) e, soln, drs, fp);
      break;

    case KHE_DRS_EXPR_AND_TAG:

      KheDrsExprAndDoDebug((KHE_DRS_EXPR_AND) e, soln, drs, fp);
      break;

    case KHE_DRS_EXPR_INT_SUM_TAG:

      KheDrsExprIntSumDoDebug((KHE_DRS_EXPR_INT_SUM) e, soln, drs, fp);
      break;

    case KHE_DRS_EXPR_FLOAT_SUM_TAG:

      KheDrsExprFloatSumDoDebug((KHE_DRS_EXPR_FLOAT_SUM) e, soln, drs, fp);
      break;

    case KHE_DRS_EXPR_INT_DEV_TAG:

      KheDrsExprIntDevDoDebug((KHE_DRS_EXPR_INT_DEV) e, soln, drs, fp);
      break;

    case KHE_DRS_EXPR_FLOAT_DEV_TAG:

      KheDrsExprFloatDevDoDebug((KHE_DRS_EXPR_FLOAT_DEV) e, soln, drs, fp);
      break;

    case KHE_DRS_EXPR_INT_SUM_COST_TAG:

      KheDrsExprIntSumCostDoDebug((KHE_DRS_EXPR_INT_SUM_COST) e, soln, drs, fp);
      break;

    case KHE_DRS_EXPR_INT_SEQ_COST_TAG:

      KheDrsExprIntSeqCostDoDebug((KHE_DRS_EXPR_INT_SEQ_COST) e, soln, drs, fp);
      break;

    default:

      HnAbort("KheDrsExprDoDebug internal error: unknown tag %d", e->tag);
      break;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Major category "assignments"                                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_ASST_TO_TASK"                                         */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_ASST_TO_TASK KheDrsAsstToTaskMake(                               */
/*    KHE_DRS_ASST_TO_TASK_CLASS asst, KHE_DRS_TASK_ON_DAY fixed_dtd)        */
/*                                                                           */
/*  Make a new fixed resource assignment object with these attributes.       */
/*  NB there is no memory allocation.                                        */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_ASST_TO_TASK KheDrsAsstToTaskMake(
  KHE_DRS_ASST_TO_TASK_CLASS asst, KHE_DRS_TASK_ON_DAY fixed_dtd)
{
  KHE_DRS_ASST_TO_TASK res;
  res.asst = asst;
  res.fixed_dtd = fixed_dtd;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_SHIFT KheDrsAsstToTaskShift(KHE_DRS_ASST_TO_TASK dat)            */
/*                                                                           */
/*  Return the shift that dat is for, or NULL if none.                       */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_SHIFT KheDrsAsstToTaskShift(KHE_DRS_ASST_TO_TASK dat)
{
  return KheDrsAsstToTaskClassShift(dat.asst);
}


/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_RESOURCE KheDrsAsstToTaskResource(KHE_DRS_ASST_TO_TASK dat)      */
/*                                                                           */
/*  Return the resource that dat assigns.                                    */
/*                                                                           */
/*****************************************************************************/

static KHE_DRS_RESOURCE KheDrsAsstToTaskResource(KHE_DRS_ASST_TO_TASK dat)
{
  return dat.asst->asst_to_shift->resource_on_day->encl_dr;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsAsstToTaskIsFixed(KHE_DRS_ASST_TO_TASK dat)                   */
/*                                                                           */
/*  Return true if dat is a fixed assignment.                                */
/*                                                                           */
/*****************************************************************************/

static bool KheDrsAsstToTaskIsFixed(KHE_DRS_ASST_TO_TASK dat)
{
  KHE_DRS_RESOURCE dr;
  dr = KheDrsAsstToTaskResource(dat);
  /* dr = dat.asst->asst_to_shift->resource_on_day->encl_dr; */
  return dr->expand_role == KHE_DRS_RESOURCE_EXPAND_FIXED;
}


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsAsstToTaskAssignsFreeResource(KHE_DRS_ASST_TO_TASK tasst)     */
/*                                                                           */
/*  Return true if tasst is a fixed assignment.                              */
/*                                                                           */
/*****************************************************************************/

/* *** currently unused
static bool KheDrsAsstToTaskAssignsFreeResource(KHE_DRS_ASST_TO_TASK tasst)
{
  KHE_DRS_RESOURCE dr;
  dr = KheDrsAsstToTaskResource(tasst);
  return dr->expand_role == KHE_DRS_RESOURCE_EXPAND_FREE;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsAsstToTaskClassIsCompatible(KHE_DRS_ASST_TO_TASK_CLASS asst1, */
/*    KHE_DRS_ASST_TO_TASK_CLASS asst2)                                      */
/*                                                                           */
/*  Return true if asst1 and asst2 are compatible, in the sense that         */
/*  they are not on each other's skip lists.  Only one skip list needs       */
/*  to be checked.                                                           */
/*                                                                           */
/*****************************************************************************/

/* *** no longer needed
static bool KheDrsAsstToTaskClassIsCompatible(KHE_DRS_ASST_TO_TASK_CLASS asst1,
  KHE_DRS_ASST_TO_TASK_CLASS asst2)
{
  int pos;
  return !HaArrayContains(asst1->skip_assts, asst2, &pos);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  bool KheDrsAsstToTaskIsCompatible(                                       */
/*    KHE_DRS_ASST_TO_TASK tasst, KHE_DYNAMIC_RESOURCE_SOLVER drs)           */
/*                                                                           */
/*  Return true if tasst is compatible with drs->expand_assts_to_tasks.         */
/*                                                                           */
/*****************************************************************************/

/* *** no longer needed
static bool KheDrsAsstToTaskIsCompatible(
  KHE_DRS_ASST_TO_TASK tasst, KHE_DYNAMIC_RESOURCE_SOLVER drs)
{
  KHE_DRS_ASST_TO_TASK tasst2;  int i;
  HaArrayForEach(drs->expand_assts_to_tasks, tasst2, i)
    if( !KheDrsAsstToTaskClassIsCompatible(tasst.asst, tasst2.asst) )
      return false;
  return true;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsAsstToTaskDebug(KHE_DRS_ASST_TO_TASK dat, int verbosity,      */
/*    int indent, FILE *fp)                                                  */
/*                                                                           */
/*  Debug print of dat onto fp with the given verbosity and indent.          */
/*                                                                           */
/*****************************************************************************/

static void KheDrsAsstToTaskDebug(KHE_DRS_ASST_TO_TASK dat, int verbosity,
  int indent, FILE *fp)
{
  KHE_DRS_RESOURCE dr;
  if( indent > 0 )
    fprintf(fp, "%*s", indent, "");
  dr = KheDrsAsstToTaskResource(dat);
  if( verbosity == 1 )
    fprintf(fp, "<%s, ", KheDrsResourceId(dr));
  else
    fprintf(fp, "<%s %s, ", KheDrsResourceExpandRoleShow(dr->expand_role),
      KheDrsResourceId(dr));
  if( dat.fixed_dtd != NULL )
    KheDrsTaskOnDayDebug(dat.fixed_dtd, 1, -1, fp);
  else
    fprintf(fp, "(free)");
  fprintf(fp, ">");
  if( indent >= 0 )
    fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_ASST_TO_TASK - expansion"                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void KheDrsAsstToTaskShiftAsstTrieBuildAssts(                            */
/*    KHE_DRS_ASST_TO_TASK tasst, KHE_DRS_SHIFT_ASST_TRIE dsat,              */
/*    KHE_DRS_SHIFT ds, KHE_DRS_RESOURCE_SET included_resources,             */
/*    int resource_index, KHE_DRS_SOLN prev_soln, KHE_DRS_DAY next_day,      */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Build shift assignments including tasst and later included_resources.    */
/*                                                                           */
/*****************************************************************************/
static void KheDrsExpanderAddAsstToTask(KHE_DRS_EXPANDER de,
  KHE_DRS_ASST_TO_TASK tasst);
static void KheDrsShiftAsstTrieBuildAssts(KHE_DRS_SHIFT_ASST_TRIE dsat,
  KHE_DRS_SHIFT ds, KHE_DRS_RESOURCE_SET included_free_resources,
  int resource_index, KHE_DRS_SOLN prev_soln, KHE_DRS_DAY prev_day,
  KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de);
  /* KHE_DYNAMIC_RESOURCE_SOLVER drs); */

static void KheDrsAsstToTaskShiftAsstTrieBuildAssts(
  KHE_DRS_ASST_TO_TASK tasst, KHE_DRS_SHIFT_ASST_TRIE dsat,
  KHE_DRS_SHIFT ds, KHE_DRS_RESOURCE_SET included_free_resources,
  int included_free_resources_index, KHE_DRS_SOLN prev_soln,
  KHE_DRS_DAY prev_day, KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de)
{
  /* save the expander so it can be restored later */
  KheDrsExpanderMarkBegin(de);

  /* add tasst to the expander */
  KheDrsExpanderAddAsstToTask(de, tasst);

  /* if the expander is still open, recurse */
  if( KheDrsExpanderIsOpen(de) )
    KheDrsShiftAsstTrieBuildAssts(dsat, ds, included_free_resources,
      included_free_resources_index, prev_soln, prev_day, next_day, de);

  /* restore the expander */
  KheDrsExpanderMarkEnd(de);
}


/*****************************************************************************/
/*                                                                           */
/*  void KheDrsAsstToTaskExpandByResources(KHE_DRS_ASST_TO_TASK tasst,       */
/*    KHE_DRS_SOLN prev_soln, KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de,     */
/*    KHE_DRS_RESOURCE_SET free_resources, int free_resources_index)         */
/*                                                                           */
/*  Add tasst, recurse, then unassign.                                       */
/*                                                                           */
/*****************************************************************************/
static void KheDrsSolnExpandByResources(KHE_DRS_SOLN prev_soln,
  KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de,
  KHE_DRS_RESOURCE_SET free_resources, int free_resources_index);

static void KheDrsAsstToTaskExpandByResources(KHE_DRS_ASST_TO_TASK tasst,
  KHE_DRS_SOLN prev_soln, KHE_DRS_DAY next_day, KHE_DRS_EXPANDER de,
  KHE_DRS_RESOURCE_SET free_resources, int free_resources_index)
{
  /* save the expander so it can be restored later */
  KheDrsExpanderMarkBegin(de);

  /* add tasst to the expander */
  KheDrsExpanderAddAsstToTask(de, tasst);

  /* if the expander is still open, recurse */
  if( KheDrsExpanderIsOpen(de) )
    KheDrsSolnExpandByResources(prev_soln, next_day, de,
      free_resources, free_resources_index + 1);

  /* restore the expander */
  KheDrsExpanderMarkEnd(de);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "KHE_DRS_ASST_TO_TASK_SET"                                     */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  KHE_DRS_ASST_TO_TASK_SET KheDrsAsstToTaskSetMake(                        */
/*    KHE_DYNAMIC_RESOURCE_SOLVER drs)                                       */
/*                                                                           */
/*  Make a new, empty asst to tast set object.                               */
/*                                                                           */
/*********************************************