/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         array.c                                                    */
/*  DESCRIPTION:  Generic extendable arrays (implementation)                 */
/*                                                                           */
/*****************************************************************************/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include "array.h"
#define MAX_RANK 15
#define ASSERT_ON 1


/*****************************************************************************/
/*                                                                           */
/*  int ArrayModuleInitialized                                               */
/*                                                                           */
/*  True if this module has been initialized.                                */
/*                                                                           */
/*****************************************************************************/

static int ArrayModuleInitialized = 0;


/*****************************************************************************/
/*                                                                           */
/*  static int ArrayRanks[MAX_RANK]                                          */
/*                                                                           */
/*  ArrayRanks[r] is the size of arrays of rank r.                           */
/*                                                                           */
/*****************************************************************************/

static int ArrayRanks[MAX_RANK];


/*****************************************************************************/
/*                                                                           */
/*  ArrayInitWithRank(ARRAY(void *) *a, int r, int item_size)                */
/*                                                                           */
/*  Initialize "a" to be an empty array with rank r and the given item size. */
/*                                                                           */
/*****************************************************************************/

void ArrayInitWithRank(ARRAY_VOIDP *a, int r, int item_size)
{
  int i;

  /* printf("[ ArrayInitWithRank(a, %d)\n", r); */

  /* make sure ArrayRanks is initialized */
  if( !ArrayModuleInitialized )
  {
    ArrayRanks[0] = 2;
    ArrayRanks[1] = 5;
    for( i = 2;  i < MAX_RANK;  i++ )
      ArrayRanks[i] = 3*ArrayRanks[i-1] - ArrayRanks[i-2];
    ArrayModuleInitialized = TRUE;
  }

  /* make sure we are getting a legal rank */
  if( r >= MAX_RANK )
  {
    fprintf(stderr, "Array internal error: maximum rank (%d) exceeded\n",
      MAX_RANK - 1);
    assert(FALSE);
  }

  /* allocate memory for *a and initialize its fields */
  *a = (ARRAY_VOIDP)  malloc(sizeof(**a));
  if( *a == NULL )
  {
    fprintf(stderr, "Array internal error: out of memory\n");
    assert(FALSE);
  }
  (*a)->rank = r;
  (*a)->avail_size = ArrayRanks[(*a)->rank];
  (*a)->size = 0;
  (*a)->item_size = item_size;
  (*a)->items = (void **) malloc((*a)->avail_size * (*a)->item_size);
  if( (*a)->items == NULL )
  {
    fprintf(stderr, "Array internal error: out of memory\n");
    assert(FALSE);
  }
  /* printf("] ArrayInitWithRank returning\n"); */
}


/*****************************************************************************/
/*                                                                           */
/*  ArrayVoidpFree(ARRAY_VOIDP a)                                            */
/*                                                                           */
/*  Free up the memory used within array a.                                  */
/*                                                                           */
/*****************************************************************************/

void ArrayVoidpFree(ARRAY_VOIDP a)
{
  /* ***
  free(a->items);
  free(a);
  *** */
}


/*****************************************************************************/
/*                                                                           */
/*  ArrayExtend(ARRAY(void *) a)                                             */
/*                                                                           */
/*  Make array a longer.  Return value is not used.                          */
/*                                                                           */
/*****************************************************************************/

int ArrayExtend(ARRAY_VOIDP a)
{
  a->rank += 1;

  /* make sure we are getting a legal rank */
  if( a->rank >= MAX_RANK )
  {
    fprintf(stderr, "Array internal error: maximum rank (%d) exceeded\n",
      MAX_RANK - 1);
    assert(FALSE);
  }

  a->avail_size = ArrayRanks[a->rank];
  a->items = realloc(a->items, a->avail_size * a->item_size);
  if( a->items == NULL )
  {
    fprintf(stderr, "Array internal error: out of memory\n");
    assert(FALSE);
  }
  a->avail_size = ArrayRanks[a->rank];
  return 0;
}


/*****************************************************************************/
/*                                                                           */
/*  ArrayCycleLeft(ARRAY(void *) a)                                          */
/*                                                                           */
/*  Move the elements of a cyclically left by one step:  the first becomes   */
/*  the last, and the others all move left one step.                         */
/*                                                                           */
/*  This is the basis of our implementation of ArrayRemoveFirst.             */
/*                                                                           */
/*****************************************************************************/

int ArrayCycleLeft(ARRAY_VOIDP a)
{
  int i;  void *first;
  if( a->size > 0 )
  {
    first = a->items[0];
    for( i = 1;  i < a->size;  i++ )
      a->items[i-1] = a->items[i];
    a->items[i-1] = first;
  }
  return 0;
}


/*****************************************************************************/
/*                                                                           */
/*  ArrayCycleRight(ARRAY(void *) a)                                         */
/*                                                                           */
/*  Move the elements of a cyclically right by one step:  the last becomes   */
/*  the first, and the others all move right one step.                       */
/*                                                                           */
/*  This is the basis of our implementation of ArrayAddFirst.                */
/*                                                                           */
/*****************************************************************************/

int ArrayCycleRight(ARRAY_VOIDP a)
{
  int i;  void *last;
  if( a->size > 0 )
  {
    last = a->items[a->size - 1];
    for( i = a->size - 1;  i >= 1;  i-- )
      a->items[i] = a->items[i-1];
    a->items[0] = last;
  }
  return 0;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ArrayVoidpContains(ARRAY_VOIDP a, void *x, int *index)           */
/*                                                                           */
/*  Return TRUE if x is present in a, and set *index to its index.           */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ArrayVoidpContains(ARRAY_VOIDP a, void *x, int *index)
{
  int i;
  for( i = 0;  i < ArraySize(a);  i++ )
    if( ArrayGet(a, i) == x )
    {
      *index = i;
      return TRUE;
    }
  *index = -1;
  return FALSE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ArrayVoidpContainsDeep(ARRAY_VOIDP a, void *x,                   */
/*    int(*compar)(void *t1, void *t2), int *index)                          */
/*                                                                           */
/*  Return TRUE if x is present in a, and set *index to its index.           */
/*  Use *compar (a comparison function of the kind passed to qsort)          */
/*  to determine whether two elements are equal or not.                      */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ArrayVoidpContainsDeep(ARRAY_VOIDP a, void *x,
  int(*compar)(const void *t1, const void *t2), int *index)
{
  int i;
  for( i = 0;  i < ArraySize(a);  i++ )
    if( compar(&ArrayGet(a, i), &x) == 0 )
    {
      *index = i;
      return TRUE;
    }
  *index = -1;
  return FALSE;
}


/*****************************************************************************/
/*                                                                           */
/*  void ArrayVoidpUnique(ARRAY_VOIDP a,                                     */
/*    int(*compar)(const void *t1, const void *t2))                          */
/*                                                                           */
/*  Reduce a in size so that it does not contain any adjacent elements       */
/*  that compare 0 using compar (a comparison function of the kind passed    */
/*  to qsort).  It works best immediately after a has been sorted using      */
/*  compar; then it ensures that no element appears twice.                   */
/*                                                                           */
/*  Implementation note.  The loop invariant of this function includes       */
/*                                                                           */
/*      a[i] exists and a[0..i] contains the unique elements of the          */
/*      original a[0..j-1].                                                  */
/*                                                                           */
/*****************************************************************************/

void ArrayVoidpUnique(ARRAY_VOIDP a,
  int(*compar)(const void *t1, const void *t2))
{
  int i, j;

  /* nothing to do on any empty array, and special case too */
  if( ArraySize(a) == 0 )
    return;

  /* eliminate duplicates */
  i = 0;
  for( j = 1;  j < ArraySize(a);  j++ )
  {
    if( compar(&ArrayGet(a, i), &ArrayGet(a, j)) != 0 )
    {
      i++;
      ArrayPut(a, i, ArrayGet(a, j));
    }
  }

  /* truncate to the smaller size */
  a->size = i + 1;
}


/*****************************************************************************/
/*                                                                           */
/*  void ArrayInsertElement(ARRAY_VOIDP a, int i, void *x)                   */
/*                                                                           */
/*  Insert x at position i, shuffling the rest up one space.                 */
/*                                                                           */
/*****************************************************************************/

void ArrayInsertElement(ARRAY_VOIDP a, int i, void *x)
{
  int j;

  /* first add x to end to make the array the right size */
  ArrayAddLast(a, x);

  /* now shuffle everyone up to make a space at position i */
  for( j = ArraySize(a) - 1;  j > i;  j-- )
    ArrayPut(a, j, ArrayGet(a, j-1));
  ArrayPut(a, i, x);
}


/*****************************************************************************/
/*                                                                           */
/*  void ArrayRemoveElement(ARRAY_VOIDP a, int i)                            */
/*                                                                           */
/*  Remove the ith element, shuffling later entries down to close the gap.   */
/*                                                                           */
/*****************************************************************************/

void ArrayRemoveElement(ARRAY_VOIDP a, int i)
{
  int j;
  for( j = i+1;  j < ArraySize(a);  j++ )
    ArrayPut(a, j-1, ArrayGet(a, j));
  ArrayDropLast(a);
}


/*****************************************************************************/
/*                                                                           */
/*  void ArrayIntAccumulate(ARRAY_INT dest, ARRAY_INT source)                */
/*                                                                           */
/*  Accumulate source into dest component by component.  The two arrays      */
/*  must have equal length.                                                  */
/*                                                                           */
/*****************************************************************************/

void ArrayIntAccumulate(ARRAY_INT dest, ARRAY_INT source)
{
  int i, x;
  assert(ArraySize(dest) == ArraySize(source));
  for( i = 0;  i < ArraySize(dest);  i++ )
  {
    x = ArrayGet(dest, i) + ArrayGet(source, i);
    ArrayPut(dest, i, x);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  ASTRING ArrayIntShow(ARRAY_INT array_int)                                */
/*                                                                           */
/*  Return a static string representation of this array of integers.         */
/*                                                                           */
/*****************************************************************************/
#define end_buff_bp &buff[bp][strlen(buff[bp])]

ASTRING ArrayIntShow(ARRAY_INT a)
{
  static int bp;
  static char buff[4][200];  int i;
  bp = (bp + 1) % 4;
  sprintf(buff[bp], "[");
  for( i = 0;  i < ArraySize(a);  i++ )
  {
    if( i > 0 )
      sprintf(end_buff_bp, ", ");
    sprintf(end_buff_bp, "%d", ArrayGet(a, i));
  }
  sprintf(end_buff_bp, "]");
  return buff[bp];
}


/*****************************************************************************/
/*                                                                           */
/*  ASTRING ArrayBooleanShow(ARRAY_BOOLEAN a)                                */
/*                                                                           */
/*  Return a static string representation of this array of booleans.         */
/*                                                                           */
/*****************************************************************************/

ASTRING ArrayBooleanShow(ARRAY_BOOLEAN a)
{
  static int bp;
  static char buff[4][100];  int i;
  bp = (bp + 1) % 4;
  strcpy(buff[bp], "");
  for( i = 0;  i < ArraySize(a);  i++ )
    sprintf(end_buff_bp, "%c", ArrayGet(a, i) ? 'T' : 'F');
  return buff[bp];
}
