
/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         array.h                                                    */
/*  DESCRIPTION:  Generic extendable arrays (header file)                    */
/*                                                                           */
/*  To create an array type it is best to use a typedef such as              */
/*                                                                           */
/*       typedef ARRAY(int) ARRAY_INT;                                       */
/*                                                                           */
/*  And then use ARRAY_INT wherever an array of integers is required.        */
/*  The supplied type can be any size (a struct, etc.) - its does not have   */
/*  to be one word.  Three such types are predefined in this file:           */
/*                                                                           */
/*       typedef ARRAY(void *)  ARRAY_VOIDP;                                 */
/*       typedef ARRAY(int)     ARRAY_INT;                                   */
/*       typedef ARRAY(BOOLEAN) ARRAY_BOOLEAN;                               */
/*                                                                           */
/*  Where it says ARRAY(TYPE) in these methods, use ARRAY_INT etc.           */
/*                                                                           */
/*  Implementation note.  This implementation of generic extendable arrays   */
/*  uses the C preprocessor to create a generic type ARRAY(TYPE) and most    */
/*  of its operations.  When an array has to extend beyond its current       */
/*  available size, its contents are copied to a new array of next largest   */
/*  size (about 2.6 times larger), then the old one is freed.                */
/*                                                                           */
/*****************************************************************************/
#ifndef ARRAY_HEADER_FILE
#define ARRAY_HEADER_FILE
#include "boolean.h"
#include "utypes.h"

/*****************************************************************************/
/*                                                                           */
/*  ARRAY(TYPE)                                                              */
/*                                                                           */
/*****************************************************************************/

#define ARRAY(TYPE)							\
  struct {								\
    int rank;		/* rank of array, i.e. which size group	*/	\
    int avail_size;	/* size available without enlargement	*/	\
    int size;		/* current number of elements in array	*/	\
    int item_size;	/* size in bytes of TYPE		*/	\
    int cursor;		/* internal cursor used by traversal    */	\
    TYPE *items;	/* the array of items			*/	\
  } *									\


/*****************************************************************************/
/*                                                                           */
/*  ARRAY_VOIDP                                                              */
/*                                                                           */
/*****************************************************************************/

typedef ARRAY(void *)   ARRAY_VOIDP;
typedef ARRAY(int)	ARRAY_INT;
typedef ARRAY(BOOLEAN)	ARRAY_BOOLEAN;


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ArrayInRange(ARRAY(TYPE) a, int i)                               */
/*                                                                           */
/*  TRUE if i is currently a legal index for array a.                        */
/*                                                                           */
/*****************************************************************************/

#define ArrayInRange(a, i)	((i) >= 0 && (i) < (a)->size)


/*****************************************************************************/
/*                                                                           */
/*  TYPE ArrayGet(ARRAY(TYPE) a, int i)                                      */
/*                                                                           */
/*  Returns the i'th element of a, without checking array bounds.            */
/*                                                                           */
/*****************************************************************************/

#define ArrayGet(a, i)	((a)->items[i])


/*****************************************************************************/
/*                                                                           */
/*  TYPE ArrayFirst(ARRAY(TYPE) a)                                           */
/*  TYPE ArrayLast(ARRAY(TYPE) a)                                            */
/*                                                                           */
/*  Returns the first/last element of a, without checking array bounds.      */
/*                                                                           */
/*****************************************************************************/

#define ArrayFirst(a)	(a)->items[0]
#define ArrayLast(a)	(a)->items[(a)->size - 1]


/*****************************************************************************/
/*                                                                           */
/*  TYPE ArrayPut(ARRAY(TYPE) a, int i, TYPE x)                              */
/*                                                                           */
/*  Replaces the i'th element of a by x, without checking array bounds.      */
/*                                                                           */
/*****************************************************************************/

#define ArrayPut(a, i, x)	((a)->items[i] = (x))


/*****************************************************************************/
/*                                                                           */
/*  int ArraySize(ARRAY(TYPE) a)                                             */
/*                                                                           */
/*  Returns the current size (number of elements) of a.                      */
/*                                                                           */
/*****************************************************************************/

#define ArraySize(a)	(a)->size


/*****************************************************************************/
/*                                                                           */
/*  void ArrayClear(ARRAY(TYPE) a)                                           */
/*                                                                           */
/*  Set the size of existing array a to 0.                                   */
/*                                                                           */
/*****************************************************************************/

#define ArrayClear(a)	((a)->size = 0)


/*****************************************************************************/
/*                                                                           */
/*  void ArrayAddLast(ARRAY(TYPE) a, TYPE x)                                 */
/*  void ArrayAddFirst(ARRAY(TYPE) a, TYPE x)                                */
/*                                                                           */
/*  Append x to the end of a, resizing the array if necessary.               */
/*                                                                           */
/*****************************************************************************/

#define ArrayAddLast(a, x)						\
(									\
  (a)->size == (a)->avail_size ? ArrayExtend((ARRAY_VOIDP) (a)) : 0,	\
  (a)->items[(a)->size++] = (x)						\
)

#define ArrayAddFirst(a, x)						\
(ArrayAddLast(a, x), ArrayCycleRight((ARRAY_VOIDP) (a)))

/*****************************************************************************/
/*                                                                           */
/*  void ArrayEnsureLegalIndex(ARRAY(TYPE) a, int i, int sanity, TYPE x)     */
/*                                                                           */
/*  Add x repeatedly to the end of a until i is a legal index, but not       */
/*  beyond sanity.                                                           */
/*                                                                           */
/*****************************************************************************/

#define ArrayEnsureLegalIndex(a, i, sanity, x)				\
while( (i) < (sanity) && !ArrayInRange((a), (i)) )  ArrayAddLast((a), (x))


/*****************************************************************************/
/*                                                                           */
/*  void ArraySafePut(ARRAY(TYPE) a, int i, int sanity, TYPE fill, TYPE x)   */
/*                                                                           */
/*  Replaces the i'th element of a by x, expanding if needed.                */
/*                                                                           */
/*****************************************************************************/

#define ArraySafePut(a, i, sanity, fill,x)				\
{									\
  ArrayEnsureLegalIndex(a, i, sanity, fill);				\
  ArrayPut(a, i, x);							\
}


/*****************************************************************************/
/*                                                                           */
/*  void ArrayInsert(ARRAY(TYPE) a, int i, TYPE x)                           */
/*                                                                           */
/*  Insert x at position i, shuffling the others up one place to make room.  */
/*  Here i may be ArraySize(a), in which case x goes at the end.             */
/*                                                                           */
/*****************************************************************************/

#define ArrayInsert(a, i, x)						\
  ArrayInsertElement((ARRAY_VOIDP) (a), i, (void *) (x))


/*****************************************************************************/
/*                                                                           */
/*  void ArrayRemove(ARRAY(TYPE) a, int i)                                   */
/*                                                                           */
/*  Remove the ith element, shuffling the others down.                       */
/*                                                                           */
/*****************************************************************************/

#define ArrayRemove(a, i)    ArrayRemoveElement((ARRAY_VOIDP) (a), (i))


/*****************************************************************************/
/*                                                                           */
/*  TYPE ArrayRemoveLast(ARRAY(TYPE) a)                                      */
/*  TYPE ArrayRemoveFirst(ARRAY(TYPE) a)                                     */
/*                                                                           */
/*  Remove and return the last/first elt of a, without checking non-empty.   */
/*                                                                           */
/*****************************************************************************/

#define ArrayRemoveLast(a)   (a)->items[--(a)->size]
#define ArrayRemoveFirst(a)  (ArrayCycleLeft((ARRAY_VOIDP) (a)), ArrayRemoveLast(a))


/*****************************************************************************/
/*                                                                           */
/*  void ArrayDropLast(ARRAY(TYPE) a)                                        */
/*                                                                           */
/*  Remove the last elt of a, without checking non-empty.                    */
/*                                                                           */
/*****************************************************************************/

#define ArrayDropLast(a)     ((a)->size--)


/*****************************************************************************/
/*                                                                           */
/*  void ArrayDropFromEnd(ARRAY(TYPE) a, int n)                              */
/*                                                                           */
/*  Remove the last n elements of a, without checking non-empty.             */
/*                                                                           */
/*****************************************************************************/

#define ArrayDropFromEnd(a, n)     ((a)->size -= (n))


/*****************************************************************************/
/*                                                                           */
/*  void ArrayInit(ARRAY(TYPE) *a)                                           */
/*                                                                           */
/*  Initialize a to an empty array of rank 1.                                */
/*                                                                           */
/*****************************************************************************/

#define ArrayInit(a)							\
  ArrayInitWithRank( (ARRAY_VOIDP *) (a), 1, sizeof((*(a))->items[0]) )


/*****************************************************************************/
/*                                                                           */
/*  void ArrayFresh(ARRAY(TYPE) a)                                           */
/*                                                                           */
/*  If a is NULL, initialize it, otherwise clear it.  Helps when reusing     */
/*  static arrays initialized to NULL.                                       */
/*                                                                           */
/*****************************************************************************/

#define ArrayFresh(a)							\
  if( (a) == NULL ) ArrayInit(&(a)); else ArrayClear(a)


/*****************************************************************************/
/*                                                                           */
/*  void ArrayFree(ARRAY(TYPE) a)                                            */
/*                                                                           */
/*  Free the memory used by array a.                                         */
/*                                                                           */
/*****************************************************************************/

#define ArrayFree(a) ArrayVoidpFree( (ARRAY_VOIDP) a )


/*****************************************************************************/
/*                                                                           */
/*  void ArrayFill(ARRAY(TYPE) a, TYPE x)                                    */
/*                                                                           */
/*  Fill a (up to avail_size) with x.                                        */
/*                                                                           */
/*****************************************************************************/

#define ArrayFill(a, x)							\
{ int zzz_yukko;							\
  a->size = a->avail_size;						\
  for( zzz_yukko = 0;  zzz_yukko < a->avail_size;  zzz_yukko++ )	\
    a->items[zzz_yukko] = x;						\
}


/*****************************************************************************/
/*                                                                           */
/*  ArraySetFirst(a)                                                         */
/*  ArraySetLast(a)                                                          */
/*  ArrayNext(a)                                                             */
/*  ArrayPrev(a)                                                             */
/*  ArrayCurrent(a)                                                          */
/*  ArrayCurrentPos(a)                                                       */
/*  ArrayOffEnd(a)                                                           */
/*  ArrayOffStart(a)                                                         */
/*  ArrayNotOffEndAndCurrent(a, x)                                           */
/*  ArrayNotOffStartAndCurrent(a, x)                                         */
/*  ArrayForEach(a, x)                                                       */
/*  ArrayForEachReverse(a, x)                                                */
/*                                                                           */
/*****************************************************************************/

#define ArraySetFirst(a)  ((a)->cursor = 0)
#define ArraySetLast(a)   ((a)->cursor = (a)->size - 1)
#define ArrayNext(a)	  ((a)->cursor++)
#define ArrayPrev(a)	  ((a)->cursor--)
#define ArrayCurrent(a)	  ArrayGet((a), (a)->cursor)
#define ArrayCurrentPos(a) ((a)->cursor)
#define ArrayOffEnd(a)	  ((a)->cursor >= (a)->size)
#define ArrayOffStart(a)  ((a)->cursor < 0)

#define ArrayNotOffEndAndCurrent(a, x)					\
  ArrayOffEnd(a) ? FALSE : ( (x) = ArrayCurrent(a), TRUE )

#define ArrayNotOffStartAndCurrent(a, x)				\
  ArrayOffStart(a) ? FALSE : ( (x) = ArrayCurrent(a), TRUE )

#define ArrayForEach(a, x)						\
  for( ArraySetFirst(a);  ArrayNotOffEndAndCurrent(a, x);  ArrayNext(a) )

#define ArrayForEachReverse(a, x)					\
  for( ArraySetLast(a);  ArrayNotOffStartAndCurrent(a, x);  ArrayPrev(a) )

/*****************************************************************************/
/*                                                                           */
/*  ArrayAppend(dest, source)                                                */
/*                                                                           */
/*  Append source to the end of dest; both must be non-null.                 */
/*                                                                           */
/*****************************************************************************/

#define ArrayAppend(dest, source)					\
for( ArraySetFirst(source); !ArrayOffEnd(source);  ArrayNext(source) )	\
  ArrayAddLast((dest), ArrayCurrent(source))


/*****************************************************************************/
/*                                                                           */
/*  ArrayCopy(dest, source)                                                  */
/*                                                                           */
/*  Shallow copy of source onto dest.  Any previous dest is lost.            */
/*                                                                           */
/*****************************************************************************/

#define ArrayCopy(dest, source)						\
ArrayInit(&(dest));							\
ArrayAppend(dest, source)


/*****************************************************************************/
/*                                                                           */
/*  ArraySwap(ARRAY(TYPE) a, int i, int j, TYPE tmp)                         */
/*                                                                           */
/*  Swap a[i] with a[j], using tmp as the temporary variable.                */
/*                                                                           */
/*****************************************************************************/

#define ArraySwap(a, i, j, tmp)						\
(tmp = ArrayGet(a, i), ArrayPut(a, i, ArrayGet(a, j)), ArrayPut(a, j, tmp))


/*****************************************************************************/
/*                                                                           */
/*  ArraySort(a, function)                                                   */
/*                                                                           */
/*  Sort a using the given comparison function, which is as for qsort.       */
/*                                                                           */
/*****************************************************************************/

#define ArraySort(a, function)						\
  qsort((a)->items, (a)->size, (a)->item_size, function)


/*****************************************************************************/
/*                                                                           */
/*  ArraySortUnique(a, function)                                             */
/*                                                                           */
/*  Like ArraySort except that the final array will be reduced in size       */
/*  to ensure that no two adjacent elements compare 0 with function.         */
/*                                                                           */
/*****************************************************************************/

#define ArraySortUnique(a, function)					\
{									\
  qsort((a)->items, (a)->size, (a)->item_size, function);		\
  ArrayVoidpUnique((ARRAY_VOIDP) a, function);				\
}


/*****************************************************************************/
/*                                                                           */
/*  ArraySortFirst(a, lim, function)                                         */
/*                                                                           */
/*  Sort the first lim elements of a using the given comparison function,    */
/*  which is as for qsort.                                                   */
/*                                                                           */
/*****************************************************************************/

#define ArraySortFirst(a, lim, function)				\
  qsort((a)->items, (lim), (a)->item_size, function)


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ArrayContains(ARRAY(TYPE) a, TYPE x, int *index)                 */
/*                                                                           */
/*  Return non-zero if a contains x, and set *index to its first occurrence. */
/*                                                                           */
/*****************************************************************************/

#define ArrayContains(a, x, index)					\
  ArrayVoidpContains((ARRAY_VOIDP) a, (void *) x, index)


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ArrayContainsDeep(ARRAY(TYPE) a, TYPE x,                         */
/*    int(*compar)(void *, void *), int *index)                              */
/*                                                                           */
/*  Return non-zero if a contains x, and set *index to its first occurrence. */
/*  Here a contains x if *compar returns 0.                                  */
/*                                                                           */
/*****************************************************************************/

#define ArrayContainsDeep(a, x, compar, index)				\
  ArrayVoidpContainsDeep((ARRAY_VOIDP) a, (void *) x, compar, index)


/*****************************************************************************/
/*                                                                           */
/*  External symbols of array.c.  ArrayExtend and ArrayVoidpFree are not     */
/*  intended to be called directly.                                          */
/*                                                                           */
/*****************************************************************************/

extern int ArrayCycleLeft(ARRAY_VOIDP a);
extern int ArrayCycleRight(ARRAY_VOIDP a);
extern void ArrayInitWithRank(ARRAY_VOIDP *a, int r, int item_size);
extern int ArrayExtend(ARRAY_VOIDP a);
extern void ArrayVoidpFree(ARRAY_VOIDP a);
extern BOOLEAN ArrayVoidpContains(ARRAY_VOIDP a, void *x, int *index);
extern BOOLEAN ArrayVoidpContainsDeep(ARRAY_VOIDP a, void *x,
  int(*compar)(const void *t1, const void *t2), int *index);
extern void ArrayVoidpUnique(ARRAY_VOIDP a,
  int(*compar)(const void *t1, const void *t2));
extern void ArrayInsertElement(ARRAY_VOIDP a, int i, void *x);
extern void ArrayRemoveElement(ARRAY_VOIDP a, int i);
extern void ArrayIntAccumulate(ARRAY_INT dest, ARRAY_INT source);
extern ASTRING ArrayIntShow(ARRAY_INT array_int);
extern ASTRING ArrayBooleanShow(ARRAY_BOOLEAN array_boolean);

#endif
