/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         ustring.h                                                  */
/*  DESCRIPTION:  32-bit Unicode strings (header file)                       */
/*                                                                           */
/*  This code assumes that all the calls to memory allocation succeed, but   */
/*  is otherwise secure (no array overruns etc.).                            */
/*                                                                           */
/*****************************************************************************/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include "memory.h"
#include "astring.h"
#include "ustring.h"
#define AFACTORY_COUNT 12


/*****************************************************************************/
/*                                                                           */
/*  Submodule "operations for constructing temporary AStrings via a factory" */
/*                                                                           */
/*****************************************************************************/

struct afactory_rec {
  ASTRING	value;				/* string being built	     */
  int		max_len;			/* available space in value  */
  int		curr_len;			/* current length of value   */
};

typedef ARRAY(AFACTORY) ARRAY_AFACTORY;

static ARRAY_AFACTORY	factories = NULL;	/* the factories             */
static int		next_factory;		/* factory to use next       */


/*****************************************************************************/
/*                                                                           */
/*  AFACTORY AStringBegin(void)                                              */
/*                                                                           */
/*  Start building a new AString in the returned factory.                    */
/*                                                                           */
/*****************************************************************************/

AFACTORY AStringBegin(void)
{
  int i;  AFACTORY af;

  /* initialize factories and next_factory if not done yet */
  if( factories == NULL )
  {
    ArrayInit(&factories);
    for( i = 0;  i < AFACTORY_COUNT;  i++ )
    {
      GetMemory(af, AFACTORY);
      af->value = (ASTRING) malloc(128 * sizeof(char));
      af->curr_len = 0;
      af->max_len = 128;
      ArrayAddLast(factories, af);
    }
    next_factory = 0;
  }

  /* find a factory and reset it */
  next_factory = (next_factory + 1) % AFACTORY_COUNT;
  af = ArrayGet(factories, next_factory);
  af->curr_len = 0;
  return af;
}


/*****************************************************************************/
/*                                                                           */
/*  void ensure_avail(AFACTORY af, int size)                                 */
/*                                                                           */
/*  Ensure that af has enough space available to be able to accept another   */
/*  size characters.                                                         */
/*                                                                           */
/*****************************************************************************/

static void ensure_avail(AFACTORY af, int size)
{
  int new_max_len;
  if( af->max_len < af->curr_len + size )
  {
    new_max_len = af->max_len * 2;
    while( new_max_len < af->curr_len + size )
      new_max_len *= 2;
    af->max_len = new_max_len;
    af->value = realloc(af->value, new_max_len * sizeof(char));
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void AStringAddChar(AFACTORY af, char ch)                                */
/*                                                                           */
/*  Add ch to the end of the string being built in af.                       */
/*                                                                           */
/*****************************************************************************/

void AStringAddChar(AFACTORY af, char ch)
{
  ensure_avail(af, 1);
  af->value[af->curr_len++] = ch;
}


/*****************************************************************************/
/*                                                                           */
/*  void AStringAddInt(AFACTORY af, int value)                               */
/*                                                                           */
/*  Add integer value, displayed in decimal, to the end of the string        */
/*  being built in af.                                                       */
/*                                                                           */
/*****************************************************************************/

void AStringAddInt(AFACTORY af, int value)
{
  ensure_avail(af, 10);
  sprintf(&af->value[af->curr_len], "%d", value);
  af->curr_len = strlen(af->value);
}


/*****************************************************************************/
/*                                                                           */
/*  void AStringAddHex(AFACTORY af, unsigned int hex)                        */
/*                                                                           */
/*  Add integer hex, displayed in hexadecimal, to the end of the string      */
/*  being built in af.                                                       */
/*                                                                           */
/*****************************************************************************/

void AStringAddHex(AFACTORY af, unsigned int hex)
{
  ensure_avail(af, 5);
  sprintf(&af->value[af->curr_len], "%04X", hex);
  af->curr_len = strlen(af->value);
}


/*****************************************************************************/
/*                                                                           */
/*  void AStringAddAString(AFACTORY af, ASTRING str)                         */
/*                                                                           */
/*  Add str to the end of the string being built in af.                      */
/*                                                                           */
/*****************************************************************************/

void AStringAddAString(AFACTORY af, ASTRING str)
{
  int len = strlen(str);
  ensure_avail(af, len);
  strcpy(&af->value[af->curr_len], str);
  af->curr_len += len;
}


/*****************************************************************************/
/*                                                                           */
/*  void AStringAddFmt1(AFACTORY af, ASTRING fmt, ASTRING p1)                */
/*                                                                           */
/*  Add sprintf(fmt, p1) to the end of the string being built in af.         */
/*                                                                           */
/*****************************************************************************/

void AStringAddFmt1(AFACTORY af, ASTRING fmt, ASTRING p1)
{
  char buff[256];
  sprintf(buff, fmt, p1);
  AStringAddAString(af, buff);
}


/*****************************************************************************/
/*                                                                           */
/*  void AStringAddFmt2(AFACTORY af, ASTRING fmt, ASTRING p1, ASTRING p2)    */
/*                                                                           */
/*  Add sprintf(fmt, p1, p2) to the end of the string being built in af.     */
/*                                                                           */
/*****************************************************************************/

void AStringAddFmt2(AFACTORY af, ASTRING fmt, ASTRING p1, ASTRING p2)
{
  char buff[256];
  sprintf(buff, fmt, p1, p2);
  AStringAddAString(af, buff);
}


/*****************************************************************************/
/*                                                                           */
/*  void AStringAddFmt3(AFACTORY af, ASTRING fmt, ASTRING p1, ASTRING p2,    */
/*    ASTRING p3)                                                            */
/*                                                                           */
/*  Add sprintf(fmt, p1, p2, p3) to the end of the string being built in af. */
/*                                                                           */
/*****************************************************************************/

void AStringAddFmt3(AFACTORY af, ASTRING fmt, ASTRING p1, ASTRING p2,
  ASTRING p3)
{
  char buff[256];
  sprintf(buff, fmt, p1, p2, p3);
  AStringAddAString(af, buff);
}


/*****************************************************************************/
/*                                                                           */
/*  void AStringAddFmt4(AFACTORY af, ASTRING fmt, ASTRING p1, ASTRING p2,    */
/*    ASTRING p3, ASTRING p4)                                                */
/*                                                                           */
/*  Add sprintf(fmt, p1, p2, p3, p4) to the end of the string being built    */
/*  in af.                                                                   */
/*                                                                           */
/*****************************************************************************/

void AStringAddFmt4(AFACTORY af, ASTRING fmt, ASTRING p1, ASTRING p2,
  ASTRING p3, ASTRING p4)
{
  char buff[256];
  sprintf(buff, fmt, p1, p2, p3, p4);
  AStringAddAString(af, buff);
}


/*****************************************************************************/
/*                                                                           */
/*  void AStringAddFmt5(AFACTORY af, ASTRING fmt, ASTRING p1, ASTRING p2,    */
/*    ASTRING p3, ASTRING p4, ASTRING p5)                                    */
/*                                                                           */
/*****************************************************************************/

void AStringAddFmt5(AFACTORY af, ASTRING fmt, ASTRING p1, ASTRING p2,
  ASTRING p3, ASTRING p4, ASTRING p5)
{
  char buff[256];
  sprintf(buff, fmt, p1, p2, p3, p4, p5);
  AStringAddAString(af, buff);
}


/*****************************************************************************/
/*                                                                           */
/*  ASTRING AStringEnd(AFACTORY af)                                          */
/*                                                                           */
/*  End the construction of the string being built in af, and return that    */
/*  string.  It does not occupy permanent memory; it will last for some      */
/*  unspecified time, until its factory gets re-used.  To make a permanent   */
/*  string, use AStringCopy(AStringEnd(af)).                                 */
/*                                                                           */
/*****************************************************************************/

ASTRING AStringEnd(AFACTORY af)
{
  ensure_avail(af, 1);
  af->value[af->curr_len++] = '\0';
  return af->value;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "operations for constructing permanent AStrings"               */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  ASTRING AStringCat(ASTRING str1, ASTRING str2)                           */
/*                                                                           */
/*  Return the concatenation of str1 with str2.                              */
/*                                                                           */
/*****************************************************************************/

ASTRING AStringCat(ASTRING str1, ASTRING str2)
{
  int i, j;
  ASTRING res = (ASTRING)
    malloc((strlen(str1) + strlen(str2) + 1)*sizeof(char));
  for( i = 0, j = 0;  str1[j] != '\0';  i++, j++ )
    res[i] = str1[j];
  for(        j = 0;  str2[j] != '\0';  i++, j++ )
    res[i] = str2[j];
  res[i] = '\0';
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  ASTRING AStringCat3(ASTRING str1, ASTRING str2, ASTRING str3)            */
/*                                                                           */
/*  Return the concatenation of str1 with str2 and str3.                     */
/*                                                                           */
/*****************************************************************************/

ASTRING AStringCat3(ASTRING str1, ASTRING str2, ASTRING str3)
{
  int i, j;
  ASTRING res = (ASTRING)
    malloc( (strlen(str1) + strlen(str2) +
	     strlen(str3) + 1) * sizeof(char));
  for( i = 0, j = 0;  str1[j] != '\0';  i++, j++ )
    res[i] = str1[j];
  for(        j = 0;  str2[j] != '\0';  i++, j++ )
    res[i] = str2[j];
  for(        j = 0;  str3[j] != '\0';  i++, j++ )
    res[i] = str3[j];
  res[i] = '\0';
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  ASTRING AStringCat4(ASTRING str1, ASTRING str2, ASTRING str3, str4)      */
/*                                                                           */
/*  Return the concatenation of str1 with str2, str3, and str4.              */
/*                                                                           */
/*****************************************************************************/

ASTRING AStringCat4(ASTRING str1, ASTRING str2, ASTRING str3, ASTRING str4)
{
  int i, j;
  ASTRING res = (ASTRING)
    malloc( (strlen(str1) + strlen(str2) +
       strlen(str3) + strlen(str4) + 1) * sizeof(char));
  for( i = 0, j = 0;  str1[j] != '\0';  i++, j++ )
    res[i] = str1[j];
  for(        j = 0;  str2[j] != '\0';  i++, j++ )
    res[i] = str2[j];
  for(        j = 0;  str3[j] != '\0';  i++, j++ )
    res[i] = str3[j];
  for(        j = 0;  str4[j] != '\0';  i++, j++ )
    res[i] = str4[j];
  res[i] = 0;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  ASTRING AStringCopy(ASTRING str)                                         */
/*                                                                           */
/*  Return a malloced copy of str.                                           */
/*                                                                           */
/*****************************************************************************/

ASTRING AStringCopy(ASTRING str)
{
  int i;
  ASTRING res = (ASTRING) malloc((strlen(str) + 1)*sizeof(char));
  for( i = 0;  str[i] != '\0';  i++ )
    res[i] = str[i];
  res[i] = 0;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "miscellaneous"                                                */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN AStringEqual(ASTRING str1, ASTRING str2)                         */
/*                                                                           */
/*  Return TRUE if str1 and str2 are equal.                                  */
/*                                                                           */
/*****************************************************************************/

BOOLEAN AStringEqual(ASTRING str1, ASTRING str2)
{
  return strcmp(str1, str2) == 0;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN AStringBeginsWith(ASTRING str1, ASTRING str2)                    */
/*                                                                           */
/*  Return TRUE if str1 begins with str2.                                    */
/*                                                                           */
/*****************************************************************************/

BOOLEAN AStringBeginsWith(ASTRING str1, ASTRING str2)
{
  return strstr(str1, str2) == str1;
}


/*****************************************************************************/
/*                                                                           */
/*  USTRING AStringToUString(ASTRING str)                                    */
/*                                                                           */
/*  Convert Ascii string to ISO 10646 string.                                */
/*                                                                           */
/*****************************************************************************/

USTRING AStringToUString(ASTRING str)
{
  int i;
  UStringBeginWithSize(strlen(str));
  for( i = 0;  str[i] != '\0';  i++ )
    UStringAdd((UCHAR) str[i]);
  return UStringEnd();
}


/*****************************************************************************/
/*                                                                           */
/*  int AStringHash(ASTRING str)                                             */
/*                                                                           */
/*  This hash function adds the characters together, but progressively       */
/*  shifts to the left one place.  If the result is negative it is negated.  */
/*                                                                           */
/*  We don't do % table_size here, that varies from table to table so is     */
/*  done in the symbol table code.                                           */
/*                                                                           */
/*****************************************************************************/

int AStringHash(ASTRING str)
{
  int i, res;
  res = 0;
  for( i = 0;  str[i] != 0;  i++ )
    res = (res << 1) + str[i];
  if( res < 0 )  res = - res;
  return res;
}
