/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         asymtab.c                                                  */
/*  DESCRIPTION:  Ascii symbol table (implementation)                        */
/*                                                                           */
/*****************************************************************************/
#include "asymtab.h"
#include <assert.h>
#define DEBUG1 0
#define DEBUG2 0


/*****************************************************************************/
/*                                                                           */
/*  void ASymInitKey(SYMTAB_VOIDP *s)                                        */
/*                                                                           */
/*  Set *s to a new, empty symbol table.                                     */
/*                                                                           */
/*  This code ensures that the condition "s->cursor != -1" is equivalent     */
/*  to the condition "a traversal is currently underway".                    */
/*                                                                           */
/*****************************************************************************/

void ASymInitKey(ASYMTAB_VOIDP *s)
{
  *s = (ASYMTAB_VOIDP) malloc(sizeof(**s));
  (*s)->empty_string = "";
  (*s)->count = 0;
  (*s)->cursor = -1;
  if( DEBUG1 )
    fprintf(stderr, "      ASymInitKey returning cursor 0\n");
  ArrayInit(&(*s)->keys);
  ArrayFill((*s)->keys, NULL);
  ArrayInit(&(*s)->values); /* no fill needed, values of NULL keys undef. */
  (*s)->enlarge_threshold = (ArraySize((*s)->keys) * 3) / 4;
  if( DEBUG2 )
  {
    fprintf(stderr, "ASymInit(%p):\n", (void *) *s);
    ASymDebugPrintKey(*s);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void ASymClearKey(ASYMTAB_VOIDP s)                                       */
/*                                                                           */
/*  Make s empty, without changing its current size.                         */
/*                                                                           */
/*****************************************************************************/

void ASymClearKey(ASYMTAB_VOIDP s)
{
  s->count = 0;
  s->cursor = -1;
  ArrayFill(s->keys, NULL);
  ArrayFill(s->values, NULL);
}


/*****************************************************************************/
/*                                                                           */
/*  void ASymFreeKey(ASYMTAB_VOIDP s)                                        */
/*                                                                           */
/*  Free the memory used by s.                                               */
/*                                                                           */
/*****************************************************************************/

void ASymFreeKey(ASYMTAB_VOIDP s)
{
  /* ***
  ArrayFree(s->keys);
  ArrayFree(s->values);
  free(s);
  *** */
}


/*****************************************************************************/
/*                                                                           */
/*  static void ASymRehash(ASYMTAB_VOIDP s)                                  */
/*                                                                           */
/*  Rehash the symbol table into a larger array.                             */
/*                                                                           */
/*****************************************************************************/

static void ASymRehash(ASYMTAB_VOIDP s)
{
  int i, j;
  ASTRING key;
  ARRAY_ASTRING new_keys;
  ARRAY_VOIDP   new_values;

  if( DEBUG2 )
  {
    fprintf(stderr, "[ ASymRehash(%p):\n", (void *) s);
    ASymDebugPrintKey(s);
  }

  /* recalculate s->count since not copying over empty strings */
  s->count = 0;

  /* make new_keys and new_values arrays, one rank larger than before */
  ArrayInitWithRank( (ARRAY_VOIDP *) &new_keys, s->keys->rank + 1,
    s->keys->item_size);
  ArrayInitWithRank( (ARRAY_VOIDP *) &new_values, s->values->rank + 1,
    s->values->item_size);

  /* fill new_keys with NULL; new_values can remain unfilled */
  ArrayFill(new_keys, NULL);

  /* insert every undeleted name in old array into new one */
  for( i = 0;  i < ArraySize(s->keys);  i++ )
  {
    /* printf("ASymRehash at index %d\n", i); */
    key = ArrayGet(s->keys, i);
    if( key != NULL && key != s->empty_string )
    {
      /* find index to insert this key into new_keys */
      j = AStringHash(key) % ArraySize(new_keys);
      while( ArrayGet(new_keys, j) != NULL )
	j = (j + 1) % ArraySize(new_keys);

      /* insert key into new_keys and value into new_values */
      ArrayPut(new_keys, j, key);
      ArrayPut(new_values, j, ArrayGet(s->values, i));
      s->count++;
    }
  }

  /* free old arrays and replace with new ones */
  ArrayFree( (ARRAY_VOIDP) s->keys);
  ArrayFree( (ARRAY_VOIDP) s->values);
  s->keys = new_keys;
  s->values = new_values;

  /* reset cursor and update threshold */
  s->cursor = -1;
  if( DEBUG1 )
    fprintf(stderr, "      ASymRehash returning cursor -1\n");
  s->enlarge_threshold = (ArraySize(s->keys) * 4) / 5;
  if( DEBUG2 )
  {
    fprintf(stderr, "] ASymRehash(%p):\n", (void *) s);
    ASymDebugPrintKey(s);
  }

}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ASymInsertKey(ASYMTAB_VOIDP s, ASTRING key)                      */
/*                                                                           */
/*  If no entry with this key is already present, insert this key into       */
/*  the table, set s->position to its index so that the wrapper macro can    */
/*  insert the corresponding value, and return TRUE.                         */
/*                                                                           */
/*  If an entry with this key is already present, set s->position to its     */
/*  index so that the wrapper macro can retrieve the associated value,       */
/*  and return FALSE.                                                        */
/*                                                                           */
/*  The insert point is the first entry whose key is equal to either NULL    */
/*  or empty_string.  However we have to search to the first NULL entry to   */
/*  confirm that the key is not already present.                             */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ASymInsertKey(ASYMTAB_VOIDP s, ASTRING key)
{
  int i, first_empty_str;

  /* error if traversal in progress */
  if( DEBUG2 )
  {
    fprintf(stderr, "ASymInsert(%p, %s):\n", (void *) s, key);
    ASymDebugPrintKey(s);
  }
  assert(s->cursor == -1);

  /* enlarge array if threshold reached */
  if( s->count >= s->enlarge_threshold )
    ASymRehash(s);

  /* check that key is absent, set first_empty_str to first empty str. */
  i = AStringHash(key) % ArraySize(s->keys);
  first_empty_str = -1;
  while( ArrayGet(s->keys, i) != NULL )
  {
    if( ArrayGet(s->keys, i) == s->empty_string )
    {
      if( first_empty_str < 0 )  first_empty_str = i;
    }
    else if( AStringEqual(key, ArrayGet(s->keys, i)) )
    {
      s->position = i;
      return FALSE;
    }
    i = (i + 1) % ArraySize(s->keys);
  }

  /* OK to insert */
  if( first_empty_str >= 0 )
  {
    /* insert position is first_empty_str, count does not change */
    s->position = first_empty_str;
  }
  else
  {
    /* using a fresh spot, so increase count too */
    s->count++;
    s->position = i;
  }
  ArrayPut(s->keys, s->position, key);
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ASymRetrieveKey(ASYMTAB_VOIDP s, ASTRING key)                    */
/*                                                                           */
/*  Search for this key in the symbol table; if present, set s->position to  */
/*  its index and return TRUE, otherwise return FALSE.                       */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ASymRetrieveKey(ASYMTAB_VOIDP s, ASTRING key)
{
  int i = AStringHash(key) % ArraySize(s->keys);
  if( DEBUG2 )
  {
    fprintf(stderr, "ASymRetrieve(%p, %s)\n", (void *) s, key);
    ASymDebugPrintKey(s);
  }
  while( ArrayGet(s->keys, i) != NULL )
  {
    if( AStringEqual(key, ArrayGet(s->keys, i)) )
    {
      s->position = i;
      if( DEBUG2 )
      {
	fprintf(stderr, "ASymRetrieve(%p, %s) returning TRUE:\n",(void *)s,key);
        ASymDebugPrintKey(s);
      }
      return TRUE;
    }
    i = (i + 1) % ArraySize(s->keys);
  }
  if( DEBUG2 )
    fprintf(stderr, "ASymRetrieve(%p, %s) returning FALSE\n", (void *) s, key);
  return FALSE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ASymChangeSymKey(ASYMTAB_VOIDP s, ASTRING oldkey, ASTRING newkey)*/
/*                                                                           */
/*  Change the key by which some element is accessed from oldkey to          */
/*  newkey.  If successfully accomplished, return TRUE; otherwise            */
/*  change nothing and return FALSE.                                         */
/*                                                                           */
/*  The two keys may be equal, producing a successful no-operation.          */
/*  Otherwise, the old key must be present initially and the new key         */
/*  must not be present initially.                                           */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ASymChangeSymKey(ASYMTAB_VOIDP s, ASTRING oldkey, ASTRING newkey)
{
  void *val, *val2;
  if( AStringEqual(oldkey, newkey) )
    return TRUE;
  else if( !ASymRetrieve(s, oldkey, &val) )
    return FALSE;
  else if( !ASymInsert(s, newkey, val, &val2) )
    return FALSE;
  else
    return ASymDelete(s, oldkey);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ASymDeleteKey(ASYMTAB_VOIDP s, ASTRING key)                      */
/*                                                                           */
/*  Search for this key in the symbol table; if present, delete it and       */
/*  return TRUE, otherwise return FALSE.                                     */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ASymDeleteKey(ASYMTAB_VOIDP s, ASTRING key)
{
  int i;

  /* error if traversal in progress */
  assert(s->cursor == -1);

  i = AStringHash(key) % ArraySize(s->keys);
  while( ArrayGet(s->keys, i) != NULL )
  {
    if( AStringEqual(key, ArrayGet(s->keys, i)) )
    {
      ArrayPut(s->keys, i, s->empty_string);
      return TRUE;
    }
    i = (i + 1) % ArraySize(s->keys);
  }
  return FALSE;
}


/*****************************************************************************/
/*                                                                           */
/*  void ASymFirstKey(ASYMTAB_VOIDP s)                                       */
/*                                                                           */
/*  Set internal cursor to first element of s (in arbitrary fixed order).    */
/*                                                                           */
/*****************************************************************************/

void ASymFirstKey(ASYMTAB_VOIDP s)
{ ASTRING key;
  for( s->cursor = 0;  s->cursor < ArraySize(s->keys);  s->cursor++ )
  {
    key = ArrayGet(s->keys, s->cursor);
    if( key != NULL && key != s->empty_string )
    {
      if( DEBUG1 )
	fprintf(stderr, "      SymFirstKey returning cursor %d\n", s->cursor);
      return;
    }
  }
  s->cursor = -1;
  if( DEBUG1 )
    fprintf(stderr, "      SymFirstKey returning cursor %d\n", s->cursor);
}


/*****************************************************************************/
/*                                                                           */
/*  void ASymNextKey(ASYMTAB_VOIDP s)                                        */
/*                                                                           */
/*  Set internal cursor to next element of s (in arbitrary fixed order).     */
/*                                                                           */
/*****************************************************************************/

void ASymNextKey(ASYMTAB_VOIDP s)
{
  ASTRING key;
  for( s->cursor++;  s->cursor < ArraySize(s->keys);  s->cursor++ )
  {
    key = ArrayGet(s->keys, s->cursor);
    if( key != NULL && key != s->empty_string )
    {
      if( DEBUG1 )
	fprintf(stderr, "      SymNextKey returning cursor %d\n", s->cursor);
      return;
    }
  }
  s->cursor = -1;
  if( DEBUG1 )
    fprintf(stderr, "      SymNextKey returning cursor %d\n", s->cursor);
}


/*****************************************************************************/
/*                                                                           */
/*  void ASymDebugPrintKey(ASYMTAB_VOIDP s)                                  */
/*                                                                           */
/*  Debug print of symbol table.                                             */
/*                                                                           */
/*****************************************************************************/

void ASymDebugPrintKey(ASYMTAB_VOIDP s)
{
  int i; ASTRING key;
  fprintf(stderr, "[ Symbol table (%d array size, %d elements):  %p\n",
    ArraySize(s->keys), s->count, (void *) s);
  for( i = 0;  i < ArraySize(s->keys);  i++ )
  {
    key = ArrayGet(s->keys, i);
    if( key != NULL && key != s->empty_string )
      fprintf(stderr, "%3d: %s  %p\n", i, key, (void *) s); 
  }
  fprintf(stderr, "]  %p\n", (void *) s);
}
