/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         asymtab.h                                                  */
/*  DESCRIPTION:  Ascii symbol table (linear probing hash table).            */
/*                                                                           */
/*  The keys of these symbol tables are always ASTRING, but the values can   */
/*  be any type.  To create a symbol table with values of type VALUE_TYPE:   */
/*                                                                           */
/*       typedef ARRAY(VALUE_TYPE)          ARRAY_VALUE_TYPE;                */
/*       typedef ASYMTAB(ARRAY_VALUE_TYPE)  ASYMTAB_VALUE_TYPE;              */
/*                                                                           */
/*  Use ASYMTAB_VALUE_TYPE wherever a symbol table of this type is required. */
/*  Notice that the type parameter of ASYMTAB is an array type; this is      */
/*  necessary owing to problems behind the scenes.                           */
/*                                                                           */
/*****************************************************************************/
#ifndef ASYMTAB_HEADER_FILE
#define ASYMTAB_HEADER_FILE

#include <stdio.h>
#include <stdlib.h>
#include "boolean.h"
#include "astring.h"
#include "array.h"

/*****************************************************************************/
/*                                                                           */
/*  ASYMTAB type                                                             */
/*                                                                           */
/*****************************************************************************/

#define ASYMTAB(ARRAY_TYPE)						\
  struct {								\
    ASTRING empty_string;	/* for marking deleted slots	 */	\
    int count;			/* number of entries in table	 */	\
    int position;		/* internal position marker      */	\
    int cursor;			/* internal cursor		 */	\
    int enlarge_threshold;	/* threshold for enlarging table */	\
    ARRAY_ASTRING  keys;	/* array of keys		 */	\
    ARRAY_TYPE    values;	/* parallel array of values	 */	\
  } *

typedef ASYMTAB(ARRAY_VOIDP) ASYMTAB_VOIDP;
typedef ASYMTAB(ARRAY_ASTRING) ASYMTAB_ASTRING;
typedef ASYMTAB(ARRAY_INT) ASYMTAB_INT;
typedef ASYMTAB(ARRAY_BOOLEAN) ASYMTAB_BOOLEAN;


/*****************************************************************************/
/*                                                                           */
/*  void ASymInit(SYMTAB(TYPE) *s)                                           */
/*                                                                           */
/*  Initialize *s to a new, empty symbol table.                              */
/*                                                                           */
/*****************************************************************************/

#define ASymInit(s)	ASymInitKey( (ASYMTAB_VOIDP *) (s))


/*****************************************************************************/
/*                                                                           */
/*  void ASymClear(SYMTAB(TYPE) s)                                           */
/*                                                                           */
/*  Make s empty, without changing its current size.                         */
/*                                                                           */
/*****************************************************************************/

#define ASymClear(s)	ASymClearKey( (ASYMTAB_VOIDP) (s))


/*****************************************************************************/
/*                                                                           */
/*  void ASymFree(ASYMTAB(TYPE) s)                                           */
/*                                                                           */
/*  Free memory used by s.                                                   */
/*                                                                           */
/*****************************************************************************/

#define ASymFree(s)	ASymFreeKey( (ASYMTAB_VOIDP) (s))


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ASymInsert(ASYMTAB(TYPE) s, ASTRING key, TYPE v, TYPE *cv)       */
/*                                                                           */
/*  If key is not already present in s, insert (key, v) and return TRUE.     */
/*  If key is already present in s, set *cv to the corresponding vaue, do    */
/*  no insertion, and return FALSE.                                          */
/*                                                                           */
/*****************************************************************************/

#define ASymInsert(s, key, v, cv)					\
(									\
  ASymInsertKey((ASYMTAB_VOIDP) (s), (key)) ?				\
  ( ArrayPut((s)->values, (s)->position, (v)), TRUE ) : 		\
  ( *(cv) = ArrayGet((s)->values, (s)->position), FALSE )		\
)


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ASymRetrieve(ASYMTAB(TYPE) s, ASTRING str, TYPE *v)              */
/*  BOOLEAN ASymRetrieveWithString(ASYMTAB(TYPE) s, ASTRING str, *vs,TYPE *v)*/
/*                                                                           */
/*  If key is present, set *v to its value and return TRUE.                  */
/*  If key is not present, set nothing and return FALSE.                     */
/*                                                                           */
/*  ASymRetrieveWithString also returns the search string, in *vs.  This is  */
/*  useful when str is in stack memory but the table's vs is in heap memory. */
/*                                                                           */
/*****************************************************************************/

#define ASymRetrieve(s, str, v)						\
(									\
  ASymRetrieveKey((ASYMTAB_VOIDP) (s), (str)) ?				\
  ( *(v) = ArrayGet((s)->values, (s)->position), TRUE) : FALSE		\
)

#define ASymRetrieveWithString(s, str, vs, v)				\
(									\
  ASymRetrieveKey((SYMTAB_VOIDP) (s), (str)) ?				\
  ( *(vs) = ArrayGet((s)->keys, (s)->position),				\
    *(v) = ArrayGet((s)->values, (s)->position), TRUE) : FALSE		\
)


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ASymDelete(ASYMTAB(TYPE) s, ASTRING str)                         */
/*                                                                           */
/*  If key is present, delete it and return TRUE; else return FALSE.         */
/*                                                                           */
/*****************************************************************************/

#define ASymDelete(s, str)	ASymDeleteKey((ASYMTAB_VOIDP) (s), (str))


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ASymChangeKey(ASYMTAB(TYPE) s, ASTRING oldkey, ASTRING newkey)   */
/*                                                                           */
/*  Update the name of an object in s from oldkey to newkey, returning       */
/*  TRUE if successful, else FALSE.                                          */
/*                                                                           */
/*  Having oldkey = newkey is allowed and is a no-op.                        */
/*                                                                           */
/*****************************************************************************/

#define ASymChangeKey(s, oldkey, newkey)				\
  ASymChangeSymKey((ASYMTAB_VOIDP) (s), oldkey, newkey)


/*****************************************************************************/
/*                                                                           */
/*  void ASymFirst(ASYMTAB(TYPE) s)                                          */
/*                                                                           */
/*  Set internal cursor to the first entry of s.                             */
/*                                                                           */
/*****************************************************************************/

#define ASymFirst(s)	ASymFirstKey( (ASYMTAB_VOIDP) (s) )


/*****************************************************************************/
/*                                                                           */
/*  void ASymNext(ASYMTAB(TYPE) s)                                           */
/*                                                                           */
/*  Set internal cursor to the next entry of s.                              */
/*                                                                           */
/*****************************************************************************/

#define ASymNext(s)	ASymNextKey( (ASYMTAB_VOIDP) (s) )


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ASymOffEnd(ASYMTAB(TYPE) s)                                      */
/*                                                                           */
/*****************************************************************************/

#define ASymOffEnd(s) ((s)->cursor == -1)


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ASymRetrieveCurrent(ASYMTAB(TYPE) s, ASTRING *key, TYPE *v)      */
/*                                                                           */
/*  Retrieve the current (key, value) pair of the traversal.                 */
/*                                                                           */
/*****************************************************************************/

#define ASymRetrieveCurrent(s, key, v)					\
(									\
  *(key) = ArrayGet((s)->keys, (s)->cursor),				\
  *(v) = ArrayGet((s)->values, (s)->cursor)				\
)

/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ASymNotOffEndAndCurrent(ASYMTAB(TYPE) s, ASTRING *key, TYPE *v)  */
/*                                                                           */
/*  If not off the end, return TRUE and set *key and *v to the current pair. */
/*  If off the end, return FALSE and set nothing.                            */
/*                                                                           */
/*****************************************************************************/

#define ASymNotOffEndAndCurrent(s, key, v)				\
(ASymOffEnd(s) ? FALSE : (ASymRetrieveCurrent((s), (key), (v)), TRUE))


/*****************************************************************************/
/*                                                                           */
/*  ASymForEach(ASYMTAB(TYPE) s, ASTRING *key, TYPE *v)                      */
/*                                                                           */
/*  Equivalent to "for( each (*key, *v) in s )"                              */
/*                                                                           */
/*****************************************************************************/

#define ASymForEach(s, key, v)						\
for( ASymFirst(s); ASymNotOffEndAndCurrent((s), (key), (v)); ASymNext(s) )


/*****************************************************************************/
/*                                                                           */
/*  void ASymDebugPrint(ASYMTAB(TYPE) s)                                     */
/*                                                                           */
/*  Debug print of symbol table s.                                           */
/*                                                                           */
/*****************************************************************************/

#define ASymDebugPrint(s)  ASymDebugPrintKey((ASYMTAB_VOIDP) (s))


/*****************************************************************************/
/*                                                                           */
/*  External functions - not intended to be called directly.                 */
/*                                                                           */
/*****************************************************************************/

extern void ASymInitKey(ASYMTAB_VOIDP *s);
extern void ASymClearKey(ASYMTAB_VOIDP s);
extern void ASymFreeKey(ASYMTAB_VOIDP s);
extern BOOLEAN ASymInsertKey(ASYMTAB_VOIDP s, ASTRING str);
extern BOOLEAN ASymRetrieveKey(ASYMTAB_VOIDP s, ASTRING str);
extern BOOLEAN ASymDeleteKey(ASYMTAB_VOIDP s, ASTRING str);
extern BOOLEAN ASymChangeSymKey(ASYMTAB_VOIDP s,ASTRING oldkey,ASTRING newkey);
extern void ASymFirstKey(ASYMTAB_VOIDP s);
extern void ASymNextKey(ASYMTAB_VOIDP s);
extern void ASymDebugPrintKey(ASYMTAB_VOIDP s);

#endif
