/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         auto.c                                                     */
/*  DESCRIPTION:  Automatic generation of certain Nonpareil classes.         */
/*                                                                           */
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "externs.h"


/*****************************************************************************/
/*                                                                           */
/*  Submodule "helper functions".                                            */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void PrintHeaderComment(FILE * fp, ASTRING str1, ASTRING str2,           */
/*    ASTRING str3)                                                          */
/*                                                                           */
/*  Make a nice header, enclosed in hashes, out of str1, str2, str3.         */
/*                                                                           */
/*****************************************************************************/

static void PrintHeaderComment(FILE * fp, ASTRING str1, ASTRING str2,
  ASTRING str3)
{
  fprintf(fp, "\n");
  fprintf(fp, "###################################################\n");
  fprintf(fp, "#                                                 #\n");
  fprintf(fp, "#  %-45s  #\n", str1);
  fprintf(fp, "#  %-45s  #\n", str2);
  fprintf(fp, "#  %-45s  #\n", str3);
  fprintf(fp, "#                                                 #\n");
  fprintf(fp, "#  This file is generated automatically, and it   #\n");
  fprintf(fp, "#  is always a bad idea to hand-edit it.  For     #\n");
  fprintf(fp, "#  more info, see file externs.h, near the top    #\n");
  fprintf(fp, "#                                                 #\n");
  fprintf(fp, "###################################################\n");
}


/*****************************************************************************/
/*                                                                           */
/*  ASTRING Vars(ASTRING str, int from, int to)                              */
/*                                                                           */
/*  Return a comma-separated list of variables.  For example,                */
/*                                                                           */
/*      X3, X4, X5, X6                                                       */
/*                                                                           */
/*  would be produced by PrintVars("X", 3, 6, fp).                           */
/*                                                                           */
/*****************************************************************************/
#define end_buff &buff[strlen(buff)]

static ASTRING Vars(ASTRING str, int from, int to)
{
  int i;  static char buff[200];
  strcpy(buff, "");
  for( i = from;  i <= to;  i++ )
    sprintf(end_buff, "%s%s%d", i > from ? ", " : "", str, i);
  return buff;
}


/*****************************************************************************/
/*                                                                           */
/*  ASTRING TypedVars(ASTRING str1, ASTRING str2, int from, int to)          */
/*                                                                           */
/*  Similar to Vars, only now the variables have types.  For example,        */
/*                                                                           */
/*     x3: X3, x4: X4, x5: X5, x6: X6                                        */
/*                                                                           */
/*  would be produced by PrintTypedVars("x", "X", 3, 6).                     */
/*                                                                           */
/*****************************************************************************/

static ASTRING TypedVars(ASTRING str1, ASTRING str2, int from, int to)
{
  int i;  static char buff[200];
  strcpy(buff, "");
  for( i = from;  i <= to;  i++ )
    sprintf(end_buff, "%s%s%d: %s%d", i > from ? ", " : "", str1, i, str2, i);
  return buff;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "tuple classes".                                               */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void PrintTupleClass(FILE *fp, int n)                                    */
/*                                                                           */
/*  Print the tuplen class.  For n = 2 the name is "pair", and for n = 3     */
/*  the name is "triple".  This makes the first three tuple classes special. */
/*                                                                           */
/*****************************************************************************/

static void PrintTupleClass(FILE *fp, int n)
{
  if( n == 2 )
  {
    fprintf(fp, "\n");
    fprintf(fp, "norename class pair{X1, X2}\n");
    fprintf(fp, "\n");
    fprintf(fp, "  item1: X1\n");
    fprintf(fp, "\n");
    fprintf(fp, "  item2: X2\n");
    fprintf(fp, "\n");
    fprintf(fp, "end\n");
  }
  else if( n == 3 )
  {
    fprintf(fp, "\n");
    fprintf(fp, "norename class triple{X1, X2, X3}\n");
    fprintf(fp, "inherit pair{X1, X2}\n");
    fprintf(fp, "\n");
    fprintf(fp, "  item3: X3\n");
    fprintf(fp, "\n");
    fprintf(fp, "end\n");
  }
  else if( n == 4 )
  {
    fprintf(fp, "\n");
    fprintf(fp, "norename class tuple4{X1, X2, X3, X4}\n");
    fprintf(fp, "inherit triple{X1, X2, X3}\n");
    fprintf(fp, "\n");
    fprintf(fp, "  item4: X4\n");
    fprintf(fp, "\n");
    fprintf(fp, "end\n");
  }
  else
  {
    assert(n > 4);
    fprintf(fp, "\n");
    fprintf(fp, "norename class tuple%d{%s}\n", n, Vars("X", 1, n));
    fprintf(fp, "inherit tuple%d{%s}\n", n - 1, Vars("X", 1, n - 1));
    fprintf(fp, "\n");
    fprintf(fp, "  item%d: X%d\n", n, n);
    fprintf(fp, "\n");
    fprintf(fp, "end\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN AutoTupleClasses(USTRING module_dir_name)                        */
/*                                                                           */
/*  Check whether file "tuples" exists in directory module_dir_name, and     */
/*  if not, make such a file containing definitions of the tuple classes.    */
/*                                                                           */
/*****************************************************************************/

BOOLEAN AutoTupleClasses(USTRING module_dir_name)
{
  FILE *fp;  int n;
  USTRING full_name;  UTF8 utf8_full_name;

  /* if file already exists, don't disturb it */
  full_name = UStringCat(module_dir_name,
    AStringToUString(NPC_DIR_SEP "tuples"));
  utf8_full_name = UStringToUTF8(full_name);
  fp = fopen((ASTRING) utf8_full_name, "r");
  if( fp != NULL )
  {
    fclose(fp);
    return TRUE;
  }

  /* file does not exist; open for writing */
  fprintf(stderr, "creating Nonpareil source file \"%s\"\n",
    (ASTRING) utf8_full_name);
  fp = fopen((ASTRING) utf8_full_name, "w");
  if( fp == NULL )
  {
    fprintf(stderr, "cannot write to file \"%s\"\n", (ASTRING) utf8_full_name);
    return FALSE;
  }

  /* write classes to file */
  PrintHeaderComment(fp, "Tuple classes: pair, triple, tuple4, etc.",
    NPC_VERSION, TimingTimeStamp());
  for( n = 2;  n <= MAX_TUPLE;  n++ )
    PrintTupleClass(fp, n);
  fclose(fp);
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "function classes".                                            */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void PrintFunNRepClass(FILE *fp, int n)                                  */
/*                                                                           */
/*  Print a funn_rep class onto fp.                                          */
/*                                                                           */
/*****************************************************************************/

static void PrintFunNRepClass(FILE *fp, int n)
{
  fprintf(fp, "\n");
  fprintf(fp, "norename class fun%d_rep{%s, T}\n", n, Vars("X", 1, n));
  fprintf(fp, "\n");
  fprintf(fp, "  builtin\n");
  fprintf(fp, "\n");
  fprintf(fp, "noncreation\n");
  fprintf(fp, "\n");
  fprintf(fp, "  call(%s): T :=\n", TypedVars("x", "X", 1, n));
  fprintf(fp, "    builtin \"fun%d_rep_call\"(self, %s)\n", n, Vars("x", 1, n));
  fprintf(fp, "\n");
  fprintf(fp, "end\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void PrintFunNClass(FILE *fp, int n)                                     */
/*                                                                           */
/*  Print a funn class onto fp.                                              */
/*                                                                           */
/*****************************************************************************/

static void PrintFunNClass(FILE *fp, int n)
{
  int i;
  fprintf(fp, "\n");
  fprintf(fp, "norename class fun%d{%s, T}\n", n, Vars("X", 1, n));
  fprintf(fp, "\n");
  fprintf(fp, "noncreation\n");
  fprintf(fp, "\n");
  for( i = 1;  i <= n;  i++ )
  {
    if( i < n )
    {
      /* print ordinary apply feature */
      fprintf(fp, "  apply%d(%s): fun%d{%s, T}\n\n", i,
	TypedVars("x", "X", 1, i), n - i, Vars("X", i + 1, n));
    }
    else
    {
      /* print last apply feature, which is a bit different */
      fprintf(fp, "  apply%d(%s): T\n\n", i, TypedVars("x", "X", 1, i));
    }
  }
  fprintf(fp, "end\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void PrintFunN_MClass(FILE *fp, int n, int m)                            */
/*                                                                           */
/*  Print a funn_m class onto fp.                                            */
/*                                                                           */
/*****************************************************************************/

static void PrintFunN_MClass(FILE *fp, int n, int m)
{
  BOOLEAN noncreation_printed;  int i, first_param, last_param, param_count;
  char buff[200];

  /* header and rep variable */
  fprintf(fp, "\n");
  fprintf(fp, "norename class fun%d_%d{%s, T}\n", n, m, Vars("X", 1, n));
  fprintf(fp, "inherit fun%d{%s, T}\n", m, Vars("X", n - m + 1, n));
  fprintf(fp, "\n");
  fprintf(fp, "  rep: fun%d_rep{%s, T}", n, Vars("X", 1, n));
  fprintf(fp, "\n");

  /* creation and noncreation features */
  noncreation_printed = FALSE;
  for( i = 1;  i <= n;  i++ )
  {
    if( i <= n - m )
    {
      /* print a creation variable */
      fprintf(fp, "  x%d: X%d\n", i, i);
    }
    else
    {
      /* print noncreation keyword and initialize if not done yet */
      if( !noncreation_printed )
      {
	/* switch to printing apply features */
	fprintf(fp, "\nnoncreation\n\n");
	first_param = i;
	last_param = first_param;
	param_count = 1;
	noncreation_printed = TRUE;
      }

      if( i < n )
      {
	/* print ordinary apply feature */
	sprintf(buff, "fun%d_%d{%s, T}", n, n - i, Vars("X", 1, n));
	fprintf(fp,  "  apply%d(%s): %s :=\n", param_count,
	  TypedVars("x", "X", first_param, last_param), buff);
	fprintf(fp, "    %s(rep, %s)\n\n", buff, Vars("x", 1, last_param));
	last_param++;
	param_count++;
      }
      else
      {
	/* print last apply feature, which is a bit different */
	fprintf(fp, "  apply%d(%s): T :=\n", param_count,
	  TypedVars("x", "X", first_param, last_param));
	fprintf(fp, "    rep.call(%s)\n\n", Vars("x", 1, n));
      }
    }
  }

  /* class footer */
  fprintf(fp, "end\n");
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN AutoFunctionClasses(USTRING module_dir_name)                     */
/*                                                                           */
/*  Check whether file "functs" exists in directory module_dir_name, and     */
/*  if not, make such a file containing definitions of the function classes. */
/*                                                                           */
/*****************************************************************************/

BOOLEAN AutoFunctionClasses(USTRING module_dir_name)
{
  FILE *fp;  int n, m;  USTRING full_name;  UTF8 utf8_full_name;

  /* if file already exists, don't disturb it */
  full_name = UStringCat(module_dir_name,
    AStringToUString(NPC_DIR_SEP "functs"));
  utf8_full_name = UStringToUTF8(full_name);
  fp = fopen((ASTRING) utf8_full_name, "r");
  if( fp != NULL )
  {
    fclose(fp);
    return TRUE;
  }

  /* file does not exist; open for writing */
  fp = fopen((ASTRING) utf8_full_name, "w");
  fprintf(stderr, "creating Nonpareil source file \"%s\"\n",
    (ASTRING) utf8_full_name);
  if( fp == NULL )
  {
    fprintf(stderr, "cannot write to file \"%s\"\n", (ASTRING) utf8_full_name);
    return FALSE;
  }

  /* write funrep classes to fp */
  PrintHeaderComment(fp, "Function classes: funrep_n",
    NPC_VERSION, TimingTimeStamp());
  for( n = 1;  n <= MAX_FUN_PARAMS;  n++ )
    PrintFunNRepClass(fp, n);

  /* write funn classes to fp */
  fprintf(fp, "\n");
  PrintHeaderComment(fp, "Function classes: fun_n",
    NPC_VERSION, TimingTimeStamp());
  for( n = 1;  n <= MAX_FUN_PARAMS;  n++ )
    PrintFunNClass(fp, n);

  /* write funn_m classes to fp */
  fprintf(fp, "\n");
  PrintHeaderComment(fp, "Function classes: fun_n_m",
    NPC_VERSION, TimingTimeStamp());
  for( n = 1;  n <= MAX_FUN_PARAMS;  n++ )
    for( m = 1;  m <= n;  m++ )
      PrintFunN_MClass(fp, n, m);

  /* close file and quit */
  fclose(fp);
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  void UStringToNonpareilSource(USTRING str, FILE *fp)                     */
/*                                                                           */
/*  Print str onto fp in Nonpareil source code format, including the         */
/*  enclosing quotes.  Printable ASCII characters are printed as             */
/*  themselves; other characters are printed in hexadecimal, between         */
/*  \[ and ] as Nonpareil source code permits.                               */
/*                                                                           */
/*****************************************************************************/

void UStringToNonpareilSource(USTRING str, FILE *fp)
{
  BOOLEAN in_hex;  int i, len;  UCHAR ch;
  fprintf(fp, "\"");
  len = UStringLength(str);
  in_hex = FALSE;
  for( i = 0;  i < len;  i++ )
  {
    ch = UStringGet(str, i);
    if( in_hex )
    {
      if( UCharIsPrintableAscii(ch) )
      {
	if( ch == '\\' || ch == '"' )
	  fprintf(fp, "]\\%c", (char) ch);
	else
	  fprintf(fp, "]%c", (char) ch);
	in_hex = FALSE;
      }
      else
	fprintf(fp, " %04X", ch);
    }
    else
    {
      if( UCharIsPrintableAscii(ch) )
      {
	if( ch == '\\' || ch == '"' )
	  fprintf(fp, "\\%c", (char) ch);
	else
	  fprintf(fp, "%c", (char) ch);
      }
      else
      {
	fprintf(fp, "\\[%04X", ch);
	in_hex = TRUE;
      }
    }
  }
  if( in_hex )
    fprintf(fp, "]");
  fprintf(fp, "\"");
}


/*****************************************************************************/
/*                                                                           */
/*  void PrintCharEntry(UCHAR first_char, UCHAR last_char, FILE *fp)         */
/*                                                                           */
/*  Print an entry in the char file for first_char .. last_char, which are   */
/*  known to share properties.                                               */
/*                                                                           */
/*****************************************************************************/

void PrintCharEntry(UCHAR first_char, UCHAR last_char, FILE *fp)
{
  int v, numerator, denominator;

  /* character code or range */
  assert(first_char <= last_char);
  if( first_char < last_char )
    fprintf(fp, "  0x%04X .. 0x%04X", first_char, last_char);
  else
    fprintf(fp, "  0x%04X", first_char);

  /* general category */
  fprintf(fp, "(char_cat!%s",
    UCharGeneralCategoryShow(UCharGeneralCategory(first_char)));

  /* combining class */
  fprintf(fp, ", %u", UCharCanonicalCombiningClass(first_char));

  /* bidi class */
  fprintf(fp, ", char_bidi!%s",
    UCharBidiClassShow(UCharBidiClass(first_char)));

  /* decimal */
  fprintf(fp, ", %s", UCharIsDecimalDigit(first_char, &v) ? "true" : "false");

  /* numerator and denominator (0 for both if not numeric) */
  if( !UCharIsNumeric(first_char, &numerator, &denominator) )
    numerator = denominator = 0;
  fprintf(fp, ", %d, %d", numerator, denominator);

  /* mirrored */
  fprintf(fp, ", %s", UCharIsBidiMirrored(first_char) ? "true" : "false");

  /* cmap type */
  fprintf(fp, ", char_map!%s", UCharCMapClassShow(UCharCMapClass(first_char)));

  /* cmap */
  fprintf(fp, ", ");
  if( UCharCMapClass(first_char) != UCHAR_CMAP_NONE )
    UStringToNonpareilSource(UCharCMap(first_char), fp);
  else
    fprintf(fp, "\"\"");

  /* upper case */
  fprintf(fp, ", ");
  UStringToNonpareilSource(UCharUpperCase(first_char), fp);

  /* lower case */
  fprintf(fp, ", ");
  UStringToNonpareilSource(UCharLowerCase(first_char), fp);

  /* title case */
  fprintf(fp, ", ");
  UStringToNonpareilSource(UCharTitleCase(first_char), fp);

  /* finish off */
  fprintf(fp, ")\n");
  fflush(fp);
}


/*****************************************************************************/
/*                                                                           */
/*  void PrintCharList(FILE *fp)                                             */
/*                                                                           */
/*  Print the long list of all predefined characters onto fp.                */
/*                                                                           */
/*****************************************************************************/

static void PrintCharList(FILE *fp)
{
  UCHAR ch, base_char, max_char;
  fprintf(fp, "  # char list (auto-generated from Unicode web site files)\n");
  max_char = UCharMax();
  base_char = 1;
  for( ch = 2;  ch <= max_char;  ch++ )
  {
    if( !UCharEqualProperties(base_char, ch) )
    {
      PrintCharEntry(base_char, ch - 1, fp);
      base_char = ch;
    }
  }
  if( base_char <= max_char )
    PrintCharEntry(base_char, max_char, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN AutoCharClass(USTRING module_dir_name)                           */
/*                                                                           */
/*  Check whether file "char" exists in directory module_dir_name, and       */
/*  if not, make such a file containing the definition of the char class     */
/*  and related classes.                                                     */
/*                                                                           */
/*****************************************************************************/
#define CHAR_SKEL NPC_LIB_DIR NPC_DIR_SEP "char_skeleton"

BOOLEAN AutoCharClass(USTRING module_dir_name)
{
  FILE *in_fp, *out_fp;  int ch;  USTRING full_name;  UTF8 utf8_full_name;
  char buff[200];

  /* if file already exists, don't disturb it */
  full_name = UStringCat(module_dir_name, AStringToUString(NPC_DIR_SEP "char"));
  utf8_full_name = UStringToUTF8(full_name);
  out_fp = fopen((ASTRING) utf8_full_name, "r");
  if( out_fp != NULL )
  {
    fclose(out_fp);
    return TRUE;
  }

  /* open source file */
  fprintf(stderr, "creating Nonpareil source file \"%s\"\n",
    (ASTRING) utf8_full_name);
  in_fp = fopen(CHAR_SKEL, "r");
  if( in_fp == NULL )
  {
    fprintf(stderr, "cannot open file \"%s\" needed for building char class\n",
      CHAR_SKEL);
    return FALSE;
  }

  /* open target file and print a header comment */
  out_fp = fopen((ASTRING) utf8_full_name, "w");
  if( out_fp == NULL )
  {
    fprintf(stderr, "cannot write to file \"%s\" (char class file)\n",
      (ASTRING) utf8_full_name);
    return FALSE;
  }
  PrintHeaderComment(out_fp, "The char class and its relatives",
    NPC_VERSION, TimingTimeStamp());

  /* copy source to target, interpolating chars at $INSERT_CHAR_LIST_HERE$ */
  while( (ch = getc(in_fp)) != EOF )
  {
    if( ch == '$' )
    {
      fscanf(in_fp, "%s", buff);
      if( strcmp(buff, "INSERT_CHAR_LIST_HERE$") != 0 )
      {
	fprintf(stderr, "format error in \"%s\"\n", CHAR_SKEL);
	exit(1);
      }
      PrintCharList(out_fp);
    }
    else
      putc(ch, out_fp);
  }
  fclose(in_fp);
  fclose(out_fp);
  return TRUE;
}
