/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         befn.c                                                     */
/*  DESCRIPTION:  Objects representing back-end functions                    */
/*                                                                           */
/*****************************************************************************/
#include <string.h>
#include "externs.h"
#define DEBUG1 0
#define DEBUG2 0
#define DEBUG3 0
#define DEBUG4 0
#define DEBUG5 0
#define DEBUG6 0
#define DEBUG7 0


/*****************************************************************************/
/*                                                                           */
/*  Back-end functions                                                       */
/*  ------------------                                                       */
/*                                                                           */
/*  As explained at more length in the implementation notes, a back-end      */
/*  function is something that can be called from within the generated       */
/*  code.  There are several kinds of back-end functions, all inheriting     */
/*  from BEFN:                                                               */
/*                                                                           */
/*      BEFN_CREATION      A creation function                               */
/*      BEFN_CREDFT        A creation feature default value                  */
/*      BEFN_FEATURE       A feature                                         */
/*      BEFN_PRECOND       A precondition                                    */
/*      BEFN_BUILTIN       A builtin function                                */
/*      BEFN_LETDEF        A letdef with params                              */
/*      BEFN_DOWNDEF       A downcast variable                               */
/*      BEFN_PARAM         A back-end parameter                              */
/*      BEFN_INVARIANT     A class invariant                                 */
/*      BEFN_CLASS_INIT    A class initialization function                   */
/*      BEFN_ENUM_INIT     An all_enumerated initialization function         */
/*      BEFN_SYSTEM_INIT   A system initialization function                  */
/*                                                                           */
/*  The be_type and be_obj fields remain NULL until code generation.         */
/*                                                                           */
/*****************************************************************************/

struct befn_rec {
  KIND_TAG		kind_tag;	/* kind of entity                    */
  CODEGEN_TYPE		be_type;	/* result type                       */
  CODEGEN_OBJ		be_obj;		/* corresponding backend function    */
  ARRAY_BEFN		inner_functs;	/* helpers; generate just before me  */
  ARRAY_BEFN_PARAM	parameters;	/* parameters (NULL if none)         */
  CODEGEN_OBJ		inline_be_obj;	/* non-NULL if to be inlined         */
  BOOLEAN		utilized;	/* TRUE when function has been called*/
  BOOLEAN		cached;		/* TRUE when calls are to be cached  */
};


/*****************************************************************************/
/*                                                                           */
/*  KIND_TAG BEFnKindTag(BEFN fun)                                           */
/*                                                                           */
/*  Return the kind tag of fun.                                              */
/*                                                                           */
/*****************************************************************************/

KIND_TAG BEFnKindTag(BEFN fun)
{
  return fun->kind_tag;
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnAddParameter(BEFN fun, BEFN_PARAM param)                        */
/*                                                                           */
/*  Add a parameter to fun.  It will be inserted into its proper place       */
/*  in the parameter list according to its kind.                             */
/*                                                                           */
/*****************************************************************************/

void BEFnAddParameter(BEFN fun, BEFN_PARAM param)
{
  BEFnParamListInsert(&fun->parameters, param);
}


/*****************************************************************************/
/*                                                                           */
/*  int BEFnParamsCount(BEFN fun)                                            */
/*                                                                           */
/*  Return the number of parameters of fun.                                  */
/*                                                                           */
/*****************************************************************************/

int BEFnParamsCount(BEFN fun)
{
  return fun->parameters == NULL ? 0 : ArraySize(fun->parameters);
}


/*****************************************************************************/
/*                                                                           */
/*  ARRAY_BEFN_PARAM BEFnParameters(BEFN fun)                                */
/*                                                                           */
/*  Return the parameters of fun.                                            */
/*                                                                           */
/*****************************************************************************/

ARRAY_BEFN_PARAM BEFnParameters(BEFN fun)
{
  return fun->parameters;
}


/*****************************************************************************/
/*                                                                           */
/*  BEFN_PARAM BEFnFindParamFromCreationFeature(BEFN befn, BEFN_FEATURE f)   */
/*                                                                           */
/*  Search the parameters of befn for one that was derived from creation     */
/*  feature f, and return that parameter.  It must be present.               */
/*                                                                           */
/*****************************************************************************/

BEFN_PARAM BEFnFindParamFromCreationFeature(BEFN befn, BEFN_FEATURE f)
{
  BEFN_PARAM param;
  assert(befn->parameters != NULL);
  ArrayForEach(befn->parameters, param)
    if( BEFnParamCreationFeature(param) == f )
      return param;
  assert(FALSE);
  return NULL;
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnAddInnerBEFn(BEFN fun, BEFN inner_fun)                          */
/*                                                                           */
/*  Add an inner (helper) funct to fun.                                      */
/*                                                                           */
/*****************************************************************************/

void BEFnAddInnerBEFn(BEFN fun, BEFN inner_fun)
{
  if( fun->inner_functs == NULL )
    ArrayInit(&fun->inner_functs);
  ArrayAddLast(fun->inner_functs, inner_fun);
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnSetUtilized(BEFN fun)                                           */
/*                                                                           */
/*  Record the fact that fun has been utilized (i.e. called).                */
/*                                                                           */
/*****************************************************************************/

void BEFnSetUtilized(BEFN fun)
{
  fun->utilized = TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN BEFnUtilized(BEFN fun)                                           */
/*                                                                           */
/*  Return TRUE if fun has been utilized (i.e. called).                      */
/*                                                                           */
/*****************************************************************************/

BOOLEAN BEFnUtilized(BEFN fun)
{
  return fun->utilized;
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnFinalize(BEFN fun, CODEGEN be)                                  */
/*                                                                           */
/*  Finalize fun for be.  This must be done once for each C function after   */
/*  its construction is complete but before any code generation (of the      */
/*  function itself or a call on the function) is attempted.                 */
/*                                                                           */
/*  Finalizing assigns a backend type and function to fun, sets the types    */
/*  of its parameters (but not their names), and recursively finalizes       */
/*  any helper functions.                                                    */
/*                                                                           */
/*****************************************************************************/

void BEFnFinalize(BEFN fun, CODEGEN be)
{
  BEFN befn;
  if( DEBUG3 )
    fprintf(stderr, "[ BEFnFinalize()\n");

  /* finalize the be_type and be_obj fields */
  switch( fun->kind_tag )
  {
    case KIND_BEFN_CREATION:

      BEFnCreationFinalize((BEFN_CREATION) fun, be);
      break;


    case KIND_BEFN_CREDFT:

      BEFnCreDftFinalize((BEFN_CREDFT) fun, be);
      break;


    case KIND_BEFN_FEATURE:

      BEFnFeatureFinalize((BEFN_FEATURE) fun, be);
      break;


    case KIND_BEFN_PRECOND:

      BEFnPrecondFinalize((BEFN_PRECOND) fun, be);
      break;


    case KIND_BEFN_BUILTIN:

      BEFnBuiltinFinalize((BEFN_BUILTIN) fun, be);
      break;


    case KIND_BEFN_LETDEF:

      BEFnLetDefFinalize((BEFN_LETDEF) fun, be);
      break;


    case KIND_BEFN_DOWNDEF:

      BEFnDownDefFinalize((BEFN_DOWNDEF) fun, be);
      break;


    case KIND_BEFN_PARAM:

      /* parameters should never be finalized */
      assert(FALSE);
      break;


    case KIND_BEFN_INVARIANT:

      BEFnInvtFinalize((BEFN_INVARIANT) fun, be);
      break;


    case KIND_BEFN_CLASS_INIT:

      BEFnClassInitFinalize((BEFN_CLASS_INIT) fun, be);
      break;


    case KIND_BEFN_ENUM_INIT:

      BEFnEnumInitFinalize((BEFN_ENUM_INIT) fun, be);
      break;


    case KIND_BEFN_SYSTEM_INIT:

      BEFnSystemInitFinalize((BEFN_SYSTEM_INIT) fun, be);
      break;


    default:

      assert(FALSE);
      break;

  }

  /* set parameter types */
  BEFnParamListSetBETypes(fun->parameters, be);

  /* finalize inner functions */
  if( fun->inner_functs != NULL )
    ArrayForEach(fun->inner_functs, befn)
      BEFnFinalize(befn, be);
  if( DEBUG3 )
    fprintf(stderr, "] BEFnFinalize returning %s\n", be->VarShow(fun->be_obj));
}


/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_OBJ BEFnBEObj(BEFN fun)                                          */
/*                                                                           */
/*  Return the C name of fun.  It may be NULL in predefined object features. */
/*                                                                           */
/*****************************************************************************/

CODEGEN_OBJ BEFnBEObj(BEFN fun)
{
  return fun->be_obj;
}


/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_TYPE BEFnBEType(BEFN fun)                                        */
/*                                                                           */
/*  Return the backend type of fun.                                          */
/*                                                                           */
/*****************************************************************************/

CODEGEN_TYPE BEFnBEType(BEFN fun)
{
  assert(fun->be_type != NULL);
  return fun->be_type;
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnCodeGenPrototype(BEFN fun, CODEGEN be)                          */
/*                                                                           */
/*  Code gen a prototype for fun, in which the parameters just have          */
/*  types, not names.                                                        */
/*                                                                           */
/*****************************************************************************/

static void BEFnCodeGenPrototype(BEFN fun, CODEGEN be)
{
  /* ARRAY_BEFN_PARAM hidden_params; */
  be->PrototypeBegin(fun->be_obj, fun->be_type);
  if( fun->parameters != NULL )
    BEFnParamListPrototypeDeclare(fun->parameters, fun->be_obj, be);
  be->PrototypeEnd(fun->be_obj);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN BEFnInitOrder(BEFN befn, int visit_num, BOOLEAN *report,         */
/*    BEFN_SYSTEM_INIT fun)                                                  */
/*                                                                           */
/*  Carry out the part of the algorithm for finding a safe initialization    */
/*  order for predefined object features that applies to befn.               */
/*                                                                           */
/*****************************************************************************/

BOOLEAN BEFnInitOrder(BEFN befn, int visit_num, BOOLEAN *report,
  BEFN_SYSTEM_INIT fun)
{
  switch( befn->kind_tag )
  {
    case KIND_BEFN_CREATION:

      return BEFnCreationInitOrder((BEFN_CREATION) befn, visit_num,report,fun);


    case KIND_BEFN_CREDFT:

      return BEFnCreDftInitOrder((BEFN_CREDFT) befn, visit_num, report, fun);


    case KIND_BEFN_FEATURE:

      return BEFnFeatureInitOrder((BEFN_FEATURE) befn, visit_num, report, fun);


    case KIND_BEFN_PRECOND:

      return BEFnPrecondInitOrder((BEFN_PRECOND) befn, visit_num, report, fun);


    case KIND_BEFN_BUILTIN:

      /* nothing to do here */
      return TRUE;


    case KIND_BEFN_LETDEF:

      return BEFnLetDefInitOrder((BEFN_LETDEF) befn, visit_num, report, fun);


    case KIND_BEFN_DOWNDEF:
    case KIND_BEFN_PARAM:

      /* nothing to do here */
      return TRUE;


    case KIND_BEFN_INVARIANT:

      return BEFnInvtInitOrder((BEFN_INVARIANT) befn, visit_num, report, fun);


    case KIND_BEFN_ENUM_INIT:

      /* nothing to do here */
      return TRUE;


    case KIND_BEFN_CLASS_INIT:
    case KIND_BEFN_SYSTEM_INIT:
    default:

      /* should never be called; not available to user code */
      assert(FALSE);
      return FALSE; /* keep compiler happy */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnCodeGenParamHash(BEFN fun, CODEGEN_OBJ be_var_sig, CODEGEN be)  */
/*                                                                           */
/*  Generate code to hash the parameters of fun, accumulating the            */
/*  resulting signature in be_var_sig.                                       */
/*                                                                           */
/*****************************************************************************/

void BEFnCodeGenParamHash(BEFN fun, CODEGEN_OBJ be_var_sig, CODEGEN be)
{
  int i;  CODEGEN_TYPE be_type;  CODEGEN_OBJ be_var;  BEFN_PARAM param;
  for( i = 0;  i < ArraySize(fun->parameters);  i++ )
  {
    param = ArrayGet(fun->parameters, i);
    be_var = BEFnParamBEVar(param);
    be_type = BEFnParamBEType(param);
    be->VarOpAsstBegin(be_var_sig, be->add);
    if( i > 0 )
      Call2(be->lshift, Cast(be->int_type, be_type, Var(be_var)), Int(i));
    else
      Cast(be->int_type, be_type, Var(be_var));
    be->AsstEnd();
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void EntryMatches(BEFN fun, CODEGEN_OBJ sig_be_var,                      */
/*    CODEGEN_OBJ entry_be_var, CODEGEN be)                                  */
/*                                                                           */
/*  Generate code to check whether entry_be_var matches fun, including       */
/*  testing the signatures for equality.                                     */
/*                                                                           */
/*****************************************************************************/

static void EntryMatches(BEFN fun, CODEGEN_OBJ sig_be_var,
  CODEGEN_OBJ entry_be_var, CODEGEN be)
{
  int i;  BEFN_PARAM param;
  be->CallBegin(be->logical_and_seq);
  Equal(Call1(NPBack_Signature, Var(entry_be_var)), Var(sig_be_var)),
  be->CallContinue(be->logical_and_seq, 1);
  Equal(Call2(NPBack_Array_Get, Var(entry_be_var), Int(0)),
    Cast(be->int_type, NULL, Call1(be->address_of, Var(fun->be_obj))));
  if( fun->parameters != NULL )
    for( i = 0;  i < ArraySize(fun->parameters);  i++ )
    {
      be->CallContinue(be->logical_and_seq, i+2);
      param = ArrayGet(fun->parameters, i);
      Equal(Call2(NPBack_Array_Get, Var(entry_be_var), Int(i+1)),
	Cast(be->int_type, BEFnParamBEType(param), Var(BEFnParamBEVar(param))));
    }
  be->CallEnd(be->logical_and_seq);
}


/*****************************************************************************/
/*                                                                           */
/*  void EntryFill(BEFN fun, CODEGEN_OBJ sig_be_var,                         */
/*    CODEGEN_OBJ entry_be_var, CODEGEN_OBJ res_be_var, CODEGEN be)          */
/*                                                                           */
/*  Generate code to fill the attributes of entry_be_var with the            */
/*  signature, function address, parameters, and result.                     */
/*                                                                           */
/*****************************************************************************/

static void EntryFill(BEFN fun, CODEGEN_OBJ sig_be_var,
  CODEGEN_OBJ entry_be_var, CODEGEN_OBJ res_be_var, CODEGEN be)
{
  int i;  BEFN_PARAM param;
  Assign(Call1(NPBack_Signature, Var(entry_be_var)), Var(sig_be_var));
  Assign(Call2(NPBack_Array_Get, Var(entry_be_var), Int(0)),
    Cast(be->int_type, NULL, Call1(be->address_of, Var(fun->be_obj))));
  if( fun->parameters != NULL )
    for( i = 0;  i < ArraySize(fun->parameters);  i++ )
    {
      param = ArrayGet(fun->parameters, i);
      Assign(Call2(NPBack_Array_Get, Var(entry_be_var), Int(i+1)),
	Cast(be->int_type, BEFnParamBEType(param), Var(BEFnParamBEVar(param))));
    }
  Assign(Call2(NPBack_Array_Get, Var(entry_be_var), Int(i+1)),
    Cast(be->int_type, fun->be_type, Var(res_be_var)));
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnCodeGen(BEFN fun, CODEGEN be)                                   */
/*                                                                           */
/*  Generate C function fun onto backend be.  Use dynamic dispatch to        */
/*  generate the body.                                                       */
/*                                                                           */
/*****************************************************************************/

void BEFnCodeGen(BEFN fun, CODEGEN be)
{
  BEFN befn;  int param_count, width;
  CODEGEN_OBJ res_be_var, sig_be_var, pos_be_var, entry_be_var;
  if( DEBUG4 )
    fprintf(stderr, "[ BEFnCodeGen(%s, be)\n", be->VarShow(fun->be_obj));
  assert(fun->be_obj != NULL);

  /* preceding letdefs and preconditions, possibly with prototypes first */
  if( fun->inner_functs != NULL )
  {
    if( ArraySize(fun->inner_functs) >= 2 )
      ArrayForEach(fun->inner_functs, befn)
	BEFnCodeGenPrototype(befn, be);
    ArrayForEach(fun->inner_functs, befn)
      BEFnCodeGen(befn, be);
  }

  /* function header */
  be->FunctionBegin(fun->be_obj, fun->be_type);
  if( fun->parameters != NULL )
    BEFnParamListSetBENamesAndDeclare(fun->parameters, fun->be_obj, be);
  be->FunctionContinue(fun->be_obj);
  param_count = (fun->parameters != NULL ? ArraySize(fun->parameters) : 0);

  /* check cache, if we are caching this function */
  if( fun->cached && fun->inline_be_obj == NULL )
  {
    /* variables for caching; fresh ones, in case clash with params */
    res_be_var = be->VarMakeAndDeclare("res", NULL, fun->be_type);
    entry_be_var = be->VarMakeAndDeclare("entry", NULL, NPBack_Cache_Fun_Type);
    sig_be_var = be->VarMake("sig", NULL);
    pos_be_var = be->VarMake("pos", NULL);
    be->VarDeclare2(sig_be_var, pos_be_var, be->uint_type);

    /* calculate signature */
    be->CommentSmall("calculate hash signature", FALSE, TRUE);
    VarAssign(sig_be_var, Call2(be->lshift, Cast(be->int_type, NULL,
      Call1(be->address_of, Var(fun->be_obj))), Int(16)));
    BEFnCodeGenParamHash(fun, sig_be_var, be);

    /* search hash table */
    be->CommentSmall("return previous result if in cache", FALSE, TRUE);
    VarAssign(pos_be_var, Call2(be->mod, Var(sig_be_var),
      Var(NPBack_Fun_Cache_Len)));
    VarAssign(entry_be_var, Index(Var(NPBack_Fun_Cache), Var(pos_be_var)));
    While( Var(entry_be_var),
    ( If( EntryMatches(fun, sig_be_var, entry_be_var, be),
	Indent(Return(Cast(fun->be_type, NULL,
	  Call2(NPBack_Array_Get, Var(entry_be_var), Int(param_count+1)))))
      ),
      VarAssign(pos_be_var, Call2(be->mod,
	Call2(be->add, Var(pos_be_var), Int(1)), Var(NPBack_Fun_Cache_Len))),
      VarAssign(entry_be_var, Index(Var(NPBack_Fun_Cache), Var(pos_be_var)))
    ));

    /* function body */
    be->CommentSmall("evaluate function", FALSE, TRUE);
    be->BlockBegin();
  }
  else
    res_be_var = be->ret;

  /* function body */
  switch( fun->kind_tag )
  {
    case KIND_BEFN_CREATION:

      BEFnCreationCodeGenBody((BEFN_CREATION) fun, res_be_var, be);
      break;


    case KIND_BEFN_CREDFT:

      BEFnCreDftCodeGenBody((BEFN_CREDFT) fun, res_be_var, be);
      break;


    case KIND_BEFN_FEATURE:

      BEFnFeatureCodeGenBody((BEFN_FEATURE) fun, res_be_var, be);
      break;


    case KIND_BEFN_PRECOND:

      BEFnPrecondCodeGenBody((BEFN_PRECOND) fun, res_be_var, be);
      break;


    case KIND_BEFN_BUILTIN:

      /* builtin functions should never be code generated */
      assert(FALSE);
      break;


    case KIND_BEFN_LETDEF:

      BEFnLetDefCodeGenBody((BEFN_LETDEF) fun, res_be_var, be);
      break;


    case KIND_BEFN_DOWNDEF:
    case KIND_BEFN_PARAM:

      /* downdefs and parameters should never be code generated */
      assert(FALSE);
      break;


    case KIND_BEFN_INVARIANT:

      BEFnInvtCodeGenBody((BEFN_INVARIANT) fun, res_be_var, be);
      break;


    case KIND_BEFN_CLASS_INIT:

      BEFnClassInitCodeGenBody((BEFN_CLASS_INIT) fun, res_be_var, be);
      break;


    case KIND_BEFN_ENUM_INIT:

      BEFnEnumInitCodeGenBody((BEFN_ENUM_INIT) fun, res_be_var, be);
      break;


    case KIND_BEFN_SYSTEM_INIT:

      BEFnSystemInitCodeGenBody((BEFN_SYSTEM_INIT) fun, res_be_var, be);
      break;


    default:

      assert(FALSE);
      break;

  }

  /* insert into cache and return, if we are caching this function */
  if( fun->cached && fun->inline_be_obj == NULL )
  {
    be->BlockEnd();
    be->CommentSmall("make cache entry, insert it, and return", FALSE, TRUE);
    width = sizeof(unsigned int) + (param_count + 2) * sizeof(int);
    VarAssign(entry_be_var,
      Cast(NPBack_Cache_Fun_Type, NULL, Call1(NPBack_Fun_New, Int(width))));
    EntryFill(fun, sig_be_var, entry_be_var, res_be_var, be);
    Stmt(Call2(NPBack_Fun_Cache_Insert, Var(entry_be_var), Var(pos_be_var)));
    Return(Var(res_be_var));
    be->ObjFree(sig_be_var);
    be->ObjFree(pos_be_var);
    be->ObjFree(res_be_var);
    be->ObjFree(entry_be_var);
  }

  /* function footer */
  be->FunctionEnd(fun->be_obj);
  if( DEBUG4 )
    fprintf(stderr, "] BEFnCodeGen returning\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void PrintParams(CODEGEN_OBJ be_obj, ARRAY_BEFN_PARAM parameters,        */
/*    ARRAY_EXPR sorted_actuals, int already_printed, CODEGEN be)            */
/*                                                                           */
/*  Called by BEFnCallCodeGen just below to print the actual                 */
/*  parameters.  Parameter already_printed tells how many parameters         */
/*  have already been printed.                                               */
/*                                                                           */
/*****************************************************************************/

static void PrintParams(CODEGEN_OBJ be_obj, ARRAY_BEFN_PARAM parameters,
  ARRAY_EXPR sorted_actuals, int already_printed, CODEGEN be)
{
  int i;  BEFN_PARAM param;  EXPR actual;
  if( sorted_actuals != NULL )
    for( i = 0;  i < ArraySize(sorted_actuals);  i++ )
    {
      param = ArrayGet(parameters, i);
      actual = ArrayGet(sorted_actuals, i);
      if( actual != NULL )
      {
	if( already_printed > 0 )
	  be->CallContinue(be_obj, already_printed);
	ExprCodeGen(actual, NULL, BEFnParamBEType(param), be);
	already_printed++;
      }
    }
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnCallCodeGen(BEFN fun, ARRAY_EXPR sorted_actuals,                */
/*    CODEGEN_TYPE res_be_type, CODEGEN be)                                  */
/*                                                                           */
/*  Generate a call to function fun with sorted_actuals for its actual       */
/*  parameters.  If there are fewer actual parameters than formals, the      */
/*  call is a curried one, i.e. creation of a fun_n_m object.  The other     */
/*  parameters are as in ExprCodeGen.                                        */
/*                                                                           */
/*  This function assumes that all the parameters are small-scale; it        */
/*  just generates the actual call, with expressions for the parameters;     */
/*  it does not generate separate, preceding code for evaluating large-      */
/*  scale parameters.  ExprCallCodeGen does that, and it also sets up any    */
/*  required assignment to a variable.  Consequently the res_be_var          */
/*  parameter that usually appears in code gen calls is always NULL here     */
/*  and has been omitted.                                                    */
/*                                                                           */
/*  The functions called are syntactically quite heterogeneous in the        */
/*  C back end, however that is the back end's problem.                      */
/*                                                                           */
/*****************************************************************************/

void BEFnCallCodeGen(BEFN fun, ARRAY_EXPR sorted_actuals,
  CODEGEN_TYPE res_be_type, CODEGEN be)
{
  EXPR e;  int missing_compulsories, param_count;
  BEFN curry;

  if( DEBUG6 )
  {
    fprintf(stderr, "[ BEFnCallCodeGen(%p: %s, actuals %s NULL, ...)\n",
      (void *) fun, KindShow(fun->kind_tag),
      sorted_actuals == NULL ? "==" : "!=");
    fprintf(stderr, "  be_obj = %s\n", be->VarShow(fun->be_obj));
    fprintf(stderr, "  inline_be_obj = %s\n",
      fun->inline_be_obj == NULL ? "NULL" : be->VarShow(fun->inline_be_obj));
  }

  /* cast if needed */
  be->CastBegin(res_be_type, fun->be_type);

  /* work out how many missing compulsory parameters there are */
  if( sorted_actuals != NULL )
  {
    missing_compulsories = 0;
    ArrayForEach(sorted_actuals, e)
      if( e == NULL )
	missing_compulsories++;
  }
  else
    missing_compulsories =
      fun->parameters != NULL ? ArraySize(fun->parameters) : 0;

  if( missing_compulsories > 0 )
  {
    /* missing compulsories, so generate a curried call */
    param_count = fun->parameters != NULL ? ArraySize(fun->parameters) : 0;
    curry = (BEFN)
      ClassBEFnCreation(ClassFunNM[param_count][missing_compulsories]);
    be->CallBegin(curry->be_obj);
    Cast(ClassBEType(ClassFunRep[param_count]), NULL,
      Call1(be->address_of, Var(fun->be_obj))
    );
    PrintParams(curry->be_obj, fun->parameters, sorted_actuals, 1, be);
    be->CallEnd(curry->be_obj);
  }
  else if( fun->inline_be_obj != NULL )
  {
    /* generate an inline call */
    be->CallBegin(fun->inline_be_obj);
    PrintParams(fun->inline_be_obj, fun->parameters, sorted_actuals, 0, be);
    be->CallEnd(fun->inline_be_obj);
  }
  else
  {
    /* no missing compulsories, so generate a direct call */
    be->CallBegin(fun->be_obj);
    PrintParams(fun->be_obj, fun->parameters, sorted_actuals, 0, be);
    be->CallEnd(fun->be_obj);
  }

  be->CastEnd();
  if( DEBUG6 )
    fprintf(stderr, "] BEFnCallCodeGen returning\n");
}
