/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         befn_class_init.c                                          */
/*  DESCRIPTION:  Back-end class initialization function                     */
/*                                                                           */
/*****************************************************************************/
#include <string.h>
#include "externs.h"
#define DEBUG1 0
#define DEBUG2 0


/*****************************************************************************/
/*                                                                           */
/*  VISIT_STATE                                                              */
/*                                                                           */
/*  The initialization state of a class initialization function.             */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  VISIT_NEW,					/* search never reached here */
  VISIT_ACTIVE,					/* search active from here   */
  VISIT_CYCLE,					/* search found cycle here   */
  VISIT_COMPLETE				/* search complete here      */
} VISIT_STATE;


/*****************************************************************************/
/*                                                                           */
/*  BEFN_CLASS_INIT                                                          */
/*                                                                           */
/*****************************************************************************/

struct befn_class_init_rec {		/* inherits from BEFN                */
  KIND_TAG		kind_tag;	/* kind of entity                    */
  CODEGEN_TYPE		be_type;	/* result type                       */
  CODEGEN_OBJ		be_obj;		/* corresponding backend function    */
  ARRAY_BEFN		inner_functs;	/* to generate just before this fn   */
  ARRAY_BEFN_PARAM	parameters;	/* ordinary params (NULL here)       */
  CODEGEN_OBJ		inline_be_obj;	/* non-NULL if to be inlined         */
  BOOLEAN		utilized;	/* TRUE when has been called         */
  BOOLEAN		cached;		/* TRUE when calls are to be cached  */

  CLASS			class;		/* the class being initialized       */

  /* initialization order */
  int			visit_num;	/* visit number for cycle stopping   */
  VISIT_STATE		visit_state;	/* visit state (predefs only)        */
  int			visit_index;	/* position in all_enumerated only   */

  /* load function (piggybacked onto init function) */
  CODEGEN_OBJ		be_load_obj;
};


/*****************************************************************************/
/*                                                                           */
/*  BEFN_CLASS_INIT BEFnClassInitMake(CLASS c)                               */
/*                                                                           */
/*  Make a new class initialization function.                                */
/*                                                                           */
/*****************************************************************************/

BEFN_CLASS_INIT BEFnClassInitMake(CLASS c)
{
  BEFN_CLASS_INIT res;
  GetMemory(res, BEFN_CLASS_INIT);
  res->kind_tag = KIND_BEFN_CLASS_INIT;
  res->be_type = NULL;
  res->be_obj = NULL;
  res->inner_functs = NULL;
  res->parameters = NULL;
  res->inline_be_obj = NULL;
  res->utilized = FALSE;
  res->cached = FALSE;
  res->class = c;
  res->visit_num = 0;
  res->visit_state = VISIT_NEW;
  res->visit_index = 0;
  res->be_load_obj = NULL;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnClassInitFinalize(BEFN_CLASS_INIT fun, CODEGEN be)              */
/*                                                                           */
/*  Finalize the be_type and be_obj fields of fun; also be_load_obj.         */
/*                                                                           */
/*****************************************************************************/

void BEFnClassInitFinalize(BEFN_CLASS_INIT fun, CODEGEN be)
{
  NAME cvn;
  static USTRING init_str = NULL;
  static USTRING load_str = NULL;
  if( init_str == NULL )
  {
    init_str = AStringToUString("init");
    load_str = AStringToUString("load");
  }
  fun->be_type = ClassBEType(ClassArray);
  cvn = ClassViewName(ClassOrigClassView(fun->class));
  fun->be_obj = be->FunctionMake2(NameRep(cvn), init_str, TRUE, TRUE);
  fun->be_load_obj = be->FunctionMake2(NameRep(cvn), load_str, TRUE, TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN BEFnClassInitInitOrder(BEFN_CLASS_INIT befn, int visit_num,      */
/*    BOOLEAN *report, BEFN_SYSTEM_INIT fun, int predef_index)               */
/*                                                                           */
/*  Carry out the specification of BEFnInitOrder on class initialization     */
/*  function befn.  The implementation here is more complex than in other    */
/*  cases because class initialization functions are coloured vertices       */
/*  in the general algorithm, and because we want to permit cycles when      */
/*  they merely report calls on predefined objects defined earlier in this   */
/*  function (see the Implementation Notes for more details).                */
/*                                                                           */
/*  This function is not actually called from BEFnInitOrder.  It has an      */
/*  extra parameter, predef_index, holding the index of the predefined       */
/*  object that triggered this call, and is called only from places where    */
/*  a reasonable value for this index is available:                          */
/*                                                                           */
/*    -1         This value is supplied by the initial, root call.  This     */
/*               call must find state VISIT_NEW or VISIT_COMPLETED, and      */
/*               predef_index is not used in either of these states, so      */
/*               actually the value here is a don't-care.                    */
/*                                                                           */
/*    0, 1, ...  An ordinary value like this is supplied by a call from      */
/*               a predefined object in a non-enumerated class, and is       */
/*               the index of that object in its class.                      */
/*                                                                           */
/*    INT_MAX    This value is supplied by calls that require the entire     */
/*               array to be initialized.                                    */
/*                                                                           */
/*****************************************************************************/

BOOLEAN BEFnClassInitInitOrder(BEFN_CLASS_INIT befn, int visit_num,
  BOOLEAN *report, BEFN_SYSTEM_INIT fun, int predef_index)
{
  ARRAY_BEFN_FEATURE all_predefs;  BEFN_FEATURE predef;
  BEFN_CLASS_INIT other_class_init;
  static int max_visit_num = 0;
  if( DEBUG1 )
    fprintf(stderr, "[ BEFnClassInitInitOrder(befn, %d, *report, fun, %d)\n",
      visit_num, predef_index);
  switch( befn->visit_state )
  {
    case VISIT_NEW:

      /* befn has never been encountered until now; visit all predef bodies */
      befn->visit_state = VISIT_ACTIVE;
      befn->visit_num = ++max_visit_num;
      all_predefs = ClassPredefs(befn->class);
      /* befn->visit_index initialized to 0 at creation, and needs to be! */
      while(  befn->visit_index < ArraySize(all_predefs) )
      {
        predef = ArrayGet(all_predefs, befn->visit_index);
	if( BEFnFeatureSoleClass(predef) == befn->class )
	{
	  /* original to this class, will code gen expression */
	  if( !ExprInitOrder(BEFnFeatureSoleBody(predef), befn->visit_num,
		report, fun) )
	    db_return(DEBUG1, "BEFnClassInitInitOrder (new)", FALSE);
	}
	else
	{
	  /* some other class, need to ensure that's initialized first */
	  other_class_init = ClassBEFnClassInit(BEFnFeatureSoleClass(predef));
	  BEFnClassInitInitOrder(other_class_init, befn->visit_num, report,
	    fun, 0);
	}
	befn->visit_index++;
      }

      /* success so mark befn as complete and report class to fun */
      befn->visit_state = VISIT_COMPLETE;
      BEFnSystemInitRegisterClass(fun, befn->class);
      db_return(DEBUG1, "BEFnClassInitInitOrder (new)", TRUE);


    case VISIT_ACTIVE:

      /* if good cycle (just an earlier predef already initialized), success */
      if( visit_num == befn->visit_num && predef_index < befn->visit_index )
	db_return(DEBUG1, "BEFnClassInitInitOrder (active but safe)", TRUE);

      /* if bad cycle, error */
      befn->visit_state = VISIT_CYCLE;
      fprintf(stderr, "%s: class \"%s\" lies on initialization cycle\n",
	FilePosShow(ClassFilePos(befn->class)),
	NameShow(ClassViewName(ClassOrigClassView(befn->class))));
      *report = TRUE;
      db_return(DEBUG1, "BEFnClassInitInitOrder (active)", FALSE);


    case VISIT_COMPLETE:

      /* already done */
      db_return(DEBUG1, "BEFnClassInitInitOrder (complete)", TRUE);


    case VISIT_CYCLE:
    default:

      /* should never come down on a cycle node */
      assert(FALSE);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void AssignInterval(CODEGEN_OBJ be_obj, CODEGEN_OBJ elem_be_obj,         */
/*    CODEGEN_OBJ val_be_obj, CODEGEN be, int j1, int j2)                    */
/*                                                                           */
/*  Assign val_be_obj to every element of be_obj in the range j1 .. j2 incl. */
/*                                                                           */
/*****************************************************************************/

static void AssignInterval(CODEGEN_OBJ be_obj, CODEGEN_OBJ elem_be_obj,
  CODEGEN_OBJ val_be_obj, CODEGEN be, int j1, int j2)
{
  assert(j1 <= j2);
  if( j2 == j1 )
  {
    /* single assignment */
    be->VarIndexedAttributeUpdateIndex(elem_be_obj, j1);
    VarAssign(elem_be_obj, Var(val_be_obj));
  }
  else if( j2 == j1 + 1 )
  {
    /* two assignments */
    be->VarIndexedAttributeUpdateIndex(elem_be_obj, j1);
    VarAssign(elem_be_obj, Var(val_be_obj));
    be->VarIndexedAttributeUpdateIndex(elem_be_obj, j2);
    VarAssign(elem_be_obj, Var(val_be_obj));
  }
  else
  {
    /* more than two assignments, use a for loop */
    be->AsstArrayInit(be_obj, NPBack_I, j1, j2, val_be_obj);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void ClassInitEnumSubTrie(CODEGEN_OBJ be_obj, CODEGEN_OBJ be_obj2,       */
/*    CODEGEN be, int trie_width, int trie_height, int trie_first_code,      */
/*    ARRAY_BEFN_FEATURE all_predefs, int *i, BOOLEAN const_predefs)         */
/*                                                                           */
/*  Generate code for initializing a subtrie of an enum trie.  A pointer     */
/*  to the root of this subtrie is to be assigned to be_obj, and also to     */
/*  be_obj2 if non-NULL.                                                     */
/*                                                                           */
/*  The subtrie has width (array size) trie_width and height trie_height.    */
/*  For example, a trie that is just an array would have height 1, and       */
/*  this is the minimum legal height.  The first code point that it covers   */
/*  is trie_first_code.                                                      */
/*                                                                           */
/*  The first as yet unassigned (or incompletely assigned) predef is         */
/*  all_predefs[*i].  These predefs are sorted by increasing code number,    */
/*  and there are no code number clashes, although there may be gaps, which  */
/*  lead to NULL pointers at the bottom of the trie.  If const_predefs is    */
/*  TRUE it means that the predefined objects have no creation variables,    */
/*  so they must all be equal, and only one actual creation is needed.       */
/*                                                                           */
/*  Implementation note                                                      */
/*  -------------------                                                      */
/*                                                                           */
/*  However the elements of this trie are assigned, the ultimate result      */
/*  must be an optimized trie, because the array_create calls ensure that    */
/*  any duplicate values are shared.  But this code tries very hard not to   */
/*  generate such duplicate arrays anyway, since for the char class that     */
/*  would be too slow.  It also generates for loops to initialize sequences  */
/*  of three or more identical values, which are very common, instead of     */
/*  tedious sequences of individual assignments.                             */
/*                                                                           */
/*  An *index interval* is a set of consecutive array indexes.  A *code      */
/*  interval* is a set of consecutive code numbers.  The numbers in a        */
/*  code interval do not have to be assigned to any predefined object.       */
/*                                                                           */
/*  Every index interval [j1, j2] corresponds with a code interval [i1, i2], */
/*  these being the codes that are handled by indexing through any of the    */
/*  trie entries with indexes j1 to j2.  For example, the entire subtrie     */
/*  has index interval [0, trie_width-1] and corresponds with code interval  */
/*                                                                           */
/*      [trie_first_code, trie_first_code + trie_width*code_step - 1]        */
/*                                                                           */
/*  where code_step is 1 if the trie has height 1, 256 if it has height      */
/*  2, and so on; in general, code_step = 256**(trie_height - 1).            */
/*                                                                           */
/*  A *uniform code interval* [i1, i2] is a code interval whose elements     */
/*  are all either contained in the code range of a single predefined        */
/*  object, or else not contained in the code range of any predefined        */
/*  object.  Either way, these codes map to the same value.                  */
/*                                                                           */
/*  A *uniform index interval* [j1, j2] is an index interval whose           */
/*  corresponding code interval is uniform.  The values assigned to the      */
/*  trie indexes in this interval should clearly be equal, since they        */
/*  all lead eventually to the same uniform value.  The algorithm works      */
/*  by identifying and assigning entire uniform index intervals, rather      */
/*  than assigning individual entries.                                       */
/*                                                                           */
/*  The main loop of the algorithm iterates over consecutive uniform         */
/*  subintervals [j1, j2] of [0, trie_width-1].  Within the main loop, for   */
/*  a given j1, the first job is to find the maximum j2 such that [j1, j2]   */
/*  is uniform.  To do this, first find the maximum uniform code interval    */
/*  [i1, i2], where i1 corresponds with j1, and then find the maximum j2     */
/*  such that [j1, j2] corresponds with a subinterval of [i1, i2].           */
/*                                                                           */
/*  Once [j1, j2] has been identified, the main loop proceeds to generate    */
/*  code to assign values to the trie entries in this interval.  There       */
/*  are quite a few cases, which we take in turn, following the code.        */
/*                                                                           */
/*  If [j1, j2] is empty, it must mean that we are not at the bottom         */
/*  level, and even the interval [j1, j1] is not uniform, so we must         */
/*  recursively generate a lower-level subtrie.                              */
/*                                                                           */
/*  Otherwise we have a non-empty uniform index interval.  If this is        */
/*  an interval not contained in any code range ("is_gap"), we may assign    */
/*  a "null subtrie" value to the entries in the interval.  This value is    */
/*  a null pointer at height 1, a pointer to an array of null pointers       */
/*  at height 2, and so on, which has been pre-created.                      */
/*                                                                           */
/*  Otherwise we have a non-empty uniform index interval of non-null         */
/*  entries.  Check to see whether the previous entry at this height lies    */
/*  within the same predefined object.  If so, then the appropriate trie     */
/*  will have already been generated and assigned to a "subtrie" variable,   */
/*  which would be a pointer to a content object at height 1 and a pointer   */
/*  to an array at higher levels.  So assign this existing value to each     */
/*  index in the uniform interval.                                           */
/*                                                                           */
/*  Otherwise the value to be assigned to this uniform index interval        */
/*  almost certainly does not exist yet and so must be generated.  Check     */
/*  to see whether the subtrie height is greater than 1.  If so, generating  */
/*  this value is done by a recursive call, so do that and then assign the   */
/*  resulting value to each index in the uniform interval.                   */
/*                                                                           */
/*  Otherwise we are at height 1 and the value is generated by code          */
/*  generating the content object; this value is then assigned to            */
/*  each index in the uniform interval.  An optimization here is to          */
/*  not use an auxiliary variable to hold the value if it is to be           */
/*  assigned to only one index;  instead, it is code generated directly      */
/*  into that position.                                                      */
/*                                                                           */
/*  NPBack_SubTrie[h] and NPBack_NullTrie[h] are backend variables whose     */
/*  values are tries of height h, where 0 <= h <= 3, as follows:             */
/*                                                                           */
/*      h  NPBack_SubTrie[h]   NPBack_NullTrie[h]                            */
/*      -----------------------------------------                            */
/*      0  "val"               "NULL"                                        */
/*      1  "sub_trie1"         "null_trie1"                                  */
/*      2  "sub_trie2"         "null_trie2"                                  */
/*      3  "sub_trie3"         "null_trie3"                                  */
/*      -----------------------------------------                            */
/*                                                                           */
/*  The variables "null_trie1", "null_trie2", and "null_trie3" are           */
/*  pre-created globals pointing to an array of pointers to null, an array   */
/*  of arrays of pointers to null, etc.  The other variables are declared    */
/*  at the start of the generated function.                                  */
/*                                                                           */
/*****************************************************************************/
#define min(a, b) ((a) < (b) ? (a) : (b))

static void ClassInitEnumSubTrie(CODEGEN_OBJ be_obj, CODEGEN_OBJ be_obj2,
  CODEGEN be, int trie_width, int trie_height, int trie_first_code,
  ARRAY_BEFN_FEATURE all_predefs, int *i, BOOLEAN const_predefs)
{
  int trie_last_code, i1, i2, j;  BOOLEAN is_gap;
  int j1, j2, code_step, code1, code2, f_code1, f_code2;
  static char buff[100];
  CODEGEN_OBJ elem_be_obj, val_be_obj;  BEFN_FEATURE predef, last_predef, f;

  /* work out the code step at this height, and the last code number covered */
  assert(trie_height >= 1);
  code_step = 1;
  for( j1 = 1;  j1 < trie_height;  j1++ )
    code_step *= 256;
  trie_last_code = trie_first_code + trie_width * code_step - 1;

  /* initialize the subtrie */
  VarAssign2(be_obj, be_obj2, Call1(NPBack_Array_Create_Fn1, Int(trie_width)));

  elem_be_obj = be->VarIndexedAttributeMake(be_obj, "elems", 0);
  for( j1 = 0;  j1 < trie_width;  j1 = j2 + 1 )
  {
    /* find i1, the code number corresponding to index j1 */
    i1 = trie_first_code + code_step * j1;

    /* find predef, the first predef relevant to i1, if any */
    while( *i < ArraySize(all_predefs) )
    {
      predef = ArrayGet(all_predefs, *i);
      if( !BEFnFeatureHasCodeNumbers(predef, &code1, &code2) )
	assert(FALSE);
      if( i1 <= code2 )
	break;
      (*i)++;
    }


    /* find a maximum uniform code interval [i1, i2] starting at j1 */
    if( *i >= ArraySize(all_predefs) )
    {
      /* gap to the end of the trie */
      i2 = trie_last_code;
      is_gap = TRUE;
    }
    else if( code1 > i1 )
    {
      /* gap to just before code1, or to end of trie if that comes first */
      i2 = min(code1 - 1, trie_last_code);
      is_gap = TRUE;
    }
    else
    {
      /* if constant, increase code2 to the last contiguous predef */
      if( const_predefs )
      {
	j = (*i) + 1;
	while( j < ArraySize(all_predefs) )
	{
	  f = ArrayGet(all_predefs, j);
	  if( !BEFnFeatureHasCodeNumbers(f, &f_code1, &f_code2) )
	    assert(FALSE);
	  if( f_code1 > code2 + 1 )
	    break;
	  code2 = f_code2;
	  last_predef = predef;
	  j++;
	}
      }

      /* non-gap to code2, or to end of trie if that comes first */
      i2 = min(code2, trie_last_code);
      is_gap = FALSE;
    }
    j2 = (i2 - trie_first_code + 1)/code_step - 1;
    if( DEBUG2 )
    {
      sprintf(buff, "index interval [%d, %d], code interval [0x%04X, 0x%04X]%s",
	j1, j2, i1, i2, is_gap ? ", is_gap" : "");
      be->CommentSmall(buff, FALSE, TRUE);
    }

    /* handle [j1, j2], the max uniform index interval for [i1, i2] */
    if( j2 < j1 )
    {
      /* the maximum uniform index interval is empty, so recurse */
      assert(trie_height > 1);
      val_be_obj = NPBack_SubTrie[trie_height-1];
      be->BlockBegin();
      be->VarIndexedAttributeUpdateIndex(elem_be_obj, j1);
      ClassInitEnumSubTrie(val_be_obj, elem_be_obj, be, 256, trie_height-1,
	i1, all_predefs, i, const_predefs);
      be->BlockEnd();
      VarAssign(elem_be_obj, Var(val_be_obj));
      j2 = j1;
    }
    else if( is_gap )
    {
      /* there is a uniform index interval, and it has null entries */
      val_be_obj = NPBack_NullTrie[trie_height-1];
      AssignInterval(be_obj, elem_be_obj, val_be_obj, be, j1, j2);
    }
    else if( code1 <= i1 - code_step )
    {
      /* there is a uniform index interval with non-null entries, whose */
      /* value will have been already generated, so just assign it */
      val_be_obj = NPBack_SubTrie[trie_height-1];
      AssignInterval(be_obj, elem_be_obj, val_be_obj, be, j1, j2);
    }
    else if( trie_height > 1 )
    {
      /* there is a uniform index interval with non-null entries, whose */
      /* value has not been assigned yet, and not leaf, so recurse */
      val_be_obj = NPBack_SubTrie[trie_height-1];
      be->BlockBegin();
      be->VarIndexedAttributeUpdateIndex(elem_be_obj, j1);
      ClassInitEnumSubTrie(val_be_obj, elem_be_obj, be, 256, trie_height-1,
	i1, all_predefs, i, const_predefs);
      be->BlockEnd();
      AssignInterval(be_obj, elem_be_obj, val_be_obj, be, j1, j2);
    }
    else if( code2 == code1 )
    {
      /* there is a uniform index interval with non-null entries, whose */
      /* value has not been assigned yet, and which is a leaf, and what's */
      /* more a unique code point; so code gen and assign directly to index */
      be->VarIndexedAttributeUpdateIndex(elem_be_obj, j1);
      ExprCodeGen(BEFnFeatureSoleBody(predef), elem_be_obj, be->voidp_type, be);
    }
    else
    {
      /* there is a uniform index interval with non-null entries, whose */
      /* value has not been assigned yet, and which is a leaf, but not a */
      /* unique code point; so code gen to "val" and assign to interval */
      val_be_obj = NPBack_SubTrie[trie_height-1];
      ExprCodeGen(BEFnFeatureSoleBody(predef), val_be_obj, be->voidp_type, be);
      AssignInterval(be_obj, elem_be_obj, val_be_obj, be, j1, j2);
    }
  }

  /* finalize the subtrie */
  VarAssign(be_obj, Call1(NPBack_Array_Create_Fn2, Var(be_obj)));
}


/*****************************************************************************/
/*                                                                           */
/*  void CallInvariants(CLASS c, int code1, int code2, CODEGEN be)           */
/*                                                                           */
/*  Generate code to check the invariants of all ancestors of class c on     */
/*  each code point from code1 to code2 inclusive.                           */
/*                                                                           */
/*****************************************************************************/

static void CallInvariants(CLASS c, int code1, int code2, CODEGEN be)
{
  CODEGEN_OBJ invt_be_obj;  ITYPES ancestor_set;  ITYPE itype;
  BEFN_INVARIANT invt;

  ancestor_set = ClassAncestorSet(c);
  ArrayForEach(ancestor_set, itype)
  {
    invt = ClassInvariant(TypeITypeClass(itype));
    if( invt != NULL )
    {
      /* generate code to test invt for code1 ... code2 */
      invt_be_obj = BEFnBEObj((BEFN) invt);
      if( code1 == code2 )
      {
	/* generate a single call on the invariant function */
	Stmt(Call1(invt_be_obj, Int(code1)));
      }
      else
      {
	/* generate a call on a macro that loops through the code range */
	Stmt(Call3(NPBack_EnumInvtSeq,Var(invt_be_obj),Int(code1),Int(code2)));
      }
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnClassInitCodeGenInvtChecks(BEFN_CLASS_INIT fun, CODEGEN be)     */
/*                                                                           */
/*  Generate the invariant checks for enumerated class fun->class.           */
/*  Code ranges are handled by macro calls of the form                       */
/*                                                                           */
/*      enum_invt_seq(c_invt, from, to)                                      */
/*                                                                           */
/*  which calls c_invt(from) ... c_invt(to).  Adjacent code ranges are       */
/*  merged into one call.                                                    */
/*                                                                           */
/*****************************************************************************/

static void BEFnClassInitCodeGenInvtChecks(BEFN_CLASS_INIT fun, CODEGEN be)
{
  ARRAY_BEFN_FEATURE all_predefs;
  BEFN_FEATURE predef;  int i, prev_code1, prev_code2, code1, code2;

  /* initialize, write a comment */
  assert(ClassIsEnum(fun->class));
  assert(ClassInvariant(fun->class) != NULL);
  be->CommentSmall("invariant checks", FALSE, TRUE);

  /* find the first predef that needs checking, quit if none */
  all_predefs = ClassPredefs(fun->class);
  for( i = 0;  i < ArraySize(all_predefs);  i++ )
  {
    predef = ArrayFirst(all_predefs);
    if( BEFnFeatureSoleClass(predef) == fun->class )
      break;
  }
  if( i >= ArraySize(all_predefs) )
    return;
  if( !BEFnFeatureHasCodeNumbers(predef, &prev_code1, &prev_code2) )
    assert(FALSE);
  i++;

  /* invariant:  prev_code1 and prev_code2 are well defined; a call  */
  /* to enum_invt_seq(c_invt, prev_code1, prev_code2) is definitely  */
  /* needed, and it will complete the coverage of all predefs in the */
  /* range 0 to i-1 inclusive                                        */
  for( ;  i < ArraySize(all_predefs);  i++ )
  {
    predef = ArrayGet(all_predefs, i);
    if( BEFnFeatureSoleClass(predef) == fun->class )
    {
      /* original to this class so needs its invariant checked */
      if( !BEFnFeatureHasCodeNumbers(predef, &code1, &code2) )
	assert(FALSE);
      assert(code1 > prev_code2);
      if( code1 == prev_code2 + 1 )
      {
	/* merge this predef in with preceding */
	prev_code2 = code2;
      }
      else
      {
	/* code gen preceding and start a new code sequence */
        CallInvariants(fun->class, prev_code1, prev_code2, be);
	prev_code1 = code1;
	prev_code2 = code2;
      }
    }
  }

  /* last call (always needed, see invariant) */
  CallInvariants(fun->class, prev_code1, prev_code2, be);
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnClassInitCodeGenBody(BEFN_CLASS_INIT fun,                       */
/*    CODEGEN_OBJ res_be_var, CODEGEN be)                                    */
/*                                                                           */
/*  Generate the body of class initialization function fun onto be.          */
/*                                                                           */
/*****************************************************************************/

void BEFnClassInitCodeGenBody(BEFN_CLASS_INIT fun,
  CODEGEN_OBJ res_be_var, CODEGEN be)
{
  ARRAY_BEFN_FEATURE all_predefs;
  CODEGEN_OBJ be_obj, predef_be_obj, all_predef_be_obj;
  BEFN_FEATURE predef;  int i;
  if( DEBUG1 )
    fprintf(stderr, "[ BEFnClassInitCodeGenBody(%s, be)\n",
      be->VarShow(fun->be_obj));
  assert(fun->be_obj != NULL);
  all_predefs = ClassPredefs(fun->class);

  if( ClassIsEnum(fun->class) )
  {
    /* declare i, sub_trie, and val variables as needed */
    be->VarDeclare(NPBack_I, ClassBEType(ClassInt));
    if( ClassHasNoCreationFeatures(fun->class) ||
	ClassEnumHasCodeRange(fun->class) )
      be->VarDeclare(NPBack_SubTrie[0], ClassBEContentType(fun->class));
    for( i = 1;  i < ClassEnumTrieHeight(fun->class);  i++ )
      be->VarDeclare(NPBack_SubTrie[i], ClassBEType(ClassArray));

    /* create class_enum_trie trie */
    be->CommentSmall("create and initialize enum_trie", FALSE, TRUE);
    be_obj = BEFnBEObj((BEFN) ClassAllEnumTrieBEFnFeature(fun->class));
    i = 0;
    ClassInitEnumSubTrie(be_obj, NULL, be, ClassEnumTrieWidth(fun->class),
      ClassEnumTrieHeight(fun->class), 0, all_predefs, &i,
      ClassHasNoCreationFeatures(fun->class));

    /* invariant checks, if any */
    if( ClassInvariant(fun->class) != NULL )
      BEFnClassInitCodeGenInvtChecks(fun, be);

    /* return the trie as result object */
    Return(Var(be_obj));
  }
  else
  {
    /* create class_all_predefined array */
    all_predef_be_obj =
      BEFnBEObj((BEFN) ClassAllPredefinedBEFnFeature(fun->class));
    VarAssign(all_predef_be_obj,
      Call1(NPBack_Array_Create_Fn1, Int(ArraySize(all_predefs))));

    /* initialize the array elements */
    be_obj = be->VarIndexedAttributeMake(all_predef_be_obj, "elems", 0);
    for( i = 0;  i < ArraySize(all_predefs);  i++ )
    {
      predef = ArrayGet(all_predefs, i);
      predef_be_obj = BEFnBEObj((BEFN) predef);
      if( BEFnFeatureSoleClass(predef) == fun->class )
      {
	/* original to this class, do the full code generation */
	ExprCodeGen(BEFnFeatureSoleBody(predef), predef_be_obj,
	  be->voidp_type, be);
      }
      else
      {
	/* already initialized, just assign to the array element */
	be->VarIndexedAttributeUpdateIndex(be_obj, i);
	VarAssign(be_obj, Var(predef_be_obj));
      }
    }

    /* finalize class_all_predefined array */
    VarAssign(all_predef_be_obj,
      Call1(NPBack_Array_Create_Fn2, Var(all_predef_be_obj)));
    VarAssign(res_be_var, Var(all_predef_be_obj));
  }

  if( DEBUG1 )
    fprintf(stderr, "] BEFnClassInitCodeGenBody returning\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnClassInitCodeGenCall(BEFN_CLASS_INIT fun, CODEGEN be)           */
/*                                                                           */
/*  Generate a call on class initialization function fun.                    */
/*                                                                           */
/*****************************************************************************/

void BEFnClassInitCodeGenCall(BEFN_CLASS_INIT fun, CODEGEN be)
{
  BEFnCallCodeGen((BEFN) fun, NULL, be->voidp_type, be);
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnClassInitCodeGenLoadFn(BEFN_CLASS_INIT fun, CODEGEN be)         */
/*                                                                           */
/*  Code gen the class load function that is piggybacked onto this           */
/*  init function.                                                           */
/*                                                                           */
/*****************************************************************************/

void BEFnClassInitCodeGenLoadFn(BEFN_CLASS_INIT fun, CODEGEN be)
{
  CODEGEN_OBJ be_var;

  /* work out whether loading trie or all_predefined */
  if( ClassIsEnum(fun->class) )
    be_var = BEFnBEObj((BEFN) ClassAllEnumTrieBEFnFeature(fun->class));
  else
    be_var = BEFnBEObj((BEFN) ClassAllPredefinedBEFnFeature(fun->class));
    
  /* do the code generation */
  be->FunctionBegin(fun->be_load_obj, be->void_type);
  be->FunctionFormal(fun->be_load_obj, NPBack_Other, ClassBEType(ClassArray));
  be->FunctionContinue(fun->be_load_obj);
  VarAssign(be_var, Var(NPBack_Other));
  be->FunctionEnd(fun->be_load_obj);
}


/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_OBJ BEFnClassInitCodeGenLoadBEObj(BEFN_CLASS_INIT fun)           */
/*                                                                           */
/*  Return the load function piggybacked onto initialization function fun.   */
/*                                                                           */
/*****************************************************************************/

CODEGEN_OBJ BEFnClassInitCodeGenLoadBEObj(BEFN_CLASS_INIT fun)
{
  return fun->be_load_obj;
}
