/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         befn_creation.c                                            */
/*  DESCRIPTION:  Back-end creation functions                                */
/*                                                                           */
/*****************************************************************************/
#include "externs.h"
#include <string.h>

struct befn_creation_rec {		/* inherits from BEFN                */
  KIND_TAG		kind_tag;	/* KIND_CLASS                        */
  CODEGEN_TYPE		be_type;	/* backend type                      */
  CODEGEN_OBJ		be_obj;		/* backend function                  */
  ARRAY_BEFN		inner_functs;	/* from body of creation function    */
  ARRAY_BEFN_PARAM	parameters;	/* ordinary params (NULL if none)    */
  CODEGEN_OBJ		inline_be_obj;	/* non-NULL if to be inlined         */
  BOOLEAN		utilized;	/* TRUE when has been called         */
  BOOLEAN		cached;		/* TRUE when calls are to be cached  */

  CLASS			class;		/* class this creation fn is for     */
  int			visit_num;	/* when finding initialization order */
};


/*****************************************************************************/
/*                                                                           */
/*  BEFN_CREATION BEFnCreationNew(CLASS class)                               */
/*                                                                           */
/*  Make a new creation function for class, initially with no parameters.    */
/*                                                                           */
/*****************************************************************************/

BEFN_CREATION BEFnCreationNew(CLASS class)
{
  BEFN_CREATION res;  FILE_POS file_pos;
  GetMemory(res, BEFN_CREATION);
  res->kind_tag = KIND_BEFN_CREATION;
  res->be_type = NULL;
  res->be_obj = NULL;
  res->inner_functs = NULL;
  ArrayInit(&res->parameters);
  res->inline_be_obj = NULL;
  res->utilized = FALSE;
  res->cached = FALSE;
  res->class = class;
  file_pos = ClassViewFilePos(ClassOrigClassView(class));
  res->visit_num = 0;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN BEFnCreationInitOrder(BEFN_CREATION cfn, int visit_num,          */
/*    BOOLEAN *report, BEFN_SYSTEM_INIT fun)                                 */
/*                                                                           */
/*  Carry out the specification of BEFnInitOrder on creation function cfn.   */
/*  NB This function should also check the invariants of cfn.                */
/*                                                                           */
/*****************************************************************************/

BOOLEAN BEFnCreationInitOrder(BEFN_CREATION cfn, int visit_num,
  BOOLEAN *report, BEFN_SYSTEM_INIT fun)
{
  BEFN_PARAM param;  BEFN_FEATURE cfeature;  BEFN_CREDFT befn_credft;
  ITYPES ancestor_set;  ITYPE itype;  BEFN_INVARIANT befn_invariant;

  /* check whether visited already and update if not */
  if( cfn->visit_num >= visit_num )
    return TRUE;
  cfn->visit_num = visit_num;

  /* explore the default parameter values of cfn */
  ArrayForEach(cfn->parameters, param)
  {
    cfeature = BEFnParamCreationFeature(param);
    befn_credft = BEFnFeatureCreationDefaultValue(cfeature);
    if( befn_credft != NULL )
    {
      if( !BEFnInitOrder((BEFN) befn_credft, visit_num, report, fun) )
	return FALSE;
    }
  }

  /* explore the invariants of cfn, own and inherited */
  ancestor_set = ClassAncestorSet(cfn->class);
  ArrayForEach(ancestor_set, itype)
    if( TypeITypeCoercion(itype) == NULL )
    {
      befn_invariant = ClassInvariant(TypeITypeClass(itype));
      if( befn_invariant != NULL )
      {
	if( !BEFnInitOrder((BEFN) befn_invariant, visit_num, report, fun) )
	  return FALSE;
      }
    }

  /* all done so return success */
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnCreationFinalize(BEFN_CREATION cfn, CODEGEN be)                 */
/*                                                                           */
/*  Finalize cfn's be_type and be_obj fields.                                */
/*                                                                           */
/*****************************************************************************/

void BEFnCreationFinalize(BEFN_CREATION cfn, CODEGEN be)
{
  if( ClassIsEnum(cfn->class) )
  {
    cfn->be_type = ClassBEContentType(cfn->class);
    cfn->be_obj =
      be->FunctionMake2(NameRep(ClassViewName(ClassOrigClassView(cfn->class))),
	AStringToUString("content_create"), TRUE, TRUE);
  }
  else
  {
    cfn->be_type = ClassBEType(cfn->class);
    cfn->be_obj =
      be->FunctionMake2(NameRep(ClassViewName(ClassOrigClassView(cfn->class))),
	AStringToUString("create"), TRUE, TRUE);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_OBJ BEFnCreationBEObj(BEFN_CREATION cfn)                         */
/*                                                                           */
/*  Return the backend function corresponding to cfn.                        */
/*                                                                           */
/*****************************************************************************/

CODEGEN_OBJ BEFnCreationBEObj(BEFN_CREATION cfn)
{
  return cfn->be_obj;
}


/*****************************************************************************/
/*                                                                           */
/*  void OtherMatches(BEFN_CREATION cfn, CODEGEN_OBJ other_be_var,           */
/*    CODEGEN_OBJ sig_be_var, CODEGEN be)                                    */
/*                                                                           */
/*  Generate code which checks whether a variable of type other matches      */
/*  the parameter values of the creation function of c;  called just below.  */
/*                                                                           */
/*****************************************************************************/

static void OtherMatches(BEFN_CREATION cfn, CODEGEN_OBJ other_be_var,
  CODEGEN_OBJ sig_be_var, CODEGEN be)
{
  CODEGEN_OBJ cfeature, param_be;  BEFN_PARAM param;  int i;
  be->CallBegin(be->logical_and_seq);
  Equal(Call1(NPBack_Signature, Var(other_be_var)), Var(sig_be_var));
  be->CallContinue(be->logical_and_seq, 1);
  Equal(Call1(NPBack_Type_Tag,Var(other_be_var)),Var(ClassTypeTag(cfn->class)));
  if( cfn->parameters != NULL )
    for( i = 0;  i < ArraySize(cfn->parameters);  i++ )
    {
      param = ArrayGet(cfn->parameters, i);
      cfeature = BEFnFeatureLayoutBEObj(BEFnParamCreationFeature(param));
      param_be = BEFnBEObj((BEFN) param);
      be->CallContinue(be->logical_and_seq, i+2);
      Equal(Call1(cfeature, Var(other_be_var)), Var(param_be));
    }
  be->CallEnd(be->logical_and_seq);
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnCreationCodeGenBody(BEFN_CREATION cfn, CODEGEN_OBJ res_be_var,  */
/*    CODEGEN be)                                                            */
/*                                                                           */
/*  Print the body of creation function cfn onto be, ensuring that its       */
/*  result goes to res_be_var.                                               */
/*                                                                           */
/*****************************************************************************/

void BEFnCreationCodeGenBody(BEFN_CREATION cfn, CODEGEN_OBJ res_be_var,
  CODEGEN be)
{
  int j, width;
  CODEGEN_OBJ param_be_var, cfeature_be_var, other_be_var;
  CODEGEN_OBJ sig_be_var, pos_be_var;
  BEFN_FEATURE cfeature;  BEFN_CREDFT befn_credft;
  BEFN_PARAM param;  BOOLEAN dfts_begun;
  BEFN_INVARIANT befn_invariant;
  ITYPE itype;  ITYPES ancestor_set;

  /* start of body, with declarations of four local variables */
  other_be_var = be->VarMake("other", NULL);
  sig_be_var = be->VarMake("sig", NULL);
  pos_be_var = be->VarMake("pos", NULL);
  be->VarDeclare2(NPBack_Self, other_be_var, cfn->be_type);
  be->VarDeclare2(sig_be_var, pos_be_var, be->uint_type);

  /* default parameter assignments */
  dfts_begun = FALSE;
  for( j = 0;  j < ArraySize(cfn->parameters);  j++ )
  {
    param = ArrayGet(cfn->parameters, j);
    cfeature = BEFnParamCreationFeature(param);
    befn_credft = BEFnFeatureCreationDefaultValue(cfeature);
    if( befn_credft != NULL )
    {
      /* code gen a test for default value and call on befn_credft */
      if( !dfts_begun )
      {
	be->CommentSmall("substitute default values", FALSE, TRUE);
        dfts_begun = TRUE;
      }
      If(Equal(Var(BEFnParamBEVar(param)),
	 Cast(BEFnParamBEType(param), be->int_type, Var(NPBack_Dft_Param_Val))),
	Indent(VarAssign(BEFnParamBEVar(param),
	  BEFnCreDftCodeGenCall(befn_credft, cfn->parameters, be))));
    }
  }

  /* work out hash signature */
  be->CommentSmall("calculate hash signature", FALSE, TRUE);
  VarAssign(sig_be_var,
    Call2(be->lshift, Var(ClassTypeTag(cfn->class)), Int(16)));
  BEFnCodeGenParamHash((BEFN) cfn, sig_be_var, be);

  /* search cache for this object */
  be->CommentSmall("return previously created object if in cache",
    FALSE, TRUE);
  VarAssign(pos_be_var,
    Call2(be->mod, Var(sig_be_var), Var(NPBack_Obj_Cache_Len)));
  VarAssign(other_be_var,
    Cast(cfn->be_type, NULL, Index(Var(NPBack_Obj_Cache), Var(pos_be_var))));
  While( Var(other_be_var),
  ( If( OtherMatches(cfn, other_be_var, sig_be_var, be),
      Indent(Return(Var(other_be_var)))),
    VarAssign(pos_be_var, Call2(be->mod,
      Call2(be->add, Var(pos_be_var), Int(1)), Var(NPBack_Obj_Cache_Len))),
    VarAssign(other_be_var,
      Cast(cfn->be_type, NULL, Index(Var(NPBack_Obj_Cache), Var(pos_be_var))))
  ));

  /* make a new object */
  be->CommentSmall("create and initialize a new object", FALSE, TRUE);
  width = TypeStructWidth(ClassType(cfn->class)) / CHAR_BIT;
  while( width % 4 != 0 ) width++;
  VarAssign(NPBack_Self, Cast(cfn->be_type, NULL,
    Call1(NPBack_Obj_New, Int(width))));
  Assign(Call1(NPBack_Signature, Var(NPBack_Self)), Var(sig_be_var));
  Assign(Call1(NPBack_Type_Tag,Var(NPBack_Self)),Var(ClassTypeTag(cfn->class)));
  ArrayForEach(cfn->parameters, param)
  {
    cfeature_be_var = BEFnFeatureLayoutBEObj(BEFnParamCreationFeature(param));
    param_be_var = BEFnParamBEVar(param);
    Assign(Call1(cfeature_be_var, Var(NPBack_Self)), Var(param_be_var));
  }

  /* insert into hash table, invariant checks, and return */
  if( ClassIsEnum(cfn->class) )
  {
    be->CommentSmall("cache the new object and return", FALSE, TRUE);
    Stmt(Call2(NPBack_Obj_Cache_Insert,
	Cast(be->voidp_type, NULL, Var(NPBack_Self)), Var(pos_be_var)));
  }
  else
  {
    be->CommentSmall("cache the new object, check invariants, and return",
      FALSE, TRUE);
    Stmt(Call2(NPBack_Obj_Cache_Insert,
	Cast(NPBack_Cache_Obj_Type, NULL, Var(NPBack_Self)), Var(pos_be_var)));
    ancestor_set = ClassAncestorSet(cfn->class);
    ArrayForEach(ancestor_set, itype)
      if( TypeITypeCoercion(itype) == NULL )
      {
	befn_invariant = ClassInvariant(TypeITypeClass(itype));
	if( befn_invariant != NULL )
	  BEFnInvtCallCodeGen(befn_invariant, ClassBEType(cfn->class), be);
      }
  }
  VarAssign(res_be_var, Var(NPBack_Self));

  /* free the backend variables */
  be->ObjFree(other_be_var);
  be->ObjFree(sig_be_var);
  be->ObjFree(pos_be_var);
}
