/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         befn_credft.c                                              */
/*  DESCRIPTION:  Back-end creation function default value                   */
/*                                                                           */
/*****************************************************************************/
#include "externs.h"
#include <string.h>

struct befn_credft_rec {		/* inherits from BEFN                */
  KIND_TAG		kind_tag;	/* KIND_CLASS                        */
  CODEGEN_TYPE		be_type;	/* backend type                      */
  CODEGEN_OBJ		be_obj;		/* backend function                  */
  ARRAY_BEFN		inner_functs;	/* from body of default value        */
  ARRAY_BEFN_PARAM	parameters;	/* ordinary params (NULL if none)    */
  CODEGEN_OBJ		inline_be_obj;	/* non-NULL if to be inlined         */
  BOOLEAN		utilized;	/* TRUE when has been called         */
  BOOLEAN		cached;		/* TRUE when calls are to be cached  */

  FEFN_CREDFT		fefn_credft;	/* originating front-end function    */
  EXPR			body;		/* body of default value             */
};


/*****************************************************************************/
/*                                                                           */
/*  BEFN_CREDFT BEFnCreDftNew(FEFN_CREDFT fefn_credft, EXPR body)            */
/*                                                                           */
/*  Make a new creation feature default function corresponding to            */
/*  fefn_credft, with the given body.                                        */
/*                                                                           */
/*****************************************************************************/

BEFN_CREDFT BEFnCreDftNew(FEFN_CREDFT fefn_credft, EXPR body)
{
  BEFN_CREDFT res;
  GetMemory(res, BEFN_CREDFT);
  res->kind_tag = KIND_BEFN_CREDFT;
  res->be_type = NULL;
  res->be_obj = NULL;
  res->inner_functs = NULL;
  ArrayInit(&res->parameters);
  res->inline_be_obj = NULL;
  res->utilized = FALSE;
  res->cached = FALSE;
  res->fefn_credft = fefn_credft;
  res->body = body;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN BEFnCreDftInitOrder(BEFN_CREDFT befn_credft, int visit_num,      */
/*    BOOLEAN *report, BEFN_SYSTEM_INIT fun)                                 */
/*                                                                           */
/*  Implement the specification of BEFnInitOrder on creation default         */
/*  value function befn_credft.                                              */
/*                                                                           */
/*****************************************************************************/

BOOLEAN BEFnCreDftInitOrder(BEFN_CREDFT befn_credft, int visit_num,
  BOOLEAN *report, BEFN_SYSTEM_INIT fun)
{
  return ExprInitOrder(befn_credft->body, visit_num, report, fun);
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnCreDftFinalize(BEFN_CREDFT befn_credft, CODEGEN be)             */
/*                                                                           */
/*  Carry out the specification of BEFnFinalize on this creation feature     */
/*  default value function.                                                  */
/*                                                                           */
/*****************************************************************************/

void BEFnCreDftFinalize(BEFN_CREDFT befn_credft, CODEGEN be)
{
  static USTRING dft_str = NULL;
  FEFN_FEATURE orig_cfeature;  CLASS_VIEW orig_cv;
  if( dft_str == NULL )
    dft_str = AStringToUString("dft");
  befn_credft->be_type =
    TypeBEType(FEFnResultType((FEFN) befn_credft->fefn_credft), be);
  orig_cfeature = FEFnCreDftOrigCreationFeature(befn_credft->fefn_credft);
  orig_cv = FEFnFeatureClassView(orig_cfeature);
  befn_credft->be_obj = be->FunctionMake3(NameRep(ClassViewName(orig_cv)),
    NameRep(FEFnFeatureName(orig_cfeature)), dft_str, TRUE, TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnCreDftCodeGenBody(BEFN_CREDFT befn_credft,                      */
/*    CODEGEN_OBJ res_be_var, CODEGEN be)                                    */
/*                                                                           */
/*  Code generate the body of this creation feature default value function.  */
/*                                                                           */
/*****************************************************************************/

void BEFnCreDftCodeGenBody(BEFN_CREDFT befn_credft,
  CODEGEN_OBJ res_be_var, CODEGEN be)
{
  ExprCodeGen(befn_credft->body, res_be_var, befn_credft->be_type, be);
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnCreDftCodeGenCall(BEFN_CREDFT befn_credft,                      */
/*    ARRAY_BEFN_PARAM params, CODEGEN be)                                   */
/*                                                                           */
/*  Generate a call on befn_credft.  The actual parameters are to be         */
/*  calls on the elements of params that correspond with the parameters      */
/*  of befn_credft.                                                          */
/*                                                                           */
/*****************************************************************************/

void BEFnCreDftCodeGenCall(BEFN_CREDFT befn_credft, ARRAY_BEFN_PARAM params,
  CODEGEN be)
{
  int i, param_count;  BEFN_PARAM befn_param, param;  BEFN_FEATURE cfeature;
  be->CallBegin(befn_credft->be_obj);
  param_count = befn_credft->parameters == NULL ? 0 :
    ArraySize(befn_credft->parameters);
  for( i = 0;  i < param_count;  i++ )
  {
    if( i > 0 )
      be->CallContinue(befn_credft->be_obj, i);

    /* find the element of params corresponding to the ith befn parameter */
    befn_param = ArrayGet(befn_credft->parameters, i);
    cfeature = BEFnParamCreationFeature(befn_param);
    param = BEFnParamListFind(params, cfeature);

    /* generate a call on this parameter */
    Var(BEFnParamBEVar(param));
  }
  be->CallEnd(befn_credft->be_obj);
}
