/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         befn_enum_init.c                                           */
/*  DESCRIPTION:  Backe-end all_enumerated initialization function           */
/*                                                                           */
/*****************************************************************************/
#include <string.h>
#include "externs.h"
#define DEBUG1 0
#define DEBUG2 0

struct befn_enum_init_rec {		/* inherits from BEFN		     */
  KIND_TAG		kind_tag;	/* kind of entity                    */
  CODEGEN_TYPE		be_type;	/* result type                       */
  CODEGEN_OBJ		be_obj;		/* corresponding backend function    */
  ARRAY_BEFN		inner_functs;	/* to generate just before this fn   */
  ARRAY_BEFN_PARAM	parameters;	/* ordinary params (NULL here)       */
  CODEGEN_OBJ		inline_be_obj;	/* non-NULL if to be inlined         */
  BOOLEAN		utilized;	/* TRUE when has been called         */
  BOOLEAN		cached;		/* TRUE when calls are to be cached  */

  CLASS			class;		/* the class being initialized       */

  /* load function (piggybacked onto init function) */
  CODEGEN_OBJ		be_load_obj;
};


/*****************************************************************************/
/*                                                                           */
/*  BEFN_ENUM_INIT BEFnEnumInitMake(CLASS c)                                 */
/*                                                                           */
/*  Make a new all_enumerated initialization function.                       */
/*                                                                           */
/*****************************************************************************/

BEFN_ENUM_INIT BEFnEnumInitMake(CLASS c)
{
  BEFN_ENUM_INIT res;
  GetMemory(res, BEFN_ENUM_INIT);
  res->kind_tag = KIND_BEFN_ENUM_INIT;
  res->be_type = NULL;
  res->be_obj = NULL;
  res->inner_functs = NULL;
  res->parameters = NULL;
  res->inline_be_obj = NULL;
  res->utilized = FALSE;
  res->cached = FALSE;
  res->class = c;
  res->be_load_obj = NULL;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnEnumInitFinalize(BEFN_ENUM_INIT fun, CODEGEN be)                */
/*                                                                           */
/*  Finalize the be_type and be_obj fields of fun; also be_load_obj.         */
/*                                                                           */
/*****************************************************************************/

void BEFnEnumInitFinalize(BEFN_ENUM_INIT fun, CODEGEN be)
{
  NAME cvn;
  static USTRING init_str = NULL;
  static USTRING load_str = NULL;
  if( init_str == NULL )
  {
    init_str = AStringToUString("init_all_enumerated");
    load_str = AStringToUString("load_all_enumerated");
  }
  fun->be_type = ClassBEType(ClassArray);
  cvn = ClassViewName(ClassOrigClassView(fun->class));
  fun->be_obj = be->FunctionMake2(NameRep(cvn), init_str, TRUE, TRUE);
  fun->be_load_obj = be->FunctionMake2(NameRep(cvn), load_str, TRUE, TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnEnumInitCodeGenBody(BEFN_ENUM_INIT fun, CODEGEN_OBJ res_be_var, */
/*    CODEGEN be)                                                            */
/*                                                                           */
/*  Generate the body of class initialization function fun onto be.          */
/*                                                                           */
/*****************************************************************************/

void BEFnEnumInitCodeGenBody(BEFN_ENUM_INIT fun, CODEGEN_OBJ res_be_var,
  CODEGEN be)
{
  CODEGEN_OBJ all_enum_be_obj;
  ARRAY_BEFN_FEATURE all_predefs;  BEFN_FEATURE predef, j_predef;
  int i, j, code1, code2, j_code1, j_code2;
  if( DEBUG1 )
    fprintf(stderr, "[ BEFnEnumInitCodeGenBody(%s, be)\n",
      be->VarShow(fun->be_obj));
  assert(fun->be_obj != NULL);
  all_predefs = ClassPredefs(fun->class);
  assert(ClassIsEnum(fun->class));

  /* declare i, code, sub_trie, and val variables as needed */
  be->VarDeclare(NPBack_I, ClassBEType(ClassInt));
  /* be->VarDeclare(NPBack_Code, ClassBEType(ClassInt)); */

  /* create class_all_enumerated array */
  all_enum_be_obj = BEFnBEObj((BEFN) ClassAllEnumeratedBEFnFeature(fun->class));
  VarAssign(all_enum_be_obj,
    Call1(NPBack_Array_Create_Fn1, Int(ArraySize(all_predefs))));

  /* initialize the array elements */
  i = 0;
  while( i < ArraySize(all_predefs) )
  {
    /* get the current predef */
    predef = ArrayGet(all_predefs, i);
    if( !BEFnFeatureHasCodeNumbers(predef, &code1, &code2) )
      assert(FALSE);

    /* find the interval [i, j-1] of uniformly increasing indexes and codes */
    j = i + 1;
    if( code1 == code2 )
    {
      while( j < ArraySize(all_predefs) )
      {
	j_predef = ArrayGet(all_predefs, j);
	if( !BEFnFeatureHasCodeNumbers(j_predef, &j_code1, &j_code2) )
	  assert(FALSE);
	if( j_code1 != j_code2 || (j_code1 - j) != (code1 - i) )
	  break;
	j++;
      }
    }

    /* either insert a loop if found at least 3, or just assign one */
    if( j >= i + 3 )
    {
      Stmt(Call4(NPBack_EnumSetSeq, Var(all_enum_be_obj), Int(i),
	Int(j-1), Int(code1)));
      i = j;
    }
    else
    {
      Stmt(Call4(NPBack_EnumSet,
	Var(all_enum_be_obj), Int(i), Int(code1), Int(code2)));
      i++;
    }
  }

  /* finalize class_all_enumerated array */
  VarAssign(all_enum_be_obj,
    Call1(NPBack_Array_Create_Fn2, Var(all_enum_be_obj)));
  VarAssign(res_be_var, Var(all_enum_be_obj));
  if( DEBUG1 )
    fprintf(stderr, "] BEFnEnumInitCodeGenBody returning\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnEnumInitCodeGenCall(BEFN_ENUM_INIT fun, CODEGEN be)             */
/*                                                                           */
/*  Generate a call on class initialization function fun.                    */
/*                                                                           */
/*****************************************************************************/

void BEFnEnumInitCodeGenCall(BEFN_ENUM_INIT fun, CODEGEN be)
{
  BEFnCallCodeGen((BEFN) fun, NULL, be->voidp_type, be);
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnEnumInitCodeGenLoadFn(BEFN_ENUM_INIT fun, CODEGEN be)           */
/*                                                                           */
/*  Code gen the class all_enumerated load function that is piggybacked      */
/*  onto this init function.                                                 */
/*                                                                           */
/*****************************************************************************/

void BEFnEnumInitCodeGenLoadFn(BEFN_ENUM_INIT fun, CODEGEN be)
{
  CODEGEN_OBJ be_var;
  be_var = BEFnBEObj((BEFN) ClassAllEnumeratedBEFnFeature(fun->class));
  be->FunctionBegin(fun->be_load_obj, be->void_type);
  be->FunctionFormal(fun->be_load_obj, NPBack_Other, ClassBEType(ClassArray));
  be->FunctionContinue(fun->be_load_obj);
  VarAssign(be_var, Var(NPBack_Other));
  be->FunctionEnd(fun->be_load_obj);
}


/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_OBJ BEFnEnumInitCodeGenLoadBEObj(BEFN_ENUM_INIT fun)             */
/*                                                                           */
/*  Return the load function piggybacked onto initialization function fun.   */
/*                                                                           */
/*****************************************************************************/

CODEGEN_OBJ BEFnEnumInitCodeGenLoadBEObj(BEFN_ENUM_INIT fun)
{
  return fun->be_load_obj;
}
