/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         feature.c                                                  */
/*  DESCRIPTION:  A feature, including all its views (instances).            */
/*                                                                           */
/*****************************************************************************/
#include "externs.h"
#define DEBUG1 0
#define DEBUG2 0
#define DEBUG3 0
#define DEBUG4 0
#define DEBUG5 1


/*****************************************************************************/
/*                                                                           */
/*  BEFN_FEATURE_INSTANCE                                                    */
/*                                                                           */
/*  An instance of a feature.  It may be inherited into several classes      */
/*  but nevertheless it represents a single limb of the feature.             */
/*                                                                           */
/*****************************************************************************/

struct befn_feature_instance_rec {
  FILE_POS		file_pos;	/* file position where defined       */
  ARRAY_CLASS		classes;	/* the classes holding this instance */
  ARRAY_BEFN_PRECOND	preconditions;	/* preconditions of this body        */
  EXPR			body;		/* body of common feature view       */
};


/*****************************************************************************/
/*                                                                           */
/*  BEFN_FEATURE                                                             */
/*                                                                           */
/*  A BEFN_FEATURE is a feature of a class, including all its instances.     */
/*                                                                           */
/*****************************************************************************/

struct befn_feature_rec {		/* inherits from BEFN                */
  KIND_TAG		kind_tag;	/* kind of entity                    */
  CODEGEN_TYPE		be_type;	/* backend version of result type    */
  CODEGEN_OBJ		be_obj;		/* corresponding backend function    */
  ARRAY_BEFN		inner_functs;	/* inner functs after manifesting    */
  ARRAY_BEFN_PARAM	parameters;	/* ordinary params (NULL if none)    */
  CODEGEN_OBJ		inline_be_obj;	/* non-NULL if to be inlined         */
  BOOLEAN		utilized;	/* TRUE when has been called         */
  BOOLEAN		cached;		/* TRUE when calls are to be cached  */

  CLASS			base_class;	/* class where feature originates    */
  FEFN_FEATURE		base_fv;	/* base feature view                 */
  BEFN_FEATURE		parent;		/* clone only, forward to parent     */
  FEATURE_TYPE		ftype;		/* type (creation, predef, etc.)     */
  ARRAY_BEFN_FEATURE_INSTANCE instances; /* feature instances                */
  int			visit_num;	/* visit number for cycle stopping   */

  /* used by creation features only */
  BOOLEAN		layout_done;	/* TRUE when record layout done      */
  int			layout_offset;	/* offset in record if variable      */
  CODEGEN_OBJ		layout_be_obj;	/* field name                        */
  BEFN_CREDFT		befn_credft;	/* default value function            */

  /* used by predefined object features only */
  int			predef_index;	/* index in original predefs array   */
  int			code1;		/* start of code number range        */
  int			code2;		/* end of code number range          */
};


/*****************************************************************************/
/*                                                                           */
/*  ASTRING FeatureTypeShow(FEATURE_TYPE ftype)                              */
/*                                                                           */
/*  Show feature type ftype.                                                 */
/*                                                                           */
/*****************************************************************************/

ASTRING FeatureTypeShow(FEATURE_TYPE ftype)
{
  switch( ftype )
  {
    case FEATURE_CREATION:		return "creation";
    case FEATURE_NONCREATION:		return "noncreation";
    case FEATURE_NONCREATION_CLONE:	return "clone";
    case FEATURE_PREDEF:		return "predefined";
    case FEATURE_PREDEF_ALL_PREDEFINED:	return "all_predefined";
    case FEATURE_PREDEF_ALL_ENUMERATED:	return "all_enumerated";
    case FEATURE_PREDEF_ALL_ENUM_TRIE:	return "all_enum_trie";
    case FEATURE_PREDEF_LEGAL_CODE:	return "legal_code";
    case FEATURE_PREDEF_WITH_CODE:	return "with_code";

    default:

      assert(FALSE);
      return ""; /* keep compiler happy */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  FEATURE_TYPE BEFnFeatureFType(BEFN_FEATURE f)                            */
/*                                                                           */
/*  Return the feature type (creation, predef, etc.) of f.                   */
/*                                                                           */
/*****************************************************************************/

FEATURE_TYPE BEFnFeatureFType(BEFN_FEATURE f)
{
  return f->ftype;
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnResetBaseFEFn(BEFN_FEATURE f, FEFN_FEATURE base_fv)             */
/*                                                                           */
/*  Retrospectively alter the base feature view of f.  This is used with     */
/*  clone features because the first back-end clone feature made for any     */
/*  set of classes is not necessarily as high as possible in inheritance.    */
/*                                                                           */
/*****************************************************************************/

void BEFnResetBaseFEFn(BEFN_FEATURE f, FEFN_FEATURE base_fv)
{
  f->base_fv = base_fv;
}


/*****************************************************************************/
/*                                                                           */
/*  FILE_POS BEFnFeatureOrigFilePos(BEFN_FEATURE f)                          */
/*                                                                           */
/*  Return the original file position of f.                                  */
/*                                                                           */
/*****************************************************************************/

FILE_POS BEFnFeatureOrigFilePos(BEFN_FEATURE f)
{
  return FEFnFeatureFilePos(f->base_fv);
}


/*****************************************************************************/
/*                                                                           */
/*  NAME BEFnFeatureOrigName(BEFN_FEATURE f)                                 */
/*                                                                           */
/*  Return the original name of f.                                           */
/*                                                                           */
/*****************************************************************************/

NAME BEFnFeatureOrigName(BEFN_FEATURE f)
{
  return FEFnFeatureName(f->base_fv);
}


/*****************************************************************************/
/*                                                                           */
/*  BEFN_FEATURE BEFnFeatureMake(CLASS base_class, FEFN_FEATURE base_fv,     */
/*    FEATURE_TYPE ftype, int code1, int code2)                              */
/*                                                                           */
/*  Make a new feature with these attributes, and initially no parameters.   */
/*                                                                           */
/*****************************************************************************/

BEFN_FEATURE BEFnFeatureMake(CLASS base_class, FEFN_FEATURE base_fv,
  FEATURE_TYPE ftype, int code1, int code2)
{
  BEFN_FEATURE res;
  GetMemory(res, BEFN_FEATURE);
  res->kind_tag = KIND_BEFN_FEATURE;
  res->be_type = NULL;
  res->be_obj = NULL;
  res->inner_functs = NULL;
  res->parameters = NULL;
  res->inline_be_obj = NULL;
  res->utilized = FALSE;
  res->cached = (ftype == FEATURE_NONCREATION);

  res->base_class = base_class;
  res->base_fv = base_fv;
  res->parent = NULL;
  res->ftype = ftype;
  res->visit_num = 0;

  ArrayInit(&res->instances);
  res->layout_done = FALSE;  /* codegen fields initialized properly later */
  res->layout_be_obj = NULL;
  res->befn_credft = NULL;
  res->predef_index = -1;
  if( code1 != NO_CODE_NUM )
    assert(code1 <= code2);
  res->code1 = code1;
  res->code2 = code2;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  BEFN_PARAM BEFnFeatureSelfParam(BEFN_FEATURE f)                          */
/*                                                                           */
/*  Return the "self" parameter of f, which must exist.                      */
/*                                                                           */
/*****************************************************************************/

BEFN_PARAM BEFnFeatureSelfParam(BEFN_FEATURE f)
{
  BEFN_PARAM p;
  ArrayForEachReverse(f->parameters, p)
    if( BEFnParamKind(p) == PARAM_SELF )
      return p;
  assert(FALSE);
  return NULL;  /* keep compiler happy */
}


/*****************************************************************************/
/*                                                                           */
/*  BEFN_FEATURE BEFnFeatureNewAnon(FILE_POS file_pos, CLASS c, EXPR body)   */
/*                                                                           */
/*  Make a new anonymous feature, and register an instance with it with      */
/*  these attributes.                                                        */
/*                                                                           */
/*  An anonymous feature is one with no corresponding feature view, hence    */
/*  no name in Nonpareil.  In fact, the only callers of this function are    */
/*  string literals.                                                         */
/*                                                                           */
/*****************************************************************************/

BEFN_FEATURE BEFnFeatureNewAnon(FILE_POS file_pos, CLASS c, EXPR body)
{
  BEFN_FEATURE res;
  res = BEFnFeatureMake(c, NULL, FEATURE_PREDEF, NO_CODE_NUM, NO_CODE_NUM);
  BEFnFeatureRegisterNewInstance(res, c, file_pos, body);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  BEFN_FEATURE BEFnFeatureNewEnumTrie(CLASS c)                             */
/*                                                                           */
/*  Return a new enum_trie feature for enumerated class c.                   */
/*                                                                           */
/*****************************************************************************/

BEFN_FEATURE BEFnFeatureNewEnumTrie(CLASS c)
{
  BEFN_FEATURE res;
  assert(ClassIsEnum(c));
  res = BEFnFeatureMake(c, NULL, FEATURE_PREDEF_ALL_ENUM_TRIE,
    NO_CODE_NUM, NO_CODE_NUM);
  BEFnFeatureRegisterNewInstance(res, c, ClassFilePos(c), NULL);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnFeatureSetPredefIndex(BEFN_FEATURE f, int index)                */
/*                                                                           */
/*  Set the predef index of f to index.                                      */
/*                                                                           */
/*****************************************************************************/

void BEFnFeatureSetPredefIndex(BEFN_FEATURE f, int index)
{
  assert(f->ftype == FEATURE_PREDEF);
  assert(f->predef_index == -1);
  f->predef_index = index;
}


/*****************************************************************************/
/*                                                                           */
/*  BEFN_FEATURE BEFnFeatureRoot(BEFN_FEATURE f)                             */
/*                                                                           */
/*  Return the root of feature f.  For any feature other than clone this     */
/*  is f itself; for clone, it is the feature that replaces f after          */
/*  feature merges.                                                          */
/*                                                                           */
/*  This code is the Find operation of the Union-Find data structure.        */
/*                                                                           */
/*****************************************************************************/

BEFN_FEATURE BEFnFeatureRoot(BEFN_FEATURE f)
{
  while( f->parent != NULL )
    f = f->parent;
  return f;
}


/*****************************************************************************/
/*                                                                           */
/*  BEFN_FEATURE_INSTANCE BEFnFeatureRegisterNewInstance(                    */
/*    BEFN_FEATURE feature, CLASS class, FILE_POS file_pos, EXPR body)       */
/*                                                                           */
/*  Register the discovery of a new instance of feature.                     */
/*                                                                           */
/*****************************************************************************/

BEFN_FEATURE_INSTANCE BEFnFeatureRegisterNewInstance(BEFN_FEATURE feature,
  CLASS class, FILE_POS file_pos, EXPR body)
{
  BEFN_FEATURE_INSTANCE res;
  GetMemory(res, BEFN_FEATURE_INSTANCE);
  res->file_pos = file_pos;
  ArrayInit(&res->classes);
  ArrayAddLast(res->classes, class);
  res->preconditions = NULL;
  res->body = body;
  ArrayAddLast(feature->instances, res);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  FILE_POS BEFnFeatureInstanceFilePos(BEFN_FEATURE_INSTANCE fi)            */
/*                                                                           */
/*  Return the file position of feature instance fi.                         */
/*                                                                           */
/*****************************************************************************/

FILE_POS BEFnFeatureInstanceFilePos(BEFN_FEATURE_INSTANCE fi)
{
  return fi->file_pos;
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnFeatureRegisterExistingInstance(BEFN_FEATURE_INSTANCE fi,       */
/*    CLASS class)                                                           */
/*                                                                           */
/*  Register the fact that existing feature instance fi has just been        */
/*  discovered to lie in class as well as any other classes it's in.         */
/*                                                                           */
/*****************************************************************************/

void BEFnFeatureRegisterExistingInstance(BEFN_FEATURE_INSTANCE fi, CLASS class)
{
  ArrayAddLast(fi->classes, class);
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnFeatureRegisterPrecondition(BEFN_FEATURE_INSTANCE fi,           */
/*    BEFN_PRECOND precond)                                                  */
/*                                                                           */
/*  Register precond as a precondition of fi.                                */
/*                                                                           */
/*****************************************************************************/

void BEFnFeatureRegisterPrecondition(BEFN_FEATURE_INSTANCE fi,
  BEFN_PRECOND precond)
{
  assert(BEFnKindTag((BEFN) precond) == KIND_BEFN_PRECOND);
  if( fi->preconditions == NULL )
    ArrayInit(&fi->preconditions);
  ArrayAddLast(fi->preconditions, precond);
}


/*****************************************************************************/
/*                                                                           */
/*  EXPR BEFnFeatureSoleBody(BEFN_FEATURE f)                                 */
/*                                                                           */
/*  Return the body of feature f, which must have just one instance.         */
/*                                                                           */
/*****************************************************************************/

EXPR BEFnFeatureSoleBody(BEFN_FEATURE f)
{
  BEFN_FEATURE_INSTANCE fi;
  assert(ArraySize(f->instances) == 1);
  fi = ArrayFirst(f->instances);
  return fi->body;
}


/*****************************************************************************/
/*                                                                           */
/*  CLASS BEFnFeatureSoleClass(BEFN_FEATURE f)                               */
/*                                                                           */
/*  Return the sole class that f lies in.                                    */
/*                                                                           */
/*****************************************************************************/

CLASS BEFnFeatureSoleClass(BEFN_FEATURE f)
{
  BEFN_FEATURE_INSTANCE fi;
  assert(ArraySize(f->instances) == 1);
  fi = ArrayFirst(f->instances);
  assert(ArraySize(fi->classes) == 1);
  return ArrayFirst(fi->classes);
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnFeatureMerge(BEFN_FEATURE target, BEFN_FEATURE source)          */
/*                                                                           */
/*  Merge source into target, as required for clone features.                */
/*                                                                           */
/*  Implementation note.  Each of these features has a self parameter,       */
/*  and source's is discarded during this merge.  This might seem to be      */
/*  dangerous, since references to this parameter will find that it has      */
/*  not been assigned a backend name.  However, the code that generates      */
/*  calls on creation features does not access a self parameter, it          */
/*  simply generates NPBack_Self.  So in clone functions this parameter      */
/*  will never be accessed.                                                  */
/*                                                                           */
/*****************************************************************************/

void BEFnFeatureMerge(BEFN_FEATURE target, BEFN_FEATURE source)
{
  BEFN_PARAM param;
  assert(source->ftype == FEATURE_NONCREATION_CLONE);
  assert(target->ftype == FEATURE_NONCREATION_CLONE);
  assert(source != target);
  assert(!source->utilized);
  assert(!target->utilized);
  assert(source->parameters != NULL);
  assert(target->parameters != NULL);

  if( DEBUG5 )
    fprintf(stderr, "  BEFnFeatureMerge(%s(%s), %s(%s))\n",
      NameShow(FEFnFeatureName(target->base_fv)),
      BEFnParamListShow(target->parameters),
      NameShow(FEFnFeatureName(source->base_fv)),
      BEFnParamListShow(source->parameters));

  /* merge the parameters, all except source's self parameter */
  param = ArrayLast(source->parameters);
  assert(BEFnParamKind(param) == PARAM_SELF);
  ArrayDropLast(source->parameters);
  ArrayAppend(target->parameters, source->parameters);

  /* merge the feature instances */
  ArrayAppend(target->instances, source->instances);

  /* now source must forward on to target */
  source->parent = target;
  if( DEBUG5 )
    fprintf(stderr, "  BEFnFeatureMerge, target now %s(%s)\n",
      NameShow(FEFnFeatureName(target->base_fv)),
      BEFnParamListShow(target->parameters));
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN BEFnFeatureHasCodeNumbers(BEFN_FEATURE f, int *code1, int *code2)*/
/*                                                                           */
/*  If f currently has a code number range, return TRUE and set *code1       */
/*  and *code2 to that range.  Otherwise return FALSE.                       */
/*                                                                           */
/*****************************************************************************/

BOOLEAN BEFnFeatureHasCodeNumbers(BEFN_FEATURE f, int *code1, int *code2)
{
  if( f->code1 != NO_CODE_NUM )
  {
    *code1 = f->code1;
    *code2 = f->code2;
    return TRUE;
  }
  else
    return FALSE;
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnFeatureSetCodeNumbers(BEFN_FEATURE f, int code1, int code2)     */
/*                                                                           */
/*  Assign these code numbers to f.  The feature must not already have       */
/*  code numbers.                                                            */
/*                                                                           */
/*****************************************************************************/

void BEFnFeatureSetCodeNumbers(BEFN_FEATURE f, int code1, int code2)
{
  assert(f->code1 == NO_CODE_NUM);
  assert(code1 <= code2);
  f->code1 = code1;
  f->code2 = code2;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN BEFnFeatureListSortByCode(ARRAY_BEFN_FEATURE features,           */
/*    BOOLEAN *has_range)                                                    */
/*                                                                           */
/*  Sort features by code number, and also set *has_range to TRUE if any     */
/*  of the features has a non-trivial code number range.                     */
/*                                                                           */
/*****************************************************************************/

static int BEFnFeatureCodeCmp(const void *t1, const void *t2)
{
  BEFN_FEATURE f1 = * (BEFN_FEATURE *) t1;
  BEFN_FEATURE f2 = * (BEFN_FEATURE *) t2;
  assert(f1->code1 != NO_CODE_NUM);
  assert(f2->code1 != NO_CODE_NUM);
  return f1->code1 - f2->code1;
}


BOOLEAN BEFnFeatureListSortByCode(ARRAY_BEFN_FEATURE features,
  BOOLEAN *has_range)
{
  int i, prev_code1, prev_code2, code1, code2;  BEFN_FEATURE f, prev_f;
  BOOLEAN success = TRUE;

  /* do the sort */
  ArraySort(features, &BEFnFeatureCodeCmp);

  /* check for no code number clashes */
  if( ArraySize(features) > 1 )
  {
    prev_f = ArrayFirst(features);
    if( !BEFnFeatureHasCodeNumbers(prev_f, &prev_code1, &prev_code2) )
      assert(FALSE);
    if( prev_code1 < prev_code2 )
      *has_range = TRUE;
    for( i = 1;  i < ArraySize(features);  i++ )
    {
      f = ArrayGet(features, i);
      if( !BEFnFeatureHasCodeNumbers(f, &code1, &code2) )
	assert(FALSE);
      if( code1 < code2 )
	*has_range = TRUE;
      if( code1 <= prev_code2 )
      {
	fprintf(stderr, "%s: enum code number clash (other feature is at %s)\n",
	  FilePosShow(BEFnFeatureOrigFilePos(f)),
	  FilePosShow(BEFnFeatureOrigFilePos(prev_f)));
	success = FALSE;
      }
      prev_f = f;
      prev_code1 = code1;
      prev_code2 = code2;
    }
  }
  return success;
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnFeatureSetCreationDefaultValue(BEFN_FEATURE f,                  */
/*    BEFN_CREDFT befn_credft)                                               */
/*                                                                           */
/*  Set the default value function of creation feature f.                    */
/*                                                                           */
/*****************************************************************************/

void BEFnFeatureSetCreationDefaultValue(BEFN_FEATURE f,
  BEFN_CREDFT befn_credft)
{
  assert(f->ftype == FEATURE_CREATION);
  assert(f->befn_credft == NULL);
  f->befn_credft = befn_credft;
}


/*****************************************************************************/
/*                                                                           */
/*  BEFN_CREDFT BEFnFeatureCreationDefaultValue(BEFN_FEATURE f)              */
/*                                                                           */
/*  Return the default value function of creation feature f, or NULL if      */
/*  none.                                                                    */
/*                                                                           */
/*****************************************************************************/

BEFN_CREDFT BEFnFeatureCreationDefaultValue(BEFN_FEATURE f)
{
  assert(f->ftype == FEATURE_CREATION);
  return f->befn_credft;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "predefined features".                                         */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN BEFnFeatureInitOrder(BEFN_FEATURE f, int visit_num,              */
/*    BOOLEAN *report, BEFN_SYSTEM_INIT fun)                                 */
/*                                                                           */
/*  Carry out the specification of BEFnInitOrder on back-end feature f.      */
/*                                                                           */
/*****************************************************************************/

BOOLEAN BEFnFeatureInitOrder(BEFN_FEATURE f, int visit_num, BOOLEAN *report,
  BEFN_SYSTEM_INIT fun)
{
  BEFN_FEATURE_INSTANCE fi;
  BEFN befn;  int i;
  assert(*report == FALSE);
  if( DEBUG1 )
    fprintf(stderr, "[ BEFnFeatureInitOrder(%s, %d, report, fun)\n",
      NameShow(BEFnFeatureOrigName(f)), visit_num);

  /* if already visited f, don't need to revisit it */
  if( f->visit_num >= visit_num )
    return TRUE;
  f->visit_num = visit_num;

  /* visit f, handling it depending on its type */
  switch( f->ftype )
  {
    case FEATURE_CREATION:

      /* depends on class initialization function if enum, else nothing */
      if( ClassIsEnum(f->base_class) )
      {
	if( !BEFnClassInitInitOrder(ClassBEFnClassInit(f->base_class),
	    visit_num, report, fun, INT_MAX) )
	  db_return(DEBUG1, "BEFnFeatureInitOrder (creation)", FALSE);
      }
      break;


    case FEATURE_NONCREATION:

      /* depends on whatever is called in its bodies */
      for( i = 0;  i < ArraySize(f->instances);  i++ )
      {
	fi = ArrayGet(f->instances, i);
	if( fi->body != NULL && !ExprInitOrder(fi->body,visit_num,report,fun) )
	  db_return(DEBUG1, "BEFnFeatureInitOrder (noncreation)", FALSE);
      }
      break;


    case FEATURE_NONCREATION_CLONE:

      /* depends on creation function */
      befn = (BEFN) ClassBEFnCreation(f->base_class);
      if( !BEFnInitOrder(befn, visit_num, report, fun) )
	db_return(DEBUG1, "BEFnFeatureInitOrder (clone)", FALSE);
      break;


    case FEATURE_PREDEF:

      /* depends on nothing in enum classes, and on class init otherwise */
      if( !ClassIsEnum(f->base_class) )
      {
	if( !BEFnClassInitInitOrder(ClassBEFnClassInit(f->base_class),
	      visit_num, report, fun, f->predef_index) )
	  db_return(DEBUG1, "BEFnFeatureInitOrder (predef)", FALSE);
      }
      break;


    case FEATURE_PREDEF_ALL_PREDEFINED:
    case FEATURE_PREDEF_ALL_ENUM_TRIE:
    case FEATURE_PREDEF_LEGAL_CODE:
    case FEATURE_PREDEF_WITH_CODE:

      /* depends on class initialization function */
      if( !BEFnClassInitInitOrder(ClassBEFnClassInit(f->base_class),
	    visit_num, report, fun, INT_MAX) )
        db_return(DEBUG1, "BEFnFeatureInitOrder (all_predef etc.)", FALSE);
      break;


    case FEATURE_PREDEF_ALL_ENUMERATED:

      /* all_enumerated features are initialized first and depend on nothing */
      break;


    default:

      assert(FALSE);
  }
  db_return(DEBUG1, "BEFnFeatureInitOrder", TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "code generation".                                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  TYPE_POINTER_STATUS BEFnFeatureSwizzleStatus(BEFN_FEATURE f,             */
/*    int *swizzle_index)                                                    */
/*                                                                           */
/*  Return an indication of whether this feature needs swizzling or not,     */
/*  and if the answer is "maybe", set the swizzle index too.                 */
/*                                                                           */
/*****************************************************************************/

TYPE_POINTER_STATUS BEFnFeatureSwizzleStatus(BEFN_FEATURE f, int *swizzle_index)
{
  return TypeIsPointer(FEFnResultType((FEFN) f->base_fv), swizzle_index);
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnFeatureCodeGenInitCreation(BEFN_FEATURE f, ISET layout_set,     */
/*    CODEGEN be)                                                            */
/*                                                                           */
/*  Do the backend initialization of creation feature f.  This is            */
/*  principally about assigning it an offset in objects that does not        */
/*  conflict with layout_set, but we also assign its backend type and        */
/*  struct name here.                                                        */
/*                                                                           */
/*  When registering with the structure scope, allow two creation features   */
/*  to have the same C name provided they have the same offset, since they   */
/*  can never appear in the same record in that case.  This allows some      */
/*  common names to avoid being swizzled, making some code easier to read.   */
/*                                                                           */
/*****************************************************************************/

void BEFnFeatureCodeGenInitCreation(BEFN_FEATURE f, ISET layout_set, CODEGEN be)
{
  ISET is;  TYPE type;  int field_width;
  assert(f->ftype == FEATURE_CREATION);
  if( !f->layout_done )
  {
    if( DEBUG3 )
      fprintf(stderr, "[ BEFnFeatureRecordLayout(%s, %s)\n",
	NameShow(BEFnFeatureOrigName(f)), ISetShow(layout_set));
    type = FEFnResultType((FEFN) f->base_fv);
    f->be_type = TypeBEType(type, be);
    field_width = be->TypeFieldWidth(f->be_type);
    f->layout_offset = ISetGap(layout_set, field_width, field_width);
    is = ISetInterval(f->layout_offset, f->layout_offset + field_width - 1);
    ISetUnion(layout_set, is);
    f->layout_be_obj =
      be->StructItemMake(NameRep(BEFnFeatureOrigName(f)), f->layout_offset);
    if( !ClassIsEnum(f->base_class) )
      f->inline_be_obj = f->layout_be_obj;
    BEFnFinalize((BEFN) f, be);
    f->layout_done = TRUE;
    if( DEBUG3 )
      fprintf(stderr, "] BEFnFeatureRecordLayout returning, assigned %s\n",
	ISetShow(is));
  }
}


/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_OBJ BEFnFeatureLayoutBEObj(BEFN_FEATURE f)                       */
/*                                                                           */
/*  Return the backend function which lays out creation feature f.           */
/*                                                                           */
/*****************************************************************************/

CODEGEN_OBJ BEFnFeatureLayoutBEObj(BEFN_FEATURE f)
{
  assert(f->layout_be_obj != NULL);
  return f->layout_be_obj;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN FeatureWrapsBuiltin(BEFN_FEATURE f, CODEGEN_OBJ *be_obj)         */
/*                                                                           */
/*  Return TRUE if noncreation feature f merely wraps a builtin function;    */
/*  set *be_obj to the name of that builtin function, in that case.          */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN FeatureWrapsBuiltin(BEFN_FEATURE f, CODEGEN_OBJ *be_obj)
{
  BEFN_FEATURE_INSTANCE fi;

  /* only noncreation features are eligible for this optimization */
  if( f->ftype != FEATURE_NONCREATION )
    return FALSE;

  /* must be single instance */
  if( ArraySize(f->instances) != 1 )
    return FALSE;
  fi = ArrayFirst(f->instances);

  /* check the body */
  return ExprFnHeadWrapsBuiltin(fi->body, f->parameters, be_obj);
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnFeatureFinalize(BEFN_FEATURE f, CODEGEN be)                     */
/*                                                                           */
/*  Finalize f's be_type and be_obj fields.                                  */
/*                                                                           */
/*****************************************************************************/

void BEFnFeatureFinalize(BEFN_FEATURE f, CODEGEN be)
{
  CLASS_VIEW cv;  BEFN_FEATURE all_predefined;  BEFN_FEATURE_INSTANCE fi;
  TYPE result_type;  CODEGEN_OBJ be_obj;
  static USTRING enum_trie_str = NULL;
  switch( f->ftype )
  {
    case FEATURE_CREATION:

      /* will have be_type and layout_be_obj already */
      /* give it name <class_view_name>_<feature_view_name> */
      assert(f->be_type != NULL);
      cv = FEFnFeatureClassView(f->base_fv);
      f->be_obj = be->FunctionMake2(NameRep(ClassViewName(cv)),
	NameRep(BEFnFeatureOrigName(f)), TRUE, TRUE);
      break;


    case FEATURE_NONCREATION:
    case FEATURE_NONCREATION_CLONE:
    case FEATURE_PREDEF_LEGAL_CODE:
    case FEATURE_PREDEF_WITH_CODE:

      /* give it name <class_view_name>_<feature_view_name> */
      f->be_type = TypeBEType(FEFnResultType((FEFN) f->base_fv), be);
      cv = FEFnFeatureClassView(f->base_fv);
      f->be_obj = be->FunctionMake2(NameRep(ClassViewName(cv)),
	NameRep(BEFnFeatureOrigName(f)), TRUE, TRUE);

      /* set up noncreation feature for inlining if wraps a builtin function */
      if( FeatureWrapsBuiltin(f, &be_obj) )
	f->inline_be_obj = be_obj;
      break;


    case FEATURE_PREDEF_ALL_PREDEFINED:
    case FEATURE_PREDEF_ALL_ENUMERATED:

      /* give it name <class_view_name>_<feature_view_name> */
      f->be_type = TypeBEType(FEFnResultType((FEFN) f->base_fv), be);
      cv = FEFnFeatureClassView(f->base_fv);
      f->be_obj = be->FunctionMake2(NameRep(ClassViewName(cv)),
	NameRep(BEFnFeatureOrigName(f)), FALSE, TRUE);
      break;


    case FEATURE_PREDEF_ALL_ENUM_TRIE:

      /* give it name <class_view_name>_enum_trie */
      if( enum_trie_str == NULL )
	enum_trie_str = AStringToUString("enum_trie");
      assert(ArraySize(f->instances) == 1);
      fi = ArrayFirst(f->instances);
      cv = ClassOrigClassView(ArrayFirst(fi->classes));
      f->be_type = ClassBEType(ClassArray);
      f->be_obj = be->FunctionMake2(NameRep(ClassViewName(cv)),
	enum_trie_str, FALSE, TRUE);
      break;


    case FEATURE_PREDEF:

      assert(ArraySize(f->instances) == 1);
      fi = ArrayFirst(f->instances);
      assert(fi->body != NULL);
      result_type = ExprType(fi->body);
      f->be_type = TypeBEType(result_type, be);
      if( !ClassIsEnum(TypeClass(result_type)) )
      {
	/* give it name <class>_all_predefined->elems[<predef_index>] */
	all_predefined = ClassAllPredefinedBEFnFeature(TypeClass(result_type));
	f->be_obj = be->VarIndexedAttributeMake(
	  BEFnBEObj((BEFN) all_predefined), "elems", f->predef_index);
      }
      else if( BEFnFeatureOrigName(f) != NULL )
      {
	/* give it name <class_view_name>_<feature_view_name> */
	cv = FEFnFeatureClassView(f->base_fv);
	f->be_obj = be->ConstMakeInt2(NameRep(ClassViewName(cv)),
	  NameRep(BEFnFeatureOrigName(f)), f->code1, TRUE);
      }
      else
      {
	/* it has no name */
	f->be_obj = NULL;
      }
      break;


    default:

      assert(FALSE);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  int BEFnFeatureOffset(BEFN_FEATURE cf)                                   */
/*                                                                           */
/*  Return the offset in bits of a variable holding f.                       */
/*                                                                           */
/*****************************************************************************/

int BEFnFeatureOffset(BEFN_FEATURE f)
{
  assert(f->layout_done);
  return f->layout_offset;
}


/*****************************************************************************/
/*                                                                           */
/*  int BEFnFeatureFieldWidth(BEFN_FEATURE f, CODEGEN be)                    */
/*                                                                           */
/*  Return the width in bits of a variable holding creation feature f.       */
/*                                                                           */
/*****************************************************************************/

int BEFnFeatureFieldWidth(BEFN_FEATURE f, CODEGEN be)
{
  assert(f->layout_done);
  return be->TypeFieldWidth(f->be_type);
}


/*****************************************************************************/
/*                                                                           */
/*  int BEFnFeatureOffsetCmp(const void *t1, const void *t2)                 */
/*                                                                           */
/*  Comparison function for sorting features by increasing offset.           */
/*                                                                           */
/*****************************************************************************/

int BEFnFeatureOffsetCmp(const void *t1, const void *t2)
{
  BEFN_FEATURE f1 = * (BEFN_FEATURE *) t1;
  BEFN_FEATURE f2 = * (BEFN_FEATURE *) t2;
  assert(f1->kind_tag == KIND_BEFN_FEATURE);
  assert(f1->layout_done);
  assert(f2->kind_tag == KIND_BEFN_FEATURE);
  assert(f2->layout_done);
  return f1->layout_offset - f2->layout_offset;
}


/*****************************************************************************/
/*                                                                           */
/*  void DeleteAbstractBranches(BEFN_FEATURE f)                              */
/*                                                                           */
/*  Feature f is a noncreation feature.  Run through its instances and       */
/*  delete all references to abstract classes.  Then delete all instances    */
/*  that have no classes.                                                    */
/*                                                                           */
/*****************************************************************************/

static void DeleteAbstractBranches(BEFN_FEATURE f)
{
  int i, j;  BEFN_FEATURE_INSTANCE fi;  CLASS c;
  for( i = 0;  i < ArraySize(f->instances);  i++ )
  {
    fi = ArrayGet(f->instances, i);
    if( fi->classes != NULL )
    {
      for( j = 0;  j < ArraySize(fi->classes);  j++ )
      {
	c = ArrayGet(fi->classes, j);
	if( ClassIsAbstract(c) )
	{
	  ArrayRemove(fi->classes, j);
	  j--;
	}
      }
      if( ArraySize(fi->classes) == 0 )
      {
	ArrayRemove(f->instances, i);
	i--;
      }
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void CloneBodyCodeGen(BEFN_FEATURE f, BEFN_FEATURE_INSTANCE fi,          */
/*    CODEGEN_OBJ res_be_var, CODEGEN be)                                    */
/*                                                                           */
/*  Generate the body of the clone function for feature instance fi of       */
/*  clone feature f, which for clone must relate to just one class.  This    */
/*  body is builtin.                                                         */
/*                                                                           */
/*****************************************************************************/

static void CloneBodyCodeGen(BEFN_FEATURE f, BEFN_FEATURE_INSTANCE fi,
  CODEGEN_OBJ res_be_var, CODEGEN be)
{
  CLASS c;  ARRAY_BEFN_FEATURE cfeatures;  BEFN_FEATURE cf;  int i;
  BEFN_PARAM befn_creation_param, p, self_param;
  CODEGEN_OBJ p_be_obj, cf_be_obj, create_be_obj;
  CODEGEN_TYPE p_be_type, class_be_type, self_be_type;
  BEFN_CREATION befn_creation;  ARRAY_BEFN_PARAM befn_creation_params;
  assert(ArraySize(fi->classes) == 1);
  c = ArrayFirst(fi->classes);
  class_be_type = ClassBEType(c);
  self_param = ArrayLast(f->parameters);
  self_be_type = BEFnBEType((BEFN) self_param);

  /* generate default test and replacement for each creation feature */
  cfeatures = ClassCreationFeatures(c);
  ArrayForEach(cfeatures, cf)
  {
    p = BEFnFindParamFromCreationFeature((BEFN) f, cf);
    p_be_type = BEFnBEType((BEFN) p);
    p_be_obj = BEFnBEObj((BEFN) p);
    cf_be_obj = BEFnFeatureLayoutBEObj(cf);
    If( Equal(Var(p_be_obj), Cast(p_be_type, NULL, Var(NPBack_Dft_Param_Val))),
      Indent(VarAssign(p_be_obj, Call1(cf_be_obj,
	Cast(class_be_type, self_be_type, Var(NPBack_Self))))));
  }

  /* generate creation call with one parameter for each creation feature */
  befn_creation = ClassBEFnCreation(c);
  befn_creation_params = BEFnParameters((BEFN) befn_creation);
  create_be_obj = BEFnBEObj((BEFN) befn_creation);
  be->VarAsstBegin(res_be_var);
  be->CastBegin(f->be_type, class_be_type);
  be->CallBegin(create_be_obj);
  for( i = 0;  i < ArraySize(befn_creation_params);  i++ )
  {
    befn_creation_param = ArrayGet(befn_creation_params, i);
    cf = BEFnParamCreationFeature(befn_creation_param);
    p = BEFnFindParamFromCreationFeature((BEFN) f, cf);
    p_be_obj = BEFnBEObj((BEFN) p);
    if( i > 0 )
      be->CallContinue(create_be_obj, i);
    Var(p_be_obj);
  }
  be->CallEnd(create_be_obj);
  be->CastEnd();
  be->AsstEnd();
}


/*****************************************************************************/
/*                                                                           */
/*  void InstanceCodeGen(BEFN_FEATURE f, BEFN_FEATURE_INSTANCE fi,           */
/*    CODEGEN_OBJ res_be_var, CODEGEN be)                                    */
/*                                                                           */
/*  Generate instance fi of feature f onto be.  First any preconditions,     */
/*  then the body.                                                           */
/*                                                                           */
/*****************************************************************************/

void InstanceCodeGen(BEFN_FEATURE f, BEFN_FEATURE_INSTANCE fi,
  CODEGEN_OBJ res_be_var, CODEGEN be)
{
  BEFN_PRECOND befn_precond;

  /* default parameter values */
  ExprFnHeadCodeGenDftParamValues(fi->body, be);

  /* preconditions */
  if( fi->preconditions != NULL )
    ArrayForEach(fi->preconditions, befn_precond)
      Stmt(BEFnPrecondCodeGenCall(befn_precond, f->parameters, be));

  /* body (will not do default parameter values again) */
  ExprCodeGen(fi->body, res_be_var, f->be_type, be);
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnFeatureCodeGenBody(BEFN_FEATURE f, CODEGEN_OBJ res_be_var,      */
/*    CODEGEN be)                                                            */
/*                                                                           */
/*  Generate the body of feature f.  If it is a predefined object feature,   */
/*  the code generated is just a variable, which is initialized elsewhere.   */
/*                                                                           */
/*****************************************************************************/
#define swap(a, b, tmp) (tmp = a, a = b, b = tmp)

void BEFnFeatureCodeGenBody(BEFN_FEATURE f, CODEGEN_OBJ res_be_var, CODEGEN be)
{
  int i;  BEFN_FEATURE_INSTANCE fi, fi1, fi2;
  CLASS c;  CODEGEN_OBJ be_param;
  if( DEBUG4 )
    fprintf(stderr, "[ BEFnFeatureCodeGenBody(%s (%d instances), fp)\n",
      be->VarShow(f->be_obj), ArraySize(f->instances));
  switch( f->ftype )
  {
    case FEATURE_CREATION:

      assert(ArraySize(f->instances) >= 1);
      fi = ArrayFirst(f->instances);
      c = ArrayFirst(fi->classes);
      if( ClassIsEnum(c) )
	VarAssign(res_be_var, Call1(f->layout_be_obj,
	  ClassEnumCodeGenTrieAccess(c, NPBack_Self, be)));
      else
	VarAssign(res_be_var, Call1(f->layout_be_obj, Var(NPBack_Self)));
      break;


    case FEATURE_NONCREATION:

      /* delete instances from abstract classes */
      DeleteAbstractBranches(f);

      /* print body, different versions depending on 0, 1, 2, or more cases */
      if( ArraySize(f->instances) == 0 )
	be->Fail();
      else if( ArraySize(f->instances) == 1 )
      {
	fi = ArrayFirst(f->instances);
	InstanceCodeGen(f, fi, res_be_var, be);
      }
      else if( ArraySize(f->instances) == 2 )
      {
	/* get both fi's, the one with the smallest number of classes first */
	fi1 = ArrayFirst(f->instances);
	fi2 = ArrayLast(f->instances);
	if( ArraySize(fi1->classes) > ArraySize(fi2->classes) )
	  swap(fi1, fi2, fi);

	/* print if statement that selects the first fi */
	be->IfStmtBegin();
	be->CallBegin(be->logical_or_seq);
	for( i = 0;  i < ArraySize(fi1->classes);  i++ )
	{
	  c = ArrayGet(fi1->classes, i);
	  if( i > 0 )
	    be->CallContinue(be->logical_or_seq, i);
	  Equal(Call1(NPBack_Type_Tag, Var(NPBack_Self)), Var(ClassTypeTag(c)));
	}
	be->CallEnd(be->logical_or_seq);
	be->IfStmtContinue();

	/* print the bodies separated by else */
	if( ExprLargeScale(fi1->body) || fi1->preconditions != NULL )
	  Block(InstanceCodeGen(f, fi1, res_be_var, be));
	else
	  Indent(InstanceCodeGen(f, fi1, res_be_var, be));
	be->IfStmtElse();
	if( ExprLargeScale(fi2->body) || fi2->preconditions != NULL )
	  Block(InstanceCodeGen(f, fi2, res_be_var, be));
	else
	  Indent(InstanceCodeGen(f, fi2, res_be_var, be));
	be->IfStmtEnd();
      }
      else
      {
	/* switch statement */
	be->SwitchBegin();
	Call1(NPBack_Type_Tag, Var(NPBack_Self));
	be->SwitchContinue();
	ArrayForEach(f->instances, fi)
	{
	  ArrayForEach(fi->classes, c)
	    be->SwitchCase(ClassTypeTag(c));
	  be->SwitchActionBegin();
	  InstanceCodeGen(f, fi, res_be_var, be);
	  be->SwitchActionEnd(FALSE);
	}
	be->SwitchCase(NULL); /* default case */
	be->SwitchActionBegin();
	be->Fail();
	be->SwitchActionEnd(FALSE);
	be->SwitchEnd();
      }
      break;


    case FEATURE_NONCREATION_CLONE:

      /* delete instances from abstract classes */
      DeleteAbstractBranches(f);

      /* print body, different versions depending on 0, 1, 2, or more cases */
      if( ArraySize(f->instances) == 0 )
	be->Fail();
      else if( ArraySize(f->instances) == 1 )
      {
	fi = ArrayFirst(f->instances);
	CloneBodyCodeGen(f, fi, res_be_var, be);
      }
      else if( ArraySize(f->instances) == 2 )
      {
	/* get both fi's, the one with the smallest number of classes first */
	fi1 = ArrayFirst(f->instances);
	fi2 = ArrayLast(f->instances);
	if( ArraySize(fi1->classes) > ArraySize(fi2->classes) )
	  swap(fi1, fi2, fi);

	/* print if statement that selects the first fi */
	be->IfStmtBegin();
	be->CallBegin(be->logical_or_seq);
	for( i = 0;  i < ArraySize(fi1->classes);  i++ )
	{
	  c = ArrayGet(fi1->classes, i);
	  if( i > 0 )
	    be->CallContinue(be->logical_or_seq, i);
	  Equal(Call1(NPBack_Type_Tag, Var(NPBack_Self)), Var(ClassTypeTag(c)));
	}
	be->CallEnd(be->logical_or_seq);
	be->IfStmtContinue();

	/* print the bodies separated by else */
	Block(CloneBodyCodeGen(f, fi1, res_be_var, be));
	be->IfStmtElse();
	Block(CloneBodyCodeGen(f, fi2, res_be_var, be));
	be->IfStmtEnd();
      }
      else
      {
	/* switch statement */
	be->SwitchBegin();
	Call1(NPBack_Type_Tag, Var(NPBack_Self));
	be->SwitchContinue();
	ArrayForEach(f->instances, fi)
	{
	  ArrayForEach(fi->classes, c)
	    be->SwitchCase(ClassTypeTag(c));
	  be->SwitchActionBegin();
	  Indent(CloneBodyCodeGen(f, fi, res_be_var, be));
	  be->SwitchActionEnd(FALSE);
	}
	be->SwitchCase(NULL); /* default case */
	be->SwitchActionBegin();
	be->Fail();
	be->SwitchActionEnd(FALSE);
	be->SwitchEnd();
      }
      break;


    case FEATURE_PREDEF:
    case FEATURE_PREDEF_ALL_PREDEFINED:
    case FEATURE_PREDEF_ALL_ENUMERATED:
    case FEATURE_PREDEF_ALL_ENUM_TRIE:

      break;


    case FEATURE_PREDEF_LEGAL_CODE:

      assert(ArraySize(f->instances) >= 1);
      fi = ArrayFirst(f->instances);
      c = ArrayFirst(fi->classes);
      assert(ClassIsEnum(c));
      be_param = BEFnBEObj((BEFN) ArrayFirst(f->parameters));
      VarAssign(res_be_var, Call2(be->logical_and,
	Call2(be->le, Var(be_param), Int(ClassEnumMaxCode(c))),
	Call2(be->ne,
	  ClassEnumCodeGenTrieAccess(c, be_param, be), Var(be->null_value))));
      break;


    case FEATURE_PREDEF_WITH_CODE:

      assert(ArraySize(f->instances) >= 1);
      fi = ArrayFirst(f->instances);
      c = ArrayFirst(fi->classes);
      assert(ClassIsEnum(c));
      be_param = BEFnBEObj((BEFN) ArrayFirst(f->parameters));
      Stmt(Call1(be->assert_fn,
	Call1(BEFnBEObj((BEFN) ClassLegalCode(c)), Var(be_param))));
      VarAssign(res_be_var,
	Cast(ClassBEType(c), ClassBEType(ClassInt), Var(be_param)));
      break;


    default:

      assert(FALSE);
  }
  if( DEBUG4 )
    fprintf(stderr, "] BEFnFeatureCodeGenBody returning\n");
}
