/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         befn_invariant.c                                           */
/*  DESCRIPTION:  Back-end invariant functions                               */
/*                                                                           */
/*****************************************************************************/
#include <string.h>
#include "externs.h"
#define DEBUG1 0


/*****************************************************************************/
/*                                                                           */
/*  BEFN_INVARIANT                                                           */
/*                                                                           */
/*****************************************************************************/

struct befn_invariant_rec {		/* inherits from BEFN                */
  KIND_TAG		kind_tag;	/* kind of entity                    */
  CODEGEN_TYPE		be_type;	/* result type                       */
  CODEGEN_OBJ		be_obj;		/* corresponding backend object      */
  ARRAY_BEFN		inner_functs;	/* to generate just before this fn   */
  ARRAY_BEFN_PARAM	parameters;	/* ordinary params (NULL if none)    */
  CODEGEN_OBJ		inline_be_obj;	/* non-NULL if to be inlined         */
  BOOLEAN		utilized;	/* TRUE when has been called         */
  BOOLEAN		cached;		/* TRUE when calls are to be cached  */

  CLASS			class;		/* the class holding this invt       */
  ARRAY_EXPR		exprs;		/* the actual expressions            */
  int			visit_num;	/* when finding initialization order */
};


/*****************************************************************************/
/*                                                                           */
/*  BEFN_INVARIANT BEFnInvtNew(CLASS c, FEFN_PARAM orig_self_param)          */
/*                                                                           */
/*  Return a new, empty invariant for class c, with a self parameter         */
/*  sourced to orig_self_param.                                              */
/*                                                                           */
/*****************************************************************************/

BEFN_INVARIANT BEFnInvtNew(CLASS c, FEFN_PARAM orig_self_param)
{
  BEFN_INVARIANT res;  BEFN_PARAM param;
  GetMemory(res, BEFN_INVARIANT);
  res->kind_tag = KIND_BEFN_INVARIANT;
  res->be_type = NULL;
  res->be_obj = NULL;
  ArrayInit(&res->parameters);
  param = BEFnParamNew(PARAM_COMPULSORY, orig_self_param, NULL);
  BEFnAddParameter((BEFN) res, param);
  res->inline_be_obj = NULL;
  res->utilized = FALSE;
  res->cached = FALSE;
  res->class = c;
  ArrayInit(&res->exprs);
  res->visit_num = 0;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnInvtAddExpr(BEFN_INVARIANT invt, EXPR e)                        */
/*                                                                           */
/*  Add e as an invariant to invt.                                           */
/*                                                                           */
/*****************************************************************************/

void BEFnInvtAddExpr(BEFN_INVARIANT invt, EXPR e)
{
  ArrayAddLast(invt->exprs, e);
}


/*****************************************************************************/
/*                                                                           */
/*  BEFN_PARAM BEFnInvtSelfParam(BEFN_INVARIANT invt)                        */
/*                                                                           */
/*  Return the "self" parameter of invt.                                     */
/*                                                                           */
/*****************************************************************************/

BEFN_PARAM BEFnInvtSelfParam(BEFN_INVARIANT invt)
{
  return ArrayFirst(invt->parameters);
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnInvtFinalize(BEFN_INVARIANT invt, CODEGEN be)                   */
/*                                                                           */
/*  Assign a be_type and be_obj to invt.                                     */
/*                                                                           */
/*****************************************************************************/

void BEFnInvtFinalize(BEFN_INVARIANT invt, CODEGEN be)
{
  invt->be_type = be->void_type;
  invt->be_obj =
    be->FunctionMake2(NameRep(ClassViewName(ClassOrigClassView(invt->class))),
    AStringToUString("invt"), TRUE, TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN BEFnInvtInitOrder(BEFN_INVARIANT befn, int visit_num,            */
/*    BOOLEAN *report, BEFN_SYSTEM_INIT fun)                                 */
/*                                                                           */
/*  Carry out the specification of BEFnInitOrder on invariant befn.          */
/*                                                                           */
/*****************************************************************************/

BOOLEAN BEFnInvtInitOrder(BEFN_INVARIANT befn, int visit_num,
  BOOLEAN *report, BEFN_SYSTEM_INIT fun)
{
  EXPR e;

  /* check whether visited already and update if not */
  if( befn->visit_num >= visit_num )
    return TRUE;
  befn->visit_num = visit_num;

  /* explore the expressions */
  ArrayForEach(befn->exprs, e)
    if( !ExprInitOrder(e, visit_num, report, fun) )
      return FALSE;
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnInvtCodeGenBody(BEFN_INVARIANT invt, CODEGEN_OBJ res_be_var,    */
/*    CODEGEN be)                                                            */
/*                                                                           */
/*  Generate the code for the body of invariant function invt onto be.       */
/*                                                                           */
/*  Parameter res_be_var is not used here, because invariants have           */
/*  void result type and there is not return variable.                       */
/*                                                                           */
/*****************************************************************************/

void BEFnInvtCodeGenBody(BEFN_INVARIANT invt, CODEGEN_OBJ res_be_var,
  CODEGEN be)
{
  /* EXPR combined_invt; */
  EXPR e;
  if( DEBUG1 )
    fprintf(stderr, "[ BEFnInvtCodeGenBody(invt, res_be_var, be)\n");

  ArrayForEach(invt->exprs, e)
    Stmt(ExprCodeGen(e, NULL, be->void_type, be));

  if( DEBUG1 )
    fprintf(stderr, "] BEFnInvtCodeGenBody returning\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnInvtCallCodeGen(BEFN_INVARIANT invt, CODEGEN_TYPE self_be_type, */
/*    CODEGEN be)                                                            */
/*                                                                           */
/*  Generate a call on invt in environment ee.  The type of self is          */
/*  self_be_type, which may require a cast.                                  */
/*                                                                           */
/*****************************************************************************/

void BEFnInvtCallCodeGen(BEFN_INVARIANT invt, CODEGEN_TYPE self_be_type,
  CODEGEN be)
{
  BEFN_PARAM param = ArrayFirst(invt->parameters);
  Stmt(Call1(invt->be_obj,
    Cast(BEFnParamBEType(param), self_be_type, Var(NPBack_Self))));
}
