/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         befn_letdef.c                                              */
/*  DESCRIPTION:  Back-end let definitions                                   */
/*                                                                           */
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "externs.h"
#define	DEBUG1 0
#define	DEBUG2 0
#define	DEBUG3 0

/*****************************************************************************/
/*                                                                           */
/*  BEFN_LETDEF                                                              */
/*                                                                           */
/*  The C function corresponding to a letdef.                                */
/*                                                                           */
/*****************************************************************************/

struct befn_letdef_rec {		/* inherits from BEFN                */
  KIND_TAG		kind_tag;	/* kind of entity                    */
  CODEGEN_TYPE		be_type;	/* result type                       */
  CODEGEN_OBJ		be_obj;		/* corresponding backend object      */
  ARRAY_BEFN		inner_functs;	/* to generate just before this fn   */
  ARRAY_BEFN_PARAM	parameters;	/* ordinary params (NULL if none)    */
  CODEGEN_OBJ		inline_be_obj;	/* non-NULL if to be inlined         */
  BOOLEAN		utilized;	/* TRUE when has been called         */
  BOOLEAN		cached;		/* TRUE when calls are to be cached  */

  FEFN_LETDEF		letdef;		/* the letdef that generated this    */
  EXPR			body;		/* the body of the letdef            */
  int			visit_num;	/* when finding initialization order */
};


/*****************************************************************************/
/*                                                                           */
/*  BEFN_LETDEF BEFnLetDefNew(FEFN_LETDEF letdef, EXPR body)                 */
/*                                                                           */
/*  Make a new letdef C function with no parameters for the given letdef.    */
/*                                                                           */
/*****************************************************************************/

BEFN_LETDEF BEFnLetDefNew(FEFN_LETDEF letdef, EXPR body)
{
  BEFN_LETDEF res;
  GetMemory(res, BEFN_LETDEF);
  res->kind_tag = KIND_BEFN_LETDEF;
  res->be_type = NULL;
  res->be_obj = NULL;
  res->inner_functs = NULL;
  res->parameters = NULL;
  res->inline_be_obj = NULL;
  res->utilized = FALSE;
  res->cached = TRUE;
  res->letdef = letdef;
  res->body = body;
  res->visit_num = 0;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnLetDefFinalize(BEFN_LETDEF fun, CODEGEN be)                     */
/*                                                                           */
/*  Finalize the be_type and be_obj fields of fun, and also the types        */
/*  of any hidden parameters.                                                */
/*                                                                           */
/*****************************************************************************/

void BEFnLetDefFinalize(BEFN_LETDEF fun, CODEGEN be)
{
  fun->be_type = TypeBEType(FEFnResultType((FEFN) fun->letdef), be);
  if( fun->parameters == NULL )
  {
    /* no parameters, so it's a local variable */
    fun->be_obj = be->VarMakeAndDeclare(NULL,
      NameRep(FEFnName((FEFN) fun->letdef)), fun->be_type);
  }
  else
  {
    /* parameters, so it's a static function */
    fun->be_obj =
      be->FunctionMake2(NameRep(FEFnName((FEFN) fun->letdef)),NULL,TRUE,FALSE);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnLetDefUnFinalize(BEFN_LETDEF fun, CODEGEN be)                   */
/*                                                                           */
/*  Undo the effect of BEFnLetDefFinalize, including undeclaring and         */
/*  freeing the backend variable.                                            */
/*                                                                           */
/*****************************************************************************/

void BEFnLetDefUnFinalize(BEFN_LETDEF fun, CODEGEN be)
{
  assert(fun->parameters == NULL);
  be->VarUnDeclare(fun->be_obj);
  be->ObjFree(fun->be_obj);
  fun->be_type = NULL;
  fun->be_obj = NULL;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN BEFnLetDefInitOrder(BEFN_LETDEF befn, int visit_num,             */
/*    BOOLEAN *report, BEFN_SYSTEM_INIT fun)                                 */
/*                                                                           */
/*  Carry out the specification of BEFnInitOrder on let definition befn.     */
/*                                                                           */
/*****************************************************************************/

BOOLEAN BEFnLetDefInitOrder(BEFN_LETDEF befn, int visit_num,
  BOOLEAN *report, BEFN_SYSTEM_INIT fun)
{
  /* check whether visited already and update if not */
  if( befn->visit_num >= visit_num )
    return TRUE;
  befn->visit_num = visit_num;

  /* check initialization order of body */
  return ExprInitOrder(befn->body, visit_num, report, fun);
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnLetDefCodeGenBody(BEFN_LETDEF fun, CODEGEN_OBJ res_be_var,      */
/*    CODEGEN be)                                                            */
/*                                                                           */
/*  Code generation of letdef's C function, when it has parameters.          */
/*                                                                           */
/*****************************************************************************/

void BEFnLetDefCodeGenBody(BEFN_LETDEF fun, CODEGEN_OBJ res_be_var, CODEGEN be)
{
  assert(fun->parameters != NULL);
  ExprCodeGen(fun->body, res_be_var, fun->be_type, be);
}
