/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         befn_param.c                                               */
/*  DESCRIPTION:  Parameters of C functions                                  */
/*                                                                           */
/*****************************************************************************/
#include "externs.h"
#define DEBUG1 0
#define DEBUG2 0
#define DEBUG3 0
#define DEBUG4 0
#define DEBUG5 0


struct befn_param_rec {			/* inherits from BEFN                */
  KIND_TAG		kind_tag;	/* kind of entity (KIND_PARAM)       */
  CODEGEN_TYPE		be_type;	/* backend type                      */
  CODEGEN_OBJ		be_obj;		/* corresponding backend object      */
  ARRAY_BEFN		inner_functs;	/* always NULL in BEFN_PARAM         */
  ARRAY_BEFN_PARAM	parameters;	/* always NULL in BEFN_PARAM         */
  CODEGEN_OBJ		inline_be_obj;	/* non-NULL if to be inlined         */
  BOOLEAN		utilized;	/* TRUE when has been called         */
  BOOLEAN		cached;		/* TRUE when calls are to be cached  */

  PARAM_KIND		param_kind;	/* self, compulsory, etc.            */
  FEFN_PARAM		fefn_param;	/* originating front-end parameter   */
  int			index;		/* index in own parameter list       */
  BEFN_FEATURE		creation_feature;	/* if param of creation fun  */
};


/*****************************************************************************/
/*                                                                           */
/*  BEFN_PARAM BEFnParamNew(PARAM_KIND param_kind, FEFN_PARAM fefn_param,    */
/*    BEFN_FEATURE creation_feature)                                         */
/*                                                                           */
/*  Make and return a new C parameter with these attributes.                 */
/*                                                                           */
/*****************************************************************************/

BEFN_PARAM BEFnParamNew(PARAM_KIND param_kind, FEFN_PARAM fefn_param,
  BEFN_FEATURE creation_feature)
{
  BEFN_PARAM res;
  GetMemory(res, BEFN_PARAM);
  res->kind_tag = KIND_BEFN_PARAM;
  res->be_type = NULL;
  res->be_obj = NULL;
  res->inner_functs = NULL;
  res->parameters = NULL;
  res->inline_be_obj = NULL;
  res->utilized = FALSE;
  res->cached = FALSE;

  res->param_kind = param_kind;
  res->fefn_param = fefn_param;
  res->index = -1;
  res->creation_feature = creation_feature;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  PARAM_KIND BEFnParamKind(BEFN_PARAM param)                               */
/*                                                                           */
/*  Return the kind of parameter this is.                                    */
/*                                                                           */
/*****************************************************************************/

PARAM_KIND BEFnParamKind(BEFN_PARAM param)
{
  return param->param_kind;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN BEFnParamIsOptional(BEFN_PARAM param)                            */
/*                                                                           */
/*  Return TRUE if param is optional.                                        */
/*                                                                           */
/*****************************************************************************/

BOOLEAN BEFnParamIsOptional(BEFN_PARAM param)
{
  return param->param_kind == PARAM_OPTIONAL;
}


/*****************************************************************************/
/*                                                                           */
/*  FEFN_PARAM BEFnParamOrigFEFnParam(BEFN_PARAM befn_param)                 */
/*                                                                           */
/*  Return the Nonpareil parameter from which befn_param originated.         */
/*                                                                           */
/*****************************************************************************/

FEFN_PARAM BEFnParamOrigFEFnParam(BEFN_PARAM befn_param)
{
  return befn_param->fefn_param;
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnParamResetOrigFEFnParam(BEFN_PARAM befn_param,                  */
/*    FEFN_PARAM fefn_param)                                                 */
/*                                                                           */
/*  Reset the original fefn_param of befn_param to the given value.  This    */
/*  is used when adjusting clone features.                                   */
/*                                                                           */
/*****************************************************************************/

void BEFnParamResetOrigFEFnParam(BEFN_PARAM befn_param,
  FEFN_PARAM fefn_param)
{
  befn_param->fefn_param = fefn_param;
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnParamListInsert(ARRAY_BEFN_PARAM *befn_params,                  */
/*    BEFN_PARAM befn_param)                                                 */
/*                                                                           */
/*  Insert befn_param into its correct place in *befn_params.  The           */
/*  *befn_params array is passed by reference because it could be NULL,      */
/*  in which case it has to be changed to non-NULL on exit.                  */
/*                                                                           */
/*  Parameters are kept in increasing order of their kind fields:            */
/*  first hidden, then optional, then self, then compulsory.  Parameters     */
/*  of equal kind are kept in the order they were inserted.                  */
/*                                                                           */
/*****************************************************************************/

void BEFnParamListInsert(ARRAY_BEFN_PARAM *befn_params, BEFN_PARAM befn_param)
{
  int i;  BEFN_PARAM befn_param2;
  if( *befn_params == NULL )
    ArrayInit(befn_params);
  ArrayAddLast(*befn_params, NULL);
  for( i = ArraySize(*befn_params) - 1;  i > 0;  i-- )
  {
    befn_param2 = ArrayGet(*befn_params, i - 1);
    if( befn_param2->param_kind <= befn_param->param_kind )
      break;
    ArrayPut(*befn_params, i, befn_param2);
    befn_param2->index = i;
  }
  ArrayPut(*befn_params, i, befn_param);
  befn_param->index = i;
}


/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_TYPE BEFnParamBEType(BEFN_PARAM param)                           */
/*                                                                           */
/*  Return the backend type of param.                                        */
/*                                                                           */
/*****************************************************************************/

CODEGEN_TYPE BEFnParamBEType(BEFN_PARAM param)
{
  return param->be_type;
}


/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_OBJ BEFnParamBEVar(BEFN_PARAM param)                             */
/*                                                                           */
/*  Return the backend variable of param.                                    */
/*                                                                           */
/*****************************************************************************/

CODEGEN_OBJ BEFnParamBEVar(BEFN_PARAM param)
{
  assert(param->be_obj != NULL);
  return param->be_obj;
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnParamListSetBETypes(ARRAY_BEFN_PARAM params, CODEGEN be)        */
/*                                                                           */
/*  Assign backend types to params.                                          */
/*                                                                           */
/*****************************************************************************/

void BEFnParamListSetBETypes(ARRAY_BEFN_PARAM params, CODEGEN be)
{
  BEFN_PARAM param;
  if( params != NULL )
    ArrayForEach(params, param)
      param->be_type = TypeBEType(FEFnParamType(param->fefn_param), be);
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnParamListPrototypeDeclare(ARRAY_BEFN_PARAM params               */
/*    CODEGEN_OBJ be_obj, CODEGEN be)                                        */
/*                                                                           */
/*  Declare a sequence of nameless C parameters, for a function prototype.   */
/*                                                                           */
/*****************************************************************************/

void BEFnParamListPrototypeDeclare(ARRAY_BEFN_PARAM params, CODEGEN_OBJ be_obj,
  CODEGEN be)
{
  BEFN_PARAM param;
  if( params != NULL )
    ArrayForEach(params, param)
      be->PrototypeFormal(be_obj, param->be_type);
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnParamListSetBENamesAndDeclare(ARRAY_BEFN_PARAM params,          */
/*    CODEGEN_OBJ be_obj, CODEGEN be)                                        */
/*                                                                           */
/*  Assign names to these parameters and declare them.                       */
/*                                                                           */
/*****************************************************************************/

void BEFnParamListSetBENamesAndDeclare(ARRAY_BEFN_PARAM params,
  CODEGEN_OBJ be_obj, CODEGEN be)
{
  BEFN_PARAM param;  USTRING name_sugg;
  if( params != NULL )
  {
    if( DEBUG5 )
      fprintf(stderr, "[ BEFnParamListSetBENamesAndDeclare((%s), %s, be)\n",
        BEFnParamListShow(params), be->VarShow(be_obj));
    ArrayForEach(params, param)
    {
      assert(param->be_obj == NULL);
      name_sugg = NameRep(FEFnParamName(param->fefn_param));
      param->be_obj = be->VarMake(NULL, name_sugg);
      be->FunctionFormal(be_obj, param->be_obj, param->be_type);
    }
    if( DEBUG5 )
      fprintf(stderr, "] BEFnParamListSetBENamesAndDeclare returning\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  int BEFnParamIndex(BEFN_PARAM param)                                     */
/*                                                                           */
/*  Return the index of param.                                               */
/*                                                                           */
/*****************************************************************************/

int BEFnParamIndex(BEFN_PARAM param)
{
  assert(param->index != -1);
  return param->index;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN BEFnParamHasCreationFeature(BEFN_PARAM param)                    */
/*                                                                           */
/*  Return TRUE if param is associated with a creation feature.              */
/*                                                                           */
/*****************************************************************************/

BOOLEAN BEFnParamHasCreationFeature(BEFN_PARAM param)
{
  return param->creation_feature != NULL;
}


/*****************************************************************************/
/*                                                                           */
/*  BEFN_FEATURE BEFnParamCreationFeature(BEFN_PARAM param)                  */
/*                                                                           */
/*  If param is a parameter of a creation function, return the creation      */
/*  feature that it corresponds with.                                        */
/*                                                                           */
/*****************************************************************************/

BEFN_FEATURE BEFnParamCreationFeature(BEFN_PARAM param)
{
  assert(param->creation_feature != NULL);
  return param->creation_feature;
}


/*****************************************************************************/
/*                                                                           */
/*  BEFN_PARAM BEFnParamListFind(ARRAY_BEFN_PARAM params,                    */
/*    BEFN_FEATURE cfeature)                                                 */
/*                                                                           */
/*  Return the element of params corresponding to cfeature.  It must exist.  */
/*                                                                           */
/*****************************************************************************/

BEFN_PARAM BEFnParamListFind(ARRAY_BEFN_PARAM params,
  BEFN_FEATURE cfeature)
{
  int i;  BEFN_PARAM param;
  assert(params != NULL);
  for( i = 0;  i < ArraySize(params);  i++ )
  {
    param = ArrayGet(params, i);
    if( param->creation_feature == cfeature )
      return param;
  }
  assert(FALSE);
  return NULL;  /* keep compiler happy */
}


/*****************************************************************************/
/*                                                                           */
/*  UTF8 BEFnParamShow(BEFN_PARAM param)                                     */
/*                                                                           */
/*  Show param for debug output.                                             */
/*                                                                           */
/*****************************************************************************/

UTF8 BEFnParamShow(BEFN_PARAM param)
{
  return (UTF8) NameShow(FEFnParamName(param->fefn_param));
}


/*****************************************************************************/
/*                                                                           */
/*  UTF8 BEFnParamListShow(ARRAY_BEFN_PARAM parameters)                      */
/*                                                                           */
/*  Show a parameter list for debug output.                                  */
/*                                                                           */
/*****************************************************************************/

UTF8 BEFnParamListShow(ARRAY_BEFN_PARAM parameters)
{
  AFACTORY af;  int i;  BEFN_PARAM param;
  if( parameters == NULL )
    return (UTF8) "";
  else
  {
    af = AStringBegin();
    for( i = 0;  i < ArraySize(parameters);  i++ )
    {
      if( i > 0 )
	AStringAddAString(af, ", ");
      param = ArrayGet(parameters, i);
      AStringAddAString(af, (ASTRING) NameShow(FEFnParamName(param->fefn_param)));
    }
    return (UTF8) AStringEnd(af);
  }
}
