/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         befn_precond.c                                              */
/*  DESCRIPTION:  Back-end creation function default value                   */
/*                                                                           */
/*****************************************************************************/
#include "externs.h"
#include <string.h>

struct befn_precond_rec {		/* inherits from BEFN                */
  KIND_TAG		kind_tag;	/* KIND_CLASS                        */
  CODEGEN_TYPE		be_type;	/* backend type                      */
  CODEGEN_OBJ		be_obj;		/* backend function                  */
  ARRAY_BEFN		inner_functs;	/* from body of default value        */
  ARRAY_BEFN_PARAM	parameters;	/* ordinary params (NULL if none)    */
  CODEGEN_OBJ		inline_be_obj;	/* non-NULL if to be inlined         */
  BOOLEAN		utilized;	/* TRUE when has been called         */
  BOOLEAN		cached;		/* TRUE when calls are to be cached  */

  FEFN_PRECOND		fefn_precond;	/* originating front-end function    */
  EXPR			body;		/* body of default value             */
  int			visit_num;	/* for checking initialization order */
};


/*****************************************************************************/
/*                                                                           */
/*  BEFN_PRECOND BEFnPrecondNew(FEFN_PRECOND fefn_precond, EXPR body)        */
/*                                                                           */
/*  Make a new back-end precondition corresponding to fefn_precond,          */
/*  with the given body.                                                     */
/*                                                                           */
/*****************************************************************************/

BEFN_PRECOND BEFnPrecondNew(FEFN_PRECOND fefn_precond, EXPR body)
{
  BEFN_PRECOND res;
  GetMemory(res, BEFN_PRECOND);
  res->kind_tag = KIND_BEFN_PRECOND;
  res->be_type = NULL;
  res->be_obj = NULL;
  res->inner_functs = NULL;
  ArrayInit(&res->parameters);
  res->inline_be_obj = NULL;
  res->utilized = FALSE;
  res->cached = FALSE;
  res->fefn_precond = fefn_precond;
  res->body = body;
  res->visit_num = 0;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnPrecondFinalize(BEFN_PRECOND befn_precond, CODEGEN be)          */
/*                                                                           */
/*  Carry out the specification of BEFnFinalize on this back-end precondion. */
/*                                                                           */
/*****************************************************************************/

void BEFnPrecondFinalize(BEFN_PRECOND befn_precond, CODEGEN be)
{
  static USTRING pre_str = NULL;
  FEFN_FEATURE orig_feature;  CLASS_VIEW orig_cv;
  if( pre_str == NULL )
    pre_str = AStringToUString("pre");
  befn_precond->be_type = be->void_type;
  orig_feature = FEFnPrecondOrigFeature(befn_precond->fefn_precond);
  orig_cv = FEFnFeatureClassView(orig_feature);
  befn_precond->be_obj = be->FunctionMake3(NameRep(ClassViewName(orig_cv)),
    NameRep(FEFnFeatureName(orig_feature)), pre_str, TRUE, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN BEFnPrecondInitOrder(BEFN_PRECOND befn_precond, int visit_num,   */
/*    BOOLEAN *report, BEFN_SYSTEM_INIT fun)                                 */
/*                                                                           */
/*  Carry out the specification of BEFnInitOrder on back-end precondition    */
/*  befn_precond.                                                            */
/*                                                                           */
/*****************************************************************************/

BOOLEAN BEFnPrecondInitOrder(BEFN_PRECOND befn_precond, int visit_num,
  BOOLEAN *report, BEFN_SYSTEM_INIT fun)
{
  /* check whether visited already and update if not */
  if( befn_precond->visit_num >= visit_num )
    return TRUE;
  befn_precond->visit_num = visit_num;

  /* check initialization order of body */
  return ExprInitOrder(befn_precond->body, visit_num, report, fun);
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnPrecondCodeGenBody(BEFN_PRECOND befn_precond,                   */
/*    CODEGEN_OBJ res_be_var, CODEGEN be)                                    */
/*                                                                           */
/*  Code generate the body of this creation feature default value function.  */
/*                                                                           */
/*****************************************************************************/

void BEFnPrecondCodeGenBody(BEFN_PRECOND befn_precond,
  CODEGEN_OBJ res_be_var, CODEGEN be)
{
  ExprCodeGen(befn_precond->body, res_be_var, befn_precond->be_type, be);
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnPrecondCodeGenCall(BEFN_PRECOND befn_precond,                   */
/*    ARRAY_BEFN_PARAM params, CODEGEN be)                                   */
/*                                                                           */
/*  Generate a call on befn_precond.  The actual parameters are to be        */
/*  calls on the elements of params that correspond with the parameters      */
/*  of befn_precond.                                                         */
/*                                                                           */
/*****************************************************************************/

void BEFnPrecondCodeGenCall(BEFN_PRECOND befn_precond, ARRAY_BEFN_PARAM params,
  CODEGEN be)
{
  int i, param_count, pos;
  BEFN_PARAM befn_param, param;  FEFN_PARAM fefn_param;  FEFN hidden;
  be->CallBegin(befn_precond->be_obj);
  param_count = befn_precond->parameters == NULL ? 0 :
    ArraySize(befn_precond->parameters);
  for( i = 0;  i < param_count;  i++ )
  {
    if( i > 0 )
      be->CallContinue(befn_precond->be_obj, i);

    /* find param, the element of params corresponding to ith befn parameter */
    befn_param = ArrayGet(befn_precond->parameters, i);
    fefn_param = BEFnParamOrigFEFnParam(befn_param);
    hidden = FEFnParamHiddenValue(fefn_param);
    assert(hidden != NULL && NamedKind((NAMED) hidden) == KIND_FEFN_PARAM);
    param = FEFnParamBEFnParam((FEFN_PARAM) hidden);
    assert(ArrayContains(params, param, &pos));

    /* generate a call on this parameter */
    Var(BEFnParamBEVar(param));
  }
  be->CallEnd(befn_precond->be_obj);
}
