/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         system_init.c                                              */
/*  DESCRIPTION:  System initialization function                             */
/*                                                                           */
/*  Implementation note.  Every predefined object in the system is           */
/*  registered with this function, including all the all_predefined and      */
/*  all_enumerated features, but not the predefined objects in enumerated    */
/*  classes.  This function will initialize all of these features, and the   */
/*  predefined objects of enumerated classes will be initialized during the  */
/*  initialization of their all_enumerated feature.  Exactly what has to be  */
/*  done varies with feature type, as does the way of working out which      */
/*  features must be initialized before which.                               */
/*                                                                           */
/*****************************************************************************/
#include <string.h>
#include "externs.h"
#define DEBUG1 1

struct befn_system_init_rec {		/* inherits BEFN		     */
  KIND_TAG		kind_tag;	/* kind of entity                    */
  CODEGEN_TYPE		be_type;	/* result type                       */
  CODEGEN_OBJ		be_obj;		/* corresponding backend object      */
  ARRAY_BEFN		inner_functs;	/* to generate just before this fn   */
  ARRAY_BEFN_PARAM	parameters;	/* ordinary params (NULL if none)    */
  CODEGEN_OBJ		inline_be_obj;	/* non-NULL if to be inlined         */
  BOOLEAN		utilized;	/* TRUE when has been called         */
  BOOLEAN		cached;		/* TRUE when calls are to be cached  */

  SYSTEM		system;		/* the system being finalized        */
  ARRAY_CLASS		init_classes;	/* classes requiring initialization  */
};


/*****************************************************************************/
/*                                                                           */
/*  BEFN_SYSTEM_INIT BEFnSystemInitMake(SYSTEM sys)                          */
/*                                                                           */
/*  Make a new system initialization function.                               */
/*                                                                           */
/*****************************************************************************/

BEFN_SYSTEM_INIT BEFnSystemInitMake(SYSTEM sys)
{
  BEFN_SYSTEM_INIT res;
  GetMemory(res, BEFN_SYSTEM_INIT);
  res->kind_tag = KIND_BEFN_SYSTEM_INIT;
  res->be_type = NULL;
  res->be_obj = NULL;
  res->inner_functs = NULL;
  res->parameters = NULL;
  res->inline_be_obj = NULL;
  res->utilized = FALSE;
  res->cached = FALSE;
  res->system = sys;
  ArrayInit(&res->init_classes);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnSystemInitRegisterClass(BEFN_SYSTEM_INIT fun, CLASS c)          */
/*                                                                           */
/*  Register class c with fun as a class which has predefined objects        */
/*  requiring initialization by this system initialization function.         */
/*                                                                           */
/*  These registrations must occur in the order in which classes are         */
/*  to be initialized, i.e. in a safe initialization order.  See the         */
/*  implementation notes for how this order is determined.                   */
/*                                                                           */
/*****************************************************************************/

void BEFnSystemInitRegisterClass(BEFN_SYSTEM_INIT fun, CLASS c)
{
  ArrayAddLast(fun->init_classes, c);
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnSystemInitFinalize(BEFN_SYSTEM_INIT fun, CODEGEN be)            */
/*                                                                           */
/*  Finalize the be_type and be_obj fields of fun.                           */
/*                                                                           */
/*****************************************************************************/

void BEFnSystemInitFinalize(BEFN_SYSTEM_INIT fun, CODEGEN be)
{
  fun->be_type = ClassBEType(ClassArray);
  fun->be_obj = be->FunctionMake("system_init", TRUE, TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnSystemInitCodeGenBody(BEFN_SYSTEM_INIT fun,                     */
/*    CODEGEN_OBJ res_be_var, CODEGEN be)                                    */
/*                                                                           */
/*  Generate the body of system initialization function fun onto be.         */
/*  It just consists of a call to the initialization function of each        */
/*  class that has an all_predefined or all_enumerated array - each          */
/*  non-generic class.  These will have been reported already to this        */
/*  function, in a suitable order for initialization.                        */
/*                                                                           */
/*****************************************************************************/

void BEFnSystemInitCodeGenBody(BEFN_SYSTEM_INIT fun,
  CODEGEN_OBJ res_be_var, CODEGEN be)
{
  CLASS c;  CODEGEN_OBJ be_res, be_res_el;  int res_len, i;

  /* find out how long res needs to be */
  res_len = ArraySize(fun->init_classes);
  ArrayForEach(fun->init_classes, c)
    if( ClassIsEnum(c) )
      res_len++;

  /* declare and initialize res array */
  be_res = be->VarMake("res", NULL);
  DeclareAndAssign(be_res, ClassBEType(ClassArray),
    Call1(NPBack_Array_Create_Fn1, Int(res_len)));

  /* init_all_enumerated calls */
  i = 0;
  be_res_el = be->VarIndexedAttributeMake(be_res, "elems", i);
  be->CommentSmall("initialize all_enumerated arrays", FALSE, TRUE);
  ArrayForEach(fun->init_classes, c)
    if( ClassIsEnum(c) )
    {
      be->VarIndexedAttributeUpdateIndex(be_res_el, i++);
      VarAssign(be_res_el, BEFnEnumInitCodeGenCall(ClassBEFnEnumInit(c), be));
    }

  /* class (i.e. all_predefined and trie) calls */
  be->CommentSmall("initialize all_predefined arrays and tries", FALSE, TRUE);
  ArrayForEach(fun->init_classes, c)
  {
    be->VarIndexedAttributeUpdateIndex(be_res_el, i++);
    VarAssign(be_res_el, BEFnClassInitCodeGenCall(ClassBEFnClassInit(c), be));
  }

  /* finalize and return res */
  VarAssign(be_res, Call1(NPBack_Array_Create_Fn2, Var(be_res)));
  VarAssign(res_be_var, Var(be_res));
}


/*****************************************************************************/
/*                                                                           */
/*  void BEFnSystemInitCodeGenLoadFn(BEFN_SYSTEM_INIT fun,                   */
/*    CODEGEN_OBJ be_fn, CODEGEN be)                                         */
/*                                                                           */
/*  Code generate the system load function that inverts system_init above.   */
/*  Its name is be_fn.                                                       */
/*                                                                           */
/*****************************************************************************/

void BEFnSystemInitCodeGenLoadFn(BEFN_SYSTEM_INIT fun,
  CODEGEN_OBJ be_fn, CODEGEN be)
{
  CODEGEN_OBJ be_all, be_all_len, be_all_el, be_obj;
  int i, all_len;  CLASS c;  CODEGEN_TYPE array_type;
  be_all = be->VarMake("all_all", NULL);
  be_all_len = be->AttrMake("length");

  /* function header "void system_load(array all_all)" */
  be->FunctionBegin(be_fn, be->void_type);
  be->FunctionFormal(be_fn, be_all, ClassBEType(ClassArray));
  be->FunctionContinue(be_fn);

  /* find out how long all_all needs to be */
  all_len = ArraySize(fun->init_classes);
  ArrayForEach(fun->init_classes, c)
    if( ClassIsEnum(c) )
      all_len++;

  /* assert length ok */
  Assert(Equal(Call1(be_all_len, Var(be_all)), Int(all_len)));

  /* load all_enumerated arrays */
  i = 0;
  be_all_el = be->VarIndexedAttributeMake(be_all, "elems", i);
  array_type = ClassBEType(ClassArray);
  be->CommentSmall("load all_enumerated arrays", FALSE, TRUE);
  ArrayForEach(fun->init_classes, c)
    if( ClassIsEnum(c) )
    {
      be_obj = BEFnEnumInitCodeGenLoadBEObj(ClassBEFnEnumInit(c));
      be->VarIndexedAttributeUpdateIndex(be_all_el, i++);
      Stmt(Call1(be_obj, Cast(array_type, NULL, Var(be_all_el))));
    }

  /* class (i.e. all_predefined and trie) calls */
  be->CommentSmall("load all_predefined arrays and tries", FALSE, TRUE);
  ArrayForEach(fun->init_classes, c)
  {
    be->VarIndexedAttributeUpdateIndex(be_all_el, i++);
    be_obj = BEFnClassInitCodeGenLoadBEObj(ClassBEFnClassInit(c));
    Stmt(Call1(be_obj, Cast(array_type, NULL, Var(be_all_el))));
  }

  /* function footer */
  be->FunctionEnd(be_fn);
}
