/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         class.c                                                    */
/*  DESCRIPTION:  Nonpareil classes (implementation)                         */
/*                                                                           */
/*****************************************************************************/
#include "externs.h"
#include <string.h>
#define DEBUG1 0
#define DEBUG2 0
#define DEBUG3 0
#define DEBUG4 0
#define DEBUG5 0
#define DEBUG6 0
#define DEBUG7 0
#define DEBUG8 0
#define DEBUG9 0
#define DEBUG10 0
#define DEBUG11 0
#define DEBUG12 0
#define DEBUG14 0
#define DEBUG15 0
#define DEBUG16 0
#define DEBUG17 0

/*****************************************************************************/
/*                                                                           */
/*  CLASS                                                                    */
/*                                                                           */
/*  A CLASS represents one class in the complete system.  It contains all    */
/*  its creation features, non-creation features, and invariants, including  */
/*  inherited ones, and including all class extensions.                      */
/*                                                                           */
/*****************************************************************************/

#define MAX_EQ_CLASS_OFFSET 256

typedef enum {
  BUILTIN_UNTAGGED,			/* builtin class, immediate value    */
  BUILTIN_TAGGED,			/* builtin class, tagged objects     */
  TAGGED,				/* ordinary class, tagged objects    */
  ENUMERATED				/* enumerated class                  */
} CLASS_TYPE;

struct class_rec {

  /* class identity and simple properties */
  FILE_POS	      file_pos;		/* file position where class defined */
  CLASS_VIEW	      orig_cv;		/* original class view               */
  ARRAY_NAME	      name_set;		/* all names this class is known by  */
  CLASS_TYPE	      ctype;		/* tagged, builtin, enumerated       */
  BOOLEAN	      is_abstract;	/* contains abstract feature(s)      */
  CASE_TYPE	      case_type;	/* used by case expressions          */

  /* types */
  TYPE_VARS	      type_vars;	/* type variables in following types */
  TYPE		      class_type;	/* type of class                     */
  TYPE		      inherit_type;	/* inherited type, or NULL	     */
  ITYPES	      parent_set;	/* the parent set                    */
  ITYPES	      ancestor_set;	/* the ancestors                     */
  ARRAY_CLASS	      inherit_parents;	/* inherit parents of this class     */
  ARRAY_CLASS	      inherit_children;	/* inherit children of this class    */

  /* inheritance */
  ARRAY_CLASS	      eq_class;		/* the equivalent classes to this    */
  int		      eq_class_id;	/* the equivalence class id          */
  int		      eq_class_offset;	/* the equivalence class offset      */
  int		      sort_key;		/* eq_class_id * N + eq_class_offset */
  ARRAY_TYPE	      coerce_types;	/* result types of coercion features */
  ARRAY_FEFN_FEATURE  coerce_feature_views; /* coercion feature views        */

  /* features */
  ARRAY_BEFN_FEATURE  cfeatures;	/* all creation features of class    */
  ARRAY_BEFN_FEATURE  orig_cfeatures;	/* creation features based here      */
  ARRAY_BEFN_FEATURE  orig_nfeatures;	/* noncreation features based here   */
  ARRAY_BEFN_FEATURE  orig_predefs;	/* predefined objects based here     */
  ARRAY_BEFN_FEATURE  all_predefs;	/* predefs defined here or below     */
  BEFN_FEATURE	      all_predefined;	/* all_predefined feature, if any    */
  BEFN_FEATURE	      all_enumerated;	/* all_enumerated feature, if any    */
  BEFN_FEATURE	      all_enum_trie;	/* trie of all enumerated, if any    */
  BEFN_FEATURE	      legal_code;	/* legal_code feature, if any        */
  BEFN_FEATURE	      with_code;	/* with_code feature, if any         */

  /* fields used only by enumerated classes */
  ISET		      enum_code_set;	/* all codes used by predefined objs */
  BOOLEAN	      enum_codes_done;	/* TRUE when all predefs have codes  */
  int		      enum_trie_height;	/* between 1 and 4 inclusive         */
  int		      enum_trie_width;	/* no. of elements in root node      */
  BOOLEAN	      enum_has_range;	/* has nontrivial code range         */

  /* disjoint sets */
  CLASS		      ds_parent;	/* parent class in disjoint sets     */
  ARRAY_CLASS	      ds_component;	/* all classes in the component      */
  BEFN_FEATURE	      ds_clone_feature;	/* if any class in compt has clone   */

  /* miscellaneous */
  BEFN_CREATION	      creation_fn;	/* creation function                 */
  BEFN_CLASS_INIT     befn_class_init;	/* class initialization function     */
  BEFN_ENUM_INIT      befn_enum_init;	/* all_enumerated init function      */
  BEFN_INVARIANT      befn_invariant;	/* non-inherited part of invariant   */
  EXPR		      min_value;	/* the min poss value of this class  */
  EXPR		      max_value;	/* the max poss value of this class  */

  /* code generation */
  CODEGEN_TYPE	      be_type;		/* backend name of class type        */
  CODEGEN_TYPE	      be_content_type;	/* if enum, the content type         */
  CODEGEN_OBJ	      be_type_tag;	/* backend name of enum value        */
  int		      swizzle_width;	/* no. of bits needed for swizzling  */
  int		      swizzle_offset;	/* offset of swizzle_bits field      */
  CODEGEN_OBJ	      swizzle_fn;	/* swizzle function be_var           */
  CODEGEN_OBJ	      unswizzle_fn;	/* unswizzle function be_var         */
  BOOLEAN	      layout_done;	/* TRUE when record layout is done   */
  int		      record_width;	/* record width in bits, when done   */
  ISET		      layout_set;	/* offsets consumed by layout so far */
  BOOLEAN	      has_pointer;	/* TRUE if has any pointer fields    */
};


/*****************************************************************************/
/*                                                                           */
/*  Submodule "establish"                                                    */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/* CLASS ClassNew(FILE_POS file_pos, CLASS_VIEW orig_cv, TYPE_VARS type_vars,*/
/*    TYPE inherit_type, BOOLEAN is_enum, BOOLEAN is_builtin)                */
/*                                                                           */
/*  Return a new class object with these attributes.                         */
/*                                                                           */
/*  If the class name is "bool" or "int", set the ClassBool or ClassInt      */
/*  variable early.  These are needed when initializing invariant views and  */
/*  enumerated classes respectively.                                         */
/*                                                                           */
/*****************************************************************************/

CLASS ClassNew(FILE_POS file_pos, CLASS_VIEW orig_cv, TYPE_VARS type_vars,
  TYPE inherit_type, BOOLEAN is_enum, BOOLEAN is_builtin)
{
  CLASS res;
  static USTRING bool_str = NULL;
  static USTRING int_str = NULL;
  static USTRING array_str = NULL;
  static USTRING string_str = NULL;
  if( bool_str == NULL )
  {
    bool_str = AStringToUString("bool");
    int_str = AStringToUString("int");
    array_str = AStringToUString("array");
    string_str = AStringToUString("string");
  }
  GetMemory(res, CLASS);

  /* class identity and simple properties */
  res->file_pos = file_pos;
  res->orig_cv = orig_cv;
  ArrayInit(&res->name_set);
  ArrayAddLast(res->name_set, ClassViewName(orig_cv));
  assert(!is_enum || !is_builtin);
  if( is_enum )
    res->ctype = ENUMERATED;
  else if( !is_builtin )
    res->ctype = TAGGED;
  else if( UStringEqual(NameRep(ClassViewName(orig_cv)), array_str) ||
           UStringEqual(NameRep(ClassViewName(orig_cv)), string_str) )
    res->ctype = BUILTIN_TAGGED;
  else
    res->ctype = BUILTIN_UNTAGGED;
  res->is_abstract = FALSE;
  res->case_type = is_enum ? CASE_TYPE_ENUM : CASE_TYPE_OTHER;

  /* set ClassBool or ClassInt if this is it */
  if( UStringEqual(NameRep(ClassViewName(orig_cv)), bool_str) )
    ClassBool = res;
  else if( UStringEqual(NameRep(ClassViewName(orig_cv)), int_str) )
    ClassInt = res;

  /* types */
  res->type_vars = type_vars;
  res->class_type = TypeMakeInstantiatedClassType(file_pos, res, type_vars);
  res->inherit_type = inherit_type;
  res->parent_set = NULL;
  res->ancestor_set = NULL;
  res->inherit_parents = NULL;
  ArrayInit(&res->inherit_children);

  /* inheritance */
  res->eq_class = NULL;
  res->eq_class_id = 0;
  res->eq_class_offset = 0;
  res->sort_key = 0;
  ArrayInit(&res->coerce_types);
  ArrayInit(&res->coerce_feature_views);

  /* features */
  ArrayInit(&res->cfeatures);
  ArrayInit(&res->orig_cfeatures);
  ArrayInit(&res->orig_nfeatures);
  ArrayInit(&res->orig_predefs);
  ArrayInit(&res->all_predefs);
  res->all_predefined = NULL;
  res->all_enumerated = NULL;
  res->all_enum_trie = NULL;
  res->legal_code = NULL;
  res->with_code = NULL;

  /* fields used only by enumerated classes */
  res->enum_code_set = NULL;
  res->enum_codes_done = FALSE;
  res->enum_trie_height = -1;
  res->enum_trie_width = -1;
  res->enum_has_range = FALSE;

  /* disjoint sets */
  res->ds_parent = NULL;
  res->ds_component = NULL;
  res->ds_clone_feature = NULL;

  /* miscellaneous */
  res->creation_fn = (is_builtin ? NULL : BEFnCreationNew(res));
  res->befn_class_init = NULL;
  res->befn_enum_init = NULL;
  res->befn_invariant = NULL;
  res->min_value = NULL;
  res->max_value = NULL;

  /* code generation vars (initialized properly at start of code gen phase) */
  /* res->is_obj_ref = FALSE; */
  res->be_type = NULL;
  res->be_content_type = NULL;
  res->be_type_tag = NULL;
  res->swizzle_width = -1;
  res->swizzle_offset = -1;
  res->swizzle_fn = NULL;
  res->unswizzle_fn = NULL;
  res->layout_done = FALSE;
  res->record_width = 0;
  res->layout_set = NULL;
  res->has_pointer = FALSE;

  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  FILE_POS ClassFilePos(CLASS c)                                           */
/*                                                                           */
/*  Return the file position where c is initially defined.                   */
/*                                                                           */
/*****************************************************************************/

FILE_POS ClassFilePos(CLASS c)
{
  return c->file_pos;
}


/*****************************************************************************/
/*                                                                           */
/*  CLASS_VIEW ClassOrigClassView(CLASS c)                                   */
/*                                                                           */
/*  Return the original view of c.                                           */
/*                                                                           */
/*****************************************************************************/

CLASS_VIEW ClassOrigClassView(CLASS c)
{
  return c->orig_cv;
}


/*****************************************************************************/
/*                                                                           */
/*  CLASS_VIEW ClassToView(CLASS c, CONTEXT cxt)                             */
/*                                                                           */
/*  Return the view of class c in cxt, or NULL if none.                      */
/*                                                                           */
/*****************************************************************************/

CLASS_VIEW ClassToView(CLASS c, CONTEXT cxt)
{
  NAME name;  CLASS_VIEW res;
  ArrayForEach(c->name_set, name)
    if( ContextRetrieveClassView(cxt, NameKey(name), &res) &&
	ClassViewClass(res) == c )
      return res;
  return NULL;
}


/*****************************************************************************/
/*                                                                           */
/*  void ClassRegisterName(CLASS class, NAME name)                           */
/*                                                                           */
/*  Register the fact that class is known to someone somewhere by this name. */
/*                                                                           */
/*****************************************************************************/

void ClassRegisterName(CLASS class, NAME name)
{
  NAME n2;
  if( DEBUG8 )
    fprintf(stderr, "ClassRegisterName(class %p, %s)\n",
      (void *) class, UStringToUTF8(NameKey(name)));
  ArrayForEach(class->name_set, n2)
    if( UStringEqual(NameKey(n2), NameKey(name)) )
      return;
  ArrayAddLast(class->name_set, name);
}


/*****************************************************************************/
/*                                                                           */
/*  NAME ClassName(CLASS c, CONTEXT cxt)                                     */
/*                                                                           */
/*  Return the name of c in cxt.  It is an error if c is not present.        */
/*                                                                           */
/*****************************************************************************/

NAME ClassName(CLASS c, CONTEXT cxt)
{
  CLASS_VIEW cv;
  cv = ClassToView(c, cxt);
  assert(cv != NULL);
  return ClassViewName(cv);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ClassIsBuiltin(CLASS c)                                          */
/*                                                                           */
/*  Return the builtin status of c.                                          */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ClassIsBuiltin(CLASS c)
{
  /* return c->is_builtin; */
  return c->ctype <= BUILTIN_TAGGED;
}


/*****************************************************************************/
/*                                                                           */
/*  CASE_TYPE ClassCaseType(CLASS c)                                         */
/*                                                                           */
/*  Return the case type of c.  This indicates whether c is an enumerated    */
/*  type, an integral type, the string type, or something else, and is       */
/*  used by case expressions.                                                */
/*                                                                           */
/*****************************************************************************/

CASE_TYPE ClassCaseType(CLASS c)
{
  return c->case_type;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "establish predefined classes"                                 */
/*                                                                           */
/*  These classes are defined in the first module of the system file, but    */
/*  these global variables give direct access to them, and setting these     */
/*  variables is an opportunity to check that the classes all arrived.       */
/*                                                                           */
/*  ClassTuple[i] is the class with name tuple-i for 2 <= i <= MAX_TUPLE.    */
/*                                                                           */
/*****************************************************************************/

CLASS ClassVoid = NULL;			/* internal use only                 */
CLASS ClassObject = NULL;		/* the object class                  */
CLASS ClassBool;			/* the bool class                    */
CLASS ClassByte;			/* the byte class                    */
CLASS ClassShort;			/* the short class                   */
CLASS ClassInt;				/* the int class                     */
CLASS ClassUByte;			/* the ubyte class                   */
CLASS ClassUShort;			/* the ushort class                  */
CLASS ClassUInt;			/* the uint class                    */
CLASS ClassReal;			/* the real class                    */
CLASS ClassChar;			/* the char class                    */
CLASS ClassList;			/* the list class                    */
CLASS ClassNList;			/* the nlist class                   */
CLASS ClassEList;			/* the elist class                   */
CLASS ClassArray;			/* the array class                   */
CLASS ClassString;			/* the string class                  */
CLASS ClassTuple[MAX_TUPLE + 1];	/* the tuple_n classes, n = 2-10     */
CLASS ClassFun[MAX_FUN_PARAMS + 1];	/* the fun_n classes, n = 1-6        */
CLASS ClassFunRep[MAX_FUN_PARAMS+1];	/* the fun_n_rep classes, n = 1-6    */
CLASS ClassFunNM[MAX_FUN_PARAMS+1][MAX_FUN_PARAMS+1];	/* fun_n_m classes   */


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN do_class(ASTRING str, CLASS *variable, SYSTEM_VIEW sv, int g,    */
/*    BOOLEAN is_builtin)                                                    */
/*                                                                           */
/*  Set one class variable; used by ClassInitPredefined below.  It finds     */
/*  the class in the first system view sv, sets *variable, and checks        */
/*  that the class has g generic parameters, is not private, is norename,    */
/*  and has the given builtin character.                                     */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN do_class(ASTRING str, CLASS *variable, SYSTEM_VIEW sv, int g,
  BOOLEAN is_builtin)
{
  CLASS c;  CLASS_VIEW cv;

  /* make sure class is present in first module */
  prohibit3(!SystemViewRetrieveClassView(sv, AStringToUString(str), &cv),
    "%s: class \"%s\" missing from first module \"%s\"\n",
    FilePosShow(SystemViewFilePos(sv)), str, NameShow(SystemViewName(sv)));
  c = ClassViewClass(cv);

  /* make sure class has correct number of generic parameters */
  prohibit2(g != (c->type_vars == NULL ? 0 : ArraySize(c->type_vars)),
    "%s: class must have %d generic parameters\n",
    NameFullShow(ClassViewName(cv)), g);

  /* make sure class is not private (except void class) */
  if( variable != &ClassVoid )
    prohibit1(ClassViewIsPrivate(cv), "%s: class must not be private\n",
      NameFullShow(ClassViewName(cv)));

  /* make sure class is norename (except void class which is private anyway) */
  if( variable != &ClassVoid )
    prohibit1(!ClassViewIsNoRename(cv), "%s: class must be norename\n",
      NameFullShow(ClassViewName(cv)));

  /* sort out builtin status of class */
  if( is_builtin )
  {
    /* class must be builtin */
    prohibit2(!ClassIsBuiltin(c), "%s: class %s must be builtin\n",
      FilePosShow(ClassViewFilePos(cv)), NameFullShow(ClassViewName(cv)));
  }
  else
  {
    /* class must not be builtin */
    prohibit2(ClassIsBuiltin(c), "%s: class %s must not be builtin\n",
      FilePosShow(ClassViewFilePos(cv)), NameFullShow(ClassViewName(cv)));
  }

  /* OK, so set the variable and quit */
  *variable = c;
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ClassInitPredefined(SYSTEM_VIEW sv, CODEGEN be)                  */
/*                                                                           */
/*  Initialize all the predefined classes above; find them in sv.            */
/*                                                                           */
/*****************************************************************************/
#define set(str, class_var, generics, builtin)			\
if( !do_class(str, class_var, sv, generics, builtin) )		\
  return FALSE;

BOOLEAN ClassInitPredefined(SYSTEM_VIEW sv)
{
  int i, j;  char buff[20];  CLASS_VIEW cv;

  /* do the non-tuple classes */
  /* NAME          VARIABLE  GENERICS  BUILTIN */
  set("object",    &ClassObject,    0, FALSE);
  set("void",      &ClassVoid,      0, TRUE);
  set("bool",      &ClassBool,      0, FALSE);
  set("char",      &ClassChar,      0, FALSE);
  set("byte",      &ClassByte,      0, TRUE);
  set("short",     &ClassShort,     0, TRUE);
  set("int",       &ClassInt,       0, TRUE);
  set("ubyte",     &ClassUByte,     0, TRUE);
  set("ushort",    &ClassUShort,    0, TRUE);
  set("uint",      &ClassUInt,      0, TRUE);
  set("real",      &ClassReal,      0, TRUE);
  set("list",      &ClassList,      1, FALSE);
  set("nlist",     &ClassNList,     1, FALSE);
  set("elist",     &ClassEList,     1, FALSE);
  set("array",     &ClassArray,     1, TRUE);
  set("string",    &ClassString,    0, TRUE);

  /* do tuple classes from 2 to whatever; first two are "pair" and "triple" */
  ClassTuple[0] = ClassTuple[1] = NULL;
  set("pair", &ClassTuple[2], 2, FALSE);
  set("triple", &ClassTuple[3], 3, FALSE);
  for( i = 4;  i <= MAX_TUPLE;  i++ )
  {
    sprintf(buff, "tuple%d", i);
    set(buff, &ClassTuple[i], i, FALSE);
  }

  /* make sure there are no extra tuple classes */
  sprintf(buff, "tuple%d", MAX_TUPLE + 1);
  if( SystemViewRetrieveClassView(sv, AStringToUString(buff), &cv) )
  {
    fprintf(stderr,
      "%s: unexpected tuple class \"%s\"; last should be \"tuple%d\"\n",
      FilePosShow(ClassViewFilePos(cv)), buff, MAX_TUPLE);
    fprintf(stderr, "  See file externs.h, near the top, for information\n");
    fprintf(stderr, "  about tuple classes, including the right way to\n");
    fprintf(stderr, "  fix this problem.\n");
    return FALSE;
  }

  /* do the fun_n classes */
  for( i = 1;  i <= MAX_FUN_PARAMS;  i++ )
  {
    sprintf(buff, "fun%d", i);
    set(buff, &ClassFun[i], i+1, FALSE);
  }

  /* make sure there are no extra fun classes */
  sprintf(buff, "fun%d", MAX_FUN_PARAMS + 1);
  if( SystemViewRetrieveClassView(sv, AStringToUString(buff), &cv) )
  {
    fprintf(stderr,
      "%s: unexpected function class \"%s\"; last should be \"fun%d\"\n",
      FilePosShow(ClassViewFilePos(cv)), buff, MAX_FUN_PARAMS);
    fprintf(stderr, "  See file externs.h, near the top, for information\n");
    fprintf(stderr, "  about function classes, including the right way to\n");
    fprintf(stderr, "  fix this problem.\n");
    return FALSE;
  }

  /* do the fun_n_rep classes: "void * (*fun_n_rep)(void *, ... void *)" */
  /* strcpy(buff2, ")(void *)"); */
  for( i = 1;  i <= MAX_FUN_PARAMS;  i++ )
  {
    sprintf(buff, "fun%d_rep", i);
    /* be_type = be->TypeMakeFunctionType(AStringToUString(buff), i); */
    set(buff, &ClassFunRep[i], i+1, TRUE);
    /* ClassFunRep[i]->is_obj_ref = FALSE; */
    /* strcpy(&buff2[strlen(buff2) - 1], ", void *)"); */
  }

  /* do the fun_n_m classes */
  for( i = 1;  i <= MAX_FUN_PARAMS;  i++ )
  {
    for( j = 1;  j <= i;  j++ )
    {
      sprintf(buff, "fun%d_%d", i, j);
      set(buff, &ClassFunNM[i][j], i+1, FALSE);
    }
  }

  /* min and max values */
  ClassByte->min_value = ExprLitFromInt(ClassByte->file_pos, SCHAR_MIN);
  ClassByte->max_value = ExprLitFromInt(ClassByte->file_pos, SCHAR_MAX);
  ClassShort->min_value = ExprLitFromInt(ClassShort->file_pos, SHRT_MIN);
  ClassShort->max_value = ExprLitFromInt(ClassShort->file_pos, SHRT_MAX);
  ClassInt->min_value = ExprLitFromInt(ClassInt->file_pos, INT_MIN);
  ClassInt->max_value = ExprLitFromInt(ClassInt->file_pos, INT_MAX);
  ClassUByte->min_value = ExprLitFromInt(ClassByte->file_pos, 0);
  ClassUByte->max_value = ExprLitFromInt(ClassByte->file_pos, UCHAR_MAX);
  ClassUShort->min_value = ExprLitFromInt(ClassShort->file_pos, 0);
  ClassUShort->max_value = ExprLitFromInt(ClassShort->file_pos, USHRT_MAX);
  ClassUInt->min_value = ExprLitFromInt(ClassInt->file_pos, 0);
  ClassUInt->max_value = ExprLitFromInt(ClassInt->file_pos, UINT_MAX);
  ClassString->min_value = ExprLitFromString(ClassString->file_pos,
    UStringEmpty(), sv);
  ClassString->max_value = ExprLitFromString(ClassString->file_pos,
    UStringMax(), sv);

  /* case types */
  ClassByte->case_type = CASE_TYPE_INTEGRAL;
  ClassShort->case_type = CASE_TYPE_INTEGRAL;
  ClassInt->case_type = CASE_TYPE_INTEGRAL;
  ClassUByte->case_type = CASE_TYPE_INTEGRAL;
  ClassUShort->case_type = CASE_TYPE_INTEGRAL;
  ClassUInt->case_type = CASE_TYPE_INTEGRAL;
  ClassString->case_type = CASE_TYPE_STRING;

  /* all done OK */
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "validate interface types"                                     */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  TYPE_VARS ClassVars(CLASS c)                                             */
/*                                                                           */
/*  Return the type variables of class c.                                    */
/*                                                                           */
/*****************************************************************************/

TYPE_VARS ClassVars(CLASS c)
{
  return c->type_vars;
}


/*****************************************************************************/
/*                                                                           */
/*  void ClassRegisterCoerceResultType(CLASS c, TYPE type, FEFN_FEATURE cfv) */
/*                                                                           */
/*  Register type as the result type of coercion feature view cfv in c.      */
/*                                                                           */
/*****************************************************************************/

void ClassRegisterCoerceResultType(CLASS c, TYPE type, FEFN_FEATURE cfv)
{
  ArrayAddLast(c->coerce_types, type);
  ArrayAddLast(c->coerce_feature_views, cfv);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ClassCheckBuiltinInheritance(CLASS c, CLASS parent_c)            */
/*                                                                           */
/*  Classes c and parent_c are builtin, and c inherits parent_c.  Check      */
/*  that this particular case of inheritance between builtin classes         */
/*  is known to the compiler.                                                */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ClassCheckBuiltinInheritance(CLASS c, CLASS parent_c)
{
  if( c == ClassString && parent_c == ClassArray )
    return TRUE;
  return FALSE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ClassValidateInterfaceTypes1(CLASS c, CONTEXT cxt)               */
/*                                                                           */
/*  Carry out the first stage of the setup of class c:  create the           */
/*  inherit type and check it for level 1 validity.  Also create the         */
/*  parent set.                                                              */
/*                                                                           */
/*  The context must contain two layers, the usual outer class views layer   */
/*  plus an empty layer into which this function can insert things.  This    */
/*  function does not remove or even clean up this layer on exit.            */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ClassValidateInterfaceTypes1(CLASS c, CONTEXT cxt)
{
  CLASS c2;  int i;  TYPE coerce_type;  FEFN_FEATURE coerce_feature_view;
  BOOLEAN is_public;
  if( DEBUG6 )
    fprintf(stderr, "[ ClassValidateInterfaceTypes1(%s, cxt)\n",
      NameShow(ClassName(c, cxt)));

  /* instantiate type_vars and class_type */
  is_public = !ClassViewIsPrivate(c->orig_cv);
  if( !TypeVarsBeginLevelOneValid(c->type_vars, cxt, is_public) )
    db_return(DEBUG6, "ClassValidateInterfaceTypes1 (1)", FALSE);

  /* sort out inherit type */
  if( c->inherit_type == NULL )
  {
    /* inheriting from nothing means inheriting from "object", or  */
    /* from nothing if this class itself is "object" or is builtin or enum */
    assert(ClassObject != NULL);
    if( c != ClassObject && c->ctype == TAGGED )
    {
      /* set c->inherit_type to a TYPE of the "object" class */
      c->inherit_type = ClassType(ClassObject);
    }
  }
  else
  {
    /* instantiate c->inherit_type and check suitable  */
    if( !TypeLevelOneValid(c->inherit_type, cxt, is_public) ||
	!TypeCheckInheritance(c->inherit_type, c, FALSE, cxt) )
      db_return(DEBUG6, "ClassValidateInterfaceTypes1 (3)", FALSE);
  }

  /* sort out parent and child classes */
  c->inherit_parents = TypeClasses(c->inherit_type);
  ArrayForEach(c->inherit_parents, c2)
    ArrayAddLast(c2->inherit_children, c);

  /* do disjoint set merges between child and parents */
  ArrayForEach(c->inherit_parents, c2)
    ClassDSMerge(c, c2);

  /* add inherit type to parent_set */
  ArrayInit(&c->parent_set);
  if( c->inherit_type != NULL )
    TypeAddITypes(c->parent_set, TypeCopy(c->inherit_type), 0, NULL);

  /* add coercion result types to parent set */
  for( i = 0;  i < ArraySize(c->coerce_types);  i++ )
  {
    coerce_type = TypeCopy(ArrayGet(c->coerce_types, i));
    coerce_feature_view = ArrayGet(c->coerce_feature_views, i);
    if( !TypeLevelOneValid(coerce_type, cxt, is_public) ||
	!TypeCheckInheritance(coerce_type, c, TRUE, cxt) )
      db_return(DEBUG6, "ClassValidateInterfaceTypes1 (5)", FALSE);
    TypeAddITypes(c->parent_set, coerce_type, 0,
      CoercionMake(coerce_feature_view));
  }

  /* all OK */
  if( DEBUG6 )
    fprintf(stderr, "] ClassValidateInterfaceTypes1 retn; parent_set = %s\n",
      TypeITypesShow(c->parent_set, cxt));
  db_return(DEBUG6, "ClassValidateInterfaceTypes1 (3)", TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ClassValidateInterfaceTypes2(CLASS c, CONTEXT cxt)               */
/*                                                                           */
/*  Carry out Stage 2 of the instantiation of c:  build its ancestor set.    */
/*                                                                           */
/*  The context cxt is used here only for error messages, and will not       */
/*  change.                                                                  */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ClassValidateInterfaceTypes2(CLASS c, CONTEXT cxt)
{
  return TypeBuildAncestorSet(c, &c->ancestor_set, cxt);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ClassValidateInterfaceTypes3(CLASS c, CONTEXT cxt)               */
/*                                                                           */
/*  Carry out Stage 3 of the instantiation of c:  topological sort.          */
/*  For an original class, or rather for any class for which the job has     */
/*  not yet been done, this consists of assigning eq_class_id and            */
/*  eq_class_offset numbers to the class, as well as eq_class and sort_key,  */
/*  and sorting that class's ancestor set.                                   */
/*                                                                           */
/*  The context cxt is used here only for debugging.  If there were any      */
/*  error messages it would be used for those too.                           */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ClassValidateInterfaceTypes3(CLASS c, CONTEXT cxt)
{
  CLASS anc_c;
  static int eq_class_id = 0;
  int eq_class_offset;
  ITYPE itype, junk;
  if( DEBUG5 )
    fprintf(stderr, "[ ClassValidateInterfaceTypes3(%s)\n",
      NameShow(ClassName(c, cxt)));
  if( c->eq_class == NULL )
  {
    if( DEBUG5 )
      fprintf(stderr, "  %s equivalence class starting now\n",
	NameShow(ClassName(c, cxt)));

    /* initialize a new equivalence class */
    ArrayInit(&c->eq_class);
    ArrayAddLast(c->eq_class, c);

    /* classify each ancestor as equivalent or not, and act accordingly */
    ArrayForEach(c->ancestor_set, itype)
    {
      anc_c = TypeITypeClass(itype);
      if( anc_c == c )
      {
	/* do nothing about oneself */
      }
      else if( TypeAncestorsContainIType(anc_c->ancestor_set, c, &junk) )
      {
	if( DEBUG5 )
	  fprintf(stderr, "  %s has equivalent ancestor %s\n",
	    NameShow(ClassName(c, cxt)), NameShow(ClassName(anc_c, cxt)));

	/* add anc_c to this new equivalence class */
	assert(anc_c->eq_class == NULL);
	anc_c->eq_class = c->eq_class;
	ArrayAddLast(c->eq_class, anc_c);
      }
      else
      {
	if( DEBUG5 )
	  fprintf(stderr, "  %s has inequivalent ancestor %s\n",
	    NameShow(ClassName(c, cxt)), NameShow(ClassName(anc_c, cxt)));

	/* ensure this non-equivalent ancestor is numbered first */
	if( !ClassValidateInterfaceTypes3(anc_c, cxt) )
	  db_return(DEBUG5, "ClassValidateInterfaceTypes3", FALSE);
      }
    }

    /* now all ancestors are numbered, so assign class ids and offsets here */
    eq_class_id++;
    eq_class_offset = 0;
    assert(ArraySize(c->eq_class) < MAX_EQ_CLASS_OFFSET);
    ArrayForEach(c->eq_class, anc_c)
    {
      anc_c->eq_class_id = eq_class_id;
      anc_c->eq_class_offset = eq_class_offset;
      anc_c->sort_key = eq_class_id * MAX_EQ_CLASS_OFFSET + eq_class_offset;
      eq_class_offset++;
    }

    /* and finally sort the ancestors by sort key */
    ArrayForEach(c->eq_class, anc_c)
      TypeITypeSort(anc_c->ancestor_set);

  }
  db_return(DEBUG5, "ClassValidateInterfaceTypes3", TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ClassValidateInterfaceTypes4(CLASS c, CONTEXT cxt)               */
/*                                                                           */
/*  Carry out the fourth stage of instantiating class c, which is to         */
/*  establish Level 3 validity of all its types.                             */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ClassValidateInterfaceTypes4(CLASS c, CONTEXT cxt)
{
  /* check constraints within the class's type variable constraints */
  if( !TypeVarsLevelThreeValid(c->type_vars, cxt) )
    return FALSE;

  /* check constraints within the inherit type */
  if( c->inherit_type != NULL )
    if( !TypeLevelThreeValid(c->inherit_type, cxt) )
      return FALSE;

  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  int ClassSortKey(CLASS c)                                                */
/*                                                                           */
/*  Return c's inheritance sort key.                                         */
/*                                                                           */
/*****************************************************************************/

int ClassSortKey(CLASS c)
{
  return c->sort_key;
}


/*****************************************************************************/
/*                                                                           */
/*  ITYPES ClassParentSet(CLASS c)                                           */
/*                                                                           */
/*  Return the parent set of class c.                                        */
/*                                                                           */
/*****************************************************************************/

ITYPES ClassParentSet(CLASS c)
{
  assert(c->parent_set != NULL);
  return c->parent_set;
}


/*****************************************************************************/
/*                                                                           */
/*  ITYPES ClassAncestorSet(CLASS c)                                         */
/*                                                                           */
/*  Return the ancestor set of class c.                                      */
/*                                                                           */
/*****************************************************************************/

ITYPES ClassAncestorSet(CLASS c)
{
  return c->ancestor_set;
}


/*****************************************************************************/
/*                                                                           */
/*  int ClassEquivalenceId(CLASS c)                                          */
/*                                                                           */
/*  Return the equivalence class id of c.                                    */
/*                                                                           */
/*****************************************************************************/

int ClassEquivalenceId(CLASS c)
{
  return c->eq_class_id;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ClassIsEnum(CLASS c)                                             */
/*                                                                           */
/*  Return TRUE if c is an enumerated class.                                 */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ClassIsEnum(CLASS c)
{
  return c->ctype == ENUMERATED;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ClassIsAbstract(CLASS c)                                         */
/*                                                                           */
/*  Return TRUE if c is an abstract class (contains one or more abstract     */
/*  features).  Naturally this is only correct after this has been worked    */
/*  out, i.e. after Stage 6, which brings every feature into the class.      */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ClassIsAbstract(CLASS c)
{
  return c->is_abstract;
}


/*****************************************************************************/
/*                                                                           */
/*  TYPE ClassType(CLASS c)                                                  */
/*                                                                           */
/*  Return the type of this class.                                           */
/*                                                                           */
/*****************************************************************************/

TYPE ClassType(CLASS c)
{
  return c->class_type;
}


/*****************************************************************************/
/*                                                                           */
/*  void ClassDebugTypes(CLASS c, CONTEXT cxt, FILE *fp, int print_style)    */
/*                                                                           */
/*  Debug print of class types and parent sets etc. in given context.        */
/*                                                                           */
/*****************************************************************************/

void ClassDebugTypes(CLASS c, CONTEXT cxt, FILE *fp, int print_style)
{
  CLASS xc;

  /* header */
  fprintf(fp, "[ %s:", NameShow(ClassName(c, cxt)));
  begin_indent;

  /* equivalence class */
  if( c->eq_class != NULL )
  {
    next_line;
    fprintf(fp, "numbered %d.%d (%d) with equivalence class {",
      c->eq_class_id, c->eq_class_offset, c->sort_key);
    ArrayForEach(c->eq_class, xc)
    {
      if( xc != ArrayFirst(c->eq_class) )
	fprintf(fp, ", ");
      fprintf(fp, "%s", NameShow(ClassName(xc, cxt)));
    }
    fprintf(fp, "}");
  }

  /* inherit_type */
  if( c->inherit_type != NULL )
  {
    next_line;
    fprintf(fp, "inherit_type: %s %s", TypeVarsShow(c->type_vars, cxt),
      TypeShow(c->inherit_type, cxt));
  }

  /* class type */
  if( c->class_type != NULL )
  {
    next_line;
    fprintf(fp, "class_type: %s %s", TypeVarsShow(c->type_vars, cxt),
      TypeShow(c->class_type, cxt));
  }

  /* parent set */
  if( c->parent_set != NULL )
  {
    next_line;
    fprintf(fp, "parent_set: %s %s", TypeVarsShow(c->type_vars, cxt),
      TypeITypesShow(c->parent_set, cxt));
  }

  /* ancestor set */
  if( c->ancestor_set != NULL )
  {
    next_line;
    fprintf(fp, "ancestor_set: %s %s", TypeVarsShow(c->type_vars, cxt),
      TypeITypesShow(c->ancestor_set, cxt));
  }

  /* footer */
  end_indent;
  next_line;
  fprintf(fp, "]");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "creation and invariants"                                      */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  BEFN_CREATION ClassBEFnCreation(CLASS c)                                 */
/*                                                                           */
/*  Return the creation function of class c.                                 */
/*                                                                           */
/*****************************************************************************/

BEFN_CREATION ClassBEFnCreation(CLASS c)
{
  assert(c->creation_fn != NULL);
  return c->creation_fn;
}


/*****************************************************************************/
/*                                                                           */
/*  BEFN_CLASS_INIT ClassBEFnClassInit(CLASS c)                              */
/*                                                                           */
/*  Return the initialzation function of class c.                            */
/*                                                                           */
/*****************************************************************************/

BEFN_CLASS_INIT ClassBEFnClassInit(CLASS c)
{
  assert(c->befn_class_init != NULL);
  return c->befn_class_init;
}


/*****************************************************************************/
/*                                                                           */
/*  BEFN_ENUM_INIT ClassBEFnEnumInit(CLASS c)                                */
/*                                                                           */
/*  Return the all_enumerated initialzation function of class c.             */
/*                                                                           */
/*****************************************************************************/

BEFN_ENUM_INIT ClassBEFnEnumInit(CLASS c)
{
  assert(c->befn_enum_init != NULL);
  return c->befn_enum_init;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ClassFindInitializationOrder(CLASS c, BEFN_SYSTEM_INIT fun)      */
/*                                                                           */
/*  Report c to fun for initialization, but only if non-generic, and only    */
/*  after all classes that must be initialized before this one have been     */
/*  reported.                                                                */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ClassFindInitializationOrder(CLASS c, BEFN_SYSTEM_INIT fun)
{
  BOOLEAN res, report;
  report = FALSE;
  if( c->befn_class_init != NULL )
  {
    if( DEBUG7 )
      fprintf(stderr, "[ ClassFindInitializationOrder(%s, fun)\n",
	NameShow(ClassViewName(ClassOrigClassView(c))));
    res = BEFnClassInitInitOrder(c->befn_class_init, -1, &report, fun, -1);
    db_return(DEBUG7, "ClassFindInitializationOrder", res);
  }
  else
    return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  void ClassMakeInvariant(CLASS c, FEFN_PARAM orig_self_param)             */
/*                                                                           */
/*  Assuming that c does not already have a class invariant, make one.       */
/*  Its self parameter will be sourced to orig_self_param.                   */
/*                                                                           */
/*****************************************************************************/

void ClassMakeInvariant(CLASS c, FEFN_PARAM orig_self_param)
{
  assert(c->befn_invariant == NULL);
  c->befn_invariant = BEFnInvtNew(c, orig_self_param);
}


/*****************************************************************************/
/*                                                                           */
/*  BEFN_INVARIANT ClassInvariant(CLASS c)                                   */
/*                                                                           */
/*  Return the invariant of c, or NULL if there isn't one.                   */
/*                                                                           */
/*****************************************************************************/

BEFN_INVARIANT ClassInvariant(CLASS c)
{
  return c->befn_invariant;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "features"                                                     */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void ClassAddBEFnFeature(CLASS c, BEFN_FEATURE f, BOOLEAN is_abstract,   */
/*    BOOLEAN original)                                                      */
/*                                                                           */
/*  Add feature f to class c.                                                */
/*                                                                           */
/*  If is_abstract is TRUE, the effective instance of f in c is abstract,    */
/*  rendering c itself abstract; if original is TRUE, f originates in c,     */
/*  that is, it is not inherited from anywhere else.                         */
/*                                                                           */
/*  If f is a creation feature it is necessary to add a parameter to c's     */
/*  creation and clone functions.  However this is done separately.          */
/*                                                                           */
/*****************************************************************************/

void ClassAddBEFnFeature(CLASS c, BEFN_FEATURE f, BOOLEAN is_abstract,
  BOOLEAN original)
{
  switch( BEFnFeatureFType(f) )
  {
    case FEATURE_CREATION:

      assert(!ClassIsBuiltin(c));
      ArrayAddLast(c->cfeatures, f);
      if( original )
	ArrayAddLast(c->orig_cfeatures, f);
      break;


    case FEATURE_NONCREATION:
    case FEATURE_NONCREATION_CLONE:

      if( is_abstract )
	c->is_abstract = TRUE;
      if( original )
	ArrayAddLast(c->orig_nfeatures, f);
      break;


    case FEATURE_PREDEF:

      assert(original);
      BEFnFeatureSetPredefIndex(f, ArraySize(c->orig_predefs));
      ArrayAddLast(c->orig_predefs, f);
      ArrayAddLast(c->all_predefs, f);
      break;


    case FEATURE_PREDEF_ALL_PREDEFINED:

      assert(c->ctype != ENUMERATED);
      assert(c->all_predefined == NULL && c->befn_class_init == NULL);
      assert(c->type_vars == NULL);
      c->all_predefined = f;
      c->befn_class_init = BEFnClassInitMake(c);
      break;


    case FEATURE_PREDEF_ALL_ENUMERATED:

      assert(c->ctype == ENUMERATED);
      assert(c->all_enumerated == NULL && c->befn_enum_init == NULL);
      assert(c->type_vars == NULL);
      c->all_enumerated = f;
      c->all_enum_trie = BEFnFeatureNewEnumTrie(c);
      c->befn_class_init = BEFnClassInitMake(c);
      c->befn_enum_init = BEFnEnumInitMake(c);
      break;


    case FEATURE_PREDEF_LEGAL_CODE:

      assert(c->legal_code == NULL);
      assert(c->type_vars == NULL);
      c->legal_code = f;
      break;


    case FEATURE_PREDEF_WITH_CODE:

      assert(c->with_code == NULL);
      assert(c->type_vars == NULL);
      c->with_code = f;
      break;


    default:

      assert(FALSE);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  BEFN_FEATURE ClassAllPredefinedBEFnFeature(CLASS c)                      */
/*                                                                           */
/*  Return the all_predefined feature of c, which must exist.                */
/*                                                                           */
/*****************************************************************************/

BEFN_FEATURE ClassAllPredefinedBEFnFeature(CLASS c)
{
  assert(c->all_predefined != NULL);
  return c->all_predefined;
}


/*****************************************************************************/
/*                                                                           */
/*  BEFN_FEATURE ClassAllEnumeratedBEFnFeature(CLASS c)                      */
/*                                                                           */
/*  Return the all_enumerated feature of c, which must exist.                */
/*                                                                           */
/*****************************************************************************/

BEFN_FEATURE ClassAllEnumeratedBEFnFeature(CLASS c)
{
  assert(c->all_enumerated != NULL);
  return c->all_enumerated;
}


/*****************************************************************************/
/*                                                                           */
/*  BEFN_FEATURE ClassAllEnumTrieBEFnFeature(CLASS c)                        */
/*                                                                           */
/*  Return the all_enum_trie feature of c, which must exist.                 */
/*                                                                           */
/*****************************************************************************/

BEFN_FEATURE ClassAllEnumTrieBEFnFeature(CLASS c)
{
  assert(c->all_enum_trie != NULL);
  return c->all_enum_trie;
}


/*****************************************************************************/
/*                                                                           */
/*  void ClassRegisterPredefinedObjects(CLASS c, ARRAY_BEFN_FEATURE features)*/
/*                                                                           */
/*  Register features as predefined object features with c.                  */
/*                                                                           */
/*****************************************************************************/

void ClassRegisterPredefinedObjects(CLASS c, ARRAY_BEFN_FEATURE features)
{
  if( features != c->orig_predefs )
    ArrayAppend(c->all_predefs, features);
}


/*****************************************************************************/
/*                                                                           */
/*  void ClassPromotePredefinedObjects(CLASS c)                              */
/*                                                                           */
/*  Promote all the orig_predef features of c to the all_predefs arrays      */
/*  of all its ancestors that have such arrays.                              */
/*                                                                           */
/*****************************************************************************/

void ClassPromotePredefinedObjects(CLASS c)
{
  ITYPE itype;
  if( ArraySize(c->orig_predefs) > 0 )
  {
    /* TypePromotePredefinedObjects(c->ancestor_set, c->orig_predefs); */
    ArrayForEach(c->ancestor_set, itype)
    if( TypeITypeCoercion(itype) == NULL && TypeITypeGenerics(itype) == NULL )
      ClassRegisterPredefinedObjects(TypeITypeClass(itype), c->orig_predefs);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  ARRAY_BEFN_FEATURE ClassPredefs(CLASS c)                                 */
/*                                                                           */
/*  Return the all_predefs array of c, containing all the predefined         */
/*  object features including features from child classes, but not           */
/*  all_predefined, all_enumerated, all_enum_trie, legal_code, or with_code  */
/*  features.                                                                */
/*                                                                           */
/*****************************************************************************/

ARRAY_BEFN_FEATURE ClassPredefs(CLASS c)
{
  return c->all_predefs;
}


/*****************************************************************************/
/*                                                                           */
/*  BEFN_FEATURE ClassLegalCode(CLASS c)                                     */
/*                                                                           */
/*  Return the legal_code feature of c.                                      */
/*                                                                           */
/*****************************************************************************/

BEFN_FEATURE ClassLegalCode(CLASS c)
{
  assert(c->legal_code != NULL);
  return c->legal_code;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "enumerated classes".                                          */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  ISET ClassEnumCodeSetInit(CLASS c)                                       */
/*                                                                           */
/*  Initialize the code set of c to contain the code numbers of all          */
/*  predefined object features in c and in c's descendants, and return       */
/*  that set.                                                                */
/*                                                                           */
/*  Implementation note.  This code ensures that child classes are           */
/*  initialized before parent classes, so that it can take the child         */
/*  classes' sets and union them into the parent's.  It assumes that         */
/*  c->enum_code_set is NULL if and only if the code set of c has not been   */
/*  initialized yet.                                                         */
/*                                                                           */
/*****************************************************************************/

ISET ClassEnumCodeSetInit(CLASS c)
{
  CLASS child_class;  BEFN_FEATURE f;  int code1, code2;
  assert(c->ctype == ENUMERATED);

  /* if code set is non-NULL, we've already done this class */
  if( c->enum_code_set != NULL )
    return c->enum_code_set;

  if( DEBUG15 )
    fprintf(stderr, "[ ClassEnumCodeSetInit(%s)\n",
      NameShow(ClassViewName(ClassOrigClassView(c))));

  /* initialize the code set and accumulate own codes */
  c->enum_code_set = ISetNew();
  ArrayForEach(c->orig_predefs, f)
    if( BEFnFeatureHasCodeNumbers(f, &code1, &code2) )
      ISetInsertInterval(c->enum_code_set, code1, code2);

  /* do all the children and accumulate their code sets */
  ArrayForEach(c->inherit_children, child_class)
    ISetUnion(c->enum_code_set, ClassEnumCodeSetInit(child_class));

  /* all done, return code set */
  if( DEBUG15 )
    fprintf(stderr, "] ClassEnumCodeSetInit(%s) returning %s\n",
      NameShow(ClassViewName(ClassOrigClassView(c))),
      ISetShow(c->enum_code_set));
  return c->enum_code_set;
}


/*****************************************************************************/
/*                                                                           */
/*  void ClassAugmentCodeSet(CLASS c, ISET enum_code_set)                    */
/*                                                                           */
/*  Augment enum_code_set by adding to it the code set of c.                 */
/*                                                                           */
/*****************************************************************************/

/* *** inlined now
void ClassAugmentCodeSet(CLASS c, ISET enum_code_set)
{
  if( c->enum_code_set != enum_code_set )
    ISetUnion(enum_code_set, c->enum_code_set);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void ClassNotifyCodeSet(CLASS c, ISET enum_code_set)                     */
/*                                                                           */
/*  Notify c that the codes in enum_code_set are no longer open to it.       */
/*                                                                           */
/*****************************************************************************/

/* *** inlined now
void ClassNotifyCodeSet(CLASS c, ISET enum_code_set)
{
  if( c->enum_code_set != enum_code_set )
    ISetUnion(c->enum_code_set, enum_code_set);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ClassEnumAssignCodes(CLASS c)                                    */
/*                                                                           */
/*  Assign code numbers to those predefined object features of c that do     */
/*  not already have them, assuming that ClassEnumCodeSetInit has been       */
/*  run by now.  Return TRUE if successful.  The only possible kind of       */
/*  failure is a code number clash.                                          */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ClassEnumAssignCodes(CLASS c)
{
  CLASS child_class, ancestor_class;  BEFN_FEATURE f;  ITYPE itype;
  int code1, code2, max_code, i, code_step;
  assert(c->ctype == ENUMERATED);

  /* if already done, just return */
  if( c->enum_codes_done )
    return TRUE;

  /* ensure that all the child classes are done */
  ArrayForEach(c->inherit_children, child_class)
    if( !ClassEnumAssignCodes(child_class) )
      return FALSE;

  /* add the code sets of c's non-coercion proper ancestors to c's code set */
  ArrayForEach(c->ancestor_set, itype)
    if( TypeITypeCoercion(itype) == NULL  )
    {
      ancestor_class = TypeITypeClass(itype);
      if( ancestor_class != c )
	ISetUnion(c->enum_code_set, ancestor_class->enum_code_set);
    }

  /* assign code numbers to the predefined object features of c needing them */
  ArrayForEach(c->orig_predefs, f)
    if( !BEFnFeatureHasCodeNumbers(f, &code1, &code2) )
    {
      /* find a gap in c->enum_code_set; assign it to f and c->enum_code_set */
      code1 = ISetGap(c->enum_code_set, 1, 1);
      ISetInsertInterval(c->enum_code_set, code1, code1);
      BEFnFeatureSetCodeNumbers(f, code1, code1);
    }

  /* c's and child classes are done, so all_predefs must be done; sort them */
  if( !BEFnFeatureListSortByCode(c->all_predefs, &c->enum_has_range) )
    return FALSE;

  /* notify c's non-coercion ancestors that c's code set is unavailable */
  /* TypeNotifyCodeSet(c->ancestor_set, c->enum_code_set); */
  ArrayForEach(c->ancestor_set, itype)
    if( TypeITypeCoercion(itype) == NULL  )
    {
      ancestor_class = TypeITypeClass(itype);
      if( ancestor_class != c )
	ISetUnion(ancestor_class->enum_code_set, c->enum_code_set);
      /* ClassNotifyCodeSet(TypeITypeClass(itype), c->enum_code_set); */
    }

  /* all done, set trie height and width */
  max_code = ISetMaxElement(c->enum_code_set);
  c->enum_trie_height = max_code < 256 ? 1 : max_code < 256 * 256 ? 2 :
    max_code < 256 * 256 * 256 ? 3 : 4;
  code_step = 1;
  for( i = 1;  i < c->enum_trie_height;  i++ )
    code_step *= 256;
  c->enum_trie_width = max_code/code_step + 1;

  /* set min_value and max_value */
  c->min_value = ExprLitFromInt(c->file_pos, ISetMinElement(c->enum_code_set));
  c->max_value = ExprLitFromInt(c->file_pos, ISetMaxElement(c->enum_code_set));
  c->enum_codes_done = TRUE;
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ClassEnumHasCodeRange(CLASS c)                                   */
/*                                                                           */
/*  Return TRUE if enumerated class c contains a non-trivial code range.     */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ClassEnumHasCodeRange(CLASS c)
{
  return c->enum_has_range;
}


/*****************************************************************************/
/*                                                                           */
/*  int ClassEnumMaxCode(CLASS c)                                            */
/*                                                                           */
/*  Return c's largest code number.                                          */
/*                                                                           */
/*****************************************************************************/

int ClassEnumMaxCode(CLASS c)
{
  assert(c->enum_codes_done);
  if( DEBUG15 )
    fprintf(stderr, "ClassEnumMaxCode(%s) found %s\n",
      NameShow(ClassViewName(ClassOrigClassView(c))),
      ISetShow(c->enum_code_set));
  return ISetMaxElement(c->enum_code_set);
}


/*****************************************************************************/
/*                                                                           */
/*  int ClassEnumTrieHeight(CLASS c)                                         */
/*                                                                           */
/*  Return the height of the trie for enumerated class c.  This function     */
/*  may be called only after ClassEnumAssignCodes is all done.               */
/*                                                                           */
/*****************************************************************************/

int ClassEnumTrieHeight(CLASS c)
{
  assert(c->enum_trie_height > 0);
  return c->enum_trie_height;
}


/*****************************************************************************/
/*                                                                           */
/*  int ClassEnumTrieWidth(CLASS c)                                          */
/*                                                                           */
/*  Return the width of the trie for enumerated class c, that is, the        */
/*  number of elements in its root node.  If a straight array would have     */
/*  less than 256 elements, return that size; otherwise return 256.          */
/*                                                                           */
/*****************************************************************************/

int ClassEnumTrieWidth(CLASS c)
{
  assert(c->enum_trie_width > 0);
  return c->enum_trie_width;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ClassHasNoCreationFeatures(CLASS c)                              */
/*                                                                           */
/*  Return TRUE if c has no creation features, implying that there is        */
/*  only one distinct object of class c.                                     */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ClassHasNoCreationFeatures(CLASS c)
{
  return ArraySize(c->cfeatures) == 0;
}


/*****************************************************************************/
/*                                                                           */
/*  ARRAY_BEFN_FEATURE ClassCreationFeatures(CLASS c)                        */
/*                                                                           */
/*  Return the creation features of class c.                                 */
/*                                                                           */
/*****************************************************************************/

ARRAY_BEFN_FEATURE ClassCreationFeatures(CLASS c)
{
  return c->cfeatures;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "disjoint sets".                                               */
/*                                                                           */
/*  This is just the well-known disjoint sets data structure applied to      */
/*  sets of classes.  We don't both with weights or path compression.        */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  CLASS ClassDSRoot(CLASS c)                                               */
/*                                                                           */
/*  Return the root of the disjoint sets tree containing c.                  */
/*                                                                           */
/*****************************************************************************/

static CLASS ClassDSRoot(CLASS c)
{
  while( c->ds_parent != NULL )
    c = c->ds_parent;
  return c;
}


/*****************************************************************************/
/*                                                                           */
/*  void ClassDSMerge(CLASS c1, CLASS c2)                                    */
/*                                                                           */
/*  Merge the disjoint sets of classes containing c1 and c2, unless either   */
/*  is the "object" class.  This is just the "union" operation of the        */
/*  well-known union-find data structure.  We don't bother with weights      */
/*  or path compression.                                                     */
/*                                                                           */
/*****************************************************************************/

void ClassDSMerge(CLASS c1, CLASS c2)
{
  BOOLEAN doit = FALSE;
  c1 = ClassDSRoot(c1);
  c2 = ClassDSRoot(c2);
  if( ClassObject != NULL )
  {
    /* can use ClassObject to tell whether c1 or c2 is "object" */
    if( c1 != c2 && c1 != ClassObject && c2 != ClassObject )
      doit = TRUE;
  }
  else
  {
    /* must use name of class to tell whether class is "object" */
    static USTRING object_name = NULL;
    if( object_name == NULL )
      object_name = AStringToUString("object");
    if( c1 != c2 &&
	!UStringEqual(NameKey(ClassViewName(c1->orig_cv)), object_name) &&
	!UStringEqual(NameKey(ClassViewName(c2->orig_cv)), object_name) )
      doit = TRUE;
  }

  if( doit )
  {
    /* merging is required; forward c2's root to c1's root */
    c2->ds_parent = c1;

    /* merge clone features, if present */
    if( c1->ds_clone_feature == NULL )
      c1->ds_clone_feature = c2->ds_clone_feature;
    else if( c2->ds_clone_feature != NULL )
    {
      if( DEBUG16 )
	fprintf(stderr, "  ClassDSMerge(%s, %s) calling\n",
	  NameShow(ClassViewName(ClassOrigClassView(c1))),
	  NameShow(ClassViewName(ClassOrigClassView(c2))));
      BEFnFeatureMerge(c1->ds_clone_feature, c2->ds_clone_feature);
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  BEFN_FEATURE ClassDSCloneBEFnFeature(CLASS c, FEFN_FEATURE fv,           */
/*    FEFN_PARAM fv_self)                                                    */
/*                                                                           */
/*  Return the clone feature of the disjoint set partition containing c,     */
/*  or if NULL, make one based on fv (whose self parameter is fv_self),      */
/*  initially with just a self parameter.                                    */
/*                                                                           */
/*****************************************************************************/

BEFN_FEATURE ClassDSCloneBEFnFeature(CLASS c, FEFN_FEATURE fv,
  FEFN_PARAM fv_self)
{
  BEFN_PARAM self_param;
  CLASS root = ClassDSRoot(c);
  if( root->ds_clone_feature == NULL )
  {
    if( DEBUG17 )
      fprintf(stderr, "[ ClassDSCloneBEFnFeature(%s, %s) making new befn\n",
	NameShow(ClassViewName(c->orig_cv)), NameShow(FEFnFeatureName(fv)));
    root->ds_clone_feature = BEFnFeatureMake(root, fv,
      FEATURE_NONCREATION_CLONE, NO_CODE_NUM, NO_CODE_NUM);
    self_param = BEFnParamNew(PARAM_SELF, fv_self, NULL);
    BEFnAddParameter((BEFN) root->ds_clone_feature, self_param);
    if( DEBUG17 )
      fprintf(stderr, "] ClassDSCloneBEFnFeature returning new befn (%s)\n",
	BEFnParamListShow(BEFnParameters((BEFN) root->ds_clone_feature)));
  }
  return root->ds_clone_feature;
}


/*****************************************************************************/
/*                                                                           */
/*  void ClassDSSetCloneBEFnFeature(CLASS c, BEFN_FEATURE f)                 */
/*                                                                           */
/*  Set the clone feature of the disjoint set partition containing c.        */
/*                                                                           */
/*****************************************************************************/

/* *** done within ClassDSCloneBEFnFeature now
void ClassDSSetCloneBEFnFeature(CLASS c, BEFN_FEATURE f)
{
  ClassDSRoot(c)->ds_clone_feature = f;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void ClassDSFinalizeComponent(CLASS c)                                   */
/*                                                                           */
/*  This function should be called after all merges are complete.  Its       */
/*  purpose is to finalize the disjoint sets structure, including            */
/*  setting ds_component in each class c.                                    */
/*                                                                           */
/*****************************************************************************/

void ClassDSFinalizeComponent(CLASS c)
{
  CLASS cr;
  for( cr = c;  cr->ds_parent != NULL;  cr = cr->ds_parent);
  if( cr->ds_component == NULL )
    ArrayInit(&cr->ds_component);
  ArrayAddLast(cr->ds_component, c);
  c->ds_component = cr->ds_component;
}


/*****************************************************************************/
/*                                                                           */
/*  ARRAY_CLASS ClassDSComponent(CLASS c)                                    */
/*                                                                           */
/*  Return the set of all classes in c's component.  This can only be        */
/*  called after it has been worked out.                                     */
/*                                                                           */
/*****************************************************************************/

ARRAY_CLASS ClassDSComponent(CLASS c)
{
  assert(c->ds_component != NULL);
  return c->ds_component;
}


/*****************************************************************************/
/*                                                                           */
/*  void ClassDSDebug(CLASS c, FILE *fp)                                     */
/*                                                                           */
/*  Debug print of c's component.                                            */
/*                                                                           */
/*****************************************************************************/

void ClassDSDebug(CLASS c, FILE *fp)
{
  CLASS c1;  int i;
  if( c->ds_component == NULL )
    fprintf(fp, "  %s: NULL\n", NameShow(ClassViewName(c->orig_cv)));
  else
  {
    fprintf(fp, "  %s: {", NameShow(ClassViewName(c->orig_cv)));
    for( i = 0;  i < ArraySize(c->ds_component);  i++ )
    {
      c1 = ArrayGet(c->ds_component, i);
      fprintf(fp, "%s%s", i > 0 ? ", " : "",
	NameShow(ClassViewName(c1->orig_cv)));
    }
    fprintf(fp, "}\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "code generation, phase 1 (initialization)"                    */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void ClassCodeGenInitAll(ARRAY_CLASS classes, CODEGEN be)                */
/*                                                                           */
/*  Initialize every class in the system (all in classes) for code           */
/*  generation utilizing backend be.                                         */
/*                                                                           */
/*  Implementation note.  This is somewhat complex because many of the       */
/*  builtin classes require specific values, mainly for their be_type        */
/*  field but occasionally also for their name.                              */
/*                                                                           */
/*****************************************************************************/

void ClassCodeGenInitAll(ARRAY_CLASS classes, CODEGEN be)
{
  CLASS c; int i, curr_tag_value;
  USTRING orig_name, tag, content_name;
  USTRING tag_str, swizzle_str, unswizzle_str, content_str, init_all_str;

  /* initialize useful strings */
  tag_str = AStringToUString("_tag");
  swizzle_str = AStringToUString("swizzle");
  unswizzle_str = AStringToUString("unswizzle");
  content_str = AStringToUString("_content");
  init_all_str = AStringToUString("init_all_enumerated");

  /* initialize backend type fields of all (known) builtin untagged classes */
  ClassVoid->be_type = be->void_type;
  ClassBool->be_type = be->TypeMakeEnumerated(AStringToUString("bool"), 8, be);
  ClassChar->be_type = be->TypeMakeEnumerated(AStringToUString("uchar"),32,be);
  ClassByte->be_type = be->char_type;
  ClassShort->be_type = be->short_type;
  ClassInt->be_type = be->int_type;
  ClassUByte->be_type = be->uchar_type;
  ClassUShort->be_type = be->ushort_type;
  ClassUInt->be_type = be->uint_type;
  ClassReal->be_type = be->float_type;
  for( i = 1;  i <= MAX_FUN_PARAMS;  i++ )
  {
    c = ClassFunRep[i];
    orig_name = NameRep(ClassViewName(c->orig_cv));
    c->be_type = be->TypeMakeFunctionType(orig_name, i);
  }

  /* generic initialization depending on class type */
  curr_tag_value = 0;
  ArrayForEach(classes, c) switch( c->ctype )
  {
    case BUILTIN_UNTAGGED:

      if( c->be_type == NULL )
      {
	/* should have been initialized above; must be an unknown builtin */
	fprintf(stderr, "%s: unknown builtin class %s\n",
	  FilePosShow(c->file_pos), NameShow(ClassViewName(c->orig_cv)));
	exit(1);
      }
      c->be_content_type = NULL;
      c->be_type_tag = NULL;
      c->swizzle_width = 0;
      c->swizzle_offset = -1;
      c->swizzle_fn = NULL;
      c->unswizzle_fn = NULL;
      c->layout_done = TRUE;
      c->record_width = 0;
      c->layout_set = NULL;
      break;


    case BUILTIN_TAGGED:
    case TAGGED:

      orig_name = NameRep(ClassViewName(c->orig_cv));
      c->be_type = be->TypeMakeObject(orig_name);
      if( !c->is_abstract )
      {
	tag = UStringCat(orig_name, tag_str);
	c->be_type_tag = be->ConstMakeInt(NULL, tag, ++curr_tag_value, TRUE);
      }
      else
	c->be_type_tag = NULL;
      c->swizzle_width = -1;
      c->swizzle_offset = -1;
      c->swizzle_fn = be->FunctionMake2(orig_name, swizzle_str, TRUE, TRUE);
      c->unswizzle_fn = be->FunctionMake2(orig_name, unswizzle_str, TRUE, TRUE);
      c->layout_done = FALSE;
      c->record_width = 0;
      c->layout_set = ISetInterval(0, be->TypeFieldWidth(be->int_type) + 
	be->TypeFieldWidth(be->ushort_type) - 1);
      break;


    case ENUMERATED:

      orig_name = NameRep(ClassViewName(c->orig_cv));
      content_name = UStringCat(orig_name, content_str);
      i = ISetMaxElement(c->enum_code_set);
      if( c->be_type == NULL ) /* i.e. if not already named */
	c->be_type = be->TypeMakeEnumerated(orig_name,
	  (i <= UCHAR_MAX ? 8 : i <= USHRT_MAX ? 16 : 32), be);
      c->be_content_type = be->TypeMakeObject(content_name);
      if( !c->is_abstract )
      {
	tag = UStringCat(orig_name, tag_str);
	c->be_type_tag = be->ConstMakeInt(NULL, tag, ++curr_tag_value, TRUE);
      }
      else
	c->be_type_tag = NULL;
      c->swizzle_width = -1;
      c->swizzle_offset = -1;
      c->swizzle_fn = be->FunctionMake2(content_name, swizzle_str, TRUE, TRUE);
      c->unswizzle_fn = be->FunctionMake2(content_name,unswizzle_str,TRUE,TRUE);
      c->layout_done = FALSE;
      c->record_width = 0;
      c->layout_set = ISetInterval(0, be->TypeFieldWidth(be->int_type) + 
	be->TypeFieldWidth(be->ushort_type) - 1);
      break;


    default:

      assert(FALSE);
      break;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void PropagateLayoutSetToDescendants(CLASS class, ISET layout_set)       */
/*                                                                           */
/*  Propagate layout_set to all proper descendants of class.                 */
/*                                                                           */
/*****************************************************************************/

static void PropagateLayoutSetToDescendants(CLASS class, ISET layout_set)
{
  CLASS c;
  ArrayForEach(class->inherit_children, c)
  {
    ISetUnion(c->layout_set, layout_set);
    PropagateLayoutSetToDescendants(c, layout_set);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void AccumulateDescendantLayoutSets(CLASS class, ISET layout_set)        */
/*                                                                           */
/*  Accumulate in layout_set the layout sets of all proper descendants       */
/*  of class.                                                                */
/*                                                                           */
/*****************************************************************************/

static void AccumulateDescendantLayoutSets(CLASS class, ISET layout_set)
{
  CLASS c;
  ArrayForEach(class->inherit_children, c)
  {
    ISetUnion(layout_set, c->layout_set);
    AccumulateDescendantLayoutSets(c, layout_set);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void AccumulateAncestorLayoutSets(CLASS class, ISET layout_set)          */
/*                                                                           */
/*  Accumulate in layout_set the layout sets of all proper ancestors         */
/*  of class.                                                                */
/*                                                                           */
/*****************************************************************************/

static void AccumulateAncestorLayoutSets(CLASS class, ISET layout_set)
{
  CLASS c;
  ArrayForEach(class->inherit_parents, c)
  {
    ISetUnion(layout_set, c->layout_set);
    AccumulateAncestorLayoutSets(c, layout_set);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void InsertGap(int current_offset, int desired_offset, CODEGEN be)       */
/*                                                                           */
/*  Insert zero or more bit fields into be to move us from current_offset    */
/*  (where we are now) to desired_offset.  Make sure that none of the        */
/*  inserted bit fields straddles a word boundary.                           */
/*                                                                           */
/*****************************************************************************/
#define WORD_SIZE 32

static void InsertGap(int current_offset, int desired_offset, CODEGEN be)
{
  int rest_of_word;

  /* calculate number of bits unused in the current word */
  if( current_offset % WORD_SIZE == 0 )
    rest_of_word = 0;
  else
    rest_of_word = WORD_SIZE - current_offset % WORD_SIZE;

  /* if word boundaries are involved, get across all of those */
  if( desired_offset - current_offset >= rest_of_word )
  {
    /* get to the end of the current word */
    if( rest_of_word > 0 )
    {
      be->StructGap(rest_of_word);
      current_offset += rest_of_word;
    }

    /* now as many word-sized chunks as we need */
    while( desired_offset - current_offset >= WORD_SIZE )
    {
      be->StructGap(WORD_SIZE);
      current_offset += WORD_SIZE;
    }
  }

  /* now finish off with a gap within one word, if needed */
  if( desired_offset - current_offset > 0 )
    be->StructGap(desired_offset - current_offset);
}


/*****************************************************************************/
/*                                                                           */
/*  void ClassCodeGenInitCreationFeatures(CLASS class, CODEGEN be)           */
/*                                                                           */
/*  Unless already done, do the record layout for class, but don't print     */
/*  anything at this point.                                                  */
/*                                                                           */
/*  When this routine is called, class->layout_set will contain the layout   */
/*  of all features of this class that have already been laid out.           */
/*                                                                           */
/*****************************************************************************/

void ClassCodeGenInitCreationFeatures(CLASS class, CODEGEN be)
{
  CLASS c;  BEFN_FEATURE f;

  if( !class->layout_done )
  {
    if( DEBUG9 )
      fprintf(stderr, "[ ClassCreationFeaturesInit(%s)\n",
	NameShow(ClassViewName(c->orig_cv)));

    /* make sure all the parents are done first */
    ArrayForEach(class->inherit_parents, c)
      ClassCodeGenInitCreationFeatures(c, be);

    /* add to class->layout_set all ancestor and descendant layouts */
    AccumulateAncestorLayoutSets(class, class->layout_set);
    AccumulateDescendantLayoutSets(class, class->layout_set);

    if( DEBUG9 )
      fprintf(stderr, "  accumulated layout_set %s\n",
	ISetShow(class->layout_set));

    /* lay out each creation feature in turn */
    ArrayForEach(class->cfeatures, f)
      BEFnFeatureCodeGenInitCreation(f, class->layout_set, be);

    /* propagate class layout set to all descendants */
    PropagateLayoutSetToDescendants(class, class->layout_set);

    /* sort the creation features of class */
    ArraySortUnique(class->cfeatures, &BEFnFeatureOffsetCmp);

    /* record that layout is now done */
    class->layout_done = TRUE;
    if( DEBUG9 )
      fprintf(stderr, "] ClassCreationFeaturesInit returning\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void ClassCodeGenInitFunctions(CLASS c, CODEGEN be)                      */
/*                                                                           */
/*  Initialize for code generation the orginal noncreation features and      */
/*  other functions of c.                                                    */
/*                                                                           */
/*****************************************************************************/

void ClassCodeGenInitFunctions(CLASS c, CODEGEN be)
{
  BEFN_FEATURE f;  BEFN_CREDFT befn_credft;
  if( c->creation_fn != NULL )
    BEFnFinalize((BEFN) c->creation_fn, be);
  if( c->befn_class_init != NULL )
    BEFnFinalize((BEFN) c->befn_class_init, be);
  if( c->befn_enum_init != NULL )
    BEFnFinalize((BEFN) c->befn_enum_init, be);
  if( c->befn_invariant != NULL )
    BEFnFinalize((BEFN) c->befn_invariant, be);
  if( c->all_predefined != NULL )
    BEFnFinalize((BEFN) c->all_predefined, be);
  if( c->all_enumerated != NULL )
  {
    BEFnFinalize((BEFN) c->all_enumerated, be);
    BEFnFinalize((BEFN) c->all_enum_trie, be);
    BEFnFinalize((BEFN) c->legal_code, be);
    BEFnFinalize((BEFN) c->with_code, be);
  }
  ArrayForEach(c->orig_cfeatures, f)
  {
    befn_credft = BEFnFeatureCreationDefaultValue(f);
    if( befn_credft != NULL )
      BEFnFinalize((BEFN) befn_credft, be);
  }
  ArrayForEach(c->orig_nfeatures, f)
    if( BEFnFeatureFType(f) != FEATURE_NONCREATION_CLONE )
      BEFnFinalize((BEFN) f, be);
  if( c->ds_parent == NULL && c->ds_clone_feature != NULL )
    BEFnFinalize((BEFN) c->ds_clone_feature, be);
  ArrayForEach(c->orig_predefs, f)
    BEFnFinalize((BEFN) f, be);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "code generation, phase 2 (actual code generation)"            */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_TYPE ClassBEType(CLASS c)                                        */
/*                                                                           */
/*  Return the backend name of class c, assuming it has been set.            */
/*                                                                           */
/*****************************************************************************/

CODEGEN_TYPE ClassBEType(CLASS c)
{
  assert(c->be_type != NULL);
  return c->be_type;
}


/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_TYPE ClassBEContentType(CLASS c)                                 */
/*                                                                           */
/*  Return the backend name of the content record class c, assuming that c   */
/*  is an enumerated type and this has been set.                             */
/*                                                                           */
/*****************************************************************************/

CODEGEN_TYPE ClassBEContentType(CLASS c)
{
  assert(ClassIsEnum(c));
  assert(c->be_content_type != NULL);
  return c->be_content_type;
}


/*****************************************************************************/
/*                                                                           */
/*  void ClassEnumCodeGenTrieAccess(CLASS c, CODEGEN_OBJ be_obj, CODEGEN be) */
/*                                                                           */
/*  Here be_obj is a variable of enumerated type c.  Generate the code for   */
/*  accessing the content record of c, i.e.                                  */
/*                                                                           */
/*    ((c_content_type) trie*(c_enum_trie, be_obj))                          */
/*                                                                           */
/*  where * is 1, 2, 3, or 4 depending on how large the enumeration is.      */
/*                                                                           */
/*****************************************************************************/

void ClassEnumCodeGenTrieAccess(CLASS c, CODEGEN_OBJ be_obj, CODEGEN be)
{
  CODEGEN_OBJ all_enum_trie, trie_be_obj;
  assert(ClassIsEnum(c));
  all_enum_trie = BEFnBEObj((BEFN) c->all_enum_trie);
  trie_be_obj = NPBack_Trie[ClassEnumTrieHeight(c)-1];
  Cast(ClassBEContentType(c), ClassBEType(ClassArray),
    Call2(trie_be_obj, Var(all_enum_trie), Var(be_obj)));
}


/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_OBJ ClassBEFnCreation2(CLASS c)                                  */
/*                                                                           */
/*  Return the C name of the second creation function of class c.  This is   */
/*  only defined in the array and string classes, which have a two-stage     */
/*  creation procedure.                                                      */
/*                                                                           */
/*****************************************************************************/

/* ***
CODEGEN_OBJ ClassBEFnCreation2(CLASS c)
{
  assert(c->be_obj2 != NULL);
  return c->be_obj2;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ClassIsObjRef(CLASS c)                                           */
/*                                                                           */
/*  Return TRUE of objects of this class are references to objects.          */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ClassIsObjRef(CLASS c)
{
  /* return c->is_obj_ref; */
  return c->ctype == BUILTIN_TAGGED || c->ctype == TAGGED;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ClassIsSwizzle(CLASS c)                                          */
/*                                                                           */
/*  Return TRUE if this class needs swizzle bits.                            */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ClassIsSwizzle(CLASS c)
{
  assert(c->swizzle_width != -1);
  return c->swizzle_width > 0;
}


/*****************************************************************************/
/*                                                                           */
/*  int ClassFieldWidth(CLASS c, CODEGEN be)                                 */
/*                                                                           */
/*  Return the number of bits occupied by a variable or record field         */
/*  of type c.                                                               */
/*                                                                           */
/*****************************************************************************/

int ClassFieldWidth(CLASS c, CODEGEN be)
{
  return be->TypeFieldWidth(c->be_type);
}


/*****************************************************************************/
/*                                                                           */
/*  EXPR ClassMin(CLASS c)                                                   */
/*                                                                           */
/*  Return the minimum possible value of this class.                         */
/*                                                                           */
/*****************************************************************************/

EXPR ClassMin(CLASS c)
{
  assert(c->min_value != NULL);
  return c->min_value;
}


/*****************************************************************************/
/*                                                                           */
/*  EXPR ClassMax(CLASS c)                                                   */
/*                                                                           */
/*  Return the maximum possible value of this class.  In the case of string, */
/*  this will be a string whose first character is beyond the allowed        */
/*  range of Unicode.                                                        */
/*                                                                           */
/*****************************************************************************/

EXPR ClassMax(CLASS c)
{
  assert(c->max_value != NULL);
  return c->max_value;
}


/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_OBJ ClassSwizzleFn(CLASS c)                                      */
/*                                                                           */
/*  Return the name of the swizzle function of class c.                      */
/*                                                                           */
/*****************************************************************************/

CODEGEN_OBJ ClassSwizzleFn(CLASS c)
{
  assert(c->swizzle_fn != NULL);
  return c->swizzle_fn;
}


/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_OBJ ClassUnSwizzleFn(CLASS c)                                    */
/*                                                                           */
/*  Return the name of the unswizzle function of class c.                    */
/*                                                                           */
/*****************************************************************************/

CODEGEN_OBJ ClassUnSwizzleFn(CLASS c)
{
  assert(c->unswizzle_fn != NULL);
  return c->unswizzle_fn;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "typedefs and structs"                                         */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void ClassTypeDefPrint(CLASS c, CODEGEN be)                              */
/*                                                                           */
/*  Print the typedef for c onto fp.  If it is an enumerated class, also     */
/*  print the typedef for the corresponding content type.                    */
/*                                                                           */
/*****************************************************************************/

void ClassTypeDefPrint(CLASS c, CODEGEN be)
{
  assert(c->be_type != NULL);
  be->TypeDef(c->be_type);
  if( c->ctype == ENUMERATED )
    be->TypeDef(c->be_content_type);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ClassHasTypeTag(CLASS c)                                         */
/*                                                                           */
/*  Return TRUE if c has a type tag.  This function is only defined after    */
/*  code generation initialization.  The result is TRUE if c is array, or    */
/*  string, or a non-abstract non-builtin class.                             */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ClassHasTypeTag(CLASS c)
{
  return c->be_type_tag != NULL;
}


/*****************************************************************************/
/*                                                                           */
/*  void ClassTypeTagPrint(CLASS c, CODEGEN be)                              */
/*                                                                           */
/*  Print the type tag of class c onto be.  It had better exist.             */
/*                                                                           */
/*****************************************************************************/

void ClassTypeTagPrint(CLASS c, CODEGEN be)
{
  be->ConstDeclare(c->be_type_tag);
}


/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_OBJ ClassTypeTag(CLASS c)                                        */
/*                                                                           */
/*  Return the type tag of class c.  This class must have a type tag and     */
/*  this call must occur after ClassTypeTagPrint.                            */
/*                                                                           */
/*****************************************************************************/

CODEGEN_OBJ ClassTypeTag(CLASS c)
{
  if( DEBUG11 && c->be_type_tag == NULL )
    fprintf(stderr, "ClassTypeTag(%s) failing\n",
      NameShow(ClassViewName(c->orig_cv)));
  assert(c->be_type_tag != NULL);
  return c->be_type_tag;
}


/*****************************************************************************/
/*                                                                           */
/*  void ClassTypeStructPrint(CLASS class, CODEGEN be)                       */
/*                                                                           */
/*  Print class's struct definition, if any, onto fp.                        */
/*                                                                           */
/*****************************************************************************/

void ClassTypeStructPrint(CLASS class, CODEGEN be)
{
  BEFN_FEATURE f;  BOOLEAN swizzle_bits_done;  CODEGEN_TYPE swizzle_type;

  /* various fixed widths and attributes */
  static int sig_width, type_tag_width;
  static CODEGEN_OBJ length_attr;
  static CODEGEN_OBJ elems_attr;

  assert(class->layout_done);
  if( DEBUG1 )
    fprintf(stderr, "[ ClassTypeStructPrint(%s, fp)\n",
      NameShow(ClassViewName(class->orig_cv)));

  /* make various useful static values */
  if( length_attr == NULL )
  {
    sig_width = be->TypeFieldWidth(be->int_type);
    type_tag_width = be->TypeFieldWidth(be->short_type);
    length_attr = be->StructItemMake(AStringToUString("length"), -1);
    elems_attr = be->StructItemMake(AStringToUString("elems[1]"), -1);
  }

  switch( class->ctype )
  {
    case BUILTIN_UNTAGGED:

      /* nothing to do here */
      break;


    case BUILTIN_TAGGED:

      /* array and string classes are the only BUILTIN_TAGGED ones */
      be->StructBegin(class->be_type);
      be->StructItem(NPBack_Signature, be->uint_type);
      be->StructItem(NPBack_Type_Tag, be->ushort_type);
      be->StructItem(NPBack_Swizzle_Bits, be->uchar_type);
      be->StructGap(8);
      be->StructItem(length_attr, be->int_type);
      be->StructItem(elems_attr,
	class == ClassArray ? be->voidp_type : ClassChar->be_type);
      be->StructEnd(class->be_type);
      break;


    case ENUMERATED:
    case TAGGED:

      /* signature, type tag, and optional swizzle bits fields */
      be->StructBegin(class->ctype == TAGGED ?
	class->be_type : class->be_content_type);
      be->StructItem(NPBack_Signature, be->uint_type);
      be->StructItem(NPBack_Type_Tag, be->ushort_type);
      class->record_width = sig_width + type_tag_width;

      /* sort out swizzle width and offset */
      class->swizzle_width = TypeSwizzleSet(class->type_vars);
      if( DEBUG1 )
	fprintf(stderr, "  swizzle width %d\n", class->swizzle_width);
      if( class->swizzle_width > be->TypeFieldWidth(be->int_type) )
      {
	fprintf(stderr, "%s: too many type variable swizzles (%d, max is %d)\n",
	  FilePosShow(class->file_pos), class->swizzle_width, 32);
	exit(1);
      }
      else if( class->swizzle_width > 0 )
      {
	swizzle_type = be->TypeMakeEnumerated(NULL, class->swizzle_width, be);
	class->swizzle_width = be->TypeFieldWidth(swizzle_type);
	class->swizzle_offset = ISetGap(class->layout_set, class->swizzle_width,
	  class->swizzle_width);
      }

      /* ordinary creation features */
      swizzle_bits_done = (class->swizzle_width == 0);
      ArrayForEach(class->cfeatures, f)
      {
	if( DEBUG1 )
	  fprintf(stderr, "  %s{%d..%d}\n",
	    be->VarShow(BEFnBEObj((BEFN) f)), BEFnFeatureOffset(f),
	    BEFnFeatureOffset(f) + BEFnFeatureFieldWidth(f, be) - 1);
	if( !swizzle_bits_done && class->swizzle_offset < BEFnFeatureOffset(f) )
	{
	  InsertGap(class->record_width, class->swizzle_offset, be);
	  class->record_width = class->swizzle_offset + class->swizzle_width;
	  be->StructItem(NPBack_Swizzle_Bits, swizzle_type);
	  swizzle_bits_done = TRUE;
	}
	InsertGap(class->record_width, BEFnFeatureOffset(f), be);
	class->record_width = BEFnFeatureOffset(f)+BEFnFeatureFieldWidth(f, be);
	be->StructItem(BEFnFeatureLayoutBEObj(f), BEFnBEType((BEFN) f));
      }
      if( !swizzle_bits_done )
      {
	InsertGap(class->record_width, class->swizzle_offset, be);
	class->record_width = class->swizzle_offset + class->swizzle_width;
	be->StructItem(NPBack_Swizzle_Bits, swizzle_type);
	swizzle_bits_done = TRUE;
      }
      be->StructEnd(class->ctype == TAGGED ?
	class->be_type : class->be_content_type);

      /* ensure the record width is a multiple of the word size, and return */
      while( class->record_width % 32 != 0 )
	class->record_width++;
      break;


    default:

      assert(FALSE);
  }
  if( DEBUG1 )
    fprintf(stderr, "] ClassTypeStructPrint returning\n");
}


/*****************************************************************************/
/*                                                                           */
/*  int ClassStructWidth(CLASS class)                                        */
/*                                                                           */
/*  Return the width in bits of the struct holding object of this class.     */
/*                                                                           */
/*****************************************************************************/

int ClassStructWidth(CLASS class)
{
  assert(class->record_width > 0);
  return class->record_width;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "executable code".                                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void ClassFunctionsComment(CLASS c, ASTRING type, CODEGEN be)            */
/*                                                                           */
/*  Generate a C comment to precede a list of functions or their headers.    */
/*                                                                           */
/*****************************************************************************/

static void ClassFunctionsComment(CLASS c, ASTRING type, CODEGEN be)
{
  char buff[200];  ASTRING be_name;  USTRING class_name;
  assert(c->name_set != NULL && ArraySize(c->name_set) > 0);
  be_name = be->TypeShow(c->be_type);
  class_name = NameRep(ClassViewName(c->orig_cv));
  if( UStringEqual(class_name, AStringToUString(be_name)) )
    sprintf(buff, "%s functions of %s%s%sclass \"%s\"", type,
      c->is_abstract ? "abstract " : "", ClassIsEnum(c) ? "enumerated " : "",
      ClassIsBuiltin(c) ? "builtin " : "", UStringToUTF8(class_name));
  else
    sprintf(buff, "%s functions of %s%s%sclass \"%s\" (C name is \"%s\")", type,
      c->is_abstract ? "abstract " : "", ClassIsEnum(c) ? "enumerated " : "",
      ClassIsBuiltin(c) ? "builtin " : "", UStringToUTF8(class_name), be_name);
  be->CommentLarge(buff, NULL, NULL, TRUE, TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  void ClassInvtCallCodeGen(CLASS c, CODEGEN_TYPE self_be_type, CODEGEN be)*/
/*                                                                           */
/*  Generate a call on c's invariant, if it has one.  This is called         */
/*  indirectly from BEFnCreationCodeGen just below.                          */
/*                                                                           */
/*****************************************************************************/

void ClassInvtCallCodeGen(CLASS c, CODEGEN_TYPE self_be_type, CODEGEN be)
{
  if( c->befn_invariant != NULL )
    BEFnInvtCallCodeGen(c->befn_invariant, self_be_type, be);
}


/*****************************************************************************/
/*                                                                           */
/*  void SwizzleCallCogeGen(CODEGEN_OBJ npsys_obj, CODEGEN_OBJ obj,          */
/*    CODEGEN_OBJ attr_be_var, CODEGEN be)                                   */
/*                                                                           */
/*  Generate "mem_offset(<npsysobj>, (CACHE_OBJ) <obj>-><attr_name>)"        */
/*                                                                           */
/*****************************************************************************/

static void SwizzleCallCogeGen(CODEGEN_OBJ npsys_obj, CODEGEN_OBJ obj,
  CODEGEN_OBJ attr_be_var, CODEGEN be)
{
  static CODEGEN_OBJ mem_offset = NULL;
  if( mem_offset == NULL )
  {
    mem_offset = be->FunctionMake("mem_offset", TRUE, TRUE);
  }
  Call2(mem_offset, Var(npsys_obj),
    Cast(NPBack_Cache_Obj_Type, NULL, Call1(attr_be_var, Var(obj))));
}


/*****************************************************************************/
/*                                                                           */
/*  void SwizzleFnCodeGen(CLASS c, CODEGEN be)                               */
/*                                                                           */
/*  Print the swizzle function of c onto fp - header and body.               */
/*  It can be static because it's called only from later in the same file.   */
/*                                                                           */
/*****************************************************************************/

static void SwizzleFnCodeGen(CLASS c, CODEGEN be)
{
  BEFN_FEATURE f;  int index;
  CODEGEN_OBJ f_be_obj;  CODEGEN_TYPE be_type, f_be_type;

  /* set up some backend functions */
  static CODEGEN_OBJ offset1 = NULL;
  static CODEGEN_OBJ offset2 = NULL;
  if( offset1 == NULL )
  {
    offset1 = be->VarMake("offset1", NULL);
    offset2 = be->VarMake("offset2", NULL);
  }

  be->FunctionBegin(c->swizzle_fn, be->int_type);
  be_type = c->ctype == ENUMERATED ? c->be_content_type : c->be_type;
  be->FunctionFormal(c->swizzle_fn, NPBack_Self, be_type);
  be->FunctionFormal(c->swizzle_fn, NPBack_Other, be_type);
  be->FunctionContinue(c->swizzle_fn);
  if( c->swizzle_width > 0 )
  {
    be->VarDeclare(offset1, be->int_type);
    be->VarDeclare(offset2, be->int_type);
    Assign(Call1(NPBack_Swizzle_Bits, Var(NPBack_Self)), Int(0));
  }
  Assert(Equal(
    Call1(NPBack_Type_Tag, Var(NPBack_Self)),
    Call1(NPBack_Type_Tag, Var(NPBack_Other))));
  ArrayForEach(c->cfeatures, f)
  {
    f_be_obj = BEFnFeatureLayoutBEObj(f);
    f_be_type = BEFnBEType((BEFN) f);
    switch( BEFnFeatureSwizzleStatus(f, &index) )
    {
      case TYPE_POINTER_YES:

	/* this field is swizzled in all objects */
	Assign( Call1(f_be_obj, Var(NPBack_Self)),
	  Cast(f_be_type, NULL,
	    SwizzleCallCogeGen(NPBack_NPSysObj1, NPBack_Self, f_be_obj, be)));
	c->has_pointer = TRUE;
	break;


      case TYPE_POINTER_NO:

	/* this field is never swizzled in any object */
	break;


      case TYPE_POINTER_VAR:

	/* this field varies from object to object */
	assert(c->swizzle_width > 0);
	Stmt(Call3(NPBack_Swizzle_Field, Call1(f_be_obj, Var(NPBack_Self)),
	  Call1(f_be_obj, Var(NPBack_Other)), Int(index)));
	c->has_pointer = TRUE;
	break;


      default:

	assert(FALSE);
    }
  }
  Return(Int(c->record_width / CHAR_BIT));
  be->FunctionEnd(c->swizzle_fn);
}


/*****************************************************************************/
/*                                                                           */
/*  void UnSwizzleStmtCogeGen(CODEGEN_OBJ attr, CODEGEN_TYPE type,CODEGEN be)*/
/*                                                                           */
/*    Code generate "self->attr = (type) (mem + (int) self->attr)"           */
/*    OLD: self->attr = (type) mem_unoffset(npsys_obj1, (int) self->attr)    */
/*                                                                           */
/*****************************************************************************/

static void UnSwizzleStmtCogeGen(CODEGEN_OBJ attr, CODEGEN_TYPE type, CODEGEN be)
{
  /* ***
  static CODEGEN_OBJ mem_unoffset = NULL;
  static CODEGEN_OBJ obj1 = NULL;
  if( mem_unoffset == NULL )
  {
    mem_unoffset = be->FunctionMake("mem_unoffset", TRUE, TRUE);
    obj1 = be->VarMake("npsys_obj1", NULL);
  }
  Assign(Call1(attr, Var(NPBack_Self)), Cast(type, NULL,
    Call2(mem_unoffset, Var(obj1),
      Cast(be->int_type, NULL, Call1(attr, Var(NPBack_Self))))
  ));
  *** */
  Assign(Call1(attr, Var(NPBack_Self)),
    Cast(type, NULL, Call2(be->add, Var(NPBack_Mem),
    Cast(be->int_type, NULL, Call1(attr, Var(NPBack_Self))))));
}


/*****************************************************************************/
/*                                                                           */
/*  void UnSwizzleFnCodeGen(CLASS c, CODEGEN be)                             */
/*                                                                           */
/*  Print the unswizzle function of c onto fp - header and body.             */
/*  It can be static because it's called only from later in the same file.   */
/*                                                                           */
/*****************************************************************************/

static void UnSwizzleFnCodeGen(CLASS c, CODEGEN be)
{
  BEFN_FEATURE f;  int index, i;  BOOLEAN comment_printed;
  CODEGEN_OBJ f_be_obj;  CODEGEN_TYPE f_be_type, be_type;

  static CODEGEN_OBJ swizzle_bits = NULL;
  if( swizzle_bits == NULL )
    swizzle_bits = be->AttrMake("swizzle_bits");

  /* function header */
  be->FunctionBegin(c->unswizzle_fn, be->int_type);
  be_type = c->ctype == ENUMERATED ? c->be_content_type : c->be_type;
  be->FunctionFormal(c->unswizzle_fn, NPBack_Self, be_type);
  be->FunctionFormal(c->unswizzle_fn, NPBack_Mem, be->charp_type);
  be->FunctionContinue(c->unswizzle_fn);

  /* variable declarations */
  if( c->has_pointer )
    be->VarDeclare2(NPBack_I, NPBack_Sig, be->uint_type);
  else
    be->VarDeclare(NPBack_I, be->uint_type);

  /* unswizzle features, possibly re-calculate signature */
  comment_printed = FALSE;
  if( c->has_pointer )
  {
    be->CommentSmall("unswizzle fields and re-calculate signature",FALSE,TRUE);
    VarAssign(NPBack_Sig, Call2(be->lshift, Var(ClassTypeTag(c)), Int(16)));
    comment_printed = TRUE;
  }
  for( i = 0;  i < ArraySize(c->cfeatures);  i++ )
  {
    f = ArrayGet(c->cfeatures, i);
    f_be_obj = BEFnFeatureLayoutBEObj(f);
    f_be_type = BEFnBEType((BEFN) f);

    /* unswizzle */
    switch( BEFnFeatureSwizzleStatus(f, &index) )
    {
      case TYPE_POINTER_YES:

	/* this field is swizzled in all objects */
	if( !comment_printed )
	{
	  be->CommentSmall("unswizzle fields", FALSE, TRUE);
	  comment_printed = TRUE;
	}
        UnSwizzleStmtCogeGen(f_be_obj, f_be_type, be);
	break;


      case TYPE_POINTER_NO:

	/* this field is never swizzled in any object */
	break;


      case TYPE_POINTER_VAR:

	/* this field varies from object to object */
	assert(c->swizzle_width > 0);
	Stmt(Call2(NPBack_Unswizzle_Field, Call1(f_be_obj, Var(NPBack_Self)),
	  Int(index)));
	break;


      default:

	assert(FALSE);
    }

    /* optionally recalculate signature */
    if( c->has_pointer )
    {
      be->VarOpAsstBegin(NPBack_Sig, be->add);
      if( i > 0 )
	Call2(be->lshift, Cast(be->int_type, f_be_type,
	  Call1(f_be_obj, Var(NPBack_Self))), Int(i));
      else
	Cast(be->int_type, f_be_type, Call1(f_be_obj, Var(NPBack_Self)));
      be->AsstEnd();
    }
  }

  /* optionally reassign signature */
  if( c->has_pointer )
    Assign(Call1(NPBack_Signature, Var(NPBack_Self)), Var(NPBack_Sig));

  /* insert into cache */
  be->CommentSmall(
    comment_printed ? "insert into cache (cannot be already present)" :
    "no unswizzling needed; insert into cache (cannot be already present)",
    FALSE, TRUE);
  VarAssign(NPBack_I, Call2(be->mod, Call1(NPBack_Signature, Var(NPBack_Self)),
    Var(NPBack_Obj_Cache_Len)));
  WhileSingle( Index(Var(NPBack_Obj_Cache), Var(NPBack_I)),
    VarAssign(NPBack_I, Call2(be->mod,
      Call2(be->add, Var(NPBack_I), Int(1)), Var(NPBack_Obj_Cache_Len)))
  );
  Stmt(Call2(NPBack_Obj_Cache_Insert,
      Cast(NPBack_Cache_Obj_Type, NULL, Var(NPBack_Self)), Var(NPBack_I)));

  /* return record length and end function */
  be->CommentSmall("return record length", FALSE, TRUE);
  Return(Int(c->record_width / CHAR_BIT));
  be->FunctionEnd(c->unswizzle_fn);
}


/*****************************************************************************/
/*                                                                           */
/*  void ClassCodeGenFunctionsInitFile(CLASS c, CODEGEN be)                  */
/*                                                                           */
/*  Generate onto be the functions of class c related only to predefined     */
/*  object initialization.                                                   */
/*                                                                           */
/*****************************************************************************/

void ClassCodeGenFunctionsInitFile(CLASS c, CODEGEN be)
{
  if( DEBUG12 )
    fprintf(stderr, "[ ClassCodeGenFunctionsInitFile(%s, be)\n",
      NameShow(ClassViewName(c->orig_cv)));

  /* initial comment */
  ClassFunctionsComment(c, "Init", be);

  /* initialization functions */
  if( c->befn_enum_init != NULL )
    BEFnCodeGen((BEFN) c->befn_enum_init, be);
  if( c->befn_class_init != NULL )
    BEFnCodeGen((BEFN) c->befn_class_init, be);

  /* swizzle function */
  if( !ClassIsBuiltin(c) && !c->is_abstract )
    SwizzleFnCodeGen(c, be);

  if( DEBUG12 )
    fprintf(stderr, "] ClassCodeGenFunctionsInitFile returning\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void ClassCodeGenFunctionsLoadFile(CLASS c, CODEGEN be)                  */
/*                                                                           */
/*  Generate onto be the functions of class c related only to predefined     */
/*  object loading.                                                          */
/*                                                                           */
/*****************************************************************************/

void ClassCodeGenFunctionsLoadFile(CLASS c, CODEGEN be)
{
  if( DEBUG12 )
    fprintf(stderr, "[ ClassCodeGenFunctionsLoadFile(%s, be)\n",
      NameShow(ClassViewName(c->orig_cv)));

  /* initial comment */
  ClassFunctionsComment(c, "Loading", be);

  /* load functions */
  if( c->befn_enum_init != NULL )
    BEFnEnumInitCodeGenLoadFn(c->befn_enum_init, be);
  if( c->befn_class_init != NULL )
    BEFnClassInitCodeGenLoadFn(c->befn_class_init, be);

  /* unswizzle function */
  if( !ClassIsBuiltin(c) && !c->is_abstract )
    UnSwizzleFnCodeGen(c, be);

  if( DEBUG12 )
    fprintf(stderr, "] ClassCodeGenFunctionsLoadFile returning\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void ClassCodeGenFunctionsMainFile(CLASS c, CODEGEN be)                  */
/*                                                                           */
/*  Generate the functions of class c (creation, swizzle, unswizzle,         */
/*  features, and clone) onto be.                                            */
/*                                                                           */
/*****************************************************************************/

void ClassCodeGenFunctionsMainFile(CLASS c, CODEGEN be)
{
  BEFN_FEATURE f;  CODEGEN_OBJ be_obj;  BEFN_CREDFT befn_credft;

  if( DEBUG12 )
    fprintf(stderr, "[ ClassCodeGenFunctionsMainFile(%s, be)\n",
      NameShow(ClassViewName(c->orig_cv)));

  /* initial comment */
  ClassFunctionsComment(c, "Main", be);

  /* all_predefined, or all_enumerated and trie stuff */
  if( c->all_predefined != NULL )
    BEFnCodeGen( (BEFN) c->all_predefined, be);
  else if( c->all_enumerated != NULL )
  {
    ArrayForEach(c->orig_predefs, f)
    {
      be_obj = BEFnBEObj((BEFN) f);
      if( be_obj != NULL )
	be->ConstDeclare(be_obj);
    }
    BEFnCodeGen( (BEFN) c->all_enumerated, be);
    BEFnCodeGen( (BEFN) c->all_enum_trie, be);
    BEFnCodeGen( (BEFN) c->legal_code, be);
    BEFnCodeGen( (BEFN) c->with_code, be);
  }

  /* invariant */
  if( c->befn_invariant != NULL )
    BEFnCodeGen((BEFN) c->befn_invariant, be);

  /* creation feature default values */
  ArrayForEach(c->orig_cfeatures, f)
  {
    befn_credft = BEFnFeatureCreationDefaultValue(f);
    if( befn_credft != NULL )
      BEFnCodeGen((BEFN) befn_credft, be);
  }

  /* creation function */
  if( !ClassIsBuiltin(c) && !c->is_abstract )
    BEFnCodeGen((BEFN) c->creation_fn, be);

  /* original creation features */
  ArrayForEach(c->orig_cfeatures, f)
  {
    assert(BEFnFeatureFType(f) == FEATURE_CREATION);
    if( DEBUG12 )
      fprintf(stderr, "  calling BEFnCodeGen on feature %s\n",
       NameShow(BEFnFeatureOrigName(f)));
    BEFnCodeGen((BEFN) f, be);
  }

  /* original noncreation features, except not clone */
  ArrayForEach(c->orig_nfeatures, f)
    if( BEFnFeatureFType(f) != FEATURE_NONCREATION_CLONE )
    {
      if( DEBUG12 )
	fprintf(stderr, "  calling BEFnCodeGen on feature %s\n",
	  BEFnFeatureFType(f) >= FEATURE_PREDEF ?
	  "<predef>" : (ASTRING) NameShow(BEFnFeatureOrigName(f)));
      BEFnCodeGen((BEFN) f, be);
    }

  /* clone, if this is its root class */
  if( c->ds_parent == NULL && c->ds_clone_feature != NULL )
  {
    if( DEBUG12 )
      fprintf(stderr, "  calling BEFnCodeGen on clone feature\n");
    BEFnCodeGen((BEFN) c->ds_clone_feature, be);
  }

  if( DEBUG12 )
    fprintf(stderr, "] ClassCodeGenFunctionsMainFile returning\n");
}
