/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         class_view.c                                               */
/*  DESCRIPTION:  A class view (the view of one class from a system view)    */
/*                                                                           */
/*****************************************************************************/
#include "externs.h"
#define DEBUG1 0
#define DEBUG3 0
#define DEBUG4 0
#define DEBUG6 0
#define DEBUG7 0
#define DEBUG8 0
#define DEBUG9 0
#define DEBUG10 0
#define DEBUG11 0
#define DEBUG12 0
#define DEBUG13 0
#define DEBUG14 0
#define DEBUG15 0
#define DEBUG16 0


/*****************************************************************************/
/*                                                                           */
/*  CLASS_VIEW                                                               */
/*                                                                           */
/*  A CLASS_VIEW is the view of one class from a particular system view      */
/*  (module).  It includes all of the features of the class that are         */
/*  visible from that module, and it has a name.                             */
/*                                                                           */
/*****************************************************************************/

struct class_view_rec {

  /* class view identity and simple properties */
  KIND_TAG		kind_tag;	/* kind (KIND_CLASS_VIEW)            */
  FILE_POS		file_pos;	/* position of class view in input   */
  NAME			name;		/* name of class view                */
  CLASS_VIEW_TYPE	type;		/* whether imported or whatever      */
  SYSTEM_VIEW		system_view;	/* the system view this cv lies in   */
  CLASS			class;		/* the class this is a view of       */
  BOOLEAN		is_private;	/* TRUE if class view is private     */
  BOOLEAN		is_norename;	/* TRUE if class view is norename    */

  /* types */
  TYPE_VARS		type_vars;	/* type variables                    */
  TYPE_VARS		ext_type_vars;	/* type vars if type is EXTENDED     */

  /* feature views */
  ARRAY_FEFN_FEATURE	uninherited_features_array;	/* may be imported   */
  SYMTAB_FEFN_FEATURE	uninherited_features_table;	/* but not inherited */
  ARRAY_FEFN_FEATURE_SET  all_features_array;		/* all feature views */
  SYMTAB_FEFN_FEATURE_SET all_features_table;		/* including imports */

  /* miscellaneous */
  FEFN_CREATION		fefn_creation;		/* creation function view    */
  FEFN_FEATURE		fefn_clone;		/* "clone" feature view      */
  FEFN_INVARIANT	invariant;		/* invariant view            */
  FILE_POS		predefined_file_pos;	/* pos of predefined keyword */
  FEFN_FEATURE		all_predefined;		/* "all_predefined" feature  */
  FEFN_FEATURE		all_enumerated;		/* "all_enumerated" feature  */
};


/*****************************************************************************/
/*                                                                           */
/*  CLASS_VIEW_UPDATE                                                        */
/*                                                                           */
/*  A class view update is a set of feature renames.                         */
/*                                                                           */
/*****************************************************************************/

typedef struct feature_rename_rec {
  FILE_POS	from_pos;
  NAME		from_name;
  FILE_POS	to_pos;
  NAME		to_name;
} *FEATURE_RENAME;

typedef ARRAY(FEATURE_RENAME)			ARRAY_FEATURE_RENAME;
typedef SYMTAB(ARRAY_FEATURE_RENAME)		SYMTAB_FEATURE_RENAME;

struct class_view_update_rec {
  FILE_POS	file_pos;			/* position of rename clause */
  SYMTAB_FEATURE_RENAME renames;		/* the feature renames       */
};


/*****************************************************************************/
/*                                                                           */
/*  Submodule "simple queries"                                               */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  FEFN_CREATION ClassViewFEFnCreation(CLASS_VIEW cv)                  */
/*                                                                           */
/*  Return the creation function view of cv.                                 */
/*                                                                           */
/*****************************************************************************/

FEFN_CREATION ClassViewFEFnCreation(CLASS_VIEW cv)
{
  return cv->fefn_creation;
}


/*****************************************************************************/
/*                                                                           */
/*  FILE_POS ClassViewFilePos(CLASS_VIEW cv)                                 */
/*                                                                           */
/*  Return the file position of class view cv.                               */
/*                                                                           */
/*****************************************************************************/

FILE_POS ClassViewFilePos(CLASS_VIEW cv)
{
  return cv->file_pos;
}


/*****************************************************************************/
/*                                                                           */
/*  NAME ClassViewName(CLASS_VIEW cv)                                        */
/*                                                                           */
/*  Return the name of this class view.                                      */
/*                                                                           */
/*****************************************************************************/

NAME ClassViewName(CLASS_VIEW cv)
{
  return cv->name;
}


/*****************************************************************************/
/*                                                                           */
/*  CLASS_VIEW_TYPE ClassViewType(CLASS_VIEW cv)                             */
/*                                                                           */
/*  Return the type of this class view: original, imported, or extended.     */
/*                                                                           */
/*****************************************************************************/

CLASS_VIEW_TYPE ClassViewType(CLASS_VIEW cv)
{
  return cv->type;
}


/*****************************************************************************/
/*                                                                           */
/*  CLASS ClassViewClass(CLASS_VIEW cv)                                      */
/*                                                                           */
/*  Return the class that cv is a view of.                                   */
/*                                                                           */
/*****************************************************************************/

CLASS ClassViewClass(CLASS_VIEW cv)
{
  return cv->class;
}


/*****************************************************************************/
/*                                                                           */
/*  SYSTEM_VIEW ClassViewSystemView(CLASS_VIEW cv)                           */
/*                                                                           */
/*  Return the system view that cv lies in.                                  */
/*                                                                           */
/*****************************************************************************/

SYSTEM_VIEW ClassViewSystemView(CLASS_VIEW cv)
{
  return cv->system_view;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ClassViewIsPrivate(CLASS_VIEW cv)                                */
/*                                                                           */
/*  Return TRUE if cv is private.                                            */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ClassViewIsPrivate(CLASS_VIEW cv)
{
  return cv->is_private;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ClassViewIsNoRename(CLASS_VIEW cv)                               */
/*                                                                           */
/*  Return TRUE if cv is norename.                                           */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ClassViewIsNoRename(CLASS_VIEW cv)
{
  return cv->is_norename;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "establish"                                                    */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  CLASS_VIEW ClassViewNew(FILE_POS file_pos, NAME name,                    */
/*    CLASS_VIEW_TYPE type, SYSTEM_VIEW sv, CLASS class, BOOLEAN is_private, */
/*    BOOLEAN is_norename, TYPE_VARS type_vars, BOOLEAN found_builtin)       */
/*                                                                           */
/*  Return a new, empty class view with these attributes.                    */
/*                                                                           */
/*****************************************************************************/

static CLASS_VIEW ClassViewNew(FILE_POS file_pos, NAME name,
  CLASS_VIEW_TYPE type, SYSTEM_VIEW sv, CLASS class, BOOLEAN is_private,
  BOOLEAN is_norename, TYPE_VARS type_vars, BOOLEAN found_builtin)
{
  CLASS_VIEW res;
  GetMemory(res, CLASS_VIEW);

  /* class view identity and simple properties */
  res->kind_tag = KIND_CLASS_VIEW;
  res->file_pos = file_pos;
  res->name = name;
  res->type = type;
  res->system_view = sv;
  res->class = class;
  res->is_private = is_private;
  res->is_norename = is_norename;

  /* types */
  res->type_vars = type_vars;
  res->ext_type_vars = NULL;

  /* feature views */
  ArrayInit(&res->uninherited_features_array);
  SymInit(&res->uninherited_features_table);
  res->all_features_array = NULL;
  res->all_features_table = NULL;

  /* miscellaneous */
  if( found_builtin )
    res->fefn_creation = NULL;
  else if( class != NULL )
    res->fefn_creation = FEFnCreationNew(file_pos, name, type_vars,
      NULL, ClassType(class), res, ClassBEFnCreation(class));
  else
    res->fefn_creation = FEFnCreationNew(file_pos, name, type_vars,
      NULL, NULL, res, NULL);
  res->invariant = NULL;
  res->predefined_file_pos = file_pos;  /* replaced later if "predefined" */
  res->fefn_clone = NULL;
  res->all_predefined = NULL;
  res->all_enumerated = NULL;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  CLASS_VIEW ClassViewImportCopy(CLASS_VIEW cv, FILE_POS file_pos,         */
/*    NAME name, BOOLEAN is_private, SYSTEM_VIEW sv)                         */
/*                                                                           */
/*  Make a copy of cv for importing, changing the attributes shown.          */
/*                                                                           */
/*****************************************************************************/

CLASS_VIEW ClassViewImportCopy(CLASS_VIEW cv, FILE_POS file_pos,
  NAME name, BOOLEAN is_private, SYSTEM_VIEW sv)
{
  return ClassViewNew(file_pos, name, CLASS_VIEW_IMPORTED, sv, cv->class,
    is_private, FALSE, cv->type_vars, ClassIsBuiltin(cv->class));
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ClassViewImportUninheritedFEFnFeatures(CLASS_VIEW target,        */
/*    CLASS_VIEW source, CLASS_VIEW_UPDATE class_view_update,                */
/*    SYMTAB_NAMED op_table, FILE_POS dft_pos)                               */
/*                                                                           */
/*  Import the public uninherited feature views of source into the           */
/*  uninherited feature views of target, renaming along the way using        */
/*  class_view_update.  Check all new names inserted for operator            */
/*  consistency, using op_table.                                             */
/*                                                                           */
/*  If any of the features are there already, it means they have been        */
/*  imported previously, which is fine as long as they denote the same       */
/*  features.                                                                */
/*                                                                           */
/*  Each newly imported feature needs a file position as well as a name.     */
/*  If it appears explicitly as a rename class_view_update, then that        */
/*  gives a position.  Otherwise, use dft_pos.                               */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ClassViewImportUninheritedFEFnFeatures(CLASS_VIEW target,
  CLASS_VIEW source, CLASS_VIEW_UPDATE class_view_update,
  SYMTAB_NAMED op_table, FILE_POS dft_pos)
{
  USTRING key;  NAME new_name;  FILE_POS new_pos;  FEATURE_RENAME fr;
  SYMTAB_FEATURE_RENAME rename_table;  FEFN_FEATURE fv;

  /* make sure that every rename does actually apply */
  rename_table = class_view_update != NULL ? class_view_update->renames : NULL;
  if( rename_table == NULL )
  {
    /* do nothing */
  }
  else if( source->is_norename )
  {
    /* rename clause for norename class */
    fprintf(stderr, "%s: rename clause for norename class %s\n",
      FilePosShow(class_view_update->file_pos), NameShow(source->name));
    return FALSE;
  }
  else
  {
    /* make sure that each rename is feasible */
    SymForEach(rename_table, &key, &fr)
    {
      if( !SymRetrieve(source->uninherited_features_table, key, &fv) )
      {
	/* can't rename an unknown feature */
	fprintf(stderr, "%s: %s not known in class %s\n",
	  FilePosShow(fr->from_pos), NameFullShow(fr->from_name),
	  NameShow(source->name));
	return FALSE;
      }
      else if( FEFnFeatureIsPrivate(fv) )
      {
	/* can't rename a private feature */
	fprintf(stderr, "%s: %s is private in class %s\n",
	  FilePosShow(fr->from_pos), NameFullShow(fr->from_name),
	  NameShow(source->name));
	return FALSE;
      }
      else if( FEFnFeatureIsNoRename(fv) )
      {
	/* can't rename a norename feature */
	fprintf(stderr, "%s: %s is norename in class %s\n",
	  FilePosShow(fr->from_pos), NameFullShow(fr->from_name),
	  NameShow(source->name));
	return FALSE;
      }
      else if( !NameCompatible(FEFnFeatureName(fv), FEFnFeatureFilePos(fv),
	       fr->from_name, fr->from_pos, "feature rename") )
	return FALSE;
    }
  }

  /* import each public feature view of source into target */
  ArrayForEach(source->uninherited_features_array, fv)
  {
    if( !FEFnFeatureIsPrivate(fv) && FEFnFeatureName(fv) != NULL )
    {
      /* get and check the revised name of fv */
      key = NameKey(FEFnFeatureName(fv));
      if( rename_table == NULL || !SymRetrieve(rename_table, key, &fr) )
      {
	new_name = NameNewFromPrev(FEFnFeatureName(fv), (NAMED) fv);
	new_pos = dft_pos;
      }
      else
      {
	new_name = NameNewFromPrev(fr->to_name, (NAMED) fv);
	new_pos = fr->to_pos;
      }
      if( !NameOperatorConsistent(new_name,new_pos,op_table,"feature import") )
	return FALSE;

      /* insert copy of fv unless a view with same name is already present */
      fv = FEFnFeatureCopyImported(fv, new_pos, new_name, target);
      if( !ClassViewRegisterOriginalFEFnFeature(target, fv) )
	return FALSE;
    }
  }
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  void ClassViewRegisterOriginalFEFnFeature(CLASS_VIEW cv, FEFN_FEATURE fv)*/
/*                                                                           */
/*  Add fv as an original feature view to cv.  We might find out later that  */
/*  it is a redefinition, but for now anyway it's original.                  */
/*                                                                           */
/*  The feature view might have a NULL name, indicating an anonymous         */
/*  feature.  In that case it goes into the uninherited features array       */
/*  but not into the table.                                                  */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ClassViewRegisterOriginalFEFnFeature(CLASS_VIEW cv, FEFN_FEATURE fv)
{
  FEFN_FEATURE fv2;  USTRING key;  BEFN_FEATURE f1, f2;
  if( FEFnFeatureName(fv) == NULL )
    ArrayAddLast(cv->uninherited_features_array, fv);
  else
  {
    key = NameKey(FEFnFeatureName(fv));
    if( SymInsert(cv->uninherited_features_table, key, fv, &fv2) )
      ArrayAddLast(cv->uninherited_features_array, fv);
    else
    {
      f1 = FEFnFeatureBEFnFeature(fv);
      f2 = FEFnFeatureBEFnFeature(fv2);
      if( (f1 == NULL && f2 == NULL) || f1 != f2 )
      {
	/* if both views defined locally or they denote different features */
	fprintf(stderr, "%s: feature name clash in class %s:\n",
	  FilePosShow(cv->file_pos), NameShow(cv->name));
	fprintf(stderr, "  %s: %s\n", FilePosShow(FEFnFeatureFilePos(fv)),
	  NameFullShow(FEFnFeatureName(fv)));
	fprintf(stderr, "  %s: %s\n", FilePosShow(FEFnFeatureFilePos(fv2)),
	  NameFullShow(FEFnFeatureName(fv2)));
	return FALSE;
      }
    }
  }
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ClassViewRetrieveFEFnFeatureSet(CLASS_VIEW cv, USTRING key,      */
/*    FEFN_FEATURE_SET *fvs)                                                 */
/*                                                                           */
/*  Retrieve the feature view set with the given key from cv, returning      */
/*  TRUE if found and setting *fvs to the result.  Otherwise return FALSE.   */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ClassViewRetrieveFEFnFeatureSet(CLASS_VIEW cv, USTRING key,
  FEFN_FEATURE_SET *fvs)
{
  return SymRetrieve(cv->all_features_table, key, fvs);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ParseOneFeatureRename(TOKEN *t, SYMTAB_NAMED op_table,           */
/*    FEATURE_RENAME *feature_rename)                                        */
/*                                                                           */
/*  Parse one rename and return it in *from_name and *to_name.               */
/*  Return TRUE if there were no errors in the parse, else FALSE.            */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN ParseOneFeatureRename(TOKEN *t, SYMTAB_NAMED op_table,
  FEATURE_RENAME *feature_rename)
{
  GetMemory(*feature_rename, FEATURE_RENAME);
  (*feature_rename)->from_pos = LexFilePos(curr_token);
  if( !NameDefParse(t, op_table, &(*feature_rename)->from_name) ) return FALSE;
  skip(TK_AS, "\"as\" keyword in rename clause");
  (*feature_rename)->to_pos = LexFilePos(curr_token);
  if( !NameDefParse(t, op_table, &(*feature_rename)->to_name) ) return FALSE;
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ParseClassViewUpdate(TOKEN *t, SYMTAB_NAMED op_table,            */
/*    CLASS_VIEW_UPDATE *res)                                                */
/*                                                                           */
/*  Parse one class view update and return it in *res.                       */
/*  Return TRUE if there were no errors in the parse, else FALSE.            */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ParseOptionalClassViewUpdate(TOKEN *t, SYMTAB_NAMED op_table,
  CLASS_VIEW_UPDATE *res)
{
  FEATURE_RENAME fr, fr2;
  *res = NULL;
  if( LexType(curr_token) == TK_LEFT_PAREN )
  {
    GetMemory(*res, CLASS_VIEW_UPDATE);
    (*res)->file_pos = LexFilePos(curr_token);
    SymInit(&(*res)->renames);
    do
    {
      /* skip opening left par or separating comma, known to be there */
      next_token;

      /* parse one feature rename and insert it into the rename table */
      if( !ParseOneFeatureRename(t, op_table, &fr) )  return FALSE;
      if( !SymInsert((*res)->renames, NameKey(fr->from_name), fr, &fr2) )
      {
	fprintf(stderr, "%s: \"%s\" %s, at %s\n", FilePosShow(fr->from_pos),
	  NameShow(fr->from_name),"already renamed",FilePosShow(fr2->from_pos));
	return FALSE;
      }
    }
    while( LexType(curr_token) == TK_COMMA );

    /* skip concluding right paren */
    skip(TK_RIGHT_PAREN, "comma or closing ) of rename clause");
  }
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ClassViewParse(TOKEN *t, SYSTEM_VIEW sv, FILE_POS file_pos,      */
/*    BOOLEAN extension)                                                     */
/*                                                                           */
/*  Parse one class or class extension as described in the header comment    */
/*  of ClassViewListParse just below.                                        */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN ClassViewParse(TOKEN *t, SYSTEM_VIEW sv, FILE_POS file_pos,
  BOOLEAN extension)
{
  CLASS_VIEW cv, cv2;  FILE_POS save_pos;
  BOOLEAN found_private, found_norename, found_enum, found_extension;
  BOOLEAN found_builtin;
  EXPR expr;  NAME name;  TYPE_VARS type_vars, ext_type_vars;
  TYPE inherit_type;
  SYMTAB_NAMED op_table = SystemViewOpTable(sv);

  if( DEBUG11 )
    fprintf(stderr, "[ ClassViewParse(-, -, %s, %s)\n", FilePosShow(file_pos),
      bool(extension));

  /* parse optional "private", "norename", and "enum" (lenient grammar) */
  found_private = FALSE;
  found_norename = FALSE;
  found_enum = FALSE;
  type_vars = ext_type_vars = NULL;
  while( LexType(curr_token) == TK_PRIVATE ||
	 LexType(curr_token) == TK_NORENAME ||
	 LexType(curr_token) == TK_ENUM ||
	 LexType(curr_token) == TK_EXTENSION )
  {
    if( LexType(curr_token) == TK_PRIVATE )
    {
      prohibit1(found_private, "%s: private keyword appears twice\n",
	LexPos(curr_token));
      found_private = TRUE;
      next_token;
    }
    else if( LexType(curr_token) == TK_NORENAME )
    {
      prohibit1(found_norename, "%s: norename keyword appears twice\n",
	LexPos(curr_token));
      found_norename = TRUE;
      next_token;
    }
    else if( LexType(curr_token) == TK_ENUM )
    {
      prohibit1(found_enum, "%s: enum keyword appears twice\n",
	LexPos(curr_token));
      found_enum = TRUE;
      next_token;
    }
    else if( LexType(curr_token) == TK_EXTENSION )
    {
      prohibit1(TRUE, "%s: extension keyword before class keyword\n",
	LexPos(curr_token));
    }
  }

  /* parse "class" keyword */
  save_pos = LexFilePos(curr_token);
  skip(TK_CLASS, "\"class\" keyword or end of file");

  /* parse optional "extension" (lenient grammar) */
  found_extension = FALSE;
  while( LexType(curr_token) == TK_PRIVATE ||
	 LexType(curr_token) == TK_NORENAME ||
	 LexType(curr_token) == TK_ENUM ||
	 LexType(curr_token) == TK_EXTENSION )
  {
    if( LexType(curr_token) == TK_PRIVATE )
    {
      prohibit1(TRUE, "%s: private keyword after class keyword\n",
	LexPos(curr_token));
    }
    else if( LexType(curr_token) == TK_NORENAME )
    {
      prohibit1(TRUE, "%s: norename keyword after class keyword\n",
	LexPos(curr_token));
    }
    else if( LexType(curr_token) == TK_ENUM )
    {
      prohibit1(TRUE, "%s: enum keyword after class keyword\n",
	LexPos(curr_token));
    }
    else if( LexType(curr_token) == TK_EXTENSION )
    {
      prohibit1(found_extension, "%s: extension keyword appears twice\n",
	LexPos(curr_token));
      found_extension = TRUE;
      next_token;
    }
  }

  /* parse rest of class header, semantics depending on extension or not */
  if( found_extension )
  {
    /* make sure we were expecting a class extension */
    prohibit2(!extension, "%s: class extension in classes file (see %s)\n",
      LexPos(curr_token), FilePosShow(file_pos));

    /* class extensions can't be private or rename */
    prohibit1(found_private, "%s: a class extension may not be private\n",
      LexPos(curr_token));
    prohibit1(found_norename, "%s: a class extension may not be norename\n",
      LexPos(curr_token));

    /* parse name def */
    if( !NameDefParse(t, op_table, &name) )
      db_return(DEBUG11, "ClassViewParse extension name def", FALSE);

    /* work out what class this is an extension of */
    prohibit2(!SystemViewRetrieveClassView(sv, NameKey(name), &cv),
      "%s: %s extends unknown or private class\n", FilePosShow(save_pos),
      NameFullShow(name));

    /* make sure enum status of extension equals that of the original class */
    if( ClassIsEnum(cv->class) )
    {
      prohibit1(!found_enum,
        "%s: non-enum extension of enum class\n", LexPos(curr_token));
    }
    else
    {
      prohibit1(found_enum,
        "%s: enum extension of non-enum class\n", LexPos(curr_token));
    }

    /* parse formal generic parameters */
    if( !TypeOptionalFormalGenericsParse(t, FEATURE_NONCREATION, FALSE,
	  &ext_type_vars) )
      db_return(DEBUG11, "ClassViewParse extension formal generics", FALSE);

    /* inherit clause illegal if extension */
    prohibit1(LexType(curr_token) == TK_INHERIT,
      "%s: inherit in class extension\n", LexPos(curr_token));

    /* ensure this is the first extension of this class in this module */
    prohibit4(ClassViewType(cv) != CLASS_VIEW_IMPORTED,
      "%s: class %s already defined or extended in module %s, at %s\n",
      FilePosShow(save_pos), NameShow(name), NameShow(SystemViewName(sv)),
      FilePosShow(ClassViewFilePos(cv)));
    cv->type = CLASS_VIEW_EXTENDED;
    cv->ext_type_vars = ext_type_vars;
    cv->file_pos = save_pos;  /* more specific than import point */
  }
  else
  {
    /* make sure we were expecting a class, not an extension */
    prohibit2(extension, "%s: class in class extensions file (see %s)\n",
      LexPos(curr_token), FilePosShow(file_pos));

    /* private does not go with norename */
    prohibit1(found_private && found_norename,
      "%s: a private class may not be norename\n", LexPos(curr_token));

    /* parse name def */
    if( !NameDefParse(t, op_table, &name) )
      db_return(DEBUG11, "ClassViewParse class name def", FALSE);

    /* parse formal generic parameters */
    if( !TypeOptionalFormalGenericsParse(t, FEATURE_NONCREATION, FALSE,
	&type_vars) )
      db_return(DEBUG11, "ClassViewParse class formal generics", FALSE);

    /* parse optional inherit clause */
    inherit_type = NULL;
    if( LexType(curr_token) == TK_INHERIT )
    {
      next_token;
      if( !TypeParse(t, &inherit_type) )
	db_return(DEBUG11, "ClassViewParse inherit type", FALSE);
    }

    /* sort out builtin status */
    found_builtin = (LexType(curr_token) == TK_BUILTIN);

    /* make and register cv and its class */
    cv = ClassViewNew(save_pos, name, CLASS_VIEW_ORIGINAL, sv, NULL,
      found_private, found_norename, type_vars, found_builtin);
    cv->class = ClassNew(save_pos, cv, type_vars, inherit_type, found_enum,
      found_builtin);
    SystemViewRegisterClass(sv, cv->class);
    SystemRegisterClass(SystemViewSystem(sv), cv->class);
    if( !SystemViewInsertClassView(sv, cv, &cv2) )
    {
      fprintf(stderr, "%s: class %s already present in %s (at %s)\n",
	FilePosShow(save_pos), NameShow(name), NameShow(SystemViewName(sv)),
	FilePosShow(ClassViewFilePos(cv2)));
      db_return(DEBUG11, "ClassViewParse insert class view", FALSE);
    }

    /* complete the initialization of creation function of cv */
    if( !found_builtin )
      FEFnCreationSetResultTypeAndBEFnCreation(cv->fefn_creation,
	ClassType(cv->class), ClassBEFnCreation(cv->class));

    /* add special things to enumerated class */
    if( found_enum )
    {
      FEFnFeatureRegisterToInt(cv, cv->file_pos);
      FEFnFeatureRegisterLegalCode(cv, cv->file_pos);
      FEFnFeatureRegisterWithCode(cv, cv->file_pos);
    }
  }

  /* parse "builtin" keyword or else creation features */
  if( LexType(curr_token) == TK_BUILTIN )
  {
    prohibit1(!ClassIsBuiltin(cv->class),
      "%s: builtin extension of non-builtin class\n", LexPos(curr_token));
    next_token;
  }
  else while( LexType(curr_token) != TK_INVARIANT &&
	      LexType(curr_token) != TK_NONCREATION &&
	      LexType(curr_token) != TK_PREDEFINED &&
	      LexType(curr_token) != TK_END )
  {
    /* illegal to extend a builtin class with creation features */
    prohibit1(extension && ClassIsBuiltin(cv->class),
      "%s: creation features in extension of builtin class\n",
      LexPos(curr_token));

    /* illegal to extend the "object" class with creation features */
    /* NB extensions of object must be in 2nd or later module, by  */
    /* which time ClassObject will have been defined               */
    prohibit2(extension && cv->class == ClassObject,
      "%s: creation features in extension of %s class\n",
      LexPos(curr_token), NameShow(name));

    /* parse creation feature list and add them */
    /* to cv; a body is required in extensions  */
    if( !FEFnFeatureSetParse(t, cv, FEATURE_CREATION, FALSE, extension, sv) )
      db_return(DEBUG11, "ClassViewParse creation feature set", FALSE);
  }

  /* parse optional invariants */
  if( LexType(curr_token) == TK_INVARIANT )
  {
    /* invariants illegal in builtin class */
    prohibit1(ClassIsBuiltin(cv->class),
      "%s: invariant in builtin class\n", LexPos(curr_token));

    /* make an invariant view */
    cv->invariant = FEFnInvtNew(LexFilePos(curr_token), cv);

    next_token;
    while( LexType(curr_token) != TK_NONCREATION &&
	   LexType(curr_token) != TK_PREDEFINED &&
	   LexType(curr_token) != TK_END )
    {
      /* parse one invariant and add to invariant list */
      if( !ExprParse(t, sv, &expr) )
	db_return(DEBUG11, "ClassViewParse invariant expr", FALSE);
      FEFnInvtAddExpr(cv->invariant, expr);
    }
  }

  /* parse optional noncreation features */
  if( LexType(curr_token) == TK_NONCREATION )
  {
    next_token;
    while( LexType(curr_token) != TK_PREDEFINED &&
	   LexType(curr_token) != TK_END )
    {
      /* parse noncreation feature list and add them to cv */
      /* a body is required if the class is builtin or extension */
      if( !FEFnFeatureSetParse(t, cv, FEATURE_NONCREATION,
	  ClassIsBuiltin(cv->class), extension, sv) )
	db_return(DEBUG11, "ClassViewParse noncreation feature", FALSE);
    }
  }

  /* parse optional predefined objects */
  if( LexType(curr_token) == TK_PREDEFINED )
  {
    /* predefined objects illegal in generic class */
    prohibit1(type_vars != NULL || ext_type_vars != NULL,
      "%s: predefined prohibited in generic class\n", LexPos(curr_token));

    /* save file pos to give to all_predefined or all_enumerated */
    cv->predefined_file_pos = LexFilePos(curr_token);

    next_token;
    while( LexType(curr_token) != TK_END )
    {
      /* parse predefined objects and add to cv */
      if( !FEFnFeatureSetParse(t, cv, FEATURE_PREDEF,
	  ClassIsBuiltin(cv->class), extension, sv) )
	db_return(DEBUG11, "ClassViewParse predefined object", FALSE);
    }
  }

  /* end of class and file */
  skip(TK_END, "\"end\" keyword at end of class");
  db_return(DEBUG11, "ClassViewParse", TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ClassViewListParse(TOKEN *t, SYSTEM_VIEW sv,                     */
/*    FILE_POS file_pos, BOOLEAN extension)                                  */
/*                                                                           */
/*  Parse zero or more classes or class extensions by reading token list *t. */
/*  Work out what's an operator by consulting sv's op_table.  Register the   */
/*  resulting classes with sv.                                               */
/*                                                                           */
/*  Parameter file_pos is used for error messages only; it is the position   */
/*  where the file containing these tokens was listed.                       */
/*                                                                           */
/*  Parameter extension is TRUE if this file is expected to contain class    */
/*  extensions.                                                              */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ClassViewListParse(TOKEN *t, SYSTEM_VIEW sv,
  FILE_POS file_pos, BOOLEAN extension)
{
  while( LexType(curr_token) != TK_END_FILE )
    if( !ClassViewParse(t, sv, file_pos, extension) )
      return FALSE;
  skip(TK_END_FILE, "end of file");
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ClassViewVoidClassMake(SYSTEM_VIEW sv)                           */
/*                                                                           */
/*  Pretend that a class "private class void builtin end" appeared in the    */
/*  current module.                                                          */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ClassViewVoidClassMake(SYSTEM_VIEW sv)
{
  CLASS_VIEW cv, cv2;  CLASS class;
  FILE_POS file_pos;  NAME name;
  file_pos = SystemViewFilePos(sv);
  name = NameNew(AStringToUString("void"));
  cv = ClassViewNew(file_pos, name, CLASS_VIEW_ORIGINAL, sv, NULL, TRUE,
    FALSE, NULL, TRUE);
  class = ClassNew(file_pos, cv, NULL, NULL, FALSE, TRUE);
  cv->class = class;
  SystemViewRegisterClass(sv, class);
  SystemRegisterClass(SystemViewSystem(sv), class);
  if( !SystemViewInsertClassView(sv, cv, &cv2) )
  {
    fprintf(stderr, "%s: class %s already present in %s (at %s)\n",
      FilePosShow(file_pos), NameShow(name), NameShow(SystemViewName(sv)),
      FilePosShow(ClassViewFilePos(cv2)));
    return FALSE;
  }
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "validate interface types"                                     */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ClassViewValidateInterfaceTypes0(CLASS_VIEW cv)                  */
/*                                                                           */
/*  Carry out stage 0 of instantiating class view cv.                        */
/*  For an original view, this consists of adding an all_predefined or       */
/*  all_enumerated feature to cv, except that we do nothing for a generic    */
/*  class.  For an imported class or extension there is nothing to do.       */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ClassViewValidateInterfaceTypes0(CLASS_VIEW cv)
{
  BOOLEAN res;
  if( DEBUG14 )
    fprintf(stderr, "[ ClassViewValidateInterfaceTypes0(%s)\n",
      NameShow(cv->name));
  res = TRUE;
  if( cv->type == CLASS_VIEW_ORIGINAL && cv->type_vars == NULL )
  {
    if( ClassIsEnum(ClassViewClass(cv)) )
    {
      cv->all_enumerated = FEFnFeatureAllEnumeratedMake(cv->predefined_file_pos,
	cv, cv->system_view);
      res = ClassViewRegisterOriginalFEFnFeature(cv, cv->all_enumerated);
    }
    else if( ClassViewClass(cv) != ClassVoid )
    {
      cv->all_predefined = FEFnFeatureAllPredefinedMake(cv->predefined_file_pos,
	cv, cv->system_view);
      res = ClassViewRegisterOriginalFEFnFeature(cv, cv->all_predefined);
    }
  }
  db_return(DEBUG14, "ClassViewValidateInterfaceTypes0", res);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ClassViewValidateInterfaceTypes1(CLASS_VIEW cv, CONTEXT cxt)     */
/*                                                                           */
/*  Carry out the first stage of instantiating class view cv in context      */
/*  cxt.  For an original class view, this consists of setting its class's   */
/*  type_vars, class_type, and parent_set to their instantiated values.      */
/*  A class and its original view share type variables, so this will         */
/*  also instatiate the type variables in the class view.                    */
/*                                                                           */
/*  For an extension we need to check the extension's type variables, then   */
/*  just forward them to the original type variables and forget about them.  */
/*                                                                           */
/*  For an imported class there is nothing to do here.                       */
/*                                                                           */
/*  The context contains the global context of all class views in scope,     */
/*  plus, for the convenience of this function there is an empty top         */
/*  layer over that, with a shadow limit, into which this function inserts.  */
/*  This function is not obliged to clean up that top level on exit.         */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ClassViewValidateInterfaceTypes1(CLASS_VIEW cv, CONTEXT cxt)
{
  BOOLEAN res;
  if( DEBUG6 )
    fprintf(stderr, "[ ClassViewValidateInterfaceTypes1(%s)\n",
      NameShow(cv->name));

  res = TRUE;
  switch( cv->type )
  {

    case CLASS_VIEW_ORIGINAL:

      res = ClassValidateInterfaceTypes1(cv->class, cxt);
      break;


    case CLASS_VIEW_EXTENDED:

      /* check level 1 validity of extension type variables */
      if( !TypeVarsBeginLevelOneValid(cv->ext_type_vars, cxt, !cv->is_private) )
	db_return(DEBUG6, "ClassViewValidateInterfaceTypes1 (1)", FALSE);

      /* check consistency of extension vars with originals */
      if( DEBUG3 )
	fprintf(stderr, "  calling TypeVarsEquivalent\n");
      if( !TypeVarsEquivalent(cv->ext_type_vars, cv->type_vars, cxt,
	    cv->file_pos) )
      {
	fprintf(stderr, "%s: in class extension %s, generics inconsistent %s\n",
	  FilePosShow(cv->file_pos), NameShow(cv->name), "with originals");
	db_return(DEBUG3, "ClassViewValidateInterfaceTypes1 (ext)", FALSE);
      }

      /* forward extension generics permanently to originals */
      TypeVarsForwardBegin(cv->ext_type_vars, cv->type_vars, cv->file_pos);
      break;


    case CLASS_VIEW_IMPORTED:

      /* nothing to do here */
      break;

  }
  db_return(DEBUG6, "ClassViewValidateInterfaceTypes1", res);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ClassViewValidateInterfaceTypes5(CLASS_VIEW cv, CONTEXT cxt)     */
/*                                                                           */
/*  Stage 5.  Instantiate the interfaces of all the features of the          */
/*  class.  These each consist of type variables, optional parameters,       */
/*  compulsory parameters, and a result type.                                */
/*                                                                           */
/*  On entry, cxt has a class views layer and above that an empty layer      */
/*  which this function uses.  On exit, this layer will still be present,    */
/*  and it will be empty.  This layer is used by this code to hold the       */
/*  class type variables.  It pushes another layer to hold the features,     */
/*  but removes that before exit.                                            */
/*                                                                           */
/*  Logically, this is also the place to manifest the interfaces of all      */
/*  the predefined objects.  However, there is nothing to do in manifesting  */
/*  those, since they have no parameters (generic or ordinary), and their    */
/*  result type is the class type, hence already manifested.                 */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ClassViewValidateInterfaceTypes5(CLASS_VIEW cv, CONTEXT cxt)
{
  BOOLEAN success;  FEFN_FEATURE fv;  TYPE_VARS tvs;
  if( DEBUG6 )
    fprintf(stderr, "[ ClassViewValidateInterfaceTypes5(%s)\n",
      NameShow(cv->name));
  success = TRUE;
  if( cv->type != CLASS_VIEW_IMPORTED )
  {
    /* push class type variables onto context empty layer */
    tvs = cv->type == CLASS_VIEW_EXTENDED ? cv->ext_type_vars : cv->type_vars;
    if( !TypeVarsBegin(tvs, cxt) )
      db_return(DEBUG6, "ClassViewValidateInterfaceTypes5 typevars", FALSE);

    /* do the interface instantiation for each original feature */
    /* use a new level so that it can be cleared out easily */
    ContextPushEmpty(cxt, NULL, FALSE);
    ArrayForEach(cv->uninherited_features_array, fv)
    {
      if( !FEFnFeatureManifestInterface(fv, cxt) )
	success = FALSE;
      ContextClearTopLevel(cxt);
    }
    ContextPop(cxt, TRUE);

    /* clear out context empty layer by popping the type variables */
    TypeVarsEnd(tvs, cxt);
  }
  db_return(DEBUG6, "ClassViewValidateInterfaceTypes5", success);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "feature inheritance"                                          */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void ClassViewAcceptFEFnFeatureSet(CLASS_VIEW cv, FEFN_FEATURE_SET fvs)  */
/*                                                                           */
/*  Original or inherited feature view set fvs has just been accepted into   */
/*  class view cv in this place in the canonical ordering.  It is already    */
/*  in the feature views table (unless anonymous), now it needs to be added  */
/*  to the array as well, then things done as required when a new feature    */
/*  is accepted:                                                             */
/*                                                                           */
/*    * If fv is not imported, it needs to be added to cv's class, either    */
/*      as a new feature, or as a new instance of an existing feature.       */
/*      This is done by FEFnFeatureAddToClass(fv, c).  We do nothing if      */
/*      fv is imported, because in that case class c would have been told    */
/*      about fv in its original module.                                     */
/*                                                                           */
/*    * If fv is a creation feature, then a parameter corresponding to it    */
/*      must be added to cv's creation function and clone feature.  This     */
/*      is true whether or not fv is imported, since these signatures are    */
/*      built for the sole benefit of the current module.  If, in addition,  */
/*      fv is not imported, then a parameter corresponding to its feature    */
/*      must be added to cv's class's creation function and clone feature.   */
/*      The class view parameters must forward to these parameters.  All     */
/*      this is done by FEFnFeatureAddToCreation and FEFnFeatureAddToClone.  */
/*                                                                           */
/*  At this stage there are no coercion features in the class view.          */
/*                                                                           */
/*****************************************************************************/

static void ClassViewAcceptFEFnFeatureSet(CLASS_VIEW cv, FEFN_FEATURE_SET fvs)
{
  FEFN_FEATURE fv;
  if( DEBUG12 )
    fprintf(stderr, "  [ ClassViewAcceptFEFnFeatureSet(%s, %s)\n",
      NameShow(cv->name), NameShow(FEFnFeatureSetName(fvs)));

  /* add fvs to all_features_array */
  ArrayAddLast(cv->all_features_array, fvs);

  /* add fvs's sole uncoerced feature view to the class */
  fv = FEFnFeatureSetSoleUncoercedMember(fvs);
  assert(fv != NULL);
  FEFnFeatureAddToClass(fv, ClassViewClass(cv));

  /* if creation, add a parameter to creation function and possibly clone */
  if( FEFnFeatureFType(fv) == FEATURE_CREATION )
  {
    assert(cv->fefn_creation != NULL);
    FEFnFeatureAddToCreation(fv, cv->fefn_creation);
    if( cv->fefn_clone != NULL )
      FEFnFeatureAddToClone(fv, cv->fefn_clone);
  }

  if( DEBUG12 )
    fprintf(stderr, "  ] ClassViewAcceptFEFnFeatureSet returning\n");
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ClassViewInheritParentFEFnFeatureSets(CLASS_VIEW cv,             */
/*    CLASS_VIEW parent_cv, CONTEXT cxt)                                     */
/*                                                                           */
/*  Global function but actually part of ClassInheritFeatures1 below.        */
/*                                                                           */
/*  Inherit all uncoerced feature views from parent_cv into cv, copying      */
/*  their types so that the substitution which has been set up for the       */
/*  parent class's variables will take effect, and checking that the         */
/*  inherited feature views are compatible with any other inherited or       */
/*  redefined feature views with the same name.                              */
/*                                                                           */
/*  Inheritance of invariants is also carried out here.  Predefined          */
/*  object features are not inherited, so this code skips over them.         */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ClassViewInheritParentFEFnFeatureSets(CLASS_VIEW cv,
  CLASS_VIEW parent_cv, CONTEXT cxt)
{
  FEFN_FEATURE_SET parent_fvs, fvs, fvs2;  USTRING key;
  if( DEBUG10 )
    fprintf(stderr, "[ ClassViewInheritParentFEFnFeatureSets(%s, %s, -)\n",
      NameShow(ClassViewName(cv)), NameShow(ClassViewName(parent_cv)));
  ArrayForEach(parent_cv->all_features_array, parent_fvs)
  {
    if( FEFnFeatureFType(FEFnFeatureSetSoleUncoercedMember(parent_fvs))
	< FEATURE_PREDEF )
    {
      if( DEBUG10 )
      {
	fprintf(stderr, "  inheriting ");
	FEFnFeatureSetDebug(parent_fvs, cxt, stderr, 0);
	fprintf(stderr, "\n");
      }
      key = NameKey(FEFnFeatureSetName(parent_fvs));
      if( SymRetrieve(cv->all_features_table, key, &fvs) )
      {
	/* feature view set already present; just accept parent_fvs into it */
	if( !FEFnFeatureSetCheckInherited(parent_fvs, fvs, parent_cv, cv, cxt) )
	  db_return(DEBUG10, "ClassViewInheritParentFEFnFeatureSets", FALSE);
      }
      else
      {
	/* fvs is new, so add to both the table and the canonical array */
	fvs = FEFnFeatureSetCopyInherited(parent_fvs, cv);
	SymInsert(cv->all_features_table, key, fvs, &fvs2);
	ClassViewAcceptFEFnFeatureSet(cv, fvs);
      }
    }
  }
  db_return(DEBUG10, "ClassViewInheritParentFEFnFeatureSets", TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ClassViewInheritFeatures1(CLASS_VIEW cv, CONTEXT cxt)            */
/*                                                                           */
/*  Build the full set of features of this class, by bringing down the       */
/*  inherited features, and place the result in cv->all_features_array       */
/*  and cv->all_features_table.                                              */
/*                                                                           */
/*  This has to be done in topological order, which will always be           */
/*  possible since by this stage we have verified that the class             */
/*  inheritance graph has no cycles.  The state of the operation on cv       */
/*  is recorded by the condition cv->all_features_array != NULL.             */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ClassViewInheritFeatures1(CLASS_VIEW cv, CONTEXT cxt)
{
  FEFN_FEATURE fv;  FEFN_FEATURE_SET fvs, fvs2;
  static USTRING clone_str = NULL;
  if( clone_str == NULL )
    clone_str = AStringToUString("clone");

  if( DEBUG8 )
    fprintf(stderr, "[ ClassViewInheritFeatures1(%s)\n",
      NameShow(ClassViewName(cv)));

  /* return if already done */
  if( cv->all_features_array != NULL )
    db_return(DEBUG8, "ClassViewInheritFeatures1 (already done)", TRUE);

  /* ensure view of non-builtin, non-enum class has a clone feature */
  if( !ClassIsBuiltin(cv->class) && !ClassIsEnum(cv->class) )
  {
    if( cv->type == CLASS_VIEW_ORIGINAL )
    {
      /* clone feature needs to be made fresh */
      cv->fefn_clone = FEFnFeatureCloneNew(cv);
      if( !ClassViewRegisterOriginalFEFnFeature(cv, cv->fefn_clone) )
	return FALSE;
      if( DEBUG8 )
	fprintf(stderr, "  just registered %s with %s\n",
	  NameShow(FEFnFeatureName(cv->fefn_clone)),
	  NameShow(ClassViewName(cv)));
    }
    else
    {
      /* clone feature view should have already been imported */
      if( !SymRetrieve(cv->uninherited_features_table, clone_str,
	    &cv->fefn_clone) )
	assert(FALSE);
    }
  }

  /* initialize all_features_array to empty */
  ArrayInit(&cv->all_features_array);

  /* init all_features_table with views of non-anonymous original features */
  SymInit(&cv->all_features_table);
  ArrayForEach(cv->uninherited_features_array, fv)
    if( FEFnFeatureName(fv) != NULL )
    {
      fvs = FEFnFeatureSetNewFromFEFnFeature(fv);
      SymInsert(cv->all_features_table,NameKey(FEFnFeatureName(fv)),fvs,&fvs2);
    }

  /* merge parent feature views (via ClassViewInheritParentFEFnFeatureSets) */
  if( !TypeInheritFEFnFeatureSets(cv, ClassParentSet(cv->class), cxt) )
    db_return(DEBUG8, "ClassViewInheritFeatures1 (1)", FALSE);

  /* add local definitions and redefinitions last to all_features_array */
  /* so that it is in canonical order                                   */
  ArrayForEach(cv->uninherited_features_array, fv)
  {
    if( FEFnFeatureName(fv) == NULL )
    {
      /* anonymous feature, make a fvs for it */
      fvs = FEFnFeatureSetNewFromFEFnFeature(fv);
    }
    else
    {
      /* ordinary feature, retrieve fvs from all features table */
      SymRetrieve(cv->all_features_table, NameKey(FEFnFeatureName(fv)), &fvs);
    }
    ClassViewAcceptFEFnFeatureSet(cv, fvs);
  }

  /* check that class name is suited to number of params */
  if( !ClassIsBuiltin(cv->class) && !NameConsistentWithParameters(cv->name,
	FEFnCompulsoryParamsCount((FEFN) cv->fefn_creation), cv->file_pos) )
    return FALSE;

  db_return(DEBUG8, "ClassViewInheritFeatures1", TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ClassViewInheritFeatures2(CLASS_VIEW cv, CONTEXT cxt)            */
/*                                                                           */
/*  Search the ancestor set of class view cv for ancestors which are         */
/*  reached by non-NULL coercions, and import into cv all their features.    */
/*                                                                           */
/*  Note that the ancestors are visited in reverse sorted order, so that     */
/*  children are visited before parents whenever possible.                   */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ClassViewInheritFeatures2(CLASS_VIEW cv, CONTEXT cxt)
{
  ITYPE itype;  ITYPES ancestor_set;
  CLASS_VIEW ancestor_cv;  COERCION coercion;
  FEFN_FEATURE_SET fvs, ancestor_fvs, fvs2;  USTRING key;  FEFN_FEATURE fv;
  if( DEBUG6 )
    fprintf(stderr, "[ ClassViewInheritFeatures2(%s)\n",
      NameShow(ClassViewName(cv)));
  ancestor_set = ClassAncestorSet(cv->class);
  ArrayForEachReverse(ancestor_set, itype)
    if( TypeITypeCoercion(itype) != NULL )
    {
      ancestor_cv = ClassToView(TypeITypeClass(itype), cxt);
      coercion = TypeITypeCoercion(itype);
      ArrayForEach(ancestor_cv->all_features_array, ancestor_fvs)
      {
	/* only want ancestor_fvs if not coerced into ancestor_cv */
	fv = FEFnFeatureSetSoleUncoercedMember(ancestor_fvs);
	if( fv != NULL && FEFnFeatureFType(fv) < FEATURE_PREDEF )
	{
	  key = NameKey(FEFnFeatureName(fv));
	  if( !SymRetrieve(cv->all_features_table, key, &fvs) )
	  {
	    /* make and insert a new feature view set */
	    fvs = FEFnFeatureSetNew(FEFnFeatureFilePos(fv),FEFnFeatureName(fv));
	    SymInsert(cv->all_features_table, key, fvs, &fvs2);
	    ArrayAddLast(cv->all_features_array, fvs);
	  }
	  if( !FEFnFeatureSetAddCoercedCopy(fvs, fv, coercion, cv) )
	    db_return(DEBUG4, "ClassAddCoercedFeatures", FALSE);
	}
      }
    }
  db_return(DEBUG6, "ClassInheritFeatures2", TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "validate expressions".                                        */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ClassViewValidateExpressions(CLASS_VIEW cv, CONTEXT cxt)         */
/*                                                                           */
/*  Do a complete type analysis of all expressions original to cv:  its      */
/*  creation function (including default values of creation features),       */
/*  feature view bodies (including predefined features), and invariant.      */
/*                                                                           */
/*  Since a CLASS_IMPORTED contains no original expressions, we skip         */
/*  over CLASS_IMPORTED classes in this stage.                               */
/*                                                                           */
/*  At entry, cxt is the class views context only.  It's left in this        */
/*  state on exit.                                                           */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ClassViewValidateExpressions(CLASS_VIEW cv, CONTEXT cxt)
{
  FEFN_FEATURE fv;  BOOLEAN success;

  if( DEBUG9 )
    fprintf(stderr, "[ ClassViewValidateExpressions(%s, cxt)\n",
      NameShow(ClassViewName(cv)));
  success = TRUE;
  if( cv->type != CLASS_VIEW_IMPORTED )
  {
    /* add class vars to context */
    ContextPushEmpty(cxt, NULL, TRUE);
    if( !TypeVarsBegin(cv->type_vars, cxt) )
    {
      ContextPop(cxt, TRUE);
      db_return(DEBUG9, "ClassViewValidateExpressions", FALSE);
    }

    /* manifest creation function, including creation feature default values */
    if( cv->fefn_creation != NULL )
      if( !FEFnCreationManifest(cv->fefn_creation, cxt, ClassType(cv->class)) )
	success = FALSE;

    /* add feature views to context */
    ContextPushFEFnFeatureSets(cxt, cv->all_features_table, FALSE);

    /* manifest bodies of unimported features, excluding creation features */
    ContextPushEmpty(cxt, NULL, TRUE);
    ArrayForEach(cv->uninherited_features_array, fv)
      if( !FEFnFeatureIsImported(fv) && FEFnFeatureFType(fv)!=FEATURE_CREATION )
      {
	if( !FEFnFeatureManifestBody(fv, cxt, ClassType(cv->class)) )
	  success = FALSE;
	ContextClearTopLevel(cxt);
      }

    /* manifest invariant */
    if( cv->invariant != NULL )
      if( !FEFnInvtManifest(cv->invariant, cxt) )
	success = FALSE;

    /* take away context */
    ContextPop(cxt, TRUE);	/* working level       */
    ContextPop(cxt, FALSE);	/* feature views level */
    ContextPop(cxt, TRUE);	/* class vars level    */
  }
  db_return(DEBUG9, "ClassViewValidateExpressions", success);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "debug"                                                        */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void ClassViewDebug(CLASS_VIEW cv, CONTEXT cxt, FILE *fp,int print_style)*/
/*                                                                           */
/*  Debug print of class view cv onto fp with the given print style.         */
/*                                                                           */
/*****************************************************************************/

void ClassViewDebug(CLASS_VIEW cv, CONTEXT cxt, FILE *fp, int print_style)
{
  FEFN_FEATURE_SET fvs;  FEFN_FEATURE fv;
  fprintf(fp, "[ class view %s", NameFullShow(cv->name));
  begin_indent;
  next_line;
  fprintf(fp, "orig_features:");
  begin_indent;
  ArrayForEach(cv->uninherited_features_array, fv)
  {
    next_line;
    FEFnFeatureDebug(fv, cxt, fp);
  }
  end_indent;
  if( cv->all_features_array != NULL )
  {
    next_line;
    fprintf(fp, "all_features:");
    begin_indent;
    ArrayForEach(cv->all_features_array, fvs)
    {
      next_line;
      FEFnFeatureSetDebug(fvs, cxt, fp, print_style);
    }
    end_indent;
  }
  end_indent;
  next_line;
  fprintf(fp, "] end class view %s", NameShow(cv->name));
}
