/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         codegen.h                                                  */
/*  DESCRIPTION:  Interface to code generator(s).                            */
/*                                                                           */
/*****************************************************************************/
#include "memory.h"
#include "boolean.h"
#include "astring.h"
#include "ustring.h"
#include "array.h"

typedef struct codegen_file_rec *		CODEGEN_FILE;
typedef struct codegen_exec_rec *		CODEGEN_EXEC;
typedef struct codegen_type_rec *		CODEGEN_TYPE;
typedef struct codegen_obj_rec *		CODEGEN_OBJ;

typedef ARRAY(CODEGEN_OBJ) ARRAY_CODEGEN_OBJ;
typedef ARRAY(CODEGEN_TYPE) ARRAY_CODEGEN_TYPE;

typedef struct codegen_rec {

  /* executables */
  CODEGEN_EXEC	(*ExecMake)(USTRING exec_name);
  void		(*ExecAddFile)(CODEGEN_EXEC exec, CODEGEN_FILE file);

  /* comments */
  void		(*CommentLarge)(ASTRING str1, ASTRING str2, ASTRING str3,
		  BOOLEAN header, BOOLEAN code);
  void		(*CommentSmall)(ASTRING str, BOOLEAN header, BOOLEAN code);

  /* files */
  CODEGEN_FILE	(*FileMake)(USTRING file_name, BOOLEAN system,
      		  BOOLEAN header_only);
  void		(*FileBegin)(CODEGEN_FILE file, ASTRING comment1,
      		  ASTRING comment2);
  void		(*HeaderFileInclude)(CODEGEN_FILE file);
  void		(*CodeFileInclude)(CODEGEN_FILE file);
  void		(*FileEnd)(CODEGEN_FILE file);
  void		(*FileCopy)(USTRING dir_name, CODEGEN_FILE file,
		  ASTRING comment1, ASTRING comment2);
  void		(*FlushCurrentFile)(void);

  /* types, type tags, structs, and their attributes (which are functions) */
  CODEGEN_TYPE	(*TypeMakeObject)(USTRING usugg);
  CODEGEN_TYPE	(*TypeMakeFunctionType)(USTRING usugg, int param_count);
  CODEGEN_TYPE	(*TypeMakeEnumerated)(USTRING usugg, int bits_reqd,
      		  struct codegen_rec *be);
  int		(*TypeFieldWidth)(CODEGEN_TYPE type);
  void		(*TypeDef)(CODEGEN_TYPE type);
  ASTRING	(*TypeShow)(CODEGEN_TYPE type);
  void		(*StructBegin)(CODEGEN_TYPE type);
  CODEGEN_OBJ	(*StructItemMake)(USTRING usugg, int offset);
  void		(*StructItem)(CODEGEN_OBJ attr, CODEGEN_TYPE type);
  void		(*StructGap)(int gap_bit_width);
  void		(*StructEnd)(CODEGEN_TYPE type);

  /* blocks and indenting */
  void		(*IndentBegin)(void);
  void		(*IndentEnd)(void);
  void		(*BlockBegin)(void);
  void		(*BlockEnd)(void);

  /* assignment statements */
  void		(*AsstBegin)(void);
  void		(*AsstContinue)(void);
  void		(*AsstEnd)(void);
  void		(*AsstOpBegin)(CODEGEN_OBJ op);
  void		(*AsstOpContinue)(CODEGEN_OBJ op);
  void		(*AsstOpEnd)(CODEGEN_OBJ op);
  void		(*AsstArrayInit)(CODEGEN_OBJ arr, CODEGEN_OBJ index_var,
      		  int from_index, int to_index, CODEGEN_OBJ init_var);

  /* other one-line statements */
  void		(*StmtBegin)(void);
  void		(*StmtEnd)(void);

  /* functions and function calls, including predefined operators */
  CODEGEN_OBJ	(*FunctionMake)(ASTRING name, BOOLEAN has_params,
      		  BOOLEAN global);
  CODEGEN_OBJ	(*FunctionMake2)(USTRING name1, USTRING name2,
      		  BOOLEAN has_params, BOOLEAN global);
  CODEGEN_OBJ	(*FunctionMake3)(USTRING name1, USTRING name2, USTRING name3,
      		  BOOLEAN has_params, BOOLEAN global);
  CODEGEN_OBJ	(*FunctionMakeFunRef)(ASTRING name);
  CODEGEN_OBJ	(*FunctionAttributeIndexed)(ASTRING name);
  void		(*FunctionBegin)(CODEGEN_OBJ fn, CODEGEN_TYPE type);
  void		(*FunctionFormal)(CODEGEN_OBJ fn, CODEGEN_OBJ var_fn,
		    CODEGEN_TYPE type);
  void		(*FunctionContinue)(CODEGEN_OBJ fn);
  void		(*FunctionEnd)(CODEGEN_OBJ fn);
  void		(*PrototypeBegin)(CODEGEN_OBJ fn, CODEGEN_TYPE type);
  void		(*PrototypeFormal)(CODEGEN_OBJ fn, CODEGEN_TYPE type);
  void		(*PrototypeEnd)(CODEGEN_OBJ fn);
  void		(*CallBegin)(CODEGEN_OBJ fn);		/* before first param */
  void		(*CallContinue)(CODEGEN_OBJ fn, int i);	/* between params     */
  void		(*CallEnd)(CODEGEN_OBJ fn);		/* after last param   */

  /* variables (creation, declaration, assignment, and invocation) */
  CODEGEN_OBJ	(*VarMake)(ASTRING asugg, USTRING usugg);
  CODEGEN_OBJ	(*VarIndexedAttributeMake)(CODEGEN_OBJ be_obj, ASTRING attr,
		  int index);
  void		(*VarIndexedAttributeUpdateIndex)(CODEGEN_OBJ be_obj, int index);
  void		(*VarDeclare)(CODEGEN_OBJ var, CODEGEN_TYPE type);
  void		(*VarDeclare2)(CODEGEN_OBJ var1, CODEGEN_OBJ var2,
      		  CODEGEN_TYPE type);
  void		(*VarDeclareAndAsstBegin)(CODEGEN_OBJ var, CODEGEN_TYPE type);
  CODEGEN_OBJ	(*VarMakeAndDeclare)(ASTRING asugg, USTRING usugg,
		  CODEGEN_TYPE type);
  CODEGEN_OBJ	(*VarMakeAndDeclareAndAsstBegin)(ASTRING asugg, USTRING usugg,
		  CODEGEN_TYPE type);
  void		(*Variable)(CODEGEN_OBJ var);
  void		(*VarOpAsstBegin)(CODEGEN_OBJ var, CODEGEN_OBJ op);
  void		(*VarAsstBegin)(CODEGEN_OBJ var);
  void		(*VarAsst2Begin)(CODEGEN_OBJ var1, CODEGEN_OBJ var2);
  void		(*VarUnDeclare)(CODEGEN_OBJ var);
  ASTRING	(*VarShow)(CODEGEN_OBJ var);

  /* attributes */
  CODEGEN_OBJ	(*AttrMake)(ASTRING name);

  /* constants */
  CODEGEN_OBJ	(*ConstMakeAString)(ASTRING asugg, USTRING usugg,
      		  ASTRING value, BOOLEAN global);
  CODEGEN_OBJ	(*ConstMakeInt)(ASTRING asugg, USTRING usugg, int value,
      		  BOOLEAN global);
  CODEGEN_OBJ	(*ConstMakeInt2)(USTRING name1, USTRING name2, int value,
      		  BOOLEAN global);
  void		(*ConstDeclare)(CODEGEN_OBJ cnst);

  /* freeing all the above */
  void		(*ObjFree)(CODEGEN_OBJ obj);

  /* if statements and conditional expressions */
  void		(*IfStmtBegin)(void);
  void		(*IfStmtElseIf)(void);
  void		(*IfStmtContinue)(void);
  void		(*IfStmtElse)(void);
  void		(*IfStmtEnd)(void);
  void		(*IfSmallBegin)(void);
  void		(*IfSmallElseIf)(void);
  void		(*IfSmallContinue)(void);
  void		(*IfSmallElse)(void);
  void		(*IfSmallEnd)(void);

  /* switch statements */
  void		(*SwitchBegin)(void);
  void		(*SwitchContinue)(void);
  void		(*SwitchCaseBegin)(void);
  void		(*SwitchCaseEnd)(void);
  void		(*SwitchCase)(CODEGEN_OBJ cnst);
  void		(*SwitchCaseInt)(int case_label);
  void		(*SwitchActionBegin)(void);
  void		(*SwitchActionEnd)(BOOLEAN break_required);
  void		(*SwitchEnd)(void);

  /* while statements */
  void		(*WhileBegin)(void);
  void		(*WhileContinue)(void);
  void		(*WhileEnd)(void);
  void		(*WhileSingleBegin)(void);
  void		(*WhileSingleContinue)(void);
  void		(*WhileSingleEnd)(void);

  /* fail and goto statements (a goto label is treated like a local var) */
  void		(*GotoStmt)(CODEGEN_OBJ label);
  void		(*GotoTarget)(CODEGEN_OBJ label);
  void		(*Fail)(void);

  /* type casts (inserted only if res_type != source_type) */
  void		(*CastBegin)(CODEGEN_TYPE res_type, CODEGEN_TYPE source_type);
  void		(*CastEnd)(void);

  /* invocations of literals */
  void		(*LiteralInt)(int i);
  void		(*LiteralChar)(UCHAR ch);
  void		(*LiteralString)(ASTRING str);
  void		(*LiteralConst)(ASTRING str);

  /* compile */
  BOOLEAN	(*Compile)(void);

  /* predefined types offered by C */
  CODEGEN_TYPE	char_type;  		/* char                         */
  CODEGEN_TYPE	uchar_type;  		/* unsigned char                */
  CODEGEN_TYPE	short_type;		/* short                        */
  CODEGEN_TYPE	ushort_type;		/* unsigned short               */
  CODEGEN_TYPE	int_type;		/* int                          */
  CODEGEN_TYPE	uint_type;		/* unsigned int                 */
  CODEGEN_TYPE	float_type;		/* float       			*/
  CODEGEN_TYPE	double_type;		/* double     			*/
  CODEGEN_TYPE	void_type;		/* void                         */
  CODEGEN_TYPE	voidp_type;		/* void *                       */
  CODEGEN_TYPE	charp_type;		/* char *                       */

  /* predefined functions offered by C (in order of index in K&R) */
  CODEGEN_OBJ	ret;			/* return, treated as variable  */
  CODEGEN_OBJ	no_ret;			/* for expressions of void type */
  CODEGEN_OBJ	add;			/* ($1 + $2)			*/
  CODEGEN_OBJ	address_of;		/* (& $1)			*/
  CODEGEN_OBJ	bitwise_and;		/* ($1 & $2)			*/
  CODEGEN_OBJ	bitwise_or;		/* ($1 | $2)			*/
  CODEGEN_OBJ	divide;			/* ($1 / $2)			*/
  CODEGEN_OBJ	divide_float;		/* ($1 / (float) $2)		*/
  CODEGEN_OBJ	eq;			/* ($1 == $2)			*/
  CODEGEN_OBJ	ge;			/* ($1 >= $2)			*/
  CODEGEN_OBJ	gt;			/* ($1 > $2)			*/
  CODEGEN_OBJ	ne;			/* ($1 != $2)			*/
  CODEGEN_OBJ	lshift;			/* ($1 << $2)			*/
  CODEGEN_OBJ	le;			/* ($1 <= $2)			*/
  CODEGEN_OBJ	lt;			/* ($1 < $2)			*/
  CODEGEN_OBJ	logical_and;		/* ($1 && $2)			*/
  CODEGEN_OBJ	logical_not;		/* (! $1)			*/
  CODEGEN_OBJ	logical_or;		/* ($1 || $2)			*/
  CODEGEN_OBJ	mod;			/* ($1 % $2)			*/
  CODEGEN_OBJ	multiply;		/* ($1 * $2)			*/
  CODEGEN_OBJ	rshift;			/* ($1 >> $2)			*/
  CODEGEN_OBJ	subtract;		/* ($1 - $2)			*/
  CODEGEN_OBJ	unary_minus;		/* (- $1)			*/
  CODEGEN_OBJ	size_of;		/* sizeof($1)			*/
  CODEGEN_OBJ	assert_fn;		/* assert($1)			*/
  CODEGEN_OBJ	array_index;		/* $1[$2]			*/
  /* CODEGEN_OBJ	attribute; */	/* $1->$2			*/

  /* a selection of limits from <limits.h> and <float.h> */
  CODEGEN_OBJ	char_min;		/* SCHAR_MIN			*/
  CODEGEN_OBJ	char_max;		/* SCHAR_MAX			*/
  CODEGEN_OBJ	short_min;		/* SHRT_MIN			*/
  CODEGEN_OBJ	short_max;		/* SHRT_MAX			*/
  CODEGEN_OBJ	int_min;		/* INT_MIN			*/
  CODEGEN_OBJ	int_max;		/* INT_MAX			*/
  CODEGEN_OBJ	uchar_max;		/* UCHAR_MAX			*/
  CODEGEN_OBJ	ushort_max;		/* USHRT_MAX			*/
  CODEGEN_OBJ	uint_max;		/* UINT_MAX			*/
  CODEGEN_OBJ	float_max;		/* FLT_MAX			*/
  CODEGEN_OBJ	float_min;		/* FLT_MIN			*/

  /* a selection of useful C operations */
  CODEGEN_OBJ	identity;		/* $1				*/
  CODEGEN_OBJ	cast_to_char;		/* (signed char) $1		*/
  CODEGEN_OBJ	cast_to_short;		/* (short) $1			*/
  CODEGEN_OBJ	cast_to_int;		/* (int) $1			*/
  CODEGEN_OBJ	cast_to_uchar;		/* (unsigned char) $1		*/
  CODEGEN_OBJ	cast_to_ushort;		/* (unsigned short) $1		*/
  CODEGEN_OBJ	cast_to_uint;		/* (unsigned int) $1		*/
  CODEGEN_OBJ	cast_to_float;		/* (float) $1			*/
  CODEGEN_OBJ	round;			/* (int) round($1)              */
  CODEGEN_OBJ	logical_and_seq;	/* ($1 && $2 && ... && $n)      */
  CODEGEN_OBJ	logical_or_seq;		/* ($1 || $2 || ... || $n)      */
  CODEGEN_OBJ	fprintf_fn;		/* fprintf($1, ... , $n)        */
  CODEGEN_OBJ	stderr_fn;		/* stderr                       */
  CODEGEN_OBJ	null_value;		/* NULL                         */
} *CODEGEN;


/*****************************************************************************/
/*                                                                           */
/*  Helper macros for making codegen access less painful                     */
/*                                                                           */
/*****************************************************************************/

#define VarAssign(var, expr)						\
  (be->VarAsstBegin(var), expr, be->AsstEnd())

#define VarAssign2(var1, var2, expr)					\
  (be->VarAsst2Begin(var1, var2), expr, be->AsstEnd())

#define Return(expr) VarAssign(be->ret, expr)

#define Assign(e1, e2)							\
  (be->AsstBegin(), e1, be->AsstContinue(), e2, be->AsstEnd())

#define AssignOp(op, e1, e2)						\
  (be->AsstOpBegin(op), e1, be->AsstOpContinue(op), e2, be->AsstOpEnd(op))

#define Stmt(expr)							\
  (be->StmtBegin(), expr, be->StmtEnd())

#define DeclareAndAssign(var, type, expr)				\
  (be->VarDeclareAndAsstBegin(var, type), expr, be->AsstEnd())

#define Call0(fn) (be->CallBegin(fn), be->CallEnd(fn))

#define Call1(fn, e1)							\
  (be->CallBegin(fn), e1, be->CallEnd(fn))

#define Call2(fn, e1, e2)						\
  (be->CallBegin(fn), e1, be->CallContinue(fn, 1), e2, be->CallEnd(fn))

#define Call3(fn, e1, e2, e3)						\
  (be->CallBegin(fn), e1, be->CallContinue(fn, 1), e2,			\
   be->CallContinue(fn, 2), e3, be->CallEnd(fn))

#define Call4(fn, e1, e2, e3, e4)					\
  (be->CallBegin(fn), e1, be->CallContinue(fn, 1), e2,			\
   be->CallContinue(fn, 2), e3, be->CallContinue(fn, 3), e4, be->CallEnd(fn))

#define Call5(fn, e1, e2, e3, e4, e5)					\
  (be->CallBegin(fn), e1, be->CallContinue(fn, 1), e2,			\
   be->CallContinue(fn, 2), e3, be->CallContinue(fn, 3), e4,		\
   be->CallContinue(fn, 4), e5, be->CallEnd(fn))

#define Indent(expr)							\
  (be->IndentBegin(), expr, be->IndentEnd())

#define Block(expr)							\
  (be->BlockBegin(), expr, be->BlockEnd())

#define Equal(e1, e2) Call2(be->eq, e1, e2)

#define Index(e1, e2) Call2(be->array_index, e1, e2)

#define Int(i) be->LiteralInt(i)

#define String(str) be->LiteralString(str)

#define Var(var) be->Variable(var)

#define Cast(res_type, from_type, expr)					\
  (be->CastBegin(res_type, from_type), expr, be->CastEnd())

#define While(condition, expr)						\
  (be->WhileBegin(), condition, be->WhileContinue(), expr, be->WhileEnd())

#define WhileSingle(condition, expr)					\
  (be->WhileSingleBegin(), condition, be->WhileSingleContinue(),	\
   expr, be->WhileSingleEnd())

#define If(condition, expr)						\
  (be->IfStmtBegin(), condition, be->IfStmtContinue(), expr, be->IfStmtEnd())

#define IfElse(condition, e1, e2)					\
  (be->IfStmtBegin(), condition, be->IfStmtContinue(), e1,		\
   be->IfStmtElse(), e2, be->IfStmtEnd())

#define IfElse2(cond1, e1, cond2, e2, e3)				\
  (be->IfStmtBegin(), cond1, be->IfStmtContinue(), e1,			\
   be->IfStmtElseIf(), cond2, be->IfStmtContinue(), e2,			\
   be->IfStmtElse(), e3, be->IfStmtEnd())

#define Assert(condition)						\
  (be->StmtBegin(), Call1(be->assert_fn, condition), be->StmtEnd())

#define SmallIfElse(condition, e1, e2)					\
  (be->IfSmallBegin(), condition, be->IfSmallContinue(), e1,		\
   be->IfSmallElse(), e2, be->IfSmallEnd())



/*****************************************************************************/
/*                                                                           */
/*  Creation function                                                        */
/*                                                                           */
/*****************************************************************************/

extern CODEGEN CodeGen_C(USTRING code_dir, ASTRING compilername);
