/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         back_end_c.c                                               */
/*  DESCRIPTION:  C back end                                                 */
/*                                                                           */
/*****************************************************************************/
#include <sys/stat.h>
#include <sys/types.h>
#include <dirent.h>
#include <string.h>
#include <stdio.h>
#include <assert.h>
#include <limits.h>
#include "codegen.h"
#include "asymtab.h"
#include "usymtab.h"
#define DEBUG1 1
#define DEBUG2 0
#define DEBUG3 0
#define DEBUG4 0

#define db_return(db, str, val)						\
{									\
  BOOLEAN xxx_zzz = val;						\
  if( db )								\
    fprintf(stderr, "] %s returning %s\n", str, bool(xxx_zzz));		\
  return xxx_zzz;							\
}


/*****************************************************************************/
/*                                                                           */
/*  Back end files.                                                          */
/*                                                                           */
/*****************************************************************************/

typedef ARRAY(CODEGEN_FILE) ARRAY_CODEGEN_FILE;
typedef SYMTAB(ARRAY_CODEGEN_FILE) SYMTAB_CODEGEN_FILE;

struct codegen_file_rec {
  USTRING		name;		/* name of file                      */
  BOOLEAN		system;		/* TRUE if system file               */
  BOOLEAN		header_only;	/* TRUE if header file only          */
  BOOLEAN		generated;	/* TRUE when generation or copy done */
  ARRAY_CODEGEN_FILE	hincludes;	/* files included into .h file       */
  ARRAY_CODEGEN_FILE	cincludes;	/* files included into .c file       */
};


/*****************************************************************************/
/*                                                                           */
/*  Back end executables.                                                    */
/*                                                                           */
/*****************************************************************************/

struct codegen_exec_rec {
  USTRING		name;		/* name of resulting executable	     */
  ARRAY_CODEGEN_FILE	files;		/* files contributing to executable  */
};

typedef ARRAY(CODEGEN_EXEC) ARRAY_CODEGEN_EXEC;


/*****************************************************************************/
/*                                                                           */
/*  Back end types.                                                          */
/*                                                                           */
/*  Immediate types                                                          */
/*  ---------------                                                          */
/*                                                                           */
/*  These are builtin types whose name is equal to their value:              */
/*                                                                           */
/*      <value>         <type>                                               */
/*      ----------------------------------------                             */
/*      char            CODEGEN_TYPE_CHAR                                    */
/*      unsigned char   CODEGEN_TYPE_UCHAR                                   */
/*      short           CODEGEN_TYPE_SHORT                                   */
/*      unsigned short  CODEGEN_TYPE_USHORT                                  */
/*      int             CODEGEN_TYPE_INT                                     */
/*      unsigned int    CODEGEN_TYPE_UINT                                    */
/*      float           CODEGEN_TYPE_FLOAT                                   */
/*      double          CODEGEN_TYPE_DOUBLE                                  */
/*      void            CODEGEN_TYPE_VOID                                    */
/*      void *          CODEGEN_TYPE_VOIDP                                   */
/*      char *          CODEGEN_TYPE_CHARP                                   */
/*      ----------------------------------------                             */
/*                                                                           */
/*  No typedef is needed for these types.                                    */
/*                                                                           */
/*                                                                           */
/*  CODEGEN_TYPE_OBJECT                                                      */
/*  -------------------                                                      */
/*                                                                           */
/*  These types are references to objects.  At some point they need to be    */
/*  defined by                                                               */
/*                                                                           */
/*      typedef struct <value> *<type_name>;                                 */
/*                                                                           */
/*  followed later in the header file by                                     */
/*                                                                           */
/*      struct <value> {                                                     */
/*         ...                                                               */
/*      };                                                                   */
/*                                                                           */
/*  Most also have a type tag, a constant which is handled separately.       */
/*                                                                           */
/*                                                                           */
/*  CODEGEN_TYPE_FUNCTION                                                    */
/*  ---------------------                                                    */
/*                                                                           */
/*  These types are functions of a given number of parameters.  The          */
/*  parameters' types, and the result type, are void *.  Each needs to be    */
/*  defined by                                                               */
/*                                                                           */
/*      typedef void *(*<type_name>)(void *, void *, ... void *);            */
/*                                                                           */
/*                                                                           */
/*  CODEGEN_TYPE_ENUMERATED                                                  */
/*  -----------------------                                                  */
/*                                                                           */
/*  These are enumerated types, each with a name, and represented by         */
/*  one of "unsigned char", "unsigned short", and "unsigned int",            */
/*  depending on how many elements the enumeration contains.  Each has       */
/*  a name and a value, and a typedef of the form                            */
/*                                                                           */
/*      typedef <value> <name>;                                              */
/*                                                                           */
/*  will be needed to declare these types.                                   */
/*                                                                           */
/*  Alternatively, if a NULL name is passed when defining an enumerated      */
/*  type, it is represented as an immediate type - "unsigned char",          */
/*  "unsigned short", or "unsigned int".                                     */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  CODEGEN_TYPE_CHAR,
  CODEGEN_TYPE_UCHAR,
  CODEGEN_TYPE_SHORT,
  CODEGEN_TYPE_USHORT,
  CODEGEN_TYPE_INT,
  CODEGEN_TYPE_UINT,
  CODEGEN_TYPE_FLOAT,
  CODEGEN_TYPE_DOUBLE,
  CODEGEN_TYPE_VOID,
  CODEGEN_TYPE_VOIDP,
  CODEGEN_TYPE_CHARP,
  CODEGEN_TYPE_OBJECT,
  CODEGEN_TYPE_FUNCTION,
  CODEGEN_TYPE_ENUMERATED
} CODEGEN_TYPE_TYPE;

struct codegen_type_rec
{
  CODEGEN_TYPE_TYPE	type;		/* identifies float as special       */
  ASTRING		name;		/* C type name, used to invoke it    */
  int			param_count;	/* if function type                  */
  ASTRING		value;		/* all other cases                   */
  int			field_width;	/* number of bits occupied           */
};


/*****************************************************************************/
/*                                                                           */
/*  Back end objects.                                                        */
/*                                                                           */
/*  Functions, variables, and "return" are all treated by this module as     */
/*  CODEGEN_OBJ objects.  The traditional syntax for invoking a function,    */
/*  "name1($1, ... , $n)", is just one of many allowable syntaxes here.      */
/*  Others include variables "name1' (no parentheses), infix notation        */
/*  "($1 name1 $2)", and attributes "$1->name1".                             */
/*                                                                           */
/*  The rather complex CODEGEN_OBJ_VAR_INDEX is included because sometimes   */
/*  we need such a thing as a result variable.  We prefer not to use it.     */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  CODEGEN_OBJ_RETURN,			/* return                            */
  CODEGEN_OBJ_NO_RETURN,		/* for expressions of type "void"    */
  CODEGEN_OBJ_CONST,                    /* name1 (but decl. is #define)      */
  CODEGEN_OBJ_VAR,                      /* name1                             */
  CODEGEN_OBJ_VAR_INDEX,                /* name1->name2[index]               */
  CODEGEN_OBJ_IDENTITY,			/* $1                                */
  CODEGEN_OBJ_ORDINARY,			/* name1($1, ... , $n)               */
  CODEGEN_OBJ_ATTRIBUTE,		/* $1->name1                         */
  CODEGEN_OBJ_INDEX,			/* $1[$2]                            */
  CODEGEN_OBJ_ATTRIBUTE_INDEX,		/* $1->name1[$2]                     */
  CODEGEN_OBJ_PREFIX,			/* (name1 $1)                        */
  CODEGEN_OBJ_INFIX,			/* ($1 name1 $2)                     */
  CODEGEN_OBJ_INFIX_SEQ,		/* ($1 name1 $2 name1 ... name1 $n)  */
  CODEGEN_OBJ_PREFIX_INFIX,		/* (name1 ($1 name2 $2))             */
  CODEGEN_OBJ_INFIX_PREFIX,		/* ($1 name1 (name2 $2))             */
  CODEGEN_OBJ_PREFIX_ORDINARY,		/* name1 name2($1, ... , $n)         */
  CODEGEN_OBJ_FUNREF			/* (*($1))($2, ... , $n)             */
} CODEGEN_OBJ_TYPE;


struct codegen_obj_rec {
  CODEGEN_OBJ_TYPE	type;
  BOOLEAN		has_params;	/* derived from type for convenience */
  ASTRING		name1;		/* as in comments just above         */
  ASTRING		name2;		/* as in comments just above         */
  int			index;		/* as in comments just above         */
  BOOLEAN		global;		/* TRUE if global                    */
  CODEGEN_OBJ		next_free_obj;	/* next object in free list          */
};


/*****************************************************************************/
/*                                                                           */
/*  Static variables describing the current state of the C back end.         */
/*                                                                           */
/*****************************************************************************/

static	ARRAY_CODEGEN_EXEC executables;		/* all executables created   */
static	ARRAY_CODEGEN_FILE source_file_array;	/* source files, compiled    */
static	SYMTAB_CODEGEN_FILE source_file_table;	/*   or copied               */
static	ASYMTAB_INT	global_symbols;		/* globally visible symbols  */
static	ASYMTAB_INT	struct_symbols;		/* struct items with offsets */
static	ASYMTAB_INT	local_symbols;		/* local to current function */
static	USTRING		code_dir;		/* where compiler code goes  */
static	ASTRING		compiler_name;		/* name of compiler          */
static	CODEGEN_FILE	curr_file;		/* current file              */
static	FILE *		hfp;			/* current header file ptr   */
static	FILE *		fp;			/* current code file ptr     */
static	int		indent;			/* current code file indent  */
static	int		formals_count;		/* count of formal params    */
static	int		infix_seq_count;	/* count of infix seq params */
static	ARRAY_BOOLEAN	casts_inserted = NULL;
static	ARRAY_BOOLEAN	blocks_inserted = NULL;
static	BOOLEAN		start_block;		/* TRUE at start of block    */
static	USTRING		underscore_str = NULL;	/* "_" ustring               */
static	CODEGEN_OBJ	first_free_obj;		/* first free codegen object */


/*****************************************************************************/
/*                                                                           */
/*  Private submodule debug.                                                 */
/*                                                                           */
/*****************************************************************************/

static ASTRING BackendFnShow(CODEGEN_OBJ be_obj)
{
  static char buff[2][100];
  static int bp = 0;
  bp = (bp + 1) % 2;
  if( be_obj == NULL )
    return "null";
  switch( be_obj->type )
  {
    case CODEGEN_OBJ_RETURN:	return "return";
    case CODEGEN_OBJ_NO_RETURN:	return "void";

    case CODEGEN_OBJ_VAR:	sprintf(buff[bp], "var %s", be_obj->name1);
				break;

    case CODEGEN_OBJ_VAR_INDEX:	sprintf(buff[bp], "var_index %s->%s[%d]",
				  be_obj->name1, be_obj->name2, be_obj->index);
				break;

    case CODEGEN_OBJ_IDENTITY:	return "identity";

    case CODEGEN_OBJ_ORDINARY:	sprintf(buff[bp], "ordinary %s", be_obj->name1);
				break;

    case CODEGEN_OBJ_ATTRIBUTE:	sprintf(buff[bp], "attr %s\n", be_obj->name1);
				break;
	
    case CODEGEN_OBJ_INDEX:	return "index";
    case CODEGEN_OBJ_ATTRIBUTE_INDEX: return "attr_index";

    case CODEGEN_OBJ_PREFIX:	sprintf(buff[bp], "prefix %s\n", be_obj->name1);
				break;

    case CODEGEN_OBJ_INFIX:	sprintf(buff[bp], "infix %s\n", be_obj->name1);
				break;

    case CODEGEN_OBJ_INFIX_SEQ:	sprintf(buff[bp], "infix_s %s\n", be_obj->name1);
				break;

    case CODEGEN_OBJ_PREFIX_INFIX: sprintf(buff[bp], "prefix_infix %s %s\n",
				    be_obj->name1, be_obj->name2);
		    		break;
	    
    case CODEGEN_OBJ_INFIX_PREFIX: sprintf(buff[bp], "infix_prefix %s %s\n",
				    be_obj->name1, be_obj->name2);
		    		break;

    case CODEGEN_OBJ_PREFIX_ORDINARY: sprintf(buff[bp], "prefix_ord %s %s\n",
				  be_obj->name1, be_obj->name2);
		    		break;

    case CODEGEN_OBJ_FUNREF:	return "funref";

    default:			assert(FALSE);
  }
  return buff[bp];
}


/*****************************************************************************/
/*                                                                           */
/*  Private submodule "variable scoping".                                    */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_OBJ BackendObjNew()                                              */
/*                                                                           */
/*  Return a new codegen object record.                                      */
/*                                                                           */
/*****************************************************************************/

static CODEGEN_OBJ BackendObjNew()
{
  CODEGEN_OBJ res;
  if( first_free_obj != NULL )
  {
    res = first_free_obj;
    first_free_obj = res->next_free_obj;
  }
  else
  {
    GetMemory(res, CODEGEN_OBJ);
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void ObjFree(CODEGEN_OBJ obj)                                            */
/*                                                                           */
/*  Free codegen object obj.                                                 */
/*                                                                           */
/*****************************************************************************/

static void ObjFree(CODEGEN_OBJ obj)
{
  obj->next_free_obj = first_free_obj;
  first_free_obj = obj;
}


/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_OBJ FunctionNew(CODEGEN_OBJ_TYPE type, ASTRING name1,            */
/*    ASTRING name2, BOOLEAN global)                                         */
/*                                                                           */
/*  Make a new codegen function object.  Not directly callable by users.     */
/*                                                                           */
/*****************************************************************************/

static CODEGEN_OBJ FunctionNew(CODEGEN_OBJ_TYPE type, ASTRING name1,
  ASTRING name2, int index, BOOLEAN global)
{
  CODEGEN_OBJ res;
  res = BackendObjNew();
  res->type = type;
  res->name1 = name1;
  res->name2 = name2;
  res->index = index;
  res->global = global;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_OBJ VarNew(CODEGEN_OBJ_TYPE type, ASTRING name)                  */
/*                                                                           */
/*  Make a new local variable with this name.  Not directly callable         */
/*  by users.                                                                */
/*                                                                           */
/*****************************************************************************/

static CODEGEN_OBJ VarNew(CODEGEN_OBJ_TYPE type, ASTRING name)
{
  return FunctionNew(type, name, NULL, 0, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_OBJ FunctionPredef(CODEGEN_OBJ_TYPE type, ASTRING name1,         */
/*    ASTRING name2)                                                         */
/*                                                                           */
/*  Make a predefined function.  These are all of a type to have no          */
/*  index, and they are all global.                                          */
/*                                                                           */
/*****************************************************************************/

static CODEGEN_OBJ FunctionPredef(CODEGEN_OBJ_TYPE type, ASTRING name1,
  ASTRING name2)
{
  return FunctionNew(type, name1, name2, 0, TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_OBJ FunctionAttributeIndexed(ASTRING name)                       */
/*                                                                           */
/*  Return a new function of type attribute-index, i.e. $1->name[$2].        */
/*                                                                           */
/*****************************************************************************/

static CODEGEN_OBJ FunctionAttributeIndexed(ASTRING name)
{
  return FunctionNew(CODEGEN_OBJ_ATTRIBUTE_INDEX, name, NULL, 0, TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  ASTRING ConvertToCIdentifier(USTRING str)                                */
/*                                                                           */
/*  Convert str to a C identifier, by an arbitrary method but preferring     */
/*  to preserve as much of str as possible, except that a fixed set of       */
/*  punctuation character sequences are converted to comprehensible things.  */
/*                                                                           */
/*  The returned value is in malloced memory, not a pointer to any           */
/*  temporary buffer.                                                        */
/*                                                                           */
/*  Macro ascii_letter(c) is TRUE if c is a suitable ASCII character to      */
/*  begin a C identifier (we don't consider _ to be suitable).               */
/*                                                                           */
/*  Macro ascii_id(c) is TRUE if c is a suitable ASCII character to          */
/*  have in an identifier, after the first character.  Confusingly, in       */
/*  addition to the usual letters and digits, this macro accepts [\]^_`      */
/*  characters.  The reasoning here is as follows:                           */
/*                                                                           */
/*      []    will occur only in "identifiers" which are array               */
/*            declarations, where they must be left unmodified               */
/*                                                                           */
/*      \^`   will never occur so it doesn't matter how they are classified  */
/*                                                                           */
/*      _     is acceptable in a C identifier                                */
/*                                                                           */
/*****************************************************************************/

#define ascii_letter(c) (((c) >= 'a' && (c) <= 'z')||((c) >= 'A' && (c) <= 'Z'))
#define ascii_digit(c) ((c) >= '0' && (c) <= '9')
#define ascii_permissive_letter(c) ((c) >= 'A' && (c) <= 'z')
#define ascii_id(c) (ascii_permissive_letter(c) || ascii_digit(c))

typedef SYMTAB(ARRAY_ASTRING) SYMTAB_ASTRING;

static ASTRING ConvertToCIdentifier(USTRING str)
{
  int i;  UCHAR ch;  ASTRING str2;
  static char pr[62] =
    "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789";
  static SYMTAB_ASTRING puncts = NULL;

  /* initialize punctuation sequences if not done already */
  if( puncts == NULL )
  {
    SymInit(&puncts);
    SymInsert(puncts, AStringToUString("="),  "eq",       &str2);
    SymInsert(puncts, AStringToUString("!="), "ne",       &str2);
    SymInsert(puncts, AStringToUString("<>"), "ne",       &str2);
    SymInsert(puncts, AStringToUString("<"),  "lt",       &str2);
    SymInsert(puncts, AStringToUString("<="), "le",       &str2);
    SymInsert(puncts, AStringToUString(">"),  "gt",       &str2);
    SymInsert(puncts, AStringToUString(">="), "ge",       &str2);
    SymInsert(puncts, AStringToUString("+"),  "plus",     &str2);
    SymInsert(puncts, AStringToUString("-"),  "minus",    &str2);
    SymInsert(puncts, AStringToUString("_-"), "neg",      &str2);
    SymInsert(puncts, AStringToUString("*"),  "times",    &str2);
    SymInsert(puncts, AStringToUString("/"),  "divide",   &str2);
    SymInsert(puncts, AStringToUString("%"),  "remainder",&str2);
  }

  /* either find str in puncts or swizzle it */
  if( SymRetrieve(puncts, str, &str2) )
    return str2;
  else
  {
    AFACTORY af = AStringBegin();
    ch = UStringGet(str, 0);
    if( ascii_letter(ch) )
      AStringAddChar(af, (char) ch);
    else
      AStringAddChar(af, pr[ch % 52]);
    for( i = 1;  (ch = UStringGet(str, i)) != UEOF;  i++ )
      if( ascii_id(ch) )
	AStringAddChar(af, (char) ch);
      else
	AStringAddChar(af, pr[ch % 62]);
    return AStringCopy(AStringEnd(af));
  }
}


/*****************************************************************************/
/*                                                                           */
/*  ASTRING PutInScope(ASTRING str, ASYMTAB_INT symbols,                     */
/*    ASYMTAB_INT symbols2)                                                  */
/*                                                                           */
/*  Register str with symbols:  return a name not already in use there.      */
/*                                                                           */
/*  If symbols2 is non-NULL, the registered value must not conflict with     */
/*  anything in symbols2 either.  This is used for local symbols, to         */
/*  make sure they do not inadvertently hide global ones.                    */
/*                                                                           */
/*  This function assumes that str is available for permanent use; str       */
/*  should not be a pointer to a buffer subject to later overwriting.        */
/*                                                                           */
/*****************************************************************************/

static ASTRING PutInScope(ASTRING str, ASYMTAB_INT symbols,
  ASYMTAB_INT symbols2)
{
  int i, val2;  char buff[5];  ASTRING str2;
  str2 = AStringCopy(str);
  i = 1;
  while( ASymRetrieve(symbols, str2, &val2) ||
     (symbols2 != NULL && ASymRetrieve(symbols, str2, &val2)) )
  {
    assert(i < 1000);
    sprintf(buff, "_%d", i);
    str2 = AStringCat(str, buff);
    i += 1;
  }
  ASymInsert(symbols, str2, 0, &val2);
  if( DEBUG2 )
    fprintf(stderr, "PutInScope(\"%s\", -, -) returning \"%s\"\n", str, str2);
  return str2;
}


/*****************************************************************************/
/*                                                                           */
/*  ASTRING PutInScopeWithVal(ASTRING str, int val, ASYMTAB_INT symbols)     */
/*                                                                           */
/*  Register str with symbols.  It may share with an already existing        */
/*  symbol, but only if they have the same value.  This is used for          */
/*  record offsets, since if the offset is the same, the two names cannot    */
/*  lie in the same record, or they mean the same thing if they do.          */
/*                                                                           */
/*****************************************************************************/

static ASTRING PutInScopeWithVal(ASTRING str, int val, ASYMTAB_INT symbols)
{
  int i, val2;  char buff[5];  ASTRING str2;
  str2 = AStringCopy(str);
  i = 1;
  while( (ASymRetrieve(symbols, str2, &val2) && val2 != val) )
  {
    assert(i < 1000);
    sprintf(buff, "_%d", i);
    str2 = AStringCat(str, buff);
    i += 1;
  }
  ASymInsert(symbols, str2, val, &val2);
  return str2;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "executables"                                                  */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_EXEC ExecMake(USTRING exec_name)                                 */
/*                                                                           */
/*  Declare a new executable with this file name.                            */
/*                                                                           */
/*****************************************************************************/

static CODEGEN_EXEC ExecMake(USTRING exec_name)
{
  CODEGEN_EXEC res;
  GetMemory(res, CODEGEN_EXEC);
  res->name = exec_name;
  ArrayInit(&res->files);
  ArrayAddLast(executables, res);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void ExecAddFile(CODEGEN_EXEC exec, CODEGEN_FILE file)                   */
/*                                                                           */
/*  Add file to the list of files to be compiled together to produce exec.   */
/*                                                                           */
/*****************************************************************************/

static void ExecAddFile(CODEGEN_EXEC exec, CODEGEN_FILE file)
{
  int pos;
  assert(!ArrayContains(exec->files, file, &pos));
  ArrayAddLast(exec->files, file);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "comments"                                                     */
/*                                                                           */
/*****************************************************************************/
#define LINE_WIDTH 78

/*****************************************************************************/
/*                                                                           */
/*  void star_line(FILE *fp)                                                 */
/*                                                                           */
/*  Print a comment line containing stars.                                   */
/*                                                                           */
/*****************************************************************************/

static void star_line(FILE *fp)
{
  int i;
  fputs("/*", fp);
  for( i = 0;  i < LINE_WIDTH - 4;  i++ )
    putc('*', fp);
  fputs("*/\n", fp);
}


/*****************************************************************************/
/*                                                                           */
/*  void comment_line(ASTRING str, FILE *fp)                                 */
/*                                                                           */
/*  Print a comment line containing str.                                     */
/*                                                                           */
/*****************************************************************************/

static void comment_line(ASTRING str, FILE *fp)
{
  fprintf(fp, "/*  %-*s  */\n", LINE_WIDTH - 8, str);
}


/*****************************************************************************/
/*                                                                           */
/*  void comment_block(ASTRING str1, ASTRING str2, ASTRING str3, FILE *fp)   */
/*                                                                           */
/*  Make a comment block with up to three lines of text, preceded by         */
/*  two blank lines.                                                         */
/*                                                                           */
/*****************************************************************************/

static void comment_block(ASTRING str1, ASTRING str2, ASTRING str3, FILE *fp)
{
  fputs("\n\n", fp);
  star_line(fp);
  comment_line("", fp);
  if( str1 != NULL ) comment_line(str1, fp);
  if( str2 != NULL ) comment_line(str2, fp);
  if( str3 != NULL ) comment_line(str3, fp);
  comment_line("", fp);
  star_line(fp);
}


/*****************************************************************************/
/*                                                                           */
/*  void CommentLarge(ASTRING str1, ASTRING str2, ASTRING str3,              */
/*    BOOLEAN header, BOOLEAN code)                                          */
/*                                                                           */
/*  Make a large comment with up to three lines of text, in the code         */
/*  file and also in the header file if there is one.                        */
/*                                                                           */
/*  In the header file we add a blank line after the comment, since          */
/*  most other entries in header files do not precede themselves by          */
/*  a blank line, as is the case in code files.                              */
/*                                                                           */
/*****************************************************************************/

static void CommentLarge(ASTRING str1, ASTRING str2, ASTRING str3,
  BOOLEAN header, BOOLEAN code)
{
  if( header )
  {
    assert(hfp != NULL);
    comment_block(str1, str2, str3, hfp);
    fputs("\n", hfp);
  }
  if( code )
    comment_block(str1, str2, str3, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  void CommentSmall(ASTRING str, BOOLEAN header_file, BOOLEAN code_file)   */
/*                                                                           */
/*  Make a small, inline comment in the header and/or code files.            */
/*                                                                           */
/*****************************************************************************/

static void CommentSmall(ASTRING str, BOOLEAN header_file, BOOLEAN code_file)
{
  if( header_file )
    fprintf(hfp, "\n%*s/* %s */\n", indent, "", str);
  if( code_file )
    fprintf(fp, "\n%*s/* %s */\n", indent, "", str);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "files"                                                        */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN FileExists(USTRING file_name)                                    */
/*                                                                           */
/*  Return TRUE if file utf8_file_name exists, but do not leave it open.     */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN FileExists(USTRING file_name)
{
  FILE *fp;
  fp = fopen((ASTRING) UStringToUTF8(file_name), "r");
  if( fp == NULL )
    return FALSE;
  else
  {
    fclose(fp);
    return TRUE;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  FILE *OpenFile(USTRING dir_name, USTRING file_name, USTRING suffix,      */
/*    ASTRING mode, ASTRING comment1, ASTRING comment2)                      */
/*                                                                           */
/*  Open a file with this directory name, file name, and suffix, for         */
/*  reading or writing according to mode.                                    */
/*                                                                           */
/*  If writing, include a comment block containing comment1 and comment2,    */
/*  and move any existing file with the same name to a backup version.       */
/*                                                                           */
/*****************************************************************************/

static FILE *OpenFile(USTRING dir_name, USTRING file_name, USTRING suffix,
  ASTRING mode, BOOLEAN copied, ASTRING comment1, ASTRING comment2)
{
  USTRING backup_full_name, full_name, with_suffix;
  UTF8 utf8_full_name, utf8_backup_full_name;  char buff[200];
  FILE *file_p;

  /* get the full file name */
  with_suffix = UStringCat(file_name, suffix);
  if( dir_name != NULL )
    full_name = UStringCat3(dir_name, AStringToUString(NPC_DIR_SEP),
      with_suffix);
  else
    full_name = with_suffix;
  utf8_full_name = UStringToUTF8(full_name);

  /* if writing and there is an existing file, move it to a backup file */
  if( mode[0] == 'w' && FileExists(full_name) )
  {
    backup_full_name = UStringCat(full_name, AStringToUString("~"));
    utf8_backup_full_name = UStringToUTF8(backup_full_name);
    if( rename((ASTRING) utf8_full_name, (ASTRING) utf8_backup_full_name) != 0 )
    {
      fprintf(stderr, "cannot rename \"%s\" to \"%s\"\n", utf8_full_name,
	utf8_backup_full_name);
      exit(1);
    }
  }

  /* open the file and fail if can't */
  file_p = fopen((ASTRING) utf8_full_name, mode);
  if( file_p == NULL )
  {
    fprintf(stderr, "cannot open file %s for %s\n", (ASTRING) utf8_full_name,
      mode[0] == 'w' ? "writing" : "reading");
    exit(1);
  }

  /* if writing, add header comments to the file */
  if( mode[0] == 'w' )
  {
    sprintf(buff, "%s (%s file)", UStringToUTF8(with_suffix),
      copied ? "copied" : "generated");
    comment_block(buff, comment1, comment2, file_p);
  }
  return file_p;
}


/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_FILE FileMake(USTRING file_name, BOOLEAN system,                 */
/*    BOOLEAN header_only)                                                   */
/*                                                                           */
/*  Make (but do not yet open) a file with the given name.                   */
/*                                                                           */
/*  If system is TRUE, the file is a system file, so that when a request     */
/*  is made to include it, we use <file.h> rather than "file.h".  Such a     */
/*  file cannot be opened for writing, obviously.                            */
/*                                                                           */
/*  If system is FALSE, the file is one of those generated by Nonpareil,     */
/*  so it can be written, and inclusion is done via "file.h".                */
/*                                                                           */
/*  If header_only is TRUE, we are going to be writing only a .h file,       */
/*  not a .c file, so don't open one.                                        */
/*                                                                           */
/*****************************************************************************/

CODEGEN_FILE FileMake(USTRING file_name, BOOLEAN system, BOOLEAN header_only)
{
  CODEGEN_FILE res, res2;

  /* make the file object */
  GetMemory(res, CODEGEN_FILE);
  res->name = file_name;
  res->system = system;
  res->header_only = header_only;
  res->generated = FALSE;
  ArrayInit(&res->hincludes);
  ArrayInit(&res->cincludes);

  /* insert into symbol table and array */
  if( !SymInsert(source_file_table, res->name, res, &res2) )
  {
    fprintf(stderr, "file name %s used twice\n",
      (ASTRING) UStringToUTF8(res->name));
    exit(1);
  }
  ArrayAddLast(source_file_array, res);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void FileBegin(CODEGEN_FILE file)                                        */
/*                                                                           */
/*  Begin writing to a file.                                                 */
/*                                                                           */
/*****************************************************************************/

static void FileBegin(CODEGEN_FILE file, ASTRING comment1, ASTRING comment2)
{
  ASTRING header_seen;

  /* make sure no other file is open, and make this the current file */
  assert(curr_file == NULL);
  curr_file = file;

  /* make sure not already generated */
  assert(!file->generated);
  file->generated = TRUE;

  /* header file, with "header-seen" macro */
  assert(!file->system);
  hfp = OpenFile(code_dir, file->name, AStringToUString(".h"), "w", FALSE,
    comment1, comment2);
  header_seen = AStringCat(ConvertToCIdentifier(file->name), "_header_seen");
  header_seen = PutInScope(header_seen, global_symbols, NULL);
  fprintf(hfp, "#ifndef %s\n", header_seen);
  fprintf(hfp, "#define %s\n", header_seen);

  /* source file, but only if not header_only */
  if( !file->header_only )
    fp = OpenFile(code_dir, file->name, AStringToUString(".c"), "w", FALSE,
      comment1, comment2);
}


/*****************************************************************************/
/*                                                                           */
/*  void HeaderFileInclude(CODEGEN_FILE file)                                */
/*                                                                           */
/*  Print #include "name.h" or #include <name.h> in header file.             */
/*  If not a system file, remember that this has been done.                  */
/*                                                                           */
/*****************************************************************************/

static void HeaderFileInclude(CODEGEN_FILE file)
{
  assert(hfp != NULL);
  if( file->system )
    fprintf(hfp, "#include <%s.h>\n", (ASTRING) UStringToUTF8(file->name));
  else
  {
    fprintf(hfp, "#include \"%s.h\"\n", (ASTRING) UStringToUTF8(file->name));
    ArrayAddLast(curr_file->hincludes, file);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void CodeFileInclude(CODEGEN_FILE file)                                  */
/*                                                                           */
/*  Print #include "name.h" or #include <name.h> in code file.               */
/*  If not a system file, remember that this has been done.                  */
/*                                                                           */
/*****************************************************************************/

static void CodeFileInclude(CODEGEN_FILE file)
{
  assert(fp != NULL);
  if( file->system )
    fprintf(fp, "#include <%s.h>\n", (ASTRING) UStringToUTF8(file->name));
  else
  {
    fprintf(fp, "#include \"%s.h\"\n", (ASTRING) UStringToUTF8(file->name));
    ArrayAddLast(curr_file->cincludes, file);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void FileEnd(CODEGEN_FILE file)                                          */
/*                                                                           */
/*  End a file, opposite of FileBegin().                                     */
/*                                                                           */
/*****************************************************************************/

static void FileEnd(CODEGEN_FILE file)
{
  /* finish off header file */
  assert(hfp != NULL);
  fputs("\n#endif\n", hfp);
  fclose(hfp);
  hfp = NULL;
  curr_file = NULL;

  /* finish off code file */
  if( !file->header_only )
  {
    assert(fp != NULL);
    fclose(fp);
    fp = NULL;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void FileDoCopy(USTRING dir_name, CODEGEN_FILE file, USTRING suffix,     */
/*    ASTRING comment1, ASTRING comment2, ARRAY_CODEGEN_FILE includes)       */
/*                                                                           */
/*  Copy file from dir_name to code_dir.                                     */
/*                                                                           */
/*  If a #include "file.h" occurs, look the file up in the source file       */
/*  table and add it to includes.                                            */
/*                                                                           */
/*****************************************************************************/

static void FileDoCopy(USTRING dir_name, CODEGEN_FILE file, USTRING suffix,
  ASTRING comment1, ASTRING comment2, ARRAY_CODEGEN_FILE includes)
{
  FILE *from_fp, *to_fp;  int ch, prev_ch;  char buff[200], fname[100];
  CODEGEN_FILE include_file;

  /* open source for reading and destination for writing */
  from_fp = OpenFile(dir_name, file->name, suffix, "r", TRUE, NULL, NULL);
  to_fp = OpenFile(code_dir, file->name, suffix, "w", TRUE, comment1, comment2);

  /* copy from to to, taking note of #include lines */
  prev_ch = '\n';
  while( (ch = getc(from_fp)) != EOF )
  {
    if( prev_ch == '\n' && ch == '#' )
    {
      /* found line beginning with #; echo it and check for #include */
      putc(ch, to_fp);
      if( fgets(buff, 200, from_fp) != buff || buff[strlen(buff)-1] != '\n' )
	assert(FALSE);
      if( sscanf(buff, "include \"%[^.\"].h\"", fname) == 1 )
      {
	/* found #include "<fname>.h" */
	if( !SymRetrieve(source_file_table, AStringToUString(fname),
	      &include_file) )
	{
	  fprintf(stderr, "include file name %s.h not declared\n", fname);
	  exit(1);
	}
	ArrayAddLast(includes, include_file);
      }
      fputs(buff, to_fp);
      prev_ch = '\n';
    }
    else
    {
      /* did not find line beginning with #; just echo the character */
      putc(ch, to_fp);
      prev_ch = ch;
    }
  }

  /* close both files */
  fclose(from_fp);
  fclose(to_fp);
}


/*****************************************************************************/
/*                                                                           */
/*  void FileCopy(USTRING dir_name, CODEGEN_FILE file,                       */
/*    ASTRING comment1, ASTRING comment2, BOOLEAN header_only)               */
/*                                                                           */
/*  Include this file by copying it from directory dir_name.                 */
/*                                                                           */
/*****************************************************************************/

static void FileCopy(USTRING dir_name, CODEGEN_FILE file,
  ASTRING comment1, ASTRING comment2)
{
  FileDoCopy(dir_name, file, AStringToUString(".h"), comment1, comment2,
    file->hincludes);
  if( !file->header_only )
    FileDoCopy(dir_name, file, AStringToUString(".c"), comment1, comment2,
      file->cincludes);
}


/*****************************************************************************/
/*                                                                           */
/*  void FlushCurrentFile(void)                                              */
/*                                                                           */
/*  Flush the current output file(s);                                        */
/*                                                                           */
/*****************************************************************************/

void FlushCurrentFile(void)
{
  if( hfp != NULL )
    fflush(hfp);
  if( fp != NULL )
    fflush(fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "types, type tags, structs, and their attributes"              */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_TYPE TypeNew(CODEGEN_TYPE_TYPE type, ASTRING name,               */
/*    int param_count, ASTRING value)                                        */
/*                                                                           */
/*  Return a new codegen type with these attributes.  Not directly           */
/*  callable by the end user.                                                */
/*                                                                           */
/*****************************************************************************/

static CODEGEN_TYPE TypeNew(CODEGEN_TYPE_TYPE type, ASTRING name,
  int param_count, ASTRING value, int field_width)
{
  CODEGEN_TYPE res;
  GetMemory(res, CODEGEN_TYPE);
  assert(name != NULL);
  res->type = type;
  res->name = name;
  res->param_count = param_count;
  res->value = value;
  res->field_width = field_width;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_TYPE TypeMakeImmediate(CODEGEN_TYPE_TYPE type, ASTRING value)    */
/*                                                                           */
/*  Make an immediate type with the given type and C value, needing no       */
/*  typedef.  Not directly callable by the end user.                         */
/*                                                                           */
/*****************************************************************************/

static CODEGEN_TYPE TypeMakeImmediate(CODEGEN_TYPE_TYPE type, ASTRING value,
  int field_width)
{
  return TypeNew(type, value, 0, value, field_width);
}


/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_TYPE TypeMakeObject(USTRING usugg)                               */
/*                                                                           */
/*  Make a type which is a reference to an object, with the given name       */
/*  as a suggestion.                                                         */
/*                                                                           */
/*****************************************************************************/

static CODEGEN_TYPE TypeMakeObject(USTRING usugg)
{
  ASTRING name, struct_tag;
  name = PutInScope(ConvertToCIdentifier(usugg), global_symbols, NULL);
  struct_tag = PutInScope(AStringCat(name, "_rec"), global_symbols, NULL);
  return TypeNew(CODEGEN_TYPE_OBJECT, name, 0, struct_tag,
    sizeof(void *) * CHAR_BIT);
}


/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_TYPE TypeMakeFunctionType(USTRING usugg, int param_count)        */
/*                                                                           */
/*  Make a function type with the given number of parameters.                */
/*                                                                           */
/*****************************************************************************/

static CODEGEN_TYPE TypeMakeFunctionType(USTRING usugg, int param_count)
{
  ASTRING name;
  name = PutInScope(ConvertToCIdentifier(usugg), global_symbols, NULL);
  return TypeNew(CODEGEN_TYPE_FUNCTION, name, param_count, NULL,
    sizeof(void *) * CHAR_BIT);
}


/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_TYPE TypeMakeEnumerated(USTRING usugg, int bits_reqd, CODEGEN be)*/
/*                                                                           */
/*  Make an enumerated type with this suggested name, capable of holding     */
/*  bits_reqd bits.                                                          */
/*                                                                           */
/*****************************************************************************/

static CODEGEN_TYPE TypeMakeEnumerated(USTRING usugg, int bits_reqd, CODEGEN be)
{
  ASTRING name;  CODEGEN_TYPE itype;

  /* find the immediate type that this enumation is equivalent to */
  if( bits_reqd <= sizeof(unsigned char) * CHAR_BIT )
    itype = be->uchar_type;
  else if( bits_reqd <= sizeof(unsigned short) * CHAR_BIT )
    itype = be->ushort_type;
  else if( bits_reqd <= sizeof(unsigned int) * CHAR_BIT )
    itype = be->uint_type;
  else
  {
    fprintf(stderr, "TypeMakeEnumerated: %d bits (max is %d)\n", bits_reqd,
      sizeof(unsigned int) * CHAR_BIT);
    exit(1);
  }

  /* return the equivalent type if anonymous, or else a new enumerated type */
  if( usugg == NULL )
    return itype;
  else
  {
    name = PutInScope(ConvertToCIdentifier(usugg), global_symbols, NULL);
    return TypeNew(CODEGEN_TYPE_ENUMERATED, name, 0, itype->value,
      itype->field_width);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  int TypeFieldWidth(CODEGEN_TYPE type)                                    */
/*                                                                           */
/*  Return the number of bits occupied by a variable or field of this        */
/*  type.                                                                    */
/*                                                                           */
/*****************************************************************************/

static int TypeFieldWidth(CODEGEN_TYPE type)
{
  return type->field_width;
}


/*****************************************************************************/
/*                                                                           */
/*  void TypeDef(CODEGEN_TYPE type)                                          */
/*                                                                           */
/*  Print the appropriate typedef for this type; possibly nothing.           */
/*                                                                           */
/*****************************************************************************/

static void TypeDef(CODEGEN_TYPE type)
{
  int i;
  switch( type->type )
  {
    case CODEGEN_TYPE_CHAR:
    case CODEGEN_TYPE_UCHAR:
    case CODEGEN_TYPE_SHORT:
    case CODEGEN_TYPE_USHORT:
    case CODEGEN_TYPE_INT:
    case CODEGEN_TYPE_UINT:
    case CODEGEN_TYPE_FLOAT:
    case CODEGEN_TYPE_DOUBLE:
    case CODEGEN_TYPE_VOID:
    case CODEGEN_TYPE_VOIDP:
    case CODEGEN_TYPE_CHARP:

      /* no typedef required, because they stand for themselves */
      break;


    case CODEGEN_TYPE_OBJECT:

      fprintf(hfp, "typedef struct %s *%s;\n", type->value, type->name);
      break;


    case CODEGEN_TYPE_FUNCTION:

      fprintf(hfp, "typedef void *(*%s)(", type->name);
      for( i = 0;  i < type->param_count;  i++ )
	fprintf(hfp, "%svoid *", i > 0 ? ", " : "");
      fputs(");\n", hfp);
      break;


    case CODEGEN_TYPE_ENUMERATED:

      fprintf(hfp, "typedef %s %s;\n", type->value, type->name);
      break;


    default:

      assert(FALSE);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  ASTRING TypeShow(CODEGEN_TYPE type)                                      */
/*                                                                           */
/*  Return this type as a string.                                            */
/*                                                                           */
/*****************************************************************************/

static ASTRING TypeShow(CODEGEN_TYPE type)
{
  return type->name != NULL ? type->name : type->value;
}


/*****************************************************************************/
/*                                                                           */
/*  void StructBegin(CODEGEN_TYPE type)                                      */
/*                                                                           */
/*  Begin printing the struct declaration for this type.                     */
/*                                                                           */
/*****************************************************************************/

void StructBegin(CODEGEN_TYPE type)
{
  fprintf(hfp, "struct %s {\n", type->value);
}


/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_OBJ AttrMake(ASTRING name)                                       */
/*                                                                           */
/*  Make an attribute function ($1->name).                                   */
/*                                                                           */
/*****************************************************************************/

static CODEGEN_OBJ AttrMake(ASTRING name)
{
  return FunctionNew(CODEGEN_OBJ_ATTRIBUTE, name, NULL, -1, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_OBJ StructItemMake(USTRING usugg, int offset)                    */
/*                                                                           */
/*  Make a struct item and return it as a codegen function.                  */
/*                                                                           */
/*****************************************************************************/

CODEGEN_OBJ StructItemMake(USTRING usugg, int offset)
{
  ASTRING name =
    PutInScopeWithVal(ConvertToCIdentifier(usugg), offset, struct_symbols);
  return AttrMake(name);
}


/*****************************************************************************/
/*                                                                           */
/*  void StructItem(CODEGEN_OBJ attr, CODEGEN_TYPE type)                     */
/*                                                                           */
/*  Add attr to the current struct, with this type.                          */
/*                                                                           */
/*****************************************************************************/

void StructItem(CODEGEN_OBJ attr, CODEGEN_TYPE type)
{
  ASTRING gap = type->name[strlen(type->name) - 1] == '*' ? "" : " ";
  assert(attr->type == CODEGEN_OBJ_ATTRIBUTE);
  fprintf(hfp, "  %s%s%s;\n", type->name, gap, attr->name1);
}


/*****************************************************************************/
/*                                                                           */
/*  void StructGap(int gap_bit_width)                                        */
/*                                                                           */
/*  Add a gap of the given width to the current struct.                      */
/*                                                                           */
/*****************************************************************************/

void StructGap(int gap_bit_width)
{
  fprintf(hfp, "  unsigned : %d;\n", gap_bit_width);
}


/*****************************************************************************/
/*                                                                           */
/*  void StructEnd(CODEGEN_TYPE type)                                        */
/*                                                                           */
/*  End the current struct.                                                  */
/*                                                                           */
/*****************************************************************************/

void StructEnd(CODEGEN_TYPE type)
{
  fputs("};\n\n", hfp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "indenting and blocks"                                         */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void IndentBegin(void)                                                   */
/*                                                                           */
/*  Begin an indented section.                                               */
/*                                                                           */
/*****************************************************************************/

void IndentBegin(void)
{
  indent += 2;
}


/*****************************************************************************/
/*                                                                           */
/*  void IndentEnd(void)                                                     */
/*                                                                           */
/*  End an indented section.                                                 */
/*                                                                           */
/*****************************************************************************/

void IndentEnd(void)
{
  indent -= 2;
}


/*****************************************************************************/
/*                                                                           */
/*  void BlockBegin(void)                                                    */
/*                                                                           */
/*  Begin a block.  If a block has currently just been begun, don't          */
/*  actually make another.                                                   */
/*                                                                           */
/*****************************************************************************/

static void BlockBegin(void)
{
  if( start_block )
    ArrayAddLast(blocks_inserted, FALSE);
  else
  {
    fprintf(fp, "%*s{\n", indent, "");
    IndentBegin();
    start_block = TRUE;
    ArrayAddLast(blocks_inserted, TRUE);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void BlockEnd(void)                                                      */
/*                                                                           */
/*  End a block.                                                             */
/*                                                                           */
/*****************************************************************************/

static void BlockEnd(void)
{
  if( ArrayRemoveLast(blocks_inserted) )
  {
    IndentEnd();
    fprintf(fp, "%*s}\n", indent, "");
    start_block = FALSE;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "assignment statements".                                       */
/*                                                                           */
/*****************************************************************************/

void AsstOpBegin(CODEGEN_OBJ op)
{
  fprintf(fp, "%*s", indent, "");
  start_block = FALSE;
}


void AsstOpContinue(CODEGEN_OBJ op)
{
  fprintf(fp, " %s= ", op->name1);
}


void AsstOpEnd(CODEGEN_OBJ op)
{
  fputs(";\n", fp);
}


void AsstBegin(void)
{
  fprintf(fp, "%*s", indent, "");
  start_block = FALSE;
}


void AsstContinue(void)
{
  fputs(" = ", fp);
}


void AsstEnd(void)
{
  fputs(";\n", fp);
}

void AsstArrayInit(CODEGEN_OBJ arr, CODEGEN_OBJ index_var, int from_index,
  int to_index, CODEGEN_OBJ init_var)
{
  fprintf(fp, "%*sfor( %s = %d;  %s <= %d;  %s++ )\n", indent, "",
    index_var->name1, from_index, index_var->name1, to_index, index_var->name1);
  fprintf(fp, "%*s  %s->elems[%s] = %s;\n", indent, "", arr->name1,
    index_var->name1, init_var->name1);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "other one-line statements"                                    */
/*                                                                           */
/*****************************************************************************/

void StmtBegin(void)
{
  fprintf(fp, "%*s", indent, "");
  start_block = FALSE;
}


void StmtEnd(void)
{
  fputs(";\n", fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "functions and function calls, including predefined operators" */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_OBJ FunctionMake(ASTRING name, BOOLEAN has_params,BOOLEAN global)*/
/*                                                                           */
/*  Make a function, using name as suggestion.  Parameter has_params tells   */
/*  whether the function has parameters or not (i.e. whether it is to be     */
/*  invoked with following parentheses or not); global tells whether it      */
/*  is global to the whole generated code, or private to the current file.   */
/*                                                                           */
/*****************************************************************************/

CODEGEN_OBJ FunctionMake(ASTRING name, BOOLEAN has_params, BOOLEAN global)
{
  return FunctionNew(has_params ? CODEGEN_OBJ_ORDINARY : CODEGEN_OBJ_VAR,
    PutInScope(name, global_symbols, NULL), NULL, 0, global);
}


/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_OBJ FunctionMake2(USTRING name1, USTRING name2,                  */
/*    BOOLEAN has_params, BOOLEAN global)                                    */
/*                                                                           */
/*  Make a codegen function, whose name ideally would be <name1>_<name2>,    */
/*  or just <name1> if name2 is NULL.                                        */
/*                                                                           */
/*  If has_params is TRUE, the function is a true function with zero or      */
/*  more parameters, to be called with parentheses.  Otherwise it is         */
/*  actually a global variable, not function, and is called without them.    */
/*                                                                           */
/*  If global is TRUE, the function is a global one, so entry will be made   */
/*  in the header file as well as the code file;  otherwise, the function    */
/*  is local, so make no header in the header file, and prepend "static"     */
/*  to the code file.                                                        */
/*                                                                           */
/*****************************************************************************/

CODEGEN_OBJ FunctionMake2(USTRING name1, USTRING name2,
  BOOLEAN has_params, BOOLEAN global)
{
  ASTRING name = ConvertToCIdentifier(name1);
  if( name2 != NULL )
    name = AStringCat3(name, "_", ConvertToCIdentifier(name2));
  return FunctionMake(name, has_params, global);
}


/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_OBJ FunctionMake3(USTRING name1, USTRING name2, USTRING name3,   */
/*    BOOLEAN has_params, BOOLEAN global)                                    */
/*                                                                           */
/*  Like FunctionMake2 but with a three-part name.                           */
/*                                                                           */
/*****************************************************************************/

CODEGEN_OBJ FunctionMake3(USTRING name1, USTRING name2, USTRING name3,
  BOOLEAN has_params, BOOLEAN global)
{
  ASTRING name = ConvertToCIdentifier(name1);
  name = AStringCat3(name, "_", ConvertToCIdentifier(name2));
  name = AStringCat3(name, "_", ConvertToCIdentifier(name3));
  return FunctionMake(name, has_params, global);
}


/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_OBJ FunctionMakeFunRef(ASTRING name)                             */
/*                                                                           */
/*  Make a function with this name that is a function reference call,        */
/*  to be invoked as "(*($1))($2, ... , $n)".                                */
/*                                                                           */
/*****************************************************************************/

CODEGEN_OBJ FunctionMakeFunRef(ASTRING name)
{
  return FunctionNew(CODEGEN_OBJ_FUNREF,
    PutInScope(name, global_symbols, NULL), NULL, 0, TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  void FunctionBegin(CODEGEN_OBJ fn, CODEGEN_TYPE type)                    */
/*                                                                           */
/*  Begin the code generation of the declaration of fn.  Only ordinary       */
/*  functions and variables need to be generated.                            */
/*                                                                           */
/*****************************************************************************/

static void FunctionBegin(CODEGEN_OBJ fn, CODEGEN_TYPE type)
{
  ASTRING gap, paren;
  assert(fn->type == CODEGEN_OBJ_ORDINARY || fn->type == CODEGEN_OBJ_VAR);
  gap = type->name[strlen(type->name) - 1] == '*' ? "" : " ";
  paren = (fn->type == CODEGEN_OBJ_ORDINARY ? "(" : "");
  if( fn->global )
  {
    fprintf(hfp, "extern %s%s%s%s", type->name, gap, fn->name1, paren);
    fprintf(fp, "\n%s%s%s%s", type->name, gap, fn->name1, paren);
  }
  else
    fprintf(fp, "\nstatic %s%s%s%s", type->name, gap, fn->name1, paren);
  ASymClear(local_symbols);
  formals_count = 0;
  start_block = FALSE;
}


/*****************************************************************************/
/*                                                                           */
/*  void FunctionFormal(CODEGEN_OBJ fn,CODEGEN_OBJ var_fn,CODEGEN_TYPE type) */
/*                                                                           */
/*  Add formal parameter var with type type to fn.                           */
/*                                                                           */
/*****************************************************************************/

static void FunctionFormal(CODEGEN_OBJ fn, CODEGEN_OBJ var_fn, CODEGEN_TYPE type)
{
  ASTRING sep, gap, var;
  if( DEBUG3 )
    fprintf(stderr, "[ FunctionFormal(%s, %s, %s)\n", BackendFnShow(fn),
      BackendFnShow(var_fn), TypeShow(type));
  assert(fn->type == CODEGEN_OBJ_ORDINARY);
  assert(var_fn == NULL || var_fn->type == CODEGEN_OBJ_VAR);
  sep = (formals_count == 0 ? "" : formals_count % 5 == 3 ? ",\n  " : ", ");
  gap = var_fn == NULL || type->name[strlen(type->name) - 1] == '*' ? "" : " ";
  var = var_fn == NULL ? "" : var_fn->name1;
  if( fn->global )
    fprintf(hfp, "%s%s%s%s", sep, type->name, gap, var);
  fprintf(fp, "%s%s%s%s", sep, type->name, gap, var);
  formals_count++;
  if( DEBUG3 )
    fprintf(stderr, "] FunctionFormal returning\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void FunctionContinue(CODEGEN_OBJ fn)                                    */
/*                                                                           */
/*  Called after the last formal parameter and before the function body.     */
/*                                                                           */
/*****************************************************************************/

static void FunctionContinue(CODEGEN_OBJ fn)
{
  if( fn->type == CODEGEN_OBJ_ORDINARY )
  {
    if( fn->global )
      fputs(");\n", hfp);
    fprintf(fp, ")\n");
    BlockBegin();
  }
  else
  {
    if( fn->global )
      fputs(";\n", hfp);
    fputs(";\n", fp);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void FunctionEnd(CODEGEN_OBJ fn)                                         */
/*                                                                           */
/*  End fn.                                                                  */
/*                                                                           */
/*****************************************************************************/

static void FunctionEnd(CODEGEN_OBJ fn)
{
  if( fn->type == CODEGEN_OBJ_ORDINARY )
    BlockEnd();
}


/*****************************************************************************/
/*                                                                           */
/*  void PrototypeBegin(CODEGEN_OBJ fn, CODEGEN_TYPE type)                   */
/*                                                                           */
/*  Begin the declaration of a function prototype (i.e. just the header,     */
/*  with nameless parameters).                                               */
/*                                                                           */
/*****************************************************************************/

static void PrototypeBegin(CODEGEN_OBJ fn, CODEGEN_TYPE type)
{
  assert(fn->type == CODEGEN_OBJ_ORDINARY);
  assert(!fn->global);
  FunctionBegin(fn, type);
}


/*****************************************************************************/
/*                                                                           */
/*  void PrototypeFormal(CODEGEN_OBJ fn, CODEGEN_TYPE type)                  */
/*                                                                           */
/*  Add one parameter to the prototype declaration of fn.  Only a type       */
/*  is printed, not a name.                                                  */
/*                                                                           */
/*****************************************************************************/

static void PrototypeFormal(CODEGEN_OBJ fn, CODEGEN_TYPE type)
{
  FunctionFormal(fn, NULL, type);
}


/*****************************************************************************/
/*                                                                           */
/*  void PrototypeEnd(CODEGEN_OBJ fn)                                        */
/*                                                                           */
/*  Finish off the prototype declaration of fn.                              */
/*                                                                           */
/*****************************************************************************/

static void PrototypeEnd(CODEGEN_OBJ fn)
{
  fputs(");\n", fp);
}


/*****************************************************************************/
/*                                                                           */
/*  void CallBegin(CODEGEN_OBJ fn)                                           */
/*                                                                           */
/*  Begin a call on fn.                                                      */
/*                                                                           */
/*****************************************************************************/

static void CallBegin(CODEGEN_OBJ fn)
{
  switch( fn->type )
  {

    case CODEGEN_OBJ_CONST:
    case CODEGEN_OBJ_VAR:

      fputs(fn->name1, fp);
      break;


    case CODEGEN_OBJ_VAR_INDEX:

      fprintf(fp, "%s->%s[%d]", fn->name1, fn->name2, fn->index);
      break;


    case CODEGEN_OBJ_IDENTITY:

      break;


    case CODEGEN_OBJ_ORDINARY:

      fprintf(fp, "%s(", fn->name1);
      break;


    case CODEGEN_OBJ_ATTRIBUTE:

      break;


    case CODEGEN_OBJ_INDEX:

      break;


    case CODEGEN_OBJ_ATTRIBUTE_INDEX:

      break;


    case CODEGEN_OBJ_PREFIX:

      fprintf(fp, "(%s ", fn->name1);
      break;


    case CODEGEN_OBJ_INFIX:

      fputs("(", fp);
      break;


    case CODEGEN_OBJ_INFIX_SEQ:

      fputs("(", fp);
      infix_seq_count = 0;
      break;


    case CODEGEN_OBJ_PREFIX_INFIX:

      fprintf(fp, "(%s (", fn->name1);
      break;


    case CODEGEN_OBJ_INFIX_PREFIX:

      fputs("(", fp);
      break;


    case CODEGEN_OBJ_PREFIX_ORDINARY:

      fprintf(fp, "%s %s(", fn->name1, fn->name2);
      break;


    case CODEGEN_OBJ_FUNREF:

      /* (*($1))($2, ... , $n)             */
      fputs("(*(", fp);
      break;


    case CODEGEN_OBJ_RETURN:
    case CODEGEN_OBJ_NO_RETURN:
    default:

      fflush(fp);
      assert(FALSE);
      break;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void CallContinue(CODEGEN_OBJ fn, int i)                                 */
/*                                                                           */
/*  Between parameters in a call on fn; parameter i is next.                 */
/*                                                                           */
/*****************************************************************************/

static void CallContinue(CODEGEN_OBJ fn, int i)
{
  switch( fn->type )
  {

    case CODEGEN_OBJ_CONST:
    case CODEGEN_OBJ_VAR:
    case CODEGEN_OBJ_VAR_INDEX:

      break;


    case CODEGEN_OBJ_IDENTITY:

      break;


    case CODEGEN_OBJ_ORDINARY:
    case CODEGEN_OBJ_PREFIX_ORDINARY:


      fputs(", ", fp);
      break;


    case CODEGEN_OBJ_ATTRIBUTE:
    case CODEGEN_OBJ_PREFIX:

      assert(FALSE);
      break;


    case CODEGEN_OBJ_INDEX:

      fputs("[", fp);
      break;


    case CODEGEN_OBJ_ATTRIBUTE_INDEX:

      fprintf(fp, "->%s[", fn->name1);
      break;

    case CODEGEN_OBJ_INFIX:

      fprintf(fp, " %s ", fn->name1);
      break;


    case CODEGEN_OBJ_INFIX_SEQ:

      if( ++infix_seq_count > 0 )
	fprintf(fp, "\n%*s   ", indent, "");
      fprintf(fp, " %s ", fn->name1);
      break;


    case CODEGEN_OBJ_PREFIX_INFIX:

      fprintf(fp, " %s ", fn->name2);
      break;


    case CODEGEN_OBJ_INFIX_PREFIX:

      fprintf(fp, " %s (%s ", fn->name1, fn->name2);
      break;


    case CODEGEN_OBJ_FUNREF:

      fputs(i == 1 ? "))(" : ", ", fp);
      break;


    case CODEGEN_OBJ_RETURN:
    case CODEGEN_OBJ_NO_RETURN:
    default:

      assert(FALSE);
      break;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void CallEnd(CODEGEN_OBJ fn)                                             */
/*                                                                           */
/*  Between parameters in a call on fn                                       */
/*                                                                           */
/*****************************************************************************/

static void CallEnd(CODEGEN_OBJ fn)
{
  switch( fn->type )
  {

    case CODEGEN_OBJ_CONST:
    case CODEGEN_OBJ_VAR:
    case CODEGEN_OBJ_VAR_INDEX:

      break;


    case CODEGEN_OBJ_IDENTITY:

      break;


    case CODEGEN_OBJ_ORDINARY:
    case CODEGEN_OBJ_PREFIX_ORDINARY:

      fputs(")", fp);
      break;


    case CODEGEN_OBJ_ATTRIBUTE:

      fprintf(fp, "->%s", fn->name1);
      break;


    case CODEGEN_OBJ_INDEX:
    case CODEGEN_OBJ_ATTRIBUTE_INDEX:

      fputs("]", fp);
      break;


    case CODEGEN_OBJ_PREFIX:
    case CODEGEN_OBJ_INFIX:
    case CODEGEN_OBJ_INFIX_SEQ:

      fputs(")", fp);
      break;


    case CODEGEN_OBJ_PREFIX_INFIX:

      fputs("))", fp);
      break;


    case CODEGEN_OBJ_INFIX_PREFIX:

      fputs("))", fp);
      break;


    case CODEGEN_OBJ_FUNREF:

      fputs(")", fp);
      break;


    case CODEGEN_OBJ_RETURN:
    case CODEGEN_OBJ_NO_RETURN:
    default:

      assert(FALSE);
      break;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "variables (creation, declaration, assignment, invocation)"    */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_OBJ VarMake(ASTRING asugg, USTRING usugg)                        */
/*                                                                           */
/*  Make a new variable, unique in the local scope and not conflicting with  */
/*  any symbol in the global scope.                                          */
/*                                                                           */
/*****************************************************************************/

static CODEGEN_OBJ VarMake(ASTRING asugg, USTRING usugg)
{
  ASTRING name;
  name = usugg != NULL ? ConvertToCIdentifier(usugg) :
    asugg != NULL ? asugg : "tmp";
  name = PutInScope(name, local_symbols, global_symbols);
  return FunctionNew(CODEGEN_OBJ_VAR, name, NULL, -1, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_OBJ VarIndexedAttributeMake(CODEGEN_OBJ be_obj, ASTRING attr,    */
/*    int index)                                                             */
/*                                                                           */
/*  Make a variable representing the value                                   */
/*                                                                           */
/*    be_obj->attr[index]                                                    */
/*                                                                           */
/*****************************************************************************/

CODEGEN_OBJ VarIndexedAttributeMake(CODEGEN_OBJ be_obj, ASTRING attr, int index)
{
  return FunctionNew(CODEGEN_OBJ_VAR_INDEX, be_obj->name1, attr, index, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*  void VarIndexedAttributeUpdateIndex(CODEGEN_OBJ be_obj, int index)       */
/*                                                                           */
/*  Update the index of the indexed attribute variable be_obj, to index.     */
/*                                                                           */
/*****************************************************************************/

void VarIndexedAttributeUpdateIndex(CODEGEN_OBJ be_obj, int index)
{
  assert(be_obj->type == CODEGEN_OBJ_VAR_INDEX);
  be_obj->index = index;
}


/*****************************************************************************/
/*                                                                           */
/*  void VarDeclare(CODEGEN_OBJ var, CODEGEN_TYPE type)                      */
/*                                                                           */
/*  Declare var.                                                             */
/*                                                                           */
/*****************************************************************************/

void VarDeclare(CODEGEN_OBJ var, CODEGEN_TYPE type)
{
  ASTRING gap;
  assert(var->type == CODEGEN_OBJ_VAR);
  gap = type->name[strlen(type->name) - 1] == '*' ? "" : " ";
  fprintf(fp, "%*s%s%s%s;\n", indent, "", type->name, gap, var->name1);
}


/*****************************************************************************/
/*                                                                           */
/*  void VarDeclare2(CODEGEN_OBJ var1, CODEGEN_OBJ var2, CODEGEN_TYPE type)  */
/*                                                                           */
/*  Declare var1 and var2.                                                   */
/*                                                                           */
/*****************************************************************************/

void VarDeclare2(CODEGEN_OBJ v1, CODEGEN_OBJ v2, CODEGEN_TYPE type)
{
  assert(v1->type == CODEGEN_OBJ_VAR);
  assert(v2->type == CODEGEN_OBJ_VAR);
  if( type->name[strlen(type->name) - 1] == '*' )
    fprintf(fp, "%*s%s%s, *%s;\n", indent, "", type->name,v1->name1,v2->name1);
  else
    fprintf(fp, "%*s%s %s, %s;\n", indent, "", type->name,v1->name1,v2->name1);
}


/*****************************************************************************/
/*                                                                           */
/*  void VarDeclareAndAsstBegin(CODEGEN_OBJ var, CODEGEN_TYPE type)          */
/*                                                                           */
/*  Declare var and begin an assignment to it.                               */
/*                                                                           */
/*****************************************************************************/

void VarDeclareAndAsstBegin(CODEGEN_OBJ var, CODEGEN_TYPE type)
{
  ASTRING gap;
  assert(var->type == CODEGEN_OBJ_VAR);
  gap = type->name[strlen(type->name) - 1] == '*' ? "" : " ";
  fprintf(fp, "%*s%s%s%s = ", indent, "", type->name, gap, var->name1);
  start_block = FALSE;
}


/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_OBJ VarMakeAndDeclare(ASTRING asugg, USTRING usugg,              */
/*    CODEGEN_TYPE type)                                                     */
/*                                                                           */
/*  Make a new variable and declare it.                                      */
/*                                                                           */
/*****************************************************************************/

static CODEGEN_OBJ VarMakeAndDeclare(ASTRING asugg, USTRING usugg,
  CODEGEN_TYPE type)
{
  CODEGEN_OBJ res = VarMake(asugg, usugg);
  VarDeclare(res, type);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_OBJ VarMakeAndDeclareAndAsstBegin(ASTRING asugg, USTRING usugg,  */
/*    CODEGEN_TYPE type)                                                     */
/*                                                                           */
/*  Make a new variable, declare it, and begin to assign to it.              */
/*                                                                           */
/*****************************************************************************/

static CODEGEN_OBJ VarMakeAndDeclareAndAsstBegin(ASTRING asugg, USTRING usugg,
  CODEGEN_TYPE type)
{
  CODEGEN_OBJ res = VarMake(asugg, usugg);
  VarDeclareAndAsstBegin(res, type);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void Variable(CODEGEN_OBJ var)                                           */
/*                                                                           */
/*  Invoke var.  If a true function is invoked by this procedure, it will    */
/*  print its name only.  This is needed when finding the address of a       */
/*  true function.                                                           */
/*                                                                           */
/*****************************************************************************/

static void Variable(CODEGEN_OBJ var)
{
  switch( var->type )
  {
    case CODEGEN_OBJ_CONST:
    case CODEGEN_OBJ_VAR:
    case CODEGEN_OBJ_ORDINARY:

      fputs(var->name1, fp);
      break;


    case CODEGEN_OBJ_VAR_INDEX:

      fprintf(fp, "%s->%s[%d]", var->name1, var->name2, var->index);
      break;


    case CODEGEN_OBJ_ATTRIBUTE:
    case CODEGEN_OBJ_INDEX:
    case CODEGEN_OBJ_ATTRIBUTE_INDEX:
    case CODEGEN_OBJ_PREFIX:
    case CODEGEN_OBJ_INFIX:
    case CODEGEN_OBJ_INFIX_SEQ:
    case CODEGEN_OBJ_PREFIX_INFIX:
    case CODEGEN_OBJ_INFIX_PREFIX:
    case CODEGEN_OBJ_PREFIX_ORDINARY:
    case CODEGEN_OBJ_RETURN:
    case CODEGEN_OBJ_NO_RETURN:
    default:

      assert(FALSE);
      break;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void VarOpAsstBegin(CODEGEN_OBJ var, CODEGEN_OBJ op)                     */
/*                                                                           */
/*  Begin an assignment to var utilizing an accumulation of op.              */
/*                                                                           */
/*****************************************************************************/

static void VarOpAsstBegin(CODEGEN_OBJ var, CODEGEN_OBJ op)
{
  assert(var != NULL && op != NULL);
  assert(var->type != CODEGEN_OBJ_RETURN);
  assert(var->type != CODEGEN_OBJ_NO_RETURN);
  AsstOpBegin(op);
  Variable(var);
  AsstOpContinue(op);
}


/*****************************************************************************/
/*                                                                           */
/*  void VarAsstBegin(CODEGEN_VAR var)                                       */
/*                                                                           */
/*  Begin an assignment to var, using just ordinary =.                       */
/*                                                                           */
/*****************************************************************************/

static void VarAsstBegin(CODEGEN_OBJ var)
{
  AsstBegin();
  if( var->type == CODEGEN_OBJ_RETURN )
    fputs("return ", fp);
  else if( var->type != CODEGEN_OBJ_NO_RETURN )
  {
    Variable(var);
    AsstContinue();
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void VarAsst2Begin(CODEGEN_OBJ var1, CODEGEN_OBJ var2)                   */
/*                                                                           */
/*  Begin a double assignment, to var1 and var2, using just ordinary =.      */
/*  If var2 is NULL, this is just an ordinary assignment after all.          */
/*                                                                           */
/*****************************************************************************/

static void VarAsst2Begin(CODEGEN_OBJ var1, CODEGEN_OBJ var2)
{
  if( var2 == NULL )
    VarAsstBegin(var1);
  else
  {
    AsstBegin();
    Variable(var1);
    AsstContinue();
    Variable(var2);
    AsstContinue();
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void VarUnDeclare(CODEGEN_OBJ var)                                       */
/*                                                                           */
/*  Here var is going out of scope, so remove it from the local symbols.     */
/*                                                                           */
/*****************************************************************************/

static void VarUnDeclare(CODEGEN_OBJ var)
{
  assert(var->type == CODEGEN_OBJ_VAR);
  if( !ASymDelete(local_symbols, var->name1) )
    assert(FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*  ASTRING VarShow(CODEGEN_OBJ var)                                         */
/*                                                                           */
/*  Show this variable.  For debug use only.                                 */
/*                                                                           */
/*****************************************************************************/

static ASTRING VarShow(CODEGEN_OBJ var)
{
  return var == NULL ? "(NULL)" :
    var->type == CODEGEN_OBJ_RETURN ? "return" : var->name1;
}


/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_OBJ ConstMakeAString(ASTRING asugg, USTRING usugg,               */
/*    ASTRING value, BOOLEAN global)                                         */
/*                                                                           */
/*  Create a codegen constant whose name follows the given suggestions,      */
/*  unique to the global scope, with the given value and global status       */
/*  (determines whether to write in header file or just code file).          */
/*  The value is an ASTRING, including the enclosing quotes.                 */
/*                                                                           */
/*****************************************************************************/

CODEGEN_OBJ ConstMakeAString(ASTRING asugg, USTRING usugg,
  ASTRING value, BOOLEAN global)
{
  CODEGEN_OBJ res;  AFACTORY af;
  res = BackendObjNew();
  res->type = CODEGEN_OBJ_CONST;
  res->name1 = usugg != NULL ? ConvertToCIdentifier(usugg) :
    asugg != NULL ? asugg : "tmp";
  res->name1 = PutInScope(res->name1, global_symbols, NULL);
  af = AStringBegin();
  AStringAddChar(af, '"');
  AStringAddAString(af, value);
  AStringAddChar(af, '"');
  res->name2 = AStringCopy(AStringEnd(af));
  res->index = -1;
  res->global = global;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_OBJ ConstMakeInt2(USTRING name1, USTRING name2,                  */
/*    int value, BOOLEAN global)                                             */
/*                                                                           */
/*  Create a codegen constant whose name follows the given suggestions,      */
/*  unique to the global scope, with the given value and global status       */
/*  (determines whether to write in header file or just code file).          */
/*  The value is the given long integer.                                     */
/*                                                                           */
/*****************************************************************************/

static CODEGEN_OBJ ConstMakeInt2(USTRING name1, USTRING name2,
  int value, BOOLEAN global)
{
  CODEGEN_OBJ res;  char buff[12];
  res = BackendObjNew();
  res->type = CODEGEN_OBJ_CONST;
  if( name2 == NULL )
    res->name1 = ConvertToCIdentifier(name1);
  else
    res->name1 = ConvertToCIdentifier(UStringCat3(name1,underscore_str,name2));
  sprintf(buff, "%d", value);
  res->name2 = AStringCopy(buff);
  res->index = -1;
  res->global = global;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_OBJ ConstMakeInt(ASTRING asugg, USTRING usugg,                   */
/*    int value, BOOLEAN global)                                             */
/*                                                                           */
/*  Create a codegen constant whose name follows the given suggestions,      */
/*  unique to the global scope, with the given value and global status       */
/*  (determines whether to write in header file or just code file).          */
/*  The value is the given long integer.                                     */
/*                                                                           */
/*****************************************************************************/

static CODEGEN_OBJ ConstMakeInt(ASTRING asugg, USTRING usugg,
  int value, BOOLEAN global)
{
  CODEGEN_OBJ res;  char buff[12];
  res = BackendObjNew();
  res->type = CODEGEN_OBJ_CONST;
  res->name1 = usugg != NULL ? ConvertToCIdentifier(usugg) :
    asugg != NULL ? asugg : "tmp";
  res->name1 = PutInScope(res->name1, global_symbols, NULL);
  sprintf(buff, "%d", value);
  res->name2 = AStringCopy(buff);
  res->index = -1;
  res->global = global;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void ConstDeclare(CODEGEN_OBJ cnst)                                      */
/*                                                                           */
/*  Declare cnst.                                                            */
/*                                                                           */
/*****************************************************************************/

static void ConstDeclare(CODEGEN_OBJ cnst)
{
  FILE *file_p = (cnst->global ? hfp : fp);
  assert(cnst->type == CODEGEN_OBJ_CONST);
  fprintf(file_p, "#define %s %s\n", cnst->name1, cnst->name2);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "if statements and conditional expressions"                    */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void IfStmtBegin(void)                                                   */
/*                                                                           */
/*  Begin an if statement.                                                   */
/*                                                                           */
/*****************************************************************************/

static void IfStmtBegin(void)
{
  fprintf(fp, "%*sif( ", indent, "");
  start_block = FALSE;
}


/*****************************************************************************/
/*                                                                           */
/*  void IfStmtElseIf(void)                                                  */
/*                                                                           */
/*  Begin an "else if" part of an if statement.                              */
/*                                                                           */
/*****************************************************************************/

static void IfStmtElseIf(void)
{
  fprintf(fp, "%*selse if( ", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  void IfStmtContinue(void)                                                */
/*                                                                           */
/*  Called between the "if" or "else if" test and the start of the action    */
/*  for that case.                                                           */
/*                                                                           */
/*****************************************************************************/

static void IfStmtContinue(void)
{
  fputs(" )\n", fp);
}


/*****************************************************************************/
/*                                                                           */
/*  void IfStmtElse(void)                                                    */
/*                                                                           */
/*  Begin an "else" part of an if statement.                                 */
/*                                                                           */
/*****************************************************************************/

static void IfStmtElse(void)
{
  fprintf(fp, "%*selse\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  void IfStmtEnd(void)                                                     */
/*                                                                           */
/*  End this if statement.                                                   */
/*                                                                           */
/*****************************************************************************/

static void IfStmtEnd(void)
{
  /* nothing to do if we are not tracking state */
}


/*****************************************************************************/
/*                                                                           */
/*  void IfSmallBegin(void)                                                  */
/*                                                                           */
/*  Begin a small if statement, utilizing ?: syntax.                         */
/*                                                                           */
/*****************************************************************************/

static void IfSmallBegin(void)
{
  fputs("(", fp);
}


/*****************************************************************************/
/*                                                                           */
/*  void IfSmallElseIf(void)                                                 */
/*                                                                           */
/*  Begin an "else if" part of a small if statement.                         */
/*                                                                           */
/*****************************************************************************/

static void IfSmallElseIf(void)
{
  fputs(" : ", fp);
}


/*****************************************************************************/
/*                                                                           */
/*  void IfSmallContinue(void)                                               */
/*                                                                           */
/*  Called between an if or if-else test and the corresponding action.       */
/*                                                                           */
/*****************************************************************************/

static void IfSmallContinue(void)
{
  fputs(" ? ", fp);
}


/*****************************************************************************/
/*                                                                           */
/*  void IfSmallElse(void)                                                   */
/*                                                                           */
/*  Begin the "else" part of a small if statement.                           */
/*                                                                           */
/*****************************************************************************/

static void IfSmallElse(void)
{
  fputs(" : ", fp);
}


/*****************************************************************************/
/*                                                                           */
/*  void IfSmallEnd(void)                                                    */
/*                                                                           */
/*  End a small "if" statement.                                              */
/*                                                                           */
/*****************************************************************************/

static void IfSmallEnd(void)
{
  fputs(")", fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "switch statements"                                            */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void SwitchBegin(void)                                                   */
/*                                                                           */
/*  Begin a switch statement.                                                */
/*                                                                           */
/*****************************************************************************/

static void SwitchBegin(void)
{
  fprintf(fp, "%*sswitch( ", indent, "");
  start_block = FALSE;
}


/*****************************************************************************/
/*                                                                           */
/*  void SwitchContinue(void)                                                */
/*                                                                           */
/*  Called after the switch test and before the first case.                  */
/*                                                                           */
/*****************************************************************************/

static void SwitchContinue(void)
{
  fputs(" )\n", fp);
  fprintf(fp, "%*s{\n", indent, "");
  indent += 2;
}


/*****************************************************************************/
/*                                                                           */
/*  void SwitchCaseBegin(void)                                               */
/*                                                                           */
/*  Begin a non-default switch case.                                         */
/*                                                                           */
/*****************************************************************************/

static void SwitchCaseBegin(void)
{
  fprintf(fp, "%*scase ", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  void SwitchCaseEnd(void)                                                 */
/*                                                                           */
/*  End a non-default switch case.                                           */
/*                                                                           */
/*****************************************************************************/

static void SwitchCaseEnd(void)
{
  fprintf(fp, ":\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void SwitchCase(CODEGEN_OBJ cnst)                                        */
/*                                                                           */
/*  Begin a case.  There may be several of these consecutively, if several   */
/*  labels correspond with one action.  Also, case_label may be NULL,        */
/*  indicating that it is the default case.                                  */
/*                                                                           */
/*****************************************************************************/

static void SwitchCase(CODEGEN_OBJ cnst)
{
  if( cnst != NULL )
    fprintf(fp, "%*scase %s:\n", indent, "", cnst->name1);
  else
    fprintf(fp, "%*sdefault:\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  void SwitchCaseInt(int case_label)                                       */
/*                                                                           */
/*  Like SwitchCase, except that the case label is an integer.               */
/*                                                                           */
/*****************************************************************************/

static void SwitchCaseInt(int case_label)
{
  fprintf(fp, "%*scase %d:\n", indent, "", case_label);
}


/*****************************************************************************/
/*                                                                           */
/*  void SwitchActionBegin(void)                                             */
/*                                                                           */
/*  Begin one of the actions of a switch statement.                          */
/*                                                                           */
/*****************************************************************************/

static void SwitchActionBegin(void)
{
  fputs("\n", fp);
  indent += 2;
}


/*****************************************************************************/
/*                                                                           */
/*  void SwitchActionEnd(BOOLEAN break_required)                             */
/*                                                                           */
/*  End a switch action.  Print a break only if break_required.              */
/*                                                                           */
/*****************************************************************************/

static void SwitchActionEnd(BOOLEAN break_required)
{
  if( break_required )
    fprintf(fp, "%*sbreak;\n", indent, "");
  indent -= 2;
  fputs("\n", fp);
}


/*****************************************************************************/
/*                                                                           */
/*  void SwitchEnd(void)                                                     */
/*                                                                           */
/*  End of switch statement.                                                 */
/*                                                                           */
/*****************************************************************************/

static void SwitchEnd(void)
{
  indent -= 2;
  fprintf(fp, "%*s}\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "while statements"                                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void WhileBegin(void)                                                    */
/*                                                                           */
/*  Begin a while statement.                                                 */
/*                                                                           */
/*****************************************************************************/

static void WhileBegin(void)
{
  fprintf(fp, "%*swhile( ", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  void WhileContinue(void)                                                 */
/*                                                                           */
/*  Called between the while test and the action of the while.               */
/*                                                                           */
/*****************************************************************************/

static void WhileContinue(void)
{
  fputs(" )\n", fp);
  BlockBegin();
}


/*****************************************************************************/
/*                                                                           */
/*  void WhileEnd(void)                                                      */
/*                                                                           */
/*  End a while statement.                                                   */
/*                                                                           */
/*****************************************************************************/

static void WhileEnd(void)
{
  BlockEnd();
}


/*****************************************************************************/
/*                                                                           */
/*  void WhileSingleBegin(void)                                              */
/*                                                                           */
/*  Begin a while statement whose body contains a single statement.          */
/*                                                                           */
/*****************************************************************************/

static void WhileSingleBegin(void)
{
  fprintf(fp, "%*swhile( ", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  void WhileSingleContinue(void)                                           */
/*                                                                           */
/*  Called between the while test and the action of the while.               */
/*                                                                           */
/*****************************************************************************/

static void WhileSingleContinue(void)
{
  fputs(" )\n", fp);
  IndentBegin();
}


/*****************************************************************************/
/*                                                                           */
/*  void WhileSingleEnd(void)                                                */
/*                                                                           */
/*  End a while statement whose body contains a single statement.            */
/*                                                                           */
/*****************************************************************************/

static void WhileSingleEnd(void)
{
  IndentEnd();
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "fail and goto statements"                                     */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void GotoStmt(CODEGEN_OBJ label)                                         */
/*                                                                           */
/*  Add a goto statement to this label.                                      */
/*                                                                           */
/*****************************************************************************/

static void GotoStmt(CODEGEN_OBJ label)
{
  fprintf(fp, "%*sgoto %s;\n", indent, "", label->name1);
  start_block = FALSE;
}


/*****************************************************************************/
/*                                                                           */
/*  void GotoTarget(CODEGEN_OBJ label)                                       */
/*                                                                           */
/*  Insert a goto label.                                                     */
/*                                                                           */
/*****************************************************************************/

static void GotoTarget(CODEGEN_OBJ label)
{
  fprintf(fp, "%*s%s:\n", indent, "", label->name1);
  start_block = FALSE;
}


/*****************************************************************************/
/*                                                                           */
/*  void Fail(void)                                                          */
/*                                                                           */
/*  Make the program fail.                                                   */
/*                                                                           */
/*****************************************************************************/

void Fail(void)
{
  fprintf(fp, "%*sassert(false);\n", indent, "");
  start_block = FALSE;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "type casts"                                                   */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void CastBegin(CODEGEN_TYPE res_type, CODEGEN_TYPE source_type)          */
/*                                                                           */
/*  Begin a type cast to res_type, but only if it differs from source_type.  */
/*  If source_type is NULL, it means to always insert the cast.              */
/*                                                                           */
/*****************************************************************************/

void CastBegin(CODEGEN_TYPE res_type, CODEGEN_TYPE source_type)
{
  if( source_type == NULL )
  {
    /* compulsory cast */
    fprintf(fp, "((%s) ", res_type->name);
    ArrayAddLast(casts_inserted, TRUE);
  }
  else if( res_type == source_type )
  {
    /* types are equal, no cast needed */
    ArrayAddLast(casts_inserted, FALSE);
  }
  else if( res_type->type == CODEGEN_TYPE_FLOAT )
  {
    /* conversion to float; ordinary cast not allowed in C */
    if( source_type->type == CODEGEN_TYPE_VOIDP )
      fputs("voidp_to_float(", fp);
    else if( source_type->type == CODEGEN_TYPE_INT )
      fputs("int_to_float(", fp);
    else
      assert(FALSE);
    ArrayAddLast(casts_inserted, TRUE);
  }
  else if( source_type->type == CODEGEN_TYPE_FLOAT )
  {
    /* conversion from float to voidp or int; ordinary cast not allowed in C */
    if( res_type->type == CODEGEN_TYPE_VOIDP )
      fputs("float_to_voidp(", fp);
    else if( res_type->type == CODEGEN_TYPE_INT )
      fputs("float_to_int(", fp);
    else
      assert(FALSE);
    ArrayAddLast(casts_inserted, TRUE);
  }
  else
  {
    /* ordinary cast */
    fprintf(fp, "((%s) ", res_type->name);
    ArrayAddLast(casts_inserted, TRUE);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void CastEnd(void)                                                       */
/*                                                                           */
/*  Call this at the end of the expression that was to be case by            */
/*  CastBegin.  It works, whether a cast was actually inserted or not.       */
/*                                                                           */
/*****************************************************************************/

void CastEnd(void)
{
  if( ArrayRemoveLast(casts_inserted) )
    fputs(")", fp);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "invocations of literals"                                      */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void LiteralInt(int i)                                                   */
/*                                                                           */
/*  Invoke literal integer i.                                                */
/*                                                                           */
/*****************************************************************************/

void LiteralInt(int i)
{
  fprintf(fp, "%d", i);
}


/*****************************************************************************/
/*                                                                           */
/*  void LiteralChar(UCHAR ch)                                               */
/*                                                                           */
/*  Invoke literal character ch.  For readability, use printable ASCII       */
/*  if available, otherwise hexadecimal.                                     */
/*                                                                           */
/*****************************************************************************/

void LiteralChar(UCHAR ch)
{
  if( !UCharIsPrintableAscii(ch) )
    fprintf(fp, "0x%X", ch);
  else if( ch == '\\' || ch == '\'' )
    fprintf(fp, "'\\%c'", (char) ch);
  else
    fprintf(fp, "'%c'", (char) ch);
}


/*****************************************************************************/
/*                                                                           */
/*  void LiteralString(ASTRING str)                                          */
/*                                                                           */
/*  Invoke literal string str (do we really need this?).                     */
/*                                                                           */
/*****************************************************************************/

void LiteralString(ASTRING str)
{
  fprintf(fp, "\"%s\"", str);
}


/*****************************************************************************/
/*                                                                           */
/*  void LiteralConst(ASTRING str)                                           */
/*                                                                           */
/*  Invoke literal const str (typically a number in string format).          */
/*                                                                           */
/*****************************************************************************/

void LiteralConst(ASTRING str)
{
  fputs(str, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  int SkipBlanksAndComments(FILE *fp)                                      */
/*                                                                           */
/*  Skip blanks and comments in fp and return the first character after      */
/*  them, or EOF if none.                                                    */
/*                                                                           */
/*****************************************************************************/

static int SkipBlanksAndComments(FILE *fp)
{
  int ch;
  ch = getc(fp);
  while( TRUE )
  {
    CONTINUE: switch( ch )
    {
      case ' ':
      case '\t':
      case '\n':
      case '\r':

	ch = getc(fp);
	break;


      case '/':

	ch = getc(fp);
	if( ch != '*' )
	{
	  ungetc(ch, fp);
	  return '/';
	}
	ch = getc(fp);
	while( TRUE ) switch( ch )
	{
	  case EOF:

	    return EOF;

	  case '*':

	    ch = getc(fp);
	    if( ch == '/' )
	    {
	      ch = getc(fp);
	      goto CONTINUE;
	    }
	    break;

	  default:

	    ch = getc(fp);
	    break;
	}
	break;


      default:

	return ch;
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN FileHasChanged(USTRING file_name)                                */
/*                                                                           */
/*  Compare file_name with its backup version and return TRUE if it has      */
/*  changed.  If there is no backup version, then it is considered changed.  */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN FileHasChanged(USTRING file_name)
{
  FILE *fp, *bfp;  int ch, bch;  BOOLEAN res;
  USTRING backup_name;  UTF8 utf8_file_name, utf8_backup_name;
  if( DEBUG4 )
    fprintf(stderr, "[ FileHasChanged(%s)\n",
      (ASTRING) UStringToUTF8(file_name));

  /* if backup file does not exist, the file has changed */
  utf8_file_name = UStringToUTF8(file_name);
  backup_name = UStringCat(file_name, AStringToUString("~"));
  utf8_backup_name = UStringToUTF8(backup_name);
  bfp = fopen((ASTRING) utf8_backup_name, "r");
  if( bfp == NULL )
    db_return(DEBUG4, "FileHasChanged (no backup file)", TRUE);

  /* file_name must open */
  fp = fopen((ASTRING) utf8_file_name, "r");
  assert(fp != NULL);

  fseek(fp, 0L, SEEK_END);
  fseek(bfp, 0L, SEEK_END);
  if( ftell(fp) != ftell(bfp) )
  {
    /* if files differ in length, file has changed */
    res = TRUE;
  }
  else
  {
    /* otherwise skip initial blanks and comments and compare char by char */
    rewind(fp);
    rewind(bfp);
    ch = SkipBlanksAndComments(fp);
    bch = SkipBlanksAndComments(bfp);
    while( ch == bch && ch != EOF )
    {
      ch = getc(fp);
      bch = getc(bfp);
    }
    res = (ch != bch);
  }

  /* close the files and return res */
  fclose(fp);
  fclose(bfp);
  db_return(DEBUG4, "FileHasChanged", res);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN FileChanged(CODEGEN_FILE file, BOOLEAN header,                   */
/*    SYMTAB_BOOLEAN file_comparisons)                                       */
/*                                                                           */
/*  Return TRUE if file has changed.  Since file actually represents two     */
/*  files, a .h file and (optionally) a .c file, parameter "header" is       */
/*  used to specify whether the question is being asked of the header file   */
/*  or the .c file.                                                          */
/*                                                                           */
/*  Symbol table file_comparisons holds all currently known answers to       */
/*  this question.  It is consulted before doing anything else, and any      */
/*  new answers are stored in it as we go.                                   */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN FileChanged(CODEGEN_FILE file, BOOLEAN header,
  SYMTAB_BOOLEAN file_comparisons)
{
  BOOLEAN res, res2;  CODEGEN_FILE f;  ARRAY_CODEGEN_FILE included_files;
  USTRING file_name;

  if( DEBUG4 )
    fprintf(stderr, "[ FileChanged(%s, %s)\n",
      (ASTRING) UStringToUTF8(file->name), bool(header));

  /* if the question has been asked before, return the previous answer */
  file_name = UStringCat4(code_dir, AStringToUString(NPC_DIR_SEP), file->name,
    AStringToUString(header ? ".h" : ".c"));
  if( SymRetrieve(file_comparisons, file_name, &res) )
    db_return(DEBUG4, "FileChanged (from file_comparisons)", res);

  /* if any include files have changed, then so has this file */
  included_files = header ? file->hincludes : file->cincludes;
  ArrayForEach(included_files, f)
    if( FileChanged(f, TRUE, file_comparisons) )
    {
      SymInsert(file_comparisons, file_name, TRUE, &res2);
      db_return(DEBUG4, "FileChanged (include file)", TRUE);
    }

  /* compare the file with any previous version and return the result */
  res = FileHasChanged(file_name);
  SymInsert(file_comparisons, file_name, res, &res2);
  db_return(DEBUG4, "FileChanged (tested)", res);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN Compile()                                                        */
/*                                                                           */
/*  Compile the system.  The complete set of source files, compiled and      */
/*  copied, is in source_files.  Each file knows what other files have       */
/*  been #included into it.                                                  */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN Compile()
{
  CODEGEN_EXEC exec;  CODEGEN_FILE file;  ASTRING com;  AFACTORY af;
  USTRING source, object, full_object;
  char command[400];
  SYMTAB_BOOLEAN file_comparisons;

  /* compile the individual source files */
  ASymInit(&file_comparisons);
  ArrayForEach(source_file_array, file)
  {
    if( !file->header_only && !file->system )
    {
      source = UStringCat(file->name, AStringToUString(".c"));
      object = UStringCat(file->name, AStringToUString(".o"));
      full_object = UStringCat4(code_dir, AStringToUString(NPC_DIR_SEP),
	file->name, AStringToUString(".o"));
      if( FileChanged(file,FALSE,file_comparisons) || !FileExists(full_object) )
      {
	/* if source file changed or there is no object file, compile it */
	sprintf(command, "cd %s && %s -c %s -o %s",
	  (ASTRING) UStringToUTF8(code_dir), compiler_name,
	  (ASTRING) UStringToUTF8(source), (ASTRING) UStringToUTF8(object));
	if( DEBUG1 )
	  fprintf(stderr, "CompileSystem calling \"%s\"\n", command);
	if( system(command) != 0 )
	  return FALSE;
      }
      else
      {
	/* source file unchanged and object file exists, so skip compile */
	if( DEBUG1 )
	  fprintf(stderr, "CompileSystem skipping unchanged \"%s\"\n",
	    (ASTRING) UStringToUTF8(source));
      }
    }
  }

  /* link object files to produce each executable */
  ArrayForEach(executables, exec)
  {
    af = AStringBegin();
    AStringAddFmt2(af, "cd %s && %s -o ", (ASTRING) UStringToUTF8(code_dir),
	compiler_name);
    AStringAddAString(af, (ASTRING) UStringToUTF8(exec->name));
    ArrayForEach(exec->files, file)
    {
      assert(!file->header_only);
      AStringAddFmt1(af, " %s.o", (ASTRING) UStringToUTF8(file->name));
    }
    AStringAddAString(af, " -lm");
    com = AStringEnd(af);
    if( DEBUG1 )
      fprintf(stderr, "CompileSystem calling \"%s\"\n", com);
    if( system(com) != 0 )
      return FALSE;
  }

  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "initialize"                                                   */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void InitGlobalSymbols(void)                                             */
/*                                                                           */
/*  Initialize the table of global symbols.  This contains all the usual     */
/*  C keywords and standard functions, which Nonpareil names must not        */
/*  conflict with.                                                           */
/*                                                                           */
/*****************************************************************************/

static void insert_global(ASTRING str)
{
  int val;
  if( !ASymInsert(global_symbols, str, 0, &val) )
  {
    fprintf(stderr, "\"%s\" inserted twice into initial global scope\n", str);
    exit(1);
  }
}

void InitGlobalSymbols(void)
{
  ASymInit(&global_symbols);

  /* from Kernighan and Ritchie, 2nd edition, p. 192 */
  insert_global("auto");
  insert_global("break");
  insert_global("case");
  insert_global("char");
  insert_global("const");
  insert_global("continue");
  insert_global("default");
  insert_global("do");
  insert_global("double");
  insert_global("else");
  insert_global("enum");
  insert_global("extern");
  insert_global("float");
  insert_global("for");
  insert_global("goto");
  insert_global("if");
  insert_global("int");
  insert_global("long");
  insert_global("register");
  insert_global("return");
  insert_global("short");
  insert_global("signed");
  insert_global("sizeof");
  insert_global("static");
  insert_global("struct");
  insert_global("switch");
  insert_global("typedef");
  insert_global("union");
  insert_global("unsigned");
  insert_global("void");
  insert_global("volatile");
  insert_global("while");
  insert_global("fortran");
  insert_global("asm");

  /* from /usr/lib/gcc-lib/i386-redhat-linux/2.96/include/stddef.h */
  insert_global("size_t");
  insert_global("ssize_t");
  insert_global("rune_t");
  insert_global("wchar_t");
  insert_global("wint_t");
  insert_global("NULL");
  insert_global("offsetof");

  /* from /usr/include/assert.h */
  insert_global("assert");
  insert_global("assert_perror");

  /* from /usr/include/stdlib.h */
  insert_global("div_t");
  insert_global("ldiv_t");
  insert_global("lldiv_t");
  insert_global("RAND_MAX");
  insert_global("EXIT_FAILURE");
  insert_global("EXIT_SUCCESS");
  insert_global("MB_CUR_MAX");
  insert_global("atof");
  insert_global("atoi");
  insert_global("atol");
  insert_global("atoll");
  insert_global("strtod");
  insert_global("strtof");
  insert_global("strtold");
  insert_global("strtol");
  insert_global("strtoul");
  insert_global("strtoq");
  insert_global("strtouq");
  insert_global("strtoll");
  insert_global("strtoull");
  insert_global("l64a");
  insert_global("a64l");
  insert_global("random");
  insert_global("srandom");
  insert_global("initstate");
  insert_global("setstate");
  insert_global("random_r");
  insert_global("srandom_r");
  insert_global("initstate_r");
  insert_global("setstate_r");
  insert_global("rand");
  insert_global("rand_r");
  insert_global("drand48");
  insert_global("erand48");
  insert_global("lrand48");
  insert_global("nrand48");
  insert_global("mrand48");
  insert_global("jrand48");
  insert_global("srand48");
  insert_global("seed48");
  insert_global("lcong48");
  insert_global("drand48_r");
  insert_global("erand48_r");
  insert_global("lrand48_r");
  insert_global("nrand48_r");
  insert_global("mrand48_r");
  insert_global("jrand48_r");
  insert_global("srand48_r");
  insert_global("seed48_r");
  insert_global("lcong48_r");
  insert_global("malloc");
  insert_global("calloc");
  insert_global("realloc");
  insert_global("free");
  insert_global("cfree");
  insert_global("valloc");
  insert_global("posix_memalign");
  insert_global("abort");
  insert_global("atexit");
  insert_global("on_exit");
  insert_global("exit");
  insert_global("getenv");
  insert_global("putenv");
  insert_global("setenv");
  insert_global("unsetenv");
  insert_global("clearenv");
  insert_global("mktemp");
  insert_global("mkstemp");
  insert_global("mkstemp64");
  insert_global("mkdtemp");
  insert_global("system");
  insert_global("canonicalize_file_name");
  insert_global("realpath");
  insert_global("comparison_fn_t");
  insert_global("bsearch");
  insert_global("qsort");
  insert_global("abs");
  insert_global("labs");
  insert_global("llabs");
  insert_global("div");
  insert_global("ldiv");
  insert_global("lldiv");
  insert_global("ecvt");
  insert_global("fcvt");
  insert_global("gcvt");
  insert_global("qecvt");
  insert_global("qfcvt");
  insert_global("qgcvt");
  insert_global("ecvt_r");
  insert_global("fcvt_r");
  insert_global("qecvt_r");
  insert_global("qfcvt_r");
  insert_global("mblen");
  insert_global("mbtowc");
  insert_global("wctomb");
  insert_global("mbstowcs");
  insert_global("wcstombs");
  insert_global("rpmatch");
  insert_global("getsubopt");
  insert_global("setkey");
  insert_global("posix_openpt");
  insert_global("grantpt");
  insert_global("unlockpt");
  insert_global("ptsname");
  insert_global("ptsname_r");
  insert_global("getpt");
  insert_global("getloadavg");

  /* from /usr/include/stdio.h */
  insert_global("FILE");
  insert_global("va_list");
  insert_global("fpos_t");
  insert_global("fpos64_t");
  insert_global("SEEK_SET");
  insert_global("SEEK_CUR");
  insert_global("SEEK_END");
  insert_global("P_tmpdir");
  insert_global("stdin");
  insert_global("stdout");
  insert_global("stderr");
  insert_global("remove");
  insert_global("rename");
  insert_global("tmpfile");
  insert_global("tmpfile64");
  insert_global("tmpnam");
  insert_global("tmpnam_r");
  insert_global("tempnam");
  insert_global("fclose");
  insert_global("fflush");
  insert_global("fflush_unlocked");
  insert_global("fcloseall");
  insert_global("fopen");
  insert_global("freopen");
  insert_global("fopen64");
  insert_global("freopen64");
  insert_global("fdopen");
  insert_global("fopencookie");
  insert_global("fmemopen");
  insert_global("open_memstream");
  insert_global("setbuf");
  insert_global("setvbuf");
  insert_global("setbuffer");
  insert_global("setlinebuf");
  insert_global("fprintf");
  insert_global("printf");
  insert_global("sprintf");
  insert_global("vfprintf");
  insert_global("vprintf");
  insert_global("vsprintf");
  insert_global("snprintf");
  insert_global("vsnprintf");
  insert_global("vasprintf");
  insert_global("asprintf");
  insert_global("vdprintf");
  insert_global("dprintf");
  insert_global("fscanf");
  insert_global("scanf");
  insert_global("vfscanf");
  insert_global("vscanf");
  insert_global("vsscanf");
  insert_global("fgetc");
  insert_global("getc");
  insert_global("getchar");
  insert_global("getc_unlocked");
  insert_global("getchar_unlocked");
  insert_global("fgetc_unlocked");
  insert_global("fputc");
  insert_global("putc");
  insert_global("putchar");
  insert_global("fputc_unlocked");
  insert_global("putc_unlocked");
  insert_global("putchar_unlocked");
  insert_global("getw");
  insert_global("putw");
  insert_global("fgets");
  insert_global("fgets_unlocked");
  insert_global("gets");
  insert_global("getdelim");
  insert_global("getline");
  insert_global("fputs");
  insert_global("fputs_unlocked");
  insert_global("puts");
  insert_global("ungetc");
  insert_global("fread");
  insert_global("fwrite");
  insert_global("fread_unlocked");
  insert_global("fwrite_unlocked");
  insert_global("fseek");
  insert_global("ftell");
  insert_global("rewind");
  insert_global("fseeko");
  insert_global("ftello");
  insert_global("fgetpos");
  insert_global("fsetpos");
  insert_global("fseeko64");
  insert_global("fgetpos64");
  insert_global("fsetpos64");
  insert_global("clearerr");
  insert_global("feof");
  insert_global("ferror");
  insert_global("clearerr_unlocked");
  insert_global("feof_unlocked");
  insert_global("ferror_unlocked");
  insert_global("perror");
  insert_global("sys_nerr");
  insert_global("sys_errlist");
  insert_global("fileno");
  insert_global("fileno_unlocked");
  insert_global("popen");
  insert_global("pclose");
  insert_global("ctermid");
  insert_global("cuserid");
  insert_global("obstack");
  insert_global("obstack_printf");
  insert_global("obstack_vprintf");
  insert_global("flockfile");
  insert_global("ftrylockfile");
  insert_global("funlockfile");

  /* from /usr/include/limits.h */
  insert_global("MB_LEN_MAX");
  insert_global("CHAR_BIT");
  insert_global("SCHAR_MIN");
  insert_global("SCHAR_MAX");
  insert_global("UCHAR_MAX");
  insert_global("CHAR_MIN");
  insert_global("CHAR_MAX");
  insert_global("SHRT_MIN");
  insert_global("SHRT_MAX");
  insert_global("USHRT_MAX");
  insert_global("INT_MIN");
  insert_global("INT_MAX");
  insert_global("UINT_MAX");
  insert_global("LONG_MAX");
  insert_global("LONG_MIN");
  insert_global("ULONG_MAX");
  insert_global("LLONG_MAX");
  insert_global("LLONG_MIN");
  insert_global("ULLONG_MAX");

  /* from /usr/include/math.h */
  insert_global("HUGE_VAL");
  insert_global("HUGE_VALF");
  insert_global("HUGE_VALL");
  insert_global("NAN");
  insert_global("signgam");
  insert_global("float_t");
  insert_global("double_t");
  insert_global("FLT_EVAL_METHOD");
  insert_global("INFINITY");
  insert_global("FP_FAST_FMA");
  insert_global("FP_FAST_FMAF");
  insert_global("FP_FAST_FMAL");
  insert_global("FP_ILOGB0");
  insert_global("FP_ILOGBNAN");
  insert_global("DECIMAL_DIG");
  insert_global("FP_NAN");
  insert_global("FP_INFINITE");
  insert_global("FP_ZERO");
  insert_global("FP_SUBNORMAL");
  insert_global("FP_NORMAL");
  insert_global("fpclassify");
  insert_global("signbit");
  insert_global("isfinite");
  insert_global("isnan");
  insert_global("MATH_ERRNO");
  insert_global("MATH_ERREXCEPT");
  insert_global("matherr");
  insert_global("X_TLOSS");
  insert_global("DOMAIN");
  insert_global("SING");
  insert_global("OVERFLOW");
  insert_global("UNDERFLOW");
  insert_global("TLOSS");
  insert_global("PLOSS");
  insert_global("HUGE");
  insert_global("M_E");
  insert_global("M_LOG2E");
  insert_global("M_LOG10E");
  insert_global("M_LN2");
  insert_global("M_LN10");
  insert_global("M_PI");
  insert_global("M_PI_2");
  insert_global("M_PI_4");
  insert_global("M_1_PI");
  insert_global("M_2_PI");
  insert_global("M_2_SQRTPI");
  insert_global("M_SQRT2");
  insert_global("M_SQRT1_2");
  insert_global("M_El");
  insert_global("M_LOG2El");
  insert_global("M_LOG10El");
  insert_global("M_LN2l");
  insert_global("M_LN10l");
  insert_global("M_PIl");
  insert_global("M_PI_2l");
  insert_global("M_PI_4l");
  insert_global("M_1_PIl");
  insert_global("M_2_PIl");
  insert_global("M_2_SQRTPIl");
  insert_global("M_SQRT2l");
  insert_global("M_SQRT1_2l");
  insert_global("isgreater");
  insert_global("isgreaterequal");
  insert_global("isless");
  insert_global("islessequal");
  insert_global("islessgreater");
  insert_global("isunordered");

  /* from /usr/include/bits/mathcalls.h */
  /* *** these are defined in builtin.c so are omitted here
  insert_global("sin");
  insert_global("cos");
  insert_global("tan");
  insert_global("asin");
  insert_global("acos");
  insert_global("atan");
  insert_global("atan2");
  insert_global("sinh");
  insert_global("cosh");
  insert_global("tanh");
  insert_global("exp");
  insert_global("log");
  insert_global("log10");
  insert_global("pow");
  insert_global("sqrt");
  insert_global("ceil");
  insert_global("floor");
  insert_global("fabs");
  insert_global("ldexp");
  insert_global("fmod");
  *** */
  insert_global("sincos");
  insert_global("acosh");
  insert_global("asinh");
  insert_global("atanh");
  insert_global("exp10");
  insert_global("pow10");
  insert_global("frexp");
  insert_global("expm1");
  insert_global("log1p");
  insert_global("logb");
  insert_global("exp2");
  insert_global("log2");
  insert_global("hypot");
  insert_global("cbrt");
  insert_global("isinf");
  insert_global("finite");
  insert_global("drem");
  insert_global("significand");
  insert_global("copysign");
  insert_global("nan");
  insert_global("j0");
  insert_global("j1");
  insert_global("jn");
  insert_global("y0");
  insert_global("y1");
  insert_global("yn");
  insert_global("erf");
  insert_global("erfc");
  insert_global("lgamma");
  insert_global("tgamma");
  insert_global("gamma");
  insert_global("lgamma_r");
  insert_global("rint");
  insert_global("nextafter");
  insert_global("nexttoward");
  insert_global("remainder");
  insert_global("scalb");
  insert_global("scalbn");
  insert_global("ilogb");
  insert_global("scalbln");
  insert_global("nearbyint");
  insert_global("round");
  insert_global("trunc");
  insert_global("remquo");
  insert_global("lrint");
  insert_global("llrint");
  insert_global("lround");
  insert_global("llround");
  insert_global("fdim");
  insert_global("fmax");
  insert_global("fmin");
  insert_global("fma");
  insert_global("modf");

  /* from /usr/include/time.h */
  insert_global("CLOCKS_PER_SEC");
  insert_global("clock_t");
  insert_global("time_t");
  insert_global("clockid_t");
  insert_global("timer_t");
  insert_global("timespec");
  insert_global("pid_t");
  insert_global("clock");
  insert_global("time");
  insert_global("difftime");
  insert_global("mktime");
  insert_global("strftime");
  insert_global("gmtime");
  insert_global("localtime");
  insert_global("gmtime_r");
  insert_global("localtime_r");
  insert_global("asctime");
  insert_global("ctime");
  insert_global("asctime_r");
  insert_global("ctime_r");
  insert_global("tzname");
  insert_global("tzset");
  insert_global("daylight");
  insert_global("timezone");
  insert_global("stime");
  insert_global("timegm");
  insert_global("timelocal");
  insert_global("dysize");
  insert_global("nanosleep");
  insert_global("clock_getres");
  insert_global("clock_gettime");
  insert_global("clock_settime");
  insert_global("clock_nanosleep");
  insert_global("clock_getcpuclockid");
  insert_global("timer_create");
  insert_global("timer_delete");
  insert_global("timer_settime");
  insert_global("timer_gettime");
  insert_global("timer_getoverrun");
  insert_global("getdate_r");

  /* global symbols inserted by Nonpareil's own code generation (?) */
  /* *** done during initialization now
  insert_global("DFT_PARAM_VAL");
  insert_global("NPC_DIR_SEP");
  insert_global("NP_LIB_DIR");
  *** */
}


/*****************************************************************************/
/*                                                                           */
/*  CODEGEN CodeGenMake(void)                                                */
/*                                                                           */
/*  Create and initialize a codegen object.                                  */
/*                                                                           */
/*****************************************************************************/

static CODEGEN CodeGenMake(void)
{
  CODEGEN res;
  GetMemory(res, CODEGEN);

  /* executables */
  res->ExecMake = &ExecMake;
  res->ExecAddFile = &ExecAddFile;

  /* comments */
  res->CommentLarge = &CommentLarge;
  res->CommentSmall = &CommentSmall;

  /* files */
  res->FileMake = &FileMake;
  res->FileBegin = &FileBegin;
  res->HeaderFileInclude = &HeaderFileInclude;
  res->CodeFileInclude = &CodeFileInclude;
  res->FileEnd = &FileEnd;
  res->FileCopy = &FileCopy;
  res->FlushCurrentFile = &FlushCurrentFile;

  /* types, type tags, structs, and their attributes (which are functions) */
  res->TypeMakeObject = &TypeMakeObject;
  res->TypeMakeFunctionType = &TypeMakeFunctionType;
  res->TypeMakeEnumerated = &TypeMakeEnumerated;
  res->TypeFieldWidth = &TypeFieldWidth;
  res->TypeDef = &TypeDef;
  res->TypeShow = &TypeShow;
  res->StructBegin = &StructBegin;
  res->StructItemMake = &StructItemMake;
  res->StructItem = &StructItem;
  res->StructGap = &StructGap;
  res->StructEnd = &StructEnd;

  /* indenting and blocks */
  res->IndentBegin = &IndentBegin;
  res->IndentEnd = &IndentEnd;
  res->BlockBegin = &BlockBegin;
  res->BlockEnd = &BlockEnd;

  /* assignment statements */
  res->AsstBegin = &AsstBegin;
  res->AsstContinue = &AsstContinue;
  res->AsstEnd = &AsstEnd;
  res->AsstOpBegin = &AsstOpBegin;
  res->AsstOpContinue = &AsstOpContinue;
  res->AsstOpEnd = &AsstOpEnd;
  res->AsstArrayInit = &AsstArrayInit;

  /* statements */
  res->StmtBegin = &StmtBegin;
  res->StmtEnd = &StmtEnd;

  /* functions and function calls, including predefined operators */
  res->FunctionMake = &FunctionMake;
  res->FunctionMake2 = &FunctionMake2;
  res->FunctionMake3 = &FunctionMake3;
  res->FunctionMakeFunRef = &FunctionMakeFunRef;
  res->FunctionAttributeIndexed = &FunctionAttributeIndexed;
  res->FunctionBegin = &FunctionBegin;
  res->FunctionFormal = &FunctionFormal;
  res->FunctionContinue = &FunctionContinue;
  res->FunctionEnd = &FunctionEnd;
  res->PrototypeBegin = &PrototypeBegin;
  res->PrototypeFormal = &PrototypeFormal;
  res->PrototypeEnd = &PrototypeEnd;
  res->CallBegin = &CallBegin;
  res->CallContinue = &CallContinue;
  res->CallEnd = &CallEnd;

  /* variables (creation, declaration, assignment, and invocation) */
  res->VarMake = &VarMake;
  res->VarIndexedAttributeMake = &VarIndexedAttributeMake;
  res->VarIndexedAttributeUpdateIndex = &VarIndexedAttributeUpdateIndex;
  res->VarDeclare = &VarDeclare;
  res->VarDeclare2 = &VarDeclare2;
  res->VarDeclareAndAsstBegin = &VarDeclareAndAsstBegin;
  res->VarMakeAndDeclare = &VarMakeAndDeclare;
  res->VarMakeAndDeclareAndAsstBegin = &VarMakeAndDeclareAndAsstBegin;
  res->Variable = &Variable;
  res->VarOpAsstBegin = &VarOpAsstBegin;
  res->VarAsstBegin = &VarAsstBegin;
  res->VarAsst2Begin = &VarAsst2Begin;
  res->VarUnDeclare = &VarUnDeclare;
  res->VarShow = &VarShow;

  /* attribute functions */
  res->AttrMake = &AttrMake;

  /* constants */
  res->ConstMakeAString = &ConstMakeAString;
  res->ConstMakeInt = &ConstMakeInt;
  res->ConstMakeInt2 = &ConstMakeInt2;
  res->ConstDeclare = &ConstDeclare;

  /* freeing all the above */
  res->ObjFree = &ObjFree;

  /* if statements and if expressions */
  res->IfStmtBegin = &IfStmtBegin;
  res->IfStmtElseIf = &IfStmtElseIf;
  res->IfStmtContinue = &IfStmtContinue;
  res->IfStmtElse = &IfStmtElse;
  res->IfStmtEnd = &IfStmtEnd;
  res->IfSmallBegin = &IfSmallBegin;
  res->IfSmallElseIf = &IfSmallElseIf;
  res->IfSmallContinue = &IfSmallContinue;
  res->IfSmallElse = &IfSmallElse;
  res->IfSmallEnd = &IfSmallEnd;

  /* switch statements */
  res->SwitchBegin = &SwitchBegin;
  res->SwitchContinue = &SwitchContinue;
  res->SwitchCaseBegin = &SwitchCaseBegin;
  res->SwitchCaseEnd = &SwitchCaseEnd;
  res->SwitchCase = &SwitchCase;
  res->SwitchCaseInt = &SwitchCaseInt;
  res->SwitchActionBegin = &SwitchActionBegin;
  res->SwitchActionEnd = &SwitchActionEnd;
  res->SwitchEnd = &SwitchEnd;

  /* while statements */
  res->WhileBegin = &WhileBegin;
  res->WhileContinue = &WhileContinue;
  res->WhileEnd = &WhileEnd;
  res->WhileSingleBegin = &WhileSingleBegin;
  res->WhileSingleContinue = &WhileSingleContinue;
  res->WhileSingleEnd = &WhileSingleEnd;

  /* fail and goto statements (a goto label is treated like a local var) */
  res->GotoStmt = &GotoStmt;
  res->GotoTarget = &GotoTarget;
  res->Fail = &Fail;

  /* type casts (inserted only if type1 != type2) */
  res->CastBegin = &CastBegin;
  res->CastEnd = &CastEnd;

  /* invocations of literals */
  res->LiteralInt = &LiteralInt;
  res->LiteralChar = &LiteralChar;
  res->LiteralString = &LiteralString;
  res->LiteralConst = &LiteralConst;

  /* compile and clear */
  res->Compile = &Compile;

  /* predefined types */
  res->char_type = TypeMakeImmediate(CODEGEN_TYPE_CHAR, "char",
    sizeof(char) * CHAR_BIT);
  res->uchar_type = TypeMakeImmediate(CODEGEN_TYPE_UCHAR, "unsigned char",
    sizeof(unsigned char) * CHAR_BIT);
  res->short_type = TypeMakeImmediate(CODEGEN_TYPE_SHORT, "short",
    sizeof(short) * CHAR_BIT);
  res->ushort_type = TypeMakeImmediate(CODEGEN_TYPE_USHORT, "unsigned short",
    sizeof(unsigned short) * CHAR_BIT);
  res->int_type = TypeMakeImmediate(CODEGEN_TYPE_INT, "int",
    sizeof(int) * CHAR_BIT);
  res->uint_type = TypeMakeImmediate(CODEGEN_TYPE_UINT, "unsigned int",
    sizeof(unsigned int) * CHAR_BIT);
  res->float_type = TypeMakeImmediate(CODEGEN_TYPE_FLOAT, "float",
    sizeof(float) * CHAR_BIT);
  res->double_type = TypeMakeImmediate(CODEGEN_TYPE_DOUBLE, "double",
    sizeof(double) * CHAR_BIT);
  res->void_type = TypeMakeImmediate(CODEGEN_TYPE_VOID, "void", 0);
  res->voidp_type = TypeMakeImmediate(CODEGEN_TYPE_VOIDP, "void *",
    sizeof(void *) * CHAR_BIT);
  res->charp_type = TypeMakeImmediate(CODEGEN_TYPE_CHARP, "char *",
    sizeof(char *) * CHAR_BIT);

  /* predefined functions offered by C (in order of index in K&R) */
  res->ret = VarNew(CODEGEN_OBJ_RETURN, NULL);
  res->no_ret = VarNew(CODEGEN_OBJ_NO_RETURN, NULL);
  res->add = FunctionPredef(CODEGEN_OBJ_INFIX, "+", NULL);
  res->address_of = FunctionPredef(CODEGEN_OBJ_PREFIX, "&", NULL);
  res->bitwise_and = FunctionPredef(CODEGEN_OBJ_INFIX, "&", NULL);
  res->bitwise_or = FunctionPredef(CODEGEN_OBJ_INFIX, "|", NULL);
  res->divide = FunctionPredef(CODEGEN_OBJ_INFIX, "/", NULL);
  res->divide_float = FunctionPredef(CODEGEN_OBJ_INFIX_PREFIX,"/","(float)");
  res->eq = FunctionPredef(CODEGEN_OBJ_INFIX, "==", NULL);
  res->ge = FunctionPredef(CODEGEN_OBJ_INFIX, ">=", NULL);
  res->gt = FunctionPredef(CODEGEN_OBJ_INFIX, ">", NULL);
  res->ne = FunctionPredef(CODEGEN_OBJ_INFIX, "!=", NULL);
  res->lshift = FunctionPredef(CODEGEN_OBJ_INFIX, "<<", NULL);
  res->le = FunctionPredef(CODEGEN_OBJ_INFIX, "<=", NULL);
  res->lt = FunctionPredef(CODEGEN_OBJ_INFIX, "<", NULL);
  res->logical_and = FunctionPredef(CODEGEN_OBJ_INFIX, "&&", NULL);
  res->logical_not = FunctionPredef(CODEGEN_OBJ_PREFIX, "!", NULL);
  res->logical_or = FunctionPredef(CODEGEN_OBJ_INFIX, "||", NULL);
  res->mod = FunctionPredef(CODEGEN_OBJ_INFIX, "%", NULL);
  res->multiply = FunctionPredef(CODEGEN_OBJ_INFIX, "*", NULL);
  res->rshift = FunctionPredef(CODEGEN_OBJ_INFIX, ">>", NULL);
  res->subtract = FunctionPredef(CODEGEN_OBJ_INFIX, "-", NULL);
  res->unary_minus = FunctionPredef(CODEGEN_OBJ_PREFIX, "-", NULL);
  res->size_of = FunctionPredef(CODEGEN_OBJ_ORDINARY, "sizeof", NULL);
  res->assert_fn = FunctionPredef(CODEGEN_OBJ_ORDINARY, "assert", NULL);
  res->array_index = FunctionPredef(CODEGEN_OBJ_INDEX, NULL, NULL);
  /* res->attribute = FunctionPredef(CODEGEN_OBJ_ATTRIBUTE, NULL, NULL); */

  /* a selection of limits from <limits.h> */
  res->char_min = VarNew(CODEGEN_OBJ_VAR, "SHRT_MIN");
  res->char_max = VarNew(CODEGEN_OBJ_VAR, "SHRT_MAX");
  res->short_min = VarNew(CODEGEN_OBJ_VAR, "SHRT_MIN");
  res->short_max = VarNew(CODEGEN_OBJ_VAR, "SHRT_MAX");
  res->int_min = VarNew(CODEGEN_OBJ_VAR, "INT_MIN");
  res->int_max = VarNew(CODEGEN_OBJ_VAR, "INT_MAX");
  res->uchar_max = VarNew(CODEGEN_OBJ_VAR, "UCHAR_MAX");
  res->ushort_max = VarNew(CODEGEN_OBJ_VAR, "USHRT_MAX");
  res->uint_max = VarNew(CODEGEN_OBJ_VAR, "UINT_MAX");
  res->float_max = VarNew(CODEGEN_OBJ_VAR, "FLT_MAX");
  res->float_min = VarNew(CODEGEN_OBJ_VAR, "FLT_MIN");

  /* a selection of useful C operations */
  res->identity = FunctionPredef(CODEGEN_OBJ_IDENTITY, NULL, NULL);
  res->cast_to_char = FunctionPredef(CODEGEN_OBJ_PREFIX, "(signed char)", NULL);
  res->cast_to_short = FunctionPredef(CODEGEN_OBJ_PREFIX, "(short)", NULL);
  res->cast_to_int = FunctionPredef(CODEGEN_OBJ_PREFIX, "(int)", NULL);
  res->cast_to_uchar =FunctionPredef(CODEGEN_OBJ_PREFIX,"(unsigned char)",NULL);
  res->cast_to_ushort=FunctionPredef(CODEGEN_OBJ_PREFIX,"(unsigned short)",NULL);
  res->cast_to_uint =FunctionPredef(CODEGEN_OBJ_PREFIX, "(unsigned int)", NULL);
  res->cast_to_float = FunctionPredef(CODEGEN_OBJ_PREFIX, "(float)", NULL);
  res->round = FunctionPredef(CODEGEN_OBJ_PREFIX_ORDINARY, "(int)", "round");
  res->logical_and_seq = FunctionPredef(CODEGEN_OBJ_INFIX_SEQ, "&&", NULL);
  res->logical_or_seq = FunctionPredef(CODEGEN_OBJ_INFIX_SEQ, "||", NULL);
  res->fprintf_fn = FunctionPredef(CODEGEN_OBJ_ORDINARY, "fprintf", NULL);
  res->stderr_fn = VarNew(CODEGEN_OBJ_VAR, "stderr");
  res->null_value = VarNew(CODEGEN_OBJ_VAR, "NULL");

  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  CODEGEN CodeGen_C(USTRING codedir)                                       */
/*                                                                           */
/*  Return a C codegen object, suitably initialized.                         */
/*                                                                           */
/*****************************************************************************/

CODEGEN CodeGen_C(USTRING codedir, ASTRING compilername)
{
  DIR *dir;  UTF8 utf8_codedir;
  
  /* make sure codedir exists */
  utf8_codedir = UStringToUTF8(codedir);
  dir = opendir((ASTRING) utf8_codedir);
  if( dir != NULL )
    closedir(dir);
  else if( mkdir((ASTRING) utf8_codedir, 0777) != 0 )
  {
    fprintf(stderr, "cannot create codegen directory %s\n", utf8_codedir);
    exit(1);
  }

  /* initialize the static variables */
  InitGlobalSymbols();
  ASymInit(&struct_symbols);
  ASymInit(&local_symbols);
  code_dir = codedir;
  compiler_name = compilername;
  hfp = fp = NULL;
  indent = 0;
  ArrayInit(&source_file_array);
  ASymInit(&source_file_table);
  ArrayInit(&executables);
  ArrayInit(&casts_inserted);
  ArrayInit(&blocks_inserted);
  start_block = FALSE;
  underscore_str = AStringToUString("_");

  /* make a codegen object and return it */
  return CodeGenMake();
}
