/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         coercion.c                                                 */
/*  DESCRIPTION:  Nonpareil coercions                                        */
/*                                                                           */
/*****************************************************************************/
#include <stdlib.h>
#include "externs.h"
#define DEBUG1 0


/*****************************************************************************/
/*                                                                           */
/*  COERCION                                                                 */
/*                                                                           */
/*  A coercion is a sequence of functions that maps one type to another.     */
/*                                                                           */
/*****************************************************************************/

struct coercion_rec {
  ARRAY_FEFN_FEATURE	feature_views;	/* the coercion features             */
};



/*****************************************************************************/
/*                                                                           */
/*  COERCION CoercionMake(FEFN_FEATURE fv)                                   */
/*                                                                           */
/*  Make a new coercion with one feature view in it.                         */
/*                                                                           */
/*****************************************************************************/

COERCION CoercionMake(FEFN_FEATURE fv)
{
  COERCION res;
  GetMemory(res, COERCION);
  ArrayInit(&res->feature_views);
  ArrayAddLast(res->feature_views, fv);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  COERCION CoercionCopyAndExtend(COERCION co, FEFN_FEATURE fv)             */
/*                                                                           */
/*  Return a copy of coercion co, extended one step by feature fv.           */
/*                                                                           */
/*****************************************************************************/

COERCION CoercionCopyAndExtend(COERCION co, FEFN_FEATURE fv)
{
  COERCION res;  FEFN_FEATURE fv2;
  GetMemory(res, COERCION);
  ArrayInit(&res->feature_views);
  ArrayForEach(co->feature_views, fv2)
    ArrayAddLast(res->feature_views, fv2);
  ArrayAddLast(res->feature_views, fv);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  COERCION CoercionCopyAndAppend(COERCION co1, COERCION co2)               */
/*                                                                           */
/*  Return a copy of coercion co1, with coercion co2 appended to it.         */
/*  This code allows both coercions to be NULL or empty.                     */
/*                                                                           */
/*****************************************************************************/

COERCION CoercionCopyAndAppend(COERCION co1, COERCION co2)
{
  COERCION res;  FEFN_FEATURE fv;
  if( co1 == NULL && co2 == NULL )
    return NULL;
  else
  {
    GetMemory(res, COERCION);
    ArrayInit(&res->feature_views);
    if( co1 != NULL && co1->feature_views != NULL )
      ArrayForEach(co1->feature_views, fv)
	ArrayAddLast(res->feature_views, fv);
    if( co2 != NULL && co2->feature_views != NULL )
      ArrayForEach(co2->feature_views, fv)
	ArrayAddLast(res->feature_views, fv);
    return res;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  int CoercionLength(COERCION co)                                          */
/*                                                                           */
/*  The length of coercion co, which may be NULL.                            */
/*                                                                           */
/*****************************************************************************/

int CoercionLength(COERCION co)
{
  return co == NULL ? 0 : ArraySize(co->feature_views);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN CoercionEqual(COERCION co1, COERCION co2)                        */
/*                                                                           */
/*  Return TRUE if co1 and c02 are equal; they may be NULL.                  */
/*                                                                           */
/*****************************************************************************/

BOOLEAN CoercionEqual(COERCION co1, COERCION co2)
{
  int s1, s2, i;
  if( co1 == NULL || co2 == NULL )
    return co1 == NULL && co2 == NULL;
  s1 = ArraySize(co1->feature_views);
  s2 = ArraySize(co2->feature_views);
  if( s1 != s2 )
    return FALSE;
  for( i = 0;  i < s1;  i++ )
    if( ArrayGet(co1->feature_views, i) != ArrayGet(co2->feature_views, i) )
      return FALSE;
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  void CoercionInsert(COERCION co, EXPR *e, CONTEXT cxt, TYPE self_type)   */
/*                                                                           */
/*  Replace *e by *e with the given coercion above it.                       */
/*                                                                           */
/*****************************************************************************/

void CoercionInsert(COERCION co, EXPR *e, CONTEXT cxt, TYPE self_type)
{
  FEFN_FEATURE fv;
  if( co != NULL )
  {
    if( DEBUG1 )
    {
      fprintf(stderr, "[ CoercionInsert(%s, *e, cxt) at %s, e = ",
	CoercionShow(co), FilePosShow(ExprFilePos(*e)));
      ExprDebug(*e, cxt, TRUE, stderr, SINGLE_LINE);
      fprintf(stderr, "\n");
    }
    ArrayForEach(co->feature_views, fv)
      ExprCallInsertCoercion(e, fv, cxt, self_type);
    if( DEBUG1 )
    {
      fprintf(stderr, "] CoercionInsert returning, now e = ");
      ExprDebug(*e, cxt, TRUE, stderr, SINGLE_LINE);
      fprintf(stderr, "\n");
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  UTF8 CoercionShow(COERCION co)                                           */
/*                                                                           */
/*  Return a string showing this coercion.  Not for code generation,         */
/*  only suited to debugging.                                                */
/*                                                                           */
/*****************************************************************************/

UTF8 CoercionShow(COERCION co)
{
  FEFN_FEATURE fv;
  AFACTORY af;
  if( co == NULL )
    return (UTF8) "";
  else
  {
    af = AStringBegin();
    ArrayForEach(co->feature_views, fv)
    {
      if( fv != ArrayFirst(co->feature_views) )
	AStringAddChar(af, '.');
      AStringAddAString(af, (ASTRING) NameShow(FEFnFeatureName(fv)));
    }
    return (UTF8) AStringEnd(af);
  }
}
