/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         context.c                                                  */
/*  DESCRIPTION:  Nonpareil context handler                                  */
/*                                                                           */
/*  Notes on the use of contexts                                             */
/*  ----------------------------                                             */
/*                                                                           */
/*  A function which receives a context as parameter is obliged to leave     */
/*  that context unchanged at exit, whether or not the function succeeds.    */
/*  It may (and often will) push new layers onto the context as it runs,     */
/*  but by exit these must all have been popped.                             */
/*                                                                           */
/*****************************************************************************/
#include <stdio.h>
#include "externs.h"
#define DEBUG1 0
#define DEBUG2 0
#define DEBUG3 0
#define DEBUG4 0
#define DEBUG5 0
#define DEBUG6 0


/*****************************************************************************/
/*                                                                           */
/*  CONTEXT                                                                  */
/*                                                                           */
/*  A context, with a stack of symbol tables, and also marked names, which   */
/*  are to be deleted by the next ClearToMark call.                          */
/*                                                                           */
/*****************************************************************************/

struct context_rec {
  ARRAY_SYMTAB_NAMED		symbols;	/* symbols in scope          */
  ARRAY_BOOLEAN			shadow_lims;	/* insert search stops here  */
  ARRAY_FEFN			fefns;	/* for hidden parameters     */
};


/*****************************************************************************/
/*                                                                           */
/*  CONTEXT ContextNew(SYMTAB_CLASS_VIEW class_views)                        */
/*                                                                           */
/*  Initialize context to a new context, initially holding just this         */
/*  one symbol table.  A class views symbol table is always at the           */
/*  bottom of the stack of views.                                            */
/*                                                                           */
/*****************************************************************************/

CONTEXT ContextNew(SYMTAB_CLASS_VIEW class_views)
{
  CONTEXT res;
  if( DEBUG1 )
    fprintf(stderr, "[ ContextInit(class_views)\n");
  GetMemory(res, CONTEXT);
  ArrayInit(&res->symbols);
  ArrayInit(&res->shadow_lims);
  ArrayAddLast(res->symbols, (SYMTAB_NAMED) class_views);
  ArrayAddLast(res->shadow_lims, TRUE);
  ArrayInit(&res->fefns);
  ArrayAddLast(res->fefns, NULL);
  if( DEBUG1 )
  {
    fprintf(stderr, "] ContextInit returning\n");
    ContextDebug(res, "new context", stderr, 2);
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void ContextPushFEFnFeatureSets(CONTEXT cxt,                             */
/*    SYMTAB_FEFN_FEATURE_SET fvs, BOOLEAN shadow_lim)                       */
/*                                                                           */
/*  Push symbol table of feature signatures onto cxt.                        */
/*                                                                           */
/*****************************************************************************/

void ContextPushFEFnFeatureSets(CONTEXT cxt,
  SYMTAB_FEFN_FEATURE_SET fvs, BOOLEAN shadow_lim)
{
  if( DEBUG4 )
    fprintf(stderr, "[ ContextPushFEFnFeatureSets(cxt, fvs)\n");
  assert(fvs != NULL);
  ArrayAddLast(cxt->symbols, (SYMTAB_NAMED) fvs);
  ArrayAddLast(cxt->shadow_lims, shadow_lim);
  ArrayAddLast(cxt->fefns, NULL);
}


/*****************************************************************************/
/*                                                                           */
/*  void ContextPushEmpty(CONTEXT cxt, FEFN fefn, BOOLEAN shadow_lim)        */
/*                                                                           */
/*  Push a new, empty symbol table onto cxt, with associated function for    */
/*  receiving hidden parameters (may be NULL), and information about whether */
/*  this table is the limit of shadow checking.                              */
/*                                                                           */
/*****************************************************************************/

void ContextPushEmpty(CONTEXT cxt, FEFN fefn, BOOLEAN shadow_lim)
{
  SYMTAB_NAMED s;
  if( DEBUG4 )
    fprintf(stderr, "[ ContextPushEmpty(cxt)\n");
  SymInit(&s);
  ArrayAddLast(cxt->symbols, s);
  ArrayAddLast(cxt->shadow_lims, shadow_lim);
  ArrayAddLast(cxt->fefns, fefn);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ContextPushTypeVars(CONTEXT cxt, TYPE_VARS tv, BOOL shadow_lim)  */
/*                                                                           */
/*  Push a new symbol table onto cxt, containing these type variables.       */
/*  Return TRUE if success (failure means they had a name clash).            */
/*                                                                           */
/*****************************************************************************/

/* *** not using this omnibus function any more
BOOLEAN ContextPushTypeVars(CONTEXT cxt, TYPE_VARS tv, BOOLEAN shadow_lim)
{
  ContextPushEmpty(cxt, NULL, shadow_lim);
  return TypeVarsBegin(tv, cxt);
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  void ContextPop(CONTEXT cxt, BOOLEAN free_it)                            */
/*                                                                           */
/*  Pop one symbol table off the cxt stack, and optionally free it.          */
/*                                                                           */
/*****************************************************************************/

void ContextPop(CONTEXT cxt, BOOLEAN free_it)
{
  SYMTAB_NAMED s;
  assert(ArraySize(cxt->symbols) >= 1);
  s = ArrayRemoveLast(cxt->symbols);
  if( free_it )
    SymFree(s);
  ArrayDropLast(cxt->shadow_lims);
  ArrayDropLast(cxt->fefns);

  if( DEBUG1 )
  {
    fprintf(stderr, "ContextPop returning; context is:\n");
    ContextDebug(cxt, "", stderr, 2);
  }
  if( DEBUG4 )
    fprintf(stderr, "] ContextPop(%s)\n", bool(free_it));
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN add(CONTEXT cxt, NAMED v1)                                       */
/*                                                                           */
/*  Add v1 to cxt.  Return TRUE on success, or FALSE on failure.  The only   */
/*  possible cause of failure is that something with v1's name is already    */
/*  present; when that happens, an error message is printed as well.         */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN add(CONTEXT cxt, NAMED v1)
{
  SYMTAB_NAMED s;  int i;  USTRING key;  NAMED v2;

  /* precondition checks */
  assert(v1 != NULL);
  assert(ArraySize(cxt->symbols) > 0);

  /* check all tables back to the most recent shadow limit */
  key = NameKey(NamedName(v1));
  for( i = ArraySize(cxt->symbols) - 1;  i >= 0;  i-- )
  {
    s = ArrayGet(cxt->symbols, i);
    if( SymRetrieve(s, key, &v2) )
    {
      fprintf(stderr, "name clash:\n");
      fprintf(stderr, "  %s: %s \"%s\"\n", FilePosShow(NamedFilePos(v2)),
	KindShow(NamedKind(v2)), UStringToUTF8(key));
      fprintf(stderr, "  %s: %s \"%s\"\n", FilePosShow(NamedFilePos(v1)),
	KindShow(NamedKind(v1)), UStringToUTF8(key));
      return FALSE;
    }
    if( ArrayGet(cxt->shadow_lims, i) )
      break;
  }

  /* insert into cxt */
  if( !SymInsert(ArrayLast(cxt->symbols), key, v1, &v2) )
    assert(FALSE);
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ContextInsertGeneric(CONTEXT cxt, TYPE_VAR v)                    */
/*  BOOLEAN ContextInsertFEFnLetDef(CONTEXT cxt, FEFN_LETDEF ld)             */
/*  BOOLEAN ContextInsertFEFnParam(CONTEXT cxt, FEFN_PARAM param)            */
/*  BOOLEAN ContextInsertCondition(CONTEXT cxt, FEFN_DOWNDEF downdef)        */
/*                                                                           */
/*  Add these various things to cxt.  On failure, print an error message     */
/*  and return FALSE.                                                        */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ContextInsertGeneric(CONTEXT cxt, TYPE_VAR v)
{
  return add(cxt, (NAMED) v); 
}

BOOLEAN ContextInsertFEFnLetDef(CONTEXT cxt, FEFN_LETDEF ld)
{
  return add(cxt, (NAMED) ld);
}

BOOLEAN ContextInsertFEFnParam(CONTEXT cxt, FEFN_PARAM param)
{
  return add(cxt, (NAMED) param);
}

BOOLEAN ContextInsertFEFnDownDef(CONTEXT cxt, FEFN_DOWNDEF downdef)
{
  return add(cxt, (NAMED) downdef);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ContextRetrieve(CONTEXT cxt, USTRING key, NAMED *named)          */
/*                                                                           */
/*  Retrieve named, when not sure what type it will turn out to be.          */
/*  Any non-local retrieves get reported to the functions they pass through, */
/*  outermost function first.                                                */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ContextRetrieve(CONTEXT cxt, USTRING key, NAMED *named)
{
  SYMTAB_NAMED s, s2;  int i, j;
  FEFN fefn;  NAMED v2;
  if( DEBUG6 )
    fprintf(stderr, "[ ContextRetrieve(cxt, %s)\n", UStringToUTF8(key));
  for( i = ArraySize(cxt->symbols) - 1;  i >= 0;  i-- )
  {
    s = ArrayGet(cxt->symbols, i);
    if( SymRetrieve(s, key, named) )
    {
      assert(*named != NULL);
      if( NamedIsLocal(*named) && FEFnParams((FEFN) *named) == NULL )
      {
	if( DEBUG6 )
	  fprintf(stderr, "  local *named\n");
	for( j = i + 1;  j < ArraySize(cxt->symbols);  j++ )
	{
	  fefn = ArrayGet(cxt->fefns, j);
	  if( DEBUG6 )
	    fprintf(stderr, "  j = %d, fefn = %s\n", j,
	      fefn == NULL ? "NULL" : "non-NULL");
	  if( fefn != NULL )
	  {
	    if( !FEFnAddHiddenParameter(fefn, named) )
	      return FALSE;
	    s2 = ArrayGet(cxt->symbols, j);
	    if( !SymInsert(s2, NameKey(NamedName(*named)), *named, &v2) )
	      assert(FALSE);
	  }
	}
      }
      db_return(DEBUG6, "ContextRetrieve", TRUE);
    }
  }
  if( DEBUG6 )
  {
    fprintf(stderr, "ContextRetrieve(%s) failing, context is:\n",
      UStringToUTF8(key));
    ContextDebug(cxt, "", stderr, 2);
    fprintf(stderr, "\n");
  }
  db_return(DEBUG6, "ContextRetrieve", FALSE);
  return FALSE;
}


/*****************************************************************************/
/*                                                                           */
/*  USTRING ContextNewName(CONTEXT cxt, USTRING str)                         */
/*                                                                           */
/*  Return a new name based on str that does not clash with cxt.             */
/*                                                                           */
/*****************************************************************************/

/* *** apparently no longer in use
USTRING ContextNewName(CONTEXT cxt, USTRING str)
{
  int i;  NAMED named;  USTRING str2;
  for( i = 1;  i < 100;  i++ )
  {
    str2 = UStringCat(str, UStringFromInteger(i));
    if( !ContextRetrieve(cxt, str2, &named) )
      return str2;
  }
  assert(FALSE);
  return NULL;  ** keep compiler happy **
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ContextRetrieveClassView(CONTEXT cxt, USTRING key,               */
/*    CLASS_VIEW *class_view)                                                */
/*                                                                           */
/*  Retrieve a class view with the given name from cxt, and return TRUE      */
/*  if found, setting *class_view to the retrieved class view.  Otherwise    */
/*  return FALSE.                                                            */
/*                                                                           */
/*  Class views only live in the bottommost symbol table, so this function   */
/*  assumes there is such a table, and it only looks there.                  */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ContextRetrieveClassView(CONTEXT cxt, USTRING key,
  CLASS_VIEW *class_view)
{
  NAMED named;
  assert(ArraySize(cxt->symbols) >= 1);
  if( SymRetrieve(ArrayFirst(cxt->symbols), key, &named) )
  {
    assert(NamedKind(named) == KIND_CLASS_VIEW);
    *class_view = (CLASS_VIEW) named;
    return TRUE;
  }
  else
    return FALSE;
}


/*****************************************************************************/
/*                                                                           */
/*  void ContextDelete(CONTEXT cxt, NAMED named)                             */
/*                                                                           */
/*  Delete named entity "named" from cxt c.  It should definitely be there,  */
/*  in the last symbol table.                                                */
/*                                                                           */
/*****************************************************************************/

void ContextDelete(CONTEXT cxt, NAMED named)
{
  NAME name;
  name = NamedName(named);
  if( !SymDelete(ArrayLast(cxt->symbols), NameKey(name)) )
  {
    fprintf(stderr, "%s: internal error, can't delete \"%s\" from cxt\n",
      FilePosShow(NamedFilePos(named)), NameShow(name));
    abort();
  }
  if( DEBUG5 )
    fprintf(stderr, "  cxt deleted %s\n", UStringToUTF8(NameKey(name)));
}


/*****************************************************************************/
/*                                                                           */
/*  void ContextClearTopLevel(CONTEXT cxt)                                   */
/*                                                                           */
/*  Remove every entry from the top level of cxt.                            */
/*                                                                           */
/*****************************************************************************/

void ContextClearTopLevel(CONTEXT cxt)
{
  SymClear(ArrayLast(cxt->symbols));
}


/*****************************************************************************/
/*                                                                           */
/*  ContextDebug(CONTEXT cxt, ASTRING header, FILE *fp, int print_style)     */
/*                                                                           */
/*  Debug print of cxt with header onto fp in print_style.                   */
/*                                                                           */
/*****************************************************************************/

void ContextDebug(CONTEXT cxt, ASTRING header, FILE *fp, int print_style)
{
  USTRING key;  NAMED named;  SYMTAB_NAMED s;

  /* null context case */
  if( cxt == NULL )
  {
    fprintf(fp, "[ %s: context is NULL ]", header);
    return;
  }

  /* header */
  fprintf(fp, "[ %s", header);
  begin_indent;

  /* symbol table */
  ArrayForEachReverse(cxt->symbols, s)
  {
    if( s != ArrayLast(cxt->symbols) )
    {
      next_line;
      fprintf(fp, "--------------------");
    }

    SymForEach(s, &key, &named)
    {
      next_line;
      fprintf(fp, "%s (%s %p)", NameFullShow(NamedName(named)),
	KindShow(NamedKind(named)), (char *) named);
    }
  }

  /* wrapup */
  end_indent;
  next_line;
  fprintf(fp, "]");
}
