/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         expr.c                                                     */
/*  DESCRIPTION:  Nonpareil expressions                                      */
/*                                                                           */
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "externs.h"
#include "expr.h"
#define DEBUG1 0
#define DEBUG2 0


/*****************************************************************************/
/*                                                                           */
/*  Submodule "parse".                                                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  FILE_POS ExprFilePos(EXPR e)                                             */
/*                                                                           */
/*  Return the position of expression e.                                     */
/*                                                                           */
/*****************************************************************************/

FILE_POS ExprFilePos(EXPR e)
{
  return e->file_pos;
}


/*****************************************************************************/
/*                                                                           */
/*  void ExprSetParamName(EXPR e, USTRING val)                               */
/*                                                                           */
/*  Set the param token of e to a new token containing val.                  */
/*                                                                           */
/*****************************************************************************/

void ExprSetParamName(EXPR e, USTRING val)
{
  e->param_name = val;
}


/*****************************************************************************/
/*                                                                           */
/*  TOKEN ExprParamName(EXPR e)                                              */
/*                                                                           */
/*  Return the parameter name token of expression e (could be NULL).         */
/*                                                                           */
/*****************************************************************************/

USTRING ExprParamName(EXPR e)
{
  return e->param_name;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ParseFactorAndSuffixes(TOKEN *t, SYSTEM_VIEW sv, EXPR *res)      */
/*                                                                           */
/*  Parse a factor and its optional sequence of suffixes.                    */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN ParseFactorAndSuffixes(TOKEN *t, SYSTEM_VIEW sv, EXPR *res)
{
  NAMED named;  BOOLEAN finished;

  /* parse the factor and make it the current result */
  switch( LexType(curr_token) )
  {

    case TK_FALSE:
    case TK_TRUE:
    case TK_LIT_CHARACTER:
    case TK_LIT_STRING:
    case TK_LIT_INTEGER:
    case TK_LIT_REAL:

      /* literal */
      if( !ExprLitParse(t, sv, res) )
	return FALSE;
      break;


    case TK_BUILTIN:

      /* builtin */
      if( !ExprBuiltinParse(t, sv, res) )
	return FALSE;
      break;


    case TK_IDENTIFIER:
    case TK_EXCLAM:
    case TK_EXCLAM_EXCLAM:
    case TK_SELF:

      /* ordinary name, unclassed call, self */
      if( !ExprCallParse(t, sv, res) )
	return FALSE;
      break;


    case TK_LEFT_PAREN:

      /* parenthesized expression or tuple */
      if( !ExprParenOrTupleParse(t, sv, res) )
	return FALSE;
      break;


    case TK_LEFT_BRACKET:

      /* manifest lists */
      if( !ExprListParse(t, sv, res) )
	return FALSE;
      break;


    case TK_LEFT_BAR_BRACKET:

      /* manifest arrays */
      if( !ExprArrayParse(t, sv, res) )
	return FALSE;
      break;


    case TK_LET:

      /* let expression */
      if( !ExprLetParse(t, sv, res) )
	return FALSE;
      break;


    case TK_IF:

      /* if expression */
      if( !ExprIfParse(t, sv, res) )
	return FALSE;
      break;


    case TK_CASE:

      /* case expression */
      if( !ExprCaseParse(t, sv, res) )
	return FALSE;
      break;


    default:

      /* error case */
      fprintf(stderr, "%s: expected start of factor here\n",LexPos(curr_token));
      return FALSE;
      break;

  }

  /* now parse a sequence of zero or more factor suffixes */
  finished = FALSE;
  while( !finished ) switch( LexType(curr_token) )
  {

    case TK_DOT:

      /* parse feature call and *update* *res */
      if( !ExprCallFeatureParse(t, sv, res) )
	return FALSE;
      break;


    case TK_LEFT_PAREN:

      /* parse function application and *update *res */
      if( !ExprCallApplyParse(t, sv, res) )
	return FALSE;
      break;


    case TK_IDENTIFIER:
    case TK_PUNCTSEQ:

      if( SymRetrieve(SystemViewOpTable(sv), LexValue(curr_token), &named) &&
	  NameIsPostfixOperator(NamedName(named)) )
      {
	/* parse postfix operator and *update* res */
	if( !ExprCallPostfixOpParse(t, sv, NamedName(named), res) )
	  return FALSE;
      }
      else
      {
	/* not a postfix operator, so not a factor suffix, so stop */
	finished = TRUE;
      }
      break;


    default:

      /* not a factor suffix, so stop */
      finished = TRUE;
      break;

  }
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOL ExprDecoratedFactorParse(TOKEN *t, SYSTEM_VIEW sv, EXPR *res)       */
/*                                                                           */
/*  Parse one decorated factor.  Consult sv's operator table to find         */
/*  whether a name encountered is an operator or not, and if so which kind.  */
/*                                                                           */
/*  In order to handle the precedence of prefix and postfix operators        */
/*  correctly, this code implements the grammar                              */
/*                                                                           */
/*      <decorated_factor> ::= <factor_prefix> <decorated_factor>            */
/*                         ::= <factor> <factor_suffixes>                    */
/*                                                                           */
/*  This will naturally give any initial factor prefix low precedence.       */
/*                                                                           */
/*****************************************************************************/

USTRING UnderscorePrefix(TOKEN token)
{
  if( LexType(token) == TK_IDENTIFIER )
    return LexValue(token);
  else
    return UStringCat(AStringToUString("_"), LexValue(token));
}

BOOLEAN ExprDecoratedFactorParse(TOKEN *t, SYSTEM_VIEW sv, EXPR *res)
{
  NAMED named;

  if( LexType(curr_token) == TK_FUN )
  {
    /* fun keyword, parse anonymous function and following decorated factor */
    return ExprFunParse(t, sv, res);
  }
  else if( !LexIsName(curr_token) )
  {
    /* not name def, so parse <factor> <factor_suffixes> */
    return ParseFactorAndSuffixes(t, sv, res);
  }
  else if( !SymRetrieve(SystemViewOpTable(sv), UnderscorePrefix(curr_token),
      &named) )
  {
    /* not operator, so parse <factor> <factor_suffixes> */
    return ParseFactorAndSuffixes(t, sv, res);
  }
  else if( NameIsPrefixOperator(NamedName(named)) )
  {
    /* found prefix operator, so parse <prefix_op> <decorated_factor> */
    return ExprCallPrefixOpParse(t, sv, NamedName(named), res);
  }
  else
  {
    /* operator (must be identifier, incidentally), but wrong sort */
    fprintf(stderr, "%s: prefix operator expected here, but\n",
      LexPos(curr_token));
    fprintf(stderr, "  \"%s\" is defined as %s at %s, for example\n",
      LexShow(curr_token), KindShow(NamedKind(named)),
      FilePosShow(NamedFilePos(named)));
    return FALSE;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprParse(TOKEN *t, SYSTEM_VIEW sv, EXPR *res)                   */
/*                                                                           */
/*  Parse one expression.  Consult sv's operator table to find whether a     */
/*  name encountered is an operator or not, and if so which kind.  Also      */
/*  register literal strings with sv's predefined objects.                   */
/*                                                                           */
/*****************************************************************************/

/* implemented by a macro call to ExprCallInfixSeqParse in externs.h, since  */
/* it implements the grammar for general expressions                         */


/*****************************************************************************/
/*                                                                           */
/*  Submodule "uninstantiated copy".                                         */
/*                                                                           */
/*****************************************************************************/


/*****************************************************************************/
/*                                                                           */
/*  EXPR ExprCopyUninstantiated(EXPR e, ARRAY_FEFN_PARAM orig_params,        */
/*    ARRAY_FEFN_PARAM copy_params)                                          */
/*                                                                           */
/*  Make a copy of uninstantiated expression e.  Any references to           */
/*  elements of orig_params have to be changed to references to              */
/*  copy_params.  This function is used to copy the bodies of                */
/*  uninstantiated feature views declared together, as part of the           */
/*  process of disentangling them from each other.                           */
/*                                                                           */
/*****************************************************************************/

EXPR ExprCopyUninstantiated(EXPR e, ARRAY_FEFN_PARAM orig_params,
  ARRAY_FEFN_PARAM copy_params)
{
  EXPR r;
  switch( e->kind_tag )
  {
    case KIND_EXPR_LIT_BOOLEAN:
    case KIND_EXPR_LIT_CHAR:
    case KIND_EXPR_LIT_STRING:
    case KIND_EXPR_LIT_INTEGER:
    case KIND_EXPR_LIT_REAL:

      r = ExprLitCopyUninstantiated((EXPR_LIT) e, orig_params, copy_params);
      break;


    case KIND_EXPR_CALL:

      r = ExprCallCopyUninstantiated((EXPR_CALL) e, orig_params, copy_params);
      break;


    case KIND_EXPR_PAREN:

      r = ExprParenCopyUninstantiated((EXPR_PAREN) e, orig_params, copy_params);
      break;


    case KIND_EXPR_TUPLE:

      r = ExprTupleCopyUninstantiated((EXPR_TUPLE) e, orig_params, copy_params);
      break;


    case KIND_EXPR_LIST:

      r = ExprListCopyUninstantiated((EXPR_LIST) e, orig_params, copy_params);
      break;


    case KIND_EXPR_ARRAY:

      r = ExprArrayCopyUninstantiated((EXPR_ARRAY) e, orig_params, copy_params);
      break;


    case KIND_EXPR_LET:
    case KIND_EXPR_FUN:

      r = ExprLetCopyUninstantiated((EXPR_LET) e, orig_params, copy_params);
      break;


    case KIND_EXPR_IF:

      r = ExprIfCopyUninstantiated((EXPR_IF) e, orig_params, copy_params);
      break;


    case KIND_EXPR_CASE:

      r = ExprCaseCopyUninstantiated((EXPR_CASE) e, orig_params, copy_params);
      break;


    case KIND_EXPR_FNHEAD:

      r = ExprFnHeadCopyUninstantiated((EXPR_FNHEAD)e,orig_params,copy_params);
      break;


    case KIND_EXPR_PRECOND:

      r = ExprPrecondCopyUninstantiated((EXPR_PRECOND) e, orig_params,
	copy_params);
      break;


    case KIND_EXPR_DEFAULT:

      r = ExprDefaultCopyUninstantiated((EXPR_DEFAULT) e, orig_params,
	copy_params);
      break;


    default:

      assert(FALSE);
      break;

  }
  return r;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "manifest".                                                    */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprManifest(EXPR *e, CONTEXT cxt, TYPE self_type,               */
/*    BEFN encl_befn)                                                        */
/*                                                                           */
/*  Manifest expression *e in the given context, leaving its actual type     */
/*  in (*e)->type.  A pointer to the expression is passed since              */
/*  ExprManifest might choose to modify the expression tree in order to      */
/*  convert manifest tuples and lists to creation calls, and in that case    */
/*  it will change *e.                                                       */
/*                                                                           */
/*  Where there is a self_type, if will be passed in self_type.              */
/*                                                                           */
/*  The value returned is TRUE if the mission was successful, and FALSE      */
/*  otherwise.  This return value is always equal to the condition           */
/*  (*e)->type != NULL; that is, success always places a result type in      */
/*  (*e)->type, and failure never does.  The sole exception to this is       */
/*  noncreation features without bodies; they may have a body expression,    */
/*  containing preconditions and default values of parameters, but if        */
/*  there is no true body then (*e)->type will remain NULL.                  */
/*                                                                           */
/*  Code generation preparation                                              */
/*  ---------------------------                                              */
/*                                                                           */
/*  In addition to finding the type, this function also carries out          */
/*  some preparatory steps for code generation: it classifies all            */
/*  subexpressions as either small-scale or large-scale, and it passes       */
/*  all inner letdefs to encl_befn, which is the C function that will        */
/*  ultimately hold this expression, hence these letdefs should be           */
/*  code generated just before this C function is.                           */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ExprManifest(EXPR *e, CONTEXT cxt, TYPE self_type, BEFN encl_befn)
{
  static int indent = 0;
  BOOLEAN res;
  if( DEBUG1 )
  {
    fprintf(stderr, "%*s[ ExprManifest(", indent, "");
    ExprDebug(*e, cxt, FALSE, stderr, SINGLE_LINE);
    fprintf(stderr, ")\n");
    indent += 2;
  }

  /* distribute the call based on the kind of expression */
  switch( (*e)->kind_tag )
  {
    case KIND_EXPR_LIT_BOOLEAN:
    case KIND_EXPR_LIT_CHAR:
    case KIND_EXPR_LIT_STRING:
    case KIND_EXPR_LIT_INTEGER:
    case KIND_EXPR_LIT_REAL:

      res = ExprLitManifest((EXPR_LIT *) e, cxt, self_type, encl_befn);
      break;


    case KIND_EXPR_CALL:

      res = ExprCallManifest((EXPR_CALL *) e, cxt, self_type, encl_befn);
      break;


    case KIND_EXPR_PAREN:

      res = ExprParenManifest((EXPR_PAREN *) e, cxt, self_type, encl_befn);
      break;


    case KIND_EXPR_TUPLE:

      res = ExprTupleManifest((EXPR_TUPLE *) e, cxt, self_type, encl_befn);
      break;


    case KIND_EXPR_LIST:

      res = ExprListManifest((EXPR_LIST *) e, cxt, self_type, encl_befn);
      break;


    case KIND_EXPR_ARRAY:

      res = ExprArrayManifest((EXPR_ARRAY *) e, cxt, self_type, encl_befn);
      break;


    case KIND_EXPR_LET:
    case KIND_EXPR_FUN:

      res = ExprLetManifest((EXPR_LET *) e, cxt, self_type, encl_befn);
      break;


    case KIND_EXPR_IF:

      res = ExprIfManifest((EXPR_IF *) e, cxt, self_type, encl_befn);
      break;


    case KIND_EXPR_CASE:

      res = ExprCaseManifest((EXPR_CASE *) e, cxt, self_type, encl_befn);
      break;


    case KIND_EXPR_FNHEAD:

      res = ExprFnHeadManifest((EXPR_FNHEAD *) e, cxt, self_type, encl_befn);
      break;


    case KIND_EXPR_PRECOND:

      res = ExprPrecondManifest((EXPR_PRECOND *) e, cxt, self_type, encl_befn);
      break;


    case KIND_EXPR_DEFAULT:

      res = ExprDefaultManifest((EXPR_DEFAULT *) e, cxt, self_type, encl_befn);
      break;


    default:

      assert(FALSE);
      break;

  }

  if( DEBUG1 )
  {
    indent -= 2;
    fprintf(stderr, "%*s] ExprManifest(", indent, "");
    ExprDebug(*e, cxt, FALSE, stderr, SINGLE_LINE);
    if( res )
      fprintf(stderr, ") returning TRUE, type is %s\n",
	(*e)->type == NULL ? "NULL" : (ASTRING) TypeShow((*e)->type, cxt));
    else
      fprintf(stderr, ") returning FALSE (1)\n");
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  TYPE ExprType(EXPR e)                                                    */
/*                                                                           */
/*  Return the actual type of expression e, or NULL it not calculated yet.   */
/*                                                                           */
/*****************************************************************************/

TYPE ExprType(EXPR e)
{
  return e->type;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprLargeScale(EXPR e)                                           */
/*                                                                           */
/*  Assuming that ExprFindScale has been run over e, report whether e        */
/*  is a large-scale expression.                                             */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ExprLargeScale(EXPR e)
{
  return e->large_scale;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "constant expressions".                                        */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprIsConst(EXPR e, EXPR *root)                                  */
/*                                                                           */
/*  If e denotes a constant value (that is, either a literal or an element   */
/*  of an enumerated class), return TRUE and set *root to the root of e,     */
/*  that is, to the expression underlying any type casts on the surface.     */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ExprIsConst(EXPR e, EXPR *root)
{
  switch( e->kind_tag )
  {
    case KIND_EXPR_LIT_BOOLEAN:
    case KIND_EXPR_LIT_CHAR:
    case KIND_EXPR_LIT_STRING:
    case KIND_EXPR_LIT_INTEGER:
    case KIND_EXPR_LIT_REAL:

      *root = e;
      return TRUE;


    case KIND_EXPR_CALL:

      return ExprCallIsConst(e, root);


    case KIND_EXPR_PAREN:
    case KIND_EXPR_TUPLE:
    case KIND_EXPR_LIST:
    case KIND_EXPR_ARRAY:
    case KIND_EXPR_LET:
    case KIND_EXPR_FUN:
    case KIND_EXPR_IF:
    case KIND_EXPR_CASE:
    case KIND_EXPR_FNHEAD:
    case KIND_EXPR_PRECOND:
    case KIND_EXPR_DEFAULT:

      return  FALSE;


    default:

      fprintf(stderr, "%s: internal error in ExprIsConst: %s %d\n",
	FilePosShow(e->file_pos), "unexpected expression type", e->kind_tag);
      assert(FALSE);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  int ExprConstIntValue(EXPR e)                                            */
/*                                                                           */
/*  Assuming that e is a constant expression (as defined by ExprIsConst      */
/*  above) which has an integral value (i.e. its type is either integral     */
/*  or enumerated), return that integer value.                               */
/*                                                                           */
/*****************************************************************************/

int ExprConstIntValue(EXPR e)
{
  EXPR root;

  /* get to the root of e */
  if( !ExprIsConst(e, &root) )
    assert(FALSE);

  /* value depends on type of expression */
  switch( root->kind_tag )
  {
    case KIND_EXPR_LIT_BOOLEAN:
    case KIND_EXPR_LIT_CHAR:
    case KIND_EXPR_LIT_INTEGER:
    case KIND_EXPR_LIT_REAL:

      return ExprLitConstIntValue(root);


    case KIND_EXPR_CALL:

      return ExprCallConstIntValue(root);


    default:

      fprintf(stderr, "%s: internal error in ExprConstIntValue: %s %d\n",
	FilePosShow(root->file_pos), "unexpected expression type",
	root->kind_tag);
      assert(FALSE);
      return 0;  /* keep compiler happy */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  int ExprConstCmp(EXPR e1, EXPR e2)                                       */
/*                                                                           */
/*  Assuming that e1 and e2 are constant expressions (as defined by          */
/*  ExprIsConst above) of the same type, return negative, zero, or           */
/*  positive accordingly as e1 is considered to be less than, equal to,      */
/*  or greater than e2.                                                      */
/*                                                                           */
/*  This code is willing to compare integer expressions with values of       */
/*  enumerated classes.  This is because the min_value and max_value         */
/*  expressions stored in enumerated classes are integers, and we            */
/*  want to be able to compare them with elements of the class.              */
/*                                                                           */
/*****************************************************************************/

int ExprConstCmp(EXPR e1, EXPR e2)
{
  EXPR root1, root2;

  /* get to the root of both expressions */
  if( !ExprIsConst(e1, &root1) )
    assert(FALSE);
  if( !ExprIsConst(e2, &root2) )
    assert(FALSE);

  switch( root1->kind_tag )
  {
    case KIND_EXPR_LIT_STRING:
    case KIND_EXPR_LIT_REAL:

      /* strings and reals compare only with themselves */
      return ExprLitConstCmp(root1, root2);


    case KIND_EXPR_LIT_BOOLEAN:
    case KIND_EXPR_LIT_INTEGER:
    case KIND_EXPR_CALL:

      /* integral types compare with each other */
      return ExprConstIntValue(root1) - ExprConstIntValue(root2);


    default:

      /* other types should not be mentioned */
      fprintf(stderr, "%s: internal error in ExprConstCmp: %s %d\n",
	FilePosShow(root1->file_pos), "unexpected expression type",
	root1->kind_tag);
      assert(FALSE);
      return 0;  /* keep compiler happy */
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "code generation".                                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void ExprCodeGen(EXPR e, CODEGEN_OBJ res_be_var,                         */
/*    CODEGEN_TYPE res_be_type, CODEGEN be)                                  */
/*                                                                           */
/*  Print e as a C expression onto back end be.                              */
/*  At the same time, make up C names for the parameters and other locals.   */
/*                                                                           */
/*  The result has to be assigned to res_be_var, which could be NULL         */
/*  meaning to not assign to anything, or it could be a variety of other     */
/*  things, responsibility for which rests with procedure VarAsstBegin()     */
/*  in the back end module.                                                  */
/*                                                                           */
/*  Parameter res_be_type is the backend type that the surrounding generated */
/*  code expects this expression to have.  The type must be a supertype of   */
/*  e->type, or else the type checking would have reported an error before   */
/*  now; but if the backend type of e->type differs from res_be_type, then   */
/*  a cast will have to be inserted to keep the C compiler happy, since      */
/*  it does not understand subtyping.                                        */
/*                                                                           */
/*  An important feature of ExprCodeGen is that it leaves the expression     */
/*  e in the same state in which it was received.  ExprCodeGen may make      */
/*  temporary changes (e.g. it sets e->be_var to indicate that the code      */
/*  has already been generated and the result placed in a variable of        */
/*  that name), but by the time ExprCodeGen returns, all such temporary      */
/*  changes have been undone.  Because of this, it is always safe to call    */
/*  ExprCodeGen twice on the same expression.  This is needed because when   */
/*  preconditions are inherited they are code generated at several points.   */
/*                                                                           */
/*****************************************************************************/

void ExprCodeGen(EXPR e, CODEGEN_OBJ res_be_var, CODEGEN_TYPE res_be_type,
  CODEGEN be)
{
  if( DEBUG2 )
  {
    fprintf(stderr, "[ ExprCodeGen(");
    ExprDebug(e, NULL, FALSE, stderr, SINGLE_LINE);
    fprintf(stderr, ", %s, %s, be)\n", be->VarShow(res_be_var),
      be->TypeShow(res_be_type));
  }

  /* if expression has a be_var, printing it is all we do */
  if( e->be_var != NULL )
  {
    assert(res_be_var == NULL);
    Var(e->be_var);
    e->be_var = NULL;  /* get back to original state */
  }

  /* else do the print */
  else switch( e->kind_tag )
  {
    case KIND_EXPR_LIT_BOOLEAN:
    case KIND_EXPR_LIT_CHAR:
    case KIND_EXPR_LIT_STRING:
    case KIND_EXPR_LIT_INTEGER:
    case KIND_EXPR_LIT_REAL:

      ExprLitCodeGen((EXPR_LIT) e, res_be_var, res_be_type, be);
      break;


    case KIND_EXPR_CALL:

      ExprCallCodeGen((EXPR_CALL) e, res_be_var, res_be_type, be);
      break;


    case KIND_EXPR_PAREN:

      ExprParenCodeGen((EXPR_PAREN) e, res_be_var, res_be_type, be);
      break;


    case KIND_EXPR_TUPLE:
    case KIND_EXPR_LIST:

      /* should be compiled away by now */
      assert(FALSE);


    case KIND_EXPR_ARRAY:

      ExprArrayCodeGen((EXPR_ARRAY) e, res_be_var, res_be_type, be);
      break;


    case KIND_EXPR_LET:
    case KIND_EXPR_FUN:

      ExprLetCodeGen((EXPR_LET) e, res_be_var, res_be_type, be);
      break;


    case KIND_EXPR_IF:

      ExprIfCodeGen((EXPR_IF) e, res_be_var, res_be_type, be);
      break;


    case KIND_EXPR_CASE:

      ExprCaseCodeGen((EXPR_CASE) e, res_be_var, res_be_type, be);
      break;


    case KIND_EXPR_FNHEAD:

      ExprFnHeadCodeGen((EXPR_FNHEAD) e, res_be_var, res_be_type, be);
      break;


    case KIND_EXPR_PRECOND:

      ExprPrecondCodeGen((EXPR_PRECOND) e, res_be_var, res_be_type, be);
      break;


    case KIND_EXPR_DEFAULT:

      ExprDefaultCodeGen((EXPR_DEFAULT) e, res_be_var, res_be_type, be);
      break;


    default:

      fprintf(stderr, "%s: internal error in ExprCodeGen: %s %d\n",
	FilePosShow(e->file_pos), "unexpected expression type", e->kind_tag);
      assert(FALSE);
  }
  if( DEBUG2 )
    fprintf(stderr, "] ExprCodeGen returning\n");
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprInitOrder(EXPR e, int visit_num, BOOLEAN *report,            */
/*    BEFN_SYSTEM_INIT fun)                                                  */
/*                                                                           */
/*  Helper function for BEFnInitOrder, which finds a suitable                */
/*  order in which to initialize the predefined objects of a system.         */
/*                                                                           */
/*  This function traverses an expression calls BEFnInitOrder for each       */
/*  back-end feature which is called from within the expression.  It         */
/*  assumes that all these expressions have been manifested, so the          */
/*  back-end functions are available.                                        */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ExprInitOrder(EXPR e, int visit_num, BOOLEAN *report,
  BEFN_SYSTEM_INIT fun)
{
  switch( e->kind_tag )
  {
    case KIND_EXPR_LIT_BOOLEAN:
    case KIND_EXPR_LIT_CHAR:
    case KIND_EXPR_LIT_STRING:
    case KIND_EXPR_LIT_INTEGER:
    case KIND_EXPR_LIT_REAL:

      return ExprLitInitOrder((EXPR_LIT) e, visit_num, report, fun);


    case KIND_EXPR_CALL:

      return ExprCallInitOrder((EXPR_CALL) e, visit_num, report, fun);


    case KIND_EXPR_PAREN:

      return ExprParenInitOrder((EXPR_PAREN) e, visit_num, report, fun);


    case KIND_EXPR_TUPLE:
    case KIND_EXPR_LIST:

      /* should be compiled away by now */
      assert(FALSE);
      return FALSE; /* keep compiler happy */


    case KIND_EXPR_ARRAY:

      return ExprArrayInitOrder((EXPR_ARRAY) e, visit_num, report, fun);


    case KIND_EXPR_LET:
    case KIND_EXPR_FUN:

      return ExprLetInitOrder((EXPR_LET) e, visit_num, report, fun);


    case KIND_EXPR_IF:

      return ExprIfInitOrder((EXPR_IF) e, visit_num, report, fun);


    case KIND_EXPR_CASE:

      return ExprCaseInitOrder((EXPR_CASE) e, visit_num, report, fun);


    case KIND_EXPR_FNHEAD:

      return ExprFnHeadInitOrder((EXPR_FNHEAD) e, visit_num, report, fun);


    case KIND_EXPR_PRECOND:

      return ExprPrecondInitOrder((EXPR_PRECOND) e, visit_num, report, fun);


    case KIND_EXPR_DEFAULT:

      return ExprDefaultInitOrder((EXPR_DEFAULT) e, visit_num, report, fun);


    default:

      fprintf(stderr, "%s: internal error in ExprInitOrder: %s %d\n",
	FilePosShow(e->file_pos), "unexpected expression type", e->kind_tag);
      assert(FALSE);
      return FALSE; /* keep compiler happy */

  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "debug".                                                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void ExprDebug(EXPR e, CONTEXT cxt, BOOLEAN show_types, FILE *fp,        */
/*    int print_style)                                                       */
/*                                                                           */
/*  Debug print of expr e on *fp.  If show_types is TRUE, annotate the       */
/*  print with the type of each subexpression.  Some of the entries may      */
/*  be NULL.                                                                 */
/*                                                                           */
/*  See externs.h for an explanation of print_style.                         */
/*                                                                           */
/*****************************************************************************/

void ExprArrayDebug(ARRAY_EXPR ae, CONTEXT cxt, BOOLEAN show_types, FILE *fp)
{
  EXPR p;  int i;
  if( ae == NULL )
    fprintf(fp, "<null>");
  else
  {
    fprintf(fp, "[");
    for( i = 0;  i < ArraySize(ae);  i++ )
    {
      p = ArrayGet(ae, i);
      if( i > 0 )
	fprintf(fp, ", ");
      if( p != NULL )
	ExprDebug(p, cxt, show_types, fp, SINGLE_LINE);
      else
	fprintf(fp, "_");
    }
    fprintf(fp, "]");
  }
}

void ExprDebug(EXPR e, CONTEXT cxt, BOOLEAN show_types, FILE *fp,
  int print_style)
{
  /* indicate when be_var is non-NULL */
  if( e->be_var != NULL )
    fprintf(fp, "**(");

  /* show param_name if present */
  if( e->param_name != NULL )
    fprintf(fp, "%s := ", UStringToUTF8(e->param_name));

  /* ascriptions if requested */
  if( show_types && e->type != NULL )
    fprintf(fp, "(");

  /* show the expression proper */
  switch( e->kind_tag )
  {
    case KIND_EXPR_LIT_BOOLEAN:
    case KIND_EXPR_LIT_CHAR:
    case KIND_EXPR_LIT_STRING:
    case KIND_EXPR_LIT_INTEGER:
    case KIND_EXPR_LIT_REAL:

      ExprLitDebug((EXPR_LIT) e, cxt, show_types, fp, print_style);
      break;


    case KIND_EXPR_CALL:

      ExprCallDebug((EXPR_CALL) e, cxt, show_types, fp, print_style);
      break;


    case KIND_EXPR_PAREN:

      ExprParenDebug((EXPR_PAREN) e, cxt, show_types, fp, print_style);
      break;


    case KIND_EXPR_TUPLE:

      ExprTupleDebug((EXPR_TUPLE) e, cxt, show_types, fp, print_style);
      break;


    case KIND_EXPR_LIST:

      ExprListDebug((EXPR_LIST) e, cxt, show_types, fp, print_style);
      break;


    case KIND_EXPR_ARRAY:

      ExprManifestArrayDebug((EXPR_ARRAY) e, cxt, show_types, fp, print_style);
      break;


    case KIND_EXPR_LET:
    case KIND_EXPR_FUN:

      ExprLetDebug((EXPR_LET) e, cxt, show_types, fp, print_style);
      break;


    case KIND_EXPR_IF:

      ExprIfDebug((EXPR_IF) e, cxt, show_types, fp, print_style);
      break;


    case KIND_EXPR_CASE:

      ExprCaseDebug((EXPR_CASE) e, cxt, show_types, fp, print_style);
      break;


    case KIND_EXPR_FNHEAD:

      ExprFnHeadDebug((EXPR_FNHEAD) e, cxt, show_types, fp, print_style);
      break;


    case KIND_EXPR_PRECOND:

      ExprPrecondDebug((EXPR_PRECOND) e, cxt, show_types, fp, print_style);
      break;


    case KIND_EXPR_DEFAULT:

      ExprDefaultDebug((EXPR_DEFAULT) e, cxt, show_types, fp, print_style);
      break;


    default:

      assert(FALSE);
      break;

  }
  if( show_types && e->type != NULL )
    fprintf(fp, " :%s)", TypeShow(e->type, cxt));
  if( e->be_var != NULL )
    fprintf(fp, ")");
}
